/*
 * Decompiled with CFR 0.152.
 */
package org.apache.accumulo.tserver.compactions;

import com.google.common.base.Preconditions;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Sets;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.accumulo.core.conf.Property;
import org.apache.accumulo.core.data.TableId;
import org.apache.accumulo.core.dataImpl.KeyExtent;
import org.apache.accumulo.core.metadata.schema.ExternalCompactionId;
import org.apache.accumulo.core.spi.compaction.CompactionExecutorId;
import org.apache.accumulo.core.spi.compaction.CompactionKind;
import org.apache.accumulo.core.spi.compaction.CompactionServiceId;
import org.apache.accumulo.core.spi.compaction.CompactionServices;
import org.apache.accumulo.core.tabletserver.thrift.TCompactionQueueSummary;
import org.apache.accumulo.core.util.Pair;
import org.apache.accumulo.core.util.Retry;
import org.apache.accumulo.core.util.compaction.CompactionExecutorIdImpl;
import org.apache.accumulo.core.util.compaction.CompactionServicesConfig;
import org.apache.accumulo.core.util.threads.Threads;
import org.apache.accumulo.server.ServerContext;
import org.apache.accumulo.tserver.OpeningAndOnlineCompactables;
import org.apache.accumulo.tserver.compactions.Compactable;
import org.apache.accumulo.tserver.compactions.CompactionExecutor;
import org.apache.accumulo.tserver.compactions.CompactionService;
import org.apache.accumulo.tserver.compactions.ExternalCompactionExecutor;
import org.apache.accumulo.tserver.compactions.ExternalCompactionJob;
import org.apache.accumulo.tserver.metrics.CompactionExecutorsMetrics;
import org.apache.accumulo.tserver.tablet.Tablet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionManager {
    private static final Logger log = LoggerFactory.getLogger(CompactionManager.class);
    private final Supplier<OpeningAndOnlineCompactables> compactables;
    private volatile Map<CompactionServiceId, CompactionService> services;
    private final LinkedBlockingQueue<Compactable> compactablesToCheck = new LinkedBlockingQueue();
    private final long maxTimeBetweenChecks;
    private final ServerContext context;
    private CompactionServicesConfig currentCfg;
    private long lastConfigCheckTime = System.nanoTime();
    private final CompactionExecutorsMetrics ceMetrics;
    private String lastDeprecationWarning = "";
    private final Map<CompactionExecutorId, ExternalCompactionExecutor> externalExecutors;
    private final Map<ExternalCompactionId, ExtCompInfo> runningExternalCompactions;
    private final Cache<Pair<TableId, CompactionServiceId>, Long> unknownCompactionServiceErrorCache;

    private void warnAboutDeprecation(String warning) {
        if (!warning.equals(this.lastDeprecationWarning)) {
            log.warn(warning);
            this.lastDeprecationWarning = warning;
        }
    }

    private void mainLoop() {
        long lastCheckAllTime = System.nanoTime();
        long increment = Math.max(1L, this.maxTimeBetweenChecks / 10L);
        Retry.RetryFactory retryFactory = Retry.builder().infiniteRetries().retryAfter(increment, TimeUnit.MILLISECONDS).incrementBy(increment, TimeUnit.MILLISECONDS).maxWait(this.maxTimeBetweenChecks, TimeUnit.MILLISECONDS).backOffFactor(1.07).logInterval(1L, TimeUnit.MINUTES).createFactory();
        Retry retry = retryFactory.createRetry();
        Compactable last = null;
        while (true) {
            try {
                while (true) {
                    long passed;
                    if ((passed = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - lastCheckAllTime)) >= this.maxTimeBetweenChecks) {
                        HashMap<ExternalCompactionId, ExtCompInfo> runningEcids = new HashMap<ExternalCompactionId, ExtCompInfo>(this.runningExternalCompactions);
                        OpeningAndOnlineCompactables compactablesSnapshot = this.compactables.get();
                        for (Tablet tablet : compactablesSnapshot.online.values()) {
                            Compactable compactable;
                            last = compactable = tablet.asCompactable();
                            this.submitCompaction(compactable);
                            compactable.getExternalCompactionIds(runningEcids::remove);
                        }
                        lastCheckAllTime = System.nanoTime();
                        runningEcids.values().removeIf(extCompInfo -> compactablesSnapshot.opening.contains(extCompInfo.extent));
                        runningEcids.forEach((ecid, info) -> log.debug("Removing unknown external compaction {} {} from runningExternalCompactions", ecid, (Object)info.extent));
                        this.runningExternalCompactions.keySet().removeAll(runningEcids.keySet());
                    } else {
                        Compactable compactable = this.compactablesToCheck.poll(this.maxTimeBetweenChecks - passed, TimeUnit.MILLISECONDS);
                        if (compactable != null) {
                            last = compactable;
                            this.submitCompaction(compactable);
                        }
                    }
                    last = null;
                    if (retry.hasRetried()) {
                        retry = retryFactory.createRetry();
                    }
                    this.checkForConfigChanges(false);
                }
            }
            catch (Exception e) {
                KeyExtent extent = last == null ? null : last.getExtent();
                log.warn("Failed to compact {} ", (Object)extent, (Object)e);
                retry.useRetry();
                try {
                    retry.waitForNextAttempt(log, "compaction initiation loop");
                    continue;
                }
                catch (InterruptedException e1) {
                    log.debug("Retry interrupted", (Throwable)e1);
                    continue;
                }
            }
            break;
        }
    }

    private void submitCompaction(Compactable compactable) {
        for (CompactionKind ctype : CompactionKind.values()) {
            CompactionServiceId csid = compactable.getConfiguredService(ctype);
            CompactionService service = this.services.get(csid);
            if (service == null) {
                Pair cacheKey;
                Long last;
                this.checkForConfigChanges(true);
                service = this.services.get(csid);
                if (service == null && (last = (Long)this.unknownCompactionServiceErrorCache.getIfPresent((Object)(cacheKey = new Pair((Object)compactable.getTableId(), (Object)csid)))) == null) {
                    log.error("Tablet {} returned non-existent compaction service {} for compaction type {}.  Check the table compaction dispatcher configuration. No compactions will happen until the configuration is fixed. This log message is temporarily suppressed for the entire table.", new Object[]{compactable.getExtent(), csid, ctype});
                    this.unknownCompactionServiceErrorCache.put((Object)cacheKey, (Object)System.currentTimeMillis());
                }
            }
            if (service == null) continue;
            service.submitCompaction(ctype, compactable, this.compactablesToCheck::add);
        }
    }

    public CompactionManager(Supplier<OpeningAndOnlineCompactables> compactables, ServerContext context, CompactionExecutorsMetrics ceMetrics) {
        this.compactables = compactables;
        this.currentCfg = new CompactionServicesConfig(context.getConfiguration(), this::warnAboutDeprecation);
        this.context = context;
        this.ceMetrics = ceMetrics;
        this.externalExecutors = new ConcurrentHashMap<CompactionExecutorId, ExternalCompactionExecutor>();
        this.runningExternalCompactions = new ConcurrentHashMap<ExternalCompactionId, ExtCompInfo>();
        HashMap tmpServices = new HashMap();
        this.unknownCompactionServiceErrorCache = CacheBuilder.newBuilder().expireAfterWrite(5L, TimeUnit.MINUTES).build();
        this.currentCfg.getPlanners().forEach((serviceName, plannerClassName) -> {
            try {
                tmpServices.put(CompactionServiceId.of((String)serviceName), new CompactionService((String)serviceName, (String)plannerClassName, this.currentCfg.getRateLimit(serviceName), this.currentCfg.getOptions().getOrDefault(serviceName, Map.of()), context, ceMetrics, this::getExternalExecutor));
            }
            catch (RuntimeException e) {
                log.error("Failed to create compaction service {} with planner:{} options:{}", new Object[]{serviceName, plannerClassName, this.currentCfg.getOptions().getOrDefault(serviceName, Map.of()), e});
            }
        });
        this.services = Map.copyOf(tmpServices);
        this.maxTimeBetweenChecks = context.getConfiguration().getTimeInMillis(Property.TSERV_MAJC_DELAY);
        ceMetrics.setExternalMetricsSupplier(this::getExternalMetrics);
    }

    public void compactableChanged(Compactable compactable) {
        this.compactablesToCheck.add(compactable);
    }

    private synchronized void checkForConfigChanges(boolean force) {
        try {
            long secondsSinceLastCheck = TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - this.lastConfigCheckTime);
            if (!force && secondsSinceLastCheck < 1L) {
                return;
            }
            this.lastConfigCheckTime = System.nanoTime();
            CompactionServicesConfig tmpCfg = new CompactionServicesConfig(this.context.getConfiguration(), this::warnAboutDeprecation);
            if (!this.currentCfg.equals((Object)tmpCfg)) {
                HashMap tmpServices = new HashMap();
                tmpCfg.getPlanners().forEach((serviceName, plannerClassName) -> {
                    try {
                        CompactionServiceId csid = CompactionServiceId.of((String)serviceName);
                        CompactionService service = this.services.get(csid);
                        if (service == null) {
                            tmpServices.put(csid, new CompactionService((String)serviceName, (String)plannerClassName, tmpCfg.getRateLimit(serviceName), tmpCfg.getOptions().getOrDefault(serviceName, Map.of()), this.context, this.ceMetrics, this::getExternalExecutor));
                        } else {
                            service.configurationChanged((String)plannerClassName, tmpCfg.getRateLimit(serviceName), tmpCfg.getOptions().getOrDefault(serviceName, Map.of()));
                            tmpServices.put(csid, service);
                        }
                    }
                    catch (RuntimeException e) {
                        throw new RuntimeException("Failed to create or update compaction service " + serviceName + " with planner:" + plannerClassName + " options:" + String.valueOf(tmpCfg.getOptions().getOrDefault(serviceName, Map.of())), e);
                    }
                });
                Sets.SetView deletedServices = Sets.difference(this.services.keySet(), tmpServices.keySet());
                for (CompactionServiceId dcsid : deletedServices) {
                    this.services.get(dcsid).stop();
                }
                this.currentCfg = tmpCfg;
                this.services = Map.copyOf(tmpServices);
                HashSet activeExternalExecs = new HashSet();
                this.services.values().forEach(cs -> cs.getExternalExecutorsInUse(activeExternalExecs::add));
                this.externalExecutors.keySet().retainAll(activeExternalExecs);
            }
        }
        catch (RuntimeException e) {
            log.error("Failed to reconfigure compaction services ", (Throwable)e);
        }
    }

    public void start() {
        log.debug("Started compaction manager");
        Threads.createCriticalThread((String)"Compaction Manager", () -> this.mainLoop()).start();
    }

    public CompactionServices getServices() {
        final Set<CompactionServiceId> serviceIds = this.services.keySet();
        return new CompactionServices(){

            public Set<CompactionServiceId> getIds() {
                return serviceIds;
            }
        };
    }

    public boolean isCompactionQueued(KeyExtent extent, Set<CompactionServiceId> servicesUsed) {
        return servicesUsed.stream().map(this.services::get).filter(Objects::nonNull).anyMatch(compactionService -> compactionService.isCompactionQueued(extent));
    }

    public int getCompactionsRunning() {
        return this.services.values().stream().mapToInt(cs -> cs.getCompactionsRunning(CompactionExecutor.CType.INTERNAL)).sum() + this.runningExternalCompactions.size();
    }

    public int getCompactionsQueued() {
        return this.services.values().stream().mapToInt(cs -> cs.getCompactionsQueued(CompactionExecutor.CType.INTERNAL)).sum() + this.externalExecutors.values().stream().mapToInt(ee -> ee.getCompactionsQueued(CompactionExecutor.CType.EXTERNAL)).sum();
    }

    public ExternalCompactionJob reserveExternalCompaction(String queueName, long priority, String compactorId, ExternalCompactionId externalCompactionId) {
        log.debug("Attempting to reserve external compaction, queue:{} priority:{} compactor:{}", new Object[]{queueName, priority, compactorId});
        ExternalCompactionExecutor extCE = this.getExternalExecutor(queueName);
        ExternalCompactionJob ecJob = extCE.reserveExternalCompaction(priority, compactorId, externalCompactionId);
        if (ecJob != null) {
            this.runningExternalCompactions.put(ecJob.getExternalCompactionId(), new ExtCompInfo(ecJob.getExtent(), extCE.getId()));
            log.debug("Reserved external compaction {} {}", (Object)ecJob.getExternalCompactionId(), (Object)ecJob.getExtent());
        }
        return ecJob;
    }

    ExternalCompactionExecutor getExternalExecutor(CompactionExecutorId ceid) {
        return this.externalExecutors.computeIfAbsent(ceid, id -> new ExternalCompactionExecutor((CompactionExecutorId)id));
    }

    ExternalCompactionExecutor getExternalExecutor(String queueName) {
        return this.getExternalExecutor(CompactionExecutorIdImpl.externalId((String)queueName));
    }

    public void registerExternalCompaction(ExternalCompactionId ecid, KeyExtent extent, CompactionExecutorId ceid) {
        this.runningExternalCompactions.put(ecid, new ExtCompInfo(extent, ceid));
        log.trace("registered external compaction {} {}", (Object)ecid, (Object)extent);
    }

    public void commitExternalCompaction(ExternalCompactionId extCompactionId, KeyExtent extentCompacted, Map<KeyExtent, Tablet> currentTablets, long fileSize, long entries) {
        ExtCompInfo ecInfo = this.runningExternalCompactions.get(extCompactionId);
        if (ecInfo != null) {
            Preconditions.checkState((boolean)ecInfo.extent.equals((Object)extentCompacted), (String)"Unexpected extent seen on compaction commit %s %s", (Object)ecInfo.extent, (Object)extentCompacted);
            Tablet tablet = currentTablets.get(ecInfo.extent);
            if (tablet != null) {
                log.debug("Attempting to commit external compaction {} {}", (Object)extCompactionId, (Object)tablet.getExtent());
                tablet.asCompactable().commitExternalCompaction(extCompactionId, fileSize, entries);
                this.compactablesToCheck.add(tablet.asCompactable());
                this.runningExternalCompactions.remove(extCompactionId);
                log.trace("Committed external compaction {} {}", (Object)extCompactionId, (Object)tablet.getExtent());
            } else {
                log.debug("Ignoring request to commit {} {} because its not in set of known tablets", (Object)extCompactionId, (Object)ecInfo.extent);
            }
        } else {
            log.debug("Ignoring request to commit {} because its not in runningExternalCompactions", (Object)extCompactionId);
        }
    }

    public void externalCompactionFailed(ExternalCompactionId ecid, KeyExtent extentCompacted, Map<KeyExtent, Tablet> currentTablets) {
        ExtCompInfo ecInfo = this.runningExternalCompactions.get(ecid);
        if (ecInfo != null) {
            Preconditions.checkState((boolean)ecInfo.extent.equals((Object)extentCompacted), (String)"Unexpected extent seen on compaction commit %s %s", (Object)ecInfo.extent, (Object)extentCompacted);
            Tablet tablet = currentTablets.get(ecInfo.extent);
            if (tablet != null) {
                log.debug("Attempting to fail external compaction {} {}", (Object)ecid, (Object)tablet.getExtent());
                tablet.asCompactable().externalCompactionFailed(ecid);
                this.compactablesToCheck.add(tablet.asCompactable());
                this.runningExternalCompactions.remove(ecid);
                log.trace("Failed external compaction {} {}", (Object)ecid, (Object)tablet.getExtent());
            } else {
                log.debug("Ignoring request to fail {} {} because its not in set of known tablets", (Object)ecid, (Object)ecInfo.extent);
            }
        } else {
            log.debug("Ignoring request to fail {} because its not in runningExternalCompactions", (Object)ecid);
        }
    }

    public List<TCompactionQueueSummary> getCompactionQueueSummaries() {
        return this.externalExecutors.values().stream().flatMap(ece -> ece.summarize()).collect(Collectors.toList());
    }

    public Collection<ExtCompMetric> getExternalMetrics() {
        HashMap metrics = new HashMap();
        this.externalExecutors.forEach((eeid, ece) -> {
            ExtCompMetric ecm = new ExtCompMetric();
            ecm.ceid = eeid;
            ecm.queued = ece.getCompactionsQueued(CompactionExecutor.CType.EXTERNAL);
            metrics.put(eeid, ecm);
        });
        this.runningExternalCompactions.values().forEach(eci -> {
            ExtCompMetric ecm = metrics.computeIfAbsent(eci.executor, id -> {
                ExtCompMetric newEcm = new ExtCompMetric();
                newEcm.ceid = id;
                return newEcm;
            });
            ++ecm.running;
        });
        return metrics.values();
    }

    public void compactableClosed(KeyExtent extent, Set<CompactionServiceId> servicesUsed, Set<ExternalCompactionId> ecids) {
        this.runningExternalCompactions.keySet().removeAll(ecids);
        if (log.isTraceEnabled()) {
            ecids.forEach(ecid -> log.trace("Removed {} from runningExternalCompactions for {} as part of close", ecid, (Object)extent));
        }
        servicesUsed.stream().map(this.services::get).filter(Objects::nonNull).forEach(compService -> compService.compactableClosed(extent));
    }

    static class ExtCompInfo {
        final KeyExtent extent;
        final CompactionExecutorId executor;

        public ExtCompInfo(KeyExtent extent, CompactionExecutorId executor) {
            this.extent = extent;
            this.executor = executor;
        }
    }

    public static class ExtCompMetric {
        public CompactionExecutorId ceid;
        public int running;
        public int queued;
    }
}

