(function(/*! Brunch !*/) {
  'use strict';

  var globals = typeof window !== 'undefined' ? window : global;
  if (typeof globals.require === 'function') return;

  var modules = {};
  var cache = {};

  var has = function(object, name) {
    return ({}).hasOwnProperty.call(object, name);
  };

  var expand = function(root, name) {
    var results = [], parts, part;
    if (/^\.\.?(\/|$)/.test(name)) {
      parts = [root, name].join('/').split('/');
    } else {
      parts = name.split('/');
    }
    for (var i = 0, length = parts.length; i < length; i++) {
      part = parts[i];
      if (part === '..') {
        results.pop();
      } else if (part !== '.' && part !== '') {
        results.push(part);
      }
    }
    return results.join('/');
  };

  var dirname = function(path) {
    return path.split('/').slice(0, -1).join('/');
  };

  var localRequire = function(path) {
    return function(name) {
      var dir = dirname(path);
      var absolute = expand(dir, name);
      return globals.require(absolute, path);
    };
  };

  var initModule = function(name, definition) {
    var module = {id: name, exports: {}};
    cache[name] = module;
    definition(module.exports, localRequire(name), module);
    return module.exports;
  };

  var require = function(name, loaderPath) {
    var path = expand(name, '.');
    if (loaderPath == null) loaderPath = '/';

    if (has(cache, path)) return cache[path].exports;
    if (has(modules, path)) return initModule(path, modules[path]);

    var dirIndex = expand(path, './index');
    if (has(cache, dirIndex)) return cache[dirIndex].exports;
    if (has(modules, dirIndex)) return initModule(dirIndex, modules[dirIndex]);

    throw new Error('Cannot find module "' + name + '" from '+ '"' + loaderPath + '"');
  };

  var define = function(bundle, fn) {
    if (typeof bundle === 'object') {
      for (var key in bundle) {
        if (has(bundle, key)) {
          modules[key] = bundle[key];
        }
      }
    } else {
      modules[bundle] = fn;
    }
  };

  var list = function() {
    var result = [];
    for (var item in modules) {
      if (has(modules, item)) {
        result.push(item);
      }
    }
    return result;
  };

  globals.require = require;
  globals.require.define = define;
  globals.require.register = define;
  globals.require.list = list;
  globals.require.brunch = true;
})();
require.register("app", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

if (Ember.$.uuid === undefined) {
  Ember.$.uuid = 0;
}

// Application bootstrapper
require('utils/bootstrap_reopen');
require('utils/ember_reopen');
require('utils/ember_computed');
var stringUtils = require('utils/string_utils');
var stompClientClass = require('utils/stomp_client');

module.exports = Em.Application.create({
  name: 'Ambari Web',
  rootElement: '#wrapper',

  store: DS.Store.create({
    revision: 4,
    adapter: DS.FixtureAdapter.create({
      simulateRemoteResponse: false
    }),
    typeMaps: {},
    recordCache: []
  }),
  StompClient: stompClientClass.create(),
  isAdmin: false,
  isOperator: false,
  isClusterUser: false,
  isPermissionDataLoaded: false,
  auth: undefined,
  isOnlyViewUser: function () {
    return App.auth && (App.auth.length == 0 || App.isAuthorized('VIEW.USE') && App.auth.length == 1);
  }.property('auth'),

  /**
   * @type {boolean}
   * @default false
   */
  isKerberosEnabled: false,

  /**
   * state of stack upgrade process
   * states:
   *  - NOT_REQUIRED
   *  - PENDING
   *  - IN_PROGRESS
   *  - HOLDING
   *  - COMPLETED
   *  - ABORTED
   *  - HOLDING_FAILED
   *  - HOLDING_TIMEDOUT
   * @type {String}
   */
  upgradeState: 'NOT_REQUIRED',

  /**
   * Check if upgrade is in INIT state
   * 'INIT' is set on upgrade start and when it's finished
   * @type {boolean}
   */
  upgradeInit: Em.computed.equal('upgradeState', 'NOT_REQUIRED'),

  /**
   * flag is true when upgrade process is running
   * @returns {boolean}
   */
  upgradeInProgress: Em.computed.equal('upgradeState', 'IN_PROGRESS'),

  /**
   * Checks if update process is completed
   * @type {boolean}
   */
  upgradeCompleted: Em.computed.equal('upgradeState', 'COMPLETED'),

  /**
   * flag is true when upgrade process is waiting for user action
   * to proceed, retry, perform manual steps etc.
   * @returns {boolean}
   */
  upgradeHolding: function () {
    return this.get('upgradeState').contains("HOLDING") || this.get('upgradeAborted');
  }.property('upgradeState', 'upgradeAborted'),

  /**
   * flag is true when upgrade process is aborted
   * SHOULD behave similar to HOLDING_FAILED state
   * @returns {boolean}
   */
  upgradeAborted: function () {
    return this.get('upgradeState') === "ABORTED" && !App.router.get('mainAdminStackAndUpgradeController.isSuspended');
  }.property('upgradeState', 'router.mainAdminStackAndUpgradeController.isSuspended'),

  /**
   * flag is true when upgrade process is suspended
   * @returns {boolean}
   */
  upgradeSuspended: function () {
    return this.get('upgradeState') === "ABORTED" && App.router.get('mainAdminStackAndUpgradeController.isSuspended');
  }.property('upgradeState', 'router.mainAdminStackAndUpgradeController.isSuspended'),

  /**
   * RU is running
   * @type {boolean}
   */
  upgradeIsRunning: Em.computed.or('upgradeInProgress', 'upgradeHolding'),

  /**
   * flag is true when upgrade process is running or suspended
   * or wizard used by another user
   * @returns {boolean}
   */
  wizardIsNotFinished: function () {
    return this.get('upgradeIsRunning') || this.get('upgradeSuspended') || App.router.get('wizardWatcherController.isNonWizardUser');
  }.property('upgradeIsRunning', 'upgradeAborted', 'router.wizardWatcherController.isNonWizardUser', 'upgradeSuspended'),

  /**
   * @param {string} authRoles
   * @returns {boolean}
   */
  havePermissions: function havePermissions(authRoles) {
    var result = false;
    authRoles = $.map(authRoles.split(","), $.trim);

    // When Upgrade running(not suspended) only operations related to upgrade should be allowed
    if (!this.get('upgradeSuspended') && !authRoles.contains('CLUSTER.UPGRADE_DOWNGRADE_STACK') && !authRoles.contains('CLUSTER.MANAGE_USER_PERSISTED_DATA') && !App.get('supports.opsDuringRollingUpgrade') && !['NOT_REQUIRED', 'COMPLETED'].contains(this.get('upgradeState')) || !App.auth) {
      return false;
    }

    authRoles.forEach(function (auth) {
      result = result || App.auth.contains(auth);
    });

    return result;
  },
  /**
   * @param {string} authRoles
   * @returns {boolean}
   */
  isAuthorized: function isAuthorized(authRoles) {
    return this.havePermissions(authRoles) && !App.router.get('wizardWatcherController.isNonWizardUser');
  },

  isStackServicesLoaded: false,
  /**
   * return url prefix with number value of version of HDP stack
   */
  stackVersionURL: function () {
    return '/stacks/{0}/versions/{1}'.format(this.get('currentStackName') || 'HDP', this.get('currentStackVersionNumber'));
  }.property('currentStackName', 'currentStackVersionNumber'),

  falconServerURL: function () {
    var falconService = this.Service.find().findProperty('serviceName', 'FALCON');
    if (falconService) {
      return falconService.get('hostComponents').findProperty('componentName', 'FALCON_SERVER').get('hostName');
    }
    return '';
  }.property().volatile(),

  /* Determine if Application Timeline Service supports Kerberization.
   * Because this value is retrieved from the cardinality of the component, it is safe to keep in app.js
   * since its value will not change during the lifetime of the application.
   */
  doesATSSupportKerberos: function () {
    var YARNService = App.StackServiceComponent.find().filterProperty('serviceName', 'YARN');
    if (YARNService.length) {
      var ATS = App.StackServiceComponent.find().findProperty('componentName', 'APP_TIMELINE_SERVER');
      return !!ATS && !!ATS.get('minToInstall');
    }
    return false;
  }.property('router.clusterController.isLoaded'),

  clusterId: null,
  clusterName: null,
  clockDistance: null, // server clock - client clock
  currentStackVersion: '',
  currentStackName: function () {
    return Em.get((this.get('currentStackVersion') || this.get('defaultStackVersion')).match(/(.+)-\d.+/), '1');
  }.property('currentStackVersion', 'defaultStackVersion'),

  /**
   * true if cluster has only 1 host
   * for now is used to disable move/HA actions
   * @type {boolean}
   */
  isSingleNode: Em.computed.equal('allHostNames.length', 1),

  allHostNames: [],

  /**
   * This object is populated to keep track of uninstalled components to be included in the layout for recommendation/validation call
   * @type {object}
   * keys = componentName, hostName
   */
  componentToBeAdded: {},

  /**
   * This object is populated to keep track of installed components to be excluded in the layout for recommendation/validation call
   * @type {object}
   * keys = componentName, hostName
   */
  componentToBeDeleted: {},

  uiOnlyConfigDerivedFromTheme: [],

  currentStackVersionNumber: function () {
    var regExp = new RegExp(this.get('currentStackName') + '-');
    return (this.get('currentStackVersion') || this.get('defaultStackVersion')).replace(regExp, '');
  }.property('currentStackVersion', 'defaultStackVersion', 'currentStackName'),

  isHadoopWindowsStack: Em.computed.equal('currentStackName', 'HDPWIN'),

  /**
   * If NameNode High Availability is enabled
   * Based on <code>clusterStatus.isInstalled</code>, stack version, <code>SNameNode</code> availability
   *
   * @type {bool}
   */
  isHaEnabled: function () {
    return App.Service.find('HDFS').get('isLoaded') && !App.HostComponent.find().someProperty('componentName', 'SECONDARY_NAMENODE');
  }.property('router.clusterController.dataLoadList.services', 'router.clusterController.isServiceContentFullyLoaded'),

  hasNameNodeFederation: function () {
    return App.HDFSService.find('HDFS').get('masterComponentGroups.length') > 1;
  }.property('router.clusterController.isHostComponentMetricsLoaded', 'router.clusterController.isHDFSNameSpacesLoaded'),

  /**
   * If ResourceManager High Availability is enabled
   * Based on number of ResourceManager host components installed
   *
   * @type {bool}
   */
  isRMHaEnabled: function () {
    var result = false;
    var rmStackComponent = App.StackServiceComponent.find().findProperty('componentName', 'RESOURCEMANAGER');
    if (rmStackComponent && rmStackComponent.get('isMultipleAllowed')) {
      result = this.HostComponent.find().filterProperty('componentName', 'RESOURCEMANAGER').length > 1;
    }
    return result;
  }.property('router.clusterController.isLoaded', 'isStackServicesLoaded'),

  /**
   * If Ranger Admin High Availability is enabled
   * Based on number of Ranger Admin host components installed
   *
   * @type {bool}
   */
  isRAHaEnabled: function () {
    var result = false;
    var raStackComponent = App.StackServiceComponent.find().findProperty('componentName', 'RANGER_ADMIN');
    if (raStackComponent && raStackComponent.get('isMultipleAllowed')) {
      result = App.HostComponent.find().filterProperty('componentName', 'RANGER_ADMIN').length > 1;
    }
    return result;
  }.property('router.clusterController.isLoaded', 'isStackServicesLoaded'),

  /**
   * Object with utility functions for list of service names with similar behavior
   */
  services: Em.Object.create({
    all: function () {
      return App.StackService.find().mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    clientOnly: function () {
      return App.StackService.find().filterProperty('isClientOnlyService').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    hasClient: function () {
      return App.StackService.find().filterProperty('hasClient').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    hasMaster: function () {
      return App.StackService.find().filterProperty('hasMaster').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    hasSlave: function () {
      return App.StackService.find().filterProperty('hasSlave').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    noConfigTypes: function () {
      return App.StackService.find().filterProperty('isNoConfigTypes').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    servicesWithHeatmapTab: function () {
      return App.StackService.find().filterProperty('hasHeatmapSection').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    monitoring: function () {
      return App.StackService.find().filterProperty('isMonitoringService').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    hostMetrics: function () {
      return App.StackService.find().filterProperty('isHostMetricsService').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    serviceMetrics: function () {
      return App.StackService.find().filterProperty('isServiceMetricsService').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    supportsServiceCheck: function () {
      return App.StackService.find().filterProperty('serviceCheckSupported').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded'),

    supportsDeleteViaUI: function () {
      return App.StackService.find().filterProperty('supportDeleteViaUi').mapProperty('serviceName');
    }.property('App.router.clusterController.isLoaded')
  }),

  /**
   * List of components with allowed action for them
   * @type {Em.Object}
   */
  components: Em.Object.create({
    isMasterAddableOnlyOnHA: function () {
      return App.StackServiceComponent.find().filterProperty('isMasterAddableOnlyOnHA').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    allComponents: function () {
      return App.StackServiceComponent.find().mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    reassignable: function () {
      return App.StackServiceComponent.find().filterProperty('isReassignable').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    restartable: function () {
      return App.StackServiceComponent.find().filterProperty('isRestartable').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    deletable: function () {
      return App.StackServiceComponent.find().filterProperty('isDeletable').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    rollinRestartAllowed: function () {
      return App.StackServiceComponent.find().filterProperty('isRollinRestartAllowed').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    decommissionAllowed: function () {
      return App.StackServiceComponent.find().filterProperty('isDecommissionAllowed').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    refreshConfigsAllowed: function () {
      return App.StackServiceComponent.find().filterProperty('isRefreshConfigsAllowed').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    addableToHost: function () {
      return App.StackServiceComponent.find().filterProperty('isAddableToHost').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    addableMasterInstallerWizard: function () {
      return App.StackServiceComponent.find().filterProperty('isMasterAddableInstallerWizard').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    multipleMasters: function () {
      return App.StackServiceComponent.find().filterProperty('isMasterWithMultipleInstances').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    slaves: function () {
      return App.StackServiceComponent.find().filterProperty('isSlave').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    masters: function () {
      return App.StackServiceComponent.find().filterProperty('isMaster').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    clients: function () {
      return App.StackServiceComponent.find().filterProperty('isClient').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded'),

    nonHDP: function () {
      return App.StackServiceComponent.find().filterProperty('isNonHDPComponent').mapProperty('componentName');
    }.property('App.router.clusterController.isLoaded')
  })
});

});

require.register("config", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.version = '2.7.7.0'; // filled out by set-ambari-version.sh script
App.testMode = location.port == '3333'; // test mode is automatically enabled if running on brunch server
App.testModeDelayForActions = 10000;
App.skipBootstrap = false;
App.alwaysGoToInstaller = false;
App.testEnableSecurity = true; // By default enable security is tested; turning it false tests disable security
App.testNameNodeHA = true;
App.appURLRoot = '{proxy_root}/'.replace(/\{.+\}/g, ''); // determines application root path name, not related to hash route
App.apiPrefix = '/api/v1';
App.defaultStackVersion = 'HDP-2.3';
App.defaultWindowsStackVersion = 'HDPWIN-2.1';

App.defaultJavaHome = '/usr/jdk/jdk1.6.0_31';
App.timeout = 300000; // default AJAX timeout
App.maxRetries = 3; // max number of retries for certain AJAX calls
App.sessionKeepAliveInterval = 60000;
App.bgOperationsUpdateInterval = 6000;
App.componentsUpdateInterval = 6000;
App.contentUpdateInterval = 15000;
App.hostStatusCountersUpdateInterval = 10000;
App.alertDefinitionsUpdateInterval = 10000;
App.alertInstancesUpdateInterval = 10000;
App.alertGroupsUpdateInterval = 10000;
App.clusterEnvUpdateInterval = 10000;
App.pageReloadTime = 3600000;
App.nnCheckpointAgeAlertThreshold = 12; // in hours
App.minDiskSpace = 2.0; // minimum disk space required for '/' for each host before install, unit GB
App.minDiskSpaceUsrLib = 1.0; // minimum disk space for '/usr/lib' for each host before install, unit GB
App.healthIconClassGreen = 'glyphicon glyphicon-ok-sign'; // bootstrap icon class for healthy/started service/host/host-component
App.healthIconClassRed = 'glyphicon glyphicon-warning-sign'; // bootstrap icon class for master down/stopped service/host/host-component
App.healthIconClassOrange = 'glyphicon glyphicon-minus-sign'; // bootstrap icon class for slave down/decommissioned host/host-component
App.healthIconClassYellow = 'glyphicon glyphicon-question-sign'; // bootstrap icon class for heartbeat lost service/host/host-component
App.isManagedMySQLForHiveEnabled = false;
App.isStormMetricsSupported = true;
App.healthStatusRed = '#EF6162';
App.healthStatusGreen = '#1EB475';
App.healthStatusOrange = '#E98A40';
App.widgetContentColor = '#666666';
App.inactivityRemainTime = 60; // in seconds
App.enableLogger = true;
App.stackVersionsAvailable = true;
App.upgradeHistoryAvailable = false;
App.enableDigitalClock = false;

// experimental features are automatically enabled if running on brunch server
App.enableExperimental = false;

App.supports = {
  preUpgradeCheck: true,
  displayOlderVersions: false,
  autoRollbackHA: false,
  alwaysEnableManagedMySQLForHive: false,
  preKerberizeCheck: false,
  customizeAgentUserAccount: false,
  installGanglia: false,
  opsDuringRollingUpgrade: false,
  customizedWidgetLayout: false,
  showPageLoadTime: false,
  skipComponentStartAfterInstall: false,
  preInstallChecks: false,
  serviceAutoStart: true,
  logSearch: true,
  redhatSatellite: false,
  addingNewRepository: false,
  kerberosStackAdvisor: true,
  logCountVizualization: false,
  createAlerts: false,
  enabledWizardForHostOrderedUpgrade: true,
  manageJournalNode: true,
  enableToggleKerberos: true,
  enableAddDeleteServices: true,
  regenerateKeytabsOnSingleHost: false,
  disableCredentialsAutocompleteForRepoUrls: false
};

if (App.enableExperimental) {
  for (var support in App.supports) {
    App.supports[support] = true;
  }
}

// this is to make sure that IE does not cache data when making AJAX calls to the server
if (!$.mocho) {
  $.ajaxSetup({
    cache: false,
    headers: { "X-Requested-By": "X-Requested-By" }
  });
}

/**
 * Test Mode values
 */
App.test_hostname = 'hostname';

});

require.register("controllers", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

// load all controllers here

require('controllers/application');
require('controllers/login_controller');
require('controllers/wizard');
require('controllers/installer');
require('controllers/experimental');
require('controllers/global/background_operations_controller');
require('controllers/global/wizard_watcher_controller');
require('controllers/global/user_settings_controller');
require('controllers/global/errors_handler_controller');
require('controllers/main');
require('controllers/main/dashboard');
require('controllers/main/dashboard/config_history_controller');
require('controllers/main/admin');
require('controllers/main/admin/service_auto_start');
require('controllers/main/admin/highAvailability_controller');
require('controllers/main/admin/highAvailability/nameNode/wizard_controller');
require('controllers/main/admin/highAvailability/progress_controller');
require('controllers/main/admin/highAvailability/progress_popup_controller');
require('controllers/main/admin/highAvailability/nameNode/rollback_controller');
require('controllers/main/admin/highAvailability/nameNode/step1_controller');
require('controllers/main/admin/highAvailability/nameNode/step2_controller');
require('controllers/main/admin/highAvailability/nameNode/step3_controller');
require('controllers/main/admin/highAvailability/nameNode/step4_controller');
require('controllers/main/admin/highAvailability/nameNode/step5_controller');
require('controllers/main/admin/highAvailability/nameNode/step6_controller');
require('controllers/main/admin/highAvailability/nameNode/step7_controller');
require('controllers/main/admin/highAvailability/nameNode/step8_controller');
require('controllers/main/admin/highAvailability/nameNode/step9_controller');
require('controllers/main/admin/highAvailability/nameNode/rollbackHA/step1_controller');
require('controllers/main/admin/highAvailability/nameNode/rollbackHA/step2_controller');
require('controllers/main/admin/highAvailability/nameNode/rollbackHA/step3_controller');
require('controllers/main/admin/highAvailability/nameNode/rollbackHA/rollback_wizard_controller');
require('controllers/main/admin/highAvailability/resourceManager/wizard_controller');
require('controllers/main/admin/highAvailability/resourceManager/step1_controller');
require('controllers/main/admin/highAvailability/resourceManager/step2_controller');
require('controllers/main/admin/highAvailability/resourceManager/step3_controller');
require('controllers/main/admin/highAvailability/resourceManager/step4_controller');
require('controllers/main/admin/federation/wizard_controller');
require('controllers/main/admin/federation/step1_controller');
require('controllers/main/admin/federation/step2_controller');
require('controllers/main/admin/federation/step3_controller');
require('controllers/main/admin/federation/step4_controller');
require('controllers/main/admin/highAvailability/hawq/addStandby/wizard_controller');
require('controllers/main/admin/highAvailability/hawq/addStandby/step1_controller');
require('controllers/main/admin/highAvailability/hawq/addStandby/step2_controller');
require('controllers/main/admin/highAvailability/hawq/addStandby/step3_controller');
require('controllers/main/admin/highAvailability/hawq/addStandby/step4_controller');
require('controllers/main/admin/highAvailability/hawq/removeStandby/wizard_controller');
require('controllers/main/admin/highAvailability/hawq/removeStandby/step1_controller');
require('controllers/main/admin/highAvailability/hawq/removeStandby/step2_controller');
require('controllers/main/admin/highAvailability/hawq/removeStandby/step3_controller');
require('controllers/main/admin/highAvailability/hawq/activateStandby/wizard_controller');
require('controllers/main/admin/highAvailability/hawq/activateStandby/step1_controller');
require('controllers/main/admin/highAvailability/hawq/activateStandby/step2_controller');
require('controllers/main/admin/highAvailability/hawq/activateStandby/step3_controller');
require('controllers/main/admin/highAvailability/rangerAdmin/wizard_controller');
require('controllers/main/admin/highAvailability/rangerAdmin/step1_controller');
require('controllers/main/admin/highAvailability/rangerAdmin/step2_controller');
require('controllers/main/admin/highAvailability/rangerAdmin/step3_controller');
require('controllers/main/admin/highAvailability/rangerAdmin/step4_controller');
require('controllers/main/admin/highAvailability/journalNode/wizard_controller');
require('controllers/main/admin/highAvailability/journalNode/progress_controller');
require('controllers/main/admin/highAvailability/journalNode/step1_controller');
require('controllers/main/admin/highAvailability/journalNode/step2_controller');
require('controllers/main/admin/highAvailability/journalNode/step3_controller');
require('controllers/main/admin/highAvailability/journalNode/step4_controller');
require('controllers/main/admin/highAvailability/journalNode/step5_controller');
require('controllers/main/admin/highAvailability/journalNode/step6_controller');
require('controllers/main/admin/highAvailability/journalNode/step7_controller');
require('controllers/main/admin/stack_and_upgrade_controller');
require('controllers/main/admin/stack_upgrade_history_controller');
require('controllers/main/admin/serviceAccounts_controller');
require('utils/polling');
require('controllers/main/admin/kerberos');
require('controllers/main/admin/kerberos/wizard_controller');
require('controllers/main/admin/kerberos/disable_controller');
require('controllers/main/admin/kerberos/progress_controller');
require('controllers/main/admin/kerberos/step1_controller');
require('controllers/main/admin/kerberos/step2_controller');
require('controllers/main/admin/kerberos/step3_controller');
require('controllers/main/admin/kerberos/step4_controller');
require('controllers/main/admin/kerberos/step5_controller');
require('controllers/main/admin/kerberos/step6_controller');
require('controllers/main/admin/kerberos/step7_controller');
require('controllers/main/admin/kerberos/step8_controller');
require('controllers/main/alert_definitions_controller');
require('controllers/main/alerts/alert_definitions_actions_controller');
require('controllers/main/alerts/add_alert_definition/add_alert_definition_controller');
require('controllers/main/alerts/add_alert_definition/step1_controller');
require('controllers/main/alerts/add_alert_definition/step2_controller');
require('controllers/main/alerts/add_alert_definition/step3_controller');
require('controllers/main/alerts/definition_details_controller');
require('controllers/main/alerts/definition_configs_controller');
require('controllers/main/alerts/alert_instances_controller');
require('controllers/main/alerts/manage_alert_groups_controller');
require('controllers/main/alerts/manage_alert_notifications_controller');
require('controllers/main/service');
require('controllers/main/service/item');
require('controllers/main/service/info/summary');
require('controllers/main/service/info/configs');
require('controllers/main/service/info/audit');
require('controllers/main/service/add_controller');
require('controllers/main/service/reassign_controller');
require('controllers/main/service/reassign/step1_controller');
require('controllers/main/service/reassign/step2_controller');
require('controllers/main/service/reassign/step3_controller');
require('controllers/main/service/reassign/step4_controller');
require('controllers/main/service/reassign/step5_controller');
require('controllers/main/service/reassign/step6_controller');
require('controllers/main/service/manage_config_groups_controller');
require('controllers/main/service/widgets/create/wizard_controller');
require('controllers/main/service/widgets/create/step1_controller');
require('controllers/main/service/widgets/create/step2_controller');
require('controllers/main/service/widgets/create/step3_controller');
require('controllers/main/service/widgets/edit_controller');
require('controllers/main/host');
require('controllers/main/host/bulk_operations_controller');
require('controllers/main/host/details');
require('controllers/main/host/configs_service');
require('controllers/main/host/add_controller');
require('controllers/main/host/combo_search_box');
require('controllers/main/host/addHost/step4_controller');
require('controllers/main/host/host_alerts_controller');
require('controllers/main/charts');
require('controllers/main/charts/heatmap_metrics/heatmap_metric');
require('controllers/main/charts/heatmap');
require('controllers/main/service/info/heatmap');
require('controllers/main/service/info/metric');
require('controllers/main/views_controller');
require('controllers/main/views/details_controller');
require('controllers/wizard/step0_controller');
require('controllers/wizard/step1_controller');
require('controllers/wizard/step2_controller');
require('controllers/wizard/step3_controller');
require('controllers/wizard/step4_controller');
require('controllers/wizard/step5_controller');
require('controllers/wizard/step6_controller');
require('controllers/wizard/step7_controller');
require('controllers/wizard/step7/assign_master_controller');
require('controllers/wizard/step7/pre_install_checks_controller');
require('controllers/wizard/step8_controller');
require('controllers/wizard/step9_controller');
require('controllers/wizard/step10_controller');
require('controllers/global/cluster_controller');
require('controllers/global/update_controller');
require('controllers/global/configuration_controller');
require('controllers/main/service/reassign/step7_controller');

});

require.register("controllers/application", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ApplicationController = Em.Controller.extend(App.Persist, {

  name: 'applicationController',

  isPollerRunning: false,

  clusterName: Em.computed.alias('App.router.clusterController.clusterName'),

  /**
   * set ambari server version from installerController or mainController, making sure version shown up all the time
   */
  ambariVersion: function () {
    return App.router.get('installerController.ambariServerVersion') || App.router.get('mainController.ambariServerVersion') || Em.I18n.t('common.notAvailable');
  }.property('App.router.installerController.ambariServerVersion', 'App.router.mainController.ambariServerVersion'),

  clusterDisplayName: Em.computed.truncate('clusterName', 13, 10),

  isClusterDataLoaded: Em.computed.and('App.router.clusterController.isLoaded', 'App.router.loggedIn'),

  isExistingClusterDataLoaded: Em.computed.and('App.router.clusterInstallCompleted', 'isClusterDataLoaded'),

  enableLinks: Em.computed.and('isExistingClusterDataLoaded', '!App.isOnlyViewUser'),

  /**
   * Determines if "Exit" menu-item should be shown
   * It should if cluster isn't installed
   * If cluster is installer, <code>isClusterDataLoaded</code> is checked
   * @type {boolean}
   */
  showExitLink: function () {
    if (App.router.get('clusterInstallCompleted')) {
      return this.get('isClusterDataLoaded');
    }
    return true;
  }.property('App.router.clusterInstallCompleted', 'isClusterDataLoaded'),

  /**
   * Determines if "Manage Ambari" menu-item should be shown
   *
   * @type {boolean}
   */
  showManageAmbari: function () {
    if (App.router.get('clusterInstallCompleted')) {
      return this.get('isClusterDataLoaded');
    }
    return App.get('isPermissionDataLoaded');
  }.property('App.router.clusterInstallCompleted', 'isClusterDataLoaded', 'App.isPermissionDataLoaded'),

  /**
   * Determines if upgrade label should be shown
   *
   * @type {boolean}
   */
  showUpgradeLabel: Em.computed.or('App.upgradeInProgress', 'App.upgradeHolding', 'App.upgradeSuspended'),

  /**
   * @return {{msg: string, cls: string, icon: string}}
   */
  upgradeMap: function () {
    var upgradeInProgress = App.get('upgradeInProgress');
    var upgradeHolding = App.get('upgradeHolding');
    var upgradeSuspended = App.get('upgradeSuspended');
    var isDowngrade = App.router.get('mainAdminStackAndUpgradeController.isDowngrade');
    var typeSuffix = isDowngrade ? 'downgrade' : 'upgrade';
    var hasUpgradePrivilege = App.isAuthorized('CLUSTER.UPGRADE_DOWNGRADE_STACK');
    var wizardWatcherController = App.router.get('wizardWatcherController');
    var isNotWizardUser;
    var wizardUserName;
    if (upgradeInProgress) {
      isNotWizardUser = wizardWatcherController.get('isNonWizardUser');
      wizardUserName = wizardWatcherController.get('wizardUser');
      return {
        cls: hasUpgradePrivilege ? 'upgrade-in-progress' : 'upgrade-in-progress not-allowed-cursor',
        icon: 'glyphicon-cog',
        msg: isNotWizardUser ? Em.I18n.t('admin.stackVersions.version.' + typeSuffix + '.running.nonWizard').format(wizardUserName) : Em.I18n.t('admin.stackVersions.version.' + typeSuffix + '.running')
      };
    }
    if (upgradeHolding) {
      return {
        cls: hasUpgradePrivilege ? 'upgrade-holding' : 'upgrade-holding not-allowed-cursor',
        icon: 'glyphicon-pause',
        msg: Em.I18n.t('admin.stackVersions.version.' + typeSuffix + '.pause')
      };
    }
    if (upgradeSuspended) {
      return {
        cls: hasUpgradePrivilege ? 'upgrade-aborted' : 'upgrade-aborted not-allowed-cursor',
        icon: 'glyphicon-pause',
        msg: Em.I18n.t('admin.stackVersions.version.' + typeSuffix + '.suspended')
      };
    }
    return {};
  }.property('App.upgradeInProgress', 'App.upgradeHolding', 'App.upgradeSuspended', 'App.router.mainAdminStackAndUpgradeController.isDowngrade'),

  startKeepAlivePoller: function startKeepAlivePoller() {
    if (!this.get('isPollerRunning')) {
      this.set('isPollerRunning', true);
      App.updater.run(this, 'getStack', 'isPollerRunning', App.sessionKeepAliveInterval);
    }
  },

  getStack: function getStack(callback) {
    App.ajax.send({
      name: 'router.login.clusters',
      sender: this,
      callback: callback
    });
  },

  goToAdminView: function goToAdminView() {
    App.router.route("adminView");
  },

  goToDashboard: function goToDashboard() {
    if (this.get('enableLinks')) {
      App.router.route("main/dashboard");
    }
  },

  showAboutPopup: function showAboutPopup() {
    App.ModalPopup.show({
      header: Em.I18n.t('common.aboutAmbari'),
      secondary: false,
      bodyClass: Em.View.extend({
        templateName: require('templates/common/about'),
        ambariVersion: this.get('ambariVersion')
      })
    });
  }

});

});

require.register("controllers/experimental", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

var App = require('app');

App.ExperimentalController = Em.Controller.extend(App.Persist, {
  name: 'experimentalController',
  supports: function () {
    return Em.keys(App.get('supports')).map(function (sup) {
      return Ember.Object.create({
        name: sup,
        selected: App.get('supports')[sup]
      });
    });
  }.property('App.supports'),

  loadSupports: function loadSupports() {
    return this.getUserPref('user-pref-' + App.router.get('loginName') + '-supports');
  },

  getUserPrefSuccessCallback: function getUserPrefSuccessCallback(response) {
    if (response) {
      App.set('supports', $.extend({}, App.get('supports'), response));
    }
  },

  doSave: function doSave() {
    var supports = this.get('supports');
    supports.forEach(function (s) {
      var propName = 'App.supports.' + s.get('name');
      var propValue = s.get('selected');
      Ember.set(propName, propValue);
    });
    this.postUserPref('user-pref-' + App.router.get('loginName') + '-supports', App.get('supports')).complete(function () {
      App.router.transitionTo('root.index');
    });
  },

  doCancel: function doCancel() {
    App.router.transitionTo('root.index');
  },

  doResetUIStates: function doResetUIStates() {
    var self = this;
    return App.ModalPopup.show({
      header: Em.I18n.t('reset.ui.states'),
      bodyClass: Ember.View.extend({
        template: Ember.Handlebars.compile(Em.I18n.t('reset.ui.states.body'))
      }),
      primary: Em.I18n.t('yes'),
      context: self,
      onPrimary: function onPrimary() {
        var router = App.router;
        App.db.cleanUp();
        router.clearAllSteps();
        App.cache.clear();
        App.clusterStatus.setClusterStatus({});
        this.context.postUserPref('wizard-data', {});
        this.hide();
        router.transitionTo('root.index');
      }
    });
  }
});

});

require.register("controllers/global/background_operations_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.BackgroundOperationsController = Em.Controller.extend({
  name: 'backgroundOperationsController',

  runningOperationsCount: function () {
    return this.get('services').filterProperty('isRunning').length;
  }.property('services.@each.isRunning'),

  /**
   * List of requests
   */
  services: Em.A([]),
  serviceTimestamp: null,

  /**
   * Number of operation to load
   */
  operationsCount: 10,

  /**
   * Possible levels:
   * REQUESTS_LIST
   * HOSTS_LIST
   * TASKS_LIST
   * TASK_DETAILS
   */
  levelInfo: Em.Object.create({
    name: 'REQUESTS_LIST',
    requestId: null,
    taskId: null
  }),

  subscribeToUpdates: function subscribeToUpdates() {
    var _this = this;

    this.requestMostRecent(function () {
      App.StompClient.subscribe('/events/requests', _this.updateRequests.bind(_this));
    });
  },

  updateRequests: function updateRequests(event) {
    if (this.isUpgradeRequest({ Requests: { request_context: event.requestContext } })) {
      return;
    }
    var request = this.get('services').findProperty('id', event.requestId);
    var context = this.parseRequestContext(event.requestContext);
    var visibleOperationsCount = this.get('operationsCount');
    var map = this.generateTasksMapOfRequest(event, request);
    var updatedState = {
      progress: Math.floor(event.progressPercent),
      status: event.requestStatus,
      userName: event.userName,
      isRunning: this.isRunning(event.requestStatus),
      startTime: App.dateTimeWithTimeZone(event.startTime),
      endTime: event.endTime > 0 ? App.dateTimeWithTimeZone(event.endTime) : event.endTime,
      previousTaskStatusMap: map.currentTaskStatusMap,
      hostsMap: map.hostsMap
    };

    if (request) {
      request.setProperties(updatedState);
    } else {
      this.get('services').unshift(Em.Object.create(updatedState, {
        id: event.requestId,
        name: context.requestContext,
        displayName: context.requestContext,
        tasks: event.Tasks
      }));
      if (this.get('services').length >= visibleOperationsCount) {
        this.set('isShowMoreAvailable', true);
        this.get('services').pop();
      }
    }
    this.set('serviceTimestamp', App.dateTime());
    this.propertyDidChange('services');
  },

  /**
   *
   * @param {object} event
   * @param {Em.Object} request
   * @returns {{}}
   */
  generateTasksMapOfRequest: function generateTasksMapOfRequest(event, request) {
    var _this2 = this;

    var hostsMap = request ? request.get('hostsMap') : {};
    var previousTaskStatusMap = request ? request.get('previousTaskStatusMap') : {};
    var currentTaskStatusMap = {};
    event.Tasks.forEach(function (task) {
      var host = hostsMap[task.hostName];
      if (host) {
        var existedTask = host.logTasks.findProperty('Tasks.id', task.id);
        if (existedTask) {
          existedTask.Tasks.status = task.status;
        } else {
          host.logTasks.push(_this2.convertTaskFromEventToApi(task));
        }
        host.isModified = host.isModified ? true : previousTaskStatusMap[task.id] !== task.status;
      } else {
        hostsMap[task.hostName] = {
          name: task.hostName,
          publicName: task.hostName,
          logTasks: [_this2.convertTaskFromEventToApi(task)],
          isModified: previousTaskStatusMap[task.id] !== task.status
        };
      }
      currentTaskStatusMap[task.id] = task.status;
    }, this);
    return {
      currentTaskStatusMap: currentTaskStatusMap,
      hostsMap: hostsMap
    };
  },

  convertTaskFromEventToApi: function convertTaskFromEventToApi(task) {
    return {
      Tasks: {
        status: task.status,
        host_name: task.hostName,
        id: task.id,
        request_id: task.requestId
      }
    };
  },

  handleTaskUpdates: function () {
    var _this3 = this;

    var levelInfo = this.get('levelInfo');
    if (!levelInfo.get('requestId') || !levelInfo.get('taskId')) {
      return;
    }
    var request = this.get('services').findProperty('id', levelInfo.get('requestId'));
    var taskStatus = request.get('previousTaskStatusMap')[levelInfo.get('taskId')];
    if (levelInfo.get('name') === 'TASK_DETAILS' && !this.isFinished(taskStatus)) {
      App.StompClient.subscribe('/events/tasks/' + levelInfo.get('taskId'), function (updatedTask) {
        _this3.updateTask(updatedTask);
        if (_this3.isFinished(updatedTask.status)) {
          App.StompClient.unsubscribe('/events/tasks/' + updatedTask.id);
        }
      });
    }
  }.observes('levelInfo.name'),

  updateTask: function updateTask(updatedTask) {
    var request = this.get('services').findProperty('id', updatedTask.requestId);
    var host = request.get('hostsMap')[updatedTask.hostName];
    var task = host.logTasks.findProperty('Tasks.id', updatedTask.id);
    task.Tasks.status = updatedTask.status;
    task.Tasks.stdout = updatedTask.stdout;
    task.Tasks.stderr = updatedTask.stderr;
    task.Tasks.structured_out = updatedTask.structured_out;
    task.Tasks.output_log = updatedTask.outLog;
    task.Tasks.error_log = updatedTask.errorLog;
    this.set('serviceTimestamp', App.dateTime());
  },

  /**
   * Get requests data from server
   * @param callback
   */
  requestMostRecent: function requestMostRecent(callback) {
    var queryParams = this.getQueryParams();
    App.ajax.send({
      'name': queryParams.name,
      'sender': this,
      'success': queryParams.successCallback,
      'callback': callback,
      'data': queryParams.data
    });
    return !this.isInitLoading();
  },

  /**
   * indicate whether data for current level has already been loaded or not
   * @return {Boolean}
   */
  isInitLoading: function isInitLoading() {
    var levelInfo = this.get('levelInfo');
    var request = this.get('services').findProperty('id', levelInfo.get('requestId'));

    if (levelInfo.get('name') === 'HOSTS_LIST') {
      return Boolean(request && !request.get('hostsLevelLoaded'));
    }
    return false;
  },
  /**
   * construct params of ajax query regarding displayed level
   */
  getQueryParams: function getQueryParams() {
    var levelInfo = this.get('levelInfo');
    var count = this.get('operationsCount');
    var result = {
      name: 'background_operations.get_most_recent',
      successCallback: 'callBackForMostRecent',
      data: {
        'operationsCount': count
      }
    };
    if (levelInfo.get('name') === 'TASK_DETAILS') {
      result.name = 'background_operations.get_by_task';
      result.successCallback = 'callBackFilteredByTask';
      result.data = {
        'taskId': levelInfo.get('taskId'),
        'requestId': levelInfo.get('requestId')
      };
    } else if (levelInfo.get('name') === 'TASKS_LIST' || levelInfo.get('name') === 'HOSTS_LIST') {
      result.name = 'background_operations.get_by_request';
      result.successCallback = 'callBackFilteredByRequest';
      result.data = {
        'requestId': levelInfo.get('requestId')
      };
    }
    return result;
  },

  /**
   * Push hosts and their tasks to request
   * @param data
   */
  callBackFilteredByRequest: function callBackFilteredByRequest(data) {
    var requestId = data.Requests.id;
    var requestInputs = data.Requests.inputs;
    var request = this.get('services').findProperty('id', requestId);
    var hostsMap = {};
    var previousTaskStatusMap = request.get('previousTaskStatusMap');
    var currentTaskStatusMap = {};
    data.tasks.forEach(function (task) {
      var host = hostsMap[task.Tasks.host_name];
      task.Tasks.request_id = requestId;
      task.Tasks.request_inputs = requestInputs;
      if (host) {
        host.logTasks.push(task);
        host.isModified = true;
      } else {
        hostsMap[task.Tasks.host_name] = {
          name: task.Tasks.host_name,
          publicName: task.Tasks.host_name,
          logTasks: [task],
          isModified: true
        };
      }
      currentTaskStatusMap[task.Tasks.id] = task.Tasks.status;
    }, this);
    request.set('previousTaskStatusMap', currentTaskStatusMap);
    request.set('hostsMap', hostsMap);
    request.set('hostsLevelLoaded', true);
    this.set('serviceTimestamp', App.dateTime());
  },
  /**
   * Update task, with uploading two additional properties: stdout and stderr
   * @param data
   * @param ajaxQuery
   * @param params
   */
  callBackFilteredByTask: function callBackFilteredByTask(data, ajaxQuery, params) {
    var request = this.get('services').findProperty('id', data.Tasks.request_id);
    var host = request.get('hostsMap')[data.Tasks.host_name];
    var task = host.logTasks.findProperty('Tasks.id', data.Tasks.id);
    task.Tasks.status = data.Tasks.status;
    task.Tasks.stdout = data.Tasks.stdout;
    task.Tasks.stderr = data.Tasks.stderr;

    // Put some command information to task object
    task.Tasks.command = data.Tasks.command;
    task.Tasks.custom_command_name = data.Tasks.custom_command_name;
    task.Tasks.structured_out = data.Tasks.structured_out;

    task.Tasks.output_log = data.Tasks.output_log;
    task.Tasks.error_log = data.Tasks.error_log;
    this.set('serviceTimestamp', App.dateTime());
  },

  /**
   * returns true if it's upgrade equest
   * use this flag to exclude upgrade requests from bgo
   * @param {object} request
   * @returns {boolean}
   */
  isUpgradeRequest: function isUpgradeRequest(request) {
    var context = Em.get(request, 'Requests.request_context');
    return context ? /(upgrading|downgrading)/.test(context.toLowerCase()) : false;
  },
  /**
   * Prepare, received from server, requests for host component popup
   * @param data
   */
  callBackForMostRecent: function callBackForMostRecent(data) {
    var currentRequestIds = [];
    var countIssued = this.get('operationsCount');
    var countGot = data.itemTotal;

    data.items.forEach(function (request) {
      if (this.isUpgradeRequest(request)) {
        return;
      }
      var rq = this.get("services").findProperty('id', request.Requests.id);
      var isRunning = this.isRunning(request.Requests.request_status);
      var requestParams = this.parseRequestContext(request.Requests.request_context);
      var requestState = {
        progress: Math.floor(request.Requests.progress_percent),
        status: request.Requests.request_status,
        userName: request.Requests.user_name || Em.I18n.t('hostPopup.default.userName'),
        isRunning: isRunning,
        startTime: App.dateTimeWithTimeZone(request.Requests.start_time),
        endTime: request.Requests.end_time > 0 ? App.dateTimeWithTimeZone(request.Requests.end_time) : request.Requests.end_time
      };
      this.assignScheduleId(request, requestParams);
      currentRequestIds.push(request.Requests.id);

      if (rq) {
        rq.setProperties(requestState);
      } else {
        rq = Em.Object.create(requestState, {
          id: request.Requests.id,
          name: requestParams.requestContext,
          displayName: requestParams.requestContext,
          hostsMap: {},
          tasks: [],
          dependentService: requestParams.dependentService,
          sourceRequestScheduleId: request.Requests.request_schedule && request.Requests.request_schedule.schedule_id,
          previousTaskStatusMap: {},
          contextCommand: requestParams.contextCommand
        });
        this.get("services").unshift(rq);
        //To sort DESC by request id
        this.set("services", this.get("services").sortProperty('id').reverse());
      }
    }, this);
    this.removeOldRequests(currentRequestIds);
    this.set('isShowMoreAvailable', countGot >= countIssued);
    this.set('serviceTimestamp', App.dateTimeWithTimeZone());
  },

  isShowMoreAvailable: null,

  /**
   * remove old requests
   * as API returns 10, or  20 , or 30 ...etc latest request, the requests that absent in response should be removed
   * @param currentRequestIds
   */
  removeOldRequests: function removeOldRequests(currentRequestIds) {
    var services = this.get('services');

    for (var i = 0, l = services.length; i < l; i++) {
      if (!currentRequestIds.contains(services[i].id)) {
        services.splice(i, 1);
        i--;
        l--;
      }
    }
  },

  /**
   * identify whether request or task is running by status
   * @param status
   * @return {Boolean}
   */
  isRunning: function isRunning(status) {
    return ['IN_PROGRESS', 'QUEUED', 'PENDING'].contains(status);
  },

  /**
   * identify whether request or task is finished by status
   * @param status
   * @return {Boolean}
   */
  isFinished: function isFinished(status) {
    return ['FAILED', 'ABORTED', 'COMPLETED'].contains(status);
  },

  /**
   * identify whether there is only one host in request
   * @param inputs
   * @return {Boolean}
   */
  isOneHost: function isOneHost(inputs) {
    if (!inputs) {
      return false;
    }
    inputs = JSON.parse(inputs);
    if (inputs && inputs.included_hosts) {
      return inputs.included_hosts.split(',').length < 2;
    }
    return false;
  },
  /**
   * assign schedule_id of request to null if it's Recommission operation
   * @param request
   * @param requestParams
   */
  assignScheduleId: function assignScheduleId(request, requestParams) {
    var oneHost = this.isOneHost(request.Requests.inputs);
    if (request.Requests.request_schedule && oneHost && /Recommission/.test(requestParams.requestContext)) {
      request.Requests.request_schedule.schedule_id = null;
    }
  },

  /**
   * parse request context and if keyword "_PARSE_" is present then format it
   * @param {string} requestContext
   * @return {Object}
   */
  parseRequestContext: function parseRequestContext(requestContext) {
    var context = {};
    if (requestContext) {
      if (requestContext.indexOf(App.BackgroundOperationsController.CommandContexts.PREFIX) !== -1) {
        context = this.getRequestContextWithPrefix(requestContext);
      } else {
        context.requestContext = requestContext;
      }
    } else {
      context.requestContext = Em.I18n.t('requestInfo.unspecified');
    }
    return context;
  },

  /**
   *
   * @param {string} requestContext
   * @returns {{requestContext: *, dependentService: *, contextCommand: *}}
   */
  getRequestContextWithPrefix: function getRequestContextWithPrefix(requestContext) {
    var contextSplits = requestContext.split('.'),
        parsedRequestContext,
        contextCommand = contextSplits[1],
        service = contextSplits[2];

    switch (contextCommand) {
      case "STOP":
      case "START":
        if (service === 'ALL_SERVICES') {
          parsedRequestContext = Em.I18n.t("requestInfo." + contextCommand.toLowerCase()).format(Em.I18n.t('common.allServices'));
        } else {
          parsedRequestContext = Em.I18n.t("requestInfo." + contextCommand.toLowerCase()).format(App.format.role(service, true));
        }
        break;
      case "ROLLING-RESTART":
        parsedRequestContext = Em.I18n.t("rollingrestart.rest.context").format(App.format.role(service, true), contextSplits[3], contextSplits[4]);
        break;
    }
    return {
      requestContext: parsedRequestContext,
      dependentService: service,
      contextCommand: contextCommand
    };
  },

  popupView: null,

  /**
   * Onclick handler for background operations number located right to logo
   */
  showPopup: function showPopup() {
    // load the checkbox on footer first, then show popup.
    var self = this;
    App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
      App.updater.immediateRun('requestMostRecent');

      App.HostPopup.set("breadcrumbs", [App.HostPopup.get("rootBreadcrumb")]);

      if (self.get('popupView') && App.HostPopup.get('isBackgroundOperations')) {
        self.set('popupView.isNotShowBgChecked', !initValue);
        self.set('popupView.isOpen', true);
        var el = $(self.get('popupView.element'));
        el.appendTo('#wrapper');
        el.find('.modal').show();
      } else {
        self.set('popupView', App.HostPopup.initPopup("", self, true));
        self.set('popupView.isNotShowBgChecked', !initValue);
      }
    });
  },

  /**
   * Called on logout
   */
  clear: function clear() {
    // set operations count to default value
    this.set('operationsCount', 10);
  }

});

/**
 * Each background operation has a context in which it operates.
 * Generally these contexts are fixed messages. However, we might
 * want to associate semantics to this context - like showing, disabling
 * buttons when certain operations are in progress.
 *
 * To make this possible we have command contexts where the context
 * is not a human readable string, but a pattern indicating the command
 * it is running. When UI shows these, they are translated into human
 * readable strings.
 *
 * General pattern of context names is "_PARSE_.{COMMAND}.{ID}[.{Additional-Data}...]"
 */
App.BackgroundOperationsController.CommandContexts = {
  PREFIX: "_PARSE_",
  /**
   * Stops all services
   */
  STOP_ALL_SERVICES: "_PARSE_.STOP.ALL_SERVICES",
  /**
   * Starts all services
   */
  START_ALL_SERVICES: "_PARSE_.START.ALL_SERVICES",
  /**
   * Starts service indicated by serviceID.
   * @param {String} serviceID Parameter {0}. Example: HDFS
   */
  START_SERVICE: "_PARSE_.START.{0}",
  /**
   * Stops service indicated by serviceID.
   * @param {String} serviceID Parameter {0}. Example: HDFS
   */
  STOP_SERVICE: "_PARSE_.STOP.{0}",
  /**
   * Performs rolling restart of componentID in batches.
   * This context is the batchNumber batch out of totalBatchCount batches.
   * @param {String} componentID Parameter {0}. Example "DATANODE"
   * @param {Number} batchNumber Parameter {1}. Batch number of this batch. Example 3.
   * @param {Number} totalBatchCount Parameter {2}. Total number of batches. Example 10.
   */
  ROLLING_RESTART: "_PARSE_.ROLLING-RESTART.{0}.{1}.{2}"
};

});

require.register("controllers/global/cluster_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var stringUtils = require('utils/string_utils');
var credentialUtils = require('utils/credentials');

App.ClusterController = Em.Controller.extend(App.ReloadPopupMixin, {
  name: 'clusterController',
  isLoaded: false,
  ambariProperties: null,
  clusterEnv: null,
  clusterDataLoadedPercent: 'width:0', // 0 to 1

  isClusterNameLoaded: false,

  isAlertsLoaded: false,

  isComponentsStateLoaded: false,

  isHostsLoaded: false,

  isConfigsPropertiesLoaded: false,

  isComponentsConfigLoaded: false,

  isStackConfigsLoaded: false,

  isServiceMetricsLoaded: false,

  /**
   * @type {boolean}
   */
  isHostComponentMetricsLoaded: false,

  isHDFSNameSpacesLoaded: false,

  /**
   * This counter used as event trigger to notify that quick links should be changed.
   */
  quickLinksUpdateCounter: 0,

  /**
   * Ambari uses custom jdk.
   * @type {Boolean}
   */
  isCustomJDK: false,

  isHostContentLoaded: Em.computed.and('isHostsLoaded', 'isComponentsStateLoaded'),

  isServiceContentFullyLoaded: Em.computed.and('isServiceMetricsLoaded', 'isComponentsStateLoaded', 'isComponentsConfigLoaded'),

  isStackVersionsLoaded: false,

  clusterName: Em.computed.alias('App.clusterName'),

  updateLoadStatus: function updateLoadStatus(item) {
    var loadList = this.get('dataLoadList');
    var loaded = true;
    var numLoaded = 0;
    var loadListLength = 0;
    loadList.set(item, true);
    for (var i in loadList) {
      if (loadList.hasOwnProperty(i)) {
        loadListLength++;
        if (!loadList[i] && loaded) {
          loaded = false;
        }
      }
      // calculate the number of true
      if (loadList.hasOwnProperty(i) && loadList[i]) {
        numLoaded++;
      }
    }
    this.set('isLoaded', loaded);
    this.set('clusterDataLoadedPercent', 'width:' + Math.floor(numLoaded / loadListLength * 100).toString() + '%');
  },

  dataLoadList: Em.Object.create({
    'stackComponents': false,
    'services': false
  }),

  /**
   * load cluster name
   */
  loadClusterName: function loadClusterName(reload, deferred) {
    var dfd = deferred || $.Deferred();

    if (App.get('clusterName') && !reload) {
      App.set('clusterName', this.get('clusterName'));
      this.set('isClusterNameLoaded', true);
      dfd.resolve();
    } else {
      App.ajax.send({
        name: 'cluster.load_cluster_name',
        sender: this,
        data: {
          reloadPopupText: Em.I18n.t('app.reloadPopup.noClusterName.text'),
          errorLogMessage: 'failed on loading cluster name',
          callback: this.loadClusterName,
          args: [reload, dfd],
          shouldUseDefaultHandler: true
        },
        success: 'reloadSuccessCallback',
        error: 'reloadErrorCallback',
        callback: function callback() {
          if (!App.get('currentStackVersion')) {
            App.set('currentStackVersion', App.defaultStackVersion);
          }
        }
      }).then(function () {
        dfd.resolve();
      }, null);
    }
    return dfd.promise();
  },

  reloadSuccessCallback: function reloadSuccessCallback(data) {
    this._super();
    if (data.items && data.items.length > 0) {
      App.setProperties({
        clusterId: data.items[0].Clusters.cluster_id,
        clusterName: data.items[0].Clusters.cluster_name,
        currentStackVersion: data.items[0].Clusters.version,
        isKerberosEnabled: data.items[0].Clusters.security_type === 'KERBEROS'
      });
      this.set('isClusterNameLoaded', true);
    }
  },

  setServerClock: function setServerClock(data) {
    var clientClock = new Date().getTime();
    var serverClock = Em.getWithDefault(data, 'RootServiceComponents.server_clock', '').toString();
    serverClock = serverClock.length < 13 ? serverClock + '000' : serverClock;
    App.set('clockDistance', serverClock - clientClock);
    App.set('currentServerTime', parseInt(serverClock));
  },

  getServerClockErrorCallback: Em.K,

  getUrl: function getUrl(testUrl, url) {
    return App.get('testMode') ? testUrl : App.get('apiPrefix') + '/clusters/' + App.get('clusterName') + url;
  },

  /**
   *  load all data and update load status
   */
  loadClusterData: function loadClusterData() {
    this.loadAuthorizations();
    this.getAllHostNames();

    if (!App.get('clusterName')) {
      return;
    }

    if (this.get('isLoaded')) {
      // do not load data repeatedly
      App.router.get('mainController').startPolling();
      return;
    }
    App.router.get('userSettingsController').getAllUserSettings();
    App.router.get('errorsHandlerController').loadErrorLogs();

    this.loadClusterInfo();
    this.restoreUpgradeState();
    App.router.get('wizardWatcherController').getUser();

    this.loadClusterDataToModel();

    //force clear filters  for hosts page to load all data
    App.db.setFilterConditions('mainHostController', null);
  },

  loadClusterInfo: function loadClusterInfo() {
    var clusterUrl = this.getUrl('/data/clusters/cluster.json', '?fields=Clusters');
    App.HttpClient.get(clusterUrl, App.clusterMapper, {
      complete: function complete(jqXHR, textStatus) {
        App.set('isCredentialStorePersistent', Em.getWithDefault(App.Cluster.find().findProperty('clusterName', App.get('clusterName')), 'isCredentialStorePersistent', false));
      }
    }, Em.K);
  },

  /**
   * Order of loading:
   * 1. load all created service components
   * 2. request for service components supported by stack
   * 3. load stack components to model
   * 4. request for services
   * 5. put services in cache
   * 6. request for hosts and host-components (single call)
   * 7. request for service metrics
   * 8. load host-components to model
   * 9. load services from cache with metrics to model
   */
  loadClusterDataToModel: function loadClusterDataToModel() {
    var self = this;

    this.loadStackServiceComponents(function (data) {
      data.items.forEach(function (service) {
        service.StackServices.is_selected = true;
        service.StackServices.is_installed = false;
      }, self);
      App.stackServiceMapper.mapStackServices(data);
      App.config.setPreDefinedServiceConfigs(true);
      self.updateLoadStatus('stackComponents');
      self.loadServicesAndComponents();
    });
  },

  loadServicesAndComponents: function loadServicesAndComponents() {
    var updater = App.router.get('updateController');
    var self = this;

    updater.updateServices(function () {
      self.updateLoadStatus('services');

      //hosts should be loaded after services in order to properly populate host-component relation in App.cache.services
      updater.updateHost(function () {
        self.set('isHostsLoaded', true);
        self.loadAlerts();
      });
      self.loadConfigProperties();
      // components state loading doesn't affect overall progress
      updater.updateComponentsState(function () {
        self.set('isComponentsStateLoaded', true);
        // service metrics should be loaded after components state for mapping service components to service in the DS model
        // service metrics loading doesn't affect overall progress
        updater.updateServiceMetric(function () {
          self.set('isServiceMetricsLoaded', true);
          // make second call, because first is light since it doesn't request host-component metrics
          updater.updateServiceMetric(function () {
            self.set('isHostComponentMetricsLoaded', true);
            updater.updateHDFSNameSpaces();
          });
          // components config loading doesn't affect overall progress
          self.loadComponentWithStaleConfigs(function () {
            self.set('isComponentsConfigLoaded', true);
          });
        });
      });
    });
  },

  loadComponentWithStaleConfigs: function loadComponentWithStaleConfigs(callback) {
    return App.ajax.send({
      name: 'components.get.staleConfigs',
      sender: this,
      success: 'loadComponentWithStaleConfigsSuccessCallback',
      callback: callback
    });
  },

  loadComponentWithStaleConfigsSuccessCallback: function loadComponentWithStaleConfigsSuccessCallback(json) {
    json.items.forEach(function (item) {
      var componentName = item.ServiceComponentInfo.component_name;
      var hosts = item.host_components.mapProperty('HostRoles.host_name') || [];
      App.componentsStateMapper.updateStaleConfigsHosts(componentName, hosts);
    });
  },

  loadConfigProperties: function loadConfigProperties() {
    var self = this;

    App.config.loadConfigsFromStack(App.Service.find().mapProperty('serviceName')).always(function () {
      App.config.loadClusterConfigsFromStack().always(function () {
        App.router.get('configurationController').updateConfigTags().always(function () {
          App.router.get('updateController').updateClusterEnv().always(function () {
            self.set('isConfigsPropertiesLoaded', true);
          });
        });
      });
    });
  },

  loadAlerts: function loadAlerts() {
    var updater = App.router.get('updateController');
    var self = this;

    console.time('Overall alerts loading time');
    updater.updateAlertGroups(function () {
      updater.updateAlertDefinitions(function () {
        updater.updateAlertDefinitionSummary(function () {
          updater.updateUnhealthyAlertInstances(function () {
            console.timeEnd('Overall alerts loading time');
            self.set('isAlertsLoaded', true);
          });
        });
      });
    });
  },

  /**
   * restore upgrade status from server
   * and make call to get latest status from server
   * Also loading all upgrades to App.StackUpgradeHistory model
   */
  restoreUpgradeState: function restoreUpgradeState() {
    var self = this;
    return this.getAllUpgrades().done(function (data) {
      var upgradeController = App.router.get('mainAdminStackAndUpgradeController');
      var allUpgrades = data.items.sortProperty('Upgrade.request_id');
      var lastUpgradeData = allUpgrades.pop();
      if (lastUpgradeData) {
        var status = lastUpgradeData.Upgrade.request_status;
        var lastUpgradeNotFinished = self.isSuspendedState(status) || self.isRunningState(status);
        if (lastUpgradeNotFinished) {
          /**
           * No need to display history if there is only one running or suspended upgrade.
           * Because UI still needs to provide user the option to resume the upgrade via the Upgrade Wizard UI.
           * If there is more than one upgrade. Show/Hive the tab based on the status.
           */
          var hasFinishedUpgrades = allUpgrades.some(function (item) {
            var status = item.Upgrade.request_status;
            if (!self.isRunningState(status)) {
              return true;
            }
          }, self);
          App.set('upgradeHistoryAvailable', hasFinishedUpgrades);
        } else {
          //There is at least one finished upgrade. Display it.
          App.set('upgradeHistoryAvailable', true);
        }
      } else {
        //There is no upgrades at all.
        App.set('upgradeHistoryAvailable', false);
      }

      //completed upgrade shouldn't be restored
      if (lastUpgradeData) {
        if (lastUpgradeData.Upgrade.request_status !== "COMPLETED") {
          upgradeController.restoreLastUpgrade(lastUpgradeData);
        }
      } else {
        upgradeController.initDBProperties();
      }

      App.stackUpgradeHistoryMapper.map(data);
      upgradeController.loadStackVersionsToModel(true).done(function () {
        upgradeController.loadCompatibleVersions();
        upgradeController.updateCurrentStackVersion();
        App.set('stackVersionsAvailable', App.StackVersion.find().content.length > 0);
        self.set('isStackVersionsLoaded', true);
      });
    });
  },

  isRunningState: function isRunningState(status) {
    if (status) {
      return "IN_PROGRESS" === status || "PENDING" === status || status.contains("HOLDING");
    } else {
      //init state
      return true;
    }
  },

  /**
   * ABORTED should be handled as SUSPENDED for the lastUpgradeItem
   * */
  isSuspendedState: function isSuspendedState(status) {
    return "ABORTED" === status;
  },

  loadRootService: function loadRootService() {
    return App.ajax.send({
      name: 'service.ambari',
      sender: this
    });
  },

  requestHosts: function requestHosts(realUrl, callback) {
    var testHostUrl = '/data/hosts/HDP2/hosts.json';
    var url = this.getUrl(testHostUrl, realUrl);
    App.HttpClient.get(url, App.hostsMapper, {
      complete: callback
    }, callback);
  },

  /**
   *
   * @param callback
   * @returns {?object}
   */
  loadStackServiceComponents: function loadStackServiceComponents(callback) {
    var callbackObj = {
      loadStackServiceComponentsSuccess: callback
    };
    return App.ajax.send({
      name: 'wizard.service_components',
      data: {
        stackUrl: App.get('stackVersionURL'),
        stackVersion: App.get('currentStackVersionNumber')
      },
      sender: callbackObj,
      success: 'loadStackServiceComponentsSuccess'
    });
  },

  loadAmbariProperties: function loadAmbariProperties() {
    return App.ajax.send({
      name: 'ambari.service',
      sender: this,
      success: 'loadAmbariPropertiesSuccess',
      error: 'loadAmbariPropertiesError'
    });
  },

  loadAuthorizations: function loadAuthorizations() {
    return App.ajax.send({
      name: 'router.user.authorizations',
      sender: this,
      data: { userName: App.db.getLoginName() },
      success: 'loadAuthorizationsSuccessCallback'
    });
  },

  loadAuthorizationsSuccessCallback: function loadAuthorizationsSuccessCallback(response) {
    if (response && response.items) {
      App.set('auth', response.items.mapProperty('AuthorizationInfo.authorization_id').uniq());
      App.db.setAuth(App.get('auth'));
    }
  },

  loadAmbariPropertiesSuccess: function loadAmbariPropertiesSuccess(data) {
    var mainController = App.router.get('mainController');
    this.set('ambariProperties', data.RootServiceComponents.properties);
    // Absence of 'jdk.name' and 'jce.name' properties says that ambari configured with custom jdk.
    this.set('isCustomJDK', App.isEmptyObject(App.permit(data.RootServiceComponents.properties, ['jdk.name', 'jce.name'])));
    this.setServerClock(data);
    mainController.setAmbariServerVersion.call(mainController, data);
    mainController.monitorInactivity();
  },

  loadAmbariPropertiesError: Em.K,

  updateClusterData: function updateClusterData() {
    var testUrl = '/data/clusters/HDP2/cluster.json';
    var clusterUrl = this.getUrl(testUrl, '?fields=Clusters');
    App.HttpClient.get(clusterUrl, App.clusterMapper, {
      complete: function complete() {}
    });
  },

  /**
   *
   * @returns {*|Transport|$.ajax|boolean|ServerResponse}
   */
  getAllHostNames: function getAllHostNames() {
    return App.ajax.send({
      name: 'hosts.all',
      sender: this,
      success: 'getHostNamesSuccess',
      error: 'getHostNamesError'
    });
  },

  getHostNamesSuccess: function getHostNamesSuccess(data) {
    App.set("allHostNames", data.items.mapProperty("Hosts.host_name"));
  },

  getHostNamesError: Em.K,

  /**
   * puts kerberos admin credentials in the live cluster session
   * and resend ajax request
   * @param {credentialResourceObject} credentialResource
   * @param {object} ajaxOpt
   * @returns {$.ajax}
   */
  createKerberosAdminSession: function createKerberosAdminSession(credentialResource, ajaxOpt) {
    return credentialUtils.createOrUpdateCredentials(App.get('clusterName'), credentialUtils.ALIAS.KDC_CREDENTIALS, credentialResource).then(function () {
      if (ajaxOpt) {
        $.ajax(ajaxOpt);
      }
    });
  },

  //TODO Replace this check with any other which is applicable to non-HDP stack
  /**
   * Check if HDP stack version is more or equal than 2.2.2 to determine if pluggable metrics for Storm are supported
   * @method checkDetailedRepoVersion
   * @returns {promise|*|promise|promise|HTMLElement|promise}
   */
  checkDetailedRepoVersion: function checkDetailedRepoVersion() {
    var dfd;
    var currentStackName = App.get('currentStackName');
    var currentStackVersionNumber = App.get('currentStackVersionNumber');
    if (currentStackName == 'HDP' && currentStackVersionNumber == '2.2') {
      dfd = App.ajax.send({
        name: 'cluster.load_detailed_repo_version',
        sender: this,
        success: 'checkDetailedRepoVersionSuccessCallback',
        error: 'checkDetailedRepoVersionErrorCallback'
      });
    } else {
      dfd = $.Deferred();
      App.set('isStormMetricsSupported', currentStackName != 'HDP' || stringUtils.compareVersions(currentStackVersionNumber, '2.2') == 1);
      dfd.resolve();
    }
    return dfd.promise();
  },

  checkDetailedRepoVersionSuccessCallback: function checkDetailedRepoVersionSuccessCallback(data) {
    var rv = (Em.getWithDefault(data, 'items', []) || []).filter(function (i) {
      return Em.getWithDefault(i || {}, 'ClusterStackVersions.stack', null) === App.get('currentStackName') && Em.getWithDefault(i || {}, 'ClusterStackVersions.version', null) === App.get('currentStackVersionNumber');
    })[0];
    var version = Em.getWithDefault(rv || {}, 'repository_versions.0.RepositoryVersions.repository_version', false);
    App.set('isStormMetricsSupported', stringUtils.compareVersions(version, '2.2.2') > -1 || !version);
  },
  checkDetailedRepoVersionErrorCallback: function checkDetailedRepoVersionErrorCallback() {
    App.set('isStormMetricsSupported', true);
  },

  /**
   * Load required data for all upgrades from API
   * @returns {$.ajax}
   */
  getAllUpgrades: function getAllUpgrades() {
    return App.ajax.send({
      name: 'cluster.load_last_upgrade',
      sender: this
    });
  },

  triggerQuickLinksUpdate: function triggerQuickLinksUpdate() {
    this.incrementProperty('quickLinksUpdateCounter');
  }
});

});

require.register("controllers/global/configuration_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ConfigurationController = Em.Controller.extend({
  name: 'configurationController',

  /**
   * get configs by tags
   * return Deferred object with configs as argument
   * @param tags {Object}
   * ** siteName
   * ** tagName (optional)
   * @return {object}
   */
  getConfigsByTags: function getConfigsByTags(tags) {
    var storedTags = [];
    App.db.getConfigs().forEach(function (site) {
      storedTags.push({
        siteName: site.type,
        tagName: site.tag
      });
    });
    if (this.checkTagsChanges(tags, storedTags)) {
      return this.loadFromServer(tags);
    } else {
      return this.loadFromDB(tags.mapProperty('siteName'));
    }
  },

  /**
   * if no sites specified then configs from all sites will be fetched
   * @param {Array} sites
   * @returns {$.Deferred}
   */
  getCurrentConfigsBySites: function getCurrentConfigsBySites() {
    var _this = this;

    var sites = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];

    var dfd = $.Deferred();
    this.getConfigTags(sites).done(function (tags) {
      _this.getConfigsByTags(tags).done(dfd.resolve);
    });
    return dfd.promise();
  },

  /**
   * check whether tag versions have been changed
   * if they are different then return true
   * otherwise false
   * @param tags
   * @param storedTags
   * @return {Boolean}
   */
  checkTagsChanges: function checkTagsChanges(tags, storedTags) {
    var isDifferent = false;
    var i = 0;
    while (i < tags.length && !isDifferent) {
      var storedTag = storedTags.findProperty('siteName', tags[i].siteName);
      isDifferent = !storedTag || !tags[i].tagName || storedTag.tagName !== tags[i].tagName;
      i++;
    }
    return isDifferent;
  },

  /**
   *
   * @param {Array} siteNames
   * @returns {*}
   */
  loadFromDB: function loadFromDB(siteNames) {
    var dfd = $.Deferred();
    var configs = App.db.getConfigs().filter(function (site) {
      return siteNames.contains(site.type);
    });
    dfd.resolve(configs);
    return dfd.promise();
  },
  /**
   * load configs from server
   * and update them in local DB
   * @param tags
   * @return {Array}
   */
  loadFromServer: function loadFromServer(tags) {
    var self = this;
    var dfd = $.Deferred();
    if (!tags.everyProperty('tagName')) {
      var configTags;
      var jqXhr = this.loadConfigTags();
      jqXhr.done(function (data) {
        configTags = data.Clusters.desired_configs;
        tags.forEach(function (_tag) {
          if (_tag.siteName && configTags[_tag.siteName] && !_tag.tagName) {
            _tag.tagName = configTags[_tag.siteName].tag;
          }
        }, self);
        self.loadConfigsByTags(tags, dfd);
      });
    } else {
      self.loadConfigsByTags(tags, dfd);
    }
    return dfd.promise();
  },

  /**
   *  loadConfigsByTags: Loads properties for a config tag
   *  @params tags
   *  @params dfd jqXhr promise
   */
  loadConfigsByTags: function loadConfigsByTags(tags, dfd) {
    var self = this;
    var loadedConfigs = [];
    App.config.loadConfigsByTags(tags).done(function (data) {
      if (data.items) {
        data.items.forEach(function (item) {
          loadedConfigs.push(item);
        });
      }
    }).complete(function () {
      self.saveToDB(loadedConfigs);
      dfd.resolve(loadedConfigs);
    });
  },

  /**
   * loadConfigTags: Loads all config tags applied to the cluster
   * @return: jqXhr promise
   */
  loadConfigTags: function loadConfigTags() {
    return App.ajax.send({
      name: 'config.tags',
      sender: this
    });
  },

  /**
   * save properties obtained from server to local DB
   * @param loadedConfigs
   */
  saveToDB: function saveToDB(loadedConfigs) {
    var storedConfigs = App.db.getConfigs();
    loadedConfigs.forEach(function (loadedSite) {
      var storedSite = storedConfigs.findProperty('type', loadedSite.type);
      if (storedSite) {
        storedSite.tag = loadedSite.tag;
        storedSite.properties = loadedSite.properties;
        storedSite.properties_attributes = loadedSite.properties_attributes;
      } else {
        storedConfigs.push(loadedSite);
      }
    });
    App.db.setConfigs(storedConfigs);
  },

  /**
   * @param {Array} sites
   * @return {Array}
   */
  getConfigTags: function getConfigTags() {
    var _this2 = this;

    var sites = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : [];

    var dfd = $.Deferred();
    if (App.db.getTags().length > 0) {
      dfd.resolve(this.extractTagsFromLocalDB(sites));
    } else {
      this.updateConfigTags().always(function () {
        dfd.resolve(_this2.extractTagsFromLocalDB(sites));
      });
    }
    return dfd.promise();
  },

  /**
   *
   * @param {Array} sites
   * @return {Array}
   */
  extractTagsFromLocalDB: function extractTagsFromLocalDB(sites) {
    return App.db.getTags().filter(function (tag) {
      if (sites.length > 0) {
        return sites.contains(tag.siteName);
      } else {
        return true;
      }
    });
  },

  /**
   * update configs' tags from server
   */
  updateConfigTags: function updateConfigTags() {
    return this.loadConfigTags().done(function (data) {
      var tags = [];
      for (var site in data.Clusters.desired_configs) {
        tags.push({
          siteName: site,
          tagName: data.Clusters.desired_configs[site].tag
        });
      }
      App.db.setTags(tags);
    });
  }
});

});

require.register("controllers/global/errors_handler_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ErrorsHandlerController = Em.Controller.extend(App.Persist, {

  name: 'errorsHandlerController',

  /**
  * @const
  */
  ERROR_STORAGE_SIZE: 500000,

  /**
  * @const
  */
  MAX_TRACE_LENGTH: 1000,

  init: function init() {
    var oldError = window.onerror || Em.K;
    var self = this;
    window.onerror = function (err, url, lineNumber, colNumber, Err) {
      oldError.call(this, err, url, lineNumber, colNumber, Err);
      self.saveErrorLogs(err, url, lineNumber, colNumber, Err);
    };
    return this._super();
  },

  /**
   * load logs from server
   */
  loadErrorLogs: function loadErrorLogs() {
    this.getUserPref('errors');
  },

  /**
  * @method getUserPrefSuccessCallback
  * @param {object|null} data
  */
  getUserPrefSuccessCallback: function getUserPrefSuccessCallback(data) {
    if (data) {
      localStorage.setObject('errors', data);
    }
  },

  /**
   * save error logs to localStorage and server
   * @param {string} err
   * @param {string} url
   * @param {number} lineNumber
   * @param {number} colNumber
   * @param {Error} Err
   */
  saveErrorLogs: function saveErrorLogs(err, url, lineNumber, colNumber, Err) {
    var ls = localStorage.getObject('errors') || {};
    var key = new Date().getTime();
    var stackTrace = Em.get(Err || {}, 'stack');

    if (stackTrace) {
      var origin = location.origin || location.protocol + '//' + location.host,
          path = origin + location.pathname + 'javascripts',
          pattern = new RegExp(path, 'g');
      stackTrace = stackTrace.replace(pattern, '').substr(0, this.MAX_TRACE_LENGTH);
    }

    var val = {
      file: url,
      line: lineNumber,
      col: colNumber,
      error: err,
      stackTrace: stackTrace
    };

    //overwrite errors if storage full
    if (JSON.stringify(ls).length > this.ERROR_STORAGE_SIZE) {
      delete ls[Object.keys(ls).sort()[0]];
    }

    ls[key] = val;
    localStorage.setObject('errors', ls);
    this.postUserPref('errors', ls);
  }
});

});

require.register("controllers/global/update_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var stringUtils = require('utils/string_utils');

App.UpdateController = Em.Controller.extend({
  name: 'updateController',
  isUpdated: false,
  cluster: null,
  isWorking: false,
  updateAlertInstances: Em.computed.and('isWorking', '!App.router.mainAlertInstancesController.isUpdating'),
  timeIntervalId: null,
  clusterName: Em.computed.alias('App.router.clusterController.clusterName'),

  paginationKeys: ['page_size', 'from'],

  /**
   * @type {Array}
   */
  serviceComponentMetrics: ['host_components/metrics/jvm/memHeapUsedM', 'host_components/metrics/jvm/HeapMemoryMax', 'host_components/metrics/jvm/HeapMemoryUsed', 'host_components/metrics/jvm/memHeapCommittedM', 'host_components/metrics/mapred/jobtracker/trackers_decommissioned', 'host_components/metrics/cpu/cpu_wio', 'host_components/metrics/rpc/client/RpcQueueTime_avg_time', 'host_components/metrics/dfs/FSNamesystem/*', 'host_components/metrics/dfs/namenode/Version', 'host_components/metrics/dfs/namenode/LiveNodes', 'host_components/metrics/dfs/namenode/DeadNodes', 'host_components/metrics/dfs/namenode/DecomNodes', 'host_components/metrics/dfs/namenode/TotalFiles', 'host_components/metrics/dfs/namenode/UpgradeFinalized', 'host_components/metrics/dfs/namenode/Safemode', 'host_components/metrics/runtime/StartTime'],

  /**
   * @type {object}
   */
  serviceSpecificParams: {
    'FLUME': "host_components/processes/HostComponentProcess",
    'YARN': "host_components/metrics/yarn/Queue," + "host_components/metrics/yarn/ClusterMetrics/NumActiveNMs," + "host_components/metrics/yarn/ClusterMetrics/NumLostNMs," + "host_components/metrics/yarn/ClusterMetrics/NumUnhealthyNMs," + "host_components/metrics/yarn/ClusterMetrics/NumRebootedNMs," + "host_components/metrics/yarn/ClusterMetrics/NumDecommissionedNMs",
    'HBASE': "host_components/metrics/hbase/master/IsActiveMaster," + "host_components/metrics/hbase/master/MasterStartTime," + "host_components/metrics/hbase/master/MasterActiveTime," + "host_components/metrics/hbase/master/AverageLoad," + "host_components/metrics/master/AssignmentManager/ritCount",
    'STORM': 'metrics/api/v1/cluster/summary,metrics/api/v1/topology/summary,metrics/api/v1/nimbus/summary',
    'HDFS': 'host_components/metrics/dfs/namenode/ClusterId'
  },

  nameNodeMetricsModelProperties: ['jvm_memory_heap_max_values', 'jvm_memory_heap_used_values', 'capacity_used', 'capacity_total', 'capacity_remaining', 'capacity_non_dfs_used', 'name_node_rpc_values', 'name_node_start_time_values'],

  /**
   * @type {string}
   */
  HOSTS_TEST_URL: '/data/hosts/HDP2/hosts.json',

  /**
   * map which track status of requests, whether it's running or completed
   * @type {object}
   */
  requestsRunningStatus: {
    "updateServiceMetric": false
  },

  getUrl: function getUrl(testUrl, url) {
    return App.get('testMode') ? testUrl : App.apiPrefix + '/clusters/' + this.get('clusterName') + url;
  },

  /**
   * construct URL from real URL and query parameters
   * @param realUrl
   * @param queryParams
   * @return {String}
   */
  getComplexUrl: function getComplexUrl(realUrl, queryParams) {
    var prefix = App.get('apiPrefix') + '/clusters/' + App.get('clusterName'),
        params = '';

    if (queryParams) {
      params = this.computeParameters(queryParams);
    }
    params = params.length > 0 ? params + "&" : params;
    return prefix + realUrl.replace('<parameters>', params);
  },

  /**
   * compute parameters according to their type
   * @param queryParams
   * @return {String}
   */
  computeParameters: function computeParameters(queryParams) {
    var params = '';

    queryParams.forEach(function (param) {
      var customKey = param.key;

      switch (param.type) {
        case 'EQUAL':
          if (Em.isArray(param.value)) {
            params += param.key + '.in(' + param.value.join(',') + ')';
          } else {
            params += param.key + '=' + param.value;
          }
          break;
        case 'LESS':
          params += param.key + '<' + param.value;
          break;
        case 'MORE':
          params += param.key + '>' + param.value;
          break;
        case 'MATCH':
          if (Em.isArray(param.value)) {
            params += '(' + param.value.map(function (v) {
              return param.key + '.matches(' + v + ')';
            }).join('|') + ')';
          } else {
            params += param.key + '.matches(' + param.value + ')';
          }
          break;
        case 'MULTIPLE':
          params += param.key + '.in(' + param.value.join(',') + ')';
          break;
        case 'SORT':
          params += 'sortBy=' + param.key + '.' + param.value;
          break;
        case 'CUSTOM':
          param.value.forEach(function (item, index) {
            customKey = customKey.replace('{' + index + '}', item);
          }, this);
          params += customKey;
          break;
        case 'COMBO':
          params += App.router.get('mainHostComboSearchBoxController').generateQueryParam(param);
          break;
      }
      params += '&';
    });
    return params.substring(0, params.length - 1);
  },

  /**
   * depict query parameters of table
   */
  queryParams: Em.Object.create({
    'Hosts': []
  }),

  /**
   * Pagination query-parameters for unhealthy alerts request
   * @type {{from: Number, page_size: Number}}
   */
  queryParamsForUnhealthyAlertInstances: {
    from: 0,
    page_size: 10
  },

  /**
   * map describes relations between updater function and table
   */
  tableUpdaterMap: {
    'Hosts': 'updateHost'
  },

  /**
   * Start polling, when <code>isWorking</code> become true
   */
  updateAll: function () {
    if (this.get('isWorking') && !App.get('isOnlyViewUser')) {
      App.updater.run(this, 'updateHostsMetrics', 'isWorking', App.contentUpdateInterval, '\/main\/(hosts).*');
      App.updater.run(this, 'updateServiceMetric', 'isWorking', App.componentsUpdateInterval, '\/main\/(dashboard|services).*');
      App.updater.run(this, 'graphsUpdate', 'isWorking');

      if (!App.get('router.mainAlertInstancesController.isUpdating')) {
        App.updater.run(this, 'updateUnhealthyAlertInstances', 'updateAlertInstances', App.alertInstancesUpdateInterval, '\/main\/alerts.*');
      }
      App.updater.run(this, 'updateWizardWatcher', 'isWorking', App.bgOperationsUpdateInterval);
      App.updater.run(this, 'updateHDFSNameSpaces', 'isWorking', App.componentsUpdateInterval, '\/main\/(dashboard|services\/HDFS|hosts).*');
    }
  }.observes('isWorking', 'App.router.mainAlertInstancesController.isUpdating'),

  startSubscriptions: function startSubscriptions() {
    App.StompClient.subscribe('/events/hostcomponents', App.hostComponentStatusMapper.map.bind(App.hostComponentStatusMapper));
    App.StompClient.subscribe('/events/alerts', App.alertSummaryMapper.map.bind(App.alertSummaryMapper));
    App.StompClient.subscribe('/events/ui_topologies', App.topologyMapper.map.bind(App.topologyMapper));
    App.StompClient.subscribe('/events/configs', this.configsChangedHandler.bind(this));
    App.StompClient.subscribe('/events/services', App.serviceStateMapper.map.bind(App.serviceStateMapper));
    App.StompClient.subscribe('/events/hosts', App.hostStateMapper.map.bind(App.hostStateMapper));
    App.StompClient.subscribe('/events/alert_definitions', App.alertDefinitionsMapperAdapter.map.bind(App.alertDefinitionsMapperAdapter));
    App.StompClient.subscribe('/events/alert_group', App.alertGroupsMapperAdapter.map.bind(App.alertGroupsMapperAdapter));
    App.StompClient.subscribe('/events/upgrade', App.upgradeStateMapper.map.bind(App.upgradeStateMapper));
    App.router.get('backgroundOperationsController').subscribeToUpdates();
  },

  /**
   *
   * @param {boolean} loadMetricsSeparately
   * @returns {string|*}
   */
  getUpdateHostUrlWithParams: function getUpdateHostUrlWithParams(loadMetricsSeparately) {
    var url = '/hosts?fields=Hosts/rack_info,Hosts/host_name,Hosts/maintenance_state,Hosts/public_host_name,' + 'Hosts/cpu_count,Hosts/ph_cpu_count,<lastAgentEnv>alerts_summary,Hosts/host_status,Hosts/host_state,' + 'Hosts/last_heartbeat_time,Hosts/ip,host_components/HostRoles/state,' + 'host_components/HostRoles/maintenance_state,host_components/HostRoles/stale_configs,' + 'host_components/HostRoles/service_name,host_components/HostRoles/display_name,' + 'host_components/HostRoles/desired_admin_state,<nameNodeMetrics>' + '<metrics>Hosts/total_mem<hostDetailsParams><stackVersions>&minimal_response=true';
    var stackVersionInfo = ',stack_versions/HostStackVersions,' + 'stack_versions/repository_versions/RepositoryVersions/repository_version,' + 'stack_versions/repository_versions/RepositoryVersions/id,' + 'stack_versions/repository_versions/RepositoryVersions/display_name',
        loggingResource = ',host_components/logging',
        isHostDetailPage = App.router.get('currentState.parentState.name') === 'hostDetails',
        isHostsPage = App.router.get('currentState.parentState.name') === 'hosts',
        hostDetailsParams = ',Hosts/os_arch,Hosts/os_type,metrics/cpu/cpu_system,metrics/cpu/cpu_user,' + 'metrics/memory/mem_total,metrics/memory/mem_free',
        nameNodeMetrics = 'host_components/metrics/dfs/namenode/ClusterId,' + 'host_components/metrics/dfs/FSNamesystem/HAState,';

    url = url.replace("<stackVersions>", stackVersionInfo);
    url = url.replace("<metrics>", loadMetricsSeparately ? "" : "metrics/disk,metrics/load/load_one,");
    url = url.replace('<hostDetailsParams>', isHostsPage ? '' : hostDetailsParams);
    url = url.replace('<lastAgentEnv>', isHostDetailPage ? 'Hosts/last_agent_env,' : '');
    url = url.replace('<nameNodeMetrics>', App.Service.find('HDFS').get('isLoaded') ? nameNodeMetrics : '');
    url = App.get('supports.logSearch') ? url + loggingResource : url;
    return url;
  },

  /**
   *
   * @param {Function} callback
   * @param {Function} error
   * @param {boolean} lazyLoadMetrics
   */
  updateHost: function updateHost(callback, error, lazyLoadMetrics) {
    var testUrl = this.get('HOSTS_TEST_URL'),
        self = this,
        hostDetailsFilter = '',
        mainHostController = App.router.get('mainHostController'),
        sortProperties = mainHostController.getSortProps(),
        isHostsLoaded = false,

    // load hosts metrics separately of lazyLoadMetrics=true, but metrics in current request if we are sorting
    loadMetricsSeparately = lazyLoadMetrics && !(sortProperties.length && ['loadAvg', 'diskUsage'].contains(sortProperties[0].name));
    this.get('queryParams').set('Hosts', mainHostController.getQueryParameters(true));
    if (App.router.get('currentState.parentState.name') === 'hosts') {
      App.updater.updateInterval('updateHost', App.get('contentUpdateInterval'));
    } else {
      if (App.router.get('currentState.parentState.name') === 'hostDetails') {
        hostDetailsFilter = App.router.get('location.lastSetURL').match(/\/hosts\/(.*)\/(summary|configs|alerts|stackVersions|logs)/)[1];
        App.updater.updateInterval('updateHost', App.get('componentsUpdateInterval'));
        //if host details page opened then request info only of one displayed host
        this.get('queryParams').set('Hosts', [{
          key: 'Hosts/host_name',
          value: [hostDetailsFilter],
          type: 'MULTIPLE',
          isHostDetails: true
        }]);
      } else {
        // On pages except for hosts/hostDetails, making sure hostsMapper loaded only once on page load, no need to update, but at least once
        isHostsLoaded = App.router.get('clusterController.isHostsLoaded');
        if (isHostsLoaded) {
          callback();
          return;
        }
      }
    }

    var realUrl = this.getUpdateHostUrlWithParams(loadMetricsSeparately);

    var clientCallback = function clientCallback(skipCall, queryParams, itemTotal) {
      var completeCallback = function completeCallback() {
        callback();
        if (loadMetricsSeparately) {
          self.loadHostsMetric(queryParams);
        }
      };
      if (skipCall) {
        //no hosts match filter by component
        App.hostsMapper.map({
          items: [],
          itemTotal: '0'
        });
        callback();
      } else {
        if (App.get('testMode')) {
          realUrl = testUrl;
        } else {
          realUrl = self.addParamsToHostsUrl(queryParams, sortProperties, realUrl);
        }

        App.HttpClient.get(realUrl, App.hostsMapper, {
          complete: completeCallback,
          beforeMap: function beforeMap(response) {
            if (itemTotal) {
              response.itemTotal = itemTotal;
            }
          },
          doGetAsPost: true,
          params: self.computeParameters(queryParams),
          error: error
        });
      }
    };

    if (!this.preLoadHosts(clientCallback)) {
      clientCallback(false, self.get('queryParams.Hosts'));
    }
  },

  updateHostsMetrics: function updateHostsMetrics(callback) {
    var queryParams = App.router.get('mainHostController').getQueryParameters(true);
    if (App.router.get('currentState.parentState.name') === 'hostDetails') {
      var currentHostname = App.router.get('location.lastSetURL').match(/\/hosts\/(.*)\/(summary|configs|alerts|stackVersions|logs)/)[1];
      queryParams = [{
        key: 'Hosts/host_name',
        value: [currentHostname],
        type: 'MULTIPLE',
        isHostDetails: true
      }];
    }
    this.loadHostsMetric(queryParams).always(callback);
  },

  /**
   *
   * @param {Array} queryParams
   * @param {Array} sortProperties
   * @param {string} realUrl
   * @returns {string}
   */
  addParamsToHostsUrl: function addParamsToHostsUrl(queryParams, sortProperties, realUrl) {
    var paginationProps = this.computeParameters(queryParams.filter(function (param) {
      return this.get('paginationKeys').contains(param.key);
    }, this));
    var sortProps = this.computeParameters(sortProperties);

    return App.get('apiPrefix') + '/clusters/' + App.get('clusterName') + realUrl + (paginationProps.length > 0 ? '&' + paginationProps : '') + (sortProps.length > 0 ? '&' + sortProps : '');
  },

  /**
   * lazy load metrics of hosts
   * @param {Array} queryParams
   * @returns {$.ajax|null}
   */
  loadHostsMetric: function loadHostsMetric(queryParams) {
    var isAmbariMetricsStarted = App.Service.find('AMBARI_METRICS').get('isStarted'),
        hostDetailsParam = queryParams.findProperty('isHostDetails'),
        currentHostName = hostDetailsParam && hostDetailsParam.value[0],
        isHostWithNameNode = currentHostName && App.HostComponent.find('NAMENODE_' + currentHostName).get('isLoaded');
    if (isAmbariMetricsStarted || isHostWithNameNode) {
      var realUrl = '/hosts?fields=',
          realUrlFields = [];
      if (isAmbariMetricsStarted) {
        realUrlFields.push('metrics/disk/disk_free', 'metrics/disk/disk_total', 'metrics/load/load_one');
      }
      if (isHostWithNameNode) {
        realUrlFields.push('host_components/metrics/dfs/namenode/ClusterId', 'host_components/metrics/jvm/HeapMemoryMax', 'host_components/metrics/jvm/HeapMemoryUsed', 'host_components/metrics/dfs/FSNamesystem/CapacityUsed', 'host_components/metrics/dfs/FSNamesystem/CapacityTotal', 'host_components/metrics/dfs/FSNamesystem/CapacityRemaining', 'host_components/metrics/dfs/FSNamesystem/CapacityNonDFSUsed', 'host_components/metrics/rpc/client/RpcQueueTime_avg_time', 'host_components/metrics/runtime/StartTime');
      }
      realUrl += realUrlFields.join(',') + '&minimal_response=true';
      return App.ajax.send({
        name: 'hosts.metrics.lazy_load',
        sender: this,
        data: {
          url: this.addParamsToHostsUrl(queryParams, [], realUrl),
          parameters: this.computeParameters(queryParams)
        },
        success: 'loadHostsMetricSuccessCallback'
      });
    }
    return $.Deferred().resolve().promise();
  },

  /**
   * success callback of <code>loadHostsMetric</code>
   * @param {object} data
   */
  loadHostsMetricSuccessCallback: function loadHostsMetricSuccessCallback(data) {
    App.hostsMapper.setMetrics(data);
    if (App.router.get('currentState.parentState.name') === 'hostDetails' && data) {
      var hostComponentsData = Em.get(data, 'items.0.host_components');
      if (hostComponentsData) {
        var nameNodeData = hostComponentsData.findProperty('HostRoles.component_name', 'NAMENODE');
        if (nameNodeData) {
          var hostName = Em.get(data, 'items.0.Hosts.host_name'),
              nameNodeModelMap = App.serviceMetricsMapper.activeNameNodeConfig,
              processedModelProperties = this.nameNodeMetricsModelProperties,
              hdfsModel = App.HDFSService.find('HDFS'),
              componentModel = App.HostComponent.find('NAMENODE_' + hostName);
          Object.keys(nameNodeModelMap).forEach(function (key) {
            if (processedModelProperties.contains(key)) {
              var modelKey = stringUtils.underScoreToCamelCase(key);
              hdfsModel.get(modelKey)[hostName] = Em.get(nameNodeData, nameNodeModelMap[key]);
              hdfsModel.propertyDidChange(modelKey);
            }
          });
          componentModel.setProperties({
            clusterIdValue: Em.get(nameNodeData, 'metrics.dfs.namenode.ClusterId')
          });
        }
      }
    }
  },

  /**
   * identify if any filter by host-component is active
   * if so run @getHostByHostComponents
   *
   * @param callback
   * @return {Boolean}
   */
  preLoadHosts: function preLoadHosts(callback) {
    if (this.get('queryParams.Hosts').length > 0 && this.get('queryParams.Hosts').filter(function (param) {
      return param.isComponentRelatedFilter;
    }, this).length > 0) {
      this.getHostByHostComponents(callback);
      return true;
    }
    return false;
  },

  /**
   * get hosts' names which match filter by host-component
   * @param callback
   */
  getHostByHostComponents: function getHostByHostComponents(callback) {
    var realUrl = '/hosts?<parameters>minimal_response=true';

    App.ajax.send({
      name: 'hosts.host_components.pre_load',
      sender: this,
      data: {
        url: this.getComplexUrl(realUrl, this.get('queryParams.Hosts')),
        callback: callback
      },
      success: 'getHostByHostComponentsSuccessCallback',
      error: 'getHostByHostComponentsErrorCallback'
    });
  },
  getHostByHostComponentsSuccessCallback: function getHostByHostComponentsSuccessCallback(data, opt, params) {
    var queryParams = this.get('queryParams.Hosts');
    var hostNames = data.items.mapProperty('Hosts.host_name');
    var skipCall = hostNames.length === 0;

    var itemTotal = parseInt(data.itemTotal, 10);
    if (!isNaN(itemTotal)) {
      App.router.set('mainHostController.filteredCount', itemTotal);
    }

    if (skipCall) {
      params.callback(skipCall);
    } else {
      queryParams = [{
        key: 'Hosts/host_name',
        value: hostNames,
        type: 'MULTIPLE'
      }];
      params.callback(skipCall, queryParams, itemTotal);
    }
  },
  getHostByHostComponentsErrorCallback: Em.K,

  graphs: [],

  graphsUpdate: function graphsUpdate(callback) {
    var existedGraphs = [];
    this.get('graphs').forEach(function (_graph) {
      var view = Em.View.views[_graph.id];
      if (view) {
        existedGraphs.push(_graph);
        if (!view.get('isRequestRunning')) {
          //console.log('updated graph', _graph.name);
          view.loadData();
          //if graph opened as modal popup update it to
          if ($(".modal-graph-line .modal-body #" + _graph.popupId + "-container-popup").length) {
            view.loadData();
          }
        }
      }
    });
    callback();
    this.set('graphs', existedGraphs);
  },

  /**
   * Updates the services information.
   *
   * @param callback
   */
  updateServiceMetric: function updateServiceMetric(callback) {
    var self = this;
    self.set('isUpdated', false);
    var isATSPresent = App.StackServiceComponent.find().findProperty('componentName', 'APP_TIMELINE_SERVER');

    var conditionalFields = this.getConditionalFields(),
        requestsRunningStatus = this.get('requestsRunningStatus'),
        conditionalFieldsString = conditionalFields.length > 0 ? ',' + conditionalFields.join(',') : '',
        testUrl = '/data/dashboard/HDP2/master_components.json',
        isFlumeInstalled = App.cache.services.mapProperty('ServiceInfo.service_name').contains('FLUME'),
        isATSInstalled = App.cache.services.mapProperty('ServiceInfo.service_name').contains('YARN') && isATSPresent,
        flumeHandlerParam = isFlumeInstalled ? 'ServiceComponentInfo/component_name=FLUME_HANDLER|' : '',
        atsHandlerParam = isATSInstalled ? 'ServiceComponentInfo/component_name=APP_TIMELINE_SERVER|' : '',
        haComponents = App.get('isHaEnabled') ? 'ServiceComponentInfo/component_name=JOURNALNODE|ServiceComponentInfo/component_name=ZKFC|' : '',
        realUrl = '/components/?' + flumeHandlerParam + atsHandlerParam + haComponents + 'ServiceComponentInfo/category.in(MASTER,CLIENT)&fields=' + 'ServiceComponentInfo/service_name,' + 'host_components/HostRoles/display_name,' + 'host_components/HostRoles/host_name,' + 'host_components/HostRoles/public_host_name,' + 'host_components/HostRoles/state,' + 'host_components/HostRoles/maintenance_state,' + 'host_components/HostRoles/stale_configs,' + 'host_components/HostRoles/ha_state,' + 'host_components/HostRoles/desired_admin_state,' + conditionalFieldsString + '&minimal_response=true';

    var servicesUrl = this.getUrl(testUrl, realUrl);
    callback = callback || function () {
      self.set('isUpdated', true);
    };

    if (!requestsRunningStatus.updateServiceMetric) {
      requestsRunningStatus.updateServiceMetric = true;
      App.HttpClient.get(servicesUrl, App.serviceMetricsMapper, {
        complete: function complete() {
          App.set('router.mainServiceItemController.isServicesInfoLoaded', App.get('router.clusterController.isLoaded'));
          requestsRunningStatus.updateServiceMetric = false;
          callback();
        }
      });
    } else {
      callback();
    }
  },
  /**
   * construct conditional parameters of query, depending on which services are installed
   * @return {Array}
   */
  getConditionalFields: function getConditionalFields() {
    var conditionalFields = this.get('serviceComponentMetrics').slice(0);
    var serviceSpecificParams = $.extend({}, this.get('serviceSpecificParams'));
    var metricsKey = 'metrics/';

    if (/^2.1/.test(App.get('currentStackVersionNumber'))) {
      serviceSpecificParams.STORM = 'metrics/api/cluster/summary';
    } else if (/^2.2/.test(App.get('currentStackVersionNumber'))) {
      serviceSpecificParams.STORM = 'metrics/api/v1/cluster/summary,metrics/api/v1/topology/summary';
    }
    serviceSpecificParams.ONEFS = 'metrics/*,';

    App.cache.services.forEach(function (service) {
      var urlParams = serviceSpecificParams[service.ServiceInfo.service_name];
      if (urlParams) {
        conditionalFields.push(urlParams);
      }
    });

    //first load shouldn't contain metrics in order to make call lighter
    if (!App.get('router.clusterController.isServiceMetricsLoaded')) {
      return conditionalFields.filter(function (condition) {
        return condition.indexOf(metricsKey) === -1;
      });
    }

    return conditionalFields;
  },

  updateServices: function updateServices(callback) {
    var testUrl = '/data/services/HDP2/services.json';
    var componentConfigUrl = this.getUrl(testUrl, '/services?fields=ServiceInfo/state,ServiceInfo/maintenance_state,ServiceInfo/desired_repository_version_id,components/ServiceComponentInfo/component_name&minimal_response=true');
    App.HttpClient.get(componentConfigUrl, App.serviceMapper, {
      complete: callback
    });
  },

  updateComponentsState: function updateComponentsState(callback) {
    var testUrl = '/data/services/HDP2/components_state.json';
    var realUrl = '/components/?fields=ServiceComponentInfo/service_name,' + 'ServiceComponentInfo/category,ServiceComponentInfo/installed_count,ServiceComponentInfo/started_count,ServiceComponentInfo/init_count,ServiceComponentInfo/install_failed_count,ServiceComponentInfo/unknown_count,ServiceComponentInfo/total_count,ServiceComponentInfo/display_name,host_components/HostRoles/host_name&minimal_response=true';
    var url = this.getUrl(testUrl, realUrl);

    App.HttpClient.get(url, App.componentsStateMapper, {
      complete: callback
    });
  },

  updateAlertDefinitions: function updateAlertDefinitions(callback) {
    var testUrl = '/data/alerts/alertDefinitions.json';
    var realUrl = '/alert_definitions?fields=' + 'AlertDefinition/component_name,AlertDefinition/description,AlertDefinition/enabled,AlertDefinition/repeat_tolerance,AlertDefinition/repeat_tolerance_enabled,' + 'AlertDefinition/id,AlertDefinition/ignore_host,AlertDefinition/interval,AlertDefinition/label,AlertDefinition/name,' + 'AlertDefinition/scope,AlertDefinition/service_name,AlertDefinition/source,AlertDefinition/help_url';
    var url = this.getUrl(testUrl, realUrl);

    App.HttpClient.get(url, App.alertDefinitionsMapper, {
      complete: callback
    });
  },

  updateUnhealthyAlertInstances: function updateUnhealthyAlertInstances(callback) {
    var testUrl = '/data/alerts/alert_instances.json';
    var queryParams = this.get('queryParamsForUnhealthyAlertInstances');
    var realUrl = '/alerts?fields=' + 'Alert/component_name,Alert/definition_id,Alert/definition_name,Alert/host_name,Alert/id,Alert/instance,' + 'Alert/label,Alert/latest_timestamp,Alert/maintenance_state,Alert/original_timestamp,Alert/scope,' + 'Alert/service_name,Alert/state,Alert/text,Alert/repeat_tolerance,Alert/repeat_tolerance_remaining' + '&Alert/state.in(CRITICAL,WARNING)&Alert/maintenance_state.in(OFF)&from=' + queryParams.from + '&page_size=' + queryParams.page_size;
    var url = this.getUrl(testUrl, realUrl);

    App.HttpClient.get(url, App.alertInstanceMapper, {
      complete: callback
    });
  },

  updateAlertDefinitionSummary: function updateAlertDefinitionSummary(callback) {
    //TODO move to clusterController
    var testUrl = '/data/alerts/alert_summary.json';
    var realUrl = '/alerts?format=groupedSummary';
    var url = this.getUrl(testUrl, realUrl);

    App.HttpClient.get(url, App.alertDefinitionSummaryMapper, {
      complete: callback
    });
  },

  updateAlertGroups: function updateAlertGroups(callback) {
    var testUrl = '/data/alerts/alertGroups.json';
    var realUrl = '/alert_groups?fields=' + 'AlertGroup/default,AlertGroup/definitions,AlertGroup/id,AlertGroup/name,AlertGroup/targets';
    var url = this.getUrl(testUrl, realUrl);

    App.HttpClient.get(url, App.alertGroupsMapper, {
      complete: callback
    });
  },

  updateAlertNotifications: function updateAlertNotifications(callback) {
    App.HttpClient.get(App.get('apiPrefix') + '/alert_targets?fields=*', App.alertNotificationMapper, {
      complete: callback
    });
  },

  configsChangedHandler: function configsChangedHandler(event) {
    if (event.configs && event.configs.someProperty('type', 'cluster-env')) {
      this.updateClusterEnv();
    }
    App.router.get('configurationController').updateConfigTags();
  },

  //TODO - update service auto-start to use this
  updateClusterEnv: function updateClusterEnv() {
    return App.router.get('configurationController').getCurrentConfigsBySites(['cluster-env']).done(function (config) {
      App.router.get('clusterController').set('clusterEnv', config[0]);
    });
  },

  loadClusterConfig: function loadClusterConfig(callback) {
    return App.ajax.send({
      name: 'config.tags.site',
      sender: this,
      data: {
        site: 'cluster-env'
      },
      callback: callback
    });
  },

  updateWizardWatcher: function updateWizardWatcher(callback) {
    App.router.get('wizardWatcherController').getUser().complete(callback);
  },

  /**
   * Request to fetch logging info for specified host.
   *
   * @method updateLogging
   * @param {string} hostName
   * @param {string[]|boolean} fields additional fields to request e.g. ['field1', 'field2']
   * @param {function} callback execute on when request succeed, json response will be passed to first argument
   * @returns {$.Deferred.promise()}
   */
  updateLogging: function updateLogging(hostName, fields, callback) {
    var flds = !!fields ? "," + fields.join(',') : "";

    return App.ajax.send({
      name: 'host.logging',
      sender: this,
      data: {
        hostName: hostName,
        callback: callback,
        fields: flds
      },
      success: 'updateLoggingSuccess'
    });
  },

  updateLoggingSuccess: function updateLoggingSuccess(data, opt, params) {
    var clbk = params.callback || function () {};
    clbk(data);
  },

  updateHDFSNameSpaces: function updateHDFSNameSpaces() {
    if (App.Service.find('HDFS').get('isLoaded') && App.get('isHaEnabled')) {
      App.router.get('configurationController').getCurrentConfigsBySites(['hdfs-site']).done(function (configs) {
        var properties = configs && configs[0] && configs[0].properties;
        if (properties) {
          var nameSpaceProperty = properties['dfs.nameservices'];
          if (nameSpaceProperty) {
            var nameSpaces = nameSpaceProperty.split(',').map(function (nameSpace) {
              var nameNodeIdsProperty = properties['dfs.ha.namenodes.' + nameSpace];
              if (nameNodeIdsProperty) {
                var nameNodeIds = nameNodeIdsProperty.split(','),
                    hostNames = nameNodeIds.map(function (id) {
                  var propertyValue = properties['dfs.namenode.http-address.' + nameSpace + '.' + id],
                      matches = propertyValue && propertyValue.match(/([\D\d]+)\:\d+$/),
                      hostName = matches && matches[1];
                  return hostName;
                });
                return {
                  nameSpace: nameSpace,
                  hostNames: hostNames
                };
              }
            }),
                allNameNodes = App.HDFSService.find().objectAt(0).get('hostComponents').filterProperty('componentName', 'NAMENODE');
            allNameNodes.forEach(function (component) {
              var nameSpaceObject = nameSpaces.find(function (ns) {
                return ns && ns.hostNames && ns.hostNames.contains(component.get('hostName'));
              });
              if (nameSpaceObject) {
                component.set('haNameSpace', nameSpaceObject.nameSpace);
              }
            });
            App.set('router.clusterController.isHDFSNameSpacesLoaded', true);
          }
        }
      });
    } else {
      App.set('router.clusterController.isHDFSNameSpacesLoaded', true);
    }
  }
});

});

require.register("controllers/global/user_settings_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

var timezoneUtils = require('utils/date/timezone');

/**
 * Controller for user settings
 * Allows to get them from persist and update them to the persist
 *
 * @class UserSettingsController
 */
App.UserSettingsController = Em.Controller.extend(App.Persist, {

  name: 'userSettingsController',

  /**
   * @type {object}
   */
  userSettings: {},

  /**
   * Each property's type is {name: string, defaultValue: *, formatter: function}
   *
   * @type {object}
   */
  userSettingsKeys: function () {
    var loginName = App.router.get('loginName');
    var prefix = 'admin-settings-';
    return {
      show_bg: {
        name: prefix + 'show-bg-' + loginName,
        defaultValue: true
      },
      timezone: {
        name: prefix + 'timezone-' + loginName,
        defaultValue: timezoneUtils.detectUserTimezone(),
        formatter: function formatter(v) {
          return timezoneUtils.get('timezonesMappedByValue')[v];
        }
      }
    };
  }.property('App.router.loginName'),

  /**
   * Load some user's setting from the persist
   * If <code>persistKey</code> is not provided, all settings are loaded
   *
   * @param {string} [persistKey]
   * @method dataLoading
   * @returns {$.Deferred.promise}
   */
  dataLoading: function dataLoading(persistKey) {
    var key = persistKey ? this.get('userSettingsKeys.' + persistKey + '.name') : '';
    var dfd = $.Deferred();
    var self = this;
    this.getUserPref(key).complete(function () {
      var curPref = self.get('currentPrefObject');
      self.set('currentPrefObject', null);
      dfd.resolve(curPref);
    });
    return dfd.promise();
  },

  /**
   * Success-callback for user pref
   *
   * @param {?object} response
   * @param {object} opt
   * @returns {?object}
   * @method getUserPrefSuccessCallback
   */
  getUserPrefSuccessCallback: function getUserPrefSuccessCallback(response, opt) {
    var getAllRequest = opt.url.endsWith('persist/');
    if (Em.isNone(response)) {
      this.updateUserPrefWithDefaultValues(response, getAllRequest);
    }
    this.set('currentPrefObject', response);
    return response;
  },

  /**
   * Error-callback for user-pref request
   * Update user pref with default values if user firstly login
   *
   * @param {object} request
   * @method getUserPrefErrorCallback
   */
  getUserPrefErrorCallback: function getUserPrefErrorCallback(request) {
    // this user is first time login
    if (404 === request.status) {
      this.updateUserPrefWithDefaultValues();
    }
  },

  /**
   * Load all current user's settings to the <code>userSettings</code>
   *
   * @method getAllUserSettings
   */
  getAllUserSettings: function getAllUserSettings() {
    var userSettingsKeys = this.get('userSettingsKeys');
    var userSettings = {};
    var self = this;
    this.dataLoading().done(function (json) {
      if (!json) {
        return;
      }
      Object.keys(userSettingsKeys).forEach(function (k) {
        var value = userSettingsKeys[k].defaultValue;
        if (undefined === json[userSettingsKeys[k].name]) {
          self.postUserPref(k, userSettingsKeys[k].defaultValue);
        } else {
          value = JSON.parse(json[userSettingsKeys[k].name]);
        }
        if ('function' === Em.typeOf(userSettingsKeys[k].formatter)) {
          value = userSettingsKeys[k].formatter(value);
        }
        userSettings[k] = value;
      });
    });
    this.set('userSettings', userSettings);
  },

  /**
   * If user doesn't have any settings stored in the persist,
   * default values should be populated there
   *
   * @param {object} [response]
   * @param {boolean} [getAllRequest] determines, if user tried to get one field or all fields
   * @method updateUserPrefWithDefaultValues
   */
  updateUserPrefWithDefaultValues: function updateUserPrefWithDefaultValues(response, getAllRequest) {
    var r = response || {};
    var keys = this.get('userSettingsKeys');
    var self = this;
    if (getAllRequest) {
      Object.keys(keys).forEach(function (key) {
        if (Em.isNone(r[keys[key].name])) {
          self.postUserPref(key, keys[key].defaultValue);
        }
      });
    }
  },

  /**
   * "Short"-key method for post user settings to the persist
   * Example:
   *  real key is something like 'userSettingsKeys.timezone.name'
   *  but user should call this method with 'timezone'
   *
   * @method postUserPref
   * @param {string} key
   * @param {*} value
   * @returns {*}
   */
  postUserPref: function postUserPref(key, value) {
    var normalizedKey = key.startsWith('userSettingsKeys.') ? key : 'userSettingsKeys.' + key + '.name';
    var shortKey = normalizedKey.replace('userSettingsKeys.', '').replace('.name', '');
    this.set('userSettings.' + shortKey, value);
    return this._super(this.get(normalizedKey), value);
  },

  /**
   * Open popup with user settings after settings-request is complete
   *
   * @method showSettingsPopup
   */
  showSettingsPopup: function showSettingsPopup() {
    var self = this;
    // Settings only for admins
    if (!App.isAuthorized('CLUSTER.UPGRADE_DOWNGRADE_STACK')) {
      return;
    }

    this.dataLoading().done(function (response) {
      self.loadPrivileges().complete(function () {
        self._showSettingsPopup(response);
      });
    });
  },

  loadPrivileges: function loadPrivileges() {
    return App.ajax.send({
      name: 'router.user.privileges',
      sender: this,
      data: {
        userName: App.db.getLoginName()
      },
      success: 'loadPrivilegesSuccessCallback'
    });
  },

  loadPrivilegesSuccessCallback: function loadPrivilegesSuccessCallback(data) {
    var key,
        privileges = this.parsePrivileges(data),
        clusters = [],
        views = [];

    for (key in privileges.clusters) {
      clusters.push({
        name: key,
        privileges: privileges.clusters[key]
      });
    }
    for (key in privileges.views) {
      views.push({
        instance_name: key,
        privileges: privileges.views[key].privileges,
        version: privileges.views[key].version,
        view_name: privileges.views[key].view_name
      });
    }
    privileges.clusters = clusters;
    privileges.views = views;
    this.set('privileges', data.items.length ? privileges : null);
    this.set('noClusterPriv', Em.isEmpty(clusters));
    this.set('noViewPriv', Em.isEmpty(views));
    this.set('hidePrivileges', this.get('noClusterPriv') && this.get('noViewPriv'));
  },

  /**
   *
   * @param {?object} data
   * @returns {{clusters: {}, views: {}}}
   */
  parsePrivileges: function parsePrivileges(data) {
    var privileges = {
      clusters: {},
      views: {}
    };
    data.items.forEach(function (privilege) {
      privilege = privilege.PrivilegeInfo;
      if (privilege.type === 'CLUSTER') {
        // This is cluster
        privileges.clusters[privilege.cluster_name] = privileges.clusters[privilege.cluster_name] || [];
        privileges.clusters[privilege.cluster_name].push(privilege.permission_label);
      } else if (privilege.type === 'VIEW') {
        privileges.views[privilege.instance_name] = privileges.views[privilege.instance_name] || { privileges: [] };
        privileges.views[privilege.instance_name].version = privilege.version;
        privileges.views[privilege.instance_name].view_name = privilege.view_name;
        privileges.views[privilege.instance_name].privileges.push(privilege.permission_label);
      }
    });
    return privileges;
  },

  /**
   * Show popup with settings for user
   * Don't call this method directly! Use <code>showSettingsPopup</code>
   *
   * @param {object} response
   * @returns {App.ModalPopup}
   * @method _showSettingsPopup
   * @private
   */
  _showSettingsPopup: function _showSettingsPopup(response) {
    var keys = this.get('userSettingsKeys');
    var curValue, self, timezonesFormatted, initValue, initTimezone;
    if (response[keys.show_bg.name]) {
      curValue = null;
      self = this;
      timezonesFormatted = timezoneUtils.get('timezones');
      initValue = JSON.parse(response[keys.show_bg.name]);
      initTimezone = timezonesFormatted.findProperty('value', JSON.parse(response[keys.timezone.name]));
      return App.ModalPopup.show({

        header: Em.I18n.t('common.userSettings'),

        bodyClass: Em.View.extend({

          templateName: require('templates/common/settings'),

          isNotShowBgChecked: !initValue,

          updateValue: function () {
            curValue = !this.get('isNotShowBgChecked');
          }.observes('isNotShowBgChecked'),

          timezonesList: timezonesFormatted,

          privileges: self.get('privileges'),

          isAdmin: App.get('isAdmin'),

          noClusterPriv: self.get('noClusterPriv'),

          noViewPriv: self.get('noViewPriv'),

          hidePrivileges: self.get('hidePrivileges') || App.get('isAdmin')
        }),

        /**
         * @type {string}
         */
        selectedTimezone: initTimezone,

        primary: Em.I18n.t('common.save'),

        onPrimary: function onPrimary() {
          if (Em.isNone(curValue)) {
            curValue = initValue;
          }
          var tz = this.get('selectedTimezone.value');
          var popup = this;
          if (!App.get('testMode')) {
            self.postUserPref('show_bg', curValue).always(function () {
              self.postUserPref('timezone', tz).always(function () {
                if (popup.needsPageRefresh()) {
                  location.reload();
                }
              });
            });
          }
          this._super();
        },

        /**
         * Determines if page should be refreshed after user click "Save"
         *
         * @returns {boolean}
         */
        needsPageRefresh: function needsPageRefresh() {
          return initTimezone !== this.get('selectedTimezone');
        }
      });
    } else {
      App.showAlertPopup(Em.I18n.t('common.error'), Em.I18n.t('app.settings.noData'));
    }
  }

});

});

require.register("controllers/global/wizard_watcher_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.WizardWatcherController = Em.Controller.extend(App.Persist, {
  name: 'wizardWatcherController',

  /**
   * @const
   */
  PREF_KEY: 'wizard-data',

  /**
   * name of user who working with wizard
   * @type {string|null}
   */
  wizardUser: null,

  /**
   * @type {string|null}
   */
  controllerName: null,

  /**
   * define whether Wizard is running
   * @type {boolean}
   */
  isWizardRunning: Em.computed.bool('wizardUser'),

  /**
   * @type {string}
   */
  wizardDisplayName: function () {
    var controllerName = this.get('controllerName');
    return controllerName ? Em.I18n.t('wizard.inProgress').format(App.router.get(controllerName).get('displayName'), this.get('wizardUser')) : '';
  }.property('controllerName'),

  /**
   * define whether logged in user is the one who started wizard
   * @type {boolean}
   */
  isNonWizardUser: function () {
    return this.get('isWizardRunning') && this.get('wizardUser') !== App.router.get('loginName');
  }.property('App.router.loginName', 'wizardUser').volatile(),

  /**
   * set user who launched wizard
   * @returns {$.ajax}
   */
  setUser: function setUser(controllerName) {
    return this.postUserPref(this.get('PREF_KEY'), {
      userName: App.router.get('loginName'),
      controllerName: controllerName
    });
  },

  /**
   * reset user who launched wizard
   * @returns {$.ajax}
   */
  resetUser: function resetUser() {
    return this.postUserPref(this.get('PREF_KEY'), null);
  },

  /**
   * get user who launched wizard
   * @returns {$.ajax}
   */
  getUser: function getUser() {
    return this.getUserPref(this.get('PREF_KEY'));
  },

  getUserPrefSuccessCallback: function getUserPrefSuccessCallback(data) {
    if (Em.isNone(data)) {
      this.set('wizardUser', null);
      this.set('controllerName', null);
    } else {
      this.set('wizardUser', data.userName);
      this.set('controllerName', data.controllerName);
    }
  },

  getUserPrefErrorCallback: function getUserPrefErrorCallback() {
    this.resetUser();
  }
});

});

require.register("controllers/installer", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var stringUtils = require('utils/string_utils');
var validator = require('utils/validator');

App.InstallerController = App.WizardController.extend(App.Persist, {

  name: 'installerController',

  isCheckInProgress: false,

  totalSteps: 11,

  isInstallerWizard: true,

  content: Em.Object.create({
    cluster: null,
    installOptions: null,
    hosts: null,
    services: null,
    slaveComponentHosts: null,
    masterComponentHosts: null,
    serviceConfigProperties: null,
    advancedServiceConfig: null,
    configGroups: [],
    slaveGroupProperties: null,
    stacks: null,
    clients: [],
    // list of components, that was added from configs page via AssignMasterOnStep7Controller
    componentsFromConfigs: [],
    /**
     * recommendations for host groups loaded from server
     */
    recommendations: null,
    /**
     * recommendationsHostGroups - current component assignment after 5 and 6 steps, 
     * or adding hiveserver2 interactive on "configure services" page
     * (uses for host groups validation and to load recommended configs)
     */
    recommendationsHostGroups: null,
    controllerName: 'installerController'
  }),

  /**
   * Wizard properties in local storage, which should be cleaned right after wizard has been finished
   */
  dbPropertiesToClean: ['service', 'hosts', 'masterComponentHosts', 'slaveComponentHosts', 'cluster', 'allHostNames', 'installOptions', 'allHostNamesPattern', 'serviceComponents', 'clientInfo', 'selectedServiceNames', 'serviceConfigGroups', 'serviceConfigProperties', 'fileNamesToUpdate', 'bootStatus', 'stacksVersions', 'currentStep', 'serviceInfo', 'hostInfo', 'recommendations', 'recommendationsHostGroups', 'recommendationsConfigs', 'componentsFromConfigs', 'operatingSystems', 'repositories'],

  init: function init() {
    this._super();
    this.get('isStepDisabled').setEach('value', true);
    this.get('isStepDisabled').pushObject(Ember.Object.create({
      step: 0,
      value: true
    }));
  },
  /**
   * redefined connectOutlet method to avoid view loading by unauthorized user
   * @param view
   * @param content
   */
  connectOutlet: function connectOutlet(view, content) {
    if (App.db.getAuthenticated()) {
      this._super(view, content);
    }
  },

  getCluster: function getCluster() {
    return jQuery.extend({}, this.get('clusterStatusTemplate'));
  },

  getHosts: function getHosts() {
    return [];
  },

  /**
   * Remove host from model. Used at <code>Confirm hosts(step2)</code> step
   * @param hosts Array of hosts, which we want to delete
   */
  removeHosts: function removeHosts(hosts) {
    var dbHosts = this.getDBProperty('hosts');
    hosts.forEach(function (_hostInfo) {
      var host = _hostInfo.name;
      delete dbHosts[host];
    });
    this.setDBProperty('hosts', dbHosts);
  },

  cancelInstall: function cancelInstall() {
    return App.showConfirmationPopup(function () {
      App.router.get('applicationController').goToAdminView();
    });
  },

  /**
   * Load services data. Will be used at <code>Select services(step4)</code> step
   */
  loadServices: function loadServices() {
    var dfd = $.Deferred();
    var self = this;
    var stackServices = App.StackService.find().mapProperty('serviceName');
    if (!(stackServices.length && App.StackService.find().objectAt(0).get('stackVersion') === App.get('currentStackVersionNumber'))) {
      this.loadServiceComponents().complete(function () {
        self.set('content.services', App.StackService.find().forEach(function (item) {
          // user the service version from VersionDefinition
          var serviceInStack = App.Stack.find().findProperty('isSelected').get('stackServices').findProperty('name', item.get('serviceName'));
          var serviceVersionDisplay = serviceInStack ? serviceInStack.get('latestVersion') : item.get('serviceVersion');
          item.set('serviceVersionDisplay', serviceVersionDisplay);
        }));
        dfd.resolve();
      });
    } else {
      dfd.resolve();
    }
    return dfd.promise();
  },

  /**
   * total set of hosts registered to cluster, analog of App.Host model,
   * used in Installer wizard until hosts are installed
   */
  allHosts: function () {
    var rawHosts = this.get('content.hosts');
    var masterComponents = this.get('content.masterComponentHosts');
    var slaveComponents = this.get('content.slaveComponentHosts');
    var hosts = [];
    masterComponents.forEach(function (component) {
      var host = rawHosts[component.hostName];
      if (host.hostComponents) {
        host.hostComponents.push(Em.Object.create({
          componentName: component.component,
          displayName: component.display_name
        }));
      } else {
        rawHosts[component.hostName].hostComponents = [Em.Object.create({
          componentName: component.component,
          displayName: component.display_name
        })];
      }
    });
    slaveComponents.forEach(function (component) {
      component.hosts.forEach(function (rawHost) {
        var host = rawHosts[rawHost.hostName];
        if (host.hostComponents) {
          host.hostComponents.push(Em.Object.create({
            componentName: component.componentName,
            displayName: component.displayName
          }));
        } else {
          rawHosts[rawHost.hostName].hostComponents = [Em.Object.create({
            componentName: component.componentName,
            displayName: component.displayName
          })];
        }
      });
    });

    for (var hostName in rawHosts) {
      var host = rawHosts[hostName];
      hosts.pushObject(Em.Object.create({
        id: host.name,
        hostName: host.name,
        hostComponents: host.hostComponents || []
      }));
    }
    return hosts;
  }.property('content.hosts'),

  stacks: [],

  /**
   * stack names used as auxiliary data to query stacks by name
   */
  stackNames: [],

  /**
   * Load stacks data from server or take exist data from in memory variable {{content.stacks}}
   * The series of API calls will be called  When landing first time on Select Stacks page
   * or on hitting refresh post select stacks page in installer wizard
   */
  loadStacks: function loadStacks() {
    var stacks = this.get('content.stacks');
    var dfd = $.Deferred();
    if (stacks && stacks.get('length')) {
      App.set('currentStackVersion', App.Stack.find().findProperty('isSelected').get('stackNameVersion'));
      dfd.resolve(true);
    } else {
      App.ajax.send({
        name: 'wizard.stacks',
        sender: this,
        success: 'loadStacksSuccessCallback',
        error: 'loadStacksErrorCallback'
      }).complete(function () {
        dfd.resolve(false);
      });
    }
    return dfd.promise();
  },

  /**
   * Send queries to load versions for each stack
   */
  loadStacksSuccessCallback: function loadStacksSuccessCallback(data) {
    this.get('stacks').clear();
    this.set('stackNames', data.items.mapProperty('Stacks.stack_name'));
  },

  /**
   * onError callback for loading stacks data
   */
  loadStacksErrorCallback: function loadStacksErrorCallback() {},

  /**
   * query every stack names from server
   * @return {Array}
   */
  loadStacksVersions: function loadStacksVersions() {
    var requests = [];
    var dfd = $.Deferred();
    this.get('stackNames').forEach(function (stackName) {
      requests.push(App.ajax.send({
        name: 'wizard.stacks_versions_definitions',
        sender: this,
        data: {
          stackName: stackName,
          dfd: dfd
        },
        success: 'loadStacksVersionsDefinitionsSuccessCallback',
        error: 'loadStacksVersionsErrorCallback'
      }));
    }, this);
    this.set('loadStacksRequestsCounter', requests.length);
    return dfd.promise();
  },

  /**
   * Counter for counting number of successful requests to load stack versions
   */
  loadStacksRequestsCounter: 0,

  /**
   * Parse loaded data and create array of stacks objects
   */
  loadStacksVersionsDefinitionsSuccessCallback: function loadStacksVersionsDefinitionsSuccessCallback(data, opt, params) {
    var stacks = App.db.getStacks();
    var oses = App.db.getOses();
    var repos = App.db.getRepos();
    this.decrementProperty('loadStacksRequestsCounter');
    var isStacksExistInDb = stacks && stacks.length;
    if (isStacksExistInDb) {
      stacks.forEach(function (_stack) {
        var stack = data.items.findProperty('VersionDefinition.id', _stack.id);
        if (stack) {
          stack.VersionDefinition.is_selected = _stack.is_selected;
        }
      }, this);
    }
    if (!data.items || !data.items.length) {
      this.setSelected(true, params.dfd);
    }
    data.items.sortProperty('VersionDefinition.stack_version').reverse().forEach(function (versionDefinition) {
      // to display repos panel, should map all available operating systems including empty ones
      var stackInfo = {};
      stackInfo.isStacksExistInDb = isStacksExistInDb;
      stackInfo.stacks = stacks;
      stackInfo.oses = oses;
      stackInfo.repos = repos;
      this.getSupportedOSList(versionDefinition, stackInfo, params.dfd);
    }, this);
  },

  mergeChanges: function mergeChanges(repos, oses, stacks) {
    var _repos = repos || [];
    var _oses = oses || [];
    var _stacks = stacks || [];
    _repos.forEach(function (repo) {
      if (App.Repository.find(repo.id).get('isLoaded')) {
        App.Repository.find(repo.id).set('baseUrl', repo.base_url);
      }
    });
    _oses.forEach(function (os) {
      if (App.OperatingSystem.find().findProperty('id', os.id)) {
        App.OperatingSystem.find().findProperty('id', os.id).set('isSelected', os.is_selected);
      }
    });
    //should delete the record on going to step 2, on going back to step 1, still need the record
    if (App.router.get('currentState.name') != "step1") {
      App.OperatingSystem.find().filterProperty('isSelected', false).forEach(function (os) {
        App.stackMapper.deleteRecord(os);
      });
    }
    _stacks.forEach(function (_stack) {
      var stack = App.Stack.find().findProperty('id', _stack.id);
      if (stack) {
        stack.set('useRedhatSatellite', _stack.use_redhat_satellite);
      }
    });
  },

  setSelected: function setSelected(isStacksExistInDb, dfd) {
    if (!isStacksExistInDb) {
      var stacks = App.Stack.find();
      stacks.setEach('isSelected', false);
      stacks.sortProperty('id').set('lastObject.isSelected', true);
    }
    this.set('content.stacks', App.Stack.find());
    var selected = App.Stack.find().findProperty('isSelected');
    App.set('currentStackVersion', selected ? selected.get('stackNameVersion') : null);
    dfd.resolve();
  },

  /**
   * Get the the repo version (to install) info, this data will be POST
   * @method startDeploy
   */
  getSelectedRepoVersionData: function getSelectedRepoVersionData() {
    var vdfData = App.db.getLocalRepoVDFData();
    var selectedStack = App.Stack.find().findProperty('isSelected', true);
    var isXMLdata = false;
    var data = {};
    if (selectedStack && selectedStack.get('showAvailable')) {
      //meaning user selected a public repo
      data = {
        "VersionDefinition": {
          "available": selectedStack.get('id')
        }
      };
      isXMLdata = false;
    } else if (vdfData && validator.isValidURL(vdfData)) {
      // meaning user uploaded a VDF via entering URL
      data = {
        "VersionDefinition": {
          "version_url": vdfData
        }
      };
      isXMLdata = false;
    } else if (vdfData) {
      // meaning user uploaded a local VDF.xml file
      isXMLdata = true;
      data = vdfData;
    } else {
      return null;
    }
    return {
      isXMLdata: isXMLdata,
      data: data
    };
  },

  /**
   * onError callback for loading stacks data
   */
  loadStacksVersionsErrorCallback: function loadStacksVersionsErrorCallback() {},

  /**
   * check server version and web client version
   */
  checkServerClientVersion: function checkServerClientVersion() {
    var dfd = $.Deferred();
    var self = this;
    self.getServerVersion().done(function () {
      dfd.resolve();
    });
    return dfd.promise();
  },
  getServerVersion: function getServerVersion() {
    return App.ajax.send({
      name: 'ambari.service',
      sender: this,
      data: {
        fields: '?fields=RootServiceComponents/component_version,RootServiceComponents/properties/server.os_family&minimal_response=true'
      },
      success: 'getServerVersionSuccessCallback',
      error: 'getServerVersionErrorCallback'
    });
  },
  getServerVersionSuccessCallback: function getServerVersionSuccessCallback(data) {
    var clientVersion = App.get('version');
    var serverVersion = data.RootServiceComponents.component_version.toString();
    this.set('ambariServerVersion', serverVersion);
    if (clientVersion) {
      this.set('versionConflictAlertBody', Em.I18n.t('app.versionMismatchAlert.body').format(serverVersion, clientVersion));
      this.set('isServerClientVersionMismatch', clientVersion !== serverVersion);
    } else {
      this.set('isServerClientVersionMismatch', false);
    }
    App.set('isManagedMySQLForHiveEnabled', App.config.isManagedMySQLForHiveAllowed(data.RootServiceComponents.properties['server.os_family']));
  },
  getServerVersionErrorCallback: function getServerVersionErrorCallback() {},

  /**
   * set stacks from server to content and local DB
   */
  setStacks: function setStacks() {
    App.db.setStacks(App.Stack.find().slice());
    this.set('content.stacks', App.Stack.find());
    App.db.setOses(App.OperatingSystem.find().slice());
    App.db.setRepos(App.Repository.find().slice());
  },

  /**
   * Save data to model
   * @param stepController App.WizardStep4Controller
   */
  saveServices: function saveServices(stepController) {
    var selectedServiceNames = [];
    var installedServiceNames = [];
    stepController.filterProperty('isSelected').forEach(function (item) {
      selectedServiceNames.push(item.get('serviceName'));
    });
    stepController.filterProperty('isInstalled').forEach(function (item) {
      installedServiceNames.push(item.get('serviceName'));
    });
    this.set('content.services', App.StackService.find());
    this.set('content.selectedServiceNames', selectedServiceNames);
    this.set('content.installedServiceNames', installedServiceNames);
    this.setDBProperties({
      selectedServiceNames: selectedServiceNames,
      installedServiceNames: installedServiceNames
    });
  },

  /**
   * Save Master Component Hosts data to Main Controller
   * @param stepController App.WizardStep5Controller
   * @param  skip  {Boolean}
   */
  saveMasterComponentHosts: function saveMasterComponentHosts(stepController, skip) {
    var obj = stepController.get('selectedServicesMasters'),
        hosts = this.getDBProperty('hosts');

    var masterComponentHosts = [];
    obj.forEach(function (_component) {
      masterComponentHosts.push({
        display_name: _component.get('display_name'),
        component: _component.get('component_name'),
        serviceId: _component.get('serviceId'),
        isInstalled: false,
        host_id: hosts[_component.get('selectedHost')].id
      });
    });

    this.set('content.masterComponentHosts', masterComponentHosts);
    if (!skip) {
      this.setDBProperty('masterComponentHosts', masterComponentHosts);
    }
  },

  /**
   * Load master component hosts data for using in required step controllers
   * @param inMemory {Boolean}: Load master component hosts from memory
   */
  loadMasterComponentHosts: function loadMasterComponentHosts(inMemory) {
    var props = this.getDBProperties(['masterComponentHosts', 'hosts']);
    var masterComponentHosts = !!inMemory ? this.get("content.masterComponentHosts") : props.masterComponentHosts,
        hosts = props.hosts || {},
        hostNames = Em.keys(hosts);
    if (Em.isNone(masterComponentHosts)) {
      masterComponentHosts = [];
    } else {
      masterComponentHosts.forEach(function (component) {
        for (var i = 0; i < hostNames.length; i++) {
          if (hosts[hostNames[i]].id === component.host_id) {
            component.hostName = hostNames[i];
            break;
          }
        }
      });
    }
    this.set("content.masterComponentHosts", masterComponentHosts);
  },

  loadCurrentHostGroups: function loadCurrentHostGroups() {
    this.set("content.recommendationsHostGroups", this.getDBProperty('recommendationsHostGroups'));
  },

  loadRecommendationsConfigs: function loadRecommendationsConfigs() {
    App.router.set("wizardStep7Controller.recommendationsConfigs", this.getDBProperty('recommendationsConfigs'));
  },

  /**
   * Load master component hosts data for using in required step controllers
   */
  loadSlaveComponentHosts: function loadSlaveComponentHosts() {
    var props = this.getDBProperties(['slaveComponentHosts', 'hosts']);
    var slaveComponentHosts = props.slaveComponentHosts,
        hosts = props.hosts || {},
        hostNames = Em.keys(hosts);
    if (!Em.isNone(slaveComponentHosts)) {
      slaveComponentHosts.forEach(function (component) {
        component.hosts.forEach(function (host) {
          for (var i = 0; i < hostNames.length; i++) {
            if (hosts[hostNames[i]].id === host.host_id) {
              host.hostName = hostNames[i];
              break;
            }
          }
        });
      });
    }
    this.set("content.slaveComponentHosts", slaveComponentHosts);
  },

  /**
   * Generate clients list for selected services and save it to model
   * @param stepController step4WizardController
   */
  saveClients: function saveClients(stepController) {
    var clients = [];
    stepController.get('content').filterProperty('isSelected', true).forEach(function (_service) {
      var client = _service.get('serviceComponents').filterProperty('isClient', true);
      client.forEach(function (clientComponent) {
        clients.pushObject({
          component_name: clientComponent.get('componentName'),
          display_name: clientComponent.get('displayName'),
          isInstalled: false
        });
      }, this);
    }, this);
    this.setDBProperty('clientInfo', clients);
    this.set('content.clients', clients);
  },

  /*
   * Post version definition file (.xml) to server, DRY_RUN = TRUE
   */
  postVersionDefinitionFile: function postVersionDefinitionFile(isXMLdata, data) {
    var dfd = $.Deferred();
    var name = isXMLdata ? 'wizard.step1.post_version_definition_file.xml' : 'wizard.step1.post_version_definition_file.url';

    App.ajax.send({
      name: name,
      sender: this,
      data: {
        dfd: dfd,
        data: data
      },
      success: 'postVersionDefinitionFileSuccessCallback',
      error: 'postVersionDefinitionFileErrorCallback'
    });
    return dfd.promise();
  },

  /**
   * onSuccess callback for postVersionDefinitionFile.
   */
  postVersionDefinitionFileSuccessCallback: function postVersionDefinitionFileSuccessCallback(_data, request, dataInfo) {
    if (_data.resources.length && _data.resources[0].VersionDefinition) {
      var data = _data.resources[0];
      // load the data info to display for details and contents panel
      data.VersionDefinition.id = Em.get(dataInfo, 'data.VersionDefinition.available') || data.VersionDefinition.id;
      var response = {
        id: data.VersionDefinition.id,
        stackVersion: data.VersionDefinition.stack_version,
        stackName: data.VersionDefinition.stack_name,
        type: data.VersionDefinition.type,
        stackNameVersion: data.VersionDefinition.stack_name + '-' + data.VersionDefinition.stack_version, /// HDP-2.3
        actualVersion: data.VersionDefinition.repository_version, /// 2.3.4.0-3846
        version: data.VersionDefinition.release ? data.VersionDefinition.release.version : null, /// 2.3.4.0
        releaseNotes: data.VersionDefinition.release ? data.VersionDefinition.release.notes : null,
        displayName: data.VersionDefinition.release ? data.VersionDefinition.stack_name + '-' + data.VersionDefinition.release.version : data.VersionDefinition.stack_name + '-' + data.VersionDefinition.repository_version, //HDP-2.3.4.0
        repoVersionFullName: data.VersionDefinition.stack_name + '-' + data.VersionDefinition.repository_version,
        osList: data.operating_systems,
        updateObj: data
      };
      var services = [];
      data.VersionDefinition.services.forEach(function (service) {
        services.push({
          name: service.name,
          version: service.versions[0].version,
          components: service.versions[0].components
        });
      });
      response.services = services;

      // to display repos panel, should map all available operating systems including empty ones
      var stackInfo = {};
      stackInfo.dfd = dataInfo.dfd;
      stackInfo.response = response;
      this.incrementProperty('loadStacksRequestsCounter');
      this.getSupportedOSListSuccessCallback(data, null, {
        stackName: data.VersionDefinition.stack_name,
        stackVersion: data.VersionDefinition.stack_version,
        versionDefinition: data,
        stackInfo: stackInfo
      });
    }
  },

  /*
   * Post version definition file (.xml) to server in step 8
   */
  postVersionDefinitionFileStep8: function postVersionDefinitionFileStep8(isXMLdata, data) {
    var dfd = $.Deferred();
    var name = isXMLdata == true ? 'wizard.step8.post_version_definition_file.xml' : 'wizard.step8.post_version_definition_file';
    App.ajax.send({
      name: name,
      sender: this,
      data: {
        dfd: dfd,
        data: data
      },
      success: 'postVersionDefinitionFileStep8SuccessCallback',
      error: 'postVersionDefinitionFileErrorCallback'
    });
    return dfd.promise();
  },
  /**
   * onSuccess callback for postVersionDefinitionFile.
   */
  postVersionDefinitionFileStep8SuccessCallback: function postVersionDefinitionFileStep8SuccessCallback(response, request, data) {
    if (response.resources.length && response.resources[0].VersionDefinition) {
      data.dfd.resolve({
        stackName: response.resources[0].VersionDefinition.stack_name,
        id: response.resources[0].VersionDefinition.id,
        stackVersion: response.resources[0].VersionDefinition.stack_version
      });
    }
  },

  /**
   * onError callback for postVersionDefinitionFile.
   */
  postVersionDefinitionFileErrorCallback: function postVersionDefinitionFileErrorCallback(request, ajaxOptions, error, data, params) {
    params.dfd.reject(data);
    var header = Em.I18n.t('installer.step1.useLocalRepo.uploadFile.error.title');
    var body = '';
    if (request && request.responseText) {
      try {
        var json = $.parseJSON(request.responseText);
        body = json.message;
      } catch (err) {}
    }
    App.db.setLocalRepoVDFData(undefined);
    App.showAlertPopup(header, body);
  },

  getSupportedOSList: function getSupportedOSList(versionDefinition, stackInfo, dfd) {
    this.incrementProperty('loadStacksRequestsCounter');
    return App.ajax.send({
      name: 'wizard.step1.get_supported_os_types',
      sender: this,
      data: {
        stackName: versionDefinition.VersionDefinition.stack_name,
        stackVersion: versionDefinition.VersionDefinition.stack_version,
        versionDefinition: versionDefinition,
        stackInfo: stackInfo,
        dfd: dfd
      },
      success: 'getSupportedOSListSuccessCallback',
      error: 'getSupportedOSListErrorCallback'
    });
  },

  /**
   * onSuccess callback for getSupportedOSList.
   */
  getSupportedOSListSuccessCallback: function getSupportedOSListSuccessCallback(response, request, data) {
    var self = this;
    var stack_default = data.versionDefinition.VersionDefinition.stack_default;
    var existedOS = data.versionDefinition.operating_systems;
    var existedMap = {};
    existedOS.map(function (existedOS) {
      existedOS.isSelected = true;
      existedMap[existedOS.OperatingSystems.os_type] = existedOS;
    });
    response.operating_systems.forEach(function (supportedOS) {
      if (!existedMap[supportedOS.OperatingSystems.os_type]) {
        supportedOS.isSelected = false;
        existedOS.push(supportedOS);
      } else {
        existedMap[supportedOS.OperatingSystems.os_type].repositories.forEach(function (repo) {
          supportedOS.repositories.forEach(function (supportedRepo) {
            if (supportedRepo.Repositories.repo_id == repo.Repositories.repo_id) {
              repo.Repositories.components = supportedRepo.Repositories.components;
              repo.Repositories.distribution = supportedRepo.Repositories.distribution;
            }
          });
        });
      }
    });

    App.stackMapper.map(data.versionDefinition);

    if (!this.decrementProperty('loadStacksRequestsCounter')) {
      if (data.stackInfo.dfd) {
        data.stackInfo.dfd.resolve(data.stackInfo.response);
      } else {
        var versionData = this.getSelectedRepoVersionData();
        if (versionData) {
          this.postVersionDefinitionFile(versionData.isXMLdata, versionData.data).done(function (versionInfo) {
            self.mergeChanges(data.stackInfo.repos, data.stackInfo.oses, data.stackInfo.stacks);
            App.Stack.find().setEach('isSelected', false);
            var stackId = Em.get(versionData, 'data.VersionDefinition.available') || versionInfo.stackNameVersion + "-" + versionInfo.actualVersion;
            App.Stack.find().findProperty('id', stackId).set('isSelected', true);
            self.setSelected(data.stackInfo.isStacksExistInDb, data.dfd);
          }).fail(function () {
            self.setSelected(data.stackInfo.isStacksExistInDb, data.dfd);
          });
        } else {
          this.setSelected(data.stackInfo.isStacksExistInDb, data.dfd);
        }
      }
    }
  },

  /**
   * onError callback for getSupportedOSList
   */
  getSupportedOSListErrorCallback: function getSupportedOSListErrorCallback(request, ajaxOptions, error, data, params) {
    var header = Em.I18n.t('installer.step1.useLocalRepo.getSurpottedOs.error.title');
    var body = "";
    if (request && request.responseText) {
      try {
        var json = $.parseJSON(request.responseText);
        body = json.message;
      } catch (err) {}
    }
    App.showAlertPopup(header, body);
  },

  updateRepoOSInfo: function updateRepoOSInfo(repoToUpdate, repo) {
    var deferred = $.Deferred();
    var repoVersion = this.prepareRepoForSaving(repo);
    App.ajax.send({
      name: 'admin.stack_versions.edit.repo',
      sender: this,
      data: {
        stackName: repoToUpdate.stackName,
        stackVersion: repoToUpdate.stackVersion,
        repoVersionId: repoToUpdate.id,
        repoVersion: repoVersion
      }
    }).success(function () {
      deferred.resolve([]);
    }).error(function () {
      deferred.resolve([]);
    });
    return deferred.promise();
  },

  /**
   * transform repo data into json for
   * saving changes to repository version
   * @param {Em.Object} repo
   * @returns {{operating_systems: Array}}
   */
  prepareRepoForSaving: function prepareRepoForSaving(repo) {
    var repoVersion = { "operating_systems": [] };
    var ambariManagedRepositories = !repo.get('useRedhatSatellite');
    var k = 0;
    repo.get('operatingSystems').forEach(function (os) {
      if (os.get('isSelected')) {
        repoVersion.operating_systems.push({
          "OperatingSystems": {
            "os_type": os.get("osType"),
            "ambari_managed_repositories": ambariManagedRepositories
          },
          "repositories": []
        });
        os.get('repositories').forEach(function (repository) {
          if (!(repository.get('isGPL') && _.isEmpty(repository.get('baseUrl')))) {
            repoVersion.operating_systems[k].repositories.push({
              "Repositories": {
                "base_url": repository.get('baseUrl'),
                "repo_id": repository.get('repoId'),
                "repo_name": repository.get('repoName'),
                "components": repository.get('components'),
                "tags": repository.get('tags'),
                "distribution": repository.get('distribution'),
                "applicable_services": repository.get('applicable_services')
              }
            });
          }
        });
        k++;
      }
    });
    return repoVersion;
  },

  /**
   * Check validation of the customized local urls
   */
  checkRepoURL: function checkRepoURL(wizardStep1Controller) {
    var selectedStack = this.get('content.stacks').findProperty('isSelected', true);
    selectedStack.set('reload', true);
    var nameVersionCombo = selectedStack.get('stackNameVersion');
    var stackName = nameVersionCombo.split('-')[0];
    var stackVersion = nameVersionCombo.split('-')[1];
    var dfd = $.Deferred();
    if (selectedStack && selectedStack.get('operatingSystems')) {
      this.set('validationCnt', selectedStack.get('operatingSystems').filterProperty('isSelected').filterProperty('isEmpty', false).map(function (os) {
        return os.get('repositories').filterProperty('showRepo', true).length;
      }).reduce(Em.sum, 0));
      var verifyBaseUrl = !wizardStep1Controller.get('skipValidationChecked') && !wizardStep1Controller.get('selectedStack.useRedhatSatellite');
      if (!verifyBaseUrl) {
        dfd.resolve();
      }
      //for redhat satellite/spacewalk the os urls will be empty
      var useRedhatSatellite = wizardStep1Controller.get('selectedStack.useRedhatSatellite');
      selectedStack.get('operatingSystems').forEach(function (os) {
        if (os.get('isSelected') && (useRedhatSatellite || !os.get('isEmpty'))) {
          os.get('repositories').forEach(function (repo) {
            if (repo.get('showRepo')) {
              repo.setProperties({
                errorTitle: '',
                errorContent: '',
                validation: 'INPROGRESS'
              });
              this.set('content.isCheckInProgress', true);
              App.ajax.send({
                name: 'wizard.advanced_repositories.valid_url',
                sender: this,
                data: {
                  stackName: stackName,
                  stackVersion: stackVersion,
                  repoId: repo.get('repoId'),
                  osType: os.get('osType'),
                  osId: os.get('id'),
                  dfd: dfd,
                  data: {
                    'Repositories': {
                      'base_url': repo.get('baseUrl'),
                      'repo_name': repo.get('repoName'),
                      "verify_base_url": verifyBaseUrl
                    }
                  }
                },
                success: 'checkRepoURLSuccessCallback',
                error: 'checkRepoURLErrorCallback'
              });
            }
          }, this);
        } else if (os.get('isSelected') && os.get('isEmpty')) {
          os.set('isSelected', false);
        }
      }, this);
    }
    return dfd.promise();
  },
  /**
   * onSuccess callback for check Repo URL.
   */
  checkRepoURLSuccessCallback: function checkRepoURLSuccessCallback(response, request, data) {
    var selectedStack = this.get('content.stacks').findProperty('isSelected');
    if (selectedStack && selectedStack.get('operatingSystems')) {
      var os = selectedStack.get('operatingSystems').findProperty('id', data.osId);
      var repo = os.get('repositories').findProperty('repoId', data.repoId);
      if (repo) {
        repo.set('validation', 'OK');
      }
    }
    this.set('validationCnt', this.get('validationCnt') - 1);
    if (!this.get('validationCnt')) {
      this.set('content.isCheckInProgress', false);
      data.dfd.resolve();
    }
  },

  /**
   * onError callback for check Repo URL.
   */
  checkRepoURLErrorCallback: function checkRepoURLErrorCallback(request, ajaxOptions, error, data, params) {
    var selectedStack = this.get('content.stacks').findProperty('isSelected', true);
    if (selectedStack && selectedStack.get('operatingSystems')) {
      var os = selectedStack.get('operatingSystems').findProperty('id', params.osId);
      var repo = os.get('repositories').findProperty('repoId', params.repoId);
      if (repo) {
        var title = Ember.Handlebars.Utils.escapeExpression(request.status + ":" + request.statusText);
        var content = Ember.Handlebars.Utils.escapeExpression($.parseJSON(request.responseText) ? $.parseJSON(request.responseText).message : "");
        repo.setProperties({
          validation: 'INVALID',
          errorTitle: title,
          errorContent: content
        });
      }
    }
    this.set('content.isCheckInProgress', false);
    params.dfd.reject();
  },

  loadMap: {
    '0': [{
      type: 'sync',
      callback: function callback() {
        this.load('cluster');
      }
    }],
    '1': [{
      type: 'async',
      callback: function callback() {
        var dfd = $.Deferred();

        this.loadStacks().done(function (stacksLoaded) {
          App.router.get('clusterController').loadAmbariProperties().always(function () {
            dfd.resolve(stacksLoaded);
          });
        });

        return dfd.promise();
      }
    }, {
      type: 'async',
      callback: function callback(stacksLoaded) {
        var dfd = $.Deferred();

        if (!stacksLoaded) {
          this.loadStacksVersions().done(function () {
            dfd.resolve(true);
          });
        } else {
          dfd.resolve(stacksLoaded);
        }

        return dfd.promise();
      }
    }],
    '2': [{
      type: 'sync',
      callback: function callback() {
        this.load('installOptions');
      }
    }],
    '3': [{
      type: 'sync',
      callback: function callback() {
        this.loadConfirmedHosts();
      }
    }],
    '4': [{
      type: 'async',
      callback: function callback() {
        return this.loadServices();
      }
    }],
    '5': [{
      type: 'sync',
      callback: function callback() {
        this.setSkipSlavesStep(App.StackService.find().filterProperty('isSelected'), 6);
        this.loadMasterComponentHosts();
        this.loadConfirmedHosts();
        this.loadComponentsFromConfigs();
        this.loadRecommendations();
      }
    }],
    '6': [{
      type: 'sync',
      callback: function callback() {
        this.loadSlaveComponentHosts();
        this.loadClients();
        this.loadComponentsFromConfigs();
        this.loadRecommendations();
      }
    }],
    '7': [{
      type: 'async',
      callback: function callback() {
        var dfd = $.Deferred();
        var self = this;
        this.loadServiceConfigGroups();
        this.loadCurrentHostGroups();
        this.loadRecommendationsConfigs();
        this.loadComponentsFromConfigs();
        this.loadConfigThemes().then(function () {
          self.loadServiceConfigProperties();
          dfd.resolve();
        });
        return dfd.promise();
      }
    }]
  },

  clearConfigActionComponents: function clearConfigActionComponents() {
    var masterComponentHosts = this.get('content.masterComponentHosts');
    var componentsAddedFromConfigAction = this.get('content.componentsFromConfigs');

    if (componentsAddedFromConfigAction && componentsAddedFromConfigAction.length) {
      componentsAddedFromConfigAction.forEach(function (_masterComponent) {
        masterComponentHosts = masterComponentHosts.rejectProperty('component', _masterComponent);
      });
    }
    this.set('content.masterComponentHosts', masterComponentHosts);
    this.setDBProperty('masterComponentHosts', masterComponentHosts);
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.setCurrentStep('0');
    this.clearStorageData();
    this.clearServiceConfigProperties();
    App.router.get('userSettingsController').postUserPref('show_bg', true);
    App.themesMapper.resetModels();
  },

  /**
   * Save cluster provisioning state to the server
   * @param state cluster provisioning state
   */
  setClusterProvisioningState: function setClusterProvisioningState(state) {
    return App.ajax.send({
      name: 'cluster.save_provisioning_state',
      sender: this,
      data: {
        state: state
      }
    });
  },

  setStepsEnable: function () {
    for (var i = 0; i <= this.totalSteps; i++) {
      this.get('isStepDisabled').findProperty('step', i).set('value', i > this.get('currentStep'));
    }
  }.observes('currentStep'),

  setLowerStepsDisable: function setLowerStepsDisable(stepNo) {
    for (var i = 0; i < stepNo; i++) {
      var step = this.get('isStepDisabled').findProperty('step', i);
      step.set('value', true);
    }
  },

  /**
   * Compare jdk versions used for ambari and selected stack.
   * Validation check will fire only for non-custom jdk configuration.
   *
   * @param {Function} successCallback
   * @param {Function} failCallback
   */
  validateJDKVersion: function validateJDKVersion(successCallback, failCallback) {
    var selectedStack = App.Stack.find().findProperty('isSelected', true),
        currentJDKVersion = App.router.get('clusterController.ambariProperties')['java.version'],

    // use min as max, or max as min version, in case when some of them missed
    minJDKVersion = selectedStack.get('minJdkVersion') || selectedStack.get('maxJdkVersion'),
        maxJDKVersion = selectedStack.get('maxJdkVersion') || selectedStack.get('minJdkVersion'),
        t = Em.I18n.t,
        fCallback = failCallback || function () {},
        sCallback = successCallback || function () {};

    // Skip jdk check if min and max required version not set in stack definition.
    if (!minJDKVersion && !maxJDKVersion) {
      sCallback();
      return;
    }

    if (currentJDKVersion) {
      if (stringUtils.compareVersions(currentJDKVersion, minJDKVersion) < 0 || stringUtils.compareVersions(maxJDKVersion, currentJDKVersion) < 0) {
        // checks and process only minor part for now
        var versionDistance = parseInt(maxJDKVersion.split('.')[1], 10) - parseInt(minJDKVersion.split('.')[1], 10);
        var versionsList = [minJDKVersion];
        for (var i = 1; i < versionDistance + 1; i++) {
          versionsList.push("" + minJDKVersion.split('.')[0] + '.' + (+minJDKVersion.split('.')[1] + i));
        }
        var versionsString = stringUtils.getFormattedStringFromArray(versionsList, t('or'));
        var popupBody = t('popup.jdkValidation.body').format(selectedStack.get('stackName') + ' ' + selectedStack.get('stackVersion'), versionsString, currentJDKVersion);
        App.showConfirmationPopup(sCallback, popupBody, fCallback, t('popup.jdkValidation.header'), t('common.proceedAnyway'), 'danger');
        return;
      }
    }
    sCallback();
  }

});

});

require.register("controllers/login_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.LoginController = Em.Object.extend({

  name: 'loginController',

  loginName: '',
  password: '',

  errorMessage: '',

  isSubmitDisabled: false,

  submit: function submit(e) {
    this.set('errorMessage', '');
    this.set('isSubmitDisabled', true);
    App.get('router').login();
  },

  postLogin: function postLogin(isConnected, isAuthenticated, responseText) {
    var errorMessage = "";
    if (!isConnected) {
      this.set('errorMessage', responseText || Em.I18n.t('login.error.bad.connection'));
    } else if (!isAuthenticated) {
      if (responseText === "User is disabled") {
        errorMessage = Em.I18n.t('login.error.disabled');
      } else if (responseText === "Authentication required" || Em.isNone(responseText)) {
        errorMessage = Em.I18n.t('login.error.bad.credentials');
      } else {
        errorMessage = responseText;
      }
      this.set('errorMessage', errorMessage);
    }
    this.set('isSubmitDisabled', false);
  }

});

});

require.register("controllers/main", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
require('models/background_operation');

App.MainController = Em.Controller.extend({
  name: 'mainController',
  isUserActive: true,
  checkActivenessInterval: null,
  lastUserActiveTime: null,
  userTimeOut: 0,
  userTimeOutModal: null,

  updateTitle: function () {
    var name = App.router.get('clusterController.clusterName');
    if (App.router.get('clusterInstallCompleted')) {
      if (name && App.router.get('clusterController').get('isLoaded')) {
        name = name.length > 13 ? name.substr(0, 10) + "..." : name;
      } else {
        name = Em.I18n.t('common.loading');
      }
      $('title').text(Em.I18n.t('app.name.subtitle').format(name));
    }
  }.observes('App.router.clusterController.clusterName', 'App.router.clusterInstallCompleted', 'App.router.clusterController.isLoaded'),

  isClusterDataLoaded: Em.computed.alias('App.router.clusterController.isLoaded'),

  clusterDataLoadedPercent: Em.computed.alias('App.router.clusterController.clusterDataLoadedPercent'),
  /**
   * run all processes and cluster's data loading
   */
  initialize: function initialize() {
    // Since we use only defaultTransaction, we can stub <code>removeCleanRecords</code> method,
    // because it would remove from and add records to the same (default) transaction
    App.store.defaultTransaction.reopen({
      removeCleanRecords: Em.K
    });
    var startSubscription = App.router.get('updateController').startSubscriptions.bind(App.router.get('updateController'));
    App.StompClient.connect().done(startSubscription).fail(function (dfd) {
      dfd.always(startSubscription);
    });
    App.router.get('clusterController').loadClusterData();
  },

  dataLoading: function dataLoading() {
    var self = this;
    var dfd = $.Deferred();
    if (App.router.get('clusterController.isLoaded')) {
      dfd.resolve();
    } else {
      var interval = setInterval(function () {
        if (self.get('isClusterDataLoaded')) {
          dfd.resolve();
          clearInterval(interval);
        }
      }, 50);
    }
    return dfd.promise();
  },

  /**
   *
   * @param isLoaded {Boolean}
   * @param opts {Object}
   * {
   *   period {Number}
   * }
   * @return {*|{then}}
   */
  isLoading: function isLoading(isLoaded, opts) {
    var dfd = $.Deferred();
    var self = this;
    opts = opts || {};
    var period = opts.period || 20;
    if (this.get(isLoaded)) {
      dfd.resolve();
    } else {
      var interval = setInterval(function () {
        if (self.get(isLoaded)) {
          dfd.resolve();
          clearInterval(interval);
        }
      }, period);
    }
    return dfd.promise();
  },

  startPolling: function () {
    if (App.router.get('applicationController.isExistingClusterDataLoaded')) {
      App.router.get('updateController').set('isWorking', true);
    }
  }.observes('App.router.applicationController.isExistingClusterDataLoaded'),

  stopPolling: function stopPolling() {
    App.router.get('updateController').set('isWorking', false);
  },

  reloadTimeOut: null,

  pageReload: function () {

    clearTimeout(this.get("reloadTimeOut"));

    this.set('reloadTimeOut', setTimeout(function () {
      if (App.clusterStatus.get('isInstalled')) {
        location.reload();
      }
    }, App.pageReloadTime));
  }.observes("App.router.location.lastSetURL", "App.clusterStatus.isInstalled"),

  setAmbariServerVersion: function setAmbariServerVersion(data) {
    var clientVersion = App.get('version');
    var serverVersion = data.RootServiceComponents.component_version.toString();
    this.set('ambariServerVersion', serverVersion);
    if (clientVersion) {
      this.set('versionConflictAlertBody', Em.I18n.t('app.versionMismatchAlert.body').format(serverVersion, clientVersion));
      this.set('isServerClientVersionMismatch', clientVersion !== serverVersion);
    } else {
      this.set('isServerClientVersionMismatch', false);
    }
    App.set('isManagedMySQLForHiveEnabled', App.config.isManagedMySQLForHiveAllowed(data.RootServiceComponents.properties['server.os_family']));
  },

  monitorInactivity: function monitorInactivity() {
    var timeout = Number(App.router.get('clusterController.ambariProperties')['user.inactivity.timeout.default']);
    var readonly_timeout = Number(App.router.get('clusterController.ambariProperties')['user.inactivity.timeout.role.readonly.default']);
    var isAdmin = App.get('isAdmin');
    if (isAdmin && timeout > 0) {
      this.set('userTimeOut', timeout * 1000);
    } else if (!isAdmin && readonly_timeout > 0) {
      this.set('userTimeOut', readonly_timeout * 1000);
    }
    if (this.get('userTimeOut') > 0) {
      this.startMonitorInactivity();
    }
  },

  startMonitorInactivity: function startMonitorInactivity() {
    this.set('isUserActive', true);
    this.set('lastUserActiveTime', Date.now());

    this.rebindActivityEventMonitors();
    if (!this.get('checkActivenessInterval')) {
      this.set('checkActivenessInterval', window.setInterval(this.checkActiveness, 1000));
    }
  },

  /* this will be triggerred by user driven events: 'mousemove', 'keypress' and 'click' */
  keepActive: function keepActive() {
    var scope = App.router.get('mainController');
    if (scope.get('isUserActive')) {
      scope.set('lastUserActiveTime', Date.now());
    }
  },

  checkActiveness: function checkActiveness() {
    var scope = App.router.get('mainController');
    if (!scope.isOnWizard()) {
      var remainTime = scope.get('userTimeOut') - (Date.now() - scope.get('lastUserActiveTime'));
      if (remainTime < 0) {
        scope.set('isUserActive', false);
        scope.unbindActivityEventMonitors();
        clearInterval(scope.get('checkActivenessInterval'));
        App.router.logOff({});
      } else if (remainTime < App.inactivityRemainTime * 1000 && !scope.userTimeOutModal) {
        // show alert 60 seconds before logging user out
        scope.userTimeOutModal = App.ModalPopup.show({
          primary: Em.I18n.t('common.timeout.warning.popup.primary'),
          secondary: Em.I18n.t('common.timeout.warning.popup.secondary'),
          third: false,
          header: Em.I18n.t('common.timeout.warning.popup.header'),
          showCloseButton: false,
          bodyClass: Ember.View.extend({
            template: Ember.Handlebars.compile('<p>{{view.beforeMsg}}<b>{{view.remainTime}}</b>{{view.afterMsg}}</p>'),
            beforeMsg: Em.I18n.t('common.timeout.warning.popup.body.before'),
            afterMsg: Em.I18n.t('common.timeout.warning.popup.body.after'),
            remainTime: App.inactivityRemainTime,
            didInsertElement: function didInsertElement() {
              var self = this;
              setInterval(function () {
                self.countDown();
              }, 1000);
            },
            countDown: function countDown() {
              if (this.get('remainTime') > 0) {
                this.set('remainTime', this.get('remainTime') - 1);
              }
              if (this.get('remainTime') == 0) {
                App.router.logOff({});
              }
            }
          }),
          onPrimary: function onPrimary() {
            scope.keepActive();
            scope.userTimeOutModal.hide();
            delete scope.userTimeOutModal;
          },
          onSecondary: function onSecondary() {
            scope.userTimeOutModal.hide();
            delete scope.userTimeOutModal;
            App.router.logOff({});
          }
        });
      }
    }
  },

  rebindActivityEventMonitors: function rebindActivityEventMonitors() {
    this.unbindActivityEventMonitors();
    this.bindActivityEventMonitors();
  },

  isOnWizard: function isOnWizard() {
    var isWizard = window.location.href.indexOf('/step') != -1;
    var isUpgrade = window.location.href.indexOf('/stack/upgrade') != -1;
    return isWizard || isUpgrade;
  },

  bindActivityEventMonitors: function bindActivityEventMonitors() {
    $(window).bind('mousemove', this.keepActive);
    $(window).bind('keypress', this.keepActive);
    $(window).bind('click', this.keepActive);
    // iframes need to be monitored as well
    var iframes = $('iframe');
    if (iframes.length > 0) {
      for (var i = 0; i < iframes.length; i++) {
        var iframe = iframes[i];
        $(iframe.contentWindow).bind('mousemove', this.keepActive);
        $(iframe.contentWindow).bind('keypress', this.keepActive);
        $(iframe.contentWindow).bind('click', this.keepActive);
      }
    }
  },

  unbindActivityEventMonitors: function unbindActivityEventMonitors() {
    $(window).unbind('mousemove', this.keepActive);
    $(window).unbind('keypress', this.keepActive);
    $(window).unbind('click', this.keepActive);
    // iframes need to be monitored as well
    var iframes = $('iframe');
    if (iframes.length > 0) {
      for (var i = 0; i < iframes.length; i++) {
        var iframe = iframes[i];
        $(iframe.contentWindow).unbind('mousemove', this.keepActive);
        $(iframe.contentWindow).unbind('keypress', this.keepActive);
        $(iframe.contentWindow).unbind('click', this.keepActive);
      }
    }
  }
});

});

require.register("controllers/main/admin", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainAdminController = Em.Controller.extend({
  name: 'mainAdminController',

  /**
   * @type {string}
   * @default null
   */
  category: null,

  /**
   * Check if access page available.
   * Turn on if YARN service is installed with Application Timeline Server component and TEZ installed too.
   *
   * @type {Boolean}
   */
  isAccessAvailable: function () {
    var dependencies = {
      services: ['YARN', 'TEZ']
    };
    var serviceNames = App.Service.find().mapProperty('serviceName').filter(function (serviceName) {
      return dependencies.services.contains(serviceName);
    });
    var isAppTimelineServerAvailable = App.HostComponent.find().someProperty('componentName', 'APP_TIMELINE_SERVER');
    return dependencies.services.length === serviceNames.length && isAppTimelineServerAvailable;
  }.property()
});

});

require.register("controllers/main/admin/federation/step1_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var validator = require('utils/validator');

App.NameNodeFederationWizardStep1Controller = Em.Controller.extend({
  name: "nameNodeFederationWizardStep1Controller",

  existingNameServices: function () {
    var isMetricsLoaded = App.router.get('clusterController.isHostComponentMetricsLoaded');
    return isMetricsLoaded ? App.HDFSService.find().objectAt(0).get('masterComponentGroups').mapProperty('name') : [];
  }.property('App.router.clusterController.isHDFSNameSpacesLoaded'),

  existingNameServicesString: function () {
    return this.get('existingNameServices').join(', ');
  }.property('existingNameServices.length'),

  isNameServiceIdError: function () {
    var nameServiceId = this.get('content.nameServiceId');
    return !nameServiceId || this.get('existingNameServices').contains(nameServiceId) || !validator.isValidNameServiceId(nameServiceId);
  }.property('content.nameServiceId', 'existingNameServices.length'),

  next: function next() {
    if (!this.get('isNameServiceIdError')) {
      App.router.send('next');
    }
  }
});

});

require.register("controllers/main/admin/federation/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.NameNodeFederationWizardStep2Controller = Em.Controller.extend(App.AssignMasterComponents, {

  name: "nameNodeFederationWizardStep2Controller",

  useServerValidation: false,

  mastersToShow: ['NAMENODE'],

  mastersToAdd: ['NAMENODE', 'NAMENODE'],

  showCurrentPrefix: ['NAMENODE'],

  showAdditionalPrefix: ['NAMENODE'],

  showInstalledMastersFirst: true

});

});

require.register("controllers/main/admin/federation/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.NameNodeFederationWizardStep3Controller = Em.Controller.extend(App.BlueprintMixin, {
  name: "nameNodeFederationWizardStep3Controller",
  selectedService: null,
  stepConfigs: [],
  serverConfigData: {},
  once: false,
  isLoaded: false,
  isConfigsLoaded: false,
  versionLoaded: true,
  hideDependenciesInfoBar: true,

  /**
   * Map of sites and properties to delete
   * @type Object
   */
  configsToRemove: {
    'hdfs-site': ['dfs.namenode.shared.edits.dir', 'dfs.journalnode.edits.dir']
  },

  clearStep: function clearStep() {
    this.get('stepConfigs').clear();
    this.set('serverConfigData', {});
    this.set('isConfigsLoaded', false);
    this.set('isLoaded', false);
  },

  loadStep: function loadStep() {
    this.clearStep();
    this.loadConfigsTags();
  },

  loadConfigsTags: function loadConfigsTags() {
    return App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'onLoadConfigsTags'
    });
  },

  onLoadConfigsTags: function onLoadConfigsTags(data) {
    var servicesModel = App.Service.find();
    var urlParams = '(type=hdfs-site&tag=' + data.Clusters.desired_configs['hdfs-site'].tag + ')';
    if (servicesModel.someProperty('serviceName', 'RANGER')) {
      urlParams += '|(type=core-site&tag=' + data.Clusters.desired_configs['core-site'].tag + ')' + '|(type=ranger-tagsync-site&tag=' + data.Clusters.desired_configs['ranger-tagsync-site'].tag + ')' + '|(type=ranger-hdfs-security&tag=' + data.Clusters.desired_configs['ranger-hdfs-security'].tag + ')';
    }
    if (servicesModel.someProperty('serviceName', 'ACCUMULO')) {
      urlParams += '|(type=accumulo-site&tag=' + data.Clusters.desired_configs['accumulo-site'].tag + ')';
    }
    App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        urlParams: urlParams
      },
      success: 'onLoadConfigs'
    });
  },

  onLoadConfigs: function onLoadConfigs(data) {
    this.set('serverConfigData', data);
    this.set('isConfigsLoaded', true);
  },

  onLoad: function () {
    if (this.get('isConfigsLoaded') && App.router.get('clusterController.isHDFSNameSpacesLoaded')) {
      var federationConfig = $.extend(true, {}, require('data/configs/wizards/federation_properties').federationConfig);
      if (App.get('hasNameNodeFederation')) {
        federationConfig.configs = federationConfig.configs.rejectProperty('firstRun');
      }
      federationConfig.configs = this.tweakServiceConfigs(federationConfig.configs);
      this.removeConfigs(this.get('configsToRemove'), this.get('serverConfigData'));
      this.renderServiceConfigs(federationConfig);
      this.set('isLoaded', true);
    }
  }.observes('isConfigsLoaded', 'App.router.clusterController.isHDFSNameSpacesLoaded'),

  prepareDependencies: function prepareDependencies() {
    var ret = {};
    var configsFromServer = this.get('serverConfigData.items');
    var journalNodes = App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE');
    var nameNodes = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE');
    var hdfsSiteConfigs = configsFromServer.findProperty('type', 'hdfs-site').properties;
    var nameServices = App.HDFSService.find().objectAt(0).get('masterComponentGroups').mapProperty('name');
    ret.nameServicesList = nameServices.join(',');
    ret.nameservice1 = nameServices.find(function (ns) {
      return hdfsSiteConfigs['dfs.namenode.rpc-address.' + ns + '.nn1'];
    });
    ret.newNameservice = this.get('content.nameServiceId');
    ret.namenode1 = hdfsSiteConfigs['dfs.namenode.rpc-address.' + ret.nameservice1 + '.nn1'].split(':')[0];
    ret.namenode2 = hdfsSiteConfigs['dfs.namenode.rpc-address.' + ret.nameservice1 + '.nn2'].split(':')[0];
    ret.newNameNode1Index = 'nn' + (nameNodes.length - 1);
    ret.newNameNode2Index = 'nn' + nameNodes.length;
    ret.newNameNode1 = nameNodes.filterProperty('isInstalled', false).mapProperty('hostName')[0];
    ret.newNameNode2 = nameNodes.filterProperty('isInstalled', false).mapProperty('hostName')[1];
    ret.journalnodes = journalNodes.map(function (c) {
      return c.get('hostName') + ':8485';
    }).join(';');
    ret.clustername = App.get('clusterName');

    var dfsHttpA = hdfsSiteConfigs['dfs.namenode.http-address'];
    ret.nnHttpPort = dfsHttpA ? dfsHttpA.split(':')[1] : 50070;

    var dfsHttpsA = hdfsSiteConfigs['dfs.namenode.https-address'];
    ret.nnHttpsPort = dfsHttpsA ? dfsHttpsA.split(':')[1] : 50470;

    var dfsRpcA = hdfsSiteConfigs['dfs.namenode.rpc-address'];
    ret.nnRpcPort = dfsRpcA ? dfsRpcA.split(':')[1] : 8020;

    ret.journalnode_edits_dir = hdfsSiteConfigs['dfs.journalnode.edits.dir'];

    return ret;
  },

  tweakServiceConfigs: function tweakServiceConfigs(configs) {
    var servicesModel = App.Service.find();
    var dependencies = this.prepareDependencies();
    var nameServices = App.HDFSService.find().objectAt(0).get('masterComponentGroups').mapProperty('name');
    nameServices.push(dependencies.newNameservice);
    var result = [];
    var configsToRemove = [];
    var hdfsSiteConfigs = this.get('serverConfigData').items.findProperty('type', 'hdfs-site').properties;
    var wizardController = App.router.get(this.get('content.controllerName'));

    if (!hdfsSiteConfigs['dfs.namenode.servicerpc-address.' + dependencies.nameservice1 + '.nn1'] && !hdfsSiteConfigs['dfs.namenode.servicerpc-address.' + dependencies.nameservice1 + '.nn2']) {
      configsToRemove = configsToRemove.concat(['dfs.namenode.servicerpc-address.{{nameservice1}}.nn1', 'dfs.namenode.servicerpc-address.{{nameservice1}}.nn2', 'dfs.namenode.servicerpc-address.{{newNameservice}}.{{newNameNode1Index}}', 'dfs.namenode.servicerpc-address.{{newNameservice}}.{{newNameNode2Index}}']);
    }

    if (servicesModel.someProperty('serviceName', 'RANGER')) {
      var hdfsRangerConfigs = this.get('serverConfigData').items.findProperty('type', 'ranger-hdfs-security').properties;
      var reponamePrefix = hdfsRangerConfigs['ranger.plugin.hdfs.service.name'] === '{{repo_name}}' ? dependencies.clustername + '_hadoop_' : hdfsRangerConfigs['ranger.plugin.hdfs.service.name'] + '_';
      var coreSiteConfigs = this.get('serverConfigData').items.findProperty('type', 'core-site').properties;
      var defaultFSNS = coreSiteConfigs['fs.defaultFS'].split('hdfs://')[1];

      nameServices.forEach(function (nameService) {
        configs.push(this.createRangerServiceProperty(nameService, reponamePrefix, "ranger.tagsync.atlas.hdfs.instance." + App.get('clusterName') + ".nameservice." + nameService + ".ranger.service"));
        configs.push(this.createRangerServiceProperty(defaultFSNS, reponamePrefix, "ranger.tagsync.atlas.hdfs.instance." + App.get('clusterName') + ".ranger.service"));
      }, this);
    }

    if (servicesModel.someProperty('serviceName', 'ACCUMULO')) {
      var hdfsNameSpacesModel = App.HDFSService.find().objectAt(0).get('masterComponentGroups');
      var newNameSpace = this.get('content.nameServiceId');
      var volumesValue = nameServices.map(function (ns) {
        return 'hdfs://' + ns + '/apps/accumulo/data';
      }).join();
      var replacementsValue = nameServices.map(function (ns) {
        var hostName;
        if (ns === newNameSpace) {
          var hostNames = this.get('content.masterComponentHosts').filter(function (hc) {
            return hc.component === 'NAMENODE' && !hc.isInstalled;
          }).mapProperty('hostName');
          hostName = hostNames[0];
        } else {
          var nameSpaceObject = hdfsNameSpacesModel.findProperty('name', ns);
          hostName = nameSpaceObject && nameSpaceObject.hosts[0];
        }
        return 'hdfs://' + hostName + ':8020/apps/accumulo/data hdfs://' + ns + '/apps/accumulo/data';
      }, this).join();
      configs.push({
        name: 'instance.volumes',
        displayName: 'instance.volumes',
        isReconfigurable: false,
        value: volumesValue,
        recommendedValue: volumesValue,
        category: 'ACCUMULO',
        filename: 'accumulo-site',
        serviceName: 'MISC'
      }, {
        name: 'instance.volumes.replacements',
        displayName: 'instance.volumes.replacements',
        isReconfigurable: false,
        value: replacementsValue,
        recommendedValue: replacementsValue,
        category: 'ACCUMULO',
        filename: 'accumulo-site',
        serviceName: 'MISC'
      });
    }

    configs.forEach(function (config) {
      if (!configsToRemove.contains(config.name)) {
        config.isOverridable = false;
        config.name = wizardController.replaceDependencies(config.name, dependencies);
        config.displayName = wizardController.replaceDependencies(config.displayName, dependencies);
        config.value = wizardController.replaceDependencies(config.value, dependencies);
        config.recommendedValue = wizardController.replaceDependencies(config.recommendedValue, dependencies);
        result.push(config);
      }
    }, this);

    return result;
  },

  createRangerServiceProperty: function createRangerServiceProperty(nameservice, reponamePrefix, propertyName) {
    return {
      "name": propertyName,
      "displayName": propertyName,
      "isReconfigurable": false,
      "recommendedValue": reponamePrefix + nameservice,
      "value": reponamePrefix + nameservice,
      "category": "RANGER",
      "filename": "ranger-tagsync-site",
      "serviceName": 'MISC'
    };
  },

  removeConfigs: function removeConfigs(configsToRemove, configs) {
    Em.keys(configsToRemove).forEach(function (site) {
      var siteConfigs = configs.items.findProperty('type', site);
      if (siteConfigs) {
        configsToRemove[site].forEach(function (property) {
          delete siteConfigs.properties[property];
        });
      }
    });
    return configs;
  },

  renderServiceConfigs: function renderServiceConfigs(_serviceConfig) {
    var serviceConfig = App.ServiceConfig.create({
      serviceName: _serviceConfig.serviceName,
      displayName: _serviceConfig.displayName,
      configCategories: [],
      showConfig: true,
      configs: []
    });

    _serviceConfig.configCategories.forEach(function (_configCategory) {
      if (App.Service.find().someProperty('serviceName', _configCategory.name)) {
        serviceConfig.configCategories.pushObject(_configCategory);
      }
    }, this);

    this.loadComponentConfigs(_serviceConfig, serviceConfig);

    this.get('stepConfigs').pushObject(serviceConfig);
    this.set('selectedService', this.get('stepConfigs').objectAt(0));
    this.set('once', true);
  },

  /**
   * Load child components to service config object
   * @param _componentConfig
   * @param componentConfig
   */
  loadComponentConfigs: function loadComponentConfigs(_componentConfig, componentConfig) {
    _componentConfig.configs.forEach(function (_serviceConfigProperty) {
      var serviceConfigProperty = App.ServiceConfigProperty.create(_serviceConfigProperty);
      componentConfig.configs.pushObject(serviceConfigProperty);
      serviceConfigProperty.set('isEditable', serviceConfigProperty.get('isReconfigurable'));
    }, this);
  },

  isNextDisabled: function () {
    return !this.get('isLoaded') || this.get('isLoaded') && this.get('selectedService.configs').someProperty('isValid', false);
  }.property('selectedService.configs.@each.isValid', 'isLoaded')
});

});

require.register("controllers/main/admin/federation/step4_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.NameNodeFederationWizardStep4Controller = App.HighAvailabilityProgressPageController.extend(App.WizardEnableDone, {

  name: "nameNodeFederationWizardStep4Controller",

  commands: ['stopRequiredServices', 'reconfigureServices', 'installNameNode', 'installZKFC', 'startJournalNodes', 'startInfraSolr', 'startRangerAdmin', 'startRangerUsersync', 'startNameNodes', 'startZKFCs', 'formatNameNode', 'formatZKFC', 'startZKFC', 'startNameNode', 'bootstrapNameNode', 'startZKFC2', 'startNameNode2', 'restartAllServices'],

  tasksMessagesPrefix: 'admin.nameNodeFederation.wizard.step',

  initializeTasks: function initializeTasks() {
    this._super();
    this.removeUnneededTasks();
  },

  removeUnneededTasks: function removeUnneededTasks() {
    var installedServices = App.Service.find().mapProperty('serviceName');
    if (!installedServices.contains('RANGER')) {
      this.removeTasks(['startInfraSolr', 'startRangerAdmin', 'startRangerUsersync']);
    }
    if (!installedServices.contains('AMBARI_INFRA_SOLR')) {
      this.removeTasks(['startInfraSolr']);
    }
  },

  newNameNodeHosts: function () {
    return this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').filterProperty('isInstalled', false).mapProperty('hostName');
  }.property('content.masterComponentHosts.@each.hostName'),

  stopRequiredServices: function stopRequiredServices() {
    this.stopServices(["ZOOKEEPER"]);
  },

  reconfigureServices: function reconfigureServices() {
    var servicesModel = App.Service.find();
    var configs = [];
    var data = this.get('content.serviceConfigProperties');
    var note = Em.I18n.t('admin.nameNodeFederation.wizard,step4.save.configuration.note');
    configs.push({
      Clusters: {
        desired_config: this.reconfigureSites(['hdfs-site'], data, note)
      }
    });
    if (servicesModel.someProperty('serviceName', 'RANGER')) {
      configs.push({
        Clusters: {
          desired_config: this.reconfigureSites(['ranger-tagsync-site'], data, note)
        }
      });
    }
    if (servicesModel.someProperty('serviceName', 'ACCUMULO')) {
      configs.push({
        Clusters: {
          desired_config: this.reconfigureSites(['accumulo-site'], data, note)
        }
      });
    }
    return App.ajax.send({
      name: 'common.service.multiConfigurations',
      sender: this,
      data: {
        configs: configs
      },
      error: 'onTaskError',
      success: 'installHDFSClients'
    });
  },

  installHDFSClients: function installHDFSClients() {
    var nnHostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').mapProperty('hostName');
    var jnHostNames = App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').mapProperty('hostName');
    var hostNames = nnHostNames.concat(jnHostNames).uniq();
    this.createInstallComponentTask('HDFS_CLIENT', hostNames, 'HDFS');
  },

  installNameNode: function installNameNode() {
    this.createInstallComponentTask('NAMENODE', this.get('newNameNodeHosts'), "HDFS");
  },

  installZKFC: function installZKFC() {
    this.createInstallComponentTask('ZKFC', this.get('newNameNodeHosts'), "HDFS");
  },

  startJournalNodes: function startJournalNodes() {
    var hostNames = App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').mapProperty('hostName');
    this.updateComponent('JOURNALNODE', hostNames, "HDFS", "Start");
  },

  startNameNodes: function startNameNodes() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').filterProperty('isInstalled').mapProperty('hostName');
    this.updateComponent('NAMENODE', hostNames, "HDFS", "Start");
  },

  startZKFCs: function startZKFCs() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').filterProperty('isInstalled').mapProperty('hostName');
    this.updateComponent('ZKFC', hostNames, "HDFS", "Start");
  },

  formatNameNode: function formatNameNode() {
    App.ajax.send({
      name: 'nameNode.federation.formatNameNode',
      sender: this,
      data: {
        host: this.get('newNameNodeHosts')[0]
      },
      success: 'startPolling',
      error: 'onTaskError'
    });
  },

  formatZKFC: function formatZKFC() {
    App.ajax.send({
      name: 'nameNode.federation.formatZKFC',
      sender: this,
      data: {
        host: this.get('newNameNodeHosts')[0]
      },
      success: 'startPolling',
      error: 'onTaskError'
    });
  },

  startZKFC: function startZKFC() {
    this.updateComponent('ZKFC', this.get('newNameNodeHosts')[0], "HDFS", "Start");
  },

  startInfraSolr: function startInfraSolr() {
    this.startServices(false, ['AMBARI_INFRA_SOLR'], true);
  },

  startRangerAdmin: function startRangerAdmin() {
    var hostNames = App.HostComponent.find().filterProperty('componentName', 'RANGER_ADMIN').mapProperty('hostName');
    this.updateComponent('RANGER_ADMIN', hostNames, "RANGER", "Start");
  },

  startRangerUsersync: function startRangerUsersync() {
    var hostNames = App.HostComponent.find().filterProperty('componentName', 'RANGER_USERSYNC').mapProperty('hostName');
    this.updateComponent('RANGER_USERSYNC', hostNames, "RANGER", "Start");
  },

  startNameNode: function startNameNode() {
    this.updateComponent('NAMENODE', this.get('newNameNodeHosts')[0], "HDFS", "Start");
  },

  bootstrapNameNode: function bootstrapNameNode() {
    App.ajax.send({
      name: 'nameNode.federation.bootstrapNameNode',
      sender: this,
      data: {
        host: this.get('newNameNodeHosts')[1]
      },
      success: 'startPolling',
      error: 'onTaskError'
    });
  },

  startZKFC2: function startZKFC2() {
    this.updateComponent('ZKFC', this.get('newNameNodeHosts')[1], "HDFS", "Start");
  },

  startNameNode2: function startNameNode2() {
    this.updateComponent('NAMENODE', this.get('newNameNodeHosts')[1], "HDFS", "Start");
  },

  restartAllServices: function restartAllServices() {
    App.ajax.send({
      name: 'restart.custom.filter',
      sender: this,
      data: {
        filter: "HostRoles/component_name!=NAMENODE&HostRoles/component_name!=JOURNALNODE&HostRoles/component_name!=ZKFC&HostRoles/component_name!=RANGER_ADMIN&HostRoles/component_name!=RANGER_USERSYNC&HostRoles/cluster_name=" + App.get('clusterName'),
        context: "Restart Required Services"
      },
      success: 'startPolling',
      error: 'onTaskError'
    });
  }
});

});

require.register("controllers/main/admin/federation/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.NameNodeFederationWizardController = App.WizardController.extend({

  name: 'nameNodeFederationWizardController',

  totalSteps: 4,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('admin.nameNodeFederation.wizard.header'),

  isFinished: false,

  content: Em.Object.create({
    controllerName: 'nameNodeFederationWizardController'
  }),

  /**
   * Load data for all steps until <code>current step</code>
   */
  loadMap: {
    '1': [{
      type: 'sync',
      callback: function callback() {
        this.load('cluster');
      }
    }],
    '2': [{
      type: 'async',
      callback: function callback() {
        var self = this,
            dfd = $.Deferred();
        this.loadServicesFromServer();
        this.loadMasterComponentHosts().done(function () {
          self.loadConfirmedHosts();
          dfd.resolve();
        });
        return dfd.promise();
      }
    }],
    '3': [{
      type: 'sync',
      callback: function callback() {
        this.loadNameServiceId();
      }
    }],
    '4': [{
      type: 'sync',
      callback: function callback() {
        this.loadServiceConfigProperties();
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
      }
    }]
  },

  init: function init() {
    this._super();
    this.clearStep();
  },

  clearStep: function clearStep() {
    this.set('isFinished', false);
  },

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      wizardControllerName: 'nameNodeFederationWizardController',
      localdb: App.db.data
    });
  },

  saveNameServiceId: function saveNameServiceId(nameServiceId) {
    this.setDBProperty('nameServiceId', nameServiceId);
    this.set('content.nameServiceId', nameServiceId);
  },

  loadNameServiceId: function loadNameServiceId() {
    var nameServiceId = this.getDBProperty('nameServiceId');
    this.set('content.nameServiceId', nameServiceId);
  },

  saveServiceConfigProperties: function saveServiceConfigProperties(stepController) {
    var serviceConfigProperties = [];
    var data = stepController.get('serverConfigData');

    var _content = stepController.get('stepConfigs')[0];
    _content.get('configs').forEach(function (_configProperties) {
      var siteObj = data.items.findProperty('type', _configProperties.get('filename'));
      if (siteObj) {
        siteObj.properties[_configProperties.get('name')] = _configProperties.get('value');
      }
    }, this);
    this.setDBProperty('serviceConfigProperties', data);
    this.set('content.serviceConfigProperties', data);
  },

  /**
   * Load serviceConfigProperties to model
   */
  loadServiceConfigProperties: function loadServiceConfigProperties() {
    this.set('content.serviceConfigProperties', this.getDBProperty('serviceConfigProperties'));
  },

  /**
   * Remove all loaded data.
   * Created as copy for App.router.clearAllSteps
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.resetDbNamespace();
    App.router.get('updateController').updateAll();
    this.set('isFinished', true);
  }
});

});

require.register("controllers/main/admin/highAvailability/hawq/activateStandby/step1_controller", function(exports, require, module) {
"use strict";

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ActivateHawqStandbyWizardStep1Controller = Em.Controller.extend({
  name: "activateHawqStandbyWizardStep1Controller"
});

});

require.register("controllers/main/admin/highAvailability/hawq/activateStandby/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @typedef {object} hawqActivateStandbyConfigDependencies
 */

var App = require('app');
require('utils/configs/hawq_activate_standby_config_initializer');

App.ActivateHawqStandbyWizardStep2Controller = Em.Controller.extend({
  name: "activateHawqStandbyWizardStep2Controller",

  selectedService: null,

  versionLoaded: true,

  hideDependenciesInfoBar: true,

  isLoaded: false,

  isSubmitDisabled: Em.computed.not('isLoaded'),

  loadStep: function loadStep() {
    this.renderConfigs();
  },

  /**
   * Render configs to show them in <code>App.ServiceConfigView</code>
   */
  renderConfigs: function renderConfigs() {
    newHawqMaster = App.HostComponent.find().findProperty('componentName', 'HAWQSTANDBY').get('hostName');

    var configs = require('data/configs/wizards/hawq_activate_standby_properties').haConfig;

    var serviceConfig = App.ServiceConfig.create({
      serviceName: configs.serviceName,
      displayName: configs.displayName,
      configCategories: [],
      showConfig: true,
      configs: []
    });

    configs.configCategories.forEach(function (configCategory) {
      if (App.Service.find().someProperty('serviceName', configCategory.name)) {
        serviceConfig.configCategories.pushObject(configCategory);
      }
    }, this);

    this.renderConfigProperties(configs, serviceConfig);
    this.setDynamicConfigValues(serviceConfig);
    this.setProperties({
      selectedService: serviceConfig,
      isLoaded: true
    });
  },

  setDynamicConfigValues: function setDynamicConfigValues(configs) {
    var topologyLocalDB = this.get('content').getProperties(['masterComponentHosts']);
    configs.configs.forEach(function (config) {
      App.HawqActivateStandbyConfigInitializer.initialValue(config, topologyLocalDB);
    });
    App.HawqActivateStandbyConfigInitializer.cleanup();
    return configs;
  },

  /**
   * Load child components to service config object
   * @param _componentConfig
   * @param componentConfig
   */
  renderConfigProperties: function renderConfigProperties(_componentConfig, componentConfig) {
    _componentConfig.configs.forEach(function (_serviceConfigProperty) {
      var serviceConfigProperty = App.ServiceConfigProperty.create(_serviceConfigProperty);
      componentConfig.configs.pushObject(serviceConfigProperty);
      serviceConfigProperty.set('isEditable', serviceConfigProperty.get('isReconfigurable'));
    }, this);
  },

  submit: function submit() {
    if (!this.get('isSubmitDisabled')) {
      App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
        App.router.send("next");
      });
    }
  }

});

});

require.register("controllers/main/admin/highAvailability/hawq/activateStandby/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
require('controllers/main/admin/serviceAccounts_controller');

App.ActivateHawqStandbyWizardStep3Controller = App.HighAvailabilityProgressPageController.extend(App.WizardEnableDone, {

  name: "activateHawqStandbyWizardStep3Controller",

  clusterDeployState: 'ACTIVATE_HAWQ_STANDBY',

  hawqActivateStandbyCustomCommand: "ACTIVATE_HAWQ_STANDBY",

  hawqServiceName: "HAWQ",

  hawqStandbyComponentName: "HAWQSTANDBY",

  hawqMasterComponentName: "HAWQMASTER",

  commands: ['activateStandby', 'stopRequiredServices', 'reconfigureHAWQ', 'installHawqMaster', 'deleteOldHawqMaster', 'deleteHawqStandby', 'startRequiredServices'],

  tasksMessagesPrefix: 'admin.activateHawqStandby.wizard.step',

  activateStandby: function activateStandby() {
    App.ajax.send({
      name: 'service.item.executeCustomCommand',
      sender: this,
      data: {
        command: this.hawqActivateStandbyCustomCommand,
        context: Em.I18n.t('admin.activateHawqStandby.wizard.step3.activateHawqStandbyCommand.context'),
        hosts: this.get('content.hawqHosts.hawqStandby'),
        serviceName: this.hawqServiceName,
        componentName: this.hawqStandbyComponentName
      },
      success: 'startPolling',
      error: 'onTaskError'
    });
  },

  stopRequiredServices: function stopRequiredServices() {
    this.stopServices([this.hawqServiceName], true);
  },

  reconfigureHAWQ: function reconfigureHAWQ() {
    App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'onLoadHawqConfigsTags',
      error: 'onTaskError'
    });
  },

  onLoadHawqConfigsTags: function onLoadHawqConfigsTags(data) {
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: '(type=hawq-site&tag=' + data.Clusters.desired_configs['hawq-site'].tag + ')',
        type: 'hawq-site'
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadConfigs: function onLoadConfigs(data, opt, params) {
    delete data.items[0].properties['hawq_standby_address_host'];

    var propertiesToAdd = this.get('content.configs').filterProperty('filename', params.type);
    propertiesToAdd.forEach(function (property) {
      data.items[0].properties[property.name] = property.value;
    });

    var configData = this.reconfigureSites([params.type], data, Em.I18n.t('admin.activateHawqStandby.step4.save.configuration.note').format(App.format.role('HAWQSTANDBY', false)));

    App.ajax.send({
      name: 'common.service.configurations',
      sender: this,
      data: {
        desired_config: configData
      },
      success: 'onSaveConfigs',
      error: 'onTaskError'
    });
  },
  onSaveConfigs: function onSaveConfigs() {
    this.onTaskCompleted();
  },

  installHawqMaster: function installHawqMaster() {
    var hostName = this.get('content.hawqHosts.hawqStandby');
    this.createInstallComponentTask(this.hawqMasterComponentName, hostName, this.hawqServiceName);
  },

  deleteOldHawqMaster: function deleteOldHawqMaster() {
    var hostName = this.get('content.hawqHosts.hawqMaster');
    this.deleteComponent(this.hawqMasterComponentName, hostName);
  },

  deleteHawqStandby: function deleteHawqStandby() {
    var hostName = this.get('content.hawqHosts.hawqStandby');
    this.deleteComponent(this.hawqStandbyComponentName, hostName);
  },

  startRequiredServices: function startRequiredServices() {
    this.startServices(false, [this.hawqServiceName], true);
  }

});

});

require.register("controllers/main/admin/highAvailability/hawq/activateStandby/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ActivateHawqStandbyWizardController = App.WizardController.extend({

  name: 'activateHawqStandbyWizardController',

  totalSteps: 3,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('admin.activateHawqStandby.wizard.header'),

  isFinished: false,

  content: Em.Object.create({
    controllerName: 'activateHawqStandbyWizardController'
  }),

  /**
   * Load data for all steps until <code>current step</code>
   */
  loadMap: {
    '1': [{
      type: 'sync',
      callback: function callback() {
        this.load('cluster');
      }
    }],
    '2': [{
      type: 'async',
      callback: function callback() {
        var dfd = $.Deferred();
        this.loadHawqHosts();
        this.loadServicesFromServer();
        this.loadMasterComponentHosts().done(function () {
          dfd.resolve();
        });
        return dfd.promise();
      }
    }],
    '3': [{
      type: 'sync',
      callback: function callback() {
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
        this.loadConfigs();
      }
    }]
  },

  init: function init() {
    this._super();
    this.clearStep();
  },

  clearStep: function clearStep() {
    this.set('isFinished', false);
  },

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      wizardControllerName: this.name,
      localdb: App.db.data
    });
  },

  /**
   * Save hosts for Hawq Master and Hawq Standby<code>controller.content</code>
   * @param hawqHosts
   */
  saveHawqHosts: function saveHawqHosts(hawqHosts) {
    this.set('content.hawqHosts', hawqHosts);
    this.setDBProperty('hawqHosts', hawqHosts);
  },

  /**
   * Load hosts for additional and current ResourceManagers from local db to <code>controller.content</code>
   */
  loadHawqHosts: function loadHawqHosts() {
    var hawqHosts = this.getDBProperty('hawqHosts');
    this.set('content.hawqHosts', hawqHosts);
  },

  /**
   * Save configs to load and apply them on Configure Components step
   * @param configs
   */
  saveConfigs: function saveConfigs(configs) {
    this.set('content.configs', configs);
    this.setDBProperty('configs', configs);
  },

  /**
   * Load configs to apply them on Configure Components step
   */
  loadConfigs: function loadConfigs() {
    var configs = this.getDBProperty('configs');
    this.set('content.configs', configs);
  },

  /**
   * Remove all loaded data.
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.resetDbNamespace();
    App.router.get('updateController').updateAll();
    this.set('isFinished', true);
  }
});

});

require.register("controllers/main/admin/highAvailability/hawq/addStandby/step1_controller", function(exports, require, module) {
"use strict";

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddHawqStandbyWizardStep1Controller = Em.Controller.extend({
  name: "addHawqStandbyWizardStep1Controller"
});

});

require.register("controllers/main/admin/highAvailability/hawq/addStandby/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddHawqStandbyWizardStep2Controller = Em.Controller.extend(App.BlueprintMixin, App.AssignMasterComponents, {

  name: "addHawqStandbyWizardStep2Controller",

  useServerValidation: true,

  mastersToShow: ['HAWQMASTER', 'HAWQSTANDBY'],

  mastersToAdd: ['HAWQSTANDBY'],

  showCurrentPrefix: ['HAWQMASTER'],

  showInstalledMastersFirst: true
});

});

require.register("controllers/main/admin/highAvailability/hawq/addStandby/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @typedef {object} hawqHaConfigDependencies
 */

var App = require('app');
require('utils/configs/hawq_ha_config_initializer');

App.AddHawqStandbyWizardStep3Controller = Em.Controller.extend({
  name: "addHawqStandbyWizardStep3Controller",

  // Used in service_config.hbs
  // selectedService
  // hideDependenciesInfoBar
  // versionLoaded

  selectedService: null,

  hawqProps: null,

  hideDependenciesInfoBar: true,

  versionLoaded: true,

  // Used in step3.hbs
  // isLoaded
  // isSubmitDisabled
  // hawqMasterDirectoryCleanUpMessage
  isLoaded: false,

  isSubmitDisabled: Em.computed.not('isLoaded'),

  loadStep: function loadStep() {
    this.renderConfigs();
  },

  content: Em.Object.create({
    controllerName: 'addHawqStandbyWizardStep3Controller'
  }),

  /**
   * Render configs to show them in <code>App.ServiceConfigView</code>
   */
  renderConfigs: function renderConfigs() {

    var configs = require('data/configs/wizards/hawq_ha_properties').haConfig;

    var serviceConfig = App.ServiceConfig.create({
      serviceName: configs.serviceName,
      displayName: configs.displayName,
      configCategories: [],
      showConfig: true,

      configs: []
    });

    configs.configCategories.forEach(function (configCategory) {
      if (App.Service.find().someProperty('serviceName', configCategory.name)) {
        serviceConfig.configCategories.pushObject(configCategory);
      }
    }, this);

    this.renderConfigProperties(configs, serviceConfig);
    App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'loadConfigTagsSuccessCallback',
      error: '',
      data: {
        serviceConfig: serviceConfig
      }
    });
  },

  loadConfigTagsSuccessCallback: function loadConfigTagsSuccessCallback(data, opt, params) {
    var urlParams = '(type=hawq-site&tag=' + data.Clusters.desired_configs['hawq-site'].tag + ')';
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: urlParams,
        serviceConfig: params.serviceConfig
      },
      success: 'loadConfigsSuccessCallback',
      error: 'loadConfigsSuccessCallback'
    });
  },

  loadConfigsSuccessCallback: function loadConfigsSuccessCallback(data, opt, params) {
    params = params.serviceConfig ? params.serviceConfig : {};
    this.setDynamicConfigValues(params, data);
    this.setProperties({
      selectedService: params,
      isLoaded: true,
      hawqProps: data
    });
  },

  setDynamicConfigValues: function setDynamicConfigValues(configs, data) {
    var topologyLocalDB = this.get('content').getProperties(['masterComponentHosts']);
    configs.configs.forEach(function (config) {
      App.HawqHaConfigInitializer.initialValue(config, topologyLocalDB);
    });
    App.HawqHaConfigInitializer.cleanup();
    return configs;
  },

  /**
   * Load child components to service config object
   * @param _componentConfig
   * @param componentConfig
   */
  renderConfigProperties: function renderConfigProperties(_componentConfig, componentConfig) {
    _componentConfig.configs.forEach(function (_serviceConfigProperty) {
      var serviceConfigProperty = App.ServiceConfigProperty.create(_serviceConfigProperty);
      componentConfig.configs.pushObject(serviceConfigProperty);
      serviceConfigProperty.set('isEditable', serviceConfigProperty.get('isReconfigurable'));
    }, this);
  },

  submit: function submit() {
    if (!this.get('isSubmitDisabled')) {
      dataDir = this.get('hawqProps').items[0].properties['hawq_master_directory'];
      hawqStandby = this.get('content.hawqHosts.newHawqStandby');
      App.showConfirmationPopup(function () {
        App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
          App.router.send("next");
        });
      }, Em.I18n.t('admin.addHawqStandby.wizard.step3.confirm.dataDir.body').format(dataDir, hawqStandby), null, Em.I18n.t('admin.addHawqStandby.wizard.step3.confirm.dataDir.title'), "Confirm");
    }
  }

});

});

require.register("controllers/main/admin/highAvailability/hawq/addStandby/step4_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

require('controllers/main/admin/serviceAccounts_controller');

App.AddHawqStandbyWizardStep4Controller = App.HighAvailabilityProgressPageController.extend(App.WizardEnableDone, {

  name: "addHawqStandbyWizardStep4Controller",

  clusterDeployState: 'ADD_HAWQ_STANDBY',

  commands: ['stopRequiredServices', 'installHawqStandbyMaster', 'reconfigureHAWQ', 'startRequiredServices'],

  tasksMessagesPrefix: 'admin.addHawqStandby.wizard.step',

  stopRequiredServices: function stopRequiredServices() {
    this.stopServices(['HAWQ'], true);
  },

  installHawqStandbyMaster: function installHawqStandbyMaster() {
    var hostName = this.get('content.hawqHosts.newHawqStandby');
    this.createInstallComponentTask('HAWQSTANDBY', hostName, "HAWQ");
  },

  reconfigureHAWQ: function reconfigureHAWQ() {
    App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'onLoadHawqConfigsTags',
      error: 'onTaskError'
    });
  },

  onLoadHawqConfigsTags: function onLoadHawqConfigsTags(data) {
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: '(type=hawq-site&tag=' + data.Clusters.desired_configs['hawq-site'].tag + ')',
        type: 'hawq-site'
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadConfigs: function onLoadConfigs(data, opt, params) {
    var propertiesToAdd = this.get('content.configs').filterProperty('filename', params.type);
    propertiesToAdd.forEach(function (property) {
      data.items[0].properties[property.name] = property.value;
    });

    var configData = this.reconfigureSites([params.type], data, Em.I18n.t('admin.addHawqStandby.step4.save.configuration.note').format(App.format.role('HAWQSTANDBY', false)));

    App.ajax.send({
      name: 'common.service.configurations',
      sender: this,
      data: {
        desired_config: configData
      },
      success: 'onSaveConfigs',
      error: 'onTaskError'
    });
  },
  onSaveConfigs: function onSaveConfigs() {
    this.onTaskCompleted();
  },

  startRequiredServices: function startRequiredServices() {
    this.startServices(false, ["HAWQ"], true);
  }
});

});

require.register("controllers/main/admin/highAvailability/hawq/addStandby/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddHawqStandbyWizardController = App.WizardController.extend({

  name: 'addHawqStandbyWizardController',

  totalSteps: 4,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('admin.addHawqStandby.wizard.header'),

  isFinished: false,

  content: Em.Object.create({
    controllerName: 'addHawqStandbyWizardController'
  }),

  /**
   * Load data for all steps until <code>current step</code>
   */
  loadMap: {
    '1': [{
      type: 'sync',
      callback: function callback() {
        this.load('cluster');
      }
    }],
    '2': [{
      type: 'async',
      callback: function callback() {
        var self = this,
            dfd = $.Deferred();
        this.loadHawqHosts();
        this.loadServicesFromServer();
        this.loadMasterComponentHosts().done(function () {
          self.loadConfirmedHosts();
          dfd.resolve();
        });
        return dfd.promise();
      }
    }],
    '4': [{
      type: 'sync',
      callback: function callback() {
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
        this.loadConfigs();
      }
    }]
  },

  init: function init() {
    this._super();
    this.clearStep();
  },

  clearStep: function clearStep() {
    this.set('isFinished', false);
  },

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      wizardControllerName: 'addHawqStandbyWizardController',
      localdb: App.db.data
    });
  },

  /**
   * Save hosts for hawq master and hawq standby to local db and <code>controller.content</code>
   * @param hawqHosts
   */
  saveHawqHosts: function saveHawqHosts(hawqHosts) {
    this.set('content.hawqHosts', hawqHosts);
    this.setDBProperty('hawqHosts', hawqHosts);
  },

  /**
   * Load hosts for hawq master and hawq standby from local db to <code>controller.content</code>
   */
  loadHawqHosts: function loadHawqHosts() {
    this.set('content.hawqHosts', this.getDBProperty('hawqHosts'));
  },

  /**
   * Save configs to load and apply them on Configure Components step
   * @param configs
   */
  saveConfigs: function saveConfigs(configs) {
    this.set('content.configs', configs);
    this.setDBProperty('configs', configs);
  },

  /**
   * Load configs to apply them on Configure Components step
   */
  loadConfigs: function loadConfigs() {
    this.set('content.configs', this.getDBProperty('configs'));
  },

  /**
   * Remove all loaded data.
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.resetDbNamespace();
    App.router.get('updateController').updateAll();
    this.set('isFinished', true);
  }

});

});

require.register("controllers/main/admin/highAvailability/hawq/removeStandby/step1_controller", function(exports, require, module) {
"use strict";

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RemoveHawqStandbyWizardStep1Controller = Em.Controller.extend({
  name: "removeHawqStandbyWizardStep1Controller"
});

});

require.register("controllers/main/admin/highAvailability/hawq/removeStandby/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @typedef {object} hawqRemoveStandbyConfigDependencies
 */

var App = require('app');

App.RemoveHawqStandbyWizardStep2Controller = Em.Controller.extend({
  name: "removeHawqStandbyWizardStep2Controller",

  selectedService: null,

  versionLoaded: true,

  hideDependenciesInfoBar: true,

  isLoaded: true,

  isSubmitDisabled: Em.computed.not('isLoaded'),

  submit: function submit() {
    if (!this.get('isSubmitDisabled')) {
      App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
        App.router.send("next");
      });
    }
  }

});

});

require.register("controllers/main/admin/highAvailability/hawq/removeStandby/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
require('controllers/main/admin/serviceAccounts_controller');

App.RemoveHawqStandbyWizardStep3Controller = App.HighAvailabilityProgressPageController.extend(App.WizardEnableDone, {

  name: "removeHawqStandbyWizardStep3Controller",

  clusterDeployState: 'REMOVE_HAWQ_STANDBY',

  hawqRemoveStandbyCustomCommand: "REMOVE_HAWQ_STANDBY",

  hawqServiceName: "HAWQ",

  hawqMasterComponentName: "HAWQMASTER",

  hawqStandbyComponentName: "HAWQSTANDBY",

  commands: ['removeStandby', 'stopRequiredServices', 'reconfigureHAWQ', 'deleteHawqStandbyComponent', 'startRequiredServices'],

  tasksMessagesPrefix: 'admin.removeHawqStandby.wizard.step',

  removeStandby: function removeStandby() {
    App.ajax.send({
      name: 'service.item.executeCustomCommand',
      sender: this,
      data: {
        command: this.hawqRemoveStandbyCustomCommand,
        context: Em.I18n.t('admin.removeHawqStandby.wizard.step3.removeHawqStandbyCommand.context'),
        hosts: this.get('content.hawqHosts.hawqMaster'),
        serviceName: this.hawqServiceName,
        componentName: this.hawqMasterComponentName
      },
      success: 'startPolling',
      error: 'onTaskError'
    });
  },

  stopRequiredServices: function stopRequiredServices() {
    this.stopServices([this.hawqServiceName], true);
  },

  reconfigureHAWQ: function reconfigureHAWQ() {
    App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'onLoadHawqConfigsTags',
      error: 'onTaskError'
    });
  },

  onLoadHawqConfigsTags: function onLoadHawqConfigsTags(data) {
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: '(type=hawq-site&tag=' + data.Clusters.desired_configs['hawq-site'].tag + ')',
        type: 'hawq-site'
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadConfigs: function onLoadConfigs(data, opt, params) {
    delete data.items[0].properties['hawq_standby_address_host'];

    var configData = this.reconfigureSites([params.type], data, Em.I18n.t('admin.removeHawqStandby.wizard.step3.save.configuration.note').format(App.format.role('HAWQSTANDBY', false)));

    App.ajax.send({
      name: 'common.service.configurations',
      sender: this,
      data: {
        desired_config: configData
      },
      success: 'onSaveConfigs',
      error: 'onTaskError'
    });
  },
  onSaveConfigs: function onSaveConfigs() {
    this.onTaskCompleted();
  },

  deleteHawqStandbyComponent: function deleteHawqStandbyComponent() {
    var hostName = this.get('content.hawqHosts.hawqStandby');
    this.deleteComponent(this.hawqStandbyComponentName, hostName);
  },

  startRequiredServices: function startRequiredServices() {
    this.startServices(false, [this.hawqServiceName], true);
  }

});

});

require.register("controllers/main/admin/highAvailability/hawq/removeStandby/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RemoveHawqStandbyWizardController = App.WizardController.extend({

  name: 'removeHawqStandbyWizardController',

  totalSteps: 3,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('admin.removeHawqStandby.wizard.header'),

  isFinished: false,

  content: Em.Object.create({
    controllerName: 'removeHawqStandbyWizardController'
  }),

  /**
   * Load data for all steps until <code>current step</code>
   */
  loadMap: {
    '1': [{
      type: 'sync',
      callback: function callback() {
        this.load('cluster');
      }
    }],
    '2': [{
      type: 'async',
      callback: function callback() {
        var dfd = $.Deferred();
        this.loadHawqHosts();
        this.loadServicesFromServer();
        this.loadMasterComponentHosts().done(function () {
          dfd.resolve();
        });
        return dfd.promise();
      }
    }],
    '3': [{
      type: 'sync',
      callback: function callback() {
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
        this.loadConfigs();
      }
    }]
  },

  init: function init() {
    this._super();
    this.clearStep();
  },

  clearStep: function clearStep() {
    this.set('isFinished', false);
  },

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      wizardControllerName: 'removeHawqStandbyWizardController',
      localdb: App.db.data
    });
  },

  /**
   * Save hosts for Hawq Master and Hawq Standby to local db<code>controller.content</code>
   * @param hawqHosts
   */
  saveHawqHosts: function saveHawqHosts(hawqHosts) {
    this.set('content.hawqHosts', hawqHosts);
    this.setDBProperty('hawqHosts', hawqHosts);
  },

  /**
   * Load hosts for Hawq Master and Hawq Standby from local db to <code>controller.content</code>
   */
  loadHawqHosts: function loadHawqHosts() {
    var hawqHosts = this.getDBProperty('hawqHosts');
    this.set('content.hawqHosts', hawqHosts);
  },

  /**
   * Save configs to load and apply them on Configure Components step
   * @param configs
   */
  saveConfigs: function saveConfigs(configs) {
    this.set('content.configs', configs);
    this.setDBProperty('configs', configs);
  },

  /**
   * Load configs to apply them on Configure Components step
   */
  loadConfigs: function loadConfigs() {
    var configs = this.getDBProperty('configs');
    this.set('content.configs', configs);
  },

  /**
   * Remove all loaded data.
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.resetDbNamespace();
    App.router.get('updateController').updateAll();
    this.set('isFinished', true);
  }
});

});

require.register("controllers/main/admin/highAvailability/journalNode/progress_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ManageJournalNodeProgressPageController = App.ManageJournalNodeWizardController.extend(App.wizardProgressPageControllerMixin, {
  name: 'manageJournalNodeProgressPageController',
  clusterDeployState: 'JOURNALNODE_MANAGEMENT',
  tasksMessagesPrefix: 'admin.manageJournalNode.wizard.step',
  isRollback: false,

  /**
   * Prepare object to send to the server to save configs
   * Split all configs by site names and note
   * @param siteNames Array
   * @param data Object
   * @param note String
   */
  reconfigureSites: function reconfigureSites(siteNames, data, note) {

    return siteNames.map(function (_siteName) {
      var config = data.items.findProperty('type', _siteName);
      var configToSave = {
        type: _siteName,
        properties: config && config.properties,
        service_config_version_note: note || ''
      };
      if (config && config.properties_attributes) {
        configToSave.properties_attributes = config.properties_attributes;
      }
      return configToSave;
    });
  }
});

});

require.register("controllers/main/admin/highAvailability/journalNode/step1_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

require('controllers/wizard/step5_controller');

App.ManageJournalNodeWizardStep1Controller = Em.Controller.extend(App.BlueprintMixin, App.AssignMasterComponents, {

  name: "manageJournalNodeWizardStep1Controller",

  useServerValidation: false,

  mastersToShow: ['JOURNALNODE'],

  showInstalledMastersFirst: true,

  JOURNALNODES_COUNT_MINIMUM: 3, // TODO get this from stack

  nextButtonCheckTrigger: true,

  /**
   * On initial rendering, load equivalent number of existing JournalNodes to masterToShow
   * @param masterComponents
   */
  renderComponents: function renderComponents(masterComponents) {
    //check if we are restoring components assignment by checking existing of JOURNALNODE component in array
    var restoringComponents = masterComponents.someProperty('component_name', 'JOURNALNODE');
    masterComponents = restoringComponents ? masterComponents : masterComponents.concat(this.generateJournalNodeComponents());
    this._super(masterComponents);
    this.updateJournalNodeInfo();
    this.showHideJournalNodesAddRemoveControl();
    this.toggleProperty('nextButtonCheckTrigger');
  },

  /**
   * Create JOURNALNODE components to add them to masters array
   */
  generateJournalNodeComponents: function generateJournalNodeComponents() {
    var journalNodes = [];
    App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').forEach(function (jn) {
      var jnComponent = this.createComponentInstallationObject(Em.Object.create({
        serviceName: jn.get('service.serviceName'),
        componentName: jn.get('componentName')
      }), jn.get('hostName'));
      jnComponent.isInstalled = true;
      journalNodes.push(jnComponent);
    }, this);
    return journalNodes;
  },

  /**
   * Override of method from <code>App.AssignMasterComponents</code>
   * Added to satisfy the requirement that maximum number of new JournalNodes to be added
   * should be 1 less that total of the existing JournalNodes
   * @returns {number}
   */
  getMaxNumberOfMasters: function getMaxNumberOfMasters() {
    var defaultLimitation = this._super('JOURNALNODE'),
        installedJournalNodesCount = App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').length;
    return Math.min(defaultLimitation, installedJournalNodesCount * 2 - 1);
  },

  /**
   * Enable/Disable show/hide operation for each JournalNode
   */
  showHideJournalNodesAddRemoveControl: function () {
    var masterComponents = this.get('selectedServicesMasters');
    var jns = masterComponents.filterProperty('component_name', 'JOURNALNODE');
    var maxNumMasters = this.getMaxNumberOfMasters('JOURNALNODE');
    var showRemoveControl = jns.get('length') > this.get('JOURNALNODES_COUNT_MINIMUM');
    var showAddControl = jns.get('length') < maxNumMasters;
    jns.forEach(function (item) {
      item.set('showAddControl', false);
      item.set('showRemoveControl', showRemoveControl);
    });
    jns.set('lastObject.showAddControl', showAddControl);
  }.observes('hostNameCheckTrigger'),

  /**
   * Mark existing JournalNodes 'isInstalled' and 'showCurrentPrefix'
   */
  updateJournalNodeInfo: function () {
    var jns = this.get('selectedServicesMasters').filterProperty('component_name', 'JOURNALNODE');
    var hosts = App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').mapProperty('hostName');
    hosts.forEach(function (host) {
      var jn = jns.findProperty('selectedHost', host);
      if (jn) {
        jn.set('isInstalled', true);
        jn.set('showCurrentPrefix', true);
      }
    });
  }.observes('hostNameCheckTrigger'),

  /**
   * Callback after load controller data (hosts, host components etc)
   * @method loadStepCallback
   */
  loadStepCallback: function loadStepCallback(components, self) {
    self.renderComponents(components);

    self.get('addableComponents').forEach(function (componentName) {
      self.updateComponent(componentName);
    }, self);
    self.set('isRecommendationsLoaded', true);
  },

  /**
   * Next button is disabled when there is any change to the original JournalNode hosts
   */
  nextButtonDisabled: function () {
    var currentHosts = this.get('selectedServicesMasters').filterProperty('component_name', 'JOURNALNODE').mapProperty('selectedHost');
    var originalHosts = App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').mapProperty('hostName');
    return currentHosts.sort().join() === originalHosts.sort().join();
  }.property('hostNameCheckTrigger', 'nextButtonCheckTrigger')

});

});

require.register("controllers/main/admin/highAvailability/journalNode/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @typedef {object} nnHaConfigDependencies
 * @property {string} namespaceId
 * @property {object} serverConfigs
 * @property {string|number} nnHttpPort
 * @property {string|number} nnHttpsPort
 * @property {string|number} nnRpcPort
 * @property {string|number} zkClientPort
 */

var App = require('app');

require('utils/configs/nn_ha_config_initializer');

App.ManageJournalNodeWizardStep2Controller = Em.Controller.extend({
  name: "manageJournalNodeWizardStep2Controller",
  selectedService: null,
  stepConfigs: [],
  serverConfigData: {},
  moveJNConfig: $.extend(true, {}, require('data/configs/wizards/move_journal_node_properties').moveJNConfig),
  once: false,
  isLoaded: false,
  versionLoaded: true,
  hideDependenciesInfoBar: true,

  isNextDisabled: Em.computed.not('isLoaded'),

  clearStep: function clearStep() {
    this.get('stepConfigs').clear();
    this.set('serverConfigData', {});
  },

  loadStep: function loadStep() {
    this.clearStep();
    this.loadConfigsTags();
  },

  loadConfigsTags: function loadConfigsTags() {
    App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'onLoadConfigsTags',
      error: 'onTaskError'
    });
  },

  onLoadConfigsTags: function onLoadConfigsTags(data) {
    var urlParams = [];
    var hdfsSiteTag = data.Clusters.desired_configs['hdfs-site'].tag;
    urlParams.push('(type=hdfs-site&tag=' + hdfsSiteTag + ')');
    this.set("hdfsSiteTag", { name: "hdfsSiteTag", value: hdfsSiteTag });

    App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        urlParams: urlParams.join('|')
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadConfigs: function onLoadConfigs(data) {
    this.set('serverConfigData', data);
    this.set('content.nameServiceIds', data.items[0].properties['dfs.nameservices'].split(','));
    this.tweakServiceConfigs(this.get('moveJNConfig.configs'));
    this.renderServiceConfigs(this.get('moveJNConfig'));
    this.set('isLoaded', true);
  },

  /**
   * Generate set of data used to correctly initialize config values and names
   */
  _prepareDependencies: function _prepareDependencies(nameServiceId) {
    var ret = {};
    var configsFromServer = this.get('serverConfigData.items');
    ret.namespaceId = nameServiceId || this.get('content.nameServiceIds')[0];
    ret.serverConfigs = configsFromServer;
    return ret;
  },

  /**
   * Generate set of data with information about cluster topology
   * Used in the configs' initialization process
   *
   * @returns {extendedTopologyLocalDB}
   * @private
   * @method _prepareLocalDB
   */
  _prepareLocalDB: function _prepareLocalDB() {
    var localDB = this.get('content').getProperties(['masterComponentHosts', 'slaveComponentHosts', 'hosts']);
    localDB.installedServices = App.Service.find().mapProperty('serviceName');
    return localDB;
  },

  tweakServiceConfigs: function tweakServiceConfigs(allConfigsDescriptor) {
    var hasNameNodeFederation = App.get('hasNameNodeFederation');
    var configs = hasNameNodeFederation ? allConfigsDescriptor.filterProperty('presentForFederatedHDFS') : allConfigsDescriptor.filterProperty('presentForNonFederatedHDFS');
    var nameSpaceDependentConfigs = configs.filterProperty('dependsOnNameServiceId');
    var nameSpaceIndependentConfigs = configs.rejectProperty('dependsOnNameServiceId');
    var localDB = this._prepareLocalDB();
    var commonDependencies = this._prepareDependencies();
    var generatedConfigs = [];
    var wizardController = App.router.get(this.get('content.controllerName'));
    var journalNodes = this.get('content.masterComponentHosts').filterProperty('component', 'JOURNALNODE');

    nameSpaceIndependentConfigs.forEach(function (config) {
      App.NnHaConfigInitializer.initialValue(config, localDB, commonDependencies);
      config.isOverridable = false;
      generatedConfigs.push(config);
    });

    this.get('content.nameServiceIds').forEach(function (nameServiceId) {
      var dependencies = this._prepareDependencies(nameServiceId);
      dependencies.journalnodes = journalNodes.map(function (c) {
        return c.hostName + ':8485';
      }).join(';');
      nameSpaceDependentConfigs.forEach(function (config) {
        var generatedConfig = $.extend({}, config, {
          isOverridable: false,
          name: wizardController.replaceDependencies(config.name, dependencies),
          displayName: wizardController.replaceDependencies(config.displayName, dependencies),
          value: wizardController.replaceDependencies(config.value, dependencies),
          recommendedValue: wizardController.replaceDependencies(config.recommendedValue, dependencies)
        });
        generatedConfigs.push(generatedConfig);
      }, this);
    }, this);

    this.set('moveJNConfig.configs', generatedConfigs);

    return generatedConfigs;
  },

  renderServiceConfigs: function renderServiceConfigs(_serviceConfig) {
    var serviceConfig = App.ServiceConfig.create({
      serviceName: _serviceConfig.serviceName,
      displayName: _serviceConfig.displayName,
      configCategories: [],
      showConfig: true,
      configs: []
    });

    _serviceConfig.configCategories.forEach(function (_configCategory) {
      if (App.Service.find().someProperty('serviceName', _configCategory.name)) {
        serviceConfig.configCategories.pushObject(_configCategory);
      }
    }, this);

    this.loadComponentConfigs(_serviceConfig, serviceConfig);

    this.get('stepConfigs').pushObject(serviceConfig);
    this.set('selectedService', this.get('stepConfigs').objectAt(0));
    this.set('once', true);
  },

  /**
   * Load child components to service config object
   * @param _componentConfig
   * @param componentConfig
   */
  loadComponentConfigs: function loadComponentConfigs(_componentConfig, componentConfig) {
    _componentConfig.configs.forEach(function (_serviceConfigProperty) {
      var serviceConfigProperty = App.ServiceConfigProperty.create(_serviceConfigProperty);
      componentConfig.configs.pushObject(serviceConfigProperty);
      serviceConfigProperty.set('isEditable', serviceConfigProperty.get('isReconfigurable'));
    }, this);
  }
});

});

require.register("controllers/main/admin/highAvailability/journalNode/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ManageJournalNodeWizardStep3Controller = App.HighAvailabilityWizardStep4Controller.extend({
  name: 'manageJournalNodeWizardStep3Controller',

  isActiveNameNodesStarted: true,

  isHDFSNameSpacesLoaded: Em.computed.alias('App.router.clusterController.isHDFSNameSpacesLoaded'),

  isDataLoadedAndNextEnabled: Em.computed.and('isNextEnabled', 'isHDFSNameSpacesLoaded'),

  pullCheckPointsStatuses: function pullCheckPointsStatuses() {
    if (this.get('isHDFSNameSpacesLoaded')) {
      this.removeObserver('isHDFSNameSpacesLoaded', this, 'pullCheckPointsStatuses');
      var hdfsModel = App.HDFSService.find('HDFS'),
          nameSpaces = hdfsModel.get('masterComponentGroups'),
          nameSpacesCount = nameSpaces.length;
      if (nameSpacesCount > 1) {
        var hostNames = hdfsModel.get('activeNameNodes').mapProperty('hostName');
        if (hostNames.length < nameSpacesCount) {
          nameSpaces.forEach(function (nameSpace) {
            var hosts = nameSpace.hosts,
                hasActiveNameNode = hosts.some(function (hostName) {
              return hostNames.contains(hostName);
            });

            if (!hasActiveNameNode) {
              var hostForNameSpace = hosts.find(function (hostName) {
                return App.HostComponent.find('NAMENODE_' + hostName).get('workStatus') === 'STARTED';
              }) || hosts[0];
              hostNames.push(hostForNameSpace);
            }
          });
        }
        App.ajax.send({
          name: 'admin.high_availability.getNnCheckPointsStatuses',
          sender: this,
          data: {
            hostNames: hostNames
          },
          success: 'checkNnCheckPointsStatuses'
        });
      } else {
        this.pullCheckPointStatus();
      }
    } else {
      this.addObserver('isHDFSNameSpacesLoaded', this, 'pullCheckPointsStatuses');
    }
  },

  checkNnCheckPointsStatuses: function checkNnCheckPointsStatuses(data) {
    var _this = this;

    var items = Em.getWithDefault(data, 'items', []),
        isNextEnabled = items.length && items.every(this.getNnCheckPointStatus);
    this.setProperties({
      isActiveNameNodesStarted: items.length && items.everyProperty('HostRoles.desired_state', 'STARTED'),
      isNextEnabled: isNextEnabled
    });
    if (!isNextEnabled) {
      window.setTimeout(function () {
        _this.pullCheckPointsStatuses();
      }, this.POLL_INTERVAL);
    }
  }
});

});

require.register("controllers/main/admin/highAvailability/journalNode/step4_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ManageJournalNodeWizardStep4Controller = App.ManageJournalNodeProgressPageController.extend({
  name: 'manageJournalNodeWizardStep4Controller',
  clusterDeployState: 'JOURNALNODE_MANAGEMENT',
  tasksMessagesPrefix: 'admin.manageJournalNode.wizard.step',

  commands: ['stopStandbyNameNode', 'stopAllServices', 'installJournalNodes', 'deleteJournalNodes', 'reconfigureHDFS'],

  hdfsSiteTag: "",

  stopStandbyNameNode: function stopStandbyNameNode() {
    // save who's active and who's standby at this point in time
    var hostName = this.get('content.standByNN.host_name');
    this.updateComponent('NAMENODE', hostName, 'HDFS', 'INSTALLED');
  },

  stopAllServices: function stopAllServices() {
    this.stopServices([], true, true);
  },

  installJournalNodes: function installJournalNodes() {
    var hostNames = App.router.get('manageJournalNodeWizardController').getJournalNodesToAdd();
    if (hostNames && hostNames.length > 0) {
      this.createInstallComponentTask('JOURNALNODE', hostNames, "HDFS");
    } else {
      this.onTaskCompleted();
    }
  },

  deleteJournalNodes: function deleteJournalNodes() {
    var hosts = App.router.get('manageJournalNodeWizardController').getJournalNodesToDelete();
    if (hosts && hosts.length > 0) {
      hosts.forEach(function (host) {
        this.deleteComponent('JOURNALNODE', host);
      }, this);
    } else {
      this.onTaskCompleted();
    }
  },

  reconfigureHDFS: function reconfigureHDFS() {
    this.updateConfigProperties(this.get('content.serviceConfigProperties'));
  },

  /**
   * Update service configurations
   * @param {Object} data - config object to update
   */
  updateConfigProperties: function updateConfigProperties(data) {
    var siteNames = ['hdfs-site'];
    var configData = this.reconfigureSites(siteNames, data, Em.I18n.t('admin.manageJournalNode.step4.save.configuration.note').format(App.format.role('NAMENODE', false)));
    App.ajax.send({
      name: 'common.service.configurations',
      sender: this,
      data: {
        desired_config: configData
      },
      success: 'installHDFSClients',
      error: 'onTaskError'
    });
  },

  installHDFSClients: function installHDFSClients() {
    var nnHostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').mapProperty('hostName');
    var jnHostNames = this.get('content.masterComponentHosts').filterProperty('component', 'JOURNALNODE').mapProperty('hostName');
    var hostNames = nnHostNames.concat(jnHostNames).uniq();
    this.createInstallComponentTask('HDFS_CLIENT', hostNames, 'HDFS');
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      clusterState: 'JOURNALNODE_MANAGEMENT',
      wizardControllerName: this.get('content.controllerName'),
      localdb: App.db.data
    });
  }

});

});

require.register("controllers/main/admin/highAvailability/journalNode/step5_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ManageJournalNodeWizardStep5Controller = Em.Controller.extend({
  name: 'manageJournalNodeWizardStep5Controller',
  isHDFSNameSpacesLoaded: Em.computed.alias('App.router.clusterController.isHDFSNameSpacesLoaded'),
  isNextEnabled: Em.computed.alias('isHDFSNameSpacesLoaded'),
  done: function done() {
    if (this.get('isNextEnabled')) {
      App.router.send('next');
    }
  }
});

});

require.register("controllers/main/admin/highAvailability/journalNode/step6_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ManageJournalNodeWizardStep6Controller = App.ManageJournalNodeProgressPageController.extend({
  name: 'manageJournalNodeWizardStep6Controller',
  clusterDeployState: 'JOURNALNODE_MANAGEMENT',
  tasksMessagesPrefix: 'admin.manageJournalNode.wizard.step',

  commands: ['startJournalNodes'],

  startJournalNodes: function startJournalNodes() {
    var hostNames = App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').mapProperty('hostName');
    this.updateComponent('JOURNALNODE', hostNames, 'HDFS', 'Start');
  }
});

});

require.register("controllers/main/admin/highAvailability/journalNode/step7_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ManageJournalNodeWizardStep7Controller = App.ManageJournalNodeProgressPageController.extend(App.WizardEnableDone, {
  name: 'manageJournalNodeWizardStep7Controller',
  clusterDeployState: 'JOURNALNODE_MANAGEMENT',
  tasksMessagesPrefix: 'admin.manageJournalNode.wizard.step',

  commands: ['startAllServices'],

  startAllServices: function startAllServices() {
    this.startServices(false);
  }
});

});

require.register("controllers/main/admin/highAvailability/journalNode/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.ManageJournalNodeWizardController = App.WizardController.extend({

  name: 'manageJournalNodeWizardController',

  totalSteps: 7,

  displayName: Em.I18n.t('admin.manageJournalNode.wizard.header'),

  /**
   * Used for hiding back button in wizard
   */
  hideBackButton: true,

  content: Em.Object.create({
    controllerName: 'manageJournalNodeWizardController',
    cluster: null,
    hosts: null,
    services: null,
    slaveComponentHosts: null,
    masterComponentHosts: null,
    serviceConfigProperties: [],
    serviceName: 'MISC',
    hdfsUser: "hdfs",
    nameServiceIds: [],
    failedTask: null,
    requestIds: null
  }),

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      clusterState: 'JOURNALNODE_MANAGEMENT',
      wizardControllerName: 'manageJournalNodeWizardController',
      localdb: App.db.data
    });
  },

  /**
   * return new object extended from clusterStatusTemplate
   * @return Object
   */
  getCluster: function getCluster() {
    return jQuery.extend({}, this.get('clusterStatusTemplate'), { name: App.router.getClusterName() });
  },

  loadMap: {
    '1': [{
      type: 'async',
      callback: function callback() {
        var dfd = $.Deferred(),
            self = this,
            usersLoadingCallback = function usersLoadingCallback() {
          self.save('hdfsUser');
          self.load('cluster');
          self.loadHosts().done(function () {
            self.loadServicesFromServer();
            self.loadMasterComponentHosts().done(function () {
              self.load('hdfsUser');
              if (!self.getDBProperty('activeNN')) {
                self.saveNNs();
              } else {
                self.loadNNs();
              }
              dfd.resolve();
            });
          });
        };
        if (self.getDBProperty('hdfsUser')) {
          usersLoadingCallback();
        } else {
          this.loadHdfsUserFromServer().done(function (data) {
            self.set('content.hdfsUser', Em.get(data, '0.properties.hdfs_user'));
            usersLoadingCallback();
          });
        }
        return dfd.promise();
      }
    }],
    2: [{
      type: 'sync',
      callback: function callback() {
        this.loadNameServiceIds();
        this.loadServiceConfigProperties();
      }
    }],
    '4': [{
      type: 'sync',
      callback: function callback() {
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
      }
    }]
  },

  getJournalNodesToAdd: function getJournalNodesToAdd() {
    var result = [];
    var masterComponentHosts = this.get('content.masterComponentHosts');
    if (masterComponentHosts) {
      result = masterComponentHosts.filterProperty('component', 'JOURNALNODE').filterProperty('isInstalled', false).mapProperty('hostName');
    }
    return result;
  },

  getJournalNodesToDelete: function getJournalNodesToDelete() {
    var result = [];
    var masterComponentHosts = this.get('content.masterComponentHosts');
    if (masterComponentHosts) {
      var currentJNs = masterComponentHosts.filterProperty('component', 'JOURNALNODE');
      var existingHosts = App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').mapProperty('hostName');
      result = existingHosts.filter(function (host) {
        return currentJNs.filterProperty('hostName', host).length === 0;
      });
    }
    return result;
  },

  isDeleteOnly: function () {
    return this.get('currentStep') > 1 && this.getJournalNodesToAdd().length === 0 && this.getJournalNodesToDelete().length > 0;
  }.property('content.masterComponentHosts', 'App.router.clusterController.isHostsLoaded', 'currentStep'),

  /**
   * Save config properties
   * @param stepController ManageJournalNodeWizardStep3Controller
   */
  saveServiceConfigProperties: function saveServiceConfigProperties(stepController) {
    var serviceConfigProperties = [];
    var data = stepController.get('serverConfigData');

    var _content = stepController.get('stepConfigs')[0];
    _content.get('configs').forEach(function (_configProperties) {
      var siteObj = data.items.findProperty('type', _configProperties.get('filename'));
      if (siteObj) {
        siteObj.properties[_configProperties.get('name')] = _configProperties.get('value');
      }
    }, this);
    this.setDBProperty('serviceConfigProperties', data);
    this.set('content.serviceConfigProperties', data);
  },

  /**
   * Load serviceConfigProperties to model
   */
  loadServiceConfigProperties: function loadServiceConfigProperties() {
    this.set('content.serviceConfigProperties', this.getDBProperty('serviceConfigProperties'));
  },

  saveNNs: function saveNNs() {
    var activeNN = App.HostComponent.find().findProperty('displayNameAdvanced', 'Active NameNode');
    var standByNN = App.HostComponent.find().findProperty('displayNameAdvanced', 'Standby NameNode');
    this.set('content.activeNN', activeNN);
    this.set('content.standByNN', standByNN);
    this.setDBProperty('activeNN', activeNN);
    this.setDBProperty('standByNN', standByNN);
  },

  loadNNs: function loadNNs() {
    var activeNN = this.getDBProperty('activeNN');
    var standByNN = this.getDBProperty('standByNN');
    this.set('content.activeNN', activeNN);
    this.set('content.standByNN', standByNN);
  },

  saveConfigTag: function saveConfigTag(tag) {
    App.db.setManageJournalNodeWizardConfigTag(tag);
    this.set('content.' + tag.name, tag.value);
  },

  loadConfigTag: function loadConfigTag(tag) {
    var tagVal = App.db.getManageJournalNodeWizardConfigTag(tag);
    this.set('content.' + tag, tagVal);
  },

  saveNameServiceIds: function saveNameServiceIds(nameServiceIds) {
    this.setDBProperty('nameServiceIds', nameServiceIds);
    this.set('content.nameServiceIds', nameServiceIds);
  },

  loadNameServiceIds: function loadNameServiceIds() {
    var nameServiceIds = this.getDBProperty('nameServiceIds');
    this.set('content.nameServiceIds', nameServiceIds);
  },

  /**
   * Remove all loaded data.
   * Created as copy for App.router.clearAllSteps
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  clearTasksData: function clearTasksData() {
    this.saveTasksStatuses(undefined);
    this.saveRequestIds(undefined);
    this.saveTasksRequestIds(undefined);
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    App.db.data.Installer = {};
    this.resetDbNamespace();
    App.router.get('updateController').updateAll();
  }
});

});

require.register("controllers/main/admin/highAvailability/nameNode/rollbackHA/rollback_wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RollbackHighAvailabilityWizardController = App.WizardController.extend({

  name: 'rollbackHighAvailabilityWizardController',

  totalSteps: 3,

  /**
   * Used for hiding back button in wizard
   */
  hideBackButton: true,

  content: Em.Object.create({
    controllerName: 'RollbackHighAvailabilityWizardController',
    cluster: null,
    masterComponentHosts: null,
    serviceName: 'MISC',
    hdfsUser: "hdfs",
    nameServiceId: '',
    selectedAddNNHost: null,
    selectedSNNHost: null,
    activeNNHost: null
  }),

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      clusterState: 'ROLLBACK_HIGH_AVAILABILITY',
      wizardControllerName: 'rollbackHighAvailabilityWizardController',
      localdb: App.db.data
    });
  },

  /**
   * return new object extended from clusterStatusTemplate
   * @return Object
   */
  getCluster: function getCluster() {
    return jQuery.extend({}, this.get('clusterStatusTemplate'), { name: App.router.getClusterName() });
  },

  /**
   * save status of the cluster.
   * @param clusterStatus object with status,requestId fields.
   */
  saveClusterStatus: function saveClusterStatus(clusterStatus) {
    var oldStatus = this.toObject(this.get('content.cluster'));
    clusterStatus = jQuery.extend(oldStatus, clusterStatus);
    if (clusterStatus.requestId) {
      clusterStatus.requestId.forEach(function (requestId) {
        if (clusterStatus.oldRequestsId.indexOf(requestId) === -1) {
          clusterStatus.oldRequestsId.push(requestId);
        }
      }, this);
    }
    this.set('content.cluster', clusterStatus);
    this.save('cluster');
  },

  saveTasksStatuses: function saveTasksStatuses(statuses) {
    App.db.setRollbackHighAvailabilityWizardTasksStatuses(statuses);
    this.set('content.tasksStatuses', statuses);
  },

  saveRequestIds: function saveRequestIds(requestIds) {
    App.db.setRollbackHighAvailabilityWizardRequestIds(requestIds);
    this.set('content.requestIds', requestIds);
  },

  saveSelectedSNN: function saveSelectedSNN(addNN) {
    App.db.setRollBackHighAvailabilityWizardSelectedSNN(addNN);
    this.set('content.selectedAddNN', addNN);
  },

  saveSelectedAddNN: function saveSelectedAddNN(sNN) {
    App.db.setRollBackHighAvailabilityWizardSelectedAddNN(sNN);
    this.set('content.selectedSNN', sNN);
  },

  loadAddNNHost: function loadAddNNHost() {
    var addNNHost = App.db.getRollBackHighAvailabilityWizardAddNNHost();
    this.set('content.addNNHost', addNNHost);
  },

  loadSNNHost: function loadSNNHost() {
    var sNNHost = App.db.getRollBackHighAvailabilityWizardSNNHost();
    this.set('content.sNNHost', sNNHost);
  },

  loadTasksStatuses: function loadTasksStatuses() {
    var sNNHost = App.db.getRollbackHighAvailabilityWizardTasksStatuses();
    this.set('content.tasksStatuses', sNNHost);
  },

  loadRequestIds: function loadRequestIds() {
    var requestIds = App.db.getRollbackHighAvailabilityWizardRequestIds();
    this.set('content.requestIds', requestIds);
  },

  /**
   * Load data for all steps until <code>current step</code>
   */
  loadAllPriorSteps: function loadAllPriorSteps() {
    var step = this.get('currentStep');
    switch (step) {
      case '3':
      case '2':
      case '1':
        this.loadSNNHost();
        this.loadAddNNHost();
        this.load('cluster');
    }
  },

  /**
   * Remove all loaded data.
   * Created as copy for App.router.clearAllSteps
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  clearTasksData: function clearTasksData() {
    this.saveTasksStatuses(undefined);
    this.saveRequestIds(undefined);
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.setCurrentStep('1');
    this.clearAllSteps();
    App.router.get('updateController').updateAll();
  }
});

});

require.register("controllers/main/admin/highAvailability/nameNode/rollbackHA/step1_controller", function(exports, require, module) {
"use strict";

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RollbackHighAvailabilityWizardStep1Controller = Em.Controller.extend({

  name: "rollbackHighAvailabilityWizardStep1Controller"

});

});

require.register("controllers/main/admin/highAvailability/nameNode/rollbackHA/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RollbackHighAvailabilityWizardStep2Controller = App.HighAvailabilityWizardStep4Controller.extend({
  name: "rollbackHighAvailabilityWizardStep2Controller",

  pullCheckPointStatus: function pullCheckPointStatus() {
    var hostName = this.get('content.activeNNHost');
    App.ajax.send({
      name: 'admin.high_availability.getNnCheckPointStatus',
      sender: this,
      data: {
        hostName: hostName
      },
      success: 'checkNnCheckPointStatus'
    });
  }
});

});

require.register("controllers/main/admin/highAvailability/nameNode/rollbackHA/step3_controller", function(exports, require, module) {
"use strict";

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RollbackHighAvailabilityWizardStep3Controller = Em.Controller.extend({
  name: "rollbackHighAvailabilityWizardStep3Controller"
});

});

require.register("controllers/main/admin/highAvailability/nameNode/rollback_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
require('controllers/main/admin/highAvailability/progress_controller');

App.HighAvailabilityRollbackController = App.HighAvailabilityProgressPageController.extend({

  name: "highAvailabilityRollbackController",

  failedTask: null,
  configsSaved: false,
  deletedHdfsClients: 0,
  numOfDelOperations: 0,
  isRollback: true,
  hostsToPerformDel: [],

  commands: ['stopAllServices', 'restoreHBaseConfigs', 'restoreAccumuloConfigs', 'restoreHawqConfigs', 'stopFailoverControllers', 'deleteFailoverControllers', 'deletePXF', 'stopStandbyNameNode', 'stopNameNode', 'restoreHDFSConfigs', 'enableSecondaryNameNode', 'stopJournalNodes', 'deleteJournalNodes', 'deleteAdditionalNameNode', 'startAllServices'],

  loadStep: function loadStep() {
    var self = this;
    this.initData().done(function () {
      self.clearStep();
      self.loadTasks();
      self.addObserver('tasks.@each.status', self, 'onTaskStatusChange');
      self.onTaskStatusChange();
    });
  },

  initData: function initData() {
    var self = this,
        dfd = $.Deferred();
    this.loadMasterComponentHosts().done(function () {
      self.loadFailedTask();
      self.loadHdfsClientHosts();
      dfd.resolve();
    });
    return dfd.promise();
  },

  setCommandsAndTasks: function setCommandsAndTasks(tmpTasks) {
    var fTask = this.get('failedTask');
    var commandsArray = ['deleteSNameNode', 'startAllServices', 'reconfigureHBase', 'reconfigureAMS', 'reconfigureAccumulo', 'reconfigureHawq', 'installPXF', 'startZKFC', 'installZKFC', 'startSecondNameNode', 'startNameNode', 'startZooKeeperServers', 'reconfigureHDFS', 'disableSNameNode', 'startJournalNodes', 'installJournalNodes', 'installNameNode', 'stopAllServices'];
    var index = commandsArray.indexOf(fTask.command);

    if (index > commandsArray.indexOf('startSecondNameNode')) {
      --index;
    }
    var newCommands = this.get('commands').splice(index);
    this.set('commands', newCommands);
    var newTasks = tmpTasks.splice(index);
    for (var i = 0; i < newTasks.length; i++) {
      newTasks[i].id = i;
    }
    this.set('tasks', newTasks);
    var pxfTask = this.get('tasks').findProperty('command', 'deletePXF');
    if (!App.Service.find().someProperty('serviceName', 'PXF') && pxfTask) {
      this.get('tasks').splice(pxfTask.get('id'), 1);
    }
    var hbaseTask = this.get('tasks').findProperty('command', 'restoreHBaseConfigs');
    if (!App.Service.find().someProperty('serviceName', 'HBASE') && hbaseTask) {
      this.get('tasks').splice(hbaseTask.get('id'), 1);
    }
    var accumuloTask = this.get('tasks').findProperty('command', 'restoreAccumuloConfigs');
    if (!App.Service.find().someProperty('serviceName', 'ACCUMULO') && accumuloTask) {
      this.get('tasks').splice(accumuloTask.get('id'), 1);
    }
    var hawqTask = this.get('tasks').findProperty('command', 'restoreHawqConfigs');
    if (!App.Service.find().someProperty('serviceName', 'HAWQ') && hawqTask) {
      this.get('tasks').splice(hawqTask.get('id'), 1);
    }
  },

  clearStep: function clearStep() {
    this.set('isSubmitDisabled', true);
    this.set('tasks', []);
    this.set('logs', []);
    this.set('currentRequestIds', []);
    var commands = this.get('commands');
    var tmpTasks = [];
    for (var i = 0; i < commands.length; i++) {
      tmpTasks.pushObject(Ember.Object.create({
        title: Em.I18n.t('admin.highAvailability.rollback.task' + i + '.title'),
        status: 'PENDING',
        id: i,
        command: commands[i],
        showRetry: false,
        showRollback: false,
        showSkip: false,
        name: Em.I18n.t('admin.highAvailability.rollback.task' + i + '.title'),
        displayName: Em.I18n.t('admin.highAvailability.rollback.task' + i + '.title'),
        progress: 0,
        isRunning: false,
        hosts: []
      }));
    }
    this.setCommandsAndTasks(tmpTasks);
  },

  onTaskStatusChange: function onTaskStatusChange() {
    if (!this.get('tasks').someProperty('status', 'IN_PROGRESS') && !this.get('tasks').someProperty('status', 'QUEUED') && !this.get('tasks').someProperty('status', 'FAILED')) {
      var nextTask = this.get('tasks').findProperty('status', 'PENDING');
      if (nextTask) {
        this.set('status', 'IN_PROGRESS');
        this.setTaskStatus(nextTask.get('id'), 'QUEUED');
        this.set('currentTaskId', nextTask.get('id'));
        this.runTask(nextTask.get('id'));
      } else {
        this.set('status', 'COMPLETED');
        this.set('isSubmitDisabled', false);
      }
    } else if (this.get('tasks').someProperty('status', 'FAILED')) {
      this.set('status', 'FAILED');
      this.get('tasks').findProperty('status', 'FAILED').set('showRetry', true);
      this.get('tasks').findProperty('status', 'FAILED').set('showSkip', true);
    }
    this.get('tasks').filterProperty('status', 'COMPLETED').setEach('showRetry', false);
    this.get('tasks').filterProperty('status', 'COMPLETED').setEach('showSkip', false);

    var statuses = this.get('tasks').mapProperty('status');
    var logs = this.get('tasks').mapProperty('hosts');
    var requestIds = this.get('currentRequestIds');
    this.saveTasksStatuses(statuses);
    this.saveRequestIds(requestIds);
    this.saveLogs(logs);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      clusterState: 'HIGH_AVAILABILITY_ROLLBACK',
      wizardControllerName: 'highAvailabilityRollbackController',
      localdb: App.db.data
    });
  },

  skipTask: function skipTask() {
    var task = this.get('tasks').findProperty('status', 'FAILED');
    task.set('showRetry', false);
    task.set('showSkip', false);
    task.set('status', 'COMPLETED');
  },

  retryTask: function retryTask() {
    var task = this.get('tasks').findProperty('status', 'FAILED');
    task.set('showRetry', false);
    task.set('showSkip', false);
    task.set('status', 'PENDING');
  },

  onTaskCompleted: function onTaskCompleted() {
    var curTaskStatus = this.getTaskStatus(this.get('currentTaskId'));
    if (curTaskStatus != 'FAILED' && curTaskStatus != 'TIMEDOUT' && curTaskStatus != 'ABORTED') {
      this.setTaskStatus(this.get('currentTaskId'), 'COMPLETED');
    }
  },

  getTaskStatus: function getTaskStatus(taskId) {
    return this.get('tasks').findProperty('id', taskId).get('status');
  },

  loadFailedTask: function loadFailedTask() {
    var failedTask = App.db.getHighAvailabilityWizardFailedTask();
    this.set('failedTask', failedTask);
  },

  done: function done() {
    if (!this.get('isSubmitDisabled')) {
      this.removeObserver('tasks.@each.status', this, 'onTaskStatusChange');
      this.popup.proceedOnClose();
    }
  },

  stopAllServices: function stopAllServices() {
    App.ajax.send({
      name: 'common.services.update',
      data: {
        context: "Stop all services",
        "ServiceInfo": {
          "state": "INSTALLED"
        }
      },
      sender: this,
      success: 'startPolling',
      error: 'onTaskError'
    });
  },
  restoreHBaseConfigs: function restoreHBaseConfigs() {
    this.loadConfigTag("hbaseSiteTag");
    var hbaseSiteTag = this.get("content.hbaseSiteTag");
    App.ajax.send({
      name: 'admin.high_availability.load_hbase_configs',
      sender: this,
      data: {
        hbaseSiteTag: hbaseSiteTag
      },
      success: 'onLoadHbaseConfigs',
      error: 'onTaskError'
    });
  },
  restoreAccumuloConfigs: function restoreAccumuloConfigs() {
    this.loadConfigTag("accumuloSiteTag");
    var accumuloSiteTag = this.get("content.accumuloSiteTag");
    App.ajax.send({
      name: 'admin.high_availability.load_accumulo_configs',
      sender: this,
      data: {
        accumuloSiteTag: accumuloSiteTag
      },
      success: 'onLoadAccumuloConfigs',
      error: 'onTaskError'
    });
  },
  restoreHawqConfigs: function restoreHawqConfigs() {
    var tags = ['hawqSiteTag', 'hdfsClientTag'];
    tags.forEach(function (tagName) {
      var tag = this.get("content." + tagName);
      App.ajax.send({
        name: 'admin.high_availability.load_hawq_configs',
        sender: this,
        data: {
          tagName: tag
        },
        success: 'onLoadHawqConfigs',
        error: 'onTaskError'
      });
    }, this);
  },

  deletePXF: function deletePXF() {
    var secondNameNodeHost = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').findProperty('isInstalled', false).mapProperty('hostName');
    var pxfComponent = this.getSlaveComponentHosts().findProperty('componentName', 'PXF');
    var dataNodeComponent = this.getSlaveComponentHosts().findProperty('componentName', 'DATANODE');

    var host, i;

    // check if PXF is already installed on the host assigned for additional NameNode
    var pxfComponentInstalled = false;
    for (i = 0; i < pxfComponent.hosts.length; i++) {
      host = pxfComponent.hosts[i];
      if (host.hostName === secondNameNodeHost) {
        pxfComponentInstalled = true;
        break;
      }
    }

    // check if DATANODE is already installed on the host assigned for additional NameNode
    var dataNodeComponentInstalled = false;
    for (i = 0; i < dataNodeComponent.hosts.length; i++) {
      host = dataNodeComponent.hosts[i];
      if (host.hostName === secondNameNodeHost) {
        dataNodeComponentInstalled = true;
        break;
      }
    }

    // if no DATANODE exists on that host, remove PXF
    if (!dataNodeComponentInstalled && pxfComponentInstalled) {
      this.updateComponent('PXF', secondNameNodeHost, "PXF", "Stop");
      this.checkBeforeDelete('PXF', secondNameNodeHost);
    }
  },

  stopFailoverControllers: function stopFailoverControllers() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').mapProperty('hostName');
    this.updateComponent('ZKFC', hostNames, "HDFS", "Stop");
  },
  deleteFailoverControllers: function deleteFailoverControllers() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').mapProperty('hostName');
    this.checkBeforeDelete('ZKFC', hostNames);
  },
  stopStandbyNameNode: function stopStandbyNameNode() {
    var hostName = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').findProperty('isInstalled', false).hostName;
    this.updateComponent('NAMENODE', hostName, "HDFS", "Stop");
  },
  stopNameNode: function stopNameNode() {
    var hostName = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').findProperty('isInstalled', true).hostName;
    this.updateComponent('NAMENODE', hostName, "HDFS", "Stop");
  },
  restoreHDFSConfigs: function restoreHDFSConfigs() {
    this.unInstallHDFSClients();
  },
  enableSecondaryNameNode: function enableSecondaryNameNode() {
    var hostName = this.get('content.masterComponentHosts').findProperty('component', 'SECONDARY_NAMENODE').hostName;
    this.updateComponent('SECONDARY_NAMENODE', hostName, "HDFS", "Install", hostName.length);
  },
  stopJournalNodes: function stopJournalNodes() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'JOURNALNODE').mapProperty('hostName');
    this.updateComponent('JOURNALNODE', hostNames, "HDFS", "Stop");
  },
  deleteJournalNodes: function deleteJournalNodes() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'JOURNALNODE').mapProperty('hostName');
    this.unInstallComponent('JOURNALNODE', hostNames);
  },
  deleteAdditionalNameNode: function deleteAdditionalNameNode() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').findProperty('isInstalled', false).mapProperty('hostName');
    this.unInstallComponent('NAMENODE', hostNames);
  },
  startAllServices: function startAllServices() {
    App.ajax.send({
      name: 'common.services.update',
      data: {
        context: "Start all services",
        "ServiceInfo": {
          "state": "STARTED"
        }
      },
      sender: this,
      success: 'startPolling',
      error: 'onTaskError'
    });
  },

  onLoadHbaseConfigs: function onLoadHbaseConfigs(data) {
    var hbaseSiteProperties = data.items.findProperty('type', 'hbase-site').properties;
    App.ajax.send({
      name: 'admin.save_configs',
      sender: this,
      data: {
        siteName: 'hbase-site',
        properties: hbaseSiteProperties
      },
      success: 'onTaskCompleted',
      error: 'onTaskError'
    });
  },
  onLoadAccumuloConfigs: function onLoadAccumuloConfigs(data) {
    var accumuloSiteProperties = data.items.findProperty('type', 'accumulo-site').properties;
    App.ajax.send({
      name: 'admin.save_configs',
      sender: this,
      data: {
        siteName: 'accumulo-site',
        properties: accumuloSiteProperties
      },
      success: 'onTaskCompleted',
      error: 'onTaskError'
    });
  },

  onLoadHawqConfigs: function onLoadHawqConfigs(data) {
    var hawqSiteProperties = data.items.findProperty('type', 'hawq-site').properties;
    App.ajax.send({
      name: 'admin.save_configs',
      sender: this,
      data: {
        siteName: 'hawq-site',
        properties: hawqSiteProperties
      },
      success: 'onTaskCompleted',
      error: 'onTaskError'
    });
  },
  onDeletedHDFSClient: function onDeletedHDFSClient() {
    var deletedHdfsClients = this.get('deletedHdfsClients');
    var hostName = this.get("content.hdfsClientHostNames");
    var notDeletedHdfsClients = hostName.length - deletedHdfsClients;
    if (notDeletedHdfsClients > 1 && hostName.length != 1) {
      this.set('deletedHdfsClients', deletedHdfsClients + 1);
      return;
    }
    this.loadConfigTag("hdfsSiteTag");
    this.loadConfigTag("coreSiteTag");
    var hdfsSiteTag = this.get("content.hdfsSiteTag");
    var coreSiteTag = this.get("content.coreSiteTag");
    App.ajax.send({
      name: 'admin.high_availability.load_configs',
      sender: this,
      data: {
        hdfsSiteTag: hdfsSiteTag,
        coreSiteTag: coreSiteTag
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadConfigs: function onLoadConfigs(data) {
    this.set('configsSaved', false);
    App.ajax.send({
      name: 'admin.save_configs',
      sender: this,
      data: {
        siteName: 'hdfs-site',
        properties: data.items.findProperty('type', 'hdfs-site').properties
      },
      success: 'onHdfsConfigsSaved',
      error: 'onTaskError'
    });
    App.ajax.send({
      name: 'admin.save_configs',
      sender: this,
      data: {
        siteName: 'core-site',
        properties: data.items.findProperty('type', 'core-site').properties
      },
      success: 'onHdfsConfigsSaved',
      error: 'onTaskError'
    });
  },

  onHdfsConfigsSaved: function onHdfsConfigsSaved() {
    if (!this.get('configsSaved')) {
      this.set('configsSaved', true);
      return;
    }
    this.onTaskCompleted();
  },

  unInstallHDFSClients: function unInstallHDFSClients() {
    var hostName = this.get("content.hdfsClientHostNames");
    for (var i = 0; i < hostName.length; i++) {
      App.ajax.send({
        name: 'common.delete.host_component',
        sender: this,
        data: {
          componentName: 'HDFS_CLIENT',
          hostName: hostName[i]
        },
        success: 'onDeletedHDFSClient',
        error: 'onTaskError'
      });
    }
  },

  unInstallComponent: function unInstallComponent(componentName, hostName) {
    if (!(hostName instanceof Array)) {
      hostName = [hostName];
    }
    for (var i = 0; i < hostName.length; i++) {
      App.ajax.send({
        name: 'common.host.host_component.passive',
        sender: this,
        data: {
          hostName: hostName[i],
          componentName: componentName,
          passive_state: "ON",
          taskNum: hostName.length,
          callback: 'checkBeforeDelete'
        },
        success: 'checkResult',
        error: 'checkResult'
      });
    }
  },

  checkBeforeDelete: function checkBeforeDelete(componentName, hostName) {
    this.set('hostsToPerformDel', []);
    if (!(hostName instanceof Array)) {
      hostName = [hostName];
    }
    for (var i = 0; i < hostName.length; i++) {
      App.ajax.send({
        name: 'admin.high_availability.getHostComponent',
        sender: this,
        data: {
          componentName: componentName,
          hostName: hostName[i],
          taskNum: hostName.length,
          callback: 'deleteComponent'
        },
        success: 'checkResult',
        error: 'checkResult'
      });
    }
  },

  checkResult: function checkResult() {
    var callback = arguments[2].callback;
    var hostName = arguments[2].hostName;
    var componentName = arguments[2].componentName;
    var taskNum = arguments[2].taskNum;
    var hostsToPerformDel = this.get('hostsToPerformDel');
    if (arguments[1] != 'error') {
      hostsToPerformDel.push({
        hostName: hostName,
        isOnHost: true
      });
    } else {
      hostsToPerformDel.push({
        hostName: 'error',
        isOnHost: false
      });
    }
    if (hostsToPerformDel.length == taskNum) {
      var hostsForDel = hostsToPerformDel.filterProperty('isOnHost', true).mapProperty('hostName');
      this.set('hostsToPerformDel', []);
      if (hostsForDel.length == 0) {
        this.onTaskCompleted();
        return;
      }
      this[callback](componentName, hostsForDel);
    }
  },

  deleteComponent: function deleteComponent(componentName, hostName) {
    if (!(hostName instanceof Array)) {
      hostName = [hostName];
    }
    this.set('numOfDelOperations', hostName.length);
    for (var i = 0; i < hostName.length; i++) {
      App.ajax.send({
        name: 'common.delete.host_component',
        sender: this,
        data: {
          componentName: componentName,
          hostName: hostName[i]
        },
        success: 'onDeleteComplete',
        error: 'onTaskError'
      });
    }
  },

  onDeleteComplete: function onDeleteComplete() {
    var leftOp = this.get('numOfDelOperations');
    if (leftOp > 1) {
      this.set('numOfDelOperations', leftOp - 1);
      return;
    }
    this.onTaskCompleted();
  }

});

});

require.register("controllers/main/admin/highAvailability/nameNode/step1_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var validator = require('utils/validator');

require('controllers/main/admin/serviceAccounts_controller');

App.HighAvailabilityWizardStep1Controller = Em.Controller.extend({
  name: "highAvailabilityWizardStep1Controller",

  isHawqInstalled: false,

  isNameServiceIdValid: function () {
    return validator.isValidNameServiceId(this.get('content.nameServiceId'));
  }.property('content.nameServiceId'),

  setHawqInstalled: function setHawqInstalled() {
    this.set('isHawqInstalled', App.Service.find().someProperty('serviceName', 'HAWQ'));
  },

  next: function next() {
    if (this.get('isNameServiceIdValid')) {
      App.router.send('next');
    }
  }

});

});

require.register("controllers/main/admin/highAvailability/nameNode/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

require('controllers/wizard/step5_controller');

App.HighAvailabilityWizardStep2Controller = Em.Controller.extend(App.BlueprintMixin, App.AssignMasterComponents, {

  name: "highAvailabilityWizardStep2Controller",

  useServerValidation: false,

  mastersToShow: ['NAMENODE', 'JOURNALNODE'],

  mastersToAdd: ['NAMENODE', 'JOURNALNODE', 'JOURNALNODE', 'JOURNALNODE'],

  showCurrentPrefix: ['NAMENODE'],

  showAdditionalPrefix: ['NAMENODE'],

  showInstalledMastersFirst: true,

  JOURNALNODES_COUNT_MINIMUM: 3, // TODO get this from stack

  renderComponents: function renderComponents(masterComponents) {
    this._super(masterComponents);
    this.showHideJournalNodesAddRemoveControl();
  },

  addComponent: function addComponent(componentName) {
    this._super(componentName);
    this.showHideJournalNodesAddRemoveControl();
  },

  removeComponent: function removeComponent(componentName, serviceComponentId) {
    this._super(componentName, serviceComponentId);
    this.showHideJournalNodesAddRemoveControl();
  },

  showHideJournalNodesAddRemoveControl: function showHideJournalNodesAddRemoveControl() {
    var jns = this.get('selectedServicesMasters').filterProperty('component_name', 'JOURNALNODE');
    var maxNumMasters = this.getMaxNumberOfMasters('JOURNALNODE');
    var showRemoveControl = jns.get('length') > this.get('JOURNALNODES_COUNT_MINIMUM');
    var showAddControl = jns.get('length') < maxNumMasters;
    jns.forEach(function (item) {
      item.set('showAddControl', false);
      item.set('showRemoveControl', showRemoveControl);
    });
    jns.set('lastObject.showAddControl', showAddControl);
  }

});

});

require.register("controllers/main/admin/highAvailability/nameNode/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @typedef {object} nnHaConfigDependencies
 * @property {string} namespaceId
 * @property {object} serverConfigs
 * @property {string|number} nnHttpPort
 * @property {string|number} nnHttpsPort
 * @property {string|number} nnRpcPort
 * @property {string|number} zkClientPort
 */

var App = require('app');

require('utils/configs/nn_ha_config_initializer');

App.HighAvailabilityWizardStep3Controller = Em.Controller.extend({
  name: "highAvailabilityWizardStep3Controller",
  selectedService: null,
  stepConfigs: [],
  serverConfigData: {},
  haConfig: $.extend(true, {}, require('data/configs/wizards/ha_properties').haConfig),
  once: false,
  isLoaded: false,
  isNextDisabled: Em.computed.not('isLoaded'),
  versionLoaded: true,
  hideDependenciesInfoBar: true,

  /**
   * Map of sites and properties to delete
   * @type Object
   */
  configsToRemove: {
    'hdfs-site': ['dfs.namenode.secondary.http-address', 'dfs.namenode.rpc-address', 'dfs.namenode.http-address', 'dfs.namenode.https-address']
  },

  clearStep: function clearStep() {
    this.get('stepConfigs').clear();
    this.set('serverConfigData', {});
  },

  loadStep: function loadStep() {
    this.clearStep();
    this.loadConfigsTags();
  },

  loadConfigsTags: function loadConfigsTags() {
    return App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'onLoadConfigsTags',
      error: 'onTaskError'
    });
  },

  onLoadConfigsTags: function onLoadConfigsTags(data) {
    var urlParams = [];
    var hdfsSiteTag = data.Clusters.desired_configs['hdfs-site'].tag;
    var coreSiteTag = data.Clusters.desired_configs['core-site'].tag;
    var zkSiteTag = data.Clusters.desired_configs['zoo.cfg'].tag;

    urlParams.push('(type=hdfs-site&tag=' + hdfsSiteTag + ')');
    urlParams.push('(type=core-site&tag=' + coreSiteTag + ')');
    urlParams.push('(type=zoo.cfg&tag=' + zkSiteTag + ')');
    this.set("hdfsSiteTag", { name: "hdfsSiteTag", value: hdfsSiteTag });
    this.set("coreSiteTag", { name: "coreSiteTag", value: coreSiteTag });
    this.set("zkSiteTag", { name: "zkSiteTag", value: zkSiteTag });

    if (App.Service.find().someProperty('serviceName', 'HBASE')) {
      var hbaseSiteTag = data.Clusters.desired_configs['hbase-site'].tag;
      urlParams.push('(type=hbase-site&tag=' + hbaseSiteTag + ')');
      this.set("hbaseSiteTag", { name: "hbaseSiteTag", value: hbaseSiteTag });
    }
    if (App.Service.find().someProperty('serviceName', 'ACCUMULO')) {
      var accumuloSiteTag = data.Clusters.desired_configs['accumulo-site'].tag;
      urlParams.push('(type=accumulo-site&tag=' + accumuloSiteTag + ')');
      this.set("accumuloSiteTag", { name: "accumuloSiteTag", value: accumuloSiteTag });
    }
    if (App.Service.find().someProperty('serviceName', 'AMBARI_METRICS')) {
      var amsHbaseSiteTag = data.Clusters.desired_configs['ams-hbase-site'].tag;
      urlParams.push('(type=ams-hbase-site&tag=' + amsHbaseSiteTag + ')');
      this.set("amsHbaseSiteTag", { name: "amsHbaseSiteTag", value: amsHbaseSiteTag });
    }
    if (App.Service.find().someProperty('serviceName', 'HAWQ')) {
      var hawqSiteTag = data.Clusters.desired_configs['hawq-site'].tag;
      urlParams.push('(type=hawq-site&tag=' + hawqSiteTag + ')');
      this.set("hawqSiteTag", { name: "hawqSiteTag", value: hawqSiteTag });
      var hdfsClientTag = data.Clusters.desired_configs['hdfs-client'].tag;
      urlParams.push('(type=hdfs-client&tag=' + hdfsClientTag + ')');
      this.set("hdfsClientTag", { name: "hdfsClientTag", value: hdfsClientTag });
    }
    if (App.Service.find().someProperty('serviceName', 'RANGER')) {
      var rangerEnvTag = data.Clusters.desired_configs['ranger-env'].tag;
      urlParams.push('(type=ranger-env&tag=' + rangerEnvTag + ')');
      this.set("rangerEnvTag", { name: "rangerEnvTag", value: rangerEnvTag });
      if ('ranger-hdfs-plugin-properties' in data.Clusters.desired_configs) {
        var rangerHdfsPluginPropertiesTag = data.Clusters.desired_configs['ranger-hdfs-plugin-properties'].tag;
        urlParams.push('(type=ranger-hdfs-plugin-properties&tag=' + rangerHdfsPluginPropertiesTag + ')');
        this.set("rangerHdfsPluginPropertiesTag", {
          name: "rangerHdfsPluginPropertiesTag",
          value: rangerHdfsPluginPropertiesTag
        });
      }
      if ('ranger-hdfs-audit' in data.Clusters.desired_configs) {
        var rangerHdfsAuditTag = data.Clusters.desired_configs['ranger-hdfs-audit'].tag;
        urlParams.push('(type=ranger-hdfs-audit&tag=' + rangerHdfsAuditTag + ')');
        this.set("rangerHdfsAuditTag", { name: "rangerHdfsAuditTag", value: rangerHdfsAuditTag });
      }
      if ('ranger-yarn-audit' in data.Clusters.desired_configs) {
        var yarnAuditTag = data.Clusters.desired_configs['ranger-yarn-audit'].tag;
        urlParams.push('(type=ranger-yarn-audit&tag=' + yarnAuditTag + ')');
        this.set("yarnAuditTag", { name: "yarnAuditTag", value: yarnAuditTag });
      }
      if (App.Service.find().someProperty('serviceName', 'HBASE')) {
        if ('ranger-hbase-audit' in data.Clusters.desired_configs) {
          var rangerHbaseAuditTag = data.Clusters.desired_configs['ranger-hbase-audit'].tag;
          urlParams.push('(type=ranger-hbase-audit&tag=' + rangerHbaseAuditTag + ')');
          this.set("rangerHbaseAuditTag", { name: "rangerHbaseAuditTag", value: rangerHbaseAuditTag });
        }
        if ('ranger-hbase-plugin-properties' in data.Clusters.desired_configs) {
          var rangerHbasePluginPropertiesTag = data.Clusters.desired_configs['ranger-hbase-plugin-properties'].tag;
          urlParams.push('(type=ranger-hbase-plugin-properties&tag=' + rangerHbasePluginPropertiesTag + ')');
          this.set("rangerHbasePluginPropertiesTag", {
            name: "rangerHbasePluginPropertiesTag",
            value: rangerHbasePluginPropertiesTag
          });
        }
      }
      if (App.Service.find().someProperty('serviceName', 'KAFKA')) {
        if ('ranger-kafka-audit' in data.Clusters.desired_configs) {
          var rangerKafkaAuditTag = data.Clusters.desired_configs['ranger-kafka-audit'].tag;
          urlParams.push('(type=ranger-kafka-audit&tag=' + rangerKafkaAuditTag + ')');
          this.set("rangerKafkaAuditTag", { name: "rangerKafkaAuditTag", value: rangerKafkaAuditTag });
        }
      }
      if (App.Service.find().someProperty('serviceName', 'KNOX')) {
        if ('ranger-knox-audit' in data.Clusters.desired_configs) {
          var rangerKnoxAuditTag = data.Clusters.desired_configs['ranger-knox-audit'].tag;
          urlParams.push('(type=ranger-knox-audit&tag=' + rangerKnoxAuditTag + ')');
          this.set("rangerKnoxAuditTag", { name: "rangerKnoxAuditTag", value: rangerKnoxAuditTag });
        }
        if ('ranger-knox-plugin-properties' in data.Clusters.desired_configs) {
          var rangerKnoxPluginPropertiesTag = data.Clusters.desired_configs['ranger-knox-plugin-properties'].tag;
          urlParams.push('(type=ranger-knox-plugin-properties&tag=' + rangerKnoxPluginPropertiesTag + ')');
          this.set("rangerKnoxPluginPropertiesTag", {
            name: "rangerKnoxPluginPropertiesTag",
            value: rangerKnoxPluginPropertiesTag
          });
        }
      }
      if (App.Service.find().someProperty('serviceName', 'STORM')) {
        if ('ranger-storm-audit' in data.Clusters.desired_configs) {
          var rangerStormAuditTag = data.Clusters.desired_configs['ranger-storm-audit'].tag;
          urlParams.push('(type=ranger-storm-audit&tag=' + rangerStormAuditTag + ')');
          this.set("rangerStormAuditTag", { name: "rangerStormAuditTag", value: rangerStormAuditTag });
        }
        if ('ranger-storm-plugin-properties' in data.Clusters.desired_configs) {
          var rangerStormPluginPropertiesTag = data.Clusters.desired_configs['ranger-storm-plugin-properties'].tag;
          urlParams.push('(type=ranger-storm-plugin-properties&tag=' + rangerStormPluginPropertiesTag + ')');
          this.set("rangerStormPluginPropertiesTag", {
            name: "rangerStormPluginPropertiesTag",
            value: rangerStormPluginPropertiesTag
          });
        }
      }
      if (App.Service.find().someProperty('serviceName', 'ATLAS')) {
        if ('ranger-atlas-audit' in data.Clusters.desired_configs) {
          var rangerAtlasAuditTag = data.Clusters.desired_configs['ranger-atlas-audit'].tag;
          urlParams.push('(type=ranger-atlas-audit&tag=' + rangerAtlasAuditTag + ')');
          this.set("rangerAtlasAuditTag", { name: "rangerAtlasAuditTag", value: rangerAtlasAuditTag });
        }
      }
      if (App.Service.find().someProperty('serviceName', 'HIVE')) {
        if ('ranger-hive-audit' in data.Clusters.desired_configs) {
          var rangerHiveAuditTag = data.Clusters.desired_configs['ranger-hive-audit'].tag;
          urlParams.push('(type=ranger-hive-audit&tag=' + rangerHiveAuditTag + ')');
          this.set("rangerHiveAuditTag", { name: "rangerHiveAuditTag", value: rangerHiveAuditTag });
        }
        if ('ranger-hive-plugin-properties' in data.Clusters.desired_configs) {
          var rangerHivePluginPropertiesTag = data.Clusters.desired_configs['ranger-hive-plugin-properties'].tag;
          urlParams.push('(type=ranger-hive-plugin-properties&tag=' + rangerHivePluginPropertiesTag + ')');
          this.set("rangerHivePluginPropertiesTag", {
            name: "rangerHivePluginPropertiesTag",
            value: rangerHivePluginPropertiesTag
          });
        }
      }
      if (App.Service.find().someProperty('serviceName', 'RANGER_KMS')) {
        if ('ranger-kms-audit' in data.Clusters.desired_configs) {
          var rangerKMSAuditTag = data.Clusters.desired_configs['ranger-kms-audit'].tag;
          urlParams.push('(type=ranger-kms-audit&tag=' + rangerKMSAuditTag + ')');
          this.set("rangerKMSAuditTag", { name: "rangerKMSAuditTag", value: rangerKMSAuditTag });
        }
      }
    }
    App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        urlParams: urlParams.join('|')
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadConfigs: function onLoadConfigs(data) {
    this.set('serverConfigData', data);
    this.removeConfigs(this.get('configsToRemove'), data);
    this.tweakServiceConfigs(this.get('haConfig.configs'));
    this.renderServiceConfigs(this.get('haConfig'));
    this.set('isLoaded', true);
  },

  /**
   * Generate set of data used to correctly initialize config values and names
   *
   * @returns {nnHaConfigDependencies}
   * @private
   * @method _prepareDependencies
   */
  _prepareDependencies: function _prepareDependencies() {
    var ret = {};
    var configsFromServer = this.get('serverConfigData.items');
    ret.namespaceId = this.get('content.nameServiceId');
    ret.serverConfigs = configsFromServer;
    var hdfsConfigs = configsFromServer.findProperty('type', 'hdfs-site').properties;
    var zkConfigs = configsFromServer.findProperty('type', 'zoo.cfg').properties;

    var dfsHttpA = hdfsConfigs['dfs.namenode.http-address'];
    ret.nnHttpPort = dfsHttpA ? dfsHttpA.split(':')[1] : 50070;

    var dfsHttpsA = hdfsConfigs['dfs.namenode.https-address'];
    ret.nnHttpsPort = dfsHttpsA ? dfsHttpsA.split(':')[1] : 50470;

    var dfsRpcA = hdfsConfigs['dfs.namenode.rpc-address'];
    ret.nnRpcPort = dfsRpcA ? dfsRpcA.split(':')[1] : 8020;

    ret.zkClientPort = zkConfigs['clientPort'] ? zkConfigs['clientPort'] : 2181;

    return ret;
  },

  /**
   * Generate set of data with information about cluster topology
   * Used in the configs' initialization process
   *
   * @returns {extendedTopologyLocalDB}
   * @private
   * @method _prepareLocalDB
   */
  _prepareLocalDB: function _prepareLocalDB() {
    var localDB = this.get('content').getProperties(['masterComponentHosts', 'slaveComponentHosts', 'hosts']);
    localDB.installedServices = App.Service.find().mapProperty('serviceName');
    return localDB;
  },

  tweakServiceConfigs: function tweakServiceConfigs(configs) {
    var localDB = this._prepareLocalDB();
    var dependencies = this._prepareDependencies();

    configs.forEach(function (config) {
      App.NnHaConfigInitializer.initialValue(config, localDB, dependencies);
      config.isOverridable = false;
    });

    return configs;
  },

  /**
   * Find and remove config properties in <code>serverConfigData</code>
   * @param configsToRemove - map of config sites and properties to remove
   * @param configs - configuration object
   * @returns {Object}
   */
  removeConfigs: function removeConfigs(configsToRemove, configs) {
    Em.keys(configsToRemove).forEach(function (site) {
      var siteConfigs = configs.items.findProperty('type', site);
      if (siteConfigs) {
        configsToRemove[site].forEach(function (property) {
          delete siteConfigs.properties[property];
        });
      }
    });
    return configs;
  },

  renderServiceConfigs: function renderServiceConfigs(_serviceConfig) {
    var serviceConfig = App.ServiceConfig.create({
      serviceName: _serviceConfig.serviceName,
      displayName: _serviceConfig.displayName,
      configCategories: [],
      showConfig: true,
      configs: []
    });

    _serviceConfig.configCategories.forEach(function (_configCategory) {
      if (App.Service.find().someProperty('serviceName', _configCategory.name)) {
        serviceConfig.configCategories.pushObject(_configCategory);
      }
    }, this);

    this.loadComponentConfigs(_serviceConfig, serviceConfig);

    this.get('stepConfigs').pushObject(serviceConfig);
    this.set('selectedService', this.get('stepConfigs').objectAt(0));
    this.set('once', true);
  },

  /**
   * Load child components to service config object
   * @param _componentConfig
   * @param componentConfig
   */
  loadComponentConfigs: function loadComponentConfigs(_componentConfig, componentConfig) {
    _componentConfig.configs.forEach(function (_serviceConfigProperty) {
      var serviceConfigProperty = App.ServiceConfigProperty.create(_serviceConfigProperty);
      componentConfig.configs.pushObject(serviceConfigProperty);
      serviceConfigProperty.set('isEditable', serviceConfigProperty.get('isReconfigurable'));
    }, this);
  }
});

});

require.register("controllers/main/admin/highAvailability/nameNode/step4_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

require('controllers/main/admin/serviceAccounts_controller');

App.HighAvailabilityWizardStep4Controller = Em.Controller.extend({

  name: "highAvailabilityWizardStep4Controller",

  POLL_INTERVAL: 1000,

  isNextEnabled: false,

  isNameNodeStarted: true,

  pullCheckPointStatus: function pullCheckPointStatus() {
    var hostName = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').findProperty('isInstalled', true).hostName;
    return App.ajax.send({
      name: 'admin.high_availability.getNnCheckPointStatus',
      sender: this,
      data: {
        hostName: hostName
      },
      success: 'checkNnCheckPointStatus'
    });
  },

  checkNnCheckPointStatus: function checkNnCheckPointStatus(data) {
    var _this = this;

    this.set('isNameNodeStarted', data.HostRoles.desired_state === 'STARTED');
    var isNextEnabled = this.getNnCheckPointStatus(data);
    if (isNextEnabled) {
      this.set('isNextEnabled', true);
      return;
    }

    window.setTimeout(function () {
      _this.pullCheckPointStatus();
    }, this.POLL_INTERVAL);
  },

  getNnCheckPointStatus: function getNnCheckPointStatus(data) {
    var isInSafeMode = !Em.isEmpty(Em.get(data, 'metrics.dfs.namenode.Safemode'));
    var journalTransactionInfo = $.parseJSON(Em.get(data, 'metrics.dfs.namenode.JournalTransactionInfo'));
    // in case when transaction info absent or invalid return 2 which will return false in next `if` statement
    journalTransactionInfo = !!journalTransactionInfo ? parseInt(journalTransactionInfo.LastAppliedOrWrittenTxId) - parseInt(journalTransactionInfo.MostRecentCheckpointTxId) : 2;
    return journalTransactionInfo <= 1 && isInSafeMode;
  },

  done: function done() {
    if (this.get('isNextEnabled')) {
      App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
        App.router.send("next");
      });
    }
  }

});

});

require.register("controllers/main/admin/highAvailability/nameNode/step5_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.HighAvailabilityWizardStep5Controller = App.HighAvailabilityProgressPageController.extend({

  name: "highAvailabilityWizardStep5Controller",

  commands: ['stopAllServices', 'installNameNode', 'installJournalNodes', 'reconfigureHDFS', 'startJournalNodes', 'disableSNameNode'],

  hdfsSiteTag: "",
  coreSiteTag: "",

  stopAllServices: function stopAllServices() {
    this.stopServices([], true, true);
  },

  installNameNode: function installNameNode() {
    var hostName = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').findProperty('isInstalled', false).hostName;
    this.createInstallComponentTask('NAMENODE', hostName, "HDFS");
  },

  installJournalNodes: function installJournalNodes() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'JOURNALNODE').mapProperty('hostName');
    this.createInstallComponentTask('JOURNALNODE', hostNames, "HDFS");
  },

  startJournalNodes: function startJournalNodes() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'JOURNALNODE').mapProperty('hostName');
    this.updateComponent('JOURNALNODE', hostNames, "HDFS", "Start");
  },

  disableSNameNode: function disableSNameNode() {
    var hostName = this.get('content.masterComponentHosts').findProperty('component', 'SECONDARY_NAMENODE').hostName;
    return App.ajax.send({
      name: 'common.host.host_component.passive',
      sender: this,
      data: {
        hostName: hostName,
        passive_state: "ON",
        componentName: 'SECONDARY_NAMENODE'
      },
      success: 'onTaskCompleted',
      error: 'onTaskError'
    });
  },

  reconfigureHDFS: function reconfigureHDFS() {
    var data = this.get('content.serviceConfigProperties');
    if (App.get('isKerberosEnabled')) {
      this.reconfigureSecureHDFS();
    } else {
      this.updateConfigProperties(data);
    }
  },

  /**
   * Update service configurations
   * @param {Object} data - config object to update
   */
  updateConfigProperties: function updateConfigProperties(data) {
    var siteNames = ['hdfs-site', 'core-site'].concat(this.getRangerSiteNames(data));
    var note = Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false));
    var configData = this.reconfigureSites(siteNames, data, note);
    return App.ajax.send({
      name: 'common.service.configurations',
      sender: this,
      data: {
        desired_config: configData
      },
      error: 'onTaskError',
      success: 'installHDFSClients'
    });
  },

  getRangerSiteNames: function getRangerSiteNames(data) {
    var siteNames = [];
    if (App.Service.find().someProperty('serviceName', 'RANGER')) {
      var hdfsPluginConfig = data.items.findProperty('type', 'ranger-hdfs-plugin-properties');
      if (hdfsPluginConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in hdfsPluginConfig.properties) {
          siteNames.push('ranger-hdfs-plugin-properties');
        }
      }
      var hdfsAuditConfig = data.items.findProperty('type', 'ranger-hdfs-audit');
      if (hdfsAuditConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in hdfsAuditConfig.properties) {
          siteNames.push('ranger-hdfs-audit');
        }
      }
    }
    return siteNames;
  },

  installHDFSClients: function installHDFSClients() {
    var nnHostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').mapProperty('hostName');
    var jnHostNames = this.get('content.masterComponentHosts').filterProperty('component', 'JOURNALNODE').mapProperty('hostName');
    var hostNames = nnHostNames.concat(jnHostNames).uniq();
    this.createInstallComponentTask('HDFS_CLIENT', hostNames, 'HDFS');
    App.router.get(this.get('content.controllerName')).saveHdfsClientHosts(hostNames);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      clusterState: 'HIGH_AVAILABILITY_DEPLOY',
      wizardControllerName: this.get('content.controllerName'),
      localdb: App.db.data
    });
  },

  /**
   * Process configurations for hdfs on kerberized cluster.
   * Secure properties will be applied to hdfs and core site after installing JournalNode.
   * For this case we need to pull updated configurations from API, merge them with configurations created
   * during `Review` step and store new configurations.
   */
  reconfigureSecureHDFS: function reconfigureSecureHDFS() {
    App.router.get('highAvailabilityWizardStep3Controller').loadConfigsTags.call(this);
  },

  onLoadConfigsTags: function onLoadConfigsTags() {
    App.router.get('highAvailabilityWizardStep3Controller').onLoadConfigsTags.apply(this, [].slice.call(arguments));
  },

  onLoadConfigs: function onLoadConfigs(data) {
    var self = this;
    var configController = App.router.get('highAvailabilityWizardStep3Controller');
    var configItems = data.items.map(function (item) {
      var fileName = Em.get(item, 'type');
      var configTypeObject = self.get('content.serviceConfigProperties').items.findProperty('type', fileName);
      if (configTypeObject) {
        $.extend(item.properties, configTypeObject.properties);
      }
      return item;
    });
    configItems = configController.removeConfigs(configController.get('configsToRemove'), { items: configItems });
    this.updateConfigProperties(configItems);
  }
});

});

require.register("controllers/main/admin/highAvailability/nameNode/step6_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.HighAvailabilityWizardStep6Controller = Em.Controller.extend({

  name: "highAvailabilityWizardStep6Controller",

  POLL_INTERVAL: 1000,

  MINIMAL_JOURNALNODE_COUNT: 3,

  /**
   * Define whether next button should be enabled
   * @type {Boolean}
   */
  isNextEnabled: false,

  /**
   * Counter of initialized JournalNodes
   * @type {Number}
   */
  initJnCounter: 0,

  /**
   * Counter of completed requests
   * @type {Number}
   */
  requestsCounter: 0,

  /**
   * Define whether are some not started JournalNodes
   * @type {Boolean}
   */
  hasStoppedJNs: false,

  /**
   * Current status for step.
   * waiting - for waiting for initializing
   * done - for completed initializing check
   * journalnode_stopped - if there are stopped JournalNode
   * @type {String}
   */
  status: 'waiting',

  loadStep: function loadStep() {
    this.set('status', 'waiting');
    this.set('isNextEnabled', false);
    this.pullCheckPointStatus();
  },

  pullCheckPointStatus: function pullCheckPointStatus() {
    this.set('initJnCounter', 0);
    this.set('requestsCounter', 0);
    this.set('hasStoppedJNs', false);
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', "JOURNALNODE").mapProperty('hostName');
    hostNames.forEach(function (hostName) {
      this.pullEachJnStatus(hostName);
    }, this);
  },

  pullEachJnStatus: function pullEachJnStatus(hostName) {
    return App.ajax.send({
      name: 'admin.high_availability.getJnCheckPointStatus',
      sender: this,
      data: {
        hostName: hostName
      },
      success: 'checkJnCheckPointStatus'
    });
  },

  checkJnCheckPointStatus: function checkJnCheckPointStatus(data) {
    var journalStatusInfo;
    var initJnCounter = 0;

    if (data.metrics && data.metrics.dfs) {
      journalStatusInfo = $.parseJSON(data.metrics.dfs.journalnode.journalsStatus);
      if (journalStatusInfo[this.get('content.nameServiceId')] && journalStatusInfo[this.get('content.nameServiceId')].Formatted === "true") {
        initJnCounter = this.incrementProperty('initJnCounter');
      }
    } else {
      this.set('hasStoppedJNs', true);
    }

    if (this.incrementProperty('requestsCounter') === this.MINIMAL_JOURNALNODE_COUNT) {
      this.resolveJnCheckPointStatus(initJnCounter);
    }
  },

  resolveJnCheckPointStatus: function resolveJnCheckPointStatus(initJnCounter) {
    var self = this;
    if (initJnCounter === this.MINIMAL_JOURNALNODE_COUNT) {
      this.set('status', 'done');
      this.set('isNextEnabled', true);
    } else {
      if (this.get('hasStoppedJNs')) {
        this.set('status', 'journalnode_stopped');
      } else {
        this.set('status', 'waiting');
      }
      window.setTimeout(function () {
        self.pullCheckPointStatus();
      }, self.POLL_INTERVAL);
    }
  },

  done: function done() {
    if (this.get('isNextEnabled')) {
      App.router.send('next');
    }
  }

});

});

require.register("controllers/main/admin/highAvailability/nameNode/step7_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.HighAvailabilityWizardStep7Controller = App.HighAvailabilityProgressPageController.extend({

  name: "highAvailabilityWizardStep7Controller",

  commands: ['startZooKeeperServers', 'startAmbariInfra', 'startMysqlServer', 'startRanger', 'startNameNode'],

  initializeTasks: function initializeTasks() {
    this._super();
    var tasksToRemove = [];

    if (!App.Service.find('AMBARI_INFRA_SOLR').get('isLoaded')) {
      tasksToRemove.push('startAmbariInfra');
    }

    if (App.ClientComponent.getModelByComponentName('RANGER_ADMIN').get('installedCount') === 0) {
      tasksToRemove.push('startRanger');
    }

    if (App.ClientComponent.getModelByComponentName('MYSQL_SERVER').get('installedCount') === 0) {
      tasksToRemove.push('startMysqlServer');
    }

    this.removeTasks(tasksToRemove);
  },

  startAmbariInfra: function startAmbariInfra() {
    this.startServices(false, ['AMBARI_INFRA_SOLR'], true);
  },

  startRanger: function startRanger() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'RANGER_ADMIN').mapProperty('hostName');
    if (hostNames.length) {
      this.updateComponent('RANGER_ADMIN', hostNames, "RANGER", "Start");
    }
  },

  startMysqlServer: function startMysqlServer() {
    var hostNames = App.MasterComponent.find('MYSQL_SERVER').get('hostNames');
    this.updateComponent('MYSQL_SERVER', hostNames, "HIVE", "Start");
  },

  startZooKeeperServers: function startZooKeeperServers() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'ZOOKEEPER_SERVER').mapProperty('hostName');
    this.updateComponent('ZOOKEEPER_SERVER', hostNames, "ZOOKEEPER", "Start");
  },

  startNameNode: function startNameNode() {
    var hostName = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').findProperty('isInstalled', true).hostName;
    this.updateComponent('NAMENODE', hostName, "HDFS", "Start");
  }
});

});

require.register("controllers/main/admin/highAvailability/nameNode/step8_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.HighAvailabilityWizardStep8Controller = Em.Controller.extend({

  name: "highAvailabilityWizardStep8Controller",

  done: function done() {
    App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
      App.router.send("next");
    });
  }

});

});

require.register("controllers/main/admin/highAvailability/nameNode/step9_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.HighAvailabilityWizardStep9Controller = App.HighAvailabilityProgressPageController.extend(App.WizardEnableDone, {

  name: "highAvailabilityWizardStep9Controller",

  commands: ['startSecondNameNode', 'installZKFC', 'startZKFC', 'installPXF', 'reconfigureRanger', 'reconfigureHBase', 'reconfigureAMS', 'reconfigureAccumulo', 'reconfigureHawq', 'deleteSNameNode', 'stopHDFS', 'startAllServices'],

  hbaseSiteTag: "",
  accumuloSiteTag: "",
  hawqSiteTag: "",
  secondNameNodeHost: "",

  initializeTasks: function initializeTasks() {
    this._super();
    var tasksToRemove = [];

    // find hostname where second namenode will be installed
    this.set('secondNameNodeHost', this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').findProperty('isInstalled', false).hostName);

    if (!App.Service.find().someProperty('serviceName', 'PXF') || this.isPxfComponentInstalled()) {
      tasksToRemove.push('installPXF');
    }
    if (!App.Service.find().someProperty('serviceName', 'RANGER')) {
      tasksToRemove.push('reconfigureRanger');
    }
    if (!App.Service.find().someProperty('serviceName', 'HBASE')) {
      tasksToRemove.push('reconfigureHBase');
    }
    if (!App.Service.find().someProperty('serviceName', 'AMBARI_METRICS')) {
      tasksToRemove.push('reconfigureAMS');
    }
    if (!App.Service.find().someProperty('serviceName', 'ACCUMULO')) {
      tasksToRemove.push('reconfigureAccumulo');
    }
    if (!App.Service.find().someProperty('serviceName', 'HAWQ')) {
      tasksToRemove.push('reconfigureHawq');
    }
    this.removeTasks(tasksToRemove);
  },

  startSecondNameNode: function startSecondNameNode() {
    var hostName = this.get('secondNameNodeHost');
    this.updateComponent('NAMENODE', hostName, "HDFS", "Start");
  },

  installZKFC: function installZKFC() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').mapProperty('hostName');
    this.createInstallComponentTask('ZKFC', hostNames, "HDFS");
  },

  startZKFC: function startZKFC() {
    var hostNames = this.get('content.masterComponentHosts').filterProperty('component', 'NAMENODE').mapProperty('hostName');
    this.updateComponent('ZKFC', hostNames, "HDFS", "Start");
  },

  isPxfComponentInstalled: function isPxfComponentInstalled() {
    var pxfComponent = this.getSlaveComponentHosts().findProperty('componentName', 'PXF');

    if (pxfComponent !== undefined) {
      var host;
      // check if PXF is already installed on the host assigned for additional NameNode
      for (var i = 0; i < pxfComponent.hosts.length; i++) {
        host = pxfComponent.hosts[i];
        if (host.hostName === this.get('secondNameNodeHost')) return true;
      }
    }

    return false;
  },

  installPXF: function installPXF() {
    this.createInstallComponentTask('PXF', this.get('secondNameNodeHost'), "PXF");
  },

  reconfigureRanger: function reconfigureRanger() {
    var data = this.get('content.serviceConfigProperties');
    var siteNames = ['ranger-env'];
    var configs = [];
    configs.push({
      Clusters: {
        desired_config: this.reconfigureSites(siteNames, data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false)))
      }
    });
    if (App.Service.find().someProperty('serviceName', 'YARN')) {
      siteNames = [];
      var yarnAuditConfig = data.items.findProperty('type', 'ranger-yarn-audit');
      if (yarnAuditConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in yarnAuditConfig.properties) {
          siteNames.push('ranger-yarn-audit');
          configs.push({
            Clusters: {
              desired_config: this.reconfigureSites(siteNames, data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false)))
            }
          });
        }
      }
    }
    if (App.Service.find().someProperty('serviceName', 'STORM')) {
      siteNames = [];
      var stormPluginConfig = data.items.findProperty('type', 'ranger-storm-plugin-properties');
      if (stormPluginConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in stormPluginConfig.properties) {
          siteNames.push('ranger-storm-plugin-properties');
        }
      }
      var stormAuditConfig = data.items.findProperty('type', 'ranger-storm-audit');
      if (stormAuditConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in stormAuditConfig.properties) {
          siteNames.push('ranger-storm-audit');
        }
      }
      if (siteNames.length) {
        configs.push({
          Clusters: {
            desired_config: this.reconfigureSites(siteNames, data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false)))
          }
        });
      }
    }
    if (App.Service.find().someProperty('serviceName', 'KAFKA')) {
      siteNames = [];
      var kafkaAuditConfig = data.items.findProperty('type', 'ranger-kafka-audit');
      if (kafkaAuditConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in kafkaAuditConfig.properties) {
          siteNames.push('ranger-kafka-audit');
          configs.push({
            Clusters: {
              desired_config: this.reconfigureSites(siteNames, data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false)))
            }
          });
        }
      }
    }
    if (App.Service.find().someProperty('serviceName', 'KNOX')) {
      siteNames = [];
      var knoxPluginConfig = data.items.findProperty('type', 'ranger-knox-plugin-properties');
      if (knoxPluginConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in knoxPluginConfig.properties) {
          siteNames.push('ranger-knox-plugin-properties');
        }
      }
      var knoxAuditConfig = data.items.findProperty('type', 'ranger-knox-audit');
      if (knoxAuditConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in knoxAuditConfig.properties) {
          siteNames.push('ranger-knox-audit');
        }
      }
      if (siteNames.length) {
        configs.push({
          Clusters: {
            desired_config: this.reconfigureSites(siteNames, data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false)))
          }
        });
      }
    }
    if (App.Service.find().someProperty('serviceName', 'ATLAS')) {
      siteNames = [];
      var atlasAuditConfig = data.items.findProperty('type', 'ranger-atlas-audit');
      if (atlasAuditConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in atlasAuditConfig.properties) {
          siteNames.push('ranger-atlas-audit');
          configs.push({
            Clusters: {
              desired_config: this.reconfigureSites(siteNames, data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false)))
            }
          });
        }
      }
    }
    if (App.Service.find().someProperty('serviceName', 'HIVE')) {
      siteNames = [];
      var hivePluginConfig = data.items.findProperty('type', 'ranger-hive-plugin-properties');
      if (hivePluginConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in hivePluginConfig.properties) {
          siteNames.push('ranger-hive-plugin-properties');
        }
      }
      var hiveAuditConfig = data.items.findProperty('type', 'ranger-hive-audit');
      if (hiveAuditConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in hiveAuditConfig.properties) {
          siteNames.push('ranger-hive-audit');
        }
      }
      if (siteNames.length) {
        configs.push({
          Clusters: {
            desired_config: this.reconfigureSites(siteNames, data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false)))
          }
        });
      }
    }
    if (App.Service.find().someProperty('serviceName', 'RANGER_KMS')) {
      siteNames = [];
      var rangerKMSConfig = data.items.findProperty('type', 'ranger-kms-audit');
      if (rangerKMSConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in rangerKMSConfig.properties) {
          siteNames.push('ranger-kms-audit');
          configs.push({
            Clusters: {
              desired_config: this.reconfigureSites(siteNames, data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false)))
            }
          });
        }
      }
    }

    App.ajax.send({
      name: 'common.service.multiConfigurations',
      sender: this,
      data: {
        configs: configs
      },
      success: 'saveConfigTag',
      error: 'onTaskError'
    });
  },

  /**
   *
   * @param {array} siteNames
   * @param {object} data
   */
  saveReconfiguredConfigs: function saveReconfiguredConfigs(siteNames, data) {
    var note = Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('NAMENODE', false));
    var configData = this.reconfigureSites(siteNames, data, note);
    return App.ajax.send({
      name: 'common.service.configurations',
      sender: this,
      data: {
        desired_config: configData
      },
      success: 'saveConfigTag',
      error: 'onTaskError'
    });
  },

  reconfigureHBase: function reconfigureHBase() {
    var data = this.get('content.serviceConfigProperties');
    var siteNames = ['hbase-site'];
    if (App.Service.find().someProperty('serviceName', 'RANGER')) {
      var hbasePluginConfig = data.items.findProperty('type', 'ranger-hbase-plugin-properties');
      if (hbasePluginConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in hbasePluginConfig.properties) {
          siteNames.push('ranger-hbase-plugin-properties');
        }
      }
      var hbaseAuditConfig = data.items.findProperty('type', 'ranger-hbase-audit');
      if (hbaseAuditConfig) {
        if ('xasecure.audit.destination.hdfs.dir' in hbaseAuditConfig.properties) {
          siteNames.push('ranger-hbase-audit');
        }
      }
    }
    this.saveReconfiguredConfigs(siteNames, data);
  },

  reconfigureAMS: function reconfigureAMS() {
    this.saveReconfiguredConfigs(['ams-hbase-site'], this.get('content.serviceConfigProperties'));
  },

  reconfigureAccumulo: function reconfigureAccumulo() {
    this.saveReconfiguredConfigs(['accumulo-site'], this.get('content.serviceConfigProperties'));
  },

  reconfigureHawq: function reconfigureHawq() {
    this.saveReconfiguredConfigs(['hawq-site', 'hdfs-client'], this.get('content.serviceConfigProperties'));
  },

  saveConfigTag: function saveConfigTag() {
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      clusterState: 'HIGH_AVAILABILITY_DEPLOY',
      wizardControllerName: this.get('content.controllerName'),
      localdb: App.db.data
    });
    this.onTaskCompleted();
  },

  startAllServices: function startAllServices() {
    this.startServices(false);
  },

  stopHDFS: function stopHDFS() {
    this.stopServices(["HDFS"], true);
  },

  deleteSNameNode: function deleteSNameNode() {
    var hostName = this.get('content.masterComponentHosts').findProperty('component', 'SECONDARY_NAMENODE').hostName;
    return App.ajax.send({
      name: 'common.delete.host_component',
      sender: this,
      data: {
        componentName: 'SECONDARY_NAMENODE',
        hostName: hostName
      },
      success: 'onTaskCompleted',
      error: 'onTaskError'
    });
  }

});

});

require.register("controllers/main/admin/highAvailability/nameNode/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.HighAvailabilityWizardController = App.WizardController.extend({

  name: 'highAvailabilityWizardController',

  totalSteps: 9,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('admin.highAvailability.wizard.header'),

  /**
   * Used for hiding back button in wizard
   */
  hideBackButton: true,

  content: Em.Object.create({
    controllerName: 'highAvailabilityWizardController',
    cluster: null,
    hosts: null,
    services: null,
    slaveComponentHosts: null,
    masterComponentHosts: null,
    serviceConfigProperties: [],
    serviceName: 'MISC',
    hdfsUser: "hdfs",
    nameServiceId: '',
    failedTask: null,
    requestIds: null
  }),

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      clusterState: 'HIGH_AVAILABILITY_DEPLOY',
      wizardControllerName: 'highAvailabilityWizardController',
      localdb: App.db.data
    });
  },

  /**
   * return new object extended from clusterStatusTemplate
   * @return Object
   */
  getCluster: function getCluster() {
    return jQuery.extend({}, this.get('clusterStatusTemplate'), { name: App.router.getClusterName() });
  },

  /**
   * save status of the cluster.
   * @param clusterStatus object with status,requestId fields.
   */
  saveClusterStatus: function saveClusterStatus(clusterStatus) {
    var oldStatus = this.toObject(this.get('content.cluster'));
    clusterStatus = jQuery.extend(oldStatus, clusterStatus);
    if (clusterStatus.requestId) {
      clusterStatus.requestId.forEach(function (requestId) {
        if (clusterStatus.oldRequestsId.indexOf(requestId) === -1) {
          clusterStatus.oldRequestsId.push(requestId);
        }
      }, this);
    }
    this.set('content.cluster', clusterStatus);
    this.save('cluster');
  },

  saveHdfsUser: function saveHdfsUser() {
    App.db.setHighAvailabilityWizardHdfsUser(this.get('content.hdfsUser'));
  },

  saveTasksStatuses: function saveTasksStatuses(statuses) {
    App.db.setHighAvailabilityWizardTasksStatuses(statuses);
    this.set('content.tasksStatuses', statuses);
  },

  saveConfigTag: function saveConfigTag(tag) {
    App.db.setHighAvailabilityWizardConfigTag(tag);
    this.set('content.' + [tag.name], tag.value);
  },

  saveHdfsClientHosts: function saveHdfsClientHosts(hostNames) {
    App.db.setHighAvailabilityWizardHdfsClientHosts(hostNames);
    this.set('content.hdfsClientHostNames', hostNames);
  },

  /**
   * Save config properties
   * @param stepController HighAvailabilityWizardStep3Controller
   */
  saveServiceConfigProperties: function saveServiceConfigProperties(stepController) {
    var serviceConfigProperties = [];
    var data = stepController.get('serverConfigData');

    var _content = stepController.get('stepConfigs')[0];
    _content.get('configs').forEach(function (_configProperties) {
      var siteObj = data.items.findProperty('type', _configProperties.get('filename'));
      if (siteObj) {
        if (_configProperties.get('name') == 'xasecure.audit.destination.hdfs.dir') {
          if ('xasecure.audit.destination.hdfs.dir' in siteObj.properties) {
            siteObj.properties[_configProperties.get('name')] = _configProperties.get('value');
          }
        } else {
          siteObj.properties[_configProperties.get('name')] = _configProperties.get('value');
        }
      }
    }, this);
    this.setDBProperty('serviceConfigProperties', data);
    this.set('content.serviceConfigProperties', data);
  },

  loadHdfsClientHosts: function loadHdfsClientHosts() {
    var hostNames = App.db.getHighAvailabilityWizardHdfsClientHosts();
    if (!(hostNames instanceof Array)) {
      hostNames = [hostNames];
    }
    this.set('content.hdfsClientHostNames', hostNames);
  },

  loadConfigTag: function loadConfigTag(tag) {
    var tagVal = App.db.getHighAvailabilityWizardConfigTag(tag);
    this.set('content.' + tag, tagVal);
  },

  loadHdfsUser: function loadHdfsUser() {
    var hdfsUser = App.db.getHighAvailabilityWizardHdfsUser();
    this.set('content.hdfsUser', hdfsUser);
  },

  loadTasksStatuses: function loadTasksStatuses() {
    var statuses = App.db.getHighAvailabilityWizardTasksStatuses();
    this.set('content.tasksStatuses', statuses);
  },

  /**
   * Load serviceConfigProperties to model
   */
  loadServiceConfigProperties: function loadServiceConfigProperties() {
    var serviceConfigProperties = this.getDBProperty('serviceConfigProperties');
    this.set('content.serviceConfigProperties', serviceConfigProperties);
  },

  saveRequestIds: function saveRequestIds(requestIds) {
    App.db.setHighAvailabilityWizardRequestIds(requestIds);
    this.set('content.requestIds', requestIds);
  },

  loadRequestIds: function loadRequestIds() {
    var requestIds = App.db.getHighAvailabilityWizardRequestIds();
    this.set('content.requestIds', requestIds);
  },

  saveNameServiceId: function saveNameServiceId(nameServiceId) {
    App.db.setHighAvailabilityWizardNameServiceId(nameServiceId);
    this.set('content.nameServiceId', nameServiceId);
  },

  loadNameServiceId: function loadNameServiceId() {
    var nameServiceId = App.db.getHighAvailabilityWizardNameServiceId();
    this.set('content.nameServiceId', nameServiceId);
  },

  saveTasksRequestIds: function saveTasksRequestIds(requestIds) {
    App.db.setHighAvailabilityWizardTasksRequestIds(requestIds);
    this.set('content.tasksRequestIds', requestIds);
  },

  loadTasksRequestIds: function loadTasksRequestIds() {
    var requestIds = App.db.getHighAvailabilityWizardTasksRequestIds();
    this.set('content.tasksRequestIds', requestIds);
  },

  loadMap: {
    '1': [{
      type: 'async',
      callback: function callback() {
        var dfd = $.Deferred(),
            self = this,
            usersLoadingCallback = function usersLoadingCallback() {
          self.saveHdfsUser();
          self.load('cluster');
          dfd.resolve();
        };
        if (App.db.getHighAvailabilityWizardHdfsUser()) {
          usersLoadingCallback();
        } else {
          this.loadHdfsUserFromServer().done(function (data) {
            self.set('content.hdfsUser', Em.get(data, '0.properties.hdfs_user'));
            usersLoadingCallback();
          });
        }
        return dfd.promise();
      }
    }],
    '2': [{
      type: 'async',
      callback: function callback() {
        var dfd = $.Deferred();
        var self = this;
        this.loadHosts().done(function () {
          self.loadServicesFromServer();
          self.loadMasterComponentHosts().done(function () {
            self.loadHdfsUser();
            dfd.resolve();
          });
        });
        return dfd.promise();
      }
    }],
    '3': [{
      type: 'sync',
      callback: function callback() {
        this.loadNameServiceId();
        this.loadServiceConfigProperties();
      }
    }],
    '5': [{
      type: 'sync',
      callback: function callback() {
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
      }
    }]
  },

  /**
   * Remove all loaded data.
   * Created as copy for App.router.clearAllSteps
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  clearTasksData: function clearTasksData() {
    this.saveTasksStatuses(undefined);
    this.saveRequestIds(undefined);
    this.saveTasksRequestIds(undefined);
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    App.db.data.Installer = {};
    this.resetDbNamespace();
    App.router.get('updateController').updateAll();
  }
});

});

require.register("controllers/main/admin/highAvailability/progress_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.HighAvailabilityProgressPageController = App.HighAvailabilityWizardController.extend(App.wizardProgressPageControllerMixin, {

  name: 'highAvailabilityProgressPageController',
  clusterDeployState: 'HIGH_AVAILABILITY_DEPLOY',
  tasksMessagesPrefix: 'admin.highAvailability.wizard.step',
  isRollback: false,

  manualRollback: function manualRollback() {
    App.ModalPopup.show({
      header: Em.I18n.t('admin.highAvailability.confirmRollbackHeader'),
      primary: Em.I18n.t('yes'),
      showCloseButton: false,
      onPrimary: function onPrimary() {
        var self = this;
        var controller = App.router.get('highAvailabilityWizardController');
        controller.clearTasksData();
        controller.clearStorageData();
        controller.resetOnClose(controller, 'main.services.index');
        this.hide();
      },
      secondary: Em.I18n.t('no'),
      onSecondary: function onSecondary() {
        this.hide();
      },
      bodyClass: Ember.View.extend({
        template: Ember.Handlebars.compile(Em.I18n.t('admin.highAvailability.confirmManualRollbackBody'))
      })
    });
  },

  rollback: function rollback() {
    var task = this.get('tasks').findProperty('status', 'FAILED');
    App.router.get(this.get('content.controllerName')).saveFailedTask(task);
    App.ModalPopup.show({
      header: Em.I18n.t('admin.highAvailability.confirmRollbackHeader'),
      primary: Em.I18n.t('common.confirm'),
      showCloseButton: false,
      onPrimary: function onPrimary() {
        App.router.get('highAvailabilityWizardController').clearTasksData();
        App.router.transitionTo('main.admin.highAvailabilityRollback');
        this.hide();
      },
      secondary: Em.I18n.t('common.cancel'),
      body: Em.I18n.t('admin.highAvailability.confirmRollbackBody')
    });
  },

  /**
   * Prepare object to send to the server to save configs
   * Split all configs by site names and note
   * @param siteNames Array
   * @param data Object
   * @param note String
   */
  reconfigureSites: function reconfigureSites(siteNames, data, note) {
    return siteNames.map(function (_siteName) {
      var config = data.items.findProperty('type', _siteName);
      var configToSave = {
        type: _siteName,
        properties: config && config.properties,
        service_config_version_note: note || ''
      };
      if (config && config.properties_attributes) {
        configToSave.properties_attributes = config.properties_attributes;
      }
      return configToSave;
    });
  }
});

});

require.register("controllers/main/admin/highAvailability/progress_popup_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

App.HighAvailabilityProgressPopupController = Ember.Controller.extend({

  name: 'highAvailabilityProgressPopupController',

  /**
   * Id of current request
   * @type {Array}
   */
  requestIds: [],

  /**
   * Title for popup header
   * @type {String}
   */
  popupTitle: '',

  /**
   * Array with Hosts tasks data used in <code>App.HostPopup</code>
   * @type {Array}
   */
  services: [],

  /**
   * Timestamp used in <code>App.HostPopup</code>
   * @type {Number}
   */
  serviceTimestamp: null,

  /**
   * Progress controller. Used to get tasks data.
   * @type {App.HighAvailabilityProgressPageController}
   */
  progressController: null,

  /**
   * Requests data with tasks
   * @type {Array}
   */
  hostsData: [],

  /**
   *  StageId for the command.
   *  @type {Number}
   */
  stageId: null,

  /**
   * During loading and calculations show popup with spinner
   * @type {Object}
   */
  spinnerPopup: null,

  isTaskPolling: false,

  taskInfo: null,

  /**
   * Get info for <code>requestIds</code> and initialize <code>App.HostPopup</code>
   * @param popupTitle {String}
   * @param requestIds {Array}
   * @param progressController {App.HighAvailabilityProgressPageController}
   * @param showSpinner {Boolean}
   * @param stageId {Number}
   */
  initPopup: function initPopup(popupTitle, requestIds, progressController, showSpinner, stageId) {
    if (showSpinner) {
      var loadingPopup = App.ModalPopup.show({
        header: Em.I18n.t('common.loading.eclipses'),
        primary: false,
        secondary: false,
        bodyClass: Ember.View.extend({
          template: Ember.Handlebars.compile('{{view App.SpinnerView}}')
        })
      });
      this.set('spinnerPopup', loadingPopup);
    }
    this.setProperties({
      progressController: progressController,
      popupTitle: popupTitle,
      requestIds: requestIds,
      hostsData: [],
      stageId: stageId
    });
    this.getHosts();
  },

  /**
   * Send AJAX request to get hosts tasks data
   */
  getHosts: function getHosts(successCallback) {
    var requestIds = this.get('requestIds');
    var stageId = this.get('stageId');
    var name = 'background_operations.get_by_request';
    if (!Em.isNone(stageId)) {
      name = 'common.request.polling';
      if (stageId === 0) {
        stageId = '0';
      }
    }
    if (Em.isNone(successCallback)) {
      successCallback = 'onGetHostsSuccess';
    }
    requestIds.forEach(function (requestId) {
      App.ajax.send({
        name: name,
        sender: this,
        data: {
          requestId: requestId,
          stageId: stageId
        },
        success: successCallback
      });
    }, this);
  },

  doPolling: function doPolling() {
    var self = this;
    this.set('progressController.logs', []);
    setTimeout(function () {
      self.getHosts('doPollingSuccessCallback');
    }, App.bgOperationsUpdateInterval);
  },

  doPollingSuccessCallback: function doPollingSuccessCallback(data) {
    this.set('hostsData', [data]);
    var hostsData = this.get('hostsData');
    this.set('progressController.logs', data.tasks);
    if (this.isRequestRunning(hostsData)) {
      this.doPolling();
    }
  },

  /**
   * Callback for <code>getHosts</code> request
   * @param data
   */
  onGetHostsSuccess: function onGetHostsSuccess(data) {
    var hostsData = this.get('hostsData');
    hostsData.push(data);
    if (this.get('requestIds.length') === this.get('hostsData.length')) {
      var popupTitle = this.get('popupTitle');
      this.calculateHostsData(hostsData);
      App.HostPopup.initPopup(popupTitle, this, false, this.get("requestIds")[0]);
      if (this.isRequestRunning(hostsData)) {
        if (this.get('progressController.name') === 'mainAdminStackAndUpgradeController') {
          this.doPolling();
        }
        this.addObserver('progressController.logs.length', this, 'getDataFromProgressController');
      }
    }
    if (this.get('spinnerPopup')) {
      this.get('spinnerPopup').hide();
      this.set('spinnerPopup', null);
    }
  },

  /**
   * Convert data to format used in <code>App.HostPopup</code>
   * @param data {Array}
   */
  calculateHostsData: function calculateHostsData(data) {
    var hosts = [];
    var hostsMap = {};
    var popupTitle = this.get('popupTitle');
    var id = this.get('requestIds')[0];

    data.forEach(function (request) {
      request.tasks.forEach(function (task) {
        var host = task.Tasks.host_name;
        if (hostsMap[host]) {
          hostsMap[host].logTasks.push(task);
        } else {
          hostsMap[host] = {
            name: task.Tasks.host_name,
            publicName: task.Tasks.host_name,
            logTasks: [task]
          };
        }
      });
    });
    for (var host in hostsMap) {
      hosts.push(hostsMap[host]);
    }
    this.set('services', [{ id: id, name: popupTitle, hosts: hosts }]);
    this.set('serviceTimestamp', App.dateTime());
    if (!this.isRequestRunning(data)) {
      this.removeObserver('progressController.logs.length', this, 'getDataFromProgressController');
    }
  },

  /**
   * Get hosts tasks data from <code>progressController</code>
   */
  getDataFromProgressController: function getDataFromProgressController() {
    var data = this.get('hostsData');
    var tasksData = [];
    var stageId = this.get('stageId');
    // If the progress page is broken into stages then update host with only stage's tasks
    if (!!stageId) {
      tasksData = this.get('progressController.logs').filterProperty('Tasks.stage_id', stageId);
    } else {
      tasksData = this.get('progressController.logs');
    }
    if (tasksData.length) {
      var tasks = [];
      tasksData.forEach(function (logs) {
        tasks.pushObjects(logs);
      }, this);
      data.forEach(function (request) {
        tasks = tasks.filterProperty('Tasks.request_id', request.Requests.id);
        request.tasks = tasks;
      });
      this.calculateHostsData(data);
    }
  },

  /**
   * Identify whether request is running by task counters
   * @param requests {Array}
   * @return {Boolean}
   */
  isRequestRunning: function isRequestRunning(requests) {
    var result = false;
    requests.forEach(function (request) {
      if (request.Requests.task_count - (request.Requests.aborted_task_count + request.Requests.completed_task_count + request.Requests.failed_task_count + request.Requests.timed_out_task_count - request.Requests.queued_task_count) > 0) {
        result = true;
      }
    });
    return result;
  },

  startTaskPolling: function startTaskPolling(requestId, taskId) {
    this.setProperties({
      isTaskPolling: true,
      taskInfo: {
        id: taskId,
        requestId: requestId
      }
    });
    App.updater.run(this, 'updateTask', 'isTaskPolling', App.bgOperationsUpdateInterval);
    App.updater.immediateRun('updateTask');
  },

  stopTaskPolling: function stopTaskPolling() {
    this.set('isTaskPolling', false);
  },

  updateTask: function updateTask(callback) {
    App.ajax.send({
      name: 'background_operations.get_by_task',
      sender: this,
      data: {
        requestId: this.get('taskInfo.requestId'),
        taskId: this.get('taskInfo.id')
      },
      success: 'updateTaskSuccessCallback',
      callback: callback
    });
  },

  updateTaskSuccessCallback: function updateTaskSuccessCallback(data) {
    this.setProperties({
      'taskInfo.stderr': data.Tasks.stderr,
      'taskInfo.stdout': data.Tasks.stdout,
      'taskInfo.outputLog': data.Tasks.output_log,
      'taskInfo.errorLog': data.Tasks.error_log,
      'isTaskPolling': !['FAILED', 'COMPLETED', 'TIMEDOUT', 'ABORTED'].contains(data.Tasks.status)
    });
  }

});

});

require.register("controllers/main/admin/highAvailability/rangerAdmin/step1_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var validator = require('utils/validator');

App.RAHighAvailabilityWizardStep1Controller = Em.Controller.extend({
  name: "rAHighAvailabilityWizardStep1Controller",

  /**
   * Define if typed load balancer URL is valid URL
   * @type {Boolean}
   */
  isloadBalancerURLValid: function () {
    return validator.isValidURL(this.get('content.loadBalancerURL'));
  }.property('content.loadBalancerURL'),

  /**
   * Define if show load balancer URL error
   * do not show is input-field is empty
   * @type {Boolean}
   */
  showloadBalancerURLError: Em.computed.and('content.loadBalancerURL', '!isloadBalancerURLValid'),

  /**
   * Define either Submit is disabled or enabled
   * @type {Boolean}
   */
  isSubmitDisabled: Em.computed.not('isloadBalancerURLValid')
});

});

require.register("controllers/main/admin/highAvailability/rangerAdmin/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RAHighAvailabilityWizardStep2Controller = Em.Controller.extend(App.BlueprintMixin, App.AssignMasterComponents, {
  name: "rAHighAvailabilityWizardStep2Controller",

  useServerValidation: false,

  mastersToShow: ['RANGER_ADMIN'],

  mastersToAdd: ['RANGER_ADMIN'],

  showCurrentPrefix: ['RANGER_ADMIN'],

  showAdditionalPrefix: ['RANGER_ADMIN'],

  mastersAddableInHA: ['RANGER_ADMIN'],

  showInstalledMastersFirst: true,

  generalWarningMessages: [Em.I18n.t('admin.ra_highAvailability.wizard.step2.warning')]

});

});

require.register("controllers/main/admin/highAvailability/rangerAdmin/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RAHighAvailabilityWizardStep3Controller = Em.Controller.extend({
  name: 'rAHighAvailabilityWizardStep3Controller',

  isLoaded: false,

  versionLoaded: true,

  hideDependenciesInfoBar: true,

  stepConfigs: [App.ServiceConfig.create({
    serviceName: 'MISC',
    showConfig: true
  })],

  loadStep: function loadStep() {
    var self = this;
    App.get('router.mainController.isLoading').call(App.get('router.clusterController'), 'isConfigsPropertiesLoaded').done(function () {
      var stepConfig = self.get('stepConfigs.firstObject'),
          configs = [],
          configCategories = [],
          installedServices = App.Service.find().mapProperty('serviceName');
      self.get('wizardController.configs').forEach(function (config) {
        var service = App.config.get('serviceByConfigTypeMap')[config.siteName];
        if (service) {
          var serviceName = service.get('serviceName'),
              serviceDisplayName = service.get('displayName');
          if (installedServices.contains(serviceName)) {
            var property = App.configsCollection.getConfigByName(config.propertyName, config.siteName) || {};
            if (!configCategories.someProperty('name'), serviceName) {
              configCategories.push(App.ServiceConfigCategory.create({
                name: serviceName,
                displayName: serviceDisplayName
              }));
            }
            configs.push(App.ServiceConfigProperty.create(property, {
              category: serviceName,
              value: self.get('content.loadBalancerURL'),
              isEditable: false
            }));
          }
        }
      });
      stepConfig.setProperties({
        configs: configs,
        configCategories: configCategories
      });
      self.setProperties({
        isLoaded: true,
        selectedService: stepConfig
      });
    });
  }
});

});

require.register("controllers/main/admin/highAvailability/rangerAdmin/step4_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RAHighAvailabilityWizardStep4Controller = App.HighAvailabilityProgressPageController.extend({

  name: "rAHighAvailabilityWizardStep4Controller",

  clusterDeployState: 'RA_HIGH_AVAILABILITY_DEPLOY',

  commands: ['stopAllServices', 'installRangerAdmin', 'reconfigureRanger', 'startAllServices'],

  tasksMessagesPrefix: 'admin.ra_highAvailability.wizard.step',

  stopAllServices: function stopAllServices() {
    this.stopServices([], true, true);
  },

  installRangerAdmin: function installRangerAdmin() {
    var hostNames = this.get('content.raHosts.additionalRA');
    this.createInstallComponentTask('RANGER_ADMIN', hostNames, "RANGER");
  },

  reconfigureRanger: function reconfigureRanger() {
    this.loadConfigsTags();
  },

  loadConfigsTags: function loadConfigsTags() {
    App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'onLoadConfigsTags',
      error: 'onTaskError'
    });
  },

  onLoadConfigsTags: function onLoadConfigsTags(data) {
    var urlParams = [];
    var siteNamesToFetch = this.get('wizardController.configs').mapProperty('siteName');
    siteNamesToFetch.map(function (siteName) {
      if (siteName in data.Clusters.desired_configs) {
        urlParams.push('(type=' + siteName + '&tag=' + data.Clusters.desired_configs[siteName].tag + ')');
      }
    });
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: urlParams.join('|')
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadConfigs: function onLoadConfigs(data) {
    var configs = [];

    this.get('wizardController.configs').map(function (item) {
      var config = data.items.findProperty('type', item.siteName);
      if (config) {
        config.properties[item.propertyName] = this.get('content.loadBalancerURL');
        configs.push({
          Clusters: {
            desired_config: this.reconfigureSites([item.siteName], data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('RANGER_ADMIN', false)))
          }
        });
      }
    }, this);

    App.ajax.send({
      name: 'common.service.multiConfigurations',
      sender: this,
      data: {
        configs: configs
      },
      success: 'onSaveConfigs',
      error: 'onTaskError'
    });
  },

  onSaveConfigs: function onSaveConfigs() {
    this.onTaskCompleted();
  },

  startAllServices: function startAllServices() {
    this.startServices(true);
  }
});

});

require.register("controllers/main/admin/highAvailability/rangerAdmin/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RAHighAvailabilityWizardController = App.WizardController.extend({

  name: 'rAHighAvailabilityWizardController',

  totalSteps: 4,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('admin.ra_highAvailability.wizard.header'),

  isFinished: false,

  content: Em.Object.create({
    controllerName: 'rAHighAvailabilityWizardController',
    cluster: null,
    loadBalancerURL: null,
    hosts: null,
    services: null,
    masterComponentHosts: null
  }),

  configs: [{
    siteName: 'admin-properties',
    propertyName: 'policymgr_external_url'
  }, {
    siteName: 'ranger-hdfs-security',
    propertyName: 'ranger.plugin.hdfs.policy.rest.url'
  }, {
    siteName: 'ranger-yarn-security',
    propertyName: 'ranger.plugin.yarn.policy.rest.url'
  }, {
    siteName: 'ranger-hbase-security',
    propertyName: 'ranger.plugin.hbase.policy.rest.url'
  }, {
    siteName: 'ranger-hive-security',
    propertyName: 'ranger.plugin.hive.policy.rest.url'
  }, {
    siteName: 'ranger-knox-security',
    propertyName: 'ranger.plugin.knox.policy.rest.url'
  }, {
    siteName: 'ranger-kafka-security',
    propertyName: 'ranger.plugin.kafka.policy.rest.url'
  }, {
    siteName: 'ranger-kms-security',
    propertyName: 'ranger.plugin.kms.policy.rest.url'
  }, {
    siteName: 'ranger-storm-security',
    propertyName: 'ranger.plugin.storm.policy.rest.url'
  }, {
    siteName: 'ranger-atlas-security',
    propertyName: 'ranger.plugin.atlas.policy.rest.url'
  }],

  init: function init() {
    this._super();
    this.clearStep();
  },

  clearStep: function clearStep() {
    this.set('isFinished', false);
  },

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      wizardControllerName: 'rAHighAvailabilityWizardController',
      localdb: App.db.data
    });
  },

  loadMap: {
    '1': [{
      type: 'sync',
      callback: function callback() {
        this.load('cluster');
        this.load('loadBalancerURL');
      }
    }],
    '2': [{
      type: 'async',
      callback: function callback() {
        var dfd = $.Deferred();
        var self = this;
        this.loadHosts().done(function () {
          self.loadServicesFromServer();
          self.loadMasterComponentHosts().done(function () {
            dfd.resolve();
          });
        });
        return dfd.promise();
      }
    }],
    '3': [{
      type: 'sync',
      callback: function callback() {
        this.load('raHosts');
      }
    }],
    '4': [{
      type: 'sync',
      callback: function callback() {
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
      }
    }]
  },

  /**
   * Remove all loaded data.
   * Created as copy for App.router.clearAllSteps
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.resetDbNamespace();
    App.router.get('updateController').updateAll();
    this.set('isFinished', true);
  }
});

});

require.register("controllers/main/admin/highAvailability/resourceManager/step1_controller", function(exports, require, module) {
"use strict";

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RMHighAvailabilityWizardStep1Controller = Em.Controller.extend({
  name: "rMHighAvailabilityWizardStep1Controller"
});

});

require.register("controllers/main/admin/highAvailability/resourceManager/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RMHighAvailabilityWizardStep2Controller = Em.Controller.extend(App.AssignMasterComponents, {

  name: "rMHighAvailabilityWizardStep2Controller",

  useServerValidation: false,

  mastersToShow: ['RESOURCEMANAGER'],

  mastersToAdd: ['RESOURCEMANAGER'],

  showCurrentPrefix: ['RESOURCEMANAGER'],

  showAdditionalPrefix: ['RESOURCEMANAGER'],

  showInstalledMastersFirst: true
});

});

require.register("controllers/main/admin/highAvailability/resourceManager/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @typedef {object} rmHaConfigDependencies
 * @property {string|number} webAddressPort
 * @property {string|number} httpsWebAddressPort
 * @property {string|number} zkClientPort
 */

var App = require('app');
var blueprintUtils = require('utils/blueprint');
require('utils/configs/rm_ha_config_initializer');

App.RMHighAvailabilityWizardStep3Controller = Em.Controller.extend(App.BlueprintMixin, {
  name: "rMHighAvailabilityWizardStep3Controller",

  selectedService: null,

  versionLoaded: true,

  hideDependenciesInfoBar: true,

  isLoaded: false,

  isSubmitDisabled: Em.computed.not('isLoaded'),

  loadStep: function loadStep() {
    this.renderConfigs();
  },

  /**
   * Render configs to show them in <code>App.ServiceConfigView</code>
   */
  renderConfigs: function renderConfigs() {

    var configs = $.extend(true, {}, require('data/configs/wizards/rm_ha_properties').haConfig);

    var serviceConfig = App.ServiceConfig.create({
      serviceName: configs.serviceName,
      displayName: configs.displayName,
      configCategories: [],
      showConfig: true,
      configs: []
    });

    configs.configCategories.forEach(function (configCategory) {
      if (App.Service.find().someProperty('serviceName', configCategory.name)) {
        serviceConfig.configCategories.pushObject(configCategory);
      }
    }, this);

    this.renderConfigProperties(configs, serviceConfig);
    App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: 'loadConfigTagsSuccessCallback',
      error: 'loadConfigsErrorCallback',
      data: {
        serviceConfig: serviceConfig
      }
    });
  },

  loadConfigTagsSuccessCallback: function loadConfigTagsSuccessCallback(data, opt, params) {
    var urlParams = '(type=zoo.cfg&tag=' + data.Clusters.desired_configs['zoo.cfg'].tag + ')|' + '(type=yarn-site&tag=' + data.Clusters.desired_configs['yarn-site'].tag + ')|' + '(type=yarn-env&tag=' + data.Clusters.desired_configs['yarn-env'].tag + ')';
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: urlParams,
        serviceConfig: params.serviceConfig
      },
      success: 'loadConfigsSuccessCallback',
      error: 'loadConfigsSuccessCallback'
    });
  },

  loadConfigsSuccessCallback: function loadConfigsSuccessCallback(data, opt, params) {
    var self = this;
    var blueprintConfigurations = Em.getWithDefault(data || {}, 'items', []).reduce(function (prev, cur) {
      prev[cur.type] = { properties: cur.properties };
      return prev;
    }, {});
    params = params.serviceConfig ? params.serviceConfig : arguments[4].serviceConfig;
    this.setDynamicConfigValues(params, data);
    this.loadRecommendations(blueprintConfigurations).always(function (recommendations) {
      self.applyRecommendedConfigurations(recommendations, data, params);
      self.setProperties({
        selectedService: params,
        isLoaded: true
      });
    });
  },

  /**
   * Get dependencies for new configs
   *
   * @param {{items: object[]}} data
   * @returns {rmHaConfigDependencies}
   * @private
   * @method _prepareDependencies
   */
  _prepareDependencies: function _prepareDependencies(data) {
    var ret = {};
    var zooCfg = data && data.items ? data.items.findProperty('type', 'zoo.cfg') : null;
    var yarnSite = data && data.items ? data.items.findProperty('type', 'yarn-site') : null;
    var portValue = zooCfg && Em.get(zooCfg, 'properties.clientPort');
    var webAddressPort = yarnSite && yarnSite.properties ? yarnSite.properties['yarn.resourcemanager.webapp.address'] : '';
    var httpsWebAddressPort = yarnSite && yarnSite.properties ? yarnSite.properties['yarn.resourcemanager.webapp.https.address'] : '';
    var trackerAddressPort = yarnSite && yarnSite.properties ? yarnSite.properties['yarn.resourcemanager.resource-tracker.address'] : '';

    ret.webAddressPort = webAddressPort && webAddressPort.contains(':') ? webAddressPort.split(':')[1] : '8088';
    ret.httpsWebAddressPort = httpsWebAddressPort && httpsWebAddressPort.contains(':') ? httpsWebAddressPort.split(':')[1] : '8090';
    ret.trackerAddressPort = trackerAddressPort && trackerAddressPort.contains(':') ? trackerAddressPort.split(':')[1] : '8025';
    ret.zkClientPort = portValue ? portValue : '2181';
    return ret;
  },

  /**
   * Set values to the new configs
   *
   * @param {object} configs
   * @param {object} data
   * @returns {object}
   * @method setDynamicConfigValues
   */
  setDynamicConfigValues: function setDynamicConfigValues(configs, data) {
    var topologyLocalDB = this.get('content').getProperties(['masterComponentHosts', 'slaveComponentHosts', 'hosts']);
    var yarnUser = data.items.findProperty('type', 'yarn-env').properties.yarn_user;
    App.RmHaConfigInitializer.setup({
      yarnUser: yarnUser
    });
    var dependencies = this._prepareDependencies(data);
    // /** add dynamic property 'hadoop.proxyuser.' + yarnUser + '.hosts' **/
    // var proxyUserConfig = App.ServiceConfigProperty.create(App.config.createDefaultConfig('hadoop.proxyuser.' + yarnUser + '.hosts',
    //   'core-site', false,  {category : "HDFS", isUserProperty: false, isEditable: false, isOverridable: false, serviceName: 'MISC'}));
    // configs.configs.pushObject(proxyUserConfig);

    configs.configs.forEach(function (config) {
      App.RmHaConfigInitializer.initialValue(config, topologyLocalDB, dependencies);
    });
    App.RmHaConfigInitializer.cleanup();
    return configs;
  },

  /**
   * Load child components to service config object
   * @param _componentConfig
   * @param componentConfig
   */
  renderConfigProperties: function renderConfigProperties(_componentConfig, componentConfig) {
    _componentConfig.configs.forEach(function (_serviceConfigProperty) {
      var serviceConfigProperty = App.ServiceConfigProperty.create(_serviceConfigProperty);
      componentConfig.configs.pushObject(serviceConfigProperty);
      serviceConfigProperty.set('isEditable', serviceConfigProperty.get('isReconfigurable'));
    }, this);
  },

  submit: function submit() {
    if (!this.get('isSubmitDisabled')) {
      App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
        App.router.send("next");
      });
    }
  },

  loadRecommendations: function loadRecommendations(blueprintConfigurations) {
    var blueprint = this.getCurrentMasterSlaveBlueprint();
    // host group where new ResourceManager will be added
    var hostGroupName = blueprintUtils.getHostGroupByFqdn(blueprint, this.get('content.rmHosts.additionalRM'));
    var dataToSend = {
      recommend: 'configurations',
      hosts: App.get('allHostNames'),
      services: App.Service.find().mapProperty('serviceName').uniq(),
      recommendations: {}
    };
    if (!!hostGroupName) {
      blueprintUtils.addComponentToHostGroup(blueprint, 'RESOURCEMANAGER', hostGroupName);
    }
    blueprint.blueprint.configurations = blueprintConfigurations;
    dataToSend.recommendations = blueprint;
    return App.ajax.send({
      name: 'config.recommendations',
      sender: this,
      data: {
        stackVersionUrl: App.get('stackVersionURL'),
        dataToSend: dataToSend
      }
    });
  },

  applyRecommendedConfigurations: function applyRecommendedConfigurations(recommendations, configurations, stepConfigs) {
    var yarnEnv = Em.getWithDefault(configurations || {}, 'items', []).findProperty('type', 'yarn-env') || {},
        yarnUser = Em.getWithDefault(yarnEnv, 'properties.yarn_user', false),
        coreSite = Em.getWithDefault(recommendations, 'resources.0.recommendations.blueprint.configurations.core-site.properties', {}),
        proxyHostName = 'hadoop.proxyuser.' + yarnUser + '.hosts',
        recommendedHosts = coreSite[proxyHostName] || false,
        newProp;

    if (yarnUser && recommendedHosts) {
      if (stepConfigs.get('configs').someProperty('name', proxyHostName)) {
        stepConfigs.get('configs').findProperty('name', proxyHostName).setProperties({
          recommendedValue: recommendedHosts,
          value: recommendedHosts
        });
      } else {
        newProp = App.config.createDefaultConfig(proxyHostName, 'core-site', false, {
          category: "HDFS",
          isUserProperty: false,
          isEditable: false,
          isOverridable: false,
          serviceName: 'MISC',
          value: recommendedHosts,
          recommendedValue: recommendedHosts
        });
        newProp.filename = App.config.getConfigTagFromFileName(newProp.filename);
        stepConfigs.get('configs').pushObject(App.ServiceConfigProperty.create(newProp));
      }
    }
  }
});

});

require.register("controllers/main/admin/highAvailability/resourceManager/step4_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

require('controllers/main/admin/serviceAccounts_controller');

App.RMHighAvailabilityWizardStep4Controller = App.HighAvailabilityProgressPageController.extend(App.WizardEnableDone, {

  name: "rMHighAvailabilityWizardStep4Controller",

  clusterDeployState: 'RM_HIGH_AVAILABILITY_DEPLOY',

  commands: ['stopRequiredServices', 'installResourceManager', 'reconfigureYARN', 'reconfigureHAWQ', 'reconfigureHDFS', 'startAllServices'],

  tasksMessagesPrefix: 'admin.rm_highAvailability.wizard.step',

  initializeTasks: function initializeTasks() {
    this._super();
    var tasksToRemove = [];
    if (!App.Service.find().someProperty('serviceName', 'HAWQ')) {
      tasksToRemove.push('reconfigureHAWQ');
    }
    this.removeTasks(tasksToRemove);
  },

  stopRequiredServices: function stopRequiredServices() {
    this.stopServices(['HDFS']);
  },

  installResourceManager: function installResourceManager() {
    var hostName = this.get('content.rmHosts.additionalRM');
    this.createInstallComponentTask('RESOURCEMANAGER', hostName, "YARN");
  },

  reconfigureYARN: function reconfigureYARN() {
    this.loadConfigsTags("Yarn");
  },

  reconfigureHAWQ: function reconfigureHAWQ() {
    this.loadConfigsTags("Hawq");
  },

  reconfigureHDFS: function reconfigureHDFS() {
    this.loadConfigsTags("Hdfs");
  },

  loadConfigsTags: function loadConfigsTags(service) {
    var onLoadServiceConfigsTags = 'onLoad' + service + "ConfigsTags";
    App.ajax.send({
      name: 'config.tags',
      sender: this,
      success: onLoadServiceConfigsTags,
      error: 'onTaskError'
    });
  },

  onLoadYarnConfigsTags: function onLoadYarnConfigsTags(data) {
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: '(type=yarn-site&tag=' + data.Clusters.desired_configs['yarn-site'].tag + ')',
        type: 'yarn-site'
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadHawqConfigsTags: function onLoadHawqConfigsTags(data) {
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: '(type=yarn-client&tag=' + data.Clusters.desired_configs['yarn-client'].tag + ')',
        type: 'yarn-client'
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadHdfsConfigsTags: function onLoadHdfsConfigsTags(data) {
    App.ajax.send({
      name: 'reassign.load_configs',
      sender: this,
      data: {
        urlParams: '(type=core-site&tag=' + data.Clusters.desired_configs['core-site'].tag + ')',
        type: 'core-site'
      },
      success: 'onLoadConfigs',
      error: 'onTaskError'
    });
  },

  onLoadConfigs: function onLoadConfigs(data, opt, params) {
    var propertiesToAdd = this.get('content.configs').filterProperty('filename', params.type);
    propertiesToAdd.forEach(function (property) {
      data.items[0].properties[property.name] = property.value;
    });

    var configData = this.reconfigureSites([params.type], data, Em.I18n.t('admin.highAvailability.step4.save.configuration.note').format(App.format.role('RESOURCEMANAGER', false)));

    App.ajax.send({
      name: 'common.service.configurations',
      sender: this,
      data: {
        desired_config: configData
      },
      success: 'onSaveConfigs',
      error: 'onTaskError'
    });
  },
  onSaveConfigs: function onSaveConfigs() {
    this.onTaskCompleted();
  },

  startAllServices: function startAllServices() {
    this.startServices(true);
  }
});

});

require.register("controllers/main/admin/highAvailability/resourceManager/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.RMHighAvailabilityWizardController = App.WizardController.extend({

  name: 'rMHighAvailabilityWizardController',

  totalSteps: 4,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('admin.rm_highAvailability.wizard.header'),

  isFinished: false,

  content: Em.Object.create({
    controllerName: 'rMHighAvailabilityWizardController'
  }),

  /**
   * Load data for all steps until <code>current step</code>
   */
  loadMap: {
    '1': [{
      type: 'sync',
      callback: function callback() {
        this.load('cluster');
      }
    }],
    '2': [{
      type: 'async',
      callback: function callback() {
        var self = this,
            dfd = $.Deferred();
        this.loadRmHosts();
        this.loadServicesFromServer();
        this.loadMasterComponentHosts().done(function () {
          self.loadConfirmedHosts();
          dfd.resolve();
        });
        return dfd.promise();
      }
    }],
    '4': [{
      type: 'sync',
      callback: function callback() {
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
        this.loadConfigs();
      }
    }]
  },

  init: function init() {
    this._super();
    this.clearStep();
  },

  clearStep: function clearStep() {
    this.set('isFinished', false);
  },

  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      wizardControllerName: 'rMHighAvailabilityWizardController',
      localdb: App.db.data
    });
  },

  /**
   * Save hosts for additional and current ResourceManagers to local db and <code>controller.content</code>
   * @param rmHosts
   */
  saveRmHosts: function saveRmHosts(rmHosts) {
    this.set('content.rmHosts', rmHosts);
    this.setDBProperty('rmHosts', rmHosts);
  },

  /**
   * Load hosts for additional and current ResourceManagers from local db to <code>controller.content</code>
   */
  loadRmHosts: function loadRmHosts() {
    var rmHosts = this.getDBProperty('rmHosts');
    this.set('content.rmHosts', rmHosts);
  },

  /**
   * Save configs to load and apply them on Configure Components step
   * @param configs
   */
  saveConfigs: function saveConfigs(configs) {
    this.set('content.configs', configs);
    this.setDBProperty('configs', configs);
  },

  /**
   * Load configs to apply them on Configure Components step
   */
  loadConfigs: function loadConfigs() {
    var configs = this.getDBProperty('configs');
    this.set('content.configs', configs);
  },

  /**
   * Remove all loaded data.
   * Created as copy for App.router.clearAllSteps
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.resetDbNamespace();
    App.router.get('updateController').updateAll();
    this.set('isFinished', true);
  }
});

});

require.register("controllers/main/admin/highAvailability_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainAdminHighAvailabilityController = App.WizardController.extend({
  name: 'mainAdminHighAvailabilityController',

  tag: null,

  dataIsLoaded: false,

  /**
   * enable High Availability
   * @return {Boolean}
   */
  enableHighAvailability: function enableHighAvailability() {
    var message = [];
    var hostComponents = App.HostComponent.find();

    if (hostComponents.findProperty('componentName', 'NAMENODE').get('workStatus') !== 'STARTED') {
      message.push(Em.I18n.t('admin.highAvailability.error.namenodeStarted'));
    }
    if (hostComponents.filterProperty('componentName', 'ZOOKEEPER_SERVER').length < 3) {
      message.push(Em.I18n.t('admin.highAvailability.error.zooKeeperNum'));
    }
    if (hostComponents.filterProperty('isMaster', true).someProperty('passiveState', "ON") || hostComponents.filterProperty('isMaster', true).someProperty('isImpliedState', true)) {
      message.push(Em.I18n.t('admin.highAvailability.error.maintenanceMode'));
    }

    if (App.get('allHostNames.length') < 3) {
      message.push(Em.I18n.t('admin.highAvailability.error.hostsNum'));
    }
    if (message.length > 0) {
      this.showErrorPopup(message);
      return false;
    }
    App.router.transitionTo('main.services.enableHighAvailability');
    return true;
  },

  disableHighAvailability: function disableHighAvailability() {
    App.router.transitionTo('main.admin.rollbackHighAvailability');
  },

  /**
   * enable ResourceManager High Availability
   * @return {Boolean}
   */
  enableRMHighAvailability: function enableRMHighAvailability() {
    //Prerequisite Checks
    var message = [];

    if (App.HostComponent.find().findProperty('componentName', 'RESOURCEMANAGER').get('workStatus') !== 'STARTED') {
      message.push(Em.I18n.t('admin.rm_highAvailability.error.resourceManagerStarted'));
    }
    if (App.HostComponent.find().filterProperty('componentName', 'ZOOKEEPER_SERVER').length < 3) {
      message.push(Em.I18n.t('admin.rm_highAvailability.error.zooKeeperNum'));
    }

    if (App.get('allHostNames.length') < 3) {
      message.push(Em.I18n.t('admin.rm_highAvailability.error.hostsNum'));
    }
    if (message.length > 0) {
      this.showErrorPopup(message);
      return false;
    }
    App.router.transitionTo('main.services.enableRMHighAvailability');
    return true;
  },

  /**
   * add Hawq Standby
   * @return {Boolean}
   */
  addHawqStandby: function addHawqStandby() {
    App.router.transitionTo('main.services.addHawqStandby');
    return true;
  },

  /**
   * remove Hawq Standby
   * @return {Boolean}
   */
  removeHawqStandby: function removeHawqStandby() {
    App.router.transitionTo('main.services.removeHawqStandby');
    return true;
  },

  /**
   * activate Hawq Standby
   * @return {Boolean}
   */
  activateHawqStandby: function activateHawqStandby() {
    App.router.transitionTo('main.services.activateHawqStandby');
    return true;
  },

  /**
   * enable Ranger Admin High Availability
   * @return {Boolean}
   */
  enableRAHighAvailability: function enableRAHighAvailability() {
    App.router.transitionTo('main.services.enableRAHighAvailability');
    return true;
  },

  /**
   * enable NameNode Federation
   * @return {Boolean}
   */
  enableNameNodeFederation: function enableNameNodeFederation() {
    //Prerequisite Checks
    var message = [];
    if (!App.HostComponent.find().filterProperty('componentName', 'ZOOKEEPER_SERVER').everyProperty('workStatus', 'STARTED')) {
      message.push(Em.I18n.t('admin.nameNodeFederation.wizard.required.zookeepers'));
    }

    if (!App.HostComponent.find().filterProperty('componentName', 'JOURNALNODE').everyProperty('workStatus', 'STARTED')) {
      message.push(Em.I18n.t('admin.nameNodeFederation.wizard.required.journalnodes'));
    }
    if (message.length > 0) {
      this.showErrorPopup(message);
      return false;
    }
    App.router.transitionTo('main.services.enableNameNodeFederation');
    return true;
  },

  /**
   * open Manage JournalNode Wizard if there are two started NameNodes with defined active/standby state
   * @returns {boolean}
   */
  manageJournalNode: function manageJournalNode() {
    var nameNodes = App.HostComponent.find().filterProperty('componentName', 'NAMENODE');
    if (nameNodes.someProperty('displayNameAdvanced', 'Active NameNode') && nameNodes.someProperty('displayNameAdvanced', 'Standby NameNode')) {
      App.router.transitionTo('main.services.manageJournalNode');
      return true;
    }
    this.showErrorPopup(Em.I18n.t('admin.manageJournalNode.warning'));
    return false;
  },

  /**
   * join or wrap message depending on whether it is array or string
   * @param message
   * @return {*}
   */
  joinMessage: function joinMessage(message) {
    if (Array.isArray(message)) {
      return message.join('<br/>');
    } else {
      return '<p>' + message + '</p>';
    }
  },

  showErrorPopup: function showErrorPopup(message) {
    message = this.joinMessage(message);
    App.ModalPopup.show({
      header: Em.I18n.t('common.error'),
      bodyClass: Ember.View.extend({
        template: Ember.Handlebars.compile(message)
      }),
      secondary: false
    });
  }
});

});

require.register("controllers/main/admin/kerberos", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
require('controllers/main/admin/kerberos/step4_controller');

App.MainAdminKerberosController = App.KerberosWizardStep4Controller.extend({
  name: 'mainAdminKerberosController',

  /**
   * @type {boolean}
   * @deafult false
   */
  securityEnabled: false,

  /**
   * @type {boolean}
   * @default false
   */
  defaultKerberosLoaded: false,

  /**
   * @type {boolean}
   * @default false
   */
  dataIsLoaded: false,

  /**
   * @type {boolean}
   * @default true
   */
  isRecommendedLoaded: true,

  /**
   * @type {boolean}
   * @default false
   */
  isEditMode: false,

  /**
   * @type {string}
   */
  kdc_type: '',

  kdcTypesValues: {
    'mit-kdc': Em.I18n.t('admin.kerberos.wizard.step1.option.kdc'),
    'active-directory': Em.I18n.t('admin.kerberos.wizard.step1.option.ad'),
    'ipa': Em.I18n.t('admin.kerberos.wizard.step1.option.ipa'),
    'none': Em.I18n.t('admin.kerberos.wizard.step1.option.manual')
  },

  getAddSecurityWizardStatus: function getAddSecurityWizardStatus() {
    return App.db.getSecurityWizardStatus();
  },
  setAddSecurityWizardStatus: function setAddSecurityWizardStatus(status) {
    App.db.setSecurityWizardStatus(status);
  },

  setDisableSecurityStatus: function setDisableSecurityStatus(status) {
    App.db.setDisableSecurityStatus(status);
  },
  getDisableSecurityStatus: function getDisableSecurityStatus(status) {
    return App.db.getDisableSecurityStatus();
  },

  notifySecurityOff: false,
  notifySecurityAdd: false,

  notifySecurityOffPopup: function notifySecurityOffPopup() {
    var self = this;
    this.checkServiceWarnings().then(function () {
      App.ModalPopup.show({
        header: Em.I18n.t('popup.confirmation.commonHeader'),
        primary: Em.I18n.t('ok'),
        onPrimary: function onPrimary() {
          App.db.setSecurityDeployCommands(undefined);
          self.setDisableSecurityStatus("RUNNING");
          App.router.transitionTo('disableSecurity');
          this.hide();
        },
        bodyClass: Ember.View.extend({
          templateName: require('templates/main/admin/kerberos/notify_security_off_popup')
        })
      });
    });
  },

  /**
   * Show confirmation popup for regenerate keytabs
   * @method regenerateKeytabs
   * @param callback function (optional)
   * @return {App.ModalPopup}
   */
  regenerateKeytabs: function regenerateKeytabs(callback) {
    var self = this;

    return App.ModalPopup.show({

      /**
       * True - regenerate keytabs only for missing hosts and components, false - regenerate for all hosts and components
       * @type {boolean}
       */
      regenerateKeytabsOnlyForMissing: false,

      header: Em.I18n.t('admin.kerberos.button.regenerateKeytabs'),

      bodyClass: Em.View.extend({
        templateName: require('templates/main/admin/kerberos/regenerate_keytabs_popup_body')
      }),

      onPrimary: function onPrimary() {
        this._super();
        return self.restartServicesAfterRegenerate(this.get('regenerateKeytabsOnlyForMissing'), callback);
      }
    });
  },

  /**
   * Show confirmation popup for restarting all services and after confirmation regenerate keytabs
   *
   * @param regenerateKeytabsOnlyForMissing {Boolean}
   * @param callback (optional)
   * @returns {*}
   */
  restartServicesAfterRegenerate: function restartServicesAfterRegenerate(regenerateKeytabsOnlyForMissing, callback) {
    var self = this;

    return App.ModalPopup.show({

      /**
       * True - automatically restart services, false - user will have to restart required services manually
       * @type {boolean}
       */
      restartComponents: false,

      header: Em.I18n.t('admin.kerberos.button.regenerateKeytabs'),

      bodyClass: Em.View.extend({
        templateName: require('templates/main/admin/kerberos/restart_services_after_regenerate_body')
      }),

      onPrimary: function onPrimary() {
        this._super();
        var popupContext = this;
        // Keytabs can either be regenerated directly or after updating kerberos descriptor in the callback function
        if (Em.typeOf(callback) === 'function') {
          callback().done(function () {
            self.regenerateKeytabsRequest(regenerateKeytabsOnlyForMissing, popupContext.get('restartComponents'));
          });
        } else {
          self.regenerateKeytabsRequest(regenerateKeytabsOnlyForMissing, popupContext.get('restartComponents'));
        }
      }
    });
  },

  /**
   * Send request to regenerate keytabs
   * @param {boolean} missingOnly determines type of regeneration - missing|all
   * @param {boolean} withAutoRestart determines if the system should automatically restart all services or not after regeneration
   * @returns {$.ajax}
   */
  regenerateKeytabsRequest: function regenerateKeytabsRequest(missingOnly, withAutoRestart) {
    missingOnly = missingOnly || false;

    return App.ajax.send({
      name: "admin.kerberos_security.regenerate_keytabs",
      sender: this,
      data: {
        type: missingOnly ? 'missing' : 'all',
        withAutoRestart: withAutoRestart || false
      },
      success: "regenerateKeytabsSuccess"
    });
  },

  /**
   * Success callback of <code>regenerateKeytabs</code>
   * show background operations popup if appropriate option is set
   *
   * @param data
   * @param opt
   * @param params
   * @param request
   */
  regenerateKeytabsSuccess: function regenerateKeytabsSuccess(data, opt, params, request) {
    var self = this;
    App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
      if (initValue) {
        App.router.get('backgroundOperationsController').showPopup();
      }
      self.set('needsRestartAfterRegenerate', params.withAutoRestart);
    });
  },

  /**
   * Do request to server for restarting all services
   * @method restartAllServices
   * @return {$.ajax}
   */
  restartAllServices: function () {
    if (!App.router.get('backgroundOperationsController.runningOperationsCount')) {
      if (this.get('needsRestartAfterRegenerate')) {
        this.set('needsRestartAfterRegenerate', false);
        App.router.get('mainServiceController').restartAllServices();
      }
    }
  }.observes('controllers.backgroundOperationsController.runningOperationsCount'),

  /**
   * performs cluster check before kerbefos security
   * wizard starts if <code>preKerberizeCheck<code> supports is true
   * otherwise runs <code>startKerberosWizard<code>
   * @method checkAndStartKerberosWizard
   */
  checkAndStartKerberosWizard: function checkAndStartKerberosWizard() {
    if (App.get('supports.preKerberizeCheck')) {
      App.ajax.send({
        name: "admin.kerberos_security.checks",
        sender: this,
        success: "runSecurityCheckSuccess"
      });
    } else {
      this.startKerberosWizard();
    }
  },

  /**
   * success callback of <code>checkAndStartKerberosWizard()</code>
   * if there are some fails - it shows popup else open security wizard
   * @param data {object}
   * @param opt {object}
   * @param params {object}
   */
  runSecurityCheckSuccess: function runSecurityCheckSuccess(data, opt, params) {
    //TODO correct check
    if (data.items.someProperty('UpgradeChecks.status', "FAIL")) {
      var hasFails = data.items.someProperty('UpgradeChecks.status', 'FAIL'),
          header = Em.I18n.t('popup.clusterCheck.Security.header').format(params.label),
          title = Em.I18n.t('popup.clusterCheck.Security.title'),
          alert = Em.I18n.t('popup.clusterCheck.Security.alert');

      App.showClusterCheckPopup(data, {
        header: header,
        failTitle: title,
        failAlert: alert,
        noCallbackCondition: hasFails
      });
    } else {
      this.startKerberosWizard();
    }
  },

  startKerberosWizard: function startKerberosWizard() {
    var self = this;
    this.checkServiceWarnings().then(function () {
      self.setAddSecurityWizardStatus('RUNNING');
      App.router.get('kerberosWizardController').setDBProperty('onClosePath', 'main.admin.adminKerberos.index');
      App.router.transitionTo('adminKerberos.adminAddKerberos');
    });
  },

  /**
   * Loads the security status from server (security_enabled property in cluster-env configuration)
   */
  loadSecurityStatusFromServer: function loadSecurityStatusFromServer() {
    if (App.get('testMode')) {
      this.set('securityEnabled', !App.get('testEnableSecurity'));
      this.set('dataIsLoaded', true);
    } else {
      //get Security Status From Server
      this.getSecurityType();
      return this.getSecurityStatus();
    }
  },

  /**
   * Load security status from server.
   * @returns {$.Deferred}
   */
  getSecurityStatus: function getSecurityStatus() {
    var self = this;
    var dfd = $.Deferred();
    if (App.get('testMode')) {
      this.set('securityEnabled', !App.get('testEnableSecurity'));
      this.set('dataIsLoaded', true);
      dfd.resolve();
    } else {
      //get Security Status From Server
      App.ajax.send({
        name: 'admin.security_status',
        sender: this,
        success: 'getSecurityStatusSuccessCallback',
        error: 'errorCallback'
      }).always(function () {
        self.getSecurityType(function () {
          dfd.resolve();
        });
      });
    }
    return dfd.promise();
  },

  getSecurityStatusSuccessCallback: function getSecurityStatusSuccessCallback(data) {
    this.set('dataIsLoaded', true);
    var securityType = data.Clusters.security_type;
    this.set('securityEnabled', securityType === 'KERBEROS');
  },

  errorCallback: function errorCallback(jqXHR) {
    this.set('dataIsLoaded', true);
    // Show the error popup if the API call received a response from the server.
    // jqXHR.status will be empty when browser cancels the request. Refer to AMBARI-5921 for more info
    if (!!jqXHR.status) {
      this.showSecurityErrorPopup();
    }
  },

  showSecurityErrorPopup: function showSecurityErrorPopup() {
    App.ModalPopup.show({
      header: Em.I18n.t('common.error'),
      secondary: false,
      bodyClass: Ember.View.extend({
        template: Ember.Handlebars.compile('<p>{{t admin.security.status.error}}</p>')
      })
    });
  },

  /**
   * Override <code>App.KerberosWizardStep4Controller</code>
   */
  clearStep: function clearStep() {
    this.set('isEditMode', false);
    this._super();
  },

  /**
   * Override <code>App.KerberosWizardStep4Controller</code>
   *
   * @param {App.ServiceConfigProperty[]} properties
   */
  setStepConfigs: function setStepConfigs(properties) {
    this.get('stepConfigs').clear();
    this._super(properties);
    this.set('selectedService', this.get('stepConfigs')[0]);
    this.get('stepConfigs').forEach(function (serviceConfig) {
      serviceConfig.set('initConfigsLength', serviceConfig.get('configs.length'));
    });
  },

  /**
   * Override <code>App.KerberosWizardStep4Controller</code>
   *
   * @param {App.ServiceConfigProperty[]} configs
   * @returns {App.ServiceConfigProperty[]}
   */
  prepareConfigProperties: function prepareConfigProperties(configs) {
    var self = this;
    var configProperties = configs.slice(0);
    var siteProperties = App.configsCollection.getAll();
    var installedServiceNames = ['Cluster'].concat(App.Service.find().mapProperty('serviceName'));
    configProperties = configProperties.filter(function (item) {
      return installedServiceNames.contains(item.get('serviceName'));
    });
    configProperties.setEach('isSecureConfig', false);
    configProperties.forEach(function (property, item, allConfigs) {
      if (['spnego_keytab', 'spnego_principal'].contains(property.get('name'))) {
        property.addObserver('value', self, 'spnegoPropertiesObserver');
      }
      if (property.get('observesValueFrom')) {
        var observedValue = allConfigs.findProperty('name', property.get('observesValueFrom')).get('value');
        property.set('value', observedValue);
        property.set('recommendedValue', observedValue);
      }
      if (property.get('serviceName') == 'Cluster') {
        property.set('category', 'Global');
      } else {
        property.set('category', property.get('serviceName'));
      }
      // All user identity should be grouped under "Ambari Principals" category
      if (property.get('identityType') == 'user') property.set('category', 'Ambari Principals');
      var siteProperty = siteProperties.findProperty('name', property.get('name'));
      if (siteProperty) {
        if (siteProperty.category === property.get('category')) {
          property.set('displayName', siteProperty.displayName);
          if (siteProperty.index) {
            property.set('index', siteProperty.index);
          }
        }
        if (siteProperty.displayType) {
          property.set('displayType', siteProperty.displayType);
        }
      }
    });
    configProperties.setEach('isEditable', false);
    return configProperties;
  },

  getKDCSessionState: function getKDCSessionState(callback, kdcCancelHandler) {
    var self = this;
    if (this.get('securityEnabled') || App.get('isKerberosEnabled')) {
      this.getSecurityType(function () {
        if (!self.get('isManualKerberos')) {
          App.ajax.send({
            name: 'kerberos.session.state',
            sender: self,
            data: {
              callback: callback
            },
            success: 'checkState',
            kdcCancelHandler: kdcCancelHandler
          });
        } else {
          callback();
        }
      });
    } else {
      callback();
    }
  },

  /**
   * Determines security type.
   *
   * @param {function} [callback] callback function to execute
   * @returns {$.Deferred|null}
   */
  getSecurityType: function getSecurityType(callback) {
    if (this.get('securityEnabled') || App.get('isKerberosEnabled')) {
      if (!this.get('kdc_type')) {
        return App.ajax.send({
          name: 'admin.security.cluster_configs.kerberos',
          sender: this,
          data: {
            clusterName: App.get('clusterName'),
            additionalCallback: callback
          },
          success: 'getSecurityTypeSuccess'
        });
      } else {
        if (Em.typeOf(callback) === 'function') {
          callback();
        }
        return $.Deferred().resolve().promise;
      }
    } else if (Em.typeOf(callback) === 'function') {
      callback();
    } else {
      return $.Deferred().resolve().promise;
    }
  },

  getSecurityTypeSuccess: function getSecurityTypeSuccess(data, opt, params) {
    var kdcType = data.items && data.items[0] && Em.getWithDefault(Em.getWithDefault(data.items[0], 'configurations', []).findProperty('type', 'kerberos-env') || {}, 'properties.kdc_type', 'none') || 'none';
    this.set('kdc_type', kdcType);
    if (Em.typeOf(params.additionalCallback) === 'function') {
      params.additionalCallback();
    }
  },

  isManualKerberos: Em.computed.equal('kdc_type', 'none'),

  checkState: function checkState(data, opt, params) {
    var res = Em.get(data, 'Services.attributes.kdc_validation_result');
    var message = Em.get(data, 'Services.attributes.kdc_validation_failure_details');
    if (res.toUpperCase() === "OK") {
      params.callback();
    } else {
      App.showInvalidKDCPopup(opt, App.format.kdcErrorMsg(message, false));
    }
  },

  /**
   * Determines if some config value is changed
   * @type {boolean}
   */
  isPropertiesChanged: Em.computed.someBy('stepConfigs', 'isPropertiesChanged', true),

  /**
   * Determines if the save button is disabled
   */
  isSaveButtonDisabled: Em.computed.or('isSubmitDisabled', '!isPropertiesChanged'),

  /**
   * Determines if the `Disbale Kerberos` and `Regenerate Keytabs` button are disabled
   */
  isKerberosButtonsDisabled: Em.computed.not('isSaveButtonDisabled'),

  makeConfigsEditable: function makeConfigsEditable() {
    if (this.get('stepConfigs') && this.get('stepConfigs.length')) {
      this.set('isEditMode', true);
      this.get('stepConfigs').forEach(function (_stepConfig) {
        _stepConfig.get('configs').setEach('isEditable', true);
        _stepConfig.get('configs').forEach(function (_config) {
          _config.set('isEditable', _config.get('name') != 'realm');
        });
      }, this);
    }
  },

  _updateConfigs: function _updateConfigs() {
    this.makeConfigsUneditable(true);
  },

  /**
   * @method makeConfigsUneditable
   * @param configsUpdated
   */
  makeConfigsUneditable: function makeConfigsUneditable(configsUpdated) {
    this.set('isEditMode', false);
    this.get('stepConfigs').forEach(function (_stepConfig) {
      _stepConfig.get('configs').forEach(function (_config) {
        if (configsUpdated === true) {
          // configsUpdated should be checked for boolean true
          _config.set('savedValue', _config.get('value'));
          _config.set('defaultValue', _config.get('value'));
        } else {
          _config.set('value', _config.get('savedValue') || _config.get('defaultValue'));
        }
        _config.set('isEditable', false);
      });
    }, this);
  },

  /**
   * Update kerberos descriptor and regenerate keytabs
   */
  submit: function submit(context) {
    var callback;
    var self = this;
    var kerberosDescriptor = this.get('kerberosDescriptor');
    var configs = [];
    this.get('stepConfigs').forEach(function (_stepConfig) {
      configs = configs.concat(_stepConfig.get('configs'));
    });
    callback = function callback() {
      return App.ajax.send({
        name: 'admin.kerberos.cluster.artifact.update',
        sender: self,
        data: {
          artifactName: 'kerberos_descriptor',
          data: {
            artifact_data: kerberosDescriptor
          }
        },
        success: '_updateConfigs',
        error: 'createKerberosDescriptor'
      });
    };
    this.updateKerberosDescriptor(kerberosDescriptor, configs);
    if (this.get('isManualKerberos')) {
      callback().done(function () {
        self.regenerateKeytabsRequest(false, false);
      });
    } else {
      this.restartServicesAfterRegenerate(false, callback);
    }
  },

  createKerberosDescriptor: function createKerberosDescriptor(requestData, ajaxOptions, error, opt, params) {
    if (requestData && requestData.status === 404) {
      var artifactName = params.artifactName,
          data = params.data;

      App.ajax.send({
        name: 'admin.kerberos.cluster.artifact.create',
        sender: self,
        data: {
          artifactName: artifactName,
          data: data
        },
        success: '_updateConfigs'
      });
    }
  },

  /**
   * List of the warnings regarding specific services before enabling/disabling Kerberos.
   *
   * @type {String[]}
   */
  serviceAlerts: function () {
    var messages = [];
    var serviceAlertMap = {
      YARN: Em.I18n.t('admin.kerberos.service.alert.yarn')
    };
    var installedServices = App.Service.find().mapProperty('serviceName');
    Em.keys(serviceAlertMap).forEach(function (serviceName) {
      if (installedServices.contains(serviceName)) {
        messages.push(serviceAlertMap[serviceName]);
      }
    });
    return messages;
  }.property(),

  /**
   * Check for additional info to display before enabling/disabling kerberos and show appropriate
   * messages in popup if needed.
   * @returns {$.Deferred} - promise
   */
  checkServiceWarnings: function checkServiceWarnings() {
    var dfd = $.Deferred();
    this.displayServiceWarnings(this.get('serviceAlerts'), dfd);
    return dfd.promise();
  },

  /**
   * Show appropriate message regarding changes affected after enabling/disabling Kerberos
   *
   * @param {String[]} messages - list of the messages to display
   * @param {$.Deferred} dfd - used to break recursive calls and reject/resolve promise returned by <code>checkServiceWarnings</code>
   */
  displayServiceWarnings: function displayServiceWarnings(messages, dfd) {
    var self = this;
    if (!messages.get('length')) {
      dfd.resolve();
    } else {
      App.showConfirmationPopup(function () {
        self.displayServiceWarnings(messages.slice(1), dfd);
      }, messages[0], function () {
        dfd.reject();
      }, Em.I18n.t('common.warning'), Em.I18n.t('common.proceedAnyway'));
    }
  },

  showManageKDCCredentialsPopup: function showManageKDCCredentialsPopup() {
    return App.showManageCredentialsPopup();
  },

  loadStep: function loadStep() {
    var self = this;
    if (this.get('isRecommendedLoaded') === false) {
      return;
    }
    this.clearStep();
    this.getDescriptor().then(function (properties) {
      self.setStepConfigs(self.createServicesStackDescriptorConfigs(properties));
    }).always(function () {
      self.set('isRecommendedLoaded', true);
    });
  },

  showDownloadCsv: function () {
    var hasUpgradePrivilege = App.isAuthorized('CLUSTER.UPGRADE_DOWNGRADE_STACK');
    return hasUpgradePrivilege;
  }.property(),

  downloadCSV: function downloadCSV() {
    App.router.get('kerberosWizardStep5Controller').getCSVData(false);
  }

});

});

require.register("controllers/main/admin/kerberos/disable_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
require('controllers/main/admin/kerberos/progress_controller');

App.KerberosDisableController = App.KerberosProgressPageController.extend(App.WizardEnableDone, {

  name: 'kerberosDisableController',
  clusterDeployState: 'DEFAULT',
  commands: ['startZookeeper', 'stopAllButZookeeper', 'unkerberize', 'deleteKerberos', 'startAllServices'],

  tasksMessagesPrefix: 'admin.kerberos.disable.step',

  loadStep: function loadStep() {
    this.set('content.controllerName', 'kerberosDisableController');
    this.loadTasksStatuses();
    this.loadTasksRequestIds();
    this.loadRequestIds();
    this._super();
  },

  startZookeeper: function startZookeeper() {
    this.startServices(false, ["ZOOKEEPER"], true);
  },

  stopAllButZookeeper: function stopAllButZookeeper() {
    this.stopServices(["ZOOKEEPER"], false);
  },

  unkerberize: function unkerberize() {
    return App.ajax.send({
      name: 'admin.unkerberize.cluster',
      sender: this,
      success: 'startPolling',
      error: 'onTaskErrorWithSkip'
    });
  },

  skipTask: function skipTask() {
    return App.ajax.send({
      name: 'admin.unkerberize.cluster.skip',
      sender: this,
      success: 'startPolling',
      error: 'onTaskError'
    });
  },

  deleteKerberos: function deleteKerberos() {
    return App.ajax.send({
      name: 'common.delete.service',
      sender: this,
      data: {
        serviceName: 'KERBEROS'
      },
      success: 'onTaskCompleted',
      error: 'onTaskCompleted'
    });
  },

  startAllServices: function startAllServices() {
    this.startServices(true);
  }

});

});

require.register("controllers/main/admin/kerberos/progress_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

App.KerberosProgressPageController = App.KerberosWizardController.extend(App.wizardProgressPageControllerMixin, {

  name: 'kerberosProgressPageController',
  clusterDeployState: 'KERBEROS__DEPLOY',
  tasksMessagesPrefix: 'admin.kerberos.wizard.step',
  isRollback: false
});

});

require.register("controllers/main/admin/kerberos/step1_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

var PreCondition = Ember.Object.extend({
  displayText: null,
  checked: false,
  visibilityCriteria: function visibilityCriteria() {
    return true;
  },

  hidden: function () {
    return !this.get('visibilityCriteria')();
  }.property('visibilityCriteria'),

  satisfied: function () {
    return this.get('checked') || this.get('hidden');
  }.property('checked', 'hidden')
});

App.KerberosWizardStep1Controller = Em.Controller.extend({
  name: "kerberosWizardStep1Controller",

  selectedItem: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc'),

  isSubmitDisabled: Em.computed.someBy('selectedOption.preConditions', 'satisfied', false),

  options: Em.A([Em.Object.create({
    displayName: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc'),
    value: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc'),
    preConditions: [PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc.condition.1')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc.condition.2')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc.condition.3')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc.condition.4'),
      visibilityCriteria: function visibilityCriteria() {
        return App.Service.find().someProperty('serviceName', 'ONEFS');
      }
    })]
  }), Em.Object.create({
    displayName: Em.I18n.t('admin.kerberos.wizard.step1.option.ad'),
    value: Em.I18n.t('admin.kerberos.wizard.step1.option.ad'),
    preConditions: [PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ad.condition.1')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ad.condition.2')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ad.condition.3')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ad.condition.4')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ad.condition.5')
    })]
  }), Em.Object.create({
    displayName: Em.I18n.t('admin.kerberos.wizard.step1.option.ipa'),
    value: Em.I18n.t('admin.kerberos.wizard.step1.option.ipa'),
    preConditions: [PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ipa.condition.1')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ipa.condition.2')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ipa.condition.3')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.ipa.condition.4')
    })]
  }), Em.Object.create({
    displayName: Em.I18n.t('admin.kerberos.wizard.step1.option.manual'),
    value: Em.I18n.t('admin.kerberos.wizard.step1.option.manual'),
    preConditions: [PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.manual.condition.1')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.manual.condition.2')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.manual.condition.3')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.manual.condition.4')
    }), PreCondition.create({
      displayText: Em.I18n.t('admin.kerberos.wizard.step1.option.manual.condition.5')
    })]
  })]),

  /**
   * precondition for the selected KDC option
   * whenever the KDC type is changed, all checkboxes for the precondition should be unchecked
   */
  selectedOption: function () {
    var options = this.get('options');
    options.forEach(function (option) {
      option.preConditions.setEach('checked', false);
    });
    return this.get('options').findProperty('value', this.get('selectedItem'));
  }.property('selectedItem'),

  loadStep: function loadStep() {},

  submit: function submit() {
    if (!this.get('isSubmitDisabled')) {
      App.router.send('next');
    }
  }
});

});

require.register("controllers/main/admin/kerberos/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
require('controllers/wizard/step7_controller');

App.KerberosWizardStep2Controller = App.WizardStep7Controller.extend(App.KDCCredentialsControllerMixin, {
  name: "kerberosWizardStep2Controller",

  isKerberosWizard: true,

  selectedServiceNames: ['KERBEROS'],

  allSelectedServiceNames: ['KERBEROS'],

  componentName: 'KERBEROS_CLIENT',

  installedServiceNames: [],

  servicesInstalled: false,

  addMiscTabToPage: false,

  kerberosConfigMap: {
    'ad': {
      configNames: ['ldap_url', 'container_dn', 'ad_create_attributes_template'],
      type: Em.I18n.t('admin.kerberos.wizard.step1.option.ad')
    },
    'mit': {
      configNames: ['kdc_create_attributes'],
      type: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc')
    },
    'ipa': {
      configNames: ['ipa_user_group'],
      type: Em.I18n.t('admin.kerberos.wizard.step1.option.ipa')
    }
  },

  /**
   * @type {boolean} true if test connection to hosts is in progress
   */
  testConnectionInProgress: false,

  /**
   * Should Back-button be disabled
   * @type {boolean}
   */
  isBackBtnDisabled: Em.computed.or('testConnectionInProgress', 'App.router.nextBtnClickInProgress'),

  /**
   * Should Next-button be disabled
   * @type {boolean}
   */
  isSubmitDisabled: function () {
    if (!this.get('stepConfigs.length') || this.get('testConnectionInProgress') || this.get('submitButtonClicked') || App.get('router.nextBtnClickInProgress')) return true;
    return !this.get('stepConfigs').filterProperty('showConfig', true).everyProperty('errorCount', 0) || this.get("miscModalVisible");
  }.property('stepConfigs.@each.errorCount', 'miscModalVisible', 'submitButtonClicked', 'testConnectionInProgress', 'App.router.nextBtnClickInProgress'),

  hostNames: Em.computed.alias('App.allHostNames'),

  serviceConfigTags: [],

  clearStep: function clearStep() {
    this._super();
    this.set('configs', []);
    this.get('serviceConfigTags').clear();
    this.set('servicesInstalled', false);
  },

  isConfigsLoaded: Em.computed.alias('wizardController.stackConfigsLoaded'),

  /**
   * On load function
   * @method loadStep
   */

  loadStep: function loadStep() {
    if (!App.StackService.find().someProperty('serviceName', 'KERBEROS') || !this.get('isConfigsLoaded')) {
      return false;
    }
    this.clearStep();
    App.config.setPreDefinedServiceConfigs(this.get('addMiscTabToPage'));
    var stored = this.get('content.serviceConfigProperties');

    this.set('configs', stored ? App.config.mergeStoredValue(this.getKerberosConfigs(), stored) : this.getKerberosConfigs());

    this.filterConfigs(this.get('configs'));
    if (!this.get('wizardController.skipClientInstall')) {
      this.initializeKDCStoreProperties(this.get('configs'));
    }
    this.applyServicesConfigs(this.get('configs'));
    if (!this.get('wizardController.skipClientInstall')) {
      this.updateKDCStoreProperties(this.get('stepConfigs').findProperty('serviceName', 'KERBEROS').get('configs'));
    }
  },

  /**
   * @method getKerberosConfigs
   * @returns {Array.<T>|*}
   */
  getKerberosConfigs: function getKerberosConfigs() {
    var kerberosConfigTypes = Em.keys(App.config.get('preDefinedServiceConfigs').findProperty('serviceName', 'KERBEROS').get('configTypes'));

    return App.configsCollection.getAll().filter(function (configProperty) {
      var fileName = Em.getWithDefault(configProperty, 'fileName', false);
      var isService = ['KERBEROS'].contains(Em.get(configProperty, 'serviceName'));
      var isFileName = fileName && kerberosConfigTypes.contains(App.config.getConfigTagFromFileName(fileName));
      return isService || isFileName;
    });
  },

  /**
   * Make Active Directory or IPA specific configs visible if user has selected AD or IPA option
   * @param configs
   */
  filterConfigs: function filterConfigs(configs) {
    var kdcType = this.get('content.kerberosOption');
    var kerberosWizardController = this.get('controllers.kerberosWizardController');
    var manageIdentitiesConfig = configs.findProperty('name', 'manage_identities');
    configs.filterProperty('serviceName', 'KERBEROS').setEach('isVisible', true);
    this.setKDCTypeProperty(configs);
    if (kdcType !== Em.I18n.t('admin.kerberos.wizard.step1.option.ad')) {
      kerberosWizardController.overrideVisibility(configs, false, kerberosWizardController.get('exceptionsForNonAdOption'), true);
    }
    if (kdcType === Em.I18n.t('admin.kerberos.wizard.step1.option.manual')) {
      if (kerberosWizardController.get('skipClientInstall')) {
        kerberosWizardController.overrideVisibility(configs, false, kerberosWizardController.get('exceptionsOnSkipClient'));
      }
      return;
    } else if (manageIdentitiesConfig) {
      manageIdentitiesConfig.isVisible = false;
      manageIdentitiesConfig.value = 'true';
    }

    this.setConfigVisibility('ad', configs, kdcType);
    this.setConfigVisibility('mit', configs, kdcType);
    this.setConfigVisibility('ipa', configs, kdcType);
  },

  /**
   *
   * @param {string} type
   * @param {Array} configs
   * @param {string} kdcType
   */
  setConfigVisibility: function setConfigVisibility(type, configs, kdcType) {
    var typeSettings = this.get('kerberosConfigMap')[type];

    typeSettings.configNames.forEach(function (_configName) {
      var config = configs.findProperty('name', _configName);
      if (config) {
        config.isVisible = kdcType === typeSettings.type;
      }
    }, this);
  },

  submit: function submit() {
    var self = this;
    if (this.get('isSubmitDisabled')) return false;
    App.set('router.nextBtnClickInProgress', true);
    this.get('wizardController').deleteKerberosService().always(function () {
      self.configureKerberos();
    });
  },

  configureKerberos: function configureKerberos() {
    var self = this;
    var wizardController = App.router.get(this.get('content.controllerName'));
    var callback = function callback() {
      self.createConfigurations().done(function () {
        self.createKerberosAdminSession().done(function () {
          App.set('router.nextBtnClickInProgress', false);
          App.router.send('next');
        });
      });
    };
    if (wizardController.get('skipClientInstall')) {
      callback();
    } else {
      wizardController.createKerberosResources(callback);
    }
  },

  createConfigurations: function createConfigurations() {
    var service = App.StackService.find().findProperty('serviceName', 'KERBEROS'),
        serviceConfigTags = [],
        allConfigData = [],
        serviceConfigData = [];

    Object.keys(service.get('configTypes')).forEach(function (type) {
      if (!serviceConfigTags.someProperty('type', type)) {
        var obj = this.createKerberosSiteObj(type);
        obj.service_config_version_note = Em.I18n.t('admin.kerberos.wizard.configuration.note');
        serviceConfigTags.pushObject(obj);
      }
    }, this);

    Object.keys(service.get('configTypesRendered')).forEach(function (type) {
      var serviceConfigTag = serviceConfigTags.findProperty('type', type);
      if (serviceConfigTag) {
        serviceConfigData.pushObject(serviceConfigTag);
      }
    }, this);
    if (serviceConfigData.length) {
      allConfigData.pushObject(JSON.stringify({
        Clusters: {
          desired_config: serviceConfigData
        }
      }));
    }
    return App.ajax.send({
      name: 'common.across.services.configurations',
      sender: this,
      data: {
        data: '[' + allConfigData.toString() + ']'
      }
    });
  },

  createKerberosSiteObj: function createKerberosSiteObj(site) {
    var properties = {};
    var content = this.get('stepConfigs')[0].get('configs');
    var configs = content.filterProperty('filename', site + '.xml');
    // properties that should be formated as hosts
    var hostProperties = ['kdc_hosts', 'realm'];

    configs.forEach(function (_configProperty) {
      // do not pass any globals whose name ends with _host or _hosts
      if (_configProperty.isRequiredByAgent !== false) {
        if (hostProperties.contains(_configProperty.name)) {
          properties[_configProperty.name] = App.config.trimProperty({ displayType: 'host', value: _configProperty.value });
        } else {
          properties[_configProperty.name] = App.config.trimProperty(_configProperty);
        }
      }
    }, this);
    this.tweakKdcTypeValue(properties);
    this.tweakManualKdcProperties(properties);
    this.tweakIpaKdcProperties(properties);
    return { "type": site, "properties": properties };
  },

  tweakKdcTypeValue: function tweakKdcTypeValue(properties) {
    var kdcTypesValues = App.router.get('mainAdminKerberosController.kdcTypesValues');
    for (var prop in kdcTypesValues) {
      if (kdcTypesValues.hasOwnProperty(prop)) {
        if (kdcTypesValues[prop] === properties['kdc_type']) {
          properties['kdc_type'] = prop;
        }
      }
    }
  },

  tweakManualKdcProperties: function tweakManualKdcProperties(properties) {
    var kerberosWizardController = this.get('controllers.kerberosWizardController');
    if (properties['kdc_type'] === 'none' || kerberosWizardController.get('skipClientInstall')) {
      if (properties.hasOwnProperty('manage_identities')) {
        properties['manage_identities'] = 'false';
      }
      if (properties.hasOwnProperty('install_packages')) {
        properties['install_packages'] = 'false';
      }
      if (properties.hasOwnProperty('manage_krb5_conf')) {
        properties['manage_krb5_conf'] = 'false';
      }
    }
  },

  tweakIpaKdcProperties: function tweakIpaKdcProperties(properties) {
    if (typeof properties['kdc_type'] === 'undefined') {
      return;
    }
    if (this.get('content.kerberosOption') === App.router.get('mainAdminKerberosController.kdcTypesValues')['ipa']) {
      if (properties.hasOwnProperty('install_packages')) {
        properties['install_packages'] = 'false';
      }
      if (properties.hasOwnProperty('manage_krb5_conf')) {
        properties['manage_krb5_conf'] = 'false';
      }
    }
  },

  /**
   * puts kerberos admin credentials in the live cluster session
   * @returns {*} jqXHr
   */
  createKerberosAdminSession: function createKerberosAdminSession(configs) {
    configs = configs || this.get('stepConfigs')[0].get('configs');
    if (!this.get('wizardController.skipClientInstall')) {
      return this.createKDCCredentials(configs);
    }

    var adminPrincipalValue = configs.findProperty('name', 'admin_principal').value;
    var adminPasswordValue = configs.findProperty('name', 'admin_password').value;
    return App.ajax.send({
      name: 'common.cluster.update',
      sender: this,
      data: {
        clusterName: App.get('clusterName'),
        data: [{
          session_attributes: {
            kerberos_admin: { principal: adminPrincipalValue, password: adminPasswordValue }
          }
        }]
      }
    });
  },

  /**
   * shows popup with to warn user
   * @param {Function} primary
   */
  showConnectionInProgressPopup: function showConnectionInProgressPopup(primary) {
    var primaryText = Em.I18n.t('common.exitAnyway');
    var msg = Em.I18n.t('services.service.config.connection.exitPopup.msg');
    App.showConfirmationPopup(primary, msg, null, null, primaryText);
  },

  setKDCTypeProperty: function setKDCTypeProperty(configs) {
    var selectedOption = this.get('content.kerberosOption');
    var kdcTypeProperty = configs.filterProperty('filename', 'kerberos-env.xml').findProperty('name', 'kdc_type');
    var kdcValuesMap = App.router.get('mainAdminKerberosController.kdcTypesValues');
    var kdcTypeValue = Em.keys(kdcValuesMap).filter(function (typeAlias) {
      return Em.get(kdcValuesMap, typeAlias) === selectedOption;
    })[0];
    if (kdcTypeProperty) {
      Em.set(kdcTypeProperty, 'value', kdcValuesMap[kdcTypeValue]);
    }
  },

  /**
   * Override App.WizardStep7Controller#overrideConfigIsRequired
   *
   * @override
   */
  overrideConfigIsRequired: function overrideConfigIsRequired() {}
});

});

require.register("controllers/main/admin/kerberos/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

App.KerberosWizardStep3Controller = App.KerberosProgressPageController.extend({
  name: 'kerberosWizardStep3Controller',
  clusterDeployState: 'KERBEROS_DEPLOY',
  serviceName: 'KERBEROS',
  componentName: 'KERBEROS_CLIENT',
  ignore: undefined,
  heartBeatLostHosts: [],
  commands: ['installKerberos', 'testKerberos'],

  loadStep: function loadStep() {
    this._super();
    this.enableDisablePreviousSteps();
  },

  clearStep: function clearStep() {
    this.get('heartBeatLostHosts').clear();
    this._super();
  },

  installKerberos: function installKerberos() {
    var self = this;
    this.getKerberosClientState().done(function (data) {
      if (data.ServiceComponentInfo.state === 'INIT') {
        App.ajax.send({
          name: 'common.services.update',
          sender: self,
          data: {
            context: Em.I18n.t('requestInfo.kerberosService'),
            ServiceInfo: { "state": "INSTALLED" },
            urlParams: "ServiceInfo/state=INSTALLED&ServiceInfo/service_name=KERBEROS"
          },
          success: 'startPolling',
          error: 'onTaskError'
        });
      } else {
        var hostNames = App.get('allHostNames');
        self.updateComponent('KERBEROS_CLIENT', hostNames, "KERBEROS", "Install");
      }
    });
  },

  /**
   * Get hosts with HEARTBEAT_LOST state.
   *
   * @return {$.Deferred.promise} promise
   */
  getHeartbeatLostHosts: function getHeartbeatLostHosts() {
    return App.ajax.send({
      name: 'hosts.heartbeat_lost',
      sender: this,
      data: {
        clusterName: App.get('clusterName')
      }
    });
  },

  getKerberosClientState: function getKerberosClientState() {
    return App.ajax.send({
      name: 'common.service_component.info',
      sender: this,
      data: {
        serviceName: this.get('serviceName'),
        componentName: this.get('componentName'),
        urlParams: "fields=ServiceComponentInfo/state"
      }
    });
  },

  testKerberos: function testKerberos() {
    var self = this;
    App.ajax.send({
      'name': 'service.item.smoke',
      'sender': this,
      'success': 'startPolling',
      'error': 'onTestKerberosError',
      'kdcCancelHandler': function kdcCancelHandler() {
        App.router.get(self.get('content.controllerName')).setStepsEnable();
        self.get('tasks').objectAt(self.get('currentTaskId')).set('status', 'FAILED');
      },
      'data': {
        'serviceName': this.serviceName,
        'displayName': App.format.role(this.serviceName, true),
        'actionName': this.serviceName + '_SERVICE_CHECK',
        'operationLevel': {
          "level": "CLUSTER",
          "cluster_name": App.get('clusterName')
        }
      }
    });
  },

  onTestKerberosError: function onTestKerberosError(jqXHR, ajaxOptions, error, opt) {
    App.ajax.defaultErrorHandler(jqXHR, opt.url, opt.type, jqXHR.status);
    this.onTaskError(jqXHR, ajaxOptions, error, opt);
  },

  /**
   * Enable or disable previous steps according to tasks statuses
   */
  enableDisablePreviousSteps: function () {
    var wizardController = App.router.get(this.get('content.controllerName'));
    if (this.get('tasks').someProperty('status', 'FAILED')) {
      wizardController.setStepsEnable();
    } else {
      wizardController.setLowerStepsDisable(3);
    }
  }.observes('tasks.@each.status'),

  /**
   * Show or hide warning to ignore errors and continue with the install
   */
  showIgnore: Em.computed.someBy('tasks', 'showRetry', true),

  /**
   * Enable or disable next button if ignore checkbox ticked
   */
  ignoreAndProceed: function () {
    if (this.get('showIgnore')) {
      this.set('isSubmitDisabled', !this.get('ignore'));
    }
  }.observes('ignore', 'showIgnore'),

  retryTask: function retryTask() {
    this._super();
    // retry from the first task (installKerberos) if there is any host in HEARTBEAT_LOST state.
    if (this.get('heartBeatLostHosts').length) {
      this.get('tasks').setEach('status', 'PENDING');
      this.get('tasks').setEach('showRetry', false);
      this.get('heartBeatLostHosts').clear();
    }
  },

  /**
   * Check for complete status and determines:
   *  - if there are any hosts in HEARTBEAT_LOST state. In this case warn about hosts and make step FAILED.
   *
   * @return {undefined}
   */
  statusDidChange: function () {
    var self = this;
    if (this.get('completedStatuses').contains(this.get('status'))) {
      this.getHeartbeatLostHosts().then(function (data) {
        var hostNames = Em.getWithDefault(data || {}, 'items', []).mapProperty('Hosts.host_name');
        if (hostNames.length) {
          self.set('heartBeatLostHosts', hostNames.uniq());
          self.get('tasks').objectAt(0).set('status', 'FAILED');
        }
      });
    }
  }.observes('status')
});

});

require.register("controllers/main/admin/kerberos/step4_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
require('controllers/wizard/step7_controller');

App.KerberosWizardStep4Controller = App.WizardStep7Controller.extend(App.AddSecurityConfigs, App.ToggleIsRequiredMixin, App.KDCCredentialsControllerMixin, {
  name: 'kerberosWizardStep4Controller',

  // stores configurations loaded by ConfigurationsController.getConfigsByTags
  servicesConfigurations: null,

  clearStep: function clearStep() {
    this.set('isRecommendedLoaded', false);
    this.set('submitButtonClicked', false);
    this.set('selectedService', null);
    this.set('stepConfigs', []);
  },

  loadStep: function loadStep() {
    var self, stored;
    self = this;
    this.clearStep();
    stored = this.get('wizardController').loadCachedStepConfigValues(this) || [];
    this.getDescriptor().then(function (kerberosDescriptor) {
      var stepConfigs = self.setStepConfigs(self.createServicesStackDescriptorConfigs(kerberosDescriptor));
      self.set('stepConfigs', stepConfigs);
      // when configurations were stored no need to apply recommendations again
      if (App.get('supports.kerberosStackAdvisor') && !stored.length) {
        self.bootstrapRecommendationPayload(kerberosDescriptor).then(function (recommendations) {
          self.loadServerSideConfigsRecommendations(recommendations).always(function () {
            self.applyServiceConfigs(stepConfigs);
          });
        });
      } else {
        self.applyServiceConfigs(stepConfigs);
      }
    }, function () {
      self.set('isRecommendedLoaded', true);
    });
  },

  /**
   * Get descriptor configs from API endpoint.
   * On <b>Enable Kerberos</b> loads descriptor from cluster STACK resource.
   * On <b>Add Service Wizard</b> first check for cluster's artifacts descriptor and
   * save it presence status, then loads from cluster COMPOSITE resource.
   * Check for cluster/artifacts/kerberos_descriptor is necessary to determine updating or creation
   * kerberos descriptor.
   *
   * @returns {$.Deferred}
   */
  getDescriptor: function getDescriptor() {
    var dfd = $.Deferred();
    var successCallback = function successCallback(data) {
      dfd.resolve(data);
    };

    this.loadClusterDescriptorConfigs(false).then(successCallback);
    return dfd.promise();
  },

  /**
   * Create service config object for Kerberos service.
   *
   * @param {App.ServiceConfigProperty[]} configs
   * @returns {Em.Object}
   */
  createServiceConfig: function createServiceConfig(configs) {
    // Identity configs related to user principal
    var clusterConfigs = configs.filterProperty('serviceName', 'Cluster');
    // storm user principal is not required for ambari operation
    var userConfigs = configs.filterProperty('identityType', 'user');
    var generalConfigs = clusterConfigs.concat(userConfigs).uniq('name');
    var advancedConfigs = configs.filter(function (element) {
      return !generalConfigs.findProperty('name', element.get('name'));
    });
    var categoryForGeneralConfigs = [App.ServiceConfigCategory.create({ name: 'Global', displayName: 'Global' }), App.ServiceConfigCategory.create({ name: 'Ambari Principals', displayName: 'Ambari Principals' })];
    var categoryForAdvancedConfigs = this.createCategoryForServices();
    return [App.ServiceConfig.create({
      displayName: 'General',
      name: 'GENERAL',
      serviceName: 'KERBEROS_GENERAL',
      configCategories: categoryForGeneralConfigs,
      configs: generalConfigs,
      configGroups: [],
      showConfig: true
    }), App.ServiceConfig.create({
      displayName: 'Advanced',
      name: 'ADVANCED',
      serviceName: 'KERBEROS_ADVANCED',
      configCategories: categoryForAdvancedConfigs,
      configs: advancedConfigs,
      configGroups: [],
      showConfig: true
    })];
  },

  /**
   * creates categories for advanced secure configs
   * @returns {[App.ServiceConfigCategory]}
   */
  createCategoryForServices: function createCategoryForServices() {
    var services = [];
    if (this.get('wizardController.name') === 'addServiceController') {
      services = App.StackService.find().filter(function (item) {
        return item.get('isInstalled') || item.get('isSelected');
      });
    } else {
      services = App.Service.find();
    }
    return services.map(function (item) {
      return App.ServiceConfigCategory.create({ name: item.get('serviceName'), displayName: item.get('displayName'), collapsedByDefault: true });
    });
  },

  /**
   * Prepare step configs using stack descriptor properties.
   *
   * @param {App.ServiceConfigProperty[]} configs
   * @param {App.ServiceConfigProperty[]} stackConfigs converted kerberos descriptor
   */
  setStepConfigs: function setStepConfigs(configs, stackConfigs) {
    var configProperties = this.prepareConfigProperties(configs),
        stackConfigProperties = stackConfigs ? this.prepareConfigProperties(stackConfigs) : [],
        alterProperties = ['value', 'initialValue', 'defaultValue'];
    if (this.get('wizardController.name') === 'addServiceController') {
      // config properties for installed services should be disabled on Add Service Wizard
      configProperties.forEach(function (item) {
        if (this.get('installedServiceNames').contains(item.get('serviceName')) || item.get('serviceName') == 'Cluster') {
          item.set('isEditable', false);
        } else if (stackConfigs) {
          var stackConfigProperty = stackConfigProperties.filterProperty('filename', item.get('filename')).findProperty('name', item.get('name'));
          if (stackConfigProperty) {
            alterProperties.forEach(function (alterProperty) {
              item.set(alterProperty, stackConfigProperty.get(alterProperty));
            });
          }
        }
      }, this);
      // Concat properties that are present in the stack's kerberos  descriptor but not in the cluster kerberos descriptor
      stackConfigProperties.forEach(function (_stackConfigProperty) {
        var isPropertyInClusterDescriptor = configProperties.filterProperty('filename', _stackConfigProperty.get('filename')).someProperty('name', _stackConfigProperty.get('name'));
        if (!isPropertyInClusterDescriptor) {
          if (this.get('installedServiceNames').contains(_stackConfigProperty.get('serviceName')) || _stackConfigProperty.get('serviceName') === 'Cluster') {
            _stackConfigProperty.set('isEditable', false);
          }
          configProperties.pushObject(_stackConfigProperty);
        }
      }, this);
    }
    configProperties = App.config.sortConfigs(configProperties);
    var stepConfigs = this.createServiceConfig(configProperties);
    this.set('selectedService', stepConfigs[0]);
    this.get('stepConfigs').pushObjects(stepConfigs);
    return stepConfigs;
  },

  /**
   * Filter configs by installed services for Kerberos Wizard or by installed + selected services
   * for Add Service Wizard.
   * Set property value observer.
   * Set realm property with value from previous configuration step.
   * Set appropriate category for all configs.
   * Hide KDC related credentials properties if kerberos was manually enabled.
   *
   * @param {App.ServiceConfigProperty[]} configs
   * @returns {App.ServiceConfigProperty[]}
   */
  prepareConfigProperties: function prepareConfigProperties(configs) {
    var self = this;
    // stored configs from previous steps (Configure Kerberos or Customize Services for ASW)
    var storedServiceConfigs = this.get('wizardController.content.serviceConfigProperties');
    var installedServiceNames = ['Cluster', 'AMBARI'].concat(App.Service.find().mapProperty('serviceName'));
    var configProperties = configs.slice(0);
    var siteProperties = App.configsCollection.getAll();
    var realmValue;
    // override stored values
    App.config.mergeStoredValue(configProperties, this.get('wizardController').loadCachedStepConfigValues(this));

    // show admin properties in add service wizard
    if (this.get('isWithinAddService')) {
      installedServiceNames = installedServiceNames.concat(this.get('selectedServiceNames'));
    }
    configProperties = configProperties.filter(function (item) {
      return installedServiceNames.contains(item.get('serviceName'));
    });
    if (this.get('wizardController.name') !== 'addServiceController') {
      realmValue = storedServiceConfigs.findProperty('name', 'realm').value;
      configProperties.findProperty('name', 'realm').set('value', realmValue);
      configProperties.findProperty('name', 'realm').set('savedValue', realmValue);
      configProperties.findProperty('name', 'realm').set('recommendedValue', realmValue);
    }

    configProperties.setEach('isSecureConfig', false);
    configProperties.forEach(function (property, item, allConfigs) {
      if (['spnego_keytab', 'spnego_principal'].contains(property.get('name'))) {
        property.addObserver('value', self, 'spnegoPropertiesObserver');
      }
      if (property.get('observesValueFrom') && allConfigs.someProperty('name', property.get('observesValueFrom'))) {
        var observedValue = Em.get(allConfigs.findProperty('name', property.get('observesValueFrom')), 'value');
        property.set('value', observedValue);
        property.set('recommendedValue', observedValue);
        property.set('isVisible', true);
      }
      if (property.get('serviceName') === 'Cluster') {
        property.set('category', 'Global');
      } else {
        property.set('category', property.get('serviceName'));
      }
      // All user identity except storm should be grouped under "Ambari Principals" category
      if (property.get('identityType') == 'user') property.set('category', 'Ambari Principals');
      var siteProperty = siteProperties.findProperty('name', property.get('name'));
      if (siteProperty) {
        if (siteProperty.category === property.get('category')) {
          property.set('displayName', siteProperty.displayName);
          if (siteProperty.index) {
            property.set('index', siteProperty.index);
          }
        }
        if (siteProperty.displayType) {
          property.set('displayType', siteProperty.displayType);
        }
      }
      this.tweakConfigProperty(property);
    }, this);

    return configProperties;
  },

  /**
   * Function to override kerberos descriptor's property values
   */
  tweakConfigProperty: function tweakConfigProperty(config) {
    var defaultHiveMsPort = "9083",
        hiveMSHosts,
        port,
        hiveMSHostNames,
        configValue;
    if (config.name === 'templeton.hive.properties') {
      hiveMSHosts = App.HostComponent.find().filterProperty('componentName', 'HIVE_METASTORE');
      if (hiveMSHosts.length > 1) {
        hiveMSHostNames = hiveMSHosts.mapProperty('hostName');
        port = config.value.match(/:[0-9]{2,4}/);
        port = port ? port[0].slice(1) : defaultHiveMsPort;
        for (var i = 0; i < hiveMSHostNames.length; i++) {
          hiveMSHostNames[i] = "thrift://" + hiveMSHostNames[i] + ":" + port;
        }
        configValue = config.value.replace(/thrift.+[0-9]{2,},/i, hiveMSHostNames.join('\\,') + ",");
        config.set('value', configValue);
        config.set('recommendedValue', configValue);
      }
    }
  },

  /**
   * Sync up values between inherited property and its reference.
   *
   * @param {App.ServiceConfigProperty} configProperty
   */
  spnegoPropertiesObserver: function spnegoPropertiesObserver(configProperty) {
    var stepConfig = this.get('stepConfigs').findProperty('name', 'ADVANCED');

    stepConfig.get('configs').forEach(function (config) {
      if (config.get('observesValueFrom') === configProperty.get('name')) {
        Em.run.once(this, function () {
          config.set('value', configProperty.get('value'));
          config.set('recommendedValue', configProperty.get('value'));
        });
      }
    }, this);
  },

  submit: function submit() {
    this.set('submitButtonClicked', true);
    this.saveConfigurations();
    App.router.send('next');
  },

  saveConfigurations: function saveConfigurations() {
    var kerberosDescriptor = this.get('kerberosDescriptor');
    var configs = [];

    this.get('stepConfigs').forEach(function (_stepConfig) {
      configs = configs.concat(_stepConfig.get('configs'));
    });
    this.updateKerberosDescriptor(kerberosDescriptor, configs);
    App.get('router.kerberosWizardController').saveKerberosDescriptorConfigs(kerberosDescriptor);
  },

  /**
   * Add/update property in `properties` object for each config type with
   * associated kerberos descriptor config value.
   *
   * @private
   * @param {object[]} configurations
   * @param {App.ServiceConfigProperty[]} kerberosDescriptor
   * @returns {object[]}
   */
  mergeDescriptorToConfigurations: function mergeDescriptorToConfigurations(configurations, kerberosDescriptor) {
    return configurations.map(function (configType) {
      var properties = $.extend({}, configType.properties);
      var filteredDescriptor = kerberosDescriptor.filterProperty('filename', configType.type);
      if (filteredDescriptor.length) {
        filteredDescriptor.forEach(function (descriptorConfig) {
          var configName = Em.get(descriptorConfig, 'name');
          properties[configName] = Em.get(descriptorConfig, 'value');
        });
      }
      return {
        type: configType.type,
        version: configType.version,
        tag: configType.tag,
        properties: properties
      };
    });
  },

  loadServerSideConfigsRecommendations: function loadServerSideConfigsRecommendations(recommendations) {
    return App.ajax.send({
      'name': 'config.recommendations',
      'sender': this,
      'data': {
        stackVersionUrl: App.get('stackVersionURL'),
        dataToSend: {
          recommend: 'configurations',
          hosts: App.get('allHostNames'),
          services: this.get('serviceNames'),
          recommendations: recommendations
        }
      },
      'success': 'loadRecommendationsSuccess',
      'error': 'loadRecommendationsError'
    });
  },

  loadRecommendationsError: function loadRecommendationsError(req, ajaxOpts, error, opt) {
    var resp;
    try {
      resp = $.parseJSON(req.responseText);
    } catch (e) {}
    return App.ModalPopup.show({
      header: Em.I18n.t('common.error'),
      secondary: false,
      bodyClass: App.AjaxDefaultErrorPopupBodyView.extend({
        type: opt.type || 'GET',
        url: opt.url,
        status: req.status,
        message: resp && resp.message || req.responseText
      })
    });
  },

  applyServiceConfigs: function applyServiceConfigs(stepConfigs) {
    this.set('isRecommendedLoaded', true);
    this.set('selectedService', stepConfigs[0]);
  },

  /**
   * Callback executed when all configs specified by tags are loaded.
   * Here we handle configurations for instlled services and Kerberos.
   * Gather needed info for recommendation request such as configurations object.
   *
   * @override
   */
  getConfigTagsSuccess: function getConfigTagsSuccess(data) {
    // here we get all installed services including KERBEROS
    var serviceNames = App.Service.find().mapProperty('serviceName').concat(['KERBEROS']).uniq();
    // collect all config types for selected services
    var installedServiceSites = Array.prototype.concat.apply([], App.config.get('preDefinedServiceConfigs').filter(function (serviceConfig) {
      return serviceNames.contains(Em.get(serviceConfig, 'serviceName'));
    }).map(function (service) {
      // when service have no configs return <code>null</code> instead return config types
      if (!service.get('configTypes')) return null;
      return Object.keys(service.get('configTypes'));
    }, this).compact()).uniq(); // cleanup <code>null</code>

    // take all configs for selected services by config types recieved from API response
    var serviceConfigTags = Em.keys(data.Clusters.desired_configs).reduce(function (tags, site) {
      if (data.Clusters.desired_configs.hasOwnProperty(site)) {
        // push cluster-env.xml also since it not associated with any service but need to further processing
        if (installedServiceSites.contains(site) || site === 'cluster-env') {
          tags.push({
            siteName: site,
            tagName: data.Clusters.desired_configs[site].tag,
            newTagName: null
          });
        }
      }
      return tags;
    }, []);
    // store configurations
    this.set('serviceConfigTags', serviceConfigTags);
    this.set('isAppliedConfigLoaded', true);
  },

  /**
   * Prepare all necessary data for recommendations payload.
   *
   * #mutates initialConfigValues
   * @returns {$.Deferred.promise()}
   */
  bootstrapRecommendationPayload: function bootstrapRecommendationPayload(kerberosDescriptor) {
    var dfd = $.Deferred();
    var self = this;

    this.getServicesConfigurations().then(function (configurations) {
      var recommendations = self.getBlueprintPayloadObject(configurations, kerberosDescriptor);
      self.set('servicesConfigurations', configurations);
      self.set('initialConfigValues', recommendations.blueprint.configurations);
      dfd.resolve(recommendations);
    });
    return dfd.promise();
  },

  getServicesConfigurations: function getServicesConfigurations() {
    var dfd = $.Deferred();
    var self = this;
    var configs, servicesConfigurations;
    if (this.get('isWithinAddService')) {
      configs = this.get('content.serviceConfigProperties');
      servicesConfigurations = configs.reduce(function (configTags, property) {
        var fileName = App.config.getConfigTagFromFileName(property.filename),
            configType;
        if (!configTags.someProperty('type', fileName)) {
          configTags.push({
            type: fileName,
            properties: {}
          });
        }
        configType = configTags.findProperty('type', fileName);
        configType.properties[property.name] = property.value;
        return configTags;
      }, []);
      dfd.resolve(servicesConfigurations);
    } else {
      this.getConfigTags().then(function () {
        App.router.get('configurationController').getConfigsByTags(self.get('serviceConfigTags')).done(function (configurations) {
          dfd.resolve(configurations);
        });
      });
    }

    return dfd.promise();
  },

  /**
   * @override
   */
  _saveRecommendedValues: function _saveRecommendedValues(data) {
    var recommendedConfigurations = Em.getWithDefault(data, 'resources.0.recommendations.blueprint.configurations', {});
    var allConfigs = Array.prototype.concat.apply([], this.get('stepConfigs').mapProperty('configs'));
    var self = this;
    // iterate by each config file name e.g. hdfs-site
    var groupedProperties = this.groupRecommendationProperties(recommendedConfigurations, this.get('servicesConfigurations'), allConfigs);
    var newProperties = [];
    Em.keys(groupedProperties.add).forEach(function (fileName) {
      var serviceName = self.getServiceByFilename(fileName);
      Em.keys(groupedProperties.add[fileName]).forEach(function (propertyName) {
        var property = self._createNewProperty(propertyName, fileName, serviceName, groupedProperties.add[fileName][propertyName]);
        property.set('category', serviceName);
        property.set('isOverridable', false);
        property.set('supportsFinal', false);
        property.set('isUserProperty', false);
        property.set('filename', fileName);
        newProperties.push(property);
      });
    });
    Array.prototype.push.apply(self.getServicesConfigObject().get('configs'), newProperties);
    Em.keys(groupedProperties.update).forEach(function (fileName) {
      Em.keys(groupedProperties.update[fileName]).forEach(function (propertyName) {
        var configProperty = allConfigs.filterProperty('filename', fileName).findProperty('name', propertyName);
        if (configProperty) {
          self._updateConfigByRecommendation(configProperty, groupedProperties.update[fileName][propertyName], true, false);
        }
      });
    });
    Em.keys(groupedProperties.delete).forEach(function (fileName) {
      Em.keys(groupedProperties.delete[fileName]).forEach(function (propertyName) {
        var servicesConfigs = self.getServicesConfigObject().get('configs');
        servicesConfigs.removeObject(servicesConfigs.filterProperty('filename', fileName).findProperty('name', propertyName));
      });
    });
  },

  /**
   * Returns category where services' configuration located.
   *
   * @returns {App.ServiceConfig}
   */
  getServicesConfigObject: function getServicesConfigObject() {
    return this.get('stepConfigs').findProperty('name', 'ADVANCED');
  }

});

});

require.register("controllers/main/admin/kerberos/step5_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var App = require('app');
var stringUtils = require('utils/string_utils');
var fileUtils = require('utils/file_utils');

App.KerberosWizardStep5Controller = App.KerberosProgressPageController.extend(App.AddSecurityConfigs, {
  name: 'kerberosWizardStep5Controller',

  /**
   * @type {Array}
   */
  csvData: [],

  /**
   * @type {Array}
   */
  kdcProperties: [{
    key: Em.I18n.t('admin.kerberos.wizard.step1.option.kdc'),
    properties: ['kdc_type', 'kdc_hosts', 'realm', 'executable_search_paths']
  }, {
    key: Em.I18n.t('admin.kerberos.wizard.step1.option.ad'),
    properties: ['kdc_type', 'kdc_hosts', 'realm', 'ldap_url', 'container_dn', 'executable_search_paths']
  }, {
    key: Em.I18n.t('admin.kerberos.wizard.step1.option.ipa'),
    properties: ['kdc_type', 'kdc_hosts', 'realm', 'executable_search_paths']
  }, {
    key: Em.I18n.t('admin.kerberos.wizard.step1.option.manual'),
    properties: ['kdc_type', 'realm', 'executable_search_paths']
  }],

  isCSVRequestInProgress: false,

  submit: function submit() {
    App.router.send('next');
  },

  /**
   * get CSV data from the server
   */
  getCSVData: function getCSVData(skipDownload) {
    this.set('isCSVRequestInProgress', true);
    return App.ajax.send({
      name: 'admin.kerberos.cluster.csv',
      sender: this,
      data: {
        'skipDownload': skipDownload
      },
      success: 'getCSVDataSuccessCallback',
      error: 'getCSVDataSuccessCallback'
    });
  },

  /**
   * get CSV data from server success callback
   * @param data {string}
   * @param opt {object}
   * @param params {object}
   */
  getCSVDataSuccessCallback: function getCSVDataSuccessCallback(data, opt, params) {
    this.set('csvData', this.prepareCSVData(data.split('\n')));
    this.set('isCSVRequestInProgress', false);
    if (!Em.get(params, 'skipDownload')) {
      fileUtils.downloadTextFile(stringUtils.arrayToCSV(this.get('csvData')), 'csv', 'kerberos.csv');
    }
  },

  prepareCSVData: function prepareCSVData(array) {
    for (var i = 0; i < array.length; i += 1) {
      array[i] = array[i].split(',');
    }

    return array;
  },

  /**
   * Send request to unkerberisze cluster
   * @returns {$.ajax}
   */
  unkerberizeCluster: function unkerberizeCluster() {
    return App.ajax.send({
      name: 'admin.unkerberize.cluster',
      sender: this,
      success: 'goToNextStep',
      error: 'goToNextStep'
    });
  },

  goToNextStep: function goToNextStep() {
    this.clearStage();
    App.router.transitionTo('step5');
  },

  isSubmitDisabled: Em.computed.notExistsIn('status', ['COMPLETED', 'FAILED']),

  confirmProperties: function () {
    var wizardController = App.router.get('kerberosWizardController');
    var kdc_type = wizardController.get('content.kerberosOption'),
        kdcTypeProperties = this.get('kdcProperties').filter(function (item) {
      return item.key === kdc_type;
    }),
        properties = kdcTypeProperties.length ? kdcTypeProperties[0].properties : [];

    return wizardController.get('content.serviceConfigProperties').filter(function (item) {
      return properties.contains(item.name);
    }).map(function (item) {
      item['label'] = Em.I18n.t('admin.kerberos.wizard.step5.' + item['name'] + '.label');
      return item;
    });
  }.property('App.router.kerberosWizardController.content.@each.serviceConfigProperties')
});

});

require.register("controllers/main/admin/kerberos/step6_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

App.KerberosWizardStep6Controller = App.KerberosProgressPageController.extend({
  name: 'kerberosWizardStep6Controller',
  clusterDeployState: 'KERBEROS_DEPLOY',
  commands: ['stopServices'],

  stopServices: function stopServices() {
    App.ajax.send({
      name: 'common.services.update',
      data: {
        context: "Stop services",
        "ServiceInfo": {
          "state": "INSTALLED"
        }
      },
      sender: this,
      success: 'startPolling',
      error: 'onTaskError'
    });
  },

  loadStep: function loadStep() {
    this.checkComponentsRemoval();
    this._super();
  },

  /**
   * remove Application Timeline Server component if needed.
   */
  checkComponentsRemoval: function checkComponentsRemoval() {
    if (App.Service.find().someProperty('serviceName', 'YARN') && !App.get('doesATSSupportKerberos') && !this.get('commands').contains('deleteATS') && App.HostComponent.find().findProperty('componentName', 'APP_TIMELINE_SERVER')) {
      this.get('commands').pushObject('deleteATS');
    }
  },

  /**
   * Remove Application Timeline Server from the host.
   * @returns {$.Deferred}
   */
  deleteATS: function deleteATS() {
    return App.ajax.send({
      name: 'common.delete.host_component',
      sender: this,
      data: {
        componentName: 'APP_TIMELINE_SERVER',
        hostName: App.HostComponent.find().findProperty('componentName', 'APP_TIMELINE_SERVER').get('hostName')
      },
      success: 'onDeleteATSSuccess',
      error: 'onDeleteATSError'
    });
  },

  onDeleteATSSuccess: function onDeleteATSSuccess() {
    this.onTaskCompleted();
  },

  onDeleteATSError: function onDeleteATSError(error) {
    if (error.responseText.indexOf('org.apache.ambari.server.controller.spi.NoSuchResourceException') !== -1) {
      this.onDeleteATSSuccess();
    }
  }
});

});

require.register("controllers/main/admin/kerberos/step7_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.KerberosWizardStep7Controller = App.KerberosProgressPageController.extend(App.AddSecurityConfigs, {
  name: 'kerberosWizardStep7Controller',
  clusterDeployState: 'KERBEROS_DEPLOY',
  isSingleRequestPage: true,
  request: {},
  commands: [],
  contextForPollingRequest: Em.I18n.t('requestInfo.kerberizeCluster'),

  /**
   * Define whether show Back button
   * @type {Boolean}
   */
  isBackButtonDisabled: true,

  /**
   * Start cluster kerberization. On retry just unkerberize and kerbrize cluster.
   * @param {bool} isRetry
   */
  setRequest: function setRequest(isRetry) {
    var kerberizeRequest = {
      name: 'KERBERIZE_CLUSTER',
      ajaxName: 'admin.kerberize.cluster',
      ajaxData: {
        data: {
          Clusters: {
            security_type: "KERBEROS"
          }
        }
      }
    };
    if (isRetry) {
      // on retry send force update
      this.set('request', {
        name: 'KERBERIZE_CLUSTER',
        ajaxName: 'admin.kerberize.cluster.force'
      });
      this.clearStage();
      this.loadStep();
    } else {
      this.set('request', kerberizeRequest);
    }
  },

  /**
   * Send request to unkerberisze cluster
   * @returns {$.ajax}
   */
  unkerberizeCluster: function unkerberizeCluster() {
    return App.ajax.send({
      name: 'admin.unkerberize.cluster',
      sender: this,
      success: 'goToNextStep',
      error: 'goToNextStep'
    });
  },

  goToNextStep: function goToNextStep() {
    this.clearStage();
    App.router.transitionTo('step7');
  },

  retry: function retry() {
    this.set('showRetry', false);
    this.removeObserver('tasks.@each.status', this, 'onTaskStatusChange');
    this.set('status', 'IN_PROGRESS');
    this.get('tasks').setEach('status', 'PENDING');
    this.setRequest(true);
  },

  /**
   * Enable or disable previous steps according to tasks statuses
   */
  enableDisablePreviousSteps: function () {
    var wizardController = App.router.get(this.get('content.controllerName'));
    if (this.get('tasks').someProperty('status', 'FAILED')) {
      wizardController.enableStep(4);
      this.set('isBackButtonDisabled', false);
    } else {
      wizardController.setLowerStepsDisable(6);
      this.set('isBackButtonDisabled', true);
    }
  }.observes('tasks.@each.status')
});

});

require.register("controllers/main/admin/kerberos/step8_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

App.KerberosWizardStep8Controller = App.KerberosProgressPageController.extend({
  name: 'kerberosWizardStep8Controller',
  clusterDeployState: 'KERBEROS_DEPLOY',
  commands: ['startServices'],

  startServices: function startServices() {
    var skipServiceCheck = App.router.get('clusterController.ambariProperties')['skip.service.checks'] === "true";
    App.ajax.send({
      name: 'common.services.update',
      sender: this,
      data: {
        "context": "Start services",
        "ServiceInfo": {
          "state": "STARTED"
        },
        urlParams: "params/run_smoke_test=" + !skipServiceCheck
      },
      success: 'startPolling',
      error: 'startServicesErrorCallback'
    });
  },

  isSubmitDisabled: Em.computed.notExistsIn('status', ['COMPLETED', 'FAILED'])

});

});

require.register("controllers/main/admin/kerberos/wizard_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.KerberosWizardController = App.WizardController.extend(App.InstallComponent, {

  exceptionsOnSkipClient: [{ 'KDC': 'realm' }, { 'KDC': 'kdc_type' }, { 'Advanced kerberos-env': 'executable_search_paths' }],

  exceptionsForNonAdOption: [{ "Advanced kerberos-env": "password_length" }, { "Advanced kerberos-env": "password_min_digits" }, { "Advanced kerberos-env": "password_min_lowercase_letters" }, { "Advanced kerberos-env": "password_min_punctuation" }, { "Advanced kerberos-env": "password_min_uppercase_letters" }, { "Advanced kerberos-env": "password_min_whitespace" }],

  name: 'kerberosWizardController',

  totalSteps: 8,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('admin.kerberos.wizard.header'),

  isKerberosWizard: true,

  stackConfigsLoaded: false,
  /**
   * Used for hiding back button in wizard
   */
  hideBackButton: true,

  /**
   * Check if Kerberos Clients should be installed.
   * Clients installation will be skipped if Manual Kerberization was selected.
   *
   * @type {Boolean}
   */
  skipClientInstall: function () {
    if (this.get('content.kerberosOption')) {
      return this.get('content.kerberosOption') === Em.I18n.t('admin.kerberos.wizard.step1.option.manual');
    }
    return false;
  }.property('content.kerberosOption'),

  kerberosDescriptorConfigs: null,

  content: Em.Object.create({
    controllerName: 'kerberosWizardController',
    serviceName: 'KERBEROS',
    kerberosOption: null,
    cluster: null,
    services: [],
    advancedServiceConfig: null,
    serviceConfigProperties: [],
    failedTask: null
  }),

  /**
   * set current step
   * @param {string} currentStep
   * @param {boolean} completed
   * @param {boolean} skipStateSave
   */
  setCurrentStep: function setCurrentStep(currentStep, completed, skipStateSave) {
    this._super(currentStep, completed);
    if (App.get('testMode') || skipStateSave) {
      return;
    }
    App.clusterStatus.setClusterStatus({
      clusterName: this.get('content.cluster.name'),
      clusterState: 'KERBEROS_DEPLOY',
      wizardControllerName: 'kerberosWizardController',
      localdb: App.db.data
    });
  },

  setStepsEnable: function () {
    for (var i = 1; i <= this.get('totalSteps'); i++) {
      var step = this.get('isStepDisabled').findProperty('step', i);
      if (i <= this.get('currentStep') && App.get('router.clusterController.isLoaded')) {
        step.set('value', false);
      } else {
        step.set('value', i != this.get('currentStep'));
      }
    }
  }.observes('currentStep', 'App.router.clusterController.isLoaded'),

  /**
   * return new object extended from clusterStatusTemplate
   * @return Object
   */
  getCluster: function getCluster() {
    return jQuery.extend({}, this.get('clusterStatusTemplate'), { name: App.get('clusterName') });
  },

  updateClusterEnvData: function updateClusterEnvData(configs) {
    var kerberosDescriptor = this.get('kerberosDescriptorConfigs');
    configs['security_enabled'] = true;
    configs['kerberos_domain'] = kerberosDescriptor.properties.realm;
    return configs;
  },

  dataLoading: function dataLoading() {
    var dfd = $.Deferred();
    this.connectOutlet('loading');
    if (App.router.get('clusterController.isLoaded') && App.router.get('clusterController.isComponentsStateLoaded')) {
      dfd.resolve();
    } else {
      var interval = setInterval(function () {
        if (App.router.get('clusterController.isLoaded') && App.router.get('clusterController.isComponentsStateLoaded')) {
          dfd.resolve();
          clearInterval(interval);
        }
      }, 50);
    }
    return dfd.promise();
  },
  /**
   * save status of the cluster.
   * @param clusterStatus object with status,requestId fields.
   */
  saveClusterStatus: function saveClusterStatus(clusterStatus) {
    var oldStatus = this.toObject(this.get('content.cluster'));
    clusterStatus = jQuery.extend(oldStatus, clusterStatus);
    if (clusterStatus.requestId) {
      clusterStatus.requestId.forEach(function (requestId) {
        if (clusterStatus.oldRequestsId.indexOf(requestId) === -1) {
          clusterStatus.oldRequestsId.push(requestId);
        }
      }, this);
    }
    this.set('content.cluster', clusterStatus);
    this.save('cluster');
  },

  saveConfigTag: function saveConfigTag(tag) {
    App.db.setKerberosWizardConfigTag(tag);
    this.set('content.' + [tag.name], tag.value);
  },

  saveKerberosOption: function saveKerberosOption(stepController) {
    this.setDBProperty('kerberosOption', stepController.get('selectedItem'));
    this.set('content.kerberosOption', stepController.get('selectedItem'));
  },

  /**
   * Override the visibility of a list of form items with a new value
   *
   * @param {Array} itemsArray
   * @param {boolean} newValue
   * @param {Array} exceptions
   */
  overrideVisibility: function overrideVisibility(itemsArray, newValue, exceptions, inverse) {
    newValue = newValue || false;

    for (var i = 0, len = itemsArray.length; i < len; i += 1) {
      if (!App.isEmptyObject(itemsArray[i])) {
        var isException = exceptions.filterProperty(itemsArray[i].category, itemsArray[i].name);
        if (!isException.length && !inverse) {
          itemsArray[i].isVisible = newValue;
        }
        if (isException.length && inverse) {
          itemsArray[i].isVisible = newValue;
        }
      }
    }
  },

  loadKerberosOption: function loadKerberosOption() {
    this.set('content.kerberosOption', this.getDBProperty('kerberosOption'));
  },

  createKerberosResources: function createKerberosResources(callback) {
    var self = this;
    this.createKerberosService().done(function () {
      self.updateAndCreateServiceComponent('KERBEROS_CLIENT').done(function () {
        self.createKerberosHostComponents().done(callback);
      });
    });
  },

  createKerberosService: function createKerberosService() {
    return App.ajax.send({
      name: 'wizard.step8.create_selected_services',
      sender: this,
      data: {
        data: '{"ServiceInfo": { "service_name": "KERBEROS"}}',
        cluster: App.get('clusterName')
      }
    });
  },

  /**
   * Delete Kerberos service if it exists
   *
   * @returns {$.Deferred}
   */
  deleteKerberosService: function deleteKerberosService() {
    var serviceName = 'KERBEROS',
        cachedService = App.cache.services.findProperty('ServiceInfo.service_name', serviceName),
        modelService = App.Service.find(serviceName);

    if (cachedService) {
      App.cache.services.removeObject(cachedService);
    }
    if (modelService.get('isLoaded')) {
      App.serviceMapper.deleteRecord(modelService);
    }
    return App.ajax.send({
      name: 'common.delete.service',
      sender: this,
      data: {
        serviceName: serviceName
      }
    });
  },

  /**
   * Unkerberize cluster. Set cluster `security_type` to "NONE".
   *
   * @returns {$.Deferred}
   */
  unkerberize: function unkerberize() {
    return App.ajax.send({
      name: 'admin.unkerberize.cluster',
      sender: this
    });
  },

  createKerberosHostComponents: function createKerberosHostComponents() {
    var hostNames = App.get('allHostNames');
    var queryStr = '';
    hostNames.forEach(function (hostName) {
      queryStr += 'Hosts/host_name=' + hostName + '|';
    });
    //slice off last symbol '|'
    queryStr = queryStr.slice(0, -1);

    var data = {
      "RequestInfo": {
        "query": queryStr
      },
      "Body": {
        "host_components": [{
          "HostRoles": {
            "component_name": 'KERBEROS_CLIENT'
          }
        }]
      }
    };

    return App.ajax.send({
      name: 'wizard.step8.register_host_to_component',
      sender: this,
      data: {
        cluster: App.get('clusterName'),
        data: JSON.stringify(data)
      }
    });
  },

  loadMap: {
    '1': [{
      type: 'sync',
      callback: function callback() {
        this.loadKerberosOption();
      }
    }],
    '2': [{
      type: 'async',
      callback: function callback() {
        var self = this;
        var dfd = $.Deferred();

        if (!self.get('stackConfigsLoaded')) {
          App.config.loadConfigsFromStack(['KERBEROS']).always(function () {
            self.loadServiceConfigProperties();
            self.set('stackConfigsLoaded', true);
            dfd.resolve();
          });
        } else {
          this.loadServiceConfigProperties();
          dfd.resolve();
        }
        return dfd.promise();
      }
    }],
    '3': [{
      type: 'sync',
      callback: function callback() {
        this.loadTasksStatuses();
        this.loadTasksRequestIds();
        this.loadRequestIds();
      }
    }],
    '4': [{
      type: 'sync',
      callback: function callback() {
        this.loadKerberosDescriptorConfigs();
      }
    }],
    '6': [{
      type: 'sync',
      callback: function callback() {
        this.loadKerberosDescriptorConfigs();
      }
    }]
  },

  /**
   * Remove all loaded data.
   * Created as copy for App.router.clearAllSteps
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  clearTasksData: function clearTasksData() {
    this.saveTasksStatuses(undefined);
    this.saveRequestIds(undefined);
    this.saveTasksRequestIds(undefined);
  },

  /**
   * shows popup with to warn user
   * @param {Function} primary
   * @param {boolean} isCritical
   */
  warnBeforeExitPopup: function warnBeforeExitPopup(primary, isCritical) {
    var primaryText = Em.I18n.t('common.exitAnyway');
    var msg = isCritical ? Em.I18n.t('admin.kerberos.wizard.exit.critical.msg') : Em.I18n.t('admin.kerberos.wizard.exit.warning.msg');
    return App.showConfirmationPopup(primary, msg, null, null, primaryText, isCritical ? 'danger' : 'success');
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    // The in-memory variable for current step should be reset to 1st step.
    this.setCurrentStep('1', false, true);
    // kerberos wizard namespace in the localStorage should be emptied
    this.resetDbNamespace();
  },

  /**
   * Discard changes affected by wizard:
   *   - Unkerberize cluster
   *   - Remove Kerberos service
   *
   * @returns {$.Deferred}
   */
  discardChanges: function discardChanges() {
    var dfd = $.Deferred();
    var self = this;

    this.unkerberize().always(function () {
      self.deleteKerberosService().always(function () {
        dfd.resolve();
      });
    });

    return dfd.promise();
  }
});

});

require.register("controllers/main/admin/serviceAccounts_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

require('controllers/main/service/info/configs');

App.MainAdminServiceAccountsController = App.MainServiceInfoConfigsController.extend({
  name: 'mainAdminServiceAccountsController',
  users: null,
  serviceConfigTags: [],
  content: Em.Object.create({
    serviceName: 'MISC'
  }),
  loadUsers: function loadUsers() {
    this.set('selectedService', this.get('content.serviceName') ? this.get('content.serviceName') : "MISC");
    this.loadServiceConfig();
  },
  loadServiceConfig: function loadServiceConfig() {
    var _this = this;

    App.router.get('configurationController').getCurrentConfigsBySites().done(function (serverConfigs) {
      _this.createConfigObject(serverConfigs);
    });
  },

  /**
   * Generate configuration object that will be rendered
   *
   * @param {Object[]} serverConfigs
   */
  createConfigObject: function createConfigObject(serverConfigs) {
    var configs = [];
    serverConfigs.forEach(function (configObject) {
      configs = configs.concat(App.config.getConfigsFromJSON(configObject, true));
    });
    var miscConfigs = configs.filterProperty('displayType', 'user').filterProperty('category', 'Users and Groups');
    miscConfigs.setEach('isVisible', true);
    this.set('users', miscConfigs);
    this.set('dataIsLoaded', true);
  },

  /**
   * sort miscellaneous configs by specific order
   * @param sortOrder
   * @param arrayToSort
   * @return {Array}
   */
  sortByOrder: function sortByOrder(sortOrder, arrayToSort) {
    var sorted = [];
    if (sortOrder && sortOrder.length > 0) {
      sortOrder.forEach(function (name) {
        var user = arrayToSort.findProperty('name', name);
        if (user) {
          sorted.push({
            isVisible: user.get('isVisible'),
            displayName: user.get('displayName'),
            value: user.get('value')
          });
        }
      });
      return sorted;
    } else {
      return arrayToSort;
    }
  }
});

});

require.register("controllers/main/admin/service_auto_start", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainAdminServiceAutoStartController = Em.Controller.extend({
  name: 'mainAdminServiceAutoStartController',

  /**
   * @type {?object}
   * @default null
   */
  clusterConfigs: null,

  /**
   * @type {Array}
   */
  componentsConfigsCached: [],

  /**
   * @type {object}
   */
  componentsConfigsCachedMap: function () {
    var map = {};
    this.get('componentsConfigsCached').mapProperty('ServiceComponentInfo').forEach(function (component) {
      map[component.component_name] = component.recovery_enabled === 'true';
    });
    return map;
  }.property('componentsConfigsCached.@each.ServiceComponentInfo.recovery_enabled'),

  /**
   * @type {Array}
   */
  componentsConfigsGrouped: [],

  /**
   * @type {boolean}
   */
  isLoaded: false,

  /**
   * @type {boolean}
   */
  isGeneralRecoveryEnabled: false,

  /**
   * @type {boolean}
   */
  isGeneralRecoveryEnabledCached: false,

  /**
   * @type {boolean}
   */
  saveInProgress: false,

  /**
   * @type {boolean}
   */
  isModified: function () {
    return this.get('isGeneralModified') || this.get('isComponentModified');
  }.property('isGeneralModified', 'isComponentModified'),

  /**
   * @type {boolean}
   */
  isGeneralModified: function () {
    return this.get('isGeneralRecoveryEnabled') !== this.get('isGeneralRecoveryEnabledCached');
  }.property('isGeneralRecoveryEnabled', 'isGeneralRecoveryEnabledCached'),

  /**
   * @type {boolean}
   */
  isComponentModified: function () {
    var componentsConfigsCachedMap = this.get('componentsConfigsCachedMap');

    return this.get('componentsConfigsGrouped').some(function (component) {
      return component.get('recoveryEnabled') !== componentsConfigsCachedMap[component.get('componentName')];
    });
  }.property('componentsConfigsGrouped.@each.recoveryEnabled', 'componentsConfigsCachedMap'),

  parseComponentConfigs: function parseComponentConfigs(componentsConfigsCached) {
    componentsConfigsCached.sortPropertyLight('ServiceComponentInfo.service_name');
    var componentsConfigsGrouped = [];
    var servicesMap = componentsConfigsCached.mapProperty('ServiceComponentInfo.service_name').uniq().toWickMap();

    componentsConfigsCached.mapProperty('ServiceComponentInfo').forEach(function (component) {
      componentsConfigsGrouped.push(Em.Object.create({
        serviceDisplayName: App.format.role(component.service_name, true),
        isFirst: servicesMap[component.service_name],
        componentName: component.component_name,
        displayName: App.format.role(component.component_name, false),
        recoveryEnabled: component.recovery_enabled === 'true'
      }));
      servicesMap[component.service_name] = false;
    });
    return componentsConfigsGrouped;
  },

  load: function load() {
    var self = this;
    var clusterConfigController = App.router.get('configurationController');
    clusterConfigController.updateConfigTags().always(function () {
      clusterConfigController.getCurrentConfigsBySites(['cluster-env']).done(function (data) {
        self.set('clusterConfigs', data[0].properties);
        self.set('isGeneralRecoveryEnabled', data[0].properties.recovery_enabled === 'true');
        self.set('isGeneralRecoveryEnabledCached', self.get('isGeneralRecoveryEnabled'));
        self.loadComponentsConfigs().then(function () {
          self.set('isLoaded', true);
        });
      });
    });
  },

  loadComponentsConfigs: function loadComponentsConfigs() {
    return App.ajax.send({
      name: 'components.get_category',
      sender: this,
      success: 'loadComponentsConfigsSuccess'
    });
  },

  loadComponentsConfigsSuccess: function loadComponentsConfigsSuccess(data) {
    this.set('componentsConfigsCached', data.items);
    this.set('componentsConfigsGrouped', this.parseComponentConfigs(data.items));
  },

  saveClusterConfigs: function saveClusterConfigs(clusterConfigs, recoveryEnabled) {
    clusterConfigs.recovery_enabled = String(recoveryEnabled);
    return App.ajax.send({
      name: 'admin.save_configs',
      sender: this,
      data: {
        siteName: 'cluster-env',
        properties: clusterConfigs
      }
    });
  },

  saveComponentSettingsCall: function saveComponentSettingsCall(recoveryEnabled, components) {
    return App.ajax.send({
      name: 'components.update',
      sender: this,
      data: {
        ServiceComponentInfo: {
          recovery_enabled: recoveryEnabled
        },
        query: 'ServiceComponentInfo/component_name.in(' + components.join(',') + ')'
      }
    });
  },

  syncStatus: function syncStatus() {
    var componentsConfigsGrouped = this.get('componentsConfigsGrouped');
    this.set('isGeneralRecoveryEnabledCached', this.get('isGeneralRecoveryEnabled'));
    this.get('componentsConfigsCached').mapProperty('ServiceComponentInfo').forEach(function (component) {
      var actualComponent = componentsConfigsGrouped.findProperty('componentName', component.component_name);
      component.recovery_enabled = String(actualComponent.get('recoveryEnabled'));
    });
    this.propertyDidChange('componentsConfigsCached');
  },

  restoreCachedValues: function restoreCachedValues() {
    this.set('isGeneralRecoveryEnabled', this.get('isGeneralRecoveryEnabledCached'));
    this.set('componentsConfigsGrouped', this.parseComponentConfigs(this.get('componentsConfigsCached')));
  },

  filterComponentsByChange: function filterComponentsByChange(components, value) {
    var map = this.get('componentsConfigsCachedMap');

    return components.filter(function (component) {
      return component.get('recoveryEnabled') !== map[component.get('componentName')] && component.get('recoveryEnabled') === value;
    }).mapProperty('componentName');
  },

  /**
   * If some configs are changed and user navigates away or select another config-group, show this popup with propose to save changes
   * @param {object} transitionCallback - callback with action to change configs view
   * @return {App.ModalPopup}
   * @method showSavePopup
   */
  showSavePopup: function showSavePopup(transitionCallback) {
    var self = this;
    var title = '';
    var body = '';
    if (typeof transitionCallback === 'function') {
      title = Em.I18n.t('admin.serviceAutoStart.save.popup.transition.title');
      body = Em.I18n.t('admin.serviceAutoStart.save.popup.transition.body');
    } else {
      title = Em.I18n.t('admin.serviceAutoStart.save.popup.title');
      body = Em.I18n.t('admin.serviceAutoStart.save.popup.body');
    }
    return App.ModalPopup.show({
      header: title,
      bodyClass: Ember.View.extend({
        template: Ember.Handlebars.compile(body)
      }),
      footerClass: Em.View.extend({
        templateName: require('templates/main/service/info/save_popup_footer')
      }),
      primary: Em.I18n.t('common.save'),
      secondary: Em.I18n.t('common.cancel'),
      onSave: function onSave() {
        var clusterConfigsCall = void 0,
            enabledComponentsCall = void 0,
            disabledComponentsCall = void 0;

        if (self.get('isGeneralModified')) {
          clusterConfigsCall = self.saveClusterConfigs(self.get('clusterConfigs'), self.get('isGeneralRecoveryEnabled'));
        }

        var enabledComponents = self.filterComponentsByChange(self.get('componentsConfigsGrouped'), true);
        var disabledComponents = self.filterComponentsByChange(self.get('componentsConfigsGrouped'), false);

        if (enabledComponents.length) {
          enabledComponentsCall = self.saveComponentSettingsCall('true', enabledComponents);
        }
        if (disabledComponents.length) {
          disabledComponentsCall = self.saveComponentSettingsCall('false', disabledComponents);
        }
        self.set('saveInProgress', true);
        $.when(clusterConfigsCall, enabledComponentsCall, disabledComponentsCall).done(function () {
          if (typeof transitionCallback === 'function') {
            transitionCallback();
          }
          self.syncStatus();
        }).always(function () {
          self.set('saveInProgress', false);
        });
        this.hide();
      },
      onDiscard: function onDiscard() {
        self.restoreCachedValues();
        if (typeof transitionCallback === 'function') {
          transitionCallback();
        }
        this.hide();
      },
      onCancel: function onCancel() {
        this.hide();
      }
    });
  }
});

});

require.register("controllers/main/admin/stack_and_upgrade_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var stringUtils = require('utils/string_utils');

App.MainAdminStackAndUpgradeController = Em.Controller.extend(App.LocalStorage, {
  name: 'mainAdminStackAndUpgradeController',

  /**
   * @type {boolean}
   */
  isLoaded: false,

  /**
   * @type {object}
   * @default null
   */
  upgradeData: null,

  /**
   * @type {number}
   * @default null
   */
  upgradeId: null,

  /**
   * Start version of upgrade
   * @type {string}
   * @default null
   */
  fromVersion: null,

  /**
   * @type {string}
   * @default null
   */
  upgradeVersion: null,

  /**
   * @type {string}
   * @default null
   */
  upgradeType: null,

  /**
   * @type {Em.Object}
   */
  upgradeTypeConfig: Em.computed.findByKey('upgradeMethods', 'type', 'upgradeType'),

  /**
   * @type {boolean}
   */
  cantBeStarted: Em.computed.alias('upgradeTypeConfig.cantBeStarted'),

  /**
   * @type {boolean}
   */
  showPauseButton: Em.computed.and('!App.upgradeSuspended', '!App.upgradeCompleted', '!App.upgradeInit'),

  /**
   * @type {boolean}
   * @default true
   */
  downgradeAllowed: true,

  /**
   * @type {string}
   * @default null
   */
  upgradeTypeDisplayName: null,

  /**
   * @type {object}
   * @default null
   */
  failuresTolerance: null,

  /**
   * @type {boolean}
   * @default false
   */
  isDowngrade: false,

  /**
   * flag which indicate that upgrade suspended
   * @type {boolean}
   * @default false
   */
  isSuspended: false,

  /**
   * version that currently applied to server
   * should be plain object, because stored to localStorage
   * @type {object|null}
   */
  currentVersion: null,

  /**
   * versions to which cluster could be upgraded
   * @type {Array}
   */
  targetVersions: [],

  /**
   * @type {object}
   * @default null
   */
  slaveComponentStructuredInfo: null,

  /**
   * @type {Array}
   */
  serviceCheckFailuresServicenames: [],

  /**
   * @type {boolean}
   * @default false
   */
  isUpgradeTypesLoaded: false,

  /**
   * @type {string}
   */
  getSupportedUpgradeError: '',

  /**
   * Restricted type of upgrade, can't be viewed in wizard.
   * It's status visible only in upgrade status label
   * @type {boolean}
   * @default false
   */
  isWizardRestricted: false,

  /**
   * @type {string}
   */
  wizardModalTitle: function () {
    var repoVersion = App.RepositoryVersion.find().findProperty('repositoryVersion', this.get('toVersion'));
    return this.getUpgradeDowngradeHeader(this.get('upgradeTypeDisplayName'), this.get('upgradeVersion'), this.get('isDowngrade'), repoVersion);
  }.property('upgradeTypeDisplayName', 'upgradeVersion', 'isDowngrade'),

  /**
   * @param {string} upgradeType
   * @param {string} upgradeVersion
   * @param {boolean} isDowngrade
   * @param {boolean} isPatch
   * @returns {string}
   */
  getUpgradeDowngradeHeader: function getUpgradeDowngradeHeader(upgradeType, upgradeVersion, isDowngrade, repoVersion) {
    if (isDowngrade) {
      return Em.I18n.t('admin.stackUpgrade.dialog.downgrade.header').format(upgradeVersion);
    }
    if (repoVersion && repoVersion.get('isPatch')) {
      return Em.I18n.t('admin.stackUpgrade.dialog.upgrade.patch.header').format(upgradeType, upgradeVersion);
    }
    if (repoVersion && repoVersion.get('isMaint')) {
      return Em.I18n.t('admin.stackUpgrade.dialog.upgrade.maint.header').format(upgradeType, upgradeVersion);
    }
    return Em.I18n.t('admin.stackUpgrade.dialog.upgrade.header').format(upgradeType, upgradeVersion);
  },

  /**
   * methods through which cluster could be upgraded, "allowed" indicated if the method is allowed
   * by stack upgrade path
   * @type {Array}
   */
  upgradeMethods: [Em.Object.create({
    displayName: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.RU.title'),
    type: 'ROLLING',
    icon: "glyphicon glyphicon-dashboard",
    description: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.RU.description'),
    selected: false,
    allowed: true,
    isCheckComplete: false,
    isCheckRequestInProgress: false,
    precheckResultsMessage: '',
    precheckResultsTitle: '',
    action: '',
    isWizardRestricted: false
  }), Em.Object.create({
    displayName: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.EU.title'),
    type: 'NON_ROLLING',
    icon: "icon-bolt",
    description: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.EU.description'),
    selected: false,
    allowed: true,
    isCheckComplete: false,
    isCheckRequestInProgress: false,
    precheckResultsMessage: '',
    precheckResultsTitle: '',
    action: '',
    isWizardRestricted: false
  }), Em.Object.create({
    displayName: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.HOU.title'),
    type: 'HOST_ORDERED',
    icon: "icon-bolt",
    description: '',
    selected: false,
    allowed: false,
    isCheckComplete: false,
    isCheckRequestInProgress: false,
    precheckResultsMessage: '',
    precheckResultsTitle: '',
    action: '',
    isWizardRestricted: !App.supports.enabledWizardForHostOrderedUpgrade,
    cantBeStarted: true
  })],

  /**
   * pre-check messages map
   * @type {object}
   */
  preCheckMessages: {
    'WARNING': {
      template: 'admin.stackUpgrade.preCheck.warning.message',
      precheckResultsMessageClass: 'ORANGE',
      isPrecheckFailed: false,
      precheckResultsMessageIconClass: 'glyphicon glyphicon-warning-sign'
    },
    'BYPASS': {
      template: 'admin.stackUpgrade.preCheck.bypass.message',
      precheckResultsMessageClass: 'RED',
      isPrecheckFailed: false,
      precheckResultsMessageIconClass: 'glyphicon glyphicon-remove',
      bypassedFailures: true
    },
    'FAIL': {
      template: 'admin.stackUpgrade.preCheck.fail.message',
      precheckResultsMessageClass: 'RED',
      isPrecheckFailed: true,
      precheckResultsMessageIconClass: 'glyphicon glyphicon-remove'
    }
  },

  runningCheckRequests: [],

  /**
   * @type {boolean} true if some request that should disable actions is in progress
   */
  requestInProgress: false,

  /**
   * @type {number} repo id, request for which is currently in progress
   */
  requestInProgressRepoId: null,

  /**
   * @type {boolean} true while no updated upgrade info is loaded after retry
   */
  isRetryPending: false,

  /**
   * properties that stored to localStorage to resume wizard progress
   */
  wizardStorageProperties: ['fromVersion', 'upgradeId', 'upgradeVersion', 'toVersion', 'currentVersion', 'upgradeTypeDisplayName', 'upgradeType', 'failuresTolerance', 'isDowngrade', 'downgradeAllowed', 'isSuspended', 'isWizardRestricted'],

  /**
   * mutable properties of Upgrade Task
   * @type {Array}
   */
  taskDetailsProperties: ['status', 'stdout', 'stderr', 'error_log', 'host_name', 'output_log'],

  /**
   * Context for Finalize item
   * @type {string}
   */
  finalizeContext: 'Confirm Finalize',

  /**
   * Context for Slave component failures manual item
   * @type {string}
   */
  slaveFailuresContext: "Check Component Versions",

  /**
   * Context for Service check (may include slave component) failures manual item
   * @type {string}
   */
  serviceCheckFailuresContext: "Verifying Skipped Failures",

  /**
   * Check if current item is Finalize
   * @type {boolean}
   */
  isFinalizeItem: false,

  isLoadUpgradeDataPending: false,

  /**
   * path to the mock json
   * @type {String}
   */
  mockRepoUrl: '/data/stack_versions/repo_versions_all.json',

  /**
   * api to get RepoVersions
   * @type {String}
   */
  realRepoUrl: function () {
    return App.get('apiPrefix') + '/stacks?fields=versions/repository_versions/RepositoryVersions,' + 'versions/repository_versions/operating_systems/*,versions/repository_versions/operating_systems/repositories/*';
  }.property('App.stackVersionURL'),

  /**
   * path to the mock json
   * @type {String}
   */
  mockStackUrl: '/data/stack_versions/stack_version_all.json',

  /**
   * api to get ClusterStackVersions with repository_versions (use to init data load)
   * @type {String}
   */
  realStackUrl: function () {
    return App.get('apiPrefix') + '/clusters/' + App.get('clusterName') + '/stack_versions?fields=*,repository_versions/*,repository_versions/operating_systems/OperatingSystems/*,repository_versions/operating_systems/repositories/*';
  }.property('App.clusterName'),

  /**
   * api to get ClusterStackVersions without repository_versions (use to update data)
   * @type {String}
   */
  realUpdateUrl: function () {
    return App.get('apiPrefix') + '/clusters/' + App.get('clusterName') + '/stack_versions?fields=ClusterStackVersions/*';
  }.property('App.clusterName'),

  /**
   * Determines if list of services with checks that failed and were skipped by user during the upgrade is loaded
   * @type {boolean}
   */
  areSkippedServiceChecksLoaded: false,

  /**
   * List of services with checks that failed and were skipped by user during the upgrade
   * @type {array}
   */
  skippedServiceChecks: [],

  /**
   * status of tasks/items/groups which should be grayed out and disabled
   * @type {Array}
   */
  nonActiveStates: ['PENDING', 'ABORTED'],

  /**
   * status of Upgrade request
   * @type {string}
   */
  requestStatus: function () {
    if (this.get('upgradeData.Upgrade') && App.get('upgradeSuspended')) {
      return 'SUSPENDED';
    } else if (this.get('upgradeData.Upgrade')) {
      return this.get('upgradeData.Upgrade.request_status');
    } else {
      return 'INIT';
    }
  }.property('upgradeData.Upgrade.request_status', 'App.upgradeSuspended'),

  init: function init() {
    this.initDBProperties();
  },

  /**
   * restore data from localStorage
   */
  initDBProperties: function initDBProperties() {
    var props = this.getDBProperties(this.get('wizardStorageProperties'));
    Em.keys(props).forEach(function (k) {
      if (!Em.isNone(props[k])) {
        this.set(k, props[k]);
      }
    }, this);
  },

  /**
   * load all data:
   * - upgrade data
   * - stack versions
   * - repo versions
   */
  load: function load() {
    var dfd = $.Deferred();
    var self = this;

    this.loadStackVersionsToModel(true).done(function () {
      self.loadRepoVersionsToModel().done(function () {
        self.loadCompatibleVersions().done(function () {
          self.updateCurrentStackVersion();
          dfd.resolve();
        });
      });
    });
    return dfd.promise();
  },

  updateCurrentStackVersion: function updateCurrentStackVersion() {
    var currentVersion = App.StackVersion.find().findProperty('state', 'CURRENT');
    if (currentVersion) {
      this.set('currentVersion', {
        stack_name: currentVersion.get('repositoryVersion.stackVersionType'),
        repository_version: currentVersion.get('repositoryVersion.repositoryVersion'),
        repository_name: currentVersion.get('repositoryVersion.displayName'),
        id: currentVersion.get('repositoryVersion.id')
      });
    }
  },

  /**
   * load upgrade tasks by upgrade id
   * @return {$.Deferred}
   * @param {boolean} onlyState
   */
  loadUpgradeData: function loadUpgradeData(onlyState) {
    var upgradeId = this.get('upgradeId'),
        deferred = $.Deferred(),
        self = this;

    if (Em.isNone(upgradeId)) {
      deferred.resolve();
    } else {
      this.set('isLoadUpgradeDataPending', true);
      App.ajax.send({
        name: onlyState ? 'admin.upgrade.state' : 'admin.upgrade.data',
        sender: this,
        data: {
          id: upgradeId
        },
        success: 'loadUpgradeDataSuccessCallback'
      }).then(deferred.resolve).always(function () {
        self.set('isLoadUpgradeDataPending', false);
      });
    }
    return deferred.promise();
  },

  /**
   * parse and push upgrade tasks to controller
   * @param data
   */
  loadUpgradeDataSuccessCallback: function loadUpgradeDataSuccessCallback(data) {
    if (Em.isNone(data)) return;
    App.set('upgradeState', data.Upgrade.request_status);
    this.setDBProperty('upgradeState', data.Upgrade.request_status);
    this.set('isSuspended', data.Upgrade.suspended);
    this.setDBProperty('isSuspended', data.Upgrade.suspended);
    if (data.upgrade_groups) {
      this.updateUpgradeData(data);
    }
    if (this.get('isRetryPending') && data.Upgrade.request_status !== 'ABORTED') {
      this.setProperties({
        requestInProgress: false,
        isRetryPending: false
      });
    }
    if (data.Upgrade.request_status === 'COMPLETED') {
      this.finish();
    }
  },

  loadCompatibleVersions: function loadCompatibleVersions() {
    return App.ajax.send({
      name: 'admin.upgrade.get_compatible_versions',
      sender: this,
      data: {
        stackName: App.get('currentStackName'),
        stackVersion: App.get('currentStackVersionNumber')
      },
      success: 'loadCompatibleVersionsSuccessCallback'
    });
  },

  /**
   *
   * @param {object} data
   */
  loadCompatibleVersionsSuccessCallback: function loadCompatibleVersionsSuccessCallback(data) {
    App.RepositoryVersion.find().forEach(function (repo) {
      var version = repo.get('repositoryVersion');
      repo.set('isCompatible', data.items.someProperty('CompatibleRepositoryVersions.repository_version', version));
    });
  },

  /**
   * update data of Upgrade
   * @param {object} newData
   */
  updateUpgradeData: function updateUpgradeData(newData) {
    var oldData = this.get('upgradeData'),
        nonActiveStates = this.get('nonActiveStates'),
        groupsMap = {},
        itemsMap = {};

    if (Em.isNone(oldData) || newData.upgrade_groups.length !== oldData.upgradeGroups.length) {
      this.initUpgradeData(newData);
    } else {
      //create entities maps
      newData.upgrade_groups.forEach(function (newGroup) {
        groupsMap[newGroup.UpgradeGroup.group_id] = newGroup.UpgradeGroup;
        newGroup.upgrade_items.forEach(function (item) {
          itemsMap[item.UpgradeItem.stage_id] = item.UpgradeItem;
        });
      });

      //update existed entities with new data
      oldData.upgradeGroups.forEach(function (oldGroup) {
        oldGroup.set('status', groupsMap[oldGroup.get('group_id')].status);
        oldGroup.set('display_status', groupsMap[oldGroup.get('group_id')].display_status);
        oldGroup.set('progress_percent', groupsMap[oldGroup.get('group_id')].progress_percent);
        oldGroup.set('completed_task_count', groupsMap[oldGroup.get('group_id')].completed_task_count);
        oldGroup.upgradeItems.forEach(function (item) {
          item.set('status', itemsMap[item.get('stage_id')].status);
          item.set('display_status', itemsMap[item.get('stage_id')].display_status);
          item.set('progress_percent', itemsMap[item.get('stage_id')].progress_percent);
        });
        var hasExpandableItems = oldGroup.upgradeItems.some(function (item) {
          return !nonActiveStates.contains(item.get('status'));
        });
        oldGroup.set('hasExpandableItems', hasExpandableItems);
      });
      oldData.set('Upgrade', newData.Upgrade);
    }
  },

  /**
   * change structure of Upgrade
   * In order to maintain nested views in template object should have direct link to its properties, for example
   * item.UpgradeItem.<properties> -> item.<properties>
   * @param {object} newData
   */
  initUpgradeData: function initUpgradeData(newData) {
    var upgradeGroups = [],
        nonActiveStates = this.get('nonActiveStates');

    //wrap all entities into App.upgradeEntity
    newData.upgrade_groups.forEach(function (newGroup) {
      var hasExpandableItems = newGroup.upgrade_items.some(function (item) {
        return !nonActiveStates.contains(item.UpgradeItem.status);
      }),
          oldGroup = App.upgradeEntity.create({ type: 'GROUP', hasExpandableItems: hasExpandableItems }, newGroup.UpgradeGroup),
          upgradeItems = [];
      newGroup.upgrade_items.forEach(function (item) {
        var oldItem = App.upgradeEntity.create({ type: 'ITEM' }, item.UpgradeItem);
        this.formatMessages(oldItem);
        oldItem.set('tasks', []);
        upgradeItems.pushObject(oldItem);
      }, this);
      upgradeItems.reverse();
      oldGroup.set('upgradeItems', upgradeItems);
      upgradeGroups.pushObject(oldGroup);
    }, this);
    upgradeGroups.reverse();
    this.set('upgradeData', Em.Object.create({
      upgradeGroups: upgradeGroups,
      Upgrade: newData.Upgrade
    }));
    this.set('downgradeAllowed', newData.Upgrade.downgrade_allowed);
    this.setDBProperty('downgradeAllowed', newData.Upgrade.downgrade_allowed);
  },

  /**
   * format upgrade item text
   * @param {App.upgradeEntity} oldItem
   */
  formatMessages: function formatMessages(oldItem) {
    var text = oldItem.get('text');
    var messages = [];

    try {
      var messageArray = JSON.parse(text);
      for (var i = 0; i < messageArray.length; i++) {
        messages.push(messageArray[i].message);
      }
      oldItem.set('text', messages.join(' '));
    } catch (err) {
      console.warn('Upgrade Item has malformed text');
    }
    oldItem.set('messages', messages);
  },

  /**
   * request Upgrade Item and its tasks from server
   * @param {Em.Object} item
   * @param {Function} customCallback
   * @return {$.ajax}
   */
  getUpgradeItem: function getUpgradeItem(item, customCallback) {
    return App.ajax.send({
      name: 'admin.upgrade.upgrade_item',
      sender: this,
      data: {
        upgradeId: item.get('request_id'),
        groupId: item.get('group_id'),
        stageId: item.get('stage_id')
      },
      success: customCallback || 'getUpgradeItemSuccessCallback'
    });
  },

  /**
   * success callback of <code>getTasks</code>
   * @param {object} data
   */
  getUpgradeItemSuccessCallback: function getUpgradeItemSuccessCallback(data) {
    this.get('upgradeData.upgradeGroups').forEach(function (group) {
      if (group.get('group_id') === data.UpgradeItem.group_id) {
        group.get('upgradeItems').forEach(function (item) {
          if (item.get('stage_id') === data.UpgradeItem.stage_id) {
            if (item.get('tasks.length')) {
              data.tasks.forEach(function (task) {
                var currentTask = item.get('tasks').findProperty('id', task.Tasks.id);
                this.get('taskDetailsProperties').forEach(function (property) {
                  if (!Em.isNone(task.Tasks[property])) {
                    currentTask.set(property, task.Tasks[property]);
                  }
                }, this);
              }, this);
            } else {
              var tasks = [];
              data.tasks.forEach(function (task) {
                tasks.pushObject(App.upgradeEntity.create({
                  type: 'TASK',
                  group_id: data.UpgradeItem.group_id
                }, task.Tasks));
              });
              item.set('tasks', tasks);
            }
            item.set('isTasksLoaded', true);
          }
        }, this);
      }
    }, this);
  },

  /**
   * request Upgrade Task
   * @param {Em.Object} task
   * @return {$.ajax}
   */
  getUpgradeTask: function getUpgradeTask(task) {
    return App.ajax.send({
      name: 'admin.upgrade.upgrade_task',
      sender: this,
      data: {
        upgradeId: task.get('request_id'),
        groupId: task.get('group_id'),
        stageId: task.get('stage_id'),
        taskId: task.get('id'),
        task: task
      },
      success: 'getUpgradeTaskSuccessCallback'
    });
  },

  getUpgradeTaskSuccessCallback: function getUpgradeTaskSuccessCallback(data, xhr, params) {
    this.get('taskDetailsProperties').forEach(function (property) {
      params.task.set(property, data.Tasks[property]);
    }, this);
  },

  /**
   * Failures info may includes service_check and host_component failures. These two types should be displayed separately.
   */
  getServiceCheckItemSuccessCallback: function getServiceCheckItemSuccessCallback(data) {
    var task = data.tasks[0];
    var info = {
      hosts: [],
      host_detail: {}
    };

    if (task && task.Tasks && task.Tasks.structured_out && task.Tasks.structured_out.failures) {
      this.set('serviceCheckFailuresServicenames', task.Tasks.structured_out.failures.service_check || []);
      if (task.Tasks.structured_out.failures.host_component) {
        for (var hostname in task.Tasks.structured_out.failures.host_component) {
          info.hosts.push(hostname);
        }
        info.host_detail = task.Tasks.structured_out.failures.host_component;
      }
      this.set('slaveComponentStructuredInfo', info);
    }
  },

  getSlaveComponentItemSuccessCallback: function getSlaveComponentItemSuccessCallback(data) {
    var info = data.tasks[0];
    if (info && info.Tasks && info.Tasks.structured_out) {
      this.set('slaveComponentStructuredInfo', info.Tasks.structured_out);
    }
  },

  /**
   * downgrade confirmation popup
   * @param {object} event
   */
  confirmDowngrade: function confirmDowngrade(event) {
    var self = this;
    if (!this.get('currentVersion')) {
      this.updateCurrentStackVersion();
    }
    var currentVersion = this.get('currentVersion');
    return App.showConfirmationPopup(function () {
      self.downgrade.call(self, currentVersion, event);
    }, Em.I18n.t('admin.stackUpgrade.downgrade.body').format(currentVersion.repository_name), null, Em.I18n.t('admin.stackUpgrade.dialog.downgrade.header').format(this.get('upgradeVersion')), Em.I18n.t('admin.stackUpgrade.downgrade.proceed'));
  },

  /**
   * make call to start downgrade process
   * @param {object} currentVersion
   * @param {object} event
   */
  downgrade: function downgrade(currentVersion, event) {
    var self = this;
    this.set('requestInProgress', true);
    this.abortUpgrade().done(function () {
      var interval = setInterval(function () {
        if (self.get('upgradeData.Upgrade.request_status') == 'ABORTED') {
          clearInterval(interval);
          self.startDowngrade(currentVersion);
        }
      }, 1000);
    });
  },

  /**
   * abort upgrade (in order to start Downgrade)
   */
  abortUpgrade: function abortUpgrade() {
    var errorCallback = this.get('isDowngrade') ? 'abortDowngradeErrorCallback' : 'abortUpgradeErrorCallback';
    return App.ajax.send({
      name: 'admin.upgrade.abort',
      sender: this,
      data: {
        upgradeId: this.get('upgradeId'),
        isDowngrade: this.get('isDowngrade')
      },
      error: errorCallback
    });
  },

  /**
   * just request ro start downgrade,
   * should be performed only if <code>abortUpgrade<code> was completed
   */
  startDowngrade: function startDowngrade(currentVersion) {
    App.ajax.send({
      name: 'admin.downgrade.start',
      sender: this,
      data: {
        value: currentVersion.repository_version,
        label: this.get('upgradeVersion'),
        id: currentVersion.id,
        isDowngrade: true,
        upgradeType: this.get('upgradeType')
      },
      success: 'upgradeSuccessCallback',
      callback: function callback() {
        this.sender.set('requestInProgress', false);
      }
    });
  },

  /**
   * suspend upgrade (in order to restart it later)
   */
  abortUpgradeWithSuspend: function abortUpgradeWithSuspend() {
    var errorCallback = this.get('isDowngrade') ? 'abortDowngradeErrorCallback' : 'abortUpgradeErrorCallback';
    return App.ajax.send({
      name: 'admin.upgrade.suspend',
      sender: this,
      data: {
        upgradeId: this.get('upgradeId'),
        isDowngrade: this.get('isDowngrade')
      },
      error: errorCallback
    });
  },

  /**
   * error callback of <code>abortUpgrade()</code>
   * @param {object} data
   */
  abortUpgradeErrorCallback: function abortUpgradeErrorCallback(data) {
    var header = Em.I18n.t('admin.stackUpgrade.state.paused.fail.header');
    var body = Em.I18n.t('admin.stackUpgrade.state.paused.fail.body');
    if (data && data.responseText) {
      try {
        var json = $.parseJSON(data.responseText);
        body = body + ' ' + json.message;
      } catch (err) {}
    }
    App.showAlertPopup(header, body);
  },

  /**
   * error callback of <code>abortDowngrade()</code>
   * @param {object} data
   */
  abortDowngradeErrorCallback: function abortDowngradeErrorCallback(data) {
    var header = Em.I18n.t('admin.stackDowngrade.state.paused.fail.header');
    var body = Em.I18n.t('admin.stackDowngrade.state.paused.fail.body');
    if (data && data.responseText) {
      try {
        var json = $.parseJSON(data.responseText);
        body = body + ' ' + json.message;
      } catch (err) {}
    }
    App.showAlertPopup(header, body);
  },

  retryUpgrade: function retryUpgrade() {
    this.setProperties({
      requestInProgress: true,
      isRetryPending: true
    });
    return App.ajax.send({
      name: 'admin.upgrade.retry',
      sender: this,
      data: {
        upgradeId: this.get('upgradeId')
      }
    });
  },

  /**
   * make call to start upgrade process and show popup with current progress
   * @param {object} version
   */
  upgrade: function upgrade(version) {
    this.set('requestInProgress', true);
    App.ajax.send({
      name: 'admin.upgrade.start',
      sender: this,
      data: version,
      success: 'upgradeSuccessCallback',
      error: 'upgradeErrorCallback',
      callback: function callback() {
        this.sender.set('requestInProgress', false);
      }
    });
    this.setDBProperty('currentVersion', this.get('currentVersion'));

    // Show a "preparing the upgrade..." dialog in case the api call returns too slow
    if (App.router.get('currentState.name') != 'stackUpgrade') {
      this.showPreparingUpgradeIndicator();
    }
  },

  /**
   * Should progress bar be displayed when preparing upgrade,
   * should show after Upgrade Options window and before Upgrade Wizard
   * @method showPreparingUpgradeIndicator
   */
  showPreparingUpgradeIndicator: function showPreparingUpgradeIndicator() {
    return App.ModalPopup.show({
      header: '',
      showFooter: false,
      showCloseButton: false,
      bodyClass: Em.View.extend({
        templateName: require('templates/wizard/step8/step8_log_popup'),
        controllerBinding: 'App.router.mainAdminStackAndUpgradeController',

        /**
         * Css-property for progress-bar
         * @type {string}
         */
        barWidth: 'width: 100%;',
        progressBarClass: 'progress log_popup',

        /**
         * Popup-message
         * @type {string}
         */
        message: Em.I18n.t('admin.stackUpgrade.dialog.prepareUpgrade.header'),

        /**
         * Hide popup when upgrade wizard is open
         * @method autoHide
         */
        autoHide: function () {
          if (!this.get('controller.requestInProgress')) {
            this.get('parentView').hide();
          }
        }.observes('controller.requestInProgress')
      })
    });
  },

  /**
   * error callback of <code>upgrade()</code>
   * @param {object} data
   */
  upgradeErrorCallback: function upgradeErrorCallback(data) {
    var header = Em.I18n.t('admin.stackVersions.upgrade.start.fail.title');
    var body = "";
    if (data && data.responseText) {
      try {
        var json = $.parseJSON(data.responseText);
        body = json.message;
      } catch (err) {}
    }
    App.showAlertPopup(header, body);
  },

  /**
   * success callback of <code>upgrade()</code>
   * @param {object} data
   */
  upgradeSuccessCallback: function upgradeSuccessCallback(data, opt, params) {
    this.set('upgradeData', null);
    this.set('upgradeId', data.resources[0].Upgrade.request_id);
    this.set('toVersion', params.value);
    this.set('upgradeVersion', params.label);
    this.set('isDowngrade', !!params.isDowngrade);
    var upgradeMethod = this.get('upgradeMethods').findProperty('type', params.type),
        upgradeTypeDisplayName = null,
        upgradeType = null,
        isWizardRestricted = false;

    if (upgradeMethod) {
      upgradeTypeDisplayName = upgradeMethod.get('displayName');
      upgradeType = upgradeMethod.get('type');
      isWizardRestricted = upgradeMethod.get('isWizardRestricted');
    }

    this.set('isWizardRestricted', isWizardRestricted);
    this.set('upgradeType', upgradeType);
    this.set('upgradeTypeDisplayName', upgradeTypeDisplayName);
    this.set('failuresTolerance', Em.Object.create({
      skipComponentFailures: params.skipComponentFailures == 'true',
      skipSCFailures: params.skipSCFailures == 'true'
    }));
    this.setDBProperties({
      upgradeVersion: params.label,
      upgradeId: data.resources[0].Upgrade.request_id,
      toVersion: params.value,
      upgradeState: 'PENDING',
      isDowngrade: !!params.isDowngrade,
      upgradeType: upgradeType,
      isWizardRestricted: isWizardRestricted,
      upgradeTypeDisplayName: upgradeTypeDisplayName,
      failuresTolerance: Em.Object.create({
        skipComponentFailures: params.skipComponentFailures == 'true',
        skipSCFailures: params.skipSCFailures == 'true'
      })
    });
    App.set('upgradeState', 'PENDING');
    App.clusterStatus.setClusterStatus({
      wizardControllerName: this.get('name'),
      localdb: App.db.data
    });
    this.load();
    App.router.get('wizardWatcherController').setUser(App.router.get('mainAdminStackAndUpgradeController').get('name'));
    this.openUpgradeDialog();
  },

  /**
   * success callback of updating upgrade options including failures tolerance. etc
   * @param {object} data
   */
  updateOptionsSuccessCallback: function updateOptionsSuccessCallback(data, opt, params) {
    this.set('failuresTolerance', Em.Object.create({
      skipComponentFailures: params.skipComponentFailures == 'true',
      skipSCFailures: params.skipSCFailures == 'true'
    }));
  },

  /**
   * run upgrade checks and add results to each method object and set selected method
   * @param {Em.Object} version
   */
  runUpgradeMethodChecks: function runUpgradeMethodChecks(version) {
    this.get('upgradeMethods').forEach(function (method) {
      if (method.get('allowed')) {
        this.runPreUpgradeCheckOnly({
          id: version.get('id'),
          label: version.get('displayName'),
          type: method.get('type')
        });
      } else {
        //if method not supported in current stack version, mark as check completed
        method.setProperties({
          isCheckComplete: false,
          isCheckRequestInProgress: false,
          action: ''
        });
      }
    }, this);
  },

  getConfigsWarnings: function getConfigsWarnings(configsMergeWarning) {
    var configs = [];
    if (configsMergeWarning && Em.get(configsMergeWarning, 'UpgradeChecks.status') === 'WARNING') {
      var configsMergeCheckData = Em.get(configsMergeWarning, 'UpgradeChecks.failed_detail');
      if (configsMergeCheckData && Em.isArray(configsMergeCheckData)) {
        configs = configsMergeCheckData.reduce(function (allConfigs, item) {
          var isDeprecated = Em.isNone(item.new_stack_value),
              willBeRemoved = Em.isNone(item.result_value);

          return allConfigs.concat({
            type: item.type,
            name: item.property,
            wasModified: !isDeprecated && !willBeRemoved && Em.compare(item.current, item.result_value) === 0,
            currentValue: item.current,
            recommendedValue: isDeprecated ? Em.I18n.t('popup.clusterCheck.Upgrade.configsMerge.deprecated') : item.new_stack_value,
            isDeprecated: isDeprecated,
            resultingValue: willBeRemoved ? Em.I18n.t('popup.clusterCheck.Upgrade.configsMerge.willBeRemoved') : item.result_value,
            willBeRemoved: willBeRemoved
          });
        }, []);
      }
    }
    return configs;
  },

  showUpgradeOptions: function showUpgradeOptions(version) {
    this.upgradeOptions(false, version, true);
  },

  /**
   * Open upgrade options window: upgrade type and failures tolerance
   * @param {boolean} isInUpgradeWizard
   * @param {object} version
   * @return App.ModalPopup
   */
  upgradeOptions: function upgradeOptions(isInUpgradeWizard, version, preUpgradeShow) {
    var self = this,
        upgradeMethods = this.get('upgradeMethods'),
        runningCheckRequests = this.get('runningCheckRequests');
    if (!isInUpgradeWizard) {
      upgradeMethods.setEach('isCheckRequestInProgress', true);
      upgradeMethods.setEach('selected', false);
      var request = this.getSupportedUpgradeTypes(Ember.Object.create({
        stackName: App.get('currentStackVersion').split('-')[0],
        stackVersion: App.get('currentStackVersion').split('-')[1],
        toVersion: version.get('repositoryVersion')
      })).done(function () {
        if (App.get('router.currentState.name') === 'versions' && App.get('router.currentState.parentState.name') === 'stackAndUpgrade') {
          self.runUpgradeMethodChecks(version);
        }
      }).always(function () {
        self.set('isUpgradeTypesLoaded', true);
        self.set('runningCheckRequests', runningCheckRequests.rejectProperty('type', 'ALL'));
      });
      request.type = 'ALL';
      this.get('runningCheckRequests').push(request);
    } else {
      this.set('isUpgradeTypesLoaded', true);
    }

    return App.ModalPopup.show({
      encodeBody: false,
      primary: function () {
        if (preUpgradeShow) return false;
        if (isInUpgradeWizard || this.get('controller.getSupportedUpgradeError')) return Em.I18n.t('ok');
        return Em.I18n.t('common.proceed');
      }.property('controller.getSupportedUpgradeError'),
      secondary: function () {
        if (preUpgradeShow) return Em.I18n.t('common.dismiss');
        if (this.get('controller.getSupportedUpgradeError')) return null;
        return Em.I18n.t('common.cancel');
      }.property('controller.getSupportedUpgradeError'),
      secondaryClass: preUpgradeShow ? 'btn-success' : '',
      classNames: ['upgrade-options-popup'],
      header: preUpgradeShow ? Em.I18n.t('admin.stackVersions.version.preUpgrade.header') : Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.header'),
      controller: this,
      showFooter: function () {
        return this.get('controller.isUpgradeTypesLoaded') || preUpgradeShow;
      }.property('controller.isUpgradeTypesLoaded'),
      bodyClass: Em.View.extend({
        templateName: require('templates/main/admin/stack_upgrade/upgrade_options'),
        didInsertElement: function didInsertElement() {
          App.tooltip($(".failure-tolerance-tooltip"), {
            placement: "top",
            title: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.tolerance.tooltip')
          });
          Em.run.later(this, function () {
            App.tooltip($(".img-thumbnail.check-failed"), {
              placement: "bottom",
              title: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.preCheck.failed.tooltip')
            });
            App.tooltip($(".not-allowed-by-version"), {
              placement: "bottom",
              title: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.notAllowed')
            });
          }, 1000);
        },
        upgradeMethods: function () {
          self.updateSelectedMethod(isInUpgradeWizard);
          return self.get('upgradeMethods');
        }.property().volatile(),
        isInUpgradeWizard: isInUpgradeWizard,
        showPreUpgradeChecks: App.get('supports.preUpgradeCheck') && !isInUpgradeWizard,
        versionText: function () {
          if (preUpgradeShow) return Em.I18n.t('admin.stackVersions.version.preUpgrade.bodyMsg.version').format(version.get('displayName'));
          if (isInUpgradeWizard) return '';
          return Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.bodyMsg.version').format(version.get('displayName'));
        }(),
        selectMethod: function selectMethod(event) {
          if (isInUpgradeWizard || !event.context.get('allowed') || event.context.get('isPrecheckFailed')) return;
          var selectedMethod = event.context;
          self.updateSelectedMethod(isInUpgradeWizard);
          self.get('upgradeMethods').forEach(function (method) {
            method.set('selected', false);
          });
          selectedMethod.set('selected', true);
          this.set('parentView.selectedMethod', selectedMethod);
        },
        runAction: function runAction(event) {
          var method = event.context,
              action = method.get('action');
          if (action) {
            this.get(action)(event);
          }
        },
        rerunCheck: function rerunCheck(event) {
          self.runPreUpgradeCheckOnly({
            id: version.get('id'),
            label: version.get('displayName'),
            type: event.context.get('type')
          });
        },
        openMessage: function openMessage(event) {
          if (isInUpgradeWizard || !event.context.get('allowed')) return;
          var data = event.context.get('precheckResultsData');

          var failTitle = Em.I18n.t('popup.clusterCheck.Upgrade.fail.title');
          var failAlert = new Em.Handlebars.SafeString(Em.I18n.t('popup.clusterCheck.Upgrade.fail.alert'));
          var bypassedFailures = data.items.filterProperty('UpgradeChecks.status', 'BYPASS').length > 0;
          if (data.items.filterProperty('UpgradeChecks.status', 'ERROR').length == 0 && bypassedFailures) {
            failTitle = Em.I18n.t('popup.clusterCheck.Upgrade.bypassed-failures.title');
            failAlert = new Em.Handlebars.SafeString(Em.I18n.t('popup.clusterCheck.Upgrade.bypassed-failures.alert'));
          }

          var header = Em.I18n.t('popup.clusterCheck.Upgrade.header').format(version.get('displayName')),
              warningTitle = Em.I18n.t('popup.clusterCheck.Upgrade.warning.title'),
              warningAlert = new Em.Handlebars.SafeString(Em.I18n.t('popup.clusterCheck.Upgrade.warning.alert')),
              configsMergeWarning = data.items.findProperty('UpgradeChecks.id', "CONFIG_MERGE"),
              popupData = {
            items: data.items.rejectProperty('UpgradeChecks.id', 'CONFIG_MERGE')
          },
              configs = self.getConfigsWarnings(configsMergeWarning);
          App.showClusterCheckPopup(popupData, {
            header: header,
            failTitle: failTitle,
            failAlert: failAlert,
            warningTitle: warningTitle,
            warningAlert: warningAlert,
            primary: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.preCheck.rerun'),
            secondary: Em.I18n.t('common.cancel'),
            bypassedFailures: bypassedFailures,
            callback: function callback() {
              self.runPreUpgradeCheckOnly.call(self, {
                id: version.get('id'),
                label: version.get('displayName'),
                type: event.context.get('type')
              });
            }
          }, configs);
        },
        upgradeShow: !preUpgradeShow
      }),

      /**
       * @type {Em.Object}
       * @default null
       */
      selectedMethod: null,
      skipComponentFailures: self.get('failuresTolerance.skipComponentFailures'),
      skipSCFailures: self.get('failuresTolerance.skipSCFailures'),
      disablePrimary: function () {
        if (isInUpgradeWizard || this.get('controller.getSupportedUpgradeError')) return false;
        var selectedMethod = this.get('selectedMethod');
        if (selectedMethod) {
          if (App.get('supports.preUpgradeCheck')) {
            return selectedMethod.get('isPrecheckFailed') || selectedMethod.get('isCheckRequestInProgress');
          } else {
            return false;
          }
        } else {
          return true;
        }
      }.property('selectedMethod', 'selectedMethod.isPrecheckFailed', 'selectedMethod.isCheckRequestInProgress'),
      onPrimary: function onPrimary() {
        this.hide();
        if (isInUpgradeWizard) {
          return App.ajax.send({
            name: 'admin.upgrade.update.options',
            sender: self,
            data: {
              upgradeId: self.get('upgradeId'),
              skipComponentFailures: Boolean(this.get('skipComponentFailures')).toString(),
              skipSCFailures: Boolean(this.get('skipSCFailures')).toString()
            },
            success: 'updateOptionsSuccessCallback'
          });
        } else {
          var upgradeMethod = self.get('upgradeMethods').findProperty('selected');
          version.upgradeType = upgradeMethod.get('type');
          version.upgradeTypeDisplayName = upgradeMethod.get('displayName');
          version.skipComponentFailures = this.get('skipComponentFailures');
          version.skipSCFailures = this.get('skipSCFailures');

          var toVersion = version.get('displayName');
          var bodyMessage = Em.Object.create({
            confirmButton: Em.I18n.t('yes'),
            confirmMsg: upgradeMethod.get('type') === 'ROLLING' ? Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.RU.confirm.msg').format(toVersion) : Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.EU.confirm.msg').format(toVersion)
          });
          return App.showConfirmationFeedBackPopup(function (query) {
            return self.runPreUpgradeCheck.call(self, version);
          }, bodyMessage);
        }
      }
    });
  },

  /**
   * open upgrade options from upgrade wizard
   */
  openUpgradeOptions: function openUpgradeOptions() {
    if (this.get('isDowngrade')) return;
    this.upgradeOptions(true, null);
  },

  /**
   * upgrade confirmation popup including upgrade options: upgrade type and failures tolerance
   * @param {object} version
   * @return App.ModalPopup
   */
  confirmUpgrade: function confirmUpgrade(version) {
    this.upgradeOptions(false, version);
  },

  /**
   * send request for pre upgrade check only
   */
  runPreUpgradeCheckOnly: function runPreUpgradeCheckOnly(data) {
    if (App.get('supports.preUpgradeCheck')) {
      var method = this.get('upgradeMethods').findProperty('type', data.type);
      method.setProperties({
        isCheckComplete: false,
        isCheckRequestInProgress: true,
        action: ''
      });
      var request = App.ajax.send({
        name: "admin.upgrade.pre_upgrade_check",
        sender: this,
        data: data,
        success: 'runPreUpgradeCheckOnlySuccess',
        error: 'runPreUpgradeCheckOnlyError',
        callback: function callback() {
          var runningCheckRequests = this.sender.get('runningCheckRequests');
          method.set('isCheckRequestInProgress', false);
          this.sender.set('runningCheckRequests', runningCheckRequests.rejectProperty('type', this.data.type));
        }
      });
      request.type = data.type;
      this.get('runningCheckRequests').push(request);
    }
  },

  /**
   * send request to get available upgrade tye names
   */
  getSupportedUpgradeTypes: function getSupportedUpgradeTypes(data) {
    this.set('isUpgradeTypesLoaded', false);
    this.set('getSupportedUpgradeError', '');
    return App.ajax.send({
      name: "admin.upgrade.get_supported_upgradeTypes",
      sender: this,
      data: data,
      success: "getSupportedUpgradeTypesSuccess",
      error: "getSupportedUpgradeTypesError"
    });
  },

  /**
   * success callback of <code>getSupportedUpgradeTypes()</code>
   * @param data {object}
   */
  getSupportedUpgradeTypesSuccess: function getSupportedUpgradeTypesSuccess(data) {
    var supportedUpgradeTypes = data.items[0] && data.items[0].CompatibleRepositoryVersions.upgrade_types;
    this.get('upgradeMethods').forEach(function (method) {
      method.set('allowed', Boolean(supportedUpgradeTypes && supportedUpgradeTypes.contains(method.get('type'))));
    });
  },

  /**
   * error callback of <code>getSupportedUpgradeTypes()</code>
   * @param xhr {object}
   */
  getSupportedUpgradeTypesError: function getSupportedUpgradeTypesError(xhr) {
    var response;
    try {
      response = JSON.parse(xhr.responseText);
    } catch (e) {
      response = { message: xhr.statusText };
    }
    this.set('getSupportedUpgradeError', response.message);
  },

  /**
   * success callback of <code>runPreUpgradeCheckOnly()</code>
   * Show a message how many fails/warnings/bypass/passed
   * on clicking that message a popup window show up
   * @param data {object}
   * @param opt {object}
   * @param params {object}
   */
  runPreUpgradeCheckOnlySuccess: function runPreUpgradeCheckOnlySuccess(data, opt, params) {
    var properties = {
      precheckResultsTitle: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.preCheck.msg.title'),
      precheckResultsData: data,
      isCheckComplete: true,
      action: 'openMessage',
      precheckResultsMessage: '',
      recheckResultsMessageClass: 'GREEN',
      isPrecheckFailed: false,
      precheckResultsMessageIconClass: 'glyphicon glyphicon-ok',
      bypassedFailures: false
    };

    Object.keys(this.get('preCheckMessages')).forEach(function (status) {
      if (data.items.someProperty('UpgradeChecks.status', status)) {
        properties = this.formatPreCheckMessage(status, data, properties);
      }
    }, this);

    if (!properties.precheckResultsMessage) {
      properties.precheckResultsMessage = Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.preCheck.allPassed');
    }
    this.get('upgradeMethods').findProperty('type', params.type).setProperties(properties);
    this.updateSelectedMethod(false);
    this.addPrecheckMessageTooltip();
  },

  /**
   * @method formatPreCheckMessage
   * @param {string} type
   * @param {object} data
   * @param {object} defaults
   * @returns {object}
   */
  formatPreCheckMessage: function formatPreCheckMessage(type, data, defaults) {
    var length = data.items.filterProperty('UpgradeChecks.status', type).length;
    var properties = this.get('preCheckMessages')[type] || {};
    var message = Em.I18n.t(properties.template).format(length, defaults.precheckResultsMessage);
    defaults = $.extend(defaults, properties);
    delete defaults.template;
    defaults.precheckResultsMessage = message;
    return defaults;
  },

  addPrecheckMessageTooltip: function addPrecheckMessageTooltip() {
    Em.run.later(this, function () {
      // add tooltip for the type with preCheck errors
      App.tooltip($(".img-thumbnail.check-failed"), {
        placement: "bottom",
        title: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.preCheck.failed.tooltip')
      });
      // destroy the tooltip for the type wo preCheck errors
      $(".img-thumbnail").not(".check-failed").not(".not-allowed-by-version").tooltip("destroy");
    }, 1000);
  },

  runPreUpgradeCheckOnlyError: function runPreUpgradeCheckOnlyError(request, ajaxOptions, error, data, params) {
    var method = this.get('upgradeMethods').findProperty('type', params.type);
    method.setProperties({
      precheckResultsMessage: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.preCheck.msg.failed.link'),
      precheckResultsTitle: Em.I18n.t('admin.stackVersions.version.upgrade.upgradeOptions.preCheck.msg.failed.title'),
      precheckResultsMessageClass: 'RED',
      isPrecheckFailed: true,
      precheckResultsMessageIconClass: 'glyphicon glyphicon-warning-sign',
      action: 'rerunCheck'
    });
  },

  /**
   * In Upgrade Wizard: update which method already been selected on open
   * Not in upgrade wizard: de-select the method with pre-check errors
   * @param isInUpgradeWizard {boolean}
   */
  updateSelectedMethod: function updateSelectedMethod(isInUpgradeWizard) {
    if (isInUpgradeWizard) {
      this.get('upgradeMethods').forEach(function (method) {
        method.set('selected', method.get('type') === this.get('upgradeType'));
      }, this);
    } else {
      var ruMethod = this.get('upgradeMethods').findProperty('type', 'ROLLING');
      var euMethod = this.get('upgradeMethods').findProperty('type', 'NON_ROLLING');
      if (ruMethod && ruMethod.get('isPrecheckFailed')) ruMethod.set('selected', false);
      if (euMethod && euMethod.get('isPrecheckFailed')) euMethod.set('selected', false);
    }
  },

  /**
   * send request for pre upgrade check
   * @param version
   */
  runPreUpgradeCheck: function runPreUpgradeCheck(version) {
    var params = {
      value: version.get('repositoryVersion'),
      label: version.get('displayName'),
      type: version.get('upgradeType'),
      skipComponentFailures: version.get('skipComponentFailures') ? 'true' : 'false',
      skipSCFailures: version.get('skipSCFailures') ? 'true' : 'false',
      id: version.get('id'),
      targetStack: version.get('displayName')
    };
    if (App.get('supports.preUpgradeCheck')) {
      this.set('requestInProgress', true);
      App.ajax.send({
        name: "admin.upgrade.pre_upgrade_check",
        sender: this,
        data: params,
        success: "runPreUpgradeCheckSuccess",
        error: "runPreUpgradeCheckError"
      });
    } else {
      this.upgrade(params);
    }
  },

  /**
   * success callback of <code>runPreUpgradeCheckSuccess()</code>
   * if there are some fails - it shows popup else run upgrade
   * @param data {object}
   * @param opt {object}
   * @param params {object}
   * @returns {App.ModalPopup|undefined}
   */
  runPreUpgradeCheckSuccess: function runPreUpgradeCheckSuccess(data, opt, params) {
    var self = this;
    if (data.items.someProperty('UpgradeChecks.status', 'FAIL') || data.items.someProperty('UpgradeChecks.status', 'WARNING') || data.items.someProperty('UpgradeChecks.status', 'BYPASS')) {
      this.set('requestInProgress', false);
      var hasFails = data.items.someProperty('UpgradeChecks.status', 'FAIL'),
          header = Em.I18n.t('popup.clusterCheck.Upgrade.header').format(params.label),
          failTitle = Em.I18n.t('popup.clusterCheck.Upgrade.fail.title'),
          failAlert = new Em.Handlebars.SafeString(Em.I18n.t('popup.clusterCheck.Upgrade.fail.alert')),
          warningTitle = Em.I18n.t('popup.clusterCheck.Upgrade.warning.title'),
          warningAlert = new Em.Handlebars.SafeString(Em.I18n.t('popup.clusterCheck.Upgrade.warning.alert')),
          bypassedFailures = data.items.someProperty('UpgradeChecks.status', 'BYPASS').length > 0,
          configsMergeWarning = data.items.findProperty('UpgradeChecks.id', 'CONFIG_MERGE'),
          popupData = {
        items: data.items.rejectProperty('UpgradeChecks.id', 'CONFIG_MERGE')
      },
          configs = this.getConfigsWarnings(configsMergeWarning);
      App.showClusterCheckPopup(popupData, {
        header: header,
        failTitle: failTitle,
        failAlert: failAlert,
        warningTitle: warningTitle,
        warningAlert: warningAlert,
        bypassedFailures: bypassedFailures,
        noCallbackCondition: hasFails,
        callback: function callback() {
          self.upgrade(params);
        }
      }, configs, params.label);
    } else {
      this.upgrade(params);
    }
  },

  runPreUpgradeCheckError: function runPreUpgradeCheckError() {
    this.set('requestInProgress', false);
  },

  confirmRetryUpgrade: function confirmRetryUpgrade(version) {
    var self = this;
    return App.showConfirmationPopup(function () {
      self.retryUpgrade();
    }, Em.I18n.t('admin.stackUpgrade.upgrade.retry.confirm.body').format(version.get('displayName')), null, this.getUpgradeDowngradeHeader(version.get('upgradeTypeDislayName'), version.get('displayName')));
  },

  confirmRetryDowngrade: function confirmRetryDowngrade() {
    var self = this,
        currentVersion = this.get('currentVersion');
    return App.showConfirmationPopup(function () {
      self.retryUpgrade();
    }, Em.I18n.t('admin.stackUpgrade.downgrade.retry.body').format(currentVersion.repository_name), null, Em.I18n.t('admin.stackUpgrade.dialog.downgrade.header').format(currentVersion.repository_name), Em.I18n.t('admin.stackUpgrade.downgrade.proceed'));
  },

  /**
   * confirmation popup before install repository version
   */
  installRepoVersionPopup: function installRepoVersionPopup(repo) {
    var availableServices = repo.get('stackServices').filter(function (service) {
      return App.Service.find(service.get('name')).get('isLoaded') && service.get('isAvailable') && service.get('isUpgradable');
    }, this);
    if (!availableServices.length && !repo.get('isStandard')) {
      return App.showAlertPopup(Em.I18n.t('admin.stackVersions.upgrade.installPackage.fail.title'), Em.I18n.t('admin.stackVersions.upgrade.installPackage.fail.noAvailableServices').format(repo.get('displayName')));
    }
    var self = this;
    var bodyContent = repo.get('isPatch') ? Em.I18n.t('admin.stackVersions.version.install.patch.confirm') : Em.I18n.t('admin.stackVersions.version.install.confirm');
    return App.ModalPopup.show({
      header: Em.I18n.t('popup.confirmation.commonHeader'),
      popupBody: bodyContent.format(repo.get('displayName')),
      skipDependencyCheck: false,
      bodyClass: Em.View.extend({
        classNames: ['install-repo-confirmation'],
        content: availableServices,
        showAvailableServices: repo.get('isPatch') || repo.get('isMaint'),
        templateName: require('templates/common/modal_popups/install_repo_confirmation')
      }),
      onPrimary: function onPrimary() {
        self.installRepoVersion(repo);
        this._super();
      }
    });
  },

  /**
   * sends request to install repoVersion to the cluster
   * and create clusterStackVersion resourse
   * @param {Em.Object} repo
   * @return {$.ajax}
   * @method installRepoVersion
   */
  installRepoVersion: function installRepoVersion(repo) {
    this.set('requestInProgress', true);
    this.set('requestInProgressRepoId', repo.get('id'));

    var data = {
      ClusterStackVersions: {
        stack: repo.get('stackVersionType'),
        version: repo.get('stackVersionNumber'),
        repository_version: repo.get('repositoryVersion')
      },
      id: repo.get('id')
    };
    return App.ajax.send({
      name: 'admin.stack_version.install.repo_version',
      sender: this,
      data: data,
      success: 'installRepoVersionSuccess',
      error: 'installRepoVersionError',
      callback: function callback() {
        this.sender.set('requestInProgress', false);
        this.sender.set('requestInProgressRepoId', null);
      }
    });
  },

  /**
   * TODO unify and move modal view into common/modal_popups
   * @param {string[]} hosts
   * @param {string} title
   * @param {string} operation
   * @param {Function} primary
   */
  showReinstallRemoveConfirmation: function showReinstallRemoveConfirmation(_ref) {
    var hosts = _ref.hosts,
        title = _ref.title,
        operation = _ref.operation,
        _ref$primary = _ref.primary,
        primary = _ref$primary === undefined ? function () {} : _ref$primary;

    return App.ModalPopup.show({
      header: title,
      visibleHosts: hosts.join("\n"),
      expanded: true,
      onPrimary: function onPrimary() {
        primary(hosts);
        this._super();
      },

      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/bulk_operation_confirm_popup'),
        message: Em.I18n.t('hosts.bulkOperation.confirmation.hosts').format(operation, hosts.length),
        textareaVisible: false,
        textTrigger: function textTrigger() {
          this.toggleProperty('textareaVisible');
        },
        putHostNamesToTextarea: function () {
          var hostNames = this.get('parentView.visibleHosts');
          if (this.get('textareaVisible')) {
            var wrapper = $(".task-detail-log-maintext");
            $('.task-detail-log-clipboard').html(hostNames).width(wrapper.width()).height(250);
            Em.run.next(function () {
              $('.task-detail-log-clipboard').select();
            });
          }
        }.observes('textareaVisible')
      })
    });
  },

  removeOutOfSyncComponents: function removeOutOfSyncComponents(event) {
    var _this = this;

    var hosts = App.RepositoryVersion.find(event.context.repoId).get('stackVersion.outOfSyncHosts');
    return this.showReinstallRemoveConfirmation({
      hosts: hosts,
      title: Em.I18n.t('admin.stackVersions.version.errors.outOfSync.remove.title'),
      operation: Em.I18n.t('hosts.host.maintainance.removeFailedComponents.context'),
      primary: function primary() {
        App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
          App.ajax.send({
            name: 'host.host_component.delete_components',
            sender: _this,
            data: {
              hosts: hosts,
              data: JSON.stringify({
                RequestInfo: {
                  query: 'HostRoles/host_name.in(' + hosts.join(',') + ')&HostRoles/state=INSTALL_FAILED'
                }
              })
            }
          });
        });
      }
    });
  },

  reinstallOutOfSyncComponents: function reinstallOutOfSyncComponents(event) {
    var _this2 = this;

    var hosts = App.RepositoryVersion.find(event.context.repoId).get('stackVersion.outOfSyncHosts');
    return this.showReinstallRemoveConfirmation({
      hosts: hosts,
      title: Em.I18n.t('admin.stackVersions.version.errors.outOfSync.reinstall.title'),
      operation: Em.I18n.t('hosts.host.maintainance.reinstallFailedComponents.context'),
      primary: function primary() {
        App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
          App.ajax.send({
            name: 'common.host_components.update',
            sender: _this2,
            data: {
              HostRoles: {
                state: 'INSTALLED'
              },
              query: 'HostRoles/host_name.in(' + hosts.join(',') + ')&HostRoles/state=INSTALL_FAILED',
              context: Em.I18n.t('hosts.host.maintainance.reinstallFailedComponents.context')
            },
            success: 'reinstallOutOfSyncComponentsSuccessCallback'
          });
        });
      }
    });
  },

  reinstallOutOfSyncComponentsSuccessCallback: function reinstallOutOfSyncComponentsSuccessCallback(data, opt, params, req) {
    if (!data && req.status == 200) {
      return App.ModalPopup.show({
        header: Em.I18n.t('rolling.nothingToDo.header'),
        body: Em.I18n.t('rolling.nothingToDo.body').format(params.noOpsMessage || Em.I18n.t('hosts.host.maintainance.allComponents.context')),
        secondary: false
      });
    }
    return App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
      if (initValue) {
        App.router.get('backgroundOperationsController').showPopup();
      }
    });
  },

  /**
   * transform repo data into json for
   * saving changes to repository version
   * @param {Em.Object} repo
   * @returns {{operating_systems: Array}}
   */
  prepareRepoForSaving: function prepareRepoForSaving(repo) {
    var repoVersion = { "operating_systems": [] };
    var ambari_managed_repositories = !repo.get('useRedhatSatellite');
    repo.get('operatingSystems').forEach(function (os, k) {
      repoVersion.operating_systems.push({
        "OperatingSystems": {
          "os_type": os.get("osType"),
          "ambari_managed_repositories": ambari_managed_repositories
        },
        "repositories": []
      });
      os.get('repositories').forEach(function (repository) {
        repoVersion.operating_systems[k].repositories.push({
          "Repositories": {
            "base_url": repository.get('baseUrl'),
            "repo_id": repository.get('repoId'),
            "repo_name": repository.get('repoName')
          }
        });
      });
    });
    return repoVersion;
  },

  /**
   * Return stack version for the repo object
   * @param {Em.Object} repo
   * */
  getStackVersionNumber: function getStackVersionNumber(repo) {
    var stackVersionNumber = repo.get('stackVersion');
    if (null == stackVersionNumber) stackVersionNumber = App.get('currentStackVersion');
    return stackVersionNumber;
  },

  /**
   * perform validation if <code>skip<code> is  false and run save if
   * validation successfull or run save without validation is <code>skip<code> is true
   * @param {Em.Object} repo
   * @param {boolean} skip
   * @returns {$.Deferred}
   */
  saveRepoOS: function saveRepoOS(repo, skip) {
    var self = this;
    var deferred = $.Deferred();
    this.validateRepoVersions(repo, skip).done(function (data) {
      if (data.length > 0) {
        deferred.resolve(data);
      } else {
        var repoVersion = self.prepareRepoForSaving(repo);
        var stackVersionNumber = self.getStackVersionNumber(repo);

        App.ajax.send({
          name: 'admin.stack_versions.edit.repo',
          sender: this,
          data: {
            stackName: App.get('currentStackName'),
            stackVersion: stackVersionNumber,
            repoVersionId: repo.get('repoVersionId'),
            repoVersion: repoVersion
          }
        }).success(function () {
          deferred.resolve([]);
        });
      }
    });
    return deferred.promise();
  },

  /**
   * send request for validation for each repository
   * @param {Em.Object} repo
   * @param {boolean} skip
   * @returns {*}
   */
  validateRepoVersions: function validateRepoVersions(repo, skip) {
    var deferred = $.Deferred(),
        totalCalls = 0,
        invalidUrls = [];

    if (skip) {
      deferred.resolve(invalidUrls);
    } else {
      var stackVersionNumber = this.getStackVersionNumber(repo);
      repo.get('operatingSystems').forEach(function (os) {
        if (os.get('isSelected')) {
          os.get('repositories').forEach(function (repo) {
            totalCalls++;
            this.validationCall(repo, os, stackVersionNumber).success(function () {
              totalCalls--;
              if (totalCalls === 0) deferred.resolve(invalidUrls);
            }).error(function () {
              repo.set('hasError', true);
              invalidUrls.push(repo);
              totalCalls--;
              if (totalCalls === 0) deferred.resolve(invalidUrls);
            });
          }, this);
        } else {
          return deferred.resolve(invalidUrls);
        }
      }, this);
    }
    return deferred.promise();
  },

  /**
   *
   * @param {Em.Object} repo
   * @param {Em.Object} os
   * @param {string} stackVersionNumber
   */
  validationCall: function validationCall(repo, os, stackVersionNumber) {
    return App.ajax.send({
      name: 'admin.stack_versions.validate.repo',
      sender: this,
      data: {
        repo: repo,
        repoName: repo.get('repoName'),
        repoId: repo.get('repoId'),
        baseUrl: repo.get('baseUrl'),
        osType: os.get('osType'),
        stackName: App.get('currentStackName'),
        stackVersion: stackVersionNumber
      }
    });
  },

  /**
   * success callback for <code>installRepoVersion()<code>
   * saves request id to the db
   * @param data
   * @param opt
   * @param params
   * @method installStackVersionSuccess
   */
  installRepoVersionSuccess: function installRepoVersionSuccess(data, opt, params) {
    if (data && data.statusText === "timeout") {
      App.showAlertPopup(Em.I18n.t('admin.stackVersions.upgrade.installPackage.fail.title'), Em.I18n.t('admin.stackVersions.upgrade.installPackage.fail.timeout'));
      return false;
    }
    var version = App.RepositoryVersion.find(params.id);
    App.db.set('repoVersionInstall', 'id', [data.Requests.id]);
    App.clusterStatus.setClusterStatus({
      wizardControllerName: this.get('name'),
      localdb: App.db.data
    });
    version.set('defaultStatus', 'INSTALLING');
    if (version.get('stackVersion')) {
      version.set('stackVersion.state', 'INSTALLING');
    }
  },

  /**
   * error callback for <code>installRepoVersion()<code>
   * show the error message
   * @param data
   * @method installStackVersionSuccess
   */
  installRepoVersionError: function installRepoVersionError(data, opt, params) {
    var header = Em.I18n.t('admin.stackVersions.upgrade.installPackage.fail.title');
    var body = "";
    if (data && data.responseText) {
      try {
        var json = $.parseJSON(data.responseText);
        body = json.message;
      } catch (err) {}
    }
    if (data && data.statusText === "timeout") {
      body = Em.I18n.t('admin.stackVersions.upgrade.installPackage.fail.timeout');
    }
    var version = App.RepositoryVersion.find(params.id);
    version.set('defaultStatus', 'INSTALL_FAILED');
    if (version.get('stackVersion')) {
      version.set('stackVersion.state', 'INSTALL_FAILED');
    }
    App.showAlertPopup(header, body);
  },

  /**
   * opens a popup with installations state per host
   * @param {Em.Object} version
   * @method showProgressPopup
   */
  showProgressPopup: function showProgressPopup(version) {
    var popupTitle = Em.I18n.t('admin.stackVersions.details.install.hosts.popup.title').format(version.get('displayName'));
    var requestIds = this.getRepoVersionInstallId();
    var hostProgressPopupController = App.router.get('highAvailabilityProgressPopupController');
    hostProgressPopupController.initPopup(popupTitle, requestIds, this);
  },

  getRepoVersionInstallId: function getRepoVersionInstallId() {
    if (App.get('testMode')) return [1];

    var requestIds = App.db.get('repoVersionInstall', 'id');
    var lastRepoVersionInstall = App.router.get('backgroundOperationsController.services').find(function (request) {
      return request.get('name').startsWith('Install version');
    });
    if (lastRepoVersionInstall && (!requestIds || !requestIds.contains(lastRepoVersionInstall.get('id')))) {
      requestIds = [lastRepoVersionInstall.get('id')];
    }
    return requestIds || [];
  },

  /**
   * reset upgradeState to NOT_REQUIRED when upgrade is COMPLETED
   * and clean auxiliary data
   */
  finish: function finish() {
    var upgradeVersion = this.get('upgradeVersion') && this.get('upgradeVersion').match(/[a-zA-Z]+\-\d+\.\d+/);
    this.setDBProperties({
      fromVersion: undefined,
      upgradeId: undefined,
      upgradeState: 'NOT_REQUIRED',
      upgradeVersion: undefined,
      currentVersion: undefined,
      upgradeTypeDisplayName: undefined,
      upgradeType: undefined,
      isWizardRestricted: false,
      failuresTolerance: undefined,
      isDowngrade: undefined,
      downgradeAllowed: undefined
    });
    this.initDBProperties();
    App.clusterStatus.setClusterStatus({
      localdb: App.db.data
    });
    if (upgradeVersion && upgradeVersion[0]) {
      App.set('currentStackVersion', upgradeVersion[0]);
    }
    App.set('upgradeState', 'NOT_REQUIRED');
  },

  /**
   * Check <code>App.upgradeState</code> for HOLDING
   * If it is, send request to check if current item is Finalize
   * @method updateFinalize
   */
  updateFinalize: function () {
    var upgradeState = App.get('upgradeState');
    if (upgradeState === 'HOLDING') {
      return App.ajax.send({
        name: 'admin.upgrade.finalizeContext',
        sender: this,
        success: 'updateFinalizeSuccessCallback',
        error: 'updateFinalizeErrorCallback'
      });
    } else {
      this.set('isFinalizeItem', false);
    }
  }.observes('App.upgradeState'),

  /**
   *
   * @param {object|null} data
   * @method updateFinalizeSuccessCallback
   */
  updateFinalizeSuccessCallback: function updateFinalizeSuccessCallback(data) {
    var context = data ? Em.get(data, 'items.firstObject.upgrade_groups.firstObject.upgrade_items.firstObject.UpgradeItem.context') : '';
    this.set('isFinalizeItem', context === this.get('finalizeContext'));
  },

  updateFinalizeErrorCallback: function updateFinalizeErrorCallback() {
    this.set('isFinalizeItem', false);
  },

  /**
   * show dialog with tasks of upgrade
   * @return {App.ModalPopup}
   */
  openUpgradeDialog: function openUpgradeDialog() {
    if (App.isAuthorized('CLUSTER.UPGRADE_DOWNGRADE_STACK') && !this.get('isWizardRestricted')) {
      App.propertyDidChange('upgradeSuspended');
      App.router.transitionTo('admin.stackUpgrade');
    }
  },

  /**
   * returns url to get data for repoVersion or clusterStackVersion
   * @param {Boolean} stack true if load clusterStackVersion
   * @param {Boolean} fullLoad true if load all data
   * @returns {String}
   * @method getUrl
   */
  getUrl: function getUrl(stack, fullLoad) {
    if (App.get('testMode')) {
      return stack ? this.get('mockStackUrl') : this.get('mockRepoUrl');
    } else {
      if (fullLoad) {
        return stack ? this.get('realStackUrl') : this.get('realRepoUrl');
      } else {
        return this.get('realUpdateUrl');
      }
    }
  },

  /**
   * get stack versions from server and push it to model
   * @return {*}
   * @method loadStackVersionsToModel
   */
  loadStackVersionsToModel: function loadStackVersionsToModel(fullLoad) {
    var dfd = $.Deferred();
    App.HttpClient.get(this.getUrl(true, fullLoad), App.stackVersionMapper, {
      complete: function complete() {
        dfd.resolve();
      }
    });
    return dfd.promise();
  },

  /**
   * get repo versions from server and push it to model
   * @return {*}
   * @params {Boolean} isUpdate - if true loads part of data that need to be updated
   * @method loadRepoVersionsToModel()
   */
  loadRepoVersionsToModel: function loadRepoVersionsToModel() {
    var dfd = $.Deferred();
    App.HttpClient.get(this.getUrl(false, true), App.repoVersionMapper, {
      complete: function complete() {
        dfd.resolve();
      }
    });
    return dfd.promise();
  },

  /**
   * set status to Upgrade item
   * @param item
   * @param status
   */
  setUpgradeItemStatus: function setUpgradeItemStatus(item, status) {
    this.set('requestInProgress', true);
    return App.ajax.send({
      name: 'admin.upgrade.upgradeItem.setState',
      sender: this,
      data: {
        upgradeId: item.get('request_id'),
        itemId: item.get('stage_id'),
        groupId: item.get('group_id'),
        status: status
      },
      callback: function callback() {
        this.sender.set('requestInProgress', false);
      }
    }).done(function () {
      item.set('status', status);
    });
  },

  currentVersionObserver: function () {
    var versionNumber = this.get('currentVersion.repository_version');
    var currentVersionObject = App.RepositoryVersion.find().findProperty('status', 'CURRENT');
    var versionName = currentVersionObject && currentVersionObject.get('stackVersionType');
    App.set('isStormMetricsSupported', versionName != 'HDP' || stringUtils.compareVersions(versionNumber, '2.2.2') > -1 || !versionNumber);
  }.observes('currentVersion.repository_version'),

  /**
   * get the installed repositories of HDP from server
   */
  loadRepositories: function () {
    if (App.router.get('clusterController.isLoaded') && App.get('currentStackVersion')) {
      var nameVersionCombo = App.get('currentStackVersion');
      var stackName = nameVersionCombo.split('-')[0];
      var stackVersion = nameVersionCombo.split('-')[1];
      App.ajax.send({
        name: 'cluster.load_repositories',
        sender: this,
        data: {
          stackName: stackName,
          stackVersion: stackVersion
        },
        success: 'loadRepositoriesSuccessCallback',
        error: 'loadRepositoriesErrorCallback'
      });
    }
  }.observes('App.router.clusterController.isLoaded'),

  loadRepositoriesSuccessCallback: function loadRepositoriesSuccessCallback(data) {
    var allRepos = [];
    data.items.forEach(function (os) {
      os.repositories.forEach(function (repository) {
        var osType = repository.Repositories.os_type;
        var repo = Em.Object.create({
          baseUrl: repository.Repositories.base_url,
          osType: osType,
          repoId: repository.Repositories.repo_id,
          repoName: repository.Repositories.repo_name,
          stackName: repository.Repositories.stack_name,
          stackVersion: repository.Repositories.stack_version,
          isFirst: false
        });
        var group = allRepos.findProperty('name', osType);
        if (!group) {
          group = {
            name: osType,
            repositories: []
          };
          repo.set('isFirst', true);
          allRepos.push(group);
        }
        group.repositories.push(repo);
      });
    }, this);
    allRepos.stackVersion = App.get('currentStackVersionNumber');
    this.set('allRepos', allRepos);
  },

  loadRepositoriesErrorCallback: function loadRepositoriesErrorCallback(request, ajaxOptions, error) {},

  /**
   * @returns {$.ajax}
   */
  suspendUpgrade: function suspendUpgrade() {
    var self = this;
    return this.abortUpgradeWithSuspend().done(function () {
      App.set('upgradeState', 'ABORTED');
      self.setDBProperty('upgradeState', 'ABORTED');
      App.clusterStatus.setClusterStatus({
        wizardControllerName: self.get('name'),
        localdb: App.db.data
      });
    });
  },

  /**
   * @returns {$.ajax}
   */
  resumeUpgrade: function resumeUpgrade() {
    var self = this;
    this.retryUpgrade().done(function () {
      App.set('upgradeState', 'PENDING');
      App.propertyDidChange('upgradeSuspended');
      self.setDBProperty('upgradeState', 'PENDING');
      App.clusterStatus.setClusterStatus({
        wizardControllerName: self.get('name'),
        localdb: App.db.data
      });
    });
  },

  /**
   *
   * @param {Em.Object} version
   */
  confirmRevertPatchUpgrade: function confirmRevertPatchUpgrade(version) {
    var self = this;
    var currentStack = App.RepositoryVersion.find(this.get('currentVersion.id'));
    App.ModalPopup.show({
      header: Em.I18n.t('popup.confirmation.commonHeader'),
      popupBody: Em.I18n.t('admin.stackVersions.upgrade.patch.revert.confirmation'),
      bodyClass: Em.View.extend({
        classNames: ['revert-patch-upgrade-confirmation'],
        servicesToBeReverted: this.getServicesToBeReverted(version, currentStack),
        stackFromVersion: version.get('displayName'),
        stackToVersion: currentStack.get('displayNameSimple'),
        templateName: require('templates/common/modal_popups/revert_patch_upgrade_confirmation')
      }),
      onPrimary: function onPrimary() {
        self.revertPatchUpgrade(version);
        this._super();
      }
    });
  },

  /**
   *
   * @param {Em.Object} version
   * @param {Em.Object} currentStack
   * @returns {Array}
   */
  getServicesToBeReverted: function getServicesToBeReverted(version, currentStack) {
    return version.get('stackServices').filter(function (_service) {
      var originalService = App.Service.find(_service.get('name'));
      var isOriginal = originalService.get('desiredRepositoryVersionId') === version.get('id');
      return originalService.get('isLoaded') && _service.get('isAvailable') && isOriginal;
    }).map(function (_service) {
      return {
        displayName: _service.get('displayName'),
        fromVersion: _service.get('latestVersion'),
        toVersion: currentStack.get('stackServices').findProperty('name', _service.get('name')).get('latestVersion')
      };
    });
  },

  /**
   *
   * @param {Em.Object} version
   * @returns {$.ajax}
   */
  revertPatchUpgrade: function revertPatchUpgrade(version) {
    this.set('requestInProgress', true);
    return App.ajax.send({
      name: 'admin.upgrade.revert',
      sender: this,
      success: 'upgradeSuccessCallback',
      error: 'upgradeErrorCallback',
      callback: function callback() {
        this.sender.set('requestInProgress', false);
      },
      data: {
        upgradeId: version.get('stackVersion').get('revertUpgradeId'),
        id: version.get('id'),
        value: version.get('repositoryVersion'),
        label: version.get('displayName'),
        isDowngrade: true
      }
    });
  },

  /**
   * @param {App.RepositoryVersion} version
   * @returns {Em.Object}
   */
  confirmDiscardRepoVersion: function confirmDiscardRepoVersion(version) {
    var self = this;
    return App.showConfirmationPopup(function () {
      self.discardRepoVersion(version);
    });
  },

  /**
   * @param {App.RepositoryVersion} version
   * @returns {$.ajax}
   */
  discardRepoVersion: function discardRepoVersion(version) {
    this.set('requestInProgress', true);
    return App.ajax.send({
      name: 'admin.stack_versions.discard',
      sender: this,
      callback: function callback() {
        this.sender.set('requestInProgress', false);
      },
      data: {
        id: version.get('id'),
        stackName: version.get('stackVersionType'),
        stackVersion: version.get('stackVersionNumber')
      }
    });
  },

  /**
   * restore last Upgrade data
   * @param {object} lastUpgradeData
   */
  restoreLastUpgrade: function restoreLastUpgrade(lastUpgradeData) {
    var self = this;
    var upgradeType = this.get('upgradeMethods').findProperty('type', lastUpgradeData.Upgrade.upgrade_type);
    var isDowngrade = lastUpgradeData.Upgrade.direction === 'DOWNGRADE';
    this.setDBProperties({
      toVersion: lastUpgradeData.Upgrade.associated_version,
      upgradeId: lastUpgradeData.Upgrade.request_id,
      isDowngrade: isDowngrade,
      upgradeState: lastUpgradeData.Upgrade.request_status,
      upgradeType: lastUpgradeData.Upgrade.upgrade_type,
      isWizardRestricted: upgradeType.get('isWizardRestricted'),
      downgradeAllowed: lastUpgradeData.Upgrade.downgrade_allowed,
      upgradeTypeDisplayName: upgradeType.get('displayName'),
      isSuspended: lastUpgradeData.Upgrade.suspended,
      failuresTolerance: {
        skipComponentFailures: lastUpgradeData.Upgrade.skip_failures,
        skipSCFailures: lastUpgradeData.Upgrade.skip_service_check_failures
      }
    });
    this.initDBProperties();
    App.set('upgradeState', lastUpgradeData.Upgrade.request_status);
    this.loadRepoVersionsToModel().done(function () {
      var version = App.RepositoryVersion.find().findProperty('repositoryVersion', lastUpgradeData.Upgrade.associated_version);
      var upgradeVersion = version && version.get('displayName');
      self.setDBProperty('upgradeVersion', upgradeVersion);
      self.set('upgradeVersion', upgradeVersion);
    });
  },

  /**
   * Build table from configs list and open new window to show this table
   * @param configs
   */
  openConfigsInNewWindow: function openConfigsInNewWindow(configs) {
    var newWindow;
    var output = '';

    output += '<table style="text-align: left;"><thead><tr>' + '<th>' + Em.I18n.t('popup.clusterCheck.Upgrade.configsMerge.configType') + '</th>' + '<th>' + Em.I18n.t('popup.clusterCheck.Upgrade.configsMerge.propertyName') + '</th>' + '<th>' + Em.I18n.t('popup.clusterCheck.Upgrade.configsMerge.currentValue') + '</th>' + '<th>' + Em.I18n.t('popup.clusterCheck.Upgrade.configsMerge.recommendedValue') + '</th>' + '<th>' + Em.I18n.t('popup.clusterCheck.Upgrade.configsMerge.resultingValue') + '</th>' + '</tr></thead><tbody>';

    configs.context.forEach(function (config) {
      output += '<tr>' + '<td>' + config.type + '</td>' + '<td>' + config.name + '</td>' + '<td>' + config.currentValue + '</td>' + '<td>' + config.recommendedValue + '</td>' + '<td>' + config.resultingValue + '</td>' + '</tr>';
    });

    output += '</tbody></table>';
    newWindow = window.open();
    newWindow.document.write(output);
    newWindow.focus();
  },

  serviceVersionsMap: {},

  /**
   * load version for services to display on admin service page
   * it should be fetched from the repo which corresponds to the desiredRepositoryVersionId of the service
   */
  getServiceVersionFromRepo: function () {
    if (!App.router.get('clusterController.isLoaded') || !App.router.get('clusterController.isStackVersionsLoaded')) {
      return;
    }

    var currentStackName = App.get('currentStackName');
    var currentStackVersionNumber = App.get('currentStackVersionNumber');
    var map = this.get('serviceVersionsMap');

    var stackServices = App.StackService.find().filter(function (service) {
      return service.get('stackName') === currentStackName && service.get('stackVersion') === currentStackVersionNumber;
    });

    stackServices.forEach(function (service) {
      var serviceName = service.get('serviceName');

      var installedService = App.Service.find().findProperty('serviceName', serviceName);
      var desiredRepositoryVersionId = installedService ? installedService.get('desiredRepositoryVersionId') : null;
      var serviceVersion = "";
      var currentStackRepoVersions = App.RepositoryVersion.find().filter(function (repoVersion) {
        return repoVersion.get('stackVersionType') === currentStackName && repoVersion.get('stackVersionNumber') === currentStackVersionNumber;
      });

      //Get service version from the current standard repo for stack services which are not installed else search by id
      if (!desiredRepositoryVersionId) {
        currentStackRepoVersions.forEach(function (repoVersion) {
          if (repoVersion.get('isCurrent') && repoVersion.get('isStandard')) {
            serviceVersion = repoVersion.get('stackServices').findProperty('name', serviceName).get('latestVersion');
          }
        });
      } else {
        var repoVersion = currentStackRepoVersions.findProperty('id', desiredRepositoryVersionId);
        serviceVersion = repoVersion.get('stackServices').findProperty('name', serviceName).get('latestVersion');
      }
      map[serviceName] = serviceVersion;
    });
  }.observes('App.router.clusterController.isLoaded', 'App.router.clusterController.isStackVersionsLoaded')

});

});

require.register("controllers/main/admin/stack_upgrade_history_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var stringUtils = require('utils/string_utils');

App.MainAdminStackUpgradeHistoryController = Em.ArrayController.extend({
  name: 'mainAdminStackUpgradeHistoryController',

  startIndex: 1,

  resetStartIndex: false,

  /**
   * status of tasks/items/groups which should be grayed out and disabled
   * @type {Array}
   */
  nonActiveStates: ['PENDING'],

  /**
   * mutable properties of Upgrade Task
   * @type {Array}
   */
  taskDetailsProperties: ['status', 'stdout', 'stderr', 'error_log', 'host_name', 'output_log'],

  /**
   * Current upgrade record clicked on the UI
   * @type {App.StackUpgradeHistory|null}
   * */
  currentUpgradeRecord: null,

  isDowngrade: false,

  upgradeData: null,

  /**
   * List of all <code>App.StackUpgradeHistory</code>. Latest one at the beginning
   * @type {App.StackUpgradeHistory[]}
   */
  content: App.StackUpgradeHistory.find(),

  upgradeHistoryUrl: function () {
    return App.get('apiPrefix') + '/clusters/' + App.get('clusterName') + '/upgrades?fields=Upgrade';
  }.property('App.clusterName'),

  upgradeUpgradeRecordUrl: function () {
    var record = this.get('currentUpgradeRecord');
    return App.get('apiPrefix') + '/clusters/' + App.get('clusterName') + '/upgrades/' + record.get('requestId');
  }.property('App.clusterName', 'currentUpgradeRecord'),

  loadStackUpgradeHistoryToModel: function loadStackUpgradeHistoryToModel() {
    console.log('Load stack upgrade history');
    var dfd = $.Deferred();
    App.HttpClient.get(this.get('upgradeHistoryUrl'), App.stackUpgradeHistoryMapper, {
      complete: function complete() {
        dfd.resolve();
      }
    });
    return dfd.promise();
  },

  loadStackUpgradeRecord: function loadStackUpgradeRecord() {
    var record = this.get('currentUpgradeRecord');
    this.set('isDowngrade', 'DOWNGRADE' == record.get('direction'));
    var dfd = $.Deferred();
    var self = this;
    if (record != null) {
      App.ajax.send({
        name: 'admin.upgrade.data',
        sender: this,
        data: {
          id: record.get('requestId')
        },
        success: 'loadUpgradeRecordSuccessCallback'
      }).then(dfd.resolve).always(function () {});
    } else {
      dfd.resolve();
    }
    return dfd.promise();
  },

  loadUpgradeRecordSuccessCallback: function loadUpgradeRecordSuccessCallback(newData) {
    if (Em.isNone(newData)) {
      var record = this.get('currentUpgradeRecord');
      console.debug('No data returned for upgrad record ' + record.get('requestId'));
      return;
    }
    var upgradeGroups = [];
    if (newData.upgrade_groups) {
      var nonActiveStates = this.get('nonActiveStates');
      //wrap all entities into App.finishedUpgradeEntity
      newData.upgrade_groups.forEach(function (newGroup) {
        var hasExpandableItems = newGroup.upgrade_items.some(function (item) {
          return !nonActiveStates.contains(item.UpgradeItem.status);
        }),
            oldGroup = App.finishedUpgradeEntity.create({ type: 'GROUP', hasExpandableItems: hasExpandableItems }, newGroup.UpgradeGroup),
            upgradeItems = [];
        newGroup.upgrade_items.forEach(function (item) {
          var oldItem = App.finishedUpgradeEntity.create({ type: 'ITEM' }, item.UpgradeItem);
          this.formatMessages(oldItem);
          oldItem.set('tasks', []);
          upgradeItems.pushObject(oldItem);
        }, this);
        upgradeItems.reverse();
        oldGroup.set('upgradeItems', upgradeItems);
        upgradeGroups.pushObject(oldGroup);
      }, this);
      upgradeGroups.reverse();
      this.set('upgradeData', Em.Object.create({
        upgradeGroups: upgradeGroups,
        upgrade_groups: newData.upgrade_groups,
        Upgrade: newData.Upgrade
      }));
    }
  },

  /**
   * format upgrade item text
   * @param {App.finishedUpgradeEntity} oldItem
   */
  formatMessages: function formatMessages(oldItem) {
    var text = oldItem.get('text');
    var messages = [];

    try {
      var messageArray = JSON.parse(text);
      for (var i = 0; i < messageArray.length; i++) {
        messages.push(messageArray[i].message);
      }
      oldItem.set('text', messages.join(' '));
    } catch (err) {
      console.warn('Upgrade Item has malformed text');
    }
    oldItem.set('messages', messages);
  },

  /**
   * request Upgrade Item and its tasks from server
   * @param {Em.Object} item
   * @param {Function} customCallback
   * @return {$.ajax}
   */
  getUpgradeItem: function getUpgradeItem(item) {
    return App.ajax.send({
      name: 'admin.upgrade.upgrade_item',
      sender: this,
      data: {
        upgradeId: item.get('request_id'),
        groupId: item.get('group_id'),
        stageId: item.get('stage_id')
      },
      success: 'getUpgradeItemSuccessCallback'
    });
  },

  /**
   * success callback of <code>getTasks</code>
   * @param {object} data
   */
  getUpgradeItemSuccessCallback: function getUpgradeItemSuccessCallback(data) {
    this.get('upgradeData.upgradeGroups').forEach(function (group) {
      if (group.get('group_id') === data.UpgradeItem.group_id) {
        group.get('upgradeItems').forEach(function (item) {
          if (item.get('stage_id') === data.UpgradeItem.stage_id) {
            if (item.get('tasks.length')) {
              data.tasks.forEach(function (task) {
                var currentTask = item.get('tasks').findProperty('id', task.Tasks.id);
                this.get('taskDetailsProperties').forEach(function (property) {
                  if (!Em.isNone(task.Tasks[property])) {
                    currentTask.set(property, task.Tasks[property]);
                  }
                }, this);
              }, this);
            } else {
              var tasks = [];
              data.tasks.forEach(function (task) {
                tasks.pushObject(App.finishedUpgradeEntity.create({
                  type: 'TASK',
                  group_id: data.UpgradeItem.group_id
                }, task.Tasks));
              });
              item.set('tasks', tasks);
            }
            item.set('isTasksLoaded', true);
          }
        }, this);
      }
    }, this);
  },

  /**
   * request Upgrade Task
   * @param {Em.Object} task
   * @return {$.ajax}
   */
  getUpgradeTask: function getUpgradeTask(task) {
    return App.ajax.send({
      name: 'admin.upgrade.upgrade_task',
      sender: this,
      data: {
        upgradeId: task.get('request_id'),
        groupId: task.get('group_id'),
        stageId: task.get('stage_id'),
        taskId: task.get('id'),
        task: task
      },
      success: 'getUpgradeTaskSuccessCallback'
    });
  },

  getUpgradeTaskSuccessCallback: function getUpgradeTaskSuccessCallback(data, xhr, params) {
    this.get('taskDetailsProperties').forEach(function (property) {
      params.task.set(property, data.Tasks[property]);
    }, this);
  },

  /**
   * status of Upgrade request
   * @type {string}
   */
  requestStatus: function () {
    if (this.get('upgradeData')) {
      if (this.get('upgradeData.Upgrade')) {
        return this.get('upgradeData.Upgrade.request_status');
      } else {
        return '';
      }
    } else {
      return '';
    }
  }.property('upgradeData.Upgrade.request_status')
});

});

require.register("controllers/main/alert_definitions_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainAlertDefinitionsController = Em.ArrayController.extend({

  name: 'mainAlertDefinitionsController',

  /**
   * Define whether restore filter conditions from local db
   * @type {Boolean}
   */
  showFilterConditionsFirstLoad: false,

  contentUpdater: null,

  /**
   * List of all <code>App.AlertDefinition</code>
   * @type {App.AlertDefinition[]}
   */
  content: App.AlertDefinition.find(),

  /**
   * Generates key for alert summary that represents current state
   */
  getSummaryCache: function getSummaryCache() {
    var res = '';
    this.get('content').forEach(function (o) {
      var summary = o.get('summary');
      o.get('order').forEach(function (state) {
        res += summary[state] ? summary[state].count + summary[state].maintenanceCount : 0;
      });
    });

    return res;
  },

  generateCacheByKey: function generateCacheByKey(key) {
    if (key === 'summary') {
      return this.getSummaryCache();
    }

    return this.get('content').mapProperty(key).join('');
  },

  contentWasChanged: function contentWasChanged(key) {
    var updatedCache = this.generateCacheByKey(key);
    if (this.get('cache.' + key) !== updatedCache) {
      this.set('cache.' + key, updatedCache);
      this.propertyDidChange('contentUpdater');
    }
  },

  cache: {
    'label': '',
    'summary': '',
    'serviceName': '',
    'lastTriggered': '',
    'enabled': ''
  },

  /**
   * Enable/disable alertDefinition confirmation popup
   * @param {object} event
   * @method toggleState
   * @return {App.ModalPopup}
   */
  toggleState: function toggleState(event) {
    var alertDefinition = event.context;
    var self = this;
    var bodyMessage = Em.Object.create({
      confirmMsg: alertDefinition.get('enabled') ? Em.I18n.t('alerts.table.state.enabled.confirm.msg') : Em.I18n.t('alerts.table.state.disabled.confirm.msg'),
      confirmButton: alertDefinition.get('enabled') ? Em.I18n.t('alerts.table.state.enabled.confirm.btn') : Em.I18n.t('alerts.table.state.disabled.confirm.btn')
    });

    return App.showConfirmationFeedBackPopup(function (query) {
      self.toggleDefinitionState(alertDefinition);
    }, bodyMessage);
  },

  /**
   * Enable/disable alertDefinition
   * @param {object} alertDefinition
   * @returns {$.ajax}
   * @method toggleDefinitionState
   */
  toggleDefinitionState: function toggleDefinitionState(alertDefinition) {
    var newState = !alertDefinition.get('enabled');
    alertDefinition.set('enabled', newState);
    Em.run.next(function () {
      App.tooltip($('.enable-disable-button'));
    });
    return App.ajax.send({
      name: 'alerts.update_alert_definition',
      sender: this,
      data: {
        id: alertDefinition.get('id'),
        data: {
          "AlertDefinition/enabled": newState
        }
      }
    });
  },

  /**
   *  ========================== alerts notifications dropdown dialog =========================
   */

  /**
   * Number of all critical and warning alert instances
   * Calculation is based on each <code>alertDefinitions.summary</code>
   * @type {Number}
   */
  unhealthyAlertInstancesCount: function () {
    return this.get('criticalAlertInstancesCount') + this.get('warningAlertInstancesCount');
  }.property('criticalAlertInstancesCount', 'warningAlertInstancesCount'),

  criticalAlertInstancesCount: function () {
    return this.get('content').map(function (alertDefinition) {
      return alertDefinition.getWithDefault('summary.CRITICAL.count', 0);
    }).reduce(Em.sum, 0);
  }.property('content.@each.summary'),

  warningAlertInstancesCount: function () {
    return this.get('content').map(function (alertDefinition) {
      return alertDefinition.getWithDefault('summary.WARNING.count', 0);
    }).reduce(Em.sum, 0);
  }.property('content.@each.summary'),

  /**
   * if critical alerts exist, the alert badge should be red.
   * @type {Boolean}
   */
  isCriticalAlerts: function () {
    return this.get('content').invoke('getWithDefault', 'summary.CRITICAL.count', 0).reduce(Em.sum, 0) !== 0;
  }.property('content.@each.summary')
});

});

require.register("controllers/main/alerts/add_alert_definition/add_alert_definition_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddAlertDefinitionController = App.WizardController.extend({

  name: 'addAlertDefinitionController',

  totalSteps: 3,

  content: {
    selectedType: null
  },

  /**
   * Do request to create new alert definition
   * @param {object} newDefinitionData
   * @returns {$.ajax}
   */
  createNewAlertDefinition: function createNewAlertDefinition(newDefinitionData) {
    return App.ajax.send({
      name: 'alerts.create_alert_definition',
      sender: this,
      data: {
        data: newDefinitionData
      }
    });
  },

  finish: function finish() {
    this.clear();
  }

});

});

require.register("controllers/main/alerts/add_alert_definition/step1_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddAlertDefinitionStep1Controller = Em.Controller.extend({

  name: 'addAlertDefinitionStep1',

  /**
   * List of available alert definition types
   * @type {{name: string, isActive: boolean}[]}
   */
  alertDefinitionsTypes: function () {
    return App.AlertType.find().map(function (option) {
      return Em.Object.create({
        name: option.get('name'),
        displayName: option.get('displayName'),
        icon: option.get('iconPath'),
        description: option.get('description')
      });
    });
  }.property(),

  /**
   * Set selectedType if it exists in the wizard controller
   * @method loadStep
   */
  loadStep: function loadStep() {
    this.set('content.selectedType', '');
  },

  /**
   * Handler for select alert definition type selection
   * @param {object} e
   * @method selectType
   */
  selectType: function selectType(e) {
    var type = e.context,
        types = this.get('alertDefinitionsTypes');
    this.set('content.selectedType', type.name);
    $("[rel='selectable-tooltip']").trigger('mouseleave');
    App.router.send('next');
  }

});

});

require.register("controllers/main/alerts/add_alert_definition/step2_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddAlertDefinitionStep2Controller = Em.Controller.extend({

  name: 'addAlertDefinitionStep2Controller'

});

});

require.register("controllers/main/alerts/add_alert_definition/step3_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddAlertDefinitionStep3Controller = Em.Controller.extend({

  name: 'addAlertDefinitionStep3Controller'

});

});

require.register("controllers/main/alerts/alert_definitions_actions_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var validator = require('utils/validator');

App.MainAlertDefinitionActionsController = Em.ArrayController.extend({

  name: 'mainAlertDefinitionActionsController',

  /**
   * List of available actions for alert definitions
   * @type {{title: string, icon: string, action: string, showDivider: boolean}[]}
   */
  content: function () {
    var content = [];
    if (App.supports.createAlerts) {
      content.push({
        title: Em.I18n.t('alerts.actions.create'),
        icon: 'glyphicon glyphicon-plus',
        action: 'createNewAlertDefinition',
        showDivider: true
      });
    }
    content.push({
      title: Em.I18n.t('alerts.actions.manageGroups'),
      icon: 'glyphicon glyphicon-th-large',
      action: 'manageAlertGroups',
      showDivider: false
    });
    if (App.isAuthorized('CLUSTER.MANAGE_ALERT_NOTIFICATIONS')) {
      content.push({
        title: Em.I18n.t('alerts.actions.manageNotifications'),
        icon: 'glyphicon glyphicon-envelope',
        action: 'manageNotifications',
        showDivider: false
      });
    }
    content.push({
      title: Em.I18n.t('alerts.actions.manageSettings'),
      icon: 'glyphicon glyphicon-cog',
      action: 'manageSettings',
      showDivider: false
    });
    return content;
  }.property('App.supports.createAlerts', 'App.auth'),

  /**
   * Common handler for menu item click
   * Call proper controller's method described in <code>action</code>-field (see <code>content</code>)
   * @param {object} event
   * @method actionHandler
   */
  actionHandler: function actionHandler(event) {
    var menuElement = event.context,
        action = menuElement.action;
    if ('function' === Em.typeOf(Em.get(this, action))) {
      this[action]();
    } else {
      Em.assert('Invalid action provided - ' + action, false);
    }
  },

  /**
   * Start "Create Alert Definition" wizard
   * @method createNewAlertDefinition
   */
  createNewAlertDefinition: function createNewAlertDefinition() {
    App.router.transitionTo('alertAdd');
  },

  /**
   * Handler when clicking on "Manage Alert Groups", a popup will show up
   * @method manageAlertGroups
   * @return {App.ModalPopup}
   */
  manageAlertGroups: function manageAlertGroups() {

    return App.ModalPopup.show({

      header: Em.I18n.t('alerts.actions.manage_alert_groups_popup.header'),

      bodyClass: App.MainAlertsManageAlertGroupView.extend({
        controllerBinding: 'App.router.manageAlertGroupsController'
      }),

      classNames: ['common-modal-wrapper', 'manage-alert-group-popup'],
      modalDialogClasses: ['modal-lg'],
      primary: Em.I18n.t('common.save'),

      onPrimary: function onPrimary() {
        var modifiedAlertGroups = this.get('subViewController.defsModifiedAlertGroups');
        var dataForSuccessPopup = {
          created: modifiedAlertGroups.toCreate.length,
          deleted: modifiedAlertGroups.toDelete.length,
          updated: modifiedAlertGroups.toSet.length
        };
        var showSuccessPopup = dataForSuccessPopup.created + dataForSuccessPopup.deleted + dataForSuccessPopup.updated > 0;
        // Save modified Alert-groups
        var self = this;
        var errors = [];
        var deleteQueriesCounter = modifiedAlertGroups.toDelete.length;
        var createQueriesCounter = modifiedAlertGroups.toCreate.length;
        var deleteQueriesRun = false;
        var createQueriesRun = false;
        var runNextQuery = function runNextQuery() {
          if (!deleteQueriesRun && deleteQueriesCounter > 0) {
            deleteQueriesRun = true;
            modifiedAlertGroups.toDelete.forEach(function (group) {
              self.get('subViewController').removeAlertGroup(group, finishFunction, finishFunction);
            }, this);
          } else if (!createQueriesRun && deleteQueriesCounter < 1) {
            createQueriesRun = true;
            modifiedAlertGroups.toSet.forEach(function (group) {
              self.get('subViewController').updateAlertGroup(group, finishFunction, finishFunction);
            }, this);
            modifiedAlertGroups.toCreate.forEach(function (group) {
              self.get('subViewController').postNewAlertGroup(group, finishFunction);
            }, this);
          }
        };
        var finishFunction = function finishFunction(xhr, text, errorThrown) {
          if (xhr && errorThrown) {
            var error = xhr.status + "(" + errorThrown + ") ";
            try {
              var json = $.parseJSON(xhr.responseText);
              error += json.message;
            } catch (err) {}
            errors.push(error);
          }
          if (createQueriesRun) {
            createQueriesCounter--;
          } else {
            deleteQueriesCounter--;
          }
          if (deleteQueriesCounter + createQueriesCounter < 1) {
            if (errors.length > 0) {
              self.get('subViewController').set('errorMessage', errors.join(". "));
            } else {
              self.hide();
              if (showSuccessPopup) {
                App.ModalPopup.show({
                  secondary: null,
                  header: Em.I18n.t('alerts.groups.successPopup.header'),
                  bodyClass: Em.View.extend({
                    dataForSuccessPopup: dataForSuccessPopup,
                    templateName: require('templates/main/alerts/alert_groups/success_popup_body')
                  })
                });
              }
              App.router.get('updateController').updateAlertNotifications(Em.K);
            }
          } else {
            runNextQuery();
          }
        };
        runNextQuery();
      },

      subViewController: function () {
        return App.router.get('manageAlertGroupsController');
      }.property('App.router.manageAlertGroupsController'),

      disablePrimary: Em.computed.not('subViewController.isDefsModified')

    });
  },

  /**
   * "Manage Alert Notifications" handler
   * @method manageNotifications
   * @return {App.ModalPopup}
   */
  manageNotifications: function manageNotifications() {

    return App.ModalPopup.show({

      header: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.header'),

      bodyClass: App.ManageAlertNotificationsView.extend({
        controllerBinding: 'App.router.manageAlertNotificationsController'
      }),

      classNames: ['common-modal-wrapper', 'manage-configuration-group-popup'],
      modalDialogClasses: ['modal-lg'],

      secondary: null,

      primary: Em.I18n.t('common.close'),

      autoHeight: false

    });
  },

  /**
   * "Manage Alert Settings" handler
   * @method manageSettings
   * @return {App.ModalPopup}
   */
  manageSettings: function manageSettings() {
    var controller = this;
    var configProperties = App.router.get('clusterController.clusterEnv.properties');

    return App.ModalPopup.show({
      classNames: ['forty-percent-width-modal'],
      header: Em.I18n.t('alerts.actions.manageSettings'),
      primary: Em.I18n.t('common.save'),
      secondary: Em.I18n.t('common.cancel'),
      inputValue: configProperties.alerts_repeat_tolerance || '1',
      errorMessage: Em.I18n.t('alerts.actions.editRepeatTolerance.error'),
      isInvalid: function () {
        var intValue = Number(this.get('inputValue'));
        return this.get('inputValue') !== 'DEBUG' && (!validator.isValidInt(intValue) || intValue < 1 || intValue > 99);
      }.property('inputValue'),
      disablePrimary: Em.computed.alias('isInvalid'),
      onPrimary: function onPrimary() {
        if (this.get('isInvalid')) {
          return;
        }
        configProperties.alerts_repeat_tolerance = this.get('inputValue');
        App.ajax.send({
          name: 'admin.save_configs',
          sender: controller,
          data: {
            siteName: 'cluster-env',
            properties: configProperties
          },
          error: 'manageSettingsErrorCallback'
        });
        this.hide();
      },
      bodyClass: Ember.View.extend({
        templateName: require('templates/common/modal_popups/prompt_popup'),
        title: Em.I18n.t('alerts.actions.editRepeatTolerance.title'),
        description: Em.I18n.t('alerts.actions.editRepeatTolerance.description'),
        label: Em.I18n.t('alerts.actions.editRepeatTolerance.label')
      })
    });
  },

  manageSettingsErrorCallback: function manageSettingsErrorCallback(data) {
    var error = Em.I18n.t('alerts.actions.manageSettings.error');
    if (data && data.responseText) {
      try {
        var json = $.parseJSON(data.responseText);
        error += json.message;
      } catch (err) {}
    }
    App.showAlertPopup(Em.I18n.t('alerts.actions.manageSettings.error'), error);
  }

});

});

require.register("controllers/main/alerts/alert_instances_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainAlertInstancesController = Em.Controller.extend({

  name: 'mainAlertInstancesController',

  content: App.AlertInstance.find(),

  /**
   * @type {App.AlertInstance[]}
   */
  unhealthyAlertInstances: [],

  updateUnhealthyAlertInstances: function () {
    Em.run.once(this, this.updateUnhealthyAlertInstancesOnce);
  }.observes('content.[]'),

  updateUnhealthyAlertInstancesOnce: function updateUnhealthyAlertInstancesOnce() {
    var alertInstances = App.AlertInstance.find().filter(function (item) {
      return ['CRITICAL', 'WARNING'].contains(item.get('state'));
    });
    this.set('unhealthyAlertInstances', alertInstances);
  },

  /**
   * Are alertInstances loaded
   * @type {boolean}
   */
  isLoaded: false,

  /**
   * A flag to reload alert instances table every 10 seconds
   * @type {boolean}
   */
  reload: false,

  /**
   * Causes automatic updates of content if set to true
   * @type {boolean}
   */
  isUpdating: false,

  /**
   * Times for alert instances updater
   * Used in <code>scheduleUpdate</code>
   * @type {number|null}
   */
  updateTimer: null,

  /**
   * @type {string|null} sourceName - hostName or alertDefinitionId
   */
  sourceName: null,

  /**
   * @type {string|null} sourceType - 'HOST'|'ALERT_DEFINITION'
   */
  sourceType: null,

  /**
   * Load alert instances from server (all, for selected host, for selected alert definition)
   * @returns {$.ajax}
   * @method fetchAlertInstances
   */
  fetchAlertInstances: function fetchAlertInstances() {
    var sourceType = this.get('sourceType'),
        sourceName = this.get('sourceName'),
        ajaxData = {
      sender: this,
      success: 'getAlertInstancesSuccessCallback',
      error: 'getAlertInstancesErrorCallback'
    };

    switch (sourceType) {
      case 'HOST':
        $.extend(ajaxData, {
          name: 'alerts.instances.by_host',
          data: {
            hostName: sourceName
          }
        });
        break;

      case 'ALERT_DEFINITION':
        $.extend(ajaxData, {
          name: 'alerts.instances.by_definition',
          data: {
            definitionId: sourceName
          }
        });
        break;

      default:
        $.extend(ajaxData, {
          name: 'alerts.instances'
        });
        break;
    }

    return App.ajax.send(ajaxData);
  },

  /**
   * Pseudo for <code>fetchAlertInstances</code>
   * Used to get all alert instances
   * @method loadAlertInstances
   */
  loadAlertInstances: function loadAlertInstances() {
    this.setProperties({
      isLoaded: false,
      sourceType: null,
      sourceName: null
    });
    this.fetchAlertInstances();
  },

  /**
   * Pseudo for <code>fetchAlertInstances</code>
   * Used to get alert instances for some host
   * @param {string} hostName
   * @method loadAlertInstancesByHost
   */
  loadAlertInstancesByHost: function loadAlertInstancesByHost(hostName) {
    this.setProperties({
      isLoaded: false,
      sourceType: 'HOST',
      sourceName: hostName
    });
    this.fetchAlertInstances();
  },

  /**
   * Pseudo for <code>fetchAlertInstances</code>
   * Used to get alert instances for some alert definition
   * @param {string} definitionId
   * @method loadAlertInstancesByAlertDefinition
   */
  loadAlertInstancesByAlertDefinition: function loadAlertInstancesByAlertDefinition(definitionId) {
    this.setProperties({
      isLoaded: false,
      sourceType: 'ALERT_DEFINITION',
      sourceName: definitionId
    });
    this.fetchAlertInstances();
  },

  scheduleUpdate: function () {
    var self = this;
    if (this.get('isUpdating')) {
      this.set('updateTimer', setTimeout(function () {
        self.fetchAlertInstances().complete(function () {
          self.scheduleUpdate();
        });
      }, App.get('alertInstancesUpdateInterval')));
    } else {
      clearTimeout(this.get('updateTimer'));
    }
  }.observes('isUpdating'),

  /**
   * Success-callback for alert instances request
   * @param {object} json
   * @method getAlertInstancesSuccessCallback
   */
  getAlertInstancesSuccessCallback: function getAlertInstancesSuccessCallback(json) {
    App.alertInstanceMapper.mapLocal(json);
    this.set('isLoaded', true);
    this.toggleProperty('reload');
  },

  /**
   * Error-callback for alert instances request
   * @method getAlertInstancesErrorCallback
   */
  getAlertInstancesErrorCallback: function getAlertInstancesErrorCallback() {
    this.set('isLoaded', true);
  }

});

});

require.register("controllers/main/alerts/definition_configs_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var numericUtils = require('utils/number_utils');

App.MainAlertDefinitionConfigsController = Em.Controller.extend({

  name: 'mainAlertDefinitionConfigsController',

  /**
   * All configurable properties of alert definition
   * @type {Array}
   */
  configs: [],

  /**
   * Define whether configs are editable
   * binds to property populated in template
   * @type {Boolean}
   */
  canEdit: true,

  /**
   * Define configs view mode (Wizard or Definition Details page)
   * @type {Boolean}
   */
  isWizard: false,

  /**
   * Alert Definition type
   * binding is set in template
   * @type {String}
   */
  alertDefinitionType: '',

  /**
   * Array of displayNames of all services
   * is used for "Service" config options
   * @type {Array}
   */
  allServices: function () {
    return App.Service.find().mapProperty('displayName').concat('CUSTOM');
  }.property(),

  /**
   * All possible values for scope propery
   * @type {Array}
   */
  allScopes: ['Any', 'Host', 'Service'],

  /**
   * Array of all aggregate-alerts names
   * @type {Array}
   */
  aggregateAlertNames: function () {
    return App.AggregateAlertDefinition.find().mapProperty('name');
  }.property(),

  /**
   * If any service selected, show related components, hide PORT and HOST field
   * If CUSTOM was selected, show no component, and show PORT and HOST field
   * @method onServiceSelect
   */
  onServiceSelect: function () {
    var serviceProperty = this.get('configs').findProperty('name', 'service');
    var componentsProperty = this.get('configs').findProperty('name', 'component');
    var defaultPortProperty = this.get('configs').findProperty('name', 'default_port');
    var uriProperty = this.get('configs').findProperty('name', 'uri');
    if (serviceProperty && serviceProperty.get('value') == 'CUSTOM') {
      componentsProperty.set('options', ['No component']);
      componentsProperty.set('canEdit', false);
      var results = this.get('configs');
      if (defaultPortProperty && uriProperty) {
        results = results.filter(function (config) {
          return config.name != 'default_port' && config.name != 'uri';
        });
      }
      if (!defaultPortProperty) {
        results = results.concat([App.AlertConfigProperties.DefaultPort.create({
          value: '8050'
        }), App.AlertConfigProperties.URI.create({
          value: ''
        })]);
        this.set('configs', results);
      }
    } else if (serviceProperty && serviceProperty.get('value') !== 'CUSTOM' && componentsProperty.get('value') && componentsProperty.get('value') != 'No component') {
      componentsProperty.set('options', ['No component'].concat(App.HostComponent.find().filterProperty('service.displayName', serviceProperty.get('value')).mapProperty('displayName').uniq()));
      if (!defaultPortProperty) {
        var results = this.get('configs').concat([App.AlertConfigProperties.DefaultPort.create({
          value: '8060'
        }), App.AlertConfigProperties.URI.create({
          value: ''
        })]);
        this.set('configs', results);
      }
    } else if (serviceProperty && serviceProperty.get('value') !== 'CUSTOM') {
      componentsProperty.set('options', ['No component'].concat(App.HostComponent.find().filterProperty('service.displayName', serviceProperty.get('value')).mapProperty('displayName').uniq()));
      if (defaultPortProperty && uriProperty) {
        var results = this.get('configs').filter(function (config) {
          return config.name != 'default_port' && config.name != 'uri';
        });
        this.set('configs', results);
      }
    }
  }.observes('configs.@each.value'),

  /**
   * @return {string|Null}
   * @method getThresholdsProperty
   */
  getThresholdsProperty: function getThresholdsProperty(type, property) {
    var warning = this.get('content.reporting').findProperty('type', type);
    return warning && !Ember.isNone(warning.get(property)) ? warning.get(property) : null;
  },

  /**
   * Render array of configs for appropriate alert definition type
   * @method renderConfigs
   */
  renderConfigs: function renderConfigs() {
    var self = this;
    var alertDefinitionType = this.get('alertDefinitionType');
    var configs = [];
    switch (alertDefinitionType) {
      case 'PORT':
        configs = this.renderPortConfigs();
        break;
      case 'METRIC':
        configs = this.renderMetricConfigs();
        break;
      case 'WEB':
        configs = this.renderWebConfigs();
        break;
      case 'SCRIPT':
        configs = this.renderScriptConfigs();
        break;
      case 'AGGREGATE':
        configs = this.renderAggregateConfigs();
        break;
      case 'SERVER':
        configs = this.renderServerConfigs();
        break;
      case 'RECOVERY':
        configs = this.renderRecoveryConfigs();
        break;
      case 'AMS':
        configs = this.renderAmsConfigs();
        break;
      default:
    }

    configs.forEach(function (config) {
      config.set('isDisabled', !self.get('canEdit') || config.get('readonly'));
    });

    this.set('configs', configs);
  },

  /**
   * Render config properties for port-type alert definition
   * @method renderPortConfigs
   * @returns {App.AlertConfigProperty[]}
   */
  renderPortConfigs: function renderPortConfigs() {
    var result = [];
    var alertDefinition = this.get('content');
    var isWizard = this.get('isWizard');

    if (this.get('isWizard')) {
      result = result.concat([App.AlertConfigProperties.AlertName.create({
        value: ''
      }), App.AlertConfigProperties.Service.create({
        options: this.get('allServices'),
        value: this.get('allServices')[0],
        isShifted: true
      }), App.AlertConfigProperties.Component.create({
        options: this.get('allComponents'),
        value: 'No component',
        isShifted: true
      }),

      //should be on next step
      App.AlertConfigProperties.Interval.create({
        value: isWizard ? '' : alertDefinition.get('interval')
      }), App.AlertConfigProperties.Thresholds.OkThreshold.create({
        label: 'Thresholds',
        showInputForValue: false,
        text: isWizard ? '' : this.getThresholdsProperty('ok', 'text'),
        value: isWizard ? '' : this.getThresholdsProperty('ok', 'value')
      }), App.AlertConfigProperties.Thresholds.WarningThreshold.create(App.AlertConfigProperties.Thresholds.PositiveMixin, {
        valueMetric: 'Seconds',
        text: isWizard ? '' : this.getThresholdsProperty('warning', 'text'),
        value: isWizard ? '' : this.getThresholdsProperty('warning', 'value')
      }), App.AlertConfigProperties.Thresholds.CriticalThreshold.create(App.AlertConfigProperties.Thresholds.PositiveMixin, {
        valueMetric: 'Seconds',
        text: isWizard ? '' : this.getThresholdsProperty('critical', 'text'),
        value: isWizard ? '' : this.getThresholdsProperty('critical', 'value')
      })]);
    } else {
      result = result.concat([App.AlertConfigProperties.Description.create({
        value: isWizard ? '' : alertDefinition.get('description')
      }), App.AlertConfigProperties.Interval.create({
        value: isWizard ? '' : alertDefinition.get('interval')
      }), App.AlertConfigProperties.Thresholds.OkThreshold.create({
        label: 'Thresholds',
        showInputForValue: false,
        text: isWizard ? '' : this.getThresholdsProperty('ok', 'text'),
        value: isWizard ? '' : this.getThresholdsProperty('ok', 'value')
      }), App.AlertConfigProperties.Thresholds.WarningThreshold.create(App.AlertConfigProperties.Thresholds.PositiveMixin, {
        valueMetric: 'Seconds',
        text: isWizard ? '' : this.getThresholdsProperty('warning', 'text'),
        value: isWizard ? '' : this.getThresholdsProperty('warning', 'value')
      }), App.AlertConfigProperties.Thresholds.CriticalThreshold.create(App.AlertConfigProperties.Thresholds.PositiveMixin, {
        valueMetric: 'Seconds',
        text: isWizard ? '' : this.getThresholdsProperty('critical', 'text'),
        value: isWizard ? '' : this.getThresholdsProperty('critical', 'value')
      })]);
    }

    return result;
  },

  /**
   * Render config properties for metric-type alert definition
   * @method renderMetricConfigs
   * @returns {App.AlertConfigProperty[]}
   */
  renderMetricConfigs: function renderMetricConfigs() {
    var result = [];
    var alertDefinition = this.get('content');
    var isWizard = this.get('isWizard');
    var units = this.get('content.reporting') && this.get('content.reporting').findProperty('type', 'units') ? this.get('content.reporting').findProperty('type', 'units').get('text') : null;

    if (this.get('isWizard')) {
      result = result.concat(this.renderCommonWizardConfigs());
    }

    result = result.concat([App.AlertConfigProperties.Description.create({
      value: isWizard ? '' : alertDefinition.get('description')
    }), App.AlertConfigProperties.Interval.create({
      value: isWizard ? '' : alertDefinition.get('interval')
    }), App.AlertConfigProperties.Thresholds.OkThreshold.create({
      label: 'Thresholds',
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('ok', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('ok', 'value')
    }), App.AlertConfigProperties.Thresholds.WarningThreshold.create({
      valueMetric: units,
      text: isWizard ? '' : this.getThresholdsProperty('warning', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('warning', 'value')
    }), App.AlertConfigProperties.Thresholds.CriticalThreshold.create({
      valueMetric: units,
      text: isWizard ? '' : this.getThresholdsProperty('critical', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('critical', 'value')
    }), App.AlertConfigProperties.Parameter.create({
      value: isWizard ? '' : alertDefinition.get('uri.connectionTimeout'),
      threshold: "CRITICAL",
      name: 'connection_timeout',
      label: 'Connection Timeout',
      displayType: 'parameter',
      apiProperty: 'source.uri.connection_timeout',
      units: 'Seconds',
      isValid: function () {
        var value = this.get('value');
        return numericUtils.isPositiveNumber(value);
      }.property('value')
    })]);

    return result;
  },

  /**
   * Render config properties for web-type alert definition
   * @method renderWebConfigs
   * @returns {App.AlertConfigProperty[]}
   */
  renderWebConfigs: function renderWebConfigs() {
    var result = [];
    var alertDefinition = this.get('content');
    var isWizard = this.get('isWizard');

    if (this.get('isWizard')) {
      result = result.concat(this.renderCommonWizardConfigs());
    }

    result = result.concat([App.AlertConfigProperties.Description.create({
      value: isWizard ? '' : alertDefinition.get('description')
    }), App.AlertConfigProperties.Interval.create({
      value: isWizard ? '' : alertDefinition.get('interval')
    }), App.AlertConfigProperties.Thresholds.OkThreshold.create({
      label: 'Thresholds',
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('ok', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('ok', 'value')
    }), App.AlertConfigProperties.Thresholds.WarningThreshold.create({
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('warning', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('warning', 'value')
    }), App.AlertConfigProperties.Thresholds.CriticalThreshold.create({
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('critical', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('critical', 'value')
    }), App.AlertConfigProperties.Parameter.create({
      value: isWizard ? '' : alertDefinition.get('uri.connectionTimeout'),
      threshold: "CRITICAL",
      name: 'connection_timeout',
      label: 'Connection Timeout',
      displayType: 'parameter',
      apiProperty: 'source.uri.connection_timeout',
      units: 'Seconds',
      isValid: function () {
        var value = this.get('value');
        return numericUtils.isPositiveNumber(value);
      }.property('value')
    })]);

    return result;
  },

  /**
   * Render config properties for recovery-type alert definition
   * @method renderRecoveryConfigs
   * @returns {App.AlertConfigProperty[]}
   */
  renderRecoveryConfigs: function renderRecoveryConfigs() {
    var result = [];
    var alertDefinition = this.get('content');
    var isWizard = this.get('isWizard');

    if (this.get('isWizard')) {
      result = result.concat(this.renderCommonWizardConfigs());
    }

    result = result.concat([App.AlertConfigProperties.Description.create({
      value: isWizard ? '' : alertDefinition.get('description')
    }), App.AlertConfigProperties.Interval.create({
      value: isWizard ? '' : alertDefinition.get('interval')
    }), App.AlertConfigProperties.Thresholds.OkThreshold.create({
      label: 'Thresholds',
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('ok', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('ok', 'value')
    }), App.AlertConfigProperties.Thresholds.WarningThreshold.create({
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('warning', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('warning', 'value')
    }), App.AlertConfigProperties.Thresholds.CriticalThreshold.create({
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('critical', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('critical', 'value')
    })]);

    return result;
  },

  renderAmsConfigs: function renderAmsConfigs() {
    var result = [];
    var alertDefinition = this.get('content');
    var isWizard = this.get('isWizard');
    var units = this.get('content.reporting').findProperty('type', 'units') ? this.get('content.reporting').findProperty('type', 'units').get('text') : null;
    if (this.get('isWizard')) {
      result = result.concat(this.renderCommonWizardConfigs());
    }

    result = result.concat([App.AlertConfigProperties.Description.create({
      value: isWizard ? '' : alertDefinition.get('description')
    }), App.AlertConfigProperties.Interval.create({
      value: isWizard ? '' : alertDefinition.get('interval')
    }), App.AlertConfigProperties.Thresholds.OkThreshold.create({
      label: 'Thresholds',
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('ok', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('ok', 'value')
    }), App.AlertConfigProperties.Thresholds.WarningThreshold.create(App.AlertConfigProperties.Thresholds.PercentageMixin, {
      text: isWizard ? '' : this.getThresholdsProperty('warning', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('warning', 'value'),
      valueMetric: units
    }), App.AlertConfigProperties.Thresholds.CriticalThreshold.create(App.AlertConfigProperties.Thresholds.PercentageMixin, {
      text: isWizard ? '' : this.getThresholdsProperty('critical', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('critical', 'value'),
      valueMetric: units
    }), App.AlertConfigProperties.Parameter.create({
      value: isWizard ? '' : alertDefinition.get('uri.connectionTimeout'),
      name: 'connection_timeout',
      label: 'Connection Timeout',
      displayType: 'parameter',
      apiProperty: 'source.uri.connection_timeout',
      units: 'Seconds',
      isValid: function () {
        var value = this.get('value');
        return numericUtils.isPositiveNumber(value);
      }.property('value')
    })]);
    return result;
  },

  /**
   * Render config properties for script-type alert definition
   * @method renderScriptConfigs
   * @returns {App.AlertConfigProperty[]}
   */
  renderScriptConfigs: function renderScriptConfigs() {
    var result = [];
    var alertDefinition = this.get('content');
    var isWizard = this.get('isWizard');

    if (this.get('isWizard')) {
      result = result.concat(this.renderCommonWizardConfigs());
    }

    result = result.concat([App.AlertConfigProperties.Description.create({
      value: isWizard ? '' : alertDefinition.get('description')
    }), App.AlertConfigProperties.Interval.create({
      value: isWizard ? '' : alertDefinition.get('interval')
    })]);

    var mixins = {
      STRING: App.AlertConfigProperties.Parameters.StringMixin,
      NUMERIC: App.AlertConfigProperties.Parameters.NumericMixin,
      PERCENT: App.AlertConfigProperties.Parameters.PercentageMixin
    };
    if (alertDefinition) {
      alertDefinition.get('parameters').forEach(function (parameter) {
        var mixin = mixins[parameter.get('type')] || {}; // validation depends on parameter-type
        result.push(App.AlertConfigProperties.Parameter.create(mixin, {
          value: isWizard ? '' : parameter.get('value'),
          apiProperty: parameter.get('name'),
          description: parameter.get('description'),
          label: isWizard ? '' : parameter.get('displayName'),
          threshold: isWizard ? '' : parameter.get('threshold'),
          units: isWizard ? '' : parameter.get('units'),
          type: isWizard ? '' : parameter.get('type'),
          hidden: parameter.get('visibility') === "HIDDEN",
          readonly: parameter.get('visibility') === "READ_ONLY"
        }));
      });
    }

    return result;
  },

  /**
   * Render config properties for aggregate-type alert definition
   * @method renderAggregateConfigs
   * @returns {App.AlertConfigProperty[]}
   */
  renderAggregateConfigs: function renderAggregateConfigs() {
    var isWizard = this.get('isWizard');
    var alertDefinition = this.get('content');
    var units = this.get('content.reporting') && this.get('content.reporting').findProperty('type', 'units') ? this.get('content.reporting').findProperty('type', 'units').get('text') : null;
    return [App.AlertConfigProperties.Description.create({
      value: isWizard ? '' : alertDefinition.get('description')
    }), App.AlertConfigProperties.Interval.create({
      value: isWizard ? '' : alertDefinition.get('interval')
    }), App.AlertConfigProperties.Thresholds.OkThreshold.create({
      label: 'Thresholds',
      showInputForValue: false,
      text: isWizard ? '' : this.getThresholdsProperty('ok', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('ok', 'value')
    }), App.AlertConfigProperties.Thresholds.WarningThreshold.create(App.AlertConfigProperties.Thresholds.PercentageMixin, {
      text: isWizard ? '' : this.getThresholdsProperty('warning', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('warning', 'value'),
      valueMetric: units
    }), App.AlertConfigProperties.Thresholds.CriticalThreshold.create(App.AlertConfigProperties.Thresholds.PercentageMixin, {
      text: isWizard ? '' : this.getThresholdsProperty('critical', 'text'),
      value: isWizard ? '' : this.getThresholdsProperty('critical', 'value'),
      valueMetric: units
    })];
  },

  /**
   * Render config properties for server-type alert definition
   * @method renderScriptConfigs
   * @returns {App.AlertConfigProperty[]}
   */
  renderServerConfigs: function renderServerConfigs() {
    var result = [];
    var alertDefinition = this.get('content');
    var isWizard = this.get('isWizard');

    if (this.get('isWizard')) {
      result = result.concat(this.renderCommonWizardConfigs());
    }

    result = result.concat([App.AlertConfigProperties.Description.create({
      value: isWizard ? '' : alertDefinition.get('description')
    }), App.AlertConfigProperties.Interval.create({
      value: isWizard ? '' : alertDefinition.get('interval')
    })]);

    var mixins = {
      STRING: App.AlertConfigProperties.Parameters.StringMixin,
      NUMERIC: App.AlertConfigProperties.Parameters.NumericMixin,
      PERCENT: App.AlertConfigProperties.Parameters.PercentageMixin
    };
    alertDefinition.get('parameters').forEach(function (parameter) {
      var mixin = mixins[parameter.get('type')] || {}; // validation depends on parameter-type
      result.push(App.AlertConfigProperties.Parameter.create(mixin, {
        value: isWizard ? '' : parameter.get('value'),
        apiProperty: parameter.get('name'),
        description: parameter.get('description'),
        label: isWizard ? '' : parameter.get('displayName'),
        threshold: isWizard ? '' : parameter.get('threshold'),
        units: isWizard ? '' : parameter.get('units'),
        type: isWizard ? '' : parameter.get('type'),
        hidden: parameter.get('visibility') === "HIDDEN",
        readonly: parameter.get('visibility') === "READ_ONLY"
      }));
    });

    return result;
  },

  /**
   * Render common list of configs used in almost all alert types in wizard
   * @returns {App.AlertConfigProperty[]}
   */
  renderCommonWizardConfigs: function renderCommonWizardConfigs() {
    return [App.AlertConfigProperties.AlertName.create({
      value: ''
    }), App.AlertConfigProperties.Service.create({
      options: this.get('allServices'),
      value: this.get('allServices')[0],
      isShifted: true
    }), App.AlertConfigProperties.Component.create({
      options: this.get('allComponents'),
      value: 'No component',
      isShifted: true
    }), App.AlertConfigProperties.Scope.create({
      options: this.get('allScopes'),
      isShifted: true
    })];
  },

  /**
   * Edit configs button handler
   * @method editConfigs
   */
  editConfigs: function editConfigs() {
    this.get('configs').forEach(function (property) {
      property.set('previousValue', property.get('value'));
      property.set('previousText', property.get('text'));
    });
    this.get('configs').forEach(function (config) {
      config.set('isDisabled', config.get('readonly'));
    });
    this.set('canEdit', true);
  },

  /**
   * Cancel edit configs button handler
   * @method cancelEditConfigs
   */
  cancelEditConfigs: function cancelEditConfigs() {
    this.get('configs').forEach(function (property) {
      property.set('value', property.get('previousValue'));
      property.set('text', property.get('previousText'));
    });
    this.get('configs').setEach('isDisabled', true);
    this.set('canEdit', false);
  },

  /**
   * Save edit configs button handler
   * @method saveConfigs
   * @return {$.ajax}
   */
  saveConfigs: function saveConfigs() {
    this.get('configs').setEach('isDisabled', true);
    this.set('canEdit', false);

    return App.ajax.send({
      name: 'alerts.update_alert_definition',
      sender: this,
      data: {
        id: this.get('content.id'),
        data: this.getPropertiesToUpdate(true)
      }
    });
  },

  /**
   * Create object with new values to put it on server
   * @param {boolean} onlyChanged
   * @method getPropertiesToUpdate
   * @returns {Object}
   */
  getPropertiesToUpdate: function getPropertiesToUpdate(onlyChanged) {
    var propertiesToUpdate = {};
    var configs = onlyChanged ? this.get('configs').filterProperty('wasChanged') : this.get('configs');
    configs = configs.filter(function (c) {
      return c.get('name') !== 'parameter';
    });
    configs.forEach(function (property) {
      var apiProperties = Em.makeArray(property.get('apiProperty'));
      var apiFormattedValues = Em.makeArray(property.get('apiFormattedValue'));
      apiProperties.forEach(function (apiProperty, i) {
        if (apiProperty.contains('source.')) {
          if (!propertiesToUpdate['AlertDefinition/source']) {
            if (this.get('content.rawSourceData')) {
              propertiesToUpdate['AlertDefinition/source'] = this.get('content.rawSourceData');
            }
          }

          if (this.get('content.rawSourceData')) {
            // use rawSourceData to populate propertiesToUpdate
            var sourcePath = propertiesToUpdate['AlertDefinition/source'];
            apiProperty.replace('source.', '').split('.').forEach(function (path, index, array) {
              // check if it is last path
              if (array.length - index === 1) {
                sourcePath[path] = apiFormattedValues[i];
              } else {
                sourcePath = sourcePath[path];
              }
            });
          } else {
            if (!propertiesToUpdate['AlertDefinition/source']) {
              propertiesToUpdate['AlertDefinition/source'] = {};
            }
            Ember.setFullPath(propertiesToUpdate['AlertDefinition/source'], apiProperty.replace('source.', ''), apiFormattedValues[i]);
          }
        } else {
          if (apiProperty) {
            propertiesToUpdate['AlertDefinition/' + apiProperty] = apiFormattedValues[i];
          }
        }
      }, this);
    }, this);

    if (Em.get(propertiesToUpdate, 'AlertDefinition/source.uri.id')) {
      delete propertiesToUpdate['AlertDefinition/source'].uri.id;
    }
    if (Em.get(propertiesToUpdate, 'AlertDefinition/source.ams.id')) {
      delete propertiesToUpdate['AlertDefinition/source'].ams.id;
    }

    // `source.parameters` is an array and should be updated separately from other configs
    if (this.get('content.parameters.length')) {
      propertiesToUpdate['AlertDefinition/source'] = this.get('content.rawSourceData');
      var parameterConfigs = this.get('configs').filterProperty('name', 'parameter');
      parameterConfigs.forEach(function (parameter) {
        propertiesToUpdate['AlertDefinition/source'].parameters.findProperty('name', parameter.get('apiProperty')).value = parameter.get('apiFormattedValue');
      });
    }

    return propertiesToUpdate;
  },

  /**
   * Return array of all config values
   * used to save configs to local db in wizard
   * @method getConfigsValues
   * @returns {Array}
   */
  getConfigsValues: function getConfigsValues() {
    return this.get('configs').map(function (property) {
      return {
        name: property.get('name'),
        value: property.get('value')
      };
    });
  },

  /**
   * Define whether critical threshold >= critical threshold
   * @type {Boolean}
   */
  hasThresholdsError: function () {
    var smallValue, smallValid, largeValue, largeValid;
    if (this.get('configs').findProperty('name', 'warning_threshold')) {
      smallValue = Em.get(this.get('configs').findProperty('name', 'warning_threshold'), 'value');
      smallValid = Em.get(this.get('configs').findProperty('name', 'warning_threshold'), 'isValid');
    }
    if (this.get('configs').findProperty('name', 'critical_threshold')) {
      largeValue = Em.get(this.get('configs').findProperty('name', 'critical_threshold'), 'value');
      largeValid = Em.get(this.get('configs').findProperty('name', 'critical_threshold'), 'isValid');
    }
    return smallValid && largeValid ? Number(smallValue) > Number(largeValue) : false;
  }.property('configs.@each.value'),

  someConfigIsInvalid: Em.computed.someBy('configs', 'isValid', false),

  /**
   * Define whether all configs are valid
   * @type {Boolean}
   */
  hasErrors: Em.computed.or('someConfigIsInvalid', 'hasThresholdsError')

});

});

require.register("controllers/main/alerts/definition_details_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var validator = require('utils/validator');

App.MainAlertDefinitionDetailsController = Em.Controller.extend({

  name: 'mainAlertDefinitionDetailsController',

  alerts: function () {
    return App.AlertInstanceLocal.find().toArray().filterProperty('definitionId', this.get('content.id'));
  }.property('App.router.mainAlertInstancesController.isLoaded', 'App.router.mainAlertInstancesController.reload'),

  // stores object with editing form data (label)
  editing: Em.Object.create({
    label: Em.Object.create({
      name: 'label',
      isEditing: false,
      value: '',
      originalValue: '',
      isError: false,
      bindingValue: 'content.label'
    })
  }),

  /**
   * Host to count of alerts on this host during last day map
   * @type {Object}
   */
  lastDayAlertsCount: null,

  /**
   * List of all group names related to alert definition
   * @type {Array}
   */
  groupsList: Em.computed.mapBy('content.groups', 'displayName'),

  /**
   * Validation function to define if label field populated correctly
   * @method labelValidation
   */
  labelValidation: function () {
    this.set('editing.label.isError', !this.get('editing.label.value').trim());
  }.observes('editing.label.value'),

  /**
   * Set init values for variables
   */
  clearStep: function clearStep() {
    var editing = this.get('editing');
    Em.keys(editing).forEach(function (key) {
      editing.get(key).set('isEditing', false);
    });
  },

  /**
   * Load alert instances for current alertDefinition
   * Start updating loaded data
   * @method loadAlertInstances
   */
  loadAlertInstances: function loadAlertInstances() {
    App.router.get('mainAlertInstancesController').loadAlertInstancesByAlertDefinition(this.get('content.id'));
    App.router.set('mainAlertInstancesController.isUpdating', true);
    this.loadAlertInstancesHistory();
  },

  /**
   * Load alert instances history data
   * used to count instances number of the last 24 hour
   * @method loadAlertInstancesHistory
   */
  loadAlertInstancesHistory: function loadAlertInstancesHistory() {
    this.set('lastDayAlertsCount', null);
    return App.ajax.send({
      name: 'alerts.get_instances_history',
      sender: this,
      data: {
        definitionName: this.get('content.name'),
        timestamp: App.dateTime() - 86400000 // timestamp for time 24-hours ago
      },
      success: 'loadAlertInstancesHistorySuccess'
    });
  },

  /**
   * Success-callback for <code>loadAlertInstancesHistory</code>
   */
  loadAlertInstancesHistorySuccess: function loadAlertInstancesHistorySuccess(data) {
    var lastDayAlertsCount = {};
    data.items.forEach(function (alert) {
      if (!lastDayAlertsCount[alert.AlertHistory.host_name]) {
        lastDayAlertsCount[alert.AlertHistory.host_name] = 0;
      }
      lastDayAlertsCount[alert.AlertHistory.host_name]++;
    });
    this.set('lastDayAlertsCount', lastDayAlertsCount);
  },

  /**
   * Edit button handler
   * @param {object} event
   * @method edit
   */
  edit: function edit(event) {
    var element = event.context;
    var value = this.get(element.get('bindingValue'));
    element.set('originalValue', value);
    element.set('value', value);
    element.set('isEditing', true);
  },

  /**
   * Cancel button handler
   * @param {object} event
   * @method cancelEdit
   */
  cancelEdit: function cancelEdit(event) {
    var element = event.context;
    element.set('value', element.get('originalValue'));
    element.set('isEditing', false);
  },

  /**
   * Save button handler, could save label of alert definition
   * @param {object} event
   * @returns {$.ajax}
   * @method saveEdit
   */
  saveEdit: function saveEdit(event) {
    var element = event.context;
    this.set(element.get('bindingValue'), element.get('value'));
    element.set('isEditing', false);

    var data = Em.Object.create({});
    var propertyName = "AlertDefinition/" + element.get('name');
    data.set(propertyName, element.get('value'));
    var alertDefinitionId = this.get('content.id');
    return App.ajax.send({
      name: 'alerts.update_alert_definition',
      sender: this,
      data: {
        id: alertDefinitionId,
        data: data
      }
    });
  },

  /**
   * Onclick handler for save button on Save popup
   * Save changes of label and configs
   */
  saveLabelAndConfigs: function saveLabelAndConfigs() {
    var configsController = App.router.get('mainAlertDefinitionConfigsController');
    if (configsController.get('canEdit')) {
      configsController.saveConfigs();
    }
    if (this.get('editing.label.isEditing')) {
      this.saveEdit({
        context: this.get('editing.label')
      });
    }
  },

  /**
   * "Delete" button handler
   * @method deleteAlertDefinition
   */
  deleteAlertDefinition: function deleteAlertDefinition() {
    var alertDefinition = this.get('content');
    var self = this;
    App.showConfirmationPopup(function () {
      App.ajax.send({
        name: 'alerts.delete_alert_definition',
        sender: self,
        success: 'deleteAlertDefinitionSuccess',
        error: 'deleteAlertDefinitionError',
        data: {
          id: alertDefinition.get('id')
        }
      });
    }, null, function () {});
  },

  /**
   * Success-callback for <code>deleteAlertDefinition</code>
   * @method deleteAlertDefinitionSuccess
   */
  deleteAlertDefinitionSuccess: function deleteAlertDefinitionSuccess() {
    App.router.transitionTo('main.alerts.index');
  },

  /**
   * Error-callback for <code>deleteAlertDefinition</code>
   * @method deleteAlertDefinitionError
   */
  deleteAlertDefinitionError: function deleteAlertDefinitionError(xhr, textStatus, errorThrown, opt) {
    xhr.responseText = "{\"message\": \"" + xhr.statusText + "\"}";
    App.ajax.defaultErrorHandler(xhr, opt.url, 'DELETE', xhr.status);
  },

  /**
   * "Disable / Enable" button handler
   * @method toggleState
   */
  toggleState: function toggleState() {
    var alertDefinition = this.get('content');
    var self = this;
    var bodyMessage = Em.Object.create({
      confirmMsg: alertDefinition.get('enabled') ? Em.I18n.t('alerts.table.state.enabled.confirm.msg') : Em.I18n.t('alerts.table.state.disabled.confirm.msg'),
      confirmButton: alertDefinition.get('enabled') ? Em.I18n.t('alerts.table.state.enabled.confirm.btn') : Em.I18n.t('alerts.table.state.disabled.confirm.btn')
    });

    return App.showConfirmationFeedBackPopup(function () {
      self.toggleDefinitionState(alertDefinition);
    }, bodyMessage);
  },

  /**
   * Enable/disable alertDefinition
   * @param {object} alertDefinition
   * @param {string} property
   * @returns {$.ajax}
   * @method toggleDefinitionState
   */
  toggleDefinitionState: function toggleDefinitionState(alertDefinition) {
    var newState = !alertDefinition.get('enabled');
    alertDefinition.set('enabled', newState);
    return App.ajax.send({
      name: 'alerts.update_alert_definition',
      sender: this,
      data: {
        id: alertDefinition.get('id'),
        data: {
          "AlertDefinition/enabled": newState
        }
      }
    });
  },

  globalAlertsRepeatTolerance: function () {
    return App.router.get('clusterController.clusterEnv.properties.alerts_repeat_tolerance') || "1";
  }.property('App.router.clusterController.clusterEnv'),

  enableRepeatTolerance: function enableRepeatTolerance(enable) {
    var alertDefinition = this.get('content');
    alertDefinition.set('repeat_tolerance_enabled', enable);
    return App.ajax.send({
      name: 'alerts.update_alert_definition',
      sender: this,
      data: {
        id: alertDefinition.get('id'),
        data: {
          "AlertDefinition/repeat_tolerance_enabled": enable
        }
      }
    });
  },

  editRepeatTolerance: function editRepeatTolerance() {
    var self = this;
    var alertDefinition = this.get('content');

    var alertsRepeatTolerance = App.router.get('clusterController.clusterEnv.properties.alerts_repeat_tolerance') || "1";

    return App.ModalPopup.show({
      classNames: ['forty-percent-width-modal'],
      header: Em.I18n.t('alerts.actions.editRepeatTolerance.header'),
      primary: Em.I18n.t('common.save'),
      secondary: Em.I18n.t('common.cancel'),
      inputValue: self.get('content.repeat_tolerance_enabled') ? self.get('content.repeat_tolerance') || 1 : alertsRepeatTolerance,
      errorMessage: Em.I18n.t('alerts.actions.editRepeatTolerance.error'),
      isInvalid: function () {
        var intValue = Number(this.get('inputValue'));
        return this.get('inputValue') !== 'DEBUG' && (!validator.isValidInt(intValue) || intValue < 1 || intValue > 99);
      }.property('inputValue'),
      isChanged: function () {
        return Number(this.get('inputValue')) !== alertsRepeatTolerance;
      }.property('inputValue'),
      doRestoreDefaultValue: function doRestoreDefaultValue() {
        this.set('inputValue', alertsRepeatTolerance);
        this.$('[data-toggle=tooltip]').tooltip('destroy');
      },
      disablePrimary: Em.computed.alias('isInvalid'),
      onPrimary: function onPrimary() {
        if (this.get('isInvalid')) {
          return;
        }
        var input = this.get('inputValue');
        self.set('content.repeat_tolerance', input);
        self.enableRepeatTolerance(input !== alertsRepeatTolerance);
        App.ajax.send({
          name: 'alerts.update_alert_definition',
          sender: self,
          data: {
            id: alertDefinition.get('id'),
            data: {
              "AlertDefinition/repeat_tolerance": input
            }
          }
        });
        this.hide();
      },
      didInsertElement: function didInsertElement() {
        this._super();
        App.tooltip(this.$('[data-toggle=tooltip]'));
      },
      bodyClass: Ember.View.extend({
        templateName: require('templates/common/modal_popups/prompt_popup'),
        title: Em.I18n.t('alerts.actions.editRepeatTolerance.title'),
        description: Em.I18n.t('alerts.actions.editRepeatTolerance.description'),
        label: Em.I18n.t('alerts.actions.editRepeatTolerance.label')
      })
    });
  },

  /**
   * Define if label or configs are in edit mode
   * @type {Boolean}
   */
  isEditing: Em.computed.or('editing.label.isEditing', 'App.router.mainAlertDefinitionConfigsController.canEdit'),

  /**
   * If some configs or label are changed and user navigates away, show this popup with propose to save changes
   * @param {String} path
   * @method showSavePopup
   */
  showSavePopup: function showSavePopup(callback) {
    var self = this;
    return App.ModalPopup.show({
      header: Em.I18n.t('common.warning'),
      bodyClass: Em.View.extend({
        template: Ember.Handlebars.compile('{{t alerts.saveChanges}}')
      }),
      primary: Em.I18n.t('common.save'),
      secondary: Em.I18n.t('common.discard'),
      third: Em.I18n.t('common.cancel'),
      disablePrimary: Em.computed.or('App.router.mainAlertDefinitionDetailsController.editing.label.isError', 'App.router.mainAlertDefinitionConfigsController.hasErrors'),
      onPrimary: function onPrimary() {
        self.saveLabelAndConfigs();
        callback();
        this.hide();
      },
      onSecondary: function onSecondary() {
        callback();
        this.hide();
      },
      onThird: function onThird() {
        this.hide();
      }
    });
  }

});

});

require.register("controllers/main/alerts/manage_alert_groups_controller", function(exports, require, module) {
'use strict';

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

var validator = require('utils/validator');

function stringify(obj, property) {
  return JSON.stringify(obj.get(property).slice().sort());
}

function groupsAreNotEqual(group1, group2) {
  return stringify(group1, 'definitions') !== stringify(group2, 'definitions') || stringify(group1, 'notifications') !== stringify(group2, 'notifications');
}

function mapToEmObjects(collection, fields, renamedFields) {
  var _renamedFields = arguments.length === 3 ? renamedFields : [];
  return collection.map(function (item) {
    var ret = Em.Object.create(Em.getProperties(item, fields));
    _renamedFields.forEach(function (renamedField) {
      var _renamedField$split = renamedField.split(':'),
          _renamedField$split2 = _slicedToArray(_renamedField$split, 2),
          realName = _renamedField$split2[0],
          newName = _renamedField$split2[1];

      Em.set(ret, newName, Em.get(item, realName));
    });
    return ret;
  });
}

var AlertGroupClone = Em.Object.extend({
  displayName: function () {
    var name = App.config.truncateGroupName(this.get('name'));
    return this.get('default') ? name + ' Default' : name;
  }.property('name', 'default'),
  label: Em.computed.format('{0} ({1})', 'displayName', 'definitions.length')
});

App.ManageAlertGroupsController = Em.Controller.extend({

  name: 'manageAlertGroupsController',

  /**
   * @type {boolean}
   */
  isLoaded: false,

  /**
   * Property used to trigger Alert Groups Filter content updating
   * @type {Boolean}
   */
  changeTrigger: false,

  /**
   * @type {App.AlertGroup[]}
   */
  alertGroups: [],

  /**
   * @type {App.AlertGroup[]}
   */
  originalAlertGroups: [],

  /**
   * @type {App.AlertGroup}
   */
  selectedAlertGroup: null,

  /**
   * @type {App.AlertDefinition[]}
   */
  selectedDefinitions: [],

  /**
   * List of all Alert Notifications
   * @type {App.AlertNotification[]}
   */
  alertNotifications: function () {
    return this.get('isLoaded') ? mapToEmObjects(App.AlertNotification.find(), ['id', 'name', 'description', 'type', 'global']) : [];
  }.property('isLoaded'),

  /**
   * List of all global Alert Notifications
   * @type {App.AlertNotification[]}
   */
  alertGlobalNotifications: Em.computed.filterBy('alertNotifications', 'global', true),

  /**
   * @type {boolean}
   */
  isRemoveButtonDisabled: true,

  /**
   * @type {boolean}
   */
  isRenameButtonDisabled: true,

  /**
   * @type {boolean}
   */
  isDuplicateButtonDisabled: true,

  /**
   * @type {boolean}
   */
  isDeleteDefinitionsDisabled: function () {
    var selectedGroup = this.get('selectedAlertGroup');
    return selectedGroup ? selectedGroup.default || this.get('selectedDefinitions').length === 0 : true;
  }.property('selectedAlertGroup.definitions.length', 'selectedDefinitions.length'),

  /**
   * observes if any group changed including: group name, newly created group, deleted group, group with definitions/notifications changed
   * @type {{toDelete: App.AlertGroup[], toSet: App.AlertGroup[], toCreate: App.AlertGroup[]}}
   */
  defsModifiedAlertGroups: {},

  /**
   * Determines if some group was edited/created/deleted
   * @type {boolean}
   */
  isDefsModified: Em.computed.and('isLoaded', 'isDefsModifiedAlertGroups'),

  isDefsModifiedAlertGroups: Em.computed.or('defsModifiedAlertGroups.toSet.length', 'defsModifiedAlertGroups.toCreate.length', 'defsModifiedAlertGroups.toDelete.length'),

  /**
   * Check when some config group was changed and updates <code>defsModifiedAlertGroups</code> once
   * @method defsModifiedAlertGroupsObs
   */
  defsModifiedAlertGroupsObs: function () {
    Em.run.once(this, this.defsModifiedAlertGroupsObsOnce);
  }.observes('selectedAlertGroup.definitions.[]', 'selectedAlertGroup.notifications.[]', 'alertGroups', 'isLoaded'),

  /**
   * Update <code>defsModifiedAlertGroups</code>-value
   * Called once in the <code>defsModifiedAlertGroupsObs</code>
   * @method defsModifiedAlertGroupsObsOnce
   * @returns {boolean}
   */
  defsModifiedAlertGroupsObsOnce: function defsModifiedAlertGroupsObsOnce() {
    if (!this.get('isLoaded')) {
      return;
    }
    var groupsToDelete = [];
    var groupsToSet = [];
    var groupsToCreate = [];
    var groups = this.get('alertGroups'); //current alert groups
    var originalGroups = this.get('originalAlertGroups'); // original alert groups
    var mappedOriginalGroups = {}; // map is faster than `originalGroups.findProperty('id', ...)`
    originalGroups.forEach(function (group) {
      mappedOriginalGroups[group.get('id')] = group;
    });
    var originalGroupsIds = originalGroups.mapProperty('id');

    groups.forEach(function (group) {
      var originalGroup = mappedOriginalGroups[group.get('id')];
      if (originalGroup) {
        // should update definitions or notifications
        if (groupsAreNotEqual(group, originalGroup) || group.get('name') !== originalGroup.get('name')) {
          // should update name
          groupsToSet.push(group.set('id', originalGroup.get('id')));
        }
        originalGroupsIds = originalGroupsIds.without(group.get('id'));
      } else {
        // should add new group
        groupsToCreate.push(group);
      }
    });
    // should delete groups
    originalGroupsIds.forEach(function (id) {
      groupsToDelete.push(originalGroups.findProperty('id', id));
    });

    this.set('defsModifiedAlertGroups', {
      toDelete: groupsToDelete,
      toSet: groupsToSet,
      toCreate: groupsToCreate
    });
  },

  /**
   * Load all Alert Notifications from server
   * @returns {$.ajax}
   * @method loadAlertNotifications
   */
  loadAlertNotifications: function loadAlertNotifications() {
    this.setProperties({
      isLoaded: false,
      alertGroups: [],
      originalAlertGroups: [],
      selectedAlertGroup: null,
      isRemoveButtonDisabled: true,
      isRenameButtonDisabled: true,
      isDuplicateButtonDisabled: true
    });
    return App.ajax.send({
      name: 'alerts.notifications',
      sender: this,
      success: 'getAlertNotificationsSuccessCallback',
      error: 'getAlertNotificationsErrorCallback'
    });
  },

  /**
   * Success-callback for load alert notifications request
   * @param {object} json
   * @method getAlertNotificationsSuccessCallback
   */
  getAlertNotificationsSuccessCallback: function getAlertNotificationsSuccessCallback(json) {
    App.alertNotificationMapper.map(json);
    this.loadAlertGroups();
  },

  /**
   * Error-callback for load alert notifications request
   * @method getAlertNotificationsErrorCallback
   */
  getAlertNotificationsErrorCallback: function getAlertNotificationsErrorCallback() {
    this.set('isLoaded', true);
  },

  /**
   * Load all alert groups from alert group model
   * @method loadAlertGroups
   */
  loadAlertGroups: function loadAlertGroups() {
    var alertGroups = App.AlertGroup.find().map(function (group) {
      var definitions = mapToEmObjects(group.get('definitions'), ['name', 'serviceName', 'componentName', 'label', 'id'], ['service.displayName:serviceNameDisplay', 'componentNameFormatted:componentNameDisplay']);

      var targets = mapToEmObjects(group.get('targets'), ['name', 'id', 'description', 'type', 'global']);

      var hash = Em.getProperties(group, ['id', 'name', 'default', 'isAddDefinitionsDisabled']);
      hash.definitions = definitions;
      hash.notifications = targets;
      return AlertGroupClone.create(hash);
    });
    this.setProperties({
      alertGroups: alertGroups,
      isLoaded: true,
      originalAlertGroups: this.copyAlertGroups(alertGroups),
      selectedAlertGroup: alertGroups[0]
    });
  },

  /**
   * Enable/disable "Remove"/"Rename"/"Duplicate" buttons basing on <code>controller.selectedAlertGroup</code>
   * @method buttonObserver
   */
  buttonObserver: function () {
    var selectedAlertGroup = this.get('selectedAlertGroup');
    var flag = selectedAlertGroup && selectedAlertGroup.get('default');
    this.setProperties({
      isRemoveButtonDisabled: flag,
      isRenameButtonDisabled: flag,
      isDuplicateButtonDisabled: false
    });
  }.observes('selectedAlertGroup'),

  /**
   * @method resortAlertGroup
   */
  resortAlertGroup: function () {
    var alertGroups = Em.copy(this.get('alertGroups'));
    if (alertGroups.length < 2) {
      return;
    }
    var defaultGroups = alertGroups.filterProperty('default');
    defaultGroups.forEach(function (defaultGroup) {
      alertGroups.removeObject(defaultGroup);
    });
    var sorted = defaultGroups.sortProperty('name').concat(alertGroups.sortProperty('name'));

    this.removeObserver('alertGroups.@each.name', this, 'resortAlertGroup');
    this.set('alertGroups', sorted);
    this.addObserver('alertGroups.@each.name', this, 'resortAlertGroup');
  }.observes('alertGroups.@each.name'),

  /**
   * remove definitions from group
   * @method deleteDefinitions
   */
  deleteDefinitions: function deleteDefinitions() {
    if (this.get('isDeleteDefinitionsDisabled')) {
      return;
    }
    var groupDefinitions = this.get('selectedAlertGroup.definitions');
    this.get('selectedDefinitions').slice().forEach(function (defObj) {
      groupDefinitions.removeObject(defObj);
    }, this);
    this.set('selectedDefinitions', []);
  },

  /**
   * Provides alert definitions which are available for inclusion in
   * non-default alert groups.
   * @param {App.AlertGroup} selectedAlertGroup
   * @method getAvailableDefinitions
   * @return {{name: string, serviceName: string, componentName: string, serviceNameDisplay: string, componentNameDisplay: string, label: string, id: number}[]}
   */
  getAvailableDefinitions: function getAvailableDefinitions(selectedAlertGroup) {
    if (selectedAlertGroup.get('default')) return [];
    var usedDefinitionsMap = {};
    var availableDefinitions = [];
    var sharedDefinitions = App.AlertDefinition.find();

    usedDefinitionsMap = selectedAlertGroup.get('definitions').toWickMapByProperty('name');

    selectedAlertGroup.get('definitions').forEach(function (def) {
      usedDefinitionsMap[def.name] = true;
    });
    sharedDefinitions.forEach(function (sharedDef) {
      if (!usedDefinitionsMap[sharedDef.get('name')]) {
        availableDefinitions.pushObject(sharedDef);
      }
    });
    return mapToEmObjects(availableDefinitions, ['name', 'serviceName', 'componentName', 'label', 'id'], ['service.displayName:serviceNameDisplay', 'componentNameFormatted:componentNameDisplay']);
  },

  /**
   * add alert definitions to a group
   * @method addDefinitions
   */
  addDefinitions: function addDefinitions() {
    if (this.get('selectedAlertGroup.isAddDefinitionsDisabled')) {
      return false;
    }
    var availableDefinitions = this.getAvailableDefinitions(this.get('selectedAlertGroup'));
    var validComponents = App.StackServiceComponent.find().map(function (component) {
      return Em.Object.create({
        componentName: component.get('componentName'),
        displayName: App.format.role(component.get('componentName'), false),
        selected: false
      });
    });
    var validServices = App.Service.find().map(function (service) {
      return Em.Object.create({
        serviceName: service.get('serviceName'),
        displayName: App.format.role(service.get('serviceName'), true),
        selected: false
      });
    });
    return this.launchDefsSelectionDialog(availableDefinitions, [], validServices, validComponents);
  },

  /**
   * Launch a table view of all available definitions to choose
   * @method launchDefsSelectionDialog
   * @return {App.ModalPopup}
   */
  launchDefsSelectionDialog: function launchDefsSelectionDialog(initialDefs, selectedDefs, validServices, validComponents) {
    var self = this;
    return App.ModalPopup.show({

      classNames: ['common-modal-wrapper', 'full-height-modal'],

      modalDialogClasses: ['modal-lg'],

      header: Em.I18n.t('alerts.actions.manage_alert_groups_popup.selectDefsDialog.title'),

      /**
       * @type {string}
       */
      dialogMessage: Em.I18n.t('alerts.actions.manage_alert_groups_popup.selectDefsDialog.message').format(this.get('selectedAlertGroup.displayName')),

      /**
       * @type {string|null}
       */
      warningMessage: null,

      /**
       * @type {App.AlertDefinition[]}
       */
      availableDefs: [],

      onPrimary: function onPrimary() {
        this.set('warningMessage', null);
        var arrayOfSelectedDefs = this.get('availableDefs').filterProperty('selected', true);
        if (arrayOfSelectedDefs.length < 1) {
          this.set('warningMessage', Em.I18n.t('alerts.actions.manage_alert_groups_popup.selectDefsDialog.message.warning'));
          return;
        }
        self.addDefinitionsCallback(arrayOfSelectedDefs);
        this.hide();
      },

      /**
       * Primary button should be disabled while alert definitions are not loaded
       * @type {boolean}
       */
      disablePrimary: Em.computed.not('isLoaded'),

      onSecondary: function onSecondary() {
        self.addDefinitionsCallback(null);
        this.hide();
      },

      bodyClass: App.SelectDefinitionsPopupBodyView.extend({

        filterComponents: validComponents,

        filterServices: validServices,

        initialDefs: initialDefs

      })

    });
  },

  /**
   * add alert definitions callback
   * @method addDefinitionsCallback
   */
  addDefinitionsCallback: function addDefinitionsCallback(selectedDefs) {
    var group = this.get('selectedAlertGroup');
    if (selectedDefs) {
      group.get('definitions').pushObjects(selectedDefs);
    }
  },

  /**
   * copy alert groups for backup, to compare with current alert groups, so will know if some groups changed/added/deleted
   * @param {App.AlertGroup[]} originGroups
   * @return {App.AlertGroup[]}
   * @method copyAlertGroups
   */
  copyAlertGroups: function copyAlertGroups(originGroups) {
    var alertGroups = [];
    originGroups.forEach(function (alertGroup) {
      var copiedGroup = Em.Object.create($.extend(true, {}, alertGroup));
      alertGroups.pushObject(copiedGroup);
    });
    return alertGroups;
  },

  /**
   * Create a new alert group
   * @param {Em.Object} newAlertGroupData
   * @param {callback} callback Callback function for Success or Error handling
   * @return {App.AlertGroup} Returns the created alert group
   * @method postNewAlertGroup
   */
  postNewAlertGroup: function postNewAlertGroup(newAlertGroupData, callback) {
    // create a new group with name , definition and notifications
    var data = {
      name: newAlertGroupData.get('name')
    };
    if (newAlertGroupData.get('definitions').length > 0) {
      data.definitions = newAlertGroupData.get('definitions').mapProperty('id');
    }
    if (newAlertGroupData.get('notifications').length > 0) {
      data.targets = newAlertGroupData.get('notifications').mapProperty('id');
    }
    var sendData = {
      name: 'alert_groups.create',
      data: data,
      success: 'successFunction',
      error: 'errorFunction',
      successFunction: function successFunction() {
        if (callback) {
          callback();
        }
      },
      errorFunction: function errorFunction(xhr, text, errorThrown) {
        if (callback) {
          callback(xhr, text, errorThrown);
        }
      }
    };
    sendData.sender = sendData;
    App.ajax.send(sendData);
    return newAlertGroupData;
  },

  /**
   * PUTs the new alert group information on the server.
   * Changes possible here are the name, definitions, notifications
   *
   * @param {App.AlertGroup} alertGroup
   * @param {Function} successCallback
   * @param {Function} errorCallback
   * @method updateAlertGroup
   */
  updateAlertGroup: function updateAlertGroup(alertGroup, successCallback, errorCallback) {
    var sendData = {
      name: 'alert_groups.update',
      data: {
        group_id: alertGroup.id,
        name: alertGroup.get('name'),
        definitions: alertGroup.get('definitions').mapProperty('id'),
        targets: alertGroup.get('notifications').mapProperty('id')
      },
      success: 'successFunction',
      error: 'errorFunction',
      successFunction: function successFunction() {
        if (successCallback) {
          successCallback();
        }
      },
      errorFunction: function errorFunction(xhr, text, errorThrown) {
        if (errorCallback) {
          errorCallback(xhr, text, errorThrown);
        }
      }
    };
    sendData.sender = sendData;
    App.ajax.send(sendData);
  },

  /**
   * Request for deleting alert group
   * @param {App.AlertGroup} alertGroup
   * @param {callback} successCallback
   * @param {callback} errorCallback
   * @method removeAlertGroup
   */
  removeAlertGroup: function removeAlertGroup(alertGroup, successCallback, errorCallback) {
    var sendData = {
      name: 'alert_groups.delete',
      data: {
        group_id: alertGroup.id
      },
      success: 'successFunction',
      error: 'errorFunction',
      successFunction: function successFunction() {
        if (successCallback) {
          successCallback();
        }
      },
      errorFunction: function errorFunction(xhr, text, errorThrown) {
        if (errorCallback) {
          errorCallback(xhr, text, errorThrown);
        }
      }
    };
    sendData.sender = sendData;
    App.ajax.send(sendData);
  },

  /**
   * confirm delete alert group
   * @method confirmDelete
   */
  confirmDelete: function confirmDelete() {
    if (this.get('isRemoveButtonDisabled')) return;
    var self = this;
    App.showConfirmationPopup(function () {
      self.deleteAlertGroup();
    });
  },

  /**
   * delete selected alert group
   * @method deleteAlertGroup
   */
  deleteAlertGroup: function deleteAlertGroup() {
    var selectedAlertGroup = this.get('selectedAlertGroup');
    if (this.get('isDeleteAlertDisabled')) {
      return;
    }
    this.get('alertGroups').removeObject(selectedAlertGroup);
    this.set('selectedAlertGroup', this.get('alertGroups')[0]);
  },

  /**
   * Rename non-default alert group
   * @method renameAlertGroup
   */
  renameAlertGroup: function renameAlertGroup() {

    if (this.get('selectedAlertGroup.default')) {
      return;
    }
    var self = this;
    var popup = App.ModalPopup.show({

      header: Em.I18n.t('alerts.actions.manage_alert_groups_popup.renameButton'),

      bodyClass: Ember.View.extend({
        templateName: require('templates/main/alerts/create_new_alert_group')
      }),

      /**
       * @type {string}
       */
      alertGroupName: self.get('selectedAlertGroup.name'),

      alertGroupNameIsEmpty: function () {
        return !this.get('alertGroupName').trim().length;
      }.property('alertGroupName'),

      /**
       * @type {string|null}
       */
      warningMessage: function () {
        var originalGroup = self.get('selectedAlertGroup');
        var groupName = this.get('alertGroupName').trim();

        if (originalGroup.get('name').trim() === groupName) {
          return Em.I18n.t("alerts.actions.manage_alert_groups_popup.addGroup.exist");
        }
        if (self.get('alertGroups').mapProperty('displayName').contains(groupName)) {
          return Em.I18n.t("alerts.actions.manage_alert_groups_popup.addGroup.exist");
        }
        if (groupName && !validator.isValidAlertGroupName(groupName)) {
          return Em.I18n.t("form.validator.alertGroupName");
        }
        return '';
      }.property('alertGroupName'),

      /**
       * Primary button is disabled while user doesn't input valid group name
       * @type {boolean}
       */
      disablePrimary: Em.computed.or('alertGroupNameIsEmpty', 'warningMessage'),

      onPrimary: function onPrimary() {
        self.set('selectedAlertGroup.name', this.get('alertGroupName'));
        this._super();
      }

    });
    this.set('renameGroupPopup', popup);
  },

  /**
   * Create new alert group
   * @param {boolean} duplicated is new group a copy of the existing group
   * @method addAlertGroup
   */
  addAlertGroup: function addAlertGroup(duplicated) {
    duplicated = duplicated === true;
    var self = this;
    var popup = App.ModalPopup.show({

      header: Em.I18n.t('alerts.actions.manage_alert_groups_popup.addButton'),

      bodyClass: Em.View.extend({
        templateName: require('templates/main/alerts/create_new_alert_group')
      }),

      /**
       * Name for new alert group
       * @type {string}
       */
      alertGroupName: duplicated ? self.get('selectedAlertGroup.name') + ' Copy' : "",

      alertGroupNameIsEmpty: function () {
        return !this.get('alertGroupName').trim().length;
      }.property('alertGroupName'),

      /**
       * @type {string}
       */
      warningMessage: function () {
        var groupName = this.get('alertGroupName').trim();
        if (self.get('alertGroups').mapProperty('displayName').contains(groupName)) {
          return Em.I18n.t("alerts.actions.manage_alert_groups_popup.addGroup.exist");
        }
        if (groupName && !validator.isValidAlertGroupName(groupName)) {
          return Em.I18n.t("form.validator.alertGroupName");
        }
        return '';
      }.property('alertGroupName'),

      /**
       * Primary button is disabled while user doesn't input valid group name
       * @type {boolean}
       */
      disablePrimary: Em.computed.or('alertGroupNameIsEmpty', 'warningMessage'),

      onPrimary: function onPrimary() {
        var newAlertGroup = AlertGroupClone.create({
          name: this.get('alertGroupName').trim(),
          default: false,
          definitions: duplicated ? self.get('selectedAlertGroup.definitions').slice(0) : [],
          notifications: self.get('alertGlobalNotifications'),
          isAddDefinitionsDisabled: false
        });
        self.get('alertGroups').pushObject(newAlertGroup);
        self.set('selectedAlertGroup', newAlertGroup);
        this.hide();
      }

    });
    this.set('addGroupPopup', popup);
  },

  /**
   * @method duplicateAlertGroup
   */
  duplicateAlertGroup: function duplicateAlertGroup() {
    this.addAlertGroup(true);
  }

});

});

require.register("controllers/main/alerts/manage_alert_notifications_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var validator = require('utils/validator');

App.ManageAlertNotificationsController = Em.Controller.extend({

  name: 'manageAlertNotificationsController',

  /**
   * Are alert notifications loaded
   * @type {boolean}
   */
  isLoaded: false,

  /**
   * Create/Edit modal popup object
   * used to hide popup
   * @type {App.ModalPopup}
   */
  createEditPopup: null,

  /**
   * Map of edit inputs shown in Create/Edit Notification popup
   * @type {Object}
   */
  inputFields: Em.Object.create({
    name: {
      label: Em.I18n.t('common.name'),
      value: '',
      defaultValue: ''
    },
    groups: {
      label: Em.I18n.t('common.groups'),
      value: [],
      defaultValue: []
    },
    global: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.global'),
      value: false,
      defaultValue: false,
      disabled: false
    },
    allGroups: Em.Object.create({
      value: '',
      defaultValue: 'custom',
      disabled: false,
      isAll: Em.computed.equal('value', 'all')
    }),
    method: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.method'),
      value: '',
      defaultValue: ''
    },
    email: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.email'),
      value: '',
      defaultValue: ''
    },
    SMTPServer: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.SMTPServer'),
      value: '',
      defaultValue: ''
    },
    SMTPPort: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.SMTPPort'),
      value: '',
      defaultValue: ''
    },
    SMTPUseAuthentication: Em.Object.create({
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.SMTPUseAuthentication'),
      value: false,
      defaultValue: false,
      invertedValue: Em.computed.not('value')
    }),
    SMTPUsername: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.SMTPUsername'),
      value: '',
      defaultValue: ''
    },
    SMTPPassword: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.SMTPPassword'),
      value: '',
      defaultValue: ''
    },
    retypeSMTPPassword: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.retypeSMTPPassword'),
      value: '',
      defaultValue: ''
    },
    SMTPSTARTTLS: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.SMTPSTARTTLS'),
      value: false,
      defaultValue: false
    },
    emailFrom: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.emailFrom'),
      value: '',
      defaultValue: ''
    },
    version: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.version'),
      value: '',
      defaultValue: ''
    },
    OIDs: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.OIDs'),
      value: '',
      defaultValue: ''
    },
    community: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.community'),
      value: '',
      defaultValue: ''
    },
    host: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.host'),
      value: '',
      defaultValue: ''
    },
    port: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.port'),
      value: '',
      defaultValue: ''
    },
    severityFilter: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.severityFilter'),
      value: [],
      defaultValue: []
    },
    description: {
      label: Em.I18n.t('common.description'),
      value: '',
      defaultValue: ''
    },
    scriptDispatchProperty: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.scriptDispatchProperty'),
      value: '',
      defaultValue: ''
    },
    scriptFileName: {
      label: Em.I18n.t('alerts.actions.manage_alert_notifications_popup.scriptFileName'),
      value: '',
      defaultValue: ''
    },
    customProperties: Em.A([])
  }),

  /**
   * List of available options for Enable or Disable
   * used in settings of SelectedAlertNotification
   * @type {Object}
   */
  enableOrDisable: {
    enable: "enable",
    disable: "disable"
  },

  /**
   * List of available Notification types
   * used in Type combobox
   * @type {Array}
   */
  methods: ['EMAIL', 'SNMP', 'Custom SNMP', 'Alert Script'],

  /**
   * List of available value for Severity Filter
   * used in Severity Filter combobox
   * @type {Array}
   */
  severities: ['OK', 'WARNING', 'CRITICAL', 'UNKNOWN'],

  /**
   * List of available SNMP versions
   * @type {Array}
   */
  SNMPVersions: ['SNMPv1', 'SNMPv2c'],

  /**
   * List of all Alert Notifications
   * @type {App.AlertNotification[]}
   */
  alertNotifications: function () {
    return this.get('isLoaded') ? App.AlertNotification.find().toArray() : [];
  }.property('isLoaded'),

  /**
   * List of all Alert Groups
   * @type {App.AlertGroup[]}
   */
  allAlertGroups: function () {
    return this.get('isLoaded') ? App.AlertGroup.find().toArray() : [];
  }.property('isLoaded'),

  /**
   * Selected Alert Notification
   * @type {App.AlertNotification}
   */
  selectedAlertNotification: null,

  /**
   * Addable to <code>selectedAlertNotification.properties</code> custom property
   * @type {{name: string, value: string}}
   */
  newCustomProperty: { name: '', value: '' },

  /**
   * @type {string[]}
   */
  customPropertyNames: Em.computed.mapBy('inputFields.customProperties', 'name'),

  /**
   * Check if custom property name exists in the <code>inputFields.customProperties</code>
   *
   * @type {boolean}
   */
  isNewCustomPropertyExists: Em.computed.existsInByKey('newCustomProperty.name', 'customPropertyNames'),

  /**
   * Check if custom property name exists in the <code>ignoredCustomProperties</code>
   *
   * @type {boolean}
   */
  isNewCustomPropertyIgnored: Em.computed.existsInByKey('newCustomProperty.name', 'ignoredCustomProperties'),

  /**
   * Check if custom property name is valid according to the <code>validator.isValidConfigKey</code>
   *
   * @type {boolean}
   */
  isNewCustomPropertyNameValid: function () {
    return validator.isValidConfigKey(this.get('newCustomProperty.name'));
  }.property('newCustomProperty.name'),

  /**
   * Error message for the new custom property name
   *
   * @type {string}
   */
  errorMessageForNewCustomPropertyName: function () {
    var isNewCustomPropertyIgnored = this.get('isNewCustomPropertyIgnored');
    var isNewCustomPropertyExists = this.get('isNewCustomPropertyExists');
    var flag = this.get('isNewCustomPropertyNameValid');
    if (flag) {
      if (isNewCustomPropertyExists || isNewCustomPropertyIgnored) {
        return Em.I18n.t('alerts.notifications.addCustomPropertyPopup.error.propertyExists');
      }
    } else {
      return Em.I18n.t('alerts.notifications.addCustomPropertyPopup.error.invalidPropertyName');
    }
    return '';
  }.property('isNewCustomPropertyNameValid', 'isNewCustomPropertyExists', 'isNewCustomPropertyIgnored'),

  /**
   * If some error with new custom property
   *
   * @type {boolean}
   */
  isErrorWithNewCustomPropertyName: Em.computed.bool('errorMessageForNewCustomPropertyName'),

  /**
   * List custom property names that shouldn't be displayed on Edit page
   * @type {string[]}
   */
  ignoredCustomProperties: ['ambari.dispatch.credential.password', 'ambari.dispatch.credential.username', 'ambari.dispatch.recipients', 'ambari.dispatch.snmp.community', 'ambari.dispatch.snmp.oids.trap', 'ambari.dispatch.snmp.oids.subject', 'ambari.dispatch.snmp.oids.body', 'ambari.dispatch.snmp.port', 'ambari.dispatch.snmp.version', 'mail.smtp.auth', 'mail.smtp.from', 'mail.smtp.host', 'mail.smtp.port', 'mail.smtp.starttls.enable', 'ambari.dispatch-property.script', 'ambari.dispatch-property.script.filename'],

  validationMap: {
    EMAIL: [{
      errorKey: 'emailToError',
      validator: 'emailToValidation'
    }, {
      errorKey: 'emailFromError',
      validator: 'emailFromValidation'
    }, {
      errorKey: 'smtpPortError',
      validator: 'smtpPortValidation'
    }, {
      errorKey: 'smtpUsernameError',
      validator: 'smtpUsernameValidation'
    }, {
      errorKey: 'smtpPasswordError',
      validator: 'smtpPasswordValidation'
    }, {
      errorKey: 'passwordError',
      validator: 'retypePasswordValidation'
    }],
    SNMP: [{
      errorKey: 'portError',
      validator: 'portValidation'
    }, {
      errorKey: 'hostError',
      validator: 'hostsValidation'
    }],
    CustomSNMP: [{
      errorKey: 'portError',
      validator: 'portValidation'
    }, {
      errorKey: 'hostError',
      validator: 'hostsValidation'
    }],
    AlertScript: [{
      errorKey: 'scriptFileNameError',
      validator: 'scriptFileNameValidation'
    }]
  },

  /**
   * Load all Alert Notifications from server
   * @method loadAlertNotifications
   */
  loadAlertNotifications: function loadAlertNotifications() {
    this.set('isLoaded', false);
    return App.ajax.send({
      name: 'alerts.notifications',
      sender: this,
      success: 'getAlertNotificationsSuccessCallback',
      error: 'getAlertNotificationsErrorCallback'
    });
  },

  /**
   * Success-callback for load alert notifications request
   * @param {object} json
   * @method getAlertNotificationsSuccessCallback
   */
  getAlertNotificationsSuccessCallback: function getAlertNotificationsSuccessCallback(json) {
    App.alertNotificationMapper.map(json);
    this.set('isLoaded', true);
  },

  /**
   * Error-callback for load alert notifications request
   * @method getAlertNotificationsErrorCallback
   */
  getAlertNotificationsErrorCallback: function getAlertNotificationsErrorCallback() {
    this.set('isLoaded', true);
  },

  /**
   * Add Notification button handler
   * @method addAlertNotification
   */
  addAlertNotification: function addAlertNotification() {
    var inputFields = this.get('inputFields');
    inputFields.setProperties({
      'global.disabled': false
    });
    Em.keys(inputFields).forEach(function (key) {
      inputFields.set(key + '.value', inputFields.get(key + '.defaultValue'));
    });
    inputFields.set('severityFilter.value', ['OK', 'WARNING', 'CRITICAL', 'UNKNOWN']);
    inputFields.set('customProperties', Em.A([]));
    this.showCreateEditPopup(false);
  },

  /**
   * Edit Notification button handler
   * @method editAlertNotification
   */
  editAlertNotification: function editAlertNotification() {
    this.fillEditCreateInputs();
    this.showCreateEditPopup(true);
  },

  /**
   * Fill inputs of Create/Edit popup form
   * @param addCopyToName define whether add 'Copy of ' to name
   * @method fillEditCreateInputs
   */
  fillEditCreateInputs: function fillEditCreateInputs(addCopyToName) {
    var inputFields = this.get('inputFields');
    var selectedAlertNotification = this.get('selectedAlertNotification');
    var methodValue = this.getNotificationTypeText(selectedAlertNotification.get('type'));
    var properties = selectedAlertNotification.get('properties');
    inputFields.set('name.value', (addCopyToName ? 'Copy of ' : '') + selectedAlertNotification.get('name'));
    inputFields.set('groups.value', selectedAlertNotification.get('groups').toArray());
    inputFields.set('email.value', properties['ambari.dispatch.recipients'] ? properties['ambari.dispatch.recipients'].join(', ') : '');
    inputFields.set('SMTPServer.value', properties['mail.smtp.host']);
    inputFields.set('SMTPPort.value', properties['mail.smtp.port']);
    inputFields.set('SMTPUseAuthentication.value', properties['mail.smtp.auth'] !== "false");
    inputFields.set('SMTPUsername.value', properties['ambari.dispatch.credential.username']);
    inputFields.set('SMTPPassword.value', properties['ambari.dispatch.credential.password']);
    inputFields.set('retypeSMTPPassword.value', properties['ambari.dispatch.credential.password']);
    inputFields.set('SMTPSTARTTLS.value', properties['mail.smtp.starttls.enable'] !== "false");
    inputFields.set('emailFrom.value', properties['mail.smtp.from']);
    inputFields.set('version.value', properties['ambari.dispatch.snmp.version']);
    inputFields.set('OIDs.value', properties['ambari.dispatch.snmp.oids.trap']);
    inputFields.set('community.value', properties['ambari.dispatch.snmp.community']);
    inputFields.set('host.value', properties['ambari.dispatch.recipients'] ? properties['ambari.dispatch.recipients'].join(', ') : '');
    inputFields.set('port.value', properties['ambari.dispatch.snmp.port']);
    inputFields.set('severityFilter.value', selectedAlertNotification.get('alertStates'));
    inputFields.set('global.value', selectedAlertNotification.get('global'));
    inputFields.set('allGroups.value', selectedAlertNotification.get('global') ? 'all' : 'custom');
    inputFields.set('scriptDispatchProperty.value', properties['ambari.dispatch-property.script'] || '');
    inputFields.set('scriptFileName.value', properties['ambari.dispatch-property.script.filename'] || '');
    // not allow to edit global field
    inputFields.set('global.disabled', true);
    inputFields.set('description.value', selectedAlertNotification.get('description'));
    inputFields.set('method.value', methodValue);
    inputFields.get('customProperties').clear();
    var ignoredCustomProperties = this.get('ignoredCustomProperties');
    Em.keys(properties).forEach(function (k) {
      if (ignoredCustomProperties.contains(k)) return;
      inputFields.get('customProperties').pushObject({
        name: k,
        value: properties[k],
        defaultValue: properties[k]
      });
    });
  },

  /**
   * Show Edit or Create Notification popup
   * @param {boolean} isEdit true - edit, false - create
   * @returns {App.ModalPopup}
   * @method showCreateEditPopup
   */
  showCreateEditPopup: function showCreateEditPopup(isEdit) {
    var self = this;
    var createEditPopup = App.ModalPopup.show({
      header: isEdit ? Em.I18n.t('alerts.actions.manage_alert_notifications_popup.editHeader') : Em.I18n.t('alerts.actions.manage_alert_notifications_popup.addHeader'),
      classNames: ['create-edit-alert-notification-popup'],
      marginBottom: 130,
      bodyClass: Em.View.extend({
        controller: this,
        templateName: require('templates/main/alerts/create_alert_notification'),

        didInsertElement: function didInsertElement() {
          App.tooltip($('.checkbox-tooltip'));
          this.nameValidation();
          this.emailToValidation();
          this.emailFromValidation();
          this.smtpPortValidation();
          this.hostsValidation();
          this.portValidation();
          this.smtpUsernameValidation();
          this.smtpPasswordValidation();
          this.retypePasswordValidation();
          this.scriptFileNameValidation();
        },

        isEmailMethodSelected: Em.computed.equal('controller.inputFields.method.value', 'EMAIL'),

        isSNMPMethodSelected: Em.computed.equal('controller.inputFields.method.value', 'SNMP'),

        isCustomSNMPMethodSelected: Em.computed.equal('controller.inputFields.method.value', 'Custom SNMP'),

        isAlertScriptMethodSelected: Em.computed.equal('controller.inputFields.method.value', 'Alert Script'),

        methodObserver: function () {
          var currentMethod = this.get('controller.inputFields.method.value'),
              validationMap = self.get('validationMap');
          self.get('methods').forEach(function (method) {
            // Replace blank spaces with empty character
            var mapKey = method.replace(/\s/g, "");
            var validations = validationMap[mapKey];
            if (method === currentMethod) {
              validations.mapProperty('validator').forEach(function (key) {
                this.get(key).call(this);
              }, this);
            } else {
              validations.mapProperty('errorKey').forEach(function (key) {
                this.set(key, false);
              }, this);
            }
          }, this);
        }.observes('controller.inputFields.method.value'),

        nameValidation: function () {
          var newName = this.get('controller.inputFields.name.value').trim();
          var errorMessage = '';
          // on editing, save current notification name
          if (newName && !this.get('currentName')) {
            this.set('currentName', newName);
          }
          var nameExistsCondition = isEdit ? newName && newName !== this.get('currentName') : !!newName;
          if (!newName) {
            this.set('nameError', true);
            errorMessage = Em.I18n.t('alerts.actions.manage_alert_notifications_popup.error.name.empty');
          } else if (nameExistsCondition && self.get('alertNotifications').mapProperty('name').contains(newName)) {
            this.set('nameError', true);
            errorMessage = Em.I18n.t('alerts.actions.manage_alert_notifications_popup.error.name.existed');
          } else if (newName && !validator.isValidAlertNotificationName(newName)) {
            this.set('nameError', true);
            errorMessage = Em.I18n.t('form.validator.alertNotificationName');
          } else {
            this.set('nameError', false);
          }
          this.set('controller.inputFields.name.errorMsg', errorMessage);
        }.observes('controller.inputFields.name.value'),

        emailToValidation: function () {
          var emailToError = false;
          if (this.get('isEmailMethodSelected')) {
            var inputValues = this.get('controller.inputFields.email.value').trim().split(',');
            emailToError = inputValues.some(function (emailTo) {
              return emailTo && !validator.isValidEmail(emailTo.trim());
            });
          }
          this.set('emailToError', emailToError);
          this.set('controller.inputFields.email.errorMsg', emailToError ? Em.I18n.t('alerts.notifications.error.email') : null);
        }.observes('controller.inputFields.email.value'),

        emailFromValidation: function () {
          var emailFrom = this.get('controller.inputFields.emailFrom.value');
          if (emailFrom && !validator.isValidEmail(emailFrom)) {
            this.set('emailFromError', true);
            this.set('controller.inputFields.emailFrom.errorMsg', Em.I18n.t('alerts.notifications.error.email'));
          } else {
            this.set('emailFromError', false);
            this.set('controller.inputFields.emailFrom.errorMsg', null);
          }
        }.observes('controller.inputFields.emailFrom.value'),

        smtpPortValidation: function () {
          var value = this.get('controller.inputFields.SMTPPort.value');
          if (value && (!validator.isValidInt(value) || value < 0)) {
            this.set('smtpPortError', true);
            this.set('controller.inputFields.SMTPPort.errorMsg', Em.I18n.t('alerts.notifications.error.integer'));
          } else {
            this.set('smtpPortError', false);
            this.set('controller.inputFields.SMTPPort.errorMsg', null);
          }
        }.observes('controller.inputFields.SMTPPort.value'),

        hostsValidation: function () {
          var inputValue = this.get('controller.inputFields.host.value').trim(),
              hostError = false;
          if (!this.get('isEmailMethodSelected') && !this.get('isAlertScriptMethodSelected')) {
            var array = inputValue.split(',');
            hostError = array.some(function (hostname) {
              return hostname && !validator.isHostname(hostname.trim());
            });
            hostError = hostError || inputValue === '';
          }
          this.set('hostError', hostError);
          this.set('controller.inputFields.host.errorMsg', hostError ? Em.I18n.t('alerts.notifications.error.host') : null);
        }.observes('controller.inputFields.host.value'),

        portValidation: function () {
          var value = this.get('controller.inputFields.port.value');
          if (value && (!validator.isValidInt(value) || value < 0)) {
            this.set('portError', true);
            this.set('controller.inputFields.port.errorMsg', Em.I18n.t('alerts.notifications.error.integer'));
          } else {
            this.set('portError', false);
            this.set('controller.inputFields.port.errorMsg', null);
          }
        }.observes('controller.inputFields.port.value'),

        smtpUsernameValidation: function () {
          var smtpUsernameError = false;
          var errorMessage = null;
          if (this.get('isEmailMethodSelected')) {
            if (this.get('controller.inputFields.SMTPUseAuthentication.value')) {
              if (Em.isBlank(this.get('controller.inputFields.SMTPUsername.value'))) {
                smtpUsernameError = true;
                errorMessage = Em.I18n.t('alerts.notifications.error.SMTPUsername');
              }
            }
          }
          this.set('smtpUsernameError', smtpUsernameError);
          this.set('controller.inputFields.SMTPUsername.errorMsg', errorMessage);
        }.observes('controller.inputFields.SMTPUsername.value', 'controller.inputFields.SMTPUseAuthentication.value'),

        smtpPasswordValidation: function () {
          var smtpPasswordError = false;
          var errorMessage = null;
          if (this.get('isEmailMethodSelected')) {
            if (this.get('controller.inputFields.SMTPUseAuthentication.value')) {
              if (Em.isBlank(this.get('controller.inputFields.SMTPPassword.value'))) {
                smtpPasswordError = true;
                errorMessage = Em.I18n.t('alerts.notifications.error.SMTPPassword');
              }
            }
          }
          this.set('smtpPasswordError', smtpPasswordError);
          this.set('controller.inputFields.SMTPPassword.errorMsg', errorMessage);
        }.observes('controller.inputFields.SMTPPassword.value', 'controller.inputFields.SMTPUseAuthentication.value'),

        retypePasswordValidation: function () {
          var passwordValue = this.get('controller.inputFields.SMTPPassword.value');
          var retypePasswordValue = this.get('controller.inputFields.retypeSMTPPassword.value');
          if (passwordValue !== retypePasswordValue) {
            this.set('passwordError', true);
            this.set('controller.inputFields.retypeSMTPPassword.errorMsg', Em.I18n.t('alerts.notifications.error.retypePassword'));
          } else {
            this.set('passwordError', false);
            this.set('controller.inputFields.retypeSMTPPassword.errorMsg', null);
          }
        }.observes('controller.inputFields.retypeSMTPPassword.value', 'controller.inputFields.SMTPPassword.value'),

        scriptFileNameValidation: function () {
          var scriptFileNameValue = this.get('controller.inputFields.scriptFileName.value').trim();
          if (!Em.isBlank(scriptFileNameValue) && !validator.isValidFileName(scriptFileNameValue)) {
            this.set('scriptFileNameError', true);
            this.set('controller.inputFields.scriptFileName.errorMsg', Em.I18n.t('alerts.actions.manage_alert_notifications_popup.error.scriptFileName.invalid'));
          } else {
            this.set('scriptFileNameError', false);
            this.set('controller.inputFields.scriptFileName.errorMsg', null);
          }
        }.observes('controller.inputFields.scriptFileName.value'),

        someErrorExists: Em.computed.or('nameError', 'emailToError', 'emailFromError', 'smtpPortError', 'hostError', 'portError', 'smtpUsernameError', 'smtpPasswordError', 'passwordError', 'scriptFileNameError'),

        setParentErrors: function () {
          this.set('parentView.hasErrors', this.get('someErrorExists'));
        }.observes('someErrorExists'),

        groupsSelectView: Em.Select.extend({
          attributeBindings: ['disabled'],
          init: function init() {
            this._super();
            this.set('parentView.groupSelect', this);
          }
        }),

        groupSelect: null,

        /**
         * Select all alert-groups if <code>allGroups.value</code> is 'custom'
         * @method selectAllGroups
         */
        selectAllGroups: function selectAllGroups() {
          if (this.get('controller.inputFields.allGroups.value') === 'custom') {
            this.set('groupSelect.selection', this.get('groupSelect.content').slice());
          }
        },

        /**
         * Deselect all alert-groups if <code>allGroups.value</code> is 'custom'
         * @method clearAllGroups
         */
        clearAllGroups: function clearAllGroups() {
          if (this.get('controller.inputFields.allGroups.value') === 'custom') {
            this.set('groupSelect.selection', []);
          }
        },

        severitySelectView: Em.Select.extend({
          init: function init() {
            this._super();
            this.set('parentView.severitySelect', this);
          }
        }),

        severitySelect: null,

        /**
         * Determines if all alert-groups are selected
         * @type {boolean}
         */
        allGroupsSelected: Em.computed.equalProperties('groupSelect.selection.length', 'groupSelect.content.length'),

        /**
         * Determines if no one alert-group is selected
         * @type {boolean}
         */
        noneGroupsSelected: Em.computed.empty('groupSelect.selection'),

        /**
         * Determines if all severities are selected
         * @type {boolean}
         */
        allSeveritySelected: Em.computed.equalProperties('severitySelect.selection.length', 'severitySelect.content.length'),

        /**
         * Determines if no one severity is selected
         * @type {boolean}
         */
        noneSeveritySelected: Em.computed.empty('severitySelect.selection'),

        /**
         * Select all severities
         * @method selectAllSeverity
         */
        selectAllSeverity: function selectAllSeverity() {
          this.set('severitySelect.selection', this.get('severitySelect.content').slice());
        },

        /**
         * Deselect all severities
         * @method clearAllSeverity
         */
        clearAllSeverity: function clearAllSeverity() {
          this.set('severitySelect.selection', []);
        }
      }),

      isSaving: false,

      hasErrors: false,

      primary: Em.I18n.t('common.save'),

      disablePrimary: Em.computed.or('isSaving', 'hasErrors'),

      onPrimary: function onPrimary() {
        this.set('isSaving', true);
        var apiObject = self.formatNotificationAPIObject();
        if (isEdit) {
          self.updateAlertNotification(apiObject);
        } else {
          self.createAlertNotification(apiObject);
        }
      },
      hide: function hide() {
        self.set('createEditPopup', null);
        return this._super.apply(this, arguments);
      }
    });
    this.set('createEditPopup', createEditPopup);
    return createEditPopup;
  },

  /**
   * Create API-formatted object from data populate by user
   * @returns {Object}
   * @method formatNotificationAPIObject
   */
  formatNotificationAPIObject: function formatNotificationAPIObject() {
    var inputFields = this.get('inputFields');
    var properties = {};
    if (inputFields.get('method.value') === 'EMAIL') {
      properties['ambari.dispatch.recipients'] = inputFields.get('email.value').replace(/\s/g, '').split(',');
      properties['mail.smtp.host'] = inputFields.get('SMTPServer.value');
      properties['mail.smtp.port'] = inputFields.get('SMTPPort.value');
      properties['mail.smtp.from'] = inputFields.get('emailFrom.value');
      properties['mail.smtp.auth'] = inputFields.get('SMTPUseAuthentication.value');
      if (inputFields.get('SMTPUseAuthentication.value')) {
        properties['ambari.dispatch.credential.username'] = inputFields.get('SMTPUsername.value');
        properties['ambari.dispatch.credential.password'] = inputFields.get('SMTPPassword.value');
        properties['mail.smtp.starttls.enable'] = inputFields.get('SMTPSTARTTLS.value');
      }
    } else if (inputFields.get('method.value') === 'SNMP') {
      properties['ambari.dispatch.snmp.version'] = inputFields.get('version.value');
      properties['ambari.dispatch.snmp.community'] = inputFields.get('community.value');
      properties['ambari.dispatch.recipients'] = inputFields.get('host.value').replace(/\s/g, '').split(',');
      properties['ambari.dispatch.snmp.port'] = inputFields.get('port.value');
    } else if (inputFields.get('method.value') === 'Custom SNMP') {
      properties['ambari.dispatch.snmp.version'] = inputFields.get('version.value');
      properties['ambari.dispatch.snmp.oids.trap'] = inputFields.get('OIDs.value');
      properties['ambari.dispatch.snmp.oids.subject'] = inputFields.get('OIDs.value');
      properties['ambari.dispatch.snmp.oids.body'] = inputFields.get('OIDs.value');
      properties['ambari.dispatch.snmp.community'] = inputFields.get('community.value');
      properties['ambari.dispatch.recipients'] = inputFields.get('host.value').replace(/\s/g, '').split(',');
      properties['ambari.dispatch.snmp.port'] = inputFields.get('port.value');
    } else if (inputFields.get('method.value') === 'Alert Script') {
      var scriptDispatchProperty = inputFields.get('scriptDispatchProperty.value').trim();
      if (scriptDispatchProperty != '') properties['ambari.dispatch-property.script'] = scriptDispatchProperty;

      var scriptFileName = inputFields.get('scriptFileName.value').trim();
      if (scriptFileName != '') properties['ambari.dispatch-property.script.filename'] = scriptFileName;
    }
    inputFields.get('customProperties').forEach(function (customProperty) {
      properties[customProperty.name] = customProperty.value;
    });
    var apiObject = {
      AlertTarget: {
        name: inputFields.get('name.value'),
        description: inputFields.get('description.value'),
        global: inputFields.get('allGroups.value') === 'all',
        notification_type: this.getNotificationType(inputFields.get('method.value')),
        alert_states: inputFields.get('severityFilter.value'),
        properties: properties
      }
    };
    if (inputFields.get('allGroups.value') === 'custom') {
      apiObject.AlertTarget.groups = inputFields.get('groups.value').mapProperty('id');
    }
    return apiObject;
  },

  getNotificationType: function getNotificationType(text) {
    var notificationType = text;
    if (notificationType === "Custom SNMP") {
      notificationType = "SNMP";
    } else if (notificationType === "SNMP") {
      notificationType = "AMBARI_SNMP";
    } else if (notificationType === "Alert Script") {
      notificationType = "ALERT_SCRIPT";
    }
    return notificationType;
  },

  getNotificationTypeText: function getNotificationTypeText(notificationType) {
    var notificationTypeText = notificationType;
    if (notificationType === "SNMP") {
      notificationTypeText = "Custom SNMP";
    } else if (notificationType === "AMBARI_SNMP") {
      notificationTypeText = "SNMP";
    } else if (notificationType === "ALERT_SCRIPT") {
      notificationTypeText = "Alert Script";
    }
    return notificationTypeText;
  },

  /**
   * Send request to server to create Alert Notification
   * @param {object} apiObject (@see formatNotificationAPIObject)
   * @returns {$.ajax}
   * @method createAlertNotification
   */
  createAlertNotification: function createAlertNotification(apiObject) {
    return App.ajax.send({
      name: 'alerts.create_alert_notification',
      sender: this,
      data: {
        data: apiObject
      },
      success: 'createAlertNotificationSuccessCallback',
      error: 'saveErrorCallback'
    });
  },

  /**
   * Success callback for <code>createAlertNotification</code>
   * @method createAlertNotificationSuccessCallback
   */
  createAlertNotificationSuccessCallback: function createAlertNotificationSuccessCallback() {
    this.loadAlertNotifications();
    var createEditPopup = this.get('createEditPopup');
    if (createEditPopup) {
      createEditPopup.hide();
    }
  },

  /**
   * Send request to server to update Alert Notification
   * @param {object} apiObject (@see formatNotificationAPIObject)
   * @returns {$.ajax}
   * @method updateAlertNotification
   */
  updateAlertNotification: function updateAlertNotification(apiObject) {
    if (apiObject != null) {
      if (apiObject.AlertTarget.global == false) {
        this.get('selectedAlertNotification').set('global', false);
      } else {
        this.get('selectedAlertNotification').set('global', true);
      }
    }
    return App.ajax.send({
      name: 'alerts.update_alert_notification',
      sender: this,
      data: {
        data: apiObject,
        id: this.get('selectedAlertNotification.id')
      },
      success: 'updateAlertNotificationSuccessCallback',
      error: 'saveErrorCallback'
    });
  },

  /**
   * Success callback for <code>updateAlertNotification</code>
   * @method updateAlertNotificationSuccessCallback
   */
  updateAlertNotificationSuccessCallback: function updateAlertNotificationSuccessCallback() {
    this.loadAlertNotifications();
    var createEditPopup = this.get('createEditPopup');
    if (createEditPopup) {
      createEditPopup.hide();
    }
  },

  /**
   * Error callback for <code>createAlertNotification</code> and <code>updateAlertNotification</code>
   * @method saveErrorCallback
   */
  saveErrorCallback: function saveErrorCallback() {
    this.set('createEditPopup.isSaving', false);
  },

  /**
   * Delete Notification button handler
   * @return {App.ModalPopup}
   * @method deleteAlertNotification
   */
  deleteAlertNotification: function deleteAlertNotification() {
    var self = this;
    return App.showConfirmationPopup(function () {
      App.ajax.send({
        name: 'alerts.delete_alert_notification',
        sender: self,
        data: {
          id: self.get('selectedAlertNotification.id')
        },
        success: 'deleteAlertNotificationSuccessCallback'
      });
    }, Em.I18n.t('alerts.actions.manage_alert_notifications_popup.confirmDeleteBody').format(this.get('selectedAlertNotification.name')), null, Em.I18n.t('alerts.actions.manage_alert_notifications_popup.confirmDeleteHeader'), Em.I18n.t('common.delete'));
  },

  /**
   * Success callback for <code>deleteAlertNotification</code>
   * @method deleteAlertNotificationSuccessCallback
   */
  deleteAlertNotificationSuccessCallback: function deleteAlertNotificationSuccessCallback() {
    this.loadAlertNotifications();
    var selectedAlertNotification = this.get('selectedAlertNotification');
    selectedAlertNotification.deleteRecord();
    this.set('selectedAlertNotification', null);
  },

  /**
   * Duplicate Notification button handler
   * @method duplicateAlertNotification
   */
  duplicateAlertNotification: function duplicateAlertNotification() {
    this.fillEditCreateInputs(true);
    this.showCreateEditPopup();
  },

  /**
   * Enable or Disable Notification button handler
   * @method enableOrDisableAlertNotification
   */
  enableOrDisableAlertNotification: function enableOrDisableAlertNotification(e) {
    return App.ajax.send({
      name: 'alerts.update_alert_notification',
      sender: this,
      data: {
        data: {
          AlertTarget: {
            enabled: e.context !== "disable"
          }
        },
        id: this.get('selectedAlertNotification.id')
      },
      success: 'enableOrDisableAlertNotificationSuccessCallback',
      error: 'saveErrorCallback'
    });
  },

  /**
   * Success callback for <code>enableOrDisableAlertNotification</code>
   * @method enableOrDisableAlertNotificationSuccessCallback
   */
  enableOrDisableAlertNotificationSuccessCallback: function enableOrDisableAlertNotificationSuccessCallback() {
    this.loadAlertNotifications();
    var createEditPopup = this.get('createEditPopup');
    if (createEditPopup) {
      createEditPopup.hide();
    }
  },

  /**
   * Show popup with form for new custom property
   * @method addCustomPropertyHandler
   * @return {App.ModalPopup}
   */
  addCustomPropertyHandler: function addCustomPropertyHandler() {
    var self = this;

    return App.ModalPopup.show({

      controllerBinding: 'App.router.manageAlertNotificationsController',

      header: Em.I18n.t('alerts.notifications.addCustomPropertyPopup.header'),

      primary: Em.I18n.t('common.add'),

      bodyClass: Em.View.extend({
        templateName: require('templates/main/alerts/add_custom_config_to_alert_notification_popup')
      }),

      disablePrimary: Em.computed.alias('controller.isErrorWithNewCustomPropertyName'),

      onPrimary: function onPrimary() {
        self.addCustomProperty();
        self.set('newCustomProperty', { name: '', value: '' }); // cleanup
        this.hide();
      }

    });
  },

  /**
   * Add Custom Property to <code>selectedAlertNotification</code> (push it to <code>properties</code>-field)
   * @method addCustomProperty
   */
  addCustomProperty: function addCustomProperty() {
    var newCustomProperty = this.get('newCustomProperty');
    this.get('inputFields.customProperties').pushObject({
      name: newCustomProperty.name,
      value: newCustomProperty.value,
      defaultValue: newCustomProperty.value
    });
  },

  /**
   * "Remove"-button click handler
   * Delete customProperty from <code>inputFields.customProperties</code>
   * @param {object} e
   * @method removeCustomProperty
   */
  removeCustomPropertyHandler: function removeCustomPropertyHandler(e) {
    var customProperties = this.get('inputFields.customProperties');
    this.set('inputFields.customProperties', customProperties.without(e.context));
  }

});

});

require.register("controllers/main/charts", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainChartsController = Em.ArrayController.extend({
  name: 'mainChartsController'
});

});

require.register("controllers/main/charts/heatmap", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

var App = require('app');

App.MainChartsHeatmapController = Em.Controller.extend(App.WidgetSectionMixin, {
  name: 'mainChartsHeatmapController',
  rackMap: {},
  racks: [],
  rackViews: [],

  /**
   * @type {boolean}
   */
  isLoaded: false,

  /**
   * Heatmap metrics that are available choices  on the page
   */
  heatmapCategories: [],

  allHeatmaps: [],

  layoutNameSuffix: "_heatmap",

  sectionNameSuffix: "_HEATMAPS",

  loadRacksUrlParams: 'fields=Hosts/rack_info,Hosts/host_name,Hosts/public_host_name,Hosts/os_type,Hosts/ip,host_components,metrics/disk,metrics/cpu/cpu_system,metrics/cpu/cpu_user,metrics/memory/mem_total,metrics/memory/mem_free&minimal_response=true',

  /**
   * @type {string}
   */
  loadHeatmapsUrlParams: function () {
    var serviceName = this.get('content.serviceName');
    if (serviceName) {
      return 'WidgetInfo/widget_type=HEATMAP&WidgetInfo/scope=CLUSTER&WidgetInfo/metrics.matches(.*\"service_name\":\"' + serviceName + '\".*)&fields=WidgetInfo/metrics';
    } else {
      return 'WidgetInfo/widget_type=HEATMAP&WidgetInfo/scope=CLUSTER&fields=WidgetInfo/metrics';
    }
  }.property('content.serviceName'),

  selectedMetric: null,

  inputMaximum: '',

  /**
   * Heatmap widget currently shown on the page
   */
  activeWidget: Em.computed.alias('widgets.firstObject'),

  /**
   * This function is called from the bound view of the controller
   */
  loadPageData: function loadPageData() {
    var self = this;

    this.loadRacks().always(function () {
      self.resetPageData();
      self.getAllHeatMaps().done(function (allHeatmapData) {
        self.set('isLoaded', true);
        allHeatmapData.items.forEach(function (_allHeatmapData) {
          self.get('allHeatmaps').pushObject(_allHeatmapData.WidgetInfo);
        });
        var categories = self.categorizeByServiceName(self.get('allHeatmaps'));
        self.set('heatmapCategories', categories);
        self.getActiveWidgetLayout();
      });
    });
  },

  /**
   * categorize heatmaps with respect to service names
   * @param {Array} allHeatmaps
   * @return {Array}
   */
  categorizeByServiceName: function categorizeByServiceName(allHeatmaps) {
    var categories = [];

    allHeatmaps.forEach(function (_heatmap) {
      var serviceNames = JSON.parse(_heatmap.metrics).mapProperty('service_name').uniq();
      serviceNames.forEach(function (_serviceName) {
        var category = categories.findProperty('serviceName', _serviceName);
        if (!category) {
          categories.pushObject(Em.Object.create({
            serviceName: _serviceName,
            displayName: _serviceName === 'STACK' ? 'Host' : App.format.role(_serviceName, true),
            heatmaps: [_heatmap]
          }));
        } else {
          category.get('heatmaps').pushObject(_heatmap);
        }
      }, this);
    }, this);
    return categories;
  },

  /**
   * clears/resets the data. This function should be called every time user navigates to heatmap page
   */
  resetPageData: function resetPageData() {
    this.get('heatmapCategories').clear();
    this.get('allHeatmaps').clear();
  },

  /**
   *  Gets all heatmap widgets that should be available in select metrics dropdown on heatmap page
   * @return {$.ajax}
   */
  getAllHeatMaps: function getAllHeatMaps() {
    var urlParams = this.get('loadHeatmapsUrlParams');

    return App.ajax.send({
      name: 'widgets.get',
      sender: this,
      data: {
        urlParams: urlParams,
        sectionName: this.get('sectionName')
      }
    });
  },

  /**
   * get hosts from server
   */
  loadRacks: function loadRacks() {
    this.get('racks').clear();
    this.set('rackMap', {});
    var urlParams = this.get('loadRacksUrlParams');
    return App.ajax.send({
      name: 'hosts.heatmaps',
      sender: this,
      data: {
        urlParams: urlParams
      },
      success: 'loadRacksSuccessCallback'
    });
  },

  loadRacksSuccessCallback: function loadRacksSuccessCallback(data, opt, params) {
    var hosts = [];
    data.items.forEach(function (item) {
      hosts.push({
        hostName: item.Hosts.host_name,
        publicHostName: item.Hosts.public_host_name,
        osType: item.Hosts.os_type,
        ip: item.Hosts.ip,
        rack: item.Hosts.rack_info,
        diskTotal: item.metrics ? item.metrics.disk.disk_total : 0,
        diskFree: item.metrics ? item.metrics.disk.disk_free : 0,
        cpuSystem: item.metrics ? item.metrics.cpu.cpu_system : 0,
        cpuUser: item.metrics ? item.metrics.cpu.cpu_user : 0,
        memTotal: item.metrics ? item.metrics.memory.mem_total : 0,
        memFree: item.metrics ? item.metrics.memory.mem_free : 0,
        hostComponents: item.host_components.mapProperty('HostRoles.component_name')
      });
    });
    var rackMap = this.indexByRackId(hosts);
    var racks = this.toList(rackMap);
    //this list has an empty host array property
    this.set('rackMap', rackMap);
    this.set('racks', racks);
  },

  /**
   * @param {Array} hosts
   * @returns {Object} rackMap
   */
  indexByRackId: function indexByRackId(hosts) {
    var rackMap = {};

    hosts.forEach(function (host) {
      var rackId = host.rack;
      if (!rackMap[rackId]) {
        rackMap[rackId] = Em.Object.create({
          name: rackId,
          rackId: rackId,
          hosts: [host]
        });
      } else {
        rackMap[rackId].hosts.push(host);
      }
    });
    return rackMap;
  },

  /**
   *
   * @param {Object} rackMap
   * @returns {Array} racks
   */
  toList: function toList(rackMap) {
    var racks = [];
    var i = 0;
    for (var rackKey in rackMap) {
      if (rackMap.hasOwnProperty(rackKey)) {
        racks.push(Em.Object.create({
          name: rackKey,
          rackId: rackKey,
          hosts: rackMap[rackKey].hosts,
          isLoaded: false,
          index: i++
        }));
      }
    }
    return racks;
  },

  validation: function () {
    if (this.get('selectedMetric')) {
      if (/^\d+$/.test(this.get('inputMaximum'))) {
        $('#inputMaximum').removeClass('error');
        this.set('selectedMetric.maximumValue', this.get('inputMaximum'));
      } else {
        $('#inputMaximum').addClass('error');
      }
    }
  }.observes('inputMaximum'),

  addRackView: function addRackView(view) {
    this.get('rackViews').push(view);
    if (this.get('rackViews').length == this.get('racks').length) {
      this.displayAllRacks();
    }
  },

  displayAllRacks: function displayAllRacks() {
    if (this.get('rackViews').length) {
      this.get('rackViews').pop().displayHosts();
      this.displayAllRacks();
    }
  },

  showHeatMapMetric: function showHeatMapMetric(event) {
    var self = this;
    var metricItem = Em.Object.create(event.context);
    this.saveWidgetLayout([metricItem]).done(function () {
      self.getActiveWidgetLayout();
    });
  },

  hostToSlotMap: Em.computed.alias('selectedMetric.hostToSlotMap'),

  /**
   * return class name for to be used for containing each rack.
   *
   * @this App.MainChartsHeatmapController
   */
  rackClass: function () {
    var rackCount = this.get('racks.length');
    if (rackCount < 2) {
      return "col-md-12";
    }
    if (rackCount === 2) {
      return "col-md-6";
    }
    return "col-md-4";
  }.property('racks.length')
});

});

require.register("controllers/main/charts/heatmap_metrics/heatmap_metric", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

var App = require('app');
var date = require('utils/date/date');

var numberUtils = require('utils/number_utils');

/**
 * Base class for any heatmap metric.
 * 
 * This class basically provides the following for each heatmap metric.
 * <ul>
 * <li> Provides number of slots in which temperature can fall.
 * <li> Maintains the maximum value so as to scale slot ranges.
 * <li> Gets JSON data from server and maps response for all hosts into above
 * slots.
 * </ul>
 * 
 */
App.MainChartHeatmapMetric = Em.Object.extend({
  /**
   * Name of this metric
   */
  name: null,

  /**
   * Number of slots this metric will be mapped into. When changing this value,
   * the color count in 'slotColors' should also be changed.
   */
  numberOfSlots: 5,

  /**
   * Colors for the each of the number of slots defined above. When changing the
   * number of slots, the number of colors also should be updated.
   * 
   * @type {Array}
   */
  slotColors: ['#1EB475', '#1FB418', '#E9E23F', '#E9A840', '#EF6162'],

  /**
   * Minimum value of this metric. Default is 0.
   */
  minimumValue: 0,

  /**
   * Maximum value of this metric. This has to be specified by extending classes
   * so that the range from 'minimumValue' to 'maximumValue' can be split among
   * 'numberOfSlots'. It is recommended that this value be a multiple of
   * 'numberOfSlots'.
   */
  maximumValue: 100,

  /**
   * Units of the maximum value which is shown in UI {String}
   */
  units: '',

  /**
   * Provides following information about slots in an array of objects.
   * <ul>
   * <li> from: {number} Slot starts from this value
   * <li> to: {number} Slot ends at this value (inclusive)
   * <li> label: {String} Slot name to be shown
   * <li> cssStyle: {String} style to be embedded on hosts which fall into this
   * slot.
   * </ul>
   * 
   * Slot count will be the same as specified in 'numberOfSlots'. Slot
   * definitions will be given in increasing temperature from 'minimumValue' to
   * 'maximumValue'.
   * 
   */
  slotDefinitions: function () {
    var max = this.getMaximumValue();
    var slotCount = this.get('numberOfSlots');
    var units = this.get('units');
    var delta = (max - this.get('minimumValue')) / slotCount;
    var defs = [];
    var slotColors = this.get('slotColors');
    var slotColorIndex = 0;
    for (var c = 0; c < slotCount - 1; c++) {
      var slotColor = slotColors[slotColorIndex++];
      var start = c * delta;
      var end = (c + 1) * delta;
      defs.push(this.generateSlot(start, end, units, slotColor));
    }
    slotColor = slotColors[slotColorIndex++];
    start = (slotCount - 1) * delta;
    defs.push(this.generateSlot(start, max, units, slotColor));

    defs.push(Em.Object.create({
      invalidData: true,
      index: defs.length,
      label: Em.I18n.t('charts.heatmap.label.invalidData'),
      cssStyle: this.getHatchStyle()
    }));
    defs.push(Em.Object.create({
      notAvailable: true,
      index: defs.length,
      label: Em.I18n.t('charts.heatmap.label.notAvailable'),
      cssStyle: "background-color: #666"
    }));
    defs.push(Em.Object.create({
      notApplicable: true,
      index: defs.length,
      label: Em.I18n.t('charts.heatmap.label.notApplicable'),
      cssStyle: "background-color:#ccc"
    }));
    return defs;
  }.property('minimumValue', 'maximumValue', 'numberOfSlots'),
  /**
   * genarate slot by given parameters
   *
   * @param min
   * @param max
   * @param units
   * @param slotColor
   * @return {object}
   * @method generateSlot
   */
  generateSlot: function generateSlot(min, max, units, slotColor) {
    var fromLabel = this.formatLegendLabel(min, units);
    var toLabel = this.formatLegendLabel(max, units);
    var from = this.convertNumber(min, units);
    var to = this.convertNumber(max, units);

    return Em.Object.create({
      hasBoundaries: true,
      from: from,
      to: to,
      label: fromLabel + " - " + toLabel,
      cssStyle: "background-color:" + slotColor
    });
  },

  /**
   * calculate hatch style of slot according to browser version used
   * @return {String}
   */
  getHatchStyle: function getHatchStyle() {
    var hatchStyle = "background-color:#666";
    if (jQuery.browser.webkit) {
      hatchStyle = "background-image:-webkit-repeating-linear-gradient(-45deg, #666, #666 6px, #fff 6px, #fff 7px)";
    } else if (jQuery.browser.mozilla) {
      hatchStyle = "background-image:repeating-linear-gradient(-45deg, #666, #666 6px, #fff 6px, #fff 7px)";
    } else if (jQuery.browser.msie && jQuery.browser.version) {
      var majorVersion = parseInt(jQuery.browser.version.split('.')[0]);
      if (majorVersion > 9) {
        hatchStyle = "background-image:repeating-linear-gradient(-45deg, #666, #666 6px, #fff 6px, #fff 7px)";
      }
    }
    return hatchStyle;
  },

  /**
   * compatible with special input value, such as negative float number or string
   * @return {float}
   */
  getMaximumValue: function getMaximumValue() {
    var max = this.get('maximumValue') ? parseFloat(this.get('maximumValue')) : 0;
    return isNaN(max) ? 0 : Math.max(0, max);
  },

  /**
   * In slot definitions this value is used to construct the label by appending
   * it to slot min-max values. For example giving '%' here would result in slot
   * definition label being '0% - 10%'.
   */
  slotDefinitionLabelSuffix: '',

  hostToValueMap: null,

  hostToSlotMap: function () {
    var hostToValueMap = this.get('hostToValueMap');
    var hostNames = this.get('hostNames');
    var hostToSlotMap = {};
    if (hostToValueMap && hostNames) {
      hostNames.forEach(function (hostName) {
        var slot = this.calculateSlot(hostToValueMap, hostName);
        if (slot > -1) {
          hostToSlotMap[hostName] = slot;
        }
      }, this);
    }
    return hostToSlotMap;
  }.property('hostToValueMap', 'slotDefinitions'),

  /**
   * calculate slot position in hostToSlotMap by hostname
   * @param hostToValueMap
   * @param hostName
   * @return {Number}
   */
  calculateSlot: function calculateSlot(hostToValueMap, hostName) {
    var slotDefinitions = this.get('slotDefinitions');
    var slotWithBoundaries = slotDefinitions.filterProperty('hasBoundaries');
    var invalidDataSlot = slotDefinitions.findProperty('invalidData').get('index');
    var notAvailableDataSlot = slotDefinitions.findProperty('notAvailable').get('index');
    var slot = slotDefinitions.findProperty('notApplicable').get('index');

    if (hostName in hostToValueMap) {
      var value = hostToValueMap[hostName];
      if (Em.isNone(value)) {
        slot = notAvailableDataSlot;
      } else if (isNaN(value) || !isFinite(value) || value < 0) {
        slot = invalidDataSlot;
      } else {
        value = Number(value);
        slotWithBoundaries.forEach(function (slotDef, slotIndex, array) {

          if (value >= slotDef.from && value <= slotDef.to ||
          // If value exceeds maximum then it pushed to the last/maximal slot
          value > slotDef.to && slotIndex === array.length - 1) {
            slot = slotIndex;
          }
        });
      }
    }
    return slot;
  },

  /**
   * Turns numbers into displayable values. For example 24.345432425 into 24.3
   * etc.
   * 
   * @private
   */
  formatLegendLabel: function formatLegendLabel(num, units) {
    var fraction = num % 1;
    if (num >= 100) {
      num = Math.round(num);
    } else if (num >= 10 && fraction > 0) {
      num = parseFloat(num.toFixed(1));
    } else if (fraction > 0) {
      num = parseFloat(num.toFixed(2));
    }
    if (units === 'ms') {
      return date.timingFormat(num);
    }
    return num + units;
  },

  /**
   * return converted value
   *
   * @param {number} number
   * @param {string} units
   */
  convertNumber: function convertNumber(number, units) {
    if (units === 'MB') {
      return Math.round(number * numberUtils.BYTES_IN_MB);
    }
    return number;
  }

});

});

require.register("controllers/main/dashboard", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainDashboardController = Em.Controller.extend({
  name: 'mainDashboardController',
  categorySelected: 'widgets'
});

});

require.register("controllers/main/dashboard/config_history_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainConfigHistoryController = Em.ArrayController.extend(App.TableServerMixin, {
  name: 'mainConfigHistoryController',

  content: App.ServiceConfigVersion.find(),
  totalCount: 0,
  filteredCount: 0,
  resetStartIndex: true,
  mockUrl: '/data/configurations/service_versions.json',
  realUrl: function () {
    return App.apiPrefix + '/clusters/' + App.get('clusterName') + '/configurations/service_config_versions?<parameters>fields=service_config_version,user,group_id,group_name,is_current,createtime,service_name,hosts,service_config_version_note,is_cluster_compatible,stack_id&minimal_response=true';
  }.property('App.clusterName'),

  /**
   * @type {boolean}
   * @default false
   */
  showFilterConditionsFirstLoad: false,

  /**
   * associations between host property and column index
   * @type {Array}
   */
  colPropAssoc: function () {
    var associations = [];
    associations[1] = 'serviceVersion';
    associations[2] = 'configGroup';
    associations[3] = 'createTime';
    associations[4] = 'author';
    associations[5] = 'notes';
    return associations;
  }.property(),

  filterProps: [{
    name: 'serviceVersion',
    key: 'service_name',
    type: 'EQUAL'
  }, {
    name: 'configGroup',
    key: 'group_name',
    type: 'EQUAL'
  }, {
    name: 'createTime',
    key: 'createtime',
    type: 'MORE'
  }, {
    name: 'author',
    key: 'user',
    type: 'MATCH'
  }, {
    name: 'notes',
    key: 'service_config_version_note',
    type: 'MATCH'
  }],

  sortProps: [{
    name: 'serviceVersion',
    key: 'service_name'
  }, {
    name: 'configGroup',
    key: 'group_name'
  }, {
    name: 'createTime',
    key: 'createtime'
  }, {
    name: 'author',
    key: 'user'
  }, {
    name: 'notes',
    key: 'service_config_version_note'
  }],

  /**
   * load all data components required by config history table
   *  - total counter of service config versions(called in parallel)
   *  - current versions
   *  - filtered versions
   * @param {boolean} shouldUpdateCounter
   * @return {*}
   */
  load: function load() {
    var _this = this;

    var shouldUpdateCounter = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    var dfd = $.Deferred();
    this.loadConfigVersionsToModel().done(function () {
      if (shouldUpdateCounter) {
        _this.updateTotalCounter();
      }
      dfd.resolve();
    });
    return dfd.promise();
  },

  /**
   * get filtered service config versions from server and push it to model
   * @return {*}
   */
  loadConfigVersionsToModel: function loadConfigVersionsToModel() {
    var dfd = $.Deferred();
    var queryParams = this.getQueryParameters();

    App.HttpClient.get(this.getUrl(queryParams), App.serviceConfigVersionsMapper, {
      complete: function complete() {
        dfd.resolve();
      }
    });
    return dfd.promise();
  },

  updateTotalCounter: function updateTotalCounter() {
    return App.ajax.send({
      name: 'service.serviceConfigVersions.get.total',
      sender: this,
      data: {},
      success: 'updateTotalCounterSuccess'
    });
  },

  updateTotalCounterSuccess: function updateTotalCounterSuccess(data, opt, params) {
    this.set('totalCount', Number(data.itemTotal));
  },

  getUrl: function getUrl(queryParams) {
    var params = '';
    if (App.get('testMode')) {
      return this.get('mockUrl');
    } else {
      if (queryParams) {
        params = App.router.get('updateController').computeParameters(queryParams);
      }
      params = params.length > 0 ? params + "&" : params;
      return this.get('realUrl').replace('<parameters>', params);
    }
  },

  subscribeToUpdates: function subscribeToUpdates() {
    App.StompClient.addHandler('/events/configs', 'history', this.load.bind(this, true));
  },

  unsubscribeOfUpdates: function unsubscribeOfUpdates() {
    App.StompClient.removeHandler('/events/configs', 'history');
  },

  /**
   * get sort properties from local db. A overridden funtion from parent
   * @return {Array}
   */
  getSortProps: function getSortProps() {
    var savedSortConditions = App.db.getSortingStatuses(this.get('name')) || [],
        sortProperties = this.get('sortProps'),
        sortParams = [];

    savedSortConditions.forEach(function (sort) {
      var property = sortProperties.findProperty('name', sort.name);
      if (property && (sort.status === 'sorting_asc' || sort.status === 'sorting_desc')) {
        property.value = sort.status.replace('sorting_', '');
        property.type = 'SORT';
        if (property.name == 'serviceVersion') {
          property.key = "service_name." + sort.status.replace('sorting_', '') + ",service_config_version";
          property.value = "desc";
        }
        if (property.name == 'configGroup') {
          property.key = "group_name." + sort.status.replace('sorting_', '') + ",service_config_version";
          property.value = "desc";
        }

        sortParams.push(property);
      }
    });
    return sortParams;
  },

  /**
   *
   * @param {string} name
   */
  getSearchBoxSuggestions: function getSearchBoxSuggestions(name) {
    var dfd = $.Deferred();
    var key = this.get('filterProps').findProperty('name', name).key;
    App.ajax.send({
      name: 'service.serviceConfigVersions.get.suggestions',
      sender: this,
      data: {
        key: key
      }
    }).done(function (data) {
      dfd.resolve(data.items.mapProperty(key).uniq());
    }).fail(function () {
      dfd.resolve([]);
    });
    return dfd.promise();
  }
});

});

require.register("controllers/main/host", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var validator = require('utils/validator');

App.MainHostController = Em.ArrayController.extend(App.TableServerMixin, {
  name: 'mainHostController',

  clearFilters: null,

  filteredCount: 0,

  /**
   * total number of installed hosts
   * @type {number}
   */
  totalCount: Em.computed.alias('App.allHostNames.length'),

  /**
   * @type {boolean}
   * @default false
   */
  resetStartIndex: false,

  startIndex: 1,

  /**
   * true if any host page filter changed
   */
  filterChangeHappened: false,

  /**
   * if true, do not clean stored filter before hosts page rendering.
   */
  showFilterConditionsFirstLoad: false,

  saveSelection: false,

  content: App.Host.find(),

  allHostStackVersions: App.HostStackVersion.find(),
  /**
   * filterProperties support follow types of filter:
   * MATCH - match of RegExp
   * EQUAL - equality "="
   * LESS - "<"
   * MORE - ">"
   * MULTIPLE - multiple values to compare
   * CUSTOM - substitute values with keys "{#}" in alias
   */
  filterProperties: [{
    name: 'hostName',
    key: 'Hosts/host_name',
    type: 'MATCH'
  }, {
    name: 'ip',
    key: 'Hosts/ip',
    type: 'MATCH'
  }, {
    name: 'cpu',
    key: 'Hosts/cpu_count',
    type: 'EQUAL'
  }, {
    name: 'memoryFormatted',
    key: 'Hosts/total_mem',
    type: 'EQUAL'
  }, {
    name: 'loadAvg',
    key: 'metrics/load/load_one',
    type: 'EQUAL'
  }, {
    name: 'rack',
    key: 'Hosts/rack_info',
    type: 'MATCH'
  }, {
    name: 'hostComponents',
    key: 'host_components/HostRoles/component_name',
    type: 'EQUAL',
    isComponentRelatedFilter: true
  }, {
    name: 'services',
    key: 'host_components/HostRoles/service_name',
    type: 'MATCH',
    isComponentRelatedFilter: true
  }, {
    name: 'state',
    key: 'host_components/HostRoles/state',
    type: 'MATCH',
    isComponentRelatedFilter: true
  }, {
    name: 'healthClass',
    key: 'Hosts/host_status',
    type: 'EQUAL'
  }, {
    name: 'criticalWarningAlertsCount',
    key: '(alerts_summary/CRITICAL{0}|alerts_summary/WARNING{1})',
    type: 'CUSTOM'
  }, {
    name: 'componentsWithStaleConfigsCount',
    key: 'host_components/HostRoles/stale_configs',
    type: 'EQUAL',
    isComponentRelatedFilter: true
  }, {
    name: 'componentsInPassiveStateCount',
    key: 'host_components/HostRoles/maintenance_state',
    type: 'MULTIPLE',
    isComponentRelatedFilter: true
  }, {
    name: 'selected',
    key: 'Hosts/host_name',
    type: 'MULTIPLE'
  }, {
    name: 'version',
    key: 'stack_versions/repository_versions/RepositoryVersions/display_name',
    type: 'EQUAL'
  }, {
    name: 'versionState',
    key: 'stack_versions/HostStackVersions/state',
    type: 'EQUAL'
  }, {
    name: 'hostStackVersion',
    key: 'stack_versions',
    type: 'EQUAL'
  }, {
    name: 'componentState',
    key: ['(host_components/HostRoles/component_name={0})', '(host_components/HostRoles/component_name={0}&host_components/HostRoles/state={1})', '(host_components/HostRoles/component_name={0}&host_components/HostRoles/desired_admin_state={1})', '(host_components/HostRoles/component_name={0}&host_components/HostRoles/maintenance_state={1})'],
    type: 'COMBO',
    isComponentRelatedFilter: true
  }],

  sortProps: [{
    name: 'hostName',
    key: 'Hosts/host_name'
  }, {
    name: 'ip',
    key: 'Hosts/ip'
  }, {
    name: 'cpu',
    key: 'Hosts/cpu_count'
  }, {
    name: 'memoryFormatted',
    key: 'Hosts/total_mem'
  }, {
    name: 'diskUsage',
    //TODO disk_usage is relative property and need support from API, metrics/disk/disk_free used temporarily
    key: 'metrics/disk/disk_free'
  }, {
    name: 'rack',
    key: 'Hosts/rack_info'
  }, {
    name: 'loadAvg',
    key: 'metrics/load/load_one'
  }],

  /**
   * Validate and convert input string to valid url parameter.
   * Detect if user have passed string as regular expression or extend
   * string to regexp.
   *
   * @param {String} value
   * @return {String}
   **/
  getRegExp: function getRegExp(value) {
    value = validator.isValidMatchesRegexp(value) ? value.replace(/(\.+\*?|(\.\*)+)$/, '') + '.*' : '^$';
    value = /^\.\*/.test(value) || value == '^$' ? value : '.*' + value;
    return value;
  },

  /**
   * Sort by host_name by default
   * @method getSortProps
   * @returns {{value: 'asc|desc', name: string, type: 'SORT'}[]}
   */
  getSortProps: function getSortProps() {
    var controllerName = this.get('name'),
        db = App.db.getSortingStatuses(controllerName);
    if (db && db.everyProperty('status', 'sorting')) {
      App.db.setSortingStatuses(controllerName, {
        name: 'hostName',
        status: 'sorting_asc'
      });
    }
    return this._super();
  },

  /**
   * get query parameters computed from filter properties, sort properties and custom properties of view
   * @param {boolean} [skipNonFilterProperties]
   * @return {Array}
   * @method getQueryParameters
   */
  getQueryParameters: function getQueryParameters(skipNonFilterProperties) {
    skipNonFilterProperties = skipNonFilterProperties || false;
    var queryParams = [],
        savedFilterConditions = App.db.getFilterConditions(this.get('name')) || [],
        colPropAssoc = this.get('colPropAssoc'),
        filterProperties = this.get('filterProperties'),
        sortProperties = this.get('sortProps'),
        oldProperties = App.router.get('updateController.queryParams.Hosts');

    this.set('resetStartIndex', false);

    queryParams.pushObjects(this.getPaginationProps());

    savedFilterConditions.forEach(function (filter) {
      var property = filterProperties.findProperty('name', colPropAssoc[filter.iColumn]);
      if (property && filter.value.length > 0 && !filter.skipFilter) {
        var result = {
          key: property.key,
          value: filter.value,
          type: property.type,
          isFilter: true,
          isComponentRelatedFilter: property.isComponentRelatedFilter
        };
        if (filter.type === 'string' && sortProperties.someProperty('name', colPropAssoc[filter.iColumn])) {
          if (Em.isArray(filter.value)) {
            for (var i = 0; i < filter.value.length; i++) {
              filter.value[i] = this.getRegExp(filter.value[i]);
            }
          } else {
            result.value = this.getRegExp(filter.value);
          }
        }
        if (filter.type === 'number' || filter.type === 'ambari-bandwidth') {
          result.type = this.getComparisonType(filter.value);
          result.value = this.getProperValue(filter.value);
        }
        // enter an exact number for RAM filter, need to do a range number match for this
        if (filter.type === 'ambari-bandwidth' && result.type == 'EQUAL' && result.value) {
          var valuePair = this.convertMemoryToRange(filter.value);
          queryParams.push({
            key: result.key,
            value: valuePair[0],
            type: 'MORE'
          });
          queryParams.push({
            key: result.key,
            value: valuePair[1],
            type: 'LESS'
          });
        } else if (filter.type === 'ambari-bandwidth' && result.type != 'EQUAL' && result.value) {
          // enter a comparison type, eg > 1, just do regular match
          result.value = this.convertMemory(filter.value);
          queryParams.push(result);
        } else if (filter.type === 'sub-resource') {
          filter.value.forEach(function (item) {
            queryParams.push({
              key: result.key + "/" + item.property,
              value: item.value,
              type: 'EQUAL'
            });
          }, this);
        } else {
          queryParams.push(result);
        }
      }
    }, this);

    if (!oldProperties.findProperty('isHostDetails')) {
      // shouldn't reset start index after coming back from Host Details page
      if (queryParams.filterProperty('isFilter').length !== oldProperties.filterProperty('isFilter').length) {
        queryParams.findProperty('key', 'from').value = 0;
        this.set('resetStartIndex', true);
      } else {
        queryParams.filterProperty('isFilter').forEach(function (queryParam) {
          var oldProperty = oldProperties.filterProperty('isFilter').findProperty('key', queryParam.key);
          if (!oldProperty || JSON.stringify(oldProperty.value) !== JSON.stringify(queryParam.value)) {
            queryParams.findProperty('key', 'from').value = 0;
            this.set('resetStartIndex', true);
          }
        }, this);
      }
    }

    if (!skipNonFilterProperties) {
      queryParams.pushObjects(this.getSortProps());
    }

    return queryParams;
  },

  /**
   * Return value without predicate
   * @param {String} value
   * @return {String}
   */
  getProperValue: function getProperValue(value) {
    return ['>', '<', '='].contains(value.charAt(0)) ? value.substr(1, value.length) : value;
  },

  /**
   * Return value converted to kilobytes
   * @param {String} value
   * @return {number}
   */
  convertMemory: function convertMemory(value) {
    var scale = value.charAt(value.length - 1);
    // first char may be predicate for comparison
    value = this.getProperValue(value);
    var parsedValue = parseFloat(value);

    if (isNaN(parsedValue)) {
      return value;
    }

    switch (scale) {
      case 'g':
        parsedValue *= 1048576;
        break;
      case 'm':
        parsedValue *= 1024;
        break;
      case 'k':
        break;
      default:
        //default value in GB
        parsedValue *= 1048576;
    }
    return Math.round(parsedValue);
  },

  /**
   * Return value converted to a range of kilobytes
   * @param {String} value
   * @return {Array}
   */
  convertMemoryToRange: function convertMemoryToRange(value) {
    var scale = value.charAt(value.length - 1);
    // first char may be predicate for comparison
    value = this.getProperValue(value);
    var parsedValue = parseFloat(value);
    if (isNaN(parsedValue)) {
      return [0, 0];
    }
    var parsedValuePair = this.rangeConvertNumber(parsedValue, scale);
    var multiplyingFactor = 1;
    switch (scale) {
      case 'g':
        multiplyingFactor = 1048576;
        break;
      case 'm':
        multiplyingFactor = 1024;
        break;
      case 'k':
        break;
      default:
        //default value in GB
        multiplyingFactor = 1048576;
    }
    parsedValuePair[0] = Math.round(parsedValuePair[0] * multiplyingFactor);
    parsedValuePair[1] = Math.round(parsedValuePair[1] * multiplyingFactor);
    return parsedValuePair;
  },

  /**
   * Return value converted to a range of kilobytes
   * eg, return value 1.83 g will target 1.82500 ~ 1.83499 g
   * eg, return value 1.8 k will target 1.7500 ~ 1.8499 k
   * eg, return value 1.8 m will target 1.7500 ~ 1.8499 m
   * @param {number} value
   * @param {String} scale
   * @return {Array}
   */
  rangeConvertNumber: function rangeConvertNumber(value, scale) {
    if (isNaN(value)) {
      return [0, 0];
    }
    var valuePair = [];
    switch (scale) {
      case 'g':
        valuePair = [value - 0.005000, value + 0.004999999];
        break;
      case 'm':
      case 'k':
        valuePair = [value - 0.05000, value + 0.04999];
        break;
      default:
        //default value in GB
        valuePair = [value - 0.005000, value + 0.004999999];
    }
    return valuePair;
  },

  /**
   * Return comparison type depending on populated predicate
   * @param {string} value
   * @return {string}
   */
  getComparisonType: function getComparisonType(value) {
    var comparisonChar = value.charAt(0);
    var result = 'EQUAL';
    if (isNaN(comparisonChar)) {
      switch (comparisonChar) {
        case '>':
          result = 'MORE';
          break;
        case '<':
          result = 'LESS';
          break;
      }
    }
    return result;
  },

  labelValueMap: {},

  /**
   * Filter hosts by componentName of <code>component</code>
   * @param {App.HostComponent} component
   */
  filterByComponent: function filterByComponent(component) {
    if (!component) return;
    var componentName = component.get('componentName');
    var displayName = App.format.role(componentName, false);
    var colPropAssoc = this.get('colPropAssoc');
    var map = this.get('labelValueMap');

    var filterForComponent = {
      iColumn: 15,
      value: componentName + ':ALL',
      type: 'string'
    };
    map[displayName] = componentName;
    map['All'] = 'ALL';
    var filterStr = '"' + displayName + '"' + ': "All"';
    App.db.setFilterConditions(this.get('name'), [filterForComponent]);
    App.db.setComboSearchQuery(this.get('name'), filterStr);
  },

  /**
   * Filter hosts by stack version and state
   * @param {String} displayName
   * @param {Array} states
   */
  filterByStack: function filterByStack(displayName, states) {
    if (Em.isNone(displayName) || Em.isNone(states) || !states.length) return;
    var colPropAssoc = this.get('colPropAssoc');
    var map = this.get('labelValueMap');
    var stateFilterStrs = [];

    var versionFilter = {
      iColumn: 16,
      value: displayName,
      type: 'string'
    };
    var stateFilter = {
      iColumn: 17,
      value: states,
      type: 'string'
    };
    map["Stack Version"] = colPropAssoc[versionFilter.iColumn];
    map["Version State"] = colPropAssoc[stateFilter.iColumn];
    stateFilter.value.forEach(function (state) {
      map[App.HostStackVersion.formatStatus(state)] = state;
      stateFilterStrs.push('"Version State": "' + App.HostStackVersion.formatStatus(state) + '"');
    });
    var versionFilterStr = '"Stack Version": "' + versionFilter.value + '"';
    App.db.setFilterConditions(this.get('name'), [versionFilter, stateFilter]);
    App.db.setComboSearchQuery(this.get('name'), [versionFilterStr, stateFilterStrs.join(' ')].join(' '));
  },

  goToHostAlerts: function goToHostAlerts(event) {
    var host = event && event.context;
    if (host) {
      App.router.transitionTo('main.hosts.hostDetails.alerts', host);
    }
  },

  /**
   * remove selected hosts
   */
  removeHosts: function removeHosts() {
    var hosts = this.get('content');
    var selectedHosts = hosts.filterProperty('isChecked');
    this.get('fullContent').removeObjects(selectedHosts);
  },

  /**
   * remove hosts with id equal host_id
   * @param {String} host_id
   */
  checkRemoved: function checkRemoved(host_id) {
    var hosts = this.get('content');
    var selectedHosts = hosts.filterProperty('id', host_id);
    this.get('fullContent').removeObjects(selectedHosts);
  },

  /**
   * associations between host property and column index
   * @type {Array}
   */
  colPropAssoc: function () {
    var associations = [];
    associations[0] = 'healthClass';
    associations[1] = 'hostName';
    associations[2] = 'ip';
    associations[3] = 'cpu';
    associations[4] = 'memoryFormatted';
    associations[5] = 'loadAvg';
    associations[6] = 'hostComponents';
    associations[7] = 'criticalWarningAlertsCount';
    associations[8] = 'componentsWithStaleConfigsCount';
    associations[9] = 'componentsInPassiveStateCount';
    associations[10] = 'selected';
    associations[11] = 'hostStackVersion';
    associations[12] = 'rack';
    associations[13] = 'services';
    associations[14] = 'state';
    associations[15] = 'componentState';
    associations[16] = 'version';
    associations[17] = 'versionState';
    return associations;
  }.property()

});

});

require.register("controllers/main/host/addHost/step4_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddHostStep4Controller = Em.Controller.extend({

  name: "addHostStep4Controller",
  isConfigGroupLoaded: false,

  loadConfigGroups: function loadConfigGroups() {
    App.ajax.send({
      name: 'config_groups.all_fields',
      sender: this,
      success: 'successLoadingConfigGroup',
      error: 'errorLoadingConfigGroup'
    });
  },

  successLoadingConfigGroup: function successLoadingConfigGroup(data) {
    App.router.get('addHostController.content').set('configGroups', data.items);
    this.set('isConfigGroupLoaded', true);
  },

  errorLoadingConfigGroup: function errorLoadingConfigGroup(data) {
    App.router.get('addHostController.content').set('configGroups', []);
    this.set('isConfigGroupLoaded', true);
  },

  configGroupsLoading: function configGroupsLoading() {
    var df = $.Deferred();
    if (this.get('isConfigGroupLoaded')) {
      df.resolve();
    } else {
      this.interval = setInterval(function () {
        if (this.get('isConfigGroupLoaded')) {
          df.resolve();
          clearInterval(this.interval);
        }
      }.bind(this), 55);
    }
    return df.promise();
  }

});

});

require.register("controllers/main/host/add_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.AddHostController = App.WizardController.extend({

  name: 'addHostController',

  totalSteps: 7,

  /**
   * @type {string}
   */
  displayName: Em.I18n.t('hosts.add.header'),

  /**
   * Used for hiding back button in wizard
   */
  hideBackButton: true,

  /**
   * All wizards data will be stored in this variable
   *
   * cluster - cluster name
   * hosts - hosts, ssh key, repo info, etc.
   * services - services list
   * hostsInfo - list of selected hosts
   * slaveComponentHosts, hostSlaveComponents - info about slave hosts
   * masterComponentHosts - info about master hosts
   * serviceConfigGroups - info about selected config group for service
   * configGroups - all config groups
   * config??? - to be described later
   */
  content: Em.Object.create({
    cluster: null,
    hosts: null,
    installOptions: null,
    services: null,
    slaveComponentHosts: null,
    masterComponentHosts: null,
    serviceConfigProperties: null,
    advancedServiceConfig: null,
    controllerName: 'addHostController',
    serviceConfigGroups: null,
    configGroups: null
  }),

  /**
   * Load data for all steps until <code>current step</code>
   */
  loadMap: {
    '1': [{
      type: 'sync',
      callback: function callback() {
        this.load('hosts');
        this.load('installOptions');
        this.load('cluster');
      }
    }],
    '2': [{
      type: 'sync',
      callback: function callback() {
        this.loadServices();
      }
    }],
    '3': [{
      type: 'async',
      callback: function callback() {
        var self = this,
            dfd = $.Deferred();
        this.loadClients();
        this.loadServices();
        this.loadMasterComponentHosts().done(function () {
          self.loadSlaveComponentHosts();
          self.load('hosts');
          dfd.resolve();
        });
        return dfd.promise();
      }
    }],
    '5': [{
      type: 'sync',
      callback: function callback() {
        this.loadServiceConfigProperties();
        this.getServiceConfigGroups();
      }
    }]
  },

  /**
   * save info about wizard progress, particularly current step of wizard
   * @param currentStep
   * @param completed
   */
  setCurrentStep: function setCurrentStep(currentStep, completed) {
    this._super(currentStep, completed);
    App.clusterStatus.setClusterStatus({
      wizardControllerName: this.get('name'),
      localdb: App.db.data
    });
    var self = this;
    Em.run.next(function () {
      if (self.isConfigGroupsEmpty()) {
        self.disableStep(4);
      }
    });
  },

  /**
   * return new object extended from clusterStatusTemplate
   * @return Object
   */
  getCluster: function getCluster() {
    return jQuery.extend({}, this.get('clusterStatusTemplate'), { name: App.router.getClusterName() });
  },

  /**
   * Remove host from model. Used at <code>Confirm hosts</code> step
   * @param hosts Array of hosts, which we want to delete
   */
  removeHosts: function removeHosts(hosts) {
    var dbHosts = this.getDBProperty('hosts');
    hosts.forEach(function (_hostInfo) {
      var host = _hostInfo.name;
      delete dbHosts[host];
    });
    this.setDBProperty('hosts', dbHosts);
  },

  disableStep: function disableStep(step) {
    this.get('isStepDisabled').findProperty('step', step).set('value', true);
  },

  isConfigGroupsEmpty: function isConfigGroupsEmpty() {
    return !this.get('content.configGroups') || !this.get('content.configGroups').length;
  },

  /**
   * Load services data. Will be used at <code>Select services(step4)</code> step
   */
  loadServices: function loadServices() {
    var services = this.getDBProperty('services');
    if (!services) {
      services = {
        selectedServices: [],
        installedServices: []
      };
      App.StackService.find().forEach(function (item) {
        var isInstalled = App.Service.find().someProperty('serviceName', item.get('serviceName'));
        item.set('isSelected', isInstalled);
        item.set('isInstalled', isInstalled);
        if (isInstalled) {
          services.selectedServices.push(item.get('serviceName'));
          services.installedServices.push(item.get('serviceName'));
        }
      }, this);
      this.setDBProperty('services', services);
    } else {
      App.StackService.find().forEach(function (item) {
        var isSelected = services.selectedServices.contains(item.get('serviceName'));
        var isInstalled = services.installedServices.contains(item.get('serviceName'));
        item.set('isSelected', isSelected);
        item.set('isInstalled', isInstalled);
      }, this);
    }
    this.set('content.services', App.StackService.find());
  },

  /**
    * Load slave component hosts data for using in required step controllers
    * TODO move to mixin
    */
  loadSlaveComponentHosts: function loadSlaveComponentHosts() {
    var props = this.getDBProperties(['slaveComponentHosts', 'hosts']);
    var slaveComponentHosts = props.slaveComponentHosts || [];
    if (slaveComponentHosts.length) {
      var hosts = props.hosts || {},
          host_names = Em.keys(hosts);
      slaveComponentHosts.forEach(function (component) {
        component.hosts.forEach(function (host) {
          //Em.set(host, 'hostName', hosts[host.host_id].name);
          for (var i = 0; i < host_names.length; i++) {
            if (hosts[host_names[i]].id === host.host_id) {
              host.hostName = host_names[i];
              break;
            }
          }
        });
      });
    }
    this.set("content.slaveComponentHosts", slaveComponentHosts);
  },

  /**
   * Generate clients list for selected services and save it to model
   */
  saveClients: function saveClients() {
    var serviceComponents = App.StackServiceComponent.find();
    var services = this.get('content.services').filterProperty('isInstallable').filterProperty('isSelected');
    var clients = this.getClientsToInstall(services, serviceComponents);
    this.setDBProperty('clientInfo', clients);
    this.set('content.clients', clients);
  },

  /**
   * get list of clients which will be installed on host
   * @param services {Array} of service objects
   * @param components {Array} of component objects
   * @returns {Array} returns array of clients
   * @method getClientsToInstall;
   */
  getClientsToInstall: function getClientsToInstall(services, components) {
    var clients = [];
    services.forEach(function (_service) {
      var serviceClients = components.filter(function (component) {
        return component.get('serviceName') == _service.get('serviceName') && component.get('isClient');
      });
      if (serviceClients.length) {
        serviceClients.forEach(function (client) {
          clients.push({
            component_name: client.get('componentName'),
            display_name: client.get('displayName'),
            isInstalled: false
          });
        });
      }
    }, this);
    return clients;
  },
  /**
   *  Apply config groups from step4 Configurations
   */
  applyConfigGroup: function applyConfigGroup() {
    var serviceConfigGroups = this.get('content.configGroups');
    serviceConfigGroups.forEach(function (group) {
      if (group.configGroups.someProperty('ConfigGroup.group_name', group.selectedConfigGroup)) {
        var configGroup = group.configGroups.findProperty('ConfigGroup.group_name', group.selectedConfigGroup);
        group.hosts.forEach(function (host) {
          configGroup.ConfigGroup.hosts.push({
            host_name: host
          });
        }, this);
        delete configGroup.href;
        App.ajax.send({
          name: 'config_groups.update_config_group',
          sender: this,
          data: {
            id: configGroup.ConfigGroup.id,
            configGroup: configGroup
          }
        });
      }
    }, this);
  },

  /**
   * Load information about selected config groups
   */
  getServiceConfigGroups: function getServiceConfigGroups() {
    var serviceConfigGroups = this.getDBProperty('serviceConfigGroups');
    this.set('content.configGroups', serviceConfigGroups);
  },

  /**
   * Save information about selected config groups
   */
  saveServiceConfigGroups: function saveServiceConfigGroups() {
    this.setDBProperty('serviceConfigGroups', this.get('content.configGroups'));
  },

  /**
   * Set content.configGroups for step4
   */
  loadServiceConfigGroups: function loadServiceConfigGroups() {
    var selectedServices = [];
    this.loadServiceConfigGroupsBySlaves(selectedServices);
    this.loadServiceConfigGroupsByClients(selectedServices);
    this.sortServiceConfigGroups(selectedServices);
    this.set('content.configGroups', selectedServices);
  },
  /**
   * sort config groups by name
   * @param selectedServices
   */
  sortServiceConfigGroups: function sortServiceConfigGroups(selectedServices) {
    selectedServices.forEach(function (selectedService) {
      selectedService.configGroups.sort(function (cfgA, cfgB) {
        if (cfgA.ConfigGroup.group_name < cfgB.ConfigGroup.group_name) return -1;
        if (cfgA.ConfigGroup.group_name > cfgB.ConfigGroup.group_name) return 1;
        return 0;
      });
    });
  },
  /**
   * load service config groups by slave components,
   * push them into selectedServices
   * @param selectedServices
   */
  loadServiceConfigGroupsBySlaves: function loadServiceConfigGroupsBySlaves(selectedServices) {
    var slaveComponentHosts = this.get('content.slaveComponentHosts');
    if (slaveComponentHosts && slaveComponentHosts.length > 0) {
      slaveComponentHosts.forEach(function (slave) {
        if (slave.hosts.length > 0) {
          if (slave.componentName !== "CLIENT") {
            var service = App.StackServiceComponent.find(slave.componentName).get('stackService');
            var serviceName = service.get('serviceName');
            var configGroups = this.get('content.configGroups').filterProperty('ConfigGroup.tag', serviceName);
            var configGroupsNames = configGroups.mapProperty('ConfigGroup.group_name');
            var defaultGroupName = 'Default';
            var selectedService = selectedServices.findProperty('serviceId', serviceName);
            configGroupsNames.unshift(defaultGroupName);
            if (selectedService) {
              Em.set(selectedService, 'hosts', Em.getWithDefault(selectedService, 'hosts', []).concat(slave.hosts.mapProperty('hostName')).uniq());
            } else {
              selectedServices.push({
                serviceId: serviceName,
                displayName: service.get('displayName'),
                hosts: slave.hosts.mapProperty('hostName'),
                configGroupsNames: configGroupsNames,
                configGroups: configGroups,
                selectedConfigGroup: defaultGroupName
              });
            }
          }
        }
      }, this);
      return true;
    }
    return false;
  },
  /**
   * load service config groups by clients,
   * push them into selectedServices
   * @param selectedServices
   */
  loadServiceConfigGroupsByClients: function loadServiceConfigGroupsByClients(selectedServices) {
    var slaveComponentHosts = this.get('content.slaveComponentHosts');
    var clients = this.get('content.clients');
    var client = slaveComponentHosts && slaveComponentHosts.findProperty('componentName', 'CLIENT');
    var selectedClientHosts = client && client.hosts.mapProperty('hostName');
    if (clients && selectedClientHosts && clients.length > 0 && selectedClientHosts.length > 0) {
      this.loadClients();
      clients.forEach(function (client) {
        var service = App.StackServiceComponent.find(client.component_name).get('stackService');
        var serviceName = service.get('serviceName');
        var serviceMatch = selectedServices.findProperty('serviceId', serviceName);
        if (serviceMatch) {
          serviceMatch.hosts = serviceMatch.hosts.concat(selectedClientHosts).uniq();
        } else {
          var configGroups = this.get('content.configGroups').filterProperty('ConfigGroup.tag', serviceName);
          var configGroupsNames = configGroups.mapProperty('ConfigGroup.group_name').sort();
          var defaultGroupName = 'Default';
          configGroupsNames.unshift(defaultGroupName);
          selectedServices.push({
            serviceId: serviceName,
            displayName: service.get('displayName'),
            hosts: selectedClientHosts,
            configGroupsNames: configGroupsNames,
            configGroups: configGroups,
            selectedConfigGroup: defaultGroupName
          });
        }
      }, this);
      return true;
    }
    return false;
  },

  loadServiceConfigProperties: function loadServiceConfigProperties() {
    var serviceConfigProperties = App.db.get('AddService', 'serviceConfigProperties');
    if (!serviceConfigProperties || !serviceConfigProperties.length) {
      serviceConfigProperties = App.db.get('Installer', 'serviceConfigProperties');
    }
    this.set('content.serviceConfigProperties', serviceConfigProperties);
  },

  /**
   * Remove all loaded data.
   * Created as copy for App.router.clearAllSteps
   */
  clearAllSteps: function clearAllSteps() {
    this.clearInstallOptions();
    // clear temporary information stored during the install
    this.set('content.cluster', this.getCluster());
  },

  clearStorageData: function clearStorageData() {
    this._super();
    this.resetDbNamespace();
  },

  /**
   * Clear all temporary data
   */
  finish: function finish() {
    this.clearAllSteps();
    this.clearStorageData();
  },

  /**
   * send request to server in order to install services
   * @param isRetry
   * @param callback
   * @param errorCallback
   */
  installServices: function installServices(isRetry, callback, errorCallback) {
    callback = callback || Em.K;
    this.set('content.cluster.oldRequestsId', []);
    this.set('content.cluster.status', 'PENDING');
    var clusterName = this.get('content.cluster.name');
    var hostNames = [];
    var hosts = this.getDBProperty('hosts');
    for (var hostname in hosts) {
      if (!hosts[hostname].isInstalled) {
        hostNames.push(hostname);
      }
    }
    if (!clusterName || hostNames.length === 0) return false;

    App.ajax.send({
      name: "common.host_components.update",
      sender: this,
      data: {
        "context": Em.I18n.t('requestInfo.installComponents'),
        "query": "HostRoles/host_name.in(" + hostNames.join(',') + ")",
        "HostRoles": { "state": "INSTALLED" },
        "level": "HOST_COMPONENT"
      },
      success: 'installServicesSuccessCallback',
      error: 'installServicesErrorCallback'
    }).then(callback, errorCallback || callback);
    return true;
  }
});

});

require.register("controllers/main/host/bulk_operations_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

var App = require('app');
var batchUtils = require('utils/batch_scheduled_requests');
var hostsManagement = require('utils/hosts');
var O = Em.Object;

/**
 * @class BulkOperationsController
 */
App.BulkOperationsController = Em.Controller.extend({

  name: 'bulkOperationsController',

  /**
   * Bulk operation wrapper
   * @param {Object} operationData - data about bulk operation (action, hosts or hostComponents etc)
   * @param {Array} hosts - list of affected hosts
   * @method bulkOperation
   */
  bulkOperation: function bulkOperation(operationData, hosts) {
    if (operationData.componentNameFormatted) {
      if (operationData.action === 'RESTART') {
        this.bulkOperationForHostComponentsRestart(operationData, hosts);
      } else if (operationData.action === 'ADD') {
        this.bulkOperationForHostComponentsAdd(operationData, hosts);
      } else if (operationData.action === 'DELETE') {
        this.bulkOperationForHostComponentsDelete(operationData, hosts);
      } else {
        if (operationData.action.indexOf('DECOMMISSION') == -1) {
          this.bulkOperationForHostComponents(operationData, hosts);
        } else {
          this.bulkOperationForHostComponentsDecommission(operationData, hosts);
        }
      }
    } else {
      if (operationData.action === 'SET_RACK_INFO') {
        this.bulkOperationForHostsSetRackInfo(operationData, hosts);
      } else {
        if (operationData.action === 'RESTART') {
          this.bulkOperationForHostsRestart(operationData, hosts);
        } else if (operationData.action === 'REINSTALL') {
          this.bulkOperationForHostsReinstall(operationData, hosts);
        } else if (operationData.action === 'DELETE') {
          this._bulkOperationForHostsDelete(hosts);
        } else if (operationData.action === 'CONFIGURE') {
          this.bulkOperationForHostsRefreshConfig(operationData, hosts);
        } else {
          if (operationData.action === 'PASSIVE_STATE') {
            this.bulkOperationForHostsPassiveState(operationData, hosts);
          } else {
            this.bulkOperationForHosts(operationData, hosts);
          }
        }
      }
    }
  },

  /**
   * Bulk operation (start/stop all) for selected hosts
   * @param {Object} operationData - data about bulk operation (action, hostComponents etc)
   * @param {Array} hosts - list of affected hosts
   * @return {$.ajax}
   */
  bulkOperationForHosts: function bulkOperationForHosts(operationData, hosts) {
    var self = this;
    return batchUtils.getComponentsFromServer({
      hosts: hosts.mapProperty('hostName'),
      passiveState: 'OFF',
      displayParams: ['host_components/HostRoles/component_name']
    }, function (data) {
      return self._getComponentsFromServerForHostsCallback(operationData, data);
    });
  },

  /**
   * run Bulk operation (start/stop all) for selected hosts
   * after host and components are loaded
   * @param operationData
   * @param data
   */
  _getComponentsFromServerForHostsCallback: function _getComponentsFromServerForHostsCallback(operationData, data) {
    var query = [];
    var hostNames = [];
    var hostsMap = {};
    var clients = App.components.get('clients');

    data.items.forEach(function (host) {
      host.host_components.forEach(function (hostComponent) {
        if (!clients.contains(hostComponent.HostRoles.component_name)) {
          if (hostsMap[host.Hosts.host_name]) {
            hostsMap[host.Hosts.host_name].push(hostComponent.HostRoles.component_name);
          } else {
            hostsMap[host.Hosts.host_name] = [hostComponent.HostRoles.component_name];
          }
        }
      });
    });

    var nn_hosts = [];
    for (var hostName in hostsMap) {
      if (hostsMap.hasOwnProperty(hostName)) {
        var subQuery = '(HostRoles/component_name.in(%@)&HostRoles/host_name=' + hostName + ')';
        var components = hostsMap[hostName];

        if (components.length) {
          if (components.contains('NAMENODE')) {
            nn_hosts.push(hostName);
          }
          query.push(subQuery.fmt(components.join(',')));
        }
        hostNames.push(hostName);
      }
    }
    hostNames = hostNames.join(",");
    if (query.length) {
      query = query.join('|');
      var self = this;
      // if NameNode included, check HDFS NameNode checkpoint before stop NN
      var isHDFSStarted = 'STARTED' === App.Service.find('HDFS').get('workStatus');

      var request = function request() {
        return App.ajax.send({
          name: 'common.host_components.update',
          sender: self,
          data: {
            query: query,
            HostRoles: {
              state: operationData.action
            },
            context: operationData.message,
            hostName: hostNames,
            noOpsMessage: Em.I18n.t('hosts.host.maintainance.allComponents.context')
          },
          success: 'bulkOperationForHostComponentsSuccessCallback'
        });
      };

      if (operationData.action === 'INSTALLED' && isHDFSStarted) {
        if (nn_hosts.length == 1) {
          return App.router.get('mainHostDetailsController').checkNnLastCheckpointTime(request, nn_hosts[0]);
        }
        if (nn_hosts.length > 1) {
          // HA or federation enabled
          return App.router.get('mainServiceItemController').checkNnLastCheckpointTime(request);
        }
      }
      return request();
    } else {
      return App.ModalPopup.show({
        header: Em.I18n.t('rolling.nothingToDo.header'),
        body: Em.I18n.t('rolling.nothingToDo.body').format(Em.I18n.t('hosts.host.maintainance.allComponents.context')),
        secondary: false
      });
    }
  },

  bulkOperationForHostsSetRackInfo: function bulkOperationForHostsSetRackInfo(operationData, hosts) {
    return hostsManagement.setRackInfo(operationData, hosts);
  },

  /**
   * Bulk restart for selected hosts
   * @param {Object} operationData - data about bulk operation (action, hostComponents etc)
   * @param {Ember.Enumerable} hosts - list of affected hosts
   */
  bulkOperationForHostsRestart: function bulkOperationForHostsRestart(operationData, hosts) {
    return batchUtils.getComponentsFromServer({
      passiveState: 'OFF',
      hosts: hosts.mapProperty('hostName'),
      displayParams: ['host_components/HostRoles/component_name']
    }, this._getComponentsFromServerForRestartCallback);
  },

  /**
   *
   * @param {object} data
   * @private
   * @method _getComponentsFromServerCallback
   */
  _getComponentsFromServerForRestartCallback: function _getComponentsFromServerForRestartCallback(data) {
    var hostComponents = [];
    data.items.forEach(function (host) {
      host.host_components.forEach(function (hostComponent) {
        hostComponents.push(O.create({
          componentName: hostComponent.HostRoles.component_name,
          hostName: host.Hosts.host_name
        }));
      });
    });
    // if NameNode included, check HDFS NameNode checkpoint before restart NN
    var isHDFSStarted = 'STARTED' === App.Service.find('HDFS').get('workStatus');
    var namenodes = hostComponents.filterProperty('componentName', 'NAMENODE');
    var nn_count = namenodes.get('length');

    if (nn_count == 1 && isHDFSStarted) {
      var hostName = namenodes.get('firstObject.hostName');
      App.router.get('mainHostDetailsController').checkNnLastCheckpointTime(function () {
        batchUtils.restartHostComponents(hostComponents, Em.I18n.t('rollingrestart.context.allOnSelectedHosts'), "HOST");
      }, hostName);
    } else {
      if (nn_count > 1 && isHDFSStarted) {
        // HA or federation enabled
        App.router.get('mainServiceItemController').checkNnLastCheckpointTime(function () {
          batchUtils.restartHostComponents(hostComponents, Em.I18n.t('rollingrestart.context.allOnSelectedHosts'), "HOST");
        });
      } else {
        batchUtils.restartHostComponents(hostComponents, Em.I18n.t('rollingrestart.context.allOnSelectedHosts'), "HOST");
      }
    }
  },

  /**
   * Bulk reinstall failed components for selected hosts
   * @param {Object} operationData - data about bulk operation (action, hostComponents etc)
   * @param {Ember.Enumerable} hosts - list of affected hosts
   */
  bulkOperationForHostsReinstall: function bulkOperationForHostsReinstall(operationData, hosts) {
    var self = this;
    App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
      return App.ajax.send({
        name: 'common.host_components.update',
        sender: self,
        data: {
          HostRoles: {
            state: 'INSTALLED'
          },
          query: 'HostRoles/host_name.in(' + hosts.mapProperty('hostName').join(',') + ')&HostRoles/state=INSTALL_FAILED',
          context: operationData.message,
          noOpsMessage: Em.I18n.t('hosts.host.maintainance.reinstallFailedComponents.context')
        },
        success: 'bulkOperationForHostComponentsSuccessCallback',
        showLoadingPopup: true
      });
    });
  },

  /**
  * Check which hosts can be deleted and warn the user about it in advance
  * @param {Ember.Enumerable} hosts - list of affected hosts
  */
  _bulkOperationForHostsDelete: function _bulkOperationForHostsDelete(hosts) {
    var self = this,
        hostNamesToDelete = [],
        hostsNotToDelete = [];
    var createNonDeletableComponents = function createNonDeletableComponents(hostName, message) {
      return Em.Object.create({
        error: {
          key: hostName,
          message: message
        },
        isCollapsed: true,
        isBodyVisible: Em.computed.ifThenElse('isCollapsed', 'display: none;', 'display: block;')
      });
    };
    hosts.forEach(function (host) {
      var hostComponents = App.HostComponent.find().filterProperty('hostName', host.hostName);
      var hostInfo = App.router.get('mainHostDetailsController').getHostComponentsInfo(hostComponents);
      console.dir(hostInfo);
      if (hostInfo.nonDeletableComponents.length > 0) {
        hostsNotToDelete.push(createNonDeletableComponents(host.hostName, Em.I18n.t('hosts.bulkOperation.deleteHosts.nonDeletableComponents').format(hostInfo.nonDeletableComponents.join(", "))));
      } else if (hostInfo.nonAddableMasterComponents.length > 0) {
        hostsNotToDelete.push(createNonDeletableComponents(host.hostName, Em.I18n.t('hosts.bulkOperation.deleteHosts.nonAddableMasterComponents').format(hostInfo.nonAddableMasterComponents.join(", "))));
      } else if (hostInfo.lastMasterComponents.length > 0) {
        hostsNotToDelete.push(createNonDeletableComponents(host.hostName, Em.I18n.t('hosts.bulkOperation.deleteHosts.lastMasterComponents').format(hostInfo.lastMasterComponents.join(", "))));
      } else if (hostInfo.runningComponents.length > 0) {
        hostsNotToDelete.push(createNonDeletableComponents(host.hostName, Em.I18n.t('hosts.bulkOperation.deleteHosts.runningComponents').format(hostInfo.runningComponents.join(", "))));
      } else {
        hostNamesToDelete.push(host.hostName);
      }
    });

    return App.ModalPopup.show({
      header: hostNamesToDelete.length ? Em.I18n.t('hosts.bulkOperation.deleteHosts.confirm.header') : Em.I18n.t('rolling.nothingToDo.header'),
      primary: hostNamesToDelete.length ? Em.I18n.t('common.next') : null,
      primaryClass: 'btn-default',
      onPrimary: function onPrimary() {
        this._super();
        self.bulkOperationForHostsDelete(hostNamesToDelete);
      },
      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/bulk_add_delete_confirm_popup'),
        modifyMessage: Em.I18n.t('hosts.bulkOperation.deleteHosts.confirm.delete'),
        skipMessage: hostNamesToDelete.length ? Em.I18n.t('hosts.bulkOperation.deleteHosts.cannot.delete1') : Em.I18n.t('hosts.bulkOperation.deleteHosts.cannot.delete2'),
        skippedHosts: hostsNotToDelete.length ? hostsNotToDelete : null,
        hostsToModify: hostNamesToDelete.length ? hostNamesToDelete.join("\n") : null,
        onToggleHost: function onToggleHost(host) {
          host.contexts[0].toggleProperty('isCollapsed');
        }
      })
    });
  },

  /**
   * Bulk refresh configs for selected hosts
   * @param {Object} operationData - data about bulk operation (action, hostComponents etc)
   * @param {Ember.Enumerable} hosts - list of affected/selected hosts
   */
  bulkOperationForHostsRefreshConfig: function bulkOperationForHostsRefreshConfig(operationData, hosts) {
    return batchUtils.getComponentsFromServer({
      passiveState: 'OFF',
      hosts: hosts.mapProperty('hostName'),
      displayParams: ['host_components/HostRoles/component_name']
    }, this._getComponentsFromServerForRefreshConfigsCallback);
  },

  /**
   *
   * @param {object} data
   * @private
   * @method _getComponentsFromServerForRefreshConfigsCallback
   */
  _getComponentsFromServerForRefreshConfigsCallback: function _getComponentsFromServerForRefreshConfigsCallback(data) {
    var hostComponents = [];
    var clients = App.components.get('clients');
    data.items.forEach(function (host) {
      host.host_components.forEach(function (hostComponent) {
        if (clients.contains(hostComponent.HostRoles.component_name)) {
          hostComponents.push(O.create({
            componentName: hostComponent.HostRoles.component_name,
            hostName: host.Hosts.host_name
          }));
        }
      });
    });
    batchUtils.restartHostComponents(hostComponents, Em.I18n.t('rollingrestart.context.configs.allOnSelectedHosts'), "HOST");
  },

  /**
   * Bulk delete selected hosts
   * @param {String} hosts - list of affected host names
   */
  bulkOperationForHostsDelete: function bulkOperationForHostsDelete(hosts) {
    var confirmKey = 'delete',
        self = this;
    App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
      return App.ModalPopup.show({
        header: Em.I18n.t('hosts.bulkOperation.deleteHosts.confirmation.header'),
        confirmInput: '',
        disablePrimary: Em.computed.notEqual('confirmInput', confirmKey),
        primary: Em.I18n.t('common.confirm'),
        primaryClass: 'btn-warning',
        onPrimary: function onPrimary() {
          this._super();
          return App.ajax.send({
            name: 'common.hosts.delete',
            sender: self,
            data: {
              query: 'Hosts/host_name.in(' + hosts.join(',') + ')',
              hosts: hosts
            },
            success: 'bulkOperationForHostsDeleteCallback',
            error: 'bulkOperationForHostsDeleteCallback',
            showLoadingPopup: true
          });
        },
        bodyClass: Em.View.extend({
          templateName: require('templates/main/host/delete_hosts_popup'),
          hostNames: hosts,
          typeMessage: Em.I18n.t('services.service.confirmDelete.popup.body.type').format(confirmKey)
        })
      });
    });
  },

  /**
   * Show popup after bulk delete hosts
   * @method bulkOperationForHostsDeleteCallback
   */
  bulkOperationForHostsDeleteCallback: function bulkOperationForHostsDeleteCallback(arg0, arg1, arg2, arg3, arg4) {
    var deletedHosts = [];
    var undeletableHosts = [];
    if (arg1 == "error") {
      var request = arg0;
      var params = arg4;
      var response = JSON.parse(request.responseText);
      var host = Ember.Object.create({
        error: {
          key: params.hosts[0],
          code: response.status,
          message: response.message
        },
        isCollapsed: true,
        isBodyVisible: Em.computed.ifThenElse('isCollapsed', 'display: none;', 'display: block;')
      });
      undeletableHosts.push(host);
    } else {
      var data = arg0;
      var params = arg2;
      if (data) {
        data.deleteResult.forEach(function (host) {
          if (!host.deleted) {
            var _host = Ember.Object.create({
              error: host.error,
              isCollapsed: true,
              isBodyVisible: Em.computed.ifThenElse('isCollapsed', 'display: none;', 'display: block;')
            });
            undeletableHosts.push(_host);
          }
        });
      }
      deletedHosts = params.hosts;
    }

    return App.ModalPopup.show({
      header: Em.I18n.t('hosts.bulkOperation.deleteHosts.result.header'),

      secondary: null,

      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/delete_hosts_result_popup'),
        message: Em.I18n.t('hosts.bulkOperation.deleteHosts.dryRun.message').format(undeletableHosts.length),
        undeletableHosts: undeletableHosts,
        deletedHosts: deletedHosts,
        onToggleHost: function onToggleHost(host) {
          host.contexts[0].toggleProperty('isCollapsed');
        }
      }),

      completeDelete: function completeDelete() {
        if (arg1 !== 'error') {
          App.db.unselectHosts(arg2.hosts);
        }
        location.reload();
      },


      onPrimary: function onPrimary() {
        this.completeDelete();
        this._super();
      },

      onClose: function onClose() {
        this.completeDelete();
        this._super();
      }
    });
  },

  /**
   * Bulk turn on/off passive state for selected hosts
   * @param {Object} operationData - data about bulk operation (action, hostComponents etc)
   * @param {Array} hosts - list of affected hosts
   */
  bulkOperationForHostsPassiveState: function bulkOperationForHostsPassiveState(operationData, hosts) {
    var self = this;

    return batchUtils.getComponentsFromServer({
      hosts: hosts.mapProperty('hostName'),
      displayParams: ['Hosts/maintenance_state']
    }, function (data) {
      return self._getComponentsFromServerForPassiveStateCallback(operationData, data);
    });
  },

  /**
   *
   * @param {object} operationData
   * @param {object} data
   * @returns {$.ajax|App.ModalPopup}
   * @private
   * @method _getComponentsFromServerForPassiveStateCallback
   */
  _getComponentsFromServerForPassiveStateCallback: function _getComponentsFromServerForPassiveStateCallback(operationData, data) {
    var hostNames = [];

    data.items.forEach(function (host) {
      if (host.Hosts.maintenance_state !== operationData.state) {
        hostNames.push(host.Hosts.host_name);
      }
    });
    if (hostNames.length) {
      return App.ajax.send({
        name: 'bulk_request.hosts.passive_state',
        sender: this,
        data: {
          hostNames: hostNames.join(','),
          passive_state: operationData.state,
          requestInfo: operationData.message
        },
        success: 'updateHostPassiveState'
      });
    }
    return App.ModalPopup.show({
      header: Em.I18n.t('rolling.nothingToDo.header'),
      body: Em.I18n.t('hosts.bulkOperation.passiveState.nothingToDo.body'),
      secondary: false
    });
  },

  updateHostPassiveState: function updateHostPassiveState(data, opt, params) {
    return batchUtils.infoPassiveState(params.passive_state);
  },

  /**
   * bulk add for selected hostComponent
   * @param {Object} operationData - data about bulk operation (action, hostComponent etc)
   * @param {Array} hosts - list of affected hosts
   */
  bulkOperationForHostComponentsAdd: function bulkOperationForHostComponentsAdd(operationData, hosts) {
    var self = this;
    return batchUtils.getComponentsFromServer({
      components: [operationData.componentName],
      hosts: hosts.mapProperty('hostName')
    }, function (data) {
      return self._getComponentsFromServerForHostComponentsAddCallback(operationData, data, hosts);
    });
  },

  _getComponentsFromServerForHostComponentsAddCallback: function _getComponentsFromServerForHostComponentsAddCallback(operationData, data, hosts) {
    var self = this;

    var allHostsWithComponent = data.items.mapProperty('Hosts.host_name');
    var hostsWithComponent = [];
    hosts.forEach(function (host) {
      var isNotHeartBeating = host.state === 'HEARTBEAT_LOST';
      if (allHostsWithComponent.contains(host.hostName) || isNotHeartBeating) {
        hostsWithComponent.push(Em.Object.create({
          error: {
            key: host.hostName,
            message: isNotHeartBeating ? Em.I18n.t('hosts.bulkOperation.confirmation.add.component.noHeartBeat.skip') : Em.I18n.t('hosts.bulkOperation.confirmation.add.component.skip').format(operationData.componentNameFormatted)
          },
          isCollapsed: true,
          isBodyVisible: Em.computed.ifThenElse('isCollapsed', 'display: none;', 'display: block;')
        }));
      }
    });
    var hostsWithOutComponent = hosts.filter(function (host) {
      return !hostsWithComponent.findProperty('error.key', host.hostName);
    });

    hostsWithOutComponent = hostsWithOutComponent.mapProperty('hostName');

    return App.ModalPopup.show({
      header: hostsWithOutComponent.length ? Em.I18n.t('hosts.bulkOperation.confirmation.header') : Em.I18n.t('rolling.nothingToDo.header'),
      primary: hostsWithOutComponent.length ? Em.I18n.t('hosts.host.addComponent.popup.confirm') : null,

      onPrimary: function onPrimary() {
        self.bulkAddHostComponents(operationData, hostsWithOutComponent);
        this._super();
      },
      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/bulk_add_delete_confirm_popup'),
        modifyMessage: Em.I18n.t('hosts.bulkOperation.confirmation.add.component').format(operationData.componentNameFormatted),
        skipMessage: hostsWithOutComponent.length ? Em.I18n.t('hosts.bulkOperation.confirmation.cannot.add1') : Em.I18n.t('hosts.bulkOperation.confirmation.cannot.add2').format(operationData.componentNameFormatted),
        hostsToModify: hostsWithOutComponent.length ? hostsWithOutComponent.join("\n") : null,
        skippedHosts: hostsWithComponent.length ? hostsWithComponent : null,
        onToggleHost: function onToggleHost(host) {
          host.contexts[0].toggleProperty('isCollapsed');
        }
      })
    });
  },
  /**
   * Bulk add for selected hostComponent
   * @param {Object} operationData - data about bulk operation (action, hostComponent etc)
   * @param {Array} hostNames - list of affected hosts' names
   */
  bulkAddHostComponents: function bulkAddHostComponents(operationData, hostNames) {
    var self = this;
    App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
      App.ajax.send({
        name: 'host.host_component.add_new_components',
        sender: self,
        data: {
          data: JSON.stringify({
            RequestInfo: {
              query: 'Hosts/host_name.in(' + hostNames.join(',') + ')'
            },
            Body: {
              host_components: [{
                HostRoles: {
                  component_name: operationData.componentName
                }
              }]
            }
          }),
          context: operationData.message + ' ' + operationData.componentNameFormatted
        },
        success: 'bulkOperationForHostComponentsAddSuccessCallback',
        showLoadingPopup: true
      });
    });
  },

  bulkOperationForHostComponentsAddSuccessCallback: function bulkOperationForHostComponentsAddSuccessCallback(data, opt, params) {
    App.ajax.send({
      name: 'common.host_components.update',
      sender: this,
      data: {
        query: 'HostRoles/state=INIT',
        HostRoles: {
          state: 'INSTALLED'
        },
        context: params.context
      },
      success: 'bulkOperationForHostComponentsSuccessCallback'
    });
  },

  /**
   * Confirm bulk delete for selected hostComponent
   * @param {Object} operationData - data about bulk operation (action, hostComponent etc)
   * @param {Array} hosts - list of affected hosts
   */
  bulkOperationForHostComponentsDelete: function bulkOperationForHostComponentsDelete(operationData, hosts) {
    var self = this;
    return batchUtils.getComponentsFromServer({
      components: [operationData.componentName],
      hosts: hosts.mapProperty('hostName'),
      displayParams: ['host_components/HostRoles/state']
    }, function (data) {
      return self._getComponentsFromServerForHostComponentsDeleteCallback(operationData, data, hosts);
    });
  },

  _getComponentsFromServerForHostComponentsDeleteCallback: function _getComponentsFromServerForHostComponentsDeleteCallback(operationData, data, requestedHosts) {
    var self = this;
    var minToInstall = App.StackServiceComponent.find(operationData.componentName).get('minToInstall');
    var installedCount = App.HostComponent.getCount(operationData.componentName, 'totalCount');
    var installedHosts = data.items.mapProperty('Hosts.host_name');
    var hostsToDelete = data.items.filter(function (host) {
      var state = host.host_components[0].HostRoles.state;
      return [App.HostComponentStatus.stopped, App.HostComponentStatus.unknown, App.HostComponentStatus.install_failed, App.HostComponentStatus.upgrade_failed, App.HostComponentStatus.init].contains(state);
    }).mapProperty('Hosts.host_name');

    if (installedCount - hostsToDelete.length < minToInstall) {
      return App.ModalPopup.show({
        header: Em.I18n.t('rolling.nothingToDo.header'),
        body: Em.I18n.t('hosts.bulkOperation.confirmation.delete.component.minimum.body').format(minToInstall, operationData.componentNameFormatted),
        secondary: false
      });
    }

    var hostsNotToDelete = [];

    requestedHosts.mapProperty('hostName').forEach(function (host) {
      if (!hostsToDelete.contains(host)) {
        var hostToSkip = Em.Object.create({
          error: {
            key: host,
            message: null
          },
          isCollapsed: true,
          isBodyVisible: Em.computed.ifThenElse('isCollapsed', 'display: none;', 'display: block;')
        });
        if (installedHosts.contains(host)) {
          hostToSkip.error.message = Em.I18n.t('hosts.bulkOperation.confirmation.delete.component.notStopped').format(operationData.componentNameFormatted);
        } else {
          hostToSkip.error.message = Em.I18n.t('hosts.bulkOperation.confirmation.delete.component.notInstalled').format(operationData.componentNameFormatted);
        }
        hostsNotToDelete.push(hostToSkip);
      }
    });

    return App.ModalPopup.show({
      header: hostsToDelete.length ? Em.I18n.t('hosts.bulkOperation.confirmation.header') : Em.I18n.t('rolling.nothingToDo.header'),
      primary: hostsToDelete.length ? Em.I18n.t('hosts.host.deleteComponent.popup.confirm') : null,
      primaryClass: 'btn-warning',

      onPrimary: function onPrimary() {
        self.bulkDeleteHostComponents(operationData, hostsToDelete);
        this._super();
      },
      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/bulk_add_delete_confirm_popup'),
        modifyMessage: Em.I18n.t('hosts.bulkOperation.confirmation.delete.component').format(operationData.componentNameFormatted),
        skipMessage: hostsToDelete.length ? Em.I18n.t('hosts.bulkOperation.confirmation.delete.component.cannot1') : Em.I18n.t('hosts.bulkOperation.confirmation.delete.component.cannot2').format(operationData.componentNameFormatted),
        hostsToModify: hostsToDelete.length ? hostsToDelete.join("\n") : null,
        skippedHosts: hostsNotToDelete.length ? hostsNotToDelete : null,
        onToggleHost: function onToggleHost(host) {
          host.contexts[0].toggleProperty('isCollapsed');
        }
      })
    });
  },

  /**
   * Bulk delete for selected hostComponent
   * @param {Object} operationData - data about bulk operation (action, hostComponent etc)
   * @param {Array} hostNames - list of affected hosts' names
   */
  bulkDeleteHostComponents: function bulkDeleteHostComponents(operationData, hostNames) {
    var self = this;
    App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
      App.ajax.send({
        name: 'host.host_component.delete_components',
        sender: self,
        data: {
          hostNames: hostNames,
          componentName: operationData.componentName,
          data: JSON.stringify({
            RequestInfo: {
              query: 'HostRoles/host_name.in(' + hostNames.join(',') + ')&HostRoles/component_name.in(' + operationData.componentName + ')'
            }
          })
        },
        success: 'bulkOperationForHostComponentsDeleteCallback'
      });
    });
  },

  /**
   * Show popup after bulk delete host_components
   * @method bulkOperationForHostComponentsDeleteCallback
   */
  bulkOperationForHostComponentsDeleteCallback: function bulkOperationForHostComponentsDeleteCallback(arg0, arg1, arg2, arg3, arg4) {
    var deletedHosts = [];
    var undeletableHosts = [];
    var componentName = arg2.componentName;
    if (arg1 == "error") {
      var request = arg0;
      var params = arg4;
      var response = JSON.parse(request.responseText);
      var host = Ember.Object.create({
        error: {
          key: params.hosts[0],
          code: response.status,
          message: response.message
        },
        isCollapsed: true,
        isBodyVisible: Em.computed.ifThenElse('isCollapsed', 'display: none;', 'display: block;')
      });
      undeletableHosts.push(host);
    } else {
      var data = arg0;
      var _params = arg2;
      if (data) {
        data.deleteResult.forEach(function (host) {
          if (!host.deleted) {
            var _host = Ember.Object.create({
              error: host.error,
              isCollapsed: true,
              isBodyVisible: Em.computed.ifThenElse('isCollapsed', 'display: none;', 'display: block;')
            });
            undeletableHosts.push(_host);
          }
        });
      }
      deletedHosts = _params.hostNames;
    }

    return App.ModalPopup.show({
      header: Em.I18n.t('hosts.bulkOperation.delete.component.result.header'),

      secondary: null,

      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/delete_hosts_result_popup'),
        message: Em.I18n.t('hosts.bulkOperation.delete.component.dryRun.message').format(componentName),
        componentName: componentName,
        undeletableHosts: undeletableHosts,
        deletedHosts: deletedHosts,
        deleteComponents: true,
        onToggleHost: function onToggleHost(host) {
          host.contexts[0].toggleProperty('isCollapsed');
        }
      }),

      onPrimary: function onPrimary() {
        location.reload();
        this._super();
      },

      onClose: function onClose() {
        location.reload();
        this._super();
      }
    });
  },

  /**
   * Bulk operation for selected hostComponents
   * @param {Object} operationData - data about bulk operation (action, hostComponents etc)
   * @param {Array} hosts - list of affected hosts
   */
  bulkOperationForHostComponents: function bulkOperationForHostComponents(operationData, hosts) {
    var self = this;

    return batchUtils.getComponentsFromServer({
      components: [operationData.componentName],
      hosts: hosts.mapProperty('hostName'),
      passiveState: 'OFF'
    }, function (data) {
      return self._getComponentsFromServerForHostComponentsCallback(operationData, data);
    });
  },

  /**
   *
   * @param {object} operationData
   * @param {object} data
   * @returns {$.ajax|App.ModalPopup}
   * @private
   */
  _getComponentsFromServerForHostComponentsCallback: function _getComponentsFromServerForHostComponentsCallback(operationData, data) {
    if (data.items) {
      var hostsWithComponentInProperState = data.items.mapProperty('Hosts.host_name');
      return App.ajax.send({
        name: 'common.host_components.update',
        sender: this,
        data: {
          HostRoles: {
            state: operationData.action
          },
          query: 'HostRoles/component_name=' + operationData.componentName + '&HostRoles/host_name.in(' + hostsWithComponentInProperState.join(',') + ')&HostRoles/maintenance_state=OFF',
          context: operationData.message + ' ' + operationData.componentNameFormatted,
          level: 'SERVICE',
          noOpsMessage: operationData.componentNameFormatted
        },
        success: 'bulkOperationForHostComponentsSuccessCallback'
      });
    }
    return App.ModalPopup.show({
      header: Em.I18n.t('rolling.nothingToDo.header'),
      body: Em.I18n.t('rolling.nothingToDo.body').format(operationData.componentNameFormatted),
      secondary: false
    });
  },

  /**
   * Bulk decommission/recommission for selected hostComponents
   * @param {Object} operationData
   * @param {Array} hosts
   */
  bulkOperationForHostComponentsDecommission: function bulkOperationForHostComponentsDecommission(operationData, hosts) {
    var self = this;

    return batchUtils.getComponentsFromServer({
      components: [operationData.realComponentName],
      hosts: hosts.mapProperty('hostName'),
      passiveState: 'OFF',
      displayParams: ['host_components/HostRoles/state']
    }, function (data) {
      return self._getComponentsFromServerForHostComponentsDecommissionCallBack(operationData, data);
    });
  },

  /**
   * run Bulk decommission/recommission for selected hostComponents
   * after host and components are loaded
   * @param operationData
   * @param data
   * @method _getComponentsFromServerForHostComponentsDecommissionCallBack
   */
  _getComponentsFromServerForHostComponentsDecommissionCallBack: function _getComponentsFromServerForHostComponentsDecommissionCallBack(operationData, data) {
    var service = App.Service.find(operationData.serviceName);
    var components = [];

    data.items.forEach(function (host) {
      host.host_components.forEach(function (hostComponent) {
        components.push(O.create({
          componentName: hostComponent.HostRoles.component_name,
          hostName: host.Hosts.host_name,
          workStatus: hostComponent.HostRoles.state
        }));
      });
    });

    if (components.length) {
      var hostsWithComponentInProperState = components.mapProperty('hostName');
      var turn_off = operationData.action.indexOf('OFF') !== -1;
      var svcName = operationData.serviceName;
      var masterName = operationData.componentName;
      var slaveName = operationData.realComponentName;
      var hostNames = hostsWithComponentInProperState.join(',');
      if (turn_off) {
        // For recommession
        if (svcName === "YARN" || svcName === "HBASE" || svcName === "HDFS") {
          App.router.get('mainHostDetailsController').doRecommissionAndStart(hostNames, svcName, masterName, slaveName);
        }
      } else {
        hostsWithComponentInProperState = components.filterProperty('workStatus', 'STARTED').mapProperty('hostName');
        //For decommession
        if (svcName == "HBASE") {
          // HBASE service, decommission RegionServer in batch requests
          this.warnBeforeDecommission(hostNames);
        } else {
          var parameters = {
            "slave_type": slaveName
          };
          var contextString = turn_off ? 'hosts.host.' + slaveName.toLowerCase() + '.recommission' : 'hosts.host.' + slaveName.toLowerCase() + '.decommission';
          if (turn_off) {
            parameters['included_hosts'] = hostsWithComponentInProperState.join(',');
          } else {
            parameters['excluded_hosts'] = hostsWithComponentInProperState.join(',');
          }
          App.ajax.send({
            name: 'bulk_request.decommission',
            sender: this,
            data: {
              context: Em.I18n.t(contextString),
              serviceName: service.get('serviceName'),
              componentName: operationData.componentName,
              parameters: parameters,
              noOpsMessage: operationData.componentNameFormatted
            },
            success: 'bulkOperationForHostComponentsSuccessCallback'
          });
        }
      }
    } else {
      App.ModalPopup.show({
        header: Em.I18n.t('rolling.nothingToDo.header'),
        body: Em.I18n.t('rolling.nothingToDo.body').format(operationData.componentNameFormatted),
        secondary: false
      });
    }
  },

  /**
   * get info about regionserver passive_state
   * @method warnBeforeDecommission
   * @param {String} hostNames
   * @return {$.ajax}
   */
  warnBeforeDecommission: function warnBeforeDecommission(hostNames) {
    return App.ajax.send({
      'name': 'host_components.hbase_regionserver.active',
      'sender': this,
      'data': {
        hostNames: hostNames
      },
      success: 'warnBeforeDecommissionSuccess'
    });
  },

  /**
   * check is hbase regionserver in mm. If so - run decommission
   * otherwise shows warning
   * @method warnBeforeDecommission
   * @param {Object} data
   * @param {Object} opt
   * @param {Object} params
   */
  warnBeforeDecommissionSuccess: function warnBeforeDecommissionSuccess(data, opt, params) {
    if (Em.get(data, 'items.length')) {
      return App.router.get('mainHostDetailsController').showHbaseActiveWarning();
    }
    return App.router.get('mainHostDetailsController').checkRegionServerState(params.hostNames);
  },

  /**
   * Bulk restart for selected hostComponents
   * @param {Object} operationData
   * @param {Array} hosts
   */
  bulkOperationForHostComponentsRestart: function bulkOperationForHostComponentsRestart(operationData, hosts) {
    var self = this;
    return batchUtils.getComponentsFromServer({
      components: [operationData.componentName],
      hosts: hosts.mapProperty('hostName'),
      passiveState: 'OFF',
      displayParams: ['Hosts/maintenance_state', 'host_components/HostRoles/stale_configs', 'host_components/HostRoles/maintenance_state']
    }, function (data) {
      return self._getComponentsFromServerForHostComponentsRestartCallback(operationData, data);
    });
  },

  _getComponentsFromServerForHostComponentsRestartCallback: function _getComponentsFromServerForHostComponentsRestartCallback(operationData, data) {
    var wrappedHostComponents = [];
    var service = App.Service.find(operationData.serviceName);

    data.items.forEach(function (host) {
      host.host_components.forEach(function (hostComponent) {
        wrappedHostComponents.push(O.create({
          componentName: hostComponent.HostRoles.component_name,
          serviceName: operationData.serviceName,
          hostName: host.Hosts.host_name,
          hostPassiveState: host.Hosts.maintenance_state,
          staleConfigs: hostComponent.HostRoles.stale_configs,
          passiveState: hostComponent.HostRoles.maintenance_state
        }));
      });
    });

    if (wrappedHostComponents.length) {
      return batchUtils.showRollingRestartPopup(wrappedHostComponents.objectAt(0).get('componentName'), service.get('displayName'), service.get('passiveState') === "ON", false, wrappedHostComponents);
    }
    return App.ModalPopup.show({
      header: Em.I18n.t('rolling.nothingToDo.header'),
      body: Em.I18n.t('rolling.nothingToDo.body').format(operationData.componentNameFormatted),
      secondary: false
    });
  },

  updateHostComponentsPassiveState: function updateHostComponentsPassiveState(data, opt, params) {
    return batchUtils.infoPassiveState(params.passive_state);
  },

  /**
   * Show BO popup after bulk request
   * @method bulkOperationForHostComponentsSuccessCallback
   */
  bulkOperationForHostComponentsSuccessCallback: function bulkOperationForHostComponentsSuccessCallback(data, opt, params, req) {
    if (!data && req.status == 200) {
      return App.ModalPopup.show({
        header: Em.I18n.t('rolling.nothingToDo.header'),
        body: Em.I18n.t('rolling.nothingToDo.body').format(params.noOpsMessage || Em.I18n.t('hosts.host.maintainance.allComponents.context')),
        secondary: false
      });
    }
    return App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
      if (initValue) {
        App.router.get('backgroundOperationsController').showPopup();
      }
    });
  },

  /**
   * Returns all hostNames if amount is less than {minShown} or
   * first elements of array (number of elements - {minShown}) converted to string
   * @param {Array} hostNames - array of all listed hostNames
   * @param {String} divider - string to separate hostNames
   * @param {Number} minShown - min amount of hostName to be shown
   * @returns {String} hostNames
   * @method _showHostNames
   * @private
   */
  _showHostNames: function _showHostNames(hostNames, divider, minShown) {
    if (hostNames.length > minShown) {
      return hostNames.slice(0, minShown).join(divider) + divider + Em.I18n.t("installer.step8.other").format(hostNames.length - minShown);
    }
    return hostNames.join(divider);
  },

  /**
   * Confirmation Popup for bulk Operations
   */
  bulkOperationConfirm: function bulkOperationConfirm(operationData, selection) {
    var hostsNames = [],
        queryParams = [];
    // @todo remove using external controller
    switch (selection) {
      case 's':
        hostsNames = App.db.getSelectedHosts('mainHostController');
        if (hostsNames.length > 0) {
          queryParams.push({
            key: 'Hosts/host_name',
            value: hostsNames,
            type: 'MULTIPLE'
          });
        }
        break;
      case 'f':
        queryParams = App.router.get('mainHostController').getQueryParameters(true).filter(function (obj) {
          return !(obj.key == 'page_size' || obj.key == 'from');
        });
        break;
    }

    if (operationData.action === 'SET_RACK_INFO') {
      this.getHostsForBulkOperations(queryParams, operationData, null);
      return;
    }

    this.getHostsForBulkOperations(queryParams, operationData);
  },

  getHostsForBulkOperations: function getHostsForBulkOperations(queryParams, operationData) {
    return App.ajax.send({
      name: 'hosts.bulk.operations',
      sender: this,
      data: {
        parameters: App.router.get('updateController').computeParameters(queryParams),
        operationData: operationData
      },
      success: 'getHostsForBulkOperationSuccessCallback',
      showLoadingPopup: true
    });
  },

  _convertHostsObjects: function _convertHostsObjects(hosts) {
    return hosts.map(function (host) {
      return {
        index: host.index,
        id: host.id,
        clusterId: host.cluster_id,
        passiveState: host.passive_state,
        state: host.state,
        hostName: host.host_name,
        hostComponents: host.host_components
      };
    });
  },

  getHostsForBulkOperationSuccessCallback: function getHostsForBulkOperationSuccessCallback(json, opt, param) {
    var self = this;
    var operationData = param.operationData;
    var hosts = this._convertHostsObjects(App.hostsMapper.map(json, true));
    var repoVersion = null;
    // no hosts - no actions
    if (!hosts.length) {
      return;
    }

    if (['SET_RACK_INFO', 'ADD', 'DELETE'].contains(operationData.action)) {
      return self.bulkOperation(operationData, hosts);
    }

    var hostNames = hosts.mapProperty('hostName');
    var hostNamesSkipped = [];
    if ('DECOMMISSION' === operationData.action) {
      hostNamesSkipped = this._getSkippedForDecommissionHosts(json, hosts, operationData);
    }
    if ('PASSIVE_STATE' === operationData.action) {
      repoVersion = App.StackVersion.find().findProperty('isCurrent');
      if (!repoVersion && App.StackVersion.find().toArray().length === 1) {
        repoVersion = App.StackVersion.find().findProperty('isOutOfSync');
      }
      if (!repoVersion) {
        console.error("CLUSTER STACK VERSIONS ERROR: multiple clusters in OUT_OF_SYNC state OR none in CURRENT or OUT_OF_SYNC state");
        return;
      }
      hostNamesSkipped = this._getSkippedForPassiveStateHosts(hosts, repoVersion);
    }

    var message = "";
    if (operationData.componentNameFormatted) {
      message = Em.I18n.t('hosts.bulkOperation.confirmation.hostComponents').format(operationData.message, operationData.componentNameFormatted, hostNames.length);
    } else {
      message = Em.I18n.t('hosts.bulkOperation.confirmation.hosts').format(operationData.message, hostNames.length);
    }

    return App.ModalPopup.show({
      header: Em.I18n.t('hosts.bulkOperation.confirmation.header'),
      hostNames: hostNames.join("\n"),
      visibleHosts: self._showHostNames(hostNames, "\n", 3),
      hostNamesSkippedVisible: self._showHostNames(hostNamesSkipped, "\n", 3),
      expanded: false,

      hostNamesSkipped: function () {
        return hostNamesSkipped.length ? hostNamesSkipped.join("\n") : false;
      }.property(),

      didInsertElement: function didInsertElement() {
        this._super();
        this.set('expanded', hostNames.length <= 3);
      },
      onPrimary: function onPrimary() {
        self.bulkOperation(operationData, hosts);
        this._super();
      },

      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/bulk_operation_confirm_popup'),
        message: message,
        textareaVisible: false,

        warningInfo: function () {
          switch (operationData.action) {
            case "DECOMMISSION":
              return Em.I18n.t('hosts.bulkOperation.warningInfo.body');
            case "PASSIVE_STATE":
              return operationData.state === 'OFF' ? Em.I18n.t('hosts.passiveMode.popup.version.mismatch.multiple').format(repoVersion.get('repositoryVersion.repositoryVersion')) : "";
            default:
              return "";
          }
        }.property(),

        textTrigger: function textTrigger() {
          this.toggleProperty('textareaVisible');
        },

        showAll: function showAll() {
          this.set('parentView.visibleHosts', this.get('parentView.hostNames'));
          this.set('parentView.hostNamesSkippedVisible', this.get('parentView.hostNamesSkipped'));
          this.set('parentView.expanded', true);
        },

        putHostNamesToTextarea: function () {
          var hostNames = this.get('parentView.hostNames');
          if (this.get('textareaVisible')) {
            var wrapper = $(".task-detail-log-maintext");
            $('.task-detail-log-clipboard').html(hostNames).width(wrapper.width()).height(250);
            Em.run.next(function () {
              $('.task-detail-log-clipboard').select();
            });
          }
        }.observes('textareaVisible')

      })
    });
  },

  /**
   * @param {object} json
   * @param {object[]} hosts
   * @param {object} operationData
   * @returns {string[]}
   * @private
   * @method _getSkippedForDecommissionHosts
   */
  _getSkippedForDecommissionHosts: function _getSkippedForDecommissionHosts(json, hosts, operationData) {
    var hostComponentStatusMap = {}; // "DATANODE_c6401.ambari.apache.org" => "STARTED"
    var hostComponentIdMap = {}; // "DATANODE_c6401.ambari.apache.org" => "DATANODE"
    if (json.items) {
      json.items.forEach(function (host) {
        if (host.host_components) {
          host.host_components.forEach(function (component) {
            hostComponentStatusMap[component.id] = component.HostRoles.state;
            hostComponentIdMap[component.id] = component.HostRoles.component_name;
          });
        }
      });
    }
    return hosts.filter(function (host) {
      return host.hostComponents.filter(function (component) {
        return hostComponentIdMap[component] == operationData.realComponentName && hostComponentStatusMap[component] == 'INSTALLED';
      }).length > 0;
    }).mapProperty('hostName');
  },

  /**
   * Exclude <code>outOfSyncHosts</code> hosts for PASSIVE request
   *
   * @param {object[]} hosts
   * @returns {string[]}
   * @private
   * @method _getSkippedForPassiveStateHosts
   */
  _getSkippedForPassiveStateHosts: function _getSkippedForPassiveStateHosts(hosts, repoVersion) {
    var hostNames = hosts.mapProperty('hostName');
    var hostNamesSkipped = [];
    var outOfSyncHosts = repoVersion.get('outOfSyncHosts') || [];
    for (var i = 0; i < outOfSyncHosts.length; i++) {
      if (hostNames.contains(outOfSyncHosts[i])) {
        hostNamesSkipped.push(outOfSyncHosts[i]);
      }
    }
    return hostNamesSkipped;
  }

});

});

require.register("controllers/main/host/combo_search_box", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainHostComboSearchBoxController = Em.Controller.extend({
  name: 'mainHostComboSearchBoxController',
  currentSuggestion: [],
  page_size: 10,
  getPropertySuggestions: function getPropertySuggestions(facet, searchTerm) {
    facet = facet == 'hostName' ? 'host_name' : facet;
    return App.ajax.send({
      name: 'hosts.with_searchTerm',
      sender: this,
      data: {
        facet: facet,
        searchTerm: searchTerm,
        page_size: this.get('page_size')
      },
      success: 'getPropertySuggestionsSuccess',
      error: 'commonSuggestionErrorCallback'
    });
  },

  getPropertySuggestionsSuccess: function getPropertySuggestionsSuccess(data, opt, params) {
    this.updateSuggestion(data.items.map(function (item) {
      return item.Hosts[params.facet];
    }));
  },

  updateSuggestion: function updateSuggestion(data) {
    var controller = App.router.get('mainHostComboSearchBoxController');
    controller.set('currentSuggestion', data);
  },

  commonSuggestionErrorCallback: function commonSuggestionErrorCallback() {
    // handle suggestion error
  },

  isComponentStateFacet: function isComponentStateFacet(facet) {
    return App.StackServiceComponent.find(facet).get('isLoaded');
  },

  isComplexHealthStatusFacet: function isComplexHealthStatusFacet(facet) {
    return ['health-status-WITH-ALERTS', 'health-status-RESTART', 'health-status-PASSIVE_STATE'].contains(facet);
  },

  generateQueryParam: function generateQueryParam(param) {
    var expressions = param.key;
    var pHash = this.createComboParamHash(param);
    return this.createComboParamURL(pHash, expressions);
  },

  /**
   * @param pHash {k1:v1, k2:[v1,v2], ...}
   * @param expression
   * @returns {string} 'k1=v1&(k2=v1|k2=v2)'
   */
  createComboParamURL: function createComboParamURL(pHash, expressions) {
    var self = this;
    var result = '';
    for (var key in pHash) {
      var v = pHash[key];
      if (Em.isArray(v)) {
        var ex = '(';
        v.forEach(function (item) {
          var expression = self.getComboParamURL(item, expressions);
          var toAdd = expression.replace('{0}', key);
          toAdd = toAdd.replace('{1}', item);
          ex += toAdd + '|';
        });
        ex = ex.substring(0, ex.length - 1);
        result += ex + ')';
      } else {
        var expression = self.getComboParamURL(v, expressions);
        var ex = expression.replace('{0}', key);
        ex = ex.replace('{1}', v);
        result += ex;
      }
      result += '|';
    }

    return result.substring(0, result.length - 1);
  },

  /**
   * @param value //value of component state
   * @returns expression //url of query of state
   */
  getComboParamURL: function getComboParamURL(value, expressions) {
    var expression = expressions[1];
    switch (value) {
      case 'ALL':
        expression = expressions[0];
        break;
      case 'STARTED':
      case 'STARTING':
      case 'INSTALLED':
      case 'STOPPING':
      case 'INSTALL_FAILED':
      case 'INSTALLING':
      case 'UPGRADE_FAILED':
      case 'UNKNOWN':
      case 'DISABLED':
      case 'INIT':
        break;
      case 'INSERVICE':
      case 'DECOMMISSIONING':
      case 'DECOMMISSIONED':
      case 'RS_DECOMMISSIONED':
        expression = expressions[2];
        break;
      case 'ON':
      case 'OFF':
        expression = expressions[3];
        break;
    }
    return expression;
  },

  /**
   * @param param ['k1:v1','k2:v1', 'k2:v2'] or 'k1:v1'
   * @returns {k1:v1, k2:[v1,v2], ...}
   */
  createComboParamHash: function createComboParamHash(param) {
    var pHash = {};
    if (Em.isArray(param.value)) {
      param.value.forEach(function (item) {
        var values = item.split(':');
        var k = values[0];
        var v = values[1];
        if (!pHash[k]) {
          pHash[k] = v;
        } else {
          if (Em.isArray(pHash[k])) {
            if (pHash[k].indexOf(v) == -1) {
              pHash[k].push(v);
            }
          } else {
            pHash[k] = [pHash[k], v];
          }
        }
      });
    } else {
      var values = param.value.split(':');
      pHash[values[0]] = values[1];
    }
    return pHash;
  }
});

});

require.register("controllers/main/host/configs_service", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with this
 * work for additional information regarding copyright ownership. The ASF
 * licenses this file to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

var App = require('app');

App.MainHostServiceConfigsController = App.MainServiceInfoConfigsController.extend(App.ConfigOverridable, {
  name: 'mainHostServiceConfigsController',
  host: null,
  isHostsConfigsPage: true,
  typeTagToHostMap: null,
  configKeyToConfigMap: null,

  canEdit: false,

  /**
   * On load function
   */
  loadStep: function loadStep() {
    var content = this.get('content');
    this.set('host', content.host);
    this._super();
  },

  /**
   * Removes categories which are not valid for this host. Ex: Remove JOBTRACKER
   * category on host which does not have it installed.
   */
  renderServiceConfigs: function renderServiceConfigs(serviceConfigs) {
    var newServiceConfigs = jQuery.extend({}, serviceConfigs);
    newServiceConfigs.configCategories = this.filterServiceConfigs(serviceConfigs.configCategories);
    this._super(newServiceConfigs);
  },
  /**
   * filter config categories by host-component of host
   * @param configCategories
   * @return {Array}
   */
  filterServiceConfigs: function filterServiceConfigs(configCategories) {
    var hostComponents = this.get('host.hostComponents');
    var hostHostComponentNames = hostComponents ? hostComponents.mapProperty('componentName') : [];

    return configCategories.filter(function (category) {
      var hcNames = category.get('hostComponentNames');
      if (hcNames && hcNames.length > 0) {
        for (var i = 0, l = hcNames.length; i < l; i++) {
          if (hostHostComponentNames.contains(hcNames[i])) {
            return true;
          }
        }
        return false;
      }
      return true;
    });
  },

  /**
   * invoke dialog for switching group of host
   */
  switchHostGroup: function switchHostGroup() {
    var self = this;
    this.launchSwitchConfigGroupOfHostDialog(this.get('selectedConfigGroup'), this.get('configGroups'), this.get('host.hostName'), function (newGroup) {
      self.set('selectedConfigGroup', newGroup);
    });
  }
});

});

require.register("controllers/main/host/details", function(exports, require, module) {
'use strict';

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var batchUtils = require('utils/batch_scheduled_requests');
var hostsManagement = require('utils/hosts');
var stringUtils = require('utils/string_utils');
require('utils/configs/add_component_config_initializer');

App.MainHostDetailsController = Em.Controller.extend(App.SupportClientConfigsDownload, App.InstallComponent, App.InstallNewVersion, App.CheckHostMixin, App.TrackRequestMixin, {

  name: 'mainHostDetailsController',

  /**
   * Viewed host
   * @type {App.Host|null}
   */
  content: null,

  /**
   * Is check host procedure finished
   * @type {bool}
   */
  checkHostFinished: null,

  /**
   * Does user come from hosts page
   * @type {bool}
   */
  isFromHosts: false,

  /**
   * Determines whether we are adding Hive Server2 component
   * @type {bool}
   */
  addHiveServer: false,

  /**
   * Determines whether we are adding ZooKeeper Server component
   * @type {bool}
   */
  addZooKeeperServer: false,

  /**
   * path to page visited before
   * @type {string}
   */
  referer: '',

  /**
   *  Host on which Hive Metastore will be added
   * @type {string}
   */
  hiveMetastoreHost: '',

  /**
   * Deferred object will be resolved when Oozie configs are downloaded
   * @type {object}
   */
  isOozieConfigLoaded: $.Deferred(),

  /**
   * @type {bool}
   */
  isOozieServerAddable: true,

  isConfigsLoadingInProgress: false,

  addDeleteComponentsMap: {
    'ZOOKEEPER_SERVER': {
      addPropertyName: 'addZooKeeperServer',
      deletePropertyName: 'fromDeleteZkServer',
      configTagsCallbackName: 'loadZookeeperConfigs',
      configsCallbackName: 'saveZkConfigs'
    },
    'HIVE_METASTORE': {
      deletePropertyName: 'deleteHiveMetaStore',
      hostPropertyName: 'hiveMetastoreHost',
      configTagsCallbackName: 'loadHiveConfigs',
      configsCallbackName: 'onLoadHiveConfigs'
    },
    'WEBHCAT_SERVER': {
      deletePropertyName: 'deleteWebHCatServer',
      hostPropertyName: 'webhcatServerHost',
      configTagsCallbackName: 'loadWebHCatConfigs',
      configsCallbackName: 'onLoadHiveConfigs'
    },
    'HIVE_SERVER': {
      addPropertyName: 'addHiveServer',
      deletePropertyName: 'deleteHiveServer',
      configTagsCallbackName: 'loadHiveConfigs',
      configsCallbackName: 'onLoadHiveConfigs'
    },
    'NIMBUS': {
      deletePropertyName: 'deleteNimbusHost',
      hostPropertyName: 'nimbusHost',
      configTagsCallbackName: 'loadStormConfigs',
      configsCallbackName: 'onLoadStormConfigs'
    },
    'ATLAS_SERVER': {
      deletePropertyName: 'deleteAtlasServer',
      hostPropertyName: 'atlasServer',
      configTagsCallbackName: 'loadAtlasConfigs',
      configsCallbackName: 'onLoadAtlasConfigs'
    },
    'RANGER_KMS_SERVER': {
      deletePropertyName: 'deleteRangerKMSServer',
      hostPropertyName: 'rangerKMSServerHost',
      configTagsCallbackName: 'loadRangerConfigs',
      configsCallbackName: 'onLoadRangerConfigs'
    }
  },

  zooKeeperRelatedServices: [{
    serviceName: 'HIVE',
    typesToLoad: ['hive-site', 'webhcat-site'],
    typesToSave: ['hive-site', 'webhcat-site']
  }, {
    serviceName: 'YARN',
    typesToLoad: ['yarn-site', 'zoo.cfg'],
    typesToSave: ['yarn-site']
  }, {
    serviceName: 'HBASE',
    typesToLoad: ['hbase-site'],
    typesToSave: ['hbase-site']
  }, {
    serviceName: 'ACCUMULO',
    typesToLoad: ['accumulo-site'],
    typesToSave: ['accumulo-site']
  }, {
    serviceName: 'KAFKA',
    typesToLoad: ['kafka-broker'],
    typesToSave: ['kafka-broker']
  }, {
    serviceName: 'ATLAS',
    typesToLoad: ['application-properties', 'infra-solr-env'],
    typesToSave: ['application-properties']
  }, {
    serviceName: 'STORM',
    typesToLoad: ['storm-site'],
    typesToSave: ['storm-site']
  }],

  /**
   * Determines whether adding/deleting host component requires configs changes
   * @type {Boolean}
   */
  isReconfigureRequired: false,

  /**
   * Contains component-related config properties loaded from server
   * @type {Object|null}
   */
  configs: null,

  /**
   * Array of all properties affected by adding/deleting host component
   * @type {Array}
   */
  allPropertiesToChange: [],

  /**
   * Array of editable properties affected by adding/deleting host component
   * @type {Array}
   */
  recommendedPropertiesToChange: [],

  /**
   * Array of non-editable properties affected by adding/deleting host component
   * @type {Array}
   */
  requiredPropertiesToChange: [],

  /**
   * Properties affected by adding/deleting host component, grouped by service, formatted for PUT call
   * @type {Array}
   */
  groupedPropertiesToChange: [],

  hasPropertiesToChange: Em.computed.or('recommendedPropertiesToChange.length', 'requiredPropertiesToChange.length'),

  addDeleteComponentPopupBody: Em.View.extend({
    templateName: require('templates/main/host/details/addDeleteComponentPopup'),
    commonMessage: '',
    manualKerberosWarning: App.get('router.mainAdminKerberosController.isManualKerberos') ? Em.I18n.t('hosts.host.manualKerberosWarning') : '',
    lastComponent: false,
    lastComponentError: '',
    fromServiceSummary: false,
    selectedHost: null,
    anyHostsWithoutComponent: true
  }),

  saveLoadedConfigs: function saveLoadedConfigs(data) {
    this.set('configs', {
      items: data.items.map(function (item) {
        return {
          type: item.type,
          properties_attributes: item.properties_attributes,
          properties: Em.copy(item.properties)
        };
      })
    });
  },

  clearConfigsChanges: function clearConfigsChanges(shouldKeepLoadedConfigs) {
    var arrayNames = ['allPropertiesToChange', 'recommendedPropertiesToChange', 'requiredPropertiesToChange', 'groupedPropertiesToChange'];
    this.abortRequests();
    arrayNames.forEach(function (arrayName) {
      this.get(arrayName).clear();
    }, this);
    this.set('isReconfigureRequired', false);
    if (!shouldKeepLoadedConfigs) {
      this.set('configs', null);
    }
  },

  applyConfigsCustomization: function applyConfigsCustomization() {
    this.get('recommendedPropertiesToChange').forEach(function (property) {
      var value = property.saveRecommended ? property.recommendedValue : property.initialValue,
          filename = property.propertyFileName;
      if (this.get('groupedPropertiesToChange.length')) {
        var group = this.get('groupedPropertiesToChange').find(function (item) {
          return item.properties.hasOwnProperty(filename);
        });
        if (group) {
          group.properties[filename][property.propertyName] = value;
        }
      }
    }, this);
  },

  /**
   * Open dashboard page
   * @method routeHome
   */
  routeHome: function routeHome() {
    App.router.transitionTo('main.dashboard.index');
  },

  /**
   * List of active (not in passive state) host components
   * @type {Ember.Enumerable}
   */
  serviceActiveComponents: function () {
    return this.get('content.hostComponents').filterProperty('service.isInPassive', false);
  }.property('content.hostComponents'),

  /**
   * List of active host components which aren't clients
   * @type {Ember.Enumerable}
   */
  serviceNonClientActiveComponents: Em.computed.filterBy('serviceActiveComponents', 'isClient', false),

  /**
   * send command to server to start selected host component
   * @param {object} event
   * @method startComponent
   */
  startComponent: function startComponent(event) {
    var self = this;
    var component = event.context;
    return App.showConfirmationPopup(function () {
      var context = Em.I18n.t('requestInfo.startHostComponent') + " " + component.get('displayName');
      self.sendComponentCommand(component, context, App.HostComponentStatus.started);
    }, Em.I18n.t('question.sure.start').format(component.get('displayName')));
  },

  /**
   * send command to server to stop selected host component
   * @param {object} event
   * @method startComponent
   */
  stopComponent: function stopComponent(event) {
    var self = this;
    var component = event.context;
    if (event.context.get('componentName') === 'NAMENODE') {
      this.checkNnLastCheckpointTime(function () {
        return App.showConfirmationPopup(function () {
          var context = Em.I18n.t('requestInfo.stopHostComponent') + " " + component.get('displayName');
          self.sendComponentCommand(component, context, App.HostComponentStatus.stopped);
        }, Em.I18n.t('question.sure.stop').format(component.get('displayName')));
      });
    } else {
      return App.showConfirmationPopup(function () {
        var context = Em.I18n.t('requestInfo.stopHostComponent') + " " + component.get('displayName');
        self.sendComponentCommand(component, context, App.HostComponentStatus.stopped);
      }, Em.I18n.t('question.sure.stop').format(component.get('displayName')));
    }
  },
  /**
   * PUTs a command to server to start/stop a component. If no
   * specific component is provided, all components are started.
   * @param {object} component  When <code>null</code> all startable components are started.
   * @param {String} context  Context under which this command is beign sent.
   * @param {String} state - desired state of component can be 'STARTED' or 'STOPPED'
   * @method sendComponentCommand
   */
  sendComponentCommand: function sendComponentCommand(component, context, state) {
    var data = {
      hostName: this.get('content.hostName'),
      context: context,
      component: component,
      HostRoles: {
        state: state
      }
    };
    if (Array.isArray(component)) {
      data.query = "HostRoles/component_name.in(" + component.mapProperty('componentName').join(',') + ")";
    } else {
      data.componentName = component.get('componentName');
      data.serviceName = component.get('service.serviceName');
    }
    App.ajax.send({
      name: Array.isArray(component) ? 'common.host.host_components.update' : 'common.host.host_component.update',
      sender: this,
      data: data,
      success: 'sendComponentCommandSuccessCallback',
      error: 'ajaxErrorCallback',
      showLoadingPopup: true
    });
  },

  /**
   * Success callback for stop/start host component request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method stopComponentSuccessCallback
   */
  sendComponentCommandSuccessCallback: function sendComponentCommandSuccessCallback(data, opt, params) {
    var running = params.HostRoles.state === App.HostComponentStatus.stopped ? App.HostComponentStatus.stopping : App.HostComponentStatus.starting;
    params.component.set('workStatus', running);
    if (App.get('testMode')) {
      this.mimicWorkStatusChange(params.component, running, params.HostRoles.state);
    }
    this.showBackgroundOperationsPopup();
  },

  /**
   * Return true if hdfs user data is loaded via App.MainServiceInfoConfigsController
   */
  getHdfsUser: function getHdfsUser() {
    var self = this;
    var dfd = $.Deferred();
    var miscController = App.MainAdminServiceAccountsController.create();
    miscController.loadUsers();
    miscController.addObserver('dataIsLoaded', this, function () {
      if (miscController.get('dataIsLoaded') && miscController.get('users')) {
        if (miscController.get('users').someProperty('name', 'hdfs_user')) {
          self.set('hdfsUser', miscController.get('users').findProperty('name', 'hdfs_user').get('value'));
        } else {
          self.set('hdfsUser', '&lt;hdfs-user&gt;');
        }
        dfd.resolve();
        miscController.destroy();
      }
    });
    return dfd.promise();
  },

  /**
   * this function will be called from :1) stop NN 2) restart NN 3) stop all components
   * @param callback - callback function to continue next operation
   * @param hostname - namenode host (by default is current host)
   */
  checkNnLastCheckpointTime: function checkNnLastCheckpointTime(callback, hostName) {
    var self = this;
    this.pullNnCheckPointTime(hostName).complete(function () {
      var isNNCheckpointTooOld = self.get('isNNCheckpointTooOld');
      self.set('isNNCheckpointTooOld', null);
      if (isNNCheckpointTooOld) {
        // too old
        self.getHdfsUser().done(function () {
          var msg = Em.Object.create({
            confirmMsg: Em.I18n.t('services.service.stop.HDFS.warningMsg.checkPointTooOld').format(App.nnCheckpointAgeAlertThreshold) + Em.I18n.t('services.service.stop.HDFS.warningMsg.checkPointTooOld.makeSure') + Em.I18n.t('services.service.stop.HDFS.warningMsg.checkPointTooOld.instructions.singleHost.login').format(isNNCheckpointTooOld) + Em.I18n.t('services.service.stop.HDFS.warningMsg.checkPointTooOld.instructions').format(self.get('hdfsUser')),
            confirmButton: Em.I18n.t('common.next')
          });
          return App.showConfirmationFeedBackPopup(callback, msg);
        });
      } else if (isNNCheckpointTooOld == null) {
        // not available
        return App.showConfirmationPopup(callback, Em.I18n.t('services.service.stop.HDFS.warningMsg.checkPointNA'), null, Em.I18n.t('common.warning'), Em.I18n.t('common.proceedAnyway'), 'danger');
      } else {
        // still young
        callback();
      }
    });
  },

  pullNnCheckPointTime: function pullNnCheckPointTime(hostName) {
    return App.ajax.send({
      name: 'common.host_component.getNnCheckPointTime',
      sender: this,
      data: {
        host: hostName || this.get('content.hostName')
      },
      success: 'parseNnCheckPointTime'
    });
  },

  parseNnCheckPointTime: function parseNnCheckPointTime(data) {
    var lastCheckpointTime = Em.get(data, 'metrics.dfs.FSNamesystem.LastCheckpointTime');
    var hostName = Em.get(data, 'HostRoles.host_name');

    if (Em.get(data, 'metrics.dfs.FSNamesystem.HAState') === 'active') {
      if (!lastCheckpointTime) {
        this.set("isNNCheckpointTooOld", null);
      } else {
        var time_criteria = App.nnCheckpointAgeAlertThreshold; // time in hours to define how many hours ago is too old
        var time_ago = (Math.round(App.dateTime() / 1000) - time_criteria * 3600) * 1000;
        if (lastCheckpointTime <= time_ago) {
          // too old, set the effected hostName
          this.set("isNNCheckpointTooOld", hostName);
        } else {
          // still young
          this.set("isNNCheckpointTooOld", false);
        }
      }
    } else if (Em.get(data, 'metrics.dfs.FSNamesystem.HAState') === 'standby') {
      this.set("isNNCheckpointTooOld", false);
    }
  },

  /**
   * mimic status transition in test mode
   * @param entity
   * @param transitionalState
   * @param finalState
   */
  mimicWorkStatusChange: function mimicWorkStatusChange(entity, transitionalState, finalState) {
    if (Em.isArray(entity)) {
      entity.forEach(function (item) {
        item.set('workStatus', transitionalState);
        setTimeout(function () {
          item.set('workStatus', finalState);
        }, App.testModeDelayForActions);
      });
    } else {
      entity.set('workStatus', transitionalState);
      setTimeout(function () {
        entity.set('workStatus', finalState);
      }, App.testModeDelayForActions);
    }
  },

  /**
   * load data (if we need to show this background operations popup) from persist
   * @param callback
   */
  showBackgroundOperationsPopup: function showBackgroundOperationsPopup(callback) {
    App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
      if (initValue) {
        App.router.get('backgroundOperationsController').showPopup();
      }
      if (typeof callback === 'function') {
        callback();
      }
    });
  },

  /**
   * Send command to server to delete selected host component
   * @param {object} event
   * @method deleteComponent
   */
  deleteComponent: function deleteComponent(event) {
    var _this2 = this;

    if ($(event.target).closest('li').hasClass('disabled')) {
      return;
    }
    var component = event.context;
    var componentName = component.get('componentName');
    var componentsMapItem = this.get('addDeleteComponentsMap')[componentName];

    if (componentsMapItem) {
      this.deleteAndReconfigureComponent(componentsMapItem, component);
    } else if (componentName === 'JOURNALNODE') {
      return App.showConfirmationPopup(function () {
        App.router.transitionTo('main.services.manageJournalNode');
      }, Em.I18n.t('hosts.host.deleteComponent.popup.deleteJournalNodeMsg'));
    } else {
      return this.showReconfigurationPopupPreDelete(component, function () {
        return _this2._doDeleteHostComponent(componentName);
      });
    }
  },

  /**
   *
   * @param {object} componentsMapItem
   * @param {App.HostComponent} component
   * @returns {App.ModalPopup}
   */
  deleteAndReconfigureComponent: function deleteAndReconfigureComponent(componentsMapItem, component) {
    var _this3 = this;

    if (componentsMapItem.deletePropertyName) {
      this.set(componentsMapItem.deletePropertyName, true);
    }
    this.loadComponentRelatedConfigs(componentsMapItem.configTagsCallbackName, componentsMapItem.configsCallbackName);
    return this.showReconfigurationPopupPreDelete(component, function () {
      _this3._doDeleteHostComponent(component.get('componentName')).done(function () {
        _this3.applyConfigsCustomization();
        _this3.putConfigsToServer(_this3.get('groupedPropertiesToChange'), component.get('componentName'));
        _this3.clearConfigsChanges();
      });
    });
  },

  showReconfigurationPopupPreDelete: function showReconfigurationPopupPreDelete(component) {
    var primary = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : Em.K;
    var commonMessage = arguments[2];

    var isLastComponent = this.getTotalComponent(component) === 1,
        componentDisplayName = component.get('displayName');

    return App.ModalPopup.show({
      header: Em.I18n.t('popup.confirmation.commonHeader'),
      controller: this,
      hasPropertiesToChange: false,
      classNameBindings: ['controller.hasPropertiesToChange:common-modal-wrapper'],
      modalDialogClasses: function () {
        return this.get('controller.hasPropertiesToChange') ? ['modal-xlg'] : [];
      }.property('controller.hasPropertiesToChange'),
      primary: Em.I18n.t('hosts.host.deleteComponent.popup.confirm'),
      bodyClass: this.get('addDeleteComponentPopupBody').extend({
        commonMessage: commonMessage || Em.I18n.t('hosts.host.deleteComponent.popup.msg1').format(componentDisplayName),
        recommendedPropertiesToChange: this.get('recommendedPropertiesToChange'),
        requiredPropertiesToChange: this.get('requiredPropertiesToChange'),
        lastComponentError: Em.I18n.t('hosts.host.deleteComponent.popup.warning').format(componentDisplayName),
        lastComponent: isLastComponent
      }),
      isChecked: !isLastComponent,
      disablePrimary: function () {
        return this.get('controller.isReconfigureRequired') && this.get('controller.isConfigsLoadingInProgress') || !this.get('isChecked');
      }.property('controller.isReconfigureRequired', 'controller.isConfigsLoadingInProgress', 'isChecked'),
      onPrimary: function onPrimary() {
        this._super();
        primary();
      },
      onSecondary: function onSecondary() {
        this._super();
        this.get('controller').clearConfigsChanges();
      },
      onClose: function onClose() {
        this._super();
        this.get('controller').clearConfigsChanges();
      }
    });
  },

  /**
   * get total count of host-components
   * @method getTotalComponent
   * @param component
   * @return {Number}
   */
  getTotalComponent: function getTotalComponent(component) {
    var count;
    if (component.get('isSlave')) {
      count = App.SlaveComponent.find(component.get('componentName')).get('totalCount');
    } else if (component.get('isClient')) {
      count = App.ClientComponent.find(component.get('componentName')).get('totalCount');
    } else {
      count = App.HostComponent.find().filterProperty('componentName', component.get('componentName')).get('length');
    }
    return count || 0;
  },

  /**
   * Trigger to reset list of master/slaves components on the view
   * @type {bool}
   */
  redrawComponents: false,

  /**
   * Deletes the given host component, or all host components.
   *
   * @param {string} componentName
   * @return  {$.ajax}
   * @method _doDeleteHostComponent
   */
  _doDeleteHostComponent: function _doDeleteHostComponent(componentName) {
    return App.ajax.send({
      name: Em.isNone(componentName) ? 'common.delete.host' : 'common.delete.host_component',
      sender: this,
      data: {
        componentName: componentName || '',
        hostName: this.get('content.hostName')
      },
      success: '_doDeleteHostComponentSuccessCallback',
      error: '_doDeleteHostComponentErrorCallback',
      showLoadingPopup: true
    });
  },

  /**
   * Error of delete component(s) request
   * @type {object}
   */
  _deletedHostComponentError: null,

  /**
   * Success callback for delete host component request
   * @method _doDeleteHostComponentSuccessCallback
   */
  _doDeleteHostComponentSuccessCallback: function _doDeleteHostComponentSuccessCallback(response, request, data) {
    this.set('_deletedHostComponentError', null);
  },

  /**
   * Error-callback for delete host component request
   * @param {object} xhr
   * @param {string} textStatus
   * @param {object} errorThrown
   * @param {object} data
   * @method _doDeleteHostComponentErrorCallback
   */
  _doDeleteHostComponentErrorCallback: function _doDeleteHostComponentErrorCallback(xhr, textStatus, errorThrown, data) {
    this.set('_deletedHostComponentError', { xhr: xhr, url: data.url, method: 'DELETE' });
  },

  /**
   * Send command to server to upgrade selected host component
   * @param {object} event
   * @method upgradeComponent
   */
  upgradeComponent: function upgradeComponent(event) {
    var self = this;
    var component = event.context;
    return App.showConfirmationPopup(function () {
      App.ajax.send({
        name: 'host.host_component.upgrade',
        sender: self,
        data: {
          component: component,
          hostName: self.get('content.hostName'),
          componentName: component.get('componentName'),
          data: JSON.stringify({
            RequestInfo: {
              "context": Em.I18n.t('requestInfo.upgradeHostComponent') + " " + component.get('displayName')
            },
            Body: {
              HostRoles: {
                stack_id: 'HDP-1.2.2',
                state: 'INSTALLED'
              }
            }
          })
        },
        success: 'upgradeComponentSuccessCallback',
        error: 'ajaxErrorCallback'
      });
    }, Em.I18n.t('question.sure.upgrade').format(component.get('displayName')));
  },

  /**
   * Success callback for upgrade host component request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method upgradeComponentSuccessCallback
   */
  upgradeComponentSuccessCallback: function upgradeComponentSuccessCallback(data, opt, params) {
    if (App.get('testMode')) {
      this.mimicWorkStatusChange(params.component, App.HostComponentStatus.starting, App.HostComponentStatus.started);
    }
    this.showBackgroundOperationsPopup();
  },

  /**
   * Send command to server to restart selected components
   * @param {object} event
   * @method restartComponent
   */
  restartComponent: function restartComponent(event) {
    var component = event.context;
    if (event.context.get('componentName') === 'NAMENODE') {
      this.checkNnLastCheckpointTime(function () {
        return App.showConfirmationPopup(function () {
          batchUtils.restartHostComponents([component], Em.I18n.t('rollingrestart.context.selectedComponentOnSelectedHost').format(component.get('displayName')), "HOST_COMPONENT");
        }, Em.I18n.t('question.sure.restart').format(component.get('displayName')));
      });
    } else {
      return App.showConfirmationPopup(function () {
        batchUtils.restartHostComponents([component], Em.I18n.t('rollingrestart.context.selectedComponentOnSelectedHost').format(component.get('displayName')), "HOST_COMPONENT");
      }, Em.I18n.t('question.sure.restart').format(component.get('displayName')));
    }
  },

  /**
   * add component as <code>addComponent<code> method but perform
   * kdc session state if cluster is secure;
   * @param event
   */
  addComponentWithCheck: function addComponentWithCheck(event) {
    App.get('router.mainAdminKerberosController').getSecurityType(function (event) {
      App.get('router.mainAdminKerberosController').getKDCSessionState(this.addComponent.bind(this, event));
    }.bind(this, event));
  },
  /**
   * Send command to server to install selected host component
   * @param {object} event
   * @method addComponent
   */
  addComponent: function addComponent(event) {
    var _this4 = this;

    var component = event.context,
        hostName = this.get('content.hostName'),
        componentName = component.get('componentName'),
        missedComponents = event.fromServiceSummary ? [] : this.checkComponentDependencies(componentName, {
      scope: 'host',
      installedComponents: this.get('content.hostComponents').mapProperty('componentName')
    }),
        isManualKerberos = App.get('router.mainAdminKerberosController.isManualKerberos'),
        manualKerberosWarning = isManualKerberos ? Em.I18n.t('hosts.host.manualKerberosWarning') : '',
        componentsMapItem = this.get('addDeleteComponentsMap')[componentName];

    if (missedComponents.length) {
      var popupMessage = Em.I18n.t('host.host.addComponent.popup.dependedComponents.body').format(component.get('displayName'), stringUtils.getFormattedStringFromArray(missedComponents.map(function (cName) {
        return App.StackServiceComponent.find(cName).get('displayName');
      })));
      return App.showAlertPopup(Em.I18n.t('host.host.addComponent.popup.dependedComponents.header'), popupMessage);
    }

    if (componentsMapItem) {
      this.addAndReconfigureComponent(componentsMapItem, hostName, component, event.fromServiceSummary);
    } else if (componentName === 'JOURNALNODE') {
      return App.showConfirmationPopup(function () {
        App.router.transitionTo('main.services.manageJournalNode');
      }, Em.I18n.t('hosts.host.addComponent.' + componentName) + manualKerberosWarning);
    } else {
      return this.showAddComponentPopup(component, hostName, function () {
        // hostname should be taken via this.get('content.hostName') because the host could be selected later
        _this4.installHostComponentCall(_this4.get('content.hostName'), component);
      }, event.fromServiceSummary);
    }
  },

  /**
   *
   * @param {object} componentsMapItem
   * @param {string} hostName
   * @param {App.HostComponent} component
   * @param {boolean} fromServiceSummary
   * @returns {*}
   */
  addAndReconfigureComponent: function addAndReconfigureComponent(componentsMapItem, hostName, component, fromServiceSummary) {
    var _this5 = this;

    if (componentsMapItem.hostPropertyName) {
      this.set(componentsMapItem.hostPropertyName, hostName);
    }
    if (componentsMapItem.addPropertyName) {
      this.set(componentsMapItem.addPropertyName, true);
    }
    this.loadComponentRelatedConfigs(componentsMapItem.configTagsCallbackName, componentsMapItem.configsCallbackName);
    return this.showAddComponentPopup(component, hostName, function () {
      _this5.installAndReconfigureComponent(_this5.get('content.hostName'), component, componentsMapItem);
    }, fromServiceSummary);
  },

  /**
   *
   * @param {string} hostName
   * @param {App.HostComponent} component
   * @param {object} componentsMapItem
   */
  installAndReconfigureComponent: function installAndReconfigureComponent(hostName, component, componentsMapItem) {
    var _this6 = this;

    this.applyConfigsCustomization();
    this.installHostComponentCall(hostName, component).done(function () {
      _this6.putConfigsToServer(_this6.get('groupedPropertiesToChange'), component.get('componentName'));
    }).always(function () {
      _this6.set(componentsMapItem.addPropertyName, false);
      _this6.clearConfigsChanges();
    });
  },

  showAddComponentPopup: function showAddComponentPopup(component, hostName, primary, fromServiceSummary) {
    var self = this,
        componentName = component.get('componentName'),
        componentDisplayName = component.get('displayName'),
        componentsMapItem = this.get('addDeleteComponentsMap')[componentName],
        commonMessage = Em.I18n.t('hosts.host.addComponent.msg').format(componentDisplayName);

    return App.ModalPopup.show({
      header: Em.I18n.t('popup.confirmation.commonHeader'),
      controller: self,
      hasPropertiesToChange: false,
      classNameBindings: ['hasPropertiesToChange:common-modal-wrapper'],
      modalDialogClasses: function () {
        return this.get('controller.hasPropertiesToChange') ? ['modal-xlg'] : [];
      }.property('controller.hasPropertiesToChange'),
      primary: Em.I18n.t('hosts.host.addComponent.popup.confirm'),
      bodyClass: self.get('addDeleteComponentPopupBody').extend({
        commonMessage: commonMessage,
        fromServiceSummary: fromServiceSummary,
        addComponentMsg: Em.I18n.t('hosts.host.addComponent.msg').format(componentDisplayName),
        selectHostMsg: Em.I18n.t('services.summary.selectHostForComponent').format(componentDisplayName),
        thereIsNoHostsMsg: Em.I18n.t('services.summary.allHostsAlreadyRunComponent').format(componentDisplayName),
        hostsWithoutComponent: function () {
          if (this.get('fromServiceSummary')) {
            var hostsWithComponent = App.HostComponent.find().filterProperty('componentName', componentName).mapProperty('hostName');
            return App.get('allHostNames').filter(function (hostname) {
              return !hostsWithComponent.contains(hostname);
            });
          } else {
            return [];
          }
        }.property('fromServiceSummary'),
        anyHostsWithoutComponent: Em.computed.or('!fromServiceSummary', 'hostsWithoutComponent.length'),
        selectedHostObserver: function () {
          var selectedHostName = this.get('selectedHost');
          if (!self.get('content')) {
            self.set('content', {});
          }
          self.setProperties({
            'content.hostName': selectedHostName
          });
          if (componentsMapItem) {
            var configs = self.get('configs');
            var params = configs && configs.params || {};
            if (componentsMapItem.hostPropertyName) {
              self.set(componentsMapItem.hostPropertyName, selectedHostName);
            }
            if (componentsMapItem.addPropertyName) {
              self.set(componentsMapItem.addPropertyName, true);
            }
            if (configs) {
              self.clearConfigsChanges(true);
              self.set('isReconfigureRequired', true);
              self.set('isConfigsLoadingInProgress', true);
              self[componentsMapItem.configsCallbackName](configs, null, params);
            } else {
              self.loadComponentRelatedConfigs(componentsMapItem.configTagsCallbackName, componentsMapItem.configsCallbackName);
            }
          }
        }.observes('selectedHost')
      }),
      disablePrimary: Em.computed.and('controller.isReconfigureRequired', 'controller.isConfigsLoadingInProgress'),
      onPrimary: function onPrimary() {
        this._super();
        if (primary) {
          primary();
        }
      },
      onSecondary: function onSecondary() {
        this._super();
        self.clearConfigsChanges();
      },
      onClose: function onClose() {
        this._super();
        self.clearConfigsChanges();
      }
    });
  },

  loadComponentRelatedConfigs: function loadComponentRelatedConfigs(configTagsCallbackName, configsCallbackName) {
    var _this7 = this;

    this.clearConfigsChanges();
    this.set('isReconfigureRequired', true);
    this.set('isConfigsLoadingInProgress', true);
    this.isServiceMetricsLoaded(function () {
      _this7.loadConfigs(configTagsCallbackName, configsCallbackName);
    });
  },

  /**
   * Success callback for install host component request (sent in <code>addNewComponentSuccessCallback</code>)
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method installNewComponentSuccessCallback
   */
  installNewComponentSuccessCallback: function installNewComponentSuccessCallback(data, opt, params) {
    if (!data || !data.Requests || !data.Requests.id) {
      return false;
    }

    if (App.get('testMode')) {
      this.mimicWorkStatusChange(params.component, App.HostComponentStatus.installing, App.HostComponentStatus.stopped);
    }

    this.showBackgroundOperationsPopup();
    return true;
  },

  /**
   * Call <code>setRackInfo</code> function to show Set Rack Id popup
   * @param data
   */
  setRackId: function setRackId(data) {
    var rack = data.context.get('rack');
    var hosts = [data.context];
    var operationData = { message: Em.I18n.t('hosts.host.details.setRackId') };
    hostsManagement.setRackInfo(operationData, hosts, rack);
  },

  /**
   * Success callback for load configs request
   * @param {object} data
   * @method loadOozieConfigs
   */
  loadOozieConfigs: function loadOozieConfigs(data) {
    return App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        urlParams: '(type=oozie-env&tag=' + data.Clusters.desired_configs['oozie-env'].tag + ')'
      },
      success: 'onLoadOozieConfigs',
      error: 'onLoadConfigsErrorCallback'
    });
  },

  /**
   * get Oozie database config and set databaseType
   * @param {object} data
   * @method onLoadOozieConfigs
   */
  onLoadOozieConfigs: function onLoadOozieConfigs(data) {
    var configs = {};
    data.items.forEach(function (item) {
      $.extend(configs, item.properties);
    });
    this.set('isOozieServerAddable', !(Em.isEmpty(configs["oozie_database"]) || configs["oozie_database"] === 'New Derby Database'));
    this.get('isOozieConfigLoaded').resolve();
  },

  /**
   * Success callback for Storm load configs request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method loadStormConfigs
   */
  loadStormConfigs: function loadStormConfigs(data, opt, params) {
    var request = App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        urlParams: '(type=storm-site&tag=' + data.Clusters.desired_configs['storm-site'].tag + ')'
      },
      success: params.callback
    });
    this.trackRequest(request);
  },

  /**
   * Success callback for Atlas load configs request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method loadAtlasConfigs
   */
  loadAtlasConfigs: function loadAtlasConfigs(data, opt, params) {
    var request = App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        urlParams: '(type=application-properties&tag=' + data.Clusters.desired_configs['application-properties'].tag + ')'
      },
      success: params.callback
    });
    this.trackRequest(request);
  },

  /**
   * Update zk configs
   * @param {object} configs
   * @method updateZkConfigs
   */
  updateZkConfigs: function updateZkConfigs(configs) {
    var portValue = configs['zoo.cfg'] && Em.get(configs['zoo.cfg'], 'clientPort'),
        zkPort = typeof portValue === 'undefined' ? '2181' : portValue,
        infraSolrZnode = configs['infra-solr-env'] ? Em.get(configs['infra-solr-env'], 'infra_solr_znode') : '/ambari-solr',
        initializer = App.AddZooKeeperComponentsInitializer,
        hostComponentsTopology = {
      masterComponentHosts: []
    },
        propertiesToChange = this.get('allPropertiesToChange'),
        masterComponents = this.bootstrapHostsMapping('ZOOKEEPER_SERVER');
    if (this.get('fromDeleteHost') || this.get('fromDeleteZkServer')) {
      this.set('fromDeleteHost', false);
      this.set('fromDeleteZkServer', false);
      var removedHost = masterComponents.findProperty('hostName', this.get('content.hostName'));
      if (!Em.isNone(removedHost)) {
        Em.set(removedHost, 'isInstalled', false);
      }
    } else if (this.get('addZooKeeperServer')) {
      this.set('addZooKeeperServer', false);
      masterComponents.push({
        component: 'ZOOKEEPER_SERVER',
        hostName: this.get('content.hostName'),
        isInstalled: true
      });
    }
    var dependencies = {
      zkClientPort: zkPort,
      infraSolrZnode: infraSolrZnode
    };
    hostComponentsTopology.masterComponentHosts = masterComponents;
    Em.keys(configs).forEach(function (fileName) {
      var properties = configs[fileName];
      Em.keys(properties).forEach(function (propertyName) {
        var currentValue = properties[propertyName],
            propertyDef = {
          fileName: fileName,
          name: propertyName,
          value: currentValue
        },
            configProperty = initializer.initialValue(propertyDef, hostComponentsTopology, dependencies);
        initializer.updateSiteObj(configs[fileName], configProperty);
        if (currentValue !== configs[fileName][propertyName]) {
          var service = App.config.get('serviceByConfigTypeMap')[fileName],
              configObject = App.configsCollection.getConfigByName(propertyName, fileName),
              displayName = configObject && configObject.displayName;
          propertiesToChange.pushObject({
            propertyFileName: fileName,
            propertyName: propertyName,
            propertyTitle: configObject && Em.I18n.t('installer.controls.serviceConfigPopover.title').format(displayName, displayName === propertyName ? '' : propertyName),
            propertyDescription: configObject && configObject.description,
            serviceDisplayName: service && service.get('displayName'),
            initialValue: currentValue,
            recommendedValue: propertyDef.value
          });
        }
      });
    });
  },

  /**
   *
   * @param {string} componentName
   * @param {string[]} [hostNames]
   * @returns {}
   */
  bootstrapHostsMapping: function bootstrapHostsMapping(componentName, hostNames) {
    if (Em.isNone(hostNames)) {
      hostNames = App.HostComponent.find().filterProperty('componentName', componentName).mapProperty('hostName');
    }
    return hostNames.map(function (hostName) {
      return {
        component: componentName,
        hostName: hostName,
        isInstalled: true
      };
    });
  },

  /**
   * update and save Storm related configs to server
   * @param {object} data
   * @method onLoadStormConfigs
   */
  onLoadStormConfigs: function onLoadStormConfigs(data) {
    var nimbusHost = this.get('nimbusHost'),
        stormNimbusHosts = this.getStormNimbusHosts(),
        configs = {},
        attributes = {},
        propertiesToChange = this.get('allPropertiesToChange');

    this.saveLoadedConfigs(data);
    data.items.forEach(function (item) {
      configs[item.type] = item.properties;
      attributes[item.type] = item.properties_attributes || {};
    });

    this.updateZkConfigs(configs);

    var propertyName = 'nimbus.seeds',
        propertyFileName = 'storm-site',
        nimbusSeedsInit = configs[propertyFileName][propertyName],
        nimbusSeedsRecommended = JSON.stringify(stormNimbusHosts).replace(/"/g, "'");
    configs[propertyFileName][propertyName] = nimbusSeedsRecommended;
    if (this.get('isReconfigureRequired') && nimbusSeedsInit !== nimbusSeedsRecommended) {
      var service = App.config.get('serviceByConfigTypeMap')[propertyFileName],
          configObject = App.configsCollection.getConfigByName(propertyName, propertyFileName),
          displayName = configObject && configObject.displayName;
      propertiesToChange.pushObject({
        propertyFileName: propertyFileName,
        propertyName: propertyName,
        propertyTitle: configObject && Em.I18n.t('installer.controls.serviceConfigPopover.title').format(displayName, displayName === propertyName ? '' : propertyName),
        propertyDescription: configObject && configObject.description,
        serviceDisplayName: service && service.get('displayName'),
        initialValue: nimbusSeedsInit,
        recommendedValue: nimbusSeedsRecommended
      });
    }
    var groups = [{
      properties: _defineProperty({}, propertyFileName, configs[propertyFileName]),
      properties_attributes: _defineProperty({}, propertyFileName, attributes[propertyFileName])
    }];
    this.setConfigsChanges(groups);
  },

  onLoadAtlasConfigs: function onLoadAtlasConfigs(data) {
    var atlasServer = this.get('atlasServer'),
        atlasServerHosts = this.getAtlasServerHosts(),
        configs = {},
        attributes = {},
        propertiesToChange = this.get('allPropertiesToChange');

    this.saveLoadedConfigs(data);
    data.items.forEach(function (item) {
      configs[item.type] = item.properties;
      attributes[item.type] = item.properties_attributes || {};
    });

    var propertyFileName = 'application-properties',
        propertyName = 'atlas.rest.address',
        atlasAddresses = configs[propertyFileName][propertyName],
        hostMask = atlasAddresses.split(',')[0].replace(/([https|http]*\:\/\/)(.*?)(:[0-9]+)/, '$1{hostname}$3'),
        atlasAddressesRecommended = atlasServerHosts.map(function (hostName) {
      return hostMask.replace('{hostname}', hostName);
    }).join(',');
    configs[propertyFileName][propertyName] = atlasAddressesRecommended;
    if (this.get('isReconfigureRequired') && atlasAddresses !== atlasAddressesRecommended) {
      var service = App.config.get('serviceByConfigTypeMap')[propertyFileName],
          configObject = App.configsCollection.getConfigByName(propertyName, propertyFileName),
          displayName = configObject && configObject.displayName;
      propertiesToChange.pushObject({
        propertyFileName: propertyFileName,
        propertyName: propertyName,
        propertyTitle: configObject && Em.I18n.t('installer.controls.serviceConfigPopover.title').format(displayName, displayName === propertyName ? '' : propertyName),
        propertyDescription: configObject && configObject.description,
        serviceDisplayName: service && service.get('displayName'),
        initialValue: atlasAddresses,
        recommendedValue: atlasAddressesRecommended
      });
    }
    var groups = [{
      properties: _defineProperty({}, propertyFileName, configs[propertyFileName]),
      properties_attributes: _defineProperty({}, propertyFileName, attributes[propertyFileName])
    }];
    this.setConfigsChanges(groups);
  },

  /**
   * Success callback for load configs request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method loadWebHCatConfigs
   */
  loadWebHCatConfigs: function loadWebHCatConfigs(data, opt, params) {
    var urlParams = this.getUrlParamsForConfigsRequest(data, ['hive-site', 'webhcat-site', 'hive-env', 'core-site']),
        request = App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        webHCat: true,
        urlParams: urlParams
      },
      success: params.callback
    });
    this.trackRequest(request);
    return request;
  },

  /**
   * Success callback for load configs request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method loadHiveConfigs
   */
  loadHiveConfigs: function loadHiveConfigs(data, opt, params) {
    var urlParams = this.getUrlParamsForConfigsRequest(data, ['hive-site', 'webhcat-site', 'hive-env', 'core-site']),
        request = App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        urlParams: urlParams
      },
      success: params.callback
    });
    this.trackRequest(request);
    return request;
  },

  /**
   * update and save Hive related configs to server
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method onLoadHiveConfigs
   */
  onLoadHiveConfigs: function onLoadHiveConfigs(data, opt, params) {
    var _this8 = this;

    var port = "";
    var configs = {},
        attributes = {},
        userSetup = {},
        localDB = {
      masterComponentHosts: this.getHiveHosts()
    },
        dependencies = {
      hiveMetastorePort: ""
    },
        initializer = params.webHCat ? App.AddWebHCatComponentsInitializer : App.AddHiveComponentsInitializer;
    this.saveLoadedConfigs(data);
    this.set('configs.params', {
      webHCat: params.webHCat
    });
    data.items.forEach(function (item) {
      configs[item.type] = item.properties;
      attributes[item.type] = item.properties_attributes || {};
    });
    var propertiesToChange = this.get('allPropertiesToChange');

    port = configs['hive-site']['hive.metastore.uris'].match(/:[0-9]{2,4}/);
    port = port ? port[0].slice(1) : "9083";

    dependencies.hiveMetastorePort = port;

    if (params.webHCat) {
      userSetup.webhcatUser = configs['hive-env']['webhcat_user'];
    } else {
      userSetup.hiveUser = configs['hive-env']['hive_user'];
    }

    initializer.setup(userSetup);

    ['hive-site', 'webhcat-site', 'hive-env', 'core-site'].forEach(function (fileName) {
      if (configs[fileName]) {
        Em.keys(configs[fileName]).forEach(function (propertyName) {
          var currentValue = configs[fileName][propertyName],
              propertyDef = {
            fileName: fileName,
            name: propertyName,
            value: currentValue
          },
              configProperty = initializer.initialValue(propertyDef, localDB, dependencies);
          initializer.updateSiteObj(configs[fileName], configProperty);
          if (_this8.get('isReconfigureRequired') && currentValue !== configs[fileName][propertyName]) {
            var service = App.config.get('serviceByConfigTypeMap')[fileName],
                configObject = App.configsCollection.getConfigByName(propertyName, fileName),
                displayName = configObject && configObject.displayName;
            propertiesToChange.pushObject({
              propertyFileName: fileName,
              propertyName: propertyName,
              propertyTitle: configObject && Em.I18n.t('installer.controls.serviceConfigPopover.title').format(displayName, displayName === propertyName ? '' : propertyName),
              propertyDescription: configObject && configObject.description,
              serviceDisplayName: service && service.get('displayName'),
              initialValue: currentValue,
              recommendedValue: propertyDef.value
            });
          }
        });
      }
    });

    initializer.cleanup();

    var groups = [{
      properties: {
        'hive-site': configs['hive-site'],
        'webhcat-site': configs['webhcat-site'],
        'hive-env': configs['hive-env']
      },
      properties_attributes: {
        'hive-site': attributes['hive-site'],
        'webhcat-site': attributes['webhcat-site'],
        'hive-env': attributes['hive-env']
      }
    }, {
      properties: {
        'core-site': configs['core-site']
      },
      properties_attributes: {
        'core-site': attributes['core-site']
      }
    }];
    this.setConfigsChanges(groups);
  },

  /**
   * save configs' sites in batch
   * @param groups
   * @param componentName
   */
  putConfigsToServer: function putConfigsToServer(groups, componentName) {
    var dfd = $.Deferred();
    var requests = [];
    if (groups.length) {
      groups.forEach(function (group) {
        var desiredConfigs = [];
        var properties = group.properties;

        for (var site in properties) {
          if (!properties.hasOwnProperty(site) || Em.isNone(properties[site])) continue;
          desiredConfigs.push({
            "type": site,
            "properties": properties[site],
            "properties_attributes": group.properties_attributes[site],
            "service_config_version_note": Em.I18n.t('hosts.host.configs.save.note').format(App.format.role(componentName, false))
          });
        }
        if (desiredConfigs.length > 0) {
          requests.push(App.ajax.send({
            name: 'common.service.configurations',
            sender: this,
            data: {
              desired_config: desiredConfigs,
              componentName: componentName
            }
          }));
        }
      }, this);
      $.when(requests).done(dfd.resolve).fail(dfd.reject).then(function () {
        // If user adding component which require config changes from Configs page then configs should be reloaded
        if (App.router.get('location.location.hash').contains('configs')) {
          App.router.get('mainServiceInfoConfigsController').loadStep();
        }
      });
    } else {
      dfd.resolve();
    }
    return dfd.promise();
  },

  /**
   * Delete Hive Metastore is performed
   * @type {bool}
   */
  deleteHiveMetaStore: false,

  /**
   * Delete WebHCat Server is performed
   *
   * @type {bool}
   */
  deleteWebHCatServer: false,

  getHiveHosts: function getHiveHosts() {
    var self = this;
    var removePerformed = this.get('fromDeleteHost') || this.get('deleteHiveMetaStore') || this.get('deleteHiveServer') || this.get('deleteWebHCatServer');
    var hiveMasterComponents = ['WEBHCAT_SERVER', 'HIVE_METASTORE', 'HIVE_SERVER'];
    var masterComponentsMap = hiveMasterComponents.map(function (componentName) {
      return self.bootstrapHostsMapping(componentName);
    }).reduce(function (p, c) {
      return p.concat(c);
    });

    if (removePerformed) {
      self.setProperties({
        deleteHiveMetaStore: false,
        deleteHiveServer: false,
        deleteWebHCatServer: false,
        fromDeleteHost: false
      });
      masterComponentsMap = masterComponentsMap.map(function (masterComponent) {
        masterComponent.isInstalled = masterComponent.hostName !== self.get('content.hostName');
        return masterComponent;
      });
    }

    if (!!this.get('hiveMetastoreHost')) {
      masterComponentsMap.push({
        component: 'HIVE_METASTORE',
        hostName: this.get('hiveMetastoreHost'),
        isInstalled: !removePerformed
      });
      this.set('hiveMetastoreHost', '');
    }

    if (!!this.get('webhcatServerHost')) {
      masterComponentsMap.push({
        component: 'WEBHCAT_SERVER',
        hostName: this.get('webhcatServerHost'),
        isInstalled: !removePerformed
      });
      this.set('webhcatServerHost', '');
    }

    return masterComponentsMap;
  },

  /**
   * Success callback for load configs request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method loadRangerConfigs
   */
  loadRangerConfigs: function loadRangerConfigs(data, opt, params) {
    var urlParams = this.getUrlParamsForConfigsRequest(data, ['core-site', 'hdfs-site', 'kms-env', 'kms-site']),
        request = App.ajax.send({
      name: 'admin.get.all_configurations',
      sender: this,
      data: {
        urlParams: urlParams
      },
      success: params.callback
    });
    this.trackRequest(request);
  },

  /**
   * update and save Hive hive.metastore.uris config to server
   * @param {object} data
   * @method onLoadRangerConfigs
   */
  onLoadRangerConfigs: function onLoadRangerConfigs(data) {
    var _this9 = this;

    var hdfsProperties = [{
      type: 'core-site',
      name: 'hadoop.security.key.provider.path'
    }, {
      type: 'hdfs-site',
      name: 'dfs.encryption.key.provider.uri'
    }],
        kmsSiteProperties = [{
      name: 'hadoop.kms.cache.enable',
      notHaValue: 'true',
      haValue: 'false'
    }, {
      name: 'hadoop.kms.authentication.zk-dt-secret-manager.enable',
      notHaValue: 'false',
      haValue: 'true'
    }, {
      name: 'hadoop.kms.cache.timeout.ms',
      notHaValue: '600000',
      haValue: '0'
    }, {
      name: 'hadoop.kms.current.key.cache.timeout.ms',
      notHaValue: '30000',
      haValue: '0'
    }, {
      name: 'hadoop.kms.authentication.signer.secret.provider',
      notHaValue: 'random',
      haValue: 'zookeeper'
    }, {
      name: 'hadoop.kms.authentication.signer.secret.provider.zookeeper.auth.type',
      notHaValue: 'kerberos',
      haValue: 'none'
    }, {
      name: 'hadoop.kms.authentication.signer.secret.provider.zookeeper.connection.string',
      notHaValue: '#HOSTNAME#:#PORT#,...',
      haValue: this.getZookeeperConnectionString()
    }],
        rkmsHosts = this.getRangerKMSServerHosts(),
        rkmsHostsStr = rkmsHosts.join(';'),
        isHA = rkmsHosts.length > 1,
        rkmsPort = data.items.findProperty('type', 'kms-env').properties['kms_port'],
        newValue = 'kms://http@' + rkmsHostsStr + ':' + rkmsPort + '/kms',
        coreSiteConfigs = data.items.findProperty('type', 'core-site'),
        hdfsSiteConfigs = data.items.findProperty('type', 'hdfs-site'),
        kmsSiteConfigs = data.items.findProperty('type', 'kms-site'),
        groups = [{
      properties: {
        'core-site': coreSiteConfigs.properties,
        'hdfs-site': hdfsSiteConfigs.properties
      },
      properties_attributes: {
        'core-site': coreSiteConfigs.properties_attributes,
        'hdfs-site': hdfsSiteConfigs.properties_attributes
      }
    }, {
      properties: {
        'kms-site': kmsSiteConfigs.properties
      },
      properties_attributes: {
        'kms-site': kmsSiteConfigs.properties_attributes
      }
    }],
        propertiesToChange = this.get('allPropertiesToChange');

    this.saveLoadedConfigs(data);

    hdfsProperties.forEach(function (property) {
      var typeConfigs = data.items.findProperty('type', property.type).properties,
          currentValue = typeConfigs[property.name],
          pattern = new RegExp('^kms:\\/\\/http@(.+):' + rkmsPort + '\\/kms$'),
          patternMatch = currentValue && currentValue.match(pattern),
          currentHostsList = patternMatch && patternMatch[1].split(';').sort().join(';');
      if (currentHostsList !== rkmsHostsStr) {
        typeConfigs[property.name] = newValue;
        if (_this9.get('isReconfigureRequired')) {
          var propertyFileName = property.type,
              propertyName = property.name,
              service = App.config.get('serviceByConfigTypeMap')[propertyFileName],
              configObject = App.configsCollection.getConfigByName(propertyName, propertyFileName),
              displayName = configObject && configObject.displayName;
          propertiesToChange.pushObject({
            propertyFileName: propertyFileName,
            propertyName: propertyName,
            propertyTitle: configObject && Em.I18n.t('installer.controls.serviceConfigPopover.title').format(displayName, displayName === propertyName ? '' : propertyName),
            propertyDescription: configObject && configObject.description,
            serviceDisplayName: service && service.get('displayName'),
            initialValue: currentValue,
            recommendedValue: newValue,
            saveRecommended: true
          });
        }
      }
    });

    kmsSiteProperties.forEach(function (property) {
      var currentValue = kmsSiteConfigs.properties[property.name];
      var newValue = isHA ? property.haValue : property.notHaValue;
      kmsSiteConfigs.properties[property.name] = newValue;

      propertiesToChange.pushObject({
        propertyFileName: 'kms-site',
        propertyName: property.name,
        serviceDisplayName: App.Service.find().findProperty('serviceName', 'RANGER_KMS').get('displayName'),
        initialValue: currentValue,
        recommendedValue: newValue,
        saveRecommended: true
      });
    });
    this.setConfigsChanges(groups);
  },

  /**
   * Delete Hive Metastore is performed
   * @type {bool}
   */
  deleteRangerKMSServer: false,

  getRangerKMSServerHosts: function getRangerKMSServerHosts() {
    var rkmsHosts = App.HostComponent.find().filterProperty('componentName', 'RANGER_KMS_SERVER').mapProperty('hostName');
    var rangerKMSServerHost = this.get('rangerKMSServerHost');

    if (!!rangerKMSServerHost) {
      rkmsHosts.push(rangerKMSServerHost);
    }

    if (this.get('fromDeleteHost') || this.get('deleteRangerKMSServer')) {
      return rkmsHosts.without(this.get('content.hostName'));
    }
    return rkmsHosts.sort();
  },

  getZookeeperConnectionString: function getZookeeperConnectionString() {
    var zookeeperHosts = App.MasterComponent.find('ZOOKEEPER_SERVER').get('hostNames');
    return zookeeperHosts.map(function (host) {
      return host + ':2181';
    }).join(',');
  },

  /**
   * Delete Storm Nimbus is performed
   * @type {bool}
   */
  deleteNimbusHost: false,

  getStormNimbusHosts: function getStormNimbusHosts() {
    var stormNimbusHosts = App.HostComponent.find().filterProperty('componentName', 'NIMBUS').mapProperty('hostName'),
        nimbusHost = this.get('nimbusHost');

    if (!!nimbusHost) {
      stormNimbusHosts.push(nimbusHost);
      this.set('nimbusHost', '');
    }

    if (this.get('fromDeleteHost') || this.get('deleteNimbusHost')) {
      this.set('deleteNimbusHost', false);
      this.set('fromDeleteHost', false);
      return stormNimbusHosts.without(this.get('content.hostName'));
    }
    return stormNimbusHosts.sort();
  },

  getAtlasServerHosts: function getAtlasServerHosts() {
    var atlasServerHosts = App.HostComponent.find().filterProperty('componentName', 'ATLAS_SERVER').mapProperty('hostName'),
        atlasServer = this.get('atlasServer');

    if (!!atlasServer) {
      atlasServerHosts.push(atlasServer);
      this.set('atlasServer', '');
    }

    if (this.get('fromDeleteHost') || this.get('deleteAtlasServer')) {
      this.set('deleteAtlasServer', false);
      this.set('fromDeleteHost', false);
      return atlasServerHosts.without(this.get('content.hostName'));
    }
    return atlasServerHosts.sort();
  },

  /**
   * Send command to server to resfresh configs of selected component
   * @param {object} event
   * @method refreshComponentConfigs
   */
  refreshComponentConfigs: function refreshComponentConfigs(event) {
    var self = this;
    return App.showConfirmationPopup(function () {
      var component = event.context;
      var context = Em.I18n.t('requestInfo.refreshComponentConfigs').format(component.get('displayName'));
      self.sendRefreshComponentConfigsCommand(component, context);
    });
  },

  /**
   * PUTs a command to server to refresh configs of host component.
   * @param {object} component
   * @param {object} context Context under which this command is beign sent.
   * @method sendRefreshComponentConfigsCommand
   */
  sendRefreshComponentConfigsCommand: function sendRefreshComponentConfigsCommand(component, context) {
    var resource_filters = [{
      service_name: component.get('service.serviceName'),
      component_name: component.get('componentName'),
      hosts: component.get('host.hostName')
    }];
    App.ajax.send({
      name: 'host.host_component.refresh_configs',
      sender: this,
      data: {
        resource_filters: resource_filters,
        context: context
      },
      success: 'refreshComponentConfigsSuccessCallback',
      showLoadingPopup: true
    });
  },

  /**
   * Success callback for refresh host component configs request
   * @method refreshComponentConfigsSuccessCallback
   */
  refreshComponentConfigsSuccessCallback: function refreshComponentConfigsSuccessCallback() {
    this.showBackgroundOperationsPopup();
  },

  /**
   * Load configs
   * This function when used without a callback should be always used from successcallback function of the promise `App.router.get('mainController').isLoading.call(App.router.get('clusterController'), 'isServiceContentFullyLoaded').done(promise)`
   * This is required to make sure that service metrics API determining the HA state of components is loaded
   * @method loadConfigs
   */
  loadConfigs: function loadConfigs(configTagsCallback, configsCallback) {
    var request = App.ajax.send({
      name: 'config.tags',
      sender: this,
      data: {
        callback: configsCallback
      },
      success: configTagsCallback,
      error: 'onLoadConfigsErrorCallback'
    });
    this.trackRequest(request);
  },

  /**
   * onLoadConfigsErrorCallback
   * @method onLoadConfigsErrorCallback
   */
  onLoadConfigsErrorCallback: function onLoadConfigsErrorCallback() {
    this.get('isOozieConfigLoaded').reject();
  },

  /**
   * Success callback for load configs request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method loadZookeeperConfigs
   */
  loadZookeeperConfigs: function loadZookeeperConfigs(data, opt, params) {
    var urlParams = this.constructZookeeperConfigUrlParams(data);
    if (urlParams.length > 0) {
      var request = App.ajax.send({
        name: 'reassign.load_configs',
        sender: this,
        data: {
          urlParams: urlParams.join('|')
        },
        success: params.callback
      });
      this.trackRequest(request);
      return true;
    }
    this.set('isConfigsLoadingInProgress', false);
    return false;
  },

  /**
   * construct URL params for query, that load configs
   * @param data {Object}
   * @return {Array}
   */
  constructZookeeperConfigUrlParams: function constructZookeeperConfigUrlParams(data) {
    var urlParams = [];
    var services = App.Service.find();
    var zooKeeperRelatedServices = this.get('zooKeeperRelatedServices').slice(0);
    if (App.get('isHaEnabled')) {
      zooKeeperRelatedServices.push({
        serviceName: 'HDFS',
        typesToLoad: ['core-site'],
        typesToSave: ['core-site']
      });
    }
    zooKeeperRelatedServices.forEach(function (service) {
      if (services.someProperty('serviceName', service.serviceName)) {
        service.typesToLoad.forEach(function (type) {
          if (data.Clusters.desired_configs[type]) {
            urlParams.push('(type=' + type + '&tag=' + data.Clusters.desired_configs[type].tag + ')');
          }
        });
      }
    });
    return urlParams;
  },

  /**
   * save new ZooKeeper configs to server
   * @param {object} data
   * @method saveZkConfigs
   */
  saveZkConfigs: function saveZkConfigs(data) {
    var configs = {};
    var attributes = {};
    this.saveLoadedConfigs(data);
    data.items.forEach(function (item) {
      configs[item.type] = item.properties;
      attributes[item.type] = item.properties_attributes || {};
    }, this);

    this.updateZkConfigs(configs);
    var groups = [];
    var installedServiceNames = App.Service.find().mapProperty('serviceName');
    var zooKeeperRelatedServices = this.get('zooKeeperRelatedServices').slice(0);
    if (App.get('isHaEnabled')) {
      zooKeeperRelatedServices.push({
        serviceName: 'HDFS',
        typesToLoad: ['core-site'],
        typesToSave: ['core-site']
      });
    }
    zooKeeperRelatedServices.forEach(function (service) {
      if (installedServiceNames.contains(service.serviceName)) {
        var group = {
          properties: {},
          properties_attributes: {}
        };
        service.typesToSave.forEach(function (type) {
          group.properties[type] = configs[type];
          group.properties_attributes[type] = attributes[type];
        });
        groups.push(group);
      }
    });
    this.setConfigsChanges(groups);
  },

  /**
   * Is deleteHost action id fired
   * @type {bool}
   */
  fromDeleteHost: false,

  /**
   * Is ZooKeeper Server being deleted from host
   * @type {bool}
   */
  fromDeleteZkServer: false,

  /**
   * Send command to server to install selected host component
   * @param {Object} event
   * @method installComponent
   */
  installComponent: function installComponent(event) {
    var self = this;
    var component = event.context;
    var componentName = component.get('componentName');
    var displayName = component.get('displayName');

    return App.ModalPopup.show({
      primary: Em.I18n.t('hosts.host.installComponent.popup.confirm'),
      header: Em.I18n.t('popup.confirmation.commonHeader'),
      installComponentMessage: Em.I18n.t('hosts.host.installComponent.msg').format(displayName),
      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/details/installComponentPopup')
      }),
      onPrimary: function onPrimary() {
        var _this = this;
        App.get('router.mainAdminKerberosController').getSecurityType(function () {
          App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
            _this.hide();

            App.ajax.send({
              name: 'common.host.host_component.update',
              sender: self,
              data: {
                hostName: self.get('content.hostName'),
                serviceName: component.get('service.serviceName'),
                componentName: componentName,
                component: component,
                context: Em.I18n.t('requestInfo.installHostComponent') + " " + displayName,
                HostRoles: {
                  state: 'INSTALLED'
                }
              },
              success: 'installComponentSuccessCallback',
              error: 'ajaxErrorCallback',
              showLoadingPopup: true
            });
          });
        });
      }
    });
  },

  /**
   * Success callback for install component request
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method installComponentSuccessCallback
   */
  installComponentSuccessCallback: function installComponentSuccessCallback(data, opt, params) {
    if (App.get('testMode')) {
      this.mimicWorkStatusChange(params.component, App.HostComponentStatus.installing, App.HostComponentStatus.stopped);
    }
    this.showBackgroundOperationsPopup();
  },

  /**
   * Send command to server to run decommission on DATANODE, TASKTRACKER, NODEMANAGER, REGIONSERVER
   * @param {App.HostComponent} component
   * @param {callback} callback
   * @method decommission
   */
  decommission: function decommission(component, callback) {
    var self = this;
    return App.showConfirmationPopup(function () {
      self.runDecommission.call(self, self.get('content.hostName'), component.get('service.serviceName'));
      if (callback) {
        callback();
      }
    }, Em.I18n.t('question.sure.decommission').format(component.get('service.serviceName')));
  },
  /**
   * identify correct component to run decommission on them by service name,
   * in result run proper decommission method
   * @param hostName
   * @param svcName
   */
  runDecommission: function runDecommission(hostName, svcName) {
    switch (svcName) {
      case 'HDFS':
        this.doDecommission(hostName, svcName, "NAMENODE", "DATANODE");
        break;
      case 'YARN':
        this.doDecommission(hostName, svcName, "RESOURCEMANAGER", "NODEMANAGER");
        break;
      case 'HBASE':
        this.warnBeforeDecommission(hostName);
    }
  },

  /**
   * Send command to server to run recommission on DATANODE, TASKTRACKER, NODEMANAGER
   * @param {App.HostComponent} component
   * @param {callback} callback
   * @method recommission
   */
  recommission: function recommission(component, callback) {
    var self = this;
    return App.showConfirmationPopup(function () {
      self.runRecommission.call(self, self.get('content.hostName'), component.get('service.serviceName'));
      if (callback) {
        callback();
      }
    }, Em.I18n.t('question.sure.recommission').format(component.get('service.serviceName')));
  },
  /**
   * identify correct component to run recommission on them by service name,
   * in result run proper recommission method
   * @param hostName
   * @param svcName
   */
  runRecommission: function runRecommission(hostName, svcName) {
    switch (svcName) {
      case 'HDFS':
        this.doRecommissionAndStart(hostName, svcName, "NAMENODE", "DATANODE");
        break;
      case 'YARN':
        this.doRecommissionAndStart(hostName, svcName, "RESOURCEMANAGER", "NODEMANAGER");
        break;
      case 'HBASE':
        this.doRecommissionAndStart(hostName, svcName, "HBASE_MASTER", "HBASE_REGIONSERVER");
    }
    this.showBackgroundOperationsPopup();
  },

  /**
   * Performs Decommission (for DN, TT and NM)
   * @param {string} hostName
   * @param {string} serviceName
   * @param {string} componentName
   * @param {string} slaveType
   * @method doDecommission
   */
  doDecommission: function doDecommission(hostName, serviceName, componentName, slaveType) {
    var contextNameString = 'hosts.host.' + slaveType.toLowerCase() + '.decommission';
    var context = Em.I18n.t(contextNameString);
    App.ajax.send({
      name: 'host.host_component.decommission_slave',
      sender: this,
      data: {
        context: context,
        command: 'DECOMMISSION',
        hostName: hostName,
        serviceName: serviceName,
        componentName: componentName,
        slaveType: slaveType
      },
      success: 'decommissionSuccessCallback',
      error: 'decommissionErrorCallback',
      showLoadingPopup: true
    });
  },

  /**
   * check is hbase regionserver in mm. If so - run decommission
   * otherwise shows warning
   * @method warnBeforeDecommission
   * @param {string} hostNames - list of host when run from bulk operations or current host
   */
  warnBeforeDecommission: function warnBeforeDecommission(hostNames) {
    if (this.get('content.hostComponents').findProperty('componentName', 'HBASE_REGIONSERVER').get('passiveState') == "OFF") {
      this.showHbaseActiveWarning();
    } else {
      this.checkRegionServerState(hostNames);
    }
  },

  /**
   *  send call to check is this regionserver last in cluster which has desired_admin_state property "INSERVICE"
   * @method checkRegionServerState
   * @param hostNames
   */
  checkRegionServerState: function checkRegionServerState(hostNames) {
    return App.ajax.send({
      name: 'host.region_servers.in_inservice',
      sender: this,
      data: {
        hostNames: hostNames
      },
      success: 'checkRegionServerStateSuccessCallback'
    });
  },

  /**
   * check is this regionserver last in cluster which has desired_admin_state property "INSERVICE"
   * @method checkRegionServerStateSuccessCallback
   * @param data
   * @param opt
   * @param params
   */
  checkRegionServerStateSuccessCallback: function checkRegionServerStateSuccessCallback(data, opt, params) {
    var hostArray = params.hostNames.split(",");
    var decommissionPossible = data.items.mapProperty('HostRoles.host_name').filter(function (hostName) {
      return !hostArray.contains(hostName);
    }, this).length >= 1;
    if (decommissionPossible) {
      this.doDecommissionRegionServer(params.hostNames, "HBASE", "HBASE_MASTER", "HBASE_REGIONSERVER");
    } else {
      this.showRegionServerWarning();
    }
  },

  /**
   * show warning that regionserver is last in cluster which has desired_admin_state property "INSERVICE"
   * @method showRegionServerWarning
   * @param hostNames
   */
  showRegionServerWarning: function showRegionServerWarning() {
    return App.ModalPopup.show({
      header: Em.I18n.t('common.warning'),
      message: Em.I18n.t('hosts.host.hbase_regionserver.decommission.warning'),
      bodyClass: Ember.View.extend({
        template: Em.Handlebars.compile('<div class="alert alert-warning">{{message}}</div>')
      }),
      secondary: false
    });
  },

  /**
   * shows warning: put hbase regionserver in passive state
   * @method showHbaseActiveWarning
   * @return {App.ModalPopup}
   */
  showHbaseActiveWarning: function showHbaseActiveWarning() {
    return App.ModalPopup.show({
      header: Em.I18n.t('common.warning'),
      message: Em.I18n.t('hostPopup.recommendation.beforeDecommission').format(App.format.components["HBASE_REGIONSERVER"]),
      bodyClass: Ember.View.extend({
        template: Em.Handlebars.compile('<div class="alert alert-warning">{{message}}</div>')
      }),
      secondary: false
    });
  },

  /**
   * Performs Decommission (for RegionServer)
   * @method doDecommissionRegionServer
   * @param {string} hostNames - list of host when run from bulk operations or current host
   * @param {string} serviceName - serviceName
   * @param {string} componentName - master compoent name
   * @param {string} slaveType - slave component name
   */
  doDecommissionRegionServer: function doDecommissionRegionServer(hostNames, serviceName, componentName, slaveType) {
    var batches = [{
      "order_id": 1,
      "type": "POST",
      "uri": "/clusters/" + App.get('clusterName') + "/requests",
      "RequestBodyInfo": {
        "RequestInfo": {
          "context": Em.I18n.t('hosts.host.regionserver.decommission.batch1'),
          "command": "DECOMMISSION",
          "exclusive": "true",
          "parameters": {
            "slave_type": slaveType,
            "excluded_hosts": hostNames
          },
          'operation_level': {
            level: "HOST_COMPONENT",
            cluster_name: App.get('clusterName'),
            host_name: hostNames,
            service_name: serviceName
          }
        },
        "Requests/resource_filters": [{ "service_name": serviceName, "component_name": componentName }]
      }
    }];
    var id = 2;
    var hAray = hostNames.split(",");
    for (var i = 0; i < hAray.length; i++) {
      batches.push({
        "order_id": id,
        "type": "PUT",
        "uri": "/clusters/" + App.get('clusterName') + "/hosts/" + hAray[i] + "/host_components/" + slaveType,
        "RequestBodyInfo": {
          "RequestInfo": {
            context: Em.I18n.t('hosts.host.regionserver.decommission.batch2'),
            exclusive: true,
            operation_level: {
              level: "HOST_COMPONENT",
              cluster_name: App.get('clusterName'),
              host_name: hostNames,
              service_name: serviceName || null
            }
          },
          "Body": {
            HostRoles: {
              state: "INSTALLED"
            }
          }
        }
      });
      id++;
    }
    batches.push({
      "order_id": id,
      "type": "POST",
      "uri": "/clusters/" + App.get('clusterName') + "/requests",
      "RequestBodyInfo": {
        "RequestInfo": {
          "context": Em.I18n.t('hosts.host.regionserver.decommission.batch3'),
          "command": "DECOMMISSION",
          "service_name": serviceName,
          "component_name": componentName,
          "parameters": {
            "slave_type": slaveType,
            "excluded_hosts": hostNames,
            "mark_draining_only": true
          },
          'operation_level': {
            level: "HOST_COMPONENT",
            cluster_name: App.get('clusterName'),
            host_name: hostNames,
            service_name: serviceName
          }
        },
        "Requests/resource_filters": [{ "service_name": serviceName, "component_name": componentName }]
      }
    });
    App.ajax.send({
      name: 'common.batch.request_schedules',
      sender: this,
      data: {
        intervalTimeSeconds: 1,
        tolerateSize: 0,
        batches: batches
      },
      success: 'decommissionSuccessCallback',
      error: 'decommissionErrorCallback',
      showLoadingPopup: true
    });
  },

  /**
   * Error callback for decommission requests
   * @param {object} request
   * @param {object} ajaxOptions
   * @param {string} error
   * @method decommissionErrorCallback
   */
  decommissionErrorCallback: function decommissionErrorCallback(request, ajaxOptions, error) {},

  /**
   * Success ajax response for Recommission/Decommission slaves
   * @param {object} data
   * @method decommissionSuccessCallback
   * @return {Boolean}
   */
  decommissionSuccessCallback: function decommissionSuccessCallback(data) {
    if (data && (data.Requests || data.resources[0].RequestSchedule)) {
      this.showBackgroundOperationsPopup();
      return true;
    } else {
      return false;
    }
  },

  /**
   * Performs Recommission and Start
   * @param {string} hostNames
   * @param {string} serviceName
   * @param {string} componentName
   * @param {string} slaveType
   * @method doRecommissionAndStart
   */
  doRecommissionAndStart: function doRecommissionAndStart(hostNames, serviceName, componentName, slaveType) {
    var contextNameString_1 = 'hosts.host.' + slaveType.toLowerCase() + '.recommission';
    var context_1 = Em.I18n.t(contextNameString_1);
    var contextNameString_2 = 'requestInfo.startHostComponent.' + slaveType.toLowerCase();
    var startContext = Em.I18n.t(contextNameString_2);
    var params = {
      "slave_type": slaveType,
      "included_hosts": hostNames
    };
    if (serviceName == "HBASE") {
      params.mark_draining_only = true;
    }
    var batches = [{
      "order_id": 1,
      "type": "POST",
      "uri": "/clusters/" + App.get('clusterName') + "/requests",
      "RequestBodyInfo": {
        "RequestInfo": {
          "context": context_1,
          "command": "DECOMMISSION",
          "exclusive": "true",
          "parameters": params,
          'operation_level': {
            level: "HOST_COMPONENT",
            cluster_name: App.get('clusterName'),
            host_name: hostNames,
            service_name: serviceName
          }
        },
        "Requests/resource_filters": [{ "service_name": serviceName, "component_name": componentName }]
      }
    }];
    var id = 2;
    var hAray = hostNames.split(",");
    for (var i = 0; i < hAray.length; i++) {
      batches.push({
        "order_id": id,
        "type": "PUT",
        "uri": "/clusters/" + App.get('clusterName') + "/hosts/" + hAray[i] + "/host_components/" + slaveType,
        "RequestBodyInfo": {
          "RequestInfo": {
            context: startContext,
            operation_level: {
              level: "HOST_COMPONENT",
              cluster_name: App.get('clusterName'),
              host_name: hostNames,
              service_name: serviceName || null
            }
          },
          "Body": {
            HostRoles: {
              state: "STARTED"
            }
          }
        }
      });
      id++;
    }
    App.ajax.send({
      name: 'common.batch.request_schedules',
      sender: this,
      data: {
        intervalTimeSeconds: 1,
        tolerateSize: 1,
        batches: batches
      },
      success: 'decommissionSuccessCallback',
      error: 'decommissionErrorCallback',
      showLoadingPopup: true
    });
  },

  /**
   * Handler for host-menu items actions
   * @param {object} option
   * @method doAction
   */
  doAction: function doAction(option) {
    switch (option.context.action) {
      case "deleteHost":
        this.validateAndDeleteHost();
        break;
      case "startAllComponents":
        if (!this.get('content.isNotHeartBeating')) this.doStartAllComponents();
        break;
      case "stopAllComponents":
        if (!this.get('content.isNotHeartBeating')) this.doStopAllComponents();
        break;
      case "restartAllComponents":
        if (!this.get('content.isNotHeartBeating')) this.doRestartAllComponents();
        break;
      case "onOffPassiveModeForHost":
        this.onOffPassiveModeForHost(option.context);
        break;
      case "setRackId":
        this.setRackIdForHost();
        break;
      case "checkHost":
        this.runHostCheckConfirmation();
        break;
      case "regenerateKeytabFileOperations":
        this.regenerateKeytabFileOperations();
        break;
    }
  },

  /**
   * Turn On/Off Passive Mode for host
   * @param {object} context
   * @method onOffPassiveModeForHost
   */
  onOffPassiveModeForHost: function onOffPassiveModeForHost(context) {
    var state = context.active ? 'ON' : 'OFF';
    var self = this;
    var message = Em.I18n.t('hosts.host.details.for.postfix').format(context.label);
    var popupInfo = Em.I18n.t('hosts.passiveMode.popup').format(context.active ? 'On' : 'Off', this.get('content.hostName'));
    if (state === 'OFF') {
      var currentHostVersion = this.get('content.stackVersions') && this.get('content.stackVersions').findProperty('isCurrent'),
          hostVersion = currentHostVersion && currentHostVersion.get('repoVersion'),
          currentVersion = App.StackVersion.find().findProperty('isCurrent'),
          clusterVersion = currentVersion && currentVersion.get('repositoryVersion.repositoryVersion');
      if (hostVersion !== clusterVersion) {
        var msg = Em.I18n.t("hosts.passiveMode.popup.version.mismatch").format(this.get('content.hostName'), clusterVersion);
        popupInfo += '<br/><div class="alert alert-warning">' + msg + '</div>';
      }
    }
    return App.showConfirmationPopup(function () {
      self.hostPassiveModeRequest(state, message);
    }, popupInfo);
  },

  /**
   * Set rack id for host
   * @method setRackIdForHost
   */
  setRackIdForHost: function setRackIdForHost() {
    var hostNames = [{ hostName: this.get('content.hostName') }];
    var rack = this.get('content.rack');
    var operationData = { message: Em.I18n.t('hosts.host.details.setRackId') };
    hostsManagement.setRackInfo(operationData, hostNames, rack);
  },

  /**
   * Send request to get passive state for host
   * @param {string} state
   * @param {string} message
   * @method hostPassiveModeRequest
   */
  hostPassiveModeRequest: function hostPassiveModeRequest(state, message) {
    App.ajax.send({
      name: 'bulk_request.hosts.passive_state',
      sender: this,
      data: {
        hostNames: this.get('content.hostName'),
        passive_state: state,
        requestInfo: message
      },
      success: 'updateHost'
    });
  },

  /**
   * Success callback for receiving host passive state
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method updateHost
   */
  updateHost: function updateHost(data, opt, params) {
    this.set('content.passiveState', params.passive_state);
    batchUtils.infoPassiveState(params.passive_state);
  },

  /**
   * Send request to get passive state for hostComponent
   * @param {object} component - hostComponentn object
   * @param {string} state
   * @param {string} message
   * @method hostPassiveModeRequest
   */
  updateComponentPassiveState: function updateComponentPassiveState(component, state, message) {
    App.ajax.send({
      name: 'common.host.host_component.passive',
      sender: this,
      data: {
        hostName: this.get('content.hostName'),
        componentName: component.get('componentName'),
        component: component,
        passive_state: state,
        context: message
      },
      success: 'updateHostComponent'
    });
  },

  /**
   * Success callback for receiving hostComponent passive state
   * @param {object} data
   * @param {object} opt
   * @param {object} params
   * @method updateHost
   */
  updateHostComponent: function updateHostComponent(data, opt, params) {
    params.component.set('passiveState', params.passive_state);
    batchUtils.infoPassiveState(params.passive_state);
  },

  /**
   * Show confirmation popup for action "start all components"
   * @method doStartAllComponents
   */
  doStartAllComponents: function doStartAllComponents() {
    var self = this;
    var components = this.get('serviceNonClientActiveComponents');

    if (components && components.get('length')) {
      return App.showConfirmationPopup(function () {
        self.sendComponentCommand(components, Em.I18n.t('hosts.host.maintainance.startAllComponents.context'), App.HostComponentStatus.started);
      }, Em.I18n.t('question.sure.startAll'));
    }
  },

  /**
   * Show confirmation popup for action "stop all components"
   * @method doStopAllComponents
   */
  doStopAllComponents: function doStopAllComponents() {
    var self = this;
    var components = this.get('serviceNonClientActiveComponents');

    if (components && components.get('length')) {
      if (components.someProperty('componentName', 'NAMENODE') && this.get('content.hostComponents').filterProperty('componentName', 'NAMENODE').someProperty('workStatus', App.HostComponentStatus.started)) {
        this.checkNnLastCheckpointTime(function () {
          App.showConfirmationPopup(function () {
            self.sendComponentCommand(components, Em.I18n.t('hosts.host.maintainance.stopAllComponents.context'), App.HostComponentStatus.stopped);
          }, Em.I18n.t('question.sure.stopAll'));
        });
      } else {
        return App.showConfirmationPopup(function () {
          self.sendComponentCommand(components, Em.I18n.t('hosts.host.maintainance.stopAllComponents.context'), App.HostComponentStatus.stopped);
        }, Em.I18n.t('question.sure.stopAll'));
      }
    }
  },

  /**
   * Show confirmation popup for action "restart all components"
   * @method doRestartAllComponents
   */
  doRestartAllComponents: function doRestartAllComponents() {
    var self = this;
    var components = this.get('serviceActiveComponents');

    if (components && components.get('length')) {
      if (components.someProperty('componentName', 'NAMENODE') && this.get('content.hostComponents').filterProperty('componentName', 'NAMENODE').someProperty('workStatus', App.HostComponentStatus.started)) {
        this.checkNnLastCheckpointTime(function () {
          App.showConfirmationPopup(function () {
            batchUtils.restartHostComponents(components, Em.I18n.t('rollingrestart.context.allOnSelectedHost').format(self.get('content.hostName')), "HOST");
          }, Em.I18n.t('question.sure.restartAll'));
        });
      } else {
        return App.showConfirmationPopup(function () {
          batchUtils.restartHostComponents(components, Em.I18n.t('rollingrestart.context.allOnSelectedHost').format(self.get('content.hostName')), "HOST");
        }, Em.I18n.t('question.sure.restartAll'));
      }
    }
  },

  /**
   * get info about host-components, exactly:
   *  - host-components grouped by status, features
   *  - flag, that indicate whether ZooKeeper Server is installed
   * @return {Object}
   */
  getHostComponentsInfo: function getHostComponentsInfo(hostComponents) {
    var componentsOnHost = hostComponents || this.get('content.hostComponents');
    var stoppedStates = [App.HostComponentStatus.stopped, App.HostComponentStatus.install_failed, App.HostComponentStatus.upgrade_failed, App.HostComponentStatus.init, App.HostComponentStatus.unknown];
    var container = {
      isReconfigureRequired: false,
      lastComponents: [],
      masterComponents: [],
      nonAddableMasterComponents: [],
      lastMasterComponents: [],
      runningComponents: [],
      nonDeletableComponents: [],
      unknownComponents: [],
      toDecommissionComponents: []
    };
    var addDeleteComponentsMap = this.get('addDeleteComponentsMap');

    if (componentsOnHost && componentsOnHost.get('length') > 0) {
      componentsOnHost.forEach(function (cInstance) {
        if (addDeleteComponentsMap[cInstance.get('componentName')]) {
          container.isReconfigureRequired = true;
        }
        var isLastComponent = false;
        if (this.getTotalComponent(cInstance) === 1) {
          container.lastComponents.push(cInstance.get('displayName'));
          isLastComponent = true;
        }
        var workStatus = cInstance.get('workStatus');

        if (cInstance.get('isMaster')) {
          var displayName = cInstance.get('displayName');
          container.masterComponents.push(displayName);
          if (!App.StackServiceComponent.find(cInstance.get('componentName')).get('isMasterAddableInstallerWizard')) {
            container.nonAddableMasterComponents.push(displayName);
          }
          if (isLastComponent) {
            container.lastMasterComponents.push(displayName);
          }
        }
        if (stoppedStates.indexOf(workStatus) < 0) {
          container.runningComponents.push(cInstance.get('displayName'));
        }
        if (!cInstance.get('isDeletable')) {
          container.nonDeletableComponents.push(cInstance.get('displayName'));
        }
        if (workStatus === App.HostComponentStatus.unknown) {
          container.unknownComponents.push(cInstance.get('displayName'));
        }
        if (App.get('components.decommissionAllowed').contains(cInstance.get('componentName')) && !cInstance.get('view.isComponentRecommissionAvailable')) {
          container.toDecommissionComponents.push(cInstance.get('displayName'));
        }
      }, this);
    }
    return container;
  },

  /**
   * Run host check confirmation
   * @method runHostCheckConfirmation
   */
  runHostCheckConfirmation: function runHostCheckConfirmation() {
    var self = this;
    var popupInfo = Em.I18n.t('hosts.checkHost.popup').format(this.get('content.hostName'));

    return App.showConfirmationPopup(function () {
      self.runHostCheck();
    }, popupInfo);
  },

  getDataForHostCheck: function getDataForHostCheck() {
    var hostName = this.get('content.hostName');
    var jdk_location = App.router.get('clusterController.ambariProperties.jdk_location');
    var RequestInfo = {
      "action": "check_host",
      "context": "Check host",
      "parameters": {
        "hosts": hostName,
        "check_execute_list": "last_agent_env_check,installed_packages,existing_repos,transparentHugePage",
        "jdk_location": jdk_location,
        "threshold": "20"
      }
    };

    return {
      RequestInfo: RequestInfo,
      resource_filters: { "hosts": hostName }
    };
  },

  /**
   * Callback for runHostCheckConfirmation
   * @method runHostCheck
   */
  runHostCheck: function runHostCheck() {
    var dataForCheckHostRequest = this.getDataForHostCheck();

    this.setProperties({
      stopChecking: false,
      checkHostFinished: false,
      isRerun: false
    });
    this.setBootHostsProp();
    this.showHostWarningsPopup();
    this.requestToPerformHostCheck(dataForCheckHostRequest);
  },

  /**
   * Shape controller's bootHosts property needed to host check
   * @method setBootHostsProp
   */
  setBootHostsProp: function setBootHostsProp() {
    var host = this.get('content');
    var bootHosts = [];

    host.name = host.get('hostName');
    bootHosts.push(host);

    this.set('bootHosts', bootHosts);
  },

  /**
   * Open popup that contain hosts' warnings
   * @return {App.ModalPopup}
   * @method showHostWarningsPopup
   */
  showHostWarningsPopup: function showHostWarningsPopup() {
    var self = this;

    return App.ModalPopup.show({

      header: Em.I18n.t('installer.step3.warnings.popup.header'),

      secondary: Em.I18n.t('installer.step3.hostWarningsPopup.rerunChecks'),

      primary: Em.I18n.t('common.close'),

      autoHeight: false,

      onPrimary: function onPrimary() {
        self.set('checksUpdateStatus', null);
        this.hide();
      },

      onClose: function onClose() {
        self.set('checksUpdateStatus', null);
        this.hide();
      },

      onSecondary: function onSecondary() {
        self.set('checkHostFinished', false);
        self.rerunChecks();
      },

      didInsertElement: function didInsertElement() {
        this._super();
        this.fitHeight();
      },

      footerClass: App.WizardStep3HostWarningPopupFooter.reopen({
        footerControllerBinding: 'App.router.mainHostDetailsController',
        checkHostFinished: function () {
          return this.get('footerController.checkHostFinished');
        }.property('footerController.checkHostFinished')
      }),

      bodyClass: App.WizardStep3HostWarningPopupBody.reopen({
        bodyControllerBinding: 'App.router.mainHostDetailsController',
        checkHostFinished: function () {
          return this.get('bodyController.checkHostFinished');
        }.property('bodyController.checkHostFinished')
      })
    });
  },

  /**
   * Deletion of hosts not supported for this version
   * @method validateAndDeleteHost
   */
  validateAndDeleteHost: function validateAndDeleteHost() {
    var container = this.getHostComponentsInfo();

    if (container.nonDeletableComponents.length > 0) {
      this.raiseDeleteComponentsError(container, 'nonDeletableList');
      return;
    } else if (container.nonAddableMasterComponents.length > 0) {
      this.raiseDeleteComponentsError(container, 'masterList');
      return;
    } else if (container.runningComponents.length > 0) {
      this.raiseDeleteComponentsError(container, 'runningList');
      return;
    } else if (container.lastMasterComponents.length > 0) {
      this.raiseDeleteComponentsError(container, 'lastMasterList');
      return;
    }

    this.set('fromDeleteHost', true);

    if (container.isReconfigureRequired) {
      this.reconfigureAndDeleteHost(container);
    } else {
      this.confirmDeleteHost(container);
    }
  },

  /**
   *
   * @param {object} container
   */
  reconfigureAndDeleteHost: function reconfigureAndDeleteHost(container) {
    var _this10 = this;

    var addDeleteComponentsMap = this.get('addDeleteComponentsMap');
    var hostName = this.get('content.hostName');
    var reconfiguredComponents = [];
    var componentStub = Em.Object.create();

    this.get('content.hostComponents').forEach(function (component) {
      var componentsMapItem = addDeleteComponentsMap[component.get('componentName')];
      if (componentsMapItem) {
        reconfiguredComponents.push(component.get('displayName'));
        if (componentsMapItem.hostPropertyName) {
          _this10.set(componentsMapItem.hostPropertyName, hostName);
        }
        if (componentsMapItem.addPropertyName) {
          _this10.set(componentsMapItem.addPropertyName, true);
        }
        _this10.loadComponentRelatedConfigs(componentsMapItem.configTagsCallbackName, componentsMapItem.configsCallbackName);
      }
    });
    this.showReconfigurationPopupPreDelete(componentStub, function () {
      _this10.confirmDeleteHost(container);
    }, Em.I18n.t('hosts.host.delete.componentsRequireReconfigure').format(reconfiguredComponents.join(', ')));
  },

  /**
   * Show popup with info about reasons why host can't be deleted
   * @param {string[]} components
   * @param {string} type
   * @method raiseDeleteComponentsError
   */
  raiseDeleteComponentsError: function raiseDeleteComponentsError(container, type) {
    App.ModalPopup.show({
      header: Em.I18n.t('hosts.cant.do.popup.title'),
      type: type,
      showBodyEnd: Em.computed.existsIn('type', ['runningList', 'masterList', 'lastMasterList']),
      container: container,
      components: function () {
        var container = this.get('container');
        switch (this.get('type')) {
          case 'masterList':
            return container.nonAddableMasterComponents;
          case 'nonDeletableList':
            return container.nonDeletableComponents;
          case 'runningList':
            return container.runningComponents;
          case 'lastMasterList':
            return container.lastMasterComponents;
        }
      }.property('type'),
      componentsStr: function () {
        return this.get('components').join(", ");
      }.property('components.[]'),
      componentsBody: Em.computed.i18nFormat('hosts.cant.do.popup.' + type + '.body', 'components.length'),
      componentsBodyEnd: function () {
        if (this.get('showBodyEnd')) {
          return Em.I18n.t('hosts.cant.do.popup.' + type + '.body.end').format(App.get('components.decommissionAllowed').map(function (c) {
            return App.format.role(c, false);
          }).join(", "));
        }
        return '';
      }.property(),
      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/details/raiseDeleteComponentErrorPopup')
      }),
      secondary: null
    });
  },

  /**
   * Show confirmation popup to delete host
   * @param {Object} container
   * @method confirmDeleteHost
   */
  confirmDeleteHost: function confirmDeleteHost(container) {
    var self = this;
    return App.ModalPopup.show({
      header: Em.I18n.t('hosts.delete.popup.title'),
      deletePopupBody: Em.I18n.t('hosts.delete.popup.body').format(self.get('content.publicHostName')),
      lastComponent: function () {
        if (container.lastComponents && container.lastComponents.length) {
          this.set('isChecked', false);
          return true;
        } else {
          this.set('isChecked', true);
          return false;
        }
      }.property(),
      disablePrimary: Em.computed.not('isChecked'),
      isChecked: false,
      lastComponentError: Em.View.extend({
        template: Em.Handlebars.compile(Em.I18n.t('hosts.delete.popup.body.msg4').format(container.lastComponents))
      }),
      unknownComponents: function () {
        if (container.unknownComponents && container.unknownComponents.length) {
          return container.unknownComponents.join(", ");
        }
        return '';
      }.property(),
      decommissionWarning: Em.View.extend({
        template: Em.Handlebars.compile(Em.I18n.t('hosts.delete.popup.body.msg7').format(container.toDecommissionComponents.join(', ')))
      }),
      toDecommissionComponents: container.toDecommissionComponents,
      bodyClass: Em.View.extend({
        templateName: require('templates/main/host/details/doDeleteHostPopup')
      }),
      onPrimary: function onPrimary() {
        this.hide();
        self.doDeleteHost();
      }
    });
  },

  /**
   * send DELETE calls to components of host and after delete host itself
   * @method doDeleteHost
   */
  doDeleteHost: function doDeleteHost() {
    var _this11 = this;

    var allComponents = this.get('content.hostComponents');
    var addDeleteComponentsMap = this.get('addDeleteComponentsMap');
    var deleteRequests = [];
    var deleteError = null;
    var dfd = $.Deferred();

    if (allComponents.get('length') > 0) {
      allComponents.forEach(function (component) {
        deleteRequests.push(this._doDeleteHostComponent(component.get('componentName')));
      }, this);
      $.when(deleteRequests).done(function () {
        if (_this11.get('isReconfigureRequired')) {
          var reconfiguredComponents = allComponents.filter(function (component) {
            return addDeleteComponentsMap[component.get('componentName')];
          }).mapProperty('displayName').join(', ');
          _this11.applyConfigsCustomization();
          _this11.putConfigsToServer(_this11.get('groupedPropertiesToChange'), reconfiguredComponents);
          _this11.clearConfigsChanges();
        }
        _this11.deleteHostCall();
      }).fail(function () {
        deleteError = _this11.get('_deletedHostComponentError');
        deleteError.xhr.responseText = "{\"message\": \"" + deleteError.xhr.statusText + "\"}";
        App.ajax.defaultErrorHandler(deleteError.xhr, deleteError.url, deleteError.type, deleteError.xhr.status);
      }).always(dfd.resolve);
    } else {
      dfd.resolve();
    }
    return dfd.promise();
  },

  deleteHostCall: function deleteHostCall() {
    return App.ajax.send({
      name: 'common.delete.host',
      sender: this,
      data: {
        hostName: this.get('content.hostName')
      },
      success: 'deleteHostCallSuccessCallback',
      error: 'deleteHostCallErrorCallback',
      showLoadingPopup: true
    });
  },

  deleteHostCallSuccessCallback: function deleteHostCallSuccessCallback(data, rq, requestBody) {
    App.router.get('updateController').updateHost(function () {
      App.router.transitionTo('hosts.index');
    });
    if (!!(requestBody && requestBody.hostName)) {
      var remainingHosts = App.db.getSelectedHosts().removeObject(requestBody.hostName);
      App.db.setSelectedHosts(remainingHosts);
      App.hostsMapper.deleteRecord(App.Host.find().findProperty('hostName', requestBody.hostName));
    }
    App.router.get('clusterController').getAllHostNames();
  },
  deleteHostCallErrorCallback: function deleteHostCallErrorCallback(xhr, textStatus, errorThrown, opt) {
    xhr.responseText = "{\"message\": \"" + xhr.statusText + "\"}";
    App.ajax.defaultErrorHandler(xhr, opt.url, 'DELETE', xhr.status);
  },

  /**
   * Send command to server to restart all host components with stale configs
   * @method restartAllStaleConfigComponents
   */
  restartAllStaleConfigComponents: function restartAllStaleConfigComponents() {
    var self = this;
    var staleComponents = self.get('content.componentsWithStaleConfigs');
    if (staleComponents.someProperty('componentName', 'NAMENODE') && this.get('content.hostComponents').filterProperty('componentName', 'NAMENODE').someProperty('workStatus', App.HostComponentStatus.started)) {
      this.checkNnLastCheckpointTime(function () {
        App.showConfirmationPopup(function () {
          batchUtils.restartHostComponents(staleComponents, Em.I18n.t('rollingrestart.context.allWithStaleConfigsOnSelectedHost').format(self.get('content.hostName')), "HOST");
        });
      });
    } else {
      return App.showConfirmationPopup(function () {
        batchUtils.restartHostComponents(staleComponents, Em.I18n.t('rollingrestart.context.allWithStaleConfigsOnSelectedHost').format(self.get('content.hostName')), "HOST");
      });
    }
  },

  /**
   * open Reassign Master Wizard with selected component
   * @param {object} event
   * @method moveComponent
   */
  moveComponent: function moveComponent(event) {
    var component = event.context;
    if ($(event.target).closest('li').hasClass('disabled')) {
      return;
    }
    return App.showConfirmationPopup(function () {
      var reassignMasterController = App.router.get('reassignMasterController');
      reassignMasterController.saveComponentToReassign(component);
      reassignMasterController.setCurrentStep('1');
      App.router.transitionTo('reassign');
    }, Em.I18n.t('question.sure.move').format(component.get('displayName')));
  },

  /**
   * Restart clients host components to apply config changes
   * @param {object} event
   * @method refreshConfigs
   */
  refreshConfigs: function refreshConfigs(event) {
    var self = this;
    var component = event.context;
    if (!Em.isNone(component)) {
      return App.showConfirmationPopup(function () {
        var message = Em.I18n.t('rollingrestart.context.ClientOnSelectedHost').format(component.get('displayName'), self.get('content.hostName'));
        batchUtils.restartHostComponents([component], message, "HOST");
      }, Em.I18n.t('question.sure.refresh').format(component.get('displayName'), self.get('content.hostName')));
    }
  },

  toggleMaintenanceMode: function toggleMaintenanceMode(event) {
    var state,
        message,
        self = this;
    if (event.context.get('isImpliedState')) return null;
    state = event.context.get('passiveState') === "ON" ? "OFF" : "ON";
    message = Em.I18n.t('passiveState.turn' + state.toCapital() + 'For').format(event.context.get('displayName'));
    return App.showConfirmationPopup(function () {
      self.updateComponentPassiveState(event.context, state, message);
    }, Em.I18n.t('question.sure.maintenance').format(state.toLowerCase(), event.context.get('displayName')));
  },

  downloadClientConfigs: function downloadClientConfigs(event) {
    this.downloadClientConfigsCall({
      hostName: event.context.get('hostName'),
      componentName: event.context.get('componentName'),
      resourceType: this.resourceTypeEnum.HOST_COMPONENT
    });
  },

  /**
   *  This controller action is called from the template when user clicks to download configs for "All Clients On Host"
   */
  downloadAllClientConfigs: function downloadAllClientConfigs() {
    var self = this;
    this.downloadClientConfigsCall({
      hostName: self.get('content.hostName'),
      resourceType: this.resourceTypeEnum.HOST
    });
  },

  installClients: function installClients(components) {
    var clientsToInstall = [],
        clientsToAdd = [],
        missedComponents = [],
        dependentComponents = [],
        self = this;
    components.forEach(function (component) {
      if (['INIT', 'INSTALL_FAILED'].contains(component.get('workStatus'))) {
        clientsToInstall.push(component);
      } else if (typeof component.get('workStatus') == 'undefined') {
        clientsToAdd.push(component);
      }
    });
    clientsToAdd.forEach(function (component, index, array) {
      var dependencies = this.checkComponentDependencies(component.get('componentName'), {
        scope: 'host',
        installedComponents: this.get('content.hostComponents').mapProperty('componentName')
      }).reject(function (componentName) {
        return array.mapProperty('componentName').contains(componentName);
      });
      if (dependencies.length) {
        missedComponents.pushObjects(dependencies);
        dependentComponents.push(component.get('displayName'));
      }
    }, this);
    missedComponents = missedComponents.uniq();
    if (missedComponents.length) {
      var popupMessage = Em.I18n.t('host.host.addComponent.popup.clients.dependedComponents.body').format(stringUtils.getFormattedStringFromArray(dependentComponents), stringUtils.getFormattedStringFromArray(missedComponents.map(function (componentName) {
        return App.StackServiceComponent.find(componentName).get('displayName');
      })));
      App.showAlertPopup(Em.I18n.t('host.host.addComponent.popup.dependedComponents.header'), popupMessage);
    } else {
      App.get('router.mainAdminKerberosController').getSecurityType(function () {
        App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
          var sendInstallCommand = function sendInstallCommand() {
            if (clientsToInstall.length) {
              self.sendComponentCommand(clientsToInstall, Em.I18n.t('host.host.details.installClients'), 'INSTALLED');
            }
          };
          if (clientsToAdd.length) {
            var message = stringUtils.getFormattedStringFromArray(clientsToAdd.mapProperty('displayName')),
                componentObject = Em.Object.create({
              displayName: message
            });
            self.showAddComponentPopup(componentObject, self.get('content.hostName'), function () {
              sendInstallCommand();
              clientsToAdd.forEach(function (component) {
                self.installHostComponentCall(self.get('content.hostName'), component);
              });
            });
          } else {
            sendInstallCommand();
          }
        });
      }.bind(this));
    }
  },

  /**
   * Check if all required components are installed on host.
   * Available options:
   *  scope: 'host' - dependency level `host`,`cluster` or `*`.
   *  hostName: 'example.com' - host name to search installed components
   *  installedComponents: ['A', 'B'] - names of installed components
   *
   * By default scope level is `*`
   * For host level dependency you should specify at least `hostName` or `installedComponents` attribute.
   *
   * @param {String} componentName
   * @param {Object} opt - options. Allowed options are `hostName`, `installedComponents`, `scope`.
   * @return {Array} - names of missed components
   */
  checkComponentDependencies: function checkComponentDependencies(componentName, opt) {
    opt = opt || {};
    opt.scope = opt.scope || '*';
    var installedComponents;
    switch (opt.scope) {
      case 'host':
        Em.assert("You should pass at least `hostName` or `installedComponents` to options.", opt.hostName || opt.installedComponents);
        installedComponents = opt.installedComponents || App.HostComponent.find().filterProperty('hostName', opt.hostName).mapProperty('componentName').uniq();
        break;
      default:
        // @todo: use more appropriate value regarding installed components
        installedComponents = opt.installedComponents || App.HostComponent.find().mapProperty('componentName').uniq();
        break;
    }
    var component = App.StackServiceComponent.find(componentName);
    return component.missingDependencies(installedComponents, opt).map(function (componentDependency) {
      return componentDependency.chooseCompatible();
    });
  },

  /**
   * On click handler for custom command from items menu
   * @param context
   */
  executeCustomCommand: function executeCustomCommand(event) {
    var controller = this;
    var context = event.context;
    return App.showConfirmationPopup(function () {
      App.ajax.send({
        name: 'service.item.executeCustomCommand',
        sender: controller,
        data: {
          command: context.command,
          context: context.context || Em.I18n.t('services.service.actions.run.executeCustomCommand.context').format(context.command),
          hosts: context.hosts,
          serviceName: context.service,
          componentName: context.component
        },
        success: 'executeCustomCommandSuccessCallback',
        error: 'executeCustomCommandErrorCallback'
      });
    });
  },

  executeCustomCommandSuccessCallback: function executeCustomCommandSuccessCallback(data, ajaxOptions, params) {
    if (data.Requests.id) {
      App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
        if (initValue) {
          App.router.get('backgroundOperationsController').showPopup();
        }
      });
    }
  },
  executeCustomCommandErrorCallback: function executeCustomCommandErrorCallback(data) {
    var error = Em.I18n.t('services.service.actions.run.executeCustomCommand.error');
    if (data && data.responseText) {
      try {
        var json = $.parseJSON(data.responseText);
        error += json.message;
      } catch (err) {}
    }
    App.showAlertPopup(Em.I18n.t('services.service.actions.run.executeCustomCommand.error'), error);
  },

  /**
   * Call callback after loading service metrics
   * @param callback
   */
  isServiceMetricsLoaded: function isServiceMetricsLoaded(callback) {
    App.router.get('mainController').isLoading.call(App.router.get('clusterController'), 'isServiceContentFullyLoaded').done(callback);
  },

  setConfigsChangesForDisplay: function setConfigsChangesForDisplay() {
    if (App.get('router.clusterController.isConfigsPropertiesLoaded')) {
      this.get('allPropertiesToChange').forEach(function (property) {
        var stackProperty = App.configsCollection.getConfigByName(property.propertyName, property.propertyFileName);
        if (stackProperty && (!stackProperty.isEditable || !stackProperty.isReconfigurable)) {
          this.get('requiredPropertiesToChange').pushObject(property);
        } else {
          Em.set(property, 'saveRecommended', true);
          this.get('recommendedPropertiesToChange').pushObject(property);
        }
      }, this);
      this.set('isConfigsLoadingInProgress', false);
      this.removeObserver('App.router.clusterController.isConfigsPropertiesLoaded', this, 'setConfigsChangesForDisplay');
    }
  },

  setConfigsChanges: function setConfigsChanges(groups) {
    this.get('groupedPropertiesToChange').pushObjects(groups);
    if (this.get('allPropertiesToChange.length')) {
      if (App.get('router.clusterController.isConfigsPropertiesLoaded')) {
        this.setConfigsChangesForDisplay();
      } else {
        this.addObserver('App.router.clusterController.isConfigsPropertiesLoaded', this, 'setConfigsChangesForDisplay');
      }
    } else {
      this.set('isConfigsLoadingInProgress', false);
    }
  },

  recoverHost: function recoverHost() {
    var components = this.get('content.hostComponents');
    var hostName = this.get('content.hostName');
    var self = this;
    var batches = [{
      "order_id": 1,
      "type": "PUT",
      "uri": "/clusters/" + App.get('clusterName') + "/hosts/" + hostName + "/host_components",
      "RequestBodyInfo": {
        "RequestInfo": {
          context: Em.I18n.t('hosts.host.recover.initAllComponents.context'),
          operation_level: {
            level: "HOST",
            cluster_name: App.get('clusterName'),
            host_name: hostName
          },
          query: "HostRoles/component_name.in(" + components.mapProperty('componentName').join(',') + ")"
        },
        "Body": {
          HostRoles: {
            state: "INIT"
          }
        }
      }
    }];
    batches.push({
      "order_id": 2,
      "type": "PUT",
      "uri": "/clusters/" + App.get('clusterName') + "/hosts/" + hostName + "/host_components",
      "RequestBodyInfo": {
        "RequestInfo": {
          context: Em.I18n.t('hosts.host.recover.installAllComponents.context'),
          operation_level: {
            level: "HOST",
            cluster_name: App.get('clusterName'),
            host_name: hostName
          },
          query: "HostRoles/component_name.in(" + components.mapProperty('componentName').join(',') + ")"
        },
        "Body": {
          HostRoles: {
            state: "INSTALLED"
          }
        }
      }
    });

    if (App.get('isKerberosEnabled')) {
      batches.push({
        "order_id": 3,
        "type": "PUT",
        "uri": "/clusters/" + App.get('clusterName'),
        "RequestBodyInfo": {
          "RequestInfo": {
            context: Em.I18n.t('hosts.host.recover.regenerateKeytabs.context'),
            query: "regenerate_keytabs=all&regenerate_hosts=" + hostName + "&config_update_policy=none"
          },
          "Body": {
            Clusters: {
              security_type: "KERBEROS"
            }
          }
        }
      });
    }
    App.get('router.mainAdminKerberosController').getSecurityType(function () {
      App.get('router.mainAdminKerberosController').getKDCSessionState(function () {
        self._doRecoverHost(batches);
      });
    });
  },

  _doRecoverHost: function _doRecoverHost(batches) {
    App.ajax.send({
      name: 'common.batch.request_schedules',
      sender: this,
      data: {
        intervalTimeSeconds: 1,
        tolerateSize: 0,
        batches: batches
      },
      success: 'recoverHostSuccessCallback',
      showLoadingPopup: true
    });
  },

  recoverHostSuccessCallback: function recoverHostSuccessCallback(data) {
    if (data && (data.Requests || data.resources[0].RequestSchedule)) {
      this.showBackgroundOperationsPopup();
      return true;
    } else {
      return false;
    }
  },

  recoverHostDisabled: function () {

    var isDisabled = false;
    var allowedStates = [App.HostComponentStatus.stopped, App.HostComponentStatus.install_failed, App.HostComponentStatus.init];
    this.get('content.hostComponents').forEach(function (component) {
      isDisabled = isDisabled ? true : !allowedStates.contains(component.get('workStatus'));
    });
    return isDisabled;
  }.property('content.hostComponents.@each.workStatus'),

  confirmRecoverHost: function confirmRecoverHost() {
    var self = this;
    var componentsNotStopped = [];
    var allowedStates = [App.HostComponentStatus.stopped, App.HostComponentStatus.install_failed, App.HostComponentStatus.init];
    this.get('content.hostComponents').forEach(function (component) {
      if (!allowedStates.contains(component.get('workStatus'))) {
        componentsNotStopped.push(component.get('componentName'));
      }
    });
    if (componentsNotStopped.length) {
      App.ModalPopup.show({
        header: Em.I18n.t('hosts.recover.error.popup.title'),
        recoverErrorPopupBody: Em.I18n.t('hosts.recover.error.popup.body').format(componentsNotStopped.toString()),
        componentsStr: componentsNotStopped.toString(),
        bodyClass: Em.View.extend({
          templateName: require('templates/main/host/details/recoverHostErrorPopup')
        }),
        secondary: false
      });
    } else {
      App.ModalPopup.show({
        header: Em.I18n.t('hosts.recover.popup.title'),
        bodyClass: Em.View.extend({
          templateName: require('templates/main/host/details/recoverHostPopup')
        }),
        primary: Em.I18n.t('yes'),
        secondary: Em.I18n.t('no'),
        onPrimary: function onPrimary() {
          self.recoverHost();
          this.hide();
        }
      });
    }
  },

  regenerateKeytabFileOperations: function regenerateKeytabFileOperations() {
    var self = this;
    var hostName = this.content.get('hostName');
    var clusterName = App.get('clusterName');
    return App.showConfirmationPopup(function () {
      return App.ajax.send({
        name: "admin.kerberos_security.regenerate_keytabs.host",
        sender: self,
        data: {
          clusterName: clusterName,
          hostName: hostName
        },
        success: 'regenerateKeytabFileOperationsRequestSuccess',
        error: 'regenerateKeytabFileOperationsRequestError'
      });
    }, Em.I18n.t('question.sure.regenerateKeytab.host').format(hostName));
  },

  regenerateKeytabFileOperationsRequestSuccess: function regenerateKeytabFileOperationsRequestSuccess() {
    App.router.get('backgroundOperationsController').showPopup();
  },

  regenerateKeytabFileOperationsRequestError: function regenerateKeytabFileOperationsRequestError() {
    App.showAlertPopup(Em.I18n.t('common.error'), Em.I18n.t('alerts.notifications.regenerateKeytab.host.error').format(this.content.get('hostName')));
  },

  /**
   * Returns URL parameters for configs request by certain tags
   * @param {object} data - object with desired configs tags received from API
   * @param {string[]} configTypes - list of config types
   * @returns {string}
   */
  getUrlParamsForConfigsRequest: function getUrlParamsForConfigsRequest(data, configTypes) {
    return configTypes.map(function (type) {
      var tag = Em.get(data, 'Clusters.desired_configs.' + type + '.tag');
      return tag ? '(type=' + type + '&tag=' + tag + ')' : null;
    }).compact().join('|');
  }

});

});

require.register("controllers/main/host/host_alerts_controller", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');

App.MainHostAlertsController = Em.ArrayController.extend({
  name: 'mainHostAlertsController',

  selectedHost: Em.computed.alias('App.router.mainHostDetailsController.content'),

  /**
   * List of all <code>App.AlertInstance</code> by Host
   * @type {App.AlertInstance[]}
   */
  content: function () {
    return App.AlertInstanceLocal.find().toArray().filterProperty('host', this.get('selectedHost'));
  }.property('App.router.mainAlertInstancesController.isLoaded', 'selectedHost'),

  /**
   * Open details page of the selected alertDefinition
   * @param {object} event
   * @method routeToAlertDefinition
   */
  routeToAlertDefinition: function routeToAlertDefinition(event) {
    var alertDefinition = App.AlertDefinition.find(event.context);
    App.router.transitionTo('main.alerts.alertDetails', alertDefinition);
  }
});

});

require.register("controllers/main/service", function(exports, require, module) {
'use strict';

/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

var App = require('app');
var misc = require('utils/misc');

App.MainServiceController = Em.ArrayController.extend(App.SupportClientConfigsDownload, {

  name: 'mainServiceController',

  /**
   * @type {Ember.Object[]}
   */
  content: function () {
    if (!App.router.get('clusterController.isLoaded')) {
      return [];
    }
    return misc.sortByOrder(App.StackService.find().mapProperty('serviceName'), App.Service.find().toArray());
  }.property('App.router.clusterController.isLoaded').volatile(),

  /**
   * Current cluster
   * @type {Ember.Object}
   */
  cluster: function () {
    if (!App.router.get('clusterController.isClusterDataLoaded')) {
      return null;
    }
    return App.Cluster.find().objectAt(0);
  }.property('App.router.clusterController.isClusterDataLoaded'),

  /**
   * Check if all services are installed
   * true - all installed, false - not all
   * @type {bool}
   */
  isAllServicesInstalled: function () {
    if (!this.get('content')) return false;
    var notAvailableServices = App.ServiceSimple.find().filterProperty('doNotShowAndInstall').mapProperty('name');
    var availableServices = App.ServiceSimple.find().filterProperty('doNotShowAndInstall', false);
    var installedServices = this.get('content').filter(function (service) {
      return !notAvailableServices.contains(service.get('serviceName'));
    });
    return installedServices.length == availableServices.length;
  }.property('content.@each', 'content.length'),

  /**
   * Should "Start All"-button be disabled
   * @type {bool}
   */
  isStartAllDisabled: function () {
    if (this.get('isStartStopAllClicked') == true) {
      return true;
    }
    var stoppedServices = this.get('content').filter(function (_service) {
      return _service.get('healthStatus') === 'red' && !App.get('services.clientOnly').contains(_service.get('serviceName'));
    });
    return stoppedServices.length === 0; // all green status
  }.property('isStartStopAllClicked', 'content.@each.healthStatus'),

  /**
   * Should "Stop All"-button be disabled
   * @type {bool}
   */
  isStopAllDisabled: function () {
    if (this.get('isStartStopAllClicked') == true) {
      return true;
    }
    return !this.get('content').someProperty('healthStatus', 'green');
  }.property('isStartStopAllClicked', 'content.@each.healthStatus'),

  /**
   * Should "Refresh All"-button be disabled
   * @type {bool}
   */
  isRestartAllRequiredDisabled: Em.computed.everyBy('content', 'isRestartRequired', false),

  /**
   * @type {bool}
   */
  isStartStopAllClicked: Em.computed.notEqual('App.router.backgroundOperationsController.runningOperationsCount', 0),

  /**
   * Callback for <code>start all service</code> button
   * @return {App.ModalPopup|null}
   * @method startAllService
   */
  startAllService: function startAllService(event) {
    return this.startStopAllService(event, 'STARTED');
  },

  /**
   * Callback for <code>stop all service</code> button
   * @return {App.ModalPopup|null}
   * @method stopAllService
   */
  stopAllService: function stopAllService(event) {
    return this.startStopAllService(event, 'INSTALLED');
  },

  /**
   * Common method for "start-all", "stop-all" calls
   * @param {object} event
   * @param {string} state 'STARTED|INSTALLED'
   * @returns {App.ModalPopup|null}
   * @method startStopAllService
   */
  startStopAllService: function startStopAllService(event, state) {
    if ($(event.target).hasClass('disabled') || $(event.target.parentElement).hasClass('disabled')) {
      return null;
    }
    var self = this;
    var bodyMessage = Em.Object.create({
      confirmMsg: state == 'INSTALLED' ? Em.I18n.t('services.service.stopAll.confirmMsg') : Em.I18n.t('services.service.startAll.confirmMsg'),
      confirmButton: state == 'INSTALLED' ? Em.I18n.t('services.service.stop.confirmButton') : Em.I18n.t('services.service.start.confirmButton')
    });

    if (state == 'INSTALLED' && App.Service.find().filterProperty('serviceName', 'HDFS').someProperty('workStatus', App.HostComponentStatus.started)) {
      App.router.get('mainServiceItemController').checkNnLastCheckpointTime(function () {
        return App.showConfirmationFeedBackPopup(function (query) {
          self.allServicesCall(state, query);
        }, bodyMessage);
      });
    } else {
      return App.showConfirmationFeedBackPopup(function (query) {
        self.allServicesCall(state, query);
      }, bodyMessage);
    }
  },

  /**
   * Download client configs for all services
   */
  downloadAllClientConfigs: function downloadAllClientConfigs() {
    this.downloadClientConfigsCall({ resourceType: this.resourceTypeEnum.CLUSTER });
  },

  /**
   * Do request to server for "start|stop" all services
   * @param {string} state "STARTED|INSTALLED"
   * @param {object} query
   * @method allServicesCall
   * @return {$.ajax}
   */
  allServicesCall: function allServicesCall(state, query) {
    var context = state == 'INSTALLED' ? App.BackgroundOperationsController.CommandContexts.STOP_ALL_SERVICES : App.BackgroundOperationsController.CommandContexts.START_ALL_SERVICES;
    return App.ajax.send({
      name: 'common.services.update',
      sender: this,
      data: {
        context: context,
        ServiceInfo: {
          state: state
        },
        query: query
      },
      success: 'allServicesCallSuccessCallback',
      error: 'allServicesCallErrorCallback',
      showLoadingPopup: true
    });
  },

  /**
   * Restart all services - restarts by sending one RESTART command
   */
  restartAllServices: function restartAllServices() {
    App.ajax.send({
      name: 'restart.allServices',
      sender: this,
      showLoadingPopup: true
    });
  },

  /**
   * Restart all services - stops all services, then starts them back
   */
  stopAndStartAllServices: function stopAndStartAllServices() {
    this.silentStopAllServices();
  },

  /**
   * Silent stop all services - without user confirmation
   * @returns {$.ajax}
   */
  silentStopAllServices: function silentStopAllServices() {
    return App.ajax.send({
      name: 'common.services.update',
      sender: this,
      data: {
        context: App.BackgroundOperationsController.CommandContexts.STOP_ALL_SERVICES,
        ServiceInfo: {
          state: 'INSTALLED'
        }
      },
      success: 'silentStopSuccess',
      showLoadingPopup: true
    });
  },

  isStopAllServicesFailed: function isStopAllServicesFailed() {
    var workStatuses = App.Service.find().mapProperty('workStatus');
    for (var i = 0; i < workStatuses.length; i++) {
      if (workStatuses[i] !== 'INSTALLED' && workStatuses[i] !== 'STOPPING') {
        return true;
      }
    }
    return false;
  },

  /**
   * Success callback for silent stop
   */
  silentStopSuccess: function silentStopSuccess() {
    var self = this;

    App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
      if (initValue) {
        App.router.get('backgroundOperationsController').showPopup();
      }

      Em.run.later(function () {
        self.set('shouldStart', true);
      }, App.bgOperationsUpdateInterval);
    });
  },

  /**
   * Silent start all services - without user confirmation
   */
  silentStartAllServices: function () {
    if (!App.router.get('backgroundOperationsController').get('runningOperationsCount') && this.get('shouldStart') && !this.isStopAllServicesFailed()) {
      this.set('shouldStart', false);
      return App.ajax.send({
        name: 'common.services.update',
        sender: this,
        data: {
          context: App.BackgroundOperationsController.CommandContexts.START_ALL_SERVICES,
          ServiceInfo: {
            state: 'STARTED'
          }
        },
        success: 'silentCallSuccessCallback',
        showLoadingPopup: true
      });
    }
  }.observes('shouldStart', 'controllers.backgroundOperationsController.runningOperationsCount'),

  /**
   * Success callback for silent start
   */
  silentCallSuccessCallback: function silentCallSuccessCallback() {
    // load data (if we need to show this background operations popup) from persist
    App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
      if (initValue) {
        App.router.get('backgroundOperationsController').showPopup();
      }
    });
  },

  /**
   * Success-callback for all-services request
   * @param {object} data
   * @param {object} xhr
   * @param {object} params
   * @method allServicesCallSuccessCallback
   */
  allServicesCallSuccessCallback: function allServicesCallSuccessCallback(data, xhr, params) {
    params.query.set('status', 'SUCCESS');

    // load data (if we need to show this background operations popup) from persist
    App.router.get('userSettingsController').dataLoading('show_bg').done(function (initValue) {
      if (initValue) {
        App.router.get('backgroundOperationsController').showPopup();
      }
    });
  },

  /**
   * Error-callback for all-services request
   * @param {object} request
   * @param {object} ajaxOptions
   * @param {string} error
   * @param {object} opt
   * @param {object} params
   * @method allServicesCallErrorCallback
   */
  allServicesCallErrorCallback: function allServicesCallErrorCallback(request, ajaxOptions, error, opt, params) {
    params.query.set('status', 'FAIL');
    App.ajax.defaultErrorHandler(request, opt.url, opt.type, request.status);
  },

  /**
   * "Add-service"-click handler
   * @method gotoAddService
   */
  gotoAddService: function gotoAddService() {
    if (this.get('isAllServicesInstalled')) {
      return;
    }
    App.router.get('addServiceController').setDBProperty('onClosePath', 'main.services.index');
    App.router.transitionTo('main.serviceAdd');
  },

  /**
   * Show confirmation popup and send request to restart all host components with stale_configs=true
   */
  restartAllRequired: function restartAllRequired() {
    var self = this;
    if (!this.get('isRestartAllRequiredDisabled')) {
      return App.showConfirmationPopup(function () {
        self.restartHostComponents();
      }, Em.I18n.t('service