/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.core.mgmt.internal;

import com.google.common.annotations.Beta;
import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.brooklyn.api.entity.Entity;
import org.apache.brooklyn.api.location.Location;
import org.apache.brooklyn.api.mgmt.HasTaskChildren;
import org.apache.brooklyn.api.mgmt.Task;
import org.apache.brooklyn.config.ConfigKey;
import org.apache.brooklyn.core.config.ConfigKeys;
import org.apache.brooklyn.core.entity.Entities;
import org.apache.brooklyn.core.internal.BrooklynProperties;
import org.apache.brooklyn.core.internal.storage.BrooklynStorage;
import org.apache.brooklyn.core.mgmt.BrooklynTaskTags;
import org.apache.brooklyn.util.collections.MutableList;
import org.apache.brooklyn.util.collections.MutableMap;
import org.apache.brooklyn.util.collections.MutableSet;
import org.apache.brooklyn.util.core.task.BasicExecutionManager;
import org.apache.brooklyn.util.core.task.ExecutionListener;
import org.apache.brooklyn.util.core.task.TaskTags;
import org.apache.brooklyn.util.core.task.Tasks;
import org.apache.brooklyn.util.exceptions.Exceptions;
import org.apache.brooklyn.util.guava.Maybe;
import org.apache.brooklyn.util.javalang.MemoryUsageTracker;
import org.apache.brooklyn.util.text.Strings;
import org.apache.brooklyn.util.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BrooklynGarbageCollector {
    private static final Logger LOG = LoggerFactory.getLogger(BrooklynGarbageCollector.class);
    public static final ConfigKey<Duration> GC_PERIOD = ConfigKeys.newDurationConfigKey("brooklyn.gc.period", "the period for checking if any tasks need to be deleted", Duration.minutes((Number)1));
    public static final ConfigKey<Boolean> DO_SYSTEM_GC = ConfigKeys.newBooleanConfigKey("brooklyn.gc.doSystemGc", "whether to periodically call System.gc()", false);
    public static final ConfigKey<Double> FORCE_CLEAR_SOFT_REFERENCES_ON_MEMORY_USAGE_LEVEL = ConfigKeys.newDoubleConfigKey("brooklyn.gc.clearSoftReferencesOnMemoryUsageLevel", "force clearance of soft references (by generating a deliberate OOME) if memory usage gets higher than this percentage of available memory; Brooklyn will use up to the max, or this percentage, with soft references,so if using any high-memory-usage alerts they should be pegged quite a bithigher than this threshhold (default >1 means never)", 2.0);
    public static final ConfigKey<Boolean> TRACK_SOFT_MAYBE_USAGE = ConfigKeys.newBooleanConfigKey("brooklyn.gc.trackSoftMaybeUsage", "whether to track each maybe soft-reference and report usage", true);
    @Beta
    public static final ConfigKey<Boolean> CHECK_SUBTASK_SUBMITTERS = ConfigKeys.newBooleanConfigKey("brooklyn.gc.checkSubtaskSubmitters", "whether for subtasks to check the submitters", true);
    public static final ConfigKey<Integer> MAX_TASKS_PER_TAG = ConfigKeys.newIntegerConfigKey("brooklyn.gc.maxTasksPerTag", "the maximum number of tasks to be kept for a given tag within an execution context (e.g. entity); some broad-brush tags are excluded, and if an entity has multiple tags all tag counts must be full", 50);
    public static final ConfigKey<Integer> MAX_TASKS_PER_ENTITY = ConfigKeys.newIntegerConfigKey("brooklyn.gc.maxTasksPerEntity", "the maximum number of tasks to be kept for a given entity", 1000);
    public static final ConfigKey<Integer> MAX_TASKS_GLOBAL = ConfigKeys.newIntegerConfigKey("brooklyn.gc.maxTasksGlobal", "the maximum number of tasks to be kept across the entire system", 100000);
    public static final ConfigKey<Duration> MAX_TASK_AGE = ConfigKeys.newDurationConfigKey("brooklyn.gc.maxTaskAge", "the duration after which a completed task will be automatically deleted", Duration.days((Number)30));
    protected static final Comparator<Task<?>> TASKS_OLDEST_FIRST_COMPARATOR = new Comparator<Task<?>>(){

        @Override
        public int compare(Task<?> t1, Task<?> t2) {
            long end2;
            long end1 = t1.getEndTimeUtc();
            return end1 < (end2 = t2.getEndTimeUtc()) ? -1 : (end1 == end2 ? 0 : 1);
        }
    };
    private final BasicExecutionManager executionManager;
    private final BrooklynStorage storage;
    private final BrooklynProperties brooklynProperties;
    private final ScheduledExecutorService executor;
    private ScheduledFuture<?> activeCollector;
    private Map<Entity, Task<?>> unmanagedEntitiesNeedingGc = new LinkedHashMap();
    private Duration gcPeriod;
    private volatile boolean running = true;

    public BrooklynGarbageCollector(BrooklynProperties brooklynProperties, BasicExecutionManager executionManager, BrooklynStorage storage) {
        this.executionManager = executionManager;
        this.storage = storage;
        this.brooklynProperties = brooklynProperties;
        if (brooklynProperties.getConfig(TRACK_SOFT_MAYBE_USAGE).booleanValue()) {
            Maybe.SoftlyPresent.getUsageTracker().enable();
        }
        this.executor = Executors.newSingleThreadScheduledExecutor(new ThreadFactory(){

            @Override
            public Thread newThread(Runnable r) {
                return new Thread(r, "brooklyn-gc");
            }
        });
        executionManager.addListener(new ExecutionListener(){

            @Override
            public void onTaskDone(Task<?> task) {
                BrooklynGarbageCollector.this.onTaskDone(task);
            }
        });
        this.scheduleCollector(true);
    }

    protected synchronized void scheduleCollector(boolean canInterruptCurrent) {
        if (this.activeCollector != null) {
            this.activeCollector.cancel(canInterruptCurrent);
        }
        this.gcPeriod = this.brooklynProperties.getConfig(GC_PERIOD);
        if (this.gcPeriod != null) {
            this.activeCollector = this.executor.scheduleWithFixedDelay(new Runnable(){

                @Override
                public void run() {
                    BrooklynGarbageCollector.this.gcIteration();
                }
            }, this.gcPeriod.toMillisecondsRoundingUp(), this.gcPeriod.toMillisecondsRoundingUp(), TimeUnit.MILLISECONDS);
        }
    }

    public void gcIteration() {
        try {
            this.logUsage("brooklyn gc (before)");
            this.gcTasks();
            this.logUsage("brooklyn gc (after)");
            double memUsage = 1.0 - 1.0 * (double)Runtime.getRuntime().freeMemory() / (double)Runtime.getRuntime().maxMemory();
            if (memUsage > this.brooklynProperties.getConfig(FORCE_CLEAR_SOFT_REFERENCES_ON_MEMORY_USAGE_LEVEL)) {
                LOG.info("Forcing brooklyn gc including soft-reference cleansing due to memory usage: " + this.getUsageString());
                MemoryUsageTracker.forceClearSoftReferences();
                System.gc();
                System.gc();
                LOG.info("Forced cleansing brooklyn gc, usage now: " + this.getUsageString());
            } else if (this.brooklynProperties.getConfig(DO_SYSTEM_GC).booleanValue()) {
                System.gc();
                System.gc();
                this.logUsage("brooklyn gc (after system gc)");
            }
        }
        catch (Throwable t) {
            Exceptions.propagateIfFatal((Throwable)t);
            LOG.warn("Error during management-context GC: " + t, t);
        }
    }

    public void logUsage(String prefix) {
        if (LOG.isDebugEnabled()) {
            LOG.debug(prefix + " - using " + this.getUsageString());
        }
    }

    public static String makeBasicUsageString() {
        int present = (int)Math.round(100.0 * Maybe.SoftlyPresent.getUsageTracker().getPercentagePresent());
        return Strings.makeSizeString((long)(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory())) + " / " + Strings.makeSizeString((long)Runtime.getRuntime().maxMemory()) + (Runtime.getRuntime().maxMemory() > Runtime.getRuntime().totalMemory() ? " (" + Strings.makeSizeString((long)Runtime.getRuntime().totalMemory()) + " real)" : "") + " memory; " + (present >= 0 ? present + "% soft-reference maybe retention (of " + Maybe.SoftlyPresent.getUsageTracker().getTotalEntries() + "); " : "") + Thread.activeCount() + " threads";
    }

    public String getUsageString() {
        return BrooklynGarbageCollector.makeBasicUsageString() + "; tasks: " + this.executionManager.getNumActiveTasks() + " active, " + this.executionManager.getNumIncompleteTasks() + " unfinished; " + this.executionManager.getNumInMemoryTasks() + " remembered, " + this.executionManager.getTotalTasksSubmitted() + " total submitted)";
    }

    public void shutdownNow() {
        this.running = false;
        if (this.activeCollector != null) {
            this.activeCollector.cancel(true);
        }
        if (this.executor != null) {
            this.executor.shutdownNow();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void onUnmanaged(Entity entity) {
        Map<Entity, Task<?>> map = this.unmanagedEntitiesNeedingGc;
        synchronized (map) {
            this.unmanagedEntitiesNeedingGc.put(entity, Tasks.current());
        }
    }

    public void deleteTasksForEntity(Entity entity) {
        this.executionManager.deleteTag(entity);
        this.executionManager.deleteTag(BrooklynTaskTags.tagForContextEntity(entity));
        this.executionManager.deleteTag(BrooklynTaskTags.tagForCallerEntity(entity));
        this.executionManager.deleteTag(BrooklynTaskTags.tagForTargetEntity(entity));
    }

    public void onUnmanaged(Location loc) {
    }

    public void onTaskDone(Task<?> task) {
        if (this.shouldDeleteTaskImmediately(task)) {
            this.executionManager.deleteTask(task);
        }
    }

    protected boolean shouldDeleteTaskImmediately(Task<?> task) {
        if (!task.isDone(true)) {
            return false;
        }
        Set<Object> tags = BrooklynTaskTags.getTagsFast(task);
        if (tags.contains("TRANSIENT")) {
            return true;
        }
        if (tags.contains("EFFECTOR") || tags.contains("NON-TRANSIENT")) {
            return false;
        }
        if (!this.isSubmitterExpired(task)) {
            return false;
        }
        if (this.isChild(task)) {
            LOG.warn("Unexpected expiry candidacy for " + task);
            return false;
        }
        return !this.isAssociatedToActiveEntity(task);
    }

    protected synchronized int gcTasks() {
        if (!this.running) {
            return 0;
        }
        Duration newPeriod = this.brooklynProperties.getConfig(GC_PERIOD);
        if (!Objects.equal((Object)this.gcPeriod, (Object)newPeriod)) {
            this.scheduleCollector(false);
        }
        this.expireUnmanagedEntityTasks();
        this.expireAgedTasks();
        this.expireTransientTasks();
        Set<Object> taskTags = this.executionManager.getTaskTags();
        int maxTasksPerEntity = this.brooklynProperties.getConfig(MAX_TASKS_PER_ENTITY);
        int maxTasksPerTag = this.brooklynProperties.getConfig(MAX_TASKS_PER_TAG);
        MutableMap taskNonEntityTagsOverCapacity = MutableMap.of();
        MutableMap taskEntityTagsOverCapacity = MutableMap.of();
        MutableMap taskAllTagsOverCapacity = MutableMap.of();
        for (Object tag : taskTags) {
            int over;
            Set<Task<?>> tasksWithTag;
            if (BrooklynGarbageCollector.isTagIgnoredForGc(tag) || (tasksWithTag = this.executionManager.tasksWithTagLiveOrNull(tag)) == null) continue;
            AtomicInteger overA = null;
            if (tag instanceof BrooklynTaskTags.WrappedEntity) {
                over = tasksWithTag.size() - maxTasksPerEntity;
                if (over > 0) {
                    overA = new AtomicInteger(over);
                    taskEntityTagsOverCapacity.put(tag, overA);
                }
            } else {
                over = tasksWithTag.size() - maxTasksPerTag;
                if (over > 0) {
                    overA = new AtomicInteger(over);
                    taskNonEntityTagsOverCapacity.put(tag, overA);
                }
            }
            if (overA == null) continue;
            taskAllTagsOverCapacity.put(tag, overA);
        }
        int deletedCount = 0;
        deletedCount += this.expireOverCapacityTagsInCategory((Map<Object, AtomicInteger>)taskNonEntityTagsOverCapacity, (Map<Object, AtomicInteger>)taskAllTagsOverCapacity, TagCategory.NON_ENTITY_NORMAL, false);
        deletedCount += this.expireOverCapacityTagsInCategory((Map<Object, AtomicInteger>)taskEntityTagsOverCapacity, (Map<Object, AtomicInteger>)taskAllTagsOverCapacity, TagCategory.ENTITY, true);
        int deletedHere = 0;
        while ((deletedHere = this.expireHistoricTasksNowReadyForImmediateDeletion()) > 0) {
            deletedCount += deletedHere;
        }
        deletedHere = this.expireIfOverCapacityGlobally();
        deletedCount += deletedHere;
        while (deletedHere > 0) {
            deletedHere = this.expireHistoricTasksNowReadyForImmediateDeletion();
            deletedCount += deletedHere;
        }
        return deletedCount;
    }

    protected static boolean isTagIgnoredForGc(Object tag) {
        if (tag == null) {
            return true;
        }
        if (tag.equals("EFFECTOR")) {
            return true;
        }
        if (tag.equals("SUB-TASK")) {
            return true;
        }
        if (tag.equals("NON-TRANSIENT")) {
            return true;
        }
        if (tag.equals("TRANSIENT")) {
            return true;
        }
        return tag instanceof BrooklynTaskTags.WrappedStream;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void expireUnmanagedEntityTasks() {
        Iterator ei;
        Map<Entity, Task<?>> map = this.unmanagedEntitiesNeedingGc;
        synchronized (map) {
            ei = MutableSet.copyOf(this.unmanagedEntitiesNeedingGc.entrySet()).iterator();
        }
        while (ei.hasNext()) {
            Map.Entry ee = (Map.Entry)ei.next();
            if (Entities.isManaged((Entity)ee.getKey()) || ee.getValue() != null && !((Task)ee.getValue()).isDone(true)) continue;
            this.deleteTasksForEntity((Entity)ee.getKey());
            Map<Entity, Task<?>> map2 = this.unmanagedEntitiesNeedingGc;
            synchronized (map2) {
                this.unmanagedEntitiesNeedingGc.remove(ee.getKey());
            }
        }
    }

    protected void expireAgedTasks() {
        Duration maxTaskAge = this.brooklynProperties.getConfig(MAX_TASK_AGE);
        Collection<Task<?>> allTasks = this.executionManager.allTasksLive();
        MutableList tasksToDelete = MutableList.of();
        try {
            for (Task<?> task : allTasks) {
                if (!task.isDone(true) || BrooklynTaskTags.isSubTask(task) || !maxTaskAge.isShorterThan(Duration.sinceUtc((long)task.getEndTimeUtc()))) continue;
                tasksToDelete.add(task);
            }
        }
        catch (ConcurrentModificationException e) {
            LOG.debug("Got CME inspecting aged tasks, with " + tasksToDelete.size() + " found for deletion: " + e);
        }
        for (Task task : tasksToDelete) {
            this.executionManager.deleteTask(task);
        }
    }

    protected void expireTransientTasks() {
        Set<Task<?>> transientTasks = this.executionManager.getTasksWithTag("TRANSIENT");
        for (Task<?> t : transientTasks) {
            if (!t.isDone(true)) continue;
            this.executionManager.deleteTask(t);
        }
    }

    protected int expireHistoricTasksNowReadyForImmediateDeletion() {
        if (!this.brooklynProperties.getConfig(CHECK_SUBTASK_SUBMITTERS).booleanValue()) {
            return 0;
        }
        Collection<Task<?>> allTasks = this.executionManager.allTasksLive();
        MutableList tasksToDelete = MutableList.of();
        try {
            for (Task<?> task : allTasks) {
                if (!this.shouldDeleteTaskImmediately(task)) continue;
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Deleting task which really is no longer wanted: " + task + " (submitted by " + task.getSubmittedByTask() + ")");
                }
                tasksToDelete.add(task);
            }
        }
        catch (ConcurrentModificationException e) {
            LOG.debug("Got CME inspecting aged tasks, with " + tasksToDelete.size() + " found for deletion: " + e);
        }
        for (Task task : tasksToDelete) {
            this.executionManager.deleteTask(task);
        }
        return tasksToDelete.size();
    }

    private boolean isAssociatedToActiveEntity(Task<?> task) {
        Entity associatedEntity = BrooklynTaskTags.getTargetOrContextEntity(task);
        if (associatedEntity == null) {
            return false;
        }
        return Entities.isManaged(associatedEntity);
    }

    private boolean isChild(Task<?> task) {
        Task parent = task.getSubmittedByTask();
        return parent instanceof HasTaskChildren && Iterables.contains((Iterable)((HasTaskChildren)parent).getChildren(), task);
    }

    private boolean isSubmitterExpired(Task<?> task) {
        if (Strings.isBlank((CharSequence)task.getSubmittedByTaskId())) {
            return false;
        }
        Task submitter = task.getSubmittedByTask();
        return submitter == null || submitter.isDone() && this.executionManager.getTask(submitter.getId()) == null;
    }

    protected int expireOverCapacityTagsInCategory(Map<Object, AtomicInteger> taskTagsInCategoryOverCapacity, Map<Object, AtomicInteger> taskAllTagsOverCapacity, TagCategory category, boolean emptyFilterNeeded) {
        if (emptyFilterNeeded) {
            MutableList nowOkayTags = MutableList.of();
            for (Map.Entry<Object, AtomicInteger> entry : taskTagsInCategoryOverCapacity.entrySet()) {
                if (entry.getValue().get() > 0) continue;
                nowOkayTags.add(entry.getKey());
            }
            for (Iterator<Object> tag : nowOkayTags) {
                taskTagsInCategoryOverCapacity.remove(tag);
            }
        }
        if (taskTagsInCategoryOverCapacity.isEmpty()) {
            return 0;
        }
        Collection<Task<?>> tasks = this.executionManager.allTasksLive();
        MutableList tasksToConsiderDeleting = MutableList.of();
        try {
            for (Task<?> task : tasks) {
                if (!task.isDone(true)) continue;
                Set<Object> tags = TaskTags.getTagsFast(task);
                int categoryTags = 0;
                int tooFullCategoryTags = 0;
                for (Object tag : tags) {
                    if (!category.acceptsTag(tag)) continue;
                    ++categoryTags;
                    if (!taskTagsInCategoryOverCapacity.containsKey(tag)) continue;
                    ++tooFullCategoryTags;
                }
                if (tooFullCategoryTags <= 0) continue;
                if (categoryTags == tooFullCategoryTags) {
                    tasksToConsiderDeleting.add(task);
                    continue;
                }
                for (Object tag : tags) {
                    AtomicInteger over;
                    if (!category.acceptsTag(tag) || (over = taskTagsInCategoryOverCapacity.get(tag)) == null || over.decrementAndGet() > 0) continue;
                    taskTagsInCategoryOverCapacity.remove(tag);
                    if (!taskTagsInCategoryOverCapacity.isEmpty()) continue;
                    return 0;
                }
            }
        }
        catch (ConcurrentModificationException e) {
            LOG.debug("Got CME inspecting tasks, with " + tasksToConsiderDeleting.size() + " found for deletion: " + e);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("brooklyn-gc detected " + taskTagsInCategoryOverCapacity.size() + " " + (Object)((Object)category) + " tags over capacity, expiring old tasks; " + tasksToConsiderDeleting.size() + " tasks under consideration; categories are: " + taskTagsInCategoryOverCapacity);
        }
        Collections.sort(tasksToConsiderDeleting, TASKS_OLDEST_FIRST_COMPARATOR);
        int deleted = 0;
        for (Task task : tasksToConsiderDeleting) {
            boolean delete = true;
            for (Object tag : task.getTags()) {
                if (!category.acceptsTag(tag) || taskTagsInCategoryOverCapacity.get(tag) != null) continue;
                delete = false;
                break;
            }
            if (!delete) continue;
            ++deleted;
            this.executionManager.deleteTask(task);
            for (Object tag : task.getTags()) {
                AtomicInteger counter = taskAllTagsOverCapacity.get(tag);
                if (counter == null || counter.decrementAndGet() > 0) continue;
                taskTagsInCategoryOverCapacity.remove(tag);
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace("brooklyn-gc deleted " + task + ", buckets now " + taskTagsInCategoryOverCapacity);
            }
            if (!taskTagsInCategoryOverCapacity.isEmpty()) continue;
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("brooklyn-gc deleted " + deleted + " tasks in over-capacity " + (Object)((Object)category) + " tag categories; capacities now: " + taskTagsInCategoryOverCapacity);
        }
        return deleted;
    }

    protected int expireIfOverCapacityGlobally() {
        Object tasksLive = this.executionManager.allTasksLive();
        if (tasksLive.size() <= this.brooklynProperties.getConfig(MAX_TASKS_GLOBAL)) {
            return 0;
        }
        LOG.debug("brooklyn-gc detected " + tasksLive.size() + " tasks in memory, over global limit, looking at deleting some");
        try {
            tasksLive = MutableList.copyOf(tasksLive);
        }
        catch (ConcurrentModificationException e) {
            tasksLive = this.executionManager.getTasksWithAllTags((Iterable<?>)MutableList.of());
        }
        MutableList tasks = MutableList.of();
        for (Task task : tasksLive) {
            if (!task.isDone()) continue;
            tasks.add((Object)task);
        }
        int numToDelete = tasks.size() - this.brooklynProperties.getConfig(MAX_TASKS_GLOBAL);
        if (numToDelete <= 0) {
            LOG.debug("brooklyn-gc detected only " + tasks.size() + " completed tasks in memory, not over global limit, so not deleting any");
            return 0;
        }
        Collections.sort(tasks, TASKS_OLDEST_FIRST_COMPARATOR);
        int numDeleted = 0;
        while (numDeleted < numToDelete && tasks.size() > numDeleted) {
            this.executionManager.deleteTask((Task)tasks.get(numDeleted++));
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("brooklyn-gc deleted " + numDeleted + " tasks as was over global limit, now have " + this.executionManager.allTasksLive().size());
        }
        return numDeleted;
    }

    protected static enum TagCategory {
        ENTITY,
        NON_ENTITY_NORMAL;


        public boolean acceptsTag(Object tag) {
            if (BrooklynGarbageCollector.isTagIgnoredForGc(tag)) {
                return false;
            }
            if (tag instanceof BrooklynTaskTags.WrappedEntity) {
                return this == ENTITY;
            }
            return this != ENTITY;
        }
    }
}

