/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.CRC32;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogPosition;
import org.apache.cassandra.db.commitlog.CommitLogReadHandler;
import org.apache.cassandra.db.commitlog.CommitLogSegmentReader;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.db.rows.DeserializationHelper;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.UnknownTableException;
import org.apache.cassandra.io.util.DataInputBuffer;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.FileDataInput;
import org.apache.cassandra.io.util.RandomAccessReader;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogReader {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogReader.class);
    private static final int LEGACY_END_OF_SEGMENT_MARKER = 0;
    @VisibleForTesting
    public static final int ALL_MUTATIONS = -1;
    private final CRC32 checksum = new CRC32();
    private final Map<TableId, AtomicInteger> invalidMutations = new HashMap<TableId, AtomicInteger>();
    private byte[] buffer = new byte[4096];

    public Set<Map.Entry<TableId, AtomicInteger>> getInvalidMutations() {
        return this.invalidMutations.entrySet();
    }

    public void readAllFiles(CommitLogReadHandler handler, File[] files) throws IOException {
        this.readAllFiles(handler, files, CommitLogPosition.NONE);
    }

    private static boolean shouldSkip(File file) throws IOException, ConfigurationException {
        try (RandomAccessReader reader = RandomAccessReader.open(file);){
            CommitLogDescriptor.readHeader(reader, DatabaseDescriptor.getEncryptionContext());
            int end = reader.readInt();
            long filecrc = (long)reader.readInt() & 0xFFFFFFFFL;
            boolean bl = end == 0 && filecrc == 0L;
            return bl;
        }
    }

    static List<File> filterCommitLogFiles(File[] toFilter) {
        ArrayList<File> filtered = new ArrayList<File>(toFilter.length);
        for (File file : toFilter) {
            try {
                if (CommitLogReader.shouldSkip(file)) {
                    logger.info("Skipping playback of empty log: {}", (Object)file.name());
                    continue;
                }
                filtered.add(file);
            }
            catch (Exception e) {
                filtered.add(file);
            }
        }
        return filtered;
    }

    public void readAllFiles(CommitLogReadHandler handler, File[] files, CommitLogPosition minPosition) throws IOException {
        List<File> filteredLogs = CommitLogReader.filterCommitLogFiles(files);
        int i = 0;
        for (File file : filteredLogs) {
            this.readCommitLogSegment(handler, file, minPosition, -1, ++i == filteredLogs.size());
        }
    }

    public void readCommitLogSegment(CommitLogReadHandler handler, File file, boolean tolerateTruncation) throws IOException {
        this.readCommitLogSegment(handler, file, CommitLogPosition.NONE, -1, tolerateTruncation);
    }

    public void readCommitLogSegment(CommitLogReadHandler handler, File file, CommitLogPosition minPosition, boolean tolerateTruncation) throws IOException {
        this.readCommitLogSegment(handler, file, minPosition, -1, tolerateTruncation);
    }

    @VisibleForTesting
    public void readCommitLogSegment(CommitLogReadHandler handler, File file, int mutationLimit, boolean tolerateTruncation) throws IOException {
        this.readCommitLogSegment(handler, file, CommitLogPosition.NONE, mutationLimit, tolerateTruncation);
    }

    public void readCommitLogSegment(CommitLogReadHandler handler, File file, CommitLogPosition minPosition, int mutationLimit, boolean tolerateTruncation) throws IOException {
        CommitLogDescriptor desc = CommitLogDescriptor.fromFileName(file.name());
        try (RandomAccessReader reader = RandomAccessReader.open(file);){
            CommitLogSegmentReader segmentReader;
            long segmentIdFromFilename = desc.id;
            try {
                desc = CommitLogDescriptor.readHeader(reader, DatabaseDescriptor.getEncryptionContext());
            }
            catch (Exception e) {
                desc = null;
            }
            if (desc == null) {
                handler.handleUnrecoverableError(new CommitLogReadHandler.CommitLogReadException(String.format("Could not read commit log descriptor in file %s", file), CommitLogReadHandler.CommitLogReadErrorReason.UNRECOVERABLE_DESCRIPTOR_ERROR, tolerateTruncation));
                return;
            }
            if (segmentIdFromFilename != desc.id && handler.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Segment id mismatch (filename %d, descriptor %d) in file %s", segmentIdFromFilename, desc.id, file), CommitLogReadHandler.CommitLogReadErrorReason.RECOVERABLE_DESCRIPTOR_ERROR, false))) {
                return;
            }
            if (this.shouldSkipSegmentId(file, desc, minPosition)) {
                return;
            }
            try {
                segmentReader = new CommitLogSegmentReader(handler, desc, reader, tolerateTruncation);
            }
            catch (Exception e) {
                handler.handleUnrecoverableError(new CommitLogReadHandler.CommitLogReadException(String.format("Unable to create segment reader for commit log file: %s", e), CommitLogReadHandler.CommitLogReadErrorReason.UNRECOVERABLE_UNKNOWN_ERROR, tolerateTruncation));
                if (reader != null) {
                    if (var8_8 != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable) {
                            var8_8.addSuppressed(throwable);
                        }
                    } else {
                        reader.close();
                    }
                }
                return;
            }
            try {
                ReadStatusTracker statusTracker = new ReadStatusTracker(mutationLimit, tolerateTruncation);
                for (CommitLogSegmentReader.SyncSegment syncSegment : segmentReader) {
                    statusTracker.tolerateErrorsInSection = tolerateTruncation & syncSegment.toleratesErrorsInSection;
                    if (desc.id == minPosition.segmentId && syncSegment.endPosition < minPosition.position) continue;
                    statusTracker.errorContext = String.format("Next section at %d in %s", syncSegment.fileStartPosition, desc.fileName());
                    this.readSection(handler, syncSegment.input, minPosition, syncSegment.endPosition, statusTracker, desc);
                    if (statusTracker.shouldContinue()) continue;
                    break;
                }
            }
            catch (RuntimeException re) {
                if (re.getCause() instanceof IOException) {
                    throw (IOException)re.getCause();
                }
                throw re;
            }
            logger.info("Finished reading {}", (Object)file);
        }
    }

    private boolean shouldSkipSegmentId(File file, CommitLogDescriptor desc, CommitLogPosition minPosition) {
        logger.debug("Reading {} (CL version {}, messaging version {}, compression {})", new Object[]{file.path(), desc.version, desc.getMessagingVersion(), desc.compression});
        if (minPosition.segmentId > desc.id) {
            logger.trace("Skipping read of fully-flushed {}", (Object)file);
            return true;
        }
        return false;
    }

    private void readSection(CommitLogReadHandler handler, FileDataInput reader, CommitLogPosition minPosition, int end, ReadStatusTracker statusTracker, CommitLogDescriptor desc) throws IOException {
        if (desc.id == minPosition.segmentId && reader.getFilePointer() < (long)minPosition.position) {
            reader.seek(minPosition.position);
        }
        while (statusTracker.shouldContinue() && reader.getFilePointer() < (long)end && !reader.isEOF()) {
            long claimedCRC32;
            int serializedSize;
            long mutationStart = reader.getFilePointer();
            if (logger.isTraceEnabled()) {
                logger.trace("Reading mutation at {}", (Object)mutationStart);
            }
            try {
                if ((long)end - reader.getFilePointer() < 4L) {
                    logger.trace("Not enough bytes left for another mutation in this CommitLog section, continuing");
                    statusTracker.requestTermination();
                    return;
                }
                serializedSize = reader.readInt();
                if (serializedSize == 0) {
                    logger.trace("Encountered end of segment marker at {}", (Object)reader.getFilePointer());
                    statusTracker.requestTermination();
                    return;
                }
                if (serializedSize < 10) {
                    if (handler.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Invalid mutation size %d at %d in %s", serializedSize, mutationStart, statusTracker.errorContext), CommitLogReadHandler.CommitLogReadErrorReason.MUTATION_ERROR, statusTracker.tolerateErrorsInSection))) {
                        statusTracker.requestTermination();
                    }
                    return;
                }
                long claimedSizeChecksum = CommitLogFormat.calculateClaimedChecksum(reader, desc.version);
                this.checksum.reset();
                CommitLogFormat.updateChecksum(this.checksum, serializedSize, desc.version);
                if (this.checksum.getValue() != claimedSizeChecksum) {
                    if (handler.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Mutation size checksum failure at %d in %s", mutationStart, statusTracker.errorContext), CommitLogReadHandler.CommitLogReadErrorReason.MUTATION_ERROR, statusTracker.tolerateErrorsInSection))) {
                        statusTracker.requestTermination();
                    }
                    return;
                }
                if (serializedSize > this.buffer.length) {
                    this.buffer = new byte[(int)(1.2 * (double)serializedSize)];
                }
                reader.readFully(this.buffer, 0, serializedSize);
                claimedCRC32 = CommitLogFormat.calculateClaimedCRC32(reader, desc.version);
            }
            catch (EOFException eof) {
                if (handler.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Unexpected end of segment at %d in %s", mutationStart, statusTracker.errorContext), CommitLogReadHandler.CommitLogReadErrorReason.EOF, statusTracker.tolerateErrorsInSection))) {
                    statusTracker.requestTermination();
                }
                return;
            }
            this.checksum.update(this.buffer, 0, serializedSize);
            if (claimedCRC32 != this.checksum.getValue()) {
                if (!handler.shouldSkipSegmentOnError(new CommitLogReadHandler.CommitLogReadException(String.format("Mutation checksum failure at %d in %s", mutationStart, statusTracker.errorContext), CommitLogReadHandler.CommitLogReadErrorReason.MUTATION_ERROR, statusTracker.tolerateErrorsInSection))) continue;
                statusTracker.requestTermination();
                continue;
            }
            long mutationPosition = reader.getFilePointer();
            this.readMutation(handler, this.buffer, serializedSize, minPosition, (int)mutationPosition, desc);
            if (mutationPosition < (long)minPosition.position) continue;
            statusTracker.addProcessedMutation();
        }
    }

    @VisibleForTesting
    protected void readMutation(CommitLogReadHandler handler, byte[] inputBuffer, int size, CommitLogPosition minPosition, int entryLocation, CommitLogDescriptor desc) throws IOException {
        Mutation mutation;
        boolean shouldReplay = entryLocation > minPosition.position;
        try (DataInputBuffer bufIn = new DataInputBuffer(inputBuffer, 0, size);){
            mutation = Mutation.serializer.deserialize(bufIn, desc.getMessagingVersion(), DeserializationHelper.Flag.LOCAL);
            for (PartitionUpdate upd : mutation.getPartitionUpdates()) {
                upd.validate();
            }
        }
        catch (UnknownTableException ex) {
            if (ex.id == null) {
                return;
            }
            AtomicInteger i = this.invalidMutations.get(ex.id);
            if (i == null) {
                i = new AtomicInteger(1);
                this.invalidMutations.put(ex.id, i);
            } else {
                i.incrementAndGet();
            }
            return;
        }
        catch (Throwable t) {
            JVMStabilityInspector.inspectThrowable(t);
            Path p = Files.createTempFile("mutation", "dat", new FileAttribute[0]);
            try (DataOutputStream out = new DataOutputStream(Files.newOutputStream(p, new OpenOption[0]));){
                out.write(inputBuffer, 0, size);
            }
            handler.handleUnrecoverableError(new CommitLogReadHandler.CommitLogReadException(String.format("Unexpected error deserializing mutation; saved to %s.  This may be caused by replaying a mutation against a table with the same name but incompatible schema.  Exception follows: %s", p.toString(), t), CommitLogReadHandler.CommitLogReadErrorReason.MUTATION_ERROR, false));
            return;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Read mutation for {}.{}: {}", new Object[]{mutation.getKeyspaceName(), mutation.key(), "{" + StringUtils.join((Iterator)mutation.getPartitionUpdates().iterator(), (String)", ") + "}"});
        }
        if (shouldReplay) {
            handler.handleMutation(mutation, size, entryLocation, desc);
        }
    }

    private static class ReadStatusTracker {
        private int mutationsLeft;
        public String errorContext = "";
        public boolean tolerateErrorsInSection;
        private boolean error;

        public ReadStatusTracker(int mutationLimit, boolean tolerateErrorsInSection) {
            this.mutationsLeft = mutationLimit;
            this.tolerateErrorsInSection = tolerateErrorsInSection;
        }

        public void addProcessedMutation() {
            if (this.mutationsLeft == -1) {
                return;
            }
            --this.mutationsLeft;
        }

        public boolean shouldContinue() {
            return !this.error && (this.mutationsLeft != 0 || this.mutationsLeft == -1);
        }

        public void requestTermination() {
            this.error = true;
        }
    }

    private static class CommitLogFormat {
        private CommitLogFormat() {
        }

        public static long calculateClaimedChecksum(FileDataInput input, int commitLogVersion) throws IOException {
            return (long)input.readInt() & 0xFFFFFFFFL;
        }

        public static void updateChecksum(CRC32 checksum, int serializedSize, int commitLogVersion) {
            FBUtilities.updateChecksumInt(checksum, serializedSize);
        }

        public static long calculateClaimedCRC32(FileDataInput input, int commitLogVersion) throws IOException {
            return (long)input.readInt() & 0xFFFFFFFFL;
        }
    }
}

