/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.sstable.indexsummary;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.lifecycle.LifecycleTransaction;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.io.sstable.indexsummary.IndexSummaryBuilder;
import org.apache.cassandra.io.sstable.indexsummary.IndexSummarySupport;
import org.apache.cassandra.metrics.StorageMetrics;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.utils.Clock;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.TimeUUID;
import org.apache.cassandra.utils.concurrent.Refs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexSummaryRedistribution
extends CompactionInfo.Holder {
    private static final Logger logger = LoggerFactory.getLogger(IndexSummaryRedistribution.class);
    static final double UPSAMPLE_THRESHOLD = 1.5;
    static final double DOWNSAMPLE_THESHOLD = 0.75;
    private final Map<TableId, LifecycleTransaction> transactions;
    private final long nonRedistributingOffHeapSize;
    private final long memoryPoolBytes;
    private final TimeUUID compactionId;
    private volatile long remainingSpace;

    public IndexSummaryRedistribution(Map<TableId, LifecycleTransaction> transactions, long nonRedistributingOffHeapSize, long memoryPoolBytes) {
        this.transactions = transactions;
        this.nonRedistributingOffHeapSize = nonRedistributingOffHeapSize;
        this.memoryPoolBytes = memoryPoolBytes;
        this.compactionId = TimeUUID.Generator.nextTimeUUID();
    }

    private static <T extends SSTableReader> List<T> getIndexSummarySupportingAndCloseOthers(LifecycleTransaction txn) {
        ArrayList<SSTableReader> filtered = new ArrayList<SSTableReader>();
        ArrayList<SSTableReader> cancels = new ArrayList<SSTableReader>();
        for (SSTableReader sstable : txn.originals()) {
            if (sstable instanceof IndexSummarySupport) {
                filtered.add(sstable);
                continue;
            }
            cancels.add(sstable);
        }
        txn.cancel(cancels);
        return filtered;
    }

    public <T extends SSTableReader> List<T> redistributeSummaries() throws IOException {
        List newSSTables;
        long start = Clock.Global.nanoTime();
        logger.info("Redistributing index summaries");
        ArrayList<T> redistribute = new ArrayList<T>();
        for (LifecycleTransaction txn : this.transactions.values()) {
            redistribute.addAll(IndexSummaryRedistribution.getIndexSummarySupportingAndCloseOthers(txn));
        }
        long total = this.nonRedistributingOffHeapSize;
        for (SSTableReader sstable : redistribute) {
            total += ((IndexSummarySupport)((Object)sstable)).getIndexSummary().getOffHeapSize();
        }
        logger.info("Beginning redistribution of index summaries for {} sstables with memory pool size {} MiB; current spaced used is {} MiB", new Object[]{redistribute.size(), this.memoryPoolBytes / 1024L / 1024L, (double)total / 1024.0 / 1024.0});
        HashMap<SSTableReader, Double> readRates = new HashMap<SSTableReader, Double>(redistribute.size());
        double totalReadsPerSec = 0.0;
        for (SSTableReader sstable : redistribute) {
            if (this.isStopRequested()) {
                throw new CompactionInterruptedException(this.getCompactionInfo());
            }
            if (sstable.getReadMeter() == null) continue;
            Double readRate = sstable.getReadMeter().fifteenMinuteRate();
            totalReadsPerSec += readRate.doubleValue();
            readRates.put(sstable, readRate);
        }
        logger.trace("Total reads/sec across all sstables in index summary resize process: {}", (Object)totalReadsPerSec);
        ArrayList sstablesByHotness = new ArrayList(redistribute);
        Collections.sort(sstablesByHotness, new ReadRateComparator(readRates));
        long remainingBytes = this.memoryPoolBytes - this.nonRedistributingOffHeapSize;
        logger.trace("Index summaries for compacting SSTables are using {} MiB of space", (Object)((double)(this.memoryPoolBytes - remainingBytes) / 1024.0 / 1024.0));
        try (Refs refs = Refs.ref(sstablesByHotness);){
            newSSTables = this.adjustSamplingLevels(sstablesByHotness, this.transactions, totalReadsPerSec, remainingBytes);
            for (LifecycleTransaction txn : this.transactions.values()) {
                txn.finish();
            }
        }
        total = this.nonRedistributingOffHeapSize;
        for (SSTableReader sstable : newSSTables) {
            total += ((IndexSummarySupport)((Object)sstable)).getIndexSummary().getOffHeapSize();
        }
        logger.info("Completed resizing of index summaries; current approximate memory used: {} MiB, time spent: {}ms", (Object)((double)total / 1024.0 / 1024.0), (Object)TimeUnit.NANOSECONDS.toMillis(Clock.Global.nanoTime() - start));
        return newSSTables;
    }

    private <T extends SSTableReader> List<T> adjustSamplingLevels(List<T> sstables, Map<TableId, LifecycleTransaction> transactions, double totalReadsPerSec, long memoryPoolCapacity) throws IOException {
        List<ResampleEntry<Object>> toDownsample = new ArrayList<ResampleEntry<T>>(sstables.size() / 4);
        ArrayList<ResampleEntry<SSTableReader>> toUpsample = new ArrayList<ResampleEntry<SSTableReader>>(sstables.size() / 4);
        ArrayList<ResampleEntry<SSTableReader>> forceResample = new ArrayList<ResampleEntry<SSTableReader>>();
        ArrayList<ResampleEntry<SSTableReader>> forceUpsample = new ArrayList<ResampleEntry<SSTableReader>>();
        ArrayList<SSTableReader> newSSTables = new ArrayList<SSTableReader>(sstables.size());
        this.remainingSpace = memoryPoolCapacity;
        for (SSTableReader sSTableReader : sstables) {
            long spaceUsed;
            if (this.isStopRequested()) {
                throw new CompactionInterruptedException(this.getCompactionInfo());
            }
            int minIndexInterval = sSTableReader.metadata().params.minIndexInterval;
            int maxIndexInterval = sSTableReader.metadata().params.maxIndexInterval;
            double readsPerSec = sSTableReader.getReadMeter() == null ? 0.0 : sSTableReader.getReadMeter().fifteenMinuteRate();
            long idealSpace = Math.round((double)this.remainingSpace * (readsPerSec / totalReadsPerSec));
            int currentNumEntries = ((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().size();
            double avgEntrySize = (double)((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getOffHeapSize() / (double)currentNumEntries;
            long targetNumEntries = Math.max(1L, Math.round((double)idealSpace / avgEntrySize));
            int currentSamplingLevel = ((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getSamplingLevel();
            int maxSummarySize = ((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getMaxNumberOfEntries();
            if (((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getMinIndexInterval() != minIndexInterval) {
                int effectiveSamplingLevel = (int)Math.round((double)currentSamplingLevel * ((double)minIndexInterval / (double)((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getMinIndexInterval()));
                maxSummarySize = (int)Math.round((double)maxSummarySize * ((double)((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getMinIndexInterval() / (double)minIndexInterval));
                logger.trace("min_index_interval changed from {} to {}, so the current sampling level for {} is effectively now {} (was {})", new Object[]{((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getMinIndexInterval(), minIndexInterval, sSTableReader, effectiveSamplingLevel, currentSamplingLevel});
                currentSamplingLevel = effectiveSamplingLevel;
            }
            int newSamplingLevel = IndexSummaryBuilder.calculateSamplingLevel(currentSamplingLevel, currentNumEntries, targetNumEntries, minIndexInterval, maxIndexInterval);
            int numEntriesAtNewSamplingLevel = IndexSummaryBuilder.entriesAtSamplingLevel(newSamplingLevel, maxSummarySize);
            double effectiveIndexInterval = ((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getEffectiveIndexInterval();
            if (logger.isTraceEnabled()) {
                logger.trace("{} has {} reads/sec; ideal space for index summary: {} ({} entries); considering moving from level {} ({} entries, {}) to level {} ({} entries, {})", new Object[]{sSTableReader.getFilename(), readsPerSec, FBUtilities.prettyPrintMemory(idealSpace), targetNumEntries, currentSamplingLevel, currentNumEntries, FBUtilities.prettyPrintMemory((long)((double)currentNumEntries * avgEntrySize)), newSamplingLevel, numEntriesAtNewSamplingLevel, FBUtilities.prettyPrintMemory((long)((double)numEntriesAtNewSamplingLevel * avgEntrySize))});
            }
            if (effectiveIndexInterval < (double)minIndexInterval) {
                logger.trace("Forcing resample of {} because the current index interval ({}) is below min_index_interval ({})", new Object[]{sSTableReader, effectiveIndexInterval, minIndexInterval});
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                forceResample.add(new ResampleEntry<SSTableReader>(sSTableReader, spaceUsed, newSamplingLevel));
                this.remainingSpace -= spaceUsed;
            } else if (effectiveIndexInterval > (double)maxIndexInterval) {
                logger.trace("Forcing upsample of {} because the current index interval ({}) is above max_index_interval ({})", new Object[]{sSTableReader, effectiveIndexInterval, maxIndexInterval});
                newSamplingLevel = Math.max(1, 128 * minIndexInterval / maxIndexInterval);
                numEntriesAtNewSamplingLevel = IndexSummaryBuilder.entriesAtSamplingLevel(newSamplingLevel, ((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getMaxNumberOfEntries());
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                forceUpsample.add(new ResampleEntry<SSTableReader>(sSTableReader, spaceUsed, newSamplingLevel));
                this.remainingSpace = (long)((double)this.remainingSpace - avgEntrySize * (double)numEntriesAtNewSamplingLevel);
            } else if ((double)targetNumEntries >= (double)currentNumEntries * 1.5 && newSamplingLevel > currentSamplingLevel) {
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                toUpsample.add(new ResampleEntry<SSTableReader>(sSTableReader, spaceUsed, newSamplingLevel));
                this.remainingSpace = (long)((double)this.remainingSpace - avgEntrySize * (double)numEntriesAtNewSamplingLevel);
            } else if ((double)targetNumEntries < (double)currentNumEntries * 0.75 && newSamplingLevel < currentSamplingLevel) {
                spaceUsed = (long)Math.ceil(avgEntrySize * (double)numEntriesAtNewSamplingLevel);
                toDownsample.add(new ResampleEntry<SSTableReader>(sSTableReader, spaceUsed, newSamplingLevel));
                this.remainingSpace -= spaceUsed;
            } else {
                logger.trace("SSTable {} is within thresholds of ideal sampling", (Object)sSTableReader);
                this.remainingSpace -= ((IndexSummarySupport)((Object)sSTableReader)).getIndexSummary().getOffHeapSize();
                newSSTables.add(sSTableReader);
                transactions.get(sSTableReader.metadata().id).cancel(sSTableReader);
            }
            totalReadsPerSec -= readsPerSec;
        }
        if (this.remainingSpace > 0L) {
            Pair<List<T>, List<ResampleEntry<T>>> result = IndexSummaryRedistribution.distributeRemainingSpace(toDownsample, this.remainingSpace);
            toDownsample = (List)result.right;
            newSSTables.addAll((Collection)result.left);
            for (SSTableReader sstable : (List)result.left) {
                transactions.get(sstable.metadata().id).cancel(sstable);
            }
        }
        logger.info("index summaries: downsample: {}, force resample: {}, upsample: {}, force upsample: {}", new Object[]{toDownsample.size(), forceResample.size(), toUpsample.size(), forceUpsample.size()});
        toDownsample.addAll(forceResample);
        toDownsample.addAll(toUpsample);
        toDownsample.addAll(forceUpsample);
        for (ResampleEntry resampleEntry : toDownsample) {
            if (this.isStopRequested()) {
                throw new CompactionInterruptedException(this.getCompactionInfo());
            }
            Object sstable = resampleEntry.sstable;
            logger.trace("Re-sampling index summary for {} from {}/{} to {}/{} of the original number of entries", new Object[]{sstable, ((IndexSummarySupport)sstable).getIndexSummary().getSamplingLevel(), 128, resampleEntry.newSamplingLevel, 128});
            ColumnFamilyStore cfs = Keyspace.open(((SSTable)sstable).metadata().keyspace).getColumnFamilyStore(((SSTable)sstable).metadata().id);
            long oldSize = ((SSTableReader)sstable).bytesOnDisk();
            long oldSizeUncompressed = ((SSTableReader)sstable).logicalBytesOnDisk();
            Object replacement = ((IndexSummarySupport)sstable).cloneWithNewSummarySamplingLevel(cfs, resampleEntry.newSamplingLevel);
            long newSize = ((SSTableReader)replacement).bytesOnDisk();
            long newSizeUncompressed = ((SSTableReader)replacement).logicalBytesOnDisk();
            newSSTables.add((SSTableReader)replacement);
            transactions.get(((SSTable)sstable).metadata().id).update((SSTableReader)replacement, true);
            this.addHooks(cfs, transactions, oldSize, newSize, oldSizeUncompressed, newSizeUncompressed);
        }
        return newSSTables;
    }

    private void addHooks(ColumnFamilyStore cfs, Map<TableId, LifecycleTransaction> transactions, long oldSize, long newSize, long oldSizeUncompressed, long newSizeUncompressed) {
        LifecycleTransaction txn = transactions.get(cfs.metadata.id);
        txn.runOnCommit(() -> {
            StorageMetrics.load.dec(oldSize);
            StorageMetrics.uncompressedLoad.dec(oldSizeUncompressed);
            cfs.metric.liveDiskSpaceUsed.dec(oldSize);
            cfs.metric.uncompressedLiveDiskSpaceUsed.dec(oldSizeUncompressed);
            cfs.metric.totalDiskSpaceUsed.dec(oldSize);
        });
        txn.runOnAbort(() -> {
            long delta = oldSize - newSize;
            long deltaUncompressed = oldSizeUncompressed - newSizeUncompressed;
            StorageMetrics.load.dec(delta);
            StorageMetrics.uncompressedLoad.dec(deltaUncompressed);
            cfs.metric.liveDiskSpaceUsed.dec(delta);
            cfs.metric.uncompressedLiveDiskSpaceUsed.dec(deltaUncompressed);
            cfs.metric.totalDiskSpaceUsed.dec(delta);
        });
    }

    @VisibleForTesting
    static <T extends SSTableReader> Pair<List<T>, List<ResampleEntry<T>>> distributeRemainingSpace(List<ResampleEntry<T>> toDownsample, long remainingSpace) {
        int noDownsampleCutoff;
        long extraSpaceRequired;
        Collections.sort(toDownsample, Comparator.comparingDouble(o -> ((IndexSummarySupport)o.sstable).getIndexSummary().getOffHeapSize() - o.newSpaceUsed));
        ArrayList willNotDownsample = new ArrayList();
        for (noDownsampleCutoff = 0; remainingSpace > 0L && noDownsampleCutoff < toDownsample.size(); remainingSpace -= extraSpaceRequired, ++noDownsampleCutoff) {
            ResampleEntry<T> entry = toDownsample.get(noDownsampleCutoff);
            extraSpaceRequired = ((IndexSummarySupport)entry.sstable).getIndexSummary().getOffHeapSize() - entry.newSpaceUsed;
            if (extraSpaceRequired > remainingSpace) break;
            logger.trace("Using leftover space to keep {} at the current sampling level ({})", entry.sstable, (Object)((IndexSummarySupport)entry.sstable).getIndexSummary().getSamplingLevel());
            willNotDownsample.add(entry.sstable);
        }
        return Pair.create(willNotDownsample, toDownsample.subList(noDownsampleCutoff, toDownsample.size()));
    }

    @Override
    public CompactionInfo getCompactionInfo() {
        return CompactionInfo.withoutSSTables(null, OperationType.INDEX_SUMMARY, this.memoryPoolBytes - this.remainingSpace, this.memoryPoolBytes, CompactionInfo.Unit.BYTES, this.compactionId);
    }

    @Override
    public boolean isGlobal() {
        return true;
    }

    private static class ResampleEntry<T extends SSTableReader> {
        public final T sstable;
        public final long newSpaceUsed;
        public final int newSamplingLevel;

        ResampleEntry(T sstable, long newSpaceUsed, int newSamplingLevel) {
            this.sstable = sstable;
            this.newSpaceUsed = newSpaceUsed;
            this.newSamplingLevel = newSamplingLevel;
        }
    }

    private static class ReadRateComparator
    implements Comparator<SSTableReader> {
        private final Map<? extends SSTableReader, Double> readRates;

        ReadRateComparator(Map<? extends SSTableReader, Double> readRates) {
            this.readRates = readRates;
        }

        @Override
        public int compare(SSTableReader o1, SSTableReader o2) {
            Double readRate1 = this.readRates.get(o1);
            Double readRate2 = this.readRates.get(o2);
            if (readRate1 == null && readRate2 == null) {
                return 0;
            }
            if (readRate1 == null) {
                return -1;
            }
            if (readRate2 == null) {
                return 1;
            }
            return Double.compare(readRate1, readRate2);
        }
    }
}

