/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.spark.utils;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.cassandra.spark.data.ReplicationFactor;
import org.apache.cassandra.spark.utils.TableIdentifier;
import org.jetbrains.annotations.NotNull;

public final class CqlUtils {
    private static final List<String> TABLE_PROPERTY_OVERRIDE_ALLOWLIST = Arrays.asList("bloom_filter_fp_chance", "compression", "default_time_to_live", "min_index_interval", "max_index_interval");
    private static final Pattern REPLICATION_FACTOR_PATTERN = Pattern.compile("WITH REPLICATION = (\\{[^\\}]*\\})");
    private static final ObjectMapper MAPPER = new ObjectMapper().configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true);
    private static final Pattern ESCAPED_WHITESPACE_PATTERN = Pattern.compile("(\\\\r|\\\\n|\\\\r\\n)+");
    private static final Pattern NEWLINE_PATTERN = Pattern.compile("\n");
    private static final Pattern ESCAPED_DOUBLE_BACKSLASH = Pattern.compile("\\\\");

    private CqlUtils() {
        throw new IllegalStateException(String.valueOf(this.getClass()) + " is static utility class and shall not be instantiated");
    }

    public static String cleanCql(@NotNull String cql) {
        String result = ESCAPED_WHITESPACE_PATTERN.matcher(cql).replaceAll("\n");
        result = NEWLINE_PATTERN.matcher(result).replaceAll("");
        result = ESCAPED_DOUBLE_BACKSLASH.matcher(result).replaceAll("");
        return result;
    }

    private static String removeTableProps(@NotNull String schema) {
        int index = schema.indexOf(40);
        int count = 1;
        if (index < 0) {
            throw new RuntimeException("Missing parentheses in table schema " + schema);
        }
        while (++index < schema.length()) {
            if (schema.charAt(index) == ')') {
                --count;
            } else if (schema.charAt(index) == '(') {
                ++count;
            }
            if (count != 0) continue;
        }
        if (count > 0) {
            throw new RuntimeException("Found unbalanced parentheses in table schema " + schema);
        }
        return schema.substring(0, index + 1);
    }

    public static Set<String> extractKeyspaceNames(@NotNull String schemaStr) {
        String cleaned = CqlUtils.cleanCql(schemaStr);
        Pattern pattern = Pattern.compile("CREATE KEYSPACE \"?(\\w+)?\"? [^;]*;");
        Matcher matcher = pattern.matcher(cleaned);
        HashSet<String> keyspaces = new HashSet<String>();
        while (matcher.find()) {
            String keyspace = matcher.group(1);
            if (keyspace.startsWith("system_") || keyspace.startsWith("cie_")) continue;
            keyspaces.add(keyspace);
        }
        return keyspaces;
    }

    public static String extractKeyspaceSchema(@NotNull String schemaStr, @NotNull String keyspace) {
        String cleaned = CqlUtils.cleanCql(schemaStr);
        Pattern pattern = Pattern.compile(String.format("CREATE KEYSPACE \"?%s?\"? [^;]*;", keyspace));
        Matcher matcher = pattern.matcher(cleaned);
        if (!matcher.find()) {
            throw new RuntimeException(String.format("Could not find schema for keyspace: %s", keyspace));
        }
        return cleaned.substring(matcher.start(0), matcher.end(0));
    }

    public static Map<TableIdentifier, String> extractCdcTables(@NotNull String schemaStr) {
        String cleaned = CqlUtils.cleanCql(schemaStr);
        Pattern pattern = Pattern.compile("CREATE TABLE \"?(\\w+)\"?\\.\"?(\\w+)\"?[^;]*cdc = true[^;]*;");
        Matcher matcher = pattern.matcher(cleaned);
        HashMap<TableIdentifier, String> createStmts = new HashMap<TableIdentifier, String>();
        while (matcher.find()) {
            String keyspace = matcher.group(1);
            String table = matcher.group(2);
            createStmts.put(TableIdentifier.of(keyspace, table), CqlUtils.extractCleanedTableSchema(cleaned, keyspace, table));
        }
        return createStmts;
    }

    public static ReplicationFactor extractReplicationFactor(@NotNull String schemaStr, @NotNull String keyspace) {
        Map map;
        String createKeyspaceSchema = CqlUtils.extractKeyspaceSchema(schemaStr, keyspace);
        Matcher matcher = REPLICATION_FACTOR_PATTERN.matcher(createKeyspaceSchema);
        if (!matcher.find()) {
            throw new RuntimeException(String.format("Could not find replication factor for keyspace: %s", keyspace));
        }
        try {
            map = (Map)MAPPER.readValue(matcher.group(1), (TypeReference)new TypeReference<Map<String, String>>(){});
        }
        catch (IOException exception) {
            throw new RuntimeException(String.format("Unable to parse replication factor for keyspace: %s", keyspace), exception);
        }
        String className = (String)map.remove("class");
        ReplicationFactor.ReplicationStrategy strategy = ReplicationFactor.ReplicationStrategy.getEnum(className);
        return new ReplicationFactor(strategy, map.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, v -> Integer.parseInt((String)v.getValue()))));
    }

    public static String extractTableSchema(@NotNull String schemaStr, @NotNull String keyspace, @NotNull String table) {
        return CqlUtils.extractCleanedTableSchema(CqlUtils.cleanCql(schemaStr), keyspace, table);
    }

    public static String extractCleanedTableSchema(@NotNull String createStatementToClean, @NotNull String keyspace, @NotNull String table) {
        Pattern pattern = Pattern.compile(String.format("CREATE TABLE (IF NOT EXISTS)? ?\"?%s?\"?\\.{1}\"?%s\"?[^;]*;", keyspace, table));
        Matcher matcher = pattern.matcher(createStatementToClean);
        if (matcher.find()) {
            List<String> propStrings;
            String fullSchema = createStatementToClean.substring(matcher.start(0), matcher.end(0));
            Object redactedSchema = CqlUtils.removeTableProps(fullSchema);
            String clustering = CqlUtils.extractClustering(fullSchema);
            String separator = " WITH ";
            if (clustering != null) {
                redactedSchema = (String)redactedSchema + separator + clustering;
                separator = " AND ";
            }
            if (!(propStrings = CqlUtils.extractOverrideProperties(fullSchema, TABLE_PROPERTY_OVERRIDE_ALLOWLIST)).isEmpty()) {
                redactedSchema = (String)redactedSchema + separator + String.join((CharSequence)" AND ", propStrings);
                separator = " AND ";
            }
            return (String)redactedSchema + ";";
        }
        throw new RuntimeException(String.format("Could not find schema for table: %s.%s", keyspace, table));
    }

    @VisibleForTesting
    static List<String> extractOverrideProperties(String schemaStr, List<String> properties) {
        ArrayList<String> overrideTableProps = new ArrayList<String>();
        if (properties.isEmpty()) {
            return overrideTableProps;
        }
        Pattern pattern = Pattern.compile("(" + properties.stream().collect(Collectors.joining("|")) + ") = (([\\w|.]+)|(\\{[^}]+}))");
        Matcher matcher = pattern.matcher(schemaStr);
        while (matcher.find()) {
            String parsedProp = schemaStr.substring(matcher.start(), matcher.end());
            overrideTableProps.add(parsedProp);
        }
        return overrideTableProps;
    }

    @VisibleForTesting
    static String extractClustering(String schemaStr) {
        Pattern pattern = Pattern.compile("CLUSTERING ORDER BY \\([^)]*");
        Matcher matcher = pattern.matcher(schemaStr);
        if (matcher.find()) {
            return schemaStr.substring(matcher.start(0), matcher.end(0) + 1);
        }
        return null;
    }

    public static Set<String> extractUdts(@NotNull String schemaStr, @NotNull String keyspace) {
        Pattern pattern = Pattern.compile(String.format("CREATE TYPE (IF NOT EXISTS)? ?\"?%s\"?\\.{1}[^;]*;", keyspace));
        Matcher matcher = pattern.matcher(schemaStr);
        HashSet<String> result = new HashSet<String>();
        while (matcher.find()) {
            result.add(CqlUtils.cleanCql(matcher.group()));
        }
        return result;
    }

    public static int extractIndexCount(@NotNull String schemaStr, @NotNull String keyspace, @NotNull String table) {
        String cleaned = CqlUtils.cleanCql(schemaStr);
        Pattern pattern = Pattern.compile(String.format("CREATE (CUSTOM )?INDEX \"?[^ ]* ON ?\"?%s?\"?\\.{1}\"?%s\"?[^;]*;", keyspace, table));
        Matcher matcher = pattern.matcher(cleaned);
        int indexCount = 0;
        while (matcher.find()) {
            ++indexCount;
        }
        return indexCount;
    }
}

