/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.planner.plan.rules.logical;

import org.apache.flink.table.planner.plan.utils.InputRefVisitor;

import org.apache.calcite.plan.RelOptRule;
import org.apache.calcite.plan.RelOptRuleCall;
import org.apache.calcite.rel.core.Filter;
import org.apache.calcite.rel.core.Project;
import org.apache.calcite.rel.rules.FilterProjectTransposeRule;
import org.apache.calcite.rex.RexNode;
import org.apache.calcite.rex.RexUtil;

import java.util.Arrays;
import java.util.List;

/**
 * Extends calcite's FilterProjectTransposeRule for streaming scenario, modification: does not
 * transpose the top filter references field which generated by non-deterministic function.
 */
public class FlinkFilterProjectTransposeRule extends FilterProjectTransposeRule {

    public static final RelOptRule INSTANCE = new FlinkFilterProjectTransposeRule(Config.DEFAULT);

    public static FlinkFilterProjectTransposeRule build(Config config) {
        return new FlinkFilterProjectTransposeRule(config);
    }

    protected FlinkFilterProjectTransposeRule(Config config) {
        super(config);
    }

    @Override
    public void onMatch(RelOptRuleCall call) {
        Filter filter = call.rel(0);
        Project project = call.rel(1);

        List<RexNode> projects = project.getProjects();
        InputRefVisitor inputRefVisitor = new InputRefVisitor();
        filter.getCondition().accept(inputRefVisitor);
        boolean existNonDeterministicRef =
                Arrays.stream(inputRefVisitor.getFields())
                        .anyMatch(i -> !RexUtil.isDeterministic(projects.get(i)));

        if (!existNonDeterministicRef) {
            super.onMatch(call);
        }
    }
}
