/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.catalog;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import org.apache.flink.annotation.Internal;
import org.apache.flink.table.api.Schema;
import org.apache.flink.table.api.ValidationException;
import org.apache.flink.table.catalog.Column;
import org.apache.flink.table.catalog.DataTypeFactory;
import org.apache.flink.table.catalog.DefaultIndex;
import org.apache.flink.table.catalog.Index;
import org.apache.flink.table.catalog.ResolvedSchema;
import org.apache.flink.table.catalog.SchemaResolver;
import org.apache.flink.table.catalog.UniqueConstraint;
import org.apache.flink.table.catalog.WatermarkSpec;
import org.apache.flink.table.expressions.ApiExpressionUtils;
import org.apache.flink.table.expressions.Expression;
import org.apache.flink.table.expressions.LocalReferenceExpression;
import org.apache.flink.table.expressions.ResolvedExpression;
import org.apache.flink.table.expressions.resolver.ExpressionResolver;
import org.apache.flink.table.types.DataType;
import org.apache.flink.table.types.logical.LocalZonedTimestampType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.TimestampKind;
import org.apache.flink.table.types.logical.TimestampType;
import org.apache.flink.table.types.logical.utils.LogicalTypeChecks;
import org.apache.flink.table.types.utils.DataTypeUtils;

@Internal
class DefaultSchemaResolver
implements SchemaResolver {
    private final boolean isStreamingMode;
    private final DataTypeFactory dataTypeFactory;
    private final ExpressionResolver.ExpressionResolverBuilder resolverBuilder;

    DefaultSchemaResolver(boolean isStreamingMode, DataTypeFactory dataTypeFactory, ExpressionResolver.ExpressionResolverBuilder resolverBuilder) {
        this.isStreamingMode = isStreamingMode;
        this.dataTypeFactory = dataTypeFactory;
        this.resolverBuilder = resolverBuilder;
    }

    @Override
    public ResolvedSchema resolve(Schema schema) {
        List<Column> columns = this.resolveColumns(schema.getColumns());
        List<WatermarkSpec> watermarkSpecs = this.resolveWatermarkSpecs(schema.getWatermarkSpecs(), columns);
        List<Column> columnsWithRowtime = this.adjustRowtimeAttributes(watermarkSpecs, columns);
        UniqueConstraint primaryKey = this.resolvePrimaryKey(schema.getPrimaryKey().orElse(null), columnsWithRowtime);
        List<Index> indexes = this.resolveIndexes(schema.getIndexes(), columnsWithRowtime);
        return new ResolvedSchema(columnsWithRowtime, watermarkSpecs, primaryKey, indexes);
    }

    private List<Column> resolveColumns(List<Schema.UnresolvedColumn> unresolvedColumns) {
        this.validateDuplicateColumns(unresolvedColumns);
        this.validateDuplicateMetadataKeys(unresolvedColumns);
        Column[] resolvedColumns = new Column[unresolvedColumns.size()];
        for (int pos = 0; pos < unresolvedColumns.size(); ++pos) {
            Schema.UnresolvedColumn unresolvedColumn = unresolvedColumns.get(pos);
            if (unresolvedColumn instanceof Schema.UnresolvedPhysicalColumn) {
                resolvedColumns[pos] = this.resolvePhysicalColumn((Schema.UnresolvedPhysicalColumn)unresolvedColumn);
                continue;
            }
            if (unresolvedColumn instanceof Schema.UnresolvedMetadataColumn) {
                resolvedColumns[pos] = this.resolveMetadataColumn((Schema.UnresolvedMetadataColumn)unresolvedColumn);
                continue;
            }
            if (unresolvedColumn instanceof Schema.UnresolvedComputedColumn) continue;
            throw new IllegalArgumentException("Unknown unresolved column type: " + unresolvedColumn.getClass().getName());
        }
        List<Column> sourceColumns = Stream.of(resolvedColumns).filter(Objects::nonNull).collect(Collectors.toList());
        for (int pos = 0; pos < unresolvedColumns.size(); ++pos) {
            Schema.UnresolvedColumn unresolvedColumn = unresolvedColumns.get(pos);
            if (!(unresolvedColumn instanceof Schema.UnresolvedComputedColumn)) continue;
            resolvedColumns[pos] = this.resolveComputedColumn((Schema.UnresolvedComputedColumn)unresolvedColumn, sourceColumns);
        }
        return Arrays.asList(resolvedColumns);
    }

    private Column.PhysicalColumn resolvePhysicalColumn(Schema.UnresolvedPhysicalColumn unresolvedColumn) {
        return Column.physical(unresolvedColumn.getName(), this.dataTypeFactory.createDataType(unresolvedColumn.getDataType())).withComment(unresolvedColumn.getComment().orElse(null));
    }

    private Column.MetadataColumn resolveMetadataColumn(Schema.UnresolvedMetadataColumn unresolvedColumn) {
        return Column.metadata(unresolvedColumn.getName(), this.dataTypeFactory.createDataType(unresolvedColumn.getDataType()), unresolvedColumn.getMetadataKey(), unresolvedColumn.isVirtual()).withComment(unresolvedColumn.getComment().orElse(null));
    }

    private Column.ComputedColumn resolveComputedColumn(Schema.UnresolvedComputedColumn unresolvedColumn, List<Column> inputColumns) {
        ResolvedExpression resolvedExpression;
        try {
            resolvedExpression = this.resolveExpression(inputColumns, unresolvedColumn.getExpression(), null);
        }
        catch (Exception e) {
            throw new ValidationException(String.format("Invalid expression for computed column '%s'.", unresolvedColumn.getName()), e);
        }
        return Column.computed(unresolvedColumn.getName(), resolvedExpression).withComment(unresolvedColumn.getComment().orElse(null));
    }

    private void validateDuplicateColumns(List<Schema.UnresolvedColumn> columns) {
        List names = columns.stream().map(Schema.UnresolvedColumn::getName).collect(Collectors.toList());
        List duplicates = names.stream().filter(name -> Collections.frequency(names, name) > 1).distinct().collect(Collectors.toList());
        if (duplicates.size() > 0) {
            throw new ValidationException(String.format("Schema must not contain duplicate column names. Found duplicates: %s", duplicates));
        }
    }

    private void validateDuplicateMetadataKeys(List<Schema.UnresolvedColumn> columns) {
        HashMap<String, String> metadataKeyToColumnNames = new HashMap<String, String>();
        for (Schema.UnresolvedColumn column : columns) {
            String metadataKey;
            if (!(column instanceof Schema.UnresolvedMetadataColumn)) continue;
            Schema.UnresolvedMetadataColumn metadataColumn = (Schema.UnresolvedMetadataColumn)column;
            String string = metadataKey = metadataColumn.getMetadataKey() == null ? metadataColumn.getName() : metadataColumn.getMetadataKey();
            if (metadataKeyToColumnNames.containsKey(metadataKey)) {
                throw new ValidationException(String.format("The column `%s` and `%s` in the table are both from the same metadata key '%s'. Please specify one of the columns as the metadata column and use the computed column syntax to specify the others.", metadataKeyToColumnNames.get(metadataKey), metadataColumn.getName(), metadataKey));
            }
            metadataKeyToColumnNames.put(metadataKey, metadataColumn.getName());
        }
    }

    private List<WatermarkSpec> resolveWatermarkSpecs(List<Schema.UnresolvedWatermarkSpec> unresolvedWatermarkSpecs, List<Column> inputColumns) {
        ResolvedExpression watermarkExpression;
        if (unresolvedWatermarkSpecs.size() == 0) {
            return Collections.emptyList();
        }
        if (unresolvedWatermarkSpecs.size() > 1) {
            throw new ValidationException("Multiple watermark definitions are not supported yet.");
        }
        Schema.UnresolvedWatermarkSpec watermarkSpec = unresolvedWatermarkSpecs.get(0);
        String timeColumn = watermarkSpec.getColumnName();
        Column validatedTimeColumn = this.validateTimeColumn(timeColumn, inputColumns);
        try {
            watermarkExpression = this.resolveExpression(inputColumns, watermarkSpec.getWatermarkExpression(), validatedTimeColumn.getDataType());
        }
        catch (Exception e) {
            throw new ValidationException(String.format("Invalid expression for watermark '%s'.", watermarkSpec.toString()), e);
        }
        LogicalType outputType = watermarkExpression.getOutputDataType().getLogicalType();
        LogicalType timeColumnType = validatedTimeColumn.getDataType().getLogicalType();
        this.validateWatermarkExpression(outputType);
        if (outputType.getTypeRoot() != timeColumnType.getTypeRoot()) {
            throw new ValidationException(String.format("The watermark declaration's output data type '%s' is different from the time field's data type '%s'.", outputType, timeColumnType));
        }
        return Collections.singletonList(WatermarkSpec.of(watermarkSpec.getColumnName(), watermarkExpression));
    }

    private Column validateTimeColumn(String columnName, List<Column> columns) {
        Optional<Column> timeColumn = columns.stream().filter(c -> c.getName().equals(columnName)).findFirst();
        if (!timeColumn.isPresent()) {
            throw new ValidationException(String.format("Invalid column name '%s' for rowtime attribute in watermark declaration. Available columns are: %s", columnName, columns.stream().map(Column::getName).collect(Collectors.toList())));
        }
        LogicalType timeFieldType = timeColumn.get().getDataType().getLogicalType();
        if (!LogicalTypeChecks.canBeTimeAttributeType(timeFieldType) || LogicalTypeChecks.getPrecision(timeFieldType) > 3) {
            throw new ValidationException(String.format("Invalid data type of time field for watermark definition. The field must be of type TIMESTAMP(p) or TIMESTAMP_LTZ(p), the supported precision 'p' is from 0 to 3, but the time field type is %s", timeFieldType));
        }
        if (LogicalTypeChecks.isProctimeAttribute(timeFieldType)) {
            throw new ValidationException("A watermark can not be defined for a processing-time attribute.");
        }
        return timeColumn.get();
    }

    private void validateWatermarkExpression(LogicalType watermarkType) {
        if (!LogicalTypeChecks.canBeTimeAttributeType(watermarkType) || LogicalTypeChecks.getPrecision(watermarkType) > 3) {
            throw new ValidationException(String.format("Invalid data type of expression for watermark definition. The field must be of type TIMESTAMP(p) or TIMESTAMP_LTZ(p), the supported precision 'p' is from 0 to 3, but the watermark expression type is %s", watermarkType));
        }
    }

    private List<Column> adjustRowtimeAttributes(List<WatermarkSpec> watermarkSpecs, List<Column> columns) {
        return columns.stream().map(column -> this.adjustRowtimeAttribute(watermarkSpecs, (Column)column)).collect(Collectors.toList());
    }

    private Column adjustRowtimeAttribute(List<WatermarkSpec> watermarkSpecs, Column column) {
        String name = column.getName();
        DataType dataType = column.getDataType();
        boolean hasWatermarkSpec = watermarkSpecs.stream().anyMatch(s -> s.getRowtimeAttribute().equals(name));
        if (hasWatermarkSpec && this.isStreamingMode) {
            switch (dataType.getLogicalType().getTypeRoot()) {
                case TIMESTAMP_WITHOUT_TIME_ZONE: {
                    TimestampType originalType = (TimestampType)dataType.getLogicalType();
                    TimestampType rowtimeType = new TimestampType(originalType.isNullable(), TimestampKind.ROWTIME, originalType.getPrecision());
                    return column.copy(DataTypeUtils.replaceLogicalType(dataType, rowtimeType));
                }
                case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                    LocalZonedTimestampType timestampLtzType = (LocalZonedTimestampType)dataType.getLogicalType();
                    LocalZonedTimestampType rowtimeLtzType = new LocalZonedTimestampType(timestampLtzType.isNullable(), TimestampKind.ROWTIME, timestampLtzType.getPrecision());
                    return column.copy(DataTypeUtils.replaceLogicalType(dataType, rowtimeLtzType));
                }
            }
            throw new ValidationException("Invalid data type of expression for rowtime definition. The field must be of type TIMESTAMP(p) or TIMESTAMP_LTZ(p), the supported precision 'p' is from 0 to 3.");
        }
        return column;
    }

    @Nullable
    private UniqueConstraint resolvePrimaryKey(@Nullable Schema.UnresolvedPrimaryKey unresolvedPrimaryKey, List<Column> columns) {
        if (unresolvedPrimaryKey == null) {
            return null;
        }
        UniqueConstraint primaryKey = UniqueConstraint.primaryKey(unresolvedPrimaryKey.getConstraintName(), unresolvedPrimaryKey.getColumnNames());
        this.validatePrimaryKey(primaryKey, columns);
        return primaryKey;
    }

    private void validatePrimaryKey(UniqueConstraint primaryKey, List<Column> columns) {
        Map columnsByNameLookup = columns.stream().collect(Collectors.toMap(Column::getName, Function.identity()));
        Set duplicateColumns = primaryKey.getColumns().stream().filter(name -> Collections.frequency(primaryKey.getColumns(), name) > 1).collect(Collectors.toSet());
        if (!duplicateColumns.isEmpty()) {
            throw new ValidationException(String.format("Invalid primary key '%s'. A primary key must not contain duplicate columns. Found: %s", primaryKey.getName(), duplicateColumns));
        }
        for (String columnName : primaryKey.getColumns()) {
            Column column = (Column)columnsByNameLookup.get(columnName);
            if (column == null) {
                throw new ValidationException(String.format("Invalid primary key '%s'. Column '%s' does not exist.", primaryKey.getName(), columnName));
            }
            if (!column.isPhysical()) {
                throw new ValidationException(String.format("Invalid primary key '%s'. Column '%s' is not a physical column.", primaryKey.getName(), columnName));
            }
            LogicalType columnType = column.getDataType().getLogicalType();
            if (!columnType.isNullable()) continue;
            throw new ValidationException(String.format("Invalid primary key '%s'. Column '%s' is nullable.", primaryKey.getName(), columnName));
        }
    }

    private List<Index> resolveIndexes(List<Schema.UnresolvedIndex> unresolvedIndexes, List<Column> columns) {
        List<Index> resolvedIndexes = unresolvedIndexes.stream().map(unresolvedIndex -> DefaultIndex.newIndex(unresolvedIndex.getIndexName(), unresolvedIndex.getColumnNames())).collect(Collectors.toList());
        this.validateIndexes(resolvedIndexes, columns);
        return resolvedIndexes;
    }

    private void validateIndexes(List<Index> indexes, List<Column> columns) {
        Map<String, Column> columnsByNameLookup = columns.stream().collect(Collectors.toMap(Column::getName, Function.identity()));
        HashSet<List<String>> deduplicatedIndexes = new HashSet<List<String>>();
        for (Index index : indexes) {
            this.validateSingleIndex(index, columnsByNameLookup);
            List<String> indexColumns = index.getColumns();
            if (deduplicatedIndexes.contains(indexColumns)) {
                throw new ValidationException(String.format("Invalid index '%s'. There is a duplicated index composed of the same columns: %s", index.getName(), indexColumns.stream().map(col -> ((Column)columnsByNameLookup.get(col)).getName()).collect(Collectors.joining(", ", "[", "]"))));
            }
            deduplicatedIndexes.add(index.getColumns());
        }
    }

    private void validateSingleIndex(Index index, Map<String, Column> columnsByNameLookup) {
        Set duplicateColumns = index.getColumns().stream().filter(name -> Collections.frequency(index.getColumns(), name) > 1).collect(Collectors.toSet());
        if (!duplicateColumns.isEmpty()) {
            throw new ValidationException(String.format("Invalid index '%s'. An index must not contain duplicate columns. Found: %s", index.getName(), duplicateColumns));
        }
        for (String columnName : index.getColumns()) {
            Column column = columnsByNameLookup.get(columnName);
            if (column == null) {
                throw new ValidationException(String.format("Invalid index '%s'. Column '%s' does not exist.", index.getName(), columnName));
            }
            if (column.isPhysical() || column instanceof Column.MetadataColumn) continue;
            throw new ValidationException(String.format("Invalid index '%s'. Column '%s' is not a physical column or a metadata column.", index.getName(), columnName));
        }
    }

    private ResolvedExpression resolveExpression(List<Column> columns, Expression expression, @Nullable DataType outputDataType) {
        LocalReferenceExpression[] localRefs = (LocalReferenceExpression[])columns.stream().map(c -> ApiExpressionUtils.localRef(c.getName(), c.getDataType())).toArray(LocalReferenceExpression[]::new);
        return this.resolverBuilder.withLocalReferences(localRefs).withOutputDataType(outputDataType).build().resolve(Collections.singletonList(expression)).get(0);
    }
}

