/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.formats.json;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalQueries;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.flink.annotation.Internal;
import org.apache.flink.formats.common.TimeFormats;
import org.apache.flink.formats.common.TimestampFormat;
import org.apache.flink.formats.json.JsonParseException;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.core.JsonParser;
import org.apache.flink.shaded.jackson2.com.fasterxml.jackson.core.JsonToken;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.data.DecimalData;
import org.apache.flink.table.data.GenericArrayData;
import org.apache.flink.table.data.GenericMapData;
import org.apache.flink.table.data.GenericRowData;
import org.apache.flink.table.data.StringData;
import org.apache.flink.table.data.TimestampData;
import org.apache.flink.table.types.logical.ArrayType;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.IntType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.LogicalTypeFamily;
import org.apache.flink.table.types.logical.MapType;
import org.apache.flink.table.types.logical.MultisetType;
import org.apache.flink.table.types.logical.RowType;
import org.apache.flink.table.types.logical.utils.LogicalTypeUtils;

@Internal
public class JsonParserToRowDataConverters
implements Serializable {
    private static final long serialVersionUID = 1L;
    private final boolean failOnMissingField;
    private final boolean ignoreParseErrors;
    private final TimestampFormat timestampFormat;

    public JsonParserToRowDataConverters(boolean failOnMissingField, boolean ignoreParseErrors, TimestampFormat timestampFormat) {
        this.failOnMissingField = failOnMissingField;
        this.ignoreParseErrors = ignoreParseErrors;
        this.timestampFormat = timestampFormat;
    }

    public JsonParserToRowDataConverter createConverter(String[][] projectedFields, final RowType rowType) {
        if (projectedFields == null || Arrays.stream(projectedFields).allMatch(arr -> ((String[])arr).length == 1)) {
            return this.createConverter((LogicalType)rowType);
        }
        final RowNestedConverter rowConverter = new RowNestedConverter();
        for (int i = 0; i < projectedFields.length; ++i) {
            this.addFieldConverter(rowConverter.fieldConverters, projectedFields[i], 0, i, rowType.getTypeAt(i));
        }
        return new JsonParserToRowDataConverter(){

            @Override
            public Object convert(JsonParser jp) throws IOException {
                GenericRowData row = new GenericRowData(rowType.getFieldCount());
                rowConverter.convert(jp, row);
                return row;
            }
        };
    }

    private JsonParserToRowDataConverter createConverter(LogicalType type) {
        return this.wrapIntoNullableConverter(this.createNotNullConverter(type));
    }

    private JsonParserToRowDataConverter createNotNullConverter(LogicalType type) {
        switch (type.getTypeRoot()) {
            case NULL: {
                return jsonNode -> null;
            }
            case BOOLEAN: {
                return this::convertToBoolean;
            }
            case TINYINT: {
                return this::convertToByte;
            }
            case SMALLINT: {
                return this::convertToShort;
            }
            case INTEGER: 
            case INTERVAL_YEAR_MONTH: {
                return this::convertToInt;
            }
            case BIGINT: 
            case INTERVAL_DAY_TIME: {
                return this::convertToLong;
            }
            case DATE: {
                return this::convertToDate;
            }
            case TIME_WITHOUT_TIME_ZONE: {
                return this::convertToTime;
            }
            case TIMESTAMP_WITHOUT_TIME_ZONE: {
                return this::convertToTimestamp;
            }
            case TIMESTAMP_WITH_LOCAL_TIME_ZONE: {
                return this::convertToTimestampWithLocalZone;
            }
            case FLOAT: {
                return this::convertToFloat;
            }
            case DOUBLE: {
                return this::convertToDouble;
            }
            case CHAR: 
            case VARCHAR: {
                return this::convertToString;
            }
            case BINARY: 
            case VARBINARY: {
                return JsonParser::getBinaryValue;
            }
            case DECIMAL: {
                return this.createDecimalConverter((DecimalType)type);
            }
            case ARRAY: {
                return this.createArrayConverter((ArrayType)type);
            }
            case MAP: {
                MapType mapType = (MapType)type;
                return this.createMapConverter(mapType.asSummaryString(), mapType.getKeyType(), mapType.getValueType());
            }
            case MULTISET: {
                MultisetType multisetType = (MultisetType)type;
                return this.createMapConverter(multisetType.asSummaryString(), multisetType.getElementType(), (LogicalType)new IntType());
            }
            case ROW: {
                return this.createRowConverter((RowType)type);
            }
        }
        throw new UnsupportedOperationException("Unsupported type: " + String.valueOf(type));
    }

    private boolean convertToBoolean(JsonParser jp) throws IOException {
        if (jp.currentToken() == JsonToken.VALUE_TRUE) {
            return true;
        }
        if (jp.currentToken() == JsonToken.VALUE_FALSE) {
            return false;
        }
        return Boolean.parseBoolean(jp.getText().trim());
    }

    private byte convertToByte(JsonParser jp) throws IOException {
        if (jp.currentToken() == JsonToken.VALUE_NUMBER_INT) {
            int value = jp.getIntValue();
            if (value < -128 || value > 127) {
                throw new JsonParseException(String.format("Numeric value (%s) out of range of Java byte.", value));
            }
            return (byte)value;
        }
        return Byte.parseByte(jp.getText().trim());
    }

    private short convertToShort(JsonParser jp) throws IOException {
        if (jp.currentToken() == JsonToken.VALUE_NUMBER_INT) {
            return jp.getShortValue();
        }
        return Short.parseShort(jp.getText().trim());
    }

    private int convertToInt(JsonParser jp) throws IOException {
        if (jp.currentToken() == JsonToken.VALUE_NUMBER_INT || jp.currentToken() == JsonToken.VALUE_NUMBER_FLOAT) {
            return jp.getIntValue();
        }
        return Integer.parseInt(jp.getText().trim());
    }

    private long convertToLong(JsonParser jp) throws IOException {
        if (jp.currentToken() == JsonToken.VALUE_NUMBER_INT || jp.currentToken() == JsonToken.VALUE_NUMBER_FLOAT) {
            return jp.getLongValue();
        }
        return Long.parseLong(jp.getText().trim());
    }

    private double convertToDouble(JsonParser jp) throws IOException {
        if (jp.currentToken() == JsonToken.VALUE_NUMBER_FLOAT) {
            return jp.getDoubleValue();
        }
        return Double.parseDouble(jp.getText().trim());
    }

    private float convertToFloat(JsonParser jp) throws IOException {
        if (jp.currentToken() == JsonToken.VALUE_NUMBER_FLOAT) {
            return jp.getFloatValue();
        }
        return Float.parseFloat(jp.getText().trim());
    }

    private int convertToDate(JsonParser jp) throws IOException {
        LocalDate date = DateTimeFormatter.ISO_LOCAL_DATE.parse(jp.getText()).query(TemporalQueries.localDate());
        return (int)date.toEpochDay();
    }

    private int convertToTime(JsonParser jsonNode) throws IOException {
        TemporalAccessor parsedTime = TimeFormats.SQL_TIME_FORMAT.parse(jsonNode.getText());
        LocalTime localTime = parsedTime.query(TemporalQueries.localTime());
        return localTime.toSecondOfDay() * 1000;
    }

    private TimestampData convertToTimestamp(JsonParser jp) throws IOException {
        TemporalAccessor parsedTimestamp;
        switch (this.timestampFormat) {
            case SQL: {
                parsedTimestamp = TimeFormats.SQL_TIMESTAMP_FORMAT.parse(jp.getText());
                break;
            }
            case ISO_8601: {
                parsedTimestamp = TimeFormats.ISO8601_TIMESTAMP_FORMAT.parse(jp.getText());
                break;
            }
            default: {
                throw new TableException(String.format("Unsupported timestamp format '%s'. Validator should have checked that.", new Object[]{this.timestampFormat}));
            }
        }
        LocalTime localTime = parsedTimestamp.query(TemporalQueries.localTime());
        LocalDate localDate = parsedTimestamp.query(TemporalQueries.localDate());
        return TimestampData.fromLocalDateTime((LocalDateTime)LocalDateTime.of(localDate, localTime));
    }

    private TimestampData convertToTimestampWithLocalZone(JsonParser jp) throws IOException {
        TemporalAccessor parsedTimestampWithLocalZone;
        switch (this.timestampFormat) {
            case SQL: {
                parsedTimestampWithLocalZone = TimeFormats.SQL_TIMESTAMP_WITH_LOCAL_TIMEZONE_FORMAT.parse(jp.getText());
                break;
            }
            case ISO_8601: {
                parsedTimestampWithLocalZone = TimeFormats.ISO8601_TIMESTAMP_WITH_LOCAL_TIMEZONE_FORMAT.parse(jp.getText());
                break;
            }
            default: {
                throw new TableException(String.format("Unsupported timestamp format '%s'. Validator should have checked that.", new Object[]{this.timestampFormat}));
            }
        }
        LocalTime localTime = parsedTimestampWithLocalZone.query(TemporalQueries.localTime());
        LocalDate localDate = parsedTimestampWithLocalZone.query(TemporalQueries.localDate());
        return TimestampData.fromInstant((Instant)LocalDateTime.of(localDate, localTime).toInstant(ZoneOffset.UTC));
    }

    private StringData convertToString(JsonParser jp) throws IOException {
        if (jp.currentToken() == JsonToken.START_OBJECT || jp.currentToken() == JsonToken.START_ARRAY) {
            return StringData.fromString((String)jp.readValueAsTree().toString());
        }
        return StringData.fromString((String)jp.getText());
    }

    private JsonParserToRowDataConverter createDecimalConverter(DecimalType decimalType) {
        int precision = decimalType.getPrecision();
        int scale = decimalType.getScale();
        return jp -> {
            BigDecimal bigDecimal = jp.currentToken() == JsonToken.VALUE_STRING ? new BigDecimal(jp.getText().trim()) : jp.getDecimalValue();
            return DecimalData.fromBigDecimal((BigDecimal)bigDecimal, (int)precision, (int)scale);
        };
    }

    private JsonParserToRowDataConverter createArrayConverter(ArrayType arrayType) {
        JsonParserToRowDataConverter elementConverter = this.createConverter(arrayType.getElementType());
        Class elementClass = LogicalTypeUtils.toInternalConversionClass((LogicalType)arrayType.getElementType());
        return jp -> {
            if (jp.currentToken() != JsonToken.START_ARRAY) {
                throw new IllegalStateException("Illegal JSON array data...");
            }
            ArrayList<Object> result = new ArrayList<Object>();
            while (jp.nextToken() != JsonToken.END_ARRAY) {
                Object convertField = elementConverter.convert(jp);
                result.add(convertField);
            }
            Object[] array = (Object[])Array.newInstance(elementClass, result.size());
            return new GenericArrayData(result.toArray(array));
        };
    }

    private JsonParserToRowDataConverter createMapConverter(String typeSummary, LogicalType keyType, LogicalType valueType) {
        if (!keyType.is(LogicalTypeFamily.CHARACTER_STRING)) {
            throw new UnsupportedOperationException("JSON format doesn't support non-string as key type of map. The type is: " + typeSummary);
        }
        JsonParserToRowDataConverter keyConverter = this.createConverter(keyType);
        JsonParserToRowDataConverter valueConverter = this.createConverter(valueType);
        return jp -> {
            if (jp.currentToken() != JsonToken.START_OBJECT) {
                throw new IllegalStateException("Illegal JSON map data...");
            }
            HashMap<Object, Object> result = new HashMap<Object, Object>();
            while (jp.nextToken() != JsonToken.END_OBJECT) {
                Object key = keyConverter.convert(jp);
                jp.nextToken();
                Object value = valueConverter.convert(jp);
                result.put(key, value);
            }
            return new GenericMapData(result);
        };
    }

    public JsonParserToRowDataConverter createRowConverter(RowType rowType) {
        JsonParserToRowDataConverter[] fieldConverters = (JsonParserToRowDataConverter[])rowType.getFields().stream().map(RowType.RowField::getType).map(this::createConverter).toArray(JsonParserToRowDataConverter[]::new);
        String[] fieldNames = rowType.getFieldNames().toArray(new String[0]);
        HashMap<String, Integer> nameIdxMap = new HashMap<String, Integer>();
        for (int i = 0; i < rowType.getFieldCount(); ++i) {
            nameIdxMap.put(fieldNames[i], i);
        }
        return jp -> {
            if (jp.currentToken() != JsonToken.START_OBJECT) {
                throw new IllegalStateException("Illegal JSON object data...");
            }
            int arity = nameIdxMap.size();
            GenericRowData row = new GenericRowData(arity);
            int cnt = 0;
            jp.nextToken();
            while (jp.currentToken() != JsonToken.END_OBJECT) {
                if (cnt >= arity) {
                    JsonParserToRowDataConverters.skipToNextField(jp);
                    continue;
                }
                String fieldName = jp.getText();
                jp.nextToken();
                Integer idx = (Integer)nameIdxMap.get(fieldName);
                if (idx != null) {
                    try {
                        Object convertField = fieldConverters[idx].convert(jp);
                        row.setField(idx.intValue(), convertField);
                    }
                    catch (Throwable t) {
                        throw new JsonParseException(String.format("Fail to deserialize at field: %s.", fieldName));
                    }
                    jp.nextToken();
                    ++cnt;
                    continue;
                }
                JsonParserToRowDataConverters.skipToNextField(jp);
            }
            if (cnt < arity && this.failOnMissingField) {
                throw new JsonParseException("Some field is missing in the JSON data.");
            }
            return row;
        };
    }

    public static void skipToNextField(JsonParser jp) throws IOException {
        switch (jp.currentToken()) {
            case START_OBJECT: 
            case START_ARRAY: {
                int match = 1;
                while (match > 0) {
                    JsonToken token = jp.nextToken();
                    if (token == JsonToken.END_ARRAY || token == JsonToken.END_OBJECT) {
                        --match;
                        continue;
                    }
                    if (token != JsonToken.START_ARRAY && token != JsonToken.START_OBJECT) continue;
                    ++match;
                }
                break;
            }
        }
        jp.nextToken();
    }

    private JsonParserToRowDataConverter wrapIntoNullableConverter(JsonParserToRowDataConverter converter) {
        return jp -> {
            if (jp == null || jp.currentToken() == null || jp.getCurrentToken() == JsonToken.VALUE_NULL) {
                return null;
            }
            try {
                return converter.convert(jp);
            }
            catch (Throwable t) {
                if (!this.ignoreParseErrors) {
                    throw t;
                }
                return null;
            }
        };
    }

    private void addFieldConverter(Map<String, ProjectedConverter> converters, String[] nestedField, int depth, int outputPos, LogicalType type) {
        String name = nestedField[depth];
        if (depth == nestedField.length - 1) {
            FieldConverter fieldConverter = new FieldConverter(outputPos, type);
            ProjectedConverter converter = converters.get(name);
            if (converter instanceof RowNestedConverter) {
                for (Map.Entry<Integer, String[]> entry : ((RowNestedConverter)converter).outputPosToPath.entrySet()) {
                    fieldConverter.addNestedFieldPath(entry.getKey(), entry.getValue());
                }
            } else if (converter instanceof FieldConverter) {
                throw new RuntimeException("This is a bug, contains duplicated fields.");
            }
            converters.put(name, fieldConverter);
        } else {
            ProjectedConverter converter = converters.computeIfAbsent(name, k -> new RowNestedConverter());
            String[] namePath = (String[])ArrayUtils.subarray((Object[])nestedField, (int)(depth + 1), (int)nestedField.length);
            if (converter instanceof FieldConverter) {
                ((FieldConverter)converter).addNestedFieldPath(outputPos, namePath);
            } else {
                RowNestedConverter rowConverter = (RowNestedConverter)converter;
                rowConverter.outputPosToPath.put(outputPos, namePath);
                this.addFieldConverter(rowConverter.fieldConverters, nestedField, depth + 1, outputPos, type);
            }
        }
    }

    private class RowNestedConverter
    extends ProjectedConverter {
        private static final long serialVersionUID = 1L;
        private final Map<String, ProjectedConverter> fieldConverters = new HashMap<String, ProjectedConverter>();
        private final Map<Integer, String[]> outputPosToPath = new HashMap<Integer, String[]>();

        private RowNestedConverter() {
        }

        @Override
        public void convertNotNull(JsonParser jp, GenericRowData outputRow) throws IOException {
            if (jp.currentToken() != JsonToken.START_OBJECT) {
                throw new IllegalStateException("Illegal Json Data...");
            }
            int arity = this.fieldConverters.size();
            int cnt = 0;
            jp.nextToken();
            while (jp.currentToken() != JsonToken.END_OBJECT) {
                if (cnt >= arity) {
                    JsonParserToRowDataConverters.skipToNextField(jp);
                    continue;
                }
                String fieldName = jp.getText();
                jp.nextToken();
                ProjectedConverter converter = this.fieldConverters.get(fieldName);
                if (converter != null) {
                    converter.convert(jp, outputRow);
                    jp.nextToken();
                    ++cnt;
                    continue;
                }
                JsonParserToRowDataConverters.skipToNextField(jp);
            }
            if (cnt < arity && JsonParserToRowDataConverters.this.failOnMissingField) {
                throw new JsonParseException("Some field is missing in the Json data.");
            }
        }
    }

    private class FieldConverter
    extends ProjectedConverter {
        private static final long serialVersionUID = 1L;
        private final int outputPos;
        private final JsonParserToRowDataConverter converter;
        private final LogicalType type;
        private final Map<Integer, int[]> outputPosToPath = new HashMap<Integer, int[]>();

        public FieldConverter(int outputPos, LogicalType type) {
            this.outputPos = outputPos;
            this.converter = JsonParserToRowDataConverters.this.createConverter(type);
            this.type = type;
        }

        @Override
        public void convertNotNull(JsonParser jp, GenericRowData outputRow) throws IOException {
            Object field = this.converter.convert(jp);
            outputRow.setField(this.outputPos, field);
            if (field != null && !this.outputPosToPath.isEmpty()) {
                this.outNestedFields(field, outputRow);
            }
        }

        private void outNestedFields(Object field, GenericRowData outputRow) {
            block0: for (Map.Entry<Integer, int[]> entry : this.outputPosToPath.entrySet()) {
                Object currentField = field;
                for (int i : entry.getValue()) {
                    if (currentField == null) continue block0;
                    currentField = ((GenericRowData)currentField).getField(i);
                }
                outputRow.setField(entry.getKey().intValue(), currentField);
            }
        }

        public void addNestedFieldPath(int pos, String[] namePath) {
            int[] path = new int[namePath.length];
            LogicalType currentType = this.type;
            for (int i = 0; i < path.length; ++i) {
                if (!(currentType instanceof RowType)) {
                    throw new RuntimeException("This is a bug, currentType should be row type, but is: " + String.valueOf(currentType));
                }
                int fieldIndex = ((RowType)currentType).getFieldNames().indexOf(namePath[i]);
                currentType = (LogicalType)currentType.getChildren().get(fieldIndex);
                path[i] = fieldIndex;
            }
            this.outputPosToPath.put(pos, path);
        }
    }

    private static abstract class ProjectedConverter
    implements Serializable {
        private static final long serialVersionUID = 1L;

        private ProjectedConverter() {
        }

        public final void convert(JsonParser jp, GenericRowData outputRow) throws IOException {
            if (jp != null && jp.currentToken() != null && jp.getCurrentToken() != JsonToken.VALUE_NULL) {
                this.convertNotNull(jp, outputRow);
            }
        }

        public abstract void convertNotNull(JsonParser var1, GenericRowData var2) throws IOException;
    }

    @FunctionalInterface
    public static interface JsonParserToRowDataConverter
    extends Serializable {
        public Object convert(JsonParser var1) throws IOException;
    }
}

