/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.assignment;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.TreeMap;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellBuilderFactory;
import org.apache.hadoop.hbase.CellBuilderType;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.MetaTableAccessor;
import org.apache.hadoop.hbase.RegionLocations;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Mutation;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.RegionInfo;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.client.Table;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.master.MasterFileSystem;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.RegionState;
import org.apache.hadoop.hbase.master.assignment.RegionStateNode;
import org.apache.hadoop.hbase.procedure2.util.StringUtils;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.KeeperException;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.wal.WALSplitUtil;
import org.apache.hadoop.hbase.zookeeper.MetaTableLocator;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZKWatcher;
import org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class RegionStateStore {
    private static final Logger LOG = LoggerFactory.getLogger(RegionStateStore.class);
    private static final Logger METALOG = LoggerFactory.getLogger((String)"org.apache.hadoop.hbase.META");
    protected static final char META_REPLICA_ID_DELIMITER = '_';
    private final MasterServices master;

    public RegionStateStore(MasterServices master) {
        this.master = master;
    }

    public void visitMeta(final RegionStateVisitor visitor) throws IOException {
        MetaTableAccessor.fullScanRegions(this.master.getConnection(), new MetaTableAccessor.Visitor(){
            final boolean isDebugEnabled = RegionStateStore.access$000().isDebugEnabled();

            @Override
            public boolean visit(Result r) throws IOException {
                if (r != null && !r.isEmpty()) {
                    long st = 0L;
                    if (LOG.isTraceEnabled()) {
                        st = System.currentTimeMillis();
                    }
                    RegionStateStore.this.visitMetaEntry(visitor, r);
                    if (LOG.isTraceEnabled()) {
                        long et = System.currentTimeMillis();
                        LOG.trace("[T] LOAD META PERF " + StringUtils.humanTimeDiff(et - st));
                    }
                } else if (this.isDebugEnabled) {
                    LOG.debug("NULL result from meta - ignoring but this is strange.");
                }
                return true;
            }
        });
    }

    public void visitMetaForRegion(String regionEncodedName, RegionStateVisitor visitor) throws IOException {
        Result result = MetaTableAccessor.scanByRegionEncodedName(this.master.getConnection(), regionEncodedName);
        if (result != null) {
            this.visitMetaEntry(visitor, result);
        }
    }

    private void visitMetaEntry(RegionStateVisitor visitor, Result result) throws IOException {
        RegionLocations rl = MetaTableAccessor.getRegionLocations(result);
        if (rl == null) {
            return;
        }
        HRegionLocation[] locations = rl.getRegionLocations();
        if (locations == null) {
            return;
        }
        for (int i = 0; i < locations.length; ++i) {
            RegionInfo regionInfo;
            HRegionLocation hrl = locations[i];
            if (hrl == null || (regionInfo = hrl.getRegion()) == null) continue;
            int replicaId = regionInfo.getReplicaId();
            RegionState.State state = RegionStateStore.getRegionState(result, regionInfo);
            ServerName lastHost = hrl.getServerName();
            ServerName regionLocation = MetaTableAccessor.getTargetServerName(result, replicaId);
            long openSeqNum = hrl.getSeqNum();
            LOG.debug("Load hbase:meta entry region={}, regionState={}, lastHost={}, regionLocation={}, openSeqNum={}", new Object[]{regionInfo.getEncodedName(), state, lastHost, regionLocation, openSeqNum});
            visitor.visitRegionState(result, regionInfo, state, regionLocation, lastHost, openSeqNum);
        }
    }

    void updateRegionLocation(RegionStateNode regionStateNode) throws IOException {
        if (regionStateNode.getRegionInfo().isMetaRegion()) {
            this.updateMetaLocation(regionStateNode.getRegionInfo(), regionStateNode.getRegionLocation(), regionStateNode.getState());
        } else {
            long openSeqNum = regionStateNode.getState() == RegionState.State.OPEN ? regionStateNode.getOpenSeqNum() : -1L;
            this.updateUserRegionLocation(regionStateNode.getRegionInfo(), regionStateNode.getState(), regionStateNode.getRegionLocation(), openSeqNum, regionStateNode.getProcedure() != null ? regionStateNode.getProcedure().getProcId() : -1L);
        }
    }

    private void updateMetaLocation(RegionInfo regionInfo, ServerName serverName, RegionState.State state) throws IOException {
        try {
            MetaTableLocator.setMetaLocation(this.master.getZooKeeper(), serverName, regionInfo.getReplicaId(), state);
        }
        catch (KeeperException e) {
            throw new IOException(e);
        }
    }

    private void updateUserRegionLocation(RegionInfo regionInfo, RegionState.State state, ServerName regionLocation, long openSeqNum, long pid) throws IOException {
        long time = EnvironmentEdgeManager.currentTime();
        int replicaId = regionInfo.getReplicaId();
        Put put = new Put(MetaTableAccessor.getMetaKeyForRegion(regionInfo), time);
        MetaTableAccessor.addRegionInfo(put, regionInfo);
        StringBuilder info = new StringBuilder("pid=").append(pid).append(" updating hbase:meta row=").append(regionInfo.getEncodedName()).append(", regionState=").append((Object)state);
        if (openSeqNum >= 0L) {
            Preconditions.checkArgument(state == RegionState.State.OPEN && regionLocation != null, "Open region should be on a server");
            MetaTableAccessor.addLocation(put, regionLocation, openSeqNum, replicaId);
            if (regionInfo.getReplicaId() == 0 && this.hasGlobalReplicationScope(regionInfo.getTable())) {
                MetaTableAccessor.addReplicationBarrier(put, openSeqNum);
                info.append(", repBarrier=").append(openSeqNum);
            }
            info.append(", openSeqNum=").append(openSeqNum);
            info.append(", regionLocation=").append(regionLocation);
        } else if (regionLocation != null) {
            put.add(CellBuilderFactory.create(CellBuilderType.SHALLOW_COPY).setRow(put.getRow()).setFamily(HConstants.CATALOG_FAMILY).setQualifier(MetaTableAccessor.getServerNameColumn(replicaId)).setTimestamp(put.getTimestamp()).setType(Cell.Type.Put).setValue(Bytes.toBytes(regionLocation.getServerName())).build());
            info.append(", regionLocation=").append(regionLocation);
        }
        put.add(CellBuilderFactory.create(CellBuilderType.SHALLOW_COPY).setRow(put.getRow()).setFamily(HConstants.CATALOG_FAMILY).setQualifier(RegionStateStore.getStateColumn(replicaId)).setTimestamp(put.getTimestamp()).setType(Cell.Type.Put).setValue(Bytes.toBytes(state.name())).build());
        LOG.info(info.toString());
        this.updateRegionLocation(regionInfo, state, put);
    }

    private void updateRegionLocation(RegionInfo regionInfo, RegionState.State state, Put put) throws IOException {
        try (Table table = this.getMetaTable();){
            RegionStateStore.debugLogMutation(put);
            table.put(put);
        }
        catch (IOException e) {
            String msg = String.format("FAILED persisting region=%s state=%s", new Object[]{regionInfo.getShortNameToLog(), state});
            LOG.error(msg, (Throwable)e);
            this.master.abort(msg, e);
            throw e;
        }
    }

    private long getOpenSeqNumForParentRegion(RegionInfo region) throws IOException {
        MasterFileSystem fs = this.master.getMasterFileSystem();
        long maxSeqId = WALSplitUtil.getMaxRegionSequenceId(this.master.getConfiguration(), region, fs::getFileSystem, fs::getWALFileSystem);
        return maxSeqId > 0L ? maxSeqId + 1L : -1L;
    }

    private Table getMetaTable() throws IOException {
        return this.master.getConnection().getTable(TableName.META_TABLE_NAME);
    }

    public void splitRegion(RegionInfo parent, RegionInfo hriA, RegionInfo hriB, ServerName serverName) throws IOException {
        TableDescriptor htd = this.getTableDescriptor(parent.getTable());
        long parentOpenSeqNum = -1L;
        if (htd.hasGlobalReplicationScope()) {
            parentOpenSeqNum = this.getOpenSeqNumForParentRegion(parent);
        }
        MetaTableAccessor.splitRegion(this.master.getConnection(), parent, parentOpenSeqNum, hriA, hriB, serverName, this.getRegionReplication(htd));
    }

    public void mergeRegions(RegionInfo child, RegionInfo[] parents, ServerName serverName) throws IOException {
        TableDescriptor htd = this.getTableDescriptor(child.getTable());
        boolean globalScope = htd.hasGlobalReplicationScope();
        TreeMap<RegionInfo, Long> parentSeqNums = new TreeMap<RegionInfo, Long>();
        for (RegionInfo ri : parents) {
            parentSeqNums.put(ri, globalScope ? this.getOpenSeqNumForParentRegion(ri) : -1L);
        }
        MetaTableAccessor.mergeRegions(this.master.getConnection(), child, parentSeqNums, serverName, this.getRegionReplication(htd));
    }

    public void deleteRegion(RegionInfo regionInfo) throws IOException {
        this.deleteRegions(Collections.singletonList(regionInfo));
    }

    public void deleteRegions(List<RegionInfo> regions) throws IOException {
        MetaTableAccessor.deleteRegionInfos(this.master.getConnection(), regions);
    }

    private Scan getScanForUpdateRegionReplicas(TableName tableName) {
        return MetaTableAccessor.getScanForTableName(this.master.getConfiguration(), tableName).addColumn(HConstants.CATALOG_FAMILY, HConstants.REGIONINFO_QUALIFIER);
    }

    public void removeRegionReplicas(TableName tableName, int oldReplicaCount, int newReplicaCount) throws IOException {
        if (TableName.isMetaTableName(tableName)) {
            ZKWatcher zk = this.master.getZooKeeper();
            try {
                for (int i = newReplicaCount; i < oldReplicaCount; ++i) {
                    ZKUtil.deleteNode(zk, zk.getZNodePaths().getZNodeForReplica(i));
                }
            }
            catch (KeeperException e) {
                throw new IOException(e);
            }
        }
        Scan scan = this.getScanForUpdateRegionReplicas(tableName);
        ArrayList<Delete> deletes = new ArrayList<Delete>();
        long now = EnvironmentEdgeManager.currentTime();
        try (Table metaTable = this.getMetaTable();
             ResultScanner scanner = metaTable.getScanner(scan);){
            Result result;
            while ((result = scanner.next()) != null) {
                RegionInfo primaryRegionInfo = MetaTableAccessor.getRegionInfo(result);
                if (primaryRegionInfo == null || primaryRegionInfo.isSplitParent()) continue;
                Delete delete = new Delete(result.getRow());
                for (int i = newReplicaCount; i < oldReplicaCount; ++i) {
                    delete.addColumns(HConstants.CATALOG_FAMILY, MetaTableAccessor.getServerColumn(i), now);
                    delete.addColumns(HConstants.CATALOG_FAMILY, MetaTableAccessor.getSeqNumColumn(i), now);
                    delete.addColumns(HConstants.CATALOG_FAMILY, MetaTableAccessor.getStartCodeColumn(i), now);
                    delete.addColumns(HConstants.CATALOG_FAMILY, MetaTableAccessor.getServerNameColumn(i), now);
                    delete.addColumns(HConstants.CATALOG_FAMILY, MetaTableAccessor.getRegionStateColumn(i), now);
                }
                deletes.add(delete);
            }
            RegionStateStore.debugLogMutations(deletes);
            metaTable.delete(deletes);
        }
    }

    private boolean hasGlobalReplicationScope(TableName tableName) throws IOException {
        return this.hasGlobalReplicationScope(this.getTableDescriptor(tableName));
    }

    private boolean hasGlobalReplicationScope(TableDescriptor htd) {
        return htd != null ? htd.hasGlobalReplicationScope() : false;
    }

    private int getRegionReplication(TableDescriptor htd) {
        return htd != null ? htd.getRegionReplication() : 1;
    }

    private TableDescriptor getTableDescriptor(TableName tableName) throws IOException {
        return this.master.getTableDescriptors().get(tableName);
    }

    public static RegionState.State getRegionState(Result r, RegionInfo regionInfo) {
        Cell cell = r.getColumnLatestCell(HConstants.CATALOG_FAMILY, RegionStateStore.getStateColumn(regionInfo.getReplicaId()));
        if (cell == null || cell.getValueLength() == 0) {
            return null;
        }
        String state = Bytes.toString(cell.getValueArray(), cell.getValueOffset(), cell.getValueLength());
        try {
            return RegionState.State.valueOf(state);
        }
        catch (IllegalArgumentException e) {
            LOG.warn("BAD value {} in hbase:meta info:state column for region {} , Consider using HBCK2 setRegionState ENCODED_REGION_NAME STATE", (Object)state, (Object)regionInfo.getEncodedName());
            return null;
        }
    }

    private static byte[] getStateColumn(int replicaId) {
        return replicaId == 0 ? HConstants.STATE_QUALIFIER : Bytes.toBytes("state_" + String.format("%04X", replicaId));
    }

    private static void debugLogMutations(List<? extends Mutation> mutations) throws IOException {
        if (!METALOG.isDebugEnabled()) {
            return;
        }
        for (Mutation mutation : mutations) {
            RegionStateStore.debugLogMutation(mutation);
        }
    }

    private static void debugLogMutation(Mutation p) throws IOException {
        METALOG.debug("{} {}", (Object)p.getClass().getSimpleName(), (Object)p.toJSON());
    }

    public static interface RegionStateVisitor {
        public void visitRegionState(Result var1, RegionInfo var2, RegionState.State var3, ServerName var4, ServerName var5, long var6);
    }
}

