/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.backup.impl;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.backup.BackupInfo;
import org.apache.hadoop.hbase.backup.BackupRequest;
import org.apache.hadoop.hbase.backup.BackupRestoreConstants;
import org.apache.hadoop.hbase.backup.BackupType;
import org.apache.hadoop.hbase.backup.HBackupFileSystem;
import org.apache.hadoop.hbase.backup.impl.BackupAdminImpl;
import org.apache.hadoop.hbase.backup.impl.BackupSystemTable;
import org.apache.hadoop.hbase.backup.impl.TableBackupClient;
import org.apache.hadoop.hbase.backup.util.BackupSet;
import org.apache.hadoop.hbase.backup.util.BackupUtils;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.ConnectionFactory;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hbase.thirdparty.com.google.common.base.Splitter;
import org.apache.hbase.thirdparty.com.google.common.collect.Lists;
import org.apache.hbase.thirdparty.org.apache.commons.cli.CommandLine;
import org.apache.hbase.thirdparty.org.apache.commons.cli.HelpFormatter;
import org.apache.hbase.thirdparty.org.apache.commons.cli.Options;
import org.apache.yetus.audience.InterfaceAudience;

@InterfaceAudience.Private
public final class BackupCommands {
    public static final String INCORRECT_USAGE = "Incorrect usage";
    public static final String TOP_LEVEL_NOT_ALLOWED = "Top level (root) folder is not allowed to be a backup destination";
    public static final String USAGE = "Usage: hbase backup COMMAND [command-specific arguments]\nwhere COMMAND is one of:\n  create     create a new backup image\n  delete     delete an existing backup image\n  describe   show the detailed information of a backup image\n  history    show history of all successful backups\n  progress   show the progress of the latest backup request\n  set        backup set management\n  repair     repair backup system table\n  merge      merge backup images\nRun 'hbase backup COMMAND -h' to see help message for each command\n";
    public static final String CREATE_CMD_USAGE = "Usage: hbase backup create <type> <backup_path> [options]\n  type           \"full\" to create a full backup image\n                 \"incremental\" to create an incremental backup image\n  backup_path     Full path to store the backup image\n";
    public static final String PROGRESS_CMD_USAGE = "Usage: hbase backup progress <backup_id>\n  backup_id       Backup image id (optional). If no id specified, the command will show\n                  progress for currently running backup session.";
    public static final String NO_INFO_FOUND = "No info was found for backup id: ";
    public static final String NO_ACTIVE_SESSION_FOUND = "No active backup sessions found.";
    public static final String DESCRIBE_CMD_USAGE = "Usage: hbase backup describe <backup_id>\n  backup_id       Backup image id\n";
    public static final String HISTORY_CMD_USAGE = "Usage: hbase backup history [options]";
    public static final String DELETE_CMD_USAGE = "Usage: hbase backup delete [options]";
    public static final String REPAIR_CMD_USAGE = "Usage: hbase backup repair\n";
    public static final String SET_CMD_USAGE = "Usage: hbase backup set COMMAND [name] [tables]\n  name            Backup set name\n  tables          Comma separated list of tables.\nCOMMAND is one of:\n  add             add tables to a set, create a set if needed\n  remove          remove tables from a set\n  list            list all backup sets in the system\n  describe        describe set\n  delete          delete backup set\n";
    public static final String MERGE_CMD_USAGE = "Usage: hbase backup merge [backup_ids]\n  backup_ids      Comma separated list of backup image ids.\n";
    public static final String USAGE_FOOTER = "";

    private BackupCommands() {
        throw new AssertionError((Object)"Instantiating utility class...");
    }

    public static Command createCommand(Configuration conf, BackupRestoreConstants.BackupCommand type, CommandLine cmdline) {
        Command cmd;
        switch (type) {
            case CREATE: {
                cmd = new CreateCommand(conf, cmdline);
                break;
            }
            case DESCRIBE: {
                cmd = new DescribeCommand(conf, cmdline);
                break;
            }
            case PROGRESS: {
                cmd = new ProgressCommand(conf, cmdline);
                break;
            }
            case DELETE: {
                cmd = new DeleteCommand(conf, cmdline);
                break;
            }
            case HISTORY: {
                cmd = new HistoryCommand(conf, cmdline);
                break;
            }
            case SET: {
                cmd = new BackupSetCommand(conf, cmdline);
                break;
            }
            case REPAIR: {
                cmd = new RepairCommand(conf, cmdline);
                break;
            }
            case MERGE: {
                cmd = new MergeCommand(conf, cmdline);
                break;
            }
            default: {
                cmd = new HelpCommand(conf, cmdline);
            }
        }
        return cmd;
    }

    static int numOfArgs(String[] args) {
        if (args == null) {
            return 0;
        }
        return args.length;
    }

    public static class BackupSetCommand
    extends Command {
        private static final String SET_ADD_CMD = "add";
        private static final String SET_REMOVE_CMD = "remove";
        private static final String SET_DELETE_CMD = "delete";
        private static final String SET_DESCRIBE_CMD = "describe";
        private static final String SET_LIST_CMD = "list";

        BackupSetCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        public void execute() throws IOException {
            if (this.cmdline == null || this.cmdline.getArgs() == null || this.cmdline.getArgs().length < 2) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            String[] args = this.cmdline.getArgs();
            String cmdStr = args[1];
            BackupRestoreConstants.BackupCommand cmd = this.getCommand(cmdStr);
            switch (cmd) {
                case SET_ADD: {
                    this.processSetAdd(args);
                    break;
                }
                case SET_REMOVE: {
                    this.processSetRemove(args);
                    break;
                }
                case SET_DELETE: {
                    this.processSetDelete(args);
                    break;
                }
                case SET_DESCRIBE: {
                    this.processSetDescribe(args);
                    break;
                }
                case SET_LIST: {
                    this.processSetList();
                    break;
                }
            }
        }

        private void processSetList() throws IOException {
            super.execute();
            try (BackupAdminImpl admin = new BackupAdminImpl(this.conn);){
                List<BackupSet> list = admin.listBackupSets();
                for (BackupSet bs : list) {
                    System.out.println(bs);
                }
            }
        }

        private void processSetDescribe(String[] args) throws IOException {
            if (args == null || args.length != 3) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            super.execute();
            String setName = args[2];
            try (BackupSystemTable sysTable = new BackupSystemTable(this.conn);){
                BackupSet set;
                List<TableName> tables = sysTable.describeBackupSet(setName);
                BackupSet backupSet = set = tables == null ? null : new BackupSet(setName, tables);
                if (set == null) {
                    System.out.println("Set '" + setName + "' does not exist.");
                } else {
                    System.out.println(set);
                }
            }
        }

        private void processSetDelete(String[] args) throws IOException {
            if (args == null || args.length != 3) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            super.execute();
            String setName = args[2];
            try (BackupAdminImpl admin = new BackupAdminImpl(this.conn);){
                boolean result = admin.deleteBackupSet(setName);
                if (result) {
                    System.out.println("Delete set " + setName + " OK.");
                } else {
                    System.out.println("Set " + setName + " does not exist");
                }
            }
        }

        private void processSetRemove(String[] args) throws IOException {
            if (args == null || args.length != 4) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            super.execute();
            String setName = args[2];
            String[] tables = args[3].split(",");
            TableName[] tableNames = this.toTableNames(tables);
            try (BackupAdminImpl admin = new BackupAdminImpl(this.conn);){
                admin.removeFromBackupSet(setName, tableNames);
            }
        }

        private TableName[] toTableNames(String[] tables) {
            TableName[] arr = new TableName[tables.length];
            for (int i = 0; i < tables.length; ++i) {
                arr[i] = TableName.valueOf((String)tables[i]);
            }
            return arr;
        }

        private void processSetAdd(String[] args) throws IOException {
            if (args == null || args.length != 4) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            super.execute();
            String setName = args[2];
            TableName[] tableNames = (TableName[])Splitter.on((char)',').splitToStream((CharSequence)args[3]).map(TableName::valueOf).toArray(TableName[]::new);
            try (BackupAdminImpl admin = new BackupAdminImpl(this.conn);){
                admin.addToBackupSet(setName, tableNames);
            }
        }

        private BackupRestoreConstants.BackupCommand getCommand(String cmdStr) throws IOException {
            switch (cmdStr) {
                case "add": {
                    return BackupRestoreConstants.BackupCommand.SET_ADD;
                }
                case "remove": {
                    return BackupRestoreConstants.BackupCommand.SET_REMOVE;
                }
                case "delete": {
                    return BackupRestoreConstants.BackupCommand.SET_DELETE;
                }
                case "describe": {
                    return BackupRestoreConstants.BackupCommand.SET_DESCRIBE;
                }
                case "list": {
                    return BackupRestoreConstants.BackupCommand.SET_LIST;
                }
            }
            System.out.println("ERROR: Unknown command for 'set' :" + cmdStr);
            this.printUsage();
            throw new IOException(BackupCommands.INCORRECT_USAGE);
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.SET_CMD_USAGE);
        }
    }

    public static class HistoryCommand
    extends Command {
        private static final int DEFAULT_HISTORY_LENGTH = 10;

        HistoryCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        public void execute() throws IOException {
            List<BackupInfo> history;
            int n = this.parseHistoryLength();
            final TableName tableName = this.getTableName();
            final String setName = this.getTableSetName();
            BackupInfo.Filter tableNameFilter = new BackupInfo.Filter(){

                @Override
                public boolean apply(BackupInfo info) {
                    if (tableName == null) {
                        return true;
                    }
                    List<TableName> names = info.getTableNames();
                    return names.contains(tableName);
                }
            };
            BackupInfo.Filter tableSetFilter = new BackupInfo.Filter(){

                @Override
                public boolean apply(BackupInfo info) {
                    if (setName == null) {
                        return true;
                    }
                    String backupId = info.getBackupId();
                    return backupId.startsWith(setName);
                }
            };
            Path backupRootPath = this.getBackupRootPath();
            if (backupRootPath == null) {
                super.execute();
                try (BackupSystemTable sysTable = new BackupSystemTable(this.conn);){
                    history = sysTable.getBackupHistory(n, tableNameFilter, tableSetFilter);
                }
            } else {
                history = BackupUtils.getHistory(this.getConf(), n, backupRootPath, tableNameFilter, tableSetFilter);
            }
            for (BackupInfo info : history) {
                System.out.println(info.getShortDescription());
            }
        }

        private Path getBackupRootPath() throws IOException {
            String value = null;
            try {
                value = this.cmdline.getOptionValue("p");
                if (value == null) {
                    return null;
                }
                return new Path(value);
            }
            catch (IllegalArgumentException e) {
                System.out.println("ERROR: Illegal argument for backup root path: " + value);
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
        }

        private TableName getTableName() throws IOException {
            String value = this.cmdline.getOptionValue("t");
            if (value == null) {
                return null;
            }
            try {
                return TableName.valueOf((String)value);
            }
            catch (IllegalArgumentException e) {
                System.out.println("Illegal argument for table name: " + value);
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
        }

        private String getTableSetName() {
            return this.cmdline.getOptionValue("s");
        }

        private int parseHistoryLength() throws IOException {
            String value = this.cmdline.getOptionValue("n");
            try {
                if (value == null) {
                    return 10;
                }
                return Integer.parseInt(value);
            }
            catch (NumberFormatException e) {
                System.out.println("Illegal argument for history length: " + value);
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.HISTORY_CMD_USAGE);
            Options options = new Options();
            options.addOption("n", true, "Number of records of backup history. Default: 10");
            options.addOption("p", true, "Backup destination root directory path");
            options.addOption("t", true, "Table name. If specified, only backup images, which contain this table will be listed.");
            options.addOption("s", true, "Backup set name");
            HelpFormatter helpFormatter = new HelpFormatter();
            helpFormatter.setLeftPadding(2);
            helpFormatter.setDescPadding(8);
            helpFormatter.setWidth(100);
            helpFormatter.setSyntaxPrefix("Options:");
            helpFormatter.printHelp(" ", null, options, BackupCommands.USAGE_FOOTER);
        }
    }

    public static class MergeCommand
    extends Command {
        MergeCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        protected boolean requiresNoActiveSession() {
            return true;
        }

        @Override
        protected boolean requiresConsistentState() {
            return true;
        }

        @Override
        public void execute() throws IOException {
            String[] args;
            super.execute();
            String[] stringArray = args = this.cmdline == null ? null : this.cmdline.getArgs();
            if (args == null || args.length != 2) {
                System.err.println("ERROR: wrong number of arguments: " + (args == null ? null : Integer.valueOf(args.length)));
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            String[] backupIds = args[1].split(",");
            if (backupIds.length < 2) {
                String msg = "ERROR: can not merge a single backup image. Number of images must be greater than 1.";
                System.err.println(msg);
                throw new IOException(msg);
            }
            Configuration conf = this.getConf() != null ? this.getConf() : HBaseConfiguration.create();
            try (Connection conn = ConnectionFactory.createConnection((Configuration)conf);
                 BackupAdminImpl admin = new BackupAdminImpl(conn);){
                admin.mergeBackups(backupIds);
            }
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.MERGE_CMD_USAGE);
        }
    }

    public static class RepairCommand
    extends Command {
        RepairCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        public void execute() throws IOException {
            String[] args;
            super.execute();
            String[] stringArray = args = this.cmdline == null ? null : this.cmdline.getArgs();
            if (args != null && args.length > 1) {
                System.err.println("ERROR: wrong number of arguments: " + args.length);
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            Configuration conf = this.getConf() != null ? this.getConf() : HBaseConfiguration.create();
            try (Connection conn = ConnectionFactory.createConnection((Configuration)conf);
                 BackupSystemTable sysTable = new BackupSystemTable(conn);){
                ArrayList<BackupInfo> list = sysTable.getBackupInfos(BackupInfo.BackupState.RUNNING);
                if (list.size() == 0) {
                    System.out.println("REPAIR status: no failed sessions found. Checking failed delete backup operation ...");
                    this.repairFailedBackupDeletionIfAny(conn, sysTable);
                    RepairCommand.repairFailedBackupMergeIfAny(conn, sysTable);
                    return;
                }
                BackupInfo backupInfo = (BackupInfo)list.get(0);
                backupInfo.setCompleteTs(EnvironmentEdgeManager.currentTime());
                backupInfo.setFailedMsg("REPAIR status: repaired after failure:\n" + backupInfo);
                backupInfo.setState(BackupInfo.BackupState.FAILED);
                String backupFailedData = "BackupId=" + backupInfo.getBackupId() + ",startts=" + backupInfo.getStartTs() + ",failedts=" + backupInfo.getCompleteTs() + ",failedphase=" + (Object)((Object)backupInfo.getPhase()) + ",failedmessage=" + backupInfo.getFailedMsg();
                System.out.println(backupFailedData);
                TableBackupClient.cleanupAndRestoreBackupSystem(conn, backupInfo, conf);
                sysTable.updateBackupInfo(backupInfo);
                sysTable.finishBackupExclusiveOperation();
                System.out.println("REPAIR status: finished repair failed session:\n " + backupInfo);
            }
        }

        private void repairFailedBackupDeletionIfAny(Connection conn, BackupSystemTable sysTable) throws IOException {
            Object[] backupIds = sysTable.getListOfBackupIdsFromDeleteOperation();
            if (backupIds == null || backupIds.length == 0) {
                System.out.println("No failed backup DELETE operation found");
                BackupSystemTable.deleteSnapshot(conn);
                return;
            }
            System.out.println("Found failed DELETE operation for: " + StringUtils.join((Object[])backupIds));
            System.out.println("Running DELETE again ...");
            BackupSystemTable.restoreFromSnapshot(conn);
            sysTable.finishBackupExclusiveOperation();
            try (BackupAdminImpl admin = new BackupAdminImpl(conn);){
                admin.deleteBackups((String[])backupIds);
            }
            System.out.println("DELETE operation finished OK: " + StringUtils.join((Object[])backupIds));
        }

        public static void repairFailedBackupMergeIfAny(Connection conn, BackupSystemTable sysTable) throws IOException {
            Object[] backupIds = sysTable.getListOfBackupIdsFromMergeOperation();
            if (backupIds == null || backupIds.length == 0) {
                System.out.println("No failed backup MERGE operation found");
                BackupSystemTable.deleteSnapshot(conn);
                return;
            }
            System.out.println("Found failed MERGE operation for: " + StringUtils.join((Object[])backupIds));
            BackupInfo bInfo = sysTable.readBackupInfo((String)backupIds[0]);
            String backupRoot = bInfo.getBackupRootDir();
            FileSystem fs = FileSystem.get((URI)new Path(backupRoot).toUri(), (Configuration)new Configuration());
            String backupId = BackupUtils.findMostRecentBackupId((String[])backupIds);
            Path tmpPath = HBackupFileSystem.getBackupTmpDirPathForBackupId(backupRoot, backupId);
            if (fs.exists(tmpPath)) {
                boolean res;
                Path destPath = HBackupFileSystem.getBackupPath(backupRoot, backupId);
                if (!fs.delete(destPath, true)) {
                    System.out.println("Failed to delete " + destPath);
                }
                if (!(res = fs.rename(tmpPath, destPath))) {
                    throw new IOException("MERGE repair: failed  to rename from " + tmpPath + " to " + destPath);
                }
                System.out.println("MERGE repair: renamed from " + tmpPath + " to " + destPath + " res=" + res);
            } else {
                RepairCommand.checkRemoveBackupImages(fs, backupRoot, (String[])backupIds);
            }
            BackupSystemTable.restoreFromSnapshot(conn);
            sysTable.finishBackupExclusiveOperation();
            sysTable.finishMergeOperation();
            System.out.println("MERGE repair operation finished OK: " + StringUtils.join((Object[])backupIds));
        }

        private static void checkRemoveBackupImages(FileSystem fs, String backupRoot, String[] backupIds) throws IOException {
            String mergedBackupId = BackupUtils.findMostRecentBackupId(backupIds);
            for (String backupId : backupIds) {
                Path path;
                if (backupId.equals(mergedBackupId) || !fs.exists(path = HBackupFileSystem.getBackupPath(backupRoot, backupId))) continue;
                if (!fs.delete(path, true)) {
                    System.out.println("MERGE repair removing: " + path + " - FAILED");
                    continue;
                }
                System.out.println("MERGE repair removing: " + path + " - OK");
            }
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.REPAIR_CMD_USAGE);
        }
    }

    public static class DeleteCommand
    extends Command {
        DeleteCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        protected boolean requiresNoActiveSession() {
            return true;
        }

        @Override
        public void execute() throws IOException {
            if (this.cmdline == null || this.cmdline.getArgs() == null || this.cmdline.getArgs().length < 1) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            if (!this.cmdline.hasOption("k") && !this.cmdline.hasOption("l")) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            super.execute();
            if (this.cmdline.hasOption("k")) {
                this.executeDeleteOlderThan(this.cmdline);
            } else if (this.cmdline.hasOption("l")) {
                this.executeDeleteListOfBackups(this.cmdline);
            }
        }

        private void executeDeleteOlderThan(CommandLine cmdline) throws IOException {
            String value = cmdline.getOptionValue("k");
            int days = 0;
            try {
                days = Integer.parseInt(value);
            }
            catch (NumberFormatException e) {
                throw new IOException(value + " is not an integer number");
            }
            final long fdays = days;
            BackupInfo.Filter dateFilter = new BackupInfo.Filter(){

                @Override
                public boolean apply(BackupInfo info) {
                    long currentTime = EnvironmentEdgeManager.currentTime();
                    long maxTsToDelete = currentTime - fdays * 24L * 3600L * 1000L;
                    return info.getCompleteTs() <= maxTsToDelete;
                }
            };
            List<BackupInfo> history = null;
            try (BackupSystemTable sysTable = new BackupSystemTable(this.conn);
                 BackupAdminImpl admin = new BackupAdminImpl(this.conn);){
                history = sysTable.getBackupHistory(-1, dateFilter);
                String[] backupIds = this.convertToBackupIds(history);
                int deleted = admin.deleteBackups(backupIds);
                System.out.println("Deleted " + deleted + " backups. Total older than " + days + " days: " + backupIds.length);
            }
            catch (IOException e) {
                System.err.println("Delete command FAILED. Please run backup repair tool to restore backup system integrity");
                throw e;
            }
        }

        private String[] convertToBackupIds(List<BackupInfo> history) {
            String[] ids = new String[history.size()];
            for (int i = 0; i < ids.length; ++i) {
                ids[i] = history.get(i).getBackupId();
            }
            return ids;
        }

        private void executeDeleteListOfBackups(CommandLine cmdline) throws IOException {
            String value = cmdline.getOptionValue("l");
            String[] backupIds = value.split(",");
            try (BackupAdminImpl admin = new BackupAdminImpl(this.conn);){
                int deleted = admin.deleteBackups(backupIds);
                System.out.println("Deleted " + deleted + " backups. Total requested: " + backupIds.length);
            }
            catch (IOException e) {
                System.err.println("Delete command FAILED. Please run backup repair tool to restore backup system integrity");
                throw e;
            }
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.DELETE_CMD_USAGE);
            Options options = new Options();
            options.addOption("k", true, "Specifies maximum age of backup (in days) to keep during bulk delete");
            options.addOption("l", true, "Backup ids list, comma-separated.");
            HelpFormatter helpFormatter = new HelpFormatter();
            helpFormatter.setLeftPadding(2);
            helpFormatter.setDescPadding(8);
            helpFormatter.setWidth(100);
            helpFormatter.setSyntaxPrefix("Options:");
            helpFormatter.printHelp(" ", null, options, BackupCommands.USAGE_FOOTER);
        }
    }

    public static class ProgressCommand
    extends Command {
        ProgressCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        public void execute() throws IOException {
            String[] args;
            if (this.cmdline == null || this.cmdline.getArgs() == null || this.cmdline.getArgs().length == 1) {
                System.out.println("No backup id was specified, will retrieve the most recent (ongoing) session");
            }
            String[] stringArray = args = this.cmdline == null ? null : this.cmdline.getArgs();
            if (args != null && args.length > 2) {
                System.err.println("ERROR: wrong number of arguments: " + args.length);
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            super.execute();
            String backupId = args == null || args.length <= 1 ? null : args[1];
            try (BackupSystemTable sysTable = new BackupSystemTable(this.conn);){
                int progress;
                BackupInfo info = null;
                if (backupId != null) {
                    info = sysTable.readBackupInfo(backupId);
                } else {
                    ArrayList<BackupInfo> infos = sysTable.getBackupInfos(BackupInfo.BackupState.RUNNING);
                    if (infos != null && infos.size() > 0) {
                        info = (BackupInfo)infos.get(0);
                        backupId = info.getBackupId();
                        System.out.println("Found ongoing session with backupId=" + backupId);
                    }
                }
                int n = progress = info == null ? -1 : info.getProgress();
                if (progress < 0) {
                    if (backupId != null) {
                        System.out.println(BackupCommands.NO_INFO_FOUND + backupId);
                    } else {
                        System.err.println(BackupCommands.NO_ACTIVE_SESSION_FOUND);
                    }
                } else {
                    System.out.println(backupId + " progress=" + progress + "%");
                }
            }
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.PROGRESS_CMD_USAGE);
        }
    }

    public static class DescribeCommand
    extends Command {
        DescribeCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        public void execute() throws IOException {
            if (this.cmdline == null || this.cmdline.getArgs() == null) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            String[] args = this.cmdline.getArgs();
            if (args.length != 2) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            super.execute();
            String backupId = args[1];
            try (BackupSystemTable sysTable = new BackupSystemTable(this.conn);){
                BackupInfo info = sysTable.readBackupInfo(backupId);
                if (info == null) {
                    System.out.println("ERROR: " + backupId + " does not exist");
                    this.printUsage();
                    throw new IOException(BackupCommands.INCORRECT_USAGE);
                }
                System.out.println(info.getShortDescription());
            }
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.DESCRIBE_CMD_USAGE);
        }
    }

    public static class HelpCommand
    extends Command {
        HelpCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        public void execute() throws IOException {
            if (this.cmdline == null) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            String[] args = this.cmdline.getArgs();
            if (args == null || args.length == 0) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            if (args.length != 2) {
                System.out.println("ERROR: Only supports help message of a single command type");
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            String type = args[1];
            if (BackupRestoreConstants.BackupCommand.CREATE.name().equalsIgnoreCase(type)) {
                System.out.println(BackupCommands.CREATE_CMD_USAGE);
            } else if (BackupRestoreConstants.BackupCommand.DESCRIBE.name().equalsIgnoreCase(type)) {
                System.out.println(BackupCommands.DESCRIBE_CMD_USAGE);
            } else if (BackupRestoreConstants.BackupCommand.HISTORY.name().equalsIgnoreCase(type)) {
                System.out.println(BackupCommands.HISTORY_CMD_USAGE);
            } else if (BackupRestoreConstants.BackupCommand.PROGRESS.name().equalsIgnoreCase(type)) {
                System.out.println(BackupCommands.PROGRESS_CMD_USAGE);
            } else if (BackupRestoreConstants.BackupCommand.DELETE.name().equalsIgnoreCase(type)) {
                System.out.println(BackupCommands.DELETE_CMD_USAGE);
            } else if (BackupRestoreConstants.BackupCommand.SET.name().equalsIgnoreCase(type)) {
                System.out.println(BackupCommands.SET_CMD_USAGE);
            } else {
                System.out.println("Unknown command : " + type);
                this.printUsage();
            }
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.USAGE);
        }
    }

    public static class CreateCommand
    extends Command {
        CreateCommand(Configuration conf, CommandLine cmdline) {
            super(conf);
            this.cmdline = cmdline;
        }

        @Override
        protected boolean requiresNoActiveSession() {
            return true;
        }

        @Override
        protected boolean requiresConsistentState() {
            return true;
        }

        @Override
        public void execute() throws IOException {
            int workers;
            String tables;
            if (this.cmdline == null || this.cmdline.getArgs() == null) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            String[] args = this.cmdline.getArgs();
            if (args.length != 3) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            if (!BackupType.FULL.toString().equalsIgnoreCase(args[1]) && !BackupType.INCREMENTAL.toString().equalsIgnoreCase(args[1])) {
                System.out.println("ERROR: invalid backup type: " + args[1]);
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            if (!this.verifyPath(args[2])) {
                System.out.println("ERROR: invalid backup destination: " + args[2]);
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            String targetBackupDir = args[2];
            if (this.isRootFolder(targetBackupDir)) {
                throw new IOException(BackupCommands.TOP_LEVEL_NOT_ALLOWED);
            }
            if (this.cmdline.hasOption("t") && this.cmdline.hasOption("s")) {
                System.out.println("ERROR: You can specify either backup set or list of tables, but not both");
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            super.execute();
            String setName = null;
            if (this.cmdline.hasOption("s")) {
                setName = this.cmdline.getOptionValue("s");
                tables = this.getTablesForSet(setName);
                if (tables == null) {
                    System.out.println("ERROR: Backup set '" + setName + "' is either empty or does not exist");
                    this.printUsage();
                    throw new IOException(BackupCommands.INCORRECT_USAGE);
                }
            } else {
                tables = this.cmdline.getOptionValue("t");
            }
            int bandwidth = this.cmdline.hasOption("b") ? Integer.parseInt(this.cmdline.getOptionValue("b")) : -1;
            int n = workers = this.cmdline.hasOption("w") ? Integer.parseInt(this.cmdline.getOptionValue("w")) : -1;
            if (this.cmdline.hasOption("q")) {
                String queueName = this.cmdline.getOptionValue("q");
                System.setProperty("mapreduce.job.queuename", queueName);
            }
            try (BackupAdminImpl admin = new BackupAdminImpl(this.conn);){
                BackupRequest.Builder builder = new BackupRequest.Builder();
                BackupRequest request = builder.withBackupType(BackupType.valueOf((String)args[1].toUpperCase())).withTableList(tables != null ? Lists.newArrayList((Object[])BackupUtils.parseTableNames(tables)) : null).withTargetRootDir(targetBackupDir).withTotalTasks(workers).withBandwidthPerTasks(bandwidth).withBackupSetName(setName).build();
                String backupId = admin.backupTables(request);
                System.out.println("Backup session " + backupId + " finished. Status: SUCCESS");
            }
            catch (IOException e) {
                System.out.println("Backup session finished. Status: FAILURE");
                throw e;
            }
        }

        private boolean isRootFolder(String targetBackupDir) {
            Path p = new Path(targetBackupDir);
            return p.isRoot();
        }

        private boolean verifyPath(String path) {
            try {
                Path p = new Path(path);
                Configuration conf = this.getConf() != null ? this.getConf() : HBaseConfiguration.create();
                URI uri = p.toUri();
                if (uri.getScheme() == null) {
                    return false;
                }
                FileSystem.get((URI)uri, (Configuration)conf);
                return true;
            }
            catch (Exception e) {
                return false;
            }
        }

        private String getTablesForSet(String name) throws IOException {
            try (BackupSystemTable table = new BackupSystemTable(this.conn);){
                List<TableName> tables = table.describeBackupSet(name);
                if (tables == null) {
                    String string = null;
                    return string;
                }
                String string = StringUtils.join(tables, (String)",");
                return string;
            }
        }

        @Override
        protected void printUsage() {
            System.out.println(BackupCommands.CREATE_CMD_USAGE);
            Options options = new Options();
            options.addOption("w", true, "Number of parallel MapReduce tasks to execute");
            options.addOption("b", true, "Bandwidth per task (MapReduce task) in MB/s");
            options.addOption("s", true, "Backup set to backup, mutually exclusive with -t (table list)");
            options.addOption("t", true, "Table name list, comma-separated.");
            options.addOption("q", true, "Yarn queue name to run backup create command on");
            options.addOption("d", false, "Enable debug loggings");
            HelpFormatter helpFormatter = new HelpFormatter();
            helpFormatter.setLeftPadding(2);
            helpFormatter.setDescPadding(8);
            helpFormatter.setWidth(100);
            helpFormatter.setSyntaxPrefix("Options:");
            helpFormatter.printHelp(" ", null, options, BackupCommands.USAGE_FOOTER);
        }
    }

    public static abstract class Command
    extends Configured {
        CommandLine cmdline;
        Connection conn;

        Command(Configuration conf) {
            if (conf == null) {
                conf = HBaseConfiguration.create();
            }
            this.setConf(conf);
        }

        public void execute() throws IOException {
            Throwable throwable;
            BackupSystemTable table;
            if (this.cmdline.hasOption("h") || this.cmdline.hasOption("help")) {
                this.printUsage();
                throw new IOException(BackupCommands.INCORRECT_USAGE);
            }
            this.conn = ConnectionFactory.createConnection((Configuration)this.getConf());
            if (this.requiresNoActiveSession()) {
                table = new BackupSystemTable(this.conn);
                throwable = null;
                try {
                    ArrayList<BackupInfo> sessions = table.getBackupInfos(BackupInfo.BackupState.RUNNING);
                    if (sessions.size() > 0) {
                        System.err.println("Found backup session in a RUNNING state: ");
                        System.err.println(sessions.get(0));
                        System.err.println("This may indicate that a previous session has failed abnormally.");
                        System.err.println("In this case, backup recovery is recommended.");
                        throw new IOException("Active session found, aborted command execution");
                    }
                }
                catch (Throwable sessions) {
                    throwable = sessions;
                    throw sessions;
                }
                finally {
                    if (table != null) {
                        if (throwable != null) {
                            try {
                                table.close();
                            }
                            catch (Throwable sessions) {
                                throwable.addSuppressed(sessions);
                            }
                        } else {
                            table.close();
                        }
                    }
                }
            }
            if (this.requiresConsistentState()) {
                table = new BackupSystemTable(this.conn);
                throwable = null;
                try {
                    String[] ids = table.getListOfBackupIdsFromDeleteOperation();
                    if (ids != null && ids.length > 0) {
                        System.err.println("Found failed backup DELETE coommand. ");
                        System.err.println("Backup system recovery is required.");
                        throw new IOException("Failed backup DELETE found, aborted command execution");
                    }
                    ids = table.getListOfBackupIdsFromMergeOperation();
                    if (ids != null && ids.length > 0) {
                        System.err.println("Found failed backup MERGE coommand. ");
                        System.err.println("Backup system recovery is required.");
                        throw new IOException("Failed backup MERGE found, aborted command execution");
                    }
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (table != null) {
                        if (throwable != null) {
                            try {
                                table.close();
                            }
                            catch (Throwable throwable3) {
                                throwable.addSuppressed(throwable3);
                            }
                        } else {
                            table.close();
                        }
                    }
                }
            }
        }

        public void finish() throws IOException {
            if (this.conn != null) {
                this.conn.close();
            }
        }

        protected abstract void printUsage();

        protected boolean requiresNoActiveSession() {
            return false;
        }

        protected boolean requiresConsistentState() {
            return false;
        }
    }
}

