/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.backup.impl;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.TreeSet;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.backup.BackupType;
import org.apache.hadoop.hbase.backup.HBackupFileSystem;
import org.apache.hadoop.hbase.backup.RestoreRequest;
import org.apache.hadoop.hbase.backup.impl.BackupManifest;
import org.apache.hadoop.hbase.backup.util.BackupUtils;
import org.apache.hadoop.hbase.backup.util.RestoreTool;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class RestoreTablesClient {
    private static final Logger LOG = LoggerFactory.getLogger(RestoreTablesClient.class);
    private Configuration conf;
    private Connection conn;
    private String backupId;
    private TableName[] sTableArray;
    private TableName[] tTableArray;
    private String backupRootDir;
    private Path restoreRootDir;
    private boolean isOverwrite;

    public RestoreTablesClient(Connection conn, RestoreRequest request) throws IOException {
        this.backupRootDir = request.getBackupRootDir();
        this.backupId = request.getBackupId();
        this.sTableArray = request.getFromTables();
        this.tTableArray = request.getToTables();
        if (this.tTableArray == null || this.tTableArray.length == 0) {
            this.tTableArray = this.sTableArray;
        }
        this.isOverwrite = request.isOverwrite();
        this.conn = conn;
        this.conf = conn.getConfiguration();
        if (request.getRestoreRootDir() != null) {
            this.restoreRootDir = new Path(request.getRestoreRootDir());
        } else {
            FileSystem fs = FileSystem.get((Configuration)this.conf);
            this.restoreRootDir = BackupUtils.getTmpRestoreOutputDir(fs, this.conf);
        }
    }

    private void checkTargetTables(TableName[] tTableArray, boolean isOverwrite) throws IOException {
        ArrayList<TableName> existTableList = new ArrayList<TableName>();
        ArrayList<TableName> disabledTableList = new ArrayList<TableName>();
        try (Admin admin = this.conn.getAdmin();){
            for (TableName tableName : tTableArray) {
                if (admin.tableExists(tableName)) {
                    existTableList.add(tableName);
                    if (!admin.isTableDisabled(tableName)) continue;
                    disabledTableList.add(tableName);
                    continue;
                }
                LOG.info("HBase table " + tableName + " does not exist. It will be created during restore process");
            }
        }
        if (existTableList.size() > 0) {
            if (!isOverwrite) {
                LOG.error("Existing table (" + existTableList + ") found in the restore target, please add \"-o\" as overwrite option in the command if you mean to restore to these existing tables");
                throw new IOException("Existing table found in target while no \"-o\" as overwrite option found");
            }
            if (disabledTableList.size() > 0) {
                LOG.error("Found offline table in the restore target, please enable them before restore with \"-overwrite\" option");
                LOG.info("Offline table list in restore target: " + disabledTableList);
                throw new IOException("Found offline table in the target when restore with \"-overwrite\" option");
            }
        }
    }

    private void restoreImages(BackupManifest.BackupImage[] images, TableName sTable, TableName tTable, boolean truncateIfExists) throws IOException {
        BackupManifest.BackupImage image = images[0];
        String rootDir = image.getRootDir();
        String backupId = image.getBackupId();
        Path backupRoot = new Path(rootDir);
        RestoreTool restoreTool = new RestoreTool(this.conf, backupRoot, this.restoreRootDir, backupId);
        Path tableBackupPath = HBackupFileSystem.getTableBackupPath(sTable, backupRoot, backupId);
        String lastIncrBackupId = images.length == 1 ? null : images[images.length - 1].getBackupId();
        BackupManifest manifest = HBackupFileSystem.getManifest(this.conf, backupRoot, backupId);
        if (manifest.getType() != BackupType.FULL) {
            throw new IOException("Unexpected backup type " + image.getType());
        }
        LOG.info("Restoring '" + sTable + "' to '" + tTable + "' from full backup image " + tableBackupPath.toString());
        this.conf.set("mapreduce.job.name", "Full_Restore-" + backupId + "-" + tTable);
        restoreTool.fullRestoreTable(this.conn, tableBackupPath, sTable, tTable, truncateIfExists, lastIncrBackupId);
        this.conf.unset("mapreduce.job.name");
        if (images.length == 1) {
            return;
        }
        ArrayList<Path> dirList = new ArrayList<Path>();
        for (int i = 1; i < images.length; ++i) {
            BackupManifest.BackupImage im = images[i];
            String fileBackupDir = HBackupFileSystem.getTableBackupDir(im.getRootDir(), im.getBackupId(), sTable);
            List<Path> list = this.getFilesRecursively(fileBackupDir);
            dirList.addAll(list);
        }
        if (dirList.isEmpty()) {
            LOG.warn("Nothing has changed, so there is no need to restore '" + sTable + "'");
            return;
        }
        String dirs = StringUtils.join(dirList, (String)",");
        LOG.info("Restoring '" + sTable + "' to '" + tTable + "' from log dirs: " + dirs);
        Path[] paths = new Path[dirList.size()];
        dirList.toArray(paths);
        this.conf.set("mapreduce.job.name", "Incremental_Restore-" + backupId + "-" + tTable);
        restoreTool.incrementalRestoreTable(this.conn, tableBackupPath, paths, new TableName[]{sTable}, new TableName[]{tTable}, lastIncrBackupId);
        LOG.info(sTable + " has been successfully restored to " + tTable);
    }

    private List<Path> getFilesRecursively(String fileBackupDir) throws IllegalArgumentException, IOException {
        FileSystem fs = FileSystem.get((URI)new Path(fileBackupDir).toUri(), (Configuration)new Configuration());
        ArrayList<Path> list = new ArrayList<Path>();
        RemoteIterator it = fs.listFiles(new Path(fileBackupDir), true);
        while (it.hasNext()) {
            Path p = ((LocatedFileStatus)it.next()).getPath();
            if (!HFile.isHFileFormat((FileSystem)fs, (Path)p)) continue;
            list.add(p);
        }
        return list;
    }

    private void restore(HashMap<TableName, BackupManifest> backupManifestMap, TableName[] sTableArray, TableName[] tTableArray, boolean isOverwrite) throws IOException {
        TreeSet restoreImageSet = new TreeSet();
        for (int i = 0; i < sTableArray.length; ++i) {
            TableName table = sTableArray[i];
            BackupManifest manifest = backupManifestMap.get(table);
            ArrayList<BackupManifest.BackupImage> list = new ArrayList<BackupManifest.BackupImage>();
            list.add(manifest.getBackupImage());
            TreeSet<BackupManifest.BackupImage> set = new TreeSet<BackupManifest.BackupImage>(list);
            ArrayList<BackupManifest.BackupImage> depList = manifest.getDependentListByTable(table);
            set.addAll(depList);
            BackupManifest.BackupImage[] arr = new BackupManifest.BackupImage[set.size()];
            set.toArray(arr);
            this.restoreImages(arr, table, tTableArray[i], isOverwrite);
            restoreImageSet.addAll(list);
            if (restoreImageSet == null || restoreImageSet.isEmpty()) continue;
            LOG.info("Restore includes the following image(s):");
            for (BackupManifest.BackupImage image : restoreImageSet) {
                LOG.info("Backup: " + image.getBackupId() + " " + HBackupFileSystem.getTableBackupDir(image.getRootDir(), image.getBackupId(), table));
            }
        }
        LOG.debug("restoreStage finished");
    }

    static long getTsFromBackupId(String backupId) {
        if (backupId == null) {
            return 0L;
        }
        return Long.parseLong(backupId.substring(backupId.lastIndexOf("_") + 1));
    }

    static boolean withinRange(long a, long lower, long upper) {
        return a >= lower && a <= upper;
    }

    public void execute() throws IOException {
        this.checkTargetTables(this.tTableArray, this.isOverwrite);
        HashMap<TableName, BackupManifest> backupManifestMap = new HashMap<TableName, BackupManifest>();
        Path rootPath = new Path(this.backupRootDir);
        HBackupFileSystem.checkImageManifestExist(backupManifestMap, this.sTableArray, this.conf, rootPath, this.backupId);
        this.restore(backupManifestMap, this.sTableArray, this.tTableArray, this.isOverwrite);
    }
}

