/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.backup.impl;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.backup.BackupInfo;
import org.apache.hadoop.hbase.backup.BackupRequest;
import org.apache.hadoop.hbase.backup.BackupType;
import org.apache.hadoop.hbase.backup.HBackupFileSystem;
import org.apache.hadoop.hbase.backup.impl.BackupManager;
import org.apache.hadoop.hbase.backup.impl.BackupManifest;
import org.apache.hadoop.hbase.backup.impl.BackupSystemTable;
import org.apache.hadoop.hbase.backup.impl.IncrementalBackupManager;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public abstract class TableBackupClient {
    public static final String BACKUP_CLIENT_IMPL_CLASS = "backup.client.impl.class";
    public static final String BACKUP_TEST_MODE_STAGE = "backup.test.mode.stage";
    private static final Logger LOG = LoggerFactory.getLogger(TableBackupClient.class);
    protected Configuration conf;
    protected Connection conn;
    protected String backupId;
    protected List<TableName> tableList;
    protected Map<String, Long> newTimestamps = null;
    protected BackupManager backupManager;
    protected BackupInfo backupInfo;
    protected FileSystem fs;

    public TableBackupClient() {
    }

    public TableBackupClient(Connection conn, String backupId, BackupRequest request) throws IOException {
        this.init(conn, backupId, request);
    }

    public void init(Connection conn, String backupId, BackupRequest request) throws IOException {
        this.backupManager = request.getBackupType() == BackupType.FULL ? new BackupManager(conn, conn.getConfiguration()) : new IncrementalBackupManager(conn, conn.getConfiguration());
        this.backupId = backupId;
        this.tableList = request.getTableList();
        this.conn = conn;
        this.conf = conn.getConfiguration();
        this.fs = CommonFSUtils.getCurrentFileSystem((Configuration)this.conf);
        this.backupInfo = this.backupManager.createBackupInfo(backupId, request.getBackupType(), this.tableList, request.getTargetRootDir(), request.getTotalTasks(), request.getBandwidth());
        if (this.tableList == null || this.tableList.isEmpty()) {
            this.tableList = new ArrayList<TableName>(this.backupInfo.getTables());
        }
        this.backupManager.startBackupSession();
    }

    protected void beginBackup(BackupManager backupManager, BackupInfo backupInfo) throws IOException {
        BackupSystemTable.snapshot(this.conn);
        backupManager.setBackupInfo(backupInfo);
        long startTs = EnvironmentEdgeManager.currentTime();
        backupInfo.setStartTs(startTs);
        backupInfo.setState(BackupInfo.BackupState.RUNNING);
        backupInfo.setPhase(BackupInfo.BackupPhase.REQUEST);
        LOG.info("Backup " + backupInfo.getBackupId() + " started at " + startTs + ".");
        backupManager.updateBackupInfo(backupInfo);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Backup session " + backupInfo.getBackupId() + " has been started.");
        }
    }

    protected String getMessage(Exception e) {
        String msg = e.getMessage();
        if (msg == null || msg.equals("")) {
            msg = e.getClass().getName();
        }
        return msg;
    }

    protected static void deleteSnapshots(Connection conn, BackupInfo backupInfo, Configuration conf) throws IOException {
        LOG.debug("Trying to delete snapshot for full backup.");
        for (String snapshotName : backupInfo.getSnapshotNames()) {
            if (snapshotName == null) continue;
            LOG.debug("Trying to delete snapshot: " + snapshotName);
            try (Admin admin = conn.getAdmin();){
                admin.deleteSnapshot(snapshotName);
            }
            LOG.debug("Deleting the snapshot " + snapshotName + " for backup " + backupInfo.getBackupId() + " succeeded.");
        }
    }

    protected static void cleanupExportSnapshotLog(Configuration conf) throws IOException {
        Path stagingDir;
        FileSystem fs = CommonFSUtils.getCurrentFileSystem((Configuration)conf);
        FileStatus[] files = CommonFSUtils.listStatus((FileSystem)fs, (Path)(stagingDir = new Path(conf.get("snapshot.export.staging.root", fs.getWorkingDirectory().toString()))));
        if (files == null) {
            return;
        }
        for (FileStatus file : files) {
            if (!file.getPath().getName().startsWith("exportSnapshot-")) continue;
            LOG.debug("Delete log files of exporting snapshot: " + file.getPath().getName());
            if (CommonFSUtils.delete((FileSystem)fs, (Path)file.getPath(), (boolean)true)) continue;
            LOG.warn("Can not delete " + file.getPath());
        }
    }

    protected static void cleanupTargetDir(BackupInfo backupInfo, Configuration conf) {
        try {
            LOG.debug("Trying to cleanup up target dir. Current backup phase: " + (Object)((Object)backupInfo.getPhase()));
            if (backupInfo.getPhase().equals((Object)BackupInfo.BackupPhase.SNAPSHOTCOPY) || backupInfo.getPhase().equals((Object)BackupInfo.BackupPhase.INCREMENTAL_COPY) || backupInfo.getPhase().equals((Object)BackupInfo.BackupPhase.STORE_MANIFEST)) {
                FileSystem outputFs = FileSystem.get((URI)new Path(backupInfo.getBackupRootDir()).toUri(), (Configuration)conf);
                for (TableName table : backupInfo.getTables()) {
                    Path tableDir;
                    FileStatus[] backups;
                    Path targetDirPath = new Path(HBackupFileSystem.getTableBackupDir(backupInfo.getBackupRootDir(), backupInfo.getBackupId(), table));
                    if (outputFs.delete(targetDirPath, true)) {
                        LOG.debug("Cleaning up uncompleted backup data at " + targetDirPath.toString() + " done.");
                    } else {
                        LOG.debug("No data has been copied to " + targetDirPath.toString() + ".");
                    }
                    if ((backups = CommonFSUtils.listStatus((FileSystem)outputFs, (Path)(tableDir = targetDirPath.getParent()))) != null && backups.length != 0) continue;
                    outputFs.delete(tableDir, true);
                    LOG.debug(tableDir.toString() + " is empty, remove it.");
                }
            }
        }
        catch (IOException e1) {
            LOG.error("Cleaning up uncompleted backup data of " + backupInfo.getBackupId() + " at " + backupInfo.getBackupRootDir() + " failed due to " + e1.getMessage() + ".");
        }
    }

    protected void failBackup(Connection conn, BackupInfo backupInfo, BackupManager backupManager, Exception e, String msg, BackupType type, Configuration conf) throws IOException {
        try {
            LOG.error(msg + this.getMessage(e), (Throwable)e);
            backupInfo.setCompleteTs(EnvironmentEdgeManager.currentTime());
            backupInfo.setFailedMsg(e.getMessage());
            backupInfo.setState(BackupInfo.BackupState.FAILED);
            String backupFailedData = "BackupId=" + backupInfo.getBackupId() + ",startts=" + backupInfo.getStartTs() + ",failedts=" + backupInfo.getCompleteTs() + ",failedphase=" + (Object)((Object)backupInfo.getPhase()) + ",failedmessage=" + backupInfo.getFailedMsg();
            LOG.error(backupFailedData);
            TableBackupClient.cleanupAndRestoreBackupSystem(conn, backupInfo, conf);
            backupManager.updateBackupInfo(backupInfo);
            backupManager.finishBackupSession();
            LOG.error("Backup " + backupInfo.getBackupId() + " failed.");
        }
        catch (IOException ee) {
            LOG.error("Please run backup repair tool manually to restore backup system integrity");
            throw ee;
        }
    }

    public static void cleanupAndRestoreBackupSystem(Connection conn, BackupInfo backupInfo, Configuration conf) throws IOException {
        BackupType type = backupInfo.getType();
        if (type == BackupType.FULL) {
            TableBackupClient.deleteSnapshots(conn, backupInfo, conf);
            TableBackupClient.cleanupExportSnapshotLog(conf);
        }
        BackupSystemTable.restoreFromSnapshot(conn);
        BackupSystemTable.deleteSnapshot(conn);
        TableBackupClient.cleanupTargetDir(backupInfo, conf);
    }

    protected void addManifest(BackupInfo backupInfo, BackupManager backupManager, BackupType type, Configuration conf) throws IOException {
        BackupManifest manifest;
        backupInfo.setPhase(BackupInfo.BackupPhase.STORE_MANIFEST);
        for (TableName table : backupInfo.getTables()) {
            manifest = new BackupManifest(backupInfo, table);
            ArrayList<BackupManifest.BackupImage> ancestors = backupManager.getAncestors(backupInfo, table);
            for (BackupManifest.BackupImage image : ancestors) {
                manifest.addDependentImage(image);
            }
            if (type == BackupType.INCREMENTAL) {
                HashMap<TableName, Map<String, Long>> tableTimestampMap = new HashMap<TableName, Map<String, Long>>();
                tableTimestampMap.put(table, backupInfo.getIncrTimestampMap().get(table));
                manifest.setIncrTimestampMap(tableTimestampMap);
                ArrayList<BackupManifest.BackupImage> ancestorss = backupManager.getAncestors(backupInfo);
                for (BackupManifest.BackupImage image : ancestorss) {
                    manifest.addDependentImage(image);
                }
            }
            manifest.store(conf);
        }
        if (type == BackupType.INCREMENTAL) {
            manifest = new BackupManifest(backupInfo);
            manifest.setIncrTimestampMap(backupInfo.getIncrTimestampMap());
            ArrayList<BackupManifest.BackupImage> ancestors = backupManager.getAncestors(backupInfo);
            for (BackupManifest.BackupImage image : ancestors) {
                manifest.addDependentImage(image);
            }
            manifest.store(conf);
        }
    }

    protected String obtainBackupMetaDataStr(BackupInfo backupInfo) {
        StringBuilder sb = new StringBuilder();
        sb.append("type=" + backupInfo.getType() + ",tablelist=");
        for (TableName table : backupInfo.getTables()) {
            sb.append(table + ";");
        }
        if (sb.lastIndexOf(";") > 0) {
            sb.delete(sb.lastIndexOf(";"), sb.lastIndexOf(";") + 1);
        }
        sb.append(",targetRootDir=" + backupInfo.getBackupRootDir());
        return sb.toString();
    }

    protected void completeBackup(Connection conn, BackupInfo backupInfo, BackupManager backupManager, BackupType type, Configuration conf) throws IOException {
        backupInfo.setCompleteTs(EnvironmentEdgeManager.currentTime());
        backupInfo.setState(BackupInfo.BackupState.COMPLETE);
        backupInfo.setProgress(100);
        this.addManifest(backupInfo, backupManager, type, conf);
        String backupCompleteData = this.obtainBackupMetaDataStr(backupInfo) + ",startts=" + backupInfo.getStartTs() + ",completets=" + backupInfo.getCompleteTs() + ",bytescopied=" + backupInfo.getTotalBytesCopied();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Backup " + backupInfo.getBackupId() + " finished: " + backupCompleteData);
        }
        if (type == BackupType.FULL) {
            TableBackupClient.deleteSnapshots(conn, backupInfo, conf);
            TableBackupClient.cleanupExportSnapshotLog(conf);
        }
        BackupSystemTable.deleteSnapshot(conn);
        backupManager.updateBackupInfo(backupInfo);
        backupManager.finishBackupSession();
        LOG.info("Backup " + backupInfo.getBackupId() + " completed.");
    }

    public abstract void execute() throws IOException;

    protected Stage getTestStage() {
        return Stage.valueOf("stage_" + this.conf.getInt(BACKUP_TEST_MODE_STAGE, 0));
    }

    protected void failStageIf(Stage stage) throws IOException {
        Stage current = this.getTestStage();
        if (current == stage) {
            throw new IOException("Failed stage " + (Object)((Object)stage) + " in testing");
        }
    }

    public static enum Stage {
        stage_0,
        stage_1,
        stage_2,
        stage_3,
        stage_4;

    }
}

