/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.fs;

import java.io.IOException;
import java.util.Collection;
import java.util.Objects;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HBaseIOException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.util.CommonFSUtils;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.ErasureCodingPolicyInfo;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class ErasureCodingUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ErasureCodingUtils.class);

    private ErasureCodingUtils() {
    }

    public static void verifySupport(Configuration conf, String policy) throws HBaseIOException {
        DistributedFileSystem dfs = ErasureCodingUtils.getDfs(conf);
        ErasureCodingUtils.checkAvailable(dfs, policy);
        Path globalTempDir = new Path(conf.get("hbase.rootdir"), ".tmp");
        Path currentTempDir = ErasureCodingUtils.createTempDir((FileSystem)dfs, globalTempDir);
        try {
            ErasureCodingUtils.setPolicy((FileSystem)dfs, currentTempDir, policy);
            try (FSDataOutputStream out = dfs.create(new Path(currentTempDir, "test.out"));){
                out.writeUTF("Testing " + policy);
            }
        }
        catch (IOException e) {
            throw new DoNotRetryIOException("Failed write test for EC policy. Check cause or logs", e);
        }
        finally {
            try {
                dfs.delete(currentTempDir, true);
            }
            catch (IOException e) {
                LOG.warn("Failed to delete temp path for ec test", (Throwable)e);
            }
        }
    }

    private static Path createTempDir(FileSystem fs, Path tempDir) throws HBaseIOException {
        Path currentTempDir = new Path(tempDir, "ec-test-" + System.currentTimeMillis());
        try {
            fs.mkdirs(currentTempDir);
            fs.deleteOnExit(currentTempDir);
        }
        catch (IOException e) {
            throw new HBaseIOException("Failed to create test dir for EC write test", e);
        }
        return currentTempDir;
    }

    private static void checkAvailable(DistributedFileSystem dfs, String policy) throws HBaseIOException {
        Collection policies;
        try {
            policies = dfs.getAllErasureCodingPolicies();
        }
        catch (IOException e) {
            throw new HBaseIOException("Failed to check for Erasure Coding policy: " + policy, e);
        }
        for (ErasureCodingPolicyInfo policyInfo : policies) {
            if (!policyInfo.getPolicy().getName().equals(policy)) continue;
            if (!policyInfo.isEnabled()) {
                throw new DoNotRetryIOException("Cannot set Erasure Coding policy: " + policy + ". The policy must be enabled, but has state " + policyInfo.getState());
            }
            return;
        }
        throw new DoNotRetryIOException("Cannot set Erasure Coding policy: " + policy + ". Policy not found. Available policies are: " + policies.stream().map(p -> p.getPolicy().getName()).collect(Collectors.joining(", ")));
    }

    public static boolean needsSync(TableDescriptor oldDescriptor, TableDescriptor newDescriptor) {
        String newPolicy = oldDescriptor.getErasureCodingPolicy();
        String oldPolicy = newDescriptor.getErasureCodingPolicy();
        return !Objects.equals(oldPolicy, newPolicy);
    }

    public static void sync(FileSystem fs, Path rootDir, TableDescriptor newDescriptor) throws IOException {
        String newPolicy = newDescriptor.getErasureCodingPolicy();
        if (newPolicy == null) {
            ErasureCodingUtils.unsetPolicy(fs, rootDir, newDescriptor.getTableName());
        } else {
            ErasureCodingUtils.setPolicy(fs, rootDir, newDescriptor.getTableName(), newPolicy);
        }
    }

    public static void setPolicy(FileSystem fs, Path rootDir, TableName tableName, String policy) throws IOException {
        Path path = CommonFSUtils.getTableDir(rootDir, tableName);
        ErasureCodingUtils.setPolicy(fs, path, policy);
    }

    public static void setPolicy(FileSystem fs, Path path, String policy) throws IOException {
        ErasureCodingUtils.getDfs(fs).setErasureCodingPolicy(path, policy);
    }

    public static void unsetPolicy(FileSystem fs, Path rootDir, TableName tableName) throws IOException {
        Path path;
        DistributedFileSystem dfs = ErasureCodingUtils.getDfs(fs);
        if (dfs.getErasureCodingPolicy(path = CommonFSUtils.getTableDir(rootDir, tableName)) == null) {
            LOG.warn("No EC policy set for path {}, nothing to unset", (Object)path);
            return;
        }
        dfs.unsetErasureCodingPolicy(path);
    }

    private static DistributedFileSystem getDfs(Configuration conf) throws HBaseIOException {
        try {
            return ErasureCodingUtils.getDfs(FileSystem.get((Configuration)conf));
        }
        catch (DoNotRetryIOException e) {
            throw e;
        }
        catch (IOException e) {
            throw new HBaseIOException("Failed to get FileSystem from conf", e);
        }
    }

    private static DistributedFileSystem getDfs(FileSystem fs) throws DoNotRetryIOException {
        if (!(fs instanceof DistributedFileSystem)) {
            throw new DoNotRetryIOException("Cannot manage Erasure Coding policy. Erasure Coding is only available on HDFS, but fs is " + fs.getClass().getSimpleName());
        }
        return (DistributedFileSystem)fs;
    }
}

