/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.replication;

import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.ReplicationPeerNotFoundException;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.client.replication.ReplicationPeerConfigUtil;
import org.apache.hadoop.hbase.conf.ConfigurationObserver;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.procedure.MasterProcedureEnv;
import org.apache.hadoop.hbase.master.procedure.ProcedureSyncWait;
import org.apache.hadoop.hbase.procedure2.ProcedureExecutor;
import org.apache.hadoop.hbase.replication.HBaseReplicationEndpoint;
import org.apache.hadoop.hbase.replication.ReplicationEndpoint;
import org.apache.hadoop.hbase.replication.ReplicationException;
import org.apache.hadoop.hbase.replication.ReplicationGroupOffset;
import org.apache.hadoop.hbase.replication.ReplicationPeerConfig;
import org.apache.hadoop.hbase.replication.ReplicationPeerConfigBuilder;
import org.apache.hadoop.hbase.replication.ReplicationPeerDescription;
import org.apache.hadoop.hbase.replication.ReplicationPeerStorage;
import org.apache.hadoop.hbase.replication.ReplicationQueueData;
import org.apache.hadoop.hbase.replication.ReplicationQueueId;
import org.apache.hadoop.hbase.replication.ReplicationQueueStorage;
import org.apache.hadoop.hbase.replication.ReplicationStorageFactory;
import org.apache.hadoop.hbase.replication.ReplicationUtils;
import org.apache.hadoop.hbase.replication.SyncReplicationState;
import org.apache.hadoop.hbase.replication.ZKReplicationQueueStorageForMigration;
import org.apache.hadoop.hbase.shaded.com.google.errorprone.annotations.RestrictedApi;
import org.apache.hadoop.hbase.shaded.org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.shaded.org.apache.zookeeper.KeeperException;
import org.apache.hadoop.hbase.util.FutureUtils;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.wal.AbstractFSWALProvider;
import org.apache.hadoop.hbase.zookeeper.ZKClusterId;
import org.apache.hadoop.hbase.zookeeper.ZKConfig;
import org.apache.hadoop.hbase.zookeeper.ZKWatcher;
import org.apache.hbase.thirdparty.com.google.common.collect.ImmutableMap;
import org.apache.hbase.thirdparty.com.google.common.collect.Maps;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class ReplicationPeerManager
implements ConfigurationObserver {
    private static final Logger LOG = LoggerFactory.getLogger(ReplicationPeerManager.class);
    private volatile ReplicationPeerStorage peerStorage;
    private final ReplicationQueueStorage queueStorage;
    private final ConcurrentMap<String, ReplicationPeerDescription> peers;
    private final ImmutableMap<SyncReplicationState, EnumSet<SyncReplicationState>> allowedTransition = Maps.immutableEnumMap(ImmutableMap.of(SyncReplicationState.ACTIVE, EnumSet.of(SyncReplicationState.DOWNGRADE_ACTIVE, SyncReplicationState.STANDBY), SyncReplicationState.STANDBY, EnumSet.of(SyncReplicationState.DOWNGRADE_ACTIVE), SyncReplicationState.DOWNGRADE_ACTIVE, EnumSet.of(SyncReplicationState.STANDBY, SyncReplicationState.ACTIVE)));
    private final String clusterId;
    private volatile Configuration conf;
    private final FileSystem fs;
    private final ZKWatcher zk;
    private final ReplicationQueueStorageInitializer queueStorageInitializer;

    ReplicationPeerManager(FileSystem fs, ZKWatcher zk, ReplicationPeerStorage peerStorage, ReplicationQueueStorage queueStorage, ConcurrentMap<String, ReplicationPeerDescription> peers, Configuration conf, String clusterId, ReplicationQueueStorageInitializer queueStorageInitializer) {
        this.fs = fs;
        this.zk = zk;
        this.peerStorage = peerStorage;
        this.queueStorage = queueStorage;
        this.peers = peers;
        this.conf = conf;
        this.clusterId = clusterId;
        this.queueStorageInitializer = queueStorageInitializer;
    }

    private void checkQueuesDeleted(String peerId) throws ReplicationException, DoNotRetryIOException {
        List<ReplicationQueueId> queueIds = this.queueStorage.listAllQueueIds(peerId);
        if (!queueIds.isEmpty()) {
            throw new DoNotRetryIOException("There are still " + queueIds.size() + " undeleted queue(s) for peerId: " + peerId + ", first is " + queueIds.get(0));
        }
        if (this.queueStorage.getAllPeersFromHFileRefsQueue().contains(peerId)) {
            throw new DoNotRetryIOException("Undeleted queue for peer " + peerId + " in hfile-refs");
        }
    }

    private void initializeQueueStorage() throws IOException {
        this.queueStorageInitializer.initialize();
    }

    void preAddPeer(String peerId, ReplicationPeerConfig peerConfig) throws ReplicationException, IOException {
        if (peerId.contains("-")) {
            throw new DoNotRetryIOException("Found invalid peer name: " + peerId);
        }
        this.checkPeerConfig(peerConfig);
        if (peerConfig.isSyncReplication()) {
            this.checkSyncReplicationPeerConfigConflict(peerConfig);
        }
        if (this.peers.containsKey(peerId)) {
            throw new DoNotRetryIOException("Replication peer " + peerId + " already exists");
        }
        this.initializeQueueStorage();
        this.checkQueuesDeleted(peerId);
    }

    private ReplicationPeerDescription checkPeerExists(String peerId) throws DoNotRetryIOException {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        if (desc == null) {
            throw new ReplicationPeerNotFoundException(peerId);
        }
        return desc;
    }

    private void checkPeerInDAStateIfSyncReplication(String peerId) throws DoNotRetryIOException {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        if (desc != null && desc.getPeerConfig().isSyncReplication() && !SyncReplicationState.DOWNGRADE_ACTIVE.equals((Object)desc.getSyncReplicationState())) {
            throw new DoNotRetryIOException("Couldn't remove synchronous replication peer with state=" + (Object)((Object)desc.getSyncReplicationState()) + ", Transit the synchronous replication state to be DOWNGRADE_ACTIVE firstly.");
        }
    }

    ReplicationPeerConfig preRemovePeer(String peerId) throws DoNotRetryIOException {
        ReplicationPeerDescription pd = this.checkPeerExists(peerId);
        this.checkPeerInDAStateIfSyncReplication(peerId);
        return pd.getPeerConfig();
    }

    void preEnablePeer(String peerId) throws DoNotRetryIOException {
        ReplicationPeerDescription desc = this.checkPeerExists(peerId);
        if (desc.isEnabled()) {
            throw new DoNotRetryIOException("Replication peer " + peerId + " has already been enabled");
        }
    }

    void preDisablePeer(String peerId) throws DoNotRetryIOException {
        ReplicationPeerDescription desc = this.checkPeerExists(peerId);
        if (!desc.isEnabled()) {
            throw new DoNotRetryIOException("Replication peer " + peerId + " has already been disabled");
        }
    }

    ReplicationPeerDescription preUpdatePeerConfig(String peerId, ReplicationPeerConfig peerConfig) throws DoNotRetryIOException {
        this.checkPeerConfig(peerConfig);
        ReplicationPeerDescription desc = this.checkPeerExists(peerId);
        ReplicationPeerConfig oldPeerConfig = desc.getPeerConfig();
        if (!this.isStringEquals(peerConfig.getClusterKey(), oldPeerConfig.getClusterKey())) {
            throw new DoNotRetryIOException("Changing the cluster key on an existing peer is not allowed. Existing key '" + oldPeerConfig.getClusterKey() + "' for peer " + peerId + " does not match new key '" + peerConfig.getClusterKey() + "'");
        }
        if (!this.isStringEquals(peerConfig.getReplicationEndpointImpl(), oldPeerConfig.getReplicationEndpointImpl())) {
            throw new DoNotRetryIOException("Changing the replication endpoint implementation class on an existing peer is not allowed. Existing class '" + oldPeerConfig.getReplicationEndpointImpl() + "' for peer " + peerId + " does not match new class '" + peerConfig.getReplicationEndpointImpl() + "'");
        }
        if (!this.isStringEquals(peerConfig.getRemoteWALDir(), oldPeerConfig.getRemoteWALDir())) {
            throw new DoNotRetryIOException("Changing the remote wal dir on an existing peer is not allowed. Existing remote wal dir '" + oldPeerConfig.getRemoteWALDir() + "' for peer " + peerId + " does not match new remote wal dir '" + peerConfig.getRemoteWALDir() + "'");
        }
        if (oldPeerConfig.isSyncReplication() && !ReplicationUtils.isNamespacesAndTableCFsEqual(oldPeerConfig, peerConfig)) {
            throw new DoNotRetryIOException("Changing the replicated namespace/table config on a synchronous replication peer(peerId: " + peerId + ") is not allowed.");
        }
        return desc;
    }

    ReplicationPeerDescription preTransitPeerSyncReplicationState(String peerId, SyncReplicationState state) throws DoNotRetryIOException {
        ReplicationPeerDescription desc = this.checkPeerExists(peerId);
        SyncReplicationState fromState = desc.getSyncReplicationState();
        EnumSet<SyncReplicationState> allowedToStates = this.allowedTransition.get((Object)fromState);
        if (allowedToStates == null || !allowedToStates.contains((Object)state)) {
            throw new DoNotRetryIOException("Can not transit current cluster state from " + (Object)((Object)fromState) + " to " + (Object)((Object)state) + " for peer id=" + peerId);
        }
        return desc;
    }

    public void addPeer(String peerId, ReplicationPeerConfig peerConfig, boolean enabled) throws ReplicationException {
        if (this.peers.containsKey(peerId)) {
            return;
        }
        ReplicationPeerConfig copiedPeerConfig = ReplicationPeerConfig.newBuilder(peerConfig = ReplicationPeerConfigUtil.updateReplicationBasePeerConfigs(this.conf, peerConfig)).build();
        SyncReplicationState syncReplicationState = copiedPeerConfig.isSyncReplication() ? SyncReplicationState.DOWNGRADE_ACTIVE : SyncReplicationState.NONE;
        this.peerStorage.addPeer(peerId, copiedPeerConfig, enabled, syncReplicationState);
        this.peers.put(peerId, new ReplicationPeerDescription(peerId, enabled, copiedPeerConfig, syncReplicationState));
    }

    public void removePeer(String peerId) throws ReplicationException {
        if (!this.peers.containsKey(peerId)) {
            return;
        }
        this.peerStorage.removePeer(peerId);
        this.peers.remove(peerId);
    }

    private void setPeerState(String peerId, boolean enabled) throws ReplicationException {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        if (desc.isEnabled() == enabled) {
            return;
        }
        this.peerStorage.setPeerState(peerId, enabled);
        this.peers.put(peerId, new ReplicationPeerDescription(peerId, enabled, desc.getPeerConfig(), desc.getSyncReplicationState()));
    }

    public boolean getPeerState(String peerId) throws ReplicationException {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        if (desc != null) {
            return desc.isEnabled();
        }
        throw new ReplicationException("Replication Peer of " + peerId + " does not exist.");
    }

    public void enablePeer(String peerId) throws ReplicationException {
        this.setPeerState(peerId, true);
    }

    public void disablePeer(String peerId) throws ReplicationException {
        this.setPeerState(peerId, false);
    }

    public void updatePeerConfig(String peerId, ReplicationPeerConfig peerConfig) throws ReplicationException {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        ReplicationPeerConfig oldPeerConfig = desc.getPeerConfig();
        ReplicationPeerConfigBuilder newPeerConfigBuilder = ReplicationPeerConfig.newBuilder(peerConfig);
        newPeerConfigBuilder.putAllConfiguration(oldPeerConfig.getConfiguration());
        newPeerConfigBuilder.putAllConfiguration(peerConfig.getConfiguration());
        ReplicationPeerConfig newPeerConfig = newPeerConfigBuilder.build();
        this.peerStorage.updatePeerConfig(peerId, newPeerConfig);
        this.peers.put(peerId, new ReplicationPeerDescription(peerId, desc.isEnabled(), newPeerConfig, desc.getSyncReplicationState()));
    }

    public List<ReplicationPeerDescription> listPeers(Pattern pattern) {
        if (pattern == null) {
            return new ArrayList<ReplicationPeerDescription>(this.peers.values());
        }
        return this.peers.values().stream().filter(r -> pattern.matcher(r.getPeerId()).matches()).collect(Collectors.toList());
    }

    public Optional<ReplicationPeerConfig> getPeerConfig(String peerId) {
        ReplicationPeerDescription desc = (ReplicationPeerDescription)this.peers.get(peerId);
        return desc != null ? Optional.of(desc.getPeerConfig()) : Optional.empty();
    }

    void removeAllLastPushedSeqIds(String peerId) throws ReplicationException {
        this.queueStorage.removeLastSequenceIds(peerId);
    }

    public void setPeerNewSyncReplicationState(String peerId, SyncReplicationState state) throws ReplicationException {
        this.peerStorage.setPeerNewSyncReplicationState(peerId, state);
    }

    public void transitPeerSyncReplicationState(String peerId, SyncReplicationState newState) throws ReplicationException {
        ReplicationPeerDescription desc;
        if (this.peerStorage.getPeerNewSyncReplicationState(peerId) != SyncReplicationState.NONE) {
            this.peerStorage.transitPeerSyncReplicationState(peerId);
        }
        if ((desc = (ReplicationPeerDescription)this.peers.get(peerId)).getSyncReplicationState() != newState) {
            this.peers.put(peerId, new ReplicationPeerDescription(peerId, desc.isEnabled(), desc.getPeerConfig(), newState));
        }
    }

    public void removeAllQueues(String peerId) throws ReplicationException {
        this.queueStorage.removeAllQueues(peerId);
        this.queueStorage.removeAllQueues(peerId);
    }

    public void removeAllQueuesAndHFileRefs(String peerId) throws ReplicationException {
        this.removeAllQueues(peerId);
        this.queueStorage.removePeerFromHFileRefs(peerId);
    }

    private void checkPeerConfig(ReplicationPeerConfig peerConfig) throws DoNotRetryIOException {
        String replicationEndpointImpl = peerConfig.getReplicationEndpointImpl();
        ReplicationEndpoint endpoint = null;
        if (!StringUtils.isBlank(replicationEndpointImpl)) {
            try {
                endpoint = Class.forName(replicationEndpointImpl).asSubclass(ReplicationEndpoint.class).getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
            }
            catch (Throwable e) {
                throw new DoNotRetryIOException("Can not instantiate configured replication endpoint class=" + replicationEndpointImpl, e);
            }
        }
        if (endpoint == null || endpoint instanceof HBaseReplicationEndpoint) {
            this.checkClusterKey(peerConfig.getClusterKey());
            if (endpoint == null || !endpoint.canReplicateToSameCluster()) {
                this.checkSameClusterKey(peerConfig.getClusterKey());
            }
        }
        if (peerConfig.replicateAllUserTables()) {
            if (peerConfig.getNamespaces() != null && !peerConfig.getNamespaces().isEmpty() || peerConfig.getTableCFsMap() != null && !peerConfig.getTableCFsMap().isEmpty()) {
                throw new DoNotRetryIOException("Need clean namespaces or table-cfs config firstly when you want replicate all cluster");
            }
            this.checkNamespacesAndTableCfsConfigConflict(peerConfig.getExcludeNamespaces(), peerConfig.getExcludeTableCFsMap());
        } else {
            if (peerConfig.getExcludeNamespaces() != null && !peerConfig.getExcludeNamespaces().isEmpty() || peerConfig.getExcludeTableCFsMap() != null && !peerConfig.getExcludeTableCFsMap().isEmpty()) {
                throw new DoNotRetryIOException("Need clean exclude-namespaces or exclude-table-cfs config firstly when replicate_all flag is false");
            }
            this.checkNamespacesAndTableCfsConfigConflict(peerConfig.getNamespaces(), peerConfig.getTableCFsMap());
        }
        if (peerConfig.isSyncReplication()) {
            this.checkPeerConfigForSyncReplication(peerConfig);
        }
        this.checkConfiguredWALEntryFilters(peerConfig);
    }

    private void checkPeerConfigForSyncReplication(ReplicationPeerConfig peerConfig) throws DoNotRetryIOException {
        if (peerConfig.replicateAllUserTables()) {
            throw new DoNotRetryIOException("Only support replicated table config for sync replication peer");
        }
        if (peerConfig.getNamespaces() != null && !peerConfig.getNamespaces().isEmpty()) {
            throw new DoNotRetryIOException("Only support replicated table config for sync replication peer");
        }
        if (peerConfig.getTableCFsMap() == null || peerConfig.getTableCFsMap().isEmpty()) {
            throw new DoNotRetryIOException("Need config replicated tables for sync replication peer");
        }
        for (List<String> cfs : peerConfig.getTableCFsMap().values()) {
            if (cfs == null || cfs.isEmpty()) continue;
            throw new DoNotRetryIOException("Only support replicated table config for sync replication peer");
        }
        Path remoteWALDir = new Path(peerConfig.getRemoteWALDir());
        if (!remoteWALDir.isAbsolute()) {
            throw new DoNotRetryIOException("The remote WAL directory " + peerConfig.getRemoteWALDir() + " is not absolute");
        }
        URI remoteWALDirUri = remoteWALDir.toUri();
        if (remoteWALDirUri.getScheme() == null || remoteWALDirUri.getAuthority() == null) {
            throw new DoNotRetryIOException("The remote WAL directory " + peerConfig.getRemoteWALDir() + " is not qualified, you must provide scheme and authority");
        }
    }

    private void checkSyncReplicationPeerConfigConflict(ReplicationPeerConfig peerConfig) throws DoNotRetryIOException {
        for (TableName tableName : peerConfig.getTableCFsMap().keySet()) {
            for (Map.Entry entry : this.peers.entrySet()) {
                ReplicationPeerConfig rpc = ((ReplicationPeerDescription)entry.getValue()).getPeerConfig();
                if (!rpc.isSyncReplication() || !rpc.getTableCFsMap().containsKey(tableName)) continue;
                throw new DoNotRetryIOException("Table " + tableName + " has been replicated by peer " + (String)entry.getKey());
            }
        }
    }

    private void checkNamespacesAndTableCfsConfigConflict(Set<String> namespaces, Map<TableName, ? extends Collection<String>> tableCfs) throws DoNotRetryIOException {
        if (namespaces == null || namespaces.isEmpty()) {
            return;
        }
        if (tableCfs == null || tableCfs.isEmpty()) {
            return;
        }
        for (Map.Entry<TableName, ? extends Collection<String>> entry : tableCfs.entrySet()) {
            TableName table = entry.getKey();
            if (!namespaces.contains(table.getNamespaceAsString())) continue;
            throw new DoNotRetryIOException("Table-cfs " + table + " is conflict with namespaces " + table.getNamespaceAsString() + " in peer config");
        }
    }

    private void checkConfiguredWALEntryFilters(ReplicationPeerConfig peerConfig) throws DoNotRetryIOException {
        String filterCSV = peerConfig.getConfiguration().get("hbase.replication.source.custom.walentryfilters");
        if (filterCSV != null && !filterCSV.isEmpty()) {
            String[] filters;
            for (String filter : filters = filterCSV.split(",")) {
                try {
                    Class.forName(filter).getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                }
                catch (Exception e) {
                    throw new DoNotRetryIOException("Configured WALEntryFilter " + filter + " could not be created. Failing add/update peer operation.", e);
                }
            }
        }
    }

    private void checkClusterKey(String clusterKey) throws DoNotRetryIOException {
        try {
            ZKConfig.validateClusterKey(clusterKey);
        }
        catch (IOException e) {
            throw new DoNotRetryIOException("Invalid cluster key: " + clusterKey, e);
        }
    }

    private void checkSameClusterKey(String clusterKey) throws DoNotRetryIOException {
        String peerClusterId = "";
        try {
            Configuration peerConf = HBaseConfiguration.createClusterConf(this.conf, clusterKey);
            try (ZKWatcher zkWatcher = new ZKWatcher(peerConf, this + "check-peer-cluster-id", null);){
                peerClusterId = ZKClusterId.readClusterIdZNode(zkWatcher);
            }
        }
        catch (IOException | KeeperException e) {
            throw new DoNotRetryIOException("Can't get peerClusterId for clusterKey=" + clusterKey, e);
        }
        if (this.clusterId.equals(peerClusterId)) {
            throw new DoNotRetryIOException("Invalid cluster key: " + clusterKey + ", should not replicate to itself for HBaseInterClusterReplicationEndpoint");
        }
    }

    public List<String> getSerialPeerIdsBelongsTo(TableName tableName) {
        return this.peers.values().stream().filter(p -> p.getPeerConfig().isSerial()).filter(p -> p.getPeerConfig().needToReplicate(tableName)).map(p -> p.getPeerId()).collect(Collectors.toList());
    }

    @RestrictedApi(explanation="Should only be called in tests", link="", allowedOnPath=".*/src/test/.*")
    public ReplicationPeerStorage getPeerStorage() {
        return this.peerStorage;
    }

    public ReplicationQueueStorage getQueueStorage() {
        return this.queueStorage;
    }

    private static Pair<ReplicationQueueStorage, ReplicationQueueStorageInitializer> createReplicationQueueStorage(final MasterServices services) throws IOException {
        Configuration conf = services.getConfiguration();
        final TableName replicationQueueTableName = TableName.valueOf(conf.get("hbase.replication.queue.table.name", ReplicationStorageFactory.REPLICATION_QUEUE_TABLE_NAME_DEFAULT.getNameAsString()));
        ReplicationQueueStorageInitializer initializer = services.getTableDescriptors().exists(replicationQueueTableName) ? () -> {} : new ReplicationQueueStorageInitializer(){
            private volatile boolean created = false;

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void initialize() throws IOException {
                if (this.created) {
                    return;
                }
                1 var1_1 = this;
                synchronized (var1_1) {
                    if (this.created) {
                        return;
                    }
                    if (services.getTableDescriptors().exists(replicationQueueTableName)) {
                        this.created = true;
                        return;
                    }
                    long procId = services.createSystemTable(ReplicationStorageFactory.createReplicationQueueTableDescriptor(replicationQueueTableName));
                    ProcedureExecutor<MasterProcedureEnv> procExec = services.getMasterProcedureExecutor();
                    ProcedureSyncWait.waitFor(procExec.getEnvironment(), TimeUnit.MINUTES.toMillis(1L), "Creating table " + replicationQueueTableName, () -> procExec.isFinished(procId));
                }
            }
        };
        return Pair.newPair(ReplicationStorageFactory.getReplicationQueueStorage(services.getConnection(), conf, replicationQueueTableName), initializer);
    }

    public static ReplicationPeerManager create(MasterServices services, String clusterId) throws ReplicationException, IOException {
        Configuration conf = services.getConfiguration();
        FileSystem fs = services.getMasterFileSystem().getFileSystem();
        ZKWatcher zk = services.getZooKeeper();
        final ReplicationPeerStorage peerStorage = ReplicationStorageFactory.getReplicationPeerStorage(fs, zk, conf);
        Pair<ReplicationQueueStorage, ReplicationQueueStorageInitializer> pair = ReplicationPeerManager.createReplicationQueueStorage(services);
        final ReplicationQueueStorage queueStorage = pair.getFirst();
        ConcurrentHashMap<String, ReplicationPeerDescription> peers = new ConcurrentHashMap<String, ReplicationPeerDescription>();
        for (final String peerId : peerStorage.listPeerIds()) {
            ReplicationPeerConfig peerConfig = peerStorage.getPeerConfig(peerId);
            if ("org.apache.hadoop.hbase.replication.regionserver.RegionReplicaReplicationEndpoint".equals(peerConfig.getReplicationEndpointImpl())) {
                LOG.info("Legacy region replication peer found, removing: {}", (Object)peerConfig);
                new Thread("Remove legacy replication peer " + peerId){

                    @Override
                    public void run() {
                        try {
                            queueStorage.removeAllQueues(peerId);
                            queueStorage.removeAllQueues(peerId);
                            peerStorage.removePeer(peerId);
                        }
                        catch (Exception e) {
                            LOG.warn("Failed to delete legacy replication peer {}", (Object)peerId);
                        }
                    }
                }.start();
                continue;
            }
            peerConfig = ReplicationPeerConfigUtil.updateReplicationBasePeerConfigs(conf, peerConfig);
            peerStorage.updatePeerConfig(peerId, peerConfig);
            boolean enabled = peerStorage.isPeerEnabled(peerId);
            SyncReplicationState state = peerStorage.getPeerSyncReplicationState(peerId);
            peers.put(peerId, new ReplicationPeerDescription(peerId, enabled, peerConfig, state));
        }
        return new ReplicationPeerManager(fs, zk, peerStorage, queueStorage, peers, conf, clusterId, pair.getSecond());
    }

    private boolean isStringEquals(String s1, String s2) {
        if (StringUtils.isBlank(s1)) {
            return StringUtils.isBlank(s2);
        }
        return s1.equals(s2);
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
        this.conf = conf;
        this.peerStorage = ReplicationStorageFactory.getReplicationPeerStorage(this.fs, this.zk, conf);
    }

    private ReplicationQueueData convert(ZKReplicationQueueStorageForMigration.ZkReplicationQueueData zkData) {
        HashMap groupOffsets = new HashMap();
        zkData.getWalOffsets().forEach((wal, offset) -> {
            String walGroup = AbstractFSWALProvider.getWALPrefixFromWALName(wal);
            groupOffsets.compute(walGroup, (k, oldOffset) -> {
                if (oldOffset == null) {
                    return new ReplicationGroupOffset((String)wal, (long)offset);
                }
                long oldWalTs = AbstractFSWALProvider.getTimestamp(oldOffset.getWal());
                long walTs = AbstractFSWALProvider.getTimestamp(wal);
                if (walTs < oldWalTs) {
                    return new ReplicationGroupOffset((String)wal, (long)offset);
                }
                return oldOffset;
            });
        });
        return new ReplicationQueueData(zkData.getQueueId(), ImmutableMap.copyOf(groupOffsets));
    }

    private void migrateQueues(ZKReplicationQueueStorageForMigration oldQueueStorage) throws Exception {
        ZKReplicationQueueStorageForMigration.MigrationIterator<Pair<ServerName, List<ZKReplicationQueueStorageForMigration.ZkReplicationQueueData>>> iter = oldQueueStorage.listAllQueues();
        Pair<ServerName, List<ZKReplicationQueueStorageForMigration.ZkReplicationQueueData>> pair;
        while ((pair = iter.next()) != null) {
            this.queueStorage.batchUpdateQueues(pair.getFirst(), pair.getSecond().stream().filter(data -> this.peers.containsKey(data.getQueueId().getPeerId())).map(this::convert).collect(Collectors.toList()));
        }
        return;
    }

    private void migrateLastPushedSeqIds(ZKReplicationQueueStorageForMigration oldQueueStorage) throws Exception {
        ZKReplicationQueueStorageForMigration.MigrationIterator<List<ZKReplicationQueueStorageForMigration.ZkLastPushedSeqId>> iter = oldQueueStorage.listAllLastPushedSeqIds();
        List<ZKReplicationQueueStorageForMigration.ZkLastPushedSeqId> list;
        while ((list = iter.next()) != null) {
            this.queueStorage.batchUpdateLastSequenceIds(list.stream().filter(data -> this.peers.containsKey(data.getPeerId())).collect(Collectors.toList()));
        }
        return;
    }

    private void migrateHFileRefs(ZKReplicationQueueStorageForMigration oldQueueStorage) throws Exception {
        ZKReplicationQueueStorageForMigration.MigrationIterator<Pair<String, List<String>>> iter = oldQueueStorage.listAllHFileRefs();
        Pair<String, List<String>> pair;
        while ((pair = iter.next()) != null) {
            if (!this.peers.containsKey(pair.getFirst())) continue;
            this.queueStorage.batchUpdateHFileRefs(pair.getFirst(), pair.getSecond());
        }
        return;
    }

    private CompletableFuture<?> runAsync(ExceptionalRunnable task, ExecutorService executor) {
        CompletableFuture future = new CompletableFuture();
        executor.execute(() -> {
            try {
                task.run();
                future.complete(null);
            }
            catch (Exception e) {
                future.completeExceptionally(e);
            }
        });
        return future;
    }

    CompletableFuture<Void> migrateQueuesFromZk(ZKWatcher zookeeper, ExecutorService executor) {
        try {
            this.initializeQueueStorage();
        }
        catch (IOException e) {
            return FutureUtils.failedFuture(e);
        }
        ZKReplicationQueueStorageForMigration oldStorage = new ZKReplicationQueueStorageForMigration(zookeeper, this.conf);
        return CompletableFuture.allOf(this.runAsync(() -> this.migrateQueues(oldStorage), executor), this.runAsync(() -> this.migrateLastPushedSeqIds(oldStorage), executor), this.runAsync(() -> this.migrateHFileRefs(oldStorage), executor));
    }

    private static interface ExceptionalRunnable {
        public void run() throws Exception;
    }

    @FunctionalInterface
    static interface ReplicationQueueStorageInitializer {
        public void initialize() throws IOException;
    }
}

