/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.spdy.server.http;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import org.eclipse.jetty.spdy.api.Stream;
import org.eclipse.jetty.spdy.http.HTTPSPDYHeader;
import org.eclipse.jetty.spdy.server.http.PushStrategy;
import org.eclipse.jetty.util.Fields;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class ReferrerPushStrategy
implements PushStrategy {
    private static final Logger LOG = Log.getLogger(ReferrerPushStrategy.class);
    private final ConcurrentMap<String, MainResource> mainResources = new ConcurrentHashMap<String, MainResource>();
    private final Set<Pattern> pushRegexps = new HashSet<Pattern>();
    private final Set<String> pushContentTypes = new HashSet<String>();
    private final Set<Pattern> allowedPushOrigins = new HashSet<Pattern>();
    private final Set<Pattern> userAgentBlacklist = new HashSet<Pattern>();
    private volatile int maxAssociatedResources = 32;
    private volatile int referrerPushPeriod = 5000;

    public ReferrerPushStrategy() {
        List<String> defaultPushRegexps = Arrays.asList(".*\\.css", ".*\\.js", ".*\\.png", ".*\\.jpeg", ".*\\.jpg", ".*\\.gif", ".*\\.ico");
        this.addPushRegexps(defaultPushRegexps);
        List<String> defaultPushContentTypes = Arrays.asList("text/css", "text/javascript", "application/javascript", "application/x-javascript", "image/png", "image/x-png", "image/jpeg", "image/gif", "image/x-icon", "image/vnd.microsoft.icon");
        this.pushContentTypes.addAll(defaultPushContentTypes);
    }

    public void setPushRegexps(List<String> pushRegexps) {
        pushRegexps.clear();
        this.addPushRegexps(pushRegexps);
    }

    private void addPushRegexps(List<String> pushRegexps) {
        for (String pushRegexp : pushRegexps) {
            this.pushRegexps.add(Pattern.compile(pushRegexp));
        }
    }

    public void setPushContentTypes(List<String> pushContentTypes) {
        pushContentTypes.clear();
        pushContentTypes.addAll(pushContentTypes);
    }

    public void setAllowedPushOrigins(List<String> allowedPushOrigins) {
        allowedPushOrigins.clear();
        for (String allowedPushOrigin : allowedPushOrigins) {
            this.allowedPushOrigins.add(Pattern.compile(allowedPushOrigin.replace(".", "\\.").replace("*", ".*")));
        }
    }

    public void setUserAgentBlacklist(List<String> userAgentPatterns) {
        this.userAgentBlacklist.clear();
        for (String userAgentPattern : userAgentPatterns) {
            this.userAgentBlacklist.add(Pattern.compile(userAgentPattern));
        }
    }

    public void setMaxAssociatedResources(int maxAssociatedResources) {
        this.maxAssociatedResources = maxAssociatedResources;
    }

    public void setReferrerPushPeriod(int referrerPushPeriod) {
        this.referrerPushPeriod = referrerPushPeriod;
    }

    public Set<Pattern> getPushRegexps() {
        return this.pushRegexps;
    }

    public Set<String> getPushContentTypes() {
        return this.pushContentTypes;
    }

    public Set<Pattern> getAllowedPushOrigins() {
        return this.allowedPushOrigins;
    }

    public Set<Pattern> getUserAgentBlacklist() {
        return this.userAgentBlacklist;
    }

    public int getMaxAssociatedResources() {
        return this.maxAssociatedResources;
    }

    public int getReferrerPushPeriod() {
        return this.referrerPushPeriod;
    }

    @Override
    public Set<String> apply(Stream stream, Fields requestHeaders, Fields responseHeaders) {
        Set<String> result = Collections.emptySet();
        short version = stream.getSession().getVersion();
        if (!this.isIfModifiedSinceHeaderPresent(requestHeaders) && this.isValidMethod(requestHeaders.get(HTTPSPDYHeader.METHOD.name(version)).getValue()) && !this.isUserAgentBlacklisted(requestHeaders)) {
            Fields.Field referrerHeader;
            String scheme = requestHeaders.get(HTTPSPDYHeader.SCHEME.name(version)).getValue();
            String host = requestHeaders.get(HTTPSPDYHeader.HOST.name(version)).getValue();
            String origin = scheme + "://" + host;
            String url = requestHeaders.get(HTTPSPDYHeader.URI.name(version)).getValue();
            String absoluteURL = origin + url;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Applying push strategy for {}", new Object[]{absoluteURL});
            }
            if (this.isMainResource(url, responseHeaders)) {
                MainResource mainResource = this.getOrCreateMainResource(absoluteURL);
                result = mainResource.getResources();
            } else if (this.isPushResource(url, responseHeaders) && (referrerHeader = requestHeaders.get("referer")) != null) {
                Set<String> pushResources;
                String referrer = referrerHeader.getValue();
                MainResource mainResource = (MainResource)this.mainResources.get(referrer);
                if (mainResource == null) {
                    mainResource = this.getOrCreateMainResource(referrer);
                }
                if (!(pushResources = mainResource.getResources()).contains(url)) {
                    mainResource.addResource(url, origin, referrer);
                } else {
                    result = this.getPushResources(absoluteURL);
                }
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Pushing {} resources for {}: {}", new Object[]{result.size(), absoluteURL, result});
            }
        }
        return result;
    }

    private Set<String> getPushResources(String absoluteURL) {
        Set<String> result = Collections.emptySet();
        if (this.mainResources.get(absoluteURL) != null) {
            result = ((MainResource)this.mainResources.get(absoluteURL)).getResources();
        }
        return result;
    }

    private MainResource getOrCreateMainResource(String absoluteURL) {
        MainResource mainResource = (MainResource)this.mainResources.get(absoluteURL);
        if (mainResource == null) {
            MainResource value;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Creating new main resource for {}", new Object[]{absoluteURL});
            }
            if ((mainResource = this.mainResources.putIfAbsent(absoluteURL, value = new MainResource(absoluteURL))) == null) {
                mainResource = value;
            }
        }
        return mainResource;
    }

    private boolean isIfModifiedSinceHeaderPresent(Fields headers) {
        return headers.get("if-modified-since") != null;
    }

    private boolean isValidMethod(String method) {
        return "GET".equalsIgnoreCase(method);
    }

    private boolean isMainResource(String url, Fields responseHeaders) {
        return !this.isPushResource(url, responseHeaders);
    }

    public boolean isUserAgentBlacklisted(Fields headers) {
        Fields.Field userAgentHeader = headers.get("user-agent");
        if (userAgentHeader != null) {
            for (Pattern userAgentPattern : this.userAgentBlacklist) {
                if (!userAgentPattern.matcher(userAgentHeader.getValue()).matches()) continue;
                return true;
            }
        }
        return false;
    }

    private boolean isPushResource(String url, Fields responseHeaders) {
        for (Pattern pushRegexp : this.pushRegexps) {
            if (!pushRegexp.matcher(url).matches()) continue;
            Fields.Field header = responseHeaders.get("content-type");
            if (header == null) {
                return true;
            }
            String contentType = header.getValue().toLowerCase(Locale.ENGLISH);
            for (String pushContentType : this.pushContentTypes) {
                if (!contentType.startsWith(pushContentType)) continue;
                return true;
            }
        }
        return false;
    }

    private class MainResource {
        private final String name;
        private final CopyOnWriteArraySet<String> resources = new CopyOnWriteArraySet();
        private final AtomicLong firstResourceAdded = new AtomicLong(-1L);

        private MainResource(String name) {
            this.name = name;
        }

        public boolean addResource(String url, String origin, String referrer) {
            this.firstResourceAdded.compareAndSet(-1L, System.nanoTime());
            long delay = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - this.firstResourceAdded.get());
            if (!referrer.startsWith(origin) && !this.isPushOriginAllowed(origin)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Skipped store of push metadata {} for {}: Origin: {} doesn't match or origin not allowed", new Object[]{url, this.name, origin});
                }
                return false;
            }
            if (this.resources.size() >= ReferrerPushStrategy.this.maxAssociatedResources) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Skipped store of push metadata {} for {}: max associated resources ({}) reached", new Object[]{url, this.name, ReferrerPushStrategy.this.maxAssociatedResources});
                }
                return false;
            }
            if (delay > (long)ReferrerPushStrategy.this.referrerPushPeriod) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Delay: {}ms longer than referrerPushPeriod ({}ms). Not adding resource: {} for: {}", new Object[]{delay, ReferrerPushStrategy.this.referrerPushPeriod, url, this.name});
                }
                return false;
            }
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding: {} to: {} with delay: {}ms.", new Object[]{url, this, delay});
            }
            this.resources.add(url);
            return true;
        }

        public Set<String> getResources() {
            return Collections.unmodifiableSet(this.resources);
        }

        public String toString() {
            return String.format("%s@%x{name=%s,resources=%s}", this.getClass().getSimpleName(), this.hashCode(), this.name, this.resources);
        }

        private boolean isPushOriginAllowed(String origin) {
            for (Pattern allowedPushOrigin : ReferrerPushStrategy.this.allowedPushOrigins) {
                if (!allowedPushOrigin.matcher(origin).matches()) continue;
                return true;
            }
            return false;
        }
    }
}

