/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hop.databases.doris;

import org.apache.hop.core.Const;
import org.apache.hop.core.database.DatabaseMeta;
import org.apache.hop.core.database.DatabaseMetaPlugin;
import org.apache.hop.core.gui.plugin.GuiPlugin;
import org.apache.hop.core.row.IValueMeta;
import org.apache.hop.databases.mysql.MySqlDatabaseMeta;

@DatabaseMetaPlugin(type = "DORIS", typeDescription = "Apache Doris", documentationUrl = "/database/databases/doris.html")
@GuiPlugin(id = "GUI-DorisDatabaseMeta")
public class DorisDatabaseMeta extends MySqlDatabaseMeta {

    @Override
    public String getFieldDefinition(
            IValueMeta v, String tk, String pk, boolean useAutoinc, boolean addFieldName, boolean addCR) {
        String retval = "";

        String fieldname = v.getName();
        if (v.getLength() == DatabaseMeta.CLOB_LENGTH) {
            v.setLength(getMaxTextFieldLength());
        }
        int length = v.getLength();
        int precision = v.getPrecision();

        if (addFieldName) {
            retval += fieldname + " ";
        }

        int type = v.getType();
        switch (type) {
            case IValueMeta.TYPE_TIMESTAMP:
            case IValueMeta.TYPE_DATE:
                retval += "DATETIME";
                break;
            case IValueMeta.TYPE_BOOLEAN:
                if (isSupportsBooleanDataType()) {
                    retval += "BOOLEAN";
                } else {
                    retval += "CHAR(1)";
                }
                break;

            case IValueMeta.TYPE_NUMBER:
            case IValueMeta.TYPE_INTEGER:
            case IValueMeta.TYPE_BIGNUMBER:
                if (fieldname.equalsIgnoreCase(tk)
                        || // Technical key
                        fieldname.equalsIgnoreCase(pk) // Primary key
                ) {
                    if (useAutoinc) {
                        retval += "BIGINT AUTO_INCREMENT NOT NULL PRIMARY KEY";
                    } else {
                        retval += "BIGINT NOT NULL PRIMARY KEY";
                    }
                } else {
                    // Integer values...
                    if (precision == 0) {
                        if (length > 9) {
                            if (length < 19) {
                                // can hold signed values between -9223372036854775808 and 9223372036854775807
                                // 18 significant digits
                                retval += "BIGINT";
                            } else {
                                retval += "DECIMAL(" + length + ")";
                            }
                        } else {
                            retval += "INT";
                        }
                    } else {
                        // Floating point values...
                        if (length > 15) {
                            retval += "DECIMAL(" + length;
                            if (precision > 0) {
                                retval += ", " + precision;
                            }
                            retval += ")";
                        } else {
                            // A double-precision floating-point number is accurate to approximately 15 decimal
                            // places.
                            // http://mysql.mirrors-r-us.net/doc/refman/5.1/en/numeric-type-overview.html
                            retval += "DOUBLE";
                        }
                    }
                }
                break;
            case IValueMeta.TYPE_STRING:
                if (length > 0) {
                    if (length == 1) {
                        retval += "CHAR(1)";
                    } else if (length < 65533) {
                        retval += "VARCHAR(" + length + ")";
                    }  else {
                        retval += "STRING";
                    }
                } else {
                    retval += "STRING";
                }
                break;
            case IValueMeta.TYPE_BINARY:
                retval += "BITMAP";
                break;
            default:
                retval += " UNKNOWN";
                break;
        }

        if (addCR) {
            retval += Const.CR;
        }

        return retval;
    }
}
