/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kylin.dict;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.util.List;
import java.util.NavigableSet;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.kylin.common.KylinConfig;
import org.apache.kylin.common.persistence.ResourceStore;
import org.apache.kylin.common.persistence.WriteConflictException;
import org.apache.kylin.common.util.ClassUtil;
import org.apache.kylin.common.util.Dictionary;
import org.apache.kylin.common.util.JsonUtil;
import org.apache.kylin.dict.AppendTrieDictionary;
import org.apache.kylin.dict.DictionaryGenerator;
import org.apache.kylin.dict.DictionaryInfo;
import org.apache.kylin.dict.DictionaryInfoSerializer;
import org.apache.kylin.dict.IDictionaryBuilder;
import org.apache.kylin.dict.IDictionaryValueEnumerator;
import org.apache.kylin.dict.TableColumnValueEnumerator;
import org.apache.kylin.metadata.datatype.DataType;
import org.apache.kylin.metadata.model.TblColRef;
import org.apache.kylin.source.IReadableTable;
import org.apache.kylin.tool.shaded.org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DictionaryManager {
    private static final Logger logger = LoggerFactory.getLogger(DictionaryManager.class);
    private static final DictionaryInfo NONE_INDICATOR = new DictionaryInfo();
    private KylinConfig config;
    private LoadingCache<String, DictionaryInfo> dictCache;

    public static DictionaryManager getInstance(KylinConfig config) {
        return config.getManager(DictionaryManager.class);
    }

    static DictionaryManager newInstance(KylinConfig config) throws IOException {
        return new DictionaryManager(config);
    }

    private DictionaryManager(KylinConfig config) {
        this.config = config;
        CacheStrength strength = CacheStrength.valueOf(config.getKylinDictCacheStrength());
        CacheBuilder cacheBuilder = CacheBuilder.newBuilder();
        switch (strength) {
            case soft: {
                cacheBuilder.softValues();
                break;
            }
            case week: {
                cacheBuilder.weakValues();
                break;
            }
        }
        this.dictCache = cacheBuilder.removalListener((RemovalListener)new RemovalListener<String, DictionaryInfo>(){

            public void onRemoval(RemovalNotification<String, DictionaryInfo> notification) {
                logger.info("Dict with resource path {} is removed due to {}", notification.getKey(), (Object)notification.getCause());
            }
        }).maximumSize(config.getCachedDictMaxEntrySize()).expireAfterWrite(1L, TimeUnit.DAYS).build((CacheLoader)new CacheLoader<String, DictionaryInfo>(){

            public DictionaryInfo load(String key) throws Exception {
                DictionaryInfo dictInfo = DictionaryManager.this.load(key, true);
                if (dictInfo == null) {
                    return NONE_INDICATOR;
                }
                return dictInfo;
            }
        });
    }

    public Dictionary<String> getDictionary(String resourcePath) throws IOException {
        DictionaryInfo dictInfo = this.getDictionaryInfo(resourcePath);
        return dictInfo == null ? null : dictInfo.getDictionaryObject();
    }

    public DictionaryInfo getDictionaryInfo(String resourcePath) throws IOException {
        try {
            if (resourcePath == null) {
                return NONE_INDICATOR;
            }
            DictionaryInfo result = (DictionaryInfo)this.dictCache.get((Object)resourcePath);
            if (result == NONE_INDICATOR) {
                return null;
            }
            return result;
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e.getCause());
        }
    }

    public DictionaryInfo forceSave(Dictionary<String> newDict, DictionaryInfo newDictInfo) throws IOException {
        this.initDictInfo(newDict, newDictInfo);
        logger.info("force to save dict directly");
        return this.saveNewDict(newDictInfo);
    }

    public DictionaryInfo trySaveNewDict(Dictionary<String> newDict, DictionaryInfo newDictInfo) throws IOException {
        this.initDictInfo(newDict, newDictInfo);
        if (this.config.isGrowingDictEnabled()) {
            logger.info("Growing dict is enabled, merge with largest dictionary");
            DictionaryInfo largestDictInfo = this.findLargestDictInfo(newDictInfo);
            if (largestDictInfo != null) {
                Dictionary<String> largestDictObject = (largestDictInfo = this.getDictionaryInfo(largestDictInfo.getResourcePath())).getDictionaryObject();
                if (largestDictObject.contains(newDict)) {
                    logger.info("dictionary content {}, is contained by  dictionary at {}", (Object)newDict, (Object)largestDictInfo.getResourcePath());
                    return largestDictInfo;
                }
                if (newDict.contains(largestDictObject)) {
                    logger.info("dictionary content {} is by far the largest, save it", (Object)newDict);
                    return this.saveNewDict(newDictInfo);
                }
                logger.info("merge dict and save...");
                return this.mergeDictionary(Lists.newArrayList((Object[])new DictionaryInfo[]{newDictInfo, largestDictInfo}));
            }
            logger.info("first dict of this column, save it directly");
            return this.saveNewDict(newDictInfo);
        }
        DictionaryInfo dupDict = this.checkDupByContent(newDictInfo, newDict);
        if (dupDict != null) {
            logger.info("Identical dictionary content, reuse existing dictionary at {}", (Object)dupDict.getResourcePath());
            dupDict = this.updateExistingDictLastModifiedTime(dupDict.getResourcePath());
            return dupDict;
        }
        return this.saveNewDict(newDictInfo);
    }

    private DictionaryInfo checkDupByContent(DictionaryInfo dictInfo, Dictionary<String> dict) throws IOException {
        ResourceStore store = this.getStore();
        NavigableSet<String> existings = store.listResources(dictInfo.getResourceDir());
        if (existings == null) {
            return null;
        }
        logger.info("{} existing dictionaries of the same column", (Object)existings.size());
        if (existings.size() > 100) {
            logger.warn("Too many dictionaries under {}, dict count: {}", (Object)dictInfo.getResourceDir(), (Object)existings.size());
        }
        for (String existing : existings) {
            try {
                DictionaryInfo existingInfo;
                if (!existing.endsWith(".dict") || (existingInfo = this.getDictionaryInfo(existing)) == null || !dict.equals(existingInfo.getDictionaryObject())) continue;
                return existingInfo;
            }
            catch (Exception ex) {
                logger.error("Tolerate exception checking dup dictionary " + existing, ex);
            }
        }
        return null;
    }

    private DictionaryInfo updateExistingDictLastModifiedTime(String dictPath) throws IOException {
        ResourceStore store = this.getStore();
        if (StringUtils.isBlank(dictPath)) {
            return NONE_INDICATOR;
        }
        int retry = 7;
        while (retry-- > 0) {
            try {
                long now = System.currentTimeMillis();
                store.updateTimestamp(dictPath, now);
                logger.info("Update dictionary {} lastModifiedTime to {}", (Object)dictPath, (Object)now);
                return this.loadAndUpdateLocalCache(dictPath);
            }
            catch (WriteConflictException e) {
                if (retry <= 0) {
                    logger.error("Retry is out, till got error, abandoning...", e);
                    throw e;
                }
                logger.warn("Write conflict to update dictionary " + dictPath + " retry remaining " + retry + ", will retry...");
            }
        }
        return this.loadAndUpdateLocalCache(dictPath);
    }

    private void initDictInfo(Dictionary<String> newDict, DictionaryInfo newDictInfo) {
        newDictInfo.setCardinality(newDict.getSize());
        newDictInfo.setDictionaryObject(newDict);
        newDictInfo.setDictionaryClass(newDict.getClass().getName());
    }

    private DictionaryInfo saveNewDict(DictionaryInfo newDictInfo) throws IOException {
        this.save(newDictInfo);
        this.updateDictCache(newDictInfo);
        return newDictInfo;
    }

    public DictionaryInfo mergeDictionary(List<DictionaryInfo> dicts) throws IOException {
        if (dicts.isEmpty()) {
            return null;
        }
        if (dicts.size() == 1) {
            return dicts.get(0);
        }
        for (DictionaryInfo dict : dicts) {
            if (!dict.getDictionaryClass().equals(AppendTrieDictionary.class.getName())) continue;
            return dict;
        }
        DictionaryInfo firstDictInfo = null;
        int totalSize = 0;
        for (DictionaryInfo info : dicts) {
            if (firstDictInfo == null) {
                firstDictInfo = info;
            } else if (!firstDictInfo.isDictOnSameColumn(info)) {
                logger.warn("Merging dictionaries are not structurally equal : {} and {}", (Object)firstDictInfo.getResourcePath(), (Object)info.getResourcePath());
            }
            totalSize = (int)((long)totalSize + info.getInput().getSize());
        }
        if (firstDictInfo == null) {
            throw new IllegalArgumentException("DictionaryManager.mergeDictionary input cannot be null");
        }
        DictionaryInfo newDictInfo = new DictionaryInfo(firstDictInfo);
        IReadableTable.TableSignature signature = newDictInfo.getInput();
        signature.setSize(totalSize);
        signature.setLastModifiedTime(System.currentTimeMillis());
        signature.setPath("merged_with_no_original_path");
        boolean identicalSourceDicts = true;
        for (int i = 1; i < dicts.size(); ++i) {
            if (dicts.get(0).getDictionaryObject().equals(dicts.get(i).getDictionaryObject())) continue;
            identicalSourceDicts = false;
            break;
        }
        if (identicalSourceDicts) {
            logger.info("Use one of the merging dictionaries directly");
            return dicts.get(0);
        }
        Dictionary newDict = DictionaryGenerator.mergeDictionaries(DataType.getType(newDictInfo.getDataType()), dicts);
        return this.trySaveNewDict(newDict, newDictInfo);
    }

    public DictionaryInfo buildDictionary(TblColRef col, IReadableTable inpTable) throws IOException {
        return this.buildDictionary(col, inpTable, null);
    }

    public DictionaryInfo buildDictionary(TblColRef col, IReadableTable inpTable, String builderClass) throws IOException {
        if (!inpTable.exists()) {
            return null;
        }
        logger.info("building dictionary for {}", (Object)col);
        DictionaryInfo dictInfo = this.createDictionaryInfo(col, inpTable);
        String dupInfo = this.checkDupByInfo(dictInfo);
        if (dupInfo != null) {
            logger.info("Identical dictionary input {}, reuse existing dictionary at {}", (Object)dictInfo.getInput(), (Object)dupInfo);
            DictionaryInfo dupDictInfo = this.updateExistingDictLastModifiedTime(dupInfo);
            return dupDictInfo;
        }
        logger.info("Building dictionary object {}", (Object)JsonUtil.writeValueAsString(dictInfo));
        Dictionary<String> dictionary = this.buildDictFromReadableTable(inpTable, dictInfo, builderClass, col);
        return this.trySaveNewDict(dictionary, dictInfo);
    }

    private Dictionary<String> buildDictFromReadableTable(IReadableTable inpTable, DictionaryInfo dictInfo, String builderClass, TblColRef col) throws IOException {
        Dictionary<String> dictionary;
        try (IDictionaryValueEnumerator columnValueEnumerator = null;){
            columnValueEnumerator = new TableColumnValueEnumerator(inpTable.getReader(), dictInfo.getSourceColumnIndex());
            if (builderClass == null) {
                dictionary = DictionaryGenerator.buildDictionary(DataType.getType(dictInfo.getDataType()), columnValueEnumerator);
            } else {
                IDictionaryBuilder builder = (IDictionaryBuilder)ClassUtil.newInstance(builderClass);
                dictionary = DictionaryGenerator.buildDictionary(builder, dictInfo, columnValueEnumerator);
            }
        }
        return dictionary;
    }

    public DictionaryInfo saveDictionary(TblColRef col, IReadableTable inpTable, Dictionary<String> dictionary) throws IOException {
        DictionaryInfo dictInfo = this.createDictionaryInfo(col, inpTable);
        String dupInfo = this.checkDupByInfo(dictInfo);
        if (dupInfo != null) {
            logger.info("Identical dictionary input {}, reuse existing dictionary at {}", (Object)dictInfo.getInput(), (Object)dupInfo);
            DictionaryInfo dupDictInfo = this.updateExistingDictLastModifiedTime(dupInfo);
            return dupDictInfo;
        }
        return this.trySaveNewDict(dictionary, dictInfo);
    }

    private DictionaryInfo createDictionaryInfo(TblColRef col, IReadableTable inpTable) throws IOException {
        IReadableTable.TableSignature inputSig = inpTable.getSignature();
        if (inputSig == null) {
            throw new IllegalStateException("Input table does not exist: " + inpTable);
        }
        DictionaryInfo dictInfo = new DictionaryInfo(col.getColumnDesc(), col.getDatatype(), inputSig);
        return dictInfo;
    }

    private String checkDupByInfo(DictionaryInfo dictInfo) throws IOException {
        ResourceStore store = this.getStore();
        List<DictionaryInfo> allResources = store.getAllResources(dictInfo.getResourceDir(), DictionaryInfoSerializer.INFO_SERIALIZER);
        IReadableTable.TableSignature input = dictInfo.getInput();
        for (DictionaryInfo dictionaryInfo : allResources) {
            if (!input.equals(dictionaryInfo.getInput())) continue;
            return dictionaryInfo.getResourcePath();
        }
        return null;
    }

    private DictionaryInfo findLargestDictInfo(DictionaryInfo dictInfo) throws IOException {
        ResourceStore store = this.getStore();
        List<DictionaryInfo> allResources = store.getAllResources(dictInfo.getResourceDir(), DictionaryInfoSerializer.INFO_SERIALIZER);
        DictionaryInfo largestDict = null;
        for (DictionaryInfo dictionaryInfo : allResources) {
            if (largestDict == null) {
                largestDict = dictionaryInfo;
                continue;
            }
            if (largestDict.getCardinality() >= dictionaryInfo.getCardinality()) continue;
            largestDict = dictionaryInfo;
        }
        return largestDict;
    }

    public void removeDictionary(String resourcePath) throws IOException {
        logger.info("Remvoing dict: {}", (Object)resourcePath);
        ResourceStore store = this.getStore();
        store.deleteResource(resourcePath);
        this.dictCache.invalidate((Object)resourcePath);
    }

    public void removeDictionaries(String srcTable, String srcCol) throws IOException {
        DictionaryInfo info = new DictionaryInfo();
        info.setSourceTable(srcTable);
        info.setSourceColumn(srcCol);
        ResourceStore store = this.getStore();
        NavigableSet<String> existings = store.listResources(info.getResourceDir());
        if (existings == null) {
            return;
        }
        for (String existing : existings) {
            this.removeDictionary(existing);
        }
    }

    void save(DictionaryInfo dict) throws IOException {
        ResourceStore store = this.getStore();
        String path = dict.getResourcePath();
        logger.info("Saving dictionary at {}", (Object)path);
        store.putBigResource(path, dict, System.currentTimeMillis(), DictionaryInfoSerializer.FULL_SERIALIZER);
    }

    private DictionaryInfo loadAndUpdateLocalCache(String dictPath) throws IOException {
        DictionaryInfo dictInfo = this.load(dictPath, true);
        this.updateDictCache(dictInfo);
        return dictInfo;
    }

    DictionaryInfo load(String resourcePath, boolean loadDictObj) throws IOException {
        ResourceStore store = this.getStore();
        if (loadDictObj) {
            logger.info("Loading dictionary at {}", (Object)resourcePath);
        }
        DictionaryInfo info = store.getResource(resourcePath, loadDictObj ? DictionaryInfoSerializer.FULL_SERIALIZER : DictionaryInfoSerializer.INFO_SERIALIZER);
        return info;
    }

    private void updateDictCache(DictionaryInfo newDictInfo) {
        this.dictCache.put((Object)newDictInfo.getResourcePath(), (Object)newDictInfo);
    }

    private ResourceStore getStore() {
        return ResourceStore.getStore(this.config);
    }

    private static enum CacheStrength {
        week,
        soft,
        strong;

    }
}

