/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.balancer;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.Message;
import java.io.IOException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.conf.StorageUnit;
import org.apache.hadoop.hdds.fs.DUFactory;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerConfiguration;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerMetrics;
import org.apache.hadoop.hdds.scm.container.balancer.ContainerBalancerTask;
import org.apache.hadoop.hdds.scm.container.balancer.IllegalContainerBalancerStateException;
import org.apache.hadoop.hdds.scm.container.balancer.InvalidContainerBalancerConfigurationException;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.ha.StatefulService;
import org.apache.hadoop.hdds.scm.server.StorageContainerManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerBalancer
extends StatefulService {
    private static final AtomicInteger ID = new AtomicInteger();
    public static final Logger LOG = LoggerFactory.getLogger(ContainerBalancer.class);
    private StorageContainerManager scm;
    private final SCMContext scmContext;
    private OzoneConfiguration ozoneConfiguration;
    private ContainerBalancerConfiguration config;
    private ContainerBalancerMetrics metrics;
    private volatile Thread currentBalancingThread;
    private volatile ContainerBalancerTask task = null;
    private ReentrantLock lock;

    public ContainerBalancer(StorageContainerManager scm) {
        super(scm.getStatefulServiceStateManager());
        this.scm = scm;
        this.ozoneConfiguration = scm.getConfiguration();
        this.config = (ContainerBalancerConfiguration)this.ozoneConfiguration.getObject(ContainerBalancerConfiguration.class);
        this.scmContext = scm.getScmContext();
        this.metrics = ContainerBalancerMetrics.create();
        this.lock = new ReentrantLock();
        scm.getSCMServiceManager().register(this);
    }

    @Override
    public void notifyStatusChanged() {
        if (!this.scmContext.isLeader() || this.scmContext.isInSafeMode()) {
            boolean shouldStop;
            this.lock.lock();
            try {
                shouldStop = this.canBalancerStop();
            }
            finally {
                this.lock.unlock();
            }
            if (shouldStop) {
                LOG.info("Stopping ContainerBalancer in this scm on status change");
                this.stop();
            }
            return;
        }
        this.lock.lock();
        try {
            boolean shouldRun = this.shouldRun();
            if (shouldRun && !this.canBalancerStart()) {
                LOG.warn("Could not start ContainerBalancer on notify, might be stopped");
            }
            if (shouldRun && this.canBalancerStart()) {
                LOG.info("Starting ContainerBalancer in this scm on status change");
                try {
                    this.start();
                }
                catch (IllegalContainerBalancerStateException | InvalidContainerBalancerConfigurationException e) {
                    LOG.warn("Could not start ContainerBalancer on raft/safe-mode status change.", (Throwable)e);
                }
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    @Override
    public boolean shouldRun() {
        try {
            HddsProtos.ContainerBalancerConfigurationProto proto = this.readConfiguration(HddsProtos.ContainerBalancerConfigurationProto.class);
            if (proto == null) {
                LOG.warn("Could not find persisted configuration for {} when checking if ContainerBalancer should run. ContainerBalancer should not run now.", (Object)this.getServiceName());
                return false;
            }
            return proto.getShouldRun();
        }
        catch (IOException e) {
            LOG.warn("Could not read persisted configuration for checking if ContainerBalancer should start. ContainerBalancer should not start now.", (Throwable)e);
            return false;
        }
    }

    public boolean isBalancerRunning() {
        return null != this.task && this.task.getBalancerStatus() == ContainerBalancerTask.Status.RUNNING;
    }

    private boolean canBalancerStart() {
        return null == this.task || this.task.getBalancerStatus() == ContainerBalancerTask.Status.STOPPED;
    }

    public ContainerBalancerTask.Status getBalancerStatus() {
        return null != this.task ? this.task.getBalancerStatus() : ContainerBalancerTask.Status.STOPPED;
    }

    private boolean canBalancerStop() {
        return this.isBalancerRunning();
    }

    @Override
    public String getServiceName() {
        return ContainerBalancer.class.getSimpleName();
    }

    @Override
    public void start() throws IllegalContainerBalancerStateException, InvalidContainerBalancerConfigurationException {
        this.lock.lock();
        try {
            HddsProtos.ContainerBalancerConfigurationProto proto;
            this.validateState(false);
            try {
                proto = this.readConfiguration(HddsProtos.ContainerBalancerConfigurationProto.class);
            }
            catch (IOException e) {
                throw new InvalidContainerBalancerConfigurationException("Could not retrieve persisted configuration while starting Container Balancer as an SCMService. Will not start now.", e);
            }
            if (proto == null) {
                throw new InvalidContainerBalancerConfigurationException("Persisted configuration for ContainerBalancer is null during start. Will not start now.");
            }
            if (!proto.getShouldRun()) {
                throw new IllegalContainerBalancerStateException("According to persisted configuration, ContainerBalancer should not run.");
            }
            ContainerBalancerConfiguration configuration = ContainerBalancerConfiguration.fromProtobuf(proto, this.ozoneConfiguration);
            this.validateConfiguration(configuration);
            this.config = configuration;
            this.startBalancingThread(proto.getNextIterationIndex(), true);
        }
        finally {
            this.lock.unlock();
        }
    }

    public void startBalancer(ContainerBalancerConfiguration configuration) throws IllegalContainerBalancerStateException, InvalidContainerBalancerConfigurationException, IOException {
        this.lock.lock();
        try {
            this.validateState(false);
            this.validateConfiguration(configuration);
            this.saveConfiguration(configuration, true, 0);
            this.config = configuration;
            this.startBalancingThread(0, false);
        }
        finally {
            this.lock.unlock();
        }
    }

    private void startBalancingThread(int nextIterationIndex, boolean delayStart) {
        String prefix = this.scmContext.threadNamePrefix();
        this.task = new ContainerBalancerTask(this.scm, nextIterationIndex, this, this.metrics, this.config, delayStart);
        Thread thread = new Thread(this.task);
        thread.setName(prefix + "ContainerBalancerTask-" + ID.incrementAndGet());
        thread.setDaemon(true);
        thread.start();
        this.currentBalancingThread = thread;
        LOG.info("Starting Container Balancer {}... {}", (Object)thread, (Object)this);
    }

    private void validateState(boolean expectedRunning) throws IllegalContainerBalancerStateException {
        if (!this.scmContext.isLeaderReady()) {
            LOG.warn("SCM is not leader ready");
            throw new IllegalContainerBalancerStateException("SCM is not leader ready");
        }
        if (this.scmContext.isInSafeMode()) {
            LOG.warn("SCM is in safe mode");
            throw new IllegalContainerBalancerStateException("SCM is in safe mode");
        }
        if (!expectedRunning && !this.canBalancerStart()) {
            throw new IllegalContainerBalancerStateException("Expect ContainerBalancer as not running state, but running state is actually " + (Object)((Object)this.getBalancerStatus()));
        }
        if (expectedRunning && !this.canBalancerStop()) {
            throw new IllegalContainerBalancerStateException("Expect ContainerBalancer as running state, but running state is actually " + (Object)((Object)this.getBalancerStatus()));
        }
    }

    @Override
    public void stop() {
        Thread balancingThread;
        this.lock.lock();
        try {
            if (!this.canBalancerStop()) {
                LOG.warn("Cannot stop Container Balancer because it's not running or stopping");
                return;
            }
            LOG.info("Trying to stop ContainerBalancer in this SCM.");
            this.task.stop();
            balancingThread = this.currentBalancingThread;
        }
        finally {
            this.lock.unlock();
        }
        ContainerBalancer.blockTillTaskStop(balancingThread);
    }

    private static void blockTillTaskStop(Thread balancingThread) {
        balancingThread.interrupt();
        LOG.info("Container Balancer waiting for {} to stop", (Object)balancingThread);
        try {
            balancingThread.join();
        }
        catch (InterruptedException exception) {
            Thread.currentThread().interrupt();
        }
        LOG.info("Container Balancer stopped successfully.");
    }

    public void stopBalancer() throws IOException, IllegalContainerBalancerStateException {
        Thread balancingThread;
        this.lock.lock();
        try {
            this.validateState(true);
            this.saveConfiguration(this.config, false, 0);
            LOG.info("Trying to stop ContainerBalancer service.");
            this.task.stop();
            balancingThread = this.currentBalancingThread;
        }
        finally {
            this.lock.unlock();
        }
        ContainerBalancer.blockTillTaskStop(balancingThread);
    }

    public void saveConfiguration(ContainerBalancerConfiguration configuration, boolean shouldRun, int index) throws IOException {
        this.config = configuration;
        this.saveConfiguration((Message)configuration.toProtobufBuilder().setShouldRun(shouldRun).setNextIterationIndex(index).build());
    }

    private void validateConfiguration(ContainerBalancerConfiguration conf) throws InvalidContainerBalancerConfigurationException {
        long size = (long)this.ozoneConfiguration.getStorageSize("ozone.scm.container.size", "5GB", StorageUnit.BYTES);
        if (conf.getMaxSizeEnteringTarget() <= size) {
            LOG.warn("hdds.container.balancer.size.entering.target.max {} should be greater than ozone.scm.container.size {}", (Object)conf.getMaxSizeEnteringTarget(), (Object)size);
            throw new InvalidContainerBalancerConfigurationException("hdds.container.balancer.size.entering.target.max should be greater than ozone.scm.container.size");
        }
        if (conf.getMaxSizeLeavingSource() <= size) {
            LOG.warn("hdds.container.balancer.size.leaving.source.max {} should be greater than ozone.scm.container.size {}", (Object)conf.getMaxSizeLeavingSource(), (Object)size);
            throw new InvalidContainerBalancerConfigurationException("hdds.container.balancer.size.leaving.source.max should be greater than ozone.scm.container.size");
        }
        DUFactory.Conf duConf = (DUFactory.Conf)this.ozoneConfiguration.getObject(DUFactory.Conf.class);
        long refreshPeriod = duConf.getRefreshPeriod().toMillis();
        if (conf.getBalancingInterval().toMillis() <= refreshPeriod) {
            LOG.warn("hdds.container.balancer.balancing.iteration.interval {} should be greater than hdds.datanode.du.refresh.period {}", (Object)conf.getBalancingInterval().toMillis(), (Object)refreshPeriod);
        }
        if (conf.getMoveReplicationTimeout().toMillis() >= conf.getMoveTimeout().toMillis()) {
            LOG.warn("hdds.container.balancer.move.replication.timeout {} should be less than hdds.container.balancer.move.timeout {}.", (Object)conf.getMoveReplicationTimeout().toMinutes(), (Object)conf.getMoveTimeout().toMinutes());
            throw new InvalidContainerBalancerConfigurationException("hdds.container.balancer.move.replication.timeout should be less than hdds.container.balancer.move.timeout.");
        }
    }

    public ContainerBalancerMetrics getMetrics() {
        return this.metrics;
    }

    @VisibleForTesting
    Thread getCurrentBalancingThread() {
        return this.currentBalancingThread;
    }

    public String toString() {
        String status = String.format("%nContainer Balancer status:%n%-30s %s%n%-30s %b%n", "Key", "Value", "Running", this.isBalancerRunning());
        return status + this.config.toString();
    }
}

