/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.container.states;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerInfo;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.states.ContainerAttribute;
import org.apache.hadoop.hdds.scm.container.states.ContainerQueryKey;
import org.apache.hadoop.hdds.scm.exceptions.SCMException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ContainerStateMap {
    private static final Logger LOG = LoggerFactory.getLogger(ContainerStateMap.class);
    private static final NavigableSet<ContainerID> EMPTY_SET = Collections.unmodifiableNavigableSet(new TreeSet());
    private final ContainerAttribute<HddsProtos.LifeCycleState> lifeCycleStateMap = new ContainerAttribute();
    private final ContainerAttribute<String> ownerMap = new ContainerAttribute();
    private final ContainerAttribute<ReplicationConfig> repConfigMap = new ContainerAttribute();
    private final ContainerAttribute<HddsProtos.ReplicationType> typeMap = new ContainerAttribute();
    private final Map<ContainerID, ContainerInfo> containerMap = new ConcurrentHashMap<ContainerID, ContainerInfo>();
    private final Map<ContainerID, Set<ContainerReplica>> replicaMap = new ConcurrentHashMap<ContainerID, Set<ContainerReplica>>();
    private final Map<ContainerQueryKey, NavigableSet<ContainerID>> resultCache = new ConcurrentHashMap<ContainerQueryKey, NavigableSet<ContainerID>>();

    public void addContainer(ContainerInfo info) throws SCMException {
        Preconditions.checkNotNull((Object)info, (Object)"Container Info cannot be null");
        ContainerID id = info.containerID();
        if (!this.contains(id)) {
            this.containerMap.put(id, info);
            this.lifeCycleStateMap.insert(info.getState(), id);
            this.ownerMap.insert(info.getOwner(), id);
            this.repConfigMap.insert(info.getReplicationConfig(), id);
            this.typeMap.insert(info.getReplicationType(), id);
            this.replicaMap.put(id, Collections.emptySet());
            this.flushCache(info);
            LOG.trace("Container {} added to ContainerStateMap.", (Object)id);
        }
    }

    public boolean contains(ContainerID id) {
        return this.containerMap.containsKey(id);
    }

    public void removeContainer(ContainerID id) {
        Preconditions.checkNotNull((Object)id, (Object)"ContainerID cannot be null");
        if (this.contains(id)) {
            ContainerInfo info = this.containerMap.remove(id);
            this.lifeCycleStateMap.remove(info.getState(), id);
            this.ownerMap.remove(info.getOwner(), id);
            this.repConfigMap.remove(info.getReplicationConfig(), id);
            this.typeMap.remove(info.getReplicationType(), id);
            this.replicaMap.remove(id);
            this.flushCache(info);
            LOG.trace("Container {} removed from ContainerStateMap.", (Object)id);
        }
    }

    public ContainerInfo getContainerInfo(ContainerID containerID) {
        return this.containerMap.get(containerID);
    }

    public Set<ContainerReplica> getContainerReplicas(ContainerID containerID) {
        Preconditions.checkNotNull((Object)containerID);
        return this.replicaMap.get(containerID);
    }

    public void updateContainerReplica(ContainerID containerID, ContainerReplica replica) {
        Preconditions.checkNotNull((Object)containerID);
        if (this.contains(containerID)) {
            Set<ContainerReplica> newSet = this.createNewReplicaSet(containerID);
            newSet.remove(replica);
            newSet.add(replica);
            this.replaceReplicaSet(containerID, newSet);
        }
    }

    public void removeContainerReplica(ContainerID containerID, ContainerReplica replica) {
        Preconditions.checkNotNull((Object)containerID);
        Preconditions.checkNotNull((Object)replica);
        if (this.contains(containerID)) {
            Set<ContainerReplica> newSet = this.createNewReplicaSet(containerID);
            newSet.remove(replica);
            this.replaceReplicaSet(containerID, newSet);
        }
    }

    private Set<ContainerReplica> createNewReplicaSet(ContainerID containerID) {
        Set<ContainerReplica> existingSet = this.replicaMap.get(containerID);
        return existingSet == null ? new HashSet<ContainerReplica>() : new HashSet<ContainerReplica>(existingSet);
    }

    private void replaceReplicaSet(ContainerID containerID, Set<ContainerReplica> newSet) {
        this.replicaMap.put(containerID, Collections.unmodifiableSet(newSet));
    }

    public void updateContainerInfo(ContainerInfo info) {
        Preconditions.checkNotNull((Object)info);
        ContainerID id = info.containerID();
        if (this.contains(id)) {
            ContainerInfo currentInfo = this.containerMap.get(id);
            this.flushCache(info, currentInfo);
            this.containerMap.put(id, info);
        }
    }

    public void updateState(ContainerID containerID, HddsProtos.LifeCycleState currentState, HddsProtos.LifeCycleState newState) throws SCMException {
        Preconditions.checkNotNull((Object)currentState);
        Preconditions.checkNotNull((Object)newState);
        if (!this.contains(containerID)) {
            return;
        }
        if (currentState == newState) {
            LOG.debug("CurrentState and NewState are the same, return from updateState directly.");
            return;
        }
        ContainerInfo currentInfo = this.containerMap.get(containerID);
        try {
            currentInfo.setState(newState);
            this.lifeCycleStateMap.update(currentState, newState, containerID);
            if (LOG.isTraceEnabled()) {
                LOG.trace("Updated the container {} to new state. Old = {}, new = {}", new Object[]{containerID, currentState, newState});
            }
            this.flushCache(currentInfo);
        }
        catch (SCMException ex) {
            LOG.error("Unable to update the container state.", (Throwable)ex);
            LOG.info("Reverting the update to lifecycle state. Moving back to old state. Old = {}, Attempted state = {}", (Object)currentState, (Object)newState);
            currentInfo.revertState();
            this.lifeCycleStateMap.update(newState, currentState, containerID);
            throw new SCMException("Updating the container map failed.", (Throwable)ex, SCMException.ResultCodes.FAILED_TO_CHANGE_CONTAINER_STATE);
        }
    }

    public Set<ContainerID> getAllContainerIDs() {
        return ImmutableSet.copyOf(this.containerMap.keySet());
    }

    NavigableSet<ContainerID> getContainerIDsByOwner(String ownerName) {
        Preconditions.checkNotNull((Object)ownerName);
        return this.ownerMap.getCollection(ownerName);
    }

    NavigableSet<ContainerID> getContainerIDsByType(HddsProtos.ReplicationType type) {
        Preconditions.checkNotNull((Object)type);
        return this.typeMap.getCollection(type);
    }

    NavigableSet<ContainerID> getContainerIDsByRepConfig(ReplicationConfig repConfig) {
        Preconditions.checkNotNull((Object)repConfig);
        return this.repConfigMap.getCollection(repConfig);
    }

    public NavigableSet<ContainerID> getContainerIDsByState(HddsProtos.LifeCycleState state) {
        Preconditions.checkNotNull((Object)state);
        return this.lifeCycleStateMap.getCollection(state);
    }

    public NavigableSet<ContainerID> getMatchingContainerIDs(HddsProtos.LifeCycleState state, String owner, ReplicationConfig repConfig) {
        Preconditions.checkNotNull((Object)state, (Object)"State cannot be null");
        Preconditions.checkNotNull((Object)owner, (Object)"Owner cannot be null");
        Preconditions.checkNotNull((Object)repConfig, (Object)"RepConfig cannot be null");
        ContainerQueryKey queryKey = new ContainerQueryKey(state, owner, repConfig);
        if (this.resultCache.containsKey(queryKey)) {
            return this.resultCache.get(queryKey);
        }
        NavigableSet<ContainerID> stateSet = this.lifeCycleStateMap.getCollection(state);
        if (stateSet.size() == 0) {
            return EMPTY_SET;
        }
        NavigableSet<ContainerID> ownerSet = this.ownerMap.getCollection(owner);
        if (ownerSet.size() == 0) {
            return EMPTY_SET;
        }
        NavigableSet<ContainerID> factorSet = this.repConfigMap.getCollection(repConfig);
        if (factorSet.size() == 0) {
            return EMPTY_SET;
        }
        NavigableSet<ContainerID> typeSet = this.typeMap.getCollection(repConfig.getReplicationType());
        if (typeSet.size() == 0) {
            return EMPTY_SET;
        }
        NavigableSet<ContainerID>[] sets = this.sortBySize(stateSet, ownerSet, factorSet, typeSet);
        NavigableSet<ContainerID> currentSet = sets[0];
        for (int x = 1; x < sets.length; ++x) {
            currentSet = this.intersectSets(currentSet, sets[x]);
        }
        this.resultCache.put(queryKey, currentSet);
        return currentSet;
    }

    private NavigableSet<ContainerID> intersectSets(NavigableSet<ContainerID> smaller, NavigableSet<ContainerID> bigger) {
        Preconditions.checkState((smaller.size() <= bigger.size() ? 1 : 0) != 0, (Object)"This function assumes the first set is lesser or equal to second set");
        TreeSet<ContainerID> resultSet = new TreeSet<ContainerID>();
        for (ContainerID id : smaller) {
            if (!bigger.contains(id)) continue;
            resultSet.add(id);
        }
        return resultSet;
    }

    @SafeVarargs
    private final NavigableSet<ContainerID>[] sortBySize(NavigableSet<ContainerID> ... sets) {
        for (int x = 0; x < sets.length - 1; ++x) {
            for (int y = 0; y < sets.length - x - 1; ++y) {
                if (sets[y].size() <= sets[y + 1].size()) continue;
                NavigableSet<ContainerID> temp = sets[y];
                sets[y] = sets[y + 1];
                sets[y + 1] = temp;
            }
        }
        return sets;
    }

    private void flushCache(ContainerInfo ... containerInfos) {
        for (ContainerInfo containerInfo : containerInfos) {
            ContainerQueryKey key = new ContainerQueryKey(containerInfo.getState(), containerInfo.getOwner(), containerInfo.getReplicationConfig());
            this.resultCache.remove(key);
        }
    }
}

