/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.node;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.conf.OzoneConfiguration;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.container.ContainerID;
import org.apache.hadoop.hdds.scm.container.ContainerNotFoundException;
import org.apache.hadoop.hdds.scm.container.ContainerReplica;
import org.apache.hadoop.hdds.scm.container.ReplicationManagerReport;
import org.apache.hadoop.hdds.scm.container.replication.ContainerReplicaCount;
import org.apache.hadoop.hdds.scm.container.replication.ReplicationManager;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.node.DatanodeAdminMonitor;
import org.apache.hadoop.hdds.scm.node.NodeDecommissionMetrics;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.server.events.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DatanodeAdminMonitorImpl
implements DatanodeAdminMonitor {
    private OzoneConfiguration conf;
    private EventPublisher eventQueue;
    private NodeManager nodeManager;
    private ReplicationManager replicationManager;
    private Queue<DatanodeDetails> pendingNodes = new ArrayDeque<DatanodeDetails>();
    private Queue<DatanodeDetails> cancelledNodes = new ArrayDeque<DatanodeDetails>();
    private Set<DatanodeDetails> trackedNodes = new HashSet<DatanodeDetails>();
    private NodeDecommissionMetrics metrics;
    private long pipelinesWaitingToClose = 0L;
    private long sufficientlyReplicatedContainers = 0L;
    private long trackedDecomMaintenance = 0L;
    private long trackedRecommission = 0L;
    private long unClosedContainers = 0L;
    private long underReplicatedContainers = 0L;
    private Map<String, NodeDecommissionMetrics.ContainerStateInWorkflow> containerStateByHost;
    private static final Logger LOG = LoggerFactory.getLogger(DatanodeAdminMonitorImpl.class);
    private final int containerDetailsLoggingLimit;

    public DatanodeAdminMonitorImpl(OzoneConfiguration conf, EventPublisher eventQueue, NodeManager nodeManager, ReplicationManager replicationManager) {
        this.conf = conf;
        this.eventQueue = eventQueue;
        this.nodeManager = nodeManager;
        this.replicationManager = replicationManager;
        this.containerDetailsLoggingLimit = conf.getInt("ozone.scm.datanode.admin.monitor.logging.limit", 1000);
        this.containerStateByHost = new HashMap<String, NodeDecommissionMetrics.ContainerStateInWorkflow>();
    }

    @Override
    public synchronized void startMonitoring(DatanodeDetails dn) {
        this.cancelledNodes.remove(dn);
        this.pendingNodes.add(dn);
    }

    @Override
    public synchronized void stopMonitoring(DatanodeDetails dn) {
        this.pendingNodes.remove(dn);
        this.cancelledNodes.add(dn);
    }

    @Override
    public synchronized void setMetrics(NodeDecommissionMetrics metrics) {
        this.metrics = metrics;
    }

    @Override
    public synchronized Set<DatanodeDetails> getTrackedNodes() {
        return Collections.unmodifiableSet(this.trackedNodes);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        try {
            this.containerStateByHost.clear();
            DatanodeAdminMonitorImpl datanodeAdminMonitorImpl = this;
            synchronized (datanodeAdminMonitorImpl) {
                this.trackedRecommission = this.getCancelledCount();
                this.processCancelledNodes();
                this.processPendingNodes();
                this.trackedDecomMaintenance = this.getTrackedNodeCount();
            }
            this.processTransitioningNodes();
            if (this.trackedNodes.size() > 0 || this.pendingNodes.size() > 0) {
                LOG.info("There are {} nodes tracked for decommission and maintenance.  {} pending nodes.", (Object)this.trackedNodes.size(), (Object)this.pendingNodes.size());
            }
            this.setMetricsToGauge();
        }
        catch (Exception e) {
            LOG.error("Caught an error in the DatanodeAdminMonitor", (Throwable)e);
        }
    }

    public int getPendingCount() {
        return this.pendingNodes.size();
    }

    public int getCancelledCount() {
        return this.cancelledNodes.size();
    }

    public int getTrackedNodeCount() {
        return this.trackedNodes.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized void setMetricsToGauge() {
        NodeDecommissionMetrics nodeDecommissionMetrics = this.metrics;
        synchronized (nodeDecommissionMetrics) {
            this.metrics.setContainersUnClosedTotal(this.unClosedContainers);
            this.metrics.setRecommissionNodesTotal(this.trackedRecommission);
            this.metrics.setDecommissioningMaintenanceNodesTotal(this.trackedDecomMaintenance);
            this.metrics.setContainersUnderReplicatedTotal(this.underReplicatedContainers);
            this.metrics.setContainersSufficientlyReplicatedTotal(this.sufficientlyReplicatedContainers);
            this.metrics.setPipelinesWaitingToCloseTotal(this.pipelinesWaitingToClose);
            this.metrics.metricRecordOfContainerStateByHost(this.containerStateByHost);
        }
    }

    void resetContainerMetrics() {
        this.pipelinesWaitingToClose = 0L;
        this.sufficientlyReplicatedContainers = 0L;
        this.unClosedContainers = 0L;
        this.underReplicatedContainers = 0L;
    }

    private void processCancelledNodes() {
        while (!this.cancelledNodes.isEmpty()) {
            DatanodeDetails dn = this.cancelledNodes.poll();
            try {
                this.stopTrackingNode(dn);
                this.putNodeBackInService(dn);
                LOG.info("Recommissioned node {}", (Object)dn);
            }
            catch (NodeNotFoundException e) {
                LOG.warn("Failed processing the cancel admin request for {}", (Object)dn, (Object)e);
            }
        }
    }

    private void processPendingNodes() {
        while (!this.pendingNodes.isEmpty()) {
            this.startTrackingNode(this.pendingNodes.poll());
        }
    }

    private void processTransitioningNodes() {
        this.resetContainerMetrics();
        Iterator<DatanodeDetails> iterator = this.trackedNodes.iterator();
        while (iterator.hasNext()) {
            DatanodeDetails dn = iterator.next();
            try {
                NodeStatus status = this.getNodeStatus(dn);
                if (!this.shouldContinueWorkflow(dn, status)) {
                    this.abortWorkflow(dn);
                    iterator.remove();
                    continue;
                }
                if (status.isMaintenance() && status.operationalStateExpired()) {
                    this.completeMaintenance(dn);
                    iterator.remove();
                    continue;
                }
                if (!status.isDecommissioning() && !status.isEnteringMaintenance() || !this.checkPipelinesClosedOnNode(dn) || status.getOperationalState() != dn.getPersistedOpState() || !this.checkContainersReplicatedOnNode(dn)) continue;
                status = this.getNodeStatus(dn);
                if (status.isDead()) {
                    LOG.warn("Datanode {} is dead and the admin workflow cannot continue. The node will be put back to IN_SERVICE and handled as a dead node", (Object)dn);
                    this.putNodeBackInService(dn);
                    iterator.remove();
                    continue;
                }
                if (status.isDecommissioning()) {
                    this.completeDecommission(dn);
                    iterator.remove();
                    continue;
                }
                if (!status.isEnteringMaintenance()) continue;
                this.putIntoMaintenance(dn);
            }
            catch (NodeNotFoundException e) {
                LOG.error("An unexpected error occurred processing datanode {}. Aborting the admin workflow", (Object)dn, (Object)e);
                this.abortWorkflow(dn);
                iterator.remove();
            }
        }
    }

    private boolean shouldContinueWorkflow(DatanodeDetails dn, NodeStatus nodeStatus) {
        if (!nodeStatus.isDecommission() && !nodeStatus.isMaintenance()) {
            LOG.warn("Datanode {} has an operational state of {} when it should be undergoing decommission or maintenance. Aborting admin for this node.", (Object)dn, (Object)nodeStatus.getOperationalState());
            return false;
        }
        if (nodeStatus.isDead() && !nodeStatus.isInMaintenance()) {
            LOG.error("Datanode {} is dead but is not IN_MAINTENANCE. Aborting the admin workflow for this node", (Object)dn);
            return false;
        }
        return true;
    }

    private boolean checkPipelinesClosedOnNode(DatanodeDetails dn) throws NodeNotFoundException {
        Set<PipelineID> pipelines = this.nodeManager.getPipelines(dn);
        NodeStatus status = this.nodeManager.getNodeStatus(dn);
        if (pipelines == null || pipelines.size() == 0 || status.operationalStateExpired()) {
            return true;
        }
        LOG.info("Waiting for pipelines to close for {}. There are {} pipelines", (Object)dn, (Object)pipelines.size());
        this.containerStateByHost.put(dn.getHostName(), new NodeDecommissionMetrics.ContainerStateInWorkflow(dn.getHostName(), 0L, 0L, 0L, pipelines.size()));
        this.pipelinesWaitingToClose += (long)pipelines.size();
        return false;
    }

    private boolean checkContainersReplicatedOnNode(DatanodeDetails dn) throws NodeNotFoundException {
        int sufficientlyReplicated = 0;
        int deleting = 0;
        int underReplicated = 0;
        int unclosed = 0;
        ArrayList<ContainerID> underReplicatedIDs = new ArrayList<ContainerID>();
        ArrayList<ContainerID> unClosedIDs = new ArrayList<ContainerID>();
        Set<ContainerID> containers = this.nodeManager.getContainers(dn);
        for (ContainerID cid : containers) {
            try {
                boolean replicatedOK;
                ContainerReplicaCount replicaSet = this.replicationManager.getContainerReplicaCount(cid);
                HddsProtos.LifeCycleState containerState = replicaSet.getContainer().getState();
                if (containerState == HddsProtos.LifeCycleState.DELETED || containerState == HddsProtos.LifeCycleState.DELETING) {
                    ++deleting;
                    continue;
                }
                boolean isHealthy = replicaSet.isHealthyEnoughForOffline();
                if (!isHealthy) {
                    if (LOG.isDebugEnabled()) {
                        unClosedIDs.add(cid);
                    }
                    if (unclosed < this.containerDetailsLoggingLimit || LOG.isDebugEnabled()) {
                        LOG.info("Unclosed Container {} {}; {}", new Object[]{cid, replicaSet, this.replicaDetails(replicaSet.getReplicas())});
                    }
                    ++unclosed;
                    continue;
                }
                boolean legacyEnabled = this.conf.getBoolean("hdds.scm.replication.enable.legacy", false);
                if (legacyEnabled) {
                    replicatedOK = replicaSet.isSufficientlyReplicatedForOffline(dn, this.nodeManager);
                } else {
                    ReplicationManagerReport report = new ReplicationManagerReport();
                    this.replicationManager.checkContainerStatus(replicaSet.getContainer(), report);
                    boolean bl = replicatedOK = report.getStat(ReplicationManagerReport.HealthState.UNDER_REPLICATED) == 0L;
                }
                if (replicatedOK) {
                    ++sufficientlyReplicated;
                    continue;
                }
                if (LOG.isDebugEnabled()) {
                    underReplicatedIDs.add(cid);
                }
                if (underReplicated < this.containerDetailsLoggingLimit || LOG.isDebugEnabled()) {
                    LOG.info("Under Replicated Container {} {}; {}", new Object[]{cid, replicaSet, this.replicaDetails(replicaSet.getReplicas())});
                }
                ++underReplicated;
            }
            catch (ContainerNotFoundException e) {
                LOG.warn("ContainerID {} present in node list for {} but not found in containerManager", (Object)cid, (Object)dn);
            }
        }
        LOG.info("{} has {} sufficientlyReplicated, {} deleting, {} underReplicated and {} unclosed containers", new Object[]{dn, sufficientlyReplicated, deleting, underReplicated, unclosed});
        this.containerStateByHost.put(dn.getHostName(), new NodeDecommissionMetrics.ContainerStateInWorkflow(dn.getHostName(), sufficientlyReplicated, underReplicated, unclosed, 0L));
        this.sufficientlyReplicatedContainers += (long)sufficientlyReplicated;
        this.underReplicatedContainers += (long)underReplicated;
        this.unClosedContainers += (long)unclosed;
        if (LOG.isDebugEnabled() && underReplicatedIDs.size() < 10000 && unClosedIDs.size() < 10000) {
            LOG.debug("{} has {} underReplicated [{}] and {} unclosed [{}] containers", new Object[]{dn, underReplicated, underReplicatedIDs.stream().map(Object::toString).collect(Collectors.joining(", ")), unclosed, unClosedIDs.stream().map(Object::toString).collect(Collectors.joining(", "))});
        }
        return underReplicated == 0 && unclosed == 0;
    }

    private String replicaDetails(Collection<ContainerReplica> replicas) {
        StringBuilder sb = new StringBuilder();
        sb.append("Replicas{");
        sb.append(replicas.stream().map(Object::toString).collect(Collectors.joining(",")));
        sb.append("}");
        return sb.toString();
    }

    private void completeDecommission(DatanodeDetails dn) throws NodeNotFoundException {
        this.setNodeOpState(dn, HddsProtos.NodeOperationalState.DECOMMISSIONED);
        LOG.info("Datanode {} has completed the admin workflow. The operational state has been set to {}", (Object)dn, (Object)HddsProtos.NodeOperationalState.DECOMMISSIONED);
    }

    private void putIntoMaintenance(DatanodeDetails dn) throws NodeNotFoundException {
        LOG.info("Datanode {} has entered maintenance", (Object)dn);
        this.setNodeOpState(dn, HddsProtos.NodeOperationalState.IN_MAINTENANCE);
    }

    private void completeMaintenance(DatanodeDetails dn) throws NodeNotFoundException {
        LOG.info("Datanode {} has ended maintenance automatically", (Object)dn);
        this.putNodeBackInService(dn);
    }

    private void startTrackingNode(DatanodeDetails dn) {
        this.eventQueue.fireEvent(SCMEvents.START_ADMIN_ON_NODE, (Object)dn);
        this.trackedNodes.add(dn);
    }

    private void stopTrackingNode(DatanodeDetails dn) {
        this.trackedNodes.remove(dn);
    }

    private void abortWorkflow(DatanodeDetails dn) {
        try {
            this.putNodeBackInService(dn);
        }
        catch (NodeNotFoundException e) {
            LOG.error("Unable to set the node OperationalState for {} while aborting the datanode admin workflow", (Object)dn);
        }
    }

    private void putNodeBackInService(DatanodeDetails dn) throws NodeNotFoundException {
        this.setNodeOpState(dn, HddsProtos.NodeOperationalState.IN_SERVICE);
    }

    private void setNodeOpState(DatanodeDetails dn, HddsProtos.NodeOperationalState state) throws NodeNotFoundException {
        long expiry = 0L;
        if (state == HddsProtos.NodeOperationalState.IN_MAINTENANCE || state == HddsProtos.NodeOperationalState.ENTERING_MAINTENANCE) {
            NodeStatus status = this.nodeManager.getNodeStatus(dn);
            expiry = status.getOpStateExpiryEpochSeconds();
        }
        this.nodeManager.setNodeOperationalState(dn, state, expiry);
    }

    private NodeStatus getNodeStatus(DatanodeDetails dnd) throws NodeNotFoundException {
        return this.nodeManager.getNodeStatus(dnd);
    }
}

