/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.lock;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.Striped;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.utils.CompositeKey;
import org.apache.hadoop.hdds.utils.SimpleStriped;
import org.apache.hadoop.ipc.ProcessingDetails;
import org.apache.hadoop.ipc.Server;
import org.apache.hadoop.ozone.om.lock.IOzoneManagerLock;
import org.apache.hadoop.ozone.om.lock.LockUsageInfo;
import org.apache.hadoop.ozone.om.lock.OMLockDetails;
import org.apache.hadoop.ozone.om.lock.OMLockMetrics;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OzoneManagerLock
implements IOzoneManagerLock {
    private static final Logger LOG = LoggerFactory.getLogger(OzoneManagerLock.class);
    private final Map<Resource, Striped<ReadWriteLock>> stripedLockByResource;
    private OMLockMetrics omLockMetrics;
    private final ThreadLocal<Short> lockSet = ThreadLocal.withInitial(() -> (short)0);
    private ThreadLocal<OMLockDetails> omLockDetails = ThreadLocal.withInitial(OMLockDetails::new);

    public OzoneManagerLock(ConfigurationSource conf) {
        this.omLockMetrics = OMLockMetrics.create();
        EnumMap<Resource, Striped<ReadWriteLock>> stripedLockMap = new EnumMap<Resource, Striped<ReadWriteLock>>(Resource.class);
        for (Resource r : Resource.values()) {
            stripedLockMap.put(r, this.createStripeLock(r, conf));
        }
        this.stripedLockByResource = Collections.unmodifiableMap(stripedLockMap);
    }

    private Striped<ReadWriteLock> createStripeLock(Resource r, ConfigurationSource conf) {
        boolean fair = conf.getBoolean("ozone.om.lock.fair", false);
        String stripeSizeKey = "ozone.om.lock.stripes." + r.getName().toLowerCase();
        int size = conf.getInt(stripeSizeKey, 512);
        return SimpleStriped.readWriteLock((int)size, (boolean)fair);
    }

    private ReentrantReadWriteLock getLock(Resource resource, String ... keys) {
        Striped<ReadWriteLock> striped = this.stripedLockByResource.get((Object)resource);
        Object key = CompositeKey.combineKeys((Object[])keys);
        return (ReentrantReadWriteLock)striped.get(key);
    }

    @Override
    public OMLockDetails acquireReadLock(Resource resource, String ... keys) {
        return this.acquireLock(resource, true, keys);
    }

    @Override
    public OMLockDetails acquireWriteLock(Resource resource, String ... keys) {
        return this.acquireLock(resource, false, keys);
    }

    private OMLockDetails acquireLock(Resource resource, boolean isReadLock, String ... keys) {
        this.omLockDetails.get().clear();
        if (!resource.canLock(this.lockSet.get())) {
            String errorMessage = this.getErrorMessage(resource);
            LOG.error(errorMessage);
            throw new RuntimeException(errorMessage);
        }
        long startWaitingTimeNanos = Time.monotonicNowNanos();
        ReentrantReadWriteLock lock = this.getLock(resource, keys);
        if (isReadLock) {
            lock.readLock().lock();
            this.updateReadLockMetrics(resource, lock, startWaitingTimeNanos);
        } else {
            lock.writeLock().lock();
            this.updateWriteLockMetrics(resource, lock, startWaitingTimeNanos);
        }
        this.lockSet.set(resource.setLock(this.lockSet.get()));
        this.omLockDetails.get().setLockAcquired(true);
        return this.omLockDetails.get();
    }

    private void updateReadLockMetrics(Resource resource, ReentrantReadWriteLock lock, long startWaitingTimeNanos) {
        if (lock.getReadHoldCount() == 1) {
            long readLockWaitingTimeNanos = Time.monotonicNowNanos() - startWaitingTimeNanos;
            this.omLockMetrics.setReadLockWaitingTimeMsStat(TimeUnit.NANOSECONDS.toMillis(readLockWaitingTimeNanos));
            this.updateProcessingDetails(ProcessingDetails.Timing.LOCKWAIT, readLockWaitingTimeNanos);
            resource.setStartReadHeldTimeNanos(Time.monotonicNowNanos());
        }
    }

    private void updateWriteLockMetrics(Resource resource, ReentrantReadWriteLock lock, long startWaitingTimeNanos) {
        if (lock.getWriteHoldCount() == 1 && lock.isWriteLockedByCurrentThread()) {
            long writeLockWaitingTimeNanos = Time.monotonicNowNanos() - startWaitingTimeNanos;
            this.omLockMetrics.setWriteLockWaitingTimeMsStat(TimeUnit.NANOSECONDS.toMillis(writeLockWaitingTimeNanos));
            this.updateProcessingDetails(ProcessingDetails.Timing.LOCKWAIT, writeLockWaitingTimeNanos);
            resource.setStartWriteHeldTimeNanos(Time.monotonicNowNanos());
        }
    }

    private String getErrorMessage(Resource resource) {
        return "Thread '" + Thread.currentThread().getName() + "' cannot acquire " + resource.name + " lock while holding " + this.getCurrentLocks().toString() + " lock(s).";
    }

    @VisibleForTesting
    List<String> getCurrentLocks() {
        ArrayList<String> currentLocks = new ArrayList<String>();
        short lockSetVal = this.lockSet.get();
        for (Resource value : Resource.values()) {
            if (!value.isLevelLocked(lockSetVal)) continue;
            currentLocks.add(value.getName());
        }
        return currentLocks;
    }

    @Override
    public boolean acquireMultiUserLock(String firstUser, String secondUser) {
        Resource resource = Resource.USER_LOCK;
        if (!resource.canLock(this.lockSet.get())) {
            String errorMessage = this.getErrorMessage(resource);
            LOG.error(errorMessage);
            throw new RuntimeException(errorMessage);
        }
        Striped<ReadWriteLock> striped = this.stripedLockByResource.get((Object)Resource.USER_LOCK);
        Iterable locks = striped.bulkGet(Arrays.asList(firstUser, secondUser));
        for (ReadWriteLock lock : locks) {
            lock.writeLock().lock();
        }
        this.lockSet.set(resource.setLock(this.lockSet.get()));
        return true;
    }

    @Override
    public void releaseMultiUserLock(String firstUser, String secondUser) {
        Striped<ReadWriteLock> striped = this.stripedLockByResource.get((Object)Resource.USER_LOCK);
        Iterable locks = striped.bulkGet(Arrays.asList(firstUser, secondUser));
        for (ReadWriteLock lock : locks) {
            lock.writeLock().unlock();
        }
        this.lockSet.set(Resource.USER_LOCK.clearLock(this.lockSet.get()));
    }

    @Override
    public OMLockDetails releaseWriteLock(Resource resource, String ... keys) {
        return this.releaseLock(resource, false, keys);
    }

    @Override
    public OMLockDetails releaseReadLock(Resource resource, String ... keys) {
        return this.releaseLock(resource, true, keys);
    }

    private OMLockDetails releaseLock(Resource resource, boolean isReadLock, String ... keys) {
        this.omLockDetails.get().clear();
        ReentrantReadWriteLock lock = this.getLock(resource, keys);
        if (isReadLock) {
            lock.readLock().unlock();
            this.updateReadUnlockMetrics(resource, lock);
        } else {
            boolean isWriteLocked = lock.isWriteLockedByCurrentThread();
            lock.writeLock().unlock();
            this.updateWriteUnlockMetrics(resource, lock, isWriteLocked);
        }
        this.lockSet.set(resource.clearLock(this.lockSet.get()));
        return this.omLockDetails.get();
    }

    private void updateReadUnlockMetrics(Resource resource, ReentrantReadWriteLock lock) {
        if (lock.getReadHoldCount() == 0) {
            long readLockHeldTimeNanos = Time.monotonicNowNanos() - resource.getStartReadHeldTimeNanos();
            this.omLockMetrics.setReadLockHeldTimeMsStat(TimeUnit.NANOSECONDS.toMillis(readLockHeldTimeNanos));
            this.updateProcessingDetails(ProcessingDetails.Timing.LOCKSHARED, readLockHeldTimeNanos);
        }
    }

    private void updateWriteUnlockMetrics(Resource resource, ReentrantReadWriteLock lock, boolean isWriteLocked) {
        if (lock.getWriteHoldCount() == 0 && isWriteLocked) {
            long writeLockHeldTimeNanos = Time.monotonicNowNanos() - resource.getStartWriteHeldTimeNanos();
            this.omLockMetrics.setWriteLockHeldTimeMsStat(TimeUnit.NANOSECONDS.toMillis(writeLockHeldTimeNanos));
            this.updateProcessingDetails(ProcessingDetails.Timing.LOCKEXCLUSIVE, writeLockHeldTimeNanos);
        }
    }

    @Override
    @VisibleForTesting
    public int getReadHoldCount(Resource resource, String ... keys) {
        return this.getLock(resource, keys).getReadHoldCount();
    }

    @Override
    @VisibleForTesting
    public int getWriteHoldCount(Resource resource, String ... keys) {
        return this.getLock(resource, keys).getWriteHoldCount();
    }

    @Override
    @VisibleForTesting
    public boolean isWriteLockedByCurrentThread(Resource resource, String ... keys) {
        return this.getLock(resource, keys).isWriteLockedByCurrentThread();
    }

    @Override
    public void cleanup() {
        this.omLockMetrics.unRegister();
    }

    @Override
    public OMLockMetrics getOMLockMetrics() {
        return this.omLockMetrics;
    }

    private void updateProcessingDetails(ProcessingDetails.Timing type, long deltaNanos) {
        Server.Call call = (Server.Call)Server.getCurCall().get();
        if (call != null) {
            call.getProcessingDetails().add(type, deltaNanos, TimeUnit.NANOSECONDS);
        } else {
            switch (type) {
                case LOCKWAIT: {
                    this.omLockDetails.get().add(deltaNanos, OMLockDetails.LockOpType.WAIT);
                    break;
                }
                case LOCKSHARED: {
                    this.omLockDetails.get().add(deltaNanos, OMLockDetails.LockOpType.READ);
                    break;
                }
                case LOCKEXCLUSIVE: {
                    this.omLockDetails.get().add(deltaNanos, OMLockDetails.LockOpType.WRITE);
                    break;
                }
                default: {
                    LOG.error("Unsupported Timing type {}", (Object)type);
                }
            }
        }
    }

    public static enum Resource {
        S3_BUCKET_LOCK(0, "S3_BUCKET_LOCK"),
        VOLUME_LOCK(1, "VOLUME_LOCK"),
        BUCKET_LOCK(2, "BUCKET_LOCK"),
        USER_LOCK(3, "USER_LOCK"),
        S3_SECRET_LOCK(4, "S3_SECRET_LOCK"),
        KEY_PATH_LOCK(5, "KEY_PATH_LOCK"),
        PREFIX_LOCK(6, "PREFIX_LOCK"),
        SNAPSHOT_LOCK(7, "SNAPSHOT_LOCK");

        private byte lockLevel;
        private short mask;
        private short setMask;
        private String name;
        private final ThreadLocal<LockUsageInfo> readLockTimeStampNanos = ThreadLocal.withInitial(LockUsageInfo::new);
        private final ThreadLocal<LockUsageInfo> writeLockTimeStampNanos = ThreadLocal.withInitial(LockUsageInfo::new);

        void setStartReadHeldTimeNanos(long startReadHeldTimeNanos) {
            this.readLockTimeStampNanos.get().setStartReadHeldTimeNanos(startReadHeldTimeNanos);
        }

        void setStartWriteHeldTimeNanos(long startWriteHeldTimeNanos) {
            this.writeLockTimeStampNanos.get().setStartWriteHeldTimeNanos(startWriteHeldTimeNanos);
        }

        long getStartReadHeldTimeNanos() {
            long startReadHeldTimeNanos = this.readLockTimeStampNanos.get().getStartReadHeldTimeNanos();
            this.readLockTimeStampNanos.remove();
            return startReadHeldTimeNanos;
        }

        long getStartWriteHeldTimeNanos() {
            long startWriteHeldTimeNanos = this.writeLockTimeStampNanos.get().getStartWriteHeldTimeNanos();
            this.writeLockTimeStampNanos.remove();
            return startWriteHeldTimeNanos;
        }

        private Resource(byte pos, String name) {
            this.lockLevel = pos;
            this.mask = (short)(Math.pow(2.0, this.lockLevel + 1) - 1.0);
            this.setMask = (short)Math.pow(2.0, this.lockLevel);
            this.name = name;
        }

        boolean canLock(short lockSetVal) {
            if (((Resource.USER_LOCK.setMask & lockSetVal) == Resource.USER_LOCK.setMask || (Resource.S3_SECRET_LOCK.setMask & lockSetVal) == Resource.S3_SECRET_LOCK.setMask || (Resource.PREFIX_LOCK.setMask & lockSetVal) == Resource.PREFIX_LOCK.setMask) && this.setMask <= lockSetVal) {
                return false;
            }
            return lockSetVal <= this.mask;
        }

        short setLock(short lockSetVal) {
            return (short)(lockSetVal | this.setMask);
        }

        short clearLock(short lockSetVal) {
            return (short)(lockSetVal & ~this.setMask);
        }

        boolean isLevelLocked(short lockSetVal) {
            return (lockSetVal & this.setMask) == this.setMask;
        }

        String getName() {
            return this.name;
        }

        short getMask() {
            return this.mask;
        }
    }
}

