/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.om.ratis;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.utils.DBStoreHAManager;
import org.apache.hadoop.hdds.utils.TransactionInfo;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.common.ha.ratis.RatisSnapshotInfo;
import org.apache.hadoop.ozone.om.OzoneManager;
import org.apache.hadoop.ozone.om.OzoneManagerPrepareState;
import org.apache.hadoop.ozone.om.exceptions.OMException;
import org.apache.hadoop.ozone.om.helpers.OMRatisHelper;
import org.apache.hadoop.ozone.om.lock.OMLockDetails;
import org.apache.hadoop.ozone.om.ratis.OzoneManagerDoubleBuffer;
import org.apache.hadoop.ozone.om.ratis.OzoneManagerRatisServer;
import org.apache.hadoop.ozone.om.ratis.metrics.OzoneManagerStateMachineMetrics;
import org.apache.hadoop.ozone.om.ratis.utils.OzoneManagerRatisUtils;
import org.apache.hadoop.ozone.om.response.DummyOMClientResponse;
import org.apache.hadoop.ozone.om.response.OMClientResponse;
import org.apache.hadoop.ozone.protocol.proto.OzoneManagerProtocolProtos;
import org.apache.hadoop.ozone.protocolPB.OzoneManagerRequestHandler;
import org.apache.hadoop.ozone.protocolPB.RequestHandler;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.concurrent.HadoopExecutors;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.Message;
import org.apache.ratis.protocol.RaftClientRequest;
import org.apache.ratis.protocol.RaftGroupId;
import org.apache.ratis.protocol.RaftGroupMemberId;
import org.apache.ratis.protocol.RaftPeer;
import org.apache.ratis.protocol.RaftPeerId;
import org.apache.ratis.protocol.exceptions.StateMachineException;
import org.apache.ratis.server.RaftServer;
import org.apache.ratis.server.protocol.TermIndex;
import org.apache.ratis.server.storage.RaftStorage;
import org.apache.ratis.statemachine.SnapshotInfo;
import org.apache.ratis.statemachine.StateMachine;
import org.apache.ratis.statemachine.TransactionContext;
import org.apache.ratis.statemachine.impl.BaseStateMachine;
import org.apache.ratis.statemachine.impl.SimpleStateMachineStorage;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.apache.ratis.util.ExitUtils;
import org.apache.ratis.util.LifeCycle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OzoneManagerStateMachine
extends BaseStateMachine {
    public static final Logger LOG = LoggerFactory.getLogger(OzoneManagerStateMachine.class);
    private final SimpleStateMachineStorage storage = new SimpleStateMachineStorage();
    private final OzoneManagerRatisServer omRatisServer;
    private final OzoneManager ozoneManager;
    private RequestHandler handler;
    private RaftGroupId raftGroupId;
    private OzoneManagerDoubleBuffer ozoneManagerDoubleBuffer;
    private final RatisSnapshotInfo snapshotInfo;
    private final ExecutorService executorService;
    private final ExecutorService installSnapshotExecutor;
    private final boolean isTracingEnabled;
    private final AtomicInteger statePausedCount = new AtomicInteger(0);
    private final String threadPrefix;
    private ConcurrentMap<Long, Long> applyTransactionMap = new ConcurrentSkipListMap<Long, Long>();
    private ConcurrentMap<Long, Long> ratisTransactionMap = new ConcurrentSkipListMap<Long, Long>();
    private OzoneManagerStateMachineMetrics metrics;

    public OzoneManagerStateMachine(OzoneManagerRatisServer ratisServer, boolean isTracingEnabled) throws IOException {
        this.omRatisServer = ratisServer;
        this.isTracingEnabled = isTracingEnabled;
        this.ozoneManager = this.omRatisServer.getOzoneManager();
        this.snapshotInfo = this.ozoneManager.getSnapshotInfo();
        this.loadSnapshotInfoFromDB();
        this.threadPrefix = this.ozoneManager.getThreadNamePrefix();
        this.ozoneManagerDoubleBuffer = this.buildDoubleBufferForRatis();
        this.handler = new OzoneManagerRequestHandler(this.ozoneManager, this.ozoneManagerDoubleBuffer);
        ThreadFactory build = new ThreadFactoryBuilder().setDaemon(true).setNameFormat(String.valueOf(this.threadPrefix) + "OMStateMachineApplyTransactionThread - %d").build();
        this.executorService = HadoopExecutors.newSingleThreadExecutor((ThreadFactory)build);
        ThreadFactory installSnapshotThreadFactory = new ThreadFactoryBuilder().setNameFormat(String.valueOf(this.threadPrefix) + "InstallSnapshotThread").build();
        this.installSnapshotExecutor = HadoopExecutors.newSingleThreadExecutor((ThreadFactory)installSnapshotThreadFactory);
        this.metrics = OzoneManagerStateMachineMetrics.create();
    }

    public void initialize(RaftServer server, RaftGroupId id, RaftStorage raftStorage) throws IOException {
        this.getLifeCycle().startAndTransition(() -> {
            super.initialize(server, id, raftStorage);
            this.raftGroupId = id;
            this.storage.init(raftStorage);
        }, new Class[0]);
    }

    public synchronized void reinitialize() throws IOException {
        this.loadSnapshotInfoFromDB();
        if (this.getLifeCycleState() == LifeCycle.State.PAUSED) {
            this.unpause(this.getLastAppliedTermIndex().getIndex(), this.getLastAppliedTermIndex().getTerm());
        }
    }

    public SnapshotInfo getLatestSnapshot() {
        LOG.debug("Latest Snapshot Info {}", (Object)this.snapshotInfo);
        return this.snapshotInfo;
    }

    public void notifyLeaderChanged(RaftGroupMemberId groupMemberId, RaftPeerId newLeaderId) {
        this.ozoneManager.omHAMetricsInit(newLeaderId.toString());
    }

    public void notifyTermIndexUpdated(long currentTerm, long index) {
        this.computeAndUpdateLastAppliedIndex(index, currentTerm, null, false);
    }

    public void notifyConfigurationChanged(long term, long index, RaftProtos.RaftConfigurationProto newRaftConfiguration) {
        List newPeers = newRaftConfiguration.getPeersList();
        LOG.info("Received Configuration change notification from Ratis. New Peer list:\n{}", (Object)newPeers);
        ArrayList<String> newPeerIds = new ArrayList<String>();
        for (RaftProtos.RaftPeerProto raftPeerProto : newPeers) {
            newPeerIds.add(RaftPeerId.valueOf((ByteString)raftPeerProto.getId()).toString());
        }
        this.ozoneManager.updatePeerList(newPeerIds);
    }

    public void notifySnapshotInstalled(RaftProtos.InstallSnapshotResult result, long snapshotIndex, RaftPeer peer) {
        LOG.info("Receive notifySnapshotInstalled event {} for the peer: {} snapshotIndex: {}.", new Object[]{result, peer.getId(), snapshotIndex});
        switch (result) {
            case SUCCESS: 
            case SNAPSHOT_UNAVAILABLE: {
                if (!this.ozoneManager.getOmRatisServer().getServer().getPeer().equals((Object)peer)) break;
                this.ozoneManager.getOmSnapshotProvider().init();
                break;
            }
        }
    }

    public TransactionContext startTransaction(RaftClientRequest raftClientRequest) throws IOException {
        ByteString messageContent = raftClientRequest.getMessage().getContent();
        OzoneManagerProtocolProtos.OMRequest omRequest = OMRatisHelper.convertByteStringToOMRequest((ByteString)messageContent);
        Preconditions.checkArgument((boolean)raftClientRequest.getRaftGroupId().equals((Object)this.raftGroupId));
        try {
            this.handler.validateRequest(omRequest);
        }
        catch (IOException ioe) {
            TransactionContext ctxt = TransactionContext.newBuilder().setClientRequest(raftClientRequest).setStateMachine((StateMachine)this).setServerRole(RaftProtos.RaftPeerRole.LEADER).build();
            ctxt.setException((Exception)ioe);
            return ctxt;
        }
        return this.handleStartTransactionRequests(raftClientRequest, omRequest);
    }

    public TransactionContext preAppendTransaction(TransactionContext trx) throws IOException {
        OzoneManagerProtocolProtos.OMRequest request = (OzoneManagerProtocolProtos.OMRequest)trx.getStateMachineContext();
        OzoneManagerProtocolProtos.Type cmdType = request.getCmdType();
        OzoneManagerPrepareState prepareState = this.ozoneManager.getPrepareState();
        if (cmdType == OzoneManagerProtocolProtos.Type.Prepare) {
            UserGroupInformation userGroupInformation = UserGroupInformation.createRemoteUser((String)request.getUserInfo().getUserName());
            if (this.ozoneManager.getAclsEnabled() && !this.ozoneManager.isAdmin(userGroupInformation)) {
                String message = "Access denied for user " + userGroupInformation + ". " + "Superuser privilege is required to prepare ozone managers.";
                OMException cause = new OMException(message, OMException.ResultCodes.ACCESS_DENIED);
                throw new StateMachineException(message, (Throwable)cause, false);
            }
            prepareState.enablePrepareGate();
        }
        if (prepareState.requestAllowed(cmdType)) {
            return trx;
        }
        String message = "Cannot apply write request " + request.getCmdType().name() + " when OM is in prepare mode.";
        OMException cause = new OMException(message, OMException.ResultCodes.NOT_SUPPORTED_OPERATION_WHEN_PREPARED);
        throw new StateMachineException(message, (Throwable)cause, false);
    }

    public CompletableFuture<Message> applyTransaction(TransactionContext trx) {
        try {
            Object context = trx.getStateMachineContext();
            OzoneManagerProtocolProtos.OMRequest request = context != null ? (OzoneManagerProtocolProtos.OMRequest)context : OMRatisHelper.convertByteStringToOMRequest((ByteString)trx.getStateMachineLogEntry().getLogData());
            long trxLogIndex = trx.getLogEntry().getIndex();
            CompletableFuture<Message> ratisFuture = new CompletableFuture<Message>();
            this.applyTransactionMap.put(trxLogIndex, trx.getLogEntry().getTerm());
            this.ozoneManagerDoubleBuffer.acquireUnFlushedTransactions(1);
            CompletableFuture<OzoneManagerProtocolProtos.OMResponse> future = CompletableFuture.supplyAsync(() -> this.runCommand(request, trxLogIndex), this.executorService);
            future.thenApply(omResponse -> {
                if (!omResponse.getSuccess()) {
                    if (omResponse.getStatus() == OzoneManagerProtocolProtos.Status.INTERNAL_ERROR) {
                        this.terminate((OzoneManagerProtocolProtos.OMResponse)omResponse, OMException.ResultCodes.INTERNAL_ERROR);
                    } else if (omResponse.getStatus() == OzoneManagerProtocolProtos.Status.METADATA_ERROR) {
                        this.terminate((OzoneManagerProtocolProtos.OMResponse)omResponse, OMException.ResultCodes.METADATA_ERROR);
                    }
                }
                ratisFuture.complete(OMRatisHelper.convertResponseToMessage((OzoneManagerProtocolProtos.OMResponse)omResponse));
                return ratisFuture;
            });
            return ratisFuture;
        }
        catch (Exception e) {
            return OzoneManagerStateMachine.completeExceptionally(e);
        }
    }

    private void terminate(OzoneManagerProtocolProtos.OMResponse omResponse, OMException.ResultCodes resultCode) {
        OMException exception = new OMException(omResponse.getMessage(), resultCode);
        String errorMessage = "OM Ratis Server has received unrecoverable error, to avoid further DB corruption, terminating OM. Error Response received is:" + omResponse;
        ExitUtils.terminate((int)1, (String)errorMessage, (Throwable)exception, (Logger)LOG);
    }

    public CompletableFuture<Message> query(Message request) {
        try {
            OzoneManagerProtocolProtos.OMRequest omRequest = OMRatisHelper.convertByteStringToOMRequest((ByteString)request.getContent());
            return CompletableFuture.completedFuture(this.queryCommand(omRequest));
        }
        catch (IOException e) {
            return OzoneManagerStateMachine.completeExceptionally(e);
        }
    }

    public synchronized void pause() {
        LOG.info("OzoneManagerStateMachine is pausing");
        this.statePausedCount.incrementAndGet();
        if (this.getLifeCycleState() == LifeCycle.State.PAUSED) {
            return;
        }
        LifeCycle lc = this.getLifeCycle();
        if (lc.getCurrentState() != LifeCycle.State.NEW) {
            this.getLifeCycle().transition(LifeCycle.State.PAUSING);
            this.getLifeCycle().transition(LifeCycle.State.PAUSED);
        }
        this.ozoneManagerDoubleBuffer.stop();
    }

    public synchronized void unpause(long newLastAppliedSnaphsotIndex, long newLastAppliedSnapShotTermIndex) {
        LOG.info("OzoneManagerStateMachine is un-pausing");
        if (this.statePausedCount.decrementAndGet() == 0) {
            this.getLifeCycle().startAndTransition(() -> {
                this.ozoneManagerDoubleBuffer = this.buildDoubleBufferForRatis();
                this.handler.updateDoubleBuffer(this.ozoneManagerDoubleBuffer);
                this.setLastAppliedTermIndex(TermIndex.valueOf((long)newLastAppliedSnapShotTermIndex, (long)newLastAppliedSnaphsotIndex));
            }, new Class[0]);
        }
    }

    public OzoneManagerDoubleBuffer buildDoubleBufferForRatis() {
        int maxUnflushedTransactionSize = this.ozoneManager.getConfiguration().getInt("ozone.om.unflushed.transaction.max.count", 10000);
        return new OzoneManagerDoubleBuffer.Builder().setOmMetadataManager(this.ozoneManager.getMetadataManager()).setOzoneManagerRatisSnapShot(this::updateLastAppliedIndex).setmaxUnFlushedTransactionCount(maxUnflushedTransactionSize).setIndexToTerm(this::getTermForIndex).setThreadPrefix(this.threadPrefix).setS3SecretManager(this.ozoneManager.getS3SecretManager()).enableRatis(true).enableTracing(this.isTracingEnabled).build();
    }

    public long takeSnapshot() throws IOException {
        LOG.info("Current Snapshot Index {}", (Object)this.getLastAppliedTermIndex());
        TermIndex lastTermIndex = this.getLastAppliedTermIndex();
        long lastAppliedIndex = lastTermIndex.getIndex();
        this.snapshotInfo.updateTermIndex(lastTermIndex.getTerm(), lastAppliedIndex);
        TransactionInfo build = new TransactionInfo.Builder().setTransactionIndex(lastAppliedIndex).setCurrentTerm(lastTermIndex.getTerm()).build();
        Table txnInfoTable = this.ozoneManager.getMetadataManager().getTransactionInfoTable();
        txnInfoTable.put((Object)"#TRANSACTIONINFO", (Object)build);
        this.ozoneManager.getMetadataManager().getStore().flushDB();
        return lastAppliedIndex;
    }

    public CompletableFuture<TermIndex> notifyInstallSnapshotFromLeader(RaftProtos.RoleInfoProto roleInfoProto, TermIndex firstTermIndexInLog) {
        String leaderNodeId = RaftPeerId.valueOf((ByteString)roleInfoProto.getFollowerInfo().getLeaderInfo().getId().getId()).toString();
        LOG.info("Received install snapshot notification from OM leader: {} with term index: {}", (Object)leaderNodeId, (Object)firstTermIndexInLog);
        CompletableFuture<TermIndex> future = CompletableFuture.supplyAsync(() -> this.ozoneManager.installSnapshotFromLeader(leaderNodeId), this.installSnapshotExecutor);
        return future;
    }

    public void notifyNotLeader(Collection<TransactionContext> pendingEntries) throws IOException {
    }

    public String toStateMachineLogEntryString(RaftProtos.StateMachineLogEntryProto proto) {
        return OMRatisHelper.smProtoToString((RaftProtos.StateMachineLogEntryProto)proto);
    }

    public void close() throws IOException {
        LOG.info("StateMachine has shutdown. Shutdown OzoneManager if not already shutdown.");
        if (!this.ozoneManager.isStopped()) {
            this.ozoneManager.shutDown("OM state machine is shutdown by Ratis server");
        } else {
            this.stop();
        }
    }

    private TransactionContext handleStartTransactionRequests(RaftClientRequest raftClientRequest, OzoneManagerProtocolProtos.OMRequest omRequest) {
        return TransactionContext.newBuilder().setClientRequest(raftClientRequest).setStateMachine((StateMachine)this).setServerRole(RaftProtos.RaftPeerRole.LEADER).setLogData(raftClientRequest.getMessage().getContent()).setStateMachineContext((Object)omRequest).build();
    }

    private OzoneManagerProtocolProtos.OMResponse runCommand(OzoneManagerProtocolProtos.OMRequest request, long trxLogIndex) {
        try {
            OMClientResponse omClientResponse = this.handler.handleWriteRequest(request, trxLogIndex);
            OMLockDetails omLockDetails = omClientResponse.getOmLockDetails();
            OzoneManagerProtocolProtos.OMResponse omResponse = omClientResponse.getOMResponse();
            if (omLockDetails != null) {
                return omResponse.toBuilder().setOmLockDetails(omLockDetails.toProtobufBuilder()).build();
            }
            return omResponse;
        }
        catch (IOException e) {
            LOG.warn("Failed to write, Exception occurred ", (Throwable)e);
            return this.createErrorResponse(request, e, trxLogIndex);
        }
        catch (Throwable e) {
            String errorMessage = "Request " + request + " failed with exception";
            ExitUtils.terminate((int)1, (String)errorMessage, (Throwable)e, (Logger)LOG);
            return null;
        }
    }

    private OzoneManagerProtocolProtos.OMResponse createErrorResponse(OzoneManagerProtocolProtos.OMRequest omRequest, IOException exception, long trxIndex) {
        OzoneManagerProtocolProtos.OMResponse.Builder omResponseBuilder = OzoneManagerProtocolProtos.OMResponse.newBuilder().setStatus(OzoneManagerRatisUtils.exceptionToResponseStatus(exception)).setCmdType(omRequest.getCmdType()).setTraceID(omRequest.getTraceID()).setSuccess(false);
        if (exception.getMessage() != null) {
            omResponseBuilder.setMessage(exception.getMessage());
        }
        OzoneManagerProtocolProtos.OMResponse omResponse = omResponseBuilder.build();
        DummyOMClientResponse omClientResponse = new DummyOMClientResponse(omResponse);
        omClientResponse.setFlushFuture(this.ozoneManagerDoubleBuffer.add(omClientResponse, trxIndex));
        return omResponse;
    }

    public void updateLastAppliedIndex(List<Long> flushedEpochs) {
        Preconditions.checkArgument((flushedEpochs.size() > 0 ? 1 : 0) != 0);
        this.computeAndUpdateLastAppliedIndex(flushedEpochs.get(flushedEpochs.size() - 1), -1L, flushedEpochs, true);
    }

    private synchronized void computeAndUpdateLastAppliedIndex(long lastFlushedIndex, long currentTerm, List<Long> flushedEpochs, boolean checkMap) {
        if (checkMap) {
            ArrayList<Long> flushedTrans = new ArrayList<Long>(flushedEpochs);
            Long appliedTerm = null;
            long appliedIndex = -1L;
            long i = this.getLastAppliedTermIndex().getIndex() + 1L;
            while (true) {
                Long removed;
                if (flushedTrans.contains(i)) {
                    appliedIndex = i;
                    appliedTerm = removed = (Long)this.applyTransactionMap.remove(i);
                    flushedTrans.remove(i);
                } else if (this.ratisTransactionMap.containsKey(i)) {
                    appliedTerm = removed = (Long)this.ratisTransactionMap.remove(i);
                    appliedIndex = i;
                } else {
                    Iterator iterator = flushedTrans.iterator();
                    while (iterator.hasNext()) {
                        long epoch = (Long)iterator.next();
                        this.ratisTransactionMap.put(epoch, (Long)this.applyTransactionMap.remove(epoch));
                    }
                    if (!LOG.isDebugEnabled() || flushedTrans.isEmpty()) break;
                    LOG.debug("ComputeAndUpdateLastAppliedIndex due to SM added to map remaining {}", flushedTrans);
                    break;
                }
                ++i;
            }
            if (appliedTerm != null) {
                this.updateLastAppliedTermIndex(appliedTerm, appliedIndex);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("ComputeAndUpdateLastAppliedIndex due to SM is {}", (Object)this.getLastAppliedTermIndex());
                }
            }
        } else if (this.getLastAppliedTermIndex().getIndex() + 1L == lastFlushedIndex) {
            this.updateLastAppliedTermIndex(currentTerm, lastFlushedIndex);
            if (LOG.isDebugEnabled()) {
                LOG.debug("ComputeAndUpdateLastAppliedIndex due to notifyIndex {}", (Object)this.getLastAppliedTermIndex());
            }
        } else {
            this.ratisTransactionMap.put(lastFlushedIndex, currentTerm);
            if (LOG.isDebugEnabled()) {
                LOG.debug("ComputeAndUpdateLastAppliedIndex due to notifyIndex added to map. Passed Term {} index {}, where as lastApplied Index {}", new Object[]{currentTerm, lastFlushedIndex, this.getLastAppliedTermIndex()});
            }
        }
        this.metrics.updateApplyTransactionMapSize(this.applyTransactionMap.size());
        this.metrics.updateRatisTransactionMapSize(this.ratisTransactionMap.size());
    }

    public void loadSnapshotInfoFromDB() throws IOException {
        TransactionInfo transactionInfo = TransactionInfo.readTransactionInfo((DBStoreHAManager)this.ozoneManager.getMetadataManager());
        if (transactionInfo != null) {
            this.setLastAppliedTermIndex(TermIndex.valueOf((long)transactionInfo.getTerm(), (long)transactionInfo.getTransactionIndex()));
            this.snapshotInfo.updateTermIndex(transactionInfo.getTerm(), transactionInfo.getTransactionIndex());
        }
        LOG.info("LastAppliedIndex is set from TransactionInfo from OM DB as {}", (Object)this.getLastAppliedTermIndex());
    }

    private Message queryCommand(OzoneManagerProtocolProtos.OMRequest request) {
        OzoneManagerProtocolProtos.OMResponse response = this.handler.handleReadRequest(request);
        return OMRatisHelper.convertResponseToMessage((OzoneManagerProtocolProtos.OMResponse)response);
    }

    private static <T> CompletableFuture<T> completeExceptionally(Exception e) {
        CompletableFuture future = new CompletableFuture();
        future.completeExceptionally(e);
        return future;
    }

    @VisibleForTesting
    public void setHandler(OzoneManagerRequestHandler handler) {
        this.handler = handler;
    }

    @VisibleForTesting
    public OzoneManagerRequestHandler getHandler() {
        return (OzoneManagerRequestHandler)this.handler;
    }

    @VisibleForTesting
    public void setRaftGroupId(RaftGroupId raftGroupId) {
        this.raftGroupId = raftGroupId;
    }

    @VisibleForTesting
    public OzoneManagerStateMachineMetrics getMetrics() {
        return this.metrics;
    }

    public void stop() {
        this.ozoneManagerDoubleBuffer.stop();
        HadoopExecutors.shutdown((ExecutorService)this.executorService, (Logger)LOG, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        HadoopExecutors.shutdown((ExecutorService)this.installSnapshotExecutor, (Logger)LOG, (long)5L, (TimeUnit)TimeUnit.SECONDS);
        LOG.info("applyTransactionMap size {} ", (Object)this.applyTransactionMap.size());
        if (LOG.isDebugEnabled()) {
            LOG.debug("applyTransactionMap {}", (Object)this.applyTransactionMap.keySet().stream().map(Object::toString).collect(Collectors.joining(",")));
        }
        LOG.info("ratisTransactionMap size {}", (Object)this.ratisTransactionMap.size());
        if (LOG.isDebugEnabled()) {
            LOG.debug("ratisTransactionMap {}", (Object)this.ratisTransactionMap.keySet().stream().map(Object::toString).collect(Collectors.joining(",")));
        }
        if (this.metrics != null) {
            this.metrics.unRegister();
        }
    }

    @VisibleForTesting
    void addApplyTransactionTermIndex(long term, long index) {
        this.applyTransactionMap.put(index, term);
    }

    public long getTermForIndex(long transactionIndex) {
        return (Long)this.applyTransactionMap.get(transactionIndex);
    }

    public void awaitDoubleBufferFlush() throws InterruptedException {
        this.ozoneManagerDoubleBuffer.awaitFlush();
    }

    @VisibleForTesting
    public OzoneManagerDoubleBuffer getOzoneManagerDoubleBuffer() {
        return this.ozoneManagerDoubleBuffer;
    }
}

