/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.common.statemachine.commandhandler;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.utils.db.BatchOperation;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.helpers.BlockDeletingServiceMetrics;
import org.apache.hadoop.ozone.container.common.helpers.ChunkInfoList;
import org.apache.hadoop.ozone.container.common.helpers.DeletedContainerBlocksSummary;
import org.apache.hadoop.ozone.container.common.impl.ContainerSet;
import org.apache.hadoop.ozone.container.common.interfaces.Container;
import org.apache.hadoop.ozone.container.common.interfaces.DBHandle;
import org.apache.hadoop.ozone.container.common.statemachine.DatanodeConfiguration;
import org.apache.hadoop.ozone.container.common.statemachine.SCMConnectionManager;
import org.apache.hadoop.ozone.container.common.statemachine.StateContext;
import org.apache.hadoop.ozone.container.common.statemachine.commandhandler.CommandHandler;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainer;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.BlockUtils;
import org.apache.hadoop.ozone.container.metadata.DeleteTransactionStore;
import org.apache.hadoop.ozone.container.ozoneimpl.OzoneContainer;
import org.apache.hadoop.ozone.protocol.commands.CommandStatus;
import org.apache.hadoop.ozone.protocol.commands.DeleteBlockCommandStatus;
import org.apache.hadoop.ozone.protocol.commands.DeleteBlocksCommand;
import org.apache.hadoop.ozone.protocol.commands.SCMCommand;
import org.apache.hadoop.util.Daemon;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DeleteBlocksCommandHandler
implements CommandHandler {
    public static final Logger LOG = LoggerFactory.getLogger(DeleteBlocksCommandHandler.class);
    private final ContainerSet containerSet;
    private final ConfigurationSource conf;
    private int invocationCount;
    private long totalTime;
    private final ThreadPoolExecutor executor;
    private final LinkedBlockingQueue<DeleteCmdInfo> deleteCommandQueues;
    private final Daemon handlerThread;
    private final OzoneContainer ozoneContainer;
    private final BlockDeletingServiceMetrics blockDeleteMetrics;
    private final long tryLockTimeoutMs;
    private final Map<String, SchemaHandler> schemaHandlers;

    public DeleteBlocksCommandHandler(OzoneContainer container, ConfigurationSource conf, DatanodeConfiguration dnConf, String threadNamePrefix) {
        this.ozoneContainer = container;
        this.containerSet = container.getContainerSet();
        this.conf = conf;
        this.blockDeleteMetrics = BlockDeletingServiceMetrics.create();
        this.tryLockTimeoutMs = dnConf.getBlockDeleteMaxLockWaitTimeoutMs();
        this.schemaHandlers = new HashMap<String, SchemaHandler>();
        this.schemaHandlers.put("1", this::markBlocksForDeletionSchemaV1);
        this.schemaHandlers.put("2", this::markBlocksForDeletionSchemaV2);
        this.schemaHandlers.put("3", this::markBlocksForDeletionSchemaV3);
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat(threadNamePrefix + "DeleteBlocksCommandHandlerThread-%d").build();
        this.executor = (ThreadPoolExecutor)Executors.newFixedThreadPool(dnConf.getBlockDeleteThreads(), threadFactory);
        this.deleteCommandQueues = new LinkedBlockingQueue(dnConf.getBlockDeleteQueueLimit());
        long interval = dnConf.getBlockDeleteCommandWorkerInterval().toMillis();
        this.handlerThread = new Daemon((Runnable)new DeleteCmdWorker(interval));
        this.handlerThread.start();
    }

    @Override
    public void handle(SCMCommand command, OzoneContainer container, StateContext context, SCMConnectionManager connectionManager) {
        if (command.getType() != StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.deleteBlocksCommand) {
            LOG.warn("Skipping handling command, expected command type {} but found {}", (Object)StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.deleteBlocksCommand, (Object)command.getType());
            return;
        }
        DeleteCmdInfo cmd = new DeleteCmdInfo((DeleteBlocksCommand)command, container, context, connectionManager);
        try {
            this.deleteCommandQueues.add(cmd);
        }
        catch (IllegalStateException e) {
            String dnId = context.getParent().getDatanodeDetails().getUuidString();
            Consumer<CommandStatus> updateFailure = cmdStatus -> {
                cmdStatus.markAsFailed();
                StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto emptyACK = StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.newBuilder().setDnId(dnId).build();
                ((DeleteBlockCommandStatus)cmdStatus).setBlocksDeletionAck(emptyACK);
            };
            this.updateCommandStatus(cmd.getContext(), cmd.getCmd(), updateFailure, LOG);
            LOG.warn("Command is discarded because of the command queue is full");
        }
    }

    @Override
    public int getQueuedCount() {
        return this.deleteCommandQueues.size();
    }

    @Override
    public int getThreadPoolMaxPoolSize() {
        return this.executor.getMaximumPoolSize();
    }

    @Override
    public int getThreadPoolActivePoolSize() {
        return this.executor.getActiveCount();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processCmd(DeleteCmdInfo cmd) {
        boolean executedStatus;
        StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto deleteAck;
        LOG.debug("Processing block deletion command.");
        StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto blockDeletionACK = null;
        long startTime = Time.monotonicNow();
        boolean cmdExecuted = false;
        try {
            List<StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction> containerBlocks = cmd.getCmd().blocksTobeDeleted();
            this.blockDeleteMetrics.incrReceivedTransactionCount(containerBlocks.size());
            DeletedContainerBlocksSummary summary = DeletedContainerBlocksSummary.getFrom(containerBlocks);
            LOG.info("Summary of deleting container blocks, numOfTransactions={}, numOfContainers={}, numOfBlocks={}, commandId={}.", new Object[]{summary.getNumOfTxs(), summary.getNumOfContainers(), summary.getNumOfBlocks(), cmd.getCmd().getId()});
            if (LOG.isDebugEnabled()) {
                LOG.debug("Start to delete container blocks, TXIDs={}", (Object)summary.getTxIDSummary());
            }
            this.blockDeleteMetrics.incrReceivedContainerCount(summary.getNumOfContainers());
            this.blockDeleteMetrics.incrReceivedRetryTransactionCount(summary.getNumOfRetryTxs());
            this.blockDeleteMetrics.incrReceivedBlockCount(summary.getNumOfBlocks());
            List<StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult> results = this.executeCmdWithRetry(containerBlocks);
            StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.Builder resultBuilder = StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.newBuilder().addAllResults(results);
            resultBuilder.setDnId(cmd.getContext().getParent().getDatanodeDetails().getUuid().toString());
            blockDeletionACK = resultBuilder.build();
            if (!containerBlocks.isEmpty() && LOG.isDebugEnabled()) {
                LOG.debug("Sending following block deletion ACK to SCM");
                for (StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult result : blockDeletionACK.getResultsList()) {
                    LOG.debug("TxId = {} : ContainerId = {} : {}", new Object[]{result.getTxID(), result.getContainerID(), result.getSuccess()});
                }
            }
            cmdExecuted = true;
            deleteAck = blockDeletionACK;
            executedStatus = cmdExecuted;
        }
        catch (Throwable throwable) {
            StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto deleteAck2 = blockDeletionACK;
            boolean executedStatus2 = cmdExecuted;
            Consumer<CommandStatus> statusUpdater = cmdStatus -> {
                if (executedStatus2) {
                    cmdStatus.markAsExecuted();
                } else {
                    cmdStatus.markAsFailed();
                }
                ((DeleteBlockCommandStatus)cmdStatus).setBlocksDeletionAck(deleteAck2);
            };
            this.updateCommandStatus(cmd.getContext(), cmd.getCmd(), statusUpdater, LOG);
            long endTime = Time.monotonicNow();
            this.totalTime += endTime - startTime;
            ++this.invocationCount;
            throw throwable;
        }
        Consumer<CommandStatus> statusUpdater = cmdStatus -> {
            if (executedStatus2) {
                cmdStatus.markAsExecuted();
            } else {
                cmdStatus.markAsFailed();
            }
            ((DeleteBlockCommandStatus)cmdStatus).setBlocksDeletionAck(deleteAck2);
        };
        this.updateCommandStatus(cmd.getContext(), cmd.getCmd(), statusUpdater, LOG);
        long endTime = Time.monotonicNow();
        this.totalTime += endTime - startTime;
        ++this.invocationCount;
    }

    @VisibleForTesting
    public List<StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult> executeCmdWithRetry(List<StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction> transactions) {
        ArrayList<StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult> results = new ArrayList<StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult>(transactions.size());
        HashMap idToTransaction = new HashMap(transactions.size());
        transactions.forEach(tx -> idToTransaction.put(tx.getTxID(), tx));
        ArrayList<StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction> retryTransaction = new ArrayList<StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction>();
        List<Future<DeleteBlockTransactionExecutionResult>> futures = this.submitTasks(transactions);
        this.handleTasksResults(futures, result -> {
            if (result.isLockAcquisitionFailed()) {
                retryTransaction.add((StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction)idToTransaction.get(result.getResult().getTxID()));
            } else {
                results.add(result.getResult());
            }
        });
        idToTransaction.clear();
        if (!retryTransaction.isEmpty()) {
            futures = this.submitTasks(retryTransaction);
            this.handleTasksResults(futures, result -> {
                if (result.isLockAcquisitionFailed()) {
                    this.blockDeleteMetrics.incrTotalLockTimeoutTransactionCount();
                }
                results.add(result.getResult());
            });
        }
        return results;
    }

    @VisibleForTesting
    public List<Future<DeleteBlockTransactionExecutionResult>> submitTasks(List<StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction> deletedBlocksTransactions) {
        ArrayList<Future<DeleteBlockTransactionExecutionResult>> futures = new ArrayList<Future<DeleteBlockTransactionExecutionResult>>(deletedBlocksTransactions.size());
        for (StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction tx : deletedBlocksTransactions) {
            Future<DeleteBlockTransactionExecutionResult> future = this.executor.submit(new ProcessTransactionTask(tx));
            futures.add(future);
        }
        return futures;
    }

    public void handleTasksResults(List<Future<DeleteBlockTransactionExecutionResult>> futures, Consumer<DeleteBlockTransactionExecutionResult> handler) {
        futures.forEach(f -> {
            try {
                DeleteBlockTransactionExecutionResult result = (DeleteBlockTransactionExecutionResult)f.get();
                handler.accept(result);
            }
            catch (InterruptedException | ExecutionException e) {
                LOG.error("task failed.", (Throwable)e);
                Thread.currentThread().interrupt();
            }
        });
    }

    private void markBlocksForDeletionSchemaV3(KeyValueContainerData containerData, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction delTX) throws IOException {
        DeletionMarker schemaV3Marker = (table, batch, tid, txn) -> {
            Table delTxTable = table;
            delTxTable.putWithBatch(batch, (Object)containerData.getDeleteTxnKey(tid), (Object)txn);
        };
        this.markBlocksForDeletionTransaction(containerData, delTX, delTX.getTxID(), schemaV3Marker);
    }

    private void markBlocksForDeletionSchemaV2(KeyValueContainerData containerData, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction delTX) throws IOException {
        DeletionMarker schemaV2Marker = (table, batch, tid, txn) -> {
            Table delTxTable = table;
            delTxTable.putWithBatch(batch, (Object)tid, (Object)txn);
        };
        this.markBlocksForDeletionTransaction(containerData, delTX, delTX.getTxID(), schemaV2Marker);
    }

    private void markBlocksForDeletionTransaction(KeyValueContainerData containerData, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction delTX, long txnID, DeletionMarker marker) throws IOException {
        int newDeletionBlocks = 0;
        long containerId = delTX.getContainerID();
        this.logDeleteTransaction(containerId, containerData, delTX);
        try (DBHandle containerDB = BlockUtils.getDB(containerData, this.conf);){
            DeleteTransactionStore store = (DeleteTransactionStore)((Object)containerDB.getStore());
            Table delTxTable = store.getDeleteTransactionTable();
            try (BatchOperation batch = containerDB.getStore().getBatchHandler().initBatchOperation();){
                marker.apply(delTxTable, batch, txnID, delTX);
                this.updateMetaData(containerData, delTX, newDeletionBlocks += delTX.getLocalIDList().size(), containerDB, batch);
                containerDB.getStore().getBatchHandler().commitBatchOperation(batch);
            }
        }
        this.blockDeleteMetrics.incrMarkedBlockCount(delTX.getLocalIDCount());
    }

    private void markBlocksForDeletionSchemaV1(KeyValueContainerData containerData, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction delTX) throws IOException {
        long containerId = delTX.getContainerID();
        this.logDeleteTransaction(containerId, containerData, delTX);
        int newDeletionBlocks = 0;
        try (DBHandle containerDB = BlockUtils.getDB(containerData, this.conf);){
            Table<String, BlockData> blockDataTable = containerDB.getStore().getBlockDataTable();
            Table<String, ChunkInfoList> deletedBlocksTable = containerDB.getStore().getDeletedBlocksTable();
            try (BatchOperation batch = containerDB.getStore().getBatchHandler().initBatchOperation();){
                for (Long blkLong : delTX.getLocalIDList()) {
                    String blk = containerData.getBlockKey(blkLong);
                    BlockData blkInfo = (BlockData)blockDataTable.get((Object)blk);
                    if (blkInfo != null) {
                        String deletingKey = containerData.getDeletingBlockKey(blkLong);
                        if (blockDataTable.get((Object)deletingKey) != null || deletedBlocksTable.get((Object)blk) != null) {
                            if (!LOG.isDebugEnabled()) continue;
                            LOG.debug("Ignoring delete for block {} in container {}. Entry already added.", (Object)blkLong, (Object)containerId);
                            continue;
                        }
                        blockDataTable.putWithBatch(batch, (Object)deletingKey, (Object)blkInfo);
                        blockDataTable.deleteWithBatch(batch, (Object)blk);
                        ++newDeletionBlocks;
                        if (!LOG.isDebugEnabled()) continue;
                        LOG.debug("Transited Block {} to DELETING state in container {}", (Object)blkLong, (Object)containerId);
                        continue;
                    }
                    try {
                        Container<?> container = this.containerSet.getContainer(containerId);
                        this.ozoneContainer.getDispatcher().getHandler(container.getContainerType()).deleteUnreferenced(container, blkLong);
                    }
                    catch (IOException e) {
                        LOG.error("Failed to delete files for unreferenced block {} of container {}", new Object[]{blkLong, containerId, e});
                    }
                }
                this.updateMetaData(containerData, delTX, newDeletionBlocks, containerDB, batch);
                containerDB.getStore().getBatchHandler().commitBatchOperation(batch);
            }
            catch (IOException e) {
                throw new IOException("Failed to delete blocks for TXID = " + delTX.getTxID(), e);
            }
            this.blockDeleteMetrics.incrMarkedBlockCount(delTX.getLocalIDCount());
        }
    }

    private void updateMetaData(KeyValueContainerData containerData, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction delTX, int newDeletionBlocks, DBHandle containerDB, BatchOperation batchOperation) throws IOException {
        if (newDeletionBlocks > 0) {
            Table<String, Long> metadataTable = containerDB.getStore().getMetadataTable();
            if (delTX.getTxID() > containerData.getDeleteTransactionId()) {
                metadataTable.putWithBatch(batchOperation, (Object)containerData.getLatestDeleteTxnKey(), (Object)delTX.getTxID());
            }
            long pendingDeleteBlocks = containerData.getNumPendingDeletionBlocks() + (long)newDeletionBlocks;
            metadataTable.putWithBatch(batchOperation, (Object)containerData.getPendingDeleteBlockCountKey(), (Object)pendingDeleteBlocks);
            containerData.updateDeleteTransactionId(delTX.getTxID());
            containerData.incrPendingDeletionBlocks(newDeletionBlocks);
        }
    }

    private void logDeleteTransaction(long containerId, KeyValueContainerData containerData, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction delTX) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Processing Container : {}, DB path : {}, transaction {}", new Object[]{containerId, containerData.getMetadataPath(), delTX.getTxID()});
        }
        if (delTX.getTxID() <= containerData.getDeleteTransactionId()) {
            this.blockDeleteMetrics.incOutOfOrderDeleteBlockTransactionCount();
            LOG.info(String.format("Delete blocks for containerId: %d is either received out of order or retried, %d <= %d", containerId, delTX.getTxID(), containerData.getDeleteTransactionId()));
        }
    }

    @Override
    public StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type getCommandType() {
        return StorageContainerDatanodeProtocolProtos.SCMCommandProto.Type.deleteBlocksCommand;
    }

    @Override
    public int getInvocationCount() {
        return this.invocationCount;
    }

    @Override
    public long getAverageRunTime() {
        if (this.invocationCount > 0) {
            return this.totalTime / (long)this.invocationCount;
        }
        return 0L;
    }

    @Override
    public long getTotalRunTime() {
        return this.totalTime;
    }

    @Override
    public void stop() {
        if (this.executor != null) {
            try {
                this.executor.shutdown();
                if (!this.executor.awaitTermination(3L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                }
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
        }
        if (this.handlerThread != null) {
            try {
                this.handlerThread.interrupt();
                this.handlerThread.join(3000L);
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
        }
    }

    @VisibleForTesting
    public Map<String, SchemaHandler> getSchemaHandlers() {
        return this.schemaHandlers;
    }

    @VisibleForTesting
    public BlockDeletingServiceMetrics getBlockDeleteMetrics() {
        return this.blockDeleteMetrics;
    }

    @VisibleForTesting
    public ThreadPoolExecutor getExecutor() {
        return this.executor;
    }

    public void setPoolSize(int size) {
        if (size <= 0) {
            throw new IllegalArgumentException("Pool size must be positive.");
        }
        int currentCorePoolSize = this.executor.getCorePoolSize();
        if (size > currentCorePoolSize) {
            this.executor.setMaximumPoolSize(size);
            this.executor.setCorePoolSize(size);
        } else {
            this.executor.setCorePoolSize(size);
            this.executor.setMaximumPoolSize(size);
        }
    }

    public static interface SchemaHandler {
        public void handle(KeyValueContainerData var1, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction var2) throws IOException;
    }

    private static interface DeletionMarker {
        public void apply(Table<?, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction> var1, BatchOperation var2, long var3, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction var5) throws IOException;
    }

    public final class ProcessTransactionTask
    implements Callable<DeleteBlockTransactionExecutionResult> {
        private StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction tx;

        public ProcessTransactionTask(StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction transaction) {
            this.tx = transaction;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        @Override
        public DeleteBlockTransactionExecutionResult call() {
            StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult.Builder txResultBuilder = StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult.newBuilder();
            txResultBuilder.setTxID(this.tx.getTxID());
            long containerId = this.tx.getContainerID();
            boolean lockAcquisitionFailed = false;
            try {
                Container<?> cont = DeleteBlocksCommandHandler.this.containerSet.getContainer(containerId);
                if (cont == null) {
                    throw new StorageContainerException("Unable to find the container " + containerId, ContainerProtos.Result.CONTAINER_NOT_FOUND);
                }
                ContainerProtos.ContainerType containerType = cont.getContainerType();
                switch (containerType) {
                    case KeyValueContainer: {
                        KeyValueContainer keyValueContainer = (KeyValueContainer)cont;
                        KeyValueContainerData containerData = keyValueContainer.getContainerData();
                        if (!keyValueContainer.writeLockTryLock(DeleteBlocksCommandHandler.this.tryLockTimeoutMs, TimeUnit.MILLISECONDS)) {
                            lockAcquisitionFailed = true;
                            txResultBuilder.setContainerID(containerId).setSuccess(false);
                            return new DeleteBlockTransactionExecutionResult(txResultBuilder.build(), lockAcquisitionFailed);
                        }
                        try {
                            String schemaVersion = containerData.getSupportedSchemaVersionOrDefault();
                            if (!DeleteBlocksCommandHandler.this.getSchemaHandlers().containsKey(schemaVersion)) throw new UnsupportedOperationException("Only schema version 1,2,3 are supported.");
                            ((SchemaHandler)DeleteBlocksCommandHandler.this.schemaHandlers.get(schemaVersion)).handle(containerData, this.tx);
                        }
                        finally {
                            keyValueContainer.writeUnlock();
                        }
                        txResultBuilder.setContainerID(containerId).setSuccess(true);
                        return new DeleteBlockTransactionExecutionResult(txResultBuilder.build(), lockAcquisitionFailed);
                    }
                }
                LOG.error("Delete Blocks Command Handler is not implemented for containerType {}", (Object)containerType);
                return new DeleteBlockTransactionExecutionResult(txResultBuilder.build(), lockAcquisitionFailed);
            }
            catch (IOException e) {
                LOG.warn("Failed to delete blocks for container={}, TXID={}", new Object[]{this.tx.getContainerID(), this.tx.getTxID(), e});
                txResultBuilder.setContainerID(containerId).setSuccess(false);
                return new DeleteBlockTransactionExecutionResult(txResultBuilder.build(), lockAcquisitionFailed);
            }
            catch (InterruptedException e) {
                LOG.warn("InterruptedException while deleting blocks for container={}, TXID={}", new Object[]{this.tx.getContainerID(), this.tx.getTxID(), e});
                Thread.currentThread().interrupt();
                txResultBuilder.setContainerID(containerId).setSuccess(false);
            }
            return new DeleteBlockTransactionExecutionResult(txResultBuilder.build(), lockAcquisitionFailed);
        }
    }

    public final class DeleteCmdWorker
    implements Runnable {
        private long intervalInMs;

        public DeleteCmdWorker(long interval) {
            this.intervalInMs = interval;
        }

        @VisibleForTesting
        public long getInterval() {
            return this.intervalInMs;
        }

        @Override
        public void run() {
            while (true) {
                if (!DeleteBlocksCommandHandler.this.deleteCommandQueues.isEmpty()) {
                    DeleteCmdInfo cmd = (DeleteCmdInfo)DeleteBlocksCommandHandler.this.deleteCommandQueues.poll();
                    try {
                        DeleteBlocksCommandHandler.this.processCmd(cmd);
                    }
                    catch (Throwable e) {
                        LOG.error("taskProcess failed.", e);
                    }
                    continue;
                }
                try {
                    Thread.sleep(this.intervalInMs);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return;
                }
            }
        }
    }

    public static final class DeleteBlockTransactionExecutionResult {
        private final StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult result;
        private final boolean lockAcquisitionFailed;

        public DeleteBlockTransactionExecutionResult(StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult result, boolean lockAcquisitionFailed) {
            this.result = result;
            this.lockAcquisitionFailed = lockAcquisitionFailed;
        }

        public StorageContainerDatanodeProtocolProtos.ContainerBlocksDeletionACKProto.DeleteBlockTransactionResult getResult() {
            return this.result;
        }

        public boolean isLockAcquisitionFailed() {
            return this.lockAcquisitionFailed;
        }
    }

    public static final class DeleteCmdInfo {
        private DeleteBlocksCommand cmd;
        private StateContext context;
        private OzoneContainer container;
        private SCMConnectionManager connectionManager;

        public DeleteCmdInfo(DeleteBlocksCommand command, OzoneContainer container, StateContext context, SCMConnectionManager connectionManager) {
            this.cmd = command;
            this.context = context;
            this.container = container;
            this.connectionManager = connectionManager;
        }

        public DeleteBlocksCommand getCmd() {
            return this.cmd;
        }

        public StateContext getContext() {
            return this.context;
        }

        public OzoneContainer getContainer() {
            return this.container;
        }

        public SCMConnectionManager getConnectionManager() {
            return this.connectionManager;
        }
    }
}

