/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.container.keyvalue;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.List;
import java.util.function.BooleanSupplier;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.protocol.proto.StorageContainerDatanodeProtocolProtos;
import org.apache.hadoop.hdds.utils.db.Table;
import org.apache.hadoop.hdds.utils.db.TableIterator;
import org.apache.hadoop.ozone.container.common.helpers.BlockData;
import org.apache.hadoop.ozone.container.common.impl.ContainerData;
import org.apache.hadoop.ozone.container.common.interfaces.BlockIterator;
import org.apache.hadoop.ozone.container.common.interfaces.ContainerInspector;
import org.apache.hadoop.ozone.container.keyvalue.KeyValueContainerData;
import org.apache.hadoop.ozone.container.keyvalue.helpers.KeyValueContainerUtil;
import org.apache.hadoop.ozone.container.metadata.DatanodeStore;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaThreeImpl;
import org.apache.hadoop.ozone.container.metadata.DatanodeStoreSchemaTwoImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyValueContainerMetadataInspector
implements ContainerInspector {
    public static final Logger LOG = LoggerFactory.getLogger(KeyValueContainerMetadataInspector.class);
    public static final Logger REPORT_LOG = LoggerFactory.getLogger((String)"ContainerMetadataInspectorReport");
    public static final String SYSTEM_PROPERTY = "ozone.datanode.container.metadata.inspector";
    private Mode mode;

    public KeyValueContainerMetadataInspector(Mode mode) {
        this.mode = mode;
    }

    public KeyValueContainerMetadataInspector() {
        this.mode = Mode.OFF;
    }

    @Override
    public boolean load() {
        String propertyValue = System.getProperty(SYSTEM_PROPERTY);
        boolean propertyPresent = propertyValue != null && !propertyValue.isEmpty();
        boolean propertyValid = false;
        if (propertyPresent) {
            if (propertyValue.equals(Mode.REPAIR.toString())) {
                this.mode = Mode.REPAIR;
                propertyValid = true;
            } else if (propertyValue.equals(Mode.INSPECT.toString())) {
                this.mode = Mode.INSPECT;
                propertyValid = true;
            }
            if (propertyValid) {
                LOG.info("Container metadata inspector enabled in {} mode. Reportwill be output to the {} log.", (Object)this.mode, (Object)REPORT_LOG.getName());
            } else {
                this.mode = Mode.OFF;
                LOG.error("{} system property specified with invalid mode {}. Valid options are {} and {}. Container metadata inspection will not be run.", new Object[]{SYSTEM_PROPERTY, propertyValue, Mode.REPAIR, Mode.INSPECT});
            }
        } else {
            this.mode = Mode.OFF;
        }
        return propertyPresent && propertyValid;
    }

    @Override
    public void unload() {
        this.mode = Mode.OFF;
    }

    @Override
    public boolean isReadOnly() {
        return this.mode != Mode.REPAIR;
    }

    @Override
    public void process(ContainerData containerData, DatanodeStore store) {
        this.process(containerData, store, REPORT_LOG);
    }

    public String process(ContainerData containerData, DatanodeStore store, Logger log) {
        if (this.mode == Mode.OFF) {
            return null;
        }
        KeyValueContainerData kvData = null;
        if (!(containerData instanceof KeyValueContainerData)) {
            LOG.error("This inspector only works on KeyValueContainers. Inspection will not be run for container {}", (Object)containerData.getContainerID());
            return null;
        }
        kvData = (KeyValueContainerData)containerData;
        JsonObject containerJson = KeyValueContainerMetadataInspector.inspectContainer(kvData, store);
        boolean correct = this.checkAndRepair(containerJson, kvData, store);
        Gson gson = new GsonBuilder().setPrettyPrinting().serializeNulls().create();
        String jsonReport = gson.toJson((JsonElement)containerJson);
        if (log != null) {
            if (correct) {
                log.trace(jsonReport);
            } else {
                log.error(jsonReport);
            }
        }
        return jsonReport;
    }

    static JsonObject inspectContainer(KeyValueContainerData containerData, DatanodeStore store) {
        JsonObject containerJson = new JsonObject();
        try {
            containerJson.addProperty("containerID", (Number)containerData.getContainerID());
            String schemaVersion = containerData.getSchemaVersion();
            containerJson.addProperty("schemaVersion", schemaVersion);
            containerJson.addProperty("containerState", containerData.getState().toString());
            containerJson.addProperty("currentDatanodeID", containerData.getVolume().getDatanodeUuid());
            containerJson.addProperty("originDatanodeID", containerData.getOriginNodeId());
            Table<String, Long> metadataTable = store.getMetadataTable();
            JsonObject dBMetadata = KeyValueContainerMetadataInspector.getDBMetadataJson(metadataTable, containerData);
            containerJson.add("dBMetadata", (JsonElement)dBMetadata);
            JsonObject aggregates = KeyValueContainerMetadataInspector.getAggregateValues(store, containerData, schemaVersion);
            containerJson.add("aggregates", (JsonElement)aggregates);
            JsonObject chunksDirectory = KeyValueContainerMetadataInspector.getChunksDirectoryJson(new File(containerData.getChunksPath()));
            containerJson.add("chunksDirectory", (JsonElement)chunksDirectory);
        }
        catch (IOException ex) {
            LOG.error("Inspecting container {} failed", (Object)containerData.getContainerID(), (Object)ex);
        }
        return containerJson;
    }

    static JsonObject getDBMetadataJson(Table<String, Long> metadataTable, KeyValueContainerData containerData) throws IOException {
        JsonObject dBMetadata = new JsonObject();
        dBMetadata.addProperty("#BLOCKCOUNT", (Number)metadataTable.get((Object)containerData.getBlockCountKey()));
        dBMetadata.addProperty("#BYTESUSED", (Number)metadataTable.get((Object)containerData.getBytesUsedKey()));
        dBMetadata.addProperty("#PENDINGDELETEBLOCKCOUNT", (Number)metadataTable.get((Object)containerData.getPendingDeleteBlockCountKey()));
        dBMetadata.addProperty("#delTX", (Number)metadataTable.get((Object)containerData.getLatestDeleteTxnKey()));
        dBMetadata.addProperty("#BCSID", (Number)metadataTable.get((Object)containerData.getBcsIdKey()));
        return dBMetadata;
    }

    static JsonObject getAggregateValues(DatanodeStore store, KeyValueContainerData containerData, String schemaVersion) throws IOException {
        PendingDelete pendingDelete;
        JsonObject aggregates = new JsonObject();
        long usedBytesTotal = 0L;
        long blockCountTotal = 0L;
        try (BlockIterator<BlockData> blockIter = store.getBlockIterator(containerData.getContainerID(), containerData.getUnprefixedKeyFilter());){
            while (blockIter.hasNext()) {
                ++blockCountTotal;
                usedBytesTotal += KeyValueContainerMetadataInspector.getBlockLength(blockIter.nextBlock());
            }
        }
        if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "1")) {
            long pendingDeleteBlockCountTotal = 0L;
            long pendingDeleteBytes = 0L;
            try (BlockIterator<BlockData> blockIter = store.getBlockIterator(containerData.getContainerID(), containerData.getDeletingBlockKeyFilter());){
                while (blockIter.hasNext()) {
                    ++blockCountTotal;
                    ++pendingDeleteBlockCountTotal;
                    long bytes = KeyValueContainerMetadataInspector.getBlockLength(blockIter.nextBlock());
                    usedBytesTotal += bytes;
                    pendingDeleteBytes += bytes;
                }
            }
            pendingDelete = new PendingDelete(pendingDeleteBlockCountTotal, pendingDeleteBytes);
        } else if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "2")) {
            DatanodeStoreSchemaTwoImpl schemaTwoStore = (DatanodeStoreSchemaTwoImpl)store;
            pendingDelete = KeyValueContainerMetadataInspector.countPendingDeletesSchemaV2(schemaTwoStore, containerData);
        } else if (KeyValueContainerUtil.isSameSchemaVersion(schemaVersion, "3")) {
            DatanodeStoreSchemaThreeImpl schemaThreeStore = (DatanodeStoreSchemaThreeImpl)store;
            pendingDelete = KeyValueContainerMetadataInspector.countPendingDeletesSchemaV3(schemaThreeStore, containerData);
        } else {
            throw new IOException("Failed to process deleted blocks for unknown container schema " + schemaVersion);
        }
        aggregates.addProperty("blockCount", (Number)blockCountTotal);
        aggregates.addProperty("usedBytes", (Number)usedBytesTotal);
        pendingDelete.addToJson(aggregates);
        return aggregates;
    }

    static JsonObject getChunksDirectoryJson(File chunksDir) throws IOException {
        JsonObject chunksDirectory = new JsonObject();
        chunksDirectory.addProperty("path", chunksDir.getAbsolutePath());
        boolean chunksDirPresent = FileUtils.isDirectory((File)chunksDir, (LinkOption[])new LinkOption[0]);
        chunksDirectory.addProperty("present", Boolean.valueOf(chunksDirPresent));
        long fileCount = 0L;
        if (chunksDirPresent) {
            try (Stream<Path> stream = Files.list(chunksDir.toPath());){
                fileCount = stream.count();
            }
        }
        chunksDirectory.addProperty("fileCount", (Number)fileCount);
        return chunksDirectory;
    }

    private boolean checkAndRepair(JsonObject parent, KeyValueContainerData containerData, DatanodeStore store) {
        JsonElement chunksDirPresent;
        JsonElement pendingDeleteCountAggregate;
        long deleteTransactionCount;
        JsonElement pendingDeleteCountDB;
        long dbDeleteCount;
        JsonArray errors = new JsonArray();
        boolean passed = true;
        Table<String, Long> metadataTable = store.getMetadataTable();
        JsonObject dBMetadata = parent.getAsJsonObject("dBMetadata");
        JsonObject aggregates = parent.getAsJsonObject("aggregates");
        JsonElement blockCountDB = parent.getAsJsonObject("dBMetadata").get("#BLOCKCOUNT");
        JsonElement blockCountAggregate = parent.getAsJsonObject("aggregates").get("blockCount");
        long blockCountDBLong = 0L;
        if (!blockCountDB.isJsonNull()) {
            blockCountDBLong = blockCountDB.getAsLong();
        }
        if (blockCountDBLong != blockCountAggregate.getAsLong()) {
            passed = false;
            BooleanSupplier keyRepairAction = () -> {
                boolean repaired = false;
                try {
                    metadataTable.put((Object)containerData.getBlockCountKey(), (Object)blockCountAggregate.getAsLong());
                    repaired = true;
                }
                catch (IOException ex) {
                    LOG.error("Error repairing block count for container {}.", (Object)containerData.getContainerID(), (Object)ex);
                }
                return repaired;
            };
            JsonObject blockCountError = this.buildErrorAndRepair("dBMetadata.#BLOCKCOUNT", blockCountAggregate, blockCountDB, keyRepairAction);
            errors.add((JsonElement)blockCountError);
        }
        JsonElement usedBytesDB = parent.getAsJsonObject("dBMetadata").get("#BYTESUSED");
        JsonElement usedBytesAggregate = parent.getAsJsonObject("aggregates").get("usedBytes");
        long usedBytesDBLong = 0L;
        if (!usedBytesDB.isJsonNull()) {
            usedBytesDBLong = usedBytesDB.getAsLong();
        }
        if (usedBytesDBLong != usedBytesAggregate.getAsLong()) {
            passed = false;
            BooleanSupplier keyRepairAction = () -> {
                boolean repaired = false;
                try {
                    metadataTable.put((Object)containerData.getBytesUsedKey(), (Object)usedBytesAggregate.getAsLong());
                    repaired = true;
                }
                catch (IOException ex) {
                    LOG.error("Error repairing used bytes for container {}.", (Object)containerData.getContainerID(), (Object)ex);
                }
                return repaired;
            };
            JsonObject usedBytesError = this.buildErrorAndRepair("dBMetadata.#BYTESUSED", usedBytesAggregate, usedBytesDB, keyRepairAction);
            errors.add((JsonElement)usedBytesError);
        }
        if ((dbDeleteCount = KeyValueContainerMetadataInspector.jsonToLong(pendingDeleteCountDB = dBMetadata.get("#PENDINGDELETEBLOCKCOUNT"))) != (deleteTransactionCount = KeyValueContainerMetadataInspector.jsonToLong(pendingDeleteCountAggregate = aggregates.get("pendingDeleteBlocks")))) {
            passed = false;
            BooleanSupplier deleteCountRepairAction = () -> {
                String key = containerData.getPendingDeleteBlockCountKey();
                try {
                    metadataTable.put((Object)key, (Object)deleteTransactionCount);
                    return true;
                }
                catch (IOException ex) {
                    LOG.error("Failed to reset {} for container {}.", new Object[]{key, containerData.getContainerID(), ex});
                    return false;
                }
            };
            JsonObject deleteCountError = this.buildErrorAndRepair("dBMetadata.#PENDINGDELETEBLOCKCOUNT", pendingDeleteCountAggregate, pendingDeleteCountDB, deleteCountRepairAction);
            errors.add((JsonElement)deleteCountError);
        }
        if (!(chunksDirPresent = parent.getAsJsonObject("chunksDirectory").get("present")).getAsBoolean()) {
            passed = false;
            BooleanSupplier dirRepairAction = () -> {
                boolean repaired = false;
                try {
                    File chunksDir = new File(containerData.getChunksPath());
                    Files.createDirectories(chunksDir.toPath(), new FileAttribute[0]);
                    repaired = true;
                }
                catch (IOException ex) {
                    LOG.error("Error recreating empty chunks directory for container {}.", (Object)containerData.getContainerID(), (Object)ex);
                }
                return repaired;
            };
            JsonObject chunksDirError = this.buildErrorAndRepair("chunksDirectory.present", (JsonElement)new JsonPrimitive(Boolean.valueOf(true)), chunksDirPresent, dirRepairAction);
            errors.add((JsonElement)chunksDirError);
        }
        parent.addProperty("correct", Boolean.valueOf(passed));
        parent.add("errors", (JsonElement)errors);
        return passed;
    }

    static long jsonToLong(JsonElement e) {
        return e == null || e.isJsonNull() ? 0L : e.getAsLong();
    }

    private JsonObject buildErrorAndRepair(String property, JsonElement expected, JsonElement actual, BooleanSupplier repairAction) {
        JsonObject error = new JsonObject();
        error.addProperty("property", property);
        error.add("expected", expected);
        error.add("actual", actual);
        boolean repaired = false;
        if (this.mode == Mode.REPAIR) {
            repaired = repairAction.getAsBoolean();
        }
        error.addProperty("repaired", Boolean.valueOf(repaired));
        return error;
    }

    static PendingDelete countPendingDeletesSchemaV2(DatanodeStoreSchemaTwoImpl schemaTwoStore, KeyValueContainerData containerData) throws IOException {
        long pendingDeleteBlockCountTotal = 0L;
        long pendingDeleteBytes = 0L;
        Table<Long, StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction> delTxTable = schemaTwoStore.getDeleteTransactionTable();
        Table<String, BlockData> blockDataTable = schemaTwoStore.getBlockDataTable();
        try (TableIterator iterator = delTxTable.iterator();){
            while (iterator.hasNext()) {
                StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction txn = (StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction)((Table.KeyValue)iterator.next()).getValue();
                List localIDs = txn.getLocalIDList();
                pendingDeleteBlockCountTotal += (long)localIDs.size();
                pendingDeleteBytes += KeyValueContainerMetadataInspector.computePendingDeleteBytes(localIDs, containerData, blockDataTable);
            }
        }
        return new PendingDelete(pendingDeleteBlockCountTotal, pendingDeleteBytes);
    }

    static long computePendingDeleteBytes(List<Long> localIDs, KeyValueContainerData containerData, Table<String, BlockData> blockDataTable) {
        long pendingDeleteBytes = 0L;
        for (long id : localIDs) {
            try {
                String blockKey = containerData.getBlockKey(id);
                BlockData blockData = (BlockData)blockDataTable.get((Object)blockKey);
                if (blockData == null) continue;
                pendingDeleteBytes += blockData.getSize();
            }
            catch (IOException e) {
                LOG.error("Failed to get block " + id + " in container " + containerData.getContainerID() + " from blockDataTable", (Throwable)e);
            }
        }
        return pendingDeleteBytes;
    }

    static PendingDelete countPendingDeletesSchemaV3(DatanodeStoreSchemaThreeImpl schemaThreeStore, KeyValueContainerData containerData) throws IOException {
        long pendingDeleteBlockCountTotal = 0L;
        long pendingDeleteBytes = 0L;
        Table<String, BlockData> blockDataTable = schemaThreeStore.getBlockDataTable();
        try (TableIterator iter = schemaThreeStore.getDeleteTransactionTable().iterator((Object)containerData.containerPrefix());){
            while (iter.hasNext()) {
                StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction delTx = (StorageContainerDatanodeProtocolProtos.DeletedBlocksTransaction)((Table.KeyValue)iter.next()).getValue();
                List localIDs = delTx.getLocalIDList();
                pendingDeleteBlockCountTotal += (long)localIDs.size();
                pendingDeleteBytes += KeyValueContainerMetadataInspector.computePendingDeleteBytes(localIDs, containerData, blockDataTable);
            }
        }
        return new PendingDelete(pendingDeleteBlockCountTotal, pendingDeleteBytes);
    }

    private static long getBlockLength(BlockData block) {
        long blockLen = 0L;
        List chunkInfoList = block.getChunks();
        for (ContainerProtos.ChunkInfo chunk : chunkInfoList) {
            blockLen += chunk.getLen();
        }
        return blockLen;
    }

    static class PendingDelete {
        static final String COUNT = "pendingDeleteBlocks";
        static final String BYTES = "pendingDeleteBytes";
        private final long count;
        private final long bytes;

        PendingDelete(long count, long bytes) {
            this.count = count;
            this.bytes = bytes;
        }

        void addToJson(JsonObject json) {
            json.addProperty(COUNT, (Number)this.count);
            json.addProperty(BYTES, (Number)this.bytes);
        }
    }

    public static enum Mode {
        REPAIR("repair"),
        INSPECT("inspect"),
        OFF("off");

        private final String name;

        private Mode(String name) {
            this.name = name;
        }

        public String toString() {
            return this.name;
        }
    }
}

