/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import com.google.common.annotations.VisibleForTesting;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.Chore;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.SplitLogCounters;
import org.apache.hadoop.hbase.SplitLogTask;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.master.MasterFileSystem;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.monitoring.MonitoredTask;
import org.apache.hadoop.hbase.monitoring.TaskMonitor;
import org.apache.hadoop.hbase.protobuf.generated.ZooKeeperProtos;
import org.apache.hadoop.hbase.regionserver.wal.HLogSplitter;
import org.apache.hadoop.hbase.regionserver.wal.HLogUtil;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.ZKSplitLog;
import org.apache.hadoop.hbase.zookeeper.ZKUtil;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperListener;
import org.apache.hadoop.hbase.zookeeper.ZooKeeperWatcher;
import org.apache.hadoop.util.StringUtils;
import org.apache.zookeeper.AsyncCallback;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooDefs;
import org.apache.zookeeper.data.Stat;

@InterfaceAudience.Private
public class SplitLogManager
extends ZooKeeperListener {
    private static final Log LOG = LogFactory.getLog(SplitLogManager.class);
    public static final int DEFAULT_TIMEOUT = 120000;
    public static final int DEFAULT_ZK_RETRIES = 3;
    public static final int DEFAULT_MAX_RESUBMIT = 3;
    public static final int DEFAULT_UNASSIGNED_TIMEOUT = 180000;
    private final Stoppable stopper;
    private final MasterServices master;
    private final ServerName serverName;
    private final TaskFinisher taskFinisher;
    private FileSystem fs;
    private Configuration conf;
    private long zkretries;
    private long resubmit_threshold;
    private long timeout;
    private long unassignedTimeout;
    private long lastTaskCreateTime = Long.MAX_VALUE;
    public boolean ignoreZKDeleteForTesting = false;
    private volatile long lastRecoveringNodeCreationTime = 0L;
    private long checkRecoveringTimeThreshold = 15000L;
    private final List<Pair<Set<ServerName>, Boolean>> failedRecoveringRegionDeletions = Collections.synchronizedList(new ArrayList());
    protected final ReentrantLock recoveringRegionLock = new ReentrantLock();
    private volatile ZooKeeperProtos.SplitLogTask.RecoveryMode recoveryMode;
    private volatile boolean isDrainingDone = false;
    private final ConcurrentMap<String, Task> tasks = new ConcurrentHashMap<String, Task>();
    private TimeoutMonitor timeoutMonitor;
    private volatile Set<ServerName> deadWorkers = null;
    private final Object deadWorkersLock = new Object();
    private Set<String> failedDeletions = null;

    public SplitLogManager(ZooKeeperWatcher zkw, final Configuration conf, Stoppable stopper, MasterServices master, ServerName serverName, boolean masterRecovery) throws InterruptedIOException, KeeperException {
        this(zkw, conf, stopper, master, serverName, masterRecovery, new TaskFinisher(){

            @Override
            public TaskFinisher.Status finish(ServerName workerName, String logfile) {
                try {
                    HLogSplitter.finishSplitLogFile(logfile, conf);
                }
                catch (IOException e) {
                    LOG.warn((Object)("Could not finish splitting of log file " + logfile), (Throwable)e);
                    return TaskFinisher.Status.ERR;
                }
                return TaskFinisher.Status.DONE;
            }
        });
    }

    public SplitLogManager(ZooKeeperWatcher zkw, Configuration conf, Stoppable stopper, MasterServices master, ServerName serverName, boolean masterRecovery, TaskFinisher tf) throws InterruptedIOException, KeeperException {
        super(zkw);
        this.taskFinisher = tf;
        this.conf = conf;
        this.stopper = stopper;
        this.master = master;
        this.zkretries = conf.getLong("hbase.splitlog.zk.retries", 3L);
        this.resubmit_threshold = conf.getLong("hbase.splitlog.max.resubmit", 3L);
        this.timeout = conf.getInt("hbase.splitlog.manager.timeout", 120000);
        this.unassignedTimeout = conf.getInt("hbase.splitlog.manager.unassigned.timeout", 180000);
        this.setRecoveryMode(true);
        LOG.info((Object)("Timeout=" + this.timeout + ", unassigned timeout=" + this.unassignedTimeout + ", distributedLogReplay=" + (this.recoveryMode == ZooKeeperProtos.SplitLogTask.RecoveryMode.LOG_REPLAY)));
        this.serverName = serverName;
        this.timeoutMonitor = new TimeoutMonitor(conf.getInt("hbase.splitlog.manager.timeoutmonitor.period", 1000), stopper);
        this.failedDeletions = Collections.synchronizedSet(new HashSet());
        if (!masterRecovery) {
            Threads.setDaemonThreadRunning((Thread)this.timeoutMonitor.getThread(), (String)(serverName + ".splitLogManagerTimeoutMonitor"));
        }
        if (this.watcher != null) {
            this.watcher.registerListener((ZooKeeperListener)this);
            this.lookForOrphans();
        }
    }

    private FileStatus[] getFileList(List<Path> logDirs, PathFilter filter) throws IOException {
        ArrayList fileStatus = new ArrayList();
        for (Path hLogDir : logDirs) {
            this.fs = hLogDir.getFileSystem(this.conf);
            if (!this.fs.exists(hLogDir)) {
                LOG.warn((Object)(hLogDir + " doesn't exist. Nothing to do!"));
                continue;
            }
            FileStatus[] logfiles = FSUtils.listStatus(this.fs, hLogDir, filter);
            if (logfiles == null || logfiles.length == 0) {
                LOG.info((Object)(hLogDir + " is empty dir, no logs to split"));
                continue;
            }
            Collections.addAll(fileStatus, logfiles);
        }
        FileStatus[] a = new FileStatus[fileStatus.size()];
        return fileStatus.toArray(a);
    }

    public long splitLogDistributed(Path logDir) throws IOException {
        ArrayList<Path> logDirs = new ArrayList<Path>();
        logDirs.add(logDir);
        return this.splitLogDistributed(logDirs);
    }

    public long splitLogDistributed(List<Path> logDirs) throws IOException {
        if (logDirs.isEmpty()) {
            return 0L;
        }
        HashSet<ServerName> serverNames = new HashSet<ServerName>();
        for (Path logDir : logDirs) {
            try {
                ServerName serverName = HLogUtil.getServerNameFromHLogDirectoryName(logDir);
                if (serverName == null) continue;
                serverNames.add(serverName);
            }
            catch (IllegalArgumentException e) {
                LOG.warn((Object)("Cannot parse server name from " + logDir));
            }
        }
        return this.splitLogDistributed(serverNames, logDirs, null);
    }

    public long splitLogDistributed(Set<ServerName> serverNames, List<Path> logDirs, PathFilter filter) throws IOException {
        String msg;
        MonitoredTask status = TaskMonitor.get().createStatus("Doing distributed log split in " + logDirs);
        FileStatus[] logfiles = this.getFileList(logDirs, filter);
        status.setStatus("Checking directory contents...");
        LOG.debug((Object)"Scheduling batch of logs to split");
        SplitLogCounters.tot_mgr_log_split_batch_start.incrementAndGet();
        LOG.info((Object)("started splitting " + logfiles.length + " logs in " + logDirs));
        long t = EnvironmentEdgeManager.currentTimeMillis();
        long totalSize = 0L;
        TaskBatch batch = new TaskBatch();
        Boolean isMetaRecovery = filter == null ? null : Boolean.valueOf(false);
        for (FileStatus lf : logfiles) {
            totalSize += lf.getLen();
            String pathToLog = FSUtils.removeRootPath(lf.getPath(), this.conf);
            if (this.enqueueSplitTask(pathToLog, batch)) continue;
            throw new IOException("duplicate log split scheduled for " + lf.getPath());
        }
        this.waitForSplittingCompletion(batch, status);
        if (filter == MasterFileSystem.META_FILTER) {
            isMetaRecovery = true;
        }
        this.removeRecoveringRegionsFromZK(serverNames, isMetaRecovery);
        if (batch.done != batch.installed) {
            batch.isDead = true;
            SplitLogCounters.tot_mgr_log_split_batch_err.incrementAndGet();
            LOG.warn((Object)("error while splitting logs in " + logDirs + " installed = " + batch.installed + " but only " + batch.done + " done"));
            msg = "error or interrupted while splitting logs in " + logDirs + " Task = " + batch;
            status.abort(msg);
            throw new IOException(msg);
        }
        for (Path logDir : logDirs) {
            status.setStatus("Cleaning up log directory...");
            try {
                if (this.fs.exists(logDir) && !this.fs.delete(logDir, false)) {
                    LOG.warn((Object)("Unable to delete log src dir. Ignoring. " + logDir));
                }
            }
            catch (IOException ioe) {
                FileStatus[] files = this.fs.listStatus(logDir);
                if (files != null && files.length > 0) {
                    LOG.warn((Object)("returning success without actually splitting and deleting all the log files in path " + logDir));
                }
                LOG.warn((Object)("Unable to delete log src dir. Ignoring. " + logDir), (Throwable)ioe);
            }
            SplitLogCounters.tot_mgr_log_split_batch_success.incrementAndGet();
        }
        msg = "finished splitting (more than or equal to) " + totalSize + " bytes in " + batch.installed + " log files in " + logDirs + " in " + (EnvironmentEdgeManager.currentTimeMillis() - t) + "ms";
        status.markComplete(msg);
        LOG.info((Object)msg);
        return totalSize;
    }

    boolean enqueueSplitTask(String taskname, TaskBatch batch) {
        SplitLogCounters.tot_mgr_log_split_start.incrementAndGet();
        String path = ZKSplitLog.getEncodedNodeName(this.watcher, taskname);
        this.lastTaskCreateTime = EnvironmentEdgeManager.currentTimeMillis();
        Task oldtask = this.createTaskIfAbsent(path, batch);
        if (oldtask == null) {
            this.createNode(path, this.zkretries);
            return true;
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void waitForSplittingCompletion(TaskBatch batch, MonitoredTask status) {
        TaskBatch taskBatch = batch;
        synchronized (taskBatch) {
            while (batch.done + batch.error != batch.installed) {
                try {
                    int remainingInZK;
                    status.setStatus("Waiting for distributed tasks to finish.  scheduled=" + batch.installed + " done=" + batch.done + " error=" + batch.error);
                    int remaining = batch.installed - (batch.done + batch.error);
                    int actual = this.activeTasks(batch);
                    if (remaining != actual) {
                        LOG.warn((Object)("Expected " + remaining + " active tasks, but actually there are " + actual));
                    }
                    if ((remainingInZK = this.remainingTasksInZK()) >= 0 && actual > remainingInZK) {
                        LOG.warn((Object)("Expected at least" + actual + " tasks in ZK, but actually there are " + remainingInZK));
                    }
                    if (remainingInZK == 0 || actual == 0) {
                        LOG.warn((Object)("No more task remaining (ZK or task map), splitting should have completed. Remaining tasks in ZK " + remainingInZK + ", active tasks in map " + actual));
                        if (remainingInZK == 0 && actual == 0) {
                            return;
                        }
                    }
                    batch.wait(100L);
                    if (!this.stopper.isStopped()) continue;
                    LOG.warn((Object)"Stopped while waiting for log splits to be completed");
                    return;
                }
                catch (InterruptedException e) {
                    LOG.warn((Object)"Interrupted while waiting for log splits to be completed");
                    Thread.currentThread().interrupt();
                    return;
                }
            }
        }
    }

    @VisibleForTesting
    ConcurrentMap<String, Task> getTasks() {
        return this.tasks;
    }

    private int activeTasks(TaskBatch batch) {
        int count = 0;
        for (Task t : this.tasks.values()) {
            if (t.batch != batch || t.status != TerminationStatus.IN_PROGRESS) continue;
            ++count;
        }
        return count;
    }

    private int remainingTasksInZK() {
        int count = 0;
        try {
            List tasks = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)this.watcher.splitLogZNode);
            if (tasks != null) {
                for (String t : tasks) {
                    if (ZKSplitLog.isRescanNode(t)) continue;
                    ++count;
                }
            }
        }
        catch (KeeperException ke) {
            LOG.warn((Object)"Failed to check remaining tasks", (Throwable)ke);
            count = -1;
        }
        return count;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void removeRecoveringRegionsFromZK(Set<ServerName> serverNames, Boolean isMetaRecovery) {
        if (this.recoveryMode != ZooKeeperProtos.SplitLogTask.RecoveryMode.LOG_REPLAY) {
            return;
        }
        String metaEncodeRegionName = HRegionInfo.FIRST_META_REGIONINFO.getEncodedName();
        int count = 0;
        HashSet<String> recoveredServerNameSet = new HashSet<String>();
        if (serverNames != null) {
            for (ServerName tmpServerName : serverNames) {
                recoveredServerNameSet.add(tmpServerName.getServerName());
            }
        }
        try {
            List regions;
            this.recoveringRegionLock.lock();
            List tasks = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)this.watcher.splitLogZNode);
            if (tasks != null) {
                for (String t : tasks) {
                    if (ZKSplitLog.isRescanNode(t)) continue;
                    ++count;
                }
            }
            if (count == 0 && this.master.isInitialized() && !this.master.getServerManager().areDeadServersInProgress()) {
                SplitLogManager.deleteRecoveringRegionZNodes(this.watcher, null);
                this.lastRecoveringNodeCreationTime = Long.MAX_VALUE;
            } else if (!recoveredServerNameSet.isEmpty() && (regions = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)this.watcher.recoveringRegionsZNode)) != null) {
                for (String region : regions) {
                    if (isMetaRecovery != null && (isMetaRecovery.booleanValue() && !region.equalsIgnoreCase(metaEncodeRegionName) || !isMetaRecovery.booleanValue() && region.equalsIgnoreCase(metaEncodeRegionName))) continue;
                    String nodePath = ZKUtil.joinZNode((String)this.watcher.recoveringRegionsZNode, (String)region);
                    List failedServers = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)nodePath);
                    if (failedServers == null || failedServers.isEmpty()) {
                        ZKUtil.deleteNode((ZooKeeperWatcher)this.watcher, (String)nodePath);
                        continue;
                    }
                    if (recoveredServerNameSet.containsAll(failedServers)) {
                        ZKUtil.deleteNodeRecursively((ZooKeeperWatcher)this.watcher, (String)nodePath);
                        continue;
                    }
                    for (String failedServer : failedServers) {
                        if (!recoveredServerNameSet.contains(failedServer)) continue;
                        String tmpPath = ZKUtil.joinZNode((String)nodePath, (String)failedServer);
                        ZKUtil.deleteNode((ZooKeeperWatcher)this.watcher, (String)tmpPath);
                    }
                }
            }
        }
        catch (KeeperException ke) {
            LOG.warn((Object)"removeRecoveringRegionsFromZK got zookeeper exception. Will retry", (Throwable)ke);
            if (serverNames != null && !serverNames.isEmpty()) {
                this.failedRecoveringRegionDeletions.add((Pair<Set<ServerName>, Boolean>)new Pair(serverNames, (Object)isMetaRecovery));
            }
        }
        finally {
            this.recoveringRegionLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeStaleRecoveringRegionsFromZK(Set<ServerName> failedServers) throws KeeperException {
        HashSet<String> knownFailedServers = new HashSet<String>();
        if (failedServers != null) {
            for (ServerName tmpServerName : failedServers) {
                knownFailedServers.add(tmpServerName.getServerName());
            }
        }
        this.recoveringRegionLock.lock();
        try {
            List regions;
            List tasks = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)this.watcher.splitLogZNode);
            if (tasks != null) {
                for (String t : tasks) {
                    ServerName serverName;
                    byte[] data = ZKUtil.getData((ZooKeeperWatcher)this.watcher, (String)ZKUtil.joinZNode((String)this.watcher.splitLogZNode, (String)t));
                    if (data != null) {
                        SplitLogTask slt = null;
                        try {
                            slt = SplitLogTask.parseFrom(data);
                        }
                        catch (DeserializationException e) {
                            LOG.warn((Object)("Failed parse data for znode " + t), (Throwable)e);
                        }
                        if (slt != null && slt.isDone()) continue;
                    }
                    if ((serverName = HLogUtil.getServerNameFromHLogDirectoryName(new Path(t = ZKSplitLog.getFileName(t)))) != null) {
                        knownFailedServers.add(serverName.getServerName());
                        continue;
                    }
                    LOG.warn((Object)("Found invalid WAL log file name:" + t));
                }
            }
            if ((regions = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)this.watcher.recoveringRegionsZNode)) != null) {
                for (String region : regions) {
                    String nodePath = ZKUtil.joinZNode((String)this.watcher.recoveringRegionsZNode, (String)region);
                    List regionFailedServers = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)nodePath);
                    if (regionFailedServers == null || regionFailedServers.isEmpty()) {
                        ZKUtil.deleteNode((ZooKeeperWatcher)this.watcher, (String)nodePath);
                        continue;
                    }
                    boolean needMoreRecovery = false;
                    for (String tmpFailedServer : regionFailedServers) {
                        if (!knownFailedServers.contains(tmpFailedServer)) continue;
                        needMoreRecovery = true;
                        break;
                    }
                    if (needMoreRecovery) continue;
                    ZKUtil.deleteNodeRecursively((ZooKeeperWatcher)this.watcher, (String)nodePath);
                }
            }
        }
        finally {
            this.recoveringRegionLock.unlock();
        }
    }

    public static void deleteRecoveringRegionZNodes(ZooKeeperWatcher watcher, List<String> regions) {
        try {
            if (regions == null) {
                LOG.info((Object)"Garbage collecting all recovering regions.");
                ZKUtil.deleteChildrenRecursively((ZooKeeperWatcher)watcher, (String)watcher.recoveringRegionsZNode);
            } else {
                for (String curRegion : regions) {
                    String nodePath = ZKUtil.joinZNode((String)watcher.recoveringRegionsZNode, (String)curRegion);
                    ZKUtil.deleteNodeRecursively((ZooKeeperWatcher)watcher, (String)nodePath);
                }
            }
        }
        catch (KeeperException e) {
            LOG.warn((Object)"Cannot remove recovering regions from ZooKeeper", (Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setDone(String path, TerminationStatus status) {
        Task task = (Task)this.tasks.get(path);
        if (task == null) {
            if (!ZKSplitLog.isRescanNode(this.watcher, path)) {
                SplitLogCounters.tot_mgr_unacquired_orphan_done.incrementAndGet();
                LOG.debug((Object)("unacquired orphan task is done " + path));
            }
        } else {
            Task task2 = task;
            synchronized (task2) {
                if (task.status == TerminationStatus.IN_PROGRESS) {
                    if (status == TerminationStatus.SUCCESS) {
                        SplitLogCounters.tot_mgr_log_split_success.incrementAndGet();
                        LOG.info((Object)("Done splitting " + path));
                    } else {
                        SplitLogCounters.tot_mgr_log_split_err.incrementAndGet();
                        LOG.warn((Object)("Error splitting " + path));
                    }
                    task.status = status;
                    if (task.batch != null) {
                        TaskBatch taskBatch = task.batch;
                        synchronized (taskBatch) {
                            if (status == TerminationStatus.SUCCESS) {
                                ++task.batch.done;
                            } else {
                                ++task.batch.error;
                            }
                            task.batch.notify();
                        }
                    }
                }
            }
        }
        this.deleteNode(path, this.zkretries);
    }

    private void createNode(String path, Long retry_count) {
        SplitLogTask.Unassigned slt = new SplitLogTask.Unassigned(this.serverName, this.recoveryMode);
        ZKUtil.asyncCreate((ZooKeeperWatcher)this.watcher, (String)path, (byte[])slt.toByteArray(), (AsyncCallback.StringCallback)new CreateAsyncCallback(), (Object)retry_count);
        SplitLogCounters.tot_mgr_node_create_queued.incrementAndGet();
    }

    private void createNodeSuccess(String path) {
        LOG.debug((Object)("put up splitlog task at znode " + path));
        this.getDataSetWatch(path, this.zkretries);
    }

    private void createNodeFailure(String path) {
        LOG.warn((Object)("failed to create task node" + path));
        this.setDone(path, TerminationStatus.FAILURE);
    }

    private void getDataSetWatch(String path, Long retry_count) {
        this.watcher.getRecoverableZooKeeper().getZooKeeper().getData(path, (Watcher)this.watcher, (AsyncCallback.DataCallback)new GetDataAsyncCallback(true), (Object)retry_count);
        SplitLogCounters.tot_mgr_get_data_queued.incrementAndGet();
    }

    private void tryGetDataSetWatch(String path) {
        this.watcher.getRecoverableZooKeeper().getZooKeeper().getData(path, (Watcher)this.watcher, (AsyncCallback.DataCallback)new GetDataAsyncCallback(false), (Object)-1L);
        SplitLogCounters.tot_mgr_get_data_queued.incrementAndGet();
    }

    private void getDataSetWatchSuccess(String path, byte[] data, int version) throws DeserializationException {
        if (data == null) {
            if (version == Integer.MIN_VALUE) {
                this.setDone(path, TerminationStatus.SUCCESS);
                return;
            }
            SplitLogCounters.tot_mgr_null_data.incrementAndGet();
            LOG.fatal((Object)("logic error - got null data " + path));
            this.setDone(path, TerminationStatus.FAILURE);
            return;
        }
        data = this.watcher.getRecoverableZooKeeper().removeMetaData(data);
        SplitLogTask slt = SplitLogTask.parseFrom(data);
        if (slt.isUnassigned()) {
            LOG.debug((Object)("task not yet acquired " + path + " ver = " + version));
            this.handleUnassignedTask(path);
        } else if (slt.isOwned()) {
            this.heartbeat(path, version, slt.getServerName());
        } else if (slt.isResigned()) {
            LOG.info((Object)("task " + path + " entered state: " + slt.toString()));
            this.resubmitOrFail(path, ResubmitDirective.FORCE);
        } else if (slt.isDone()) {
            LOG.info((Object)("task " + path + " entered state: " + slt.toString()));
            if (this.taskFinisher != null && !ZKSplitLog.isRescanNode(this.watcher, path)) {
                if (this.taskFinisher.finish(slt.getServerName(), ZKSplitLog.getFileName(path)) == TaskFinisher.Status.DONE) {
                    this.setDone(path, TerminationStatus.SUCCESS);
                } else {
                    this.resubmitOrFail(path, ResubmitDirective.CHECK);
                }
            } else {
                this.setDone(path, TerminationStatus.SUCCESS);
            }
        } else if (slt.isErr()) {
            LOG.info((Object)("task " + path + " entered state: " + slt.toString()));
            this.resubmitOrFail(path, ResubmitDirective.CHECK);
        } else {
            LOG.fatal((Object)("logic error - unexpected zk state for path = " + path + " data = " + slt.toString()));
            this.setDone(path, TerminationStatus.FAILURE);
        }
    }

    private void getDataSetWatchFailure(String path) {
        LOG.warn((Object)("failed to set data watch " + path));
        this.setDone(path, TerminationStatus.FAILURE);
    }

    private void handleUnassignedTask(String path) {
        if (ZKSplitLog.isRescanNode(this.watcher, path)) {
            return;
        }
        Task task = this.findOrCreateOrphanTask(path);
        if (task.isOrphan() && task.incarnation == 0) {
            LOG.info((Object)("resubmitting unassigned orphan task " + path));
            this.resubmit(path, task, ResubmitDirective.FORCE);
        }
    }

    private boolean needAbandonRetries(int statusCode, String action) {
        if (statusCode == KeeperException.Code.SESSIONEXPIRED.intValue()) {
            LOG.error((Object)("ZK session expired. Master is expected to shut down. Abandoning retries for action=" + action));
            return true;
        }
        return false;
    }

    private void heartbeat(String path, int new_version, ServerName workerName) {
        Task task = this.findOrCreateOrphanTask(path);
        if (new_version != task.last_version) {
            if (task.isUnassigned()) {
                LOG.info((Object)("task " + path + " acquired by " + workerName));
            }
            task.heartbeat(EnvironmentEdgeManager.currentTimeMillis(), new_version, workerName);
            SplitLogCounters.tot_mgr_heartbeat.incrementAndGet();
        }
    }

    private boolean resubmit(String path, Task task, ResubmitDirective directive) {
        int version;
        if (task.status != TerminationStatus.IN_PROGRESS) {
            return false;
        }
        if (directive != ResubmitDirective.FORCE) {
            boolean alive;
            long time = EnvironmentEdgeManager.currentTimeMillis() - task.last_update;
            boolean bl = alive = this.master.getServerManager() != null ? this.master.getServerManager().isServerOnline(task.cur_worker_name) : true;
            if (alive && time < this.timeout) {
                LOG.trace((Object)("Skipping the resubmit of " + task.toString() + "  because the server " + task.cur_worker_name + " is not marked as dead, we waited for " + time + " while the timeout is " + this.timeout));
                return false;
            }
            if ((long)task.unforcedResubmits.get() >= this.resubmit_threshold) {
                if (!task.resubmitThresholdReached) {
                    task.resubmitThresholdReached = true;
                    SplitLogCounters.tot_mgr_resubmit_threshold_reached.incrementAndGet();
                    LOG.info((Object)("Skipping resubmissions of task " + path + " because threshold " + this.resubmit_threshold + " reached"));
                }
                return false;
            }
            version = task.last_version;
        } else {
            SplitLogCounters.tot_mgr_resubmit_force.incrementAndGet();
            version = -1;
        }
        LOG.info((Object)("resubmitting task " + path));
        ++task.incarnation;
        try {
            SplitLogTask.Unassigned slt = new SplitLogTask.Unassigned(this.serverName, this.recoveryMode);
            if (!ZKUtil.setData((ZooKeeperWatcher)this.watcher, (String)path, (byte[])slt.toByteArray(), (int)version)) {
                LOG.debug((Object)("failed to resubmit task " + path + " version changed"));
                task.heartbeatNoDetails(EnvironmentEdgeManager.currentTimeMillis());
                return false;
            }
        }
        catch (KeeperException.NoNodeException e) {
            LOG.warn((Object)("failed to resubmit because znode doesn't exist " + path + " task done (or forced done by removing the znode)"));
            try {
                this.getDataSetWatchSuccess(path, null, Integer.MIN_VALUE);
            }
            catch (DeserializationException e1) {
                LOG.debug((Object)("Failed to re-resubmit task " + path + " because of deserialization issue"), (Throwable)e1);
                task.heartbeatNoDetails(EnvironmentEdgeManager.currentTimeMillis());
                return false;
            }
            return false;
        }
        catch (KeeperException.BadVersionException e) {
            LOG.debug((Object)("failed to resubmit task " + path + " version changed"));
            task.heartbeatNoDetails(EnvironmentEdgeManager.currentTimeMillis());
            return false;
        }
        catch (KeeperException e) {
            SplitLogCounters.tot_mgr_resubmit_failed.incrementAndGet();
            LOG.warn((Object)("failed to resubmit " + path), (Throwable)e);
            return false;
        }
        if (directive != ResubmitDirective.FORCE) {
            task.unforcedResubmits.incrementAndGet();
        }
        task.setUnassigned();
        this.createRescanNode(Long.MAX_VALUE);
        SplitLogCounters.tot_mgr_resubmit.incrementAndGet();
        return true;
    }

    private void resubmitOrFail(String path, ResubmitDirective directive) {
        if (!this.resubmit(path, this.findOrCreateOrphanTask(path), directive)) {
            this.setDone(path, TerminationStatus.FAILURE);
        }
    }

    private void deleteNode(String path, Long retries) {
        SplitLogCounters.tot_mgr_node_delete_queued.incrementAndGet();
        this.watcher.getRecoverableZooKeeper().getZooKeeper().delete(path, -1, (AsyncCallback.VoidCallback)new DeleteAsyncCallback(), (Object)retries);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void deleteNodeSuccess(String path) {
        if (this.ignoreZKDeleteForTesting) {
            return;
        }
        Task task = (Task)this.tasks.remove(path);
        if (task == null) {
            if (ZKSplitLog.isRescanNode(this.watcher, path)) {
                SplitLogCounters.tot_mgr_rescan_deleted.incrementAndGet();
            }
            SplitLogCounters.tot_mgr_missing_state_in_delete.incrementAndGet();
            LOG.debug((Object)("deleted task without in memory state " + path));
            return;
        }
        Task task2 = task;
        synchronized (task2) {
            task.status = TerminationStatus.DELETED;
            task.notify();
        }
        SplitLogCounters.tot_mgr_task_deleted.incrementAndGet();
    }

    private void deleteNodeFailure(String path) {
        LOG.info((Object)("Failed to delete node " + path + " and will retry soon."));
    }

    private void createRescanNode(long retries) {
        this.lastTaskCreateTime = EnvironmentEdgeManager.currentTimeMillis();
        SplitLogTask.Done slt = new SplitLogTask.Done(this.serverName, this.recoveryMode);
        this.watcher.getRecoverableZooKeeper().getZooKeeper().create(ZKSplitLog.getRescanNode(this.watcher), slt.toByteArray(), (List)ZooDefs.Ids.OPEN_ACL_UNSAFE, CreateMode.EPHEMERAL_SEQUENTIAL, (AsyncCallback.StringCallback)new CreateRescanAsyncCallback(), (Object)retries);
    }

    private void createRescanSuccess(String path) {
        SplitLogCounters.tot_mgr_rescan.incrementAndGet();
        this.getDataSetWatch(path, this.zkretries);
    }

    private void createRescanFailure() {
        LOG.fatal((Object)"logic failure, rescan failure must not happen");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Task createTaskIfAbsent(String path, TaskBatch batch) {
        Task newtask = new Task();
        newtask.batch = batch;
        Task oldtask = this.tasks.putIfAbsent(path, newtask);
        if (oldtask == null) {
            ++batch.installed;
            return null;
        }
        Task task = oldtask;
        synchronized (task) {
            if (oldtask.isOrphan()) {
                if (oldtask.status == TerminationStatus.SUCCESS) {
                    return null;
                }
                if (oldtask.status == TerminationStatus.IN_PROGRESS) {
                    oldtask.batch = batch;
                    ++batch.installed;
                    LOG.debug((Object)("Previously orphan task " + path + " is now being waited upon"));
                    return null;
                }
                while (oldtask.status == TerminationStatus.FAILURE) {
                    LOG.debug((Object)("wait for status of task " + path + " to change to DELETED"));
                    SplitLogCounters.tot_mgr_wait_for_zk_delete.incrementAndGet();
                    try {
                        oldtask.wait();
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        LOG.warn((Object)"Interrupted when waiting for znode delete callback");
                        break;
                    }
                }
                if (oldtask.status != TerminationStatus.DELETED) {
                    LOG.warn((Object)("Failure because previously failed task state still present. Waiting for znode delete callback path=" + path));
                    return oldtask;
                }
                Task t = this.tasks.putIfAbsent(path, newtask);
                if (t == null) {
                    ++batch.installed;
                    return null;
                }
                LOG.fatal((Object)"Logic error. Deleted task still present in tasks map");
                assert (false) : "Deleted task still present in tasks map";
                return t;
            }
            LOG.warn((Object)("Failure because two threads can't wait for the same task; path=" + path));
            return oldtask;
        }
    }

    Task findOrCreateOrphanTask(String path) {
        Task orphanTask = new Task();
        Task task = this.tasks.putIfAbsent(path, orphanTask);
        if (task == null) {
            LOG.info((Object)("creating orphan task " + path));
            SplitLogCounters.tot_mgr_orphan_task_acquired.incrementAndGet();
            task = orphanTask;
        }
        return task;
    }

    public void nodeDataChanged(String path) {
        Task task = (Task)this.tasks.get(path);
        if (task != null || ZKSplitLog.isRescanNode(this.watcher, path)) {
            if (task != null) {
                task.heartbeatNoDetails(EnvironmentEdgeManager.currentTimeMillis());
            }
            this.getDataSetWatch(path, this.zkretries);
        }
    }

    public void stop() {
        if (this.timeoutMonitor != null) {
            this.timeoutMonitor.interrupt();
        }
    }

    private void lookForOrphans() {
        List orphans;
        try {
            orphans = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)this.watcher.splitLogZNode);
            if (orphans == null) {
                LOG.warn((Object)("could not get children of " + this.watcher.splitLogZNode));
                return;
            }
        }
        catch (KeeperException e) {
            LOG.warn((Object)("could not get children of " + this.watcher.splitLogZNode + " " + StringUtils.stringifyException((Throwable)e)));
            return;
        }
        int rescan_nodes = 0;
        for (String path : orphans) {
            String nodepath = ZKUtil.joinZNode((String)this.watcher.splitLogZNode, (String)path);
            if (ZKSplitLog.isRescanNode(this.watcher, nodepath)) {
                ++rescan_nodes;
                LOG.debug((Object)("found orphan rescan node " + path));
            } else {
                LOG.info((Object)("found orphan task " + path));
            }
            this.getDataSetWatch(nodepath, this.zkretries);
        }
        LOG.info((Object)("Found " + (orphans.size() - rescan_nodes) + " orphan tasks and " + rescan_nodes + " rescan nodes"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void markRegionsRecoveringInZK(ServerName serverName, Set<HRegionInfo> userRegions) throws KeeperException {
        if (userRegions == null || this.recoveryMode != ZooKeeperProtos.SplitLogTask.RecoveryMode.LOG_REPLAY) {
            return;
        }
        try {
            this.recoveringRegionLock.lock();
            this.lastRecoveringNodeCreationTime = EnvironmentEdgeManager.currentTimeMillis();
            block7: for (HRegionInfo region : userRegions) {
                String regionEncodeName = region.getEncodedName();
                long retries = this.zkretries;
                while (true) {
                    String nodePath = ZKUtil.joinZNode((String)this.watcher.recoveringRegionsZNode, (String)regionEncodeName);
                    long lastRecordedFlushedSequenceId = -1L;
                    try {
                        long lastSequenceId = this.master.getServerManager().getLastFlushedSequenceId(regionEncodeName.getBytes());
                        byte[] data = ZKUtil.getData((ZooKeeperWatcher)this.watcher, (String)nodePath);
                        if (data == null) {
                            ZKUtil.createSetData((ZooKeeperWatcher)this.watcher, (String)nodePath, (byte[])ZKUtil.positionToByteArray((long)lastSequenceId));
                        } else {
                            lastRecordedFlushedSequenceId = SplitLogManager.parseLastFlushedSequenceIdFrom(data);
                            if (lastRecordedFlushedSequenceId < lastSequenceId) {
                                ZKUtil.setData((ZooKeeperWatcher)this.watcher, (String)nodePath, (byte[])ZKUtil.positionToByteArray((long)lastSequenceId));
                            }
                        }
                        nodePath = ZKUtil.joinZNode((String)nodePath, (String)serverName.getServerName());
                        if (lastSequenceId <= lastRecordedFlushedSequenceId) {
                            lastSequenceId = lastRecordedFlushedSequenceId;
                        }
                        ZKUtil.createSetData((ZooKeeperWatcher)this.watcher, (String)nodePath, (byte[])ZKUtil.regionSequenceIdsToByteArray((Long)lastSequenceId, null));
                        LOG.debug((Object)("Mark region " + regionEncodeName + " recovering from failed region server " + serverName));
                    }
                    catch (KeeperException e) {
                        if (retries <= 1L) {
                            throw e;
                        }
                        try {
                            Thread.sleep(20L);
                        }
                        catch (Exception ignoreE) {
                            // empty catch block
                        }
                        if (--retries <= 0L) continue block7;
                        if (!this.stopper.isStopped()) continue;
                    }
                    break;
                }
            }
        }
        finally {
            this.recoveringRegionLock.unlock();
        }
    }

    public static long parseLastFlushedSequenceIdFrom(byte[] bytes) {
        long lastRecordedFlushedSequenceId = -1L;
        try {
            lastRecordedFlushedSequenceId = ZKUtil.parseHLogPositionFrom((byte[])bytes);
        }
        catch (DeserializationException e) {
            lastRecordedFlushedSequenceId = -1L;
            LOG.warn((Object)"Can't parse last flushed sequence Id", (Throwable)e);
        }
        return lastRecordedFlushedSequenceId;
    }

    public static boolean isRegionMarkedRecoveringInZK(ZooKeeperWatcher zkw, String regionEncodedName) throws KeeperException {
        boolean result = false;
        String nodePath = ZKUtil.joinZNode((String)zkw.recoveringRegionsZNode, (String)regionEncodedName);
        byte[] node = ZKUtil.getDataAndWatch((ZooKeeperWatcher)zkw, (String)nodePath);
        if (node != null) {
            result = true;
        }
        return result;
    }

    public static ZooKeeperProtos.RegionStoreSequenceIds getRegionFlushedSequenceId(ZooKeeperWatcher zkw, String serverName, String encodedRegionName) throws IOException {
        ZooKeeperProtos.RegionStoreSequenceIds result = null;
        String nodePath = ZKUtil.joinZNode((String)zkw.recoveringRegionsZNode, (String)encodedRegionName);
        nodePath = ZKUtil.joinZNode((String)nodePath, (String)serverName);
        try {
            byte[] data = ZKUtil.getData((ZooKeeperWatcher)zkw, (String)nodePath);
            if (data != null) {
                result = ZKUtil.parseRegionStoreSequenceIds((byte[])data);
            }
        }
        catch (KeeperException e) {
            throw new IOException("Cannot get lastFlushedSequenceId from ZooKeeper for server=" + serverName + "; region=" + encodedRegionName, e);
        }
        catch (DeserializationException e) {
            LOG.warn((Object)("Can't parse last flushed sequence Id from znode:" + nodePath), (Throwable)e);
        }
        return result;
    }

    private List<String> listSplitLogTasks() throws KeeperException {
        List taskOrRescanList = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)this.watcher.splitLogZNode);
        if (taskOrRescanList == null || taskOrRescanList.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<String> taskList = new ArrayList<String>();
        for (String taskOrRescan : taskOrRescanList) {
            if (ZKSplitLog.isRescanNode(taskOrRescan)) continue;
            taskList.add(taskOrRescan);
        }
        return taskList;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setRecoveryMode(boolean isForInitialization) throws KeeperException {
        List<String> tasks;
        if (this.isDrainingDone) {
            return;
        }
        if (this.watcher == null) {
            this.isDrainingDone = true;
            this.recoveryMode = ZooKeeperProtos.SplitLogTask.RecoveryMode.LOG_SPLITTING;
            return;
        }
        boolean hasSplitLogTask = false;
        boolean hasRecoveringRegions = false;
        ZooKeeperProtos.SplitLogTask.RecoveryMode previousRecoveryMode = ZooKeeperProtos.SplitLogTask.RecoveryMode.UNKNOWN;
        ZooKeeperProtos.SplitLogTask.RecoveryMode recoveryModeInConfig = this.isDistributedLogReplay(this.conf) ? ZooKeeperProtos.SplitLogTask.RecoveryMode.LOG_REPLAY : ZooKeeperProtos.SplitLogTask.RecoveryMode.LOG_SPLITTING;
        List regions = ZKUtil.listChildrenNoWatch((ZooKeeperWatcher)this.watcher, (String)this.watcher.recoveringRegionsZNode);
        if (regions != null && !regions.isEmpty()) {
            hasRecoveringRegions = true;
            previousRecoveryMode = ZooKeeperProtos.SplitLogTask.RecoveryMode.LOG_REPLAY;
        }
        if (previousRecoveryMode == ZooKeeperProtos.SplitLogTask.RecoveryMode.UNKNOWN && !(tasks = this.listSplitLogTasks()).isEmpty()) {
            hasSplitLogTask = true;
            if (isForInitialization) {
                for (String task : tasks) {
                    try {
                        byte[] data = ZKUtil.getData((ZooKeeperWatcher)this.watcher, (String)ZKUtil.joinZNode((String)this.watcher.splitLogZNode, (String)task));
                        if (data == null) continue;
                        SplitLogTask slt = SplitLogTask.parseFrom(data);
                        previousRecoveryMode = slt.getMode();
                        if (previousRecoveryMode != ZooKeeperProtos.SplitLogTask.RecoveryMode.UNKNOWN) break;
                        previousRecoveryMode = ZooKeeperProtos.SplitLogTask.RecoveryMode.LOG_SPLITTING;
                        break;
                    }
                    catch (DeserializationException e) {
                        LOG.warn((Object)("Failed parse data for znode " + task), (Throwable)e);
                    }
                }
            }
        }
        SplitLogManager splitLogManager = this;
        synchronized (splitLogManager) {
            if (this.isDrainingDone) {
                return;
            }
            if (!hasSplitLogTask && !hasRecoveringRegions) {
                this.isDrainingDone = true;
                this.recoveryMode = recoveryModeInConfig;
                return;
            }
            if (!isForInitialization) {
                return;
            }
            if (previousRecoveryMode != ZooKeeperProtos.SplitLogTask.RecoveryMode.UNKNOWN) {
                this.isDrainingDone = previousRecoveryMode == recoveryModeInConfig;
                this.recoveryMode = previousRecoveryMode;
            } else {
                this.recoveryMode = recoveryModeInConfig;
            }
        }
    }

    public ZooKeeperProtos.SplitLogTask.RecoveryMode getRecoveryMode() {
        return this.recoveryMode;
    }

    private boolean isDistributedLogReplay(Configuration conf) {
        boolean dlr = conf.getBoolean("hbase.master.distributed.log.replay", false);
        int version = conf.getInt("hfile.format.version", 3);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("Distributed log replay=" + dlr + ", " + "hfile.format.version" + "=" + version));
        }
        return dlr && version >= 3;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void handleDeadWorker(ServerName workerName) {
        Object object = this.deadWorkersLock;
        synchronized (object) {
            if (this.deadWorkers == null) {
                this.deadWorkers = new HashSet<ServerName>(100);
            }
            this.deadWorkers.add(workerName);
        }
        LOG.info((Object)("dead splitlog worker " + workerName));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void handleDeadWorkers(Set<ServerName> serverNames) {
        Object object = this.deadWorkersLock;
        synchronized (object) {
            if (this.deadWorkers == null) {
                this.deadWorkers = new HashSet<ServerName>(100);
            }
            this.deadWorkers.addAll(serverNames);
        }
        LOG.info((Object)("dead splitlog workers " + serverNames));
    }

    static enum TerminationStatus {
        IN_PROGRESS("in_progress"),
        SUCCESS("success"),
        FAILURE("failure"),
        DELETED("deleted");

        String statusMsg;

        private TerminationStatus(String msg) {
            this.statusMsg = msg;
        }

        public String toString() {
            return this.statusMsg;
        }
    }

    static enum ResubmitDirective {
        CHECK,
        FORCE;

    }

    public static interface TaskFinisher {
        public Status finish(ServerName var1, String var2);

        public static enum Status {
            DONE,
            ERR;

        }
    }

    class CreateRescanAsyncCallback
    implements AsyncCallback.StringCallback {
        private final Log LOG = LogFactory.getLog(CreateRescanAsyncCallback.class);

        CreateRescanAsyncCallback() {
        }

        public void processResult(int rc, String path, Object ctx, String name) {
            if (rc != 0) {
                if (SplitLogManager.this.needAbandonRetries(rc, "CreateRescan znode " + path)) {
                    return;
                }
                Long retry_count = (Long)ctx;
                this.LOG.warn((Object)("rc=" + KeeperException.Code.get((int)rc) + " for " + path + " remaining retries=" + retry_count));
                if (retry_count == 0L) {
                    SplitLogManager.this.createRescanFailure();
                } else {
                    SplitLogManager.this.createRescanNode(retry_count - 1L);
                }
                return;
            }
            SplitLogManager.this.createRescanSuccess(name);
        }
    }

    class DeleteAsyncCallback
    implements AsyncCallback.VoidCallback {
        private final Log LOG = LogFactory.getLog(DeleteAsyncCallback.class);

        DeleteAsyncCallback() {
        }

        public void processResult(int rc, String path, Object ctx) {
            SplitLogCounters.tot_mgr_node_delete_result.incrementAndGet();
            if (rc != 0) {
                if (SplitLogManager.this.needAbandonRetries(rc, "Delete znode " + path)) {
                    SplitLogManager.this.failedDeletions.add(path);
                    return;
                }
                if (rc != KeeperException.Code.NONODE.intValue()) {
                    SplitLogCounters.tot_mgr_node_delete_err.incrementAndGet();
                    Long retry_count = (Long)ctx;
                    this.LOG.warn((Object)("delete rc=" + KeeperException.Code.get((int)rc) + " for " + path + " remaining retries=" + retry_count));
                    if (retry_count == 0L) {
                        this.LOG.warn((Object)("delete failed " + path));
                        SplitLogManager.this.failedDeletions.add(path);
                        SplitLogManager.this.deleteNodeFailure(path);
                    } else {
                        SplitLogManager.this.deleteNode(path, retry_count - 1L);
                    }
                    return;
                }
                this.LOG.info((Object)(path + " does not exist. Either was created but deleted behind our" + " back by another pending delete OR was deleted" + " in earlier retry rounds. zkretries = " + (Long)ctx));
            } else {
                this.LOG.debug((Object)("deleted " + path));
            }
            SplitLogManager.this.deleteNodeSuccess(path);
        }
    }

    class GetDataAsyncCallback
    implements AsyncCallback.DataCallback {
        private final Log LOG = LogFactory.getLog(GetDataAsyncCallback.class);
        private boolean completeTaskOnNoNode;

        public GetDataAsyncCallback(boolean completeTaskOnNoNode) {
            this.completeTaskOnNoNode = completeTaskOnNoNode;
        }

        public void processResult(int rc, String path, Object ctx, byte[] data, Stat stat) {
            SplitLogCounters.tot_mgr_get_data_result.incrementAndGet();
            if (rc != 0) {
                if (SplitLogManager.this.needAbandonRetries(rc, "GetData from znode " + path)) {
                    return;
                }
                if (rc == KeeperException.Code.NONODE.intValue()) {
                    SplitLogCounters.tot_mgr_get_data_nonode.incrementAndGet();
                    this.LOG.warn((Object)("task znode " + path + " vanished."));
                    if (this.completeTaskOnNoNode) {
                        try {
                            SplitLogManager.this.getDataSetWatchSuccess(path, null, Integer.MIN_VALUE);
                        }
                        catch (DeserializationException e) {
                            this.LOG.warn((Object)"Deserialization problem", (Throwable)e);
                        }
                    }
                    return;
                }
                Long retry_count = (Long)ctx;
                if (retry_count < 0L) {
                    this.LOG.warn((Object)("getdata rc = " + KeeperException.Code.get((int)rc) + " " + path + ". Ignoring error. No error handling. No retrying."));
                    return;
                }
                this.LOG.warn((Object)("getdata rc = " + KeeperException.Code.get((int)rc) + " " + path + " remaining retries=" + retry_count));
                if (retry_count == 0L) {
                    SplitLogCounters.tot_mgr_get_data_err.incrementAndGet();
                    SplitLogManager.this.getDataSetWatchFailure(path);
                } else {
                    SplitLogCounters.tot_mgr_get_data_retry.incrementAndGet();
                    SplitLogManager.this.getDataSetWatch(path, retry_count - 1L);
                }
                return;
            }
            try {
                SplitLogManager.this.getDataSetWatchSuccess(path, data, stat.getVersion());
            }
            catch (DeserializationException e) {
                this.LOG.warn((Object)"Deserialization problem", (Throwable)e);
            }
        }
    }

    class CreateAsyncCallback
    implements AsyncCallback.StringCallback {
        private final Log LOG = LogFactory.getLog(CreateAsyncCallback.class);

        CreateAsyncCallback() {
        }

        public void processResult(int rc, String path, Object ctx, String name) {
            SplitLogCounters.tot_mgr_node_create_result.incrementAndGet();
            if (rc != 0) {
                if (SplitLogManager.this.needAbandonRetries(rc, "Create znode " + path)) {
                    SplitLogManager.this.createNodeFailure(path);
                    return;
                }
                if (rc == KeeperException.Code.NODEEXISTS.intValue()) {
                    this.LOG.debug((Object)("found pre-existing znode " + path));
                    SplitLogCounters.tot_mgr_node_already_exists.incrementAndGet();
                } else {
                    Long retry_count = (Long)ctx;
                    this.LOG.warn((Object)("create rc =" + KeeperException.Code.get((int)rc) + " for " + path + " remaining retries=" + retry_count));
                    if (retry_count == 0L) {
                        SplitLogCounters.tot_mgr_node_create_err.incrementAndGet();
                        SplitLogManager.this.createNodeFailure(path);
                    } else {
                        SplitLogCounters.tot_mgr_node_create_retry.incrementAndGet();
                        SplitLogManager.this.createNode(path, retry_count - 1L);
                    }
                    return;
                }
            }
            SplitLogManager.this.createNodeSuccess(path);
        }
    }

    private class TimeoutMonitor
    extends Chore {
        private long lastLog;

        public TimeoutMonitor(int period, Stoppable stopper) {
            super("SplitLogManager Timeout Monitor", period, stopper);
            this.lastLog = 0L;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void chore() {
            long now;
            Task task;
            String path;
            Set localDeadWorkers;
            int resubmitted = 0;
            int unassigned = 0;
            int tot = 0;
            boolean found_assigned_task = false;
            Object object = SplitLogManager.this.deadWorkersLock;
            synchronized (object) {
                localDeadWorkers = SplitLogManager.this.deadWorkers;
                SplitLogManager.this.deadWorkers = null;
            }
            for (Map.Entry e : SplitLogManager.this.tasks.entrySet()) {
                path = (String)e.getKey();
                task = (Task)e.getValue();
                ServerName cur_worker = task.cur_worker_name;
                ++tot;
                if (task.isUnassigned()) {
                    ++unassigned;
                    continue;
                }
                found_assigned_task = true;
                if (localDeadWorkers != null && localDeadWorkers.contains(cur_worker)) {
                    SplitLogCounters.tot_mgr_resubmit_dead_server_task.incrementAndGet();
                    if (SplitLogManager.this.resubmit(path, task, ResubmitDirective.FORCE)) {
                        ++resubmitted;
                        continue;
                    }
                    SplitLogManager.this.handleDeadWorker(cur_worker);
                    LOG.warn((Object)("Failed to resubmit task " + path + " owned by dead " + cur_worker + ", will retry."));
                    continue;
                }
                if (!SplitLogManager.this.resubmit(path, task, ResubmitDirective.CHECK)) continue;
                ++resubmitted;
            }
            if (tot > 0 && (now = EnvironmentEdgeManager.currentTimeMillis()) > this.lastLog + 5000L) {
                this.lastLog = now;
                LOG.info((Object)("total tasks = " + tot + " unassigned = " + unassigned + " tasks=" + SplitLogManager.this.tasks));
            }
            if (resubmitted > 0) {
                LOG.info((Object)("resubmitted " + resubmitted + " out of " + tot + " tasks"));
            }
            if (tot > 0 && !found_assigned_task && EnvironmentEdgeManager.currentTimeMillis() - SplitLogManager.this.lastTaskCreateTime > SplitLogManager.this.unassignedTimeout) {
                for (Map.Entry e : SplitLogManager.this.tasks.entrySet()) {
                    path = (String)e.getKey();
                    task = (Task)e.getValue();
                    if (!task.isUnassigned() || task.status == TerminationStatus.FAILURE) continue;
                    SplitLogManager.this.tryGetDataSetWatch(path);
                }
                SplitLogManager.this.createRescanNode(Long.MAX_VALUE);
                SplitLogCounters.tot_mgr_resubmit_unassigned.incrementAndGet();
                LOG.debug((Object)"resubmitting unassigned task(s) after timeout");
            }
            if (SplitLogManager.this.failedDeletions.size() > 0) {
                ArrayList tmpPaths = new ArrayList(SplitLogManager.this.failedDeletions);
                for (String tmpPath : tmpPaths) {
                    SplitLogManager.this.deleteNode(tmpPath, SplitLogManager.this.zkretries);
                }
                SplitLogManager.this.failedDeletions.removeAll(tmpPaths);
            }
            long timeInterval = EnvironmentEdgeManager.currentTimeMillis() - SplitLogManager.this.lastRecoveringNodeCreationTime;
            if (!SplitLogManager.this.failedRecoveringRegionDeletions.isEmpty() || tot == 0 && SplitLogManager.this.tasks.size() == 0 && timeInterval > SplitLogManager.this.checkRecoveringTimeThreshold) {
                if (!SplitLogManager.this.failedRecoveringRegionDeletions.isEmpty()) {
                    ArrayList previouslyFailedDeletions = new ArrayList(SplitLogManager.this.failedRecoveringRegionDeletions);
                    SplitLogManager.this.failedRecoveringRegionDeletions.removeAll(previouslyFailedDeletions);
                    for (Pair failedDeletion : previouslyFailedDeletions) {
                        SplitLogManager.this.removeRecoveringRegionsFromZK((Set)failedDeletion.getFirst(), (Boolean)failedDeletion.getSecond());
                    }
                } else {
                    SplitLogManager.this.removeRecoveringRegionsFromZK(null, null);
                }
            }
        }
    }

    static class Task {
        volatile long last_update;
        volatile int last_version = -1;
        volatile ServerName cur_worker_name;
        volatile TaskBatch batch;
        volatile TerminationStatus status;
        volatile int incarnation = 0;
        final AtomicInteger unforcedResubmits = new AtomicInteger();
        volatile boolean resubmitThresholdReached;

        public String toString() {
            return "last_update = " + this.last_update + " last_version = " + this.last_version + " cur_worker_name = " + this.cur_worker_name + " status = " + (Object)((Object)this.status) + " incarnation = " + this.incarnation + " resubmits = " + this.unforcedResubmits.get() + " batch = " + this.batch;
        }

        Task() {
            this.status = TerminationStatus.IN_PROGRESS;
            this.setUnassigned();
        }

        public boolean isOrphan() {
            return this.batch == null || this.batch.isDead;
        }

        public boolean isUnassigned() {
            return this.cur_worker_name == null;
        }

        public void heartbeatNoDetails(long time) {
            this.last_update = time;
        }

        public void heartbeat(long time, int version, ServerName worker) {
            this.last_version = version;
            this.last_update = time;
            this.cur_worker_name = worker;
        }

        public void setUnassigned() {
            this.cur_worker_name = null;
            this.last_update = -1L;
        }
    }

    static class TaskBatch {
        int installed = 0;
        int done = 0;
        int error = 0;
        volatile boolean isDead = false;

        TaskBatch() {
        }

        public String toString() {
            return "installed = " + this.installed + " done = " + this.done + " error = " + this.error;
        }
    }
}

