/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.storage;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.hadoop.hdds.client.BlockID;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.datanode.proto.ContainerProtos;
import org.apache.hadoop.hdds.scm.ContainerClientMetrics;
import org.apache.hadoop.hdds.scm.OzoneClientConfig;
import org.apache.hadoop.hdds.scm.XceiverClientFactory;
import org.apache.hadoop.hdds.scm.XceiverClientReply;
import org.apache.hadoop.hdds.scm.XceiverClientSpi;
import org.apache.hadoop.hdds.scm.container.common.helpers.StorageContainerException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.storage.BufferPool;
import org.apache.hadoop.hdds.scm.storage.ContainerProtocolCalls;
import org.apache.hadoop.ozone.common.Checksum;
import org.apache.hadoop.ozone.common.ChecksumData;
import org.apache.hadoop.ozone.common.ChunkBuffer;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.security.token.TokenIdentifier;
import org.apache.ratis.thirdparty.com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BlockOutputStream
extends OutputStream {
    public static final Logger LOG = LoggerFactory.getLogger(BlockOutputStream.class);
    public static final String EXCEPTION_MSG = "Unexpected Storage Container Exception: ";
    private AtomicReference<BlockID> blockID;
    private final AtomicReference<ContainerProtos.ChunkInfo> previousChunkInfo = new AtomicReference();
    private final ContainerProtos.BlockData.Builder containerBlockData;
    private XceiverClientFactory xceiverClientFactory;
    private XceiverClientSpi xceiverClient;
    private OzoneClientConfig config;
    private int chunkIndex;
    private final AtomicLong chunkOffset = new AtomicLong();
    private final BufferPool bufferPool;
    private final AtomicReference<IOException> ioException;
    private final ExecutorService responseExecutor;
    private long totalDataFlushedLength;
    private long writtenDataLength;
    private List<ChunkBuffer> bufferList;
    private final List<DatanodeDetails> failedServers;
    private final Checksum checksum;
    private int flushPeriod;
    private int currentBufferRemaining;
    private ChunkBuffer currentBuffer;
    private final Token<? extends TokenIdentifier> token;
    private int replicationIndex;
    private Pipeline pipeline;
    private final ContainerClientMetrics clientMetrics;

    public BlockOutputStream(BlockID blockID, XceiverClientFactory xceiverClientManager, Pipeline pipeline, BufferPool bufferPool, OzoneClientConfig config, Token<? extends TokenIdentifier> token, ContainerClientMetrics clientMetrics) throws IOException {
        this.xceiverClientFactory = xceiverClientManager;
        this.config = config;
        this.blockID = new AtomicReference<BlockID>(blockID);
        this.replicationIndex = pipeline.getReplicaIndex(pipeline.getClosestNode());
        ContainerProtos.KeyValue keyValue = ContainerProtos.KeyValue.newBuilder().setKey("TYPE").setValue("KEY").build();
        ContainerProtos.DatanodeBlockID.Builder blkIDBuilder = ContainerProtos.DatanodeBlockID.newBuilder().setContainerID(blockID.getContainerID()).setLocalID(blockID.getLocalID()).setBlockCommitSequenceId(blockID.getBlockCommitSequenceId());
        if (this.replicationIndex > 0) {
            blkIDBuilder.setReplicaIndex(this.replicationIndex);
        }
        this.containerBlockData = ContainerProtos.BlockData.newBuilder().setBlockID(blkIDBuilder.build()).addMetadata(keyValue);
        this.xceiverClient = xceiverClientManager.acquireClient(pipeline);
        this.bufferPool = bufferPool;
        this.token = token;
        this.refreshCurrentBuffer();
        this.flushPeriod = (int)(config.getStreamBufferFlushSize() / (long)config.getStreamBufferSize());
        Preconditions.checkArgument(((long)this.flushPeriod * (long)config.getStreamBufferSize() == config.getStreamBufferFlushSize() ? 1 : 0) != 0);
        this.responseExecutor = Executors.newSingleThreadExecutor();
        this.bufferList = null;
        this.totalDataFlushedLength = 0L;
        this.writtenDataLength = 0L;
        this.failedServers = new ArrayList<DatanodeDetails>(0);
        this.ioException = new AtomicReference<Object>(null);
        this.checksum = new Checksum(config.getChecksumType(), config.getBytesPerChecksum());
        this.clientMetrics = clientMetrics;
        this.pipeline = pipeline;
    }

    void refreshCurrentBuffer() {
        this.currentBuffer = this.bufferPool.getCurrentBuffer();
        this.currentBufferRemaining = this.currentBuffer != null ? this.currentBuffer.remaining() : 0;
    }

    public BlockID getBlockID() {
        return this.blockID.get();
    }

    public long getTotalAckDataLength() {
        return 0L;
    }

    public long getWrittenDataLength() {
        return this.writtenDataLength;
    }

    public List<DatanodeDetails> getFailedServers() {
        return this.failedServers;
    }

    @VisibleForTesting
    public XceiverClientSpi getXceiverClient() {
        return this.xceiverClient;
    }

    @VisibleForTesting
    public long getTotalDataFlushedLength() {
        return this.totalDataFlushedLength;
    }

    @VisibleForTesting
    public BufferPool getBufferPool() {
        return this.bufferPool;
    }

    public IOException getIoException() {
        return this.ioException.get();
    }

    XceiverClientSpi getXceiverClientSpi() {
        return this.xceiverClient;
    }

    public ContainerProtos.BlockData.Builder getContainerBlockData() {
        return this.containerBlockData;
    }

    public Pipeline getPipeline() {
        return this.pipeline;
    }

    Token<? extends TokenIdentifier> getToken() {
        return this.token;
    }

    ExecutorService getResponseExecutor() {
        return this.responseExecutor;
    }

    @Override
    public void write(int b) throws IOException {
        this.checkOpen();
        this.allocateNewBufferIfNeeded();
        this.currentBuffer.put((byte)b);
        --this.currentBufferRemaining;
        this.writeChunkIfNeeded();
        ++this.writtenDataLength;
        this.doFlushOrWatchIfNeeded();
    }

    private void writeChunkIfNeeded() throws IOException {
        if (this.currentBufferRemaining == 0) {
            this.writeChunk(this.currentBuffer);
        }
    }

    @Override
    public void write(byte[] b, int off, int len) throws IOException {
        this.checkOpen();
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException();
        }
        if (len == 0) {
            return;
        }
        while (len > 0) {
            this.allocateNewBufferIfNeeded();
            int writeLen = Math.min(this.currentBufferRemaining, len);
            this.currentBuffer.put(b, off, writeLen);
            this.currentBufferRemaining -= writeLen;
            this.writeChunkIfNeeded();
            off += writeLen;
            len -= writeLen;
            this.updateWrittenDataLength(writeLen);
            this.doFlushOrWatchIfNeeded();
        }
    }

    public void updateWrittenDataLength(int writeLen) {
        this.writtenDataLength += (long)writeLen;
    }

    private void doFlushOrWatchIfNeeded() throws IOException {
        if (this.currentBufferRemaining == 0) {
            if (this.bufferPool.getNumberOfUsedBuffers() % this.flushPeriod == 0) {
                this.updateFlushLength();
                this.executePutBlock(false, false);
            }
            if (this.bufferPool.getNumberOfUsedBuffers() == this.bufferPool.getCapacity()) {
                this.handleFullBuffer();
            }
        }
    }

    private void allocateNewBufferIfNeeded() {
        if (this.currentBufferRemaining == 0) {
            this.currentBuffer = this.bufferPool.allocateBuffer(this.config.getBufferIncrement());
            this.currentBufferRemaining = this.currentBuffer.remaining();
        }
    }

    private void updateFlushLength() {
        this.totalDataFlushedLength = this.writtenDataLength;
    }

    private boolean isBufferPoolFull() {
        return this.bufferPool.computeBufferData() == this.config.getStreamBufferMaxSize();
    }

    public void writeOnRetry(long len) throws IOException {
        if (len == 0L) {
            return;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrying write length {} for blockID {}", (Object)len, this.blockID);
        }
        Preconditions.checkArgument((len <= this.config.getStreamBufferMaxSize() ? 1 : 0) != 0);
        int count = 0;
        while (len > 0L) {
            ChunkBuffer buffer = this.bufferPool.getBuffer(count);
            long writeLen = Math.min((long)buffer.position(), len);
            if (!buffer.hasRemaining()) {
                this.writeChunk(buffer);
            }
            len -= writeLen;
            ++count;
            this.writtenDataLength += writeLen;
            if (this.writtenDataLength % this.config.getStreamBufferFlushSize() == 0L) {
                this.updateFlushLength();
                this.executePutBlock(false, false);
            }
            if (this.writtenDataLength != this.config.getStreamBufferMaxSize()) continue;
            this.handleFullBuffer();
        }
    }

    private void handleFullBuffer() throws IOException {
        this.waitForFlushAndCommit(true);
    }

    void waitForFlushAndCommit(boolean bufferFull) throws IOException {
        try {
            this.checkOpen();
            this.waitOnFlushFutures();
        }
        catch (ExecutionException e) {
            this.handleExecutionException(e);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, true);
        }
        this.watchForCommit(bufferFull);
    }

    void releaseBuffersOnException() {
    }

    private void adjustBuffersOnException() {
        this.releaseBuffersOnException();
        this.refreshCurrentBuffer();
    }

    XceiverClientReply sendWatchForCommit(boolean bufferFull) throws IOException {
        return null;
    }

    private void watchForCommit(boolean bufferFull) throws IOException {
        this.checkOpen();
        try {
            List dnList;
            XceiverClientReply reply = this.sendWatchForCommit(bufferFull);
            if (reply != null && !(dnList = reply.getDatanodes()).isEmpty()) {
                Pipeline pipe = this.xceiverClient.getPipeline();
                LOG.warn("Failed to commit BlockId {} on {}. Failed nodes: {}", new Object[]{this.blockID, pipe, dnList});
                this.failedServers.addAll(dnList);
            }
        }
        catch (IOException ioe) {
            this.setIoException(ioe);
            throw this.getIoException();
        }
        this.refreshCurrentBuffer();
    }

    void updateCommitInfo(XceiverClientReply reply, List<ChunkBuffer> buffers) {
    }

    CompletableFuture<ContainerProtos.ContainerCommandResponseProto> executePutBlock(boolean close, boolean force) throws IOException {
        List<ChunkBuffer> byteBufferList;
        this.checkOpen();
        long flushPos = this.totalDataFlushedLength;
        if (!force) {
            Preconditions.checkNotNull(this.bufferList);
            byteBufferList = this.bufferList;
            this.bufferList = null;
            Preconditions.checkNotNull(byteBufferList);
        } else {
            byteBufferList = null;
        }
        CompletionStage flushFuture = null;
        try {
            ContainerProtos.BlockData blockData = this.containerBlockData.build();
            XceiverClientReply asyncReply = ContainerProtocolCalls.putBlockAsync((XceiverClientSpi)this.xceiverClient, (ContainerProtos.BlockData)blockData, (boolean)close, this.token);
            CompletableFuture future = asyncReply.getResponse();
            flushFuture = ((CompletableFuture)future.thenApplyAsync(e -> {
                try {
                    this.validateResponse((ContainerProtos.ContainerCommandResponseProto)e);
                }
                catch (IOException sce) {
                    throw new CompletionException(sce);
                }
                if (this.getIoException() == null && !force) {
                    BlockID responseBlockID = BlockID.getFromProtobuf((ContainerProtos.DatanodeBlockID)e.getPutBlock().getCommittedBlockLength().getBlockID());
                    Preconditions.checkState((boolean)this.blockID.get().getContainerBlockID().equals((Object)responseBlockID.getContainerBlockID()));
                    this.blockID.set(responseBlockID);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Adding index " + asyncReply.getLogIndex() + " flushLength " + flushPos + " numBuffers " + byteBufferList.size() + " blockID " + this.blockID + " bufferPool size" + this.bufferPool.getSize() + " currentBufferIndex " + this.bufferPool.getCurrentBufferIndex());
                    }
                    this.updateCommitInfo(asyncReply, byteBufferList);
                }
                return e;
            }, (Executor)this.responseExecutor)).exceptionally(e -> {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("putBlock failed for blockID {} with exception {}", this.blockID, (Object)e.getLocalizedMessage());
                }
                CompletionException ce = new CompletionException((Throwable)e);
                this.setIoException(ce);
                throw ce;
            });
        }
        catch (IOException | ExecutionException e2) {
            throw new IOException(EXCEPTION_MSG + e2.toString(), e2);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, false);
        }
        this.putFlushFuture(flushPos, (CompletableFuture<ContainerProtos.ContainerCommandResponseProto>)flushFuture);
        return flushFuture;
    }

    void putFlushFuture(long flushPos, CompletableFuture<ContainerProtos.ContainerCommandResponseProto> flushFuture) {
    }

    @Override
    public void flush() throws IOException {
        if (!(this.xceiverClientFactory == null || this.xceiverClient == null || this.bufferPool == null || this.bufferPool.getSize() <= 0 || this.config.isStreamBufferFlushDelay() && this.writtenDataLength - this.totalDataFlushedLength < (long)this.config.getStreamBufferSize())) {
            this.handleFlush(false);
        }
    }

    private void writeChunk(ChunkBuffer buffer) throws IOException {
        if (this.bufferList == null) {
            this.bufferList = new ArrayList<ChunkBuffer>();
        }
        this.bufferList.add(buffer);
        this.writeChunkToContainer(buffer.duplicate(0, buffer.position()));
    }

    protected void handleFlush(boolean close) throws IOException {
        try {
            this.handleFlushInternal(close);
        }
        catch (ExecutionException e) {
            this.handleExecutionException(e);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, true);
        }
        catch (Throwable e) {
            String msg = "Failed to flush. error: " + e.getMessage();
            LOG.error(msg, e);
            throw e;
        }
        finally {
            if (close) {
                this.cleanup(false);
            }
        }
    }

    private void handleFlushInternal(boolean close) throws IOException, InterruptedException, ExecutionException {
        this.checkOpen();
        if (this.totalDataFlushedLength < this.writtenDataLength) {
            this.refreshCurrentBuffer();
            Preconditions.checkArgument((this.currentBuffer.position() > 0 ? 1 : 0) != 0);
            if (this.currentBuffer.hasRemaining()) {
                this.writeChunk(this.currentBuffer);
            }
            this.updateFlushLength();
            this.executePutBlock(close, false);
        } else if (close) {
            this.executePutBlock(true, true);
        }
        this.waitOnFlushFutures();
        this.watchForCommit(false);
        this.checkOpen();
    }

    @Override
    public void close() throws IOException {
        if (this.xceiverClientFactory != null && this.xceiverClient != null) {
            if (this.bufferPool != null && this.bufferPool.getSize() > 0) {
                this.handleFlush(true);
            } else {
                this.cleanup(false);
            }
        }
    }

    void waitOnFlushFutures() throws InterruptedException, ExecutionException {
    }

    void validateResponse(ContainerProtos.ContainerCommandResponseProto responseProto) throws IOException {
        try {
            IOException exception = this.getIoException();
            if (exception != null) {
                throw exception;
            }
            ContainerProtocolCalls.validateContainerResponse((ContainerProtos.ContainerCommandResponseProto)responseProto);
        }
        catch (StorageContainerException sce) {
            this.setIoException((Exception)((Object)sce));
            throw sce;
        }
    }

    public void setIoException(Exception e) {
        IOException ioe = this.getIoException();
        if (ioe == null) {
            IOException exception = new IOException(EXCEPTION_MSG + e.toString(), e);
            this.ioException.compareAndSet(null, exception);
            LOG.debug("Exception: for block ID: " + this.blockID, (Throwable)e);
        } else {
            LOG.debug("Previous request had already failed with {} so subsequent request also encounters Storage Container Exception {}", (Object)ioe, (Object)e);
        }
    }

    void cleanup() {
    }

    public void cleanup(boolean invalidateClient) {
        if (this.xceiverClientFactory != null) {
            this.xceiverClientFactory.releaseClient(this.xceiverClient, invalidateClient);
        }
        this.xceiverClientFactory = null;
        this.xceiverClient = null;
        this.cleanup();
        if (this.bufferList != null) {
            this.bufferList.clear();
        }
        this.bufferList = null;
        this.responseExecutor.shutdown();
    }

    void checkOpen() throws IOException {
        if (this.isClosed()) {
            throw new IOException("BlockOutputStream has been closed.");
        }
        if (this.getIoException() != null) {
            this.adjustBuffersOnException();
            throw this.getIoException();
        }
    }

    public boolean isClosed() {
        return this.xceiverClient == null;
    }

    CompletableFuture<ContainerProtos.ContainerCommandResponseProto> writeChunkToContainer(ChunkBuffer chunk) throws IOException {
        long expectedOffset;
        ContainerProtos.ChunkInfo previous;
        int effectiveChunkSize = chunk.remaining();
        long offset = this.chunkOffset.getAndAdd(effectiveChunkSize);
        ByteString data = chunk.toByteString(this.bufferPool.byteStringConversion());
        ChecksumData checksumData = this.checksum.computeChecksum(chunk);
        ContainerProtos.ChunkInfo chunkInfo = ContainerProtos.ChunkInfo.newBuilder().setChunkName(this.blockID.get().getLocalID() + "_chunk_" + ++this.chunkIndex).setOffset(offset).setLen((long)effectiveChunkSize).setChecksumData(checksumData.getProtoBufMessage()).build();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Writing chunk {} length {} at offset {}", new Object[]{chunkInfo.getChunkName(), effectiveChunkSize, offset});
        }
        long l = (previous = this.previousChunkInfo.getAndSet(chunkInfo)) == null ? 0L : (expectedOffset = chunkInfo.getChunkName().equals(previous.getChunkName()) ? previous.getOffset() : previous.getOffset() + previous.getLen());
        if (chunkInfo.getOffset() != expectedOffset) {
            throw new IOException("Unexpected offset: " + chunkInfo.getOffset() + "(actual) != " + expectedOffset + "(expected), " + this.blockID + ", chunkInfo = " + chunkInfo + ", previous = " + previous);
        }
        try {
            XceiverClientReply asyncReply = ContainerProtocolCalls.writeChunkAsync((XceiverClientSpi)this.xceiverClient, (ContainerProtos.ChunkInfo)chunkInfo, (BlockID)this.blockID.get(), (ByteString)data, this.token, (int)this.replicationIndex);
            CompletableFuture respFuture = asyncReply.getResponse();
            CompletionStage validateFuture = ((CompletableFuture)respFuture.thenApplyAsync(e -> {
                try {
                    this.validateResponse((ContainerProtos.ContainerCommandResponseProto)e);
                }
                catch (IOException sce) {
                    respFuture.completeExceptionally(sce);
                }
                return e;
            }, (Executor)this.responseExecutor)).exceptionally(e -> {
                String msg = "Failed to write chunk " + chunkInfo.getChunkName() + " into block " + this.blockID;
                LOG.debug("{}, exception: {}", (Object)msg, (Object)e.getLocalizedMessage());
                CompletionException ce = new CompletionException(msg, (Throwable)e);
                this.setIoException(ce);
                throw ce;
            });
            this.containerBlockData.addChunks(chunkInfo);
            this.clientMetrics.recordWriteChunk(this.pipeline, chunkInfo.getLen());
            return validateFuture;
        }
        catch (IOException | ExecutionException e2) {
            throw new IOException(EXCEPTION_MSG + e2.toString(), e2);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            this.handleInterruptedException(ex, false);
            return null;
        }
    }

    @VisibleForTesting
    public void setXceiverClient(XceiverClientSpi xceiverClient) {
        this.xceiverClient = xceiverClient;
    }

    void handleInterruptedException(Exception ex, boolean processExecutionException) throws IOException {
        LOG.error("Command execution was interrupted.");
        if (!processExecutionException) {
            throw new IOException(EXCEPTION_MSG + ex.toString(), ex);
        }
        this.handleExecutionException(ex);
    }

    private void handleExecutionException(Exception ex) throws IOException {
        this.setIoException(ex);
        this.adjustBuffersOnException();
        throw this.getIoException();
    }

    public int getReplicationIndex() {
        return this.replicationIndex;
    }
}

