/*
 * Decompiled with CFR 0.152.
 */
package org.apache.seatunnel.engine.core.classloader;

import com.google.common.annotations.VisibleForTesting;
import java.net.URL;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.seatunnel.engine.common.loader.SeaTunnelChildFirstClassLoader;
import org.apache.seatunnel.engine.core.classloader.ClassLoaderService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultClassLoaderService
implements ClassLoaderService {
    private static final Logger log = LoggerFactory.getLogger(DefaultClassLoaderService.class);
    private final boolean cacheMode;
    private final Map<Long, Map<String, ClassLoader>> classLoaderCache;
    private final Map<Long, Map<String, AtomicInteger>> classLoaderReferenceCount;

    public DefaultClassLoaderService(boolean cacheMode) {
        this.cacheMode = cacheMode;
        this.classLoaderCache = new ConcurrentHashMap<Long, Map<String, ClassLoader>>();
        this.classLoaderReferenceCount = new ConcurrentHashMap<Long, Map<String, AtomicInteger>>();
        log.info("start classloader service" + (cacheMode ? " with cache mode" : ""));
    }

    @Override
    public synchronized ClassLoader getClassLoader(long jobId, Collection<URL> jars) {
        String key;
        Map<String, ClassLoader> classLoaderMap;
        log.debug("Get classloader for job {} with jars {}", (Object)jobId, (Object)jars);
        if (this.cacheMode) {
            jobId = 1L;
        }
        if (!this.classLoaderCache.containsKey(jobId)) {
            this.classLoaderCache.put(jobId, new ConcurrentHashMap());
            this.classLoaderReferenceCount.put(jobId, new ConcurrentHashMap());
        }
        if ((classLoaderMap = this.classLoaderCache.get(jobId)).containsKey(key = this.covertJarsToKey(jars))) {
            this.classLoaderReferenceCount.get(jobId).get(key).incrementAndGet();
            return classLoaderMap.get(key);
        }
        SeaTunnelChildFirstClassLoader classLoader = new SeaTunnelChildFirstClassLoader(jars);
        log.info("Create classloader for job {} with jars {}", (Object)jobId, (Object)jars);
        classLoaderMap.put(key, classLoader);
        this.classLoaderReferenceCount.get(jobId).put(key, new AtomicInteger(1));
        return classLoader;
    }

    @Override
    public synchronized void releaseClassLoader(long jobId, Collection<URL> jars) {
        String key;
        log.debug("Release classloader for job {} with jars {}", (Object)jobId, (Object)jars);
        if (this.cacheMode) {
            jobId = 1L;
        }
        if (!this.classLoaderCache.containsKey(jobId)) {
            return;
        }
        Map<String, ClassLoader> classLoaderMap = this.classLoaderCache.get(jobId);
        if (!classLoaderMap.containsKey(key = this.covertJarsToKey(jars))) {
            return;
        }
        int referenceCount = this.classLoaderReferenceCount.get(jobId).get(key).decrementAndGet();
        log.debug("Reference count for job {} with jars {} is {}", jobId, jars, referenceCount);
        if (this.cacheMode) {
            return;
        }
        if (referenceCount == 0) {
            ClassLoader classLoader = classLoaderMap.remove(key);
            log.info("Release classloader for job {} with jars {}", (Object)jobId, (Object)jars);
            this.classLoaderReferenceCount.get(jobId).remove(key);
            DefaultClassLoaderService.recycleClassLoaderFromThread(classLoader);
        }
        if (classLoaderMap.isEmpty()) {
            this.classLoaderCache.remove(jobId);
            this.classLoaderReferenceCount.remove(jobId);
        }
    }

    private static void recycleClassLoaderFromThread(ClassLoader classLoader) {
        Thread.getAllStackTraces().keySet().stream().filter(thread -> thread.getContextClassLoader() == classLoader).forEach(thread -> {
            log.info("recycle classloader for thread " + thread.getName());
            thread.setContextClassLoader(null);
        });
    }

    private String covertJarsToKey(Collection<URL> jars) {
        return jars.stream().map(URL::toString).sorted().reduce((a, b) -> a + b).orElse("");
    }

    @VisibleForTesting
    Optional<ClassLoader> queryClassLoaderById(long jobId, Collection<URL> jars) {
        String key;
        if (this.cacheMode) {
            jobId = 1L;
        }
        if (!this.classLoaderCache.containsKey(jobId)) {
            return Optional.empty();
        }
        Map<String, ClassLoader> classLoaderMap = this.classLoaderCache.get(jobId);
        if (!classLoaderMap.containsKey(key = this.covertJarsToKey(jars))) {
            return Optional.empty();
        }
        return Optional.of(classLoaderMap.get(key));
    }

    @VisibleForTesting
    int queryClassLoaderReferenceCount(long jobId, Collection<URL> jars) {
        String key;
        if (this.cacheMode) {
            jobId = 1L;
        }
        if (!this.classLoaderCache.containsKey(jobId)) {
            return 0;
        }
        Map<String, AtomicInteger> classLoaderMap = this.classLoaderReferenceCount.get(jobId);
        if (!classLoaderMap.containsKey(key = this.covertJarsToKey(jars))) {
            return 0;
        }
        return classLoaderMap.get(key).get();
    }

    @VisibleForTesting
    int queryClassLoaderCount() {
        AtomicInteger count = new AtomicInteger();
        this.classLoaderCache.values().forEach(map -> count.addAndGet(map.size()));
        return count.get();
    }

    @Override
    public void close() {
        log.info("close classloader service");
        this.classLoaderCache.clear();
        this.classLoaderReferenceCount.clear();
    }
}

