/*
 * Decompiled with CFR 0.152.
 */
package org.apache.sedona.common.raster.netcdf;

import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import org.apache.sedona.common.raster.RasterConstructors;
import org.geotools.coverage.grid.GridCoverage2D;
import org.opengis.referencing.FactoryException;
import org.opengis.referencing.datum.PixelInCell;
import ucar.ma2.Array;
import ucar.nc2.Attribute;
import ucar.nc2.AttributeContainer;
import ucar.nc2.Dimension;
import ucar.nc2.Group;
import ucar.nc2.NetcdfFile;
import ucar.nc2.Variable;

public class NetCdfReader {
    public static String getRecordInfo(NetcdfFile netcdfFile) {
        ImmutableList<Variable> recordVariables = NetCdfReader.getRecordVariables(netcdfFile);
        if (Objects.isNull(recordVariables) || recordVariables.isEmpty()) {
            return "";
        }
        int numRecordVariables = recordVariables.size();
        StringBuilder recordInfo = new StringBuilder();
        for (int i = 0; i < numRecordVariables; ++i) {
            Variable variable = (Variable)recordVariables.get(i);
            recordInfo.append(variable.getNameAndDimensions(false));
            if (i == numRecordVariables - 1) continue;
            recordInfo.append("\n\n");
        }
        return recordInfo.toString();
    }

    public static GridCoverage2D getRaster(NetcdfFile netcdfFile, String variableShortName) throws FactoryException, IOException {
        Variable recordVariable = NetCdfReader.getRecordVariableFromShortName(netcdfFile, variableShortName);
        List variableDimensions = recordVariable.getDimensions();
        int numDimensions = ((AbstractCollection)((Object)variableDimensions)).size();
        if (numDimensions < 2) {
            throw new IllegalArgumentException("Provided variable has less than two dimensions");
        }
        String latDimensionShortName = ((Dimension)variableDimensions.get(numDimensions - 2)).getShortName();
        String lonDimensionShortName = ((Dimension)variableDimensions.get(numDimensions - 1)).getShortName();
        Variable[] coordVariablePair = NetCdfReader.getCoordVariables(netcdfFile, latDimensionShortName, lonDimensionShortName);
        Variable latVariable = coordVariablePair[1];
        Variable lonVariable = coordVariablePair[0];
        return NetCdfReader.getRasterHelper(netcdfFile, recordVariable, lonVariable, latVariable, false);
    }

    public static GridCoverage2D getRaster(NetcdfFile netcdfFile, String variableShortName, String latDimShortName, String lonDimShortName) throws FactoryException, IOException {
        Variable recordVariable = NetCdfReader.getRecordVariableFromShortName(netcdfFile, variableShortName);
        Variable[] coordVariablePair = NetCdfReader.getCoordVariables(netcdfFile, latDimShortName, lonDimShortName);
        Variable latVariable = coordVariablePair[1];
        Variable lonVariable = coordVariablePair[0];
        return NetCdfReader.getRasterHelper(netcdfFile, recordVariable, lonVariable, latVariable, true);
    }

    public static Map getRasterMetadata(GridCoverage2D raster) {
        return raster.getProperties();
    }

    private static Variable getRecordVariableFromShortName(NetcdfFile netcdfFile, String variableShortName) {
        Variable recordVariable = NetCdfReader.findVariableRecursive(variableShortName, netcdfFile.getRootGroup());
        NetCdfReader.ensureRecordVar(recordVariable);
        return recordVariable;
    }

    private static Variable[] getCoordVariables(NetcdfFile netcdfFile, String latVariableShortName, String lonVariableShortName) {
        Group rootGroup = netcdfFile.getRootGroup();
        Variable lonVariable = NetCdfReader.findVariableRecursive(lonVariableShortName, rootGroup);
        Variable latVariable = NetCdfReader.findVariableRecursive(latVariableShortName, rootGroup);
        NetCdfReader.ensureCoordVar(lonVariable, latVariable);
        return new Variable[]{lonVariable, latVariable};
    }

    private static ImmutableList<Variable> getRecordVariables(NetcdfFile netcdfFile) {
        ImmutableList<Variable> variables = netcdfFile.getVariables();
        ArrayList<Variable> recordVariables = null;
        for (Variable variable : variables) {
            List variableDimensions = variable.getDimensions();
            if (((AbstractCollection)((Object)variableDimensions)).size() < 2) continue;
            if (Objects.isNull(recordVariables)) {
                recordVariables = new ArrayList<Variable>();
            }
            recordVariables.add(variable);
        }
        if (Objects.isNull(recordVariables)) {
            return null;
        }
        return ImmutableList.copyOf(recordVariables);
    }

    private static GridCoverage2D getRasterHelper(NetcdfFile netcdfFile, Variable recordVariable, Variable lonVariable, Variable latVariable, boolean findCoordIndices) throws IOException, FactoryException {
        Attribute noDataAttribute;
        double lastVal;
        ImmutableList<Attribute> globalAttributes = netcdfFile.getGlobalAttributes();
        Group rootGroup = netcdfFile.getRootGroup();
        double minX = 0.0;
        double maxY = 0.0;
        double skewX = 0.0;
        double skewY = 0.0;
        boolean isIncreasing = false;
        HashMap<String, List<String>> varMetadata = new HashMap<String, List<String>>();
        Array lonData = lonVariable.read();
        int width = lonData.getShape()[0];
        double firstVal = NetCdfReader.getElement(lonData, 0);
        isIncreasing = firstVal < (lastVal = NetCdfReader.getElement(lonData, width - 1).doubleValue());
        minX = isIncreasing ? firstVal : lastVal;
        double translateX = Math.abs(lastVal - firstVal) / (double)(width - 1);
        Array latData = latVariable.read();
        int height = latData.getShape()[0];
        firstVal = NetCdfReader.getElement(latData, 0);
        lastVal = NetCdfReader.getElement(latData, height - 1);
        isIncreasing = firstVal < lastVal;
        maxY = isIncreasing ? lastVal : firstVal;
        double translateY = Math.abs(lastVal - firstVal) / (double)(height - 1);
        String lonVariableName = lonVariable.getShortName();
        String latVariableName = latVariable.getShortName();
        List recordDimensions = recordVariable.getDimensions();
        int numRecordDimensions = recordDimensions.size();
        int lonIndex = numRecordDimensions - 1;
        int latIndex = numRecordDimensions - 2;
        if (findCoordIndices) {
            lonIndex = recordVariable.findDimensionIndex(lonVariableName);
            latIndex = recordVariable.findDimensionIndex(latVariableName);
            if (latIndex == -1 || lonIndex == -1) {
                throw new IllegalArgumentException("Given record variable does not contain one of the latitude or longitude dimensions as the participating dimensions");
            }
        }
        Array recordData = recordVariable.read();
        Double noDataValue = null;
        if (recordVariable.attributes().hasAttribute("missing_value") && !Objects.isNull(noDataAttribute = recordVariable.findAttribute("missing_value"))) {
            noDataValue = NetCdfReader.getAttrDoubleValue(noDataAttribute);
        }
        int[] strides = recordData.getIndex().getShape();
        int[] pointers = recordData.getIndex().getCurrentCounter();
        int numBands = 1;
        int numValuesPerBand = 1;
        for (int i = 0; i < numRecordDimensions; ++i) {
            Dimension recordDimension = (Dimension)recordDimensions.get(i);
            String dimensionShortName = recordDimension.getShortName();
            if (dimensionShortName.equalsIgnoreCase(latVariableName) || dimensionShortName.equalsIgnoreCase(lonVariableName)) {
                numValuesPerBand *= strides[i];
                continue;
            }
            numBands *= strides[i];
        }
        Array[] dimensionVars = new Array[numRecordDimensions];
        double scaleFactor = 1.0;
        double offset = 0.0;
        for (int i = 0; i < numRecordDimensions; ++i) {
            if (i == lonIndex) {
                dimensionVars[i] = lonData;
                continue;
            }
            if (i == latIndex) {
                dimensionVars[i] = latData;
                continue;
            }
            Dimension recordDimension = (Dimension)recordDimensions.get(i);
            String dimensionShortName = recordDimension.getShortName();
            Variable recordDimVar = NetCdfReader.findVariableRecursive(dimensionShortName, rootGroup);
            if (Objects.isNull(recordDimVar)) {
                throw new IllegalArgumentException(String.format("Corresponding coordinate variable not found for dimension %s", dimensionShortName));
            }
            AttributeContainer recordDimAttrs = recordDimVar.attributes();
            varMetadata.computeIfAbsent(dimensionShortName, k -> new ArrayList());
            for (Attribute attr : recordDimAttrs) {
                Array values;
                varMetadata.get(dimensionShortName).add(attr.getStringValue());
                if (attr.getShortName().equalsIgnoreCase("scale_factor") && !Objects.isNull(values = attr.getValues())) {
                    scaleFactor = NetCdfReader.getElement(attr.getValues(), 0);
                }
                if (!attr.getShortName().equalsIgnoreCase("add_offset") || Objects.isNull(values = attr.getValues())) continue;
                offset = NetCdfReader.getElement(attr.getValues(), 0);
            }
            dimensionVars[i] = recordDimVar.read();
        }
        double[][] allBandValues = new double[numBands][numValuesPerBand];
        ArrayList<List<String>> bandMetaData = new ArrayList<List<String>>();
        for (int currBandNum = 0; currBandNum < numBands; ++currBandNum) {
            if (dimensionVars.length > 2) {
                ArrayList<String> currBandMetadata = new ArrayList<String>();
                for (int metadataDim = dimensionVars.length - 3; metadataDim >= 0; --metadataDim) {
                    double data = NetCdfReader.getElement(dimensionVars[metadataDim], pointers[metadataDim]);
                    String metadata = ((Dimension)recordDimensions.get(metadataDim)).getShortName() + " : " + data;
                    currBandMetadata.add(metadata);
                }
                bandMetaData.add(currBandMetadata);
            }
            for (int j = height - 1; j >= 0; --j) {
                for (int i = 0; i < width; ++i) {
                    double currRecord;
                    int index = j * width + i;
                    int dataIndex = currBandNum * (width * height) + ((height - 1 - j) * width + i);
                    allBandValues[currBandNum][index] = currRecord = scaleFactor * NetCdfReader.getElement(recordData, dataIndex) + offset;
                }
            }
            boolean addCarry = true;
            for (int currDim = dimensionVars.length - 3; currDim >= 0; --currDim) {
                int maxIndex = strides[currDim];
                if (addCarry) {
                    int n = currDim;
                    pointers[n] = pointers[n] + 1;
                    addCarry = false;
                }
                if (pointers[currDim] != maxIndex) continue;
                pointers[currDim] = 0;
                addCarry = true;
            }
        }
        Map<String, List<String>> rasterProperties = NetCdfReader.getRasterProperties(globalAttributes, varMetadata, bandMetaData);
        return RasterConstructors.makeNonEmptyRaster(numBands, width, height, minX, maxY, translateX, -translateY, skewX, skewY, 0, allBandValues, rasterProperties, noDataValue, PixelInCell.CELL_CENTER);
    }

    private static Variable findVariableRecursive(String shortName, Group currGroup) {
        if (!Objects.isNull(currGroup.findVariableLocal(shortName))) {
            return currGroup.findVariableLocal(shortName);
        }
        for (Group group : currGroup.getGroups()) {
            Variable ret = NetCdfReader.findVariableRecursive(shortName, group);
            if (Objects.isNull(ret)) continue;
            return ret;
        }
        return null;
    }

    private static Map<String, List<String>> getRasterProperties(ImmutableList<Attribute> globalAttrs, Map<String, List<String>> varAttrs, List<List<String>> bandAttrs) {
        HashMap<String, List<String>> res = new HashMap<String, List<String>>(varAttrs);
        if (!Objects.isNull(globalAttrs) && !globalAttrs.isEmpty()) {
            ArrayList<String> globalAttrString = new ArrayList<String>();
            for (Attribute globalAttr : globalAttrs) {
                globalAttrString.add(globalAttr.getStringValue());
            }
            res.put("GLOBAL_ATTRIBUTES", globalAttrString);
        }
        for (int band = 0; band < bandAttrs.size(); ++band) {
            int currBandNum = band + 1;
            res.put("BAND_" + currBandNum, bandAttrs.get(band));
        }
        return res;
    }

    private static Double getElement(Array array, int index) {
        double res;
        switch (array.getDataType()) {
            case INT: {
                res = array.getInt(index);
                break;
            }
            case LONG: {
                res = array.getLong(index);
                break;
            }
            case FLOAT: {
                res = array.getFloat(index);
                break;
            }
            case DOUBLE: {
                res = array.getDouble(index);
                break;
            }
            case BYTE: {
                res = array.getByte(index);
                break;
            }
            case SHORT: 
            case USHORT: {
                res = array.getShort(index);
                break;
            }
            default: {
                throw new IllegalArgumentException("Error casting dimension data to double");
            }
        }
        return res;
    }

    private static Double getAttrDoubleValue(Attribute attr) {
        Number numericValue = attr.getNumericValue();
        if (!Objects.isNull(numericValue)) {
            return numericValue.doubleValue();
        }
        throw new IllegalArgumentException("An attribute expected to have numeric values does not have numeric value");
    }

    private static void ensureCoordVar(Variable lonVar, Variable latVar) throws NullPointerException, IllegalArgumentException {
        if (latVar == null) {
            throw new NullPointerException("Provided latitude variable short name is invalid");
        }
        if (lonVar == null) {
            throw new NullPointerException("Provided longitude variable short name is invalid");
        }
        if (latVar.getShape().length > 1) {
            throw new IllegalArgumentException("Shape of latitude variable is supposed to be 1");
        }
        if (lonVar.getShape().length > 1) {
            throw new IllegalArgumentException("Shape of longitude variable is supposed to be 1");
        }
    }

    private static void ensureRecordVar(Variable recordVar) throws NullPointerException {
        if (recordVar == null) {
            throw new IllegalArgumentException("Provided record variable short name is invalid");
        }
    }
}

