#!/bin/csh

# Copyright Intel Corporation
# SPDX-License-Identifier: MIT
# https://opensource.org/licenses/MIT

set command=($_)
if ( "$command" != "" ) then
    set script = `eval "readlink -e $command[2]"`
    set script_dir = `dirname "$script"`
    set product_dir = `cd "$script_dir/.." && pwd -P`
else
    printf 'ERROR: This script should be sourced\n'
    printf 'Usage:\n'
    printf '\tsource %q\n' "$0"
    exit 2
endif

# Export required env vars for compiler package.
setenv CMPLR_ROOT "$product_dir"

set INTEL_TARGET_ARCH="intel64"
set INTEL_TARGET_PLATFORM="linux"
set USE_INTEL_LLVM="0"

while ( $#argv != 0 )
    switch ( $argv[1] )
        case "ia32" :
            set INTEL_TARGET_ARCH="ia32"
            shift
            breaksw
        case "intel64" :
            set INTEL_TARGET_ARCH="intel64"
            shift
            breaksw
        case "--include-intel-llvm" :
            set USE_INTEL_LLVM="1"
            shift
            breaksw
        default :
            shift
            breaksw
    endsw
end

# This environment variable to switch il0 compiler to support ia32 mode.
if ( ${INTEL_TARGET_ARCH} == "ia32" ) then
    setenv INTEL_TARGET_ARCH_IA32 "ia32"
else
    unsetenv INTEL_TARGET_ARCH_IA32
endif

# Compiler driver and other executables are in intel64 (host), include ia32/intel64 targets.
if !($?PATH) then
    setenv PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin/intel64:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin"
else
    setenv PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin/intel64:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin:${PATH}"
endif

if ( ${USE_INTEL_LLVM} == "1" ) then
    setenv PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/bin-llvm:${PATH}"
endif

if ( ${INTEL_TARGET_ARCH} == "ia32" ) then
    if !($?LIBRARY_PATH) then
        setenv LIBRARY_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/ia32_lin:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib"
    else
        setenv LIBRARY_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/ia32_lin:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib:${LIBRARY_PATH}"
    endif
else
    if !($?LIBRARY_PATH) then
        setenv LIBRARY_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib"
    else
        setenv LIBRARY_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib:${LIBRARY_PATH}"
    endif
endif

if !($?LD_LIBRARY_PATH) then
    setenv LD_LIBRARY_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin"
else
    setenv LD_LIBRARY_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin:${LD_LIBRARY_PATH}"
endif

# intel64 libraries also be needed in ia32 target
if ( ${INTEL_TARGET_ARCH} == "ia32" ) then
    setenv LD_LIBRARY_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/ia32_lin:${LD_LIBRARY_PATH}"
endif

# There should be only one OpenCL CPU / FGPA emu runtime is loaded.
# Backup GPU runtime path to variable $gpu_icd_tmp_path and unset OCL_ICD_FILENAMES if GPU runtime is set before this vars.
if ($?OCL_ICD_FILENAMES) then
    set __icd_tmp_arr = ( `echo ${OCL_ICD_FILENAMES} | sed 's/:/\n/g'` )
    foreach __icd_tmp_path (${__icd_tmp_arr})
        if ( ${__icd_tmp_path} =~ *libigdrcl.so* ) then
            set __gpu_icd_tmp_path="${__icd_tmp_path}"
            break
        endif
    end
    unsetenv OCL_ICD_FILENAMES
endif

# OpenCL FPGA runtime
if ( -e "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/oclfpga/fpgavars.csh" ) then
    setenv FPGA_VARS_DIR "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/oclfpga"
    setenv FPGA_VARS_ARGS "${SETVARS_ARGS}"
    source "${FPGA_VARS_DIR}/fpgavars.csh"
endif

# OpenMP offload and CPU LLVM libraries and headers (includes OpenCL CPU run-time)
setenv LD_LIBRARY_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/x64:${LD_LIBRARY_PATH}"

# adding OpenCL CPU ICD filename, OpenCL FPGA part should be prepend here, added in fpgavars.csh
if !($?OCL_ICD_FILENAMES) then
    setenv OCL_ICD_FILENAMES "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/x64/libintelocl.so"
else
    setenv OCL_ICD_FILENAMES "${OCL_ICD_FILENAMES}:${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/lib/x64/libintelocl.so"
endif

# The order in OCL_ICD_FILENAMES should be: 1) FPGA emu; 2) FPGA device (if exists); 3) CPU runtime; 4) GPU runtime.
if ( $?__gpu_icd_tmp_path ) then
    setenv OCL_ICD_FILENAMES "${OCL_ICD_FILENAMES}:${__gpu_icd_tmp_path}"
endif

if !($?MANPATH) then
    setenv MANPATH "${CMPLR_ROOT}/documentation/en/man/common"
else
    setenv MANPATH "${CMPLR_ROOT}/documentation/en/man/common:${MANPATH}"
endif

if !($?PKG_CONFIG_PATH) then
    setenv PKG_CONFIG_PATH "${CMPLR_ROOT}/lib/pkgconfig"
else
    setenv PKG_CONFIG_PATH "${CMPLR_ROOT}/lib/pkgconfig:${PKG_CONFIG_PATH}"
endif

if !($?CMAKE_PREFIX_PATH) then
    setenv CMAKE_PREFIX_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/IntelDPCPP"
else
    setenv CMAKE_PREFIX_PATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/IntelDPCPP:${CMAKE_PREFIX_PATH}"
endif

if !($?NLSPATH) then
    setenv NLSPATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin/locale/%l_%t/%N"
else
    setenv NLSPATH "${CMPLR_ROOT}/${INTEL_TARGET_PLATFORM}/compiler/lib/intel64_lin/locale/%l_%t/%N:${NLSPATH}"
endif
