package SNMP::Insight::Device;

#ABSTRACT: Base class for devices
use Moose;
use namespace::autoclean;

our $VERSION = '0.001';    #TRIAL VERSION:
use SNMP::Insight::Session;
use SNMP::Insight::MIB::Utils qw(sysObjectID2vendor);

has 'session' => (
    isa => 'SNMP::Insight::Session',
    is  => 'ro',
);

has model => (
    is  => 'ro',
    isa => 'Str'
);

has os => (
    is  => 'ro',
    isa => 'Str'
);

has os_ver => (
    is  => 'ro',
    isa => 'Str'
);

has vendor => (
    is      => 'ro',
    isa     => 'Str',
    default => sub {
        my $self = shift;
        return sysObjectID2vendor( $self->sysObjectID ) || "";
    }
);

sub get_all_mib_roles {
    my $self = shift;

    my @roles = grep { $_->can('mib_name') }
      $self->meta->calculate_all_roles_with_inheritance;

    return @roles;
}

with 'SNMP::Insight::MIB::SNMPv2';

__PACKAGE__->meta->make_immutable;
1;

# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

SNMP::Insight::Device - Base class for devices

=head1 VERSION

version 0.001

=head1 ATTRIBUTES

=head2 session

An L<SNMP::Insight::Session> instance used to retrieve SNMP info.

=head2 model

Guessed device model. May be overridden by device roles.

=head2 os

Guessed device operating system. May be overridden by device roles.

=head2 os_ver

Guessed device operating system version. May be overridden by device roles.

=head2 os_ver

Guessed device vendor. May be overridden by device roles.

=head1 METHODS

=head2 get_all_mib_roles

Return all the MIB roles for this device.

=head1 AUTHOR

Gabriele Mambrini <g.mambrini@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
