unit module Transit::Network:ver<0.0.2>;

use Transit::Network::Helpers;
use Transit::Network::i8n;

my %stop-id2name;
my %stop-name2id;

my $basepath = "";

my @routes;
my %down;
my %up;
my %interval;
my %file;
my %iv;
my %start-at;
my %end-at;
my %rows;

our %commands = (route        => &exec-route,
                 summary      => &exec-summary,
		 timingpoints => &exec-timingpoints,
		 tp           => &exec-timingpoint-single,
		 vehicles     => &exec-vehicles,
		 stops        => &exec-stops,
		 batch        => &exec-batch,
		 path         => &exec-path,
		 lang         => &exec-language,
		 help         => &exec-help);
		 
our @commands = %commands.keys;

our sub set-basepath ($path)
{
  $basepath = $path ~ "/";
}

class stop-enroute
{
  has $.stop-id;
  has $.platform;
  has $.stop-name;
  has $.stop-number;
  has $.travel-time;
}

class Route
{
  has $.route-id;
  has $.interval;
  has $.up;
  has $.up-actual   is rw;
  has $.down;
  has $.down-actual is rw;
  has @.source;
  has @.stops-down;        
  has %.stops-down-lookup; 
  has @.stops-up;          
  has %.stops-up-lookup;   

  method init
  {
    $.down-actual = $.down;
    $.up-actual   = $.up;

    @.stops-down        = ();
    @.stops-up          = ();
    %.stops-down-lookup = ();
    %.stops-up-lookup   = ();
    
    self._init-down;

    if ($.down ~~ /^(.*)\[(\w\w\d\d\d)\]$/)
    {
      my $down    = $0.Numeric;
      my $time-at = $1.Str;

      if $.stops-down-lookup{$time-at}
      {
        $down -= $.stops-down-lookup{$time-at}.travel-time;
	my ($interval) = $.interval.split(";");
        $down += $interval while $down < 0;
        $.down-actual = $down;
        self._init-down;
      }
      else
      {
        die "$time-at: Not a stop on this route";
      }
    }

    self._init-up;

    if ($.up ~~ /^(.*)\[(\w\w\d\d\d)\]$/)
    {
      my $up      = $0.Numeric;
      my $time-at = $1.Str;

      if $.stops-up-lookup{$time-at}
      {
        $up -= $.stops-up-lookup{$time-at}.travel-time;
	my ($interval) = $.interval.split(";");
        $up += $interval while $up < 0;
        $.up-actual = $up;
        self._init-up;
      }
      else
      {
        die "$time-at: Not a stop on this route";
      }
    }
  }

  method _init-down
  {
    my $count           = 1;
    my $travel-time     = 0;
    @.stops-down        = ();
    %.stops-down-lookup = ();

    return if $.down eq "off";
    
    for @.source -> $row
    {
      next unless $row;
      next if $row.substr(0,1) eq "#";
      my ($id, $down, $up, $name) = $row.split(/\s+/, 4);
      next if $down eq"-";

      my ($platform, $min) = $down.split(":");

      $min = dehourify($min) if $min;

      if $count > 1
      {
        $min.defined ?? ( $travel-time += $min ) !! $travel-time++;
      } 

      my $s = stop-enroute.new(stop-id     => $id,
                               platform    => $platform,
                               stop-name   => $name,
			       stop-number => $count,
			       travel-time => $travel-time
			      );
      $count++;

      @.stops-down.push: $s;
      %.stops-down-lookup{$id} = $s unless %.stops-up-lookup{$id};
      
      die "Redeclaration of stop $id (%stop-id2name{$id} -> $name)"
        if %stop-id2name{$id} && %stop-id2name{$id} ne $name;

      %stop-id2name{$id}           = $name; 
      %stop-name2id{$name} = $id;  # Can overwrite !!
    }
  }

  method _init-up
  {
    my $count           = 1;
    my $travel-time     = 0;
    @.stops-up          = ();
    %.stops-up-lookup   = ();

    return if $.up eq "off";

    for @.source.reverse -> $row
    {
      next unless $row;
      next if $row.substr(0,1) eq "#";
      my ($id, $down, $up, $name) = $row.split(/\s+/, 4);
      next if $up eq"-";

      my ($platform, $min) = $up.split(":");

      $min = dehourify($min) if $min;

      if $count > 1
      {
        $min.defined ?? ( $travel-time += $min ) !! $travel-time++;
      } 

      my $s = stop-enroute.new(stop-id     => $id,
                               platform    => $platform,
                               stop-name   => $name,
			       stop-number => $count,
			       travel-time => $travel-time
			      );
      $count++;

      @.stops-up.push: $s;
      %.stops-up-lookup{$id} = $s unless %.stops-up-lookup{$id};

      die "Redeclaration of stop $id (%stop-id2name{$id} -> $name)"
        if %stop-id2name{$id} && %stop-id2name{$id} ne $name;

      %stop-id2name{$id} = $name;
      %stop-name2id{$name} = $id;  # Can overwrite !!
    }
  }
  
  method summary
  {
    my @summary;

    return "$.route-id: Summary " ~ Transit::Network::i8n::translate("not possible")
      if any($.interval.split(";")) >= 60;

    if @.stops-down
    {
      my $show-hours = @.stops-down[* -1].travel-time >= 60;
      
      @summary.push: "Route $.route-id: { @.stops-down[0].stop-name } - { @.stops-down[* -1].stop-name }";
      @summary.push: "----------------------------------------------------------------";

      my $down = $.down-actual;

      for @.stops-down -> $stop
      {
        @summary.push: "{ print-minutes($stop.travel-time, :$show-hours) } | { print-time($down + $stop.travel-time, $.interval) } { $stop.stop-name } [{ $stop.platform }]";
      }
      @summary.push: "";
    }

    if @.stops-up
    {
      my $show-hours = @.stops-up[* -1].travel-time >= 60;

      @summary.push: "Route $.route-id: { @.stops-up[0].stop-name } - { @.stops-up[* -1].stop-name }";
      @summary.push: "----------------------------------------------------------------"; 

      my $up = $.up-actual;

      for @.stops-up -> $stop
      {
        @summary.push: "{ print-minutes($stop.travel-time, :$show-hours) } | { print-time($up + $stop.travel-time, $.interval) } { $stop.stop-name } [{ $stop.platform }]";
      }
      @summary.push: "";
    }
      
    return @summary;
  }

  
  method summary-raw
  {
    my @summary;

    if @.stops-down
    {
      my $show-hours = @.stops-down[* -1].travel-time >=   60;
      my $show-days  = @.stops-down[* -1].travel-time >= 1440;

      @summary.push: "Route $.route-id: { @.stops-down[0].stop-name } - { @.stops-down[* -1].stop-name }";
      @summary.push: "----------------------------------------------------------------";

      my $down = $.down-actual;

      for @.stops-down -> $stop
      {
        @summary.push: "{ print-minutes($stop.travel-time + $.down, :$show-hours, :$show-days) } | { $stop.stop-name } [{ $stop.platform }]";
      }
      @summary.push: "";
    }

    if @.stops-up
    {
      my $show-hours = @.stops-up[* -1].travel-time >= 60;
      my $show-days  = @.stops-up[* -1].travel-time >= 1440;

      @summary.push: "Route $.route-id: { @.stops-up[0].stop-name } - { @.stops-up[* -1].stop-name }";
      @summary.push: "----------------------------------------------------------------"; 

      my $up = $.up-actual;

      for @.stops-up -> $stop
      {
        @summary.push: "{ print-minutes($stop.travel-time + $.up, :$show-hours, :$show-days) } | { $stop.stop-name } [{ $stop.platform }]";
      }
      @summary.push: "";
    }
      
    return @summary;
  }
}

my sub print-minutes ($min is copy, :$show-hours, :$show-days)
{
  if ($show-hours)
  {
    my $half-minute = $min ~~ /\.5$/;
    my $hour        = 0;

    while $min >= 60 { $min -= 60; $hour++; } 

    unless $show-days
    {
      return "{ $hour ?? "{ $hour.fmt('%2d') }:" !! '  ' }{ $min.fmt('%02d\'') }" if $half-minute;
      return "{ $hour ?? "{ $hour.fmt('%2d') }:" !! '  ' }{ $min.fmt('%02d') } ";
    }

    my $days = 0;
    while $hour >= 24 { $hour -= 24; $days++; }

    return "{ $days ?? "+$days" !! '  ' } { $hour ?? "{ $hour.fmt('%2d') }" !! ' 0' }:{ $min.fmt('%02d\'') }" if $half-minute;
    return "{ $days ?? "+$days" !! '  ' } { $hour ?? "{ $hour.fmt('%2d') }" !! ' 0' }:{ $min.fmt('%02d') } ";

  }
  return $min ~~ /\.5$/ ?? $min.fmt('%02d\'') !! $min.fmt('%02d') ~ ' '; 
}

my sub print-time ($min is copy, $interval = 0) is export
{
  $min -= 60 while $min >= 60;

  unless $interval
  {
    return $min ~~ /\.5$/ ?? $min.fmt('%02d\'') !! $min.fmt('%02d') ~ ' '; 
  }

  my @interval = $interval.split(";");
  my $index    = 0;
  my @times    = $min ~~ /\.5$/ ?? $min.fmt('%02d\'') !! $min.fmt('%02d') ~ ' ';
  my $curr     = $min + @interval[$index++ % @interval.elems];
  
  $curr -= 60 if $curr >= 60;

  while $curr != $min
  {
    @times.push: $curr ~~ /\.5$/ ?? $curr.fmt('%02d\'') !! $curr.fmt('%02d') ~ ' ';

    $curr += @interval[$index++ % @interval.elems];
    $curr -= 60 if $curr >= 60;
  }

  return @times.join(' ');
}

sub get-stops is export
{
  my @content;

  for %stop-id2name.keys.sort -> $stop-id
  {
    @content.push: "$stop-id %stop-id2name{$stop-id}";
  }
  return @content;
}

sub is-stop ($stop-id) is export
{
  return %stop-id2name{$stop-id}.defined;
}

sub ensure-stop ($stop) is export
{
  return $stop                if %stop-id2name{$stop}.defined;
  return %stop-name2id{$stop} if %stop-name2id{$stop}.defined;
  return;
}

sub get-file ($filename) is export
{
  my $basepath = $filename.IO.dirname;

  set-basepath($basepath) if $basepath;

  return $filename.IO.lines;
}

sub exec-language (@args)
{
  my $lang = @args[0];

  $lang ?? Transit::Network::i8n::set-lang($lang)
        !! say Transit::Network::i8n::get-lang;
}

sub exec-help (@*args = ())
{
  say "See https://raku-musings.com/transit-network/ for a description of this program.";
  say "Sample networks are bundled with this program. Unpack the following zip file in a";
  say " suitable location: { %?RESOURCES<examples-0.0.1.zip> }";
}

sub exec-path (@args)
{
  my $path = @args[0] // return;

  $path.IO.d && set-basepath($path);
}

sub exec-batch (@args is copy)
{
  my $silent = False;
  my $file   = @args.shift // "";

  if $file eq "-silent"
  {
    $silent = True;
    $file   = @args.shift;
  }

  unless $file.IO.e && $file.IO.r
  {
    say Transit::Network::i8n::translate("Unable to read the file") ~ ":$file";
    return;
  }

  my $basepath = $file.IO.dirname;

  set-basepath($basepath) if $basepath;

  my @rows   = $file.IO.lines;
  my $first  = @rows.shift;

  return unless check-first-line-setup($first);
  
  for @rows -> $row
  {
    next unless $row;
  
    my @items   = $row.words;
    my $command = @items.shift;

    given $command
    {
      # when "batch" { warn "Cannot execute batch command inside a batch command"; }

      when $silent && $command ne 'route' { ; }

      when $command eq any(@commands) { &(%commands{$command})(@items); }
      default                  { say "{ Transit::Network::i8n::translate("Unknown command") }: \"$_\" (Legal commands: { @commands })" }
    }
  }
}

sub exec-route (@args is copy)
{
  my $command = @args.shift // "";
  
  if $command eq "list"
  {
    for @routes -> $route
    {
      say "{ Transit::Network::i8n::translate("Route")} $route: { (@(%iv{$route}.stops-down)[0]).stop-name } - { (@(%iv{$route}.stops-up)[0]).stop-name }";
    }
    return;
  }

  if $command eq "delete"
  {
    my $id = @args.shift;
    
    if %iv{$id}
    {
      %iv{$id}:delete;
      @routes = @routes.grep({ $_ ne $id });
    }
  }
  
  elsif $command eq "set"
  {
    my $id = @args.shift;
    
    @routes.push: $id unless $id eq any(@routes);

    my ($file, $start-at, $end-at, $reverse);

    %interval{$id} =  %down{$id} =  %up{$id} = Any;
 
    for @args -> $item
    {
      my ($key, $val) = $item.split(/<[=\:]>/, 2);
      
      given $key
      {
        when 'up'       { %up{$id}       = dehourify($val.Str); }
        when 'down'     { %down{$id}     = dehourify($val.Str); }
        when 'file'     { $file          = $basepath ~ $val.Str; }
        when 'interval' { %interval{$id} = dehourify-interval($val.Str); }
        when 'start-at' { $start-at      = $val.Str; }
        when 'end-at'   { $end-at        = $val.Str; }
        when 'reverse'  { $reverse       = True; }
	default         { say Transit::Network::i8n::translate("Unknown command") ~ ": $_"; }
      }
    }

    unless $file ~~ /\.def$/
    {
      say Transit::Network::i8n::translate("File does not have the mandatory extension") ~ " '.def'";
      next;
    }
    
    my @rows = read-start-stop(:$file, :$start-at, :$end-at, :$reverse);

    next unless @rows.elems;

    %iv{$id} = Transit::Network::Route.new(
          route-id => $id,
          interval => %interval{$id} || 10,
          down     => %down{$id} || 0,
          up       => %up{$id} || 0,
          source   => @rows,
    );
  
    %iv{$id}.init;
  }
  
  elsif $command eq "append"
  {
    my $id = @args.shift;
    unless ($id eq any(@routes))
    {
      say Transit::Network::i8n::translate("Unknown route") ~ " $id." ~
          Transit::Network::i8n::translate("Cannot append");
      return;
    }

    my ($file, $start-at, $end-at, $reverse);

    for @args -> $item
    {
      my ($key, $val) = $item.split(/<[=\:]>/, 2);
      
      given $key
      {
        when 'up'       { %up{$id}       = dehourify($val.Str); }
        when 'down'     { %down{$id}     = dehourify($val.Str); }
        when 'file'     { $file          = $basepath ~ $val.Str; }
        when 'interval' { %interval{$id} = dehourify-interval($val.Str); }
        when 'start-at' { $start-at      = $val.Str; }
        when 'end-at'   { $end-at        = $val.Str; }
        when 'reverse'  { $reverse       = True; }
	default         { say Transit::Network::i8n::translate("Unknown command") ~ ": $_"; }
      }
    }

    unless $file ~~ /\.def$/
    {
      say Transit::Network::i8n::translate("File does not have the mandatory extension") ~ " '.def'";
      next;
    }

    my @rows = read-start-stop(:$file, :$start-at, :$end-at, :$reverse);

    next unless @rows.elems;
   
    %iv{$id}.source.append: @rows;
    %iv{$id}.init;
  }
  
  elsif $command eq "source"
  {
    my $id      = @args.shift;
    my $reverse = False;
    
    if $id eq ":reverse"
    {
      $id = @args.shift;
      $reverse = True;
    }

    if ( %iv{$id} )
    {
      say $reverse
        ?? nicify-source($id, reverse-route(%iv{$id}.source)).join("\n")
        !! nicify-source($id, %iv{$id}.source).join("\n");

    }
    else
    {
      say "{ Transit::Network::i8n::translate("The specified route does not exist") }: $id";
    }
  }

  else
  {
    say "{ Transit::Network::i8n::translate("Unknown command") } 'route $command'";
    return;
  }
}

sub exec-summary (@args)
{
  my $raw = False;
  
  while @args && @args[0].substr(0,1) eq ":"
  {
    my $colon = @args.shift;
    if $colon eq ":raw"
    {
      $raw = True;
    }
  }

  my $out-file = @args.shift // "-";

  my @summary;
  
  for @routes -> $route
  {
    $raw ?? @summary.append: %iv{$route}.summary-raw
         !! @summary.append: %iv{$route}.summary;
  }

  print-content(@summary, $out-file);
}

sub exec-timingpoints (@args)
{
  my $tabulated = False;
  my $merge     = False;
  
  while @args[0].substr(0,1) eq ":"
  {
    my $colon = @args.shift;
    if $colon eq ":tabulated"
    {
      $tabulated = True;
    }
    elsif $colon ~~ /^\:merge\[(.*)\]$/
    {
      $merge     = $0.Str;
      $tabulated = True;
    }
  }
  
  my $in-file  = $basepath ~ @args.shift;
  my $out-file = @args.shift // "-";

  unless $in-file ~~ /\.def$/
  {
     say "{ Transit::Network::i8n::translate("File does not have the mandatory extension") } '.def': \"$in-file\"";
     return;
  }  

  my @stops = $in-file.IO.lines;

  print-content(do-timingpoints(@stops, :$tabulated, :$merge), $out-file);
}

##!! Merge the argument parsing

sub exec-timingpoint-single (@args)
{
  my $tabulated = False;
  my $merge     = False;
  
  while @args[0].substr(0,1) eq ":"
  {
    my $colon = @args.shift;
    if $colon eq ":tabulated"
    {
      $tabulated = True;
    }
    elsif $colon ~~ /^\:merge\[(.*)\]$/
    {
      $merge     = $0.Str;
      $tabulated = True;
    }
  }
  
  print-content(do-timingpoints(@args, :$tabulated, :$merge), "-");
}

sub exec-vehicles (@args)
{
  my $days = False;

  if @args[0] eq ":days"
  {
    $days = True;
    @args.shift;
  }

  my $in-file  = $basepath ~ @args.shift;
  my $out-file = @args.shift // "-";

  unless $in-file ~~ /\.def$/
  { 
    say "{ Transit::Network::i8n::translate("File does not have the mandatory extension") } '.def': \"$in-file\"";
    return;
  }

  print-content(do-number-of-vehicles($in-file, :$days), $out-file);
}

sub exec-stops (@args)
{
  my $out-file = @args.shift // "-";

  print-content(get-stops, $out-file);
}

my sub print-content(@content, $out-file)
{
  given $out-file
  {
    when "-" 
    { 
      say @content.join("\n") ~ "\n";
    }
    when /\.txt$/
    {
      spurt $out-file, @content.join("\n") ~ "\n";
    }
    when " "
    {
      ;
    }
    default
    {
      die "{ Transit::Network::i8n::translate("File does not have the mandatory extension") }: '.txt': \"$out-file\"";
    }
  }
}

sub do-timingpoints (@stops, :$tabulated, :$merge)
{
  my %values;
  my @merge = $merge.defined ?? $merge.split(';') !! Any;

  for @stops -> $stop-candidate
  {
    my $stop = ensure-stop($stop-candidate) // next;
    
    for @routes -> $route
    {
      my $r = %iv{$route};

      if any($r.interval.split(";")) >= 60
      {
        say "$r.route-id: Timingpoint " ~ Transit::Network::i8n::translate("not possible");
	next;
      }
      
      if $r.stops-down-lookup{$stop}
      {
        for $r.stops-down -> $s
        {
          next unless $s.stop-id eq $stop;

          my $stop-platform = "{ $s.stop-name } [{ $s.platform }]";
	
          %values{$stop-platform}.push:
	    " { _fix-dep(print-time($r.down-actual + $s.travel-time, $r.interval)) } | { translate-route($r.route-id) } {$r.stops-down[* -1].stop-name  }";
        }
      }
      
      if $r.stops-up-lookup{$stop}
      {
        for $r.stops-up -> $s
        {
          next unless $s.stop-id eq $stop;

          my $stop-platform = "{ $s.stop-name } [{ $s.platform }]";
	
          %values{$stop-platform}.push:
	    " { _fix-dep(print-time($r.up-actual + $s.travel-time, $r.interval)) } | { translate-route($r.route-id) } {$r.stops-up[* -1].stop-name  }";
        }
      }
    }
  }

  my @return;
  for %values.keys.sort -> $key
  {
    @return.push: $key;
    @return.append: $tabulated ?? _tabulate(%values{$key}, :$merge) !! |%values{$key}.sort;
    @return.push: "";
  }

  return @return;

  sub translate-route ($route-id)
  {
    return $route-id unless @merge;
    return @merge[0] if $route-id eq any(@merge);
    return $route-id;
  }

  sub _fix-dep ($dep-string)
  {
    return $dep-string.words.sort.fmt('%-3s').join(" ");
  }
  
  sub _tabulate (@rows, :$merge)
  {
    my %all-minutes;
    my @result;
  
    for @rows -> $row
    {
      my ($minutes) = $row.split("|");
      my @minutes   = $minutes.words;
      @minutes.map({ %all-minutes{$_} = True; });
    }

    if $merge
    {
      my %dept;
      for @rows -> $row
      {
        my ($minutes, $destination) = $row.split("|");
	%dept{$destination}.append: $minutes.words;
      }
      for %dept.keys.sort -> $destination
      {
        my $string  = "";
        my %minutes = %dept{$destination}.Set;
        for %all-minutes.keys.sort -> $minute
        {
          $string ~= ( %minutes{$minute} ?? $minute !! ($minute.chars == 3 ?? '-- ' !! '--') ) ~ ' ';
        }
        $string ~= "|$destination";
        @result.push: $string;
      }
      
      return @result;
    }
  
    for @rows -> $row
    {
      my $string = "";
      my ($minutes, $destination) = $row.split("|");
      my %minutes = $minutes.words.Set;
      for %all-minutes.keys.sort -> $minute
      {
        $string ~= ( %minutes{$minute} ?? $minute !! ($minute.chars == 3 ?? '-- ' !! '--') ) ~ ' ';
      }
      $string ~= "|$destination";
      @result.push: $string;
    }
  
    return @result;
  }
}

our sub do-number-of-vehicles ($def-file, :$days)
{
  unless ($def-file.IO.f && $def-file.IO.r)
  {
    say Transit::Network::i8n::translate("Unable to read the file") ~ ": $def-file";
    return;
  }
  
  my @def-lines = $def-file.IO.lines;

  my %dep;
  my %arrival;
  my %dest;
  my %time;
  my %pause;
  my %dept-first;
  my $total-vehicles = 0;
  my %total-vehicles;

  my @rows;

  my ($route-id, $destinasjon, @deps);

  for @routes -> $route-id
  {
     %time{$route-id}{'D'}       = %iv{$route-id}.stops-down ?? %iv{$route-id}.stops-down[* -1].travel-time !! 0;
     %time{$route-id}{'U'}       = %iv{$route-id}.stops-up   ?? %iv{$route-id}.stops-up[* -1].travel-time   !! 0;
     
     %dept-first{$route-id}{'D'} = %iv{$route-id}.down-actual;
     %dept-first{$route-id}{'U'} = %iv{$route-id}.up-actual;
  }

  my @output;
  my $series;
  
  for @def-lines -> $row
  {
    next if $row.substr(0,1) eq '#';
    next unless $row;
    
    if $row ~~ /^\= (\w+\d\d)$/
    {
      my $new-series = $0.Str;
      do-print;
      $series        = $new-series;
      next;
    }

    my ($route-id, $dir, $to, $pause) = $row.split(/\s+/);

    my $arrival = $to;

    ($to, $arrival) = $to.split(';') if $to.contains(';');

    @rows.push: "$route-id $dir";
     
    %pause{$route-id}{$dir}   = dehourify($pause);
    %arrival{$route-id}{$dir} = $arrival;
    %dest{$route-id}{$dir}    = $to;
  }

  do-print;

  sub do-print
  {
    return unless @rows.elems;

    @rows.push: @rows[0];

    my $counter = 0;
    my $add-time;
    my @depts;

    my $dept;
    my $first;
    my @interval;

    my (@a, @b, @c); # , @d);

    my $indentation = 0;

    for @rows -> $row
    {
      my ($route-id, $dir) = $row.split(/\s+/);
    
      if $counter
      {
        $dept += $add-time;

        @a.push: %arrival{$route-id}{$dir};
        @b.push: 'arr';
        @c.push: $dept;
        # @d.push: %iv{$route-id}.interval;

        $indentation = max($indentation, %arrival{$route-id}{$dir}.chars, %dest{$route-id}{$dir}.chars);
      }

      @interval = %iv{$route-id}.interval.split(";");

      if $counter
      {
        my $new-dept = %dept-first{$route-id}{$dir};
        $dept += %pause{$route-id}{$dir};

        my $index = 0;

        while $new-dept < $dept { $new-dept += @interval[$index++ % @interval.elems]; }

        $dept = $new-dept;
      }
      else
      {
        $first = $dept = %dept-first{$route-id}{$dir};
      }
      
      @a.push: %dest{$route-id}{$dir};
      @b.push: 'dep';
      @c.push: $dept;
      # @d.push: %iv{$route-id}.interval;
      $counter++;
      $add-time = %time{$route-id}{$dir};
    }

    @output.push: "";

    my $one-round = $dept - $first;

    my $vehicles = ($one-round / ( @interval.sum / @interval.elems )).Int;
    $total-vehicles += $vehicles;
    %total-vehicles{$series} = $vehicles;
    @rows = ();

    my $l  = " " x ($indentation + 5);
    my $id = $series;
    for 1 .. $vehicles -> $vhcl
    {
      $id.=succ;
      $id    = 0 ~ $id if $id.chars == 3;
      $l ~= '  ' if $days;
      $l ~= "  $id";
    }

    @output.push: $l;
    @output.push: "-" x $indentation + 5 + (6 * $vehicles) + ($days ?? 2 * $vehicles !! 0);

    my $dept-rows = 0;

    for ^@a.elems -> $index
    {
      my $mins     = @c[$index];
      # my @interval = @d[$index].split(";");

# say "!!! @a[$index] || @b[$index] || { nice-time(@c[$index], :days) } || @d[$index] !!!";

      my $l = "@a[$index]" ~ " " x ($indentation - @a[$index].chars + 1)  ~ "@b[$index]  ";
      my $i = 0;
      for ^$vehicles -> $col
      {
        $l ~= " { nice-time($mins, :$days) }";

# say "!! { ($dept-rows * $vehicles + $i) % @interval.elems } -> { nice-time(@interval[($dept-rows * $vehicles + $i) % @interval.elems], :days) }";

        $mins += @interval[$i++ % @interval.elems];
        # $mins += @interval[($dept-rows * $vehicles + $i++) % @interval.elems];
      }
      @output.push: $l;

      $dept-rows++ if @b[$index] eq "dep";
    }
  }

  @output.push: "";

  for %total-vehicles.keys.sort -> $series
  {
    @output.push: "$series: { Transit::Network::i8n::translate("Number of vehicles") }: %total-vehicles{$series}";
  }

  @output.push: "---------------------------------------";
  @output.push: "{ Transit::Network::i8n::translate("Total number of vehicles") }: $total-vehicles";

  return @output;
}

our sub read-start-stop (:$file, :$start-at, :$end-at, :$reverse)
{
  unless $file.IO.f && $file.IO.r
  {
    say Transit::Network::i8n::translate("Unable to read the file") ~ ": $file";
    return ();
  }

  my @rows = $file.IO.lines;
  my @save;
  
  @rows = reverse-route(@rows) if $reverse;

  my $start-at-id = $start-at;
  my $end-at-id   = $end-at;

  my $start-at-platform;
  my $end-at-platform;

  if $start-at && $start-at ~~ /^(.*)\[(.+)\]$/
  {
    $start-at-id       = $0.Str;
    $start-at-platform = $1.Str;
  }

  if $end-at && $end-at ~~ /^(.*)\[(.+)\]$/
  {
    $end-at-id       = $0.Str;
    $end-at-platform = $1.Str;
  }

  my $from;
  my $to;

  for @rows -> $row
  {
    next unless $row;
    next if $row.substr(0,1) eq "#";
      
    my ($id, $down, $up, $name) = $row.split(/\s+/, 4);
    next if $start-at-id && $start-at-id ne $id && !$from;

    unless $from
    {
      if $start-at-platform
      {
        my ($a, $b) = $start-at-platform.split(";");
	$down = $a;
	$up   = $b;
      }
    }

    if $end-at-id && $end-at-id eq $id
    {
      if $end-at-platform
      {
        my ($a, $b) = $end-at-platform.split(";");
	$down = $a;
	$up   = $b;
      }
    }

    $from = $name unless $from;
    $to   = $name;
    
    @save.push: "$id $down $up $name";

    last if $end-at-id && $end-at-id eq $id;
  }

  return @save;
}

our sub reverse-route(@rows)
{
  my @new;

  for @rows.reverse -> $row
  {
    next unless $row;
    next if $row.substr(0,1) eq "#";

    my ($id, $down, $up, $name) = $row.split(/\s+/, 4);

    @new.push: "$id $up $down $name";
  }

  return @new;
}

our sub nicify-source($route_id, @source)
{
  my $max-down = 1;
  my $max-up   = 1;
  my $first-stop;
  my $last-stop;
  my @new;

  my $down-off = %iv{$route_id}.down eq "off";
  my $up-off   = %iv{$route_id}.up   eq "off";

  for @source -> $row
  {
    next unless $row;
    next if $row.substr(0,1) eq "#";

    my ($id, $down, $up, $name) = $row.split(/\s+/, 4);

    $first-stop = $name;
    $last-stop  = $name unless $last-stop;

    $max-down = $down-off ?? 1 !! max($max-down, $down.chars);
    $max-up   = $up-off   ?? 1 !! max($max-up,   $up.chars);
  }

  @new.push: "#";
  @new.push: "# $route_id $last-stop - $first-stop";
  @new.push: "# ", "-" x 30;

  for @source -> $row
  {
    next unless $row;
    next if $row.substr(0,1) eq "#";

    my ($id, $down, $up, $name) = $row.split(/\s+/, 4);

    @new.push: "$id { $down-off ?? '-' !! $down }" ~ " " x ($max-down - $down.chars) 
                ~ " { $up-off ?? '-' !! $up }" ~ " " x ($max-up - $up.chars) ~ " " ~ $name;
  }

  return @new;
}

our sub check-first-line-setup ($line) is export
{
  return True if $line eq '## Networkplanner ##';

  say "{ Transit::Network::i8n::translate("Wrong first row. Should be") } '## Networkplanner ##'";
  return False;
}

=begin pod

=head1 NAME

Transit::Network - Plan and set up public transportation routes or networks

=begin code

$ networkplanner --help

=end code

=begin code

$ networkplanner
networkplanner: Enter «exit» to exit
> help

=end code

=head1 SYNOPSIS

=begin code

use Transit::Network;

=end code

=head1 DESCRIPTION

Transit::Network (and the bundled programs) enables you to plan and set up public transportation
routes or complete networks.

See https://raku-musings.com/transit-network/ for a description of this module.

The module comes with some sample networks. Use the «--help» command line option, or the «help» 
command, to get the location of the zip file.

=head1 AUTHOR

Arne Sommer <arne.sommer@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright 2021 Arne Sommer

This library is free software; you can redistribute it and/or modify it under the Artistic License 2.0.

=end pod
