package Icon::FamFamFam::Silk;

use 5.006;
use MIME::Base64 qw[];
use URI;

our $VERSION = '0.002001003';
our $_data   = {};

sub new
{
	my ($class, $icon) = @_;
	bless \$icon, $class;
}

sub _load_data()
{
	return if %$_data;
	local $/;
	eval <DATA>;
}

sub name
{
	my ($icon) = @_;
	return (ref $icon) ? ${$icon} : $icon;
}

sub list
{
	my ($class) = @_;
	_load_data() unless %$_data;
	return map { my $icon = $_; bless \$icon, $class; } keys %$_data;
}

sub base64
{
	my ($icon) = @_;
	_load_data() unless %$_data;
	return (ref $icon) ? $_data->{${$icon}} : $_data->{$icon};
}

sub uri
{
	my ($icon) = @_;
	return URI->new('data:image/png;base64,' . base64($icon));
}

sub raw
{
	my ($icon) = @_;
	MIME::Base64::decode_base64(base64($icon));
}

sub attribution_data
{
	my %data = (
		-url             => 'http://www.famfamfam.com/lab/icons/silk/famfamfam_silk_icons_v013.zip',
		-title           => 'Silk icons',
		-attributionName => 'Mark James',
		-attributionURL  => 'http://www.famfamfam.com/lab/icons/silk/',
		-licence         => 'by',
		-type            => 'Collection',
		);
	return %data;
}

1;

=head1 NAME

Icon::FamFamFam::Silk - embed these famous icons in your code

=head1 VERSION

0.002001003

(Version 0.002 of this module and version 1.3 of the icons.)

=head1 SYNOPSIS

 use Icon::FamFamFam::Silk;
 my $wrench = Icon::FamFamFam::Silk->new('wrench_orange');
 print $wrench->base64;

or

 use Icon::FamFamFam::Silk;
 print Icon::FamFamFam::Silk::base64('wrench_orange');

=head1 DESCRIPTION

These methods can be called as functions taking the name of the icon (a string)
as their argument, or can be called as object methods on an instantiated icon
object. (See L</SYNOPSIS>.)

=over

=item C<< Icon::FamFamFam::Silk->new($name) >>

Returns a blessed icon object.

=item C<< Icon::FamFamFam::Silk->list >>

Returns a list of all the icons.

=item C<< $icon->name >>

Returns the name of the icon.

=item C<< $icon->base64 >>

Returns base64-encoded PNG data.

=item C<< $icon->raw >>

Returns raw PNG data.

=item C<< $icon->uri >>

Returns a "data:" URI for the icon.

=item C<< $icon->attribution_data >>
=item C<< Icon::FamFamFam::Silk->attribution_data >>

Returns a hash of data useful for giving credit to Mark James.

This is in a format suitable for passing to L<HTML::HTML5::Builder>'s
C<CREATIVE_COMMONS> function, should you want to do something like
that.

=back

=head1 BUGS

Please report any bugs to L<http://rt.cpan.org/>.

=head1 SEE ALSO

L<http://famfamfam.com/lab/icons/silk/>.

=head1 AUTHOR

Icons by Mark James of L<http://famfamfam.com/>.

Module by Toby Inkster E<lt>tobyink@cpan.orgE<gt>.

=head1 COPYRIGHT AND LICENCE

Mark James distributes the icons themselves under the CC-BY 3.0 licence
L<http://creativecommons.org/licenses/by/3.0/>. Applications using
Icon::FamFamFam::Silk (even indirectly via another module) must provide
credit. Including a link to L<http://famfamfam.com/lab/icons/silk/> in the
documentation of the application is probably sufficient.

Toby Inkster releases the additional Perl coding into the Public Domain.

=head1 DISCLAIMER OF WARRANTIES

THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.

=cut

__DATA__

$_data = {
accept=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKfSURBVDjLpZPrS1NhHMf9O3bOdmwDCWREIYKEUHsVJBI7mg3FvCxL09290jZj2EyLMnJexkgpLbPUanNOberU5taUMnHZUULMvelCtWF0sW/n7MVMEiN64AsPD8/n83uucQDi/id/DBT4Dolypw/qsz0pTMbj/WHpiDgsdSUyUmeiPt2+V7SrIM+bSss8ySGdR4abQQv6lrui6VxsRonrGCS9VEjSQ9E7CtiqdOZ4UuTqnBHO1X7YXl6Daa4yGq7vWO1D40wVDtj4kWQbn94myPGkCDPdSesczE2sCZShwl8CzcwZ6NiUs6n2nYX99T1cnKqA2EKui6+TwphA5k4yqMayopU5mANV3lNQTBdCMVUA9VQh3GuDMHiVcLCS3J4jSLhCGmKCjBEx0xlshjXYhApfMZRP5CyYD+UkG08+xt+4wLVQZA1tzxthm2tEfD3JxARH7QkbD1ZuozaggdZbxK5kAIsf5qGaKMTY2lAU/rH5HW3PLsEwUYy+YCcERmIjJpDcpzb6l7th9KtQ69fi09ePUej9l7cx2DJbD7UrG3r3afQHOyCo+V3QQzE35pvQvnAZukk5zL5qRL59jsKbPzdheXoBZc4saFhBS6AO7V4zqCpiawuptwQG+UAa7Ct3UT0hh9p9EnXT5Vh6t4C22QaUDh6HwnECOmcO7K+6kW49DKqS2DrEZCtfuI+9GrNHg4fMHVSO5kE7nAPVkAxKBxcOzsajpS4Yh4ohUPPWKTUh3PaQEptIOr6BiJjcZXCwktaAGfrRIpwblqOV3YKdhfXOIvBLeREWpnd8ynsaSJoyESFphwTtfjN6X1jRO2+FxWtCWksqBApeiFIR9K6fiTpPiigDoadqCEag5YUFKl6Yrciw0VOlhOivv/Ff8wtn0KzlebrUYwAAAABJRU5ErkJggg==',
add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJvSURBVDjLpZPrS5NhGIf9W7YvBYOkhlkoqCklWChv2WyKik7blnNris72bi6dus0DLZ0TDxW1odtopDs4D8MDZuLU0kXq61CijSIIasOvv94VTUfLiB74fXngup7nvrnvJABJ/5PfLnTTdcwOj4RsdYmo5glBWP6iOtzwvIKSWstI0Wgx80SBblpKtE9KQs/We7EaWoT/8wbWP61gMmCH0lMDvokT4j25TiQU/ITFkek9Ow6+7WH2gwsmahCPdwyw75uw9HEO2gUZSkfyI9zBPCJOoJ2SMmg46N61YO/rNoa39Xi41oFuXysMfh36/Fp0b7bAfWAH6RGi0HglWNCbzYgJaFjRv6zGuy+b9It96N3SQvNKiV9HvSaDfFEIxXItnPs23BzJQd6DDEVM0OKsoVwBG/1VMzpXVWhbkUM2K4oJBDYuGmbKIJ0qxsAbHfRLzbjcnUbFBIpx/qH3vQv9b3U03IQ/HfFkERTzfFj8w8jSpR7GBE123uFEYAzaDRIqX/2JAtJbDat/COkd7CNBva2cMvq0MGxp0PRSCPF8BXjWG3FgNHc9XPT71Ojy3sMFdfJRCeKxEsVtKwFHwALZfCUk3tIfNR8XiJwc1LmL4dg141JPKtj3WUdNFJqLGFVPC4OkR4BxajTWsChY64wmCnMxsWPCHcutKBxMVp5mxA1S+aMComToaqTRUQknLTH62kHOVEE+VQnjahscNCy0cMBWsSI0TCQcZc5ALkEYckL5A5noWSBhfm2AecMAjbcRWV0pUTh0HE64TNf0mczcnnQyu/MilaFJCae1nw2fbz1DnVOxyGTlKeZft/Ff8x1BRssfACjTwQAAAABJRU5ErkJggg==',
anchor=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGdSURBVDjLpVMxa8JAFL6rAQUHXQoZpLU/oUOnDtKtW/MDBFHHThUKTgrqICgOEtd2EVxb2qFkKTgVChbSCnZTiVBEMBRLiEmafleCDaWxDX3w8e7dve+7l3cv1LZt8h/jvA56vV7DNM20YRgE/jyRSOR+ytvwEgAxvVwui/BF+LTvCtjNwKvj/X8CbgXPOHMEZl559HsTu93uPQi7jBiNRgMEx8PR0GIxRB+y2eze2gqQeAXoSCaqqu5bpsWIdyzGvvRrBW7rdDo2I6ZSKeq7B8x0XV/bwJWAJEnHSMwBDUEQWq5GfsJthUJhlVuv11uckyiGgiH2RWK73RYRb2cymbG7gnK5vIX9USwWI1yAI/KjLGK7teEI8HN1TizrnZWdRxxsNps8vI3YLpVKbB2EWB6XkMHzgAlvriYRSW+app1Mpy/jSCRSRSyDUON5nuJGytaAHI/vVPv9p/FischivL96gEP2bGxorhVFqYXDYQFCScwBYa9EKU1OlAkB+QLEU2AGaJ7PWKlUDiF2BBw4P9Mt/KUoije+5uAv9gGcjD6Kg4wu3AAAAABJRU5ErkJggg==',
application=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFiSURBVBgZpcEhbpRRGIXh99x7IU0asGBJWEIdCLaAqcFiCArFCkjA0KRJF0EF26kkFbVVdEj6/985zJ0wBjfp8ygJD6G3n358fP3m5NvtJscJYBObchEHx6QKJ6SKsnn6eLm7urr5/PP76cU4eXVy/ujouD074hDHd5s6By7GZknb3P7mUH+WNLZGKnx595JDvf96zTQSM92vRYA4lMEEO5RNraHWUDH3FV48f0K5mAYJk5pQQpqIgixaE1JDKtRDd2OsYfJaTKNcTA2IBIIesMAOPdDUGYJSqGYml5lGHHYkSGhAJBBIkAoWREAT3Z3JLqZhF3uS2EloQCQ8xLBxoAEWO7aZxros7EgISIIkwlZCY6s1OlAJTWFal5VppMzUgbAlQcIkiT0DXSI2U2ymYZs9AWJL4n+df3pncsI0bn5dX344W05dhctUFbapZcE2ToiLVHBMbGymS7aUhIdoPNBf7Jjw/gQ77u4AAAAASUVORK5CYII=',
application_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH9SURBVDjLpZM9aFRREIW/ue9FF2IULDQuaYIGtTBRWGFJAgqSUsEmjZVgo4mFWBiwVVjBHwjGwsbCShExIAghoEUMARGNGFJYhIC7isXGRbORvJ0Zi/dWY5fFCwOnuHz3nDsz4u78z5HTlx6NDB4v3KjWvd0dMMPNUFPcHHPDVTF3XBU1Y/uWZHVxsXzl6e3hibgwUBhvy7WH3bmWHm5fres4MBHXEw/16s+Wra8lHgBiV+f6mX0tA86VlkkBbgCsNxQH3Bw1MBwzR83Qhqflxro63Z0dqGkKIOuCBEHc8SC4OGJCCIJIQESRyIksEDfS+9bIAE1SAFwEBCIHEzBzIocgEbGAiqMhdWxqWQTL5kAE3P8BiYCrYwIuQBAii1JAM0JTpAxJxQaQxUJsxvTbSV7NP6e2ukLSSFjT/cBJ4kaS/HEggLsjIvgG0Is3T3hfnuLYwFG6dvbwcuEZcx+nKY7mbwbPskSAZC4k00GEIMLk64ccPtCHBqVvzxAqCcVD/QAjwcz+Rsg+M4gQbahv37/QJts4dfAiAJdP3Gfvrl6AXFxeWn58/k4ybKqYGqqKmaFJgplh7lRrKyxUZpmvzDA29IDS1Fly0VaAX7KZbSyO5q91de+42t87SE/nET59fcfshxk+L9VuyWbXuTiaLwEXgA7gB3Bv7m5l7Dd8kw6XoJxL0wAAAABJRU5ErkJggg==',
application_cascade=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGeSURBVDjLjZLdKwRRGIf3X1wXuNAikiKfWXaVLO1SIjYlG9LYaGKxNSXWx1rnwoUo+Zhdu2vkRsp32ePCvH5nYoaGNaeemjkz5/ee85zX5W6VS4bjuc3uhSzzz4NohnlnT1nHzAlrnz5mbZEj1jJ5yHwRxsS/ROT6jiu4lEuF12+YE5pHd1O2AFHZKXVDSWYL8EcvxKQjaga27AG+ubTxUUllMlOJq9fB1Us+sAJieR5azPJ+Oc0DC2e8N3rCmyYOOFxocOGxAiTVCBhTtMJ08pYXY1i55nChwUXeDGgM7xsBovJ/dErnHC40uNDMAGynr35kj3VJKn98eQOcPzwLCib3gqcCf3l9e8QiDS6sgK8HuBCTWnxHvRtT8joqEfqC0BeEYxJ6g9AXhL4g9AXBBaF4gxUgqUZAKJYjnNMRcPFuBsCFESAqOwUurPvEdsbhQkNfkNMBFz+b4tPFnwt0gS7Qjfeq4MYvARBWbHyFiOEJrNkD4MLxEdxtsj0ALmS4MATVDm5TdTBBlf3rVNGjUHl3nMp8y1TqjYkrFMgf+hUje+AiV2IAAAAASUVORK5CYII=',
application_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH0SURBVBgZpcFPaM9xHMfx5/vz/Ywfv6ZwGC1mtRsXtRvFQSmlKK3JkRJymAul5EJcrFZrcnO0QpELDgrlKJcd5M9PtpHawew39v2+3y/7rlZuLI+HSeJ/2OGhO2d27+2/PtNWUwIiUAQejkKEArkTEnLHI1i3qpybmJg8f+/GwGju39U/0tFopq4GK9Gca/sIMJrbpVJ75gcrNV8qsSjLxZVjfazUiWstalkKaguVI0AhPCAQEcIj8Ep4JVzBgoveTZ14OLWMRM2SYRJKhkxYGCkZZgkzxwpRRCJXohaVU8seTi0BMgODQhAGEaIQJCvIBm7CU1ALD2pZIZaYgUQCZAYGZiAXYSADklFEQS3CqeUIZ5mZsUQiATIjspEjmHp4k+mHY7Q/vaWjq5sdq/cAB8lVWbLEDAMkYWaIRRIJmH50i9lX4+w8cpzVvduZf/OYNc+f8mRfx9kkD2oFYICZYYAByYxkRuvuMH17DtF49wy7fZS17+/T27MBmYZyRLDMAGORGX9amGrR2NQLB86xLF/eTBG2LU9+aI2fGi4Hwp3wwN2JCLwsiQhCYqCxnvbrRzQfnObX/BfawOz3Ai+YMkn8zYvB7surOpsXt2ysck6fmf1W8fFr4eVPXTJJ/IuXg1svzM1MnSzcerzQpGBs/+Pq6m9gRhRi/EsJnAAAAABJRU5ErkJggg==',
application_double=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGnSURBVDjLnZOxilNBGIW/mXt3CZsYQQtjJYqPkE4L8Q20WARbmxWEFQvBXrByQdDKF3CL1QcQH8DOKmVIkWIFbdybTe7M/x+Lm+zeFELcAz/DwJnD4eOf8OD5p4d37w2f/qrUwR25k3PG5cgMl5AZ5k5/O81Ho+mHo7e7RyxVDu8M97c63TjosIk61cz2gfOAWVKc/T5hU50mxfa9lInXj29vHPDkzYT1ADkAi2x8/jq6fpy7N37+8eJfPHqX+zx7/1397VSNRtOXJRIAMcB4tnOr19thcHWjMt1qZu9KcwMghEBVi+o/eZSW81nARXiUOaXzgBYPuTCH7I65Y1nNyKlN3BxcwtwoLTUNItDmoRhQECWRECIhGKEQhUfK3Pg8G+V0PPm2d5Du5zpxZXDtrA0BCoEkCkEMBWUAC8Ji09TNG8NqXnz8IUnK7sruSmaqzTQ30yIlndZJszrpZJ4kSY9efVnfqjaP9hmBECNFEQkxEIuVP1O2A9Z4LB8Xy3OlrbbfbD1gOp4c7h2k3VwnzAx3Jy0WzY90Q6ZmK93xBsNh0JL8RfUXD1Ut4VHY1QEAAAAASUVORK5CYII=',
application_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJRSURBVBgZpcHda81xHMDx9+d3fudYzuYw2RaZ5yTWolEiuZpCSjGJFEktUUr8A6ZxQZGHmDtqdrGUXHgoeZqSp1F2bLFWjtkOB8PZzvmd7+djv5XaBRfL6yVmxv+QjQeu7l25uuZYJmtxM0AVU8Wpw9RQU8w51AxzDqfKhFjwq6Mjdbj1RN0Zv2ZFzaloUdwrL2Is4r+y7hRwxs8G5mUzPxmrwcA8hvnmjIZtcxmr3Y09hHwzJZQvOAwwNZyCYqgaThVXMFzBCD7fJfv8MpHiKvaV3ePV2f07fMwIiSeIGeYJJoao4HmCiIeIQzPXifY+paJqO4lZi/nWPZ/krabjvlNHyANMBAQiBiqgakQMCunbxHJviM9bQeZdBzHJUzKhguLJlQnf1BghAmZ4gImAgAjk++8jP56QmL2GXG8zsfFCz8skA1mQXKbaU3X8ISIgQsgDcun7FL7cJjFnLUMfLyLRr0SLS4hbhiup5Szd19rpFYKAESKICCERoS95neyHmyTmbmAodQ4vGpAfmEn6YTtTahv4ODiRkGdOCUUAAUSE/uQNfqTaKFu4jvynJiIxIzcwg/SjF1RsOk9R+QJMlZCvqvwhQFdbM4XvrynIVHpfn2ZSWYyhzHS+PUtSueUC0cQ0QmpGyE9197TUnwzq1DnUKbXSxOb6S7xtPkjngzbGVVbzvS/FjaGt9DU8xlRRJdTCMDEzRjuyZ1FwaFe9j+d4eecaPd1dPxNTSlfWHm1v5y/EzBitblXp4JLZ5f6yBbOwaK5tsD+9c33jq/f8w2+mRSjOllPhkAAAAABJRU5ErkJggg==',
application_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIiSURBVBgZpcE7SJZhFMDx/3neV/u8ZlhoVxAkESoyJYoa3BojDCFc25psaS8CWxoEhxAagiCpHCpqaa3AyiIISwjTtHIou3wX/b73nFOPIEG0SL+fuDv/Q04Mjp052ttz6WvR69wBM9wMNcXNMTdcFXPHVVEzGqsrhamphXPjl/tH0p4jPcNVubrQkmM96gpFHQZG0mLFQ/FrnvUqVTzwW+rqXBxoZ71OD80Spe5GVM4UB9wcNTAcM0fN0MzRzFE3yuq0tTagpkQBdyIJQhAIQQgJJCKkIZAmKf7zBeV3Q1RJidqqlMgyJQpqShQAEUGCkAQhJIIECF5ieW6c+uZ9VD7dJ60ORKZGFNycVSJEAQgihCAkiVD88IDa5i4at3ZRmHsI+RkiMyUKZsoaEQERogBofoFv7+7RsLkJ/XGHLZ2n+P72Bm4ZZkYUskqFVSKICJGIEH15c5Pm9uOwPMnEtevUN5X4MfOI77OPySoZUXA1ogQQQEQQoPB5Ei0s0bCpiK3MgBuaf0pb71nmn1yhimWiYGasESAA4sris6s07dqPFV/hVqK7rwMrfySXm6ZxxyG6aiaI5MTg2FjLzm39poqpoars2fCUkwdztO6uQfMTuJd5fnuK7r5OJNkINcd4NHphpdpLB8Td+dvE8OH5vQPXtyfhPZ4tAc4fgaSmg8XXL5m+e/5Wyj9kK+Xc5Ghfyc1xM9wMN8PNcTPcHMxw99ZfSC4lgw+6sSMAAAAASUVORK5CYII=',
application_form=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFlSURBVBgZpcGxTpNhGIbh+20+DUkTRhkcNLBIOmG6+Q+egg4cgWExhDi5sDm5SErCoB4BDB4KjnVTBpooCTKUOvx9n4d+ISQOQNL0usI2i4hX7w7fNi/7H88n7tqAhCVSiWVk4Uxk40xSYvlhezkcnr7/9mnzoPRf9PcfLHU7K0vMo3s5yX3goExadybnY+b1r3WHmeI0u68fM6/twQlVsYVtTGUwEMH342Pu8nxjg1RSFWwyk63BL2583XlK1TQNtxmPx2iaVCWVyObz9hMIgwPZ3EcSSlEVyyhFBFjMGML0ej0uLv5i858ATABSUhUpkRJzLZgxDH8Mucuz9XUkUZVp25KZ7Hz5zY3B1gpV0zTc5uzsD9N2SlWcQhJ7bx4BppLEfTKFJaoiCcmACMBcW11dYzQaAQFhgsCGCIMD2VTl9OfJ0YfDdlOZKEVmIolsWyQhGytxGllYQqI6YiZss4gOC7oC3Q3wOtNMt7AAAAAASUVORK5CYII=',
application_form_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHiSURBVDjLpZPNaxNBGMb33/DWoxcPngr+B4pQggdL8WNbaxQbGtugkESE0IZEkiYhpC2GtJQatKW5pFYl2yrKCrZVrMYo2jbHfJElm/1IPJXHmdklbZGURBd+PPO++84zM+/ucAC4/4GzjC2PTKXzuvvZPlxPCcldOBd/4v7Cd9ybz8Exl8V4fAd3H3+GfeYjbLEtuOLvdTqPGXhT+YNopoxusUfeHTADuvK/YA28BTNwJffQbDa75obvtWHgfPILjUYDOkOHrutsLIpiWxqkhvcKpgFpmKqq6J/IttBITAvbPZqm4ZrnlWFAu11XFCh1glI3VTnRgL6/8vCFYUA/Va0mQ5Zlpgy5RrRGcoYeIht5wsCDVcPAkfgKSZJQNaFjqSr9de54KozxmauwhvrAPzoPi3PMMLDPfkKlUjnWAxofPcLzD0vwrAxiLTeNL0UB4fU7uBw+g96RU0FuNLaNUqlEKJtqcNRgKHAB6W8RpH9EWRzcuInwxi1q8JuzRTdRKBQJBRQJhRbFll50n8XLXOJYI1ez09QA9C4s3w6J7M8a9r/BoHcd1ycypMtrrFH97jTO2XoQEIbgE3g22ZfhD3fQyY0jhZOXgqcxJQyzlanSmPWg02tLiv0ElW7bVD/N/wGu4yJFlYuLaQAAAABJRU5ErkJggg==',
application_form_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHvSURBVBgZpcG/S9RxHMfx5+fuq10d/SJKQ+rUaLoWw60bIoIgaKhBpDksDBFbCooQiqAlURCp/gIjHIQWa4h+QDRYtFhDqOEdSqCSelbfe79f+UUEg35Jj0eQxP8Ip7sGLxaONt+eLSsrAe7IHXNDLlyOzHAJmWHubKuOl8bGipeH7rT0R81HmvuqMtlUTYaNyC6VrQ/oj8qxUuXZRTZqOVaKFZFMXDtTx0Z19E6SiCRHEiIhEBACb0ZH+Z3DTU2YG4kICTOjrXeCNfc760kUCgV+ZXFxEa8YicjccIm7HTkIAgVc4k/cHTcnEcmFmxMCyFkhCCKfzzM/P4fEOgEQAXA3EpG74W6IVYEVgrH3Y/xkdBh7PURl+hPp3bUc2nwMOEVUiWPMjM57M6zpbashUSgUSEwNDzBXfMXBs+1sasiz/G6E7PMnPD5e1RHJHHen59weQCTcnfXGH/bQ3HqBzMen8OwmW7bvoCGXY/SDuiJ3x12AEwCxqrHxAKVSCQh8L02QqW2Ak5dYE3XvJe2hPiqOTz64MRi3uBlujpnh7lgc4+64REtmJ+W3j8gOt/NteZoysPAljaUpBUn8zYvWuu7qrdmr+3ZVoig1xcLnChMzaYu/6nqQxL942br/ytJs6XzaQs7SKgoGToxUbv0ALswWDRrf9Y0AAAAASUVORK5CYII=',
application_form_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLpZPvS1NRGMfvv9Db6FUvIpDqLwj6AZER1EhBkvTFTALNlWJrkwKRoWKGjZltaUmuYmsIq7SEtrSt1qakzh/TtunSdq9j8253u3fL1vh27t2KNAqsAx+ecw7n+z0PzzkPBYD6HyjZZVNNpzXINz0OQP2IYPwI1YMFXOmfQ+O9WTT0eVFvmMQl/Qcobo+jVueG2uDkRZ1koLEEc9qRNWwXRddYTjIQb/4XqjtGIRmojX6k0+ltI2+15Q1UA4sQBAG8BA+e56W5w+H4jQmbDgsPy+C3ajDVdwzT3UUVlFiwZDKJshbvT1JkLQp+HQJtBWNXIuF9DqTCiHstcN04vkaJ1U5wHLgEgUsUIrfJQKCHwM43Y2P1KZjRDqRnBvEt5MTM/eoMJT4Vy8YRj8elKBFnSWTJHovoEhHPXkMu7UQ6WIt1z0X4BuRw6+XwaGV7qIbeacRiMUQLiPNYNCZl4LF1g51RIZdxQQhUIrNSjuS8GqH+06i73psvoqJnApFIZFMNxPXrwXYw48qCuAKZT2fBzTZhyXASjM+JkkZL3qBO5wHDMIS1QmTge2fE/MtGkvYbZJbl+LJyDgmSSUB/Al85WjpzpsGcN6jVvkc4TBPCoAnuoR7MDasx/aILn92lEJbLsT51Ff67p7C6OAmaFs/RkNWb8KMXTBduOqSfVdVmh7n9KJAMYcFQArt6J97eKcZw8yHUqPQk7SfkZhOIRsQkGWztrpbzRdkNlw4b7ltwaw7DXLUrNaLaf+CP3bh1o/Tgjkxr5d6sra04+6rzyNgz5b7df2vn7z39IYTVO0xbAAAAAElFTkSuQmCC',
application_form_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH2SURBVBgZpcE/SJRxHMfx9+/uuT8qlgR6mvTHqyGwKMWpbmhvskFa2spFRJxaGlprKBQcKmhqcWloaQ2ixcFaDhUKLDDJSFHvT+fzfD/ffJC2Go5er+Du/I8wPrs4Vbk+9nC74V3ugIRLmAyXIxduhtxxM0ziWD6ur6xs3Hv1eGIhGrs2Np8rdmVKRdrRVW/YPLAQNWLPNLZrtKsZe4ZDkZtz/+Yg7Zqe+0IqchfujpNycCAEPiwv8y+jIyOYjFSEO2bG5Nw6fzyfOUuqUqnwN7VaDSVGKjIZcufp9BkIDh6QO0ecd9UNtvaa1FoFCrmI4x3i6lABmUhFLkcmQgAXhxyCMzw8zNuP69TVxeWhE0S5LFu7B+zst1j6XEMyUhnJkIzEDJMhGUpEtVrlR7NIudTJTiuwuWckIctgXzfbzYAkUlESx5gZM8++88fcZIlUrXlAPp8jaSXIoZU4xShL4hFJnJCK3IQkntzpA5yUJFKdhRwHsREbJHJSHqBRr+MSqUgSkgMiAM6RcvkcvzZj9ptGd0eeljlRJrC5HeM/1zifXSMVxmcXF0unTk7IDJkwMyRhcUwuH9Hf18PoxdPIYbfWwrZWqPR+Y/fT++Tr6tLt4O604/WDG3f7B/of9Q5d6VldetMI7k67Xk5duDVQvvRCSjp+A5XMMGcdZp9aAAAAAElFTkSuQmCC',
application_get=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHXSURBVBgZpcE/S5VhGMfx7+8+T2VI0SDVKvYSrKE/0FsIwgZpCFqiISSiIcoigkIosWyrOR16AU2NETSGZUQoiNYgBJ1znufcz3VdeUvOIX4+igj2QhemFq6fPT/+ZLMXwxGAO+GOuREeeDhhhkcQZpg7h/fn7tLS2u23Tyfmq/Ez43P7hobTsSF2Y7jbszlgvurlSL3NP+xWP0diSxUWPJo8wW5dfbxCUUU4xaA1AggPzMEJ3ANzx9rA2sDCGVgwevwQ5kZREUGhJBRBJBEK5CIlISUkQ52g44mqDQpvjaIyN4oEhASCToAL3INOQFKHSmAKLDmFm1NU4cE2CSJIQEggkCAscMHsp4d4G9w4eY/C3SiSu7FDEkgUCUgSqhIzH+7SH3TpNr+ZfjdF4e4Uqc2ZbRKSKCSBhHnL/fc3yblhbGSM0aNj1LnLlVeT5NxQpDCn6AACJCFAwPOPM/zcXKeuG+p2QN02HNh/kNWNFX6lBYrk7uwQkIAk0ZG4dfoOry++YXn1G02uaXLN8vdlZi+/ZCRfoqjWfqwsXnuWJ9wMN8fMcHcsZ9wdj6B/pKbfNmTLbKxvMD37hS2LbFFE8D/nHpyKpsnkOjMYZD6/+Cr+UUSwF38B/pkb32XiUiUAAAAASUVORK5CYII=',
application_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIMSURBVBgZpcHNi05xGMfhz/07hzTDiKZmEmLYeM3iKTKUiFhY2EhZ2NjIBgsWYoUoSWr+B7NhY6GkJBRhYSMvJYRSFDPPi3N+9/01Z2Jvcl0mif9h+46PH92yrXXpe0f9EhCBIvBwFCIUyJ2QkDsewcDsuv3y5adTN67sHytbo61rs+b0p6E5zER/u+PXgLGyUyt1vk8yU91aiSmlXJw/uJKZOnzxPY1SChpVdgQohAcEIkJ4BJ6FZ+EKKhfLh+fh4TRKJBqWDJNQMmTCwkjJMEuYOVaIIhJlFo3ITiN5OI0EmBmWjCIZqTAsQZFgVlFw/tZuTt/cjIqaRnjQSAoxzYxGApIZKRlFYRQGKcGvXLF4cBXHxjdS5R4RTqOMcP4yM6ZJnLy+DSlTRabKmUULVrJqeCMTvTZ7x0ZYoKs0ylzXTDPDAEmYGTkqdq45hCvwcALx+cdH1i0eZbLq8qx7iPXnDswv5UGjAMQUM5Do5QpX8P7bG+rI5Kipvebnrwk2LNnKZN3h8bsH38qI4C8DjClm9HKP7JmhgaXkcFzBlx8fWDh3mOcfH/L47Qs6Tsv2HR8fH1qyaH+4Ex64OxHBz8Ej9KqKKip6uWLF4Go2jezi6YdH3H/1hGXdE7fvXD6zxyTxL9aeS+3W0u19917f/VQFOz5f0CummCT+xchZa3sUfd3wka8X9I4/fgON+TR7PCxMcAAAAABJRU5ErkJggg==',
application_home=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI/SURBVBgZpcFdaI1xHMDx7/Oc/3ms8xiZxLxcGIXF3cnN5KWUXIrc4EZcSC5QKC+1xjLKW5ELLzfIS8iNS0qUtxqllkYstllsNufsPKc9z+/3s/9q5Vb7fAIzYyKCDXvv7F6xutg2ULHYDFDFVBEVTA01xURQM0wEUWVKlA53dHQffHBm80VXbCpeyNfE4cwa/kc8XJELwEVXSS2sDJT5l8uGaeptYSRLSRLlw6LjZLmYfyWphYxyJsaJLQsZlyZ/eHejhXx9HaVywuyGOqJPR1m//wb5mlrG7TjZhReaKd5IJgyXB3l/ax+hM4ZKVWoa1vLr+1em18/lYetWvnb18OVHCU9U8ELM8IIw4OP9Q4RRQClR6tccYMmqbcxZuYdvnZ0UptTy5Nx2CnmHp5ngOVHBC4FqdYhUI+avP8a0GXNIK7+Z19iE2RGeXj5Mlma4KMRTUTxnaowJApbvvMnjR/eZNbeB022tDA4OEccxzc3N3L5X5EDLaVDGqApeqCqMC4KAVAM0HSGfj2hsbCQIAjAjF9fhchDmGKOqeC5LU7xXvS94/e0Nn8odLO1fRm1cS6FQYFJ+Eu0Dbykt7+H8szYWTl3MpuJGsjTDcyaKd/PldZJKwkDvENcuXyWnjr6ffVSrVa5cuoosqPK5/zntSTubihsxVTynqngt606RiVDuL2NqiAiqiqemOOeYPG0yURThqRme6/7SdXfX2XSziqCiiAiqiqQpqoqaYSqYGGqKqaKKd5dRgZkxESET9BfqVjgB6mRiLwAAAABJRU5ErkJggg==',
application_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIwSURBVDjLpZLNS1RhFIef897rNDSoUEFFIhpCi1okzKKyRSFBFC3aCFERRQTSItrUHxBFbQoEkYJaBdoQ7kqQwm1BREjmF2JWFuKgNTozzr3zntNiRmthC+nAb3fOc37nQ8yM/wk5ff3Z1cNH0vcWCpYyA1QxVbx6TA01xbxHzTDv8arUJeL86Ojszf77Hd1hui3dVZNMue3JDTVO5Qu+C+gOC7G5wsLyhq0XY3MAoXnj9tmWDQMu352hAjAFICp7DDA1vIJiqBpeFV+2ikyJvNG8oxavvgKgegVxgphhTjAxRAXnBBGHiEecJ/rwiLA4RW6xHS1vBsCtkhwgIogTAie4QBAHgYOaIKA8liGV/MLOtguszL2k0UYqdabVPxBZAzmpdA8CIRAoTvQTrAxRvydNfmaAILmLc81vGLi2+0yoVQdUHQBghgNMBA2FcvY9W1tP4gsjJLc1sOK/0Xh0H1Kc7wnLcbzmQAAzQ0Swv0BBqoF48TMuyCIuT6IuCysFln+UgtB85QoBVIpEWF1slP1EbvghkojRchGnP8HVYnHEx74pxscXroSq+meEqhBhafIFudHnbGk9iEWDiF9ibDCiplQiEOHp/Am6em/1hrPTM5nOB3GHeo96xXuPqtLZ1Mf+9mNE2R7CsJ7h1yV6xw4wl9iLGUSJTZnqvmxdfe0//rY0d8fy46fsXdehpccXW86vlxf+61W/T2df5X4NNRUW8sWJifkbl55MZtbL+w0EmEFfv0ZNiwAAAABJRU5ErkJggg==',
application_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIiSURBVBgZpcHfa81hHMDx9/l+n82x0zadyBabMFtKSS03XGxXUlMspqxcufGrcOMPQHHhR1v8B26Wxi7ccOlKLUlqTJvQ0H5gv8737DzP5/OxJ5RcyPJ6OTPjf7juCwOn93a0X/tSsoIZoIqpIiqYGmqKiaBmmAiiSl21XxwZmbg4eKPntmvf095XlS8k6/OsRGGxJH3AbVfylpS+LLBSmbeEZc7EuNLbwkqduPqOyJkpUSUIBoT5Z5Q/9mMSMFE0KCaKeI9Rg990mc3NWxEVIocZUS7JkVNB5p6wqmY9adU6zAfMB6SSkX19hTX0UFVoINIgRE5UiBJAwhSV6UfUbugizA6T5HchMkP52yhp4zFcQxdiSqSiRM7U+EFYmnpIdW0LkKLZZ3BLzH54zOotZ8g3HUINUk2JVIXIqQpRKI1S/nSfuqZ9WMggbQTvMXXownsWR26hPsOSOvJtR1BVIhe8J1oYv0l1/TaymVdoUKprWkAd9Q2dZDMvEF9h7uNzirsvka5aQ/CByJkoUb7YSShPQ5KQSyt8HR+irrGDUP5GpTRNUr+T5u7ruNomkjTFVImcqhIVNvbyi4YFZl/fpTT5kvL8JMUd56hvPczv1IzITbx9N3Dypu9REVQUEaF3Yz9ta4s8fZMw9Pks2XCKySBqiqmiSjTAMjd4o+cocJSfxu9t369e7mQzcv7gqbEHB/k7xx/Uy4R6OdB6fOwl/+A7Obk497M21x8AAAAASUVORK5CYII=',
application_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJPSURBVDjLpZPta1JhGMaFPvZ/CH3rL1BqICLzg0EMKmSwmZXWskPhSwQJ2yQzVEYyaRI1ikkOCqISJDU/9IZRKyqn9SWYuBedb+d4ju7quZ/jBvvYuuGC+1z3dX7nOc9zjgaA5n+ksbiWHKGnlbbvcRneR0yLJXge/MC1+99wNfEVwsIXXIl/wuX5IqbufoBz7h288UKb7uOA6SeVQfRVFf+qqXBuwAH05IPobDALDvAuruIgNTGbUQGehz+50VP6kJjEnoK2qKApymh0ethsiajVu1hb7+BPrYVfa02eH59ODwFsw6jkwQByv89BXVlBR1JB210FW20J600GaoiobnR53nrzhQqg3abqM4BCEKYeA4kM1KHVMFCLVtRRUG/J2GhKPH/mxnMVQEfFATs7HLAL6jFJffWVdkHNIYjq1PVnKkC493kPQGp3u3iZTiMUCiGRSCAYDGK1XEZtcwu3hx7NxoUwdDrdYc1U7KMKYJJkGQssEIlEUK1WUalUkMvl+PV8PI5SqcS9QqHAvdHR0VuaS3Pv947mdTYLn8+H4soKZgMBmM1m2Gw2RKNReDwe3pMXYDOCCILQ0Dijb/cA4XAYMzMzHCBJEoxGIzKZDIrFIvL5PFKpFPdoRgC/3y/Sv7B07s4b/mWZJ/ywn3dywEnrRRw7PgKDwQCLxcJFPXknTl/gALvdXt/3Zx1hxZb4nW1SPZlMbi4vLyMWi8HlcsHtdvOePJpRhrL7AFqt9pBerz9qMpl+T05Oth0Oh8jUGxsbszKNsL5JHs0oQ9m/1aYTg/t8sGkAAAAASUVORK5CYII=',
application_osx=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF5SURBVDjLpZMxa5RBEIafvdu7RPCMhYLBxCoiaqO/IHZWNvkBooh1WrHUwk4s0sTGQrsQwf9gK1gpQoJoY++d5Ludd2YtvlwuaWLODAyzsPu88zLMplorp4n85t3W5vLS4sPc6/VqgMKxIiRRJEoRxYzx2NpajFKMvaaxH993X+flpcXHl169SAv3P/Gge42Nm2//2TUlOH/ubO/p85eXc879dHF1lfmrXe50bgNwfeXKsQJfd37S7XZw95TNxa+7a8Aa94BxKXz+sgOA5JgJuWNyJEfe1m+7gUvkGsGtGyv/NcDt9x8uZCkA2GvGM8Fn5ueIGmR3B2Cu3z/yoNbaJpWI9uwR1NreAXgEuZhmAicJEO5kk2YGD5qFTx1EjRODk4gIsu0LuMeJwSMCxexgIIfhduPSsQI1gjwajYZN0wx+D/8QEfsC00kfdjMxVKn0cwcPV1p/8mx9MFh4BJ1BcU23T0JmyA1JuBlyx2WEO2Y2jPCP6bTf+S96U2WlbWXHPgAAAABJRU5ErkJggg==',
application_osx_terminal=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGfSURBVDjLpVOxSgNBEJ297B2iiUmToKABCxEVKyGVoJ+QLp1RxE4sRAQL26RVsLHRQouAkMoq/yD5Ads0ErDIQe5udu+c2eTCGU4MuPCYmbvZN292d0QURfCfJZ9eXh9WV5aPpW3bUQigQg0YKFBKQcAgP0AE38eRDcgShp6HX/3PS0mbT5dumyJ/+A5HmQ24337+s6oQAIXFrN28e1yXUjqifHYO9uYHXIgyFGR2JukL83OACoVEreDg6ppYRSosyzKYjhlrO3sgozCETCYD/X7/R1LahmRcLBYhny+ApVQ4kdVut6HRaMx8A0iHbGmtgS7AoFarQaVSgWq1amLHcSb/kqCDMzbAAGSAynxgtFot6Ha70Ol0TMK07GRb3Db6PkiWEVer1+smgeM4kdf0GZgHRAWVGiuIZSUPMFkxLWYFih6WxHELcdVpklgBx0mfLTIBP09mY4LRK0uvGCMm4T2+IgLXdQekIFcqlVKTfyNh6/ueJ3u93s3W7v4J8eYCzUNEw4SjYeIelUbja/KRrlxT1VBzDg68ofsm/jvO35HitdVS/1ysAAAAAElFTkSuQmCC',
application_put=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHbSURBVDjLpZM/a1RBFEfPnfciq4uChLgrWChaihBYbAxqBD+AgrGwVBDRxipgH1AMEQJpbawU9SvYCRaiEmOaBUkRDRYBwf3zZt691+K9DWuXxYELwzCce37MXHF3/mfJtYcv789d7jzZ7XvTHTDDzVBT3Bxzw1Uxd1wVNePIgdTb3NxefLuysJZ3LnRWpxrN0GpM1LjZ6+sqsJb3k4f+7p+J1QfJA0Du6izdOjMx4M7jLSqAGwCxVBxwc9TAcMwcNUNLr8qNqM6p9mHUtAJQv4IEQdzxILg4YkIIgkhARJHMySyQl9V9K2vAiBQAFwGBzMEEzJzMIUhGLqDiaKiMTa2OYPU/EAH3f0Ai4OqYgAsQhMyyCjCKMNpUDKk2YyDLhcV3t0mxJMZEGiZe3HyDWWUQypT2DEYAkbp9Ha0YJi6dnGf+9BUG/QKAMpV1hDpLBvhYlHGjYlBQlAVREw1psPlzA68N8pEKgNQ16r78folvP9aZOXiMYRmJGpk52uLB87ukvAXcIN/+vvXq3rO0YKqYGqqKmaEpYX6WeGKDQ9MFRRoSNbLza4fe7ymOl9dfA8h+pvHq8sUv7en2uRgT3W7388en67N71vsd5/OPZj+kYUqfVr7OjZ//BRjUGmnYsJxdAAAAAElFTkSuQmCC',
application_side_boxes=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFwSURBVBgZpcG9apNhHMbh3/ORWiztpLSrQ86gmw6egkuXjE7i5CQ0OAZxUSj0DCxiB8/A07BDBXEoKEoGbaP45n/f9im+SwhCyXUl26wiPXjy9vG9+7svpjNv2ICEJUKBZWThCGTjCEJia627ODk5e/ru5d5h3b27ezBY38jb61zHxsUsDoDDOuucZ9NzrutX50yz//rUy4zffPT/PHz+3peotmjGR6f0JqMhven5b2JuYm7C4k+YOzubhIImY9PbvrXJoloqdZCpNzJra5Wbg0qjedDUUND7+v0ni0qGnAo1QSQTWTQK0VTLNJPRkGVKAiVwAnKiqNBIQVOloBkfndKbjIb0Us1UCRkyoMQVSTR53nX0dm5vsSgDKWdKyaScyIUr825OUx2i9+XbDxbllGgElJSwRGOJpkqimYyGLFNSoin8UwqNbJp69unz8aNX3Z4iUIiIQBLRdYyefUA2VuAwsrCERHPMpWSbVWRW9Bcyl/ydiZeC0gAAAABJRU5ErkJggg==',
application_side_contract=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG1SURBVBgZpcG/S9RxHMfx5+dz3+LKsKxBC4KQXIvArYb+BQmuIYgGoaBJiAoiLCqwxUGwkBr7gaGOTu5NZhBkBRU3NEV2Q3d69/m+X+/6DEJcNoiPR3B3diKMjM1ePXN2+OFay3vcAQmXMBkuRy7cDLnjZphE7+7UXF39dmNhsjZdDJ8entpV7Yn9Vbajp9myKWC6aCWPrbVfbNd68sgfhZtz6UKHQBYIBCKRO3O3uHhylBOHT9G39yBFLPjb6ESdLLqLTSbj5etnZOutNlYa7s5WTEZW4E7mwMzSI340vlM/Xqe30ktZGjjg/EOlkUWTkT1ZekzFIscODfJ0cYZqZQ8YSI7jdJOJrHA5E/P32V/tY2hgiNKNA/v6WP34gXang7sIBLpJRhYl4+a52zSbLd58WmH5/TJzi/NstDew0pCcrUgii2VKZNdq12n8bPD5y1ee333Byru3BA8EAu5OtzKVZNFNbLp3+QGDA4Nk7Y1EJyXcRQiBbi6RRUlsKioF41fGyVI7IRP/I3eyMDI2O9t/9EhNZsiEmSEJSwlJyB2X4ebIhUtIZK8WJmvng7uzE5Ed+g22TgZNyTAeRAAAAABJRU5ErkJggg==',
application_side_expand=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHXSURBVBgZpcG/S1VxHMfh1/me41USwpa0XwTpUNToECTR4D/g0G1oiKChCAchChqiKYzCIbGWtlpuhBBBS7TY0FoUSZOVSIo/Crz3eu895/N511cQDWoQnyeRxE4kQyOVqwNn+u+u1NUpAe7IHXNDLlyOzHAJmWHu7C7ltenpuRuTY+WJrP9U/4O2js7Q3cF2dNbq9gCYyOq5Qn2lynat5Qr8kcnElfMpkCCJWqvK919feT3zinsnH/E/l0a/EQXJ+ZciL4juv7vD/M9V5pfqzC1WmZlfJTI3ooDEBiEgQYJWMyf6svCZ0bfXydoDpVLGrraMyAsjCuaG2CQ5RVFwsOMwF56fpau9CyRuvRymLU3JSoHIzYmCXCRsIbDCaDabDPSd5sjeXnoP9NGwOuXxQUppSuRuRJm7sUESLiEXa40GUx+n6NnTw+LKEqomvLj2BnfWuTtRKPKcrSTRauUsLC9wc/A27z99oLbc4PHlpyQhIaSsK/KCKMgcsUlyrDBmf8xybN9xjnaf4MlwBZKENATSJCGSO1Hm7kRCRHJRFAWNepNo/OJD/pKmRC4RJUMjlUr3of1lN8PNMTPcHctz3B2XkBsy4XLkjjvRs8mx8rlEEjsR2KHf89sagbhDhaEAAAAASUVORK5CYII=',
application_side_list=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGQSURBVBgZpcG9alRhFIbR5/tRA0Fbbe0tp4tFsHFKLSaNjSIGJlapDIq12BgIxJRaCGIK8QaUNN6BmEJBLNIGCcko55z9bs8WpxlECLNWcnfmkW6uv7l/dXnw9HDii+6AhEuYDJcjF26G3HEzTOLC2fZkf//gwdtnK9t1sDTYOrOwmC8ucBqLJxPbArbrpPU8OTzmtH62ngkPX33xf3n0+qv/z90n771HdRdhvAcjfaBpGobDIVOHx7+wzrHOMReNOZcvncdkhIo7YWeZ3jVm1VJJyUjFKcrUzgnqjFBNRhjvwfUf72jbltFoxFTJkFOhJrDkWBZBJkJ1OWFnmd4NZpUESuAJyImiQpCMUCUjLL2Ae+klTdOwurrKVKqZKiGHDCjxhyRC7dqW8PEOvdvMyvRypgDmTk5O6NqOUN1EuLIJa+eec3R0xMbGBlM5JYKAkhIuEVwiVEmET+v01phVUiIU/iqFIHdCPfj2fXe82a7IDJkwMyRhbcutx5+ROy7DzZELl5AIu/SSuzOPzJx+AwKoFtIrwFHLAAAAAElFTkSuQmCC',
application_side_tree=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF1SURBVBgZpcGxalRBFMfh35yZGwK72mrrO2yxoIXFvIBNGksrsbISFGuxMRDIGwhiCtttrq9hCgWxSLtVsor3nvN3RwxcggTCfl+SxC7So+cfnz14uHi73mgmAREoAg9HIUKB3AkJueMR3N4bLk5Pz158endwXBb3F0fd/szu7HMTs4uNHwHHZTPINutzburnIKN5+f6r/ufVh2+6zpM3n7VFkYJmtVrRdR21VqbW57/wUfgoXMFvF/fu3sLDaQoSTdd11Fq5quRCSk7KIodRRtHE6DTFw7nU9z1NrZVL2cBSpiTwJNyCJjxoTCGa5XJJU2tlKicwA8uQS2IvZ5oIp7EIZ6rve6ZSMYpBNigZLPNXRNCUcRiYqrUyZWyZkQGXsCSacRhpTB5M9X3PlKWEpQQpkc3IKdEogqZEBM18PqfWylU5JZrMPznThERTzr7/OHl6OByEO+GBuxMR+DDw+PUXQkLhyEUoUAQRNCdsJUnswtjRH6iT+f+z/UsKAAAAAElFTkSuQmCC',
application_split=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGaSURBVBgZpcG9alRRGIXh99vnKIGgrbaCl5BOC2/BJgi2NmIlFoK9oI1CIJ39kInxhxReh1VKCZmfQkiVTOLs863l7GAau5DnCdtcRzx+ufPi4aON98cLr9uAhCVSiWVk4Uxk40xS4vbNenpwMH395cPmdr/xYGPrxtp6ubPGVayfLnIL2O4X1WVxfMJVnVUXVnqnefv0Plf17N0hTW+LZjkkBiyTAmEkkxI5mBxMWizT3Lt7i1TS9Ng0UYKwcQkcJhSUEkQUIpLoTKdCP5hGQ9L0qaQpgCMgoDMoQDKdoURHH5BhsohGKZpimdFoxGQyYXdnh9nREXvjMbPphO97u8ynE/a/7jKbT/ix/5nf8zmj0QgpufDq0083g+RB8iC5Zrpmepnp80z/qdVny+rFsvrkvLp58uabV+iHWrkQQQC2iQjMik1hpRQ6IG1KmGaoA03vFE0HmJUIsGkigksCuggs0Vii6SVxKYBgJYL/dfzTdTSyafrpr8Px8491U5koRWYiiawVScjGSpxGFpaQaMashG2uo3BNfwFx+DLsFQ4W2wAAAABJRU5ErkJggg==',
application_tile_horizontal=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFCSURBVDjLpVOxSgNBFJxcrkmC+AV+gP5BwEKwtLESW0mZD7CyEyViJVhbG7DURvEXBAsDKgpGxIjIecnF29t9byyyAZu9wkz12HnMm3m7WyGJWRC3Dq7aq83FTpKzMcwyqFNYVczHNrvpDc5XmktrXz9siAJpksCKYK5qs9u7t+3T/fXjysn1q9TrtWhqhCSUAAkMR2PUGzWoAkqCBJST+qF3r4ft5Wo8toyy77En/zQpoSRSk/kaEH+mCoxyiQAARxcvDGHv7DnIbe1ekiQiUQ0uyIkEucJaAEAkUiLgwpzNvUDhyqa4sLiZcFHplBJxN41QNsWWOXDegTFFsMnkJixujI9QlOUMR1DxDopC/hVhentx//Gp2+rkG2IdksEHnDioE4goqOxv7uQLUjiIE6Sf76AI/NPpAkBl1t8YYUb8Ao9lHyy2IyRjAAAAAElFTkSuQmCC',
application_tile_vertical=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF+SURBVDjLnZOxalRBFIa/e3djWGyE1GJhlcogRLCwshPrgI2NjaCktZZUCVaB5A1s8gI2PkMKGwu3CIIgKGuuu5ts7pzzH4t7k5tZSVw8MDPMmcPHnP+fKSKCFzsfXz1+uLp9PIubEoSEJJI3q0nIHFMw4IRPn7+/fv/26R5AH+DR+upub2lQrixBRBBARDsARRABiub8zth2gQ4wTVFO6hMioi2CIJDIcueg8cxL2ugDuAWbT26zaLx8d0QGMPcriyMAuhb6vYI6pRxQVeMrAUUBUFAUULS56sfo4rwESNfcIIO1azLPb3A0/Hq4/+HW/V+n4Arkjlwkc9yFyfHkuMRyTJHpMAOsP7i31lsesHKjs+qy8vMu/K7qtQwwPlNZzSZIlz2Pv/cBUjCdtzElsfXs7sI2brz5kotodr2I3cuMRsRkOcBd/1a/m/CzuXdw/HO0mIWtj5NqlAO+DYcHz7dmG26GJ8dSan5fXSNv7JMbYSJZDcHBBfi8r/+NP5dvHwJcdb23AAAAAElFTkSuQmCC',
application_view_columns=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF/SURBVDjLpZN/S8JgEMd9i1JURPSDIoIkS1FQI4iQgihStExrhGmydGObyZyyYRaybBRFQb/8vxcgBIF92275ApoHx7jjns/37p49LgCuQdzlXmEXd8RON1L4QPjM9NwbQtkXBE+eEWCe4D9+hC99j+XDO3j3b+FJ3CCcvu5a5wgQLXV6ceUT/3Xv3mWPAJayE5/fboAA4dw7nNjspmoDQqevlDAMA+12G61WC1/fP1BVFfV6HbIsUyyKIgRBAMdxVD8drf0BzIU5scl12QZY27ZM13VSbzQapFir1VCtViFJEsUsy6JQKCCfz1P9xFrFBlhX5cTGVyUb4D96oESz2SR1RVFIsVKpoFwuo1gsUpzNZsEwDDKZDNWPhQUb4D0wHHUwHCjZgKVEmxKaptHc/ZmtL8/zNLMVp1IpJJNJxGIxqh/yn9sAT1x31IHbx6L/FtiF3Sv6s+a2NMxE65jaUMwtX9CixiIiRkM8RoKc2XbRVGZhnrGcJcDAr3FQwC803UMOARws7QAAAABJRU5ErkJggg==',
application_view_detail=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHSSURBVDjLpZPbSxtREMbzL4pSpYj1VilUmmpQSCKCSLAgihFTo9Eg0TRZY5LdxJgbKYo+2AstLRbU1nfxrlmTdLPJ55nZ+OCjyYGPYQ8zv/PN7DkmAKZGZGoakN5OKSeqPfAPtk9Ca8ew+g4xvPIHQ94DWJZ/Y3DxJ94v/ID54zf0z32BbXFPpToGODZOKrPpUzxX5pmdCgPo5HrUN7kNBtjW/qKe1TORMwDW1SPeeJ0ucMzlcshms0gmkyhqVSQSCVzmdSiKAlmWEQ6HOa/TkakBxMBolUQyFRRIpQruijp/3xR0XN/ruMiXcXar4fRG4/yOsaQBoGmzpa08x0wmg1QqhVgsxoBoNMoQSZIQCATg9/s5r300YQDoV9HS9Cr+l6vspFRzQgBVOCE3j06uVJ3zX47EDIBl6RdvdG9ec6Se4/E4QqEQA4LBIO5FSz6fD16vFx6Ph/PabLIBMM9/541ypYpyzQWpqD2VKiB54YJEq2VowwC8m/tqTFU+50i9RyIR7pUK6WSKbrcbLpcLTqeT85ot6wagf3a/rnvQNCjh8S1Ib6Z3+Wb1fviMLkcWr8bTYsqbPKg2u4JWawQvhsPCdkicLEHUkCQGNPwaGwU8AG9RQVkc+5PeAAAAAElFTkSuQmCC',
application_view_gallery=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG9SURBVBgZpcG/S9RxHMfx5/f8fPXyrMyOhGgoWrLJ4m6Ii2hrCyRwiKa2fg06aKNjLRVHQkNEtOnQUNReg5RLBJHUEhe4iEiUnj8+3/f7lR/j/gDz8cgksRfZyNjMrfMXa/dX2qpIgDtyx9yQC5cjM1xCZpg7B7rj2sLC4uTLB6PTodaoNfNypTRYZjcqa21rAtOhHVVqr6yyW+tRJbYFmRi71M9u3X3aIgmSk8zPz/Oi5yg9eZnBrpOYZxQOheDaoSec7W/xe2uAua+naTQamBtJQKKj6H3Ot83v7IunGOoap6xjRIe3v67yc+MdR0pLZFlG4oWRBHOjoy9b50Q50Nf9heg3iXEIbZ3h8+ow75cv4JvGxP6PJG5OEuSiozbQQ54fpxIOYtrgjy2xaa8Rb3BEXM/JWldI3I0kuBtJvV6nTp3E3emQRIcqQodF4u4koYiRarWKxA7xjwSSSCQQYA5ZBuU8o4gFSZA5yfVXyyTPLldZ/TRK0js8w725KZLJc1PkE3fY8egxcmfH7eYH/Y+R8RltIyz+aM3eeBhH3Qw3x8xwdyxG3B2XkBsy4XLkjjvJLNsySexFiT36C4QDM7+0SJboAAAAAElFTkSuQmCC',
application_view_icons=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFuSURBVDjLpZPdS8JgFIf9F6UoieiDogKllUNhM4IIKYhCRAqlIkqKmYVGERZFUZB9kBgJ9uH9dARdeDfn6T3ndbJFN7kfPDDGe55zONvrAgCXE1zuacW7nK3UQ3ufIO8yku8gbZchuFWCwOYriBsv4E8UYCr+DMLqI/hi9yAnbutYR4JwumJEcyr8FyFybZAAO3fC2NIVkEBOfkAnGVm44AJp541eFItFAjOU0QhMvBAhms2m/Uz4rCVgC+tEMDB3ygW47b+CBSaGYdjA9M+ecAF+qt9dzU5YDCmRwMLv0jiB6Zs54gJx3T62dVTq1hI0Gg2bwCNnuEBYe2qPbB0TC6zout4G0x1Ic8Fk7IELBC+Bh83F4bM5DRZal9glprjAF83zreXvOCzlrxKBUVUVqtUqPddqNdA0vie3XwHzLigTKzf0Z40uXsJw+BwG53Nsy8e0KE8oC73SIfQED9jY+6yzAqwGUUjg+DY6FfwA5i1AjZKvAWgAAAAASUVORK5CYII=',
application_view_list=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFrSURBVDjLpZO/SmNREMZ/uVxFCGy7L+AjpAgoIUW4hZjCCDZCbCQgVlY+gaDNLghibbmy2EgQTCOp8gaWYmFrpbE4Z+azuPe6UVE2ZGBgzoHz/Zk5U5HELFFZ2/uzu9ysHT2OVZUAd+SOuSEXLkdmuITMMHd+zIfn29uH/YtfGydpbal2PLdQTX4uTEVcfR7bMXCSjoOS8ePT1NJfghKAVCYONhenBtg+vCcHkAOwcwOnTRiNRkjCzHD3TwmQZRnmlgNQTOG0mSPX6/V3TJNTmqw9FgAl0tY1nGUwHA4xM2KMbyom1Uii0+ng5oUFz1HPshy50Wh8yTp59tJCWaxfwt9VMRgMiDG+y48Kut3uWz/SGAKQP5ZEq9X6lrmMGGJhofCyclGhvyb6/T4hBEIIxBgJIXyaQq/XQ6WC8vJqHSCh3W7/1z/wQlH6cHd/vvM7bLgZbv86biWzhNyQFXvhTsF5DlCZdRsTZoxXOgYqlSAcLRcAAAAASUVORK5CYII=',
application_view_tile=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFjSURBVDjLpZPdS8JgFMb9FyUpieiDwoKklUNhUwIJKYhCRFrah1FSzDRaFGFRFHbRF0mSkJX3QTddeKfz9J7z5mbtIpoPPLww9vzOw9leBwA4OrHDOaWOL2jVWnDnDeRt5vQLSJvPENgog3/9CcS1EviSRZhMPICwdAfe+A3Iyasa5ggQyVb1WP4d/msheqkTACfbsWf+Agggp1/BjoZnzzhA2qrQg3DBNErQPIZRy0XTqMHIyTeALawFSJV+AhLFKJ3NZpOCuYoJ6A8fcwBu+68Guq5bGvRNH3EAfqrfwoktY7jdjUaD3ukNHXCAuPpomYhBUFyGMfhZHjWMcsv7HCAo9wYgVVKMyhTOiHTiVAp+rBgAlz/LARPxW0sDqtnWoF6vWxp0iRkO8Maubf0HTp8Krbugji0W6M8amTuHocgpDMzk2ZYPaVHuoAY90h50B3Ks9i6brALLoFUCdHwbOwV8AQoMLgCS+2NeAAAAAElFTkSuQmCC',
application_xp=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAE8SURBVDjLpVM9SwNREJzTpx4JaGMn/oJrRMtU/g2xsLD1F/gDbK0lpaAgNmJnI1YWasBOELs0QgQDfsQ4Mxb3vEvgipwuLAsLszszb19iG/+JsHf6dDU3g9WXdzdtABIsAQZowjJkwSRkwyQoYX52+PYx8F0w0FrPFqfuuwP0P1W5ZW2hi9vXpViXsXOyieOtw+b1zUMr2T16tGnYBizYhqR8a2QjquxRkAjJsIhgGhsrg4q9CYDpmGWMerZ//oxgC1mW/clAnl0gIE6UqvXbLlIqJTYaDeibCBRrAX97ACAKwXIt4KgHEhEUGdQBlgOE4Khd0sTAMQZkzoDkxMBiAI1g5gzSNK39jJYQJKHT6SBN00KGpDFGVfJ6vR5kIyQetg8uh9tiH+IIMNb8hPOzNV2cuATAX+3kv9/5B7T5iPkmloFJAAAAAElFTkSuQmCC',
application_xp_terminal=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGNSURBVDjLpVM9SwNBEJ297J1FQBtzjQj2dgppYiP4A1KZRoiFrYWt9rHyH6QUPBDTCimtLNSAnSB26YKg4EdMdsd5611cjwsIWRhmZ3f2zZuPVcxMsyx9fPF0NRfS2vM7lx2WtcQiJHvDRvZMluXMGNHstJH7+Wj09jHkOy1+tc3VxeC+P6TXT1sYZX2hT7cvS6lepv3zHUp2T8vXNw81dXT2yGwEGeERSbSVCC5qysYa+3vm9sJGmLFojceXJ9uklCqUIAic5G3IytahAAhqqVSiwWDwx6nogW9XKhWphaGAvC50Oh1qtVr/7oAdCwBQwjB00mg0qFqtUr1ed3YURZM7X7TWTqM2Gm3CASRJEur1etTtdp1DnrafFtJGMbVNGSBas9l0DrAzR6x8DdwASUB0RqNNGS2/gH7EInvCwMhkZTnlnX0GsP09tJER0BgMoAEAa1rETDIQvBkjBZeHMIjjuNB5Ggg0/oZWPGrHGwd7Fp9F2CAlgHKqf0aYXb6Y2mzE8d/IfrXVrN/5G81p6oa2mIEUAAAAAElFTkSuQmCC',
arrow_branch=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHYSURBVDjLlVLPS1RxHJynpVu7KEn0Vt+2l6IO5qGCIsIwCPwD6hTUaSk6REoUHeoQ0qVAMrp0COpY0SUIPVRgSl7ScCUTst6zIoqg0y7lvpnPt8MWKuuu29w+hxnmx8dzzmE5+l7mxk1u/a3Dd/ejDjSsII/m3vjJ9MF0yt93ZuTkdD0CnnMO/WOnmsxsJp3yd2zfvA3mHOa+zuHTjy/zojrvHX1YqunAZE9MlpUcZAaZQBNIZUg9XdPBP5wePuEO7eyGQXg29QL3jz3y1oqwbvkhCuYEOQMp/HeJohCbICMUVwr0DvZcOnK9u7GmQNmBQLJCgORxkneqRmAs0BFmDi0bW9E72PPda/BikwWi0OEHkNR14MrewsTAZF+lAAWZEH6LUCwUkUlntrS1tiG5IYlEc6LcjYjSYuncngtdhakbM5dXlhgTNEMYLqB9q49MKgsPjTBXntVgkDNIgmI1VY2Q7QzgJ9rx++ci3ofziBYiiELQEUAyhB/D29M3Zy+uIkDIhGYvgeKvIkbHxz6Tevzq6ut+ANh9fldetMn80OzZVVdgLFjBQ0tpEz68jcB4ifx3pQeictVXIEETnBPCKMLEwBIZAPJD767V/ETGwsjzYYiC6vzEP9asLo3SGuQvAAAAAElFTkSuQmCC',
arrow_divide=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI3SURBVDjLpZNNSJRxEMaf1XVVyA/8iH1fKdEKIzwpSIeCLmkRUaaXSsE6tB3qYLIQeTUMCwqEKNI6hJHRNamwEDSXCENKoaKNNjLCd9PDRuX7f2b+HUSzcLs4t2Hm+Q0zPBOw1mItkbEy6RiP9LSPnuhK19x6r6nn8MDBrlUBHc8iveFsN+rkOJ2nnhy78q+4ZbCp180ri7qFbmfjjb3L9UDH+MlMq9oXznHaNuVvgYjg3dwbJOYT/UKJkAIx0ucUlLVVOVWgJabiU4jPxPtpJBJU1bMqWpuFEFQVFhakgEbqhBKlkUyStUJCrEBVEQxmwfisEyPRgLUWp58e3xgKhBJ1znaoKh6/fYRbjYOBlSscuN5g62vrQSsYjg0j9T1VPnIu9ikDAISyOFUEtITS4sidxtCy+FpDSI2CSlAJYwj68ueIQoEYQixBEYTzwqBh9xLA+Ox2SlwYXazTX4QAQBAASJlZMP6cl/KK8nMLUF5agR+/fp7Zf3V3tVDgFLv1lRsqQCVmkx5SqdQcKTMAEFgyUsvdQ82kDFSUVIbcYhcKC2/egwhRWFAEqkH84wdMTU/7Qjk6cfHV/b8AANB8c99mGukk2bq1fFvm+uJSqAo+f/2CickJEcptGjn/8tLr98s+WM3Key7v2rEuN2+0proGYhWx5zF4yeTOFxcmx/5r5aV42D4y5n1LLvhi4NNg1vMWVhOnBQAAfWarVSgUNJKdri+YFmCIB0NDECMQStpvDKz1nX8D4+Fd1+gIFK0AAAAASUVORK5CYII=',
arrow_down=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAENSURBVDjLpZM/SwNREMTnxBRpFYmctaKCfwrBSCrRLuL3iEW6+EEUG8XvIVjYWNgJdhFjIXamv3s7u/ssrtO7hFy2fcOPmd03SYwR88xi1cPgpRdjjDB1mBquju+TMt1CFcDd0V7q4GilAwpnd2A0qCvcHRSdHUBqAYgOyaUGIBQAc4fkNSJIIGgGj4ZQx4EEAY3waPUiSC5FhLoOQkbQCJvioPQfnN2ctpuNJugKNUWYsMR/gO71yYPk8tRaboGmoCvS1RQ7/c1sq7f+OBUQcjkPGb9+xmOoF6ckCQb9pmj3rz6pKtPB5e5rmq7tmxk+hqO34e1or0yXTGrj9sXGs1Ib73efh1WaZN46/wI8JLfHaN24FwAAAABJRU5ErkJggg==',
arrow_in=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHqSURBVDjL3ZHbThpRFIZ5i3kcLRYPqIgUGcDhNKBAqyKCobTR2NhiKmCstcWmBmtLPaCO4CQ6SBWVKInx0N70KbjhCf7O3ia0ZS686F0vVrL3Xvv7VvIvFQBVuOITQxfe6tj5IEPu9xW/ZxGcu2aJnAksxW9eYP42hmB5oBY48zAjJ240QoP7HH3j8xYhWgwiUgiAyxpFlTxZmL2ewvrPNBJX0wid+TF0zCsEHtEKGcbT4igWK0k8OwzBumGo0uZoeUCYuZzE0vUcVn6k8OSbUyFwyfDbSgKvShOIFsZgWTfU2K96pv5huOSm8KfvS/AXHAqBQ2CxcJFAsjwDe5YFgWkGdzCPoSMXHhed8BXs8B7YFALbVh/6Nx+RyWAzevR91qEu+Jf6XwRuecdkTSRp27YcVtaoCLE33Qn9sha6D+3oSrVB+07zO0RXzsx4chxmT18ifhqjSTcKej5qMbkfRVQM12EqILA8uRaRgnguhRE7mqJrahR0y5MjYgi+TTfsq1a0vVELVODYMVUJ/Lo0jZG8768d/1md71uhS2nBZxwYzwXRn2bxMNksqLgtoxgQ/RjOe2HK9FCrYaVLIehY1KB9oYVpnVfXnKscrMsmqBNNEm2a13ol05c7+L7SzD1gWpLNVXW8SST3X7gvtJUuvnAlAAAAAElFTkSuQmCC',
arrow_inout=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG5SURBVBgZpcG7b41xAMfhz/v2SJ1qI2nVedXtD0ClsdpIdDARBlKRaEIMLP4OCZNYJGIQl0EQiYnJytSRRA8nLnGrSy+/z1dHObH1eapLL86FVUlIQgyX91+v+I+zT2aiQcUSVGpWdQa3sL29kxhMmovPZ0Ofs09momkQOu2GzlAHi9RJSEJnsGFrewct1vU09LOIxd6m9jhbh7cxULVQqWPo/X7Pm4U3DFcb2L1xkij9imFq8z5GWiN0v3d5/fk1pUhtQimlefezy2CrTZUKS+hnkQTWt4bofuuiNhapknDh2ZnEoEElhmvTNyv+cere8ahosIjKnZkHVZWEtahZo5o1qlmjFqtmH52MBouoaLh17G7FP47cOJwYLKJB5fH5p1U9++hkLDYDGWDX2B4mx6ewSD9LmJzYy66J3ahomumrB1NbRNMbWz/Ot99fWV5ZwiL9VBbLEl9+fWHT8DhV6KnUFmmGJhgdHOXH4gKv3r9EpZ9F5ubnWFpeobOxw+jwGDG0ivJn+Q/zi/N8XPhAKTZqjz4aiovN/Ke3PUshBhOqE7ePRsUiGlTun35Y8R+HrhxIDCbEkIS/gKBKja+GF3wAAAAASUVORK5CYII=',
arrow_join=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIESURBVDjLpVLPS9NxGH7mjw3cZmvGclsFxcwQpFsQCRLBvIZJEC7LkyVdO5gnDx0i2qk/IAipyA5GBYoQkX0rWIaxTYvad9E85IbVcd/P834+HcZWKZtRz/V9n4f3eZ7XZYzB/6Cp0XB8/tzrsSeJxX8SuDg3stzZFj7S6Y0cO//g9Nt6e67NFi4tjLpFJBNuC8e6OrqhjUZ6LQ173f5AJb0zo4+chheQ8phK9pACGoKa8Lq9oMN9dPhw2wuqGLk/ZI53n4A2GtaKhdKP0tHZsblXm/da6nmjkrIjyqONoPS9VJ69sJVcN8Qz0yf7fG6fRxsN0QKfx++JJ/v7tg0xce9UTJRMkjx7KNrTHNoZgmii8HUNS5kloZLbJK9aU6mPWwQSdweHSJnev+uAO9IRgYZB8VsRIkRgRxDUCp/yOaQzGUcow2+uv5upCQzfGWwmud6793Cw3dcOUiFrryBfyM+LEkR2R+NdsRgMXCgW1/Fi0doQSih98700VQJjtAWtQb/XDwqxaq8i/yWfXLj8fODpFWsgZ+eSmWwWoolAMIBWtztISrQWolIEFaGk0rtdyEMpTlR9KsWJXM6GGAG1QJRAKL9aoEMop0KmEE7ZwbPJl7WPS11bdpyyArVA6wpZRP8ZYvxGv6EiqAQkYU2lXL/X1TN+0FSJWjRytz67Gn7i3+In2xhLsvVnPqcAAAAASUVORK5CYII=',
arrow_left=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADrSURBVDjLY/z//z8DJYCJgUIwyAwoPZHJBsS7STGABY1/9e+fvzKkGMAIiwWgzRfF2ST0/vz5w/Dw/UOGXz9/M/z6AcK/GH4CMZj+jmCD5C70X2VkgWo+KcYqrqfArcTw598fBhluOTD9++9fIP7N8PsfEP/9AxUD0b8ZVq9ci/AC0Nm//zD+Yfj19xdY0R+got9gxb8RNNQAkNyf/0CxX39QvZC5M+68MJuIAQczJ8PDlw8ZXr9/g9XZIK+BNP/5/Yfh/sJHjIzIKTF2VchNoEI5oAbHDWk7TpAcjUDNukDNB4nVjOKFEZwXAOOhu7x6WtPJAAAAAElFTkSuQmCC',
arrow_merge=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF2SURBVDjLlZLNK4RhFMXPaMzU+EhJLyUKjYUSykeRsRCWUnbKzpTd8AfYWGPNRtnJQorNbMdsJ8xiForYSFNKM2Pe9zn3PhYoTfN56m5uPef+7n2Oz1qLctq5jSZUtPlo/mQaVdRUrhlLbKWcgDPrBJ2p7fjmXTUD33+C3dtoQFUfnGB3eKB9CGoVmWwGrx8vj0IZOV0796oSCOWKlD4RgehvkaDHXuPxuibBn6I3G3auPwKringmjrP1C1+lFfzlmjQ/01UF9IiGj2g8QkRASzhtDlaPV2INGhjQEjk3h872LhjXrC0fLsQaIqAQ2UIWbaFWjA9PzPV09hxE9mfs7N6krZsg/5XH88cT/EE/woNhLEaWYEpuUt7AJYwIMulHfL7lUSy4MDQQS9Cw9i8Yz0BUMDE6hkQyifv0/Tt86CAlINT6DKhEqCUE1ytCRPtTB+li3TkwLkFLqCqMISo9rkLwkwOxWjNIFZJIXF5eQSjQkp1L9Q34mO6FTUohDAAAAABJRU5ErkJggg==',
arrow_out=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHkSURBVDjL3ZNvT1JhGMafb3G+TQqKECNFRIEDcvgXmB5IPNJmTdbC1SQ0S1xzZKXyT41TdpCOMyYtiXS9aW2uD8EbPsHV87RRmyLrdc92vbt/1/U8930/ZLYxASbpSwgz9SCin2+CHtJJwYoLgbITvvcOeN7a4S6NgTB45+cmCucvu8JMFOZCZQHpr0tYO12Ga9cKwpJz5xvIfH+GR2dxRGp+uSOs8Jxv39GKV+/gYS2OlXoSfNECMnMSRKw+hdS3BLI/Mlho3MPUR88lE+++ozlfjWG1kYJUCcNRsMCWM4NM02vf/hTgwsf+1uLpfTw4mcOtQ0G9aCDINiWmRiAdiAz+HTC6Nfi3QKx6uckjT3Pi0K1c1QPnzojahtsi3Zr2L/rfDGin5fE3o+pVxeYXRmVw3dA0Pddzfwz8Co82LFVERMuTbEyXJjGUMaqBgoBQ0Qfjmq5lWO3n9E/76IK8s4PCYHCytoDZgwhsWXPzosGNdYPszY1jTonBnxVgSuuhe6KhyfRDJGsJ3P0gQSqLDG7RBeE6PeF6Wie7X/MI5N2YLonoX+oFce1ZsXicQOJoHs68FdbNznBbAytaREthSHIE2lQPCF8cgT0/jLHtIQbD8sqEbrBuWYM+mqx93ANN8hp+AQOPtI0tirA3AAAAAElFTkSuQmCC',
arrow_redo=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIDSURBVDjLpZJPSJRRFMXPN46GWqRjZjkOhILU2lXUotzYqglmEQURZLtQbFEK2VLIlQVu2kXQIsOghDaFZS1yI4Vhf3CcwWSgNlmRY3z3nPu10GxEF2UXHo97ee/AuecXRFGE/6nYvzw+M5LpO3XnRNmWBRjqNI03S2dBqYXuZ50pp2ckdYhqE1VPCjKBFBprknAKc4XcjbELj3vWCXQ/7TwoqTdZ1ZSurUygurwa8VgcigS5w11gJJiIN9lpZD/ODTy59KI/DgBd4+dSLu/dnziQbqjeg2UWEQvKQBe0ejzSWm9G0FgBAHEAEJVJbm9K11ftBp0ISWQ/v0P+Ux5rFoxo3JWEJMzN54Ynrry8XCrQsXNbDYq2BMkx/nZ8QdToyNmxi6ULax88PC3j1ET/ZNe6FEi1VZZXIUAMhS8F0Ljh80oKvGvG86WzOADQCIoIggAmgiE3jfH51cmBTUFiqKnFH4tYtiISO+pgxsyx60eH/oaNIIoinLx9vKexNjnUsrcFihxLy0uYnZ9FfiEP2h8ORK30EmaGPwRrFsw4mivkjlSUVaTrEw0IEaK1uRXN+1rgkeDuoAsOh9zx8N7Yegv3Ox8tWMjBV+9fP5jJzuDb1+8o/iyu7EOCuaBI4CpQojZHuf3aoRRNGZIdMrWRqpMpJgqS4/ftcuRuzQcbBLZSvwCJx2jrjVn/uwAAAABJRU5ErkJggg==',
arrow_refresh=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI/SURBVDjLjZPbS9NhHMYH+zNidtCSQrqwQtY5y2QtT2QGrTZf13TkoYFlzsWa/tzcoR3cSc2xYUlGJfzAaIRltY0N12H5I+jaOxG8De+evhtdOP1hu3hv3sPzPO/z4SsBIPnfuvG8cbBlWiEVO5OUItA0VS8oxi9EdhXo+6yV3V3UGHRvVXHNfNv6zRfNuBZVoiFcB/3LdnQ8U+Gk+bhPVKB3qUOuf6/muaQR/qwDkZ9BRFdCmMr5EPz6BN7lMYylLGgNNaKqt3K0SKDnQ7us690t3rNsxeyvaUz+8OJpzo/QNzd8WTtcaQ7WlBmPvxhx1V2Pg7oDziIBimwwf3qAGWESkVwQ7owNujk1ztvk+cg4NnAUTT4FrrjqUKHdF9jxBfXr1rgjaSk4OlMcLrnOrJ7latxbL1V2lgvlbG9MtMTrMw1r1PImtfyn1n5q47TlBLf90n5NmalMtUdKZoyQMkLKlIGLjMyYhFpmlz3nGEVmFJlRZNaf7pIaEndM24XIjCOzjX9mm2S2JsqdkMYIqbB1j5C6yWzVk7YRFTsGFu7l+4nveExIA9aMCcOJh6DIoMigyOh+o4UryRWQOtIjaJtoziM1FD0mpE4uZcTc72gBaUyYKEI6khgqINXO3saR7kM8IZUVCRDS0Ucf+xFbCReQhr97MZ51wpWxYnhpCD3zOrT4lTisr+AJqVx0Fiiyr4/vhP4VyyMFIUWNqRrV96vWKXKckBoIqWzXYcoPDrUslDJoopuEVEpIB0sR+AuErIiZ6OqMKAAAAABJRU5ErkJggg==',
arrow_refresh_small=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGMSURBVDjLY/j//z8DJZiggtx9Sasyd8Yxk21Axo7YSymbow4QZUDJ8QyHoiNpB/IPJP/P3pPwP3177P+mQ5X/6/aV/o9cFrATrwHFxzIcCg+nnplzacr/TbdW/19/c8X/tTeW/l91bdH/5Vfn/y/ZkvPfb7rbHZwGFBxKnTn9fN//jTdX/W8+XPU/cX34/5iVQf8rtuf/L9mc/d9nqutuvC7I2Zv4AOjf/0D//o9fG3YIJh4wy+OS9xTnQ2699kyO7VacRAUi0L/wUPea5LTGtceW9FgA+ncNyekgfJEfZ9AcTyagfw+59ztcgolbVBsdMi7V/a+Xr/lfK0v1AV4XAP27O2tl0v/UJbH/rRtM/5tVGf6PmB74v/dE0//khdH/VVMUZ+I0AOjflxnLE/5PP9v7f8rprv8TT7X/7zvZ8r/nRON/kLhKssIZxXhZB7wusGu22Bk3N+x/1Mzg//qFWv+1s9X+q6cp/1dOUjigEIeqGWcgAv17AOjfS2RnJt08DWbNTNVVVMmNhDAANau2t3wToKQAAAAASUVORK5CYII=',
arrow_right=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADvSURBVDjLY/z//z8DJYCJgUIwxAwImOWx22uSExvZBvz68cvm5/dfV5HFGEGxUHoiExwVf//8Zfjz+w/D719/GH79/A3UAMK/GH4CMYiWFJJk+PXrN8PN27cunWq/oA/SwwIzyUrYluHvP6AB//7A8e+/f4H4N8Pvf0D8Fyb2h+HLl696WllqJ69Nu2XOArMZpBCuGajoN1jxbwT9FyH36/dvkCt/w10Acvb+h3uxOhvoZzCbi4OLQVJSiuH1q9cMt2/cvXB7zj0beBgQAwwKtS2AFuwH2vwIqFmd5Fi40H/1BFDzQaBrdTFiYYTnBQAI58A33Wys0AAAAABJRU5ErkJggg==',
arrow_rotate_anticlockwise=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHySURBVDjLtZPvT1JxFMb5O/gHskVzrrV+mFZomDdEDGkYKSXlRleY6IzcFdQ7lBgYeaELBNjFEpbWi9psRU7JnCa3VYTV/WfY01davkFk0/XivDp7Ps/Zc86RAZAdpmT/BWDLmun+5ZuS5X0P+paMML82SKZXeroqYGDttty22it6Po8iWeCxIAlI/5pF9Osj3M8MwPCsXex8ekVeEWAlYn+OxaKUxNx2FKmfcTzfjiH2ncNsnsfIOzu00RZxT4B1pZee3GTw4vdfVyEfxkTWAdfyMMJfHiL2LYgImcSyeAstgQt0GeBuxiQl8iEIP/iSW/eCrtiV0rLXkm3s1ThVnN6cQkj0w511osl7TioD9L29QcaNY64QhWvlHrrmtey/niasclCcEqrp81B669HoPo0yAEmaBBcpuTOZQegF9S6gdUaJqms0vdRL3JYXQdEHLueD9snlovpxc2qnd8nfiIues9gXYEx30INLFvAksB1IIPcAd9LdaPY1oEcw4HqiE2ecJ7DvHegSlGh/Y0FgywP3uhPeDRae9TG4P7nArjHQ8W2oG1KgIkATUcmpYJNonjeC+TCMyZJwFOMfR+BadaCdo3DcdhRVT5kkTZOkC/VjJ3GKqUNHSA3NTCsR1+BAz1RrPwaFtQYH/kZF/5GKa/wDDtK86rC6fMkAAAAASUVORK5CYII=',
arrow_rotate_clockwise=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHsSURBVDjLtZPpTlpRFIV5Dt7AOESr1kYNThGnSomIihPoNVi5Qp3RgBgvEERpRW1BRBAcMEDUtIkdjKk4otK0Jdr2vgxZ3kA0MYoaG3+cX2evb529zt4sAKz/OawnASgCBNm5LaE7vjVDutkA4mMdLV4TkvcCuvba2Iqd1pDhWA33mQU+2oXVv07YfpoxuNWFuqVXoeqFCnZcgJwRm04p+Gk3Fs9t8PyZx/K5Hfbf03CGLRj62g2+rSR0K0D+vZXUB1Xw/ou5usJWjAaU0Gz3w/rjHey/ZjDLvKTD34KSyXzyBkC2JaYd4feMqyNa3OQTREQePlXjrqSq5ssj5hMjTMd66ALDKDLm0jcA0s+NID6JIFmvQaNXANEKX3l5x7NyqTcb7Zg8GYtCOLoXuPcbha6XV0VlU4WUzE9gPKjF2CGFbE3G3QAmafDnShETF3iKTZyIblcNza4Syi/deD6USscFCJwV6Fwn8NonQak5Hy1L9TAcjkJ/oAG1p0a1hYdnfcnkrQCBoxyyNYLp1YCJoB7GIwqGgxGod/oZsQoNDiHSepNCceeAN8uF1CvGxJE25rofc+3blKPqQ2VUnKxIYN85yty3eWh216LeKUTOSCayVGlIH0g5S+1JJB+8Cxxt1rWkH7WNTNIPAlwA9Gm7OcXUHxUAAAAASUVORK5CYII=',
arrow_switch=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI9SURBVDjLpZNBS9RhEMZ/u60aZAdNSXdLrcxNS82DaRQVRBCUGngwwkOnvkB0yEt0qy/QKSrq5DUSQgLTSi01d80gcrXSTTdTViTU//+ded8ORihFYD4wl+FhYOY3T8A5x2YU3Ij54qOmp833zmX+14CWh431vm9OGs+8W9sPXOm49HsHqxarFhXbZ9W2EQxeECNnxUh0W2Y2kdwIcwtzJCbHY8+uvagBCAG0Vl3G4XDOYZ1jbPbj0ffJ0S6xQrT4AFszsxC1qFPycvJYXl45fOxG7ctXNweOB51zWBzW2V+l7MnbS21JLemFNBmhDIwIxhqMGowKxgjGNxkAISuWB2/uoqIE7Rb255dxMHKInO07CLkMxpMTpOZnmE7NEN4ZQUVITIyPDNyK1wEE1mJsud+QLUavl4cr2o5E64glhumJ9ag629TV1ttRd7VGNWQ/Dd6Ol/6VgguCDTjiYzGWvCWiReX4Pwxe2gPAX/Lx5rx1dAKt7c1OjCBGcOIoyC1kMb1IWTjKvqJSJqbGGR6Nk0gkOBitQMQyNDg0kmj/XA0QMr7hRPkp1ClqBbHKXNY88Q9xineVEC6IUFgQwZ62qFUsFm/Fq9p9Pvx66sl0XdD46y8sKiwuLZL6/o3nvd3Mp+cRJ4gVxCliFRFFjBqAQMOdM06MYHxB/FVEYqRPPG3z0/7qI/kazc/Pp7K6kuSXJEP9b2MznbM1f1D4l4oaI/Uq2qViJ1Ods9ENZ2Hy8dd+NdqtRivXUdhsnH8Cn6RstCM01H4AAAAASUVORK5CYII=',
arrow_turn_left=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGWSURBVDjLpZG9S5thFMXPG2NUxE8oxFAtKFnyLwiCHaxOnToodmoVh0qJFBVcRXQLuOhWLLQoWtsIfkCzmNLioA52EYz64mBKFAJKCXnuuU8HWykaS3i92z3Dj/O717HW4j7juxm8+TZQMvS1f9QzgNRZUnuKBTj/KkSTfbGG8tBrVYWbdUEqKMzQcFuEGzRc+tD76aQgILrZNx6sCI01V7XAcQBahaoiJzlkf2WRzv5E6jT1mUamlvvXv99SIDVAEgqFKEESYgU+x4fyQBnCwTAiDyNPjZGRzlh7Y0GFgbXn08HKhlck4Z65ECFC1SE0PXiEUn8AqsRe6gcO3IPol+Fk7NYRZ7reDbrn7tvjjLs392zRed+97Bymj2KJncTJZe4SF/kL1FbXwhh5cucXxMhLMTL/d//4YjVq8rK0f7wPv68UdTX1kLx0FlT43zyebLUdbR2gKuKrcWxN7DoA4C/23yYvMBSoVYjhdV40QIxAlLCWECNeAAT1TwPx2ICWoCroTYFXCqqglwYUIr6wAlKh1Ov8N9v2/gMRLRuAAAAAAElFTkSuQmCC',
arrow_turn_right=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF7SURBVDjLpZI/S5thFMVPXt9aakBjIYJvyFycnZw7KDhkcCi41BK30tpkqA5d/QClLv0IKuIQB/8Mje5dVLCCxhQtkg+QFnLPuU8HqYY0Q4h3fLjnx+9ynlQIAY+ZqNfFxcr8ypvtVwN9A2icp/Fr53uq84SlajEv+ZyoacknRWVlwvhwDk6h1qh93lzY+dAV8L5anHL6cpLOFTJPR5F+kkYcxfDgoAt04rR+gtqvq9XK24NPABDfh78V85KWX2QmCmPpMfyx34iiCHRCLngIkAtGwoyD/3L3AFFzyVBSyA5lQRdMwtntOX426qAJyfMEpHB1U1vbLVU//gcgNT08mEHTmqCEox+H1zRubbyulABg9svLY5q+75Wr77q2IOPMyLMRDKRiXDYu0B4GADOum3Gxs4UHAxMoQsFRb9SxubBTal/cLx+udqu3DUAwEJKDLfb8E+M2RRgFQTDrA8CW7gxc/RnQCBPhoU8DaxF0wh9jsH+0d9cGewf8BRKi/IUanjYRAAAAAElFTkSuQmCC',
arrow_undo=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIJSURBVDjLpVM9aJNRFD35GsRSoUKKzQ/B0NJJF3EQlKrVgijSCBmC4NBFKihIcXBwEZdSHVoUwUInFUEkQ1DQ4CKiFsQsTrb5xNpgaZHw2Uog5t5zn0NJNFaw0guX97hwzuPcc17IOYfNlIdNVrhxufR6xJkZjAbSQGXjNAorqixSWFDV3KPhJ+UGLtSQMPryrDscPwLnAHOEOQc6gkbUpIagGmApWIb/pZRX4fjj889nWiSQtgYyBZ1BTUEj6AjPa0P71nb0Jfqwa+futIheHrzRn2yRQCUK/lOQhApBJVQJChHfnkCqOwWEQ+iORJHckUyX5ksvAEyGNuJC+s6xCRXNHNxzKMmQ4luwgjfvZp69uvr2+IZcyJ8rjIporrxURggetnV0QET3rrPxzMNM2+n7p678jUTrCiWhphAjVHR9DlR0WkSzf4IHxg5MSF0zXZEuVKWKSlCBCostS8zeG7oV64wPqxInbw86lbVXKEQ8mkAqmUJ4SxieeVhcnANFC02C7N2h69HO2IXeWC8MDj2JnqaFNAMd8f3HKjx6+LxQRmnOz1OZaxKIaF1VISYwB9ARZoQaYY6o1WpYCVYxt+zDn/XzVBv/MOWXW5J44ubRyVgkelFpmF/4BJVfOVDlVyqLVBZI5manPjajDOdcswfG9k/3X9v3/vfZv7rFBanriIo++J/f+BMT+YWS6hXl7QAAAABJRU5ErkJggg==',
arrow_up=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEGSURBVDjLpZM/LwRRFMXPspmEaGc1shHRaiXsJ5GIRixbCr6SikxIlqgJM5UohIiGdofovHf/PZVmYwZvTntPfjnn3txWCAFNNFE33L/ZKXYv+1dRgL3r7bu0PbucJp3e4GLjtsrXGq9wkA8SU7tPk87i/MwCzAyP5QNeytcnJl46XMuoNoGKDoVlTkQhJpAgmJqcBjnqkqPTXxN8qz9cD6vdHtQMxXOBt49y5XjzLB/3tau6kWewKiwoRu8jZFvn+U++GgCBlWFBQY4qr1ANcAQxgQaFjwH4TwYrQ5skYBOYKbzjiASOwCrNd2BBwZ4jAcowGJgkAuAZ2dEJhAUqij//wn/1BesSumImTttSAAAAAElFTkSuQmCC',
asterisk_orange=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKKSURBVDjLpZNdSBRRGIbnzOzubSxBRReBYhTRDziQQlKxbmoKItp0YVRUsBB2UVQsWdkfilHaj6GuZqEkhJaSf6knISqUYIgooogWS2uRwjFd25yZ3Xn7NlKS3bzp4jDMzHne73zPfCMAEP5nzbux6gU5UifwsE+AWSMos89DVczz4xpD8ArjkxUsMW4AwZ7InSWwetJh8Vzo1YzPviNYjfTmQL8rY+KSqI1fFJWYAKrsjjSvgPV4F/DsAGbqFyF0nSVOX2Xu0M3lwKMdCHdlgGDtW5kox23BqGFes2UdBeyD2ZYKgn1Tlcynt6YAPB/TDUkg2PNPB9H1s4pxozWZTlIIgjX9XipVL0CoaW0U9sVINGsF2ahm8l/9OkmWZg3shNWXC/TnwnzgwtdSUR27IDpn942cluSPxZIsRGXpt5eCTINg7Y9pNdy1DejbDjzMhNm+BQSrgXMS/1wi+UdOSQiUOeH32rgwc4PxSH8eMFSECC+A2Z0Ns5PAgXygNxPoTqdrFoz2dMy0bKLTuCk0B6HmjXh3hALINCdZCFYyTFaIKn0mTqa50baZNmZQgAvG/TSMlkjqp5MSHz4h+T8ct+HtYRteFdl5jMTxctFJsjSrLw/hDtfvEL01DQSrBDsXnMToIphPN66H0ZGJL2ckf7ApGejJglazCu+P2XwLBpDp8smG1dS/gonalSDTHjLtm7q1AehyIXA5AS8P2r1xAwhWvtcm0Bjn08Rlg0xrBDvJtHukdBnQuRU6SXxzdDGG9jpiJ3HsvKgEzkpasDEZE3VrMFwszVV6fciuTjWmYLQ8CYN7HNrTQocStwUynUiyWkgWJ9Nzf90Lj115vt/BB3c7vE8KHfNE/gKM7aCNx0eNYwAAAABJRU5ErkJggg==',
asterisk_yellow=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ5SURBVDjLpZPNS1RhFMaff2EWLWo5tGnRaqCFRBAM0cZFwVSQpVHNQAWVMQwaSSZWtimLiKnsO5lEjKzs4y1zRK3oItfMj1FnnJkaUtNrjo45H3eejpCKNa5anMX73vs855zfOS9I4n9i2SHbCpvph8q8A9PNcCzcz76EM9EETj+DmmqENaeBiJ3mRyuzQy5mwyVMKqiFbzNN0MxgKZOd2zj5GMZE/ZL5ooHZAntGW89s7Bw5Ws25llWcfQHrzHPYE/51ZOQ0M4Fiitj4UQdbzhZSb+FJ63ZypJqp7p0UsTf+FN6kvoMMl3GmNY9jj+BckcF8/HoFldLzpZIqxhthJPVdkr2cifdb5sXefyAKLFvyzVJJAssisIxstILZ0DEyeJzpHifHfNBGamFZ+C9yC7bhG7BBxCrZZqWQpoiNP6S1TMBFDh4gA0VMdxfy+0NosftQX+8gGKkBY741HLoGhbnXUOZwKTn+gGa4nOlBN9MDxdJzCTmwj+wvEKPDTPUc5Zx+kOk+NxmqZOJTIXsviYGQVgKLAos/n0CbbIAS0ir1eY9kF4O+3UzpBYzehhaugQpdR3DwKth7EeyqEoO/oYzXwyKwDDN0ipme/VKFi0l9L8M3oYW8SwxWnIKI1XT7Vqb6i/ntLoLTHdulhROcUJsZuJJjCsvEPpyf8m8io5U0VB6FtFNIe6da84XFEcYaNrDzLDw5DUZ9cEwqm6zxGWYGPBTShogtQtoerV0rLA5JKy5+ubya7SdzbKKMyRG7ByPeIfvebKfAWszUdQFavKOI0bqNbCuF4XfneAvzIaStQrpOxEpIL746rQKOD2VQbSXwtLiXg/wNTNvAOhsl8oEAAAAASUVORK5CYII=',
attach=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEZSURBVCjPY/jPgB9iEVoqPefllFPdNk2GWBUsVpz9ctL1rkcNW/v+59VhKFgkPfP+xI0dF+uC/jPkWCR/Q1MwX2TGvf7Nretr/UG8BO2I5ygK5olP/dCzpWV+dVAhd+bB+JawrT7ICubIT3nbvaFpVkVqgVDa0diO4CneN91E4Qpmq0560jW/YXp5XB5nyq2YrqCFno9cJeG+mKk48UHHjLruMu8czuSbkfUBizxeucrDw2GGev/71uW1jMVrsq4nPIto8F/g8caFDymgetxbHlVLgDjxnWExPjPdb7sIoYRkk17FywJRECdY1Xux201nMbSgLufO25qyJUY1yNrzsus9JxkscZHMG+kVcN7jqWueowARkUWiAgBEUvolGfpITwAAAABJRU5ErkJggg==',
award_star_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALnSURBVDjLfZNvaNR1HMdf39/97m63nbmdf3BpRrqYt9G2SEJEDFZCEkhCIj4ZDhFpIT7wQUKINCQz0Ee1BxIJjWXPlBwhjdK2dW02/HNTp7FIZa1zu+08b57b7/vn04O1MQR9w4fP58nn/fn3/igRAeDdo93SvK2K3+9N4oxgrcNah7MOYxyb1i6hreMKA20fKBbAnwuctTgnGP1/opn11syaMYI1lqfhzQXz1bTFBJZkuI+mFz5DBxat7Szh8wicsTwual5cFEUHlsbSc6xP/MUKuc1Li2NMFWew9rkd2NY/Bh+wNOpTW/GINZE0XqKRd+I/s7wsQv/VEZyxrc8k6D6+9ciNP8db+9MZ3iy9iF+5A698I7VeF5ev3WPwTqZ1qGP3kacJVNC7OiYSqhYnISdw+LemffvqLuxdXbcXvAjB/e9p7wmd31mT/lTEwznB6uJ45Y6x+wBq5tKKESlNrhQBEUGc4FVshtKXEZ1DdB49kcYWxxAUTmuCwl2CyWzTKx9Ku28mMi0uk+kI1bTE/ZW7kCCDC3I4nQVbBOUIL2sg4i/CGU1+4DtyA9kfiHIWQIkI+XZV5SydkXXbq0Or3keCB4h5jLgiYp+ACvHLUIbufy0FrSkU84WHU5Nf/Hjs+lE1p8TcGRV38G2s/uPtKprAmRxii+Cm6fpbMzgDbyTXsyrxKhdvnqPvRg93h7Mn5q9QsUumVKysS+eHEQWIQ6FALJ1DIzSsq8d6lvrKLVil2fDaRoCPvIUnEUreC1c0oJSPGEMwOYoKJxh7lCWs4mxL7gfg4NunWLu8DqBk/hcmzi6NOaExVF5D/vIJJlKpYRS9ZdVVuwtTM9wcTXF9tJdDW07z+U/NlISiANMLlGjeUvE1sfHOg4xfSp0ymteTx6T54e3hrZtUvtCXThHB5/xgGxHl03vtV4Cv5pf4z9dqz/QYn0xnaak9KRcWjtZ/QC3+5kl5z61wSa1WygMKQFvfl6OH/gNPtZHfxghrXgAAAABJRU5ErkJggg==',
award_star_bronze_1=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJvSURBVDjLhZNNSFRRFIC/N++9eWMzhkl/ZJqFMQMRFvTvImkXSdKiVRAURBRRW1eZA9EqaNOiFlZEtQxKyrJwUS0K+qEQzaTE/AtLHR3HmffuvafFNINDWGdz7z2c7+Nyzr2WiFAIffaMBDW1+B0diAgYgxiDiCDG4DU1QfcLos+fWAXGYUGIUsXiAliUFER+sBAhVCIIVB7QGtEat1oTbcwVz2LMfwR+gPg+oY0bEa3x6sHdUoVdniMUj0M2i/j+PwVJa2QUu7YWp34D7mqNWdNApD6Ks24dpvcL4gfJRQXevbutjI4lGRzCS9iYukPo5dvxVqWQvn6k/2uyoudd60LGEhG43VBGyI4j2ADZ7vDJ8DZ9Img4hw4cvO/3UZ1vH3p7lrWRLwGVneD4y6G84NaOYSoTVYIFIiAGvXI3OWctJv0TW03jZb5gZSfzl9YBpMcIzUwdzQsuVR9EyR3TeCqm6w5jZiZQMz8xsxOYzDTi50AMVngJNgrnUweRbwMPiLpHrOJDOl9Vh6HD7GyO52qa0VPj6MwUJpNC5mYQS/DUJLH3zzRp1cqN8YulTUyODBBzt4X6Ou870z2I8ZHsHJLLYNQ8jusQ6+2exJf9BfivKdAymKZiaVdodhBRAagAjIbgzxp20lwb6Vp0jADYkQO6IpHfuoqInSJUVoE2HrpyRQ1tic2LC9p3lSHWPh2rJfL1MeVP2weWvHp8s3ziNZ49i1q6HrR1YHGBNnt1dG2Z++gC4TdvrqNkK1eHj7ljQ/ujHx6NyPw8BFIiKPmNpKar7P7xb/zyT9P+o7OYvzzYSUt8U+TzxytodixEfgN3CFlQMNAcMgAAAABJRU5ErkJggg==',
award_star_bronze_2=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKFSURBVDjLfZNNSFRRFMd/b96Mb0ZHEbFPTfqQNKpFkVZYFO1MSFpEkBAYhBht2rnKHIogCNq0qIX2QdHOKFHJooIKlDIrykxLM0VzbMYZx9F57360GC0lpgOHc7j3/H8czrnX0FqzYKefVOuDm8rpDr1CaoVUct4VQglKcvfQdP8W7WeeGQsaN4tMOAKlk8VSSYRS8/Gv27ZYLMG1BGALtAZHSmwp2BgJc/T7V2zpIOT8XcJJDXBsgUU6lsuLIwV7JyfZLvysjEawTB8+7cex/w8IfBrqZY1VyAphsn5mGrW6jLLxMOu8RXS/68GxRSAl4O6x5vrnn54GJkZCHJgxYMNhZG4JJeEwE/1B2jpbA28uva9frDG01nC7zIfLLEJjApyPumpqsq2TWVtqkY4b60cz9/peP6xaldNAsgTEXJATL4aTgJulI+QU52kM0Bq0Qi7fTcKdj4pNYooprHg/xlwo2bR0IDaGKxo+ngRcXHMIoe+o/bV+WXgEFQ0iopOo6SAqPoW2E6AVRlo6JgL3hxa8gwMPyPBUGX8e0tm8QhQtamdlUaKgEhkeR8bDqHgEPRNFGxpLhPC/fSyJiXqaxi8sHWJgdAC/Z4frc3uze+ojWtnouRl0Io4Ss7g9bvy9T0PYunxB/M8WqBuKkZ3V4ZoeQgsHhANKgjMf09wxro12pFwjAKa3QmYXJ1OPwGtGcPmykcpC5iwroKF4a2pA4y4f2jgg/Wvxfmsj81HjQPrLthuZwU4scxqRtQ6kUZEaINU+mZHv87SeI62r6zpCb+PqSLVnbLg8o6d1VM/OgqOXAJb8RiJTeeaX8UF+2ado/Nn+5/zyUDt1RZu9fe+vICldLPkNkalgK6EwqgIAAAAASUVORK5CYII=',
award_star_bronze_3=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKESURBVDjLfZNLSFRRGIC/O3Pn1cyUlLXIlB6SQrXo/YSiRSQDualVUFZE1KJtrTKHIloEbQpqYUW4DipSehBIYWr00MIs0ckUR6Z8jqNz7/nPaTEqI2E/HM6D833n5/znWMYYZuLglUZz4lApTT+H0MogohHRaNEopdmzZgm36z7w/vZha4axyQstgtYG5U6DKteLyjWlDKIkH8GTP5k9zRWUI6xzP3PKuYvrCK4rOeH/BFoJExmX5dEAriMcMK/YER6gaKqb4kUh0pksIv/NQOKt7YMUBmzWRydYa36gl+8mZjWxLOyn+WMfWkl8XkHj9YrqL99T8ea2JLtohTWVSOFWNjlNtHz6SXtnMt5RV1Wdz1jGGHi4O4THW4bBC3ChM3bm/Op3pws3H0dcm8CvRzz8oJ9UlSZqyG0BNZXi5JvenODBtj4WlxcZLDAGjEaW7SRrr0Cnf+NVIwQyP7CmhnJJiwvpATxjw8dygmvFh1CmTu87G5HSI+ixFGrsN3o8hc6MYJwsGI3lX4AXhd3+lGBP12PCvqPW7EO6VFSK5qneXlmWLalEhpNIZhidGcVMjGEsQ0ANEfn4Ukirau4lr869xHh/FxHfFs+3hkf2yFeMdjBTE5hsBq0msX02kY7XQzimYgb+pwpcTKQpWPjCM57AKBeUC1rAne79dpo7/S/mLSMA3mBMCspzQ58i6B3FEypAdABZvLSEmvIN8wtqd4Qw1n6JrCTYXU/0eW3Xgrf196OpZgLecdTCVSBWbH6B6L0SXhHyPbuMv6XlLsps5FbfCd9Ab0X407N+MzkJrpkjmPMbGR0p8n5P9vDHOUftYMPs+o1EAxfL1gU7224ibMtH/gIKIWcO8vV/HwAAAABJRU5ErkJggg==',
award_star_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALjSURBVDjLfZNfSN11FMA/3/v7ea/36prXWWirTZdlKtiC6CGi4BLVCqJB2/BFNmKMfInVQw8RK4lVD/VoEL2UWL0Za8HQJHNL5ya17lZz4aItXb957/V6r9v13t/3z+nBNBm0A4dzXs7n/FciAsCz747LgRdambyyiDOCtQ5rHc46jHE8ft8W+gd/Yrr/JcUG8dccZy3OCUb/G2hWrTWraoxgjeVWiaw569m0xYSW9qrT9NxxFB1atLarwNsBnLHcLGnu3hRDh5ZU4mseqb9Mo8xw7+Y4N0oVrL1tBbbv7PnrNMR8OpNFdkTTROpTPFU7yl01UaZ+nsMZ2/e/gPEPdh258Humbyod8Gjie/ymPUTqHqMzMsKZc1c4fynouzi4/8itABWe2hYX8drEiecE3vqx59ChrhMHt3UdhEiU8OpXDJz0vtnXkX5HJIJzgtWlTNOehasAqjLWOCeJ9q0iICKIEyLJJyCxHdF5RBfQuTS2tICgcFoTLv9JuJjtaXlFBnyTC3pdEAx6Hb21/tZuJAxwYR6ns2BLoBxVd+4k6m/CGU1h+gvy09ljxBgCUCJCYUC1Osvx6IO727x7XkTC64i5ibgSYldAeQTfTRP88AuVpbJYT+aA/qeHzftq7RLzX6paB5/HH3pjt4rV40wesSVwZTKT8yxfyvJAai+xlk5W0sP8dnLE5mcuHl7fQrJbbqh4zYguzCIKEIdCgVj+Gp3k/tReqi+PoT7rJvHHEM3bt3ii5LC/cSVC9fNVyZ0o5eOMQS9dI9rQjM4WqW5sgede++8H3m7Cc6p5HZAbaog7IeXVdVA48yG5iYlZFKdq2lr3e3UJSue+peZYL5WVgBKwXPSwHtc2XKJ5UtXuiGeOv05mbOITo3m4/T05sDQzu6u6nuVfT4/zdzlO0asiv6iYnVdWoH99iPOfqpfLC7xZztLb+ZGc2Nja1KtqcyWTHC3nil2eUb71ZF7g42eGzdF/AONCmjwFNk4lAAAAAElFTkSuQmCC',
award_star_gold_1=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKDSURBVDjLhZNfSN1lGMc/7+939ORU1C0iszwSG2duOFqNBQ2meJUoY4MixiiKiCCGMMauZYJ00252uYIuRlsgKJRsDLE/VBIhu3B/knVmInpy/jnrpB71fX+/99uFO3KkrAfei/fh+Xx4eJ/nNZIoRtx9Vq6xCTs0hCTwHnmPJOQ9ya4u+PYHKkdumSKToCQURVvFRXBLUhRZV4oQbBO4aBOIYxTHBOlVyt6Y3brL+/8RWIesJdi3D8UxiWN5glfrYM8yQToN6+vI2v8U9JrZLGFTE+HLKcwLqwS72wmOGRKpFP7X35B1vTsKktev9ZD9o5epacJX1gga3iKofQ2lsmjiAXow2Vt773ZPKWMk4X5KVUhhWl6hF/j+5z5U2+IHZUcvQFCOnf4SNzD+tVrDi1KA9yJ2hYX6N+enjSTs9/Uz2tXcIPFkCiKoOw67Usg9Ri6PWxonLswjDN457PIUNrf4jpFEYcCc8I4vwgMfVSUaTiM7h7eP8S4HcQEUgUlgEtX4yJEfu0Zu7O5XJDljiouUv2r2+pih8v2n0uHzJ5F9hKJV5AsoXgMTEuWyzN3sj+0yPYcuq2/bI9a8rQxJjqz/Pjjo/7oPEtIG8hsgiyHk0Tf9ObdBRxH+xxTqTmvFVFQOu3wGGUAegwFtLhEhKy2faHjHMQKIpzrL6l7CmASKImwuiynbTVBeQ/kzzzbeuWhaSuu3/YWlwacrvGgPaw+Q/+USS6OjGQw/Vqb3vlu9v41k/UFyD+c6gTv/2kEcR62m6sWKhaHzLHw3eiVyHG7+WO/9OZHpmBz4bNauWdZX6dzesrR1Zj7l/Uwfk3fP8XppXhI/d1MzcobPb5ziXmn+b0pcjPW7AMpFAAAAAElFTkSuQmCC',
award_star_gold_2=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKUSURBVDjLfZPta9V1GMY/39/vbGdnTubEhsOcFIPjDEdCvijEwletQWRRYFIoEkFEIv4Bw4EEYm96WYFQ+ACCixINxFKokWFmakvwzESX7fGnR7ezne/T1Yu2OqPWDRc33Pd9XdzcD0YS8/bu2Z16obObS9kAQZEQwxwiPno2rtjEoc8/5as958w8J0eNeeeJ+qs4xICPcc7/A2t9LYVkgYD1SOBCwAZPh/2Dl6cvYIPDh7lc1S0u4KwnTyP5pAEXPE9Xh+hKEx6ZnSCfFiioCWf/X6Bv8NavrM530CrR7iZIlm9h49QwjzUUufTzZZz1fYsKHHm9v/f84Dd9Y8MZm6mQa3uNZNkzdM3cYfTGOKcvnOr78cCV3lqOkYT7bk1BSouKSqPgwFDr2ztWV95aWdwNST329jGO/nT5y1fa6/ZJCTGK4Crjba+O3TaSsOfbhtXYuUoCSSiKpGUzNK5B7h5yZdzkFUJlDGGIzmEf3sJmE28aSVROmBej43C67p2m3KptyI4Q7T2iyyBUQB5MDpNbSvSO8sUjZBevfUGe7Wb+kMqfmY4YOFm/dmsxffQlZEeRn0axgsIMmBSf3WXk9PFgH9Lb9aH2Lxhi8xsqkeep2d/6++ODQZCQqihWQRZDyujXxzNXpXue/K8ttGzTlCksOePKJWQARQwGFFCMkDK1/qDOLLpGANHQU9fyJMbkkPfY7C6mbjlJfTP1rSvbr+4z62vrF/zCZP+KQhRb0mXrKP/wAZMDAyUM3y4pduxYuvY58m1PkA2N9ABX/7ODEPyzpunxwvjJvYyfG/jIOzZ0vq+d96+Xum+e+OR3O2OZnaZnYcvS3xj+mF2l/dy8tofna+OS+P49ms9u59CprfxSG/8TX76Rzx39RVIAAAAASUVORK5CYII=',
award_star_gold_3=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKfSURBVDjLfZNLaNRXFIe/O//JTCePxqQiRq2CDcQkGFoQF0VaCS4MAVGoiEhDpEigi0JxXYKhthTaTRcuVOhCIgXBFCtVCLb1laoNKkl8tIyPSCoTY8aOJqP538evixiZUOKBw+Uezve7h3PONZKYs81fntPuLfX8MZonOOF9wPtA8AHnAhveeYsDvVcZPPCRmWOSlFjwnhCEsy9BN3t6N+vOCe98KUKi9PLqNetxsaex7BIdb36FjT3W+lnB1wkE55kuWpZVpbGxp7X8J9bV3mGpbvN2dYap4gzev7YC3/Pn8DiL00maa56yOjVEoraVTZVnWFKR4vK1MYLzPQsKnPumrXvk74mey0M51pf/RrJuO4lF79Oc6OfK9VGG/8r13Ort7C5ljCTsxVUZKWpQUBQEX1zs6OpqOb1nZcseSKSIH/zIkfPRzzuahvZJCUIQ3hYn6rY/emAkEZ+tG1N543IJJKEgEjUfQPkqZJ8gW8BODuGLjxCGYC3xs/vE+ccdRhLF42ZLsPRGTZ9WJpfvRHGOED8h2Dz4IsiBSWKSVQRnKQweJT84coI0u8zcIhWOmPrgOZlas60hWrEVxePITaNQRP45mAiXf0ju1DEfP6O75Xvtn9fE6o+VJc26F/f6+sLTmyAhzaAwA4oxRIz/eixvZ2ibg/83hZqdmjKZin5byCIDKGAwII9CgIiptd+qf8ExAog32stq3sWYJHKOOP8QU1ZLIlVNasnSlcP7zNrS/Hl/YbJvcSaI1mhRE4Ur3zE5MJDFcKGiob6zas1G0nXN5O/k2oHhBTbRfWgqV2cmTu5l4veBg87yXuPX2v3v7Wzb3eOH/4mfx7yYpn1+ydIrHzvEJ9n93B35nM2lcUlc+ozqM7v44Zdt3CiN/wel+5Gy/cSN+gAAAABJRU5ErkJggg==',
award_star_silver_1=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLfVI7aJpRFD7/7y8qdlDwQdQYwQYJJdCSNmSwlBQcQjK4Zil0KIUgXVwFUeji1Lk4dLGh7RCoDhYtRQoODnUqjVY0SJKKT3zg83/03Jv+YlLMhcO595zzffe8GEmSQD7CK580sztgmkgAtYsiSETwTrTq4ADg23fQfv3CyBgOFo7E8/NgGTgnkYmms0UIsNcIZvwVQBCoDDY24PzwcP6mJLcSoFOaToFdX6eAhtsNo50dGJpMwLpcAOMx9d9GEGYuLkHhcICwtQW91VXQ6/XQ2N0Fbm0NxF+/acxSAtXx+yBc/gnDWRXaSGBaWQGdTgc1ux3E0xJIxXJY9/NHcBHDkMbk83kNz/MuFMVsNgPHh08vzx67X9i2H9GgWq0G0vHH+On9zRDxT7EM1A2/31+lBLlc7lypVFoFrBtJqJDUWZaFyWQCHMdBvV6HZrNJgNTf6/Wg3W4/o2NE49FoNIpZrdY7JOVutwvD4ZAK+YAQm81mcDqdoFKpyIdQqVQ+I+6EkRcplUrdRUPCYrG41Go19Pt9CqSzRyE2EceYTCYFjAuGQqHX15ro8XhKWNvDYrF4QtJUKBQUIP5bICyRgNsYsyeD/5uC1+sdYECq1WpRAtIDohmGoSTYj0E4HE4tHSM5GLRvMBhoylqtFndnTBtK7qjt2PnNpQTRaFSDGTy12WxQrVYhHo+XMpnMu2w2C51OBxy4YFje/lICBD8xGo0arJV0+i2+HwSDweeFQmEvFotdkH7cJOBupG8tl8sVBB4FAoGkbI9EIkmfz3cvnU6/wR3YXsT8BaM2jDnYpij1AAAAAElFTkSuQmCC',
award_star_silver_2=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJwSURBVDjLfVJdaJJRGH4+dXyagylpDj8rhwuRfoXqqogCL2w3u+4i2EUEw5vY7UAUuvGq6xgUgUUULDJKcFHeSA2a1cXCbWjIgmVT0f36/fae0yS3sAMv7/nO+zzP957nPYJhGOiu2NsJ43ooioVGAZqhQ9O1vdCh6iouuC7hwYtHyN55L3Q5FvQsVVGhG3/AjKjq+l7+G7Ks9lJg2idARdaQohFQU+HWJITly7RXoGp7tY7SX0AhARGHIJqsJKJiRA4hYAnBrgxBNNtgMwYJ83+B5OL3bzgqjsIBF4bkw3A6nTiunMCINYiFL585pq/A4xuz8fziu2RttYGQEcawZxgOhwNHtiT8XK7jzcfXyU+pr/FejsCmUCwWbaqqBinMiqLg+fKT21e9kVtng+c4aG1tDc8+PM2cGjqTYHVZlkH519TUVJULzM/Prw4MDEgamUciPFjrJpMJnU4HFosFtVoN6+vrjMjr7XYbjUbjJh8jHU7u7OykJUkaZC23Wi1sb2/zYD9gwh6PB4FAAKIosh+iUqm8JN6s0H1IuVxulA5eeb3eoNVqxcbGBieyOgt2ptO7yGazGuHiiUTi7j4TI5HICt3t/NLS0ixr02w2cwILJkBXZOQGYaJd8j9TGB8f3yRArl6vcwHmAcuCIHAR8mMzmUzm+o6RLQKNuVwu3rLdbsfu7i43lO0pHyPnT/cVmJmZsVEH13w+H6rVKjKZzEo+n39YKBTQbDbh9/uZ4WN9BYh8xe122+iuzOn79B2Ox+MTpVIpmk6nfzA/DgpYDrQvlcvlChEnp6ens93zVCqVjcViJ+fm5u7RG7jYy/kNO+KHFL4q4agAAAAASUVORK5CYII=',
award_star_silver_3=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ0SURBVDjLfVJNaFNBEP5e8tLXkKqtSa0kVYKplFIsFNSTIgiFhoL04FnsQYTSW66BkIcezMlTD1LQS7wWjIdA6qEVKq1ixAqSWBqMFWJ/EtvUtMnb9+LM1kispAvD7s7M983Mt6vU63U01uiDhfrErT68+VqEJeowTUuaRSaEhWsBN6bj7/Fu+rbSwKhoWpZpwrLqEMYfoDjcTXFogkmF2QyBrfnyt5phQtRM+DQT1901GHQ2yCcJjyOwKPirYsB7QpOggY4aBtwqTioGzp1yYq9SpSLHdmDqb1d+wKOp6DvdBrejhq6uLgy5LZxxtWEpvc5F9JYEC4+CkU/ZTX3pYwEXnAbO9vSgs7MTbqWMpQ/fsJIp6J/jdyPNGIVfIZ1OO4UQ/WR2wzDw9PXP+6OD2r0rQxdlUqFQwPNXXxLDXhHleK1GuhjGZigUykuC5eXldYfD4eP5iEQat26z2VCtVqGqKjY2NrC1tcVAGd/d3UWxWLwjn5Gck/v7+3Gfz9fBLe/s7KBSqUjjAkzcQ+MEAgFomsYFkcvlXhBuVml8pFQq1UeOl16vt7+9vR3lclkCOc7GPsuykEwmTcqLRKPRh/+IODIyskqzXc5ms7Pcpt1ulwA2JqARGVyknGAD/N8rjI+P71FCant7WxKwBrwriiJJSI89XddTLZ+RFyWNeTwe2bLL5cLBwYEUlM+0nyflL7UkmJmZcVIHN3t7e5HP55FIJFbn5+efLS4uolQqwe/3s+BjLQkIfKO7u9tJs7LST+g+HIlEJjKZTDAej39nPY4SqEfa962treUIOBkOh5MNfywWS05NTQ3Ozc09pj9wtRnzGyK4jfbwxX10AAAAAElFTkSuQmCC',
basket=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIvSURBVDjLpZNPiBJRHMffG6aZHcd1RNaYSspxSbFkWTpIh+iwVEpsFC1EYO2hQ9BlDx067L1b0KVDRQUa3jzWEughiDDDZRXbDtauU5QV205R6jo6at+3lNShKdgHH77zm9/f994MHQwGZCuLI1tctgVKpZJQLBYluxj6ty3M3V+alfvNG1Efzy03XGT9e3vu+rkD9/5rAiTPiGI/2RvZNrrSkmWL52ReGNw9f+3hzD8LIHmC9M2M4pHI2upbEqD18tdPnwmzOWJlpi/fmrAtcMrfnld5k+gvdeKTrcXT07FJxVovMHuMtsiUv3/xjzOoVCo3Lcs6DEi32xVAIBKJ0MzCY3My6BN1XSeqqpKnperGiamDUi6Xa3U6nTemaRLoGodEy+v1hlEsjBdXBEGg+Xz+2fgORazVamclSSLVavXMnjGHlM1m78iy7Gi321dDoVAYRQK0UCiMU0pfN5vNXShggH2gDud21gloeNahO6EfoSr4Iopi3TCMBS4aja40Go1vmOA9Ao7DsYgORx0ORxkadzqdS9AYdBn+uKIoTI9omsa28GTzO3iEBeMCHGyCIPQDdDd0lU0AaswG7L0X52QAHbs/uXkL6HD7twnKrIPL5Sqyjm63m00U93g8JdjHoC9QJIYCdfB8+CWmUqkuHKMI8rPThQahr8BeUEWwBt4BFZ33g0vJZPIQ/+s+kcCDDQSTn1c0BElD2HXj0Emv13tg+y/YrUQiITBNp9OdH302kDq15BFkAAAAAElFTkSuQmCC',
basket_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKCSURBVDjLpZNJTJNREMc5GTy5J8LBiwcJiY3iwRAPxoMnTUyKpkGFIG6hilXTRtI2Slla4OsqpS3UJsWWpouVbgj9KKWVpGVTCIV68SbxxkGCQjTx77zG9YJEX/LyMvNmfjPzf3lFAIr+Z296OTc3t21mZmb7PwEan76ubXJOrD5PZNdaQ0trd12v6rcMoOQqqW8WMu8kuMgCHsXegNmXuVjVXwGULGh0ZtE6uoQ6tR/B6Mi8xDFRsGu5EM7cswk2BYy9zHjb3SlcNQ1BafLMMt/D3tAks+9rPdCZLOY/ALlczkZi5UmsfCaTeZtOp7+urKyg1xXcmJqags/nA/mgNtk/USwMBsPHzs7OfEtLS16pVKaZ0ubl5WUQAKlUSpjNZhGLxbKJRAJer1cUiURgt9svDAwMQK/X2/v6+qBSqS4xsEwme1dEVQ5OT09jfHy8NB6PF0ej0YpgMLjf4/EInE7nDko40t3dvVOn05Wr1erdzc3N5XK5vIQgkEgkQ4U5ksnkBwKAqgl5ngdVqwmFQrDZbCICgeM4UbtVjjvm6vUb+nO42HZq/YFejIaGhqYCYHh4eCQcDpf6/f5it9stcDgc+6xW6zGj0birq6urQqw/36EKXEFssQfz73kYRsUQGQ7jpPjQswJgcHBQSAC4XC5hIBCAxWKpZjOTWNVs5hrN6c/hnBHh/GOwpU1chyFxE8dvlWz8fI7+/v4vJFax2WwuI7H2aDSaSppzr0KhqDyrOIoXi0/w+4os9DDAr/dklUgskFggsUBigcSCVCrFidsHwPH10PB1hWRNvO5HB+tb+nEU2FalLYOOv1aozE5mk1+75W9LwR20V1nb388O5v8GEnKwdx3xuSsAAAAASUVORK5CYII=',
basket_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKXSURBVDjLpZNdSFNhGMefdzues7ODc2wqW5ofE5WVqNgHdRM0AkPsSykWlBddGHXhRRdRBuFNQRR0E/QduG4kjCDSCx0omKzBpNmyi5XuVG5N5o6207ZztrOdnkkJXWRCB/78zvvx/7/veZ/zElVV4X8ezUaDfr+f9vl87EZzyN920Pf0bQ+XT97ZXUlp5n4YIC5KfbdPtT3Z1A7Q3M0w+cGcrqh4PsVxCqXhKFp9fPrWSPc/A9DcDHl5uMTEQmzhC9hIZHZ1aRkKbQ0ow50X7jdvGHC0Ruq3UDLwH3io5JSZrs721hIl7i20S0kKHDX53j/OIBAI3FMUZR8KstksjbI1NTWR4dFJubWhkuF5HiwWC0z7g+lDjj2s2+1OZTKZz7IsAzKmQaNSVlZmxzA7dlykaZp4PJ43ddYSJhQKOVmWhWAweKK6VM+Oj48/4jhOL0nStcbGRjuG2IjX660jhHxKJpMVGCCgtqEiOFheWAlVi+88cgsyirSgVhiGiQiCMLpWxomJie8YYhBFsVun0z2PxWI9uJIrGo06DQbDUDgcdtpW3g8Z517FVTFhzhRphXTLEZNbNF2mCgeByR5MPYMUEolEC/IbcidyAbljl//hYZPVnG3o7Tcztdsh/W7MNDflhr1fV6vWqoDf9AARRkMHbm0WeQBX9iEPGo3GmXR88Vy943iRbn4SyOBJ0C+8gNpqE5RLkY71P9HlcmVxF8VoqimcLrIB+RFVv99zY9px8xkhbcfWy7c6YIWZqWWV+t2BBgqVRgP8KtG6kjoOUv4R4F6eBzkdhRTOFxNayGkhQjZzG187KwboYu7KVrNCUZpFEGMK8EvaXFZSr5LNXudpZ9WlpBA5q82R6pxWDaPrbvuYcv0nJj6CVSYYRbYAAAAASUVORK5CYII=',
basket_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK9SURBVDjLpZNbSNNRHMf3JPYUBUFIT/USRCsiCKIeWhc06WUmicvNXdVITLDUKWmMZTJKZQ7mZDbbhV2au9aczjkvbU4tp3MqaWi28KWZ9zmWfvtPIgrEoh5+D9/fOd/P75zf+R0SANL/xJ6LIyMjSUNDQ/v+CVD44h29TNG30ubyrQnMobV7yresvwYQ5owS3TDuawcgso6hyj6JhL4tsmf8EUCYyYUKHwSdIeQ+1sNoaw8Uyft2NF1kRnqxlLwnoKvXqxWqPGA3vEJlg2Y4kXvYZB5IaIm0GlPqLExbhAjIr2FUcoJOCgaDUqJZE0SzJrxe70xPT892JBJBk9K46ff7odPpQOSgfV4UW3CXYmnMDqx+xtexl+gVpccSnW4Mh8MgAPB4PFSfzwe73e5zuVzQarW3rFYrvAbqsy/jVYh9smLBI8JGsA3f5voRaOFtk4gqxwYHB9Hd3Z3idDqTbTbbGaPReFij0ZAVCsX+4bbrdxbHK7e2Nvqx8eEuIoOFmFSyMdyUi0cPCjw793a73csEAEQ1akdHB9RqdY7ZbEaXKqtmMViOragP6zMMROdpWJngY0ZKWdK31KOgoKBsB+BwONotFkuKXq9PVqlUZLlcfqit9kLZwlDpDzMd0Y/ZWB6vwLSEstYpuJqan58f43A4qTsAk8lEJQBQKpVUg8EAnehGXai9BFsbvYjOsrE5n4Ml4iTvJVdidUI+8vLyctls9iyDwTj48/laW1vjzc3NybLisxWh13yMOuoR9mdifZaGSKAMU2JKxFSdfo7H451msVg0Op3e/9scyGQyiMVi6GsvAytzmJLdhJufgjfSNDirz6OyiAkulwsmkwkCABqNZtt1kATcE/GYrxExfwP8wkvQc46sOsvJp37dk52dnZSIXQGZFw9EaxjH411P0uKup5QeW+nJo3v9xu8f/sicYEnItwAAAABJRU5ErkJggg==',
basket_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKsSURBVDjLpVNdSFNhGH7P2dnZz3FOxfnD/Ns0pzXNEgbdSEhlhBeV9AP90IUV3XhR0EV3CUEXgRBBfxS18CIYXWStKJIo2NxSmw1zGLYlOja3Ua25nW1n39c7KVEwCzrw8vC97/s833Pe7/sYSin8z8euV/R4PPzo6KhqvR7mTw76Hrw/LpDFa5Yqjp1MFELsh9g3cHTr3X9ygOQehYLczynlmpmkIEgcK3A8vXPsytOevwoguRVI2qYtUUHk8ywYmeDEt3AU8msWJFv32Zut6wrsrRMvVHBpCHwMQJUgje3v7mrTSjFXfl3KJKGzjpxaNQOv13tDkqQODMhmszyG0Ww2Mzb763RbY5UiEAiAsTwFTMJFlNUn2JfDjmQmk/mSTqcBMcIiUdLpdM0o1oyJ8zzPM06nc6S+Uqvw+/2HBRUHheKreYOpk4357nkFQVCLonjJZDI1o4iRRdJAMBiERCKhx4Q9Go22I+5bWFjYjI3Pq+mT6yW1XfrCyi1Qxodq9OzIIawP+3y+vIMPrMVimUFyHB3MY2GPQqEYQ+JOtVo9UcT4e2W55BlNaRHk4o+hvLVXq5JJD0k21WUwGPICb5eGiERnPB7XI9H+e+dwaM5Sr/RYKjYdABDHwW0dhIKiFBTTebqrZopzOBwZnJdnSQAJt1Y4mEDcUalJuQtKzQc1xUk8VT8AJZBLvAPD9nOMnCxeZIgYQgH38k20Wq1ZtKRBcp2ahL+3CK7hho6TTUq5C0hm7teZyUEmtEP4EwvP7EOzpy+/qF2+B0jk0EkKcYqXIsGyDbub1JooUCmGRBmMPZrGLgJEnAZdYy00qCPlb/rNG9d8C+6r2+ZajgzqZWwABb5ihq66ezKVCcKTHpge6rdxa70FKZ1Rjt/uSVFC8dfJiqDLCHmktOInriB9Oz6CFbUAAAAASUVORK5CYII=',
basket_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKbSURBVDjLpZNdSNNRGMb/EcjsxiCkMKhGCEaxTAoh60qyIiKaiDTNUklTWatwoLnSbbiaa5vDMefGcv/+29gHa3ObqWvpprTp5kckrpvu2q0XhVZU7OlsF2I32sfFe/G85zy/l/c5HAoA9T+15eHS0lJOIpHI/ScA//liXYd55vOLUGxN6l1Zu8ssNPwxgJgr2x3zENpnofC9Q3fgPTK6VhGo3BZAzBy+OQbpqxXclDnh9o+/FZhmsrpO4cWle3rOloDX01F7ryWMRs0oRBrbfKb3aMg7m9H3n9qg1Oi0vwGWl5f1JKwkCSsZjUY/RCKR9OrqKoYY97e5uTk4HA6QHmQa4xdyF2q1el0ulyclEklSJBJFMklrU6kUCADhcJgbi8UQCARioVAIdru92ufzwWg0VvGZMtwaPgG9QQexWFyTAQuFwo8UmXI4Ho9jamqqYGJiguX3+0vcbvc+m83GMZvNeQaDoXhgYGB343DJV7Gfh2p9Ee6I69kEAoFAMJrdY3Jy8hMBgEzjBoNBWK3W6wLrWfCZ0nQzfRL1puNpkacKzoQG7a4rOKfemzbRRrS0tHRkAWNjY+MjIyMFTqeTZbFYOCaTKb+NKYV3UQf3gjZrtCf60R9qBx3rw21rBU49ZoHXyq3MAjweD5cAwDAM1+VyQafTXWsk+2bMqqAA8vE29L5sQo+/HpLRZuinpag1n0FhN/Vz4zlomv5BwmJptdoilUq1p8Z4BI54P6yzStDRPjx784RAWjE4LUGT9QIKu3K+7++kijcAJCyQsKBUKiGTyXB1kI2Lmv0oV+ejTJGHuuHT0EV60MBUoPBBLi43nA9v+5k211HpjvUbdDkOdlGpQ8Jdx3g8Xs5fAdgPqfUDXTuR30mxN/d/AaKgrAVJC0Z9AAAAAElFTkSuQmCC',
basket_put=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJvSURBVDjLhZNdSJNhFMfPM9eW+3BjbGIKuik5toZCgYRIF/axEUbSRQUjiqigi7roIiiCrrwIwusKbybtJuquZLSBWcSaFU0slC0/arbC6PVjn+/Htv7PKKlwduDH/32ec55zzvPxskqlQpvZtUf7K2JJpJXCGuXkfPnBubm6zeJUVMMYUxSv+zwNdl2iYkmqFVY7AalLamIqqjAiheSacep/J25HBlazcoFaTd3V/OUyI4fFTbuH6zOZvCgmb5StWyao2yaZ3OZeajLtoeVMitYKAvXZj2FPOsPT2Ynif7ew8PlW92xunOaFCbLom8lqbKP48nNKrRupd/t968CVu11bJhi0F6+rl4ZoMveEZn6EaEZ4Rq9TyfXG7Cmysjz128sX/jrs6enpO4qi7AMky7IGtHs8HvZwbFwM2c5oVYKTLrcM08t4onCkf299JBLJS5L0SRRFgn5XYaFis9lcSObCxFWNRsOi0eirjh1m7YnVkZOnDTcpkUgcb7Pq6sPh8Iher9cVi8Uhp9PpQpJ2FovFOhhjH3O5XAsSCMAN0nA28krAge9FaDP0G7QJrGi12rQgCGOqnp6euWw2u44OviDgMBxvUeGgTqebgvoMBsM7qBf6AX6fyWTiesDhcPAtvGD8KYdgGJyFg3fQCf0KbYXO8w7AAh8DPm/DOQlgEbs/Wr0FVLj3RwdTvEJDQ8MbXtFsNvOOfBaLJY7xIeh7JPEiQRpMst8/0+joqAyHEUF2frrQTmgS7AQJBDvAEmhC5V3gYiAQ6Nt4iVigBgUE068r2gCLNuDXjUOnUqn0uPoOav3OW5nf79dwDQaD0k8mKaZoCMdoNAAAAABJRU5ErkJggg==',
basket_remove=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ0SURBVDjLlZNbSBRRGMe/2XRWZ2/D7G4sSuYaKVurhMZaIRbm6iaGERSSbz0EPfTSQ4j0ZNRLQq8WPaQgPfQaIWmUJWwrSYolkW2ul7S0Rrbcy+zM7vQ/EovJ2uXAj/+c833nu5xzhtN1nXKN+h6Jc1Sqma/fhPHn574cpG2GYTuDt9quHbe0U0vRiZrqXvHZfwXoHqnTfaZWg8ceII90jPy7mo5W9Vjv5fLltrZwa6RJLzbVkstWQys/FiiakEng7TQ6N0iD7x4vhK+mSjb7522NmFb56PRqKF+OyYLbcQQlFtCT8H0aW5ygHUTiX1uYX75WL690C/PRScwyZDDoNCtPU1vlKB0ueGBpvXy76o8BTpUmu1x5CpGWT6Rn0CMrk6fIdIQcXJwaSjMXfjuDqampXk3T6gGpqsqDMq/Xy918eoaUtEJriSjF1QQdiHcmTjYcKhweHo6nUqk5RVEIumrARs3pdHoQzIOFKzzPc8Fg8GWj2EW1+qX209IN8lPn2d0OoXBoaOiuyWQSksnk9YqKCg+ClHGhUGgPx3EfYrFYMQLIYB9YgnEnywTc+I5Ai6CfoS6wZjQal2RZfmTw+Xzh9fX176jgExxaYBhHBr8gCJPQgNlsfg1thr6FPWCz2Zg2ut1u1sKLjXcwiIHJeRhYBeXQZWgJ9COrAMyyOWDrTpyTDCLovm3jFpDhzqYKJlkGq9X6imUURZFVFJAkaQLzJugbBGlGgCUwln2J/f39KgwWOJWy04WWQ2fAXvAezm6wCFzIvB9c7Ovrq8u+RGzIAwk4068ryoJNWdh149ApnU4/zPkv/Mvo6OjgmQ4MDKR+Apt6owU5Oz7IAAAAAElFTkSuQmCC',
bell=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKnSURBVDjLlZJbSNNhGMaFroKCbgqCHW37W24tT6mFRZq1sougqIzwQoKoCyMxSMWyZkV4KkemCUtJcSibkznn1A3pNGYTNY+bOptb2g42TzS1kU/fAodRZl088H3wPr/vfd/vCQIQ9D8aaaJUg3Kuz1AarPDfNzYoqe0mJRVKjNtMSm6eVRUBd3MiWvLYvg0BlhbqOTHahhXUHHn1u029H/rH7BV9ER/yHFbTugBi5I6qqUVnTxqWXFosO1sx25UOV1M8BsrDoMxl5a7W/sl81tpxAO6hfHxzqTHXfR6eNwlwKnhwNMbAoTkKtYhl+g1AjDuJ2qeMyViw1mHJ/hJzxiTMvI3HtO4g3JpIuFQCuLQn0VXM8QUAoyqKS4xTZNYVd/8d+GaN+Gq5D7deCKfuMCabYzBWuxd2GR/ORtJF6wl0PAheDgCG5Vytu+8clh0SeCcy4TWlYrH/DFyv4jFaH46hSh4+lFGwSkN+jjGlPo7GbJYtAOir4kzOW65h3iLC+xo+eutDMVgXjTEipyYaxhIOup/sgr2WA3fbMUzI4lB3kykLADqfBleMqOLgMedgoOE0VPdioRMfgbaAjY8yATytYegTs2GRMOFoSUTPMx5qrjOEvyzxdTFb3yONIF1kQ3FLAK+1EF96M6HJ56OziIGZZooAWGQfJEC32Z61vxY4tD1kmw1V4TC8uIBxXQa84yKMqC6iJGUrdHd3YEHJha3hEKQ3mIN/BPhFAtKgK96HtsJYKDJ50JcloPTSFjxK2oxuMQ0WaRSqrtIn1gX4Jc9mCeVZTOhJ4uyGU/j8TgiDZA8+qXejt0yAisv0zr8CViXNYIqk6QzoCngwV0fBXBmJpqwQlKbQRP8E8Ks6jbFJcoWeU55Kd4pTaNMlybR2cTKNtbbmB+pfvh6cSOd2AAAAAElFTkSuQmCC',
bell_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALCSURBVDjLlZJ7SJNRGMaF/i3on/6wUDfbvsLpUiO1tItamRUISdmFYEYaGQaikNlIvIdOp9JSI0ssXOrmdH5ONzek29AmanPenU2H5rYyNZqa5dP3CY1EQzrwcHjhPL/3Oee8TgCc/kfDjQTZJ2Uvt4vcZXS9uUFO7BiUEx6UcfugnJ1hIn1hazqO5gzm8qYAYzNRRhknBmTEHNX114TiALRC5oo23xNSPqPxnwDKyB5REAuW7ngsWtVYsigx25kAa2MwDKXekKcyUv+c3cgcaWo7CFt/Ln5YFZjruoCZNyGwyDiYrvfHdEsoFOmMwXUAyuhMqXVKdxHfTNVYNFdiTncaX98G47PmEGwt+2ElubCqw9FZwFp2AEZIgk0Zp6i7rth672N5VofvxkzYtGGwaI5gsskfo1VeMEs8YamnUihPoS3LfckBGJCy1Tb9eSxNl8M+ngz7YDQWes/B+ioYIzU+6H/GwYcSAibxntVrTClOoj6FMeEA6CtYk/PGm5g3puP9C0/01Higr9oPo5QsLX7QFbHQVbgb5ioWbKoTGJcEoTrJTeIAdDx0fzxMBmFmiA9DXQTItABoio9BncfERwkXM0pv6IuZeFkWiKRHUYgtiMDlzKP28GQu3/GIrwuY2m6xL5UiBbI7XNhNAnzpSUZLric68l1RWxmEtNpokAYReqZUELbGIUroBf9bzgLHL6iymUPtFT5ofxqFMU0i7GPpGCYvoejqNvByQtHQW4iG/mLQS6COgVB9gwYsrJkBakDqNAX7oBIEQJbMgbYkBKIrW3H2ng8Uhif4e8n1IhqwfgqlKYww6V03aKmJM7efwad3YTh8exfyVNeQo+KtmnOUvI0T/C1xolu6OMEVmjwOsrIJRAr2Il91fbUzvdP1mjfYSM/jXbeUx7rwS6NdLLyYnfbAOOefdGxK85Qe0Gd+Azr0uSlwBNX8AAAAAElFTkSuQmCC',
bell_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALKSURBVDjLldNbaJt1HMbx73tok7Q5tukhmzXNShuHXgwvlIEIgpp5pgqiIN5PUPBw4RClKjjwxhvZcDeCeKEgu1mKMUWYVFQUPFRZOrtmW0zWxKZdm3fNm/f0/3nVguIm/u6fz83veTQR4f/cSjFf9FwpWM2geOdzq7PmfwbO5EcUjOgaV5TIy2E99lAqPERtc/VhgBsC1VL+AzQeEJ+EpyQaiyT1+vm2oFyt60jpukC1lJ9WwlI8Uwgn9j+GJgH2HyXctZ+JRzyturY19/jbF9/8V6Bayj9hhIc/i4/Nkkjfhl0/RbDTxmu3EC1KenKY2p9bTwN/B6qlfAb4KJK+/d7YyCx9hoN9+X2UY6NcBz0SRnwbzCFGo+bUbs68MJ+f1g2+CnzGU5NPacmJR3A3vsC6soiybfyeg73dJdQv9JuCBIJlK7UH+I6cTE8fysRHjxA4K3jNE+jeNuK5dDYsvB0Xr+dhJjUwTFSg2N5RrT3As+RgaDCNs9Ng+dsi/f2KPokSAuKJPmprFoYIhmjogzfT63RxXPl+F9Dta2q+WfkV33cZGJiiXonTbA1wqbZO91qPqVuimLpis+Lx+4c/sXLiOxJLjbvL95uvAmgiwuJ7B76JZVKHp+44wpenihSOPou91eaHcpGU0WHIN+mujzBzz5OEcrdiL5U5t7gQXF2uvKjtVnnh+IHz8X3JGdQMo9mbGM8lqJ+r8PmnRQ5edbjr6HEiq2eh8TUkkrTNLD+WFy/uvfG+Y9X8mbnc6cHE8uyFzcv8smAxlh3DVILeVYTHc/DgS3t9MecyGEqb1P45ptOv5QqIlDLZFBOH9mMGPr+9e5bDjz7DYG0ex27SBayOwfIqDe16a/zklcm3UPL66L4YqY6P11RMDPmYeh1r3edSywi8nryh3WjOH7+QNVxHjnkezw87Eh3YaGkhT8KBIQ2Bk4Wy/85fhGJYxwKt7REAAAAASUVORK5CYII=',
bell_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK/SURBVDjLldPPb5MFHMfx9/O0K+3W51m3tbBmkU0XfFLdgmjUjMTEgwSNIIwdhBiPcJAT4WC8IfwHxGjUgwkJN6KJDK1HJQYzdRIMgtW2MMfWbZ3r6Npnz6/v1xONJFPj9/z9vPK5fAxV5f/cb9POdODr/mYtmn7+rfJk/D8Dl52cQM40WBDV00nTerUv2c/cn+UDAP8KVIrOhxi8oiG9gWjaSmXM+V/rivhG29PiPwKVorNLlBt2fn+yd+gQhka4fxTxF69jpwKjstg4c+Rc9d0tgUrRmYolBy7ZOybpzY7hzn9E1KoT1JdQI012ZIC55cYx4GGgUnTywIVU9umXrNwkXTEP9+57iOcivoeZSqKhC/F+tqfjow9y8d+vOLvMGF9HIYN9I0eNzCMH8Ve/orlwFXFdwk2PTS+B2TNMKipDpDRdkQeAGXr6gT20Oz/87NtGj+0T1N7HbM+igc/91SbrdRdJZLEfO4ZnZpFIWG/JUqdB0NTCtp4sXuset69Nk0gIXZpmG2D3drHspcmNTmHn97By00K8Bp6vM50G7oZcqd36mTD06e4eZf6WTW2pmztzK7RaQsLqx8pmiO5/zvbx4zRbIGJ+0gGeO1k+USvdu3Z39gt27n6SoN1m4rUD7Nn3OqvrFgOFN2FzlpkLF0lnXKIohlMYSXYAgBdOVfauVNdKv1z9krxToFZeRMUnO7wXq6+NeFVQIdr4nkdfPE3oeedmzk9YHQBg3zsVZ7m69lmrcZvyT99S/u4HMjufQto3UHF5ZspB/AWSyRI7xo8WQs8/9RAAcPBM9Uijvv6y+hsMjh2m26qj4SoYMX78tAQIslki9/gwkRec/Obs2BPGVmucOT8xP/7GxaGYeQcN14C//5jEUg5LN69Tunz20pZbCD0/OfvxlKuiiEhcI4mrCCqqqPgqqoigqoN/Abi5XyBiV8YvAAAAAElFTkSuQmCC',
bell_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALWSURBVDjLlZNbSFRRFIb/c8bLjM7NcUadvDZig1kgUVmBEpRplIj1ki/1XFAQXUAq6EL1UvSQFQk9FD4ERRGNZBcJ8UaQGgWpqSOOk46XyctxPHPOPmfvtoJCkUX7ZT+s9X2svdZeAmMM/3P6fV4fUVmZFNJ9RccGq2L+Cbz0uijgEgWMUsZOGUXLviSjA4Efg/sX438V+Bu99yFgL9NgI5SZLSa7GOybYqCqsKCwxlUFHMyjDJ+t7jKjLb0SAtMhjzRCHfsEq4kI/rGZiweuDF36o4DDBw3G5KfW1CrYnBsgB+ugR6ZApsbBBDOcOckITMxU89RfBRx08+uRyblpt8VVhViDAnm4FlSRecUKRJMRTJM54UCKOSZ3mYsZaPDmiQY06xrSknIOCfbMCqjh15BGW0BlGVqUi2YXEB/HEBfDwHQGSaZ0RaAp7J4zr9BtTSmHrvSDhO5CJLNgRMVcWAKJqCBRglqzHRGN4HysjtkIHV8REInlxyc6oUS+o7fDh7g4ilhmRjwPWm2xCIxJMPC/EjWqyHCux9mRLlRorHNZIMrztCHU8wWapiIhIRfBHitC4wm4rcXj6qSAB3Ybai28efY85KcVITPZi4fuaGXJDYd5USAs/sSWW552iztpe+7WcjTV+VB29DDOtDxBacER6IxCpzooGEKzQSQlpqDD/w6dgWYa1VTH0hSKT/p3vL3u6VOUV+vc3nyEBsfAg0vwcJj3hWrQKAHRCeYUCYWZxZgnC+KHodawuPyW0hq/d2Jo+nlkphcD3W1cEIXGR5NqzUKaNRtum4eP1gSXJR1dI21oG2ydXFCxWfh9mZ6dW1sGxhrrs0QoqgqVqkvVeJwF2ObZg4+BdjR9exNSNewavca+Cqtt4+PTOZdB2YWUNRZkpJlQPdyNjZk70dH7XpJAt3C4b6WJq536E9kGVWE1hOD4zfSgS2GiUDidUPLizlzLcs5P3E1ndhQ2sD8AAAAASUVORK5CYII=',
bell_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALkSURBVDjLlZJrSJNRHMaFvtWXvvShT6ltb6Qz1EorZDZv5ZwNJ4UQQYJLnfZhmTJDtthUUpi2UcyELKI0l3My57zkEE2kmszyOi8jLyi75GVaXpr59L4vNBgV0oEH/ud/nud3zuGcAAAB/6OpVsI4pmN63z8O1lPz/QMG4ojNQISQwcM2A1Mxa4yEuy0B7Yog774AezvxhAzOT+gJD7nrz3nTWQxUB+0NKFnQlQS2/hNABpnTJmLLOXQb265u7Dg7sTYohquVg9GacBhkgbLf3r+F02d7zsM9XokfLhM81mtYeRcHpz4UjpZoODriYZIH2v4AkMGjpN4uWTKwMduI7YUX8Fi4WO3n4Kv5Atwdp+EynoKrOxmDVQyvDzBtJJhkcIm86557RArvmgXf7aVwD1yC08zGYls0ZurDsNDEgrOFPEXnZfSUBe/4ABM6Zrd7+Cp2HE+xOSfBpi0TWyMCuHo5mNZGYPxZKD5rCMw2nKCvsWRKQsu9wHkfYPg5Y3Hdnot1uxwfX7LwSRuCscYozJBydkTBomLA+vA4FuoZcHclYq4pBo13jzX5AB8eBddOGWOwMlmC0WY+9NJz0NxPR0WZFHV1dahQFKC7MgLDNQQqSwvpXnmZYkMkEhWy2exDNKWvKmhgqCESy5MSFN9kQ6VSweFwwG63o7e3F9UPCqFWCDFpG6N7/f39tIfH41X6XqGrPGiyVhoBSZEY1pERlEmzkMpLgVAohFqtRnFxMV3zUrjIyMgwUZCCgoI1vz+QeSV0qlQuowFvilhIjI+F2WyG1WpFX18fdDod4jnsPT6ff5ACyOXyLT9AWFhYbk72rQ0KcEeYjIQEDpKSkpCWlkaLquPiLu4JBIJOCpCdnb3qBzhJjtTU1AmlUrmqff1qs7m5GRqNBmKxGBKJhK6pnlarXaY8lNcPQBDEgdjY2HAul/slKyvrW15e3lZ+nmhXduOMR36d5ckX5eySvW1qjfJQ3l9oR6i5wBWqZAAAAABJRU5ErkJggg==',
bin=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFuSURBVBgZBcG/S1RxAADwz3teyp3XFUUWNVSoRGQR3dLQIESBbUZt9gekm9XW2lRbNDv0gxbJWoJoCcT+ABskTgcDDwLpOD19d+/73rfPJ4kAANaejUx03t5eBZIIgKe34r3JB7OTVVvZuzf9lderiKIoip7MLba+xY24H4v4N36PC635uSgFIJ2/Pz7ppH19w66aHk/nqQCfk8LU1BWJAyMyo3Y1bV2nwpeh8nxxthg+Vm+ZUFVKHDjhK1UqlJeK52E61LOkasOhRDAic8EWKp/qxaupmdOO6Fi3bVyiEAQdA6Th7tjMGYcyDTcdtWlUoqYtypHmjy/atadrX6JpU5QaMhDlSPNTFX9kMj0H6rr+gYFCjnSw3XNZ2y9dPfT1lUq5UkA6+Phb3TU3NJArHFeKhtTkSBc+rC//0NBQVbNmwphzGu5oCztUGDz8udydbSrlVmI9eSkIirzYKZokESw+yl+EdtgL75eWAID/yIWfXhcZhKEAAAAASUVORK5CYII=',
bin_closed=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAD9SURBVBgZBcFLLkNRAADQc1/fU6qkhFRMMNIQn8TUQCJWYBE2YBkWYQWMJIZmNuAXCSoGFUGIInJf33VOSAAAAIAcgLOFt3079flaEdTS50M6nT7YeggJwPFle6nhAoVhc370rnaXcwBSp62GTdxoGdPrkAPQD5OSbRFr6oLvjByA53CqY9YUvjy68YQcgELTuTd/khENbQk5ANGqFUSFnq6WW2QA5Op4VuhreJVEZACUAKiJkogMgIEKANFARAZAKQKolColMgA+f7vVkBkRSeYjvf6QAfB1cnnXNWTUhHHrXuLoESEBYO/aYjNUSqX3snk/2DjshwQAAAD4B9GUWR0G4scKAAAAAElFTkSuQmCC',
bin_empty=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFtSURBVBgZBcE9b01hAADg55x72mrdnqrE1SCkCUEivu7SxGKppGMHNhKj2KRisYkY2MTcRFQMGh8/QGLxMRikJklJkN5B0N72cu95z3uO50lqAAAAQAYACxP16KF8vhotvxSPNgfF/QFJDWBhOF7Yfyk9EXemRn73P359GJce1BkA1Y1918+MtxSiRmtrtjfzc9qtpAYvkmhl4/L4pNKGnglDfng6uLMt42WjOhD3xOGTzQ/acpVa0PDe5AgZ1eF4szxbNvvJlHeCTKEhOOUVsmfNeO/Y3G5D1q3giERUWreuQFqea81N+acvN2Pcqu0SYzpqAWm4Mu2XTV1bEm2raqmGQi0gDbsy3/X19fzV1PUHFKKAtPjWc1THJ109DAxUKkGlRFo8+azpuNNyBNEOlVrDmID06uOV5ddyuVFj3jioZa/crI5yjYzi/Nvl7nxbJXheN5O7SqUY4lpsk9Tg2sVwu+yUm+XS4iIA8B+6i5xffIyBpQAAAABJRU5ErkJggg==',
bomb=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKrSURBVDjLZZPPTxNREMe/265tbbEEkmpD6wGCGolwkYh6gIZwa28evHrTGMPFxL+AmxqT3ogxejDePGhQYxqh8VC1tCZSBVLFCsovgdKW/u7ue85s3BV1ksm8fW++n5n3YxUpJUzTclE7hTa1e7xozsnXSqcyLPNiWrkkdTwl18mbB8KyxuuKCSDxAIWX5IuQYp7iY0DEbSvXz0PHNRIFyJNCx2lHRIbMAvsBNykMQcqz1YYsNFvC3W6b98RnN6qhwPODantIEYV1aLXFNwR6IDTE3BdlzoY/Ni+EPDO3VNPvvdB8qSWPp6kcxXDPglvtHFWUZhvs9SOwVX3nRAsOFrPIApDoUL7YcJ7qcbm21r/g2fRHTL/NQVTWIUqrQJm2vFeDbLSwlnde3buv+C3AxMTE4LfvWzfuPtlEqaLB3daBRCKBzOZx3H53BfXCZ1T2XqEhFpEsHsaFyRNy6ObJoAVoNpuRrq6uwPu5r7j1aBvZpVUUi0Wsra3B71zBww+jmP3RjTvpAWwXVBzrDvZVq9UIa9XfAONUfT4fYrEYaBFjY2Pwer2Y27ChXPZi5afEwoYbiVwHnE6npTEA9Xq9XwgBv9+PcDgMHrdaLVQqFcNpHZ/yHjQaDWiaZgBovd8CcFKhUKBKZUOs67oBYEGtVjMA7FTVAPAcRwtAExkCjLCAE3mRISzgb57nMbvNZkOpVOKcjAWgynGHwzGiKIrRDVdnCEcWmZGhLpcLOzs7PI5bt0AnPpXNZvn5GlvgFvkg/23fbrcbbedyOXp0YsoCzMzMpEgQTSaTJW5RVVUDZFZn48oMSKfTJaoeJWjqr3+Brbe39zIJxoPBYJ/H4zH2a97G7u4ulpeXuXKUwJP//UymBQKBQTq0CJ1BiCoZ10ueIY9T7hSJU/vzfwHG+eOypzQIsAAAAABJRU5ErkJggg==',
book=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHjSURBVDjLdZO/alVBEMZ/5+TemxAbFUUskqAoSOJNp4KC4AsoPoGFIHY+gA+jiJXaKIiChbETtBYLUbSMRf6Aydndmfks9kRjvHdhGVh2fvN9uzONJK7fe7Ai6algA3FZCAmQqEF/dnihpK1v7x7dPw0woF64Izg3Xl5s1n9uIe0lQYUFCtjc+sVuEqHBKfpVAXB1vLzQXFtdYPHkGFUCoahVo1Y/fnie+bkBV27c5R8A0pHxyhKvPn5hY2MHRQAQeyokFGJze4cuZfav3gLNYDTg7Pklzpw4ijtIQYRwFx6BhdjtCk+erU0CCPfg+/o2o3ZI13WUlLGo58YMg+GIY4dmCWkCAAgPzAspJW5ePFPlV3VI4uHbz5S5IQfy/yooHngxzFser30iFcNcuAVGw3A0Ilt91IkAsyCXQg5QO0szHEIrogkiguwN2acCoJhjnZGKYx4Ujz5WOA2YD1BMU+BBSYVUvNpxkXuIuWgbsOxTHrG3UHIFWIhsgXtQQpTizNBS5jXZQkhkcywZqQQlAjdRwiml7wU5xWLaL1AvZa8WIjALzIRZ7YVWDW5CiIj48Z8F2pYLl1ZR0+AuzEX0UX035mxIkLq0dhDw5vXL97fr5O3rfwQHJhPx4uuH57f2AL8BfPrVlrs6xwsAAAAASUVORK5CYII=',
book_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLbZPfa85xFMdf36/nebbGBcLaMmtM2DybElsjQm5c+PEnoKSQCzfEnZtduLRdSJFc4UK5sEizSLtQNCYXJMXKlMeG7fv5cc5x8X02w06dzqdPn/M673M6n8TM2H/6WruZ3TaoYPQYhhlgRh5s1lUCwU18GLpxfjVAgfzBMYP15bZVyfjXCcxmkiCHKabwfeIX085QK7RQtRwAO8ptTcmujiZWNZSxnICa5lU1r758cR11tQW2HTjOXwDMlpTbm7n//B2VyhSmCoDOqDDD1Pg+OUXmPHOt2gJJoVRg7cZmWuuXIgJmiqohYogqUY3pLHDrzuP5AIaI8nF8klJaJMsygvNEze8jCygUSyxbWIOazQMAVJQoAecch7a25vJzdZgZ1wffEmqL/JP/R0EQRUIkSsrNx29wIRLFkKhEEoqlEj7mQ50XEKPiQ8ArWFpDUixCamiiqCpeErz8D0irQyREIWYRF4QsClkQXIi4KDgvuKlnfP50iZZ1A5R3j7PvXOeFvxWIElzABcnbEcOLEkWR6ac01r9h84YuVi5dy+DoXYZfP7nYfbJxcTq3heBzgI/KdBB8EFxUpn48YtP6TiQVOhv2Ikmgu9wDcCKdWRgfhegiLihZEELQXL4TKj+/UEwWsX/DKQDO7LnCmhUdALWzMxAxxAsu5J6FiHOK98q3yQqjY8/ofXgYgN4Hh3k/PgKQzc6ANGVLVweWJIgYUQytxsQdZHhkgJ6O7dx71U8pKfD05RBAX2Jm7DzSd9WMo/nPm7P/GFTP1A5BzQtIPMAPoH/48tjZ3wRCz4QMKdc8AAAAAElFTkSuQmCC',
book_addresses=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKUSURBVBgZBcG/b5R1HADg5717764trfQoVUpasAfU+CtRgmERXEwcHFmMgwwmjo7+BUyOxsXVmGidDHGQEAc1NsEQxQsm4g9sKYWgUErLXe967/f78XmKc+fOPR8RFzE3Pz8/3u/3Tey/SgA5iAiQM6MR25vxz/IXv3agjIj30Ol0Onq9nkajYaZNQBCJrAA7jyHUG8UiQImzCwsLTp486cyZM9rtKRARIAQB1OtN+y5ssI/PP7h+w+Ejr5UR0Z6cnHTp0iV/3vxQTkAEEEEOYKfHO9UByye+Utv8d2mXqRJFrVYzOzvr1DOnKbLIIYSQRQ4RCaS05+tr90GzUdhlpoSIUBSFejkFUvTMHz2tXraUZcvNG5cNhzsiEhHiv3uM1UAJEWE0Gtl6+Lux8WOOP3vWrZvfG/QfaLbGPX38Tdd/WVavZwe3C998vN9wmP74jM0yIqSUpJTAk4cW9Xv3HZ4/pT3Tsbc3cHvtO0c7r1hf/VHkMGVMbXyw9P6uqEWEnLPhcAim2ws2bl1RL1uurnyk37vr3sbPJiZnpVTJgFqzgJka5JyNRiOQo5KrgZwro+FA5EqVBlK1JyKT2YkBgDIi5JxVVQUeb99x4OAxE/tmPPfyW1qtKYsnXvfwwV+yrIyeU8UF8Wi41XR4s4Scs6qqwOrf33rhpbdd++kTT+w/IvJIrRy3cfsHjWZbDqZqT9mZuD891s9RRoSUkpyzKg3AnfUrFpfeECkZVbt+635JnTInABpNmCkhIkSENBrIaWh9bcXa6oooiKBoAFFkge3dW+zCISXA3NycZmubIiOEIJIoQkQGOSWPJgZefXHl0263ez4rpsuIuLy6uvouPB7elREZAeQAoEBwsdvtnoea2Pof+BV374GHrjgAAAAASUVORK5CYII=',
book_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJhSURBVDjLdZPda85hGMc/v+f5Pc8e2wE2QtiMyZh5K4RSpCkHxJnkSElOxJFy4kDyF3DopR15yUst2pRpJw4cSJLETCTb0mxre57f775eHPyevdBcdXfV3X19ru/36rojd+fwuZtt7n7PYQRnt+O4A+5kyaePaSAko19e3rm0GiAme3DaobV9Q2M0NDyK+1QRZDDDDX6PTlBOHPO4mWpkANjbvmFltG/TShqXteMZAXPLulrWffGCWmpLMXuOnOEvAO4L29uaePr6EyMjk7gZADalwh035/fYJJUkZXZULRDFxZi1G5toWVKPKrgbZo6qo2aIOeVK4O793rkAjqrxdWiMYq5ApVIhJCli2b2QJy4UWVRXg7nPAQBMDdFAkiQc3dGSyc/U4e7cevGBUCrwT/2MgqCGBkE0R2fve5IgiDoqhhBRKBZJJRvqnAARIw2B1MBzNUSFAuQciwwzI9WIVP8LgCCKVIQkKKJGUKvmDL5+4BFrPj5g29AAv4olujviix3dcm1GgRohCSRBMzvqpFVIa/9jdiV9tJ48Q01zG+W33bzv67nSc6AwkZttIaQZIBWjHJQ0KIkYy991sm7fMUqfe4luH6e2/yGrmhryHvn5eGphUlEkEZJgBDNUnGBKCM788UFKS5vh0IUZ75eXkbdo1fQMVB1NNbNghogh4og4Y7UNTL7pou7JWZLyTyaB8bE8mufH9AzI5di+cxMeRag6oo5V8+iWE7x71UVj/TzifIHxYWFgMFLHr08Bep51vTqV/bxZ+4+Dw3NfwX7byuZvPTSkYPncd8dvHOyWq38AFgvYQWlFsCQAAAAASUVORK5CYII=',
book_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK/SURBVDjLbZNNaFRXFMd/72U+dDLNRItGUSeJiagTJ6IQhdhaWoopFCJiF10UBAXpSlHcddHi0oUbkXYRFURE/NiIIjSkpCpdtGoTJG20iUMsMZJokhmqee/de8/p4jmDggcuFw73/s7/nPu/nqrSe/hch6peUZhD6VYUVUCVeNPaEmcwYbn06/nv1gIkiA8cVNhQLOS96ZkyqtVLEMMEFZgvv2IhVEQTrbyJGAA7i4U13qeda8ivLKIxAVGJq0pcfVljhsyiBDt2f8s7AFSXFDuauXVvjLm516gIAFJVoYqKMl95TRBGvB1vWsBLpBKs29RMe9NSnANVQURxTnEiWFEWAsPlq4PvAyjOCRPTFVJ+kiAIMGGElThvqSORTFFID3Oy+xfqdnUyfLZHvWByX3UGiBOsM4RhyJ5t7bH8WB2qyp27fWxLP2dx8RtyrVuYL61n9Oe+EzUFxgnOWKzzuTD4F6GxWKc4K7Sk/2DPpjINuR3Mjv9Nyov4oGEF2Q/zuRrAWiEyhkhA/TReMgm+sjr1gL0bZ2lc20M4dYlUxmNiaBQTRC+Dhf+6q0PEWIcNLKFxWCcYJ6zkPl93lMi19RJM/oSfsiSzzQSzI4j1P+862v/YrylwggkNoXEExrGkfJuv2sbJtfcSTP6InzRElRaeDtzj+4EGth7tHwLw327BRDGgsXKXL/LPWN7xJdHzPupSSlhpZur2fX4Y+Yyx+XTtGf2qYSLrsKGl/lk/vflphFVMPTyFEPBqdhWlwYdcW3SYF1H2vUaKDRM5CjpA4aMzPLp0jMd3fiOd30x5ZoqbyYNkMktRxhCRp+8oUFXwfbq2d/JofIZo5Aatmz+mvn49//75D0NNh8g2tWGtoAphENbs6Kkqn+w/3afKAUVZ8eQ4W1uX0bWhhYmonqulTuZMtvYzUa7/fvHI7irgf/y+taODWkwAAAAAAElFTkSuQmCC',
book_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJwSURBVDjLbZPLb01RFIe/c3pbvdU2RYsmfUS1KrdaqkGQCANDzyZMjEhEYiBiYGCkf4LExEQiDDxiQARh4DWoegxERG4IQkQ8cnsb2rPP3mstg6OlcleyspKdtb71++2dHZkZ24+c7TOzywYljA2GYQaYkRWbSRWPd+V398+d6ALIkTUcNFjeX+iIvn4rYzY9BBlMMYXx8i+mnKGWW8KfyACwqb/QHm0ZaKejtR/LCKhptlWz7S1NddTV5ti44xCzAJjN6+/r5ObTN5RKk5gqADqtwgxTY3xiksSl/Bt/LBDlanL0rOike9F8RMBMUTVEDFElqDGVeC5duVcJYIgoH75OUBNXkyQJ3qUEzc4DVeSqa2ieOwc1qwAAVJQgHuccu9Z2Z/IzdVj5Gd/eXuSF7OW/+b8KvCjiA0Fizt97hfOBIEYsCduar7C0Zwul4iPM8rMA8TQgBCX1nlTB4jlE1XmozlOof0nnkg00tg7Smo7R3jheCQA+CCEJOC8kQUi8kJcv9DJKQ3MTMnGNtoF9bG17TVVcSYEo3vkM4AXvhYI9pK1vNyTPGTt3gfqmKZbVf2R1m6tswacZIA3KAimyclFEw7xJ1L0DU+TnE7o2H2NXocTYqfUNMwA1Iw1CcAHnldSnrJbbNHWsQidfYDrF0HAvmn6mtrbI4MZhgkuPzroDEUNSwXmh24/S2zNIXcN3LPyAqIpnV4uAokmRlmWdiPOHH4ysKMw8I3HMmnUDWBQx9OkuLYXTEL8nbmwBjLX7d864rsovpWvr8YXF6yMnpwF3bt0YPZD9PGNoeZnHZ/ZgapjqrBppkNgkRRUzW/wbVUOsic+TyncAAAAASUVORK5CYII=',
book_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ7SURBVDjLbZPLa1RJFIe/e9MdExXRJhIHTcQXjIkd3PhAR3EWulPxL3AhiCsFcSG4HVyIS/cj4krFjYqCQRONqPhgNkbxgcRAAoomHdvOvVXn4eKmYxw9UFRR1PnO71d1KnF39h37t9fdLztM4GxzHHfAnWLy2WEaiXnt/eCFU6sBShQHDjv8We3pTj5+quHeTIICZrjBZO0b07ljXlrFTBQA2Fnt6Ur+7uui+48qXhAwt6KqFdWXLp7P/LYS2/cf4ScA7kuqvSu5+fQtExMN3AwAa6pwx82ZnGqQ5YG5MWOBpNRaYt2GlaztrKAK7oaZo+qoGWLOdBa5dGXgdwBH1Rj5OEVrWibLMmIeECv2hRZK5VY6FszD3H8DAEwN0Uie5xzYvLaQX6jD3Tl/9xWxrcz/8n8oiGpoFERTLg4Mk0dB1FExhIRGepKx8Tokh34CpE2AiBFiJBh4Oo+k3A7ldqzcjrW0kklgRcd6FvedI2mxXy6RKIpkQh4VUWMyHMUsEkwIIiyvrGP9si18zb7xZe99dp6tLLx34kt91oKoEfNIHpWohlhgd+9B1A01xXDGaqNUV2ynHqZ59mGw1vdPWinNtRBDARBzMg2oGyOf3xBNEItEjUzlX9nYtYN6bKSP3w99LjUbJogiuZBHI5qRWYao0LmoGzFF3RivfaCycBnPRx/w4N3QpyDsmb0DVUeDFhbMaHjO+YdnCRbIJLCmo4etq/fwZGSIOy8GNU/YNXbah2ctkKZs2tKHJwmqjuhVTB1Rx9W5PvoXSdrO/ZfP8Cebr4313xue2we3b914dKj4eXP6H4eZdX2Vcvf1AEvGj/f/13/mQPMZvwOnSeJok49LlgAAAABJRU5ErkJggg==',
book_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKdSURBVDjLbZNNTFxVFMd/781jICiCCFoSC5YgGGAQG/E7EhckJFqMe5suNI27Ji7r3qXbLk2rSRuFhU1asESgxvoRm5qQoIAikECMdcIMAzO8++6957h4MyjKSW5OcnPv7/z/554bqCoTFz4eUNXPFQooLymKKqBKmvRwibdYs7t++8oH3QAR6YHzCk/l+juD+3/tolq7BClMUIHibpkDo4hGp6hGCoBXc/0ng9eGTtLZkUNTAqKSVpW0entLI40NES+/+R5HAKg+nBvoYvrubxQKFVQEAKmpUEVFKZYqxCbh31G1QBBlI54c7KLnsVa8B1VBRPFe8SI4UQ5iy2eTC8cBFO+FzfslsmEdcRxjTYKTdN+RIarL0vZAPaJ6DAAQLzhvMcbw1nM9qfxUHarK5flfeOJgkovPrjJZKfxfgfWCtw7nQz5d+BljHc4r3gmOgCE3zzODJZr73+fEIx9x78Pe8dMXV2cOAc4JibUkAhrWE9TVQahIIPTG84z1rNLcN0Z5c5rWR/vIdJqp6Qvd79aaiHUeFzuM9TgvWC/VrHSGv9KSex1fWaKh7XFiv0XH6HCjKxcu/aPAC9ZYjPWpHa8kVcifrhVb2CDM5AnCMtmH8hBX2P/DZI5YsEkKcKIkTmhz64w/+CVNTYq4bUIpQtiE2oTlayvx0krpfFQbmMR5nHEYK1gR+twPjETf8PQrY2hyi8DvsXjzgKRQJBDL1kbp7bNX16cOe+C94hOfWhDhheafGB49Q5K/RBQ1sziXMLv3Bl9vRuTX7mzdu/3t1JFnJAwZeX4IDQK8V7JkCTLbZOrbWby1z1flCXbaX8Ss3aVYcXP/nYPZmRvfv5P+vOr8n7CY/B1KO4ZPlvtZtQHKd6Bc//3HL87VAH8DX5rXmGdCnY8AAAAASUVORK5CYII=',
book_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKnSURBVDjLdVJNSFRRFP7e+H7MZ06TyjCSDpZCaS93BpX9EUEEhUJkBS0KJDBoES6ibRujIqJcBUW0Khft2qQZtWigEYLBUgxz0BnHqWbUeTPv/3Xue/5k2eXdex73nO873zn3cK7r4tS1J61kX7pADi72uXTQNdjhG3d1O7YJU1+Yevfs5nZygYcf0EOBO5WWBm4+u+CBfQDgkzmgD/kFFSWdSFy+EcvLJwAOKi313JE99WiIKB6YoR0GZNbxs9duqUBFOY/9p69gHQF5Q0prFK8/TSKXK1I2x7t2VlQwSyT5xSI03cCfa7kEcLzIo3l3FE3hrbBteLJZZtumTYQW/Zc0Ey8GRzYiYIEOpucXIQYEaJpGjTI8ELu3UAZeEFEjS56qfwmYXBZIHdZ1HZ3tTb58X52X4OnbrzDLBfyFX1NgEoFtWkQSwPORMejePymwmAIOgijCsPymbkhgkdMwTRjUPzcggRMEIEAvwLFekM/maP+XADAtG5ZmoVgsQZ0ZQ9DIoDFcg2/JNNTqNkCSsTT1Eef2hjEdaUVvb29fIpEYWFNA7EZJQ3J0CG11m3D2wkWoqoodMzMYeh+D5pbh/JkuiAKPdDqNeDx+O5lM1q4rIZccR7O8hI6jJ9F36z5+fU+gLhKBoige6N7dO0il0oBYiRvXr4IIeviVgTGoBHU+iSqaRrbKW06g+CWO7u5uhEIhFAoFZLNZPHg4gB8F2YsJBoPSag/YwHC8jFQ64znzo6+8B+7v74cs+wBWUoADtm22NJZjdnZWWy0BgQDaDx/D5IdBDL8ZRtfxA5D4DmQyGUxMTECSJESjUYTDYRRLGheLxfKpVGqOY+BDlx49JnOZjS+MPCoLn1EXriaQSBMAjOeqYLkidoV+kgLXMXStROAsKer8DdsBr2sFe8jtAAAAAElFTkSuQmCC',
book_next=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJQSURBVDjLpVNNaBNBFH6RTVNimmQtpg1EpCFVXONPTD1UViQeoihIPXgXURTFXjyKiAh6EkFEFCweCiKIh4IWWqihULC2aRpqC62bSlqRQjZ6CEm6OzOb9c3UBoM5CB34eMMw3/e+x3vPYds2bOVIF59NxC3LgsGbJ2b4w/kH7+OMUqACBCjZiJ+eXp1pJuC4/XraNvFTaiJ9rYZueo8oz8G2gDEGXIjH+eU1cLW6elDsy/SLG6RBwDAMeySzAgNji2uccCW5P3jmaLghS/zyE/B4PFBdr+7u6h0dIAY5O9z/UQhJmUwGEtEodO2Qgm1tXtjVIUM2mwVeVrlcBlVVgRATS2kBYpqAZNVcJwvI7RYOCoWCres65PN5UBQFAoEA3J27JTJbDEuhDMkMyRTJFFqICwUpLGlf56YeZg9JmqaBz+eDUqkEJmbYPMfaj4NVQ4Eaq4Oiq5/VXxgZuqscVK7v+bwtEokIwt/t5JmphRktsgFGwBQwEFgGB+8SYVTK5XLg9XrrZC7EbadWxuq2sW4wETzu9HeCXtBBW1zOai+/qVI4HIZisSiImy7uHX4k2sfLCoVC0J7oxzK9sD042ZdvTb/BzKtIjokuyLIMbrcb/H6/gCRJ4HQ6GwTFTCC+L+ydDR34MY4Oz9UnMXbpsZi2PnWfIAwOT4op5ATrD5G3lJfF75j5dMMo98Si4vJufEp8OpVMNJ35oaEPQvCfXUjPzgsHnR0BkWlkNNXUQQ3RbPEc/LH7wv1XlUr5JB9ln19e0t7eSf7vNjq2us6/ATm4jnKH3lxFAAAAAElFTkSuQmCC',
book_open=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIASURBVDjLpVPPaxNREJ6Vt01caH4oWk1T0ZKlGIo9RG+BUsEK4kEP/Q8qPXnpqRdPBf8A8Wahhx7FQ0GF9FJ6UksqwfTSBDGyB5HkkphC9tfb7jfbtyQQTx142byZ75v5ZnZWC4KALmICPy+2DkvKIX2f/POz83LxCL7nrz+WPNcll49DrhM9v7xdO9JW330DuXrrqkFSgig5iR2Cfv3t3gNxOnv5BwU+eZ5HuON5/PMPJZKJ+yKQfpW0S7TxdC6WJaWkyvff1LDaFRAeLZj05MHsiPTS6hua0PUqtwC5sHq9zv9RYWl+nu5cETcnJ1M0M5WlWq3GsX6/T+VymRzHDluZiGYAAsw0TQahV8uyyGq1qFgskm0bHIO/1+sx1rFtchJhArwEyIQ1Gg2WD2A6nWawHQJVDIWgIJfLhQowTIeE9D0mKAU8qPC0220afsWFQoH93W6X7yCDJ+DEBeBmsxnPIJVKxWQVUwry+XyUwBlKMKwA8jqdDhOVCqVAzQDVvXAXhOdGBFgymYwrGoZBmUyGjxCCdF0fSahaFdgoTHRxfTveMCXvWfkuE3Y+f40qhgT/nMitupzApdvT18bu+YeDQwY9Xl4aG9/d/URiMBhQq/dvZMeVghtT17lSZW9/rAKsvPa/r9Fc2dw+Pe0/xI6kM9mT5vtXy+Nw2kU/5zOGRpvuMIu0YAAAAABJRU5ErkJggg==',
book_previous=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI6SURBVDjLpVNNaFNBEP5e8hJiSjVVTComsdYUMbRVUhSVgFawgqgo6FFBaMEeehQ8WAQFj57FCh4KVixCERXipQhCa6kaEz00uSQIghibNn++t7tv4+6mTU2bi3Rh+WZn95v5ZndWq1ar2MzQ1zuGHs85xwaPEIF9qz5uWbBW5vjIiY/Sd+n+qz5GKbT1CgRxnwCPmPPBHW5wLolcBTEJxfT7+RtccI5Fwg9RtdYU3Jwddgp4DVwfrXJrBpoNt87trwfmnCP2KYvU9z13ZObTB/04e7izoYRvFrP8qwspV45kMqlsxhj6u7uxd7u+q7V1KwK+NsTj8VoJIvsXn7O9Vx7K5rMgJkVpqQzTICjmSwrl+unQJDKZDMLhMLxerwqqC/IHr8PX29HSCcYZ/C1BhRVigHKKP1SgxTAx8QwyWaFQgGmaSl0qlYIuZFOmMRCLKCITh6lA0zIFkcJkZs1HmCL9e+mhUAj6g+ij6HDs2udypXLIZd+C7M8sfuVzDdJlSYyyBrK00+n02jNefX55gRgkyAo9I05ycmx5aRlTty/AMAxVKyEEuVwOiUQCkUgEgUBA+eqvIMg9IuNLe/H4V2arEeRwuVz1jG63Gx6PR01d1+FwODY20vm7U0ftNm1m8fciKCWidrqCNfti9IAKNv5mVvjpxlbWgB9yo2P3zqa9/+LdnLqPMwP9zf+ClC4zZgrFpgrafV7VWLG300qB9j+/sevKvSflcumUbOVtnraF9OTogLbZ7/wXRdt3lZxkvhIAAAAASUVORK5CYII=',
box=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG9SURBVDjLpZO9apRREIafDVuIhMjGhPwJukashNjoNdgIqQQbG8U7ECy0i4UXIMQLEKxtrCwsRMRKbBSCoBhQwRjwZ3e/M/O+FufbTYRYZWA45wznnXk4Z6Zjm8PYFIe0LsDDG/1pm03jy5gpAzbIxga3q2wMv2Q/uPXo8wZAZ/P6qVmbrd7iyd7cUh86HWhFMvvcpKBE4fv2B358+7Rx+/H23a7Nq+PL/d7c8ipf3r+kjH6jhDSkTAjCRoISZmbhNDMLq4S4c+/K8rmu8fzahYu8fvaEwc+dKm5FIZMJIVMSIsXu1ltmhw1nzq6x8/XjeteG+ZVF1q/dRKMhVqBInElG4igoApXxPlEJpo4t8eaF6drgEIPdd6j5g0KoqCYpSRShkq0LlZps+ugJZOjWxxEuSQ6zVohETZIh1LTiNqYQGTVmtwQqiUZBjgKVICfVsj0Ll7GwpYvcI1AkOSyUYTkQN4twCjWB0jgryYTAjYhRkIPyH1zVilETOV19QlCSHAQ5bA7GTaEUDuFxZ9EmsCGLOLJyvv5AGmvvstVWlGt/7zNjOvevrjy1uST90+8Hz4HBVYkrwfPOYcf5L9lR/9+EMK8xAAAAAElFTkSuQmCC',
brick=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFWSURBVBgZBcE/SFQBAAfg792dppJeEhjZn80MChpqdQ2iscmlscGi1nBPaGkviKKhONSpvSGHcCrBiDDjEhOC0I68sjvf+/V9RQCsLHRu7k0yvtN8MTMPICJieaLVS5IkafVeTkZEFLGy0JndO6vWNGVafPJVh2p8q/lqZl60DpIkaWcpa1nLYtpJkqR1EPVLz+pX4rj47FDbD2NKJ1U+6jTeTRdL/YuNrkLdhhuAZVP6ukqbh7V0TzmtadSEDZXKhhMG7ekZl24jGDLgtwEd6+jbdWAAEY0gKsPO+KPy01+jGgqlUjTK4ZroK/UVKoeOgJ5CpRyq5e2qjhF1laAS8c+Ymk1ZrVXXt2+9+fJBYUwDpZ4RR7Wtf9u9m2tF8Hwi9zJ3/tg5pW2FHVv7eZJHd75TBPD0QuYze7n4Zdv+ch7cfg8UAcDjq7mfwTycew1AEQAAAMB/0x+5JQ3zQMYAAAAASUVORK5CYII=',
brick_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJrSURBVDjLdVNLaxNRFJ6VP6GgG7eCuBAEN65LFwUXLlyIEsUqKLgQdwXBhaal1YiYNK8mNCXBPBqIMYJJE/IqWeRlkyYpDc2DBJMmpU4TSqNJ+nnPpS06qRc+5s6Zc77vu+fcEQAIUoTD4Tdut7tuMpmOCLSn2Fm5I4GVlZUxq9X6G5JFMYvFcuFMgmPFMlMbMsDlciGfz2M4HGIwGCCbzfIYfaMcyj1xxAkYe+9vtUqlAofDgfX1dQ673c5jEkc9TuB0Oo0MyOVyXJEQi8Xg8XiwvLzMQXuKkZt+v49MJgOqWVxctAhMqb+5uYlkMolUKsUVSV26ThwlEgmEQiEEAgHodLq+wOyJ3W4XOzs72N7eRqlUAjsftra2Th3RPhKJoFAowOfzcaTTaWi1WlGw2WycYH9/H3t7e6hWq9xuPB4/7QGpUhOpMBgMcpBbjUYjUgM5gSiK2N3dRavVQr1e56p0tGg0ygvJMjkjJ0RAAmq1WhTYbH8dHByg0+mg0Wjw4nK5jGKxyEdJyVSodSjwXHUXj97dxD35OOZML6FUKnuC2Wz2EXutVuPHIBLqBTkgAvr28dMsXtnv48uGEt9/eKHwPcFtxRXcmZ6oCWxM59jlmFpaWipSd5vNJtrtNm8mEZB92ewEXNn3cOU/8InMr05BsfoY15+ePzy9kkajccxgMLxms/25trbGSYiMGjg5fRVfN/T/jPVzRkkEoz+HXq+/zOZrZiM6orvh9/tx49lFzHkfQO6V8WL5N9moAylYh8cXFhYCKpUqOvnimvXW/CW89T7kyvSkd0Yw/18CKVjyDEOHbB8/Zyj+B1XaG3VPBqIRAAAAAElFTkSuQmCC',
brick_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ7SURBVDjLdVNNaJJhHH/D2T2IMaIOIzoFHbt0iWrdRoeCDh3boMNOwVp16dQORRCB31+4obW5CTKMmVN0c0ioEx1+MFMnSk4d7nVmWa/66/k/tFHOXvjxPvzf5/fx/P/PKwAQ+rG+vj67srJSmpub6xFoTbVBe08UlpeXhxcWFn6h76Ga1Wo9N1Dgj2OeuXUZ4HA4kEwm0e120el0sL29zWv0jfbQ3qNEXICpt/92293dxdLSEmKxGIfNZuO1vkRtLmC3200MSCQS3JEQDAbhdDoxPz/PQWuqURpJkhCPx0Ecg8FgFZiTlE6nEYlEsLW1xR3Jvf85ShQOh+H3++H1eqHT6SSBxRObzSYqlQqy2SxyuRzY+bCzs3OciNYbGxtIpVJwu90c0WgUWq1WFBYXF7lAo9FAvV5HoVDgcUOh0HEPyJWaSESfz8dBaTUajUgN5AKiKGJ/fx/VahWlUom70tECgQAnUmRKRklIgAzUarUosNn+bLVaODw8RLlc5uR8Po9MJsNHSZs58e00fA8uYu2GHJ/ujsD/egoKhaItWCwWN6kXi0V+DBKhXlACEuDOjBybvoYfzjfoJVfR+vAYoakreD81/kVgYzrNLsek2WzOUHf39vZQq9V4M0mA4rvuncd3Rsa7cWDmDDA7itqr63DdGsofX0mTyTRsNBpfstkebG5uchESowau3ZSjF7H/M9aDFyPsOEO9E/+CXq+/zOZrYSPq0d3weDz4eOcsvhkeAozUfiKgzlB4JINrTFYSBv1hBNbh2yqVyqtUKgOuyavWzxOXpK8zo6g+kyM7cQqe8aHO6pjs+X8F+hG4f+Epc8xTbPYuEpnqvwHnmxpBHuHptgAAAABJRU5ErkJggg==',
brick_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKZSURBVDjLhVPfa1JRAL5PPfTWyx7WS/QURA/9AREVRGMPvTSIRu1hLBq0oJdWK9hDWFshVDps0dWm06FzlplFP3abtiVOJlOnE51Tp+E2NvMHd2pX93XOYY0wRwc+OHz3fD/OOfdwALh6OJ3OBzabLa3Varcp6Jxyjdb+Q0xMTDQZjcZfqBuUMxgMzQ0NdhLjJK1GAKvVilAohFqthmq1ikAgwDj6ja6ha/80YgbEvfx3WiKRgNlshs/nYxgfH2dcXaMyM7BYLBoCBINBlkjhcrlgt9uh0+kY6JxytI0kSfD7/aAanucNHEmSwuEw5ubm4PV6WSJNrx+UC37nEdZfRNQqwzx/Fq+Hb1Y5Ui9XLBaxtraGWCyG5eVlkP0hEonsNqJz79dHyAi9yPnfAcUf+Ok345v8/DZnMpmYQT6fRzabRTKZZHU9Hs/uGQSmFdhc6Ecl9RaZqcfYClhQTUzDp7nGDpAZ5HI5bGxsYH19Hel0mqWyrU2pkF24h9rWNLZi17E524NFXScRd+GVSl7gyN1WRFFEoVBAJpNh4ng8jmg0irBHi2zgDmolF8SlDpRW2lEI9WHpZStmhTcYGhoqc3q9/rMgCEilUmwb1ISehXdyGBlP7474CkrJS8gv3EX0eQtm7KMgwVAqlQ6OXNM+8nN0jYyMRB0OB1ZXVxFxjyH08Rap7UQp3onyymXkSJOI6hysYzwVJgh6FArF/t1fUqPRNKnVahk/0C0G3/fB9+EJ0u42iPF2bM7fxqKqFWrlgEiEciI8uOdbMA2eAQoJhF9cgNDXjBlSWZCdhO5Z/yciPv7fx3S/66hUcSlRcT+FW3YKxquHqqOD3TcavcSGBm0nDpQedhyRJgdapC/y0w5b77HDe4kpfgO2GyDntdvjkwAAAABJRU5ErkJggg==',
brick_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKwSURBVDjLbVPPaxNBGH27m5jdaH5ZjU3SFkRiWkwt0l4qFupFEY/Bg/jj6MGLIPQigtCDf4BXLYZKK6UJSL2Ih4AGLGltTSUh1pgaJBXTEkNMY9NNdtf5Rlva2IFvv9mZN9978z5GMAwDrSMejz+sVCo3y+Wyl/5dLtd3h8MxPjQ0dO8/MBXYHdFo1D01NaUaLYPWJicnva14gT7/GK8yxi5WU2Rs8Pv9CAQCHJTJZLCysgKGIU6dKfrGMM+5IgKw6vXdbPl83ohEIsbS0hKP6elpvtaiiM5A6uvreypJ0gD9tLW18YLLy8toNptgh1AsFmG326FpGjweD3RdRzqdJjWm2dnZgIktXA8Gg6jVamBsYMWwurqKUCi0xyvmDaxWKxqNBsc6nU6USqUrImOs+Xw+dHR0gO5us9ngdruRzWY5G0Xu4wyCh9/BYmqA+QRVVakzpLZm2m6jLMswm82oVqs8k2GpVIq5rEIuvcbRrmHkc69gls/t6Z64XYAysSmKgs7OTq7GYrFAK8Xh8g7C7jkDpZ6CRV/nePKEF2ATRRRFUND9SB5lQRCg13/gQHUetiNOaL9m0N57A+byG+hak5OxszIpeJtIJLhkMolMpE3DYLEWw7EeZmZ9EXPjEzjk3IRU+QTLZha5XI4UJEQGvlwoFG7FYrEvyWSSM1PbhI0srJIGm+s39K2v7I6McWMex4fvQsu/aGz8LIywApeEbQ/C4bCbze+wuN0d8DudxWfwnr4A2ZyArhb+9lIwQzrYj2JWRCERfnB2ZGFUaH1MY2Njpzz6h3Cw+8RA+0mFsc4xqSoWohn0h3ogSA5AuYj3T+6vGbp+XtjvNc49Giz0XpvwSWIeRrNMPdq1K0JSAiimk/j8cjRiwj6juaXKi49Dm4bOek2G7oSxk8GNNtr/AJE93CiYMK0yAAAAAElFTkSuQmCC',
brick_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKoSURBVDjLdVNdSJNRGD4RRJfdJP2ReBH9eGMQGWVXwoggoosIKooMf0jQG2/EJFOstB+VcpvObX6ub7TNWTIW4Wxtc3PR3GTKVAyd4mA6xT6VgbWtp3MOKTL1wAOH95z3eZ73fc8hAEg6XC7XM4vFEunq6vrLwPYsttPdbQGz2ZxhMBj+IG2xmF6vP7IjwX/FMFVLUaC3txdjY2NIpVJIJpMYHR3lMXbG7rC7G444AWVf36o2MzOD7u5uBINBDpPJxGNpjtY5QU9Pj5YCoVCIKzJ4vV5YrVbodDoOtmcx5iaRSGBkZAQsR61W6wlVSkxMTMDv9yMQCHBFpp6+NhwNDQ3B6XTCbrdDpVIlCLUnra2tYWFhAVNTU5iengatD5OTk5uO2H5gYADj4+Ow2Wwcw8PDaG9vl4jRaOQEKysrWF5exuzsLLfr8/k2e8BUy97noVB7Fl+/2eBwOLjbtrY2iTWQE0iShKWlJcRiMUQiEa7KSnO73VyxuDMXTy23cUt5Cv32Pi6gVColQmf7Ox6PY3V1FdFolCeHw2GUi5dRpruAEuEcCtQ5ePzxJoy+FlSYrkPWdAiuQQdaW1vXiSiKNtaQubk5XgYjYb0o1eXiU0AOs/8dT/zga0ZzfwUEbyNKRBnOP9+PV28bPISOaR99HIWCIPxk3Z2fn8fi4iIe0npZ8pu+cjR8KUX95yLUWB6g1loMpasOdzvzcOIJSW4+Sa1Wm6HRaOrpbH95PB7cUZ2G4UczxO+vIQw2QuN5QUkeQeGqRZF4BVnVJHa0kuRs+wsdHR3ZdL7iDUUWrrYcQ35TBi69PIB72ouQO2tQoJMhs5pED1eSMzt+pg3QDssUCoVdLpe7abOuZdftid8X8pFZRSI0+eSuv3E3UMvx41V7cbCSZG2N/wO9JP1FmMH3ngAAAABJRU5ErkJggg==',
brick_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKOSURBVDjLbVLNS+JRFP3BwPwJLWY9BLMI5j+IGEmQWlSLdrPScnAVbapFRB8DUi0qIitSShIsKyJaTFL0BS5KxaKMhj5Ew5Ssnz+nyVI7885jksnpwYHLffeec959VwGglGJ7e/v7yspKbGZm5plgzNxbtf8lFhYWytxu9xNKDnMul+vDmwR/FS+EWkEAy8vLOD4+RqFQQD6fx+HhoczxjjWsfXEkCQR79l+1y8tLeDwehEIhifn5eZkrcZSVBIuLiw4BHB0dSUXC5/NhdXUVTqdTgjFzdJPL5XBwcAD2TE1NuRShlDs5OYHf70cgEJCKVC89L4729/extbWFjY0NTE5O5hRhT81kMkgkEjg7O8P5+TnE+3B6elp0xHhnZwfhcBher1ciGAxiYmJCVebm5iRBOp3G7e0tIpGItLu3t1ecAVU5RDZubm5K0O34+LjKAUoCVVVxc3ODZDKJWCwmVfm03d1d2UjLdEYnJKCAzWZTFfG3j/f399A0DfF4XD5DkKK/vx92ux29vb1YWlrC2toa+vr6ZM5qtXKAGBoayiqzs7NeskejUaRSKYyMjPAC19fXkowDY8PAwIB0xBxdscZsNkcV8U3vxXKYpqenf46OjqKjowMBsTidnZ2ora2FyWTC8PAw2tvbZVxTU8PG3yRpbW1NF1fS4XCUNTQ0+Hp6ep5JcHV1Bb1ej/X1dTkwvl+sOaqqqp66uro+kqC7u/vh1V5XVFR8E+waCZqamqDT6VBdXY26ujoJxiSor6//QYLm5ua7VwSfxBG2w4ODg3fiezVu29jYGFpaWtDW1iZj5sRdijWsfUVQXl7+rrKy8rPBYLgwGo2/LBbLg8BjY2PjV4EvItaY4x1rWPsHDKf5+VmuapYAAAAASUVORK5CYII=',
bricks=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALLSURBVDjLpZPfT1JhGMfprnnd39B1986rtja3TEWGBJocEYbya+oBzgJRt8xwKjoCf1SgqKSiGUqlYiZzGpY6XebMXFOxocIFK8p1c76d8zpXq9lNF8/27H2f7+f5Pu8PAQDB/8RfC2/u1WVErHTmS1oniyko7Bfmk4iVlmCG1t6KWOhMvuZcwCpTHU0a9EioVTjiRCmGQYqmSZ7QlCNZacCyuSp6LuCjVMR+sVpx4uzAkboMMYmQBJ/za/zeB3EOGzbqLhNAndeWYXRVZxocWpl5UI9d4XWkfT4ktRU45kbgRWewY5UScaqIGykXHOAKAdz2GKMNIStsYyYwAQP2RDfwIxjCN7cLiQoVsZys0uNzkZSI+VH2Jfm/AJqeMtYxfRe+xS5YR2lC5wFpdwfSTidSLc04lBcR8SFVfAr43YHSW4zZzSk4phvBjFRi52YB6Zx2d+JreztSTXbES2REyAMOuO5bomzWZHNfNTf7MwTqHjmWPi1gaiOE4aV+dAfqMa+jsCvJIzfBu+DP4kAmwY4wG7MKOWrtflR2R6BpGIgKyntLsbA9h4m1UQRXA/BHPTAN62CzuBClpNwN5GFXnItoiQR3mDao2sJwTKzD4ltELjPECjReBTu2MoSRZT+GlnzwLnSBHtIg1/oET6M7qPe+grJlkoj44AHq9hnIm15AaAtCUOXQZ2nd6ri2T4WuOSf6X3vIWeTbxjCzFkNgfhueyXeo6XsNQ2cERfbnRFzaOg1RXfD08ZycnFwwtOkUSheVrvApUN5LEfrkyh4G5rbwmItH0xuQN08SMcU54gHEwZ8vUd+qvaR3aC+Ka0dZb/g9POFNPOQc3OfsF9tPO/MAcf04coyD7Lm/TGXtyJJZeuISrtAeeAtXaB2K1jCkjc+QYw5AbHyQoEzOa//8qvxo6pruskKmNy2uG+csj6HA2PtdbnZVn9X8BLLktmdBdpY6AAAAAElFTkSuQmCC',
briefcase=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKrSURBVDjLfVNNTFNBEP5aqlItmlcoqG3EP4QQhFitxFNREwwnSbx50aiJBwkQD54gxsTgXQ+K4QoXogduRGPExCAaCCDRQhBCtaX8RKCl7et7u2+d3RZSjHGS6Uz3zTf7zc/aXtw+KvCXCDqx6EeplfMtAW7JMyirfFKHBFy71Yp/CqEFBQlOyixYpILznM/R09etEkRWl6Jed7EGPTaRA0pQDmByWAbpljUYCpzF2FNWKfNzmeDBXOhrb0nwCr6Pz4MT3Q9TCwhU+EAYGHRThgn8iMZQ5vbAMC00XQpgavKLLO2l/W7PfF94dno0PBuC70wD9OS6IpFKbSBJfjIZJz+eO9uEt+ocEmvLGAuNr1CCDnuWsWif+TYJt+aG3eVRwWlTIG1YSBlCqRRGFVeVlGJ6PqQwTwaWf9uEyH58eqO893Rd/XXnfg90XgCd6jVVtwnIs10vcjjAVhcwPDUyROAGiVNT6L5zzLXLYds7Mf6Jcy5szBL2/FHtGCfd6NztGNsalI32wE12tuTQcc3jrdg5RTnzvLlLNobJsBiexmp07vHDV4udksEIAbVS3ykszXwGN9LqJglkOfrmtgJFnnJohythMqujs/lgjUzgqaz1IzT8Fhnq+uSvOP4naxtJuNMMvhNnsRKda1Y92E9LFLjYCJ5Jo54xWiJTLQ43DbIspwb9l1tI32gb92kujBFThxoC1WfEw2B6EsPJ89gULnVbdfojCvU1vGEBbKZ02oMUHt3PMlx5n30zDtVZKtZZWgtOtvBnIS43XlXBo6NuRCIRXKirQzAYRGvLPcTeeVFgt22XZHt288g6jejA1otDcQ2qm9rg9/uhaZoKSiQSGBwcxNDr59AyIYi897u9SPnS1dXVput6C7E4KZmQHyMdME2zvb+/P50f+wdCqc9c4Pf4aQAAAABJRU5ErkJggg==',
bug=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKYSURBVDjLnZPJT1NRFMb5G1wDHV5boNiqdHrvFYolCAtsGSSWKpMFKhYqlDI6oAEKaVJwCIgSphaKtLYWCgSNBgRjMNHoxsSFS3cmJmA0NMTw+R6JKKZl4eJL7sm953fOd3JPHIC4WMpcppG5SGnZc8ZjVVF6QLn975sDgfaZmvg71oRJZIRUYcuAnq/2KWroGfm3QwEn2YpLVPPvOD2oiqj9yq/mGznegl56mx6T7ZbY1M6YAM0CuZkxT0b2Wg6QW/SsApRXDsotR+d6E9Y/h9DuqoCuJq0lKoDxqU1/pITGR27mBU4h+GEcTz5OY+ClA5JbyahYzof/9TBO9B/FcWcqpA4xU3We3GJ87ntnfO5meinMvruNnqcmXA2XoDVcCc0wCYkzBaZpA7ILRJ/2O2B87jA+QT9UeDRe8svZYAG8b/txc6kc9mA+yqayYPQXwvdmBEOrA5B2p0BtFIYOWKCm5RukWwZyXIbA+0F0LpaiKaBHmVsLw4we99ccsM8a8GClF5JOMcQdou8prULrgRmQo7KI0VcE13MrGv06lE5kodhzGvdWu2GdKkTVWC4DcELcJkKyXbCb1EhAVM//M0DVUNqP2qAJd1baUDaZjTMTeXAttsPi0cM0mgvHvA0NkxYk2QRIrieOsDmEmXttH0DfVfSluSToWmpD8bgOroUOWNw6VI7koGfOBuq6EqLLTNU6ojrmP5D1HVsjmrkYezGIrlA9LjKgnrlGXJlpgbCOD0EtD0QNN8I3cZqjAlhJr4rXpB1iNLhrYffUQWoT7yUKzbxqJlHLq0jc5JYmgHMunogKYJVqF7mTrPyfgktMRTMX/CrOq1gLF3fYNrLiX+Bs8MoTwT2fQPwXgBXHGL+TaIjfinb3C7cscRMIcYL6AAAAAElFTkSuQmCC',
bug_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK4SURBVDjLfVPtT9JRFPbfKNssUVQUSgH5AYriTNcU3xey+YIUaqilYr6ks/INWQjmJMv5EigmCJGomeg07YNttWrl6kMf2/rQ1qZWk7ny6fdzZaLS2c7OfXbvec55zr3XD4CfL4+bFSBumi+m1rEPeVkx9uitg2e8gHheSNuPRS4CsU6eS6VPCxRauUrBGGfzvwTxVMUZfs1fHOPgeYS26K/F15MsGR2CLcEgeye3Sqj1SSCaItZjJwnPbst2YkMwzgXfwgHfxEHzSjVWPjlRryuERBl19UgCUqc4ZiIaIiuxnmI/C8faEJ58GIX+WTuYhlAUzqbD9rwPkV0ROKMNB6udQVadJDZInXvaSZ07cRY+xl91o21OgUZXLmpdcoj6CDC1YVCMSpGYQf+41wGpc5vUCcEDrllkIb6cd2TA8rILN2YKoHakI38kATJbJqwv+mFc0oPVGgahLNjpJYE/ylklTGwQQ2zYX/eieToP1fY05JvEkI6l4e5yO9TjUtxb7ACzmQFGA/17WG1whdcMiAG2R2bNgm6hApU2CfKGE5BjTsWdpVZUjGTiwmAySaAFo46OUHXQTkglDfTywH8D5BmjfpQ6FLi9WIf8+4nIHk6BbroeKnMaFAPJuDaixGWjDMX6dMja4r+dq2c10YpPNu0RCHq4migdEy0zdcgZkkA31QCVSQJ5fxJqh4twc6II02tGvPnshmFeBamBCV7ZMZ3XNbI1p5dpNScx+LQXLc5ylJBEbY8qUdCZhMl33Zh83wPKdAtKGBZKKIKtQ++f1chYZjUwcMVUCrW5DKwqBlIbI/F4bQD7zfXWSBEc/YnC1XRTSEXgz6BLp0DqBFHu/+uWWwGNW76brJmT++7goAcWBaxGKP2RrQtFl/vibmUqUvjQDHx5gOzEdoTiuIdM2KTa/hM7qb3flzV5wwl+LjkAAAAASUVORK5CYII=',
bug_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALWSURBVDjLhVNrTJJRGLZB/a9ftbl5R8WgEpEPUFTy0jS85UKbpuElUykFE8mmLW+joZXTLOclUMwLSKKkpmnaD93M+tPqR6vWdP2wVdo0yTmePtwyLbWzvTvn3Tnv8z7Pc86xA2C3UwQM8BDQzw2yrf17OLF+et+Vv89sSYKG+S6bc4GJgL+RY8qqjnTmd7LTee0+33cFCLR1NHPzf+d+Bo6F3+X7JaMkVBddwVvhNbGs8Xl81Y4Agj5iwb+XsKxT1hOLvA42uDofcDU+KJ6QYWLWiEJ1MkTp3pe3BSB1Bvl1+0LQSSxE6ENgeNWMwTdtqH5WDsZNDyQPRKFrqh7Hqg7jqMoLzHI62bWXWCR1bmgndVoDdFx0vLiFsiEJikzxKDClQFBPgKHyhKRNjOBo2tsNBqTOVVIneA/YWoGOmD9tiIbueRWumc9CbohCUqsQCV0x6JxuQN1YNZilnuAnuBm3SOC2+UwSGhaIZhb0L2tR3J8ImT4SSZogiNsjcXe8HPIOMe6NVoBRTAddSVvyLHCTbvGAaGRZEjpjoR6RIrdLhMQWIeK0J3FnrBTS1hikNoWRACrQFTR4yF2t7rkuoOU4/zGQU+e9nGmQ4PaoAkn3g3GqJQLq/kJkaSMhaQxDa2U4HokPYVhIxWAIZW4ohFLokuF4dQOAV8Ou9FYzcN2sQFyzCOo+JbI0IqQ0hEJbEY4ZOYEVcxWsrwex3CHDdDZj7fFxqnTLNbIqj4y75Dui6WktrhtzcJ4EKnuYi96YA/hBFqNGBCj2AxVO+HwjEEPBlPf/vH9mEX2cqaTjkiYTcm02mHn0ddrWmR5sHt9KDmJESLVu+4m85DSNu9R5zfWCE0id6D6x7+dSUypAFlkK7PCVjI+ZFJA+zNnt9htt4XzOYVJ2xh5TaW5rnxROmFfuxbu0PXgioq6RZl75L4AtHBLsVy+K7S1kxw822uQ8ayu27f0CZyF3rEtmMUcAAAAASUVORK5CYII=',
bug_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL7SURBVDjLdVP9L9RxHP/8A/1glR6WlpsUh1Me7nvuvs7RHDaZp2XkoR1WQlbJ01J5GHbOU4fuPM1TK3Uh8pCHIs2wlkgphJ70gMTcZbJXTs1y6Yf39vq8ttdrr71f7w+hq9nEtoVHbO/zCP8eh9CtVoTbxnnqVM33DcpzJsZtpl8E1yktYQ1NDiboEwAbhqgN1GDVRGLTzGGqMV/BfuhQSSH0otBuW7E23DyMaE3huoF1DbUG+I1sCOp479SYU8GCRSETzlIOFC/ykJEZDq4fQ3tTA34dZ/1BlTFBFbGQ15WAW91ieN9wgalMFzf7MmFwZT+Msw3xb4JaDni32VxBifkWbqEZsh/E4GrzKQSUsHHy1jEcL/eEUGwJoY/+hU0T8BSWJ2xKLWAhN4C8MxFZTcHwlpvCudgGyY1nUNweDzrVDKxI3bE3NYfdhyu9MFKbjP4iezzLZfoR6wYOoWRG+d4VLpC1RsBLfggucjuIm6IhktGQtkbCQUwjLIeBqfYozA3UAwsf8G3gNrrTHT4R83QWsco0CfQpPQpJSwSEUgqXFCEILOAjvMINIUX+iMrSfz07FL+09O4uph6KoRy8g58TXRgsCVYRXs7vGk2SGJMu2fYoe5SBiHJHpNQH43zlabjHamN6MA4ryi4ox0Ix0xuGl+UiDBT4TvfmuOoTbo7l+kIOnNP56pfvisuKcDgk0QhK0vs4OxiDFVU3Fkf9oXrrg/kXsRiX2Xx6IjliurbEvw3Us1O0tWpnwI4Wp5PbMdUX9UfsB9WkN74/j8Noru38q3RL8/UW1AZ0LkXM0ljEONqQ6IXuJQ2Ju9yHGs8uryg7oRoX4cdbX8ytJhnJtVMOp1P7NtSo2WvhKR3RUEMsnjVm4X2PJxbHfTDTH43XUsHnYQnF+OcONImqNDtgfgLDcg+0x+7G42uO6Egw/zCQyjba9JA0icQg5vJStxRLPdnoSRagKnDPQnOMCeu/n0mT8KS1VCn+B5fbUh2XWyW2HXVRxoz/idXzCw2TfDKYV5fXAAAAAElFTkSuQmCC',
bug_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALbSURBVDjLfZHrM5RhGMb9A33uq7ERq7XsOry7y2qlPqRFDsVk0c4wZZgRckjJeWMQRhkju5HN6V1nERa7atJMw0zMNLUow1SOUby7mNmrdzWJWn245nnumee+7vt3PRYALI6SZy8fnt08kenu0eoW4E666v9+c6gQDQgYB2thJwGPNrfOmBJfK0GTSxT/qfP3/xqcNk3s4SX9rt1VbgZBs+tq9N1zSv98vp5fwzWG3BAUHGkg7CLWPToIw97KJLHBb3QBT+kMXq0zMrQJ0M63IbUoAuIozk2zBjSnyL3FFcImYt2HPAvVlBx97+pRMpoH1n1bRPT6oXmsEk7Fp+BYYA+HPCY9tYPYoDn32WlOo6eSh8bxUuQ+lyK9MwTJnZEQVhJgFdhBWn8Z3v42uv0NaM4dmhP8Bpc6oZJYuqTyh/JNMTJ7wpGo8oPkiRfyO4IxOXId1cOFcMixgyDUuu0QAq/e+RVRywUh54KcqEBGdxgSSF9IakUIb/DD24FIrOpaoO6PBSuDCWaazaZdsnXcoQyIR1xDaFMAigbjEN8sRpjCC0F1F9A3EIdlOofdzWlMtgfDN5sN28QTxpPxDNjEWv0J0O0BZ+uaSoqyoRRIHnsjUOGDqu4ETLRehGG5G4bPJVib6YHioRDiVPvjph5GtOXtfQN+uYuMU8RCdk8KguRiFHelobVBjJX3JAzz2dDe42JnlcSE/IxxvFoUaPYbuTK2hpFkiZqRClSRUnxUp2N7qQ7U9FVoZU7Qz6VgffYZBkuJxddlxLF/DExySGdqOLfsMag4j290cPpPSdj6EPJLOgmNUoo5TTnac9mlZg1MypJxx+a0Jdj+Wrk3fUt3hUbg7J3UbAyoLx3Q5rAWNVn2TLMG9HoL1MoMttfUMCzRGSy1HJAKuz+msDBWj6F0mxazBi8LOSsvZI7UaB6boidRA5lM9GfYYfiOLUU3Ueo0a0qdwqAGk61GfwIga508Gu46TQAAAABJRU5ErkJggg==',
bug_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALRSURBVDjLfVNbSJNhGP6li24C6UIiDHWah83UdJv7d7R5oiiPWGp5yIUHNNF0acNjrmWe5gx1Ujmd+pubW8upaSWZRilZdhN00WVBEAgalSK5p18hczm7eOF7eb/neZ/n/d6PAEDsFWETAoSN8aWbZ/EDXqLIFLr67x27RPpU6LUzl1hJiC08a7461lNo4GYLBjnf/ktwYrPjOF/+JxeZeWtCY+hSTk00FX9TsCroZttSrggb9iSQjJLL4hFybUuyiVwRDHHBpzjg6zmoni3B7CcLrjVnIiY75KpDAtqnVDQcComBXD5tioL5vQ6THwagfqFCQKsfMifiYJzXIrjlGI43+CNQxaK7jpArtM9t77RPWxjFx9CiBjcey1BhTUGZ9SIkWhIBDUzIBpIRGe/zcVsB7XOd9gnBfW6fhCK/njPHg3rTgtrxDJSa45DeH45UYwIMC3fQMa1GoJIJYaq3xc4Cf4AzR+rZIHVsmN61o3osDSWmWKTrpUgejEXXjAoF/SSye4IRVMMCS+HznVnmXWg3A/Ieey3VkIjmqUIUG2OQ1hOOpL4z6JxWorA/AZd6QnB99DySu/zAlB+1+RZ7weey598B8jpCfuSZZWh7Vo703kgUUWIaSCKXVpbVHYRKy1kYF9ogH45DVOuhDXHTwQNeOR6V2wSC29z6kOYA1I2XI0kXQ0vm4eFiJ8xv27eAQwsaaKbk0M81Io+KBrd+/0aA0snZ7hnZ9UEzXnIPdD9vpyUHb4HVT4rQMFkA1aMc1I5m0Q1y0TWrRFqvCN41xK9d+x9YwZoJVLBw4S4ThtcaUPMt0L9qhO7lLZokH9rZOuRQJ8GoIpZcFcRxh5/Iv9RHn6Bl4FSbKyJaXSBsckZGjwCdM7WQ9UfDvYr4clhBsHZtoqPwzHKfY2S4wV/p9DNTHwH3CuIzDfZ1uMp7hXvqkXVGpZPNrWIfXBQEY2ftN8xTb5GsXWfEAAAAAElFTkSuQmCC',
bug_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALhSURBVDjLfZHtT1JhGMbd+kPaNElNnCHqARQ4TmnjRYRVSPk2hShMJ5IK2hQxdDqxZBnVkuYLpgg4TaeuzElutdXogyw/NNeHvrg258t0mCuuzmHLomVnu7frvp9rv+e+nhMHIO6kEsxzIZjlkLTO9Wcrc7xZ4b89MQ35gpfwZ8+fIZA7lT2j7yuM501karlj7L3/AoT0jXOcW7/6HF/2Ic+TtXWtTeQu6uSGuYMZkWIDr/tEAP85sZM7TRxGV/YSu9zxTHDcbHCG2GgN1CPwZQqm3nLItKyGfwKonGTOZBb4E8SOxFsAX8iFhfVR9L22gXk3CeXzcnjeOnHenor07nNIs6VQt04Tu1TO4+xUzojAzcF48B7uLFaiZaYYjTMV4DsJMLuTUTmqQn4R49PxBlTOIyonuM8yh/lu4utlXxHc7+2wzJXB6JOjdCQPao8CE+8eY2C5D2kdyeCpE6diInBG2W+IoQwQrgx4P9xH62wJ6r2FKB0ioRorxMMVG4zjKjx61QlmawpSzIz95MbEmpg3IJ5kHKonlOh9WYM6jwwlT/NwcViKB8sdqBlRoGrwAgXoRkoTA0nGM5GzdQlg3Iz//YDZA6yD675K2BcaUNV1CdaudrhcLjRbmqBxyNExaUBze3N0ZrPZtvV6fUNiVYL1GMB1ZHaxepjQ3q5Af38/Njc3sbGxgZWVFbR3taG6RY/Qx1B0trq6GvVIpdKemN/I0qWHao21CK6tQWMog0QqgVarhcPhgNlsjmqJRIyiK/J1GmI0GndiASxWj8Vq+UEDal06kPkklpaWEAwGEQgE4PP5wBfyI/FXT0e3sFqt4RgAk8m8odPptmmASF0QEeYJIRKJoFAookVrkiSPlErlIg2gvTGAZOqTyWTrdrt92+PxbPn9fjidThgMBphMpqimZ/QZ7aG9MQAGg3FKIBCki8XizxqNZr+6ujpM1Td2Met7mir1gNJ79Iw+oz209ye6Km+xbu69pwAAAABJRU5ErkJggg==',
building=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG+SURBVDjLnZNNSwJRFIbfGccPCKJMoqSwdBcEobUQWrmIChJs07pVuA36By0KWrTtB7hqGSQSFBGCbdq0ECkhSmgxldmHOuP9aO4VpSEV6cDhHu7c85z3nLkXnHP048/RKO+0r6JPO9nc7LivCErL8vk8p5SCEILWKlzXdWSzWfj9foRCIcTjcaWVo/2mNRoNBINBGf8GizgWi8k1k8nYFNgAopqwi0KzM2oxKAMYZ5YiYHmGwDCM7gDTNGWV8Fi54yCBof4AudKwrEpYUwGxqjPGkZgjqNfrvQHCFsZfuvzO4d4KxEdx8PLBa/XPpQJGm7PgloKN+UZvBRX6LgGLk3oXBV588c/OgJ1ccso15MEKX7X6ZUilUlBVFYqiyDWRSKBivONxvGgDtG+ilTStOh3IV25lNU3TEA6HEYlEZCyg5VoZrgEX1o6W122A7ezWIKhyPuL04ayUlodFkvCWihqpYv9qFwFfAMSkB0uHsdk2gBKaNEyT3+kFfL9Wkbk5BXMyOByONiR9nYZWd+Ht40VcOD8lbLc9A3rs2ZMPw/IqKAq4l/JKxZKtX68yitenMibMgNtteuJ/HtN/7AcakErwOe40HwAAAABJRU5ErkJggg==',
building_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIJSURBVDjLjZNPSNNhGMc/7+/3Ww5LmRaho0AdQgWpbRUqHmohVIdBp6Cbp7DWpQi8VyRBt+jQpdNuHYLAWEEiORl0CCsbIlGH5YKFbXPE/rzv+3QYipIOH3h5eB94Pu/3ff4gIuzl5EZGZKe4wx7t1cTEjnElIpuXTCYjxhi01mx4rTX5fJ5UKkUwGCQUChGLxdRGjreVVq/X6evrA2ArWESIRqOICMlkcpuCbQCtNQCzy42fGQFjwYrFGLh4QlOtVncH1Go1RIRw158dCwmBvQHS2Q6sWLRtKNAGrBWuDGkqlUpzAMCZ7t+7tLOjuYJqtYqIMPejEyOCtmBNoxZihaun680VFE0BEWHsaP6/1z98m2P6xTyl1gLXHpwvFcprj2YeLt7fBNxNT/bsC/i5JJex1pJIJHAcB6UU6y05Wnv+cm7sLEc6+5ldetmW/vL+3nA8GNicRGttr+NzyRQ/IyJ4nkc4HCYSiZDTKwwdG8Q4hsHucYyqM3xyFOCmA3A7db0do94d9B3ibfY11lo8z8PzPBzHYa2cx6cOEDt+C4A7F54ROjwA4PcAjDaTRltZKS+r/bV2ktkZrM/iui5KKQqlNZZWF1hcnWdq/DnTbybwuy0AFSUixONxabZIP/mK25tjdGCM/q5TrPz6yMKnebLfi4+3LVMzG44Hp4EbQBuwDjxNP1md+gdPcFmX7csAmAAAAABJRU5ErkJggg==',
building_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAILSURBVDjLjZPNS5RRFIef+753ypJEs4UJftOmImosSGgREn0tBBdRLV2EGLooCIoWbSL8C2rdQmwXGBUzQmkx4LIPaJLSQiYtRkyz0Xln7j2nhShKOnjgcrgHznN/93ygqmznzLS16WbxgG3as66uTeNGVdcu6XRavfc451j1zjmy2SypVIra2lpaWlro6Ogwqzl2Pa1YLNLc3AzAerCq0t7ejqqSSCQ2KNgAcM4B8Hp85WdewQuICt7D+YOOKIq2BhQKBVSVeM3vTQsJldsDjGWqEBWcrChwHkSUzqOOfD5fGgBwYv/sFu2sKq0giiJUldHve/GqOAHxK7VQUS4fL5ZWsODnUVVO1WX/ez33ZpDxJ4OcnPpKcvTeFPDwbNL1rwFujfU07qgs44JeREQYGBggCAKMMTTMvudIbIb4pWvsbDrE8odk3ae3w/eHz8Rya5MoIk1BLCS98BFVxVpLPB6ntbWVyokRDpzupGxiBPP4Krsnn9LYUB2q0RsBwM1UdwXevKqO7WM48xIRwVqLtZYgCNiVm6espgn6hqB/Du5MYnOfCcU0WgDvfI93ol/+jpvyQgWJzAskJoRhiDGG5fI9LL17TvnQdaLlnywBi39CfMi0UVV6e3u11CIdXkpzLMxQX13EBhkWs45vvywuL3c3LFMpS12pv52bm+4OvWnwof5QeHQu6R78A6o+ZiAxKKNtAAAAAElFTkSuQmCC',
building_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJtSURBVDjLjZPLa1NREIe/m1dr1aY22tb4qEkEX4hYaKG+7UIFS5auBOlG6bbgHyC4EHEhKCjiVhBcaVWsBSVgikLF2iqxPqI1llrTxiba9N6Tc+64CAaDFRw4DDNwvt/MOTOICP9zpjo7ZbG8h/+0/p6eRfOWiFSCVColxhi01vz2Wmuy2SzJZJJwOEwsFiMej1u/7/j+pJVKJaLRKAB/gkWErq4uRISBgYGqCqoAWmsAHo+XOzMCxgVXXIyBI1s1juP8G6CUQkRoa/m+6EOq7Cgd3vu8v5OR+ZkElj15YlHA0y8rcMVFu+UKtIFmfY+dnhe0bD9OMLKTuY+bePPw+vm/AADtq2eqlK1CgqB/gqX1u8l9SBGwFMvrW1gWWh+sAjiOg4iQ+NSIEUG7ENJD7A+nCUYO40zdJFBnMTHyhpKtZu2Fn7uqAHkzh4iwZ122rF54RqNvjGAsjj15FU9A41/WSm1xbLr403ekvW/wbWWQTj/t3fC6YRgRwXVd7t84w1L9kuDGOPbkFTz+EqqwgczjZ6gfxe62vsGRql9wXTfi8XtJ5ceo+/yOg+2raNp2FPX1Ot6AYOdb+fbkOdeCIT54fev7YRgoj3Jf8lQ9xnoU8q9kKHGZgJ3GsI6psUu42Mzn1jA99IoLfi8NzTG0MhcOXezaXgEYbXodpeRddpzW6Rxb95ykJvOEt7eTjAzkeNGf5IG7mYJaQq4wg9Y6bLR7ttKCuVV7DsACxtU06vVdIjv2ks/Okh5OMVp3gMLXIo1WE7OZ76xVrTU1qjb+1zIBHNvXuNAWbfZ1bIkgfmdo4Vu2p/vcaPpfW/oLvSN/oHdKKLQAAAAASUVORK5CYII=',
building_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIfSURBVDjLjZNNSJRRFIafcRJSnGEszTQddQZUJCGTCiGjXLkaiIKgTUhECK6K1oGroE20adEmQnfVIqEYhKRgyiwlFBkcqVzYHzNiVo7f982957QQpxk08cDhcA+c57735R5Uld3kt+5u3a5fwi5jtL9/275PVfOHZDKp1lqMMWxWYwzpdJpEIkFdXR3RaJRYLObbnNlTSMvlckQiEQAKwapKb28vqko8Hi9SUAQwxgAwPr/xMqtgBUQFa6Gv3eC67v8Bnuehqhw9uLKtkRDaHWBiqRJRwciGgko7Tb0dI9hxBcdxdgYAHKvN/LvZOujSY0LhHrJf4jsrcF0XVeXl4j6sKkagMTdKZ/g4wdpO0uM3KfOFiwBF/2DV/kRVOdmQ5lRDmjM1szS4zwhUhbC/nnKg/SLR4Cdmh/tKtwBuTAw0zYXeo6qICCPDwyy8uU9161lwppl8OEJFaJ0afwbjeLEtABFpLin1k1ydRVWpDfymrfkQgcos4n4GFeyfdzSfvs6a692ZvNsdyAOuJa4Gsb4X+0urGFt6jtgcLRUfCYWPINkZVNbpOteKeF/ZuzdFuONC/XfP3MoDrLEDrufpQnqeteUsM2/v0dTWQ3kgg5pl8PmZepICBHFSVLc0UuZ4l18NHW73qSqDg4NaaOb5yAwnLj3AX7KImhVAi3z3l7XyY+4DqdGhR0XLtBmvb3dl1Eq5iqIiBan5igiqOvUX9fhxNiezxvQAAAAASUVORK5CYII=',
building_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIrSURBVDjLjdNPSNRBFMDx764aarTaZqmb9mcXDyaF9IeS/kBCUR6kbt3MU9gxCjp1iujgpQ4eutS9SxAUS1Barm6EYnRYTEPNpaAtbXPd/f2Zea/Doij+wQfDMA/mM2/eMKgqWxk/29p0vXyQLcbL7u518wFVXV6kUim11mKMYWk2xpDJZEgkEkQiEWKxGJ2dnYGlPaUrNd/3iUajAKyEVZX29nZUlXg8vqqCVYAxBoB348WbWQUrICpYC5cOGVzX3RjwPA9V5Wjd/LqNhOqtAcn0TkQFI8UKjAUR5WqrwXGczQGAE/W/15z+OHmd5EyOJu/axoDruqgqA9NhrCpGQGyxF3nf0FDTzIj7lCt9fRUvbk4X1gBZ+xdV5UxjhkfDXaj4eGLwjCESbqK57iQLziKf08ncud5w1fvbc7ll4E6y58C26nIuawcighGPCy1dWBWsWATlR3aWww2nyXmF4Mj3geyR+8HwMiAiB4NlJaSyX9ijERzjYVWY+TOBLwYjPr71+ecu0Np4lpyfD36cGvwTBLiVuBHCBt7uKqvhTfo1IoJjHIw11Ib2URfaT31VlLKSCnbv2MvobIKhb4n5vMfxUgBrbI81ohO58cB2L0Q8/Yq8cXg23IsnHo5xidW0cCp6kU8zg/RP9mttZevY0N3RsSLwvPwhQADIYxlnkggdq57rQ+EJgWAFQ1+HOOZ2BMJztefXfKbNInovsGilpLIgNvrrgU4t5f8DTGqAX1cDO6cAAAAASUVORK5CYII=',
building_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJTSURBVDjLjdNNSJRBGMDx//vu6mp+pKaii4mupCQUIVgKgWH0TXaumySFp8Cog9coooN000t0EssIokgzoQiyLEiQlHU1ykDxsOqquKvvuzPzdBAXFzdwYBjmYeb3PDPDICLspS80NkqquM0e25vW1pRxS0QSk2AwKFprlFJsj0opwuEwIyMj+P1+qqqqaGlpsbb3eHdq8XicQCAAwE5YRGhubkZEGBoaSqogCVBKAfAxtHUyLaANGDFoDedrFY7j/B9wXRcRoa4kkvIiIW9vwOhcPkYMymxVoDQYozlr93A8a4yZ7sfXD7VPPEkJANSXLu7Krn+/IjNrgdrLt1n82vVo7EH1fF3n9LskwHEcRIRPswVoEZQBo6Fk9TUNRePsrzlD9O8gaTk1BXnlzsvBW4G2JGBVryAinDwYTspuTX0j78gldGySjMIyNvUcpU3H9qlopDsB3Bltr0jPy+CCXMQYQ29vL7ZtY1kWR7OXKamdxfYsYtlR0nMXYTPG+oLjSQDGmEo7zUNw9SfF4sfr9XLicCFl8hkrTWPUJLZZATsHibtMPQttTobWbngBOkZu5lrG+nAgs5DhuUGuFrdSmTmHP/qD8qZziPseSy8zPrCGtbGEaKW/hOOdHX1/+mwArXS747oyEw4RXYoxNDZAbUmYwOkrqJWnWO46E8Ob9C9n8ryimns+n3pbUXwq8Yz6RcZDAAuIoQnxi4b6OJZnHo+viPHBCAPBIpy0UlbGIxS7lT6fm9Gy6zPtbN+76u9nF+S3xZajG9PT4bvXeqb7U637BwW7d+CBmtk1AAAAAElFTkSuQmCC',
building_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIuSURBVDjLjZPBaxpBFMY/E5u1MdZuNFgVaqsHoSCE0BzEm6eehFxaafGQQ1oCtodCj14tFq8pveVP8NCTUForCLnYSwVbexHrrhFtm4jiqjuznRkxRDDBgeENb9/83vfe24FhGFhmt8JhY5F/BUuuj/v7C/0mTpmtarVqEEKg6zpmlu9Op4NSqQSPx4NAIIBYLGaa3TFfpk0mE/j9fnG+DObnaDQqbD6fn1MwB+DZ+Pryc1oZYQxCAWpQpgh49EDHaDS6GjAej0WWnTv/FjYSuL0c4KQpi6w6nSrQWXZKDext69A07XoAX7vu7hXjlK9XwD/ywK/1TVa/IRRQMu2FwRQ8eTi5XsE5OcNwOIR0WkSj0YDD4UC33UYkEoHNZsPx8ScxynQ6/VdRlLeVSuX9BeDNyeE9s03Cea4nxplIJDAYDNBsNlEoFGC32/n8YTab0Wq15HK5/I4l2boAUErv/6mdAb8bePz0GVKpFA+E2+1GKBQS52w2C7WlYsNh/fby4NUOAzwXgNelF7dM1PRZq4+xvrEugPyHqtfriMfjkGUZ/X5f/JFHH45AtkebPIapkgSA6OSQ6NTor/VM6qkiALVfNa4KmUwGVqtV+HhJYI1d+37zLnYB1gdNvIVkMmnMAnq9HsLhMLxeLywWC9qsibVaDZIkwefzweVyYagNoSoqisXij7nHFAwGV1nNIZYxx7q9xS6tMvdKt9s9YFZxOp05Zm+wcRNVVTss4d5/6EF77ncyqQUAAAAASUVORK5CYII=',
bullet_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACwSURBVDjLY/j//z8DJZhh1ADsBuRPiazJmxLxKa3P/39ki8Mnzwq9GqINyOgPbGhcnfh/y5Wp/y882/W/f3fW//B+3f/m2ZI9RBkQ2+7yfePlCf83Xpv0HwR69qT+79+TDjLgO1EG+FQb/t92Zc5/ZLDp0lSQAf+JMsAyR/p7966k/+27EsCa23cmkOYCoMKW4B6N/727UsA2g2gQn+gwgBrSAcSfQM6G0h2jSRk3BgASP+M7I0sypgAAAABJRU5ErkJggg==',
bullet_arrow_bottom=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAB3SURBVCjPY/jPgB8yDCYFh/7v/b/1/9r/i//P/N/3DYuCvV/3/j/4f8//Hf8X/m8+j0XBVom1X9YDzVjwv/lbpSJWNyyWmPl9DUhaDosb1mYvPjvzbN/G5h+Vm/NXpq/EYsLMX31/mv9U/sn/k/4n/s8gDUlcEAA/mhEkVy1MFgAAAABJRU5ErkJggg==',
bullet_arrow_down=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABbSURBVCjPY/jPgB8yDCkFB/7v+r/5/+r/i/7P+N/3DYuC7V93/d//fydQ0Zz/9eexKFgtsejLiv8b/8/8X/WtUBGrGyZLdH6f8r/sW64cTkdWSRS+zpQbgiEJAI4UCqdRg1A6AAAAAElFTkSuQmCC',
bullet_arrow_top=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAB4SURBVCjPzZChCoAwFEX3GSbBLzMaDOblBduqrAlDYcKCqNFiF39Gp8iDq91plhPvgQOXgX3D/iRM50gDWdKkSNJDmNJxHmbb6kN10gjjTdhA7z2kE6E3cc9rDYsC3GWRR9BbhQYVSuRIFo+gICHAkSFB7H765BsXhQcRTCg+5ikAAAAASUVORK5CYII=',
bullet_arrow_up=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABbSURBVCjPY/jPgB8yDDkFmyVWv14kh1PBeoll31f/n/ytUw6rgtUSi76s+L/x/8z/Vd8KFbEomPt16f/1/1f+X/S/7X/qeSwK+v63/K/6X/g/83/S/5hvQywkAdMGCdCoabZeAAAAAElFTkSuQmCC',
bullet_black=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABlSURBVCjPY/jPgB8yDC4FilKKDfJXZa6KNwhKYVfQkPW/63/b/+T/XA1YFchd7fqf/j/2f+N/5qtYFUhe7fif9D/sf91/BuwKhBoS/jcBpcP/M2C3gluKrYHhKhA2MEgN2pDEDgEb0f5zlvXgVgAAAABJRU5ErkJggg==',
bullet_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACzSURBVDjL7dI9C0FxHMXx8zruG2SSUjKgXwklw2WxSB4yGC2iDFyDpwj1T1LK00jq+hduOt6AwU02w1k/deoLkvhm+APvAVRpoEpBxVEoaoX8SZDbG24AkcWTrZ3D+ubByPBCmEv5HCjfVXPrMNq/0WdpZuaaSI3U50DhomrrG/2WpqdzZWJiE7G2CyB3lPDgTHOmGR/bDHUPRLDk4kJ2ZSA9FSR7CtJQCOQF3rjxL/FHwAu8X+2ABKJChQAAAABJRU5ErkJggg==',
bullet_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADGSURBVDjLY/j//z8DJZhh1ADsBhyJkKs44Mv3cI8Ty7+drsyPgLiCaAOOhMuVXyy2+Pl9a+//f9d2/P+6ouj/6WzdP7ucWXKJMmCnC/Pdb0DN/yf5/v9fLvj/f5vi/9ddDv+B4veIMgDk7H9n1/1HBu/rJf6DxIlzgSvz4y9zk///B2r6Ucbw/x0QP8xg/g8Uf0KUAYfDpRpOpqj+flau+P9VJev/uymM//f6svzZ4cpcRXwshMtWAG28D42Fx7g0jyZlCAYAAc7hFvdEsKgAAAAASUVORK5CYII=',
bullet_disk=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF1SURBVBgZpcE9axRRFIDh99y5s1nHjawQVEgQGxErIWgES9MGBDuxlrBVfkG6lBEsrUQEEXu7gNgIUSTBQrBSyYdgFjXMktmdmTvniLCB28/ziJnRhqMlR0ueyNrTLzY7A0UVON/r8PNojDjoZQlqwt98zNvdw9vfX97/yJQnMi4m3F26xr3FHm+2D1h/eJ3Yi3dDVOUDIEw5IkVZ82cU+G/r0y9OTQLkJQyPK0SUmCfinJAY1I3yeLCIqmFAgpF5KMuS/KQi5ol0U8ekrNl4/QMnEBpFG8UAMwODLFVinkhQo2wCiUsBQ2gQEUwNMyWEmqoKxDyRUVGTimewMkvQCnBgYKJ0fYfNV3v8zk+IeSKNGgkpyphnO6vMZZcRHMNin0e3nuDoUpQVMU/E+4Qrl+Bib8TyjSX6nXlwcDyZ58LZQNYBL0JMzIxTNwdb768unLnTn+nT9ecIWiOqiDWoFHzd+8ZRXvP5+QNhSsyMNhwtOVr6B/fLolbgemG4AAAAAElFTkSuQmCC',
bullet_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFYSURBVDjLY/z//z8DJYCJgUJAsQEsuCQeHIgP/f/vX/H/f//9lFyWvCLJBff2xPD9+/27kV/O3xxIl5HsBaCmAj5Zb00+SUOGPz9/J19fF2BKtAG3NoVoATXl84oIMPz9tIlBXC9F4O/PX7WXl3iwEjQAaBPTn5+/KkW1ooUYfpxjOLVoKQOPwHeGPz9++QCxH0EDgDa5cQnrxfAKfmP49/M+A8P/fwx/v5xmUHQoZvzz82fzqUmWvDgNuLjQjQ1oS4uAnAHDv2+XgHq/MxgHqzP8+/WMgYPjFoO4boQm0HWFOA0A2p4qpOJtzMX7huH/n7cMDIzMDGfX3QIFKcO/H7cYRNXkgWp+Zx9q0tHCmg7+/PgJ9Ls/0MgHDEx8okCR/wxmSQFwe5g5lRmUXMvFbm1uagQKhGIa8PMXx7nZwd+BCQfo/H9I+D+cZgDR//9LILuAcehnJgBMs6gZ4tipDAAAAABJRU5ErkJggg==',
bullet_feed=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACYSURBVDjLY/j//z8DJZhh1AAcBvSLXQHi//97xD797xLb/b9V1IZ0F3SJ8QE1pv6vFz3yv0rk/v9SYRviDWgVvQLU+A+o8Q5QY8r/AqEj/zOFdpPmglLhIKDGOUCNd/4nC6b+jxP4RLwBBUL7/tcq/wdqTAdq/Pc/UoDvfwj/f+INSBacDNR4AswO4b8Cpn35roymRBoZAADgYeRxtD76EQAAAABJRU5ErkJggg==',
bullet_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEsSURBVDjLY/j//z8DJZhhmBuQvdj8a+p8w//xc3U5yTIgeb7h18bNUf/DZ2j8958qz0nQgPyltv9zF5v/TV9o/Ddxrv7fmvWh/1ednvi/ZLX/f9d+8b+23YI8eA0AOvn/hnPT/q89OwWsccXpCf8n7Cn5v/B41/+MpW7/TdvZ/+o2M/LjNADoZLDmvl35/zt3ZP9v3Zb2v2Fz4v+mren/Zxxq/h+zwOa/aj3DH5wGRM/W/L/y1IT/S0/0/l94rOv/vKMdQEOy/k8/1PQ/banHf8VahlfSlQwGOA0InK74z3Oi9D/nftF/1t38/+LmW/2fdrDhf9Jit//ytQzPJSsZtEiKBe1mxq/xC53/y1czPAFqVic5GoFO/ipXzfxftJJBkeyUKFzOwDm48wIAh5XH+g7drOwAAAAASUVORK5CYII=',
bullet_green=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAC5SURBVDjLY/j//z8DJZhh1ADsBjjsspIC4gb77ZZX7TdbXLVda9Zgs8xEihQDGmZfm/7/0KOD/3ff3/V/6plJ/y3mGjYQbYD9NsurBx4e+D/10tT/nWc6/i+5sui/+RS9q0QbYLfB/OrWO1v+d5/p+t96qvn/3Auz/pt0aRNvgPVyk4appyf+X3xl4f/ZF2b+n3Co579+mSrxXrCcZyhlMV2/wbRP56pRq+ZV3SLlBq1EOanRlEgjAwAXIuIDq5qm/AAAAABJRU5ErkJggg==',
bullet_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFGSURBVDjLY/j//z8DJZhhcBtwa5ou062Zhp0PV7udvjVNO5lkA25M1Cx4czL//+/PJ/8/3xX+9myrqgfRBtycppf26mDs398fFv//cKXh/5vjdf/vLg74ui1PMZIoA+4utF7/893K/z+e1f3/8WLW/w8X6/5/e9Tw/+YM+w9EGXBrpknbl7ud/789yPr//VHx/2/34/9/uxX6/2y9xWe8BtycqmvycJXbqac7Au59uhn3/9ttv//f70b//3oDqLlB7/uiCAXcXrg1wyDm7jyrc98ezfz/9U7w/2833f6fmWD+5Xyz6YtLLWYvlscpBOMNxIerPc7/eLbw/6dLFv+/XXP/f3aK9deZ0cpJO4KleA/EyfISjMbH6zxO/nzZ9v/rTd//ZyZZfZ6bqBJLUkI62WvSenW+68vTvVYPlqarhg2DvEAMBgD1ZuZTUbWrEgAAAABJRU5ErkJggg==',
bullet_orange=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACtSURBVDjL7dIxCwFxHMbxX8mmDAa67tV4D7wDZTolMlzpn+QoouNlyMZ0Vyc5g1v/JQPdYlJGmb5egbpLNsOzfuqprwDyzeQPfABc08A1FdOSZlLQODlFL2OkARThGGIfLhsIujxtUcmBWVETe3AcQNCGaMG9KTo5MMprTkvYdsCzYD/kWk8D9LMK34ZoDqED6waHaooLL1uMR0vUzRJ9roneVUStymL8S/wR8AaM7e7IrixORwAAAABJRU5ErkJggg==',
bullet_picture=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFoSURBVDjL1ZMxSBxREIa/WdecLssuRFRUiGkiNmKbaGlhY2UjooVoaZVAiqQTLFUQYi2Wkso2tZ2NioXtqQiKuO557u3u896kMAdecEG4yqlm3sx8vJ+ZEVWlFXNo0VoGuM+Dn7/+rAeeNxfVtENEECFUS9zIi1iytLa98W3y64uAwPfml6aGuz3Pl2fPYcOpZoat34cLwMuAOKHkeb6cXlbYL38nDG5IcsPZdRdj/at8Hh5ApE0KJYgQPAKP1uL713S2v8eSYd+VuYoSrFpENCgEqFJRJbxPcqI0pa4PVE3GbZISkYEKqlQKAf8g3KeW8uUnbOmEqkm4iwYZ6i29YoxCYC3c1eqM9v5gMe5h4tBh4uMaruuigIgUS0AVQTm+WMYYw0Gek/e1k5/MYIxhdvyA/xfXbf6OPdd6/mFleo+nOm1qMMaAalwISJJ0d3P36IvFGXEcQSBUiGlAxPJQre00qX77x/QXu32e+E+qGcwAAAAASUVORK5CYII=',
bullet_pink=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACwSURBVDjL7dI7CsJAFIXhrCMLOm5rlmAjgiA+CtNF7AQrsVBEtNBBBBERC4kBIZm8JjHHuAAhInYp/vJ+cOAaJI1fMirgA1DLzSKRIZUxIvmAJ264m98A4tnPma0003lCv6t4wkWUBlJo+T7WvYBxo2gYcYuDLA14UFJPE8ZNxageMLBCLrApDzhwhdfxGdshw4Gi03Y5wqT8hDOu5h5HscZOzrCUNsaiBcusPvFPwAtRJv5/iiq/OwAAAABJRU5ErkJggg==',
bullet_purple=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAC4SURBVDjL7dK/CkFhHMbx5zrOlbgE5Q4MBgulhIn86Wc4os7AbjEaLThhQekYbC/JYMBIp3jrcNLjBgxHshme9VNPfUES3wx/4D1ggYYFSgO+MqGV4CwFHIxPAJnLk3vb56734KjsMouNBAbquKv90Oei4nGc1nRamkksVWCgCletux4nGc1B4sZp7coYxsGBIk5ily50mpoz88pe/sgIOsEv5LA1UlhJHDMVRV+F0ZYQTONf4o+AF+Us6n1474tIAAAAAElFTkSuQmCC',
bullet_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACxSURBVDjL7dItC8JQAIXh491HWbDvD1pEDIIKm1zDxqKi0WIULAZFYc1msF3ErEZlMC9Mhxx/gTAR28KpDxx4QRK/DCXwAbDg0oLMTShtQF0F5AlwvwHkqy+Zxxs+lwsmvs8DKrIw8DCh8njNLOxRtxrU4yF3MFRhIBFQ2XxG3W7yXq8xjULGsIsDFwF58zzq0YBpFPDc6XIKp/iFI4S7h5BbWGoFW03gyABVtyzxT8Ab8Xzei+tkcykAAAAASUVORK5CYII=',
bullet_star=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADdSURBVDjLY/j//z8DJZhhmBvw41KQ07dzbk5kG/Dtgu/Jb2fsT5JkwI+bqZw/rsfYA21v+XE97f+PS5H/vx5Ra/98QN7+824ZTiIMSJr580bW/x+3iv//etD9/+fdpv/fzwX+/3LY6P/n7TIzCRtwPYYZaPvGH7dKgAb0AA1o/v/tQsh/oO0bP26TZiYqDIB+1/1+wef/z3vN/3/erPr/5aAOyHZdogMRGPIe38/7gvz+Gej3z18OG/8H2u5BvAFn7GO/Htdv/3pAQejzXjkhoO3tH7dIxY7EpEwMBgAr6O5Q8udliwAAAABJRU5ErkJggg==',
bullet_toggle_minus=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABhSURBVCjPY/jPgB8y0FHBkb37/+/6v+X/+v8r/y/ei0XB3v+H4HDWfywKtgAl1oLhof8TsClYA5SAgEP/27EpWIxkQj02BbOQ3FCGTcGEdV3/W4B6K/+X/M9fNzAhSbYCAMiTH3pTNa+FAAAAAElFTkSuQmCC',
bullet_toggle_plus=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABjSURBVCjPY/jPgB8y0FHBkb37/+/6v+X/+v8r/y/ei0XB3v+H4HDWfywKtgAl1v7/D8SH/k/ApmANUAICDv1vx6ZgMZIJ9dgUzEJyQxk2BRPWdf1vAeqt/F/yP3/dwIQk2QoAfUogHsamBmcAAAAASUVORK5CYII=',
bullet_white=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABbSURBVCjPzdAxDoAgEERRzsFp95JbGI2ASA2SCOX3Ahtr8tuXTDIO959bCxRfpOitWS5vA+lMJg9JbKCTTmMQ1QS3ThqVQbBBlsbgpXLYE8lHCXrqLptf9km7Dzv+FwGTaznIAAAAAElFTkSuQmCC',
bullet_wrench=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFSSURBVDjLzdPPS8JwAAVw6Z/o0J/TqUOHKA/RIfDSP5CHyIi5DRWatrWGsh8o4kAkJoNBYJiBaebJb8xDZSbK0DQMIkKoXn+BpYTQ4V0/h8d7LgCuv8Q1c0DTNI+qql1ZlkexWEyYCtA0bUFRlE6r1QIhBJIkZaYCZFlesywLjuNAFMWhIAhGJBLROY47nAiIRqNu0zRRr9fB8/xtOBx+KxQKCIVCn78CkiTNiaJ4Wa1WQQhBpVIBIQSlUgksy3Z/BPhsm+IyjVHSKMK2beRyOei6jng8DoZhOn6/3zMW4Iy2N5B+/MqW+/AlGjhOnSEYDBosy27SNO2mKGp+bIl0qundTTRH2fIA1vUzkvkePJz9MfEOvGrj/aTYh3k1QPriCb7EA1Zp+2BiYOvobqicOkid97At32N5/4aZaonrAXtjhbZflvZqr4s7tcD/+MLMgW9PPe6C6jWXgQAAAABJRU5ErkJggg==',
bullet_yellow=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACxSURBVDjL7dI9C0FxGAXw8znu5/IdZDGYjI+BzWSwKFnlpW5dcaPkbgalPInyzqSrG65w0/EJlH9Ww1l/dU4HJPFL8Ac+AB4sepCoDw1dqG9DdlVYJoC8VsLI7/B5dBhMhNMy5Gvg0YNGJ5f3bYbhPMVw3+SwAP0aCBzo49jmbZnmVeO8rOvs5gyAQw3ij4XhweZ50+BiIKwkDSrMKrBGRYiXh7Yy0FICko2ZjPh/olHeFCjwX864N2kAAAAASUVORK5CYII=',
cake=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI2SURBVDjLhVPfa1JRHPepnnrrT/Al6KG/YG9RD0EPFXsJCkaMHjMKIamVNQhqQW3LFqtZq9Yg1KXVcBhdZ9ZDgyblT9y8Z1fdvXo3Ua9D1E/ne6c3bUIHPtxzPr++5164JgCmDsJ+0/FI2BTu5v6n9xgSEZNWLh0BN9r6FfTTewyx1f3QqsOIre5r9ZvY0aM/d/U9Be+WHiO4PIg5n70mCEIizgM0MRQ4W+Bn93PPOJY+n8H4G6vUU8BFM8fYtL8I17ctTH7IQ9M0GBP5s1AowP5WguOjjIsTSYUyRsFXweNkjOHJooL5oIoJrwJazve7E2c8o/r52ksJDxc2YZlKgzJGQVAINPjC6y8qN8jwr5T0wJ35LByfZNx4JelnhyuPq9MMroCMZWFxxygICb5WvV7Hv+v6rIRH3k1YXzCDazabkGUZye+2hlHAVizNRDwKeo3Oohs53DlYnzEsCEWdU1UV8dhv5NM+KOFDfwu2QgcatcxtpJJR/WPlcjkwcQ0bG0wHFSuKgvW1FEqZpyAvZYyC7MjhVmFmGJXUXShMQEmcRU0cNaCJ97HN5lAV70FL2UFeyhgFRe/BhvzgHCTLKSiTQ9j2XkLlh003E2hPHGnkIS9lul9hp5a5hVLgCpSpC8jaBiEOncD66aM6aE8caeQhL2W6C5zlXye5cLPn6n3BPeSlTHeBmWOMo1aOHEMlfh5a+jI3j+igPXGkkaftNe/5Fzg5wGHjcHMkOKptJNocaQPdmT/bXw90YXDpsgAAAABJRU5ErkJggg==',
calculator=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGxSURBVDjLpVM9a8JQFL0vUUGFfowFpw4dxM2vf9G5newv6OIvEDoVOnUQf0G7CEYQHVzUVZQoaKFugoW20EUaTd5L+u6NSQORdvDC5dyEd+499ySPOY4Dh0TEK8rl8n0mk7lOJBIpVVWBMUaJAzCFEMA5B8MwPpfL5VOlUrklonegWq3qEr+c/2Nbq9VWHs9XkEwm0xLUy/Lzn5KbD1exaDR6FlpBURSq4/E4HJ2c4jMwmYpcw6vf31be2bAHQTPVHYEFyAr7VeEACzfAQKPuSmlCy7LINBcteifSx3ROWutzlCAZ3Z9Op9ButyEWi8F8Poder0drXTQ1SNUeqalt22EFQrgvC4UC5HI5mow1EjA/SjdEjEQiYAd+HV8BF5xwNBpBo9EgBZPJBDqdDimYzWbQ7XapmeA8rIDLiRjFYpEm4zTEfD7v19lslhSgJ2EFXBAOh0Oo1+vk/ng8Bk3TyBtd16HVarkrCRFWYFqmrwAzqMDzBhMVWNaeFSzT5P3BQJXI3G+9P14XC8c0t5tQg/V6/dLv9c+l3ATDFrvL5HZyCBxpv5Rvboxv3eOxQ6/zD+IbEqvBQWgxAAAAAElFTkSuQmCC',
calculator_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAImSURBVDjLjZLda9JRGMf3b3TTdTcFXXXRTbeDIijo5W7GmtWqCysIF1JM9DfffvMN+fk284UpKqRIMCfIDCO7kNSRooJvOBUvgkFEKPv2O8+WsWmtB77n4TznPJ/z5eHMAZiTyWQat9vdCQaDCIfDiEQiiEajlNme1f1+PxwOx1ChUJhYz2/RYrPZCmL+htPjpyAIe1MAr9c7EvPBNdkm/iUWolNMAZg9FrdW3mFRu4Ml/QdI+Swern/EsukTHls+0xkLj8czDfD5fBOAVGx+cKL5qfUPYMPzdjZgPB6jWq2iXC6jWCwin88jl8shm80ik8kgnU7/3cFRkSZvMBhQqVQQCASg0WgI1H/+DLU7NzAajWbPwOWiIjqdDlqtFur1Omq1GrkplUqkQqFAd5yuGQDBbqfDWCwGo9GIdrtNbnieJ1goFIJerycHgmCfBlitVgL0+330ej10u11yw0DNZhONRgObW068ECRYMlzHXeWV/fmXF15NAEajiQCJRAIWi4Ug8XgcZrOZQLqN13gTliCxa8WXvST47Ue4zZ/HpeUzOgJodToCDIdD0mAwmLhhkmjmESutI/b18CFd6j74lJQBfhBApVKNdjKZg1QqBabtE7q6chHvd53H/nS8aGWAw0HI5fLS6qryu0qlhlotiuPAra2B45g4XH5yFtrkPaiTC9Ss3lo47uA0iReVN3XnoE8u0ssss/1kBv8J4UTtM9tHmWP1X8ma/9q6R1ZmAAAAAElFTkSuQmCC',
calculator_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJGSURBVDjLjZLda5JhGMYN3R9QRxF04EEnQf9AR1HtLE86aNFhSVtsdWCjLQMLjNd3bmp+IWqa+UGKGikGswyZy8oiQ4UUXfk1pya0k9YHilfv81BW01U3XO/Nez/P9Xsvbl4eAJ5EIlHa7fa61+uF3+9HIBBAMBiknbyTudvthtVq7chkMh3x/BR9mEymDNc38e/6ZjabN4YATqezy/W+SHIPfxMpLimGACQeqVPXHkC8uILzqlVMapK4cOsZpnUvMGN4Sc9IORyOYYDL5RoAJjnz1DbzJeMvwB3H3dGAXq+HYrGIfD6PbDaLdDqNVCqFZDKJRCKBeDy+c4IfQ7p5tVqNQqEAj8cDpVJJQa3ZyyidPolutzt6BzYbHaJer6NarWJtbQ2lUommyeVyVJlMht65bRsBMFss9DAUCkGr1aJWq9E0Go2Gwnw+H1QqFU1gNluGAUajkQJarRaazSYajQZNQ0CVSgXlchkZuxzPzx5E7OgYouP8Gqf5AUCr1VFAJBKBwWCgkHA4DL1eT0FRZgZvZg/jy0MV+m+XseWV4NX0od6jY4KLFLC4tEQBnU6Hqt1uD9IQxc8I8ZkzQycC5nYDjBAd5RFEj/PfUwDDMN2VRKIfi8VA9HibSOz+6/t//NOb1/dyc0GfAqRSaU4uv7nFMAooFJxYFuzCAliWiEXkxB58sp0DONPXKzx85FSd4pNdrPN+3+hOWp3YdyMlPtDdmBPiw9UxvBPvwhORoLc8zpf+F4Do6cT+ee6LZRKb63ViJvPvI7z+j7d96tcAAAAASUVORK5CYII=',
calculator_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKRSURBVDjLhZHfT1JhHMb9F7ptXXXR2lw/Llp3blnNZauLtmwtp15oWsu6oJZ5bKyFQiGIEIRIKoEsJtikWM1JmiQhtpieo3ISUoEINiJNE2SgT5x3ZiVZ3+15v3t/PJ89+755APJ4PJ64s7MzZDKZYDabYbFY0NvbSzq35867u7uh1WpjfD5fwXl+iixqtXoi2xfw/0ppNJrPOQC9Xp/O9vXTvCf4l7jKJkUOgIvH1bmGPlQ1D6Na+gY1Micut77FFcUoapVj5I4rnU6XCzAYDJuAmqz50hbzNdUvQJfu8d8BmUwGMzMz8Hq9oGkaHo8HbrcbTqcTDocDQ0ND+B62gzWeh8/ahPGOIkyo8ssJYCMWmXxLSwtYloXRaIRYLCag6I3rmKUKERmswyJtA5bDWKAtcElORAmgo4MMBqFQCIFAAH6/Hz6fj6RhGAasuw3xqTtIhZ4h8roZCeYpMvMjYLqqkwSgaW8nAKvVCrlcjmAwSNLIZDLM0ibEJ29jLTGCxMdaxMeuwmuoBK0t+zKmOLOHAFQqFQFEo1FEIhGEw2GSZp4x4ytTj7WkCyv+CiSDpViapjCnORJ9Lz1+cHOIcrmCAGw2G5RKJYHY9HxE3tVtmMuRDFzAt8kGsKoifJAcPvTHLzRLJAQQi8WI2FEjpvtvZmM7kJyrxGqwDIvZJH7NSTyUCnK/USgUpocdjnW73Y6+R3xMvaAw8bIVn9wlWJkrRXz8FrzqUxgZ6FsXikSJHABFUYxA0LgiFIrQc/8YsDQPtv0sBqmdcLYVY0BQgAfCetwVNK5m37pyAL9LcDE/nXIpkXLL4W4qRE/VruX++v0Htr7bFlBSsCMpqtibfnWvOG2XHh1+Xrdv93ZmTj8Aff0H4WdEl0kAAAAASUVORK5CYII=',
calculator_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJtSURBVDjLhVPfS1NRHP/ce91wW9MZrjLLKCNxTEgcG9FLj0EgmbApPfQHBEHsL9B6CnrpRcQeQpIiWgwjlN42X3Kza7nKGEXRxFpMhVq7uz/OvZ1z5ubGpL7w5XO+536/n/M53/O9gmVZiEajd/x+/7jT6TwmSRIEQeDOvjEnhMAwDCiKUsjlco8mJiZuoGosYWpq6i3FHev/pk5PT29WiZm3MBKXy+WjIA1HH+NfNn93zG6z2brq9ziBKIocHQ4H2jwdLIZAXaTXqK5/ft/Ebi6aCJiUqkm7BUJdsSjsFVgQmgmYsUbdvubjqOs6b1oFdb5HfO08j7a2gUCsFrPuZzIZLCwswG63Y319HYlEgl+rN/0EPT9moJa2YZpmswJCKpvBYBCBQICfzNZMBdH/oDQo4ODxEfz+9gJmo4CKAoMYPJBlGfF4nCtYW1vD4uIiWhUZnq4Q2roGsf35JZwo7ENAT2QWCoUQDof5dZiCK5fOw8gn4O70gPyah7d/HEdbPiDz8KKtkcAgPFhZWUEsFuPdX12V8enVfXT2jQBlGanZORzwKGgnORhlbbiBQNO1moJIJMIJzvY6carbC3dHCab6hb6fCVJM4+SFKAxVvZW6d85da6KuaUZyaUmiKFTmgsCTn8Xp0GWYpWVaq2BotA+mtolWVxaHB8b6N5Yf3KSpk1xBsVj8mEwklVQqjXQ6jfy7OA71DMHpLsAytujjS3j9LEszTZjlLLxnToCo+vXkpN8n1E9h1ai8jYGrc92S+JUS7PD5q++75OhD/v0bZJ9PPm3Z76cxVK1VnhlVLProFh2cPbdqCIaWdeQvTLNXD529QmkAAAAASUVORK5CYII=',
calculator_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJlSURBVDjLhZPfa5JhFMcH3dX/Meiuv8CBoRdjIvOieRUYmjkzmBaowXCbU9E5U5KhtmX7QbrpyF3UuhDRVFoXgnrhmJO2CSFELaIYw7FvzznU+uGqA+d9znuecz7v9zkPbx+APrPZ7F1YWGgnEgmsra0hlUohnU7zSu+UX15eRiwWez8+Ph6inh/Oj7m5uapYD/F/O45EIu96AIuLi12xnirMT/EvJxNK0QMgeWTX7j+D1pfHTf8r6AMlGB6UYQy9xu2Hb3iPLB6P9wKWlpbOAHrRfOuP5jvhn4DH8SfnA05OTrCzs4NGo4FarYZKpYKtrS2USiUUCgXkcrm/K/ie5MnPzs5ie3sbKysr8Hq9DOrctaCpVqHb7Z4/g/l5TqLdbmN/fx+7u7toNpuspl6vs1erVa55NH8OIBKN8mYmk0EwGMTBwQGrCQQCDEsmk/D7/awgEon2AsLhMAM6nQ729va42efzsVyPx4NyucwKCEK56enpj6Ojo/ckEsklBgSDoTMFJpOJVRCs1Wohn8/D7XbD4XDwkClXLBa5ZmhoyMsA38wMAzY2NmC321ERZ56YmIBCoYBOp0MoFILNZuNYNEGtVj8niMVi+cQAl8vVzRcKp2NjY3A6nQx4sbkJmUyGbDbLN0FXSUeTSqVQKpUXCTA5OXnEAPHV+tSU86tGcwMGg4EBGo2Gi+VyOYaHh9kpFrlTlUr1kgB6vf6w79eJXhYmZDfEsA5XV1c/rK+vQ/xoIGVWq5VjytEe1VDtb4D+/v4LAwMDVwYHB99qtdovRqPxSPjxyMjIdeFXRfyZcrRHNVT7DWZq3D+QvMywAAAAAElFTkSuQmCC',
calendar=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI1SURBVDjLpZPNS5RRFIef98NRG3M05gOaqMhCAoskEFtVO2kT5DZo1SZoW9EiFGrVok1/QhRGmlHYJoQy8wuJIoLQ0iH8nCGcMmzm3ntOi1edWbQJLxzOPZvffc7v3OOpKjs5IcCDkbVbqlx2KvtEBCdgRXBOcKJYJ1XhcFZxIlN3Lh7pCAFE9Wp3x+7k/7zc0zd7fJvAiSQB4v1XQBWcBXFgTHQvl8CUoWTAGH7fGMYYV1slsOWDwt4UiI1C7aZICUwJbBk+LwFgrKt44JxsCniwmAeViEQkInFbGchkAChvCqCq3H+5pOsbRh+OrKiqat/bZVVVffwuqp+MRfXgeJTzxQ29dG9KVTUiME4QVcIA+sdWCQKfgfFVwgCeTa4yunKbN4OCilAz3cPR7C7KJiLwt1pQIAw8uk+lCX240JmmxofzHWl8L6C15TC+H3DuZCYarZWKB8YJKhD68GI6z2ThLhNDiqoSvr+O73vMzecI/ICnEytkEz5KtYAVVJWaALraU4wNwaEDB/mWm6erPUXg3aSztRlxysxCkT2NtcRrw0oLViK1wPd49aEAwFwuh4fH8McCRkBEmV0s0tQQI9lYtz25EMBaG3ngw9ljScLgGmfakrz+VOB0W5KB8TxfF36SiMdIJeoQUQKfCoFxUQtt+xtY+rFBS6aeueV1ss0xvnwvkk1AoiFGuqk+Ig08RFSqPpKO9j6aOVF2EjfWYazDWcE4wTr7z11Y+/XnOYC303X+CyNCWo09HetwAAAAAElFTkSuQmCC',
calendar_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJlSURBVDjLpZNbaM9hGMc/v8NsTqF2yMhZKIexmpFyRZIiblwouZDauLYoTVEryg2uXIoIOUU5zJkh4e9QDmtJbWxrbUzz/72/9/m6+P3HLslbT9/3ufm+n/f7Pm8gif9ZMcDxe717JLZ62UQzwxukZnhveBOptyHl8anwZk/3b5pZEwOYtGNDzejSfzm58dTH+b8JvFkpwMizdSCBT8E8OJftkzy4BPIOnONHQzPO+eIhBoM5CCrLwNKslBZM8uDykCbwtgMAl/o/GXhvBYMA2rtAlpGYZSR+UIGKCgCSggGSOHy1Q/0DTifufZUknbr/RZJ0+mHWn3mU9edbMu3qG9DmQ08lKSNw3jCJOIKzjzqJopBzLZ3EEVx40smDr/u4e96QGUXPGpkzYQSJywjCwSsIiKOADUvKiUNYX1tOUQhra8oJg4hZ02cQhhGrqyuyp03tTwbOGzKIQ7j8rIsn3Qd4fEVIIn6+kzAMaH35Fn37wbZD68gnCUl+EbAkI3CpIYmiCNZUlwEwbfIUgiBg1cIyJqbzGFPiWbl8GXUb66mqnkrJ2IvUbq88GEI2dQBRGHDjZTcAbZ8+ERDQnOvm+fszVM1egA89C8avwAeO2nlLAeqRxK7j79TzPa/mXJck6darTG8XdM3uhbry+piGrou5I1pcP17h7wwk5k4aRUfPANMrhtP2pZ8J44bx7nMfff29vGl/SNP1LQA0XdtCa2cO4GdhkPRg78kPVYm3kS71uNTjU8N5I/UpxSWracndZOn8ZVx6dZRhQcz9F3cAjgR/+51rt1c2AXXAaOA7cLTlcHvDL6y6kIpO9lqsAAAAAElFTkSuQmCC',
calendar_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ4SURBVDjLpZNdaM1xGMc//5fDZqTZ2c6Y9mIkmZelll2hlCVF3FDKlZRyQWJXmsKNCzeLcqdE5D0mL/M2s6HlJSlhs2Rv51iYOZ3z+/2ex8X/zHZJfvX0/J6b7+/zfH/P46kq/3NCgNNt3w6ost2pzBYRnIAVwTnBiWKdTAiHs4oTeX5467y6EEBUd22qmxb/l5ebzn1Y/IfAicQBCi7uBFVwFsSBMdE9mwGThYwBYxhtvIcxbvIEgTEfFGYVg9go1OZEMmAyYLPwth8AY924B85JTsCDviSoRCQiEYkby0AiAUA2J4Cq0nyzX3+mjZ5pG1RV1XOPB1RV9fyTqL7QEdVXOqOc/J7Wbceeq6pGBMYJokoYwMWOIYLA51LnEGEAV58N0T54iEdXBBUh1tXEgrIpZE1E4I+1oEAYeGyqLyH0YePyEmI+rK8rwfcC5lfPxfcD1i5LRF9rZdwD4wQVCH243pXkWeooT1sUVSV8sR/f94i13KHh1Utam0+iRaUsKVwH1OcErKCqxAJoqC2mowXmVFTS3fuJhtpi8p6WUTTcxbyte5lctZD069tMb2vlzurYLh+iqQMIfI+7r1IA9PT24uFx73WKzP0TzF2xgbyPD/BObWFK92WqKgpRT3eHANbayAMfVi2KEwb7WFkT5+GbFCtq4rR+/UxeaRWs3TO+A00zCcSrHPdAlZryqfQPp6lO5NMz8JOywkm8+/wdnZHg18sbFFzbSSY9wC9g5EeAC+jLDZK2Hzz7fmnWSYGxDmMdzgrGCdZZ6ks3MrXzFhUz8gmDGCNJS8+gr4oc9/52nds3lzeODvftCJxX4QL9onBizW175DdAmHVGgBeCfwAAAABJRU5ErkJggg==',
calendar_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKbSURBVDjLpZNLbIxRGIaf/9Lfpa1rZ0hNtYpEVRBiNkUiEZFoKjSxQmLBglizIQiJILEgWHQlgkpLGg1BU6E3imorLkElLmOmM1XV6mXm/8/5LH5GI7EQJ/lyztm85/ne872GiPA/ywa40NC3X4TtSnRIa43S4GmNUhqlBU/pUaVQnqC0fnRk89ywDaBFdpeHs3P+5eUDlW8XpgmU1jkAmdU7QQSUB1qB6/rnVBLcFCRdcF0G99bjumrMKIFfPgjkBkB7fon3UyQJbhK8FLyIAuB66rcHSumfAgZ8ToBon0Rrn0T5+6AzyKd5eVi3j7HDuUfnmchWRITTN6PyfdiViw3dIiJS2RgTEZErzf69qiUmg59rJFq/R/o6a0UGIvK1s0paTqyN2QCu0mgRbAuqW+JYlsnVB3FsC2pa4yQSuxjKLmbK3BJ6u17iGCmyJ0wna+rMiekWBLAtgw3hADWtCdaHg9Q+jrO64BVDmUVMKlxDMnoZZ7zB+/ZX9A+ZGMmeRWkC0WCbUPskQWvPcR7eEEI6xvDYPCbNKWMkcg7T8cjIysfxPnC+dwun95a/Nn1HNSJChgWlSwMAFNtDbJw+g4lzyhiJnMXMcEn1F9B9vwNZto/vTggAE/ypA7BMg7qOHnL6PrBYxhEsXkcqVoHlCMn+fLob2mDVSQIFC9M/ZwN4nud7YMKMgTpyJ8/GkyDRZ6eYHHQY6c2jp/Ul5qqTBHLz0VqwTH4TuMpvYcHMLDoaK5i/fAdjPjbyuqaJ9lu9PL/zFKPkEMFQoU9qGWgtetQgSdPBS28Wp5TOzO1KkHpey6xFK/iW+EJX2xvuhg7z5do34GE6C30DI9cBjD/jvGnllOElhdPscNEsJCPZPBxPbCs92vnub6H6AWmNdrgLt0FkAAAAAElFTkSuQmCC',
calendar_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKtSURBVDjLpZPrT5JRHMfdetn/4evet3QrhGayZathrbwNppMU8ckFONEhoiGFeMNL4o3yBoi37lQQ89rYypaLIitTQVSUWKC9+Haeg5P5svXi+5zf7/t8f59zzrM9SQCS/kf0MeAKKfudodWe19vofhlE54sg2p4F0Px4A43T69BN/ET92A+oLd9QPexD5YMvUAx4F44Afc6dYCT6B/+iW73LsSOA6dUWNWEWAQNCoDcXMN0AOrKAtstAIx9o4AHqs0DVGZqVdH3AEaDLETwEkGGHHHheDjyVAk+KgclCwJ4HjF4DBgms8jTNFhnfJQDt5L7HTtCfD/SRoZ4coPs60EmG2wVAKwG0XKJZYbMnAWghH4s1H74JsCaG3X66jszEe8tsvLfPxdfgXhQ5+sUEQD+1hvDvAzpgYTW7CetsALb5AOxEUvtNlIwVodhaiOm3fnzdCOOqdi4B0I2vYo8AbPOb8Z0O18nF+AmYcQla39+DZExM+89re7iimUkA6mzfsRs5wMTiJmwzq2CaS6GuV8NkMqFGcweMWQJmVIKKGgX1tFodsuVdSElJOUkBqpEVhH7tY2phnQYMBgP8fj98Ph+cTifqG/QwtnfC6/VSz+1200xGRoaWApRDPgpo7JnAbZkCnqUl5DO5JHABIpEITU1NkMvltObz+VDXaiiEYZhdClCYP2GHAMoq6qDRaCggFoshjcuDw+GAx+OBy+WC1WoFj3cega0wBahUqigFlPctYzscw/1eMySSUgqQVSjB4XDA5XKRmZlJxdYcThqUVdUUUFBQEKKAMtNHd0nHUiRb/QgXs/KgvWuAecgCm80Go9EIqVQKmUxGa9YzD1qg1ekPyHWWj/2aycnJJ1JTU0+lp6evCIXCiFgsjhLtCwSCbKJzpA6zHvuOzbDZv+ti+O9p0FSiAAAAAElFTkSuQmCC',
calendar_view_day=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHOSURBVDjLpZOxa1RBEIe/d/e8FKeFEA2IGBWCICZYBSESBCFglUDSCJZaRBBbK1HQ0s4/QQlCgoKdoBA9sVBshCBETCNRiUUg5PDt7MxY7HuXdxgEycKwyzJ88/vN7Gbuzl5WDvDozeZtd66p21EzQw2iGaqGmhPVaqFodNTs/f0rI+M5gLnfmB0/MPg/le88+TLWU6BmgwDtpevgDhrBFETSORQgAQoBEbZvvUJEB2qAqg8ORw6BxRQeS0gBUkAMsPIdAIm60wNVKwEZrG+AW1JilpRotQNDQwCEOiCWgIXhe1w+f/if3hffrXMhxH4Fooa5kzdT0rNPi3TWlrl6bp7PP1d4ufqCiyNTzIzOUYiz1RWCJECjsuBA3swAmBmdoxu6APza3uDB9EM6a8sAFFEJYsRoOwBRww3yxt+Su6FLq9nqAQuxst11QDTcnX2lhc7XVO3jtw8cOzjMzafzTJ26RJUL0B7Ia020dNlsJAsTJyaZODlZziVj+swsWZb1AarJJUCMeCnn8esfaWruiIKoEtQIkry3mlUx+qfg7owd389prd6+9/7CbsvMrfaQ/O3dhdWzQa0tUZGoaDREjahxV8Dm1u/nANlev/MfAjw0JrMu09AAAAAASUVORK5CYII=',
calendar_view_month=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHlSURBVDjLpVPLShxBFD3T3WOUUUGYByhkI+50o6Bmo3ErAVG3Qvb5giyThets8gmC+DYQzEaGBMUnBhQRJAEXQuZJ8Mlodz28t6rbGY2b4IXqU1XcOvfc01UxrTWeEx5/ptbOssQzILVylVKQChCMkucagrA6JKTQtK+uJyc6Gg2B0npwvLfJ+Z/KH2Z+J+4VEJs5nFh4B3BLUhCrBILAzv1bmvvAbWD2rt9nCSRqCCIfCFtTdFjYoUVIwgQ0BJEc5UxmIGoJuGkTMeBPiQ4qq4R8MEpkhJSSyZhMPyQA/4XP33L6qhLoKGbW8wZnNwoG5zftennLYum8ot9+2uWpVRBQBTISc5tFsBmu62BxqwjPBb7sFNFQ52DlZwnxOOFeAe3pevihB07UArvguTGMv0rDo92x/jQoHyO9aZM43J2y2GNbEEJVCVgBt80Hv+5RJdLFFR0nZpAjwqXtQmh3LQGxcT9xkvymJ/WwIiEridajfVZB4oVXJeBbx+FSxdX98oOK2YMyAoV/lER/zrP9COsB0Q11JfH9sIzXnUn8IBwkfBzHp+dUrOYtGA+ohc6Xjcj9raA904CT/BXaWupM8lOhwssXXiR98XH6V7NPRHzDeEjyJTCPRzxJcHZ5kzdX77nP+Q6ZHT+VaotBJwAAAABJRU5ErkJggg==',
calendar_view_week=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFySURBVDjLpZM9a1RhEIWf7C6mWASLSEBsbUWb/AdrW8HKRrC3EgWttbCzUgQRtRK0ERVEm/Q2aiPoBtJEosZ7Z+Yci3tvdosgKxkY5m3eZ77OrNjmMDYBePhu55rNpbJOSqIEKVElSiZLC15UmpI2b104tTEBkH3l/MbRtf/JfP3x59P7FZS0BjB9dhlsqAQVRHTvtoFooQmI4NfV10TU6gJgmIPhxHFQdu7sIQ1EA9nCxxkAkTWfQZV6wAp83warq0TqKqkhAuvrALQ9ANvcfTnzz72wbd9/M/tnfPLhm7d/7Pni7U3bZgQQJbTkOpswu7+DNroKRkMLy6qhyaINkak5IEpYSwJC/bgXASmWVWT0maerkzkgpaUFNACGzU0AMhMDD95uMR51UTb3Xs2IKu68+Eobxc2nXzgy7j6ORwu3ENW1cO7sMbIG7Xv/Fg4yqZtaLyS/v/Ho05m2NI0sIotKESWy8kDAzu6f5wArhz3nv90eObvZ0Hw1AAAAAElFTkSuQmCC',
camera=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIrSURBVDjLxVPLjhJBFD3V3dCYSZjwDipDRjCOCcTgLGalJmxYufAD0E/QHV9g3LjSsGDhik/QjTEaTdgaEsQQINEFJjwEecmr6Yf3lkJmPwsruXWrq+85dc+9VcJxHFxkKLjg0HgqlUpP1+t1drvdXrIsS2w2G9dqtXKzt21b6Lpu+P3+N4VC4QnH5/P5U3Lfy+XyL1EsFg9VVZ1kMhmwHAJIa1Tf4eH96zC+vcKLDyfwRm6i1Wo/Go/HLwnsJWPtZ2oikainUil/JBJBp9NBr9fDcDjE5aMTfPz8E835bdw6vQMhBLrd7gNFUXSPxwNN08RsNrur0alJr9eLer2OcDiMdDoNkgKSglAohEajgUqlgkAggGw2i36/j8lkImOazeYNjQMHgwGCwSC2B0d4/n6FhSlgU366cOHe1Wvwz+eUfgucJQPb7Tai0aiUqvC0WCwQi8Xw+ouBjaPA4yJzK7AUFz79OJDA0WgkwYZhgAq8z1JjAqq2tN+Wm4AqNGouXw9Bk+G4QR2SxmCO4zV7mQGz7Fh5OI79F0wkCpmqCCyXSxlzPoM9wS4DqijcjgHTJv22Q+k5ksiNFabTqQSapilJOJ79noAZa7UaMoc9qOYSG9oziUSxlriy+YpqtbrXvSPZEWjMxidwewLjMc6Oj2VxOGBO1WdivhvJZFLusf54PC5lMYHI5XKPqa/P6EP3+XyCry4T8E/2BHDoEIdv6fmH92/9Vvz31/gHd9iUVZFEDKoAAAAASUVORK5CYII=',
camera_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKySURBVDjLpVNfSFNRGP+de++2a66Fm3PNdKArUnCaiSijPzCR6qGg3i2itygQetDHXiIf6qWaYBA9+BL0EGhPlZUwQoQo1mpsAxss1Klzc3O72527t+9cUXor6MDvfBfu9/u+3/l95zBd1/E/S+Lb1NTUvXK5HKhWq3W1Wo1VKhWToihmHjVNYxaLRbXb7a/HxsZGef7IyEgfhZ/T09ObLBgMHhJFMdfb2wuuhggGol/e4urFY1CXnuHR+w7YXJ2IxxPXstnsYyLbCFz6gOj1eiNdXV12l8uFVCqF1dVVbGxsoNnTgY+f1xErnERP32kwxrCysnJZEASLLMuQJInl8/kzEnU9arPZEIlE0NTUBJ/PBzoK6ChwOp2IRqMIhUJwOBwIBAJIp9PI5XJGTiwWOy7xxLW1NTQ2NqJa78GDOQXFHQaN9FmYCWdb2mEvFEh+HFwlJyYSCbjdbuOoAt+KxSJaW1sx+01FRRcgmwhmATXBhPlf9QYxk8kYZFVVQQbvq5R4AXLbwHbNTEQRkkAOkUWMNlU3gyZkgJN5Hv/m0VDAq+xV5UvXtV0yFREIosBQKpWMnD8V7BXYV0COwqzXYUeTIfAJ6bsqzFCwtbUFq4chXJpDqW4bB/ryWM8uGQXE7u7uu1ar1XDW46xHWjGjysTdW6YpOKJ+R2L5A9r9NpzqH8BQ/3lU5QxSahjZ3DYk3p134ONxZLMYaGszzOFyC+R+OByG5NvEiQ4/mVpDj3sY7368xKDPj2R8FhJ1Hk0mk/dJjqWhoYEtLi4yXoDL45EM0w97a8zErLjQecNQdmfoKU1skkya4Ub//TH5b7coVy6dk3fodowPP8fEm+uQRQtevJopC//y4jRde7gQ/kSGSkZnM5MQ+jrPfwXZvz7nwVvNExRuEg4SCoTJhSfL478BoeOJpjqa+ZsAAAAASUVORK5CYII=',
camera_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKvSURBVDjLpVNNTxNRFD1vZtoZICkyhdYilFQ0QKSayoJEoyY10UTjgh0LUXcuJW5wSYwfG1caNCxY8RPQxOBHEDEkREJSS2qpIKHWllLol+10pp0Z3xsCupPEl9x3J5N7zj3v3Fximib+5wjsGh8ff1ipVILVarVO13WiqqpNURQ7y4ZhEFEUNVmWX46MjAyz+qGhoT6avk9OTu6QsbGxRp7nc4FAAEwNBVgRWXqDG9eOQ1ubwNP33XC4e7CyEruZzWafUbCDBpPez3d2doZ7e3tlt9uNeDyOVCqFTCaDVm83Zha3EC2exqm+cyCEIJlMDnAcJ0qSBEEQSKFQOC/QrsccDgfC4TBcLhf8fj/oU0CfgpaWFkQiEczNzcHpdCIYDGJzcxO5XM6qiUajXQIrTKfTaG5uRrXBiyfvFJRqBAbVJxIbLrQdhVwsUvkrYCoZMBaLwePxWE/l2FUqldDe3o6pLxpUk4Nko2HnoHM2fPjRYAG3t7ctsKZpoAbvqxQYAXXbil+6nQJ5CBx1iFpE6KWZdtAJWcHArI59s2wpYCx7rOyYprELpiQcDZ4jKJfLVs3fCvYI9hVQR2E361AzJHBsQuauCjsU5PN59JEEvK9H4c38RBcvIp4+ifk9AsYYCoUQaPVjKe+CKkjg6dhshoIj2jKwOIWrPh49F+9A9J2AEprG8se3WNvIQ2DdWQc2Hmc2i36fzzKHyS1S9xnxwNZndF+/D2l1Bph9gPrGQ/B1dOBMJAaBznR4fX39MVUiNjU1kYWFBcIImDKWqWGm7FCJdNgHXLn7ZwdGPaCDZ0b/e5mmLwnxs4O32ho2XkFVUijTf8UCj6+rSHAH2bh62T2xPD9bS1bqUOBtyO4QfEsQnbZ+Tg66zp8GvfdKOz9v8zrp0HkzQVEvLk/XHv0Gq/ySugg7yEwAAAAASUVORK5CYII=',
camera_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL6SURBVDjLpVNLaFNBFD3zvtHUpEnaJIhNm6aaVhtKqtWFqBDUgJ+CC11ZqyCKq7oQshdEEReiFCnaVTdFcSF2I8Uv9QPWD2lKmqpVqW1f09CXNP+X5D3nRVdVVw7cGWa4595z7p1LNE3D/yxO3wYGBi4VCoVAqVRaU6lUSLFY5PP5vKCfqqoSURQVq9U6EgqFzuv+PT09W+nxdWhoaJn09/ebWZZN+v1+6GwooGrRD6M4cXgjlJlB3HjSCpOjDdPTn3plWb5JwSZqOvUdrMfjibS3t1sdDgdmZ2chSRISiQTWu1rx7N0SYulOdGzdBUIIFhYWjjAMIxoMBnAcR1ZWVnZzNGuLyWRCJBKB3W6Hz+cDlQIqBfX19YhGoxgbG4PNZkMgEMDi4iKSyWTVJxaLeTndMR6Po66uDiWjC9ce55EtE6iUn0h47NnQDGs6TelPw+uUsdMQBtviQzbxHAlXHoyuN5vNoqGhAQ8nFBQ1BgaemsCgwvB4/sMIXd6GmijazCk4fcfREjiJxq4z2O+tBacHoNWuWqYiUCALjqEVoiUidFM0ATWVcfQc7YLZ0oXlL1EIRME6kxN1DheqEhRFAW0bJS1QoEqNAaFBGBqkVXyPbZtZ1DYHUVwYhrCW4PvHKWSyJVy8/+WXBD07rSgETUFZpfpVjRZRg5sbx1n/JGo9B1GYvw3Cy+Br1kGRZ/BgioOUVH4F0LOHw2H4zRLYcg5F+ubMPUXv5gmYW7pRmLsFhi9BWWnC3JO3uP5Ug5Rhqv+F07OnUqlqe2yyjB1uN8RcBJ2beNi3dEOR7oAVNBRSjZBevMXdWS9Qm0Uul6sGIMFgsI/29TK9iBaLhTTVyKTv9D6UiR0260tY7AIFN2Bm9DVC95fVxbRanYHfM/SIrB6me1f3akfPDSI2fAHz315BdHWgHF+atwhs0Bd6E1k9TMzqh9hnqaxMjsDdsQtGoxdz4xOZQl458Dewvv5gcGy3Nd/Z7OC2t7mh8cVX+fjSqUNXwjP/GuefEpx5owIvdjoAAAAASUVORK5CYII=',
camera_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALVSURBVDjLpVO7T1NRGP/dR18iF+iTQktEkHdDCjHgIGAX40AMMhrkD3AxcWBxMnExTigDPhYJcdFFHHxEo5HBVKlaCk1bFJJiaQuhpbS0ve29x3Ov0jiweZLf+ZKT7/t9v+9xGEII/ufwyjU7O3urUCh4SqWSQZIkplgsavL5vFaxsiwzOp1ONBqNL6ampq4p/hMTE/3UrM/Nze0yMzMzNRzHpd1uNxQ1NEBF8OsbXBk9BfHnI0y/64Bg60Q4HJlMpVJ3abBAoUgf4FpaWgI9PT1Gm82GaDSKeDyOnZ0dNDR14P3SNkL7fejtPwuGYbC1tTXGsqxOr9eD53kmk8kM8TRrqyAICAQCsFqtcLlcoKWAlgKLxYJgMIjFxUWYTCZ4PB4kEgmk02nVJxQKtfOKYzKZhNlsRqmqCXfe5pErM5CpPh2jwbDjJIz7+1R+GIpKJTASicBut6ulssqVy+XgdDqxsCyiSFjoNRRaFhKrwYfNKnQ4JYwNboNIWYiiCNrgikqVgHZbRVZiaW0cBQsN+wccR2Dl/ejuuwgLG1T96MRUqypQWA5ZlUOIDGU1GBag8RgUPsF2YhiC3Y065geq2JTqWyE4VEA7Ci0RUZZp/TKh8giMbBJu7UdUm2shZZ6jsXcSrcIGyqWiWkKFQGH0+/1w18TBlQ9QpG8SxWnNazi7LwEFH7yP53G8Ng8Ll0CbtVQh4JXse3t76nhMqRQGmpvV5lQjBredoLrugGZfp7VR0uxnNI9cx4Xd23jyPasScA6HI722tjYci8U4umXKPjCrK8sYaoqifWAUnOwDKe+ioctMOQ6gPaaDQd+FoPcVWf1Veskc9Zl890eumjsv3qtvM9CsXtpUEUvPgugf7wTD1QCG8/jy8EaSyPK5Iwm802c2XZfnGzl2g2ZP/V37w0NHa2hHYuUbwgs3n/JHfdFyUdT7HoznCZ0GzfIPSMVCsYTU/wbkK6iCy8xjQgAAAABJRU5ErkJggg==',
camera_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK7SURBVDjLpVNLT1NBGD1z7+2DR2poS0uR1tBKhNBGKyaYGDSpkZ0LXZogK/fs2LlSdOFKQyILE5P+BF2pGEx4KApEbGjaRkGqtFRKX9y29+3MNeDKhMRJvpncyXfOd77z3SGGYeB/lsC2mZmZ+81mM6YoSoumaUSSJEuj0bCyU9d1YrPZZKfT+XJycnKC5Y+NjQ3RYzMej++T6enpEzzPl6PRKJgaCjAjufYat6/3Qf72DI/f9sPhHUA6nRkvlUpPKNhBg0kf5kOhUCIcDju9Xi+y2Szy+Tz29vbQHejH3MovpGrncXZoBIQQ5HK5GxzH2ex2OwRBINVq9bJAq552OBxIJBLweDyIRCKgrYC2gs7OTiSTSczPz8PlciEWi2F3dxflctnMSaVSZwSWWCgU4Ha7obQF8Gi2AVEl0Kk+G7HgSk8QzlqNyk+DqWTATCYDn89ntsqxTRRF+P1+vPgiQzI42C00rBw0zoJ3P9pMYLFYNMGyLIMafKTSJKBum3GgcbQ3ngYHC/cnVGIFnRDaY2m8qT9EQ66b3yzfVMBYDlnZMgwd7NcgHEDx4DmCep2CVBk97gGstj+nJOIRgXCogDoKq9ECVbeDoxOy2+6A5xVAV7Goqjjp7MNA1zBqTRGfR95D+ur7S8Cqr6+vI9odwVrFA0mgY9JlXBsch0YVabpGTTWwU8ki0nMJB3IDK1Ic+ko/BFa9UqmY43GVShju7TXN+UglM/D3YgYKVaHqChRNQVWq4Zx/BAdKHR9GFyDQmU5sbW09oEpsHR0dZHl5mTCC7ltNqJoKryNAwZpJlqtsw9nehdXsAhY3liAsnVog/3pMN58G9SY1V6atMAND7kFyMTiKT9uLmE2/yssqru5MGRvkuK8xfI8TLwRirXPp2Z+yboJT7P7YBMG7RNR0vrWha8HClLF5eP8bbVCla6rK198AAAAASUVORK5CYII=',
camera_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALZSURBVDjLpVM9TFNRFP7eT/9SUktLW6hQEqkUCMVABycUm+DmwGoEw+DQDSZWGdSQGJJqGBic2GXQgWjUmuBoTLSmKTWVHymlBQqlpfT3ec4lEJx9ybn3vZfznfud73xX0jQN//OovCwuLj45PT0NVqtVU71el8rlsq5UKul5bzQaksFgqNhstrczMzNTnD8+Ph6g7ffS0tKBtLCwcEVRlMPBwUEwGwKIiH17j4l711FJvsKLjz2wuHqxtpZ4mMvlXhLYQsHUbypdXV3R/v5+m8vlwtbWFtLpNPb29uD29CDyNYv48RBuBIYhSRJ2dnbGZFk2GI1GqKoq5fP5Wyqd6rVYLIhGo3A6nfD7/aBWQK3A4XAgFothdXUVdrsdwWAQu7u7ODw8FDnxeNyncmImk0FLSwuqZg+efyihWJPQIH4GSYfb7ddgOz4m+mtglgxMJBJoa2sTrcq8FItFdHR04M2PCsqaDKOOQi+jLuvw+Y9ZAPf39wW4UqmABL5gqXIBUltEoa4noAJVJoVIIomWiqYHTUgEgzmP33kXDLjKeVV+NK1xBqYiMoUiSzg5ORE5lxmcF7hgQIpCr5koQYKS+wm3vI2rLjt+pdLI6DRBd3NzEx6PB4FAQEyLscrAwMDjpqYmoWy7zYjk9y8Yaq1g4sF9uN1uOO1WVhsmkwmTk5OgsYvpMJtkMgmVTz86OhLjqVOip7kZw8EJzM7OCl+w2jxaZjA/P49UKgWr1YpQKHRWgE6eWl9ff0Z0DHq9Xh4dHRVa8PzZmdPT02imooVCAdlsFuFwGBsbGzxlmf431JWVlTB9cIDaCZHbntKrlbz/jmx7Z25uTmc2m0VRHjfZvtrd3f2JPu9Sbl66fBv7+vp6vV7v65GRkVbyRYPsauPW2ER0odDZ2Sk8UavVDqg9ORKJpP8p4PP5FOrZTycuk4AOAilMle7GI9q3ya3LtOtItzppkSVGY38B11WimHp9JJ8AAAAASUVORK5CYII=',
camera_small=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF7SURBVDjL1VM9a8JQFD3PiEGSCO1ih1gnB0EkdCkKzp0E/0IHIRQc3bJ0LTh28me00CJ06SSddGmQGKpYHbQGVEz8wte8QIqQoYND6YXHvXC5h3POvY9QSnFMhHBk/D1A2C80TbtzU9X3hGX/bbdb1Go1UqlU6Gq1Qr1eJwEGu92uWiwWkU6nkUgk0Gq1cKtmoF29Qdd1qKpKJUmCKIoolUpGgIEgCGi321AUBbFYDPl8Hi/vQzSbBZTLCpbLJQaDAXieZ/V5AIChZ7NZPPRP8dxjxCIonMm4SKXQaDSQy+XQ6XQgyzKTFAlIYMjRaBRPZgizPY8ZeLx+CeA4DvP5HEy74zhYr9dMbnALtm1jsVi4pu1BmUUcAQ0RjMdjr/crwHQ6hWmauDyxIBEHIrGRCfdhGIY35A+zjRxe748Hk8kElmW5G/jEtWsiY9PTe/gYDhGPx71eMpn02DCQAMBoNLp3ad50u13i736z2XiZuX94F248+nPk/3+mb7TL1kbVXJ5aAAAAAElFTkSuQmCC',
cancel=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHdSURBVDjLpZNraxpBFIb3a0ggISmmNISWXmOboKihxpgUNGWNSpvaS6RpKL3Ry//Mh1wgf6PElaCyzq67O09nVjdVlJbSDy8Lw77PmfecMwZg/I/GDw3DCo8HCkZl/RlgGA0e3Yfv7+DbAfLrW+SXOvLTG+SHV/gPbuMZRnsyIDL/OASziMxkkKkUQTJJsLaGn8/iHz6nd+8mQv87Ahg2H9Th/BxZqxEkEgSrq/iVCvLsDK9awtvfxb2zjD2ARID+lVVlbabTgWYTv1rFL5fBUtHbbeTJCb3EQ3ovCnRC6xAgzJtOE+ztheYIEkqbFaS3vY2zuIj77AmtYYDusPy8/zuvunJkDKXM7tYWTiyGWFjAqeQnAD6+7ueNx/FLpRGAru7mcoj5ebqzszil7DggeF/DX1nBN82rzPqrzbRayIsLhJqMPT2N83Sdy2GApwFqRN7jFPL0tF+10cDd3MTZ2AjNUkGCoyO6y9cRxfQowFUbpufr1ct4ZoHg+Dg067zduTmEbq4yi/UkYidDe+kaTcP4ObJIajksPd/eyx3c+N2rvPbMDPbUFPZSLKzcGjKPrbJaDsu+dQO3msfZzeGY2TCvKGYQhdSYeeJjUt21dIcjXQ7U7Kv599f4j/oF55W4g/2e3b8AAAAASUVORK5CYII=',
car=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH0SURBVDjLxZPPS1RxFMU/7800vglR0hkNRDKHUAwpgyxSiSgwsiFqEbRIWkW47l+IYNa1aOGqoFYJYRiBkeAwYQjmj5GBHJOyVAYZR6b3Zt73e78tZMDKnQsPXDjcczlwL+daxhj2A5t94uANghUST6SiwHMR6RQtIa00O6UcpZSvfbWgfP3o89Nb7/Y0AFK97ZHY6ZYj2FhoEZSABZSVJru61fti7MtbIPCfQTyRqg7axJzNBV5OLHIxfo/ZqQl8bVhd+Ur31btkUyNszGVsuP33CvFE6qiIjHa11vNtNklNQwsFV1H0FGvfsygxbBUVoqGpoYbmSw9NVaQto5W+sTzyIGNdezz5uq8tevNCe5SldY980aeshLIylHyh5O/w2sMBYo1hlIZPM0u8n5ibWnkzdC4oWgbOnoiQ+elScBUlJZR9oayEkm92zHyh8Ntnec1FDPSebOFjarG7+fqTuqBWOmQMFH/MMPbqGaEqhzN9A5w6348Sg9KCFlian2JydJhgKMyv8H1ChwJ4HtVW15U7Rm2vo7Y3iEbqsG2b9Vweq7YV4+aoRN0qbRKN1CMi5PJFtNOE29h/zKoMdHR0ZHt6eo47jsP4+Ph8Op3u3H3tiu55HtPT0x/S6fTlf3MwmEwmE4ALDO8Rut36UKVpHfg3/gGTgwnlYQ1oPAAAAABJRU5ErkJggg==',
car_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI3SURBVDjL3ZLtS1NhGMbPH9CnIOhFIijIIwouMg1rDjOY5cuoKHqhbJzjQqfJMiqW5MbaObKTG+VIlpmWTMuaLpfNuRgtGQhF6WTNuRSDabj0Q32wPsTVcw4VgyKEvvXAxfN237/75noeCgD1L6L+I0BJU2gNkXcfN5JQmoLJvY2BZOFlf7Lgovdzfr1nKe9s/4vtVY+UfwPEeXcM3rEkfGMf8fT1AgZeLcBD5Bqdh+CKIpu9/+2PAJK8SmUJwXanD6d0ZrQH5lDX1I1qsxMqjRHmvhkw5zmsz6vAbwCSvJa0/dLQ+xZ1DRbU8E7YBt+DNdxDKWNAsboRF7qmoNZxyCmrRZriHDYfckQ3qW5ulQAk2cW5JvE8siRVbvbMgu+fgfHhNPTdcdTfjaG2fRINPXF0BBJo8yfACEFsLLWPSgBi2Fd/eBEOciEMzOIqaffKg3e45JyCrjMG7e0oKlsj0LS0oqblJCqby3GCU2IHyyJt/43VFHEbw+OLsHY8wR5lGYrLD0MvdKJn5AO6gvNSVaPzFgy9p+GZsOPNnA/W4SocsWYh58wWByUrOoqs3ELQGZmQy+VQKBSgM7ORkX8A9DY50mW7cdxUBHfYBnfkumgiBD8Lq1+D3Op1X365SdP0NMMw0Gq1IOtwqtMlehkGJ9qQOh6P20UAUgG7iEJEz4iOpQJ2ajcsW3xqcD7pGcENVfzsYHlF35UEmg4K6bjmY6TK4izuybmw4j9PgnmiT2LbP2ZePP8OiAqyZHfdgY0AAAAASUVORK5CYII=',
car_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJDSURBVDjLxVNdSFNhGH7OD3NmuaVnapg5lVCJKINWlBJFYGTSz0XYReHFIJFdehEEFkEF3kd00ZWFXSlIYgyDRlsLJVkbEgu3JjpxZqaOOde+8329ZyIs6kLwohee873fy3me73nPeT9JCIGdhIwdxv8XULeS9j6/jZZ+zvlhrnOTznRsgpkZY1k9y6bo+XDiybU3/xSg8Dc3aHVH7XvJlgSdczAOyoBfJBSNrza/GP08QlvlLwE6fbcqo868PIUBzxecae9EcNyDrC4Qn5mG48JNRP1DWAyFqeXrf7ZA5Aqy/bqpthSxoA/FZXaspRlSGwwLs1FyIbCaYuA6UFlWjKqzPaJAqw9Te5e/DXWFpYuPvYMt9barpxpsiCQ2sJLKkmVOEMhkeQ5GbtmloK68ENQNPgYicHtC4zPD3SdU+mBtxw9qCM+ncydnDHKOZJDFphjta76+hBYZgPlHDPus5TAJh6PqEi9RyYrJGMbUXACjr57BVGDGsZY2HDnZmrPPdA7LZD/s7C0aOrtQUHMI6aAb2vsxnA77XFLT+RuCJRNgyUXYtBLIsozE0gokSy1EegnGqPdaJ3Gl5wEKI++AuBewWLGkVmPC/WE294KBxsbGqNPpFC6XS1Ae2qobGDuncv5pUOTHz3sVwqjnz8Etn8/XR2ua8Dz/V+mKiK8HRvYXDXcjk17AOtWSawrVMS9t5zZ6Oyrvm/YU3a0qZaoqzyH5nSGWUPTshuiVtnudfR0H7qSW528rulRtOCLW01Y3e/QbBKBEL0GVKsYAAAAASUVORK5CYII=',
cart=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAE3SURBVCjPhZFfK4NxGIZ/R458Dz7B7wv4Go5WpJW8819mSw3bos202DsTNpT3FTbCRs0YkYPFasvKJhw5kpVEu5y8W0uZ7sPnup+e+34EorH+HIRZQsXfLtKkOSPJCXEOiLGNxgarpPniGW9WnPPN5y+9E3p5I4+n6DaLFHmeuOSRFEUSFNgjj04WDVcbQiCOuWBfnrIlE0RkjKDU8ck1VBytCIE45JYkD8QpECOHTpYIGVzYmo0UO5Q5IiyjqFLDKyNMyUVs1GJq1mvuiZJjkzvCZAhyw3ClrocVQiwwhwc3E4xjZ5f+SoOihpr66Hk1gOWWYGAeHzM4cWBnlEF6QZjLBqBaEbPKdMmpVP0WpbtkVjo/DMBv9aJzxTo2RhjAQoA0fkyFuhsmGWestqFL6cDEP9+s6gcdbFPqiEGjeAAAAABJRU5ErkJggg==',
cart_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJZSURBVDjLpVJda5JhGPYH9Ac66bSdFHTydtDhIAiCiA6CEbGiL1bUTgaO9pHk1+y11ymyNT/wYypMxzanqa8Km9PJxg5EJcWBJq7OzUHIRlfP/VQyKWLRCzcP78113dd1X8+jAKD4n/pngtfrhdPpxMLCAqxW6x1FLpcD1dbWFjY2NpBOpyHLMmKxGNbX17GysoJgMIhAIACPx8OxR0dHODg4gMlkKiuy2SyOj4/R7Xb/Wp1OBw6H41O73Ua1WoUkSQ2DwTCiyGQyvNFqtZDP59FsNkG9RqOBZDKJ/f19RCIRjgmFQiiXy9zRzMzMYC+DVCqF7e1tRKNRYXNzE8vLywKRFxcXBVrDZrMJRDabzYLP5+P7q9Xqgd6AeDyOYrHIM6jX6zwDUiZypVLpKbOBKBQKpI6pqakzfbewurqKw8NDJBIJsKSFcDhMSgLZZWEJRNbpdILdbicyfrtGBpzY3d1FrVYDkUl5aWkJpVKJBnJltgr29vagVCq//fEduN1uShrz8/OwWCyUNFjS0Gg0UBqe44VlCI/e3sDQ60FcU16cOPVDeiLdfKUK3kOkbEXhswwpOYLb0gVcfnpW5ACXy3We2Xs3NzdHScNoNEKv11PSmJ6exl3dVayVTFj7YKbdIaYeQko9pgFf+QAWFrczOzs7KoriR0YePeng+stLeF+24+QXLlppwA8Ae9MTLGl+XTs7O/D7/Tzp8fFxjI2N4cqzc3gj34dOHuZkXWK438Gv0mq1UKlUmJyc7HPAgOpb4gCM8gOuTCf99zI4TTGwntUXsv3z1FP/O6UL4ZoSeea0AAAAAElFTkSuQmCC',
cart_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ4SURBVDjLpZNLiFJRGMe/q1biA99UpqOilqALmYW6yM2MFLQtwU27bLbNLLSgidEethxcGLRTXIkbFyEYQlQu7qJNMD5AYXxcUZMRMhtf19s5NzRnExMdOPfAOff//f7fOd9HMAwD/zN4/ypIJpPMbDaD+XwOaL1PFAoF1sJisQCaps9M/NP6xEKj0QgOhwO63S6k0+kjHk7B5XKxgr+N6XQKqVSqbbPZ1LVaDbLZ7DEKGONhcrVaBaFQCK1WC9RqNdTrddBqtey+Xq+HSqUCJpMJJBKJutlsQqlUwgEfBAKBPM/tdhP5fJ4RCAQwGAyc6IDs9/vOyWRCIpvO8XhMdjoddm232+z+aDTC6VDYGQd/cH4ikQi7IDFZLBaTmIyIJCbLZDLSYrGAXC4nrVYrBmEHLawlls+YyWQYj8cD6FKh1+s5sRiTsZiiKKdSqSSRfadKpSIbjQaEQiFi5QAPZGm/WCyCwWBgyWazGaRSKUtWKBQkujzAQex2O6aviodYL6REIsEsn2vtrdmp6X6ByxQJvEEPRnwh8GfDJ7dy89fEeSqx4NMFxRp1+PqW9+IlgxVOv+ag+Ok9PSiXdtlKjMfjNxBlDxEfLonrDjZ/jGBzywv82geAjy9AIJGCXqfjnlSY3wFQTl6/378TjUZLSPAICQ+DweDh0kF+++WCf8VAwJ29Pz1wcBW4C0LPphCLxZ4i4XONRsMWEK60crm8cnHz6C1s370HwsY7mJx24CcKMPzOhXINqDN3EIlElo2yGw6HVw4++64dXBCL9jcUMw6P04Lhtzkcd7n0bMw8I87bzgXfxuPRSXuHSxM6mstQSPXmdm7+6heR5oijWAuHSQAAAABJRU5ErkJggg==',
cart_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKnSURBVDjLpVPfS5NhFN6/0F1EF9FFBf28GhF0UWrkVQQZkamgbckoGEUsbVJgLl2K++F0bcx0UwZT07bZnG6yzc3hJND56ZYr59zELjSdm36NTZ6+9yOEQQyii8MLh/M857zPeQ4HAOd/4p8BBoMB3d3d0Gg0UKlUDzg+nw8kpqam4HK54HQ6MT4+DpvNBovFguHhYQwMDMBoNEKv17O12WwWiUQCMpmM4ni9XuRyOWQymYKRSqWg0+nWk8kkwuEw2tvbo1KpVMDxeDxsIh6Pw+/3IxaLgeSi0SgmJiYQiURgtVrZmsHBQVAUxU7U0tJSdKiBw+HA9PQ0RkdHuW63G0NDQ1wC7uvr45JvaLVaLgH7jbdE4f57iHxqwpzuBuZVZytZgrGxMQSDQVaDlZUVVgPSmYBDoRDb+WtAhY1JEXaCViC1ju0gQ9h688ehuiMjI0in07Db7WCU5prNZqI0l4wbMBXVbVKvkImbseF6h/2Fj8iterHwgU8fEjCF4kAggOXlZRAw6WwymRCe1WOLEuNg34v974+xFXiCkKEGQW3FZkBx+1Tejnt7e4nSUKvVUCqV6FcL8XOhDge0H3vfqkCvlWN3qR6LnSX40lZ8qaCRbG+O3t2YFf0BV4KO3UeSeomQshiS5zXbeU7s6ek5zSj9vqurCwqFAhrpI1C2Z8zYHtDRGvxaq8AOM0moowTh1ssnBAJBOo+AEUtMXrlcLlQ2VGwufq7HvE2GxEwZ9qLl2Jp7gSVlUarhKS/BgIV8Pp/OI2A8LWZsya7LRPyxu4qw5g4m64/Bpy6F/fUVGDqbWRsztaiuro789ZgkEgkaeeeQ8XcgMyPHTNM1mB4eT9nrLlysra0V8ng8AkbBayy7eoR+W3Um62wuzTrarrstovMnC13nb3Lp9V3T7PhSAAAAAElFTkSuQmCC',
cart_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKoSURBVDjLpZLbT5JhHMdZ1/VP1Cqrtco21kU3zevWVa3NXOuuedVmoU2jUR6CoRwcw1SYHIzJQQxBUXgLXg6Ga+mIwkBBBp08pYGvEOa393nbWC5Xa1389mzPnu/v+/t+nh8PAO9/6p8FBoMBWq0Wvb29UKlU13ihUAikAoEAfD4fKIrC5OQkxsfHMTo6CrvdDovFApPJBL1ez70tl8vI5XKQy+UxXjAYxPb2Nkql0h8rn89Do9G839jYwNzcHGQyWVoikdTzaJrmLrLZLKamppDJZEDu0uk0PB4PkskknE4n98ZqtSIWi3ETicXimgoDr9eLcDgMl8vF9/v9sNlsfCI2Go18EqOvr49PxEqlkj84OMjlb21trao0cLvdiEajHINUKsUxIM5EHI/HQTmUmKcFGHqixezsLHGHUCjcv+sXRkZGUCgUMDExAZY03+FwECf+sNWEhLs2vZq0YMZeZ+zv7ydi/PaNbK6W6elpJBIJEDFxNpvNiIdUWI4bUS7M4/XwFbwKO9DU1LSz5x7odDpCGj09Peju7kafqg1R62UUl50ofujC2oILkaGbENxp2PnrIr21Xdr3xnzRsPLOimL2AehHZ/Ft1YoZbQ1kwutfdzUYGBg4ypJ+rFarCWl0dnZCIxcgTTWjtKQHM38DdMcZbGUasZ4ag6frwveI4tyBSgMWVgs5FQrFLalUuigVtzWwTi+/sOC2Fm9jM3H1ZyXr2ChyZPxKhCTVwkoDdqdb2LXkFiUSiWBM14wM3YXSJzXnvpmsZSNUcyeTqgfz8Snohyc/+0Unju/K3d7eDpFIhJD8/DqzsoDSGoXiEstgyfJL2VDOx5B7YcSz5iOWPQGy460EO04zgbZTDOvEsE6M7/4x5vm9KoYVMdTdwwzVeIjxCg4GfgDxYPxXmKLFvgAAAABJRU5ErkJggg==',
cart_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKNSURBVDjLpVNLTxNRFP4GCkZK6QtftZVHaURLCHFBu7Cu0H/g0rhpdKEbjOlOEzCBjYmycS2kbggLWbHQmmho8GIw0YRWIhX6TKlFgvKYx5253juhBGIkMZ7kzEnOzHe/757zjcQYw/+E5V8B8XicaZoGSil4vSElk0lTgmEY0HX9UIqPDqYA+v1+9Pf3Y21tDVNTU4sWcYVwOGwCjgpVVTE5OVnq6enxZDIZzMzMrPIDn1kE8/LyMqxWKwqFAjweD7LZLHw+n9lvb2/H0tISurq6YLfbPfl8Hul0WhwYjcViCUskEpESiQRramrCxsZGiL8g1Wo1pCgK4TJDsiyTcrls1lKpZPa3t7fFdYpCWZ14iPs1NzcLFUQw22w2Ipg5IxHMTqeTdHd3w+VykWAwKIiEgoLASrU1Tk9Ps4GBAfCholKphARYMAtwsVgMrbhfkh26iZB+B7lcDkNDQ9K+AhFc0oNUKoWOjg6TORAIwOFwmMxut5vwvcDbegGv6GPBvm8e6aCRJiYmWG1dadsLMEODalCovOdxBdDrvYz51Tf4XCDYpUrdu/s/mPQ3J96Nh9nV4E3ojPvD0GGAobxZgNN6EnPfXmMh95bKVG00nTg+Pn6es97j7LcEu0jZpZrg7PpXaFwF5Wo0XcNP5Rf6fBFsaTsWsjIr1+2Z5Ho0Gr3N6yDPLD9gUKYyqE5xquUcTre04Yy9Ew31x3HCdhYf80kkM7NfdlQ4LHsDxNjYGPN6vaaBuNOekE0Fz+f4wAwVXCr8rRcR7ryGD9lZvE/NqzLQVxphyqEZjI6O1n6UweHh4acHZ9L7qHHrUtsV68LiJ6zXV49xsPrHFo6KzofSd92ob9019IbKCKO1/m/zF57K8A8dwwAAAABJRU5ErkJggg==',
cart_put=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKNSURBVDjLlZJfSFNxFMf3FD1kT/VSTz2l1EtE14h6SbB6Ch9CqCgYbI3F2Kabf3A6F/vncOxuNre5Ob3MK7U5mgtDZQ9uOpdzhehAmbAUfetJGIRs+e33u5C0nEQPhwvnnu/5nt/nHBEAUa1Q8vfwOtSIZ/4GtHiu/DytTnTaD0WosRz7Ooz3WRaPnJf/v8Er7iaiX96C/2xHk+MifudDoRCCwSBGRkbgdrufi9LpNGgsLi5iYWEBav4+5Nxd6D48wbtVFsG0FS/G7uCW+Tyu9dcJteVyGfv7+2BZNi9aWlpCpVLB4eGhEJKxGzB8fIrwqhNsQgPTJzmGk/14yTXhuv5C5eDgAFtbW3A4HN9sNptclEqlhMTe3h4ymQx2d3fR6q2HJvIYXMYGT+oNxKFmMlUrpqamkM/nEYlEMDAw0HTMIJFIYHl5GTMzM0wymUQ0GmUesJcg45sh5R+iQV+XpmKXy8XwPC+832g01h83mJ2dxfr6usCgWCxifn4ehe0CGMtZtLhvHztPTExgbW2NuqOvr+9c1RZisRhKpRLm5uZASDPxeJw6MXRcAouhYovFwgQCASrGiTWSQl02m0WhUAAVb25uIhwOY2NjgzYUnP1+P3K5HLq6uo5q3gHHcRgdHYXX68XQ0BAlDUIaJpMJBoMBer0e09PT0Gg0R/88pFrR2dl5pr29HUql8ntVg/Hx8atkPJ/H46GkYbfbYbVaKWnBtaenBx0dHWhraxPeLpfLS1UNCCwd/TqdTtXg4OAOEav+dler1SqFQrFDxCqpVPqjqgG5aR0hLaxrZWUFk5OTAunu7m5otVoQMXw+n3DGpBZisXi7JgOz2SzA6u3tPTGBTCZTSSQSKsaf+V/YTMmdVR+nAQAAAABJRU5ErkJggg==',
cart_remove=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKTSURBVDjLlZJNSNNhHMfXpUtGBJGXbgV1sIOX/0WMXugQ4aGDnYJCUtYfxM2XueXmGrgX39Jtyl7VmdvAveAmCXMo/uc2zTXL3GBjgSl66yQMQnz59jzPQVpOosOPB37P7/v9/Z7P8xMBEJWL2v6rF55FKlDruZ4+r4bGuRdvFm8djyRfQ7/0EtWWK8J/GWiEGkx8kmAxH0Do2wRkH+txt/+yi95NTU1hfHwcdrsdo6OjL0TJZBI04vE4BEHAe+Expj8rESvMwL9ugjPxDt6UGbz/Kar1laz28PAQe3t7GB4ezooSiQSOjo5wcHDAwrhUD120DhaBRyTrRnDdDt73APdGrqFKe+l4f38f+XweQ0NDP/r6+njR8vIyS+zu7mJ1dRU7OzugOUW4hhh8wMxXG55YK1lNIBBANpuF3+9Hb2/vo1MGCwsLWFlZwdzcHBeLxRAMBrm3wfuIZFwIfbGhznrjmIpNJhPn8XjY+3t6eu6cGkQiEWxubjIGW1tbiEajkAceQjpdg1euKjx33GSd3W43NjY2aHd0d3dXlPxCKBRCsVjE/Pw8CGludnaWduLouAQWR8V6vZ5zOp1UjDPfSAqVqVQKhUIBVJzL5eDz+ZDJZKgh6+xwOJBOpyGXy0/K7sHk5CTGxsZgtVphNpspaRDS0Gq10Gg0UKvVCIfDaG9vP/nnJpaLzs7Oi21tbWhpaflZYuByuW6T8WwWi4WSxuDgIAwGAyXNunZ1dUEmk6G1tZW9nef5YokBgaWkp9FolAwMDGwTseTv7lKpVNLc3LxNxJKmpqZfJQZkp5WENFuUtbU1eL1eRlqhUKCjowNEDJvNxtaY1KKhoeF7WQY6nY7BUqlUZyYQi8WSxsZGKsaf+d8YX83UE3LzXQAAAABJRU5ErkJggg==',
cd=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIzSURBVDjLhZNbbtpQEIazgaygG4nUjXRH3QAySvvSKokEgeaBSBGFXJqAQkMxYCA03EJMzcWxCb6AAYP9d46BhqRURfqw5Zn5z5y5bAHYWufd++hbwkdkCYUYEBXCz2yv/dcDtwmOsL/yIkotHU11irY5g9QfIp5tgdmWPtsvBJbB0YOLCuaOC0kHjtIGvhQMfO9PMSYnh2A25sN8VyIrAY4ZNBvezyTvvUsNn66fIGgOXPpGD+jOwr4U4TwBetkhHLFvYy+loqounE74MfxnKupDeBn06M+k55ThukzAYbHe6TG630lBx8dLBbsXCooSUOsBqapJ15mgPwFkEtAplcEcMIjYoiYcE8gLoobPyUcSkOH/JiOS1XGYqDOHLiOcbMCkoDZlU30ChPYcgqgze54JqLfSiE6WsUvBH0jkpmEyY4d4s6RT6U0QoaKGMppHUbKYj/pHwH8ugzvtwXfaRfr+b4HiLwshXsf+zYDoo7AmkM8/DMCdd73gIKlXVRcs7dUVDhMNJBssgyGOSxai5RFyzecreEW8vh9DkIGWBTQMQgMqjxOUOhOkmjOEciPs02wEMiYSJLZeRK+NNrVGph7dDQC+C1yJQLw+x/HtFOG8hQBv4eCHiSBvkrD93Mb1QVKoXYICJCg4VnMRKc8QFsYIZhfBAd5AWrRfDtLrUZYMFznKIF6bI1JcnH4k0C7cWfgp25tHedMyZR90lLtTrwYZKgj79s9l+s86K8t336Z1/g2YLh6PHfCmogAAAABJRU5ErkJggg==',
cd_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKISURBVDjLjZPLTxNRFMb5B/gDkJ0rNy5cYWLcsNNoWLjQlaLxkcjCmLBzYVgVY6WGQBvEhYqaSIsJ1VSYVhBLQIitM20h9oW09jV0SjsttJ3Szue500eKxMRJfnNn5nznu+fec6cDQEc7V3RzZ4gxwk2kiSzhJ0ws9re+PbGTMBKVeVcEvqiMX5KCeOEAscwe7O4oWKyh6Txk0Eiee7PoR7WmIiYD79fzmPHk8S2joESiGsFiTMO0TZOmgZEFchVoV4HUU0s5vFjZhZCrQaVvNECu1eMNE6NmQA+niVokU8HUqoSAVBfZXCWYPksI7UGrIEW3Ao0KQ1WZQY3larNztD6bR8bkUhoTi2l4Y0AwBawGCrScMjJlQCQDmUrJVoE8wdX3xMgMeCGSw8vlHTIQYfoiwuqWMe0MMUGSYVkOo0BJcaomsAsI8SqEiMziPDOQNmL7NLOICUp+RiZr4QILJojuBgl/qgwrbar5e07DGysyjdQyMC2IMDpSGHMksb511MD7uwizS8brtSyRgafNgOe3szAuJLXkd+QekFSwsptLmHaGMcmZcd/Uj9sjfbg6fA4jM6OtJWibuLJVgiAC0SIQzhM5wL9Thi9RhunTWwyZr+Ojbxx8goPBfheXDSfRO3iRb7WxQq0RqUc/s4ArCXyNAPZQFR82FFx7dB6z3qeY3RzVWqx33ILBcQc9A8eUQwcpTe0S0oCTkrmgCuvmASxCCRcenILN9xztl9UzTgZdOHKUY3kVP6gCe7AKq1eBhS/i7L3jeMzdgI7r15J18/2NCrpK//yZ3NsyNpOKtgcPXw3hkv4EnnA3tZnZyN7JQN/xH78zw9072OfqGejeZ2UTeWKY6f8AEAIH5OE7cHsAAAAASUVORK5CYII=',
cd_burn=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKGSURBVDjLhZNdSFNxFMBHCdGDr0sIg3qwiF6jwMeefY0giB56CoMIeuhN8KHXCJuoEYUZpJW0QMyPmWl+ldM5FTedujn34e7c7ma6z/vr/O9maRINfrtj55zfPed/7rUAloPceNxbKzQJTiEmJASP0Kxif+cfLKwUbELu87Sf+YDOmpZlM50nGN+h3xlAxco5lYcE5eLejiEPhaJBUIcPUynez6WYiGfZk6SioGIqR+XuS/YFNhVI5jA/acluH07ycmwbV7KIIf/JBb1YipclNlMgP64KRX88R/u4hlcrJfVM79E8qLGyg9lBRL7Scs0qDEMJiqrWvHufzNczp9M2HKNlKIY7CMsRGPemZZwM8QxERaBLK4kCpIS+0pnYlGDW5U/yanRLBFGav0SxO3U6R1ZUQljxbtRHWoo2pRvvNrg2C7j8uorPKoG2EPwpd47SIsWtIpn0pVUwJJwuE/JEMtjlULt+JE3cwV2Vo/0WNDui2AYiNA2EmVo9KnBv7NI1rfN6MiHEmTsgmJ1dT2BzhM3it2L3agaq7f0ROkd8LDXcYdDuYNTxjUDLfWbW/oxgHuLY6h6uKAR2wZcSkuDZyjAfyjC+lsf/0Ers3gmMF1XkOi4eOkRzjTlZTVR2tJSA6TB89UP/SoFPC1m+Nz5Ae1YBz8/D+gUYrsb+6LZhrvHggxSTdbliMCLFfcsG9sU88111RJ9ayHSeEoEUh4SAiAarYcLafeRRDqYMZqSD/uUCdneWufrrxCQt011VEniEiOA+Bzetb/75MjnXdRbDWVYb6tm6ayExcQyjrUZmKwtaz0Cd9Ynlf6/zrcaP2sbZCiN0yUK45niBWmuea1a4Ys1y2XryF6CZCaxnm2/nAAAAAElFTkSuQmCC',
cd_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKRSURBVDjLjZNLSFRhGIYn1IULF+2CiJA2tWob1CIKa+WqoghaRBERrYKS2kSLCoKIQmbQKAwLSs3FRDbeM03TmmnGG3kZc3ScizPjzJnRuTlznr7/eGFMgg485xzO977v//2XYwJMhZx7aDssVAsOISREhQnBomp/6wuNZYJZyLbaPYzOafwOZ1hIrOKNLNPumEPV1jVlWwLWzbY33RPk8jpeDZqH4rwfjvMtkiElorygakqjtBshGwFmVYhlMa6EqOt7YtT1L+GK5dHlmzzQ8mv19RCzESAvh4S8J5KlfiDMZHhN1GJPYekMM72M0UFAbgl5ZhS6rgLyymuM3ibzaxnWeN4ToqY7xIgXpgIwMJmQ6aSJpCEoAZq0Es1BXGhbWxOzCnC6PDFe9S1KQBDL5yBWh0ZD77QS+BVNfW4SYlqQbiaXwLWQw+XRVN2pAsJj3hUZOUiNmGslZNCdUEWfsHsd30QgjVUWtfFHzGDEm1Sa8GaApSuIuSNAdYefoZntASPzSRrtGq8Ho0KE4YIAp3M2irnLb5jfSfpkWEe1vTGFhl43fS+f0nXhAB3HS2g9s5evlnubUzAWsX8mhSsIc0lwx4UYTCymGfWl6a+rxnnzCKmPj9HHbay8vcH36wd5cvFU7+Y2ZmVrgrJHv6Jg98MXD7RP5/gwluHT2X0kxcyzSqjaCQ/KCT06SsuJ0oUtBykk2+UKQa+Y26Z0rOOrNLlSRtu6vZnCK3p3Fx3HivVtR9kb1/kpHbRP5bCOZGhyJrGd3sPyi0sgpvQtE0uC52oRrRVF3n/+TI5ZjXF/xliDgdr7DF7Zj6+qnMXbJbgv76Czsjhnqyi6Y/qP31nhqL12vr/lZKlPtS0jzyuz0v8BvOcGre/IsB0AAAAASUVORK5CYII=',
cd_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKoSURBVDjLdZPBTxNREMb5B7h41LuJB+8mJiaGxIheOEhCJMiBaAjRiwdR9KAkBEFiNDYtYFQMEiIBDIQgYGlBS0tLwW1LgUILtBbL0m67bWmXXdh+zisFCsFNfvs2OzPfm/dmpgBAQT5ljSOXCQ0xR4SJGOEhdMx20j8/sJDQEsrorB/zgTjWIjI2krsICtv4MRcAs+V8Co8J5IJHuowe7KkZBONAvy2BPmcC04IMiZxUgtmYD/M9EDkQ0DKDqCD7JMm7c1JEhzkKh6giQ/9oQVzdt+dEtFkB+rhEqH5BQaclguXIvtPwrATdeATebWQz2KRXklaZkckwAZXFZncfo/MNO+N4PxlGmzEMVxBY2QQsy0k6zg6EHYAngfCGHktdZVgZaAD34Ro0rx+OMwHO4Rfx2bRFAjx0EzwG5+Lo+eVlu4QYvSYfhOAAQoZaiM4hSmUDMWcvjC0lu0wg4g6maGcebRTcTiJWX5IF/yXOMZp09dGo+wXkP4MITbxC2tWPvXUTuI/VmUMBnYGHVr8JjT4E2+qRgKWvqFxwPYOaNiHtuw/B9gCLnVVwdlSjpqk7lj0Ctx6D1hDKBn+1i3SRGbC0n79rjkZdT6BKFqS8lZAC5Ugs1GHlUwl+cxzbhDu8RPOqBAcPBNKALwFwdjrzTG0u+A4k/23E55/C03oTFjuHsf3G0h6WUaHS8FSjpRhgNg9hYfQRpf0T0loVdgIVECkTj64Y36a88GwpR2XMb6QwlUs/2g33cB0c398gaC1Faq0cAvcY7rYS9Js8sPmV4410spV7moqAxDqW2m/BUHcWU63FMDZfh9HmxiKvnN7K+cNUf+8iZIsGsvUtrA1X0VtzHtMzdrAB++8w5VN65YzcWHlB1b+8kelqqVDuNnyJ5kZbc9o4/wOexAeGRUz8AAAAAABJRU5ErkJggg==',
cd_eject=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKkSURBVDjLdZPfS5NRGMf9B/wL+keCbr3wStArL/RKEdQLCxGEGqLJgkQx0A31IsUmKRbohSxds2Ws0ja3ynLzR20sdb1be3/YptPt03OmSRN64fOel/f5Pt/znPOcUwFU/Ev9A/ctYUQICpqQESKCU8Wu6/9NrBQcQv5FIMbnuMG31Ck/rDMS6WNWgnFU7FJTWWZwmex2rUY4LxRJGPB83eTZR5N36VNyIioIKqY0SvvX5K+BQwX0PKXHEvW0T2fS/4uwXqAo/2TAKFzEL00cJQP5uCkUYuk8029TRFMXoqVADufLFLvHlCo4kpcl46miWFQGBZVbmn1Z1rf00WDCpzG2qvEpATtH8DZqyXJOSJ9AUgwMKSVzDqawfLEnDmUQCsd0pt78FIMkzldJFoMGc2u7SnComH+zhyVJTqeTr9oZ4R/nhGOGioeUQWor8VtmTjImyeNi8n7PUsED4cYlB+MTE3i9XoYejTC3npYqs0qTujJwepM4PEeMeA5Z3y83eDw5ic/nwzAMPB4P9+0PCcWsK4NQ6HsGh/ewlDz7QZeNLKLKVuW7XC78fj+WZbG1tUUmk2FhYYG7th7q7Uvhq0307+cIJyGehT1T0GFmZoaNjQ2y2SzxeJxIJEI0GkXXdZRxc3MzV23MS2uS0qPtDAQO4XUMpqafMCFrHx0dZXh4mIGBAex2O729vdy+fYeGhsZs2UHSpF1hDdYkeXmnyOKXM+bDOWaDWVwbFk8DJuuxfPlBun6UE2aRTalgZeecwcFB+vr6sNlsdHV10d7eTmtrK42NjdTW1uZqamr472UKfjfo7+/H7XZjmibb29sEAgE2NzfRNI2hoSGqq6vLr+b163zP1lPs7u6ms7OTjo4O2traaGlpoampibq6urOqqqrjPwDsCp2+T9HiAAAAAElFTkSuQmCC',
cd_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKrSURBVDjLfZNbSFNxHMcX9WLgQ1h0kXrp/lhQdjGIEotegiCKiiIjjSx88SEpzZRIpWKFnGmCOubAS1YkmZlpXnLedtrUIpdTJ9N53Jnb2cy55fbt9z9OmQke+Jz/4fy+3+//rgCgCOfik7qjRD7BE3bCSQwQKlb7Xx9ujCQ4wv9Jb0H/qIRh0Ycxz19YHdP4zI+C1UKayCUBIXOdtmkAc4EgrBLwpsuN6l43Ohw+eEkUIFiNaZh2IWQhgGMFlx/y4yG1ptmF0vYpGF0BBOkfNZAC8/VQCCcH0MchImBx+KHRiTCJ86JavReqLyIGpyGPYIJeHmp9jGCQBQSYV+69nuZX2yuhqNmOwiY7+qzA7wlAZ/LQdGbhmAUECpBoKM45wE3Uz68JxwIMRosL6rZJChCg+irgPS+hsnWQCWyM121meMg0RqMxTQHGsTkYLRKrG1iA+MP6h3oWUEjmVxTSafaw4jgRHWI8tTweN0v2oarHJdNnnWEacTFA1SiAa5hAfoMNXUPLA+6UHcejmku4ULAHZZ0O9IYFGAwjTnCNNtlcQekmMYgU7THc1RxGkvoAEor348Hb86jsfoHUqrM4pdwCfti1OAV5EduHvDAKwOgMYHYDyZoYvOM5VOvzZWN5txLKhlSodXm4pY1HTM5aCvjILW6jn7ZGoD365QT0NuAGzZeZn9enILcuGY9rE5FZcx1ZH5JQ0JKNK6Wx2PlwlX/JQbLTdhntQKsFuFy0FxVdSmg7nkHdnofibzkUchuqliwkak9jd0aUMzpNEbHsKFvdQXynEZwr2I4zL7ciTrkRsU/X4WrJEXDNmUjQxJN5s/tE1r31K14mfkTCT5sP/eOz6DG7cDA3CtfUJ7ErfdNkXHZG1LLbuMJ1ZvA70iO82+6vxoY0xZpw/T9xhOmhB93shgAAAABJRU5ErkJggg==',
chart_bar=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGvSURBVDjLpZO7alZREEbXiSdqJJDKYJNCkPBXYq12prHwBezSCpaidnY+graCYO0DpLRTQcR3EFLl8p+9525xgkRIJJApB2bN+gZmqCouU+NZzVef9isyUYeIRD0RTz482xouBBBNHi5u4JlkgUfx+evhxQ2aJRrJ/oFjUWysXeG45cUBy+aoJ90Sj0LGFY6anw2o1y/mK2ZS5pQ50+2XiBbdCvPk+mpw2OM/Bo92IJMhgiGCox+JeNEksIC11eLwvAhlzuAO37+BG9y9x3FTuiWTzhH61QFvdg5AdAZIB3Mw50AKsaRJYlGsX0tymTzf2y1TR9WwbogYY3ZhxR26gBmocrxMuhZNE435FtmSx1tP8QgiHEvj45d3jNlONouAKrjjzWaDv4CkmmNu/Pz9CzVh++Yd2rIz5tTnwdZmAzNymXT9F5AtMFeaTogJYkJfdsaaGpyO4E62pJ0yUCtKQFxo0hAT1JU2CWNOJ5vvP4AIcKeao17c2ljFE8SKEkVdWWxu42GYK9KE4c3O20pzSpyyoCx4v/6ECkCTCqccKorNxR5uSXgQnmQkw2Xf+Q+0iqQ9Ap64TwAAAABJRU5ErkJggg==',
chart_bar_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIESURBVDjLpZM9S9ZhFIev/+Pz+BKGUBAZJUQEOqRCi6hDYAYtfQdXySki+wYubeHQEoHQWoRLYBGIOFhkJTg0tCShgfq83W/nPqfhH9SgIXTGM1zn+p3DKcyM/6nqUc3553uWVYkCOStRlCDKs9lLxYkAISqTg6cQVdRAsvFyo35yA5eUmJW9QyFlo6+ng6bTkwNaToii+KRINkK1QsPJ0QB7eL/coiqWBEtC+/IDQjR8MpIo3bVM3ed/GEzdBFWKnClyprGpBDFcyKQMPTWjflwES0IhAh/egyQYHqXpIj4p7VhG8J0F4tIxgBBLQPCQBJJwGIyQFBeUlI3eLkVbytzKjKUoxJhIPhFCoqI+gAj4AN5DjDRbio+Gi6WFT8ZQ/xqdXzO23UC29xnQAXzLU1X3e3IIECOIIC6VBlGJ2QjtZW5MbHB9aIyLZ67ydusF619WONgVKtoup+JcaZAS2lJ8LAEuKu3GC0YHR8iVzEj/NLlIjF0bJzYOqVjb/RWhjKNOcekPYL/5g1rRy52hOQDuTT3hyrlhMKOqbQcpwfgE5AwimBOiGOf7aojC928HbO2ssbmzyvz0UxZez9Dd0VVe4VHXLXRpFwuCpYyljPZmlpd2ICqWhbMDt1n/9Ibx4UlefV6ks6iy+vEd9Z9DFCd957G7FxaAWeA00AAW1x/vzP8Cqr99v3YC63EAAAAASUVORK5CYII=',
chart_bar_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIOSURBVDjLpZM7aFRhEIW/m73r5gWB9UFEzAMsjIIIgkViFzFgJbHR0kqI2Ahi7KzsrEWwCQZsBFFIExWCqJWKClppEDTxEVCym73/a/75LW7AFIksOOXA+eacA5OllPifyTdbTt9dSVEVLxCj4kVxosxM7c3aAjivHNvfjaiiCSQmHrxstO/ABMVHZWVVCDHR11VhzWj7gJYRvCg2KBITLu+gaWRzQLp6uWxRlRSEFIRi+ArOJ2xIBFE6q5GGjf9wMH4cVMliJIuR5lvFScK4SIjQVU00toqQgpCJwOtXIAEOHWbNeGxQCl9GsNsyxIQtAM6XAGchCARh1SVcUIxTQkz01hRtKRefnEvBC94Hgg04F8jVOjpEwDoIAbxnraVYnzBe8bHs4pTc4/TMU+LyF6Rex41OcLv2jVzN+mXnwHsQQUwoHawD9n28w9jgAgfGL1AbPoh5N8+HZ48ZwdChhS2FxoC1EALaUqwvAcYre97fYmR8ks5PC2QzZ+levM/QQJ0jn7+Sp8LAxggiqFHMBgd9zSU6+4fh5KW/5V3bTb0I5FqYUjg6BjGCCMkIXhL9fVVEodGzi+LNHD0Pp3DmOwXQbFT4XcvJb9ROoLM/SU5IIZJCRHsjc7PL4JUUhZ3bJ+l/Mc/Qji7ySpXmirD4o4NH7ihZu+/8/MzAdOvX8vlKzAZjJS0luDkxL9f/ALqCe8YKiajkAAAAAElFTkSuQmCC',
chart_bar_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKESURBVBgZpcFPaNZ1HMDx9+/Z79l8xubWhDVmppuynpJFS4nQ6UVDlCFh0EkQKaJ5t/KmJ0O6deiQHTrlQcjlaVGK+I/SyZwH10gtxT252WrP9jzff5/v5yveRJ5OvV5ZSon/I6eBz76bT1EVLxCj4kVxonx7aE3Gc3IacF4ZLrciqmgCiYmx61UayWnABMVHZX5RCDHRUWpi2SiN5DRQM4IXxQZFYsLlBZaM0EiejhxOPKVKCkIKQr3vU5xP2JAIoqwoRqo2cm9s03u+NnC6qW2Q2uMLZPbhgZynduwEVbIYyWJk6abiJGFcJEQoFRNvd52jtPKd01392+joG+Lfe68w/ePJE3kKQiYCNyZAArz+BsvGY4NS94rERG/reT4cmaPzha0s3LlNc+ZpX9lD26qXO/LkPJkIOAtBIAiLLuGCYpyyruVX9pan6ezfhaucork148/JaarO8U1TdUVBrQMRsA6sBe9ZrinWJ14qXGV08wSd6/dgZ78mK/5Dsa2dPFSolLdTMYZcjYUg4Bx4DyKICXQujrN/0206NryLffgVhaLgq33MXrzGeM8gq0rtmJol17oF78EYCAFCYCi/yO619+neOIL/6yRNzQm7uJZHl67z/eoh/qZAW3DYmqWQ6gZEwDqwjpm5CU7smyXSS+XWlyiW2sJqfhu/xdTgPuYVjDN48Zi6o6B1AyHAlq0wvI2p/A6vDX9Ey4NLzIxdZnJ8gV/OTjD6w36WiiXK3QNs7CkTxOOMIzu683jSICQnpBBpGTrDJx98DIXI5E9nmPn9AV9MH+OPhV66yz8jQYkSiaJoVLKUEs96f3uXebP/xfytV/tIRXfFzM0fHPl86i7/4QmxWH8VvTzD6wAAAABJRU5ErkJggg==',
chart_bar_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIxSURBVDjLpZNLiM5hFMZ/33+GPsMwI5SFcakxI9QIuZcyCyVKdpMdK9vJbcVOsrexEJGFUpRSiA3FjGsoU+65DZLxff/3dt5zLD5pyqQpp06nzuJ3nufUUzEz/qeax1oeOPfFsipRIGclihJEObVnTmVcgBCV9d0tiCpqINm4ODgyfgUuKTErX34IKRvTJjVRczp+QN0JURSfFMlGaC746WRsgB3c2/iiKpYES0I5fz8hGj4ZSZTqhMyIz/9QsKkXVKnkTCVnfj5SghguZFKGrsmPOLrqKi+vvZ+1oPfM8GhAYUlABAYHYOAOiFBzEZ+UMipZShY3X2Z1z0o0pX1/WwiRiggED0kgCT+CEZLigrK69SZz5q1g6uxlvHt6rv/Q5b7+d6IknwghUagPDQU+gPcQI7W64qMxxT6yrHqD1hlt5JFLdCzdxRba2bagj62LduLrnkLd78vhN0AEcYmYMiurV+hYsh38fe6ePsuUNsfE4ed8enGdKAFX9xRaNq7iXAOQElpXZuQnrOmItLaXaHgFpuTaAPM39tM5/Izkaw0FVrpRFhp2ipDo0fO0dfSg5WNMHct3dKHxA9XqEJ3d25j2dhBXBgotHaQEa9fB+g0gwubZt+juXExL61dMvkGliXsXhgBF/RAzF86l/c0DpsdI5XDvEdMkWBAsZSxl1vXdZ+Pu4zQVrzH5DoxObEHTpC4+P33ImZMnqIwV59vHln+1rC2mhqmOavszUcXM7v0CRRWCf6vfaXAAAAAASUVORK5CYII=',
chart_bar_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJaSURBVDjLpZM7aFRREIa/u7mbNW5IjEQ3WQlBIkaLiBYWvkBi7H0EDGgjaiNY+uoECwlop2glWGknNnZBIRKCZMWooCz4Im6uya5iNnsf555z5lishYVKwCmH4Zv//5nxnHP8T/l/al66X3VWhNSAtUJqBGWEe2f7vBUBVCrs3bIaI4I4MNbxaKa+cgWxFlIrVJcM2jo621poxLJyQBgbUiMkWjDWofwMy7H5M8BdPt9MUQSnDU4boo0XUakj0Q5thFVZSz2x/1BwYARE8KzFs5blWUEZR6ws2kJb1lH/mwWnDZ4x8KIERsO27TTilEQLUdq0kLR6mFj/BaDSJkAloA1ow5JyKC3EStDW0Z4TJBTOTZx0OjWkqUYnGqU0viSKjDGQKNCaJIoohDN45a/sKqyl/GmRSIZpTVN6ZjZRXF9ksbaI35Pj6dJjfIl/bVYKrRQPXr/h0JZ+jh07QRiGDAx8YWLyGQf7YPTwKFk/SxAElEolzDvwJUogTSGOmap95/1AF0eHh7lw9Q7fPpco9vYwNDSEDgNuXL9BEAR0b17H6SOn+Dj3Ad9FMfyyMFtfJt/R0Qxn63Eab6cYGxujq6uLRqNBtVrl5u1b9O1szqzpWIMvUQxaw+499Lx6zXQQALD88i4ZTxgfHyefzzcPLAzJ4KGfR9AHc5UveFdGrjnRBqcMdfWDSqHCjv37KBY30NmeYWFhgXK5TC6Xo7+/n0KhQD1MmK9UeDI5jff7Ow8ODrb09vYO5fP5h8VicV0ul2sBMrVa7QxQ6e7ufghklVJ2fn6+Gobh4Z8+rmPjNq74hgAAAABJRU5ErkJggg==',
chart_curve=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJYSURBVDjLY/j//z8DJRhMmJQd+x89/W4IRQbY1x5L8590dzmy5PuIqC4gfvA+PPIyEMfhNqD06H+L9gfG9p33/jr23OMEiX30DTj8yT/oFxCf+hAYfBeIfwPxIyBWwjSg5Mh/tYZHzDr1D34aND7Y9tXOsf2Lg/O/z85uNjCFn908lT56eH985xXwzXvygwYUA4yLD/9Xcm+QlS572JWesP7XVyOL79/MLKci22Rc/6DXvPH+X8um+79t2u7/tOu4/w9ugFHxof8wha+1LP89NHT9iaxZIf/BCpWie7/Vi+/N/25kqvrN2Oz/suiO6QgDig6ADfgtJrX0p6TMb1u/Xd+5Eh9M4k16yCyQdH+HYOK9H6JJd+tgBv7U0j3wXVvvA9wAg8J9/6sNAvT/8gr++8Mn1MYQ8aCFIfzBf6bwB3+Zwx/8Ywu7H44e+j8VVX4hDMjf+/8/I6v/fya2OyghHHCn3GuRw3TvJTZnPJdYnXVbbA436Le49Aa4Afp5u///ZGAJ+c3AIg5T4DXT0stjpuULj1nmD9xmW6x1nWu2z2W+6RenBcbxIHmga6XgBujl7vw/R1TDAabZscNommOn0UeHLsNFDj2GPDBxh37DDrtJ+u8x0oFu9vb/liU6khal2jPNS3UfAem3FmU6Gej+tqjX5rBo0rln1qI9GdWArG3/jTI0/Q0z1N3UAyxdgTQ4NQpreMjCFAqpOoHZRvnqUhpROhmmxRo8cAO0M7f8187Y/F8rYxMQb/yvlbYBiNf/1wTh1HX/NUA4ZS0Ur/mvkbwajOEGUIIBf5BxjDvwFIUAAAAASUVORK5CYII=',
chart_curve_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKLSURBVDjLY/j//z8DJRhMmJQd+x89/W4IRQbY1x5L8590dzmy5PuIqC4gfvA+PPIyEMfhNqD06H+L9gfG9p33/jr23OMEiX30DTj8yT/oFxCf+hAYfBeIfwPxIyBWwjSg5Mh/tYZHzDr1D34aND7Y9tXOsf2Lg/O/z85uNjCFn908lT56eH985xXwzXvygwYUA4yLD/9Xcm+QlS572JWesP7XVyOL79/MLKci22Rc/6DXvPH+X8um+79t2u7/tOu4/w9ugFHxof8wha+1LP89NHT9iaxZIf/BCpWie7/Vi+/N/25kqvrN2Oz/suiO6QgDig6ADfgtJrX0p6TMb1u/Xd+5Eh9M4k16yCyQdH+HYOK9H6JJd+tgBv7U0j3wXVvvA9wAg8J9/6sNAvT/8gr++8Mn1MYQ8aCFIfzBf6bwB3+Zwx/8Ywu7H44e+j8VVX4hDMjf+/8/I6v/fya2OyghHHCn3GuRw3TvJTZn4mZ7P82dEv4trc//f2SLw6cp/mrX4Abo5+3+/5OBJeQ3A4s4TLPXTEsvj5mWLzxmmT+ImuJ+rXF14v8tV6b+v/Bs1//+3Vn/w/t1/5tnS/aAFevl7vw/R1TDAabZscNommOn0UeHLsNFDj2GPLHtLt83Xp7wf+O1SaCw+t+zJ/V//550kAHfwRp0s7f/tyzRkbQo1Z5pXqr7CEi/tSjTyYAZ6FNt+H/blTn/kcGmS1NBBkAU6GZt+2+UoelvmKHuph5g6QqkwalRWMNDFkRb5kh/796V9L99VwJYc/vOBFQXaGdu+a+dsfm/VsYmIN74XyttAxCv/68Jwqnr/htkJP4P7tH437srBWwziAbx4WFADAYq7gDiTyBnQ+kOkDgAwll4PHHRgLAAAAAASUVORK5CYII=',
chart_curve_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKgSURBVDjLY/j//z8DJRhMmJQd+x89/W4IRQbY1x5L8590dzmy5PuIqC4gfvA+PPIyEMfhNqD06H+L9gfG9p33/jr23OMEiX30DTj8yT/oFxCf+hAYfBeIfwPxIyBWwjSg5Mh/tYZHzDr1D34aND7Y9tXOsf2Lg/O/z85uNjCFn908lT56eH985xXwzXvygwYUA4yLD/9Xcm+QlS572JWesP7XVyOL79/MLKci22Rc/6DXvPH+X8um+79t2u7/tOu4/w9ugFHxof8wha+1LP89NHT9iaxZIf/BCpWie7/Vi+/N/25kqvrN2Oz/suiO6QgDig6ADfgtJrX0p6TMb1u/Xd+5Eh9M4k16yCyQdH+HYOK9H6JJd+tgBv7U0j3wXVvvA9wAg8J9/6sNAvT/8gr++8Mn1MYQ8aCFIfzBf6bwB3+Zwx/8Ywu7H44e+j8VVX4hDMjf+/8/I6v/fya2OyghHHCn3GuRw3TvJTZnumq0n+4OFHi9x4nl305X5kfXDHmvwg3Qz9v9/ycDS8hvBhZxmGavmZZeHjMtX3jMMn/QW6p5+XyJxd/vW3v//7u24//XFUX/T2fr/tnlzJILVqyXu/P/HFENB5hmxw6jaY6dRh8dugwXOfQY8ux0Yb77Daj5/yTf///LBf//b1P8/7rL4T9Q/B5Yg2729v+WJTqSFqXaM81LdR8B6bcWZToZMANBzv53dt1/ZPC+XuI/SBxiQNa2/0YZmv6GGepu6gGWrkAanBqFNTxkQTTQz4+/zE3+/x+o6UcZw/93QPwwg/k/UPwJ2ADtzC3/tTM2/9fK2ATEG/9rpW0A4vX/NUE4dd3/sriU/8dS1P8/K1f8/6qS9f/dFMb/u33Z/u9wZa4iOtcdCZetANp4HxoLj0GaQeIAMa12DZDYXzMAAAAASUVORK5CYII=',
chart_curve_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALISURBVDjLpZNpMNRhHMc31ExNmjRTDdVMsxXLkntnTcmxJTKFMK6JnK1KcheKDDZXubbSoRdlNFFNGFTIbeiNsyjHnzI0ZrFk/621+21bx6wx9caLz/Pi+T3fz/Ob56AAoKwH2WAU2QyP+wNO6xKY3WgOsMseKJQvTrm6p0ohplzcuqR4/lsQ0QQmhzA0SxkUW6QPbv47xz9t3zBjd3ZeStu0g+OAFJGUESnUtYLwRqjHjyhqxxFCvVtE+dwxC84vc5ZklmV1dHnhrJUNlW9ty588ZS+wzSHiVwkMwxpAPRm/b0/kcOqF82/m5wyYpIBhwpXfyTCOyAjKLJT0Frji29sktD+xQgeX7ikrGoTVY3nhhJaJZFj/hFA+vD+YeMFOe7QwVhOF6c4yYHYUU53FaEm3Hl8UhNbKBKJdagVC1b0i0zPvyS3eRLayz7Didp+hSteb+fMT3XELwu8lGKtNg6DrNRaIRnQ9DSBlAr2QGsTo2euKlVUkC9t2JFNciUSKCwEFF0LMCi2S8LpjJWJBIwQDl8FrC8KXZ77oyXZDW7aD+qIguBrYsNEOCpv65VsPiE3Kn+y6DjHZgrl+L5AjHpj5HI2hPPPxKeNDH1cOUffKBwgpSk4iitLu5fDd9IOcsU9RS2FPkMPu4HfHoI9rSfalM4yk3aqtCA4HvcPjnTTz5XBkskZlZ0UIxIJ6kEO++D1yDtPSTnq5LJFjlh5/zTvQuVQBk3BtVWYEPc/7qjqvpzwaHRWZ+NHqjLkhD/Dar+FrrsXPqjSGJjNBe5CRSM9ZLbhYDgO2pp0+W8PqZQoLmCHQ+9AJNdFqaHpgg7oEo9E7/keOy24sWEON5q7NNg6jbV0R0APLQGeXQotdggR/HQhbciFszUJrkgWe+x0AK/AeaH6vligGzbdIxopAHmdTFZLjRRNV37YRVWVY1pVG6VD/9xv/AJGzmhVs7+fUAAAAAElFTkSuQmCC',
chart_curve_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALXSURBVDjLY/j//z8DJRhMmJQd+x89/W4IRQbY1x5L8590dzmy5PuIqC4gfvA+PPIyEMfhNqD06H+L9gfG9p33/jr23OMEiX30DTj8yT/oFxCf+hAYfBeIfwPxIyBWwjSg5Mh/tYZHzDr1D34aND7Y9tXOsf2Lg/O/z85uNjCFn908lT56eH985xXwzXvygwYUA4yLD/9Xcm+QlS572JWesP7XVyOL79/MLKci22Rc/6DXvPH+X8um+79t2u7/tOu4/w9ugFHxof8wha+1LP89NHT9iay5aNKKo9vXVvyzrz27/LuRqeo3Y7P/y6I7piMMKDoANuC3mNTSn5Iyv239dn3nSnwwiTfpIbN85tU9e9fk/Xt7e/X/29vCekDqfmrpHviurfcBboBB4b7/1QYB+n95Bf/94RNqY4h40MIQ/uA/U/iDv3ltc/69vr7k/+8vd/5fXx/4/tpaf1OwIYoqvxAG5O/9/5+R1f8/E9sdZKc75m7uvbgu4MeP11v+/3jW9//d3a3/Ly/z3HRpsTvrb3HpDXAD9PN2///JwBLym4FFHKbZb5aV99Ilbt/f3Fzz/8fjpv+HOoz//3q75v/5ec7/zs12DAa6VgpugF7uzv9zRDUc4DZ3GE3LmWT59f7e6v8/Xy36/+1O4v9D7Yb/vz8s///h3rb/p6ZYXTs50YIXboBu9vb/liU6khal2jPNS3Uf2ZTpvD04y/Hhe2DAfX9Q8v/rrQgIvh0L9MqE/w8PTvp/tMuoDmFA1rb/Rhma/oYZ6m7qAZaux6bbFTw81Pf/54vpYNu/3o4GesEITH+7l/X/2/ON/w8267462KitBTZAO3PLf+2Mzf+1MjYB8cb/O3pc/397c/f/z3d7//94BQyDV6uR8Nr/vz9f+f/kxJL/+6rVVmPNYUDnvTnSbvDtcKv+t0PNut+ANn07UK/5bX+txjegpm97K1W/7S1X/ranTOkwAORfnQagLb+TAAAAAElFTkSuQmCC',
chart_curve_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALJSURBVDjLpZBrTJJhFMdfxUzazHQzp5aK84IgysULmNc5SXMFloVp09CJqDOXt+XMiViKlnkpxVZG5ijLlZZmZpeZpjVdfelrwqsfa3nJlAB5T680kNba2vzwe852nvP/PWcPAgDIdjAeIRXTkCGfS92WIKZ6WsRrn7tnebmUlt6Egy4JTn7Cyfy3oHwK2A0oK6ZRZYi7rCJu9lYO8ye/847qcGaWU47N4ehxFnC8/xaUvQU/yQIhsAbV0mvRkbXouIYfsfHYajw30jS4yk3yXklMXlk8xF9PvopK/hCwSifB+6Bkv3vFfFPe6QHdGpOtWQ/jdFi+xKpBm8Nr1QaOVK2PrFdro2VqzCxglk6AafArhYPNMxK0lmGvYrTPp0Sl9y9VKTTMUN91VhjczZDJtwQl40aBfq+bUuu6Tx91ZEyzS4i222fPE/Zkq0cdhaqf+XcidLkKBmR104haCm1cQw1aNgvoZ19DFZ0fbLB3xDZ2O9UjaegFRICCtQA1EAQoZntCLchRMNZqh9JB0EUGXocnUUvy0W0Jil8BWO3ggbXtZ9PaxcooKOplY3k9LEzYHQznB47Dg9k2KOvnQUKLi2HR023YLAg+8wK0iE2qHrFxMQkKe8Nh8GMnPPxwzRjsm22F1pdl0POuCcRKLoQ27DTQ6qwcjMNBRc/hpjM51hSOkzE78ZWN4StjxdA4WggXR0QgGRKC9GkedE3UwanbkeBbg2wYA7TCZ8ApC3Rll1Ovh5fTFvD6LeNGANyfaQXl+2bomW6CW1MyXFIA8gkpiJSJQKpGvrhXIvTfgoIRYIoDeAyxP9efz0nAKzFFTsKS2tyx+BZn7MAlByxTEQGdbySQ3csFzyrrFY9zdiHmP6DmDwNVPAQU8ROcx0ARDeIMQMAmuY+AjEOV2kFWTzx4VTqBn6gNyDn9YBb8D/jKax5VBHCuREiW/V/ef2+t0gzxSwAAAABJRU5ErkJggg==',
chart_curve_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALPSURBVDjLpZFdTFJhGMdJyq2cLttss2wrZ1TiBx9m0gwkJyofIuiCcrl0IgZLh6iDPhQpxajmpBU3jtaF1WraRc2yRlMkL7pgLi1vAo5e2dpERSUkeDrvaX7VuvLid/bs//zP7313DgkASNuBeGS1jEGF1V2+LQHnxlit2OJ+unnpk18w42A+2fkJnMr/C5o/Qo4JY3Jue8Lcu57dKFsQlY4uiqWrOJ/mJWVunBDODE7yv4ImJ1AMM+S0NixIa8cGl9lc01JefsSfz8tdK/p5xckLRYKFOX7piuA+ZtgiYGpHIbnQcOhgy7RZeenl6jIjJ7CSzXqw+SRmG3bvVLs3zDJ6Q7md3iC7yxtZFzC0Dlgr/khlRabpBcHNLx9uwJ6lNHpCx7SeRwHGyaMrzGx4UtFl3RA0DhOC0P4DfcHEpNCZkneBPVWYJbZ6mry32vs2vsrzM6Ha3bomDKamDweoGfPrAprmA1yjlWaGY+Mjv+L2dZLk2C2SDIMoGRYmy7BI9Dmv7O+vHzySsrohaLAD7Nglhqjob2w2O0alUjV3dHTM2Ww20LXp/GUPi6ZEtrzP+lbdd5Sh3YuCwknUJQSZ9e8hSNpZPh4TlyQUCs09PT0wOzsLbrcbRkZG4PrNq0ua9nrfl6lJInM6nYA6qEsIMq4MQW/C8Tw6nV6i1WrnXRMTIFWUhAQCQUShUIDFYgG9Xg9oFggFIJfLB5EEdQlBuvoNsJrSErOKaeMGoyGCBKcbM+s5HI7fbreDy+UCh8MB/f39wDnLDucY0q1IYDQaA38EqkFg1J0QZ7CpllplrZ+4gVQ6xOVyQzweDyQSCQGaUSYsE31FAqVS6SME1MuvgVr3ClIre6FIchFMd7rhcd9zGBgYAKvVChqNBnQ6HTGjDO1M5m4QiURTW34NhUIh49em8fl8b01NzZJarQ7gBGUyWSVOPj4vogztUAd1fwN412+yCB2ePAAAAABJRU5ErkJggg==',
chart_line=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGgSURBVDjLY/j//z8DJZgsTV+9fAu+uHo8+GzvXECWAV+c3R//mTn9/ydLu4eka3ZyY/ts63T3k4Xt+4/GlqS74JONY+9Hc5tdH4wsmAmGgWv9xQKX2nMPnapOF4A1WzsEfjSzefLB0FwUHoi/szPX/05P/f0rOWk9ugHONWefzNl44X/B/L3/o7LXnn1h4fitN6i22Tx7W5tpxqYHxmnrChh+p6X+/rd10/+fsbF/f0REmiE0n7F3rDz5wb7s6Bu3gt3Vz80db69zTd1mlr11tUnGxt89Cw/8N0ha9YDhZ2LC+p8xMb9/hEdc+h4Ucu+br//JFXFNi5zKjz20KztiDzIMGFgzP+iZboQZbpSypsAgaeUjvfilqIEI9C9bf8rk3Wd8kz59sHV+BQysa8DA+vNe1+Trew0DfrwJCehfCceqU8fsy48ttS05xAkMLANgYP39N23K/3fq+n9wpkTXugsFQP8+B/r3DdC/pciS77WN1r9T0/v9Vkl7PU4DnKrPPJi85uJ/oH9fkpUXHCqOF9iVHn5gU7S/gG6ZiaoGAADG9LhB7Kzu8AAAAABJRU5ErkJggg==',
chart_line_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIhSURBVDjLY/j//z8DJZgsTV+9fAu+uHo8+GzvXECWAV+c3R//mTn9/ydLu4eka3ZyY/ts63T3k4Xt+4/GlqS74JONY+9Hc5tdH4wsmAmGgWv9xQKX2nMPnapOF4A1WzsEfjSzefLB0FwUHoi/szPX/05P/f0rOWk9ugHONWefzNl44X/B/L3/o7LXnn1h4fitN6i22Tx7W5tpxqYHxmnrChh+p6X+/rd10/+fsbF/f0REmiE0n7F3rDz5wb7s6Bu3gt3Vz80db69zTd1mlr11tUnGxt89Cw/8N0ha9YDhZ2LC+p8xMb9/hEdc+h4Ucu+br//JFXFNi5zKjz20KztiDzIMGFgzP+iZboQZbpSypsAgaeUjvfilqIEI9C9bf8rk3Wd8kz59sHV+BQysa8DA+vNe1+RreV94S96UiE9pff7/I1scPnlW6NWgBCLQvxKOVaeO2ZcfW2pbcogTGFgGwMD6+2/alP+rYhz+Na5O/L/lytT/F57t+t+/O+t/eL/uf/NsyR4G17oLBUD/Pgf69w3Qv6XILnqvbbT+nZre74RWlz8bL0/4v/HapP8g0LMn9X//nnSQAd8ZnKrPPJi85uJ/oH9f4opOn2rD/9uuzPmPDDZdmgoy4D+DQ8XxArvSww9sivYX4DLAMkf6e/eupP/tuxLAmtt3JiBcQEzqAypsCe7R+N+7KwVsM4gG8cFhQGwSBiruAOJPIGdD6Q6QOAAJO6JfeUJqowAAAABJRU5ErkJggg==',
chart_line_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI1SURBVDjLY/j//z8DJZgsTV+9fAu+uHo8+GzvXECWAV+c3R//mTn9/ydLu4eka3ZyY/ts63T3k4Xt+4/GlqS74JONY+9Hc5tdH4wsmAmGgWv9xQKX2nMPnapOF4A1WzsEfjSzefLB0FwUHoi/szPX/05P/f0rOWk9ugHONWefzNl44X/B/L3/o7LXnn1h4fitN6i22Tx7W5tpxqYHxmnrChh+p6X+/rd10/+fsbF/f0REmiE0n7F3rDz5wb7s6Bu3gt3Vz80db69zTd1mlr11tUnGxt89Cw/8N0ha9YDhZ2LC+p8xMb9/hEdc+h4Ucu+br//JFXFNi5zKjz20KztiDzIMGFgzP+iZboQZbpSypsAgaeUjvfilqIEI9C9bf8rk3Wd8kz59sHV+BQysa8DA+vNe1+Tr8VD5hgO+fA/3OLH82+nK/AiIK1ACEehfCceqU8fsy48ttS05xAkMLANgYP39N23K/1t26n8vFlv8/L619/+/azv+f11R9P90tu6fXc4suQyudRcKgP59DvTvG6B/S5Fd9F7baP07Nb3f+xzZv3wDav4/yff//3LB///bFP+/7nL4v9OF+R6DU/WZB5PXXPwP9O9LXNEJcva/s+v+I4P39RL/QeIMDhXHC+xKDz+wKdpfgMsAoH8ff5mb/P8/UNOPMob/74D4YQbzf6D4E6JS3+FwqYaTKaq/n5Ur/n9Vyfr/bgrj/72+LH92uDJXEZ2Ej4TLVgBtvA+NhccgzSBxAIJCoGJnoURSAAAAAElFTkSuQmCC',
chart_line_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJgSURBVDjLY/j//z8DJZgsTV+9fAu+uHo8+GzvXECWAV+c3R//mTn9/ydLu4eka3ZyY/ts63T3k4Xt+4/GlqS74JONY+9Hc5tdH4wsmAmGgWv9xQKX2nMPnapOF4D4zxotum4sjfh/e2Pr/wtz3f5fnKodx/A7O3P97/TU37+Sk9ajG+Bcc/bJnI0X/hfM3/t/YlfJ/ef7yv9/uLTl///PT/+/v7Tm/8Eun08Mv9NSf//buun/z9jYvz8iIs0Qms/YO1ae/GBfdvTNtMm5y99dqv338/Gm/88PdP//dnnd/z8Pjvw/PzfjH8PPxIT1P2Nifv8Ij7j0PSjk3jdf/5Mr4poWOZUfe2hXdsT+yXYVj3eXa/79/Xbk/7e7Of/fnsr9f31x8v/j05P+Ruf1tqI4Gehftv6UybvP+CZ9+mDr/OpVnPGjd5cr///9fvz/1zvx/78/iv7/6VrV//szHV6c7XUxQAlEoH8lHKtOHbMvP7bUtuQQ5/F8pbLnp8uhmuP+f38Y9f/jler/d6Y6fbrZY2YCT4mudRcKgP59DvTvG6B/S0GC25olg6/tKP7599uh/9/vJ///8Sj2/wegS25Pdf52o8dCASUpO1WfeTB5zcX/QP++BAnMyZRNvrqt6v/F7RP+PzkZ+v/r/ej/by9U/L81xfHljV4LJYy84FBxvMCu9PADm6L94Lhe1en8//+nB/9vzAr5v69K6v/RGZ7/DzaZPL3UYa5DVGZqTtX+/fP4lP8/T078f7LV8f+qFJnPOyv19InOjaG2gt/b4zV+7+3w/L2n1+ng5nJdJXypFQA6mcPFnqo5OgAAAABJRU5ErkJggg==',
chart_line_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ3SURBVDjLpVJdSJNhFD7bp7Js6qaUDqmxEFfRQFmZ00wjXYK2kgX+h114Ed5siZf9aIWFUSLRReVFgUElGVkURDCkKI3NlKIaRYmWmJmSue3b+9f5DGuBBNoHLwfe7zw/57yPSggB//NFLQcUKHG4BCEuESbt6uUQCDncqNm3x4gEbtVSR5jbuStGEPoaHSRibV7yCDxMWhH8HsHpCd6n7J8E9mPDLsGZmzN27tHJze2z23aUIbAcCTITfM+Y0qMiDQd7gNJSQdnd6MudZZEEhYd9Y5VbpFRZ9kJmlG/OdOGNdC0+58wNg03ijFZxTnGJhJZKjt1RuBCHXFmV9QfszccmbXf/9Lfc2MeTZkvBytFiw/h1Q/Z6xkhTuS3eyCh1qeQDdT0Kya/FUC3am7yjt769aCjMp4Lv7yzoyNeZHM26Ndnw7mHTjODcXnO/NpdzdggFzv71CkVHBmNKxp/cy5sY3Jo2MxKiejY7VZGwzlhUD0D8EAia4VP/+V7BuNNS84AoGHXEvCmMUc/tJOsXt7kuGdddPJsZbUqy1gKEfDBwtQu0uiDQULgUj2MBp7YfHXLhvONo5yWnpMdzylbd15YXHG3QrobtWao4fQC4/AHTw4H9eA6mgkYVleXjAx22uHkCVHXtzYhGdcI8p3PalMuhK/YYVDmhW5sBPDCM2CBYnWY09Rk0Gj8kWyo2UDnsnifgjLTf7P8+guqtC7aYHK5PTCuxxsZ9BUGn8LEl8N7yKzECHvLDqnQj9pCGvpZNGxeNMtobs1R3pUrqj0gwraQ/4q8apBVmmHj1Avy9Ld2LJhHtaXyXnEHBBdrnEUf8rqBUIVJ+AugPahHelS39AAAAAElFTkSuQmCC',
chart_line_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ/SURBVDjLY/j//z8DJZgsTV+9fAu+uHo8+GzvXECWAV+c3R//mTn9/ydLu4eka3ZyY/ts63T3k4Xt+4/GlqS74JONY+9Hc5tdH4wsmAmGgWv9xQKX2nMPnapOF4A1WzsEfjSzefLB0FwUHoi/szPX/05P/f0rOWk9ugHONWefzNl44X/B/L3/o7LXnn1h4fitN6i22Tx7W5tpxqYHxmnrChh+p6X+/rd10/+fsbF/f0REmiE0n7F3rDz5wb7s6Bu3gt3Vz80db69zTd1mlr11tUnGxt89Cw/8N0ha9YDhZ2LC+p8xMb9/hEdc+h4Ucu+br//JFXFNi5zKjz20KztiDzIMGFgzP+iZboQZbpSypsAgaeUjvfilqIEI9C9bV1LfvnnRaT96Ghr/zZs3739Xccn/S9aO3+cEh6nV1NTcBYm1tra+y8rKKrWzs+NG9q+EY8XR4+7RFdcmTpz4/8WLF//v3r37/+DBg//7S8v+V1dXH7p58+YPkNiRI0f+g9T4+Ph0MbjWXSgA+vc50L9vLEJq5hcXF384d/ny/4iIiG0+3t7/UlNT/0+aNOl/ZWXlfxDb29v7D0gOZAhILYNT9ZkHk9dc/A/070sjI6Oupqam7yAD/P39uezt7T/v3bv3/7lz5/4fOnTo/9q1a/+DxEByIANAahkcKo4X2JUefmBTtL9AV1c3Mz09/T3IgKCgoJ2Ojo6/3Nzc/gcGBoIxiA0SA8mBDACpRQlETSDw9fW93tvb+37VqlVv161b93/69On/CwsL/1dUVIDZIDGQHEgNSC2KAWpqasxAJxp4eXndT0lJ+ZKdnf0diH+Gh4fHAbEzkP0JJAaSA6kBqQUAGVqFjOjvxXwAAAAASUVORK5CYII=',
chart_organisation=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFOSURBVDjLtVK7SgNRED0b9iuM2lr4QK1DQIyk0FZsJAj+gH+ilRZb2NjaRHTLmA9QFKz9huzm7t37Hu+u7IJgQjR6YLjDzOXMmcMERIR5EE5qXA4z4sqACYWEC5wfLQXf/WtMIuDSoL0A7DZDjBj/uYI0l8jzEEJYJMkvCEZM4PqZIxlzpGk+kSCY18TGtGYcx9Tv96dOqBUMBgNyzsFaC621312Ac+59yJFlGRhj5VvVoigKvniglEK32w1mkd3r9ejPPAjOhqdknYX18p1/rzo3pYqTh0OSRkJI5UMgPn4s61sX66SkhtEGcISGsQad5gH2FvehfV5BaIF2cwet5RZyKeu68pe5ubKG7dUNP5AQGltMN57Mosgr5EIiVQmYGvtc1PVicqHY+dXpk8Dg7v22XKFo1ARe9v1bDOlXKKKCs4Sn1xdU1v3vIc2CD3bN4xJjfJWvAAAAAElFTkSuQmCC',
chart_organisation_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG5SURBVDjLpdHNa9MAGMfx/BtBvO/i26XBw0DEocLUSift2Lp2LupYFh2CVLA6rIMVmqaiqxDZxaJQNehSspCksdYXRNGJzOKmNz0IDpRSvH+9SBVEaNgDD8/hgQ8Pv0cAhM30fxfl5k8KfpvZ2gYz1S+EBgpem1etNk9XfpBeXA8PXFz6RvP1d9xnGxwtvg0PqLc/kzLWiGor7L30PDyw6RABwXEcLMuiJ6DRaBAEAZ7nYds2pmlSqVQwDANd18nlcmQyGVRVRZZl/gFc16XXs5PJJKEzOLMwnD29kOic1I8wPLenc/D89iwgCDNPJlAfp5l6NMZkfaQrp5aHSFiHiN7bT8I4wOX749itMu+++pTqU8RL29hxbivCdCOF9cnk4ce7TLjxLhBfGuTGGx3t5RVG8/uw3l/F+nANAC04QSk4RWRaRJisj/JgvYq5dofU8lAXOFwdIP9ilmzzLIMXduG0Fvm7aqtlJEVEkN0E484xxuwYI7VoFxi41U//zQiR6zvZrW6h4B9n3k8DMO+l/1zQS4CSIs7FtD6KvkxttUzRl4lpfUiKqPX8BUkR85IidiRF5PfMA8IvzWTWMhb2/CMAAAAASUVORK5CYII=',
chart_organisation_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHFSURBVDjLpZI/aFNRFMa/m/eI0eBUBNuYhgrqICg4l0Jra0AFp2C7BMFBdBKngiCdxM2tHSIUwbUIkRpIh1jUVYoUB8HiUEuLUPL+3L/vnnvdooIpqT3jOYff9/Gdw7z3OEqF/QaL69wLY5Eqg0goPL9dZv/ay/UDCG0xMQJcLYXopuLwDmKpIWUIpQhR9B+Abqqw/EkgSgTiWPYFsKOGmDto2Gq1fLPZPFCh56DT6XjnHIgIWZZBKQUhBKSU4JwjTVNwznu9RqPB/srAGINqtcoGsV2v1/2hM/g4V5m3vHvfclGmwG8DWLzWts/Yw/W7nhyBnINzhKXpVwwA7ryteW01lDa48mUPt5KCOT9Vyx8buwj5uY3N92t4ke0iZ8liunQTM2duICPbU1SZwkRpEuOj47iw8RXnpmr5wrd3YC/ncGLrNc5WhjC5EyO0RCBn4TzBEvUAUmnEJkJqElyODQqnx4Drj34/0MIwTglCaMnizdYKyDmQ+wOgFVY3W9Da4FIxgNhYRbH5AFruQgBI4gD7x4PBQvwwW1rInyw+Lg/ZMMxtI/lp8X0voEz5J4NfYXZ0nu/v3AuIVSjwPzywVG3bp78AeAkDORpY/RgAAAAASUVORK5CYII=',
chart_pie=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMoSURBVDjLndPLb5RVHMbx5515pxfnnaadQi9MSykzRSFSqyWjITVoUIMhLmClLky6YcempKBGgitDJEgaIlVjMGkkLkCqIUTjyo7YQs00dYYZpsNoufReaWn7nvM757yX4wJJJBoXfv+AT57NY2it8bBT2fct6aoeodQeoSgplISQYpSE+i6onv2gWr9e/tEbMY6/ZTwETmaO7ZKO+uKZ6q7WoFkBx/BQV7keN6fyuJj7qj9mfJJVjturlNf9+YH40CPAiV+P7tsYSlysDW/AtLqHcTuPoA5gp/U0zl39bKnS3ZeMGC+NJhNWNHdrFbdn7f3nD7cPAkDgw/GjUaHEQJ21EWN2Ean7I7jvrCBR2YL5ubtgjN4L692HttRVROtrKtDWaIFIDbzy9nAUAAJcUk9n9S6rRFPI8wlwV2B9MApLhPBN5sJ4LHj6miDnQKI5jMKMQLSqHG1NEUtw6vkLYHuDoXJk7QKUK1EVsNBe9QRGiz+D1sRBR5p9HZsjQeX4mLqnUJyTaKgNQ5DYCwAmJ7FNGi4CMNBhPYlN5THc+b2EdCl9tjUysIFIdsZqKzFS5ODMA1v1sDUWghTUCgAmI071FevKuiI7MD9zF/1jJ5ckU33Hll87M7GSNn8IP15aWBbbTRjgzIf2fUhlQpEIPljAafKXG8Mdl64PLkqSxw/PNp3fvRR+S/PcxPM8/cKlbb0Q0gPnGsQ81NaEML1gQ0kqPQAYfflt5uv+U1Ntl7esBHs0p7yzudkyir/BX7NBRODCA3ENYbtojj+G4aslOJIuA4A5WOo4qzkd15xOO/GWMifeAt/zYI5lAcYguYSQHoiAzu0RFCbnkcllbM9RfQBgapuZPuNvqp3JMremGuJGHqHGJvg2g2YEKThIeEjEwigUp1HM3YQrRffs0JFFAAiEPj6z6K+xbuNaGsgVEGpohE8cHhE8ElAksMocXEll8FMqNTkzd+vV2aEjF/7xhbWuF1/WQnyq4pta3fp1CPw4Ar3wR/tzO9455ylJrqu+91x1Yj71rv2vZwKA5a1PWZ5UvVqpPb5yktp12xuWZrL4jx4B/k9/AolT0+iTfsOYAAAAAElFTkSuQmCC',
chart_pie_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANhSURBVDjLfdPfT5V1AMfx93N+8GPnOYSHhMM5KCEcE5eA0ZAxGm2Uc3O12VV10cZFXhTe4HBu5eyiNVaRcxY012xjtS5QsezHpnPNMlQaRNAhwKNUIiLE4eA5z/P9Pt9znufpwtz0pvcf8Nrn5qO5rsv9jky+rVs51SWV2iWVaJLKQlpyREj1vVfteLfYfSn/w5ejJg+k3Qd6Jw63WVn12ZPFrVVeXwFZzaa0cD3X5qc4Hf+yP6p9MqmyuW6l7I5P91ZffAh4/7dDezb6a06XBCLcUiuMZ6bwuh5a9O18ceV4sjC3pymoPTvSVKOH4n/e5a/bmRcHD9QNAXjeGz8UkkoOlOobGcvM8mPqMqnsGjWFldxZvIlhiLcCbvv+zaUFobJ1BcTKdYRQAzsPDocAPKYluhqL2/SEmGfKnMHMSdZ7Q+jSz5mJk+NR77GrUmT31mwIML0gCRXlE6sI6tIUXf8Bxm6vP5/JzDQqZ1Hk0akr2sLI7M+ItNyXtXxHGzYFvSrrML+imF20CJcEkELuBvCZQm61tBweNBr0J3gsP8rfNxKMJkZPVAUHIkJYjdGSQi7PmpiGjXHXpjbqx5KiCsBnCFOUFTya1xp8ijsLN+kf601ahjp6ePX5vpm1Ud+5wOOJpVW5zYeGaTi4joOlfCghvfcWmGLulz+GG87+PrRsCavnwO2KwfZk4FXXjM88bY4+c3ZrN9KyMU0XYdiUrPNzaymDskTiHmCIz7+aONV/ZD727eY1b5driqnspg26NnsdJ51BCIEpbXLiAuGiiyiZYtkQlEUCa/AavqFEwwnXFD2uKY5lqyvzstWVOLaNb2wSDAPLtFhNfUc4PEJjbRMVoRg/xM/g6j+1NXdGPvC4GcPnpDOvWDsa81SsCjEzjWMInIyBbQgsaZJMfU3Dlnpsj019+XPYWpbmbS0Ab3j8H/ctO2mjQ7s6CvFp/OFyHGFiC4EtJEpIkulF/JrOC7X7ANjffpzq0jqAAg9A4anBk+7MjZ3eb87PceESTmIO25TY0sJSgn9SK058YZie8x0A9Jzr4PrSBIDUHnzjam29bluq21Vql6OyTW4uVxdOLkw2d0beqah65M2WulZi4e1cW/yV4YlLzM+t9T4E/F/NnZEe4HUgCKSBvisfLRz8F8J11bR5XdMKAAAAAElFTkSuQmCC',
chart_pie_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANpSURBVDjLfdPdb1N1AMbx5/ScvtHTZbSwDcrWlXXIUOdkZiRmioQXSYgX82p4YUKMkJjswpkRjCJckUWDhBCZGoJhYrwYMIUQzbhReZOZLtvqateVgdKVjYV2W3t+b+fl5wWSwI3fP+CT5+ZRpJR43LHkYZ1bopsJsZMJ2sYEB+NsmDLxkyo2HamUnd7Pd0cInkh5DBwdP7SZm+KbjZXtMVXzwVRsVPlXYiqXwoWJ7/siypdJYVo9Qth7Tu1t+PUp4LOxgx117viFcGA1ZsRDjJZTUKULL+sv4rvfvy74rY62oLJtuC2uhybuLuHv++U3B/Y3DwKA69PRgyEmWH+VXoeRcga/LdzEgrmIuD+Kudl7MAz6cUBu/WBdlS9UvdyHxlU6KBX9Ow7cCAGAi3Da3Vq5Wc/SHFJkEsRiWKmGoDM3fhg/NxpRT9xi1Nwbrw0gnWcIVXjRuCaoM0K7/wOMXarbi2Q5DWFxVLh0NFesx3DmOmiJdZlcO96yNqgK00HuoUBmlqMmHACjbBcAaISyDVyx4IKCFv051Hsj+Gc6i0Q2cToW7F9NKW+NhP24mSEghg1jyUZTxA3OaAwANIMSWu1b4WkPvoS5/D30jRwtcEMcP1R84+TkYkIbCjyTfVBkz2tQQAwH0nHAhQZBmfpoAaF3/vjrRsulPwfnOeW9+++vGdhaCLwtycTkKyTx2qUNPWDcBiES1LARXu7GzIMyBKfZR4BBz/44fr7vWK7x8rpFtVsSmjLX1upK5jacUhmUUhBmIz51FtunzmDZwh0s+MIImdEi8C60wWzLaUloryT0hNkQ9ZgNUTi2DW0kCRgGOOGIpL7FRvtnNL2zD97Ys6DjQ4hevfLqlW3uLpcsG5pTKr/FN7V6RGMMdDINx6BwygZsg4IzgvDYKazf0gHf7V+gnNmNZdODqI+GVanI913uL07OOyVjj3IrAUyk4a5ZBYcS2JTCpgyCMgRLM/DVxICui0BvAfhwGpqRhuoo9RoA+M8PnCu1b9mhjqW+shvqY071CoAwSMbBBUVRDQgyetkTuPgeOJ0FAVBaUmGryCtPvrHY9IJuc9EjhdjpCLNNWlZzTSGfvNYZOewJBj6qDVua5sqhNG/h7pxqm0x+8hTwf13vrDtgFPL7VFuJ2qqckUDf60PWkX8BnoTdhvAVaYUAAAAASUVORK5CYII=',
chart_pie_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANsSURBVDjLdZNrTNNnGMWb+M3wRaObc/plSxYToiZzc94quCGCVRfMnwYtUiyuUhUQKqtcCmhatAgEaKlcBNFSBYQ5lSIgQ0GFttRCL0BBoAVUVFKo/UtLBXJsiZp5+3CS98N7fjnnyfNQAFDeK1uf6nVGm5CSquS28VqPzMY0RcweVjDawmqC+QevZi6IvfJk4f//e/ThkalL8RFqTg7dHqhFo6UJiuEGdLzU4oq2HISMJo0pH+VwLpqHIgoHfD4DZHQlB1V0l+GOpRFl/VdxXMsH91Eqavr+xd5LO62MkuIfI0vN1tLWcXAvD4IQ6YI+AESdyYtPq0+QzcPNEBklYKmjEa6KxvmeUkhbxNgh3cZhXxiSZteOQWEgUXDnBWhpHeR23sPF8wB3X4Gi/xaKTJfBVEchpI2NeE0aZFoZ/MU+naxC489h4r7Zmzo7shrGUaWy4fgFE6hRTYJ5QHxLZGe9uRFRmkTsc5vZyjjI+isQVREJavpvWw7kme5nK56hWmODpPIaTPIQPL4hRFeJP3T53mGUo/XhrhuWOsRokiDS56Gyrwbn6kXYJPi1hJHbS3f3dVQqJ1FcXYaxZh5s+lqAfIpJfTXaMwOeU8Kv023K52pc67sOyd08+GZtsm48/UtKfeypJbnx5cvcffU1dXKMG9PgGr2JsXvn4DD8g1nLAxgusp0Uunx3p/hujqfvS5+MDXGKWGLlNJOZ5AymW6doe1bzMnLMViMfc44HcAweg9U9p15ZBJTSgzPqvKCfKLuK/Lh+uVS2IZ71vYv9V9Z0aChJpiTjdcg+jGZ6cyYMCZhztmNqgAnnCAP2nkTo82kgGAnF80Oc+fvEojfHjha6WCzXa6EAkxUyWOVlGGRuwVgH7505DM7h/XhlTEK3JBB+BH/qO9+MpfOAN0c4S92RSXthPiaq5Hh2Kgn94mj0KuLcsVvhNEdgeuQAbO4kPZIA+IcWYNnWs8RHm+jYSxAki4WJVD406Wx01yVCdzsHT1TBmDIzYO06iUc5NKzfnTbyLTU94Iu3YN/su/3Vug1DVaI/ALsFpiICzYnL8bAgELX8za4/6dzz31CFXl89Jo8mVq3xEhzynnO1S+BS5UIl3IaqQyvIhoQ1az81fhHgUTB1kfMMc9XMf2cDZ5qyfm+5xVv9w9fMHr0Fh4yy26byoRwAAAAASUVORK5CYII=',
chart_pie_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANvSURBVDjLVZNbbFN1HMe/p+d0be3p6Lr7unWMbuCQbR3FQWAEyPASkYe5mHhJTIhhPPngDAQTEGOMIZBpFiKoIWiIhAdgYxI14cFEo1M6O+Zmx1ob2MbYhbEb6zn/y7n8fcCZ7ZP88nv7JN+HjySEwDKfDX6oMpO3Uc5fpJw0UM5AGY0Ryn+U+dZP/OI116evB3WsQFoWtA+c2MUM/vVmf2OFrLhhSBYKPPn4Z3wInYnL54LSF4PcMA9zbh043xr+eZXg9F/Hm0POys5cbwke8Fn0Z4YgCwe2q/W49MdXcx6zucEn7Y01VKqBxMhjjE5mXrlypLYLAByn+o8HKKcXC9QQ+jIp/LLwOxaMRVR6yjE9dR+aRo55RdN76wvcgcIcN6qKVRDCLz5/tCcAAA6dkbaof5eaJuMY0pPQTYp8OQCVOnF94Gp/UD5zixKjtbLMi+EJikC2C1WlPpXqpO0/gbZPdrowmBkGNxmyHSpqs59GLPUbyBJ9x2BKR2SdT+aGjfFZjtQUQ1GuF5TQfU8EhG5kkgkHJETUTWjK2YGpu6OIp+MXQt5vSghh0Rp/EsFHpwC6iLFJBpeigFFSAQCKRnRS6M7LavRtwfTEfZzra59jGu84Mb//bHIxrvy6Zu3dAq37mdLwDtTP/oSbCy+BcQWcUBkAFF0n93rv9ERu/N01wwg7eWSy9ErTnPctoSeSO/X4buf+7XklZVuQXVyPGtcHSPvq8OBhITgj6ScTNPJt98C1Q+1j4fqb6Zpg04x7yCgv+9iS5cBCnRXelns7z5fnh/X4O5TXv4md6g8YSY3BYOR7AFC60pELQicnhU7OGOHyLCNcDtuyoNwexFIV3g7VviqD9iF2qRvbDr6PPNIL76JJTSO/AwAcIqMp9lLmDbY1msWrKkCSw7A1At2vwxWKvOzL0WGze4CwYWV6UbG7DYdqYrPXmzspADicn5+dsZe0A9KtOJAYhrOoGBbJYG6TCX8oAlsfgLAJoi0bYPMJuN0prNvcEjQZf3dVC0uNe54TlH7Jw2srFqslOJueRdF6D6xMDEJwxK/dQbSlGpK8BvC8gD/PH3sobHuPtLLG+eo61WL88Ghr9tGNBy9nyY4RCHMegFjRnwOyZwOmE/1I3fjo6irBMj2no4+EZT8lbAFh2ytO/P9h2xBCxP8FbMDM8CUkkoMAAAAASUVORK5CYII=',
chart_pie_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAOPSURBVDjLVdPra1t1AMbx51zS03iSNk16y0m2tF27bp0d3SodjLjqJqNl+qLDF3WDjTEsKIhQ3fCFY74Sq3ZjyFZRcVCUCd1WpYgiglXYZiudtV1vMXhJmrTpbW1zfud3yeX4Yg627x/w4XnzSLZt42EXJ99x8azoZkK0MUFbmOBgnI1SJr5TxL53PXanduGlgIVHkh4CvRPnW3lGXN3rCVcraiEyUg7lzjL8OT+Nm1PX+gLSx5Mikz0jRO7UZ13bfn4M+OCPcx1bHbU3fbqBhFjFuDkNxZax37UHX/76yZoz29Hilp4bbal1eaf+2cS/C+bRgbO7BwFAfn/8nJcJ1l/u2oq7ZgS/rN/BemYDtc4QUotxEELf1u1Db2wvL/RWlBSizu8CpaL/8Fu3vQAgW5x2N3taXVE6j2lrDlaWoUzxwsUc+Hri+nhA+WiE0UxX7RYds0kGb5GGuqDbxSza/T9AjigODZPmLESWo0h2YXfRDoxGboGm2WsZrl5qqnErIpPH/KpAZJGj0qeDUXYEAFSLsgYuZSFDQpPrSVRpAcT+imIsOvZ5tbvfoJQ3B3xO3IlYsEgOZDOHnQEHOKPVAKASatGKwtKCsPsppJJx9N3tXeNEXDp//4Urcxtj6g96fXTpPmtUIcEiedj5PLhQIShTHiyw6N+/zdxuGro3uMwpf+/sQnDg0Jp+wram5p62xp4ZajgDxnOwLBuU5OArcSCxZEJwGn0AEPrFNxM3+i7O1327fUPppsSa+dEo02fdRfAGgxN7oz9lNpYdYGk3augQdvjLEU8uY5dnfbW1tVVXpnYenT2e8p3wbWavWqFQ+CutoGAluAXHTp6EYRioLPcqsZkRSCSG08efx7aaGpSVlsAppWsSiYRDtU2i5ol1TOxvKbi1sop43saLBw/i01dexYjgeb9hyI2NjeBmEhd6P0QyuQDHE97f33y9a08sFuuSHZevLOfT5JQ0MoZ74xPQiz0AgNMLSzBNk3V2diIcDqO9vR1tbW0wTZPaYiMMAMXFxZoKAM4bA9fT4WcPe0n62lSJ1wcAlz1uyJwV9PT0QNd1AAAhBLIsqYqiDAJAIpFg0qNv7GjY1axXVX2/78CBkmBlpYLiIqRSKUQiEWiahlAohIqKCmSz2bV4PC4PDw8vPgbU19crfr+/Udf1QcMwyjRNUwDIKysrLwNIlJaWDgJwcM5zyWRymRDS8R+TYbuVB9P4ZQAAAABJRU5ErkJggg==',
clock=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMESURBVDjLXZNrSFNxGMYPgQQRfYv6EgR9kCgKohtFgRAVQUHQh24GQReqhViWlVYbZJlZmZmombfVpJXTdHa3reM8uszmWpqnmQuX5drmLsdjenR7ev9DR3Xgd3h43+d5/pw/HA4AN9zITSPUhJ14R0xn87+h2ZzJvZVInJpzAQOXQOQMt+/5rvhMCLXv9Vjrt1rSXitmwj+Jua1+Ox+2HfGNdGf6yW8l5sUKPNVcRsiaPDA22Ahv6/7Ae/0aKdviQ0G7B/c6f8Zg+gbfh079Mjno0MhS58lflOsgEjh3BXc+bM/0DzbvDwj314znt/bjof0HdPw3FBq6kP+oCxVNfdDZvqPsrQmf6zdFRtyPJgbrFoqUTeS+FnPrekpmiC2lS+QcUx+qrf0wmFzodYfgC0nwhoYh9oegfdmLsmYXHj7JhV23erS7ZNYHyibGLiLtXsO19BoHSiwu6Ok09gwFg/gy8BO/STOkKFBk7EWh2YkLeh5Hy4Ws2B2w157iDvOpxw4UPRPRTSfL41FIsow7ZeXwUFF4dBQ1L96A/xLEFf1HMC/LxAt25PH+VN0HXH1gh2dEwdBoBGO0OKvW4L7hCdIvavBSsMIRVHCi0ArmZZl4wbYrz/yHSq1Ql9vQLylUEoE7GMal3OuxMG/7CO848N6n4HheK5iXZeIFmy88Nu+8aYJG24G3ziB+0Ee7wwqemlvQ5w9hcAJwyUDtpwBOFLeBeVkmXpB0qlK9RV2HlLsCsvUivHRhQwoQjhCkA1TgJX1OK0JVzIN5WSZesPZ44XKia+P5BqSS4aq+BzZXABLdhyQrsJPOqv4MVcEbMA/zsky8gLHyYO7hI9laecOZWuzLfYXU2zzSblmQerMZqjwTknOeY9dlIw5kVcrMG/8XpoQgCEkOhwNNJn5i7bFSrFDpsCrFEIPpLacr0WxpibYIQpS86/8pMBqNswnJ6XSivqHBv3R3pmbxzgwz4Z+EaTXtwqIogrzjxIJ4QVVV1UyihxgjFv3/K09Bu/lEkBgg5rLZH+fT5dvfn7iFAAAAAElFTkSuQmCC',
clock_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMvSURBVDjLXZNtSFNRGMcvgQQRfYsoCD/0waI3ikqzIoTojV6IgiwrtBejzGRlVGi1QaHZamZpaaalNnPWyjmNlrZ11dnyZc6luWauXNmWm267zvT68u+cSy3rwO/e5z7n+f/Puc+9hwHADFQwkwhigonwjjCZ5idCcjN+zxsIIX/yTL+SCSKkDTREfeVdWngb9jkNqq21iVW8juD+je6tagfrMx5zDbYnu0m9gTBbMHAWM0lew/5vw44K9NbH9DcqwrnUWhduNzhxv+W7AI1vsF1oUSz1e8wSP9dy+gfRNRGCGPsD5oLPlOx21MT064vCRzLqu1Fi6oGc/YwsZRsyStvwoLoLcuNX5L3R4oNqw9igvXTU8XyehWhDmE/ZzJqOnCmWutxF/jRtF4oN3VBqbei0e+Hycuj1DsDS7UWhphN5NTaUlElhkocNtedMaybaEKERiffLr519akZOrQ0KshodfR4PPn77jp8kpnDjwF11J7J0VlxUsDier78i9IBeorKbdGeemHH3hQXtZGX/yDg4vx+ZeflwEiPf0BCevnwN9qMHKYpW0FqqCRjsSmfdInkzrj4ywTnIo29oDMNk4rxYgiJlGc5ekkCjN8Ds4XEqywBaSzUBg20pL9yxuQaI843o5nhiMga7x4fL0uuCmDW2oncEaHTxiE+vB62lmoDBxotPdJEyLSSFTXhj9aCHvLTdx6NSV4cutxeOUaBEr4ToTjRib2zHnivrsSkp1hYwiDjzULxZ/BwJ9/RIVVjQSxrWxwO+MQKJS+seQ1IaA/X7TLT0aCB7dRy7ZQsRGjdTKhisis9aRmhbf6EcomwWVxUdMNr6wZF+cH4eMWmbUGZOR1l7hvCFpFVHIKs6Sg1+Bv71FUekR4+lFvrXnXuGA9JXEN1ikXizFiJZDbYkLUHl+1xMHKrWTGrw97Do9foIs9mMai07uiouF8tPyBGaoBRYeSIY1zQHkaKJFsQpL6P/3YFarZ5O4KxWK1Tl5e7Fe5MlCyKTdAQ3ZemhiC87pXNxXXNYWJne6XOgBwUFBVMJHYRhwvz/jzKFFKeGxc0aoNtefTJ4fG3CnFs0/wsC49wP713enQAAAABJRU5ErkJggg==',
clock_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANKSURBVDjLXZN7TFNnGMaPg2aJWUyMMfMf/9A/hkbjksVtRmMy75ekJCZL1DldzNwmc8TgEHSitEYFEaeiNsK4OHB1FihyE0Gw9VALFbGUBkYpFKSIrW3p5XCwHNo+e8+Ja8y+5Hfy5vne5/nO9+YcBgAz1ch8QCgIC/GM+FDU34e0j9/tm4ik/3TGr2VkRN5U175xwatDsOtbt6lObkhvFfSE7x36zrpdbMic4p3uz/JRv4lYLAW47zKngqb9r2ZcjfB0HPQ/16zlcg1e3Ohyo6TntYRY/8460KP5jA9YlTzX8+sb8nUTMsZ5mzkdsmT5XO0H/cY7a2cLOsZwzzIBNTsKlbYPBZV9uN3mgNo8jtInOvxTty067ayMuO4vt5E3iRkuZNYPFM21PS1exefpHLhrGoNWN4IhZxDeIAdPcAq2sSAqWoZQ2j6Ce7X5sKjXhPuL5r0gb5I0iPSS+kuZ1VYUGUagodPENRkIYPDVa7ylWoSLAbcahqDS23FGw+LnMuN5aQbiY19ht/54lRW3mmzop5P52Rg4nsfN0jK4KSgUDqO6+THYwQByNL0Qe0VPPODrq6wvTf0CF/+ywD0tYDIcxQxtnFQocUdbi8xsJVqMJlgDAo6pTBB7RU88IDmnyfdjsQmKMjPGOIFConAGQjiXf1kys+ZeeGaB514BqVc7IPaKnnjA9jNV+j1XdFBWdOOJPYAJurQzJOCB/ikcviBcEaC38iYe71+G1o0yaOWLcP67XY54wIbjfyp2Ku7j6B9G5Gps8NDAJgUgFCWotteo0JuxHm8bLyPW/xD838fw7MiqWMumxFQpYF2qajXRt/V0PdIKWVzUDMA84gdH8+B4Aa17l2KazCiQA5nzgQtL4Mn7Cs2bExzxb/2LH/J/Ssmt4DefqMGB/EdIu84i/ZoBaVfapdeOdWvx/vJnLyI9MRYPMBqNG6xWK9p0bGTdkWJ8/osaXx7VSlTuWIipku8BMoUzGEwSLw8noHlLwrhkbmhoWEhwdrsddfX1vk+/yVKu3HNKT/hEzu7eNNp56JPoROYSvDkpw/ChOWiTJ0Yebkn4TQooLy//iBggZogV//+VRQy7F59o3ipzitdpSl4QeZC8IEfU/wVxe9rJJAH4hQAAAABJRU5ErkJggg==',
clock_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANZSURBVDjLVZN7TFNnGIdPTMwSY/aHyTJjYkJmIjNT/zCLMS4uoomKRhMTTcjwMoJCWDTGeUFAtmIkoKIoA4QqoJRLrKyzLQ63qS2ltDtcpOXYQi1SbMFSodALp1fbn+9pgGwneU6+vN/7e75LzmEAMHNPmSWEiDASPcRnQv2/UO3L+XmWSF6oM7MyZilxfa43fTwyrYK394iLVezXnn8eURPuedT/Kg5qfAM50wHzZTf1s8TqhMDVyhR42aMT4cmnmNJnzPZJt/pLtdOo7HWhzuBMIIxvaUZhkG7iPVwR7zec+0C5fmIp43jAFPqMl92TXRmzuqat0Qq9HY+M79GiGUO1zISKxyY8eDGKloFx1HeqMKTYHQs4Hn+cfLLOQtlk5m0ts21YvMzSfX8jf101ilbWDpnKhhGHF9NeP6a8c7DYvZD8PYL6LhseyctgbNkSMos/f0XZ5MRFnK9T3sj9nYNYa4OUVhOeGY8HbyacCNJYwB8H/lA2Y7g5DVZ5MQx1u2Cs+uZYQpBe26++0MahpsMCM63MR+Pw8zyq6hvgIpEvFIL6WQmcL3PhGWwn2wRmB9ugL9szmRAcuq1xn215hWvNRrgCEcyEYgjTRJ6oCE0yOe7dyYDbJELYoYBTfQMBToaPY1pwDVnBhOBASYc76z4LUcMA7P4ISWJweHy4WnYT1eXZmHldiFhAi8DbU3D3nMaQJBOsODPeU3FwbUKw55c2dVq5CkWSfnRaPXhPh3b4Iuj8pxozXB5iQT34keMI2tPhM+eDq9mH7CuVfYnvQHilXHgo2it6gjP3dCiVWjBFF2YdlMPZmzsfPobgux/gfV0Ac1Uqhe9CyCwKvjtd/S1h2lWoxNlaDWrqxDD/dZG2rUHQlomQ/Sg8tJOhqt1I+7UJQq+QWRQIbD5Zlp1TKuGzfr4I05/5MHbcxjh7GLwtHW7DJQz8tg8/lUhwovghL/Qu/gsLA51Ol8JxHKTXdgK+MQyLD+Fl/ip016RCKdqOjLxydGm74906XZx6v/+foL29/QvCb7VacTVrA8L6SoTZO2CLU9CcmRTf/2NO3/q0ApFCqfRZLBZQb5RYsyhobGxcTgwT4cPbVoRKjn8dfVGaGn1+c0enMnfDVwur0XwS4SEmiJVC7RNi1+PWy99nOwAAAABJRU5ErkJggg==',
clock_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANLSURBVDjLVZBrTFN3GMZPlpgli/Gb8ZNftkRE42VsWYxuJsZFzZKZmPjBDYzhw7ZothgciMplRWe4WMGQrRNEwMJA24rSFjBLa8uhVDqGba1QaksrlFvphV44pxekz95TtdGT/M755/2/z/Oe92EAMCt9zAeEiLAS/xIfCvV3odqmN/cmIudtnVnuYdYRdSuj+bOpgA6R0QKfSfmtoViT0hPBN+hHlMfYqPl0gJ8oD1K/idicMfB1M2UR08m55GIf/E8Kl/+T7Y3VGAL4Y9SH25aFDMK5nnXDIsvjwrYqLmb5dYl0Y8Q6xtvOVESt5cHFocJlY+fe1cYnM7hnnUcX+xKSnnE0ysfRrnWjyzyL1kEd7MrDa7xX/mrxYa6DtDnMVBPz1WTzR47hlp1cnc6NbtMMenQeuLwRBCIx+CMrcMxE0PGPC61DHtzrFcPatScx0bzhKWlzMkEU31ZdK71vQ7PBAxlNE55QOIwXcwuI01kglgZuql2Q6J2olLE402a8mslAeOU3jelLFDbcHHBggiZzq2nEOA5/trbBR0aBuWHYNEUw2L2olj2D0CtosgbHb7DBoq6nqP3bCh+fQiixhiRdXBRVofuBAmO9pxB0ymG3tOOcxAShV9BkDY5WDwR/bDFB1GbGTCxFJmvwhqP4XXwdd5t/ht/eidUVFyZUBbgi6YHQK2iyBkcqFfoTDTpUdYxh0BnGPC3tjaageayEnUQJvxqJ+XqEpvrQLz2L7xu0EDRZgwMld0TfiB7i7C0jamQO+CmwUDINz8h1BBwKJLyXwdZ8hlRQAXPrQZTUiiFosgb7fpF8TowfqlChqIlFrWwSFrMeHm0ZkktS8K5CsNWfIj5dirC7H/0NhxPfFV/enzUQ+OIH8U+nazq4ry88QKH4ETQtBVim4OIvi8G9OPEa50la5QamBxsxXJdX+Z6B0Wg8YLPZoNWxry5UlmCarUdy8a/MdM6ZTyvkZb68+wz4hV6wV3b4Bqu2b8uI1Wr1RiLmdDqhVKmC+vovo3xgCsmQFoklymBJ/g73sRp7jtmRTjwu2yLPGEil0vXEJJEkttPvBQzVu/mhq7t4msTTJF7/Wy6vq9ga15ZvS2su5aa1pZ/ENec/HvofoDruXRuQqRYAAAAASUVORK5CYII=',
clock_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANRSURBVDjLXZNPTFt1AMc/77WltH0F5I9U1sqgc0qFxYNGlODiEj1sh0kC0WkUD2qiNxM9mCVc3IhGd0E9mOjiZdFkzJg5iI5ljARoNPvDwsZgDLvRCaxA4bV97/X9/Xkxhvi5f7/J55t8JSEE2qgkAwNAD2AC3ZGDwmQH2qjUCJwHysBbkYNiAUDa+pkAcDyQeOP1ipZ3dhmZ73Pzq+ri6fAZB+j4Nz/bq78qpxKxlD9YJ5nzx+4AvTU9Iis9+JGjla1vflDZ/FpTITO8fS+74L/QdFZRgi6hgAeAYctYhsaBtV59d9thfG5B02+dyAKdsmUiS5XNlVuLw9uLSwvKdOInpSVqUWsYmH/l0RbzKAWNWLSCq40nwquZ6UpfXWetCLUplkmrtPQt3bYU/m5T3hOfSv4SToT9BHWXfXtqqamSEUjkVZc/59exQz4i+TM8oQ2bgdLcXFAUjkhCCD4+ee4LX83uj5KNCjWmoO9AC1uqyoZmkGiKAeAIODW6hKcI1nKrbGj+wW/efu6oDLDqND3jCfCKFu1769AdQSAQYOz33yioKrZpcn5snNTeetScgSE/hGoFuwD8AKahdTiWhGp61NWFMF1BJBzm/vIyYxcnmL0+g/HYBYyVEooxBDURTEPrAJABLEND020My8N0wfYgpxaJKAqz12c49EoPlnCI17exIr9LSdewDI2dBbOFoo7k98is6biSzJcjh8g1nkZN/soP1/qpV+K0xZ4lUfc4d4t9mOX1uZ0Klwqyf39eCZK+9oBUshrHs3jpyX5c4eF6Lh6CFTVLR7yLkmVwpfxe175j/dV+AKusnbPKWt8t15eS4w4nhwVlx8IVHvc2F7E9B8ezsV2bglnkqUQ3JVuX/shMbsoAk0PvX7bL+lAysq3PZja5uZyl7JRxXIfGqkeJVTXzSHUrAV+IhugurmanmFqaXNctnpaEEACk0+kXo9HoxdzGljswfNv3cPsgZcvC8izKjkWyPkVn68tcXp5m4s54Xrfs7pVBMecHGBkZaQDONjQ0oJVUtbR192t14sj+nWe6kvrseUkOBW7cv8En7V9Vx0Jx878R8/m8AfydTqdbgBdmTn16k//ROnBcG799KfBhy+fFCiNczBt5DeAfIAavfKIsB2UAAAAASUVORK5CYII=',
clock_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANTSURBVDjLTZJtTFNXGMdvlpgl27Jvi5/2meyLWZbFGBdBBqJpi0njEtzEJUZa7LBGrARwTkvZFFpqodmKMC0CCrG1jSJqfCktl65YkUKtIFVKl6KzlVZa2vJShP/OOXHNbvK7ee5znv/v3HtyOQBc6hb3AUFJ8BIeET6k/f9Dehvfr7sJOf/1uXkrt4GgTo3se5mJ2pEYKY24+4qdxx9kHITYexwP+8T8wpgsujh5Mkbm3YTPmSDSy/2ccO9/tRK+hbnhA/OPTVuTDc4ofh+J4OL4awatz/EzGDd9lY776tLJccUbkhslbOBmL3G/LHhPxsJDB+Zdl7eu6odDuOr9Bz383zBYJ6A3T+CSbQY9Yy9hHLTjWd/OtcVZ87vw9S/8JJvDBdq4bVPtH/n/urAprbbPoNcdgtUexPRsAtFEEnOJFPyhBLrvTcM4FMTVG03w9mxZnmz/1EOyOewgjl+8qam2+NDuDMJEdqPX23gcz1+9xhKpKcl14Hz/NAyOFzhl4vFTh+s3dgb0tq9t1FF1zYfzd/yYJDunV9eRTKfxh7EDESJaWF6G5e4A+OdxnDU9AZ2lmazgu2Y+VtnjQeMVLyKLGbxdXsMKWahV1uGy9QaqT9fhnssNXzyDYwY36CzNZAW7z96JSS+4oewYQyiZIZI1zMYX8GuTloX5sSeYWwUeRzOQNw+DztJMVrDr1DXHXp0ddd2juP80DMvt+9CQsNFoREOjGqOBIJ6GE6hv0LCe6kwjflA0BnJzcz9mgvyqTqVAeR1H2oZwqFqLlpYWhMNhBAIBDA4OQkeeDe1/YnLKz3pOp5PNiEQiNRN8Izd8TZjYLtVAUVUDj8+HE6frUVxcDIlEAr1ej9raWlYLRSKI9x9KUolCoYhn//XNkqZy4V5JRqmqZ4KjzTbkF+yAzWaDx+MBz/OwWCzI/7YQWw6ekVOBSqVaygpcLle+TqfDYbmcCQTfy8hwAYqKiiAWixm0LiwsRGlp6SMqKC8vn2fh/v7+zwjJgYEBuvhOrWla6r7Su2q1WtHa2orKykrU1NSwmvbMZjO0Wm2CfOIzJujq6vqEMNXZ2bkik8n2CASCYFlZWaqiomKJsFJSUvIjoYDUKTl5Q6lUui4UCkN5eXlf/gvG8MXqEEqGrAAAAABJRU5ErkJggg==',
clock_pause=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMxSURBVDjLTZNdaBxlFIafmd3tTmYnJU26TYgNbZoQ7GIMBH9iNQj1QimIFCL+EVEQQa+VXBQKBfVGIlL0QtDeBFEwFSlN0EaqgkaQ2tYG89OYxiaN2d1ssplNZmfONzv7ebEx9sC5fB/e874cQ2uNN26YwGngJCBAf+qEFu4ab9xoBi4BAfBy6oSeAzCKX5MA3k20vfTinvbX7vEXP8vPrrrzX9nnK0D3jn5qoPycmWlrycSTTYbMvvMXMNBwUi8buS84ZR0ZfNM69HxraXF08/byXPz71guOk4yoS1QB8EMT5Xsczw6UDx99hlhU8sozw8tAn6kE07AOWcX50c35hTlnsu1Lp71e0ej7yK0NvPkNnJJHS/0erjYP26uLk1asqa9R1x11lHDEWPiE/tCwP103Ow/+0vGN3WbHSZYj7u9spGGvicZgw434bXaNsC5GauM893qjktienk7q0guG1pq3z118P9Zw+K2OZocG0Tx7vJ2i61LwfNpaWwCoaPh8fIGqo8nmVyl48fc+fuWRUybAaqX1waqG6pbivq4myhVNIpFg4rtvKbkuoQiXJn4g07UfN+/jm/twVfJRgDiA+F53RRm4UqWpqQ6JNCnb5s7SEhOXf2Lqj+s88eRT7Dtgs1bw0Q0JxPe6dwHK9/DKSfyYgUS13vLuFinHYXstR+fBVm7duEZ2ZYW0FIndWWd/GDaeOXP5d3MHMFXaKmPEqyxmy0SGCaZF7wN9xEyDTCZDV1cXxWwWQpdMJkNPTw8i0mvunPBjyXXZCIRfr+VIJCFlJXis/xhhGFKpVAjDEK/sI0pQSiEiiAg1B4F30V3/Z3pmaZ1cMce50Tlur2xiSIhSandFBN8PdsUiUsvg57NvXHn49eGzHanND6YWK/b6lpDN5YlVTQ7cJQ5EiELRSilDa/0/AODDV4/drK+vt/OFYnR69GZsxrAxYxZPp2uAMAwxq4qqjgyR2p8ppWqAsbGxNHAhnU7jbbvudvHvj6Kqfhzolr1mo4gQBAFBEGittRGGIVprlFIYWmtGRkYc4ArQDvQODg7++Z+zoaGhG0qp7p0wJy3LeqhQKMR3gr36LzexzEk6GOJQAAAAAElFTkSuQmCC',
clock_play=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANBSURBVDjLXZNrSFNhGMcPQQQRfYv60meJoiCtROuDZRcEA6MP3ckwS0tlZVlptnXRqavMaqZ5Ka1Jq1bOmWiXs466mqZzLs0103CmLp27HOdlaf+es2xFL/xe/jzv8/+/vA/nMACYsWpmDiEmjEQTMU+o/wvVFs+e64mAP3XGoWLmEtljzXv7vSMsXM37bHp1ZEPyK6+WsM+ifa+O4tyGuJHxzjQ79euJpb4AWwWT6tLv/zY1VI3hd9GOD8oQXtowglvNNhS3DfoQ9DWuB23K1R6nSeLh205+J18LMZex3mPOu41p9qH6aIfuQciPvHd9eGQcgIL7CrmqA3mPO3DvdQ8Uhn6UvGXxSb11Ztz6eHro+TIzeQOYLwXMhq7C+ebGopWebLYHFfo+qNhedFtdGHHxGHaNwdznQnldN0rqe/GoUgajIniys3BhK3kDfINILq7KSXlqQmFDL5R0m7BGnU58/jaICdIC/E/gjqYbcq0F6UoO8aW6K74ZCNveghbtqScm3Kkxo5Nu9vz4Cd7jwe2SUtgoyD05iae1b8B9diJT2Q6hV/D4A3bmcnaRohVZD42wjXsxOjmDKTo4K5bggaoSKRckqNPpwQ5acEKuh9ArePwB2zNr7LFFeohLDejjvRQyA6vTjcuyqz4zZ2hHWtMJiOpjkfDmEGLL1BA8/oBt6U+0u66zkJS34K3FiQF6tNXtxQttI3rsLgxNAymNiSjvzsfVVgkSa2MQmXWrxR8Qduq+OEL8HEl3dZAqzRimgY16AfcMQdpBASfZeJSY81BMSBpTEK3cjUj55rW+gNAEeRDRseV8FUQFHLKUXTD0OsDTPHiPF0bShyujkd8hwyXDaeR9lCK57hjCczb8/dbXHpYdiZOWe8LPPMMB2UuIbnJIvtEA0fV6HM9lsU+xG7ntGTjXlIgc40UkaGKwXrxmwh+g0+nCTCYTXrPcdOixIqw5rsC6JJUPQe+4G4Ws1guQGtIRrz6EkPQgb+Dplb+foNFoFhG8xWKBuqrKvmpPmmTFrlQtYZ9FG3Fj84Sk6QyOVh5EcGogDmTv2eEfYllZ2QKii5gilv//KwtslIaORuRuQvC5QEjzM4apb4lQ/wXCx9fe4QKeWQAAAABJRU5ErkJggg==',
clock_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMLSURBVDjLXZNbSBRRHMaHSHrpKehCYTch06AoTEiyy0MPEVFIUbYkqOFmgoZZBK2uli3VlrHeVitzUzPqoTT00VoqLe1CIomL3XPD3dnZ2ZnZcXac2fk6M+VY/eF3OPzP//vgfIdDAaCivdQcQiVhmDBEmKf3/4b0Fv85HyQkz/Qp9iGVQLgSfW2ZmA49hfgqOyDcXP1CLS7yKkWFjGItYKbz87x8/fJn4puC0NSojSHzg4REwyBwjzrHDR71y5O94F9Y2PCNpYLorAbffB1sRxPYdje4xqvgL5Qh5F4gRt/bROH9qSDRvSUkUD88VDk/bGPCXgsbaFimcLfrwHd1QnY5oJUWQis+htjlcnD374BpOo1gx8a49P2+OtmV4iPaZOpTM5X52Z3gm3AtErnbtRAfeBBtv4XYxHco8jTUmAzp21cIrhqILXWYbDiEieak2Oe6ue+INvl3QKeOOKNOO/hHdxFtaYReqqqCpmnMlBaPg3NcRKT9JkKluWC3b7loZKAvalGhl2uqQcxRjqmPX0xRa2srZFk29n19fWDfDUM6kQ/GUQUuLcNrGuhJs21uaHnZiE8rpoHdbkd3dzcqKiowNDQEgWGh7cgA7b6OyIZ0xjQgz8SwdxqhHdoHRYoZYkmS4HQ6DfHIyIjR4yZpaJlpCDaSF0rdNGsg5+R4I7WXESs5jvDbYWNYURQMDAxAEAQzk8CTfkiHs0BX2RBes372CrHsI5V8+VkwnR6wZ8oQV+P4v/QeY7XiZ2sLQrm5YFalVpoG0oGDaVP7sz6Ea68g7LpmmERHx6BpmkF0zIdwaQmClxygHeeJOOVDKHFNmmmgI+7Zaw2UnRFplxM/OzyQ8rOh7c6AtisdkiULPzwe+Otd8B+wiERsNf/CzIbcd6ce1mDfEzVUUoxQtR3BhhoD2m6Dv+QkXj5/rg3092tkdts/Bj09PQsJwvj4OHofP2Yi6VurIus3e/WkSVgMs3qdl1mxtpKc8T6fD2RWISSZBm1tbfMJYwSZsO7/rzwDOVtJiBD8hCV67xfvC+0h/fUq5wAAAABJRU5ErkJggg==',
clock_stop=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMsSURBVDjLXZNrSBRhFIaHIIIyu1q/+hdYJBVhJUmE0EWiC0U/ukM/KroRllGR1S4UWVnZTdJMS2ujrbZSK7ruNK5rm+m6bV7WNbfc1F3d2d3ZcUxH7e18k1k08AyHb87zHuYwwwHgOp9yQwgd4SA+EMPY+b/Q2cSB5zYi9s85FzJxQ4kznRUbvqsBM6SKjX5b0XJL6muVJ8QB+PdFq4SIfUegqzZNpH4bMUkL8N/ljki2TS09vqfoKN8S+micJ6dbArhS4ceN6jYNVp8XmlBtnKWEnXpFrt7fTl4lMZTz3uSORhxpoq90S8h6e17vpfJm3HO0wiB8RZapBpfu1+DmmyYY7N+R986MuqIl/V3e+32+x1Nd5MZyX7K5+fU5w11ludOVM+Ym3LU1w2T2oNErISDJ6JA64WqWUPiyEXmlHtx7kgGHIaG7Nie6itxYbRGpN4rPHnzoRI7FAyNNY1cwHEZDSxt+UM2QfwLXShqRxbtxzChgZ771pLYDdtuQXckfeODEtecu1NJkpfcnZEXB1bx8+Cko0t2Nhy/eQmgI45TxE1gvcwYD1mQKYoqhCqfvOODvUhHs7kcPPTis0+O26QkOHtfjpdUGZ1jFviwbWC9zBgNWnHoubsu1QZdvR7OsUkg/vOEITmSc02TB/gkdvcDHgIo9meVgvcwZDEg+9oBfe8EMfWEl3rnDaKWX9kZUPOPL0CRK8PUBHgV49DmEfdnvwXqZMxiQdOCWbqnuMfZetyLd6EIHLSyoApF+guoQBUjr1kOeMAFSTAyCRGD8eLSNG+fRAhL3ZMUTNYuPFiMlW8BpYz3snhBk2oesqHBQzURl9Wp0rlwJadkyhJKT8XXs2L/f+pytGdt3pBcqCw89wuaMV0i5LCD1ogUpF0qxO9OsTWSiOHMm2uPi0J6UhLrRo/8GWK3WJKfTiTdmoS9xVy5m7zZg7l6TBqu9NE1ctAhtU6agZfJktCYmoio6+rdcUlISQ8hutxtFxcXijPVp+ri1R3hCHIBvGDMGvgULNNGbkIBv8fEoi4r6HVBQUBBF1BM9xLT/f2WGc9SoOjaxYuRITbQQ/IgRwV81l9Ohf930iQAAAABJRU5ErkJggg==',
cog=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGSSURBVCjPVVFNSwJhEF78Ad79Cf6PvXQRsotUlzKICosuRYmR2RJR0KE6lBFFZVEbpFBSqKu2rum6llFS9HHI4iUhT153n6ZtIWMOM+/MM88z7wwH7s9Ub16SJcnbmrNcxVm2q7Z8/QPvEOtntpj92NkCqITLepEpjix7xQtiLOoQ2b6+E7YAN/5nfOEJ2WbKqOIOJ4bYVMEQx4LfBBQDsvFMhUcCVU1/CxVXmDBGA5ZETrhDCQVcYAPbyEJBhvrnBVPiSpNr6cYDNCQwo4zzU/ySckkgDYuNuVpI42T9k4gLKGMPs/xPzzovQiY2hQYe0jlJfyNNhTqiWDYBq/wBMcSRpnyPzu1oS7WtxjVBSthU1vgVksiQ3Dn6Gp5ah2YOKQo5GiuHPA6xT1EKpxQNCNYejgIR457KKio0S56YckjSa9jo//3mrj+BV0QQagqGTOo+Y7gZIf1puP3WHoLhEb2PjTlCTCWGXtbp8DCX3hZuOdaIc9A+aQvWk4ihq95p67a7nP+u+Ws+r0dql9z/zv0NCYhdCPKZ7oYAAAAASUVORK5CYII=',
cog_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALASURBVDjLdZPLS1tBFMaDf4D7LLqviy66aulSsnBRaDWLSqFgmvRBUhG7UDQoJBpiSGpKTQrRIIqvYBRMKJHeRuMzPq/GNAbFx8JHLwoKLZau7v16zlDTBuzAMAx3vt/5zjdzdQB0N821tTXz0tJSamFhIUXD/L9zRZutra2yjY2NUhKXkPj89PQUJycnGBsbO08kEiXxeLx0fHy87EYAiXtIrK6urirpdFo/NzenHB4e4uDgACRUYrGYnkDKyMiIOjAw0FMEyGQy9v39fVxcXGBvbw8kvpqentby+TxyuRwmJiY0El+RMyiKgsnJSXi9XnsBQFVbqFeNISzY3d0VoGsA77PZLBiwvLyMpqYmrb6+vqWohcXFRRcfXl9fx8rKCiRJQjgcRn9/PzsCtYXZ2VlR3ePxuAotEFGm6sczMzOXOzs7kGUZyWQSTqcz3djYaGhtbTX4/f70/Py8APF3n8936Xa7j9va2mQdidWzszNhlytTkBgaGkJ7e7vhukp3d7eBMgCdFc7YDYdrs9lUHd2xenR0JHrkD1yBEkdXV1cBEAwGDZFIRDjgFsitOG8ymVQdXYlMFo/7+vouNzc3BYRz6O3tTYdCIUMgEBAthKKdePvxGV52PsJTZ7n2+HX5d6PRKBdCJIsuClIExSs9JIyOjoLuHYGIB46oCZ9yQWS+SfB/seKJ/w7u2fQ+IY5Goy3Dw8Pa9va2EPN10cMSmTCoxlOB2Nf3iOU/gIcv+QL+5CsG/BKAwcFBOyfPL49AoHSvXC6XxqFx3w/td5HIhfHviGeDDPj7ph0OR09dXZ1qsViUhoYGPUEUdsIOHry5pXml53BLNULs/lxT7OB6EqDMarWWNjc3lxDwfGpqiv8DVFju/zT6buOdZBGVeeV9IYObZm1trbm6ujpVWVmZqqqqMtPhDpo/2PaftYPP/QZledsx50IwXwAAAABJRU5ErkJggg==',
cog_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALhSURBVDjLdVNLSBtRFL1vZpKYTCQdUZNUMGahJbTF0roxG7NoqXQZKG6EQrV0Z6kLF0pDDSguxO7aRejKLLW7brRgiSWKVcFgq2JDSUJsYmKi0fw0M+m9Aw0KNnBnwnv3nHPPeW9YtVqF637r6+sDlUqlX5ZluLi48Ltcro/X9bHLBKFQyIGAONYZghOtra1NtL+6upoyGAwWXDMiWYvb7d75h+EugX0I3MamPWwyY8nlchmotFqtTGvn5+d7WNt+v993hWBra2tMFMVBm83GSZJESvtY5lKpBMViERRFIcJ93LegFa6xsXFwenp6jLACPZCVSyQS5IWRotlsFjmOI++qisPhoHWR9iKRCAQCgSqSc1cyCAaDEyaTaZRUKbiTkxOIRqOg0WjAbreDIAgqYaFQILuTIyMj6gQMA9rAcZuxRIvFIuXzechms0S4gv89Op0O0Ja3q6urm8TQKiwtLWWRLI+YQwEf99rb27mjoyNVlRSTySSgBc/U1NQXUvH5fLS2iAJAfU6nU0Kb0szMzE2OxiJvSKSOTipGoxHQTu14qYf2MUy1BzMDmpTssNnZ2Y10Ot2MY4qdnZ0SpY7HRj5XEOghMJb3AYt3K2ufoJKIwZm2TtnTtp2uaTrCtRDn5+cnrFbrKClRYHQKsVhMVWxJboIpHoQOlxt09ttQDC3Az+VFObu781olmJube4Mq4zgBOz4+ViegohOhsfPv+sD57BXow18B4t8ATDcgLdhgc2H5t3oPMAMFA2J6vZ4uFYTD4TxOYOjp6WFEAOkDqLPYAZ4M13IR3lqBV1hbzYLX6/VlMpnnGM5hQ0PD/fr6+s3e3l66lZAcfyQ/fNrHi9HPUC4moID9pzkedsMQr30LHo/nBb7u4Oi3MIMkKvO5XA5SqRREpLvlHyuByp+SHnK8BrIZBr/iTEbp9+x/n/PQ0NAAgvvpiBlj/mHN96Z85uAlLzObzFfjiPrweKEy+RdV4845vRqFoQAAAABJRU5ErkJggg==',
cog_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALzSURBVDjLdZPdT5JhGMaZf4Bbh3ZaBzn/gU46MM1iNrf8WExNVyz1IDUOHCFpfq4PmmZmtTTT8GMJNGAO0hgi8a0oIBAGfoS4N12gmUpmL1fPy9LGpu/27N2zPdfvvu7rfh4WANZRa2pqims2mycMBsME+bjHnUvYOJ3O1JmZmWQiTiLi9dXVVYRCIUil0nWVSpWkVCqTZTJZ6pEAIu4mYtpms1EmkylFr9dTi4uLWFhYABFSCoUihYCo4eFhWiwWdycAHA6HMBAIIBwOw+/3g4i3tVptzOv1wu12Qy6Xx4h4mzgDRVFQq9UQiUTCQwCpWkd6jTEQRjA/Px8HHQCYvcvlAgOwWCwQCAQxHo9Xl9CC0WhsZQ5PT0/DarVifHwcPT096O/vZxyBtAXbx6fwDXLgV7TC8ToLzq60EhYh2kn1FZ1OF/H5fLDb7dBoNGhqajLx+fzM+vr6zPb2dtPMxCNQWj42XaPAz1VsuKTQiy7/ZhExvba2FrfLVCZBYnBwEC0tLZkH7qzS7Kbv7nvYW1GC0omwO/cef5YNxEUZWGTGdDAYjPfI2GQckMTR2dkZB4TUpy9F3Hdj9K4Buwu3ELZV4rOYC9ebMvAqSmkWGYmdWFzp6+uLzM7OxiFMDr29vabJoSu1kTkB6KgZO4FSRINF2PLWwtOVQd/m5n/Ly8uzH4ZIxtRKgsTk5CSYv17+GNQU/5+4BNGvhfjhFmL++UW01XB6E+6BRCKpGxoaink8nrjYMNoB71gN6F09oktc/ApewyZx8oWI7Z/GUFVVFSsvL/8/xoGBASGTPHPz3rbx4FHVwql+gpC1ADtLRQg77sDZcSEmE7+IZ9XQ0IDc3Fxhwj1obGzsrq6upkceZgBby/C9yoe29iSML9n4cO8sLRGcT+dwOFROTg7NZrO7j3xMBJDafDNtf8/8DHvWDlhb0zFScQqV13MjhYWFScXFxckEkHrsa2RWwbkT0fulZ/Y1D9j0u+asjRtXsy3E7rHP+S+qJels2qSd5wAAAABJRU5ErkJggg==',
cog_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL3SURBVDjLdVPdT5JRHGb+Ad57W6tFrpZ200035bpuuTk268K0LuDC2VpEhAXCyHRaudacjvxA3QKcOsVSQEABv/CDIYl8FIK9iUKt9FVyL0/nvC0Wm53t7OzsnOd5fs/zO0cAQHDcXFhYqPZ4PFMzMzNTZFT/717eZnV1Vbi0tFRIwAUEnNza2kIikYDRaEyazeaCkZGRQpPJJDyWgIA7CJibn59n3G53kdPpZKLRKCKRCAiQGR4eLiJEzMDAANfb29uRR7CysiIPh8NIpVIIhUIg4D2bzZYNBALw+/0YGhrKEvAeqQwMw2B8fBxNTU3yHAFRVRCvWUpCAcFgkCf6S0D3Pp8PlGB2dhYymSxbV1enyLPgcrk09PLi4iLm5uYwMTGBzs5OdHd304pAbMHhcPDqjY2NmpwFwugl6nG73Z5eX1+H1+uFxWKBSqVyS6XSsvr6+rLBnsfBdds9zDg+8OfNzc1prVYbb2ho8AoImNve3ubLpcokSPT19UGtVpdRhchkZeHG+5uxVMiAVcszvjJaDQ1XIpFwAtJjbnNzk/dID6gCSRxtbW08QchcUb/zUY+jvTACQyJYx7pBquXvV1VVcQLSEm9ra2u8q6srvby8zJPQHHQ6nXu0V14dGLzx43BnFIdfWpCOjGHBKMHdO9VpkUgULy8v9+ZCJG3SkCD5oOjq8bjheleL3aARh3EVnM8u4lfKiGXdVbQrK0x578BgMCj6+/uza2trPJi2y+cy4JNVjkyyB2y4Ck5tCQ5iUnyPmjHZcgV1ktvqHIFer5fT5OnLI0TQqFV79vbr+EaCO/h8H/sboj8zdItYeYGY4xXePrxsyXsHSqWyo7a2lqupqWGcr689ijlbkPn6hlffD1USC6X8ykbFYJlhOBvObTuUxWfzPhMhEIrF4sK5l5cS7G4EmbQVh0mSQdLwzzTh6KcfiVk9bPLThmO/qOt56e60tiRjV1/gplTnOfvT4oz9iZCdUpxhCYi1yk6xVulJ1vLgxPRvyXT2ghRTB18AAAAASUVORK5CYII=',
cog_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALtSURBVDjLbVPtS1NxFD73Nid6R7o13XxpY8WgJWTIPhQskMj+AKFvQpB+HtYHIUd9EJQC8R8YftoQhNk/kGStNV+a1cSyMXQ5tcbm1DX3du/93ds5FxQlL5wxfr/zPOd5nnMvp6oqXPTEYrFBWZYHGGMgSVKwt7d3+qI+7izB2tqaCwF7WMcIzthstha6X1payjU2NlrxzIBkHf39/RsnGP4M2I/AdWxKYJMFi9VqNaDS6/WMzkRRTGCtB4NB/zmCeDzuEwRhyG6380ajkSYlsSzVahUqlQooikKESby3ohXebDYPTU5O+girox9k5TOZDHnhaKLFYhF4nifv2hSXy0XnAt1tb29DOBxWkZw/l0E0Gh1vamoapakUXKFQgHQ6DXV1deBwOECn02mE5XKZ7E6MjIxoCjgMaBXltmIJVqvVWCqV4PDwkAgX8f/L+vp6QFtjbrf7Lg1Dq7CwsHCIZCXEZLlIJMKcTiefz+ehWCxqEzc2NiCVSvX5fL55muL3+x/8aJx5W5IK8KjjNRgMBrIJU1NTCk+yyBuyadJpCjWgndP1Uk9VFqHT7ILpLS9lBqSU7HCBQGB1f3+/FWUK3d3dRkod1waBxFNMX2SiIoMkM77d5ORudXpg5dc7iO8sw5WV3l0kzp6GODc3N97W1jZKSiiwma1n0Nf1GJiqAFMYKKBCprALRqEVFrfmYTX9QUZVem2NoVDoBbI9b25uhqOjI00BSSbwdj4JEqqQFQkkJsHfWhFuX70Hx1JZt5yKVDUCzEDBDXANDQ30UsHm5mapaq0KMpPBctmGYKaR/SmkwWSwwpedT7Cc+JwrA9hPLYyNjfkPDg6eYDhZk8nUk3SEftcwLFERoYJqrptvwp1rDyGWjkJ4/eNBia+0/55Qa+c+puHhYRda2UMrx7j/jMfjaaG0Z2dnc99uzHBu231z+Of77/iq9SBY/O9rPPt4vd7BXC43QCvmOC74tevNK6ZcMlcUVpedUOWTvn/PQ+bC95cxOwAAAABJRU5ErkJggg==',
coins=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJuSURBVDjLjZDLa1x1GIafc8uZqUlMMmmsLV7SC2hLCoJQ6tou3Lj0T+jGtQjusnLlP1Bw01UJgrqUoAiC2aixDUQl2oC9TjuZSWbOOTPn/L6La5MRfOHbvTy8zxe5O8fT3Hv9opt/784ZN0vcqN18F2P9hesPv/5X2d1P3Hj71VF4ctu92nEvttyPNj10b/vwh7N/Hu+mTImrzaYLb8PkMcgAwoA4n8PELhzvTgWYgtUPicIh+AQd70Mdo3JS9z8WODr8mdD9BqsLrDoi61zDBP7nAiPOz5HNv4nXT7HsFOaGip0E1Nuvzbv5rpudcSNx9TryCBn9hvR38EmBViPa569OVzC1T9KVj85lL70PPgEt81D+RfXHOu3ld/DWU5J8AC5oYBrAP05X3gMZgg5BC9L2CqE8IIl38fEILUdk0QoapiioAFbiUoA3WP0cmjEixsyLF/HWMzTvk8wuoNOeaGJouYce/oI1Xbx+QDJ/Hm2cuv87XpVEzQAvH3F6Keboq2VXpVaxXVPWUw1OlHVI2qvE2SKedXAfIMHJFy9hrS5N7znt618Qp7PABA/DfHJ0963ed59+FqsYURwj1R4yvIcMfyWdvYI0Tih7NAfP0EaJ82UIAxg/Ipo8obVwiabxC7EGNsa9bbK5y6Rzl8mWrlEd3CfJl9BTZ2m98S6Wv0z14A7uExxB5JDR/gZN7RupBNuq+3c/iE9fIckSwrig6O9RHfa+LX/8csHF12Zmom5n7qdXoCBOHSkfU3T/JtS+Fd2/01k14aap3VBlzYQdU9805dbVDwf7APufL66K+E0NfkOFNRXfUWPThFv/APJzrlrFns7aAAAAAElFTkSuQmCC',
coins_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKnSURBVDjLjY9NbFRlGIWf+zfToTMFWjKUQrTlJyk1FBYYlI0KkuimceHKsGGhG2JcGKNxRySBmBj3JG5YGMJKF5poNRAbfxJDCy1UsJZGbaCddmY6c+femXvv970va9ox8SRnd/LkPI6qsjnp7PBBFb2pyqCKeCokKjqPcKH3xeWvnxqr6pa2Z54Js8dXVeM51davqo1JzVavanNq6K/NW58uUStFf8dx6DwCU4esjpsvIUYObN52BYgFSZZxsg3QDra9BImLNVt1/+OBYpu3yFa/Q5IWEjcIBk4gBv7nA8HN7yXoO4wmK0iwDVHBGtkKSGae7VPReRUZVMFTq4mjDib8A1ObQzstbBxS2D/eXUGsfOyXP9gb7J4A7YCN8lm0SPzgAoVdJ9GeFbx8HdRgM7oB9EO//DqYJtgm2BZ+oUwWVfHcebQdYqOQwCljsy4K1gASoaYFmiLJGqRtjBFy2w+iPRVsvsbNapEbG6dofv6GhnEj3GjVPv320p2LvhjBRgvYjWkkXUWTf/H69mNTJandR+OIH1Z7udvr8/JLr7Cv/xA37n1V+u3u1CdnPxo87tpMcYIBvMIIfu8ofvEYqMVkSn7nIXL9e/hmpcSx0aNY13J0zxmsk/HCkZP8HbkTrjWC47qYeAHTnMU0b+MXn8OkShatk1YrVMIagVNk4vC7ALx/+goHyuNkiuPajOvt9RmC0hh+aYyg/wRx9SFevh+7bYie0VcJwzXuPfqFy5PnALj8/TkWK7N4DtZ7783CPjfX95rbU8TxHbK0Rrj2J+Hywo+Nf+7Xqw+md7Uhmq5X88O7h1mPl2lEFX6+M8VQO/7JeXhtYEQM74iVM9ZyRAxzYnVSLFfGz9eXAJa+2Dny2WLuy9mW93wqeDkHO1Ywv7+9o/PWEyzemnF7ZfggAAAAAElFTkSuQmCC',
coins_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKZSURBVDjLjY/Na1R3GIWf+zV3JolJ40iMptZMGrGtVLoQpK66qagLV0VwI3XThS4EQQQ31Y34F7gQFOqqdCMtdGGCRXAhQmskgVSNmYAOmkySmczHvTP33t/vfbsUkyl44OwOD+dxVJXNSWfHJ1X0oSqjKuKpkKjoPMK1/m8rv38wVtUt7cx81sre3VWN51Tbj1Ub05qt3NXmo92vNm99ekStDPifHILuWzB1yOq44TbEyOebtz0BYkGSCk62AdrFdpYgcbFmq+7/PFBs8x+ylftI0kbiBkHxMGLgIx8IbjhGMPglmiwjQR+igjWyFZDM7B1U0XkVGVXBU6uJow6m9S+mNod229i4RWHiYG8FsXLFH7k0Fuw8CdoFG4VZtEj84hqFHUfQ/DJeWAc12IxeAL3sjxwH0wTbBNvGL4yQRet47jzaaWGjFoEzgs16KFgDSISaNmiKJKuQdjBGyA1NovkqNqyxOrtB5S/D4u1ArKcV4ObRKXPDFyPYaAG78RRJV9DkDd7gBDZVktpzNI5Ye9Ygqo1x6MzPhKUDTmd2as/8o+nrT84WJlybKU5QxCuU8Pu/wB/4BtRiMiUc3kdu+y7e/F1l8rtT5Bcf4vxymr7yPcb3Fp24Zn70rREc1yWLF9DuOzRdIRw7gUnvkUVr2HoVUxfyoyU4cfG9+9VdSJvAtxm/ddZmTuW3fYUEw6DjxOtlvHA7tm83+Z0H8IZeEj/7k/4/zpF0lomBVtNDC07Hu/BD4VM3N3jMzQ/g+A5ZWqO1+pJWZeFB4/Xz+vqLpzt8vy+qvqqGbuCSeRGNdaW87OEPuVNO+ddiSQw/iZXvreVrMcyJ1Wmx3Dp4vr4EsHR7uFSby9/ZKK8dISKnBdKg6D0e2J87+x98zpgrhVsXPQAAAABJRU5ErkJggg==',
color_swatch=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABjSURBVDjLY/j//z8DJZiBagb8y8/+D8NgsVXF/+EYyP9wNf0/DA9SAygOgwuvN/2HYRA/4EzufxgG8RM2vP4Pw4PUAIrDIKJqw38YBvFvzr77H4bBaso3/ofjwWnAwGcmcjEAc0v+JGPFQvwAAAAASUVORK5CYII=',
color_wheel=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMOSURBVDjLVZNNaBxlAIafb+ab2Z3N7Oxv/nYTEyv2LzQJpKBgrQqNUKmY4kUIXqUHT70p9iB48CKIiN5E0It6KFiwiv9FpAVpKUggNc3mZ7vpJpv9n93ZnZ35PNRI+8B7e9/n9gqlFAeIVUfPeN3zh0R0eVpYM1OanhvTCEY0f3tU79+ctnpfHM73fuQhxIHAWHnmkOGXPjgZyS09l5hnNv4YOdMhoQmigzqGt4nhfeub1fpnVsl/e+hMv/q/QKy+Me0EO5dfso/OvzB8grgV4HGXJC7jwAQ2oxxDuC36xZ+Rhe+v6iutZf2iqklReNe0tPSHZ2Nz84ujR7ht3iJKjcexiOIQI8SiixxcR7QtRORFlK7O9t0rlyy4KBEj5+YisVeez85wy9zGIUeGDDYhDhYOITYuoh2BvTJ68y7B0GnCym8XGq+KL2U0MrE8Z2SRVhqdPmlCsvgk8RlCkgAivRbUNKj1YPMeeu4wcnjRql7/+jVpyvxsPjbK3whi5LEAB0WWgBRgqwAaFah04X4V7puwdwddz+FXjJMSbXI8aSTYCgU2oKMwEdgCEoDhug/G5SjsmFDUoV+DXJ7BnpiUVCNBaJqEXfDVfwG6CjoKnF4crZGCVvNBug0IPXzPZOCnAunfk8W6ro7H2gK3A02gGoDeA1MDGx2nkYG6C24bvDaMSzq7ZfxBsiC7O+aNDaWOn0oLfl0HMwDlQRCAHYUkEGvFkLsp2G9Bo0n41AiNG6sMBvY1yZr6/JsV//XZZ3WZaEp2t6DvgWFA1QRHQbwjSDeTUGvCiSPU1ovU/typQPIrTV0yrrl3vE+/+8XlaCIgq8H+BtSLUN2C2ibsl8ArR+HYGE0rwvbvRTr96HsL6od1CUDDf+enK92JwT+982cWEswvRmiug6qAr0E4AV4uoFXosnV1g8bN5kcp7E8eOZOYKtmUqm/ZiDdfPhV3Zp6IM5k0SIUBstwmXKvCX5UdM6y9n2b34wV1IXxEcEBU3J4dprU0zODpjFBTIyoIxgjXxlB/PIl1eUmdLjzc/xceOVXddrB6BQAAAABJRU5ErkJggg==',
comment=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEvSURBVDjLY/j//z8DJZiBagZEtO8QAuKlQPwTiP/jwbuAWAWbARtXHrz1//efv//xgS0n74MMuQ3EbHADgBweIP7z99+//x++/fv/8tO//88+/vv/5P2//w/f/ft/782//7df/f1/5xXE8OoFx0GGmCEbIJcz9QBY8gVQ47MP//4/Bmp+8Pbf/7tQzddf/P1/9RnEgM5VZ0EGeGM14ClQ86N3UM2v//2/9RKi+QpQ88UnuA2AewHk/PtAW++8/vv/JlDzted//18Gar7wBGTAH7ABtYtOgAywxBqIIEOQAcg1Fx7/BRuMFoicuKLxDyzK5u64Cjfo/ecfYD5Q/DLWaMSGgQrvPH/3FabxOxDXEp0SgYp7Z267AtL4BYgLSUrKQA1KQHwPiFPolxcGzAAA94sPIr7iagsAAAAASUVORK5CYII=',
comment_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGkSURBVDjLrZJPKINhHMd3UsrJTcpBDi6UC+3ookQ5OOBAaCh2cFQ40GqTzURTihI54DRp8dqBg3BQw5BtNmxe/4flT9re5+v3vJvF2l6Kp749Pe/7fj7P7/09jwqA6i9R/ZugVr+cSZmlvFOgEIGSl0xgnVt3IRyRoDSWtn1c4qakxQW0yKBEJMbw+MpwHWIQnxgCDwxnQQbvHYP7RoLnJirvntrkkuKvghytZU1+eUWg+MjgJ/j0nuEkBh9dSTgQo4KB+R0uqEgquCD4PBiDbxlc11HYSfBuILUg/gu8fB/t6rmVcEzw4aWEfYIdAS6IyILe6S0uUCdtIpd8Hbwah1+SxQlNTE91jJHPI5tcPoiLrBsL6BxrQOtQFep0pc/lXYU9P14kkngugy/onxlF30ITlpwWOEQB5tV21JgLUNKRZVSCTeM2J6/kuV5fFrbuD8N6OCJXY7S3wGxv44K3VHAuxUvR8HVldxFszolvvVncs3DB7+67Wpv9Nig0Qy80yrB+pVG5gsTQh7pqYz5Mgkbemc98rdiDJBIDJcTLjs0G/vwDCw/6dFwBuzsAAAAASUVORK5CYII=',
comment_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG2SURBVDjLrZNJKIRhGMdHxpTtwIUoRRxdlZvcJsuFcBAHFHGSSBKlbCUZOTgolLKXskyDshwsJWQLYx+DGcaSsWS+5+95Z5jQzJjirf/hfb9+v+d5l08GQPaXyP5NkFGnDuT0cF45cBENJ9KRYKRvdg9vFgmuxujSkZDscxR2AU/8OBaJCHdPhKsHgv6eoLslnJgIh9eEfYMErcEmr+hcEJKYr4KworYZ68dLBvV3hDOGj28IBx/wzqWELb1N0NC/IgQJDgXnDJ+aPmAjYe/KBm8yvK5zLrBvQbR/xFW1Rgm7DG9fSNhgeE0nBBaroLJ7UQhiHR6ikHwdopu1M8kq/nGI3s6u0fJ5ZR3qLbtIoyrARFoQpuLlGE70oZb0QM2vD4kl2guTGV3VmVgticXzWBNoWw1zbzGWC6NRk+o/7Qpuah/fFJ08DiX50RPDUCUBZQFAbTiMjXHoUyrIGRzBOeTkirlom1aGv53NbVUwJnndrfc+wJUeO3IAhl5KZTBxTvI9Maj0IrcErVkhcwt5UdCXhcNQ7oWDXA9MJctRn+I77/Zf15wdOtOvVEii7QGuzPCsWH8HCxz5pzmy7lQAAAAASUVORK5CYII=',
comment_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIWSURBVDjLjZNPSBRRHMf32rVTdFOsDkJEhODNLGqXukgJpmiEURBGdEnbskNktrhCRQuaLEEikUhlbK5EiZmxjbWwfxvL0dHdtdlCx3VtZxyaed/eG5qwZct98DnM4/f9vN/M+40NgK1Y5p7tPTY9UIeZ4Q6EvIcQ9pQ3FR1O+kvqpbFWZCI+YG0RK5EhBNz2dFHhxIvSWjl+TdOSzyGNd0GJPoE+P4nogzPqpuGUv8wux64ahjIJZbYFy1Pnwfc3I9LXuDR1t2bnf8PC0xKHHL0MQw0gJ5yEmmhA9pMTYm9VOth9cA+rsdV1jm6lDFA0Cizabl6H9KH1d7gJ6kI9VmNXIHiqs5/dFfusQ5hg+PGbL/ipG7CWxPvAv7wEQ5mAKjZjPdGIDO2E9xwmgS7Hjo1dMoFuEIKMQvAtS8C9eoT4iBNh/22kuFrkxAYsh9ow661Bp9fHuqv4S9DiGTdPTa8SfM0QDLoOANl5TN8/jjHndrzrceCt2w71uwDXYJAJjhQULNJwQia4cXY3tMA9aNwdcB37MXRuF4Ih3qwpKLBegbUvLhGcqN6GW6fK8dp1FBP9F/AxvoBwSjcF7Q/fM0FlvsD8iEyycbFuQknDFLPl40QWnqFsyRdY16hbV+gdjf8Rraytm890P0opy5+VggNECwVJzllBldL+r2ErFO7uHYmx4A/Kxc1GPT9cSpmjnC72L/0FRS76cD+dhSEAAAAASUVORK5CYII=',
comments=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG/SURBVDjLjZK9T8JQFMVZTUyc3IyJg4mDi87+GyYu6qB/gcZdFxkkJM66qJMGSNRBxDzigJMRQ1jQ4EcQ+SgVKB+FtuL13EdJxNDq8Ev7Xu85797T51nwhqeAH5w6cAxWwDgReX7jwYfdaCIraroptB7NLlVQrOoiGEsL1G06GZyxuILicsMUH3VTlOqGKNUMUdTacj+j1Nng0NGAT2WxYosK1bbIVVoiW27J9V8G57WWKVSczMV5iK+Tudv1vVh5yXdlLQN+os4AFZss2Ob82CCgQmhYHSnmkzf2b6rIhTAaaT2aXZALIRdCLgRtkA1WfYG4iKcVYX52JIs7EYvFmJ8wGiEXQi6EXAhdyn2MxQaPcg68zIETTvzyLsPzWnwqixVbhFwI3RFykes+A9vkIBKX4jCoIxdCLrI4/0OcUXXK4/1dbbDBS088xGGCCzAJCsiF2lanT8xdKNhHXvRarLFBqmcwCrbAhL32+kP3lHguETKRsNlbqUFPeY2OoikW62DNM+jf2ibzQNN0g5ALC75AGiT59oIReQ+cDGyTB+TC4jaYGXiRXMTD3AFogVmnOjeDMRAC025duo7wH74BwZ8JlHrTPLcAAAAASUVORK5CYII=',
comments_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIaSURBVDjLhZFNaBNRFIWzFdy6EIrQhdCFLlyJ4KLgShFciOhGTbVVUFf+rNxYEFPtmPhDKWI3diP2DxJF22mfirQrMVpbRKUohJi0SUwySZvJzBhOz32dlCimHfh4897cc+67ZwInQuNtJEyiTXhGzpIdAAL/EuCH+69nk6pUcZVVZ2WNIlkqVtTo9IJi3Y1mBjERF1icX3bV77KrsmVHZUuOWrKq+jyRKYvBYFMD6SrijC9aLFZVqmCrZN7W+80MnpdsV+XYWYrTFL+dS72/8ng6f8qY9E4TWVnnkIJPktyS/MRgJEeh49W0WDpfG5gpMhdwNFh1VtZgLmAuYC6gdlQMOo2RuIovZJT7p6Y52Wt6Iub84GhgLmAuYC7gLfU5xxKD73oOvuwnQ5L4xIeEzOtJVxFnfBFzAW8H5qL3fxn4Jk/MuBaPkzJzAXPRxekGcSJXQZrvv3LLYvCzLt4iYZJXZCdZZC6oerV18fC7IVx9FMS58BF03DmE/thDMfhaN9hGukmrvw+Fxz5i9kcWzATRmafoHu7Ai/k+fEqZiExewPHIbrRfPvAm8L9/65scI5ZVcRC8fRDRuXuIfnkAeYypLkSmzmPvxe12UwPf5BtzweHre/ByfgCNT+xznxhsKN4qNyD2vkstTq95BiEzqMWhieDmN6CwhYyRXSy8edRow12zU3eWVfY8NzYcoREW95CSXNtfe+R8FZtm9C3KURomAAAAAElFTkSuQmCC',
comments_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIwSURBVDjLhZHLaxNRGMUjaRDBjQtBxAdZFEQE/wEFUaurLm1FfEGzENwpturG6qIFrYUKXbUudOODNqIiTWqvFEwXKo1UUVRqS2NM0kmaZPKYPKbJ8XzTiUQxceDH3HvnO+e73xnH8X7fLjJInjbgEekiOwA4/sbBD0Ov5sIqY5SVXiO/Rpospw01HphXrOttZPBMxCkWJ3NltZItq3i2pOKZklrWi9Z5SMuKwf2GBtJVxJotiqWLKpIqqHCyYO3/Z/A8UyirBDtLcZTi6Y+RdxdHAsnTAy/NM0TerCuRlE2Y9El+YjCWoLBkViyxdL40OpNmLuBo0Gvk12AuYC5gLqB2XAw8A2NBFZzXVHm1YnHq1qQpYs4PjgbmAuYC5gLe0jrnWGLwzZqDi33ksSTunw3JvKZ0FbFmi5gLeDswF2v/h4Ftcm8yaIl9JMtcwFys4midOJQwEOX6ZyInBos18QYJk0yQVhJjLiiald/iTw+GMHN2N6YOuTB9YieCozfE4EvNYDO5Ttz2vn/Q+x5zC3EwEyw9GcaH7v0ovLiN6mcf8g8v4O35vRg+edTr+Ne/tU2OEV03SvB3uGFQjDvtQM8moM+N+M0D8B92LjQ0sE2+MhdMHXShOutF/ZO6toXnLdVm4o1yA1KYOLI+lrvbBVBU7HYgSZbOOeFvc4abGWwjXrLndefW3jeeVjPS44Z2xYXvnnVQ7S2rvjbn1aYj1BPo3H6ZHRfl2nz/ELGc/wJRo/MQHUFwBgAAAABJRU5ErkJggg==',
compress=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKQSURBVDjLbZNNaJxVFIafe7/5KTPzpUloNK0tIsowCtbiQgRRQReudCMVqYrdiLgQ01UrWUgXXZQumoU2myyKii66dOFCEUo3IiL+VRMFHactYpsMmsy0mbnnx8X8tEn7wuHAudyH97zcG9wdgKWl9zNgl7vvrVar51T1PndHVQHDzBCRFGNhqd1ePXb06PF1gALAhbONF+7PanPtymtP9G2iVK3WmJjYibtjZuNupsWVlYtviaRTwABw4WzjEPDRVGMy/vt3QLpCu73G2toqZoKZE2Mkz3PyfBKxgKplDFUA3rz7wL5Y2lnigdrHiDhuRlaoYJslrv3cWb7cfehka/3BxUY93+EGqolbAU/tqz+K2V/MzFQAHZYQ4146v55v/NPd81UxL6uKQgyY2RgQB025fOUPCC9COAjhJVqt38BlcKpKb/M65kbq9YfB3nQAGOVSxqXWCXDBSZTLBWAAMDOKsYibYURE0naAMjOzC5gc2Pc0vDwApJTQGx3UDJHNLQ7GK1xq/Q7hFQivQjzMn82LY4CqhiwWw8BBQNW2OxBK5Yxm812whNNnx5YVtBBkoxICkLqYbcugt9Fh9+xj4/RHtblxA7EMVZsOYZC+qqMqWwBHfvr829OjgRNIWkIsIhb54cr+r7Ms+3Bqanr0GjHzm4AnDy8vAAujwfz83NTs7O7z3W7nYTOjH3uPp7RuWZYNHdhtDrZIVda/8+fPWa06nfWvfjJxdfFTEd2zvPzLZyn1CCHSrx954/UPWi8DC2H0G2/VM8ebzeceqd375fer/9WvnTgDVET0oLsWzJDmPe/lzx64K//ix43WHQH1t1fmgLkC/TNPy8lFM4vuWhGx6G72TXX+UAqVd4DT/wMfm3vSJoP5ygAAAABJRU5ErkJggg==',
computer=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAItSURBVDjLfVM7bBNBEH27d7alOKfYjsM3gFLjRCAgiAoFBAIhQUNJh0SLqGgpEQW2a6LQ8VGgAAqUBqWk4bAbDEgoNCALJNtJlKDzfZaZ2bNFUJI9zc7d7c57b3ZmlTEGuw3f9x9HUXQjDEOXPMiL9ft99s/UTgDNZnOMAuYLhcL1XG4EAQUhSSC7KaZYLGBp6S3c7YIbjcYlDi6Xywfz+TxWvv8AsyeJQWISAjKICSwIAritViuI4zhLJpsGMtl3u93/JaPT6RJQggsXL8s/l4MnJw+j11sVdsOPYZVGjD+IE6XiGN68foWjlePCzmuigFE5+O68T9sUlKLZTuLZ1tfW8ODWKWH86L8Hq91/5ZpVwFKZlTcWS+PQWkOR6dT4nQFMYhkrMyfl3aRnoFkBfROAhuM4W0ynngcfHjP+9law0KtJWqIgTMujtILjukN28ZwCeVs5y7jw5RE21iNRIQA88YFwCsw4tWdE8rdD4edqlCqwjHfG7yEpWUAmFwCd5sn27ev2HeloRwBsL9hKDRVkMi7u3zwm5QnDCJubgTBksxlKw0j3aWXXYo5MyygKKK+Hy8vvzg4ahXzJ87wprk673Q5IXY5T47jK9AyOHDogivbtnZBm23IX6vX6bQK5Onv6zDnPK+Dli6d/qOZP6Hxm6f/0v13KRmufhwC1Wm2CSvZrbu48Rj2PNsRwHU2g1Y1qtTq6020dXiaS3iH7sLj4/MSg/1PGT7td97+G8aA4FJOt1wAAAABJRU5ErkJggg==',
computer_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKfSURBVDjLfVNNaBNREP52s0m0cc2PUaiWQLGKYtWKCkUPohJJFfTiIQcv2tKDCB56sAfPUjyUFKQHrS2e1NaCthapSFuxlFyMifYHRWkvDZHapj/aZLubXWfeJgV/X5g3u5uZ7/tm5j3Jsiz8byUSiS7DMC7puq6QB3lha2tr7B9J/wJIpVJeSujw+XwX3e4yaJQE04SIphy/34fBwRdQ/pacTCYjnBwMBnd4PB5Mz8yC2U3TgmmZBGShQGCapkGZmprSCoWCi0wElWSyz2azv0vGwkKWgEyEz9SJbwonV1SEsLi4JNgt/lms0hLGL8SJgN+L/r5n2Fd9SLDzf0IBo3LyjY4EhUmQJNrtTXi2leVl3G48KhjfJeJgteXnLtgKWCqzcqA/sAWyLEMik4vGzwxgmTZj9YEj4tkq9kBmBfROADIcDscvJhc9L24eM86p0+hcjImyhAK9OB5JluBQlHV24bkE8vbkbMbOj/fwY8UQKgQAb9wQLoEZK7eVifrtJSG9ZNgHauYN5ryTqJoPIpfPYWi8V0xFAMjFOtm+fPrzXGz2TmAmN4vwyeOoCOzC8MRTxMe7kdcCNoDTqaClvkaMR9cNrK5qQrLL5aQyLLQPP0DNnggKcgEHy8N4NdmD2v3H0P35ORSqq3VkZOhE6aCQD6iqWsnTyWQyGqlzf8t9hVPahLq99UJR0+m76P/QTo3sgxKNRptKUtva2q4TyPmdVbsrVdWH3icPTZr5/ez3+SsT6TEplR5Fc7gLLS8vY4PDTR2SLbmUHIvFthJbLBI5eyoUCkFVPWhoaNyYz+ejhqXfir8fg4uuDjO7JAWjydd0eszW9ctE0hfI3vb0PD5cOv/FezEZv5O+WXttu9I9PXAVGFApfIWsnb43/wTyL6VPiDb06AAAAABJRU5ErkJggg==',
computer_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKZSURBVDjLfVNLSFRhFP7uYx44XbyjFommzaKVEkVWRoFY9IYKLGgRLQpqKbhxEYSLiAipcVMLxXalWYuKCIVkiIQ2jaORUiRa+Jg0nNHJx/W+Oue/MwP2+i/nP/+995zv+85//l9yXRf/G/F4/IFlWRdM01TJg7ywtbU19l3SvwCGhoYKKaFD1/WzgUABDEqC40BEU044rKO39xXUvyUnEoljnFxSUlIWCoUwPjEFZnccF47rEJALm8AMw4A6Ojpq2LbtJxNBOZnsU6nU75IxP58iIAeHjxwX31ROLi+vQDq9INhdflxW6QrjF+JEUbgQL54/Q1X1TsHO/4QCRuXk5o44hUmQJJq9SXi2zOIibl/ZLRgH4+/AaktPnvYUsFRm5cBwUTFkWYZEJmeN1wzgOh5j9fYasXazeyCzAnonABmKoqwzOet58OYx45w2js50VJQlFJjZ9kiyBEVV8+zCcwnkvc55jJ2f2rGUsYQKAcATbwiXwIyRTQWifm9ImF6wxGp1oBs1g+2o7Z+FpRdjyeygrkgegJytk23s85/nojYTg//LR9RdakIgUoWV4T6MvH2NslTYA/D5VNy6vEO0xzQtLC8bQrLf76MyXCy0NmPbmUYEx2LAmxsoKNSxtaISk2MJqFTXnVisvy53UMgXaZoW4e4kk0mD1AVqf8wguDkCnGjKq1JbSqEZP7HuLrS1tTUSyKk9e/cd1DQdT588WqGeP9w/fO9ifcM5X+jbSxgrSSxTbGZRQWI8aOUBotHoRmrZbH39IWzQNDrKNlRFJtC7Sw0zj1v9WujalmJLVeVJZOYsTHxXbHPVvZ6/TCR9nux9T0/3rtz5z96LkQNdUy0D5ytWP3ydvqrYUqWtyFN0lO4f7bNu/gJT+aqduOCVCAAAAABJRU5ErkJggg==',
computer_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKqSURBVDjLfVNdSBRRFP7mZ2dz3dwfx3VZTVuRiizMwBAiJCixh34gEwLfiqCXegl66D2kB1mJygcjevO1FGKFUhMKI5fZNFYlV31wN7F2nZVk52dnuveuKxjaHc49d2bu953v3HMuZ9s2/jdisdgr0zR7DMMQiQfxzHRdp36I248gHo97CGDQ6/V2OZ0uaAQEywLbTTA+nxfR6DuIe4EVRemkYFmWa8rLy7G0vAoa3bJsWLZFiGwUCJmmaRATiYRWKBQkYmxTSSb12Wz2X8nIZLKEyMLFjkvsm0jBtbV12NhQWXSbPjZVaTOjLyQm/D4Pht++QdOJFhad/mMKKCsFPxyMkW0cOI7MxYl5apu5HJ7caYWPn0coNQrBfRJ/Pk9AthrATpZGpRt9/krwPA+OGL9tdE0JxM33aKsDyuQeeMIt2Fg6CjX6uqiAnAkh4CEIwi6Ckj8dmIXf5YA7eBaZxQQkTsfBiiA8laEiAS0Px3MQRHE3mKg6XKbgZhsHb/gCtPQQJBeHFWUOhilgJOmGSE+SHghNgScKwgEXy5+OKnxFZ2MCnsYryK8OgJdMONz1EPNJ5I/dQ3Z2tEjAb+dJbXGh2AuNkoLu6yDgawT8ArzDhJ4LY+3jNJKV3Qi4a4plpJPDIaL31ilWHsMwkVsZQ3WZhUDTZeg/ByFINvJqPdITX/AjeBuGUMFU0/RFUsu+8fEP7aVGsdem6tvPhGTDrkF65il8AQn5zCGkJmOYdV3FkYbjTGGwuoo126670N/ffz+kDUdu3H2J+aEHSC1/grOuGVtrqjG11ez4pTl3upQaacK5nbsQiUSqSBkjC8l16N9HEG4+B3X9N5LTCdsvV7Q+6n0e3+ve8KUFSSGjqur0t7kU+gaeYXIsihXODeXA+ZmOx/H4ftf9L53Qf7mz5LNnAAAAAElFTkSuQmCC',
computer_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKiSURBVDjLfVNdSFNhGH7Oz7Q5N3dshrKRGP4wFElMKALL7McIDDIKopuIorsuIqLLuopdxKSbIqmroKIuVCjtIoQk03BtF7FYyCxIUMmpyXQ7O+frfb9Nxai+8X7vztn3Ps/zvc87RQiB/61IJPI4l8udN01TpwzKMrLZLOenyr8AYrFYGRX0eb3e08XFJchQEWwb8jTVGIYXw8Ovof+tOBqNdnGxz+fzu1wuJKd/gNltW8AWNgEJWASWyWSgx+PxjGVZRRTy0LpMzqlU6k/JWFhIEZCNI0ePy3c6FwcCO7G4uCTZBX8EqxQy+IE4UW6UYXCgH41NLZKdf5MKGJWLb/RF6JgCRaE9v8nM8Wt5GaHLbZLxU+QDWG3ViZN5BSyVWfmgUb4dqqpCoVALwd9rSxNwzD0ErAo0NbdBUA9EoQcqK6BnAlChadqWUCmcDgtdu+Iwqvag2vMd8+4kHi2G5bWkArNgj6Iq0HR9g11mUtXsHkNdsB2eqhb4nc/w/GsM35bWpAoJwBs3hK/AjDU7SuT9eZViHkExCrfvHKzlAQRaLuDK5CvMVndTjS1dkQDMyI3imErkZ0GhvpypHUOg8xSwFsHEk37svXQTRStfUIwGAvBvKnA4dNy5uFvaY5o5pNNkz2oClaoHbiNN7Emy04a18hE1B68h/eY+RG5fXgF18u7IyNsD64NCudzjdtXUiSF428/CTo9T7SpaexpgZ2ewzZVAIHgYKz/fyWHb8l/o7e29SiDd++uVQ35DRWW9k1gnSG4Wky/jBBKEopUBzmMYfXALQzO1yQ2AcDhcQdbNdXR0Yu39dTT23IOmTpPUlJzPzUV2Oxsw+zmKxODtFxsAoVBII/njpKC1wzUM1TKlVTz4ishleHDWB6jwnt2b/A3aCqjQmVNP6QAAAABJRU5ErkJggg==',
computer_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKbSURBVDjLfVPfS1NhGH7O2THJubmtWUZrYNGFzELRsggLFSUJqosuxaCiy7rsHwiJiDYvCykvC68qIiSsXVgU1NTIZkpZmgulnL82d35+ve+3TTCqb7zfe87O9zzv8/74FCEE/rcSicQ9y7K6TNPUyIO8NMMw2N9X/kUwNjZWQYA+n893trS0DDqB4DiQpwnj9/swOPgU2t/Ao6OjJxgcDAZ3ud1uTH+dA0d3HAFHOEQkYBOZruvQksmkbtv2FjJ5qCiTfTqd/lMyFhfTROSgvaNT/qcxOBQKY2lpWUYX/BOsUkjjF4qJgL8Cjx89RKS2Xkbnb1IBszL4al+CjilQFNrzm/RsqysruHHpoIw4kngNVrvz5Om8ApbKUfmgP7ANqqpCIVMLxs9MIJx8xNoDjfJZFGqgsgJ6JwIVLpdrk6kFz+v2yHl88PZj3j2Fu0sxmZZUYBbao6gKXJq2EV16ToE8r5xlIhSswZP3NzFjeaUKScAbF4RT4IjV28tk/l+yF2E6VH3HQjBiodK7DzVVTVjNZZD+/oYIM7IrkkAt5Mn2eTI/C+F6A+2Rc7Cp77Zjg0cotTyL/aGjWDPWcfnZMTSY3XmCkhIN1y/UyfaYpoVsVkf/lCHB335NkRKaD1Jj2iZW9FXU7W7GmpnF0PodaFTJW/H48+PFQSEf8Hg81Tk7B8u2sMMbJrAtyX4szyBQXoXE7EuMp5Komz+DTXeht7f3CpGcOtR0pDU23o2coVMNDMrXwN5gBIf3dODtzCsMTwyj/mcntuqeiQ2CWCxWSS1baGlpQ7nHQ6NsQ3OpRBrNRKPR8tpraqYx3FoWnxyaMxy0pXrEJ8ZtXCaSvkj2bmDgQUNx/gv34iN/zxoCLybjWHfQvNAjpou43xdyww7T9h6SAAAAAElFTkSuQmCC',
computer_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKVSURBVDjLfVNLSFRRGP7umTs+xubpTJQO4bRxoUGRZS1CCCa0oghatkpo4aZN0LJttIhx1UZs0aZ0UxaKgSFC0KZxhnxRaGqjiU7eUZur93n6z5lR0Kxz+e93H+f/vu//zzkK5xz/G+l0+rlt23csy1IJQSjDNE2BL5V/EWSz2SAl9IRCoduVlT4YlATXhZxNOeFwCMPDQ1APS85kMu0iORqN1tfU1OD7/BKEuutyuNwlIg6HyAzDgDo9PW04jlNBISft2hSoadpBy1hf14jIRfJKh/ymiuR4/AQKhQ2pzsXFhUsuQ7yQJiLhIN4OvEFT8xmpLv5JB4JVJD/sSdM0BYpC99JNooitzU08uXdOKo6nP0G4PX7tZsmBsCpUxcRwpBaMMSgUrBziWRBwx0WD8xGJBEPeaQQv94AJB9QTImDweDz7gpVRjsUBtLREcDLZhWOBLJzVdMmBVV4ehSnwqOqeukRRAuGFQAZR308EG5MoLgwhGCAHc68R2vZCFSyiIaIEoZg46pP1l4aC5Q0bTZFlBE9dh6NPoioax46TQ92lJiQ3xkoErFyniNmvf++LhmgAljZPAnlyVERFIA/s6Ciu7JQIvF4VjztPy+WxLBu6bpArF9VWDuGtQXirXbj2JJhbAJgf3DIx0zeHd7k4VOrk09HRD227G4Uw4vf7E7XWFHyY4HUdtxRuvofibGFiUIfXKMJDJaqtD7CyOIJ9Z6G7u/s+kdw433rxcrzQi/qWNpj5Z1DVICZGdAxOxqCxGO0DG9s2xH6Y2TsLqVQqRkuWam+/iiN+P5heAcWzBE9lDFPDv35/GV/tetQ79uJgf/YIyPo6xef+/ldnRSmNVWto/rGAoqabudm1zru93/oOO3h/ANOqi32og/qlAAAAAElFTkSuQmCC',
computer_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKqSURBVDjLfVPRS1NhFP/du+suJLFN3VRcrQjZQxor7DECwSh6CHHgMgNB6kF871+IHtxeJTFiENFLVA/hy9yLYLWtLbSRFGbqbJtt624T57Z7O+fbEozqG+c7996d8/v9zjnfJxmGgf+tWCz2qFarjVerVYU8yAs7ODhg/1T6F0AikbBQwpzVavWq6jFUKAm6DhFNOTabFQsLr6H8LTkej1/l5I6Ojp7W1lasf90Gs+u6Ad3QCchAncAqlQqUZDJZqdfrZjIR9Fsm+3w+/6dk5HJ5AtIxdOWa+KZwstN5EoXCT8Fu8M9glYYwfiFOtNksePXyBc72nRfs/J9QwKicfG8uRmESJIn2xiY8W1HT8ODuRcH4PrYMVtt9/UZDAUtlVg60tbVDlmVIZHLT+JkBDL3B2HduQDwbzR7IrIDeCUCGyWQ6YnLT8+LmMWP2+DrmCwFRllBQbY5HkiWYFOWQXXgugXxjcg3G+U8PUS7WhAoBwBs3hEtgxhM2E7TUKtTKNroc7fjyLQ3VaEepvIf9/X2M1yeRKWWRiEcbTWQAuVmnVsjj848PGBrowejEbZTLZZzZ2oIRWsKb5SWMjY1BVVXs7OwgGo0ik8k0AFpaFNyf9CASiSCZNOHS4CBm/AF8XF1Fd3cX+vv7RZLf7xe+t9eNkZFhbGxsQCEZM+Fw6DIDLS4uOt1ut4PnefqUC5F3b+Hz+ejY2lAqlZDNZjE7O4tR3y3UqnuwWCw4che8Xu8zh8MxfGdqSnnyOEgXKQKFGsvHmReXxGOnGExPTyMYDNYPAQKBgF3TtAzL8ng86HE66VgB6XQaa2tronaXy4XOzk4GyW1ubsrhcPj74WWiEnI0iRjdvguhUAh2ux1ms1lMcHd3d4L8drFYfL6ystJCZaupVCpLim7+AjUfrZnK+fliAAAAAElFTkSuQmCC',
connect=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ+SURBVBgZlcFLSBRhAAfw/858s860aygiptIp9NChoDzkxYNUYIcO0TUIOghFUNElvEad6lBkdEqCIugi3QQfLR2yQumBaPh+7Li7o/uanfmes1972IOEQv5+Ma01DuPRh+U+StlEhSsZUnElprXGQd6kdomUsoOJaiflojXk4mIM+pZjaXCp8GslTwkOMDLlOVyoCamiXhkpVCOJRDyGpG2CCYlsgSPvh4TgACGVt21L97Y0meBCg0kNyiW28wHiJrC8VYAo0wsE+3g1vtRdquYeHyXHUfAldkohKJcIuUSjbWI5XYKXLQ5/fnk1RbDHyJTnSKHeCbF6SbVMGCteH5pxAk7cQLbAQZmAGbOQccuQZTqGGoK615M5woX6aRPdZTkn4a+7kehMmdOzMmptaDOTNkEuzxE3gaAcQITMQ42BugpVHUzIrqRjwCJVOA3nzPLvMzKScujPxnK04RbRdIQgYBxhIYSs0DRqDNSFnHUKIUG5xKZXQTweg5Potmyde9hz/quZ9RbgukWsLWQQlvxFFQkXNQbqKgFvDRhHyCRCKrC27cOxYmhrPksyP5rQMzAPd3FJZVdzoyrip+cn7yvUENSVQnajvclCSAUqlIMyCa8oYVsmoPsxM/pJRVVxam7ywTz2IKi5+WLmXqNjXI4TA5lCgIRtwjI1GqwYhJBY39hFLt0+NPtxcB7/IIPPvt9N2MaTRNwAZQKWqbGeLmFnxwf1GZhPwXz+RXH2HPsgPuVP25qT0DrCZtbHpltEwQuGlRBjEedexFVaCenOjd9R2Acp+RQb2xFMaKS3iiju+v3Tb69N4T8RGtBjK/lSRoWKKsYGvr2/nsIh/AUG0IfiieuuUQAAAABJRU5ErkJggg==',
contrast=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFESURBVBgZBcG9S9RxAAfg565DI51uCBocpGhoqM1VaAjByXAImvoXDtr6D4JAaG2oyXtpKJGEltYcGntDErEhEI3kvDP7fb+fnqcVAAAAQAeg39XLqsVcyl62bTw8AkTE5tqb8WHOU1MzzUFej1+uR4SIzeWPOcu/TPI7JznNecZ5ngcrEa3YnJ/7fHehY6Kqqiq+eedgP7cH4zZ6dxZmnamKoiqGnpjTXcxj2tSVq/4qGkXRGOlrfDcvK7TJ0qypoiiKob5G9cWsukSHoCiqamQgiiqKoE12p2YUxVBf0aiK6ybs0qbu/HJZMTRQFEWjuOFU3aFNnn06vLCnr1EURbHq1PF+ntIKXiz/+fDTFV/90HHNTWdOTO69fU8rYH0tr7rzc2YUF8aOx3m0NYJWAPe76VmttzK1bzsbW0dAKwAAAID/tYu/URIDsoEAAAAASUVORK5CYII=',
contrast_decrease=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJJSURBVDjLpVPNi1JRFD+Og58PAss3NSAUOS2SkJGQokCmlTCrERFBkP4GoU34HwwMBNLOTbPxaxvMclpqZkIDiokmWYih4uf7uE9v5z56j+e0aNGFwz333PP7na97TZRS+J+1e92Qz+dduKWQ+Bj3e7ibUdoo7/H8JpFI/DL6m4wZIPjEbrefBwIBp9vtBovFotolSYLhcAjVanW5WCxeJpPJkg5iBExyuVyoXC7T5XJJCSF0tVrRyWRCx+Mxnc1mVBRFimCayWRoPB4Pa7gdRoJgzuFwnPt8PjCbzTCfzwEJAIlAlmVAIIxGI6hUKtDr9WCz2byNxWIcw+78SSTl9/s9VqsVMAPmAOv1WhVNLxaLkE6nwel0gsvlYr15pROgU5jnebVW5qwoik7A9FKpxPqj6q1WCziOY+mHdQI8BFl0QRB0oCYssgZm2TQaDWC+qAf1MWqTMKbNhEUuFArqvWbXdA2jEVQw+lM2NlaGEayRPTv1A5EVbCqBvtgCaELF2IOLwWAANptNT5uBjaVIIoHQ3SM4uv8ChJUa5MLYg7Narfadjazdbus1bxEIEkiKBIIigs1kA+Hx9HLrJWaz2dB0Or3s9/uAbwKazSZ0Oh3wJj1g5Xdh78Zt8O4/AHktQ/fHN6jWP4qSRD5sPeVoNHqC53c4Z47Nm/VE2Sjw09uFW+6bcOA5UAlqV3X42m63iEQema7/xkgk4kJbCtVjrPMh6gLqXfHJbH/vDs+zJmKZ9U+nXw7/+kz/WsHXh2UiEvL57Oq5ZvsNyKLICFmyXRwAAAAASUVORK5CYII=',
contrast_high=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFFSURBVCjPhZG/K8RxHMZfdy5OnEEMhhuMJCMldVl0JcOV7Bb2K/+CRakbTOdOsSjWK2WRZMDiR1eK7sJEuYsv98P3/XkM93Uj72d51+vp6aknJP6+yO+z3U/azTGsDt2rQGb5NQBCiHxqz3tUU5LUUFm73uZCi7Rw4lQ1OdVVVUUfaupLa5pJChESud5YcT4e4RPD4TDEDfs8lNz4iRcG0pPxKB7WVp4VYgwMaxXC4JJD1PHb2ibLN7f0oSSEQROdfGL4GEaOLN84rojiJiACAgwfhyPPFsIwhCFaCed1ujB8cmTbSSN4cA5hsMNnuvHJs4UFPYxRqthhK2Hj7KlBkWyADJ9FKryUtA4hAZnE23GZHq65I0KcMTxe8WYvjgIDTKe0M9gboxOjwTsvnpYuD6BtgKl+pZmzUdUoUVDmMhgr9N/cP2kRvjgCZVaBAAAAAElFTkSuQmCC',
contrast_increase=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJfSURBVDjLpVNNiFJRFD6O4j8EMpoMTGU6LUbCMJAmmFzYQpjViIgkSNtoldDOzUCbCIJA2rlpNv4thiCY5dSmNJOiYhwRJ8nIMdT8/3nq7ZxH7/GaoE33cbjn3ne+737nnHtljDH4n6E4vZFMJg04RZB4C2cLznK0MtoLXD8JhUI/pPEyqQIEb2s0ml2n06kzGo2gVCr5/clkAo1GA/L5/KDf798Oh8MZEUQEZIlEwp3NZtlgMGAcx7HhcMja7TZrtVqs2+2y8XjMEMxisRgLBoNeAbdEJAjWa7XaXbvdDnK5HHq9HiABIBFMp1NAIDSbTcjlclCtVmGxWDwNBAJ6wi79FhJxOByrKpUKUAEFwHw+503w0+k0RKNR0Ol0YDAYqDb3RQIM8ppMJj5XCp7NZiIB+ZlMhurD+6VSCfR6Pcn3igS4cNHpo9FIBApGJwtgUnN4eAgUi75LbKPQCalsMjo5lUrx/4V9wRcwAkEOT9+gtlEaUrBARgD7nYsglylg+GlIsJy0Bvv1eh3UarUom8DSVOx3rWBZs8DKeTN8t/Gd2JfW4HGhUPhKLSuXy2LOvGz8Lt9bgwuWc2BdRhKTFZbNBhhutm+5H1xTijcxHo+7O53OQa1WA7wTUCwWoVKpgC28CiqTAs6eMYNt5RJM51M4/vYF8u/fjicT7uUfV9nv92/j+hn2WU/9ppqQClJWXT8Cz1UPT3Dw+hWcnDQ28g8/vJGdfo0+n8+AexF0tzDPdfRH6B+Pbvy84rl+E7g5B3vP9+Ddo4+yvx7Tv8bmjouhZODGdL05+Bw74gl+AetZvIbkaCwtAAAAAElFTkSuQmCC',
contrast_low=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAE3SURBVCjPfZHNSgJhFIa/C3AldD2ugraCF2E2OdX6c0ZtQAj7nUJoUxZtKn+mwE20CoRW4/hTTukg3xUIugmCp8UMYhs5u/M+L+fwvgKxegRCCSWUcOKO1XSb08as4dat+lq4XQBOsjXvEqBQBHg8z2upJcBJvDFmwhgfn28CRtyxvxEBzVhLjQjwGfLBgD59XrEpBoVYCBgeE4YLucste1xSplBACESjHTCKxB5dbthFx8Im/44QiPqv4jPyelTZQSeLRgWTBTCgF8k6OttobHGBEQK19hgfjw7XkVcjg8kxRnji0XAZ0+Fq4c2Q5pwSufDJh9iT+uKFbCRuksamjAxkLArqPlHllAp5NHRy2BxSRK4vRZ1PmrMDTrA544gSciZT/7pQwowbluHmfuRUutKSy2Wtnj+jSVcdCo7izAAAAABJRU5ErkJggg==',
control_eject=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHtSURBVDjLpVPJqiJBEHwf1f/UH+DydBTtLxBRUAS9eFH04MGTINIHUQQRt3I5eHBfW20XFBVzKgq75TGPNwxTEFSTXREZmVn1QUQf/4M/Av1+X+r1ekq321U7nY7GGNNarZbabDaVer0u/SjAyTIns/V6TefzmR6Ph8DpdKLFYkG1Wo1Vq1X5W4EXWb9er4SF/XA4kK7rdLlcRAyilUpFL5fL8heBl21mkHe7HW23W1ouV7Tf72mz2RBcGSKqqrJCoSCZArxexThgkEejMbndbrLb7S+xpQDWYDCgbDarmAK8WSqUYVXTNJrNZoJos9nJ6fzFd5uIow/oBwTT6bRqCrTbbQ3Ngl0c/Px0CDKIgMPhJKvVKsqAi9vtRolEQjMF+JiEAOzj0Gq1Mi0jKxxNp1MBw0U8Hn8LNBoNFR1HGSAhKzICFotFwOVy0WQyEZMZDocUi8XeJfD5Kvj5fD5FBq/XS4qikMfjMXfERqMR3e93KpVKFIlE3k3kc5WKxSJDD7AMuxAdj8eCiKxIgG9OZhzSl4uUz+flXC6nY+Y4eDwehZv5fC4uEzJDhBP1YDAof3uVM5mMnEqlGC9JNA49Qc2YO788xInM7/fLPz6mZDIpRaNRJRwOq6FQSAsEAhonqT6fT+Hf0l9f47/iN5+1McdPrPQwAAAAAElFTkSuQmCC',
control_eject_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJpSURBVDjLpVNLbxJRFO4P8BfwO2Dp0g07q42iGxM3JmxMXDR15aRWjdYqZMYSxbRpbaCSoEnVNjW2Ip2CNE2BUsrwELBAoTCMMDAwQDnOucCIj52TfJnk5nyPc+65IwAwMoxzs6BRoFOgV2DoQ98/0/xZP0w8o0B7+03F6Nw+Yrz+GLt/cMgHQ2F+azfCvnYlmXFH2Yg1WPubQJ981u7OUcHDBJcvVUCUWtA+7RJU6zJkC2Xw7Ia4ufUEhbUDkYGAFsmhyDex0WwDfpLcgbLYAl5BrdkhZ6Ikw5YvIFpXoyiiJQLYF8ZG5wG5WJGh8EOGo1IDStUW5IQmZMtNVWR908PdXMpjOxoU0DnZ7wzGHiZHs3W4/GgHLtx1qWII/CLJY6AdXga5KKD37EVZ7Bmj5gUZkgWJEEcnXTB236v8P5PzdLEBlXobTgQJ5pffschFAUNg/4DHYWFcLLw4xRIyEhFj97ZhlNogbWAKuX0KS7ZlHrlEwB8MEQGMj0UZvqFGRldMlMj30EvRgvmFV6qA/stOmBXEJlSlNiERZ8URcf7OJ4JLD30QP66Tm4llBKCtNrUFnW0jzqSzJeh2gThcmfHD1ScBMDzeU/94hoNtdbqw+TUMlOWDOkTNLXvRuOH2cXylN+VBXHSM5eqEGMnWiEFccTc/X+SuW2K9axwsEv02SK18dIv5cq9QqLVJmtSJRJYJnWOZMpisdnGCXv21SMOrPG3zUYuOFc4biEM6XyEzwZ7DKR7W3EGYpue4cfP7v1d5+DHdeJkyTr5YY2ZmF1j6mYV/amb4ByYrO2FyMteYyL8f0/8855/NvQk/FX0nyAAAAABJRU5ErkJggg==',
control_end=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH/SURBVDjLpZPLiupAEIbnofJOeQCjoqusJjoLEQVF0I0bRVEXrgSRFnMUQcRbe1m48H6NGi8oKtbpasbIcIY5HE6gSOj0/9VfVd1vAPD2P/HHQq/XE7rdrtzpdEi73dYopVqz2SSNRkOu1WrCjwAmFpmYrlYrOJ1OcL/feRyPR5jP51CtVmmlUhG/BXyK9cvlAvjge7/fg67rcD6f+RpCy+WyXiqVxC+AT9v0KVbVX7DZbEDTNB7r9RrQ1RNCCKG5XE4wAKxe+blhu92C1WqFQqFgiM1mCzidH9wNPv1+H9LptGwAWLMIktEqiiwWKwsL5PN5WC6XDGDmgN1ux/uB7uLxODEArVZLw2bhBgRgRgy73Q6JRAIkyQwOh5O7WywWcL1eIRKJaAaAjYkDcAOWgmKbzcYzJ5NJMJkkUBQH/4/TQBfhcPgFqNfrBDuOZWAGFGNWdDSdThnAxAFoHfcNBgMIhUKvEth85fF4DI/Hg2eQJImLJ5MJD0VR4P1d4eXdbjcoFosQCAReTWRzFVRVpSh6TgKFCB2NRsY4UTwcDlFMWQhfDlI2mxUzmYyOY0Mnh8OBu5nNZnx8KEbrTKh7vV7x26OcSqXEWCxGWUm8duwJ1oxzZ4cHmJC63W7xx8sUjUaFYDAo+/1+4vP5NI/HozERcblcMvsW/nob/zV+A0hzMNxKeHjMAAAAAElFTkSuQmCC',
control_end_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJzSURBVDjLpZNLbxJRFMf7AfwEfA5YunTDTnwk3RlfMWFj4qKpKyfV2EVjhMwAUUyb1gYqCdWgQmqkRToFaRoLBcoUEGiB8hoQpgzPcpxzgREfO2/y39yc/++87p0CgKlJXTKCQpJKklrS9Ejq0Z3iz/hJ4wVJyofrda1954Tx78fZg8ghHwpH+e29GPvGk2JmbFUtxmDsb4CR+aLVm6dCh0muUKmDIHahdz4gajQ7kCtWwbcX5hY3khTGjiFjgBLN4dh3odXuAR6x04eq0AVe0lm7T+4EsQPbgaBgdh4hREkA2BeWjZnHZsduCYo/OlCoDZWvtSFXbcuQjU0fd3+1gO0oEKCys8cMlo2nXO/A1SdeeBcoymbNnAfuGiOkGjyx1CnQNj+DXgSofd+OWOwZS0XTlcdeSR5Y9xchy7ckwBYBVBpdqDd7UKqJsLT2nkUvAqaDBxEeh4UBCMCMqOvzfmCcGdBQW3DHECbVnVRa0Omdw6pljUcvAeyHwgSAAdgrmq893SGZDa5juPzIDbeZISBTbklVdGFp+bUMUH/ZjbI1oQ0NsUcyELOUFStKFUUJ8JkAcLC4mXi2BrTZIregsrgTTCZXgcEASAYN5SbmZEEkQvNNOkza6/YHsPk1CpTpozxExQNrWev2Bji+3pI3gcbEaRPi+aa8TjQnpOz6FyvcLVN8uMbxQ6LfhijHJ69QqJ6RSmpnPVJNuiSS9aE5nq2CzmwVZmnnr4c0+ZQXLAFqxebg/MEEZAp1MhPsOZrmweUNwQK9yM3oP/z9lCc/071Xae3cSxfzzLjM0gYT/1zP8PM6MzurszM3mNi/P9P/fOefb4UIeuRftTUAAAAASUVORK5CYII=',
control_equalizer=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFCSURBVCjPfZHNSoJBFIa9qLmMiNbdwGwKClq0ieYGCvGjnw8Kyk1RixZtCiJmEUkQUtn4s3Bh/pZO+pkfSYpvr6OEGMXhwHDOM+9550wEkf/j55ATWZXRaWtsSj+pBzEF5GTW1BGiz+ighqS5lxMA20EXQBdtBPjkKcRdkJBjgOJm2G4iDss4RN0h2lwLB2RU3bUbWKP4FuawgVdW8jhXDkjrkLIWb1DIYJm5SLCDd5xqBzzbPloEalhnHmAGPirU+MKRdUCKQJMlD7MUL2IFZcZQIz4CHnWblgyWkMUCCg4o8T0F7I9GJFUJA1Tpfh67VFhlvqCHW/gjkwlxY1p0XUaMN4u44N0BEd/4YryoK3kZNFj84OQqV9Uj4geenFj1mTwxSRoNOTsPDc9E5dRnHYs9taO3bcxG9aaKiV+/+Vd8AypJdaR1UheDAAAAAElFTkSuQmCC',
control_equalizer_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKOSURBVDjLpVPbThNRFOUD/IJ+B6gvJr740mg0JsQUX0yMiUl9MJJIwGBsMOgDQSEz0mANhEIKoqhBBTEpahm5SWxLKTO9WBAKBTqtw9DpDbqcfUprvbw5ycpJTtZae+09+1QBqKrEqS4YdNToMOowHcJ4eGf4k18pPKKjuumFYh75vMbPuIPC4tKy7PX55akFUXj6McI3DCfMxCHubwaH4hODrg2LdzksxeIKVC2H/EGBYTeVRXQrgekFn9QzEbYQt2RSMqgmsU/8pqYzedCnZfeRUHOQdexl9tmdqmUxNedRbWMBMqlmBtQXxabKJfGOksW952HEklmG1mdhRBOZssnE5LR0fSBG7RjIoGZE+M5T7JJ4I5mBqW0Bqztp3OyTcLyuBde6/ViLpxlHjGyCG57hSUsGxumvAYF6pqhUcV1O4+IDD+aDCs7dfsfOM02jzFBJ5bGd1NA79FogLRmYPItLMg0rvptjBkSsa3ez864e/2htM24NBBDZ0liKbP4AA44hmbTMwO31MQOKT6T6XhHHLtxhsYMbKZy3OBGOaQzFFDn09vWXDYyf5v1CUs1gV8tjNvADZ5vH8SWk4HTjK4jRvbJBaDPF/kxwPQnO5ii3UONwhvjVaByFArCyrbGpn7zCoak/wBLUtgrsDERTyO0XMDnrh8X6tjxEQ/3gjtnpmpNkpThlqnajR2QVSWj/EGVJqEBIr97ZbZcuW4PF31haJO6l1zL63qXGEkVici/P+qVEtExUObieQIdtUG3kxn4tUuUqtznmLPbhUWnGE8JqTGEzoZ79KzLGXV60cT1SQ+ebv1e58jFdfbJibnk8zrd39QncI6v8sJOX73fYhMaOEf4SL/77Mf3Pc/4JB00Hw5wBy+AAAAAASUVORK5CYII=',
control_fastforward=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHxSURBVDjLpVPJiipBEJyP6n/q+7TL6Km/QERBEfTiRdGDB0+CSB1EEUTcyuWg4r622i4oKuZUFKMyvGEej9cQdFeTEZmRmfVGRG//gz9+tNttpdVq6c1mkzUaDYNzbtRqNVatVvVyuaz8KiDIqiDz5XJJx+ORbrebxOFwoNlsRqVSiReLRfVHgS+yeT6fCQ/eu92OTNOk0+kk/0G0UCiY+Xxe/SbwVTZ/kEejEa3XazIMQ2IwGBCqeogwxngmk1GeAsKv/gjYbDZkt9sFaSjJq9WKrFYb9Xo9ms/nMqbT6VAymdSfAqJZDMooFSSbzS5go36/T4vFQghYJbrdruwHqovH4+wpUK/XDTRru91KAWQEnE4nDYdDslisAhb6+HDI8+VyoUgkYjwFxJikAMqHFZAdDofMClFNs0hbmqbJM6oIh8MvgUqlwtBx2IBPkJEVwZPJ5ElGAsTBWigUelkQ89XR+fv9LueNckEej8cS7++a9I1pXK9XyuVyFAgEXk0Uc1Wy2SwH6TEJECEKzyAjKxJARJC5gPJtkdLptJpKpUyMDYH7/V5WM51O5TIhM0QE0fR6veqPq5xIJNRYLMaFJekdPYFnzF0sDwkid7vd6q+XKRqNKsFgUPf7/czn8xkej8cQJOZyuXTxrfz1Nv4rPgFlRDELuo2JKwAAAABJRU5ErkJggg==',
control_fastforward_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJySURBVDjLpVNLbxJRFO4P8BfwO2Dp0g27VmvSnYkbEzYmLpq6sZP66KIxQmYsUUyb1gYqCZrU2qamVJwOIKaxUKAMD4HyKBQGHAaGx1COcy4P8bHzJt/i3pzvO+c759wJAJgYx7Vl0KjQqdCrmBlAP3jT/Bk/TryiQnv/rWhwuDOM9zjGnYROhUAwLBweRbg3riQza68YMAZjfxMYkK/a2DwVOE3whbIIktwB5bJHUGu0IVesgOcoyK/sJSiMHYoMBbRIDka+S82WAnjkdhcqUgcEFfVWl7xJchsOfX7JshNFES0RQF9YNmYekuPnDSj+aEOh2kc014BcpTUS2Tvw8Hc3CmhHgwI6B3fGYNl4SmIbbjxiCQnJ+WoLphZcEMpIkCk3SUwkeQ603csgFwX0nm9RDj1jqUi6/pBV4YJIrg5ZoakKfCIInkkgNhS4qMqwuvmeQy4KzPhPQgI2q1zrEAHMiLi56IVYvgFTlCpAOWH6sZvc28olbFg3BeQSgeNAkAhg+egVydNP3CQrik7OO4mtyfl9chcbHVhdez0S0H/+GuaqUgtqskJ8EjLVJyeL8oiMCXAysWwVaIt1ZEFndcaZdK4MvR5AutQk5SI5UZAJJh/sk6lgYzvdHhx8CQNl/jBqouaerWRwsj5eEJujSSARx4mekYwNxQRxNbvpxTp/2xzrj3G4SPS7ALX1kZUKlX5gta6QalIXMlkmzBzLVsBosUlz9M6vRRpf5SWrj1q3b/FefxzSBZH0BD2HUwLssgFYolf4WdP236s8/pnuvEoZFl7uMk+X1zj6uVl4ZmKERaOFmzM6mFtM5N+f6X++80/jPghfk2d02wAAAABJRU5ErkJggg==',
control_pause=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHoSURBVDjLpVM9awJBEB2DKIoocoI2goXVgVhoYam9bep0XqX+DU1pZ6EhpZ2FYGsnot2BqJUgaiGCd/j9tZlZvc2ZjyZZGPZmb+bNe49dC2MM/rOsXw9Go1EcQdMYSYwIDbheryruLdwb0Wi0Z663mBlgs4J5TpIk2el0gt1u5+f7/R7W6zXMZrM+gpRisVhZNBEAxXA4VAaDgYbFjBbtuq4zTdPYbrfjZ5vNhnU6Ha3dbitGH2dwp/0eCoVkmrparQSweYjP54PtdgvdbpeYvKRSqd4TFWCSJtpGc6FQgEqlKpqLxVeoVt8AGQFJCwaDMrJJCxPJMPpBWhEMboPZ/ZvxbwMMG8Hr9VJtks4MBhGafjgcHigbAIRhnBGAy+WC8/kcEQBmrdR0+3402VxzPB7hdDqBmYFK5thstjsAExI+81szSV0ulwSgCoDL5dIigxwOBy+SJN+DBHLfYEE10+mUJLSEiai9MR6Pn1Gb7Pf7QVEy4p7QBcpkbrnb7Yb5fA6qqvYxbQgGiUSih66W8JLoRM/j8YDVauWm0k5BzXgTodls6uhBCUF7365yvV5XUFsuHA7LgUCAA6E8WCwWMJlM+GTMS/l8vvzjW6BVq9XiSDuNQPwxoVZuMkYLmTSy2ezvj+kv6wM24KX3CYk6PAAAAABJRU5ErkJggg==',
control_pause_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJjSURBVDjLpZNLbxJhFIb7A/wF/A5YunTDzuqmO+PdsDFx0dSVpJq4aIyQmUIU06a1gUqCJlXb1Ehb6QhiGgsdKFNAwAIOhWEcBobLUI5zPpgRLzsneRfz5bzPec93mQCAiXGdc4FJk0WTVdPUSNbRmunP+nHjGU3muy8lW+DjMR3ZTzMHiUMhziaF3b0U82InR0/76zaswdrfACPzWV+obI8fZjm+JoGs9EA9HRA1Wl0oVeoQ3mO5hc2sHWt1iA4wo5lNfZXbHRXwU7p9qMs9EDQ1O32yJitd2I3GZM/6EULMBIBzYWzsrJurUhcqP7rAi0OVxQ6U6h0DsrkV5m6v8DiOCQGWAPONxti6+eKDENxyJw3z5OwOXHclSBr8UrnvQPkjNHoRYA1/OWJwZoyKpgv3EZAgXYtCWwNsE0Ct0QOppcKJqMDi6msGvQiYih0kBNwsLEAAdryhAY5rbaJJ+zZcm2dJOvzvqqew4l0V0EsA+3GWALAAu+qRsbhQbcP5e0G4Sg8B+C+1erC49NwAWD98TjKi3IGGog47ksgs5E8UyFUUDfCeAHBj8WTSRREoj9cYweINZuhCqQaDAZAOGBeV5RUiNF+mWDJerz+ArU9JsLvfGptouuOr2oKhKCdIbeMkdKXLLeM40ZzRujufLHNX3OnhMeoXiXoVt6+9C8l8vUmSiE2VpMEx8PjQnC7WweHxyTPU+q+LNH6V57xR+7J/jYvEMlDgJbInOHMyL8BGKA5z1AI37Xzz91Uef0w3n+Vts0836EeuJYaadwuPnbTw0OFhZhwB+hKd+vdj+p/n/BMZPwxzcSL1lgAAAABJRU5ErkJggg==',
control_play=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHiSURBVDjLpZPLquJAEIbPQ+Wd8gAZnTirPIGIgiLoxo2iCxeuBJGgoggq3trLwoX3a9R4QVGxpv/mJCJzOMMwDUVCur+/qv7qfBDRx//EHx/6/b7U6/W0brerdzodgzFmtFotvdlsavV6XfpWgMMyh9l6vabz+UyPx0PE6XSixWJBtVqNVSoV+UuBT9i8Xq+EhefhcCDTNOlyuYhvEC2Xy2apVJLfBD7LZha82+1ou91SPp8nwzBos9kQqrJEdF1n2WxWsgV4v5p1wIIBOp0/KZfLCXi5XIrAGgwGlEqlNFuAm6VDGaUCtLI6HE4RPKOA4QP8QIJEIqHbAu1224BZ+/1ewMi4Wq047BDhcv2iarVKs9lMCN1uN4pGo4YtwMckBFC+BeMgYFV1kaL8EHvT6dSuIhKJvAQajYYOx9GG1SsOqqr6Bk8mEzGZ4XBI4XD41QKfr4bN5/MpwPl8LspVFEXA2BuPxzQajeh+v1OxWKRgMPgykc9VKhQKDB5gIRsCsAUiKxLgncOMh/R2kTKZjJxOp024j4PH49GuBpcJmSHCQdPn88lfXuVkMinH43HGWxItwBP0jLljlBxkHo9H/vZnisViUigU0gKBgO73+w2v12twSHe73Rp/l/76N/5r/AZGRj/URbdFDAAAAABJRU5ErkJggg==',
control_play_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJfSURBVDjLpZNNbxJRFIb7A/wF/A5YunTDrpouujNxY8LGxEVTVyU11UVjCmEsUUyb1gYqEWuqtqmRWukUimksH6UMHwIW6FCYwWFgYBjKcc6FGam68ybvZuY87/m4544BwNiobiyCQZVJlVnV5FDm4TfDn/Gj4DVVxgdvBIvv4IwKHafp2MkpF40nuP2jJP1qL0dNeXkLxmDsFYMhfN0TKFujp1mGrQkgSl1QLvtEjZYMpQoPwaM4s7STtWKsZqIZGBGOJ7+L7Y4CeCS5B7zYBU5Vs9Mj30RJhv1wRHRtpdDESAywLywbM2twVZCh8lOGt+EKsHUZyvUOlPiObrKzG2TurbHYjgENTD76B4Vlj8II3noYgI3DCoHPam0iPMncOTi8IQpZNDAHv6Vo7BlLRVDLenN2j+h1iCVwodoGoaXARV2C5fV3NLJoMBmJnXA4rFqjS2DMWOTaKvyZaOJRCPwxDnIViRjJyiWsudc5ZInBcTRODLB8DcZAAs8dwPiMn/zLstKwii4sr7zUDcxfviboutiBhqTovWLgxBx9Bc6ct8jNpIt1cLjcegsmtz9DFUo16PeBgPkLiZQ7PvOJwAimyy1IlVrQ7fVh9zABVucHfYiG+56qxR8IM5wwmDJmQyGsgclSkyTIqNntz1aZO8704Bq1RXJsRK2bHwMiyw8C601FrwaXCTOnizzYXB5x2rH1e5FGV3neHbauejeZUCQDBVYgM8GeE3kOtgNRmHcsMVP293+v8uhjuvsib5l9vk09WVyhHU+d3IKd4h7bXPS0zUfdppL/fkz/85x/AR14FVfMwp4lAAAAAElFTkSuQmCC',
control_repeat=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAE4SURBVCjPfZFNS4JBFIX9UfOfpl306XZ+QYhG9UJtiihq4cJNQdQsIulLKhs/CAlNy9JRX+slKel0nFdCjOJsLvc8c+6dmQgi/+unKIqCyuucNTarb9SVGAOKsmAaCNCn3lBHxlzIEYC23wPQQxc+3lkFOPXTcggw3AzsNlqwVBMNh2hzKByQV4NGClGneWoOK+yUkFIOyOmAsVFcurMveMYZZrlJC7vaAbe2jw6B0HxEDPeYYfWBLeuALIE2g8/ZrNKexhGBOjPWQ+Bad7lSEhNs7tGewiQWeZ8y1sIRGVXFF80nxtewSeAOFXziBF64ZFocmw63rlFVPGADRR6owDOeGD7Ugdz3m2y+uhyfp8vw/IQceeqk3DEZjgg4uwSNhInJsc/aFqtqWS/ZuI3pBRUXv37zL30Ddxx1NEzXzZIAAAAASUVORK5CYII=',
control_repeat_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKASURBVDjLpVNLTxNRGOUH+Av6O2Dp0oVdGEE37EzcmHShiRoCMWqDJiQSI6QjREtAHilQU0lQ3hawjEUM2pZSOn1QHi0M0GmdDp2+oMf5LnSsj503OZubc853vu9+twZATTUudcGgoU6DUUPjOYznd4Y/+dXCCxpqW97JJsfnXW7ZE+bX1jcknz8gLa0G+dHFGNdkT5mIQ9zfDM7FF4dde2bfRlQQkzIUtYjSaZkhky0gcZCCe9Uv9M5EzcStmFQMaknsD24quXwJdNTCCVJKEZKG4/wJu1PUApZWvIp1MkQmtcyA+qLYVLkiPpILOPhRgJg+w146j0Qqr5vMzLuFO0MitWMggzoHv8NRbDo9c3Fcf+rSce3JIkND6wJaBkOME4ztw2Jf5khLBkb39xBPPVNUEjnXJL1qXMphN5nDrCeJBrMTcraEw7SKvpH3PGnJoNG7ti7RsJKZIjOoFsYOVNzu2cD6roL6xx/ZXaF0iiHbiERaZuDx+ZkB9U5x57xJRozsZ5m4/tEs3rpFZrB9lNNSFNHXP6gbGD99DfBpJY+MWkLX9A4u3xtlxIGFBBNffTiNKw8mcP9NkL1MOJ6GxWrTW6izOSPcdiKJchlMuHWosuhRUcWzsRgz8MQyCCWyKJ6UMf8lAHP3hD5Ew93hI5PTtSJIco5NmYQEaiG8l0WbYxPfojIrENGqd74aEG52h8+esbJIljGfeXzWpYipY0ZMH5f0NLRMVDkcT6HDOqw0WyZ/LVL1KrfbVswD9nFh2RvBtiizmVDPgS0JUy4f2i29QlPnh79Xufoz3erZMrW+nuKed/Xzlpfd0otOTmrrsPLNHQ7uBhf892f6n+/8E/bIBuJgfmmXAAAAAElFTkSuQmCC',
control_rewind=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH4SURBVDjLpZPJiiJBEIb7oeqd6qQgoiNaavoCIgqKoBcvih48eBJE8iCKIOKWLgcP7mup5YKiYnRG0l2OdNPNMAlBlZn+X/wRkfUGAG//E182er2e1O12SafToe12W2WMqc1mkzYaDVKr1aQfAVwsczFbrVZwOp3gfr+LOB6PMJ/PoVqtskqlIn8L+BBrl8sFcOFzv9+DpmlwPp/FHkLL5bJWKpXkF8CHbfYp3m63sFgsQFVVEev1GqbTqQ6hlLJ8Pi/pAF4vQdt/i51O54tYURSxj6vf70MmkyE6gDeLIhmtYq1W6x8BQOhkMgGj0SgAeIb92Gw2kEqlqA5otVoqNgsPLBYLD6sQYGaDwcDDCA6HQ/xGF9frFeLxuKoD+JgEYLfbgdtNwGw2CwACbTYbmEwmsNvtws2ni1gs9gTU63WKHccylssluFxuXoILZrOZECBMUZwwHo/FZAaDAUSj0WcJfL4EDx+PhxCgTY/HIzJioHVCCAyHQ7jdblAsFiEcDj+byOcqFQoFhiXgQhFmR+hoNBJCfGICfOdixkN6uUi5XE7OZrMajg3/eDgchBsE4WXCzGidC7VAICB/e5XT6bScTCYZL0nYxp5gzTh3fnmAC5nP55N//JgSiYQUiURIKBSiwWBQ9fv9KhdRr9dL+Lv069f4r/EO4RAxpm00KCQAAAAASUVORK5CYII=',
control_rewind_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ7SURBVDjLpZNNbxJRFIb7A/wF/A5YunRDovEjalEXJm5M2Ji4aFoTI6kxjcVaCGOJYtq0NlBJ0AS0tcbWSkeQhhQopQwfAhYotAwIAwPDUI5zLh9SdedN3kzmznmfc86dc4cAYGhQZ2ZAJkkhSSlJ1ZWyuyf7M37QeEqSfOxNWW37uk+5fVF6Z3ePDQRD7KY3TL/eSFAj1qIaYzD2BKBrPm1xZjWBvTiTK5SB45sgHreJKjUBMvkiuLxBZnY1rsHYHqQHkKM5GP7O1Rsi4OKFFhS5JrCSqo0W2eN4ATY9fs60HEGInACwLywbM/fMR2UB9gt1yJUEomypAYk834esrruYO4s5bEeGAIWN/kFh2YNmldZ7wjw8uUX2cYUTB2Cwuin0IkDp2o7Q2DOWmjqqw6WHTgLIFBsQz/Fw7p6DAPBbuSbCYYmHuSUHjV4EqPw7uyweVv6nABfHP0vaIAbMfHbMLskBVx97yDtWIYjHsGheYtFLAL5AkAAKlSZcm/LDhQefCACBlx/RcP7+B7gy4SbVdKpowtz8qz5A+WUrRJe4BlR4EdKs1P8Tn9TCNiQPOwaEDU96IXZQI38mmi6BwWTut6Awr8WoVKYA7TYQA5Z5YzpAMqKw9OtP/RDJ1KDZasP6txBojO/7hyi7azlSrzk9DFvunDKaMDtmjGZrxIhPTBCTsuufLzC3jNHOb+wNkuFtQGP/6ORyxSoJLFVFUg2CcJgwczRdBJ3Jwo0aln8P0uAoa80ezYLVzrj9MUjlyuRMsOdQkoUVZwC0hllmRP/u71EevEy3XybV4y9WqKmZedrwzMhO6yl2QmeiR3U26iYV/vdl+p/r/AvMhAk86cw6LgAAAABJRU5ErkJggg==',
control_start=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHuSURBVDjLpZM7iyJBFIXnR/V/6nADdaINOprWDUQUFEETE0VxDIwEkRJEEUR8lY/AwLeirbYPFBXv1ilmWmSHWZZtuHR1dZ2vzr236oWIXv4n/pjo9XpKt9vVOp0Oa7fbBufcaDabrNFoaLVaTfkWIMSqEPPlcknH45Fut5uMw+FA8/mcqtUqr1Qq6peAD7F5Pp8JD9673Y5M06TT6STnAC2Xy2apVFKfAB+2+acYDlarFRmGIQPjfD5vQRhjPJfLKRZA5KtBhAc7Ohyv5HL9ehLrupMWi4Vc0+/3KZ1OaxZAFIuBDKvb7VYAHBIAKGN5+mH/KQGoA+qxXq8pkUgwC9BqtQwUC+LNZkN2u4OcThclk0k5RgAwnU6li8vlQtFo1LAAok0SADHCZrNLwft76gkwmUwsF5FI5AGo1+sMFUcasGez2aRgNptJV5+A8XgsOzMYDCgcDj9SEP3V8PN+v8vC6bpOb2+63BEBVwAMh0O6Xq9ULBYpGAw+iij6qhQKBY7dsOCzfaPRSAaE+MYGGAsxF6E8HaRsNqtmMhkTbcPC/X4v80UaaC3AsC6Eps/nU788yqlUSo3H41ykJCuOmiBn9F0cHhJC7vF41G8vUywWU0KhkBYIBJjf7ze8Xq8hRMztdmtirPz1Nv5r/AZMKDFYuHxjqwAAAABJRU5ErkJggg==',
control_start_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJiSURBVDjLpZNLbxJRFMf7AfwEfA5YunTDwsTuujO+YsLGxEVTV066cdGYQmaAKKZNawOVSE2qltRIW+kI0jQWOlCGh0Bb6FAYcBgYGKAc51wegrrzJv9k5ub8f+dx750CgKlx3bCBTpNBk1HTzEDGwZ7uz/hx4zVN+icbksnz9YwJHiXZ4+iJGOFi4v5hnH2zl2Fm3RUTxmDsBGBgvu7yF6jISZoXyhLIShs6Vz2iWkOFfLECgUOOX9pOUxg7hAwBejRz8R9ys9UBXIrahYrcBlFTvdUle7Kiwn4oLDu2EgjREwD2hWVj5qE5X2lBodoCoaoS4fdGsDiCbO8E+EdrArajQ4DBw54yWDYuzDg9vwcPbNEJ8z2Gg7Nyk8TEMxdAu4MMehFgDHxPsNgzllquIWCXALCSt5r5JnVAALlSE6RGBy6rCiyvv2fRi4CZ8HFUxGGhuSSpME3twn0rB8xWjnyjEJApKqQKtXMFa851Eb0EcBThCADNqFtPfcRg9Z5OANKCMqiiDcsrr0cA45eDGFuVW1BTOlD8iYDPxJC9VPotDQCpiwY5meR5FWiHc9SCwelLMbl8GXo9IIPD4Dt0PyMKq8K9RL4B7W4Pdr7FgLJ/HA1R99hVMvn8IV6UmiRgeHzJQoMIjfiPCVJadsuLVf6uPdk/xuFFot9FqM1Pflmo1Elgtd4h/WIbeLQITp5XwOxwyXP01u+LNH6VF5whatW9yQfDKcgJEpkJ9hzLiuD1R2CBXuJnLR/+vsrjj+nhq6xp/qWXeW5bYWmrXVy0MOIzs4OdM3uY20z834/pf57zL6w/CRKnLd7XAAAAAElFTkSuQmCC',
control_stop=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAElSURBVCjPfZFNSwJBHMb9UPOd5pR9hUnvIQrFQl26FHXo0CmImEMoQUhl48vBw5qWtY461pKU9PTsrJRsFM9l2N/v/7IzOeT+z/ehI9qqpZvW2Ia+VdciI3Rk20SIsWBeMUTdXMkVgdjNAcwxg8MbTzEuXU0uBTY3CZ5gDMuMEHlFm3PhhZaKMviJAbo4UV5o6phtE7jO5FEkHnKTMY60F+7sAlPiiDDvhUc8UHrHvvVCg8KE+NnjNRQwYJIee6lwo2dcKZlbJCxgg7jP/wmxm46oqz4+WZE0Hnh4jx4+UEWQLlkTF2bKrX9gyIIeAhOI5UWdyVM34scX38exOkTgKnLlqo/loalzRMzZXWhUTElmHutA7KhtvWXLtqQ3VVn8es2/8gUo3nl2LXz6SAAAAABJRU5ErkJggg==',
control_stop_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJJSURBVDjLpZNNbxJRFIb7A/wF/A5YunRDovsmRk3cmLAxcdG0uiFuXDSmkBlLFNOmtYFKgibUtqlJG6UjiGksU0oZPgQs0KEwMw4Dw8dQjnPuMCNq48abvJub87zn4547BQBTk7q2CDZdDl1OXdNjOcd3tj/jJ8Eruuxzb2RX+NMpHT/MMUfHJwKbSgv7Bxnm9YciPRMSXRiDsb8ZjOGrwWjNzZ4UOL4pg6IOQLsYEbU6fajWRYgdpLilnYIbY00T08COcCrzTen2NMCj9ocgKgMQdLV7Q3KnqH3YTyQV/1YWTezEAPvCsjGzCTfkPtR/9IGXDNWkHlTFnmWysxfj7q/x2I4NDRxh5juNZf8LPm12ifBkimdAheI0smjgjH3NMtgzlmqCNx5tGnq4Abe9LIHLjS7IHQ3OJRWW1zcYZNFgOnl0LOCwmq0BgTEjgqbQoHSuQrGuEqO+dgFrgXUBWWJwyKaIAZaPcEXoWvD1uQjc8rBQ4FUio4oBLK+8sgycH7+kGUnpQUvVrF4xK4KomwuGQf6sQ14mV5GA8gesFhyB3TxdrjZhNAKSwSzXzIpgrtaBbLUDg+EI9j6nwe3btIZoexBsuHajCU6QjSlfBmaqbZIgr2f3Pl/l7vpyxjOai0S9Zd2R91GFF41Aqa1Z1eAyYeZcRQSPP6jMUlu/FmlylecDCfdqKMLFk3ko8zKZCfacLgmwHWVhnlriZrzv/l7lyc9072XJ9fjFNv10cYWhnvmEBS8tPPH4mVlPmL5DZy7/TP/znX8C6zgR9sd1gukAAAAASUVORK5CYII=',
controller=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLnZPNi1JhFMbvKtoHBa1atgmCtv0VrVq0aCkGCn6mYH47ip8IflAKhibpRke00BnnKiKoiKA7qSkF08FvvToak/f0ngu2qBYzXngu3Jf3+b3nPee5VCAQcPj9/ucAQB0iyufzPXS73Wd2u/3RQQB8Wa1Wiclkqms0mrsHAQwGwy21Wn2qUCjOxGLxHVyrVCpHpVKJpWmazeVy20wmQyeTyaf/BaAKhcIrkUh04XA4vhSLxTIxX5IHULMCDd+PkxCLxbaRSETxD6DVamUbjcavWq22LZfLMBqNgGEYuJgs4TxbhG9PHnManuQgGAyypOnv/wCazaat2+1yJ735pOCMy+USBuMFvPzIwosPAMW3xzDwemA+HHL78vk82Gy2Iw5APtZoms/nHGCv2WwGP4Zz6AwWsFgsYLVacUI47jUajTvS9GcUaQ6LgL/Ne3U6HSBVgtPpZFHT6ZSrst1ug1Kp/EolEokdUveGPWAymUA2m4V0Og1kD5AxX1osFo1er2fxGpvNBiQSCVDxeJzp9/tcWWjEcsfjMVSrVUilUth5IEYgo/6Md1apVDSu46FCoRCoaDR6gp1HIwLQ7PV6ezKZbMnj8YBoKZVKUzqd7h4C5HL5bZKVU4FAMOHz+U4qHA6/RiJOAgFIJvFmrp3EUCj0gMyVqdfr0Ov1YL1eg8vl2t0oyh6P5x2JKZAwAQkVNuznjQDkb7xPgnFuNpuvyHyvtFpt+bqA3zDZAQQexaeGAAAAAElFTkSuQmCC',
controller_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKJSURBVDjLnZPfS5NRGMffq/6BIMhuuvRGKLqooItuugiCoi4qiNDoh2RS6BCnkA6mG+3dD0bvHDXB0iTtwm1M4X39sQ0ZqAzJJYGUNWFuMje3vdtSW9u38xxaDOvKA9/3cA7P93Oe9znPEVwul8XpdN4CIBxGwsDAQL3dbp8zm80NhwLQx2Qyafr7+8O9vb3HDgXQ6/VHenp6ZrRa7Vx7e/tR2ltYWDDMz89X/H5/RVGUvcnJSb/b7b7wXwApEAi0tLW1bVksli/BYDDEzD/YACkT8OO7x42xsbG9kZER7T+ASCQiLy8v/1paWtoLhULY3t5GPp/HVlrFuhzEtzOnuJLTCgYHByus6G/+AlZWVsSNjQ1+0vMpLTeqqopEKofG9xXceQcEX3mQcEjIJpM8bnZ2FqIoGjiALYpkymazHFBVJpPBZjKLaCKHXC6HQqHARXCK7evrK7OiXxdYcSoEOGiuKhqNgmUJq9VaIe3s7PAs19bW0NXV9VWYmJgoE7VqqALS6TRkWYbP5wOLQbf52c+nL2/vP7RcRdOLy3grS9BoNBDGx8fz8Xicp0VGSjeVSmFxcRFer5cqj06xFboPTfCtSvgYV2Cdfoyb1gZca70IYXR0dJoqT0YCkNnhcMQ6OjrU5uZmMKl3DZfKnk82eD7bQUOceQDrzCOca6krCcPDw510Et0EAeh3WHvna+/6SvdpTK26UDu8EQlnW45DGBoaOsnuNR8OhxGLxVAsFmGz2cq1gPNPTuyalHswKI3cbJAbeQYMsMsDJEl6zdoUrJnAmgrsXezXAlig/oZYD7Nyn59MM63ZvsgD2GusY42xbjQaS+x+SzqdLnSw51mwkUmltP/MRtr/DeMW8yghqDBkAAAAAElFTkSuQmCC',
controller_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKUSURBVDjLnZNfSFNxFMdvaL0EQQQSQpQPQS9R0FPQQwUSPfhQBBb4EOTDIILYxnR7aBPmlPZPBptSDoxNUR9yLhduOrexBs6WpA+BlbnRmrL/f22lu99+58JiWE9e+F64h/P9/M495/y40dFRw8jIyH0A3GHEDQ8PXzCZTEt6vf7ioQD00mq1Uo1GE1GpVC2HAqjV6mNKpXJRLpcvSSSSUxRbXl4eCAaDvM/n4z0eT9XlcvkcDse1/wJIfr//sVgs3jEYDF8CgUCImXfZA1LO78PWrANTU1NVu90u/wewvr7uXl1d3V9ZWamGQiGkUimUSiXsZIrYdAfw7cplQckFD6xWK8+a/uovYG1tTReLxYSTnr2VC8ZisYjtdAEPJ3l0jQOBF7PYtpiRTyaFPK/XC51ONyAA2EeFTPl8XgDUlcvl8COZR3S7gEKhgHK5LIjglNvf319jTb/LsebwBDhorisajYJVCaPRyJOy2axQ5cbGBhQKxVduZmamRtS6oQ7IZDJwu92Ym5sDy4H96b3f3jst+cWbzfA/OIvPNjWkUim46enpUiKREMoiI5WbTqcRDofhdDqp8xgXd+Kj5Grtp0sP/tM8KpNivH9yCcauG+AmJiYWqPNkJACZLRZLXCaTFUUiEZiKztsnKrvMDFMH0HMS0LQh9fw63tw6XuBsNlsvnUSTIAD9DlvvUuOsWdk8/+E1Gp+c8jQozo2NjZ1jcy1FIhHE43FUKhUMDQ3VGgHu9qbvZesjgJmqMg5ZppioCSweFxLMZvNLtqZgywS2VGD34lcjINjZqgp3n99L9LQhKT+Kze4j8HY078+3NymEBHYbW9libA4ODu6x+e719fWFDu78u84zvezELSqbKiIzxf8AGfvxrzvUBvIAAAAASUVORK5CYII=',
controller_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALBSURBVDjLnZPfT1JhHMa56h9o66KrVqtZrWV60U0Xtf6DtjY3a6u7wi1TWIYusQKhBDIWwiY0S2IKOtCFS5DfsaGSTVZrhCAsROX3DzlgJE/vOY3WqivP9pyzc/Y+n+/7Pt/vYanVaplKpWoDwNqPWEqlskkul9ulUumZfQHo29DQEHdwcNA/MDBwaF8AgUBwgM/nL/B4PDuHwzlIf/P5fCKPx1N3OBx1i8VSNZvNDpPJdOG/AFpOp7Oju7t7SyaThVwul5eYKXKBVs7pwPqMCZOTk1WtVsv7BxAIBOZXVlZ+LC0tVb1eL1KpFEqlErYyRYTnXYi0NjNKWi3QaDR1Evqr34DV1VVJLBZjKvXP8RhjsVjEZrqAGxN1XHsDLOsUiJpvIZuIMOtsNhskEomIAZCXMm3K5/MMoKFcLoeNZB7RjW1EPV3IhAxIBDQMnF4rFAr3SOhXWCScOg3429xQZFmN1BctajtrCExfxXb0A7PLYDCI3t7eNZbRaNyjqQ1DA5DJZGB/N4HPxjZUU29RTciQDZthHW1HeaeESqUCLpcLll6vLyUSCWZbtLFQKCCdTmNx0QffdBfSwSlUvz2CW9yK75kpfHx5GVaDlDlKZ2cnWDqdzkonTxtpAG0eGRmJPxfcptZtfdhNvga1dhNu0TlUYj3IR+bgGL60w7nbkWWz2c9Y4+Pj92dnZ0F3ggbQZJVSUSKV/DkSXCXKRflr2y+FrpOjDCPmksP7tKWf6cLY2NgR0teS3+9HPB5HuVyG/sWdeswtw+6WkqleDrWTI7QwTyrCBrU5A/fjM9uuh6dPMcOgUChGyZiCDBPIUME9fLFOpcPYzdpQTZIMkoY/NI1a6RPiPi3sfScMDID8jYfJYITFYnGN9LfmenK+9l7UTHmEZylSiSKVKCf/JOV40EQRE2XjHadsPceohXtHPT8Bse8J0M1mZB4AAAAASUVORK5CYII=',
creditcards=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJHSURBVDjLfZJPa1NREMV/972ksYmx0bbUpzS1ValpKVQohgguRLuxIBRBcKMbwS/g2l026jfQpWtTcCEFN+Kii0AoSFNpQUirlFprXmybNi/v3nHxYv6Q6oFhLsydc+aeO+rWs8UX08nYkx7bigOIAGIQEcQImCCLMRgjFEuVt+9fzt+jgdC10fjT00PnAQukdbkra0H7PhcOardpQwgBRIEECjSUxAiiTaCsWyQ9Fqc6CB5dP8P4+DCfVnYZONVDtabb66SG4ywWfjCfcQBYWVEddUtEANjYOeTVYql5/hurm3vklrZY3dwj8EjofEIDNyb7AYhGbKIRm+RgL1++7bOxc8h8xuHnb4/joIrFoqRSKQCWl5epVCpEo1Fs2z62QUSoVqu1Uqn0oVAoPA8dbb9DTrwBI5TLs6TTaUKhEEop/gXP8yKO44waYx6HRPvQcL+vr49wOIy3vo4sLCC1GlYqhT19EWKrUPsKGKzIBM7Q7MTIyMhl++Gd/rM7h87M1i8bFbvCoFKobBZrdxe7XMZaW4OPS+iMjSVV0DVU/Tth26dcG7JVu6uFQkEmNjYglwtW0hgwhr25S8SvHoAyIBrEx05k+Lw9idVlkueB1uD7zYjnivh1C0w9CF0PyNu/sUkwNobSuqmO1uynz3HSPgDjNxp9IFi4rgnCU1N4yWRrAq2JztyEiANiAAO9w6iBue4JXNelrjXRbBY5OkI8DxWPE2zE3dbyKIXnebiu20mQz+cfGGNeJxKJmGVZ/A+u65LP5+//AbkTRxnEop0TAAAAAElFTkSuQmCC',
cross=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIhSURBVDjLlZPrThNRFIWJicmJz6BWiYbIkYDEG0JbBiitDQgm0PuFXqSAtKXtpE2hNuoPTXwSnwtExd6w0pl2OtPlrphKLSXhx07OZM769qy19wwAGLhM1ddC184+d18QMzoq3lfsD3LZ7Y3XbE5DL6Atzuyilc5Ciyd7IHVfgNcDYTQ2tvDr5crn6uLSvX+Av2Lk36FFpSVENDe3OxDZu8apO5rROJDLo30+Nlvj5RnTlVNAKs1aCVFr7b4BPn6Cls21AWgEQlz2+Dl1h7IdA+i97A/geP65WhbmrnZZ0GIJpr6OqZqYAd5/gJpKox4Mg7pD2YoC2b0/54rJQuJZdm6Izcgma4TW1WZ0h+y8BfbyJMwBmSxkjw+VObNanp5h/adwGhaTXF4NWbLj9gEONyCmUZmd10pGgf1/vwcgOT3tUQE0DdicwIod2EmSbwsKE1P8QoDkcHPJ5YESjgBJkYQpIEZ2KEB51Y6y3ojvY+P8XEDN7uKS0w0ltA7QGCWHCxSWWpwyaCeLy0BkA7UXyyg8fIzDoWHeBaDN4tQdSvAVdU1Aok+nsNTipIEVnkywo/FHatVkBoIhnFisOBoZxcGtQd4B0GYJNZsDSiAEadUBCkstPtN3Avs2Msa+Dt9XfxoFSNYF/Bh9gP0bOqHLAm2WUF1YQskwrVFYPWkf3h1iXwbvqGfFPSGW9Eah8HSS9fuZDnS32f71m8KFY7xs/QZyu6TH2+2+FAAAAABJRU5ErkJggg==',
css=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGeSURBVDjLxVNNS0JRED3qA0MRoqCFouBGN9Yia9HGRa3b9x/6M62jf9CuTS0EV0arXASG0gcUYWIgmvi6792P7sz1WUI7Fw0Mc70z59wz88aYMQbLWBxL2tIEXrN5+mcPWkvrBsZQVNYDSKmglLTZ0J4lwjCER8XZ7OYcSDMxRs/cEdCZSKKoNeUU7u/rjoBMiE8GuKQrcCA1A0XuFK2sZKwC3xE4Zo1UahX5/Dam0yH6/Q4KhV17H+Lu7gKVyiESCQ/dbgPD4QvfSykQlzKcMxP4+fnGJr4seAdPT01MJh8oFve4uNOp20fWQBilQqvAEtBQqE+6IBuPe3h8bML3hyiX95FOr6HXayOT2UCpdIDR6I1r6VF6KK61z5N1ROAkvdBuX+H6+oznksttodE4wevrLbdC8h1GwCMZJF+pgIdSrR6xtCCYWLnrnBuP31GrHfN5MHhgcDRUj3pzbAFarfOFSUf++4tEA3dRwhNCsKRkMv2r+Oe7R7+jvbArNotu/6wC3/Z7yX3TdhkjbDS8eUTi5EoGuLhosX//N34Dm6aVPfzbYjIAAAAASUVORK5CYII=',
css_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLpVNNaBNREP42Wck2MVUilJr+YJC2KG0kViGWHsTSS0HBg+DVq+hJkIJXD714E8FePHnKpdSDf4EebEJOBq1NSbUNiCYNRmPSmHSz+/Y587YpCB4KGfiYx5uZb76Z3adJKdGNedCldU2gp1KL/53BcWyChJTsBaEN2xYQwqaoRWcblmVB5+RweOKgkHcipbMPl4DPTNLxjsMxgY2NpEvAZpq7qsANuglukdgv6sBVZBhBUtByCVxmB37/cQwNnUezWUW5nMfw8EW6t7C2tozx8avwenVsbq6gWv2q7m3bhMe2rQNmLi4UMhTYo+IL2N5OodGoIBK5pJLz+SQ1CYFrhLBIARHwUnhOvmCr10vY2kqh1apibOwKAoEQSqUcgsE+jI7OoFb7rnK5KTfSHaelNusSQQWFEMjlXqmkWOwGBgaieJq4jy+/dlBv1rFn/sHZ/lPwCxM6y2D5QrTVUiYnbypp7XaD5J5QseX0c5S1Ci5PxzEYGsHK+hIyn97B+OlA59lcBW1ks4l/Nt3B6/dvcX1uDsIjcO7kLJK5BOITU0gsvYDHNE21UZ8voGAYAfWJGD09vaSiF5X6DxzRjuLambtqT/dmFnG6LwpBM5OCFs37Um2W/y4pTfJS/XncndX9JoL1YhofiquYn32GhTe3YHh98GoatMO8xvid8MPByLEHU9FpjPTH8Hkni/THVXwr1B5ph33ORLJA7jYhSNglPMk8Ls7/Be/8gsufCT5oAAAAAElFTkSuQmCC',
css_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIgSURBVDjLpVPPaxNBFP6SXZuYZWlsQbQhqRFsD1ahVgUPIuhBEDyKAU+ehB49FQTpQbwKggj+Cx6UgigloJQsqJcWCykppJVqTILi2iRms7szO86baVMFD4U8eMyP975vvvdmJiaEwCAWx4A2MIHpOM/+W0MUMekCQtDIpQdgjINzJqOhnDOEYQiTksfGTvWB1BMhoh3XBDQnkt0xiijGsbZW1ARkvt9WAB3UCRrEd0C7rhUlk7ZU4GkCzRwhlUojmz2DbtdFs1lBLndO7odYXV3A1NR1GIaJ9fW3cN0ttc+YjzhjYZ+ZwJub72WgJ8FnsbHhoNP5gXz+gkquVIrykBEQhvNQKpAE1BSqkzbIWq06qlUHnudicvIyLGsE9XoZtn0YExNXsL1dU7l0KB1kRpGnOquJoIKcc5TLb1TS9PQNZDKnsfToFuIrH+DXpPxhG73zl8BTPkySQfI5D1RTZmYKSloQdKTcURXbevEYhxpfcOLmLBL5k/A+LaJcKiJtSAVUm1YQYHn5+T+d7vvLJ7h4ew7J6jtg6QFSw2kcy42jUfoI0/d91dFEwvrr2vbundae+wvJI3ng2t29Fzh/FAc7ASnwZL2vVWfpdQnhy1Gol0ckpO546gC6K69gLczC9xroSoJ2y0DPGkJsP7+xVMjMD9nWvewoM834V7S/M3xuGjzsifux/X5np5Cb+/3z2x2Dx8a5IWoS9fTqInv4B7QMlwnqx2E+AAAAAElFTkSuQmCC',
css_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIhSURBVDjLpZM9SJVhFMd/Vy8pV40yIRHvBY0QUcHUqAgXo7YapKGm1oJ2scGWcI2mIBojh5YIgoYkl5sNkZGUaH6QkhejED/vfd/nfDS8emloEDxwOHCe8/9xnuc5J+XuHMYqOKQdGpDO55/89w5mgpnjLpgpZjEiiqoAAREhhEAaoKmpqyx0d9xtzxOAu2Gm5WhmuCszM28TAEAUbeHue4dJQSLSPdG+Jx1VV9cRQjEBJGQjkzlGNtvD7u46a2uz5HJnMQtMT7+is/MqlZVp5ubesb6+jFlAJKJCJJTJ2WwPS0sfECmRy/WxuJhne/s3LS0XuPfyGnef93Okug6RgGoghIgKVcFdUA0AbG4WWFjIUyyu09Y2QE1NPYXCNyINNDe0MzJ+m0iKmCkiJdJmRVSFBASqAVXl/uubuAViF2IRmo6fpr3xHFulHR5+fcTQmWFUI9IhRJgJqjEhFOntvUEIEc/mH3O54xbqhppiOKsbK3Q1X2Q7LjI8OcT16j7SItFeBzFTUy/K71GSGHXjx5/vBBPEAkEDm9EW3dl+tsMuT5fypKMowixQVVXzz7cZJSkhKpw8mkNMUTcKG8vU1zbyaSXPl5XPDNZ0kxobu+PuMSLJdLlHiDhvbIpSHBNbTEliTjV0cL71Ch+X3zM5P8FgbScnKjOkDrqNnQ8qdvpyA5mJufGfsXFpddRngYMDWkdSO2qVmaJp669RX9rP/wXZZaE2eCuTIAAAAABJRU5ErkJggg==',
css_valid=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAInSURBVDjLnZPPS1RRFMe/b3zk5DCVmlq+0ZBCoTEXWkEELmzhqiCocNWyTav+gfoX2gq2bhET/VgUIczKoVUGxthIKqX1tAmG+cGM7/56nXvezDCtDC9czpl7z+ec7zlznxOGIY6y5p/N3ZVKLcb+J/j+23uphczt6x3wLYKfe0Nne53DFBB8QQiZFkK8FFLNSynjBL86MzDY9X13t3qoAoJ7Cc6kBrwYwR8Ifn3OSzFM19POYu4xSzAmpG2gaRutQYFQStOW+H1Qhh9WcTKZRK1exy9/D+nEEI4hBjdtPAwPX2pXtC2FoWlu6yv2Pxe/YulLBk7o4MnVBzgdP4H19WW4LTAIqgxYFRGsyddtez7ej0cX71AFg+NaQWsJKRtRAgtYsKfnFEZGplGvl7C/X8Do6BU6l1hbe4PJyZuY6nKxsZFFqfSDz5UKELM92grGKIa3tz/SxQHBl7G1tYJa7Q/Gxq5xcKGwTEX6eC6RAkqgteI+7YFdlYqPzc0VNBolTEzMIZHog+/nkUwOYnz8Bsrlnxxri9pCMWMa0NyTTQS+tBXy+ffI5ZZ4Lp43hWz2KXZ2PnErVn7EBHCtDCtfa8FDmZlZYGlC1EhuP99VKnuYnX3IfrH4jeGobQ3X9hZlE1hdfdGeRyug85+wtjXwyCq4QRCwpO7uREewafqm/bv1LujJNW30CZCCBvX7jvtWys4hIGvfg+AkkVzFQGT/Xc5RP+fW+gsEgchGXj0/PQAAAABJRU5ErkJggg==',
cup=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAILSURBVDjLrVM7ixNhFB2LFKJV+v0L24nabIogtmItin9ALBS3tXNt3CWgVlpMsAgrWRexkCSTd0KimYS8Q94vsnlrikAec7z34hSibON+cJjXPeee79xvFADK/0C5UIFyubxLUEulklooFNR8Pn+Sy+VOstmsmslk1HQ6raZSqd2/BCqVyh4RW/V6HePxGJPJRDCdTuU6Go0EZ2dnIFEkk8lWIpHYEwEi24lszGYzjHptfPvsgvbuEJ9ePMPH548Epwf70N4f4fuXY6rpYDgcIh6PG7FYzM62dSav12spfHXn2rk4fbmPxWIhIpFIRFfIzk+v1wvDMLAhka9vD+B88gCv79lxdPeG4M39W/jw9KF8q+oJzOdz2VIoFPqhOJ3O7mAwwHK5xGazketqtRKws3+Bto1arYZgMFhTHA6HC78XW6P0wYJmcAy2y+9arRYoPCHTpOD3+w8Vm8122xTgQhobqtUqms0mGo0GeDLckdOnESIcDqPdbnN3aJp2VbFarTfN7kxmUqfTkSLuyM8syB3pLMj7fr8Pn883kTFaLJbr1EHfbrdilwm9Xg/dblfABNMF3/NWisUiKPjHIkDrMou43e4CF+m6LkUMU4idcFc+WJwRkbU/TiKtS4QrgUDgmGZrcEcelXkKORsWJ9sGkV3n/kzRaHSHgtrQjEGCHJSAyBuPx7Nz4X/jL/ZqKRurPTy6AAAAAElFTkSuQmCC',
cup_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJdSURBVDjLrVNLaFNBFB0XWYiusnPhwp0bCy6kumkWRVQQFF24kdqKH1QQQcWqq67S1mpaa1sFqeALLmqlbWgLxpY0/5BoXmrSkJT8P9IkbRKbRSC/49whQVERQQcOl5k359xz733DALB/AfuvAsFgsI1DCgQCkt/vl9bW1mZ9Pt+s1+uVVldXJY/HI7nd7rZfBNbX1zs4MR6JRLC5uYmtrS2BfD4vYi6XE8hkMuCicLlccafT2SEEOLmTkxuFQgG5dAIf56dgmBzGXP89vOu7KaAb6IXh1Qg+LU7zO0lks1k4HI6G3W7vJNsykavVqrioOX3oj9AN9qJUKgkRq9UqM25ne3l5GY1GAzUuon8+AO2dCxg/34mRM4cFJrqO4s3dHvEtJDtRLBZFSWaz+SvTarWpjY0NlMtl1Go1ESuVigA5+x142QiHwzCZTGE2Ojo6heYia7z7IMFW4whkl87i8Th48wSZTworKyvDTKVSHWsJ0EU+NoRCIcRiMUSjUdBkKKO08AK3J7pw5ckpdA8ch3ryAQwGw26mVCqPtLITmUjJZBKJREJkpP3rhXH0ve3BvG8Mni96aD5cxznNAbTf2DPEFApFO88g1+t1YZcI6XQaqVRKgIR6Bk9gzstH638qnA4tXYZm6SoJlBlfO0lkZmbGT1ZlWRb1ElpCJx8exKLvJX5cus9jJND8HRnbwbHLaDRO89k2qG4aVesvVN3ah0f6i1DruwVZ/b77u4OfH4fNZttrsVhqfMbggtQo3H92DWeH9uOx/pLITJH2ogd/++r45X6ObbLdjP10/g1HrwhirOEKWgAAAABJRU5ErkJggg==',
cup_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJtSURBVDjLrVNLSJRRFL4yziJqk6sWCbmIdraKCkKjaWglRBsJpIgKXFiLyLQ2EkGlEWkDFUG2mCFCrDGpUHFmnJfaTPo7zAOVeT9i3jNlNTGP/+ueWwOBEUFd+Lj89z/fd75zzr0MAPsXsP8qsLa21sqhXV1d1fp8Pq3X653weDwTbrdb63K5tCsrK9rl5eXWTQLr6+ttnBgJBoPIZrPI5XIC+Xxe7JlMRiCVSoGLwul0RhwOR5sQ4GQVJ8uFQgGZRBTvX4/BNDqMV7f78OL6RYHJwX6Yno5g6e04j4khnU5jcXFRXlhYUJFticiVSkUE3ju+74+YHOrHxsaGELHb7RLjdj4ZDAbIsowqF5l5NAjd5dN40KXCyIkDAg9PqfGs94z455ccKBaLoiSr1fqR6XS6eDKZRKlUQrVaFXu5XBYgZ78DLxuBQAAWiyXANBrNGH4ussa7DxKsN45AduksEomAN0+Q+aQwNzc3zNrb24/VBSiQjw1+vx/hcBihUAg0Gcq49HgAlq7dmFUpYezcCeOtbphMpm2sqanpYD07kYkUi8UQjUZFRvp2jd6Aq/cQSm/uQvZN4cvzS3D27MWMqvECUyqV+3kGqVarCbtESCQSiMfjAiRkPLkLXzkZ9zuAvu3AzRZkhg5j+qgiyPjaQiJ6vd5HViVJEvUS6kJkW156iV9XYWAHZo80yj+uI2MNHFvNZvM4n61MddOo6reQav785CzASd+uMOQ5It0KTKsV8U2PY35+vtlms1X5jMEFqVEw3+nBu/N78KGvBemrSgTONcDQ0VidUiuu/fWrs3U29/OMIbLN9xiR6fw750IHMReXp6cAAAAASUVORK5CYII=',
cup_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKcSURBVDjLrZNLTBNRFIbHBQujG1mauDEuTIysNOpCiDZKZINRSYwoLIhGE18LYxHjgigRVCINRlEjYqZqKChP8dVObQMptAhtp522MPZBQR6lLdLACG3n984NmBgjLvQkf+7m/t/955w5DADmX8T8V4DP58siYr1eLysIAut2u1tdLlcrz/Osw+Fg7XY7OzAwkPUbYGhoKJsYQ36/H9PT04hGo1SxWIyekUiEanJyEgQKm80Wslqt2RRAzCpiluPxOCJjI+jv1MFYX4O2SjVelZ+naq8qhfGpBp+7msmdMKamptDb2ytbLBaVEntQMSeTSXrx7sHtf1TjhW3wPj+K4bYK2J/sh7XtRoghcWYNBgNkWUaKQD7UVUF7qRj3j6ugObST6kHRPujv5GGcU2PG2QkkxhB3NsOiOSwzWq12dGJiApIkIZVK0XNxcZFKSUY/TWxCTCjHQrgd459uY55/jVSwG3zDGZmpra3VYakSiQRI96EAfzZO7EDMdQ3p+W7MfzmLqPUcPGwJnA2nYH7z4hmTk5OTuwxQmkPGBlEUEQwG4Xc0IsZfQVqyYE4shjRSiFmhDIGGfHS/I802GtcymZmZu5ZfV8yBQADhcBi8uR7jNvWSuQhS6Bi+ua5CfJiHsMsMjuOidIwZGRk7yCgH0+k0jawAhB4WwvvLJLYZUqAE30dOYIYkGa47gK+iEx6PB6TxFymA1GoF0tLSIhAQPr68BXdXGRxvazDaV4C5QCGi9lIMP85HJCTQHhGz8Zc/kdQqojUmk6lZV6UCZoPwPjoCrmw9esirPdW58PVzSmyZmHUrLtP1k1uSC5Z7WOjToK9iD5pObwLXrlXMKb1ev+Gv21iwe510s3hz0lB5IKmv3mvqUG/duNI2/gDNuAiQNZy2IwAAAABJRU5ErkJggg==',
cup_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKoSURBVDjLpVM9TFNRFP7ubWvb1x+QQAtRIFaRmEakaYiGiJaAurioiYsxXXVxMHESB40TJI5OOihxq0sZTDBq6kANikAJNLSBAokQ29q/1x/69673PoSoMS6c5Oa8c979zv2+k3MIYwz7MYp9mvb3IBKJ9HB3T7BSFAXcW8Q3P/KvWOTHXC5XaBdDRDIajZ7jflyn03VYrVYQQnZ+cr9bTJjwyWQSxWJxg+dv9vX1fST81SEevLXZbKRWlLE++wlyfBO5+BZq5ZIK1BkkWGxtsNoPoePUaTCdHqurqxzGLpDl5eVZDu41m8148+QBVoLv/qv5aP8QBm/fR6lUEsznNF6vd2xtbU3vcDhw7IwHBTmrXqxXymC1GpdB0XnSDvfFw7C0n0XXwGUYG5pUOYlEQtJyBrnh4WFLuVwG7wEGvHeg0WjUIoVCAay+jVx4FJbWfjQVMjB1diEej6t3uIQkTafTU3a7HQaDQaUVi8WQSqWQzWZR4wwK3yZhanbD2uZCfuMDVhYDKrharQoWE9Tn8z3f1ScKCGr5fF59XU6uIL8+CUtzI+o5P2zOG6CJ99BpqcqCMxihCwsLOQEWIJE0mUzQ6/WglKC6NYmW7ivA9ldMv3wFc2MJJL2E2o95wS7l8XjyVJZlRXRTkiQIKep4Uop6JowDrAzLwSKUcgxgCur5zzjiuYvU/DhQyYyqd7mWkNPpvOX3+8O8H8hkMqCEoRD1obGjF0oxxLEluK91Q6ls8l5F0OI4D33osX5vEsnO6EmBQOCFVqu92lRbIhKR0XrcyF+d5lormHkd5kVOgGgaAOMlfHk2EmeKMkj+3sZgMNhO5x5u9Fx/Cg1d47OQ5ln2x/5pjN34vjiHyMQjH/nXOk+NuZOsrkhM4YsklmjvsD2PneWa+QnIJn6IP3aTNQAAAABJRU5ErkJggg==',
cup_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKeSURBVDjLrZN9SFNRGMaPhGAUFEKB9CEG0QckCpUWpn+sLPqnECIoUTRSSdD+MFJM8gNJjXLDUCHS1t2kTJlpSYhjbroPttxV1DHHptgczW1+lISgc0/nnBoERRB14MeBe8/z3Od933sIAPIvkP9qMD09HU8R7Ha7YLPZhKmpqZ7JycmeiYkJYXx8XBgbGxOsVmv8LwYOhyOVCudmZmYQCASwuLjIWVpa4rvf7+csLCyAmsJiscyZzeZUbkDFEioOLS8vw+/5iA9vO6Fpk+JN3V10VxVxeutLoWmXYbS/i55xw+fzwWQyhYxGo4TFFpl4Y2ODH2y8fOKP9DaUYnV1lZvo9XqR0Dhf1Go1QqEQgtRkoLUeipJsNGdKIMtI5rRknUPHnRz+zimasbKywksaHh7+TBQKxbzX68Xa2hqCwSDf19fXOSxZmNsdqbjZnghfwAtaNlwuF3Q6nYs0NTV14sdi0Wj3wQzDjWOwuAUvklHVdw1XWw/D7rCBTgpDQ0NSkpaWdj5swA7SscHpdKJYeQZFQjIK5MeR+ywB91RX0GmRoeT1JaRLYzCoGYBGo9lOoqOjT4W/zsSzs7Nwu90oFJLQY21G9+gTLnxpkUI6WAK5sQEFynScfBCFYzURO0hkZGQSrUnc3NzkcZmBx+PBDVovEz8eKEb9+0LU9uehsi8H1e/y0aqrQebzFBy8T4KErq3MRKVS2VhzRFHkPbj+9AhemaVQmh5BbmhAm76OmtxCi64aecoLiKsgvj1lJOH770hIBGWbVqvtorMNsT8yo+UALsr24mzjbqQ83Ims9tNo1lYiV0hHbAX5FFNGjv72MhkMhn0jIyNBOmNQQ9YoTmJtFLLlEsSWk3kqPvTXt5FG/rq/fAt2lZG4n59/Ay6e4tcw3s+GAAAAAElFTkSuQmCC',
cup_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKaSURBVDjLrZNdSFNhGMdPF15EdOO94E230kXR6sIZZl4VkRiVMdOLiqC6sMgr08hQQUqUnBFJbRqKpa6yrw2bzm162lduY2dubjvbbF/Oj+WZuu38e8+hdRNGUA/8eDnw/P/v83FeCgD1L1D/1YBhmBKCwuVyKZxOp8LhcIzZ7fax+fl5hc1mU1itVoXZbC75zcDtdpcSYWBxcRGJRALLy8siyWRSPOPxuEg0GgUxBU3Tgbm5uVLRgIjLiZhfWVlBPMziy5thTD59iPG223jZcl1E1d6Iyf4umCZGSE4QsVgMRqORNxgM5ULZFkGcyWTExAenDv4RVUcjUqmUaDIzM2OhSDnrGo0GPM8jS0w+ytuhvFmLRxfK0XVaItIrq8DgrTrQfefhV9Xhm0kptjQ9Pb1GKZXKUCQSQTqdRjabFc/t7W0RobI8Yb0cidkbyKRmsaS+CJduCFNTU16qu7t7GD9DKI1MH4JhfnACQWM/YloZMqsKrNqbkTA0wTtQjU/yhneUVCqtzBsIfZG1wePxwO/3w+fzQdiM5/U1bCWHsLnUhM3IY6zamsCxzWDkZWtUYWHh4fztglgQBYNBsCyLQCAgfi98uIfv3nZw/qtIsw3gfLXg3NUw3ZGkqIKCgkNklZZcLieWKwjC4TBCoRBY+yQCE5cQ0tRjnZGBWziJtLcGGy4ibtmfeX62+BxFYrdgMjo66iRGsFgs4gyC9CB8IzWk1D5seKrAMcdh6pHC2nEMtjbp1gtZcdWvP5HELsIerVY7QnbLC32H3l8hPT/D+lcJOGclzL1HMSa/y2vejr/6LCvau+Nj0uv1RTqdLutR1WMreh8bzAnQPWVQPWnNqdXqor9+jbOdB1od/RVRuvOIf+DyvjM75f0AOEMKvrn+ie0AAAAASUVORK5CYII=',
cup_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKKSURBVDjLrVPPS5NxGH8jPES3HfoThC7SJaqLgjMHojB38SJGoHgQA7Fsgr8SwamMbQotPKzDRsjY1EYkBaLODWXWpkyXijO2CqZzbmkwmts+Pc9THqToUl/48D7v5/s8n+fX+yoAlH+B8l8FdnZ2ygj27e1teyQSsW9tbc1ubm7OhsNh+8bGhn19fd0eDAbLfhPY3d0tp8DY/v4+jo6OkEqlBMfHx/JMJpOCg4MDkCjW1tZigUCgXAQoWE3BxXQ6jeSXON69cmLBZsZLw2O4nzwQeEb0WHhuwfvXLvL5hMPDQ6yurhZXVlbUXHaIg8/OzsTRpL35V3hG9Tg9PRURv98fUqick/n5eRSLReRJ5O2zETge3sPTRjUsutsCa9NdvHh0X+72QgFkMhlpaXl5+avicDg+JxIJZLNZ5PN5eeZyOQFX9idQ24hGo/B6vVFlYmLCiV+HS6PpgwXPB8fgcpmLxWKg4UkwbQqLi4tmpaKiQnMuwI60JkxNTWFsbAw2mw0Gg4Ez8eQxPDwsHN9NTk6ir6/vmqJSqe6cZ6c9w2QywWKxSEbOtLS0BKPRKBwNXDifzyfvtbW1o0pJSckt6ilUKBTg8XjQ3d2NYDiMgYEB1NXVoaWlBePj48KzzVxPT4+IdHZ2ZhQ6V1hkZmYm0t/fj6GhIRGIx+PQaDTgDXFb3Ibb7UZ1dTXm5ua8LDA4OJj9+TkqyiXC1a6urmB7e7sI9Pb2oqqqSgLq6+sFbFdWVhZ0Ot0bFmhtbU1f+Beu09FqtXtmsxkulwvT09OwWq3o6OiAXq8Xmzmn05miuaSpnQ8XBEpLSy/TVm7U1NR8bG5u/tbW1pYlfG9oaGgiqMk+YY7v2Id9fwCSFd62xayLzgAAAABJRU5ErkJggg==',
cursor=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAD0SURBVCjPfdExSwJxHMbx/yTc1NrUy+h1+AZ6GUJBaYdiKVwopjmYOASiINJgEVFwUFHo4BIiDtql/SPU5BDUQb8Nomh3J8/we4bP8MBPIOYpexdtPcvyyrO6ETxR5zGwAeiMeOfmxBE8MOKXKsWwA7hjSJceZbJhW1DC5BvJDy+kNRtwzYA2BgYSnUTEAgr0+aBJC0mbe85i/0AOkw4Gn8SH0Yo2CRGMrYEralyOq/SJzrRtBEJVvMoKyJCSyd3zZh2dUMZmZOotuYOIuAuYBKbqlgVcKPN7KhvccnRsAYv49/I0ODA9Lgfgcx1+7Vc8y8/+AURAMO9/VDEvAAAAAElFTkSuQmCC',
cut=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIaSURBVDjLY/j//z8DLqyaNVPLrnr5PMnESay41DDgM8Cuellm+7rTT0RiJ3Aii4snTGIiygDnupV5c/dc/QF0AT9MTCl9hq5P67qtRBng3ri6ZN2Je/9lU6bKgfgSCZPVJ2+7+CR9+u5tRBng07K2+sCVZ//lUqepCMX0y87cefnO9B2XH4rGTZQgyoCA9vUt5+69/a+QNj25f/O504evPf+jkDbNmuhADOna1Hn50cf/fZvPf7vz8ut/87JFOUTFAq9tHDiUI/u3dd8Fatxy9tH/xCk7FxCMRiGXNCmjzLmrneo2XtLJmLckffqesxcefPgfP3HbUcHgRha8Bgg5p0kANd5OWHXnf8i8C59TN7/6P3PXjf8PX//4H965bg+vZbgjXgOMsuasiVt67a+Ub4GdhHeef8LaJ/9n773zf+nZ9//Tt7//H7vsxn9Zz7QUnAZ4de375Fi3Ahy/RnnTpqdteP6/ZNGpf+kbn/7XjZty0Ld3x2XrgvVfuA08ObAa4NK09XnUkmsvHJvWHU3b9ua/Wd7yG+Y5a14HTj3yGSSvHlZW5lCx/b+QRZA0VgPkgsvDAqcffxO17MY/s5xlp7lMAyVMM1Y8DF9w8RenlqOcWVbfHPvSLX94jX0FcMaCiGu6hJhHlgKMrx83/1jypuf//Sftf5q0+u5/o6RFN0jKjTyGXuyGiQuu25dt+26SuuQBj5G3CLoaAMk4ntedg7qJAAAAAElFTkSuQmCC',
cut_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIcSURBVDjLY/j//z8DLqyePcvQuW7VQomESay41DDgM8CpdkVm94YzD0RiJ3Iii4snTGIjygD3xlX58/de+y6ZOFkAJqaQNk3Hu2HlaaIM8G5eU7r+5L3/silT5aE2q9UsOfw6ecJW4gwIaF9fc/Dq8//yqVO1hWL6pZuXH30+d8/VhymTtyYRZUBw58aWc/ff/VdIm55as/jQraM3Xv71a1pZTXQghvds7rzy6OP/vk3nvjx4/fW/V/2yuUTFQgc3AxOIjurbMuHuq2//t5x9+D+2d+N+gtHYw8ssv1xNeuYuE60zc2UEOzNn7D5z4cGH//ETtx8Vie3nxmsASPNOU53r591s/p+yM/503cf+/6w9N/4/BLogatLu/RJJU6ylEicy4jRgpYbsnDNu1v8nCHFYTRBkD77iYf1/zt7b/5edeff/no/D/4vOVv9nSPJn4jTgsI3Rm7V6yutB7BWqUjMvBTj9L118+t81f8f/ixXF5h6w1L+yz0LvfjcvExdWA/Za6d87aW/6dLOB6qG7Xvb/12jJH9pkoHL6mI3RpzYuBua50oLlJxzN//cJcShgNWC6JH/ycVujH+eBilZrym8AapLYoqu8/oyD6d9WTgaJFeoyU4/amfzo4WMRwRkLvfys8n0CbGpd3IzMIP5sGaGCm94O//eb6ty96G77f6W67BKScmMP0L/L1GXmA732fb2u0k5gTMmgqwEAFiuGP1TJcvYAAAAASUVORK5CYII=',
database=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEYSURBVBgZBcHPio5hGAfg6/2+R980k6wmJgsJ5U/ZOAqbSc2GnXOwUg7BESgLUeIQ1GSjLFnMwsKGGg1qxJRmPM97/1zXFAAAAEADdlfZzr26miup2svnelq7d2aYgt3rebl585wN6+K3I1/9fJe7O/uIePP2SypJkiRJ0vMhr55FLCA3zgIAOK9uQ4MS361ZOSX+OrTvkgINSjS/HIvhjxNNFGgQsbSmabohKDNoUGLohsls6BaiQIMSs2FYmnXdUsygQYmumy3Nhi6igwalDEOJEjPKP7CA2aFNK8Bkyy3fdNCg7r9/fW3jgpVJbDmy5+PB2IYp4MXFelQ7izPrhkPHB+P5/PjhD5gCgCenx+VR/dODEwD+A3T7nqbxwf1HAAAAAElFTkSuQmCC',
database_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIkSURBVDjLpVNNiFJRFP7eU1E0KSLTMpAwYSxyaidDtChm0WYQ3NSutv2s2kwwm2igNgMtooUQEQhhA9GqhSDTQsZZFDbNDBgVg5bSw9J8rzFF33udc+HGg0ladOHj3nPe+b7zc99VbNvG/yy30yiVSl4SnCNcsixrivYEgY7WJu0faX9EKGUyGVNyFFkBkY/T+WkoFEpFIhEEAgH4/X7w916vB8Mw0Gg00G63y+S7mM1mm4LIAYxisbhSr9c5nT1pjUYju1qt2oVC4YnkqbIUMk6Ew+F/9hyNRkFJLuyaATmFoqZp8Pl88Hq98Hg8wtfv99HpdNBsNhGPx0XsRAG3241ut4vBYCDs8XgMXdcxHA7FN/b9VUD25HK5RAUczKC+hYgcNpNN05xcAQdLkqIoIlj6VFWdXIEUkAQGV8M2k2vaG3z6sYGfVR39XzsHlm/dX3h5d31xlwAHM5goBd5+LuO75z3OnU3jyP4EVrZeKGub2p309cP7VKcAQ2Znoiz3deMVTk1Nw1RNTB+ahamMkD45w7RrfwSYwFdFf6K4Quf6pmvwKHswl7wh7Jvnc4gfTPHR52zhcqVSeZZMJgOxWEyI8BC5CmOnh63WKtZbZczPPsa94hX4XCLJQHG+xnw+f5SEFghZmvhefgvcTqn2HN3gBmZSZ5CInMaHr1Wsvivjy3ZvSZn0nHO5XJDIxwgWDbW2vL10m9xXCUGCQXi49qA1/xvyq6BCh7yZeQAAAABJRU5ErkJggg==',
database_connect=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKNSURBVDjLpZLLa1NBFMa/m/uITfpIKzFJYxGaxkQDFnFRsrLgxm5i3dWl+Ae46Kr+DWbZTcBNhboQ1KW1uYhQhVKECFE0SEvSBrXR2Nyb+37EmYHG0hZceOEww5n5fvOdcw/X6/XwP59wWlKW5SABF0jc8X0/S9Y0CbL1q2T9StZHJOT5+XmPO+6AiC+T3JNoNHolHo8jHA4jFAqB3ut0OlBVFY1GA61Wa4PkFk4A1tfXX2cymdmJiQlwHHeqbdd1Ua1WUavVVgLHDwkwF4vF/ll7MpkEKWPuRA+cnhjY3NyEpmnI5/MIBoMQRZGVoOs62u02ms0mUqkUBfxt4upGe8B2vFXXGBnje/uQJAnlchm5XI6JqW1FUWBZFgRBYLk+YOXNL8FxnA9BkUsLkTg+7vzAZESB53msiVRAATQOe0bF9Jz1wDDtccNy0yEJ4Dkf/Fga7+pBJFJXsba2huXlZZAHGICuVHjogAG6up20bJuCsNvSIPI+pMg4Ss+3mOXp6WkUi0XY5M4hiAL6DlTDiOpErJHo6iZ2vik4IwKJCym8/aJje3sbiUSCCY4C6J4BOqpx0XYs6KYDRbegaiZqe20cqBYGzl/Dy/c+Crdu91+lAGqfOuJb5wqSqltyNCLxChFKPAeBJzMe4OD5Hho7Lczlp3BpMsJ+4+joKLLZLJvGer2uCweq0R0OBUSPUDXDwm6TjOtvDUbXhNm1MJUI497CDCSRZ2XQUaZTWKlUvhMnBeGgq8uBnnjz008Ve3ttx9HMB7EhvzOTQeHsMHd9ZNAeksuvmG06RKZpUuFjUsrDpaWlfW72/osB13CeuYZ1wzXswa2nd+2jk1kqlYaIYIqET5r2eXFx0Tp6/ge8rrdXLiWBdQAAAABJRU5ErkJggg==',
database_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIlSURBVDjLpVM9iBNBFP5mf0jIIsIdMSsR90L8iwfWlhaiaCHBNB42ioWg1QmChYiFXG8l5LjiIKCxMSg2ewQFlRQBc4iIiIoJ5kgMJCTh8mOyu743sF70Llg48PFm3rzvzffezAjP8/A/Q5tc5PP5ACU8R1hwXfcI2YMEmrrvyX4mu0LIJ5NJx+cIXwGRj9L8YTgcPmaaJgzDQCgUAu+32210u11UKhU0Go3X5LuQSqWqksgBDNu2X5TLZT7OmzZGo5FXKpW8bDa76vMUXwot5iORyD9rjkajoEPObOsBOWXGer2OYDCIQCAAXdelr9frodlsolqtIh6Py9ipCTRNQ6vVwmAwkOvxeIxOp4PhcCj32LdjAr8mVVWlAg5mUN0yid9sJjuOM10BB/skIYQM9n2KomxToPydwCcweM5gsnj7DFi+gk+X4zBzN2ftU9qtHRVwMIOJXA5bt5jDzI8iDl+8jkBsHv13tvLh1dq9tZP65h8KGJOn+3IHbx7h0InzCH55CbG6gNDXJ5izZlVPeIu/FTCBr4peImq1Gvr9/ladTbpaMwacvbHVvLt7obpibrKES4VC4XEikTAsy5LvgJso38YeE7315zCeXsOwX0OP4rsdKk/Fhpj8jZlMJkaJbhNS1PHd/Be4HL2Uw77WOvbP/ISmfEe3Mca3uuqMBt4dMe07p9PpXUQ+QHCpqR+PF+8vbjY3rqqOsBzVqxLrwWl7vPQLOvKpkCFdDpkAAAAASUVORK5CYII=',
database_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKRSURBVDjLhVNLTBNRFD3TTju1FCcBaxuaQEr94ydiZIHGpcFISBOi0YREZWHCVoyRxKUxxq0LXcACQyLsjO6KjSEiDfHDz0S0CLSxlFKd2g7MTDudGd+bMAQCxJucuXfuu+fcO/PeYwzDALVoNMqRuI3guq7rR4g/SEBC/Svxc8T3EUTD4bCGTcZQAUI+RvxLr9d70u/3o6KiAm63G3Qtn89DFEUkk0lks9lRkrvW3t6e2lCgRZFI5F0ikaDtjN1MVVVjYmLCGBoa6qccC7Z1kQafz4f/WSAQAGlyaXOOpQ+SNNUymQxcLhc4joPD4TBzkiRBEASkUimEQiGzdlcBlmWRy+WgKIr5Xi6XUSgUUCwWzTVN+IAzeOOde71orP0eAaOkbrDWf6Cw2+3mBLSYgny3KULXPOUY2BUB/hMd4IOn8XfhMGYjvU+2TECLLRLDMNA0zYw5JYa6Ghke/hyEn9/gZEqo3OuHp7qW3yJgESjoNPSdlb8gWCOCr29BMT0Ip5tBYnIWqlL6o8irzVsEaHcKSqQCen4cweok+FAblNRz2JxlODx1cEkzGWmVbTl7Z/jHhgCF1Z3GYjIKf+U8+ANhQn4Gm6OMUiGI9MhHg5Gl1sbu8UnKNc8B7Ui3ipxEcwvlpVFw6hz2N1xGabkXdqeBYqEOmfefEZWac4e6xz9Z22hbn+BmLBZbi8fjEBdG4NF/QdUDSM88hQ4FawKJR6cxLDZl86qzZdtdoDYwMBAkQg/2LL/ovNLVh++Dd7G0OAau9hTkrKgnnE39GW3f/Z6enpUdBSx7ePu4eq+zi4VNw+TbV0gsxFd5b9X5i4+mpnY63tsErl6okhvrfWzT0SAMR3FMXsnean08Pb/b/fgHqpjCspi90kkAAAAASUVORK5CYII=',
database_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI8SURBVDjLpVNNiFJRFP70ac80w6FEJ2FqsDKmSRpmU6taBRGE4aaCYPatat1qtu3aRosJI4Ro0Q+EIP0QCTM4WhSBBqmkJJr2cvx53vfu69w3vXIapUUPPs69553zne+ee67NMAz8z+cY3aTTaZkIzxMucc6PkD1EoCV/T/YT2TuEdCwW060cm6WAkudofd/v90eDwSA8Hg/cbjfEf0VR0Ol0UKlU0Gg0XpPvYjwer5qJIkAglUo9L5fLopwx6WOMGblczkgmkytWnt2SQpujgUDgn2cOhUKgIme39YCcJmO9XofL5YIsy3A6naav1+uh1WqhWq0iHA6bsRMJHA4H2u02BoOBudc0DUzJw8PygHTG9I0lsM4kSZKpQBAJMHUDanMNe2ZOQS3lKXkeuv77Ev70wFJgVmTMhAjUGi8xte8Edk8vwNl9C32jtEXBNgIziUhMdGsYfn0B714f9B+PMH3sCvrlJ+A6m0xgVdc0BvXLM/gjF4DBOlbv3sMuXx+DWhZevSSPJRCwquvKR8i2IbxTPXD1MzWJk/w1zJ6+jiBb96zeOundQiCki6uiSYS8QwKvPIVv5jh47x3l9rEYj4APa9TgAg5Ez0maOrz2t4KlTCbTLRaLcH7PUuOicHubMLRvNPASsg8LIgp8UID/8H7oKrv6anl+zjb6GhOJxCwR3TiorCwtXL5tl+wlImiLSx6ZRTuknRHUP+RReLz8wDbuOb+5udg0dO6mY9sN0Vyu801Ls/LLYrPp2Z9W3anPTwD1kQAAAABJRU5ErkJggg==',
database_gear=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFmSURBVBgZBcHPa85xAAfw1/Psy9bEFE+MgwyzrLS4KqUQRauVODohB/+As7OzGilZrpQki4OLgyiEImFWmqb5sfZ4vt/P2+vVitn+nMyZMpZdKeV1PpTpMjvZALQe7clMZ+9mawyKJb99sfA0p6e+AR4+/pySJEmSJOnlRe7cjIhoZ3wTAICtyjGAqojvBvRbJZYt+maHAqAqovLTiqj90lWJAqCK6DOgUumpBTPqDkBVRK2n1tJ477tRI+LKoe71pQdXz7eLaNRqjcaCA2LEqLHZY9uac8cHqyJ6ehp9Gpux5LEB+zSGbtxfbhdFrdaIuzYa9spFnYW3y1tMnL2QdmNRRz/4a1HXBvN60vttzry+qTdfJ9urh3WsM+GHrvWe5V/G1zXuTy8cbsWt7eVymWoPDaq9c9Anu634aMS0uaoVwLW19c66PL/05+zQif33fnh5unt7+dGToyIiIiIiTuVIIiL+A271xrBxnHZ+AAAAAElFTkSuQmCC',
database_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJMSURBVDjLpVPPaxNBGH3ZJCQkVSEkJJja0MSCabHoSUEKerAgggRy0ZP9F4QeDAhepKeeBA8GPFQCEutdCMRSSYkimouIVqUkNW1iyJZN0mR/zvrNwtbUGjw48JiZb773vvfN7jhM08T/DNfwplgsekjwBuEWY+wMzVMEWrKPNH+j+QmhmEqlDJvjsB0QeZrWz0Kh0GwkEoHf74fP5wM/lyQJ3W4XtVoNrVarRLGb6XS6bhF5AkehUFirVqu8nDlqaJpmVioVM5/Pr9g8wbZCm5lwOPzPnqPRKKjItSN3QEFLsdlswuv1wuPxwO12W7F+vw9RFFGv15FIJKzckQIulwt7e3uQZdna67qOTqcDRVGsMx77q4Ddk9PptBzwZA7q2xJZrWYw0PaRmXwBwzj4CL/vwHbAkzmJgydy8JiiqxgPJpF5eR0aUxwjW7AJD98swGQaVKZDpf3JwBSSkQvoyvtY/XE/+HT57tjrRbF3RMCurjMVV2duwzAZDGaATrEjbePs+CX01AHe19al2QdC4JAAB6/OIZNlTq62v5JlipEbzdDQUbo4d2oOPa0vvN0qtQ8EuHX+qehPRKPRIAEZuqEjfHyCyIYltivVEBiL4MP2Bja+l1qqjvlhBwvlcvl5Mpn0x2IxDHQFK+VlugPVchMPTuNifB7vqiWsbRbbso7LO0vmJ8fwa8zlcpMkdI+QFgThBH8LvB3u7LF4xzw/MedY33y1qzDzCpG/HHpMf45sNnuMyKcJjC718yNpUTSY0zdgRvznkrll5/0CZpfQA8IRXj8AAAAASUVORK5CYII=',
database_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKOSURBVDjLhZNPTBNBFMa/3e0/ty1tiZUChkZAESnGA54k4WRATQxJL2piYpR4UBNvnrwZPXjwiiHRBCVBUMJJ0UrlYtJEjI2JBCkEbLGhWGxh2+5u6W7XmY1LCkJ8ybcz+2bm9968mWE0TUM4HLaS9jzRxXK5fJS0h4lIt/yNtAukfUIU7u3tVbHDmMnJyWNk0rDX6z3u8/lgt9vB8zwoeGNjA7lcDolEAul0+iPxXQgGg8lthFAoNBWPx2k4bS8rlUpaNBrVRkZGBim4Uiz5tNXU1OB/Vl9fDxLkzE6/iTh10urqKmw2G6xWK8xms+4TRRGZTAbJZBJNTU0UgD0BJpMJ2WwWsizr/4qiQBAEFItFfYz6dgOwxl44jtMzcDgc8Hg8cLvdejEtFgtYlkVZVdFQnLInXnVPz/cHrm0BjAxoRFIsXSqZTGX4aL+0MI62didf13Ovw9kcePjlwZGebVswFlDRbPRFf4Gu/DQc3nm4Wk6jEJ+A2dlS7W4ojk3cbuzbAlRGNwBGBryyBFf7OajiDGz7D0JWf6K26wSvFLL9OoDKiG4AKzMT1SqUsj/Acmtg2AIsVWuALCK/UuRMdAI9KnITkUqlIEnSVoUt8jICmMK+WlJEJUMKtk6q5oRW2sT3F3PyzJxwnWZwJRKJjLa2ttr9fr9+DxiGgbj4Frz0Hgc6OqFthsCoOcy8FiRTbkXgyHhsUbh5eXhpjKFpDg0NHSKgu0RBcmQuenxt2jgaO7uxudZP7oELsx/y0udI6pZfll7a7By6BhM5/TFRQKUNDAw4SS2az/rePKvrPBVQ1iOYffc7/zX668bVp/PP/3mNOwGGfXp08r6j2tMnZgpSLJa+c+lxbHS3eX8A58zTPyvL4X4AAAAASUVORK5CYII=',
database_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKZSURBVDjLpVNNSFRRFP7evNE3P6n5MzYTDVOi+QNZFrUIIlEIaiGCEBltgqBdYK6Eli2DFrYyalFCFETQdhiCCKIoMJxIxppKCQfnOc7kjG+c+9e5zxz6EVr04HAv99zvu98553uGUgr/83m3O0wkEhYRD1GMSim7aO2goK1M0vqR1jsUieHhYWH8qYDAPXT2IBQK9YbDYQSDQQQCAeh7hUIBa2trWFhYQDabfUFn5/4iiMfjzzo7O/uj0SgMw9hWNuccyWQSqVTqXpXAfntyUAk1WeYN+32W34QElJCQnMHwBNFy+CZM364qwerqqlab9f4EmwQe8Tcf695R2+qCFBOQFQdO7gN8e86Cox7ZTMYtQ4NjsRioD5tNlEJGieBCTXAveOENPL4+esVGOZ/CEj8OsRKFWJ6FbdvweDzVnrgEy69P1JLUS/6mQ3UAKXcyNJsN5BfjMMIX0dx0Bl6vF4wxdy2VSm4ZGiyEgJfAfRSXrfoOKO4QRwR0G0oSWXER4vttCMUhEIA0j1DKclVUFSguJ/2hoy2OPUelKNQG2onei52RQap/HpJVULRnoWJXUJGWq8Q0TZfAVUD1Pyp+e/lcccFoX8P4u7GW2CmPKOfB1ldQ8rTBPHgXsiYERk1kbN0F63DL2t3/6sbWfL8+PVAPXr7q2O9RzC+BhUbBG/vh9/lJCau+utWDSqXyu5Ull098/rBR4o1oHbiF5ZUSuONU89pYkUgE2mTpdHpTwVby8+Oe01RC22K++XpaDYx1f8kE9awty3KB+nUN1lbWLpyZmcmQoYaqTvz0sKtXCSHbz88np6en95HEaxQj1PEGPXftvlwuh3K5rIH3qZQbExMTy8a/fuepqak6ArRTUBvY3Pj4+Mav+R/mW7NMaC1PRQAAAABJRU5ErkJggg==',
database_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI5SURBVDjLpVM7i1pREJ7rYxXFiKjxVYhIMBhMbWGbIk0wasCsCCkD1vkR5gcEmxSpsrLIxVaQCKYQRAW1kgTxrVEUn7i+bmYO3MUlkRQ5MJwzc2e++Wb4LicIAvzPkZ072WxWgYCv0N6eTqeneD9Bw+epjvcPvD+jZf1+/1Gs4UQGWOzG91ej0fjcbDaDWq0GlUoF9H0+n8NyuYR2uw3j8fg7xsLBYLDHCimBLJPJfGu1WtROuHT2+71QqVSEZDL5RayTiFTQeWYymf45s81mA2zy8o8dYJAhjkYjUCqVoFAoQC6Xs9hms4HpdAq9Xg+cTifLvQggk8lgNpvBdrtl/uFwgMViAXd3d+wbxf4KIM4klUoZA0omw7kZiLhsKj4ej5cZULJYxHEcSxZjEonkMgMRgOYtFArQ6XRAr9eznbjdbpZTq9Wg2WzCcDg0xGKxD/V6/dMDgN1uB+l0mvnRaBTW6zV0u10GSOChUIgtdjAYcKVS6SPqwngvpFQq9QuFY8zn8/Dm+hpSNzdQrVaBROXxeKgIJpMJdQer1brDBlc8z8/vGdCsxWIRNBoN8wOBAPPD4TDodDpYrVakQkgkEuBwOBaYYtBqtYrzEd6hZHlc1hX5NAqxi8fjTNZ0aCRqVC6Xf/p8PgPqYsud/42RSOQFbvzW6/U+QsVxNC8tsdFoMGHZ7XYgtWLOFJcsyeVywwcALpdLarFYPNiRxzkfYxEx5FCF75Fhy2Aw8OjLUVTHfr8/RkavfwO7WaXhrjM2EgAAAABJRU5ErkJggg==',
database_refresh=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKUSURBVDjLhVNdSJNRGH6++enSTc3Rl9Y0KzRcoSHShURERRldlLGLMsqrCO2iLuwuL7qKLr0MoQvDHJigJkEbG5mGdzKU1PnvN7I2Fgu33Or7W+858YkrpRee8x7ec97n/TtHyGQyCAQCVtJXCS2GYdSQribQ1vhEeon0C0KgublZx18i+P3+43TJI0lSXVlZGWw2GwoKCsCINzY2kEwmEQ6HEYvFPpLtptvtXs9i8Pl872VZZuEyu4mqqplgMJh57O1Ya/e25jByExZaTpSWluJ/4nQ6kdZSTlXRAtvtIkXmTNPT07Db7RwlJSVYSS7infwGa8llaJoOTdXhLCwX7Zr97C3PdW9fy2BTFoHD4WB1IhKJIJH3HZPKBA4UOXHh4GXoGR0GQTd0vk+l0peuPW9aGm7zVolmLSyyKIoUTUMgPoN9uRIk635MRYNY+bYMVVFRXlTBz0PhBf/Ifd9FloHFzIAdULM4FhMh7jiyMIREfBOte9vwtLEL65+/anNyaFz5qTSdf3Y6P6sEXdc5CcO9Qw85UTQaRfJHktvYnTzFqpPzOV/HmEG+6awMzOgmyRahquHtlyF+p0FoTJCz/s8UGExHk1DRflHjDAxHBpBSNrldURQ0djaMUT/O0DgZuSyySPF4HPQS+QTS6fQfZksuBuQ+5BXlQFDzceflDWqkJh2tOCydqq/H7Pw8xsYnvILH47lC7P0ul8tWWVkJq9UKQRBw99VtFBcW4+Sx2q3xmeNcWl2F3z86qWv6I4Gl29vbe4RIOglui8VSzP4CK2dQfo09ksgf0kxoDhpPm/VG/0DOT1Z7wqOcYLt0d3cXknMVwaCmhjzRHi+l7pjqmq3b8Y1v/xg7ofZBTY6rvbp/t/PfI0AjgZ0qo+wAAAAASUVORK5CYII=',
database_save=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKFSURBVDjLhVNNaxNRFD3vzUwSJ622YEwgYoVaNBUVilZwqStBtJBNxC4EEel/sDsXLhRcVxSUQo07QVy0jbpQqiANsS5ciNpowBhM2kk7nWS+vPdJqi0tXjhz39x595zz7syIMAxRKBSilM8TLgZBcIjyAIGWwQfKnyjfIxRGRkZ8bAoxOzs7SJumEonE0VQqhXg8DtM0wcTLy8toNpsol8uo1WqvqJbLZrOVDQzT09MvFhcXWS7cLlzXDYvFYpjP5x8w8b+QdDmcTCbxv0in0yCRs5vrOhUVU7VaRSwWQzQahWEYqmbbNur1OiqVCvr7+5kA2xLouo5GowHHcdS953mwLAutVks949qWBJ2zaJqmHPBmxs0ndXRHe2G3PfR2RfBo/geEHEy8v1sKg1CgYa3hebFyct0BK9KwVBZCYM12cHr4IC4MdeHpm+8Yv5TZoPzwZY0cibeyQ+D7vmpm8Npuuag3PbV55l11vdGhktUCakttEgr+zoDVGdzMx5FSQAsB1w9we2yI1OioRKDR1dShZmOttv8QMDrqHcKYIeGQixv5ryAueEQUEJiEn/PCNAJIVuRXRV+ieoWd8Eix5XvQpEFWdZAfyho1SiIQcEmsTQNmB5fn5uYeZzKZeF9fnyLhITbtKgxqHDvXTWRtopRKNaRzx/QIbk2V8ctahZ7L5Z5NTk4eWVhYuF4qlbJSyl38L/hBijQNBFjD/flr2G3uIxcSNfsbrp64Q6sYDZpmwHZHR0e/ULrCmJiY6F5ZWTmg6+n5/Skg2dXEmWPD6ImklYklJ409cQ9mhD4icirUQLaI42Mzrwf27jjVE+0hyzvpGC4EDViEPgJh42P5M35aLn4DnlayCCcx84IAAAAASUVORK5CYII=',
database_table=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJoSURBVDjLhZJZTxNhFIbnRq/9E/4Rf4JRE+PChV4YYjQxLglRQkgDWhGXRo2VNJWu0IWh0hY6iEvpjl1AWuoImqFCO6V0uqRrzOt8H0mxBvQkTyYnOfOcd775mPn5+WMcx12dm5v74Ha7806ns+JwOOIsyyptNttxAMy/YDwejz6ZTKJQKKDZbKLdbkOSJKTTaVgslrX/CmZnZwu1Wg3VarUjID3BbDZD5d7GE+cWRhwZ3J8SoLD+wMDEOu4ZvqFP9zXMuFyuXLlcphszmQwEQUAwGESpVILBYEC13j6Um9pUg5mZmck2Gg3wPI9isYh4PE4hNT4+DlXkIUZDw3jgH4TC24+Bj324u3CbCq6//gJmenqaClZXV6kgEolQSGk0GhxWRND7MgHGbrdTQSKRwM7ODnw+H/x+Px1Sq9UwenMHQgSXVVEwVqtVarVaSKVS9PvD4TBisRgVqFSqzkZrINuVIF+qo+dxBMyEXCSyKIr095EDJUmI6OlzNeyhHFgZIiBPNpiFcymLje0yziqDYIxG41GdTtc7pp/CpMWCMa0eJpMJYyYXKpXKoQn4nyWcHvLvXQatVntEaV0Dv7GJCW4Ztk882MAm3i6JFHdUpAKaQk5gl1kTJJwaWty/UYOT31GsNOkwKS6e79roiYko19qdngh6HgX3Bf3mdSrwyC9yf/EukYfzs9gFEZxX+vcFffo0dmXBwvLe5vcr3QlsAbGrpwlG/hDcepNCodyAKNWxVahBEKvySVfAZ0p0+CAuKH2/OoIbmuTitVcr1SsvErj0LIqLoxGcU4ZwZjiAkwrvgZy4w7G/AXhUV4qmXai6AAAAAElFTkSuQmCC',
date=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIESURBVDjLlVJtaxpBEH7uvNSL50skFBqCiDVYpCWiIAjtx4Ih4I/zs78jkD9QioVAUBGNWigqRfpBxSO+3LnbmY13mNQWOvAwuzszz7zsQEoJBomWzWY/V6vVb5lM5oruBr/tYBQKhU+1Wu0r+/CbF6cOA02Tv9jr5gbn+TyGd3cQlQpe40nYFry9xZvLS/y8v8fm+lrZ0lJqukbCTlYwCCsWw3a7RTgex3EggLiuK5jkYkYiynYcjcLcEXOsvjvDNAx0BgPl1O31IIjEPjmBHQ5ja5rodLvK1nl48Ang9dgHRIyyN87O0LNtXFD2FLWmU4B0HKxdF99JDwhvhUCB9CPZLwDd2K/gw+kp3lsW5GYDl5wEg8heEdG7oyNkSGuE4GKBRyL1q6jX69J13b/CcRy5XC4VWPiNYzjWwAFZr9dot9tIp9Po9/uq9/l8jnK57H25L/ohAg4ejUaI0ORzuRxSqRRCoRAosw+P6BmB95inXfAWhdFqtVQ1Dg+UqqNW/Jg/WnhZ4mw2g6DJc/BkMlFnhud3cAb7ZNwOrbaaQzKZ5OXBcDiEQb/GA9XljoqU2A+u0CqzqVgswqKv5awcPB6PfSJ/Bgv6V5uEjoIN+wjQHrDmCjhzIpHAarVSLfktdGlNyTHKZf1LvAqYrNlsolQqPRFMp9MvjUbjI/5D6Dd+sP4NLTpNB1cxufkAAAAASUVORK5CYII=',
date_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJRSURBVDjLhZNdSJNhFMedDj8QFT9QQlFIEHexi+ZFFwrddJERRHTRRTdb9AFJ0E0QiBfBQrLVmvBCsVAG3RXvMN1Ya03YJvsqFbMJNXNjsBbhhox9r/17zrMpih8d+L/nPc95zu+c93nftwpAFYmZRCaTndfpdL6BgYELLJbSWkVShUIxotfr3bSH1vbq6BJkzs3kN5kQjUbhnZ/ncbAiul+2WHjui9m8l6s0rpJQsFZXh4DLhUgkgm/Mr9bUYLu6mmtFIkFgaYnnaM9KBVCZrgz43tgI0WBAKBSCODuL5Y4OhNvaEG5uxueGBogzM+Uc8xtHAf60tODT4CBe9vTA1teHzd5ebHV342dnJzYYyNTUhOn6eizU1iLKJjoESLa34/fwMGJDQ/gllyPCYOH+fmwx0CYDBbu68INNFWxtxbZUehDg9XpRLBaPVaFQQCaT4SKjNao5FpBKpeD3+xGPx+Hz+eB2u2Fhb4GsVCr9H0DFRqORd87lcojFYnA4HDzO5/PIZrPweDyHAbRhFyKK4t5mm82Gd3YDHrxS4vbzy1A9GcWUYeJowK52AdRd9+YxHr1VYWFdwGrUCu3Hu7imlePs2CnNiQAa3263QzU1irmvLzAXmObnoLHdgtZ2hwAZ/ikTgBX93Q8g0fjJZBKXxs/AvP4a++39mkCA8g9Bz5hIJHYIsv8waYJ0Oo1z90/jqfUGJq1KXjz5QXlwAkEQsLi4CKfTeaTGhTFc1QzimfUm70yeYn4GBFCr1S72G+MkqSauYOReHx+b/PWHF3eo9h8NpaVzO3ihRAAAAABJRU5ErkJggg==',
date_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJeSURBVDjLhZNdSFNhGMc3z9lUREVnGmgzEZa7kGBe6kWYg26syz7uoqKgEOpiUjfdjALJiwmDYKSMburm7EInG8yNttm+UitwQm26sVgrcSPWvtf5977HbSyc9sB/73ne53l+7//snCMCIKIiIVYqlVM6nc6vUCgukZylexWxKpVqwmAweGgP3avN0Z8QWTxEAbMZ8XgcvuVlIQ9VRK83LRahtrG6WqtVDhaJafK5uRlBtxuxWAzbZP3IMDhoahK0JRYjuL4u1GjPVgVQcXcI+NLWBs5oRCQSAbe0hM2eHkS7uxHt6MCH1lZwi4uHNbLuNALsd3ZibWQELwcGYBscRFgux15/P3Z7e7FDQOb2diy0tGBFKkWcODoCSMtk+DE+jsTYGL6PjiJGYNHhYewRUJiAQn19+Epchbq6cMCy/wJ8Ph/K5fKxKpVKyOVygmjQPTpzLCCTySAQCCCZTMLv98Pj8cBCngINnuf/D6DDJpNJOLlQKCCRSMDpdAp5sVhEPp+H1+s9CqANVQjHcbVmm82G7dfP8f6mEraLEjiuyfFu7n5jQFVVAD3d8eIhPmkmkDPPgw9akHnzCIEH52GdZGdOBFD7drsd9utnkSXDWJgGZruAZ0PYn7sA6xSzK7zKFECG/tQDqKj9dDot2OY3ONRH6ulp2CZZXgDQe0ylUr8opP7PpA6y2SwcV8/g96tbABnKa0RIEkXvMbCqmW8CQK/Xw+FwwOVyNdb8DHx3ziE+O4SfjyUI3xZj7bIUFjXzRABotVo3+Yxxkt5qbmDlyiliWwLztAzGu+oUnf0LtxKkWPCjmmgAAAAASUVORK5CYII=',
date_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKxSURBVDjLdZNfSFNRHMc3FVNEBTWFFH0wTAVf9NEeiiLNh+ohKZIykKSHegssJXqZVFbiKsOUNAkhMu7wz8zgqjk3LndbOnVuorPdbboZsc0/zKs4+XbOdU4jPfC9v3PO73w/53cu58gAyKhIk+fl5Z1XKpX6nJycUjKOonMhRRUWFp5ua2vj6Bo6F/bRj40EjsigVsPtdoPv65PGtpBof3xwUMr9HBgI50Iby+R0MHXsGCxaLVwuF2ZINEVGwhsRIWlCLodFp5NydM1ECBCqbhcwFxcHprMTgiCA6ejAeEoKHElJcCQkwBgbC6a9fTdHovUwwJ/ERAzl5qIlIwNsVhYWMjNhT0/Hr9RUWAlIHR+P1zEx6I+OhptU9B9gPTkZv4uLsVxUBE9BAVwE5sjOhp2AFgjIlpYGW0kmZruuY76nHqYPFzDxJr9SAvA8j2AweKS2t7exInTDM1yDlal+YH0J/qmv4F6WLh8KCAQCMBgM8Pl80Ov1MI02wWt+gi1XLzw/XmBjmkFQ0GK6o1o8FEDNKpVK2nndPQSf+TF2NrTYWLgHr/4+rJ+qMNl+B9rGy6f+AVDDHoRhGKwusfBNP8KOyCFgq4TorMCapRZC5xXwQ6r9n3gQsKeej3XwGGpC5lsQHTewaq6D7X0Z5g3fwXHc0YAlcw9mBh+QsjUQ7VXYdN7ECqmEmjf9i9J9oMcMX2UKIMYdarZoujAzUIvJb01Y5MsRsFfAa3qIudZL2PS54HQ6qRFGo3EfwLIs/H7/KoV8eX4OWBMw23oVw7UnoGu5CN2rElj1rLQzbaIoSkcIA5qbmzEyMoKxsTEoqguwxb3FFq8EX38W3XdPYlT9WcppyTvQkTeh0WhAPWGAQqHQkmcMqmtnjuPp7Xywz8rQ21iOd40K7OUOqqGhgZYj+wu8wrLULIN/YAAAAABJRU5ErkJggg==',
date_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKDSURBVDjLdVNLTxNRFP5mOrVT2tLWSgIhEVEsaSUNrSujJi5cGBLiwoTEv+HeDfEPsOmmO6Ou0IUGQ4huTAwPrSUlYDtQ1LRKQKBmOgy2zNzrudMHmMBJvpxzz+O75845A845BEikWCx2d2pqaikajd6jsyJ8LSipVOpWJpOZFznC165zjJIk8d8ia2YG/ckkytks2Pg4etAUEfPMzqI3kcDW8jIaY2NO7ArnkiyRiCSfxwNfMAjbtuEPh+F1uRCWZQcqpaiBgBPzdndDbRGLWrllQ1UUFEolJ6m4vg5GJEYoBMPvh62qKBSLTqygaR0CtN+4AbAg3Z7v68O6YWCIbh+kp8lUwI+OULcsbJIuES4zhhRpk+JDgKyc7GAkEsE1nw+80YBFSUyAyM4R0bDbjShpieCp1WASaaeLxcVFblnWmdC35nl56TE/0He4EOETNaJWxilSr9eRy+Wg6zpy2QVsrT5DVzgB8+dse+QdOZVgbW0NlUoFAfryA8EKzvffQHdfEvulORzsFkAddIj+I2g7k7QLwm7UyqhuzCBwIQRbf42e2EPsrLyAaRpnd3C8mRx7xZeIXL0P/P2CpafP4Q8dQv+xgO3C+9M7OCldrALb/INA2ASrfyM+Btv4hME7j3CoTUORbEHCZd6iIsU6XTALl9xfEbo4Cmbm6XyI6w+GwRq/oKoaIoO3wcrvjjuo0VwNEjKZ2Oz94lt4w1F0BXbBrT0algvZV5oIg/3V0BMdQH1zDh8mR+LOz5ROp3k8HoeiNPfKvfIEiYk0XPJ3Iqg63+Pk4FzeYWyvLkN7MzntVFSr1Y/5fP5mO2W0oeNzZoJa503Q6zhjrXNTg9lcYlbvPxXdfEoTGuB5AAAAAElFTkSuQmCC',
date_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKDSURBVDjLhZNrSJNRGMc3HeoQL1ObhEMhcWyB5IWScn1KRPoSBBFUFBmpJOQXv4RhXlasRbZFXmDkXLowIxesrSGTodvaLWcEKdTMyWAtQteFsWSLf+e8zlHo6sD/fd7nPOf/e877cg4LAIuKDLZYLG5QKpVuoVDYRHIOnUuIU1NTI1GpVA66hs4lffThI8FB5DEYEAwG4dLrmdyXEH33mkxMbcFoTNYSjVlsmrzNzMSSzYZAIIB3JL5JT8d6WhqjRTYbS3Y7U6NrFhOAxO62AO+zszGt0cDv92NarYa3qAhrBQVYy83Fay4X06OjWzUSl3cDfMnLw6xIhBGBAOayMqyUlmK1pAQf+XwsE5AhJwf3s7LwIiMDQbKjHYAfhYX4XF+PUG0tPlVWIkBga+XlWCWgFQLyFRfjA9mVj8fDOofzN8DlciEej6dULBZDNBplRAedo56UgEgkAo/Hg42NDbjdbjgcDlydkOCyuho/Y9H/A6hZp9MxnTc3NxEKhdA6Vode/RmcHhHhe+QrnE7nTgA1bEM6tEdJ18Noe3QQzQ+rcF13ClMeJTqfnkCjYi/m7JbdAdtqH6/Dc+8Qni08YIyTHgUU5k5oHHK0aRtxSMbF/i52fkrAJfK91Dww04HbpnbcNLagR38RfYZWjMz349yYBBU3WHHmKFMAMf36E3BWJcYTtwJa511oXskxapcRyBUMz/ehRduEA1IeSq6xqhiA2WxGOBz+RiHb/+Dk8D4cVwrQcI8PyZ18nFcfwdBcD5rHG1ErK8bUzGMkL9Pg4CAsFgusVmtKVd/i4oLmGKr792Dy5QSoJwmQSqU2co3xL1V0Z6C0Kx29A91MLpfL/dT7G1NDhMW9KO0jAAAAAElFTkSuQmCC',
date_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKOSURBVDjLdVNNTxphEH5YVhZFRCCNmB6UYmgIRSJJD6QcS0QTT7145NT/4KUXY4yJHjzzB/gDJl68NE0IJCbaA+GzJBQBQ1RU1ALLbmfeAqmJTjJ5331n5pl5Zmah6zpYSQw+n+/z4eFhxuv1xuhb5rehyqFQKJJIJFLsw2+jOHEpGwx6i72OjvB2ZQW/T0+hbWzgDf4J25TjY7iWl9E4O0NvfV3YPLpukAwk7GRRFFhsNgwGA0zb7Zg0GmGXJKFmcjFbrcI2OTMD8xCYY6XhHWZZRq5cFk75YhEagXRmZ9GZnsbAbEYunxe2XKEwBsCIYwnQbJT95/w8ip0Olii7m6hJFKD3++iqKn7RWSZ9p2kI0flI9iVAkv+v4IPTCb/FAr3Xg0pOGiuBmQjo/cQEvHQaSJX7ezwS6LiKdDqtq6r6qvb7ff3p6UkoC79xDMfKeEG63S6y2Sw8Hg9KpZLgfnt7i9XV1dHIxyK9BMDBtVoNVup8MBiE2+3G1NQUKPNYR0DPKuBHzn59fY08df3m5gZXV1dYXFwUQLRkcDgcwm6nKYXDYcszAOKKZDKJh4cHxONxcXIlqVQKzWYTG7RcMjW70WjglJbN6XR+ewbAjpVKBV82N7G3tyf4u1wuBAIB1Ot17O/vCyBaa0SjUVSr1a+SPiRDh3Z+fg4LjZFle3tbVLRJYJFIBGtra4jFYujRiLe2toQP7Y4iKrinuXZIKJs1k8mI1T44OBDd50pGoEyJ8+3u7ooKLi4u/ggAbpjRaJxhfu12G99PTuD3+0Xpl5eXKND6KvSvLCwsYG5uDhptIyViWk3xN+7s7PygkX3i+93dHZgKNQgmk0lknqV/gu+tVktUwDTortI0Pv4FgnWCE6f7EysAAAAASUVORK5CYII=',
date_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJZSURBVDjLjZNfTFJRHMdBmX/m1Pln+qDTNjcmD77oo62trYd667H10nNPvrbWIy+u0FFAOhrMzWZKXaYE8YAx4+Ll8kfL/NPoIuBdmAVjCQQb1LdzLkg2wzrb5557zu/8Pud3/xwZABmFNLlKpbqi1Wr9SqXyKhkr6FwFxejo6EWj0cjRNXSumkcvAuk4QsBuRyKRAG+zSWOhAr3fcDqlWMjhqMYqG8vkdLDV2IhdloUoitgh/dv6eqTq6iQ25XLser1SjK7ZrAgq1ZUF4ZYWMHNziMViYMxmbHR3I97ZiXhbG4LNzWBMpnKM9Ht/E3xtb8fq8DBm+vvhGhxEZGAA0b4+7Pf0YI+I7K2teNjUhJcNDUiQis4IMl1dOBofx+exMRyOjEAksvjQEKJEFCEiobcXH0lVQkcHUgrFnwKe51EqlWpSLBaRz+claKNzNKemIJfLIRAIIJVK4smSA48ZN6YtPPTWEGYsr1HIZ88X0GSr1YoF+xosngjCiSyiyQL4yDFWeBHzK2/g8/nOCmipJxKGYaBn/NgRj+GPfcf6fg5ewtanPHRL3tqCE6hAs+iD8KWA9UgWrJDF6ocMQgcFPHoRBMdx/xbonvPYFjNYC5eTKUGxgPvzrPSY1V+ZCkjSj9MCyuIrDq7QAYKxHLhYHgGyu+1dBtOzC7A/1f0WuFwupNPpb1Ry+mUmjpJwsu9JJRwM5AtonvGYmrUgsGyAffIm7l2/cEMS6PV6uN1ueDyec2HJOWBMU3BobmF7+QGME5dykkCtVrPkGON/mbx7G+Y7134aJi7jF1A6sIfsK39SAAAAAElFTkSuQmCC',
date_next=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJCSURBVDjLhVJLa1pREP7u9ao3PmLEBEy7sorFtkQUpIuWdlOwBOy+m+zzN7ovRejOZemm+9CsC6FYCIiI+CwUbUJoqVo1Pu7jdOZUxUKSfjDMvWdmvjMz54MQAmwEJZFIPMvn81/i8fhz+tf4bGFaOp1+XCgUPnMOny3r5EdbUcQPzjo6wu1UCp3TU9i5HHbwFxxzHx8jvLeH81IJ8/19GYsKoagKgZO8bje8gQAsy4IvGMSGw4GgqkrTKUX3+2VsY3MT+oKYa9XFN3RNQ63dlkn1ZhM2kYy2tjDy+WDpOmr1uozVGo0VAZYztgA7QLeXd3fRHI0Qo9sjNJpKBcIwMDNNfCXfJrtj20iTv6R4DFC19Q4ehEK47/VCzOcwKclmIzIXEd11OhEnr5C5h0NcEumqi2KxKEzTvNYMwxCTyUQag8+4hms1XIHZbIZqtYpoNIpWqyVnHwwGyGazyydfQb2KgIu73S78tPlkMolIJAKPxwO6eWVLon86kMKgJ0uxFjodLMVSqVRAAsKLNzWotPbDp0FsX9fBmjIl+v0+bNo87QEUwa2QC28/XqDUvWGEdTIeh6Qt92DPTLiow3DQjQ8nNp4cfnqpisV15Oz14hxJmUOZTAZeelruwpqaMOYGNFVgZ9sF3Yn3cgdDetcRwUdYVyfvw0E6YM8dWGMimFoY/57jrNGDObUOJEGdZEqJm5qm3TQRrImJYU/DeeMXHkYv8PrVwTtZ0ev1Tsrl8iP8B9b4Hr5XqTh2hrD/5zc++wOzk1RA9fA6WQAAAABJRU5ErkJggg==',
date_previous=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJiSURBVDjLhVNNTxNhEH52u7RLFyi1QiCGNLWkptFAqGk86E0TDAl/wIM/gLOJBy9evHiy5x71RLyS4M2DxFRDQ6AS+qWRgkJMdAm17XY/Xmde2AUTjJM8mXffmXnm452FEAIMEiWbzd4rFAofMpnMffrW+O4UWi6Xu1MsFt+zD9/5cfLQVBTxg71WVnBlbg6t9XV4i4sYw4mwLbK6iomZGXzf2EB/YUHa0kIoqkLCTkYkAiMWg+u6GIrHMRgKIa6qEjq56MPD0jY4MgL9lJhj1dMzdE3DTrMpnar1OjwiaY+Ooj00BFfXsVOtSttOrRYQwO+xAXgxyr45OYl6u41pyp6i1lQKELYNy3HwmXSTcNXzkCPdIfs0oGrnK7iRSOC6YUD0+3DIyWMQWZiIrg0MIENaIUSOj9Eh0qCKUqkkHMf5J2zbFt1uV4KF7ziGYzVcIJZlYXt7G+l0Go1GQ/Z+dHSE+fl5/8kDUS8i4OC9vT0M0+RnZ2eRSqUQjUZBmQP4REEFbz+ZeP56F27fxptnN9FqteAvS6VSAS0QqBVZjXOuf0mw+U3BcvkLJhJhtJq//yrTNE14NHkOPjw8lGdGUMHdJx8fLJc8jI9FoNHCOJ0zdnbidmi1ZeZkMsnLI6vT6NV4oKoRxqtxyqypArblwO1YMniRVpkJ8vk8DHpazsrB+/v7AZEconnQfdiqmmj/tGD3XLg9Rxp9hGgPWHMFnHlqagq9Xk+2BP+vevT0pVirXEYiaeCg9gsvHicuepyAlMm2trawtLR0sgfxgd21Wxn79rvKJVlquVzG/4R25SvrPyvWchiOM0HOAAAAAElFTkSuQmCC',
delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJdSURBVDjLpZP7S1NhGMf9W7YfogSJboSEUVCY8zJ31trcps6zTI9bLGJpjp1hmkGNxVz4Q6ildtXKXzJNbJRaRmrXoeWx8tJOTWptnrNryre5YCYuI3rh+8vL+/m8PA/PkwIg5X+y5mJWrxfOUBXm91QZM6UluUmthntHqplxUml2lciF6wrmdHriI0Wx3xw2hAediLwZRWRkCPzdDswaSvGqkGCfq8VEUsEyPF1O8Qu3O7A09RbRvjuIttsRbT6HHzebsDjcB4/JgFFlNv9MnkmsEszodIIY7Oaut2OJcSF68Qx8dgv8tmqEL1gQaaARtp5A+N4NzB0lMXxon/uxbI8gIYjB9HytGYuusfiPIQcN71kjgnW6VeFOkgh3XcHLvAwMSDPohOADdYQJdF1FtLMZPmslvhZJk2ahkgRvq4HHUoWHRDqTEDDl2mDkfheiDgt8pw340/EocuClCuFvboQzb0cwIZgki4KhzlaE6w0InipbVzBfqoK/qRH94i0rgokSFeO11iBkp8EdV8cfJo0yD75aE2ZNRvSJ0lZKcBXLaUYmQrCzDT6tDN5SyRqYlWeDLZAg0H4JQ+Jt6M3atNLE10VSwQsN4Z6r0CBwqzXesHmV+BeoyAUri8EyMfi2FowXS5dhd7doo2DVII0V5BAjigP89GEVAtda8b2ehodU4rNaAW+dGfzlFkyo89GTlcrHYCLpKD+V7yeeHNzLjkp24Uu1Ed6G8/F8qjqGRzlbl2H2dzjpMg1KdwsHxOlmJ7GTeZC/nesXbeZ6c9OYnuxUc3fmBuFft/Ff8xMd0s65SXIb/gAAAABJRU5ErkJggg==',
disconnect=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKuSURBVDjLlZLLS5RRGMbf7zrzebdw8FaSqJAbF0GgRBvptinoPxDKRdDKQjBw2TKClgVJCwmUyMBKydDKS9TCME2tRhs1cZwZ57ud853bfB0HiiAVOvBuDuf5ve/7nEcJwxAOOgPTtk4Fr6ZU1OCAVyBCm2Td9jEdcxG5pBwEeDyZtaRwjAvWSpkAxjkITsGKqJBIuvB903upH9QdE3rd1MLW0gIVCMsBoYq8U8H2CUQNBdJZp33fCe6PbJo+4XZVmRHFhEDKCXaB4Accii0NFlfT8GNt56a6X/er56qog/Cd1aQjRRQictasR8B2EXgIQyblAbX95X9WeDSRPiriz3oZY1pvZ2dH590Z7GB+q7LcjBZHVdhMBaCpEfBsCXDR9p8V+t9lLGlUP7PXLxyJbkUMw4DZ2dm+rq6ujjPdz09xTEZrY8VWYZEh/WAwNxWHwEV1eYDsqsuOCxFDaYRQwGZ8ljeUE31+fr4PY3xFVulM5mQzC4LRypoSy037kEykvuZytDnvAQ5oNSa8scAE0JQcGIeb9LcrJl02Tj+U4gcIoanG8MU35qKzK58SaCux9ZSLoGVxvJvnPfAQrQEQEhTCRhpBYVQB61CNyZY+v6qvrzdisRgMDg6O1+kjbUt+23EpTPz2LA9wMa7QFJBuhxIWQHKHQWmBDrXHGozJuTfQ4sWBEDI9NDSUkc8zf5ueB9gubiqyVJBacBDZXQm2MhSiugZW7QkYfj/NuGi5ttd3a9uxi6bM9FhFmak5fgCmHEXqQFcVEDkBiZVt+edhz8fh7om9AHrWxV5JgWoImXMfE1jbsMHd8QF7AQQyONjxp4UQ9/YLnJ710JgaGucXUi6sr2cY84MeQfmyCOg2p3RD5PjPL69v8H0ByEWXnSR7IoPSzjEt+jDQQeE/zi9kq6pv7shelwAAAABJRU5ErkJggg==',
disk=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH+SURBVBgZBcE9i11VGAbQtc/sO0OCkqhghEREAwpWAWUg8aMVf4KFaJEqQtAipTZWViKiCGOh2Ap2gmJhlSIWFsFOxUK0EsUM3pl79n4f12qHb3z3Fh7D83gC95GOJsDe0ixLk5Qq/+xv/Lw9Xd+78/HLX3Y8fXTr2nWapy4eCFKxG7Fby97SnDlYtMbxthyfzHO//nl85fNvfvnk8MbX5xa8IHx1518Vkrj54Q+qQms2vVmWZjdiu5ZR2rT01166/NCZg/2PFjwSVMU6yjoC1oq+x6Y3VbHdlXWExPd379nf7Nmejv2Os6OC2O4KLK0RNn3RNCdr2Z5GJSpU4o+/TkhaJ30mEk5HwNuvX7Hpi76wzvjvtIwqVUSkyjqmpHS0mki8+9mPWmuWxqYvGkbFGCUAOH/+QevYI9GFSqmaHr5wkUYTAlGhqiRRiaqiNes6SOkwJwnQEqBRRRJEgkRLJGVdm6R0GLMQENE0EkmkSkQSVVMqopyuIaUTs0J455VLAAAAAODW0U/GiKT0pTWziEj44PZ1AAAAcPPqkTmH3QiJrlEVDXDt0qsAAAAAapa5BqUnyaw0Am7//gUAAAB49tEXzTmtM5KkV/y2G/X4M5fPao03n/sUAAAAwIX7y5yBv9vhjW/fT/IkuSp5gJKElKRISYoUiSRIyD1tufs/IXxui20QsKIAAAAASUVORK5CYII=',
disk_multiple=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJFSURBVDjLpZPNS1RhFMZ/5733zkzjR/ZBCUpoJdUiBCkll4m0CUKJIGpVSLjyL2gntDFop6shAolWbcSNIW0ircHBUHCloo3VjNY0jjP3831bWA5ai8Bnfc7vPOfhHDHGcBjZAENji7N1cSj7IcdqY2zkKoiC2qSFNsKPYoXpTPbBynj/4j8BlbLL9c4L3OqoZWLmM4/vXdpX9OJtHq0lBXQdBIgxhvtPZmZ7ui+yspZrjwKfWExxtMbh66YLAgj4geZnyd2YzmT7Vsb75/c5UEqwDLgVl55r57hxuYY3c18Y6mtDgO1KSBBETMwV0VpeA2f3ARKOwvUCcgWX9bzH0NhqvC4Okx9zBzNpPdGQ4OHIrJnOZLtWxvs/2AChNnhRiFIKy8j/ZjILiALYLgc4YnO8zsJSIWUv4Pt2CMBU+tteoxtC0YN8wUdEV1eItMHCIdSagru5l0kQaZ4OdqC1wQAWhqQNnudR3PGrANu2aGmE9FJATSxJwinhegHDr1ZRAmGk0ZHGAMYYMJB0dh0ogOVs6VNqcoGtosYv1+9lYikHERvBQsQCozBGCMIQ3w+rDtKjvQMAd4bfL59vFqYzQasjNoM36wi1vzvHgBFNwo4x8nKNreJOFfBHy9nSXGpyoSPSYOGgqZCae8TJ5BkERb68zsDVZygSlD3/b0B6tPf2byempRFO127T095JQ6wJFBTcJk7VhCRjYItUT/mgrgxOvWtrPtLdEG8gYdcT6gDRGjERWsosrS2TKwbMP78rcth3/gX/0SEvLZFG1QAAAABJRU5ErkJggg==',
door=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEuSURBVDjLpZM9SgRBFIRr/EHXwGVFAxUPYmTgMTyEoblX2NwjGG9i5B0EMTIQFWTBaHTeqyqDHmdX0FlhHzSvO6n+qvp1ZRvL1AqWrLX5w93VuSXBJkhCKovMrpOJk4vr6lcBidg7PgVMWIYkWATEbv9wc/03AZkwiY/3J7i93STcEmxu7yOz6ReQCDFhFTExIRJWgox+gcwot2UUAmY5kzADzkBEL0Er0PUZReUGIhcRNAU5muI/E1JiZzjF4cEbHp+HyIx+C2otdPgqfTodoP5c/w9BgNn8sPD6sgFzFVu76ieIaGYhMtrwWiGVPRkLCLJkYJX0vyms8rQLMigWBqMj2IKZsFSWCbGM83xVtjGZTFzXNeL2spv3+fmf/QnCJu5HZwCA8XhcVct+5y9H3H2zjxE/HwAAAABJRU5ErkJggg==',
door_in=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJHSURBVDjLlZPNi81hFMc/z7137p1mTCFvNZfGSzLIWNjZKRvFRoqNhRCSYm8xS3+AxRRZ2JAFJWJHSQqTQkbEzYwIM+6Yid/znJfH4prLXShOnb6r8/nWOd8Tcs78bz0/f+KMu50y05nK/wy+uHDylbutqS5extvGcxaWqtoGDA8PZ3dnrs2srQc2Zko41UXLmLdyDW5OfvsUkUgbYGbU63UAQggdmvMzFmzZCgTi7CQmkZwdEaX0JwDgTnGbTCaE0G4zw80omhPI92lcEtkNkdgJCCHwJX7mZvNaB0A14SaYJlwTrpHsTkoFlV1nt2c3x5YYo1/vM9A/gKpxdfwyu/v3teCayKq4JEwT5EB2R6WgYmrs2bYbcUNNUVfEhIfFYy69uci+1fuRX84mkawFSxd/4nVWUopUVIykwlQxRTJBTIDA4Pp1jBZPuNW4wUAPmCqWIn29X1k4f5Ku8g9mpKCkakRLVEs1auVuauVuyqHMo8ejNCe+sWPVTkQKXCMmkeZUmUZjETF1tc6ooly+fgUVw9So1/tRN6YnZji46QghBFKKuAouERNhMlbAHZFE6e7pB+He8MMw+GGI4xtOMf1+lsl3TQ4NHf19BSlaO1DB9BfMHdX0O0iqSgiBbJkjm491hClJbA1LxCURgpPzXwAHhg63necAIi3XngXLcRU0fof8ETMljIyM5LGxMcbHxzvy/6fuXdWgt6+PWncv1e4euqo1ZmabvHs5+jn8yzufO7hiiZmuNpNBM13rbvVSpbrXJE7/BMkHtU9jFIC/AAAAAElFTkSuQmCC',
door_open=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGOSURBVDjLlZPNapNBFIafSdOvtbFgSi1dREEtguDSnTfStbgW9A6y9BICinfkRosRFw1mE5BoS4rNzPlzEfOrYjJwOGfzPvO+h5kUEWx6zt6+eO1ur8x0VN9E+Ondyy/udlLdPua8d8ZBrdIZoN1uh7szLTOb9WePgxpOdXjMzXsnuDlx/gGRzAxgZrRaLQBSSks94iPNJ0+BRL4aYpKJcER0GbAqns5mhptRRgNMC1Aj3P50sChanFULboJpwbUAiXCnlPEcoKr/BJgWQhWXMnEQE4DKmNrfHKyW/L7ZJBNyzVGzR4RSSp4DFh2sOhEpmCpWMo0bPzi4NWR76xqR/0SYA8a4ZkwyF9+3cD0kl8HyEqeA1fwpJUrJuAouGRNhmOvgjkhZD6AynuxABdNMSnXcHdU1AUXyRCwZl0JKTsQGAJFJhL3mHVwFzT8hBpgpqdPpRLfbpd/vL73/xX56v0djf5+d3QbV7h7b1Q6jqwu+fn7/La3znd88v3tkpg/M5JGZPnS3Vq1enZrky19GcE/tIr8QhwAAAABJRU5ErkJggg==',
door_out=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJCSURBVDjLlZO7a9RBFIW/+e1ms0kMmOCbRKKGaCBBUCsttNM/wU4UsRQUQQSblIKWFpGIiLVYWQgWsRIRxFc0PhOMhSjRDZFkZ+5jLFazWWx04HKq883cw5mQc+Z/z9T105fc7ayZLpb/x/j6xpl37jZYWb+JmdkpeouKrgDGxsayu/NnzGxFT4xkCpzKuk2s2TaIm5NnXiASWQGYGX19fQCEEFo055f07DsABOLPeUwiOTsiSrEakMlM1u+tmP+MmeFm1GufkaUFXBLZ7e8X3F++y0KqETqbZgDVhJtgmnBNQCC7k1K9CZjufcqWjZvpsbXc+jiBqaFimBpX+/eQVXFJmCbIDYDKb8CRK4eeD/QPMDo0irqya3An4oqYcPv2HeT3zSaRrHU2rv/K+6ykFCkfvnzw5sCWgdHRoRFq9RpLsoSYkFzoKq9B1RBJmCqWIt1dP+hdO09baZlFqVPcO/fg2JuPb6cePXtMEUq0l6pUyx1USx1ES6gYInVcIyaR2vcSs7PriKmtGeLkxYcjB8/vz8v1ZVSVDx9mMHVMDTcnpYir4BIxEeZjGdwRSc0Qt3/dyUx4S5FLnNt7oaUL+upaIwMVTCMhlHF3VFOzB6rK8eFTZMstHQghkCQ2zBJxSY0e5AagvBpQFAUndp9q6UAIAZHGCp09/bgKGpcgf8FMCePj43l6epq5ubmW/q/Wo9tn6erupr3aRaXaSVulncWfNT69efIt/Mt3nji5dYOZ7jCTYTMdcre+olw5ahIXfgHcTaP3d3vNvQAAAABJRU5ErkJggg==',
drink=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJGSURBVDjLpZNLTxpRFMdd+EX8BixZ6VfxQ9imq3YzSdNVE2MNyEMYAeUNofIKEBjA8CoiM4BCgiQgaJqWtjySAc2/585iogG76eI/czN3/r9z7jnnbgDY+B8pj3w+v5nNZncEQdhLp9N8KpUqJhKJYTwel2OxmByJRIbn5+fFUCjEB4PBPZ/Pt+PxeDZVAJm5SqUCURTRarVUNZtNdd1oNFCtVkHBEA6H4XK5OBWQyWQwnU4xHA7RbrdRr9eVn8vlsiK2ZnC2NxqNMB6PcXZ2BhVAacu3t7eYTCYQbr4jIP2ErzWHt/0I780jnOIUjsoDYlcDjH//UYAOh0NWAXTmbTrzUmpew3bRA196gONqAndrARfJevkLXzJ9fI5dwxkvwG63L09OTrZVABMVTBuNRpfVegPWlIRPvhI+nF7gHZ/FG4sAzl2AP1V8YX4BYKJKa6nSy8srEZakiPeneby1CvjoKeJrurRiXgEwUZu0fr9/+a16iVStC9/FNSLCevNaAJPX69W63e6nxWKhdIfMT+vMrwKYnE6nl7WtVCqB53nbPyfxuSjyFvV4l9pU6Xa7yOVysFgsebPZvGs0GrdeBdBoami6ioFAADQXoPHFYDBQ3lQXUHFxfHwMnU5XPDw81KwAyPxDkiTIsozZbIa7uztFDHJ/f698Y3vJZBIHBwejFQClzbFIhUIBnU4H/X4f8/lcUa/XQ61WU+7A0dER9vf3ubU1sNlsGqvVylGqWZPJ1DEYDNDr9SztLqWdpcgcmTXPPX8BpLUNr3FYePgAAAAASUVORK5CYII=',
drink_empty=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFDSURBVCjPfdBLSkIBFIfxM3Aj7sCho9yKi3AJdwUiYk/LTMyyuohZYmKZkD0uptUlDSt8BWVmPuB2ka+BYdog/sPvNzgcQf6fIDlL1nHiygTS+VQzaRwaiWY8Hwuorh3HtgURJKtcUkJHR+cOHZ1bNHLsE1EQQY7p06RMEY0LLtAoUaZFhzCIIBnjkR4fdHinzQvPPFCjh0bIQARJz6XMMj06vNLkkXuqfKKxYa7PIYIgSfuBqdOjxQNFbulO8g9A4vaYeUOXa7SZPAGIat81C7zR5WoqTwEkat8afdFnY/SbZwCyGS1zTiD455PjbVnDztBllVNWcyvOJesM2LZF8ntkSFAngUoMP7681zYBkfYNBgMaNKjzwgCDIzytCQgrKmdUqDFkyBMF9lnArUzdELStKf7scmWReXxVb9ajuG3j8g3ex2UlnmJ3nQAAAABJRU5ErkJggg==',
drive=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADsSURBVCjPpZE9SgRBEIW/XnvWwB+ExUAXBDNvsTCTeAtP4AkEz+MNnGSHuYHBRgObCAaCmAiCU93VZdC9g5HJVlBJvfe+osoZ/9eMfQUeYFhrrUQCgYAgCCPS3TXgjGE178+RyZO3quh5be47D/HxjHcOAStD441Trhke6NxmNe8XfJXgiOE4oOKEI1q2jXtZX9SGYCXcpl7xyVPnhW8+GDESSsJIJBTDc4zgpQZB2eCYFa+RuEJZIrUfiUQUxZEKZpcRCPgRRUnckEjYH4yRsmCBTsHZvwNdEvE/7fOtMObbIUi5Z8Za6/b+5i8QHogENhlMkQAAAABJRU5ErkJggg==',
drive_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIBSURBVDjLpVPPaxNBGH2zvyGbEAw5aEogByE9RG9ScxESgvUi6D8QxJuYk5dCwXMv3rSHXrwqHlqikFBqXEEkx2KMEppLIGokSjESu/mxG+ebMsvWi4UOPL6dme99730zO2yxWOAsQ8EZhyY/Op3Oa8/zChyYz+eYzWYBptNpgMlkQrFRLpeLxGPUAidfMwzDSSaTIunfEW5T13U4joNer1esVCoN4YArPozH4xgMBjBNMyCFifTd7/cRi8WQyWRIdJ0vN1ir1RLqiUQCo9HohF1qhYiMMaiqKtSj0SgikQjq9Tq63W5RI3WyTr1SkmVZAlI97ISi7/twXRe5XA7tdntdI6XxeIzhcCgOSCbRYVKUc7lGc03TYNu2cEkFClSdJpTAWxKWFUU54UAWSqfTIi+VShGnoJEq9UqgDQIVoOTwDRyyL/ilf8PnnoMj9w8uja+KtkUBSSRSNpsNLBOZ1ruH+1BjP1FazmPp3EW8ae+g+XEbI8U+LkA3IPsL25Xq1YMt3LqyCk/xcPl8CXufXmAll8fzg1fQ+InWa7XaaugvC6LED/M7dGbjxvJdUfBBcQsvW5tYsOrxn/i/ka8sHd2+ed2aw8da6Sk2du/AUk082666p3pM/sJ/1PzwHgZ/OqRsMA3v9t/S1hN22ue8cv/CBg/3OKIcvzk2m4+/rv0F4ux6ZTGbxgwAAAAASUVORK5CYII=',
drive_burn=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHySURBVDjLpVNBaxpBGH2z7rrb6poNG0lSi0VyiA1I6a2XUtBLyD/oyV/gvSD03Gv/Q7El5NZDhBDx1lPpQUyRbg9SoUYTsEs21XVdM9/QWdZePGTgY2beft/33pudYcvlEvcZCu45VLno9Xrni8WizANBEGA+n0fh+34Us9mM5la1Wq1QHSMLvPhVMplsZ7NZkfT/iNvUNA3tdhv9fr9Sq9VaQgFnfGtZFobDIXRdj4rihbQeDAbIZDIoFApEWudwi3U6HcFu2zZc112RS1aokDGGRCIh2E3TRCqVQrPZhOM4FZXYSTp5pSTDMERI9rgSmsMwxHQ6RalUQrfbravE5HkexuOxOCCZRIdJs9xLjPaqqiKdTguV1KBM3WlDCdySkKwoyooC2Sifz4u8XC5HNWWFWMkrBX2QITGJP//2CdbVT1gTB896H6JfrFIDWUQMxWIxkkysEn/y6wJP/3yFzTYxe5TGFceiBvQHpL+4XHl45uf3SO15sPU9oMxg9D0cOCdCHWs0Gqfcy2HslkUzxaF9jH3NwcbONnR3Eziii8Mb/7jF98nDS7buMf1+wRraG7w2sQP92gJecnCDx5jf2Hc3H9c/Jh+j5Rnwd3fELYXA5T/8SwC4GK1X8Jg94E9uAhNJxVeC7ewWYHDkhrcOYd0B0mCFUhT4PX8AAAAASUVORK5CYII=',
drive_cd=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJwSURBVDjLpVNNaxNRFD2TzOSTpFgZCg0JJPEDCaK48wNaEzS4cOlCsIgEXHWrCC3FRQXBP+BWCrpyKcZImhZmYdwVUaolIWXSGGPJpLHNJJnJjPdNnTBFXDlwuPe9eefcc98HZ5om/ufj7aRcLhdHo1Fa13VQBItOaJrmzFdzuVyG8TjmgMgzfr9/LZFIYDgc/lXF6VIQBJRKJciynJmfn1+1HJDq0uTkJJrNJrxe75jkJLK8Xq8jHA4jmUyiWq0u0PQqJ0nSTCAQWIvFYuh2u5YDG8wuI3IcB7fbbVUPhUIIBoMoFouo1WoZnvpZEkXR6pEt8vl8FuzqTicsGoaBfr+PVCqFra2tBZ5VUVUV7XYbg8FgvIhtJIv22J5jY57nMTExYTlkAmlWWVEU9Hq9I4R/CbAWIpEIc53m7eMpFApHBJyWnTmLjMxOzHaAr419tKayWLw7i82WjoDHhV/tH/jwZQf13X1cPR/FyUQc7Z6BM1MCll+UIFV6loCLOagpJu5lU2h0DfSGJrw8Z5GvnRLw4LqI9Q0Zfg+HgW5iW9GxNHcR8h7GAvnKdh0+Dw9tdLjbLg6YPRfF+28anhV+YoZydXj4TxsBfc1Eq9Vie5K3buLtp+/uXElNr1y+cBq6YWI67IZfcEFwgxwZ2Gho6Kq0kaQRPcbThZLxWqo8Jr0nrj8X7a30uZGX6zs4cZy3yCMSPqCqHdWEcmAg7HNR/zw6u01G/kicl68eZXXOviTkIk5hmXDr/o2UcDYuQvAIGJDd74qKzl4Xz9980lkxwkMib44fk/2RiEDhEmGOcJMgsjUEmbBOWCGsEXn84n4DYCewx+p/CvMAAAAASUVORK5CYII=',
drive_cd_empty=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADnSURBVCjPtZG9SgNREIW/604wMaig2CRgEUsfIrLb+B62vkEgr2UQsklhk2cQRNDCSsHC6N6ZOxa5rKm0cgZOc36Kc4Lz+wnAam6lYmj+uMH6qoLgrMa95Yim9WwyOyx4qq5rAZ0e8cIu4Jl0njngjIcJdbgb7y1PeaehoUFxAgUd9ukz57GSOD0hUtCli7cpTuKTc+4noqx55QsnYSScRMJwhEMU0bLgjY9MbAv6DImlRJTbLGALnSEjFFGMC9at30hAwumRUCRyjOF49pEzHGeAInF2cxl/2mvbNBTDZuGvLXb4d8E3CxSGOVhPWycAAAAASUVORK5CYII=',
drive_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIGSURBVDjLpVNNaxNRFD1vZpJMmo+mhmBLZOq4sAHJL+hCSECLP0C6i65cBcWVUBAX4k/QlfvuVRKCDUNpEVy4MKQYTMHUpA5EUpiSNsnMJM595Y1TNxZ64XLfu3PPPee+eY/NZjNcxiRc0hSxaLVaW67rFjyH4ziwbdv3yWTi+3g8plgvlUpFwjEawQPfDofDRiaT4UX/WnDMUCgEwzDQ6XSK5XK5zhV4jM9TqRRM00QkEvFBQSCtu90ukskkdF0n0g0vXWeNRoOzp9NpWJZ1Ti6NQkDGGGRZ5uyJRAKxWAzVahXtdruoEDtJp1mpSFVV7oI9qITidDrFaDRCPp9Hs9ncUIhpOByi3+/zAxJFdJgUxV7kaK8oCuLxOFdJDQrUnTZU4I3EJUuSdE6BaKRpGq/LZrOEKSjESrOS0wdyakDFwT+w2PuM9I8dSO9M2MkFHK/e52PzBgJIoFwu50smMOWjex+xhAOsPHyCiH4Lp19r2NvZwo2j6FkD+gNivqBcwT7YfI+bpcdQ9w1g+yXm5lO4ri1j5fsXKN6JViuVylrglvlR+NrvQ6iLOnDv6d8r/GIJ8/bJ2U38n9XuKD9X1x9cix18wPjUxImXO7ZkfNtH70KPae7K1bfNT9vOr1EUlhzC0YCh3WOuR/2aXfQ5765rz4aDw0eyy5ZdedbzUG/u1pxXfwB4L36S0dG+qwAAAABJRU5ErkJggg==',
drive_disk=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJJSURBVDjLpVM9ixNRFD1vvhLzxUiMCitiCjFWwrJuYaGQICgIgl2wSCWSIr9AFAsbQcVfYGEj9kICizGFgoUIIUENBDFrkLiDukw22fke7501MetHtY+58x6Pd+455973RBiG2MuQsMehzBa9Xu+57/tFCnieB9d15+E4zjxs2+a5WalUSowTbIHA5zRNa+VyuejQn2PRpqqqaLVaGAwGpVqt1owUEOMtXdcxGo0Qi8XmoEUgr4fDITKZDPL5PJPeoO2m6HQ6EXs2m4VpmrvkshUGCiEgy3LEnk6nkUwm0Wg00O/3Swqzs3T2yofi8XgUD+pbSJOYqeNhf0rDl41tCMlHKjFGEG7hh1mAZw7vKsw0mUxgGEZUIGYMggDb0wyKqydweTmFZ6+HuHn15K66PG4ZeBGcX5EoQZE3OBF3oN1uo9vtYmq7+D72osNrb77OgRZtmTZgbDpkLYDCrOyVgxNwsGdJIt9UQ9cPcL+6TKqoqJRApn+CSs84c+LsJJgBWXqhUIjmdz0JFqm48/QTKBc8ShRQhLO20pdQfyngDnCCWes4vPc2bN+DLKkMgcCOspCVhAEp5gvmQbEsq1Gv1y8s3LJoHrtnoQoF1UtpeIGzc+uZmHzHFQ33nqzjmzmBUi6XL/7rjl+5/TKkzoP6gUdvr+NA4iipkGBMP+Pa6Ye0ilOhnd9v4a9Hosg4dhg4lBqjdGoVurYUidi0lnAw6SGh0RmyJP73nFeqa6+OH9l3Ro/pJDlDNlwIKq4Iqdhiig/rH7FhuvgJMpVtkQoe5WAAAAAASUVORK5CYII=',
drive_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLpVPPaxNREP7ebna3bX62aZMmYnVLxZRSior1IFVIC1at3uo1ePRQ8CB4EDyr/4AHFS8SCl4FC9IQf1WRhmij2Ei1Dabp0rSxRsMmm23WnZWEWBQPHRjmvcd83zfz3htmGAZ2Yxx2abb6Ip1Oz25vb4dNh67rqFarDdc0reGVSoViLBKJjBKOUQsm+KQoivGuri4raac1tykIAuLxODKZzOjU1FTMqsBUvO7xeKAoCiRJaoCagbTOZrNwuVyQZZlEr5nHMZZKpSx1r9eLYrH4R7nUCgEZY+B5Hm3VBfDLT8A7BlHaeIq1rdpNlkwmZwOBQJgSCVRXbY7k3I8YWgpLaO0cgVs+hK3lV0jN3NdsBCqVSsjn89YFUXKtVgNdJkXaO2sJhHo0OA4cR+HzR4hMg9PVDXfnHhsRhEmJiAhktmSVzHG/X9jXuoKjwy3w9I6jsjYNsY0h83YRVZ1D9J2ZRqrUKzkR1J32HcISxoYFePrOopy7AyZ8g+BwgldzqOyfwGbJgI0I6iAqORQKWRGFNzjSL8Hddx7l1dvgBB1aUYbybB6L9tPwt/qtP2IR0AsQQf3C1NxzuPwqfAPnoCl3wYsGyt/3Yf1FAsWBy/BL7QgGg1aVLBqNPjb7H6//MmwkET7WDZ354O14iXafaIL34ms8gQefgthUuUbFptgM2zlMD2+NGZOX7iE9fQW5lTlIPUPQ1/O5dpE/NXj19fv/DlN6SdG1D48gD43Abj+I1fnUz7KqnfkbuDELzXbhRId6uNdvG+6XYQiVOXU9f3HixsKXf03jL2qvd7dZXvRWAAAAAElFTkSuQmCC',
drive_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJTSURBVDjLpVPPbxJBFP5md4Gl/HAJULHogRqUJiVpw8GYmFjhoDcPHIzRBO+Gk8c2HvwTjCfryaQcjDZR08hFwsWD1DZGQhSLaUk4UDkYUQSWZdd504Ct4dZJvrzZzXzv+957M8yyLBxnSTjmUkabarX6djgcJjlgGAYGg8EYuq6P0e/3KRYymUyKeIxK4OTLdru9GAwGxaH/1+EybTYbisUi6vV6KpvNFoQDrnhf0zQ0m004HI4x6TCR9o1GA16vF5FIhESX+e8CK5fLQt3v96Pdbh+xS6UQkTEGWZaFusfjgcvlQj6fR61WSymkTtapVjqkqqrASF3ETgX4+Q6m9yZM00Sv10M8HkelUllWSKnT6aDVaokGEYEOUTMpMkvHGbaB6dkUdqrraBqLUBQFbrdbuKQESVKiDyLxkoRlSTqY8FmtjkDiErynFjH16RnsmEF3GEA4HCZOUiJVqpVACUagb5W1EXbuwRPQMGy/wumFO/DpWxga+njEIsGIRJZjsRii0ShmZyOYD7UwM38D6G2j9HQNbq0LtfsNU/quOEsJFEpAE6AEo9EJ/PoCTZ+Cx/eHq+/ybvK+/N5EZOkeBoVVnJxOC5csl8u94bVcO3TLMNC7uODbxELqNlTbe5h64+AyMBtkVwL7OxLWX25gs+nPs0mPafvx0t3A3PVHoXNOrlrijnRsvfiMRHoOTD4BOK/iw5OV75ZpXpmYoPTwYiN+ay0sS3uwjB90G468P9l5HvuVj/j6+sFzZdILM/q6ur2a7lom7wVv1j9Y4wiKlhX6C0eqiNj1HMM7AAAAAElFTkSuQmCC',
drive_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAInSURBVDjLpVPLahNRGP5mcrWZhGgaLEwtJFJJqQEXgoKIkKBkKfoAeYKA4M7S4kLyDF34AF4eIIOQMC7sooqbECExImnSGgwqTTtmcubm+Y9MSAWh0AM//7l9l/9cJM/zcJYm44wt6Hfa7XbdcZwCD9i2DcuyZsEYm8V0OqXcKJfLRcJJVAIH3wmHw3o6nRab/m3zZYZCIei6jl6vV6xUKg3hgCtuJZNJDIdDRCKRGWgeSP3BYIBEIoFMJkOiG3y6ITWbTaGeSqUwHo9P2KVSCChJEgKBgFCPx+OIxWLQNA3dbrcYJHWyTrXSpmg0KsJXn3dC2XVdmKaJfD6PVqu1ESQlwzAwGo3EAfmb6DAp+2PtZxUTy0Ap+QSR0DkoiiJcEkGB2GlAIF6SsCzL8gkHE2WK5fQaXu1v4uHiU6iqSphCkFSpVgoioCCCL6nXXJlfoWuD8TX1/CrWlm7gyDTwvP8Im+pLUbYg8IFkNZfLidz9xXB3vQzH4+W4fA0eDg77yC/fwjGb4PHbEq449/8S0A0QgW+XwhwxAe79+AyLu7C5G8uxMJ4e4dql2zi2fmN38gJBfqJarVYrzb0ykc20CduxcTGxwsGOIPt2uIcLyhI+9t9ht/Meq8PijvS/z/RgO+ua9B5cBtNmuLy4Lt3M3sOHvR3UO2+GzEbxoOp9kk77G68+k43rK4UFvVPfZ64At2d/4TQtuyUZjhtYmLhO9nvV++rP/wHfnZcUJl+kcgAAAABJRU5ErkJggg==',
drive_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI7SURBVDjLpVPPaxNREP7e7uaHhmxD09AaDzYgGCIBoeBFpJCAKIoXT54Cgh6EXL0U/C/EmyD2ZHMTTBSN6akXoWiIJU1qTU3L4kZjk+xms5vNurN1Q1rtqQMf895jZr5vZneYZVk4iXE4oQnuoVqtvjdNM2UDw+EQhmGMoev6GIPBgHwxk8mkKY9RC3byotfrLUUiESfoqE226fF4UCqV0Gg00tlstugosBkfh0IhSJIEn883TppMpHOz2YQoiojFYkS6ZD8XWblcdtjD4TA6nc4hudQKJTLGwPO8wx4MBhEIBFAoFFCv19MCsZN06pWC/H6/A5d9rMTGqL4MfrcJVbyMZHIBlUplSSAmRVEgy7IzIAoejUagYZJ372J7FfHzGqYS9yGtPUG/daCSCqSIiS6UZLfkSOY4bqzgLNvAQrKLqQvXoDTyOBWcx5y8isjQSHHESr0SqIAL940w65MRSt6Cqdbgn4lCOA1Er17E7cQ+BCrgJpHUeDzueFc+vVu9rzDa38DxLTBOgVdsAZoKVRocFKAv4AT+HZjbN7MTxfZreObsgsMKuNFv+98NwjJ0VFe2kduahqBpWiGfz1+f+MscH+W3kBC/48zNO7D0t2BmF59fdWF2ZPD2jHK1GexhusCOW6ad3I312St3L+mtpxCEKWwUe+rHNSl7Tuuv+AM8Fp/vdA/twj9bxkNn/C54XwRf3vzsfVr/8fDBcv3Fsct01Pa2W+86+x/m1V9Kf3NTfnTvWe3l/+L+ANeBhMdSVgxaAAAAAElFTkSuQmCC',
drive_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI5SURBVDjLpVM5q1pREJ6r1w33qLihIgoS0Z8QYhRCAjapXBqrFCn8BQ9SBixs1N+RVyqBGBvttFELxUJxRwm4gc81ZwauaNp3YJhz7p2Zb7nnctfrFV6zRPDKxQubTqfz+3w+h1nA6XSC4/F4i8PhcIuXlxfM5VQqFcE+DiWw5vdSqbRiMpmo6P91L1MikUClUoHBYBBJp9NlYsAQv+t0OpjNZiCTyW5N9424H41GoNFowO12I+gTe1zmms0moRsMBliv1w90UQo2chwHYrGY0NVqNSiVSiiVStDr9SI8oiN11IpFcrmcQkC/Z4L5crnAfr+HYDAI7Xb7iUek3W4Hi8WCDBKK0EzMwll4hmee50GlUhFLHBDG6XjAAiaJKItEogcGwiCn00l1drsde8I8oqJWDHyBgzabDcmwWCzQ7/dBoVCQvO12C1qtFubzOaxWK2JMA7ARA31AI41GI8RiMZLm8XigVqsRQDKZJCOn0ynU63UCogH4BXBAo9GgwnfhMORyOZKDLNAwbMpms5QDgQBEo1G8C8AzR0vFYvETUq9Wq4RIV5QZhQzi8Tjo9Xqij0YXCgUYj8dUw+Rc+EQi8Vm4LPl8/hvT/oNtda1W6xcz7kMmk5Hgd8eFA9k6svyH5Y+TyWTN3d82v9//1uv1/gyFQhaHw3FhLN6gYd1ul26oy+UCs9mMMv8Oh0MRu9KzhwE+n09stVqDDPHZZrOZWJMY/9jlcvmV5TEz9xl/B+bbmaEvGJMv/wD5tI7A5/InTwAAAABJRU5ErkJggg==',
drive_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAITSURBVDjLpVM9ixNBGH5md/YjH3u3EnKX5BDcKyRNUAs7QUwawT9gZSptJL/gQK6ysLYUsfAX2CSNIVgKV4VwBqIQCGc06mFiYnazu+O8o1mihaA37MPM7L7v+zzPOztMCIGzDA1nHHy96Pf7L6MoqkogDEOsVqsEQRAk8H2f5na9Xq9RHiMLMvm6aZqdfD6vgv4cmzYNw0Cn08FwOKw1Go22UiAZH7iui/F4DMuykqTNRFqPRiNsbW3B8zwiPZCv26zb7Sr2XC6H6XT6m1yyQomMMei6rtgdx0Emk0Gr1cJgMKhxYifp5JWCbNtWWLNvKqE5jmMsl0tUKhX0er0DTkzz+RyTyUQ1aB1EzRRCBqeLCDQb33wLlsGxnYohTo6QcVylkgpUqTptKElaUpI1TcPOxatIp3ZxqZQFN3R8/BrgdOZjlrqA0ragnKpGrOSVQAXWoD2y57G/m8apz/B+Kt8xHXs7DiLTTY6YU4F1Ekkvl8tqJnz6HsA0DYR+iFi2wQ8FbK4jFFx9TwrQCfz0LBJQwGJhIFhFkA/C+FcjGbCQPSt5JaWSy462ms3mzY2/LJn3KjcwK1yBkzLhRwJck1a+rCA+9/Hk4TNJZLfY3y7T4+bQszjunMuahyRgOvfj8MMxu5Y/YfO3r2bvjo/usX+9jS8Ob90tFAuP8t5l983r5oL9z3V+fr98u7hfeRrHYeoHJYR682gpzWwAAAAASUVORK5CYII=',
drive_network=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHbSURBVDjLdVK9qhpREJ7j7vrLGlhRiaCFFikkT2BxQZs8Q7pgJ9rcVuIj3MYHSJdXUAj+FDa2wSaIIihcJVUs1t91c75z7yyLd+8Hw5k5M9/MnDkjXNclYDqdDhzHqV6vV5In4fTL5XLx68N6vV4DTyCBJD/EYrFxsVik8/lM9+AigGEYNBqNaL1e15rN5lDHpczasSyLttstRSIRj+QnQt9sNpRMJqlUKtFyuWzL66GYTCYP8Xh8XCgUaL/fqw5Y0C6IQgjSNE1VN02TEokEDQYDWq1WNV2+p5NOp9UbERSNRpVwdX8nOG+3Gx2PRyqXyzSfz9s6qhwOB9rtdmTbtkeAcAcgASgCW9d1ymQyyo8EVVSezWaqVSbjCaFQyEsAMvsBDFzeVXX+nnw+T6fTSQVDkBQ+fCnPATZ8sJmn87/LqaqKnIBnwglgc0cAYlQCOFKpFFUqFS+Yg2BzMMh+fy6X8xL0e73eF27pbuPebCRvKU4pfeFfFkar1focDod/Z7NZZcutu3W7XY0CIBqNhhvkIC1inzVLraXh/LOFY5tBYWqVsZr3WCwW8WK1ofTNr+/mOzEvCaDc469buHywHYN1GWQEPiFoBsDHr3+e5PHt1fzx/PPTY1Dcf079mla6MmR7AAAAAElFTkSuQmCC',
drive_rename=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGASURBVDjLxVO7agJRED3XN4gPkEWwET/AysoqoE3+wcIPCEvEVsgnhJXgF4j+QraJLFjkAyxEsFCwsBYU32bPyF7WkEDAIsMOcxnmzJx5rLpcLrhHArhTQt5jMpl8nE6niqs4Ho84HA5a9/u91t1uRzuo1+tV4hRbcMEPkUjEMQxDgr6Lv81wOAzHcTCfz6umaQ6EgVvxJZ1OY7lcIhqNapAfyPdisUAymUShUGDRluseqNFoJNUzmQxWq9UNXbZCoFIKwWBQqicSCcTjcdi2jel0Wg2weiqVkl4ZFIvFpAoTZrNZ0W63i+FwiF6vh/P5jO12i2KxyHm0Qqy0Xq+Rz+d/nXS73Uaj0YBlWdo3m82EJRNUPOfz26fQhdDmiOXD61MZm83mJilZExviWtgrrvEu8ArWFj8fmrdiScDdUyyz/OcD4ix0Ag5sPB7r1VEZ4K2x0+mgVCqh2WyiVquJP5fLCXPV7/ff3V4efVemraf+qySIjKluIlv9+890d4IvgMkS1vrQFCsAAAAASUVORK5CYII=',
drive_user=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJaSURBVDjLpVPNi1JRFP89v2f8RHHGaqOCJFNuohki+oBxE7Sp1oFQLXPXbqDFUG1btOgvyIKBoFmUixSJqERIzbFJ05lRpHn6xoYcP0af+nrnlmKBqzlwOPe+d3/nd37n3MtJkoSjmAJHNNVokcvlIoPBYFl29Pt9iKI49l6vN/Zut0sxGggE/ITjSIIMvqzRaGJ2u50d+t8mZarVasRiMZRKJX8wGIyyCmTG+xaLBTzPQ6vVjkGTQFpXKhWYTCa4XC4iXZE/R7lMJsPYbTYbGo3GP+WSFAJyHAelUsnYjUYj9Ho9wuEwCoWCX0XsVDpppUM6nY75iL3T6eDt86c4TL3E4VDeW0/h2t1V+Hw+ZLPZFRUxtVotCILAGkTA4XAIaibFr58i6Hx5hYEkQuKUaJYTePbkAW7cuceqpATLxEQbAsmSWMkKxZ8J86kI5ubdsJmNpBtmxzHUhTzMci8IqyJW0kpOCcgpAbGTGRxO3Axch35Gh4P6LlQGG16vr0P8O2qWYAQkkNfrZZGc5HzYrWEzGceZpSWYrHPY2cojJehwUv4/TkAToASj0Y36kE6nsbVdRHRmAfG195hVA8WDWTQlLRKJBKuaC4VCb2QtVyZuGYtCrcbGxVeraLfbOHf+AuYdDqy9CLFR0kj39oRv3LTHtPHw7DZ//KrzXVmD5q86qnIiYqSLptbqcem0HYvix/7Ux2SwnYjv72RQrvyA1WqF2+1mYI/HA3EwRHnzM/QmY0o1LYFkdd7mftYfFQvfbzX3qxflSg0kLZlMDip8fWNh0f6YszjyvwFmK4mzFto0SwAAAABJRU5ErkJggg==',
drive_web=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJASURBVDjLhZLNa9NwGMe/WRKXpp19L6WrdVsZHgq9aBF2cFDRDi8WxJPssqMXDx4H+xuGFDz1YC+eBvUgnWDc0FqoL+CmHlylthurdWqG7bY0zUvtL5CSxYOBh+f5/fJ8P89LQg0GA9ifu68XMzOB8INJ/kL8WKGwf/y5WW817z/KrBXtuZQdMBRfuz5z+emcb4E97LvwtXsG3aMOfiiP1Y0Pwu3ineenIGN24nm//+GsN8U2dQ3bf4BnByJe0luIhsKM1+Fatecz9ovZs9NT7+QaPFoKG3sStOgOPrFPQP92YtoTif4XoOkyTmTgTUvHN5EBdxKFo7sEyr2Jnlr7Z1+jEarVqlCpVAa7P0U6pEg4kmqgxjgcfPdAP9xDnAPqu7/oQqEwyOfzwinAUDzvcDjSyWQSVzxZ7Oy/RSZE45JXw9w5BTeTW/jSfo+l1D1ks1kEAoF0LpdLj0ZQVXXF5/Oh3W4jPD6Ji+O3UNxeg6q9AsP28bHVwo2pRfBdHo1GA/F4HPV6fXkofUGVy+V5nuc3Y7EYOp0O+v3+yIZgkM9MURRomgbLspiYmIDT6YQgCAR2lVEUZSUYDGLojSSO4wwz/w/irbGu6+j1ekgkEqjVassMqSJJEkRRhCzLoyRN0wxvns07cmYYBm632+iQANKkMmnZTLAL7GfiXS4X6TpNRjBIxMyq1sp2iPnO1DGm0BTbIfZRzJ2Q2AAQkt/vH1WyJpjLI7F1ocQikcgIsF4qlRbMlqwjWWPrmJau1/8CtF7RM3ksOU0AAAAASUVORK5CYII=',
dvd=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKOSURBVDjLlZPtS1NhGMb3D8y/xY+FwQKXr4mWFYT2wQhNQqEXKohIlNLIT2NNxLLYVtAiGIGSL9nm1lTmW3rczpk7287cdNTci9vU7Wy7Os9DR3J+qQMX93Nu7ut3nvt+nqMAoPhbDoejwmKxaKemppbHxsZ+mc3muMlk4oxG49A7w9DZ0vqjhdPpVEpmnWTOBYNBxGIx7O3tIZPJIJlMwu12YejlU3FY12fQagbKjgH+mL/wPI9isYiDgwMQiCAIiEajyOfzKOb3Ie5zsEy/x4vnD2wD/Y/LjgDky8Scy+VAHlEUQd69Xi8SiQSFZve3cZh2oXC4icnxt+h90mWgAMl8Stp2IZ1OU0MqlaKQcDgMlmVB8vm8iEySRTbtprvISbpz+2bh3t1bFQrJrNva2gIRx3HURHomoJ2dHbrOHiYlACMZg9J6V9phFjabDR0dHTpFeXk5/lVkNgQcj8fh8/nQ1ta2SgEMw8Dj8cBut0Ov19NiEkOhEM3LOTJYv9+PQCCASCSClpaWKAWQItlEIEQkJ+fl6NjwY4HxUrHCNi40X9k9ASCSzaUAKyNgdiMIuzuEiUUWNY2XXCdakEEklrYws8pj5gePuc0IXn0cR835xpH/GqJr049vaz4sCXGoztUV1FXVp+lF0mg0OqvVikhCOsrwGlzCCtZ9i1jxLGCJdWDR/R0enqMDFCJR9Gtfo7K6buToJg4ODir7nvVOf/hshO+nm0IIYNkzjyVujkJYP0OPb3R0FJXqqkmVSqU89jP19PQoHz66P9LZ3S6aJ0yYX5+lkDXeCfvyVxg/vUFtXW1OrVbrZPMxgKzu7q4z7Z03hq9db2UuX70Ya2puiDU01TP1DbXDkrmitP43h3Pjic5IKdYAAAAASUVORK5CYII=',
dvd_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKmSURBVDjLjZPdS5NRHMf3D+h9XXVZRGBXBpYEdSGUaNkLoZWPmNNEI4tekMg0c+HKGLoldhFuBa2bKSm57WI2Wcryhfm4t/biptNRc29uzu2Z+/acU4+Y3njgy++cH+f7Ob/zJgIg2imTyVRkMBh6dTrdzMjIyG+NRhNRq9UOlUql+KBUnN49f7tjNpvzeLOcN2f8fj/C4TDi8TiSySRisRhsNisUfZ1cv7xD2SuT5P8H+Gf+6na7kcvlkEqlQCA+nw+hUAjZbBa57Aa4DQcM+o/ofvnQKOl6kr8NICsTcyaTAWkcx4GMXS4XotEohaY3VrCZsGJr8ye0o+/R/rRJSQG8+QRf9lYikaCG9fV1CgkEArDb7SD5bJZDMmZHOmGjVWR4tdyt37p/r7FIxJvlS0tLIHI4HNRE9kxAq6urtJ/ejPEAljf6+f4aX2EaRqMRYrFYLiooKMB+Rc6GgCORCDweDxiGmaMAlmXhdDoxMTGBwcFBOpnE5eVlmhdy5GC9Xi8WFxcRDAZRWVkZogAySTARCBHJCXkhmha8mGJdVHbfCi5UXFnbAyASzDsBZcxZtChuou51GW5IStCh7ERJ2SXrni0IIBKFLXS+fYxnnxl8Yfswt6JFj+42rvYcwynmqGVfh1j1/AyG5t9gyCqjVyzV30KPXozCxgNp+pBkMpl8fHwcwSh/lQELrL5ZzHt+YNY5hWm7Cedbj2OUfYedbdjSRwB/37NUKs3reNGu/zSsgueXjUIIYMY5iWnHdxTfOYRubQ26tNXU3DVWLVSQ2v5MbW1teY9aHww0NNdxmjE1Jue/UYjFbUbrQD0qpIfxSltLVyaRjHmAVLT7ezY3N52sa6jtv15TxV6+djFcXlEaLi0/xxYzR2YLGw8mSdm84rwkZP4fYOfdUwjREaAAAAAASUVORK5CYII=',
dvd_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKySURBVDjLjZNbSNNRHMcXuoceFHrzpQdfsjd9FOzBMimIkWUKYT5Y5gVT0ES7EIqSWq2XZROJIHTd9hBDWKhbMJssbU1D527t4q4Onbvo7pv7ds6hiemLB76c3/nx+37O75z//3AAcA5KpVKVKRSKUZlMtiSVSrckEklALBYbRSLR2PvJsfLD9fuBWq3OI2YhMScdDgf8fj92dnYQiUQQCoWg1+sw9vppalw4ODkqGMn/D/DPPG2xWJDJZBCLxUAhdrsdPp8P6XQamXQUqagRCvkHvHjWqxwZfpK/D6A7U3MymQQdqVQKdG02mxEMBhk0EfUgHtZhL/4Hs1/fYaCvfZIBiLmUtL0XDoeZYXd3l0HcbjcMBgNoPp1OIRIyIBHWsy6SRF2dLXvd99vKOMQsdDqdoDIajcxEz0xBGxsbLE7EQwSgJUYHibdJhwkolUo0NzcLOSUlJTiu6N1QcCAQgNVqRUNDw28G0Gq1MJlMmJ+fx8TEBCums8vlYvlsjl6szWbD+vo6vF4v6urqfAxAi7ImCqGiuWw+O6vWbFjUmpkMdg+uVdduHwFQZc0HAT0VZ6GoL4K8govZ2tOY4Xfj8tUbuiNHyILonD3CVF8jNJ2liElfIqObRuRTF37eK8bwpTOqY13ix/MnESVmCHhA7ylgqBBbz8shvch1sx9JIBAI5+bm4A2ST+legc6+jFXrLyybFqExqFjbGc0XHByBvgLIL+RmGIDP5+cNDg3IP0+JYN3UMwgFLJkWoDH+gKymAOG3dwBiivdw4Ceyt+RgpjLHtf+Y+vv78x487nnT2tGUksyIsbD6nUFWLGooBW1YuFsET28hNh9yYWk8gW+83PR0Zc4jzuHn2dHRfq6ptXG8/vYtbc3N6/6qap6fV3VF+6q6eFFayfXQtsnOTmqm9X8BjufboeKd+1kAAAAASUVORK5CYII=',
dvd_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALeSURBVDjLjZNLTxNRGIb5A2Xh0rULjJqw1BCDprXQAuICDBEBE7xFJQY1giCKEARCF9rUljuEFiOIBokFhYIFK6Q2Ei6V3oW21BKFdnq/93VmQomACyd5M+d8Oc9zvjM5kwQg6e/Mzs6myeVywcTExLxUKv09PDzsHBwc1EkkElG/WHRq7/qdgUqlYpCwkITDFosFDocDbrcbPp8PLpcLGs0KRC8aIm3CerGA35S8S7ANfzCZTIjH4wgEAqAkZrMZm5ubiEajiEf9iPh1kMteoqW54nNTY03yjoDamYLD4TCoJxKJgJobjUYQBEFLQ/6fCHpXEAsaMD7ai7rHZWJaQMLHybZjXq+XBjweDy2x2WzQarWg6tFoBD6XFiGvhu4iTOZO+fXYvbs305JIWGi1WkFFp9PREHVmSmS32+lxKOgiBWq4Vgeg7S+A4V0DFrrY6HzEmUxKTU3F/+T2pSOwT1WAWHoPeGxwLg3hUzM3QgvUajX0ej0UCgX6+vpogHqvr6/T9bFXFfi1+BAh6wjs8hb4l98iuqYgu7gapwXUogRESahQNSql54/Boa5BzK+A33QLW1/LoBGXYqn3Gi4X5RP7BFQSsGG+H47lB4gF5uAzliBgKYR7pQqmnlxMymXIzM1b2XeEhKg4OwV2VcU2XIyA+QJc6mroW7lQqRTofD2KzOzcjn9+xItZh7EkLSfbnkFgtRRBSxEIshODiIMZpRLfzE6kn+HGWOyME/RF4vP5wunpaWwQVkxLefg+WoXFsWdYV+bDt1qIrYVK6NvP4odaCfPGJhoFXWBmcDt2biKPx2PUP62TDYxIMNjMBNxr0LbnYarqIL60kru2MKFZ+Ain04nu7m4wWezx9PR0xq6fqba2llFZfb+j/spRhOYECCmfQ9lwGkM3DmFmSgzJmx5wuJwwi8USJuBdgkTyTx4INZakxGRNnLjkCTtSXJBJ5JzLUmflcNpIOG3v+j9k/eSwcE1V7wAAAABJRU5ErkJggg==',
dvd_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALoSURBVDjLhZP9S1NRHIf3D+gfURRhL2gFFQYFiSkEIkZFI8kfyjIUNNFCrOnMFxKJMTfCKNEl5AxUVCx1bm6NdMMkx97ci5vTjTn3otvdy9326dxbiinRhQ/n3MN9nnO+557DAcDZH7VanSuXy4VTU1OL4+Pjm8PDw4HBwUGTRCIRf+wXXz34/V5Ho9FkEFhE4ITT6YTf78f29jYikQhCoRAMBj3E3a/otyJ+v1DQnvmX4A88abVakU6nEY1GwUgcDgd8Ph+SySTSSQo0ZYJ8egCvO+qV7W2NmXsCZmYGTiQSYB6apsG8WywWBINBVhqnNhAL65GKreDrRC+aX1b2swICXyTLToXDYRbY2dlhJevr6zAajWDGk0kakZAR8bCBXUWCpKb6Uar26ZNcDoFFa2trYGIymViIqZkRud1uth+PhYhAR0An6W+RFcahVCpRXl4u4mRnZ+N/qbh/BZMfShDZ9rLiQCAAm82GsrKyJVag0+lgNpuhUqnQ19fHQkzrcrlgNi5DN/EAWytS2Ba6Ybfbsbq6Co/HAy6X62MFDLwLMRImzBiTlppr2DRIQIct0I/chVY7i3mdBUbHBopLbm0dEjDZhc3LKmgGihDbHENsowt+6zgWx+qh0jvwRWtEQdFN/aESdkU5OUQ8y4fPNITYWjPm2s8hsTWEH+/zMK8exTvpBApuFPX8cxNLiy/APtOAuLcPlKUMc205iDrqEbRNYKH3NvILC1N5+dcvsQdJIBCIFAoFPEHyK9d/Qm/XYklaigDZuOhqLSLmO7+zco+U8gYOhQDC6lzt3kns7OzM4Lc2T38alcDmNUD3TQjHXBfiHjE7e2SFS0o4y7aUrQKUewQK/mmvoulk1t5l4vF4Gc8a6noeVz2k1d15oHxWxP0ziHnJHnil+/IZ9I4Oru8SyBqOSzkHr2dVVeVlRcf5qKI1JyVvyU7Lms6kZbxTKdmLLFrWeCJGIGrm+TFqpv4oNV13RPkLngD4bMIOcuMAAAAASUVORK5CYII=',
dvd_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALoSURBVDjLjZNrSFNhGMdHJfVl0peCKAKNoL7NopQ0CCqRQrxlUVoxK6eZk0oy03Ap5rXSYZsiedsWuYhGopgaqNNlrTR0uUtuuOl0lE6n87Kb/857aKJJ0IE/7/M+PP/feZ73nJcBgLFWCoUiuLOzs6K9vb2/ubn5l0wmm5FKpVqxWCyUiIQn/q5fDZRKJZMyCyiz02QywWq1Ym5uDgsLC7DZbFCrhyF8nu+qEuSJKviFvusAf8yter0eKysrWFpaAoEYjUZMTU3B7XZjxb0I16IWnR0vUVKUIS8seOi7CiBvJman0wnyuFwukP3IyAhmZ2dpqGNxAsv2YXiWf6CtpQ65OakiGkCZg6i2PXa7nTbMz8/TELPZDI1GA5J3u11YsGngsKvpLpyU7txO8qTfTQlmUGbB2NgYiLRaLW0iMxPQ5OQkHTuWbRRARRlNVDxNdeiAXC4Hh8MRMFgsFv5X5GwIeGZmBgaDAWw2+xsNUKlU0Ol06OnpQUNDA11M1vHxcTpP4uSaQNyoDYBOr8Xo6CgsFgvi4uKmaAAp8poIhIjkvHmysqsD8KjpEi5UHkDv4HdojBOIijk/vQFARPZpkuPgioLAqTuMhBcsZL+NhVRZjvTXEQgt24WmTwMIizg3vGEELyhFFAhZvwBvvlbQxlfKMpR1pKP+YzGSJKE4WrANp6PC6v95iNeoeYn5aVsailpTkN+SCF4TG7nNHFR25yG+LgT7cxge+kfi8/mCrq4uWGapT2kexLBxAHHVB9H4uQySvieoVxSjpreQgtyEsDsXiZIw7HuwZWl3JoNFA0pLS5l5j3M7Gt+JYfippiHRQn+cKd+DU892IqRkO67UHoOgi4cEUSj8M30cfqnMI+suE4/HY97PuledzE10yd5L0TfUjSHDFwzqlejp/wBW/lZcrT8Jvwwfu98t30MbbqNXXG5qSGLy9arLCfGq2IvR1siYcGt45FmVf/Ym196szdiRyfBbW/8bMwbKi+1SPUYAAAAASUVORK5CYII=',
dvd_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALCSURBVDjLjZP/SxNxGMf9B/QfiVB/ytCaoS7MDEoMYlgswi+FkzQsjNZQNKX1g0s3ZkTppuB+smJSOnOiGTa1lLHt5rZ0c05trX29fblt7+5zdJIK0cHD57mHz+t1z3P3uRwAOX/H4uJisdFoHJienl7V6/U/JiYmful0Okqr1apGNarSo/sPEpPJlMvCShZOud1uBAIBhMNhxGIxhEIhWK0WqAa7GbWySzOg6M07JPgDv3c6nchms4jH4yCSra0t+P1+pNNpZNM0GJqC0TCGp33353ufPMo7EJAnEziVSoFcDMOA3DscDgSDQU6apHeQiFqQSWxgavI1Oh83azgBCxexbWei0SgHRCIRTuL1emGz2UDq6TSDWMiGZNTKdZFio/VuY+Ze253iHBZWejwekKAoioPIzETk8/m4PJkIsQIzC7rZ/CfbYRLz8/NoaGhQ5hQUFOB/orCwAL6FZ/BONcG3rIHL5YJYLP7GCcxmM+x2OxYWFjAyMsIBZN3e3ubqJO8QnYR/qQVMZAm+GTE2l99AJBL5OQHZxENEQoLU+Lp9dhB7xjowQQ2CZhn8n6Vwjl5Fe11Z5JiABA/zdce7ZiQD40jsSJHYHUJwTQraLcP6QGnm2Ai8iKz8CLMvGxF19oHevI24uw309xug7bVYkp5O/fMlXqnIh6H/HFbHLiJMXQe9cQlxpwgxG4Hz0VNzQsodJIVCoZybm8NukP2U3nVYtr7C8mkAdl0t26oaMUcNaEqIlednsSYXwtQtQMvlU4aDkyiXy3O7ejoN42+1cO1bOYlLL2ZnHkZ4vQi05TxWVGwnk8N4NaRClbDso0AgyD30M8lkstwHD9tfNEnqmYkPOjgnbyK514MYVY3lwVIM97ei8kJlqry8XMnDhwR8SCTNJfVNt9RDHRX7X9TC7ExPSUZy7YynqrpSzcLFR/f/BmKF4lxymSkBAAAAAElFTkSuQmCC',
dvd_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALFSURBVDjLjZPfS1phGMeFXdef0bVXMeiiQS0hNeeCuuhKUFzKEnU/lDEKxZazaEop20Ub1QpaRhdrjFVgGY4RyEYuj80y7SfMNH9l/vzufV9mrO1mBx7Oc77n+X7e533OeXkAeH+Gz+dr8ng8Y8vLy/6lpaWfi4uLybm5udD09LTr7ZTr1t/1V8nm5mYdMTuJuRiLxZBIJJBOp5HL5ZBKpRAMbsM1bim9dJqnxhxD9dcAv80fd3d3Ua1Wkc/nQSHRaBTxeBzlchnV8gVKFyF4VmZgsz72Dj17Wn8FoCtTc7FYBL1KpRLoczgcxvn5OYMWLo5xmd1G5fIHPn14A1P//SkGIOabpO1KNptlhkwmwyBHR0fgOA5UL5dLyKU4FLJB1kWRhE57r/JAr27iEbPz4OAANEKhEDPRPVPQyckJywuXKQIIEGOM5GekwwK8Xi+USqWTx+fz8b9BZ0PByWQSe3t7kMlkXxkgEAhgZ2cHGxsbmJycZMX0fnh4yPSaRgcbiUSwv7+P09NT9PT0xBmAFtVMFEKDajW9dvd9j+BLIMyCix5D2tl19g9gYmICs7OzsNlsLNfr9VhfX0djYyMsQ1amWZ/bMP56BoKOTu7aFtbW1tDc3Ay73c5apJ+SahaLBb29veDIkKlGO6Q1QpH42z9D1Gg08G9tQSwWQygUQqFQwOFwwGg0slxEdGO/mUF0Ol2G/UikwElXGrSa2WoU4Oc+o63tNlZXV+H3+9k23G43BAIBLPZXDGAymfIMMDw8XGceNK08eqJHn6aPAR4a9GhtbSWQNkilUhY0b2lpgUQiiVMA+Q+SV4dpYGCgTqGUv5PelVZHX4xi3j2PhYUFuFwuaLVaGAwGllONnM7EyMhIkmwzeO1oNjQ03Oju7pKJRKK0XC6vqNSqqkqlIp/rzqFE2vFerVbnSOTJu2x7e3uEDJz/C4Myz4QSsAdYAAAAAElFTkSuQmCC',
email=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAITSURBVBgZpcHLThNhGIDh9/vn7/RApwc5VCmFWBPi1mvwAlx7BW69Afeu3bozcSE7E02ILjCRhRrds8AEbKVS2gIdSjvTmf+TYqLu+zyiqszDMCf75PnnnVwhuNcLpwsXk8Q4BYeSOsWpkqrinJI6JXVK6lSRdDq9PO+19vb37XK13Hj0YLMUTVVyWY//Cf8IVwQEGEeJN47S1YdPo4npDpNmnDh5udOh1YsZRcph39EaONpnjs65oxsqvZEyTaHdj3n2psPpKDLBcuOOGUWpZDOG+q0S7751ObuYUisJGQ98T/Ct4Fuo5IX+MGZr95jKjRKLlSxXxFxOEmaaN4us1Upsf+1yGk5ZKhp8C74H5ZwwCGO2drssLZZo1ouIcs2MJikz1oPmapHlaoFXH1oMwphyTghyQj+MefG+RblcoLlaJG/5y4zGCTMikEwTctaxXq/w9kuXdm9Cuzfh9acujXqFwE8xmuBb/hCwl1GKAnGccDwIadQCfD9DZ5Dj494QA2w2qtQW84wmMZ1eyFI1QBVQwV5GiaZOpdsPaSwH5HMZULi9UmB9pYAAouBQbMHHrgQcnQwZV/KgTu1o8PMgipONu2t5KeaNiEkxgAiICDMCCFeEK5aNauAOfoXx8KR9ZOOLk8P7j7er2WBhwWY9sdbDeIJnwBjBWBBAhGsCmiZxPD4/7Z98b/0QVWUehjkZ5vQb/Un5e/DIsVsAAAAASUVORK5CYII=',
email_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKLSURBVDjLpVNLaBNRFD3vZTKZfCaftmltmghGLHUhVUQoRRApRVyo4M6luBBEQXDTjftu3IhW0IWgCNaN2qrUuGjFUmpR/EORgpbUhk7zaTOdJPN5M76ZVFBXhT44vDsz79w59553ieM42M6i2OYSrt55OymF5ANF1QxvNCxqc0E2HDAe2Fwd47Dt5vMmHBBmmrX1Yn5+YUFIJmKZ86e6o7rpECng+yc7+T8mzb2uW766zlJnhvUGVapW1rBs8mCygHzRgKY7WCzZyJdtLK3ZKKzbUFQHRc2ByYClkoHrzwqoaDqVk5ndVNMZCfgpujqjyL1XsLZhoiNK4OdiRB+BKLgA4kGCUtXAo+kVxFuiaI0HPE201rA8idkdEaQ7oph4p6CimmiLUI8o8kQxiaCsumQFba1RZLsiIJvmUa3Bmt3kB7OpCJKJEEan8h7BJcocJR7fe5VHLBbyzgSFv2zU6k0FhHfHMi1Igo2dXXE8n1OwVGx4eDqrIMPfySIDdSxP2Z/OCjWdwVVjGBZWyioyHTJE0Y9CWcKb+ap3UbozCSyvvsTc93GotQoahoEjvScA56CbwHK4t0QpcXJSRlDyw824qz2EnRyubTOfRvGjksPRw4eQbtmDyW9PkPt6C9VSPy+h/Ounzn3sSQedVIKiJcx4AxnaZYZU1EYnx9SH+9jf0wtGGXo7B8GIib59/dCqLxzB2FhdHLg8kQjI4bAQ8BGBd5Ny+3xcO6UElNfr0wqCn0RwfO85r/QrA7cx/mWER2OEbGWY+i+l66dPHpMsfsmHBu9iOHcWki+Ah4/HGlsaJtuxr81+noEIwfuzSARMf3ztfrpJtjrOfRdTw3y7wCFzqBwjszeWh34DRKkiE0cpxFcAAAAASUVORK5CYII=',
email_attach=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKrSURBVDjLpVNNSFRRGD33zZs3P85vOs3kOEUK4qqQCqzcDE0F2eBQgbQIhMAWGWREQdAyaF2LFkaLJCLbBBVoCga5KNEwChK00kYdHWfGnzfjzPu93ffGooTceOG89937vnPued93L6GUYjuDwzYHf7v7w5Dd6W7MiEpFvqRyOjOkg0Jjgc7caQy6Xp5vgIJoirK+mklOTE3xAb83cqm13iMplNhtln/UyeaYlN9FSbUUJa36/F2pxKXX1FpZ1cmToRSSGRkFiWImqyOZ0zG7oiO1qiMtUmQKFIoGzGZl3HuVwnJB4tyBSB1XkDRis3II7/LgzVgaK3kFQQ+BlZkRLAQCbwDwOQiyazJ6hxfh2+FBpc9meuLWS6ppsTbkQk3Qg77RNJZFBVUuziQKTMhrJ8iJBjmNqkoPasMukI3mcYWSVq4mS6ytdiHgd+LZ26RJMIhuhiyLHw8k4fU6zRwH/1cXCsWyA8Kqoyoq7LyO3WEfXo+kcbyxylzv/5hBhK0VM5PofngfLpcLPaoKPhwDvy5pMNzIsorFnIhI0A1BsCKVs+PdxJp5UOojfmhiEv3Dz9F46AiaWi6iIQjcuH7NEFAp6y1JZxk54IbDboWhuHenE7sZjLYlv33Fy95u7D/YjH3Np3Hz8gW0xk9iITXHfiE3Ny3J6p6GGgdxOThCOM3c1bBO2OPzp3G8eNqDpqbDSJxN4NyZBBKnWujo+2FSqkv85OX80syxq31+m7uigrdZCM+qybH2WZhKiM5w5McA9yUNOhUKa3eORi3NsTjGRkfItP9Ecml64TuMy/Q/dHR0UEEQHsTj8bzf7xe7uroetbe301gsZv2dQ7a6jYw8Jsvygc7OzhDLuyKK4q35+Xnn4OBg8U/SVg42xMedTudkW1sbjUajls3ffwGqPWPVeFFgygAAAABJRU5ErkJggg==',
email_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKGSURBVDjLpVNLaBNBGP5mdpNsHptH25i+0toIRRAFD3rxqCjowaPtzYNYvIleehE8iHgTBO1BBCl4aG+CxZIeWqrFBxVEUXpIwZqmsWuyaZNusu9xdlPxcSp04GP/Heb75pv/QRhj2M+i2OcSbz1+Ny9F5OOVhhXd0W3qckMuGBweuNydw+G67f9dMBDHsprbleJKoSCmU4ns2MXhuGExIoWEf9TJ/zFpf1uGLbQMp3f0nqFTpW7nTNslz+bLKFZMaAbDWtVFUXWxvuWivO1CaTBUNAbLAdarJh68KKOmGVROZw9RzXBIKEDR1xNH/oOCrR0LmThBgJsJCgRB0QOQDBNU6yamX28i2RFHZzLke6JN3fYt5rpj6M/EMbusoNaw0BWjPjHIhRISgdrwyAq6OuPI9cVAdotHNd1pZ5MfzPXGkE5FMLVQ9AkeUeao8nhyrohEIuKfCYt/lVFrtR0Qnh3bsiGJLgb6kph5r2C9ovt4/lZBlu/JQQeU2b6z35kVm4YDz41p2thUG8hmZASDAZRVCa9W6n6jDGdTSHx8Cm3uIcxSAXpnL8oXrgLsjCdgM15bolQ5OS0jLAXgKQ4diGCAwyublp8APk/h5KUrCA0dQetTHl+XpnGU1ZioqaVvhmkPHu4Pk1iYEkId/1bvSYS0O2H55X2cGL0GaXUBWLyDSCKJg9lBHFuYdERz5+fa6euzqZAcjYohgYg8m5SXT+AqlBJQ/t6bG2ui1D0EnL/xp4Vv9yCp10Syl2HKnxWLp0Yu90e/z8Bo/UCT7zXqAlZWUdrTMEU6Mk++vFm0y3oYdSGAmkpQKBGHX/2I7HWcl0YGxjV1Y0xwyKAjsBJnTZzL23d/AbqhIfYa8w35AAAAAElFTkSuQmCC',
email_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKGSURBVDjLpZNPSFRhFMV/33vfzDjOvBmlRkuz0THKIjGN2qSrFkFEBUFBG1fhtkW7aB1UUNDCyIJaRC3aRAXWxkroHxpCRGaY2uS/GWfUGUfffPPe+1poUbvCs77n3HPvuVdorVkP5P8Ujz3ae0IVtj80w80U5l4i7MlO8a8OfvQmTsvAyXvBjR1EG1pZGHvD8PNbs/JCz7u+snKrdS5fCi3ZjuFp8NC4nsbTGldrmq234kx7p4hWtJEd/YxfKKzIJsoq4zEZq4zWdR3bHimWtCgLmH91FYDKvEKlM0QThyhOP8BfLpgYGsb1/Fwe25c0UjknoRxP3OubJjmnKBQ1ExmPZNYjOdaHSvUSbTyMPdWD8M3jC1tgz2Hu7WK5YvdWo1B0RcBnULs5wvPBFAtLJaojgpXxx5QvPCO67Sj2ZDeGr4TK1TP1YoiB6vPE6psAhFy2HQASm8IIDb0DKdo3DOLLvaaq/Qhq5hamX2Mvxpnp/8DgtmtsrGtE6FWeUbDd1TxNSNSEiWaeYWbfo9wapj9ex8OmkK0l2f+JgcQVahsaCf4RviysrCoJAU7JwTd9n13Hb/PlwTlG+l8T2NqCPZ9mvu0ivnAMQztIn/y9ZWO56KIBpRxms3lGvqVRn57Q0NJBKLSDyaFR9IFLNDXvoX6zRXYhj+c4aA1ogVwuOtr1tEhl8tTFLO58TXH1Zjf7dzbgj7fQfOou/sgWPDSy3I+ssphK51ipCIL2tCxkJ8eLyok3bQmKcNAQN54mMdZGEkKsOfUQvw4DSbzS8sZn8iqX/jEl1VJ64uDZ3sqAFQrJgCmkNDFMgWmAYQgMucpb00KAdh2lVhbnM+nR5Hex3m80WCd+AqUYHPPwkaN5AAAAAElFTkSuQmCC',
email_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKqSURBVDjLpVNNT1NBFD3z3uvHo339gFZoCxpqRAlBJayMG4wLl2hYqT/AhQsT1xoTV65cEFeiG5MuJLpAJTEmWo26oBbUhSIfEUhLsLUt0NL2fc44r0UDhh2T3NyZ5M6Zc8+ZSxhj2M8SsM8l3RybSrpblIFCxfBsqaZAOSEKBotvKGdn8aC0ed4OBmIZRm2zkPmxuCiFg/6uK8M9Ps1gxO0Sd6GT//ekmeuaKdY1K3rxjqYK+bIZ101KEsk1ZAo6qhrDSpEiU6LIblCsbVLkKwyFKoNhAdmijtEXa1ivaoIS7josVDWLuBwCYhEfXk3nsbFloN1H4OBknCKBU7IDCMgExbKO8Q85BFp9aAu4GpyEmmo2KMY7vOhs9+FlOo/1ioGQV2hcdNpAW5+Qm76FiY8/EWrzIR7zgmybJ1RVq6kmL4xHvQgHW/D4bQalig6/m8AjqSgtjEMJ9aNPft+okaUdNlbrTQaEq2MaJtwSxcFYAJOpPLIFFauzE1DCA/BFBhCuvYOzNtdg9ldZoaZZsNnouolcqYKudgUn4kEcORRE+vscaiuT/PUArPIzRPsvQVhOgBoaGv+PERvAZLa/+SK/HFbg9zg5RYLuAzKOS68R6T0PqDNIPUrAG6jDVUxBXXnDrTQ4AGVCtbS6rHEfj3XKLBoU0OqxuIAWlFoabrMEJVgD1ZbsYlhczO6h62j7lUAuX1LLv7NLpO/C3aSzNX7SpXg8kkskElfTwXW4Fnkonj47TNyOKVA9u92zA6JnELkFAYnk/Gbia/wz2WuYZu4PXQ31Dt/r6JH5qyner47pp7MYHOkFEf2AfA7pBzfyjNIzewKkRk9l+y8nYqKwDGau22rtmj9RPorcty+Yf377ibTXhJma7p4ZG6lzjXjrdEewfxl2ZqzjD9JZU0+1XOyXAAAAAElFTkSuQmCC',
email_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKESURBVDjLpZO7axRhFMV/38zsy+xuEt2Q50bYgChGUBAVxUbRRlS0sxJsbP0HFAQJ1jYqtlqYTlARNZoiPlELCxVd0LgxwcnuxmT2MfPN97BIxAcIggcu91EcLudwhLWW/4HDf8I7feXZw/Sq3JZqEHc0QuUYCwaLNhZjLdpajFneV8oidBy3FquVd+Wy19PdWTx5eF0+iq1Ip9zf2MWfs1ju7Ui57UgPHDsfhY6/pEpSGXHt4RyVqqQZWaZrhkrdMPPNMLdo8ANLtWmJNczUJBduzrHQjJxcT3HEaUZapBIOg/157r70+daI6c0LEi4kXUHSEyQ96MoIakuS8amvdK3Os6YrBSCcVqgAKPVlGerNc+eFz0IQU8g6JD1IutCZFtQDyfiUT2FNntJgFrFintMM9bKaLpQGsvR0r+L6ZIV6IOlMC3JpQS2QnL99gHcLhygNZMl4v9jYbC9/IASoWJH2DMODXdx67jNTDZmphtx46hNbxVBhAxcnRkl6P5X1WpHGAlIqvtYDir05kskElx8c4c19jTQKqRUD3SNs6NtOEDY5e2MtV0+UwQq8VqSsNlb4tYBiT45MOgEWlJHs23gcbQ3aaAyW2cUKm4Z20ZBtDl4s0Ofcs16z/uVTJNXa9UMZkc04QjgaBwiVRFvDdO0DsVEoExPrmKUoYHNxN424xfOPu/FkY35676k73alcR4eXcoXnuTiuoL8YekorevPDKKPR1jC3+JnV2T5eVR7xrFyOE2/PPRF/C9PRSyUdSok0klBJRgobnR2l/bz4/JiJ93dfS8W22TEbiX9N4+g5Z37r8J7C5PuJl9Kwc3bMSoB/JiidEfPauIW20Ql/zKof9+9pyFaERzUY+QAAAABJRU5ErkJggg==',
email_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALHSURBVDjLpZLtS5NRGMYf6H8RIoT8VB+CZaKUsHIGGYgRpYn4IVdgmhW+zNnUTTZwC6l9SahmDjNTLJdtTtO11tBiOed8mS9zbW5zm9szd56r5zxKtM8euODmuq/7d87hHAYAcxwxxwY86pudlPX/itTpnAeVPXZyS2UnN1XfSEW3jZR3zZHrnbPk2tOv5Kp8hpTIpom41ZoRt5mTFxuHfackqklGPeDypNKEiyYyYDPIUvo/HVARIMMrtn+AQDhFCh4MLDFNLxa4Pd7Qjvjg8bPYTXBw7xB4AgTeIMFaiGAjzGE7ymE/DawGWLQb1rG4EUVJ9yzH1GmdYHn0u+9RaD/44N1hEU1yWYOBGIckP+wLsug0rqN3IoJgPEUBYO6o7AJgYZNg2H4IobskWAiDoTjH94HNEAvlkA96cwRjrgxCsSNAuXxOALj8BD+3CAZmQlAYPMJudJDef2uXRcdrD55PhvBlKSPoH6Dk8bQA+M3f27nG4r3tD15NB/Fs1Ifl7SS8/iR0fN0/FcSb6QAs7hRmVo4ASh5QVG9BigfM+1iM/wjC5o6g3/gRis5u6PV6NMsUeGtywbYUhlxx6Cm6lBgaGcMVxRSYc3c/cwn+zSacQcyvxtHTq4darYbf78fy8jLMZjOUPWpotH1YXHQLntVqFTKXa2Rg8ipeendjKbIRTHCj4yY8bGqCY2EB7fIOiMViVFVVQaPRoLGxUaip19rWzlGI9H49Yehvyrs9FD5b9yl9RnKPNLfKBEDhE9OB6EIhTCYTHA4HLBYLBgcHcZ73LrWMxymguaWVZP3r3Nzcmurq6jAFlJaWjufn56eLioogkUgE0Zp6tEcBNJsFOMkv/ogupVIZNhgMIaPRCJ1OB6lUioaGBqGmHu3RDM1mAXJyck6IRKLTxcXFK5WVlfHa2tokL7asrOwGrwK+3qMe7dEMzf4FvOYAdxCFF/YAAAAASUVORK5CYII=',
email_open=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKhSURBVDjLjZLNTxNBGIf37/Bf8CI3DxwQo8aPA3g1XDTxwAFIJMWIIiJCUCQaZT1ICQYCISWRbyMgHwUUpJXyIQVa+kFpoWVpu922u9vt7s+ZgRCJGJ3kyUxm5n3edyYvB4A7jZaR4IW2iTDaJyMwj4Zu/+3eqZsNlq1nlpkwfGEFoaiKoQUBzy2erv8S1HS6JocXIkjIBtQsoBEUzcDMzyiedLi8jR+3z5wqeDu4fY4fCqh2twSJBEsKEIpnGSmVyIhkM5QGPxxEXY+/8ITgVa//XvvELu63Lh9iXoKpZQnl7x0oe/cDxW9seNi2RNZ2lBI6rWFUdfp4JuAHA/rQwgF2SSYa/K9BJfR5I4sx1Hb7de61xelTMjrEVJZlpYNfO0nJVAY3B2R2VsLbIaU19M0GcLHC4uUqzStGgmyYWlaPBQcJFVFJZXM4JsO3J2HVG2NnhY+tKG5ewcaOiILGOYMr4x3kg3T02UVUmB3sEg2OEeJJIiKSvWga7mAChgEUNcyheSwOIalQAbi7TTYmWAnq6LeJaO53s+AYCRZTGQYVhoQ0ars20DoVxydnFgfSkeBW3TwTOPd0rIZ0fF4UUPnBieqO9RM8aHNi3iNj0pVlHAsKHs0ywXpYh8OvYuD7PpkVJvsdutc9G4F1U8FX75HgJRFcNllJp+lYDqgsO724tqv/IaDQCnpJghmXAiFBK5gHl1s6bqRIz445BKztqPAKBrz7BjwEF8FN2IqQdYRUSZ5p9ykYIK0e2JdQ8OKbweUUtXuikqLvCCkjntIgyhppZQ1JRQMVU9IEmZI5RBBl3bYRkfNNli3ubGHTRM6d3tj5stFMbsW4llc5peVXWbVL1VbtSs20drVuWrtGuF5/yI366czVp1+SeeU9fhr7C4feT2DQtSX8AAAAAElFTkSuQmCC',
email_open_image=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK9SURBVDjLbZJPaFxlFMV/93vfmz/NjElpp5rEyUggVijWP9BKVYQacSVIdeWqUBXRjYILaXetRFC7caHFhXYjBsyiuIiIirXQja0UQcSWmjaZxMFk7CSZyfx77/vedZFpSEgu3Ls4HC73nHtEVdmpvvixctSG9udAhCh2r554bvDLHYmquq0/nJqZmLq8qLOLHa3Uujp99T/9aOrW5E7cbcCZr2/+8t3VJa23E+061dipduJEL/9Z09Nf3bx99kJ5aDNf7kr4dHr+USPy6xP7B1IPDveBCI2OB6A/a7AG5qodfvp9mZWWe+nkyyMXgPUFn3xbfndPf+rs33NzqCo+Ubz3OOeJnSN2HkGJnCNJlPEnD3N9oX3u9Cult+y56QU/Npwzj4/189fMLU48O8i+QgERiwJxIiTrbuFdxKnPL/HCoQJ786tvTnxTfsPG7bX58UeGSt1YiWNHYW+B6sefkclkNoyWwOJsQOa146Ag6mm31rh47UrZLtXdSOQSJiZv4LzDGIvk8gw9fBBSAdgArKFy5RpJAsvdAd47f523XxwhXyiWTLPrJR0annqsyEq9QQKkQwv9fZBNQ2hBTE+Esss2ObC/yJ6BNICYVscBMHpfjmPPH+X98xc3pUQ2NUz+8AfjzxxhdDiH9PJnm71X2QBGh3Kknz7MB5eyBL+tbgmcY4zjhw7S9uEW3Dbb6xeIgIsdUay8fuwINtxKdHHMjYU6g4V7sKHdONC0uh4FosixWGtQvDdPKhVui7wNQx4YzFNbaZA4hyqggm11nfpEZelOg2IhTzYTgoIC/q4DCgmK3ZXC7stTqdZpD2RBE7XN2j+z3ciVHro/K7msETEe05MkIlusXB+W0u58MvtvI6pXFyo2WqvOjb/z/e50vq/PpgOxNsAEQmDAGMHY3gLZkK3eRVF7dflOdWa+/D+ThmoKzy7qoQAAAABJRU5ErkJggg==',
emoticon_evilgrin=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJpSURBVDjLpZPLS5RhFMZ/7/eNM6Xf6GhYSeU0FUFGdMUWRdIiEqSgqDataldLwU3WXyC0jxYFEUFKbTIkyLCgoIhuRPcGC7LQHPPS5+f7vue0CCa70KLO8nDOj8N5nseoKv9TqV8b7oaJVGlXoVWEZlVQ4a4IfSqczO7SydnzZvYF7oZpUeGMaewomFwzmq4D9Wg8gi89IHnVVRThUO0eHfgN4PrNHs21XAoWHiQOMjwduMn8hpCqyPH2mWNhIU9DwwKSd90knwb21h/Qy2WA7Td1qgyGay5EEozjp4uUhko8vD2GKizJC4Xlgsk0EmbW8OXO4UkR8g0HdTQAUKE9yHdGmgZJivT3FKmpSWhpzbBlm7I0P8HN6wY3/gSfvKFyVWcknnaAAECENlO7GZkuoi5m++5avoMtYcoCsGXrCGoncWP3SeVWIp62sgoiNJGpRuIxXj+uBZMF84s8qqgbZXHdNSDBewplgAqxuuk03lGX30jlvOWYIPx5XzxTnx6h472gigjhjws8RWy8DtJURTHZBSsIUhU/AbxN8J+vYoJKxKewCa9n/6DXjjzApBsJkgHs1w/AbIcqdvwNfL5Iqnot8cggdobesoylS6ZehLfR5nOR2Be4qUHCXBsVuQ2gnmT4FjPvTxPOXYSp3snglSOTzrJs/TEdLhtp+KLZl6rf0T2ncS+EE7jSPXz8AXXTmFSWMLsOzzJKz69Setm3f0On9vxm5aHzZod4TlU1dRTS81YTVAQgDpd4poZe8vFOV9FZjm48rn1/zALA+7MmEk+HCK3e0+xmwFnuOkufs3RtOvGXMP1LfQPtZUr+1SCKsQAAAABJRU5ErkJggg==',
emoticon_grin=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLpZPNS9RRFIaf+5tx/JqxUZGKQtEkyJBKw1ZhG0mSFkW1aVW7Wgpusv4CoX20KIgIKowgZSgwrIVghFQbM2swQRM/xmx05jf33nNaBJNWtKizPJzzcDjv+xpV5X8q+mvDPTdxVXpV6BahQxVUGBchpcL1xEnNbp43my9wz02nCrdMfV+jSXagsRpQj+aW8JkJwg8DaREuVJ/S0d8AbsSc0mTnYLDjPFqxDb8xiYTLqMuDKSMoa8bYgPDzA8KF0dN15/RREWBHTI0qM5HWe3EJ1vD5NNhviAtRn/8BEcWU1hMpbeXr2MWsCA07z+tKAKBCb9DQH9cYSJgGt87I4yWQAvgCL56B2Cxu7R0+/EjFvv64eHoBAgARekz1ESSfRl0OFcuxE+WoWNRbjnZmUQlRm8Wtviaa3It4eooqiNBCaRWSW2X6bTWYBJhf5FFF3Qq7a54CId7TWASokFOXj+EdNQ3tVNTuwQSRrfviWV94g64NgSoiRH5e4EljcwchRmU8R2J7M0G0ZAvA2xC/PIwJKhAfxYZMb/7BkF2awMTqCcJR7MYcsNmhil37CMv3iVYdILc0gy0wVJQxM2jqRPgUP3InLvY9bn2GSLKHkmQbqCdcfElh9iaR8l2YquPMPLmUdZamQ1d0sWikxfvmTLSu60FZ/WmIfMNlXuFzc6jLY6IJIomDeJrITA6TmUqdbevXh79Zef6u6RLPjcqWvsZY7X6CkgDE4ULP+vwUX8YG0s5yuf2qpv6YBYDZ2yYunj4Rur2nwxXAWcadJeUsA4ev/SVM/1LfAVOtXW4xpbWpAAAAAElFTkSuQmCC',
emoticon_happy=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJtSURBVDjLpZPtS1NxFMfPH+Arwb+jFz1Q9GK+aLBXkWQvgoRehRDYAyxEY2StB8SILBJnWZY4HyhDrbR06YssW3O54dRNr7a2Zbq2u3u33bt7b347LhhqYlA/OJwfB87nd873nB8BoP+xPwK6i4q0EbLl3tC4MkRGdpCMzEsalwfIJvVR0Y4ATjZpwyToQSuMlR7o4gj05GtosQ4o01aIz0lIPCPTtgB9hMo0jwlG1MFJXVCjdcgKVcgETyETqoLytRGqcBepURNWuqhsE4BLLs4Nk2x8d0JbbcbQncNorzHD3VkOt/No/j54y4z07Dkoi61Y7iA51k7FBQD3a9dDtfyyE2qkOp8guCry/vf9RN6LHw8xpBaStxaRNrIXACyW14j3IfftGpSlM8guVCIzdxLpwHHIU2WQPBZOLkXy3X6IHyzILrYh3EreAiDzinKGPMp9XoC/7xL8/TfhH9hi/Q3w9dYgMbaHAQ4stpBYAKRfkKgnB/n184hO9SMZCUCMzW2y9VjE48SPt7tY3GYsNJNcAPB8vfpKLwOqIS5046euYesxNBXxqQau4ADSoQeYbdzQAs/Xng3YoMbuQQqchppa4vDahvQ1KPFpxF37WMBKrL63wd+wQcTEUyqJ95Cshp+wgBch+SqQCXdCk4LQUjOQ5x1c+l4W8AhSgSb46kmevEolmxaJl6M84TIjO9/EItVDmjzGqh/kknezL2XoWSR9jRC6LfDYqXzbVeblMPN8haTbiozwiKfyGMqXVsjBFiyPWfH5BgmfrpBlx88UfkhFS/epTnDQRKiJMHOb1vud8F6nOvflv3ymf7Ff4B/4xZL2LgEAAAAASUVORK5CYII=',
emoticon_smile=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJnSURBVDjLpZPNS9RhEMc/z29t1d1tfSmhCAwjioqoKNYuYkRRFB300MWT3eooeMn6C4TunYoiOgSKkGAUhh0SjJCwsBdtfQMN17Ta2v39nueZ6WBtktGh5jLDMPPhC/Mdo6r8T5T93nCPTUqVDhVOi5BRBRVGRBhQ4drGc5pfO2/WKnCPTbMKN0x9Z4OpzqDxWlCPFnL45VHCd91ZEdprWnRoHcANmhatbu4JtrShiSr8t9dIuIS6IpgKgoqdGBsQztwj/DDUWndee0sAO2hqVZmO7b+bkuAzvpgF+wVxIeqLqxBRTHk9sfL9fBq+kBdh+9Y2/RgAqNARbO9KaRwkzIL7ymBfDiQCH/HkIYjN4z6P4cNJEnu6UuLpAAgARDhrahqRYhZ1BVQsx85UomJRb2lqzqMSojaPW3lOWfUuxHN2LWAv5WnErZSWVCzqItRHP2qL+ggJc0CI9zSUACoU1BXBOx71PmXq7dzqorc/csj05BKDD+ZQsaCKCLFfCjxZbAGIc7R5N+9ezTI7uYD6EBXLTHaZiTfLZBrTmCCB+DJsyETJSCL029zowaC6nkRynqNNDYw9m2L8xSx4S7LSkMlUkUzEKEsfoJCbxkb0l8643GPqRHifarydEvsGnx9HohXUhYj7eUaIJXdi0qeYvn8x7yw7Dl3WxQCgplUXRWj/NnELdBuxdCMmVouKgihBfDMb6k6gieMsvezDRrQfuqyL66w8f8ecFM/15N7OhvimfQQbAhCHCz1f59+yMNyddZZLh6/owB9/AWD2pkmJp1OE096TcRE4y4izDDhL95Grf3mmf4nvrQOLvcb/mlMAAAAASUVORK5CYII=',
emoticon_surprised=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ3SURBVDjLpZO/a5NRFIaf+yWmNU1iolQsSqRiBZXiT1ocpC5FURwUdXHSTcdCF6t/QcFdHBREBBVFUCkIFV1KK+LgorYaqtCa9rOpaUzy5d57joMQi4qDnvFweDjnvO9rVJX/qfivDffUpFQZUOGQCD2qoMKECCMqXE4f1cryebN8A/fU9KlwzeQHO022B02sBvVoLcSXXhFNDhdEOJM7ps9+A7hRc0yzffeCdafR5Cp89Q0SfUFdHUwrQetmjA2IPt4hKj473n5K7zcBdtSsVmU61n0rJUEZXy+AXUJchPr6D4gopiVPrKWbr2NnKyJs7DitCwGACgPBxqGUJkCiArhvjD4IQRrgGzx/AmIruPJrfPSe5NahlHgGAAIAEY6YXC9SL6CuhorlwOGVqFjUW/b3VVCJUFvBLb4knt2CeI40VRBhGy0ZpLaIiqW65Jn/nGBxoQ1Fyaa/kksXaQkaqAuBCO/pbAJUqKmrJ/AOfIPiTJzUhhPk93UDEE6/Jpy6yvpcESUAVUSI/TzBU8DWgAQqloUvK+jo6iWZWUsys5aOrl5KS2tQaWCCJOLj2Iip5T94ZMNXmEQeFcjlqsxOjlMtz1EtzzE7OU42WQQNiGd2UAunsQ0eNWUs3TPtInxI9d5IiX1LJZwinIuxWE6DCpnkPLnkDG3ZDkzmINMPz1WcZdOuCzrfNNL8bXMi3t5/pzV/HGJLuNILfG0GdXVMPE0svRPPJkpvHlN6N3Jy95De/c3KszdNv3iutG0b7Eys2U6wIgBxuMjzbfYdn8eGC85yfs9FHfljFgA+XTcp8QyKcMh7elwDnGXCWUacZXjvpb+E6V/qO6evcx8oUQKYAAAAAElFTkSuQmCC',
emoticon_tongue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJpSURBVDjLpZM7aFRRFEXXffPyJokzmkSFWBiJiGBEDH5io8QmIIqFojbaaGkZSBVtbAOCpZU2IhhRLJSAEFELISIpRBB/wxgwakYnn5d5n3vvORZCjB8sdJeHw2Jzzt5GVfkfhb8O3ANTUmVQhQMi9KmCChMijKlwsXxY4+X7ZrkD98D0q3DFdA11m7Y+NOoA9WhSw9cnyV6PVEQ43X5EH/4GcOPmiLb13wo6T6Ktq/CNl0j2BXUpmGaC5k0YG5C9HyX79PDo2hN6ewlgx02HKtXCtuslCebxaQXsAuIy1KffIaKYYheF4jbmnpyJRdiw7qR+DQBUGAw2DJc0Askq4BYZv1MDycHnPLoPYmPc/HN89pbWLcMl8QwCBAAiHDLte5C0groEFcv+gy2oWNRb9vXHqGSojXGzzwjbNiOeQ0tfEKGH4kokmeVT9SstLdAUOUJy1OXYzJIuepJFWN1RAzK8p3sJoEKiLo3wjpmpORoLOXnqsVawVaEUBuAD4kTZe8qCKiIUfjjwVLBJL0T07G5HXYa6hM+VmNl3RdYYg00MU/UEE7QiPsRmvFl+g7u2NomJulAB9TnqLFGzJ2ukaJLjFhIkbBCu3E5Sq2Jz7v4AeC41XlyIcWWCaD2oQSWjvMrRtGWetzN1qvNzrD/ciRa3M/34QmwzLv0UpJkb5li4dmC0uesoFBZw9af45APqUkxYplDuxbOR+st71F+NHd8xrDd/i/L0NTMgnssreoa6o9VbCZoCEIfLPIvTr/j4ZKTiLGd3ntOxP3YBYOqqKYlnSIQD3tPncnCWCWcZc5aRXef/UqZ/0TcrHX7i2ZbMyQAAAABJRU5ErkJggg==',
emoticon_unhappy=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJlSURBVDjLpZNNSJRhEMd/z7u6666b6VaQFoohgYpYFHoKo2+SDnbo4smOHQUvfVy6CoHHTkUQHYLCgyAEK32AUUSHCqKMxYIs3ErXV3ff93memQ7FJhkdak7DMPPjz8z8jaryP1Hze8HNmKwqoyqcEKFPFVR4IsK0Clc2ndJwfb9Zr8DNmAEVrpnWsXbT2Icmc6AeLRfx354TvR0viDDSNKT3NwBc3gxp48CdYPswmtmMX3uNRF9QVwFTR1DXgbEB0fvbRJ/vn952Ru9WATZvcqrMJ3puZSUo4SsFsCuIi1Bf+QERxaRaSaR6WJ49G4rQ1jysXwMAFUaDtgtZTYJEBXCr5CeLIDH4mAf3QGyIK73AR+/IdF7IimcUIAAQYdA09SOVAurKqFgOnkyjYlFvOTAQohKhNsQtPaOmcTfiGVwP6CLVgLil6pCKRV2M+vhnblEfI1ERiPCe9uoZVSirqyTxDnxMuFzixdM5St9CEEd9xtDd3UAm5VECUEWExC8FngK2DCQJl1eYzb+iZUeWw8d3cehYG80taR4/WiBcKWOCDOJrsBFzVQUiTNni8z1BYyvp2rf0D3RSX1tGfAX1ETtbkuSae0n7j9Q09FIuzmNjptYrmFh7dTnEbSKR7SSbBpUIfFzdR9p/JFHfgaZ6WXh4ObQRE1VA02ldFGFkbe4G6A4SDf2YRA4VBVGC5FZqtx1BM4f48nISGzOy97wubnjlhZvmqHiu1neNtSe3dBPUBiAOF3lWF97waXa84Czn9l3U6T96AeDDdZMVz5gIJ7ynz8XgLE+cZdpZxvdf+ouZ/iW+AxEldjf3XZxpAAAAAElFTkSuQmCC',
emoticon_waii=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJzSURBVDjLpZNLSFRxFMZ/944z42MqH2MllWJEkT1UjHETSFJiVAtdtHFly5aCm6xNWyFw2aoIoiAoWgjSIqkWghAuVFBTJJVGGV/pzPznzr3/c9qEaUaLOsuPw4/z+D5HVfmfKvhdCIadmCo9KrSLkFAFFUZFGFLh0YFbmt7d7+yeIBh2WlR44lT31jqlCTRSDmpRs4rdGMP70j8vQndZh37YBwjeOx1a2vLaPdqFFh/CZqcQbw0NcuAU4haewvFdvIVXeCsfOitv65sdgP/eKVfla+jCi5i4Wwy/HCSVNJy5eAhUmR7fIl4Jl6+fJxS9wPeRO2kRaqq6dN0FUKHHremLmewKm/PvCLuGtptFLM1usDS3SdvNIiIRsNsTmPUxis/2xcTSA+ACiHDDKWtGtj8TdbZoSkA0bLjSXkTr9QjRsKEp8XNpb4qC0tOI5cZuQB3Rg6xML+z5yOZymo1ls0dLLRrAw1pqdwAqGA1yePkSVhc9xCr5nJDJKEuzhuScIcgryTlDdjMLqogQ2vGBWObxTUP8ZDVzI5OMf0xREHGJH48SPxJi/VuGmdEtKqosx07FEFuA7zH7CyAM+qtjDWU1F2koNai/jfo5xObQwHC4CrQ+AFXCFY2Y1a/4eQZ/3cAykJ18mCY4gBs5Aeqg4oHNo9ZHxQdcQrEzaLSe5KeHad9jYAdQ1qkpEbqzs89AjxE62IwTKkdFQRQ3EidceRUtbmVt4i1+nu7Ge5raZ+Xkc+eaWB6X1PXWRirO4YZdkIDAs2SSMyyP9M8HPneb7uvQH7MAsPjUiYmlV4R2a0kEeQh8RgOfocCn/9KDv4TpX+oHaI9cJDajhlcAAAAASUVORK5CYII=',
emoticon_wink=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJaSURBVDjLpZNPSFVREIe/c9/r+tD36vW/IA2rTUYlZbp0FQTRQqmgXFW7loIra+NWCFxFENQmgsyEQnGlSIFUVIsURKuHBZmlWWnee985Z6aNaWa0qFkOwzczv/mNUVX+J9K/J9yAyarSrMIxEWpVQYUnIvSpcCV3Qud/rTe/TuAGTL0KN0xFS6XJ16LhBlCPRtP42Rck4+0FEc6tb9DBVQDXbxo0X38v2NaElq7DL4wiyQzqYjAZgswejA1I3naSTA02bj6t3UsA2282qDKR2n87K8E3fFwAO4e4BPUx32YWePk84kBdJfktNXwdOj8vws7tTfo5AFChOdjZmtUQJCnQdfUZXdfH6L45Ab7I2MgCmRLP8ONxfPKa0r2tWfE0L4kowvHU+jp8PIy6iMYLu1EXoz5BbcyhGhBfRG2M+/KMcOsZxHMcuPQTUEXJWiT6gorlzcgmTDqHQUEVWNTJOXbk7wMJ3lO5NIEKkbo4xDvwRcqPnAUTrjiviqc0v525x12gigip5RU8BWxUDSEqlmy+jCBTsco06mNMUIr4NDbhFUCwuEKPnX6BCStQAff1EahbBbAzg6TXHiSansAW6VkGeDoWRtrmcTmCsJzixwckk7eR4qfFzhHFqV6S991oyUEmH7bN24SOFUb6dMecTG8+2pmpaITUHG72KT56j7oYk86RylXj2cXsaC+zY32nDrXq3VVWnrxljornWllVS2W4cR/BmgDE4RLP98kxPgy1F5zl4uFL2vfHXwB4d9NkxdMiwjHvqXVFcJYnztLnLO01l//yTP8SPwD79F9Uvnnx1AAAAABJRU5ErkJggg==',
error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLpVNLSJQBEP7+h6uu62vLVAJDW1KQTMrINQ1vPQzq1GOpa9EppGOHLh0kCEKL7JBEhVCHihAsESyJiE4FWShGRmauu7KYiv6Pma+DGoFrBQ7MzGFmPr5vmDFIYj1mr1WYfrHPovA9VVOqbC7e/1rS9ZlrAVDYHig5WB0oPtBI0TNrUiC5yhP9jeF4X8NPcWfopoY48XT39PjjXeF0vWkZqOjd7LJYrmGasHPCCJbHwhS9/F8M4s8baid764Xi0Ilfp5voorpJfn2wwx/r3l77TwZUvR+qajXVn8PnvocYfXYH6k2ioOaCpaIdf11ivDcayyiMVudsOYqFb60gARJYHG9DbqQFmSVNjaO3K2NpAeK90ZCqtgcrjkP9aUCXp0moetDFEeRXnYCKXhm+uTW0CkBFu4JlxzZkFlbASz4CQGQVBFeEwZm8geyiMuRVntzsL3oXV+YMkvjRsydC1U+lhwZsWXgHb+oWVAEzIwvzyVlk5igsi7DymmHlHsFQR50rjl+981Jy1Fw6Gu0ObTtnU+cgs28AKgDiy+Awpj5OACBAhZ/qh2HOo6i+NeA73jUAML4/qWux8mt6NjW1w599CS9xb0mSEqQBEDAtwqALUmBaG5FV3oYPnTHMjAwetlWksyByaukxQg2wQ9FlccaK/OXA3/uAEUDp3rNIDQ1ctSk6kHh1/jRFoaL4M4snEMeD73gQx4M4PsT1IZ5AfYH68tZY7zv/ApRMY9mnuVMvAAAAAElFTkSuQmCC',
error_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJYSURBVDjLlZNdSFNhGMcH3nrl3ZAgyr4YoiPTNXJ30UVBUOTGXGVin6OLLrqQ6KIwJrm2KBUjE5xRaEnNpuZxpLCLpG7KHbfpprPmPt3a99r3v/cchxi5Wi/8eDnvOf/f85z3nJcDgLMd/pnGsnX9EdpHCQ2e94fLij3HKXaDhHvC9D2Evt6Fe7zh6n8JSLjCSwkjuXQY6eACXGOHAmtvDlaULPBOCXURSzfy2SjyaTdCRjUcr/m9JQlIuNYzKcghl0LK2430+gCR+PFtuCa7+qK69p8CEqZj9mGSD8A0JGLJxmmEzRqsDPEMfxV4JgTSwNwNIJ9CzHYOCxoRS2xRhnwmAOf0FVj790q3FZBwuUtX7//pm0Mu7UJs6SwWBhtZImYxstGPiDlmsfikas3Su6v8D4HrXf1IaF5JKvmQdHUTgQzL2mMFQRPiVhlyyRW4Z2+DVu1Q/CYg4Sqnti6Tz0SRiRiQsMlYQcLehvXPYkRMZ6CbbMLNvmZcUp3E+Q4RpO081abA+bbuU9j8jGycG0lnFxJWKRE0b77C6Ngp3Hl1ATq6B19cFNTT1yBWV6NBzlUy4eOemVbyzWPkpxlHfEnCErWQymYJqS5B6/2j0BofQmt6BGYo9Reh1l+GQM5NcRyjfEfc8YEV5DNBwo8NssENyNqJW3xM0P3YOsbme5gOwPk+UqNZfVkN+3MelgcPwDawD0tP92CxbzdMj3fCqKyE6HoluqhWKKgWNqyYamE7EMq52aKHaSukUsdp5X48oNrYyszMXLN7UIqgIOkkRJi2C3Mns/4Lp3nrFHdnUnEAAAAASUVORK5CYII=',
error_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJrSURBVDjLlZNbSFNhAMcXZ3uTIJ+iMLK0CBkqmXlJicR8SB9rY62C8sGIHgrKCqIIKkgrKJPCC6hRKd0sLacrjQVa0kO6i5vOacdd3MW57Zw5d87Ov2/HECMtO/DjwHfO7/d937lIAEiWw9tbQHm0e/Tu7jydqyuXWuk+yUoXiPwgoL+O2e/X4OzcffK/AkROnO7OC8aiAUT9Bjje7PJNvdqZuOrAtCavIzhSA4EPQYg6MTt8F/TzzNpVBYic4XqfE0NsHvPTNYh6GknEi8nWdH7iiTzjnwEi6xlbK/F9MLYUivCsHgFTM8Zb0nR/Dbje5ah8A2cAYR7M2FEYmgtFGLMaAueDvacCo/XbVMsGiJzg6Mj2zrkHEIs6wFiOwNBUIBI0KcCH+sHQfTA/Spkaqd2S8EfA8Ta7bXaomszkRsRRQwJqWNtLfgUOgR1VIxYZh7PvMvR3km7+FiByir09ixO4ELigDuExtRgI28rhGVQgaDyI8fp89B/bBG2RDNpSmaAtk1UtBuyvs74GTA3kwTkRsVchPKoigcOLW7DW5WPoXC7mOm9DMHaBfXYWg6fkQneR9HRcPuDqPU7eOUM+mk6wFqVIaITMbFKS2ZX4pFyLMJFxrwyoXAfcSIbn1l5o9lMuCf0ik2bpj2JA4PyEmQV4/wJkLL5s4dtLLD38V9ZDu08KyY+29OaJp3LYHqfB2rQDY43bYalLhfnhVhjvb8Zw9Ub0lMrANJwAiBQ5L8EMYbKCgqaEYlf8mZaiU2y4+qU8lXNUJsN9UQZr+Rp8KJPyXcXUpVUF4nxWJF3QFFM2smyBnOm4HB//CRnU6d05um8NAAAAAElFTkSuQmCC',
error_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJwSURBVDjLjVNdSJNhFH62fWbNWboGKtmipVksf6AsN2dE0K833ZQwiu6CbgLpxgg0QiS6NKG6SMJCyAujCCz7UyhRySgixLT8W9NtLpkuN9/vfc/pwtZFzvKBw4GX5zw87/kxMDOSIdy138SKPzLRLBEfyDrSo5LxjFgBrLhxTfYx55qsox5WdA4rEpmXReiFxxrodM8pEWEx+5n9j0vDvoe7rcm4SR2QopZ1dm+6wWiElmaFeavXyorqV+Ug8MxdMt1Rplgt8mKgiUWomUnM8PiDYjnWWljyXwdMdN+yo9pIMoqvnW0YeXoXpE8jo+iiiRTd+GcTAx0ub0qmy5m25QRik9VgBpiBuO8a0vMqkZpd4Rm5U+BNKhDocFmIqNHsqALJMEC/q5lBpKOm/SSu+PpAiq4P3dxmWSZAiprN9lMbUzMd0GfaATDWZpgTH0Ncn0OubSdqWdsk4/rlRJ2BmTH1ZF8eEw3mHH+tqdgH6MHbuNQ3DrAOQRJCSuRk5qMo14P+sVf45OvF1YjcvrduZlhbWhpqtRSc15iiUPO9ABMkCRxynoVigiIFAsMfmURhbjmiIoYL8e6hhXqjSfM/Kq00bSgqTc+vgpzvhvo5AACISwHFhPHwMHSSkKRDVzrmFudRsrkCUX3B0D/6Rmik1K2MvNMAAM3ihmZxLQl0l0Mqiaz1dkhSUEyYikzAasnG+8m3+B4cRVPQ9M0w0VbcQorOsCKQIiRyjZSICwFBAnEp4LA5UeY4jHcTPRgefI66H0aYpLpnWOka/8auemNoj/2grevLywFBcPsbWPyZwmrgqDWEFJlsMVIpwQaWifdfoJOVMsp2n0QAAAAASUVORK5CYII=',
exclamation=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJPSURBVDjLpZPLS5RhFMYfv9QJlelTQZwRb2OKlKuINuHGLlBEBEOLxAu46oL0F0QQFdWizUCrWnjBaDHgThCMoiKkhUONTqmjmDp2GZ0UnWbmfc/ztrC+GbM2dXbv4ZzfeQ7vefKMMfifyP89IbevNNCYdkN2kawkCZKfSPZTOGTf6Y/m1uflKlC3LvsNTWArr9BT2LAf+W73dn5jHclIBFZyfYWU3or7T4K7AJmbl/yG7EtX1BQXNTVCYgtgbAEAYHlqYHlrsTEVQWr63RZFuqsfDAcdQPrGRR/JF5nKGm9xUxMyr0YBAEXXHgIANq/3ADQobD2J9fAkNiMTMSFb9z8ambMAQER3JC1XttkYGGZXoyZEGyTHRuBuPgBTUu7VSnUAgAUAWutOV2MjZGkehgYUA6O5A0AlkAyRnotiX3MLlFKduYCqAtuGXpyH0XQmOj+TIURt51OzURTYZdBKV2UBSsOIcRp/TVTT4ewK6idECAihtUKOArWcjq/B8tQ6UkUR31+OYXP4sTOdisivrkMyHodWejlXwcC38Fvs8dY5xaIId89VlJy7ACpCNCFCuOp8+BJ6A631gANQSg1mVmOxxGQYRW2nHMha4B5WA3chsv22T5/B13AIicWZmNZ6cMchTXUe81Okzz54pLi0uQWp+TmkZqMwxsBV74Or3od4OISPr0e3SHa3PX0f3HXKofNH/UIG9pZ5PeUth+CyS2EMkEqs4fPEOBJLsyske48/+xD8oxcAYPzs4QaS7RR2kbLTTOTQieczfzfTv8QPldGvTGoF6/8AAAAASUVORK5CYII=',
eye=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKASURBVDjLxVNLTxNRFP7udDp9TCEtFSzloUBwY4FUF0ZjVDYsTDSw0/gjXBii/gk2GjZudO1G4wONK40CGkQSRKTybqGAfVHa6dy5M/d6WwMhccnCk3yLk3u+L9+55xwihMBRQsERQz2crK+vX3Txyn1SyfXDMnyE24AjwR0Q4qLQw1M82H4vGo1+3OeQ/RZSqdQTV2XnhkKzmqaoYJaJQj4P27LgcQGNdTocRmFzyWiJv2zqil0/EJDkt67C0oAGhtTmJpLpHEwSAPNEwBwCy+bQ7W1EsYlYWxiKdMSjvbPhniu96tra2ohmbAxovILZxCq0E5dh6M1g0jllAqYEZRw7lhRp1ZDdewW9tILAykRPingfk9Ti7BbJJ47viiC645cwNm2gYPAaefhWH4TgGB79JoU4vG6Cu0MNyMx/Bv8+hkzJtlWWW27yRfrQ0dhS+4sq0aAOqHQgOK8JGJbMKZf9/h1asPssyv56sBejqupuinEtEHI5jgNFURCuA5JZB6a0fPvBF1BLClbsmoPT7X5wKVqrbWhFqDMmFFHcKLLiNmzbBmMM7WEFAY2jbDCUJbFsMpQkjgUI4ifVWk21lqaXoBQ2mMJ94adi6wes5AxoMYOw7uBcl4JTEQFVULhhId5GcO2MJtuUEykXQRc+gb1/hLTl/VobY2JmctyfnTvvUwlEqCMPvdGEHrKgevj+wlTrxO8VL1+ebLaSc1gwA2kj9bPlYJGmPrx7bm0lrkbIrhrwewFPPbjbj+pzdSPtUh7YXsRqpiT2gp1T9NfEhcGR1zY5fEzjo3c8ud3SIKV0SJrp1wgCLjiS7/CKaU5LPCOcj918+Gb+n1X+b9f4B22tbKhgZZpBAAAAAElFTkSuQmCC',
feed=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJFSURBVBgZBcHda5V1AADg5/d733Oc7tjOaNs5GC6KdrEwmpPRxG7spoKghOim7oK8y0MIEQRL+geGEIQ3UXQvSJ8IafZxUbjQhRDZoU60iYsSc9t5v87b84TsVe3mrBWpHoCICIAIACixYTUfOJM2Z62YO97TOULSIKaEQAyESAzEgISAgLpi48de87MLUqmezhGyhO4SCW7f4O81YiSJiCQIkbqmNcXMIjMXeilIGsQxDp8AnKDY5teL3PyU6h4CdY3Av7cYu58R0QghZWeT9fP0v2V7i8Y4j77As2c5sAwIFAXDgjInJxURAzub/PwxMZBGphZYeIWJWZ44xdo5bl4kK8kzioohUUREd4kXP+Kpd3nkee72+epNBleAxdfoLJBlDEuKkpxoBAkBjXGm53n8ZZ45S/shrr7P75eBo6eo9zAsKCqGRBEB/1zj89e5eo7tLRr7ePJtWg9wZZV7t2i2OPQcw5JiRE4UESN1ZPc2g0tceos/LtPYx9HTaPDNe8Dhl9gtyStyUiMIJDXLp2m0GHzN2gdMzdPq0F3k+pcc/4+x/UwepKzIiSDWTB/iwBLT8xw8xt07rJ8HHj7GbkX/B+DBxyhrciIQ2N2i2AG2fiPL+OsXoNVlWPDnDaC5l6qiJJWjLlHxxRs0JhhcIyvp/8SHJylKdiu++4Tr31NW7B8nkrwzp627d9nkHM0Wsea+GSY6tDvESEyY6TIxyZ4GSUp/nTubqyF7WrvZtaKrZ4QSQ+TIMUSJHCVypGhaHW448z+h1tLAgvKk7gAAAABJRU5ErkJggg==',
feed_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKNSURBVBgZBcFfSNx1AADwz/f7u7uZntNKPVtlixIytmlbkrVg/UEqiGrQS7SXCOqlmlRQD5HFXuptDILwZRQ9Vtb6yxb9cdZLachcEW0SUm5T2dL0dne/u/v1+YTq0zoLfcbljAGIiACIAIA6lhyuLXozV+gzrv/eMaU9JHlijhCIgRCJgRiQEBCQNVj6eazw5TE5OWNKe6gm9A6TYPk3VmeJkSQikiBEsoxiFz276Tk2lgNJntjCbfsB+0k3+fMEZ76gsYFAliFw6TwtV9MkaiLkKJ/j1CQL02yukG/j1sd46AjbRgCBNKWSUq9RIyciBsrnmP+AGMhFuoYYOkBHH3ccZHaCMyeo1qlVSRtUiCIieod5/H3uO8TNj7K+wLevsDgF7H6G0hDVKpU6aZ0aURMSAvJtdA8w+CQPHqHzRmbe5a/vgTsPkm2hkpI2qBBFBFyc46sXmJlgc4V8K3e/RvFapg6zcZ5CkR0PU6mTNqkRsrdl9h1icZ4EIZBvY+cBbriHjQt8/JzjuZJvNsrWNi9Ky2tG29o9Mf2ZnCYEkoyRl8kXWfyB2aN0DVAs+TwpmS2U7dt7u+uu6vfd6U98On/SSutWEcSM7h1sG6Z7gOv3sv4vpybB5NJZQ7cMasSGwWtGNUJqZOddPmxvFYHA5RXSMrBylmqVf/4Ay2sX5EPRIwPPg5fun3BTzy61EOTUkNXR4OsXyXewOEe1zsKvvPes/9ZXnV76ydzStFdHj3rr+FNaki0KWSZ5o1+n3itGXNlPoUjM2NpDR4nOEjGqpKmp1d9tL223Wv7b2uayH+dO2rW0NhOqD+gs9BrXa0wTdVRQQw0V1Hmnrd1H7W3KMWhtZkYuV395fe7S6P8L1PAuhFtHgAAAAABJRU5ErkJggg==',
feed_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ8SURBVBgZBcHda5V1AADg5/d73/d42g66bF8ZTgbbxcpoKaNJXWQ3fUJJdRF5JxRF5EGEboIZ/QPiTeFNGF03GFiRUSZiUGxlSlnqtsZya6ustY9z3vPx9jyhfkRXacCEVBVARARABAA0ccvJfME7aWnAhOGDVX37STJiSgjEQIjEQAxICAgoWtz6rlr6ZEoqVdW3n3pC/xgJVn7izxliJImIJAiRoqDSTe8+eqeqKUgyYpkHDwEO0djg+jlunKW1jkBRIHB7mfJdtInaCCmbS1yZZPYiG6tkndz7HE+eYtc4INBoUGvQzMlJRcTA5hJXPyIG0kj3KKOH2THAQ0eZOc2Nc9Sb5HUaLWpEERH9Y7zwIY+9y9CzrM3y5VssXAD2vULfKPU6tSaNJjlRGxICsk56RnjgZZ44Rdcg0+8zfx44cJRiG7UGjRY1ooiAvy/z6ZtMn2ZjlayDR96mcg8XTrK+TKnC3meoNWm0yUlFxEgR2Vph4SuWvuX+w+x5lAPH+fgNi++9ZvHarzZ+uy4rp3avtw3mpNoQSArGj5NVWPiamQ/oHqHSZ3EluL2ybPT5I7YN3mfrx8/9fPEL6WYUQSzo2cuuMXpG2P0wa/9wZRLMTX9j6OCLyjfPC2de0jE7ac/ATvONIAUCWys0Nsk6WL1Jvc7vv4B89Q/l/kGeOgYgPXG3vB2kchRNtPjsGNkOFi5TbzL7PWdelXWWbf5wVufU6+pbyzbx31oiTUhODOvSf8e4O4cpVYgF23vZ0UdXHzEKMTN/7aqYRY1kw79/FeaWEt3t9qVQf1xXqd+EflVtNFFDjhw1NFm03dz6hrwVZElhZywuDa20n/4fCNbrcsCV4KMAAAAASUVORK5CYII=',
feed_disk=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ0SURBVBgZBcFLaFxVAADQc+97Mw5JaiPmR4uhG5GI0Jg0tStBXYggaNe6Uop00w7FpVIElwrBZRcupCBuFIo/CKKC4LeBoIhoLTKWRBNtQ2Jm8ua9uddzQvWiyfa8y0pdABERABEA0GDT6rDntbI977L7H+uaXaZoEUtCIAZCJAZiQEFAQB6x+X23/fE1pVLX7DJVwdwKBbZ/5p91YqSIiBQIkZyZmGJmiZlr3RIULWKHh88CzlIf8NsaNz5i9B8COSNw5y8695KIEkJJf4sfP+DmVxzs0BrnwWd56i2OnQEE6prDmmbIkFJEDPS3+OkqMVBGphZZfJ6j8zxykfUr3FijahhW1CMOKUVEzK3waJfdHpvrLn696MjNff3hhnsm2ja3TwnxlImxQsrBnY2BC4P3lRIUBLTGmV5gesHgs+seP/2AZ5YmfPjNLa8+twAA3vlix2q6KooIuL3BJxe4foWDHf2qdnu/AWs//A3gsGGvYmd3KISkFBEjOTLYpvc5W9+J8ZwiU4+SN88vSSnLKGRjJVVV2TsYKiUIFJkzL9OaoPelzq/RYVV7/b0/xEAzStIoycg5kxlrJSWImemHOLYC5EaTBqpRo4gtZMFICEFOWc5J09SGw0YJBAbb1H1aY+z8br8/rRVK558+oklDRDI5JJ2y7Y13e/7dO1AaIjcY8eklWkfpbRilVxRakoG3118yNTYviHb6fzq3sirq6FdDpVtWbX7bNbNCB6OaEyeV24UTc8xO7Hvi5GmT7eNEdg+PmxlvjLUpQxCqJ02251w2pyuhwSEvpFWD+5ZN3jWpU96tSbWQkpBHUuj7pXfT9l7tf4r+Cu34pbphAAAAAElFTkSuQmCC',
feed_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKzSURBVBgZBcFNbJNlAADg53v7tdR1o4vup/yk20IQxqZsLIskCsaTGi7sYDzIzYtctDEkXkwWEhOvix4kRG8eOKkh/iUmDgJiREVh6PidcRnb2Ngcg3Xr1671eaLKm1ozRaNiJQABAQABAFDDrLFk2sk4UzRq90slnUOk0oSYKCJERIEQESKkiBChscnsr6XMt2fFYiWdQ1RSFIZJYeFvHlwhBFIBgRSiQKNBcxsdB+g4W4pBKk3IMjgCGKG6xu0fuPMNm48R0Wgg4r95sk9RJ6gjiinPMfElUxdZWySdY99RXv2I7QcBEdUqG1VqCQmxgBBRnuP654SIONA2wMAx8kWee4crp61NfWWmmpfKD1ibOq+4pVUsIKAwzOESK9PMXmF6nB/fY+g4xcPK2woePXxWoe2QfM+glX/2uPugJlaHFBHSOdp7ae9l3wgXP+D3U8orl1XSZU/uft7y3UmZKNGytSDb1iMWEGH5Kt+9TUc/e0fItfPC+zbG31JJPZTvfkVl7oxMU+TfP2+oV6p2/fSFWEAINALrC0yPM3eZZ44pZ6ls3ym/64iNe6eETE26uUvu0TVtTbtlVx+L1SEi1eDgCdLNTJ83f+5D+gZ19B+1ce8TIV2TrPZYunTVtnKrbC4mIYDQoL2f7cO095qvNax09uvoOyKZ/1Qq01BZ7bJ44Tc7Xz8t2zVErUFCACLWF6mW3bp0xvLyjETR3MTH6jasLe+wNP6z4o4XpbduI/MEm5vUiCVo1LDJ9++6OnnHa8c/c/PMCbcuXLKluF9YmjKwUpNePMeNa9Q2ackRCGaMmf2F5jbynW7fvS/562s9+w/J5fa4/8ekvs6nZfsO0N1DazMtOaYmyBiLKi9rzRSMKiip88blVvu7Ow3v7bE1WdB787qm9YQENSSIkTG2Mevk/++B+Jm41JzeAAAAAElFTkSuQmCC',
feed_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKUSURBVBgZBcHda5V1AADg5/29Z/NMNzvW5qbhwmLpQkpbkqBFXvVBF0pXUXdddVMHu5Agsv6D8KbwppC6LRAqQWhdRKTmoAwvnJSOOsqmTmU7H+/H79fzZIN3tIanndDQBhAQABAAABU6PiuWfNoYnnbCzOG2yTnyIUKDLCNkZIGQETLkZMiQajoX28M/nNHQ0DY5xyBnaj85lq9we4EQyAMCObJASoyOs/VZtp5pN0A+RGiy7yjgKOU6i+e49j31GjJSQsbqLZqPEAkisgbdm1z+jr9/YX2FoU08dYRXT7L9ACCjLOmXVAUFDQEho3uTv74mZDQC43vZ+zYPTfP8+yyc4to5BhXFgLKmT0NAwNR+Xmxzb4nOAkvz/HScuXetjzTdjx2Tk3vki+cZVJQVBUGEnAxDm5iY5Zm3eOUkrZ3iwhfuXvlKs7XH6vgO0gb6JWVNnyAgw90/+PE9Lp1ifYWhjRz6yL2RUSOTB23ets/q9Xm9XYfoV5SRgiAgBFKgt8zSPPMfcuNnRf+uO1XP2HhL/eCM8dk3Ldf3pH5FUVMQRMjIEwc+4NDHbHuOhS+tXD7t4Zkj9BdcOP2N0VbP6vXf3J/YSlVTEEBITOxh+34mZtlx0NraqrLfNbalKw7+IUX12kU7Xzrm3w19dVlTEICM3gplF6TlRZ2ypzW9V+z+KcWeuTd2iUVHs3nV5tnXdbodKoICqULN2WOcPe72r59rzrxm49htqbpDlrv07VVEsX/VxJOP+e9BR3d0g/yTGS1TIwdsmWF4lJAsLt8wffi40MhlQzuE5m6P7ntBaO6WDT+hsWlGPva4653zi9ngZa3hKSdMaYuouDD5tDqSYpLqKMUoxSTFJMUoxSSlWKWUfv8fIbEaFyrl/WgAAAAASUVORK5CYII=',
feed_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKLSURBVBgZBcFJaFxlAADg7/3zZkjbYBfTZIg1SNVCFjW2hgbUQwWXHkQDHkUPgkcdiuBFiEVvXoJ4kHiqiHhQLAUXqCjGiNGaSG1Ra9pqp5JI4tY0y5u3+n1R5xm7GgMmxVoAAgIAAgAgx5KptO143Bgw6fYjLX2HqNUJMVFEiIgCISJEqBEhQlWwdKbV+PiUWKyl7xCdGs0xalj5ib8WCIFaQKCGKFBVdPfQe5DeU60Y1OqELu6eAEyQbbB4mosfUawjoqoQ8e+fdN1ISVAiitlc5tyHXJ5lY5X6DoYe5+jr9I8DIrKMJCNPSYkFhIjNZc6/Q4iIAz2jjD7JzgEOP8/CNBdP08lJO2QFCUFAQHOMJ97mgVe47THWLvP5i7RngIPP0jdKp0OSk+WkxEqoEaG+g72D7B1kaILZV5l/k7J0bO5l68maN6oBjWSLrCAhCIjwz1k+eY75aTZWqW/nvpfovomZKUnWsa9n0FPlBWknIytJiQWEQBXYWqH9Bcvfaa39qqpyaZlJ80z/ngMGm4ddTzY8ujXnZGfEtpRYCRG1ivEXqHfT/lL+33kPDj+tqEpFWShVlq5ddce+e62nW460T/isIAahYu8I/WNAlUvOTSuq0pW/F2VlLi8zWZFZ61w3evP91rNNY3fOioGIrRWyTerbWb0kyRN5keu7YUBeFoqqtHytbU9308LVr/382/fe/4FYiipHwafHqO+kfdZWmTjxzWvSMpXkqf09w8b3P+TMlVk/Ls54d/cjDiQnxf4wZenblt4xulBk3HKXD7IRsoK0ICvc8/tborDNL5fmvLf7qFvbbRqmos7DdjWaJjW1lMiRIEWKBDnDQ6yr+Wq+MFCgYSpZcvx/t+Akg61CC8wAAAAASUVORK5CYII=',
feed_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKVSURBVBgZBcFPaNV1AADwz/f7+73XnNveK9+2x6RVqMVGoaYjg6DsEtGhhLpEtzrXECGCYIkdO4wulZcgunQKhFpkan92iXBoWrRmI4e4tf9ue2/v/6/PJ9TfVMwPm5AaBxARARABAC3cNdlYcDbND5tw6OS4wWMkOWJKCMRAiMRADEgICMja3P1tPP/tBanUuMFj1BPKYyRY/pPVGWIkiYgkCJEso6fEwJMMXBhPQZIjdnH0FOAUzQpzF7n1De0dBLIMgY0luvbRIeogpFQXufE189NUVsjtZfQVXvyYoROAQLNJrUmrQYNURAxUF7n5JTGQRkpHOPIGhWGeeoeZ89y6SL1Fo06zTY0oIqI8xqtf8Pw5Dr7M1jyX32XhZ+DoW1ZjYim/a6u+RbNFg1QHEgJye+kfoX+E0VNMf8jVT+l0bGwuCMNlpdHT1qc/Ym1JX40oImD9OlNvc/U8lRVy3TzzPj373bt8TmvrksJjx1VuT0mKI9ayqsXQI4qIkSyyu8zCFa68x+0fyXXz9Bm1NCg88ZJ2dU5XaUjaTfnZwyrHD0t1IJBknDhDroeFn5j5nNIIPYPS+8qaG/+KyaoQK/J9q9Sqdv5rSEHM6H+coTEga3FzSu3Xz2xmK0J/qtP6Q+xsEntlzYa/v5pTvDYnBQK7yzSr5LpZ+cd2/Z6tO7944ORrssb3QnvbX1MVuVZd0ti1Lzfsweo1qQayFtp8d5pcgYXrKr1dSs+9rrH6iTQtmL1U0T3fcWCwT2jvlRT6iCQfHFJU3nPC/YfI9xAz+gZUbdvzUC+tdbM/bOmtljz6yAGxKy+mOeZvsLk4GeovKObLJpSN66CFGneK+1UfPqi2tqP4+6zh7R1aaCBF3mTtrrP/Ayh5+MxrJCCJAAAAAElFTkSuQmCC',
feed_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK4SURBVBgZBcFfaFV1AADg7/zO2Zy7A3fVebdEQ2yCoeVfmmhQGlRgmCg0sEAoeivvWy/BiigYvoi9hPYgEfQYKGVgiIhaaJPSVlC6hU7v9Pon166755577+n7ovRtvZ3LjUiUAQQEAAQAQBO3HWrc8HHSudyIwRfLShuJOwgJUUSIiAIhIkSIiRAhb3H7Urnz++MSibLSRtKY/s3EuPsH9y4TAnFAIEYUyHN6FrNkA0uOlxMQdxC6WL8bsJusxt+nuPYdrVlE5DkiHk7TtYg2QRtRwuMKV79l4hy1Kh0Fnn6dVw/zxBAgIsuoZzQbNEgEhIjHFX7/mhCRBBavY92bLFjOcwe4fIRrp0ibNFKyFnWCgID+zez9iu2f8NQuZiY4/QE3zgIb3qW0jjSl3iRr0iBoQ0yEjgJ9q3l2H68cpncFY1/wzxlgywHyedQzshZ1goAID37j5PuMHaFWpaObbR/Ss5Szh5idprOHNTupN8naNEgEhEAemLurfv20cxfGXElXKi7qc/9en709tyz88XNfTixSLBY99JJnJqu2pbFEGyLiXLap7JsT5/2Xd9r31htqtZqpqSnHf0p136l6bd9+cRyrVCrGxsbcbKySgJDTt8aFyYbrlX/tGR527NP3nJ+c0z8wYO3atSYezbl48KDp6WlbV5Zs27vfyckdEiBi7q5fxy8pFAqgvHrGmT8bhoeHFYtFs7OzqtWqo0ePKj8/6BcUehdKNJA30TJw/5yfHw2A0cvztNoNo6OjCoUCqNVq8mZm9MRFO97ZqnrrpvijQb365w8pDpo/v9vZ8WnpXGrV+i02DW21bNkyrVZLqVQyNDTkhe3bDawY9Nf4VdfGrzyI0pf1dvYb0a/cajE52+Oz25t1L3lSx7wu7Zw9d36wMJtzbOkurSjI0rp7lamZBTMPdv4PY0MOogadRGMAAAAASUVORK5CYII=',
feed_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJzSURBVBgZBcFdaJVlAADg532/75zN/XSstrM1YVBUpEmtycigG4sQu7FBl0VE5GUdooiuRj/3RRdddBNE94FgFkJqehPBQLLInEMHbXZmDvd7vr/z9Twhe8v+5rQFqQ6AiAiACAAoseqLfMXHaXPagseOdUwcIWkQU0IgBkIkBmJAQkBAXbH6W6f5w2mpVMfEEbKEyTkSdP/kziIxkkREEoRIXTMyRnuW9ulOCpIGcZBn5gHzFDtcP8fSGaptBOoagY3bDD5In6iPkLK7xu/fs3yZnXUawxx6hRNfMnUUECgKegVlTk4qIgZ217j6HTGQRsZmmHmN1jTPvsvi1yydIyvJM4qKHlFExOQcr37LC5/y6Ek2l/n5Q1Z+AWZPMTFDltErKUpyUn1ICGgMM36Q8YMcmufyZy7dLHT/uW47GzDQeFNr9g3HL7wuFhU9UhEBd69w9h3ah3linuFxl8ZO2SyGPD01Im0kuvdyG1uZi49/4FjvJ3KiiBipI3tdVs5z/iNuXXB7e8AjE0M2smBts1KGxIH2qG7rSfKKnFQfAknN0fdpjLBykcVvbLc+0Ww2lFmpX5OVtcE0UdYpZUVOCmLN+GGm5oC65OpZQ+2GvKgUFWW/BnVgd2eHsiYnAoG9dYpdYP0GWebA3jVbe5XRfamhwdR9Qw1rdwv1f9esbt2jJJWjLlHx43s0WqxcISs99cdXljZPuvHQi/o1Wzs9xb9/eX68a3WzqWi1hPptn3v5REd7jhJVQVFRVBQVeUVRUVZU3NrccmffoAcenvH3r2fKkB23vzlpwaSOPkr0kCNHDyVylMi5ef+o5edeKqu6n/4PaywNA5LOKLcAAAAASUVORK5CYII=',
female=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHgSURBVDjLjZLNbtNAFIXrJ8mL+QVQoVUWICFEFZCAbrJCAdpFWCC1i1QgVLEoZYFAgrY0rhPHTmwnThP/BKetU9vxT7I5zHWEhEBuWBxpFnO+c+6dWQGw8qfUisQNdnR+tD+ojQ9txz2wHPvdea1f7fDtssj9ff8fs7VnlCbCJRIzwsxNMfuZIuoGuPriwtjqlOSndS4XQMlkJuPcm2F+yXSxUGrHcD9YUDcbfC6AalMyGaNuiIvDEdx9C0FjwqAJfMGD/qRZywXQvFk6SxwfOOhvdwpGRSnYb/pslASxMYX6UHRuaDB0aGYCuO9N9J4rhW65VTC39AWgN0XnnpAPMHeN2lT1F3XrHqzXBsxXOq4+u0idGN7XMeTiaf4IvRcKP/5oIxlEWSKJjCTayeClhubaUf4SlWcCp5el0ujtEGHrOtt8asUIxQmGzCzdOS6JxW/5z5hBHguc9qjJdzekfqQGSM4jdFaFsHX7hG+sf7/5I5HaG2dV7UFD0+6K2lTxs8W1b536yuoPTVo72lwKUO+Lc6pP5nQYZ/uIeyGCMw+shbYUoBTrGYBE9UmhdI1A9CD/D0BaP6nKrC6rrZGRRGc5G+H401LAb5HBZ7WpOp3z7uUC2HtXqDKJJe/k3fsF8TkmWq5zP6gAAAAASUVORK5CYII=',
film=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIfSURBVDjLpZNPaBNBGMXfbrubzBqbg4kL0lJLgiVKE/AP6Kl6UUFQNAeDIAjVS08aELx59GQPAREV/4BeiqcqROpRD4pUNCJSS21OgloISWMEZ/aPb6ARdNeTCz92mO+9N9/w7RphGOJ/nsH+olqtvg+CYJR8q9VquThxuVz+oJTKeZ63Uq/XC38E0Jj3ff8+OVupVGLbolkzQw5HOqAxQU4wXWWnZrykmYD0QsgAOJe9hpEUcPr8i0GaJ8n2vs/sL2h8R66TpVfWTdETHWE6GRGKjGiiKNLii5BSLpN7pBHpgMYhMkm8tPUWz3sL2D1wFaY/jvnWcTTaE5DyjMfTT5J0XIAiTRYn3ASwZ1MKbTmN7z+KaHUOYqmb1fcPiNa4kQBuyvWAHYfcHGzDgYcx9NKrwJYHCAyF21JiPWBnXMAQOea6bmn+4ueYGZi8gtymNVobF7BG5prNpjd+eW6X4BSUD0gOdCpzA8MpA/v2v15kl4+pK0emwHSbjJGBlz+vYM1fQeDrYOBTdzOGvDf6EFNr+LYjHbBgsaCLxr+moNQjU2vYhRXpgIUOmSWWnsJRfjlOZhrexgtYDZ/gWbetNRbNs6QT10GJglNk64HMaGgbAkoMo5fiFNy7CKDQUGqE5r38YktxAfSqW7Zt33l66WtkAkACjuNsaLVaDxlw5HdJ/86aYrG4WCgUZD6fX+jv/U0ymfxoWVZomuZyf+8XqfGP49CCrBUAAAAASUVORK5CYII=',
film_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ1SURBVDjLpZPvSxNhAMf9M+pF4AuV1DU3cBvoLLexzbWJbdEGuWRnm256m9MCxQwdBFppi2QQVBLWu8owYuipCPmjYIqnnVMWsjWpF1mvoksH7ttzvrD09FUHD8/d8Xw+99z3e5cDIOd/xv5Ja2sr19LS8tPv928ct9hms8Wrq6t3TCbTmkhA4G2aph/5fD44HI4jB4FRVVU1oNPpVkUCAoPAPzweT4ZmvvDXJ7/zgbEt3hvd4hdjPfy3tR7eaDRmtFrtQkVFBUQCr9e7TOBbFEVxyulu+OY2EHwHNE4DL2Mj2El5odFo4mq1mlapVKxI4Ha7ky6Xa8bpdLL62RpIGDNcU+uoZwDPOHBz4hPKy8tXlErljFwuT4oEBE4QeJi865IzVoPONQealrWoex/EhbEozo3EQGCWwMMSiSQhEtTW1nIEDpGkF4fSbXi+2YWnmx2IpK4i8vkyBtOXIJPJ2OLi4lBBQQEnEtjt9rTVamVJFse2UFRUxOfn57O5ublpkYDAnMViCZGkFwLz22j/ALTNAfQssLT+EK+nOtEctu823KuBvfvsrqlD2nVAYDabEwaDYZgkvXS4hTuvuhF6QeEtFwH7lUF4ogmO8Bmomk/27wv0en2ysrJyhiQtauHK7fMY/Xgfo/EHQv/on/QgPNkoCH7vCwjMEpgmSa8ebsFyQ44o9xj/Hm9WIoLg73deVlYGhUIxT5L+9SQZyD5LtWeHUteygxt1WXXwFO4y9ehlXHtw77hLvIPS0tJMSUkJRSSi9A2UChf7T2OAce89WZiF6wMZSKXS9cLCwkxeXt7CUX8iWdynaDyxK2xb0bA39wn3/wBLOHSjT9MDogAAAABJRU5ErkJggg==',
film_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJsSURBVDjLpZNdSFNhGMf/787Ome8SF7mxvuecJGU6SLuWIGZ0E7WgFRiBdZFE1CDwJoyI6iYvhIqKoqibrsIgwUkUjDCkC7UQ03IW5EfS8muTvec95/SsXE3dXQd+8HLO8/+d5zkPh1mWhf+57LlDNBr9YJqmj/je3t4eKFQcDocHdV0PSClHOzs7ty8TULDCMIzHxMlIJFKwLQpnaSP2reqAgg7iENl1T1ObLLI5IKQFYQKnPDexpRhoPNdjp3A9sSOXs+UOFBwgbhPDvepdnuKz3OZ0c4u7eQI13MUnuRDiE/GQ6F/VAQVLiHpCutQ+xFPvUKfcgM2oRFfyIPpnqiHECUlvP0K4Cgl0IkEPq70OYHdpMWZEMxbSNUjOhjA878nObxLZGu8qAd0US4KqBm8AGnNCwo+UaxrY+AQm03FfCCwJdhUSlBAHvF5vsOvCtwI7sNEIYlu2JltbSDBHdCQSCVnZ2lHLaQu6AQhaaJP7Dpy9PRiaG1ecmYXgT4vpsZC9JRST1/8KyK6RwE8obzPXUKsdh8Mo/y2ZfPkem74MI3TmoubwV2FxIKYOxruvdO9VU7Y8gUr4CfZnC2cxo3+kzoDFVzFU1B9G0efXYI+Owjn6DGW+UsVi1vn8EWaJp8TplVsoSY6haL0f2B/99w9c2gDFZGX5IwQpfIwo2+P2WRrj0PlmpIqnkXE7WLrvBdY8b0ZmcRJpqp+fU2AoGM8XUFa/p2nag1jL1LLvL9hOrOuJw1fKYVdUzE9LjE0xw4J1i+V+52AwOESCcmJgZGSkbuUS30S2tnydmLjssUP9YVhyrYbWhpi8+gtmOYLutSnwxAAAAABJRU5ErkJggg==',
film_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALpSURBVDjLdZNrSFNhGMf/Z9uZnjldy+lKKF2ThsqULgZdLAIrKTHKMLvYhS4fgiKLoi/Rx6LID0aIaRAUpQSFEcIMiS6mhoFmN0duVpo6azd1m+852+lZSeSlF36fHv7/532e//ty5eXl7yKRSCrhqqysNGOWU1xc/EEURbMkSY7GxsaMf2sqEqaHw+HbxOHS0lJ5NgMSR6kgCqbXVCSMIbaTu5h0qEKKVcSASTJYBDiSdB0LtEDZyVYVidcRmdMNFCR8S1QR9tf8DWFc8AkKjUGQBYPgRLagE4YExthn4hbRNeMGJEwg1hGSju/Ei/EOLFdehSJsgc29DV1eKxg7IFH3nYRuNgORcFLRaowBchO18LJjGAtkw+3bCPtoUnT+COFcnRU299zdJSu1Voz/eAYuNLA/OhubNMjaZDRDzWkgwYRx3QiQcgcRTsRNxrAnPxC3d/O2+YIhDzrTEnidFnxqqr0cNUggthqNxhzbmYFZMlBgZUYwY/++kmydfgXcvR+pCUN8wjxoExfqogZ+osHpdEqWCw3LBEpBDAOMAj1kqEaK3AellK7WmwswMVgHtYbDl85PEEPsZyg4tkpBm1UTJkLZNnER/rADkXA0e2BkxA1e4qBPL0Toew043gNeG4/YQO+w6B/Mzz3+wB414CcNuD8pnIBX7MG84UewxvqgSy9CaKAKCl4E86fB9bRtlI0GCpeeetL5e0C6vo+oJ/hoCkX0csxiGXbMeY7krC1gQ7VQqmVM+FPR39wakYKB7MWn2zv+xkidc0i8m0hbb0iVld3fkavawElcMga7r0GfrEbIvQBfm1tCJRVdsfZ+b9+UFZOBSNTwPK9pOjfMaT56uMw1RxHz7SXsDS3otLnx4p4NVx72nu1zBbwzMiKxg6j2eDxv6urquJ7PQxJ7/ximnDzExVkw0NE9dr7e+eV+u6uS/szP6QacLE/9gCVr5waXLjKqVmSYIPMTr4KukYOFl9468J/zC7f3tq13JhENAAAAAElFTkSuQmCC',
film_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKySURBVDjLbZNdSFNhGMf/Z9txO8t9iBuHvvzIaszcXKlBV5qUehFWLmkVQWAhGEQjom4isIugyIuBRHUTpRdBSFYIdllQoRlq2ccWbpCGms7PTXc+e0Yqw+3Ajxfe9/3/znPO876M3+//qihKPjEVCASKkOHxer3fRFEskiRptKenx5m6pqPgTlmWnxLnfT6fmklA4SRtRN3GNR0F9UQD2UV7U5tk0OghSCoEBbhgb8f2bODs5Q86ClcSxRsFGgoOE/eJYB/7kItx85zGaONUzsaF4eYs3AQnCMIv4jExlFYBBc1EJSFZ2EG8i31CufYeNLIDvdHjGJpzQRDOSfT2k4Qlk0AkwrTo4vVARW425oQWLMXdiM7XILhoT36/QiT38GkCmhRWBXtq+SJkMUZIKETM8hfY0gGFERHxbLI2HrZX+++MLGUSmImjPM+X9l4dT+uAntWgqTYnb5enkTlRPbmYSbBAdIfDYclxs7uMoy6IMiBQQ5tsD2CLRWAwljDmzXtR5bEYv3cdq3A2vOhf7wL92SyikNB+TNzGgjwKRU72Hhib1SM+HoLJZoW88BKF+5u1ckK48aWjjk0VsKsC5n8XLmFO/AlJpLMwEUWu4xSw8hl9TzqRbV2GtCIcIerXBVT+PPGMYJNdqKeTY7S1YJuuGfvMUZhy4lASYUBVIC/1o7DqCiMlErf6AgdMaxWUEqeJgoO2fNXJFagVJo96aOY3rHkeKPFhyi6jzOuAIvyBwRAE7/I5pYTgXxOIxCOWZY1vrk8yr69FmKn2EJPH18BomoYqzQCMFgNdQdquQFkJwr47H3JCvPi2taSYcbvdP6j8HcRwKBQqT1qpvDHXmc6tWk2EBLM0k3rHNNByDkyODCL4qvU5o6rpF/D93bJpVVaMqqJS+UoK6vqI5KiqA/8ArTiuh9VnDK4AAAAASUVORK5CYII=',
film_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK/SURBVDjLbZNdSFNhGMf/52xunTldrM2JaeomlZnTalJQOSmyLithSRBFH0ReJX3QlVfhTeWFUJFQCHVTV9WFVCCUaZaYpIb4kc1mmihOp5vO97znnJ4jFqI78Lt5n/P/ned9n/MK1dXV31VVzSYm6+vrPUjwVFRU9Mmy7OGc/2xqaspfXTNSME9RlKfExcrKSi2RgMI6dcSxtTUjBc3ESbLLzgt1fINoBuMamApcct5HlhU4c7XdSGE/sWOtQKRgD/GQGOxIapBiUkQSLQ5JkxxSEF7JJk1IjLEfRCPRva4DCqYSfoLbkr7hY6wTPsM9iMo2vA2fQPdsIRg7x+nrpwhbIoFMBKlY6DIDJZusmGVViC54EY6UY3Deqe9fJfR3XOsEtMhWBAVHXR6YBAs4chGzTQEZz6AKMh4zhhXB7rUCIRAIhKgQdrlcRTMzM4mGgN7e3kX/DauZQxaZyi2N53sWV3cwR7wKBoN8e83rPZLBBKbQ6GgKAecbbLYaUF5W1b+kyjuz0/LFvrGuheMPciwvq0aWJUY6WRMJcglDR+wRiqUKmLUMiKOleDFC89c4DtxK3pVhz0N++l7Mx2Po/v05WnrXbmu5Ho7qgqQVgZBiHEZ7+A6Kkq/BqjIcKTgLRVOhqApUaBiPjKIwcz+ibFH8GvoQ8d4W7foWIsRz4orDbEKxfSNm4g2Ic7Yc/jU9RNvh4Cp1o8iYW5pHcdZBROUF8UuwdVrvoIjCp4kcf1qmloQUyBYDWnhc4AqHK3ULhZVl2Z9ICHZrOrpG29A23DrFOMoFn8/HKHzZZDI9cbvd/0+elXUgTuNjtBW9G4+jAPvc5egMfULz4LsJCh8er9X6BK/X208CN9EzNDTkSzRGSZIG3DeXtpZ4DuH9YPMY3RM9PLD8H2hawgu47nHXCDFFNVgWVcU9WasF/63/Ban+u4K8LTKZAAAAAElFTkSuQmCC',
film_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALVSURBVDjLdZJdSFNxGMafc3Y2PdN2pGYjsdS2WFu1oiy8KKSINPqAGtgwhMgKrMAkiIigu7oIvRDCPjCKgj7og27sg6ALg75EW2roMldGpbk2t7nV/uer90RF2Drwuznv+zzn+f+fwzU1NfVpmlZCfGltbXUiy+P3+1/LsuxUFGW4o6PD8/dMIKFLVdVLxK5AIKBnMyCxQQtRPXUmkDCH2ErucmF9i5LL54ApOpgG7C48hdn5QN2BJwKJKwnvVAOehK+INiL03HxWTIlxkbfaRV20i2H4REkcFRljQ8QFIvhPAhLaiEpCkcwv0ZnqQrmpGbzqxv3oFgQnFoGxHQp9fRshZTOQiTANFzlygOUz8jHB9mIy7UM0vg6hZKFxfo0wdhz/GNBL9stgQZXDCQtnhYIypKRxoOgyNE5GO2PQVPld/Spu/ciNqheZ8Y+n5zX0tRsGXE1NzQiJow6HY3EsFstWAnp7e7/5l6ixxsbqIslbi8iTluinF93blx4J3TMSJIg74XBYcR+7s0ykFmQVYFRovf0MivM5nDz4KrJ/z4piyV2O1Pu7ME9zTy+Yk7l5t3HuLp5u1kKUEaanmRNIqMMU1+geeJucCZvSg3K31Vbg28ip6TfItRdBsAKzKpdYnd45bQIJzZSgjOD+10Lex81Mjr0Db4qA41Ow2CLA9zQmP2dMxhHixDWiYWoL1i9erGFPYVtTWqAp/eC1CfpzpkGXGQauDn7vH0zsMRIsJnEtUbraXqJbOBGyWAwpGcLcTA/nrqiFzh6YOXUU3bdjyMt8HjNxHELDiX11V8I3DQPSyucsFsv5B4fH/tz8gbUaPBt2gkXaIAgS+h9OZo62D6mHF4rzcvNMCFwcSf5c9Pl8Ax6Ph7lcri5d1/GbD7eqn2XGjuupwU364+YKdaW7QOd5fujvHQMhGAzOz9b9p3DkYSL+qDQdTX17H/p6qHMgdj3b3g+UOcQZKOdjkwAAAABJRU5ErkJggg==',
film_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALQSURBVDjLbZNdSFNhGMf/55xtx6P7aG1rbiSbORwpm/QJQeGQPq/SFmiFFZjBQiiF6CboUrxIUqioKIO66SYRQVAiBIkiilJGyJQGrjl1pq42c+ecnfUcEQrdgd/Fed/n/3+e533el2lvbw8riuIiFnp7eytQ4AsGg98kSaqQZfn70NDQ7v/3NCT05HK5F8SVpqamfCEDEqt0Eyc372lIyBNnyF2ytXTLRSwPUc5DVIBW232U6YHmG+81JK4lqjYbsCScIB4SkY/ax0JGSAlssVXIC1YhCr9gEuYEURSniefE+JYKSGgkagnZpP2Kscwn7Ofugs15MbzUgPEVH0TxskzZGwlTIQOJiNKmz84DByx6rIjXkF71Yyl1HJHfNrV/hVBj7FsMaFHcMKg+Ya+AjimGjHJkTEnA+RIKI+GpKGLDYG8hAyNx2m631wzfjBeYAUstiJVqjBpbyOAXMRCNRmXvnYF9Ak1BygEiDbTF+gg79QwOBT5PUpWDFBfcYkDuOtooJ7gP2U7sQSN0U3HYlyfwxWbGm4VpGI1GfTqdrqqvr6/s6+vLJxKJ5Xg83hkOhx9wbre7QxUTvOFoqjTxdgSHtUZcvHAOTqcTBtMO0D0xOBwOVygUEjweD2OxWNTRHiMTrdpCinhFhIqn1uCXqnGkrg4d924hFf4Jp8MBn8+n4zhO19PTg9nZ2ZzBYBhpa2s7NTMzc1VtoYbE5wm3NV6EbXrzem8HLyn5wetphq43zGYzqAUkk0nQe/mj0WjOUkjGZDLxqgFppSc6ne5Z7N0aOPePdYOx26uMIjHo6upCSUnJ+lomkwHLsjxV06/+UwtrjN/vnySDXcSEVqttph5fBwKB0rKyMoUybZ+fn0ckEgHP83C5XKBxgyayFIvF2NHR0Tkmn//3AL1eL0eH5aOM/XSANhJx6kVYXFxsVRNarVY1szabzeboLJJUUcNfnRWk+01xXkAAAAAASUVORK5CYII=',
film_save=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK4SURBVDjLbZJfSFNhGMaf72znbGfNaTRboKViYQUqmFp0IxVEFyMpoaQVBCWhV3kRdBPdCN2kiBBRgURdpBddCDIwIYoILcwyJEJiC8tIJZuzze387TmLxGxn57dzOO/3PN/75xOdnZ3TlmWVkYW+vr5K5LlaWlo+6LpeaRhGLBqN7lkfc1O40zTNh+Ria2urnc+AYocecmxjzE2hh5yku158ocfwSh5ohg3NAtqKb2G7Hzh3ecxNcRPZu9FAovA9uU1mXst31ZS6rEq+oGqrQTWOGrVQ/a5qmvaJ3CdT/2VAYYA0EaNQfocXqQnUu7ohmVUYWTqBqUQ1NO28wd1Pk8J8BjqJM1gd8gANW/xIaB34la7B0vJRzKwUO/VbxFkT2mggIpHINA0eMxgO95bXKcIHg7/J4bMooGFaM7DZr+DbwiqEBPh9Lli2wM/kKp6+ndvvNCdAmkOhUO3Ilbk150BjBocbq9Bc58fw+Fdci/wzPTx4tgjLEq8cgyQZisfjRtX1oX0qp6CbwPzzcSytGLnFoxPzCB8ozb1n+EljfDGhQQgLEjurkAriGs/eQNKMweICSRJw8VTopoXu9jruZsMkLAA+t4VsNotkSoObQpkZVBCxfgpeWUImq6Nr8DPoBYNGFnFOmm3zn7dPZgYULpNBIjtTOM6T4wt2wOBuWdOAS5KZqhvMh08XhRINBDg6jtfIZVBL8RlSfihYZitCha6WIprWhUxhe7iAZppz5nK72qzb61Zw89EsfiRTOQNq9XuKovQ/uTq/1mVzt1OvzIpX0T95CUHfDmYhYTH9BW0NvXzzIp3904MYDe6kUqk3AwMD9X8NTnWN2eXbgJB/BUdqG1GklOSSSGRKsHWTAZ/CUygERK4hea769tGXu0rVg0WeIqYcYBk6hGVRYMISaXycjWEhqeM3NCmjdDDSuwAAAAAASUVORK5CYII=',
find=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIlSURBVDjLY/j//z8DJZiB6gY09895tGTDnv+tE+f+B/EnL1gHZofGpt4iyoCK5r5H63YcBmkAG5BRVPO/b/aK/0CDn+A1ICm75H/X9CX/azun/m+bNP+/iaUd2AAHN5//WSV1/wuqWsFiVvauyxWUVHkEhUWZwAYsWLOTo6i23aOpbzbYqYXVbf89/MP+u3gF/M8pa/gfm5b3PyKn6X/txGX/S1qmgOW4uXmq2NjZGcEGTJi7mmXKwvUPF63b9T+3vAmMqyeu+j9l+a7/fUu2/2qcvuF/be/8/9G5zf/DkwvBLmRmYXnAwMDADDYA6FxWkM3TFm/8n11a/x/k55Tc8v/RyTn/1bT0wDaCXAITj0svAOpi+AfErGAD0goqWf1CY35a2Dr99wqM+G9sYftfW9/4v6yC8lMRMXEDSRm5rWISUv+B/v4vKi75n5eP/z8jI+M7oAFM8ED0CYo6DAq4XYfP/F+15cD/7hnLQAG2AiSnqqmzorJlwv+1Ow6B5UAxwscveBglFtx8gv/kVzSDDQC66H98RuF/PWPzqyA5oM1XQTEAMiA1v+J/emH1fw5Orj8oBji6+/6HGQBTpKGt/1NRRZ1RQlr2HSjgYAaAwoKVle0/igHWjm7geAYlIJACUGDqGpn9B/qfX0lV4wrIAFAsweSAYYBqACiBGJhYggMP6Of/QJv/S8sq/AcGohTQv7c5ubj/A+MdFH2gGABj2mUmUjEAnjJojQ5aPHUAAAAASUVORK5CYII=',
flag_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIxSURBVDjLpdNdSFNhGAdwKSNyg6ALC8GLoE8so49VRHVRFFQISlJeREXQBxWSSZDUpNrOTDoxSGvhmokV1TazLeekTBda9rVKmW5lYq6slgutOI7j1vn3vKc4rCAv3MXDgfPy/73Pc3hOEoCkROq/B6v2GdIWHLnhmK1v7ZtZ3PIuo9DmOr17iaUkLx1Ud6g2jgqo+JCU4x7Bs5AEe4+EhbYYMsv9iEYGcU+/VEZkYNkew7iJnHBrgl4YSeYEJJcIGF8qoKw9Bv8g8GkY8IaBthDgqatCsNGAq4czGbBLBhbv5VWT+EiL2Q9cfg2YA0DDe+AxBeqDQPvX3+/PdwKmfA0+PDDCuGM6A9JkYP5Byyy1SexgQM5dIJvqpJdCb4DWz8BDKguhhzxDor1Ig+7afBaG8hFnFDiyp1ZFgxa6JbcR2NoEnCLg2ltqfQBwUzcVhJc1+4c8/Br0urV/A9OKvJyxQ/qmfQ5so/D2ZoB7CVh7gN4fwP1+wEWjGK/XoKt6C9rOrWeATwHUugEn3RBjrW9oAI4TdJPCno80RlfsZ27d9+Eslxitcdpk4HbxCgboFEB1JvKk3CfhSjdQTXM7+yRorCLUZ8PSposvvMZX0bydtf2Vi9JT4avcjIr9GQxYrQBzC2zmVG2nkGIISyncF2mKLiDOKbQ+it8JCqy9dGCe3EH8/KMu0j9AqePEyoSAwFNTVkKAHG7i1ykrPCbAfmw5A46OBbjw5y9kz8nxZ78A90vOcDVd+i0AAAAASUVORK5CYII=',
flag_green=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIySURBVDjLpdNdTFJhGAdw120X3XXR5kU33fQxS0+5Yl24lFnQKsvl2nJLM0fmXLNASceKgAv8yBGgJPEhkcIShEEYKuKU1IxcTm0WUDiJ1Fpbm1tZ/855186oLS/k4r/34n2e355z9rwZADLSyX8vCm+WU6fqT38+21S4ztPy1rmK4lXF5Ry//Hwm6LjoHN8QOGOgUOe9iGByCJ7FJ5BMX0ORiosfa1/wTHqQIAQ4VCHbwpXL53iWHPAe7QefJAvq4G2MLY9gcnUcQ0kf/AkvAm4DPvhl6Lq+jwEuESD7inLrCWXJ10BygC56SgpHlofxfGUMjvhjDH7sR1e0Hfq3VmiqKSwOt6CldCcD7CDA3qrOXfRo37tjRojC5SRt81KYIxp4lxx0+mCOaqEON8NeR2Ght5ppBvsTT9Yqai60F/y0vTehPlyBW+FKAliiOvQnPGQKY+Q+TOOdCCjzEPU2/A1wxIaH3a8N0C20ouGVAI3TVVC9kcEa0yO0MgrfkptM0mprwqypGKG2AgaYYYEsqfGFI94D4csy1E6VonlWgt64Fb6EG7aYGTdGK1ETEv6yu+wEcDQeZoA7LHBEJfxkiejQQxczccZtEE8JwHNRKLMK1rRzng6R3xU8kLkdM/oidAh2M8BRFsi7W/Iu38wBty8bXCcdSy6OyfjfUneCbjj34OoeMkHq92+4SP8A95wSTlrA/ISGnxZAmgeV+ewKbwqwi3MZQLQZQP3nFTLnttS73y9CuFIqo/imAAAAAElFTkSuQmCC',
flag_orange=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIvSURBVDjLpdNdSFNhHMdx6YVIhaALC8GLoFcso5dVRHVRFFQISlJeVFYQgdKKiEprldrmS8KgWYK2raZGbaOcafamLVxq6qxs5qxIHYpkUiYsmVvn2zmnGBbkhbv48Ryeh//n+Z/D/4QBYaHkvweagxujb6cttzlOLuqtP7Wgx3I0tjr38Gp9TnIMYu6L2TEh8DkjQhgzJSL0tSC4rAR0K+i8EId/9BtPLq2RERnQ7Fs7xZs/4643b/qYN3caXrWY7KkEGnQw2AkjA9DnhN5G7FU38DzVUHYiTgIOyUBByqqI0ZyZ9bSUgNMIzeL6/iF4mqDrAQy8+b3fdJUipYK+51q0KfMkIFoG9EeWLfRlRrbLQFkilCZAbSa0ikU9DvHmF+KznmHzcZ81XcGHe0qpmOBHtB2bn+BXz/HQoofyJLi1B+qy4FU59Iutd9WIXRXTWaEbthdsprtG9TfgzJirFhza7zxWgXkvWPbDMzW8NcPXbvhYC+5qWiv1vDPtpvHKNglwBYEvmshK8YaA3LphOzw6B+134JOdQKvx54gx6YfPGO9/XZ4uAxXn10tAdhAYzY94KTQWQlupGBNCRyW+QgVDqkih7fJOp79em9x/84BhZUwULsMuilNjJWBTELAol5R0qKK8Q1nhwmBmuOA+PdtnTl3cMH4mxIIt19OWyh2Mf/8JB+kfIM92cUNIgLu5KD4kQC6uK9gaHOFJAdaz6yTgzGSAa3/+QmmdNf7sF2A4ynPOLQFtAAAAAElFTkSuQmCC',
flag_pink=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIdSURBVDjLpdNfSFNxFAfwX+GbSGjhg+JDED2YBYVGSPiQJigYRRZGUBD0UuBDT2rkIjOV/LsH5/5oIlE9yHSbFsy5TV3zH3soZcQWtD+CmjPXcru62f127q9YM8gH93D4wf1xPuece89lAFgy8d+LrtOyLGO2WudOe+t1pQ55LJnq0ea7+b1NVTmgMFCU7wmEmE1EmRewh4E3G0DeZwQz5hETNjD29CxHOCDPlR2MsnHtFjNFBTYOgVkQYVag7SuwIAD+GDBL51QY1pF++EzP8PLBKQm4wwHlsYZUkb2fQs86oPkG9FCMhgAbVTbQ6RB+P5cHoKguwNJEBzpuH5WALA5os9uPI+XDRw5c8gEVFPWrlERtWwmZoFDR3a3l7cHaAriHqqVkxF/idJrmMtKdPqioyhU/ULkEyFaAgSC1HgFGqAvFOjxNzqC19QK+vHu0G/AzbSOer31HHVW9QcBNAp7Q/K8JcEcB4w9AH8Jwiw7OgeuYlpdKwGIcCLMxPVXY4a2X0luvJegVJZs2AWXgJ0q8EZR4YjPX9BwYri+UgIa/e3DANovOANBPM7+gMbTU8kkXfQm76M2fdKB5rWqrzNV3JicTi31Xobp3QgKK4oDliFK9ygzhTWYWQ8wkrjDjtvmwxp64E5RQrLmfxztInH/PRfoHaNE9Pp8U8GlOUZEUwJPNrRfjK7wvYPDhOQmo2Q/Q/ecvlM5DiXe/ADHD2LkNLqYxAAAAAElFTkSuQmCC',
flag_purple=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIiSURBVDjLpdNdSFNhGAdw6baL7qUuuonA6Cojoiw6qwth0VUsDIxKggohIXGtpA+1DxQhwoltsXKjNpbruC9q5jypMwf2sdlqbc120ixzfqzV8Lhz/r3vWRw0yAt38eeF857nx/McnlMEoKiQ/Pei5ayluP2YnzUdeZd8XpMWzef4hRtV24zXNRtA4iQpXxGwMvPSgFbEzHsJU6+BoTrgUVUIi9lZ+Bq2y4gM3DplWePdK3R59giCu0yAk4TdLeCjXUI6CWRTQJoH5hJAn8sEvqcJ5pqtFDguAy0nrGtd+3L9Yy5gzAt8Iue3IJCKAJMvSWEc+BoAvvgBfXUpxrlWtFZupECxDNyp9GxyqMQQBQIXgUEdEDHlsR9hYJpkYpA8M4uwa0sRc1TTYigf0aAJHGLV4BNuMmc9yRXy8n0g6QNmoqSLYQL0A7GeDPqaGQJfWg48PBhrjNowP2oEgg0kTQTozLecmQS+j+S7eOVNImy8gKHbBygwqgDdqp/dCSdytHWuFggbAL4XmHpDxnBLIqfL/uZqc4v+q7N429aJJ/U7KXBNATxMbjj+GPj8jOQpaXcA8J0UYVNlJPZ8fCRqFTVcY+peyfrNCLVr0XG6hAJlCtBx9MVdm5r/5WAyUheTlizlEwv6Ci6wdCdIAWM4swWRB4eXzb/iIv0D3GQv7yoI+BDUqwsC5OLe5v3KCq8KsOt2UKBuNUDb37+QnuuW3v0BGUzmBpilPwcAAAAASUVORK5CYII=',
flag_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIrSURBVDjLpdNfSFNhGMdx6Q9RDoIuLAQtYVuWpZSunKQXjYIKwdBqkJkEYRhIBEFRRmVbGYNB2hTnVKaGzWFq/im0mLgyLTNjpdSFJkX/hCxYY26db+ecYliQF3rx44Xz8nzO8748bxgQNp/8d8OoS41s0Ca0uBPXvu3VqMYbk+Parx5Nsl3RRyHmjpjdswKfosOF6ey9CENPEFqdBNM2MaKNJ+D7StflLTIiA8bUrQu8sUuavOrF017lIrwxYqIXErSWwOsR+PgBhgZhoA9XWw0T3UbqTsZLwBEZMKUkhvtUS3uxW6G+GmrEtfsuPH0MXR3gGf79vfIGZQUa3vWYMR+OkYBIGbBpN6r9qxUvZEBsmYMZUHwR6sSiPjf0P4RaG1OnTvidZzS8uV0gFRO6xBaNMiOgXjmB3QY5WZB7AK5dAkc9PBdb7+oUu6pgpLRkymXazlhn4d/AYMIqg2Axf8NQCHnZcCwHTAZodsD4GPTch3vtDJeX88q+n77rOyXAEwK+rFe0in8Iyq1n7oKic9B0C9wugjerf34/lPXDr08PuPJyZKD5fIoEFIUAX2x4v2AthYZaMXaEjlb8Og2TaxTCs317BgMWs/59fm7V5qgIPFWZVOTHSUBaCGhMXmd9GR/hnVQuEz6LGVWt8DuSYh/NnAmxQFd5fIPcwczzzzpI/wDFLRe2zQsYHShLnxcgFz8w7QiN8JwA59lkCTg9F8Dy5xVK6/KZe78AQiW2y4SvvaoAAAAASUVORK5CYII=',
flag_yellow=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIxSURBVDjLpdNdSFNhGAfww0Cri+gyKLowpMC+LsooEy+SgqJuKqRIiIQKkryoi4zaUmbWSHKdPkYz05xdnKNobmwW6Vi6tbk2TDYl82PTTSr3PXe2s2T+O+dgYwV54S7+vBcvz4/neXleAgCRTf570UXdLda9ORUytW1LDbbkp1TK8h8PLu1rvn92C7houBxfEbA/E+Hn4C6wAQMYTxO8vbkwvMjBYiKED3X7BUQAaFqao6XLgxZyDaxyAp9JArYnBCLjd5CM2bDIupCI6MEEtRjQtWK2rx7t13fzQMUfYHNfx7H4wtQ9xFwPEZuuR+I7jWSgH9H5FrBRI4KeGgTcN6CoKoT3YyMaL+TxwCYBoOi6M5+6i37xgM9YICQ8elnAmKCai4YDJHCPnEDnrUJMdFfxxUg/Ik2JlSPq7anYtAw+0x74zXs54AqYGRLxMN9FK/yem5hySpcMDYfh6hX/DXRR15yhcclS2FEBv+Ugl0OIjFWCmVUgGR9FzE8h6mvGF7MMY21lMJNHecCZBrRUWXhhcrn9ga0IOy4Kxey8BoGZWnwbKsCkbSOGX+cJwFtJEQ9I04C+o5SNTojBuOXc3I8Qn1Nh7v062BUiWHXnWLtD+1TVTxt7anPhfHUayqs7eKAkDajbz3tN5HpYH4swJBfBQq7Fu6aSROZOcAWlLyt3Ch1kzr/iIv0DyHpqirMCvloVJ7MChGJ9w5H0Cq8K6Lx9gAeqVwM8X/6F/Lkh8+43zznRPkqpYfEAAAAASUVORK5CYII=',
folder=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGrSURBVDjLxZO7ihRBFIa/6u0ZW7GHBUV0UQQTZzd3QdhMQxOfwMRXEANBMNQX0MzAzFAwEzHwARbNFDdwEd31Mj3X7a6uOr9BtzNjYjKBJ6nicP7v3KqcJFaxhBVtZUAK8OHlld2st7Xl3DJPVONP+zEUV4HqL5UDYHr5xvuQAjgl/Qs7TzvOOVAjxjlC+ePSwe6DfbVegLVuT4r14eTr6zvA8xSAoBLzx6pvj4l+DZIezuVkG9fY2H7YRQIMZIBwycmzH1/s3F8AapfIPNF3kQk7+kw9PWBy+IZOdg5Ug3mkAATy/t0usovzGeCUWTjCz0B+Sj0ekfdvkZ3abBv+U4GaCtJ1iEm6ANQJ6fEzrG/engcKw/wXQvEKxSEKQxRGKE7Izt+DSiwBJMUSm71rguMYhQKrBygOIRStf4TiFFRBvbRGKiQLWP29yRSHKBTtfdBmHs0BUpgvtgF4yRFR+NUKi0XZcYjCeCG2smkzLAHkbRBmP0/Uk26O5YnUActBp1GsAI+S5nRJJJal5K1aAMrq0d6Tm9uI6zjyf75dAe6tx/SsWeD//o2/Ab6IH3/h25pOAAAAAElFTkSuQmCC',
folder_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIuSURBVBgZpcHNi01hAMfx73PmzHWUO0QmbiOhyQyZMYWGRhbYSfkLpKwYpSRTSikLxE7IxsrGSmJBGtSQnXfJ5CUvg+G699x5u/ec53l+zilFNtR8PkYS0xEwTQHTFJJ5dX3to6hlxQpjAn4TjbHXH5ytrgca/MmQm1i+7akNyRgFHYv6LjYbY0BkBMZg69+Xfn109INA/NJUaJFcOjr+efAgcDkkZ1XHJzMaX87hkiYIWjCmSFTaTGndiQIS4EEeECaYtWD4at8R4HJILjWBfIJLCsgLP/WedOIr46N3aI4WglLwCZIFLMWOQwXkF5MJyRlF3k6RTIKSCdKxGsWOnUTzOgEDCPAgAcKEc8AFIZmQXBoQzmxlTuduwIOE8PjkE7Z6C7kY2RjZGnLjRG2HoSFyIblUkqvjJ58gFyM3hmwVn1aQi8FWGRyucPedI56aIrX76Qlc2A6E5BpI3uLTb8jGyMXIVpGNkatw42WFp/UimzauoW1uO7efX+Hes5HwUn/pVEAukQwO2R/IllH6HdkysmVkK1x7UWV1RzcucHQv3IozKb2rNpDZG5BR4it2shyn4wVvJ4vYxnxc0oZ3y/BuJaO1Ms1mFts795E7sPkCy1q7yEQhGdUbJ9+c37EOsQVDkb/UapXo+ch9Ho8MMbD1Isdv7iJqmkGmbiTxL739pWNtS2Yf3tDVR/uCHoa/POT+kyE+vo1PG0n8j97+0nFgD1AExoCzD86MDPwE/3Mt+7fw600AAAAASUVORK5CYII=',
folder_bell=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKfSURBVDjLlZNPaFRXFMZ/982bSSZ2YmKiQpoIQSsihiDVYsGFBUHUhmLQLLsQLCgqtLhyoRtdWCwU7KLdqBBXRVyUCK2CiGYXC9UiamIaO0nGzGgmybz5c+97797jYtIGEWJ74OPAWfy+78A5iAj/V5dP8nXuZ6rXT/CLz3+sn46pr7o+ZGfKp3FXL3uCSdKleVa81+3kZ7Tfv8SQeXZEJHte3P2PpXjZlx8Oca1/Ix3ecq7ffak+P3Z83Z1P+87tT/lVyF6A4u80d6yg5yO2tzSQfGeFw9uUt2EDu/r2cvHoqf1bmrr6k+QGMblhzFyML9CYikglyCQ8Uv7oze1/NDZv3qxUPcw3Zw8qW7yb6Nn3hdKFKeKxM9hSiVI+ZjYLHZ2ATTKRY3K2TM1X4m3q2nklqZQCgZa5PE+HTmBeTTAzOsXfD/M0r4xZ3wltLWA1BBX4c5QbQY28RywaF2Kmv6f64hKJ4FfW9e5h5HaBZNqntWMbW/uP8mTcp6KhtQ2msmV5PsPNW9MS+UTKExdiwxTiBFfLouYnWNveSXV8ljSWsaEHrMpsJQLGZiCVRkXRiAD4KGl0cY3KywImP44LqyQaVrJ+YBCxltzTB9j5J7SudcTa8EGmjexvVzERpg6IPPz0GlbvOA04EEFcQFwaxukJ1rQbyskihclplC5Qae4DoKyZXASIiNW46iPELiA2wOkXIA5xGlyNprQm3d2A2DbS3QOMP77GvdeymMAg4mJc9AqJF+oQZ0BiEIO4KmIruDhAbABKvXU3PqGIwiJxEYnn67ILS7A4QGwJsRXEaZzJAuh/ARK6ubg62xSVUxlcxhNJgsuAtCPWACHi1bvyLFZrkdCZJYA23/7144FPEHajyCz7kgKo4RAng/+M3gCnSKmLCc4yugAAAABJRU5ErkJggg==',
folder_brick=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJxSURBVDjLfdHLaxNRFMfx752ZpOkjfUgiYkykC0G6a8EXuhBxmY1r/wTBpaI7l7oXXQguuvAPEKF/QA3WorSWtmCxldLGR6NpMpnpzGTuOS6iVYv1B4cDF+7ncO41qsq/UqvVvK2trWIQBKPj4+Ov2u32MEA+n2+vra1dGBsba5ZKpR1zGDAzM3OrUqncj6KIMAwpl8uICNvb2wwODpLL5VhfX7/tcEiCILg5OjrK1NQUmUyGpaUllpeXyWQyTE5OMjIyQqPRuO0dBqRpWgrDkMXFRUSEarUKwNzcHPPz8wRBwNDQ0JFDAcdxKJVK7O7uUq/XWV1dRUTodDq4rovneb3+/sWZhdzwxIQxf26jGFPdh4rFImEY0mg0cF0X13UBsNbiGXVOly89zRhjQHuXMYaF589ZWVmhXC6jqogI1lqy2SyqSrPZpNVqffdINUKSvvjzI2zigjOMMXmuXL7I24VVarWXVConKRQK9Pf34/s+X7/Ute2Hz0TkjkfXOCoJNsmiosjeJns7G0RvpjlWr3MMYBtioPCzxsGcuvH6OoCH0ZykewSfvhJ/+YAkIW7fCCeqTzAYpBuDARXBGIM7cJy1h+f2X8uj6+D1H6V4/i4goIqKT9qeRaIN0Bi1IWoDVEJylXt//ZZHV1VthITvUNtCrY9EH0EFlQhkD5UOajuoDUHlABCjKinS3UHTVg+RGDTtTZfedEl91PpgzAEgUTVYNP2Opru9sq3fWOqjtv1zhQiJNwGifUATaabht4FuJ5tH8o5qBiQPWkBtDCSo0+vGsdgoUk0k/g1E8YP1x9fOolzFkOd/UcDMJohO/zr6Acl3eEJ9hHHwAAAAAElFTkSuQmCC',
folder_bug=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALPSURBVDjLhZJvaJV1FMc/v7tnc1e9Cs6Jl9oNFWmtOU2lzTbxX2CWohLthYq+FImolHrhm6SgF77yRSNfFIJRvQgssEgwNkipRpG6FP+1ze6cbvdud+4+9z7/n3N6cVsrmHrgy4FzDh++h3NQVR6nDefXndxwrv2WqtLc3aKbT7fVTfUSPCQ2d3es2PLD+m6AwA67cqX88jcPb93pZm3qviq3Ts0ZVZ0RsOn7FxbGSl7KYdIOAzc99wmKI+Oc2HOMgTN9fNjb9VK4dG55Rgcbv1vX0rPtpzG36Bzonxh2333mIEeW72FjQxutH7yMNs8hms25mtrqUzM6WP91m/qufzgOq955q+VA+v7o79wp5ekr5Ohs3MFHZz+l7oZkfj6bHTI3v117uXZeU5Mx02b6/JI5dL3XOvTsXpJhgX47R+/oPTqXtlNtCV/+9gt/FYaY7ybftowmGhs6TlUbY0ABlAZjqL++hkxqEZeGB7g2VmDvsi1cuneBlU+1Uy441LpmhYj8YRGphwSz/JGPiYMqSMzDmBSvPr2dby734JiIbemV/Jn7lbWZV3jv8xO83rRg5P0jd68CxiI0CZWAOKhBRRE3i5sfZF9pgtf8kcpOQ2OVXDzD1tUZgMVL9md6Bk9nN1kYrZXIpXw/hz/ajwQOVbPm8+T2TzAYuq+cJ/dgHCdwaUwvo/25XdzuaoWINwAswgRWchH1bUcBAVVUbKLiRcQbpGOxj9Y7DE+M8eONCzQvWQXA4BfZq/8AVDX2EKcPjSfR2Ea8O6CCigfiolIiPcejc1WSZDJJ7j8nt/BRlQgJ82g0WYGIDxqB+qg4aFxGIhuNbTDmfz9jEagaYjQqoNGDiuLJaVhko3ERjcuoeIifBfD+BWggE5EzPjss1aSQVEK1GiQFuhCNfSBAE5VsEjGx56kG4k8DPP/4wMndz6O8iCHFo0IBczFA9LOp0t/gFKTUltQw1wAAAABJRU5ErkJggg==',
folder_camera=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJrSURBVDjLpZPPS1RRHMU/982bGUcZLemHkAougmhhIhgqtUqKgggXtWnt0mUg4V9gKNGmwKJFbYMwoxDaRAsXLrQyUlMricSycmbee/e9++79tvBHZNGmLxwOnMXhcDhfJSL8z/m7hcHBwSHgyraxiOzAGMPw8LDq7+8XrTWjo6NK7U4wMDAgvb29LCwsUKlUGBsb4+HNyyRLd7h0rYrm5mbq6+uJoojl5eV3fySoqalhZmaGtrY2amtr6e7u5tmbT0xOnqSvr40gCFhZWSGfzxMEQbM//7hjuqr26FGlPACKxWO0trby6EM9T997QI6TDY20Hz7MxMQEXV1dzM3N0djYiDEm5yvxjjSduJtVSoFA/tUtCoUCTxY9SioPHjz/Ch0NGUqlElproigijmPSNMUnFY1L8vHqTWySIQwN5XIZkRrEA5VRiKdYW1sjDMO/GBjliUuwSQ5xwvr6OqsLk1yrm6ZJ5n/r51wPwH3OnoGR14cQEXyUVLk0Ivi8RrA0xam6iJZMNbnz1/EzWUwSo7UmSRKy2Sz56j0k4xcJwxBjDD7Gwy8cYH/nVfZ3Og5GEVU5w/iDEcTGAGgdYFONtZrs3gu0A1NTUwDjPkZErMaFLxG7gRe+JRHD6RP7EKdRSnDGIdYi1lBo6WHx9g1mZ2fV5hJjRFyKM1+QdAOUj9gKSAoSI+IhNsClZcSWQaldU05EFBZJvyHpj03YDSTd2OIyYkuIDRCncfFHAL1jIIn7nobr1aaSK+KKnkgWXBFk31YHCeJtsvIsVmuRxMW/DHQ8tHSr9zhCD4riP19PAPUiwcm9bUn97zv/BLX9cx0txHrHAAAAAElFTkSuQmCC',
folder_database=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJBSURBVDjLhdKxa5NBGMfx713yvkmbJnaoFiSF4mJTh06Kg4OgiyCCRXCof4YIdXdxFhQVHPo3OFSoUx0FySQttaVKYq2NbdO8ed/L3fM4JG3tYPvAcfBw9+HHPWdUlf/V0tLSqKo+EpEHInJFRIohhDUR+RBCeDM7O7ua55QSkRfVanVufHyckZERrLV0Op2Zra2tmXq9fg+YsmcAdyYnJykUCke9OI6ZmJgghHAZ4KwE3ntPs9mkVCohIjQaDWq1GiEEAM5KoHEcY62lVCrRarUoFotUKpUjIL/y/uqXYmV62ph/LSVrr30P4bEFcM4B0Ov1jk547/uAUTs1ceNdZIwB7V/GGHz6+9LXxY96eDiEgHMOY8xJAK8p4grZz5cElwNbwZgyxYu3EFM01lriOCZJEqIoIooiALIsGwA9Y1UcwcWoKNLdpLu9zvbnBWqNBhuvn5EDUmB0EH/1E2TZw5U+YLQovkun+Ytsaw1xCbnCOap334LC7s4Oe/ttvA+ICLmhMXRxDufczUECS37oAuevPwUEVFFp4/eXkXSdYc2IopSepnjtUh5/wg9gfn6+OQBUNaRIUkfDHhraSLoBKqikIF3yHJDLHaAkFOLciVHnyVAVj/S2Ub/XRyQD9aAZKgkaOohvo6ENgykcA07VEFDfQv1uf4W9Y8y30bCPhg4qKZJtMnjTPqBO/vhkZ7h3EJeRslWNQMqgY2jIAIfa/m5sIKSpqpPsGEiz599e3b+GchtD+bSvjQJm2SG6cNj6C+QmaxAek5tyAAAAAElFTkSuQmCC',
folder_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLpZPPS1RRFMc/970341N7Ehjij/wVQWqrFgmBiyAhiDYtAqNFmyAIWhSEgSC2CeofsF20bVMIblRkAtsPCgmGEpk246+ZcX6/d+89LcZf0UbwbO7lcM7nfM+95ygR4SzmcEY7M8ADWJm+nvSbBgaUOskTqvnVdaOzN4DqP1kKgOKVu0vaA1Di9HUOfYgppUBqySiFruxcSidfr8uBF8CNN4mYaKvwZ/4l8MkDQEsFG9ZVU5OY0AWnCaUC/PZbtA++jSMCWBALCMo51/pjamj8GBApR2yICeOIFWz5F1ExTWErQcxvA4nAhohoQBP0jcYR2330BijxrS4TlkDCIlF+n6DvEX5z/0HDhwqkpsA7D8bxjgGRg1ffwvn+x0eBgsWGG+jsHGJyiM4heh8xBfyLY1AVTgBExFSwpcVasMkjOouNMojJgc6SSiyR+rpMaXOPWMs89R2ux8NDQBURq7HRdq2SySE6e3DPkE4sU1jRXBsZpa73KuXFGb4vzHqzw7FntY8PRRQG0XuI3kWindqpdxGdYX0uyeWb9/FXE6iPD2hY+0xPVzOi5LkHIKHN6NJuQ1SIB9jAEYmBDUAuIKZKtPMFv7UX7rw4nsCJNlyremqASvXd2vt7gwjDKIL/5rXerSslp1Xj1FOq5RQlIL/vYlw21Wm2cWGkYyIeNI51NmvPc36T39b8TLsmqsi4Ou06fxvpelXc23ziGtVtXNkQmLw9o9/8BXTmMmWfiWeXAAAAAElFTkSuQmCC',
folder_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJvSURBVDjLpZM5aFRRGIW/+2bmZYzJTEzMuCbRGIjRiAsYEIKNIoIiWGgluCCChZ3aCyJqK6hFxMLCpbKxERE3DIpFEgtjonGJYSYzMcYsb7nvvvtbTMC9yil/OB8/nHOUiDAXOcxRcwYkAQbube5JZ9asUepXnhBOvR+OzcQWIPzNpQCYad392iQBlDirGzqvp5RSIGUzSmGCsebRnjPDMnsFSLgZkTgqTucfngLuJAEwEmB1RVi4QqwT4GRQqpr00m0s7bjgIgJYvPwjvjy5SaJqXaM/tvB23+X2dBkQKUesJtYuYgXrfyaaGWW6+IhUeglIhJkZxPjLWbzuANmVG5n40Er//a6LZYCStDU+2gPRM0RTk1SvPki6rg1Q+IX7SFKRWb6R8fdvcJWmOrOYqrrG7OwHDsl5OWrajgIWRBAsVo/gDXdhojTZ5p2E+Vu4lYpPPf2Y0Bff89fPAkQkDrBeHxJ/R+IpxEzgjfUiUkW2ZRfByFUc15CqamK+/YhHKWo/0T1Qzi1ExBpsVMLqElaPUhh4RuBXkm3ZQzByBScVoSdXUHrSTW3nIRwT259F0iKKGDHjiPlK8d1LPL+O3Npd6EIXCVcIJ5soPe2mZm2eivoGML8USbT9ZryvldG0Wz3UN+gkKpZhVI7860ssyLkE4w18e/GK2g2rSKRbiYNARNvwJyAILw5d3duBsL3fydXvO36Nt7dOMvD0ORWN67GlflmYMrr4uCAIoJ5prNwoB/jHGs8ea49OHzmexInpeXCXTx8Gp7P1tZ07zvX2/msLfwH2b631NzUvSna0rURS4XO/WDq8+3zf0P/G9AOyUDsBCTaamwAAAABJRU5ErkJggg==',
folder_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJpSURBVDjLpZNNSJRhEMd/z7vr9m7t2oopZl/UoVWpQ0RC0EFL6FAQ4a0OXbrVLaJDEdStjhEUFQVBHaIuSocIomtWJvYFRhb51eqaH7vu7vu8zzvTQbENvDkwDAzMj+E//zGqymrCY5URBxh6vm/Ar21rM6aapwSF7yORm90PBP9NGQAWskc/ujiAUa9ly4EHNcYY0MVhjMFV8jtyA1dGdKkLEEvUqkbhZHHi1XngSRwApxXErgl+3yKyMfBqMSaN33yI5vZrCVQBARVAMV6q6VvPgcv/AKHxVCyRTaCiSPkX4UKO4uRravyNoCGIRdUBjnTLhQQq25Y1wKgvrowtgdoFwsI86ZZT+PWtgKGU72du5CWNbWeJJVKYeAYiL/4PEHrEk41kWk8vraoogtgxwj8vmP7+gVTzEfKDl0jXCf7mixAoVQBVjSpIaRCN5tCogLpZJJxh/vcwaxu7qN24h+Evj0nEF/A1gLDaBwGq4pBwCrFTiM0hNoctjjE7kSO9IUM030PDrtPMTIRIFCwfdhFgVQ0R6v6gbhoN86ibZnZ0jPrsCaj00/fwEalMmULeY378E7gqgFqZcaXpubCYEFdK44IGivkk4u0kXVdCgh+gQlR8y/aOc4y960Ws2GUNtBJcH759vB2lC0MaYwjq6hJbO896UnqDSpm93VnEjuOvGyKz7SBTE/e/Lh5whWfqv9NxZkPrsZtNO5NExT5ULe+ffWVvdysmth6Sh3l379KkinSuCOi7sX9098lHm2LeT9TNUOVkwCOWzJL7PMBQ79Wn8ZU+zAXW77/bXVZRVKQqdbkigqo2/QXSf1oX6zFeUgAAAABJRU5ErkJggg==',
folder_explore=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI5SURBVBgZpcE9SFVhAMfh33vue49X85ih1tUI0cXbF7QkCA5BQVAtbU3VUC3O0dbHWHNQUxE0NQYREUU0BoHUYB9qVJRdLe/V+6HnnPe8/4xu5NIQPo+RxEbYdw/2Txa6du0yJuAvEddmPmeuOgbErGf4pTFy7LVjjTUKSjvGb+eNMSDWCIzBrX4fLk9e+SwQLbmwS8rS+frc0/PAPdZYnFbxSVv87QZZkoOgC2MiCgMHGRi9GiIBHuQBYYLO4vv74xeBe6yxpCaQT8iSEHnhVz6RNsrU55+RL/SDUvAJkgMcUelCiPwgLRajgncrJE1Q0iCtLROVTlHo2QkYQIAHCRDGdkMWWFosaYBt30r3zjOABwnh8ckXXPUJ04u9fFgeZGGlSHtbnp5NdQbcFkOLJZWUreKbr1C2hLIaclV8WmG6UuRjeoDSUCd78jnmlxIqtZjZztN2N78FxEje4dMFfLKAT8r4pIzSBabqBxne1kElNswtZziTY/vWiObmsRwtlkQyZMgtIldFroqyJeSWqK8khGEeFzu8IHaiYHM4Wf6wSnzFNX90pPUwwkeBlAcfgXrpaMuTpBlpBs6LX2Sg2Wjwh9VqfG325vFRxCEMEetEI8P5WvFILmoPiTNhA8PcYop+vNWjSxOnDl95fMdI4l+uP/w41GY5uaUzvOwFy43Yu/KUGe/7ahozz2uzUy/PGUn8j/uXj54t9hev9Q3t637z4mHTSOJ/3Z0onegf3nvLe9duJLERPwFUpzZM2BWatgAAAABJRU5ErkJggg==',
folder_feed=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJFSURBVDjLpZNNSFRRFMd/986dUaeZIS370BRbCKbgpiwIKYhWfSwKgqhNQau2bTOodq6CFkWbqBYtgmAIg7CgxL0iaBYZSYIfYfP9nHffe/e0mElrMW26m3s55/z+58/hHiUi/M8xjRLuprqiE9zFsQ0NNNerdRy0gzCquHlGVEMHd9QKx27tpucEeKtQ+gRhGUwCdAz8Ary/t9rQAY4kmQ5IdUDnIfCPw8IbWBwHBHb0gCXZWCBBiM3B/Guw67BvCAYvQ7IVpp6B50GZ0HweG5puzvT3K6X/oIXOmInpTA/0noXiMnx8AaXvcPA6rC3C2iwUwSjRfV3Dj+NKKZAajFIwlYXiLEzOQO9pOHoD3t2GuSwMXoC3C5AHQyhVnG3yVx4Q2RjoDEqlaQFoaoX2AVicAOdg4BJ8uA99pyC1t+aAQGlxlsgmECfY3Cze8heCeDduOovoMSTYgLkssVQbUi1gXl2ke8958DAGJc0u3MB6ILZCUBbah0dJBD5kusFFEFmIIkikAMW352dgaRwMniHQmJZdbD9wDXAgguDwZq6iKkuIjtVm4wQcJPc/gkIBXj5ZcoYRQyAiURXnzSBRAYlKSJhHtx9B2vpQYb4eLyJRBenoIkglhAnp0oDBR8SFuOAHEhZqxWG+/s4hYbEO1wVcFRTh1i5YEUWEhD/rYL3jpljpb1iqELL5/41Ylwu99WRQTqRxaS0SB5cG2YlEPmARXbuVjoj8soh1/pZA1R/9+vDcYYSTKNL/3F0B1KTFydPfoV+5q0ZcY8+j5QAAAABJRU5ErkJggg==',
folder_find=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKtSURBVDjLfZNNaFRnFIaf785PpqMzE8JEbYgaFZOggoLWaHFRrCIUi1qEdudGaGnpqqWLUi1ZKeKuaF0UKgiCdlNairRuXNiFCyUG/xLzVycmk2Scycyd3Hu/O/f7jovRWoXOgcOBA+c573nhKBGhVXx3+oeO4tPCTN11F7ds2/HFo3tDlyfHRob+vvHXAIDTavj4l9/kEonkiZ4NfW2FqfHE8J1blOaLyXff29+/fdeewY39W5ItAV2re7IPhu8cL83NIcRy3T2957RxIJ7KZtvzJ+v1ar4l4NqvVz7zvaXl6/o3s2rNuljdb6zMZjPkO1eRTi9DB+HnqpUHHx/7tOLEYu2L5WdMT41NzD4tXOzqXvtVrqMj53s+9+/eHouP/vHOUCq7aZNS/xUjaHe8kMnuTh/88BAT/xT4/ZeLD9uTxetv9279+tDRT5idW+D82cFKXInTv3rPzwmlFEhzGKWIgtL66LdTRMYw8mCYfC72wUeH9+6/dndZUjkOk2OPCLVeGyeSABu26eKPmDAGThalMqS63mfJrfFkeobA95kthSrZeSBZrfzJ6MgIpfkige+1q9GrO2sbDl/PBNOXECtYXcZfmCQoV9AzM//rz8D3U6O1aqUUR0nKRj5Ls/PouXFs6BFry9F98CcUCtvQoIAXZsfSXTw+N0C5NN8HEKfhEH9rBZ27vgUsiCDWJardxAaTIBoxHmKWEOuRWjP4mpI4DRExAdYbRkwVMS42mAKxiA3A+oitI6aOGA/EvgHQiNgI21hAomoTYjVI1Nxum9tt5CLGBaXeAIQiCoNEZSRabKapvoJFLmJqL04IsPoJQPAvQEJbibxn6UY9mcFmHJEE2AxIHjEaCBGnWZVjMEEgElr9ChDoMxMXjuxE2Ici0/K3BVA3Q6xcetl6DtPMgEIgxcFhAAAAAElFTkSuQmCC',
folder_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJISURBVDjLpZPLS5RhFIef93NmnMIRSynvgRF5KWhRlmWbbotwU9sWLupfCBeBEYhQm2iVq1oF0TKIILIkMgosxBaBkpFDmpo549y+772dFl5bBIG/5eGch9+5KRFhOwrYpmIAk8+OjScr29uV2soTotzXtLOZLiD6q0oBUDjY89nGAJQErU3dD+NKKZDVYpTChr9a5sdvpWUtClCWqBRxZiE/9+o68CQGgJUQr8ujn/dxugyCSpRKkaw/S33n7QQigAfxgKCCitqpp939mwCjAvEapxOIF3xpBlOYJ78wQjxZB2LAa0QsYEm19iUQv29jBihJeltCF0F0AZNbIdXaS7K6ba3hdQey6iBWBS6IbQJMQGzHHqrarm0kCh6vf2AzLxGX5eboc5ZLBe52dZBsvAGRsAUgIi7EFycQl0VcDrEZvFlGXBZshtCGNNa0cXVkjEdXIjBb1kiEiLd4s4jYLOKy9L1+DGLQ3qKtpW7XAdpqj5MLC/Q8uMi98oYtAC2icIj9jdgMYjNYrznf0YsTj/MOjzCbTXO48RR5XaJ35k2yMBCoGIBov2yLSztNPpHCpwKROKHVOPF8X5rCeIv1BuMMK1GOI02nyZsiH769DVcBYXRneuhSJ8I5FCmAsNomrbPsrWzGeocTz1x2ht0VtXxKj/Jl+v1y0dCg/vVMl4daXKg12mtCq9lf0xGcaLnA2Mw7hidfTGhL5+ygROp/v/HQQLB4tPlMzcjk8EftOTk7KHr1hP4T0NKvFp0vqyl5F18YFLse/wPLHlqRZqo3CAAAAABJRU5ErkJggg==',
folder_heart=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ3SURBVBgZBcHPi5RlHADwz/POOzu7o21bIWVsiiAR1kYdJOwHdAgKO3QQDMqCTv4XdQikg1BCUnTQCIK6REVdOnWTClIRjAwNCdFy1t1xdnZm3nee59vnkyICAOSTJyu5HCsdR6PESju4fXG6vvFBOxzOVd7r7tn7Zju4vbF58dJnk7Y9VQMAaNu3PbTr03rtyR6hGqw/N/3u+6ct9mYrrx5+cXHfXpHLzqVffn3/2pnPl2oAgNJfPN5de7xnNGK8re733Xf0yCFtq2oKl/+UcO/aE1K3+1YNAJBTtdptC+sbRGHzrqruUDLDIaUw3mbnDlFKrwYAyFujmbZhe8w8kzOzGfM5udDMmGyjaPM86is/HrywuHzgQEoVWKh0ujf2WFjewfUb5DklyC2TGbMpqw+7c/6CzeHdH+oU1WOPPH+2m1IiyPtvWf/oQ/es7ra0Z5Wr19gYkVt6XQ486s6li37/+tvfxpPJido8pkrTm936RG46VMv6x9YMf7rhvzNn7T78soUqsT3R7Fzyx6mPjfdv5eF4/NqRweBmrU0VlHkfYevaOcPL5+nhFa6XL+mhB/TfqfWtdJ796spNqJWqo+oY/X1Fc+cfZbbtgYOvW3nqDUlS5q1mMKEU3fsX1Mv7/HX6GQC1OUnHrhfeJTJRkDWDb8hjZEkjTDSDic6OEwCg1kZEZM2/X6AQgUwUES3RijIhjykTAAC1JgqBIAqRRTTEnGhEmZK3lTwSeQsFANRmUaREFBFzoqXMRJkRU1GmooxF3qJMUAGAOtrSy+NN0oNSSuiITiXpgKRCIhJqeTICAHXMZj9fPf3SIYEEAAAACCSUcg7gf+9HV+4TlzZTAAAAAElFTkSuQmCC',
folder_image=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI3SURBVDjLfZM7aFRBGIW/mXt3swmuaMjGBwqCBlQsRPCB2GknCIpgJSKKkNZGxM7CQrSw0Ngp2vsoLBS0srERUSMiKkFRYkLMO7t778x/LO7VxFdOMWc4//xn5j8wThL/wqnz9w6BrgIrhQNBuSzEQMp/odtnjm7t6m00cC5FQG4OK2rEkHHuyuP+/xpIdDV6GvRfvsWGvr5feuIdn96+4sLpfkxikReA9yl969exZecuEp/gkwreeyqJxwzMFjEwCQM6OmrUOpfAH1kJEU2k7x5sf1Fbunmzc/73sg4QDJ6/HmZUQ4UaiwQ+v/nKsb0QzEid/Ma1e25UnFuQtHPEh3eQwCdVfNJRBlBSpYaAEIyUoBaWdbSHB4hZAn4pztUJ0UAwMjT413gjQ4Og/eTRSMmdl2XErIpMWPMT+ew3zm4bofnsETePGDBUZmBYFsl2V5vgOkM0UpxqFppkc6Bslnx6ivrGY1Qbq1ixehnOuWIsqeCkzucnJ5KKz2KMmkzJPWlnL8s2nQQMJISxJP/I+8EHxDCLxRkUZpA1qfYe58uHMS7dvT8uOJySS4otbO4lipMoTqMwQbfGWd49CWGi1KdQnKVzTUqlNmwPrx1tAKS0kSxg+SgKk8XhMFHux1GYKpsLAylAez7QlExyRBS+l43ljb/MpuebrVWMGRYYKLPxMDfWlc9U61jdSxWwOqgHxTaQIV+w85HYakmZtecNWu2LH68f3IHYh6O+2N9AgHuaYbr9U/oB0sFcUlVzMrwAAAAASUVORK5CYII=',
folder_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJiSURBVDjLjZJNiI1RGMd/59x35l5T7x2DGRGJ0CBfpUlRJMlCSRZsUExTsxpr1lIWyupakIUFFnbSNAnJRlJMhhmM7zHkcr/vvO95zzmPxR3urHRPnc7p6fx/z/88/ZWI0Mp6m9ukSQXn0wu698Q/py6vGXx5FQARaWmPX1p3Ov9kSJLKE5keOfLr2bk1+0UE1YqDsVzfxZ4NvUNdm/fp2tdJbMVTejdan3j6oj9oxX6QTvd1bjygXX2MzKJlRO4rS3Zt6bC1Qq4lgDP2U1L4uEOn8ihdoz2bh6hOdTpOqYk7255nsuvXK6XnSIS4MvnFT5cGM+GS80G2Y2G4Mrsi0EWUDvGJYfzmRDQ2Xj4ZKNG9y3dea1NKgTTEKEXx1a1VheKV4e7th5WYEZT7zuvhKkE975S11Tfvy6eO3fhwO8BKhDfp+HsOZ1KgsygVMjP1kKW7jyuTzxEEnUw8snRsGGB130E3eXfft6NnPtwG0CRKizc40454jat9ZubHfbz5gUpNkUp3M/6gjursYXHXKKRMuw5k+b8BoyTj7QymDmJqJJUyYe8JurauBaBtwSBbBpqzUcF8cDpoAhJNMK+H+ev6AQ8iCB5vprDFe4grIbaE2DLiqmSWnYW4mZ2ARERchK+PNh67CmKL+KSAuBLY4my9jLgaSAzJnIwQI+ItPvnZ6ORKiC3O3guzncv/ACIW4rkAI6JwiP09Kyw2bbsSYitNsY8a37RzAGJ8wdZ/dSTV9hAfapE28CHIIsTFgEF041Ta4aJIxPi4CYjiC+8vH+pD2Isi/G+mBVCPDV6u/y39AQ7XjBmT8uenAAAAAElFTkSuQmCC',
folder_lightbulb=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ3SURBVDjLfZLNi01xGMc/zzm/c9/NDMZLudOENGPyLkphw5aFkrKgLFhbWZCV/AeyVBMlpCQbFEWRBRKGMcMMo7yOe+fOueeec+7v91jc8TLT8Cy/PX2+377PI6rKbDN0/XhQ7Gw7XChkDmWydrm6sGDDcWsnK9V6JboZ1jix4sCl9zIbYOTW6Uxbh3++1LVqr18sI0EB8Qzqmrhogvj9fcLh26MTSecuM5t7Pp+cKpU37g0617cE9UEASfFKUFy9GzLzu+3jCzv/AYgOmfYeIAWKYHIIPuAghWb0inx3L/HQvCOzAjwXtUuQxTXGkKCE/RGCKprW8doNLnqOV1qKZOb1mMEbm57m2vr6RLw/CeKmn4vqoCNEr+/i6lnED3CNGl4mxZQNfn4ztvoFI+r1dm09F4gIKIAy+fYO0evLFNftIR29SG7zGrwgRjJdTFy7Qn7lMWovBpj4/PWhoakNXJKNP53FJj54bfjBHMI4QF7cI+hYi+cUsTkkVbKLN1AfjKg87P+47Oi9LYa0ld01C1PuD6i+fDK9lIGZLQ2w4uijMoDBeT6eT+3dIMn4B1xcZ/6mfXSs248gVGuKw8e5Bs4F5AslPvXv+I0yNEHwWbDtJKgFdYAl+XYVbEi9YvFziwkycxmvxLypLmThX1kMqaqqJfnc37qzKtACqaYUsvBmdIxqOMa37yHbdxwkfPY3IFHXql9b7mpRTUCboAlZ02BVdx1na6idpLikzNC0BLE6RKYcm6ApuBh1MWgDdQ3UhaidBBcB3rQ6jaYua8MKyCJEBPBR35t6XRA8QEAFMNioNgMQx3eHz+zcgrb2/ju/dpx78Ev6CYWKMs7gLifFAAAAAElFTkSuQmCC',
folder_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKjSURBVDjLrZLdT1JhHMfd6q6L7voT3NrEuQq6aTircWFQ04u4MetCZ4UXGY2J1UoMdCoWxMZWDWtrrqHgylZ54RbkZtkiJ5aAku8MXXqQl3PgAOfb8xwq5LrOzmfnd34vn+d5tqcMQNm/UPZfBMHXx2ZWvI386odLe7jIL7w5EQ68qjhEOFjCmMj+v4LQmCwtCHkSCuSlFOIst4X1KU1mbUqT/kPki57bmL6xEnx55HxRMCqNCTkO6fUBpH5YkFoeBLsyAiHLEFcSQi5B2C38Z3eAPJ8JjcrmigKnLJ7nd8mwDcnFh4h/68T29FVsfW4F4zeCmb0LZqYDO191hOtkZ5sIuY8lioJhKZ9lo2DmbNjx9WDTowW7+YmsGv+9Ov3GijsgxwsNy7iiYOg4L54/nyawQC4lDubYANIRG7g1I9glHVILl5EMNCCXnEfouXSP4JksI+RY5OIfkWXGwf8cQSb6hAz2gV2+BXaxFangBSS/n0PCfxq5xAxCg3sFj2TpPB8Hvz2G3dWneOvqhLnPCIfDgd5uPebfNyAyrUR/t1bMmft7MdR1NiuXyw8UBDYpJ/AMkhsOPLa2wmKxIBqNIhwOw+Px4EG/Hvb7GoSCc2JucnJS7FEqlb2FizRwNMLHFmPvXnQJN/U6+Px+3LvdApVKiebmZlitVuj1ejFWqc7AZNCJEq1WGxMFAVPFtUCPZKhDXZUyGu6IAr+pklOclGNiYgI+nw9erxculws0N2uqjFOBwWDgSu61RCK50tLSwlBBfX39eE1NDa9QKFBXVydCY5qjNSqgvSWCw+RRqVTzZrOZcTqd2263G3a7HW1tbWhvbxdjmqM12kN7SwTl5eX7qqurq2pra5eampqSGo2GI2TUanUj4RSJ4zRHa7SH9v4C8Nrl+GFh7LoAAAAASUVORK5CYII=',
folder_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJASURBVDjLpZO9T5NRFMZ/9+3bD4pFDIgaJAI68BHDQCQmxc3ERCf+AgaNk5uTMQ6u6kDC4kJMdHJxIzHGARIlaowxOhBNFAOB0oq0b0vb9+PeexxaaScYPMu9Ofc8z7nPc+5VIsL/hMN/hnvQ4fOFm7Gt+LXljszRrG/SJFyXmPaI3t/uvPXwU+1Qgi336vLA8PnsxNkTxBMuBS8kv+PxrXz/FTB9qIRkV392bPA4pdAhVzZoFWPwdC8dmZ7svoTvixc+p7rGxpRq5xKCyo8NP3pEMpnAq2isQKCFlBtDYsmWB0qckYHpJ3GlFEgDjFJof2c4ubiGH0RERqFtY1qiwK/7bVPQ4mNDgs05ar/mqa0/pb7+gpgbp++IYbtQpDOhSKdcutJxcrsRbnWDuetTsw0TI+WIDTFhArGCra8TVfPsFZaY7O5ndfMY79YmcGIJ6kGE8ja4dLJI/tyZ+XszQ4GLkpTVdcIaSFglqpTJjMyS6hmlF8UQAlgQAYSvK9tsfViSwfGLmVy+sOASObgdfXSP3tgvFCw23ESXXiPGQ7SH6DJi9hifvEvu7ar+uOLpiFjaJRIR42NrXxrFpoLoEjYqIsYDXWrmy4ipggQMn0qZK3dephseBIhYjY1+NzoZD9Gl5r7Y7FzeJxDRELQ/5VBEYRC92wSWWtc2HqIrLbD1GzJ1G4GEtqhrf9LRXiKDzTgicbAZkF7EBECIOI1VOQbj+yKhDVoEfvDg5+OZKYTLKDIHfj0B1JsQK8/+pf4CzT1ZpPPtCXkAAAAASUVORK5CYII=',
folder_page=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJCSURBVBgZBcFBi1VlGADg5/3Od+/cYWjUTYlRS43Zi1BGuGlVizZB0EJaFf2JNpHgPt1kBf2EXFlEZFFCUJsIsWmhI07iqOPM3HvPPed7e57ITAAAcO3mw1wOg2Fo4PbOo6NoGfuL4d7du4tv+r29yz9dfXsemQkAAK78cD8/vHDKw4Mm0DKtxqZ2fP3bE7/f2vn2wb2d9yoAAMA4psdH6c7DVEpaDc3+fPDG6XXnzxy3MS1vXf/u4LMCAACQ6IJZZdqFaRdm0+K/J3NnTnDx3DEb07WPCwAAAEQw6ahB7cKsFtt74eb20tN5mtSi3r5+9o/Z5tZWRAFASp8KoSsFiNRastaJErquk6iR5ZWXzn85iQgSkghu3NdACE0XTGsRmVoLESGTasiF1q8tH1wx9h1lU8Rzfrz1souvv6gWShQt6YLSMGW9kpmqVZRsvbGfypYOt3/29O8/XTrO7hcEEoEOHWZoH/xCC1XkrA1z+9t3rPZ2tNXCibPvq1sf2dzoZBZAyqQU/vn8nOVwIFqJalXU9eedvHAJjUypOXrwlf4ZKWQWhBTq5mtgWja1HPpqlZnjQr97DQloDudFP7BcsRpGi34wX/aOv/BYxbuf/Lp7bGOyXi1ltoFAJhptZXNtxXQpxwXtUBv35fDU7NSb/sWNy6+ehKrPDCOZ5Ej2si1pC5lzOR7J8UAO+3J8hgYAavatDkePtGFCFrKTOaGtybZBrmT2RE8ZjIsFAKi5WP61ffWd0xIBAAAASMT3tLwN8D9pITwp1Smo1gAAAABJRU5ErkJggg==',
folder_page_white=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIRSURBVDjLhZM7a5RBFIafme9bv2TjxkS8IN5SWGgKu8QmhYWFbf6DtT8kVoGAKQRBC8HextrGQhDBxoioSJSV7Gbdze53OTOvxV6yaogHXoY5HJ55z5kZJ4mjwjl3BTgPuHEKGBc3ga+SIpKOFLBSlmVhZppWnufa2traBJYAfxxg1cw0GAzU7XbVbrfVbDZlZur3+9re3t4EltwxLaya2WszI8aImVEUBYuLi5OaNE1vef4TSZLgvSdJErIso9Pp0Gq1DiEfXqy8nZlfXnZumiWePbjZHDkhSZKxK5xzmNkhwMlfv7z2uOacG81Y4BxXeXlxqh2SJME5N9lPAJhyYpkVPx4SygT8PM418MnC3zP5BzQEVM7HUGBFDUVRtd/T//6Rxm6TLztzSAFZCUR8fQEUSE6d5tLdpyOA04xCTjVwqDyg6omzaxsspheoz51k+pbGJ39+vj7twJPOnmPhxj0ggoSIFDv36ac1UPhjuLPXHiGJEEIPCCmVpJAT++9Q6KDQRbZPYXWqQY7HhmACMQbK1i69vW/Ksuw20EwpkKIRq5/IOkOI7VOfdZC5oSMZqEKxot6oMZN5k/Rm2EIpOQKyFrL9oUIH4mAE66LwC4UDFHNQDjb5VKQqY9v6e/Wqd6JBbHipBrEBOoNCAZTID1fnA6HoSWUsDgF5sfFpe30VcQdH49h3LcC9Kol6Mk79BmoIbLI/IOsSAAAAAElFTkSuQmCC',
folder_palette=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALISURBVBgZfcFLbFRVHMDh37mPeXQ6M8VaGSvYilRgqmLSKErcSdSVaGIQoytkx0pDQjRx5Up3btSY+IjGFRhWJCQaNYZEQA1YMGWqlXYs6XPK3LnzuI9zzt8aazQG+T4lItzMK8/uHbuzMnhkqNz/mGDMStD+cnZh9YO3Pz8/xTolIvyfNw7tO7T9js2vj+24e3TzsICUWF3qUb9aW7g0t/Tmldnld5SI8F8vPzXhjG0f+WR8pHywujXj3rLN4A7WsEmOYP5FbMZn5soP5uTX3x/0uIFto8PvP/LAxAvjlSYqvkBad6Cvn05q+Tj+jLzOs3/Lk+79u5ZecriB4YHC8zsv/Y4+eQ1ty9jsKK35A9Rr47iOQNLDj0KGSvmqN33qwYu5UrWqlMPfstmCLwnYjEen+AS1qXOtUt+kL5Vd+RNzeyiYhOcGariKoqfE2bn10Y98pRQI64SL3x5m9fE99PuWsLnGYmO6P1hzVd9glR4lHGmDsoB4HloibJKNF9/FJC44JbxNBS78eBbfRty3e4J779nr+K5DLpzhWOMcI7dvIfTz/Lpy/ZRHqhyxCSbJIFawvToT1Tad5QZ6eZFoZgqfv0TA7gLQmuS7a3ed/+Ls9KseSnJW9+gsLBMvzWCTLm62zMjTH6JQzP/yEznTJe/n0CYldgq0vjrKiW9+PnD89OU5j9TBy9/G0MOvARZEEBuiW2ew0VXKuQ6NxSZtkyKqy6axo/zp+OnLc6zzSEXERNjuJGICxITYaBbEIjYi4/SoVBLEtBHTJT9QZpV/eMSIWI1NVxAdICZAbAyiQWLEdhHTweoQMSEoxb95JCIKg+g1RDcR3URMgOgAMQGiQ8S0ENNBbISN66yL2OBJYq/rbqMvbWeK2KIj4oMtgtyKmBhIECcGEpRjMFEkktiYDZ5E8Vu/vffMQwj7UBS5GQHUmQQrn7LhDzEafwIN5+bTAAAAAElFTkSuQmCC',
folder_picture=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJbSURBVBgZfcE9iFxVGMfh33v3zM7uODMr6iaoqJWypBELZfEDCy3TWElQQRAstLKwMJ1iJVqpaBdCyq1tLCVWsVhihBAwyy4SdWMyc+fjzr33nPP+XWEkW7g+j0niv5z/6vsvhr3eG6OF1swMMzbklCyZOU29uBA4wbDfe/Ods1ubvV7fuGeDpVkT+Xpn9+3ACcqKbq/Xt+u3Jlze/5CN4V9UbeTg8EGef+RTtrcexWzFAicwY5iA5E6/f8h65wGcBl/d589Rhcsx0zBwAomJxMa0ahnVNVlzZrHhblUzogEZEpNw47tnd9eGZ86YFdwjOsVbKxJMa2f/1pN49xqzWDEePcFTp7v8K5iKrcdevNAxMxBHBGakn3/EHcaLzNOnP+KZ/Q/45cYB45c/J8WEADMbBpJqvO02f3xDblegGGI2AJ3CEFd/e58YI1falvbhDu2114kxcu6FK0gQiFbIW3K7ilz44oDF7T3OWsWdS3u8xzErwHqA9UCwCFIZMK15WjD//ZD5zZ9QiljosnXuIimJpolg4A5FARbuY7HzCl/u7Jbz2eJiIBaE9VNsbp9nc9v5h/KENPkBb/aAiOIM5TnyirXHP+ZX4JN3X7qfI4EoKdd4dRXlklxdB0WQI68xE8pjlGcoVyDnuECD5AmPt1EqwQLKM1ACNUgFynM8TVGeghnHBVrJyCjdRWmM0hjlEqUS5RKlKcoTlOfIa7w54EjNUlDro1Td6cXZ6gAfFFIHfAB6COUGaFHRAC1WZHJdS603LAXVzWc3v33tOcSrGAP+jwC73OK6xNLfULttUqzYsnAAAAAASUVORK5CYII=',
folder_star=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKFSURBVDjLjZPPb4xhEMc/z/vublvt292m1TZoo4RI4yAhgvhx4OBHSFwkHCQuuEq4SBxcHPwD7EEcHCQSiQMJCbcmRCKkhBQtitKu3W139919fz0zDos2Iphk8iQzmU/mOzOPUVX+ZW/u5IZS6b6J0sTngfUna58W5hz+wzTRxx19hxCrL37Ppf5WOH4v24XlmNd/uCe3bAeLV77IPs23n27rMlfWHKqVAcyfJLy5nYuNo65xc6aj5yBe/zZcLRNHDuUPD/ALD7CNqvjTon+UUPqYnI3iLSY3eIbuoQO4zBKXRnC1SO/qvXQOHMWP1jrFWTlnxm5veNbaOTxszEKWUvn8uJRqGegb2nqKuHAXaUyBxjjd+3l1P09h4tmRPRfC6ymjzpqBrVfTxhjQZjHGkATf+sZubpfK5EOnvWsQt30Y67+lMvWc6peXo3suhNebQ0w0QKKW8OslbOSC04kxHq1LdrKoe7njpD0St5fy6/t0r9hMJvOOlHFXzW8hNo5KhI0yqCjSmKRReEfhyTWSqSLF8Ru/hE2NvQSgv2t12zzAaKskDfwvM4TT40hUx23JsmxfnpnnN8hkV5Jdug5+bMsvF5i+dXzBHcQOqbZeFm86CwioolIlnhvBy1VItYwSzzxCrY9KnY7B80wDd0846d15iVPEqmoDpD6K2jnUVpHgPajgpgVNfFRqqK2htg4qAOzOS9zsIERVEiQuoMlcEyIhaAIaolJHrY8kVdRWwZjfTjlSNVg0KaHJbNPt3DwsqaK28kNCgISTAMEvgEZSTurFRXEt4yGeo5oG8UB7UBsCEeo0X+NYbBCoRhLOA4Lw4sTlgxtRdmHw/v4tATMSIXrtZ+g7IylnbI9OutsAAAAASUVORK5CYII=',
folder_table=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI1SURBVDjLhZNNSFVREMd/53qfPo3nwq8W1SJaVBKtVAxs56KVIdKiaBct24dBtIkg2tcuLCFavCCiNm2ChAwJQlvlR+LC/OhDffrevfecmWnxUlGe9YfhDDPwmzln5jgzY79G36/dNuO6mB5VVcppIPVCEP1rggRDVCdiakjNbgz1FNr4j+48nzlbEyCqbQAv50YIGthISpR9BS+BoIEggZvn7uK9NBwAqF7rSue1A6tvJQEfhNoAUQCeja0cCBjoaiELQvz1dffnfHNnp3PRTjLIvR3/cl8HxfFlhnoP7wH82EiqHTiLTh3re5xzzoEBGP7NEmpGXAfFDyvgoDi+TBw5MCMXw+kjTWReiAmWoFlDuvQQyeogakakHwPiOsdgT3vNDmYWNwhBifEuMs2QrB5TQysLHEpmKU284MzUKNMTcBaY/rRv1ANvMZQYZ3kNFba+r5Auz6JZmQu5eVq7H9DSdRXJMrwo2/sW5VtZHTkPQGPsNMZHxI0dtPcOAwpmTH5bZvHLO7xPEVXSLEMkxdSTa73ICWD4yRRrpeRVjDczSdDyJCbrmJQ42TgPecU0Aa1guonJJiZlGo9fYvYjFG/1OYCYFDMNqF/FwnoVoilYAEsxLWOyhYYSJiVwbs9bxGRmDsHCLyysVU3Wd2GhhMkGJluYJmi6AJDsACzT36H8s8lv1hfQQmSWAy2AtWGSAhkWVU8XCZIkZpmmu4AkvT/3aLAHox9H4Z/fzwA3lqH2dDv0B6mSc8HU1qcrAAAAAElFTkSuQmCC',
folder_user=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJsSURBVBgZfcFLbExhGIDh9z9zztw6marQUm1DNZEQEdUpsRC2lLgsJDZNGmFjY2nBErGzENewIEUQRBBBJFW3IC6xqqYSd4tWtOPMnMv/fdqYUIvxPEZVqWbn6sUzm+tyV3OpTMFaQUET2eSN7YdvdFHhUMWudR1OY23NxVwyXbAiWFUiK8YfLa3ZvXFpDxUOVbTUanZKpmZ5Jl9L67w5NDY1IKKICHWZzAkqXKpI6seoftZsae/a7Ez3ikwoaZox63H/7LGACocq9n3fE06dlh3L4hOaFMbLMiH48FTntjWGVLgD1wt+Oj8/Y4zDX8r5rZ/ibYcWvjxV09/R3NJKyXh4JuTDx2HtHVrU28lvZuBap7ateYQxBpRxih9Ydhx4QuvceRQKAU9vncWYNF4yT1NHN6eP3olvv3q/JX7cc8EMXC5I29o+E349jA0T4OTZ2xtRmrqaKZ2zGCyCtfzhKkxDeXHuXvDm7etNLpFBJcSGSVSUn0MP6M6/gvg4PKSq45uupA4+33PJxaiRuMTo0CDRyCckKlO3YD0/2nfxL2VCzoPRk8v4kpxJ86IVSZfIwc3UM2PVfkBAFRuHJL7f5KtfxqqlZBP4McQ2orlhCQaIn70ojXj9sUukqC0TfjsJKL8JKbFkyTEmSYwYPBWsRLgEWCDR0d40Z2XfiEsAKjEYQBUQkAhHIxrcH9QbHzVF1CmiDJPOtPMOOLOSEca5hIrBgiqoBQ1RCUDKqJZQ66O2iMajqPUBYTJXQyH2h5HYA3VAE6h6IClUakAjVEMwITgxtlxmMlfLweehIxsaUBwMhv9RwNwF0WEqfgE9XTQvEQ+I/gAAAABJRU5ErkJggg==',
folder_wrench=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ2SURBVBgZjcHNb1RlFMDh33u5c52BTmtTipkSS8JXFF2YRumYsGxYKcF/wUV3EBJ2LJpAoomsuiG1bibR6JbEuHNFw8JqTYosCK3Ur9IWRuidzte977znHEsyxq6kz+PMjP2o1Wofm9mnqjqsqvPT09OX2RWxD7Va7aiq3piamnptcnIyEZEKfTH7ICLvj42NHS0UCiwsLKSqGs/Ozn6jqk9j9kFEnIiQpikiUheR8xMTEwcXFxc15iXm5uYiVb1SqVQIIVCtVk+VSiVarRYhhLp7+N27y8XBM2eci/iPkTcf/fV9b/521xcvHYk3k4nTw2xsbFCv1/Hes76+/lhVZ2Jn0Ruvn6sVnHNg7DJwjh8fPDneuJ9ffXP8EEsrsLP8G69mP0hlOEs3t7e+FBn/bGZm5olbuf1eevLDhSG/NYf4AxAN8vPmae6sjfHOiUEKBag3AneW/+aLSxVcNOBXvz23euri4tvsium5yNQjPsHU+OmPMvd+7/JB9AmHlu7xwnFgMoFf53khAd6iL8ZZUUOX9uZT2mtLHGgd5sLANkMXvsbU0ex4SgmMDsaAESVDrN46y79iehFx6Qij1WuMVpWRrjKQtFh5dJdnaYtjw1sMFevkO21MOxTHr7NXTM/MJEM7v2DSQNJVfCGj1y0zUsoYKTWwkGLSwqQDpuwVk2OmAe3VsdCg9IrDQs7JkTZYDhZh0kZDE5MmOMdeMd7MIVh4joUUCykmDSw0MGlgoYnJDiZtTDM0/5NdGX2xed0OnWcHe62kjJYjswJoGewwJjngsSgHPC4SJMvMvOb0xZblN9c+/+gsxhSOMv/HAHfXo/YVff8AB5pkhgXUuFcAAAAASUVORK5CYII=',
font=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHJSURBVDjLY/j//z8DJZiBZgY4tN9wcO6+0erZd2uKc+fNfoeWGxMcW27Msiq+3GWUdIZXL/okI14D7JqvB+csf3Rv4p6X//t3Pf/fvf35/8Ilj3471V3bph9zmougC6xrr8mETbu7q3jl40/FKx5+LVzy8Ltd+eUZBvGnOYjygk3llfKCZY++u3fcWutcd21B07on/61yz88kKgwsCi8qJc++9yhu2p37ppnnQ4C4oWblo/9WOReXEjTANOsCs1PD9VVZ8+9/N0k7m6Yfe5LLOPFMR+Wyh/9dqq5eUvc6xIbXALOs8zEZc+9/C+q+ddEw/rSfXuRxLfP0swuqgAYEt934pOq2nxenAUbJZ0TjJt9+Vbn80X+v5huXrbLOb7LMOLfVterqjcYVj/+Htd38qey4TxqrAQaxpxntSy7PBvnVPO0MSmCZJ5/ZWL7g/v+ozlv/lex2K2EYoB9zigsYPS6lSx7+j+i59UYn6JgtTIGK635hdY/D9dnT7vxP6L/9X9F+b4icxTYmFAMsMs6ti+2/9S9hwu3/Ac3X32oHHOlVdtoroGS/R0vb9/Aip8ILrwLrrv33rbn63zD02F5Zy22GtM8LdDMAACVPr6ZjGHxnAAAAAElFTkSuQmCC',
font_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIMSURBVDjLY/j//z8DJZiBZgY4tN9wcO6+0erZd2uKc+fNfoeWGxMcW27Msiq+3GWUdIZXL/okI14D7JqvB+csf3Rv4p6X//t3Pf/fvf35/8Ilj3471V3bph9zmougC6xrr8mETbu7q3jl40/FKx5+LVzy8Ltd+eUZBvGnOYjygk3llfKCZY++u3fcWutcd21B07on/61yz88kKgwsCi8qJc++9yhu2p37ppnnQ4C4oWblo/9WOReXEjTANOsCs1PD9VVZ8+9/N0k7m6Yfe5LLOPFMR+Wyh/9dqq5eUvc6xIbXALOs8zEZc+9/C+q+ddEw/rSfXuRxLfP0swuqgAYEt934pOq2nxenAUbJZ0TjJt9+Vbn80X+v5huXrbLOb7LMOLfVterqjYp5C/7nTo7/n9bn/z+yxeGTZ4VeDYoBBrGnGe1LLs8G+dU87QxKYIXUNt9oXJ34f8uVqf8vPNv1v3931v/wft3/5tmSPWAF+jGnuIDR41K65OH/iJ5bb3SCjtnCNKu47heOaXP7vfHyhP8br036DwI9e1L/9+9JBxnwHRLqGefWxfbf+pcw4fb/gObrb7UDjvQqO+0VULLfo6Xte3iRT7Xh/21X5vxHBpsuTQUZQFx6t8yR/t69K+l/+64EsOb2nQmoLiCEgQpbgns0/vfuSgHbDKJBfHgYEGlIBxB/AjkbSneAxAHhJKUiPzK8ogAAAABJRU5ErkJggg==',
font_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAInSURBVDjLY/j//z8DJZiBZgY4tN9wcO6+0erZd2uKc+fNfoeWGxMcW27Msiq+3GWUdIZXL/okI14D7JqvB+csf3Rv4p6X//t3Pf/fvf35/8Ilj3471V3bph9zmougC6xrr8mETbu7q3jl40/FKx5+LVzy8Ltd+eUZBvGnOYjygk3llfKCZY++u3fcWutcd21B07on/61yz88kKgwsCi8qJc++9yhu2p37ppnnQ4C4oWblo/9WOReXEjTANOsCs1PD9VVZ8+9/N0k7m6Yfe5LLOPFMR+Wyh/9dqq5eUvc6xIbXALOs8zEZc+9/C+q+ddEw/rSfXuRxLfP0swuqgAYEt934pOq2nxenAUbJZ0TjJt9+Vbn80X+v5huXrbLOb7LMOLfVterqjYVNbf+3BKv+2+PE8m+nK/MjIK5AMcAg9jSjfcnl2SC/mqedQQmstpjsq+dLbP5/39r7/9+1Hf+/rij6fzpb988uZ5ZcsAL9mFNcwOhxKV3y8H9Ez603OkHHbGGaVVz3C693EXz3Daj5/yTf///LBf//b1P8/7rL4f9OF+Z7kFDPOLcutv/Wv4QJt/8HNF9/qx1wpFfZaa+Akv0eLW3fw4v2OLH+/3d23X9k8L5e4j/IO0Sld6B/H3+Zm/z/P1DTjzKG/++A+GEG83+g+BOiDDgcLtVwMkX197Nyxf+vKln/301h/L/Xl+XPDlfmKqJz3ZFw2QqgjfehsfAYpBkkDgCxi6Nciw8o3QAAAABJRU5ErkJggg==',
font_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJOSURBVDjLpZO7T5NhGMXP+30fbalAUVtBbmIRE6RgFdIoSuIlODkYByMx6oI64ECMs4uk8ieIE8YmDu4OEIgaMKBUKSJRwCBgmsod7OX93qsTAwmICWc8yfnlyfM8h2itsRsZ2KWsrczzHd/PmRaaHKbhYVxzyTUhgNte4at0xX4sbJmMRUJ6W4ASen+gfE9zpc95WCkFIYH4EhcxluxJ/dZyI7wtQDI9NL/Cp5wG8UIrUwkY0S/rz9cTtG3kRT3dcQdEqBtF+VmNY9OZnpHx1Ku9btMl0oJ87toc3nKC0w9G/dXHcluXV1lieSb1EkCAcQVoI2fHK4RaY6bb4+hwWsQ3PrL2hCXZa0WlS0qN4/Vt1+52ndT/BBDo5qOlrsuJRT4hbZXQQpVbDlJIANhSmCXeKlzvrNoEIRuPVNcS9QWCnq/FXqcvNpUeq/TfCUgpNNcCXEhStK8CtSVn8eFnH2K/BgUVzPHu4bK2AODErWHiKXCFyw44fT29888GO+vu3Y8w3RS4TaRWkEpCQSO+NoeakjNIsowVnX3LatsNhxW8+dFNTNIQCua1TM3RpdQiiwAAFQxSK8wsTYIrAaE4uORYt/8gWNqIJE9bQ9P91MjOsSI1QU/3wiIDzUiilbpy5GJfPhUUQgoU5JWhMO8QDnr8yDKz4cstxqe5AQz86P+WZsgn25Xp6lO/pIyBKQYqGCq81cYp/yUMz75H70T3KBMIxcPaJv/bxkC7sVBfdsH7ZqI3yhQa4mHNNl1hJ/kfkQWpTG9Gyaz5sBYb/l+aZznkhSoxiAAAAABJRU5ErkJggg==',
group=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKDSURBVDjLjdFNTNJxHAZw69CWHjp16O2AZB3S1ovOObaI8NBYuuZAhqjIQkzJoSIZBmSCpVuK/sE/WimU6N9SDM0R66IHbabie1hrg0MK3Zo5a8vwidgym8w8PKffvp89e35RAKJ2ipp7WDxvjltZ6jwCr5W2bpHHtqUnx+77877jsZxzlO3roAWXuw5ha1pl9MZdAW2ig8RyXyL8rnx8G6uH387AMnUMC2b6l10BJPdAfWDGhZVREuszT7D6hsTStBNDurO+XQEZnEypx1a28XW2F8HFPqwtOBAYJlCde9EeEZCy4sTN4ksrRA4LZB57vZCfMElUyH4E7Ap86r+LwIAGIy03cDr/lDNJGR/zDyBiHGc3i1ODjUIWtqbdIIexVY86kwZ3HijR/86GmqFqJGhPWs8oTkRvAgb+uZGHhVfRV3UNni41OhU8EDlstBSkwjKjhnmqAg3uUtS6y9Dzvg0ljmKkFCaRm4CJT+/5OERtG4yqZMEwdQt1biV0EyW4PVEE1dsiiMk8eMn0/w9Wp+PCNK1CQ6iBYeommkIpH5Qhy5AF/6Mrf4G955tUJlXxtsHieeWQ2LJxvVuAAkoASUcmLugZPqW0qsprEQjDx3sY3ZIMhXt1+DNw77kdmnYKSsKKx+PfoTQtYX9KtzWG2Rod6aujaJwWHk8+uDawGITeA+SPA7nDQOYgwKcAYhQQajyIY9eQEYE5feLPyV4jFC8CELkAkWMDQmoDPGsQaWYgzRjEU8vL8GARAV8T099bUwqBdgzS14D4VaiBA8gZALJ/t6j1Qqu4Hx4sIvChoyDFWZ1RmcyzORJLJsDSzoUyD5Z6FsxKN+iXn/mM5ZLwYJGAX0F/sgCQt3xBAAAAAElFTkSuQmCC',
group_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK5SURBVDjLjZBbSNNxHMVXDz3UQ0892AVSs4Q0shQRA5uzCFGL5Rxqc1u5mZpNnWNN533eaDr976blZeFllc7mMLMl6IOGqdOpS4vAPaRbb6FiQjpPm5QahvmFw5cfP87nezgkAKS9JI4+zp5Wey3Ot57AnMZ9rYnn0RAV6HHoz/+eZl74SYq12d2x0OaGnapL9azeF6CBeYxY6PSHrZeDH8OVsOmCsaA9BYva8/u+AKroo5V2cy8Wh1RYMz/D8nsV5id60F/sZ90XgBoew51pydxYmuyAY7YTKxY97AMEihKu6v4J4JK92Ep26CLBIEPFoqwl033HCGHqT7uOj69dhbAbcjFY+wAXOOd7AgQ+R/4CMIPPUJTsMEd1PBk71SjjQV4nQYUiF/lSAbo+tqCkvwi+eec0F/lnD28BZPRLg0+Sb6Gz4B5m2sRo5dNAMCioTQpDk1kM9bgQVaYMlJsy0f6pAen6NAQlB6i2AAq6Z/uXfu2uwrTZZMjGH6HCJEDxaDpyRlMg+pACtoqFOVXU/wurKI6GYkKEfMN9pKvjwK26ibjSUFAl12B7GrENOHi5RqQQpe0qzIeWBW5dDArb2ei2KGG2GSF7lwK6zBcMoffrTfOB4OeJVL5peeAbUPpSh9xGLQSEBvUjqxAo5hFfcn29a7oaXTMEXCPt40DWl4TAVLdVknt4LY3G614xzDogmQE4I0DCABDTDdC1ADEEROT4ocdSj51jmFK6ACBNSfzXxzrk4L+yg9kLMPUbiNdugKZxIFINRModuPLwNB4b76LMyNo0l71lbSew1oTYOkoyEJs3DK4RYL9xJtADDANwx5WifA6xvCjclnqj0pi4edm1XW8nQEr63JwU1FNEzQ6ktej900dBzptyahpk8SRCsk3wvPHCKs9KLEgQehuchiVX7N+73NXfL+Zkqi9OGtlWAAAAAElFTkSuQmCC',
group_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALNSURBVDjLjZBbSNNRHMdn0x7qQXwoyoqYS3tIJUsR8UHnFEKayXCO8Dp0mje8zLVm3vJSGpk6539azQteZt6aYmuKoQ8q3uY9LSJHqFsUJGpqpNu3v0FmGOaBLwfOOZ/P+fGlAKAclFQ/S960zHplse4M5qtoW5XxVnIfZ6ujv+8PhOO9zzK1NTTDUv1p7E15DL3oUAJ5yAnJUqsjdGo+NgYLoGtxxZLiHGZk9OVDCQg/8wL9hBor/QS2JqqxNkBgcVyFnmwH7aEEbG//iNnaJOPqZDMMc61Yn1FC3ytBVrB7yz8FEQxrXinPY0USxAARytyK4tqNSkQxP/QtAiy03YO+PQ19ZbGw519SOQltj/8lCHG9wCzleRqKAhjYm4rCeJSU5yBfmoaMR0K0valFbk8W7NIvVl0W2BzbFRRyr/Q9ifJFa2YYZutTUSfgQBLERFmkJyonUiEbE+GxJhF5miQ0vZUjQRkHlygnYlcg5dKb3vco9hWmSGGgcOwO8jVCZI8k4O5INMRD0eARoZgnfP5fWH62H6TjYigIFroCLdHNNEUb2xwPYh2ge3r9j+DI1WKxVBy3rzBbTjKKM90wnuyCzZcFMM6qsd6QhOEYe+MA73z1L9jEtSGcLdCs9X4C7je2IK1CAaGkCs+GNyGULqKRabG6QcKQsACRBfmIhi8P3dHpSZ2n0LzLOJz4jvX2OQNyZgH+MBDcC/h3AFwFyfQD3R5mMGpasXctZ5wiz02NlKkcx+3R5hIIXugRogZClEYEKIzgVBnAkgGsEgNUN07imzwMIKHvtyn4SubjLSo6vaiLFG2xm645NxE30wcR0QXwXpETKIGgdiBwZ4q8eVTzfTHEt4FORMNnsRk+hJvgNct0W+1FTaG8q4l0UWWxU5w5tUrHhBEw0qfITIOROgm3FA3o155rS5LDM0di7RvIH7U7Y5P7wg68099P+T6oezGZAe0AAAAASUVORK5CYII=',
group_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKjSURBVDjLjZJbSNNhGMZXF912FZEVpOaVFFSCmhe2/iNqhYVsmo1tDXUiJjNnrC0RMxVFxTm3ecJDeSrblorpSFAJzxptOrfFzJnuUJDplG2ZrqchtaSGePFcffx+vN/zviQApL2STQvgzFaFOCytx7HQFLjVyAuqjwkPOvTnfU+YRz1BmJsDPda2Y9idmrRg8b4E9ewjEqsqDDZ1MlzjZbApo2BtPwldVfDqvgRy2uEyu0YNx4gcW5qn2BiVw/K+F4NPzpn3JYilxnH1LZk/17UKeAwqOHWdsA9JkMe6pPQr4JJDODLOZYeESYb8LrGVGn9mWiJI+2FX8rHc9RiWrmQYW2/D1FUATf0VaGWhLB/MjjpNyDgUj5hBxu40lPNQWZOPF01M2AcEWJvpATasWJ1RYLT06hefoDz+/HBt6i2ochOhb8tGK58OCZNAdQoFr8dY+DqXg83lbu8XSuCaVWF7cRiahuRtn0AaH/zSNNj+X2Hqkkis6LLhcQ3D9fEeVibTYWhOxGQdEyZZzN6FWfoz8W1WCI97DM55NtxLDKzrRZiroyCjPA62uht/BQcvVAilwnRfYYZnDNinBL9hFtyf7sChewS9nACn8KLtATc3d6ExgbEDH4h6nhTLf7cx9Bko7FBCLBNC15flHfst3OZEfF9iYs07iUlGuIylEad2b44USK2m03k9zm6DB/l6QNTSjrleEbR9Ylgm6HCaGVjRPIRWTHEZyyKC/r0V0kx+2Pa0ohL8V3aw1UBHMQGsL8JYS8OAKAAj1dcwUEDGvJQKf8dGMldE2xQF95GQMw7uGyCPexabY1JsTlRgwgu2cUNQnCXYKcyv4ENzSmRvXqwonN7SGZYxBVr0URSyQ9FfdB0NRUmIuFm1XJmVtFOYP8EvH33H/9EvkWcAAAAASUVORK5CYII=',
group_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALcSURBVDjLjZFpMNRxGMfVi17UNL3qRdcLRI2jiUgyHdp6ozIlco3Whm1c49gwmyNndGBtLB3uRdPuCrOtnWoPokGMs5WwMth1S9q/Y+PbzjaJqTFefOb34jffzzPP99EBoLMRUY57KZ3ZBrPDJfsgL9BV5wfp5dpb6W37879hOMhuP2mgWHd5pHQP1vLEX5+xKUEueTdzpNwCCqEPiIZUKHg2GCk7gK5s/ZlNCViOu1KVbULM1rOgbivE3AcWhlsFkCSYDWxK4GB3nSpjh658b+diubscqq4KKKVMxN84y/uvgGprQMminJtletiC5UlS+zqbNjMj/JeUPBqGKuOgrIpGXU4AjvgYCyzDTHasE5BtDpKyKOeXGe62WEteehC4pRFo5nshmUFD5Sc2kiTxMI05VHCUZrh9VZDubF731PcKymO9ICuNQgnNCUwPEp4HXECnmIypXg6E0ptIaQkF53MugisCYe1ryVoVZDrrc3olZf8UVpNzERPdbKh/9KGj8hrS6r1Bb/QDheUJOct+48JGxClo517FwgQfC4o0TPe/hqTKGeF8P7ilu0Hx7NJfwdZjGfRMeuC6wlrZLpjs4WJhKB61KcewNMVFax4JIY9OjYVRY2Pl+a7u2vAWmxfeDrSWOekocO8lD9F5ZWDkJGNAFInF8SIQfRTUJpthfjAC3+QCNGWelDVmnNipvYKuXY6TUxBfVdW9jEQZ4NMEUCRqVBf5YkZT3PzX21B9cflNr4dmFQYGa5iof2AeoxV0JFr8bOY+Bu2VEmQhQK5YwcPCcgzWpmFxNFs7XdXrrlnBXPsScj8QykrUJpiO1cQZG+kMZJxRcJNC4BrTAOobzfRqQJx5GcRkPxanRVgY13QwzlkDD+q5Lgw3sCGONOTo9BTfshbEO9yxcmJXWAR/hG1MB97ePw1J8nGIkywhijNbkcaZqKV3jRYl0YcJTYgQ0Q0IUYQ+8S5c7/0vfNrCXhlwpm0AAAAASUVORK5CYII=',
group_gear=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALKSURBVDjLjZJbTNIBGMWpth7soa2th24PStZD2bronPPByB6aa805iTFFYonGTC3Imk6YeEGnmXJVCg0UhFIMDf8Sm8IcNa9LEdF0y2kqvKU524rwBDZNW3M+nKfvO7+d7+wjASDtpsLk46zxuvCVBd0JfFaH+l7mhjXcig47uDnf1ZybcDJ+tjnUv9hyDNulzCLX7gnQwDwqWWyPxJKFje/91VgyxmJRfwquOvLXPQEUyYerPaMWrLxXwDeqweoHBRY+ErCVXJrdEyAp4XaGW8td/zbWBv9kO9ZcJnjsEhSnXTX+F5BBCWfJWddWJAwKFHfifRxaxLDkSdZPj5GHLx1CeDr5cNTfxwX2OSIq7/yhHQBm7Ol4Oeu6vzaFgu1qrMmFVFmK7m4CZrMZHRNalNmKESE4q77IOxOyBaihXXY85ySivegu3C2F0PGoMDwVwGohoB2qgN1uR29vL8QOIfTGFjwzVCOGE6XYAsho5NYZm35HYVarFdPT07DZbHC73ZiamoLFYoHL5UJFRQVm5Im7FzZi1cHpdKK1tRVCoRCVlZUbZwwODoIv5mPpxc2/gP1XxPmy/OytwjYXvV4vRCIR7hnoyNTTIauTYn5+HgRBoLy8/I95X6whPYk3smr3AqLXRvAb9dDpdIHiujE3NweVSoVDccqQ4K5cLl+emJiAUqkEm80GKTShnkrNNa91TvpR6gbYg0CaHUh9+wOC5hEMDAygp6cHckW9TyaTLXd1daGvrw/Z2dmqAOAIyVka+Wu4TQreGw+YFoBpWkeKfh1UtR9qtXoDEEwTPCdo1Gg0cDgcQQBycnIOkGbFcUttZQ9BF/QjwwqwugMJTACjE3isJMDhcMDPK0FTUxOqqqqCsWkMBsNPp9PfbTzSp+bMGKI4qSCaqjVFPhgCReAMaByUwjHEFYyAfOPVrPRRehGXy0Vqair+ffffyy28O7WhyP8AAAAASUVORK5CYII=',
group_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALcSURBVDjLjZJpSNNxHMb/VhTUi3pjVFoyj+wyPDG1sDUTE7ES1zCPuXRbZjZrSk63qVPzKNuc+ncmHmPOeU3nkYkEmnjhlVdaBPoiddGbMDEoj6clZEYlvnhe/fh8+P4eHgIAsVWE/kdY4wVWC7NqE0yXUZZLeebFvs7mu3+9bwnzvE1pMyrK6lzFYWxOYaSFbFuCYqaxfK7OEfOtbHzty8a81g1zmqOYKLD4vC0B6b8/Wz/SioVuEssjSiz2kJh93YL2FLuZbQn8vG9wJssfrH0ZrcXqVB2WJnTQd8ghCbmo/aeAQ7Vi5bMuLciDqSBDacsRDJtB+cPI73otHx8akqFvFKFLcRdn2adbnGLP7PtDwHSzpOWzPFZlgVRsTomUh9zCVGTmiZD4JBYNb8qR1i6Bjdi6zJZ/fO+GQMqw73oWcQ11SWGYrBBCzadDHkyDguuB0hEhotSuYJfaIX0gGjVvixGti4JLhBO5IchjWNS8b9f8VZgmngrpcBzClY5IbroJhuIE4nq5YJGhmCZ9/18YT30B91TnwFU6gFVsC2E9HdUDOYipuYrL0kMIkDIwX+TzW7DDIUeQJ4jaKCxS5QzdMAntUN46WDkgg+xlDJS9Wbit9oRT+p61riIf9jps5FYZ7scfWuz4CDyq1kJUokGY4b8/4adtPGS2RiKthYOkJhYkz7lQdKYgqOw8rBKJFYLiraDTec1LjVOrSJ0E2P1ASAcQWHQSVf0yqA0LVPZkoaQ7wyC5g4JOCThqL1BExCcTAWFLjKU6rgzW5oJfrwezFWDq1hCoWcP1AnNcyTEFTXoQbo8PIKTUFeSrJNxSecJMROgPC4hT6zuYyXGfr027jwBxHzhtAOuF4QIdENwIBDUDjIxpiPnpsE/dBaaSBrMEYtYAW28s8Z2K69Ii8Yt3ppfrHKMHQBWPGTIOqnAU7vFDsPCqmsmNCU+yFBt9O5awE8YCgrJ5vT8AXgaV02he+4MAAAAASUVORK5CYII=',
group_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK/SURBVDjLjdJZSNMBHAfw1UMPhfTUQ5bgkfWQSodmYplroiAkXkvXcnOVipqsWhLKXEvnUEHUOecU72tec8vEVgQamUfeUzzThcc2QUqd+09b+u0pMxTz4fv04/d5+PIlASAdFG6wJWsk3351ofoMZstszKVs22I/V9tjf+4HPrN9z1I0lTZbizWnsTsFsXbZhwKKmadEi03O0KoiQHRnQit3x6LMCqP5dj8OBUiCT2bqhlRY/SyBeagchk4JFgZb0ZZyWXMoIND3buRY1bPtteFGbI03wTiqhK5dhGSGp3xfIJJsz8pj3V4VhZEhCaeYo0Mc+0QvYn/q5BzMv34FXXMSOqSP4RRxsdUl3uHEPwDT/Rwlj+W1lU0nY3dKstjILRAgQ8yFMtcf4y001CjC4ci7UHaJc/74DpAVcqWjMNofTfyHGKvhoppDhSiMAmmUF0qHuGh5Q8VyDxtmQw/mP9xHRhUNbtEukh1AHGLXMN0m21OYLJEMueoelj6GwbxSiZVRPpa7eJioCMBQmsf/C0tPCUanwg+b3+uwoeVhQ1+IlWEeiDk+pqSef4GjV3MSxAlxewpzoD5HRYkP1mfSQXyLgWmOA0LDBDFFRT/fzUQCQDriXvsokNNvaNcDwno5kkpkiBeVobZtAL3VUVDLQw1rkwwQ034wzdBhnKCin+9kqgi1ppFsfKVUKrvF2Dy+BcEYEPEFYLQDwvoWfCoLBzFXAOPXIBCT3ujLdl0fTHHRqwXX9DKGdRAAEkktcP7V15gLjkIHpgpgKrdBl22jqy4GG9pyrKmvgxjzwYD4Bgrodg9UQZYW7Qwri50haXJuaRtTn4LG60bke4D1FmAogS4FG5tLQhgn76A7xwO9wpvYb62kycoot9bkwERXapXS+UkvyDw1yLwRpKW+RHdRAN4Jvc1FcV4Ns6U0+n7Ab/dSu26WPRQHAAAAAElFTkSuQmCC',
group_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALsSURBVDjLjdBbTJJhGAdw6qKLuuiqi04XZoeLsnXQOecFfWGu8ECQJs08UCADLMQvUVI0UcksQj4NzM1DU6RUEFtj1eZMR02nkOWxNeUiFdfWTM1sKv2Lb+u0WvPivz173vf5bc/DAMD4X/LitgkGTXvmJs3bMVEfsFIn31UTG7prw4/3/w7L2TtYnoYA31TTVvyeu7LA8jUBNSlbqClbMKYfi/C5R4dpazimLDsxZAqcXRNgjNus8w48xtxzI1YG7mHhhRGTLx3oLDrsWRPAY59NG2nM/Dr/qhW+URsWh+zwPqOgST5m/SeQRuwR3BEcn6OSCBhTWSuShKB+Klu27LWSeNdeCO9DNZxV6Tgo2u8IyTqw6Q8gJXw3644gwleeSOD31OrlqLhbjBuVahTcykL7cCNKOjUIyt9Xf4jcu/EnoE844qyWnIbt2kWMNOXBTMaDSmKhShyBuoE8mNzZuO1SoNSViZaxGmTYLyFMEmL8CVQmBLa87bT8dTDLVQJ6dw5uuLJQ1JeB3D4pVL1SCIypmDDG/n2w930WPDXrcausFLW1tchXZ+PmUyVuPlei4Hou3SsuLYIgXwCPKeoXsP6oQUVlpy9XqJJhMBgwMzOD8fFxdHV1Qa3Ng1JLYmRsmO45nU76z+UUziN6eF34fSGPdC1U2p/hilIJ9+AgUiWXEB0dA5FIBIqioFKp6DoqKmqVz+c7/AhJkh8ZAeyq+Hj5o8WHoz7w1eUoLNbSwFn7FxyLOImOjg643W50d3fDarWCyWQucDicjX5Ao9EsMV4XB6/2t1aAbPOCXfAAaekZNHBGXAiCdQKRkZHgcrl0/DVBEMs8Hu+JHxCLxbMMj4E53VqiwLn8HpxvGMfpZAnK9JUwt9hhs9lgMpmgUCiQk5ND1/5ec3PzB51ONxsTEzPKeNMgDnNoeFdD4xvtwfJeEJkOnOIm4oJIAqnsMqTS9K+kMKmtOpevlclk89+zJBQKP7HZbM/3dQ59A+o8mPeL9bcZAAAAAElFTkSuQmCC',
heart=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ/SURBVDjLpZNLiI1xGMZ/3/edOWZojjkiudbMuEVMM4SMBVlbSUpZyAIhG4qlkcJWycZOSslCMTNFuYVpRrkzLuMy5tA4xulcv//t/Vt80ZTLxltvz7N43vd5F+8TeO/5n0r9JNLTs9A7t8FbO0WsfSvWdtdv2VIAKJ45kxWtt4rWh5xSQ6LUyeldXVcAAu890t29zzt3hPp0ljBCyiVMofhMjNkmWldE64t1U5qWTpjXiiuVqDx8RDX35ZxTalfgrl7d6K2+HC5cQBBGYAyk05jhYWrPX350WpcbWpsX17e0QGEMwgiasnzv7eX7oyfHUmLt3mjWTIJqFXJfwAlYS13zHKKV7XN9rInqG6D/AYgkBo0TyXSuId/Xvz0lxiyJMhkYegfGghdwDl68JpycgSiAwTeAgLYJ5scIWgUXx5mUGJPGOYgVKJUs0CZZMpIDaxNnEfAOlAFxYDSilKRE66K3dlpgDcQ1sC4ZtjbB8dxacBZSIYQhTqkwFKWu28FBmD0TKmWo1SCOwagEdZxgrZYYlEowv4X8jVuIUudDp9SJyodP7+NPI9C2FNJRIipXk4FqDVQM1QrUhbB2FYXRMXJXusdE667Ae0/++PFlotTZhmzjiknLlxOO5mDgCQRBcnq1Cm2L8M3zGO3p5fPte0/FmN0d/f13gp+v/Pnw4clOqQOi1P5sR1tj46wZcPceFMuwdjXFbwXen7+gRevTYsyxjoGB/K9PHF/vduxY4ZQ61dQ8d/XUDevBWfJ37jJy/eaQaL2z/f79a+P1wZ/C9Grz5ian1FHRek92zozg68s3l0Trg+19fUO/ib33f+3H69ZtetjZuf9fmuB/4/wDFoO2ZVesLdkAAAAASUVORK5CYII=',
heart_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALGSURBVDjLpZNdSNNRGMZ330V2GXXVRXVTICRd2IVIIFGSoJDWRUssRREJiswP1La16aab3x8tyoWl+T11tqmYmJbZh61pfvWFKZmoqZmp236dv1MxKrrowMP5n/95n+d5z3veIwNk/4PND1dz8z5nY2P0al1d0nJVVdhSebnXxt5cYeGO2ezsmGmtduyLUtnxOTn5+C8CLosl1tnQMONsseJsa2WlvpbF0lLHgtHoPVdQsHfWYLB/M91mtbuTH1YL0+lqxuLi7nyIitomkQOd5jrcQwMwMgQDDhgdZqW9jbn8/I8zen3/ktjHYYdHD0GISDEz+kzeyuVK2arZbHU/fwovn0FTI5jNUFMj1r24ertxdgpSbw/cugU3b0JREZSZcD59zHBo6Lhsubr6k3tkEKzNUCecagW5shLu3vUIPmgCo1GgBAoKBPIg24DrSRdvgoIWZKJYX9yD/VAvyBUVUH4PTCaPY8k6KU+QcnIEUQ8ZGaBR4+psp//YsTnZosk06nK8gmrhWnrbk+YGMTcXDAbQ6SA9HVQquJYG1xW4ujqw+/svyBZu3Cherr4PPV2e9La6abXCUQNKJaSmQnISXL4kjljGpEpBn69vsexrXt6emays90uSiFClpNDjJEFxTRBT1ohWVSSXc09zIesk51RH0YYd+v7Cx2fXWh9MqdUHJ1NTe+ezM3FJV1UjCphwFRITIP4KDSlnSas8R6Mjn74JG/qWaE7pD3A4ZqdusxMn4uO3j128qPgYHT0/byyGZnGdyUIkLpZwTQD1rw3UD4ijiaFrPY++NVISWPqtt9+Fhx8aOXPm8VSSILfboNXCiURvLA4jW4fZni8J/PmBDIWEeA0EBuY6AgLc4xFyjsTsdmpt4aht8jWy2ir/ewZbYffzCxaVjhOBymDdfjJtEWvO0iytf6nBvyCCNQLzUtrrs0b6/xNhTevE6BlD4wAAAABJRU5ErkJggg==',
heart_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALJSURBVDjLpVNbS1RRFP72nHGaUedmpamj5pgKSUli0eXJfOipoELs+hA+FFT44oMSRBkZ1GNC/oAoCJ+C8QaSZhdCTEU0rbyP05jTzOTczv20zkz6IBFBBxZrs9f+vv3t76zFNE3D/3zGjYXa01OuKUqtJstZqizPUnSb6+vDem29o8OpiuIFimZFEOZUQXiY09rq0WtMV6B2d98k8F2YTU4YOKjRCKTw+qQqSZcJFKPoTMty7Nu2pwRKJILY2DjiPv9TIrvGlK6uk5osvjSUl4ERGJIEmEyQlpeRmJpeUkQxaikp3mt2u4FwEPoFcDgR6u1FaHzivpGk3uDy88DiccDnBxQVkGWkFReAO3SgUONFcGYLMDxC71RTF1jTYTt2FIEPw1eMJLOCs9mAuXkqyoBGhxQF+PQFBjvtcwyY+aq7BIhyKgeCYCUqFJ636QSmJIAXAEFIEYhSimTFl1STvFkPjfYEqqmUJRFkpmokg9bJ+Z1MpgKfIICSAutAZcs6mSmMBvLCADLRYCCWfnlmBnDlAbEokCASnqcbhFQW+VRO7lPQX0CpG4GB17qCZyzc3u4mFf2WItdu864cYHIKWAv+bg5lU4V3bYxiFLHQd3CZNqT9zIynx7PLuAceTyg2OPiK9/kPqqEfeWmV+8HSqb8WvSkwue5dHUXIHkbFmQaUXmxETrYVIX6FC0YmFtlGK39rabHTm5pIVqOzqtJqzc8F3r2nNoxiaLkT1Q1NsMwOkLFvALsDAWMRPvYNzbOtszDf0FBNRI8dxYWHd9TWJE3rbzuF449egFWd3jwXvpOLkaGAxv40TJ/r6hxEco+8ue4syGVTC88TR85fsmQseSAk/KCWQ2Sdw/QsVtjfpnGipuYs9Ykr4pp3mqwZtwq2y0ajwYvImoyFVU6ReO02+9dxfnuusDkW9F3lFFakcNoKoZ6c6JPbfgHHGqU/+iLy1wAAAABJRU5ErkJggg==',
help=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKkSURBVDjLpZPdT5JhGMb9W+BPaK3matVqndXWOOigA6fmJ9DUcrUMlrN0mNMsKTUznQpq6pyKAm8CIogmypcg8GIiX8rHRHjhVbPt6o01nMvZWge/k3vP9duuZ/edAyDnf/hjoCMP2Vr3gUDj3CdV6zT1xZ6iFDaKnLEkBFOmPfaZArWT5sw60iFP+BAbOzTcQSqDZzsNRyCNkcVoaGghzDlVQKylOHJrMrUZ2Yf52y6kc36IxpyoH1lHF7EBgyMKV4jCJ5U/1UVscU4IZOYEa3I1HtwI01hwxlDLhDoJD/wxGr5YGmOLAdRIrVCuhmD3JdA6SQabx12srGB0KSpc86ew4olDOGjH4x4z0gdHDD9+c4TaQQtq+k2Yt0egXYugTmoVZgV9cyHSxXTtJjZR3WNCVfcK/NE0ppYDUNu2QTMCtS0IbrsOrVMOWL27eNJtJLOCDoWXdgeTEEosqPxoBK/TwDzWY9rowy51gJ1dGr2zLpS2aVH5QQ+Hbw88sZ7OClrGXbQrkMTTAQu4HXqUv9eh7J0OSfo7tiIU+GItilpUuM/AF2tg98eR36Q+FryQ2kjbVhximQu8dgPKxPMoeTuH4tfqDIWvCBQ2KlDQKEe9dBlGTwR36+THFZg+QoUxAL0jgsoOQzYYS+wjskcjTzSToVAkA7Hqg4Spc6tm4vgT+eIFVvmb+eCSMwLlih/cNg0KmpRoGzdl+BXOb5jAsMYNjSWAm9VjwesPR1knFilPNMu510CkdPZtqK1BvJQsoaRZjqLGaTzv1UNp9EJl9uNqxefU5QdDnFNX+Y5Qxrn9bDLUR6zjqzsMizeWYdG5gy6ZDbk8aehiuYRz5jHdeDTKvlY1IrhSMUxe4g9SuVwpdaFsgDxf2i84V9zH/us1/is/AdevBaK9Tb3EAAAAAElFTkSuQmCC',
hourglass=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ6SURBVDjLjZO7T1NhGMY7Mji6uJgYt8bElTjof6CDg4sMSqIxJsRGB5F4TwQSIg1QKC0KWmkZEEsKtEcSxF5ohV5pKSicXqX3aqGn957z+PUEGopiGJ583/A+v3znvPkJAAjWR0VNJG0kGhKahCFhXcN3YBFfx8Kry6ym4xIzce88/fbWGY2k5WRb77UTTbWuYA9gDGg7EVmSIOF4g5T7HZKuMcSW5djWDyL0uRf0dCc8inYYxTcw9fAiCMBYB3gVj1z7gLhNTjKCqHkYP79KENC9Bq3uxrrqORzy+9D3tPAAccspVx1gWg0KbaZFbGllWFM+xrKkFQudV0CeDfJsjN4+C2nracjunoPq5VXIBrowMK4V1gG1LGyWdbZwCalsBYUyh2KFQzpXxVqkAGswD3+qBDpZwow9iYE5v26/VwfUQnnznyhvjguQYabIIpKpYD1ahI8UTT92MUSFuP5Z/9TBTgOgFrVjp3nakaG/0VmEfpX58pwzjUEquNk362s+PP8XYD/KpYTBHmRg9Wch0QX1R80dCZhYipudYQY2Auib8RmODVCa4hfUK4ngaiiLNFNFdKeCWWscXZMbWy9Unv9/gsIQU09a4pwvUeA3Uapy2C2wCKXL0DqTePLexbWPOv79E8f0UWrencZ2poxciUWZlKssB4bcHeE83NsFuMgpo2iIpMuNa1TNu4XjhggWvb+R2K3wZdLlAZl8Fd9jRb5sD+Xx0RJBx5gdom6VsMEFDyWF0WyCeSOFcDKPnRxZYTQL5Rc/nn1w4oFsBaIhC3r6FRh5erPRhYMyHdeFw4C6zkRhmijM7CnMu0AUZonCDCnRJBqSus5/ABD6Ba5CkQS8AAAAAElFTkSuQmCC',
hourglass_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALASURBVDjLjZNJTFNRGIW7ZOFON2xMiBtDHOJKgom6dYgsWKCJTRgUIkSCYYOiiURiq1ZKsEUmAYMUEgm0MrWFIHagiEALhYJCKW2lpZOU9nUejve9CIJTvMn3huSe7/15N4e11HYnhVBK6CcYCRQhMdd4C5O11zH2+Eqiv/Ic1V120thamNYvZB8q5eccSAHAomGRzar1YQ5sE0I4ta/g1r+Ga64dm1PN2FDUwyLnwyjhYKHjLlS1eeitOAsiUO0KDB335nYEjplmQgvsmkZ8/SDEurQGRvETLHVVQdtcDgWPzQhq2alzuwL1vDl9Rj2O1eEmLIoeYEpYgDFOFsjYIGOjregIGgoOo6nkGLoeZaNJwIWgczh9V0Bfxlai0hlrBG5/DKFoEuFYEp5AHIu2EKbNQZjcERhdEQzMuiAYMkl3wrsCGpkh2CczBJLrZDMVTsDmjWHJHsYaCaq/+PBSZkm+GDT17g3vE9CItdsZEq3X+NHoh+VblAkP6Tyol5lX6gbXMn4N/ybYQTThVM6aKUyb/BBKzYo/7fmnoHvCodFZKcwQQd3AmvK/BSK1I1P8yWmet/jhoeKwb8cwOO0At2d5tbyhrLFUcNVXWJOFa9XnfRcqTtzfJ+hQbop7Jh3JNWeIOYlIPAlfKAGLJwp+Xyuq3uZhYEEInU0O/kgxcvjHcboklceE2xV22Yjegw1vFIFIAlESjieSoMiz1hpE3tOLkOhrITHUgV680ZvgjxbRghCra0Sf3qm0YdywBacvxoRJlhF4g3F83gzjcuUpDC20YO96Ny+kBWC6sCBrgEqjhmbZDasriO0AOUK7H6L3Jjx8o8OZ22l4Js8HR57LhDmy3J8T7C3T37pQzb2EbN5RPJffYL5M3+l35h/srTOpsJFUmPpRYaYLpMIJUmGKnX9wK7M4NUaPTfARuPT/+w5sF/jkpVJK3QAAAABJRU5ErkJggg==',
hourglass_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALPSURBVDjLjZNbSFNxHMcXm+BDBEFBvgRS9CBFrxJRoUlPs4ceJMiHsIshSTdSi7AI1K6KOfOSaZgXQmyic5u3TJ3zNue8TEvP5tl0czed7uy+nW//DV3ajQ58D+fh9/nwP/8fX85s9e1okkySVhKKhCEJqspvYKjoEnrykoOtD08zjbeOU++vxbYKUvdlFqbsjgbACYVDhgcWxfkwDApgVlbBOvUBFlUNVkYqsdxXCl1HIaiWfEzX5mCg6DKas0+BCAYiAnXtA9WWwKSoJHkHo7wcS18FWJS8BiV8htmGJ1BW3kXfy9SwoCg1RhURyCbpOIWsFwviCszUP8KIIA09+edBjg1ybFRfP4SytIOoyDiKhqcXUFFSgJI6cVxEEHr1zPskCr0XVocfbh8Lj5+FzRnAjMGNMdoFrdULyuJF27gFJe1ayRYcEYQiVbs+S9VOdpEMM54gDHY/Zo0eaAgo+76Bt1Id+0akbd4O7xCEIlSux7co7dQw5YBu1ReG2ydsKJXS88UiTfyv8G+CrdQPmvvHaQZjWgcEErrvTzP/FDQOmuQTegYKIihu0/T/t6BeZjohHDXTkzoHbEwAxnU/RGMmFDTNLYjTE0p7+XvorgQeK03i6kiydwhq+1eETUMmVmN2hzfhDbDYcAehs/nQXfYCynsn4Ra9AquWwNl4B6MZxwIdibybYbimzyjtnLJh2e6D0xuEj8CBIAuGfCv1LnRdPAwXgVHMB7L2AnmxsDw/A+lZrobT0DkVV9dvQK96DeYNfxgmbFhgdwXwbcWDrsQosIrQBn8+a7kHEPqdcBempWUYkMsgn7NCb3Fh3UlWaHSg/osWjz9OoCU5BkxVGkAgz30OVknodC7IPSztKNPfuqDI5WP46hEYsmJhzokCdWUXuvm8gCSJ+4Czvc6kwhSpMLNZ4XAXSIWDpMLMp5T9NtE5nnNzC0shOHR/PwBGKPcL7gZY5gAAAABJRU5ErkJggg==',
hourglass_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL0SURBVDjLhZNbSNNhGMb/EqGGUEEUUhgWUkmWQpFZ1oUkdZNQhFAeMkktQY288EgeKk+l05zHdI41tZNO5mHTEJ3OmTrPh8imTvM0tTxs6qbb07c/OTSjLp7v4uV9frzw8aMGWI9MSAJI+CQyEiWJtiv7AZoZbqh9fl3LD7+sLA46Lcv3seQz3fcFpLiamQCg9KHIcuNIVRwmmphQdORhtoeNma4CTLXkYlyUgdHqFMjK4tDLCUUjwwslIZdAAI0GQD8nrGsDMC3NJXmNSUk2vtczMSJIhoyXgIGiaHTkPobohTsNYLibdxkA4m65tVRch29VOegrjEQL0xu1cS4gZ4OcDZbvUWR5WyDH/ySKYm8iJz0e6dwqawNA/9QOagTSMTVml9awotFhdU2HOdU6+iZW0CZfxvCsGrIZNcrbZ5BeOSzYKBsA+gj7l0uF/SrdCFlWrmoxMb+GgclVDJGi+OsiMoWjulcVwyWby1sA+vA6FuzLOuZln2VLGP2hocuVnXPIEMoH0yqG7P8sbwNspLBJ0dAuV6JteAlMgVykn/lzzqnus+zgmWdj+l9AcdO0pHNMCSkBpJUPNehn3iw7VTT/NlyzjsOFedj0r4BC8bQDr1Uh7x5dQiDXEQFvzsOPfQZeebaIKL2Fd62pCH7vgispB7SOSXvNtgA4DVO8D83TuiHFCv0T5GTw2jPwUZpOF4tbGWB8CgZbkgg/rjPOxhlrbWKNdtPlAtGksKZnDuPzGqjUWmjWdSAn0+Xk6kAkCPzxrNIHUXwvxFT4IksUC7eCi7B6Qq1TRTU91tyGCdT1/4RicY0ua3XAndwTeNvCALf5JdhNicgXxxPIQ2SKYuDDvQrLSEpxMJSypV3oFWahUSKG5MssxmaWsaDS4EbmEVxLPQSnlP24kLQHHiwHZNRH4R7HGcfCjNTmoZT1Npn+5YLd053wZDvhVLgxQjx2STfbaNCZKCwjCit/K0y7QBTWEoWVVhGUziJ8B4LumkZv1vkXO/PSRSy+XJ4AAAAASUVORK5CYII=',
hourglass_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL5SURBVDjLjZNbSJNhGMd3KdRlN90E1Z0URVfShYKKic7ChFRKyHRNp3lAgq1ppeam5nFuno+JJ8ylzNMSRec3LdFNm7rKPuemznm2nMfl/n3fBw6tiC5+Lw8Pz//H8148LH1VvBNFDIWCgqSwUhxNlETiQ94D9IluHymEbtbGuGtk5eOLClnIuZjcwLNOAFg0LGqYmOsSwzwkw4q2Amu6GqxOVMMyUoZFVSFM73NBtokxWSsAkRcKOd8VlIBwCKZrn00cC5bHyijKsTRcgoUBGea6c0C2ZkDfkAxtWQJUWSGMIC/k/IRDoP5kdB5T9+NbVymm6pMwIgtDn/gOqLVBrY0q7mUUh11AadQVNKQGoFSaDmldl7NDQD99M4fdY/MHWNu2Ye/Qjn2bHes7PzFl3sOocReGtQOQqwdo16xC2mnoPg47BDTK6d13yukd+xw1bN0/gnnLBv3SPmapoPrrDxQpTfaCDoP8ZPiUgKZV+92lTbtFfiS3Ydo4ZMKd4+soVBpnJB2zLr+H/xAcUz+0MqgxWjFq2Ias26j628w/BY1Dy8Pj81aMUQJJ++zgfwvq1cs3mwmT6U1zO7KyslFZWYnUtAwkl/ctCKUK38TERJLupaWlbfB4vKeurq5nHOHaQUtrE7Foz5WWIj8/HxaLBSRJYmBgAOmvc5H4Kg/6z1+O6B5BEMwMm83OZMLVqiVlj24d8s5eCIVCaHQ6iMXp8PPzA4fDgUQigUAgYGpfNtseFBTUSUsSEhK2WA09Oue6QTP6pzchysyBSCRiBDu7e7jl7Y3e3l5oNBqoVCq0tLTA3dMLvCTZDVqQkpKyx9zCpLIYxLAa6ZIKxMbGMQK+8Dk8PDzh5eUFf39/Brr2cHfHwwD3TVrA5XI3Tx3TiCIDnNBgFOTnQP62CXK5HEVFRYiPjwefz2dqutdUV2PLzs7epL6oZ508Z21xBNny8t5u8F1fcDmP8CQqEtEUSfev7r8IvGSO5kXYoqJ4h+Hh4VYfHx+Dm5vb9V9HN9N1j9T0nAAAAABJRU5ErkJggg==',
house=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK4SURBVDjLjZPrT1JhHMfPq/NH+K6ty2bhJcswzUa2hTMaEmCsZmWuUU0HQuAVEWHMgCnLy2yOhiOKIs0L08ByXgab1TTRNlO7ULwylTOZ9iL9djiVrdLmi++bZ7/P5znP93kOAYDYKt1F+0k6cR4ZK86jSCS3m9sW7pGxwh5FwlqfOmnNW34w7NUcInck6Ck+QNJgZNjExYTzOl67iuG/nQuf7kjEp2eT/xV45AlknyopMmLJweRDGR05Jt1KBDvLMdoiRp8uLeKpTiO3FHiUiWR/WTI12sBD8JEC/kYBvLXpeGrIwHCTGOPuKgxYRXCXsan7ilTyD0G/Opn0lqdQfisfwccq+JuEGKjjYHpqkklvLQc+iwiBe2p06/mwSVOo5kvJjISgCyLpgij/bQGCHWX0p4rgNZ7AyxdjWFxcxPLyMuZmZ+BUHUOHlodnd26g/eYpmIQsSn86niR81akBf9PZn+fMo+EsTIy/wtLSEgP/yuzbGbQUsnFXdhJd5gtoLMxAOWdvgKCb5Xr1aevDzXno0WZufHg3t7nz3/n08T1qclhouHgUZZl71ulwfxRYncZ9omGPWOstG6urq8ywUqmESqWCWq1GaWkpsxaNRmEx3YKaEz8Vg/+5RpPJtEJRFDMcA1tbW9HW1obKyspNQUwqkUj2bfkODAbDSiQSYYYrKipgs9lgt9tRU1OzKZDL5RAKhb8FRc8vJxR0nTsvtvMGxBb+N8dQO2ISjUYDh8MBp9MJWsysPXhjR0GnBGIbbzrbytGmaw/zCRr+LOu9iqrBEhT1FqDAmo9wOAydTgeXywW32426ujqEQiFoBlSoH9NDO6REvkOERFl8lKB3HqRtIdoWOC5Lp3jXchakUum80WhkQLoXmM1mCASC+dySMwvZtVlf0zWpYzT8ZfeVXYPEdr/pTvMdjX2sh+52/VQAAAAASUVORK5CYII=',
house_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALvSURBVDjLjZNvLNRxHMd/Tft50AOPqs1aTapJzp/8yYWaLM5k+a/F1F3FxXAidnIc53adc7j5uxo6OqX8PxcdZY7uWhJRD1pqrXWPCveb8ATvfq4yitaD95Pvvq/X5/t9f/clABCbpSv5CEnHWsNjWGvSncit9m0Jq3kMoybdcbEny3lRm33UqM11I/9LoE5xIGnQpJOxMN6UiNfNKdCXh6Kv4Jipr9CT/KdAk+ZI9mQ6m4bkQZh4wKOThomWDEy2Z2O4Ogo9BUyTJo9JbirQZDiRvXwXargsGJMP06GvCINW5IXHYm/oKqMw1iJAvyISLXxP6l66B7lB0JvlQmqzXSm9IgSTrZnQV0agX+qLt28mzOkW+aJPHgmDKgtdhSGo47pSVRddzBKCLoikC6L05WGYbOPTR42EVnIKL0deYHp6GrOzs5h6/w5NmT5oEwbjya0kNFwPgCyCQRWesSeJvjwPg74y/Nc9o2nYD+Njo5iZmTHDv5Oq8sGVehfUXvNDZ3EsKi57I9v3kIGgm2VpC5nLuqpoqIUnVj59nFqbvD7cBk/kq88jusYOJWwm+CcOLtNh/Swwj8nqyPUcUpTKVxYWFtYmJjceQ4LSDexaZ+S0R+LBiAIZD8/idMlu8AL3h/71jDKZbI6iKLMgiYY7XlWhdbTCDN4fKUNZfwaUhiJwVf5wl1guM0TbrDYIxGLxnMlkMgsu0fddhUu0qZD2JkH8KB5CNRsFmgTU6ESIveONg3nEEpH8lO3I6TwXE6UM7o+ShyzdHWzAqiTm9mE0vyiD6rkcSn0R6p7dpCWJqNYVIF7Fgm2uxTxDsC+NoOEvvO54CAauIbmbA44iDkajEaHVNghU7IFf6S54yawQV38cVYNCcBr9YSfagfDaADjx7L8T9OSBQIXvZy+hu+Ekz4sKvhr0lcvlfpBIJJBKpaB7QXFxMRzyt69cUPrBNsdyxV3gMEHD3w5cshkgtvqmf8ZGQMzvvWGBnXzCZv36D8sKlHMs9WAJAAAAAElFTkSuQmCC',
house_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL2SURBVDjLfVHrS9NhGB0Evy/9HQlZmmbe0iabspbbwJm6IKnMLWND2dzcnKDb3CJyjpy3QR+MgoxMV06HNl3iHBO6qDkjvGVlgn5Qt1yOCD393l8X8EIvHHg4z3nO+7znZQFg7Yer/ATVp4xbcavjowO6hKin5vSKp+4MdZj2ANFXcZKiB0M+Kx9TnXK8e1KBQIsYQ/WpoSFLGvVfA7cqnhrQJoTGbAJMdylpqDDdrUHwWQ38jiIM1KeH3IZ06lADt+YUNahPDPubRAg+VSPQmg+POQMvbmXC11aEye5aDNsL0a1PCz9Wp1B7DAZ1iZSnJikcsOch2KNFoK0Aw3c4eD8zzaDfzMGQrRDjj3RwWfLQcTMp3H4tkTFh0QFRdEDhQEs+gk49vWohPLe5ePP6FdbX17G5uYmF+Vl0as/BaRTBe0+Bh1XnYS2IC1uEsRRryJAyHmi7+OedEno4G1OTb7GxscEM/8X83Cwc0jTcV+agt7EYrdJM1HBixll0snyPJX3H1y5Bn5G9++njwr+b92P58xJMgjg0XT0LPfvYDg3+7wAN6fzndWlj9ru23e3tbUas0Wig1Wqh0+lQXV3NcJFIBDZrA3Sc2BkyfOAbrVbr1traGlwuFywWCzo6OmA0GqHX67G8vIzGxkaGo3thuVxexWazj+4xMJvNWw6HA3a7Haurq1hcXMTo6ChMJhMIPzc3x3B+v5/RCIXCBlb5y5L4672XLhc9EA0LlLwdbbUWE8EgSktLiQAymQzNzc3MFqQWCAWQyAu3iEmluvI7ix7+quy/gdqRSogq+DDVmxgDsnpOTg68Xi8mJibg8/nQ09ODbB4XdV41s4XBZACLvnnkgp3zJcOYPJ5akByVlcl2iUFJSQm4XC54PB7EYjEDUnO4HIiKhSFiIC2T/tyTwXH6iESiDzabbbOrq2vd6XQyb1epVMxPkJpwpEc0RLvHICYm5khWVlZCbm7uklQqjSgUiiiNHxKJ5AqNbLr+RjjSIxqi/QVm6pQnUTAFywAAAABJRU5ErkJggg==',
html=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHUSURBVDjLxZM7a1RhEIafc3J2z6qJkIuCKChItBNSBQ0iIlZiK4gWItj6HwRbC7FRf4CVnSCIkH9gJVjYiCDximCyZ7/zfXOz2A0I2qVwmmFg3rm870wVEezFavZoey7Q3Hv+/Z87qDsiTlZFBJIGKStZlFSCTpyUlAZgfXXfH9BAPTCberVANBB3RAJRR8wp6jzd/DotALA9UcyZgZxis2QNijpZjSJBVqeIszTfkMY65cAjuHxmgSzGlbUFrp1d5ObGErcuLLNxep5hU3H93AqjYcXti4cZZ2OSDU9CnVURddqmIovTDmoev/5GVcGDF585tjzg1JGWo0tDDgxrThxq6XojieOd0nRZ6dVpBxU3zi/T1BVdViKCcTbcYX11ngB6cca9MSlGlprojHqcglycVJyHL79Q1Jn0TgBdb1gEbz9OeL81IYsRAakYvQSeC/WvVOiLE8GsM4xnvsuGe/Do1RY/dpRenIP753hyZxURJ3JQXbr/Lq6uLfLpZ6aIk9XJssv8VK5dNcQcmcl7fKVl89kHmu0dJRVjYTRHGVSMpELaQLVCtEY8EAvMHHUwn067+0LVybtvok9KSODZiaKEOJENihPm01gD3P+62Oq/f+Nv2d9y2D8jLUEAAAAASUVORK5CYII=',
html_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJMSURBVBgZpcFLiI1hGMDx//ed78yZ4RzTzGHcRpEZUpqajbsmaYpcdpRYYGEhbGwUG2VhMwslScnCwoZSChE2khW5LliImDFk3M4333nf93mfB6VsFDW/X2JmTETKBKVMUHb04kfjL0SVEBQnQghQiFE4wQWh8EYelKIQMn5a2tvGH4aoEaMhakg0ghhBlRCMIEqIihflzO1RMn77Ni5EBVFDouKjEUQJYnhRnER8MJwoPiid1YyiIaT8pGYM9tVwIbKhv8bW5R3sWNnJzoE6KxdWackStq2YSmtLwu41XTRcZNxFtAikToQgSiVLcEGplFNO3/xAksDQlRG662UWzKwwu7OFyS0pc6dVyJuRIiiaC1nuhKYolXLC9tV1sjQhd4KZ0XARVVjaW8WAZlAazci4j7iQYnkkaxSG80rhlRNXRzmwfjrjTcWAvBmJZjx5XfBm5DqjI9c4cvYLk0OTsU8DqOsn+1p4mr4NM8idYAYNJ5hB7iKqxrmb5+mZ84DBgSV0d/Zy59ll7j+9QH36C5K1x57bpv4O3o45fFCcKC4YXhQnihelXQ+yZeM6KKVsXrSfoVt7KJFy4dI10m/fhcJHaq0lapNKtLeV6KiWqFdLTKtmdE0p86UxSjmpsnnRfn45uPYM87v6SNJANlYETl59hwVDnWJesKCYi+AVi0rP4s88G77Ho+G7HBo8x/Ebu2gtVTAtk5gZ/7Js36xj3fPaD6/oW0XvjH5evn/Ivcd3efvq61BiZvyPZftmHQf2AjXgO3Dq/snhQz8A9uxhvZij7OIAAAAASUVORK5CYII=',
html_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJCSURBVDjLpZNNiI1xFMZ/73vfuXemmQn3IhqfCQul2ExIkq98bSw0QmSBLCyslBIlO4UUWdkpWREJSc1Y2PmIhVKUGTNmTJm5733v//8//3MsZsTCYmqezbM55zmd55wnMTOmg5RpYtoC2cX7w//dQVQJQXEihACFGIUTXBAKb+RBKQohA+he3vZPqyFqxDjBEo0gRlAlBCOIEqLiRbn9YmhCAGCsIURlsknxcbJYDC+Kk4gPhhPFB6XakVHUZcIDNWPb6k5ciOxe08n+dbM4tKHK0U01NqzsoJwlHFg/m9ZywrHNc6m7SMNFtAikToQgSiVLcEGptKTcevaDJIErD7+zoNbCivkVuqpl2sspS+ZUyJuRIiiaC1nuhKYolZaEgxtrZGlC7gQzo+4iqtC9vAMDmkGpNyMNH3EhxfJIWi8M55XCK9ceD+JFaTQVA/JmJJrx/muDD3evs/beVt7ta+dI73aWfbyNOk/2q/A0fRtmTE6G+iTnLqJqvL5zlZ3pczYdPk5l6SqKd0/50PuMt9Yg2XLpo+1dM4tvow4fFCeKC3+cnzjX4b5d7Dl5mrbPL6G/D2bMZCRbzJMnn8jGxoXCRzpbS/iWhNaQECqGSEKQlKBGNR+gdd5S2HXm7wdemM88fpKNFoEbj/uxYKhTzAsWFHMRvGJR2Vqu0njziPYHp3DFIA1gfKzESFIjmUoa+3q6LpQ7288trEmWpd8YHxa+DJViaNr5ZKpxftWz6Gw+OnCiFJPFsWT9Bjd3PJXLvwGPS3Y/UZc50wAAAABJRU5ErkJggg==',
html_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJGSURBVBgZpcFLaFVXFIDh/5yc5EZJhN6E+koziC/UtqgIUYOIbZWGgkNLUVAcOHToRBCU6qygExGHQmfViSCCD0RQELVQCoW2UDW2iimlJrnn7rP3WnstMxA66UDI9xXuzkKULFDJAlWnfvjb+R9qhogRVRGBoE6IShQlJKcWIwSlYt74mkX8x1FzcnbUHM2OqCNmiDiihmQjqXHpzmsq3pntKtlAzdFspOyIGqJOUiNqJokT1UhitAcqQkcpmWfu7Pl0kCiZrzYPsn/7BxycaHN41xAT6wboqwq+2TFMf1/Bkd0f0omZbsxYEMqoiqjRqgqiGK3ekos3pykK+O7aK0aGelm7vMWFu18yNbuPlW2om0wQw2qlrKPSqNHqLTiwc4iqLKij4u50YsYMxtcMEDUxMryeo5e30E2BKBmvM1UnODEZIRnnr7/m2ORSuo1x4upuDOHM9UwSZUV7NeuXjTPX1PwUJunKDSwmqpmQaNIi3KGOijt0okJfYs/GQ2Q3smUM5+XMCz4ZmaCTAk+mtlPYZao3XSGkzOkrf5LEOP79M6I4PWUiu/H8n98RU9QEycJsnGPTRzvpSJc3X3xNNTunhJQZ7O8h9Rb0S4G0nOnUoFlZumQUtUx249XMFO2BZfz44j4P//iFf++do1h17Ik3QXFxLBqeFBdjYvIITUokSzSaWDW8gW1je3k89YBbPz+ifngG74xSuDvv4+Nvy3rr6GeL7/52+69kfP7yrP/KvMLdeR9jJ4s6W8/iYHls+qw/5Z23QlWChZhP1DoAAAAASUVORK5CYII=',
html_valid=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJSSURBVDjLnZPPaxNBFMe/s0mTVJuWtI21JFhB60UUKmJB8aJo8WBBRREEb179Azx7EepBvVjBmygeKpaCYIV6EhHxJBTspRRj1Cpq22R/zMzO+J1JA8VLpcMub3ZmPu9935u3wlqLrYyxhycuKq0ng/85fHXmUvXy1LljG+Bxwk8qA4MlsZkCwnulVPullM+k0mNKqQLh5zvLOzJLtdrapgoIlwhPVcuVgPAs4emhStXD3D4kbsw+8hKMNTCpQWr40qpUQ8kUiVIIdQ22cwE9xSIaYYjP9WUEyVFo1YlsbuU0Roc7N8S00MbSSctqWqUtFlff4fXSBKwROLPrLnJiAA/mviPbxlZDzehYhwykB42HJa3VBzDSfcuv//hdQm+XQdTQLQeGhTx1sIjp938wfrgH+Y7ACeFjMV+L8ebTGs4f6cfTt8CF0T5MzHxBoUPARIqpaO0j5bOC+RoP33+1DCGA2zNfUe3rwL7BPCq9OWzPBdhdzqMZp4h41jSpoJloxM4BPV453odsIODW3PU2khSsKWvU5QQhJtQgHPriBrDNFEEjskgk8+F758U3n28YGw+4SCkdfVwKsVAPCaVwbRPRQawsTCIRrEQSMWG30YoMRtbr306Bxb2Xdfxc015B97YMJq8NQ3FuEwtx8ua8PTtSQu1XAsnFhAoS1aq8m0t/E63bUL4/Wtc71J/H3ONFZFfp2UkqFjKQrENBCag8D2laHUAZ6yHXXPTjG82pa/8BYs/1DzaOKNnnxE3p5k5eyj6mJei/qcBX9J8htvo7t8dfFzerp+lA/ogAAAAASUVORK5CYII=',
image=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGWSURBVBgZpcE/a1NhGMbh3/OeN56cKq2Dp6AoCOKmk4uCn8DNycEOIojilr2TaBfRzVnESQR3Bz+FFDoWA2IjtkRqmpyc97k9qYl/IQV7XSaJw4g0VlZfP0m13dwepPbuiH85fyhyWCx4/ubxjU6kkdxWHt69VC6XpZlFBAhwJgwJJHAmRKorbj94ewvoRBrbuykvT5R2/+lLTp05Tp45STmEJYJBMAjByILxYeM9jzr3GCczGpHGYAQhRM6fO8uFy1fJQoaUwCKYEcwwC4QQaGUBd36KTDmQ523axTGQmEcIEBORKQfG1ZDxcA/MkBxXwj1ggCQyS9TVAMmZiUxJ8Ln/kS+9PmOvcSW+jrao0mmMH5bzHfa+9UGBmciUBJ+2Fmh1h+yTQCXSkJkdCrpd8btIwwEJQnaEkOXMk7XaiF8CUxL/JdKQOwb0Ntc5SG9zHXQNd/ZFGsaEeLa2ChjzXQcqZiKNxSL0vR4unVwwMENMCATib0ZdV+QtE41I42geXt1Ze3dlMNZFdw6Ut6CIvKBhkjiM79Pyq1YUmtkKAAAAAElFTkSuQmCC',
image_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIfSURBVDjLpZPNS5RRFMZ/577v+L5jmlmNoBgE4iLIWkgxmTtx4R8QLXLRB1GYG4lAwlWkCH1sShcRuIgWYUQoBIUVgojLyowWLSRhSCNtchzn672nxYxT6hRBD/cuzuW5D+c5H6Kq/A9cgM6+0VtBTk4tJwM/kS7BspvDsAc7w4w8uXGyxwUIrHRev9AcqYlERMRFAS3+E1RBdSNWglyGs9eenwbyAsuJwIvsjUjX7QfU7duF51gC9cBUYYT8NYJjhM8fZ+nvuUg2EClaSKbBGJfGhv0cjLbiGAfVAMQFEYwIIgZjDCHHYO2WGmzY9DwfP1yRz/cv0KLJLQLZTIpsah1EULVYDbDWIICq4khALpNE1W7PQBW+xmN8W4qTtTmsBvxIL5IJ6pECp8ZbYX0tDmpKC3xZLCe0kPr1oBFUU0XyCmEWFnT7HNgC3zhlGMcr6TtITJBLvKK6+jtX7z/ElDV4cGJzBn9COv6MPZXTNDcfpX53I6/nnrL+ftKPdtfddAHUWgRYmp8rKRAKPabtSAeBCThc287Eh1GiTS3Mfxq75OZnLd+coYG+YvQ7rtzpJyQVdBw4B8DltnuMzw4DY74LsDNs4jaXqqotl3wLC4KFw+panLnYNG9jU/S2jzD44gx+vlYpF2CHZx6dH3h5LJnVJmtL7dJxf+bdtNdyqJXx2WHKxGXqzSTAkPzrOke76waBLqASWAWGZ+7Gen8CJf/dMYh8E3AAAAAASUVORK5CYII=',
image_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIfSURBVDjLpZNNSJRRFIafc79v+r7RUKFGMrLRQgqqVUQWbYogaB8VGURQltDC2gS5iDA3UYugWha5iqBF4CIr6JegRUQSTEQ/JIo/pIw/83/vaTHjmDpF0Mu9i3N578v73nOuqCr/Ax+grev+VVuQIxMpG85kK7DcwjIaQE2U2w+uHOz0AayTtu72rbH6WExEfBTQ8j1BFVTnasUWchy/9OgYUBSYmLFBbGVMOq71srqxjsBzWA3A1GKE4jaCZ4QfiQEud54ib0XKEVJZMManZX0Tm1t34RkPVQvigwhGBBGDMYaIZ3Bu0RvMxQyCkDC6vOj3L9ByyEUC+VyGfCYNIqg6nFqcMwigqnhiKeRSqLqlDlRhPDnMz7EkeVfAqWUqO0rOrkFKnPpgkvRsEtRUFhgZrSIymJk/0BiqmTJ5kiiDg7p0DlyJb7xlGC+omLslcZfGgTtExr6S+HyBVrMzgAMLHfwJLYletk32sfHoSYLmTaQ/9FP98kn4eG/kjA+gziHA2PePFQW2v73OhtPnCL88gxfdVNXWsS4e590nLQ6SlJpzo6erXP2OoeeHCVc1w/6z89kvNuA5afIBaqIm6QqZ2oYqKbawJFhajMQaSL3vo/phB9n0CClgesrDegz7ANWBuXei5+mOVF63OLc0wp7o7jB88yqIr4jiexGmxwt8GzWquJvyr9/59aG152cnhts9K3Hr6ZDCrX39hZ5fKo7iczfDI18AAAAASUVORK5CYII=',
image_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKhSURBVDjLpZNZSJRRFMd/937fN58zqWPpjBWamYmtlG0YLW+RVEQPLQ9JVNBCENR7UEhFJPUSZZAR0UsLCEE9VGQLLVCG5kJm2CZajjg2rjPfzHdvD4NTRj31h/NwDvf++Z17zxFaa/5HJkDFkVtn3ITYHh5x04ZifzmlxqdeGzK9XKmt2nbYBHCVqDi+b3EgGAgIIUw0oFP3BFqD1mO5xk047K68txNIGoSHXDuQExAHzl5jan4WtqFwtQ3SjxQkQwoMKfja1syJw/uJu0KkWhiJgZQmxUXTmVe2EkMaaO2CMEEIpBAIIZFSYhkSpf54g7E2bTuNNG96kvcvyuivY5NzmcGnL6la+MTfdOHYjnEGcSdKPDoKQqC1QmkXpSQCyB68S3GkAe/8CvyFpfz4VELb/ZrTKQOtoTfSTV8oQlwlUNplINaD4+ZRqB4xL6+DzOIVhDve4REOGZmTSc+e5h9n8L3Hh9UZ/VXQAQrMZ6yb1U7WjLXEvl3H4xN8aWwjMqy1dPoXyDF8rUEaHqRhJ8NMo8DTSEXJW7KK1hHtvoSw+rHSM/CqPi5+WDK89GBtu/yd4HcFBx6wJf8V/pkbiXZVI604zsB0OuveYJQdpcvJcQEkgFYKAYQ+t9LZ+hLZWk355BaCc9fjfK/B8GhiAwV0Pa7nWHMpMrsk9ZVmctaSE3b+5BF6Gm4S0IoEeXxrPsfEoIdoOJ++122I8stUbp4COClSCZDplRGViDLF5xBqqmHOyr3Ync9ov/2cxnth3j9sxL/mBLnBbHJ9cSzpYFtCpwgm2PLGnpMPl4/E9fxV0ZDfab1D4YJVRHr76Kh/r+/4Dg33VLW40AKAbYHX5CqA+HOdt66eNLpoRq65bHYh2oq9GA317tpwqunjv9b5J0pAFDVe4XHbAAAAAElFTkSuQmCC',
image_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKXSURBVBgZBcFNiJR1HADg5//Ou767jblWM9qOyWZSXlqCNMpYWbKo3VMEhYJKEn1Rlzx0CSmIIq20o5fCQ3QSskOXCPqEuhQWXkLsg3V3xtVN23LW2Z1531/PkyICAAAAAAAAAOSw//CpY+Ug7buyXA5fWwEAQAUAjBSsG3Hy0/f2HMqhrNL+t17Y3tzQbKaUcoFABZIIIqhAKAernnnzi4M4lMOVa2XRbDTTS8c/1tq8XlGrlFGQjcoSWSLLklqWzP521tuHXtQvU4IcllfIstydW2939wOTallNREnKSUmWkpQyWZYZqmWqCoAcoEJRDBseWUsEAAAACIEAOUCF/mpPv3edlERUqihVVSYhItRSabC6LKICkANEcHmp7e9LS/rVQBWlf1cWrJa3SYANxVXXu0tEBiAHiODiwg2GLvSACKIpogfgqhEXLgQAyKFCBFltDRVl+4zG6h9aG27x+9wlvc0PS8OjivOfmdnS8MnJE26OKKampuo5QARRlRZ//dy9t4Y9Bw/odru2zs356ofv9a2x76nHFUO5TqdjzdDPw1/+te6NHKKqJLTPfmPHlhvt2n3AK4eP+O/iea2xMRMTEzqdeR8ce1+73XHf/Ts9snvK7Ozs8xkkEB7afpfGTevBa6+/o98f2Lt3r8nJSTMzM6anp/UHA08/+zIYHR0tclg3ki1Vg97oHZuafjnzEzj14RFVWTp69Kh6vQ663a6qCh+dOG76sUfNz8/3UkR48tVT7w6qtHPpn4V71l79sb7rwR2ptWmT+nCeFhYWnDt3TlEUxsfHbdy40XJvEJ323Op33379Z4oIANu2bauNjY1N1Ov1061Wq1kURQ3Z4uLic5hvNBqnMbSyslK22+3L3W73if8BE1ob/QNlaFEAAAAASUVORK5CYII=',
images=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAInSURBVDjLhZPda9NQHIbzVwlWryzthpWuIpWOieKYZXO2q1vC0KFr9aZM3Yr40QunspU2TVYmYhVRvNErwQtR3E0JTq3G2o80mc0Ql9dzTr/SYdnFA8k5yft78nLCjcxJNwKzsuoOiZoj2GKsi3NS1I7y4hIA7n9wgQvyz4KiWLphwNgyoRMq+jZ+MUyo1ToOR6Ra3wA6ua4b8F/2gL830WF8YRGB2VX4hBwOBEWrnxl3kGzQyXzyLJbfLuL+uwQevr+Jk7EsiBn2MmMBdbJ58UEEKx9vYfVDE89MBtTsTVjA53iiy/XbeD4XRaluwhWSNRZQIYmeay6cSsYxfCmFwfMpEGW4wjk4gxm4J7IECd6IhOW7z/AlkYRaawXQbyuTtCOJAQzPp/bU9gtrLOBHrUECJI3bP5bWypoJx7l9cE+tMO0TsTuIpl90uCq+xJnoEtP2hUV8Cp7G90orwMECGthQd5gynRxLPUWuoOOR8huPN//gyde/iMuvmLZvKgtlfBTFdsBgSNwslavQiOIACaCF0ofzRQv5bzsd6BrV9obSyI8EUCw34JwkAcd4aWFoWn5N00ihFi30+HwaM5LCmM4UGH5SLtX28uvMtlg2mwH2U9UuNHBlDUKu2ANdo9pDwjqqpNQSOwdyrSegXeih0Rh7wQ5da2lbdDI5RBqxT/Qa2ArdUK1ddLV7/gX7jb1QzdhGjVAl10262n0D7IXSSbtpa9vf+QeB6/JTIb6VuwAAAABJRU5ErkJggg==',
information=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKcSURBVDjLpZPLa9RXHMU/d0ysZEwmMQqZiTaP0agoaKGJUiwIxU0hUjtUQaIuXHSVbRVc+R8ICj5WvrCldJquhVqalIbOohuZxjDVxDSP0RgzyST9zdzvvffrQkh8tBs9yy9fPhw45xhV5X1U8+Yhc3U0LcEdVxdOVq20OA0ooQjhpnfhzuDZTx6++m9edfDFlZGMtXKxI6HJnrZGGtauAWAhcgwVnnB/enkGo/25859l3wIcvpzP2EhuHNpWF9/dWs/UnKW4EOGDkqhbQyqxjsKzMgM/P1ymhlO5C4ezK4DeS/c7RdzQoa3x1PaWenJjJZwT9rQ1gSp/js1jYoZdyfX8M1/mp7uFaTR8mrt29FEMQILr62jQ1I5kA8OF59jIItVA78dJertTiBNs1ZKfLNG+MUHX1oaURtIHEAOw3p/Y197MWHEJEUGCxwfHj8MTZIcnsGKxzrIURYzPLnJgbxvG2hMrKdjItjbV11CYKeG8R7ygIdB3sBMFhkem0RAAQ3Fuka7UZtRHrasOqhYNilOwrkrwnhCU/ON5/q04vHV48ThxOCuoAbxnBQB+am65QnO8FqMxNCjBe14mpHhxBBGCWBLxD3iyWMaYMLUKsO7WYH6Stk1xCAGccmR/Ozs/bKJuXS39R/YgIjgROloSDA39Deit1SZWotsjD8pfp5ONqZ6uTfyWn+T7X0f59t5fqDhUA4ry0fYtjJcWeZQvTBu4/VqRuk9/l9Fy5cbnX+6Od26s58HjWWaflwkusKGxjm1bmhkvLXHvh1+WMbWncgPfZN+qcvex6xnUXkzvSiYP7EvTvH4toDxdqDD4+ygT+cKMMbH+3MCZ7H9uAaDnqytpVX8cDScJlRY0YIwpAjcNcuePgXP/P6Z30QuoP4J7WbYhuQAAAABJRU5ErkJggg==',
ipod=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFhSURBVDjLjVJRSsNAEJ3WpkKgIA2hVIL/fngJv/3rNbyD4BU8QD57AOkBjOQCPUHIX2m1YAtCku64b5NZtnSrThhmNzvz5s1jesxMsDzPX5RSj7jrSBIPh4ONzvlpNps9m0IkwrMsW+73e/7L1us1z+fzd6kbUGdAhr0uv0hpUowPUTteWP9EvL9happGyo4BaoUyouurARDMmZ1YfFama13XpwCYD5mK2RZR192AGU1a2l4GLQAfdbSRW2aqO59lcNHXUSeVH1U7N7dFwkoAqqryA0CHh9tL20GUJuqZ+3A4pO322z+CW4QOq9WKNpuNuUdRRHEcUxAErdjnRpClQnFZlpbBbrcz76PR6ESDvo+BdE6SxDjOeoHsu3cE/BQGoCnu6uOO6NVAAMbjMem1pqIo7BjQ4VcGLsBkMjFdXRGn0+n/AMIwNA4QnyHHLJ1Hg7c0Te+ALo5Eia5rPRZS9wNIBbXTpSVpqgAAAABJRU5ErkJggg==',
ipod_cast=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJZSURBVDjLhZJLSFRxFMZ/995xXpYzaRn0AsloEUlQLSwqS5ShhGjTwtpZ0Toogly0iB4uWrWIVkLgvsVEi4wKsUULIwQxLGyU8TE6vmbm3v+9/0cLnXA06ducA+c7P74DxzLGUKHB3sPALZQ6h5HHkBKM/Ib2PyH8V3Q8GF1vtyoAg713UOoRsWSMSBQsB4wCtwD+EsxlXEzQzZVnzysBg71hjHlDtDpFVQSWcuAug6XAF6B9cCIQr4FCHhYm3iHEZW689m0AlOohVp0CIDeu8BZa8d04bXctLnVbiGKchalWJocVSkF4ewpRfLKa4OPLo6CHiO90WJ5QlLyDWHIe6d/HBBfWEnxAeE9Rfh2B+El9o0NmSKGDphDK7yKadCjOgldqRwuJkd+JJRso5MGxIVTTTDHfiRBnUH47s2P91O13mB7pslHe6t2FOQjcLwSl24RriuR+9bE000I+28L0jz7iu4rI0nWUO8jitCS6DdxSKoRwG8BAUAJfiJE9bTlL6/HF2vMdWutOpRRaa8JefkbUttW3DD0WBH4BnCTa3xtCuiAFuEVQXqhQtePqseMnT7NZu/v73zfheTaOsbA0WMYJUVzJMDXaiFdwkZ4h8E4A9LydrNi+d3EftrdyCktaGLPM768JIBPi2otD641iYCBS7s8eqQdgfN4HwLUicR4OS+BA2RPamFMp9bcfm/UqZlLKTXdtApRNN5urgVVY+d3Xw8uyt0pg2zbZbJZsNott21sC/pVgEUgmEgkSicRG+OJ/AUqpdDqdblNK1Zd/YK0WtNafN/r/AJRSSvzM+v9SAAAAAElFTkSuQmCC',
ipod_cast_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKuSURBVDjLdZLdS1NxGMdPRZLgPxBC0pVhWHgRXhiMdlEUQUXBLhLUMnux0gxUtBeX8yXzuCVbKS5Htunmppt7cW5OXGeerZZhObVA8DIaBtoLljr89pwjWm32gy8/Dvyez/N5Hg4DgPknvC6dwoLTjuNlWwwj6hh8qnF4m1g4FOnx7+OLy6hwCW8swKQTiLjX79dGINAOWKuX0F9elgjgdUkY63Rj3AS8twFcB+BhQV0BZy1gvwu4FMBoK+CoAbouutGRm/QHwGlVeEvFAsCnilGhlHSTNzuZy5KhL5biRVEMNoIZSwH1GXYd4G/LhP9JDOFeKmZjsNelwSFPIV0F6QZhLAmi+6oCuoIUaM+n4em5GPoqAeWxGFhpBoORVhX4LtJTC7pS2O6kwlo5h6FGwFIOmG/RXQF0yOagPp2KxyekBKExCoGmHJYh3Y8Im9Zns1Ylk24tXPUR0jXgWZ4E7TIJ6RpguB6B6mgFWo7sQsPhVfTSGPKsaYZ0lxGmLfdTN2PJ9hnOefOD3+4IhUJfeZ4Hx3Hw+/0IDvV/HnUPsKjP3kaFC+i9Ddzfv8iQ7jKCzwFDsbDdpHCIH1tZWcFWcbsHh3Hv4A7IDyyKo9VkfGdou7PoqwLpLpHuzjA38kt4rLDN/ZPBVxaUaXLXilpOoUAhWet+cEgATDPxf1YgEMAGgJv9IabOrIPcXADnlAbvPnmhHL4GmTIT2cW7mxMAwrwbgE7+i5i8h8cxEFFhYIZ+JDrNvktQ+i4LgJ8JAJ/PJwKi0ehmTlZnYXBKi7+PfVIjAJAA8Hg8ImB+fh4TExNiJKV78ch7AQ3efLG4wZP/fwOXy7UQv32NrR5nm/eB9RaKnYVb+N5yB3a7XW+1WqMWiwUmkwk9PT3Q6/W4opAt59zYsypoU75RGoX3vwEsUe6Qo5mAvAAAAABJRU5ErkJggg==',
ipod_cast_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK7SURBVDjLbZJRSFNhGIaPTS3BG+8CA7ObwrDoqosKa2ZBIRQUBgVRSRRWloGalSYtC+HUMqlES7KZm83NpmvOZq6jU1qGpGiBNIgiMiON1NQOPv1noqXrwMt/87/P/3wfRwKkefFVrhSRUSq6eHFXpaVUxWPsorlYpsGwcuH9heUsURznlRV6GqHXNXO+NENbGdjPj2PLzgoF+Cojab/vossCb+pBKQe3jHgVGi+D4yI4DdBaAg2XoOqIi/IDkX8BSoWR16KsATxGVRT1Qjdq7qXHWVGYMvQ8PKpSL2Dm01C6W54BeO8m4r2t4q8VZVnFcSWOhsJooWsQuh2YMzt4dNxA5aFoKvbHcWePSl0u3NiuIusTJFpKjPiqhF6ppqun/kIs9twATdfAmg2Pz4gzB8rTApTuiuXmDr2AiDHSoXiDLAndd/gtM7PZ86KE7mWcRb1Ct5p7B5MoS0sSutVUn+jFuC2H61uWcHXjb2rFGIXr+iShO4lfbNkmXjNnLupXGk+99ToaOjs7f/h8PhRFwev10tFk+9LqeiJTtD5MFIepPQsFq0ckoTtJxwOoztC2G+nv9LVPTU3xv7hcT5+Rv1ZH4ZqR4GiXEn5KYrsD1OUhdMeFboRfaZnQLhvqA/PywVZC677l0x59+LR352I1kBGvAfqkhX9WW1sbswBlYDQYxy2ZnuxN/HLKTPc3MWbO4lVGotqcHH4yBKDNOwu47/sWjHPvCsZFmZJUyImBoniGijfj3qoLhAA8Hk8QMDg4OBdPcgTTr238+w0XLEUbJwTgdruDgKGhIbq7u4N5nraM0XtHxNaXMpEt8V3kwzEd7hTdpxCA0+kcXrj9gLkY/9FVfM6J5+u5CN6nh9GSGq42pejyQgAOh8Nkt9sHrVYrFouFmpoaTCYTjvzDk67UmDFNW7z8UStr9/8Aj3HtMz4GxWkAAAAASUVORK5CYII=',
ipod_sound=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI4SURBVDjLhVPNaxNREJ8k2yxuDMGEkGi0QltREC9+HQQPKl5EkBz8D7xI/QDxIh4KpeA5lII3IbeAF6UKHjU5SS2op1JKTFiN5GOD6WY/s/t8M5u3bkvBWWZneJn3m99vdhJrNBqrjLEH3MH3fRBRuOd55NN8qVwuL0PU6vX6N13X2f+s3++zWq1WxwZRlxAZbf37CBiP/IzcxwL4l1+bZcRkvxGAG2DAscxMAIAPRaBXS3NIguu6VNdefHoPDPP97Ku1X1KAynXzwuGuBdudMSiHkmHnoxmJxwRdnEwm0L7/5HkK3BV9bGzwo0uBhKmeTEqGCwvyHimYDwcBgxPq7yKMjTe2Y68cjnsXf1y/s0gMEjEAj1e2h+4e/X5kHv6Xr3C+rS7ounHbNa3HcpJVYGw+JADsfuuMHGok6VNWaMmkDNrnFEx2deAMXpz88DrWvHyzApZ1WkJdwhzHgW63C5qmEeVsNgv5fJ4D8JkguGEBGxtBsWECs20gAPEp8bKqquFC8f2gPJ1Og692QHIdYIb1rHn2yqO0IsHoj70VFwywcDAYUCyVSuQI0uv1KMZuXIVP2SM7YJrrM75XYY4NYNmr8agE0VloF/MRM9mcP95hln1XTjAY/RxuzPW21iQcnJCAmpF2q9UKgfBMAGCzuebm0k7uVAecyVvaxCgDHBh25XtPF3K5HBSLxRBAfKX5wfbLcJXxEH9UFIW8UCjAQYY1B/4XOIOP1Wr1HDLBgmgUecTf7Qf4C2kj+HVimC2aAAAAAElFTkSuQmCC',
joystick=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHBSURBVDjLlVM9aMJQEP6eNEF0sbiUouLgoLRkKXS1IG4dC6Xg2LXQRXATHbqVzq4iQjc3sVscnUSnYIdIB9GC4L/xL333IEFsBj04jpf77nt3l+8x0zRxaMViMbTdbtXVahVer9dYLBY/0+k0mcvltEPsGRzMMIyPQCAQ9ng8IAJd14OdTuedp+4PsS4ngslkctFoNNBsNgWB2+3GaDQKOWEdCTgY2WyW9Xo9QbBcLoUfTSDLsoiMMUFgkRxNwHeAdDpt+nw+8EUKp29O5rhEvnEoigJJktBqteD3+0/rgINNulHTNCjzGR5++1Bvb67x+vLF/dmxg3K5HOZB2+12MncxfzAYxJ25wcXjE5ixZCu9m/wufybfUqnLUqmUtwmomAtKi0ajcrVaxWAwQKFQEHOfK1dQajUwrwdSrw8ZEiKRSC4ej0NV1TwjJXI2IxaLyZwA4/FYFHL12T6fz+3o9XrhcrmQyWTQbreZ6IAnZS5dVCoVEpFYmFVEPpvNxJm+0zmRSIhoj0AJunU4HNogq3C/EwtHuqBfaxNQkhJ8NpGwAPtxs9n8c5ug2+2iXq/bojl0S41URKPuv2Dm9JxPsT8W0mO2IJm2EgAAAABJRU5ErkJggg==',
joystick_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIvSURBVDjLjZLbi1JRFMb9N3ro8mYgPqSBCIEPoQSGYYVREkxxTLQsREkjlRG18doxZbyGF7y8HAONQUgnENEHQyFmJqEnoQf/gBDKl/na+1RDEz6cDYvF3uz1W9/+9hIBEP0f1Wr1XKlUWuRyOaTTaUSj0W8+n0+86a5o02GhUHjX7XYxGAzQ7/dRLBZhMpn2BAMSicR4e3sbyWQSFFSr1WAwGA4EA4jcMc0ulwudTodXoNPphAOCwSAPcLvdaLVayGQy0Gg0wgFer3fsdDoRDofRbDb5p6hUKuEAu90+LpfLqNfroCpisRgUCoVwgMViGdEvJErQfGrD4rkTP23WYzx70iPBbAQ0Go0LJNbUcbPZjHw+D/KdWHpe4PiQNJ9+wrrF4cvN2zAajb5TAFpM5K6n0ykCgQD0ej1kMhkkEgm4uwZ8jcfQzLjhTN/Do9c3YAxehZq5/PEEQLryxavVCsvlEovFAvP5HLPZDKPRCG/qIfi5B9g72sXnZQ9s34o7rBSKx2fifycPk8mENy2bzYJlWYRCIXg8HjgcDtx/pUbnMInOPAW64vsmsPtmCvjBA1KpFIbDIWgmcw+/38+7b7PZwDAMrr+8hO7RW/y73h/sUsBvIyKRCD/zdGzb7TY4jgPxBZVKhTfyiu08Yr2H2Olt8cU7H7ZOK7BardBqtVCr1XRgoFQqIZfLIZVKIRaLIb12FrfiF5HoMXxnmun+xAMhQS6HSXynsv/kMD3/Bc9MubHqnCOyAAAAAElFTkSuQmCC',
joystick_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIxSURBVDjLlZNBiBJRGMf/r9FZc6lcjLBQdzaElI25LNWhSx0kiD1FUMEegyAqakHYS7hG7C261TEQETpEEoRgS7QDRXgQkShmrVCi1W3JUMcZRx2nN4+0kDlsH3x8w8z3/d57//k/YpomJiOZTAYNw9jo9XpCv9+HpmnfFEWJJhIJebLXAZvQdf2h3+8X3G43LEClUgmUy+UH9NPiZO8eO0C73fbl83kUi0UGcLlcaDabQbteWwBtRjweJ7VajQG63S7LXQN4nmeVEMIAI8iuAVQDxGIx0+PxgArJ0npnF7YiUsUhiiKcTidKpRK8Xu//7YA2m9aKsixDVDu4uLONjZMLx3H7Zo7mVdsdpNNpgRZ5OBzyNNn5A4EAzpgD+C5dAdG7pFepRj+nn0bXlpaOpFKpe0wny0jWMK1yOBzms9ksCoUCqtUqO/eqOA9x4QS0nyUom6/R/bUN1b0PZW1qfXm9FmUAStMjkQhPAWi1WmyQum+c/ffPMFPP49jZC5iam4dWyuGD9ApN+dMtdgTaxFPrIpPJWCZigqmqOgac/vgEp27chevLG0C6D/cBD44Ksyhsmnccf0RjqzYaDfY8Gux0Oqzu1xW4fHPA+eW/4q0eBjckwmgHDBAKhdg/t4zzb9V/PIdafInpF9eha3Wolt1bHAwOWwxgCSZJ0tg0k7l4aBoz7yQIB/fCwTnR3hnga53AhPmI2F1nu3h7ObjSaWxd4wwya3Dmdzr1+FxusPYbDlZf5OWGzT4AAAAASUVORK5CYII=',
joystick_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJZSURBVDjLjZNNaBNREMf/25Q1Lqmk2NYamo+amI+mREOhFhRS0CKKN8FT8NSeiiBIDiJYk4MXEcGTwVtI48FQouLNix48RCoh4GETqAmioTFG0qT52HT3ObtgqiGHPBgGdmd+M2/m/zjGGPpPLBazyLL8XpIkW7fbRavV+tZoNFbC4bDYHzuKAafT6TyZmZmxCYIAFVAoFMz5fP4x/brWHzsyCFCv16fT6TQymYwG0Ov1qNVqlkGxAwEUjI2NDa5UKmmAdrut2dAAnuc1z3GcBvgLGRpAM0AoFGJGoxE0SM3Ub0MDaOLwer2gQSKbzeKEoYIHa1P2nXfBqaEA1C5TK4qiCL/UwOLxInznbgiGgi6N27dWB64xkUjYyImKovBk2v3NZjOW2E8I7ss4dtKPnYmktdIsPX8YDJri8Xik14GaTIISXS4XTztHuVxGNBrF1otnqBt3MTZhhLz3GpO+m4C7C+dpezgQCNzvdUAVRY/Hw7vdbphMJgSDQVJfEwfFBKZP+elOn5HefIWltbuQxvdx3jqOl8lqmFIjGoCkypN0kUqlVBFpKzOw77h05gjGxptU/SvAFMiNT5hdvoO9t0/B5PbhENUEdWjValUDNOo12PU5GC1noTSzlNvCwnUXFOkHqTIHm+8qVuZ0hwDqQAM4HA44nU4sziqwzy9DGKuAHfwiRemwvZVTLwulncOk04qLrlF8iMzPceprXF9fZ8VisSeae1c6uLC6Cd1IgQC/KZH9t3ndURd2v2SQexNJcoOe88dHCxUmKwJTGLWv/GOs56F6xrb/AFceRXFTtLBJAAAAAElFTkSuQmCC',
key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH2SURBVDjLjZNLTxNRGIaJv6ZNWeBwkZFLQtGAC4l/wKULV+7YILeSYukEUhJoSASVuCI0hpAYDSUQCJBSoAaC0wbBUi4aWphpO52Zlpa+nnOqCAptJ3k3M3me73LOlAAoyZfDqQdOEvyO89/vRcGZ5HeWmySFYdWHVOQN0vE58jrLJMFJ82hewVU4+bMfqdPxP9VBn+A4D88wP59PwFqmsH7UgeTJEMlsTuIyI5uRsDfCMcmtAtoyhVmOu5kkHZuFsiNA3XuEi+QCdhxluL0D/SvpoO+vhIksiItNiPqqyXgfIL403gjfoTsIL70gQBdim3VQvz2FFnwOxf8E8kYF0rIVYqcRM70Vgf/Pe/ohwsutOJdcpBpP4Mek+jPEfbWQVzkG+7tNcNsqt68tkcLZTIzM6YZ21IbolgHq9j1o+z04nKhHRnlH2p6A32LCvFD55fIYr960VHgSSqCFVDJBEeugh+zw2jnpc0/5rthuRMBaioWBqrVrFylXOUpankIi0AjJY0DC3wD9oA9rAnc2bat+n++2UkH8XHaTZfGQlg3QdlsIbIVX4KSPAv+60L+SO/PECmJiI1lYM9SQBR7b3einfn6kEMwEIZd5Q48sQQt1Qv/xFqt2Tp5x3B8sBmYC71h926az6njdUR6hMy8O17wqFqb5Bd2o/0SFzIZrAAAAAElFTkSuQmCC',
key_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJRSURBVDjLjZLta5JRGMZH/0X0DzhG0Wy1PdVG6aIGtYI+9CVhRSt6gT4Vgavl2vI9LVIWrWL0KZTYpBqpFMUMZ03aMpXVMttLpPX47qPT7eqck2NWzjpwcXjOc67ffZ37nBoANdU0M9RqJkJJ5j///5e5mPvEVAnyb3PGi3z0LgrJZ2R5iUFCFslAVUC5OfdVj/z3weXqoCM0yMFt4p5XA7DI1CzMXkLu200i5y+IVYKlIo/pfjGDrAqgkamZaU7BIIWEE6kpLTLTbVjMvcCUcSNWTyC8JwnUKxAG6kHSvxtxbxM53kP4u2ormtfQHkRGzxLDZSQmmpH5IEM2dBKpwAHExhtQiPXB310L+9WG4N/3bduFiOs8FngrqcYR8z5SvRNJ73bExsTMHFDUwaHa7PutidS8VEyQczqQnZUjPilCxteI7OcrmHnQghHXRZy7dRjHr+2HrHfHYtuFDV0MUP7S8hELUsF2UqkOKX8zhLAGHo2Y16ulvMJyBI98Jrydt8PgOIVDhvVoPL1WV6ocJ5GHkA5KwbtFSAd2Qviixmut+IdN1XRfpmwVht8ZMey/wd6BztkJg/MEBQgUkFyIOUizOPAuEbIf24m5Dx6tmH+s5W7TmHvl9XjiG0D5sE2aKKB05+lXSPilpGF7kAn3wK2qj4/ouf7l/nBn1gla+1Eo7R3MrHzasZIgbJWMC9FRZMPdEObvYUwjjtmNW6+XXy3Z2HtQJ4LefoxVpjP9Zj3w3GmRT5i3zL0xborSM780bVNWelxks4ooSWOXZhVd/wllWOl+7RbUkgAAAABJRU5ErkJggg==',
key_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJmSURBVDjLjZJdSFNhGMdXW9BtdxEEeRFMJKiTladSajoqSOnOqy5CgyIkKFDLyI/Q5sZgRYpQFtJNpYgIkm5mVFpTc7Cw47Dmml/RZufs65yzr86/9z0qrVLrhf/N8/L7P//neV8NAM1GmuspaiXCilr/vP8vOB2fUbWWyb9h0YVE8CFSkSFSVlQTX6fhwYYGmXD8qxWJpY7V7qDH18HC2cK+3MhAjUxhef4m4t/uEQ0um3QZoKR5eNsY1WRdAxqZwqoW6lWTVHgQ0WkLRO8p/Ii/wrRtL9ZPIH8kCZp/mahGDYhwJxBy5ZHxusHdyF4T3kx3EBi+TIA6hN35ED+dheS7iOjUGQgTuUgJjeBqs2FvyvX8/d69RgRGKpHku0g3lsDFpPt5RFxHIYwyKjxVnwOH6cDkb0uksJIOkzkdkOZrEPqghzh5CNKXW5h7WoDZZ2V4d24XBg062E9vTQ4YtdWqQeZPSwQ6EfWUkE45iHL5kP1mjJkZfqhCv+S+mge5zwqF64f45ArGL+1J2wt1FSudQyRyD2KeQvBOPWJTxyHPNmPcwnzvNeU9HijSeiUC424xULUNaMpC0HwMpD5DDSJJwUGWxYIf0UP6XELgRoxZGL7Pwt6nMUlsRZnoRuYRarfTcZTlN4+9RZgrJAs7CdHfAKdpX+i5lW1b3Q+Zdy7WXgYQKF6pAU/kv6AFqc9r/F2GCTk4DMlfC3nxEUbNjGC3Hb6T+bRvSnfUjZbvTi1WZSFwbQu85ZvwoliX7jdqr2vG2gtq3K0HF97b9gfpzK9bjtxe63MNl+6sJh19NDZNRGFa/wllZefQFCrWMAAAAABJRU5ErkJggg==',
key_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ6SURBVDjLjZNdSJNhGIa/VSQdRQcSISQeBI4R5ZrZNIO2Mj3I6CCCfinRfoR2IoJM020sp8OcoHNQassG1laNpAgdhqI2f0GoOSwTS4u22p+b29zP3ft9Wi2b1gf3yftyX8/9PM/7UQCo9fTJeERNhBWpV9//lzkc+MAoHuTfZt84gvZWhDw95DjKQGb0grvrAmLNgS91CH7T/qwO+pvR8mFu4r9aD8BEps3+uZsIfG0kMi1DDAJEww5Ma7gMZE0AHZk2M5qXMJCQ24SFKSV803mIBHoxpdqLtRP435IENb8hDEgKj+UYXOMHSHtPYClnxzVvoGdg6xcRQxXcE9nwvTuP0o4sFLalwTnGQ8gph6WSja5bPOvf+352FLaBUiw5DKQan5iPk+qFuNHOhaTzDE5rUjEp4aBbkf7mjyHS5mjYTfrsxuKcGCJdNkQPMnDtPg8FLXtQ/vQUHo00oER/Ajn126PZym0sBhD70oI2PRas+XAOcVDcngHjuBqPxxoZY8eICipTCbSva3FVl4P06oTQbhmLtVLZRSIb4bUK4TCnwjt5GAWkX9p8u0uEmpfFkL8oQlXnJUifX4GmT4Zz9w5iVyUVpAGeJWc3GRYfjoFULL7Ph/+jHGfvsPFwWAWduQ7awVq0DigI5Dqa+6Qo0uUipYKaTCqjtizv3DsIt0VIBpYL36wUZkWa62RzSjSvISkirE+MZCm3Ri60ZULdW4XL7TlIrqAmdpRRCcwMZg2CMb+9H4uzlfB/bsNQLdfZpcpsWL1ejoxlv6gVIllMjRLz5l//wnDLIfGEev/8qGqffUTJ/d7XlFUd73GRyPad4o1ILKM2xZ7/AGAf1Jkquq5mAAAAAElFTkSuQmCC',
keyboard=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHMSURBVDjLxVNNSxtRFD0jEYqgBCMU00bEWFsqSNoi1EU2WiiluHEluuumfyH0B3RbuvQHCIpbaaFSxZULQYMUAklJCQZDjUTyMfl48z5675t0AgqCuPDC5dw3M+/c8+6Z5xhjcJfowx0j9L/IZrMfPc/7QtmhaLbb7Xqr1SpTppvN5jlhlTBHuZNKpeQ1AvrgK20EZSqZTK7dWkGj0einrt+JaPM2R3D28/KU8LGb2wMRIPz8LZTSkDYVPKkgPQVB6Hm8lhaFXxeZwDwM1QNGdoWN0Zza2LUi5JqfKa1tTfzYz1R6LkxGB1E8d/Hk0RAKf+t4FhtC/qyG6fEwcsUqEpPDyBQuMft0xD5jhUJIOHu/BSlooFh2LTO/4I6SuwRHMQEm4hG8nIpg97iEnydl9EnpS5p/MYo3r6J0Vo33r2PoCIWl5DjaQmNlIU5rjQ/vpuxmDibkeTjffrkm+qCFP6UapOTOva6swAuQlKmespmJCHbSJYTslKnr4twYNnbzWJ6fuNG2z+tpfFpNYPvg1DZytg4rJjYgoLpT11rbCQdoug5YF8gVdkr7+OPoDKGOkBcZ14xc8devu/+AVamUP2BKTdm9ghfOvd/Gf3hhfpp0UP3EAAAAAElFTkSuQmCC',
keyboard_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI9SURBVDjLpVNfSFNRHP7u3MiHZqJCumZZamVRaSas0V40i4gKeoreIjKI3iK40GsPEpQEIRRBj/17kYKMSodlZk8rmy42mUjjLrfVnPt/77nn9jt3NSFfFH/w8f3OOfd+v3/nSIZhYD1mwTrN+s8JBoOXNU0bIBTJcoVCIZ3P5+OEL7lcLkacIg4R3sqyzFYI0Ad36UcQZI/Hc3/NGWQyGRtFHSahp2spQRoLsx/EzmzICxJA9Z7j0HUOZkKHxnQwTYdKrGlizUxWS35ECBibremyopiKGAwX4Ia51omFL/Z0zk2f9DEW+L08hRaHHZFYFq1bqjC/kMbuxiqElSXsbapGKJJCe0sNAvNJdO2qM/dEhqrKIHlnVcogg0g8ayqLAxGRiSjlUowytzfX4uDOWoz6ohiZisPCWCml7o4GHO10UK0cJ12NKKo6znqaUFA5zvc0Q0m8R0i5hYev+3Dpzml8mnls9kN65c8ajso85qJLYExEXo4qMtCIo79GUGH3obPtEJw1rfBOD2HS/wHFpBtWs8sU9dThrXgyGsa57h0rRtU3cB1nuo5Bt+g40NCLdzPP4drnxrOhYVhVjYHZOD5/T2BbvR0f/QtmhznnJaYpxBajsEkbcaLtoil4recBXn4bpLMXsBZVlghkjbr/5lvy/96BZHoR08oEvirjkHsfof/NBVRWbIAk2QxpNa/RddVx07l90w33/iNore9A6KcPE1PjiMylbkurfc4k0k90hWAniJs3OHlPkf8A2YeC/G6HEpkAAAAASUVORK5CYII=',
keyboard_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI7SURBVDjLpVPPaxNBGH2bbGq0pgYTsE0a05pWg0Wov9BLDlaloCfFQ/BQ8FQUvHgKCNKTV/Gi+B9UL6KgQiWNxUTwZCzBSFKjwTQ1TWxpkv2R3ZldZzaagL20dODxvpnded/3zZsRTNPEToYNOxzivyCXy03ruv6AocWGrKpqQ1GUKkNaluVVxhuM8wxvY7EY2STAfnjINoIhFolEnmy7gmaz6WBZ3zChp9tpQVgokJ+MB6V8AkwA7qOToNQAsUChEwqiU2iMdZ3PicVaOy5xAfOA2Ogocle4MQaHYVpzypjHfI0ahhUzfSxk17oujPhcKK1KGPX3oVhpIBzoQ6Fcx9iQG/nSBsZH9iNbXMfpI15rjVeoaQRCYkljFTRRqkqWMv/AMxKepdOK2eHxkAcnDnsw/2kF8cUqbIS0S5o4PoALJ32sVwOXzwbQ0iiuRoagagaunw+h7/Msjj2/hvrtQaSmQqCJR9Z5CK8zkulzKvi+UgchPHM3K69AZ+zJPMPoehzhc1ewa3gMyuIcviTjeN9zEcKLdN30O1Wrx9n5AqIThzZZ9eFGGKeiN7H72ztgOQnsc6MmBpFMLkHUdALiMPDxaw3BfhdSmYp1woZhtJm54CgX4ewfBi7d6V6gmQG4mr8htjRSy0qm9z9/2/HfO3Bmrxdy+hV6X95CS/kFmQk06nZIezyGsJXXmIz6Z3pcvXcDHiKKthIaVYIfFTvVVfOesNXnnIoejElr5Wk7FYLUbi6zXY8n58j9P8T+gmJP+OvyAAAAAElFTkSuQmCC',
keyboard_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIdSURBVDjLpVNLa1NREP7uzb2hkSR2Yc2jxtKg2BYExY2C2RTBRXHfPyAu/ANZunYjLt24ElToShddiC1dSNWVSCE1bSJqsOTRxvTmvs/DOSdaQVexA8PMuWfuN/N9wzGklDiOmTimWb+Ter1+J47jB+QhmRcEgeP7fpf8g+d5HYoDijvkr6rVKvsHgAoe0o8gr1YqlUdjTzAcDm3qukpAz8ehYGw02TeKZ9yddRAAJhdugnMBpp0jZhws5ogoxrE6Mx2jUd5SADJnOUeIaitqMUK5kPrMKapcfeNC6JzwsVE7+LOFc8UMWh0X56ez+NJ2MFfK4vP3AQa+h2avj609H439GP0gxkIppSeMIgZjfTeiCYZodV2NrC5UR0Zumgx26iQuFNOw7AQ6gwh9J4QtfXBm4PXHLkzGRiMtXi7gxpUicRVYulpCGBFHM41y7gT6oYG9Q9LCSGD6dAZdZ6SR0sPSIlkSLze/gjHVWeDpWoMKJAI/QjJpg4VM8w/pfsJKgEmLhGZUz38BUNdb187i2VoTy4vlI0FX3rZHanNoSlpkA/BcF+lMUk9hRTEh2QLvtnuYyWfwZqutFRY0SRIcjs+RSSUR0kSWSVQOYsj9TzAdh2pmYIUR69Vceeqv/epcibnbDjBbyGoKh24I1q7h+lQHPxqbyDnzjjHua3xxb+l2vpC/PzV7aXL7/apn/M9zfnJ3brlQvvhYCJb6CQyniQeOtrI4AAAAAElFTkSuQmCC',
layers=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHnSURBVDjLpZM/a1RRFMTPfXv3L5sYIXYSxMbCWsTCOo1JGlPYWKeyshE7v4JNwCZfQRBBBBsLERsbv4K1hGz23TMz91i8t4muleQ0h1vM3B/MnBQRdpVJz78c3Y8wq5KJMlJGyOg0OAyrXbr9/tmnr38a5IjYFOtnUUawE+NvkZfVGw/XCTIhE5Ums/GL0WRklZckWCM5P1vuHBzvPuhJXvUGNEJpPB3d8sLXgoYEh6IGdGYAGc4Gjrw4XbbD0eA7Ch5fEMBpdCY4Ex2T2cb0TcQ4VSqRSiQToUSnTefTiailt37t4Hj3GxyWUWBwNMTY4Byb2Z2zX4uXVdoi61zSlqg5wZkv/UatsShLH1iynyjYz966oSMwFI4IWUS9LdWNKm0ImhEYwzV050xSeMHALBovsFzajgAFqRSfEDSx3hQ1AzShYwTnAI4BWh+SGnoLC4uKQusJcK9dtNuL0+XdzetzK63vEMwXwi7GhAITa+PFI8IqHJbfHn34aGZ2eLL3pMnNDxRYlYogEWzgSl0KNDgrnIHOgAQtr+LoSR5569ae+7YXJDhq/3ugIAAanQFnNTMQtLR+C4cnewHHOxQEHOGFtTOg4BBBsaMLUU/zejUvSMpllVf3QO+r3re1Sv8S/O80dsX5DaeOJ5G2UDDRAAAAAElFTkSuQmCC',
layout=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFySURBVBgZpcE7a1RBGMfh3xwnRziwBlOoBEG28FYriMSU9oKkTop8AcFWEKystRDEL2BnIVbapdVC8EbAYCEokgsL2T05M+/fd9CFRTcYyfMESRxGxK08fHMLOANcAy6BhAiYqKpAFcDMsMz2TNT6+dPHLmPi9s0LAUksP3j9WP/p/tN3ckR+WcQ9WdtkTOxvdWGOD1+2W1ykMJ3it7MnjlKEwFQfv7UUw1FX4yJOUoMT8Ol7y0Eo54CLOJkibnVhjoOybBQRJ1PALd19ycgMEzy/d53PX3eYpj8/iyxTVDiZUfwYdly5eJLNYWKs19T0mppeU9NrasaUMkWFk4liMMp0ErttYlKXRZfFJLNEUeGCGUXbdgy7TB51/IvSHkXECTEm8ZeZI4E/SYki4iyZgIB7sbbBpMHuHtPIJFzEZbMNoP/20Q0m9edn2Y+kLVykyPZs6c6rc5KuYnbcLIEMWUaWQBmZIQmQwAZQvccFSRzGT6Th9AjEKBwhAAAAAElFTkSuQmCC',
layout_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHTSURBVDjLY/j//z8DJRhMxE8+VwjEE4D4DBD/j5989l/8pLP/4yec+Z8IpJMnn/2fOPH0//i+0+9TJp063b3uxv/uNdf/Ixsw+z+JoHPVNRQDboB4c468hePZeDAIJPYc+4EwYOLZDzADDt76AsaHbmPHMAMiWg7+gxsQN+HML5DgbAI2I7sgvHEfwguxfaf/kRoGoXV7EAbE9JwCC4bU7/nvU7vrv1fNLjD/3tMPWDFYbc1OhAHRXSfAgg5l2/83L7v436J4G9yA1++/oWCYAcEV2xAGRHUcBwsa5235X7f0wn+9nE0oBjx78/X/moMr/5fMTPif1uf/P7HL679veRGSC9qOgjXopK//X7rg7H/tlLUoBizfu/h/4+rE/1uuTP1/4dmu//27s/6H9+v+N8+W7IG4oO0I3ICS+ZgGgGzceHnC/43XJoHFe/ak/u/fkw4y4DvYgIimQ/9gBoA0IxsAwj7Vhv+3XZmDEgubLk0FGQDJEKENB+7hizL7QqX/3buS/rfvSgDz23cmoLogtHZff0jN3q3B1XveBVfu+h9Yvu1/YNmW/wElG//7F63975wX/z+4R+N/764UsM0gGsSHhwExGKi4A4g/gZwNpTtA4gDRWxxGCQ/RXQAAAABJRU5ErkJggg==',
layout_content=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGZSURBVBgZpcHPi41hHMbhz/P2oJijzEKSQn4uFRsNK8lKKWZtFvMPKFulrKxZKGzs2MhismI3S5Qpg0YNZUFiaso5Z973e9+eJ45mYWo015VssxGZYur2qyvAXuAUcBxsTEKmaRJNAkkoWNqUvXBkz/YTyFy9eDRhm8u3Xt71f7r56I0LMr+dprg/+50Rs7bpiXHefloaUmQqeRd/HNq5hSol/undlyFVf9BupsgUtrdSGHj/dch6OCJRZArLmWJ6Ypz1UogqU1hOFJPXnzGQkGHmxlnO3ztHr9dDEpKQRETweGoGK6gyhSWqb/2WM8d2M/PiM1WvN8bBA/uobGOL13PzVO6CqqGwTLU8CFqbn8OOKkIkEqt1EVRSR5UpkkQ1HLb02yAGLZUkbAMGG9tEF1TuVqgyhTEjNn91XSAHlW1kEwoqu6PKFOpkIFE8nV1kpDe2jYUPH2nblpW2JbqgSQ2VZVNkipAWgf1zdy6w2oNLD1mL7R8UmSr0ZPLa88O2TyLtkDqwsAKrAweWsA3YoGVo5imSbTbiFyrGMBGmEu5TAAAAAElFTkSuQmCC',
layout_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHySURBVDjLpZNLaFNREIa/k3vTR4qKFdpaxZhFfeBCUEF8oaAiulIkoCtdFATBRcWFG6EgCG5VfFBduBVBFIIQHwga3GlREJVqS4OhrRKjwdxc7z0zLq5JGwwSceCszplv/n/OjFFV/idcgKOXXg4BSWArsB5UUQyixGKGmAERQSyluKtjK5fO34Aopw6uMqgqRy6+GNF/jPO33qiqRgqAbQA3csW6tL8ZG9zSzdvJkl+3gGhf7XKgpx0AY5onv5v2AfCqQVsdoKqJWtX3M35LzVNrzSxA1K1JazXECnMBBiA9/IiqCKKQObub8cK3psmp/gWoWABiESCiffECNq7upeiF9cfzEm0Np24htA0KAChXLYEqFT9sqBhYpZQd4Wv2Gl5+jHzvEtbGdwB7I4D5rcD3A7zAYqtBA6CUvU4wept16UHaU2vwXmXpfPqQB7viJyILc3692WTPZC4wsP0AHR+eYG4eJvHxDqlkN2p0yAWQUBQwAPdzEw3J5cpPwqlJOvpSsO/k7A4ML8YRs9wFsCITQOr11f1/dBsg39NPZTRD173j+N4UFaD83cE6FKJJtHI3febxClXdhMhCkRBUULGohGzu2knb82ckF3XiOnHKn0PGp2OqyGXT6jrnDi07/aNYOOZYk7SOflK4sicbnvsFhzwbXdu8qEIAAAAASUVORK5CYII=',
layout_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJeSURBVDjLpZNLSNRRFIe/O81o+WjISM0epuarEHuDqIFEDyoqEFtFD4gWQVDQoo0QhFARbowKNNpKi0DJRYVGqRmY5oPUBs3S1GnMcdR0/v8Z554WM44RGURne7nf+X6cc5SI8D9lBTh79/0VIBkoAHaCCIJCCxaLwqJAa40O4LFZpT9z/cpdaOFqcZZCRDhT0V4p/1i3HveIiAQNgEKAh83usNrfgp3Pj6NvyGOGI6AlceExPT4SAKX+/PnjNxMAr+GPCANEJGqhq8NlLtk53myk0FlN/0QO19a+Ul33Lp4OArRYF9SWqrmxWqb7WliRcwp7ynY8g5n0Pa+6vQBQACXX6zG0RgvU3djP4OhUMI7nBXZ6iEvPxz3QS4TyEbsykZjVG+0hgAbgu9fPvm1J1LWNhDtH+1qxSRf21IOYY9VERCm+dPQxPatQvolcS8gAgBkjgF+EOXM+OImpZmw/GrCnHcYYrUTZJrHFxBItbh4N5bH70hOHBUCFDEzTj9cfIGD4cfbWEjX7GvvmYxgj97HY/PimN+Fq7GTNgTKchh2AoMEvUxeBnKgOPF+bid96BJ+zimURgjmdzHhTO6qonOUJ2YjMLwL0vA4ThluqKT0UwBdIYqy7Ao3BrHsdrre9qKJyVHQCodgSBgS0/gzQ/eAExWntbCm4QORwE46aZjqeuXG87GTD8TukZmSRkmQPmcrk4iYGdE1JaUOGiOTlulyrfB+ekpJbyNT4BANtDupjLzNe9g6R1lBIPQOWXgD1+zmf3Bvn3ZGaYN2TnYLYzDde1/i5oze7Pi21YD8BVSdMJ0n4cQkAAAAASUVORK5CYII=',
layout_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLpZNNSFRRGIafc2fGP3Iaw58SSVyYJRHGCJUVtGldJLZoUS3atIqkdYStwkWLWhjiJghCpIWRLSKMQENFC/uhhsixn0HLTFPn3uvc+30tprlOIVH0wuEsDuf53vc73zGqyv8oDHD6+tPzQC1wAIiDKopBFMsyWAZEBPFZiIT1bUNNtBlRLrRuN6gqp65NdOs/6krvK1XVrAPgIEDP0Hxg7U/BzuzfxOv3C24QAdHNucP6ykIAjFn/8ptZFwDbyRQEAFUtyVVNfHb/qnnq+2YNIBrOWVtPK19GWZweoGpXO6GCaNa0L+QDDEDbpYc4IojCvcuHmUotgjjou9vEqltITvZiVbdRV70RFR8AKwvI0ubsDHt2VDFve0H1yMIg0co40S278WYeIctT2QjeL4Bsz5ccn4wqaTcLUDvFyvQApeUx/O/9VDaeYDXZh4qHiLcGMD8duG4GO+PjOxlQJfOhn/L6I+BMMHrzFhtiNu6nERanh1FvNc9B3qvnJnt5ZoJCWaK0LI24U6CCvzxG3aF2Pj65QcSk1wDiSUC4P5QkbITUWA+xrU1IehIVm3hrA7KaoqgoQbRmL83FYxoAfJEkwPOuo7zoPsaDcyGKyuopKZ1Dva9gQozfSQCCOAkqttXSFBnRxx07G42qcvzi4FVV3aaq+xApO1vRScvJLkJWEvW+/TbYFqHiBmZfPiNxt6PPrPedhzvjc+pLiYqiInlLgx0RVHX8BxTzXjKQ0/OBAAAAAElFTkSuQmCC',
layout_header=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGGSURBVDjLpZM7a1VREIW/fXKTIhAfVSQIIY2PWkEELe0FiZVgWkvBVhCsrLUQ/AdpxCKIhelstRCiAcFExEbRhAu5j71nLYtzvSeHXF9kYLOb2d+etWYm2eYw0QFYefTmNrAIXALOgY1JyFRVokogCQU70x1/OH3yyHlk7lw7k7DNzYevn/g/48Hqhm3XFQCXAW6sXscIy4SEJCKCCBEKIgJJvLi1zvtPO4OxBOQTAMM8ZH7h2B81b25sA9Dr55kxwPYsQM6Zz9tfsY0MViC51m/j0Q3giNQA5A7A05W1f3ZfIfYDEsDyvZf0JWRYu3+Fj192Jz5eWjiKFQBUNaCmfetlLpyd53uvjJPnZmda51e4tAC1rm4/yDZ7g9L6MYfJ0R44qTSANKpgMMj0chD9/FcPXIb7PKChT5rs6al0EODSAFRkIAE8f7XVSuzuDSdXMOpnByCkLWDp7eOrB9z+rQT7RzOJoWfLd9dP2b6IdFwqYGEFVgEHlqg31wZ1oXoHkA67zj8BVEcprN9nEQAAAAAASUVORK5CYII=',
layout_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAImSURBVDjLpZPfa1JhGMcHXfZ/eBUVdNOVBqGDgwaSy1HLg6R5IUVyYCi5li6Wv+dRysAQoiG4ple1aMyRYj/wF8Ugl8PVTSCM0GI0K/DbeR9RJ4xi9MLnnPf9Pg+f877ncMYAjP0PdOGjVZtEWKIsAT5a6fKRCvhwGUbpfiVagVEsgQ+VWqZIseTPbMK/XMN+QRyHHN6lDyOCTbZ6WPg6IP4X2DAGXneGArHS7gty9V0iv3UwfcHknVx3IDCEy79YGP/Hk/fvQO9aHx7hcqjUPew7mLi1NhRMBYoU6mbXoJ5ZBedcpfX2l/aBUK/zxVBwyfeWwjPTzzGXfI/TwspAsNP6MUJfcN6+MhRc9Lyh8NT1p7j5qAjOKiIYDCKRSMDr86H8roatz034/QHKWG3qhgi5XH60t4P5VyQ4dnUZvHAXoiii2Wyi0Wggl8shtBDGvfsP8LFep6xQKFCPWq329XYwXyDB8QseCNN2VDc24PF4oNFoYDabEYlE4HA4aM4yVmMSQRDaJJh05+krnOCsmHXNkaDT6UA1Po5sNotqtYp8Po90Ok0ZqzGB2+3eI8HE7ZfbTJBKpWCz2UjgcrmgVCqhUqmg1WoJNmcZqzGBxWJp9QQz6ws6Z/aZ+trjb+d0BngDYSwmnyCTySAWi5HUbrfTnGWLyaWuLxD6LR2nNvJrymSyIwqF4iTHcZ9MJtOu1Wrdk/ip1+sNEmel+XeWsRrrYb1/AB4L/elcpleiAAAAAElFTkSuQmCC',
layout_sidebar=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFxSURBVDjLpZO/S1ZRGMc/5/aqIFhYKCKBOGTttYT1JziJsw26uQit/gHOOgQR7tHi0tjmWIGDPyIoGgQd1BCqe+8532/Dfd/7evEtFR84nOE853O+z/c5T7DNTaIF8GL98zIwATwDHoONCchkWSALIAklTvta/vrw/u0nyLycfRSwzfzap9e+Zqy+3bHtSgHwHODN1nEt7X+FLUzfZe/HaV6XgDzWOXwwOgBACL0v7x/mAPz+U/bXANuDnVe/HOVXMs8phS5AbnWkXTWUxHlAAFh6v4iUkM2rmQ2+HfzseXly/A5WAiCrABWtKApGxu5R5GWdPDTY31h1CbEBqDwviwhAnheNF8tkytTsixS7gFArKNug8nITY3HOg3bXY4ztPTWS+25d7KkduwBFGQgpJrY/7mGrTjz7VfRWoGqIWgBJ+g5MvlvYvOD2P0uwT7o/MWlzbuXDlO2nSMNSBAsrYUVwwhLV5NqgM8h2AcJNx/kvz3X5EBChVawAAAAASUVORK5CYII=',
lightbulb=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKgSURBVDjLlZLrS1NxGMd90ZvovdEfEBEUEhZIb0xMjdyLIuyGkiHGUFKydFKKJiRegjIyFJRwojMxzfJSaVOYeTfxtpSNuZ1tXnY2z27nsss5334uWloG9uLD7/A7z/fzPPx4IgBE7ISl3qWyelUvu9JIueZqeOdUmcCMFDgcQ3fntjSK0j/rwx+csesIZ3jbL1j6EbCPIej5DpE3QRIoBJ3LEFb74BjIxkbXVYNdrTixS8Ca3h/y6pSTfloD0UcRjCS8BJGbRdA7QRgjd1pIfhruyeewKOMdm+rCw2GBV1tXKZh7SIEVoqAjpwVS0AlIvhBSkCGyeQRcPYDogO1DNixvrveFBa6ZCkuAmSe1OtJpFVLATkJboWCIAE3+GYngI6ENgnUK+hcxfFiw9fWRT+RWEWTHEeRmyPhaMvYCgu5ZEpgkbzCCgPszBNsr8NY8iF4Ky5WnpLDArs41+zYnSPdF8OYi0qEcTHc6mF45mJ4M2Ftl4C1lYPU34KerwFNTWKmO/j2BfbiwghmvJuPawZsUsNVHgTPlEx6ANcjJeR9r5QfhWUqEJOlhbc+FoV42FBY4R0sPbPbKlz2LLeQB9aCbYkJhzpIFlkoDZ8zDRk0kRHYYrm8d0JYeEyyduUd37QH9pTBqvSOV9iy0wtmZ+VNAOm+HOeM92JtlYDQN0JYcD1BtmTf/WqRtbJ/yTxtUt9fXGhPBq5MhriVBtMYhoLkMQ1Ek5sqi3eb2O4l7buIvhlRPkmsfZ/ibax+iruosnpacQUFOOq7Fn5TUypJz/1zlnRQr5JSypRVKZRvq6htR/ewlriTH03vV7ilQ5NwaHRgchM1GY3p6Bq+bmpEii9XtWzCgqkhLuXSBTUg4L8XFxUoXk2K57obirH0L/ocfNQ8V8wE+uE0AAAAASUVORK5CYII=',
lightbulb_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALZSURBVDjLfZJrSJNRHMb90Jfoe+GnPkVBF5GKJIKS0gqFbnTX0swaSkqWTdJIBXXqygtMFPRDC29lmuYlp6mgeTfzthTHnHObuoub0/fitvd9Ou+KpWYdePif97z/5/d/OBwvAF7rRWk/hlCq8gZqqkS7PJLH2AZSWWvXM4ul/dHIUqc4eXO/Z0Nrarxp9YcmVtcEp7kHrpUf4JhZ8KwWLtsk2JlGWFqisVBzTW1uEx/YAKBm63asTsv7HaZOcGtaIg0xT4Cjh+Fa7SPqIWdK8A4T7P250MlPWxbbEnZ5AKvKwkx2rp406MGx06TqwLtsAL/mFu+yEtgonMv1AGeB8VM0dJU3Gj2A5SGJzmkdJb3TZNIMeKeZmJbcRrecJvJPQwCfiSrA6gegyvdjPIClr4lrHD0DF9ULFz1E4itJ7DG47MPE0E/uoAtOuwKssQCMPg7cqhaTmb68B2Bui51bW+wj08fBzCWRCemw1obB2iCCtT4C5rJgMLpUUKqbcJiywGgHMJV9+E8Cc0eCxNqbTeKawcyKYSzyAT0bT/QUlFpE6hMY0rdjZSIQPK+C/l0s1EXB7R6ArTt522KDaHJlvJRcoAqmN35uM62LAqUNBa2Jw0LeTnBUB5a/VUGZvJfVVcfu2fAOTF8SfOarQkwrY2WwVUf+ApDJgpnWPEZN8WXEF9zGg9cXEJLmz5xPOJS0ASDI2Bx/RF0ePm8oCQTTFgTOcBac3h+1lZeQ8j4c9eMyfDcokNMShes5B3Es2lvqtflptpenBclSIhxvZc9RmHUc0pdHcSc9ALVjuahV5kNY0tZI5LQ+FADMXwBBL8Qirby0DHJ5BQqLShCc6IvG8WKsX3WjMgGALQHimLvdLa2tMBpNGBwcwomY3chW3EOGIsxtzmgO+3+ClnJJ6NWL56iAgDO8v/9J/tSt/c4r0n14pbjvnixU4XvLO/iXSLOEyC7E/l0lwvlPxOcI8qKVSZYAAAAASUVORK5CYII=',
lightbulb_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALrSURBVDjLbZJZSFRhGIYNRyK6iC7aLiK8qG4MifZuylxaVOymjZQKDYbEJE1nskITEpeCFAy98KIJl8LUyn0Hzd3EbVKcxlnOuMyZcUads4xzznn7Z4JJzQMv3+E/3/t87/n4fQD4rBdjqIlmNOV1zHSJYXk0n7MPZvK27lSrtSNhdKlLkbG53/vC6qoPsNovDTzVAJelF8LqL4icHhJvgGCfAj9bD2tLPBaqb2ot7YqADQBG/22nY0Y1sEZ3QXQaiHTEPAmRHYHg6CfqJWdqSGs0VgbegVIFWxfblfu8AIe6KIc31pIGE0R+hlQKkmAHJKdHkmAjsDG4lmsB0Qrz93hQn27XewHLw9mUyzZGemfIpFlILgsxLXmMHrlo8k1HAI1EFeBNg9AUnOW8gKUfz50iOwuB6YPADpP4ahJ7HMLKCDEMkB10w7XSDN78HpwpCaLDgKmc45IXYGlPNDoX+8n0CXDGF2RCFmxf78NWJ4etNhaWsghwVCYYzR2s0bngDIOYzjvxL4GlU5lt68sjcS3g9AqYiwPB6lOInoLRyklNxlzWDqxOhkGSNDB9ToS2OKLDC7D3ZMgW6+RTqxOlZIEa0B/Oesws9QiMIQasLgkL+XshMp1Y/lkJdcZRnqpKPLzhHtBtysD5ymh6dbwM9qqHfwFkstvM6p5gOuM0eu4eQuslGVqubrc0hfoqNwDcMjelnNSWP5ifKwkD1x4Oce4yRFMQ9PnnMJp8BlzdW0jqRjAVSRiMPyY0B8sSfDZfzY7y1+GFr2LXPhamoSj3PN6kn0LNtV1giRkFkYBiN5DlDzr3IppCfLX/Adx6qZAbVKVlUKkqUFRcQmL7QRquwvrHlr7f/TvSlgDF43s9La2tMJtpDA0Noz5qDxwlsQAx8ak+WCLSy31B9kBtCWgpz465cf0KExoaIgUFXZCyowJc/XFHxDmFP8zP/PA7bhvaImVCY6hv2paArdR966CSTJx1xybV6Da7z/8A2VoHSyUsj6sAAAAASUVORK5CYII=',
lightbulb_off=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJOSURBVDjLlZLdTxJQGMa96K4/oPUHdFfrpntyZrXsoq25tlbroi6qi7ZuYsuZ0UXRWiv72NS0gjIgDQ1LS0wkwU/UVEREUkEIBBFE+V48ve/ZICza7OLZOTt739/znHPeEgAlhZpyB8+MLa58HHL63H2zy4muycVku8UZahl2TNJ688/6wsbd31yBLps3BNdqFCvrMYRjSURIvOdzzdAcmozWhTaLc+8WADXvHHb6RhYCEdEU2kiIJu/aBtwEywE3k2lQKjz8NB7Sjs7vygPMDu9ddogmUliNxsWaSGfwM5sViqcy+BHeFCl4r6YkzwzTnXlA9/SSh924md25qFDszMnmfGuga4pEd3QjiTxAN/49xY0c10MgjsuOuSssBdfh8IdBSUG1AibTDmbzAHrhZab6IzHQq6N3xo3+LyqY+1phMmig/9AISolm8yyMdo9IcKtt6HcC+h653uoScTsJ0K65jw5yYrWOOISrol6Kht4pcUV+g0efJwx5ADXtUA3a7aMLflHQoa0VzfTSoHMBUClqwL9EM4Lrb01JOt+zZQ7ob/c/N1qDDGEHBugKxO6mOS+qWswZRb/t9F+DxCLHAzQovsfdEyAYXn6d4cHBa7r7NXU/brwbiCpNtsNFJzEnaqp4KjufblDU4XbtJVTJL+BqjQynyvZl6e8P/nOUC1UtvehWNr+BUqlGXX0T7j14gpMVZcFitUUB0ivnBvQ9PQgEgrBYxvBC8QqVxyXz2wboVfKzlSeOxsrLD2VLSyXZY0ck8feN1Ze3Dfgf/QJBCig+4GhFlwAAAABJRU5ErkJggg==',
lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIMSURBVDjLpVNLaxNRFP6STmaKdFqrgYKWlGLSgoiKCwsKVnFRtBsVUSTNyj/gxv4Bl678AyKCoCulgmtd+W7romgzKT4QMW1G+5hMpnPnnuuZm6ZNawoVBw7n3pn5vvP4zkkopfA/j9F8cafO3FekCjGpIgKIvayftTXOkr71jkz2/UXA4HxXfz72gIx/lBsWSfiVtwiWHK8B3kRQeX/6lmnnkuDAwn0MJSKQEFChQCp9CcHixxgsGWw3B01uRKfx9t1HIP1POpoSdUulLyD0vqO26IAkDW7tgSZYeHPqcmpXxkTChKzOaAKSEdo6jnEWVY5ehFxdHs2cn55rScDR73H6DKyyRWs1R0haGdR+z8YZ3MyMTj9rpUKi/PLkUJuZfmX3nkNYmQBxzYprpyCA2XMRrvNAcdfDhgKkm6ttKTdW6jH4w4RpD/ALAaNzhH2kSwALoSJCd9+VhIqEVVeD4C1MclaOT0Ke0Cowq+X9eLHapLH23f1XreDzI27LfqT2HIfvzsRAyLB2N1coXV8vodUkfn16+HnnvrPDhrmXsxBY+fmOwcVlJh/IFebK207iuqSShg0rjer8B9TcWY7q38nmnRstm7g1gy9PDk2129mjinjy3OIvJjvI4PJ2u7CJgMEdUMmVuA9ShLez14rj/7RMDHzNAzTP/gCDvR2to968NSs9HBxqvu/E/gBCSoxk53STJQAAAABJRU5ErkJggg==',
lightning_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ8SURBVDjLpVLfS1NhGPYP8D6im8DKwrYhUaa1tEyyIXlRWtFFJlZQERIGUZKGI5dO7ZeUEl1YaKUYkyU1J0hE/ppzuOnUDbdpbp7Nue2crZ2J9fSeQ4LhdtUHLx/fx/c87/M+z5cEIOl/6p/DsjGnzWfIhnf0CJjhw2AGD2HpWxY8Xw/CPXAAi/378aNvHxY+p7viEhD416q/FTFfC2JLL8AvPkd0/gl+OhoRsdXDN1gsgLm4CghcE5opw6qvFeHpfHDm4wgZsxEcykLEroZ/tFQAryUcwTsij8WYZ4i6boGz5IE1HkWQxojY6xAwlZN0OVyfZClxCbzD8jMBywXEvC0IT50AazqG4Kgc3ORNcNYqeAYUcGllioQmklnhiKsavLsR3EQuQmPZCAxmitK9388RWFqRMAUCZyyPFSLGvKSOCoTG8xAcycEKOR+eeSSAfzs1e3lHdxo/17WHt79P5W3tO/nZNymMSEAxMezsbepO8y+Q484Gce6IrQ5hqwqsWUmkVQgaKhEYvosFbT4IHJl+vV30I4kyDlLGPGXMU8Y8Oc3P98p4zvoQvl4ZlvWkyliNro4iVDQX40pjIc4rc9iTd6SVm/7Bejl7JAMrhnKwEzUEvo/2tlN40HkJWkszTG4dmvqu4WyTBBnXt6rjEjg+ponSPf1FmPsgxUVV7prG/BiaqacQllp/GU36qwJBNB543KMvhFtXAHvHLr/t7Y4tBffS0Wt5hY2rZ6JZINgETnZ0SzDXmQZyum79PvPGtmi9rhS1uhIRXPulJL4CimmSYmIInLzxnh4qT6t3o0FXJnYWduG8yQP7u9SMRB+GHquoWEH2310l3P8B4M3c7jDaDNsAAAAASUVORK5CYII=',
lightning_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ7SURBVDjLpVNLSFRRGP7u3Dt3JnMsdcAeNjrgAyoqWiQUZBYi6aYiFbEWbdsZJNEqaBO0bxdkENRKMXobmQVa4SPBfMyoPdQp9To6zfvec07/vTo4TbPrwMd/zn/v9/2vcyQhBP5nKekHbbjmvuDioikqOAc4Wba+t5DaM/69uG6o5B8BIrdu87aaFmDmj2wTBkN0+RPia75wivyXwPLgiZuqq9wGCqxrnRC6Aa7rEEkddvc5xFfHTTIjsis9qG0zOm93bj8IFp2wogl9HXb3GSTDc4it+sAZr8zsgSWw9PH4eXuOR4WkgkXGLAHODMi5hymLCEWfAkuEGjz1I9NZBSj6PUqfiBGCsVGzAZvDg1hw0szgqqdh5Gm2KUi/+o9Vyap7wFV8GsnlbnCqWVDtPB6HWnQWmu+BoK4nUxPgVnMtrJVf8Bcp9KFbdVWQQ4eSV0fWsEoADUIYHPklTZIwdMf6NDjCS0OUlS/KdXbUmgKpOsILbxNpM7ZsvrfZEZ99RG3ZDXvBESy8eQ1tcBKJxRCYLDTiNlZewi0p20389vhAb96uU9WKWkhZ6JjreQZjUUZFTSMc3n2Ijb7El3evWHBivM2WrTGURbXicCOyNArN34VA3+e1ciI7p3shdbQgZ6YTpSWFspBEm5JJ/tq1f9jpKkM4MICoNrVCYnsR5QHnDi9Qf2XzDdzYCZlLpUoGORfCdigW9IPpydtlLVPtpn+mQ5mPjjwp3tp9GYnYT0TJ9zskUy+wYMtI/QMRFwmuFNlcOQVFd8f6+4xAfAtCsh3BFQn+eYnajjuwXt4G/A8rq9LP6XjfvOfai1p5tuekwsn+eF4rXzf9fwCYD6X48OnVRgAAAABJRU5ErkJggg==',
lightning_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ1SURBVDjLpVLdS9NhFH724WbTWcpKiTUxWyJKRTdGURaSUt5URDcW3XYZ9AER1UXhTQT9CREE3URi1EUkaGhq5ZSgDzc1tWmaGzrd5m/v+55zurBy6u468MD54Dzn0yYi+B9xZhvxwWOPheWCiECYARYIrejCvKoTT/qbQuUbCISlZXNFC4QFIIYQrcIQ0rEPsBKR5N/kNQSxgaN3Xd6gHQzo+HOINmCtIUojz3cG1sJXWIkI+ZtC3uyi9tXqfD1/y15Q+hvEEESvIM93CioZxfJCBExctX4HdgCYe3/kbJ4n4ILNBUp9hhgCk4GjcD9Yp2AthEGZxebAyaHRnATC/MjlDUJMCmLMn5kN7O4AlueHoZLRa4HmoVe5rmCb7T1U53D5+rz+E1CxdrDSEK3BlgVX6WnEI09EiNXDxQJ3yihc8aQzDmIIcSJ4fqTUKSTtLu9uCGk4i5ogZCCGAAHEMIrLz9nEaHfm00v4fdW4MxVy3y7gtEPTQQBwCrM7Od2VyboxhBkP5ovcwgqKDZQhbC8JorqsDktWCpejffmWUbNvAdhyfeLEiz2d9xOe+uM1F0HCICYwBDOJKIoLtqF37A0GJrvYMqrEmWsxQlxvGQUSxkQ8As0GhjU0aSxmlrBvx2Ekddre/707voFgvK12MN+7C1YsDEMGpUUBGCaQMH4mJlFSWIbQjx70jHbPKYPGNSOMt9UWQuxLQoyrsTSWlRLFCpZRqPTV2A7sbMTHyXfoCL+eUQYN063yxbmu9X4m80uIK59dGktmx2rv2VM2+yZPZ7hjSjEapltleCVJ5B9GnlbVZdvZqLiFVOCmQ7beQEW2/zeRQcHUmJPKOgAAAABJRU5ErkJggg==',
link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADpSURBVCjPY/jPgB8y0EmBHXdWaeu7ef9rHuaY50jU3J33v/VdVqkdN1SBEZtP18T/L/7f/X/wf+O96kM3f9z9f+T/xP8+XUZsYAWGfsUfrr6L2Ob9J/X/pP+V/1P/e/+J2LbiYfEHQz+ICV1N3yen+3PZf977/9z/Q//X/rf/7M81Ob3pu1EXWIFuZvr7aSVBOx1/uf0PBEK3/46/gnZOK0l/r5sJVqCp6Xu99/2qt+v+T/9f+L8CSK77v+pt73vf65qaYAVqzPYGXvdTvmR/z/4ZHhfunP0p+3vKF6/79gZqzPQLSYoUAABKPQ+kpVV/igAAAABJRU5ErkJggg==',
link_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHMSURBVDjL3VK9S0JxFBWChvoHinap4UG6RIsihYMfiTboUFGhPVIbxAJFG5TEKM1U1CWENjEUigiyHBRnicrCwaIlXPqggldRnd6VkNqMti4cfvede875Xd57AgCCv0DwjwIkEkmn2Wxe8Pl8t8lkEm63+8pqtQ7w6OL7GnE0Iw1pfwSIxeJ2lUq1Eg6HUa/XUavVUCgU4PF4LlwuV7FarT4TVyqVQBrSkqcZIBKJRux2+32lUrk1GAx7SqXyzWQyIRKJwOl0gnriaJZKpa5IS57vG6x4vV4uGo2yGo2mQyqVPubzeZTLZRSLRWQyGRBHM9KQljzNAIZhZlmWvYvH4/M6nW5fJpO9yuVyaLXaBqgnjmakIS15mgF9fKnV6vNgMHiXTqdvstksEokEbDYbHA5How9t+mCLjX3MrGlg8Mreh+eYcDNAKBS28Sv2KxSKS6PR+GSxWDgeL3q9foLH0OzixItnawq7pzEcXecQOjBDH2IwYOkOtPStx/3D3PbJOrbPIqAKHJoQOmQpgGspQOUSYe90A99r5zhGAa39bYPWHm41Nw1/brJh9u9P/m4DXrg0GuhFMGds3EwnPbf8Dr5Clnk80Npf5zLxn1E7ljyteCJyAAAAAElFTkSuQmCC',
link_break=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIjSURBVDjLY/j//z8DJZiBqgb4dN1jDpnycL5Xx111z7a7/JVrnn8Aibs13DKrXv38t0/b3XkEXeDVdjetZOWzJx7Nd4y82+5McKm9pVm56tnPgK67a4n2glvjraicRU/vudTc5AzsurcmdOKDg3i9YGdnx52VlVXa2tr6bt68ef9ramoeJqXnXwHJ5eTkSAD5d0HiIHmQOpB6uAFGRkZsPj4+XRMnTvz/4sWL/3fv3v1/8ODB/42NjfdACqqrqw/dvHnzB0j8yJEj/0HqQOpB+sAGGBoa+hUXF3+4evXqu4iIiG3e3t5/UlNT/0+aNCkPpCA/P/8/iA8SB8mvWLHiIUg9SB/MBV1NTU3fJ0+enA5U+Mne3p5j7969HOfOneMAKTh06BDH2rVrOYDiakD5JyB1IPUgfWADdHV1M9PT099PmzatJCgoaKejo+MvNze3/4GBgf9BCoC0PogPEgfJg9SB1IP0gQ3QBAJfX9/rvb2971etWvV23bp1/6dPn/6/sLAQbEBFRQWYDxIHyYPUgdSD9IENUFNTYwY6z8DLy+t+SkrKl+zs7O9A/DM8PDwOpCAhOfc6kP8JJA6SB6kDqQfpw5kOPKtvHHTIu7JGL/wMZ0DzrXvaIaejiM4LTgVX1yZOuvdTN+yMplHk+QmaIaeNAhpuPlEPPJFG0ACr9Ivz4ife+60TesYMxA9tu/UBqJFfPeCEulHk2fmqfseZqZ4bAf27e9aCOQHGAAAAAElFTkSuQmCC',
link_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHqSURBVDjL3VJBaJJhGDZ+F7Sduo0gYoeEDkJ6iV00t/Yf1H+il99TMPrFUjuY1XQKpeAOggMV9DQPnUTQTg0Xc6B5GSM7WXj4N6QIKdKgNd0yn753B1m3H7r1wQMvz/s8z/fA96kAqP4Fqv8owGAwzHg8nifxeLyXz+cRiUQ6Pp/vFsMsm2XiaEca0v4VoNfrL1qt1kQqlUK324Usy6jVaohGowfhcLjebreHxDUaDZCGtOSZBOh0uuVAIPC91Wr1nE7nlsViGblcLqTTaYRCIdBMHO0KhUKHtOQ53yARi8UGmUzGbbPZpo1G449qtYpms4l6vY5SqQTiaEca0pJnEqDVah+43e5+Npt97HA4tk0m0ynP87Db7WegmTjakYa05JkE3GBHEIQPyWSyXywWv5XLZeRyOfj9fgSDwbO5su7Brnjt987CFF7y06cvTJc2JgEajYZjFW+azeZDSZKOvF7vgOFEFMW7DIvZFX74LjCPwaskxu8r+Fl4hH2vdvR6Uf1Q0Vtv3+HkY2ZGWgBWLwPrc/iauA3GHygK2FlQj8dvyzh/+s9mQbyyBkvcx6PNewAzDZ+q0GPo3OfA+E+KAt6IV57vSdd/fV6dw5fQFGTpAqqCelRZ4tYU//mGeDXIbjyk2tSIzMT/ASTzlHJFEjXFAAAAAElFTkSuQmCC',
link_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJRSURBVDjL3VNNaJIBGBaCDnXqVlEQO0ixrE1a26G2uYU0tyVaA2lsHtS5HzuYhTkl0CHZmpT/0coO0UVS6AdrkCPFkC0am+ambea2GjOYuubmN9vG0/eNLl1s0K0XHnh53h8eXp6XBoD2L6D9Jwuqq6v3dnd3X9fr9Rmn0wmNRjMnk8kqSewn8wTFUTWqh+r9YwGTydzd1NTUbzKZkEqlkEgk4Pf7odVqv6jV6kA8Hl+nuGAwCNfgVcSeCjD9XI/xR2xM2ErbaeXl5RcUCsVyNBrNCAQCb2Nj46ZEIoHZbIZKpQKVU5xVWzu+OKzEcvgVkFtANvwMoYHzKUpBv06nIywWi5TL5e6pqanJ+Xw+jI2NIRAIwO12Q9lZQWSiNwuFry+w+O4O8hEPNmeDiDzuIGgMBqNLKpVm7Xb7NT6fP8RisX6y2WzweLxtKDrOIB3RYCsfRD4hQ3r0CqaeiBAebFsaNfPotGNkNDc3TxmNxqzL5Up7PB44HA7I5XLYDEJkIipsESGszQhBzLdiZbIXM47apY/Gc2XbR6TT6btI2WUcDicpFotXe3p6CBKFu3KmdfGD8vdwO4i5y/jxSY1pa91qxFBxuqgPvH0HLk6+URS28gEQSRHW59uwTCqZttXnYwNVR4oa6WHXYVHU24uJ1/fwbaQFa8lWpMdv4LOV9T1mrCr5qxNdt+uBlVnEHlzCcO9BvL/fAL/u1ELYUHl8R1buk5RuFEJWFEZMGNGz4BIfyg2pTpzc8S+0nN1H3BIe3fAZGjbeGuv8L5WMkmK/8AtkdLda3u0iOQAAAABJRU5ErkJggg==',
link_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJMSURBVDjL3ZJfaJJRGMaFYBdF910XSSukaRdFF7Mxkpozc0UI1UXksNQ7V0O0QocshElqTYL4ilhjzE0oahT4ZToHFWUb5Za5zyWNsk3dYNvxX/R0zgJZsHXTXQdezuE9z/N73/dwRABE/xKi/wjQ2Ni4xWAwXHI6nQWO42Cz2TImk2k/jW30LLAcu2Mapv0DIJPJ6lpbW10ejwfZbBaCICASicBut6etVms0mUyWWC4Wi4FpmJZ5agCpVHrMbDYvJhKJglarHVEqlT/a29vh9XphsVjg7jqP8APtzws6bWhgYCDDtMyztgOXw+Eo+nw+vVqt3iyXy5d4nkc8HsdoNITEUz3yqQAmH53yMA3TMk8NIJFILur1+oXe3t6Otra2Z01NTRWFQgGNRoNbXScwP9WH6vI0JgKa6jBn8zAt89QA9XSpVKqpnp6ehcHBwXwwGITf74fDZsJ44CRK849R+upGQXiC8N0zleNq1UfmqQHEYvEm2nZDS0vLjE6nWzYajUWTyVgO31GP5ZJDKH1xIHp9Hyr5IcS5ZvTbD5mZ56//4H3/0SMzvBXlufsg0+cQ7ZaimOnEYnoEr28enHzlObB1Q8D4vcN177jmNwv04YqfO7DySfs7UmfpKDeQiXgx5pJd3RDw9rbcmIm6Uc76V6uvpE7TEWSrO0kbQL49RLRL8j1i37N7XQBtb5bkBJQLPEpz9A3mAmtiGNWlD5h92YfnVnFgXQBtLxfrbiCjzr2EViK0EnlxrZ6Er+wi1ER4y07Cd+4gocvbR38Bt2OvTVFKHBsAAAAASUVORK5CYII=',
link_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIhSURBVDjLY/j//z8DJZhhGBlgZ2fHnZWVVdra2vpu3rx5/2tqah7m5OSYA7EEkH0XJAaSA6kBqUUxwMjIiM3Hx6dr4sSJ/1+8ePH/7t27/w8ePPi/sbHxXnV19aGbN2/+AIkdOXLkP0gNSC1ID9wAQ0NDv+Li4g9Xr159FxERsc3b2/tPamrq/0mTJv2vrKz8D2KDxEByK1aseAhSC9KD7IKupqam75MnT0739/fnsre3/7x3797/586d+3/o0KH/a9eu/Z8xx+Jf6nzD/yA1ILUgPXADdHV1M9PT099PmzatJCgoaKejo+MvNze3/4GBgWAMYifMMPrfuDnqf/gMjf8gtSA9cAM0gcDX1/d6b2/v+1WrVr1dt27d//yltv9zF1n8T19g8j9pruH/mvWh/1ednvi/ZLX/f9c+iX+a2hpacAPU1NSYgc428PLyup+SkvIlOzv7e/Zi8/8bzk37v/bsFLDGFacn/J+wp+T/wuNd/zOWuv03bWf/rdvMyIgzfpOB/gVp7tuV/79zR/b/1m1p/xs2J/5v2pr+f8ah5v8xC2z+q9Yz/MRpQPRszf8rT034v/RE7/+Fx7r+zzvaATQk6//0Q03/05Z6/FesZbguXcnAidOAwOmKfz0nSv917hf9a93N/zduvtX/aQcb/ictdvsvX8twUbKSgZ2kpKzdzPg6fqHzf/lqhjNAzWwk5wWgk1/LVTP/F61kYEEWBwAP7or0z//OfQAAAABJRU5ErkJggg==',
lock=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ/SURBVDjLbVJBaxNBGH2bpEkTmxi1NTRKTZtoQUHEWz0Igj2I4kG9eVNQhEBO7bEHc+yv8JAiHnr2B4gFqVrQRhObljQolBSTJqZJdnZmfbNr2rU68DEz33zfm/fejGHbNrxjaWlpRCk1J6WcYZxkgPGTsWJZ1mIul/vlrTe8AIVC4Qqbl5PJ5GQsFoPP5wP36PV6qNfr2OIg0L35+fm1fwDYPMLDj+l0OmOaJmq1Gjqdjr4dgUAAiUTCqSsWixvMXV5YWOjqvW+AxOSz8fHxjBAC5XJ5s91up7gO6tDrUqn0QwOTXYZSsoO+wGDB5EwkEkGlUgGb7mSz2apHajWfz9+sVqvFVCrl1P4PYExr5m16vYUjQ+c0O11DtmN/ebD95pG9UpnGzl7Y0Xz30ir8toAtLdiWG0JIvFi76piaGG7g9plVTD/5YLgMCPLg/g0YtMTwhznfApRBfsP6kAYJSKuN57Md5oXTsvHy7aEEfZMutHZfIRAahWGMsHAICMeZVsD+HmTrG8zudyhrH+HJLGyz7wEgRSh9k4nm+nvqPIb4xWuovV5k/2lMXJ9F8+s6ARqIpk6QsIQtTC+AcGTYpBqfvgBfcJTuKMi+xKfdMCZgIp6eRK8TYu2+w2oA4PwDm+5qVK218XmNLN7xxILqKfS7pGqTWekLmuVtV65STs8hA73RqJQQP5+CP3KKACamHj7FlGBDawfH00kEW0MuA8o9AmA6qMrSHqwTIAoM08hAkHkN0ES3UYfotBGdiNFu5cr2AmgJobOPET7nhxEMuU/o40soSjO7iHbbVNgnUen6pY0/AOCTbC7PuV44H0f8Cetg5g9zP5aU7loDcfwGcrKyzYdvwUUAAAAASUVORK5CYII=',
lock_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALKSURBVDjLbVJNSFRRFP7em+f8ZTKJOswkNTZOmpFKBaWEFDVBFC7KRdCqgiKccBFJKxe6CWwXtWhRtNCihQURRAUliEJYTeoMzOQk05D9jGTjOH/v73bum9Tp58Lh3Xfu+b57vu8egTGG0jU8PFyu6/plTdPaKCopQPGDYkJV1cGenp7l0nqhlGBoaKiVwCNut7uuoqICoiiC/pHP55FMJjFHi4iO9/b2Bv8hIHA5Hb7zer31siwjkUggk8nw2yFJEpxOp1EXDodnKdfc19eX4//iChMlL7hcrnpFURCNRj+m02kP7c08+D4SicxzYuqunqQEVnDSyoaSbXa7HbFYDAQ6FggE4iVS4wMDA4fi8XjY4/EYtf8jqOaa6Ta+n8Nfi+d4d7yGuq3+w4NPY2fYRKwB31I2Q3Pn9kmYmAKmqWBqMRRFw93gTsNUp3URRzdOouHcW6HYAZF0nTgAgSwRTDb6HgF0gfqz8kMySIGmptHvz1BeMSCz98fXJPCbeKG68ACSpQqCUE6FZYDNQWkdyKagLX2AnPsMXc3CVhcAkwslBNQidH6TjJ+hN6RzHRxN7Ug8HSR8DTZ1+PFkMobx5TIsEU4ZHURLTkTTGoFiyGDUqqNhK0RzFbmjQytomF6wYSocw7Tdgf27dqO20oeXoUeYmPmCewH3NWMOGLnLmGZoXZwJUhevaUJU6HkdhVwBD6ciaG1sgSZqaHH5oQkK9u5o59DuVQKulZEEh88DR6OPCGRsOXUeXd2H8T2VRBn50rntoqH70sFb8NY08621KEGRyQOdDOIehGCyr6cjMlIyU16l2VhAaH4c7+fHcMV/B1efnYbVZOHQ/CoBl2CpPQvbZhMEs6X4hCK9hK7h5NcNeDH1HO3N+/B4+ibM9PpjwVEOvWEMUvT2HsaNNLwwBkf5HerqdyRrwitZQp4JsNKTdwhZ9F9PCb8A2smdT0Nb9SIAAAAASUVORK5CYII=',
lock_break=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKVSURBVDjLZZJPaFRXFIe/92aSyYyTZGL9lygxcYwBi1hcqQtRIYoobtpNaVcVLIHArIKb4EJEwazEveAiQSq4EASzcCOiICqCDjHRWFQsmAYyZjIzzrv3nNPFs/EZDxzu3ZyP3/nuDcyMZE1OTuZVdVRE9onIWhHBOdflnNtdKpWWWVVBEjAxMfGTqt7s6enp7+joIAxDVJWZmRnm5+dr3vtdpbtnRgGXmbISQDoxnBeRG8VisT+KIqanp6nVaogIZsZAunH1yNTYXWADcOL/uRWAiAx3d3dvd84xOzv7xnt/WET+ERF+vXdxuKu+MA68AjZlpmz5O4D3fl8ul2Nubg7n3ImRkZG3AM2jwXHgEnD1ytD45Xw+f3vh/HkbGxs7uBqwPgxDqtUq3vu/E54+AL9f2Hv2VrC83Ozs7KRerz/6RuK7+3/Yw7lBPn7KIiKc/PExKXOYeMzH7Zxw7dkeVJWNbYsc3/yYwdNPgziBGb/8fIiAkCCVJeAYaADpNsBAHOKrnBuqgToAXl9/8NWBiQcMv/AX6cw6giAP2gLZAphC/ROy9Iqo8QH1dbL9I1jUTAC8BzWQiEr5CWG4hsLO/by/M47pBnoPDFF5WSZqLNLe14WZYC5KAhyYYeIoDO4gbF0HgSJN4flCll4iCsV+PtcymK+D6QogBDDnMBMQx+KLZ1TKjyD06Gel2WiCRVRmpqnMvovXVcWcSyRwLqZKRGGgj1TuBwgjtv32J9uch6WPdBZ7aF1qiROorQZEoIr6iEq5TCrXDm15SLeCeliq0Fj8F1er0t7bgaHx2kmAmZDZcors1hRBayZ+wrAFVCBq0N6oIq6JqcS+vE8AoiZvbo7GLrzHvPvSfuVEJP5YIvFdBYD/ALt6pUq4OEUGAAAAAElFTkSuQmCC',
lock_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALBSURBVDjLbVNLSFRRGP7uw3lpw5TaNCZ5R6eEhIhahBFRoIsetEgJImhRUBQTUpCLFhK5dN8ycDESLSqCULLoSVZYDZYDTo7TOCTJTOk4zeu+Tv+5OjZWB37uOf/5/+9833fOFRhjqBxDQ0M1pmleNQyjnWIDBSh+Uozpuj7Q09Pzq7JeqAQIhUI7qfluQ0OD3+12QxRF0BrFYhGpVApxGgR0vLe3N/wPADXX0ObHlpaWgKqqSCaTyOVy/HTIsgyv12vVRSKRacrt6OvrK/C1WEai5AWfzxfQNA3RaHQmm80qNLfx4POpqak5DkzsAiQlWO6TyxNKtrtcLsRiMVDT0WAwmKiQmujv7+9IJBIRRVGs2v8B1HPNdBqfx/HX4DnOjtcQ2/o1Hsy+OsPGYq2YzzgtzcfaxiExDczQwfTl0DQDg+FdlqlexwKObB5H67kPwjIDAunuOgiBLBEkJ30PAaZA/Bx8kwzSYOhZ3OjMUV6zWqZvv/4jgZ/EC/X0Hcj2OghCDRVWAU4PpU0gn4Gx9AVq4RtMPQ+nPwimlioAiCJMfpKKxcn3pLManu17kRwZoP6N2LK/E/H7z5GemEExnYFc/xZ2zxoAzZLBiKqndRtEWx25Y8IoGfiUdkJ8+gbqoozdp6/B7m9DYeIRIi9HMdpRdcl6B4zcZcywtC58DhOLd/RCdJhFE6VCCfGRxwgc6IYj9gzC4Em4Zu5BaaoFE9hluQzAtTKS4NmqQHLVEoCK5lPn0azpeHJiGI5NfuDwldVrla/7IJmCsgKgkgcmGcQ9mCSAdYCDjJRtlNchue3Ihx+i+sFFYvQdeerJLkkwJMytAnAJ9sazcDZJEGz25SsU6SZMA81ddYi8GEbjeidkqQrZlI6v8wLdG7tpPaTorT2MG2l5YT0cbSX01a/6Q0ZmdgGgX4g5GBwehn0hQ/gNd0qgkPVltHcAAAAASUVORK5CYII=',
lock_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALvSURBVDjLZZJdSFNhGMf/5+zo2pyb5cdyxXRNW2WkhhJGXlReFEZEBV0UVARBIAiCXnojdFE3QUQR3k6iC6GIoK+LMDNBaVpOW04by/yYzuk8O9v57DnHJqteeM57zvue5/c+///zMpqmITv6+vpsqqp2KorSRLGDAhRxiiFZlu+2t7dv4J/BZAF+v7+OkvtdLpfHbreDZVnQN9LpNGKxGGZpEOh8V1dX4D8AJdto87PX660SRRHRaBQ8z+ung+M4OJ1O4+dgMDhNa4e6u7uFLIDTH7R4q7y8vEqSJIRCoRkq9wSt/dIBgiC4EonER4/H46qtFKqqmXBq+vlt8MvvwaTnrhoASmiyWq0Ih8MgyJm2trZITpWRnp6eFmbtbbChuhiWkitweOqRmPVh6nXvnSygVNecTCb199l/jbpc56+3ey7BXtSAeHgS+YyIQvtO2IrdDiYycF0bCvuwuGYxNJ+tGYFJk6ApMjRZJpPWUVTVDMeeU8jMP4GwwmDpWwpSWlxJCxtHOZCJFy8cBwMWjMlC82lAZcidbUjFhpFJBODwtiI99whsvow8WwXM/BhSfH5LY8ebEKefBGiQl5+CM5eAYWwEyMPCHClhVJQdPEfJD8HmyRDXPVgZHEWaX8LhjkmjnaxeJlS6C4qIxMQoEsERLEQmsRrPoKymFeJCL0z5GjLrFYgNfILz5DWoUmrLHwJI0GVoioQi314siSziCQskzY35L/dBVwl8fBeWB4ex3cuAK7BDk8QcAPVe0xSqQMLq1wDGxn/gwLGbMEc/IPRsEIFXcUy9fAfWtAaWU6laFXrOXwBotEgSiqor8X1mEeLEC3hqm1FQQN0Zn4LviJtOL6auiIbcXABnlENUVdY9mMBEaB73Hj9A475KWEvNaNrvIx9+QuKTKHRT+STKkJ0L0CWYd9+ApcIEf4vZaCHZTmCSJgpQhCQpzFChyqZfuvFbADGDmf5Ooyx9Q6dvhrw10w3bvFiKsvmug/6M39LTvtXHnYlaAAAAAElFTkSuQmCC',
lock_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALPSURBVDjLbVLdS1NhGP+d7bS5tcZYkzmzWmqKSSISpElEUEEkXfRxGRFBEA2MSC93o3TT6Ka/wECJQG+7CfrSLFLzIkd+TDsOjFg559nH2XvOe07Pe9Qy64XnfDzv8/ye3+/3vpJlWdhaQ0NDPtM0ezjnHRRBClCsUowbhvGwu7s7jx1L2gIYHBxspeaR6urqQ36/Hw6HA/QPTdOQyWSwRIuALvX29k7/A0DNPtr8VFdXV88YQzqdRqFQENMhyzLC4bBdnEwmFyjXEo/HS1sADvGg5O1IJFKv6zrm5uYWVVWN0rdLhPienZ1dEcDErp6kxLYzkMWDkh1erxepVArU1BWLxZRtNUpfX98ZRVGS0WjUrv0fQKXQTNPE99JOo0ROsBM1xLbyLw+Utzes8VQjvuc8tuaLzRNwWjosbsAyNkLXOQam22xTwxVZXNg3gcZbU9IGAzLxyuXTkMgOyemh93nApD25grbphLgObqiU6kG2mEV/VwILT9/9kSAmiULjxzPI7hAkyUcAuwBPgNImUMyBr89DY+uoCTXh2vAdxJmxDYAowhSTGNZmJknnbgSOnMDd548pz8AsDkb6I8EGNFUdh6oVcK/0HsVEUHpzf9UiAB1ChkVUA40NcLhC5IwJg5rPNl8HJxbc5DCJ5UoujaM1ncizEiaXX7OWfodLtgjdoilCa/bzNJxuPwItndAMZjcrP+ehmwYB6tCpZr2sonX/SeT1ovxhaVSzAYRWiyQEDkfh9O6l68UIQINB/oT9B6iZ22DfcssI+qowlR7DWGr0C1nRRgCMtJowDeHBDAHsASp8KBHAwHgCzCzbbGpDzWivPYePyihSsy+gcbSuPLDKNoCQ4K65Cc9BJySX2z7C4XY6CZoM0stLKk49uQrJ4UEm+xWJghPHHvHyximwMhZHemB7YV8cfTOM32+6Ycg7Vbxce4WRAt0YAby5fgEeKcjVvgWNOgAAAABJRU5ErkJggg==',
lock_open=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJpSURBVDjLdZLLaxNRFIe/O2kTqxmNxAopUjXWB7pwrUIXggs3oispCoqCFWG0G6H9D6Su3IjrQEAExYULF+LKlagUNL5qs4i0jRhokj6mM/dxXIyPIdgDh3s43Pvx+517lIiQjmq1etJaeyuKomPAFmPMC2PMvSAIXvKfUGlApVK57vv+/aGhIeV5HgArKyvMzc1Jq9W6MTk5+aAX0Jd6fCifz0+XSiXVaDRoNpsA+L5PqVRSYRhOl8vln/V6/XEa4P0put3uq2Kx6M/Pz9NsNi8GQaCCIFCLi4uXZmdnKRQK+bGxsTu9CrxUnTPGsLCwsBQEQfVPc2pqqgK0Op2OGhwczG9oAchYaxER23tpYmJikA1CiQiNV1fk2cxRjFNYazlz5A0Z0Yg1iElSa/vUddtPgfMKOe2J4eC1dypRIML45WMoPFRmAMVpcAr6NgECVvOxevEscBZg5Nwdvj28+c+CWAMIpvWIvtwOlMqD64eBAoiDtQ4jJ0aJw3mcWQPnkDhKAYwBJ2Bj2rW3eN4WCoeP8/35XcTtZHj0FO3PNeJwCX/PdkQsouM0QIMIYjWFgwfwsjtAOWxked8aYJiYwr69rK/mELMG4v4CPADRGhELVrP0YYZ27TV4BrfuiMIIJKb95RPtr43ErnOI1ikFWidUG1PYv4fM5iJ4MeUL45S1ge4Ptu0bItvtTxQ46QXE4BzOxLRrNTKbfdiUh74sOAPdNuHST/TqMv7wVgSX2E4DRCy5XVcZ2J1BZXPJF3r94CzEIX64jNUR4mwyL2NSgDii/uR2MgtjEKN/p/l7Ym2yWNYmtUsW9hfAtnFXLnJPWAAAAABJRU5ErkJggg==',
lorry=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHYSURBVDjLxZPNS5RhFMV/7yhGhsL0oaSBfRDF5CIFKchVRCHVNugPiEDa9AcUBEEgQosWUVC0ixYuQtoUmAuLWlSbehmpFGGQQEhCafK5Xy3mdYIKWrjobs498NzDec7lZhHBRqrEBqu1+uByzcx63Q0zxUxQFVRTE0USqunpmevTp/4QMLPermMjRBhhTrjiboQpYYKbEqa8nbx/8q8OVAV3Ze3bImGKq+AmDdSEa2JT507udIxx+9rzMDVMzVR0dGb89N1W1UR35ew//2rTU1w9P4h5UFtabbk18eYG0BC48mSO6uIKlhxNhiVD16zJD/ZsQUVJGhzqKzOwp8zNh6+2ArSKrHFxxwzfSwuFZcEK6+u8fVsfF2pDdG7OWPiyzIFdZSQJTQGXhKV6MSjNQdfUCFIFSYKok9dWqSf7JZDSD0wTbR3djcS1SN612IARKmgSRIO9Xe2IBSoKQHbuxOGIcNydCCcK9IgGj4AI5koDRATh0cAIZl8/zkqPnr3L3i/KfHnfUXr6j1Ndyup9gyN07j7Cp69tU5MvPmeTL+eyDzMTGcvV+eH+7Qzt76Bl5eMUQBYRVCqVYWAMqAP3gEtFP5rn+ez6Kn97N5rn+Wz2349pwwI/AbeHbP+UHpVDAAAAAElFTkSuQmCC',
lorry_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJDSURBVDjLpZNLSNRRFMZ/f53UUoc0SnyQhYRmRWVFYrvKQqpt0KaIiKDcRBs3BUGQSNGiMhJ6rIoWQiVtJlDDmXDRWx1c2EOQIXqJKI3ee869LeafCQUJns13v8X5OOf77gm89yykslhgRYbvnh5T1XLnFFVB1SJiETGzaK1BxMT2X+jd+5eAqpav2NGE94pXh3eCc4pXwavFqeBVeNV1e88/JxCxOCfMTKTwKjixOLUZFIMTQ260lJuFbdw43+NVFBVVsXIyfmlfR0TEUFJ74L+7am835w7Voc4z9nUq+2rny4tARuDskw8MpyZR4xCjqFFkRmd5TVk+YgUjnnWVRWxeXcSV+/3FABFrZzixPM7PrNFwZIuGo//mS5ZVcnxsG9HFAaOfx6muKMIay6yAswY16bDRzjY6MRkjxWKNxYojOTZF2ugfAWOmUTHkFJZkHJfQeSdhAooXS9nSQTp7HjGVHqfvhSFasBI4QnBw9ybvvcM5h/cOH6LzPsO9J1pbQvXeGras3UpF8Rp6hh7SP9jHp5Fvl7MePH0dDKbsx6KqesrW72T4a5CurGsiumo7Iz9yursS74PsqvzpTTUb0SxlY2kjGljqNzQAnIqEKR1OJBJtQBq4FYvFmsP3SYDxqe95i4ICmtYeA+DMrg66BtqBx3kRgGQyGQca5sR+b+4f+D7xZXoo9TzvbSpOS+MdWmNHycvOBZie1zE57y73v3tODhG6BtrJCSLE3zwDuB7M95zrm8taw5UKgUmgvf9aquUXPDCAeT1Y9gAAAAAASUVORK5CYII=',
lorry_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI9SURBVDjLpVNLSJRhFD2f/+QThVHKJ41Ti0pblCAKrtJQ7LGT0E21CUEisFUtioIeIEGLICkwahUtooW4aAQTcahNBWE1gWOY4yQIio8c5/vuo8X8jYuCBO/mfmdxD+ee+x2jqthJ5WCHFYg97U8wc7UIg5nA7EDkQGSz3TkLIhs5dWu84y8CZq7e09IJVYayQIUgwlAmKDsIE5QJH4aftP9TAZGDCCG9koQyQchB2GU6WQhZ5JVU4lHxAAZvvlEmBhMzOeqbvHfycYDIorzu9H935fExXO9pAIsisbjuPXj5/i6ADMG1kRnEkmtgKyDLYMugNGfxwaoikCNYUtSHgjgaDuL+83elABBwLo3e3ZPYyJn1JTuwL/0PLiwL4UKiESUFBrMLyzhQE4SzDlkCcRZsU/6gyw4K2YyR5OCsgyPBl8Q6Upa3CKzdBJNFbnF5xnHynRfyL8BQcji29hru9lWEk3HY0gq0ppsAnIM5c/yIqgpEBKoC9buoZrAqmosFnfWVqGvrQl64HqlPEUxNjGLl29dLOS9GP5qppPse3N+MqsOtiC2aVKihEyW1TZheyh0bjsZNI8/NHGrrQn58HOZZDwpnXmFfbRnUaH/Av9LZaDQ6ACAFYCgSiVz0330A4IkJ51eEgROXtz7QjUp4YmrNdsIUaQ/MtXSfryn6MYJ0agEbANZWPcTimN9WmApLy4c+v52gn5sFWPV2YXnJYHresAIPzXbjHO3ee+XXUrLXYxNiT+cVGOyI0J3frMiI4RHtXVwAAAAASUVORK5CYII=',
lorry_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ1SURBVDjLpVNdSFRBFP7u3S3Xh1U3LN0MQ5NsrcifKEMCC8wkpWCJKKGXiqAgiF4KLCiCIIKggtBCgpAI8qEkAmXVyCW13BbJWiRchWUzV4pSd91778x05q5cihZ6cIaPMzPM+c4535lRhBBYzlCxzGEPPTofYYwVcM7AmEHQYRgSmmV1Xa617sbr/fX/EEjnNTUNEIJBMA7BDUgyQWSCyLhpDQS62velzUBG4eSU/Bk1L3K5l47SUnSJjCw3Wp03cf9qH12hTA3KVTfODNw60GaXKeaVNf23VtbfiytHK8G4QCQ2b7vbOXKDjlMEl19OIBSdA9M4DI0iEIwks/aHyr7iYvkQ7UuwqagQFUUu3H4yuMosQdeTOL16AHF1aillnVLUrPQF15HtFnAVNyI6242paS9K17mga3qqjZKAk8pMS1C0FJiJRRMrMhLILqxDlrsCtlg/wuExBCdmLQK7Ji9SpJXOvJTixpLyXKq/AFWZhDM3B+zXC+RvPob4p1f4bpSAREwRtA/MQLxppU5QC4Vs45KlebjOjf1HzgGLAQx3PEf1qUtQY0N43BGEOmkn9xNQn/Z8UD5G9bBrQzXWbtmLUExJrK9swLbtlaip2Q2nKw6eDIOYwebfobj2ApqrFpClfXn951M+7vf73/p8vl5an+zr7RncWTA9l1NYDh4fJd8Eqryl4FoUDsc4PDu8aGnKGJWOSrrPFGirPZvrOXgvf2MmRR2mkjSMdH4mEg8UWzaQWY/3D1tmqNw9aQmG7+yKbG3uKLCpkyTqDzoRf/0/W2Ypvo0FMd517Zk93aszkpoj8MCbEPTqTFEtCMvCFFvk/wbcn5vkMx8lqQAAAABJRU5ErkJggg==',
lorry_flatbed=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFUSURBVDjLY/j//z8DJZhh1ADiDXBvPvjftWHff6ea3f/tK3b8sS7ekkaSAS51e/8fuv7+//6r7/4vPvDov1nu+rckGeBQufP/nsvv/l989v//o3f//xulr/pPkgFAJ/8/def9/6WHHv+/D7RbN3EpxIDr8ws3XJ6d8//ctNT/JybE/D/UFfx/V5Pn/83V9v/XlJj8X5Kj9X9emiLIyf+P3nj7f/aeh//3X3n9XyN6AcQAkOZn51fD8aOTC//f3T/5/82trf+vrq+BY5CTD1x583/vpdf/d1189V8lbA7EgHhf2/+x3lb/oz3N/0e6mfwPczH4H+Kk+z/IXvN/gK3afz9r5f8+lor/tayD/mtaBf7XsAj4r27u/1/NzA8RBpqamvdSUlL+5+Tk/Aeyv+Xn5/+PiYkBsfcihwNMHbIcTMIGiI+BBIE4ComtjmaADbrcwCdlAI+lwaKCJjy7AAAAAElFTkSuQmCC',
lorry_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJNSURBVDjLpZNLSNVhEMV/f7UCy+iaqWkpKVFqUIlRUS3SHkSPRYugVbuCaFPrEgIxsKCFix5QtIsWraIIS5MQa6G1CTN7iKa3SAyt7P7v983M1+Kqm4oCB4bDmcXhzBkmCiEwl8pijpXTf+v0iKqWmimqgqpHxCPiZtF7h4hrO9DUufc3AVUtLdy2jxCUoEYwwUwJKgT1mApBhRf3bu75owMRj5mQnkwSVDDxmPoMisPEsWDxcq7ltXDl/JOgoqioipeTXZf2X88RcRRVH/znrtrZQePRWtQCI2M/slvv9l4AMgLn7n+gP/kddYY4RZ0iaZ3la0sWIl5wEqgpT7BxVYLLt5/nA+R4n+bEsi5+Zg1NW/botPUZnru0nNb8du70THFs3gNqyorxzmfO6H0a8w51KWS61aVQF6MuxnyMiScWx4qCKlo7d9LzbnRWIMe5GBXH/LyiTOKSSf7qtzZCtseZ4Cb6Kc1fTVXxZr7HU1zs2ITqWQCiI7s2hBAMMyMEI0xj6fEEu2uOocFQU4zA58kREgsLefbhMb1DT4k1vSTrzqOX0aukH0xUbqFkXT39Y1GqvHYfsTg0GEPjbxkcH+D9WB8TqXGGvr5nw8odrCmpI5YwHoUQqK6u3g60ACngBnCq9gxbD60/gZgipmgwPk0Ok7+omJcfu3n0+uFXpzREf3umw1crLHYOZ45YHJUFNdGWij30DHfTPtD22QkNyebQF/3vN65rypqqK6vP7RxoH3VGQ7I5vAH4b4GKxmhKLTs3ZVrxpTkMzsx/ARtuob3+yA7oAAAAAElFTkSuQmCC',
lorry_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKZSURBVDjLpVPdS5NRGP+97tVpw9nm5tpShiSsFhMMImiCQlAOk+im7Ma6UKK86g+oCMKwgi66CLuQ7rqqBRVRQS2aFIFeCA7xQjC3qbkcus/345zTc5aNoqALX3h4znPO+X085z1HEUJgJ18Ndvip84+uJBljezlnYMykMGCaMvRqNgw51t+cvBk78ReBBLeEIxCCQTAOwU1IMkFkgsh4JZuYeT55/J8OpAonkLaZrmzkspZAmUldhtXuxUTjbTy48Z62kFOTvBrm5fjd/oeqtOgJDvy3VxZ7h2vnDoFxgeR63nL/yfQtmv5JcPXlIubTOTCdw9RJgcLUWLXe77OBFKGbAgf9DnS1O3Dv8WdnpQXD0HDRHUexZmnbskEW9ap9We9q9mMkeRj2BgVLq1kEWh0wdANVAk6nzPTSNtCoAmUW2+chAYbJkUjmUSJXVQJdL1eAdY2eyomXS2UklvJIFRvhavFiNb2II5YSFL2A2LNJeFua8WVmA62qhp6eHps6Gf8G8XGC/gQHFxyszoFj/WcwdOEsCoUCksl9eP3qBbpqp3FqYBgN9fVYWVlBvXUab7/bryvyKgeDwcVwONxeLBbhdrvF+eFh5c7YGLa2toTP51NCoVAFlMlkkE6nYbfbMTo6img0uvnrKg9NTU19mpub++p0Ok05MTs7+yGfz5cGBwfR3d2NSCSCvr4+5HI5JtfknqamJqsqB4lEIk7paGdn5yVSGKPx7kAgoGWz2drx8XHYbLaKimzJYrFwuSbrVCpVVn5/jdTKgY6Ojqe9vb172trauKqqzrW1NSwsLMBqtcLv98Pj8dC7MDeWl5drYrHY6h8ExGzxer0hUoxS724CWeSLpd5HpKDL5YpSrtU0jZHTdXJ0+gdZxaA90+h/PAAAAABJRU5ErkJggg==',
magifier_zoom_out=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIjSURBVDjLpZJvSBNhHMdlwV4n9CIsCAxaEEQEvSh6ExQEva1eFBFY2YuQ/pERUUZUkiihy47U9WdlrWSzoqW1rW6ZbrY/zoazOXfX7hZe3jZm2+12h+Pb3YpgtUXWiw88bz6f5/d7eKoAVP0PPw9Oal7zZlombGGJHQzlCs+D2UL/2Bzb50kRD0Z4zR8DpCK/jsikj5FAJWQwKQlMUsLUlzwcE2ncJmfILjurqRhwRGTCy+TBpmSEZkT4mRw8sRwCbA7RWQnWsQT0L2iiYuDlpMhSvFSUvYr4/pMANyVgOCp8DzECWp9E2IoBdd+YMrIqj9ICXIr8bjoL6cJ5zB9tgHygDuLO3chu2465DZuQWrU2VRKw+NMFOiHBrcgjyq1Diuycyhblrw87kb7bjmR3G3h9M7jWi0jUrERJwOROsJM/xn8byYJUZMfHDDLHT/56c1GerV4ulgSMQxwxEEgizOWLu6uyLZQpxj7ERRhsNDquG0T7pR26sm/Q7YhrbgzGSMsoB19MQPCziHEFdzSDnlc02k0uTFiuwHpuC99/eqOu7Ee69iyqabGEicuPQmxTb7Bw9k6g0Njji7fojZTXeAqcpw/++40wHVvH3zuyWvdboBIDTVu1T89sNg8TBxF3GuC6WQ/DoRX8XwdUHp9Yr+1tWGO2t+1C1HoVxP6lWFBA5VZ9rbarbpnZcLgW+r3VzQsOqHTuW7KoY8/imrKP+K98A227/eEeskW1AAAAAElFTkSuQmCC',
magnifier=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH5SURBVDjLpZK/a5NhEMe/748kRqypmqQQgz/oUPUPECpCoEVwyNStIA6COFR33boIjg6mg4uL0k0EO1RFISKImkHQxlbQRAsx0dgKJm/e53nunnOwViR5leJnuZs+973jHBHB/+D/ah7X2LXWloilyMw5YgtD3CDiBWN4Zno8bQcJHBFBucauZfsolZDCru0OfFcAAUISrLZDfPzSKxuiibOT+T6JCwDMtrQzYQvZHQ5Cw2h3GK0OI9AWBzJJZFOxgtJUGpTABQAiLu5OOviuGIEWkBUwC7pasNZj7N2ThNJUjBQY4pznAoEWsBWwxU+JFXSVRTzmQWvKRR5RG4KVGMgKrAVYflexAAugDCEygdbUCI2F7zobk7FZY76DIDQgrT9HCwwt1FsBhhIu4p4D3kiS8B0MJz28ftfGSPfl8MPLxbGBAqVpptbslJc+fEPMA7JDPrIpH3FX8LzaROdrE5O51jalgid3Lh4b6/sDALh6971riErGcFET58gwDPGndG9JT6ReHcwfPorGygu8rdxvGxMeP3XtzcofgigWZ0/EtQ7n0/sOTe0/Mo7V5WeoVu61z1yvZzZX+BsnZx9opYLpevXp7eXKIrL5UWit0n0r/Isb50bjRGreiyWmgs76lfM31y5tSQAAc6czHjONXLi13thygih+AEq4N6GqMsuhAAAAAElFTkSuQmCC',
magnifier_zoom_in=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI6SURBVDjLpZJbaJJxGMaHgdcFXURdBLtZrGitiFh0uhjRVRTVWI1as7mQakhjyyEkRAcaHSCrj0xrWGuuoVsr25qzfeYObh6yJJdzavoZs3Sy8PhJ8vR9EoHkotXFA/+b3+//vC9vEYCi/8mvh8H7nTM8kyF0LpoacCazLxzxbM/bb1S3OUo8GQtz/iggGfi1O0NaAzS8kQwCURqBORrTX9LQf5jHQ3KWlA1RnAUFeneGsATSoKIZOGdTsAWSMPuTsFNJeL7SEOoF4GtrUKuuShUUvJpKUd4wnYMtDDj5KQGTN4FRTyInOvH8MDonL6BKuRcFBey8fqYyC0/4Ehhn4JGZOBp1AtT1VkOkrYfMKIKgsxq7b+zErssV0TyBxjaf9UVomBh4jPnVyMCG6ThbGfKRVtwebsK1wdO4+JIPce8xbBGXI0+gMkWoqZ/137jjIBlY/zEGnqoO+2R7wGvfj/N9x3FAWonNojKUCUtTeQKlMUT02+fgCqVzs7OwzhnLyd4HU2xlCLsOYlPz+sI7uK8Pcu4O+EnNRAhWfwKOzym8Y2LyxCAf9GGHZDvKm9Zha2NptudcRUnBQ7rZ5+G0aVzEpS4nJelwZMXt9myL3Bpskyq9FmUzQuZu2B63QCXcEH50ak3Jb4KF0i+p5D5r3aYeJeoRNCgwfq8BCv7q8F8L2Dw9u5HbcWateuj6IXi0V0HUrsCiBGweNBRzZbxVasXJYkhrll1ZtIDNnaPLl9w6snRlwSX+a34AgPPwSZzC+6wAAAAASUVORK5CYII=',
male=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIHSURBVDjLpVNNaxNRFH0fgzNxRlNIm6QVsSRQ1IUiVCSCS8FF4kokO/9BF7oUstBll4IuunBpl61RNy4suhgF21WhJVU0WiHUksmQr2bex/TeMJEoxog+uMzjzT3nnXPmDg3DkPzPMsY1PHrpdaWUZFBCDJ6C3L91NjaWAAH5SwkLhaJaTfBJyOOnmwf4no0lkIJUqg2hAKUAqKMKQMFfEdy5PhN7X/Ge1L22QqACCUFPhCIIvv6UwdJaiyopi1KqAnjMRZ5dUDBzbta5fDxuc99vafuYw9SQgj7B0qsmBeDdTJKXUhMWi5mcaRKSVrt30uCcHrFMtrv7Xb5wq8+uzp/Ip1MJLoYJ4KZiZsoozU5bBqWM9r8slO0c5bjvdrr6uVt9B6CF1dcfrgm5JVHhDwJIuoA3QyS0tteUGzueEtBwIRvnyXTC6HR76PnLw9tX0HdsOCMWJZ0zLZANt63v1NV+ozO377Xn3M2a0hCabdsMPOdGDhL60dFASthjQHgmOOufK/Az8PzrYhGB22wdaGw+n5ngjkkrjsUqF88kOZ75dV9jz2gFgSh/+ubfOJ012VR60sinJg0UhGApgnDrYw0IZHkkAUheXt/ey/aCoHRqOs4cx+7n4TcaehvAbzY+34PUl39HQAd/48KDtxT8FyH9ghQyhyOMgwRVRvDK4s3wjwT/ug4BPyZOkZTlSM0AAAAASUVORK5CYII=',
map=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK2SURBVBgZBcHLjxN1AADg79eZttvddttll+WxIGrAV4QYTbxoYsSLgjc5arx68qCGf8G7J+NBEg8e9GqURDEmEiOJAWNUwIi8sryWDbSdmc60Mx2/Lxw78caZ1fXVo1EUAYAAACAIgICymrt/7/6P8er66tHPPzklNIJyPgNAENRqAWSzBEFdN9TzWq3y0ckPj8ZRFKmUwuxL6fcXrJzJbJ18xFKv6/LlX1xMP3Nra6jXW3Z3eyiKm0Zp7t1jtWYcaQAMh9uGT7eMkkR2+m9JcluSJEJxQTEZm2Rj00liMkmUeSqEOYI4APr9ndQzi+/v4OPz2m+tWd+zV2d2xaQ8pDfoaUUNcbMlyXIhFAgaQDAcDiXJ2MP1ymilNPn6X6q5OvvZrEhk49SsyBR5alpkQqhBDNDvr1PPDPrLhu89Y+XTbeUre7TXCo9MtzW7+y22I81W0zibYkSgQQB5XkiTzHA0NF6qPHiC/Iv/1FWuMf1WPklMi1SeJubTVABBHACdzkB3OdVfXgbNtx/V+eCsuDpg7+pf8s7ERvcP7daW6eSaqPGOgDgAxsNUkhRI/bZ5x41Zw66DlSdPXbZ5PLUr+kYx+slg8Yj2uW1hNRPQEIKA5cFuFgZ+z0rXpk2DwU53XjtgX7xisrXXoDuycX1B54dLbrQqdbMFYqjr2rmbV13YvKTb7cnzXJKkkqLw61MNh7+6Ijl/3eZK0+3nHzdqBvtbbRAHZGVqq5ppt7qWOn15MdVd6Or/ed2+W5lWkwfPHnbzsZZ2e8HStKCuQUwQQkPPsknItKuOxdC16+I9rc2Jqy8dURxMvHz2oX/27xa1m5irZqWAuCwrVVXZ2PGcjVUCCMIhvMpaCCC8OPdCNBMttESNSDWvlFUlvHni+Hc719dej5oxggAIAAAIAIGyrGzd3Tr9P5JrNp8Zt4rCAAAAAElFTkSuQmCC',
map_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALWSURBVBgZBcHLi1V1AADg73fuuXMd5947jzuO+WgSylIhdROEJNFIkBptclmEuza1yPBfyHWLKBcJLdq0KFBMEotKxCBSNDVJ0xzHxzyqM/ec+z739H3hwKH95xozjblSqQQAAgCAIAACBvnQ8uLy93FjpjH3+ccnhCgYDPsACIJCIYBWP0VQFJFiWCjkPjx6ZC4ulUpyA6H/pezsJZPnWpaOzhqrVd28ecGN7LgHS4lare7xSqIUl61mHe8cKJTjkgggSVYk20espqnWmevS9KE0TYXuJd12U7vV1Gun2u3UoJMJYYggDoDx8XUUfWvfn+Kj31TenDazYaPR/m3twVa1iZqRUiQuj0hbHSF0EURAkCSJNG36bya3OjnQ/uoW+VDR+lm/m2o1M/1uS7eT6XVbQihADDA+PkPRNzFel7y7w+SnKwYvb1CZ7prtrShXn7S2UlIeKWu2elglEBFAp9OVpS3JaqI5lvv3WTpf/KXIO6LeaZ12qtfNdLLUsJcJIIgDYHR0QrWeGa/XQfmtLUY/OC/On7KxcU1ntG1T9YrKyJJe+65S9LaAOACaSSZNu8j8uvDIvX5k/TO5507ctHAw83j+M1fvzUvbbb120ws7KgJiIQioTzxhddB1udVyt1c2NTnl0b5g3+mBk38OtWvL5vbutXlqqx+ufeOn37+WDesiKIrCL/N3nLp13f1/VnQ6HWmaSbtdF7cVbjx6YPe2XfIot2vDq/LQ9+LzeywP74oCWoPMUt5XGakaGx23pjKmuqZqy3xi8/0FS9mKcqh6Y/t74Mi+456e2akwFBGEEKmpq4aaSj5qbahaf2PRxoW2O3t2Sporrj244NjZw+DYd4fdXrwiEokHg1ye5zZN7bapQQBB2IpXmA7B/seJi1dO2bPzJSevfmIkxM5f/tGUWeH1Qwe/XTcz/VqpHCMIgAAAeDj8w/Lwb0MDkVjDrGhx8sz/ABdNCpr3mlcAAAAASUVORK5CYII=',
map_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALVSURBVBgZBcE7jBRlAADg79+dvb2929u9F8cbIeHhK8TeggiNHnbQaDS2xsICDXbGzt7KWEi0sLHQGEUiYhTQYEKEEDlCcrzUg4PjgL2dmZ3Z3bnx+8L80VfOzMzNHKxWqwAgAAAIAiBgWKx7+ODhL9HM3MzBzz85IVSC4foAAEFQKgWQDmIEZVlRrpdKhfePv3cwqlarCkNh8JXk9CVTZ1Irx3cYn2i6fv0P15LP3F3pmJhoub/aUY1q1pLMW/OlWlRVAeh0VnWeGbEWx9JTC+L4njiOhfySvNfVS7v6vVivFxtmiRDWEUQB0G5voBwYe3eaj/9SPzJrbvMWjcENveEeE5MTRqoVUW1EnGZCyBFUgKDT6YjjridzhbWpod7XixTryvScQR5Lu4lBnsqzRD9PhVCCCKDdnqMcmGy3dN5+1tSnq4YHNqvP5nb0V9Wa243Vq2ojNd20jzUCFQLIslwSpzprHd3xwuO9ZF/eVBaZSv+krBfr54ksia33EwEEUQA0GpOarUS71QK1N3ZqHDsvKp6yZeaqrNGztXlFfWRFv3dbtfKmgCgAup1EHOdIXFxa9s+gYuPuwr4T1y0dTsxc+FD8288eLz8SmmPW5xMBkRAEtCY3WRvmLqep2/2a6alpy4eCQyeH7py9Z7Rz3/7XP1Df9ZzelZ8snD9j95OmCpRl6c9/b/l+ccF/j1ZlWSaOE3Geu/B0qXb2d3sOHDF641fhi9eM3fzGzh1Tdj+6KgpIh4mVYqA+0jTeaMvyvuZoU/vvO7bdTS0NMqObdjF/DED00WatLBYRhFAxoaUXUvWiYSw0bbz2wMhSz60X96tcOC29/IPx796R95al6K5VxfUx0XBYKIrC1ukXbJ0hgCDswUvMhiA+klg4961tUw1Rtaa7MnR7pWpxcp/w6tHDP26Ym325WosQBEAAADyf3LL97kUTWSIeHbPY3uvscNOp/wGooE3b/ShD8gAAAABJRU5ErkJggg==',
map_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL+SURBVBgZXcHPbxRlHMDhz3femdkunZ3Zdrublgptg7W0aig28SAYk8aDUuNFjkRvhBMHNfwL3vGgQcWEGGJiTCQxgjFoECTUqG1AKZj+MqUUum1xdmdnZ3dm9tW1TTQ+jxw5+vLlQqkwqZTiv4T/E4RtAiRpi431jW/NQqkwefb0x4ghJK2YfwmCRiO0hXEACFob6JZGk/L2qbcmTaUUKQkSn6f2zQxdl0PKp/bSmXO4e/c6c7Uz3C/75HIuDzd9lGlRqUW8cURjmQqDHb6/iT9qUwkCwku3CYI1giBAGjM06lXqYZVmPaBeD0iiGiItQDCFbZ5XBB2z62Q3vPMLmdd6KPXtJhsvUE+GyeVz2MrAtGyCMEKkAQgm/xB83ycIqlDKEnUlOJ/Nw3MtdHiVuHGYsGpQbF7hQOUiynma2oUr7Hc7MNnheSXQMXnPxT8xRtf7myQv9JHpabC3uUlB/cqAeZtszzG8oYP8uTTCMxtnMUFoi6IGtSBERFPtTFFPQHRuEf16gXz9DMN9wzj9h9hamMOWJjm3l+6exzCFbdlsHset4bkubdaxQbJvXkNVFaO9Q3hDL9JY+xR7l/DH7B2SluKTuRhD2Fb1awRBA79S46fVB5xfWOLmwS2ypou3b4ro/geI9QjLydERLmNPTFGJFQYiCODme6Ejz2yYsNy0KFYfMXZoHO/xV4lW38OwYpqVQcrfTXPPKGIUB2gz+ZvWmumVJWZW7+A4Oax784w4g5SenKL54EOUrYn8Ada/v8GP+0Ypd3YwZWdoMwUIkxrlNCZjO9iLK4xnR0haJdZuvUtXySba2sPG9Z+58dQE2uuis9kArWkzQBAxyOHiSA5rfpmxw8fJrFzj9ws/MPv1Fje/uoq7uBvD7MTGwsYijRMEMJMkJU1T+rvH6S/A5x+do/nblwwdeB6/vMnKrXn2nzyNld3DhIpRHTbKUKStlCRNkVeOTl0slnpeUpYJCJXZLxgfLPLs6CDlljC9XiDUGdqEHQJJklJ+WL70F3VJO/3fH/GfAAAAAElFTkSuQmCC',
map_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALcSURBVBgZBcFNiFR1AADw338+dhl3ZmfdXbf1a4NCwUSQOiREBVJhpmRoByGQbp26FJ66d4gI6iIFJRYRdQjCSiglsDIxNEgsK61tcc3WrWbmzZs3b95Hv1/Ye+jx0zNzM7ur1SoACAAAggAIyPLC7b9vn6nNzM3sfvv1d4RKkBUjAARBqRRAPIoQlGVFWZRKuRePvrC7Vq1W5TJh9L7+F5esPR1bObpgotV09eq3fuq/aXmlo9WadGu1o1qr6/YTR/aW6rWqCkCns6qzbUw3isSnroiim6IoEoaXDAc9g7gnHUQGg0iW9IVQIKgFQLu9jnJkzfPTvHzR+MFZc+s3aIyuGWRbtKZaxqoVtfqYKE6EMERQAYJOpyOKev6by3XXZgYf/UZeKOOzRsNI3OsbDWPDpC8dxkIoQQ2g3Z6jHJlqT+o8d4+1x1ZlD683Pju0kK6qNzdbM15VH6vrxSm6BCoEkCRD/SjW6Xb0JnL/biU5cV2ZJyrpZ07+uN+X1/fpR/8o0r4AgkoANBpTmpMz2u15reY69Wd2aizWNPI7bZi5YZinNs1uc/LaAdvn31KtFAJqAdDr9EXREH3f3/jLqSsXlI+k0u9+kWaZDdNbbZu/Xy/pe+mr97z21H4BNSEImJya182Gfohjf6R1WZF6dPsReVnIi1yhtNxZsmPTA6J04NkP9tgRnlaDsiydX/rdpRs/azZbkiSRZKm8LCyu/mpUZLJiZJSPdIc9Ozc/KBrFzg7eVQuIs76VfGR8rGmi0ZYMU0mWyPLMHZMLsiKXl4WbnT9NN+ddXPrG5eUr7u0eUCMIoaJl0iDExvOGNaEpzhLHz70iLVJJlrp7drtddz3mwuLXzi+e8+rBEz4+9qlaluXyPLdxeqeNMwQQ7NryJAghgMPH7hMqDZeXL3vj8Ic2Ti3I8k+EfYee+Hzd3Oyear2GIAACAOBM67i0LD3UP2RCS5blVm6tnPofTwlmPORvTlwAAAAASUVORK5CYII=',
map_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKvSURBVBgZBcG7ixx1AADg77czu5vL7Su3d5fEmHBBElBBEkvBVyqNdgYrwdbKRsm/YG8lFgoiNpYWChJBI4IPTAKSByQm5p17aOZ2ZnZ2Z2bH7wunz7x+brw+PhVFEQAIAACCAAio6oXtze0f4vH6+NRnH38utIJqUQIgCBqNAPIyRdA0Lc2i0ah9ePaDU3EURWqVUH4l+/6CfedyW2ePWO73XLv2iyvZp+5vJfr9gUc7iShu280K755utONICyBJdiRPd+ymqfy7y9L0gTRNhdkFs+nENJ+YT1PTaaoqMiEsEMQBMByu0ZT2vr/CR3/qvrVq/eATlsobptUx/VFfJ2qJ2x1pXghhhqAFBEmSSNOJx+u13X2V6dfXqRea/LxylsonmXKWmxWZ+SwXQgNigOFwnaY0Gg4k7z1j3yc7qpcP6q7OHJnvaPcO29uNtDttk3yOXQIxARTFTJbmQmhMlmvRcaZf/O2nk6dNm4Xi/lWdOBZVie7SqiBGEAfA0tJIb5AZDgag/c6G8192LK++5IWn9mt3YpvJ3KPtxKNbP4paJwTEATBJMmk6Q+aPew/dLluqw297ZWPN43lLVdSaENl4ctXtf44KWS6gJQQBg9EB9oxczCu35m2j0ZqiXOh2O6p6YbFoFPOFsg6aqKtpd0AMTdP49c5NF+5d1ev1FUUhTTPdKChmpbIOqkUDmkAxLTSdMYgD8iqzVZe6nZ7lpaFiNtfb05MlFz3cPGg0HitF4lbw4N9SnN1x9/fLICYIoaVvYBpy3XrJ3tCz/8qm0cO/XL9BdfM5ragjn5VayR0vHvjP1qWf7ckm4qqq1XXt0MoJh8YEEIRjeJWjIYCVycJm1HL30o7ta785cvx5z979RnjzzBvfrq2vvha1YwQBEAAAEACd/IGTK7WimPkf4mk8xSgTQvcAAAAASUVORK5CYII=',
medal_bronze_1=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAISSURBVDjLfZM9a1RBFIaf2d27H1kLTRUQEtAUojYpxA/yA1JYCKmtxE60XSxSxCb/wCJiY2NpK1GwCIEgpBBCQK1CihVMcM3m7p2ZM+dY3E3YuKsDw5kZ5nnn5Zwzzsw4HenpE+P6TWRjAxPBUkJjRIcxW1rCPnyi/fG9O2VqjIxRyEZAFSn3IliIo8hfAlEwEdR7VKQUEjlb1yYIVM4JhIh5D3NzJO9R70nek4oCNz+P5gMshP84CHHVHe6tNG71ad2tQ606PHdIXhA+75JCXP2nQP3+/gFS6ccbDy/4+iwpVTApcJKTDfZpVt723b10MMq40yrYmzu3wb0LC89nfGig+S+kOMFigUkAV6XRdEx9fdkl2oPs2e72+Rwk6ci1RzNeWqS8RwolaDGgRY70fpIf9Shml2f0OHXGkmgqi7F9BR0cl6AEkIhGj0lA4wD5fchApsCnxfEcqE2rVbFYoBKwFFEp4RSGIr4geUGTm55QRiubSOKZ9bOZhlECljymE/rA1I5c7IOm89bFl3mIHlAQD5GjcQcpbtZ6e7hahqZwznr5cqKStWj6H0hgc9zBQNdqO6+7mcupZE1QxVL5JzCj2mrTaGfUvmx1TStrY30AEF7MP0ZYDQvLM0X9MkkU8znEnPrJPrWdra6KW7m4/m19ogBA0bl6W/vSIeqimpsmgZk7UmFT69napVfft0fv/wFUf661fqqpcgAAAABJRU5ErkJggg==',
medal_bronze_2=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIgSURBVDjLdVI9a1RBFD3z9r3sVwrdWCwYUkSLWKaQWOwf0EpIKSIi6QSxWyxSrE3+gUVEQREsU1ooiIRImhSCCImNIcj6kcVNNm/fzP2yeJu46yYXDpdh7jlzOHOdmeG47r+7azeuXMdWZwNiClEZQMHKuHqhgedrL/Dm4Xt3zIkxVEwMtXxYVMCqg/4PIfAw5T+BwDADSGRIJIeq5neeRgSi4QMFRhEVFKMSSBhBCEEIJIxioYyyTYICne2AArc62x+X72Q/MXXQRplzu2kUoV2ZwttvH0CBW2cKvLJfe/3v+7147vYkTc/gSCIYZ3CcYrq/i1s7r3tLs25vmOOOf8FeXlsA3FqYf1T3oQhN/4CzIxhlMA6AK6BYcqhsP2mD7Gby4PPmaAbCTZ67V/dchqRdSMiJRgGapeDub6SdLrKZxboeSnMsRFNuUHUW2j/MiRwAJih5GAco9cEH++hzBfDSGM9AraZWgFEG5QATgnJOljAQ8RnEM1Rc7ZRvNJgIlOnE+glk0DnAxMP0lD0wtY6jHqAyap19ngN5AAqwBwidcQdC63H3C1ycQCWMWM9fFkRJGSX/AxywPu6gryvx1rN24lJESQlQhQnBiAAzFMpVFKsJ4k8bbdNoZWwPACA8vrwERivML9aziYsQVphPAUoxcbSLeGujreyWz63urJ4qAABZ89KC9rgJ0oaaq0EAM9dRxrpOJCvnn37dHJ7/C8hDreOQ71qxAAAAAElFTkSuQmCC',
medal_bronze_3=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIYSURBVDjLfZK/a1NhFIafm+YmTauoFTEodmgdxK1THfIHOIlQBwcXkYKb4FQcOlSHTo51qOggXcW/QUpLUQoKVWhd7GKEGoxNbu693/nhkLYkJnjgcJbvec/L+53I3Tmum8/e+/1bV9n83sDEUTVUDVNDxKhNn2dlbZuPK3eiY6ZIT5kqZo6EI1C6U6XbIo6K9iL9AifbgnahHvhE8B+BQp8DUdpJ4NLpMiHXvr5ypkIryVD9rwNdOvjyYfHehU0ulvcYtTYAiVf4wRSvd2YwKS71MlFviL5Wm0+0/Dy+fvdUKE2iWsAlJZKEuLNPvvu2NU7yuPjw0+qAgL+5MQvRu3zmSTXLy1jyG0nbeEhxySEaoTwaMbb7ok7w2/Gjna3+DFQW5NqDaiYVNGmieRf0kGNpgjQPSBpN0sm5qh3qwkCIblIL41NY57ALSg4SsJDhkmOhg/z5RUfGINPaYIjmE+YjeEgxyXENmHRhzY9EshTNBNNoYsg3Oq6KSTixftJ6NCXHNcNtyB24eSMKLTDtty5ZN4eQAQaSQaAx6EDDerH5lagYY5r3We9uVgpxhdHsJ5KzPuigY8vF7Vf1OEooxKNghmvAQwB3RirjlMdjip836m6F5aGHlD+9Oo+wlM/MVdPSZVQMzxIICaX2PsXtjbpJtHh2dW91qABAujA9ay1ZIFjNPJpAwT1qmLBupXj53MtvW73v/wKskrUnmSsmJgAAAABJRU5ErkJggg==',
medal_bronze_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ9SURBVDjLfVNLSFRRGP7OeO88fJAMGmqikAaalbUIJ3NRROAiwmjdQmqXFdRmaOGiB4nrMmjRpk1LIQjK7CmmCFHWIAguMpGhnMGZud7XeXXOndF8YAd+/p9z/+873/n+e4iUEnrx6wMSBw+BjY1BMgbJOQSlEKVs9vZCvnmPivFXBJuWsV5sBslNQKH317NPsX39I6AsIBGeFzQHRDqXamMXgtAGgfooFRjNzeCaRIXO3HVBWlshbEf1+P9R4NM7JDM3GDluIXYirL6UlfYJmO3Cn0mBq55dCcLnFpfAQhbtuFTphZvAeUhdQZ3ObJjOIqKh5xbp5kvbCYiegnyW6FLlqH/sdp3nR5TcVTB3TfniKhIlm5QhEiUon3+cBpV95o3U9FYPOEuytst1HouB2zkltQiU1IdwbbDcCuxsDm7TxTpR4MkdJkrBemjFfginUATqU5kaHfWCWlAHLJ+Bw8oBj/fs9EDIuJBlgWShCbieexHM/WL+7OUws5xGPt5RU7h/Or9qZYdfPvh2r2SiLP5EjG5I3wju46OTwUK9gVPtJ9EYP4B3qdGqqR+f7iYGGqpLV5BZQi2lhG+VrlUokvG1NI62dYKHODrrz4ITisThbg29WjKRThi5ORDDVBz+Fuk6/lgrMEklzrdfC9pvnXmClr1HdBktKnDEkPHladokNkJmVCkRgQ/6DUCNOa8MTC1PYmisPyAYet2Phd+zunQDAuPm/LRw+GDk7XA65v+Euaca4Zp9KhoQidfiQm0jpmYnEVaev/g+gjAxMPH1g4Y+IuvPOaBLtnQJiyVBRY+QJA6uBZCsYJi4UsULv4joU21VKgoqRqYeLif/Al7ruK6zYN/dAAAAAElFTkSuQmCC',
medal_bronze_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJsSURBVDjLfZO9a5RBEIefvfcuucRUanGiJkYjiNoEET9II0gUsRC0sbASO1G0MVhIiCL+BdoJYmOnjYWJQsA0QRBRYkBUJESJ+dJLzvfe3Z3dsXiTS6LowDKzC7+H3ww7RlUBCJcuKrv3IkNDqAgaAtF74lIuHT+OPh9m3YtnhlVRXC5Wi3SVMIrkdxHUef6MFYAXVIRoLVEkB4k06uI/AIUGwHnUWujoIFhLtJZgLSHLMF1dxLSOOvcfB84PmLnxG837a7QcaoJisvRukDTDvRojOD/wT0DTyYlJpFDze8612aZ2QiigkmEkpVSfoFx4VDOHw+SfAKOq6MODB8A8cd3XK9Y1E9OfSPYL9RkqDkxCc9nQ+uHeFF5PlS6Pja6dQZA+2XW+YqWFkFYJLheqd8QsRaqzpPNVsvbTlbgY+v4aokbp8eu2E+uLuVAciCd6i4oj+jqyMEddWsGGnr9nEHV91AT1GVEcGjxRcnFwea5+GqP6Ypxsempj6C1OAHd7B+XO0hA1/0TiG9YbJziqn8eJUmPfuSs0d+6h/nZw6/uXQ7eGjpZ+LbWg88bXIIa11sWi3jEzNsrOI2cofxrGPDhL6+fHbOvYkKjRK7mD4EeK1fFTrriZGNwa6yqOWKtRrnTCiasrvfdvIolmW+6gHu8UX9+fKpmUQqkMMaIh3wlUMW2tpG+eQv8m7DXDj2uGhe8zhES/meVtdDe7LiAMuO7TlaxpM0EialPwKdV3wyzMTrNlg1AsTLI4I3z5ngSf6Y0GACDr23Eg1qQPH3uimvUEUDXzURj5KHybm/16IgmmIyT6VeHesUG5/Rt5eNFI8xvNdAAAAABJRU5ErkJggg==',
medal_gold_1=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIHSURBVDjLfZKxa5NBGMZ/35dEamtFMkgGpaAVQcTRLp1LoYubi6MOgqiTZOrQxYKjOIh/gIuim5YqdChKl4KbQ0XESoPYaGq+fHff3fu+DklM2lRfeLg7uOd3z929iZnRL7l9y7hwkbi6isWIiaAhoL2xMj+PvVlj4u1K0veUGaphkw0ZNcbuOkasCMOWA4AQsRhR79EYu6AY/87LhwDSfYAiYN7D1BTiPeo94j3iHMn0NNrJsaL4T4IiLNH+sFhZ2KFy9Qcc8aga0knR3S/oWrO751+A8p332+bSNmfuHmP8PEZKqjlJaKF7HynGHrZ1wW0Pe5L+L8R3p2YwXjL9qGal42hsQvyNSYaJgyQBCbQ36w0puFK7Zhv73sAk1Dl9r2blE1j8BZJj4nrKUP8d04yxczdrYY/6yBU0FrOliUtY3MPU9eR7cph20PCTdPwkIWN25BdUpWppGdMc1PfkBhIH0gYLxILqKEAULIIWg+j902WQCPPE4pA+0ChNCy3MwoHobgBBkaJDyGmOAoJbl90NkqS0P770Rguk6VHyna90WqyPAMSx3N6831C3C6UxzCImvSQoSXmS6BI+PX/VyDOWR/oAYOdxckMCS5OXr9cq1bOQGBYzpGiRfdti6+nrRp6xOPfMnhwKAPj8IJkpWtSjY1Yi1eDBdWi6jHURludedBuoX38Av56vLTwJJBoAAAAASUVORK5CYII=',
medal_gold_2=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAITSURBVDjLdZI/aNNBFMc/vySSpLZFg5QguvgHcXHTImQu6OQq6CDoIDjoIpk6dLHgKA7iIOjgotjNLqJCKXQpooMOFSlWGpRGU/NLLnf33nNIQtKmPvjyOO7e57537yVmRj9uvblmF09fYLW+jJgiKj0pUSNnD1V4svCUxTvvkn5NjqGIIaLWPSwqRNVeHsj7OFyyC+AjZhBEhiBdqWp3rxN2ADLDi+AjecbIZwoEiXgJeAkEieSzRYo2TvDh/w6Cj3P19fezV/OfKYV1CuZQNZqa5Zce5u0nJfg491/As2MfNlrxY7M4dXucsVMYGRJtMxkajG9/4WD9QfPyebcxXJP0uxCXj0xjLHDiYdmyk2isQ/yLSYqJgyQBCTRXqzXxXCpfsZUdf2ASqhy9W7bcASz+AWlj4npK0c5PTFMKJ2+WwzbVkSdo9JXs/jNY3MbU9dTpyWHaQsNvMmNThJTKSBdUpWSZHKZt0E5PbiBxIE2wQPSURgGiYBHUD6z3b5eBI6xD9HvMgUapW2hgFnZZdwMIivgWoU19FBDckmytkCTZnfally2QyRRpb36n1WBpBCCO+ebqvZq6LcgWMIuY9JygJLkJokv4+vJ1rZ0yPzIHAJuPkhsSmJs4d728r3QcEsNiivgG6Y811p4v1topszMv7PGeAIBv95Np36AaHRWJlEIHXIu6S1kSYX7mVXeA+vEP7PyqQia3ZfwAAAAASUVORK5CYII=',
medal_gold_3=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIMSURBVDjLfZI9aJNxEMZ/bxJp+hGViBLEgtKKTo52yeBUnETBSVwEO4ibg2TqUAQLgptF6SjiouimOEqpFqSLCB2qIiINYmPT5s378f/fnUOSJmmKBw83Pb97uLvAzOjUxXvv7calST78qKHeEFFEFBXFe6U8cYSFZ6t8WrgadDw5ekpFUDW8axt9q4tvyXtDvPRa+gG705y0TD3mXeAeQKYvgRfCpuN4YQiXSp/GDw3TaCaI/DeBzP35tjx7beIdRwufyQd1VI3QjVJ1Z3n65QLqx+Z6PUHvEuXjqZkwHXo4cvL2GCNnMDKYRpiro9trbK09bgyxc6d4JVkcAPjlE1MYr5l8VLLsQdTXwO9gEmISQxCAOBqrlaqkXC5dt5W+HZi4CuN3S5Y7jPktkAiTuK0QTX5jGpI/favktqkM7EB9Ws6OnsP8NqZxW0lbMaZN1P0lM3IMF1IeuIKqFC2TwzQCTdqKu5IYpAHm8CnFQYAomAdNu9E706WbCEvw6T5/oF5q5uqYuT3R4y4ERdImLqI2CHDxkmyuEATZ/vjS7ubIZIaJNn7SrLM0AJCY+cbq/arGm5DNY+YxaSdBCXIFfBzw9eWbahQyv+8jbTwJZsQxVzh/s3SgOAGBYT5E0jrhr3XWn7+tRiGz0y9scV8AwPcHwVRap+JjyuIpugTiJrU4ZEmE+elXrQfq1D+32LPJr7b2hgAAAABJRU5ErkJggg==',
medal_gold_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJvSURBVDjLfZPNS1RRGMZ/984dmawk7IMpjKiMPuiLCIowWkQgGNHOTZuW0ScFMSsXFSi1CUoX/RVBVJRFRWK6iewDIpAKysR0ambunTn3no+3xR3NFHvh4X0PnPM7z3PgeCICgD13Rti2HdPfjxiDWIvTGlfv2fZ25MlzFj995DGrgulh9iGZddAZk66NQRLN3PoL0AYxBhfHOGNSkDEzc7AAwJ8BJBqJY1i3DhvHuDjGxjFWKbzWVly1hiTJfxwk+grhSFe24wfZzkloiHFOsFUfN/UV97yY7lkIEJx/9U2UH7LhwhIaNyP4+K6Gp0u48keS3K3QdahvcwGeiGAGW/Yh3KW1Ny+ZJpwpgqkgNkKsAs8DqwlfF8ZtwvH8CRn+5w3E6gJrL+clWIaY32BriFV1Rbh4AnERuU2n8rpMYV4EZ5K2zOKdiCkjTtUV16UQV8XpX/iNq9ARbfMi1J422WDPA9+pMbDV1LoJ6xFCxIQ8+xLx8juUahGVaqnyOyxef9A9ci11YB2IAZf8tT59u1U8Hi3z3jRy6OBeWpo38ezD3aVD719e3X9mzTI/jWCLokuI6DnWU93/VGT3ll1Y37Jr9RGsp9m/4wDA6RSg1YCdGsbzMuBicCqVTftEeZKst4RjW88CcOnwHTau2gmQ8wGsoid83T3u1BRkcogYxNad4KiEk3wYG6Sn/yQAPY9PMjrxFkD5AM2dMmwiuqYeXhxPJt7hZZrwF7Xg59ZCZiVH1y9n6O0gDQTce9dHgxcw8OYFQK83/Z0BPt/w9iUlCkbRZg3NOgZVpagiBm6uWFn5GQTHgaVABegbuj1W+AMfFb4tKjBFuAAAAABJRU5ErkJggg==',
medal_gold_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJmSURBVDjLfZPLb8xRFMc/9/ebqVHvEkaiqTJCEIkNmy4shKY2FhIkliwk4ploV00jQhNLQSz9BSxY6CBNtBWbBgsh3lrVd3U6j/v73cexmGk7Kpzkm3Nucs/nfM9NrhIRANzZM8L2ndhsFrEWcQ5vDL6Sk83NyJNuljx9rKiKxGxR3SRVjd7a8tlaJDYsjHmAsYi1+CjCW1sGWTtXJ/4BCOYAsUGiCBoacFGEjyJcFOG0RmUy+GIJieP/OIjNFfKv25OHfpI8Og41Ed4LrhjgJ77huyfLd/4FSJx7MSg6yLPp/FJqtyIEBL6EMtP43Dvi1M28P6QHFwKUiGD7NuxFeEDmVlrC5Xg7CXYGcQXEaVAKnCHf3zbsYg6nT8jLP95AnGmj/nJaEisR+wtcCXG6ogI+GkV8gdSW02mTo+2vFbyNm8IluxCbQ7yuKKpII76IN1MEtWsxBZr+BnhXFwQJxJfARxXpeTnNaO97RvseUvyRWPMxm/gO3D7QZTvLAOdBLPh43vrsdKcZ7ftA8bNh97FLLGrcQelNV/3b59mr2f3JQlBewU2KmUbELLBe1sCzfjL7jpD61I26d5zaz/fZ2LA6FCUXyg6M7nETLw8HK7Yh1fZdOduJIql0I7RcnN+9Yz2hVxsDAKfpzPdfH/Z6AsIUIhZxFSd4wlWLKb56BB3riVoVU62K3MgYLpQhNfsbf95Vp5zhyrI9J9PJus2gBLEFXDzNwKNeCp9mqF/tSASDzIxZvo6EzmhpnwMAfLmh9sbTtFlNk7PUmQh0kUldoKcmXDdUmhpvCZ1qcKH8ELhzsMte+w2tS8iXJ5jooAAAAABJRU5ErkJggg==',
medal_silver_1=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHfSURBVDjLfVI9TwJBEJ2DMxwf1xASSCxsrIwN1TUUFhaUFlr4AwyNsaZQCioqepB/AIWtUaMYGhNiSWdHQUAp4Djue505b83yoZtMZm/vzZs3b1dijAFf3tUlg4NDcB8egLkuMM8D33HAD/NOsQjs8QWST/cSr5FBWGIREwp9OufZdsSSNQLHDUh8ywrAARHlcC9vIYisEOBPhsWwtwcekWBQ9kwTpP198I0lYux/FNhO1TTNyvT0DHQ1BWY0Cg5Jn89B+fyC7PMrqIj5k+Dj5nrIDENP7+6mFPz2fR+4yUgM79GobhwfDc+FGokDBoOB5rruXTabzc2x43K5DAgoIpEIxGKxILrd7siyrJNSqfS24gFKLWcymdxsNgPDMMDDG6AgAhpD1/XgPJ/P51BNecNEBBVIDXV2yfWwOwUnWywWkEwmARUUNjywbTtNYF4Mggfid+hHehvBL1jM63sKi656fQQ8nHJD+excPt9LkhQoJOw2gh7NT07zrmLQWTweh/F4TGp7GwQ4V63f748URYFEIrHiAXUm82RZhk6nM0KC2sY7oNVqtS5QSVXTtJyqqj+vM1QzmUyg3W6P8LYqjUbjdisBrXq9riFJGbsU6GbIXCyaYvQwas1m803EfwO4vsGvr9ICrgAAAABJRU5ErkJggg==',
medal_silver_2=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHqSURBVDjLdVI9ayJRFL0zkZ0ZPxoRFPwF6dZimcYm5W6z+Qspgk2aQAoDwSKwYGWvKyxsq4VlmoUVJBCQNAEJW7swUbDQcdR5b97svaMv+8yYB4d5H+eee96Zp4VhCHJc/DoLvxx/hsfZPQShgEAEOwjggsOnXBl+9H7C3eVvTdYkQBmccRDhlkyFXIjd9z98n6slbwTwkAyxIFBEthAoFp1t2J6Ari4YChiQBEM3UYSDH7AINDeOLLDCNHLY+w5Q4Pbv2KmdZL7Ch9AEjenABAMvWMJCzOHP03PEeVfg5uO3sed7bjFbTNN6a3sb8nq9Bv+FuVel67Fao0nCaDSyOee9fD5fWCwWsFqtIgGCrutgGEaEfr/vbDab00ql8rCXAWOsmsvlCvP5HDzPgwCDJJAAnoHrutF+qVQqoJtqLEQklckNdUYnr90JUmy5XEIqlQJ0UI5l4Pt+lsiyWM1AXe/yyB4SeCWr37dzAjqIvwPcnMlA5d2lfTnXNC1ySNxDAgO6PyUtu6qgPcuyYDKZkNtBTADvVR8Oh45pmpBMJvcyoM4UXiKRgG6366BAPfYOaLTb7XN0cmvbdiGTyUR70s10OoVOp+Pg36o1m83vBwVoNBoNG0Wq2KVMf4bCxaIZYoCot1qtB5X/D8V+vgtcQDdeAAAAAElFTkSuQmCC',
medal_silver_3=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHnSURBVDjLfVI9bxpBEJ1DKMeHkRUbCWIlSuE0KamuIVUKp4pSuI9cWPwFKhcoBRWlFTBtKij8CyIRIUWWUCQXprdEQcBCCtwdHPtxmVm8aM9HMtLT7e3OvHnzdq0wDEHHhy8/wrOPb+Dn/QwkD0EIqSARnEsoHx/C5bdfMLg8tXRNEoyQQoCUIXD2WMg3X8E34ETKhVkSJdh2Y2JTZBRvCZ8QJCIK8NDzGRzlbGBrEcGr/TS4foBE/1UgaqP7h4v3b/fgpPAMLMmA4Th+IOFP4MH3u9/UpPZPgq+fX4+8FXNfvni+pxRJCdrk1WoFbH7rvjuyR2aNpROGw6HDOb8uFArFxWIBy+VSERASiQTYtq3Q6/XGQRB8qlQqNxEPGGPVfD5fnM/n4Pu+mlWoW5F0Bq7rqv1SqVRENdWYiZhUJjXUGZVsuxM0med5kM1mARWUYx6s1+sDStbFpgfm/6MfB7sItsnm9+magAri7wA3Z9pQPbuWr9eWZSmFlLuLoE/zk9O6qwnaS6fTMJlMSG0/RoBz1QeDwTiVSkEmk4l4QJ3JvGQyCd1ud4wE9dg7oGi32+eopOY4TjGXy6k9rWY6nUKn0xnjbV00m82rnQQUjUbDQZIqdinTzZC5WDRD9BH1Vqt1Y+b/BXpYxDgsNaz2AAAAAElFTkSuQmCC',
medal_silver_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJpSURBVDjLfVLPTxNREP52C+mvbSKE2FYkkIiJioo9KMT0QKImnAwHPXjk1IPoxUsPaoiaSDw0MTEe/Cs4mWhVLGCDlWgwFCMJiYeCSGstu9ttt/vLeS9dLBCc5MvMezvzzcz3VnAcB8ys2xMOTp2GmU7DMU04lgXbMGA3ffvoKJw37xF8+0pAi7W5QWuR01Jos3vXNwzstX8EhslJbF3nyZyI+WbcdgCBuENAHx0qRm8vLEZCYN6q1yH098PWapTT+M8EDeNBvV6/X752HWpIQt3jgcFGVxT4Sr8RnplFiHIOJFi7d7fgaJra2d0t+ehs2zZcgYkYnz0eVbs8Urixh0BgSSsrK0OmaU6Hw+GIQh1rtRonYBBFEV6vlyOTyWzquj6WSCQ+7tKARk12dXVFZFmGpmmw6AUYGAFbQ1VVfh+LxSI0TXKfiJQUZ5OwziZTvdmdwSWrVqsIBoOgCeL7Vshms1ZPT49YLBZ5EVo0cM/5nzl8L36BrFWgaNtKRS0/efl46REXsUHP4ya3ejdeWs+iJH7DSPwCjnYex0x+OrSwPPdweOLIIb4CjVXe+aWbu7vjM//pxzucOzEIS7QwGL0CSzAwfOYiS7/pEsyz/ZnSbtdWlORfaBckXD15ize5c+kFjh0+y0Kf2HznqcXFxU2fz4dAILBLA0EQIKsV5DeymEqP829Tr8extvWVl3omJycxMDCwnsvl5NXV1fPRaFTq6OiAJElcdb/fj4r8h3SYRV+4DyWtgO3qFj4szUGu6E8Fd3dmqVRqiNZJkqhxQicTl564TJhfRlrRBXWM0kIEhfB84dlG8i8v6tBqmkd4owAAAABJRU5ErkJggg==',
medal_silver_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLfVPPaxNBFP5mtzVL0nhIA4mUmBaqB/WSUy459FBs6amIIv0DJBcR9GIOWqSI9JSjesjFi1eDoNBEsdEaEFsRD0EPLYKxNMREyY/NbvbH+GbdTdOWOvB4M7Pvfe9735tlnHOIZd24znHuAsxiEdw0wS0LtmHAdv3o/Dz4q3UEXq8xDK0RbzOcxIcSbXHv+b6Bw2sfwDAdEFvXnWAHSHh3P3IMgDQAoI+ckhGPwxIgZMJbmgY2PQ1b7VFM/z8M+saKpmnLzctX0AmOQZNlGIJ6uw3lVwORN28RpJhjAbbv3qlyVe2EJibGFDrbtg1PYALGJ1nuqLMz1aVDAEwEVSqVpGma+UgkEm1TxV6v5wAIkyQJPp/PsVKptKfr+mI6nf5wQAOimgmHw9FWqwVVVWHRBIQJANFGp9Nx7hOJRJTYZI6ISEEpwURUNoXqbnVhHli320UgEAAxSB1poVwuW7FYTKrX604ShjQYnD/mwbfy0Ks7sGT+g64eXiyYq46IfRqPFzzsB/ut5zhZfY+zV9PwTZ1H70shVnlXvF+cHe06LRCt5uBJu7179IXvbTzFmZlLULbXwZ4swb/zDJPxcZkzfnPEBdig/heF0s7sDzGRGjUo0Slg4db+/O+dgmyzScmd8+rm5uaeoijw+/0HNGCMgYfCUD+/AChJv83wm6xVqwstdplHPZfLXSMmK8lkMhoMBv+9TpdF4+VjnPi2htPj9E9IVbTrJr7XZMvQ+PIAQKxsNpskkAyJmiILCXGppSbZxtyf0q5S/7ogWyxOlX9S1qO5gvngL9401yPDHgg9AAAAAElFTkSuQmCC',
money=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ0SURBVDjLlZPdT9JRGMe5qFu2Lrt1a63LWv9ATRdN5xvLsnLRipzZpIVpigjyIs3XAOUHgopoWkggP5QXSRJwJQmtm/IlAWtt3XXTfubS+nZ+P1eby6ldPGdn5+zzfb7Pc57DA8DbL9rjrYxuVsXf7W5fuC2mYawpE7QRJZpDDfz/EngYVTN9qR4EPvlgXjCiKVCPWvou/0ACxDJjSbIwDefqMPxrEzC87IDUW4Pq8Vv8PQVaX7Qw5qQRgY9ePP0wDMeSFfWTUkxmPeiI61DlFOP6SAV/VwFtRMFQCwb4CdwW10IbVcK+aMHgohmPlwdBZ11oCctx1X5p/R8B9Uzzuum1ntj1Iv1tGRtb3zH2dgSa2eZtOOOCMizD5cGyzR0lGBNdx1TP5T96E4+4WttiWg6mYr3Ifk1DF1PBmxmHYlrGZkbFUDku2oSHOAFjolOuIpZ65rs5+MmKg9hWcJlZWB1UbsOhRjYz5r/MoSn4AKWWQg0nwFoyzndhijRobGWIq3XgPQU1sa2LqjCRHoc81IBK9w0OnvscRWQtBGFfEc4b8o7wNDMKOwnY3lDwZZ+h1idB/zsThpf6CezkstVN3yNwHFMrNGqCVRvlA2UQ6POkud1nTvE0EcVR1gU7JNSCnrPrWLRtw+RM7BKBXnJDP9eOYqogVNAj0Av0uTk7mtjov2+1p2yQ0hIYXnXCs+qEzF+HC9YSyIiIsK84XWTKP5tvPHdi11GupSXHW8JNW+FMAHdclSCCKDEX/iKdDgotRY17jTu31LhvHybT5RGPin5K3NWs1c0yW+lp0umc/T7b383NUdHJa44rSfJU+Qf54n/iNzi8zBtL0z1zAAAAAElFTkSuQmCC',
money_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKiSURBVDjLlVJbTNJRGOehXtl6dD20tbV66cGtrfnUU7WW09Sl3TRsokucuGGZInIXRRAFEQhMBNMCUkEDuWiKrTC1uTJvqVQP9dZDDW1p/TrnT2u5XNbZvp2d75zf5fvOxwLA2i2aJuoTqjEZe6e7XcHqqCJhnWmHclSCumA1+78IGsblidszbQi8G4R5yoDaQBUqfRXsfyIglhOWaQr2wbXihP/NAPRPNBB4y8FzF7P/SlD/SJowTxsQeOvF/ddOOBatqBoSYCjeD82ECtddXBR257N3JFCOihOmKT38BKyeUEI5LoF9wYLOBTPuLnXCF/dAGhbhiv3C+h8E8pG69fZnrcSuF6ufl/BlawO9s91QjNUlwWseSMJCXOrM3dxWgmGy+ZAsIvpqnGxhalVHlQzYFDUi/nEVqqgM3jU3xCEhVUZ+10Xk2XL2MASGSa1IRiy1xXQM+N6yg9gWM8oULB+WJMHBGqqM2IfHqB2+hWxLhoIhoJYMsWY8JA3qXe5iar0zb4Kc2FaNyzCw6oYoWA1eVwEE5gIUN2eioPEUMhpP4Iz+5D6WYkRsJwHbcxMG4w9QOchHx6t2OBc7CNjFqPG7CyF1F2JozojZ9yHowqXI0x1FWsUBJ0sxKk6hLuiQmKZaGbuOBVsSTHJcDwcc9Wl4X7bAO68HXZoIF7pICY7zUjaYJtb4b1jtMzYIfHzon2rRv+KC0H8T561ZEBKSdGEq/HM2/L58L4yUIPkVlT7+YWm4diu8FkCZpwSEEFnmjO+k08M5lsyatLL9G02ha1CFOAxYFeRsd1DeV7qXTFc/t4fzjd/HwzlT+mauLfsY6fRBek8eKnI0R6ANFTHKdKdnktf8GqKiHk7qVcflafJVZ3cab/K4gcQnavvn3kDzPwBTBMCdhxN/5wAAAABJRU5ErkJggg==',
money_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK4SURBVDjLlZJdTJJxFMZp6C2bd+bWhTXXReuirbtuWuZFMyt1iX04bWlfNmlqoqCCfDZSS1ABwUQ0NSQVVD5ETcCVqLS6yK8EslpttdlFoS6Vp//7urVaLuvdzs357/c8zznvYQBg7FR3vbKwfEzE2u5tR1jhkYR1/kZIR4WodJax/kvgjlscbvLXw/62H5opFcrtXBRbb7H+SYBEDmunKdgK02IbbG/6oHxajSJLAfK7r7D+KiB7UhXWTKtgX7Lg0es2GOd04A4UYSDUi2qvHNdNebjUnsXaVkA6Kgirp5SwEVjhlULqFsIwq0XLrAYP51tgDZlR5arARUPmyh8C4pHKlcbJOhLXgsDXeaxtrKLzRTskY5VbcNAMoYuP8y0Z67+NoPLVJoiGK743+O7Tsyo8UhpWexoQWg5A7hHBEuyGYIhPOSOr9RzY+vQoWkDlq6kQkUj1E/douGvBSGILaGcKFjuEW7CTRzlj4uM4yh2lSNOektACVCTVRC0GyYI6F1rpWR/MqCEmseVuEfoC3ahwlkElS4Q7ex+GE6MxxI6FuPAATiiTYhiSEYGBFPTP1egPPUZxPwfNrxrRNtdMYBPt1lSdjJclR7A6WIPIjAPhrkJM5h/cbE2PUTAko4LdVArqSNRTdXRc46x+Cya9PHMOHJlxWCEwlCkANwaQxeOz4iicx5lBeok8222dwa9HkZUD5bMa9C6awLeV4KwuFXwiMnwsChF/D379vghiqX6EFii2cvZXuco3XEE7bpqvgggiVXMqQjbtSNee5jmTmO++NV8GCLRWwsAyqaVrTJD+e1qgoOdGNLmu3ryOnE1OTz7OqE+uZ+jTDpNN76Xevew4oS83Yf0DNx6fyqIRyN2FkZSoDUcSk/fziHI7cg5lGy9Mk1+VvN15j7P3lBLHEBWbSkTBVP8HrJC/O3IOxUMAAAAASUVORK5CYII=',
money_dollar=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIISURBVDjLlZPRT1JxFMf5E+6fwL9Ri8lsI5cSV7swL82w0SZTB6zWXOuB0cLU8HKhuAooTNrFupcAsYWjh1sRIaDgTLGXxmubD2w+9Prth29tXMWH83J+Z5/vOd9zfhoAml7h+mg3ReuhUxIdR37CrVanUXvgvvsOtk4kbJ+kEaos/bkSYCZv0wcri7/zrTS2f32AUOX+2nPWACvd1V4KmM7fnxQP1pE+2kSuJUM+EpFpvUOS5MJVHgQSuBCwWuU72eP3EA8TWCx523NFl+Iv+zrxRgRr+wKeFJ1NVYA9y+o3mjFskbkj9SDGpTGqm2dSJmosZfRYZXPClLxNqQJsGYt2bS+MbEtCF2SVmQCTukOPikaqbxPnik4l3ohC+ilivbGKcC0Af/klXAVHczhuoC8FmDdpyl2YUrjyAlmfHytklATpJronwP9jAYbYIN3XHXTDuDGkJ6qeRzsz7XCNh1AjvshmRRXQnZWVmIQxOfTf5RFV/fw3LyJkC+6d2U5PwOjbEe3Tz4/bQp0/b92WY5VbsZtuQ3SQfpC71+R3/eAqr2ASR7I9AUSVepibUHhSFCVKQv31uXm+0nPwVQ5dgOfLM+jeXNdf6AFRnZz9NNVeKs8jtr+CCDHvRcmL8bSlqQtdo/v+TBaZ+RrcXUaQqLMZy+GVf+OAcGPaWXCckW7OBgTdslrdPxtwvK6n/CCRAAAAAElFTkSuQmCC',
money_euro=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHvSURBVDjLjZPfSxRRFMf3T9g/wRdRkfKmRbkq3gfpYSvUHiIVZHqRiH7MbogW2GD1KOxrIVuUoOiKI+2isMleJXPZ3Np2NegstHNpGJg7K8yfcLquUII7sz183+7ne875nnMDiBjwEmz0ECkKqRCFZHew3pv64GbvkJkbN+zSExTFp1LTaBciWE72xUC/HPQ1kBUVcTiDzo9ZCUWRbw8y8/OIIb5Po1Oawd/bwwVPAwk32cUpdA6e4a/0wFv4cOVvNVi7NGRl77iQ6NK8DZIh1TnQ0MyOGnVHW+kkdTOAVE+wkgnrVn7CPfo5h8ct88wNxreuM/7xmlSYVTYGdM8Qy5t9Mbs4idXDF1IvURQmUXx7LBVFkY+g2FcRlojmuwWZLrGPAQlD4iKVs1JY7qSwdEGKUK9VB06FROyvkVpVOauET0BY7CB+t3IKVrFa0rBa1Goti/2HKHIPEBbOq40NEl0KT4eZtTvmHpWeo/VpxOWpq6yy3q/7wWfXuNihiC9RtHZuu/D+3JnWYb5VhfkW4nuJ5tawUc1PoZW55ZYXSAzetFGItyl8jTJn7x7aO+MIr5ubvE/5XTsx04OGyN5HJydD3Z1AsXcXnewjtNI3XQkrDT9TzSjeqlSWQzpfpYyv9rNyvD0Gr/5Vbmjwv/oDiJrRGbut70sAAAAASUVORK5CYII=',
money_pound=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHHSURBVDjLpdPNTuJQGAZgL4FL4BK4BFcuNEZ03Mwk41+MMbNQok7UqCkGJWrGiZKiYJXKERgLpUVEIagcULSTii3KxC2X0Et4bXcmisq4+DYn53ve89sCoOUz9WJgnJXs7nBeJrlb8NlbBFKKMelL84PLcfu7wJhPcnDHipEs3SNz8wipVEPq8h/+nOnYjJeNb+6Y402Ala7qyeIDhEIVfunaWODydC1arB/kNERzOqbYLG0I/FgXnbEzDfJlDV5S0PuXBJs1/pWJ2ZZ4WuczFbAJBT2TxP4qMLKWYA4vdETMtD6PMPB8Uu9MtPXLFGG6XcTVNRa2vQoMeeOuSF7DQVaDmepq+ha+ewQHl1YRv3jAr2jJaBrYEhUzXYdYqGEnpeJ3rGxCZaySMkaWU/RdgE1cIyirIKfWid9jW1TN5it4+RIGFz8AWNU9QZxs4i+2kyo6R0NM0y9xdCVN944q2DxU0D4cGvgw4BwP22f8+bpPUEBOquDkO6xHbuAOUjABivktijl/AR3DPN9wBdZeSaaK/WMdobSGvSMNu7IGTrpD0KyAWMG07xwNgX5Gph6u+CJ11myyGqc3zvFz4w2grW/H9j/f+Qn6r94u36IRBwAAAABJRU5ErkJggg==',
money_yen=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHESURBVDjLY/j//z8DJRjO8F3q3d9ypOF/9b7y9x7zXAXQFTpPcaiv2VX+v2Bzznv7HlsFDAM857sJFOzIuT/hRO//wMW+81E0T3JQAGls29f8377Lph6rC0DYfZZLQOfh1v9tB5v/u0x1coCJB8zzW9+xv/1/xOKw8zi9AMPBSwPX9xzu+h+/KhqsGOjcgMZddf+rt1X+t26xNCBoAMS5eUDntvx3meDYn7Qy7n7rrsb/9h22/XgDERkDbS1o2d3wv31vy//+A73/S9YXvbesNRcg2gAQLtlU8H/KoUn/e/d1/89YnnafYDSC/T8vcH/ppqL/xRsK/tdvr/0/6cCE/9MPTf3fsr35f/byjP9Zy9L/5y3PeYnTgIBJ/g1+E3wbfPu8G5IWJR7o39v3v3h1wfvIuZEHnJudGhwbHBrs6+0aiPKCe4dbQ/XGqv/Ji5KeOzY6NBDlBd8pPvtzVmb9z16Z8b9hc/3/CXv7//fv7vtfu6Hqf8r8pP9J8xP/A124D6cBbm1uDa6tLg0g54bNDD3Qs6v7f/ayjM9BUwIPWFdaNViWWzZYlJsT5wW7WtuGnGXZ/8Nnht23rLAkzgsU5UZyMQAcp633iiwCvgAAAABJRU5ErkJggg==',
monitor=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH2SURBVDjLnZO9a1RBEMB/++5d1LxgPjQYkpxdLEVItFEU0mgay/wDCtfpvyBpIgommCoHqSy0s5Jgo2kEjRa2R/CrM0HzJd7lvZndsXiX8+VDFIddZncZfjOzM+PMjHq9biEEQgh47/Heo6ptXdwi0taTk5MuBvDeMzg4zOb2NphBvsAMA8ysfcaM3t4eFhefAdAGbG5tcevhK/5F7lfPk2XZb4CqtlzCcOU0XZ0dVPoTjh6J2RFDNCAevMHr5fdYMNI0BSACEJE8TMA5R2NH+bq5QyP1mIF6I9NAqrmNGXsjEBHCbnzO4ZzjR0MJltKTlInjElEwrAUIFg4CLOQIV4A0M08mgaSzg1IpInK5DwuHACik4JzLIYA5RyP1uMjydyDsTyHLMqJSBMCXz5/+WoVSXDoI6CjH3L15Du+NcrnM2uoqSddxzIyNjW8MnBqg2Wy2fj4cADxYWnp5pdBlfSLyYmho6IaqsrKysqCq4yKyXujKpwBut3xFmZmZua2q1y9eujze3d3Lk8ePmqp6Z3p6+t5+2z2Aubm5fhFZGxk5Q9+Jk3h1iCpJcoyPH+osL7/5OTs721UExMVLmqZjlUqFiYlrqCpmhpkRRRFjo2dZX/+eVKvVq/Pz888PjQBgamrqnYiM7p/C1mS+rdVqF/6Ywv/IL/azYiamBPboAAAAAElFTkSuQmCC',
monitor_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJGSURBVDjLjZNNSFRRFMd/970ZNbX8KCHUCVxM4CYSDaKkoAgywk3gPoyCyFq0cW0bSUTIcNEm2uWqUEIIrEUqUoFZuZAEFcPGD9LRGOe9+3FajE5jCnW4l3O53Ps7/3PPPUpEmJ6eFucczjmstVhrMcZkfe7UWmd9S0uLigBYa6msrGZ9YwNEIDNABAFEJLtGhLKyUoaGXgGQBawnk9x9NMr/WNetU4Rh+AdgjNkOCdWxYxQX5hGrKKIgP0JaC9o4tAUrMP7+E+KEIAgA8AC01hmZgFKKVNqQWE+TCiwiYKwQGkdgMmdE2K1Aa43b0acUSik2UwYnAaVFUSIRH88Jsg1w4vYCxGUQKgeyFVpC7SgqzMP3PTyViSFuHwA5KSilMhBAlCIVWJQnmX3A/Z1CGIZ4vgfA/NzsP6vgR/y9gLxohM4bJ7FWiEajLC8tUVR8CBFhbW2VRDDD5MIom1tJugaeIfZARrGI0N/f3621Pp/zy8q11m+qqqpajTF8WBieKKmxdfW1DVSXx3k79ZLxr++Ym1ntVjvly7Wenp57xpjms43nLpSUlPFwoI1rV6+A79Fc20b38E18PJ6/GEhHci/29vZWaK2X4/HjlB8+gtGQSKzw89cKUVVMU20rAPcvPmHwSx8wULALEARBQywWo6npMsYYRAQR4cFgkqnFMSYXR2i/9JTO19cp8PMB0ntS6Ojo+Ki1rs/twHk9iVfzgzMnGokfreNbYoKxzyN8n03u/wb72ek7lZ3AbeAgsAn0jT9ebP8NoAhq3YVujicAAAAASUVORK5CYII=',
monitor_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJFSURBVDjLjZPPS1RRFMc/9703ao40jqbkj9EkjKBFQdaioiCKskVLqXWBJFTQIqJFhYsIgoRctG3TPxCIoFCiCZERUisXplLa+IPUkcZ57/44LUanMYX6ci/ncjn3c865514lIkxMTIhzDucc1lqstRhjCrZ4aq0LtqOjQwUA1lrq6xtZyWRABPIDRBBARAprREgmK+nv7wOgAFhZXeXW81H+R087jxFF0R+AMWYjJDSmmqgoLyFVE6esNCCnBW0c2oIVeP9hHHFCGIYAeABa63yagFKKbM6QXsmRDS0iYKwQGUdo8j4ibM1Aa43bzE8plFKsZQ1OQirjMYLAx3OCbACcuO0AcXmEKoKsR5ZIO+LlJfi+h6fyMcTtAKCoBKVUHgKIUmRDi/Ikvw+4v0uIogjP9wCYmZ76Zxf8wN8OKIkFPLl+BGuFWCzGwvw88YrdiAjLy0skvw6THXmF/jFD+mEtDWX7twCeDQ29PVP0yqq01m8aGhquGWPwhl+O1ZRn2o5fvaFKWw6x/nmAxMigGzwXu6k221esnp6e28aYyydPnT6bSCSZvHtCznfdV7smh2D2HSQqWQqa+TQwMhUUH+zt7a3RWi+0th6gqnoPRkM6vUg8l1Fle1vg0p2Cb/CoDt+pfVsAYRi2pVIp2tsvYoxBRBARxmrryI73EX/dRbieJgusZXysz9y2Erq7uz9qrY8W/8CDq1847H+juVoTeN9ZWzRMz/tW5+TBjnewk0avNN379XOu07eq2foyK/DiwoB5/BuCJmv1SL6PpQAAAABJRU5ErkJggg==',
monitor_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKTSURBVDjLjZNLSJRRFMd/95uHOmrjo0zM0RTR7EFvg4oeEtQiKoRcRBRFaKs2QatW0cIeFCRUWERBgYsWFYQUUfawh20GW+SDHIcSdaRSq5n5vu/e77aYnD6JosO9nHvh3t85/3vuEVpr+vr6tOM4OI6DUgqlFFLKtHdP27bTvrGxUXgBlFKUlJQyMTUFWkNqgNZoQGudXqM1+fl5dHTcByANmJic5MiFLv7HzjSvxrKs3wAp5a+QUBoqIyfgJzQnm8wML0lbY0sHW4HS8Lo7jHY0pmkCYADYtp1KExBCEE9KRieSxE2F1iCVxpIOpkyd0ZqZGdi2jTOdnxAIIfgWlzjaJC/bh9frwXA05XSzZ2UHnoFBNuU+pefilX1pgHZSCOGCJCyFZTtkB/xUZz1jS2GUrNl7CVYsZyJSQ+/Dq6fTAFwShBApCKCFYJ7xnG21MWYF1/Hlw3v8wiJ3VjE5hWVBL6T0GB4DgOhQZMaLLyocYPdai7zKbZgj7fgDgmi4l+8/JNqaWpoG+H1eWg4tQymNz+cjNjZGnmeAUDBBsGoHyeHLGH6JL6ecDBnldmQex1uu908DznV2Ptno+mUFufEXkepNZfXBql0khy9h+CTWVAUjnW/UvU/l/bG49xaAmC6f226eari2Ymnl/gUbdxrW2A2ExyI5Wcr4s+6ESvxYWHP01VC6aG5Aa2vrHPE5HKtfU4wURRQWdJFf5Cc5GWK0q4f2wVD8xNnL2e5ghntjmuaquYEoC9c3kfHxBf13uwg/+EL/4zDlDecpmr840NzcvNV95w8JJ5uWOMcOHhYYivCjOwwORXiXtZmvdiZSyrdtbW11/wQ0bihIrKic662rrUD7zJeJ2PiB7S09g39rrJ+Ib2CJHdkBMQAAAABJRU5ErkJggg==',
monitor_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLjZJNSFRRFIC/N29m1PEHNSUc/5WUoj+zAXe6ctmiQNtEFIWLoAhXbYcWkZCge2vTqghKyk2oREVa1hAI6SINQhpRGs3Gee++e0+LGZ9jtuhwD+cuLt/9zr3HEhEWFhbEGIMxBq01Wms8z/Nrfiql/NrX12cFAbTWRKN1pDY3QQSyC0QQQET8PSJUVJQzMfEcAB+Q2tjg+sgb/ieGBmK4rrsL8DwvdyXU1TdQEglTX11MYUGQjBKUZ1AatMC72QRiBMdxAAgAKKWymoBlWaQzHj9SGdKORgQ8LbiewfGyZ0TYa6CUwuz4WRaWZfEr7WHEobw4RDBoEzCC5ABGjA/YNTBZhJUH2XY1qymHSubprbhPxP6dNTD/AJDXgmVZWQgQshXHi6ZobY3RFnqdM/irBdd1CdgBAL4tL+158d6WLzS2dFFW00Fz4hF1Jd3YQXs/IBwKcufKSbQWQqEQq8kkpQUOhclxSqti6M1nNJ26yED4JQHp2Qe4Nz091Z03ZZWecqdi0eXLDSfOQuYjsw+f0nX1FrL+mcnE3a9KlY35gP7+/sF87eHh4RvV4ZVLNQdKKK1IozeXQAx66z3NPYOoF7edWLkZ8QE7MTo6Wq2UWm071Ep06xPlDWcw6RnEbNN5rh3jrlBYvMjBY+cPf595cBOIB/IBjuOcrq2tJdbkUVXXQaR0DfHWwbKZe7IIGExmkeq2RrSjrr2KHz1i7UzgTsTj8Q89ZZOdsQtj2IFlxPuJP+e5n7eL2knOJ1gcjz/eBwB4O9S5JtpExAhiTF6KXzEGEZn7A0dmdT7nVaIZAAAAAElFTkSuQmCC',
monitor_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJKSURBVDjLjZNLSNVBFIe/+d97LdTCK2ki3qJLBipE9KBWtZAkW4ebFkEF7nLZroWQBBVGbgpa9FhJ65BCIUGxl9EmwSJLKzMl8IF2//M4p8W16zVbeJjhzGLmO7/fmRmjqoyPj6uIICKEEAgh4L0v5OLpnCvktrY2kwQIIVBbW8f84iKoQn6AKgqoamGNKul0BX19TwAoAOYXFrh0e5jNxPX2I1hr1wDe+9WSUJfZRXlpCZmqMrZuSZJzivOCCxAUXrx6h4oSxzEAEYBzLi8TMMawkvPMzOdYiQOq4INivRD7/B5V1itwziF/9RmDMYalFY9oTEVZimQyQSSKrgJEZSNAJY8wRZDfNmCdUFZaQiIREZl8DZU1QMECRRaMMXkIoMawEgfmpJ1pdxYTeeRfC9ZaokQEwOSXz//tfFVTzO7qBhb3XkWixxsBJakk1y4eIAQllUpxZ/Q8Kg4rHus9tZX1NNQcZSm3zOX+FhrdmbxiVaW3t/emc+5E0SurHNv+cM/JpnMEFYIEBGVm4RvpsmpGJvoZnRrUnLdp8/f6iqO7u7vjffrRrdP7LzD56yNOPF4cLjiiqITqbXUMTTzl5eehkCw+2NPTU+Wcm62v38fozxw+eHZu34WXQFDhx8IUleU1vP06zPCnoTnraVkHiOP4cCaTobX1FPfvdfBg5AZWLDlvye5o5Fi2hdeTQwx8eDZjPc3TXTq2wUJnZ+cb59yhf3/hYPVdDu4+zvMPA9+t0DzdpeOFJm4mslfMcpBE6W8J2dkuLdz1H8DrioJLLPMsAAAAAElFTkSuQmCC',
monitor_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKSSURBVDjLnVPdSxRRFP/N7Pq1U+q2an5tPqSVPQjhF1hYrElJBBE9RPQY7Ev00h9Q7EMLRSAGhUIP0UNqQkaYCOVDUJEZ9iDqurhGsaDpbOvq7uzO3I/uzK7rWi/R4f44Z2bu+Z3fmXuuxDmHaYFAgDPGYIJSaoEQkvW5MAwj6+3ImLmxuroW0VgMMEnTy4q55Xg2Nr3TWYrx8bHdBNGNDdzoe49/sXveVui6vkNgp/NwREdx59wWbLINjDLBKsAIknohRlevI85dkGUZi/Nz4IwjlUqlCdSvHlsBaYfDVYODrnJwg1igugbt1wJe/uxCgjkhyRIgSch0YSmQrQfK3AqbRl5RHVh8TmxUhAJRObqIV8udmE90gEuyyJUsmMY4SxOsfzmVLwiuFZUeFq+FdG1FqE5hbXkCY6E2zCZO7iSaPtOyOLL0P2CUHuOEeQv2NoATTXBUQZyPWDJK5FV0OAZhYzqSVEEQp5GQKjIK0i3YRfKDQldLmaYuiMoM+Y568dWOqrpulKpBUENHNDyD598vYDb+AzpftQhsdltGAaHD8fDHd2Gt/WZlRTliwddw1naBJKPQE+vYIAeQOvIIHXV5uLi/EpqmWQQy0i1I25M4NDR0n5NNT2thX5OjuF6OrCyR5c3GD8mKS53m1AWDwcfCe8T0RXKm8kWWwLTQyNG3+Y5qj6Z+G55Ur87EDKX7+IlOT0mJE4PPnmoi4bbf77+bO1BZgtBIYw8z6MMtvfjWpHrlSUPDIexzlYESCYaopihFCC0FMDX1Kd7b27snO4DbgUgOC5x/s3bZ7XbXoKfnrCXTLGDCnMCW5iZEIqri9XrP9Pf3T+xSkGs+n29a9Nr85y3M3MzPAwMDbX+18L/2G23Jn5HeUDltAAAAAElFTkSuQmCC',
monitor_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJySURBVDjLjVJNa1NBFD3vvXxhWprShrSxbTbGYEs2hUoXBaELaTaCICEQJd2Im+JfcF2EUgT/gH8gKymIttkIIlRS3dgs4kfSpMFUmqRJXvI+xnsnJiZ2oZc3794Zzpx7zswoQggcHx8L27bBw7IsOUzTHOThYRjGIMfjccUBCgYGg3M4r9cBIkTvk7WQSQxqzpOTPuztveQZBgTntRoeP3uL/4mnj1bQ7Xb/ELCkHj0wN7+AsSsuzPu98Lgd0A0Bw7RhWNSIMO/eZyFsgU6nI/Eq/9iPlEmhKApauonTcx2tjiUdmbSzSyQds4fhtREFTGD39REBkzRaJmzRgc/rhMOhQaWu4jeBLezLBHQNvf1DJO2uha5hw0uWNE2FqvR6MPYSAYYs8JAkDGZLZEVRRW9dKvjLAk9UTR4Hvn398s9b0MhSn0Dhw0un02Jl5SbqjQbabZ0e1mcUC9/h9/tRLp/iWvg6Qgsh7O+/xsxMANXqGYrFArLZ7FhfwU4mc3BL13Xl6OjoajgcDqRSKTSbTQIWkclkGp8+Zp3JZNLtcrmUcrmMw0MX8vn8E6V/fRzLy8t31tfXXyQ3Nyee7+6eVSqVqdnZWUSjUfCmarWKUqlkjY+Pv9ra2oqR8po67M3tdt+eoOD6/oPUFD+WRCKBtbU1xGIxbGxs4OLiok1xjzEEdTuGCXw+H6irrDMHb2Te3t6G1+uVNVtSVdWjaVqa5ycnJ/qIhcXFxRtLS0sfVldXPcFgEB6PRxLmcjlWh1AohEAgwE//Z6FQUOlsTkcIIpGIRp6j1DFNBH7apPFzJ+8PueH09DR3dpI1i87iBym6+wuQ3GjC/VvaWwAAAABJRU5ErkJggg==',
mouse=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIMSURBVDjLpZLLaxpRFMZHCEIp/V/SfaGL/gXdZxMIpd0kgo6jINoH6iIIBd1kFRdmoZUOs4pVXDhKfJCA+AQfAz4QCQSf41tP77lwB2mTLtILZ2Zzft/3nXMvBwDcc0uW5XfPAtPptD4ej9skSeo/2lCpVF6VSiVXPp+/zWazEnF6s+f6+pqccDis+v3+v+FyufyCwLedTgcmkwn0+33I5XIQiUR+xWKx78RVqdVqkEqlwOPxXP3prC8WizetVgvw7HY7YKdarUIoFFJJAvB6vQ9ut/vUZrO9ZKCONHwoFAodRVFgvV7DdrulAljD4RBUVYVerwc+nw9cLteR2WzWI4uRdaR+ttttGhkBjI3Nq9UKME29XofpdEpFo9EoOJ1OnqXmyKI+s8gsNiZoNptAkkG324X5fE4LBXF+h8NxrQlkMpkURmRx0WWz2cBsNqNz4zIxPqZDYWIIVqs1rgkkEon75XKpgeiyWCyoIwojgH+EyZWCxWLZGI3Gt5pAUhTv0Q0bGMgcR6MRFWw0GiCKIgiCUCbw4f7NcbmLCxl3gO77ILqymcl13RH47LFHx9UF86V6fg6rZJIC4/GYFo6FQoFAAMiVvX/qWXOtk5ODxtnpzezbV7o0dB4MBkDeBASDQeB5/ovBYNA9KYAf5fj4oPjp46UkCPIVb3qw2+09sukfJpPp6F+wJvA/9RsZICZTCkof6AAAAABJRU5ErkJggg==',
mouse_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJrSURBVDjLnZPda1JxGMcVxiCiPyD6B7op2FUE3UR0EQRddBHBGDNsvbKE1CWJtpqewbSNdMUg2IjtQieNE5XuOC98Yb6wC/EVNAVfEBkMN+fxXZ9+z4+OSXUR+8HDj3N4vt/v53k4RwQAopOWx+O5diKh3+8fdblcapZly/9sSCQSZ2KxmD4cDu8Fg0GWJF0ZSh2zk2Oz2fi1tbW/xfF4/BQR7xUKBTg+PoZyuQyhUAgcDse20+lcIqnZVCoFPp8PjEbjxp/Jo9FodDeXywGefr8Pwkkmk2C1WnlCACaT6YBhmGm1Wn1aEIpJw1QkEilks1nodDrQ6/WoAdbh4SHwPA+lUgnMZjPo9fpxpVI5ilpEFpP6nM/nKTIKEBub2+02IE06nYZarUZNOY4DnU6nEKhFZFGvBGQBGwkymQwQMigWi9BoNGihIc6v1WrtA4NAIOBDRAEXU7rdLtTrdTo3LhPxkQ6NSSCoVCrXwMDtdu+3Wq2BEFOazSZNRGMU4G0P2uD5hwmYensL7s5d5W+8uKimBt6trX1MQ3dBKCQeHR1Rw0/f3sPspgS+xpYhXOJg0fkY7ixegEtPzhpEoZUVD+4A04eFmCrMPMFc77HRJWAT7+ieDDv3YXHnARo0ROkZ5Sq/sABtr5cKqtUqLRwLjdbX1+HmyzH4HvsIw+dLZBkNQJSTSkd+PJverb95TZeGyZVKBcg3ARaLBRQKxezlp+caC9w9YLhJKma2J38T4CKyEslI9NHDVXZmxrOhkB9oNJoS2fSmXC4fl8lkYtI4d9twHoyclCbjjc90B//7B5LmeVJVxP51z+P7nwtGFih8vipxAAAAAElFTkSuQmCC',
mouse_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ3SURBVDjLnZNPaNJhGMd/ohkRQd0i6DCoY4fOdYjWOgQ777LLSGIRLQn/zJC5GnPQECK97FDbwUFbw2EQ/lke/MM2bYFNpzDTcIrYYjmdP3X++/Y+L/zGqA6xBx5+8PJ+v9/P8/D+BADCSTsQCPSeSLi2tqb0+XxGp9NZ/OeFRCJxLh6Pm6PR6EY4HHaypJvHUq+7WC0tLYlzc3N/i7e2ts4w8UYul8PBwQGKxSIikQjcbrdnZWXlFUvNbG9vIxQKwWKxzP+ZrIzFYqvZbBZU3W4XUiWTSSwuLoqMAFardW9qamrEaDSelYQyduHB5uZmLpPJoNVqodPpcAPq/f19iKKIQqEAm80Gs9k8qNPplKQlZBlrx87ODkcmAWHT5WazCaJJpVKoVqvc1Ov1YnJyUitRC2xR4xKyhE0E6XQajAz5fB71ep03GdL8JpPJdWSwvr4eIkQJl1La7TZqtRqfm5ZJ+ERHxiwQBoPBd2Tg9/t3Dw8Pj4SU0mg0eCIZk4C+WYcV/sEr+HRbAdfd0z88ffJRbhBcXt6lNHKXhFJiuVzmhl/fTiCquYH6Rwu6CTfEd0/x+dG1trdX8ViIzMwEaAeUflxIqdLMH+6db9SYGK/7Af0FwNyDny9vwXNHnhFSet2sOD2NZjDIBZVKhTeNRUZ2u52wu90vDhyvkukiPxeyKpXi25OR1drEC740Si6VSmBvAgsLC9Bqtc/ZvLnqm/sAEzV0An6xzg7Lwc7zfJOZoSFF7OHwrFOvD8xrNXtjY2MFtun3Go1mUK1Wy4IDl8bDqqutgr4Hu4ZTSKtk8PUr2u4++bP//gNDA5dHWeJ3wiYiEtP5byK4FJQG5P/0AAAAAElFTkSuQmCC',
mouse_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKoSURBVDjLnZPdS5NhGMYVRIjwnyiKPojSTiKCCP+EDoqyA0H6OrAI3VjOVso2STTLQUmUI/REVzbQXNqqvVu6LSvbJ+7L3k3Wmpturj37dFfP88aGlEc+cPOe3Nd1/e4L3ioAVTsdjuMadyScn5+v1ev1Uq1WG9l2weVy1TkcDsXi4uKCxWLR0qRTW1Lrp+nTaDRptVr9v9jpdO6i4oVQKIRUKoVIJAKr1QqdTvd2dnZ2gKYGPB4PTCYT+vr6Rv9NrrXb7XM8z4O9UqmE8nO73RgbG0tTAgwODsaVSmWrVCrdXRZW04XLNpstFAgEUCgUsLm5KRiwSSQSSKfTCIfDUKlUUCgUTSKRqJZpGXI1nVfBYFBAZgKGzZbz+TwYjc+mRcjSiXxmHTMzM5DL5e1l6ipa1N0ychmbEfj9flAyrAR94E23EPeO45dTDXa/TCabrhiYzWYTQyzjMvRisQhCiHD38pdnWHWPoPDbh6XJJny3TEEikegrBgaDIZrL5SpChp3NZpHJZBALu+DWXkB2dRLZcD/W/FPQPTlXui2+2VgxME5MRFkawy4LWWGpjQ0sm3oQW9IgG+oC13Mc+bgG3543lr4+PXO2YmAdGuJYByxdENIik8kkIl4DlvUdyEVfgPiawSnrkeHFSATewKo66bI8PFEnGHjFouF0by/yRqOAv0GTk4k1uF9fwTotLvOjDWnP+b/jvURPGQBveIRP9xvuCAZ8S0uN70brHOnuEkpjhfKfR8Fz/chFHgvpae9FekKD8CWB6yA/teC6j/wy3Dt8SLgj0NxcY792dVgrFnOj7W1x7sHpIon5kVvTIxulHUTHt8xLFFIOrJhH8L5j//i2PxPFi5mUx4hRfpTQJEKTyEfZQfKh8wChIqKX7CN68V7yTrTH+Aes5yJtFoFr+wAAAABJRU5ErkJggg==',
music=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAETSURBVBgZfcExS0JRGIDh996OFIQEgSRhTS1Bg0trw937B9UPCAT3hnJ1kYbGhrv0BxoaXSsMhBCsyUEcoiTKUM/3HU8Fce4Q+DyRZz5DcOkdiqIIiiAo7xiCMXs4HI4ZisPhOMcQOJQbOoxxKHm22UUxBBbHM1cRfw58GUtMIAieTIwgxAQWRclMEZSYwCIIGYsixASCYsl4pgiGwDFF+HWUaDopbfCGHRp+nCWSTktFXvFDOKyuNNYp4LhFriPPaXW5UWAV5Y6HNH+/dbHJIjN6NHlJzMnxWqNIDqFHh8/U7hiEJbp0+ar0m2a4MGFEjie6jCrtJs1y57FuI21R6w8g8uwnH/VJJK1ZrT3gn8gz3zcVUYEwGmDcvQAAAABJRU5ErkJggg==',
new=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEMSURBVDjL3ZLBSgJRFIYvtO0BfIPeI3qBNj2Cy1rWzlWbkcBNYhC0TletJKOFq1lIILhQJCywaDZOkINiGl/n3DNj6LaF4MDHGebc/5tz544D3H9w2yAI3LkQp7UgREJRSIS+0BJqwr6QTzkWulqdD09juD3Ah5PI7r8TiPvw0YJeDUq7cJ83NDzqwmUOFUyYT/ASfasGm6d4kQo1OB3JszN4fTDujuBrqP2hW4baVxbMBIuZTfAeQucGxm/w+WzB6AleGipo/Am06hTrEwQupLhjwkFdtlOFnzlc72n/cFWgQb3WJ8i22a7A44mtCfQQ7BSyL6617BtWZ+kphMKFlwSusrJmW/7ETQt+AQhq/TxibW0lAAAAAElFTkSuQmCC',
newspaper=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIkSURBVDjLjZPLaxpRGMWzsHQTpLvusgiF7rvuqn9L/4Muu/UxvlAUFa0SUWsU32+jg0iLi1KIohhREWnxTdU4aG2E5vS7AxHSWJvFwJ3H79xzzv3mCMDRoSudTksTiUQ+Fotx4XBY8vf7x8CFaDS6CoVCQiAQ4Hw+n+RRAqlUShqPxxm8Ho/HWK1WIPjG4/FwLpfr6UGBOzgSiayHwyFoDYoAQRBwdnb22+FwJG0225O9AslkUkof83cwA8/Pz0FxsFgsRCcECxaLhTOZTJK9MJW1HgwGIPvMNsgRLjsD6NMNxL5+g9lsvjUajSW9Xv9sB1NZUgJ4KkuEyQG8Xi9IFLPZDOaLJt5++IJ3nkt87w9BsKDVal/vhckBqCwx+3Q6RalUwo/ZHO8DFQQ+NZHgP0Oj0Ww5jnsjCtBufDAY3E4mE5AI3G63mJ3BJI5isYher4dGq4uPcZ7BNwS/2pVIcN7v99/2+310Oh3k83kRZg4ymQy63S4ajQayuQuo1eqNUqk8vXeM1LJQq9VwfX2NdruNVqslOmDNX11doVqtIpvNQqVS/ST45MEgUVmjcrksfjSfz9FsNkURtmulUhFdMFihUJzsHWWarOdOp3NbKBTEo2OtL5dL1Ot18DwPyvtLLpef7hu63cJutx9brdYp240dIcudy+UYvCH45b9G/t4NTdaxwWAYsfysQMq7kclkLw79cA8e6HS6Y2p6RLDwP5hdfwD40PFHzWBC4QAAAABJRU5ErkJggg==',
newspaper_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKASURBVDjLjZJfaFJRHMf3sOhlSG8RvfbQS0G9RNBLPRUEPfQQPQQrVkH1VC/BegkEL1O7ouhwORS1beI2///Z9TKk2EMEM3QqTkQKdU5S58VhE9q33zmkrRhjB36ce889n+/5/r73jAAYOaoikYgqGAxKfr9fWFpaGv3/+3HghM/n6y4uLioej0eYm5sbPZZAOBxWBQIBBu/W63V0u10QvOdyuQSHw3HySIEBvLy8vFur1UDPoBagKArsdvsvm80WslqtJw4VCIVCKtosD2AGzs/Pg9pBu93mTghWLBaLYDKZRg+FKazdarUKss9sgxxhvViFGMnC/+UbzGbzvtFoTIqieGoIU1gqAmQKi8PkAG63GySKZrMJ80oeE+8/45VrHd8rNRCs6HS6a4fC5AAUFu+90WggmUziR7OFSU8Kno95BOVP0Gq1fUEQbnABOk32er397e1tkAicTifvncEkjtXVVZTLZWQLJXwIyAzeI/jyMESCpYWFhf1KpYJisQhJkjjMHESjUZRKJbgiVry0PMCTd3dwX329e+v1xTdDAUpZSafT2NnZwebmJgqFAnfAks/lcrD5DHjrfYjwhhlfaxLExDPcEy/gyvMzOi5AYW2tra0hFouh1Wohn89zkWw2i1QqhXHhJgIZAwI5I9jQyY8hyk+ZQI8L0M06PTs7208kEvzXsdQ7nQ4ymQxkWcbtyUuIbthwcATTZibw9w7MzMyMTU9PN1jP7BeyvuPxOCis3tUXZ39qpUfQSOMc1qyM/+tgUHSzxgwGwxbrnwWo0Wh6arX6HG1U39Wdh16a4Cezmb0PMzhYer1+bGpqaotghcGDddosUCnM9p9ZYOu/ASUg4G4xOdG6AAAAAElFTkSuQmCC',
newspaper_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKZSURBVDjLjZJLTBNRGIUxbeOGNLrSaFxgdOnGrS4MijtWLkjcysLEuNEoqDG4IDIpNCUlLVLbQFrsI0Dpg0IfIzYiCyRQA2mb0tQG6QOqbemkFanY438nofFBCJPczOTOfGfOOfdvANBw2PJ4PHKXy+V3OBzcxMSE9N/3R4EDk5OT5fHxccFms3Fms1l6JIGpqSm50+lkcGVzcxPlchkE75pMJm5kZOT4oQL7sN1ur2QyGdAzKAIEQcDw8PAvvV7vHhoakh0o4Ha75fQxvw8z0GKxgOKgWCyKTggWtFotNzAwID0QprIq6XQaZJ/ZBjnCcjwNlScMx+I6NBpNTa1WB1Uq1Yk6TGXJCeCpLBEmBxgdHQWJIp/PQ+OLov31Ah6alvEllQHBQl9f39UDYXIAKkvMnsvlEAwG8S1fwDNbCLb3Ubj4OfT29lY5jmsWBehv/NjYWHVrawskAqPRKGZnMIljdnYWyWQS4VgCb5w8g3cJvlwvkWC/1WqtpVIpxONx+P1+EWYOpqenkUgksKTrwrvb58E3y+C9Kcv4WiSddQFqWVhZWcH29jbW1tYQi8VEB6z5SCSCj6+e49OjK9jxKFGLeFGxPsDivUt7/uvS+6IAlZWdn5/HzMwMCoUCotGoKBIOhxEKheC7dRbfCYa6Feg4CbxswlfFNfhuSD6LAjRZpwwGQzUQCIhHx1ovlUpYXV0Fz/Oi7dqSHX9exa7TtC+t1Y9Rp9M1Dg4O5lhmdoQst9frBZW1Q5mzZcMdgKAfjxtQoLV+VwLqIfXXINFkNfb392dZflZgT0/PTnd394W5tjMvFtov/sx0NCH3RIZE+zG8bZXueVskT/8bZaVS2ahQKLIECwze3//Qdq6T/phktum+wWC2/xtU6N71e0LFhAAAAABJRU5ErkJggg==',
newspaper_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKdSURBVDjLjVNdSFNhGH7OPOtnbicKXVrmbFEg05QS9hPrYlF4E9FuuuzOorsg2IXRTUO6ki6kG28SEm+aELpBkdJIndOyQJpL05HSaNkudG47nt/eb7iRJOGBh3O+77zP8z7vc77D6bqO/12RSERQVfWlpmlzdH/o9/uVv98b9kEOETyKotwjBIeGhvh9CYTDYYEIJbLb7a7u7Oy00PN9QnBgYOBguY7bawRGLnd2uVym2dlZsDqfz4dQKKTRfoTg7+rqkv8RGB0dZeRhmtnNyDMzMygWixAEAR6PB0ajEYODgzmqecYy4fciE0rkeDxeIdefa8fzqTROW82g0cxU4yKYKxmMjIywmYcJFXI+n4fFYgFlgNjKJr6kt/A+mcX1Gzc5Il8gtBjK5HJnukzT09MVstPpxPz8PJayd8Dxd3GxsQqfk8vMxSHCgZLATufLXq/XFIvFdnVmYpQHREVCQ00zXi3cwsKnCYmNEAgExksZ0IIly8uyjLa2NqRSKYTXH2P89RNImgLpt4ITR8+iuc6JnJjH2+JTXoS0GEAA/I4Dl8Ph4MxmMwqFAmw2G5SMhKuO21B1DSQPDTrSG2tobbiELalo+Lga3TgfNBwrO8jncjkLs89mzmQyJcuM/D27BJlcKJoMWZWxuZ1D+ykvtuSCIZ6ayJYzaE8mkzLP84hGo7BarSQgQlEVHBcaUSfYUH/EDmPVYdRaTmJubRKTyxPrBQkdlYPU39/Pvu0K2a8VRREvMg9QlCTKYJu50c/UODiX/Ro+rE5hbPHNT0nBlXSPnth1Evv6+pjIUlNTUx3HcUgkEiKtW7u7u7+1BA35jkaf6d3i2A9JK5G/7vkv9Pb2lkQI1eywMDLbtz/i8qpWZSpqqv1Xj54q1/8B08jE6o+fnvoAAAAASUVORK5CYII=',
newspaper_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKlSURBVDjLjZLdS5pxFMe9aAy2cLvbvzF2PYocddELUTd5464MISEIGRRGF4H0sBKpMMnZC9kqe/UlF+UL6upiDJ6BUSEisqHOXKY8s4Wy/O78flDQaC8XBw7nOd/P75zvc2QAZH8Kr9crd7vde06nU9jY2Ki5q+dfYt/W1lZpfX1dcjgcwtLSUs1/Aba3t+Uul4uJL7LZLEqlEkhcttvtwvz8/P2/Aq7Fm5ubF5lMBpSDVoAkSZibm7uy2Wye6enpe3cCPB6PnJr912ImXF5eBq2DQqHAJyGxNDU1JUxOTtbcAlyLyayLdDoNGp+NDZoIYjwNk/cIzo+fYTabqxMTEyGTyfT4BkBmyUngJ7O4mCbA4uIiCIp8Pg/z7gnUbz5AZxfxJZUBiaWxsbHnHPC7mCYAmcV3z+VyCIVCOMufQ+/4BEfkBG7/e4yOjlYEQVBwAL3mX1tbq5yenoIgWFhY4LszMcERDAaRTCZxFEvgrcvPxGUSP7sxkcR7Kysr1VQqhXg8zg1j+1MTZmdnMTIywiFOlxtDQ0NVVjMYDOc9PT2v6urqHsrIZSkajaJYLOL4+BhGoxHj4+Ng/z+RSCAcDnMYia9isViZ1fb393lPa2vraxmZ9fXg4AA7Ozvc8YGBAYiHhxgcHERbWxu6u7tBrvM6y1taWn4qlcp3DKLT6YoyuqwnMzMzFZ/PB61Wy8bjAFEU0djYiEAgwPNIJMINrq+v/97e3v6AAYaHhy+5EVartdZiseRoTA5hgN7eXigUCjQ1NaGjo4MHyxsaGiqdnZ27DKDRaAo3h0SXVUuAbyqViv1nMGPZPRAYfX196O/v5zmrra6u5smrAq14cuuUqemRXq8/o9eu1Gr1D5rmkqLc1dX1kuIF5RKr0bdSc3NzktZ5+gtxf8ZLfEnITAAAAABJRU5ErkJggg==',
note=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGGSURBVDjLxZO/alRBFMZ/c6MmomKhBLv4AIJiYekjCFopKSzyCnkGW99BbMTOQhsrBcFKsLCJhRYBNYYsWXNn5s6Z81nMGu+626XwFDOHge/PmfkmSOIk1XHCOvWn0ZdXsulPpAFZQbUgG5BlVDOURLWELEJJXLz3JMwTVOP0tfsLChIEmC2A4OD5g0UHebLLWQl5bAcBJAcC4i9D6FZRiUtGMMOHb9j0PXhGGtruA3hCnpBHzly+i5d+CUHNgCFPoDIDjcEJeQ8yNCxxYL/2m+U55Yh7mpFE8NhE7GiRwGsi7bzF8meoA8io6ZC1jfWm7AnVCPLld1DjPna4y/kbm4Djw1emH56h2oN6VFNzIKOOCI6DFCTKj48cvN6m9jtQC64yAjcXrjrnoBu/94VbDymTPSZvHs/A6RgsT0gZqC1M/46AJcJKx7mbW8RPL5m+e8HKpeusXbmNI1AFDHBkmZHzFpO9p3fkJSNLqEQsfgc6uhCQJRgy7qlF2ypXHynMEfy33/gbubc6XKsT2+MAAAAASUVORK5CYII=',
note_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAITSURBVDjLpZNNSJRBGMd/s7q50ZoWISZSRAgF5VpELFJ0CK9RneokEl1qO3URpFNCXqIO0aFLEF0iiAj6JBB0WYwOoikRJiXUYqbsumv7zvu+M/N02BV30YPgc5h5GPh/zMz/USLCVirCFqt+tZGfb8UUFxEJEBMiNkRMgBgfsT6EGms0YjwINU0Xn6haAmuIHrm0TkEEFFQWQCD3/PJ6B37+N9tFEOeVDxSIOEAhrDGoSAMSehtcwRhcMI8pfgLnIxKUdxeA04jTiPPYtucCLixtQGB9wCBOg4QVUDVYI64EYpBgAwdmZalsuUbZwzldIfHAeWUR8289gbMaPTOK8b+DDUAMVheI7W8pKzuNWA/E1byBWg3S4oteibZ0EO86DzhcMEdx/BkN+3aBlBie1YzMOZY9j6CU489K/tabOxOD9VVMhAuT5D6m2dl9FaUUTkKQEu+/FZny45w5fYL23R0MT79kbGr0djLV1hyp/u/Gk72E+b/kR+5VwBqxmtdfc3QdSmAjlsTeHqwKSR7tBri+FmWjUXURdhy/gphmiplX1MUSxFr7WCgsEVVxzh2+AcDNs4842NIJEKvKgSb37j5iNBJ6BN4XmM1Q+vyUQiFgOpthIpumv+cxQx/6iNU1AGi1mWlMptoG2w80DXR3nqKj9Rgz8+NkJtP8+rF8V212nJOptiHgGtAIFIGHYw+y/f8B3ntD1Kp2NbQAAAAASUVORK5CYII=',
note_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIJSURBVDjLpZM9SJVRGMd/532vetOswQgjSoW2gj4Gh4ZoiIKGwGi4EtQQ0VxLQhAOEW2NzUVLOQRBBioUakGT5CBiGhZe8aameLv3nPc9H0/DvX7hHQSf4ZzDgf/v/3/O4VEiwl4qYo+VWT/I7EdxxSVEUsRZxFvEpYhLEJ+ANXhnEKfBGg5ef6W2A7yj7lRuh4MIKKgugMBKX/fOBMlqnn0iSNCVCwUiAVAImwQVNSBW12jBOUK6gCt+g5Agklb2kEIwSDBI0NQf6iLYcg2ATwCHBANiq6KtYoOEMohD0hoJ3L/lSuRtzpoQTBWiIeiKiSvtBARvMD9GcMk0+BTE4c0a2bbDFedgEK9BQu038HoZt5Zn/5mbQCCkvyiOvUF8GaRMYXiawvAk5fm3RI2K7/2ZnssD7tkGQIlg/4yzMjTKgfN3UUoRxG6IS1OWs7mHNHScRI8PMDEy+GTwUl0p2vrfzZ23sauLrA4/r4oN4g1zQ2OcuHiD7Mxn1MtuGn++o72tJRYl9zcS4Awqjmg6dwc90U/x63viltNkWy9gl/rItnbA1QebvfceIQ6qXa0P0+LraxJsgjiDWI3TBSAiUoqpT3N0dt2i6fcHEr1AGSiuxUzOkFe7mcbR3NHe+uamR8daXCYTzVFcdMwWYm+NPFa7HecvueM9pb/z92Kv2nwseYEXVwbc0/9TqFA0aM0H7QAAAABJRU5ErkJggg==',
note_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJtSURBVDjLpZPNS5RRFMZ/7zgzWmqTFpObMiuyKDOFCoJaCUKKUFAUBRKtWrSpoP8gwk1BixbpppURtIigqAjKkgSp0KJP86NGbaYZpxmd9773ve89LTTKj1bdxblw4Pmd53Kf44gI/3PCADJyT0z+ByIaMT4S+IjRiPGQwANfERiFGBd8RezQDWc+IDBEth9dRBcBB+YKIDB169hiB142wTIRxLqzXQdELOAg/CE4oWLEd5d4gjFYPYnJ94H1ENGzt9VgFWIVYl2iqw9i/cISgMADDGIViD8n+lusEFvATfaTLq4ie+eizPx4gqMS7WEAM52etTxvsou1ag7ionPDeD+XU1V3glhNA9nhWt4/6OwIA9hAoT71YLzPEGgQQ6BylFTHEVtA58agpIHK+C4yQ++IOpryFVWUrVoXCwMEbhqTS1C28zhgsXqU/KubSFDAn/kGxTuI1TTjTXQTXe4w+vo9vtJp5U7vDQE4IvjJAaYenScofILAx4qPl/+KhLcS29iCGr+OE5kiUlZOSWHou5+baNp15vbH0O//Lt/djp9NkX16GSs+mfQ42m4htqkNlbhGKOKjc+tJPn6OzhdaG88+fA0QAsAonKIQpY2nELOSsftd2JV7iG9rQU92UhQVvFw1qWf9RFJ9bD7X178gB4qp+1cQoxhLZihrbMNInInBq1TEo6jMWjJPewinX2K1mpcDZ+Ey3epoksOnu/jQfZ7xkV6K19VjkqnximhRc92FF28Wxj20sPHh86TRb+9SU7+P0tJaEv2D08rVB5YSL+ngyP5Kt3HDmvDurTVIxOt1k6mTrZcGvvxrnX8BwNty8Brb9FgAAAAASUVORK5CYII=',
note_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI6SURBVDjLpZO9a5NRFMZ/b9Jq0iY20qoVxY/FELGgFoRuFQR18atLtUgHRyfR0Un/AKU4qVOxg6hLHXRwUhQaNUiLCK1iQSs1GvP9vvd933vvcUixCcnmhXMPHHh+POfA44gI//O6AGT5mejqb0QCRIeICREdINpHjA+hwmiFaA9CRd+5aacVYDTdB8bb6CLgwNoHCBQfnW934JdWiIsg1mtMHRCxgIOwTnAiG5HQ67CC1thgFV2dA+sjEjS6DcAqxCrEemwYOIsN3Q4A4wMasQokXBM1ixViXRCNBK0OIgC6VmhYlqBJ7GGth1gPVS9TXC1jwiqi6+0OrFGopVdo/zOYAERjVIXY7q1YXadWqJLYeYbySq7tBhEA4xXQlRUSmQkSBybpSR8DGyLGpV4s0zN4kk3bD1H6tkAQibQDHBHC/DzFF9cw7hKYECshWtUo512SAylMZZYt+y/i9vaz8OBEdwtABJJHJglLvyi9vIWVEERRWq3Qn74AKkd2eoZEysOL9KJVcKoFgFY40Qi9hy8hOkX1zSwhe5BohuRmF+t/BbGY2lv2jl5F+/7N7NRIct2BVhSf38ade4j9k8f78prCco7UroNYdx6xHsNjaWzwg1hskW1D4xntB1fW1m8PU+7u6OWBzOk7g/vimFoWkYD3Tz4xPJbBifZB/Djv7l/Pi7VHOwKyUyPfhyZmdkQjy4guNkLQdPdoPM3Pjx9YfHrjcVeniGo/iOXujXliBbG2qeRfx1pEZPAvfWh3mkWk9eIAAAAASUVORK5CYII=',
note_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAInSURBVDjLpZNBSJRBFMd/37papinFUgZmYESIFiGRYdQhynueig4eomNE0NmDidc8du7arUNFKJSpIKQHAwlNKkUzTVdd3W++mTfzOuxK2RoEzeENPHi//3/mvRepKv9z0gD65aVK7geqFhWHeoeKRSVBfQLO4MWgEoMz1HY+jXYDvFDecrOErgoRFAOgkH12q9RBsr5ApSoa4kI2AtUARCi/CFFqH+riPZ4gQrBLSG4MQoKqLdzBQjBoMGiIqcjcILj8HgCfAIIGA+qKRb8XGzTkQQW1eziQrdWC5V3KMSGYIiSGEBdEZLsUELzBzLxDkk/gLajgzSb7TxwpKAdDz8QUWfOAR3/8QQrAx6vI5gLVTbepbuniwOlrEBzq8xDyqM9jxFKfaeJ+ZXmpg0gVtzxJdmCYmva7RFFEz/c5WJrFBsGKcOzQKZrq2siZbTr665wRWzH0cE3TO/0+2NbF2kA360OPqb1yDwmW681deA344AkoixvznKm/xJaN0+Nzb+3Z3lRFGgAxRGUpqlrvEE+9IDf6HCMWr4GvqzO4IEhwOO/YTHKcO36ZLZdPj30eNsU5MGRf9aNiUBdj4w+Y2irEC0drGpDg8Rr4tjHH4eo6JuZHGJkd/miF1uhvy9T5pNEba7HBYsRyMtOcutjYwfu5UQanX09a4cJinybRv25jS29q5XzD1cyb6cFxG2hf7FNbbMC/ARq7oxUfyjJx8OXLfSo7+Z9JyXr5I2wfSAAAAABJRU5ErkJggg==',
overlays=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJeSURBVDjLpZNLSNRRFIe/O81o+WjISM0epuarEHuDqIFEDyoqEFtFD4gWQVDQoo0QhFARbowKNNpKi0DJRYVGqRmY5oPUBs3S1GnMcdR0/v8Z554WM44RGURne7nf+X6cc5SI8D9lBTh79/0VIBkoAHaCCIJCCxaLwqJAa40O4LFZpT9z/cpdaOFqcZZCRDhT0V4p/1i3HveIiAQNgEKAh83usNrfgp3Pj6NvyGOGI6AlceExPT4SAKX+/PnjNxMAr+GPCANEJGqhq8NlLtk53myk0FlN/0QO19a+Ul33Lp4OArRYF9SWqrmxWqb7WliRcwp7ynY8g5n0Pa+6vQBQACXX6zG0RgvU3djP4OhUMI7nBXZ6iEvPxz3QS4TyEbsykZjVG+0hgAbgu9fPvm1J1LWNhDtH+1qxSRf21IOYY9VERCm+dPQxPatQvolcS8gAgBkjgF+EOXM+OImpZmw/GrCnHcYYrUTZJrHFxBItbh4N5bH70hOHBUCFDEzTj9cfIGD4cfbWEjX7GvvmYxgj97HY/PimN+Fq7GTNgTKchh2AoMEvUxeBnKgOPF+bid96BJ+zimURgjmdzHhTO6qonOUJ2YjMLwL0vA4ThluqKT0UwBdIYqy7Ao3BrHsdrre9qKJyVHQCodgSBgS0/gzQ/eAExWntbCm4QORwE46aZjqeuXG87GTD8TukZmSRkmQPmcrk4iYGdE1JaUOGiOTlulyrfB+ekpJbyNT4BANtDupjLzNe9g6R1lBIPQOWXgD1+zmf3Bvn3ZGaYN2TnYLYzDde1/i5oze7Pi21YD8BVSdMJ0n4cQkAAAAASUVORK5CYII=',
package=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALnSURBVDjLfZNLaFx1HIW/e2fuzJ00w0ymkpQpiUKfMT7SblzU4kayELEptRChUEFEqKALUaRUV2YhlCLYjYq4FBeuiqZgC6FIQzBpEGpDkzHNs5PMTJtmHnfu6//7uSh2IYNnffg23zmWqtIpd395YwiRL1Q0qyIfD56cmOvUs/4LWJg40auiH6jI+7v3ncybdo2Hy9ebKvqNGrn03Nj1+x0Bi1dHHVV9W0U+ye4d2d83+Ca2GJrlGZx0gkppkkfrsysqclFFvh8++3v7CWDh6ugIohfSPcPH+w6fwu05ABoSby9yb3Kc/mePYXc9TdCqslWapVGdn1Zjxo++O33Fujtx4gdEzj61f8xyC8/jN2rsVOcxYZOoVSZtBewZOAT+NonuAWw3S728wFZpFm975cekGjlz8NXLVtSo0SxPImGdtFfFq5epr21wdOxrnMwuaC2jrRJWfYHdxRfIFeDWr0unkyrSUqxcyk2TLQzQrt6hqydPvidDBg/8VTAp8DegvYa3OU1z+SbuM6dQI62kioAAVgondwAnncWvzCDNCk4CLO9vsJVw8xqN+iPiTB5SaTSKURGSaoTHHgxoAMlduL1HiFMZXP8BsvkbO1GD2O3GpLOIF0KsSBijxmCrMY+FqgGJQDzQgGT3XrJ7DuI5EKZd4iDG+CHG84m8AIki1Ai2imRsx4FEBtQHCUB8MG1wi8QKGhjEC4mbAVHTx8kNYSuoiGurkRtLN76ivb0K6SIkusCEoBEgaCQYPyT2QhKpAXKHTiMmQ2lmChWZTrw32v9TsLOyVlu8Nhi2G4Vs32HsTC9IA2KPRuU2Erp097+O5RRYvz3H1r3JldivfY7IR0+mfOu7l3pV5EM1cq744mi+OPwaRD71tSk0Vsp3/uLB6s2minyrIpeOf7a00fFMf1w+MqRGzqvIW/teecdqV5a5P/8ncXv9ZxUdf/lCae5/3/hvpi4OjajIp4ikVOTLY+cXr3Tq/QPcssKNXib9yAAAAABJRU5ErkJggg==',
package_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMVSURBVDjLdZNLaFx1GEfPnZk7cyfNkElS0jTVKKRpE2PSpAhKHyqo2QhtShUiCPWBLhTdFKUUlxYUqggGxYqIbsSNFKQmVqMhTVujSQuhtnmMaR5NJs0kncz7ztz5f5+LgguNv/WPszkcS1XZbFPnDrUh8q6KRlTkrdYj/Vc3+1n/Bkz3H65T0TdV5PXapiNRU1jjztxgVkU/UyMfPtg7uLwpYGagx1bVF1Tk7ciO7p3bWp/BJ4ZsfAw75Gc1NsTGrfF5FTmtIl90Hhsp/AOYHujpRvSdUHXnwW0tR3Gqm0FLlJMz3Bw6xb0P7MdXcR/FXILbsXEyiRujasypva+Mfm9N9R/+EpFjW3f2Wk5NO25mjVTiBqaUxcvFCVlF6ht3g5vEX9mIz4mQjk9zOzZOPjn/TUCNPL/ryT7Ly6yRjQ8hpTShfIJ8Ok56cYm9vR9jh7dAbg7NxbDS09Q2dFBVA1d+mH02oCI5xaoKOiEiNY0UEtepqI4SrQ4TJg/uApgguEtQWCS/Mkp27hLO/UdRI7mAioAAVhC7qhk7FMFdHUOyq9h+sPJ/gU8prfxMJr1BORyFYAj1yqgIATXCXQ8GtAiBLTh1XZSDYRx3HVn5iZSXoexUYkIRJF+CsiKlMmoMATXmrlA1IB5IHrRIoHIHkfpdpO6M4fkcLiyFuLwWJu26lNwUB5MTtBghoCJhn20DYSivgxRBXDBFcBooK/yyEGTKruXxRx/inppmfv3zLOevXWByw630qZHh2eGPKCQXINQA/gowJVAPENQTflzw6GzZg/EZ9mx/CmN5PNK+j4s5z/KJMU9nFkdenRw4GZv//WsMQYjsBjsMCqbokcisY1uVHGp9A4DjT5yhqa4Do/j8n343b+o7X7oSHzvzbT4x48UnzrVj+Z1I48NY9lZEwnw1OkT1dpvh2bMcaOrhvfMvkimsc21yyv1PTH/0dbWpkZMq8lzTYy9bhdU5Pr84yPVomX0dB2iu72Jm5SqXJka4dTP1gfV/OV8+3datIicQCarI+8eXc/uB14AIkAE++a1v+cTfDyOvKVPjhy0AAAAASUVORK5CYII=',
package_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMNSURBVDjLdZNLaFx1GMV/d+bO3DtJxkwSSWNq05YkrTVa04qIiOiiRIxgE2ohhUIKRQQFBcGiIiJiFaGIYEFERFy5dCE1pQ8JIbSEJhG66GM6idOk6Uwyec7zztz7/z4XlSw0nvXhx4FzjqWqbKXb517rQeRzFY2ryPv7Bkf+3Mpn/RuQHDncqqLvqMjbLZ2DCVNZZjV9uaii36uRr58Yunx/S8Cd8wMRVT2hIqfi2/u6tu17nZAYiplJIk6YpdQo6/em7qrIGRX5sXd4vLIJSJ4f6EP0Y6ep94Vtjx3BbeoGrRGs3eGv0dPsePx5QnU7qZZyLKamKORuTqgxpw++MfGbdXvk8E+IDD/cNWS5zU/iFZbZyN3E1Ir4pQyOVaWtYy94a4QbOgi5cfKZJIupKcprd3+x1cjxPYfOWn5hmWJmFKnlcco5yvkM+fkFDg59SyRWD6U0Wkph5ZO0tO+nsRmmf589aqtISbEao65DvLmDSu4GdU0JEk0xYpTBmwMTBW8BKvOUsxMU01dwdx1BjZRsFQEBrCiRxm4iThxvaRIpLhEJg1WegZBSy16ikF8niCUg6qB+gIpgqxEe9GBAq2DX47YeIIjGcL0VJHuRDb9A4DZgnDhSrkGgSC1AjcFWYx4UqgbEBymDVrEbthNv28PG6iR+yGVlIsfKtTm8xXVCD0VpfY5/EojEQpEIEINgBaQK4oGpgttOoLA6sUIt6/L08Q9xdvdQuX6BG+OX8IP1+pAaGZsd+4bK2hw47RCuA1MD9QFBfSFzJUn3S0dxZ0axfj5G3eyv7Opopja3HthizKuF+fHhW+mxU82dh7oe3d9POL4XyinwSpiqj1mr4bbthv73Nidsf/oIIU+czSlP//Bsq4q8q0bean9qINHe2w++R37+KtOffckzrwxSP3eOaiVLGSjkw9yaYeE/Z7p29kCPGvlIRY51vnjSqiylmb/4B3be0x0tgWWH7lHIBaQXw8b39BPr/+589UxPn4p8gEhURb7ierWntHr/zbCxdpqwLih89/KF4Iu/AXSvuZLBEiNYAAAAAElFTkSuQmCC',
package_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMUSURBVDjLbZNLaFx1HIW/O5ln6jSZJE3SJE1L+pym1aSbirXoogah1KZUIYISRerCot34orjTBsQigt2I+EAEFwouRBMogdgniUmkIMZMpulMY2eaSdqYed659/5/PxelXWjO+vBtvnMsVWWtzP78TDciH6hoVEXeih8b/n2tnvVfQGL4aLOKvqEiJxu3Hqs3lWXupkaLKvqZGvl4z8BoZk3A3Eh/QFVfUpG3o+1921riz+ITQzE7SSBUQy45xj9/T6VV5KyKfNEzeKnyAJAY6e9D9L1QrOdgy67jhGPbQR28lTlujJ1h0+4D+Go3Uy0tsZicorA0M67GnNl3Yvwna3b46FeIDDZtG7DCDXuxC8usLs1gnCJuKUvIqtLauRPsFWoe6sQXjpLPJlhMTlFeSX/nVyMv7Dh0znILyxSzY4iTJ1ReopzPkl+4xb6BTwlE1kEphZaSWPkEjW0PU9cA07/MP+dXkZJi1QXDIaINnVSW/qQ2Vk99LEKEMtg3wQTBvgWVBcq3xymmrhDechw1UvKrCAhgBQnUbScQimLnJpFijkANWOXr4FPemfiBlWqV97e0QDCEuh4qgk+NcM+DAa2Cfx3h5l6CDXHC0Vrk9nlW0z9iey4dTXFOzaaQso04HmoMfjXmnlA1IC5IGbTKmxPfg7o44uJ4HhtjO4i37qdgl3h5fpxvmx5HjeBXkYgvEAAi4N0BqYLYeOLwVPcgRgUjBkHJrC6wt+MARafCkYtDfCTtYb8auTB/4ZNDbbueJhLrAM2AcxfbczAqpO/M4YqHJy6ucclXC/RsOkjRLXPixuWgT4w5XFi49OpfI6eT6YlvMAQhuhPbs/GMR8v6TlrXb2ZjXReBmggbou1ML1zm2vWri4elfsODKU9/vr9ZRU6pkdfaHumvf/2Pr7EdB0ccbM+hq3E3j3b1MXnzCr8mzs+UPe3NDGn1f2f67Vxvtxo5rSLPb33iFauSS5GZucbJ0sXqns4nQ2OJ0SlHeCwzpM6ab7yfq2e7+1TkXUSCKvLhi3bySyM1TRUxgdyQevd7/wI8lcwGoqfWawAAAABJRU5ErkJggg==',
package_green=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMSSURBVDjLfZNJaFwFHIe/N+ubdCb7UpK0FbOQRdvUi4cag1iC9NA2tGoKQgTxoIKeDFLqQTAHofZiL1aK4MVTT9WmoMGIJRLN0qpJmmQas00nmcky+7yZee//92QPEvzOH9/p9zNUlYNY/P5sNyKfqmhIRT7sHBidPcgz/htYGj1Xr6Lvq8h7NS0DlU5+h73VsYyKfqmOXHtmcOzxgYHlu+e9qvqmigyHmvpbGzov4hKHTHQKr99NLDxOYnN6TUWuqsjNnqF7+SeBpbvn+xG94q/q6W3ouIBZ1QZaxN5f5u/xEY50ncJVdoxCNs52eJp0fGFSHWfkubcnbxuLo+e+RmSotnXQMKufxUrvkIwv4BQzlLJRTFeRhiPtYO3jDh7FZYZIRZfYDk+T21/71qOOvNF++rpRSu+QiY4jxRT+XJxcKkpqI0Lr4Bd4A4cgu4pmwxipJWoaj1NRDTN3Vl71qEhWMSp8pp9Q9VHy8XnKqiqprAoQIAfWOjg+sCKQ3yC3NUlmdQLzqQuoI1mPipglKeDHi7eiDa8/hBWbQjIxvG4wco/ApRS3fiSdSmAHKlE3IKAiZR51xB1OjuKRPdqCr+DyHMKsP4ntC2Bau8jWDyRLaWwziOMPIbkSe1aOiFqo43hd6jh2ua+e5eTPjEU/IZK9B1rAE2widLidnBeKfhO7YJPOWfxSiPGdJ8G+4aCOqEdFXM3lfTT4Ovhpc5jfEreo8zZwIvAiQbMRW6Fg2cykk/zpZHAMeKn8ZbqSwkORovudgWN9md3Np4OhZjoOX6LOXctcepxH1hy27rOyvc2dxDZrYtEZ7OK12rOUJ4X1uVnsQnzCuP9Nr09FhtSR4eqW063Nx8/g8nj4I36Dh5kZ/lrcp85Xw+tNF3HnCkTmH5Devb+mop8jcvPJlGe+er5eRT5QR95tPHG+srHnDKVigtjGFEHHRXR+jt31iYyK3FCRa70fr0QOPNPv1092qyOXVeRSS99bRj62yuOFB9j5zVsqOvLClfDs/77xX3692t2vIh8h4lORz05dXr59kPcPXS/CRago2Z8AAAAASUVORK5CYII=',
package_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAM9SURBVDjLdZPrS1NhHMcPREH/QC96nxURWS+iktC8bEt3skHoi95EbYk5KStqdtPGLKcuczSELnYhK0bzhpfK5nSZed1MPZtzzpq3pnN5tOY2p+fbsxMEUj3wgx9fvr8Pz+X7UACof9VwPb1juC6l2l6T/N5WJdr9P99fgqPxyCYyrLLXpczPMg/xrbcQzOukH0P6xJLBl/Gb/wsYaUpdT4Zlw/Vi55RVi5XgNLilCSy6qhGYrIO79Tw+P4/92v/soNz6JGbjGoCjKVXgaDhi/tpxA4Hvn4m0BHAswr4ejBiOImAvRsitx6JNB2fdSVge7e/su7+X5gFk+LGjgeZ8jkr4vQPwjbVgrIsYP6hhe3MOrreZ8Nvvwm/NQ9D5CMsTesx1q8C8kKBHt+dF5LLCXNCNkLcPvgEtvL0qTJnOwlmbhs57MVieswB+BzD7FtwXHcBcBiYrER5VoUu7K0yRy2JXg+PAjyEsT9ZgwXoL/v48UgpM1op5DTONgPsBOJsCfmMcZhoOYoG5i87SnSxlqznMri4RwM8RAmEArxEBRg1/VyZm6sUIj2iA0RKE2kWYa9wHj0kET3Mq2P4SfNLsYCnGIGRXeIAdWCTbne8kkHcIO7VYaEtDyCwCa4zB3EchZoxJmG6Ix3StEN+7C9FRtI2lyPv+BpAjgO1CYOoNmqu10JQUoqKiAkUFl2AlRxltFKJIdZHXim/no+aBAibV1gVq8FV8iAt/Iy/nwrK3BRW66ygrK4PH44HL5UJbWxvuqHOhU8vhGGZ4rb29nfcoTx9YoQYq45pHjZexNGVC67uXuHpFAcvgIArz5aBpMWQyGbRaLXJzc/meFouRf/4ED7l08VyYIsnaQJIlI+FwKi8cw60CFQ8IjldCJEyA0WiExWKB2WyGwWCAICEOLcot7ghAqVQG/kSZJGtTzvHopuwzUi4CuHnjApISEyEQCCCRSPiK9Anxh1bTjh1tjQAyMjLm13yM7WRJUsVjpRp16PWrp6iqqkJ5eTlycnKgUCj4PqLp9XqfRqOZp2navgYQFRW1LjY2Njo5OfmLTHoqkC3PXM2Wn+GuZQhK09PTE7KyshZJBaRS6c+IJ+L9BchY24ysm0a5AAAAAElFTkSuQmCC',
page=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAINSURBVBgZBcG/r55zGAfg6/4+z3va01NHlYgzEfE7MdCIGISFgS4Gk8ViYyM2Mdlsko4GSf8Do0FLRCIkghhYJA3aVBtEz3nP89wf11VJvPDepdd390+8Nso5nESBQoq0pfvXm9fzWf19453LF85vASqJlz748vInb517dIw6EyYBIIG49u+xi9/c9MdvR//99MPPZ7+4cP4IZhhTPbwzT2d+vGoaVRRp1rRliVvHq+cfvM3TD82+7mun0o/ceO7NT+/4/KOXjwZU1ekk0840bAZzMQ2mooqh0A72d5x/6sB9D5zYnff3PoYBoWBgFKPKqDKqjCpjKr//dcu9p489dra88cydps30KswACfNEKanSaxhlntjJ8Mv12Paie+vZ+0+oeSwwQ0Iw1xAR1CiFNJkGO4wu3ZMY1AAzBI0qSgmCNJsJUEOtJSMaCTBDLyQ0CknAGOgyTyFFiLI2awMzdEcSQgSAAKVUmAeNkxvWJWCGtVlDmgYQ0GFtgg4pNtOwbBcwQy/Rife/2yrRRVI0qYCEBly8Z+P4qMEMy7JaVw72N568e+iwhrXoECQkfH91kY7jwwXMsBx1L93ZruqrK6uuiAIdSnTIKKPLPFcvay8ww/Hh+ufeznTXu49v95IMoQG3784gYXdTqvRmqn/Wpa/ADFX58MW3L71SVU9ETgEIQQQIOOzub+fhIvwPRDgeVjWDahIAAAAASUVORK5CYII=',
page_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ1SURBVBgZBcFNiJVVGADg5z3fmTujY5NZBANBIVkZ2J9IKkZFGKRuglq1KaqVtoqoVbSKFi1LoY2QEVSbcqiFWWJqPxL044wUGESQVqJOKerce7/z9jyRmba++tXTy2YmnyphPaYQIJBBNuPWfls8l1/EfxdeOrJnxxAgMtO2148d2ffC+rWlxMqkkwBkQjp7aeT97xf99cfS5ZPzv6w6umfHElQoXdw+qN3KhX90JYIgG30243G6Muo9tOYa999WfdfOLs92x4UHd3163eG3ti8ViIgVmdkNumKiUIOu0AURFIFmdmZgx4ZZt9w6uazOTO+FAklAQQlKhBKhRCgRShfOnL/i5hUjd64Kz2+6XjfRPQkVIJPaEUJGaH1SQu0YZHHqXBq2sdaGHlg9KWoZQ4VMEjWKlBJRQiAb2RUGlBZa66RCFFAh0RBBCIlENiY6QBTRhyypIROo0MZk0hDITFAKWqhdkkGSQt/oG1ChtZSZJCkBSCCEE79+Yv7UnIuXLxiNR8rwnsomFfpGn2SjAUjQkuPzHzp98XMPb9ngplVrHFr42OX5ubpx1943K7Rxaple+2EopBZkBo2MNL3wnie2P6ovvbtntzp48iMb1232+6n9OyuMx72+Z3Zmwn03Fi3pkz5oyWffnjERKzy29lnw4iPvmDuxG/unKoyXWhu3lsNefPNnr0VKAVpy/tK/Fk5/7afTR72yda83DjxjqpuEqxVGV/u/pwfdDS+vG05nZpE0wLXLqn2Lzzn287s237XF3IndBlEd/fEwvB2ZacPOgzvHo3w8Iu5NuRxAkkhpovug1u5Q5SoGfWurDxzf/eW2/wEnITFm/fHryQAAAABJRU5ErkJggg==',
page_attach=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKsSURBVBgZBcFLiJZlGADQ87zfN/N7GSe7UBiIEmUaRKRJRogGpmEmRglRCIZEi2oTRbuojQQRdDGVCDICN4ELISMoJG/jhNm9DLOyMLMyLSXHme9/n86JzHTns3vWTx7uPVTCAkxCgEAGWXW1Hjt7Oj+Kf888vXfLqnGAyEwrN+7f+84TC+aVEtOTRgKQCenP8xO2Hzrr918u/vft10cu27dl1UVooTRx/WDbTP/mD02JIMhKP6uuSxcm+pZeN82tc1qj9c8pWeeeWfL4e5d+vOnuiwUiYigzm8GmGCi0QVNoggiKQDVjeNCqhTPMvrY3uR2e+hYUSAIKSlAilAglQolQmnDy7wtmDU244bLwyG2XawaatdACZNI2hJARaj8poW0YzOKH02m8dmodt/ianmhLBy1kkmijSCkRJQSykk1hkFJDrY1UiAJaSFREEEIikZWBBhBF9EOWVJEJtFA7MqkIZCYoBTW0TZLh1PHvvfvGC4aGhsycmJi0bt3mewrUmjKTJDMBSCCEU8eP2rtzm4WLFnvx5U2OXb1hrNfrvd1Cv9JPslIBJKjJrz8e8eGON82bf7ublqzx6Ib1+jGnPXnm5GALtUs103OfjQupBplB5ZLTX5j2+XbdrEV2TV/l1XVrPP7gaqMHR9rfum52C13X1+8zY3jA/CuLmvSTs9+P+Gr3Vj9NusryFQ849NgKy+6+3yejI74cWDv26QcrTxToLtba1ZrjfUZO9I2c7Iz+3nf0/W0Onxvir589v2ymW1av9893B7y2eWudyLaDFibG+qemDjZXPHPj+NTMLJKK+w4dtvvjA86fO+fg/j127dzhlde31sm9gXP9rp6AFiLypeVP7bk3Im5OOQUgFmw0c9bcXq/lrjsWxnfTHh5b+uS+sVrr4bbYDv8DSZ8/CcPtqE4AAAAASUVORK5CYII=',
page_code=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALESURBVBgZPcFLaJxlFIDh95zv+//MJZObITUUq/QSG42orcULSnQhgmRjQRCyqaC4qLqrLgQ34gV040JKcOGiEMS9C0E3iVGEWIu0pWClWAlpmsaZSRuTmfn+c8zY0ecRd6fr2XcXTpQH+mZVOAqUAKFLABdwI5n93tjw72Szfmrx9EybXeLudD3/wdLimTeOTqrKkEPA+Z87u5z1Wx3mlxtcu9r6++L5SyPfn55pRXo0yL15DEMXrhNUBATcoHAjJWe7U/D0oRqPTkR+svWK2+H69OtfDys9ItLv7iEPSqYQBYJCEBABRQBjfCBn5tg49xzsK8eB6hdKj4NstR0FVEBFUBFUBBVBg7D61zZ393e4b0R49fE7CFl4MdJjKB8tNGkWzqnHRvn0XIOYYHaqRgxC7srlDadtCbM2T+3vQ6ImZddDH14Z8YGxPGaKtkBFIEGtDGfONmm1nSwofXmglAdKeYajIIo++P6Vl6YOVVcm9/Vrs1HwyiMjqAgn7h9kvVFQKcPnP9dZ/m2LLCp5CPRFxQB30MkD5bl9tVBarXd4+UiF4VrEgeGKcnyixlozMV4Vlv7cQoCoQgxKYVAY6Lnlzdcur7Z2RvLA3GKTjUYHd2fjVuKrX+oMReWPtTZPjlfIVIkqBBVKmVAkRy99MvHl8lJ97/mLdSuVlM++uUZhMLdwg2pJ2dro8Ob0GE9MDeCAOTiQBSW1E+LudE2/88P2/jv3lm60oXJ4D62r62zehMGDo2gudLmDcdv8cxnPvLW4E+kxS7w3u4ePL+QcGVPswF0UDoWAOTjgDu7w6/WEm9PZSUR6UsssmXm7QH5cKTBxHKHLHATHHFwFNSFGsVRYivR0doq1ah5G336gXXV3xcG4bbAc6XKHciaIYFmQm0WyFXF3uo6d/PZk6vgLIvKw4xX+4+CA4/zL6doxs7MK8/8A73I7wFFcAY8AAAAASUVORK5CYII=',
page_copy=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIpSURBVDjLddM9aFRBFIbh98zM3WyybnYVf4KSQjBJJVZBixhRixSaShtBMKUoWomgnaCxsJdgIQSstE4nEhNREgyoZYhpkogkuMa4/3fuHIu7gpLd00wz52POMzMydu/Dy958dMwYioomIIgqDa+VnWrzebNUejY/NV6nQ8nlR4ufXt0fzm2WgxUgqBInAWdhemGbpcWNN9/XN27PPb1QbRdgjEhPqap2ZUv5+iOwvJnweT1mT5djZKjI6Ej/udz+wt1OJzAKYgWyDjJWyFghmzFsbtcY2gsTJwv09/Vc7RTgAEQgsqAKaoWsM8wu/z7a8B7vA8cHD3Fr+ktFgspO3a+vrdVfNEulJ/NT4zWngCBYY1oqSghKI465fvYwW+VAatPX07IZmF7YfrC0uDE8emPmilOFkHYiBKxAxhmSRPlZVVa2FGOU2Ad2ap4zg92MDBXJZczFmdflx05VEcAZMGIIClZASdesS2cU/dcm4sTBArNzXTcNakiCb3/HLRsn4Fo2qyXh3WqDXzUlcgYnam3Dl4Hif82dbOiyiBGstSjg4majEpl8rpCNUQUjgkia0M5GVAlBEBFUwflEv12b/Hig6SmA1iDtzhcsE6eP7LIxAchAtwNVxc1MnhprN/+lh0txErxrPZVdFdRDEEzHT6LWpTbtq+HLSDDiOm2o1uqlyOT37bIhHdKaXoL6pqhq24Dzd96/tUYGwPSBVv7atFglaFIu5KLuPxeX/xsp7aR6AAAAAElFTkSuQmCC',
page_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ2SURBVBgZBcFLiJVlGADg5/3+b87cbLyFNBJ4oexGQYqIi6hFQambgohoE0aLUqGCaBcuonWLUFe1CIJolWCLaiK1C0FUREpRBgmWNpqi4XjOnP97e57ITI+8fuLZ6bnJZ0rYhikECGSQzbi1M1cu5UJcvfzqycN7RgCRmXa9+dXJ9w5su6uUWJV0EoBMSIv/LXv/uyvOnx1eP/3zL2u+PLxnCBVKF3cMarfq1D+6EkGQjT6b8TgtLfceuv0mO7ZU37bFmWx3Xn5w/7HVx9/ePSwQESsysxt0xUShBl2hCyIoAs383MCe7fM23jY5Xedm34UCSUBBCUqEEqFEKBFKF/7+d8mGFcvuXhOe37lWN9E9CRUgk9oRQkZofVJC7Rhk8fulNGpjrY08sHlS1DKGCpkkahQpJaKEQDayKwwoLbTWSYUooEKiIYIQEolsTHSAKKIPWVJDJlChjcmkIZCZoBS0ULskgySFvtE3oEJrKTNJUgKQQAj950eMFg5ZPvebU+vW2zH9WGWnCn2jT7LRACRoyY2FI6ZOfeC+p54zuekeSz99YubkQv304YkDFdo4tUwHfxgJqQWZQSMjPX30Lbv3vmDqzBeceMPMylU2b9jg+1/z5Qrjca/vmZ+bsHVd0ZI+6YOWrL7yp6lbNrHrFQD14LyuxcYK42Fr49Zy1ItvzvVapBSgJetXzrv+4zGzR180XDrvOq5d7fSdvyos3+gvzA66m1+7dzSbmUXSACunq4vn9zt9/B23rp5WuwnXFsf+uNBJ/aHITNv3fbZvvJyPR8T9KWcAJImUHh0eq1sXP+zWDi/G1cHc8Oxgy8cvffT1E/8D2iAtJW5RUGAAAAAASUVORK5CYII=',
page_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK5SURBVBgZBcFPaJZ1HADwz+95n3e6uTnREGdljRKtGCYiHTLxkIUmQeeCOnXzVnQIoi5BQV08TMo6GIiHiKI6ZEWgszzEmtpqSDP7s9ycm9NN977vnuf37fNJEWH/G6df6l676vki2YXVSCAhEpFVOU8uzMX36daNV88MH+oApIhw8O2zZz45vOuhokjrgoYAIALC7NKKEz8vmP67fee3XyfWjwwfakMJRSNt6yob68avaRQpkYhMHVlVheWV2r6tffYPjNi4eLyncWCodf7jI1Jr6sUSUkq9EdHoajQkIZALZOpEIWlPf27r4jndQy/oH9xp4c9tJk4de7eEIEGBlAgJREqKRP/yKXVcsH7r4+Ynf9eVOvrWbtK7YUt/CRBB2SBJIiW5Doqkd3nEllWj+gef1r56UldP8tfYhJt3UhTtuR0FRBAoU6FISYFGkaxePG1LfKv/gYNa/30oNW9o9vbpzvOOXj+wsvvwZ5cKCGSkRJGSIiWtK19af/uU/gef1ZoaVjRXdG7db+bMed173zJVD2QoIFdEkBG4fflrPYs/2vjIMzrTxzS6QvvWfWZGRs3tGZY2bFdnoICcQ0QQTI+e1L3wk5W82dWLR2Qtt+fvNnNuwuLeo1LvgNXNpK4CFFBn6iAysxc/8vCel636Z8SlL84a+2be+Hdjlh57R9WzWaDZKFSdCpSQq5AjvPlLx9DkrM74VwZ3POHm7JzJsUk/7PvU9Sv3yipwYlPTSjuDEqqqVtcMrG0a/+Oa9z8Ytnv7oOXNOyw9edyjffeIIIIL1yqRw0qrAiVU7ZyrnKNTS+te/9flFCYlkJdIS5UcRJEUOSnLlKs6V1DCSqueWdPVuOu1oc6aiCgEGdDfXYIIuptJSnKzkRbrKk9BCSnFe0+9cvq5lNLOED0AgkAIIEAr5zxaFk7A/5IUWNTkV3l/AAAAAElFTkSuQmCC',
page_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKrSURBVBgZBcFLiJZlGADQ87zfP/Or4wyjeWkgMSvH0lqYSUg3CSIoISoKokVtgsjaBG0jXES1CwJpFRgF0Y2ScKNBmoVWJpQhUlaWVuo4Nphz+7736ZzITPe8uP/JhSP9x0vYhAUIEMggq7bWXy5O5L6YmnzhwM5tcwCRme57+eCBt5/bdEMpMZo0EoBMSOcuzXv324v+PjV7+acfjy/9cue2WehBaWLdYK8ZPXZWUyIIstJl1bZper6zde2wW8d7DtVzi7JeP3nXs58t+eKN+2cLRMTizGwGm2Kg0AuaQhNEUASqsZFB2zaPufq6/sLeyNBbUCAJKChBiVAilAglQmnCXxemrV48b/3S8NSWKzQDzSPQA8ik1xBCRqhdUkKvYTCLnyfSXG3VOueOa/qiV1ookEmiF0WJUNCUMNCEXoThmSNWXXjVUHPZgsEBqRAFFEhURFAilAglQmQYjBlD5z+0fOVNRv/do98rKjKBArUlk4pEZiKVwsD5PZas3GhkbKPBs3v1Z0/qKl0FCtSaMpMkMwHE5T85vdvwslHd1Keu3PCY/h/v6Tedrk1QoKt0SVa6SlfpalVPvW/5+APMHHF41zsWj07LMwfNnjmknWtBD2qbaqaXvp8TUg2unT/q6eFJw0tW6KZ+Javu0jfWbH3e71+9qcw/AXrQtp2uY2xkwM0riqyt8R92Gd1wt3r5kKzTNj28Tp07Y8HQCaOrtrjz2Oc97lWgna21rTXnOr4+3Zk6/omrxtZaNHxethNE47uPTqCqMycsH1/ttv6BZv+OG9dHZrrlmX0n97xy+7L/5nMoM8vEx4/a8NDrmvKbbCeRACiahev8c+yoE7t3fBCZafP2vdvb+XwwIjamXPTamh39gZyNrClrlbXKWmVNWWuqNUu2bWR3+H8rUk+Grcb3xQAAAABJRU5ErkJggg==',
page_excel=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALDSURBVBgZBcFNiFVVAADg75x777z50RmdDJG0phpTIwq1cqP9IBqlLaxNpYVSVIvahLVLCqFFoGEZQkQhgdGilUghaqRNIKgUZEmQlCBlmmOm772Zd+85fV/IOVuz7ejmgeHWxhgsRz8CCMiBnNQp/Xbln3w4XJ18/die9dMAIefssXcmjn326vIlMYZZmUIGIGfILl7r2Xfiir/OTbV//unM6Hd71k9BCbEIi/rKYtbpvxUxBAI50eSkrrNOr/HQwplW3FE6ni4O5rR48sFXDsz+dve6qQghhBk556KviKpIGSgiRSAEooBk3nCf9ffNMzbeGiiHhz6F8NSO1WdTHh2bNZhCk4Nl44+7fP2Sb37cK6NVzdCk2rplz9j0wEtaVandnbbpvZP1wbdXVSVOvfzI5ls7rT/9fvmMUyf3q1PbsoX3mG5q7XZHMmp8wdOOn6ulNG3VbS2hjDVEbPzw64PNDXnc8NCwRXfNU8ZBl65e1m53lcVcW9a8b3hoRH9fob+vkkVCBPHz1w5NtZsne19M7LVkYLWZ/QPGF92i2+mq69ILa3caqFqqMuorCq0ySsgZiNBuHy6+//WIXQe2u3/OBk3ZceeSu031Jp3+45CyoCqCMgZlETWJJgHx3jduevFa5+NqxeKVchXs3P+WRxc8a9Il88du99WJDzy/a0zIQRmDIgb9VdDUGURsI5s4fcQvZ3/QmW58cuQjT4w9Z2TmbKM3L7D01pUyUiajKqJ6ugbliXfPz3/4zYnOvq3L+y9eq8C/1y/4cmK7691JIUQjgzeqIlUMIOWsN5VACXXdaBoARobm2rJ2NwAAgJyyXrcGEeqplOqUMgAAAABAWcZUN6mGEnrd5sJQXzFH6A3lnKNMAowMlCBnBqooBKkqwn9Nnc5DCSHkHWu3Ht0QQlia5UEAmYwsAxl0U0qnymgf/A8eWStYAg6kAQAAAABJRU5ErkJggg==',
page_find=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMBSURBVBgZBcFLaJxVGIDh9zvn/2cmmWTSJk1oxUu1xCZoi1qvi6gggpRIdd2Niy6KVRChiCBeNiK4KOiiGxGKkNiCpKIhQrzUxiINkmJqQqdt2iQzk8xkkiZNOpfMnHM+n0dUFYBXP774VksmedQIh4AUIAAIoAIacCHMbazpb7K5fnLi9GADQFQVgMOfX5r47t1D/cbIDgWLAgCAKoBSvtdk6J8Niovb1dn/rnX+dXpwOwIAMFb2JyK7Y2YFa0RAQAN4DTin1Jqel3vbee7RiMuh3Kqhb/2ld0Z3iqry0Rdfdy4vFdYsgTeOHufst1/RdJ73Pz2FD4GGU+oNz4u9KWbKitPA+L9Ffp+YG7b5tWrHzq7uTzq7up+/OjVJ/4GnufTHGP0Hn2AuO8O+/Y+DWDxKrE160sqeNsvBPUnOTxb7zX0P7M3MTk8dWy2VuFepc/XKJNve0AyGP8dHqWyuE1khYQ0314TLi44LN6okY4tExpmx82eP16qVtof7HqPn/oco37lLJtPOru7dtLam+encGSIRYmtIJiyphCWViFEMiME80tv3dro9wy8jQ1RW83S2JSjl5vj53DeUlnOMjQwTWyE2QhwZEtaSjAwBUIWoPdPRMvj6EW4t5FiazzLwymEKuXleGzzCcqnM+OgPGBEiq6ACCorgA/gAkXMu6bwnOzvNxmqJ4lIesQYxhts3r1FYmAdAEEQhMhCAVAzeKaaytclifol6rUY+l6OytcXd9Ttcz2ZZXSmyVi4SVPEBFAgKCsTW4BqOKL94+/rI92eoONv7YE+X/LjSwq0bC8zOF6hVq/iuvXx2pQmAKgQAYGh3THM7IKoKwMCHF2vDJ59NfTmT4KkeQ1DwCl4gKCigCqowveI49QIMvHehHgEAuO0QXAja8MjfBU8QRREAgoKgBAU1gglCFElwPrgIAKBZ96V0wu764EAjraoGhQAAdLREAKhCSyyIEGIrW96FgqgqAM+c+PWEa+qbIvKkoq0AACgooCgAKAD1EMKUgaH/AfiQhzSolJNMAAAAAElFTkSuQmCC',
page_gear=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALTSURBVBgZBcFNaNdlHADwz/P8fpvbfJnO+M+XtKz1ovSqSEaJIUWRXoQKIoI81MU6dAgPQXQphS4dIiOQhMp7h4hAD84iArVSK8FU0NRtbro52+b2/z3fPp8UEZ77YPCN7kXzXsvJBnQhgYRIRNEu5dz4WBxON2+8d3Tf9lmAFBFe/Pjno1+/s2FtzmlxUAkAIiBcuzXn4LFxQxdvT/11+kzfT/u234YacpUe6KyrxX+OqHJKJKLQRNFuh+m5xjP3LfTE/bVfy7WeKA/e2PL290uOfLbtdoaU0oKIqDqrrCNTJ6pMlUiJLKFYvqjT9o3L3T0wr7teNP8ryBAkyMiJnJKckpySnJJcJVevT7trwZx1fcmbTy5VdVQvQw0QQV2RJJGS0gQ5qSs6I/tnLMyWtlJmbb5nnlTnNtQQQaBOWQiBlJOEKESVTV0aFHOzuga2CpmUQQ2BgpRIkkDgv9GLJkcuWLDyUV3zOpWOWpm+7sih4zYt1QEZSpsICgIRgTBx9azVq+40ffWkVqtlxYoV2sOnDQzcq+fm39WePXt6aiglRARBCAA9favMzMwYHR01ODiou7vbunXrTE1NGc2rm092vzqVoSk0QRSaQlM4f/wHnWVSq9Vy9uxZ23a+b8sr7/r38hXLli2z9aHF1d69e6OG0g4lwoe/zUpCSeyYmFBVlf7+fq1Wy0e/zynYWpLJyUmXLl1y/vx5NbTbjaZh+aIO61tZCabbL7l89bS+oSFr1qzx/LlvtUuxZvVKY2NjDp+82fTSqqF9u5R2KTHbSL9cbpQUQuXpK6foXWtoaMj6xx4xMzPjwoULent79eeLVerunqhhbqYZnt9Z3bH74dn5EZEFBcfGlzpw4ICdb+1y5tQJw8PDNj21Ob784vM03iy59d03nzYpImzcdWhXey52pJQeD9EDIAiE8OzCH7tGRkac7Hp9vJRyInPwj/0v7P8f4TBXams7dlUAAAAASUVORK5CYII=',
page_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKdSURBVBgZBcFLiFZlGADg5/3OmRlHbbwkpmQ37WZlmBYWIrnRRUnRolVQbcRN7Yp20SLauSvENgaB1C6KdpnlBbHMFl2IMsK7Ntg43mbm/8/53p4nMtO2dw+9Pj4x9koJGzEPAQIZZNXV+vfVK3kgrk29fXjPjgFAZKbnPjh6+NM3N64tJRYnjQQgE9LkjaH9J666dGbu1u+//rH0yJ4dc9BCaeKh0bZZ/Nu/mhJBkJU+q65LM8Pe1gdus+nB1vE6OT/rw1PPvvH1ku8/fH6uQEQszMxmtClGCm3QFJoggiJQrZwYteOple69f2y8nViwDwokAQUlKBFKhBKhRChNuPjfjHsWDj2yNOx85nbNSPMytACZtA0hZITaJyW0DaNZnLqSBrVT68CW1WOiLR20kEmijSKlRJQQyEo2hVFKDbU2UiEKKJCoiKBEKBFKhMgw0oSREr448YLPjm421hYVmUCB2pFJRSIzkUqhRGgb5rqhVcvW+uib9fpKX4EWak2ZSZIS7P12K9kb1M6g66xcssbaFZtcn73p1X336bvPQQt9pU+yUgFdHdj26Gv6rPraq9KF6bPWrdrsxmDGT7e2z3v8/ZlooXapZnrv54GQarCoG+izOn3lL8Pa6erQsB+6Nnfd+ru2uDG85fg/R2Zb6Lpe37NyYsSG5UVNDlyY1fWdOybu1tVen9XF6TOWLlzh5Nmjjv15OmfT4ha6uVq7WnPQi2PnezXSrW7gk2O7DerAbDewZtlaT6/e7sfTR5ybvmR8cs/NUx8/P9PCcLa/vGC0WfbOusGCzCyS+tgvYNF4C17ce6co4yavT/ly1w/TG3YePA8tROTu7W8deikinkg5H0CSSOnG8rmxc1PfxeLLX119ctfBk22xH/4HCmFTpxr5rC8AAAAASUVORK5CYII=',
page_green=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH/SURBVBgZBcE/jN9zHAfg5/35fn6/6/XqVEmkISHif2KQRsQgLEg4g4TJYrExYTXabJIbDZJOFoNRQssgERNiYJFeOE21QfTuft/v5+V5Konn3r/0xvbu1uutXMApFCikyDCP8euNa/mi/r7+3uX9vROASuLFD765/MnbFx5prc6GSQBIIK7+u3Hxuxv++O34v59++Pnc1/t7x9ChTfXQuk9nf/zT1KooMlgyzHPc3CyefeAWTz7YfTuuns54+Pozb31+21cfvXTcoKrOJJnWU7Nq9GJqTEUVTWE4v7u298R5996/td13dz6GBqGgoRWtSqvSqrQqbSq//3XTPWc2Hj1X3nzqdtNqeg06QEKfKCVVxhJa6RPrNL9ci5MxG+PE0/dtqd5m6JAQ9GoigmqlkEGmxpo2yhiTaFQDHYKBKkoJggxWE6CaWkpaDCRAhzGTMFBIAlrDKH0KKUKUZbAMoMMYkYQQASBAKRV6Y+DUimUO6LAMlpDBABAwwjIIRkixmpr5ZAYdxhwjsf/lywAAAADgsVc/szkeoMM8L5YFuPuuOwEAAMCVg0MZsTmaQYf5eIx5jKCuHBwCAAAA6L2NeRkzdNgcLYc76+mOd174dCdJEwbg1u0OErZXTZWxmuqfZR4H0KEqHz7/7qVXqurxyGkAIYgAAUdjjO97cxH+Bxjp+96ObDbVAAAAAElFTkSuQmCC',
page_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKzSURBVBgZBcFfqN5zHADg5/P9/d53/87OmdnU2jIttiGJEZO1SSh2YcWFXHCBEpIibsiVO3GhlhIXascF7maoiY200iSbIYZxmv3/55yd877v7/vxPJGZ7nplz6MLxuc9XMIGzEeAQAZZjWr9/eyp3B3nz7ywd/vWAUBkpntf+2bv+89suLqUWJI0EoBMSCf+G9rx3Vn/Hpmb+enAz0u/3r51DlooTazrt82Sg8c1JYIgK11Wo1G6OOxsuWqxW9a29tUTC7OuP7P56Z2XfPXWfXMFImIsM5t+U/QKbdAUmiCCIlCtGO/bevMKV1w5b0E7vug9KJAEFJSgRCgRSoQSoTTh6OmLVo8NXbM0PL7xUk2veRBagEzahhAyQu2SEtqGfha/nUqDOlLrwKY180RbRtBCJok2ipQSUUIgK9kUetXS39421h12sW4hVoECiYoISoQSoUSIDL0m9EoYO/KB1Uv/tmbzI5z+zFo/FChQR2RSkchMpFIoEeb/87FV7Zcm1t1k+q9degtWemr1F/1dz655qIVaU2aSpAQggd65703cep9u5qD5y1aZ7f6xesu1yszx7S10lS7JSgWQoCbD/krDM38qzUlRpvXHTzI747+jc00LdZRqple/HwipBplh5ewhD3rHxKKhOppV6lnKYjkc+HHysF9/OfVEC6NRp+tYMd5z42VFTZYd2+ny6Q+t3bhRDj4X3QUHPh0YTs9Z0g9vTj0wnJx8abKF0Vyto1pz0Ilvpzo10mPnPrF+0x0GJ7dr2wk/ftn5qPeks9ff443bSz343Ldz0MJwtju2qN8se/G6waLMLJL2h0Y0U5p5yx36YkZv3ZNevu1+Eer8frnQjeoUtBCRr9/9/J5tEXFDyoWwbXm/vfuPfc350yPvHr5zdGj/4s7kbpitte5vix3wP04jPQSDxpk/AAAAAElFTkSuQmCC',
page_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALZSURBVBgZBcFLaJxVHMbh3znfmVsmmaRJehlQUqE1EQu1xKAuxForgmahFhXsxoXgokZwISKIIEJ3guii3bkQstCFuHAhrWBttRZsiaIlaFNtICbmOqbNfDPznfN/fR4nCYAn3/v+lVqjctI7JoEq4ABwgBzIiGYLrQ1967a33rp4ZroH4CQB8PTpHy5+NjN5n/duSJAhAACQAMTanYLZn1usLHbb13+bH750ZrobAAB85sbLIRv6fZXMOwcOZJBkxCjyInH04AAP3Ru4Ymt9somtx17/elcAADg0utDoLZzNxoqIkrBkKCUsFkSrszhyGqjRbJSZnmpyrrxSu7lc/zQAbMwdyyb3TIVKfS99pd2oiKiIpF5OvjXP8uCLWGU3y5s5Y/0ZlVJg/yMjXPjp7xc8gJLd/fDotaxUG8N2ruN8HUuRTusP1hsv0x1+hnIWuLHhuLIY+e7PNpVShgs+hvWrR8tK9urgyDiQYfkKhC5bi+fo7JvB9jxPKIQyD2Xw5jDLEB6cJ1hKRxTttcrAQRRzyJpQFJgFXHuR8l8fUSpyCgZo730WV24iLwyQICjaJ9WRB0fzjXksGuW+A2CB4ebj5Bu/kIoe2//MoYkPCNVdCIdwJINk4C2mz3eWLn/4xdVG7PoDrN/4htjdoei06LXX+c/dQ3z0PIy9hMvqBO/IvKNacqQonCQAnnj/x3x2ZqLavnCcav9+brdWKcbeQHedwATJwIBkQg4ONz3H373UCQAAFkU+9yalyjCXN5t8WT9LvjYI/0bkBIAEBgDM7itRdI0AAHB/fc6n7Vt8VXuHoQee4pggCZIDEwiQQIJfVyMyUXQiAQBg7c6g+cMfa/7WIaelhDkhHAAmcAgTyDu8OUJwFpPFAACwsNlc7h8ZH3270atL8ggMABisBQAkqJUczmGlzN1O0ZacJACmTp0/FQs955w7ItQHAIBAgBAACICOmV3zMPs/Y958nDUklyMAAAAASUVORK5CYII=',
page_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALQSURBVBgZBcFfaJVlHADg532/bzvNo9MtY1oXRlRWsCREIlFXoaBmUEIkaH9uCsokEIK6qbtg4UVBIN0kFETQnVQ3FmXOxMwg1BApA9GdTdFJbLqzc87763lSRNjy/i+vDgw2dudkLe5AAgmRiKJbyj83r8eP6b+Zd44d3LEAkCLC9g+PH/ty39qHc07LgkoAEAHh2mzHV7/fNHWpfeuvs+eHJw7uaEMNuUqr++tq2bmrqpwSiSj0ouh2w+1Oz5MPLPH4g7WT5dqiKA/NjL313dDRT59pZ0gpLY6Iqr/K+jJ1ospUiZTIEoqVg/12rFvp3vsbA/Vg8xBkCBJk5EROSU5JTklOSa6S1o3bVi3ueGQ4ee2JO1V91QtQA0RQVyRJpKT0gpzUFf2R/X09LJSuUhZsvK8h1bkLNUQQqFMWQiDlJCEKUWX6ySUppRIyKYMaAgUpkSSBQBT6KkDKUi+JHAoigBpKlwgKEiIC5IyS1FUQiSAkvUKvADWUEiKCYL5927k/jpu8eMby4SFTV69b9/ROA0uGHDt8yMhdQ36dmTE0O1iPjb3brKFX6AWdhY4jh7/WiFkv79ltbm7O5cuX/Tbxrap/wM7nnlXXlVarpe/06frI+cEPaijdUCK8980xq69d9NKeXd7+6HOzF064e+UKo6OjWlf+deDAKZOtKevXrze2aaNLly69nqHb7en1qKfOGh5sgqde2W9+oWPXrl02bNhg27Zttm7d6la7440394GlS5c2aui2S+mWEnnpinS5dRL8dGhc9HrGx8c1m00wNzcnlfDJxwdiy+bN6cqVK/M1dOZ7083+avn+7WuaX3x2NE/8fNSLY4+yadT09LQLFy5oNBpWrVplZGREztnEiVO9ycnJqRQR1u39YW+3E88n8VhemF68/Mb3ffeMLEuNRp+EM3OrO920KNYs+rM/KdFuL5RWa3rm1uzMlv8B/jBGW3bkYMYAAAAASUVORK5CYII=',
page_paintbrush=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK/SURBVBgZBcFLiJVlGADg5/3+/4yOllro1BRZjpmXTAy7rIKCKAQJWrQpAhfRLtokroIiWkRUBEFE7QKpiCjaVBBhdqer3bQyyTE1bzPjeJk55z/f2/PEnY/v3j66ZMEDJWzBQgQIZJBVV+uB6VP5cZyZ2rHn5W19AIitT3+25/VHtqwvJZYljQQgE9KJswO7vp127ND8+d9+2XfpZy9vmwdoSxNrR9pm2a/HNSWCICvDrLouXRgM3b7mYrde1/q6nliUdd3Umy/evHb1WLwye3Jyf4mIizKzGWmKXqENmkITRFAEqvElI7bdPG7tqunRq9bd8+N1tz26NXPRw20SUBBBCpARSlCacPT0BVdf1CiDE1aM7TG+evOl+z951uzJs6+1kEnbEEJGqMOkhLZhJIu/TqXB3BErz79tzfUbnT7whn9+/uPVwSB2tJkk2ihSSkQJgaxkU6iHXXnhHRMbNpg+9Jb3PzxZl82Vnfc9c6RfEhURlAglQokQGXpNaOcnrZh518T69c4ee8+vnx/0Qf+hwX3PHJmCUjsyqUhkJtLg/EkzBz+24O8nTWyYMH/6Q99/9J2Vd7xgOsYToK01ZSZJSjDz334zk3vVbtr86TMWfvmSmam+ZuNTxq7ZZNh9A6AMK8MkK8PK5NFDvti7T3vJVTbfdIula+/30+8Xm1q83eI1d+s1RdfvALS1SzXTEz/0LesfMzb7p8uWX+uryXmv7Z7VxiJ19fPOlSv4qbPr8p7BfAXQdt3QcMj4kp7mq0+cyZ5TR884d7avNzpmYtMGK1atksne452saTDXAWi7+Vq7WrM/FLnlQTVSCgtQk8PSoaOdLKHU0LZRu2HtANrB3PC/xSPN8p039BdnZpFUwNLRFmQy2gsRaq+J2WFX/wVoI/K5ux779N6IuDHlIgBJIiWQYK7W+n1b7AL4H1KjWV696jEzAAAAAElFTkSuQmCC',
page_paste=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJRSURBVBgZBcExiJZlHADw3/O+732enKYdQVeklWBeoSBFtmgNNrRcQ9DSFBQZDU4OEYSNbQUtCg5FEAUpDdXSUliBQRBEBBXeVuFZnvKd1/e9z/P/9/uVzATnX37wEfwCAICbpy6s7wUAACjnXnrgUbyFtV3Ld3vhjbO2rv8Alu465sOzr9veugUf4dKpC+sXAWDApWdeObNv+Z57/fPV+zJTm22BzHTiyD5LR0/KzLXPzr/3LC4CwID7l1fus/n7FTHetv7JO2QiXc8fpbTx83eWV4/tBgCAAbLORR11+w+LVmWmj9tpLUMEcPO3LeX401599/O8MVv59c/1vx67fG5te4Boo6ijGGfa7D+kNoQ3n1u1MQ0FkWlsYeiP+ODK5sN96a8++doXBweIOhOtkqEUMum7zo3b6Y+N1HVprOHWdvXUQzsdP7TX0qRb+TbbTx1EnYs618a5qE3UBvrC4sCkLyZ9sTjpXNvcduhOXnxijzrmgQFinMlxLmuIsZGpLaZSWOjJJPticehc/TdN/555fP8OC0NngKhzUZsYm6hBpMhUFH3XASVFJDt6pSv6vpcYIMcm503UJmojgABFEfrCZOiUTBFFKUUmA9SxamMTrYmxkURLBUNHVzqR9IUuMGHnQGYaIOdVjE22JmvISNCiYgAAAJGVKAZc3p5OT+zatyprE7WRicGsTrEXAADM6lSJrgx4++svP92NowBw7fDzFroD9iyOMulKUQpQ0Hd3iKzzkpkAAODkme+/6btykG6F3KIgQVFKZJvuWVrY+T+vNUkTODP9hQAAAABJRU5ErkJggg==',
page_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAITSURBVBgZBcG9r59zGAfg6/4+z+/Xnp6qKol2IuKtNmlEDMLCQBeDySIRGxuDRExiszXphkTS/8Bo0DJIREQQSVkk4qWqDY32nPM898d1VRLPvHPx5Z1jh14a5QwOo0AhRdrS/fP1q/m0/rn25qXzZ/cBKonn3vvi0sevnzk9Rh0PkwCQQFy5ceDCV9f9/svefz989+OJz8+f3YMZxlQPbefp+Pd/mkYVRZo1bVni5sHq6Qdu8/iDsy/7ypH0w9eeeu2TOz479/zegKo6mmTaTsNmMBfTYCqqGArt1LGts4+dcu/9h3bmY7sfwoBQMDCKUWVUGVVGlTGV3/6+6Z6jBx45UV594k7TZnoRZoCEeaKUVOk1jDJPbDP8dDX2e9G978n7Dql5LDBDQjDXEBHUKIU0mQZbRpfuSQxqgBmCRhWlBEGazQSoodaSEY0EmKEXEhqFJGAMdJmnkCJEWZu1gRm6IwkhAkCAUirMg8bhDesSMMParCFNAwjosDZBhxSbaVj2FzBDL9GJk2+/IoqgVyrSgLQK0B985GCvwQzLslpXtneftHv6NOsqvdIlCR3SKu3G5cvScXBrATMse91Ldxzs141vv5EVvYIKLSoINWKeRy9rLzDDwa31j93tdNdfb727m2QIDbh9ZwYJO5uhSm+m+ndd+leYoSrvP/vGxReq6tHIEQAhiAABt7r763m4AP8D8JchsxSnIZQAAAAASUVORK5CYII=',
page_refresh=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALsSURBVBgZBcFtaJVlGMDx/3U/9znz7LjjfMGXYiq1VkkrUiRoZQUhhoOKrAi/hEH0IUHFZQSREu1TQh8CP0TaEBQtIYikfRD3ciSWc0rlWpuzEmbT5TZ1O8fzPPd9Xf1+8uLHvW8XSnXbnbABWAAIAAKYgClBdXz2lp2ROzMd/YfaUwAAAHnps3L/0Z0bHnVOGg0SDAAAzACMqbmMY4OzTF6rVYZ/H1lSPtReAwDwLpGH8z5pvHyTxImAgClEU0Iwqlnk+YcaeKrFM6BT9aaPzDz3/o+Le7/cWgNwIrLQzJJ84sg58AKJg0RABBwCKKtKeYorvmdtc67gS8UjAADOQAAc4AScCE4EJ4ITwSXCv9NV1izMmKtOEOq+JsklrwMAeAAz8AkIgomg0biRjjH03w9cn/uDMB4JaaCpYTVZmKd+5QkHzwLgzcAALw7DMGCyNsb5qe+4r34Jjy9tR02JFlGLBI1UKgNuyxcvXPlp19lmb4ACIiAIBozdHmBZvsjywgqGJgYZmxwlSwNNi1cTQ2T4SqrdHWebAZwGMAMFDDAzrs5cYuj6BU799i3zFcfuZ06ye9Mpbk1XCRVHqLyTtu3fWADwqoaZgYFhAOxYdxAzUIOokeODnWxb/xGL8is58MonvNw5yLlPz1cBfFSIBqagAAAYQVOEHF+V93G3OosavNV2gA+Ovsdc7ZcFT+wKFrLwj9dgqBn7L6YIhgqYCagjvfkhTQ23qcsH9nzzNFkaeLBpLdu3vcHwyJ/09p/r9iFEYoRVpRzrlzvUIBqc7nuN+xtLtDa3Ei3yWMs6okXUIqNXx+npK1+IIR73oaYaVC2NyM8TERXDEJa2nOTvy3uozA+QhcDwyAghDYQsouGBGELc+1fXtR6f3Ys3ivlk2b7WtGhmDgMFABZt/JyOrnep3Jmm3PkrImgh7+627uieGO3a0gPgRezg5r19r4rIk4bVAwBgYIDxJjE5nGvbeSYD7qnqkHccAwD4HyXFhGV2sNBxAAAAAElFTkSuQmCC',
page_save=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKYSURBVBgZBcFNiFZlGADQ87z3zjfjzDiMk5VimERmBmFpKgVRVATRQLRo5aaNhFA7pZW0qV1Ci0BoUVRQtLc2UmQaJVhBVChUhP2YDuow5sz3c7/36Zx46uipF9fNTR4oYQ82oC+QQVZdrb8tX83PY+X6kdPHF4cAAPHMG2dOf/jKnp2lxHymBiAT0tJ/Ix+dW/bvxcHqLz+dXzhzfHEAAG1pYkevbeZ/vqIpJWRlnFXXpbXR2OPb19t/T+tsXZrOeu/1x17+dMOpt58dAJSImM3MptcUbdAUmiCCIlBtnutZ3LvZtrsn17VzM+8BQEkCCkqEEqFEKBFKEy5dW3Pn7Mh9C+Hgw7doJpoXAKCFTNoGEeo4KaFt6GXx69U0rJ1ahx69a1K0pQOANpNEG0WVooTA+ycuWD/J6rCzYbbnnytrPivs2jQ7dfidH/P6ypovfvh7f5uoiKAIiaysrfY9sW+H53bPOvHtX44e2AkAPvhySa1xttSOTCoyE6kUVgcj12504OS5ywD6HSsDlpaHIqq21pzKTBIASSmhSUbj6tih3WpNiUaabhkMBlZuDrWjYfZHNade+24oI2UGlYWJoj8Yef2TP5SgG1d1XCUyk2R6omq7bmw8ZvPchAduLcbJODh7IQ3GnaZMIIWxiJA1ZVZdNzIcdtpuUGtXaw7H4ptLnRTgxurIRLQOLa7X1SEKSUY11fa8+fFFV1duakf98eWZXrPx1fuHM1mzVMCRU6kxoVrz7vcv2Ti9VSiWVv90cO9biimrg6E2Io89ffir5yPiwZQL6Eu2b52d2raJ22dveHLXPvO9LRSW+1vcNtOZ7tFGiMwEAAAPHTr59fY71j0yPzlvqp3T1ZGoVeRYjVXnL/7uysrI/62cRssvlMuuAAAAAElFTkSuQmCC',
page_white=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAC4SURBVCjPdZFbDsIgEEWnrsMm7oGGfZrohxvU+Iq1TyjU60Bf1pac4Yc5YS4ZAtGWBMk/drQBOVwJlZrWYkLhsB8UV9K0BUrPGy9cWbng2CtEEUmLGppPjRwpbixUKHBiZRS0p+ZGhvs4irNEvWD8heHpbsyDXznPhYFOyTjJc13olIqzZCHBouE0FRMUjA+s1gTjaRgVFpqRwC8mfoXPPEVPS7LbRaJL2y7bOifRCTEli3U7BMWgLzKlW/CuebZPAAAAAElFTkSuQmCC',
page_white_acrobat=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHhSURBVDjLjZPLSxtRFIfVZRdWi0oFBf+BrhRx5dKVYKG4tLhRqlgXPmIVJQiC60JCCZYqFHQh7rrQlUK7aVUUfCBRG5RkJpNkkswrM5NEf73n6gxpHujAB/fOvefjnHM5VQCqCPa1MNoZnU/Qxqhx4woE7ZZlpXO53F0+n0c52Dl8Pt/nQkmhoJOCdUWBsvQJ2u4ODMOAwvapVAqSJHGJKIrw+/2uxAmuJgFdMDUVincSxvEBTNOEpmlIp9OIxWJckMlkoOs6AoHAg6RYYNs2kp4RqOvfuIACVFVFPB4vKYn3pFjAykDSOwVta52vqW6nlEQiwTMRBKGygIh9GEDCMwZH6EgoE+qHLMuVBdbfKwjv3yE6Ogjz/PQ/CZVDPSFRRYE4/RHy1y8wry8RGWGSqyC/nM1meX9IQpQV2JKIUH8vrEgYmeAFwuPDCHa9QehtD26HBhCZnYC8ucGzKSsIL8wgsjiH1PYPxL+vQvm5B/3sBMLyIm7GhhCe90BaWykV/Gp+VR9oqPVe9vfBTsruM1HtBKVPmFIUNusBrV3B4ev6bsbyXlPdkbr/u+StHUkxruBPY+0KY8f38oWX/byvNAdluHNLeOxDB+uyQQfPCWZ3NT69BYJWkjxjnB1o9Fv/ASQ5s+ABz8i2AAAAAElFTkSuQmCC',
page_white_actionscript=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIqSURBVDjLjZPNi1JRGMan/ooWDbSKNq2sgZqh0UgqKVoOU7OooEWLgZi+JIaYGolaRAS60JXuxJWoIC6E0KAgAzGbCqpFmua393qv9+PoPJ33THPHcYy68HDPvee8v/e8zznvFIApEn8Octm4Zv6hQ1z7rbgRgE3X9S5jbDgYDDBJfB5er/flKGQUMEPBiqJAVVVLkiSh0+mgVqsJSLVahc/nsyDbwfsIQAs0TYNhGNDevIX29BnUxx50u13U63UB6Pf7oCR+v38LMg6gYCOdhnb1GgaeVajnL0CWZTQajT0lCU/GAea379AWFsHu3kJ/4TLUO/etUprNpthJpVL5C4Ax6I/WwVbvoe9+AMazMvrHzSMI7YT8aLVakwHs8xdoS1eguC7CeJUBa3fEwkKhgEwmI+pP8/Ly+fxkgP78BZj7NgYP3ZDn7FDXPGJhKpVCuVwW/tA7HA7vBawdPrJEmZl7hQc7IJ2YtwCxWEyUIgzmCgaDuwF157kDlVOnC+bKMmS7E8a79zA3PsEs/0Q8Hkc2m4VpmkLkB5URjUa3AMpZ1+uew/lVmnMw/cZ1qOtPrGOirKVSCclk0gKQQqGQOFYB6NnPKPKsfdNYvgnJdQnsV23XWRMkkUig3W6LMSkQCOyUIJ+ch3R8Fj+O2j6YHzc2J/VAsVgUEBpHIhHkcjkaDy0P/hh5jBuk0sQ4gO4AXSIa09b595Cv7YnuHQFME+Q/2nlb1PrTvwGo2K3gWVH3FgAAAABJRU5ErkJggg==',
page_white_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGSSURBVDjLlZO7S8NQGMVb/weRgoI4iIuCVuoiuBfBwUFwaosPqJODk5OLBVutSlMsuDg4OVWkYO3o4Cha/BNS+qLP9K3HnEtT07f94HATcs8v33eSawBgUGtSlVmVZYimVY3Ro8nQBJgrlUq2Xq9/NxoN9JL6HD6f76oTogEsNBeLRSiK0lIul0Mmk0E8HheQWCwGSZLaICwjAdxQLpdRrVbFShUKBWSzWSQSCQEolUrgSwKBgIB0AWjWRAAN+XweyWSya6RmJsY2gDpGawOvObc2SiqVEp3Istwf0Ck9hJ0wj3Q6/X+AHsJxmAlBIwGoWq0mciGEGhnALkJvDzgK2LB3sQH7mRWrjtmbgQCaNAVf73HyYMdTVMK7HIb3xYkt7zxWDkyeoQC273BbEfy8RPDrGixPZBfeyD4B5aFfgVo/XkQoegt9PX5IBEAP+OmXwdrhDNxhB1xhmzC7nm1/HTR/x2U1ZUXd3PMw+YOn2PTM4Ty8I97MlfcigyZgipBBR3lhe/zO4jQpbJvrkn3CT+8vh7avwsYhJlIAAAAASUVORK5CYII=',
page_white_c=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHdSURBVDjLjZNLS+NgFIad+R0KwuzcSQddunTWXraKA4KCuFKcWYqgVbE4TKJWNyqC2oHKoDBeEBF04UpFUVQqUoemSVOTJr2lrb5+5xsTUy+jgYdc3yfnnOQrAVBCsK2U4WFUvUE546OTcwk82WxWz+fzt4VCAS/B7kMQhB9uiVtQReFkMolUKuWQSCSgaRpkWeYSSZIgiqIjscMfSEAPZDIZWJbF94RpmtB1HYqicEE6nQa9xO/3/5OQoM57/qm2a3PGtyzDtxzF/FYMe6c6F1DAMAzEYrFnLfGZ1A9devqC8o2wpmL8jwJhRcbw7ygGAxJYS7xvuxVVVXklkUjkUdAshgP+DRVfureXbPPcuoKe2b/QDKtIQpXQPOLx+KOgf0nGCCu9smHiu7u8IGuDBHRsS6gdmgmJHEHfLwn9wSgqagc6Xvt8RC6X48MlCeEI2ibDIS8TVDYGBHfAO3ONowvTOacqSEBQNY6gpvOkp3cxgq8/Q8ZxyISWsDAwfY32sSscnhk8SFAFBIWLBPQZq1sOvjX5LozOqTBaxSu0jF5iYVV+FnZTJLB/pN0DDTv7WlHvtuQpLwrYxbv/DfIJt47gQfKZDShFN94TZs+afPW6BGUkecdytqGlX3YPTr7momspN0YAAAAASUVORK5CYII=',
page_white_camera=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIiSURBVDjLjZM9i1pBFIbfuSqu+AFJY+PFNCkkQYSQLAprm8r/kGYhprNYG9MErNIFFPR3pIqQJpWk0iYqKnHXqKhcvz9Xr5MzE6+6sAQHDkfvzPvMOe/MMM45xGCMeSi5KUz4/+hR3JFuJ/8JwB7yarVajTabjb7dbvljQfM8lUp9obXPKBSpOwG8EeLZbMbn8/khxuMxHw6HvNvtSkin0+HpdPoAMcRMAMSC5XLJ1+u1zCKm0ykfjUa81+tJwGKx4GKTbDYrISyRSHymHzeGF0ZFIqgiJJNJxONxUPnIZDIPzDCbzZcKUW8ikQh8Ph9UVUWhUMCn9y/x8e1PlEolxGIxOJ1OOBwORKNRDAYDUDVHiN1uR7FYRCAQgMvlQigUwvdfLeTzV7i+DoB8QLPZhNVqRblchsVigcl0PCizoPv9fny9fYrcnUK2WHHlVhF4riGXyyEYDKJSqcDj8YA8EWUbvsmhCLLNZsO3hoIRLjBmF/ihOeQuk8lE9i6EZCyoXfldQA4AchXkNHYSysXpgpu57FPMPQZQFOUI0DQN9Xodl080ONhKxgvcolqtSpEhFidycmeOHvT7femsqv7BOzJRVNNoNPC71YLb7ZZzXq9XViMgp7tLQLvdTlOZH2q1GjPO/v7+Xmbh/um9CIfD4s08AEgR+/f1NfWY39/Kc8aOzAwq+9snGtvpur4U+RwxrV1Q1tnJc1b3z1k5swJdPO2/GZyOpOSiQJ4AAAAASUVORK5CYII=',
page_white_cd=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLjZNfa9NQGIdnP4cDv8Nkn8PL6UfwSgQZOoSBYkUvZLN1lMFArQyHrsIuWkE3ug2t1K3O0LXrZotdlzZp0qZp/qc9P8852qyyigs8F8nJ7znveZN3DMAYg14XKROUyf9wiRIKckOCCcdxNN/3+71eD6Og64hEInPDkmHBJAsbhgHTNAM6nQ7a7TYkSeKSer2OaDQaSAbhC7efJGY28gZWPrUQTyt4l2lCKLfR7XahaRpkWeYCy7LANonFYr8lqzt26PUXIxzf7pCfioeS5EI2fVQkG+GVH0hlRVqFjmazeeZIvCc0PBXf1ohu96GZBEnBQMMmcAjgeH3cWRKQyTf4URRF4ZWIongqoOFURXZpUEOt1YNm+BzDI6AeFKo6IqsF3g9d13k/VFU9FSytK9V8zUJiR0WbBh+/2cVich+trodvNQeFEwvTsa/8C7Dzs54wUSBYeN+ofq+ageDZmoBX64dQdRcbByaEqoGbTzPwPA+u63IJIxDMrR2nDkUTR6oPxSJ8ZxYuNlxsHtnYLal48DIH+om5gMGqCQSP3lam7i+XSMfp40AFsjWCrbKHdMlGpeng2uxHpHM1XgGDhf8S3Fsuhe4+3w9PL+6RvbKGguhAODaRLSq4OvsBL5JFvutAMCAQDH6kK9fnZyKJAm4tZHFj/jMexnPYzJ3w0kdxRsBu6EPyrzkYQT8Q/JFcpqWabOE8Yfpul0/vkGCcSc4xzgPY6I//AmC87eKq4rrzAAAAAElFTkSuQmCC',
page_white_code=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHtSURBVDjLjZM9T9tQFIYpQ5eOMBKlW6eWIQipa8RfQKQghEAKqZgKFQgmFn5AWyVDCipVQZC2EqBWlEqdO2RCpAssQBRsx1+1ndix8wFvfW6wcUhQsfTI0j33PD7n+N4uAF2E+/S5RFwG/8Njl24/LyCIOI6j1+v1y0ajgU64cSSTybdBSVAwSMmmacKyLB/DMKBpGkRRZBJBEJBKpXyJl/yABLTBtm1Uq1X2JsrlMnRdhyRJTFCpVEAfSafTTUlQoFs1luxBAkoolUqQZbmtJTYTT/AoHInOfpcwtVtkwcSBgrkDGYph+60oisIq4Xm+VfB0+U/P0Lvj3NwPGfHPTcHMvoyFXwpe7UmQtAqTUCU0D1VVbwTPVk5jY19Fe3ZfQny7CE51WJDXqpjeEUHr45ki9rIqa4dmQiJfMLItGEs/FcQ2ucbRmdnSYy5vYWyLx/w3EaMfLmBaDpMQvuDJ65PY8Dpnz3wpYmLtApzcrIAqmfrEgdZH1grY/a36w6Xz0DKD8ES25/niYS6+wWE8mWfByY8cXmYEJFYLkHUHtVqNQcltAvoLD3v7o/FUHsNvzlnwxfsCEukC/ho3yUHaBN5Buo17Ojtyl+DqrnvQgUtfcC0ZcAdkUeA+ye7eMru9AUGIJPe4zh509UP/AAfNypi8oj/mAAAAAElFTkSuQmCC',
page_white_code_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHdSURBVDjLjZNPaxNBGIdrLwURLznWgkcvIrQhRw9FGgy01IY0TVsQ0q6GFkT0kwjJId9AP4AHP4Q9FO2hJ7El2+yf7OzMbja7Sf0578QdNybFLjwszLu/Z2femZkDMEfI54FkRVL4Dw8l8zqXEawMBgM2HA6vR6MRZiHraDabH7KSrKBA4SAIEIahxvd9eJ6HbrerJKZpotVqaUkavkMC+iCKIsRxrN6EEAKMMViWpQT9fh/0k3a7PZZkBUPmqXAKCSjAOYdt21NLUj1JBYW7C6vi6BC8vKWKQXUXQcNA5Nh6KY7jqJl0Op1JwY/Hi7mLp/lT/uoA/OX2WLC3C9FoQBwfILKulIRmQv1wXfevwHmyuMPXS5Fv1MHrFSTmhSomnUvw/Spo3C+vg3/+pJZDPSGRFvilNV+8PUZvoziKvn+d3LZvJ/BelMDevIZXK2EQCiUhtMDM53bY5rOIGXtwjU3EVz/HM5Az8eplqPFKEfzLR91cOg8TPTgr3MudFx+d9owK7KMNVfQOtyQ1OO9qiHsWkiRRUHhKQLuwfH9+1XpfhVVfU0V3//k4zFwdzjIlSA/Sv8jTOZObBL9uugczuNaCP5K8bFBIhduE5bdC3d6MYIkkt7jOKXT1l34DkIu9e0agZjoAAAAASUVORK5CYII=',
page_white_coldfusion=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHiSURBVDjLjZNfT9NQGMbBb+C9GD+DmH0BvQIXECJKEGUmXpp4TyAQAtdotsTFxAs/AFeEhBsTjHEOZFKmMOBCuNnfzHbrunb9cx7f99SVbtkCTX5p057nd3qe9gwAGGDouEUME5EruEPcCHIhwbBlWZrjOJ7ruugFPUc8Hn8bloQFEQ43Gg0YhhFQq9WgqipKpZKUFAoFJBKJQNIOD7KAB5imiVarJc+MruvQNA3lclkKms0meJJkMulLugUcbsMCDigVA0u5Fl4foWNJspNuAS0jGMDXimrjVVZg5gB4fiDkm+Tz+f6CbhZPBaZ/AisnLo4rDdlHtVrtFExmzNv3P31Z2yh52Cx7MB0/rGgepjPA033g47mLYt3vpF6vXwqmfuFu7DeKL7PA3CEwqwCq5QtWzwQm9oCxNBD9DnwuOrITJhDMZLHDwcmUnlv/I/D+Qvifq+niyQ8/vJwTWCVONVuWy//DpUDB32dUUOTd1nx73RXTxcKxwMMU8EYRskzbtiUc7hCMZfBhitb4KO1ZMSrrBfF4Fxj5RvfotQ9VNwiHCQQPdnFzNI3taMqzxikQpVlHKRzbF/ha8eTsvej3GUW/fdADLxD8l9yjggx+cJ0wjdXl7g0Jhlhyje3chrf+0D9WXtarnqRU7gAAAABJRU5ErkJggg==',
page_white_compressed=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJmSURBVDjLhZNNS5RRGIav8+HMvDNO5ZhOYqXQF2FgVNRCCKM2EbQ1ok2b/AG16F/0ge5qUwitghbWooikiIhI3AQVFER+VsyMztg7vuecp4U2ORH5wLM5cK7n5r65lYgAoJTaDhQBw/9nAfgiIgEAEWENcjiO43KSJN45J//aOI5lZGTkBtALaBFpAhxNksRXq1Wp1WqNrVQqUiqVZH5+XpxzMjs7K6Ojow2Imri9Z1Dntjwo2dObZr7vpKXFoDVAwFpNvR6za9du+vr6KRQKrKysEEJgbGzs5vDw8DX1/N6Rrx0HOrpfvOqnWs0CCgQkaJTJEkIAHENDFygWi01mWGuP2Vw+KnT3djPUM0eLzZO4L6ikztQz6Dl2i4ePxgk+IYoylMtlQgg45+js7FyFKKUk/llhevplg9zTtR8RC0AmSlGtrGCMxVqF9x5j/gRlRQLZbIbt3fvW4lwmpS0IhCA4FwgEjDForVFK/Ta9oYDa8jdmpt83Hndu86DaEQkgHgkBrXXT5QaA4FuiqI3itl4IPzHWk7G5NQUBQgISUEoBYIxpVlArle9+fCbntFY6qM2Z4BOWazFzS13UPrwjlUqzuFhtXF9NZZ0Cn7hLc59mrly+/uPQ+OO3T+6PP8W7OpH1fJ6cpLU1hUsSphcqRLlNFHK6GXD84nuvlCoDS1FrgZn28+T5zom933jzeoKpyZeY9oPceOJpz1e4erbtLw/WTTBZWVpaVNmcYuvWDk6eOsPAwCCLseHOpCOfNg0vgACg1rXxSL1enzDGZAC9QSOD9345nU4PrgfsWKvzRp9/jwcWfgF7VEKXfHY5kwAAAABJRU5ErkJggg==',
page_white_copy=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADHSURBVCjPdZFNbsIwFAa/5B5FuQPCOQ9HQuqCq3SHUAtrrgFCSKQk8Y80XTjYJqiajRfj55GftNBKJtGoRiXSytlAZORzM1ckExjouHEm0LPdqHkTRnruXAgEPGpVq1JVCJ6RB3dOfHFEa7W5RzLx3kjPDse8ZxKisue9JwkByzevPQGZQnD8kHuunEvB43EcyD0dt/kEzzGdLD2/k9Ckb261zs9ZhiggVRO12jzN4Z5C+tQq90T+ETK20/J1tU2xeCOjpT7+APfbTaDnTb/mAAAAAElFTkSuQmCC',
page_white_cplusplus=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH/SURBVDjLjZPNaxNRFMWrf4cFwV13JVKXLuta61apIChIV0rblUqhjYpRcUaNboxIqxFTQgVti4hQQTe1C7FFSUmnmvmM85XJzCSpx3efzmTSRtqBw7yZ9+5v7rl3bg+AHhK7DjClmAZ20UGm/XFcApAKgsBqNptbrVYL3cT2IQjCnSQkCRig4FqtBs/zYtm2DdM0oaoqh8iyDFEUY0gUvI8AdMD3fYRhyO8k13VhWRY0TeOAer0O+kg2m/0LIcDx9LdDgxff5jJzKjJzCmbe6fi0anEABTiOA13Xd1jiNTlxfT01UVB/CfMG7r/WILxScaOo4FpeBrPEfUdWDMPgmVQqlTbgtCjls4sGjl16PxuRny5oGH3yA7oZoPjR4BDbqeHlksLrUa1W24DJWRU3Wer9Qw/Gk+kVmA2lGuDKtMQzsVwfl6c3eE3IUgyYeCFjsqCgb3DqQhJwq/gTY7lyV61Jdhtw7qFUSjNA/8m8kASkc5tYXnN4BvTs1kO23uAdIksx4OjI19Grzys4c7fkfCm5MO0QU483cf5eGcurNq8BWfD8kK11HtwBoDYeGV4ZO5X57ow8knBWLGP49jqevVF5IKnRaOxQByD6kT6smFj6bHb0OoJsV1cAe/n7f3PQRVsx4B/kMCuQRxt7CWZnXT69CUAvQfYwzpFo9Hv/AD332dKni9XnAAAAAElFTkSuQmCC',
page_white_csharp=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJOSURBVDjLjZPbaxNBFMarf4cFwb9AIgXBR18Enyw+i1gs4g01kphSlPjQeAtNzNqGNLVpNCGhEvBS21Rr0ZIK6ovFiKbNbXNpdpNsstncUz9nNiauErEDHwMz8/1mzjlz+gD0UZGxh0hFNPAf7SXa3fUpAKparVZoNpvbrVYLvUT2YbFYTEqIEjBAzZIkoVwud1UsFiEIAjKZjAxJp9NgGKYL6Zh3UQA9UK1WUa/X5ZmqVCqhUCiA4zgZUKlUQC+xWq1tCAUM3v6+74hu2cH4eUz6OcwFcvgYEmUANYiiiFF3Aq5XHIJRCeqHLOJbFcg5OW6Mqm495fL2NznYl7OwveYxsZSF6QUHEpIc9+eQgOvuFL6EMjC6wrg4GZZfIwOGbazX8TaPY/qAr5Ms72oOBt8WknwVem8KWmcCY0/S0E1HcXYyhjNMBAYH2waYF8izl3I4eGLqmjLjz9by+PRNxCMS0k0C0c+yMDjj0MwmMOGJ4+Vqtg0Yn+dwf5HH/sG75/4uWzAiwbfCQ+dMYSGQxdhMHMPmMFY+8MgX623AiDu9+YAADg35LErzHU8SGkcSI4+T0DoSuGRnoZ5mcdIUwdC9zd85OHpjQzP+nMOVmZj4NSZBKNVh9LbN6xslnGai8CxmMP+Ol81criwntgugZTysDmovTEXEUVcKV8lt520s5kjJvP4MTpkjyApVXCZmvTWKRqMh6w9A5yO9Xy9ijUgZCi1lL/UEkMUf/+qDHtruAn5BDpAvXKYbOzGTsyW5exWAfgrZQTt3RFu//yfHVsX/fi5tjwAAAABJRU5ErkJggg==',
page_white_cup=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIRSURBVDjLjZPJa1NRFIera/8ECy7dV7txkb2UOoDgzo0R3YuLrFwWIVglWQRtN0GCLkIixJDJQJKGQOYBA4akmec5eSFT/XnPsXlNsWIffOTdd3O+e+6PezcAbBDiuS7YEmz/hxuCq3LdmmBrOp32F4vFyXK5xEWIeWg0mnfrknXBNhWPx2NIkiQzGAzQ6/XQaDRYUqvVoNVqZQkXGwyGm2q1+k00GkUkEkE4HEYwGGQCgQDS6TSKxSILJpMJaBGdTvdHYjKZHvp8vuNsNot6vc7QavRLq1UqFcTjcbhcLrmLFZyJ2+0u9Pt9hC1f8OHpDt4/uoO3928zmscKHD5/gKPPB8jn8yxpNpuoVqtnAqPRiOFwiPGgB/fhPr7uvcJH5S4Ont3Dp5dP8G3/NX4cfedCi8XCeXQ6nTOBzWaT5vM5J0yTFFy325WhtmkbhN1ux2g04gVlgcfj+UmDUqkEh8OBcrnM7xRaLpdDIpHgcSqVYihEYr0DL61O6fv9fhQKBd4vhUrpk6DdbsNsNrN8Nptxt7JApVK9EMW9TCbDEgqI2qUOSELvJPF6vbw9Kj4nEM81pVJ5V6/XH8diMQ6IaLVaLAmFQnA6nfyNslohC05P4RWFQrFLHVitVoYSF2cEyWSSgxOn9Bx/CWggPv761z24gBNZcCq5JQKSaOIyxeK/I769a4JNklziOq+gq7/5Gx172kZga+XWAAAAAElFTkSuQmCC',
page_white_database=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHVSURBVDjLjZPLaiJBFIZNHmJWCeQdMuT1Mi/gYlARBRUkao+abHUhmhgU0QHtARVxJ0bxhvfGa07Of5Iu21yYFPyLrqrz1f+f6rIRkQ3icca6ZF39RxesU1VnAVyuVqvJdrvd73Y7+ky8Tk6n87cVYgVcoXixWNByuVSaTqc0Ho+p1+sJpNvtksvlUhCb3W7/cf/w+BSLxfapVIqSySRlMhnSdZ2GwyHN53OaTCbU7/cFYBgG4RCPx/MKub27+1ur1Xqj0YjW6zWxCyloNBqUSCSkYDab0WAw+BBJeqLFtQpvGoFqAlAEaZomuc0ocAQnnU7nALiJ3uh8whgnttttarVaVCgUpCAUCgnQhMAJ+gG3CsDZa7xh1mw2ZbFSqYgwgsGgbDQhcIWeAHSIoP1pcGeNarUqgFKpJMLw+/0q72azkYhmPAWIRmM6AGbXc7kc5fN5AXi9XgWACwAguLEAojrfsVGv1yV/sVikcrksAIfDIYUQHEAoPgLwT3GdzWYNdBfXh3xwApDP5zsqtkoBwuHwaSAQ+OV2u//F43GKRCLEc5ROpwVoOngvBXj7jU/wwZPPX72DT7RXgDfIT27QEgvfKea9c3m9FsA5IN94zqbw9M9fAEuW+zzj8uLvAAAAAElFTkSuQmCC',
page_white_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGqSURBVDjLlZM7S0JhGMfVox+gqYZuQkMETYZNQmi2+QGKligiCBoalFragoqGzDM41NRQQy4VKDhUSyC0NLR1EeKIt7wePV7/vc/BI97NF36cA+f9/97neQ6vCoCKrVGGgWHswyRDQxkFVU1gkCQpWSqVKuVyGZ1g3+Fyuc5aJYrASOFsNgtRFOukUikkEgmEw2FZEgqFwPN8k4SWmgS0IZ/Po1AoyE8ik8kgmUwiEonIglwuBzrE7XbLkjYBhRVIQIF0Oo1oNNrWUm0m6iYBa6O+gd6pb6WVWCwmVyIIQndBK40SqoTmEY/H/y9olFA7NBMSDSQgisWiPBeSEAMLqIrvWyde1mbgt+jwtDIBfl7D9xRQSCHoOceb3YT8wymq716I17sIbM9WfGbtTl8Blf+8OoUcC8NpAxxDwKEe0eMF+Ba5z75/gaCyq68eNK7EwQj8Zm21UVDtNoPH5XFkL9YBFpLsKvwyglscfFbuR7kLc2zKItvc8TJ93ZwgsDkNwaFHZE+Hjw01/DZtxWvl9hXBGEl6XeXLpWH+zsIJVPa9hQtfmbgjyv4BPlWugike25IAAAAASUVORK5CYII=',
page_white_dvd=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIQSURBVDjLjZPbbtNAFEVLvoNK/ENRv6W/guChbzwXJRJBDRIgVQglUjClddxUpCEF00RJE8fgXKBtZGzHud+vzWbOgE2gRdTSfrBn9jr7nPGsAFghsec20xrT+n90h8nj+pYAa+PxuD2bzS7n8zmuE1uH1+vdWoYsA9bJ3O/3MRgMXHU6HbRaLVSrVQ4xTRM+n8+FOOZbjHy/VCohnU4jmUwim81C13X0ej20223Yts0Bw+EQVMTv9/+E5HI5TyaTeZhKpRbNZpNvJFOj0YAkScjn8zxFrVa70hKfCTNvkHk0GoGkqiq63S5YO1yCIKBcLnNIvV7nBQzD+A1gZpGqKYrCo1JE0mQy4QDLshCLxfg8CEzzoP0uQJblCg2Geh/2WwiFQjw6GS4qOooXFl69OeQnQGBqj0AuIJF4XzHKu9BST9EzJeztBxGPx3FudZA4PUNKM7ATPsB0OuWpnIQugMUTbbMAw/yK/PckTvWPOLeLMCwbn5QznHzWIURivB0CkCiNC4hGoxu7EWGRN5I4+XaEY+0AcTUCtaigatexvfMaXwolnoBE5j8Aoih6gnsvHz1/+3hxXIhCLr3Dh8IhZC2GQCAANiNe1QE4cgHOj/Rg897m/pGAF8I2noWfICwFoRU09zj/1hUAvbCPi3/dg2t06QJ+Qe6yqANauImZ7e3x27sEWCXIDa6zI7r6qz8AeSLtQ3VwWP8AAAAASUVORK5CYII=',
page_white_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH8SURBVDjLjZPfS1NhGMdXf0VEQhDUhdCN4X0IYT8ghIJQM0KoC4vushZddLELKyRhQQkSFIKEGEkUCI2oxVhepG5zi1xbc0u3cDs7Z+ec/ezT+x62scmmHvhwDrzP93Pe57znsQE2cR0SdAm6d+GwYL/M1LBVBV35fF4plUqVcrlMK8Q6TqdzYrukJuiW4Vwuh67rdbLZLJlMhmQyaUnigVlC05f4+dbB0tQplp92DsnwPimQBaZpUigUrLtE0zQURSGVSqHF37DhGkVZeQdagszKLJ7HvZtNAhmuIQWGYaCqKps/ZkivPqCwPs/Gp0cYvjnKUTe+F9fMJoFoo96zfJZ9K+sLpP33qRhujPANtr7dJPhqmO/PBxX3+PljTYLtqImPpH13qZge9LUrmLEB1FU7sZd9jJw5MljNthYk/KLnxdFqeAjzdz9Z/z3Ck2fRE36qx9pakAjME1y4Lbb9GTMyTD52GUXsZO3ZadTkL6umrSD4ZZrAezvLH54Q915EjwywtXSH8FQf+t+I9V12FLwe6wE1SmjyAi77Qb6Kt3rGe9H+hKzwrgLH9eMUPE4K3gm8jpPMjRwlHfNTLBbr7Cjo7znA2NVOXA/PsThzi2wyah1pI+0E/9rNQQsqMtM4CyfE36fLhb2ERa0mB7BR0CElexjnGnL0O2T2PyFunSz8jchwAAAAAElFTkSuQmCC',
page_white_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIBSURBVDjLlZPNTxNBGIeLV/8FNUYisaIRqOHswat3DfHgzXhUIwQIQWKUgyFq2pgm6oHgxcKBiJ76YRcS8CJJNVFoqZSaXbof3e3H7s52qz9nRoZIkRQneTKT7LzPvL/JTgBAgI5jlBClvw0nKUdYjSCwIwgRQizf9382m038C/od4XD4aatECPpZcb1eh23bu1QqFZimiVKpxCWKoiASieyRsNHBBGyD67rwPI/PjFqtBsuyoKoqFziOA3ZINBrlkn0CVixgAlZQrVahadq+SDt30rFHQGPsbmBrlltE0XWddyLL8sGCVoSkXFzE1sooDG0LhmEcXsBjuVUUlu7AyMYgZ17ySP8l0NfeQPs6A7+ew9rbAZjK58MLHGsT6wvXQbQFEHkK5Y13yKfG4DdIe4HfaKC4/Jh2MAtSnIA0GULDmMXqq8swcqn2Aqv4Ed8TI/DUaTi5G5Ae9cItDMLKv0fm9TUETxy9dKCAtfht/iZMenHu5l3Y61f/kKVx5CcopJ9h+nYw2ir4JQTbmTkUpCl428/56XZ2gEbo47OTvwVHmUdy/Fw5fb/7rHgLF+nfZ9Ni/pg+vbgCR9+AV06AqPQO1NhfzMGvfcGPlRkkR7piQnCcScSzlSZ7LelhD0k/uEA+TJwnqfFukhwLkuToGZIY7iLxodMkPtjpxu+dWvwNhj+uekyCXgUAAAAASUVORK5CYII=',
page_white_excel=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIpSURBVDjLjZNPSFRRFMZ/9707o0SOOshM0x/JFtUmisKBooVEEUThsgi3KS0CN0G2lagWEYkSUdsRWgSFG9sVFAW1EIwQqRZiiDOZY804b967954249hUpB98y/PjO5zzKREBQCm1E0gDPv9XHpgTEQeAiFCDHAmCoBhFkTXGyL8cBIGMjo7eA3YDnog0ALJRFNlSqSTlcrnulZUVWV5elsXFRTHGyMLCgoyNjdUhanCyV9ayOSeIdTgnOCtY43DWYY3j9ulxkskkYRjinCOXy40MDAzcZXCyVzZS38MeKRQKf60EZPXSXInL9y+wLZMkCMs0RR28mJ2grSWJEo+lH9/IpNPE43GKxSLOOYwxpFIpAPTWjiaOtZ+gLdFKlJlD8u00xWP8lO/M5+e5efEB18b70VqjlMJai++vH8qLqoa+nn4+fJmiNNPCvMzQnIjzZuo1V88Ns3/HAcKKwfd9tNZorYnFYuuAMLDMfJ3m+fQznr7L0Vk9zGpLmezB4zx++YggqhAFEZ7n4ft+HVQHVMoB5++cJNWaRrQwMjHM9qCLTFcnJJq59WSIMLAopQDwfR/P8+oAbaqWK2eGSGxpxVrDnvQ+3s++4tPnj4SewYscUdUgIiilcM41/uXZG9kNz9h9aa+EYdjg+hnDwHDq+iGsaXwcZ6XhsdZW+FOqFk0B3caYt4Bic3Ja66NerVACOGttBXCbGbbWrgJW/VbnXbU6e5tMYIH8L54Xq0cq018+AAAAAElFTkSuQmCC',
page_white_find=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI2SURBVDjLjZPda5JhGMZX/0AHnRQldFwjCs0maplQE8mJ5Wxr6WiWa7XaAoVFrSTbWFpslQ6NDjroAzJYnZQSHQQj6DBaBx7awcQv/P52XT3303pn5GgvXAcvz3P97vu+eO4OAB0k9u1kEjNJ/6NdTJsFXwtAXK1Ws41GY6XZbKKd2Dm8Xu+DVkgrQErmYrGIUqkkKJfLIZPJIB6Pc0gsFoPP5xMg3OxwerZaRx0122UHvn6PYHB4DEO2S6hUKigUCshms0gkEhxQLpdBRQKBwG8IM25x3vPPumafQHP8BBY/f0G3zojr03N4NO/nHeTzeSSTyX9G4plM3vWJTBZbYWxiCvIjWkzO+KDSmmB3PYSqW49o9IcwSiqV4p0sLy+vASSyQ1M64wDmnr6BwTyC8/Zp9PRZMeN/Ba3hNG677vDwCEKdUB7pdHoNwKpn+s6OQKM3QaPVwe32QKZQQaE+BnGXErv3dIKFyyE0P2VCIAHAMqguhBdx//FLTNxw4tvSEiznRvFs4QM8/hfQG07yy/V6HbVajUNIAmDwwlUE332Cbfwa+s1DeB8KwWy7gudvP+Ki/Sb27hPzy9QFAUjUjQAwDlh5dcvwOOSqowgGX6On9wxuuefBxsO27Tu4mTogkfkvQJdSHVGoNZHO/Qeivaf6EQqHIZMroTyshkR6EHKFQjC3SgCsvsJN9NPu+VL1dloP8HO9PWijFQGwCpGwgEp0sBEzu1vg29sCEBFkA+v8R7T6ol/92Z1dPFeoPQAAAABJRU5ErkJggg==',
page_white_flash=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHYSURBVDjLjZPLSxtRFMa1f0UXCl0VN66igg80kQZtsLiUWhe14MKFIFHbIEF8BNFFKYVkkT9GKFJooXTToq2gLkQT82oyjzuvO8nXe65mmIkRHfg2c+/3O+d8l9MBoIMkvi6hkNDAA3om9MTz+QAhy7JqnPO667poJ3GOdDr92Q/xAwbIrOs6GGOeFEVBtVpFoVCQkHw+j0wm40Ga5k4C0AXTNGHbNsxv32Hu7YNtp1Cr1VAsFiXAMAxQkWw2ewNpBZDZPjiA+XYebioJ9nIKqqqiVCrdGUlm0gpwzs5hzrwGX1uGMTMLtvrBG6VcLstOcrncPQDOYW3tgCffw0isg4uqnP6J8AhCnVAelUqlPYD/PYE59wZ67BXsL4fg/6ryYhNC82uaJkFtAdbHT+CJFbgbCagjYbDNlDev4zgyH4KQ7gA2n/fMUWWeiAtzBMrgWABAXciAhaibAKAYnXyaGx3/5cSXoIajsH/8hHP8B87llTSSqAMSmQMAfSL2VYtET5WRCLcW3oHt7Aaq+s1+eQAt/EJXh8MNe2kRSmwa/LoQeOsmpFUeQB0ag9I/jIve0G/n6Lhx3x60Ud3L4DbIPhEQo4PHmMVdTW6vD9BNkEesc1O0+t3/AXamvvzW7S+UAAAAAElFTkSuQmCC',
page_white_freehand=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIRSURBVDjLjZO9a1NhFMabozgIgqjgYkGwY7dKRScXBf8C/QfUycnBDpk6dHATkiGziEtFcBGkaZrmQ5M2sWlD0zbahgSS5t6ENLm5Nx837eM5r016k7boC8/ynvf5nXMeeMcAjIn43GJNsab/odssGvgcgKl2u31g2/Zhr9fDWeI6PB7POyfECZgWc7PZhGmaA9XrddRqNZTLZQUplUrwer0DSN/sEoA8aLVa6HQ6SDcaeJ7L4dH2FkoM0DRNASzLgjTx+Xx/IaMAMX/QddxMb4CSSdxY+4ly/QA6342upDIZBaSYfm09xeYE7m9u4v3+Php8J6tUKhU1SbFYPB/wbPc3KLGKh1sZWLYNzkWFJ3k0eC3Jo1qtDgNoJT7pcrtfzHPhCne+uBLHR11DgB9/5bsFlkBkf8MwFGgYkEzMS1cGgeIxPEinMZPbA0XCoHAIV6MRdLtdlY9ARMOAeOw1+f2rFPsB+h7F20IBc/k8JgS4HMQdvpdVBCCSaU5l4PJ6ZsV8mZU1m+rB49QaaCmAl5mMmkAk5iEAdyUe9QkFl3QZ91U2C5uLs3u7uBBYxCVWmDPoA/o6AUQjXxhwJOZ7HKDG9De/sqCAH7S4gJmdbTX+qE4A4VCeQssGff70rWBZR1J4urGO68Eg3DyNddztDB0OMjjO4S4HZErhvM/kNPNbQ/1eB2BcIP/xnfuSrz/+BzfCswvzr5lXAAAAAElFTkSuQmCC',
page_white_gear=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEkSURBVCjPbdE9S0IBGIbhxxobWxP8D8r5I60RLg0NNTS21VBRQwg1aA4VOAWBpBVCFhKUtkVJtPQx9GFFWh49x3P0bvAjjsWzXrzvcAtpREEZfQtoACEkpKBVdpouv7NYi3SJkAynWcXExKTCJ6+4PLPeIZJPhksdmzp1vilTwqVGlWhEgR6wsbGpU+OLt94rGfJ1gIOLi4OFSYV3Sjx5QXdtkiHFx//gjiwlTshyT5LV3T8gwy3HFLnhkCuWmB3qA0Uu2WGOZVIUmN/ru5CiwAsLNLCI8cg+i3hAggMeiNOgwQbXRJnwghoX5DkiTow0OcLJ8HAbtLpkkzwJCuTY4pQppgeFFLJNtxMrzSRFtlnhvDXO6Fk7ll8hb+wZxpChoPzoB6aiXIYcSLDWAAAAAElFTkSuQmCC',
page_white_get=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGWSURBVDjLlZNNSwJRFIa1Tf8hElq1qCDJxk22CdsErVvbLotQbGFkZBYWRqCghqv2tXHRokACw4IICvoHbRS/8HPGGbXe7hkcGTU/uvAwA3Pf555zhqsBoGFrkmFgGIcwxRijjIKmJTCIolhsNBrfzWYTf8G+IxgMBrolisBI4Wq1Cp7n25RKJRQKBaTTaVmSSqUQCoU6JLS0JKANtVoNkiTJT6JSqaBYLCKTycgCQRBAh0QiEVnSI6CwAgkoUC6Xkc1me1pqzUTbIWBttDfQO/WttJLL5eRKkslkf0E3aglVQvPI5/OjC9QSaodmQqJ/CYh6vS7PhSTEvwVUBQm8iUOcxl2jCSikQBW44064Hh1wPuzCEd0ZXVATBThjNhzE7Lj+DCPyEcDWjQUr3qUvzj4/PvQv+BInsNxuwHFnxdW7H4E3H2xRK0xHnMTt6+/Vgp9Bc1gPr8L/eg7fiwdrl2bozBOb6ruwyIbEs419L5P5Yhlnz8fwPLmwsDdHIYNaoCPJoKtscnNgJUNvn8XM9jSFdJT9BVHxpMfQmzHDAAAAAElFTkSuQmCC',
page_white_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH2SURBVDjLldPLSxtRFAbwm7ooUheuSn2iQqHtLkZTFGsXShA3LYUi2FJR8VWhuhC6iFAfiEZRE0jUUNDGNKVVxIJUSnGj4APFQhf9D0pCXs1z8tav9wyJjI9ovfAxA3PPb+4Z5jAAjK98HgWP8ooU8dygmlRYElBEIhFvPB4/SiQSuCj8OfR6ve4skgKUVBwMBiEIwkl8Ph88Hg/sdruI2Gw2GAyGUwgtGQG0IRwOIxqNildKIBCA1+uFw+EQgVAoBHqJ0WgUkXMAFadCABX4/X44nc5zLSW/iewUwNs42UD31HeqFZfLJZ7EarWmB85GitBJ6Hu43e7/B6RI76dqtC3I4fY4rwdQYrEYuswVGFxrRMPcPYQiwauBHssjvPlYgc7FcrTMy9G/+hxLBzr0LT+BSpuDrLyMx5cC3eaH+PpzBiuHerHw84EW2o0+mHbH0WlRoXz05tEDtSw7LdDK+6XiqR890Hzvxsh6OwbWmjH0rQNzW8N4+aEKd9+xRFrgxfv7+LKvhWVvEqadccxvj3HkNWa3htBuqUORmv3NfcvkUuBYCjybLUG9Lh+107dRNZGNVwuVmNkcQItZhTLNHWRVsgbpLJTxv0/ghWmHSTF2C02mGig1efj955dAAygFCgi5bJSL+1m4UJ2BzFL2NDn6BVT7D+X3feV2c5mYAAAAAElFTkSuQmCC',
page_white_h=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHtSURBVDjLjZNLS9xQFMe138C9A/0OynyBUjeFQjduROi2MMtCEalS0ToLEdQMdEShoKDWRymKigWxII7PhaB9aBFUJjHJpHlnnvbfe27NJcVIDfwIyT3nd885cOoA1BHsaWQ0MZL/4SHjgciLCJpKpZJVrVava7Ua4mDnkCRpKCqJCpKU7HkefN8X2LYN0zShqiqXKIqCTCYjJGFyPQkooFgsolwu8zfhui4sy4KmaVwQBAHokmw2+1cSClpSUmr12MP7LQunii8klOA4DnRdv9USn0koePRiJDW+aTGBjcOLgAewlnjfYSuFQoFXIsvybQF9jG2avIKFPQtzOyZmcyZMtywkVAnNwzCMeMG7jV+YyFmQ1g30L2kYWitAWtZFJdQOzYREsYLhzwZGGF+OHez/9PD2k4aeeYUHVyoVPheSELGCwRUdA+zG/VMPeycu3iyo6J5WxDxIQFA1QtCauUwPrOpIPh/vSC+qSC/qPHn3u4uu2Su8nsrzZKqAoOR/BO2j+Q+DTPC0/2CdSu79qOLVlIyXk3l0zsjomJYxv6ELQYgQPOk7a2jpOnmcaG57tvuD3fzNxc5XB9sEm0XuyMb5VcCriBI7A/bz9117EMO1ENxImtmAfDq4TzKLdfn2RgQJktxjnUNo9RN/AFmTwlP7TY1uAAAAAElFTkSuQmCC',
page_white_horizontal=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAC6SURBVCjPfdHhDoIgFAVg6jlq6x10vGdbrQfsR7bcskSTgE73gjAtirP5Qz/hsiNEWIuvpLUWhZCZbMQygEK31ll85rCNRPJnjQc6SosrzjgSUNiNRLLXGNATUmhQo6I3PU7YMwngSSBE4YbLeMwAmiUC3sP4Z0ekpjCbAeP/Mp60uBObAONjE+GR1RTYCbD+KL7XDMwTZvoDjL9XBC4PaGzHoNSdfWV3cUNDPVFZZbYsTiFWv8pOhb8BUJ5M7qO6PVQAAAAASUVORK5CYII=',
page_white_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH6SURBVDjLjZPPaxNBHMVT/wgRsVDPgpdIya3QitWDepAqWKgHQaSiPfij4CHtRSEXRUmwKYW2emkRoYgHUbQEkVLFSmoJXWtRUtlkk7WZ/TW7yyY+57t0wyY1moHHLuz3fWbesC8CICLWAaGoUPd/1CW0hzyBIjuAqOM4zPO8WrVaxd8kviOZTD5ohgSAbjKbpgnLsurSNA2VSgWKoviQQqGAVCrVAKHVQQAasG0bruv6T5JhGGCMoVQq+QDOOWiTdDrtQ3YByByIAGTQdR3lcnlXpJ076WgAiBj1AXqn3EEUVVX9k8iy3BrQrDBE1xh+ZhLYenkZlbWn7QHCEOXDFNTlEXjGMoqLF3Hn7L6RtgCkX6tzKGWG4LEnYGvjUJfi2Hh82n5x9eD5tgDym1G42/Nw5Dic4iRYNg6eH4c00cNaAujogZSlhzA3E+A/hmHnr4N/vwD+dQCfxmLGPwGmnIX8+gqKmUvQpSHwjVOwNwdhrQ/g49hhd+ZcV+sIbP05thbETvk0rG9nwKVjWEn14Mu9fuTuH8fN3r2jzZf4uyH3q2GReRb6agw814/PE33IvZ2HazC4plYjT7gLR8TfZwljvUzFxWtwlbuwpJNYedQH6d2zAF4Ts5Q9GgZ0EiRc3blbh6azU0e33ydihRsn9t9uqjVVv5O8fwBNdbGhhwEriAAAAABJRU5ErkJggg==',
page_white_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIvSURBVDjLjZPLaxNRFIeriP+AO7Gg7nRXqo1ogoKCK0Fbig8QuxKhPop04SYLNYqlKpEmQlDBRRcFFWlBqqJYLVpbq6ktaRo0aWmamUxmJpN5ZvKoP++9mmlqWuzAt7jc+X2Hcy6nDkAdhXxbCI2Epv+wlbDeyVUJGm3bzpVKpcVyuYyVIPcIBAL3qiXVgiYaNgwDpmk6qKoKRVEgCAKT8DyPYDDoSCrhdYHrO9qzkdOQvp+E+O04hC+tED63gBs+QiDnhQgTWJYFWiQUCv2RUEH/g4YNXwdcT/VEJ6xkF8zEDRixq1CnriD94SikH08gikJNS2wmVLDwybONH3GbNt8DY+YMrDk/tGkvhOFmKPE+pxVJkpDJZMBx3JJAHN+/MTPq8amxdtj8fWjhwzB+diH5ag9y8V6QubDhUYmmaWwesiwvCYRRtyv9ca9oc37kk3egTbbBiPowP+iGOHGT0A1h7BrS43ehiXHous5EjoCEx3IzF6FMnYMcPgs95iOCW1DDXqTfnEBqsBnRR9shTvYibyhsiBRHwL13dabe7r797uHOx3Kkm1T2IDfhhTRyAfMDh5Aauox8Ns5aKRQKDNrSsiHSZ6SHoq1i9nkDuNfHkHi2D9loHwtSisUig4ZXFaSG2pB8cZBUPY+ila0JV1Mj8F/a3DHbfwDq3Mtlb12R/EuNoKN10ylLmv612h6swKIj+CvZRQZk0ou1hMm/OtveKkE9laxhnSvQ1a//DV9axd5NSHlCAAAAAElFTkSuQmCC',
page_white_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH4SURBVDjLlZM7i2JBEIUd4/kJu7D+g4FZxjXSRFHQwMBsMxFFE8VEMVEDhRURRREDTY18pAYKirHJBAui0YJv8fp+jme7mrmDjvtsONzuqq7vdp2mJQAkbHxgemR6+os+MUmpRpTkFfC42+2E4/H4cjqd8CuxPJLJZPw9RAQ8UfFqtcJ6vX7TfD7HbDbDcDjkkH6/j1QqdQWhcUcA2rDdbrHf7/mXtFwuIQgCRqMRB2w2G9BPMpkMh9wAqFgUAahgsVhgPB7ftPTqyd0VgLXxtoHm1LfYymQy4Sfp9Xq/B7zXJYROQn5Mp9N/B1xCqB3yhED/BSAdDgfuC0FIfwSQy5VKBdFoFLlcDpFIBJ1Oh3sgxuhrMpkSCoXi/uYas9ks4vE4BoMBut0u6vU6X7OrQ7vd5rFms8ljer3+2xWgVqvB5/Oh9fyMcDgMg8EAi8WCRCIBr9fL5xSjHEHcbrdwBYjFYgiFQhxAzmu1WlSrVbRaLTQaDRQKBR6jHAGCweD2EnDO5/NwOp0cEAgEoFarodFoYDQauWhOMb/ffyaA1WqdiW/hM3N5zXp8sdls9GhQLpdRLBaRTqfhcrng8Xj4nGKlUunMzF2wdr6LgI8EkUqlX2Qy2Vd2zJ7ZbN7Y7fa9w+E4qFQqv1wud7D1mmKU0+l0P5RK5cNPtqSTQgo+48AAAAAASUVORK5CYII=',
page_white_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG8SURBVDjLjZNJS8NAGIarv0PBuzcV/4MHwYMHL/4CPQsexYvoSW2RioiguJ9cigtFKhpxqVtBXGqtVRO62TbJNFMb+zpfNKJVMQMPCWTeZ+YdMi4ALjGqBPWCxn+oEVRSxsb1IajnnGeLxeKraZr4DfEdbrd7sFxiCxoprOs6GGOf5HI5ZDIZxONxS6IoCjwezzcJjQoS0ATDMFAoFKwnoWkastksEomEJcjn86BFvF6vJfkhoLANCSigqiqSyeSPSh9nUvFNIGp8TqB36m1XSaVS1k5kWf5bUM5XCe2EziOdTjsXmGYRgVAMi9I1JrbuMbPzBF/wAS8F5kywfX6PlWAcNwrDXYpj/1bF2mkS/pOYM8G8JOPiUcNBNA8pwrArCMkcs9vR/wXUf9wfRTjBId3q2Anr8F9qCMY4pgKPzgSzovPFE0Pg+j1MHD1wjPqunFUIhBTsh1Uci9Be1MChWH35TIN3cgl97XU95YJSueBZ4zi8ecaCOIu5XRljm3cYmfQhtDYGabidTXfWttl3oUH8fUyE/rxMNpGD1dLReEcpsj4EX28TswXVJHFwnS26mqu6NwdajY3+FrwBN5GpoomTEloAAAAASUVORK5CYII=',
page_white_medal=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJUSURBVDjLhZPPS1RRHMU/997nzDihTx0zFxrRxkqyQKNaBBGCWzetXUhYCUKLfrhtEdFGCqEWQf9AuK5FQouiokWrFoYuBioZ480o47x57/5q0cw4ReUXzvKce8/3fI/w3gMghBgCDgGK/08JKHrvHQDeexoi4/V6vaK1tsYY/zfU63W/vLz8EDgCSO894sKtF2Z4IKcS5XHG4qzHGEdeKDaKEasPpnDOkaYphUKBra0tVlZWHs3Pzy8BxSCXzzJ+cpC1qEaaGoy2OOMYKXSytl5CSon3HiEEAGEYMjMzsxAEAXNzc0vSGmsq2zFHuzpIYk1SN4z0dVKuxOAsSimklC2BTCZDLpdjdnZ2ARiQ1pibqx++plE55nghz4n+PFF5l1dvNrhx+TRSSpRSZDIZKpUKURRRKpVaGxX23VD4fnPs+bGe4uQBVcVZTznO8M1PcerSXVBdWGt/g3OOMAwJguBs4CyLZ0bHJkXfHZxLkGabXr1N1+ZL4s/3yY/dQynVjBohRDM1AKTTtWuifxqrK9i4iN1dx8YbyL4Jdj4+aRGbVpRSBEHQEgis0d3Sg7dVsDW8reF1BYQiKe/seW28rJT64wfG4X2Kt/Evsqvh7S7YOjrZO7RmlO1kAGl0uuP1DjjdEInBW7yponV361pbBCmRUrYJVPXj+MszhOxAZg8ic4MI1cOPT2/Jj179bXntaFn7/pQwiVisRVx3KV06BeN6Cc9d4fD0bYJ8+K9SuSAIzou2Nk4kSfJaKZUD5D6NdNbaWjabvdguMNyo837k5lig9BMTlFjmy9KhMwAAAABJRU5ErkJggg==',
page_white_office=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKdSURBVDjLjZP9S1NRGMdH9GckFFT6Q5ZhWVGZEGhFSFmaoiCB72VJxYwowSAhKUqY0gZplOR+MQqCwtQSC19wXcXFfjF1zXa3u817d+927t2Lfjv3xF6Eoi58fjrn+dzn+XIeAwCDDv22UHIpef9gK2VTsi5NkKtpmhSLxdbi8Tj+BD2HyWTqSpcYbGdLnbMFeTFX+aF1ofkIxKYs+K8fhRLwIRwOIxgMQhRFeL1eJuF5Ht3d3UmJYTJzO5bqjk+bKvONisMGiRuH0n4YwR8LUFUViqJAkiQIgsAEhBCEQiGYzebfkm/HsrBy/1ydPp9+IRKJgAych+xeRscbB1oH7ajumUSblaMjRDeMxDLxlGdj4VZ+WUIQi6iIDJZC8brQ/HwO3KIfjmUvLlmmsLjsZp243e6UQLqYhaU7Jw8mBDqipREabbP91TyUsMrCu/bsKwT/KssjEAikBL7GvevEeCBOHhbFyJNiyL0tUEY6ockSjNZ5GF/acLWfQ1PvHERJZpnIspwSKN8tm93jhZmXq3eUaSEX4lGCqOpjF0JklYYmg6gifNISwqrCQtRJCjThcY06tQ2e8WLinSiBMFUFYaISGi3sG6uBebQKlk9V6BktxQ3rCRayPlJK4DTXyJ+zYe6ob0tksMo1IEo7eTJ6AW+5e6CPCxP292i2FLLiDQKy2Fcf+JiNm42nGxKCsL0YROFhGi6BdeY2gqEARmYHUPuggIqjjKRgZaar2vthN770V9idti74HI9AJneByDz6xzrR8qIIrU/P4IrpFGrvFrFudJIC7nX7Ts/QfoF/lwNhKAf+4T0QpytoBgr7k8fvBu/7CRfvREDypz+kNSZIW6Z9NKCwfvC3ZUovpncVtr1pggxd8h/rnEBf/Yxfiwq9OE8N8XAAAAAASUVORK5CYII=',
page_white_paint=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJCSURBVDjLjZPNTxNBGIerBy/eOAgmYoz/gCYoJ/XgxZsxnrygFw8eJJGDiQc0MZEElFBL2igkoBBEFLUWaYNQiB+gKd1WbVKUCgVp2O3H2N3tfvYDf+5s7KbGNjLJc5r5PTPvm3ltNpttn0GTQfN/OGCwE4CtErqadF0XisXiVqlUQjWMfTidTkc1CV3NNCzLMhRFsRBFETzPI5VKmRKO4+ByuUyJt6dub3D0qG+ut8FuCugBTdOQz+ehqBoERYMkSRAEAel02hSoqgp6ycO+mwPR2asRMTGCWcdBxRLQcELUEE6qWGRlsKKCXC6HTCZjlaKKCfxg7NDIBD6PH8fL63sclsAoA1GiY35TxfuEjDAnW6UQQsBuRLH6sRN53guOaYHnRn3/+LX6XZaAEud1TK9LeL2WQ4hTzOZRCeG+Ih7ogp59hdSXC3jSvp8ZutJQZzWxLFjJavAs83B/yyIp5c1XiSSGtUC3GSZLF/Hm3gmcOrT7rJHb8Y/AHxcwFsnAvUTwkyQRDU9hefq88ewXEFcuG007jPTaJ/z5F38LYkTFcDiJwUUWUwEGfu8YfO77mBk4g5jvJIKPjmGVmTAvqioIbebQ92EDdl8Q3UPP4Z9fAJsIg1l4Cs/d04jO9Zs9qSnISLoRDqFjeBK93ghuPQ7iXMdbtPVMIsWuo1AomNQUUNpuP0Br1wgudT5DS/soWu/M4B3z3WxmmVqCX7XmoApbNFM5C0eMX6jQje2EjbMSHcBKQSOVbGOcy9DRbywLfgOaoblOxI0zHQAAAABJRU5ErkJggg==',
page_white_paintbrush=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH8SURBVDjLjZPLaxNRFIfHLrpx10WbghXxH7DQx6p14cadiCs31Y2LLizYhdBFWyhYaFUaUxLUQFCxL61E+0gofWGLRUqGqoWp2JpGG8g4ybTJJJm86897Ls4QJIm98DED9/6+mXNmjiAIwhlGE6P1P5xjVAEQiqHVlMlkYvl8/rhQKKAUbB92u91WSkKrlcLJZBK6rptomoZoNApFUbhElmU4HA4u8YzU1PsmWryroxYrF9CBdDqNbDbLr0QikUAsFkM4HOaCVCoFesjzpwMuaeXuthYcw4rtvG4KKGxAAgrE43FEIhGzlJQWxE/RirQ6i8/T7XjXV2szBawM8yDdU91GKaqqInQgwf9xCNmoB7LYgZn+Oud0T121KfiXYokqf8X+5jAyR3NQvtzEq96z4os7lhqzieW6TxJN3UVg8yEPqzu38P7xRVy+cPoay52qKDhUf0HaWsC3xRvstd3Qvt9mTWtEOPAJf/+L8oKAfwfLnil43z7Bkusqdr2X4Btvg1+c5fsVBZJ/H9aXbix/2EAouAVx4zVmHl2BtOrkPako2DsIwulexKhnG/cmfbg+uIbukXkooR/I5XKcioLu+8/QNTyGzqE36OidQNeDJayLe7yZBuUEv8t9iRIcU6Z4FprZ36fTxknC7GyCBrBY0ECSE4yzAY1+gyH4Ay9cw2Ifwv9mAAAAAElFTkSuQmCC',
page_white_paste=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH+SURBVDjLdZI7aFRREIa/k3u5cclGwiXEF9GInSCIoF0qa1/BynJdyBYhtaRJSrsFK4XtBCGFQRALm4BYKQg2ti5JQEh2hay66ObMmbG49+4jSwYOc4rzfzP/nHFmBsCL6uWrwDfGo7PcaM5wQrjnjxduABvAnXJ6hkdr63TbnwGYmr3Fy/Un/O3+AngFbC03mq+HATGwdfv+g4ul8jT++1fMjNDrAmBmLF6bZ+LcFczs4fs3b+8CY4BLpfJpjn4fICo0N+tgBhht+4Ka4ff3ODV7ITFInHPOCt85ADRgIkzMnEWDYKb8aHf52fmHAbp/iNsTdlo9gJvOuQNg18w0BjBVVAT1PdT3wIxW65B7a++I4qTf7hKwIfKp0Wg8W1lZqTvndnOAoMFjprn37ERxgvceEUFE8N6TpimVSmU1jmNqtVp9IgMEVDwmHg0BDYppBnPOEUURURThnAMgSRKq1eoqMBcXAMRjQTFRVA2VDBBFUT9PTk7S6XQIIZCm6WCIpllVFUUlYGpYCCMdFHfnHCIy8o3ZEH3ARDEJqIKKDbYthxQWitwHZNUzgOYWig6GIcdBgz0IhnrNZ1BY0PG9z4WFpQLw0R8dLU6dn8+EqphCybf6j4YWb6R6AXj6YXt7GrjeFwA7rR5LJ4gAQgh/gOCG6cfaXQDmODkCcPAfs2Qwc59hLO4AAAAASUVORK5CYII=',
page_white_php=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGsSURBVDjLjZNLSwJRFICtFv2AgggS2vQLDFvVpn0Pi4iItm1KItvWJqW1pYsRemyyNILARbZpm0WtrJ0kbmbUlHmr4+t0z60Z7oSSAx935txzvrlPBwA4EPKMEVwE9z+ME/qtOkbgqtVqUqPRaDWbTegE6YdQKBRkJazAjcWapoGu6xayLIMoilAoFKhEEAQIh8OWxCzuQwEmVKtVMAyDtoiqqiBJEhSLRSqoVCqAP+E47keCAvfU5sDQ8MRs/OYNtr1x2PXdwuJShLLljcFlNAW5HA9khLYp0TUhSYMLHm7PLEDS7zyw3ybRqyfg+TyBtwl2sDP1nKWFiUSazFex3tk45sXjL1Aul20CGTs+syVY37igBbwg03eMsfH9gwSsrZ+Doig2QZsdNiZmMkVrKmwc18azHKELyQrOMEHTDJp8HXu1hostG8dY8PiRngdWMEq467ZwbDxwlIR8XrQLcBvn5k9Gpmd8fn/gHlZWT20C/D4k8eTDB3yVFKjX6xSbgD1If8G970Q3QbvbPehAyxL8SibJEdaxo5dikqvS28sInCjp4Tqb4NV3fgPirZ4pD4KS4wAAAABJRU5ErkJggg==',
page_white_picture=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIcSURBVDjLjZO/T1NhFIafc+/trdRaYk1KUEEWjXHRaCSik+E/cDHGzYXB2YHRhMRFY1SYmRgYHZ3VxIRFDYMraMC2hrbQXm7v9+M4UGobiOEk7/adN+9zvnNEVQEQkYvAGBDy/6oBm6rqAVBVeia30jRtGmOctVaPU5qmuri4+AaYAgJVHTKYNsa4drutnU6nr1arpY1GQ6vVqlprdXt7W5eWlvomMv/uw6tSofB4p+NOF0biYtc48tEAhXiuTZzh/s1xyuUyWZbhvWdlZeXt3Nzca14sf6zW6nXf7uzrcfq9s6sLy5+1Xq8fQQKmo1ZCvlAoyo+tXT5tPGO09IckM2zWznH3/AJ3rl5ACInjmGazifceay2VSgWASISSBaz3FIs1RnJlPF18vEG1keDVk1lLFEWICM45wvAfYqTKriqje0lGI01x2qFtuuwkKQ26oEKcCwnDEBFBRA6HfmBw8JWwl3o2ti7j8+u0TUKzcYkrY/n+wyAIEJEjSxEglLyH5r7j+tg8T1oVZr8GzE69JIoiFMiM7zeHYUgQBAMJVBGU77+eYoxhLcvIxnNk6w8xxvDo3hqH+yIieO+HEkQB/qe6bPL5g/cckCkDiBhjOJULhlCGDJIkXX2z+m3GeW4UCnExyxxxHIIOLNLk2WP5AaQXTYDb1tovgHCy8lEUzQS9g1LAO+f2AX+SZudcAjgZOOeJ3jkHJ0zggNpfYEZnU63wHeoAAAAASUVORK5CYII=',
page_white_powerpoint=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHeSURBVDjLjZO/i1NBEMc/u+/lBYxiLkgU7vRstLEUDyxtxV68ykIMWlocaGHrD1DxSAqxNf4t115jo6DYhCRCEsk733s7u2PxkuiRoBkYdmGZz3xndsaoKgDGmC3gLBDxbxsA31U1AKCqzCBXsywbO+e8iOgqz7JM2+32W+AiYFX1GGDHOeen06mmabrwyWSio9FI+/2+ioj2ej3tdDoLiJm+bimAhgBeUe9RmbkrT5wgT97RaDQoioIQAt1ud7/Var1h+uq+/s9+PLilw+FwqSRgJ1YpexHSKenHF4DFf/uC3b7CydsPsafraO5IkoTxeEwIARGh2WwCYNUJAOmHZ5y4eY/a7h4hPcIdHvDz/fMSnjviOCZJEiqVCtVqdfEl8RygHkz9DLZWQzOHisd9OizfckcURRhjMMbMm14CQlEC/NfPjPd2CSJQCEEEDWYBsNZijFkaCqu5Ky+blwl5geaOUDg0c8TnNssSClkER1GEtXYZcOruI6ILl1AJqATirW02Hr8sFThBVZfklyXMFdQbbDzdXzm78z4Bx7KXTcwdgzs3yizuzxAhHvVh4avqBzAzaQa4JiIHgGE9C3EcX7ezhVIgeO9/AWGdYO/9EeDNX+t8frbOdk0FHhj8BvUsfP0TH5dOAAAAAElFTkSuQmCC',
page_white_put=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGdSURBVDjLlZNLSwJhFIa1Rb8iIWhRQUlluuoftDEtC5TKSgINily1CmoT0kJBqwlSaBGBLVxItGgZQQQVFe3bKN7wOjqO2tucwRGvqAMPMzDf+8w5ZzgyADLhGhJQCWi6MCwwQBkJWVWg4jguVSqVKuVyGe0Q3sPtdruaJZJAQ+FcLgeWZWuk02kkk0lEIhFREg6H4fF4GiR0yUlABwqFAorFongnstksUqkUotGoKMjn86CPMAwjSloEFJYgAQUymQxisVhLS9WZyBsEQhu1A/RMfUutxONxsZJQKNRZ0Ey9hCqheSQSid4F9RJqh2ZCor4EBM/z4lxIQvQtoCp2HtexfW+CObAM062uu4BCElSBJWjEzc8Vrr8Y6L3zvQsoTKz6F+H7PAPz7oLRp8eodmSjp7/geDqG2b8Me9CK8zcnXK8O7AWsmDtUF9UHUw/1gr+2O8BzsPm3YLvbhPPlBI7nI6xc6jC9P/Gr3B0flHZhVpgyKwQ6LpPFtwaTdwmGCy0MpwsVWsD6ZVKQpNs6z9iV35PWsY/q6iso+w9crJoc0rRwaAAAAABJRU5ErkJggg==',
page_white_ruby=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIESURBVDjLjZNPTxNBGIexid9CEr8DBr8CHEiMVoomJiQkxBIM3dgIiaIESJTGGpVtyXIzHhoM4SIe9KAnEi4clQtJEczWFrbdP93d7s7u/JwZ7XYJBdnkyRxmfs/MvO9OD4AeDvuuMPoY/f/hKiMR5WKCvlarpRNCwiAI0A02D1mW38QlcUE/DzebTdi2HWEYBhqNBqrVqpBUKhUUCoVI0g5f4gK+wHVdeJ4nRo5lWdB1HbVaTQgcxwHfRFGUvxIuCKYfzmqZyZ2wKIO8fQ3/1Uv4Sy/QWliAO/sU9qMZmFMS3HfvT1xJ1ITOZJ9RpQi6+RH0y2fQb19BP23CVhRo+TysXA71+XkcMIk6fAfHK6tQVfWEoESXngNra0C5DHZJYGMDZiaD35IEi41qOo3vc3MoJ1Ooj92HpmkdQZiVEsHUAzl88hjY3gYIAdbXYQ0MoDo4CH1kBHssvH8jCf3eGKzDXzBNsyNoF/HH7WSJZLPA7i6wtQVnaAhmKoXjxUX8vDkMY3Qcnm6IInJOCS4nEte9QhF+RhInIRMTcFhYvZWCcXcUPmsl7w6H/w+nBFEb5SLc8TTo8jLq7M4m25mHfd8X8PC5AtHrXB5NdmwRrnfCcc4VCEnpA8jREasp6cpZAnrWO+hCGAn+Sa6xAtl84iJhttYSrzcm6OWSCzznNvzp9/4BgwKvG3Zq1eoAAAAASUVORK5CYII=',
page_white_stack=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADPSURBVCjPdZFNCsIwEEZHPYdSz1DaHsMzuPM6RRcewSO4caPQ3sBDKCK02p+08DmZtGkKlQ+GhHm8MBmiFQUU2ng0B7khClTdQqdBiX1Ma1qMgbDlxh0XnJHiit2JNq5HgAo3KEx7BFAM/PMI0CDB2KNvh1gjHZBi8OR448GnAkeNDEDvKZDh2Xl4cBcwtcKXkZdYLJBYwCCFPDRpMEjNyKcDPC4RbXuPiWKkNABPOuNhItegz0pGFkD+y3p0s48DDB43dU7+eLWes3gdn5Y/LD9Y6skuWXcAAAAASUVORK5CYII=',
page_white_star=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHHSURBVDjLjZPNSxtRFMX9WPk/VHBpoZRiJEtx0S5K20XFTVEKCq3g0hYXLhu6SjeFZJFNcSluxH03IiJocYINpgkYxEXSfDWTzMd7b2bs8d3XzviGRHTgMIt55zfnnpk7BGCIJK8HUgmp5B2akBqJfBogwTk3fd+/CoIAgySfI5PJfNUhOiBJZtu24ThOpG63i06ng3q9riC1Wg3ZbDaChOZhAtABxhiEEOpOsiwLpmmi0WgogOu6oJfkcrl/EB3gd/Pgf44VIISQodfrodls9o2kOtEBrLQGdrYSHZAjqbnDUVqtlkpSrVZvALz0bswuLMwaO0+/8eJ78NM34JU0vPYBAs+OQSgJ9dFut3XAck78WgUvf4B3kYY4/wR28hr2/hTYz/VYEhqHOiHQDaC4OMpP53Z5+aMEfJGAFNz8PFxjCYEvonE8z1O9EIQU68A8fvGY5V9CVFIQpQ3Ye4/gmYVYYZQiLJfSxADWj2fPmfEKzuETWPsPZfwExO/vyhSKEpDI3A84mnlrbE1uBrwFX4oVP4NfbveZdQ38jIN+Xz2FrtsAf2/bgwG6igD/IdOyIIce3Mcsz1pqezXAOEHusc6haPXHrwEWI98D8WYq8gAAAABJRU5ErkJggg==',
page_white_swoosh=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIMSURBVDjLjZO9apVBEIaf2f2iRxMJUfEHFEStBCtRC0sLSzsbEcFCrQQVL0AvQAgkd2FhZyOkNRAsrERQRMFwonDM/35nd/a1OCfHRKI4MEwx7LMvM++YJADM7ARwFIj8O5aAL5IqAJIYQi6klH7mnL2Uot0ypaSZmZlp4BQQJO0AXMo5+9ramtbX10e5vLysXq+nbrerUooWFxc1Ozs7gjRD+TasIcZICIFaK7nCSomkYlAinQydiUlu3rr9IDZjdv/e3ecmaQtwsZQyv9F3Pq8ZH1eNzSzGGzEenA4txw8eYH+ETgNjASb2Npeb7dNZ+A5vfzTkKvYE6EQIDgEjhDGWeqvsC06jwvEjhwEYAR6+/PBo7pshREDkAKVCFQwQYBGMQCdGNot+A57O5ye56noqQgwa0USJRqngMg7vNcZiZLNAJbDSH3zcPFuoU1n2uHWFKDEAi2hw/iCcnTTOTRm1DjbmLlYzpKHypnVdzc5Uv4IJXKIKbpwxrhyz0XzMDEnEGJkM4lAz6DV953TfZX2HiihVXDsZuHLMtvwxMpyZUWvdYcsmuazvCn2XXIPBnRjf6d3toBDCjl5oXbQOrWOtQ3Lx6msdyd6qf+Y2AL3WZcnFpotUxJtuZfqd//XxEFABQip6nQYqLLlIDq2LF5+cO3OZ9z3tdpHV3TcAt23nfHJ4zoH/CweWfgFQJVPOGSHTggAAAABJRU5ErkJggg==',
page_white_text=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADoSURBVBgZBcExblNBGAbA2ceegTRBuIKOgiihSZNTcC5LUHAihNJR0kGKCDcYJY6D3/77MdOinTvzAgCw8ysThIvn/VojIyMjIyPP+bS1sUQIV2s95pBDDvmbP/mdkft83tpYguZq5Jh/OeaYh+yzy8hTHvNlaxNNczm+la9OTlar1UdA/+C2A4trRCnD3jS8BB1obq2Gk6GU6QbQAS4BUaYSQAf4bhhKKTFdAzrAOwAxEUAH+KEM01SY3gM6wBsEAQB0gJ+maZoC3gI6iPYaAIBJsiRmHU0AALOeFC3aK2cWAACUXe7+AwO0lc9eTHYTAAAAAElFTkSuQmCC',
page_white_text_width=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADNSURBVCjPhZBNCsIwFIQnWvAGunHhSrBbQT2Op3EvCJ7JCxTcuhB0YytaW9N0fElbbIs/GZI8eF+GzFPE79X504dXXEq1GyytFYYYoPvh6RkH5raYJqE2GZtKuFljhI61mWlz55URY95FES88CXIsEQEs/xCl7rwx5FmAWKqtIEqAncEe2gWiyOCJRZVh7hVZ/erv0rYKkSNDv4qpELzjSTt3e1yfg1+fgAMMWAeCxghyB02+ORRf5W8HC/mfHdi61XJ162WoS7utXc30BXX/jFnkwiD5AAAAAElFTkSuQmCC',
page_white_tux=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIuSURBVDjLjZNPiFJRFManVo24jYISClqli3KGp0VY7mSCBqKoIHIIahFtStwUCPVGIiloRJmkqQERiqBFIZGtcgrHrCGaFo+hCQlCU9/T558n/v+699Z7PBmjDnzcxbnnd8/53jtjAMaoSOwkmiDi/qFdRJu1Oh1gotVqyd1ut9/r9TBKJI9QKDSnh+gBXKfT6cfjcbhcLvj9flQqFVSrVXYWCgUGyefzCIfDGkQt3kQBklTGvv022A84yWlFJpNBvV6HLMsoFosM0Gw20Wg0EIlEfkP0AFEUEb53EYnnbpw5MYV0Os0KarUaSqXShpGYJ3pAWfyJ3IcjKH5y4NIpK5aX37O5FUVho9AHaCe5XG40IJlcwv1gAMLnFSw8fASfzwfiiwahnVA/JEnaCOA47mw0GkWvDxbZbBZmsxk8z2sQOg71hIKGAB6PZ9xms60KggA16AWv1wuDwcBgFNJutxmEaghwbPr4Ubd7hhUOBgMNkkgkYDQakUqlQP4PBqCi3QwBzp+bPulwHEaHXKIJNW4H7mDLuAHr699YB+ooQ4DCu8u7f7yaeum0b8ObpbRW/H31KSatFph2bCfGiRpAlQZYix16lnuyF8Gre/BgYRFKkzjekJGcd+L66a14ccuM8pebbAS9NMDHxzMX1hYt+ZV5S+3aFTcCd+cwO8sjduMg3gat+BqzQ3jNj9qNvubBn085SQxSaOJvy6QvJnfrbHt1ABOF/Mc6q6Krb/oFGtGkE2IcdecAAAAASUVORK5CYII=',
page_white_vector=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIWSURBVDjLhZNPbxJRFMWhRrYu3NrExIUbdzWte6M7d34Eo2Hjxm8gwZUxIYEARUKAWgwbV0BpxAW11bpQFrCoCVEMDplhQMow782/enx3WsiU0jrJ2bz7zu+9e95cHwAfSXzXhFaEVv+j60JLM58HsGIYxsi27SPHcbBIoo5oNBrxQryAVTJPJhPouu6q0+mgVquh0WhAlmUX0uv1EIvFZpCp2U8A2sA5h2maYIyhUChA0zTU63UoiuICaJ0OSSaTx5B5AJnpqqVSCbmNTWxVt9FsNtHv98+05GYyD7AsC5VKBZvFd/j2k6Etc6gjHfLgELKiujeRJGkxQGSAYDCIx8+eI/ORIb3Lkf0sWvmio9aaoC2NoQ7+QFUHCwFr5XIZ8bfvhZFhq2XgU9tEb2Tj99DCgcTx9YeOg64GZTCGPQdYEnpaLBbxZl9HfIejo1rg5nGvti3CMyxouonhIYM8ZG7NBWSz2YepVKobiUR+UXjrwry+wzBm9qnAqD03YHohbsASUP+ly2u+XC7XzmQyt9LpdJc2xuscr0ULU9NUFC6JDiFRCy4gn88/EWqFw+EEmfL7HK8+8FOAqdmrWYjC7E8kElcCgcAdWmx2LbzY5mCmc+YWXp33H/w1LQehKhPPZuK8mTjR0QxwArktQtKpsLHHEarwC81ir+ZOrwewTBCiXr157/7d0PfqjQcvH10w1jT6y/8A/nHJHcAgm2AAAAAASUVORK5CYII=',
page_white_visualstudio=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJQSURBVDjLjZNvSBNxGMeX9O+FOAkaLbehozdGRGiMQqTIlEqJMIig3oxl0YxcgYt6FUZRryLYwpFWCr2wXgjBIMJMYhFjgZSiEXOg5c5N593udne7u+2+3V3tT22SBx/uxe/5fu7uuefRAdCpKJdJoVHB9h9qFSryuSJBYzqdpiRJymYyGZRDOYfH43lULCkW2NRwKpUCy7J5kskkSJJELBbTJARBwOv15iW58AZVoBbwPA9BELS7CsMwoCgK8XhcE3AcB/UhPp/vtyQnGBi03pYXjyAbPQuRD2sSbmUFVN9NLJ5ux9DryZJP0nqiChzjl48Oh9oYRPTAXBVksgnS0hRWu7uxXG/EfL0ZZ9yjGHgb1t4kGo0WBO6AvcUVsFP9oTZZjlQCP7ZA/r4JpHM3lup2Im6pRsRai2PX/GjoDWEk8BWJRKIg6P147mfP+CW63d16RUyOQP5SA6rLAsKyA0TNNizvM4D9/A4Tk2Ec7nuPE0+vgqbpgqBnzLl6vv8N3+x4eEsS0mAvHAJhMoAw6kHUVUF4rkeWHAKXZtA15kDL6C6tkXmBffiZs/P+NE7dC4pBhwsJY6USVjBtBO/bCswrbfq2GS+Ce9DwyooHoRvaPPzVxI67IVfHnQA+2JqQMFQgur0anP8J5IVmYEopmdbh5YQO1wMu0BxdKlB/44GLg48/HT8J8uBesH6/ViDxC5DnWiHPWjAz0wleYCGKokaJIDdI/6JMZ1nWEshr7UEZsnnBH8l+ZfpY9WA9YaWW0ba3SGBWJetY5xzq6pt/AY6/mKmzshF5AAAAAElFTkSuQmCC',
page_white_width=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADHSURBVCjPhZFBDoIwEEV/peLWlW5ceAcTvY6ncW9i4pm8hgsT2QgGEFLKdygQkKD2Z9pJ5nUyv1XE7zX5U4euD6WGBTatFVZYwhu5GuDKsko2WWhswU9lPB2xxqRqszU24ZMRUyaiiA/eBbk1iAAV/xLlbo8ZMhAglewsiBLgYmUI4wwRJSxyzFsPO916ndazu/ARClhg0drsPKrGkA/bZHrorkKUE8cBuKI3fMkhAkH4/S+IbjI9Vux/jNof4lmBvowL43Lmb/8gdgK2+FpkAAAAAElFTkSuQmCC',
page_white_word=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIdSURBVDjLjZO7a5RREMV/9/F9yaLBzQY3CC7EpBGxU2O0EBG0sxHBUitTWYitYCsiiJL0NvlfgoWSRpGA4IMsm43ZXchmv8e9MxZZN1GD5MCBW8yce4aZY1QVAGPMaWAacPwfm8A3VRUAVJWhyIUsy7plWcYQgh7GLMt0aWnpNTADWFX9Q2C+LMu4s7Oj/X5/xF6vp51OR1utloYQtNls6vLy8kjE3Huz9qPIQjcUg/GZenVOokIEiSBBCKUSQ+TFwwa1Wo2iKBARVlZW3iwuLr7izssPnwZ50DLIoWz9zPT+s/fabrf/GQmY97GIIXGWp28/08si5+oV1jcGTCSO6nHH2pddYqmkaUq320VECCFQr9cBsBIVBbJcSdXQmK7Q6Qsnq54sj2gBplS896RpSpIkjI2NjVZitdh7jAOSK6trXcpC2GjlfP1esHD+GDYozjm893jvSZJkXyAWe+ssc6W5G9naLqkaw/pGxBrl1tVpJCrWWpxzI6GRgOQKCv2BYHPl5uUatROeSsVy7eIkU9UUiYoxBgDnHNbagw4U6yAWwpmphNvXT6HAhAZuLNRx1iDDWzHG/L6ZEbyJVLa2c54/PgsKgyzw5MHcqKC9nROK/aaDvwN4KYS7j959DHk2PtuYnBUBFUEVVBQRgzX7I/wNM7RmgEshhFXAcDSI9/6KHQZKAYkxDgA5SnOMcReI5kCcG8M42yM6iMDmL261eaOOnqrOAAAAAElFTkSuQmCC',
page_white_world=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJwSURBVDjLjZPdT1JhHMetvyO3/gfLKy+68bLV2qIAq7UyG6IrdRPL5hs2U5FR0MJIAqZlh7BVViI1kkyyiPkCyUtztQYTYbwJE8W+Pc8pjofK1dk+OxfP+X3O83srAVBCIc8eQhmh/B/sJezm4niCsvX19cTm5uZWPp/H3yDnUKvVKr6ELyinwWtra8hkMhzJZBLxeBwrKyusJBwOQ6PRcJJC8K4DJ/dXM04DOswNqNOLybsRo9N6LCy7kUgkEIlEWEE2mwX9iVar/Smhglqd8IREKwya3qhg809gPLgI/XsrOp/IcXVMhqnFSayurv6RElsT6ZCoov5u1fzUVwvcKRdefVuEKRCA3OFHv2MOxtlBdFuaMf/ZhWg0yt4kFAoVCZS3Hd1gkpOwRt9h0LOES3YvamzPcdF7A6rlPrSbpbhP0kmlUmw9YrHYtoDku2T6pEZ/2ICXEQ8kTz+g2TkNceAKKv2nIHachn6qBx1MI5t/Op1mRXzBd31AiRafBp1vZyEcceGCzQ6p24yjEzocGT6LUacS0iExcrkcK6Fsp6AXLRnmFOjyPMIZixPHmAAOGxZQec2OQyo7zpm6cNN6GZ2kK1RAofPAr8GA4oUMrdNNkIw/wPFhDwSjX3Dwlg0CQy96HreiTlcFZsaAjY0NNvh3QUXtHeHcoKMNA7NjqLd8xHmzDzXDRvRO1KHtngTyhzL4SHeooAAnKMxBtUYQbGWa0Dc+AsWzSVy3qkjeItLCFsz4XoNMaRFFAm4SyTXbmQa2YHQSGacR/pAXO+zGFif4JdlHCpShBzstEz+YfJtmt5cnKKWS/1jnAnT1S38AGTynUFUTzJcAAAAASUVORK5CYII=',
page_white_wrench=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH3SURBVDjLjZNPaxpRFMXTfo4E+h1S8hmy7bJ0G+i60ECg2QXaIqVaqwttplXCSEMNDlZTqdEaEkyslIo4s9BG4yIOYzIo/v+Xnrz7UofR2DYDh1m8d37v3vPenQMwR2LfPNMi09J/dI/pruEzARZ7vV59OBxejkYjzBJbh8PheGOGmAFLZG61Wmi328jlchBFEYIgIB6PQ9M0DlFVFU6n04CMzXcIQBu63S5qtRp8Ph/K5TKy2SxCoRCq1SoHdDod0CEul+saMg3o9/vI5/MIh8OoVCpwu92QJAmBQACxWGyiJZ7JNIC1gUKhgGAwCEVR4PF4YLfbkUgkYLFYeCUE/ifA7/cjnU7z8lOpFP8nk0lYrVbenq7rswFJpQ7bThFbgUPIsoxoNMqzoCpsNhuHNZtNNBqNm4Anr3btL7dPIR2dY917Aqe4xyvJZDL8RuhkyodCJE0Ann8srW2Ipd+fj3VEfujYimtYea1MBEatEYBE78EAMOPqM2+pLx1dIPxdx6cDqqCID19VbhprMBhwkXkCsCqcdHcOzxE8vsD2fhVr74vwRlXjVLPZLAPw+G2hthmpQPym4em7XxAiZzdKnyUD8PCF/OjBhlxfXv/ZcX85Y/Qh/jYLU7o0QvxzE/dZQG1auI2Z7W3y6TUBFghyi3Eei0Z/4QrVT8W6WBitpQAAAABJRU5ErkJggg==',
page_white_zip=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEUSURBVCjPXdFNSsMAEIbh0Su4teAdIgEvJB5C14K4UexCEFQEKfivtKIIIlYQdKPiDUTRKtb0x6ZJ+volraEJ3+zmycwkMczGzTE3lwkbxeLE5XTqQfTIjhIm6bCy9E/icoOoyR4v7PLDN+8ibxQHxGzE3JBfHrgUalDnQ6BNk1WRFPjs66kDNTxqg0Uh5qYg4IkrjrS9pTWfmvKaBaGaNU4EY+Lpkq88eKZKmTAhbd3i5UFZg0+TzV1d1FZy4FCpJCAQ8DUnA86ZpciiXjbQhK7aObDOGnNsUkra/WRAiQXdvSwWpBkGvQpnbHHMRvqRlCgBqkm/dd2745YbtofafsOcPiiMTc1fzNzHma4O/XLHCtgfTLBbxm6KrMIAAAAASUVORK5CYII=',
page_word=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKbSURBVBgZBcFNiFVlGADg5/vOuffOdcb5cbKcjUn5V7poFJEiKMKybDYFgdSmKAiiokW1aNHSXWQQuGwR2KJFq2hTkFqUEbZRMzQIJS1nbPxjnHvvOd/b86SI8OSHx1/uT/ZeyslujCGBhEhE0ZTy5/Vr8V26ufzeiSMLQ4AUEQ4c+vHE52/tfiDnNB1UAoAICIu3R47+et0/FwcrZ0+fW/fDkYUB1JCrtK1bV9NnrqpySiSi0EbRNOHOqPX4lrX2bq2dLItromxffuzNr2eOffrsIENKaSIiqm6VdTJ1ospUiZTIEoq5ya6FPXM2be7168nxz6De+c7vhx+a39h/48tF8zN9nYqfrtyx956+4aj49swVMxPTtsyuc+9EpdepbXp41rGf/3oBL+ZmEKfunqiNBfv3TDl+acXqMDyxa8rlxaEUvH5gvQvXkpMXG9+fX9HrVFKdG8iKX26sFM/vGFen7PbSsq29WpWT01eW7Ns2Z02v0utWxrqVsW5HyKQM6mht7lX8duGOj7+5ZPv96y3913r/i8sU9j86Y9QGKUttEjkURAB1jMq4YClluWV6NUlTterSooN7N8opqasgEkFI2kJbgBxDvWiTs+evOvTKg/qdZDgqbq0Uj+yaBEmSIqlzUuVkrJO0TYC6jGKuSnSa0O9mf1weuNHe9PTWDUqEErSFQAki0amyZtiAOrVp6tz5qw6/ep8miqlOdnG59cy+WSWAKlOhkxMoEUaDAvKFr3Z+MBhdX51eW4PXDm7wyds7AAAAQJQwWm1AhmZQSlNKAAAAAADUdS5NWxqoYbTa/jvere6SRuMRkQUFMNWvQQT9TpaS0qnSrbYpf0MNKcVHT717/LmU0nyINQCCQAggwGop5VSdHYX/AYn4JwmEykruAAAAAElFTkSuQmCC',
page_world=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMZSURBVBgZBcFLaFxVHMDh3zn33Dt38poknTSJTVva1LaKxpQ07UJLKlQRTYVUcKNCwWUtCuJWXLkQxIVCl1kIIaCCG0GwXdgkamJS0hLHWs2znbym05m8JjP3nnv+fp8SEQBe+fT21XRT6h2t6ANCQAGgAFEgDuvcfLkot9R26ZOxG4MRgBIRAF7/fGLs2+t9z2itmgU8BAAARACEwm7MyHSZ9ZVaJTd3v3X8xmDNAABoT50KjNf81yaeVgoUiINEHNYK+3HCxacbOX/SMOkKdeJOlwY++KnFAAAotdQ0/PuEV4qeUE0snvg0+Yfp6XqD+rAdjQIcnU0Bg/2d/BKspxfW6oeViHDlm9euvvTsueGXT14Ev5WNSsjufo2NJzkWV6c4f+x9WjPddDc7GtMeKd9QqUa8+9WM9f5sGLsw0Nv33atnLnmSbqa8H7ARBcwXfKCF57sCbt0d5anMc1RshpVywmIx4njW5/vJtUh3tmc/7j/WG/yjEwqR5b8tw9/5mIe1efIHbpNvzHO64yCTuR9IBR5h4BEGPoIGpdGHs50XSiZiubKK5xqYXdwlCUvsHv2NOTPN1KMchw50UCwv4BtN4HmkjMYBImAC46fL1RLFxGeiuEdp2yMIY1pSLxBtHmX/8R/ICUecRPieAgFBkThIHJjt3Z3iKcKuOtvB0rrD+S0s5QP2ZqAuzNLeukZlfwctKZQojAYHhD4kVtC55fkfo60qtfI4qdQKKooIwibCjCXVPMXxTI2Zfx9wpK0XAZyAAL6nsZHFFFe3vvx58tc3L53tP1Jo81mq1YicQRpX6c4+pFaGO/kM2Z63mZuNcQDASIdPXHOY8S+mls5e7728+djMDvR56kRDiE0CGrOWuwtFVosHeevyZ6TSGURABO5tWsQJcdViAKa/nr137sPR6nbhQfrm8ijOWYyfhvoXOdTzHnPlOlTZ4gREK7RTGKOcTZw1AAAublv/aGgouxdLvYhoBBwAkEk7AEQg7SuUwvme2kmsyysRAaD/2s1rNpYhpdQZQeoAABAQQBAAEACqzrk7Gkb+B+5ReIstDYkxAAAAAElFTkSuQmCC',
paintbrush=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG2SURBVDjLY/j//z8DJZgsTUdmCkodmSV7eO8EkQayNN8+WPry3YOV/3d2ib0nVbMsUPPrT8/3/n9+Nun/1hbxP6Rolr99sOTtZ6DmD7cLwZrXVUt5kKb5xb7/P17U/b+4xu4/UHMRUYEI1KwK1PwOpvnSOgeQ5vlExQJQs8atA8UfPr+EaL662QWk+diSPDlWnAZsWjufedOaORyHZ0lrgzR/ebkfrPnWbm+Q5odAzYJY0wFQI+OmtXN5N62ZVbJpzYzrB2bIfX5zaxJY86NjYSDN34CaVbEmpN4lK5hWrJonBtS8ddOaeT82rZn9b8vSmn87u6X+393n///gdKP/QM3OOFNi95Jlks0Ll6+bOG/l//OXzv7/8+ny/09PD/zfPD/vHtTmVJxJuXfxErbW+UuyG6Yu+T9t15X/rQt2/k/t2vK/ctKa/0Utk7YuyFeXxpsX6qcvXdswe/3/tpXH/neuv/a/cu7J/9E9V//7Fi57n1w+QY1gZsppnfMvqWb6/8iSyf8Dcyb8907r+R+QO2tbbNHEIKJz46bF5SybFhVZbVqY17BpfqbEpnmpfJvmJfESYwAA/ZPGvT+K5AYAAAAASUVORK5CYII=',
paintcan=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJVSURBVDjLlZPbS5NhHMd3tbvyov6HIJRuhKAFgRci5GXYVQkG0c6HJGvD1FVMjdxC22wlQ9iG7uBh0rtDmxtb77vFoh2c7KgMtiGbU9xdgezb3hf0Ii3cxZfngef5fn6H5/ewALD+Vv8Upeh7HfzdM+pb4QwT7PPunOg8M7tlPh63JtE/4UM3f7WnLUDL3NE7vgmFKYbeF050Di3dbQvQSrvjzogTIm0Yt0WrePp25VNbAHcgrLopXMOgyo/uR8v4bAnAbrdLLwSIRCLK1HYaHKEdnQ9MuHZ/Edv5MpYcHljc1K9Fgvo+uxqanLYEr5wBhMNhZTqdRr1ex6jOBz8VxzONC+VyGQcHB5g3rcFGZWH+lscrc9ArWdjseqzzXWLMFEWdmnO5HEwbEYzpvmKZ+IFKpYJSqQS/3w/l+4XmlCXYnHGmMfTBOzyg9lxmkSQ5RpsbjQYKhQKy2Syz7u/vM0Z6T5+73W44nU7IlOrmS9tPPJn3VfsmXVdZLfKbra0tVKtV7O7uYmdnh8mEBtCwVCqFRCIBgiCg0Wggk8ko7sdNCMwJ3JogRpgSHA6HOhqN4ujoCIeHh6jVashkMkgmk4jH4wgEApibm4NUKqVEIhH74awXrfSPbyi+dJ020Wg0vguFQtjb20M+n2eixmIxBqDX6+nIpFAoZMb63owHHCXhOvOMWq1W7fF4UCwWQZdFN9RqtdKRSYFAcPonrj/fGPznIKlUKvX6+jrTfZvNBolEQvL5fPaFJ5GWXC5XGwwGiMViksfjtfcbT9Rq1gCXy/2vmdYfaGviUGKvapgAAAAASUVORK5CYII=',
palette=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALqSURBVDjLjVPbS9NRHN9T/0BWFBQUVIQKXigCpVIyygcDJUrnZdNEc5hlmZc0TdSF0bxszs3LvM3pXE6Xa3NL0TJTl7pcTY1Ec5rX/WxqCrHhp7NBPaRGD5+Xw/lczud8vzQAtJ0wWhLp+4kbUThSFGrQFwTrB59dE+ryA/3+vreNOMaPOmLkMeUT4mTMaoph1klh7pPApOLAUH4LPTn+X7qzLwXsKDDGj0wy8hibM+oCrI9pYTWGA0ZnWEd8sWZQYvXDC5g0XAzyo6BJP5f/R2C89OYeErlquiUPP9vogNgF1iYfbH10B0zxRMQFC4oszMsz8F3XBOqdBKqUs7a2B6fdHAIkMnu6le1w3WrwBLrjHSKWrhhYh72w2kVHjTIIae3eKFJexkp/I0YlKWhJdKsgZIanoTjMtlHPxSY9BD/YgbA2eGPteRjmWzOJazrmZKl4rL4AQT8TD4nIfPMjzKgKIUtwNtJIyxXftISclICN3GxYfHyw3FEEy1ALLIPNsOhkWGzLw5umCHCUflBLr2O29i4WXgnQwDpB0YY5NyapASmoxlxQrGAsFrAIWQ6D6Da0GecxXBaLFfLmuHI+TgrkCBCIYKqIwVKHEHWxxzZp758GbTrc9AqYu4WYb8kkRcnsLcPejzL5DKi3dfAQSEFX9RKRZkzxQklKIaqjD4PW9+QqVy+IxmdpOkwvOaB6xVjpa8QQOSMtY4DHAPW6GuLSVFwprUJxSQYWlRyMS9JQGXlw3PELZDB8OzN9c0hkdXua1/pYfTKonloHkeoWYVachCkuHZNFwZhrTMeCmov2rIsoY+wL2TaJJLKr4r6HzUyIpso4R9yp4mB8LWFgScPHtJyNjhx/CCOcCnccZTua77jKRkiJy51lmKlJxJK2lJBLoOMxiet+myDcKWXXXbBDGn/KTcI6brO7TUgzMcBl4Pk9d3tkhSB8r+s/l+k36mKOJpKW10VRh/TlzAOFJLLnTvd2Ffhf/AKfTM1hskDhXAAAAABJRU5ErkJggg==',
paste_plain=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHvSURBVDjLjZI/aFNRFMZ/Ly+GlCYlaMF/1D+4FQRJREigk7OiIAguHSwUkrkgLhVcpIurQjdBcLAITi4FETpIAw46CGIHJ0tFamlt3rvnfA7Ji2kTwQOXe4b7/c73HW4kCYCnc+engU8M1/b88kaFf1T05O65KvAAuFY6epw79xfZ3XoPwPjkFZ4t3uP37i+A58DK/PLGy0FAHli5euPmmbFSmfTrByRhnV0AJDFzcYrcyQtIuvXm1evrwBDg7FhpgmRnk+CBjRePQQLEltoIsb/5jeLk6QJQOBwhD4AbCoFc5QRuAckP5tzfI9neG7mDPIDc8RDwtIOnnZ6D/6seIOCW9ie7Ij7+mGDfYlB3F4oKfD8yTavVWnJ3zAwz+9wDGB5SsICbkYQc8dgkt+cekiTJ4aELkojjOG02m0t9ACFF5ig4bsLN6XQ6rK2toYFIWV+v1wkh8NeBOR4cD4ZbhJkBUK1WDwizXtIgwPHUUHAUDLcc3hO02+2+eBBSr9cxsy6gO70L8CwChqSRDrLqO8CEp97bgSGL8CgAsL6+PiSURKPR6Dt4lybJzPipKeRC7uRTiH+WkUStVhsSZ3cGePR2dbUMXMoeGXl2KpcpFotEUTTyA8VxjJkRjco2OzubM7MFMzvW+zD94+6D58sfRXpka4kRkDcAAAAASUVORK5CYII=',
paste_word=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJPSURBVDjLjZJPSFRRFMZ/7804ODhT42iaxZR/NiEFFVESGFHWTilo1SoqEGplQUQEtnSV4KrARRAELZKgVRshgoggUKgoDCUEIxumxpyZ996995wWM/knDfrgwoUDv++c7xxPVQG4f7mjG3jPRhUHx+cy/EPevUvtB4E7QH8q28r5W8OU8m8AaGg+zMPhm1RKSwCPgInB8bknawFxYOLkmbO7kqk0ZnYKVcWFJQBUld59Ofy2LlT13POnzwaADYDdydQWol+LWLHMPR4FVUDJ61sUJVicp755ZwJI/D1CHABxqLX4me2Is6jK+jmDMlGxvGkGcQAVQaxFTIiYsNbB/6kGsIgz65zvLl8jZpYJIks6244ECgeGuDL2TsUK1ijOuj8Ah1gDziLOgYJvK4xe7SMe8zd1LhQjbox9WAVgDeoEtYKqopGhLuZz+8EnioGjuyXJzEKFdF2MTCrG9GwZZxR/pQMnOCs46xDrEKcoEIRKQj1yrUl+lIRtmThB6NAIvFWAIMahtSdG0KgaZD0gofJ6+icmEha+hXyZj+jZ24BvawBxUnW1DrHVv4uqgZpQ+Vp25AuGjOcxs+DwPeX00VbE1QA4RYwgVlDrUOuQSEGhVBH8UDl1JEt2a5xk0ufYoUaaMgnEKXHgpYmi3oYdOVQUFQEFCnX4MXCR0NFUR//xNqY+LpFWy4meFmK+h2gVMPJicjIN7F+7pqXYAPlCyMjQHlCoBJbrF7pW6t8LITZSPP3H1fVdfPU5Cmxow6C+M9fYKVINWxVUFBEPVPgN0HNzn605xt0AAAAASUVORK5CYII=',
pencil=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFUSURBVDjLrZM/SAJxGIZdWwuDlnCplkAEm1zkaIiGFFpyMIwGK5KGoK2lphDKkMDg3LLUSIJsSKhIi+684CokOtTiMizCGuzEU5K3vOEgKvtBDe/2Pc8H3x8NAM1fQlx4H9M3pcOWp6TXWmM8A7j0629v1nraiAVC0IrrwATKIgs5xyG5QiE+Z4iQdoeU2oAsnqCSO1NSTu+D9VhqRLD8nIB8F0Q2MgmJDyipCzjvYJkIfpN2UBLG8MpP4dxvQ3ZzGuyyBQ2H+AnOOCBd9aL6soh81A5hyYSGWyCFvxUcerqI4S+CvYVOFPMHxLAq8I3qdHVY5LbBhJzEsCrwutpRFBlUHy6wO2tEYtWAzLELPN2P03kjfj3luqDycV2F8AgefWbEnVqEHa2IznSD6BdsVDNStB0lfh0FPoQjdx8RrAqGzC0YprSgxzsUMOY2bf37N/6Ud1Vc9yYcH50CAAAAAElFTkSuQmCC',
pencil_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHfSURBVDjLY/j//z8DJZhohU926XPeXuX/6tTkgL8nJvj9vzRX//69JaaSRBtwc0XA/xtLs/5/f3jy/4/Hp/+fmur0/1CT4Tpibf//5cry/z8eHv3/8/EZMP5+e+//kxP8/xKl+cfbw/9/PFrx/+66nP9fzi8FY5ABpycHfydK858vm/9/vZnx//P5/P8X5kb+v7u6+P/Jif7/8QYiiuY7Cf+/XHP5/+td3/9nW2L/3+y3/o83FojVjNWA/RPUidaMYcDOTrX/n57tw6p56USv/3lTIj6l9fn/j2xx+ORZoVcDN2BGsoICSPPD0xv+n1iZjqF5Qb/b/8bVif+3XJn6/8KzXf/7d2f9D+/X/W+eLdkDNmByrtz/Tw9P/P/1/OL/rbUm/w/PNPx/50ju//NzvP4fbzb5H9vu8n3j5Qn/N16bBPLC/549qf/796SDDPgON+AnMHW9WZX4/8UMh/+H0kX/r0oQ+7+lTAfsZ59qw//brsz5jww2XZoKMgDi90gnof9X5sT+/3p+2f8351f+P9jtAdcMwpY50t+7dyX9b9+VANbcvjMB1QURDsL/45xE/8/JVAVr3NVtvRY5cIEKW4J7NP737koB2wyiQXx4GBCDgYo7gPgTyNlQugMkDgAAPd9LkvTMtAAAAABJRU5ErkJggg==',
pencil_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHtSURBVDjLpZNfSFNhGMYXZ110E1QYFRR6Yd0kIhgE3ozlIII1oQvzwjAUtD8jSFILoigISqrVCAy2u6y5aAQpuJH9cUabK06GSEOnHCcrzAzsrJ3Z3K9zDjGMhhzog+fm4f093/e97/eZANP/yHDhXKhyw6TfMT/qrluJuA7z0Vs5M/1g33bDAXFfHZ96T5KRoijJGKP3rAxfqQoY3R15/BGK9IZs8p2uzOQQUZdjxRCsfAujzPpIBE4ji726tICY+0jGEJyTn5GOt/FDPMMHbwOJx+1E7zhYs4l/wVNNyBO1LC/eItXfSPx2DWtOwShcNOCla49h+J+A4PXdLKVeFIXFcxW8sm+UnlvN+aBNmFXVVQjoaS4t1WAp9pRIX2tReKx9fzYzcJP8xCBp31lipypyoQNmpx7gdu5iSYqw/HmMgYvVhO9XMTXiRPQc4u3VaoK1QuKnCnPXDp2b4FoZX29YNH+6EJBVX9eC/zhfeiwMt5bgb9pKf8de/c7asfPvA6xe3y9tQ/P1gAbrZsY9jaTFhyyIfbzuPliA9d7YhKTsbQYVUjpMLKqS2gRUf04vOGrZwjFrCZ4T5ToY6q55srq54fodl6Mt5b9SnWXMn19PomUdQ3ZzbtAmXDD8mUbqd3apO878mUJSgzX/N4hT3VGIyxnZAAAAAElFTkSuQmCC',
pencil_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLpZNdSFNhGMfPKBKvomKECH5c2AcVNpgILUNmid40KaKstGzk/Eh2oQjDRrYRpSN3xkoXuWzpqW2iCdOokYYTa9uxliLVMItlWKR1YZNt6uHf2bkYGUtOdPHcvLy/3/99n+d9CQDE/xTvjZ+dmYlTdtk3r7GIcZOHMWHO/PihKyuJt8BvLcI7qgqhgAfhGRrem1K4NKJevukITj5AODCKyMwYV6GpQXhIGcMLDn8fQfiTFdO9FxD0UVxFBbTxaIgXvBJ0YNFfgZ8+JV6bizHdXQuPQYY1m7gKfn8WwTcHsfSjBbP9JfDrJVhzCvHgetsBnO8QrYLjCp6R2+Mm11jEuOw4ieOmHX8XPGnahoXZIQ5WUjlQdmWj8p4YcvNeXHx4DHbagLpuGQ7pty7n6DYJYgKTPC0tCgfoPrhtCi65ujMbfa9a0fPyBgdaaRLk0zpYXjSjgspH1tWE5T1agYATGGtSsBBwY+nLOAbUYozcEkHO3jcKtziVaHpcjSuPytHoKINmQAGTS4vTd/cj4xIRiQki7Ouat5fhqykXLoUQp27vhM1LgnJfh+V5M+6MXmMlVWhzaVBOFSBdTbxNVhGJnKBYuhmT7SVY9N3HvM+GYV0BjrSlo9CQzOTphYxEt5Ep7diH1uFGnOvMR6qaGE9SEQmxHpzI3YJSqRDtlRnor98Np07S8+d0dmkFc2cseUhtIMZYeMM//0b2yHMpDesgVBHrf1//BaVfyc6lPey4AAAAAElFTkSuQmCC',
phone=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF6SURBVDjLjZO/S8NAFMe/l8Q0ASUUXRwK9R9wFDfdunV19Q9wcmg3/wHp4FLo4CA4Ce3o6OLWUZwKpbRLMdDFCKH5dd73SkvQkvTgeLnLe5/3vXfvhJQSu4xutyuDIEC73Rb5fQM7jizLMBwO/+1b+UWv1+soRZdCiGO1PFJzT33r4Hq9DsuyigFRFN02Gg1UKpWNc5qmehJimmYxgE6e5+GsX4VrZQgzHlfiwI7xdP5VroAOzCZMidaFgGVIENH5sPAdZeUAwzAQxzGECrSpVt0Qq0ygErKbAh5DqOC7dxWj0gtKEGSl5QAWiYCX009t18Wj9UxvK8DYBugHz3hN+hiNRnp9+PAINlzpLawBTedqlflkpcC/uUYVKFewrsF4PNZ2MpnozLPZbJOg9AgMYNdx0BJUq9U2CQoBvEYGzOdz2LYN3/fhOA4Wi4UG839hDVTf/4RhuJ9XwLdAy/5Qr1EWAqbT6f1gMGgul0sdmAMjSRK4rvv2F/ALQmi5wbpDa1QAAAAASUVORK5CYII=',
phone_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH/SURBVDjLjZLNS5RxEMc/8zyPz25vLFkEUodfhpBFGpQhXSoivFl/Qp46RJeC7OQpSAJZglow6BDsKRCCoEMdKgwTUShfLtZidFhpsyV7xM1dfjPdZLXQndt8GT4z850RM6ORyOVyliQJ/f39Uq8HNBiqysTExD96VJ8MDw8Pmdl5EWkB9gNNIoKq4pwjiqKtAdVq9WZPTw+pVGq92HuP9x5VJQzDrQGqSiaT4czIXnZESkUFMPbENZ52f99+giiK8N4joXH7nBAFhgBD0xG/q7o9IAgCarUaEhhxCJgRABIaGI1NoKpICAOjYAIigICZ3x4QhiGqyrOOWVR13TzvPZkw0zhgJMmTSqU4/PkYra2t7HvwhMmr3UjbEteyVyxZXU5+rZTvv7z36a7Uf2I+n7euri7K5fJ6ZzNjpviBctM8p9pPc6i5jTdzzxmfHeXrl6Wh4H8eFAoFVJWFhQXMjMlvbzl5tBMfeDpbLuGlRveJswDXg80rmBnOOQCcc6gqP5MSTbKb3vYbANy6+JgjBzoA0sHmM6oqi4uLxHFMqVQinU6znJSZK44x+LoPgMFXfRRK0wB/NgBEZKVSqeCcIwgCnHPEccyF45cZnx4jJuLFTI5YIt5/fAfwaIOJ2Wx2YGpqqndtbQ0RqQeztGv+4OrOH82GxkAC5MYfFu/8BdnT67i+1n1kAAAAAElFTkSuQmCC',
phone_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH5SURBVDjLjZE/SNRxFMA/7/f9eadWnmGBIl3fk1oKGsMh6KA/Bw5CUzS0NbVlkFs0RATh0HLi0CDUEmgQtCiBJA05hf1Z6pQT7Wqw4C697s7vew2BKYreWx7fx+PzPu99xcxoJvL5vFUqFYaHh2VrPaLJUFXm5uZ21OOtj7GxsREzy4pID3AEaBERVBXvPXEc7w2o1+tDuVyOZDK52RxCIISAquKc2xugqqRSKc5OHKYtVqoqgHEo0WC8/8f+BnEcE0JAnHHnvBBHhgAj8zHluu4PiKKIRqOBREbCAWZEgDgDozkDVUUc3J0FExABBMzC/gDnHKrK8zMfUdXN44UQSLlU84CJylOSySSZL6fo6+uj6/ET1i+lGVie4PWFUQ3OloH85amNh7sCBluv/puc+Weweu4onQszZG/cJpk5LdX5qWOfZ6fvT19sWYt2u0GhUEBVWVxcxMxYe/OMk9krtBZmkPFrtC+8wB/vciZ2a4eBmeG938whBBLlVVq7MzAw9H/YvR6cit/xjapKqVQinU6ztLREb28v5Y5O1t+/4sDLm9Sq31kHKmVHcHzbtoKI/K5Wq3jviaII7z2JRIJU9jqf3r2l9KeNsmvh10/h64oEg/w2g2Kx+GhycnKwVqshIlvB9FtH94nih4PtDesILloxbDQ3tfHgL9Xv6UA7GgE/AAAAAElFTkSuQmCC',
phone_sound=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJRSURBVDjLhVNNaBNREP7eZvNHU9aqBDxkiScvIh78Aw8qleTgz0lvHjwoivW/YBBBDxY8iIeiBiMoeBPBQIUqXnrQgxBy0l6aEkIWbDA0JTZtNrub3XVmQ7YEbRx4O+/tvPm+mXkzwnVdDJNsNuu2Wi1kMhnxL7uE/4jjOCgUCpvaZf7kcrknFMlRIcQOOm6nFaS955xMJiHL8nAA0zRvp9NphMNh/7Jt295ikEAg4DtoVyYvoGN8VF8/W/IB+JKiKDjwfgxR2YHucLouRkMW3hz65YNql2/dG5HsqfV2p0jH/X4N+AKziYCLO0cEHo67mBqn/4qMVdPB8ZYF7dLNi9A7M+bvNmKys09LnZnwASRJgmVZEJKLEIFIxB6gxYCkcHDdRHB55SXanZPWavsGpQDaX/NT4Ag4DUGp3v9KPpSB4Cxoua6NmWQc52pNytV8pM6+FdrhE9MEsssH4CIxwLs9857uF4+1ElAgEZosS+g213uVbHcAw9jogz7A7Eoec/onlEol77xt+hW40VLaMmTbYse72t5j12MjFKpuLPwVQWr0VI95Zy+C+tXzGCN7aWsMWxZ/TlDYX4LC+QGTCtMxng68AjuUy2VPVyoVj1nTNC/MYiIOtfA5S6xnw0S5pjWKau3784EI2IG7rq85/0QigT4Bi7r47YEW312DaX0YmAV+Rmau1WoIhUKo1+uIRCJoNBoeINv7otbnX6jNhaWBVqa+X9N1PcbMLKx5Flhzf9A0ukNnoVqtPs7n86cNehohNqaW991uF9FodG4zgD87aSjZlUUofQAAAABJRU5ErkJggg==',
photo=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHfSURBVDjLpZO9a5NhFMV/bxowYvNRjf1IoCDo0KFJBVHEVbeCi5N/gM6KruLi6KiDKA6KIC6CQwdtBxfRrUGHFlTQIlikjTFpkua55zo8r7aDipALd3keOOdwzrmJuzPMZF/cOPFXBMmRHJMTTJiJYCIEESy+ZQGqczPIDNxxd/AMDriBu+MSCkJmSA4CJ8Pym+UIIAs0177S3Wz9F3O+WGCiMrmjwM3pbrZ4fvo17kR237XAtcolRvdOA+L+9TscHB/HTGQAlLqwuHWbxa1b9JMVTBDSHRi82qijbgPXNsGEpx5kouYo+2jpI/3kCUudiwzUJBgMAoQAjf4ZFtZP0mq/x0xIYPJUQQoQLHAsX8fMeNk7y4DVCGKw0q7ytHmByx/u/lYgOVnJUbBomAa8azWYr5b50unRGZln48ccYzrH5/VTtHuTKIxQk8dUdgMEE/XyN2YPTFHJHaZWFPIan/KriEccqT5ExJi15FiwWCSTo+CYiYk9h5CL4NvIhSOmctOxCwgh3J3vauAWnc8GEzInt2+U3s1nuEWwmPlOByzthuSUSyV+XUDWTOAJxbEyhcJ+pPgxc/4KnbUFQOTKx3n74B5uQhI4JEkMMHl8ddZ3d/tfzH+aZNhrzDDk/ARfG6G/LNZPQgAAAABJRU5ErkJggg==',
photo_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJVSURBVDjLpZNLSJRhFIafGf/QTJ008zKKIWTgQkdByMxW5SJQgmjVToKCcNNlIQTRosCNu3IhgbsWCSG6ELwUlJi1KS3FCymUDCWh43VG5zvntPgnaxMIfXA23+J9H973nICZ8T/PG3l0+p8KqoaqIWo4UUQUJ4pzihP/zwMoqalERcAMMwMLYoAJmBmmijpFRVA1UDCCTL6f9AVUHLHlH8TXNg7knB3KoTBc9IfAxIivbTB84R1m+O721wD3w7fIOlwGKD0PujleUICIEgTQVAqjO12M7jxhNzCHKLjUJAXerkbQ+BSmezhRLJVB0Gf2sWuPLrEb6OXl9g2SGsMJJB04B1O7TQyunGFj6wsiiiqIWoogJeDEUZcdQUR4nbhEknlfRGBuq4S+2HVuLz7dJ1A1PFVDnfiBaZLpjSmaS/KJbifYTmtmdbOGXL3Ct5WzbCWKUJdGtZrfyt8CTpRI/k+qjhUTzjhJdUhRq+Zr9jzKM8p2n5OIecR3Enw8dYJEfB0P8EQNdYaIUphejpribA81xVCKM8qIzqyTuRXkXGMdpXkVvJruY+LzG7xMxXOiqBgZR7JIdA5g4ov5nfs7sFhRzuWWFiQoRIqbGJnppb6qgd6FfjwRBQsQys0nJycPVb/Syqt32V4eBJShF8McCmRxsfIaAHfOdzPwqQsJ9PsEi7Oz+7v923myvdUnUCMWckxHx5mMjtHe1EPHUCsZaemkGQQOco31beGHpeWhew3VjVQU1bLw/QPjU2MsL613Bg56zvVt4Q7gJpANbAJdE4+j7b8A7WGuGfrlZ+8AAAAASUVORK5CYII=',
photo_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJRSURBVDjLpZNLSJRhFIaff+Y3x3S8pKmjpBlRSKQGQUTbLhBCmxbSrk1CiyCJEAJxUe6qVQURtGvTJooQNLtYEbVKoVDJMUrFS6iMzs35zjkt/indBEIfnM23eM/Ded/XMzP+5/kvbhz5p4KqoWqIGk4UEcWJ4pziJPjzAerbmlERMMPMwEIYYAJmhqmiTlERVA0UjBAjn0YCARXHyvQ86eXEljZHy0qpqavdIDAx0ssJBo9/xIxgu20aoKfuMiVFDYDysPc+O6urEVFCAJq/wlDqLkOpO2S9cUTB5Scn8GGpFU2PYrqOE8XyNwgFzAH2ofIpst5jXiY7yekKTiDnwDkYzZ6gf+EoibVJRBRVELU8QV7AieNwtBURYThzhhwTgYjA+Fo9T1Yu0BV/8JdA1fBVDXUSHExzfEmM0l5fxWwyQzLcztJqGxV6lp8Lx1jL1KIuTIta4MpmASdKa9UvDlbGqIvspaVMUWvhR3QC5REn47eIvYnjzy6T8UPIVCQIkqihzhBRagqbUFOcraOmGEos0kDlqwn2TITY39FNYdMB0qMDfH07SHV6Ed+JomJEikvI3HyGSSAWeB5koGB2hn2X+ohMvobh62wvK2d3YyNz38bxRRTMo6yiitLSHagGljafu0Jyuh9QJq/eJlLbBKe7NjrQG6MopwFBfGzsb7b/bB7pPh8QqNHse6Q+P6f46UWy6TlSwGoiTKYgjLeVNr7rqO/dFi2+tqvS+X5omtVFx/f5sOQy1uNttc7vOxq6k0uznWHxGiVsMwb3Tg24vt9oKbFtb+0ZHwAAAABJRU5ErkJggg==',
photo_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKiSURBVDjLpVNdSBRhFD07M7nF+rc67s+ILqS4ubBpEERoIFKwC0IIlfYQRlE+lz3UQyFElBD1VA8R9NhDP/YmkYbUQ1T4oCStC7rF6G5ri5o6rePOfNOdb7YfH4KgGQ6XuXPvOefe+cZlWRb+55JGr+37KwNjFodJMEwGk2BHw7Cjk5PswtrWZjDTBMgNd2QJsFktnqIcY2DUZNfYhGCUh4DJd5MOATMNrMxnkV9e/Sflsopy+JXAbwcWJe3mFwffghsAnPgT9HxFOYfSHfU2PR4M3kONz8eJBa5Y3MLY97uEO9BdM6QOGEUUaJQ3Sy1g+SkaZ5MrW8UdCI5nx/aeyhQ1P8JLrR8FtkIF1GwQCWFKP4SRxf1YXZ/lyrQWPqLjoEhg0C72lrVQgYlXG4dRQNIhIcys1+LZylmcn7v/y4HdI/GFURVfGCtgenUKXbUy0toGNLELS2ut8LIjUBfbsL4RoFoRu3kP20pgs7bIOURKAxA/1qM65YW/+hhS2RlYbY/REH4IPG/CTt8uZJr6kPhGe7JckExOYPG5ZDGED6NZ1NPdc6IHmqahYb4Br9+PgZXoONrdhxLJjUymCRMT1VCTKUi2MqN5tntKMX15BKKvEwdOduLqhVP4vOxCUFEQjUaRSWdw6+ZtigsIVQk43n8R6qdZCLYyyEqFV8aay4vyCi//MJeuD6BQyKO3txft7e2Ix+OIxWLQ8xriIZ3XlFd6HQdziQQ/YZtrQHpB5S8HB27wkzc0NASPx8Nz9kj23E8SDKdjQIZqXX/+jZFIpLmxsfFpR0dHoK6ujkmSVJXNZpFMJuF2uxEKheD3+4nYWFJVVRgfH/+yhSAcDovBYDBKisOKotRQk0hpIZfLnaG4IMvyMMVtuq6b6XT6Kznq/gFyr64cpzvFtgAAAABJRU5ErkJggg==',
photos=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIZSURBVDjLpZPPS1RRGIafe+feSccfacxACUFYDGaKlEmrqFVEi6Bdy7YmLqL6A1oEtQiCglZBtYhKKqiEFiGRUERQthhEjKi0UNSbkk73zjnfd1pMM2VpBB64i8OFh/flOa/nnGMtx7tzoq3g1HnqHKoOVUXUIaqoOkTK9+PXJtpXAgSq6vV0dyALBuOKWJdgBVSUb0lAfWMDz1++XjVBIOKMiebC8x2P8DxwDqxV5qOY6aklLtOHFf0HQNUPvVpMSfB9D3WOg0MH8iqKqPJeF8k113G9d+vMCrVygRXFqvI1igkCv/xThJ1dbdgFQ5qI2CzheakVawXWKsYIM9NF/JSHqqMkvitFkde7Z5I6r4i1isukqQnWka1t5uRjrdYKrIjGkDo1eWi7U0fFxuh4RN/Y7zaKWdElxs7mZ0OdwIpUABoOjxTYlGvk/2y0YIxg7XgZ0H/jczvAzf58YqK59LH2e2wJN5Cx8MnAlZ4L7M5+5NWld1hRMnWGIFisVvArOio2Utmj3He7iC1kgSdf9rNoNhNqhBXhyMAoSRIj+gegYqOplKGrYZ6p5jzWv8tAoZuGW6cxpgVrlcGHbxgcfotIeQJBFfDTRseO9XTW91HDDCPfz5Ekt2lt2kZwsRz7zIP53LKH9CuBaAwcvjqFF87Sum8je+nkw7MJCF6QJFKNvQpA08MjBUQVEcfToeWjqnx/rXGtc/4BfOeC6F88S7oAAAAASUVORK5CYII=',
picture=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHwSURBVDjLpZM9a1RBFIafM/fevfcmC7uQjWEjUZKAYBHEVEb/gIWFjVVSWEj6gI0/wt8gprPQykIsTP5BQLAIhBVBzRf52Gw22bk7c8YiZslugggZppuZ55z3nfdICIHrrBhg+ePaa1WZPyk0s+6KWwM1khiyhDcvns4uxQAaZOHJo4nRLMtEJPpnxY6Cd10+fNl4DpwBTqymaZrJ8uoBHfZoyTqTYzvkSRMXlP2jnG8bFYbCXWJGePlsEq8iPQmFA2MijEBhtpis7ZCWftC0LZx3xGnK1ESd741hqqUaqgMeAChgjGDDLqXkgMPTJtZ3KJzDhTZpmtK2OSO5IRB6xvQDRAhOsb5Lx1lOu5ZCHV4B6RLUExvh4s+ZntHhDJAxSqs9TCDBqsc6j0iJdqtMuTROFBkIcllCCGcSytFNfm1tU8k2GRo2pOI43h9ie6tOvTJFbORyDsJFQHKD8fw+P9dWqJZ/I96TdEa5Nb1AOavjVfti0dfB+t4iXhWvyh27y9zEbRRobG7z6fgVeqSoKvB5oIMQEODx7FLvIJo55KS9R7b5ldrDReajpC+Z5z7GAHJFXn1exedVbG36ijwOmJgl0kS7lXtjD0DkLyqc70uPnSuIIwk9QCmWd+9XGnOFDzP/M5xxBInhLYBcd5z/AAZv2pOvFcS/AAAAAElFTkSuQmCC',
picture_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ7SURBVDjLjZNPSFRRFMZ/9/2Z90anRlKzMS1UjFyoWS3U2oVQ0KIWrXJRFC7KTdRCqUWQkBt34aKN1a5FEUkRLbKFiBAoVi4kUQIbtfwz0zjjezPv3ttCwyYr+jibs7g/vu+cc8XDwbE+pcT5TFa5fsB26fzWtsC1Gbh85vA1AEtp0X76WGWp67pCCJN/yVMggxxP30xfADYAGV85juOKR29X8VgmJaaoKvtK2E4SaMXK9zCT01EKdB0Wxdw4V4VUQvyEWtkADMPEEJA1Fqgq+YoT+kzSTxHIAMtxqK6MMTtTSFGoBKXyXVkACjAMga+/EbJXSawn8aVHNggIdBrHcUj7YYrDBhqdN5gtgBDoQOHLHF7gs57zyaoAqQCRQyuJZQi0zp+qAaD1BsCllFS6EI2NryR+IBEiRDoVIRIqxzQN0GJ7BK03IkTMvXxZWCTqzlNQaOCIgLWVAhYXYsSi1ViG2LYZS/8KsHdTHm5ibnyIokgcISW2V8q+mnYibgyp1O9nseVgarkDqRRSKQ7432ip3I8CZuYXebXWjZkoxFQRbj/wyHjezhfd87de3p3osbTWCODkxmEBYNYnyKSXceffU9LaQcP0GEuZKY7UHaViVy1Dk8/E6Mf4nebO8qLNVPnGZLgIv6SGZP1ZtGnzbvIthw42Ig1JY6wNKXI017cCXLVcWyRRuWhjWTMIsYnSPwuApeQitohwqu4SANdP3GfwQz/w3LVClnj8ZGimJSt1vdZ//gOJVCI6GR9hIj5MV9sAva8v4poOgCf03179oubO8p6KqujN1obj1O5p4tPCOCPvh5mbTfb9F2AT0gtcAXYAKaB/9F686wcCdBKN9UyNSAAAAABJRU5ErkJggg==',
picture_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ6SURBVDjLjZM7aJNhFIaf77/k/9OkJpBWjSWtaUEREVEXL6uoiIOLULCigyi4VRy8gHQo4uIsuIhOOuigCJKCLWpx8kLRQqFWhV4SQ1PTNGn+5Ls4RGlrq/jyLd9wHs57znvEvafvbmktTlZq2g8kq2VWfl0HfJe7Z4/v7gVwtBE9xw6kWn3fF0LY/EtVDUrWefxi/AzQAFQC7XmeL+4PzVFllpIYI73hO2G3iDSawnyYT+Mxmsw2HBJcOpFGaSF+Q52aBMuysQTUrCzplu94oW8UgxJSSRzPozOV5MtEhHioBa1XduUAaMCyBIHJE3Ln+LFYJFBValIiTRnP8ygHYRJhC4NZMZglgBAYqQlUnaoMWKwH1LREaUDUMVrhWAJjVk7VAjCmAfBppVSOYHAJtCKQCiFClEtRoqFN2LYFRqy2YEzDQtRuYyqbI+bP0BSx8IRkodBELpskGevEscSqzThmOcBdz6bwLibfDxKPTiOUwq220t7VQ9RPorT+MxZLHYzNnkNpjdKaLUGefakONDAxk+P5whW2f5xj6/gko7fnSTcnmjN38pcPZeRNxxiDAI40ggWAveMHlfIs/swILfvP0TOcYV39DVtOX8VLb2dxJGONvhroHzjolh2xRl5VOI4KxwlauhoJzDxgz6le/M9D8LKfpliczR0ddmHM9Dq+K4roemznhr0gxC+U+f0AeJvP4W9Mw9GLS977kthabHZCjnj4aHBiX02ZHcasfQOd0URz5cMzK/LkAsFilgpQmrdRNtPC/K1qmV53t/WFmiPXUgnpONYkpbzka85W9aq5/l8AgOHu9svlwvR5W4kOZZspA7cPZ+SNn/95GW/b/Tx4AAAAAElFTkSuQmCC',
picture_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALMSURBVDjLfZNLaJRnFIaf779PMuYPmSSjCUmYKBXREBLwEhUXIm2otlRBN1WsIoJLRcRdN0VEcKWgC1trV7pQBHGjokKrYBFNWsQk2gwpphNzGWcymcx//T4XIWpQ++4OHJ7zcnhfcenGk9NSiu9nA+n4ER9LLRxNAxyTiwe+6z4MYEgldm/b0NLgOI4QQuf/5EmIo5Brd1/+AMwBZn1p27Yjfrv/Bo8pSmKQTHqchFkkUpL8dIJnL12q1AoMUhzdmSGWQsxDjSACTdPRBATaGJn6cWxrhKJfIoojDNumvWUJ2eFqaq16pFzoygCQgKYJfDWBZb6hUCnixx5BFBGpMrZtU/YTpBIaCrXgMe8BQqAiiR+HeJFPJfQJZEQsARGiZEy79SeFmz/yVbLD7f/luBLe6F4DQKk5gEMDpXIOpZv4skwQxZh6gnIpydq6SXob8yTqd+NmuihklzNw68Kp9wBNkNSbGR17jevkqKrWsEXETL6KVGmGXWvqqXG7yf/zHEsELKpZTDLV6hrqQ4DZSFOii1dP71Gb/A8Rx2Rsm+2bV1PbvgU/dxmrSjDSN0DoBVNeZWb9OweDUweJpSSWki/8CXpa2hBmkabOLtylX+ONnkezIsxkG5qXVeG0t2X1kdtDhlIKAfTOBQsAvaNAYfg+y5qW4i77Fm/0HJoZEUxnmHzQx6Pq/eU9B3b0AWjiE3n1J4dIORaNK7cSjF1AtxT+dBuv/3iM3XuWopWJ53c1xxRFZEhneh2di3tY9O8r0pUioWwm9/cZJB7lfDOTjwao++Ycdk0KQxfvLhqWIa5cvTfcE8SqQylIZ391dx76mcHLRxn6/SF2aydePidHWg/Plh8UYkMvYGpcmgcIpRba/+ngqvDY/kMGWkzfneuMZF/MuA11G7880d//qYJ9BNi1qa7S3Z421qzIoEz/YWV8Yt+2k38Nf66hbwFu+Dui0xbh3gAAAABJRU5ErkJggg==',
picture_empty=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFhSURBVDjLpZMxbtZAEIW/N/bP7w4aSiQuQEODEFdIAVI6KFJwgeQenAEBFQVU1CBOkAsgroDSRMof77wUa68tK5GQstqVd1fz3rw345Vt7jN6gM8/zj9k6u3lIYer8ZaoTY5dD8OOj+9fPz/tAdJ6d/TqyeNhGCR1eMIkAMIGez6bMl7z/eefE6ASXF7lfr8f9OX3P0oxY2b9lmQspkznkibTnB0/paQEEACHESI6hKhTTa7mrepegsxNDWhyadAaLIQJCQssiAA3kxuCBpKRRMhkCBlCVW8a1p1rBPYCXjKKTrNRkOvCuougkkTULA4tHRQ4IVb1aQSeCJbMJlZgTdlTqsRwt4LqddUFJms2YWPfpsBugRFTRWffEkojs4CnH6sRaLoNQbImEWlXZV7L3xRx2OmCvH745sUj0Ozd89wMMY4H+k5uBA96ff326+/LQ/Gz/3mcfQe74FNt7T2f8w1Fi68/h3owMgAAAABJRU5ErkJggg==',
picture_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKFSURBVDjLfZM7aBRhEMd/3+7t7W7OmJjnxRjDKUYNnPEJHjZqo4WIEAQhFhY+asVGsLIUbGwVRMFGfKDpRWzEqCEKIko0GlQ0MV4e5u5279sZi+AjRh0Yhml+zPznP+Zy/+A5EdNXiiWILAtD57deCgKPS4f3bTwOkBI1B/ds62gOgsAY4/K/qAgktsrNu8OHgDlAKRLf9wNz5V6RChPMmJfkWscIvSmsCl+nQ54P11Gja0nRyMn9ORIx5gc0FVtwHBfHQOx8Itc0hp9+x1Q0g00sKd9nRUcbI28y1KebEJk/VQpAAMcxRDpO2isyWZ4iSirE1mJ1Ft/3mY1CGkMHRecJ8wtgDGqFKKlSsRHlakQslkQAU0UlIeUYVOer6gCozgECmpmZzaB4RJIQ2QRj0rRUXI6tGiXjlUHNPwCOYZHbzodPrVQmFlGjDdSaBkyxgbzOsKZrO93+/QWXcfR3gNfC0nADxaEi9ukIDL0lP/mF3MqdLG7bwDJ9gJ1+vVBEVXg5cZREhESErmicQkcn6oJTE1PbVE8yfYdl6/oYH76GYe8vgKpigN1zxgLAzU9Smv1C5v1dsvkDUBlk4Optth45xbvRBzT4OQ82z61g/uLXJKxHPJfMkhXULikh0QiokHx7RG77Cdqn+4OB84VaACfwzBRSpad1Kz3ZAuuyBfItm6kbfUj98vVI6RkqZTb1rkbijwTBKzq7dzg2io8DmIu3Bs+KmkKcaP7Hidui++ktnRJmu0KSbwOoxjy58YJNvWsxbh2Eu3h88fSYiuwwfxoDYOB84X2+72q767xFbfGPFR3ccDWfnw/xqv/M9dTfvs5GcTB4obesoqjIb6k/KyKoavY7re8z/KbjU2AAAAAASUVORK5CYII=',
picture_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKISURBVDjLjdLNa1RXHMbx77lzZ+6dOBqZxMa3BBJBiG1alb4o4sZSceGiQl1VaBeluBV060pCKe0/UAqlrtUuisuKLVEQNMSFYqxpTFJjSJjEyWQy95x7zu/nIoqOL6XP7pzFh8NzHvPr76M/ipgvV52k1vN6tP1YjCEt8ss3n+89BRCLmhNHD/RuStPUGFPgv5IJBJ9z6cqDr4E1YNVKkiSpOX91iYwaDTNOf8885WIdr8Licpk7Dzrp0EFiujh9vJ8gxjxHY+chigpEBlw0R3/3PElpirpt4IMnThIGercw+c86Npa6EWl/VQwgQBQZrC5QKi7xpFXHhgznPV6bJElC05bpKkco2lbMC8AY1As25GTe0sotTjxBAJOjEogjg2p7qxGA6hqQsolGcx1KESsB6wPGlGg2KlRKWykUIlDzFiAyVArbeDTXQ1ar0KFV1psqfrGLuZktROt/Yio7Th5sO6AvA8V32Frew9LYEv72JIw9pHhvgb6uj7EhZ3v3IGcu7ieoNW0dqMJ47VuCCEGEnXaBOztSVBxOJrCtk2yr7mRw8yc0siaX/z224cIPtvLX6cWVWFUxwJG1YQFQGHrC7bHv+OzdrwgqBAkIymx9hqHtB1hxLW5N/1l//1xUjc0b9hrKG8m8I6gwVfubXDxecvKQs2wb7O49yEq+Gt2YHKnFadHUkbzzg559YMwzSsl8hg+eng19eAkEFR7Xp6lWNjM6c41rEyMLznPY/Pzb6PeiZr8LOvTyF1+pfdGZOYcTR+YdO7p3sW/gMDenrzMycX21YVc+mh3Wu+bVYbwt752Lmh/2Heq4ev+PR074dHZYxwH+NzBw1jSDFDpaEgbmh3Xy+f1Tcg5RdvF2jMwAAAAASUVORK5CYII=',
picture_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKsSURBVDjLfZPLa1RnGIef75zv3OLEDObWqcaYYJUWgqa40MmilioE2kI3XVXRFsmqUAqi/4G7brpxpaLgpUIEcaEW23RbKRYqASfKJOIlMbfJZDyTc/kuXaS1BmNfeBcv/Hjeuzh/494PxoivmpkJU8WbZteGnoTQ49yxLz78HkAaKw59NtTTGYahEMLl/ywxoFXOtV8fHQVWAc3UBEEQigu/1UhYoCEq9HXPEnl1lDUsLkeMP2qjxb6PpJ3jX/ahjRD/QmWmwHFcHAGZM0NfxyyB/5h62kBphQwC+ntKTFY3UPQ7MGZtVRLAAI4jSO0cvldjaaVOqhMypVA2JggC4jSiPXKw2DWD+Q8gBFYZUp2TqJSVPCUzCm0AkWONRjoCa+2bFVi7CgjppBFPY12P1MRkSuO5EXGjQKtfYjC/RH53li1Jvw+DADivAI6g4G7m2Uw3yUKBFruJVrEJtdjOzJMSZa9K+YMapfIRBrvuh/dO7RgGkPZ1gNfFu9EgT/8co1h4jtAaL+nk455eDvRVadt5kPjxTTZs7BVqa3305nf9x161UFkYQRuDNoYd6Rz7enoxQHX6BZG9QXHgW3RznLBjC4l+Sumj3S0qrp2W1loEMLx6WAC4A0s04wXC6b/oKI9QmLpDXpvCcecRToy/cR6SJi+nU1eKde5VR0V0VMT4Hu88vIgXWYyawTFL4LRi84wHVyrJeGV5RIaeqGPytl3de0GIf1CWpHqLZOoXuvYMYbOfEbrBg9spMk3Ic20nqsuHDl+eHJW+FD+NjlX3ZdoOvL7i/eJ64b2hT9xs/jRStjE+ltm7zU8TubWc++QXj57cP7qay9p1/cm14d/TF6dsXPnc/vFjuXHm6+2H19PJt33e88n5O8v1sW3NxXhlYmLuxDdnH15dT/c3MC9g0QGCji8AAAAASUVORK5CYII=',
picture_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALVSURBVDjLfZJNaFxlGIWf7853597JTJzQzDi9U6chPxCaOmi78idIlglEsEJJqFmIaDddlEK7EF3pKmJcdiUU3bmwXQiiGxsQQdBaF9HYUFucZP6SNHWS3uncO9+PiyDNGPDs3s3DOec94rOvflkyRrzRjo0fKQ7L9p6uBN/l2tuvnb4EII0VC7Mvl/K+7wshEvyfOga06nL9u7tvAvuAdmQ8z/PF58sP6fCAPXGH4cImKbeFsoad3RS/3c3SZ08gGeTy2WG0EeJfqIwVOE4CR0DsNBjObeIl/6IV7aG0QnoeI6WA+/fSDCRzGNPrSgIYwHEEkd0i6T7k78ctIt0hVgplQzzPI4xSDKYcLLanmCcAIbDKEOkuHRXxuBsRG4U2gOhijUY6Amt7W3UArN0H+OTZC9NYXCKjiZRGiCThXoZMskgi4YAVhyNYux8hkzhGtdEk69fpSzt4QvFop49mIyDIjiAdcegz0h4EuE9TTJ1i4/ZNBjI1hNa4nTzHRxfI+AHamP/O4omDOw/Oo42hG2uOtCTSfYVc4Rk27q9S2/wAIyMGt07xze5JdpvbT1248OmVlZWVq9JaiwCmT19CK833X/9A//EB5ubmCMOQjdFRfvw5R8JxmD33Kq50qdfr4tYt76NKpZKX4sBeV3/9g/ZWm9lzr/PJ1SV+v71KcDSgXC5Tr9dZ+niJRqPBsZGTauHsrKxUKucd3xUtTJfnCi/waD2ivz8LwOWL79GNFfPz80xOTjIzM8P09DRxHPPsS2dCgGw268mkFF98efPei7G25T83VTLRrvuAePf9D5VSSi4uLpJOpwEIwxCtDT99e631/FsL2Wq12hEHhzExMXFibGzs+tTU1NFSqWSklEeazSZra2t4nsfQ0BCFQgGl1M76+rqzvLzc6AGMj48ngiAop9PpG8ViMe95XgJwtre33wGquVzuBuBGUaRrtdpWGIZn/gGotkJJF2DBHwAAAABJRU5ErkJggg==',
picture_save=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKFSURBVBgZBcHLi1V1HADwz+/cc1/j2Az4qiYjhQoXIg5lr11tWgi5aWWLwB64FMJdu5Ya/QVFbkKiNrVKMAmipLI2UZJMZcmMM+k8LnPnnnPP+X37fNLHX9y4mHM6Pa7zoGoAABAAQLdk0PXRG6cWz0GZI7128oWD+waDQUqpAwAAACaZtpn6/Oqt13EOynGV+/3+IF26tm7inlG66dCBVcPupiay+1tDv96aMxNHlPZ459VD2pwSQFk3FEVHkaiLFYf2rur3/rZZjTRto+z3HT74kD+Xdpnv7ZUzAFBCRlEkVazpdddt7Gyq2om6aTSxrd/v266G9gwLIRAASsgoUhJNVrVTk6ayM63UudFmpKnIrbJIIgIAlBBBkZKBfUbby6LTtfzbebv7jOtGOdszk3es/Dfy/qd/yNGZO3Phelz9+c4zJURQFMlsZ8GdlbvmBst2xhMvnnjSK4uzvvz+X++ePgIALl1bk3O6XgQiKIpktrvfw8Pj1n9ZN66m7o8acOXHuwAmDVsVaxu1lLISIrh57y1tztqcPVGtWe4lnWDaZhfPLso5BDrCTElVVba2a2VESHh58RyAztENP3xVmFRT713+S5Fo2iy3WSAiCGa6WZlAAIB2OK/JoWobnaKLkLRSSiKHiKxppuq6UQ66aVOezh078CwpCRBG4590U+nsyd2aXKMgiJQNyp4Ln9x2b2tb2SvT5c++XnqubuNoBABtjrmOrmzHhzfetnfmUUlhbfyPN5/+QGFgXNXKM6eOnwcAgG9ufhePPciB2ZGXjp0w31ugYGOyYP+uxkyPMiUpIgAAwFNnr3z7+CPD5+f78wblA5o8lXKWopXT2O+3l6xuTf0PNZJB+2NWN98AAAAASUVORK5CYII=',
pictures=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJSSURBVDjLpZM9aBRhEIaf2c3dJZvNxT+I0cJK0SJGwcKfwvIQFUsRG9HSKkGLKGhjKZGICViJ5RkQtLSQGOxsElEIFsKJCsZ4t7ls9u/b/cZCLxqTzoGBgRmeeV+YEVXlf6IL4MaDFyNVzzvVirUsIojgqyXsDIlY0iR+Pj5aG98UUPW9U1fO7N/qeb781fM7RZgaJqfnzgKbA5Yjyp7ny5OXC4Pfm1+2tDN1FLBFt1txeotyycUYsWNTswtpEtfHR2u31wFE6M2BlTDberF2oMvzqihKYS0uvlsuuSRp7hZodXJ67jywHqDKqip+Kyqku8fn6cxHd6m57ASxICKoreCI4DrOzszIwNjUbJAm8aPx0dpIV4ekCkWhbmZdgnbuXD59CM/r+9eyABKmpn9yeu4S8Bsg9FoLIIUjPW4QKp7Xx8LXNq8b1+mvLhFlhk+L2zm+6w5H9+9GxJU1C6giKFnxgzwPKaySA7m1+P4iPaVtWFJsucG3VoRVi4hW1wAO9psW2U6vvMPtLlVxHAdVWIkyWklCoauEJqUZJbRIQQVV2muAKEpeTNTnDxorQ2lqKGz8C5BYGl/3YivvCE1E0NrDvoHKxju4d612H+Dm1KvhSpfdDZVBayGIC4YHxjjcGOH9h08EJ++SmxwFROSPhU5EUfJsoj5/BJVzgvL281WMMbzJMrLBEtm78xhjuHDiDWsvpKob8sbkTGOpGehymGgQJhqEsbZW/uRis623Hr5uqep6BX8pqU/U549ZnCHHEQT6FZbpbBXLahg/BpD/feefqppLG2N7xVoAAAAASUVORK5CYII=',
pilcrow=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAD7SURBVDjLY/z//z8DJYCJgULAgi6gUvvEWEOVY6aqJJsxw79/DAxIDrxw+9ee/blirnhdYKjHtcpKmd1YiZ+JQZKbmeHivV97+j0EGEGaGf4T4QIZPiYlXhZGsM2g4Pn/FyL+/x+I/Ec4DEA2vv32jwEetjAa6B2YYXgNeHD/Z9iOM19XP3j3h+Hbz/9ATRBbwbH19z9hL9zrkn0PpMIUCh4Jaqpz7IZF8/8/DAwMWKIcZzQ+mCD3/tu3v+8Z/sC88h8aDgRcgAzAfoa54C9WB+A3AORnmCYw/ZdEA/4hO/kvAwMDyS74j4j6//+w6ifkBYQmXAmJccBzIwCU7Hm5Y0odkQAAAABJRU5ErkJggg==',
pill=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJhSURBVDjLpZNLSBtRFIaVIvFVsKSCG7UuWkVcGIiiGxE3YgUhgvgIirgJiIssspBK6cK0iIgbCyVgNyWGqkiLK63KiG100ZbYQiUP8zBxJuOMyaRJiCHB/j0zC0GsROjiwOVy/u+c/5578gDk/U/kTAicnDwOBoPP/H7/W4/Ho+N5/t6dAU6X63mY5yFJkhIsy8LpdHIOh0OdE/Dt8PAlGw4jk8kgnU4jkUgglUohGo2CAJL94KDwVgBjt7/yh0KKOJlMKtVFUcTZ2ZkC8ni9+LK/b/ongGEYs5sSZLGcLFeUxeRdsSCHeH4OZm8vcgPgsVqnue1tZEgQj8cRiUQUcZisnJ6eIhAIwEtw+X5jZwfXAOGpqenEygqyu7u4mJ1FjGEgCAI4jkOI7NAUcHx8rABpMlhbX3ddATyDgwZpcRFZol7MzCAxMYHfY2MQlpZAI4TP5wONECcklN9hY3MT72y2SgVgr6pSecfHkaXLlNmMOJ1jo6OI9vdD7OlBaG4ObrdbaV8g8fetLaxZLANXY9yoqXniNxqRWV1F3GBAbGQEkb4+iN3d4Ds6wLW2giVLAj3ij+VlHExODlz7SO8rKh4dUbW0xQJJr0ektxdCVxf49nZwzc0QCBCj8JMtZnh44MZXXigtzf9YV/dTNBr/JE0mCJ2dCLe1gdVqIRBAamqCr6UFX+vrdbfuwkJZ2cN9rTYo6vWZ5NAQBALEqYOoRoNfDQ34VF2ty7lM8yUlD6zl5R+OGhslVqO59NbWXn6urMza1Oqnd97GFwUF+fOFhdo3xcWm10VFunmV6n6ubf0LxSvsH2lz/wcAAAAASUVORK5CYII=',
pill_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKvSURBVDjLpZJdSFNhGMcnIX4GhglGKHlRemHhYBoSiEkgZhgaIjqwNaqBGuxiF5FFhDOHzRmZlGBdhI5SopQInZOJtXkTKGYyN+fm5s7mOW7HtjHHhv17zynLqDDowMN7znOe/+/5egUABP9jewY4VlePOp3OG3a7/YnVaq32er37/hlgXlq65fF6wbIsb263G2azmZqdnU3fE/Bhbq7d7fEgGo0iEokgGAwiHA7D7/eDAFjjzEziXwEGo/Gu3eXixaFQiM/OMAzW19d5kNVmw3uTSfFHgMFgUFpIACfmgrmMnJj0zrfAGbOxAcP0tO83gHVgoI3S6xElgkAgAJ/Px4s9pJW1tTU4HA7YCJzzj01O4heAp7W1LTg0hNjUFLY6O7FpMICmaVAUBRdph2wBy8vLPJBsBi9HR5d+AKz19TK2vx8xQt1SqRBsacFnqRT04CDICrGysgKyQqwSITeHsfFxPNNqs3iAMTs7wdbUhBhxhpVKBMj7pkQCf10dmKoquNRqWCwWvJh4CsXji7iqOY8G5elwxfUTN3nAWE7OMbtcjujwMAIyGTYbG+GrrQVTWQlvWRmo4mJou67hzvAlvFnoxRylQ/dEE+q6j+Nk8yG14Hlm5pFFki3S1wdWLIavpgZ0RQW8paWgiopAE4C0/QxGPt7HyOIDbnBQ66+gWy/jAFuCntTUuNd5efOMXP4lpFCALi+Hp6QEbpEINAGwhYU41yrE24V+7H5G53s5wLcN9KSlHTSJRE5GLI6GGhpAE0CAVOAXCvEpPx+nmg9H7+mk6NBJeHHHuORnBTtr1KSkHBjIyHi1WFDAuoXCbVtu7va7rKyYNj39LAlUXlDnoUt3mc/Mndw3P4PdF+l2fHycJjFR9Cg5WfEwKalak5Cwf+cfCVYRC3Blfz9VnP8rovbZoQ8oWiIAAAAASUVORK5CYII=',
pill_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK3SURBVDjLpZNNSFRRFMfP+5zxY2a0l81Mg6NDiosKRKFNYC6kIERpUaSbVuGmja0yp5A0aRG0CdolIa5CAqmNEzQQfoCamhDT4ETNjDOWOOW8eR8z7753u/eBE6LUogeXw7v3/H/nnnPuYTDG8D8f/y+Hb8lkM8swV03TPIUQeu1yuWa8Xq+5f8787Qaf4/F7Ho/ngdPhsP9VVQVZlrOapp1tbW3dtTcp4Ki1vLb2cCubxaVSCeu6jokQEwDO5XJ4dXX159zCgpP6sUdFjs7Pj0uSdLdOkoAAgEQEAqHRQRAEqHa7a4j4FvU9BIhGo2MBn28o4PVCsVi0ASR3IDWwbT6fhxqPBwyEhg4BNqemRltMc7jR7bYjUvG+kC7DMOx9UlQoGsaxA4DtcHjUJ4rhOp4Hc2ICrJUVW0xF+2Jq3QQuFwqgKEq8DNjs7x9whkJhJ8kZLS4CSiaBmZwEiEQOiGn+lmVBLBYDpVDosts4Fww6fN3derC3F4ylJUCZDFikXRYpnEmL19EBWk8PFOdfQv7dJGipBCAHv8ug0uOLs+gRL3Ncw0lRBEyKg1KpspBCqOWmpwl0GTCfhrZrN8EROg3ax1np0/vIWKRLUNhfmlbSslmwcjkwSW7lpShgEUuf6o/1GWi6cAWciSgwL/qg8ssraGyQOMzgQX6nUPgWX1/fqPP7z1T4/Yy6sVG+gcCyINAO4BI4fSGAy7f/zMCIHziLabSf8tPa2uPtTU0fmltafBUYC2o6DU4iRnt7kCUFTPtiu+f7bkhVyTdQ1LZBJQA5z0EsAVvlWXhSXV17orLyeVsg0OlhGLdObpBRVSulqr31XY5zoqtquF5CPM+mQd5B8PU7Zxo6vn9gmEZEkfFwXHsFy3aaGCdKlvV2UNdlejZ3PXhHyWUGOJNpMDm8RVTPLs2i8d9nr6w0wqZqDgAAAABJRU5ErkJggg==',
pill_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALDSURBVDjLpZNLTBNRFIb/6RMqtEDBFopFXiFGIERYmKgJUXBFQly4gI1LXbghcYPiI/JYmbhxLQvC3hjjQkSL4SnvYCRUCNJCSwSq0OnMdObeud6ZBIzB6MJJTiZz7/m/c89/5gqMMfzPY/tXwkYkUmkRhOuU0nJCyKvs7OyXPp+PHu4LfzvBSjh83+PxPM5wOs1vSZKQTCbjsizX1NXV7ZmLBuBPMbOw0LsVjzNVVZmiKIwLGQewRCLB5ufnv49NTGQYeZY/VQ6Nj/d5vd67BV4vOAC8IjjEqA673Y4stzuHi28buccAoVCoJ+D3dwZ8PqTTaRPAewf3wHwfHBwgx+OBRkjnMcDq4GB3FaX3TrvdZkVDfCg0QtM0c52birSm5f0G2O7q6vY7HF0FNhtofz/02VlTbIgM8ZORdvS+v4bME04kRRGpVCp8BFhtb7+ZUVralcF7JpOTIJEIhIEBYGjoV2Wiojj/DDpeNOHT8hJSothkjnEsGHT6W1qUYGsrtOlpkFgMOh/Xw+Y1MKpCZRQqhxTlVaK2+CI+fn2Hxc0pppC0+8OdhGhLWq0lRQ4HGDeHRKPQueOUB9FVNFffAGU6qE6hgyG2H0VN8QWIqizMRkb2a3ssebYfsqzK8Tj0RAKU93YIMI5siDf2vkDTuYm6Bo1qOEgnUXfqEkRNskytj+5ZdkRxI7y4uJRcXmb2wkLQVMoMhSgglMDnDsLvLkGhpwx2ayYKsgOYi45hbG10R1LRYP7Kz3Jz8+srKuYqq6r8mYzZpc1N3GqLQuFjVHkrMlFZef5Z4XzZVcxExjEcfrOtElyJ9bHPR3fhaVZW7kmX6/m5QKDRIwhuhRsZkyQ9Kkmtbbu7r6t7LKmG4GVXKDy8peqmeOXYZXrkcAgeq7U+02JppIytqbr+tkNRksZe2QMhRXWrS9Zp2bc+tn6o+QkxLL87j8znVAAAAABJRU5ErkJggg==',
plugin=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHhSURBVDjLpZI9SJVxFMZ/r2YFflw/kcQsiJt5b1ije0tDtbQ3GtFQYwVNFbQ1ujRFa1MUJKQ4VhYqd7K4gopK3UIly+57nnMaXjHjqotnOfDnnOd/nt85SURwkDi02+ODqbsldxUlD0mvHw09ubSXQF1t8512nGJ/Uz/5lnxi0tB+E9QI3D//+EfVqhtppGxUNzCzmf0Ekojg4fS9cBeSoyzHQNuZxNyYXp5ZM5Mk1ZkZT688b6thIBenG/N4OB5B4InciYBCVyGnEBHO+/LH3SFKQuF4OEs/51ndXMXC8Ajqknrcg1O5PGa2h4CJUqVES0OO7sYevv2qoFBmJ/4gF4boaOrg6rPLYWaYiVfDo0my8w5uj12PQleB0vcp5I6HsHAUoqUhR29zH+5B4IxNTvDmxljy3x2YCYUwZVlbzXJh9UKeQY6t2m0Lt94Oh5loPdqK3EkjzZi4MM/Y9Db3MTv/mYWVxaqkw9IOATNR7B5ABHPrZQrtg9sb8XDKa1+QOwsri4zeHD9SAzE1wxBTXz9xtvMc5ZU5lirLSKIz18nJnhOZjb22YKkhd4odg5icpcoyL669TAAujlyIvmPHSWXY1ti1AmZ8mJ3ElP1ips1/YM3H300g+W+51nc95YPEX8fEbdA2ReVYAAAAAElFTkSuQmCC',
plugin_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJFSURBVDjLpZNfaM0BFMc/v7s/pv25tnsXlrzo2nav3d3y4CaelqLw4l1JEVaUl1GeUHtQXpiSUvKqZFFWJtFajO62K/KwlXQn7syfe+3PPX883AkNKefl1KnzOed8zzmBu/M/Vvm74OnMiayZJlTNVfXO2fT5nX8ChJYm9zRhJFrrWok1xAJRTf+tgyWAU6neDwuyUCx5ieJCEREZ+xsgcHfOjJ50M0XV0LL39sa2QEwYnRr7JKKqqiER4cru641LNFBT1tfGMDfMHccCNcMd4s3xsLribjyeePp7EVUVdcPcyBVe83HuI+KCuRMKKjBz1oVjiMgfAKJk81kaqsKsrG3h/dc86loex+dRUwQlUhdhz7VdLiKIKLcPDATBz3dwbPCgx5vjZKczqBnmirihrjRUhVlTvxYzxzEGRx5w99Bg8MsdiCjqimjZ62KymmIz87x5+YLZ2SLF+QJuxR8jHL13wEWUFTUrUDNKXiprYoqYUZ13ossr2Lh1E2uaYtx/fpPh7EPS3S3nQt8rJ1a2syq8isnPE8SbkiSakiQiKTqiKWSqSKqtEw0pnau3oUGJdMdmgCOVACURBCXz7hkbop1MvJ0kl59CVYmGo8x8zlMV1LGjfT8Ax7su0z/eB9yqqQSQkqBmJCJJRI1cfoobe/sDgO2XurxQmOZ5bojR3CN6tl2ld2AfNRXLAObKABGevBpBVFlc0dwPYcWorw2Gx4aCzckt9I/3UR1U8ijzAOBi8K/vnO5u6QUOA/XAF6Bv+EKu5xvVXGolRpHH+AAAAABJRU5ErkJggg==',
plugin_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJGSURBVDjLpZNLbMxxEMc//+42Sh/b7cOjEVXV165UDw7iSkIEFxcuEpc2JOJxQkIkHnFz4+Ikri6kRBsbKhLR6ksvJEokdotVVbZLdx4OW0pajcRcJvkl8/3N9zMzgbvzPxGe7/Hs4IkRM42rmqvq7fMbL+34m0DB3OLjFRjxppImGsoaAlHduFAHcwROtV0cn5bpTM5zZKYziMjwQgKBu3Nu6KSbKaqG5rO3RJsDMWEoNfxZRFVVC0SEq7uuR+cwUFMaixswN8wdxwI1wx1i1bGIuuJuPBl9Oj9EVUXdMDeSX98w8W0CccHcKQhCmDn1kQZE5C8CooykRygrjLCsuIYPU2nUNW/Hv6OmCEplSSW7r+10EUFE6WzvCoLf9+BIosNj1TFGPg6iZpgr4oa6UlYYYWXpKswcx0j0PeDugUTwxx6IKOqKaD7rTLGa0vjsJWsH7hBKjZGNljJZE521cPheu4so5UXlqBk5z+WZmCJmtD1LsSldQPPeoyyqi5Md7mLpw266txQeCv/8Ob6sBcV5NTlKrKL110TMjfreBE37T1P08j70nGNJpJw1tbX0P/ejYYCcCIIy+L6fdVXrGR17RTKdQlWpilSx4VOGouV1sP3YLP0zKwhZsDrfQU5QM+KVrYgayXSKG/tuBQDbrmz2bHkJU4OdFN88yPfsGFPAl8kQGiI5Y0HofdGHqDIzom+zYMUGVtcEVY97gtqKxYRDhXz5ILx+F6jjl4N/PedHe1Ydz4wnO0Ia1GrI3zpc2dolF34Ah+h1f9LfEqAAAAAASUVORK5CYII=',
plugin_disabled=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADtSURBVBgZBcFBSpVhGAbQ8/18JNEFu1dEW4MbaAFOhDbQPBoE7ag91MhRtYgop9KgIsIfDcQQ3+ftnNEAAACACcCnrzlJ5/zsBQAsAB93fbKzG3kOACwAp+vDbdyrLwDAaJ87ItKHI37eVGUpL7fAJLZaM6Id7bf4DmBSWrtxJ9qiHSgAk/htz8atiAfRNt51KW/G5GzwoZ8oLVpcObDFBSZQIqK1iAgKk/ddHmsRLdpTv6z39SiYlGNtdaS1diVWb/eASWk/PPPHtdg4FAVgocSxuPZqvB6rFgVgUi5FqX9Q+SZ3+QswGgAAAPAfKnCHO1UwyPEAAAAASUVORK5CYII=',
plugin_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ8SURBVDjLpZNLaFwFFIa/O8m0lTxm2hJHU61Omuk0M2o1aBOl1a3QkKXQlWRhXImCGxWkCxVCobhQcGFwJeJKKIgQQSFqrWBS0zRFE2xTOiZpHpMmITNzc+95dBHQlKTZeDYHDvwf5/GfwN35P1G/U/GDsXcnzLSoaq6q333U/XHP/QCJ7eJ3DmAU8415cs25QFS7d+tgG+D9pweWI4kqscdUogoiMr4bIHB3PrzynpspqoZuZu/YfywQE67Mja+KqKpqQkQY7P1y/7YdqClHG3KYG+aOY4Ga4Q6FlkJKXXE31qeHmfzqjNc1PkllaZggnHl1E6CKumFuzK7fYiVcQVwwdxJBHWbOS3uhK/0ED7SfIpV9hpXpPH99P3huEyDKxNIEzckUmYZWFqtLqOvmOL7BUVngZEsXTY88y/L1P9kTRDQ1P0TjwcOpYKsP3vrxdS+0FJgoj6FmmCuP2x1ea+0k3fYyG3NfUysHLExWWQ1DErVq/p4riCjqiqgiJhyOy/Q/fJz0kdOEs58TJO+QbGyCqMSn1X947o1vpuoB3vyh30WU9L40akbsMa3lefqOnCDV3ks48xmJpBCtZSkNX+J8HFEy++8KIkox04HiTK/doCtq4Pl0Jw8WTxPdHqRujxOuPsbsT79RX6tmv3j70s17jBSLICh/LIyQnVnnhSCD+qPMXf0EI6SyfIhbF0c5G4fkt4j/BUgsqBnFg0+RKc1SONnP3tIvTF24yNjQMqNDPzOUzXFbbednEhF+nxpBVKlenye69i3Z46dYXSzz9+VrXMgdsrmrl2tqurajlbfGKy8eqHW2ZepPdGTx5MavtYXFvp6B8Rv3+4W77h1o9knDsw0AAAAASUVORK5CYII=',
plugin_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJQSURBVDjLpZM7SJUBFMd/371FVnp9VVppoaI377UwKrgVREMQ0WPIoQgKGnJxiGjJoMUSpIaguanHFkFGDUFLQ0NlpRiBg4qV1e1mJr7u/c6jIVDDx9KBw4ED58f/vAJ3539s2ULJq+9be800qWquqk/bUzePLAaIzC++VIKRjOfHqY3VBqKaWkrBPMCVxo6RnOQmQg+ZyE0gIj1LAQJ351r3ZTdTVA39G72+eEsgJnR/7fktoqqqERHh9rF7xfNmoKbUra7F3DB3HAvUDHdIrE0Uqivuxqv+roWHqKqoG+bG8PgQo9OjiAvmTiSIUmPC7iBKt+kiAFF6M73ElhdStnoDPyYzqCtqStSz7FRhc9VxDkx00nTnqIsIIsqT5mdBBKA9dTPo2HsrSI+nUVNCCwlVEBMaJaSq8jCx9dupD6c4WJNkz9YUojJ/CyKKuiKqiAkxFRqnshSsKULHOqloOEt1ZghTRURnWzj/vNlFlKK8ItSM0EPMjZ2TWSoTp2H6La/uPyJ1rpXSkTTpz19QmaNAREmW1VNeWM7AWD+Jkm3sW1HJrlgDBcWTWHYA3NDx11Tvv8gp5OP1qMVmAKEIgvIu/YaN+RUMfutnXX8PRZsascke3KbY0RTHcsPk5fVRtvVkvWRzF2YVhIKakSzdhqhR/v0TyZpDrCrI4PITgihdD/sAw6b7WFu3Gc2GLS/aGhKBu3PmwYm/q1FFRGgZ++V7Tt0NopFBXH4B/s/1R1fG+f7hPX2P2x4EC73zyxs7Mq62ys1xsznuMxEz3L3rDwilfn3qWP0kAAAAAElFTkSuQmCC',
plugin_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJISURBVDjLpZLLS5RxFIaf7/NCZqiZYYW1sEbTKWsTDUS0sFpF/0K7aBftulCbTIQWQi1r0aK/IKhFIBUodlNKKtBIy2qsKUPLacbvdy4tRmuhSdDZHDhw3nOelzdyd/6nypcbXnx25oWZplXNVfXOpUzvkb8JxEuXT9djpFvXtJKqSUWimlnpgyUC53f3fEskyQcP5JM8IjKykkDk7nQ9P+tmiqqhpe5ta7dHYsLzqZFZEVVVjUWE60dvrl3igZrSUp3C3DB3HIvUDHdoX99eq664G4/Hh5Y3UVVRN8yN7NwkM8UZxAVzJ47KMHO21qYQkeURzj085apKTUUtjdWNvJoeQV1LOF7Cqsy9ZT4pMB1vQkQRUW4fvxvFAJcyvVHPvitRbi6HmhIsEFQQE4IJwYVCmKepoY2Kwhsy6T2IytIciCjqiqhS+fktZglugoqwsb6Ftg17+VHMc2/wGlq27Y/Ayb7jLqLUrapDzQgeiC3hUPrYbwTDyc6+Z2fTPuaSAkOTD+joimvLFy+nG9tQnInv47TXd/Dywyjqxrvp1wQTxAJBA9/nf7B7837mwk8eTfRPlwMEEQTlWW6YHQ27GP80QVGKiAqNNVuQBTOnZiepX7OB4fcDDLzp/5IIh0sfBEHNSK/rQNTIfp3Cpeg3Bi+TWBIVJWFrQ5pM82GevOunb+zup0TozHb7qwUE4cnYU0QVEUFEi7dOjFcBHLx6QCtQhj88jKO4ivtjfR8TozPb7aO/c/Av1XwhyquVrS6YNue6fWJx/gvSRpXk80tR+AAAAABJRU5ErkJggg==',
plugin_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKJSURBVDjLpZLfT1JhGMfduqu7/giv+wsk5mKGB53UBHHiDzjQkQjhmJyT6OiQBhommKJLaVrLZcPWpi3To0KspiSYv8AL160XOVy5uVrz23lPV0225rp49j77Ps/zed5937cIQNH/REFRyHBb3rXb6FhlT+58bJk5E0DIuC8Kaxwmd0cxkRtGW9L+9cw38Ky4jiZyEUQ2gnAtM4v/BPjWedxNt6Ez1Yr2FSe4D46T8WwEY9sDuLVAHza/bTqwztbnTa9r8wUBXmn4+e5jPMuNYCI7jPHsEKLbgxjdGsDw54cYXH+AR5keGGM6FAR0pFg8lYbJ1vspHu73DNgEDWfcLJ1WOJctEqAXhhfXCgMkp+FO2tG14pG3uuI0HMtNuLlYD0asg2XegP6MH/ZZC7TjFCrGyqAeuYJTJjpEC0Y2+mFfakCzaMSNhVqY52vQ+K4ajiUavZ98CKwK8K96oRpSngbY5kwIp3rADJngCwiIRqPgfW40TOphmzWhs7tD1roCXbju1aKkpOSCPCg5DclpcGIL7H4GoVAI+/v72NvbQzweh8ffjs4+D3ZyO7KWTCblHoqiemQAGR6UTGp94gDLuZDe3ISVp0FpKNA0jXA4DJ7n5by8vPyXXq9/QyAsyx7KAHrGiJBkUllLKYR7ggzQRilcLlVAFEWk02kkEgnEYjEoFIrvlZWV5wlAEIRjGdD4yoC+tW7QvQ1g7MwfgFY7p1Qqf6pUKlRVVclBcqKRGgFYrda8DKh7WY2aSS0q+q+iwqBBoC9wMjU1dTA9PY1IJAKnU/qdHCfnRCO1YDCY12g02b9eobi4+Jx0xUtqtfqL2Ww+stlsx1L80Ol0RilKpfwb0UiN9JDe3w8qq6SmnYkEAAAAAElFTkSuQmCC',
printer=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJtSURBVDjLjZPfS1NhGMfPXfh3OG/E/yBImNkqrYGCzAthh+WNgXihwQYb2CoYukGwsdRLoYUWQbRAhqzc2Q91IrrVhlhLqznL5Tyb23m3s317z1szBzM68Lk47/N9Pud5XjgcAK7OVfM7/a2piE87HalRoLVHStrp1VKvLVi7fE9wns/WaXi58UgoH4kl/CxIyOZ/cyRKSKRFmF/tw/B4p3jl7utLFwp6baHiySnBxheZUkHkM8HKrgSpUsVGWsaDN/tQG/1PLxT02EIlRbBJBZtfZaztlSF8JEgdFqBMdnh8im7LSqWpYHJysqXHFiS5AkGMfi12UP0zRRm+D6fwxvPI0dWu3Q8QvV7f0iCgzQZKnl4WjqkgcVDDeyrYpqLoXoWtsbxTpLUyrlsFDA4O5vv7+w1MQBu7Z2dnEY1GcXsqjCwVJDM1JCixb1Vs0VXCdIoAXSVLBTcfhhEIBDA+Pg6NRtOtCLbpg0wmA7PZ/F8oWUEQMDAwsKsIiCzLUFhfX4coiv8kFAqhnh8bG6txFosFhBDG4uIiUqkUEzVDqc3Pz5/leZ4HZzKZkEgkGG63G8lkEn6/vylKxuFwnOU7OzvBTUxMwOfzMex2O+LxOJaWlpoSi8VgtVrP8u3t7eDoHvB6vQyXywV6Jwyj0YjR0VE2Zl9fH7q6uqBWq9lZPd/W1gZuZGSk6vF42IHSuPD8JZbfBpvybOEFOjo6WHZubg6tra3gDAbDzNDQ0LZOpwPvCqNYIjg6IfhBOcxJSGdL2PtewKeMiKJUBu8MQ6VSKc1bFFPDv8C7ItXhJ2sYdv/lDmOVodR4Z6R6vucXuxIEyKz+W40AAAAASUVORK5CYII=',
printer_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKgSURBVDjLjZPfS5NRGMd3F/4JRRfRVQRdeJOE1EVkaq52MfDHhbS3aeEk82IGMzaaVrrcsh+umUQ3QYsWlOKEbcrKzW3qxNqWG2IsreYkl/pOt7379e28pyQXGR348HI4z/M53+fAKwAg2OaU8q3kbI/HJtJ68gSIdJ6kSDuZFGom8qVXnX07a7cp2FTecqZX2SS+b6YQi/9klU0hGGGhHFqC9N584kTz8J5dBUKNK7GxxWHmc5aQgecTh/GFFFKZHGYiWdwYWcJJhf3ZroIqjSvJC2aJYPZLFlOLaTg/cgivbIJPtrK2hTLVeOavArVaXVSlmeDWNzn4yW3+5dyvFGnY5rZgDsSxTkY73eHgJBJJUYGANEsJcfJYWCOC4HIeH4jAR0TexQwdY2w+Qc7SqOh0oq6uLi4Wi6VUQBrLBgYG4PV6ca7HjRgRhKJ5BAn+rzm8J6O4SQoHGSVGBGduuuFwOCCXy1FeXl7GC3xkIRqNQqlU/hd8rdPpRHV19QIv4LLZLHimp6fBsuw/cblc2K5vbW3NC1QqFTiOo5hMJoTDYSr6kyev+yDvP4+Ld0RgblfiqfUhGIaBoL29HcFgkGIwGBAKhWC32wt4YOyG2sRgOKDHu4gVvTYZanqPoLTmMARtbW2w2WwUnU6HQCAAi8VSgERTgUH/XQzO3Qe/tKON6B29hJKmfRCQOWA2myl6vR7kTSgKhQItLS00pvBaMUYCj7FzDfn0KGneC4FMJssZjUYq4BtfvHyFsTcTBRy/cgA91gvoskpoc5dFQhMcbSICqVT6qL6+3ldbWwtG70YiyWF1g8M3wsp6CpFYEprnnRBrD0FnbaA3819+X1yz31HwLzB6T66xfwqNht80UCYhvN6MY5cP0tgEltDN9/wAJ277Y3yZERAAAAAASUVORK5CYII=',
printer_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKqSURBVDjLjZNdSFNhGMdPqJU3QXdCQXhT3XUdQqS0rNR2sRB3YWxMIyxEYtM22MJKY+kCo7ESgi5CK8MgWrimTN1xm24izuWGKUutOdP5se+zj/PvPackFxW98Ls45zz/3/s878uhAFA7nFGPSKo6HGZhp4MlQKhzJISd44lK7Rhb0kw/2l27Q87D+XY6tR5OYCOaRCjyg/VwEt5AGOq3S5B1zcVPXXu376+CSq0tvh1jMLmcIaTh+MxgdD6JZDqLyUAGd98voUxp6fmroEJrS3CCKSKY+pLBxGIK9AID/2oUXGermzEINKPpPwpaW1sLK7RjzFaUwQzZbWYl+7OLFMyzMRg9EWyR0c7etjISiaQwR0DCMkKEHBY2icC7wuIjEbiJyLWY5scYmouTbymcu0NDLBZHRCKRjBeQoKC7uxsulwsXO+wIEYEvyMJLmPmaxTQZxU66sJJRQkRwoc0Oq9UKuVyO8vJyASdwk4VgMAi1Wv1fcLU0TaO6unqeEzCZTAYcTqcT4XD4n9hsNuzUNzU1sZRGowHDMDx9fX3w+/286HccehXoy8cwWFaAYfERfHreBqlUCkqlUsHr9fIYDAb4fD5YLJYcRh7IMa0oQcKoAzs7gNiLG3BePwF56VFQCoUCZrOZR6fTwePxwGQy5XLpEOIkjIdVQMtBoL0Ya/dPo6d0PygyB4xGI49erwc5Ex6lUonGxka+zcGyfLCT/di9Nm8VwUzeUw0NDdne3l5ewAVfvX6DoeGxHEyiIkSfkmsnoWQzhQ3C4tU8vCzdC0omkz2pra1119TUQKq3I55gsL7NYI2wupVEIJSA+9k9jF85jkBLMb4pC7BQtweDVfls28kDH3L+Baneka1/PIF6wy/qeMbRpbyJfuFh/hZMgrzlAUGeist8B7UP+hUJif4NAAAAAElFTkSuQmCC',
printer_empty=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADwSURBVCjPpdGhSwNhHMbxX93f8WpRs/VAGXYF0SA4DgyGsaDBwdQfJoO23dzumsFgfoNcMEwss3gKr0G4uuzSC6dfy8KdDBHkU79PegT5nfw5SEiIiRnQp0dEJdCaBgmO4ZQjYi/YrU0DDXVyVVlHdDlhe7IRIqL1AU/kMww5YK0ummWM6cww5oHNd1FfUDDi44dHCgpaX3KMx3NLzqgk5xqPp4G0cTh6vHFf4rjE4QiQQ1JSLnjlruSFM1JSFpAWFksXRTmiSYN1VlmhicUyj+x/3mBRzitOWcISY5Cwv5NtVdbLLGIwzGGeTVv+/eY3onJlSlwTV9MAAAAASUVORK5CYII=',
printer_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALoSURBVDjLbVNbSJNhGH7+7z9scyqepi6JobKmE5ZigSWpYV5UskYZkRJk0lVeVN5GF96EF0kXZXRVFHWTKynqKijFQmvqTYRMcWuYodtS59zxP/T9vwdc9P08vP93eJ738L0foygKdkbb7bHLhlxdF2HQSqcM/RJQGEiSqFsNK0PjA429+GcwewVO3fmcetZbzxOqsLOs2mA0hReeNSz5EvE5rzd/9P7p5A6H7FVjWSLyLIFvlYN/jcVcmMGPFaDcZITr0D6UW/UGLtf4eC8nQ0BRw95eJAyzi99/4rBkp3H1SCFYnj3/X4H+/n4DlSBqyByrggFLU1HtPI1kMiBCx7NgOEbu7u42ZAhQcg81K9S9oKbOMUTb4CmZoxHoBBZ6CoWu0oiEZDK50tHR0aOlTQhpM5vNL5ubm4WxnwrOHDYjlqaeKFGR1VSo6qYHBeEnMBTWYsSzipsd9cLy8rJzcHDwC0dF7jY0NKC4uBgHIw9wb+B9xjXxrIz22kWYatox7r6F+oQJVus1uFwuBAKBh6qAzW63a4edTidsNluGwGbgLa1DNXLNdagqGUGptQ1FRUUahoeHKzhJkgQK7bDf79c2QqHQVoHEEAwr71BxtBNS5A1M9k6EJl5DTJ8EQ1isr68zRBRFLCwsaFCJtECIRqMUG7SDPqK46iyQmMbXp8+RnRdHVtKHec/ILodLp9NYXFzUPMZiMaiCqVQKurQfBUYeOfkx6t0HtaJS9BvKW/ow++ERopZcBIPBLQGVoA69Xg+3200rr6DRNI28E5cgxyYpN476czbIqSXojV6Yba2Y932CyiWULEciEU1ATaG6xoHjjhxU1rQgKycERQzT/mQx9cpLT8iQE16YDlhAfo2hNEcB63A4ymZnZ4WZmZmSzeou3LjQhLWJAViaroPQlmT4/SD6KpTVHdMsI1SCM1qhy7YgPzz6PeM1XhmalDjaWhc3+sBK9CXLyjbkbWz9EykZhzpXlKm/wwxDbisZJhAAAAAASUVORK5CYII=',
rainbow=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIhSURBVDjLjZPfS1NxGMbPjX9Bl/0PXqQzIYIEWWIpRFFBQwyNQqhG4VY0dUK0gXQh+KMhu2nKIAkpAu0wkBA0f0QRtoLuFgSbOmSbZ+estsf3+e6ssE7hxXNz4PM+7/d9nqMB0A6jr3Var2hJlBFZorKochhwUpQmkO65iC3/DWwP3sJO0Av59l/QI0qlmuux5buO7EMvcuM+5AInsRdqxo/5ART92j/hqMhIX7uMbOgudu+7YYRdsMaPozRZ1c/EIKwHmiM8KyptD9xEbsyHQvAYSjZozZyC+boDxbeXYKUmkF9vcHQu7QzdRn7KD/OxqwrGW1B8cx7GZheML1eVrO8R5N+5/nqzQWfC1miTgs1X7TA+eBT0bdOD5yudCCRaMPF+CEej2oEBKb6Za9ecTb0TRrIbewLPLnegd/4E2l824vSLBoQ3AjgypR2IqpJ9dAeF4cbfzgJnPnVhZLEVZ23wSsyHvkgcMf0jzvTP/RqQZlSF6D11ML6Za9OZcJuA555dQN+TOKb1JGb0z3i6kKwOsBtWZs6Miu7qYPbadCYcjCUUGJ5eQ09IJ2yKVjlgiQ1jSZgzo+K1eTC+mWvTmbB3dLEGumu344AM68mGqbdLznTntXkwvplr05nwn73hAIvdZj3V+lISDmBUyj1SdbfXdjsNKPPHYLdVPaVhLAlzZlS8tn0w06n2HFDhX8Ufg91mPdkwloQ589K2Vp0G7AOR2a7+EgKeFAAAAABJRU5ErkJggg==',
report=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIbSURBVDjLjVPPaxNREJ79Qena1EO6u/GQ9CiFouCp5FQQET0KQvBQbA/tqf+BCEXoyauCl7KFHkoOvYimUpToRTyISVtsliImpCwkLUGqxvzY3bfOvO2+bOgljx32vdn5Zr4336wUBAGUy+V7f96/3PVaDnjNKty17DkYbZ1KpVLppu/7n5nbnVDAh7NXK3Bn4/tIaFVV59R8Pm9ns9nV8aOClZhCbwDguu5QIGMMiGn8rGlamCSXy80ggxfMXAAFPPj9qXipkizLHBQtSZJEQsFg7KBgTZroZGEArWc7TSAchXIA4w+sPdQH1xAMDGQgeXD+4aNIQODZjHaRILT9Wpt/Q8wwA3X/rXVVD3glkQD3h7V/vGrA8Bvz0Rf2AK/F7zRQoY8qIAPn+TLczx/xRPF709nzPOFHayeTyfkBg29vrEkj5BkFPdlu4NtHugH4wYUSqNBaziQGE5hXifXgMVfh115RdHr90TUOIkPNBZtutwvVahUURZFlYuA4zmqzsAl/v24BFhQSRXJFDYvAlUoFUqkU+VmMwSLIyKC1W4ypwISRr9PpgG3bkMlkQNf1YRXkL6+thIlN8y9PIDGgygROp9NgGMZgqOIqEIPa0yV4sPeDgwlIne/1etBoNHhV0zTjExn+Cxh041bl3c8rSY0PCzWIgGQRCxpnSlKv1/m+3++HSaKGLV2fmp9OjN122u7JxnHrYNTf+T+76nzVPsi2lQAAAABJRU5ErkJggg==',
report_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLjZPfS5NRHMYHQn9AIeRd0za9KEMi7U4oiIZRVFSUVJvbHEyo1GXrB8XsYpDWwjKardKZ3WjKa+p0iTo1jVgs1kusWKQ3XYwhw/3+xdP7fWXvfHEXHng4nIvn+X7OczgSABKPx3N8vkODGaMCTlUFJBJJ9Ta1h8yVbrc78nV5Ed+WXZi8Wo5MJrMt8SFGo3EHwzDKKbMOn3t0mLxSjng8LlI0GkUkEhG0vr6OdDq9ESAmmIOjXr4lIBeSUzgcRiqVEhM4HuqwaG2E46IciUSC152BVRj7V3GLU5t9BW29KzBwIgohQCBY2iAYPy9HMpnkReb+uQDsnPpmA+idCaD5zV8+gKhEBOPtjVh4ocXYORmfTto8tZUzttj8uP7Kz3cQi8WyBQlGz8j4giggdxUSTaS7kzkUCmFtbS3CBRzMEzzQYv65BqOn9wrm1rcraH79B9d6/Giy/ob+5S/on7EIBAIIBoMFOliaA3OyjA+gDnKTOVxMfBnCTasSjU9OQfVIAcv7dhQVFR0WCD7e08DVpcZIXZlQIonMH1x2mAZVGGO78f2fE5ZPelyw7MeB+uJ3IgI3RzCsKBVKJAJqvIGbyPx4CuZnF2h1TmthmdahRl+SzBPcVmPWosLwsXwATafiTtytwgRrw+Y16u1GTVMJthAMHZUim83yL5EjqL1Rig5nA8xOJW82TynFBAaDQTpQtw8jlw9h8IgUfr8fPp8PLMvC6/XCZGvB2c4KPHZq+Mm005nvgL4zSSnbWXu/ardJLd+lKfR1Ky8V91XrS6KETTud6Tv/BxVpxDzfgUo/AAAAAElFTkSuQmCC',
report_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJrSURBVDjLjZNdSFNxHIZPnfS+usiborU2JaKIPvAmig0kkYSCCpHKsTnDolLXnBFJBQV9YC2VtDRThKiYzA9coW6abWSTwRixYqPtIqHTWGM7+3Bfb+d/ZEeHXjh4Lt/395x3/CkAlMPhODHzSIUpXTk+KUpAUdSRdbKThPfZ7XZ2zjaLeds0TBeLkU6n1wVfotPpCo1GY83HB3X40l0H04VixOPxPKLRKFiWFQiHw0ilUksF+QYWjFdLVxXkSnJEIhEkk8l8g/F7dZjtUmO8SopEIsFzc9AP3YAfzRzafh+0fT5oOIiFUCAYWJcMxs5Ksbi4yEPCAxYG/RxvzAz6phg09P7iC4hVnsHYXTU+d9Zi9IyEbyesvNrEBRtfeXDtpYffIBaLZdY0GD4t4QciBblPIZCL5NtJOBQKIRgMslzBwWWD1lrMtKswfGq3EG567UNDjxdXuz243PUT9S9+oP65CwzDIBAIrLGB1QJjpZgvIBvkLnO68Bv0sCn2YEJeAHPVDljbroOm6VLBYOSWCtN6JYYqxMKIBBL2vnsKp/Yo4mNPkP1uQvRtI75d2Y9nh+jHeQZ2zsBQvksYkRiQxS3VYsS4MPQngebNwH0R/j48jhEZ/XvZoEUJc5sChrLlAnKdDEe0s/MGrPz9ay3ChGxTdpXBB7kImUyG/ydyBuZz28H2KAEulNBSCHL4L9EYldMMb6DRaESDFXsxdP4w3stE8Hg8cLvdcLlccDqdmGu/ga/qEiw0i8C0FMCr2oDJykJ0lG7spMhzJtRIthy7faDojlK6VbXW0+0t29YxIqcXiDZ3+Q8f5p7zf7M8wtRUBE6BAAAAAElFTkSuQmCC',
report_disk=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKKSURBVDjLjZJLaFNBFIa/ufcmJjHa1tYoiCIiPhCfoNKVoCIuFAWFLqS+QKXo1qUg4lLBhYIgNiiCBBRBxbe0Iqjgo7WgFsWmtmbRhlhNSczNnTnj4ta0xY0Dszhwzj/f+f9R1lq6u7u3jj65cF8XcuihLFvSvev5vzOsurq6VhhjXkhQmepiyF88zObLH/5r2vO89V4mk+ltbm4+Fvt4L51sBCwEQTCpUUSw1k6q4/F4KNLS0rLEGHNeUq24aIrPOv55yXEcRKRWK6VqgjWCaM+99LSUBQkbAE7dHEIELIQDAgKc3N00vkaNYGYrrtL8etpZExCBZXPjYwLhfd9fAsAYA4Db0NAwUldXN1h5eWOnU+yhku1n4a6jHG//QkxVGRj6hV/x+fS1QG64SMyt0tEzws3n3/lW0C8nEOzFVZqRJyHB73KFjesWs2NNkruvvnNiz9JJvlztzCPS1u5kMpneXC53LP/wCqW318BarLWU/YAfoxqAx2+GaoMVDUUf8j+rKCUTPdiHi2bkUceY8wrXQmCEs21rELFYwMWS8MD3fYql6ngKzus76ekpi5jQ8VjEoeIHnM704yjQRhAj44lYSEQmEezHQVO434FSCi0W32hcJwJYFCbMXyzWCloHVKt6AsGr2+lkSiBMh9FyQER5tG2bhpYq4IAFq4SYF+XM9QEKxdIEgu0HQoK7nbiuixGLSwThN+3vjtCUmIfCIV8e5NDaczjEKPtjHmitl6/+9KAvMSOOtZDNZvE8l/mzYVZylE0r11EfnQMO/KzMITVVk4iCpxTq758+sKhxw7xkdGOuFAxe/lzoWXXw1qUlC5pW1E+pJ+ZNR0uAEkFZg6gyvQN9DBcD/gCIAXUVRPlHKwAAAABJRU5ErkJggg==',
report_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKMSURBVDjLjZNtSBNxAIcHQt+rL9GnlU2FIigo+lJRQTYGoaFRDd/Y1IL81JhzvVqaiEQZEmn2omtIZdr5Nh2lc04v3ZbOLRVS3NR8g9nci7ebk193Z16uQXjwfPw999z/7gQABIODg+eMZXJ0qsTQZyVAIBAc3SJCdnzQYrH4B0gTrGQ32jPiEQ6HtwQnUalU2wiCyOwoyUVvVS7a0+NBUVQEgUAAfr+fx+v1YnV1dV0QWWCAThoXJdiQbODz+RAKhSILdA9yYarMge5yHILBIIda64JK40I+g7LWCeUbJxQMbAUv4Av61gtaL8aBpmkOdqwxLKKWoaZrES0dDRjTXsIPohhDL8+iMk94jy9ovZ+DnmfZaEkVcXaWzXd9UVeNuc58eIZbAN9P/BquR09pojuqoOmCiDsgVrDxKMtTBJZGCkFPN2HOUIYVewPCThNsr7LpvwV3s2GskKMpeR8/vvHaifK37+B23MbaigkrE9fhHsjDqEYGR81VSE+LUiIL+gwgzsdyAvYMPNN6LNkLsEaRCIxngJqSwjuihqsmCUOmNsTExBzjC5pvydH9VIZGSSw3nh7+iDlz/p9xOijXFSw7bmKiSgLLlw+Yn5+PfgsWpqBBvBczdgKjeiWTbQQ1KUNwKg0epmT8uRgWYztmZ2ejv8TmAhm6HmehLkWI721q2HRPMNOfisCkFO4hFcark5ixjhuzh8wL/i14X3oG8DoxVpWCTvVu9DJ37X2UCMdXPRYWFrD54gsUCsUereQAGtOOoEi+HzRZAbq/HP3Fp1B/TQRrdzPsdjtsNhvMZjOsVitIklwXsCKWTNGOk3cO7SpMPb499DAjIfy5RBz+VHTim+TwzuT//c6/ASolxdSxQ5KqAAAAAElFTkSuQmCC',
report_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKGSURBVDjLjZNLSJRRFMd/3/eN6eSDGCeLHkLRy8SoTSZhUfRclVC4Ep1FLcJ9QQUSEUHQqqCCmFXELCrooWFFBRVB4WMi7YVjipqJiWOT4/fde26LL2ea2njhLi6c87+/c/7nWMYYurq69k09vtymxodQown2RD9UM7fz3ers7NygtX4lXrrQQTN25Si7rr+fU3YgEKgOxGKxDzU1Nc0FPa3RolLAgOd5OYEigjEm5x0MBn2R+vr6dVrrS1LWgIMi+fzpfz/Zto2IZN6WZWUEMwTz4q3R4jID4gcAnLk1iggY8BMEBGg5FM6WkSFY2IBjKSafPMsIiEDl8uAfAf9296cA0FrnEgS6H0RLwgZjTFbAGN71/0IAIwYjGjF+D2zbFoCsC2660LEUQxeOsD/WgzEmp+4Tt/eSmkly/mA7jpWHiKRCodD2LEHH/WjxQp9zNvnkzW+IaLQ2pPJmKA9XcCxWy8UDj8jPCxYCzl89aMRBMdHuu3D81m5MnsIVhWsplixYRcXiaqbSKSKxzZzddgfHcewMgf3mXrSkzCDa77gSl92VjWgjaNEIhuHJQaqWbeWnO01z+w7yl+r5fxE0YaMYb3uKZVmklYs2wtfxz3iiUOLhaY/kzBQbl9fy0/vFROTFoyzB67vRojIB3x3SKo3SikUl5SjRaCOMTA4QKlpMx+BLekc+MvqMxhwXbBT9pyPUtX/h8LXVpF0XV1zSymVleD1bVu7h7cAr4sNxbjQ9Ye2iSn8XlFJVm3of9s0PBTEGEokE53a0opTK7EHkwRYsO0h8IE5L7VV+9CX9sZ6d6cia0u3lRfN2DqW8weufxuP/7sOKUzzXxikYa9N10x0Mz67zb4UIk7Pj5YsYAAAAAElFTkSuQmCC',
report_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKKSURBVDjLjZPdS1NxHId31R9QF0lXmW0GRRC9eVNiL6ZIRoVCSFNzL+BNiIe1ZCzLKKRAyLIc+Y6hFcpMmUo4NXXszbUIy9p0Dp1ORGRnO2dzG5/O7wyPDQs88Fwdvs95vj9+RwRA5HA4csafyTCizsVw6RGIRKIzu+QgGT5us9mCFtME7KYxDBanIxaL7Qpeolar9+j1+pKhp0pM6pQYlKaDZdkkQqEQgsGgQCAQQDQaTQiSC0ZhKJLsEGxJtqBpGpubm8kFhholJhoVMNySIBwO81R1LkDdsYB7HKp2D1StHlAcpEIQCAVTiYKBQgkikQgPGe4YXUU7R5txFa0jq6honucFpCqpYOCRAl8a5OgvEPN2wt9fpZrn8F5XDUuXHP7pTjAME/9nQd8NMX9ARLC1CsFvacKa+S6itBkrI3ew+LWf5QQntwseyDH+Uoa+64eF4coWDyqa3Gh80wD/mBTRjQ5sfK/GmkmLuXeFqLmWokkumBqFPj+NF5AzIBKy69JnFSLr3Qj7tAiv6LDh1ILxVmOm4TwtFHzSyDD2ogy9eWnCIRK4XeGbqEPQXQvGUw7WWwlmvhjMrwJYtWeZpAIbV9CTe0g4xMCiA0tD5Vg2yhGYlYL5nQ/WXYTQzwJMPz4FKnO/ZrvgfhmMdaXoyU4I1n/o4e3lhryNCLlugpnNhq3+HJzPL2OmLgcec//Oe0AKPl5MRTweh2+4nNu5DYFvGWBmrsD+OguWviZEGRoxlgZ5hHtAUVRqZ94x9N4+jQ8XUuFyubBgUCDif4LQ7FXYXmXB2F0Pp9MJq9UKu90Ok8mUEBARoUS8N1N7IuVhmWSfjLzoUh1tcb69tD5Zm7FM5R6o+t/v/AcPwsfW2HYHkwAAAABJRU5ErkJggg==',
report_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKESURBVDjLjZNPaJJhHMc9ddulU8ei/WkQRQwanbLD1JweDDaI/q2cDhQWOjGNwXTCBk5BbeRczZZjNx1sNWaBS802CNPDOpgo2y4h4kT8O53x7X2eoUvaYYcvPO/vfX6f5/P+3vdlAWBFo1FecGYYm5q7+Pz0Clgs1s0z5iJpvhYOh4vft0P4sR2A90kX6vX6mUIhGo3m3Orq6tCn6RF8mx+B93EXKpVKS0qlEorFYjP5fB5HR0fHgFYDPzYedP4HaEAaKRQKqNVqrQYbhhGEHFJs3O/E4eEhzcvlfWiW9vGCidq1B/XiHlRMiEUT0DTYOjZYH+xEtVqlIc1L/jRcTN5/SWNxMw2Fc5cCiFWLwfqkFF9fS/BxoIPSSf49dYxpVL5N4PmbBJ1BuVz+c6rB2r0OOiACaDwKCTmRPDtpzuVyyGazRQbQc2IwIUFwdhhrovZm89i7PSgWkhidT0DuiEM29wuyVz+RTqeRyWROmcGWH25hO7xeL8xmM5xOJ4xGI2KxGA4ODpo1k8kEl8uFtra2O02DD+PMl2h5Bq3gFqxWK1KpFJLJJAKBACwWCxwOB+LxOK2FQiG6h81mL7UYLMzNQq0YRWRnB1NTUxAKhZBIJLDZbNBqtXQtEAig1+spRKlUFk4MtGKMPeLBoJugADIwHo8Hn8+HSCSCYDAIj8cDLpdLXyMBMKBqi8HMtAFyiZgCdDod+vr6wOFwIBKJaMia1BoGUqk0Tw1UKtWl5f6rcAz04GE/hyqT01ZWVmC326FQKMDso2tSc7vddAZ8Pn+XRX5nkqGO87fHb1yYHLx+Wc+o/xaLxWWZTFaVy+U1Zli63t5eOXNdIjVyj+zp7u7m/wVntrWV38u6ZgAAAABJRU5ErkJggg==',
report_magnify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ0SURBVDjLjZPLa1NREIe/+0jaNE0VTRvBF1YQwRciUgpCQXwt3AhiVz66ceXSpYL4Nyi4kQiuruLGR32gVBeKithWUatUW6pR21KjJmma3HNmXFy9tQrigbMYmPmdb35zxlFVBgYGdpVun75upgqY8RF25Ic6+L8z4fT396+31j6QcCbtYZk8c5htZ1/8V7Xv+x1+EARDnZ2dRxpf9uabFwIKYRjOSRQRVHVOnEqlIpHu7u7V1tpT0rYfD8P3e31/veS6LiISx47jxIIxQfJZbz7TpiBRAsDJS+OIgEJUICDAib3Z2TZigtb9eI7h2527sYAIrFma+ikQ3cHRCgDW2rkE/uC1fEtWUdVZAVWej04jgIqiYhGNPHBdV/4gOIDnGIq378Y9ntyX49GbCSbLNcq1BhoSSealBBtWMSZRBewswdOr+UxrxKmqiAgX7n9gwfwWNqzI4ic8Jr7VKZZqPH5bYlN7SxrwfiM4iIeheKsvdroxlaY910Sx5mCqFnU8FrdleDFcoVAo4Hme6wZBMFQoFI6M956j/OQ8YolHVK7WSSYTGCuIKDN1oW7AqE8ul8NaK78RHMLFMHW9LzaxqSFBPbSEFoxEourAdKVCNrty7hTch5fzzW0CdvYDLco4lKqWTCpJzSq+6/DpS4hOvebT4Luo1XgX6jNpF8Po8R723BpGRPharjP8ucKHokEUSpUa4fgrtrR+ZGbsMRev3DzmB0EwZIxZt/HVjXdNC1KowsjICMYYjDGkRViV1ug7NwtfS2NMvx+kdflaujaPHXN+GdazamHXsubk1kIlfH/2zdSzf23h0d1Ltu/s6jiuaht+AFR4fcZEARLxAAAAAElFTkSuQmCC',
report_picture=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJvSURBVDjLjdM/TFNRFMfx733vtdBa0QiUqFEHI3FRw4TooDGGuOpCHPy3sOji4OCgicTRERMHSWcGB/wDhmhwMEZDIkj8g0YFqmCtYJFi7Xv3z3GoUjAO3O0m59587j2/o0SE0dHRI8UH1wfM3DTm6wTtmfFWVrfyamRkZLe19onT5TU+lm83Ojnc82pVp4MgaA16e3vH29raztW+7s+k6gEBrfWKQuccIrJin0gkAPhHYMh3d3Lw5gsGn31kcrpIfkETBD6httQEfvVW5XBzr7uWBPGx/szatIADpRSTX4ocb99OMpn6L38x1NzsU5eDjo6Ondbabtd4Al8Zfjx8hFKKwqIjmUwxPrPA46kLrKubpRRpsvl69m26yt6dm1H4LAmCF/cydQ2CiKCUItIWAxjnSKXyJGIbcIS4+BRfCyWcOCJjWCY4ia8MhQcVQTzmIwLFUkShXMbKTxZ1yPdSmQIhSKWmKnh+N7O2UUBk6cdFoFh2TM3swNW8ZFGXmC9so7mpptrKquAUPobC4BAAobY4B/O/LHuaLtIydZ5X77LMH7iG0QYBIu2qAm/4TqYuLThLRSCCQhj7fBatNcNRRLQxRvSyA601x/cPI7JCcBoPw9zAEEop1qdiiI24cqyPPw9iWZbQWlMb85YJnt7OpNIObKWgecs6bg194MtsiSAeEEWWeNwHqQbJzL7tqiYxKq/xMExeOsPRwfc457DWYowhDENyuRwNDQ2k0+lqGJWqzIIxZlfLm/sfkxsSiMDExATGGIwxS3PgeR65XI5sNovneURRVLnkb8vONNcf2JqKH5r+qT/1vJsbW+04/wbW84QVgHdG+gAAAABJRU5ErkJggg==',
report_user=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKjSURBVDjLlZNbSJNhGMfnTZddJHS4iFJzdhFSRAc6WClJNmbMZrA+0x0+dQesXGpTnDo3aylmSToSgqiMqNTmFC1aOVMHuSG6DS22SK8aMQPZwanj375vMPuoC3vhz3P1/Pi9z/u8LACsqampc6MtJD6ocvBOtBcsFuvwBrObak632Wz+z9Yx2K0WDBelYW1tbUOhISqVapPRaBS+vV2K8a5SDBemIRQKMRIIBOD3++NZWlrC6upqDMA0GMEQwWY0+3w+tKvL0MLZCm3ONqiILHyZm8PKygrTYEhbirGHJRgSsLG8vEynpnselZUN0HN3QHM+EdpoLTu5GdcLL6wD4gYTMYPBS2yEw2E6qqfzqMo7gTtkBh5X5qI8exeq+ftBZiYjGAwwDQYbS/CpsxgD+ak0nUrVk++olpHwOYwIzprw09KBXy4TepoKooBg5J8G/Xmp9IAoAHWNtvudGDdIEXC+QGj2DTwmHWqUCiwuLvqjgIPrBvXFGH1Aop+3J95M1j8HJzcPdTo9tEoh2m4Kobh6A8VSOe62tiIhIeEI02BiBMbcFBpgNpuh092CRCIBn38Rhq5HGBh+Dy6XC5FIBJlcgaPHjhviBqZaEpZ2Cfo4KfQAv907A8szHdSNeiiV5cjn88Hj8VBQQKBILEW3oQme1tMRhoEtatCbk0wbeAfq8bKWi8tiBfR6PTQaDQiCQHNzMwiRFGpxNjwdXM+6QbUEH9tE6M2OAcIhP34sfIW6oQlyuYy+ikAgoGuJ4hoW3C5kpO88+5fB66wkRCIR+iWoQVKrS22jy+WC1+vFnye+BxUVFUndnH3ou3IIrzKT4Ha7MRddV6fTiZmZGUxPT8PhcNB1cnISdrsdVqs1BqBAVISpW07VHdiukbATyf/5zr8BNamLpjmUSloAAAAASUVORK5CYII=',
report_word=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJtSURBVDjLjdNNSJRBHMfx7zzP7uKTGtLqSoeCCEKigiBQI0qkpEuHTp4svXjyYp2COhedDTok9g577aAlhtUhPBi+RLVlaBhr69uumuvus8/M/DusbikdnNvAzPDhP7+fEhHGx8cv/h66N6CXk+j5GVr6EvXsbi2osbGxE8aY9zbIl7sYFu93cr73065uh0Kh+lA8Hk80NjZ2lX3u76uIAgJBEGw7aK1FRLbtPc8DYIdAs9DTSdODCbqfTxMWIZfP4UU8rBEwYA1YbdGBsDa9vlISRCb7+ypjAhaUUoSAu211hFznv/z0aoGuG6OEWltb64wxPbamDVdpVl+/QSkFRgi7DjcffmU1bzga85iay1EZdqmqcJmY3gDAicfjiWQy2bU49JiNsWeICEopxAoC5H0hIooDtR6ZrKWmKkTeN0hhc5B/BVdwlSYzVBTI5hzLgKwvjEysEBQsc/M+6Yyh4Vg5796u/yN49Yjsh6cggohgChaAwBd+bRiW0gFVSjE1Z3CU0HK6dqfgKi6azOAwABIICGRzFscXLpyJMpFYY81Yzp7aS7Qqsn0G8/0PWR99gjUUBQE4LpiC5VA0zKWm/QhQKZrmhhiuo3YK2nHQLA8Mo5QiyFuW0j53uutAIJfXXG8/XPrGxbRffGArB87Ii76KmAWzmbbAcu32R3LrObwyD2tBrEUExArWKtS2JBby5Q6aH7c6uDz4HWstxhi01vi+TyqVorq6mlgsVlIopYpd0FofP/nl5fSefR4iMDMzg9YarXWpB47jkEqlmJ2dxXEcCoViENRWSTqORM8drIg0J7PBz95vy5O7rfMfwdWGR/X6rdQAAAAASUVORK5CYII=',
resultset_first=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGcSURBVDjLxVM7SEJRGP5vmVfpAWGFERJUONQWLtEQOUQRbUK09IDW1mp1KKPBsdGlWqKIqGiLoq0aggqxraFARMpX95535xxCNLouDh04cM7//3yvwzGEEFDPaoA6l6vyEj3MaDmcc0CMX23NdY9X9iPxlxijsHK8GmxxUMAh1O+BAb8bEKZVnZntVMyisFYiotnRgkU4UMaByC0Hy/XpzWQ80GGuTwy1GUULO1tQrIQJQBLIsriuTW4kE71d3qWg3wNM9gqIOYeoWDHlGuALYQhHn/cCPnNpsMeEjyIBW9bzP8B/KlCsSAJ4TQNyiI0M97WaPe1NkM4TyJco+FpcQC3irECxYsVuM/iyaBFTplltzHU+6sx/KagCKCDlX0A6R5TXp5O77NHl4wd43QYYoOwJEAg7A6iEEZHsMijpVbzuhCLPqdzBzcMnNHsadT7CZrUUMB2gjVnZ6/v+6GzqMbt7e5/V+fBaIaqEbfUKEqByMH0enu8cu+AE0QVBqDOAYk2cvUmfUiqrHsxcTy36QqcE3K7lyrrx77/xG/TSBY2ALCinAAAAAElFTkSuQmCC',
resultset_last=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGeSURBVDjLxZO/axRRFIW/N7o7CZIIKhosAoIIopWNqEX0L7ARG0EU0V4UVLBJJVjY2FgsgiykCOY/SIw2tgHJqEELUUzQJpudcd/c9+M+ixBYhNkmhae7X3E498AxKSV2o4xdau/wMfv6V6XK89mrRx4N8zud78ve6UXxivjAwv0TpinBvqg8uNf98WQYWqucmh7j0ukJSknNL1ivnDmWm8OT7Ye3Xnx7tsMH4hAfcV6prGs2EBdQYPpgm6n97buXn669BCglYV1CvFJKbO7gj0+4oPRt5PhUi02b3zz/+EMbDLUbo24Z+labDaxVJCi9QSAzcPLoOC24tlj0ZHI8Q3wkWN9sMBCH80rtInsMZCZRB6USrco65hmgoxKUkhCfIEGeG94WW6x87S+QOPR7080cmGiRZESJld1uO28Zllb7rKz15n92zl5Rq0kk4oKS6jgqQUS88q7oUXwpu+vdc9cBkjiC2+5n5At9qywXPT593nq1Pnfhxg5PdcR7xUcl+dBsEKzn42rZ2ZifuT3M1Srv32yQgvLv+Mx/X+Nfknb5JPA+HGwAAAAASUVORK5CYII=',
resultset_next=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEdSURBVDjLY/j//z8DJZiB6gY0rH7xpW7li3YKDHj1v2bli38lix61k2VA5fJn/9eeeP+/fcOL/wlT7/aRbEDegkf/Vxx/93/xobf/S5c8/u/ecm0eSQYkTX/4f+HBN/8nbX/xf+bul/8Tp9/9r1N0dgnRBgT33QZqfPW/YdXj/42rH//v2vjkv3fHtf9SScceEWWAc8u1/xO2Pv9fsvjB//IlD4CGPPrvXH/5v2Tksc1EGWBaful/+/on/4sW3gfGxsP/9lUX/ksEH1gj6rqdhSgDlPPO/q9b8fB/5bIH/23LL/wXD9i7kqRAlEo6+b908f3/NiXn/4t57V1EcjRKRB75b1145r+o684FZCUkMb8D/0Uct88euMxEKgYA7Ojrv4CgE7EAAAAASUVORK5CYII=',
resultset_previous=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEXSURBVDjLY/j//z8DJZiBLgZkz37Ynjrz4ReyDEideb89afrDf5ET7v4n2YCEqXf7qpY9/T9r76v/Xu03STMgasLteaVLHv+fufvl/6k7X/y3qrlCvAHBvTeXFC54ANbctv7p/95Nz/5rFZ0nzoCAzpuPsuc++D91x4v/jasf/y9aeP9/89rH/6VTTxJngGPDtc3xU+/879789H/5kgf/02fd+V+17OF/yZhjxBmgVXCaRT3v7BqP1mv/a1Y+/J824/b/woX3/osHHSAtECVjjqy0Lb/wP2/+3f+Zs+/8F3XfS3o0inntXWSeffJ/0tRb/0Ucdv4nKyEJW25ZYBh/5L+w5fb/ZCdlQYMNs4WMt/wfuMyEDwMA0Irn/pDRT58AAAAASUVORK5CYII=',
rosette=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIzSURBVDjLhZNLbxJRGIZP22la6KLpou1GN11QKGW4DFTH1phorZcYi8R6w2ooAlUSjUStC39A49I/4sK9f8OFLoyCwZAAgsMwc87x9RuoSXWgLt7MvGfO+3zfnAsDwAaJT0094+PjlojHTc7YpWHz/g4x5hELCx16Qm5eg9zOQGZzEMeOQyQSsLze8v8Ab2TqOuTT55C5R5CPS5ClPYhsAfJJCXY0htbkZH4ggFp+LYIq5I00RDoDQz+DRlxHM3YSxsoq+K27EDt5WDMzlsmY5gIIn0/Il6+ocpECazDip2BrOuzYCViRBMzoCgT9El+/gEYgUHN3MDLSlMktCFKTQk5QRCgUjoOHYugsRyC3bqObvodv09NlF8DWtKYsvYDM5NCmijycgFA1iOUoeDAMM6BCXEmiq6+ioihv3YC5OdlbtJ0cOlHqgKryYAR8SYXtD/UAMr+LzvZ9VP3+7y6AMT//rq1R1UIRfOMyTAKYTnAxSOEQOLXe2kziq28RHxnbG7iNrdnZ991kCuJmGnK3CJmiNblKPvOAdiCHaiSKT4pybug5qDO2VB4bQ23tNIyNi+CFhzDObqC+fh5lVcXnQIAfeZB68nqloSiojI7ii67jA73XJibQIG8zJo8GeDxZUAeShi3ST+fzP/4HY9nhAMYscTC5Q2oxViFfOeQdgDUQ8IuxO04l+wBg9kP75PcP+V4XbZrrAlC4TmpRxarTBV0sy3ZuZ18W78OrpBYB639yvwHcmnk0jmx5QwAAAABJRU5ErkJggg==',
rss=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGkSURBVDjLxZPPi81RGMY/59zvN1chC0lSViZJzJQk/4BYTWNBkyhZsrDCgprl7JQFpSxsbEZdG5qFJjvKRLFTU4MsrGjouvf9ZXHOnTKyugtvnc6PzvnU8zzvSRHBOJUZs8YGNCxM/UODgziogDrIENTKmQHDgAE0AOyb3fDW/xzmYFbWYnWt8OJRBQCsfQF3wuoFVcIURAmVshchVEGFdud+6McIEBABR26Q2q3E23uEGfnwJdA+9uwq+eRtUttFl+aJT29ApAJUi7ZwaLcRL+dIx26RbIC/vgvm5KOXQX4iT67QOXQWXV1eB2RECsADcgNb9oIr8f0z+fh18tRF/MMiaccEzen7+MdX4EUigyDT/1X0uQOQDl4gVhZhuIY9niUI8oEZ5MEp/H2P5sRcMVOtAgZaaO7ggi/fIU1Mw4+vdGYekrrbsXcLtOd7dCbPoE9vVoCCQIr53cHkOYJvxXVVQorTIdX9UQqqYEaYsWnzHug9p6EfIEKyLmkUoVnJPqq52aFjkAIaL4nVSnFtVyCx3llYwHDDrJTu87+bNv333/gbANMZYUMccT8AAAAASUVORK5CYII=',
rss_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIbSURBVDjLpZNfZNVhGMc/7/v7nTq1P9Vkso0kJy1ZjZWZ6CKTlMW6qYmUFNkuqpslYnSxXYwolXURXdRNTRszRU2aWTRbaTctUs1pSjSn/M7vvH+eLn6nsYixh8f7PK/3+fD9eh8lIiwnNMuMZQNCHtb/R4MH48EasB5MAaxL7hxQEIghBCDT9s+sX5zOg3NJbVyxtvDiQREAkMuC94grPrAWcRaMRaxJemMQa8EaUpVbIZK/AAERaLiESpUhU7cR59A7ToONcMPn0QeuoVJp7EgP8mUSjCkCrE20iYdUOTLehWq8gnIx/vUtcB69ux3Mb8xAB0HdUeyniQWAxpgE4AV0CKUbwVtkfhbd1ImuP4WfeYJav4XwyB3851fgE4nEQkiUT/R5jwLU9hPIhyEorcY9akPv70Vva2W4r5URX8J89AsTaZrjrxyLhZDYJjTvwRv8xHX0rgvI+yGC1nvgDIODXUxVrGNvbQM1FRlGph8z8O4l38sDlPRUCTuPI/xMXLcWMYnTYhL3z87lOXzwEASaltoOep+dIUDT3z+IJhIwBpVPo/Or0PFqAlNCYMsJZS2hVPAt94OUKqWltgOAi/v62FxZR0FBSCTw/P7Cz8IJFBafueo009kx3mRH6Wy+S/fTk6SDlawQUEvZxsb2qqs1m9ZcbqrbQ2ZDPTNzk4y9HWX243yvWuo6N7ZXdQPngDIgB9wcv5Ht/APUTCZid72ZzgAAAABJRU5ErkJggg==',
rss_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAILSURBVDjLpZPNS5VBFIefmXduKVcrKKKSFBdJVJSCSfQPSC1tIxIEUQR9Qa6UQHQTuAiCFhbtWtSmoAKJJMpAqSBJ2gh9UaYXw4jU9N73zsdpMVfDIBDugWHODDMP/H7nHCUilBOaMqNsgOFe0380BLABnAUXwBbB+XjngaJACgaAXR3//A2rlw/gfcytL+UOXtwtAQAWchAC4ksPnEO8A+sQZ+PZWsQ5cJbM1t2Ql2WAgAg0d6My1cj4DcR79IFT4PL4x5fQR66hMhW45/3It7dgbQngXNQmATIbkFd9qEM9KJ8S3gyAD+iW82AXsQ8vkOxvx30dWwForI2AIKANVNVBcMjcFPpwF7rpJOHDE9SWBsyxW4TJ1xCiRFLBkC9EfSGgALXvBPJxEKpq8Pc70K1X0XvamOxpIfd5lsWZHCZrqG34TX0qGFIXaSFAsISx6+iDncj7QZK22+AtkzfPMZdW0thxkfX1e8m/G2Ji5CmmUERJ/w6h8TjCr+i6c4iNTouN7r98MEzz6T4qPw3D9Ahs3MQPU8fE0CiGvIC1KF+BWi6h97H2Es0tzheo2FYPRzv/dmDvdopBSoBnd1Y6Cy9QXL1njGJpfJDso7Ok+RmWgIX5BJNo1FqmcaS9pndddfbyzs3OGD3Fwqzjy/fE24L0qLWO82h7bdfiz9yZxKs6n8i0wEDrkLvyB9KrOCqdUGipAAAAAElFTkSuQmCC',
rss_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAINSURBVDjLpZPNS1VhEIefc7wXTNQopKTM4IKRaB+GliQtKgpau3NjRcsIzEW1EQIR2kRUUP9AtHFhi4rAqKBCUDOEkDIQPzIIWpjX7jnvO+9Mi3uvUBAIDgzMDDMP/JiZyMzYjMVs0jYNyDDc9h8NCl5BPIiCdyChWAuAM0ghA0BTzz+z+rcHhRCKsQ+lWODN4xIAYHUZVLFQahDBgoAXTHwx9x4TAfFkd+yHgpUBBmbQfoMoW4N9fIiFQHzoEkiB8LyP+Nwdomwl8uoWtjgF3pcAIkVtppCtxcZuEnUOEIUUnXgAQYmPXqb/WQ/5NM/dg/3E85PrgBjviwA1iDNQvRdUsJUl4uPXidsuorMvSILQUNfM+fFBnAqIQGpkKCRFfapEQNTai319St/UfUwdzgJOhF3bmmiuP8Zqskb30hgjfjdbUiNDWqKpgnp08h5xx1Vk8jZnWnoJpgQNKMbyyiIHGrrIuwInF0YY1SoyFAxSDyg6egVECE8ukIgjmDL/cxavgqjHB8+vdJXDe06Q97/pOPK2BPCeKFQSlVcYAokkSBB21jYiGgimfF9ZYHt1PR8W3zEzN87w9FYiu1ZveFu/LIKBg+6uahLncOpIxJGra6Ezd5aJhfdMfxrl0UwN+5IKoo1+Y+tgvNbeeKrq9ZeX35xyennIPgMbB+QGorWgFVUFDbkfQzZXrv8BKB9RM6Fzq8AAAAAASUVORK5CYII=',
rss_valid=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAImSURBVDjLnZNNiI1hFMd/73vvzSXjKwzmRopJyFcaYmczPko+8pE0SpYsrLAglnbKAvlYSNkYjYhmSrIRRRQbKblc07Cii3Hf82Xx3jsxG/Kvp9NTz/mdc/6nJ4kI/kfdl9ftENUL6b887rm9s7K7d+va35I3i+r1jvaZk5O/ddBze+e8LJNFWZbdzES7RaQsqn0zpk0vVGu1+l87yDKZnGVZb2VaRyoiA6J6a05HpVCt1erAiuTOwOYA8AjcHHPHzVA3LFNMhHea8rI8hYltbXz78YOhwY+s1++0qVPc9PU9zN/zZ1n3P485j7584mT1FQUPrs3tolIswcPrFEeS6oPgTpiBGagSpiBKqLBKlUvjJoAa7UOvYfoCGI4WICACVh4jKbURL84TZqRLD4AOY/cOk244w+xSGX1wmvjwHESaAFVQg3AoTSAenyJZfYLEGvjTc2BO2nUQ5Dty6xCFJbvR6rMRQIpIDvCAtAjj54Ar8bVGuuYo6fL9+Jt+kqmdFLdfxN8/Ac9HpBGkDP/M53UHIFm8j3jbD1kd691DEKQLtyFXNuKv+ih2n8qNVWsCGprT3MEFf3aWpHMLfPtMYdtVkvIk7OUNSj19FJbtQu8ebwIUBJI4PStYtpfgS+66KiEKKoRIftdWVDAjzBgztgJ99ykyHCBCYmWS1grNwByiaW7qUDBIAoqeb6ypJI7MCCQgC2gAFpCNigoYudGjlPzvd27pFy1NYQEtEp0PAAAAAElFTkSuQmCC',
ruby=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHiSURBVDjLzZPdS5NRHMf3Fwy67yYqepnzrSgJetEyl8s0GqvYEHLDFiGr4VwQZTcOpzfeVgQiCIJ4IWJUN+WNUhQqqAVjpg0c4vY8p7l89uLzfDpbV0FE4Y0HPpff7znf3+97TIBpJ5h2h4HxoNMlSUlEET0YEHrAL7Y77orCndsi7/WIXGuryN64KbRrDrF1uTmZuXjJXTKQAqckw+tXMD0N8/MQjcLSEvq7tySHh1GGhkgPDqI+fUZ8YID44242z5zPpGtONxcNPvLiOUxMwOgoLC7C3ByMjaEHg8R9Pta8XhSPhw23m09NTXzp7SXqv4+oOhkvGlQYXYEPRiQM8jZWVymdqSl0h4M1u531xka+S5br6vjc08Oytx1RcSKhHqlylmYg8+6ReV/qT7phfBwSCZidxXC5SEqhKk1Um41YOMzXW15E+fGYcriy8rctFHzt5nxb2+R2KAQzM7CwgCGfLurryTQ0sNHXx4oUq2XHYsqhcusf15h1uc2a8/pkviv0y0DOIFNbS7a/n28dflRL9bpy0Gr9aw+2rrSYf9jsb3IPH2GMjFCIRFDuBZB5NeVAWcs/FWnz3IW96VNno7lAJ5oUy7xaar/F+V9NFNU1Fpn3vcy7ktp39Oou/ws74Sc149q/X6rjygAAAABJRU5ErkJggg==',
ruby_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJFSURBVDjLzZLdS9NRGMf3Fwy6jLrPKKerSAsqXzKn0xY4ZrIhbvulKwtrOCdYuWDOza02Ai96AxEEaXixTMkllDdKUjhrrmDMlMFGuLbTXO7N/b6d3+rCYFLQTQ98eTiH83yel/PwAPD+Rbz/A8D2dsupvlIRTjmdluS0XWT7WifJXu4gGUZN0q2tJHWxhSSbpGSrQRJJnKtT5AE0QEaVwMwLYH4eWF4G/H7A50Pu9StExsYQHR1FfGQEsQcPEXQ4ELzdj83T1Yl4+SkJB3iLJ4+AyUnA6QRWVgCPB5iYQE6nQ1CjQYhhEFWrsaFQ4F1jIz6ZzfB33QARlgU5QAnbo11kLSaAZsP6OvI2N4ecVIqQWIwv9fX4RrVaVYWPAwNYZdpBSo6HYweFsvwMaL97aL/TOUM/4HIB4TCwtARWLkeEBsYoJCYSIWAy4bOSAREcC0SLSkt/+4Wspp2fUammtvV6YGEB8HrB0tJJTQ0StbXYGBrCGg2OHT4aiB4QFBf8xpRcwU/KmqcyPfqfADqDRGUlUlYrnhoYdNtlbPs9CVqMFfG6XsHNgnuwdf4C/7tI7E733QI7Po6sxQKnQYk7TiWee4fhCblhf3kFzfZilHXutRVcjs2Ks/vjJ8/409puJK9roTJWw/XBAZfvfn6+ttlLsM92cIDkrhtGjpQfov2+of2uNfQJMe19jJ327P0wB/i7dT1xdV/S6lZh0N2WDx6caftzBTtFHxqbbEW462bymTnPnXedwS4QM1WcK/uXN3P3PwAfNsr5/6zP/QAAAABJRU5ErkJggg==',
ruby_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJSSURBVDjLzZNdSNNhFMYXs4tuBLuKqAsvoqJMrSgJKr+WH2sZJFpsCLk/thAxhzorSm8cfhBIdxWCCIIgBiKGSpFSKVaihlo05kcLpbTtTaebbvv/emddFBgI3fTCc3Pgec55znleDaD5F2j+DwG1otQo8U1ChBEqs4qQtVgEiwpF4Po1sa6YxVpenvBfviJ8l7LF6vmsRe+5DNOGgCTkSHjp6YaBARgbA4cDJicJ9T1nsaUFd3MzS01NeB48xNXQgOtuJctnUrxLCaezwgJvaXwEnZ3Q1gYTEzA6Cu3thMrKcFkszCkKbrOZBZOJYYOBDzU1OIpLEPEnXWGBWLXc+lqttYPsxuwsG6+/n1B2NnN6PV8yM/kuMZWczPvqaqaUAkTsiXnPwficjR1Iv1HS75NQVSV0dMD8PIyMoBqNLEqiR4p40tNx2u1MX1UQR4473Qfi4v64QsBSELmen98VtNlgcBDGx1Hl6EKnw5uWxkJdHTOS7Dl8zOnefyRm0zP6jaZIX05u13q57aeA3IE3KQl/fT3j5gxeZewMPk2OUHt02k8SFZvmYPXCxciVdH3v2u07qK2tBGpr+ajoeVeSgK/rHupkNyutVt4UxgR7UyKKNg3HcmLqnqVTZx1r1lJ8N6z0Je5QVyWZ+wawRYE9WlpKpCdVO/XXhImjCYek3yHpdyY8tjr8mN+fp3IX4fqW4ir9uryNZpAkf7kGt8SsRYusf96SwIvc3VVDyr7AnC2arze341S28cwQEezWaW9t+dO8zN1bITtO/7qCK0wO138A9zXJPLDAEBEAAAAASUVORK5CYII=',
ruby_gear=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJeSURBVDjLzVNbSJNRHBd660Go597qJQrEl4qgCIoopyk0ZstN0G22EZu1dAuEbA/Otb0Iu+jciinsaUiI7uJgF4cojkIdXtYG0TbYFuk86Ly0+e3X+QY9CCZEL33we/nOOb//+V1ODYCaf0HN/0FQed3TRrFFQVgwvUrCKLvJkfw5KcuekZJYRH62t5PDJ3xy8JhL9hubN4v3HwqqBPQAj6KIGR8wPw+srADJJLC+DiYcwqbTicL4OHYcDmxbR5EZGkLmTT92b98t7ty41cwSfMJ7GzA1BbhcwNoasLwMTEyA6e1FRipFVixGQSTCD4EAn5uaENfpkOx+CVJ/PcMS1FVUymjlnRag05BKofrNzoLhcpHlcPC9oQF+txvT09PYGBjAV3EXSN213Pblel7VA6r3HNXrYd72A5OTQC4HLC3hm8GAgNeLTEcHwuEwgsEgYj0qfBy1lV1mi+tYCmVpV22ps9N9pFYDCwvA6ir8fj8SiQRCoRC1ZB3xeBw+n48urUJHZWi12rPHIjlsE9Qe8FrdJZW6SrAxPIxYLEatcUGj0UCv11dlRKNRGI3GDyf2YP9RS+3eA47/98Z8Ps9OAtV7ULh0tcViseyl02l4PB4MDg6eXI7dO/cuOKmhXqo/RU212+3YuniFx66ZzWayRpOyWq2QSCR/btgX3tObsy9eZRcXFxEIBDAyMrJnMpmIm6YRiUQgl8vtlOD8qTV10PKwBOxtWDnswbGxMczNzbEEUCgUZ04loBptMpkMfX19jeO0jQYaK53aKhQKGT6fP/NXj0mpVEIgYOt//P8vZQLgm35VBV4AAAAASUVORK5CYII=',
ruby_get=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJGSURBVDjLzZNvLNRxHMc96qGt5z3uH3FIrEUlItel5VLtbjbcdJhwOBoiRpy1zhNNVsymKbNl0mit0hbTH0dDtRul21iju19cjuN+r760tbWVZT3pu70ffl7f9+fzeX+8AK9/kdf/AZAL8zRCX4SkNXnyDZLHkCWtZmZIK2nnJbcuRVpOTJSWzp6TXPFqafF43Jzz6DHtOkAUJAg56e2B/n4YGQGrFcbH8Tx9wlxrK/aWFuabm3E03MBmNmO7VMpC+BHnfGhY3BrgFTcboasL2tthbAyGh6GjA09+Pja9nmmdDntKCrNaLa9VKt5VV2PNykEKCLGtAfxlo+GFXFMF4jemplh/fX141GqmlUo+x8byVWgyIoK3lZVM6lKR/PfNOHYFJKzPQPS7VfT7wFNWCp2dMDMDFguyRsOcKHQIiCMmhomqKj4k6ZD89k7YdyoUv2xhRZ/q7U5O7l4tKICBARgdRRbWpagonNHRzJpMfBTFDt+gCfsOvz2/XeOSRuvtSjjT7TYW/ACIGTgPH2aptpZPmVmUN6rd+junhzbMweKJk97fYpQPl4tKkNvaWKmpwZ5toKRRLRc/zqWw9wLxt5SWDYO0cChy2/z+g9ZlQx5OgwFjU7xc9CiHpjf1NFjqSLubRGRt2PvwipAtf0yYFBjqI/odrLimciXePuXOvZ/O9SEzdS9NZN9L50BZsCv4oqLnryOrqo/CPFiNqb+c2KuRBBn9Qjd1C8IyV55fpvxZMYF5vmz6mIRlhGUUOT7sztj+E/Ad9GPLsXC6ErkAAAAASUVORK5CYII=',
ruby_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJiSURBVDjLpZPJS5RxGMc/77wzlTM6pthuGhMSaVhBQaAdzOzQoShoAYug/oEgizoUHcJbENEhCDoFXYLo0q5tOpm26CXRcslRMffRGd93fmuHCKKNwA88PPA9PJfn83WstcyHAPNk3geCAPbc6VrgKuACWG1Aa6xUWKW+byG/j5TYjFRGiJORR/dvOeZs3UHgplO1M0JOFCIRCIdBSszoV6aGhgloTdAYdEaQ8uZgcorFz5vSVojaIHDGKVkXQUgYGoLSUkinoacHWlrwZmdxlWKRtehMhtFkkkhFBf7m8siSl/FrQeCE7eq8wfjYVqewGLKzoagIkkno68P1PFxrcYFp3ydcXc2Cz73kt7WPWCFPuRdfNX+1Dx/csdPJcjyvxHEcyM0FKaG1FeF5hByHiwcmadyYorJjIQVt7b1GyD15nR8aAgCBy1emrJSH9UDivo7Hob8fQiHIyiIoJSGl8K2isGA957fGSWR7h/I6P7QDOD+LlDlyNGqlvO2uWbP79MY41giEVgitWZlfQnlhJa39jXQMthhfidyXdZMp51cTvb37olbIO2dqx2tqyo6hrUEbjcEykhwkL7KU171PeTfwwvhK5Ad/FSPr3t2ZVFXNcV+JhLaGLxOfkEahjERqyUxmlk2rt5OSc4E3fU0TfzQx+9mTQV/5KK1YFi1iebSYFbkxQm4WS3JW8T7RTHNP09icYIvztzLtvx4zvhAII/CVYG1BmbMttou3A3Eauh+PCEX1cL396PxvGzdcCqS3FO0IP+9uGBKG6uF62/XbF/5F7IKT1sYNe0bHRutt34/8G0frVWL/2YcDAAAAAElFTkSuQmCC',
ruby_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJuSURBVBgZpcH7a05xHMDx9/ecI/OcXWzZRDPCqMlcFqklKZeSwi9spIRcSwrDL5K/QFJLSoTIJX4yJOQ6lkSYjfV4MGwe2549l7Pn+z3nfIz8INlPe72UiDAYFoNkMUgO/WT/njXAEcCmnwQhBAFifMT3EeMj2iDaIMYgWeOHWu90b1w7o8J9u1cCJ9T8BS55+eC6EImAMYSdHXS3f8EKApwwJMhqUl4GuroZfvdBWrRe4wB1qnyyizbQ3g4VFZBOQ1sbNDbiJZPYvk+OCEE2S2cigVtdTd+MSrf43qOjDrBBWpqPE/8+S5WOhdxcKCuDRAKiUWzPwxbBVtA1qZuixaMJXj1mxN2v30SbXfbB+w875HrDJelJVOJ55UopKCgAY+DpU7TnMUQp0mN6yF07m5K5Wwj8J2Hc+7yvtOHlSSUi/OJv3Zwv2j9nFRcvsZcvh7w85PBhemMxsuP7UKvHUThtEenPbfjJkMT7l5mWphcbLf5w6o/1hllda6IfrpkrV/mtsBDH9/GrRlIwdSlB5h05I0bjRGDUvOmRCRVl9UpE+Ju3bEW+aHPJqZq5cMjUKQSxGN+CRgpXzcGyYyhrGBLEwWRoPvsppUSEf6XmLywVbe4wb/LEVHkH5ASSW1WkHKsHZeURGs3b8y19r9/2rlciwv98OFS1V7lDD5SsXBcRfRMVJHnTkEjbyTBlK8Xr1s7tNaeilx0GYFWU1Iysro3oeD2OU0Dz7XTmWVN8x9g+72KOa1Nz6mOSfg4DsGy0stuxhxbz5saP1Ivnnds2nXl/mn84DOBLNH6rN3FnXKYr7bW2fq9bf+LdBf7jJ8/eN9kzWRDgAAAAAElFTkSuQmCC',
ruby_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKRSURBVDjLzZHfS5NRGMeFoIsCoX/CEpq6GeWNGypO24/WpuliQ0hf3dyQta3lvGkibthgpF4kFYkI3igKIoleSCmiEDUVf5ZMk4Eazr0nXU3n3n07Z3eL7rrpwBcenuf7fM7znJMFIOtflPV/AFLuJwaqYyrCJLgcRHDYSLLFSi6aTSTBNZDzujpypn9I4lXV5JdKE4nJ7xrTANpQQxXD9BSwsACsrADb28DGBoQP7xEZGkJ0cBAnAwPgX71GuLsb4WcenErLYidFxRoG+IS3b4CJCWBkBFhfB5aXgdFRCC4XwmYz9jkO0YYGHBmN+KxWY6urC9s2O4j4TpgB8lNPHR9Tz30AvQ17e0if2VkI1dXYVyrxXaHAD6qd0lJser3Y4ZpA8m8f8LnimvQb0H2v0X0nhXYPMD4OHBwAS0tIGQyI0EaeQvjKSoR8Puw+4kDyboWiNwoKMn7hwtyUnaivf5dsbQUWF4G1NaTo6KS8HLGKChz5/fhGm/mbhaHo9TzRX7/xzGDMjjyomZp6bEeANvT398NP3+FrZyc2W2zwO5xJlvN6vVGLxeKSSqVXMwASieRylUrV3dPTg8PDQ4RCIfoUs3jh8eCl2SJ82dpKsNz8/DyYR6VS+TMAYrH4ntPpJMHVVbTf1/1U090bGxvR29uLtra2dKxUKpN6vX6SQZj3zwn8HR0dcQZYLiouLJfKkjMzMwgGg5ibm6M/OwqZTHaq0WiuMADzZgBEIlGzyWTiGUCn002XlJQk5HI5tFptWixmOVZjAObNAOTSo1arNwOBAD88PHw8NjaGvr4+2O12uN3udMxyrMY8zJsByMnJuURHLFAoFLscx8WsVmuc6ry2traOqozGJyzHaszDvL8B+W6qBSeP34AAAAAASUVORK5CYII=',
ruby_put=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJISURBVDjLzZNdSBNQHMV98lHovfcSbS7NkshK0zSXKbqmtTHSLTUt1OVHEaUUis6H5kul4QeCIpig+IFFUEIo9jUttWJoyUAJbbvpcjrdfl3tpaAs6qUL5+XCOfec//9cP8DvX+D3fwj4LhVpJT5JiA14i03Ca8oX6xfyxNq5bOExGsSqXi9W0k8Jd6paLB9PWnAdPabbFJAEjYSL+wMwNARjY2CzweQk3sePWGhtxdHSwmJzM866euwWC/ZrZSwdPOJajIhM2hB4TsNd6OmBjg6YmIDRUejsxFtcjD0nh1mjEYfBwLxOx4vERN5WVWHLL0Ts3mffEAjxlZie+qorQb7GzAybZ3AQr1rNrErFx4QEPktMR0fzpqKCaWMWImTvnDNwt2ZzBjLvNpm331teBt3dMDcHVis+rZYFSXRKEWd8PFOVlbzPMCIUe6YcO5XKH7awlpMV4MnM7FsvLYXhYRgfxyeti9hYXHFxzJvNfJBkZ3DYlGOHYtdP17ii1QW4NWl9npLSbwJyBq6oKFZqajj/8AzZ/ToyutLQtafwyx4sn0gO+BKverB65Sq+9nbWqqtxFJg426ulbbKBltd1pDaq2LJIS4djti/uP2RbNRXhlmSZ163vUNP06hZ11lq0TanE1ERGbNlEERoRJPOO3LiZ6Na3pXgu9uZy+6WF2mdmCrpyOVAe7g6/rBz4bVVPtyX7S8u2vHsGLCNVmIeuk16fQliJ4p0iP9D/jzsvLY/pGk+iuZNMaFGw9a8+k7Q8oiwMevL93Ve6KsnKrbmJHAAAAABJRU5ErkJggg==',
script=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ+SURBVBgZBcExbFRlAADg7//fu7teC3elQEoMgeDkYDQ6oMQQTYyGxMHZuDA6Ypw0cWI20cHJUdl0cJLIiomR6OACGhUCpqGWtlzbu/b97/3v9/tCKQVc/e7RRXz+7OrSpUXbW7S9tu8ddv0M+3iCjF1s42v8WAP0XffKi2eOXfro9dMAYJ766SL1092jfDa17DfZgycHfvh7/hau1QB9161PhgE8epoNQlAHqprRIDo3iqoYDSpeOjv2zHRl7atfNj6LALltJys1Xc9+CmYtTxtmR8yO2D7kv4MMPr7x0KULK54/NThdA+S2XTs+jOYN86MsxqBGVRErKkEV6BHynp//2fXbw9lGDZBTWp+OK7PDzqIpYiyqSMxBFakUVYVS2dxrfHHrrz1crQG6lM6vTwZmR0UHhSoHsSBTKeoS9YU8yLrUXfj+w9d2IkBOzfkz05F5KkKkCkFERACEQil0TSOnJkMNV67fHNdVHI4GUcpZVFAUZAEExEibs4P5osMeROiadHoUiIEeCgFREAoRBOMB2weNrkmbNz+9UiBCTs1yrVdHqhgIkRL0EOj7QGG5jrZ2D+XUbADEy9dunOpSun7xuXMe7xUPNrOd/WyeyKUIoRgOGS8xWWZ7b6FLaROgzim9iXd+vXvf7mHtoCnaXDRtkLpel3t9KdamUx+8fcbj7YWc0hZAndv25XffeGH8yfuvAoBcaHOROhS+vLlhecD+wUJu222AOrft/cdPZr65ddfqsbHVyZLVlZHpysjx5aHRMBrV0XuX141qtnb25bb9F6Duu+7b23funb195955nMRJnMAJTJeGg8HS0sBkZWx1suz3Px79iZ8A/gd7ijssEaZF9QAAAABJRU5ErkJggg==',
script_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK9SURBVBgZBcHLi1VlAADw3/edc+fRmOP4YEzERxQYZGaQaQ8qRDCK+gPcGC1rYbjRWtqiTaAULWrRItwVVAaFBUIvhqjAyixIE41JB8fxzsy9c+855ztfv1/IOYPDH1/bg5N3rxnb169b/bpVt62Vpu1iCTeRsIB5fIizJUDbNI/s2rhq39EnNwCAXtVO9qt2cmGQNlc1S8Pkys1lX1zqHcCREqBtmunVIwFcu510QlAGipLRTrRlNCpi1CnYvXncpsmJte//OPtWBEh1vXqipGlZqoJuze0h3QHdAfMrzC0ncPz0Vfu2T7h/fWdDCZDqeu2dI1FvSG+QxBiUKApiQSEoAi1CWjRzecEvV7uzJUCqqunJ8UJ3pdEfZjFmRSSmoIgUsqJALtxYHDr11d+LOFwCNFW1dXp1R3eQNZApUhAzEoWszFGbSZ2kqZrtn7762K0IkKrh1o2To3pVFiJFCCIiAiBkcqYZDqVqmKCEgye+HC+LODLaiaqURBlZRhJAQIzUKVnu9RssQgnNsNowMTEmBlrIhEAU5EwIXLx0xl+XP7fUXzAV+0V3+cbrHHyjhFQN7ygnRpSRIgapDeSsRQj8+udH5vtfe/rxh21ee69zFz4JM79fP7H3lU1r4hNHTq9vqurEnh1bXF/MrtxIbi0lvYqUsxCyny6c9uCOXVJMdt11QAq1vTsfhZfLVFX78ezPF/+xsFJaHmZ1yoZ1UDWtJrWWuv/phFWeue8lcHT/e8789i4+GytTXT/0wlMPjL92aC8ASJk6ZVXD88e7Lsz+4Pzsd44d+MCbZ180VozCoNi48+A9U5MTz80v1a7O9cwtDiz2a3WTFTEa6QQpDX3zxxnbpre52f9Xtzfn+/PfWrw9PBV2Hzq5HkewFeuwDlOYwuTYSKczNtYRRs5ZSTPaPEDok9+eeWf22P/PLlOL9Py8xgAAAABJRU5ErkJggg==',
script_code=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALtSURBVBgZTcFLaFxVAIDh/5577jwzj0wSUmqMtKIiBltbbJ1FUCxVoQu3FrHGVRU3BVcKrkTcKOhCUOtOAyJ23WIQtFawpoooZWKJpnbsNJN5PzP3PO5xArPo93nOOfasXCgfAz48mE8UhzpiqCN0FLFrog7QA+qABVpAA/gC+FYyERlz/NC+qeIbT85xt4GKckMV5Voju6A09ELLzXqfi38PTgLnJBORMfPZmMeectsSeB7SA19CPBAsxgW+EAQ+PLaQZH8uXTj/S+UDwYTVOitxmAh6yqOjoR1CZwSdETR2Yadv2fPm6i2KB9IszQZzkgkVmvnLZcuP21VeO1rgs+tdAu1YOZxlKiHw8fA9iADPdvn5nxa/3epUBGOH39sqjETu2UJG4oUwDB2RcmRSHuevdtjpWgZhxEBH4KDaDflobbNrlVoRh97demHpgfTth+5J5ZpNw5kjWQxw6mCa7aYlk4bPr7X54XqfkfGIHNjAYpQ6cOH1x9fEw/cnP13M+Ik7bc3ZYxniMR9PQCElObmYptox7E97XK0MscbhHJgwxKrQMiZ+v9Y9u3knHBUCn08ut6m2DQJHe6C5WOqQl4KbVcXR2QSxwENbS38wNEapLmNi4/0Hv/r3zxvHN0p1YnGP1e/r4ODr9TbZlKBTU7xSnKG4lCUZQKMfYkJVvfT2c44xyVjKr6lpEUI3g3UOPIE1lu6O5aUTcyRjPjhISUGttYtVYYUJuXxudRZ4p/jIvZx+eoHvSopmz/Ly8jyJwBFIkD7EfMimYLM8xChVZUJapU4Ap34tbdHalfRDh7aOUHsoE2FsROQchVyOV5/Zx3ZjiFWqxoS0Wh95/qlHk2+9+AR3sw60dSgDOPj4UoVUAL3+EKt1gwlptd7arnf4cq1EfipJPpsgn46TS8fJpGLEY4K4FJxenicuodbsYbX+jwkZGfPNlfWNhSvrG/cBM8AMMA1MA7lELAgSiYBsOkk+m+KPv8o3gJ+Y+B9yFXCQeyJWrQAAAABJRU5ErkJggg==',
script_code_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL2SURBVDjLVdNNaBx1GMfx78zO7Lu765qQWIPteqiH+IKCjTmkClIVPHi0pVDSXBT0sGhLUOvJl0OL4AuVBC+CLXpoD1KqweBBWFFajOChSTU20q2h6e66nX2Znfm/ejCJ8Qff6+fy8DjWWgCmz9f3AR/cV0pPhtIQSoM0hoEyAdAFmoAG2kAL+Bz41mNzRqmJh0fzk689MczO9YUphsIU25EeExK6sebPZo9v/ugfAKo7gZFC0gGgflvjOw6eAwkPUr7LvSmXhOviJ+CRsQy7irnyp5fW33e3AC1loWQilIGucAgkdDoRQQRBBK0B3OppAF4/e53JSo4HhvzhbSAfBiP7v5pHHZ2mH2kqJ2cZ+uxjwqBPNzb0IstAgAEc3eGna22WrgfrLoA4Ol2eXfrimZw3QDoeYWyRePh0GZ0/RdTu0o8NfWnAwkYn5sPF1Y4WYtoVR44cNGH01+iwXzT1FvVj76CApUNVxI2ApNNn77EZUrXviZSDsaB9jRKicv7lxxZdM4jm433jabO0Asdn6eTLOC7oQpnfX3gFfrmKnBjnnrmTuFJiLag4RotYA7gmHLyY+LoWmcounBNvYm81cLEkgyaVuXdRe+7GuVijPlPFSfpIren1Q6WE6AC42YsXvjy9e//EtdXbyHyKyqszYOH+946TKHiEbcOVuXMEUwfI+NDqxahYbCy89awFcAEWhvaKj6YO48WQaDTAcXGbDcIA1k6cQuaKYCHruTTaA7SI17eu501Vzw4Bbz84OU6veobvlgW5rubHTy6Q9i2+B8nEvxWysFoPUUJsbANaiKeA535eXqM98OjFFqktsXQQyqC0wVhLuVjkpadHudkK0UI0/gOkfPT5Jx/KvHH48f/9gLYgtUUowMLphXWyPnR7IVrK1k5g7WYz4MziMqV8hlIhTSmXophLcUc2SSrpkvJcDk2NkPKg8XcXLeWNbcAoda52eWWsdnllN3DXZnduVkwnfT+d9inkMpQKWX69Wv8N+GEL+AdfSH+n2ppttwAAAABJRU5ErkJggg==',
script_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK9SURBVBgZBcFLiFVlAADg7//PuXdmGp3xMeIokk1USG8jKmlRYJJU1K6NRILQopXVImoVFBGBpLteu2gVLYyiUALFRSVk0aKC0nyE5uA43pm598495/zn7/tCzhns//LSQzh867rxXYO6NahbddsaNm0Py7iGhEUs4DMcKwHapnn4vtk1u157bBMA6Fft9KBqpxdX07aqZnmUnL+24tuz/T04WAK0TbN5qhvApRtJJwRloCgZ60Q3j0VFjDoFO7dN2Do9ueGT05cPRYBU11OTJU3LchX0am6M6K3SW2VhyPxKAm98ftGuuUl3z3Q2lQCprjes7Ub9Ef3VJMagRFEQCwpBEWgR0pIfzy06c7F3uQRIVbV5eqLQGzYGoyzGrIjEFBSRQlYUyIWrSyNHjv+9hP0lQFNV2zdPdfRWswYyRQpiRqKQlTlqM6mTNFUzd/SVR69HgFSNts9Oj+lXWYgUIYiICICQyZlmNJKqUYIS9r793URZxO5YJ6pSEmVkGUkAATFSp2SlP2iwBCU0o2rT5OS4GGghEwJRkDMh4ORHhic/9MO/f3lpfF1YU11/nea9ElI1uqmc7CojRQxSG8hZixBw4mNTf37hjucPGJu7y/C3Y8Xvp46/c/yJTr/4/sbtM21Kh3Y/uOPOua0zfjnfSG2WBUXMioLRpy+6/9kXTJw9IZz6QGd4XnfDlnjl3IUdZaqq3Xj65z/+sTgsrYyyOmWjOqiaVpNaB65eMD47x1OvAijf2qJowy1lqusHnnv83ok39z0CAFKmTlnVcOanrQa/fmPyq5eNhv8ZYHmpkAqXi9l79t62fnrymYXl2sX5vvmlVUuDWt1kRYy6naAbWv+cOip2grro6y1k567ElBrvh537Ds/gILZjIzZiPdZjerzb6YyPd+xJp+248rW1/QVVGeeL3Bx58ljz7v/pCEpK8wRGcAAAAABJRU5ErkJggg==',
script_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMCSURBVDjLTdBNaFxVGMbx/zn33plOZjIT8zmGOnFM1VoNJYFWYi2CtVSMdKUbK+LKhSAGXYhd2m6FQkEXLbiwETcWiq3WpiiSBLRqTa122pEmMZPmw8lkMt93zrnnXhdJbB54l++Ph0cEQQDAm1/l9gOnHmnbMVzXPnXto32fhueXgAqwChigCBSAz4ErNpvxPe/pvcnY8PvPdbE9NeUn6spPFF2zU2moNA1zq1W+vVs7DIxuB3riIQFAbt3gCIEtwLIh7EhSYYklJY4Fgzsj9Cai7WeuLX4stwCjdTxqg+dDRQlKGtabUHI3rtCAf6sGgA/H5hlOR3mq0+mytwHtrSFJrQk11yClwAYsC6QFFgJLgA8IU+anmSLX50uL9wGlehIRi1LDo94MkDLAkiCNwJJgEbCj/AN/j3/G250D1CZ/5BWdHPsf8JTq64k7lNwADyAAywhksLF9vPI17WvXiAy8TiI9yPrs4zSunH1jW4NmXzIRJrNiEBIkG88SaKlcJuX8SezRA6zdzRASitZ4klhHKmEDvHjicsS2ZCjsSJQxSAIgIADCtSnS9i8k0kdoLn1JqEXwz/RttKsKbqP6jATwmqorLEBujkQAAohUJtglrpLofwl38QzCKeLEWtHVRV+Xl17Y9875rNys32LjY0uwpAAhMfOXSJmrJHYdxb33KdLRqPLDrEzc4PTC4dtD741PA8iDo2OdnlIn9u9OsVwOmFsxlLKXSOqf6X5yBLV8FisU0Cz3kZ/8ndzAR2Sq3TNb29lGqUPAyG+ZWYoNG2fhG14dyOP5vSzdPM0D3SHctYfITd1CHvqEhZyLUSq/BUij9dDLB56IfHF8hJOvPcYeLrLn2bcI5ybJXphi+rs17nx/g4n2D4i09VKp1jFaF+430Hp2ebXEufEMbbEI2Zk86q+LpPcepJQvcO/mDM8fv8CDoX7CNuTXKhitF7YAMXjsVCcwCvQBHf25k0eG0l1i3+60mFPR4HxuSLhOB/FohLZ4C3/cyWWBY9fPvfsrwH+7HFmMUqkOrwAAAABJRU5ErkJggg==',
script_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALvSURBVDjLTdJNaFxVGMbx/znnzkwm05lJ0xlm7FeIbW3UxDotVoOEFkTMotKFK+mmOxfdRMWFggutOxGquHOnBFw4CIq0WEEsihJNBEWKUmuTYMjXJPM9c8+957wubNM88G5/8Dy8SkQAuFhdPg1ceXBoYLIbebqRJ/KeXuwbQAvYBBywDdSAT4CvA+7Gx/GTJ8p7Jl89U2R3Otbnu9bnt/vuoI2gFTrubLa5+nfnWWBmN1DKJRUAy3VHQikCBSaAVEJzOKUxWpMwUDmYZn8+M/zR3Mp7+h7goiiXCSD20LKKRgT1EBr9/6/Wg/W2A+D12SUmRzOMFxLFYBcwnE1qOiF0+g6tFQFgDGgDBoVR4AHlmvx0e5uFpcbKfcDaUj5taPRiuqGgtWA0aKcwGgyCMYAY1poh71+/1QQu7lSIrR0p5RI0+kIMWAHrFNaBavxCZukdbNjCC7iEI7Z2tHrpiev3N7DhSDmfomMFpcEohQaM75GrVSmWJkhvfIkIxGGIs6ED0ADTl6+lcXEyldBYBxoBPIInvX2V4VKF3AMVzNo3dGp/0u5049ja5g4Qh7aYUqDvjoSAApLhCgOrX5EtDOGaX1B+9EXWfptFbH/92pvTsgM4Gw4GeAINRitQGgRSq1VKx89Df4G5j2fZM9Sjt/wjj6d+796rrqdmZguxtZdPjx1mtSncWXNstRyu9ivDqk52bxcf/gPice2fGT37Cs9kvy/PfTCZBTAHTp57Xpx7rWejxK0NxeJqndv/1ji28SFHTkxh/AISb7H/kQLiuyQHU6STx5P1xfn+oadf+i5wUXTy/NnH0m9ceGrn/zf++Jze1gSD2U1cuwbKMF+9yakXHsb3/6L40HMs3ogu3Xh7/DNTnpg+ujefOVdrRSytd1hv9mn/8BajZ15GBwaVOIQeGONAZQo9MIZKHiHIHCOZPZrZvPltSVUuXCkAM8AIsA/Y9+74p6eMRFq8KPGCeA/iwYN3sWhxVnnnRWT+P1A8aPl/RBTPAAAAAElFTkSuQmCC',
script_gear=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALvSURBVBgZBcFNaNUFAADw3//jbe/t6d6cc2/kUpeXsEgUsSSiKIzAQxDdvCgdulgagmBXLx4K7BgRWamnOgSDIj3EusRangwlbVvOyba25tvH23v/z36/oCxLcOr7uaO48sxA9Vg7LbTTQloUtrKihXUsI8cqVvAtfo4Biix78eDItmPnX90FADaTotFOisZqJx9NUta7udnlDT/+vXkc52KAIsua/T0BmHuSqwSBOCCK6a2E9vSGojBUiTg0WvNUoz74xeTjT0OAPE376zFZwXoSaKU86dLq0OqwssXSRg4uXn/o2Fjd80OVXTFAnqaD23tCm102O7kwDMSIIsKISCAKKBDka36bXnX7YetxDJAnSbNRi7S2Mu1uKQxLUUiYB6KQSCmKUEYW17o+u/lgDadigCxJ9jb7K1qdUgYlUR4IS+RsPfhFliaeGzkhr+SyJBv74aOX/wsB8qS7d6TRazMpBSFREAjWH0lmflV21lR7e/T19fl3acmbAw+9MzT7CQRlWXrr0k+1OArb3104bvKfVKEE6fSEffv2mZ+f12w2hWFodnbW6Oio8fFxRVHUY8i6ya56vSoMKKAkCAi279bpdCwvL5uYmFCr1Rw4cEC73Vav1786c+ZMO4Q86fbFCnFIFAYEoY17tzSiTcPDw+7fv+/1kxe9e/q8R/PzRkZG7N+///Tly5fL+JVz14dw6eizeyyslWYXc/UqnVZLFEWazabh4WG1Kv19lGVgfX3d3Nyc6elpcZ4kb+DEH3dnrG7FNrqlNC8V2UEjG/MGBxeMjY2ZHP/aVFDa8/RuKysr7ty58yUuxHmaHn77tRdqH598CQDkJde+mcKAhYUFRw4f1Ol0zMzMaDQa8F6tVns/ztN0ZmG55drNuwa21Qz0Vw3UezXqvQYGh1y9etUHH5419fukxcVFy2XTrVufl1mW3bxx40YeHDp5ZQjnsBc7sRM7sAONak+lUq1WHKrds7S05M/yyF84efva2Sn4HxcNUm7wsX3qAAAAAElFTkSuQmCC',
script_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALZSURBVBgZBcFLiFVlAADg7zzuPLzjzDjOMINMitIie5gF+UAkIZSgRQuXLZIWrY021dYIggJdJURElJsoqlWRYA9GshGFCNQeOjoTk6bjeOd5zzn/f07flzRNA459ObcHJ3cM9+1fq2prVa2qa+uh7mAZ9xCxiAV8iu9zgDqEvU9ODOx//dkxALBa1kNrZT202I2TZcVyEd28t+Lb66uHcTwHqEMYH+xJwNyDqJUk8oQsp7eV2tqbytJUK+OpyX5bhtojH07Pv58CxKoabOeEmuUy0al4UNDp0umysM5/KxG8eWbW/u1tj4+2xnKAWFUjG3tSqwWr3ShNEzmyjDQjk8gSaiRxyYUbiy7PduZzgFiW40P9mc56sFY00rSRpaQxkaVkGlmGJnNnqXDq7N9LOJYDhLLcNj7Y0uk2AjRkMZE2iGQaeZOqG2IrCmXY/s1rB+6nALEstk0M9VotG0lKliRSpEjw+YUjPjq3RxkKoSjEsoiQwvMnvusXQ09vK1VGUg1qjVrUqDWKUJoc3emVj3dbWeuEUJZLkEMoyrF2u0+aUEPD19OHNXVQ1kEZgy2bHrZzYq/l7qr766/m3VC0ub+SQyyLDXm7R56SpYlYJ0JdOvzYy2JTi3VUa8x35jwxecBKue7S7E+dXW+nI/nB42dGcWLPI1vdXmrcvBO1++iGUmxqtxb+UtVBqCtVrCwVy3Y/dNBKtZb+OjO1kMeyfA4vXLo6Y3E9t1I0qtjo6goxGB/cKtRRbGr/dmaNDEy4PHfe+etTd8vgSB6r6ukXD+3qf+ulfQDg6OnCJ7+8p6xL3VDaMfqofTuOuHhryrk/fl4tokPz7zRX8lhVM7fvdXx29qrhgX7Dg32G271OHv3dxg09entSvXnqmXcHJGm/6Ru/ad89dmrm9AdXIK9D+GLq4rXJqYvXtmEzNmMTNmGor6fV6utr6YxWfvjzR0P/vDGTh7GvAP4H2uh1wse2x/0AAAAASUVORK5CYII=',
script_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALnSURBVBgZBcFdaNQFAADw3//jbred7pY6NLe2FKGgKOthtiJMyiiEgnoMwacIglKih3r1OSh68MFe+hCjsqgXU9N6WJFJBZU5I8/5kbqYzbvtbnf/z36/oCxLsOfIlSm8vXmkNt1NC920kBaFlaxoYQkLyLGIm/gQx2OAIsu23b9h1fRr20cBQCcpGt2kaCz28vEkZamfm1tYdvRCZyf2xgBFlq0frgbgyq1cJQjEAVHMQCU0MRCKwlAl4oHxQRsb9TUHf7r2VgiQp+lwPSYrWEoCrZRbfVo9Wj1urvDvcg7eOHTZ9Ka6e9dVRmOAPE3XrK6GOn06vVwYBmJEEWFEJBAFFAjyth+bi3653LoWA+RJsr4xGGmtZLr9UhiWopAwD0QhkVIUoYzMt/veOfF3G3tigCxJJtcPV7R6pQxKojwQlsiJytxw84D5c01bq5Nm+tObvtz3yH8hQJ70Jzc0BnSSUhASBYEQIQIMzh22sXHJ6Lbdnt/8u1dWXt8BMTy1/+vBOAqrA5VQkudCJUolcoGhq1+4o/Kdxl07dS4d1Ri506aJ9gdHX91cjSHrJ6P1ek0YUEBJEBAKlCX1pV81Ht4l755VWzeul191+/atQ1ln8UAMedIfiutVcUgUBvIioCwVCAL68Zh0cU4YLQjCjurwAr2u5ev9KH5076F12D9194Qb7dLcfK5eo1YNDa/8acP8e6q1VJH1hMUtwtXKNDH78fne2fPtF+M8SR7Hrp/PXbS4Elvul9K8tCWbMVU5aezJHcrkuCBf8uexvnK5q0h6K81me/fuwxePxHmaPvjsY/cNvvnCQwDg+jfvWnPP05KFA+K44Y9vM72Jl5ztTfj02OmDJ9/fdwTiPE0v3lho+ejEOSOrBo0M14zUB4yIBNE/ooFRs6e68i0v2zL1jFOf/6CVVZsAcZFln82cmR2fOTM7ibVYi9ueG2uNPdFsDrUXE5/MbbVwoWvk9Fd+O3/lL3wP8D9kTlQPa94WnAAAAABJRU5ErkJggg==',
script_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMBSURBVDjLTdFNaJt1AMfx7/OePGnypE27LbNrcbpZKagTqQ6ZjsmGOETxoKiXwURlgkz04sGDeFGYMhEv7ii7DAU9dXOgMCcOy8bYRLtRN03b2JekadO89Pm/eulqv/C7fi4/x1oLwJFvp8eAkzuLmb0daehIgzSGrjIrwCpQAzTQAOrA18APPusZpR59cFvP3nefHGBzbWGSjjBJY00PCgmrqebvWovxv9oHgeObga2F0AFgelkTOA6+A54PUeAyFLl4rkvgwZ7BLNuTXN+p36qfbgBjhQujh6KfWLpu8YXBaEOqDVZLIIdz3wkIt3BX4vH+6QqfvDrMr7eCAR+gfvWAN5obHS31lvGCATJSYaVCiy7dxiQrpRexth9PgAEc3eTSrQZXKitVF8Bqs2Mk/nNnkB3GtP/AcXMYrVhbvsl8zyus5J+hnUJbGrAw30z5/PxUUwtxxK9d3h9abV7L94144GG6c+CnNP45T630Fq3keZDgYfGti7GgA40S6u7v33l8yTda77HKvBHld2FVF7wySIk2Hn63QjJzEtQaxsmjyi9g7RAqTdFSaQDfKvNFpvRIf7c+iVGGML4XjE9p+wHi+nW0FDSrV2ne8yFuVERqTavdUUATwDVKn6nevnbquysxNtzNwtQ5VNpGri0jOjUW9DD1sXOIHS+RzcbUWykqFfNnP3jaArjlJy6dOPbjsc/GZw+S7Hod0hadhd9ZrFzm396jNO//GJUZBgux77LY6KJFWr1zv7/v+Ol+4KOxkSFmfn6bMOpjtlumNvIVYTZP4FiC0CH0oBDD1HQHJcT8BqCFeAo4vFoZZ3nLFBfSo9xQj5HO1hFqEaUNxlr6koQ3D21jrt5BC7H4PyDlw8/tfyD73uFeLM/yUHE3ANqC1BahAAtfnq0SB7Da6qClrG8Gbs/VVjgzkVDsyVIszFHMRSS5iHwcEoUuke/y8r6tRD4sLq2ipZzZAIxS31ycmBy8ODE5DJTW17u+JBMGQSYTUMhlKRZirt2Yvgn8cgf4D/BEgoyc1axMAAAAAElFTkSuQmCC',
script_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL+SURBVBgZBcFNaJtlAMDx//ORjzZbs7TJkmowbJcdZqr1oNavCiIIMraBh0IY7uZx8+OiVw9SQZgXp3gR3A5OtIigcxMcylyqVPADh0WNpO2bpk2bvm3e5P163sffT1hrATj/2drDwKXjR7JzwyhhGCVEScIoTlzgAOgBBugDO8DHwA0NAJDE8SMPVA7NvTpfAgAAwAuT/DBM8n3fVMMIDgLDf70BX//jPQtc1AAASRyXJ9ICgLU9Q0oItAClIZOS3JeRKClJKZitjnFPPjf54U/OOxIAwETRRE5DnMBBKHAj2AvA9cH1YWcEWwMDwOtX28wdy3F/MVXSAAAmiiYPpyVeAJ5vkFKgAaVAKlAIlIAEEGaf5r99fmm7jgYAMGFYzo8p3FHMMLBIaVESpBEoCQqLUoBVdPcD3r359z5wXgMAxGFYK0+kcH1LDGBBGYG0gAGFRVtJYsGkDHEYH/vi5cd3JQCACYNaJZ/BCy1CghICCUhAAADCgrUQBwEmDAyABnjuzetjWsl0JiUJjUFiAYsFDAIAAUgJkTEMvGEM7ANogDgIS7lcFinAD3xav/2Iu/4npakCTneHk0+d4dDhSW5f/4jfiwUek1uy67Rfm59/6z0NYMJgXOfSWBOxfONT8tLjxXMNPM9jfX2dZvMrVCrL2dOn0FrR6XTkysrK2+12uySeuHClCFw+Mz/7wvHsFs3vv2WhscDVT77kr1/vMF2pUK/X6XQ69Ho9OpubpI9Ut155qXF0aWnJ1SYMnwGeX7nb4k77Z2aq4wD0y6cYDG+xsLBAoVBgMBiwvb3N5fc/YHf8wW+Ac/l8PqNNFD10+umZsTcaj3Ltmkez2QSgtvs5a9KyuLhILpcDwPM8bJIwtXv7STjJxsaGr00UtTZ7Lldu3iXU0/TdAT98d4v6zAz1ep1ut8vq6iqZTIZarUa5XMYPo6PLy8t7juNsitnGpSJwEahhk6KK9qpToz9O3Fsp6kw6LYSA1qhEdnyCaVpYm9go8H3Hcbqe5539H/YvZvvl5HpaAAAAAElFTkSuQmCC',
script_palette=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMnSURBVBgZTcFJbBtVAIDh3+MZj+3xFsdxVmo3qiCspU3axIKGpWqphNScCpReekTiUoQEgjPXSvTABTgglQghNUAViS09gFTEElJEcmgSkjaxm92JHY/HjmfmzSMRPvB9PiklBy6PFU4CH/Ymgrma41FzPBzPo+56u4AJFAEBlIBt4Drwo0qT57qDRzsiubefa+P/LNuL12wvXtoTPbYDZkOwVKzy3aJ1Brii0uS5bnvQWuPnsW8o5f8BnyTU8TC9A6dpfyjLIV3BryhofjjWE6IrbiQ/+WP1qkJT6/xXQ8m/P8WoF+g/qtP/ZJq0t8b8+FV+u/U1W6bLZlVw4L3RPLnDBk+ktDb/7p1x5d7kD9eHO63zQ73S1/tIhUTPFNHoMiGG6c5k2ctPkrcCRNuyJAyFiTsFItEAN6c3V9XebNfHuaf7Lz3eUcbX+Asnr0A4guV4fNYYJeSGGOk+x9rKJNZjwyD9bFQaXJtYqACX1a6EcbFvpoD7awXfuTgy2ErlQY6V8gz+2AzYdTTXxBA77Lk+PAlCE7i2e/jmW8/sqLpuhKUNXkDFir7E3MI0UX0W0n3cWB7EEDavJuZQxB4+CVKC22ggHFewT5U+SfHsIBHNwyzvcG/HJSLLtLQcoU4MRVbB54EUKAo4QlC1ai5QYZ+a31j/ojwxnotFjOzx/hOczMbR/C0EzUXe3f6dTGcPphZiU0+S1mC72sBt2Bu3Pjgv2ae8ee3b178sD7wyVnuWmZkpOtq6CZtFlN1NzhxKkdL2+HOzTvTRC4RVha1SHWE3VmlST10ZTaFn3nlh4Bhe3WFlY4l0KoNlW+RLWyz6M7SdeI1UupNYGBYKNVzb3qBJFbZ9Gnh56u59iu0DrO8uk1SXmTY7WZIvYkf68H6qkIz7eONsB+vbNYRtb9GkCsc5PvL8U6H3Lw3xnyEOjEhwhMR2AQkffb9KWAOzWkM4zjZNqnCc++vFXT6fuEsiEiIRC5IwdOKGTjQcQA8o6KrCxVPt6Cps7ZgIx3lAk+q57o3bk7M9tydnM0Ar0Aq0AC1APBjQtGBQI2aESMTCTM8V5oFfaPoX2glqeEUaXMAAAAAASUVORK5CYII=',
script_save=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK2SURBVBgZBcFNiJRlAADg532/b2Z2Z9fdUdfWUlwNooJIFNryEEUSBAVCVy9eIrwJngqCwI4Jdg46VBIdukRQtBAdOliUHQ0qlbXE9afdmXVn5vt7v54ntG0Lznx1axWXHh/MnBhXybhKqpRM6jTENu6jwSYe4DN8nwOkun7+6P75E+df2gcAdsq0OC7T4ua0OVhWbBeNm/cf+vbvnVdxLgdIdb280A3g1lajE4I8kOX0OtGhXpTFqJNx7OCsxxbn9nz8y+2LEaCpqoW5nDqxXQbDiq2C4ZThlAcT7j5swDuX1504MueZpc6+HKCpqj27utFOwc60EWOQI8uIGZkgCySEZuTK9U1X14e3c4CmLJcXZzPDSW1ctGJsZZHYBFkk08oytJmNUeGjtb9GOJMD1GW5srzQMZy2any99qddPcZlbfd81+27EyEy38882u/aHE0Wfvj932EO0JTFyv7FnmsbjRCZjKdeWX3SqePzvrnyj/dOPw0APv3xnpTCzxFeu/DdrKbu9jpR2RC1xkXlv+0arP26AWBaMyq4t1UKIYlQF+W+XiAGErTEGGQtVZNcPHtcSq0mtTJJP0+KojDaKeXQlEU/n+vKI1kMmhTMdKJpUfngy5tioG6S1CQt2ralpd9J8hfPXV7ChdWnDrkzat3caMzNUKdW0dSy2EEraIQQtKnVtkldV8qyljdleRKv/3bths1J7mHRqprW9rjSCbmzb+xSpxKRljYkM3nXh1+sezDakTdVdfzUy8/Ovnv6BQDw5vs/yXQkE59cfdtS/5Aguje+5a3nLolmjItS3lTVjTv3hz5fu2YwP2uwMGMw15PnmcP7WZ7fdvLoqkH3AJGt6QGPzNX6XfIQhGOnLy3hHFawF3uxO/aPHH5iZU9n0BuYyRfUqRJSEtpGCmN/rF93d1T5H4CHTHMseNtCAAAAAElFTkSuQmCC',
server=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGkSURBVDjLpZM/T8JAGMZf/gRU0lgU1EZDok5KSIwOfgKJhsGlgyaOfAE/Abo7dNfQ1W9AmNk0kjYO6ISLk4tBDJb27nzfKyWNFhLjJZeXlnue5/feXWNCCPjPSEa9bLfbpud5Z67rJrECVjmHwyHVW13XTyMNbNueR8GNqqp6Oj0HDoqAc5CMSJrNqtBsNk7w6beBZVmHJM7lcquZTAa6L69A6ZwL4IKjkQCGZo7jRLeAixuFQkH+Hgy+wEXxo/2ABnw8D8pHso1IA+oxlUpBr9fzkxmD7eKOTBejdNrwiQTkzFBEi+hgCN227rEymU7/aZXj6QT+Xgm5mAyKpV2ZHlCIiD2IhwkCA2lCyczv/U3pQv3dAB5aF9nCWDwSBun152v4/PDk80QDQgtuJYn4CJnju/PFC+ALRMfHrUYS0CkkEgk/mXEpIAOfxN/cnwTxEMFdq9WCfr8PiqLAzGwab2BMGjHh0xDhRIJqtbpvGEal0+lcapq2t76xCUv5NSkIWltZzss7Eh6xqK+xVquVUXiFsxR8TFRp4hE/maa5NdXgL+Mbb/xsAcKofWEAAAAASUVORK5CYII=',
server_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI2SURBVDjLpVNNaBNBFP42aZNqXEw1iv0xFUpBLWmrF0v1IpKiKOohBwUvhYDgz0G8eIugYBBFD+JBpV49ekqpHoQklGBpzNb0UIwkoFarIvnT5m9mfDPJhlJTEVx4vN3Z9/28eTOaEAL/87StXYjH409rtdq5arXaRhmUVVQqFZmf+Xy+sy0JDMPYTIAnTqfTZ7dvRJlA4BzKH7ns7HRienrqDH39SZBIJI5KsMvl6nE4HEhnPkGqcy7ABSciAUZk5XK5dQtUPOV2u9XCykoJVQK/NeaIgDfDO35MtdGSQPZos9mQz+fryoxh7+CIUhcNdbnZ6zqQzIxAskgORVo3ErOUmVKX/7qOn/q7g/peCVUsCQY9+5W66UKsswcW04FJoEikMqv3/k1PYzJ7H3xVXcsWmuAG0FSfXHyMn4UaEpkwcttSOH/vtCj8yhWyxR+3Q7eMm4pAWjNPpATxhmVOa1e2XsciZpEpxeA9fBC9WwbwauG5HktGboxe6nY2W5BTsFqtdWXGV41QIPk5jJHdw2AWhuEuL5hWxahnTEIvWhoOXkciERSLRei6jo4NdjqBmiJi5OJ7YRnt2iac3HNZubx65BH6tw/J1w5F4Pf7D6RSqROhUGguGg3DbtOwq68XbncP+nZ2I1/MYmFpBsGXE4og+GIC77/Oy9eStvY2BgKBcRrrHQqPeZk+8CTa+5cxNnQIAzv24d2XN5iZj+JjOndX+9frTBsWpHSBQqcoUDyMPVi69hviC4VQKCIpUgAAAABJRU5ErkJggg==',
server_chart=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIzSURBVDjLpVO7a1NRGP/dm9ubpGlMioGiLdFBHCyFtksHZ6tiQZAMEbqZwTGgkw7B3cmugoOL/0ATujadlIZchCwGWlERF5XbNOc+zsPvnNvUQmMWAx8f5+b8Ht/jWEop/M/POXvodDpvOOebcRw7lEHZRBRFOr+rVCoPxxJ4nlcgwOtisVhJp6cREghSwngjh7OzRezstKp0Ok/Q7XbvaHCpVJrP5XI4OPwGrS6lglSSiBQEkYVhOL4Eutwql8vmwFiAmMAfvX0ikKdxa/2uKWMsga7RdV34vp8oC4Ebi8tGXZ2o60b/04FmFgTSl/RAtHWv+4GyMOr6v0v37k92kPRKmcuaYHFp1aiPXKgJPbBHzIkDbZlIxEn9dgRf/UT6+wGezRxCvXqsxNMN/xzBKVj8bZwm2vq0gha7jedf1oCpLHBxgZTsqUe96gzFpiHQ1kbbqC2b8ckkz81lca1gwc24oPEAEcWx0Fd/2Zbztuo9+GEc6CmkUqmk7rMuIglOFfIhfWccKiTwkIPx2CmggCAILmgH79vtNgaDAfL5PDLZNG2gZYhiAvKQSjsmhwE1m+ngBAzJTEx7E2bsWq221u/3N5rN5v7e3i7SroWrVxZQLs9DDEmdaQIYIAJyEQmwIMBRNEAcxclbqNfr25S2G43Geq/Xe0mjXdJLJS6/AM9RbwIaJyP700TCpdlY3z4CCxmsSc955clnZSnznnDz967KOrC+Dp2wc104yh6mZJzlfwCf3q+o0qkR9wAAAABJRU5ErkJggg==',
server_compressed=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJjSURBVDjLpVPPaxNBGH2zu026piEpTayhJVBzswR/BBVvXiyKiJeACv4LnnsRQ/DqIYcilApevQuWHIpgqIhiSJQiCrEBf9EGarqm2eyP2fGbSZMWjCC4MHwMM+99771vhwkh8D+fcXhTrVYf+75/2/M8gyqoquW6rqxP8vn8rZEE9Xo9RoBH8Xg8Hw4fgUMgBAGUNlI4ORlHubx6k3Z/EtRqtcsSnEgkZiKRCDab3yC7B4FAIAIiEuBE5jjOaAt0eTWdTquNbffgEfh9/S0RBMN1aeGKsjGSQHoMhUKwLKvfmXOcmD+luov97jLovyqQzJxA8pIciJRer72hyimgLsAYGo0owqaG5ZUl8gQmk2nvtKYXF4stpaCflVBEkmA+ewaatYZo6BmMCJ0LDsdOYmv3Ivthxa7R9ae9nn2MassYeOsrkJK5kj4VXsdMbpb2LepJ1qwuxhvPxdef13d1nUm7akiaJBiC+UFw+hgdGiac7fOwv+dgTpiqUdfVlWTf7ys3ZDiDv1EGJy0IGZwG6Bp1TdYphi40Nib7EdBlno/hVJQCOQVd15X0YF8FYwE87mLn8xzeradJnaUA3HPpjq8mpghIwetKpYJOp4NoNIpxM0x/IEMgLToaYqkaTp7dAHqaAuiMjwnuCd/tKNlMyi+VSldpGsVUKpWbO55BLDYFu/UCem+NQjTgenvwHBsvN5JfYtOnZ51u2//QbGVXlh5+ZIdfY6FQWCCiB7Syg8cka+TcXVi2D/PTfS5FKCuZe+LXHhfsX57zjeIr3t7aRDooX2C+lbF5aKJz9M6ytd0UvwHmbqDcpFnnSAAAAABJRU5ErkJggg==',
server_connect=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKFSURBVDjLpVPfSxRRFP52Z3aXXdOd1V11NbYHUUQRtRIi38pCCQoi6KWX/oRefeq1t94TgyCJhKiHWgwki4pgU3YxonzQ1BXBrXHd2Z2Ze2fuTPdef0Sl9NDA5dyBc77vO+c7N+D7Pv7nU/+VsLCwcN913RuO46g8gkd5KKUiPgocpaBQKMR5wYSmadcikRgIL4LnQWbzmkRCw8xM9nAF+Xx+VBQnk8n2uro6rHzbgGD3PB+e73EgH4yDEUIOB+DJ2UwmI++WZcPhxYuFeQ7gHZwLF8dkG38BPHj9I1Ovr0PXdXR3d+8yM4ae3gHJ7u+xi9Z/UzD1Vo9Sh005O8Wx5mgUiqIgl8thYGBQSi/kczwyyc44YPrSlV8KOKvKJ1qIhAKdqtaCT8ub6EhUZIJQwJiH3r6Tkn1fhb83g6Ds06ZtFnE7Y2FACXhQm7rwfjWCdMcg5uZeIfviKZgcoodS/Qomy3fBRykJJEDVpO2E/3AgrJdqCCkewlobJp7NS8+Hhk5j+vFDuMzF5Nd7WC1tShUHLRiWlVICQM32OZiNrW0H8ZiK9IkOvFt8g/7qMhKNTcJ+3Gq6Da9RrIInwaWCHcPqog6BaTuomARGzcZSUUfZIIgeP4XsR4bh4XN7FvoyCjChQCk1Xw4bJplNaWGlwgvDXIqq8B0PBrhdDGsrJZztaUUmHZV2it7FRsbjDdylD1DLhlVtiAVDYkg1i2B9YwfGdg1W1YZdJWiMEIxc70cymZKS91e/tSUlHVLLVXM26IdGP383UCzqjlOzxxl1l5hNSy6lGyNnyr0vZ57f4cV9+49JxF172RfVNMyrlS3niWuR865Fj+Wmb9I/lnONn+xRr/UnsVG4KayFAQcAAAAASUVORK5CYII=',
server_database=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLpZPNbhJRFMf/w9fwNUiVlZWPncSmqemm7FjZaGJcsdCkS17AJ8AH6BvU6BKfgFIXLqArTQngSoPUDSGEhAUCLQN3xnPOMLSJGBdOcnK45J7f+Z//vVezbRv/8/luLxqNxvvlcnm0WCx8lEFZwjRNzh8KhcKrjYBWq3WHCt7G4/GCrocxpyJYFkQbKdzaiuPs7PQlrf4ENJvNp1ycSCS2I5EILn/2wN0ty4ZlWwSyoQg2n883j0CbT1OplCyurq6xoOKvrQsCWOt4cvhMxtgI4BkDgQDG47HTWSk82nks3e1Vdzb6rwqYrKiIN/GBsPRW8wtlJd35ixl5JJNJVKvVIf0XpP0/KH+kfCIKHK9sATFgZ3dfuosH9gJGNIx8Po9oNJrweDyYTqd7g8Fgr91uv/C4szkKWDJBlDP70LjEtZoik8lA1/W1bB6ZFVHDhwJYF6sb4xj07tsJzKUp3vR6PQyHQ/T7fVBnuR+s2MPmuLeRJcvxWU5+fe8NQlpYOrJ0PubRaIRgMIhYLCYAMZE3eL1eZ27FrhOAVdEathK4O6rrGX+sjBV8rtfrmEwmMAwDwZBON1ATkFoZ627m36trfQMoFosHnU7neaVSuTg/r0EPaMikHyCV2kY6eZ/tFfmscjabwe/3IxQKCYDH126/xlKpdEj0Y4pd9zFls1nkcjmk0+m1F6yi2+2iVqt91/71nMvlcoCkFymOKO4S2EedfxHkE8Xxb6/5rMKMZgtBAAAAAElFTkSuQmCC',
server_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIuSURBVDjLpVNNaBNBFP62u8nGhpCU5GAQk67ipaEgFQS9KKipsQcRcmjBmwfP1UtPpgXpyUNuPbRUvNWTJwlNKQihF4sh6x96qK3FtGqwamqTbHZnxplJNy01FcGFx2Nn3ve97703T2GM4X8+7eBBsVh86DjOTdu2Ne7BvbRmsyn8XDqdHulIYJpmkANmQqFQWte7YXEQKIXUx1X29IQwP58b5n9/EpRKpasCHIlEjvn9fqyulSGyU8pAGeVEDISTWZbVuQQenIvFYvKgXm/A5uBX5gtOQNt2JZmSZXQkEDV6vV5Uq9VWZkLQlzgts7Pd7KLZhyoQzISDRJAYipBulpa5JzK7uIsOXf+7glavmAwWBIn+AZndVcEO6UGXq8AlkCQiM2nVXgmsYvZHFnRfXEeCNpjsNU4Qzb6fxsfKJqylOZx/PYPFSx6aT2rr3MbaJQhp7osUkumuZMrPRsPj0Ddz0L8t4sKtu9CNhFJ/mT/+trBwf+GyZ6etQExBVdVW3ftUyLew/BinLt6Ab+UZlEcj6P7wBL3xsMoUNuoqeF4oFM4ahoFAIADfkZ+o1RpynKJ27ftX+I4awLU7e90fj0KlSq/iSs9ms0N8GhPRaPSMceIkgsGwnI6435ocxLlUCv71p7Dqn1Hj8dtVFe9WUFYObmMmk0ly4ANu/e4y9f16gwHPBuJhG1rXJ2xXHKx9UYndYPeUf13npeHY2M7Wxm2VKHGisjJHTQ3mncnfQiCLpXc8FWIAAAAASUVORK5CYII=',
server_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ/SURBVDjLpZNfSFNRHMe/293umptu01GJMF1WQ5dRQgVl0UthYNHDiAQhChF6bdBLrz1Erz3Ug7CewqeoF0XzZVialnaXSHOYM2Ko6Szn3P2ze+/pnDMdodZLFw7n3nPP9/P7/n6/cyyEEPzPY9u5MDU1FdN1vatYLNroDDrzoWkam/sikUjnnoBEIuGhgl6v1xtxOCqgUhFME9wfdenzeTE4OHCTfu0GSJLUzsR+v7/O5XIhvZABi26aBCYxKYjAoDBVVfdOgW4eCAQCfEGWFRSpeDoxSQFmeVy6fIWnsSeA5SiKInK5XCmyYaA5fIJHJ1vRWbGrhRRmX3QSwd2CzdU4LErmFgcwskFFbBNrCrOekD7Q2eDR2b/AxSqcb7TD6e+CJ3gSv9IhJId6H5cdlGpF+GYGCLe08ujMRaUxAb9rCe7ac1j7+gWiRUNl1UG4awKesoNtAIfQyEzM353DCPkIvMF2qIt9ECss+CYlUVS0rCLnz5YBZbFRKhoDOPUEWmp1eBqvQck8g1XUYXfXY19hermQt7WfuvcmxQGsPdsnkllmKZDsKI4eWYfn8HUqfgqrXYeWC2IlPrZhyIWO1ui4VO4Cc8C6IAhCKe/V92io+Yn94avQlnohiATKej0W4+O08oXjoej4wnYbrVsOJkZGRpDP57GRjqPBR9tJ6rA4/QQmFGyu1WF5dAbPZwMIRccW/jwHHNDd3X1mbm6uo7+/fzKTfInmth44vr9F6vU7SINrmBn6CGfbA2Rl666DZNl5Gx/2HNPv37krwGpAGn6F+XQan8Q2ZFUHa3EyFos1/RNw40K13HrogO10UxDEro7KP1Zudzz6PP+36/wbU7aHpoSFkuUAAAAASUVORK5CYII=',
server_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI4SURBVDjLpVPNaxNBFP9tNx9N0zW7ENRFE6ygpYaCNXrQkwWtFUUEc6hgD/0TxKsS1KOX4LlQEAUPfoAKpScDAcHUhI1VCr2kF4VWsRBCk93szPhmYqJt0pMLj8fMzu/jvTejCSHwP19g90a5XF7wff9Wq9UKUAZlFZ7nyfw8k8nc7EtQqVRiBJg3TTMTDg/BJRA4h/JHLi3LxNLS4gytegkcx5mW4Hg8figajaK6/g1SnXMBLjgRCTAic123fwl0eDGZTKqNRqOJFoFXKiUi4N24OHVZldGXQNYYCoVQq9XayozhROqkUhd/1GWz93QgmRmB5CE5FGm94ixTZkrdjv3CSNCHxs29HbR7JRSRJEiNn1LqEE0cFq9h2ZM4auZ7CAY6DjoEikQqs3btgUAe8cQ57LMnkDA2sPrq+pm+BF0w+9s4nf2AsfUFRtwEq71BYmIOzPXurTydDu4gkM3p3EjZODU+cmE0PuJIahZollF88gzDZgN+07tKca3HgZyCruuqbmk/7FWxPxaCYW2Du1Wqj4PVlzFy/o7mu+7D4uOzxr8OioVCAfV6HYZhYHAwgOHae5hJGuX2Z8I2kL4xCu59p39rODA+M+a73m2J1TrWc7ncFZrGfdu208fMTRqdhoPHI6RapPI8lF6uEskYND0GRC7h0/zdTcH5pLb7NWaz2akLVv7d6dmFoD6wDuFvyfns6LseGcXGVwdrbx+80Po95w+P0j8F40OyH0Lewm6Ibkb7dpZ+A2HofmbUgVesAAAAAElFTkSuQmCC',
server_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJUSURBVDjLpVPNaxNBFP8lqY1JSJOGFBtSU11BaEulUUFB7EUalB70kINCQfBv8FgNgvRUJCdPUj36FxgqKahpi18NSQXBikRbE6UfYpNtsruzO+ubTTaHNoLgwuMxs/P7eG/eOEzTxP98Xfs38vn8Y13XpxhjXZRB2QpN00R+mkwmb3QkKBaLAQI8CgaDSbfbC5VA4ByWP3LZ2xvE/HzmOq0OEhQKhcsCHA6Hoz6fD6WvZQh1zk1wkxORCYPIVFXtXAIdzsRiMWuj0VDACPyhuEIEvB0TiStWGR0JRI3d3d2oVqtNZcPA8MiYpW621EWz/+pAMBsEEofEpQjrxcI7yoalLv5FJq92dOAQoGw2a8bjcdRqNTrEUK5st9XtnNm+g7omQ+O698mt1YZN4LQdNJttWmEIZaNZ+5a/hLnfaSg6w0B4CApj9WsPj3kOlNAGt4DPNqdJmUH9weCjvvSFTmKo/xxqyh6K31/L47OhwKvbv2SLQDTHnkhhWfRA5xomRm7CoLXliCaisruB0YELkLWGc2X95e6p+85Q24G4BZfL1aybXCi6ZoG/7XwG4zoR0kQaDFW1hrGjFyGzuvNNaXHH2XLwNpfLQZZl+P1+HPa4iUCBbug40hNDf88gIgEJh1we9PmjyG8sYenL4lZdw1mHbT2dTk/SPNyLRCJnjksn8GB1Cgo506gU4UYKD+O8lMD79WUsrD3/qem4VJkxPzr2v8ZUKpUgolmKUfsxibwcncPpwXG8WFsoa9wCf2rPwb980l3HnsFd3gY3pM0Zs2Tv/wEq26vP9fcF2QAAAABJRU5ErkJggg==',
server_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ8SURBVDjLpZNLTBNRFIb/edgWamkLsylqsaASIDUEBHfCBqohcWG6UIM7YmJMNG5cuKkudGEMYQeJYqMbCZGFYVHrStMYTIm1RU3amsgjQVQaHsXSztt7p7YqlpWT3JyZO+f/zn/OnWF0Xcf/XPzOjVgsFlQUZVCWZZ5EkGgsSZJonPD7/ecqAhKJhJ0IHjgcDr/ZXA2RiKBpMPwRl06nA+Fw6Cx5+hcQj8dPUrEgCPusVivmF5ZBq2uaDk3XCEiHSmCiKFZugSSH3G63sZHPFyAT8fvEWwLQyquv/5TRRkUA7dFkMiGbzRYrqypa29qN6vqv6nTYuzqgZJWIaBI9FGo9EZ8lUTWq03eugdM4xEax9NQ3K64ujx2+9GH8LwfFWelGMgW0eTuM6iUXzOI0jncLsLdeOZaZGb4bu3NkueNG+nnZQQlgQEhlKqL3qzXz+JJ6jI6D+2Fv7kNuMYQ9tuZah1ucCl1tHGJLgLJY/T04CnqYuo8GpQC7dwDq9idYhHrw1YCrp726qdU9ajigwyl9kdQybUGnELJ3re4mhM0w5PUFsFwGDJuDqSYDFLbxY0Xkyi3QU+A4rti3SqeuwSQu4YD2EuZaAla+g9U2ANYGXZaQnEgVPqayF0sOopFIpNvj8cBms8FStQk+8xp1+Tdo6PVBl16AUbcwN70Fc27lG8cwSH/OXr7wZH6KKVkfGRkZIKdxy+VydXoam1C/No76zhOQMqPgeTuSr2RMRqvQm31XY7Fy6Hm0tEV1zM6/MRAI9BPQvTMtSe9RXxeUjRnMhdbxbM6Br5qQDAaDLX/mM7v9ztHhrtt7a51D22u5fDq9ev38WHqyUt5PJr6PrWVr/2QAAAAASUVORK5CYII=',
server_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJrSURBVDjLpVPNS1RxFD1vPpwPefoGG+1FZllWEkZSLYIWpiiJSEgt1VUQ/RXSMmgx4LLQRS1CgmpT2DLogz5khoTRwKaiMckZmtHnPN97v4/u+80M5SJc+OBwNr9z7rn33atJKbGXL7Tbg4WFhVnG2LjneSFiECu4ruvzQ+1/CTKZTDMJ7hmGcS0SicMhEYSAek2aRMLA/PzzvwmKmf4BycU0ISh4uLGFN7YFtEAIqxyOG0YhehMeDAgpyEiCk5njOFWDYro/KIW4Gk2c7w6Gk5AeU+CuDfv3EsrBS0h/WiGRpBBCYXBoWLWhDKhqO2E8HOsAK39AINoLzovYLn1GpfEyNitn0H0KqrqsVfdbVwkKH/saSHw9YpzWgSCEvUajdVD6/gIiOQHRPAhh/UIm/Z4qc1Wdcw5z5Eo1geC8VzJxI6J3QTKbPEzQeKkSmVXykJszaCqVcKE9Bit2EV4ooVLI2gwCJJ6OtpzbZxeXUCksQ4scU3/XMAcQd3KI21nECs/gch1Mi2Fdz2GmlAKNspaA8bmt/JuXxB4hLNnrSaNjLMm2S3ArBVjyMFbN22ChVjWDmeW72NpkKoVvsGMPvj7taZKM/4w2nYxXymuQ5ji2jT78yK+ryELKGgOH2k08uD+7cxOFxx83xA/Ev61xnBiew8YWh71hQXB/6nUDWWVZbSFQF3951D0sPNZpF3Nji87ou1dv07AsC7quIxqL0AZqyojXUvjJ/ZUO/VM9Txg9Prmy2AU8SaVSI9ls9pZpmmePdB5Fa/KgEtRb3t+WhH8b2m7XODU1NUTCO4Se+jH57IP2YUnb6zn/Ad9Qrbi4Y2W9AAAAAElFTkSuQmCC',
server_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJUSURBVDjLpVPfa1JRHP/Ma15JRQ3dsEAfBkoOIXqoJ3uqaOwhAgObQXuwnvw/Yhuh+LCnyNfe9jisEYI1RuFQGxv5stCciTKHzTnn7r19v8ccq1wvHTgc7j3fz6/vOWdM0zT8z9D/+WNjYyN1cnLyuN/v62kFrWIeHx/z+joUCj0aSVAoFKwEeGmz2UKyfBE9AkFVIfyRS7vdhnR6JUxffxPk8/l7DHY4HFdMJhN2vlbB6qqqQdVUItKgEFmv1xsdgYpX3G63+NHtHqFP4M+FHBGop/PO3WkRYyQBZzQYDGi32wNlRYF/6ppQ136pc7PPdcDMCoG4iA+FrRfyn2hVhDrvuWbu/9vBoFeaKGaCqcB1oT50oZ3TA93QwZBAkLCyMsjesOzg1X4C6pm6kRGG4MPDLkpftvCjvY/xcSe2y1tomto4dHeEu1QqpdVqtVa1Wn2+ubm5JAjYGoO5gaurbyHLBszNPUGn08Hkt0lcWnNiff09IpEI7ckgAnsul1sol8vOUwd8CnSZ0Grt4eHsLBYWX5CTbbhcLgQCAQYhHo9jd3dXsVgsb2Kx2DQRPBs6+JjNZm8Ui0WYzWaRLXjrNoqFPMLhMN1COw4ODtBoNJBMJrt6vT5EJR2r1SoLgmg0ejORSMxUKpUlIhA3au3DO24r5ufnwbeTB0fS6XSyJEnL/E19OBo7+xr9fv9Vr9ebDgaDl2lIRqMR9XodpVJJZPd4PJiYmOBe7ZGYLpPJfP+NwOfzSZQ5QIrLROAkkMRH3Ww2n7IgvRVWvkCRFepFgxw9+AkiS4SDy9ee+AAAAABJRU5ErkJggg==',
server_uncompressed=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIvSURBVDjLpVNNi1JRGH7OdfKaH5MTQkFhYeCiYTDaSAs3SVJQOxcFLf0TguAfCFwHQdt+gcPsIgcXxYiXXGQKxWSMFor5/XXv6X2P3stABkEHDu895zzv8z7Pe88RUkr8z9g5v6hUKq9Xq9Xz5XK5QxEU1VwsFhzfpNPpZ1sJDMO4RAmvgsFgWte9mFMSLAtKGync2wvi6OjwKa3+JKhWqw85ORQKXfP5fPjy9Tu4umVJWNIiIgmTyObz+XYLBD4Mh8NqMZ3OsKTkj8YJEVjOfJB6pGxsJWCPbrcbg8FgXdk0cXv/jqouN9W50X9VwMwmJTGIfwhLN6ofKJpqv9VqwePxKHA2myVPENwZsnsll8v91FjBuldSJTDB/sFd7O5eRqPRwGQyQa/Xw3A4RCQSEWT3CZNMp9OrjgKbQJFQZZZuGBXE43GMx2OnF7VaTUaj0V+MJ7vSIXCSzTWQCTRNQNd1+P1+dRdYwSZxuYnrHnBz7NvIjWMLctM4BnFz7UbzoG/BVm3lSgEfulwuVdkyOdmywUoFV2dl9h4T2Ao0UvC+VCphNBohEAjAc1GnGygcVf1+n+xoEEKoNcULRCBns5kCaJlMJt5sNh8Xi8WT4+N30N0CN29cx/1kCp/qn3F21sbp6TfU63X8oOH1et+SCpOUKQni/GvM5/MpOnxB88B+TByTySSoIsrlskkwF2MTiYRSIf7lORcKBbPdbqPb7d6jpFu05Y/FYi87nY78DZN2pgrwMw41AAAAAElFTkSuQmCC',
shading=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABzSURBVDjLY/j//z9Dzqzj0UB8E8puA+KnBNgVQNwEYoM4WSRqBrEfQdmeIOIhiZph7BYg3sZApmY4myLNIFcwUKAZzB5wL7SNeoFKXiinIDlXgohUpLRNiuZakOUwL3gB8TEcGh5D2TVAfAnKLgbiDhAbAJ0nqHy8Qq7oAAAAAElFTkSuQmCC',
shape_align_bottom=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEgSURBVDjLrZMxSkQxEIa/J2IhVmKlF/EWVl7FW3iNrS28wna29iKIKNgIvrfZzPy/Rd7uIu5jFzQQApnMN18mpLPNX8bhrgO396+bCoaUubk67/YGAFycHiEbCZ7fFz9iB/sAVslKU9Pbr3D30NuAEgzYcH153EWaDJBMpqhlApCCs5MDBFjw9ikAIkSkUJqQKDW2A2xIt1WGaPnUKiJa9UxPG0SCtAFktv1l9doi05TI7QCNBtJPg2VNaogcLWrdYdAARtkItSRRW/WQqXXCINQaKbsZjOdKNXWsLkFMPWOmSHWbHnjVxGzJ2cCSJgwMLx9Jji+y+iKxNI9PX78SV6P7l880m81cSmEYBhaLBX3f/5rDMKzjpRTm83n3LwbfdX8jZ1EmeqAAAAAASUVORK5CYII=',
shape_align_center=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAESSURBVDjLtZMxTsNAEEW/k4gCIRdQcgNuwQ18CG7AOdL4AJyBgmNQ04JE4Q7REGHPzP8Uuw52nJggxEqr3dFq3vy/s1tIwl/GYu6wrusf6cVQwf3jZhsEAQqIAJyA5/Xm+rQYAla7xIuzBdgnE4i8ksLrW0wUjAACQH4nkqkqqRRP88eAXi6VJCeYRrBZgAfQvHPrl5H2oQTlvluUdHBWVaW5c0mpC+uHZtIud8GdMCdaC1hLtEZYR3QeuLu9KkYWLs9PsqLsewAwX8JNMCfciaeXj/13QAkMwCMnWwK4EeZCBOEhBDkFULlqKFcLuBFdVuIukAJ3nv6q9/vcfB70653gQei3XSjL8rgu/NtvPGZ8AUm6WPSP0fwsAAAAAElFTkSuQmCC',
shape_align_left=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEkSURBVDjLrZM7TkNBDEXPixAFQimgpGYTdOyCBaRjLTSR2AcFy6CnoAOhdAgpeR9m7GuKebwkSvgowdJo7JF8fG1rqohgHxsBTKfTnSkHAF3XAXD30AwgFyjAHUxg/T25PKo2AE3TDA+nxyP0lSzw/paClzffrqCuawACkJaJUqkqRYn9mxYWi0WR3ctUFMkFFmuwHwHmMHvX0K+8+B4FKn4BXF2sD+jPW5jP5wDc3M821ukemInsQU5Byk7OImVxe31erQEAzk4OAVAE8hWACXNhNsKyeHyuN7cAENEPzgN3YRZkE56D3MfuovvQti2UauaBWe+byL3vpgEurQDatgUgJfH02pKSk630mVORn1PgsUxckRyMx+OICHY51b/8xn3sE1maQHPqA+BuAAAAAElFTkSuQmCC',
shape_align_middle=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEwSURBVDjLpZMxSkNBFEXPD2IhVmKlS8gGXIBdsHQRkt5dWGYfFmYXtsHaQkTQRlDIzHv3WsyPIfiDSh4MU8zlzJ375nW22aX2fhPc3D1v3JA21xcn3Z8BAKdH+9hgm8fXMuzg9v7TBpRgwIbLs4MOQDYSWCarhgEpOD4cIcCCl/cmzDRKkEzKlNgCsCHddhlWugiRaVIiE+oyhwGRNJs9IHtdTTeITOZPByOA6XRq9Q5C6zWZTFyLqSFqiAhRqhiPx94AzGazbuWgLaMU8/m8KyWJFFlFDVOLWCwWP9sYakHKbhn0Tyghovr7CXVbiJki1a0z6E3WZbMvNU0t3hKi4ektyb4jqx9eQkQay62V3gK4Oj/shn5hrUmGkVcQ/W8WIszD4weyBs+7XadxxI71BUieEw+8ru7iAAAAAElFTkSuQmCC',
shape_align_right=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEjSURBVDjLxVM7TsRADPVEK4oVSrGU1FyCjltwADrOQhOJe1DsMbbfYjsQ2g4h5cvM2MaeT1BIJNBSYMl69iTv+TOJYWb4ixWnEquqCpVX3x887bqxJSQAkgwRwEvsE97drM0wDLAooHZxXgBlsiAmJDl4ecPwTtd1ywJaXsmZqLFP5JBHPrRtuyyQ29Xq2nIU44mYWtM0ywJKOr7TOC9hjIOAOvwgcHu9Nr+5hYnAw/bI81EYvLTgBJ1lsA7BORIkeLy/MnVdTzu43JwFJPmwdFGjgNcRxH0BXsj757i8mQBzWhaqK4EDGZ1gyvV8+KD5LeRqPmCKQ/UYo3oSJ6L5DqwlOLz2ghiq6pzOxvZ1fuQvYra+73PrfJKXZcmK5t/+xmyf9PZAc1Cvzt0AAAAASUVORK5CYII=',
shape_align_top=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEoSURBVDjLrZMxSgRBEEXfiBiIkRjpRTyAuZGH8ALewmtsbOAVNjM1F0FEwURwZ9vq+t+gZ3aR3RFEC5qGLv6v11XdnW3+ErsAs9nMpRT6vme5XLJYLDZW3/erfCmF+XzeAXT/QgBwc7ewASUYsOHidL+7vn1eVzCkzNX5cbdhkIKjgx0EWPDyrpXu5HAP2Ujw+LrcTmBDuu0y1LV+JVaaSG83qAnS2iBzPDdZQTKZIsqEgQYC6TtBraKmUJoqUaL+TNAMjLI5RIhaW/VMTxNUtUbKbgTDFT7DK4pMU8bExhRSpLp1DzwaJFFFDhQRUwSGp7ckh4mM7ytKUqNVrzIREwSXZwfdtpdWwsRQXWpT2WowFRHZxNl6I+l3BvXT3D98bAjH+PNn+gIL+yQjrYYUIQAAAABJRU5ErkJggg==',
shape_flip_horizontal=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAElSURBVDjL3dM9SgNRFAXgM/kxPjQRLQIWFoHZTSAuwNbOeiCNbsBmQBBEmyCCLsJCQbEVwcI0KUxpKZlJ7jnXwomNipFg41nA9947977I3TFPSpgzfwOc3Y3D6W3eB4C9i2F/9/w5/AogPTEhBgDJY8qTmYHeTd4yedf4Xq4kkOomvUFrJsDo6fpKKUwBk2NjrRKMSn8ETq6zdiNEnUYtwhQghVAGVpfKnZ2jp/a3wPFVFoye1msRcgNM0xsJrxmxWAbMlG4fPIYvAaMnzXoprpYBo4MqnmDCaCyYEc1GNTYq+QQcXo5aRu8uL0SYFKdT04kQ2ZgY5QLkINXd2r//KLRSFJVSCA/DCeiABKhYcaMweMkgCpRD7kHyFMAmAET/9C/8Jm9+37CM1tkN3AAAAABJRU5ErkJggg==',
shape_flip_vertical=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAE0SURBVDjLtZHPSgJRGMXdzmPM88w+gqBF0KZF0CIIstkkhVhSIGFYYWHSHxKmCSdhYAiCIEJTk4ibNJbUA1jjXDenMy+gzkSLD+7i/s4533diAGJ/mVCfs05fzdieEVpg76avENYJe1uVH4QSIKwRFqUHH3ZzgI3y93gC+VtfJWyc3fuw6hL2k4T1KJG8GiFQuJMKYZ2wZ9YknNYAlYZEmW+zKrF22RsuQFgQBmODOyPYOYgdOAdwwughcgsja1w56WocwcFy8QNLhQ4WD10sHLQxnxOYy75gNvM8PAFhhbBO2Nu1PlF0vrB/3cWO+Y70hYuZ7dZ4K9BZpbOROHWxWeogdf6G1eM2ptONcDdgbI2xRfzoFfG8wFSqHv6IjK3QWSfsTSZr0VsgrE6sV43/qzHK/AJ0lPqXO1KzBQAAAABJRU5ErkJggg==',
shape_group=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG7SURBVDjLjZMxi1NREIXPC9HCKGGN2MgWdio2utZxA0KKVPkBwqaR/IS0dqYx/0LIVpsuBMFKA5LdQotF1kJWcMvYmBfemzPH4r4X3+pDMjDM3MvMx5m53Gg4HJ5L2pV0OhgMHmAL6/f779I03Sd5WgGw2+l0QPIetrQ4jvd7vR7iOL5fJXkymUweXb1+OzparCQATiCPDsAFkMDz5rUIANbr9dvRaPTMzN5HkgAAR4uVdmoVQKFJHhqlEC9+Og6eBkDRqnlCD8V5Q+4S4A6Yl4+zAeSFyiS7/wVhOWEzwuGHlW7eqEACPp0vgyoJJCA6zAE3hxlgJF4d3I0uKTABLIxwZ+dKUOECXXCGSApff6QlO2BBNpWdBXcHHSAdZAAkqZcABFwsHXQPxS44HSSyuz9K0lT/AopP9PJwKZpvJNMdbgp7oMOsBFA0M8EKkjc7yHIz/h+QJoRZddNoVLZMhxOXFXS73UWSJHskj6fT6RMASM1x9v0XUgdoDiNgRjjD03588wK3Xn+RmZ2h3W5rNpup2WxKErbxRqOh8XisWq2mqNVqHSdJ8pjkyXw+39vmN9br9c9m9pDkt98JJaJgEg+kbwAAAABJRU5ErkJggg==',
shape_handles=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGsSURBVDjLpZIxixNRFIW/kVHQQlwEQdgUdkkgTX6BkMIi/yJY5geE/ISAZIuUSb9lgnYWNsraaOkkTSSz6MLWJhmY9+7ZIm8mCUYLvfDg8d49h3POvZEk/qdigMFgkEqqSEp6vV79b4But/shz/OXzrlkMpnU4/BeabfbzGaz2pt3NzIJAd4DAplhAm+w3W7pdDqMRqNaqcB7/3U6nTbjR095fnYfGQiQCRPogOT2yTnD4RDn3CeAqMhg8Panzs8e7AAGkpDApJLITJiJ5U3GoPMiKhUQ5HoDJMwCsLjbMUnu7ThEAHnbNQrMDBkBYCWwIHTuBIEzMC8sNO2ai7vtlQjM6XcCc4Y3lT514FnBUqHEnSJwDry30v8xyV6RBM5OWfAe7+OQegAqjPIgQEl4f1LB7qOUa2DB++E4FbIqKpJEq9VK8zyvPHz8jFedi6PED6VL4vLiNd+Tz3jvkyzL6vcAsiyr9Pt9bn8sWVyvWVxvmKdrFqtfzNM189WGZLUmSTcsv10xHo9xztV285doNptfGo2GVavVK4UF+tOJ4/h9FEUCPkrar/K/1h35Nbw8S+DdbwAAAABJRU5ErkJggg==',
shape_move_back=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEdSURBVDjLjZIxTgNBDEXfbDZIlIgmCKWgSpMGxEk4AHehgavQcJY0KRKJJiBQLkCR7PxvmiTsbrJoLY1sy/Ibe+an9XodtqkfSUd+Op0mTlgpidFodKpGRAAwn8/pstI2AHvfbi6KAkndgHZx31iP2/CTE3Q1A0ji6fUjsiFn8fJ4k44mSCmR0sl3QhJXF2fYwftXPl5hsVg0Xr0d2yZnIwWbqrlyOZlMDtc+v33H9eUQO7ACOZAC2Ye8qqIJqCfZRtnIIBnVQH8AdQOqylTZWPBwX+zGj93ZrXU7ZLlcxj5vArYi5/Iweh+BNQCbrVl8/uAMvjvvJbBU/++6rVarGI/HB0BbI4PBgNlsRtGlsL4CK7sAfQX2L6CPwH4BZf1E9tbX5ioAAAAASUVORK5CYII=',
shape_move_backwards=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAD4SURBVDjLxZMxTsRAFEOdaAU1ogs0XCDKkTgKdBwFcZooJRINK+UGNJtvm2I20WQzEcUWWBppUvj971hT2cY1qnGlDgAwjqMlYT4kQXJz77quKgIkoWma3Sm20ff9/gYkMYNK5rquERF/Ay6NuUji9eNo0hCBCOLt+anaAC6N87ckPNzdQDJI42uM7Qa5ubRBhBJAxjRxDZiz7im1kcyiEeE1YBiGYoV5tRECleKcckDbtku/L+/ffry/XbLOKyu/C2AeIVcEVllFL1MlQEqgIMqA00SQh+JU8QyxEaKLgCmEz+MPQgApkDj/g9Q/bViCgQoAqn9/jb8mSSzHKz3sXAAAAABJRU5ErkJggg==',
shape_move_forwards=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEPSURBVDjLxZM7TsNQFERPrIielg5FFNkVy6FhN6wiG4hC5AoJVkAR+84MhWM75FNRcKWRXnPP3N9bJOEv0fDHWAK8vn1NZSghAgUsIwcpWFAlXp4fFxcAgIf7O5LgQBxskI0NPkLaz7pegRLsIdnOiUDyAHDoe90AiDnhzHVMtkJVbgDKlK67WkEG23QV9vt9bGOb9Xq9WAJUeXY7c53eBvVitXoiCdvtdq6gaoBccx3bsUMJJNE0DbZnQNcLaXnV1TpCEuR5iJJmQF/m/eObOvY/DNXT/pUQmwDj5Y4VkORCbdtGUrqum3Q4HCZVVTabTZLMh3QakkhC09y+9F8tnIdtdrsd47puCWDx77/xB7F6hU6PdBGYAAAAAElFTkSuQmCC',
shape_move_front=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFFSURBVDjLfZNNasMwEEafHfcA7coUsigEssmZep2ue5VCobfIJgsbuklLS6DrLhJrvulClpF/0gExI8v6ZkZ6Kk6nk0siH2Y287vdrmDBKjOjruulNdwdgMPhwDWrJAGQ/HRzWZaY2XWB6WLamMdT8cUKAJ7ffobY3DEDNyHVvL5/eBCEYDw9PhSzCooifru/vcEdJJAck0dv0b9/hXkLTdPE0/Y7TOByJDAJM5AJ60XO3bjlarvdDuW8tN+eMsl82GQSMjA5XedjgXwSJCwIE1ifNQkl300ryCddJ7rQZ3Oh4ASlFuJthOC0besJsLHAxQihGmUcWupHMGez2QyAjQTOF9F8/qLQt2Px2mJFjguciHsCrMjBye14PPp6vR4gmgK2Wq3Y7/eU1wjLAVuiE5ifQW45YAmypX/+FRgA65/1NJbEH0d3cad+jVEKAAAAAElFTkSuQmCC',
shape_rotate_anticlockwise=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIjSURBVDjLpZNNaBNBFMffJNumS61t1cRYUYwGvHgRj+pNUYmkh54UPFnEgrESaA0oloKC1UuEKmoRD6HgoSIRK1ooUhUMoYpaBWvS9IvmUCMFYbs7Xzu+bU2s9kOwC392Zva937z57xuilILVPK5/BbSkmtzR16di/w0QXNzhlB9f7jtZ6QjNA41xf8WmZoTAyHTuG6fsLbX4c2axh4+b+iZXBJx9cfLyxgr/hR01QZC2BIMZUPjxHfKFPAyPfk0ipGPgYuqNtlTy+cR4iG1g5czFgEkGM+YMaO5y0DwabNu8HUCR+vS7Qdgd3RVZ5EFrYlyX0o7fPpJozeazne9HP4BlUchMZIDimwoK671eqPPX1VOTNiyqAJNbUEFn3H3sUeRo54E1Q8Of9mDJ/Xj+hv17921xKqmurQFG+aE/ANH7uYC0VQwBpTXTsBoxOfaqLX0FS4anvc/OcSZAoKSQh8ExsahIVybZ0TOmTt/6opx5zyBLogILY/5WyYNIVza0rtIdLiP4739XELYVfH6QZpe6U0xftpHO3M2gcTLuwZlhyRIAk2Gnj+i+KtIupPp476UVWhLgGOerKgtyboPFbCh6gH7MQao9BLbWkuBanTy52W8mb/TNBooADc87Zxx2FJiYjDstAAAwoXANwO0i4K0kzo7hsYI4eK3XuIp51zUsN44DPTc9+2tHlD3fnQ5saAqh9vy4KATrGNuOYSfIaq/zT68cX09iiVY0AAAAAElFTkSuQmCC',
shape_rotate_clockwise=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIzSURBVDjLpZNPSFRRFMa/NzPaPNSpsDQVF4JUi3YtomW0MBCchbSxhS3chAS6mJpFBYmugpjSRS5cWAQFSUmBGIYiCbYYbOyfOAqKgwgqRNF79953zzktxoEspyAvnM095/zOdy/fcUQE+zmhYomOl23J9mcXw/8N0J5u8z09+C+AU3hC58Tl+kDbVqODpkCb09UVNUeNMciuLt0b757q+ivgyuv2s4EOrteW1cZj0YOIlrpw4ECRxvsvGWSXlvpme9M39gKEN84s1xtt+o5XnojXHapDAAsWxpa3BUcc5NZz2NzYfLf+NV567lQs+8cfKN+0HnNr4pVlR6CshlYai6tZGN8inZnD4kJ2YO7ux2sAUjef5NzfARHlqabyynIoq/DN+47x6fE1rYKRQJvzRgfpD/cXrgIAszSySAJAz24Fnr7wNjOD529eYGxyDL6nR2Z7093KN0+1CjoKhcwMIk4mhlcadkkQkaLxYNJvGJjwRkUEiYcr0v8qJ11Dy6O/1uzpg6Fp5Q5OqVuW5JMlaQEAIoYbBg6XhVs6BxebixppaFo1W8Z8VYVz+2R1xLWU94klxg9FiIYBaznV0f/ZBYBIoXF4RjdYklTMdVoqDjgoCQOBFRDn89YyfMMgZlTFShrXtv0EgJ7I41njWpKEJUlWx0JueakDQwITACQAcV4BEUGZEAqKiDh56U7mUcSSzFuSRsvA6jbBksBSfjKxYKcflhgr2wpMvHMvLrOknP2u80/X2WfmmbX8IwAAAABJRU5ErkJggg==',
shape_square=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADzSURBVDjLxZMxTsRAEATL6EROSsDD+A4xv+EzJCQEiA9ccjtdBF7ba9nZBWw00k63qmd2J5V7zgN3nstSvH/8rChRBKoAwYQIlbmuwNvry7QzAHh+ekTEgICRCA4mCXz9Xo8EpWgXBFS003SjRBKp20mEqhkRJenCpc7e5FY5GliZG4UkGLogq3AxbO3EoAVSkt40Ny91NhIhzaNBWqi45nTIbI+0kLQzg9agKmv+vclGpNByFqGKqkufehfaVzkMUKXqlGC+WHED6dnHddpndUJgn8F+4iP68k4G/UgQPr+va+YkWBIgtT1vE8bvN/37b/wDV/Vlq22fLE8AAAAASUVORK5CYII=',
shape_square_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGtSURBVDjLpZO/a5NhFIWfL1bNoFQdREMRRAo62CqIhOImWRz8A9xcNU4uFd1asEs3cXDxX+ggFkTQoZRsUrGDCDpoY4VSEoPRku+e4/B9+UUyFHqnM9zzcM77chPbHGQKHHAmumL51c9eFNkYiAAMlpAhlOkQLNw5lwwBAEonj2CMBQYsI4MHIBJ8/dUeTRA2dm4Q2MbO0+QgyUgmOmMqRGQRsZFyY1drGNIJjQIcyhYNkrDIDeoZu8A0HQNIBQqjfClb7mr1kxiUehSgVITc6+mBzs4rdZOk4wBpChHq9R+GQLP1luaf9/zda9Bq73HrUf3J66cbi31ABBET+avnRmdfudNYpThZo3LtOlOnpnm3uULt09ZCuVo6UegnMBFGKUQ4+5UQaZitnRWuXJwlCsHs2QqRdChfngO43weECZlUzrWIyJLstrY5nBzj9qUHADy8+YILp2cAigNvID5/b/c6S8JhBOz+brBZX2ejvsZ85SVLb+5SPHQU4F+yn2ssV0uLU+cnH8/N3GD6zFW+bH9g/eMaP741l5P9nnO5WloC7gHHgRbwvPasPv8fJZl0Xd9fi4EAAAAASUVORK5CYII=',
shape_square_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGrSURBVDjLpZM9a1RhEIWfu3eNYVcLiYqrmA8QjPgrBFGws4uCrYKdVgFBAorYWFtYBe0FiQqbRlBbERGDhUEwxkRBiCEbZd85x2LvfrFbBDLVFHMezplhMtvspkrsssrt5sHCj44V2RiIAAyWkCHU6kNw5/J41gcAOHpgBGMsMGAZGdwDkWB5vTHoIGzsQiCwjV24KUCSkUw0h0SIaFnERiqE7V79kGZoEOBQa9AgCYtCoI6wDUxpCCAJFEbFUGu43avrxKDkQYCSCLmT0z2ZXURqO0nDAClBhDr5+yFQW3rMwY/zlH4uc6J6mPrLtdlz9XS/C4ggolxsvRC6dcra0hNObbxg+spV9k6dZvtDnU+vF+8unt2zVeo6MBFGCSLcukqIFGbfu0ecPHOR0S+vyOYvUVl+yuTEWO7MN3ocuNhB/8ZtU9lYYfTIFFy42c0+VyNXNtmzA/H5W6OTWRIOI2C6cojG++dUn13n3/YaDWDzT07krGY7+cY3M8fmRvZXbx0fS+VyaYXNX4mv63k0//p2ttN3fjszPrv1e/VaHtlE5P5ueHi+nu79Bx7reDBYdnW7AAAAAElFTkSuQmCC',
shape_square_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAImSURBVDjLpZNNSFRxFMV/M40TjdqUqZmCn0GURSSUVFIUVhtx2SICqUXRVqptRBERtGpRC2vVwo0QEUWaUElSRDEW5EekiU1+RKKDOuPMu/e2eM+Xpq28q7P433PP4X9OwMxYzQRZ5YQWwO0no74UNcMAEcDAVFEDUReLwrVTpYElBADFG8MYhikYYGqogS0iUYXB8bnlCsQMM29Bwcww89R4RKqGqiGZFSyIuBIxQ9VbXMDqkhTIG6oTbdTn76LnwSsLpOJNPoGJug8NVBVTvKuu9GJrpza7l3Vlp4lW7GFqaBt97S23fAJHQcVQ76p73cVb6KSu/Du50YNMfuslHEiTu76InE2lUZ9AHUXUfJ/meS7gHYcrBthQeYL50VbCkQDDsT4yqfTvVHLmwF8FDoio71/VyNe3HKv6QLSqkVT8HsGwQ1ZOGTI7TGZ2rn5vc8eAHyRHBBFDVFExsic7OFLynujWRlLxuwSzMqQT5fx8GeNh4jw1zR2xJb/gOIaIm4Hs6U5qi3oprG4gPdbCmrCRmi5jrOsjzyJXiU9FlkfZEUPUCP54zP7CfsSKGf18ByXF7GQJI6+/8DRyhUQwH7EVcuA4Sv/IHEeTbeyou09/60UGurpZW7qbxMQ4z0OXmJjKwXSGxfUL/NvG6+d2Zi6fvRAiKMRePGJ46OtMtCCv7viNnp6VyrSM4OShvGRN5ebQvu0VWNZ8d3Li15mGm58G/9fGP3sKXaMRKZvBAAAAAElFTkSuQmCC',
shape_square_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHMSURBVDjLpZNNS1VRGIWf67HQ8MZVEtLserNIsk8JA0f1F8JZ/oRG9RNs3ESaNQscFQ5KCoIiopGhhFDgHZRgZEIRSN3PvddqcI5memZu2Ow1WQ/Pu+Et2OYgp4MDns7tcH9hY0dFNgZiBAyWkCEqzVFw71a58B8AYLD3MMZYYMAyMngXRILPm7X9BtHGzgoC29iZTQaSjGRiO2eEGFNFbKSsuJ31P6Qdtf8THZXSBVFC0Sk0iv7CCtcOPSDxFlEmhBxAEFlJmU2aC7HBaZ4zXBmn4tcoGgXvBygoK21D0nzSbxgsT3B0YJyB8I6euEbIA4TAv6JMiKJbGwyFVxSPlYhbTxm6NM1IWEBq5wBizMrp/I6i3HrB4NhNaCyz+GiOnlKdvvoSw8lKnkGqrAAxmlL7E2f6GhR7a6j5BSzi7/ecunGXiWSexdnJ4h6DTF1GsU2l9phS+QqqrWDVuTo1ilrf6Oqqcu7ydUKzdWePgVhdr7G6/ofk+0sqI+c5UvyBw08oJCzNVwGhRpX+s8PEZvv225kLY4W8bVycnfx6cXruRNKxhsOvdCF2TZ10j7L58QPVZzNPOvM2LDRbXcsPp+qWsbTreudFwvbxvyYHcoBEg0hXAAAAAElFTkSuQmCC',
shape_square_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHISURBVDjLpZM/a5NRFIefxCJYSsFQsGjNEBGsVXEQFEQHxQr6BaSLX8DR2cWSSUe/haOroKUVFC24CDVoMcUUrRFikubtfe/5HYe8+VOSodC73B+Xcx6ecy835+4cZuU55JroheevtvsqcscBM8DBJeRg6mYTPF0q5vYBAE4eP4rjuMABlyMHH4JI8P3X7qiBueOeNQjcHffMJgNJjuRYOmYEs64i7khZYy9rPyQ1jQLc1C10kISLrEH9xh4wxjGAKJA5yoq6xb0sNhpLJGmTc9MvUfTRZ1QUJsfMMVO2ZzlCEgNzM/N82LlPGpMxBhHM1J+/0niAlBIUCTFyqnCW+dmrNJM2nzt3uflsb2rl8d/WAGCG2UR26xAVuLPwEHNhMoRTa2xxce46rdDhU/Vt49JyvjBk0FX2zCCJAXPxo14hVSQqJbWUf3tNLp++QSvdzb/fXK0PGTimwbMlMSFa5MR0kSjDXGw3qhSmZlnfWmPt2+pOiCzmep/p0YuKmwY3Xz92jyQEggJJDJyZOc+10iIfq+9YqazTTv8s1Mr+JXfQ33hhOd++Urw1+ebr659B3K6VfQPgwIDSk1zbdGSyIyv9Lvtm7/w/ZAmOh9Ycu/MAAAAASUVORK5CYII=',
shape_square_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHxSURBVDjLpZPLa1NREIe/PKhSNK2g9UEIKqEFRSgooi4sCJqCuHEpduEDwZU7/wGhu27cdKWoG7WQrsSgKAV3SnFTAqZCW22xlAbTpL0xyb0z4+Ke5mHrqgMHfouZ3/lm5pyImbGTiLLDiG+KsTfLTRQ1wwARwMBUUQPRUIvCoxupSIcBwJF9XRiGKRhgaqiBtZmowtxKdSuBmGHmChTMDDNH44xUDVVD/G1aEAkRMUPVFW5q7TTxRbcamGiYaKCqmOIK1FEp6fIzDu36SSwYBO53biFQUDFE1NG0tIqSLE1yOr3KicwdMse+8HW0f7iDQANFXI8tklAnN3KcS03TM3AZ70eORG8KS61lcw+O320aBAGIaLP/dpM+8vSeuopU8+zen6QmSxweGuwOvNJ4y0AEkbiburvdwlWWgz780gLRWJFI1KMrUYRalY3leqyNwBBxxW76Cb/AmdgE3QcEDXyiugbRvZjf4NurQi1fqNxrIzA3g/D2ZH2K/uAdA5eGsMZ7IrLOzNsa5lUxhYW5ys2Rl/PZNgKlsFh1g4OLBz9x8kqGRnGceLyHmY8+r5euU9xzHgNePD+bDffvHsy/Z3Fy+HN9ZdS8wjWbfnxh/cmt9Mh2efH//bJf88UPlfLU0epv78/s7OrD20+/T2yX9xfxXpKxy4ipWwAAAABJRU5ErkJggg==',
shape_square_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIUSURBVDjLpZO/T1NhFIafQss1aaSpUmgbCxpriE3q6Ayjk5oYpf6ILk5d9S9wwsRRDVFXN22ijg6YaBxMiUPjUBqCLe2l9goxcpHC/c5x6C20KRtneofzPnnfk+8LqCpHmSGOOMGuePLB3o8iqihgDKCgIoiCkY42Ao9uTgb6AADJ6AiKogIKqCiioD0QEVhpbg8mMKqo+gYBVUXVT+ODRBQRxewdUsGYTkRUEfGNXS39kD0jgwA10llUEBFU8A2yb+wCPe8QgCcgRhF/qbPc1XKQREE8HQSIJxjR/Z7a01n9St0k3mEAzwNjhN12m83qEqHtFeLjJ1ld+8Wx1CzD1ihu5R2nT8XY+eeQz68+LJVKzw4AxrC3C8vf3pMeV27cu4PrupxdW+PTl8/sMELu+hVGQkFs26ZYtB5Xq9VYoPuUH7xc0cDmdwK/i+Ru51h49YZ6pUgiESebzWLbNo7jYNvrHE9cIH/3EoVC4U9PAmXj5w/OxcMAnLl4i+XSV+bm5ohGo2xtbdFqtXj6fIHRyWuASyQSsXpuIPw1Eep2BYCljy9ADPPz84TDHajruqgIu5XXMHuZer2+E+j9jZlM5nw6nX47MzMTT6VSEgwGTzSbTcrlMpZlMTU1xcTEBJ7nbdRqtaHFxcX1PsD09PRwIpHIhsPhQjKZjFmWNQwMOY5zH6iPjY0VgFC73TaNRqPluu7V/620dUeTJTHnAAAAAElFTkSuQmCC',
shape_ungroup=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIsSURBVDjLpVMxbBJhFH6HR1sKNG01jbkEGYgDjMLk0GgYiZODBEZ1cHDASAJMLiwkspAwECRMOhkhcXNwUTGmZWhs0RqtXmutBrGDHoT7//987wfaQrv1T77/3d3/3nff++6dYlkWnGaptOVyOR2JPIhWKpUKTCal0+mXpmleEUK08vl84BgBLk8kEoF6ve5/tmJYggOQLooCY7fbhVgsBpVKxX+iAs55s1arXXLMLcHirE0WWbgJZKEO585dgFKpBIyx15MEysiDp+8M66zLJgvEEch7JNvdF3Dr6qxyogJao8Q3z4vQ+bUD6owbLl+7C/YZ1+CMC5kXDodX0I8gYrXRaIRsIwLZLya293SIRqPw++cOrG+3Ye1bB9a22rD3Zx8ePNGtXq8XzGQy5EtwTAHDYo6wO+ahWCzCmSknLLlVmHbYgSOzwEOKzgUNkskkoILmgQeJRELKcs6fh+Xr9+E9vlVbmMKWsBD7IvUcN04kiM3vBhTuXJR+yBZITjwehx/6Jux22CBZWDIyhuoYRQvYkMA0D4dPEmBfzUKhIOPtsEsxKZGKOBUJCT4iGcaxr1CtVoNHP83gbYeSDzwYXjPGj8/BaIVCIb3f73vcixrcuPdIFpIS8oOjH48f3oQvG29p+FqoOGCbHAx86Mlms+jHJ/i4bcAH/R/CgI2vf6G1ZcDn9QaUy2WaSv9wgKwx+Hy+Va/XKzRNa0yeEVRVfaEoCsl+RffKaX/n/x+oi531jRZtAAAAAElFTkSuQmCC',
shield=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJQSURBVDjLpZNLiI1xGMZ//+98Zy6G4zIJkyg0xEyWLlkMC6JkIYoke1kQUrKwsJGFlaVLsbBAuYSJBTEJCTkZQ8clJsYxx5y5fZfzfx+Lk8swbDz1rt635/319rxOEv+j4F/NzzeW5Xval/5zg5NE6c5yyYSkRZhmybRK0ra6prVIYujNBWS6JLOrMnWYtw6ZxszY+Ng5SfTeblNt01oqpTxB7WTCCQupm7oCMmMBQ2mJ6FMHUc8jkt4X1E9v4+vTk8zc9NSFADLlou728sTFx0GGfD9p3z3C3BKcC0l6zpId00LN3C0gT/H+Yczbyh83aGy71W/pIMhIi+dIi1ew6C3IAGHxR+LuMwy9PAAy4t4CMnsIEH4/hkxVXJ+ApeAMG+7AZVLko2pVYpAnyOaoDJWnAaWfBt5AAkuQT3GBB4bBpTjFoAgsrhJ5j3mzPwlUJZBSUAaIgBgRI58iHyMZmEPesiNyIG+YxQT1s8ESkIcggaBKISUgQIaP+7GKlUYamB0tPz9FOH4xLjsJFzpcmOLChKBG4FIaFhyh72U7cbn4bN72wvsRBk3rHu4cfHOdctdZaqasJ6hvIKhJceEgmbEhDc2HiHry9HXdwCfphlGjLG+TBwrXGChcJ9u4GRfOwQWDZBv3ERW7KHVeY/jzuw0tu953jojyr3p9umV+kM3lc82ryc1fQ6a2jnLXTb48uczA2/y61t0fLv7xC7/r1Yl5U+RtRxA27B/X3EbxwfmCTypbW/d8uDvqM/1NncdmTfJJ5aAllb2te7uHR5v5BtnsZt4skcPVAAAAAElFTkSuQmCC',
shield_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKISURBVDjLpZNLiM1xFMc/v//877yYMYwxjzzKW8MMG8YkMwgrLJRXFAsbj4VkUpLsJMoCCwuPokyShYRhgabbhBg3N+8xpGtct3ncedz7///v7xyLG9NEs3HqrE59zqfT+RpV5X/KGW348+GyaLylftQNRlXpaV2hKoqqLkF0uoquVdWd+VXrUFWGOm+hordV5K6KhsVKWEULp25qN0ZV6X7SqHlV68j0RHHyynBLasmvWAk5YwFBgx7SP8Kk4y/wu99RMLmR3shlpm2NGBdARYvTsZbk+LqLoILafoK+NtzipRjj4sebCRXOJ3fOdlBL4ulJxMqaPzcobXzcL8EgqBAkbhIk7iDpL6ACKOJ14cWuMfThKKjgdXegIs8B3N/HUNGsrvVBAjCCpMKYnAC16WxnPFCLEyomM5SsBHqGAVZAFcRHbYBxLJACE2DUA02DeFkjaxEr8reBZg1UA9AcIA14KB5qA9R6qAqIQa2ERgKsIOLhFMzADkTA5ILjg5O1eBTLo/X7ePqencRPCamhwh3XockZNpAzyTdXcMfVYUITMK7BuAHG9Xnw2dKeKaNheQN7tuyldnE1TORQ3b6qU38AVRueHxjsvE/yfTO55RtxCsbg5AYYd5Dbr5MsnFuLdSy1lauxJqBuQT3A3hGvrFbKBjruMdBxn1DpNow7E+MMEu9LEDJjWT9vPwAHV11gxqQagPwRgKmbXyUyqWR1b7SZ3shVrD8FzG76B7qJxsKceLALgBMtu/gUjwCkzb/S+PHS3HK1st9xxxwpmt3Iubs3eqMlRSX1NcuYVbGID10vCUda+fa577QZLc5vz0+fYP3McfEzTbu/cgzYAxQB/cD5trOxw78Ay4ZsAbSg5bYAAAAASUVORK5CYII=',
shield_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKSSURBVBgZpcFNaNt1HMfx9+///2VptppujuE21LoNOjcHY6D4gGAUcToY6KVgFdHTTqIICirqdvHgyYPozQfQg9DhQYaSeaiu1DGkaKE6mXuwtqVNSqNJm/wf8vt+zCGDIbjLXi8niRsRcR317x6crVUfENfhJNGYfFgyIeleTLtlOizp+YGdR5FE+8pXyPS1zL6RacqCTcm08fbRn52TxOoPFRV3HqXbmCUqbsNvPsjA9kcgHgQM5Q2S5SmS2jTZ6u+Ubq3w98ynDD894zw9MpWTxWpzy30fgwyFFvk/Z/Hl+3HOk9W+pLDxABv2PgsKrJx7Dwv2GD0RPVsr37csXwcZ+cpJ8pVTWPInyABh6RLp4he0L7wFMtLVS8jsJ3o8fTIBhkIGloMzrDOFi3MUEhQS1E1BgahQpttu7gAanj4FAwksQyHHRQHogMtxSkEJWAoICwELZvR4+mQCGQoZUg6KgQRIESkKOQopkoE5FKxAj6dPwTBLiUp7CGsz4DZAlEHUAZdTO1ejfnae9lIFN+iRwtjIMV739Mns/eZvn708tO8FLJ3HRes4n+N8Rv3HBTpzJQ6NvUFx1110Zqr8eub0q6cfLSw6SVz11/ghDe55ivLeUcLaSQpbihA1mHzuA+5+5h1KFydgYRKGNrPih5munrnsuYaCbVu79G09KgxR3jeGi34BJsiWOwxs3wVHXuEqf3wHsbk7nCSudfnzA/ujQnm2PPIE5f1HiIsDTIw+xD2PP8mmuVOknSXaQKsZc/4iC04S//XHJ3feomAvRn7TmzeNVJgbH29Yq1S+bWs39tE8rXqXK8txyBO97STxf85/uPvmkHVPWNZ9rTntX1pfXTwWBzccYi0IPjpc7b77L4KQewoBSj4ZAAAAAElFTkSuQmCC',
shield_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKZSURBVBgZpcFfaJV1HMfx9+/Zc9QtO0unOMdasEibGf65SLOCJWRX4p0gCCVddBVYF0FEF4Hsom668jIN6iJJvYgoRoOMeTBdEoORf2rqJseca8ftHM+e5/f8ft9P5+JcSJA3vV5OEv9HwiPc+/HlqbnRPeIRnCRq469KJiTtwjQo0+uS3lzVtx9JNG+eRaZvZfa9TBWLVpGpa+Dgb85JYuHnYa3s20+oTZGsXE/6xDZW9e6FjtWAoaJGdrdCNncZv3CVzv5h7k+e5KlDky6lRaZyVh1dWrP7c5ChWKdYvEBafhHnUvzc15S6trJi82FQZP7iJ1i0fbQktPQMn6tb8QBkFPOnKea/w7JbIAOE5X+RV7+ief0jkJEvTCOzCVpS2mQCDEUPVoAzbLmC6yhQzFDMUMhBkaRUJjSXNgK1lDZFAwnMo1jgkggsgytwyjl2tUotzzm+SViMWDSjJaFNJpCh6JE8kgcyoInIyUJB/7ohDp86AuZQtBItKW2KhllO0vk0sTHJBxcvgcbwFvAhsHHNMwz17qKePeDt2xf4tNR5n5aUNpl9tvT7F0e7h45g+W2CeV577g2ijGgRQ1QXZ3m+/yUafpm3Zs5NZ8eSFSltfQcm3p39ZsdR19FFefNBsnCaKOPW39cpLBCsoIgFS3md7U++QqNopr/cGM9SHqJo6xvTP9xLSt1kISPEwIbyAMEiUcadxRnWru7l8ux5zv85fsUHdjpJPOzGl1u3JKXy1Hu1u2SFx5snC57Bni3sHtzHxEyFsWujkz7wQnVEuZPEv/1x4tkNivZOkj724eObhpm/dGb6UH22Z8fA3u6fro396o091RF5Wpwk/suV44Nrow8fmw/vH2jcmYnWsW7ZYmluRIG2fwAIBqNZGcz/tQAAAABJRU5ErkJggg==',
sitemap=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACoSURBVCjPY/jPAIMMCgxmYKiAEAOKwhmMDGZ//v/5/+M/qhI0BW/+gxQBlTDiUPDl/3v8Cn7//4VfwR/cVkB8kfI27S0WR+7rm/ui70X7i9YX88O7whtflANh7ouUPqgCBsbZLyBGAq1hRLBiX4BZiODBxgIpgAWPOYM/BgukGCl4En9jsswYkIIHKIzBgiiABg9QGINlxoAUPEBhDJYZwhdmDDZYWAoAtTEEdnXdy7IAAAAASUVORK5CYII=',
sitemap_color=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEoSURBVDjLnZIxS8NAGIabf5JfURf/g2NxyCCEbAGXcuCWIQSC3C4hZHPXttopKI5S6KaC2E2umDRSg6SXwGcubeAuCeVs4F0e3nu43Pf1AKDXTPmpZfpc1K5e1e04rLBDRVEAS5ZlsE+yVxBFEdSinUT5lyBNU0iS5HBBnudAKT1cwEdawE9B1/XYMIxY6hHDMMS+7xOMMXEch9i2TYIgGLiuO7AsiyCEqpimSUoxFgTsap7nEf6Ku19RupimaYRj7aWRZbWguTRHZU4kWC3eCsKXGBbxpn7xMzbC9fMNFJ+vArtdjuAjW/CT2Qqe3tfw8JYI5d/5HfzMRgK7X01h/DVpCy4fv+FiuhLK9Poc6NWpwIZLBEOC2oLG0lRlCdbvmgLLsSRT/wAPFzLO5ovpLAAAAABJRU5ErkJggg==',
sound=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH0SURBVDjLxdPPS9tgGAfwgH/ATmPD0w5jMFa3IXOMFImsOKnbmCUTacW1WZM2Mf1ho6OBrohkIdJfWm9aLKhM6GF4Lz3No/+AMC/PYQXBXL1+95oxh1jGhsgOX/LywvN5n/fN+3IAuKuEuzagVFoO27b1/Z+BcrnUx4otx7FPLWsJvYpIM2SS9H4PqNWqfK1W8VKplHlW/G1zs4G9vS9YXPx4CaDkXOFES4Om4gceUK2WsbZWR72+gtXVFezsbKHVamF7ewtm/sMFgBJZhd6pvm4kDndaAo2KOmt5Gfv7X9HpdNBut9FsNmFZFgPrMHKZc4DkjHyi6KC3MZNehTOuGAH5Xx5ybK/Y3f0Mx3Fg2zaKxSIMw2DjT0inNQ84nogcUUQJHIfZquNT3hzx46DBALizg2o01qEoCqLRKERRRDAYhKYlWRK/AJdCMwH2BY28+Qk8fg667wdXKJjY2FiHaeaRzWYQCk1AEASGzSCZjP/ewtik5r6eBD0dM+nRSMb1j4LuPDnkFhZymJ/PsmLdazmV0jxEkqKsK+niIQ69mKUBwdd9OAx3SADdHtC53FyK12dVXlVlPpF4zytK7OgMyucNyHLs8m+8+2zJHRwG3fId9LxIbNU+OR6zWU57AR5y84FKN+71//EqM2iapfv/HtPf5gcdtKR8VW88PgAAAABJRU5ErkJggg==',
sound_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI+SURBVDjLtZPPaxNBFMcX+leoIB4VFFQEcXtIMXpILbQhYprUmGyym2SbX00aSSCGWrfLhmSTTQI9tDGBtlgIIqVFBNec7NGLaJGCvczBQKG5eGi8fZ1dbVHagiIOfHkzj3mf951hhgHA/IuY/wYoFgsuWZa+/DWgVCoO0GJJUeR9SZrFcUVETPGEi5w2AZVKma1UVFOqWmJp8edms4G1tReYmXl0BEBCU7k9MQbiDLwzAeVyCfV6DbWahmpVw8rKEtrtNpaXl5DNPPwNQIIJgTwIX+y6A+hNcCC3HRFquYDNzbfodDrQdR2tVguSJFFgDalk/BBA+Di/J0RA7vqyZMQV7zncIIN3thl6VqyuPoOiKJBlGfl8HqlUis7nEIuJJmB3zL1D3IJ110W7DjvNHGGHQa5awRgX1WgsQBAEeL1eOBwO2Gw2iGKIKngA6BG7x0ojyNDoD8DlmyAXBsHkclksLi4gm80gkYjDbh+DxWKhMA9CocDhEWr58Y10dRzByijuPxn61nSxIOeubTPpdBLT0wlaHDEtR6OiCeE4L3XFmYCwZn88+5zDy615vO/q0N5Mwqldgv/e2Q6TnIqykckwGw7zbDDoZwXBt2OAMpkUeN5nAjzKrf76xyrWP9WNNdSOAK0TwvXIqf6RR0K7DvABn0y1fwAYyV3Bq62n+HVsfJg3ACc/UQqaoOoa8xvRM/2S7oeim0Aor30nOzhOdOOcQz2Pss6bnY1orGle/eNfRzcXqL4atn/GgpH/DtOZnBuaeVfLAAAAAElFTkSuQmCC',
sound_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJZSURBVDjLtZNfSFNRHMdvzHqPIEIo8qEgKIrwIa4PE+ceZo0cC8Ymtq3duz/un7uz7spkxe022dx0e9OxgUqSQdlLYLZemkEhPvRQCGlxqAaW5kutovDbuTeUwgIjOvDlHH7w/fy+v8M5DADmX8T8N0Ay2WuVZenNXwNSqaSGmqVEQq5K0mX8zkR8Akec/loVkMmk2UymT1VfX4ql5ueFQh4TEzcRj1/cACCezu4lXxDE4ppRAel0CrlcFtlsPwYG+jE6Oozx8XGMjAwjJp79BUDcYZ6c9h6s2FxYbnOC6M1+GrkX5fIDlEolTE1NoVgsQpIkCsxCiITWAYQLcUu8H+SUI0ZOWEPLZhtIw/E5hs6KsbFrSCQSkGUZPT09EASBnq8gGPSpgMVW2zyx8bpFK+3aYlFrhG0BOaoDo1xUPj8Inudht9thNpthMBjg83mo3GuAZWJq19EdpPHkD8CRJpADDWC6u2MYGhpELCYiHA7BZGqFVqulsHZ4PK71EWbNh28/NO5Eqalm9V7ztpWnjftB9tbPMV1dEUSjYWr2q5EDAZ8KcTrtNJVTBUxb9ohPose+fLqTxuqzSXy8LmCm4xAe1e+4wUQ6A6y/w8t6vRzrdp9hed4xr4BEUQDHOVTA3WbNQpWakTMC4nbgah3eJRtB6y82PBLaVcO5HDJVdQ2gxF6dvYWf10p8lzrOH58oBbVRVdQEes2rDwV6H9T0+RyD91TEqwGtv97Uhylbai895vZ9rYh1eHt+Kxa4LbhvrPk2qddc2PSvm7bsjtGOL5XYSiLFrNS/A2oTmihPyHNrAAAAAElFTkSuQmCC',
sound_low=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGeSURBVDjLxdO9SwJhHAdwIdpqrKmxNZqKuEUhSIyixCI0yjPv1PM1z8KDEonrOPF8OzcVhYoCh2gXp/wzcnmGgiDX1m+PQsZR9EJEw5fnWX6f5/e8mQCYfhPTnwHZbMapKPL9jwFNy47QYllVlWdZPsGXQKGQZwqF3CC5nMbQ4rt6vYabm2uk08fvAOLfHzcA+byGclmHrhdRKhVxcXGGZrOJ8/MzSMlDA0B8MZ7sBqYMgKZl0Oncot1uo9VqodFoQJZlCuoQ49EhQLgo98SHQDZYiaw4J4YA3Suuri6hqioURUEqlYIoinR+ikhEGACP664ucfGLj04PyPKWoStT/6BqtQp4nofb7YbD4YDNZoMg+Gl8r0CP2HcW6QhiWTMCR0cSqtUKJCmJWCwKu30dZrOZYjvw+71vW7BuCr3VTZB5q0hmLWND4OAgjkQiRotDg5bDYWGAeDxu2pXHeIhzS0EyY540dBDfDzOhYIAJBDjG59tjeJ7t9qFkUgTHse+vcXph9NOHRFcd4bysQvP8EfDtp0yhbZqH//tM380L5ZG0STFOrDMAAAAASUVORK5CYII=',
sound_mute=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFsSURBVDjLxZO/SwJhHMYF16b+gP6GZiehwcm7hBORKLXzPT1SIhMUHCKO48TT88emhwchHTiEERQhTrlE1FIhQS1BGRTU5vr0ntgS6BFBDR94eeH5fPk+L68DgOM3OP5MUCjkg7IsPf9YoKoFJw1LiiKPJGkX7wyToCxMFWhayaVpxTHFouqi4ftmU0enc4CTGLEE15T5qYJSSUWtVkW1WkalUkartYd2u43zbBZPPp8lMGeuoKp59Ptn6PV66Ha7MAwDp6KIIcfh1u+3BHMzBXRXmOY+FEWBLMs4FoTx5LtgENuJOGxLtIrS9ToIITADATwyDC69XmzGBYiiYC/I5bJoNOo44vnx5CuWgcftRii0iliMtxek01s4jIRoeBk3dO/URhw+eo7QO0Ii9oIBx+lvLPvxwrKDnfW1JULCD8mkiEwmhWg0PFtAG16kvFIuvtqmU51RPixTRraCicTz/akmohXK8P8+0zQ+AXBHwZp9sfnqAAAAAElFTkSuQmCC',
sound_none=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEzSURBVDjLxdOxasJAHAZwwbfKkjlbJVMpJaYmxtSoNVoSsCLlekQSjcZNRUFFIUNxD5nqY7Rr+wiuX89M3a62lA4f3PL97n/HXQ5A7jfJ/Rng+/1LSsn72UAQ+HlWJp5Hj4Q8gguE4VAIw0GWwSAQWPl1sZhjv39Gr/fAB4bDAJNJhCgaYTweYbNZIY5jrNcruM49HwiCPg6HF6RpiiRJsFwuQQhhYAS7WecD7KzY7bbwPA+UUnS7Xdi2zdZPqNVMPnC6qPl8Cl3XoSgKZFmGJEkwTYOlzAc6HRez2RSu66DRqKNQuIAoigy7hmGU+EC73USr1WDlajayZZkZoqoKm0rlA807S6jeVoRKRRPK5RtB14tvJ8hxbGhaEWc/JLZrXisVKcvxR8AX6Irl4/8+03fzCbreyRfHFw9qAAAAAElFTkSuQmCC',
spellcheck=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHtSURBVDjLY/j//z8DJZiBKgY49drM9J3idhLEtu+xjvea4nLNqsVspnWr2S6QmF6+Zol2ltpq5QSlmcpxijMxDABp9pjkuMuu28rIpsMi3rLZFKzIus38mm6OuqRxpf41nC5w7rOJd+i1ngnUXGLTbj7Tsskk3rbL8ppZreEu7Ry1mWpJSvHK8Uoz0TWK5U/nYIg8y8rgPsl+l12P1WqgbTPdJtk/AtoWb1CkBdagnqyyWilawVM/Rw/FBQyx540ZGm/eYIg8P43BdYLdSZiEcYXeTJB/TaoNroH8q5OldVIhXE5SKUqhXSNRfZdKvPKVkOrED+L9d/8wN998w+B4XIL40I48K8FQf/O6+7In/7mbb35hsD2qjBKNDLU3ExjKb7pi1Rx61ke89+6fwBVP/jPXXn/HYHlYGiMdMJTe1JJc/PgHQ/X1xQyplznBYuFnmRiiz062nPfof8DSJ/8ZSq8/ZzA9KIEzIQE1Vvuuf/6fufv2M4bgsz4MxVdPui8Cal4C1Jx/+RGDPqpmTANiz7MAvXI+bO2L/5ZzHvzP2Pjif8DCx/8ZMi/fY9DcL0FUUmbwPKkg3Hr7T+WOV//95j/8z5B6/jaD6l4JkvIC0J9FTtPu/2dIPn+PQXG3BFmZiUFzbweDLH7NVMmNAOGld33BRiNUAAAAAElFTkSuQmCC',
sport_8ball=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF8SURBVBgZBcFNiIwBAAbgN5etvQiJOPiuyk/YwqLk52v9NbTtarDGNpbhoCknHKgtOXxDIU60mVI4OOxVbtuchaTkoByUMpoc/NTjeSIi0irPd5q9U/2J/uHe7s7mUkQkImeLi1VrcM+cZ56oXLJusKZaVohEThcz3fve+Oaz1+bdctdNNUPdFBKZqu54658v2q54pKvlsmt2SCWpleODl75aMKvtu3MWNJ02oymDlCk7N7zwR9tHc9pm/TDtpHFNa6WT0d4d930y54E583inoe6YmhHpZX3/oVnP/fTKvF/+emzKpJo9tkk/Rf+2q9qe+uC39x5rqBt30E4bpJ+lvUuuaLmg5ZymhhMm1OwzarX0sqiz33UN06Y1TKmbcNSY7UYMSycpM5hxxnF1dZPG1YzZaZOVMkgZSbXOpGMOOeKIA/baaqNVhqSSSIp01xhV2mWrLTZZa7Eh6aaQiKRIlcESK6y23LAhGaRKIRIRkZTppJd++umlk1JE5D9AhzZjNC9N+QAAAABJRU5ErkJggg==',
sport_basketball=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANjSURBVDjLbZNdTFtlAEDPd+9tKS20HS1/tbAuDNyEEMffIv5hMkW3RU2my96ci/HF6IPZgz65LEZnYnzQxFezJYtREzaJ06hkA8zGgs4hIOOnAyZdadpy296WS2+5vdfXYTzv57wdYds2D1JQE62amjyiZycfMwvzdaWsXsL76Iw/2HGxubP/Nv9BPBhQ48uvpefOfewPtTRKspui6WJ7s8i2niMRX425Gl7+rC4U+jKyr8vYEdALmje9On9RMc8fLd+W2TU4SNlQEdsq+XQcWYQo5jdI3/2DRF5bjHS98VzHoZP3AER+c1NJR7+9oYi13kBdLxtjo1S1ezDUGbJTcTwtu6j2N6L49iM5aogv/cnC32tzT5w61+8NtuXkd996/Z3UF2dPNT1/jPLmIoYxQeqnW+j3szQ+eRCH0we2DcUUhhqjOtSNc2u9Njo96/d4nCPym6+Gf/D19bgzP49R3eykkIyRm82w+/ABHLKAchnJtLDMEnapRDGrU9nQydK1S10YxSnFKs4FvOHjbKjnSc2ukJ9XaTvezf2bsxRzZYRTwRV0ozideNwKFY4c5epmlGCHdHloZFBJz8YI790m/MIzLF4YpmyBwxUk8tTjYJYQNmjJJJZhkk1qbOdKONYnqfVUoydXDyj6+CT603EU28LhdxJoDXLr61/pOjaAjMAWMr7aemzTwOtzIUwDXAGyWeiPVOWl0CsDaMsZ5i9fhapKwi3tNPXsYeL7CdRUHiHZ2AhsIYEMWCaSqwajkCGRtdYU05T0itYGd1W6HT21RCGToi7cgMsXYPq3eYSQ2d8XwV8FkmUiXB6E4mHj3iqFItfEX9988lFmc+X9UGA3lcY0mfUkbY+EQAgsRSGnFrnz+xqYFhVuiYbuw8yMjPPdldFL0aJ0UqTuXK+6MfT5dUPb6jx64ggz4zd5OLKFSyqBDbYsYQsbWfFTkFr5Z2KSqxeGNK2jvv3MUCImBff1F7yhrpcWlu9Gp0Z+IfhQE/nKXjYdESylHlHRjBzsB9ce5OURtOiVFa25+tkzQ4nYjpmGPz3tSSSi79VkZ9/2+lt8NXvbUO1KPLaGV1/HiI0Z0ThfzSWsD86Oa8n/vRHgwxM9A35LPeRUswf9TjxJtzM9GbUWVhPZ0Rf7/D+eHk7uEP4FZ6iVh4fSzgIAAAAASUVORK5CYII=',
sport_football=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAL9SURBVDjLddN9MNNxHAfw31/Vf6LoLq4HRVdxpaurDnEdESepUxTCJVe5U52eyFNFp9iMeS5n19ZankZu2q3IUOZh2WweaiOnyeSI8tBF775bl0vmj++fr+/78/l8vx8KALXUUTDpyzqYtF1dLHr0YJOoWvNWKFVw0lr5Ab4clqtjJG3bRguDsDObvkKZw/BXZjNaZEzabBc7B70CNsZ6lRj7IENXaR4Ytpt/3TE3bl+Mc+imnbkZdIJnx9VKaFtEGGx4jh4eDd3P0qGVv8Gw/C0UHBri1xp/W4CVzDQPZRa9U80rwJBEAG2zEFPDn6CuzEdbeiQaEwJRF3sKfTWl0DSLwD9tNzOPO+j3PDsyUlV9/EJM9CuhEZdhUtuPbpI8M6pFK+MKqs85oOSYNarDXdBfVwlpfvxPPZalJK6UpSaVq0vy8LmxAgMiDqZHNOhkJaHuqifkBQlQV7P1mOdtBe4RK4iiTqLxfsRPqi3++nLp3diMtqS4uXG1XI+/D/ZCxc8lA2vHu+xoVPhtn8dPvDaB7WmF4kBHJFqYTlKSG5d9JDFRfR946fqyp79ooChMwKsIZ9Tf8oeqqmgBfnzYEmUhzuD62eP8mpUjVENk+IN2Rize81Ix/K5WPzBdsjTrBvgnti7CLHdLkn5Ah+fC1hg1UuLLwbWyzBjIc66hJSUMr6M8ybQDDCaz3DaiJMgJT/wdQPBkqJnRBUolYA8pHt2GJDkE4ptHIQx3NJxMcKHrBjwNcNThOYKbg8yMllOqF5yv2vYG1Mf5QRC8G+W+Wwxjl/XgkdKz3e1A8ATBoboXpJroUaNaqRgfa8ogOGtvEBe5b9bjzEM7dHiAYN+//4fKXLda+b7iITQSEdRCLirO7AfX2xpcHxvw/Pb86fmUA9IP2uqwhmCPf38vlWphUld8fB96ygv0l/S9LEbVRS+wffch120nkvda6XqeIriMYKf/d4dKNjd2JltVmWhu8ll4yWuqlRn9QxwfNBltvqqbPFU5wdcJtllq5X8DFM+Gyf3nvIYAAAAASUVORK5CYII=',
sport_golf=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGKSURBVCjPPVFNSwJRFH1SPyR00V+xIqJVREIEUm6VFm1ahFvLiiI/oE1QRlQgUtgHlKhp41dlokVFpKmjo41M69MZlbi8d+Gec8897z4BoYcwREdj67fJsHqunkp728cjvToR/UoYs66I9og6OvjEA0o41FzeXWOfkB+6CZQIqajhGz/MRWQRh+dg3USCMNyvvbG3xVDQ5JFRJkXFNZZ8JNyZw1qbfTrUZuhwk1mihktbGRNxd45QqxvtroqMBip4pZcYHB4Rkhr44oh2H9bzJ/srHBLEYkoE1RZF5X8PLZ4qnmm3SqsOlQSpC1X6KhU+ssacwxlvW0ccSWnObOCdkK7ygUcSmijQ5BNsKeH1FMnOkK2wWOCtII9LDlCxBeuWCJh3tTynKVRJItG1meUOyqTMaTMT3KTTdwUNUYa+i3uCMvcSgR+2VTFAwo5xcz/EV6dps06VKLu/EMDUyeRw/7NcRoffqcXp+4I2X+DB9K/VbTH9/6Yey6MLfrtkV62d2cz8hmVcDPbqfy6NlFRFHkA7AAAAAElFTkSuQmCC',
sport_raquet=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJhSURBVDjLY/j//z8DPnzu3DmhkydPZh45cmTd/v37i3fu3Jm7efPmorVr1wqB5PFqPnv2rPCJEycuHz58OH7Pnj0pO3bsCN+0aVP8mjVrFi1duvTO/Pnz1XBqPnPmDM/x48dfHzt2zAVoSNeTJ096Hz16tPXp06exGzZsCJw7d+6M6dOnF2DVfPr0aSagxomHDh2qAzq/F6hp1uPHjzc9f/68EmjIbiA7C2i7V39//yasBhw9elQOqPnAunXrKm/dulUE1LAbaEgbyAVAl/Q8fPhwG9BbDu3t7SewGrBm636v9EkHXq9atSrowYMHa16+fJkLshmIpwP5m4CG1AIDt7yhoWEHVgMWr9sZ7NO098eyZctKgYphNrcDNYOc33H37t11nZ2d2ZWVlXVYDZi0aIu5c9WOn4sXL7bZu3cvKPCm379/f/OzZ88KgZq3b9u2TbuiouJWUVGRFFYDolo2K9uVbPk9Z86c88CQdl+9enXPrl274oFh0jdt2jSf8vLy14WFhbZY00Fu20Ip66LNZ/JmXfxvl7/6tU328mcW6UueGCYtuumaPuVWaWnpzIKCAg+YehTNXb0TWhrben44FK56H9974r9z3uKspPLJtSml3XOj8zpb/dI6vPLz83WQ9cAZ7V29LUC//T948OD/grK6fzbp8ychKywuLubIy8tjQncxmGhp72rZsmXL//fv3//fuHHj/6y8khOZmZmchPIJ2IDW1lbJ7u7u/x8+fPgPTKL/07MLT6SlpRGlGe6CnOLaFcDQ/p+SkXciJSWFaM0oYZCclumbmJjITopmEAYAfvG+1+IMmV0AAAAASUVORK5CYII=',
sport_shuttlecock=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI9SURBVDjLjZM7aFNRGMdTA+Jg8UFx0qGDi4PgoERE6qCIUJxEwcFOdhPEdhB0FVQsmRRbaM1mhdIhtUVBwzU2mPf71bzfuTEUb9repiVNrr8rNsQaJcOf795zz//xffccjaIoml3IsrwP9KnPlUrlWOe3f0GzR0Cr1mq1eqhcLk8Xi8X5fD5/ricB1bnD/Qnkl7lc7mEmk9lIpVKWeDx+o6vA79gqtGotlUoHIOcQuZnNZvWQX4HxaDSaDIfDbwKBwP69An2dgHyH2JtASCaTfoSGVlZWPodCoXuQjV6v1+d2u093nUGhUNAS20b/C7huETsM2RiJRCZIdtHn8y27XK5vDodjuasAkS+DFCk+pdNpN2QRZxmBEZxtfr//fiKRuG61WtcsFsuDvwQgzeO8GovF1MFZqRHaMHo8nh0wS/RxyF6VbDabi4IgDLTJEE6hnqDGcZQZ2A9amsJVJMlHu92+ZbPZRNYvITAKOWsymUbaAvT7mknXgcSmPANbJfYGKT7gXGRNUAUgrzmdzq/sG263gPFxXDI41yA2+ehiYGUGtslmiRROyOr7AjM5/8dvbL57q9teWvwiLr6vBYPBdZJ42NjEVaKW6LlGnWUGg10P0s7MtKE1N6c0nj5TKhN6kdg1XOv0vA5xivrfO6HZfvRYak1OKi29Xqnfuq3Enr9oEHOJI3yyp8skXxs2NO6OKq2xMaV+5aoiXhgSOJH9vZB/CUhnzuq+nxg0FI4MSLmDhw35/qO63cvVi8BP3IoJNAAt5FMAAAAASUVORK5CYII=',
sport_soccer=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGXSURBVBgZBcHPi0xxAADwz3szfoxRGqzV7C6hVhwcJU6Ii5xxWBJFDuIitaX8FViExMVNIoe9IAeSH+vHJqsVOWBmlpmdNzPv++b5fKIcMLU5HMp2/xttLUrm5p+1bp5+DEQ5PhSz8ezs4LKypodKhs2E5v3o5JnfRLkPC7LrlbEBsY55P701Y70RX16U9h39E+XeXlh+cbWgLxXJJWp6lqibupceiN5szF6tKk+KbLVOoi3R1dNUNuvb/jiMrSxf8sCMr/oymUxHW+qXqt6pOOzp+2yJlo/m9HR05L6b1FSQbiuGDU11bX/l5sUSwbSb/qk5qFeI03jAiJqKIxZq6/nkqjreq0sV0x8LK+Me2WlASx9z2mIULRbE6ZOGQQes0BUEHcOWiuTWKUnfxent130SqSCV6olUlVTt8kW4HOXuXhs9tkZNQaJpXksiNaTn0fOwu0h67sWm+vbPGtYakiuoqGh4OJsdu9KJcpyvdm8M7a1oKNmhoGXay6fh5MRHohxw4nD3eLolL1ZD1g9T4VZ2Z6IL/wGvx8Nbuo22qgAAAABJRU5ErkJggg==',
sport_tennis=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMGSURBVDjLVZNLaFx1GMV//3vnzkymk5lEm/GRzWgVOkVcWRcRLK2YVHwg1geFQJfJIgWhmy7UQl24bcxQGjcqRHfiQikmiIKGKEWE1NqkQmjStNVOZ5J53Ll37v/porQkZ33OD77vcIRzjp0yV94bBTFmnRvBqIqVkbDFl+qksn/Lm1+eLx7644edfnEfYC6/VsYvTDkTTXild/PWBysUziqE9xiEGyRr1dBZOWtkuzr4am39AcAsHy3j7flYlN4et2mN6f6MlWs4tYV1D5F59CzdWzOkCy/jy4TuXx/MGRV/OPSOWveEEJ6x/kkxdGxci1X01jQ2XgRzG4hB1cGlUI2rdDZn6MoNspUz46anpwC8xu/H3xLkJm26i21/j7Mt4N5ZwhMgQuT2LxQOfIaJPXq1eXSQxygmNi+IUS8/PDbpP/F+TgRP4hjCWbvrqV4QoJsXSO5cZPDAOWSrQVT/lcLzn+RlmzHP3fg0b5Nr6M4i6UfO4KQPO5sRApFOkTS+wFlFas9heturuHQRIxnxsHK/9T1U4ytMvIkInsMZA9zjOOsAgUj5xP/9RtD/LKqzjcFHSyqe0RFYg3UC3bqMlzuINfZB2FqwBox2aJVgDTgDVktUDJ7VvRUnDU4MosKreH3PYHoaoyxGg5GgI40K+yjsO0747xIiO4CQEUmXFc9ptUQSIdJPITt/Ejz8ItY7TLIVI1sJSTOi195LoXKO8NYlZPMS6fzT6LBG0mVJbH8rRq1x32QPVvOdm+dxci/9+04h/Aw6uoufKeFI07w+T3NtjqDYT7F8gn+mT4cy5lhq4E27UP9azLp27VRQOEKv/hN3lycRYhjnBtBJEy3vgDEExSLZwRfQtWt0t5h95Tu34AEYSbW5eHYurfvJDU/QVzoCGYGxGwi/RVB8nNzwIYrlE/ixZnn68zkrqO4a042qKJuQKWOYGDp6Ou/6Slg/g9MGl4So5m2uzFTDXptZl6H6xkW3vgtwX6sfidGkxZhVjKiEShJBErEiE5aMZv71H93CTv//ct+662PTZOEAAAAASUVORK5CYII=',
star=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIwSURBVDjLlZLNS5RRFMafe9/3vjPOjI1jaKKEVH40tGgRBWEibfoPQoKkVdtoEQQF4T/QqkVtWrSTFrVsF1FgJbWpIAh1k2PNh+PrfL4f95zTQk0HHKkDD/cc7vP8uHCuEhF0q/KnmXNgGR248PZFN4/GISXMC8L89DBPV0Dp4/SsazJjrtfb9/vdxfn/BgjzY5M8Aq8nBya+V3h93vtnQHFxat4kszntJAAAxus1YvnZQV5V/jyTEZarwnwFLGeFZdT0ZFOJdD84qoCDOpQ7grZfRNj020JSEOKvwvxGiF+q0tL0N5PuO+Mk0nC0B0BDsYCCImyzAIktBBloMwKJLSgKYcMAcdhC2KpVlIig+H5qxcv0n0xmj4Gbq+BwC2wtJLbgHUlMEFJwUpMIGpto16u+kJzSACAk+WCzvNbe+AVljkOYIcQQou3TbvdOJo+g4aNdqzaF+PT43HJVA8DQpcVIiPPtaqlEUQzlDELsTpgYwgTAQIjQqlUCtpQfn1spdmxh+PJSQyw9CrbKgM7tvcISQAxlBhC3GuCYXk3cWP25m3M7dk88qbWBRDVApaATOSjPBdXXwYEP5QyCgvjE/kwHgInHtHYBnYA2owhrPiiuw0sOw3EZFEagIB7qChDiYaUcNIoFtP1KxCTPhWiDw7WbXk9vKpnOgsI4exjg6Mbq96YQPxm79uPOvqvbXx4O3KrF6w8osv2df17kr5YXJq7vnw/S0v3k7Ie7xtud/wAaRnP+Cw8iKQAAAABJRU5ErkJggg==',
status_away=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKsSURBVDjLhZJfSFNxFMcHPUUP0aP0EoSEVEaY5QixFzMTRtLDCH1QqIbEKBCzTEyxPwotsbLUETSHmTUTm86mrk03N9eW253Oua2pW8OcS6Z3d7uzWd/uZi0qaQfOww/O93N+53wPCwDr7yyoJ0oqRG61oM9P3u3xkUUCQp3C7SnZqnZLsUDqXzK6wpjxrGPKHcHYDAVei3NpW86TkqSAcqZzTOz8vI4PLhp6ZxjmBRpycxAZZXJ1UkBjr4+c9myKJxw0NLYQlNMUxmcpFDZayaSAum4fScxHMGEPQx0XhzBMUBhlxsivI5IDCutNaqU1CNM8jRELhUFTrHsIL3Wr2J7XmXyEnQVdNUUCKz06E4LUSOK1fg1ygpmfPxpl7eI3/xdwmK/j8oUum/jtHERDC2iQeHHrhQePemfR2mNBToV6hZVyu2ZLwH6e4tClNqdVwyzL5o3AOEfD8DEMLfN96ZtevOuqhk7WAKXkJoRVJ+1VnPTcPwDZFTqhbHINVg/N+B6C0RUD0Bjol8KqbYduahEq8wqGDH7I9XY8rT0VOp994EQCUNpkd4zbg1BaSIxZKbxnuusdYfQ/vYIxsxerwa+4U1mNWCwGIuge1OAa56AqAeAL3bTBEYJ42AcV47uOsTG2fZnoKmSG5bjwFyC6sYEOlRvNl9mrCQCvxRXR2kPoVCxDYQlu3sAUhVcPStGtWcT3n4BYfmMeDwdmcY+XGUgAcq+bXGKlH31afxygYm5BbiZxv/wcOmUqfApE45BYzH+J4nHHM5SxUxW/7Tgmzd/B1beWNdsxwgjHbRSzTBIdXQo08dkQSSR4PkRAPDCJdlEbqjhpgTOZaex/Dmk3d3gpg2/w5t0gAjmVRDT1gi6653Rr8OzxPLq2+OhGfXF64GLWXjnnyL6sWP0PKd/8SStdk8YAAAAASUVORK5CYII=',
status_busy=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKBSURBVDjLhZJdSFNhHMZfkQLxIhKKJPWum6KkpIxAh9KQIWTrZhfe2EWIxK5CIgJBTEphhJBkCYGI9EFLZFAtlTnnNudm80yP82xO3RzqcZp5ds7OhuTTeyYsMmkvPHfv83v+XwQAOazaNqahuS9sMwzFhKdGXqg3MLZCnbHhqL9Hmg2m2IYnlMB8JIXZcBLj8yIau4MbuaqXDVkBD2iyYg6upzAdkuEKJjCzIsM8E0dZk9mWFdAxyAtzkQPzZEDGhF+CZU6EfUGEtoMVsgJa3/MCs5zEJJeALW2WMMyIsNI2NK1MdoC2zWuzsHF4l2WM+ER88SrpEj44fyKvZiB7Cydq37bUG1jZOi/B5BHwybULM0P711v3yEl9138Bl/VOnb435O//uoS+byt49jGKJ+8ieDG4gB6jD6pm2zYpbG85EnChcbT0/qsgO0GH5Y8m4VmS4V5MwEHLd7a/xuKx4wjS74oWSI7MEaL5C1DR7Oz9/H0XbESme5fgCSkAGdOdb7BafAZ81TVs31FjW3sTa9dLsXSqAH5C1BnA3edcwM7FYfEJGGdFTNF0VyCRTuZVV7GlqcR6SQnWiouxRSHR8otgCdnJAPS9YdkdkNA/zGOM7t1J16hMXylZST78NuuqMEvIfgbQ2B1KOjgJA6ObGPXFD25gVjwA0MSU243U1FRG/K1DAPUjb6jfEsOQI5YGjNFbMM8I4PLysVZ+CbHb1YgWFaXF11UjXHZeAfz4s45ykyZf5+pp6uIwQo12v0iHKcDabcTi6QKs0p6VspXk8JW0eZ8hpPKfQzqrG94o07ujNY+ZHdVDZu/cPeee+kZnfDI3/5eP5KSNVFvUXKH8/w2+ut3dWHFeZAAAAABJRU5ErkJggg==',
status_offline=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAE4SURBVCjPZdBLSwIBGIXh/lHQb4guyza1CEIqpNoIQdHKXEQQrkS6IUSLFhYFtpCIwUAG07IstTTnqjNTjnSRZmPg4m3lpYZvd84DB74BBjq36zkXk07CORB9nl7aVydtkwZ1NKL2tMcFYqLJOxYGb1QIiC5w5dhYGOgo6EQcFxCcOjV0VCRUdtxgX1R4RaZClTzz7okF/2FLo0SRChvtkdA/sDl1Wk6RQuASAYHg54S/D6wPnjzrNLAwqVJBJsfax/BoFwQjZWw0LEx0SmQocsGk2AVHko6MhoGByhMZEqSZ++qCs5bBLSo1qkgUSBMny1K7C45/qtwho6NQ4oFr4mRZ7IGwmqWAjMILee65IUWMmd6Ed3xlL4qEjkqZR9KE8X2PDf151Kq9ZW03Q+1Ae7np1WZznfwXGfNkzblrzUIAAAAASUVORK5CYII=',
status_online=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJkSURBVDjLhVLPSxRhGH5mf8yOs9O6aa2b6BJhsW3RilAXDSW65clDdgwkEBH/gIiI6FC3uoRBQYeooP4Aw9isQ2xG5YZEVFrINmnFto67s7sz33xf76wedEfwgxdm4H1+vO/zSkIINL7Bax/PpxLRkXhUTVuMY/7Hci4z++2e/njofmNvYDvwqe726/2pcJsa9MMhgd7D4T5NUQ8GBibBZka3kPgaCZKk7IKbVT8qNodpcUToe6g33tadOjCyo4NYREkrpGyYHLYDMEfArHFoioTE/o70jgRVC3AIZDMqLogA9fKR12qVefblGWHui54rmDZCsoSaLVClUkMSVlYZZl7P53YkyGQ/T9+dWqoaFY6K5ZaDEo1w42GOVWaz7xv7pc0x9kxkh/uOxa6c6JSSnDz/MgJgFGM0ZCLALTzKrhZePnh1S+gXr3p2cHQ0kx7oSVwePtmWbNUCKFsCKb6+i3K1GXKQY2JfrCW/XJqQfGNvBL/9bMsILRF1/MzxWGo3RfbHoK3VjUkgDlhEsqDXEKJ0Lgx2tSJ56JJnB13tLf3NYR9+F20CCwJSuSnw9W8hJHxdMtHeqiAYix/xEGia0ilLPuRXKnVVx41vYwRG6XEOGGsMst8PWVF3eXZgWUyixChvCc6GMiNwja7RJjR3x3GLRFwyj4PFvPFzQTehNUn1f4e6LIfXCdxDovGR2BvEh+9lVArFNQ/BdCY/Pjq5eGfqbQGC1IPkpEkGwnREMvl09/DkxQpuPs0beDd3ets7cF/HuefL8ViU7YnIYbpcTS+Y0P9apXLe+IeSWRSfzvZs7v8PV6U0ly704DwAAAAASUVORK5CYII=',
stop=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJOSURBVDjLpZI9T1RBFIaf3buAoBgJ8rl6QVBJVNDCShMLOhBj6T+wNUaDjY0WmpBIgYpAjL/AShJ+gVYYYRPIony5IETkQxZ2770zc2fGYpflQy2MJzk5J5M5z/vO5ESstfxPxA4erL4Zuh4pLnoaiUZdq7XAGKzRJVbIBZ3JPLJaD9c/eCj/CFgZfNl5qK5q8EhTXdxxLKgQjAFr0NK0ppOpt9n51D2gd2cmsvOElVcvOoprKvuPtriNzsY8rH+H0ECoQEg4WklY1czP8akZby51p6G3b6QAWBl43llSVTlUfuZE3NmYh9Vl0HkHSuVq4ENFNWFdC+uJ5JI/9/V2Y//rkShA1HF6yk/VxJ0f07CcgkCB7+fSC8Dzcy7mp4l9/khlUzwecaI9hT+wRrsOISylcsphCFLl1RXIvBMpYDZJrKYRjHELACNEgC/KCQQofWBQ5nuV64UAP8AEfrDrQEiLlJD18+p7BguwfAoBUmKEsLsAGZSiFWxtgWWP4gGAkuB5YDRWylKAKIDJZBa1H8Kx47C1Cdls7qLnQTZffQ+20lB7EiU1ent7sQBQ6+vdq2PJ5dC9ABW1sJnOQbL5Qc/HpNOYehf/4lW+jY4vh2tr3fsWafrWzRtlDW5f9aVzjUVj72FmCqzBypBQCKzbjLp8jZUPo7OZyYm7bYkvw/sAAFMd7V3lp5sGqs+fjRcZhVYKY0xupwysfpogk0jcb5ucffbbKu9Esv1Kl1N2+Ekk5rg2DIXRmog1Jdr3F/Tm5mO0edc6MSP/CvjX+AV0DoH1Z+D54gAAAABJRU5ErkJggg==',
style=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK/SURBVDjLY/j//z8DJRiFozbrLk/aqkc76/a8eDft2Ou/Ew69+lm/8/n7pMUPTsuXXlAgaAAIK/fe9Kg7/ubmsaff/h99/O2/48y7q+Tyz2vKZJ5hJGiAUucNRv0JNycuuvLho/WU24tytz67aNl5fZFM8mlhoryg0HAlcePNz7+06670y2aftaja8fy224SbW6SzL1lrNt+aY95776BJ593dJq13dpu13jqoWXptGUJz1WXVkp0vrs48/e6NTNoZM+n4kzpTDr5+7T/l9gHpzAvOyhU3J/vMe/w5e+OL/5lrXvzXKb2xTjz2QhncAKOWqzM3X//0Z97Jdx8mHHj1YsbB128P3Pz0P3bW3TNiXgfk9BturQ+Y9+ifU+/du4nLnvyXiD7fLBZ+lo0BGEAswACKXXLm3We/aXf2SoYejZQIPBws7ncwb+qeF29TZt+9LJlwNiNmydP/tm13LwNtdY+Y+/i/TNT5XnAYAANIL3vN40uTDrx6JRF0xBDmIlHPvepJM+5czJh174Hb5Pvv3SbceykWdd4aaGtQ5MyH/1UTLywDG9Cx8/n3aQdf/W/e+uxL8ozb20CCIu57jIN7bpxcdujN/+hJ9/4nLnnyXyzibC1YLuS0d/jU+/+1ky9swZoOkDHQuTHR8x//T1705H/MnIf/ffvu/Q+ffO9/ytyH/7XiLmwR9DoijFtz9Hkz6/qbl716736Tizo/XSTgZIGw34kc9ajz65JnPvivF3/+oIDbYQ2cBmhmX1qTMO/Rf7Hgk83C/ie4YOKCnkeCXSpvfNCLPn+A3+WgEoZGYCAZi4aeKXZvu/PBo+3OV6CtwUI+x1nBmj2OKAJtbXCrvPbVNufSYz6nA/EYBrh33v3k23f3v2/Pnf8+HXf+G6VdPAa0lRMkZ5Zy8aJXzY1/QPzfq/rGf/fyaz8ZKM3OABiskbcwY1E6AAAAAElFTkSuQmCC',
style_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALeSURBVDjLY/j//z8DDKvNusuTturRzro9L95NO/b674RDr37W73z+Pmnxg9PypRcUkNXCMIaAcu9Nj7rjb24ee/rt/9HH3/47zry7Si7/vKZM5hlGggYodd5g1J9wc+KiKx8+Wk+5vSh367OLlp3XF8kknxbGphnDAIWGK4kbb37+pV13pV82+6xF1Y7nt90m3NwinX3JWrP51hzz3nsHTTrv7jZpvbPbrPXWQc3Sa8sQmqsuq5bsfHF15ul3b2TSzphJx5/UmXLw9Wv/KbcPSGdecFauuDnZZ97jz9kbX/zPXPPiv07pjXXisRfK4AYYtVydufn6pz/zTr77MOHAqxczDr5+e+Dmp/+xs+6eEfM6IKffcGt9wLxH/5x6795NXPbkv0T0+Wax8LNsDMAAYgEGUOySM+8++027s1cy9GikRODhYHG/g3lT97x4mzL77mXJhLMZMUue/rdtu3sZaKt7xNzH/2WizveCwwAYQHrZax5fmnTg1SuJoCOGMBeJeu5VT5px52LGrHsP3Cbff+824d5Lsajz1kBbgyJnPvyvmnhhGdiAjp3Pv087+Op/89ZnX5Jn3N4GEhRx32Mc3HPj5LJDb/5HT7r3P3HJk/9iEWdrwXIhp73Dp97/r518YQvWdICMgc6NiZ7/+H/yoif/U2fM/58zJf5/Wp///6hWj/82ucU3BL2OCOPWHH3ezLr+5mWv3rvffKvrTzeuTvy/5crU/xee7frfvzvrf3i/7n/jdJ15OA3QzL60JmHeo/9iwSebY9tdvm+8POH/xmuT/oNAz57U//170v+bZ0v9wNAIDCRj0dAzxe5tdz54tN35Kux3Itin2vD/titz/iODTZemAg2QxLTZvfPuJ9++u/99e+789+m4898o7eIxyxzp7927kv6370oAa27fmQB2gUW2/H+8gQjDQJtagns0/vfuSgHbDKJBfKB4D1EGQA3pAOJPIGdD6Q6QOADllJVJzH+HzQAAAABJRU5ErkJggg==',
style_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALzSURBVDjLjVJ5SNNhGFac9U//iDlnTtNEYyRKKnZIUOaVeeWSeZY6E2VUkJk2Sg0PNFwtL5zlgiUjxBRLZcMJm1eubWzeTvebxzzWptM8Zoft7TfBEtTqg4cPnu97nvfh4TUBAJMduNQgR1LrZ7g5PLWuqlf7k96p+ZbLXVhOfjMlOp4pc9j9dwd7CCeaPCjn46K8d04PPSo9XGIg9fZ3pQR8utj0nwYnSsZM3enyF6yhlS8+FROs263z/edKRll4sshyP/EeA4e8oaRm+dr3UzlDz+0okrNUzsJEAF3eYksZ8CHkj786Q1MKvEqQdq9CRbt34biAkDnC/iOmDjrf56qHGSLdIj5V7G17U+haIdBqwysm+LbpsstO2fLyEKZqjdKshvQGNbhmjjVaJ8ge/DbwKBhmfBhd3WIKdSt0vkZdLdAu8eWrkFCDiLHBfHv3vPGmCOaMwZeGIEnsWcDFSfOxJMkhE7QgDFpQQp1YtxZWpeiwieqJwV3rIlqHCe5U8tRLKS+RQZtESVp83RxcKEIG0amB0bUqwMdKadsdoAW5URpUA2V8jQYX2X16J5HVlY6TydWK/rQa5VRA+eRyAF35GRsr9UGnRsYwpsE5ScbeNijmLmxWCTSQ3zq/Tq6eaDOSRwN5nsTSMSG7cxHiypSQVDcL2GjJ4+2366KrpMpJOEWWtey7B7uBxo2Pe60CMmsWaI8K4X2UC/B8zaEpHA/UiBujFsHdlgeL46TePrnywWAaon8SfatPmnF+a7OVBoYRDmy8vQefKG6GWj/nggMNCJSBhkTmDGCJwnyOH0apR8VQFgqQZQFQ5Ajapxehze/wzB4hWpKnVZQ4I7BIsRJUpNiwDOsj8nwxBoOkEXaf5VwcGPk9BoElyGroMwRCSxUQUqwAj9T+Xo4/Zna9lgyAir6iq6NDMZ1mBu+CbQx/LXEHXaRjecIU5x/zWY6geWgOSIopdIRitjj+ZtT/MjCim2SXzfU3mzTGRm+VUWzkfwHZspOnT6PVHAAAAABJRU5ErkJggg==',
style_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMxSURBVBgZdcFraFtlAAbg9zvnO5fcTk7Smda2SRvbLbau69aSUXWdE4rTuTlBnUxkYzgUFX9VLKJ4mQoyEMShwqygqKWKjKEDYSiuk152payKTdvFZZCmSdauTXNyTs4ln/vhjxLc85D1n82BASAMcqcqfPDgeu+BWpmXzl3XUqNThWPJtHbc55MRUCSUNQeT78WwFnWKNsAAw6oYaUryFUrErOEYl+a0C8mU/o2iyvD6ZNwOdW6aYAIPn8z1PBBx7x88OX8k0iBvpgyCKIv3KiH3DoFBqlgMt9gA3sAaHCs5KK9anr425UhzUKrnjUqqK+SK1MhcoKxZNTLPHdwYdb1W76cvLGZMEVU43aigRRUON/po15Wp5cvtdeKzDQp/j+RUmAeVWp9I1NlZbaxYsPP6sjWEKtTv4h55KB58e/z84vDpM9m3WMmULNP5VA25tzTdrXYspPSvbiwYo7FWz8dHdxzfkxhaucx7O6DdGAEx0gfpgb7QywEvnwsoVPPAXlIa3H13BD0BItKwV7AKJcLO3hkSor01vwZ3xdR3Xev2wB/dguV/Ypg+PXiU+j10Lw84AVUgstfFVsr0t1MXV8ObN/l6rKKd4xkb2Bkd3fDitqzkV+/H0tW/IRITPqUO3pqIn/I8caaTJfx+ocBMwQXKkbbWJvn1ZKLwcypZeuf7p97cFO/u+VZteRjlzDBEN0FqchqWYS4aevE+bva6gZ/OLCOTtznKk8fr1gmfuCWuZTFbHrt0eH94a3f8O7X1UWLMfwEi3ITg9UEuXc1ahUxf/JUTM3Rm3kJ92AVR4CjAnnbLXC1z2OyTjT/0ehr3feRvfQxG+nNwgg2zEEV+ZHzV0Uu7u/rPTeIWwhhDtV/er3+iOf7MUGz7XtHMfg3CmzBWGpE/e153dK091j9+Df/hUOXLlyLPNXUf+tFiYTEzdQwVGNCWGpAbmcg5hrYx1j9+DWtQVFGaNwy2b3seieFXMfPHGKRIJ+xcYj4g8js7BiaSqMKhSmJuwTb/OoVoZy88nhjSF6eKhm7u6hiY+BP/gzDGsNa+7UG9665aurUtCiaUx/Rc/tDuD68kcRv/AvFKVJouQOXGAAAAAElFTkSuQmCC',
style_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALwSURBVDjLhZJJTBNhFIDfTKcdumGtBYqtbSgQkE0DaggqiyhBPchBjWLiEuPBaAyHetEEE6NouOjFE5407gESExKNVlyCCYsLRkNYQrqwWVRKO22n//zz/w4kEgKi3+29/39fXt57DKUU/pDTMmqoNKlbrWbNZqtOtQphimficiwQTIx4voQOeJs3eGEJ7OJg8JRL8ATiN0DPTW906tnNGQbNW2/sxcu+maOygH3wF7jFQWbzIGPkmd1ZRs56vn38bnGGfoMYQiIRcDBwexP9r4DE8fHLdc7TF+75boV/Jh5VZBnuGnnWTJP43LwrwyeSdVy2jCkCTJXWiSYSwuMLgoyLX7P3V6S4p8JSeHZKfEATcmytljNpOdYAhCShKI2n2NTFzjVqA5EA3nWH2n9Noc8LArOOdVc4dDnTYSy46x1PkxhGnWvlzd2DrEHyhYYtW3R2Dqh+YCQ66kznXT8m4t8oJjc5Z8NnjiD58PWTrvqWzuCb3s7vt5UHRGViazyXc4lngFelGvcUZunrfH7x69BAxJ1q0jzjKOgCD4sRqwwob1912vlfAo73vJxyTzwuezDZtq2VSPj5hxFhjKOMsbDEfM1AjpLc9P0FFMl6BhPQ8kz6/BrPHrR356fwhbNRrNtbk3Z1LplS6ykpr7bd2ZlnKoqI4LStVptEJIh2y3qoqjrSShSBhoXkub/M4kNaTMP9ckqIJCNCVAhjsJkzoci+DXq8r6B/rBfGP3V0TP6IHuNgBZQxwK78YyqZEpCJDAQoTMwGoNC+FQQUh9D6sj3wuiN1RYGIEcwV+34Og0SwIpRAkiUIJyKwcd12EKQYEyqt7l8mSDv0sYQSWllTIwKWMaQlO5RieV42OesHs8EKHwNd0D3sx8G+tjPLZlDbPBrWqMHIaXeBiCRAJKF0kyCZlny21FUDff730DXkAxd5AgRzaMUhLqXgCju9ybHD8nrI8wERKJtoUk76X1tYiquRmZaJyhInsjrYRPGf/G9oAXqKbs089gAAAABJRU5ErkJggg==',
sum=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACzSURBVCjPY/jPgB8yEKmg0aHxf93/6jPlR4vP5P/I+p9yFMOEuvP7/pedKerJ7cmYnNwTOx9DQaVB9/8J/3McILyw/VjcUFA//3/a/QQBPI5MOT/7f2QBHgWxCRHvg2bhVBAjEHY/8DaecAhd73/GswfCNvmPoSA4we+8x2kQywLoTP33aAqCDHzeu79xrLepN+83uq/3Xwvdm94Jrvsd9lvtN91vuF93v+Z+tX5S44ICBQA4egHkwuNCKQAAAABJRU5ErkJggg==',
tab=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADVSURBVDjLY/z//z8DJYCJgUIw8AawgIj58+c7A6lWIDYnUt89IC5MTEzcxAIVmKyvr6kpLi4C5jAygkkoG0FD2IwMr1+/VTp9+uJUIAdugIiQED/Do0cvGX7//gvxGxMTXBMIw/gsLCwM0tLCYD1wL0AAIwMzMzPD37//4YqRDUEYwAxkM6OGAcxGZmYWoAIGFA3oNDMziGbCNAAkCJL8/58Fp+0QS1ANYJw3b95/BQVZBj09bXjgIQIMxkelQeD8+UsM9+49gLjgwYPHYEwOYBzNCwwAGT0uf+Tb34kAAAAASUVORK5CYII=',
tab_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF6SURBVDjLxZM7S0NBEIW/fWCnQgyIT0QQDKJBFBRbsRCt/Ae3VRs7wVpIY6WxkqTXxgek8IGIIQRsYjR2SkhAMJoipDAkXNfi6k18YcDCA8vMLDtnds7uCGMMf4Hkj/h/Ag0QDocngVVgrM68O2DJsqx9/bax7vf7fK2tXgCEABBvftU6vuDxMd97cXEZBFwCr8fTTCbzQKViO71J6SYJIdxYa01HRwuA123BgUAphW0b93AtSZVAIaX6qMF7RaU0WvMh4bNVSiKE/EoghEQpiTH62+qJTIzLbIzic4FypYxXdmuwEKFQyPT0dDE0NOCKVxXMiU8SB6Seooz4Run09HGa2iV+fU5Tsd+5QTqdJZ3O/vhmZ7lt5mamsaWNv22K45sdxgcn2NmLgDHm1zW7Mmwi11umFvvJoBlbaDN1/cR8IVdK3ccIHFkABA4tbnNJgFJdBC/mZS2ejNGA5uBqkwahiSbOAIKi3nEeX2wPAPNAI1AENuMb98uviwGZtIAuD3IAAAAASUVORK5CYII=',
tab_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAF/SURBVDjLY/z//z8DJYCJgUIw8AawgIj58+c7A6lWIDYnUt89IC5MTEzcxAIVmKyvr6kpLi4C5jAygkkoG0FD2IwMr1+/VTp9+uJUIAdugIiQED/Do0cvGX7//gvxGxMTXBMIw/gsLCwM0tLCYD1wL0AAIwMzMzPD37//4YqRDUEYwAxkM6OGAcxGZmYWoAIGFA3oNDMziGbCNAAkCJL8/58Fq+2/jq9h+HZkOcOvZw8YXotKMPBKmwP1JjIwzps377+CgiyDnp42PPAQAQbhv9m1kOH72Y0MqvaBDOyK2gzfL+1iuHZkL8N9bl2IAYTiTHxTFYNjaikD590DDAxPjzAw8AswvGGRZzh97DoDAygvEMJ7nFj+/Tu77j8yeF8v8R8kzkJMqvnL/P/ptwtbZbg3ZTH8/P6C4RtQ7PMnYIwxMzwjKilzCYnPvXr80J/nPzgZPjGzMrx/x8hw5ykjMMIZpjESm52PRshVfH33LJ35L6M8yEVAXdPdd/1pAwD4EZ6cdwTiAgAAAABJRU5ErkJggg==',
tab_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHWSURBVDjLzZPdS1NxGMf3L3TbXV5EEN5030UJpTdClBBKSgh2Y5cyW0QXISY2eiGxklYgGoaE2YtFdTjHvZyO25i6uReOuRc3T7TNnOFOw8bHs2MmZUEQRRefm9+P74fn9zzPzwJY/gTLPxUsjB04Hh06ifq4i+m7R5jp29/82+HFiT2NmmBlZfYpfMrwcXYU+Urte/PS4XDUGLw14Gc8G+4gF7pIaXEcTeylGHzEl4SL4L02fUsQ9vtl0mnVJJOpML9JbITl0AXKRRfFd+3kp84SGWwlMHC6PHXj2N4twYd4PIzH40KSJBOn04lX6GM5eI6yLrM234KeamI1bCNxv54HA/bStyZuCiIoimwG3W430lgvmtf6NdyMnmykEDqPeqsOLSJWnqZ/J0gmY/h8XmRZZnL8KuEXHUbZk+jxVj6nTrFiVKL21zLnFclmMzsFqZRKIODn5VA3c89tzExcI600sBZvIj/dSex2vRmORiPkctq2oNJlQXhlHC6Rzy/xsKcGVhNE75xAsO3GbZTssR8lu+CjUMga5ExEUTAnZPlxZJfaqinJNykp11G6DjFyporB/h5+NeIdC9NwcJfe3bJv/c3luvXX9sPSE2t11f/zF/6KYAOj9QWRU1s5XQAAAABJRU5ErkJggg==',
tab_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAG6SURBVDjLxVNNKERRFP7ue0/JiIxXyE8aq/HTSP4iG2JpYSEL0qzYWFgrG8zCVopSs5CVnZVIWbFAYuG3NGaKmCz8z9z37rvuvTN+htSUhVPfPefczvd1zrldwjnHX0zDH+3/BQx5BIPBDuGmBJrS5F0IjPr9/hUjeTHj83m9BQWmSghRZzL+9ImYIBq98+zsHMyK5EPAdLtzEQ7fwLJYYjZN+yBJvOeGYaC4OF9xvu2AQNd1VSiLZCz9O5YvxrBwNCxiWaP/XKIk6rqRQv4qEmcUJaYXge0+0YmWusTEjFJAA+eGanXxdBTgNqgjYDMU5VXAW9iEx9gzBpYa0JszkRAQL8DLy0uRleWCy+VSo0gB+5iis2oQjDtgDoMDjqv7CGpKWvFEXzEfGclEEFx1EApFFL5ajFBFvrw7hyW6sB0LFrPwEH9EbWkbnqwXTIeGQH77C/0LlbzbNySITEGKXd+H4c4uxH5kC+snqyfURt2vAj1zHhajVOyAImZTVJhVWrOnC7vhLWycrR0KcuNVgMdJur+xelKL1pe1m5tnG3vUQYsgU7X8dAU84yTKHN18dVjGbUA8T9LeAB6et/fbYA/BAAAAAElFTkSuQmCC',
table=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHISURBVDjLpVPNK0RRFP+9D98syMwUspHkm9I0YkFZWBFKkZ0s7a3Ewh+ilChK7FgoZCJFKYlYKB8zk2+Z5t0P577He29kQU7dd+6575zf+d1zztWklPiPmOozt/U4SThjXIoyIQS4AJjSXO0lGGlvcXAm6Vzsz4xUhm0AIeX4QLig+C+ZpxbOG1wGhGYHr1zMUmZGWRgs0ha3PE1nX/8mWmdgWTzLB+DUYbhm9FfZ35IEyrhXA3VXJfPbsV8B9LQUIeUHYJ8ASobag1jcucNgW8g9W4reYSDi2YnnZDoDiwCokDANct6NwTB0LEdj0HRA/wxa2SN25JNBEdWluUhZ366gqmAaGvrCAXKOozccTGPgt8+vn8GYSGcgyTYp3dpBnBg42nbQPRBTo5bTvqYkmxL6AQhNTWQGBXY3B7BxlEBXozcW64dxRKoKUZBju+P06gl5WaaviMJBM3TNDlbypemIZgHYOnlwASsCmW7nHADGnBoQ3c76YmweJ9BR5zFYjsbRHwm4tmJg6PhWA7pCXXk+bu7fURHKweXtq/sWaksz7SC/CCGFrwtyZ3r+rCnFRZ7qr1qc6mLZj4f9OEyPL8lVpbX/PucPv5QPKHB1TdEAAAAASUVORK5CYII=',
table_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIpSURBVDjLpZNPSFRRFMZ/749/Kt3IqFTSRoSMmrGIYTTbpEJtjBCCok1Em9JVG1dRC8FFEES5aGFEgRRZWq1iLKKxBiNqLDcltQgmHR9hY6LOu+feFm+YGVsZXbh8nHO53/nud8+xjDH8z3IB7r5avGgMZ8XoBq01okFpjYhGtEGJLtmCKINo/XbgVFPUBdDG9PVEq0P/UvnSvdlwQYFoHQIY/3obpRVKFL5W+OIXUVThrL91AN+XihKCwIeTu85sqPryqsJXUvRARAMwkshsiKB7fw25UgKVJwA40V7H/cl5jh+oL+RGk/P0xIqxl11dr8AXjTYG14HRNxkcx+ZhMoNlg52/ND6VAWMoc6F5+2Zy/l9PMIDrWByL1jI+tcDRaN06BaXxbDqLUnq9AqPBteHpuwUcJ0AIcgBXH93h+/wEyyuLrPk5cmv7gNY8gdIYYyhz4PDeWuIpj85IsS2ujQ2zJAk6DkZpqGnixcwYyU+PifUOX7Eh6DoAx7aIpzwA4imPeMrj+bTH+88PaNkZQWwhsrULsXxie9oAzgcESgUe2NAZCeE6AXZGQhwKh/Cyc5RZVXQ39wFwoeMmjXVhgMqiB8awe0cVP36u0Fi/iW9zvwuzkF3+xUz6Nal0gv6uWww+O02lUwGwmv8FM3l55EtLTvQWXwm+EkRpfNEoUZRXHCE5PUFbuJ0nH4cot1wSH14C3LA2Os6x3m2DwDmgGlgChpLX0/1/AIu8MA7WsWBMAAAAAElFTkSuQmCC',
table_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAImSURBVDjLpZNPSFVBFMZ/9859+C9J9PmPwpeUi8qyIB5JLQqEoqAQIUqCFiFB4aJWrsIgW0TLyHYRBFJgYBiUKUkmPYwio02kuZBMny8JzT95Z860uI93n62MBoaPc4b5zne+meNYa/mf5QE8GPp51VpajJXNIoIR0CIYIxixaCNZ22C0xYi87ThbE/cAxNrWpnhh9F8qtz8c251RYESiAD1f76NFo43GF41v/BCNzpy11Xfg+yYniyDwoXnH+XVVX1zR+NqEHhgjAHS9Tq6L4MS+YlazCXSaAODMwTIeDc9w6kB5JtedmKFpfxin5lfWKvCNINbiKeh+k0Qpl8eJJI4LbvpSz0gSrCXiwfZN+az6f7VgAU85NMZL6RmZ5WS8bI2C7Hhsah6tZa0CK+C50PtuFqUChCAH8Oz2TXIS95DpcZxoJXVFx4H6NIEWrLVEFBzdW0r/aIqGuvBbDN69RclkLzXNLeRU72T5Yx+FQwO8aIi0BiZKIEe5Dv2jKYAMug74LzvZdu4SueOD8Oo6+RuLqI7FeP/ZXg4ItA48cOHwriiDn1Icqg0VDPyYJLeiGo5dCWegvRIlzpbQA2uprdrA97lltpbnMTH9KzMLlFSw9OEpBU8u8nt5miVgYV5hFFPpV7DD17q+7Fk1UuBrg68NRgu+EbTR1Jc3UpB4Tqw4D09FWJjVTMy41iJ3nPWO8/DpqrbFuakLyjgxo+w3C51H+vSNP9H7LzNBaB8uAAAAAElFTkSuQmCC',
table_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ6SURBVDjLpZNZSNRRGMV//2XGsjFrMg2z0so2K21xIFpepYUiAsGIICLffI8eWiBBeg3qQV+KwBYKLB8qpHUmrahcKLc0QsxldNSxdPz/79LD1ChBUXTh8sG93POdc75zDa01/7NsgGvPR09rzQmpVZZSCqlAKIWUCqk0QqoZWyKFRir1uvxIbsAGUFqXHQqkpP1L57M3Pm5MMJBKpQHUdF9BKIGQAlcJXOlOVykSdye3leO6MmkGQNyHw+uO/1X3bzGBK+S0B1IqAKqDg3986HeCZPffwvJtoNT7lOZLvUdtAPEDAKBkRzo3QwMUb89InN1uGGD3spdE214xe8MRUnM2MfppNW0Pqy7YAK5UKK2xLbhdP4hlmdxpGMQwwQT8ziNiI534c7cT6WrFazikzF2Eb8HS1IQEDdiWwcHAQmpehTkQSAcgNvSMiYFW5uUUMdV3HW+ywefGNqITJsbUUL75k4FWYJtQ+yaMZcXrk1ANk/33mbdiD7EvlRieETy+FJLkMFcjRRSW3emIAwiF1hqPBfu2LGSWbbA1uZ41SfWkrtxPrPcypsfFiWYzFGzGKTjFV28WEJeIUHETLdOgrmkI1VdHpCdEet5enP4qLK9mKrqMgedv6cyrAP+qxOTiUxAi7oEJi8frELoFoTLpa7nI/HQvscgSRt+0kV1SSW7qYtp7xrBMphm4Mi5h/VIfTcEq1u0oJaknSEdNiMYHET7UvcMpPEN31Ed7zxgASmk1I0g6dK66s8CRak5mVxjnfS05+TsZCw/T9baTx1nnGb47DrQksjE6HrsHYPz6nYt3+Sc3L8+wA2tz0J6pF5OD4WP7Kpq7f5fO79DfSxjdtCtDAAAAAElFTkSuQmCC',
table_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJBSURBVDjLpZNLSJRhFIaf/1JO3lAbZyjDbpiXRC1FFFsUtJSwhMBsFy3b2CZaRAiuWre2opDCBCtalBl5qzSjhAwkqlWljRZizuj/fee0mHKmaFF04PDCd+A557wfx1FV/id8gKvDX8+pctKqbBERrIARwVrBimKspKXFGsWKTHQdL6n3AUT1VGt9TvhfOp+//qZqbQIrEgbof3sZIwZjDYEYAhuk1Jq12pnGLoLAZqQBkj4cqzjxV92/JQyBsSkPrBUAekbm/gpwqK6A1XSA+QEAaNsX4cboLEebomtvg6P3KcsYJVrVgbc+l9hi4tcJAiuIKr4HNx/P4XkufU/mcFzwNcHOxF1C0SaejfYwn9tCeVEmq0ES4P5cQQHfc2htjOC7cKQhwuH6CE0bx4kU1ZG7aQ8FyyMc2LGY/FojKUBgBRXwXbgz+RnPS+q9p1N8mu4nJ5yHXbxFuLyNqaFurDEo6QAjqCrrPGiuLSTkOzTvDbPbGSBa1gKJ54xfuUZ2XpyMhQns3DhZGX4KYCRJ81yHgZcxAIbHHhIsLZCTv4ysvAMV7NIE2/d3EJ++hGfjaQBjkh64cLA6jO8aovO95BXXIMtTqMSpbS1FVj8QCs1QuK2JamfwNw9UqSzO5uNCnM3Ljwjll5CZE0PNPDgek30zgCCJGQp3baVs5QFDnZUVjqpyuvv1iAlszaqVrMBY2vUsDe0X8dz3qPkCpF+si7ehlNlXL5i53dnr/Omcxy7UxtRKpoqiImmpa4oIqjr5HZAwT/Ha9ghQAAAAAElFTkSuQmCC',
table_gear=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJcSURBVDjLpZPtT5JhFMafrW997I9rscA+FFu2QRurtlw5cQ4InLpwBogIPNFSiNJ4C+JVkj0QTBHQKFPQlJfwlanY1tXz3ARkn2jd27Wz++yc33XOvd0UAOp/RNGR/X5zeH9rOlTDVKAK3fsqJrxlqN27GHPuYHh+G4rXRQzZNjEws47Hli/oo/PxNsAU3qvWT3/gX3TPuHrWBhiC30nSktXDtKLB1NI4NKkxqBMqjDByPFkcxNBCPwbCfXgUeEBq705m0AZM+qsk2e3hau88W+4ANOy+XPLFQrkrcbW31KkOYJx9rBaAOzPR0gVHW6x593q9cDgcqB6e4sZoogMYdXzD0ck5ZhfLsHGKVfAqVoadKcMdzcLr82PuwwZCoRACgQCWVzdhoK2gaVpDAMNzWzhkAXamQpze/I4t13w+j2AwiFwuh7W1NXg8HmQyGSgUCshkssuU3F7AQf0c84kK3n68KFc4hXQ6DavVCqlUCqVSSdaIx+NQq9UGMsHg7Ab2jxtwp5rOvqUqia3CUqnEObWn0mp1KBaLcLlckMvloPpfrhOAl230/SGLxQK3241CoQC9Xg9nskKk1emQzWZZkBZCoRBU3/NP2GMBgXTTObjSjI1GA8lkEgzDwO/3E4iObXY6nYhEIhCJRHoWcIW6b1pF7egMlYNT7NROUKzU8XX3GJ+3D2E0GgmAm4Zbh2s0mUyIRqMcAGKx+BIlMeSiYu1K/fbEMm4+TaFnJIHrSgZX5TFIZNPo7e1Fj9QOs9kMlUqFaw9pCASCnzwe7x15xG6/rUQiAZ/Px9/5XyhZOMVGKlOdAAAAAElFTkSuQmCC',
table_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI9SURBVDjLpZPNS5VBFIef+77X4lYayfVqpQY3XPiRGogp9rEpa2NRQpAELaJl5LKVJGSrFm36A6RMCpSilu5StETLaGcophTeNMyv+/rOmTMtrp87o4FhOOdwnvObHzMR5xz/s6IAz98vtDnHHeu0UFWxCqKKtYpVh1jdti1WHFZ1uONmSW0UQJ2721ybHf+XyQ9efqvcVGBV4wBvJjoRFcQKRgVjzdZpZbN2v74DY+zebYCMDy1lt3c1fSUQjNgtD6xVALr7U7sCXK7JJdwOkHUAwI3TCV4NzHK9IX8z1zM0S3PdVjy3GOxUYKyizhH1oWcwhe979A6liHjgrTfd6zpH2izRVP6aiqJDhCYD8Dau4ICoH6G5PkHUg2t1Ca7WJrhSmwDAOENhvJSu0YsEJkBEdypwClEP3o38wveh9cVZnBpCFUIRjuSWUFpwiqVghdbeKrJ5tg0ginOOLB8uncyjb2wO0ZAL5bewTrFqURw//kxzorCB5TDNyPcmKh8GBzMmakaO70XoG5sDIJAQ65Sp+XGMCqIGYw2La0tUF51h2azyYbJ/3gMQkYwHHpyvihP1IZAAsUJ+TjEFOcc4fDBJlh8jL/soo9MDfJz4ympIzZYHzlFRfICfv9Mcz4+RljU6Bx8TakggIcl4GXXJRoan+hmdGaMi9lR72ls+rz8kN9DePV4dWt1vxGLEEpNuolYRK+QAn2YaiXgxBsa/ULLnCQsribcAkd1+52RbZMWqvy+tNpl65CY38n8B/ZBP7EKNHesAAAAASUVORK5CYII=',
table_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ8SURBVDjLlZJZSJRRFMd/39JipZXZmCURrRBjKshkJW20CFGSQlAotENQr9FT1IP2HoFPFRFhCSMYGS1GEUUrkZHY2DItUs30kTnjrN9dehhzFAzqwuHcc+6f3z3ncIyL9/uPa80BqVWxUgqpQCiFlAqpNEKqESaRQiOVetZYv8gHYCutj9T5cgv4j3Piyrtlf+62VKoAoP3DBYQSCClwlcCVbtZLMfx2bEUjrisnjABoAHYt3fdPv8eSAldIsgCpAGh5EP4nwLaKfNIjAWIIALCzykPrwxA7VhUO5/yPQ9RVZmMnkhxdgSsVSmtsC/yPwliWSdvjMIYJ5pCo/WkYtKIodJ7C8V/wMh+oAMCUUqEB2zKoW+HBNqG20sN2n4canweAGp+H1RNusWBWiDlVu6ld8poXTYurAYxTbZ/0oU1F3O/5hWGM7tc2IZZS5DvX8ebdY3rpRmJ97xFRxcC7V/HAs679tisUWmvGWVBdPpPOLocNpdm1uP3yB4tzAkwt2YKMdzOxoJik7KNoTdkkEetvtoXKDNEyDTq7HIBhbxrgKuiLz2Ba/0dMy8EwY4zPcyAZZ/BbyrKFEJkZmLCupIB7rx3WejMVJMPdBB+dIX+6RokEpvoFZi7aTfPmciDZHYgctF2ZacE7dwrffiZYUJhD8Psg8vMNjM9Xmbe8Cp2+hSGj9NxMYSWTJFKSjx8i9Q0tQb8tpX54suVtWVqqya6QuEIiheLw7A7K128g7TRj21N5dSfNpU+bCedVMBiNd1y7UOMHQGs9pn1pq36SCjXpWGCrfn56ZfTsnoUNY+nsv63s16DTGRm4Oy/+M5bo7f1xdO+5t61j6X4DmUx477d3qncAAAAASUVORK5CYII=',
table_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJySURBVBgZpcFNbNN1HMDhT//9d926NWRb65BtB8MWAZGwZIGRcNAZD2qA6E6cOKDhAjFmBzwYYzQeOXHHiwfFAOHAZScUEmeUAPFlWzMDyOtm27XdoF1/3xdZ0iYcvBCeJ+HuvIiYlm+vVD535yN1GzIz1EDMUDXUHFFD1BA1RBUVR81+jWkx9xNTe7I5nsMX3y/uimnpjW7mGn+fYa1RxtQwMUwFF2VdI37s2kvVU4gJosKn+74mBE3HPFW6MZncnHybdGaAzKadeBA8CNqsU1+Zp2f0KK8PvguJiLbHDSGIEvOUqw0PRZdJdR1Aqr8RdY6hWqJRKfBnOMTS7T1wu8izDo730RQlLl57o8PVPuzuHQWSWP0RxOuU78zQ9+rHTL5ymA3nZpeYmhigrVhrEESJTXXMxY6ls6O41CH5MoSASJK/CvNY4SsiWSfv3Vy6+h6SGiAVw/bBDM2gxC52urN/PFcvzWNidGRGwGLyQ2/RUyqgoUlt6Qb3XjrJO3tHiFIZNiw+qCFixCZ69vH9n3/6vX5oevdwmpXCRXLDbyKNCs0nRR7KNmrbP6Oa2MKFa6vEiVUM2LGlE8fA3XF3vjx7y8srZV88N+YPZt73ue/2eWXhB2+bub7stSfB2+b/qfiRU7Me0yJmrF3/hHRnH8uNPKXRU9yrZ+FmkSgBweDK3AptW/MdqBoxLZvtF0LtDsv9x5nYP8XlP4pM7szRdn72Xz6YyNO2cLdKMoKYlqr0kh0/TbZnhIflOlsHurj1aA1VQ815bbCDhbtVnmXmlnB3Nkx/M3dVgu5uqnUHUYIoKkZQQ1T4P5XVxsWEu/Mi/gPrlHrAGd9XNQAAAABJRU5ErkJggg==',
table_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJqSURBVDjLpZNLSJRRFMd/M9/okF86IONzClciCRNGZS/BFoIa9BBDrSSC0F2rIFxZUAaDrWrXoggEqVBxHRSmofighUk2jEXjPBxHRbQv1O8+Wozo2MrowuFy7uH+z+/+OdehteZ/lgugZ3i1U2vapFaHlFJIBUIppFRIpRFSpYVECo1UaqKrtbTSBaC0vtNYme39l84PXoeO7hBIpbwAg99fIZRASIGtBLa0d3cpdmodZ7qwbelOE0j5cL389r66WxsCW8hdD6RUAPSOLO5L4NKJXLbSBcS2AMC1qnzefErQdK5g56xvLEHj6d18aW1jL4EtFUprXAb0jS5iGE76xxZxOMG5fWlwfBG0JsMFR3xZbNl/PUEDLsNBQ2Ueg+NJLlfm7yFIz0OxNYRQewm0ApcTBkYjRL5NEhiaI9+bS3Qhic9fQ++HX8yN9eMr9LK8vMKBdQ/V1R1mSkAotNY4tWB19h16fZ1bN1uxLItIJMLo+Ec82SbNV69gGAbxeJzMqSlWcnLup0xUKZzQzCTxYIjmG9d48vQ5s9MTFBUW4vf7Cf/8QXf3BPGFBY4dP0VtzXnC4XC7E0AIgQbCoRk8OQcBeNR5D2HbtLS0UFVVRX19PXV1dQjbpqm1HQCPx+Pe9UBryksPM/15MjWqDx8jhCAQCGCaZmqALAulFD0vnnHxQi3RaHTDobXm7suvI8KWFavJeXPty1vOnqyg2OcjK9NJIpEgGAzidrspKSmhoKCA35uKWHTeHh56P+dI/85lZWVGUVGR3zTNgeLi4jy3220AzqWlpTYg6vV6B4CMzc1NGYvFkpZlNfwBMWlOUI5ySkcAAAAASUVORK5CYII=',
table_multiple=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH2SURBVDjLpZJBS5RRFIafb+aOMzoaoqiYtEvBCaSFiEH7IKIQF1FIINJfiFaiC5f+iCIcXAwURX+gIkGigghBbSloEeYwM9/nveecFk5fmiMFHbjczeU5z3nvicyM/ykH8OTV/n0zpsS0X1WpJ4HEC0G0eQQJhqh+XpoZvncKoGa3pye6uv/WbWF162JLA1HtBnj6aX1Ak6GeIC4nqjSOmbRlChQLOeZXtndE9cPSzPD1Y4CjHDqs1HttvOhadTeDWiwUC1kWVrdKJw1EAfhRCy7xxsuN9xmNL2RE8lGLTAY729uYX9k2UV13AKEJEDXyuQjnR7I3xjv5h0zGHIAXRc0QVRJv1BMB4NmXRwQNHMRV6r6Bl0DQQJDAwytLeC/5dARr3vlclGZytzR3ZvdaHPBBSA1MIagSe00zKb/eOxNwc7yHw1+A79U6okoEZDNRmgnAnav9VNZ2mZ4cOAH4dhD/NhCxj4vlzZGB3i5qDUkzcVmovN2DCCpru7hMBGbkHIwOdXDom4Dl2dE5gAePN3YKbREiOmiAy0ZMTfS1NNjaOSAE5cTSiNj6Ynnz0mDfOUzBZeDFu6/AkQGAi0CB0vkCxh+A5dnRW00TE1Umh7sIYtQaPv2Z49XuIm25tiL2ZrG8eflQtOiDUK0lBAmn3u1X4+c/AQHoQ0fGFnyFAAAAAElFTkSuQmCC',
table_refresh=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKtSURBVDjLlZJvaNV1FMY/vz/3zrt77xjs7s7mVZNS3HAuFIWwXpRWexEYDANFKQpfyagE8Q8iUfinKHwTFL0wzBcWMWjpCxWJJQ6hHGOSNtzcTYmma7ipu97tfs85X1+s1DcmHnjgcB54zsNzTnDkzPge79ms3gpmhhqIGaqGmkfUHoKi4lGz3/ZuXLgSIDbvO9pXZnM8QX34/dDS//pYzXIAXcOHERNEBWeCU4czoTFTIFedpy6V48TVn9jSsgvntOohAQ/AhuZ3H7v5+JVOvuz7BCdreSCgBsDRs6P3hw21SZLZv+gdP0Hx1gAiijhlTrpASe5wu/pb4DMAQvlXAGD9C3miwDN/3ii/jneRiVKsbmxjzdw2Xl3QxuJcE00Nzcyum2btV68NAcRODfOeOILOc6NEUcjp4R6qqaa+Kk//jT6Gx67gKo5CzVxEhMGi0tVx8lmAUNXwQBwFtD+fJw7h77uX6L/Rx7HLP1KZCjn0xnccebOTcmkaK0ckJzfx8oFVqfsOvEEcwvHef4giePvpj8jMiggCuHmnwtDIXUSNFDk+eGUnW4b6+HlHTxkgdGJ470lE8PryembFAQvyKeprkuSySeqySX648DmLCxn2t39MGISkq+IHVxCbCTEKA073jwEwOFIim4JEmGDfqe1MTE5weaQEwKddHfx+/Tyt74sXJ1dDEZnJIIQ1rTniCBY+lWZ2bZqD3buZsttECeWdr19i0xcvUmKCDe3rWLFsGa4iJ2cy8J4l8zKM3CzzTEOK4vVJdh17i9psDa2LWlBTlixqRr1iXhksDvPLmZ5eFT0abD106aw4fa6ilnaiOFFUDKdGIvsNUWoCEeXiwB9IRRCnhDSOlaf+XFc8fK0b7/3/Yvm2lu6l7zVdeBQfPu7/K9NutavIwKP4e594fYOHzxflAAAAAElFTkSuQmCC',
table_relationship=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIpSURBVDjLpZP7T1JhGMfPn9RaznVZa7Zhl1WoOI1ZtNlmq5Wrma1jMTSG5li1ahWSFJKmjuhEYzVJCDGQUNJI7WYX7ALnhFwiKFvn2zkHKw6d33y27y/v830+++5535cAQCxHhN7+AR23I9Ba30EzMIeTva9BWl4+ljJbRhLqHk9i/trDOLpdDLoeMCAyuZ8oVtP1WVYKYPYsfCv2Eqd9bdB61dB4SJxwNQuHjcZnkAKY3F+Efu/0VZjDV9A9eVFoiIo37L88JQkwDjNCv7CIPm8MheINey+ERIC6/kpFtXkbdhjKUdtVIfITVn9URGRSOajOBv8ClH1yRZVpK9s63IL2kVbIz20RBvkaGI3mAVQgBmosCsd4FG8+p7Gzc0wA1Fi2KyqMm1nyfhNqjHKsP1WKct1GDPpisPLy0/8nePUxhWqdD1xkJReZbXY0oqxjLbtOU7JJf2ceqewibAFa8FKBJYCQgktg49Rg3QMuMupv1uGw/QA26Faza9SrZHyidtt7JDOLsAdp3B3Pixh6QiOd/bdZVY8SGjeJg1QDH5ktbVkp+7OPtsG3SHz9gXuhfALnJPeQHBM0ClVrqOIjg4uMkuMrZIW3oe6fEwBD3KBzScQtPy3awfNIEiq9T/IdkDdeYIEDuJ4ygtcd5gD8QLF2dT76JQU4ap5FPP0ddDKHT/EsInQGRKXWi2KVHXNSUoAjppnRQ4ZwZt+lKdSfD2H3meDyvjKv3+cfGcwF4FggAAAAAElFTkSuQmCC',
table_row_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIHSURBVDjLpZPPS9RRFMU/3/GbmhVBWVJGLSQoi5AIM+o/CCowBKVFP3AZ0SKUFlELiaBNhNugEKTIIRFc1iJFs2iRm8QoSiIdQ2Z0Zr4z8969t4U644ALwwuP++57j3PuO5wbmBmbibDvffKeGZ1iekBVEQWviogianjRNUsQb4jqx54rh5sBQjW72dq8o/Z/WO+//Hai2IGo1gIMfn+OV48Xj1OPE1fK4ot33Wd6cE6q1gAsa9DReGND7Jmcx3kpaSCiAPSPJDYEcOHULgprAfwKAED7ub28Gp2j7Wxd8WxgfI7WllL9dzFX3oETRc0IK2BgLEFFRYz4eIIgBrGVR4MTCTBjSwhH62souBIAjwdnLJUtWPxDwszM3qzk1Xg9NltWT/9OWeHaVTMzzIzgYfyndU08BUCjPJrOIKk0klpEkotIKo2mM4zER1GgcX81h253otmImuGhIHR+WQNruwQixHJ5gmxEmM5gmSwWRaS6H3HxdB359g40G5HcXY1UVuFOtljoVZeZe5+ty6xRHoDhz/Nw5wkNeyrZd/k8ms2xc/JTwN2+KVtYytvbL/NmZvZucr7szwNj5Zp8/ZW0hSNNRQ1CJ4qZcfzgdv4sRDTUbeXHbLo4C8fqK5maSZV54db1Xn1RMpKNPuifbiqIbnNecF4QrzhRvPh1zZRcyg2t7oPNjvM/JWOW3I/zgm0AAAAASUVORK5CYII=',
table_row_insert=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAITSURBVDjLpVNLaBNRFD2TjIq/TdGM0OJGXPhBFENq0aUrtQYJCJaC4GdZxI1IF1XBdiEuXSooBNKKisXfTkGtFqsIdmOxdaEUaSIlTtp05v2u902aTgIuKr3wuOfdee/c88685xARVhJu/k25jznOazJtxhhoAyibtcUExTkeGloR181Yf/f2TERgiHpymY2b/qfr1aHJPUsKmC3aPPz9HndW3EVBcpZaxplr9W+XO/ohpV7TQFDzoGvn2WV1nw+YVOnYA3tWG4W3xWURHE+3QDQSqEUCG6cOpXB/ZAYnD3pLtYejM8gdiOe//aBZgWQCNhJukhe/LyKZTODRaBFOAkgsLhr+wOp4zSoX2NG6DkLGBAl7BOuCm3SQ60jB5V13P3fjRCaFLA8bNmfbPRzZ79V+rTLNCojnduPTTyXc/tgFJVSEH09fgBQSD/ISYRAiXBAIqiECxulLgmzNlcxmb2NnejOO3TqMLS0eS5S48bwTSipcPzPAXTWqsoo5OYdK6KMifMbzGMwPwekbnKKLR9swNuXDYUkDL7LcVeFK9hnujJ9r7lytYVsTgYzUoTc/QbOVkF5+KZGNV+Mlau/dR/VgY6kxvv4o0+mb7yyMlNc8YLB76wb8ml3ANm8tCj2vMTntR4btal2NiZ9/mu6CMWQaLhKNXCt82yu0WW//rx2afZHR41H/vEzlSvCkjp2VPue/lFt5YsuGFGsAAAAASUVORK5CYII=',
table_save=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJlSURBVDjLlZJJa5RBEIaf7q8nGceJRMkiKCJIEEVUXMYtp+gxKBIQXC4iIjl496IomJuiv0BQD0HBQyCHgCIqxn29CYqKGjWZaGRiJjNfd1d5mJhE8KANzdvVTT1db1Hm8p2xE6ocjiqLRYQoEESIUYiihCizdiQGJYo87jnQVgBwonq0q9DQxH+sk1ferP59dlGkCaDv7UWCBEIMeAn46Gc0hum3Y1t68D7WzwIoAPtWHvqn3ycqAR8iM4AoAPTeHfknwM4NC0hnA8IUAGBvewtXB4fZs611+u7ag2G6Ns/Eo6XKHxWYc/1DeqijmYEXY1gAYzAot18UaaiHchqYn6/j88gkxkI+lyBqGCtNcvP50CYXo6CASwy7C830PSqyq9DCwL2PdBSWs2tdnv4Hnzi+f8UfVi7dKiJiHlofBRVwFvqfFkmSmparnu/jAYDrT4anEysBSlUo/kgxRrA+CKpKJoHO9c1knaFzfTPWGhIFH4Wz3esQUaIoCULOCdVqldJEigtSa2JiDTdejgJw4+Uo2YylUvWcvvIeayBEQabsqioo5DKCDSHUemBhx5omXFLTIEo1BhKbwRiHIcGYBNSiavAhkKYB52PNwqoleb58n2RZ6xzeff3JeNmTMY7uzgaCpIAFBTVC1tVxpvcD30oTuBh18FTv67VplLk+RHyIxCBTfjMIk1x4doSm3BIMlmL5I4c3nseSpVxNcWcPrmj/28TtOX1fly6E1vw429cUaKxbBBZ+VBbRMjeQqwNnDEZV/zqyG7qvD7YtnrO1sb6RrJtHEI8RwWhETJlXH94yUvL8AgiPWj6e64RKAAAAAElFTkSuQmCC',
table_sort=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAI4SURBVDjLlVJNaBNBFP52d2J+tB5iGjFpvUgJtaVGhIA/p+pBsRcRD4o38ejdQ5F66C235CKYg3goCh4KCh7sRVoETQwGhBhFCfgTG9GC2WazO7Pjm9E0S7CgDz7e23n7vvfeN2PcffrjhpS4KqQ/5vs+hA9w5YWKJTj5AQQEl3Tuv1i8PJEDGfOlvHY+N5LAf9jCvXcz/ZgRmy5efn+HOnPqwuGR94Q38HTWz10/ugjPE+EAgdTBpYNX/qm77RApFxgQqKXJllbXty1iooPwzwac6DjOHs/ADRLwPwTKLp5I4v7aV1w4lkSn09FnT161Edl8g6mZKdRqNXz8NIKItFEqlQ57nveBeURAQoJZwINn67AsE49W69jDNugmBBKUG5+cRCqVgrqlRqOO2TEbnO9+2Wq1VvQKSgVmGTiXG8Xy8zai7mdMZ3OIxWK6SIG6IZlMaujRGUOhUDipJ5C0BTOBh5U2TQB0u11dzDnXxZKmGPYKruvC9PjvgxAVzh0ZRYQZW4Vqhe28qlFTMfXqlFmmQYJ90zH3DfR6PR0Pdw/GegLVTWtAK5w6lNBihkMmHMfRPw53D5KoCUytAX1M79+FL9+7OLA3CnvTIS0smKap0d+5L14oFNKemgu6Bbl2c+lt1hX+TvXCFCZ2AIlKZatzJpNBPB5Hs9lEtVqFbdtaJ8Ito88ctGKxuI+Sp9WIBDOdTt/OZrMol8uo1+tzlNtQxITXfyUYtvn5+VkqWiGcyefzj4O5Xx9ItHsmdOWEAAAAAElFTkSuQmCC',
tag=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAEXSURBVDjLY/j//z8DJZhhmBpg2POQn2wDDDof8HvOe3osYtXzDzCxuM2vP3gvfn4MJIfXAP22e0Ies58eK9r2+r//3Kf3YOIhq17eK9v95j9ITrv2jhBWA/Ra7kVEr375vXDrq/9+s57eUy+4IY0kJx2w6Nk9kFzE0uffgXIRKAboNtxlC1/+/GPljjdABc9+q+ZcM0Z3qmb5LWOQXOmml/8DZz7+qJB0hQ3FBerFNyNC5z/9nrXqxX+Pvgf35OMuSSPJSXtPfXQPJBc089F3oFwE1jBQTLkiZNtw51jq4qf/XVvuwsPAa9Kjexkrnv8HyclFXxTCGwsyERf4LctvHvPuvAePBf8pDz/Y1N45BpIbKUmZFAwAR3nW32nUrY0AAAAASUVORK5CYII=',
tag_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHcSURBVDjLhZPZihpBFIbrJeY2wbcQmjxdIGSSTC4zQxLyAK4o7igoKm7TPW49LoiYjqLG3DWpZmx7/tQpsR1xycW5qTr/9/+n+jTTdR3dbhftdhutVgvNZhOapkFVVTQajSsA7FKxTqcDx3GOajqdSki1Wr0IYeRMAsMwpPNkMnEhdCZSoFQqnYUwikzN5EYH9XpdNU0Ttm3LcwJWKhXk8/mTEEauu0YhfhKRDcuysDBt5H5tk4zHYxSLReRyuSMII+dd5M1mAxL//uvgw8Mz3t4DWWN7NxqNKAXS6fQBhIkZ+Wq1kk3r9Rpz4XytPeNLF/iqAx8f9pDhcEgpEI/HXQir1WpvxIx8uVzKps7Kls53AvCjB3x7PIQMBgNKgUgkIiGSUi6XFTEjXywWsunxj433qoM7fQ+51oDMzy2k1+tRCoRCoSt3lkKhoIgZ+Xw+P4J8F4DPTeDm3oK92aZIJpMIBAKvD15UzKdks1k+m81cyDsB+SRGuG2tYVpPL8Ued4SXlclklFQqxWkTCaILyG3bgWXvnf1+v8d9xFPLkUgklFgsxmkTd5+YxOL8QHwWQBWNRr3ipTktWL/fPym+CKAKh8PeYDDISezz+TwnV/l/v6tw9Qrxq3P3/wBazDrstPR7KQAAAABJRU5ErkJggg==',
tag_blue_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIxSURBVDjLfZLPi1JRFMffPzGLNkW7Ni1aJUitI4IWLVpm0RTUohazqkVU0GhjGcGU1NA6dFQ0FX09QVHxVzr+eE9RRMw0NVslPcmn8517nulkOj44XO6953y+33Pf4SRJgiiKyOfzyOVyyGazyGQySKfTSKVSawC4VcEVCgWMx+OFaDabKiQej6+EcKRMBY1GQ1Wu1+szCJ0xF4hEIkdCOLJMyaRGB8lkMt3v96EoinpOwFgshmAwuBTCkeo0kRX/YZYbg8EAnb6CwLeJk1qthnA4jEAgsADhSHlqeTQagYp//B7j+d4+nn4BhMbkrlqtkgv4/f45CMd6lHu9npo0HA7RZsqGzD7eiMA7CdjaO4RUKhVyAY/HM4NwiUTiHOtR7na7alKhp6jKZgb4UALeF+ch5XKZXMDpdKoQlRKNRrWsR7nT6ahJxZ8K9OkxzNIhxJAB+K8TSKlUIhew2+1rs15CoZCW9Si32+0FyA4DPPpkx/23Otx6eRk6/QU8MW9gd3f3xNyLsv60giDIrVZrBnnGIA8cH/HYeh1ucRvZ7zxMn+/gquk0zt49Zlz4rzzPa30+n0yTSBCJQa4ZLsJZeAVn8TXNCozCOkzCbQIMlk6X1+vVut1umSaRIJcenoFX3MG/nyu/TYCjZ9zlcmnYS8s0YOfvncQWfwObvE4t3vTrVjuYhsPh0NhsNnnDtI4rxlN4wd9UlWml/dI3+D+sVqvGYrEcZ8l6Fr/I9t9VT/cHUXogzRNu46kAAAAASUVORK5CYII=',
tag_blue_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJPSURBVDjLhZPNi1JRGMZv6PwBLQaiqBiIoEV7QQaiaZazrNm0qlkMRJugptpEBNJMtAmapK9lhZ8oKnq5FxQVv7qOX1dRzBxHU8eohc219OrTOVfGSZyxC4cL73nf3/O857yHEUURmUwGqVQKyWQSiUQC8XgcgiAgFovNAmCmLSadTqPf70+sarWqQMLh8FQIQ5VpQaVSUZTL5fIIQmPEBQKBwJEQhlqmyVSNBqLRqNBut9Hr9ZQ4BYZCIXi93kMhDFXdTyTFf4jlSqfTQaPdA78zdFIqleD3+8Hz/ASEocr7lmVZBi3e3etjY2uAJ58BrjLcKxaL1AU8Hs8YhCE9Sq1WS0nqdruoE+X1+ACbGeC1CDzbOoAUCgXqAk6ncwRhIpHIPOlRajabSlK61VOU9QTwPge8yY5D8vk8dQGbzaZAFEowGNSSHqVGo6EkZb/38FToQy8eQNbjALs9hORyOeoCFotldtSLz+fTkh6ler0+AXlLAB/1L8FevwBuYQb8tVNwP74Bk8l0duxESX9ajuOkWq02gugI5MOrTSTuzqPjfI5B1o29T3cQu3VRZhfUtyfulWVZrdvtlugkUohIII7lc5BIMV4sAWvHAd0cWhuX4LmiKh06XS6XS+twOCQ6iRRCbQ8EC/79fj46Ae6yenDkjNvtdg05aYkOGHf1JH69uwmQot/3GPwga3tVBc+iqjr1pVmtVo3ZbJZ43SoiK+flb2tz2H0wgy8rx8AvqWX3ouoh87/najQaNQaD4Uxg+fR9oviV2ib/HVpM9/8Cz3kffqwCPcsAAAAASUVORK5CYII=',
tag_blue_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ+SURBVDjLfZPNTxNRFMXnX3DhxkRN48aEf8CkYaFGAy7cKImRCAv8iInGnSDGhSFEsJqoUWxABGNQ0zZFGrWRfqS1rf2ype1MC5FaytBmirWK1M7YTofjm0GaUAqLt7lzz++ce98bKh6Pg2EYxGIxRKNRRCIRhMNhhEIhBIPB3QConQ5F0zQkSdpyMpmMAvH5fDtCKNlZFrAsqzin0+kaRK6RFHC73dtCKDmy3Cy7yYVAIBAqFosQRVGpy0Cv1wuHw9EQQsmuG41EXCaRWUEQkCuKsC2tJ0mlUnC5XLDZbFsglOy8EblarUIWfy9JuDuzhr4vgJVd/5ZMJhGwPMLcxBnMT/UjMnoc0SdNHRSZkc/n80pTpVIBR5wHw2sYYoDhOKCZWYf8yUyCs3djJfYOKGbxK2aA915LjvL7/c1kRn55eVmB0HlRcdYSwPNZYCQBvPn8HoX4bZSXTOAcGvC0EdW0G/TYRUGZw+PxqMmMfC6XUyCJHyIGQhK0JIEp7ESBuQWJd4P/dgWFwFXMvuwiI5yHc+TaodoynE6nmmya5ziuBhl32/GTvgFJ8KKU7ITAtmM10YvU2ElYJl/AYDDs37RRsmW11Wrls9ksssxbcMHu/+IOCItn8Zu5iaT2BOzGUVmsUm6h/lqmp6fV5ld95cTH6yT2JwgLXfjLnsMKSTL/tAUW02vo9XpV7RrrAc8u7+2Kf+hF1PwAGX8bSgvtKER68HWYxJ6a2CRuCNANHgVW05gbPg177x54tK1waY7BYhzfIm4I6LvQJJa9j1H2P4S//zB0lw7ArB+FTqdTNXzK9YW25l3Cnc6Dom2gVbTeP+I0DvWcIuJ92/1M/wCZISaoLgB85AAAAABJRU5ErkJggg==',
tag_green=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH3SURBVDjLhZNJjxJhEIb7T8xVw78g6fh75mJGY/SqTjRGTXTcbu7xxtItPew2YxMgbGkCNMgSQkgDAiHRKNpkWPL61ZfQGcLioS5V9T711tfVQrVaRaVSQblcRqlUQrFYRKFQgK7ryOfzBwCEfSEYhoHlcrkRvV6PQzKZzF6IQJNJYJomn9zpdGwI5ZgLJJPJnRCBLFMzTaNELpfTJ5MJ5vM5zxMwnU5D07StEIGmrhqZ+JxZNqfTKUY/u4gbMq+1220kEgnEYrENiECTV5YXiwVIPP71Hc9Or+ORdAjNkHit1WqRC0QikTWIwHa0xuMxb5rNZhj+6OLk9Bpef7mN92f38JyBVpBms0kuEAgEbIiQzWavsB2t0WjEm76ZOTyWD/FWvYtP2kN8OLu/Bmk0GuQCPp+PQzgllUqJbEdrOBzypnq3gKe+I7xTj23IiXKEWMnF67VajVxAkqQDe5d4PC6yHa3BYLAB+fj1AV4Fb+KlcgvzxYy7CIVC8Hg8l9delO0nqqpq9ft9G/Lk81W88N/Am+gx/vz9fVHssFe4GNFoVAyHwxZdIrdr6kx8B9Nzyxa73W6H/YjbjiMYDIp+v9+iS1x9YhKz/Jp4J4BCURQne2mLDqxer28V7wVQyLLs9Hq9FoldLpdj6yn/73dlU51MfGlX/R/5GCirExPTUwAAAABJRU5ErkJggg==',
tag_orange=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHcSURBVDjLhZPbahpRFIbnJXLb4lsIQx+sF6G0kMsmpZQ8hEeUGWcUTbQqnlDUUTwgIkaj4kUI0j3ozObvXjs4jXjoxbpZe//f/689a5Rut4tOp4N2u41Wq4Vms4lGo4F6vY5arXYFQLlUimVZ4Jwf1Ww2k5ByuXwRopAzCabTqXSeTCYehHoiBQqFwlmIQpHpMrlRo1qt1jebDRzHkX0ClkolZLPZkxCFXPcXhXgrIk9t24bz8gyna8qz8XiMfD6PTCZzBFHIeR/ZdV2QmL+u4Bpf4cY/C4ghz0ajEaVAMpk8gChiRrZer+Wl3W4nnd3EF/CH7+C5n+ACtIcMh0NKAV3XPYhSqVQ+iRnZarV6gzw1pTN/vAPP3x9BBoMBpUAsFpMQSSkWi6qYkS2XyzfI3IKjixSPP/5BRCrH0uR5r9ejFIhEIlfeLLlcThUzssVicQz5/Qs8eYM/+g2468gUhmEgFAp9PHhRMZ+aTqfZfD73IDvtGtz8Bjtzhy3bvBf7vBHeVyqVUk3TZLSJEjJrw3m4Bd/anjgYDPq8Rzy1HIlEQtU0jdEm7j8xiUX/QHwWQBWPx/3ipRktWL/fPym+CKCKRqP+cDjMSBwIBHwnV/l/v6tw9Qvxh3PnfwF+wjbwD++YrQAAAABJRU5ErkJggg==',
tag_pink=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHVSURBVDjLhZNbi9pQFIXzJ+a1xX8hhP7Q0k47zFPHK4p3FBQVbyhe8YIVsTpJGid1SIiXRFbPPmI6jpc+7Jdz9vrW2ic7Qq/XQ7fbRbvdRqvVQrPZRKPRQL1eR61WuwMg3Cqh0+lgv9+f1Xw+55ByuXwTIpAzCWazGXeeTqcOhM5YChQKhasQgSJTM7nRQbVareu6Dsuy+DkBS6USstnsRYhArsdGJt6yyLP1eo2tuoFRWfG7yWSCfD6PTCZzBhHI+RjZtm2Q2HrZQvsm4eWzBKN8gIzHY0qBZDJ5AhHYjKaqqrxpt9txZ+1ewurHb6x8KrTvsgMZjUaUAtFo1IEIlUrlE5vRVBSFN5kjnTuvPCpeg0u8voMMh0NKgVAoxCGcUiwWRTajKcvyAfLTwPLrO8i9DL14gPT7fUqBQCBw58ySy+VENqMpSdI5xL/En0cF8sMv7K1Ding8Dp/P9/HkRdl8YjqdNheLxT/Il2doDzLUp2dsjM1bscsZ4W2lUikxkUiYtIkcMjagPSmwN7Yj9nq9LucRLy1HLBYTI5GISZt4/MQkZucn4qsAqnA47GYvbdKCDQaDi+KbAKpgMOj2+/0miT0ej+viKv/vd2Wubib+cO3+L9t+PiarnEW1AAAAAElFTkSuQmCC',
tag_purple=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHpSURBVDjLhZNbbxJhEIb3T/RWw78g2fjLvLE2ppe1TYNtvGuNRo6BcA4kIBBOgXCU3QXploCAmNQE/VY55PWbj7CWcPBibuab95l3ZmelZrOJRqOBWq2GarWKSqWCcrmMUqmEYrF4BEA6FFK9XsdyudyKfr8vILlc7iBEos4k6PV6orOu6yaEctwF0un0XohElqmYulGiUCiUptMp5vO5yBMwm80ikUjshEjUdV3IxX+45Z5hGPj29RcykbF463a7SKVSiMfjWxCJOq8tLxYLkPj72MCbEw3nz1WkwytIp9MhF4hEIhsQic/IJpOJKJrNZqKz7aWGm7Mu3l/quDppmxBN08gFAoGACZHy+fwzPiMbj1dFSvVBdL49v8PHq/stiKqq5AJer1dABCWTych8RjYajURRu/EDtmMV7y7+QWzHGj4FV++tVotcwO12H5mzJJNJmc/IhsPhFuSDTcfb0w6uTz/zr7MQLkKhEJxO59ONjfL55FgsxgaDgQm5fKHg+lUbtxdt/Jwaj8UWc4THEY1G5XA4zOgSxeLqD7h5/QW/jbkpdjgcFnOJu44jGAzKfr+f0SWuPzGJeX5DvBdA4fP5rHzTjA5MUZSd4oMACo/HY3W5XIzEdrvdsvOU//e78q5WLn6y7/0viZYv/mL7AwwAAAAASUVORK5CYII=',
tag_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHiSURBVDjLhZPJbhpBEIbnJXxNxFsgjfJ8VixjYscXx7EdKzdWgdgFDggQm0CsZhFCCIEAcRx7EN2Mfnd1xATEkkNdqvr/6q/uaqXVaqHZbKJer6NWq6FaraJSqaBcLqNUKp0BUE6F0mg0YBjGXozHYwnJ5/MnIQp1JsFoNJKdh8OhCaGccIFMJnMUopBlOkzdKFEsFsuapoFzLvMEzOVySCaTByEKdd0cFOKVsDxaLpfg8xlY6o+sDQYDpNNpJBKJPYhCnTeW1+s1SLxeLMC+2cC+noMlX2St3++TC0Sj0R2IImbUF0JAhxhj4DPR2W4Dv78Df3oAu74yIb1ej1wgGAyaEKVQKHwRM+rz+fwv5LUFdnEO/nAP/vsZ/NfjDqTb7ZIL+Hw+CZGUbDarihn1meguIZ0OVleX4I9bEPslWCIu6+12m1zA4/GcmbOkUilVzKhPp9N9yPMT2O0NtO/XMMTrkItwOAyXy/V550bFfGo8Htcnk8k/iO0C7MaO958/sHrTtsUWc4TtiMViaiQS0WkTJaT9ipUQG+J1NmKn02kxL/HQcoRCITUQCOi0iZsnJrHI74iPAij8fr9V3LROC9YRoxwSnwRQeL1eq9vt1knscDgsB1f5f99VdLUK8adj9Q9ogTPkuLLcmwAAAABJRU5ErkJggg==',
tag_yellow=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHcSURBVDjLhZPZihpREIb7JeY2wbcQmjxZbrIQ5nKSIYQ8gyuKOwqKihuKKy5xnJ5GRzteTIjTp51e+HPqDDaKSy7qoqvq/+qvYykNBgP0+310u110Oh202220Wi00m000Go0rANKlkHq9HhzHOYr5fC4g1Wr1IkSiySRQVVVMVhTFhVCOu0CpVDoLkcgyNdM0StTr9eZms4FlWSJPwEqlgnw+fxIi0dRdIxe/cMuqYRgw2SO2v9OiNpvNUCwWkcvljiASTd5Ztm0bJLa3GvTpZ+iT9xySErXpdEoukE6nDyAS35Gt12vRZJomTP0R+q9PYPc3MB6+C9AOMplMyAXi8bgLkWq12ju+I9M0TTRtnzp45pOZ8g2G+vMIMh6PyQUikYiACEq5XJb5jmy1Wr1C/vQ55CMM5XYPwr+1hKgPh0NygVAodOXuUigUZL4jWy6Xx5CHH2B313gaXcOxLeEimUwiEAi8PXhRvp+czWbZYrHYg3yAfvcFf6e3eDE2+2KPu8J+ZDIZOZVKMbrEV0gPz/df4ViGK/b7/R73EU8dRyKRkGOxGKNL3P3EJOb5A/FZAEU0GvXyl2Z0YKPR6KT4IoAiHA57g8EgI7HP5/OcPOX//V35VC8XvzlX/we1NDqN64FopAAAAABJRU5ErkJggg==',
telephone=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKpSURBVDjLpZPNa5xVFIef+877Tmcyk2Q+M0onSVOKnVQRMaBWNwq6c6NQ0IV/gLjspuBGkEKh2JUb14J24UZol5EKRaqhtjiNCamtDcSkNpkknUzn/bj3nOsibbF02bM5cBbP+R0ejvHe8ywVfnj6YqXdKP88NpJv4w14RbxHrCVNLdZasiQlTVNcnCJJTBYn7G7urK8u3X43nJ4Y7R5/cbI906oBoA8TiXpU/T5MFFGPiCDqsSL8fv3P2qW0vxQerJfazZEcK6t3cSJc7d7hjbkOC9dWeOWlKZxVrt24w+zsIS5f7jJ5aAIR5YX2OM3nnq+GxuxvjXKG7YEydIahRGwPLT9duYmIoj4go0hq8vS2+ky3qzhVjDGYL779xQcuJlCPCXIcf/UoO1keUX14BjhRnCpj4ZD5+QXSQQ+XWTb/6RP+urj6Safpz9THS7lms93K5Ytm/eYS63d7ZE5wThBRrMB777zGkWOH9dbiXndjr69/Lf12zjzS+Nm5C9+9/vLMx53D04yXRzg6ETyh65sLf1AYKbK59S9XF7oXvz/76fsAAcCJL3+cqlfKH7SadTKB8oGnfTcqoyyv9qhW6kTlxpuP5sHbJ8+beHv3h85Mq4AJwQSMFcxTgFZtlJ37A8JCiamDY9W5t059DhBc+uoj35mdmatWqmzsWkSVYvQkwHtPqRiRxDG9PaFYnWK0VjkBEAIkmQYbW7vc2nhAkjaYv7JInGQ4UVIrWGtJk5QsSVlZXqEUKUHQDx8DnLVYUTLnWL69hqjHOUVlv4sYCPIE+Rz9BxnjtYgwnz/yvwQZcZwx2TiAuBD1HhXFOYf3EU4EFY/3iqgyHMS4LOMx4O/rN5aH9zY7UaGEikNVUSt4l6HOgxfEOvCKqpIMBtxbW/sawDzrO/8H1LyIqK9H3tEAAAAASUVORK5CYII=',
telephone_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALuSURBVBgZpcFPaJtlHMDx7/PmTZq26b8laZ2m69qJS6dWUVa7oqDOQg9DUPTgwYPgRaenXeZNGEJB2UmLiiCi6MBd1O6iqzgota5sq8uy1nWbrX/Sbm2TkmZJ3vd9nt/jWxARvQh+Pspay//hPv3GqfZMKnGmtSmWwSqwgrEWEwR4XkAQBPh1D8/z0DUPU6/h1+psrpUKy/PXn3B7OltyB+7uzvR27WCbWMs2IxYRi7EWYwQjFmMMRiyBMZyfu7zjO688796RbM6kmyJcWV5FG8O53BJDD2aZvXCF++/ZhQ6EC5eW6O/fzdRUju7dnRgj3JVpI33bzg5XKRBriUYUxYpQ1YqqiVKsBnw7s4gxglgHn0Y8FWNjvUxPpgMtglIK9frH09bRNRyxKCfCgQf2UvJjGBGMWERAG0GL0OpWmZycxatsoP2Atd/LuD/kl5/Ppu1Ysq05kk5nuiKxRlVYnKewuoGvDVobjBECAyOPDXLnvj65lt/KrWyV5er82ePKWsu2w8cnPn1ooPe5bF8PbYkm9nY6/N17ExeJNzWytn6Dc7O5U5+9+dIhQg6hZ499sSvZnniqK53EN5Bo4F9S7S0sLG/Q0Z4kmkgN8yfn0SMnVK24eTLb2xVHuaAcWuOKfyqWp6lWxzmbfw2JftDx+Iuj3xCKLE2f5Pwv5Xf37cmo1bKhrTlGX7oBR/GXie9PsFg8zSP7Bzm4fxQ/dpMS+b63PjnW4hKq++KsrG9ybeUWdS/F5EyeWt1HG8ELDJXqRzxzaBTjGO7bOcLpy58zdO8wS1e/POwQ0kFAYARfaxau/0bpVkDFE6qexfOFUuUGUZXgyf5X2Xbk4Pvs6RwgFHcJ1X2fWs2nO9WA0S5iLWIErTXWRvnp1xL5wjQ/FqY4OvIhY1+/QDzSQKjuEvp57tJC9eZaNhpvRoxGRJDAYLWPaEujGmTm4jTDAw/zVW6cmHKZmjtD6B1lreW/GHrl9jHgZaAF2ALGZ94uHP0DwIeG5DahstsAAAAASUVORK5CYII=',
telephone_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALqSURBVBgZpcFLaFxlGIDh9z/nTDJJJsmkM5OkOMlkSklT6w3FSxW0XooibioUKuK6qKBYXbgsqMULdFnRrWBd6KLSIBYjtcZaDW1C0zSxbdoG2lyaS0OSzplzzv9/nxNQEd0IPo9RVf6P4Pn3+rPFfOaHlsa6ImpABaeKSxKiKCFJEuJqRBRF2DDCVUPisMry/M3pqfHLTwWl9ubR7du6iuWODawTVdY5UUQUp4pzghPFOYcTJXGOMyPnNxyPVsaD23JNxUKjz4WpWaxznB69ykP39TE0fIF77ujGJsLwuats3drD4OAoXT3tOCf0FlspdG5sC4wBUSXlG5bWhIo1VFyKpUrC96cu4pwg6hHTQGTqWFxYoVRsw4pgjMHs/+ykejbEE8V4Ptvv3cLNuA4nghNFBKwTrAgtQYWBgSGitUVsnDB/fYXgl7Gpl/oK+n6utckvFIodfl2Dmb44zvTsIrF1WOtwTkgc7Hz8ATbfvkkmx1ZHZ1ZX5NL4rweNqrLu1YNHP3/wrvILfZtKtGYa2dLu8XefHD1LurGB+YU5Tg+N9h/+6OXnqPGo2f3Oke5cNrOro5AjdpCp51/y2WYmphZpy+ZIZfIP8wdvx5tfmHBp+cu+ckcaE4DxaEkb/qlj/Cse/eZFej64m93Dr7V9+MSd31JjVJW3Pj3hnry/15tbVcqdzTyyOUPg8ZdrRw6xdPIwvY/tor68jfDsMcZ+/E6XJ86/HlBTjcWbWVhmcuYW1SjPwKkxwmqMdUKUOHYOHOCZvftITx6HE+/S2JqlXCqZM7/pGwE1NklInBBby8TlazhRrBXEKdYKLWs3SHeW4dl9/CnYvxFfTE9ATTWOCcOYrnw9zgaIKuIEay2qKcLmHJWRfpq+foUonKUCrK74OJ/pgJorI+cmKjfm+1LpJsRZRARJHGpjxCqp7A7yPw9SyjUQ+ClW5y1X5jxV5JBRVf6Ln/Z0v31raXqv70zJ+Xpd4eOnj9kDvwPD/42l5s+BKQAAAABJRU5ErkJggg==',
telephone_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMPSURBVDjLpZPba5tlAMZ/75cvadOmzdI2jTNp03TDJM457fAwmYeLyUBU2EBRmBfzwgvFK/0DBBF2oxc68MKK4EAKiqBUGV60qY4yWbVh2cza0nXFmqQ5NS1pvuP7fl50ThG88vkDngM/HuF5Hv9H+un3vtuXGArN9fcEEngCPIX0PKTjYFkOjuNgmxaWZeEaFtI0sA2TVm2rtF68eUJPDvcVjh0aSaRiAwCo242k8lDK2zOTCqk8pJRI5eFIya/53wZy1k5Rjw/2JqI9PpbXK7hS8kvhFo8ezXBlcZkH7hvFdRSL126RzY5x6VKBkbFhpFTckwgTvWt/RBdiL9XvEzTbio4r6Eg/zY7DzOUVpFQoT8MmiCUCNOo7JBMRXKUQQiDeuTDvaa6BpjyE5uPYRJotO4BU6vYMcKXCVYqknCVRmcYXOsxufY4vF+PoP19ffyUT9c4Nhnt90Wgi5gsERWmlSKnSwHYlriuRUpHuX+RQWhI8fIZw6kFaa2lO1ie3xV8Y3/hg+otH7k+9nBlPEg71kB7W7qDqlL/HaF6hLzxBc3WJ8OABAn3DFGc/N3WAF979ZjQbD5+KRQexJYS6/uZs1n/Eal4mMn4SqzxFoEewnr+BY9oN02g/pj311pQwmq2vMqlYN0IHodHfLfaSN3MYlYuEDzyDWfoE4d/CH+rD61T48Ifg1ENvfr2s5d5/yctkU0cj+yKUWw5SKYJ+QaX4LTu/XyR88HnMPz5G8zvYO2OU5vLMhl5nxUgdB9ABTFtp5XqL1fIupjVEfmaSE2Mlsk8+h12ZxBfwMLeTlOYW+Mw8i7QFmrajA2gAruPgSIXtumwuXODpZA1JnHLhIxQmu804a7mrnN86w8ZuD7om0AOBg/9oYGMYNiNDXeyv57j3+KcsTb3N8k/zdI0eYbdRphB9jaG7h4koRadt4No2dwzW8tdudKq1jL+7l0ytin19mtSRx9muNVhdWGIm8CrV1Rp4myilMNttqhsb5wHEv+/84hMDxsR4TH84m8LzW/NGtXb22XNXb/7Xnf8EvI+SIusIR2cAAAAASUVORK5CYII=',
telephone_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMGSURBVBgZpcFNaJt1HMDx7/MkbdMkzZK0SZxLmlZc07pN+3KoVcEdehR0ysAd9OZBvCmIIIhMPMjQswheFF+QIVitJytTRi0r2crqbE1dbddtVdq0WdbmyfPy//1MQRS8+vlYqsr/EX767elkvif+QyLankctUMGoYnwf1/XxfR+v6eK6LoHjYpoOntOktrV7e31pdTJczHYtThwr5PtzaQ6IKgeMKCKKUcUYwYhijMGI4hvD5YVf0hfc+lL4SHcsn4mGqKz/QWAM5cU1Hh4bZP5KheHjvQS+cOXnNYaG+rh4cZFCXxZjhIH8ITL3HE6FLQtElbaQxc6e0AgsGqaNnYbP93MrGCOI2nh04lrtVLfrFPMpAhEsy8J68+NZtQMHWxTLDjExWmLXa8eIYEQRgcAIgQiJcIOZmXncvSqB57N1q06oszR5PW7vjUc7aGRzmVgmk7V+W/mVlcoKGxs32LixTnTvJ06Ep3DjIxCJyd3a9tXqnfrmcnnuVUtVOfDSe998Ov5g/5nB+4ocikcpZW0OSOBw69IbxLLjLK5uMl8boTy/OP3ZuRefoMWm5fRbX/V2J+OncpluPAPxDv5RW5umMz1M4vAIqf1ZCgmHtnjPI/zNPvnK55azUzs/2J+LYIXBsklELA54d29SrUzR1ZPE1KcoPPQceWeGviPx1Nijr71Oi33h3Wd1cKh/LJVMsVnzMSJ0tlmgytbSF6TvfxKal7n00SfEkw6xWplirEZXOnmaljAtTU/sze0a1zf3abo9zMxdIx0s81TfDpn+DKb+O6hg9ubpP/ky1W/fJ9peaqfFpiXwfXwjeEHA8upN7uw3OabfkewdRhpXUXEYe6aEeLeJRCocPTHJY4WNo7TYtDQ9D8fxKPR0UMxEmEgsUBoYJdq1jQZVsEKUv6wAgjQrZAaKjCZXQj+ePf6Apao8fuadpdy9ucG2SAwxAS/kPmTi+Q8I2WtosAso/7IJdZb489oCla/PnrdUlf+aPTe2rUaiKoqKoCKoCCqKiqCiIIKqlv8CsLWQKDaepcQAAAAASUVORK5CYII=',
telephone_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALzSURBVBgZpcFdSN11GMDx7/+cv06Px3ePujrqNHG6RYxazbEaxdquuinoIiLqtoIQ1nUrogiqXdTdroLIIroJ2l1GDW0uNzWcHdGpO2tTm1M3X875v/ye5+kIuwi6Cfp8PDPj//Bf+OB8XbYp/UtNqjyLeWCKmCFxTBjGxHFMFISEYYgrhkhQJCoG3F3dWMrnFp71O5qrp44ebMt2tjSwS83YJWqoGmKGiCJqiAiiRizC+OQfDT+Hmzn/wcaqbCaVZDa/ghPhytR1+h/rZWxilkMPt+NiZeLqdfr69jE8PEXbvmZElJ5sLZnWvfW+54GaUZb0WN9WCs6jIGWsF2J+Gp1DRFFLEFFJ6JWzdmeTjmw9ThXP8/De/fJXS7giCTW8RJKjj+5nIypHVBE1VMGJ4lSp8QsMDY0Rbq/hopjVW5v4l6bzr/Rm7KPG2qpkJpNtSZZXektzOZZW1oic4JwgosQCJ595gu4DXTo/vTW1vLWp13K/nfXMjF1vnv1h8MgjnS/1dnVQm06xvznBP7311XGCeJvu1KdMjM2c//rj15+jJEHJi+9/395Yl36+JdNIJJDew7+ELibb1Mf4xhv41bXHuM9/+vQ3XlrD73qf7KnA88FLUFPhMTB4HDNHpI7IOR6o76av9QhbwQ6/F0/XHT4x+t7loc/OeGbG2+cuyInHexJ/bRmdrdUc604zMNjPyYOvIqaICoqxcu8m9VXNXFz4kSs3LhC4sM6nJIg0sXznLvPLOwRhE0Oj0wQaIabk1+aI1eE0JpaYzXCLQ21PsR0XuLQ4vOZT4uKYWJTIOWYWbiJqBGUBThwtNe04FcSU5Xs3aEi3Mv7nCKOz41Y0DvuUBFFEsRjR1rQHcT5qxq3VkC8ufkKkEYGLeKjpAP1dpxjLDzNybYzmuZfjkW/PTfqULE5enSncXu0tq6hCxaGqJOMzpFxEhTNqTBgvDuAlKhnJXWZv/jXW59c/p8QzM/6Lrne8HdFkqqjSdftDW+S+vwHrxbCSH7ilcAAAAABJRU5ErkJggg==',
telephone_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMDSURBVBgZpcFNaJt1HMDx7/PkSdombdO0Tedo2q6H2VSnlZXBylAUN/HiRRBksNsQ8eLNsyBFL9tJUA+CMnUiXmSbgqOjgyF9YVu3pmvarG+sa+eatmmb5snz8v/9zA4Dkd38fCxV5f9w3h2+0pJpb7zeHI9lUAtUMKqYIMDzAoIgwK96eJ5H6HqYqovvViltbK+tzC6edHo6mqaHXuzK9B5o5QlR5QkjiohiVDFGMKIYYzCiBMZwa+pe66i3O+t0tiUy6XiE+ZVHhMZwc3qZ44NZJm/P88qRbsJAuJ1bpr//EDduTNN1qANjhOczSdLPHUw5lgWiSjRisVUWKqFFxUTZqgRcGytgjCBq49OAZ8XYLO7Sk0kRimBZFtanF/5SO3SxRbHsCENH+9j2YxgRjCgiEBohFKHZqTAyMolX3iT0AzYe7uKMz6ycyab1i7ZkIpJOZw5EYg3WWmGWtUeb+KEhDA3GCKFRzmTzfPjCEpNL9Q+uLGaK92cnzqOqqCofnbv00/dXczq+sKf5v43+1+UfPtfi+Mca7I3r8u+n/ZvDh99WVSxV5b3Pfuvu70zOnRjM1ifijfR2xOlM2jy1c+9X/OIlUgOn2F9dINwTdu7frcxN3jkbGd3LWnZl/+obx/t64vEmYrEYPW1R6hyLp3ZyF2gZeA1x8ziJTsTfpSV7MBopbZ+0R8+9r9n+3sFUS4r1UoARoSFq8W+RRIZgexnxi6g/R6x5BaozlNe9iENN1Rd7vVhiYX2fqtfOyNgMbtUnZZZ5MzlCY5PQ3WJhSwnsJjTwyV0s6Nxc6QOHmjAICIzghyH5xVWMKIdlgmN1Y7z86inU/xPL7HHncgVT3iVm23ybHwi/+fm7iw41Vd/HdX262uswoYOo8rrm6DvxDn7xKxwnSe6az/X9t6i0H8WtuBTcW0KNQ83SVC5febyRjdYnEBMiIgwdCbAiD4nUpZn6Y4cf777EA00ihQLVcpnHq6tfUmOpKs8ycf7YcGNr6mxla9+dn9/45PTX87/wDP8ABifE0eTdPioAAAAASUVORK5CYII=',
telephone_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMfSURBVDjLpVNJaFNRFD0v/ydpM5imSZom/Ro7YNJqRVScQQV14wCtFRvRpYK4Edy4FIpgEetCRF0LDiCIoihCxWqtQ+mAsY2kDk0wP4lJrMQ0yZ99/VZRXPrhcd5/7w7n3nMf0TQN//Oxnafv13Bu28ACi4mDRgBNhUKDKpIEQZAgURQrAt0LkMsClEoZYrmCb9kZPh79uI0N1Nkj65cu5Bq9tXpEdZ6RompQ6dKDKar+ryiKjhLF0fHJ2idCIco2uKycx8IgFk9DphcjkWmsWxXC8FgMK5YtgiypGHs7jdbWxRgcjGDh4jo94BLOAU+9z8kS8jOrkSH4WlRRkglKihFfSxIev5zSjVXNABHVEIgJ+VwBAc4JWVVBqDM5dXVIM8hlGCg1YmCwfmUQM6KJUlXnywBlpuoOC9gS+vuHIRTzkEUJ2WQB7KuJ+KGQRzvjclgZj4fzMqZqwk9FwafzEGUFMl1zLCQF2L51DVramtQPE98jqe8F9X30dR/5JeOxvnvX1i5vDIeaAnDYLAjWGf6S68q9N6iyVCOby2BkOHL/+tmju+bOdat9PXcWuWpsHV6PCyLNZDP/q7e7xo538TycNS4Ybe4Nv+dgy4kbxKYKt0KbllSBsBBFEeMvXuDmZAROpxP5fB5dXV1wVNGGxftpRj+Cpi/O/eHjD9P82F5meugWRhOFy23NHEnOCJgYvIvyTBrhcBh+vx92ux0DAwPIJD+hc89ONDc3we2qBctUWpLJpJGdo1ERVUMq9w0vnw/BWZnG7oMHcLKnD9n4JPw+H9rb2/GFT+B83znwqRTcXBCHD3YgkUgc0QPIdFwl2unc5xgdFLtem7llB0rvxtDd3a2XUiwWkc1mceHiJbSGtug2DofDPM9ARLksor7eg3Q6rl+KUw9AoKG3txdWq1U/m52dhYFoSEUeAav3g5ZQ0WXcHO6Nev3ekCKVoKSGsHHtKvgbGmCpMiKTySAWi8FsNiMQCMDr9WK2JCDFfxaePXv6ifz5nIPBIOPz+dppxtu0gR7qxMxJncvlDlNMut3u2xSN9GUqPM9nKaOOH954iqSVHtHeAAAAAElFTkSuQmCC',
television=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJKSURBVDjLdZJNSFRRFIC/82YazQJpkVgbQV9aU23cuQtatIxaREi4C9zmQvrBpEUGLtvnQkqIVi0jgmzjT5PWZJYwIiEG2iInnR/ffeeeFuNMjtaFwz1wz/fde+69Ymb03Z1Ine88uZxMSuP84lo4PtKb5x/j0rX7zafPtee2torlxWymY/rVWCRmBlAneZ/9Hk6M9tVJLl693dx5tiNXKBbLix9nOzJvnkUANQHAjTtPU+n248uYNc7MLIYvnwzkAS5cvtUcnjmVKxZK5a+fZzvm3z6PqkydAODKzceprs4TOfXx4Q/Tc2EUFelMd+UK26XSty+Z8NO7F9HeejEzBgcHHwD3qjIzo6WlJQGgqnjvWV9fVzPDzFBVCoXCw/Hx8eHkLjAUXn8k+y/NDNTAe8OXNLH221FSMODXWO8QMBwANDU1ScsRIZCDcKzGj7xjNe+IvZAQCADnnEAlx8xoTELrUSEZ/IXLkbK6GbEVeRIiJIIKEIigqtQEzjmcVsBjjYJIBf65HWOeXVgIEAIRAqMmSAJEUUTkgd2dU2LkywoIIkYAeKOSG3jZJ1BVnFaK1Hu2nKcpFeDUCAJQBcQQE6qPXieI45gdNcxDKTbUV/o8lDBiJ3VwNbz39S0UdgznoeSMWEHNUBNMKmf2tgfG6gUNDQ1svh5lZWWFkaUlBtracM6RTqdZmJuju7ubqakpenp6mJycJAzDWgtiZvT391trayuq+t/w3tdm7z3ZbJZMJiNJgI2NDRYWFmqL3nvM7EBe/crVvwPwB5ahnKoTKjg4AAAAAElFTkSuQmCC',
television_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK7SURBVDjLbZFNaJxlEICfeb+vu5tYCApd3CgsdLdpu9pWEKEUEaEUb4o5iBQpePCkHqwQqlKLB6sUDyLoQbBC0IoI4g8iKv5SSlpj2qYmWtgapCaSiGY3m9399nt33vGwydaoAwNzmOcZZkbMjENPn8rsGhm+EseSOz87Xx4/frDO/8Q9Dzw7tOPWrdVGo5XMTk+WJj47mYqZAWyQfD/9a/nUiUMbJPtHjwyN3FKqNlutZPbCudLkl2+nAH0BwENPvZWpbN1yBbPc2bOz5Q/fOFwHuPu+J4bKO7dVW8128tOlc6Xz37ybrjMbBAD3P/JKZvtIoaqhO/DDxFQ5TVuMVLZXm6vt9s8zk+WL372X/rNfzIyxsbHngGfWZWZGPp+PAFSVEAKLi4tqZpgZqkqz2Xx+fHz8WLwGHC0/+IL8+2hmoAYhGKGt0fyKp61gwPLJg0eBYw5gcHBQ8tcJTv4Ld9VYqHuu1j3dIEQCDvDeC/RqzIxcDDduFmJ3DU5S5WotpZEGIhEi1wOcCKpKX+C9x2sPvD4niPTgP1a7WGANFhyCE8EZfUEMkKYpaQDWJmfEqCcKCCKGA6z2CdHKpyTJMu0kxRWa1wSqilcIBhoCDR8YzDi8Gs5BZ+l9tgx8xe133cHNN2zj65kPmBj4jb2PDb/kALrdLh010q6xmhoaentuisBMWFl4h9t27EGdsqdwABXP3l37AB516ys0O0a9Y9QTY7UDLW+oCSbwV+N3Nslm7t35OABP7n+dUn43QC4GyGaz1L44wdzcHMcvX+ZwsYj3nkqlwo9TUywX/2Rm4QwXF05z5MCbvPj5w+SiLEAS97/gPYVCgZfzeW5SRVWp1WoUi0VcWGFi+gz7dt/Jx5deIyMxpy98C/CqmBmjo6O2tLRECKGfZrahjooN4uEEiQxTwc9nmfroF/kb8GeNaWNAJ70AAAAASUVORK5CYII=',
television_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK8SURBVDjLbZFPbFRVFIe/897rdDpgKqKTDi6GtmOLg67cuGwixsSNoQujhLDSjYkLumhQg+BCTFgpCxeCJDZKQggxmrjQxoRgjG2thRaskkxptECYEmWG6fzpu+/c42LKaNWb3Jvf4n7fyTlHzIwDb55JPTm0YymKJH1p8WZh4ti+Kv9znnvx7d5dTwyUarVGa3FhdnDq69OxmBnAJsmPC78Vzhw/sEnyzOih3qHdg6V6o9FavDwzOPvtZzFARwCw/41PU8WBR5YwS09PLxa++HisCjDywsHewuOPlRr1ZuuXKzODly6cje8zmwQAe189kRoeypXUJz0/Tc0V4rjBUHG4VF9rNn/9ebYwf/Fc/M//YmaMj4+/A7x1X2ZmZLPZEEBV8d5TLpfVzDAzVJV6vf7uxMTEkWgDOFx46T3599DMQA28N3xTw5v3HE0FA+6e3ncYOBIAZDIZyW4RAvkvnKhxq+pYqToSL4QCAeCcE2hnzIx0BH1bhSj4G27FykolphZ7QhHCoA0EIqgqHYFzDqdtcFtaEGnDd9YSzLMBCwFCIEJgdAQRQBzHxB7YqJwSo9pSQBAxAmD46kn65z8iVS5R680x3Ubbr6riFLyBek/NeTKpAKdGEEBh/hRP/3GeXftfobt/N82Fb9hxcZLJPV2vBwBJkrCuRpwYa7Ghvt1nVwhmQt/MCYZH9pJeuoB88jKZ658zsHM7Jnaw00J93XAems5IFNQMNcHEeODu76T7+uH5sc6WoqM5Qi87I4Du7m4qk8dZXl7m2LVrjOXzOOcoFotcnZtjpGcrjctfseXL11hv3qYB1O6FaMitqLMF58jlcryfzfKoKqpKpVIhn89zo7bGlR++I7+9hyjsonYn4Xo5wPAfipkxOjpqq6ureO8718w25WcfavCUrPAwCX8S8n39QT6Yui1/AWOZi6sZoXAuAAAAAElFTkSuQmCC',
text_align_center=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAB8SURBVCjPY/zPgB8wMVCqgAVElP//x/AHDH+D4S8w/sWwl5GBgfE/MSYwMORk/54C0w2FOcemgmSIMyH1P7LNCHiLBDcEZ/+agqwXaFbOIxLc4P0f1e7fUPiZGDcw/AdD02z9/5r/Vf7L/Zf8L/Kf/z/3f/ZsiAwjxbEJAKUIVgAswNGVAAAAAElFTkSuQmCC',
text_align_justify=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABjSURBVCjPY/zPgB8wMVCqgAVElP//x/AHDH+D4S8w/sWwl5GBgfE/MSYU/IfpheiEwTNEm5D6H9lmBLxFtAmR/3+h6YWY95xoE7z/o+uHwM9Em2D7/yeSzSAICdc/xJhAMLIA+V1VH3Z4v2kAAAAASUVORK5CYII=',
text_align_left=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABjSURBVCjPY/zPgB8wMVCqgAVElP//x/AHDH+D4S8w/sWwl5GBgfE/MSYU/Ifphej8xbCLEaaAOBNS/yPbjIC3iHZD5P9faHqvk+gGbzQTYD76TLQbbP//hOqE6f5AvBsIRhYAysRMHy5Vf6kAAAAASUVORK5CYII=',
text_align_right=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABjSURBVCjPY/zPgB8wMVCqgAVElP//x/AHDH+D4S8w/sWwl5GBgfE/MSZAQNL/31CdMHiGaBNS/yPbjIC3SHSD+3+EXoh5z4k2wfs/qt2/ofAziW7Q+v8brhsSrn+IMYFgZAEAE0hMH/VkcbsAAAAASUVORK5CYII=',
text_allcaps=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACuSURBVCjPvdChDcMwFATQD4rKwgxNwypjAxNTN6hmNakMKkWKFGDqEQo6QEbICH+EjBBeXnoFUQ0MAqOjT7rTEWg/dAhInDi9Eo9TP8dvWP3LsZ31pNa228CSLskM6DMofPwbZFkzqM0yb6ADjeaJmEE+OgnSrBgEEl3Z0JsHQv73Km65GhnNHb6AlmUNgrnBFSBZ1MCbK2wBYmlq4CbLelYGBBJDw2c+DUdevZ8ffsX6A70Y4hwAAAAASUVORK5CYII=',
text_bold=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADCSURBVCjPY/jPgB8yUEtBeUL5+ZL/Be+z61PXJ7yPnB8sgGFCcX3m/6z9IFbE/JD/XucxFOTWp/5PBivwr/f77/gfQ0F6ffz/aKACXwG3+27/LeZjKEioj/wffN+n3vW8y3+z/Vh8EVEf/N8LLGEy3+K/2nl5ATQF/vW+/x3BCrQF1P7r/hcvQFPgVg+0GWq0zH/N/wL1aAps6x3+64M9J12g8p//PZcCigKbBJP1uvvV9sv3S/YL7+ft51SgelzghgBKWvx6E5D1XwAAAABJRU5ErkJggg==',
text_columns=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACISURBVCjPY/zPgB8wMVCqgAVElP//x/AHDKczMjBE/f/F8JvhFxDuBfIY/xNjQsH/P0A9ILgYqMfn/y8GCDxDtAmp/3+Dbf3NsAGoxw5uwi2iTYiE69kP1KMP9cVvhudEm+ANN+EsUI/i/99QP30m2gTb/z/B+n8z3AbqEQWaAAnXP8SYQDCyACM6Tx8bGCIyAAAAAElFTkSuQmCC',
text_dropcaps=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADMSURBVDjLY/z//z8DJYCJgUKAYUBE+440IHYh1gAWLGIzgXgPFINBVFTU/1+/fjH8/v2bAUSD8N69exlBcozIYQCyHUgZAzGIdl1R6bGHVBeEAjW5Qr1QDnOFj4/Pf5jNMHzmzBlUFwA1hQIpkMZ7QKxErCtYoJqVoDaGATXcg/JBBnQAsYmdnR2GC27duoUZBuQAeBhERkZi2IKOYbEAop8/f05lF3h7e/8nZDsy/vz5M5VdYGtr+//nz59Y/QvDf/78QcbUcQHFuREAOJ3Rs6CmnfsAAAAASUVORK5CYII=',
text_heading_1=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACmSURBVCjPvdCxDYQwDAVQSxQ0NFRUl4ouHVV0TapIiIYCKRnEE2QCNrgJ/iIs4jW4KOR0qSiRK+s/2U7opPuiR4CPHh5bOGkJDhYmnqTnca8meNlwtSmWFL9HKKnAJmsBBlMOFA81WGU5HFs2PB06BwP3NVjElQkaYw567mrgxBbw291xWwMrpgCFa3fLzR/YmE6DTs9UYUCPLrah+RBop9dTX31fX9NT9CS3ZDF4AAAAAElFTkSuQmCC',
text_heading_2=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADCSURBVCjPvdCxDYMwEAXQkyhoEBGKIipcRHTuXFk0riwhGgoke4JMwARMkA2Y4G+QCdggE9waxAKHkhJd+Z++z0crnQ9dAtzk4DD4lTpvYaAnJeVcQ7RHg+MBuzdQrCq51JP4PLioIhi4j0DjydLXISibG2dNBD13ix3NqEe1SN5pgeyb5hF0bGODRL2B4p0hlccOlk0EYTXe4tdKSU7/HQzrCATuXDShHAlooXYDZtJQkOGbwpcIb89bDJqvO/X5/ABgCuuOdgJr8AAAAABJRU5ErkJggg==',
text_heading_3=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADESURBVCjPvdCxqcMwFIXhw/MDF3HgqRIErEqdOlcijSqDcZPCYE+QCTSBJsgGmeBu4AmyyFnjpnjCZcrwtx+nOFB8Dl8Ba1lllWVTzNsoSWIZQnh6cdOxsHKRf58kUhF2X9xueYCFtwqiDFT4XmHvZj/AjfNrzCnHPLwCFa63cmaXDzBzrAtBPBUK03d7y2aqYGSqwMuFNpi7ou1/iVxBYqzAyR9NPrG9NuGHuCqgSCXKIGFTuM2Kke7RluaJB6bvXf25N1fx7E1Sq2rLAAAAAElFTkSuQmCC',
text_heading_4=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAC3SURBVCjPvdCxDYMwEAVQSxQ0CImKKldE19FRITeu3LihiGQGYYKbIBtkgtuACdiACW4NcgEnpKJE11j6T98+m9Wcj7kERIqsM6ymHwJ7dvQJmhvSryFK5N1rLFtc4gT8Bx4JOO42gC+Y6wM8pJ/D6Ec3dnOrAJ9ga64O0EtIDS3fBS0sGi/FklMCQXwCjQIoa1vZYsqnrEnAi0sAGWQ/5Zx9r/CkT+NW18QBWMu39TIydN1Xn88bUK7xEQPM95QAAAAASUVORK5CYII=',
text_heading_5=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADCSURBVCjPvdCxCcMwEAXQAxcuIoIFAUMgqtypcyXSqBIYNy4M0gSZQBNoAm2QCW6DTOBFbg1HTo6QyqW5QsU9vj4HK+wPHAJ88uhxDiuMwaFFk/qksUOF7cAJnmb8+rKmFXiN8sxgpomBwb6A7qUe7e2vw0Tj4qKNJvaLLkDRhRoS+QdGcpxQwml7pRaxpiowcGQZdHilVssoyu9VhsjAkmGgsCEZT1Rv/RHuH2BTqYa6xKlQmqPIda6ekGA47tT78wZ72Oy4vOPLEgAAAABJRU5ErkJggg==',
text_heading_6=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADISURBVCjPvdCxicQwEAXQzzlwcMZg9jhFKy5QpsyRcKLIYJxsYLAruApUgSu4DlTBdLAVbCO/jblgxcImDs1P3/wZBorj4BSw7qussmyKeRslStgVCjfYP3MtDSsXefoogQrfOrHZtK8VC28FBOmpcHcrbzfcOD/GFFNI/eOHbjI0ubs3uW4LmDmWCS/ftMlQ0VxrVr8FjIwFOLnQpI4KRUWkAiJDAVa+2PlP1kPlP4hBAUXcg/TiN4XdjHTS7PVUZWRM5736OP+SP+v5etuPyQAAAABJRU5ErkJggg==',
text_horizontalrule=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADPSURBVDjL3VIxCsJAEJyTaCmIYCFWfsMuviNPsAzkB/7E3gdob2+V1kKwsUmOmLu9c+8wASVocalcWGY5dmdmlxPWWoTEAIERNUWWZa0VIkJd16iqClJKn2VZ+iyKwmOe58L1itAVWgdpmtpfqs37q6dnB0mS2C+qxKonxtUnQT8OtvvbYTEdxl0NRBbaGChGpcBIeGjgfJHH3Wa59gRueDZut4ExFmTcIKcWjG4Q0JHhOvKk88kofrvB9a46lRRPKybQ2nii3m8Q/JX/gOAJ1Om5dnjvy8QAAAAASUVORK5CYII=',
text_indent=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADzSURBVDjLY/z//z8DJYCJgUJAsQEsMEZ5efn/f//+Mfz58weOf//+Dce/fv2C0yC8d+9eRpA+RkrDAO6Czi3vrpT7COnA+LGxsdeRbUTHZ86c0UQx4MfvvwyZi55cvXX7a8jeZvXrQEWKuDSDMAyAvdCy+cV/EW42hk/ffzOcvvP1zMNbX+JOTdW7TowX4GGQs/jFlVfvvzPdvfop+OxM/euenp5XYLb9/PkTbjPMWw8fPtRB8cK3n/8YVuUpasG99OOHCrqzkWMDwwUUx4K3t/d/fIGGnCZA+PPnz1ROB7a2tv+xBRayrR8+fGDEGQsDlpkACSYJhTJIjokAAAAASUVORK5CYII=',
text_indent_remove=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADxSURBVDjLY/z//z8DJYCJgUJAsQEsMEZ5efn/f//+Mfz58weOf//+Dce/fv2C0yC8d+9eRpA+RkrDgAWZ07rx3ZVqfyEdEDs2NvY6so3o+MyZM5pwAwL67msqSLCv4WFjgTsHqEgRl2YQhgFG3867mpJirIs0JdlNmBiZGR6++c7QGyXDSKwXwGHgWHldU1KOYy03B8e/2YmSYC94enpegdn28+dPuM0wbz18+FAH7oX97ZrXgZRW9MxnV2Am//jxQwXd2cixgeICqsSCt7f3f3yBhpwmQPjz589UTge2trb/sQUWsq0fPnxgxBoLA5qZANTo8jopO/z6AAAAAElFTkSuQmCC',
text_italic=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAABxSURBVCjPY/jPgB8yUFtBdkPqh4T/kR+CD+A0Ie5B5P/ABJwmxBiE//f/gMeKkAlB/90W4FHg88Dzv20ATgVeBq7/bT7g8YXjBJf/RgvwKLB4YPFfKwCnAjMH0/8a/3EGlEmD7gG1A/IHJDfQOC4wIQALYP87Y6unEgAAAABJRU5ErkJggg==',
text_kerning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGBSURBVDjLY/j//z8DJZhhcBoQMuUBJxCzkWxAeXl5PYiOm/toTtDEB0U5OTkCqamp/XFxcfvDwsISYOpsbGwUjI2N12tpaQUgazYoKSn5D8QJgRPuqdevf/nKp+MOf3R0tEJISMh/Ly+vBGTL1NXVC1BcADQgobi4eH5ubu59ED9o4r05Hs23WkBsd3f3+XZ2dudhalVUVBxkZGQUUAwoLCx0ADpZAejk/0AnB7g13JTOX/z4lW3pVWmgkw0sLCz+A53sAFIrKSkZgBIG+fn5AZmZmQIgNtDJ+4ODg/eD2Pbl19odK65Ogzp5v7y8/HxxcXEFAQEBBxQD0tPTAxISEhwiIiIcAgICEry9vf8DnexgnnNJKHbGvVcGyedVgE5OUFZW/s/HxzcfJRaATnYAOtkBWRDo5PvAUAa7Qj/xfIV1waUVIDY3N/d9NjY2lMAEObkA6GS4nywtLRUMDAzqgU6uBzrZQTv2DKdX442r6uGnhVhYWAoYGBgEhkBSpqsBAOTifxrCztZUAAAAAElFTkSuQmCC',
text_letter_omega=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGvSURBVDjLxZPbSgJRGIV9BB+h+yikuomIJigo6UBnKtJOUFSkSBIhMUZmBywUNDtQG6KMCrITXVnzCANSYUNNOoMzzEjuR/jbXWjQjN0UdLFuNnt9e/9r8RsAwPAbGf4c0BsSi8b2JOS5UN9cpwo7d6Kw82fqW19IRK0rqaIfAb1B0eS7zeC1mwzu9AtU7pwYKfe5iukzBZsXeJMuoCcoGH3EGI5loXPjy5yTeZGnCBhmj2Vc53oxagBdfsG+y2BwRhS20LzD2yK7eq0C5eTsGsD0gczs3GeBfJcuBKid5WjvpQrta0lGA5hAEhO+y0KThy8IqHZw9GJUJY/oALr8KRSOvUN3QIgWApjdr1FPVPkcAWkAjW6eWr7KwExExj9kgB2HEpSNPlK6NTYv8WjpQoGaGW7wu7li7GnQeSRDtf0Z6dbYHUgxxGhqcPNofD+NK6cS+arKR5+M/SEBV9kSqNT6YKp3cdoMnBEZquzPdOV0gupYT7JtvmS+zhYvz5Jw2RJLnCoeiNPWTRE0AMeRBLYDCaZQGiaJxvfS+Usj2yIMEVm3RLAQ84Ae4N+28QM8btMbbDzl6wAAAABJRU5ErkJggg==',
text_letterspacing=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGJSURBVDjLY/j//z8DJZiB6gaUl5fXw9g5OTkOqamp++Pi4uaHhYUpwMRtbGwKjI2N52tpaQmgazYoKir6n5+fnwATi46Onh8cHHwfWZ2BgUGAurq6AoYLiouLE3Jzc+enp6fDNQQEBCh4enr+t7OzC4CJqaioJGD1AtTJCkAn/wc6OQDJyeuBTt4PYisqKipISkoaYBiQmZkZkJycLABiA52838vLaz+Skx20tbX/y8vLK4iLiydgDcSEhISAiIgIB6CTHdzd3ROcnZ3/m5mZOcDkZWRk7ouIiOzn4+PDNADoZAegkx2QJYBOvg8MZbgrhISEEri5ue9jRCMwlAWATi4AOjkAyckKwFCuBzq5HuhksMFcXFwCLCwsBTjTQciUB0KV655fIpRwoqc9uGqec0kTxQCgZt2uHa++zDr49v+E3a//9+549b9j26v/zZue/89b/PhZ2rxHz/IXPf6fMe/h/6Ilj/+71984rR5+WhPFBYET7gnlL3tK0AWeDTeuwTTTJi/Q3QAAViCHY9kyiUsAAAAASUVORK5CYII=',
text_linespacing=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAD9SURBVDjLY/j//z8DJRiFEzLlgS7ZBgA1C3XtePUFRJNlQOW655dmHXz7H0SD+OXl5f9LSkr+FxQU/M/Ozv6fmpr6PyEh4X9kZOT/4ODgbKxemLD79X+KwqBvxyvKDGjb8hJuQHp6+v/ExMT/0dHR/0NCQv77+vr+d3Nz+4/XgNo1zyhzQfLch88oMiBq6j24Af7+/v89PDz+Ozo6/reysvpvZGT0X1tb+7+Kisp/OTm5lVgNSJr5gDIvRE+9R74Bng03rmXNf/jfo+HGVRDfxMTkv66u7n81NbX/8vLy/yUlJf8LCwvjjgX18NOartXXToNosr1AqmYMA8jBAE8T0gwxA4F9AAAAAElFTkSuQmCC',
text_list_bullets=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADqSURBVDjLY/j//z8DJZiBKgbkzH9cMHXX6wcgmiwDQJq3nv/4H0SD+OXl5dlA/L+kpOR/QUHB/+zs7P+pqan/ExIS/kdGRv4PDg7+T10XDHwgpsx8VNC56eWDkJ675Hmhbf3zB0uPvP1fuvQpOBDj4uKyIyIi/gcGBv738vL67+zs/N/Gxua/iYnJf11d3f9qamqogRjQcaugZPHjB66V14ZqINrmXyqIn3bvgXXeJfK8ANLcv+3lfxAN4hsZGWVra2v/V1FR+S8nJ/dfXFz8v5CQ0H8eHp7/7Ozs/5mZmVEDEWQzRS6gBAMAYBDQP57x26IAAAAASUVORK5CYII=',
text_list_numbers=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAD3SURBVDjLY/j//z8DJRhM5Mx/rLLo8Lv/ZBsA0kyRATBDYOzy8vJsIP5fUlLyv6Cg4H92dvb/1NTU/wkJCf8jIyP/BwcH/8fqgkUHSXcFA1UCce7+t/9n7Xn9P2LiPRWyXRDae0+ld8tL8rwQ1HVHpXPTc7jmuLi47IiIiP+BgYH/vby8/js7O/+3sbH5b2Ji8l9XV/e/mpoaaiC2rX/+v3HN0/81q54OUCCWL3v8v3Tp4//Fix+T7wKQZuu8S+THAkgzzAVGRkbZ2tra/1VUVP7Lycn9FxcX/y8kJPSfh4fnPzs7+39mZmbUQARpBGG7oisddA9EAPd/1bRtLxctAAAAAElFTkSuQmCC',
text_lowercase=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJXSURBVDjLlZDhS1NhFIeF/oJBEK5ISiIJsaxkTAQTR8wSacY0xdlmbZqkTZ3cmmKprU3naltOd2u5UtFSAiuSaAUVGEKDvphtXuVu79xttfB7Ef16FwRhG+mH8+H9Hc7zPuekAUj7u9omQ2ieIGj0hqB1B76s76+vf4KmCeFh3wzBxekYVGykYdMALbsEtZsfrR+NQ+mK5m8KUOchUNk/vqlk41srB6MosxLDhgHqkdhAw/AilKb3/YrO+cKqQQK5OTS2IYDKE9uvcQZQ3u0vSrz1r7T3au/3obh3Zf6/gGp3dEjpJFCYPuC4Tdimf33Wa39ngfVtLxS3ulHuKdkid1RFi52EOWInvgIb8eVbyZTUHNYlPWLj89M2y9wVzC7PoPNZO446rn8/NrQKzVgMBVaCmhEBFx58RgW7igM9vC6pVvWkot842wL73CyUtwVox2OQ9hFLopd3lbdovAJOuaPY17HCpNytxFH0rfXxC9TejUHS8/JnnokXJfKcLp7VUIsyZxh7GE6XajhaaJJCYszFQeNl5Fxy/aC6vuwuHiecEVx7EkcWs4yMVk6U0kBiDouorq+Cqp50/db172W4qTN3BDTRlXbqg/6kR/xTuT28v4oVoBxeRVYHxySyXQaOKR0Io8QaxvbzASYlgOrKZDYCy9OvyDYuI9PAiajubvrrSsu4gMM0E9cHZCkBVFdXeiOM3kdxlDkiyGxbQpZhCXJLCPrRT6i5GcGhdg7iukVZUgC9rojqshn6IHY0BxO6fvG5AEN/ZcTaxTU6uJZet8CmqxdEvwDWpa/ASC8BSAAAAABJRU5ErkJggg==',
text_padding_bottom=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAB/SURBVCjPpVFBDkBADJyKJ3mEk1d4goPPeYDgN2QtidFdFidZ0UnbZDszbbJCvEeCv4TUlVr3EKvCKmYYhau9AMIYh4oLFq8N6lYCIc6h5PzYbLyTVc8p+oaCQWu81mL8eEPzYNEnsWnP5SQA2fnsBkcSw+1AdJfqGN4hv39zB9EXSdykB4lxAAAAAElFTkSuQmCC',
text_padding_left=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAChSURBVCjPpZExCgIxEEVfZI/jGWy8gFewX6w9h7C1V1iwXRsvYCN4DUEEJ3HzLbIRF4zNZsq8/+bDOPH/zZgKVABHASzdVj0vAp6A4bH60CBEp5s6iV9TZZWAjULO0rqvFekbdq7QQUQisFZKG08Mw+prMwL2JUOkJwIr2Sd/cSMgGdqyIZVcDIbUJBDqR+6QgFPJAGcA5spZz32A3eRrvgFwMGHf7+AlJwAAAABJRU5ErkJggg==',
text_padding_right=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAChSURBVCjPY/zPgB8wMVCqgAVEFP7/w/CH4TcY/gLh59ul9oLtdmZk+I8D7vn/4f+e//8hJqT/h+kGwqu/GA7oQIz/D7NiJiM22/8j3BD9/xdMPwQ+vyL1n+EfEEIVLGXEph9Jge9/JN1XfzPcAbrhH8NfhILNWEz4h2yCPZIJYP88fyf1D9mRB7G6AUmBAdQXYN1X/zB8AYfDebACRopjEwDsBmruXDxiUwAAAABJRU5ErkJggg==',
text_padding_top=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAB+SURBVCjPpVHbCYAwDLyIIzmEX07hCH64nAOIuo1Sq+CZ1ueXVsyRBJrc5aBCPEeEvwuxK9XtDn0Si/ZU9gUg2Z/dYEuiuxSI5mRtwyuEIR5KOpVZYRUjjMLVVkIVCk6YPPdg1/LNQ87xdtl4JauaQ7CHjAfXeK5FH+7h9bNWB/9J3PASf8kAAAAASUVORK5CYII=',
text_replace=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJFSURBVDjLlZLfS1NhGMcF/4Jz6YqkSTRCLDUZipDWSBdjpDFxIdEWLoJ+TJscmqGxtXZqadtqrZP9QoVqV1k0gtWl3rRLkc2z4zxn22lhSLdG9O05w4TC4bp44eWBz/f9vM/zVAGo6ooqjq6IAvNjBZ1BWavWKj1bFwLRMSmL/wNvBRDYbHpQQOttKdHiy8YaPdlEw1jWUnEAgax1qoBNkDsZzqFpXESdS9BWFLAJ4k+RwGSbdwW7L6cNOwYQyBwcW8F+VoiphdphgemdCaJn2rdhn+uv3jGAQMtR/yoI5NSCNeZNBz9zCCx4cfplz6veJ8bq7pC1cCwssx1BOdE+ISfaAnKs1S85SgEE8qQq7rqYEjUXUjBGQ9+5+RuIZ97g+ocRHA9N/jjxMA/bTBHtARkDzxRcef0VfXwehzxZx7Za9PIdd3wIwfk4LFMKBmeLNCG5ZNhyM8vZnivof1TAgVGRLfs3Y6hzY/jtR5x5UYTe8+kXjZdR69Qz3kYW5rCEfazgKAcXjvhaoXc3osk9joZrkZ+km6inSakjvvVuDTo2o/aNKWug90sM6Sb6SPVUpKSbVCd17qmCS/SlPc508q9V/vfQbiStvAJLNA/dqMCqtb0ugTXdlWAMSKCms2UDSNdgmJDBvf+GendG3UiGdLX0qjg0q+Aw1TTnU4ayAaTrMN2T4J1bgzmUQ93VZehcy+jmVuGc/oKB+zk0jwjQ2JcM2wZQdxnS5WudaXWdVd0k7QhLr7KawaV1Atdr7It8zdlF5jcS1qLOWBIaUgAAAABJRU5ErkJggg==',
text_signature=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGeSURBVDjLvZPPK4NxHMe/F4dFymE8hx2YpEn+gWcHR20au4gdbCcnu3l6TqL8yO9oYSJmYbNSFsraZCSlbHJANEabaZtFYUzW29cOojzyK4f37ft6vz71/XwIAPKbkD8r4Hkeb6PX66HVaqFSqSCXyyGTycAwDLKyJchk8vGjCTTjATHNPk2MpujDCT4zl7etot8ZwYDjAuqBg6ii01v0LXMfBV27McxthMGbDsByTvvrI2rmqJmjZo6aOWrmqJnLkZY0V/Rsh7uXgim4w+bH+PI5ajs3UayzTgtaEy7C+BfztHZLzeWw3f0O1rSsoaB6wpVbaRQJwcoHN4N7jxoJXxdi8+mw2EZTcFXTCijokigNIsFfiC8T9QucvFlHMjKCRNCK0Ewa+NZuUPAVFiy4XSKauEeDp8ggnk50SEbNuNqqR2OD4ugtLFhwbSd1t9463JyZ8BgYwZ1vAmFHGbyGDNWXVjk2R4YOF0oRXShEYJLsnI4R87GR9O71EbFgAcuyszRrcpZdt7VLE6EpMhYwkcJvH1PYSsQUFv/bNT4D22mzuYC6IhwAAAAASUVORK5CYII=',
text_smallcaps=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACWSURBVCjPY/jPgB8y0ElB+YHyA8UTcg+kLYjfEP4Bm4ILxQa5Dqn/4xv+M/hdcHXAUFAc8J8hzSH+fzhQgauCjQJWN8Q7RPz3AyqwmWC0QfO/wgKJBWgKwh0C/rsCFRgBTVP4/59BMABNgZ+Dx3+bBghb4j8WK1wdHP4bQRUIYlNgs8DogOYGBaAPBB24DrA40Duo8UEA+kT4W+XS/8wAAAAASUVORK5CYII=',
text_strikethrough=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACfSURBVCjPY/jPgB8yUFNBiWDBzOy01PKEmZG7sSrIe5dVDqIjygP/Y1GQm5b2P7kDwvbAZkK6S8L/6P8hM32N/zPYu2C1InJ36P/A/x7/bc+YoSooLy3/D4Px/23+SyC5G8kEf0EIbZSmfdfov9wZDCvc0uzLYWyZ/2J3MRTYppn/14eaIvKOvxxDgUma7ju1M/LlkmnC5bwdNIoL7BAAWzr8P9A5d4gAAAAASUVORK5CYII=',
text_subscript=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAE4SURBVDjLY/j//z8DJZhh8BhQXl5+oLi4+EBubu6BtLS0A/Hx8Qrh4eEH/Pz8Dri6uh4gaABQcwBQ84eUlJT/QM0TQGJAzQ1AzQtsbGwUiPIC0GYHoOb/kZGR/4GaC/DZjDMMgM6eEBgY+N/Nze0/0GYBkg0A2iwA0uzi4vLfyMhoAskGgJwNtLnA2tr6v4GBwX8FBQUHkHjIlAcKpaueX2jZ/PKDb9fdBgwDQDZDA6wAxNfU1JwAdMF/CQmJD4KCggbJ8x5vAGpU8Gq71dCw/vl/DAOgNh8AORuo2QBo8wGg5gNAzQe4uLgOsLCwGIDUJc56eCFl3qMHZCUk+4prDWGT7l0wz7lkQLIB1kVXApyqry0wybggYJh8wUEv/qwCSQZ4t948kD734f/kWQ/+h028+2HwZCYAjxChYziQ1VwAAAAASUVORK5CYII=',
text_superscript=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAE3SURBVDjLY/j//z8DJZhh6BgQMuWBQumq5xdaNr/84Nt1t4FkA5LnPd4A1Kjg1XaroWH98/9keyFx1sMLKfMePcAwoLy8/EBxcfGB3NzcA2lpaQfi4+MVwsPDD/j5+R1wdXU9AFJjX3GtIWzSvQvmOZcMMAwAag4Aav6QkpLyH6h5AkgMqLkBqHmBjY2NgnXRlQCn6msLTDIuCBgmX3DQiz+rgOEFoM0OQM3/IyMj/wM1F8BsBmHv1psH0uc+/J8868H/sIl3P+AMA6CzJwQGBv53c3P7D7RZgORoBNosANLs4uLy38jIaALJBoCcDbS5wNra+r+BgcF/BQUFB6IMANkMDbACEF9TU3MC0AX/JSQkPggKChoQNABq8wGQs4GaDYA2HwBqPgDUfICLi+sACwuLweDMTAA2jKFj5WHetwAAAABJRU5ErkJggg==',
text_underline=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACjSURBVCjPY/jPgB8yEKmgPKH8ffn/0n4IL3F99P+QAjQTyveX/IexIwWCz2NYUbw/7z/CYK/9GApy92cgKXDEVJC+PxFJgQWmgoT9kUgK9DEVROwPRFKghqnAv9/7v2MAhK3iINePocBNwf69xXlDhf8Myg4y58UUsISkmYL+fI39ivul+0UMSA/q/wza/1X+y/0X/y/0n+c/+3/m/6SbgAsCAM8i/W7eee6fAAAAAElFTkSuQmCC',
text_uppercase=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ9SURBVDjLjZHdS1NhHMcF/4G8FgoWEQgVIhFKkNZC7CLadL0osxwy1CbNt522JAzfljrb9Ax1LtfUTTcSCkTCJV3VRe2it4vV2Vp7zs45bpAXkZAv9O05A6E3bRe/qx/fz+f7e54cADnlY6LmLJuMKUd5qF0izgwTFPbErfLuf5NT7hQV58dFlA6TWMkgyTveG7de84gouBULZQUotRPFSRsJFfcnGGpljt6Oh6+MJ3GwncuuAbVqqDVWNpSQrWHDjISSO5+Qb4goswKcHk6AWtcOm7kiatVogyNQeXs3dI8v52YF0DiTOMRwaweMHFM5O/TV/sqKwRfdUHvqnqndFbkVjrLEzYfiABMUHKaA9L49IJIWv0CM04TNAKg1tt/4AfnXI2H1g56X1uddWIo+QueTDtDw+qne4i1TQJDa54VvI6E0RpdTuLsowTD1eTUD+LNS9ZxqwLLUgoWIH753UzhhKdxuC6YVRr+weWOa33Y+XUNHII0Gr6T8J0Aeat6YfjuJ+6+dOGYs+G70S1yzl2w0echy/+IXtPpTqHfvAqBhgdaWzXIY5/r6tjqDSTS642+o1WwOpmGaT6HOJTK7NtiZZl+qqHVWQMNkfFPvFiqplWmj9lbfKrROYW9A00wqr9Er8l1BHvXO6IKO5TLfqh0TYKKQ6lF+Yk8ArWs1+xLQsdH1Oge3Umv/uFJji6wYXHGY51K4aOdDuwL0U5K+w0erj8dAg0d+3V2y87BQQJWNhP8C6D1SXuZOeuNVB/dDa48GalhhXybo4BXUyjS4BHQvpFHLJnFhgCh/A+gmRT19XdTSO2tYIt8JGiySd1X3+IlKG4FqiMjBnZn4CaGVtXQxzDRMAAAAAElFTkSuQmCC',
textfield=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAArSURBVCjPY/jPgB8y0EHByf/4IVDBHzzwBUTByUGgAK8jX/zHDxkGQVwAACuKdiqzzuaTAAAAAElFTkSuQmCC',
textfield_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADTSURBVDjLY/j//z8DJZhhGBhw8uTJ/5RgsAF//vwhC7948QJhADkGDTEDtp9c+790ZsL/tD7//4ldXv+X7pmBagA+vOnosv+NqxP/b7ky9f+FZ7v+9+/O+h/er/u/fXHZfwaQKYRwYpfn/42XJ/zfeG3SfxDo2ZP6v39P+n/bfHniEotPteH/bVfm/EcGmy5N/W+eLUmcAZY50t+7dyX9b9+VANbcvjMB7AKgAd+JMgCosCW4R+N/764UsM0gGsQHivcQneaBijuA+BPI2VC6AyQOAPdpPzVHO/APAAAAAElFTkSuQmCC',
textfield_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADhSURBVDjLY/j//z8DJZhhGBhw8uTJ/5RgsAF//vwhC7948QJhADkGDTEDHq2f8v94ktb/Pc6s/w9Gyf+/s6wd1QBCmi+V2f7/vrX3/79rO/5/XVH0/0yu/v8rC9v/M4BMIYT3Ryn9/wbU/H+S7///5YL//7cp/n/d5fB/f6QicYlljxPLv39n1/1HBu/rJf6DxIkyYKcr8+Mvc5P//wdq+lHG8P8dED/MYP4PFH9ClAGHw6UaTqao/n5Wrvj/VSXr/7spjP/3+rL82eHKXEV0mj8SLlsBtPE+yNkgF4E0g8QBOvk+kKwjj48AAAAASUVORK5CYII=',
textfield_key=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFZSURBVDjL3ZM9S0JhGIbPH/APBEFQ7Q32gS0tlQ21iDVYJ/qAooaGliYRtKBBimrIwKSkooTGaiitFnvJjzLLPGXZCfVVRMWPo6ZwNzaZmlvDtT1cw831MACYWmD+gYAQglpgCCEoFot/glL6IygrKnwhYl1B6HwWcbepekHUbkCUzKGQIqCWCcQ5S+WC2MMRIlcsCgkjEm4VolYl3vaH4LkwViYImReQjx0iF1QiR7eQuFdC4FXg9H2VjRi5WUfatwzBP4MsPw/hfRQCJ4dD0wWGUopSBDzX4E+nETBPIullIbwMIOtTIPMsh0Mtxt3JbulAuM2WYd+2xCHwOmReZRC8PbCttqed6lbq0rTRA7ZB9muJHyapMxfcQdLVAeGpF/aNzoxO0Th+JqsTXbL1orIpfx5LST68hIy3H7Y1SUo/1jRS1S8QrXjx0dAdvtVK/HtTzYOl7r4BQSM7uFIKnq0AAAAASUVORK5CYII=',
textfield_rename=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAACjSURBVDjL7ZNBCsIwEEVz1mwTo1YjiHdIqyWgFBGPonWTC8T2BjlE4JsUwU0ILe7ExUtgPvNmNkMAkG8gPyAwxiAHYwxKKUgpk/kg8N5n4Zwn6865j4CVLXj1AA//rArsW4hAzCil4wTFsUdx6rBuLLaXJ+aH+zTBqukDFpuzxe5qsagnCIbV32vHybF5Wd/GC3JkBfHJEZu11hBCJHPyvwXyAt6tONifnq6xAAAAAElFTkSuQmCC',
thumb_down=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHrSURBVDjLjZOxaxNhGIcTtRkUJ/8AB9FBHBT8U3ToLAouxkUQLEIVnRxcKiithIjBBsSgRLgMGlNjvSOh4XJ4XLhwIRwpd+a4L1xMjEMfhy+aS6W2w2/4Pngefry8bwJI7EoSOAykaHdOTt+JvTL/UY+SNAYXML1POO4OnS5YLTCtMYb5FcO8j26cR7MX0OyFeYE2OkQjuESrBWEI4wmMxjAcgRiAH4Bu7GBaUDcFm5YzL9gcnaHdAUUBVQXPk4JoCCKSklDI+AG8Lv2YF5QbJepbEgiFBIYjGMTgQEAQQj+A/BtmAk2k0JoTKhtQrYJtQxTJhH/gEPpT2O9DLh8TbHQu0zRkZSHAdiQsdsF+AF4fPB9e5GKCiv6ZwluoVOTUe9sSjlf2+xJ2t8GyYTUTE+i2J4EQnA7UahKIV/Z8KS8W4eG6zlJGnQm+OB+wTDl5MeCvLF65aUC2AFfyadL5s9wpnJ4JitYpsrW7vKyqFNTvKLUh7rRy14V3H2EpMyG9tsj1anKvTUwCR2gExylZy1jfwO1BuQy3159xtXh0/1WGBO+7F6lqv3B70NDhwast0qVzB7sFxTmGYj3HNOWkH61G3MovHvyYFP0EiuZgt+Hx05/cyC7/D/5XkLNSrKg3ufcErq2t7AcDid88lUyCVhHVfwAAAABJRU5ErkJggg==',
thumb_up=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH9SURBVDjLlZNBaxNREMcTtTkonvwAHkQP4kHBj2LBngPiqRUPgpdiDYKlLYVKBRUU2psXQwNBCrVtaowbEjasocuGDRt2l112fUs2pFLroT8Pb22MNdAe5vDezP83M2/mpYDUkalxBjV6gG6B5i0P+UbY8IXmXaJpW8Q90M2fqM7M6QCquIAWvMX3Ie6BZvapuhMnB0AKJbrNbusXURdCAYqpsunfOAkgDZyjs3+RmjOD68gqbBvK1ms2vPOjAWpwhbo/zTdPYdf5jmbtIXrQjaUZFpT1A7b0CT546eOAuvMJz4E4hv4e9PpSGMUQdUFEYDug6pA3pijo18k3rw4AmhkQ92Sw1YFaTfYvEnEoIAglpNGAYl2jUFUGgM3GZ/JrUCqB0QLXk7AwgiAR+wF4vvSZbXi3ygCwYY5Tb8jSo64M6MYS4IfgBeAmYtuVlSy9/AuwLjLsKAdslaBchlYr6V0kWX1wEnHHAcuGuSWGx1isrlOucDT/UMj+PR+cJGvHlm/UtuD5wj+A9941KgoUP0KlIkUiktn/iNsdaLWhqcPj+R/DgBX3DCuNOxQKYBhSHAgJMkz4osDs4iG5WcjmYu7mrOOr/MpIM1+/xdzaNm9WD3mxDNNP4OGjfe5PfeXeZI7s5E3Gn46RXRj7/1+QK30WyPBs8XJyHvmZfgPxTEl50XYktwAAAABJRU5ErkJggg==',
tick=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGrSURBVDjLvZPZLkNhFIV75zjvYm7VGFNCqoZUJ+roKUUpjRuqp61Wq0NKDMelGGqOxBSUIBKXWtWGZxAvobr8lWjChRgSF//dv9be+9trCwAI/vIE/26gXmviW5bqnb8yUK028qZjPfoPWEj4Ku5HBspgAz941IXZeze8N1bottSo8BTZviVWrEh546EO03EXpuJOdG63otJbjBKHkEp/Ml6yNYYzpuezWL4s5VMtT8acCMQcb5XL3eJE8VgBlR7BeMGW9Z4yT9y1CeyucuhdTGDxfftaBO7G4L+zg91UocxVmCiy51NpiP3n2treUPujL8xhOjYOzZYsQWANyRYlU4Y9Br6oHd5bDh0bCpSOixJiWx71YY09J5pM/WEbzFcDmHvwwBu2wnikg+lEj4mwBe5bC5h1OUqcwpdC60dxegRmR06TyjCF9G9z+qM2uCJmuMJmaNZaUrCSIi6X+jJIBBYtW5Cge7cd7sgoHDfDaAvKQGAlRZYc6ltJlMxX03UzlaRlBdQrzSCwksLRbOpHUSb7pcsnxCCwngvM2Rm/ugUCi84fycr4l2t8Bb6iqTxSCgNIAAAAAElFTkSuQmCC',
time=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKrSURBVDjLpdPbT9IBAMXx/qR6qNbWUy89WS5rmVtutbZalwcNgyRLLMyuoomaZpRQCt5yNRELL0TkBSXUTBT5hZSXQPwBAvor/fZGazlb6+G8nIfP0znbgG3/kz+Knsbb+xxNV63DLxVLHzqV0vCrfMluzFmw1OW8ePEwf8+WgM1UXDnapVgLePr5Nj9DJBJGFEN8+TzKqL2RzkenV4yl5ws2BXob1WVeZxXhoB+PP0xzt0Bly0fKTePozV5GphYQPA46as+gU5/K+w2w6Ev2Ol/KpNCigM01R2uPgDcQIRSJEYys4JmNoO/y0tbnY9JlxnA9M15bfHZHCnjzVN4x7TLz6fMSJqsPgLAoMvV1niSQBGIbUP3Ki93t57XhItVXjulTQHf9hfk5/xgGyzQTgQjx7xvE4nG0j3UsiiLR1VVaLN3YpkTuNLgZGzRSq8wQUoD16flkOPSF28/cLCYkwqvrrAGXC1UYWtuRX1PR5RhgTJTI1Q4wKwzwWHk4kQI6a04nQ99mUOlczMYkFhPrBMQoN+7eQ35Nhc01SvA7OEMSFzTv8c/0UXc54xfQcj/bNzNmRmNy0zctMpeEQFSio/cdvqUICz9AiEPb+DLK2gE+2MrR5qXPpoAn6mxdr1GBwz1FiclDcAPCEkTXIboByz8guA75eg8WxxDtFZloZIdNKaDu5rnt9UVHE5POep6Zh7llmsQlLBNLSMTiEm5hGXXDJ6qb3zJiLaIiJy1Zpjy587ch1ahOKJ6XHGGiv5KeQSfFun4ulb/josZOYY0di/0tw9YCquX7KZVnFW46Ze2V4wU1ivRYe1UWI1Y1vgkDvo9PGLIoabp7kIrctJXSS8eKtjyTtuDErrK8jIYHuQf8VbK0RJUsLfEg94BfIztkLMvP3v3XN/5rfgIYvAvmgKE6GAAAAABJRU5ErkJggg==',
time_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALNSURBVDjLpZLZT1NBFMb5N3zx2RcTedGYGB6MLyZuSTWoiVgMimFL2KRQUKsClR1MKbKVpUhbBKmQlhbK0pQKpFDaAtLShQKytdTblrJchM97G3PVYIyJJ/kyZ86Z+c2ZMxMBIOJ/dCwwJCm6qO94qpns4WxN93HJyd4cUidNXVc1pHZ3v805/1eAVlZQbVZz9petY9hcc8Dv94EgvFhZNMOsk6CvPn5bWp6U90fAsCS/0m4Qwudxw+r24f2gC9Wds6iSzaBVaceUbR0uqx6KuscQ5cdl/gZQtfIvGHoySO+GC1rjKrqGXLAv++H1B+Hxb8O65Eer2o4PI07MG5UQv7wdqitIiGQAA81ZigUq8XlxCzKNE7T5CAK2L2vYpXxawSOgptcOncmNfnE6anLvtDKAwZbktVW3BWLVAuaok0MHRwiGQhA0irBBgQJ7e+hUDUJrI1DcboJlQoo6LsvFADTNSbs+7wqK2kzY2CHh2zvEPpXIfs6DuEuOrGc8qPXjsBAk0gTjWHKNo5F7c4cB9NXG73o3HeCJjFgKkhTkEMtEAK9KSsObtUYzPAeAwUsiueIT3I4RNGSzfgI6y2KcDosSFTITRhYIrFKXXg6QUAyPwrnlx/o3QKrvQqqQjUdl13G/8BJSsqIIBtCUHyMalnKgN9nAl1nhoRrmI4HAISXK7xiVgNcRi94ZAaZX1SjvT0B0+WmcSzxREgY0FCaeaXkRvTNvaEGbchKvZfMwur4iSPUjGCIRW3QZcksF5HNV4RcqGXiI8oF4GrDLfKRaXiznHf8W5saqMTRhQIFoDE+qRpFeocMVbiQUM/X41T6aBTTg938tyL2XV8u5EZQL72JKkw/nnBjO2SZEpZxEsfoBCtXs8OZCFft4BQwkL/ZsZSar/U3aNbcw4+oOLXZKpJ9Vcgql6rjwyfRIz5ke/IuoxXxKAbrsHyOfjn8HcjT9RjDIr5sAAAAASUVORK5CYII=',
time_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALnSURBVDjLpZNbSNNxFMcN9b3opZce7CHqIagegqAgLB+NyMIMFCRBSX3wPm+Z1ZZZlhpT03ReZs7ZvKDpps7LmKa2uTbTnO7inM3L5vxvc3P+1X37zYeVGBF04Mvvxzmcz+/8Duf4AfD7Hx1yDPIKg0dbHonlnYz1r8JsWt6VRUubk1ZE1Unt7e+yLv8VIOGzylS9jG2jegxry1rYbFZQlAVLCyqopDwI38duNr9JyP0jYIjHLNHIymE1G6A2WPFxQI8ywTRK+d/Q0KPB5NwK9OpRdFfFgcOMSTsAEDUUXJF1ptKWVT0kChNaB/XQGG2w2Bww2zahXrShoVeDtmEdZhU94D6956xiPbzgA/TXZXTPk8D3hXXwxTp4zUpRmPuxjC1y98rhASq6NJAqDejjpqAiJ6LBBxioT1w2GabAFc1jhrzs3PHA4XSCXcPBKgHZ3W4IRAOQzFF42aTE1EQzqrLD9D6AuC5hy2pZQmGjEqsuGlb3HrZJIPNxPritHcjIy0fv6DimKBrJ7HEs6sdRk33H5QMIK2O3LGta5HMUWHTQBLIHI2XHs1dF+8kShQrmHUBmoZFY/BkG7TCqM8N+AQSvo3TaqR4U85UYnqdgIp822ml0D41At27Dyi6gamFjIPIM+oMD0R92HE2RZykfoJYZxRlqZmBUOYcCvhpm0jArDdj3iMh9vq0MqvSr2PpUBM+MEE5eMr7En/P0Xg9I3AdUP48/X/8k3DUrq0djjxwv+LNQ6DfgIP1wOGn0R5yCiySjNBTIOAawgmAuvAbRDX+db5Aq86MZHwruYmasDIMTMrA4Y0gvHUFKsXS/bI+8Fb/bRt4J4g/wHBhldk5kbiXjlqOj/D4mxUzoZrjQTdei7/ZRbFY/AEiSO90PViJDnD9EIf5Lh5aJnRt9qSQtrOlt8k1DeWqoyytBzEVqLOa0x5QRhLXMQGhjjkAcGrArDPHP+ue1lYafZJAX9d6yyWn0Jnv9PwH2GPv45gRecwAAAABJRU5ErkJggg==',
time_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMESURBVDjLpZNrSNNRGMb/ViR9qw9FUSQFQX0IMshI9EMXJIIuZGFZGWZpqSHmxM25dNq8pk7LzTSdc1u6malo3vI6bHmZrmmp0+mclrfNuTmvk3r6zw8rKfzSgQfec16e33vew3kJAMT/aN2msTDxjKzoaX1nOXXmczXd0lkRbmkRB0/W5ASXlr4KP7UhQCqJ5XTXUlfGVK2YnhiCyWSA0ajHt5FudLcUovq137w4NZDxT0BTIStNLefCoNNCpTXgbYMGnOKvSJd8gaBKja6BSWhUMlRmPwSPdT90HaBGEO8iL6dY9FMaSBXjeNeogXrMBL3JDJ1pHqpREwS1apQ0D6NfUQVhzI2F7Fh/Rxugjh9WOUgm+kZmIKkfhnUZjEYMfJ/AEhlbZf4JZFao0aLU4oMwBJkRNwU2QEP+44lxbQ+ENYPoJSsvrP6EeWEBGbk8TJGgueVlFNc0QDpgRFKBEj3tYmTT3TU2QD0/cMmg/4ZEkRJTixYYln9ghczQopgQvitDWCQTtbI2hIjd8IDniFFNG3Lp1xZtgOosvyX99BCYPAVGzRYS8gNjxjk8e568ZpYquqFbBQJEp8Es94RH5mHk0Nx/A4pTvIaHeqrAlijRPGjEONn02JwFQSJXBAmd8SjfCT65xxFRch2SjjRQii7DLXU3XJN22K0B8lhevCYxFTLlAOIlKujIBzNYyIqCkyjt4qC48+WasbCDDXYdBfxPiXgocsOJOPvVozF2dkROXMCx/GiPxX55PkRVnUiQ9EOhmYUP2a/VnFIbhITqALAqfRFV7o3o937IlMbgdp4LDkUSK2t9ZDG9qW/ir6O3lYPGdjliea24lX0E4nY2RK3J4MsSkfsxnoT4gyuNhq/oPA4wiL69NGKb7UdlRNxhZFGvmMu4nuiqZ+Eq9yAupO/DudRdcEnaDi+eMzjNUbgncIMDg1DuoRH2fw1TBsPbKS3UveDFk0taLuXiolXWmE25Kj4as2n2Lv8sHOiEnDRv/ec0biTyyrr99M3YSSO2/Hn+CweZ2kRiltpxAAAAAElFTkSuQmCC',
timeline_marker=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADZSURBVBgZBcFBbo1hGAbQ83+5kRCJhA2IEQMrYAPMxRZMbcLG2ICRGTeUtE1jpr1p3/95nLMV8PnL3eOT07Pr79f+/f34GuAAcPfqgXseunlx6ysADsfC0+3k1n03ODP41oiX2+IReO7KH7sfLr1HPBEsCOKNc0cXPghGDFZUUPHWb+/UIKpYUUXU+LRFBbsYLCqICkbsiArWroKIQVQQFayIYFRQFYwKVtQgqhgxiNixooJdDKIgCtaIHVFB1KAqWFFBVDCiiAoOuzMwfgnqpyCOYCsAAADgP4mZnXDW2crZAAAAAElFTkSuQmCC',
transmit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ/SURBVDjLhZJdSJNhGIYHkjAQgk49CMMwJKPIkyIcDIJSFA+UkqKDQM0y/CFRdKmlMVGWQ1O3ZR5sKjKXP4wSayyFJYpSdhAEhRDYQXUSGAvry6t7mYU/1cHDBy/vdT/X87yfCTBtqmlPispBuGeB6S6DkNNgsm2BiVsORupTtt7fClcKjDI3BC/GVUFYHIOnPpjqgkBVFH955faAaU88U+4JZvvhWUCXe+BhC+oKY9dhpBrG6yF8G0Zr4e65CXry4/8EhHuczAmOBUy2GQKt0jX/7jRwxYy30ErfBYOAwnwl4Mx0rAeEu9J40mEwO6iOdoOxxr2M1iVIt1m6M/SXzOC92IyrIAF33l46sw38mqLVYtByPNXEY4eTSJ/0nDFdK6PViQxXLvGgCYbKYfCyvhXgyl2i43QibVarQqD3PDQfcZik+4qZ/vXZAtfM0h0meAPp+vGctXAn1yJdP74idc0YpvWYmZtHkRnUprw0SXeVGW3ZXx47jJNuFN8lpGuZn5/PiUQiqdK10JkFTelRGg/FUXcABq4qYN8nk3RXidzTYgpj2zXjOhPT/cBQ2QnB30Kh0Fe6805J9y0NaVC930xt8vpotqQVk7b7mvtVSDcq3XjpZsS2K9jH4iS8eYTf7/f+fA1b8kF13SVwmZb0WIBGcBWUSbdIi9m98WyC96jzSjAYXBO85vV637nd7oRNP50tyaLKN237lVWCuwUbfHwO7+cRbLS3t9t3ursTfFjw91+dI4IjgtfsdvvnhoaG5P8GCA4J/iK4d+NMsE/wSk1Nzcg/AwRnCV4WbNt6UXBjRUXFVGlp6cm/Bgj2CM7ZadZYCc4uLi7OVMVtnP0A9SbJ2btHXdYAAAAASUVORK5CYII=',
transmit_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK1SURBVDjLbZJtSFNxFIcHgjAQgr6VhBSFIhhJQa842JdeLBMyVIpSSUtFZpIIaqa5UNTpUKebmR82FZmmzpXiHKawhqKUlUVQCEEE1ZfQmFg3n85VLN8uHC738H9+5/mfTQNoNtR4S7iUidHmacYtCl6zwnD1NEMPTPSWhG8+vxnOEzDAZBe8ckm5YaYfnjtgzAI9+QGcuXlbA8ZbghmzDTHRDi965HAzDFYiU6H/LvQWgKsERmuhrxAeXhmi+XLw/4DRZjOTAqsBw9WKgHrR1f6b1JGtxZ6up+2aQo+EOTLBfM60GjBqieJZvcJEp0ysUOgvDaOvKER0jaLrpz3Tjz3NiDU5BFtCGA0XFJxyiyqdQuWJSA0jJjO+NtEzq7p6+gpC6c6b42k5dOVCZ5a8b4M1fo76s6FU6/USAq1XwRht0ojue/ztq3fruaMV3W7cZYiuk5YkHY3xOtF14siQqTHdVB3Xcv8wYgaF4W81oruEX7bszFWbQaIbwHEL0dVNTU3F+Xy+SNHV0RAL5UcClB4MoigCOnIkYN8Pjegu4Xski0lXt6vFmqjqfqPLcErg316v9xdNCWdE9xP3oqDggJbC/atXK967oJHtfuBxPqIbEN1g0Y1Rtyuwg5lh+OjB6XTaDY3JxYbGpJ8ZtRdJNeqWO0uj1QC5gjXZILoZspgdaz+bwDtl8oLb7V4WeLncWjBf1p3Kk1kLM1881I1kkVgXRaohdECz5a8sJXCTwArfX8LXKa5Xnsb1xozrXT3qU+NNp857k6PZuxa3gw8J/EedbLfbfTabzXe+KJrB2VbWPwOvLWrA1ukCewVeFLh1rXcyZ49S7UmjwpOyAlcMp2xvIHCswJ8FLl7fl4PGSzURmDw3Viarb/Vb+jUbAgRuEThuu73I4cpj2bsDqrbUvPqt9v8CPKvGd70s+8YAAAAASUVORK5CYII=',
transmit_blue=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALASURBVDjLhZJdSFNhGMcPSMJACLr1wptAkIIuC6zFKjJDb6QurNiFaCH2gWiFNh1rLodNJ/iBS5k5RdvcSSunpTYrc+QHSHFwbmzNbWd+bE7FWFgnn/6TFrKsLh4OvLy///N7nvMyRMTsrq7xUCpK0/EuNK23BgXd66DQPLw8XW9Z0jwcCKTG34+HiwFG+mfWaGh2g4Y/bdDL2XVip1apfSxI91k+UmHki/8IAJjYOR4aZCfD9GJmnTreBqnx1RKhK9U8D5DqKY/vIrUMr1B1f4Cu6z2DV1vdib8D0FXLfgiTeXKNoCsAlEBXFOtS3uMTlXZ5JbceewRVH09lPV663OTS7ATo3wQPt2LOvukwNQwtCbWWQIr6WSAJukroTtzt9k0Ud3iVhXpPUoHOnSJtdgnyXp5y6pxCdo0jjWkZWdY+sYXo0ehKVFdSxfLJchPvrhtYokqTn8q6fSTv9VNes8t9qcGVfLHeKUEI3Wj30FmVXcNA185OhUndz5PCzIuga6odWCToGgvb3OK8FpcYusbbXV7KqXWYstQOUYZ6HsFeEis4joHulhnLgy5BNwG6kTvoCl2xzWbLtlqtadAVX2l0UabaHjldNZdwUsGRzOijYzJunYHuVvf7VSrt9Ea3K7rW9jmqu1Jn/pgO+LvFYvlWrR/NgO7CKeUcHVdwovRKbme0o+XcJoPtOpWsn6AbgW4idE9EtwvYYJ93kH3eSQaDoSN6ll7BHULXfQD951Tz0QCOwXZvQrcAi9kf+22AD6Dzptls3ga8rdPpeK1Wm7T7AQEWoy4w8U8zWoCbAAs+/yIteHkCLKhUqgd73d0LPgL4x6/O44DHAW/LZLIvJSUlB/8bAHgE8FfArbEzwAbAm0VFRew/AwCfB+wHfC/+ImB5fn7+mFQqPfPXAMBgddl7zRotwFm5ubmZqITY2U/WPMCPgs5K+QAAAABJRU5ErkJggg==',
transmit_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALNSURBVDjLbZJtSFNxFIdvzKSBIfWlKCqMygiMoj5UhItR9CKaHxSNIqhMtIhsZSsrtbISZTE0nbP0w2Yh03xhmK2WKd3EmGV+KIJKiCTIoBesmXXd07mIlS8XDhf+/J/fec65VwGUcdVRGS1lo83RTUeZht+u4SvupvWSjYbc6In3J8IWAYM8qYXeZikvPG+Cx25oL4P67CCeLMvkgI7KcNqdrXTVwLN6ueyAO4VIV2g6Bw1WaM6FtqvQmAPXd7fiSA7/F9DmsPNEYD3AV6wJaBZd499ONw8bcR00U71Xo17C3Jlg32EbDWgri+FhiUbXLel4RaMpfxGNZyJEt0B0O6nJ7MS1v4CKXRE4kxZRGq/hkSmKTBqFG1Yo3LfZUatFz67rmmm0zqfO0kfLRajNgluH5H0MKhL7KNk+n2KzWULgxh4oWG1TRPcVnTWjs9WfMIpuHd7ziK6HylQT1xJNouvBnS5dY+soWm/kwhrEDHKiXyiiO0ynbNmTpR8aRDeIOwPRNQUCgQRVVVeIronSOLi4Nkj+SgNnlsPNIxKw+KsiusOoVbKYg/p2jVSk6LoD1B7dKPBvv9//i/KkbaL7jrwYsC41krNkdLSzUYOKbPc1t7MR3aDohoturL5dgd0898Gbe3g8HpeauvBUR0LkB785LNQeF671ZSzQA2SEil1HRTddFhM59tkEni2dB71eb0jgkPfcvm+9x9cND7XYCL28y49aC4HDMSM9yTOrlUm/spTA5QJrfOqBjwFads4hKDAlsn3rLLgcxaeiTfg2G/qmglcJPKJ3drlcqtPpVP3m6YSeNvD/8yVvLvo4kwIE9gs8JPCNsbOWHTN/fK86AAL9PKnwWepdhgHfFkP/RDhO4H6Bz/5//ihlXn5X2rKRD9YoBk5P523aNB7Eh2l3txhyxgUIXClwwlR7UVMWnLqzdcZHXVs6v9dh/fwPHS3E3E4GyQ4AAAAASUVORK5CYII=',
transmit_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALiSURBVDjLhZJdTJtlGIa/hISExGRHO1BjthEiRp1HzHgEo4vGYTAkGwElohkpyIYRyFhZgQIBAorVqtSyCmxpC8Gu42dFkLBVWBijShYYPxJBQYSQEMeQn69rbLl86uaMMN3BffLmve7nen8UQPlXBqzREiMeywgD5gBXTAF6a0foqTLSZojeuX8nnC+gircVxjolbhjtgOt26DeDq0DFmZu/u2DAGk7/uR6GHXDTJZst0F2DTIWOEmjTQacBPB9Dux6+TOvBkhz+T4HHYsIrcKigtzYgoEZ0Ix5Maj4VgU2r2XQkBaebU5nprGK08RXGzM+lC2w+yLefBRhukYnVATrK9tFe9JjoVoruEI7sIWwnKpe6nj6x7NGxdqsLNpa4c8vF8Eev/qbQZzQx2CR6ppCuhnbdk1zM/5mvK6A1F1pOsuVKYXWiZNv/62WW+2tRx9sIzA8ycV77hyK60ww57p3NdTpCdC/iLoemdCfW1LhV56GC1Yni7aA6iPpTDre/e48f7BlMNqSjGl6YUUTXz5DcslOmObLDRFfF/i6cOx432XeqfHX8LEHfDbZm38a3kMb6lJ6FL2JRL2hBH7mmiK6fwUawa0O3G0F9CtQnrdz8JMaw/L3uPpyO75c3+X2iiFmzZt1neOavo1F8YEOR253hUgGiq1KXFI4pIba74vFjU9+cDgbVa/jmMri78BZrYjJjPnJ39sMXowRcpCYmVDCpUP/G+6KbyeeJe0JP1pD9VMZkt56xHhOL3mS25tK4PVrIj3XxK9PGlyIfPG3xgThJsrLzazo/OALr80xbj+PRP8H1+qNcrYzFZjzTsOvb7/rKkgrt8wH/jTr83k/xVsXzVeb+gKW2ZLu6unqztLQ06pEFKYf3BmveeZarNQnBK0bNgFt3MFJgu8AbhYWFbf9b4Ha7X3M6nYs2m61450aBy/Ly8vpzcnJe/s8Cga0Cv/6ws4YicGJWVlaCJOzvtT8B/ZHVkQhA0SkAAAAASUVORK5CYII=',
transmit_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMFSURBVDjLXZNbaFxVFIa/c84kPZM4M0lNE2svw1g1KG1NEKt9kAYLZbRKlBrQvhYbURG0L1V8MQYUES/QIhrQJwXvtAbxJQVHYiC9DYWCTpVOgkwzzSSTuZ199jln7+1DQDr5YL2t9a/FWuu3jDG0kZsaBI6j1AEcPUQYggrz6OA3hPycZ97+69Z0q00gN/U6Sk3S3RvHjQM2GAWtBoR1qBQFOnyLsY8+bBfITXVizBnc7iydLtSXwauBpUD6oANwNkGiB2oVqCz8ShSM8uK3gQ2AUu8T784CsHxdIVYP4ntdZN+0GJ2wEM0uajcOUswrlAI3mUW23l2f4NzpPVjqMvE+h+qCQspdmHAFFZxEh48RSNDyHJ54Dyu8nUD+w8A9DsWLCh3ujaH8Y8R7HbybIL1DmCAiCq/Q1ZuhsQK2BbHUfrzVo0j5KFIeonxths07HJauHrOJ/Cx2J9RuQiTm8Jof46YylK99R31phNXSSD1anCulk5lIy08xYo5qCdwEeF7WxvcyYCD0wG8FKHGYtRsQeqcvDJ9Ind/7vFjp67jTvfsp1lIdj+O1AmQLrBgQbrOJxPqWZQtEvROvFae1tExqm5JS/mCL4h+J9JPp5NZhqgk39vfg9v2YaP1CtnFsRGOR8p8gGgLZUviNA4z/1D+7fWz8oU3/xpLyspPo60HVz9I3fBypzUnj6BKL8wCLMbzmKURVEMhveOVsAORmZ2c3+754Ol+eZ9fuZ8G/xPxXZ3jkhTcIZfBEfmd6bHihWAH6Y4x//Qkb8H1/kkahq78nTqLXQ9Wvg9Go5nkyIyeswi+T78xvGXh436tzDXtj8czMzJAvmuO31X63e3YOob0rGC148MggOijhugUG9jx3XySD11h/9naEEB90rF1QW+/aR1eigolWwHK4+GMB0Gi/wJZ70ygZvpyb2H1/m5mmp6cPCyE+27F0KvHA0S+Tjl3ERFXgVsfaOPFBylfzFH6e+D62ofuo7/svab/5xaWpI8Jog9H6/9Aax2gNWiu0xhhzx3/je6YQnMUxcAAAAABJRU5ErkJggg==',
transmit_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALcSURBVDjLbZNrSFNxGIcPCJNBIAQRISRGYQWF0g27OJgQVmQFmpViZHhLaiaGeCtTQ1OW09QtS83NiR2XF0bZZZjKXNYq8kMUBX7qU30JIsM68+ldUpH64eXA4f/83ud9z/krgPJfjbVFSZkZsb5grEXDY9F4WP+C4Stm+i9GLTy/EC4QcIZnvTA1JOWG14Mw4YDRFnBdmEHNL1gcMNamY/TGMJPd8Molh61wvxbpCoPl0F8EQxdh5BoMlMDN1GGsybp/ASNWC88EDgY8rNcENIqu/m8nZ54ee6aRjnQNl4Q5csGy3zwfMNKyiSdNGpM90rFGY7AigoHSZaJbLbo+unN92DOqsR1fxo2kCK4f1FBlijqDRu3OjQqPzRa8HaJnCeoaGSgKp69gmntV0JsPPWfkeR5sh6dp2hdOvdEoIXArDapjzIrovsPXPT+bq1Avun24LyO6Km3HDDQfNoiuiiNLusb1URerp3ILYgYlUW8U0Z3FJ1tW84MvQ0R3BkcOomvw+/2JXq93o+gaiuyxZHbGQMXmEErXg/OsBKz5oojuLN52WUxmcLt6bClB3U/0mnYL/NPj8fygNSnhXHtM4LL7BCk2gUvWzo9WFvlVke1+4O4FRHdGdHWiGxfcrsm5B1N3LLn2bWS0R8+VDSSj+hsp7DtEQsNK5mq3BgNkBNtxk+hmyWLC/nw26bw8z7GDwVet3H3Z/Bvs9VuweArpelpHjnMv22tCtUBZ5FFl0a8sJdqtp2XeIHztkYmrD/K4cj+LCvcpKu9lYxuvIu32btZdUmaXgqPdbncg9eYG7jy34Jw00+Wro2OiVkLOYB2vJMuZQGS58ja8WNEvChDYo6rq9yPWNXP7GsMD8Q0rArvqw+bSO3fSOlZBhmMvEeXK1KpiJXTRZRL4gMAf7XZ72cLgzVW6bye74llbEvpeYN2St1HgNoETl9qLKH9eXRrC6ZyMg9nZ2SF/3v8CIIKyHGFPw/kAAAAASUVORK5CYII=',
tux=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJKSURBVDjLldFfSFNRHAdw88mXnuqtHu/dufe2e/dHjUl/YEYRRsFaLViYFkUY+CcDwcqiJAjKoCiKCqRw+GCEa6vIdKOJLBPaiqy9CBlkYUUbq83Z3f32uzMX0kZ64AvnHM79nN/53RIAJYXC8+wux/EZQRDiFsu6Y8XOFdxkjG3lOE6jgABYrVbwvFS3BEA6YbNVQTLKWL9hI2RZAWGPFw3IcuVh184axMbDePrEC7vdrgOhJVQgb488MEGdMCH9zozOlgpwBtMr6kvZogBRUvYH7jdjMrQF09HjON3RDoulgvrAnP8FqFTW1NT8PvkjhamPn6BqQCj0jPog6894azCwVUUBuqGUDg15vV7oQ1WzFBWJRBzV1Zt0QK/CT8iyggAdsLlce9RkMkFLDdmsmos+Hx4OwWazgX6xRoi9GHDd4/Hkbs9k0nT7rzygx+FwUBU8hXX+AzBeXG21mOPBYCgHpNMpAmb/ANncXm3tvtwzCLi6ABi5pazwX1QORHsFtedGC6Y+f899+BcgIuJE/W69kQyN9fLNBUCsz9o/E5aBiILROxycjm14Mx7D3NAwSwWkAmvxoYdh9LKAn37xa7LfuCsPTNxT/OqYgpkRGVpYwu2z5Xj+IoL54fUNYLCrHBgUKCI0yrc+YywPvO42RqcfykgO6YCMLz4TTrbW4Whrm+Z279UYW4PzhwxAQELKJ2HsghR81GbenAd626WV1xrFmq4j4jmav7zUIExWKnwVNaxsLsLygzukjoFTQrq7Qbxyxi2WzvfgNzL+mVcak7YgAAAAAElFTkSuQmCC',
user=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ3SURBVDjLpZNtSNNRFIcNKunF1rZWBMJqKaSiX9RP1dClsjldA42slW0q5oxZiuHrlqllLayoaJa2jbm1Lc3QUZpKFmmaTMsaRp+kMgjBheSmTL2//kqMBJlFHx44XM7vOfdyuH4A/P6HFQ9zo7cpa/mM6RvCrVDzaVDy6C5JJKv6rwSnIhlFd0R0Up/GwF2KWyl01CTSkM/dQoQRzAurCjRCGnRUUE2FaoSL0HExiYVzsQwcj6RNrSqo4W5Gh6Yc4+1qDDTkIy+GhYK4nTgdz0H2PrrHUJzs71NQn86enPn+CVN9GnzruoYR63mMPbkC59gQzDl7pt7rc9f7FNyUhPY6Bx9gwt4E9zszhWWpdg6ZcS8j3O7zCTuEpnXB+3MNZkUUZu0NmHE8XsL91oSWwiiEc3MeseLrN6woYCWa/Zl8ozyQ3w3Hl2lYy0SwlCUvsVi/Gv2JwITnYPDun2Hy6jYuEzAF1jUBCVYpO6kXo+NuGMeBAgcgfwNkvgBOPgUqXgKvP7rBFvRhE1crp8Vq1noFYSlacVyqGk0D86gbART9BDk9BFnPCNJbCY5aCFL1Cyhtp0RWAp74MsKSrkq9guHyvfMTtmLc1togpZoyqYmyNoITzVTYRJCiXYBIQ3CwFqi83o3JDhX6C0M8XsGIMoQ4OyuRlq1DdZcLkmbgGDX1iIEKNxAcbgTEOqC4ZRaJ6Ub86K7CYFEo8Qo+GBQlQyXBczLZpbloaQ9k1NUz/kD2myBBKxRZpa5hVcQslalatoUxizxAVVrN3CW21bFj9F858Q9dnIRmDyeuybM71uxmH9BNBB1q6zybV7H9s1Ue4PM3/gu/AEbfqfWy2twsAAAAAElFTkSuQmCC',
user_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ8SURBVDjLpZB/SFNRFMftj5IiW9saIQjGQrBiUagEFWLDydrUhAWzNNuWTZ1oKIUzf4RlzZY6sWyrLZXhfFszQ1eac2SQkYWW0yH0l1AWITSw1IXK+/beK0RBptCFD+fcyz2fc+4NARDyP6x5qInbVVEn5sw2SHdCL2ahQsiey4jhVW9IkBPDKbmfyibN6Rw8oLgrY0MnYaEofgcpPcitWldglLLQQhXqqSKdlIaNm8k8XDnBQWYMa2ZdgS5+O14YyzHVq8eQpQiFCTwUJ4YjX8SH+hh7wapNCQ0qMGdF/gh8/4SZN0Z87a+H13ENk89vwz85AiJ378xYq2ZLUEFjxv5B//t2TA87MT9KUNiZ3D9C4KFKMBz0Cbults1RxzVWoiAWv4ctCPieMsx/tKHzciwE8blPeCLz1jUFPAkRyhW35UWIPfB9noWjLBX2shQGOn898QsRSS/BET66xBWatq0ScE86NoUlORSRyYOYmJpH2xRQ7APy3gEXXgHnewCtsxPFRgXU9acgvyEMiEsOVS4LDsia0xJP6+EcWoLJCxS8JZE7QCK7j0RWFwmlmUCVU4lnviaMfnPD0K+B3CDAkfzwWkbwoTx6adqlxb1mFxS9VFeqo7KbxLkOEmdsVKyRoGu8AV0TjaBXreciDJ4cWhBgBN6KfaTffR3p6hZU988howM4aycht5KQWUgklx1Gj8+Clat7rIkW/P2IcWtB6Uhp1KJSeWsxTjEAJTW6agVHC/m441ZB51Ywxbo+xeoJaCbteWGVV6u5e9JcpsiE1i980eM5flLHAj/RuSCQZy7KaqNR585mOtOR3i//wUagLtdQ/KTH/hdr6PM/RhGjA91Gi1AAAAAASUVORK5CYII=',
user_comment=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ5SURBVDjLfZJtSFNRGMenRkgY1BKiL30yEkqJrCjrgxBB5Qtmyy3NcGoUuqD5skEm+ZZizpTUmZEw33ML06lzGoQKtRRETXM2Z1LOTBs6LNNw9/w7d+IiuevAj3vO4fx/z+E5lweAtxVRvp5Pqaf8psAF3RQfngtBa1OvCet2Bq5Ge/80K5nkCntR7AwhsP0imF8msCwRfF4k+GQlmFxgYF7YEKerDJzV90vKexwHZm0EX2hw6juBaZ6B8RuDsa8MRiwbggL1IP57A7b6NK36kYbH5xiM0vCwhRXYHYKMmnd/gwlH+dvunPTOehy623ZLlrfO9oCVbA72JsMzjEPK2QP5Gb5UGewJxcXtKBLsQ2JKBkR5OkfHq/QfnKKlH2uONd0f/ecVioM8OzXyC+hRRKFAeBC3A3dAfHwn7ob71tCD5rnFlc3gKiVjM+cUlEbsqZ4xqLE81IT3Lx6gXyXDUMsjpGQqRip1Y2zwJ0W6tWfOyZUQQepEYxpZHW8FTFqsGdvRX5dORLlaKw0mcP0vTsHekAYPXkDFE3VxNplU3cREXQrMdRKoCnOI+5Gycu9zlR4uBbvON7l5nNbkykunGL0VkGvfQqo2QFJtwLNhIDHfZHc/UZvpFVThxik4FfEwNS2nDc+NBMkDwI0+4LoeiNQAV+sJcrsIxMnNJDD0noxTMFt4CAPqUiSp5xHbAcRoCIQ1BBFVBGFPAYFiAYPNSkxl+4JTYFYGv6mVxyBU2oe4LiC+GxDrKPR7rQU4G9eBl/ejMVEW1sspMDUk8V+VxPsHRDZkHbjcZvGL7lrxj+pe8xN2rviEa63HLlUVvS6JPWxqlPC5BH8A3ojcdBpMJSoAAAAASUVORK5CYII=',
user_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKRSURBVDjLpZNrSNNRGIeVuaSLrW2NCozlSsrCvqifKrG1vyznRDLQMi9TsamsUCzvSWJKC0Ms0/I2hratmVbi3bLIysRZlgh9qFGuCKOF5KaonV9n+yAGokIHHs7hhd/zvofDcQHg8j8sW0wN2FpQJuVNl8u2QC3loEDMtUX7CYrXJDjrx8u6FcYlNVE83KbciOCiNISD9MDNRHaQf3lVQZWMgwYaVNNQqcwBF1dCBbhwlIczfpypVQWlgZvQVZUPS6cag7XpOBckQIZkB9IYEZIPcee02XL3FQU1scKfM98/YOpFFb72XseooRDm9quwmk3QKXdPvdOkrltRUBG9f8A6dBeTw0bY3+ooeufZatLhToLv8IpX2CZrYnsfTtXqVP6YHa7FzFirE/ubJrRk+sM3UHlfwNSsX1YgCNG586WNKZ7SPox9mYYhLwz6PLkTx/n5+G94Bj8BT1x3ni+u3vCPgH/c4OoRbIgXhg5g3GJHowXIGANSXgOJT4G4DkBTXolnMT7oFbPxgNlo7WDYuYuCAxH14ZKTahgHF1A9CqheESj7CZK6CWIfElwrqsRI5hHMtJeBjHfBps/AUJrvn55jbiqnYCR/38JkWzZu1rchvpN2pR0VjwhimglONREYw/fATsOokANZXKDECz/UQeiWsD45BaMFPsTaU4So5AYU99oQ3Qyc1hNEagkiagn66NjE1IKl61fhdlp3I07Be60qx5TjPa9QlMwHxPdDQUdPWELrCSGm6xIBGpq96AIr5bOShW6GZVl8BbM+xeNSbjF/V3hbtTBIMyFi7tlEwc1zIolxLjM0bv5l4l58y/LCZA4bH5Nc8VjuttDFsHLX/G0HIndm045mx9h0n3CEHfW/dpehdpL0UXsAAAAASUVORK5CYII=',
user_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALTSURBVDjLpZN9LBRgHMf1R/NPuo7Uam0aaV7G2uSvYki6c4hhWZe3mJw6i9FxnMj7RUNeIzYvnUtLuXKGXDV2yMm9JK10iDHlvOTuvN6369Zs1kZbf3y2Z8/2+ezZ8zw/PQB6/8NfG1H2B1n5JMPlAsoBsEkEsFyISqqdccY/Ba7ZGTJKvYiaygBDVGi570tEtjsBMY77NRRbo7RdA2UUAmq0IlsrZVN+Q0SmhzHinQ1xxY6wuGsg23Ef2sqSMclno7cqBtFOxoh1PYLr500RcYa4Vpvgqb9joDLIZE498wmLPWWY6rgHMfc25C9zoZCLwIk0Wxxttr800hCAz88zMfTQDeIS66BtgSKqVbei/xFmB5qgGuJoadStFSIO+BkWX6e7GFiQvAB+TmFe8gTCPNLMlnyY0rDX/GxULYd+GisDVVDLmnWo3jdAwLbFd2nK5uq3Fky/vguV9Ck2xrohrYlQ62Rjd46+EamedozUCdnEMrhJXmhM8tTRnucChYyFTVU3VKM3MNdPx8e6MEgqA3/0F/mc1DMic/cYuHFDTDy6MTypQv0kECsDaH1AVocACmkiNtVCKL8EQz1BxdIwE/IKpxlRvusp3SVa+1Z7u/qx0dS7gXIxQBdqECnQIJXzDNPvGH/kIKjHL2NRlgRZoRtiIyJTt15hMNliY5aXgOJqHkL4QFgrwKrjQdp2S3vst1DLw7AyEYgF7UlGSi5gtiUewjjLta2AmGWpUbTfQUBEDTI6lIgr4uBDKxNifgEm+/yhlFMxN5QASakPAsNLMd+Zjn6GlWYrIK2lJ4oSzddDQ7PW7UMEeJx7Dlgaw8gDP3Qxj6KnnAx+DhkuflWghzOVgym2K1onfdtHkjfSDFKYGUbHvXnlaeE2WBUWY7WvEH2Zzqi/agYHcq7ixMWW9pvRqYfGuTSDHafR34Gozg62WH+VQ17vzHd5w2PYmO40zr8A5dG3M3vHNHcAAAAASUVORK5CYII=',
user_female=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIpSURBVDjLpZNdaFJhGMcX9EExPVNn3Xcd3Ui3urnJbK4guiioiyjKPFm2QRSINZl2RDuprTZNaTLQ06ajizVhDILW+tpmYxXW6mKNXTSKGS2Xm+foP18vBjLRoosfLzzv//nxPC+8NQBq/oeyRfpAvYXVStMeXR2cWgoWtWT1hEJu+yuBXiG92nNYkg8cl8JfoPuoBEwrhXalOK/bL7NWFXRrqSSrEYNR1YJRi8DoJLC3yXGlUYqTCupnVQGjrIVLU4/RrmN4F7Ph85gfj90GXNDshaOByvdfO7SjqiCzMIVfk31YnuKwnBjE0qswUvMJuNQU3obo7RUFDpUol5qMIDUTQ3p2sEAU36ajSCU4uJrqhIor7NGFt9mVYv514CLWpoPIvH9U5PdMGM/vnoKjSb4m1wR2lhXIW7nibp2q3eCffMK4z1gCP/YB5uZ9IBmZ2rerRCA7OLCFnG/OMPCdbUAu/hHCracQrCMQLEMQbnDI9Y4g2HEEJEOyVGPv1pIJyEV2dBzpoQkIwWfgncPgLRyynWEIbBRZsw+CNwrhXmhDsiEgIxb3vd2HOdqNjDOGdWsY39vv4IvJidXrfqx3sJg7bUOmJ1LMkp5NghVXAMl2LxZNbnw1schc9mDF6MAizWBJb0fyEosfN/2bBS/uGxOkED9nz0/oHeDNkbKQ0eP6LoFkCz2zJW8w/9AgCrXQHq7NNEyC5ehvPv/yQQvtXRgwiCr+xn/hD7c3w4UciyonAAAAAElFTkSuQmCC',
user_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKrSURBVDjLpZNrSJMBFIY/u40wW5sNkS7LiaCJgakF3tDUMZ2aYZE4M6c0p2KiGLl0ec+cqLPErTTnbW1zltHwkgqSaYo6zRKjX0IZlJEg6Uwne/vcjyhKDfrxwOHA+3AOh0MAIP6HPxrJHgfF5Rz6kpR7ABIOFeIztGWeG6PonwSJbvTrNeE0U20UHfdI7kbSUBJCRbrvfhP3hHX+tgIZlwoFGZSQoRLuBjQUhzJwzZ+OGDfq4raCEt996JblYK5LgpG6dFz1YyAj0BYpQSwIvGhrTVlhlC0FtbHMryuf3mFxSIaPvRWY0uRitqMUC7N6qIT2i68bkvdsKbjDOz64MPoQ8+NaGF6pSNTmekGvwv14l/EtV7DhKnc7eCc3qVLd8X28DivT7WYMk0o8znSH8IHn6hWFK/4qYISoKNaclqTDnD5Mv1+CJjsc6uwwMxv1i5lvuCT3Rr4uGhfljr9JCOtgjYUVWxPHDB3EzJwBLXNAxjSQpvRBarMnEhtPgV9/EjntF9A6VoVM7VkEVdoYfcpoFmaBc2R9ROB5CbQj65BPAanDJgj7TUhpPo0nEzV4pK82B9VjUkj7MtE4LIFQyYZHCcXoUmhhQUzkOK7P67JQXa9DXBeQ0Anwn5qQQO67Ea7oSUNpdwqKOwXI0/FR0JEI+UAhYhq84ZBLrBJTYifTQk8BogQKFPUug9cGRKtN4NU6QTMqhXKkHI0vJagfuk1KkiEbKIBAyYGdmHh7SETsJd40pYr0Igcjn3/L6BHXD/4zE+JJzsnsEVx1BAGVNvAqoyFW4Yma53mIb2bDNd/ys62IoPy8wqw6yermjSLrYxE6OdOv4QMrqHWZxW5bYwVq1+z8VQamj2LeOc9y7XJjAFg5OybJ8J5Nv3EzyJG/HM3eCYaI2PVr/we4bY/dzdCujgAAAABJRU5ErkJggg==',
user_gray=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJUSURBVDjLpZNbaJJhHMZHUES51SoiqssWm5SMQekoFwvnYR4RP02dh/yMMhJpW3PaFOuzURfNi8GUDusw3QGDgl0k0Y3QcbjVqtsY3cVAu9uMwqf3ey+EwXBFFw/vgef58X94eesA1P2P1r284DDfDp/ajUhHPQZOCuBr3wXWrLv/VwAf64pFtM0YO3sUN1U7MOo+gr4OAdzSA2Cd1pENASGjGKO2JgyQ0A3TIRJuQJw5DF/HXhha91Q2BJw/3ojLaiHGr2gwwp6A/VgjrhqbYW0/CKZtJ0b6zmyvCRhU7ltdfH4XxfcT+P76AeYf9ePrs2tYmB1DVLP/56eHF7fUBCQcre9Kc5NYLmSx8nGKaJruS/NTuOMRFWpWaJP7tkql0ux4oBPlwj2sfnlKtfIhg8mBTojF4iei0+e2rQtQKpUNKpWKU6vVSKVS6OnpwcQQQ6XRaOidTqfj93HiFawBkOCmYDB4izfypmQyCavVimw2i0wmA5PJhOHhYXg8HnR3d1dkMtkggWyuAkjwEsuySKfTMBgMMBqNsNvtyOVyyOfzsFgs0Gq1sNlsiEajcLvdFblc3lcFLAw1/16eHUQ4HAbHcdTkcDhAJqPjMwwDr9cLl8sFv9+PYi6Kt/0t5SpgMdJSKb24Tg2JRIKCSCVq5iv19vYiEAggHo9T2I+XHOaCwkoV8PmxPzQfavrldDqh1+upkVcsFqP9+an4M+mPrq4uLERFZZLh1rzC0rSvnnRsIs/4ivRdIuOXFQoFePEhshYlEskbs9ks/Dbjq6/5G/9FfwAGy37p9rgYIQAAAABJRU5ErkJggg==',
user_green=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJkSURBVDjLpZNbSJNhHIeli4jAKOhun9KNbUpaURFRREkFVjpNRcssXOKYZ9J0ihnN05zSUpflzMOnW5tuammajUkWpCbbrOxwEzZJw7Rt2pxJh/16/YSBILPo4uE98P8974nXA4DH/7Dq5GQn+5NdxcK8lsK8msIczcKYlv35rwST3ZwxRw8Ljh5qmU4i0VCYbfWC+T5nfE2BQ+fFBOdbqeUdEOyEWZrCRBWFNQX2ehZMGh4mHkkwM3AYlgYKVoUXZpTe+CKnoJcd2uJW0Jbi/c32vg+WF82Yel4Hm+kYbMYYzH1Qoy9v28LrxsT1bgU34nYMWYfvYdqgwcJLFUHN9K1GFWp4/ga3R0hX0huDLp8ercnci0XDXXx/08GwMKIEnbsPJxJPjsbfrt28qiBNSW+KrcpRhhSxMeTogrQ0CI3XTjEs9XVTLQgu3I6z5RntqbR86wpBqkKxTmvukMTK9mPgdwN6bRVoeCdEtTEN0gEBxE94uDOSgcezMoSJdyGiRNic3Fi7wSVIrissTmo6jhaziGEpeHNQgLJn8SjSX0B+dxSy20NRMZQEST8fCXIu4iuzql0CUx7n13SXENd7z6G0n8dQ3HeRGed1RSKrjYt0VRD49UeRqQmGpTcfg5m+P1yCV1d9nVadiBQcwS1DKgnGkFWjkfsgHFe0XNJGEEkYJE/5OF95EDZ9AYaz/ZwuwWhTSo4xx+cnvy4Q0dLdpPDSqoRL/BEu3glTfsAiyRSseIWPaoEnrZf6RJUnvI0sC3SeKd1DLiwAoSX+CCkKIBxwhoi4X2selvuNtwg83f7Gf+EP0qq8jpoy//YAAAAASUVORK5CYII=',
user_orange=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJlSURBVDjLpZNrSJNRGMdlr7qat+zKaoYhjqYphRCWCtKFoCj60CeRpDB0djHUSMOZm0JhGEmlNhczXXunDgy7TNcVcsjGpk6dNsiJyyjWNg3ZnJfef69vMBBkFn34wXMu/995DocTBCDof1h1cvBJnM5RTsBVyYLzBgvfigjopbGDfyUwK+Nfu2RsTNcTDO5aAk4RC1/KQ2BqjetbU+AiOZip/xNyLndQSeCHmMBUIQFzTjDWFDiu0O0qzmJKU4OvPSmYuETAXhKM8WshsOYR0NZlRAUUtOXt+Dk99hYufSu+6x7D8fEAnLozmLEq0V3M8ww1F4QGFEhz+Aa3QQmHsQPeQZJGxdRuEwnp+SRjwCs0FpwIf3guwfayKBE+owxzI50M3oGn0JbuQW323vE7uac2rSpoFB6Pll/M0FjEofDZe2Go2ocu0VGG5dpjUWOEXpPlp72X5h/irhBIcrNYNunp5s+31gFTWmCsAfiQDWiOgXq2H1Q7H1TPSVCfmjBaHY4HFzJfNOQd5vgFZGHmo5n7bEBfQlPMBNGVCqgTQZGxWGjhwivbCKorHb/UybDf5UFekE76Bf3lu5ccz0uxpIgBOvgMlGoXPeZhvnkbHY7GbEMYnHVseKQb4OquQF+JYMEvMIsElFsroTfQL/TqCBYVOzHfsh0++RZ4mqIxJ98Kj2wzc7qtJhLTb6pguJ5A+QXDLZfLTGXxi45762G7TUs6BKtirWZjWByG/opkH52pWvEKEyphRK8oLan9aurkgCSGslRHYVTCwQjNkDgSpptcqrMwafZd2cGUyTZhRMDf+C/8Blefvm4GxFC9AAAAAElFTkSuQmCC',
user_red=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJfSURBVDjLpZNrSFNhGMelD34op67Lp4I+RKBREkUNIR2po7LOztlxbY1amstyXWw2m06b6Ra1llpqBJrlJdNFVyqxixZmIQ5npVE20DCkYqaDDBVy77+zDQbCmEUffvC8L/x/z/Pw8oYACPkfAl5mKmWl+cJFMMTzoNsUBnXsQqhk4qt/JVCrUosMVBQs2yJg5igWhUMbH4a0uKVQ7VWUzSnQswJc4II6LqT1Eg6NkI99GyPArF1M5hRoBZGkpTIPI60WdFYexO4NfKTGLoEidhl2rotEmXbXgqCCqj3LXa6P7Rjrvo7vr2thr8/B4P1ijPa3ojFjxURf3aHQoIJqxWrbuK0Jzp5bmHzbzGH11uP2ZlSnx/QEXcGaxM5/tnlrx5NMAaZ7ajD1/p6XyTc38FwjgFWY/KJRKOUFFJQnpfE7RFSNk6Ux5fiEvmPJaMnd7sVT/7J14ytDozMx+WJ9nCJylsCcIp03oNHWfpMwgOMD0PUSaKoFrlSAVJwDMRfCfe0ySPcrfGEY8iCBKq1LpEL9grYtjJGky4BHd3xwQVRagBIjcDofKMgGjh8AuVQCd4kJP9Nk5K6IPusX9J6MmnE+zANOnQAsRT7OFPjO+iwgOwNQK+FWSoAsFcYeF6IrJ3raL3hniCbjT40gSm6FqnIQLkg8XXWHQTT7QXRH4OYm8HT/IWfhajPBlruK+AX9DUf1dv3K3zOcYDSFBs4XB2SEZuCgGPQWxkxzGdOsV/hsVfPa5dI1TSLl8AArJ0M0iyGxBIOUBI4dLPrFMnI7QTHRyqasH76p5gX9jf/CH9NZtVjmGMuRAAAAAElFTkSuQmCC',
user_suit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJ+SURBVDjLpZNdSJNRGMeFoC50flVXBV1EoF3YdzKYG0u30FTsY9bQnLpqrSg/5uZe29RtimvkxBbCTJupmzpLl8tNxfwObUxTF2RQRNJFGOqFYELx/nt7LwaCzKKLH5zz8Px/POccThCAoP9hy+LNq+nVJZzdULMZULBCIGVGQpye2vhXAqlYVK5OiYIhMQx6Cg0vFDJ2CLLj9kGcJTRuKyAuxKKGCiqokIwmFPmcCOScDkPa0T3ktgJZbDjZa1Liq9uAcZMEGaciIGLuhZC5H4Lj4TDKrgQHFJgzD6yuvn+F5Tct+PbagumnRfjk0OC7z43W6wfX5ptu7QwouC9ielY8Nix5O7E+20bRTq9Xpttgzj3iDXgEvuJZMPt23Wht4UVseBvw4103zfpbK+qJ82BdezDCL7AythSw8+yRPIW1kaN+gaahBRRo69BaKEBvVjxERDNV+4Az5S/Bl7c/ji+whW8SMItcO/LrxyxcdQ/d9GT4I5INQxh0TWBKkoUUjRN6hw/C2jGwS7pJbp7FyJXZd/kFScVWXcbDcTg8i0jQusDTuSHQ92E2Iwnz2WlIrhoAt8yJVMMw1B2zuFw9RPLzGqr8ghlV1K8lpxKEbQa6rnm6Sah3w5aaAlVuMc5VuJBpmoDAOApJ/SSW+0oxWRS94RfMqaPJlQEtBDWjeNS/ACUlkrd4kW8aQEJZD+5aPJA2TKGyy4fEij6sDurgKT5M+gW+5jvENHHo5yXjCOI1vajs9tGUdc7hbGU/PdWfPeeeAyxFB2ZKYzaojG7TK3xulzIk2saYkzfMi1zqouKILrCUz2mYcjtYcjt5LMe0JlaZT3zpkDIC/sZ/4TfeSKfmHj5WOQAAAABJRU5ErkJggg==',
vcard=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAGnSURBVDjLxVM9T8JQFD2lBRQRFPAjEhIwXRgwEUjERRMnZPYXuPoHSJz8BQ7+BCcHJzZHiST4MWC0fsQBJUKCEQnGWihtfa/QBgK6MHiT2/tu3+1559x3y2iahlHMghFtZACuUqmkSdyiiaqqoJJo7HVFUQZi1w+Zcrnc9vl8LAXo7YexHhaNdaFQUDiSsFarFftXClriJz4eTqHJX7DMhDEbiiDlfALP80if1CFKLYjfxCUZR9s8ZcByvadG7w/QLl/guVRC8U2Etnutv5dlGXvr44S+3ZRFjcrgjEQvnFvG2pIfGzYOZ5c3KLCsWZjNZvual0wmOwD0oX/cVnFeXQDHfcDpcuGqtYJaTQKcHYBEImGebrDuY1AvPWImtIhjLoraSwtjHhvmv17Nwnw+/zcD/0QDt+9TUD1TmJ6zQWpImFTfyM64XhiLxdAr17h2E2Bnc9XcEEURzWaTFARM2na7HW63+3eAEul8tVqF1+tFMBiEw+EYmDpBEFAsFhEIBBCJRHQAhgyDQEDCFCgejzO5XE4bNn2Gp1IpJpPJaF05d8y//40/Nlww9OOVWvkAAAAASUVORK5CYII=',
vcard_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAInSURBVDjLpZPLS1RhFMB/M/eOr0YzH2mjghPjwoWGInWTMDCkmVZB/QPVrty1GXAVtHATLRIXtWgRBGGB2iKxIEgbopJhTMse9Jrpjoz4zuudx/d9LWxGh1oIHjicc+B8v/PgOw6lFHsRJ3uUPQP0eDweBM4DSClRSiGlzFMhxD/2r95zmKaZqaqq0gB27iPr/89m/UgkInSllOZyubg5JUhZ6yx/fIFKb+Csbuagt4Uz7i/4fD6C4ytYdgprM4Vlp3lw0YcQQtN3Vm2fu0XGfMOPaJTvCxaqbxqAdDrNtZPFSFmYGwtACIGeDQDSNW10tdbRXaDz8u0MEU3LJU5MTOQtz+/3bwGEEFuPM5LXCQ+6voy7rIyp1DGWlmxwbwEMw8hVz3ad18FK9BPV3sM80ttZ+pmiqKKA2o1fucT7Y3f4ujLDb3sNO7mJ/+g5ykXDdgd1+9aYXSxHVpRzoKYAe82mVC4AxTwND7PomqO7y6C+oonns8OMTA9QK9q3AVcCx3PzWZZFMplEygaklIyNDnE2cBrhFBw51MOz90MYLZ08HHmyDYhGoyQSCSorK2lsbKSkpCQHXFidx+VwE2i+BMDVU7d5/G6QjBxFF0J8CIfDzUIIOjo6HKFQSMVisbxft7q+zKwZImJOEuy5S//4BYq0QjSnrhy7uUaj13O93ru/r7P1BE21bXyeDxOaniT2bfWGY7fnbPR6+oHLQCmwDgy+GjCDfwBGL0x8usOKBQAAAABJRU5ErkJggg==',
vcard_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIdSURBVDjLpZNPSFRxEMc/b9/P9b9uu5qWVG6soERCIringi6J5wKhTl2CvFQnQQi9dO3QoUudOkSHLkUQniL/ENWyrSRbUqC9dc1dXUXxubvvzft1sF1d6iA4MMwMzHznO8OMobXmKOLjiHJkALWysjIKXAXwPA+tNZ7nVaiI/GP/6jMjnU67LS0tJsDBfZT8/9mSn0gkRGmtzaqqKh7GhKK9zcb392hnB19rD8fD5xlq+EEkEmF0chM7X8TeLWLnHV7cjCAipjrYte/bI9z0J5Ysi8WsjR6bA8BxHCYu1eJ51eWxAEQEVQoAnLYLXOzt4LJfMfP5KwnTLCdOTU1VLG9wcHAPQET2il2Pj5mTKLVBQ1MTseIAuVweGvYAotFouXuJdQWDTWuB1vBZXqo+cr+K1AT9tO8slxPjTybwxV8jqxYSCNJ17S7S3LvPoKN+i/n1AF4wwLE2P/mtPI1eFqgl+/YprWsxum6MUB0+x+7cJMmZ5xhtSxjJZFJ3d3dXzGfbNoVCoUx5/k6Ugesj1P58B8vT0BxgTZ0h9mFhn4FlWWQyGUKhEJ2dndTV1ZUBZdWipj0MQ/f2L3D8BGpzHSUiyXg83iMi9Pf3G7OzszqVSlVeX3MQ+8sb6l/dprD7GxvY3jJxGwOucZhvnB7uGPc31o+dCrlK+VJsZ10WV01x8vq+cdh3nhk+PbqTS98yxTgjpl7W8PjKpPvgD7bjUD5Jjh8/AAAAAElFTkSuQmCC',
vcard_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKZSURBVDjLpZM7TFNhFMd/t/f2IqVQqAWM72IVMUEjIRoiYnTxEWEyTjqoiYNuxkSjk5uJg4ODDjoYE6ODm4sOJlopqNRY5VXC09oCRaCg3t572++7DspTnTzJyTnfyTn/739O8lccx+F/TBsdHb0MHAOQUuI4DlLKJS6E+CP+9gdKKpXKBwIBFWAxm7n8b3Euj8ViQnMcR3W73dyMCmzjG9PxVzi5H7jKa6gI1nLE208oFOLy8wyGaWNkbQwzx+PTIYQQqrb417reW+RT7xhJJBieMHCufgQgl8txbV8hUhbMrwUghECbewDkKnfStH0NB3SN1o5OYqo63xgOhymWXQQyHajeWka+vsRdth9NCPFrOC95m16Npk3jLSkhau9masoE7y+A+tA0+cQEhetO4AvuJDNUTc+LhwsMMok+yoNVPNHqmPpss8Kvs+pHEgAr/QzViuPfvIepgR50xaa4ZBXe0soFBmuKZumaLEX6Symr1DFnTYrlBGq2G83di6/qINboI3SPwsiHXqSjk/Q1LgCcP9wwfwvDMLAsC2syQYHZiW9TC2byDi49j9u7gSLnC4FDNxho78Y1B5BIJIhGowwPD+PxeLDGwpBpxRdqwUzexuXOYc9uZOzle2aqTlFYvgkpJUosFusWQtQIIaivr1cikYhjj7dR4Rlna1Mz9vh9FNXGnFlLOvweacwE+7ZcGfp9ux5luRbunVt/pqH55N28UsFKfytlFTrmzDomX79JSyvbUH2hbXCJFpaLo2TjlrvbGs8Sf3SRvnCEgvU7yKfjqTJdPVh7qX1web9reSHeP5a3u54S3LGXoqJqkh2fvptZ+0jtpfbOv6nxjxWON/mzdVWV2q6aII7bimTTE6eOXv84+C85/wR0RnLQ/rM7uwAAAABJRU5ErkJggg==',
vector=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAFzSURBVDjLY/j//z8DPlxYWFgAxA9ANDZ5BiIMeASlH5BswPz58+uampo2kuUCkGYgPg/EQvgsweZk5rlz5zYSoxnDAKBmprq6umONjY1vsmdeamvd9Pzc1N2vv/Zse/k0a/6jZWGT7hWGTLhrEdR7hwOrAfPmzWtob29/XlRc9qdjw8P76fMeTU2c9WBi5LQH7UB6ftS0B9MDe+7k+XfeCvRpu6Xr1XJTEMPP2TMvlkzZ8fhn9JSb+ujO9e+6ZebbcSvMu/Wmm2fzDSv3hmuGsHh+BAptkJ9Llj3e2LDu2SVcfvZqucHm0XhD163+mplLzVVtjHgGar7asO75bUKB51R9Vdih4ooqRkprXPfsXsfm558JGQDCtqWXmDAEi5Y+PjNhx4v/QL8aE2MIhkD8zAcbJ+189d+z5UYOWQZ4t9xsnLjj5f/A3ltLyDIAGDXe7Zue/89b/OiZY8UVNpINAEaNUOWqp38qVj3+DwykQEIGAABS5b0Ghvs3EQAAAABJRU5ErkJggg==',
vector_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH6SURBVDjLY/j//z8DPlxYWFgAxA9ANDZ5BiIMeASlH5BswPz58+uampo2kuUCkGYgPg/EQvgsweZk5rlz5zYSoxnDAKBmprq6umONjY1vsmdeamvd9Pzc1N2vv/Zse/k0a/6jZWGT7hWGTLhrEdR7hwOrAfPmzWtob29/XlRc9qdjw8P76fMeTU2c9WBi5LQH7UB6ftS0B9MDe+7k+XfeCvRpu6Xr1XJTEMPP2TMvlkzZ8fhn9JSb+ujO9e+6ZebbcSvMu/Wmm2fzDSv3hmuGsHh+BAptkJ9Llj3e2LDu2SVcfvZqucHm0XhD163+mplLzVVtjHgGar7asO75bXSN+VMia/KmRHxK6/P/H9ni8MmjwqrNoeKKKkZKa1z37F7H5uefkTVn9Ac2NK5O/L/lytT/F57t+t+/O+t/eL/uf/NsqV4MJxYtfXxmwo4X/4F+NYaJxba7fN94ecL/jdcm/QeBnj2p//v3pAMNkPyOYUD8zAcbJ+189d+z5UYOTMyn2vD/titz/iODTZemggzADCTvlpuNE3e8/B/Ye2sJTMwyR/p7966k/+27EsCa23cm4HYBMGq82zc9/5+3+NEzx4orbCAxoMKW4B6N/727UsA2g2gQHyjeg2EAMGqEKlc9/VOx6vF/29JLgTBxoOIOIP4EcjaU7gCJAwAM9qYI32g+agAAAABJRU5ErkJggg==',
vector_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAINSURBVDjLY/j//z8DPlxYWFgAxA9ANDZ5BiIMeASlH5BswPz58+uampo2kuUCkGYgPg/EQvgsweZk5rlz5zYSoxnDAKBmprq6umONjY1vsmdeamvd9Pzc1N2vv/Zse/k0a/6jZWGT7hWGTLhrEdR7hwOrAfPmzWtob29/XlRc9qdjw8P76fMeTU2c9WBi5LQH7UB6ftS0B9MDe+7k+XfeCvRpu6Xr1XJTEMPP2TMvlkzZ8fhn9JSb+ujO9e+6ZebbcSvMu/Wmm2fzDSv3hmuGsHh+BAptkJ9Llj3e2LDu2SVcfvZqucHm0XhD163+mplLzVVtjHgGar7asO75bXSNRyLkKg748j3c48Tyb6cr86MNnsJNDhVXVDFSWuO6Z/c6Nj//jKI5XK78YrHFz+9be///u7bj/9cVRf9PZ+v+2enMlofhxKKlj89M2PHiP9CvxjCxnS7Md78BNf+f5Pv/f7ng//9tiv9fdzn8B4rfwzAgfuaDjZN2vvrv2XIjByYGcva/s+v+I4P39RL/QeIYBni33GycuOPl/8DeW0vgLnBlfvxlbvL//0BNP8oY/r8D4ocZzP+B4k8wDABGjXf7puf/8xY/euZYcYUNJHY4XKrhZIrq72fliv9fVbL+v5vC+H+vL8ufHa7MVRgGAKNGqHLV0z8Vqx7/ty29FIgISNkKoI33obHwGKQZJA4AVQ2j4x4gIJMAAAAASUVORK5CYII=',
wand=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHMSURBVDjLlZLBSyJhGMa/UxTUIWJ0ZVmlwxLLEiEhurCoKeqCOtZN7J4ZRZdd9rSG6NFbSOegDp5aqWWI3UGm6KBUxsq2LLj+CzV9jDOH8NlvJtqLjuXhBy/z8Xvel4chAMhTKGfOMeVsbqXf2wBp3s5Yf5hno8rp24YxS9PTVHq18mTAgzj3k4mCIs0cqZeLUCTHJ1q13VKRSz0v4PRNVr1KQfu9Aa31BZ2LKKg42aHfJ8ZNA9i5L9hWUZFeQ73kof3N42SPR6OyjFZ1FZ36AuQfo5CPyc7gDiRHttNYwsl+Apqmodvt4uJrCur1GmSB/GI4TAOo9JKjVasQi8VQr9ehqiqazSaqu1Fofz5C/kYow9M3gJVkp+JUJZFIIJ1Oo1gsolwu42hngcmfdfmecS4fki3TC3ieN2SPx4NAIIB4PA7lPIo70YY7YQJyhdhNS3yU3W43/H4/LBaLvnWbbbxnvGNyQz4gmb4ByWQShULBkH0+HziOg/6die+ZKOjzzQEZYXzoCYhEIsjn8z3yI0wKmf7KwWAQuVwOLpcLXq+3Rx4EyWQyaLfbcDqdCIVCQ8n/A2q1GkqlklHYMLIREA6HN/WzrVbr0LLOP1AMs7UPAa92AAAAAElFTkSuQmCC',
weather_clouds=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHXSURBVDjLzZNNi9pQFIbzA+YXDP0zLV3Nb3E9a3d1JQh+g7oQLaKCimL8QGKiMdF0OjUTjB+N0fi9Ghim7aa8vScwglBKabvohZfccM95zntObjgA3N+I+2cARVGuJEnydNjief5LpVLpFAoFTyaTufotgCiKtw8POizrMzaOjfnMhCz3kUgkbn8JkGX5utvtelut1telNYf+ScPHDzL0+yEW8wnC4fCT3+/3+Hy+nzrhBEHwTiYTvCRrQwma2sVIFXCnDaAqA7TbbdRqtcdSqZTIZrOvLwCNRsNY2RbGrKI2FN1kddCB3OtAFAU4joPT6YTj8cjas5DP58epVOrtGcCGZVD1+zuFJYusYh/9noQe03a7xW63w3q9drXf77FYLPCerTOA7b00LMMYYzRS3YDD4eCKksmBbdtYLpfuk5zkcrnvyWSyFAwG33DMzjUblJcNymDtfKMAqkbBlEwu6J0AJNoT3DRNRKPR6sVE2RUwCUCJq9XKDd5sNmfAixOaBbUTj8efLwD1ev3dbDZzDymR9tQSuSAgfa3pdOqe6boO1gJ/AWA371W1Wg00m801gznlcpkvFoutdDp9CoVCx1gsJjFpkUjkORAI8KztG+7/+Zn+VD8AV2IaSQGFiWoAAAAASUVORK5CYII=',
weather_cloudy=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJISURBVDjLpZPfa1JhGMftuqvC41l1Hf0JrTtvuyropp8QOdCFWYJgoNC0yOwiiTFtYGCIhBXTs61yc7ZdzAhHiLpTFi21+fP4Y66ZLld9e1/rCIdVBLt4eJ/n8H6+3+9zDkcGQLaTkgzt+YEDYt+KsI/Efj3M7v4vgdaLgeMEbG/Msfs+h1nQZ83ZX+c/BQh0aCPCCrSn4Pos++NL8gzWZtj3jZCiJ1B7pghXnyp2/TUBiVmjbhTcKo+ju3ob3cJdEEgQphWoTCkm/5iAgCoKErexzoer+Jq7ic7bi+jwF7D5Tofup1toLp1AiWNUxSBzuBBg9mxLQGKyjchB4jhK4GF0ls+jkzqHdvIUmYfQyV5HPsB8W52Qn96WgOx2jMRstJaHifuN3/BZAp9E5fUV8C/HsBDh8Jx7sDX15F7Q5/MpJQJkv71kP2V5klnr5u9g880Q2gkKX8arhYfIZDKo1WqoVqtIp9Pw3HfxLpdLKVmhyDHXCkEGwpIKmZQPsUUO85Fp5HI5NBoNCIKASqWCer2OZDIJh8MxLhHITzCj9EzNXMLKykrPkV6mZ7lcRqlU6hXtqaDNZvtusVg8JpNpsL9L9rH86OKctx+XOoogrWKx2CtRJBaLwWAwePoCH/3yI6FQiKewKECj06KQWGISaqTT6ZqST8Jx3AjdkV6gbqlUColEou8ej8d7MzWIRqPQaDQeiYDf79/v9XpH3G4373Q6efKyPHa73Wu1WrNmszlrNBoDer0+pNVqm2q12qNSqQZlO/2dfwL4RvrQAqV2MgAAAABJRU5ErkJggg==',
weather_lightning=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAITSURBVDjLlZLNi1JhFMaHiP6AvN0Ls4vob2jauYiIFoGBtnFlG1fSws1gDBdXLhIhdaEuvILfjtfK0cyvRflBeHWUEBW1nHGKpmkcpWw+ani6r9DCutn0wuGFl/P8znPOeZcALP0ekUhE7vf7HR6PZ+ByuQZ2u91hsVjkUrl/PITDYbnP5xMajQb6/T46nQ5KpRJMJpNgNBrlkoB4PL7M8zwbCoWaXq93RMStVguVSgXlchmCICCXy8FgMDgkAdFolK1Wq+j1emi326jX6ygUCsjn80ilUkgkEigWi9Dr9ac6nY7TarUrc4BAINDsdruo1WpzQtEZRDiCwSDE1pDJZBCLxaDRaLg5gDispnhmvRKrJJFU/SUWBwqO4+B2u5HNZqFWq8dzAKfTyRIh6ZVAksnkrDpxkk6nIW4F4nxmrghMpVLNO7Barctms5m12Wx46bw23XRf/TF5JsP4qQwHT2QYRWXYW7+Ix6vXT5VKJadQKFYk1/g1x5z/kmUU0+otnHy04Hi4hu8HHD4n6elegr7/z38wyTA3xy+Y7mHvAb69UWDauI0PiSuQEkoCRil663CwhuMddlad3EfbD/F+4zIWAvaf0+dEm48+bdDYjdMYC3dn4snmvYViya9MYoe/NNx/fQdb69R4EKGYMwOGPHVhO0qt7r66gXdhKrJIKAkQq6nehqijflCmOov4ry38T/wEpFjkJMz8PioAAAAASUVORK5CYII=',
weather_rain=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIESURBVDjLvVO/a9tAFD5ZdWwZg11w2iR1idMhxhm8ZOlS3K1rlwwZRD2EQCd7KIUOpmvoj7E4BBxElw72kMFLh5osBkEp3fwXlMY4TlRJtk4/7kffqW6wcDMFevDxTu/u+/S9d3cS5xzdZMTQDcet6xY6nc7jIAh2AU9830eAz4BP9Xr9dH6f9K8S2u22IL8rFovb6XQaEULQeDxGuq5/A5EXjUbjdMFBt9tdA9I+YAewWiqVbieTSWRZVigg5uVyebvf7+/C9kUBUN7P5/OvM5kMopQiz/OQYRhoZj/MpVIpkd+r1WoJyB02m019XmBH2J1OpwhjfEUEN1fRtm1UqVRipmk+6/V6ghYRCCHIruuGfxQk4URE8S3WJ5MJyuVyYv40coywsT0cDv+cbSyGHMcJhRRFCcEYQ5IkoWw2i0ajkRA4ifQABI4Gg0FYyszNV4AMeDQr4TtAATwEnEBjDxeOsaadvYnJSGEUYRFdj2PGmTLxOSaEKZ7LMCVccWzy8svBJo6U8Pz458pWPlF1A97aXE1UL2zS2rgbr54bQevBnXj114XfKkDevPQO/pIjDuofz94TymU3YNQnXMYeozRgUAKjxGdyABH6KLsOfaV/2MKRt7B39OPe+nJcPbeIVlheUg0j0AorS6p5GWj31xKqZRJtfSOlAvntPPnaq/xfX+NvE6ltVjnyz4AAAAAASUVORK5CYII=',
weather_snow=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAQAAAC1+jfqAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAADnSURBVCjPY/jPgB8ykKTggMPemTsfbHmwfuZqBywKDjjsOXP//6f/H/9f/7/0zAIHJAXHpQ437L+6+93j/9//fwHDC/9nz0RScKjhBlDf5/9fgRhkwgcgnvZ34oIeC6iCvVffwyXf/3/3/y0Qvvt//H/7AqiCnVffAqU/wqXf/H/9/9X/l/+bP0AVbG449f/F/6f/z/0/AyQf/z/9/yyQf+B/HcyEdVIrG5ZcXXB19oJpiyc96H3Qub51R+OH2gVVFnBv+uj6+ProYtIIBb6f/08v8OFHp9FMQMdoIYmpG10Bhv1ExSYAuRQri6NL7YwAAAAASUVORK5CYII=',
weather_sun=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAIBSURBVDjLpZPdS9NRGMe98zLw/H47QxDCLoqgC/+E/oJpL1qtNxMlRkhIYO7CoJuwQLqILoPwpWnkluKknNWVXVTWXnzZ3Ku65ftsm7/z+w3p23NGQbFZji4enodz+DzP9/twTgWAiv+JkofinblS5t235vbdN2a17AYEQebcFEfWw+/I+tskxz8bEOj5DQwI/0UQOL7zmu+lX/FNebc1YTq66TbVlGyQneLfMx6OzCQP5VOPIEMPd0JP3AOBLgKxMW5CkQKaxgm8JWuahvzXxwR2QczboC/cgBGxQ4t0Y23MhH0tSJBkIue1wojfhZhthQg0Q/gvU1yCEbVj52MjUi4VSaeK5RG1ssgCyUwXZNNUEbhaADXfeWjes6TmGnLBDgIVZ5EC8uaW3jIzF5BP9kLMtUEuUPOdI/gMtC8WUmQlG7ex8d6C+HMFsWGlqkjB6qj6MOu/Dj3YTqCETxdgzVtPe7iJ9WkLokNKS8TB2sIOdviPBqmXqjVJ/rY/NMFYfkBNbKSiCcJ3CvqiDVufriDuqkX4GUPJJa6MqE9kXnqh3E+6jyPtJRvRLhgxO7Y/tyDmrMXiIKsODbC+4AB7uu9LJG9S5pHE6DGQzMTadANCg6yHQAT7meOvL5HAE+TvkKxpWkEqTdMX+lm3rOf7qoYP9Bd+gr+gOoqTZX2mcuIHSo3eNm+GAIoAAAAASUVORK5CYII=',
webcam=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJqSURBVDjLnVNLa1phED1X1CS+Aokv0Ks0D2jQnbRJi6UJLdidBLsvXfQ3ZFnaXXfd5Re4KiG4DNKNaI1YUVDoUryVxIiPGOP71ZmvGBKkm14Y7nyXOWfOnG+uNJvNcP+pVquySqUKTafTwHg89lFgNBplut3uWavVOvH7/b/v10v3Ca6urp5JknSk0WiC9AYRgQkGgwFub2+hKErk5ubmy+HhYXKOUc2Ty8tLmboeLS8vB5eWlsDEmUxGxHA4xMrKClwuV/D6+vro+PhYXiAgmSG1Wi06M4BJ/P4XeLm/D6PRiF6vB1IGh8MRrFQqoTlOPU/6/X7AYDDwW3T7VaziPPlDKNnd3YPbvsr+wGq1otlsBgjy9YECmtPHxTyvlrqfJxMwuZ7A9OgAP4tjmEwmVgm9Xo9Op+NbUEAuC8NYPrXFdAoMocNkosVQWhU1k8kEZCITYMEDIsi0221RRNeF53tP0Wm30Or28erxDPV6XdSVSiX2I7NAQO6ekTkiTyQS0KCHd68d+HBgwKhTQSwWEwqz2SyPebYwQqPROCmXy288Hk+AbgO5XA7pdJpGmYp9YGP5TATf6dvJggIq2CWDbPF4PMLnnZ0dbG5uYmtrC16vF0SO09PTyPr6uo125f2DTQyHw28pD5PL2nw+76KlUrgrz82eaLVa1Go1kDKX2WxWiASpVOpjoVD4LEYgSd+2t7fBHsiyrNjtduh0OlxcXIjb4QUqFotsnrK2toaNjQ1Eo9FPBP1LwMvDAKfTCYvFIs4cNpvtLne73eKKeRfYFzb0zkRmJ0axRBxzEMf8G4MZxOB//o3/8/wBTrxxLgOn/DMAAAAASUVORK5CYII=',
webcam_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAKkSURBVDjLjZLfS5pRGMf7N3bxslsvtsWuJBi0i4UNRrtYIeIumuwHbAgvXa0mMWGjMDebmSzTQtNuXGZLZr5ZXRQUJjm1jWmGhT9Wr/1UQnQX351zNkbt3cUOPBzO4Xk+z/f5ntMAoOF87O/vc6VSiSd7sFAoiLu7u2ImkwkmEgl+eXmZ+zv/wmFvb6+JFPqPjo5wfHyM09NTHB4eolgsIp1OIxwO+30+X9M/ASSJI+GnRWdnZ6CQxcVFCIKAnZ0d5HI5BhkbG/NbrVZOAiBSeVEUUS6XWXcKqdd/oFavMxW0OJvNIhQKobe3l5cAUqlUkCZSSKVSQXQzC8uoB0M2NyLxDMhoSCaTiMVi0Gq1QQmAmCQeHBwwqbSrZdQNVygN53wBVv83VKtVbGxsMCUajUaUANbW1kRiIra3t1Gr1WAeccMeymNkroTBme8MsL6+jmg0CqVSKQUQw4JbW1usAx0jGk/j3YcvMPhySKaKyOfzDDA1NYW2tjbpCOR56DszwPT0NOLxOE5OTpihVPrk5CRWV1eh0+mgUCikJtrtds5sNs8tLCxgdnYWTqcT5A42m43tA46X4C1qPH57F6pXNyu3n1/TXQB4PJ52h8MR0+v1fgqIRCJYWlpif+G91wi99wECm8P4XBRgmn8Kpekq5M8uGVmx2+1uJ1EbHx9HV1cXp1KpoFar0draipaWFtx/fQszyUHMfDWDLmP4EUzhJxRQZQCXy8Xmo7ObTCYYDAZYLBb09PSA53nceXEdnzbtOL8+JoYp4Jd8Oic1jbq8srJC/zwCgQC8Xi8mJiZwQ3sZA4IGfUInK+4LdV5U0N3djY6ODvo81GE0NzdDLpejsbERMpkMVxQc7hlleCM8ZJ3pTs9/PPifIMn9JMpU9u+9n97/BG848JbqijsUAAAAAElFTkSuQmCC',
webcam_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAK3SURBVDjLdVM9TFNRFP5e+0p/KcQIrZZXYCCBdIMoaGqESGxCTBqCg5suxsRF44IjgzG6mZjYwTB2Mg6YOLQdNKUFTKwYEJ2koYZSqi20j9ff91rPvbEEbXzJyb03Oef7vvOd84Rms4mTXzablXQ63Vyj0fCpqjpGgXq9niiVSqFCofDa6/X+OJkvnATY39+/IAjCvMFg8NMJAgIDqFarODo6QiqVWioWi09nZ2dXWzW61mVvb08i1nmTyeQ3Go1gwIlEgketVoPZbIbb7fYfHh7OBwIBqQ2AZM6JosiZWQED8Xov4fLkJDo7O1Eul0HK4HK5/JlMZq5VJ7YulUrFZ7PZ2MnZviWzWFtd4UrGxyfQ7+xi/qC3txcHBwc+Knn2lwLqc4wls347iH1tNQ67+xzsg1P4mFRht9uZSlitViiKMtamgFzmhjH5RItGA6jBAk3rQE3o4jmapoFMZABo84AAErIs8yQaFy5OnIciF1AoVXBluIlcLsfzdnZ2mB+JNgByN0Tm8Hs8HocBZdycduH2lA11JYNoNMoVrq+vszZDbXuwuLgokYqXHo/Hx9rJ5/O8Zxot3wfn7gcYv4Qg5NJQ9UgLaD6/GlafHHtAzo/TCB2xWGxpdHTUPzIywntlBKatCMzyFoZv3YNx0IPyRvjs1+XIo8i0QeEKgsHgdcIIEmPH5uamm5YqxVhZ38yT21jDtfsLMH9/D+zGgK5u/BL78Sm8nOQKSOaroaEhMA8kSUo5nU5YLBak02k+nVMb72ByDgIzD47dFxfOQN8QBsQ/S8QL+vr60NPTw98sHA4HP2vb3Sh9fgvrm7uoljMoUY1c1EMjLzhAMplEJBLhS8SiBcCCvWdOm9G9EsUAnaLeAPmniu0M2YjmC+Hf3/l/X/yG+6GST9/Ra0K/pm/uUlXAF1Yf/wakxYbML/JgHwAAAABJRU5ErkJggg==',
webcam_error=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAALHSURBVDjLbZNdSJNhFMf/m/v+atayydzEFk5rlbrmLBcpRV4OW9hVdNNtV14IuxQqIoLuvI1ouykLE4rdhnNivTmKikywbaBurM19f77v2/O8MJmtBw7PA+97fuec/zlHxPM8Wk8ymTSLxWIvx3FTjUbDQQz1ep0plUrBbDa76Ha7463/i1oBiUTiokgkmpNKpR5yg4BAAdVqFYVCAbFYbCmXyz2anp4ON33Ezcfu7q6ZRJ1TKBQeuVwOCmYYRrBarQalUgmLxeLZ39+fW1hYMLcBSJpeiUQiRKYOFOJ2X8aViQlotVqUy2WQzGAymTx7e3vepp+k+ahUKlMajYbeQrQf20mshVeFTFyuMfQaj1B90NXVhUwmM0Vcnh7KgNTpoD/TemUk+lo4BJ3FCV3fJD5tN6DT6WiWUKvVKBaLjrYSiMqCYDR9EhYcB9SgQoWVwSCLIvX1AfhGHkRECsD/AEw+nwfLsiDtwqWxURTzWZTKWQwr30PVeQ7y4jqi0SjVg2kDEHWDRBzhHQqFIEUZd66ZMHMqDL3xAnTdwxDnGMR+rtIyg22AdDq9uLGxEYzH4yDdQCQSwbs3z5D+tQytQQ829xZG+21YVd9TBg231AYgyruIQCdWVlaEj4ODA7Ad20H3mRnSos9Yf+6HRl+GWVczjPSJHx4C+P3+m6QDfjJ5Q6T+e4FAAMuBJziqlkLbWQJX3SbCcmALH9E3MQu7sXprfPTs/QMAmcCXVqtVptfrYTabY46R87huZ6G3DIErfSG+ZTi8NnC1HSgUmxhw3sCope47ANDhUalU6OnpgdPphOskYOkfh0qbIq37QzamA8zrTRoKXGUTx/t7cdUmwYd5+2lhmXw+H7+1tSUMEbXZyRzcd1+gQ/ybADLEsXVjxehQ2pD4FsHm8vwr0b/rTM/qY0eKZzkVz/Ekfa7F+IObThrxZf4CSlpy3yYZtPsAAAAASUVORK5CYII=',
world=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAMtSURBVDjLVZNLa1xlAIafc5szk8xkMkkm5MKY2EpT2qa2MTVCmoLS2gq6EKooimAW7iQb/0I2bgTRIog0oFW7KQpCS7VqrSmmJGlSQtswqWlLLmbGmcmcZM6cy/edz00r6bt8eXh4N6+mlGJnxiZHR4APgSNAFjCBKjClInXm05Gzl3by2mPB2OSoCUwAp1/LHbcziSyO24gbgJAegg2urF8UUsifhZBvfvXK99v/C8YmRy3gt8G2/cMv517E8Wx8ApYcjZiyKbkRSgQkcFn3rzG9Nn1LhOLYt2/8UNUfLZkYaN0zfLRrkLIMCHUNIXTqIoZLjLJvU/ASrFQtnko+z2BH38HAD78DMConHh4FPn5nz6vGgqyxTp16JNj2kpR9C8eD/OoW1VoNO1NCS+d5oW0vV27f2PX11MS8MTR6+JOTXUMHNCPBui5AtdMpk8xsGNQ9ndur20TxCnbPIn5TnmJUwaxIDrTm9Jn7d1tM4EiuqZs5d41iXGefsZsIwYNCgOfVSXconJbLLEWb4CuahU2+6HO8d4DQF/0m0NpgNvLAXaPgu6QadrEZpKhUItJZj/aMS1EewvHnsdUWW/+WKG82kEykCAPRbCqlNE1B4DsocpiW5OJfIVoiyfqSQFdNdGXrpLZGcFZDPKYJg2VQCiGEZkoRlZ3A6W41mknFn2WlaOKFFrG4Tbw9wb2/S3g3miHySLdbNDd2kzYKVGpVpIiqugjF7P3yQ55pyLFWmCSyVokZPqHnEoYmsWQGuyWOGdexNIkRFOnqbGN5bRngjh4G4rMLd6+KnmQW012lWrpOJuNjCh9LU9i6gRkEZHIrpNv/QK8vcijXz5lfLijgS+PmuYV75+fPDXr1Wt9znfsouy5x+2miuoltW1iawBJV0o0/wT8lBvbv5WZ+gaWNlasz43MfmQChH777e37uT78eHDx5+BiLBROjqhDaFmGkQ1KS6+mlr7+XX2evc+nWVB54+4kznfr8pZQIxXkRyhPvDb9vIjtQqgFN12hLO2yUZ/ni8o8SuAa8NTM+t/GE4HGGx4del0J+IGXUH8ko86iuAneAszPjc9/s5P8DuO6ZcsXuRqAAAAAASUVORK5CYII=',
world_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAM+SURBVBgZBcHLT1xVAMDh333MkxmYAYZOwbGGPqRaHq0gpIBGba3GR2KibjQmsHDXsHDjP2DSmBgTU7swUUk0ajemK1HapjYIoYRqi4DoUAsNjwIyw9wZ7tzHOff4fZpSCgAAgJHJ4UHgPNADZAATKAG3VKAufTb49c8AAACaUgqAkclhExgF3nwtdyaSjmWw7BpsD4R0EGxxfXNMSCGvCiHf/vLlHyoAmlKKkcnhEHCju/HJ/hdzz2E5EVw8li2NsIqwawco4RHDZtOdYHZjdk744pnv3rpS0gGA0VMNx/oHmrspSA9f1xBCpyrC2IQpuBG2nRhrpRCPJp6mO/t4h+f63wMYxbMPBoCP3zn2qjEv99mkSjUQVJwEBTeE5UB+vUxpf59IehetLk9fYxvXF2dav7k1etfoHT756bnm3hOaEWNTF6CaOCgT3N4yqDo6i+sVgmiRyKG/cWvz7ARFzKLkRENOv72yVG8CPbnaFu7YG+xEdZ4wDhMgWN32cJwqdVmFVT/OcrAHriIlIuR3XM48dgrfFe0m0BA3a1i1N9h2bZLxVva8JMViQF3GoSltsyO7sNy7RFSZ8n+7FPbiJGJJfE+kTKWUpinwXAtFDjMkGZv20WIJNpcFuqqlOVMlWR7EWvdxmMX37oNSCCE0U4qgYHlWS4ORIhntZG3HxPFDhKMRok0x7v27izOTIhOeIROdJ+JZlJ0yY1O/IEVQMoUvfl8pPGg5Es9x7eEkqfgRwkYO37FRRopwIk2tO0FbdomnjvfxSP1RbixcYXp+AqNa8XTfExd/XLopDiUymPY6pd0p0mkXU7iENEVENzAr1+hq60Tqks6DZ5GaT1/7aXTPyepfvXJ53HP9n8YXb/JsSxd1Rg3pREBdWFIbdkiGXIqVLUJagtePnwfggxe+4HBTB0BIB/Bd/91f83fm/lz5i3NtPbSmTA7EFY1GmQbdplgusrAxxYWrQwBcGB/i3vYcgKMppQB46fPnk8IXl4Uvz77XP2QisygVR9M1Fv75ltXiFKc7BjiaPUn+4R9Mzf3G2v3SJ5pSCgAAgP6Pet+QQr4vZdAeyCANAJSMatnTveoBIAqUgUvTFzc+/B+ww5qo63KzbgAAAABJRU5ErkJggg==',
world_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANDSURBVDjLXZNLaFxlAIW/+5hn5pkXMWFSU0uN9mFaUlNIUqmkjVIFFRVEUczChVKyduNOETeC+FipEcXajS0VrcQWW2zEllSSUkNK0pjUJNPJZGYyd5I79/H/93dTS/EsD4ePs/k0pRT3ZnRiZBA4DhwAWgATqAKXVaA+/Wjwy5/v3Wv/AUYnRkxgDHj+6dxQJBtrwbIbsD0Q0kFQ4Hz+rJBC/iKEfPHzJ7/bvAsYnRgJAb/2Nu/qP5o7jOVEcPGYtzTCKkLJDlDCI4ZN3v2NydXJa8IXh7594XRVv/NkbH/Tzv6B9l7K0sPXNYTQqYswNmHKboQ1J8ZyNURn4lF62x7c67n+CQCjcuTWAPDByzufMq7LLfLUqQeCTSdB2Q1hOTC3UqO6tUUkW0JLz3GwuZvzM1e2f315bNroG9n34XB7327NiJHXBahW7pMJrhYM6o7OzMomQbRCZNsN3NQcxaCCWZHsbsrpVxdnG03gQC7VwZS9SjGq87DxAAGCpTUPx6mTblNYjePMBxvgKjIiwlzRZej+/fiu2GMCTXGzgSV7lTXXJhnfzoaXpFIJSLc4tGZtirIHy50momrU1kuUN+IkYkl8T2RMpZSmKfBcC0UOMyQ5+4ePFkuQnxfoKkV7S51kbRBrxcdhEt/7G5RCCKGZUgRly7M6mowMyegjLBdNHD9EOBoh2hrj5kIJ50qGwcJpjuZ/ILb+D1upGKUNiRRB1RS++HOxfKtjRzzHudsTZOI7CBs5fMdGGRnCiSwHF09xLDTJQ6+9RaRrF/Vr4/x16RxDVtHSfU98/P3sRbEt0YJpr1At/U4262IKl5CmiOgG3de/ofvwc0RvXkD76iXiC6fo6mxkoLCe0b84dnLcc/2fxmcu8lhHD2mjgWwiIB2WpMIOyZBLdrNAtK0Ljp+B98vw9gLm1ixxXzWbAL7rv3JhbuqSW/f2Du87xI01E6OqEFoNP9CxU43YUz/ScOZN3PptbKBmGUiD1bsyPfHJ40nhi5PCl0de7X/dRLahVBxN18hNnyA1fZHORh9TX6ZWFCwWDOk76h3t/zr3v9v3rBTyDSmDPYEMsnfq6jMl2+5ZWkzHPdUkDbWi4LPhcfHevzBSqkykNJyOAAAAAElFTkSuQmCC',
world_edit=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANDSURBVDjLZZNvUBN0GMf3ot7ldb2q63UvuuuFd93Vq/ROMYyu9C4CCwE1uqHFOD3D1SxPUFCb5OBwBwMJMf5GjT/bMBhjILjGNraxv47JcGrIakIYAxT89BtXHdqLz7vn832eF99HAkg2UjhyYIugXRAR/CVYFswKumXD+995en6j+Izgh6TQf6sJe6wP0/QoutAonYEBOgLNHDLlPJL2ZRk+Mex57okAIT0rGGkMnufeooNQ3IsnPo424kQ/5afR6+Wya5x21wgV1jPkaNPdH7fvfn5jQFND4AxTC2b8CxZ8cz6csy6aJ338OBVG5ZhE+WuA74asaH0DVFgUpF9517AeIOS3BA+jD8zo5vR0zOvpjQ/Re9tLXfAGFz0hZHoHsh4Lp216NJEqHDNa3q9NXU1Tb9+dDPhJP1WN/Xc9LXNdtN23Yo25KXf6UNpuINU5yDcaOeLXoHYeJNj0EZNdpbgupVJS+YYnGRCd+KOHxmgt5bFLYrsXS8xF4S9OpF12iqw28iePUeLYxYxJzvyEDh7cZW6iA+P5nWvJgMXwvIlTwWJkk3I0dwdpidjJaR2jYGAYZegqSruCmO8EK7e7mTErSXh+ZnV6hPH6T5HIru1P3Jwz8Y37KAVuFfXTRnJbLOR2ekivuc6F/mri3q9ZS4yQuFlAfExG4EoeEw15vFf85rLks8HcO87ZTlTeU3wbHqTEdo3MRjvZ2hAntc3c93zF2pKFxfA+lqJ7WfAr8GtSMI+peF2x+Z5EFKNbG6hEN9XA4eunOelqJbtpFHltJTM2+T9yLku3svjTexy/+m2+139Onak4GTAoOaDLTBXFeOSJ6fnCeIgjA4WUNp3Fc7VInD3MUiSP5WgO8+KSgJCLLmZiDjcn5ceCrPUi7Wnd1VVuPkZPoAZlfQY+gwJ3r4o71gwWI3uJu77Epd6BXJ1BzVAxx9vy17f/18QPLqdtEsVwl/Uepv1cCixME9R8iEnxMqPVafSVbafBfBRjqAFFqzQphwQvPfFMO6u2bUq5sMVQIn2NFUsVK9YKrKXbaDv4Ct3D1WjMJ5LianKz4MX/feO/ZGx94WHZvlcf959No+7c1rUdis0JIfwmGBBkPz3/Nw6S0St8tXZAAAAAAElFTkSuQmCC',
world_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANCSURBVBgZBcHbT1t1AMDx72lPKS29UCiz0BUMQxwwJptMSIAZzRgu6oOJ+jKjkQdjTAx/gI9mezAmJkb3ppKYqHsxe9oMOh0ZODd3xYVtFOLK1dG0pYWensvv4udjaK0BAACYmp8cAz4GjgEtgAmUgeta6XNfjn33CwAAgKG1BmBqftIEpoE3X8+cCCZCLVSsBiwXhLQRPOHy1iUhhfxVCPn2N6d+2gMwtNZMzU8GgD8Gk30jJzMvUbGDOLgsVwzqdJCCpdDCJYTFlnOVm5s3F4Qnjv/w1oWyDwCYPtrcPTLaNkhRung+AyF81EQdFnUUnSDbdoj1coD2yAsMpp497DrejwD+0vjqKPDZ6e7X/PdllS1q1JRgz45QdAJUbMhu7FKuVgkmChjxLMPJg1xevNH5/fXpe/6hySNfTLQNHTL8IbZ8AvQ+WmWEW0/81Gwfixt7qPoSwY5HOLEseVXCLEkONWd8tx4/bDKBY5lYmrvWJvl6H73+AygEuW0X264RT2kqTTMsqx1wNI0iSDbvcOLpo3iO6DeB5rDZQM7aZNuxiIY72XGjlEqKeIvNvoRFXg6QvnMOaVfJZw5S3AkTCUXxXNHo01obhgbXqaCtVkxPcukvD6M+xNayydpqjDYnhPA0+5M9BJfv4Nk10BohhGFKoYoVt5Ju9jcSrX+O9byJ7QVoVR8RD0ucDY/dnCDd1EVPaohdu8rC+u8UqxNIocqm8MTtx8XVdFc4w2//zdMY7qLOn0Eol/G+95BaIZVEodksr9G/f4Q9t8YnFz4Euh/4PFd89fPDWdERacG0NigX/iSRcLCFi9SKXCHLv4UlVvKL7NQK5IorDGTGeCb1PLuBe6O+b189P+M63sWZxVleTA8Q9zeQiChsYSOk4KlYO6lYB63xTgL+EC3RNLfX5rm2csOyXGImgOd471zJ3p1zau7hiSPHebRt8o9wmL72Oa5ysYXLgWQvw50n+Ts3x5WlWScs23uWz2ZrhtYagFe+fjkqPHFeeHL83ZH3TWQKrcMYPoNkvMKnF0/T1zrM1aW53Qbd3rtwZmkdwNBaAwAAMHJm6A0p5AdSqn4lVQIAKO/47yeFIlBTMrB9VgsAgP8BON24AjtZfcoAAAAASUVORK5CYII=',
world_link=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAANPSURBVBgZBcHdT1tlAMDh3zltORT6Ob4mtWDGMpgiU8LcEooJyiaEGbNkCkaNCVfeGP4Dr7zBG42J3hiVZInTeTMvFAPBYRhmGDBjEYaAMhhtVzraUjin5+M95/V5FCklAAAA4wtjfcCHwHmgAfADh8Ci9OSXn/d9+ysAAIAipQRgfGHMD0wC115PDmjxYANloxbDBuGaCHLMZqeEK9wZIdy3vh76/hhAkVIyvjAWAG731D/XeznZT9nUsLDZKitUSY0Dw0MKmyAGWWuepczSfeGIl79789ahCgBMdted6U0191BwbRxVQQiViqjCoIqCpbFvBtk7DNASeomek+1dtuXcAPAVL+2mgE/eOXPF97erk6VCxRMcmyEKVoCyCZvpIw51HS1+gBLd5GJ9B7Nrf566vji54rsw9uKnrzVf6FR8QbKqANnIU26I5ZyPiqmylj7Gqy6itf6DFdkk7xXxF10665Lq8sP1E37gfDKS4J6RIV+t8qyvDQ/Bzr6NaVaInpSUT0yz5ZXAksSExmbeYuCZbhxLPO8H6mr8tewYGfYtg3DNKUp2mGLRI9pg0hg3yLsvULZW0OQRR08OKJRqCAXDOLaI+aWUiiLBtspIkvgDLlN3HZRgiOyWQJURmhsqhI/6KKcdTJZw7G2QEiGE4neFVyjb5USdL0a4+hw7aQ9lZ502nvB0Yx3rd7LcpwNHFZzzVuloaSOTq2Zx/gGeJct+4Yi/HhZ2E6drksyk59H/OKY7mGBk5D10Xadtbw///CK6A++PXqO6KkA2m2V5eZloNm75ukbOHqzub789fDql3p6ZJb4f4sobV/nos6+4deM629v/0daSwDrM89vsLDd/vEnRyNLfd4nibimgfjP8w7RtOb9Mr/1O+CBINBwFIHZxCMO0GB0dJZVKMTQ0xODgIKZVwdduAhCLxlQ/gGM5785t3rtTT6SLfA4A4+5PKNJjYmKC2tpaAHRdR3qwMvXIGP6AmnQ6bSpSSgAGv3glbKTNnyP/xlOv9g4oiUSSgOojl8uxsbGBpmm0trbS1NSEI5zS3qM95ubmHitSSgAA2tvbfY399eOhx5GPmxubq7UqTVFQeKCsllyfu90pus4qKFiW5WYymbyu61f/B/q4pKqmYKY6AAAAAElFTkSuQmCC',
wrench=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAH0SURBVDjLlZPLbxJRGMX5X/xbjBpjjCtXLl2L0YWkaZrhNQwdIA4FZxygC22wltYYSltG1HGGl8nopCMPX9AUKQjacdW4GNPTOywak7ZAF/eRe/M73/nOzXUAcEwaqVTKmUgkGqIoWoIgWP/fTYSTyaSTgAfdbhemaSIej+NcAgRudDod9Pt95PN5RKPR8wnwPG/Z1XVdB8dxin0WDofBsiyCwaA1UYBY/tdqtVAqlRCJRN6FQiE1k8mg2WyCpunxArFY7DKxfFir1VCtVlEoFCBJEhRFQbFYhM/na5wKzq/+4ALprzqxbFUqFWiaBnstl8tQVRWyLMPr9R643W7nCZhZ3uUS+T74jR7Y5c8wDAO5XA4MwxzalklVy+PxNCiKcp4IkbbhzR4K+h9IH02wax3MiAYCgcBfv99/4TS3xxtfepcTCPyKgGl5gCevfyJb/Q3q6Q5uMcb7s3IaTZ6lHY5f70H6YGLp7QDx9T0kSRtr5V9wLbZxw1N/fqbAHIEXsj1saQR+M8BCdg8icbJaHOJBqo3r1KfMuJdyuBZb2NT2R5a5l108JuFl1CHuJ9q4NjceHgncefSN9LoPcYskT9pYIfA9Al+Z3X4xzUdz3H74RbODWlGGeCYPcVf4jksz08HHId6k63USFK7ObuOia3rYHkdyavlR+267GwAAAABJRU5ErkJggg==',
wrench_orange=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHaSURBVDjLlZO7a1NRHMfzfzhIKQ5OHR1ddRRBLA6lg4iTd5PSas37YR56Y2JiHgg21uoFxSatCVFjbl5iNBBiMmUJgWwZhCB4pR9/V4QKfSQdDufF5/v7nu85xwJYprV0Oq0kk8luIpEw4vG48f/eVDiVSikCTobDIePxmGg0yokEBO4OBgNGoxH5fJ5wOHwygVgsZpjVW60WqqqWzbVgMIjf78fn8xlTBcTy736/T7VaJRQKfQoEArqmafR6Pdxu9/ECkUjkglje63Q6NBoNisUihUKBcrlMpVLB6XR2D4df3VQnmRstsWzU63WazSZmX6vV0HWdUqmEw+GY2Gw25SC8dV1l1wrZNX5s3qLdbpPL5fB6vXumZalq2O32rtVqVQ6GuGnCd+HbFnx9AZrC+MkSHo/np8vlmj/M7f4ks6yysyawgB8fwPv70HgKG8v8cp/7fFRO/+AllewqNJ/DhyBsi9A7J1QTkF4E69mXRws8u6ayvSJwRqoG4K2Md+ygxyF5FdbPaMfdlIXUZfiyAUWx/OY25O4JHBP4CtyZ16a9EwuRi1CXs+5K1ew6lB9DXERX517P8tEsPDzfNIP6C5YeQewSrJyeCd4P0bnwXYISy3MCn5oZNtsf3pH46e7XBJcAAAAASUVORK5CYII=',
xhtml=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAHlSURBVBgZpcE7a1RRFIbh95yZXEaSSLwWFkFEkICKhWhhIV7AxlKsbSz9DQpa+gfsbERQsUhnEYOFFiJoYSrBO6IBY5I5c2bvtfb6jCIIYjfPU0liFDUjqhlR99r9FfEfHoFZkNwxg9ZFm5xkTptFY0HbOl02Hdvf4y/hIUoRHsKLMBcWgZkwD6wE2YNbi1/p8sf6wCkBHsJLkIswD8xF9iB5IZtIHmQLtk11aftOl03nDk/x6NUGpw9OsTYo3H26yoXjs/TGK8Qmwav3A5aW17h0cjfJg9tL34jWqJM7gxTMTnWIgImxmjYXeuMVNxe+UAFX731kbuc483t67Nk+zt5dk7QWROPUTXKevWk4um8LD5+vMjlWcfnMTrqdin4qCGhSIQJOHJjhl41hIVlBTaHut+LU/DSPX69z9tAMgxTcePCZZKIZFiRohoWQePmuz4eVhARDE5Ey9VqbsSKeLK/TqSsk6CdHEk0qIGhyIQQ3Fz7xY+Bs7XW4fnEOJVGdvr6s80dm+fQ9kS1IHiQT2YPkQfbAPDAXVgIrwkPM7Zhg8c5buusbTpsL05Md8ljFpFXYhHCvMK+xEFZEKYEHlAgkkPit2nflhYatIxORAmVHFigVyIFKIAvkggj+VUliFDUj+gngimmFTeOsxAAAAABJRU5ErkJggg==',
xhtml_add=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJRSURBVBgZpcHda45xGMDx7/3yzB6ebc3ayJS8rBDTkixRMsoByoE/wIkDcbQTckJRTnYmB07kiPIaRa28JM2iaLLJS0mYtzLbnvu597t/1++6TCknitrnE5kZMxEzQzEzlB679M34C1HFe8WJ4D3kYuROcF7ICyPzSp4LKdPWd5T5wxA1QjBEDQmGF8Or4r3hRfFBKUQ5c+cLKb9N1ISgIGpIUIpgeFG8GIUoTgKFN5wohVfmVlLyqpAybfuaCv1Dk/SsrjBeC1wYGGNPdzPlughjmsHQuxr3RsbZu3keTpRz976iuSd2ItSc0lxJUIVZpZi8CJTrIvpufCICjl58z6LWOla2l2lvqWNxWz25VzQT4swJD99krFs6m6uPx6gvRezb2kqaRFRdwIDMBVRh4/JGfpmcCjgfsCyQVnNjy8oG7g5PsK2zkZpTTlz5yOHd7WRTATPIpgKDI9d59OIm1XyMQgqqsgl1XaTjeYEPxv2RCdYumYMZVJ1gZmQugEEk/bz89oQtm9axcG4Hd4evMfj8PC3zXhH1HB+xnV3NfPjuKLziRHHeKERxohSiNGkve3ZshyRm14qD9N3eR0LM+cu3iCcmhbwINNQnNMxOaConNFcSWioJrZWUtsYSP6pfKEUVdq04yC+9PWdY2tZJFHvS77nn1M2PmDfUKVYI5hVzAQrFgrJs1RjDowMMjT7g0LaznOzfS30yC9MSkZnxL90HFhxfuLjpyIbOjXTM7+L156cMPHvAh7fjfZGZ8T+6Dyw4CewHGoBJ4PTgqdFDPwHX2V3XB5aEeQAAAABJRU5ErkJggg==',
xhtml_delete=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJKSURBVBgZpcFdaI1xHMDx73POs7NzbMO25qXFGikRkmQXLuQlL3GpuHDhwkuSC1dKaQp37pTszhWFFJFGLG1cuDG1cyHKsnm3tXP2nGf/5//7/36myI2i9vlEZsZM5JihHDMUn7351fgLUcV7xYngPaRipE5wXkgzI/FKmgox0zYsK/GHIWqEYIgaEgwvhlfFe8OL4oOSidLz+DMxv1RqQlAQNSQoWTC8KF6MTBQngcwbTpTMKy2NMemkEDNtx5pGegerbFnVyEQtcP3ZOHu7mikVIoxpBoPDNfrKExzcNB8nytW+L2jqyTkRak5pbsyjCvV1OdIsUCpEXLz7kQjovvGejrYCK9pLtLcW6JxXJPWKJkIuccLzNwnrl87i9otxinURh7e2EecjJl3AgMQFVGHj8tn8VJ0KOB+wJJCbTI3NK5p4MlRh2+rZ1Jxy/tYozhvJVMAMkqlA5eEVysdXM7CzyNCRZSwt96AuI55IM3wwnpYrrFvSgBlMOsHMSFwAg5XD17CRR3TtP0R950rSV700DTxk0GpEW86Vbc/aZkbGHJlXnCjOG5koTpRMlAP9u9h99ASlt30w2g9z5vIt7uDBg9fElaqQZoGmYp6sLqLoI3y9IRLhJYdXoyX5QHFBJ+w6yW9x90IW8J14LPVcuj+KeUOdYplgXjEXIFMsKFsLLdRe3qPhzjFc+okaUK3k+Ra1EpkZ/9K/r7270NRwelGrxHFuhOpX4d3nfPBTdiYyM/7HwL7Fp5KxD0fyIeoIeRs1uLy9Vy78AKt+cH41HYv2AAAAAElFTkSuQmCC',
xhtml_go=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJLSURBVDjLpZNNiI1hFMd/773vjJlxR2iMKUyZSfItHxlJaQyxYCcbJYqlpR1R2Ck7ZTdWytfCQsnHpJCGkWTKR2QmZMS4d+573/s8z3nOsZhRFhZq/ptTp/P/9f8vTmJmTEcFpqlpA9LT177/s4OoEoLiRAgBcjFyJ7gg5N7IgpLnQgqwaUnzX1ZD1Ihxcko0ghhBlRCMIEqIihfl0v1vkwCASk2IypRJ8XHqWAwvipOID4YTxQdlbiklr04l2LWmxJ2XE2xfVaJci1x5PM6+njk0NybYZChefqoxMFzm0Lb5OFH6B8bQPFBwItScMqdURBVmNBTIfaS5MeH8ra8kwKmro1x/sYeRyl7aZxuL25vIg6KZUMic8OR9xsbuFm4OjtPUkHC0bx5pMaHqIgZkLlIXz8K2ZRzpX8fPLMOFiGWRtJobvctbefC6wo7Vs6g55eyNz9STAyiB41cjvhhY0LqEZR2bmKhnHL68lPb0Nuo8aTn3hGg8HK6wvmsmZlB1Ao2eHSsOEk2JGlGML+VRVi3cQtXnDI1uJtHLpL9qgYt3vlF1kRNXRnCiuGAUC55oyqcf7wgqiAZCDFTcBGsXbaUaaoz37SetTAi5j7Q2FfENCU0hIcwwxnwdicL8WZ2IRqIpX8sjzC11MDT6iKcfhhl/eIGk+9hzq+eCBUOdYl6woGzZfZi693j11MXT3bacnq6dPBt5zN1Xg2RPz2LVTpL//caVZwrZhs7eloG39z57ZfuXc/YG+H9A18kki1psyTV2jZ2zj3/2vwEmE35dgxn8EgAAAABJRU5ErkJggg==',
xhtml_valid=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJgSURBVDjLnVNLa1NBFP7m3jQPSVKiTaskGKQKUqlSQZTqqlWrCwUX9bFx6dIf4MpFXQi6cafgQhAVJD7oLou2uFBEBLswCxFqbFRswZjHvbd3Zu6MZyYGiptKBy7f3DnnO+c7Z84wrTU2s6buT0wLKe86/+N8efZ88WL53NF15LNEflwY2pFjGykg8m7OxT7O+TMu5JQQIknkF9vzg26tXm9vqIDIOSKXi/mCQ+QKkV+WCkVLJvNBdq3y0EpQWkFFCpGij1BEEoJHCIWAL+vQqU/oz2TQ8X0sf1+BE45DihRi8eZJHN6TWpdTQypNQbooCYXUWGq9xXztFrRiOL3zDuJsCPfmfiLWo7V8Sdnxl6TALVFZMifUchRj2Zv2fLWRw9a0QtCR3QCnDqRRWWxjcjSNph/hyesGpo/kkIoz6K4oLNZ8LFSBS+MDCCngg4UVqEBQKVLCDxVyaRdUPhJ9DgKq3ZBvz/4AI/71p8so5eMYKaRQ2BbHrsEkAkE98yQcL5R489nDoeEteP6ugWQfw5XjecRchk4YWQUeoQl+bG/WltteM80lmxfB6QQaEyMZzH9s4cT+rFVzo/yNHDQ8cjRjYlDR5sOXDr6uhvZsjewq5HCaAacr03hVbcF1mDV2SJUZMJPZSPC4CQAqqY7f1Oz+lIuZCyXoUINNzlT1mbEc6r9CcKrLNMhkN503e25vonsbws5H93pLAwnMPVpCrNWWtmmZpAtO9ScFg0iQkySUDoTSlmSGi+LYQTMqey+ADV99r9cCkmxrIiM3e0IjnxMS0f6TAtvJfxbb7HPurT/OV55dSDB+9QAAAABJRU5ErkJggg==',
zoom=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJGSURBVDjLjdJLSNRBHMDx78yqLZaKS75DPdgDDaFDbdJmde5QlhCJGxgpRJfqEEKnIsJLB7skQYQKZaSmdLaopPCgEvSCShCMzR5a7oq7/3l12RVtjfzBMA/4fWZ+MyOccwBM3g8HEbIdfCEhfAFnLVapOa28Uevpjrqz/WOsERJgsu9Uq5CZQzgqrJfo9BajNd5irEYn4p3OUiFExtCLmw2tawFi4l5zUMjMIau9u7K+qxeoAcoAA0wDb2OPwmfA16LiiaOHLj1edRLpkO3WmIis7+oBDgJbgQ2AH6gC6jY19N62RkcctKeVIJAhp9QgUA3kJXdONZVcq9JxPSgQoXRAyIDRth8oAXQyKdWnoCKrTD9CBv4GMqx1WGNZkeRWJKbG2hiD1Cb9FbTnzWFdY/LCdLKlgNQ84gyNKqHm0gDjqVHnxDHgA/B9RQkpaB6YklkZl62np9KBhOqwjpKFgeY2YAz4BESBWHI8Hhs6PVVSvc3v98ye4fP7T676B845nt040ip98qpWJmI9PWiU6bfWgXGN2YHcKwU7tsuc4kpUPMbU0+f8+vKt+Pitl7PLAMDI9cNBoB0hQwICzjqUp6MZvsy8yvp95BRuQUjJ75mPvH4wYo1NlJ64Mza7DPwrhi8cCOeXl/aUB4P4c/NJxKLMvpngycCrzxVFG2v/CwAMnguF80oLe8p27cQh+fnpPV/fTc95S6piXQDAw7a9YbWkezZXFbAwMx/xPFXb1D3+Y90AQF/L7kAsri9mZ4lrTd0TcYA/Kakr+x2JSPUAAAAASUVORK5CYII=',
zoom_in=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJnSURBVDjLpZLvSxNxHMf9C3rqw6QHcVaIKHVEKQQuVtuaOrelbldTj5mrliQjYUtreqUrbYaObf6k9oOmTAIz8TbS6c7FdqPChBCLIq64hz0+Pn2/MaI6C6EHLziO7+f1eX8+328RABT9Dz8/+LCJ5CNUnI9YxHy0TeLDLVJ2xixujBvj6TEd+U8BHzHT+ahF4MMUn51pcmYmjMRG0EBsBPROLmDkOZ9RWPXW0rsKcGdcnHvczOQ/fitFNCDsiMsIDeJAylvLpLw6ITmgliUpyoWpeC5E8egggbiE6EY4EF2ITkQzomRlSMsnBtTxXRJcFLPTjU50qB5xo1B8vVCMU3QgahKMypkc0IgyQT7SImXGjbi77ZeuP0awPjOBZd4Ipmg9LN9SEslBrSQfIWSROL8eC6yFwqsFWfuFOQOENydAN6mC5zcVRIJRywWZqWaR8zXgEZSFwo7Wp41AxRrAsXgFJl+Ngm22FWo8VVDtxjv8Q5D2G9A96/ESSxAGBI0jT732QSA/Asy6Cx68vAuetBuOdpfLBamROnJtVC+sDGvxNe5HnMKRNWMKoCMUeLh+aArqoNJRBmX2Upi3n2yUPaQX987Rq8O1QnJQw7N9Z5xLPaeJRZeCwJFvr7qgousIfMr6YWftPrBuLcy2Hy+WPWW2/yyJiLOMWsTLYvtUEo5c3nkYDtkOwpfNR/B1KwTvlnsh1lYlRVuOFf8m+Bvz16rNK0Pn4f36MAhvpuED9xAyQQvcURLbexJg5jpOmNm+OthacMHbhR5IeQ0Qs5I7exZgntBkU8hU+XmpV4lGILdDVMU+/P87L+2y1u3sopMAAAAASUVORK5CYII=',
zoom_out=>'iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAYAAAAf8/9hAAAABGdBTUEAAK/INwWK6QAAABl0RVh0U29mdHdhcmUAQWRvYmUgSW1hZ2VSZWFkeXHJZTwAAAJWSURBVDjLpZLva1JRGMf9F4JgEEGjIG5vetGoSz8hmGVN023qbTklTdmyHxaN2OLuF5nDsSYWNZyLHM0fZOVetGx4dXNuU0GvL6INKioKutWtKDZaRLenc+AWxbUY9OIDh8Pz/T7f5zlHBgCy/+HXgQ01kmzYFGPDZr4UsQpsyCIUho18boiKzV6tJ/9pwIaNtlLEzLEhE1sYPkTnr1FEzq8ncoM6OjtIsdkBipvyamxlDXBnLC6OGFylFwsbEFqEA3EcoUKszXg1roy3nku5lZIksmLIFCsGTSwqJBBHEW2Is4gWxGmEAVGZ7lezSbcyVibBYb4QaKBRUR2iVRSfEcU4hR1RnXTV0Cm3ipcYlMIWIT9E4e7Hfuv6cwScqAmnSHQriFSvWpCOEDQLWZ+OEAux8KRo1izeWRHU/Q45kXQppQb56wY+O6DFIyhEof17awt8c5yAr0cssKTVw6JcAZ82b4MP6zeCxGDWp0fvrMNLrEToETYs/pKbgKXJOHy+dwcWoyOwEPDDu1XrpAaZS7Xk9BUdl/ao8TOuQewu1xmL365YDaOOHQ2SjzTZd8A25dFwqV4Vyzj30eOde4h4u5yIn6um0311wtzdNnhZ8MHT6YvAnFfDreatFZKvzFzYTyJijEvJ42Uxzhoh3rH3Y6JbCY8neuD1wxvwZj4IjxJdELXuFCKWLRV/GPyN0VO7jOn+g/BsxgPcgwA8z16GvN8MPQriybIMMLft242Msxbmx9phbqwTMl49RJvI98s2wNy0kcZgYxWMdynQCOSroGnTSnz/A5Fyv6NS2MnjAAAAAElFTkSuQmCC',
};
