/*
 *  $Id: laplace.c 28906 2025-11-24 16:45:11Z yeti-dn $
 *  Copyright (C) 2004 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>
#include <gwy.h>

#define RUN_MODES GWY_RUN_IMMEDIATE

static gboolean module_register(void);
static void     module_main    (GwyFile *data,
                                GwyRunModeFlags mode);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Removes data under mask, interpolating them with Laplace equation solution."),
    "Petr Klapetek <klapetek@gwyddion.net>",
    "2.1",
    "David Nečas (Yeti) & Petr Klapetek",
    "2004",
};

GWY_MODULE_QUERY2(module_info, laplace)

static gboolean
module_register(void)
{
    gwy_process_func_register("laplace",
                              module_main,
                              N_("/_Correct Data/_Interpolate Data Under Mask"),
                              GWY_ICON_REMOVE_UNDER_MASK,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Interpolate data under mask by solution of Laplace equation"));
    gwy_process_func_register("zeromasked",
                              module_main,
                              N_("/_Correct Data/_Zero Data Under Mask"),
                              GWY_ICON_ZERO_UNDER_MASK,
                              RUN_MODES,
                              GWY_MENU_FLAG_IMAGE_MASK | GWY_MENU_FLAG_IMAGE,
                              N_("Fill data under mask with zeros"));

    return TRUE;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    const gchar *name = gwy_process_func_current();
    GwyField *dfield, *mfield;
    GQuark dquark;
    gint id;

    g_return_if_fail(mode & RUN_MODES);
    gwy_data_browser_get_current(GWY_APP_FIELD_KEY, &dquark,
                                 GWY_APP_FIELD, &dfield,
                                 GWY_APP_MASK_FIELD, &mfield,
                                 GWY_APP_FIELD_ID, &id,
                                 0);
    g_return_if_fail(dfield && dquark && mfield);
    gwy_app_undo_qcheckpointv(GWY_CONTAINER(data), 1, &dquark);
    if (gwy_strequal(name, "laplace"))
        gwy_field_laplace_solve(dfield, mfield, -1, 1.0);
    else if (gwy_strequal(name, "zeromasked")) {
        gint xres = gwy_field_get_xres(dfield);
        gint yres = gwy_field_get_yres(dfield);
        gwy_field_area_fill(dfield, mfield, GWY_MASK_INCLUDE, 0, 0, xres, yres, 0.0);
    }
    else {
        g_assert_not_reached();
    }
    gwy_field_data_changed(dfield);
    gwy_log_add(data, GWY_FILE_IMAGE, id, id);
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
