/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

/*!
    \file   lcas_plugin_example.c
    \brief  Interface to the LCAS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code for an example LCAS plugin and shows the interface
    the plugin has to provide to the LCAS. The interface consists of the following
    functions:
    -# plugin_initialize()
    -# plugin_confirm_authorization()
    -# plugin_terminate()
*/
/*!
    \defgroup LcasPluginInterface The interface to the LCAS plugins

    Here the interface is shown that the plugin has to provide to the LCAS.
    The interface consists of the following functions:
    -# plugin_initialize()
    -# plugin_confirm_authorization()
    -# plugin_terminate()
*/
/*\@{*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcas_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "lcas_modules.h"

/******************************************************************************
                                Definitions
******************************************************************************/

/******************************************************************************
                          Module specific prototypes
******************************************************************************/

/******************************************************************************
                       Define module specific variables
******************************************************************************/

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCAS_MOD_SUCCESS : succes
    LCAS_MOD_FAIL    : failure
    LCAS_MOD_NOFILE  : db file not found (will halt LCAS initialization)
******************************************************************************/
/*!
    \fn plugin_initialize(
        int argc,
        char ** argv
        )
    \brief initialize the plugin.
    
    Everything that is needed to initialize the plugin should be put inside this function.
    Arguments as read from the LCAS database (argc, argv) are passed to the plugin.

    \param argc number of passed arguments.
    \param argv argument list. argv[0] contains the name of the plugin.
    \retval LCAS_MOD_SUCCESS successful initialization
    \retval LCAS_MOD_FAIL    failure in the plugin initialization
    \retval LCAS_MOD_NOFILE  private plugin database could not be found (same effect as
                             LCAS_MOD_FAIL)
*/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    int i;

    lcas_log_debug(1,"lcas_plugin_example-plugin_initialize(): passed arguments:\n");
    for (i=0; i < argc; i++)
    {
        lcas_log_debug(2,"lcas_plugin_example-plugin_initialize(): arg %d is %s\n",
             i,argv[i]);
    }

    return LCAS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_confirm_authorization
Description:
    Ask for authorization by passing RSL(JDL) and user credential
Parameters:
    request:   RSL request
    lcas_cred: user credential
Returns:
    LCAS_MOD_SUCCESS: authorization succeeded
    LCAS_MOD_FAIL   : authorization failed
    LCAS_MOD_NOFILE : db file not found (LCAS will deny authorization)
******************************************************************************/
/*!
    \fn plugin_confirm_authorization(
        lcas_request_t request,
        lcas_cred_id_t lcas_cred
        )
    \brief Ask the plugin authorization module for authorization.

    Ask for authorization by passing the RSL (later JDL) and the user credential.
    The user credential will contain information on the role the user wants to have.
    In the RSL (JDL) the user might specify the resources he wants to use.
    The authorization decision has to be made using this information.
    The LCAS provides no library for parsing the RSL (JDL).

    \param request   LCAS (RSL) request
    \param lcas_cred LCAS credential
    \retval LCAS_MOD_SUCCESS authorization succeeded
    \retval LCAS_MOD_FAIL    authorization failed
    \retval LCAS_MOD_NOFILE  private plugin database could not be found (LCAS will deny authorization)
*/
int plugin_confirm_authorization(
        lcas_request_t request,
        lcas_cred_id_t lcas_cred
)
{
    char *  username = NULL;

    lcas_log_debug(1,"\tlcas_plugin_example-plugin_confirm_authorization():\n");

    /*
     * check credential and get the globus name
     */
    if ( (username = lcas_get_dn(lcas_cred)) == NULL)
    {
        lcas_log(0, "lcas.mod-lcas_get_fabric_authorization() error: user DN empty\n");
        goto fail_example;
    }

    if (username != NULL)
    {
        lcas_log_debug(0,
            "\tlcas_plugin_example-plugin_confirm_authorization(): OK, what the heck, I'll authorize Mr/Mrs %s\n",
                 username);
    }
    else
    {
        lcas_log_debug(0,
            "\tlcas_plugin_example-plugin_confirm_authorization(): Authorization failure: invalid username (%s)\n",username);
        goto fail_example;
    }

    /* succes */
    return LCAS_MOD_SUCCESS;

 fail_example:
    return LCAS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCAS_MOD_SUCCESS : succes
    LCAS_MOD_FAIL    : failure
******************************************************************************/
/*!
    \fn plugin_terminate()
    \brief Whatever is needed to terminate the plugin module goes in here.
    \retval LCAS_MOD_SUCCESS success
    \retval LCAS_MOD_FAIL    failure (will result in an authorization failure)
*/
int plugin_terminate()
{
    lcas_log_debug(1,"lcas_plugin_example-plugin_terminate(): terminating\n");

    return LCAS_MOD_SUCCESS;
}
/*\@}*/

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/examples/lcas_plugin_example.c,v $
    $Date: 2010-05-03 10:42:49 $
    $Revision: 2.10 $
    $Author: okoeroo $
******************************************************************************/
