/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!  
 *  \file pdl.h
 *
 *  \brief General include file.
 *
 *  In this include file all general "things" can be found.
 *
 *
 *  \author  G.M. Venekamp  (venekamp@nikhef.nl)
 *  \version $Revision: 17818 $
 *  \date    $Date: 2014-07-03 13:42:13 +0200 (Thu, 03 Jul 2014) $
 *
 */

#ifndef _PDL_H
#define _PDL_H

#include <stdio.h>

/*!
 *  The evaluation manager defines its own boolean type. It first
 *  undefines any existing type defenitions before it defines it
 *  itself.
 *
 */
#undef TRUE
#undef FALSE
#undef BOOL
typedef char BOOL;
#define TRUE ((BOOL)1)
#define FALSE ((BOOL)(!TRUE))


/*!
 *  Different levels of error logging.
 */
typedef enum {
  PDL_UNKNOWN,  /*  Unknown error level. */
  PDL_INFO,     /*  Informational level. */
  PDL_WARNING,  /*  Warning level. */
  PDL_ERROR,    /*  Error level. */
  PDL_SAME      /*  Repeat the previous level. */
  /*  Keep PDL_SAME last as it is used as a sentinal! */
} pdl_error_t;


/*!
 *  Guide the selection of the next plugin.
 */
typedef enum {
  EVALUATION_START,        /* The evaluation process has just started. */
  EVALUATION_SUCCESS,      /* The evaluation of the plugin was successful. */
  EVALUATION_FAILURE       /* The evaluation of the plugin was unsuccessfyl. */
} plugin_status_t;


/*!
 *  \brief Structure is used to keep track of strings and the line
 *  they appear on.
 *
 *  When lex finds a match, this structure is used to keep track of
 *  the relevant information. The matchig string as well as the line
 *  number are saved. The line number can be used for later references
 *  when an error related to the symbol has occured. This allows for
 *  easier debugging of the configuration file.
 *
 */
typedef struct record_s {
  char* string;   /* Hold the symbol that lex has found. */
  int   lineno;   /* Hold the line number the symbol has been found. */
} record_t;


/*!
 *  \brief Structure holds a plugin name and its arguments, as well as
 *  the line number the plugin is first mentioned.
 *
 */
typedef struct plugin_s {
  char* name;             /* Plugin name. */
  char* args;             /* Arguments of the plugin. */
  int lineno;		  /* Line number where the plugin is first seen in the
			     configuration file. */
  struct plugin_s* next;  /* Next plugin, or 0 if there are no-more plugins. */
} plugin_t;


/* since pdl_lex.c already declares these, skip 'em here: this has only effect
 * for when we are currently being included in pdl_lex.c which is created by
 * (f)lex */
#ifndef FLEX_SCANNER
extern FILE* yyin;
#endif
/* Also declare yylex() for pdl_yacc.{y,c} */
extern int yylex(void);

/* Cleanup function for lex memory, dependent on flex version. Only called for
 * flex. */
extern void pdl_lex_cleanup(void);

extern int lineno;

extern int lcmaps_pdl_init(const char* name);
extern void lcmaps_free_path(void);
extern char *lcmaps_pdl_path(void);
extern int yyparse_errors(void);

/* Next one is wrapper around yyparse() since RH5 does not declare yyparse() in
 * the created header file, while newer yacc/bison does */
extern int pdl_yyparse(void);
extern int yyerror(const char*);
extern const char* token_name(void);
extern void lcmaps_set_yylval(record_t* r);

extern void lcmaps_set_path(record_t* _path);
extern record_t* lcmaps_concat_strings(record_t* s1, record_t* s2);
extern record_t* lcmaps_concat_strings_with_space(record_t* s1, record_t* s2);
extern const plugin_t* lcmaps_get_plugins(void);
#ifdef __GNUC__
extern void lcmaps_pdl_warning(pdl_error_t error, const char* s, ...)
    __attribute__((format(printf,2,3)));
#else
extern void lcmaps_pdl_warning(pdl_error_t error, const char* s, ...);
#endif
extern void lcmaps_free_resources(void);

/* void set_active_yyin (FILE * my_yyin); */
/* FILE * get_active_yyin (void); */


extern char* lcmaps_pdl_next_plugin(plugin_status_t status);
extern BOOL  lcmaps_allowed_policy_rule(const char* label);

#ifdef HAVE_FLEX
extern void delete_lex_buffer(void);
#endif

#endif
