/***************************************************************************
     testqgsgradients.cpp
     --------------------------------------
    Date                 : 20 Jan 2008
    Copyright            : (C) 2008 by Tim Sutton
    Email                : tim @ linfiniti.com
 ***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "qgstest.h"
#include <QObject>
#include <QString>
#include <QStringList>
#include <QApplication>
#include <QFileInfo>
#include <QDir>
#include <QDesktopServices>

//qgis includes...
#include <qgsmaplayer.h>
#include <qgsvectorlayer.h>
#include <qgsapplication.h>
#include <qgsproviderregistry.h>
#include <qgsproject.h>
#include <qgssymbol.h>
#include <qgssinglesymbolrenderer.h>
#include <qgsfillsymbollayer.h>
#include <qgscolorramp.h>
//qgis test includes
#include "qgsrenderchecker.h"

/**
 * \ingroup UnitTests
 * This is a unit test for gradient fill types.
 */
class TestQgsGradients : public QObject
{
    Q_OBJECT

  public:
    TestQgsGradients() = default;

  private slots:
    void initTestCase();// will be called before the first testfunction is executed.
    void cleanupTestCase();// will be called after the last testfunction was executed.
    void init() {} // will be called before each testfunction is executed.
    void cleanup() {} // will be called after every testfunction.

    void gradientSymbol();
    void gradientSymbolColors();
    void gradientSymbolRamp();
    void gradientSymbolRadial();
    void gradientSymbolConical();
    void gradientSymbolViewport();
    void gradientSymbolReferencePoints();
    void gradientSymbolCentroid();
    void gradientSymbolReflectSpread();
    void gradientSymbolRepeatSpread();
    void gradientSymbolRotate();
    void gradientSymbolFromQml();
  private:
    bool mTestHasError =  false ;
    bool setQml( const QString &type );
    bool imageCheck( const QString &type );
    QgsMapSettings mMapSettings;
    QgsVectorLayer *mpPolysLayer = nullptr;
    QgsGradientFillSymbolLayer *mGradientFill = nullptr;
    QgsFillSymbol *mFillSymbol = nullptr;
    QgsSingleSymbolRenderer *mSymbolRenderer = nullptr;
    QString mTestDataDir;
    QString mReport;
};


void TestQgsGradients::initTestCase()
{
  mTestHasError = false;
  // init QGIS's paths - true means that all path will be inited from prefix
  QgsApplication::init();
  QgsApplication::initQgis();
  QgsApplication::showSettings();

  //create some objects that will be used in all tests...
  QString myDataDir( TEST_DATA_DIR ); //defined in CmakeLists.txt
  mTestDataDir = myDataDir + '/';

  //
  //create a poly layer that will be used in all tests...
  //
  QString myPolysFileName = mTestDataDir + "polys.shp";
  QFileInfo myPolyFileInfo( myPolysFileName );
  mpPolysLayer = new QgsVectorLayer( myPolyFileInfo.filePath(),
                                     myPolyFileInfo.completeBaseName(), QStringLiteral( "ogr" ) );

  QgsVectorSimplifyMethod simplifyMethod;
  simplifyMethod.setSimplifyHints( QgsVectorSimplifyMethod::NoSimplification );
  mpPolysLayer->setSimplifyMethod( simplifyMethod );

  //setup gradient fill
  mGradientFill = new QgsGradientFillSymbolLayer();
  mFillSymbol = new QgsFillSymbol();
  mFillSymbol->changeSymbolLayer( 0, mGradientFill );
  mSymbolRenderer = new QgsSingleSymbolRenderer( mFillSymbol );
  mpPolysLayer->setRenderer( mSymbolRenderer );

  // We only need maprender instead of mapcanvas
  // since maprender does not require a qui
  // and is more light weight
  //
  mMapSettings.setLayers( QList<QgsMapLayer *>() << mpPolysLayer );
  mReport += QLatin1String( "<h1>Gradient Renderer Tests</h1>\n" );

}
void TestQgsGradients::cleanupTestCase()
{
  QString myReportFile = QDir::tempPath() + "/qgistest.html";
  QFile myFile( myReportFile );
  if ( myFile.open( QIODevice::WriteOnly | QIODevice::Append ) )
  {
    QTextStream myQTextStream( &myFile );
    myQTextStream << mReport;
    myFile.close();
  }

  delete mpPolysLayer;

  QgsApplication::exitQgis();
}

void TestQgsGradients::gradientSymbol()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer test</h2>\n" );
  mGradientFill->setColor( QColor( "red" ) );
  mGradientFill->setColor2( QColor( "blue" ) );
  mGradientFill->setGradientType( QgsGradientFillSymbolLayer::Linear );
  mGradientFill->setGradientColorType( QgsGradientFillSymbolLayer::SimpleTwoColor );
  mGradientFill->setCoordinateMode( QgsGradientFillSymbolLayer::Feature );
  mGradientFill->setGradientSpread( QgsGradientFillSymbolLayer::Pad );
  mGradientFill->setReferencePoint1( QPointF( 0, 0 ) );
  mGradientFill->setReferencePoint2( QPointF( 1, 1 ) );
  QVERIFY( imageCheck( "gradient" ) );
}

void TestQgsGradients::gradientSymbolColors()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer color test</h2>\n" );
  mGradientFill->setColor( QColor( "green" ) );
  mGradientFill->setColor2( QColor( "white" ) );
  QVERIFY( imageCheck( "gradient_colors" ) );
  mGradientFill->setColor( QColor( "red" ) );
  mGradientFill->setColor2( QColor( "blue" ) );
}

void TestQgsGradients::gradientSymbolRamp()
{
  QgsGradientColorRamp *gradientRamp = new QgsGradientColorRamp( QColor( Qt::red ), QColor( Qt::blue ) );
  QgsGradientStopsList stops;
  stops.append( QgsGradientStop( 0.5, QColor( Qt::white ) ) );
  gradientRamp->setStops( stops );

  mGradientFill->setColorRamp( gradientRamp );
  mGradientFill->setGradientColorType( QgsGradientFillSymbolLayer::ColorRamp );
  QVERIFY( imageCheck( "gradient_ramp" ) );
  mGradientFill->setGradientColorType( QgsGradientFillSymbolLayer::SimpleTwoColor );
}

void TestQgsGradients::gradientSymbolRadial()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer radial test</h2>\n" );
  mGradientFill->setGradientType( QgsGradientFillSymbolLayer::Radial );
  QVERIFY( imageCheck( "gradient_radial" ) );
  mGradientFill->setGradientType( QgsGradientFillSymbolLayer::Linear );
}

void TestQgsGradients::gradientSymbolConical()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer conical test</h2>\n" );
  mGradientFill->setGradientType( QgsGradientFillSymbolLayer::Conical );
  mGradientFill->setReferencePoint1( QPointF( 0.5, 0.5 ) );
  QVERIFY( imageCheck( "gradient_conical" ) );
  mGradientFill->setReferencePoint1( QPointF( 0, 0 ) );
  mGradientFill->setGradientType( QgsGradientFillSymbolLayer::Linear );
}

void TestQgsGradients::gradientSymbolViewport()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer viewport test</h2>\n" );
  mGradientFill->setCoordinateMode( QgsGradientFillSymbolLayer::Viewport );
  QVERIFY( imageCheck( "gradient_viewport" ) );
  mGradientFill->setCoordinateMode( QgsGradientFillSymbolLayer::Feature );
}

void TestQgsGradients::gradientSymbolReferencePoints()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer reference points test</h2>\n" );
  mGradientFill->setReferencePoint1( QPointF( 0.5, 0.4 ) );
  mGradientFill->setReferencePoint2( QPointF( 0, 0.2 ) );
  QVERIFY( imageCheck( "gradient_refpoints" ) );
  mGradientFill->setReferencePoint1( QPointF( 0, 0 ) );
  mGradientFill->setReferencePoint2( QPointF( 1, 1 ) );
}

void TestQgsGradients::gradientSymbolCentroid()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer centroid reference point test</h2>\n" );
  mGradientFill->setReferencePoint1IsCentroid( true );
  QVERIFY( imageCheck( "gradient_ref1centroid" ) );
  mGradientFill->setReferencePoint1IsCentroid( false );
  mGradientFill->setReferencePoint2IsCentroid( true );
  QVERIFY( imageCheck( "gradient_ref2centroid" ) );
  mGradientFill->setReferencePoint2IsCentroid( false );
}

void TestQgsGradients::gradientSymbolReflectSpread()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer reflect spread test</h2>\n" );
  mGradientFill->setReferencePoint2( QPointF( 0.5, 0.5 ) );
  mGradientFill->setGradientSpread( QgsGradientFillSymbolLayer::Reflect );
  QVERIFY( imageCheck( "gradient_reflect" ) );
  mGradientFill->setGradientSpread( QgsGradientFillSymbolLayer::Pad );
  mGradientFill->setReferencePoint2( QPointF( 1, 1 ) );
}

void TestQgsGradients::gradientSymbolRepeatSpread()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer repeat spread test</h2>\n" );
  mGradientFill->setReferencePoint2( QPointF( 0.5, 0.5 ) );
  mGradientFill->setGradientSpread( QgsGradientFillSymbolLayer::Repeat );
  QVERIFY( imageCheck( "gradient_repeat" ) );
  mGradientFill->setGradientSpread( QgsGradientFillSymbolLayer::Pad );
  mGradientFill->setReferencePoint2( QPointF( 1, 1 ) );
}

void TestQgsGradients::gradientSymbolRotate()
{
  mReport += QLatin1String( "<h2>Gradient symbol renderer rotate test</h2>\n" );
  mGradientFill->setAngle( 90 );
  QVERIFY( imageCheck( "gradient_rotate" ) );
  mGradientFill->setAngle( 0 );
}

void TestQgsGradients::gradientSymbolFromQml()
{
  mReport += QLatin1String( "<h2>Gradient symbol from QML test</h2>\n" );
  QVERIFY( setQml( "gradient" ) );
  QgsVectorSimplifyMethod simplifyMethod;
  simplifyMethod.setSimplifyHints( QgsVectorSimplifyMethod::NoSimplification );
  mpPolysLayer->setSimplifyMethod( simplifyMethod );
  QVERIFY( imageCheck( "gradient_from_qml" ) );
}



//
// Private helper functions not called directly by CTest
//

bool TestQgsGradients::setQml( const QString &type )
{
  //load a qml style and apply to our layer
  //the style will correspond to the renderer
  //type we are testing
  QString myFileName = mTestDataDir + "polys_" + type + "_symbol.qml";
  bool myStyleFlag = false;
  QString error = mpPolysLayer->loadNamedStyle( myFileName, myStyleFlag );
  if ( !myStyleFlag )
  {
    qDebug( "%s", error.toLocal8Bit().constData() );
  }
  return myStyleFlag;
}

bool TestQgsGradients::imageCheck( const QString &testType )
{
  //use the QgsRenderChecker test utility class to
  //ensure the rendered output matches our control image
  mMapSettings.setExtent( mpPolysLayer->extent() );
  QgsRenderChecker myChecker;
  myChecker.setControlPathPrefix( QStringLiteral( "symbol_gradient" ) );
  myChecker.setControlName( "expected_" + testType );
  myChecker.setMapSettings( mMapSettings );
  bool myResultFlag = myChecker.runTest( testType );
  mReport += myChecker.report();
  return myResultFlag;
}

QGSTEST_MAIN( TestQgsGradients )
#include "testqgsgradients.moc"
