% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map.R
\name{map}
\alias{map}
\alias{map_lgl}
\alias{map_chr}
\alias{map_int}
\alias{map_dbl}
\alias{map_raw}
\alias{map_dfr}
\alias{map_df}
\alias{map_dfc}
\alias{walk}
\title{Apply a function to each element of a list or atomic vector}
\usage{
map(.x, .f, ...)

map_lgl(.x, .f, ...)

map_chr(.x, .f, ...)

map_int(.x, .f, ...)

map_dbl(.x, .f, ...)

map_raw(.x, .f, ...)

map_dfr(.x, .f, ..., .id = NULL)

map_dfc(.x, .f, ...)

walk(.x, .f, ...)
}
\arguments{
\item{.x}{A list or atomic vector.}

\item{.f}{A function, formula, or vector (not necessarily atomic).

If a \strong{function}, it is used as is.

If a \strong{formula}, e.g. \code{~ .x + 2}, it is converted to a function. There
are three ways to refer to the arguments:
\itemize{
\item For a single argument function, use \code{.}
\item For a two argument function, use \code{.x} and \code{.y}
\item For more arguments, use \code{..1}, \code{..2}, \code{..3} etc
}

This syntax allows you to create very compact anonymous functions.

If \strong{character vector}, \strong{numeric vector}, or \strong{list}, it is
converted to an extractor function. Character vectors index by
name and numeric vectors index by position; use a list to index
by position and name at different levels. If a component is not
present, the value of \code{.default} will be returned.}

\item{...}{Additional arguments passed on to the mapped function.}

\item{.id}{Either a string or \code{NULL}. If a string, the output will contain
a variable with that name, storing either the name (if \code{.x} is named) or
the index (if \code{.x} is unnamed) of the input. If \code{NULL}, the default, no
variable will be created.

Only applies to \verb{_dfr} variant.}
}
\value{
\itemize{
\item \code{map()} Returns a list the same length as \code{.x}.
\item \code{map_lgl()} returns a logical vector, \code{map_int()} an integer
vector, \code{map_dbl()} a double vector, and \code{map_chr()} a character
vector.
\item \code{map_df()}, \code{map_dfc()}, \code{map_dfr()} all return a data frame.
\item If \code{.x} has \code{names()}, the return value preserves those names.
\item The output of \code{.f} will be automatically typed upwards, e.g.
logical -> integer -> double -> character.
}

\itemize{
\item \code{walk()} returns the input \code{.x} (invisibly). This makes it easy to
use in pipe.
}
}
\description{
The map functions transform their input by applying a function to
each element of a list or atomic vector and returning an object of the same length as the input.
\itemize{
\item \code{map()} always returns a list. See the \code{\link[=modify]{modify()}} family for
versions that return an object of the same type as the input.
\item \code{map_lgl()}, \code{map_int()}, \code{map_dbl()} and \code{map_chr()} return an
atomic vector of the indicated type (or die trying).
\item \code{map_dfr()} and \code{map_dfc()} return a data frame created by
row-binding and column-binding respectively. They require dplyr
to be installed.
\item The returned values of \code{.f} must be of length one for each element
of \code{.x}. If \code{.f} uses an extractor function shortcut, \code{.default}
can be specified to handle values that are absent or empty. See
\code{\link[=as_mapper]{as_mapper()}} for more on \code{.default}.
}

\itemize{
\item \code{walk()} calls \code{.f} for its side-effect and returns
the input \code{.x}.
}
}
\examples{
# Compute normal distributions from an atomic vector
1:10 \%>\%
  map(rnorm, n = 10)

# You can also use an anonymous function
1:10 \%>\%
  map(function(x) rnorm(10, x))

# Or a formula
1:10 \%>\%
  map(~ rnorm(10, .x))

# Simplify output to a vector instead of a list by computing the mean of the distributions
1:10 \%>\%
  map(rnorm, n = 10) \%>\%  # output a list
  map_dbl(mean)           # output an atomic vector

# Using set_names() with character vectors is handy to keep track
# of the original inputs:
set_names(c("foo", "bar")) \%>\% map_chr(paste0, ":suffix")

# Working with lists
favorite_desserts <- list(Sophia = "banana bread", Eliott = "pancakes", Karina = "chocolate cake")
favorite_desserts \%>\% map_chr(~ paste(.x, "rocks!"))

# Extract by name or position
# .default specifies value for elements that are missing or NULL
l1 <- list(list(a = 1L), list(a = NULL, b = 2L), list(b = 3L))
l1 \%>\% map("a", .default = "???")
l1 \%>\% map_int("b", .default = NA)
l1 \%>\% map_int(2, .default = NA)

# Supply multiple values to index deeply into a list
l2 <- list(
  list(num = 1:3,     letters[1:3]),
  list(num = 101:103, letters[4:6]),
  list()
)
l2 \%>\% map(c(2, 2))

# Use a list to build an extractor that mixes numeric indices and names,
# and .default to provide a default value if the element does not exist
l2 \%>\% map(list("num", 3))
l2 \%>\% map_int(list("num", 3), .default = NA)

# Working with data frames
# Use map_lgl(), map_dbl(), etc to return a vector instead of a list:
mtcars \%>\% map_dbl(sum)

# A more realistic example: split a data frame into pieces, fit a
# model to each piece, summarise and extract R^2
mtcars \%>\%
  split(.$cyl) \%>\%
  map(~ lm(mpg ~ wt, data = .x)) \%>\%
  map(summary) \%>\%
  map_dbl("r.squared")

# If each element of the output is a data frame, use
# map_dfr to row-bind them together:
mtcars \%>\%
  split(.$cyl) \%>\%
  map(~ lm(mpg ~ wt, data = .x)) \%>\%
  map_dfr(~ as.data.frame(t(as.matrix(coef(.)))))
# (if you also want to preserve the variable names see
# the broom package)
}
\seealso{
\code{\link[=map_if]{map_if()}} for applying a function to only those elements
of \code{.x} that meet a specified condition.

Other map variants: 
\code{\link{imap}()},
\code{\link{invoke}()},
\code{\link{lmap}()},
\code{\link{map2}()},
\code{\link{map_if}()},
\code{\link{modify}()}
}
\concept{map variants}
