#!/usr/bin/env python3
# Script to check that headers are in a consistent order
# Canonical header order is defined in scripts/HeaderOrder.txt
# Requires Python 3.6 or later

import pathlib

patterns = [
    "include/*.h",
    "src/*.cxx",
    "lexlib/*.cxx",
    "lexers/*.cxx",
    "win32/*.cxx",
    "gtk/*.cxx",
    "cocoa/*.mm",
    "cocoa/*.h",
    "test/unit/*.cxx",
    "lexilla/src/*.cxx",
    "lexilla/test/*.cxx",
]

def IsHeader(x):
    return x.strip().startswith("#") and \
        ("include" in x or "import" in x) and \
        "dllimport" not in x

def HeaderFromIncludeLine(s):
    #\s*#\s*(include|import)\s+\S+\s*
    return s.strip()[1:].strip()[7:].strip()

def ExtractHeaders(filename):
    with filename.open(encoding="cp437") as infile:
        return [HeaderFromIncludeLine(l) for l in infile if IsHeader(l)]

def ExcludeName(name):
    # LexCaml adds system headers in #if to be an external lexer
    # moc_ files are generated by Qt and follow its rules
    return "LexCaml" in name or "moc_" in name

def SortLike(incs, order):
    return sorted(incs, key = lambda i: order.index(i))

def CheckFiles(root):
    # Find all the lexer source code files
    filePaths = []
    for p in patterns:
        filePaths += root.glob(p)
    # The Qt platform code interleaves system and Scintilla headers
    #~ filePaths += root.glob("qt/ScintillaEditBase/*.cpp")
    #~ filePaths += root.glob("qt/ScintillaEdit/*.cpp")
    #~ print(filePaths)
    scriptDirectory = root / "scripts"
    headerOrderFile = scriptDirectory / "HeaderOrder.txt"
    headerOrder = ExtractHeaders(headerOrderFile)
    originalOrder = headerOrder[:]
    orderedPaths = [p for p in sorted(filePaths) if not ExcludeName(str(p))]
    allIncs = set()
    for f in orderedPaths:
        print("   File ", f.relative_to(root))
        incs = ExtractHeaders(f)
        #~ print("\n".join(incs))
        news = set(incs) - set(headerOrder)
        allIncs = allIncs.union(set(incs))

        m = 0
        i = 0
        # Detect headers not in header order list and insert at OK position
        needs = []
        while i < len(incs):
            if m == len(headerOrder):
                #~ print("**** extend", incs[i:])
                headerOrder.extend(incs[i:])
                needs.extend(incs[i:])
                break
            if headerOrder[m] == incs[i]:
                #~ print("equal", headerOrder[m])
                i += 1
                m += 1
            else:
                if headerOrder[m] not in incs:
                    #~ print("skip", headerOrder[m])
                    m += 1
                elif incs[i] not in headerOrder:
                    #~ print(str(f) + ":1: Add master", incs[i])
                    headerOrder.insert(m, incs[i])
                    needs.append(incs[i])
                    i += 1
                    m += 1
                else:
                    i += 1
        if needs:
            print(f"{f}:1: needs these headers:")
            for header in needs:
                print("#include " + header)

        # Detect out of order
        ordered = SortLike(incs, headerOrder)
        if incs != ordered:
            print(f"{f}:1: is out of order")
            fOrdered = pathlib.Path(str(f) + ".ordered")
            with fOrdered.open("w") as headerOut:
                for header in ordered:
                    headerOut.write("#include " + header + "\n")
            print(f"{fOrdered}:1: is ordered")

    if headerOrder != originalOrder:
        newIncludes = set(headerOrder) - set(originalOrder)
        headerOrderNew = scriptDirectory / "NewOrder.txt"
        print(f"{headerOrderFile}:1: changed to {headerOrderNew}")
        print(f"   Added {', '.join(newIncludes)}.")
        with headerOrderNew.open("w") as headerOut:
            for header in headerOrder:
                headerOut.write("#include " + header + "\n")

    unused = sorted(set(headerOrder) - allIncs)
    if unused:
        print("In HeaderOrder.txt but not used")
        print("\n".join(unused))

CheckFiles(pathlib.Path(__file__).resolve().parent.parent)
