/*
 *  acm : an aerial combat simulator for X
 *  Copyright (C) 1991,1992,1997  Riley Rainey
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */

/**
 *  Some data structures to perform fast 2-D interpolation
 *
 *  A lot of aircraft information is in tabular form.  We perform linear
 *  interpolation to determine values.  In order to speed up that process,
 *  we'll create preprocessed tables.  Each entry in the table contains the
 *  upper bound of the domain value (x) and the equation of the line that
 *  defines the data for that interval (y = m * x + b).
 * 
 * @file
 */

#ifndef _interpolate_h
#define _interpolate_h

#include <math.h>

#ifdef interpolate_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

typedef struct {
	double    x;  /* upper x value for this interval */
	double    m;  /* slope of line in this interval */
	double    b;  /* y-intercept of line in this interval */
} interpolate_Entry;

typedef struct {
	int       count;  /* entry count in the interpolation table */
	double    minX;	  /* minimum x value that can be interpolated */
	interpolate_Entry *entry;  /* vector of interpolation entries */
} interpolate_Table;

/**
 * Allocates a new interpolation table.
 * @param size Number of entries.
 * @return New allocated table. Can be released with memory_dispose().
 */
EXTERN interpolate_Table * interpolate_new(int size);

/**
 * Determine the value of a function of one variable by interpolation.
 * Interpolation tables are built by the 'ibuild' utility.
 */
EXTERN double interpolate_value(interpolate_Table * table, double x);

/**
 * Returns a copy of the given table.
 * @param oldp
 * @return Copy of the given table. Can be released with memory_dispose().
 */
EXTERN interpolate_Table * interpolate_clone (interpolate_Table *oldp);

#undef EXTERN
#endif
