-module(pseudoknot).

-export([?MODULE/0]).

?MODULE() ->
    R = loop(1, 0),
    abs(R-33.7976) < 0.0001.
    
append([H|T], Z) ->
    [H|append(T, Z)];
append([], X) ->
    X.

atan2(Y,X) when X>0.0 ->
    math:atan(Y/X);
atan2(Y,X) when Y<0.0 ->
    if
	X == 0.0 -> -1.57079632679489661923;
	true -> math:atan(Y/X) - 3.14159265358979323846
    end;
atan2(Y,X) ->
    if
	X == 0.0 -> 1.57079632679489661923;
	true -> math:atan(Y/X) + 3.14159265358979323846
    end.

% -- POINTS ------------------------------------------------------------------

%pt ::= {X, Y, Z} where X,Y,Z are floats

pt_sub({X1, Y1, Z1}, {X2, Y2, Z2}) 
    when is_float(X1), is_float(Y1), is_float(Z1),
         is_float(X2), is_float(Y2), is_float(Z2) ->
    {X1 - X2, Y1 - Y2, Z1 - Z2}.

pt_dist({X1, Y1, Z1}, {X2, Y2, Z2})
    when is_float(X1), is_float(Y1), is_float(Z1),
         is_float(X2), is_float(Y2), is_float(Z2) ->
    Dx = X1 - X2,
    Dy = Y1 - Y2,
    Dz = Z1 - Z2,
    math:sqrt(Dx * Dx + Dy * Dy + Dz * Dz).

pt_phi({X, Y, Z})
    when is_float(X), is_float(Z) ->
    B = atan2(X, Z),
    atan2(math:cos(B) * Z + math:sin(B) * X, Y).

pt_theta ({X, _, Z}) ->
    atan2(X, Z).

% -- COORDINATE TRANSFORMATIONS ----------------------------------------------

% The notation for the transformations follows "Paul, R.P. (1981) Robot
% Manipulators.  MIT Press." with the exception that our transformation
% matrices don't have the perspective terms and are the transpose of
% Paul's one.  See also "M\"antyl\"a, M. (1985) An Introduction to
% Solid Modeling, Computer Science Press" Appendix A.
%
% The components of a transformation matrix are named like this:
%
%  a  b  c
%  d  e  f
%  g  h  i
% tx ty tz
%
% The components tx, ty, and tz are the translation vector.

%tfo ::= {A,B,C,D,E,F,G,H,I,Tx,Ty,Tz} where all elements are floats

tfo_id() -> {1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0}.

% The function "tfo-apply" multiplies a transformation matrix, tfo, by a
% point vector, p.  The result is a new point.

tfo_apply ({A,B,C,D,E,F,G,H,I,Tx,Ty,Tz}, {X,Y,Z})
    when is_float(A), is_float(B), is_float(C), is_float(D), is_float(E),
         is_float(F), is_float(G), is_float(H), is_float(I), 
         is_float(Tx), is_float(Ty), is_float(Tz), is_float(X), is_float(Y), is_float(Z) ->
    {X * A + Y * D + Z * G + Tx,
     X * B + Y * E + Z * H + Ty,
     X * C + Y * F + Z * I + Tz}.

% The function "tfo-combine" multiplies two transformation matrices A and B.
% The result is a new matrix which cumulates the transformations described
% by A and B.

tfo_combine({A_a,A_b,A_c,A_d,A_e,A_f,A_g,A_h,A_i,A_tx,A_ty,A_tz},
   {B_a,B_b,B_c,B_d,B_e,B_f,B_g,B_h,B_i,B_tx,B_ty,B_tz}) 
   when is_float(A_a), is_float(A_b), is_float(A_c), is_float(A_d), is_float(A_e),
        is_float(A_f), is_float(A_g), is_float(A_h), is_float(A_i), is_float(A_tx),
        is_float(A_ty), is_float(A_tz),
        is_float(B_a), is_float(B_b), is_float(B_c), is_float(B_d), is_float(B_e),
        is_float(B_f), is_float(B_g), is_float(B_h), is_float(B_i), is_float(B_tx),
        is_float(B_ty), is_float(B_tz) ->
    {A_a * B_a + A_b * B_d + A_c * B_g,
      A_a * B_b + A_b * B_e + A_c * B_h,
      A_a * B_c + A_b * B_f + A_c * B_i,
      A_d * B_a + A_e * B_d + A_f * B_g,
      A_d * B_b + A_e * B_e + A_f * B_h,
      A_d * B_c + A_e * B_f + A_f * B_i,
      A_g * B_a + A_h * B_d + A_i * B_g,
      A_g * B_b + A_h * B_e + A_i * B_h,
      A_g * B_c + A_h * B_f + A_i * B_i,
      A_tx * B_a + A_ty * B_d + A_tz * B_g + B_tx,
      A_tx * B_b + A_ty * B_e + A_tz * B_h + B_ty,
      A_tx * B_c + A_ty * B_f + A_tz * B_i + B_tz}.

% The function "tfo-inv-ortho" computes the inverse of a homogeneous
% transformation matrix.

tfo_inv_ortho({A,B,C,D,E,F,G,H,I,Tx,Ty,Tz})
    when is_float(A), is_float(B), is_float(C), is_float(D), is_float(E), is_float(F),
         is_float(G), is_float(H), is_float(I), is_float(Tx), is_float(Ty), is_float(Tz) ->
    {A,D,G,
     B,E,H,
     C,F,I,
     -(A * Tx + B * Ty + C * Tz),
     -(D * Tx + E * Ty + F * Tz),
     -(G * Tx + H * Ty + I * Tz)}.

% Given three points p1, p2, and p3, the function "tfo-align" computes
% a transformation matrix such that point p1 gets mapped to (0,0,0), p2 gets
% mapped to the Y axis and p3 gets mapped to the YZ plane.

tfo_align({X1,Y1,Z1},{X2,Y2,Z2},{X3,Y3,Z3})
      when is_float(X1), is_float(Y1), is_float(Z1),
           is_float(X2), is_float(Y2), is_float(Z2),
           is_float(X3), is_float(Y3), is_float(Z3) ->
      X31 = X3 - X1,
      Y31 = Y3 - Y1,
      Z31 = Z3 - Z1,
      Rotpy = pt_sub({X2,Y2,Z2},{X1,Y1,Z1}),
      Phi = pt_phi(Rotpy),
      Theta = pt_theta(Rotpy),
      Sinp = math:sin(Phi),
      Sint = math:sin(Theta),
      Cosp = math:cos(Phi),
      Cost = math:cos(Theta),
      Sinpsint = Sinp * Sint,
      Sinpcost = Sinp * Cost,
      Cospsint = Cosp * Sint,
      Cospcost = Cosp * Cost,
      Rotpz = {Cost * X31 - Sint * Z31,
                Sinpsint * X31 + Cosp * Y31 + Sinpcost * Z31,
                Cospsint * X31 - Sinp * Y31 + Cospcost * Z31},
      Rho = pt_theta(Rotpz),
      Cosr = math:cos(Rho),
      Sinr = math:sin(Rho),
      X = Z1 * Sint - X1 * Cost,
      Y = -X1 * Sinpsint - Y1 * Cosp - Z1 * Sinpcost,
      Z = Y1 * Sinp - Z1 * Cospcost - X1 * Cospsint,
      {Cost * Cosr - Cospsint * Sinr,
      Sinpsint,
      Cost * Sinr + Cospsint * Cosr,
      Sinp * Sinr,
      Cosp,
      -Sinp * Cosr,
      -Sint * Cosr - Cospcost * Sinr,
      Sinpcost,
      Cospcost * Cosr - Sint * Sinr,
      X * Cosr - Z * Sinr,
      Y,
      X * Sinr + Z * Cosr}.

% -- NUCLEIC ACID CONFORMATIONS DATA BASE ------------------------------------

% Numbering of atoms follows the paper:
%
% IUPAC-IUB Joint Commission on Biochemical Nomenclature (JCBN)
% (1983) Abbreviations and Symbols for the Description of
% Conformations of Polynucleotide Chains.  Eur. J. Biochem 131,
% 9-15.

% Define part common to all 4 nucleotide types.

%nuc ::= {
%          tfo,tfo,tfo,tfo,
%          pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,
%          pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,
%          A/C/G/U,
%          nuc_specific
%        }

%  dgf_base_tfo  ; defines the standard position for wc and wc_dumas
%  p_o3'_275_tfo ; defines the standard position for the connect function
%  p_o3'_180_tfo
%  p_o3'_60_tfo
%  p o1p o2p o5' c5' h5' h5'' c4' h4' o4' c1' h1' c2' h2'' o2' h2' c3'
%  h3' o3' n1 n3 c2 c4 c5 c6

type({_,_,_,_,_,_,_,_,_,_,
          _,_,_,_,_,_,_,_,_,_,
          _,_,_,_,_,_,_,_,_,X,
          _}) -> X.

nuc_C1_({_,_,_,_,_,_,_,_,_,_,
         _,_,_,_,X,_,_,_,_,_,
         _,_,_,_,_,_,_,_,_,_,
         _}) -> X.

nuc_C2({_,_,_,_,_,_,_,_,_,_,
        _,_,_,_,_,_,_,_,_,_,
        _,_,_,_,_,X,_,_,_,_,
        _}) -> X.

nuc_C3_({_,_,_,_,_,_,_,_,_,_,
         _,_,_,_,_,_,_,_,_,_,
         X,_,_,_,_,_,_,_,_,_,
         _}) -> X.

nuc_C4({_,_,_,_,_,_,_,_,_,_,
        _,_,_,_,_,_,_,_,_,_,
        _,_,_,_,_,_,X,_,_,_,
        _}) -> X.

nuc_C4_({_,_,_,_,_,_,_,_,_,_,
         _,X,_,_,_,_,_,_,_,_,
         _,_,_,_,_,_,_,_,_,_,
         _}) -> X.

nuc_N1({_,_,_,_,_,_,_,_,_,_,
        _,_,_,_,_,_,_,_,_,_,
        _,_,_,X,_,_,_,_,_,_,
        _}) -> X.

nuc_O3_({_,_,_,_,_,_,_,_,_,_,
         _,_,_,_,_,_,_,_,_,_,
         _,_,X,_,_,_,_,_,_,_,
         _}) -> X.

nuc_P({_,_,_,_,X,_,_,_,_,_,
       _,_,_,_,_,_,_,_,_,_,
       _,_,_,_,_,_,_,_,_,_,
       _}) -> X.

nuc_dgf_base_tfo({X,_,_,_,_,_,_,_,_,_,
                  _,_,_,_,_,_,_,_,_,_,
                  _,_,_,_,_,_,_,_,_,_,
                  _}) -> X.

nuc_p_o3__180_tfo({_,_,X,_,_,_,_,_,_,_,
                   _,_,_,_,_,_,_,_,_,_,
                   _,_,_,_,_,_,_,_,_,_,
                   _}) -> X.

nuc_p_o3__275_tfo({_,X,_,_,_,_,_,_,_,_,
                   _,_,_,_,_,_,_,_,_,_,
                   _,_,_,_,_,_,_,_,_,_,
                   _}) -> X.

nuc_p_o3__60_tfo({_,_,_,X,_,_,_,_,_,_,
                  _,_,_,_,_,_,_,_,_,_,
                  _,_,_,_,_,_,_,_,_,_,
                  _}) -> X.

rA_N9({_,_,_,_,_,_,_,_,_,_,
       _,_,_,_,_,_,_,_,_,_,
       _,_,_,_,_,_,_,_,_,a,
       {_,_,X,_,_,_,_,_}}) -> X.

rG_N9({_,_,_,_,_,_,_,_,_,_,
       _,_,_,_,_,_,_,_,_,_,
       _,_,_,_,_,_,_,_,_,g,
       {_,_,X,_,_,_,_,_,_}}) -> X.


%nuc ::= {
%          tfo,tfo,tfo,tfo,
%          pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,
%          pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,pt,
%          A/C/G/U,
%          nuc_specific
%        }

% Define remaining atoms for each nucleotide type.

%nuc_specific
% a {N6,N7,N9,C8,H2,H61,H62,H8}
% c {N4,O2,H41,H42,H5,H6}
% g {N2,N7,N9,C8,O6,H1,H21,H22,H8}
% u {O2,O4,H3,H5,H6}

% Database of nucleotide conformations:

rA() ->
     {
      {-0.0018,   -0.8207,    0.5714,  % dgf_base_tfo
        0.2679,   -0.5509,   -0.7904,
        0.9634,    0.1517,    0.2209,
        0.0073,    8.4030,    0.6232},
      {-0.8143,   -0.5091,   -0.2788,  % p_o3'_275_tfo
       -0.0433,   -0.4257,    0.9038,
       -0.5788,    0.7480,    0.3246,
        1.5227,    6.9114,   -7.0765},
      {0.3822,   -0.7477,    0.5430,  % p_o3'_180_tfo
        0.4552,    0.6637,    0.5935,
       -0.8042,    0.0203,    0.5941,
       -6.9472,   -4.1186,   -5.9108},
      {0.5640,    0.8007,   -0.2022,  % p_o3'_60_tfo
       -0.8247,    0.5587,   -0.0878,
        0.0426,    0.2162,    0.9754,
        6.2694,   -7.0540,    3.3316},
      {2.8930,    8.5380,   -3.3280}, % P   
      {1.6980,    7.6960,   -3.5570}, % O1P 
      {3.2260,    9.5010,   -4.4020}, % O2P 
      {4.1590,    7.6040,   -3.0340}, % O5' 
      {5.4550,    8.2120,   -2.8810}, % C5' 
      {5.4546,    8.8508,   -1.9978}, % H5' 
      {5.7588,    8.6625,   -3.8259}, % H5''
      {6.4970,    7.1480,   -2.5980}, % C4' 
      {7.4896,    7.5919,   -2.5214}, % H4' 
      {6.1630,    6.4860,   -1.3440}, % O4' 
      {6.5400,    5.1200,   -1.4190}, % C1' 
      {7.2763,    4.9681,   -0.6297}, % H1' 
      {7.1940,    4.8830,   -2.7770}, % C2' 
      {6.8667,    3.9183,   -3.1647}, % H2''
      {8.5860,    5.0910,   -2.6140}, % O2' 
      {8.9510,    4.7626,   -1.7890}, % H2' 
      {6.5720,    6.0040,   -3.6090}, % C3' 
      {5.5636,    5.7066,   -3.8966}, % H3' 
      {7.3801,    6.3562,   -4.7350}, % O3' 
      {4.7150,    0.4910,   -0.1360}, % N1  
      {6.3490,    2.1730,   -0.6020}, % N3  
      {5.9530,    0.9650,   -0.2670}, % C2  
      {5.2900,    2.9790,   -0.8260}, % C4  
      {3.9720,    2.6390,   -0.7330}, % C5  
      {3.6770,    1.3160,   -0.3660}, % C6
      a, {
      {2.4280,    0.8450,   -0.2360}, % N6  
      {3.1660,    3.7290,   -1.0360}, % N7  
      {5.3170,    4.2990,   -1.1930}, % N9  
      {4.0100,    4.6780,   -1.2990}, % C8  
      {6.6890,    0.1903,   -0.0518}, % H2  
      {1.6470,    1.4460,   -0.4040}, % H61 
      {2.2780,   -0.1080,   -0.0280}, % H62 
      {3.4421,    5.5744,   -1.5482}} % H8  
    }.

rA01() ->
     {
      {-0.0043,   -0.8175,    0.5759,  % dgf_base_tfo
        0.2617,   -0.5567,   -0.7884,
        0.9651,    0.1473,    0.2164,
        0.0359,    8.3929,    0.5532},
      {-0.8143,   -0.5091,   -0.2788,  % p_o3'_275_tfo
       -0.0433,   -0.4257,    0.9038,
       -0.5788,    0.7480,    0.3246,
        1.5227,    6.9114,   -7.0765},
      {0.3822,   -0.7477,    0.5430,  % p_o3'_180_tfo
        0.4552,    0.6637,    0.5935,
       -0.8042,    0.0203,    0.5941,
       -6.9472,   -4.1186,   -5.9108},
      {0.5640,    0.8007,   -0.2022,  % p_o3'_60_tfo
       -0.8247,    0.5587,   -0.0878,
        0.0426,    0.2162,    0.9754,
        6.2694,   -7.0540,    3.3316},
      {2.8930,    8.5380,   -3.3280}, % P   
      {1.6980,    7.6960,   -3.5570}, % O1P 
      {3.2260,    9.5010,   -4.4020}, % O2P 
      {4.1590,    7.6040,   -3.0340}, % O5' 
      {5.4352,    8.2183,   -2.7757}, % C5' 
      {5.3830,    8.7883,   -1.8481}, % H5' 
      {5.7729,    8.7436,   -3.6691}, % H5''
      {6.4830,    7.1518,   -2.5252}, % C4' 
      {7.4749,    7.5972,   -2.4482}, % H4' 
      {6.1626,    6.4620,   -1.2827}, % O4' 
      {6.5431,    5.0992,   -1.3905}, % C1' 
      {7.2871,    4.9328,   -0.6114}, % H1' 
      {7.1852,    4.8935,   -2.7592}, % C2' 
      {6.8573,    3.9363,   -3.1645}, % H2''
      {8.5780,    5.1025,   -2.6046}, % O2' 
      {8.9516,    4.7577,   -1.7902}, % H2' 
      {6.5522,    6.0300,   -3.5612}, % C3' 
      {5.5420,    5.7356,   -3.8459}, % H3' 
      {7.3487,    6.4089,   -4.6867}, % O3' 
      {4.7442,    0.4514,   -0.1390}, % N1  
      {6.3687,    2.1459,   -0.5926}, % N3  
      {5.9795,    0.9335,   -0.2657}, % C2  
      {5.3052,    2.9471,   -0.8125}, % C4  
      {3.9891,    2.5987,   -0.7230}, % C5  
      {3.7016,    1.2717,   -0.3647}, % C6
      a, {
      {2.4553,    0.7925,   -0.2390}, % N6  
      {3.1770,    3.6859,   -1.0198}, % N7  
      {5.3247,    4.2695,   -1.1710}, % N9  
      {4.0156,    4.6415,   -1.2759}, % C8  
      {6.7198,    0.1618,   -0.0547}, % H2  
      {1.6709,    1.3900,   -0.4039}, % H61 
      {2.3107,   -0.1627,   -0.0373}, % H62 
      {3.4426,    5.5361,   -1.5199}} % H8  
     }.

rA02() ->
     {
      {0.5566,    0.0449,    0.8296,  % dgf_base_tfo
        0.5125,    0.7673,   -0.3854,
       -0.6538,    0.6397,    0.4041,
       -9.1161,   -3.7679,   -2.9968},
      {-0.8143,   -0.5091,   -0.2788,  % p_o3'_275_tfo
       -0.0433,   -0.4257,    0.9038,
       -0.5788,    0.7480,    0.3246,
        1.5227,    6.9114,   -7.0765},
      {0.3822,   -0.7477,    0.5430,  % p_o3'_180_tfo
        0.4552,    0.6637,    0.5935,
       -0.8042,    0.0203,    0.5941,
       -6.9472,   -4.1186,   -5.9108},
      {0.5640,    0.8007,   -0.2022,  % p_o3'_60_tfo
       -0.8247,    0.5587,   -0.0878,
        0.0426,    0.2162,    0.9754,
        6.2694,   -7.0540,    3.3316},
      {2.8930,    8.5380,   -3.3280}, % P   
      {1.6980,    7.6960,   -3.5570}, % O1P 
      {3.2260,    9.5010,   -4.4020}, % O2P 
      {4.1590,    7.6040,   -3.0340}, % O5' 
      {4.5778,    6.6594,   -4.0364}, % C5' 
      {4.9220,    7.1963,   -4.9204}, % H5' 
      {3.7996,    5.9091,   -4.1764}, % H5''
      {5.7873,    5.8869,   -3.5482}, % C4' 
      {6.0405,    5.0875,   -4.2446}, % H4' 
      {6.9135,    6.8036,   -3.4310}, % O4' 
      {7.7293,    6.4084,   -2.3392}, % C1' 
      {8.7078,    6.1815,   -2.7624}, % H1' 
      {7.1305,    5.1418,   -1.7347}, % C2' 
      {7.2040,    5.1982,   -0.6486}, % H2''
      {7.7417,    4.0392,   -2.3813}, % O2' 
      {8.6785,    4.1443,   -2.5630}, % H2' 
      {5.6666,    5.2728,   -2.1536}, % C3' 
      {5.1747,    5.9805,   -1.4863}, % H3' 
      {4.9997,    4.0086,   -2.1973}, % O3' 
      {10.3245,    8.5459,    1.5467}, % N1  
      {9.8051,    6.9432,   -0.1497}, % N3  
      {10.5175,    7.4328,    0.8408}, % C2  
      {8.7523,    7.7422,   -0.4228}, % C4  
      {8.4257,    8.9060,    0.2099}, % C5  
      {9.2665,    9.3242,    1.2540}, % C6
      a, {
      {9.0664,   10.4462,    1.9610}, % N6  
      {7.2750,    9.4537,   -0.3428}, % N7  
      {7.7962,    7.5519,   -1.3859}, % N9  
      {6.9479,    8.6157,   -1.2771}, % C8  
      {11.4063,    6.9047,    1.1859}, % H2  
      {8.2845,   11.0341,    1.7552}, % H61 
      {9.6584,   10.6647,    2.7198}, % H62 
      {6.0430,    8.9853,   -1.7594}} % H8  
     }.

rA03() ->
     {
      {-0.5021,    0.0731,    0.8617,  % dgf_base_tfo
       -0.8112,    0.3054,   -0.4986,
       -0.2996,   -0.9494,   -0.0940,
        6.4273,   -5.1944,   -3.7807},
      {-0.8143,   -0.5091,   -0.2788,  % p_o3'_275_tfo
       -0.0433,   -0.4257,    0.9038,
       -0.5788,    0.7480,    0.3246,
        1.5227,    6.9114,   -7.0765},
      {0.3822,   -0.7477,    0.5430,  % p_o3'_180_tfo
        0.4552,    0.6637,    0.5935,
       -0.8042,    0.0203,    0.5941,
       -6.9472,   -4.1186,   -5.9108},
      {0.5640,    0.8007,   -0.2022,  % p_o3'_60_tfo
       -0.8247,    0.5587,   -0.0878,
        0.0426,    0.2162,    0.9754,
        6.2694,   -7.0540,    3.3316},
      {2.8930,    8.5380,   -3.3280}, % P   
      {1.6980,    7.6960,   -3.5570}, % O1P 
      {3.2260,    9.5010,   -4.4020}, % O2P 
      {4.1590,    7.6040,   -3.0340}, % O5' 
      {4.1214,    6.7116,   -1.9049}, % C5' 
      {3.3465,    5.9610,   -2.0607}, % H5' 
      {4.0789,    7.2928,   -0.9837}, % H5''
      {5.4170,    5.9293,   -1.8186}, % C4' 
      {5.4506,    5.3400,   -0.9023}, % H4' 
      {5.5067,    5.0417,   -2.9703}, % O4' 
      {6.8650,    4.9152,   -3.3612}, % C1' 
      {7.1090,    3.8577,   -3.2603}, % H1' 
      {7.7152,    5.7282,   -2.3894}, % C2' 
      {8.5029,    6.2356,   -2.9463}, % H2''
      {8.1036,    4.8568,   -1.3419}, % O2' 
      {8.3270,    3.9651,   -1.6184}, % H2' 
      {6.7003,    6.7565,   -1.8911}, % C3' 
      {6.5898,    7.5329,   -2.6482}, % H3' 
      {7.0505,    7.2878,   -0.6105}, % O3' 
      {9.6740,    4.7656,   -7.6614}, % N1  
      {9.0739,    4.3013,   -5.3941}, % N3  
      {9.8416,    4.2192,   -6.4581}, % C2  
      {7.9885,    5.0632,   -5.6446}, % C4  
      {7.6822,    5.6856,   -6.8194}, % C5  
      {8.5831,    5.5215,   -7.8840}, % C6
      a, {
      {8.4084,    6.0747,   -9.0933}, % N6  
      {6.4857,    6.3816,   -6.7035}, % N7  
      {6.9740,    5.3703,   -4.7760}, % N9  
      {6.1133,    6.1613,   -5.4808}, % C8  
      {10.7627,    3.6375,   -6.4220}, % H2  
      {7.6031,    6.6390,   -9.2733}, % H61 
      {9.1004,    5.9708,   -9.7893}, % H62 
      {5.1705,    6.6830,   -5.3167}} % H8  
     }.

rA04() ->
     {
      {-0.5426,   -0.8175,    0.1929,  % dgf_base_tfo
        0.8304,   -0.5567,   -0.0237,
        0.1267,    0.1473,    0.9809,
       -0.5075,    8.3929,    0.2229},
      {-0.8143,   -0.5091,   -0.2788,  % p_o3'_275_tfo
       -0.0433,   -0.4257,    0.9038,
       -0.5788,    0.7480,    0.3246,
        1.5227,    6.9114,   -7.0765},
      {0.3822,   -0.7477,    0.5430,  % p_o3'_180_tfo
        0.4552,    0.6637,    0.5935,
       -0.8042,    0.0203,    0.5941,
       -6.9472,   -4.1186,   -5.9108},
      {0.5640,    0.8007,   -0.2022,  % p_o3'_60_tfo
       -0.8247,    0.5587,   -0.0878,
        0.0426,    0.2162,    0.9754,
        6.2694,   -7.0540,    3.3316},
      {2.8930,    8.5380,   -3.3280}, % P   
      {1.6980,    7.6960,   -3.5570}, % O1P 
      {3.2260,    9.5010,   -4.4020}, % O2P 
      {4.1590,    7.6040,   -3.0340}, % O5' 
      {5.4352,    8.2183,   -2.7757}, % C5' 
      {5.3830,    8.7883,   -1.8481}, % H5' 
      {5.7729,    8.7436,   -3.6691}, % H5''
      {6.4830,    7.1518,   -2.5252}, % C4' 
      {7.4749,    7.5972,   -2.4482}, % H4' 
      {6.1626,    6.4620,   -1.2827}, % O4' 
      {6.5431,    5.0992,   -1.3905}, % C1' 
      {7.2871,    4.9328,   -0.6114}, % H1' 
      {7.1852,    4.8935,   -2.7592}, % C2' 
      {6.8573,    3.9363,   -3.1645}, % H2''
      {8.5780,    5.1025,   -2.6046}, % O2' 
      {8.9516,    4.7577,   -1.7902}, % H2' 
      {6.5522,    6.0300,   -3.5612}, % C3' 
      {5.5420,    5.7356,   -3.8459}, % H3' 
      {7.3487,    6.4089,   -4.6867}, % O3' 
      {3.6343,    2.6680,    2.0783}, % N1  
      {5.4505,    3.9805,    1.2446}, % N3  
      {4.7540,    3.3816,    2.1851}, % C2  
      {4.8805,    3.7951,    0.0354}, % C4  
      {3.7416,    3.0925,   -0.2305}, % C5  
      {3.0873,    2.4980,    0.8606}, % C6
      a, {
      {1.9600,    1.7805,    0.7462}, % N6  
      {3.4605,    3.1184,   -1.5906}, % N7  
      {5.3247,    4.2695,   -1.1710}, % N9  
      {4.4244,    3.8244,   -2.0953}, % C8  
      {5.0814,    3.4352,    3.2234}, % H2  
      {1.5423,    1.6454,   -0.1520}, % H61 
      {1.5716,    1.3398,    1.5392}, % H62 
      {4.2675,    3.8876,   -3.1721}} % H8  
     }.

rA05() ->
     {
      {-0.5891,    0.0449,    0.8068,  % dgf_base_tfo
        0.5375,    0.7673,    0.3498,
       -0.6034,    0.6397,   -0.4762,
       -0.3019,   -3.7679,   -9.5913},
      {-0.8143,   -0.5091,   -0.2788,  % p_o3'_275_tfo
       -0.0433,   -0.4257,    0.9038,
       -0.5788,    0.7480,    0.3246,
        1.5227,    6.9114,   -7.0765},
      {0.3822,   -0.7477,    0.5430,  % p_o3'_180_tfo
        0.4552,    0.6637,    0.5935,
       -0.8042,    0.0203,    0.5941,
       -6.9472,   -4.1186,   -5.9108},
      {0.5640,    0.8007,   -0.2022,  % p_o3'_60_tfo
       -0.8247,    0.5587,   -0.0878,
        0.0426,    0.2162,    0.9754,
        6.2694,   -7.0540,    3.3316},
      {2.8930,    8.5380,   -3.3280}, % P   
      {1.6980,    7.6960,   -3.5570}, % O1P 
      {3.2260,    9.5010,   -4.4020}, % O2P 
      {4.1590,    7.6040,   -3.0340}, % O5' 
      {4.5778,    6.6594,   -4.0364}, % C5' 
      {4.9220,    7.1963,   -4.9204}, % H5' 
      {3.7996,    5.9091,   -4.1764}, % H5''
      {5.7873,    5.8869,   -3.5482}, % C4' 
      {6.0405,    5.0875,   -4.2446}, % H4' 
      {6.9135,    6.8036,   -3.4310}, % O4' 
      {7.7293,    6.4084,   -2.3392}, % C1' 
      {8.7078,    6.1815,   -2.7624}, % H1' 
      {7.1305,    5.1418,   -1.7347}, % C2' 
      {7.2040,    5.1982,   -0.6486}, % H2''
      {7.7417,    4.0392,   -2.3813}, % O2' 
      {8.6785,    4.1443,   -2.5630}, % H2' 
      {5.6666,    5.2728,   -2.1536}, % C3' 
      {5.1747,    5.9805,   -1.4863}, % H3' 
      {4.9997,    4.0086,   -2.1973}, % O3' 
      {10.2594,   10.6774,   -1.0056}, % N1  
      {9.7528,    8.7080,   -2.2631}, % N3  
      {10.4471,    9.7876,   -1.9791}, % C2  
      {8.7271,    8.5575,   -1.3991}, % C4  
      {8.4100,    9.3803,   -0.3580}, % C5  
      {9.2294,   10.5030,   -0.1574}, % C6
      a, {
      {9.0349,   11.3951,    0.8250}, % N6  
      {7.2891,    8.9068,    0.3121}, % N7  
      {7.7962,    7.5519,   -1.3859}, % N9  
      {6.9702,    7.8292,   -0.3353}, % C8  
      {11.3132,   10.0537,   -2.5851}, % H2  
      {8.2741,   11.2784,    1.4629}, % H61 
      {9.6733,   12.1368,    0.9529}, % H62 
      {6.0888,    7.3990,    0.1403}} % H8  
    }.

rA06() ->
     {
      {-0.9815,    0.0731,   -0.1772,  % dgf_base_tfo
        0.1912,    0.3054,   -0.9328,
       -0.0141,   -0.9494,   -0.3137,
        5.7506,   -5.1944,    4.7470},
      {-0.8143,   -0.5091,   -0.2788,  % p_o3'_275_tfo
       -0.0433,   -0.4257,    0.9038,
       -0.5788,    0.7480,    0.3246,
        1.5227,    6.9114,   -7.0765},
      {0.3822,   -0.7477,    0.5430,  % p_o3'_180_tfo
        0.4552,    0.6637,    0.5935,
       -0.8042,    0.0203,    0.5941,
       -6.9472,   -4.1186,   -5.9108},
      {0.5640,    0.8007,   -0.2022,  % p_o3'_60_tfo
       -0.8247,    0.5587,   -0.0878,
        0.0426,    0.2162,    0.9754,
        6.2694,   -7.0540,    3.3316},
      {2.8930,    8.5380,   -3.3280}, % P   
      {1.6980,    7.6960,   -3.5570}, % O1P 
      {3.2260,    9.5010,   -4.4020}, % O2P 
      {4.1590,    7.6040,   -3.0340}, % O5' 
      {4.1214,    6.7116,   -1.9049}, % C5' 
      {3.3465,    5.9610,   -2.0607}, % H5' 
      {4.0789,    7.2928,   -0.9837}, % H5''
      {5.4170,    5.9293,   -1.8186}, % C4' 
      {5.4506,    5.3400,   -0.9023}, % H4' 
      {5.5067,    5.0417,   -2.9703}, % O4' 
      {6.8650,    4.9152,   -3.3612}, % C1' 
      {7.1090,    3.8577,   -3.2603}, % H1' 
      {7.7152,    5.7282,   -2.3894}, % C2' 
      {8.5029,    6.2356,   -2.9463}, % H2''
      {8.1036,    4.8568,   -1.3419}, % O2' 
      {8.3270,    3.9651,   -1.6184}, % H2' 
      {6.7003,    6.7565,   -1.8911}, % C3' 
      {6.5898,    7.5329,   -2.6482}, % H3' 
      {7.0505,    7.2878,   -0.6105}, % O3' 
      {6.6624,    3.5061,   -8.2986}, % N1  
      {6.5810,    3.2570,   -5.9221}, % N3  
      {6.5151,    2.8263,   -7.1625}, % C2  
      {6.8364,    4.5817,   -5.8882}, % C4  
      {7.0116,    5.4064,   -6.9609}, % C5  
      {6.9173,    4.8260,   -8.2361}, % C6
      a, {
      {7.0668,    5.5163,   -9.3763}, % N6  
      {7.2573,    6.7070,   -6.5394}, % N7  
      {6.9740,    5.3703,   -4.7760}, % N9  
      {7.2238,    6.6275,   -5.2453}, % C8  
      {6.3146,    1.7741,   -7.3641}, % H2  
      {7.2568,    6.4972,   -9.3456}, % H61 
      {7.0437,    5.0478,  -10.2446}, % H62 
      {7.4108,    7.6227,   -4.8418}} % H8  
    }.

rA07() ->
     {
      {0.2379,    0.1310,   -0.9624,  % dgf_base_tfo
       -0.5876,   -0.7696,   -0.2499,
       -0.7734,    0.6249,   -0.1061,
       30.9870,  -26.9344,   42.6416},
      {0.7529,    0.1548,    0.6397,  % p_o3'_275_tfo
        0.2952,   -0.9481,   -0.1180,
        0.5882,    0.2777,   -0.7595,
       -58.8919,  -11.3095,    6.0866},
      {-0.0239,    0.9667,   -0.2546,  % p_o3'_180_tfo
        0.9731,   -0.0359,   -0.2275,
       -0.2290,   -0.2532,   -0.9399,
        3.5401,  -29.7913,   52.2796},
      {-0.8912,   -0.4531,    0.0242,  % p_o3'_60_tfo
       -0.1183,    0.1805,   -0.9764,
        0.4380,   -0.8730,   -0.2145,
       19.9023,   54.8054,   15.2799},
      {41.8210,    8.3880,   43.5890}, % P   
      {42.5400,    8.0450,   44.8330}, % O1P 
      {42.2470,    9.6920,   42.9910}, % O2P 
      {40.2550,    8.2030,   43.7340}, % O5' 
      {39.3505,    8.4697,   42.6565}, % C5' 
      {39.1377,    7.5433,   42.1230}, % H5' 
      {39.7203,    9.3119,   42.0717}, % H5''
      {38.0405,    8.9195,   43.2869}, % C4' 
      {37.3687,    9.3036,   42.5193}, % H4' 
      {37.4319,    7.8146,   43.9387}, % O4' 
      {37.1959,    8.1354,   45.3237}, % C1' 
      {36.1788,    8.5202,   45.3970}, % H1' 
      {38.1721,    9.2328,   45.6504}, % C2' 
      {39.1555,    8.7939,   45.8188}, % H2''
      {37.7862,   10.0617,   46.7013}, % O2' 
      {37.3087,    9.6229,   47.4092}, % H2' 
      {38.1844,   10.0268,   44.3367}, % C3' 
      {39.1578,   10.5054,   44.2289}, % H3' 
      {37.0547,   10.9127,   44.3441}, % O3' 
      {34.8811,    4.2072,   47.5784}, % N1  
      {35.1084,    6.1336,   46.1818}, % N3  
      {34.4108,    5.1360,   46.7207}, % C2  
      {36.3908,    6.1224,   46.6053}, % C4  
      {36.9819,    5.2334,   47.4697}, % C5  
      {36.1786,    4.1985,   48.0035}, % C6
      a, {
      {36.6103,    3.2749,   48.8452}, % N6  
      {38.3236,    5.5522,   47.6595}, % N7  
      {37.3887,    7.0024,   46.2437}, % N9  
      {38.5055,    6.6096,   46.9057}, % C8  
      {33.3553,    5.0152,   46.4771}, % H2  
      {37.5730,    3.2804,   49.1507}, % H61 
      {35.9775,    2.5638,   49.1828}, % H62 
      {39.5461,    6.9184,   47.0041}} % H8  
    }.

rA08() ->
     {
      {0.1084,   -0.0895,   -0.9901,  % dgf_base_tfo
        0.9789,   -0.1638,    0.1220,
       -0.1731,   -0.9824,    0.0698,
       -2.9039,   47.2655,   33.0094},
      {0.7529,    0.1548,    0.6397,  % p_o3'_275_tfo
        0.2952,   -0.9481,   -0.1180,
        0.5882,    0.2777,   -0.7595,
       -58.8919,  -11.3095,    6.0866},
      {-0.0239,    0.9667,   -0.2546,  % p_o3'_180_tfo
        0.9731,   -0.0359,   -0.2275,
       -0.2290,   -0.2532,   -0.9399,
        3.5401,  -29.7913,   52.2796},
      {-0.8912,   -0.4531,    0.0242,  % p_o3'_60_tfo
       -0.1183,    0.1805,   -0.9764,
        0.4380,   -0.8730,   -0.2145,
       19.9023,   54.8054,   15.2799},
      {41.8210,    8.3880,   43.5890}, % P   
      {42.5400,    8.0450,   44.8330}, % O1P 
      {42.2470,    9.6920,   42.9910}, % O2P 
      {40.2550,    8.2030,   43.7340}, % O5' 
      {39.4850,    8.9301,   44.6977}, % C5' 
      {39.0638,    9.8199,   44.2296}, % H5' 
      {40.0757,    9.0713,   45.6029}, % H5''
      {38.3102,    8.0414,   45.0789}, % C4' 
      {37.7842,    8.4637,   45.9351}, % H4' 
      {37.4200,    7.9453,   43.9769}, % O4' 
      {37.2249,    6.5609,   43.6273}, % C1' 
      {36.3360,    6.2168,   44.1561}, % H1' 
      {38.4347,    5.8414,   44.1590}, % C2' 
      {39.2688,    5.9974,   43.4749}, % H2''
      {38.2344,    4.4907,   44.4348}, % O2' 
      {37.6374,    4.0386,   43.8341}, % H2' 
      {38.6926,    6.6079,   45.4637}, % C3' 
      {39.7585,    6.5640,   45.6877}, % H3' 
      {37.8238,    6.0705,   46.4723}, % O3' 
      {33.9162,    6.2598,   39.7758}, % N1  
      {34.6709,    6.5759,   42.0215}, % N3  
      {33.7257,    6.5186,   41.0858}, % C2  
      {35.8935,    6.3324,   41.5018}, % C4  
      {36.2105,    6.0601,   40.1932}, % C5  
      {35.1538,    6.0151,   39.2537}, % C6
      a, {
      {35.3088,    5.7642,   37.9649}, % N6  
      {37.5818,    5.8677,   40.0507}, % N7  
      {37.0932,    6.3197,   42.1810}, % N9  
      {38.0509,    6.0354,   41.2635}, % C8  
      {32.6830,    6.6898,   41.3532}, % H2  
      {36.2305,    5.5855,   37.5925}, % H61 
      {34.5056,    5.7512,   37.3528}, % H62 
      {39.1318,    5.8993,   41.2285}} % H8  
    }.

rA09() ->
     {
      {0.8467,    0.4166,   -0.3311,  % dgf_base_tfo
       -0.3962,    0.9089,    0.1303,
        0.3552,    0.0209,    0.9346,
       -42.7319,  -26.6223,  -29.8163},
      {0.7529,    0.1548,    0.6397,  % p_o3'_275_tfo
        0.2952,   -0.9481,   -0.1180,
        0.5882,    0.2777,   -0.7595,
       -58.8919,  -11.3095,    6.0866},
      {-0.0239,    0.9667,   -0.2546,  % p_o3'_180_tfo
        0.9731,   -0.0359,   -0.2275,
       -0.2290,   -0.2532,   -0.9399,
        3.5401,  -29.7913,   52.2796},
      {-0.8912,   -0.4531,    0.0242,  % p_o3'_60_tfo
       -0.1183,    0.1805,   -0.9764,
        0.4380,   -0.8730,   -0.2145,
       19.9023,   54.8054,   15.2799},
      {41.8210,    8.3880,   43.5890}, % P   
      {42.5400,    8.0450,   44.8330}, % O1P 
      {42.2470,    9.6920,   42.9910}, % O2P 
      {40.2550,    8.2030,   43.7340}, % O5' 
      {39.3505,    8.4697,   42.6565}, % C5' 
      {39.1377,    7.5433,   42.1230}, % H5' 
      {39.7203,    9.3119,   42.0717}, % H5''
      {38.0405,    8.9195,   43.2869}, % C4' 
      {37.6479,    8.1347,   43.9335}, % H4' 
      {38.2691,   10.0933,   44.0524}, % O4' 
      {37.3999,   11.1488,   43.5973}, % C1' 
      {36.5061,   11.1221,   44.2206}, % H1' 
      {37.0364,   10.7838,   42.1836}, % C2' 
      {37.8636,   11.0489,   41.5252}, % H2''
      {35.8275,   11.3133,   41.7379}, % O2' 
      {35.6214,   12.1896,   42.0714}, % H2' 
      {36.9316,    9.2556,   42.2837}, % C3' 
      {37.1778,    8.8260,   41.3127}, % H3' 
      {35.6285,    8.9334,   42.7926}, % O3' 
      {38.1482,   15.2833,   46.4641}, % N1  
      {37.3641,   13.0968,   45.9007}, % N3  
      {37.5032,   14.1288,   46.7300}, % C2  
      {37.9570,   13.3377,   44.7113}, % C4  
      {38.6397,   14.4660,   44.3267}, % C5  
      {38.7473,   15.5229,   45.2609}, % C6
      a, {
      {39.3720,   16.6649,   45.0297}, % N6  
      {39.1079,   14.3351,   43.0223}, % N7  
      {38.0132,   12.4868,   43.6280}, % N9  
      {38.7058,   13.1402,   42.6620}, % C8  
      {37.0731,   14.0857,   47.7306}, % H2  
      {39.8113,   16.8281,   44.1350}, % H61 
      {39.4100,   17.3741,   45.7478}, % H62 
      {39.0412,   12.9660,   41.6397}} % H8  
    }.

rA10() ->
     {
      {0.7063,    0.6317,   -0.3196,  % dgf_base_tfo
       -0.0403,   -0.4149,   -0.9090,
       -0.7068,    0.6549,   -0.2676,
        6.4402,  -52.1496,   30.8246},
      {0.7529,    0.1548,    0.6397,  % p_o3'_275_tfo
        0.2952,   -0.9481,   -0.1180,
        0.5882,    0.2777,   -0.7595,
       -58.8919,  -11.3095,    6.0866},
      {-0.0239,    0.9667,   -0.2546,  % p_o3'_180_tfo
        0.9731,   -0.0359,   -0.2275,
       -0.2290,   -0.2532,   -0.9399,
        3.5401,  -29.7913,   52.2796},
      {-0.8912,   -0.4531,    0.0242,  % p_o3'_60_tfo
       -0.1183,    0.1805,   -0.9764,
        0.4380,   -0.8730,   -0.2145,
       19.9023,   54.8054,   15.2799},
      {41.8210,    8.3880,   43.5890}, % P   
      {42.5400,    8.0450,   44.8330}, % O1P 
      {42.2470,    9.6920,   42.9910}, % O2P 
      {40.2550,    8.2030,   43.7340}, % O5' 
      {39.4850,    8.9301,   44.6977}, % C5' 
      {39.0638,    9.8199,   44.2296}, % H5' 
      {40.0757,    9.0713,   45.6029}, % H5''
      {38.3102,    8.0414,   45.0789}, % C4' 
      {37.7099,    7.8166,   44.1973}, % H4' 
      {38.8012,    6.8321,   45.6380}, % O4' 
      {38.2431,    6.6413,   46.9529}, % C1' 
      {37.3505,    6.0262,   46.8385}, % H1' 
      {37.8484,    8.0156,   47.4214}, % C2' 
      {38.7381,    8.5406,   47.7690}, % H2''
      {36.8286,    8.0368,   48.3701}, % O2' 
      {36.8392,    7.3063,   48.9929}, % H2' 
      {37.3576,    8.6512,   46.1132}, % C3' 
      {37.5207,    9.7275,   46.1671}, % H3' 
      {35.9985,    8.2392,   45.9032}, % O3' 
      {39.9117,    2.2278,   48.8527}, % N1  
      {38.6207,    3.6941,   47.4757}, % N3  
      {38.9872,    2.4888,   47.9057}, % C2  
      {39.2961,    4.6720,   48.1174}, % C4  
      {40.2546,    4.5307,   49.0912}, % C5  
      {40.5932,    3.2189,   49.4985}, % C6
      a, {
      {41.4938,    2.9317,   50.4229}, % N6  
      {40.7195,    5.7755,   49.5060}, % N7  
      {39.1730,    6.0305,   47.9170}, % N9  
      {40.0413,    6.6250,   48.7728}, % C8  
      {38.5257,    1.5960,   47.4838}, % H2  
      {41.9907,    3.6753,   50.8921}, % H61 
      {41.6848,    1.9687,   50.6599}, % H62 
      {40.3571,    7.6321,   49.0452}} % H8  
    }.

rAs() -> [rA01(),rA02(),rA03(),rA04(),rA05(),rA06(),rA07(),
          rA08(),rA09(),rA10()].

rC() ->
     {
      {-0.0359,   -0.8071,    0.5894,  % dgf_base_tfo
       -0.2669,    0.5761,    0.7726,
       -0.9631,   -0.1296,   -0.2361,
        0.1584,    8.3434,    0.5434},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {5.2430,   -8.2420,    2.8260}, % C5' 
      {5.1974,   -8.8497,    1.9223}, % H5' 
      {5.5548,   -8.7348,    3.7469}, % H5''
      {6.3140,   -7.2060,    2.5510}, % C4' 
      {7.2954,   -7.6762,    2.4898}, % H4' 
      {6.0140,   -6.5420,    1.2890}, % O4' 
      {6.4190,   -5.1840,    1.3620}, % C1' 
      {7.1608,   -5.0495,    0.5747}, % H1' 
      {7.0760,   -4.9560,    2.7270}, % C2' 
      {6.7770,   -3.9803,    3.1099}, % H2''
      {8.4500,   -5.1930,    2.5810}, % O2' 
      {8.8309,   -4.8755,    1.7590}, % H2' 
      {6.4060,   -6.0590,    3.5580}, % C3' 
      {5.4021,   -5.7313,    3.8281}, % H3' 
      {7.1570,   -6.4240,    4.7070}, % O3' 
      {5.2170,   -4.3260,    1.1690}, % N1  
      {4.2960,   -2.2560,    0.6290}, % N3  
      {5.4330,   -3.0200,    0.7990}, % C2  
      {2.9930,   -2.6780,    0.7940}, % C4  
      {2.8670,   -4.0630,    1.1830}, % C5  
      {3.9570,   -4.8300,    1.3550}, % C6
      c, {
      {2.0187,   -1.8047,    0.5874}, % N4  
      {6.5470,   -2.5560,    0.6290}, % O2  
      {1.0684,   -2.1236,    0.7109}, % H41 
      {2.2344,   -0.8560,    0.3162}, % H42 
      {1.8797,   -4.4972,    1.3404}, % H5  
      {3.8479,   -5.8742,    1.6480}} % H6  
    }.

rC01() ->
     {
      {-0.0137,   -0.8012,    0.5983,  % dgf_base_tfo
       -0.2523,    0.5817,    0.7733,
       -0.9675,   -0.1404,   -0.2101,
        0.2031,    8.3874,    0.4228},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {5.2416,   -8.2422,    2.8181}, % C5' 
      {5.2050,   -8.8128,    1.8901}, % H5' 
      {5.5368,   -8.7738,    3.7227}, % H5''
      {6.3232,   -7.2037,    2.6002}, % C4' 
      {7.3048,   -7.6757,    2.5577}, % H4' 
      {6.0635,   -6.5092,    1.3456}, % O4' 
      {6.4697,   -5.1547,    1.4629}, % C1' 
      {7.2354,   -5.0043,    0.7018}, % H1' 
      {7.0856,   -4.9610,    2.8521}, % C2' 
      {6.7777,   -3.9935,    3.2487}, % H2''
      {8.4627,   -5.1992,    2.7423}, % O2' 
      {8.8693,   -4.8638,    1.9399}, % H2' 
      {6.3877,   -6.0809,    3.6362}, % C3' 
      {5.3770,   -5.7562,    3.8834}, % H3' 
      {7.1024,   -6.4754,    4.7985}, % O3' 
      {5.2764,   -4.2883,    1.2538}, % N1  
      {4.3777,   -2.2062,    0.7229}, % N3  
      {5.5069,   -2.9779,    0.9088}, % C2  
      {3.0693,   -2.6246,    0.8500}, % C4  
      {2.9279,   -4.0146,    1.2149}, % C5  
      {4.0101,   -4.7892,    1.4017}, % C6
      c, {
      {2.1040,   -1.7437,    0.6331}, % N4  
      {6.6267,   -2.5166,    0.7728}, % O2  
      {1.1496,   -2.0600,    0.7287}, % H41 
      {2.3303,   -0.7921,    0.3815}, % H42 
      {1.9353,   -4.4465,    1.3419}, % H5  
      {3.8895,   -5.8371,    1.6762}} % H6  
    }.

rC02() ->
     {
      {0.5141,    0.0246,    0.8574,  % dgf_base_tfo
       -0.5547,   -0.7529,    0.3542,
        0.6542,   -0.6577,   -0.3734,
       -9.1111,   -3.4598,   -3.2939},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {4.3825,   -6.6585,    4.0489}, % C5' 
      {4.6841,   -7.2019,    4.9443}, % H5' 
      {3.6189,   -5.8889,    4.1625}, % H5''
      {5.6255,   -5.9175,    3.5998}, % C4' 
      {5.8732,   -5.1228,    4.3034}, % H4' 
      {6.7337,   -6.8605,    3.5222}, % O4' 
      {7.5932,   -6.4923,    2.4548}, % C1' 
      {8.5661,   -6.2983,    2.9064}, % H1' 
      {7.0527,   -5.2012,    1.8322}, % C2' 
      {7.1627,   -5.2525,    0.7490}, % H2''
      {7.6666,   -4.1249,    2.4880}, % O2' 
      {8.5944,   -4.2543,    2.6981}, % H2' 
      {5.5661,   -5.3029,    2.2009}, % C3' 
      {5.0841,   -6.0018,    1.5172}, % H3' 
      {4.9062,   -4.0452,    2.2042}, % O3' 
      {7.6298,   -7.6136,    1.4752}, % N1  
      {8.6945,   -8.7046,   -0.2857}, % N3  
      {8.6943,   -7.6514,    0.6066}, % C2  
      {7.7426,   -9.6987,   -0.3801}, % C4  
      {6.6642,   -9.5742,    0.5722}, % C5  
      {6.6391,   -8.5592,    1.4526}, % C6
      c, {
      {7.9033,  -10.6371,   -1.3010}, % N4  
      {9.5840,   -6.8186,    0.6136}, % O2  
      {7.2009,  -11.3604,   -1.3619}, % H41 
      {8.7058,  -10.6168,   -1.9140}, % H42 
      {5.8585,  -10.3083,    0.5822}, % H5  
      {5.8197,   -8.4773,    2.1667}} % H6  
    }.

rC03() ->
     {
      {-0.4993,    0.0476,    0.8651,  % dgf_base_tfo
        0.8078,   -0.3353,    0.4847,
        0.3132,    0.9409,    0.1290,
        6.2989,   -5.2303,   -3.8577},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {3.9938,   -6.7042,    1.9023}, % C5' 
      {3.2332,   -5.9343,    2.0319}, % H5' 
      {3.9666,   -7.2863,    0.9812}, % H5''
      {5.3098,   -5.9546,    1.8564}, % C4' 
      {5.3863,   -5.3702,    0.9395}, % H4' 
      {5.3851,   -5.0642,    3.0076}, % O4' 
      {6.7315,   -4.9724,    3.4462}, % C1' 
      {7.0033,   -3.9202,    3.3619}, % H1' 
      {7.5997,   -5.8018,    2.4948}, % C2' 
      {8.3627,   -6.3254,    3.0707}, % H2''
      {8.0410,   -4.9501,    1.4724}, % O2' 
      {8.2781,   -4.0644,    1.7570}, % H2' 
      {6.5701,   -6.8129,    1.9714}, % C3' 
      {6.4186,   -7.5809,    2.7299}, % H3' 
      {6.9357,   -7.3841,    0.7235}, % O3' 
      {6.8024,   -5.4718,    4.8475}, % N1  
      {7.9218,   -5.5700,    6.8877}, % N3  
      {7.8908,   -5.0886,    5.5944}, % C2  
      {6.9789,   -6.3827,    7.4823}, % C4  
      {5.8742,   -6.7319,    6.6202}, % C5  
      {5.8182,   -6.2769,    5.3570}, % C6
      c, {
      {7.1702,   -6.7511,    8.7402}, % N4  
      {8.7747,   -4.3728,    5.1568}, % O2  
      {6.4741,   -7.3461,    9.1662}, % H41 
      {7.9889,   -6.4396,    9.2429}, % H42 
      {5.0736,   -7.3713,    6.9922}, % H5  
      {4.9784,   -6.5473,    4.7170}} % H6  
    }.

rC04() ->
     {
      {-0.5669,   -0.8012,    0.1918,  % dgf_base_tfo
       -0.8129,    0.5817,    0.0273,
       -0.1334,   -0.1404,   -0.9811,
       -0.3279,    8.3874,    0.3355},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {5.2416,   -8.2422,    2.8181}, % C5' 
      {5.2050,   -8.8128,    1.8901}, % H5' 
      {5.5368,   -8.7738,    3.7227}, % H5''
      {6.3232,   -7.2037,    2.6002}, % C4' 
      {7.3048,   -7.6757,    2.5577}, % H4' 
      {6.0635,   -6.5092,    1.3456}, % O4' 
      {6.4697,   -5.1547,    1.4629}, % C1' 
      {7.2354,   -5.0043,    0.7018}, % H1' 
      {7.0856,   -4.9610,    2.8521}, % C2' 
      {6.7777,   -3.9935,    3.2487}, % H2''
      {8.4627,   -5.1992,    2.7423}, % O2' 
      {8.8693,   -4.8638,    1.9399}, % H2' 
      {6.3877,   -6.0809,    3.6362}, % C3' 
      {5.3770,   -5.7562,    3.8834}, % H3' 
      {7.1024,   -6.4754,    4.7985}, % O3' 
      {5.2764,   -4.2883,    1.2538}, % N1  
      {3.8961,   -3.0896,   -0.1893}, % N3  
      {5.0095,   -3.8907,   -0.0346}, % C2  
      {3.0480,   -2.6632,    0.8116}, % C4  
      {3.4093,   -3.1310,    2.1292}, % C5  
      {4.4878,   -3.9124,    2.3088}, % C6
      c, {
      {2.0216,   -1.8941,    0.4804}, % N4  
      {5.7005,   -4.2164,   -0.9842}, % O2  
      {1.4067,   -1.5873,    1.2205}, % H41 
      {1.8721,   -1.6319,   -0.4835}, % H42 
      {2.8048,   -2.8507,    2.9918}, % H5  
      {4.7491,   -4.2593,    3.3085}} % H6  
    }.

rC05() ->
     {
      {-0.6298,    0.0246,    0.7763,  % dgf_base_tfo
       -0.5226,   -0.7529,   -0.4001,
        0.5746,   -0.6577,    0.4870,
       -0.0208,   -3.4598,   -9.6882},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {4.3825,   -6.6585,    4.0489}, % C5' 
      {4.6841,   -7.2019,    4.9443}, % H5' 
      {3.6189,   -5.8889,    4.1625}, % H5''
      {5.6255,   -5.9175,    3.5998}, % C4' 
      {5.8732,   -5.1228,    4.3034}, % H4' 
      {6.7337,   -6.8605,    3.5222}, % O4' 
      {7.5932,   -6.4923,    2.4548}, % C1' 
      {8.5661,   -6.2983,    2.9064}, % H1' 
      {7.0527,   -5.2012,    1.8322}, % C2' 
      {7.1627,   -5.2525,    0.7490}, % H2''
      {7.6666,   -4.1249,    2.4880}, % O2' 
      {8.5944,   -4.2543,    2.6981}, % H2' 
      {5.5661,   -5.3029,    2.2009}, % C3' 
      {5.0841,   -6.0018,    1.5172}, % H3' 
      {4.9062,   -4.0452,    2.2042}, % O3' 
      {7.6298,   -7.6136,    1.4752}, % N1  
      {8.5977,   -9.5977,    0.7329}, % N3  
      {8.5951,   -8.5745,    1.6594}, % C2  
      {7.7372,   -9.7371,   -0.3364}, % C4  
      {6.7596,   -8.6801,   -0.4476}, % C5  
      {6.7338,   -7.6721,    0.4408}, % C6
      c, {
      {7.8849,  -10.7881,   -1.1289}, % N4  
      {9.3993,   -8.5377,    2.5743}, % O2  
      {7.2499,  -10.8809,   -1.9088}, % H41 
      {8.6122,  -11.4649,   -0.9468}, % H42 
      {6.0317,   -8.6941,   -1.2588}, % H5  
      {5.9901,   -6.8809,    0.3459}} % H6  
    }.

rC06() ->
     {
      {-0.9837,    0.0476,   -0.1733,  % dgf_base_tfo
       -0.1792,   -0.3353,    0.9249,
       -0.0141,    0.9409,    0.3384,
        5.7793,   -5.2303,    4.5997},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {3.9938,   -6.7042,    1.9023}, % C5' 
      {3.2332,   -5.9343,    2.0319}, % H5' 
      {3.9666,   -7.2863,    0.9812}, % H5''
      {5.3098,   -5.9546,    1.8564}, % C4' 
      {5.3863,   -5.3702,    0.9395}, % H4' 
      {5.3851,   -5.0642,    3.0076}, % O4' 
      {6.7315,   -4.9724,    3.4462}, % C1' 
      {7.0033,   -3.9202,    3.3619}, % H1' 
      {7.5997,   -5.8018,    2.4948}, % C2' 
      {8.3627,   -6.3254,    3.0707}, % H2''
      {8.0410,   -4.9501,    1.4724}, % O2' 
      {8.2781,   -4.0644,    1.7570}, % H2' 
      {6.5701,   -6.8129,    1.9714}, % C3' 
      {6.4186,   -7.5809,    2.7299}, % H3' 
      {6.9357,   -7.3841,    0.7235}, % O3' 
      {6.8024,   -5.4718,    4.8475}, % N1  
      {6.6920,   -5.0495,    7.1354}, % N3  
      {6.6201,   -4.5500,    5.8506}, % C2  
      {6.9254,   -6.3614,    7.4926}, % C4  
      {7.1046,   -7.2543,    6.3718}, % C5  
      {7.0391,   -6.7951,    5.1106}, % C6
      c, {
      {6.9614,   -6.6648,    8.7815}, % N4  
      {6.4083,   -3.3696,    5.6340}, % O2  
      {7.1329,   -7.6280,    9.0324}, % H41 
      {6.8204,   -5.9469,    9.4777}, % H42 
      {7.2954,   -8.3135,    6.5440}, % H5  
      {7.1753,   -7.4798,    4.2735}} % H6  
    }.

rC07() ->
     {
      {0.0033,    0.2720,   -0.9623,  % dgf_base_tfo
        0.3013,   -0.9179,   -0.2584,
       -0.9535,   -0.2891,   -0.0850,
       43.0403,   13.7233,   34.5710},
      {0.9187,    0.2887,    0.2694,  % p_o3'_275_tfo
        0.0302,   -0.7316,    0.6811,
        0.3938,   -0.6176,   -0.6808,
       -48.4330,   26.3254,   13.6383},
      {-0.1504,    0.7744,   -0.6145,  % p_o3'_180_tfo
        0.7581,    0.4893,    0.4311,
        0.6345,   -0.4010,   -0.6607,
       -31.9784,  -13.4285,   44.9650},
      {-0.6236,   -0.7810,   -0.0337,  % p_o3'_60_tfo
       -0.6890,    0.5694,   -0.4484,
        0.3694,   -0.2564,   -0.8932,
       12.1105,   30.8774,   46.0946},
      {33.3400,   11.0980,   46.1750}, % P   
      {34.5130,   10.2320,   46.4660}, % O1P 
      {33.4130,   12.3960,   46.9340}, % O2P 
      {31.9810,   10.3390,   46.4820}, % O5' 
      {30.8152,   11.1619,   46.2003}, % C5' 
      {30.4519,   10.9454,   45.1957}, % H5' 
      {31.0379,   12.2016,   46.4400}, % H5''
      {29.7081,   10.7448,   47.1428}, % C4' 
      {28.8710,   11.4416,   47.0982}, % H4' 
      {29.2550,    9.4394,   46.8162}, % O4' 
      {29.3907,    8.5625,   47.9460}, % C1' 
      {28.4416,    8.5669,   48.4819}, % H1' 
      {30.4468,    9.2031,   48.7952}, % C2' 
      {31.4222,    8.9651,   48.3709}, % H2''
      {30.3701,    8.9157,   50.1624}, % O2' 
      {30.0652,    8.0304,   50.3740}, % H2' 
      {30.1622,   10.6879,   48.6120}, % C3' 
      {31.0952,   11.2399,   48.7254}, % H3' 
      {29.1076,   11.1535,   49.4702}, % O3' 
      {29.7883,    7.2209,   47.5235}, % N1  
      {29.1825,    5.0438,   46.8275}, % N3  
      {28.8008,    6.2912,   47.2263}, % C2  
      {30.4888,    4.6890,   46.7186}, % C4  
      {31.5034,    5.6405,   47.0249}, % C5  
      {31.1091,    6.8691,   47.4156}, % C6
      c, {
      {30.8109,    3.4584,   46.3336}, % N4  
      {27.6171,    6.5989,   47.3189}, % O2  
      {31.7923,    3.2301,   46.2638}, % H41 
      {30.0880,    2.7857,   46.1215}, % H42 
      {32.5542,    5.3634,   46.9395}, % H5  
      {31.8523,    7.6279,   47.6603}} % H6  
    }.

rC08() ->
     {
      {0.0797,   -0.6026,   -0.7941,  % dgf_base_tfo
        0.7939,    0.5201,   -0.3150,
        0.6028,   -0.6054,    0.5198,
       -36.8341,   41.5293,    1.6628},
      {0.9187,    0.2887,    0.2694,  % p_o3'_275_tfo
        0.0302,   -0.7316,    0.6811,
        0.3938,   -0.6176,   -0.6808,
       -48.4330,   26.3254,   13.6383},
      {-0.1504,    0.7744,   -0.6145,  % p_o3'_180_tfo
        0.7581,    0.4893,    0.4311,
        0.6345,   -0.4010,   -0.6607,
       -31.9784,  -13.4285,   44.9650},
      {-0.6236,   -0.7810,   -0.0337,  % p_o3'_60_tfo
       -0.6890,    0.5694,   -0.4484,
        0.3694,   -0.2564,   -0.8932,
       12.1105,   30.8774,   46.0946},
      {33.3400,   11.0980,   46.1750}, % P   
      {34.5130,   10.2320,   46.4660}, % O1P 
      {33.4130,   12.3960,   46.9340}, % O2P 
      {31.9810,   10.3390,   46.4820}, % O5' 
      {31.8779,    9.9369,   47.8760}, % C5' 
      {31.3239,   10.6931,   48.4322}, % H5' 
      {32.8647,    9.6624,   48.2489}, % H5''
      {31.0429,    8.6773,   47.9401}, % C4' 
      {31.0779,    8.2331,   48.9349}, % H4' 
      {29.6956,    8.9669,   47.5983}, % O4' 
      {29.2784,    8.1700,   46.4782}, % C1' 
      {28.8006,    7.2731,   46.8722}, % H1' 
      {30.5544,    7.7940,   45.7875}, % C2' 
      {30.8837,    8.6410,   45.1856}, % H2''
      {30.5100,    6.6007,   45.0582}, % O2' 
      {29.6694,    6.4168,   44.6326}, % H2' 
      {31.5146,    7.5954,   46.9527}, % C3' 
      {32.5255,    7.8261,   46.6166}, % H3' 
      {31.3876,    6.2951,   47.5516}, % O3' 
      {28.3976,    8.9302,   45.5933}, % N1  
      {26.2155,    9.6135,   44.9910}, % N3  
      {27.0281,    8.8961,   45.8192}, % C2  
      {26.7044,   10.3489,   43.9595}, % C4  
      {28.1088,   10.3837,   43.7247}, % C5  
      {28.8978,    9.6708,   44.5535}, % C6
      c, {
      {25.8715,   11.0249,   43.1749}, % N4  
      {26.5733,    8.2371,   46.7484}, % O2  
      {26.2707,   11.5609,   42.4177}, % H41 
      {24.8760,   10.9939,   43.3427}, % H42 
      {28.5089,   10.9722,   42.8990}, % H5  
      {29.9782,    9.6687,   44.4097}} % H6  
    }.

rC09() ->
     {
      {0.8727,    0.4760,   -0.1091,  % dgf_base_tfo
       -0.4188,    0.6148,   -0.6682,
       -0.2510,    0.6289,    0.7359,
       -8.1687,  -52.0761,  -25.0726},
      {0.9187,    0.2887,    0.2694,  % p_o3'_275_tfo
        0.0302,   -0.7316,    0.6811,
        0.3938,   -0.6176,   -0.6808,
       -48.4330,   26.3254,   13.6383},
      {-0.1504,    0.7744,   -0.6145,  % p_o3'_180_tfo
        0.7581,    0.4893,    0.4311,
        0.6345,   -0.4010,   -0.6607,
       -31.9784,  -13.4285,   44.9650},
      {-0.6236,   -0.7810,   -0.0337,  % p_o3'_60_tfo
       -0.6890,    0.5694,   -0.4484,
        0.3694,   -0.2564,   -0.8932,
       12.1105,   30.8774,   46.0946},
      {33.3400,   11.0980,   46.1750}, % P   
      {34.5130,   10.2320,   46.4660}, % O1P 
      {33.4130,   12.3960,   46.9340}, % O2P 
      {31.9810,   10.3390,   46.4820}, % O5' 
      {30.8152,   11.1619,   46.2003}, % C5' 
      {30.4519,   10.9454,   45.1957}, % H5' 
      {31.0379,   12.2016,   46.4400}, % H5''
      {29.7081,   10.7448,   47.1428}, % C4' 
      {29.4506,    9.6945,   47.0059}, % H4' 
      {30.1045,   10.9634,   48.4885}, % O4' 
      {29.1794,   11.8418,   49.1490}, % C1' 
      {28.4388,   11.2210,   49.6533}, % H1' 
      {28.5211,   12.6008,   48.0367}, % C2' 
      {29.1947,   13.3949,   47.7147}, % H2''
      {27.2316,   13.0683,   48.3134}, % O2' 
      {27.0851,   13.3391,   49.2227}, % H2' 
      {28.4131,   11.5507,   46.9391}, % C3' 
      {28.4451,   12.0512,   45.9713}, % H3' 
      {27.2707,   10.6955,   47.1097}, % O3' 
      {29.8751,   12.7405,   50.0682}, % N1  
      {30.7172,   13.1841,   52.2328}, % N3  
      {30.0617,   12.3404,   51.3847}, % C2  
      {31.1834,   14.3941,   51.8297}, % C4  
      {30.9913,   14.8074,   50.4803}, % C5  
      {30.3434,   13.9610,   49.6548}, % C6
      c, {
      {31.8090,   15.1847,   52.6957}, % N4  
      {29.6470,   11.2494,   51.7616}, % O2  
      {32.1422,   16.0774,   52.3606}, % H41 
      {31.9392,   14.8893,   53.6527}, % H42 
      {31.3632,   15.7771,   50.1491}, % H5  
      {30.1742,   14.2374,   48.6141}} % H6  
    }.

rC10() ->
     {
      {0.1549,    0.8710,   -0.4663,  % dgf_base_tfo
        0.6768,   -0.4374,   -0.5921,
       -0.7197,   -0.2239,   -0.6572,
       25.2447,  -14.1920,   50.3201},
      {0.9187,    0.2887,    0.2694,  % p_o3'_275_tfo
        0.0302,   -0.7316,    0.6811,
        0.3938,   -0.6176,   -0.6808,
       -48.4330,   26.3254,   13.6383},
      {-0.1504,    0.7744,   -0.6145,  % p_o3'_180_tfo
        0.7581,    0.4893,    0.4311,
        0.6345,   -0.4010,   -0.6607,
       -31.9784,  -13.4285,   44.9650},
      {-0.6236,   -0.7810,   -0.0337,  % p_o3'_60_tfo
       -0.6890,    0.5694,   -0.4484,
        0.3694,   -0.2564,   -0.8932,
       12.1105,   30.8774,   46.0946},
      {33.3400,   11.0980,   46.1750}, % P   
      {34.5130,   10.2320,   46.4660}, % O1P 
      {33.4130,   12.3960,   46.9340}, % O2P 
      {31.9810,   10.3390,   46.4820}, % O5' 
      {31.8779,    9.9369,   47.8760}, % C5' 
      {31.3239,   10.6931,   48.4322}, % H5' 
      {32.8647,    9.6624,   48.2489}, % H5''
      {31.0429,    8.6773,   47.9401}, % C4' 
      {30.0440,    8.8473,   47.5383}, % H4' 
      {31.6749,    7.6351,   47.2119}, % O4' 
      {31.9159,    6.5022,   48.0616}, % C1' 
      {31.0691,    5.8243,   47.9544}, % H1' 
      {31.9300,    7.0685,   49.4493}, % C2' 
      {32.9024,    7.5288,   49.6245}, % H2''
      {31.5672,    6.1750,   50.4632}, % O2' 
      {31.8416,    5.2663,   50.3200}, % H2' 
      {30.8618,    8.1514,   49.3749}, % C3' 
      {31.1122,    8.9396,   50.0850}, % H3' 
      {29.5351,    7.6245,   49.5409}, % O3' 
      {33.1890,    5.8629,   47.7343}, % N1  
      {34.4004,    4.2636,   46.4828}, % N3  
      {33.2062,    4.8497,   46.7851}, % C2  
      {35.5600,    4.6374,   47.0822}, % C4  
      {35.5444,    5.6751,   48.0577}, % C5  
      {34.3565,    6.2450,   48.3432}, % C6
      c, {
      {36.6977,    4.0305,   46.7598}, % N4  
      {32.1661,    4.5034,   46.2348}, % O2  
      {37.5405,    4.3347,   47.2259}, % H41 
      {36.7033,    3.2923,   46.0706}, % H42 
      {36.4713,    5.9811,   48.5428}, % H5  
      {34.2986,    7.0426,   49.0839}} % H6  
    }.

rCs() -> [rC01(),rC02(),rC03(),rC04(),rC05(),rC06(),rC07(),
          rC08(),rC09(),rC10()].

rG() ->
     {
      {-0.0018,   -0.8207,    0.5714,  % dgf_base_tfo
        0.2679,   -0.5509,   -0.7904,
        0.9634,    0.1517,    0.2209,
        0.0073,    8.4030,    0.6232},
      {-0.8143,   -0.5091,   -0.2788,  % p_o3'_275_tfo
       -0.0433,   -0.4257,    0.9038,
       -0.5788,    0.7480,    0.3246,
        1.5227,    6.9114,   -7.0765},
      {0.3822,   -0.7477,    0.5430,  % p_o3'_180_tfo
        0.4552,    0.6637,    0.5935,
       -0.8042,    0.0203,    0.5941,
       -6.9472,   -4.1186,   -5.9108},
      {0.5640,    0.8007,   -0.2022,  % p_o3'_60_tfo
       -0.8247,    0.5587,   -0.0878,
        0.0426,    0.2162,    0.9754,
        6.2694,   -7.0540,    3.3316},
      {2.8930,    8.5380,   -3.3280}, % P   
      {1.6980,    7.6960,   -3.5570}, % O1P 
      {3.2260,    9.5010,   -4.4020}, % O2P 
      {4.1590,    7.6040,   -3.0340}, % O5' 
      {5.4550,    8.2120,   -2.8810}, % C5' 
      {5.4546,    8.8508,   -1.9978}, % H5' 
      {5.7588,    8.6625,   -3.8259}, % H5''
      {6.4970,    7.1480,   -2.5980}, % C4' 
      {7.4896,    7.5919,   -2.5214}, % H4' 
      {6.1630,    6.4860,   -1.3440}, % O4' 
      {6.5400,    5.1200,   -1.4190}, % C1' 
      {7.2763,    4.9681,   -0.6297}, % H1' 
      {7.1940,    4.8830,   -2.7770}, % C2' 
      {6.8667,    3.9183,   -3.1647}, % H2''
      {8.5860,    5.0910,   -2.6140}, % O2' 
      {8.9510,    4.7626,   -1.7890}, % H2' 
      {6.5720,    6.0040,   -3.6090}, % C3' 
      {5.5636,    5.7066,   -3.8966}, % H3' 
      {7.3801,    6.3562,   -4.7350}, % O3' 
      {4.7150,    0.4910,   -0.1360}, % N1  
      {6.3490,    2.1730,   -0.6020}, % N3  
      {5.9530,    0.9650,   -0.2670}, % C2  
      {5.2900,    2.9790,   -0.8260}, % C4  
      {3.9720,    2.6390,   -0.7330}, % C5  
      {3.6770,    1.3160,   -0.3660}, % C6
      g, {
      {6.8426,    0.0056,   -0.0019}, % N2  
      {3.1660,    3.7290,   -1.0360}, % N7  
      {5.3170,    4.2990,   -1.1930}, % N9  
      {4.0100,    4.6780,   -1.2990}, % C8  
      {2.4280,    0.8450,   -0.2360}, % O6  
      {4.6151,   -0.4677,    0.1305}, % H1  
      {6.6463,   -0.9463,    0.2729}, % H21 
      {7.8170,    0.2642,   -0.0640}, % H22 
      {3.4421,    5.5744,   -1.5482}} % H8  
    }.

rU() ->
     {
      {-0.0359,   -0.8071,    0.5894,  % dgf_base_tfo
       -0.2669,    0.5761,    0.7726,
       -0.9631,   -0.1296,   -0.2361,
        0.1584,    8.3434,    0.5434},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {5.2430,   -8.2420,    2.8260}, % C5' 
      {5.1974,   -8.8497,    1.9223}, % H5' 
      {5.5548,   -8.7348,    3.7469}, % H5''
      {6.3140,   -7.2060,    2.5510}, % C4' 
      {7.2954,   -7.6762,    2.4898}, % H4' 
      {6.0140,   -6.5420,    1.2890}, % O4' 
      {6.4190,   -5.1840,    1.3620}, % C1' 
      {7.1608,   -5.0495,    0.5747}, % H1' 
      {7.0760,   -4.9560,    2.7270}, % C2' 
      {6.7770,   -3.9803,    3.1099}, % H2''
      {8.4500,   -5.1930,    2.5810}, % O2' 
      {8.8309,   -4.8755,    1.7590}, % H2' 
      {6.4060,   -6.0590,    3.5580}, % C3' 
      {5.4021,   -5.7313,    3.8281}, % H3' 
      {7.1570,   -6.4240,    4.7070}, % O3' 
      {5.2170,   -4.3260,    1.1690}, % N1  
      {4.2960,   -2.2560,    0.6290}, % N3  
      {5.4330,   -3.0200,    0.7990}, % C2  
      {2.9930,   -2.6780,    0.7940}, % C4  
      {2.8670,   -4.0630,    1.1830}, % C5  
      {3.9570,   -4.8300,    1.3550}, % C6
      u, {
      {6.5470,   -2.5560,    0.6290}, % O2  
      {2.0540,   -1.9000,    0.6130}, % O4  
      {4.4300,   -1.3020,    0.3600}, % H3  
      {1.9590,   -4.4570,    1.3250}, % H5  
      {3.8460,   -5.7860,    1.6240}} % H6  
    }.

rU01() ->
     {
      {-0.0137,   -0.8012,    0.5983,  % dgf_base_tfo
       -0.2523,    0.5817,    0.7733,
       -0.9675,   -0.1404,   -0.2101,
        0.2031,    8.3874,    0.4228},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {5.2416,   -8.2422,    2.8181}, % C5' 
      {5.2050,   -8.8128,    1.8901}, % H5' 
      {5.5368,   -8.7738,    3.7227}, % H5''
      {6.3232,   -7.2037,    2.6002}, % C4' 
      {7.3048,   -7.6757,    2.5577}, % H4' 
      {6.0635,   -6.5092,    1.3456}, % O4' 
      {6.4697,   -5.1547,    1.4629}, % C1' 
      {7.2354,   -5.0043,    0.7018}, % H1' 
      {7.0856,   -4.9610,    2.8521}, % C2' 
      {6.7777,   -3.9935,    3.2487}, % H2''
      {8.4627,   -5.1992,    2.7423}, % O2' 
      {8.8693,   -4.8638,    1.9399}, % H2' 
      {6.3877,   -6.0809,    3.6362}, % C3' 
      {5.3770,   -5.7562,    3.8834}, % H3' 
      {7.1024,   -6.4754,    4.7985}, % O3' 
      {5.2764,   -4.2883,    1.2538}, % N1  
      {4.3777,   -2.2062,    0.7229}, % N3  
      {5.5069,   -2.9779,    0.9088}, % C2  
      {3.0693,   -2.6246,    0.8500}, % C4  
      {2.9279,   -4.0146,    1.2149}, % C5  
      {4.0101,   -4.7892,    1.4017}, % C6
      u, {
      {6.6267,   -2.5166,    0.7728}, % O2  
      {2.1383,   -1.8396,    0.6581}, % O4  
      {4.5223,   -1.2489,    0.4716}, % H3  
      {2.0151,   -4.4065,    1.3290}, % H5  
      {3.8886,   -5.7486,    1.6535}} % H6  
    }.

rU02() ->
     {
      {0.5141,    0.0246,    0.8574,  % dgf_base_tfo
       -0.5547,   -0.7529,    0.3542,
        0.6542,   -0.6577,   -0.3734,
       -9.1111,   -3.4598,   -3.2939},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {4.3825,   -6.6585,    4.0489}, % C5' 
      {4.6841,   -7.2019,    4.9443}, % H5' 
      {3.6189,   -5.8889,    4.1625}, % H5''
      {5.6255,   -5.9175,    3.5998}, % C4' 
      {5.8732,   -5.1228,    4.3034}, % H4' 
      {6.7337,   -6.8605,    3.5222}, % O4' 
      {7.5932,   -6.4923,    2.4548}, % C1' 
      {8.5661,   -6.2983,    2.9064}, % H1' 
      {7.0527,   -5.2012,    1.8322}, % C2' 
      {7.1627,   -5.2525,    0.7490}, % H2''
      {7.6666,   -4.1249,    2.4880}, % O2' 
      {8.5944,   -4.2543,    2.6981}, % H2' 
      {5.5661,   -5.3029,    2.2009}, % C3' 
      {5.0841,   -6.0018,    1.5172}, % H3' 
      {4.9062,   -4.0452,    2.2042}, % O3' 
      {7.6298,   -7.6136,    1.4752}, % N1  
      {8.6945,   -8.7046,   -0.2857}, % N3  
      {8.6943,   -7.6514,    0.6066}, % C2  
      {7.7426,   -9.6987,   -0.3801}, % C4  
      {6.6642,   -9.5742,    0.5722}, % C5  
      {6.6391,   -8.5592,    1.4526}, % C6
      u, {
      {9.5840,   -6.8186,    0.6136}, % O2  
      {7.8505,  -10.5925,   -1.2223}, % O4  
      {9.4601,   -8.7514,   -0.9277}, % H3  
      {5.9281,  -10.2509,    0.5782}, % H5  
      {5.8831,   -8.4931,    2.1028}} % H6  
    }.

rU03() ->
     {
      {-0.4993,    0.0476,    0.8651,  % dgf_base_tfo
        0.8078,   -0.3353,    0.4847,
        0.3132,    0.9409,    0.1290,
        6.2989,   -5.2303,   -3.8577},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {3.9938,   -6.7042,    1.9023}, % C5' 
      {3.2332,   -5.9343,    2.0319}, % H5' 
      {3.9666,   -7.2863,    0.9812}, % H5''
      {5.3098,   -5.9546,    1.8564}, % C4' 
      {5.3863,   -5.3702,    0.9395}, % H4' 
      {5.3851,   -5.0642,    3.0076}, % O4' 
      {6.7315,   -4.9724,    3.4462}, % C1' 
      {7.0033,   -3.9202,    3.3619}, % H1' 
      {7.5997,   -5.8018,    2.4948}, % C2' 
      {8.3627,   -6.3254,    3.0707}, % H2''
      {8.0410,   -4.9501,    1.4724}, % O2' 
      {8.2781,   -4.0644,    1.7570}, % H2' 
      {6.5701,   -6.8129,    1.9714}, % C3' 
      {6.4186,   -7.5809,    2.7299}, % H3' 
      {6.9357,   -7.3841,    0.7235}, % O3' 
      {6.8024,   -5.4718,    4.8475}, % N1  
      {7.9218,   -5.5700,    6.8877}, % N3  
      {7.8908,   -5.0886,    5.5944}, % C2  
      {6.9789,   -6.3827,    7.4823}, % C4  
      {5.8742,   -6.7319,    6.6202}, % C5  
      {5.8182,   -6.2769,    5.3570}, % C6
      u, {
      {8.7747,   -4.3728,    5.1568}, % O2  
      {7.1154,   -6.7509,    8.6509}, % O4  
      {8.7055,   -5.3037,    7.4491}, % H3  
      {5.1416,   -7.3178,    6.9665}, % H5  
      {5.0441,   -6.5310,    4.7784}} % H6  
    }.

rU04() ->
     {
      {-0.5669,   -0.8012,    0.1918,  % dgf_base_tfo
       -0.8129,    0.5817,    0.0273,
       -0.1334,   -0.1404,   -0.9811,
       -0.3279,    8.3874,    0.3355},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {5.2416,   -8.2422,    2.8181}, % C5' 
      {5.2050,   -8.8128,    1.8901}, % H5' 
      {5.5368,   -8.7738,    3.7227}, % H5''
      {6.3232,   -7.2037,    2.6002}, % C4' 
      {7.3048,   -7.6757,    2.5577}, % H4' 
      {6.0635,   -6.5092,    1.3456}, % O4' 
      {6.4697,   -5.1547,    1.4629}, % C1' 
      {7.2354,   -5.0043,    0.7018}, % H1' 
      {7.0856,   -4.9610,    2.8521}, % C2' 
      {6.7777,   -3.9935,    3.2487}, % H2''
      {8.4627,   -5.1992,    2.7423}, % O2' 
      {8.8693,   -4.8638,    1.9399}, % H2' 
      {6.3877,   -6.0809,    3.6362}, % C3' 
      {5.3770,   -5.7562,    3.8834}, % H3' 
      {7.1024,   -6.4754,    4.7985}, % O3' 
      {5.2764,   -4.2883,    1.2538}, % N1  
      {3.8961,   -3.0896,   -0.1893}, % N3  
      {5.0095,   -3.8907,   -0.0346}, % C2  
      {3.0480,   -2.6632,    0.8116}, % C4  
      {3.4093,   -3.1310,    2.1292}, % C5  
      {4.4878,   -3.9124,    2.3088}, % C6
      u, {
      {5.7005,   -4.2164,   -0.9842}, % O2  
      {2.0800,   -1.9458,    0.5503}, % O4  
      {3.6834,   -2.7882,   -1.1190}, % H3  
      {2.8508,   -2.8721,    2.9172}, % H5  
      {4.7188,   -4.2247,    3.2295}} % H6  
    }.

rU05() ->
     {
      {-0.6298,    0.0246,    0.7763,  % dgf_base_tfo
       -0.5226,   -0.7529,   -0.4001,
        0.5746,   -0.6577,    0.4870,
       -0.0208,   -3.4598,   -9.6882},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {4.3825,   -6.6585,    4.0489}, % C5' 
      {4.6841,   -7.2019,    4.9443}, % H5' 
      {3.6189,   -5.8889,    4.1625}, % H5''
      {5.6255,   -5.9175,    3.5998}, % C4' 
      {5.8732,   -5.1228,    4.3034}, % H4' 
      {6.7337,   -6.8605,    3.5222}, % O4' 
      {7.5932,   -6.4923,    2.4548}, % C1' 
      {8.5661,   -6.2983,    2.9064}, % H1' 
      {7.0527,   -5.2012,    1.8322}, % C2' 
      {7.1627,   -5.2525,    0.7490}, % H2''
      {7.6666,   -4.1249,    2.4880}, % O2' 
      {8.5944,   -4.2543,    2.6981}, % H2' 
      {5.5661,   -5.3029,    2.2009}, % C3' 
      {5.0841,   -6.0018,    1.5172}, % H3' 
      {4.9062,   -4.0452,    2.2042}, % O3' 
      {7.6298,   -7.6136,    1.4752}, % N1  
      {8.5977,   -9.5977,    0.7329}, % N3  
      {8.5951,   -8.5745,    1.6594}, % C2  
      {7.7372,   -9.7371,   -0.3364}, % C4  
      {6.7596,   -8.6801,   -0.4476}, % C5  
      {6.7338,   -7.6721,    0.4408}, % C6
      u, {
      {9.3993,   -8.5377,    2.5743}, % O2  
      {7.8374,  -10.6990,   -1.1008}, % O4  
      {9.2924,  -10.3081,    0.8477}, % H3  
      {6.0932,   -8.6982,   -1.1929}, % H5  
      {6.0481,   -6.9515,    0.3446}} % H6  
    }.

rU06() ->
     {
      {-0.9837,    0.0476,   -0.1733,  % dgf_base_tfo
       -0.1792,   -0.3353,    0.9249,
       -0.0141,    0.9409,    0.3384,
        5.7793,   -5.2303,    4.5997},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {3.9938,   -6.7042,    1.9023}, % C5' 
      {3.2332,   -5.9343,    2.0319}, % H5' 
      {3.9666,   -7.2863,    0.9812}, % H5''
      {5.3098,   -5.9546,    1.8564}, % C4' 
      {5.3863,   -5.3702,    0.9395}, % H4' 
      {5.3851,   -5.0642,    3.0076}, % O4' 
      {6.7315,   -4.9724,    3.4462}, % C1' 
      {7.0033,   -3.9202,    3.3619}, % H1' 
      {7.5997,   -5.8018,    2.4948}, % C2' 
      {8.3627,   -6.3254,    3.0707}, % H2''
      {8.0410,   -4.9501,    1.4724}, % O2' 
      {8.2781,   -4.0644,    1.7570}, % H2' 
      {6.5701,   -6.8129,    1.9714}, % C3' 
      {6.4186,   -7.5809,    2.7299}, % H3' 
      {6.9357,   -7.3841,    0.7235}, % O3' 
      {6.8024,   -5.4718,    4.8475}, % N1  
      {6.6920,   -5.0495,    7.1354}, % N3  
      {6.6201,   -4.5500,    5.8506}, % C2  
      {6.9254,   -6.3614,    7.4926}, % C4  
      {7.1046,   -7.2543,    6.3718}, % C5  
      {7.0391,   -6.7951,    5.1106}, % C6
      u, {
      {6.4083,   -3.3696,    5.6340}, % O2  
      {6.9679,   -6.6901,    8.6800}, % O4  
      {6.5626,   -4.3957,    7.8812}, % H3  
      {7.2781,   -8.2254,    6.5350}, % H5  
      {7.1657,   -7.4312,    4.3503}} % H6  
    }.

rU07() ->
     {
      {-0.9434,    0.3172,    0.0971,  % dgf_base_tfo
        0.2294,    0.4125,    0.8816,
        0.2396,    0.8539,   -0.4619,
        8.3625,  -52.7147,    1.3745},
      {0.2765,   -0.1121,   -0.9545,  % p_o3'_275_tfo
       -0.8297,    0.4733,   -0.2959,
        0.4850,    0.8737,    0.0379,
       -14.7774,  -45.2464,   21.9088},
      {0.1063,   -0.6334,   -0.7665,  % p_o3'_180_tfo
       -0.5932,   -0.6591,    0.4624,
       -0.7980,    0.4055,   -0.4458,
       43.7634,    4.3296,   28.4890},
      {0.7136,   -0.5032,   -0.4873,  % p_o3'_60_tfo
        0.6803,    0.3317,    0.6536,
       -0.1673,   -0.7979,    0.5791,
       -17.1858,   41.4390,  -27.0751},
      {21.3880,   15.0780,   45.5770}, % P   
      {21.9980,   14.5500,   46.8210}, % O1P 
      {21.1450,   14.0270,   44.5420}, % O2P 
      {22.1250,   16.3600,   44.9460}, % O5' 
      {21.5037,   16.8594,   43.7323}, % C5' 
      {20.8147,   17.6663,   43.9823}, % H5' 
      {21.1086,   16.0230,   43.1557}, % H5''
      {22.5654,   17.4874,   42.8616}, % C4' 
      {22.1584,   17.7243,   41.8785}, % H4' 
      {23.0557,   18.6826,   43.4751}, % O4' 
      {24.4788,   18.6151,   43.6455}, % C1' 
      {24.9355,   19.0840,   42.7739}, % H1' 
      {24.7958,   17.1427,   43.6474}, % C2' 
      {24.5652,   16.7400,   44.6336}, % H2''
      {26.1041,   16.8773,   43.2455}, % O2' 
      {26.7516,   17.5328,   43.5149}, % H2' 
      {23.8109,   16.5979,   42.6377}, % C3' 
      {23.5756,   15.5686,   42.9084}, % H3' 
      {24.2890,   16.7447,   41.2729}, % O3' 
      {24.9420,   19.2174,   44.8923}, % N1  
      {25.2655,   20.5636,   44.8883}, % N3  
      {25.1663,   21.2219,   43.8561}, % C2  
      {25.6911,   21.1219,   46.0494}, % C4  
      {25.8051,   20.4068,   47.2048}, % C5  
      {26.2093,   20.9962,   48.2534}, % C6
      u, {
      {25.4692,   19.0221,   47.2053}, % O2  
      {25.0502,   18.4827,   46.0370}, % O4  
      {25.9599,   22.1772,   46.0966}, % H3  
      {25.5545,   18.4409,   48.1234}, % H5  
      {24.7854,   17.4265,   45.9883}} % H6  
    }.

rU08() ->
     {
      {-0.0080,   -0.7928,    0.6094,  % dgf_base_tfo
       -0.7512,    0.4071,    0.5197,
       -0.6601,   -0.4536,   -0.5988,
       44.1482,   30.7036,    2.1088},
      {0.2765,   -0.1121,   -0.9545,  % p_o3'_275_tfo
       -0.8297,    0.4733,   -0.2959,
        0.4850,    0.8737,    0.0379,
       -14.7774,  -45.2464,   21.9088},
      {0.1063,   -0.6334,   -0.7665,  % p_o3'_180_tfo
       -0.5932,   -0.6591,    0.4624,
       -0.7980,    0.4055,   -0.4458,
       43.7634,    4.3296,   28.4890},
      {0.7136,   -0.5032,   -0.4873,  % p_o3'_60_tfo
        0.6803,    0.3317,    0.6536,
       -0.1673,   -0.7979,    0.5791,
       -17.1858,   41.4390,  -27.0751},
      {21.3880,   15.0780,   45.5770}, % P   
      {21.9980,   14.5500,   46.8210}, % O1P 
      {21.1450,   14.0270,   44.5420}, % O2P 
      {22.1250,   16.3600,   44.9460}, % O5' 
      {23.5096,   16.1227,   44.5783}, % C5' 
      {23.5649,   15.8588,   43.5222}, % H5' 
      {23.9621,   15.4341,   45.2919}, % H5''
      {24.2805,   17.4138,   44.7151}, % C4' 
      {25.3492,   17.2309,   44.6030}, % H4' 
      {23.8497,   18.3471,   43.7208}, % O4' 
      {23.4090,   19.5681,   44.3321}, % C1' 
      {24.2595,   20.2496,   44.3524}, % H1' 
      {23.0418,   19.1813,   45.7407}, % C2' 
      {22.0532,   18.7224,   45.7273}, % H2''
      {23.1307,   20.2521,   46.6291}, % O2' 
      {22.8888,   21.1051,   46.2611}, % H2' 
      {24.0799,   18.1326,   46.0700}, % C3' 
      {23.6490,   17.4370,   46.7900}, % H3' 
      {25.3329,   18.7227,   46.5109}, % O3' 
      {22.2515,   20.1624,   43.6698}, % N1  
      {22.4760,   21.0609,   42.6406}, % N3  
      {23.6229,   21.3462,   42.3061}, % C2  
      {21.3986,   21.6081,   42.0236}, % C4  
      {20.1189,   21.3012,   42.3804}, % C5  
      {19.1599,   21.8516,   41.7578}, % C6
      u, {
      {19.8919,   20.3745,   43.4387}, % O2  
      {20.9790,   19.8423,   44.0440}, % O4  
      {21.5235,   22.3222,   41.2097}, % H3  
      {18.8732,   20.1200,   43.7312}, % H5  
      {20.8545,   19.1313,   44.8608}} % H6  
    }.

rU09() ->
     {
      {-0.0317,    0.1374,    0.9900,  % dgf_base_tfo
       -0.3422,   -0.9321,    0.1184,
        0.9391,   -0.3351,    0.0765,
       -32.1929,   25.8198,  -28.5088},
      {0.2765,   -0.1121,   -0.9545,  % p_o3'_275_tfo
       -0.8297,    0.4733,   -0.2959,
        0.4850,    0.8737,    0.0379,
       -14.7774,  -45.2464,   21.9088},
      {0.1063,   -0.6334,   -0.7665,  % p_o3'_180_tfo
       -0.5932,   -0.6591,    0.4624,
       -0.7980,    0.4055,   -0.4458,
       43.7634,    4.3296,   28.4890},
      {0.7136,   -0.5032,   -0.4873,  % p_o3'_60_tfo
        0.6803,    0.3317,    0.6536,
       -0.1673,   -0.7979,    0.5791,
       -17.1858,   41.4390,  -27.0751},
      {21.3880,   15.0780,   45.5770}, % P   
      {21.9980,   14.5500,   46.8210}, % O1P 
      {21.1450,   14.0270,   44.5420}, % O2P 
      {22.1250,   16.3600,   44.9460}, % O5' 
      {21.5037,   16.8594,   43.7323}, % C5' 
      {20.8147,   17.6663,   43.9823}, % H5' 
      {21.1086,   16.0230,   43.1557}, % H5''
      {22.5654,   17.4874,   42.8616}, % C4' 
      {23.0565,   18.3036,   43.3915}, % H4' 
      {23.5375,   16.5054,   42.4925}, % O4' 
      {23.6574,   16.4257,   41.0649}, % C1' 
      {24.4701,   17.0882,   40.7671}, % H1' 
      {22.3525,   16.9643,   40.5396}, % C2' 
      {21.5993,   16.1799,   40.6133}, % H2''
      {22.4693,   17.4849,   39.2515}, % O2' 
      {23.0899,   17.0235,   38.6827}, % H2' 
      {22.0341,   18.0633,   41.5279}, % C3' 
      {20.9509,   18.1709,   41.5846}, % H3' 
      {22.7249,   19.3020,   41.2100}, % O3' 
      {23.8580,   15.0648,   40.5757}, % N1  
      {25.1556,   14.5982,   40.4523}, % N3  
      {26.1047,   15.3210,   40.7448}, % C2  
      {25.3391,   13.3315,   40.0020}, % C4  
      {24.2974,   12.5148,   39.6749}, % C5  
      {24.5450,   11.3410,   39.2610}, % C6
      u, {
      {22.9633,   12.9979,   39.8053}, % O2  
      {22.8009,   14.2648,   40.2524}, % O4  
      {26.3414,   12.9194,   39.8855}, % H3  
      {22.1227,   12.3533,   39.5486}, % H5  
      {21.7989,   14.6788,   40.3650}} % H6  
    }.

rU10() ->
     {
      {-0.9674,    0.1021,   -0.2318,  % dgf_base_tfo
       -0.2514,   -0.2766,    0.9275,
        0.0306,    0.9555,    0.2933,
       27.8571,  -42.1305,  -24.4563},
      {0.2765,   -0.1121,   -0.9545,  % p_o3'_275_tfo
       -0.8297,    0.4733,   -0.2959,
        0.4850,    0.8737,    0.0379,
       -14.7774,  -45.2464,   21.9088},
      {0.1063,   -0.6334,   -0.7665,  % p_o3'_180_tfo
       -0.5932,   -0.6591,    0.4624,
       -0.7980,    0.4055,   -0.4458,
       43.7634,    4.3296,   28.4890},
      {0.7136,   -0.5032,   -0.4873,  % p_o3'_60_tfo
        0.6803,    0.3317,    0.6536,
       -0.1673,   -0.7979,    0.5791,
       -17.1858,   41.4390,  -27.0751},
      {21.3880,   15.0780,   45.5770}, % P   
      {21.9980,   14.5500,   46.8210}, % O1P 
      {21.1450,   14.0270,   44.5420}, % O2P 
      {22.1250,   16.3600,   44.9460}, % O5' 
      {23.5096,   16.1227,   44.5783}, % C5' 
      {23.5649,   15.8588,   43.5222}, % H5' 
      {23.9621,   15.4341,   45.2919}, % H5''
      {24.2805,   17.4138,   44.7151}, % C4' 
      {23.8509,   18.1819,   44.0720}, % H4' 
      {24.2506,   17.8583,   46.0741}, % O4' 
      {25.5830,   18.0320,   46.5775}, % C1' 
      {25.8569,   19.0761,   46.4256}, % H1' 
      {26.4410,   17.1555,   45.7033}, % C2' 
      {26.3459,   16.1253,   46.0462}, % H2''
      {27.7649,   17.5888,   45.6478}, % O2' 
      {28.1004,   17.9719,   46.4616}, % H2' 
      {25.7796,   17.2997,   44.3513}, % C3' 
      {25.9478,   16.3824,   43.7871}, % H3' 
      {26.2154,   18.4984,   43.6541}, % O3' 
      {25.7321,   17.6281,   47.9726}, % N1  
      {25.5136,   18.5779,   48.9560}, % N3  
      {25.2079,   19.7276,   48.6503}, % C2  
      {25.6482,   18.1987,   50.2518}, % C4  
      {25.9847,   16.9266,   50.6092}, % C5  
      {26.0918,   16.6439,   51.8416}, % C6
      u, {
      {26.2067,   15.9515,   49.5943}, % O2  
      {26.0713,   16.3497,   48.3080}, % O4  
      {25.4890,   18.9105,   51.0618}, % H3  
      {26.4742,   14.9310,   49.8682}, % H5  
      {26.2346,   15.6394,   47.4975}} % H6  
    }.

rUs() -> [rU01(),rU02(),rU03(),rU04(),rU05(),rU06(),rU07(),
          rU08(),rU09(),rU10()].

rG_() ->
     {
      {-0.2067,   -0.0264,    0.9780,  % dgf_base_tfo
        0.9770,   -0.0586,    0.2049,
        0.0519,    0.9979,    0.0379,
        1.0331,  -46.8078,  -36.4742},
      {-0.8644,   -0.4956,   -0.0851,  % p_o3'_275_tfo
       -0.0427,    0.2409,   -0.9696,
        0.5010,   -0.8345,   -0.2294,
        4.0167,   54.5377,   12.4779},
      {0.3706,   -0.6167,    0.6945,  % p_o3'_180_tfo
       -0.2867,   -0.7872,   -0.5460,
        0.8834,    0.0032,   -0.4686,
       -52.9020,   18.6313,   -0.6709},
      {0.4155,    0.9025,   -0.1137,  % p_o3'_60_tfo
        0.9040,   -0.4236,   -0.0582,
       -0.1007,   -0.0786,   -0.9918,
       -7.6624,  -25.2080,   49.5181},
      {31.3810,    0.1400,   47.5810}, % P   
      {29.9860,    0.6630,   47.6290}, % O1P 
      {31.7210,   -0.6460,   48.8090}, % O2P 
      {32.4940,    1.2540,   47.2740}, % O5' 
      {32.1610,    2.2370,   46.2560}, % C5' 
      {31.2986,    2.8190,   46.5812}, % H5' 
      {32.0980,    1.7468,   45.2845}, % H5''
      {33.3476,    3.1959,   46.1947}, % C4' 
      {33.2668,    3.8958,   45.3630}, % H4' 
      {33.3799,    3.9183,   47.4216}, % O4' 
      {34.6515,    3.7222,   48.0398}, % C1' 
      {35.2947,    4.5412,   47.7180}, % H1' 
      {35.1756,    2.4228,   47.4827}, % C2' 
      {34.6778,    1.5937,   47.9856}, % H2''
      {36.5631,    2.2672,   47.4798}, % O2' 
      {37.0163,    2.6579,   48.2305}, % H2' 
      {34.6953,    2.5043,   46.0448}, % C3' 
      {34.5444,    1.4917,   45.6706}, % H3' 
      {35.6679,    3.3009,   45.3487}, % O3' 
      {37.4804,    4.0914,   52.2559}, % N1  
      {36.9670,    4.1312,   49.9281}, % N3  
      {37.8045,    4.2519,   50.9550}, % C2  
      {35.7171,    3.8264,   50.3222}, % C4  
      {35.2668,    3.6420,   51.6115}, % C5  
      {36.2037,    3.7829,   52.6706}, % C6
      g, {
      {39.0869,    4.5552,   50.7092}, % N2  
      {33.9075,    3.3338,   51.6102}, % N7  
      {34.6126,    3.6358,   49.5108}, % N9  
      {33.5805,    3.3442,   50.3425}, % C8  
      {35.9958,    3.6512,   53.8724}, % O6  
      {38.2106,    4.2053,   52.9295}, % H1  
      {39.8218,    4.6863,   51.3896}, % H21 
      {39.3420,    4.6857,   49.7407}, % H22 
      {32.5194,    3.1070,   50.2664}} % H8  
    }.

rU_() ->
     {
      {-0.0109,    0.5907,    0.8068,  % dgf_base_tfo
        0.2217,   -0.7853,    0.5780,
        0.9751,    0.1852,   -0.1224,
       -1.4225,  -11.0956,   -2.5217},
      {-0.8313,   -0.4738,   -0.2906,  % p_o3'_275_tfo
        0.0649,    0.4366,   -0.8973,
        0.5521,   -0.7648,   -0.3322,
        1.6833,    6.8060,   -7.0011},
      {0.3445,   -0.7630,    0.5470,  % p_o3'_180_tfo
       -0.4628,   -0.6450,   -0.6082,
        0.8168,   -0.0436,   -0.5753,
       -6.8179,   -3.9778,   -5.9887},
      {0.5855,    0.7931,   -0.1682,  % p_o3'_60_tfo
        0.8103,   -0.5790,    0.0906,
       -0.0255,   -0.1894,   -0.9816,
        6.1203,   -7.1051,    3.1984},
      {2.6760,   -8.4960,    3.2880}, % P   
      {1.4950,   -7.6230,    3.4770}, % O1P 
      {2.9490,   -9.4640,    4.3740}, % O2P 
      {3.9730,   -7.5950,    3.0340}, % O5' 
      {5.2430,   -8.2420,    2.8260}, % C5' 
      {5.1974,   -8.8497,    1.9223}, % H5' 
      {5.5548,   -8.7348,    3.7469}, % H5''
      {6.3140,   -7.2060,    2.5510}, % C4' 
      {5.8744,   -6.2116,    2.4731}, % H4' 
      {7.2798,   -7.2260,    3.6420}, % O4' 
      {8.5733,   -6.9410,    3.1329}, % C1' 
      {8.9047,   -6.0374,    3.6446}, % H1' 
      {8.4429,   -6.6596,    1.6327}, % C2' 
      {9.2880,   -7.1071,    1.1096}, % H2''
      {8.2502,   -5.2799,    1.4754}, % O2' 
      {8.7676,   -4.7284,    2.0667}, % H2' 
      {7.1642,   -7.4416,    1.3021}, % C3' 
      {7.4125,   -8.5002,    1.2260}, % H3' 
      {6.5160,   -6.9772,    0.1267}, % O3' 
      {9.4531,   -8.1107,    3.4087}, % N1  
      {11.5931,   -9.0015,    3.6357}, % N3  
      {10.8101,   -7.8950,    3.3748}, % C2  
      {11.1439,  -10.2744,    3.9206}, % C4  
      {9.7056,  -10.4026,    3.9332}, % C5  
      {8.9192,   -9.3419,    3.6833}, % C6
      u, {
      {11.3013,   -6.8063,    3.1326}, % O2  
      {11.9431,  -11.1876,    4.1375}, % O4  
      {12.5840,   -8.8673,    3.6158}, % H3  
      {9.2891,  -11.2898,    4.1313}, % H5  
      {7.9263,   -9.4537,    3.6977}} % H6  
     }.


% -- PARTIAL INSTANTIATIONS --------------------------------------------------

%var ::= {Int, Tfo, Nuc}

absolute_pos({_I,T,_N}, P) -> tfo_apply(T, P).

atom_pos(Atom, {I,T,N}) ->
    absolute_pos({I,T,N}, p_apply(Atom, N)).

get_var(Id,[{Id,T,N}|_]) -> {Id,T,N};
get_var(Id,[_|Lst]) -> get_var(Id,Lst).


% -- SEARCH ------------------------------------------------------------------

% Sequential backtracking algorithm

search(Partial_inst,[],_) ->
    [Partial_inst];
search(Partial_inst,[{F,Arg0,Arg1}|T],Constraint) ->
    try_assignments(p_apply(F, Arg0,Arg1,Partial_inst),
                    Constraint, 
                    Partial_inst,
                    T);
search(Partial_inst,[{F,Arg0,Arg1,Arg2}|T],Constraint) ->
    try_assignments(p_apply(F, Arg0,Arg1,Arg2,Partial_inst),
                    Constraint, 
                    Partial_inst,
                    T).

try_assignments([],_,_,_) -> [];
try_assignments([V|Vs], Constraint, Partial_inst,T) ->
    case p_apply(Constraint, V, Partial_inst) of
        true -> append(search([V|Partial_inst],T,Constraint),
                       try_assignments(Vs, Constraint, Partial_inst,T));
        _ -> try_assignments(Vs, Constraint, Partial_inst,T)
    end.


% -- DOMAINS -----------------------------------------------------------------

% Primary structure:   strand A CUGCCACGUCUG, strand B CAGACGUGGCAG
%
% Secondary structure: strand A CUGCCACGUCUG
%                               ||||||
%                               GACGGUGCAGAC strand B
%
% Tertiary structure:
%
%    5' end of strand A C1----G12 3' end of strand B
%                     U2-------A11
%                    G3-------C10
%                    C4-----G9
%                     C5---G8
%                        A6
%                      G6-C7
%                     C5----G8
%                    A4-------U9
%                    G3--------C10
%                     A2-------U11
%   5' end of strand B C1----G12 3' end of strand A
%
% "helix", "stacked" and "connected" describe the spatial relationship
% between two consecutive nucleotides. E.g. the nucleotides C1 and U2
% from the strand A.
%
% "wc" (stands for Watson-Crick and is a type of base-pairing),
% and "wc-dumas" describe the spatial relationship between 
% nucleotides from two chains that are growing in opposite directions.
% E.g. the nucleotides C1 from strand A and G12 from strand B.

% Dynamic Domains

% Given,
%   "ref" a nucleotide which is already positioned,
%   "nuc" the nucleotide to be placed,
%   and "tfo" a transformation matrix which expresses the desired
%   relationship between "ref" and "nuc",
% the function "dgf-base" computes the transformation matrix that
% places the nucleotide "nuc" in the given relationship to "ref".

dgf_base(Tfo, V, Nuc) ->
    {_I,_T,N} = V,
    tfo_combine(nuc_dgf_base_tfo(Nuc),
                tfo_combine(Tfo,tfo_inv_ortho(process_type(type(N),V)))).

process_type(a,V) ->
    tfo_align(atom_pos(nuc_C1_, V),atom_pos(rA_N9, V),atom_pos(nuc_C4, V));
process_type(c,V) ->
    tfo_align(atom_pos(nuc_C1_, V),atom_pos(nuc_N1, V),atom_pos(nuc_C2, V));
process_type(g,V) ->
    tfo_align(atom_pos(nuc_C1_, V),atom_pos(rG_N9, V),atom_pos(nuc_C4, V));
process_type(_,V) ->
    tfo_align(atom_pos(nuc_C1_, V),atom_pos(nuc_N1, V),atom_pos(nuc_C2, V)).


% Placement of first nucleotide.

reference(Nuc,I,_) -> 
    [{I,tfo_id(),Nuc}].

% The transformation matrix for wc is from:
%
% Chandrasekaran R. et al (1989) A Re-Examination of the Crystal
% Structure of A-DNA Using Fiber Diffraction Data. J. Biomol.
% Struct. & Dynamics 6(6):1189-1202.

wc_tfo() ->
  {
      -1.0000,   0.0028,  -0.0019,
       0.0028,   0.3468,  -0.9379,
      -0.0019,  -0.9379,  -0.3468,
      -0.0080,   6.0730,   8.7208
   }.

wc(Nuc,I,J,Partial_inst) ->
    [{I,dgf_base(wc_tfo(),get_var(J,Partial_inst),Nuc),Nuc}].

wc_dumas_tfo() ->
  {
      -0.9737,  -0.1834,   0.1352,
      -0.1779,   0.2417,  -0.9539,
       0.1422,  -0.9529,  -0.2679,
       0.4837,   6.2649,   8.0285
   }.
         
wc_dumas(Nuc,I,J,Partial_inst) ->
    [{I,dgf_base(wc_dumas_tfo(),get_var(J,Partial_inst),Nuc),Nuc}].

helix5__tfo() ->
  {
       0.9886,  -0.0961,   0.1156,
       0.1424,   0.8452,  -0.5152,
      -0.0482,   0.5258,   0.8492,
      -3.8737,   0.5480,   3.8024
  }.

helix5_(Nuc,I,J,Partial_inst) ->
    [{I,dgf_base(helix5__tfo(),get_var(J,Partial_inst),Nuc),Nuc}].

helix3__tfo() ->
  {
       0.9886,   0.1424,  -0.0482,
      -0.0961,   0.8452,   0.5258,
       0.1156,  -0.5152,   0.8492,
       3.4426,   2.0474,  -3.7042
   }.

helix3_(Nuc,I,J,Partial_inst) ->
    [{I,dgf_base(helix3__tfo(),get_var(J,Partial_inst),Nuc),Nuc}].

g37_a38_tfo() ->
  {
       0.9991,   0.0164,  -0.0387,
      -0.0375,   0.7616,  -0.6470,
       0.0189,   0.6478,   0.7615,
      -3.3018,   0.9975,   2.5585
   }.

g37_a38(Nuc,I,J,Partial_inst) ->
    {I,dgf_base(g37_a38_tfo(),get_var(J,Partial_inst),Nuc),Nuc}.

stacked5_(Nuc,I,J,Partial_inst) ->
    [g37_a38(Nuc,I,J,Partial_inst) | helix5_(Nuc,I,J,Partial_inst)].

a38_g37_tfo() ->
  {
       0.9991,  -0.0375,   0.0189,
       0.0164,   0.7616,   0.6478, 
      -0.0387,  -0.6470,   0.7615,
       3.3819,   0.7718,  -2.5321
   }.

a38_g37(Nuc,I,J,Partial_inst) ->
    {I,dgf_base(a38_g37_tfo(),get_var(J,Partial_inst),Nuc),Nuc}.
   
stacked3_(Nuc,I,J,Partial_inst) ->
    [a38_g37(Nuc,I,J,Partial_inst) | helix3_(Nuc,I,J,Partial_inst)].

p_o3_(Nucs,I,J,Partial_inst) ->
    generate([],Nucs,I,J,Partial_inst).


generate(Domains,[],_,_,_) -> 
    Domains;
generate(Domains,[N|Ns],I,J,Partial_inst) ->
    Ref = get_var(J,Partial_inst),
    Align = tfo_inv_ortho(tfo_align(atom_pos(nuc_O3_,Ref),
                                    atom_pos(nuc_C3_,Ref),
                                    atom_pos(nuc_C4_,Ref))),
    generate([{I,tfo_combine(nuc_p_o3__60_tfo(N),Align),N},
              {I,tfo_combine(nuc_p_o3__180_tfo(N),Align),N},
              {I,tfo_combine(nuc_p_o3__275_tfo(N),Align),N} | Domains],
             Ns,I,J,Partial_inst).


% -- PROBLEM STATEMENT -------------------------------------------------------

% Define anticodon problem -- Science 253:1255 Figure 3a, 3b and 3c

% anticodon_domains() ->
%     [
%      {reference, rC(), 27},
%      {helix5_, rC(), 28, 27},
%      {helix5_, rA(), 29, 28},
%      {helix5_, rG(), 30, 29},
%      {helix5_, rA(), 31, 30},
%      {wc, rU(), 39, 31},
%      {helix5_, rC(), 40, 39},
%      {helix5_, rU(), 41, 40},
%      {helix5_, rG(), 42, 41},
%      {helix5_, rG(), 43, 42},
%      {stacked3_, rA(), 38, 39},
%      {stacked3_, rG(), 37, 38},
%      {stacked3_, rA(), 36, 37},
%      {stacked3_, rA(), 35, 36},
%      {stacked3_, rG(), 34, 35}, %<-. Distance
%      {p_o3_, rCs(), 32, 31},    %  | Constraint
%      {p_o3_, rUs(), 33, 32}     %<-' 3.0 Angstroms
%     ].

% Anticodon constraint

anticodon_constraint({33,T,N},Partial_inst) ->
    check0(dist(34,{33,T,N},Partial_inst));
anticodon_constraint(_,_) -> true.

check0(Dist) when is_float(Dist), Dist =< 3.0 -> true;
check0(_) -> false.

dist(J,V,Partial_inst) ->
    pt_dist(atom_pos(nuc_P, get_var(J,Partial_inst)), 
                   atom_pos(nuc_O3_,V)).

% anticodon() -> search([], anticodon_domains(), anticodon_constraint).

% Define pseudoknot problem -- Science 253:1255 Figure 4a and 4b
pseudoknot_domains() ->
    [
     {reference, rA(), 23},
     {wc_dumas, rU(), 8, 23},
     {helix3_, rG(), 22, 23},
     {wc_dumas, rC(), 9, 22},
     {helix3_, rG(), 21, 22},
     {wc_dumas, rC(), 10, 21},
     {helix3_, rC(), 20, 21},
     {wc_dumas, rG(), 11, 20},
     {helix3_, rU_(), 19, 20}, %<-.
     {wc_dumas, rA(), 12, 19}, %  | Distance
%                              %  | Constraint
% Helix 1                      %  | 4.0 Angstroms
     {helix3_, rC(), 3, 19},   %  |
     {wc_dumas, rG(), 13, 3},  %  |
     {helix3_, rC(), 2, 3},    %  |
     {wc_dumas, rG(), 14, 2},  %  |
     {helix3_, rC(), 1, 2},    %  |
     {wc_dumas, rG_(), 15, 1}, %  |
%                              %  |
% L2 LOOP                      %  |
     {p_o3_, rUs(), 16, 15},   %  |
     {p_o3_, rCs(), 17, 16},   %  |
     {p_o3_, rAs(), 18, 17},   %<-'
%
% L1 LOOP
     {helix3_, rU(), 7, 8},   %<-.
     {p_o3_, rCs(), 4, 3},    %  | Constraint
     {stacked5_, rU(), 5, 4}, %  | 4.5 Angstroms
     {stacked5_, rC(), 6, 5}  %<-'
    ].
  
% Pseudoknot constraint

pseudoknot_constraint({18,T,N}, Partial_inst) ->
    check1(dist(19, {18,T,N}, Partial_inst));
pseudoknot_constraint({6,T,N}, Partial_inst) ->
    check2(dist(7, {6,T,N}, Partial_inst));
pseudoknot_constraint(_,_) -> true.

check1(Dist) when is_float(Dist), Dist =< 4.0 -> true;
check1(_) -> false.

check2(Dist) when is_float(Dist), Dist =< 4.5 -> true;
check2(_) -> false.

do_pseudoknot() -> search([], pseudoknot_domains(), pseudoknot_constraint).

% -- TESTING -----------------------------------------------------------------

list_of_atoms(N) ->
    append(list_of_common_atoms(N),list_of_specific_atoms(N)).

list_of_common_atoms
 ({
        _,_,_,_,
        P,O1p,O2p,O5_,C5_,H5_,H5__,C4_,H4_,O4_,C1_,H1_,C2_,H2__,O2_,H2_,
        C3_,H3_,O3_,N1,N3,C2,C4,C5,C6,
        _,_
  }) ->
    [P,O1p,O2p,O5_,C5_,H5_,H5__,C4_,H4_,O4_,C1_,H1_,C2_,H2__,O2_,H2_,
        C3_,H3_,O3_,N1,N3,C2,C4,C5,C6].

list_of_specific_atoms({_,_,_,_,_,_,_,_,_,_,
                        _,_,_,_,_,_,_,_,_,_,
                        _,_,_,_,_,_,_,_,_,a,
                        {N6,N7,N9,C8,H2,H61,H62,H8}}) ->
    [N6,N7,N9,C8,H2,H61,H62,H8];
list_of_specific_atoms({_,_,_,_,_,_,_,_,_,_,
                        _,_,_,_,_,_,_,_,_,_,
                        _,_,_,_,_,_,_,_,_,c,
                        {N4,O2,H41,H42,H5,H6}}) ->
    [N4,O2,H41,H42,H5,H6];
list_of_specific_atoms({_,_,_,_,_,_,_,_,_,_,
                        _,_,_,_,_,_,_,_,_,_,
                        _,_,_,_,_,_,_,_,_,g,
                        {N2,N7,N9,C8,O6,H1,H21,H22,H8}}) ->
    [N2,N7,N9,C8,O6,H1,H21,H22,H8];
list_of_specific_atoms({_,_,_,_,_,_,_,_,_,_,
                        _,_,_,_,_,_,_,_,_,_,
                        _,_,_,_,_,_,_,_,_,u,
                        {O2,O4,H3,H5,H6}}) ->
    [O2,O4,H3,H5,H6].

var_most_distant_atom(V) ->
    {_,_,N} = V,
    maximum(map(distance,V,list_of_atoms(N))).

distance(V,P) ->
    {X,Y,Z} = absolute_pos(V,P),
    distance(X,Y,Z).

distance(X,Y,Z) when is_float(X), is_float(Y), is_float(Z) ->
    math:sqrt(X * X + Y * Y + Z * Z).

sol_most_distant_atom(S) ->
    maximum(map(var_most_distant_atom,S)).

most_distant_atom(Sols) ->
    maximum(map(sol_most_distant_atom, Sols)).

maximum([H|T]) ->
    max(T,H).

max([H|T],M) when is_float(H), is_float(M), H > M ->
    max(T,H);
max([_|T],M) ->
    max(T,M);
max([],M) -> M.

%%
%% The map/2,3 functions rewritten to use a list comprehension,
%% just to cover the letrec handling in the inliner.
%%
map(Func, L) ->
    [p_apply(Func, H) || H <- L].

map(Func, Arg, L) ->
    [p_apply(Func, Arg, H) || H <- L].

% p_apply implements higher order functions
p_apply(sol_most_distant_atom, S) -> sol_most_distant_atom(S);
p_apply(var_most_distant_atom, V) -> var_most_distant_atom(V);
p_apply(nuc_C1_, X) -> nuc_C1_(X);
p_apply(nuc_C2, X) -> nuc_C2(X);
p_apply(nuc_C3_, X) -> nuc_C3_(X);
p_apply(nuc_C4, X) -> nuc_C4(X);
p_apply(nuc_C4_, X) -> nuc_C4_(X);
p_apply(nuc_N1, X) -> nuc_N1(X);
p_apply(nuc_O3_, X) -> nuc_O3_(X);
p_apply(nuc_P, X) -> nuc_P(X);
p_apply(nuc_dgf_base_tfo, X) -> nuc_dgf_base_tfo(X);
p_apply(nuc_p_o3__180_tfo, X) -> nuc_p_o3__180_tfo(X);
p_apply(nuc_p_o3__275_tfo, X) -> nuc_p_o3__275_tfo(X);
p_apply(nuc_p_o3__60_tfo, X) -> nuc_p_o3__60_tfo(X);
p_apply(rA_N9, X) -> rA_N9(X);
p_apply(rG_N9, X) -> rG_N9(X).

p_apply(anticodon_constraint, V, P) -> anticodon_constraint(V, P);
p_apply(pseudoknot_constraint, V, P) -> pseudoknot_constraint(V, P);
p_apply(distance, V, P) -> distance(V, P).

p_apply(reference, A1, A2, A3) -> reference(A1, A2, A3).

p_apply(helix5_, A1, A2, A3, A4) -> helix5_(A1, A2, A3, A4);
p_apply(wc, A1, A2, A3, A4) -> wc(A1, A2, A3, A4);
p_apply(stacked3_, A1, A2, A3, A4) -> stacked3_(A1, A2, A3, A4);
p_apply(p_o3_, A1, A2, A3, A4) -> p_o3_(A1, A2, A3, A4);
p_apply(wc_dumas, A1, A2, A3, A4) -> wc_dumas(A1, A2, A3, A4);
p_apply(helix3_, A1, A2, A3, A4) -> helix3_(A1, A2, A3, A4);
p_apply(stacked5_, A1, A2, A3, A4) -> stacked5_(A1, A2, A3, A4).

loop(0, R) -> R;
loop(N, _R) -> loop(N-1,most_distant_atom(do_pseudoknot())).
