// @HEADER
// *****************************************************************************
//        MueLu: A package for multigrid based preconditioning
//
// Copyright 2012 NTESS and the MueLu contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#ifndef MUELU_SAPFACTORY_DECL_HPP
#define MUELU_SAPFACTORY_DECL_HPP

#include <string>

#include "MueLu_ConfigDefs.hpp"

#include <Tpetra_KokkosCompat_ClassicNodeAPI_Wrapper.hpp>

#include "MueLu_SaPFactory_fwd.hpp"

#include "MueLu_Level_fwd.hpp"
#include "MueLu_ParameterListAcceptor.hpp"
#include "MueLu_PerfUtils_fwd.hpp"
#include "MueLu_PFactory.hpp"
#include "MueLu_TentativePFactory_fwd.hpp"

namespace MueLu {

/*!
  @class SaPFactory class.
  @brief Factory for building Smoothed Aggregation prolongators.
  @ingroup MueLuTransferClasses

  ## Input/output of SaPFactory ##

  ### User parameters of SaPFactory ###
  Parameter | type | default | master.xml | validated | requested | description
  ----------|------|---------|:----------:|:---------:|:---------:|------------
  | sa: damping factor                     | double  | 1.33  | * | * |   | Damping factor for smoothed aggregation transfer operators |
  | sa: calculate eigenvalue estimate      | bool    | false | * | * |   | Force calculation of eigenvalue estimate during prolongator smoothing |
  | sa: eigenvalue estimate num iterations | int     | 10    | * | * |   | Number of power iterations to estimate max eigenvalue |
  | A                                      | Factory | null  |   | * | * | Generating factory of the matrix A used during the prolongator smoothing process |
  | P                                      | Factory | null  |   | * | * | Tentative prolongator factory (should be a TentativePFactory object). Note that if "P" is not provided the FactoryManager tries the factory which is generating the variable "Ptent" as input for "P". In the standard case using the default settings this is the non-smoothed tentative prolongation operator. |

  The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
  The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see SaPFactory::GetValidParameters).<br>
  The * in the @c requested column states that the data is requested as input with all dependencies (see SaPFactory::DeclareInput).

  ### Variables provided by SaPFactory ###

  After SaPFactory::Build the following data is available (if requested)

  Parameter | generated by | description
  ----------|--------------|------------
  | P       | SaPFactory   | Smoothed prolongator

*/
template <class Scalar        = DefaultScalar,
          class LocalOrdinal  = DefaultLocalOrdinal,
          class GlobalOrdinal = DefaultGlobalOrdinal,
          class Node          = DefaultNode>
class SaPFactory : public PFactory {
 public:
  typedef LocalOrdinal local_ordinal_type;
  typedef GlobalOrdinal global_ordinal_type;
  typedef typename Node::execution_space execution_space;
  typedef Node node_type;

 private:
#undef MUELU_SAPFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

 public:
  //! @name Constructors/Destructors.
  //@{

  /*! @brief Constructor.
    User can supply a factory for generating the tentative prolongator.
  */
  SaPFactory();

  //! Destructor.
  virtual ~SaPFactory();

  RCP<const ParameterList> GetValidParameterList() const;

  //@}

  //! Input
  //@{

  void DeclareInput(Level& fineLevel, Level& coarseLevel) const;

  //@}

  //! @name Build methods.
  //@{

  /*!
    @brief Build method.

    Builds smoothed aggregation prolongator and returns it in <tt>coarseLevel</tt>.
    //FIXME what does the return code mean (unclear in MueMat)?
    */
  void Build(Level& fineLevel, Level& coarseLevel) const;

  void BuildP(Level& fineLevel, Level& coarseLevel) const;

  void SatisfyPConstraintsNonKokkos(RCP<Matrix> A, RCP<Matrix>& P) const;

  void optimalSatisfyPConstraintsForScalarPDEsNonKokkos(RCP<Matrix>& P) const;

  void SatisfyPConstraints(RCP<Matrix> A, RCP<Matrix>& P) const;

  void optimalSatisfyPConstraintsForScalarPDEs(RCP<Matrix>& P) const;

  bool constrainRow(Scalar* orig, LocalOrdinal nEntries, Scalar leftBound, Scalar rghtBound, Scalar rsumTarget, Scalar* fixedUnsorted, Scalar* scalarData) const;

  //@}
};

}  // namespace MueLu

#define MUELU_SAPFACTORY_SHORT
#endif  // MUELU_SAPFACTORY_DECL_HPP
