/* BSE - Bedevilled Sound Engine
 * Copyright (C) 1997-2004 Tim Janik
 * Copyright (C) 2001 Stefan Westerfeld
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include "bsemathsignal.h"


/* --- frequency modulation --- */
void
bse_frequency_modulator (const BseFrequencyModulator *fm,
			 guint                        n_values,
			 const gfloat                *ifreq,
			 const gfloat                *ifmod,
			 gfloat                      *fm_buffer)
{
  gfloat *bound, fine_tune, fm_strength;
  gboolean with_fine_tune;

  fine_tune = bse_cent_factor (fm->fine_tune);
  with_fine_tune = fm->fine_tune != 0;
  fm_strength = fm->fm_strength;
  
  bound = fm_buffer + n_values;
  if (ifreq && ifmod)
    {
      if (fm->exponential_fm)
	{
	  if (with_fine_tune)
	    do {
	      *fm_buffer++ = *ifreq++ * bse_approx5_exp2 (fm_strength * *ifmod++) * fine_tune;
	    } while (fm_buffer < bound);
	  else
	    do {
	      *fm_buffer++ = *ifreq++ * bse_approx5_exp2 (fm_strength * *ifmod++);
	    } while (fm_buffer < bound);
	}
      else
	{
	  if (with_fine_tune)
	    do {
	      *fm_buffer++ = *ifreq++ * (1 + fm_strength * *ifmod++) * fine_tune;
	    } while (fm_buffer < bound);
	  else
	    do {
	      *fm_buffer++ = *ifreq++ * (1 + fm_strength * *ifmod++);
	    } while (fm_buffer < bound);
	}
    }
  else if (ifmod)
    {
      gfloat signal_freq = fm->signal_freq * fine_tune;

      if (fm->exponential_fm)
	do {
	  *fm_buffer++ = signal_freq * bse_approx5_exp2 (fm_strength * *ifmod++);
	} while (fm_buffer < bound);
      else
	do {
	  *fm_buffer++ = signal_freq * (1 + fm_strength * *ifmod++);
	} while (fm_buffer < bound);
    }
  else if (ifreq)
    {
      if (with_fine_tune)
	do {
	  *fm_buffer++ = *ifreq++ * fine_tune;
	} while (fm_buffer < bound);
      else
	do {
	  *fm_buffer++ = *ifreq++;
	} while (fm_buffer < bound);
    }
  else
    {
      gfloat signal_freq = fm->signal_freq * fine_tune;

      do {
	*fm_buffer++ = signal_freq;
      } while (fm_buffer < bound);
    }
}


/* --- windows --- */
double
bse_window_bartlett (double x)	/* triangle */
{
  if (fabs (x) > 1)
    return 0;

  return 1.0 - fabs (x);
}

double
bse_window_blackman (double x)
{
  if (fabs (x) > 1)
    return 0;

  return 0.42 + 0.5 * cos (PI * x) + 0.08 * cos (2.0 * PI * x);
}

double
bse_window_cos (double x)	/* von Hann window */
{
  if (fabs (x) > 1)
    return 0;

  return 0.5 * cos (x * PI) + 0.5;
}

double
bse_window_hamming (double x)	/* sharp (rectangle) cutoffs at boundaries */
{
  if (fabs (x) > 1)
    return 0;

  return 0.54 + 0.46 * cos (PI * x);
}

double
bse_window_sinc (double x)	/* noramlied C. Lanczos window */
{
  if (fabs (x) > 1)
    return 0;
  x = x * PI;
  if (fabs (x) < 1e-12)
    return 1.0;
  else
    return sin (x) / x;
}

double
bse_window_rect (double x)	/* a square */
{
  if (fabs (x) > 1)
    return 0;
  return 1.0;
}

/*
cos_roll_off(x)= x>fh?0:x<fl?1:cos(pi/2.*((fl-x)/(fh-fl))) 
*/


/* --- note factors --- */
static const double semitone_table265[132 + 1 + 132] = {
  0.00048828125,             /* 2^(1/12*-132) */
  0.0005173159640426246409,  /* 2^(1/12*-131) */
  0.00054807717202606102609, /* 2^(1/12*-130) */
  0.00058066753662242239586, /* 2^(1/12*-129) */
  0.00061519582514398103748, /* 2^(1/12*-128) */
  0.0006517772725439620922,  /* 2^(1/12*-127) */
  0.0006905339660024878168,  /* 2^(1/12*-126) */
  0.00073159525238119213809, /* 2^(1/12*-125) */
  0.00077509816990634739978, /* 2^(1/12*-124) */
  0.00082118790552120560843, /* 2^(1/12*-123) */
  0.00087001827943392510229, /* 2^(1/12*-122) */
  0.00092175225847821630531, /* 2^(1/12*-121) */
  0.0009765625,             /* 2^(1/12*-120) */
  0.0010346319280852492818, /* 2^(1/12*-119) */
  0.0010961543440521220522, /* 2^(1/12*-118) */
  0.0011613350732448447917, /* 2^(1/12*-117) */
  0.001230391650287962075,  /* 2^(1/12*-116) */
  0.0013035545450879241844, /* 2^(1/12*-115) */
  0.0013810679320049756336, /* 2^(1/12*-114) */
  0.0014631905047623842762, /* 2^(1/12*-113) */
  0.0015501963398126947996, /* 2^(1/12*-112) */
  0.0016423758110424112169, /* 2^(1/12*-111) */
  0.0017400365588678502046, /* 2^(1/12*-110) */
  0.0018435045169564326106, /* 2^(1/12*-109) */
  0.001953125,              /* 2^(1/12*-108) */
  0.0020692638561704985636, /* 2^(1/12*-107) */
  0.0021923086881042441044, /* 2^(1/12*-106) */
  0.0023226701464896895834, /* 2^(1/12*-105) */
  0.0024607833005759241499, /* 2^(1/12*-104) */
  0.0026071090901758483688, /* 2^(1/12*-103) */
  0.0027621358640099512672, /* 2^(1/12*-102) */
  0.0029263810095247685523, /* 2^(1/12*-101) */
  0.0031003926796253895991, /* 2^(1/12*-100) */
  0.0032847516220848224337, /* 2^(1/12*-99) */
  0.0034800731177357004091, /* 2^(1/12*-98) */
  0.0036870090339128652213, /* 2^(1/12*-97) */
  0.00390625,               /* 2^(1/12*-96) */
  0.0041385277123409971272, /* 2^(1/12*-95) */
  0.0043846173762084882087, /* 2^(1/12*-94) */
  0.0046453402929793791669, /* 2^(1/12*-93) */
  0.0049215666011518482999, /* 2^(1/12*-92) */
  0.0052142181803516967376, /* 2^(1/12*-91) */
  0.0055242717280199025344, /* 2^(1/12*-90) */
  0.0058527620190495371047, /* 2^(1/12*-89) */
  0.0062007853592507791982, /* 2^(1/12*-88) */
  0.0065695032441696448674, /* 2^(1/12*-87) */
  0.0069601462354714008183, /* 2^(1/12*-86) */
  0.0073740180678257304425, /* 2^(1/12*-85) */
  0.0078125,                /* 2^(1/12*-84) */
  0.0082770554246819942544, /* 2^(1/12*-83) */
  0.0087692347524169764174, /* 2^(1/12*-82) */
  0.0092906805859587583337, /* 2^(1/12*-81) */
  0.0098431332023036965997, /* 2^(1/12*-80) */
  0.010428436360703393475, /* 2^(1/12*-79) */
  0.011048543456039805069, /* 2^(1/12*-78) */
  0.011705524038099074209, /* 2^(1/12*-77) */
  0.012401570718501558396, /* 2^(1/12*-76) */
  0.013139006488339289735, /* 2^(1/12*-75) */
  0.013920292470942801637, /* 2^(1/12*-74) */
  0.014748036135651460885, /* 2^(1/12*-73) */
  0.015625,                /* 2^(1/12*-72) */
  0.016554110849363988509, /* 2^(1/12*-71) */
  0.017538469504833952835, /* 2^(1/12*-70) */
  0.018581361171917516667, /* 2^(1/12*-69) */
  0.019686266404607393199, /* 2^(1/12*-68) */
  0.02085687272140678695,  /* 2^(1/12*-67) */
  0.022097086912079610138, /* 2^(1/12*-66) */
  0.023411048076198148419, /* 2^(1/12*-65) */
  0.024803141437003116793, /* 2^(1/12*-64) */
  0.02627801297667857947,  /* 2^(1/12*-63) */
  0.027840584941885603273, /* 2^(1/12*-62) */
  0.02949607227130292177,  /* 2^(1/12*-61) */
  0.03125,                 /* 2^(1/12*-60) */
  0.033108221698727977018, /* 2^(1/12*-59) */
  0.03507693900966790567,  /* 2^(1/12*-58) */
  0.037162722343835033335, /* 2^(1/12*-57) */
  0.039372532809214786399, /* 2^(1/12*-56) */
  0.041713745442813573901, /* 2^(1/12*-55) */
  0.044194173824159220275, /* 2^(1/12*-54) */
  0.046822096152396296837, /* 2^(1/12*-53) */
  0.049606282874006233586, /* 2^(1/12*-52) */
  0.052556025953357158939, /* 2^(1/12*-51) */
  0.055681169883771206546, /* 2^(1/12*-50) */
  0.05899214454260584354,  /* 2^(1/12*-49) */
  0.0625,                  /* 2^(1/12*-48) */
  0.066216443397455954035, /* 2^(1/12*-47) */
  0.07015387801933581134,  /* 2^(1/12*-46) */
  0.07432544468767006667,  /* 2^(1/12*-45) */
  0.078745065618429572798, /* 2^(1/12*-44) */
  0.083427490885627147802, /* 2^(1/12*-43) */
  0.08838834764831844055,  /* 2^(1/12*-42) */
  0.093644192304792593675, /* 2^(1/12*-41) */
  0.099212565748012467172, /* 2^(1/12*-40) */
  0.10511205190671431788, /* 2^(1/12*-39) */
  0.11136233976754241309, /* 2^(1/12*-38) */
  0.11798428908521168708, /* 2^(1/12*-37) */
  0.125,                  /* 2^(1/12*-36) */
  0.13243288679491190807, /* 2^(1/12*-35) */
  0.14030775603867162268, /* 2^(1/12*-34) */
  0.14865088937534013334, /* 2^(1/12*-33) */
  0.1574901312368591456,  /* 2^(1/12*-32) */
  0.1668549817712542956,  /* 2^(1/12*-31) */
  0.1767766952966368811,  /* 2^(1/12*-30) */
  0.18728838460958518735, /* 2^(1/12*-29) */
  0.19842513149602493434, /* 2^(1/12*-28) */
  0.21022410381342863576, /* 2^(1/12*-27) */
  0.22272467953508482619, /* 2^(1/12*-26) */
  0.23596857817042337416, /* 2^(1/12*-25) */
  0.25,                   /* 2^(1/12*-24) */
  0.26486577358982381614, /* 2^(1/12*-23) */
  0.28061551207734324536, /* 2^(1/12*-22) */
  0.29730177875068026668, /* 2^(1/12*-21) */
  0.31498026247371829119, /* 2^(1/12*-20) */
  0.33370996354250859121, /* 2^(1/12*-19) */
  0.3535533905932737622,  /* 2^(1/12*-18) */
  0.3745767692191703747,  /* 2^(1/12*-17) */
  0.39685026299204986869, /* 2^(1/12*-16) */
  0.42044820762685727152, /* 2^(1/12*-15) */
  0.44544935907016965237, /* 2^(1/12*-14) */
  0.47193715634084674832, /* 2^(1/12*-13) */
  0.5,                    /* 2^(1/12*-12) */
  0.52973154717964763228, /* 2^(1/12*-11) */
  0.56123102415468649072, /* 2^(1/12*-10) */
  0.59460355750136053336, /* 2^(1/12*-9) */
  0.62996052494743658238, /* 2^(1/12*-8) */
  0.66741992708501718242, /* 2^(1/12*-7) */
  0.7071067811865475244,  /* 2^(1/12*-6) */
  0.7491535384383407494,  /* 2^(1/12*-5) */
  0.79370052598409973738, /* 2^(1/12*-4) */
  0.84089641525371454303, /* 2^(1/12*-3) */
  0.89089871814033930474, /* 2^(1/12*-2) */
  0.94387431268169349664, /* 2^(1/12*-1) */
  1.0,                   /* 2^(1/12*0) */
  1.0594630943592952646, /* 2^(1/12*1) */
  1.1224620483093729814, /* 2^(1/12*2) */
  1.1892071150027210667, /* 2^(1/12*3) */
  1.2599210498948731648, /* 2^(1/12*4) */
  1.3348398541700343648, /* 2^(1/12*5) */
  1.4142135623730950488, /* 2^(1/12*6) */
  1.4983070768766814988, /* 2^(1/12*7) */
  1.5874010519681994748, /* 2^(1/12*8) */
  1.6817928305074290861, /* 2^(1/12*9) */
  1.7817974362806786095, /* 2^(1/12*10) */
  1.8877486253633869933, /* 2^(1/12*11) */
  2.0,                   /* 2^(1/12*12) */
  2.1189261887185905291, /* 2^(1/12*13) */
  2.2449240966187459629, /* 2^(1/12*14) */
  2.3784142300054421334, /* 2^(1/12*15) */
  2.5198420997897463295, /* 2^(1/12*16) */
  2.6696797083400687297, /* 2^(1/12*17) */
  2.8284271247461900976, /* 2^(1/12*18) */
  2.9966141537533629976, /* 2^(1/12*19) */
  3.1748021039363989495, /* 2^(1/12*20) */
  3.3635856610148581721, /* 2^(1/12*21) */
  3.563594872561357219,  /* 2^(1/12*22) */
  3.7754972507267739866, /* 2^(1/12*23) */
  4.0,                   /* 2^(1/12*24) */
  4.2378523774371810582, /* 2^(1/12*25) */
  4.4898481932374919257, /* 2^(1/12*26) */
  4.7568284600108842669, /* 2^(1/12*27) */
  5.0396841995794926591, /* 2^(1/12*28) */
  5.3393594166801374593, /* 2^(1/12*29) */
  5.6568542494923801952, /* 2^(1/12*30) */
  5.9932283075067259952, /* 2^(1/12*31) */
  6.349604207872797899,  /* 2^(1/12*32) */
  6.7271713220297163442, /* 2^(1/12*33) */
  7.1271897451227144379, /* 2^(1/12*34) */
  7.5509945014535479731, /* 2^(1/12*35) */
  8.0,                   /* 2^(1/12*36) */
  8.4757047548743621165, /* 2^(1/12*37) */
  8.9796963864749838515, /* 2^(1/12*38) */
  9.5136569200217685337, /* 2^(1/12*39) */
  10.079368399158985318, /* 2^(1/12*40) */
  10.678718833360274919, /* 2^(1/12*41) */
  11.31370849898476039,  /* 2^(1/12*42) */
  11.98645661501345199,  /* 2^(1/12*43) */
  12.699208415745595798, /* 2^(1/12*44) */
  13.454342644059432688, /* 2^(1/12*45) */
  14.254379490245428876, /* 2^(1/12*46) */
  15.101989002907095946, /* 2^(1/12*47) */
  16.0,                  /* 2^(1/12*48) */
  16.951409509748724233, /* 2^(1/12*49) */
  17.959392772949967703, /* 2^(1/12*50) */
  19.027313840043537067, /* 2^(1/12*51) */
  20.158736798317970636, /* 2^(1/12*52) */
  21.357437666720549837, /* 2^(1/12*53) */
  22.627416997969520781, /* 2^(1/12*54) */
  23.972913230026903981, /* 2^(1/12*55) */
  25.398416831491191596, /* 2^(1/12*56) */
  26.908685288118865377, /* 2^(1/12*57) */
  28.508758980490857752, /* 2^(1/12*58) */
  30.203978005814191893, /* 2^(1/12*59) */
  32.0,                  /* 2^(1/12*60) */
  33.902819019497448466, /* 2^(1/12*61) */
  35.918785545899935406, /* 2^(1/12*62) */
  38.054627680087074135, /* 2^(1/12*63) */
  40.317473596635941273, /* 2^(1/12*64) */
  42.714875333441099675, /* 2^(1/12*65) */
  45.254833995939041562, /* 2^(1/12*66) */
  47.945826460053807962, /* 2^(1/12*67) */
  50.796833662982383192, /* 2^(1/12*68) */
  53.817370576237730754, /* 2^(1/12*69) */
  57.017517960981715503, /* 2^(1/12*70) */
  60.407956011628383785, /* 2^(1/12*71) */
  64.0,                  /* 2^(1/12*72) */
  67.805638038994896932, /* 2^(1/12*73) */
  71.837571091799870812, /* 2^(1/12*74) */
  76.10925536017414827,  /* 2^(1/12*75) */
  80.634947193271882545, /* 2^(1/12*76) */
  85.429750666882199349, /* 2^(1/12*77) */
  90.509667991878083123, /* 2^(1/12*78) */
  95.891652920107615923, /* 2^(1/12*79) */
  101.59366732596476638, /* 2^(1/12*80) */
  107.63474115247546151, /* 2^(1/12*81) */
  114.03503592196343101, /* 2^(1/12*82) */
  120.81591202325676757, /* 2^(1/12*83) */
  128.0,                 /* 2^(1/12*84) */
  135.61127607798979386, /* 2^(1/12*85) */
  143.67514218359974162, /* 2^(1/12*86) */
  152.21851072034829654, /* 2^(1/12*87) */
  161.26989438654376509, /* 2^(1/12*88) */
  170.8595013337643987,  /* 2^(1/12*89) */
  181.01933598375616625, /* 2^(1/12*90) */
  191.78330584021523185, /* 2^(1/12*91) */
  203.18733465192953277, /* 2^(1/12*92) */
  215.26948230495092302, /* 2^(1/12*93) */
  228.07007184392686201, /* 2^(1/12*94) */
  241.63182404651353514, /* 2^(1/12*95) */
  256.0,                 /* 2^(1/12*96) */
  271.22255215597958773, /* 2^(1/12*97) */
  287.35028436719948325, /* 2^(1/12*98) */
  304.43702144069659308, /* 2^(1/12*99) */
  322.53978877308753018, /* 2^(1/12*100) */
  341.7190026675287974,  /* 2^(1/12*101) */
  362.03867196751233249, /* 2^(1/12*102) */
  383.56661168043046369, /* 2^(1/12*103) */
  406.37466930385906554, /* 2^(1/12*104) */
  430.53896460990184603, /* 2^(1/12*105) */
  456.14014368785372403, /* 2^(1/12*106) */
  483.26364809302707028, /* 2^(1/12*107) */
  512.0,                 /* 2^(1/12*108) */
  542.44510431195917546, /* 2^(1/12*109) */
  574.70056873439896649, /* 2^(1/12*110) */
  608.87404288139318616, /* 2^(1/12*111) */
  645.07957754617506036, /* 2^(1/12*112) */
  683.43800533505759479, /* 2^(1/12*113) */
  724.07734393502466499, /* 2^(1/12*114) */
  767.13322336086092739, /* 2^(1/12*115) */
  812.74933860771813107, /* 2^(1/12*116) */
  861.07792921980369206, /* 2^(1/12*117) */
  912.28028737570744805, /* 2^(1/12*118) */
  966.52729618605414056, /* 2^(1/12*119) */
  1024.0,                /* 2^(1/12*120) */
  1084.8902086239183509, /* 2^(1/12*121) */
  1149.401137468797933,  /* 2^(1/12*122) */
  1217.7480857627863723, /* 2^(1/12*123) */
  1290.1591550923501207, /* 2^(1/12*124) */
  1366.8760106701151896, /* 2^(1/12*125) */
  1448.15468787004933,   /* 2^(1/12*126) */
  1534.2664467217218548, /* 2^(1/12*127) */
  1625.4986772154362621, /* 2^(1/12*128) */
  1722.1558584396073841, /* 2^(1/12*129) */
  1824.5605747514148961, /* 2^(1/12*130) */
  1933.0545923721082811, /* 2^(1/12*131) */
  2048.0,                /* 2^(1/12*132) */
};
const double * const bse_semitone_table = semitone_table265 + 132;

/* --- fine tune factors for -100..+100 cent --- */
static const double cent_table201[100 + 1 + 100] = {
  0.94387431268169349664, /* 2^(1/1200*-100) */
  0.94441967335506765930, /* 2^(1/1200*-99) */
  0.94496534913211618524, /* 2^(1/1200*-98) */
  0.94551134019490267099, /* 2^(1/1200*-97) */
  0.94605764672559590751, /* 2^(1/1200*-96) */
  0.94660426890646994096, /* 2^(1/1200*-95) */
  0.94715120691990413357, /* 2^(1/1200*-94) */
  0.94769846094838322441, /* 2^(1/1200*-93) */
  0.94824603117449739035, /* 2^(1/1200*-92) */
  0.94879391778094230692, /* 2^(1/1200*-91) */
  0.94934212095051920932, /* 2^(1/1200*-90) */
  0.94989064086613495337, /* 2^(1/1200*-89) */
  0.95043947771080207655, /* 2^(1/1200*-88) */
  0.95098863166763885907, /* 2^(1/1200*-87) */
  0.95153810291986938497, /* 2^(1/1200*-86) */
  0.95208789165082360322, /* 2^(1/1200*-85) */
  0.95263799804393738893, /* 2^(1/1200*-84) */
  0.95318842228275260453, /* 2^(1/1200*-83) */
  0.95373916455091716100, /* 2^(1/1200*-82) */
  0.95429022503218507919, /* 2^(1/1200*-81) */
  0.95484160391041655104, /* 2^(1/1200*-80) */
  0.95539330136957800103, /* 2^(1/1200*-79) */
  0.95594531759374214748, /* 2^(1/1200*-78) */
  0.95649765276708806401, /* 2^(1/1200*-77) */
  0.95705030707390124097, /* 2^(1/1200*-76) */
  0.95760328069857364694, /* 2^(1/1200*-75) */
  0.95815657382560379022, /* 2^(1/1200*-74) */
  0.95871018663959678045, /* 2^(1/1200*-73) */
  0.95926411932526439013, /* 2^(1/1200*-72) */
  0.95981837206742511631, /* 2^(1/1200*-71) */
  0.96037294505100424222, /* 2^(1/1200*-70) */
  0.96092783846103389896, /* 2^(1/1200*-69) */
  0.96148305248265312728, /* 2^(1/1200*-68) */
  0.96203858730110793932, /* 2^(1/1200*-67) */
  0.96259444310175138040, /* 2^(1/1200*-66) */
  0.96315062007004359091, /* 2^(1/1200*-65) */
  0.96370711839155186816, /* 2^(1/1200*-64) */
  0.96426393825195072828, /* 2^(1/1200*-63) */
  0.96482107983702196821, /* 2^(1/1200*-62) */
  0.96537854333265472764, /* 2^(1/1200*-61) */
  0.96593632892484555106, /* 2^(1/1200*-60) */
  0.96649443679969844984, /* 2^(1/1200*-59) */
  0.96705286714342496425, /* 2^(1/1200*-58) */
  0.96761162014234422567, /* 2^(1/1200*-57) */
  0.96817069598288301869, /* 2^(1/1200*-56) */
  0.96873009485157584337, /* 2^(1/1200*-55) */
  0.96928981693506497742, /* 2^(1/1200*-54) */
  0.96984986242010053851, /* 2^(1/1200*-53) */
  0.97041023149354054658, /* 2^(1/1200*-52) */
  0.97097092434235098615, /* 2^(1/1200*-51) */
  0.97153194115360586874, /* 2^(1/1200*-50) */
  0.97209328211448729528, /* 2^(1/1200*-49) */
  0.97265494741228551852, /* 2^(1/1200*-48) */
  0.97321693723439900559, /* 2^(1/1200*-47) */
  0.97377925176833450047, /* 2^(1/1200*-46) */
  0.97434189120170708655, /* 2^(1/1200*-45) */
  0.97490485572224024929, /* 2^(1/1200*-44) */
  0.97546814551776593878, /* 2^(1/1200*-43) */
  0.97603176077622463245, /* 2^(1/1200*-42) */
  0.97659570168566539775, /* 2^(1/1200*-41) */
  0.97715996843424595493, /* 2^(1/1200*-40) */
  0.97772456121023273979, /* 2^(1/1200*-39) */
  0.97828948020200096649, /* 2^(1/1200*-38) */
  0.97885472559803469042, /* 2^(1/1200*-37) */
  0.97942029758692687108, /* 2^(1/1200*-36) */
  0.97998619635737943501, /* 2^(1/1200*-35) */
  0.98055242209820333873, /* 2^(1/1200*-34) */
  0.98111897499831863174, /* 2^(1/1200*-33) */
  0.98168585524675451960, /* 2^(1/1200*-32) */
  0.98225306303264942693, /* 2^(1/1200*-31) */
  0.98282059854525106055, /* 2^(1/1200*-30) */
  0.98338846197391647262, /* 2^(1/1200*-29) */
  0.98395665350811212383, /* 2^(1/1200*-28) */
  0.98452517333741394660, /* 2^(1/1200*-27) */
  0.98509402165150740832, /* 2^(1/1200*-26) */
  0.98566319864018757467, /* 2^(1/1200*-25) */
  0.98623270449335917291, /* 2^(1/1200*-24) */
  0.98680253940103665527, /* 2^(1/1200*-23) */
  0.98737270355334426234, /* 2^(1/1200*-22) */
  0.98794319714051608649, /* 2^(1/1200*-21) */
  0.98851402035289613536, /* 2^(1/1200*-20) */
  0.98908517338093839536, /* 2^(1/1200*-19) */
  0.98965665641520689521, /* 2^(1/1200*-18) */
  0.99022846964637576952, /* 2^(1/1200*-17) */
  0.99080061326522932245, /* 2^(1/1200*-16) */
  0.99137308746266209128, /* 2^(1/1200*-15) */
  0.99194589242967891017, /* 2^(1/1200*-14) */
  0.99251902835739497389, /* 2^(1/1200*-13) */
  0.99309249543703590153, /* 2^(1/1200*-12) */
  0.99366629385993780037, /* 2^(1/1200*-11) */
  0.99424042381754732964, /* 2^(1/1200*-10) */
  0.99481488550142176449, /* 2^(1/1200*-9) */
  0.99538967910322905982, /* 2^(1/1200*-8) */
  0.99596480481474791428, /* 2^(1/1200*-7) */
  0.99654026282786783423, /* 2^(1/1200*-6) */
  0.99711605333458919778, /* 2^(1/1200*-5) */
  0.99769217652702331884, /* 2^(1/1200*-4) */
  0.99826863259739251122, /* 2^(1/1200*-3) */
  0.99884542173803015276, /* 2^(1/1200*-2) */
  0.99942254414138074953, /* 2^(1/1200*-1) */
  1.00000000000000000000, /* 2^(1/1200*0) */
  1.00057778950655485930, /* 2^(1/1200*1) */
  1.00115591285382360350, /* 2^(1/1200*2) */
  1.00173437023469589396, /* 2^(1/1200*3) */
  1.00231316184217284163, /* 2^(1/1200*4) */
  1.00289228786936707150, /* 2^(1/1200*5) */
  1.00347174850950278700, /* 2^(1/1200*6) */
  1.00405154395591583449, /* 2^(1/1200*7) */
  1.00463167440205376771, /* 2^(1/1200*8) */
  1.00521214004147591243, /* 2^(1/1200*9) */
  1.00579294106785343092, /* 2^(1/1200*10) */
  1.00637407767496938663, /* 2^(1/1200*11) */
  1.00695555005671880883, /* 2^(1/1200*12) */
  1.00753735840710875731, /* 2^(1/1200*13) */
  1.00811950292025838709, /* 2^(1/1200*14) */
  1.00870198379039901323, /* 2^(1/1200*15) */
  1.00928480121187417556, /* 2^(1/1200*16) */
  1.00986795537913970359, /* 2^(1/1200*17) */
  1.01045144648676378139, /* 2^(1/1200*18) */
  1.01103527472942701245, /* 2^(1/1200*19) */
  1.01161944030192248469, /* 2^(1/1200*20) */
  1.01220394339915583542, /* 2^(1/1200*21) */
  1.01278878421614531640, /* 2^(1/1200*22) */
  1.01337396294802185887, /* 2^(1/1200*23) */
  1.01395947979002913869, /* 2^(1/1200*24) */
  1.01454533493752364145, /* 2^(1/1200*25) */
  1.01513152858597472769, /* 2^(1/1200*26) */
  1.01571806093096469807, /* 2^(1/1200*27) */
  1.01630493216818885868, /* 2^(1/1200*28) */
  1.01689214249345558626, /* 2^(1/1200*29) */
  1.01747969210268639364, /* 2^(1/1200*30) */
  1.01806758119191599497, /* 2^(1/1200*31) */
  1.01865580995729237127, /* 2^(1/1200*32) */
  1.01924437859507683576, /* 2^(1/1200*33) */
  1.01983328730164409940, /* 2^(1/1200*34) */
  1.02042253627348233639, /* 2^(1/1200*35) */
  1.02101212570719324976, /* 2^(1/1200*36) */
  1.02160205579949213692, /* 2^(1/1200*37) */
  1.02219232674720795532, /* 2^(1/1200*38) */
  1.02278293874728338810, /* 2^(1/1200*39) */
  1.02337389199677490985, /* 2^(1/1200*40) */
  1.02396518669285285230, /* 2^(1/1200*41) */
  1.02455682303280147013, /* 2^(1/1200*42) */
  1.02514880121401900679, /* 2^(1/1200*43) */
  1.02574112143401776038, /* 2^(1/1200*44) */
  1.02633378389042414951, /* 2^(1/1200*45) */
  1.02692678878097877927, /* 2^(1/1200*46) */
  1.02752013630353650722, /* 2^(1/1200*47) */
  1.02811382665606650935, /* 2^(1/1200*48) */
  1.02870786003665234616, /* 2^(1/1200*49) */
  1.02930223664349202878, /* 2^(1/1200*50) */
  1.02989695667489808505, /* 2^(1/1200*51) */
  1.03049202032929762572, /* 2^(1/1200*52) */
  1.03108742780523241063, /* 2^(1/1200*53) */
  1.03168317930135891498, /* 2^(1/1200*54) */
  1.03227927501644839557, /* 2^(1/1200*55) */
  1.03287571514938695719, /* 2^(1/1200*56) */
  1.03347249989917561889, /* 2^(1/1200*57) */
  1.03406962946493038044, /* 2^(1/1200*58) */
  1.03466710404588228876, /* 2^(1/1200*59) */
  1.03526492384137750435, /* 2^(1/1200*60) */
  1.03586308905087736785, /* 2^(1/1200*61) */
  1.03646159987395846655, /* 2^(1/1200*62) */
  1.03706045651031270103, /* 2^(1/1200*63) */
  1.03765965915974735173, /* 2^(1/1200*64) */
  1.03825920802218514564, /* 2^(1/1200*65) */
  1.03885910329766432300, /* 2^(1/1200*66) */
  1.03945934518633870407, /* 2^(1/1200*67) */
  1.04005993388847775587, /* 2^(1/1200*68) */
  1.04066086960446665901, /* 2^(1/1200*69) */
  1.04126215253480637458, /* 2^(1/1200*70) */
  1.04186378288011371099, /* 2^(1/1200*71) */
  1.04246576084112139095, /* 2^(1/1200*72) */
  1.04306808661867811845, /* 2^(1/1200*73) */
  1.04367076041374864571, /* 2^(1/1200*74) */
  1.04427378242741384032, /* 2^(1/1200*75) */
  1.04487715286087075226, /* 2^(1/1200*76) */
  1.04548087191543268106, /* 2^(1/1200*77) */
  1.04608493979252924297, /* 2^(1/1200*78) */
  1.04668935669370643814, /* 2^(1/1200*79) */
  1.04729412282062671789, /* 2^(1/1200*80) */
  1.04789923837506905201, /* 2^(1/1200*81) */
  1.04850470355892899603, /* 2^(1/1200*82) */
  1.04911051857421875864, /* 2^(1/1200*83) */
  1.04971668362306726905, /* 2^(1/1200*84) */
  1.05032319890772024444, /* 2^(1/1200*85) */
  1.05093006463054025745, /* 2^(1/1200*86) */
  1.05153728099400680369, /* 2^(1/1200*87) */
  1.05214484820071636931, /* 2^(1/1200*88) */
  1.05275276645338249856, /* 2^(1/1200*89) */
  1.05336103595483586147, /* 2^(1/1200*90) */
  1.05396965690802432148, /* 2^(1/1200*91) */
  1.05457862951601300320, /* 2^(1/1200*92) */
  1.05518795398198436013, /* 2^(1/1200*93) */
  1.05579763050923824245, /* 2^(1/1200*94) */
  1.05640765930119196488, /* 2^(1/1200*95) */
  1.05701804056138037450, /* 2^(1/1200*96) */
  1.05762877449345591872, /* 2^(1/1200*97) */
  1.05823986130118871317, /* 2^(1/1200*98) */
  1.05885130118846660974, /* 2^(1/1200*99) */
  1.05946309435929526456, /* 2^(1/1200*100) */
};
const double * const bse_cent_table = cent_table201 + 100;


/* --- cents & init --- */
void
_bse_init_signal (void) { /* FIXME: remove */ }

/* --- bse_approx_atan1() --- */
double
bse_approx_atan1_prescale (double boost_amount)
{
  double max_boost_factor = 100;	/* atan1(x*100) gets pretty close to 1 for x=1 */
  double recip_tan_1_div_0_75 = 0.24202942695518667705824990442766; /* 1/tan(1/0.75) */
  double scale;

  g_return_val_if_fail (boost_amount >= 0 && boost_amount <= 1.0, 1.0);

  /* scale boost_amount from [0..1] to -1..1 */
  boost_amount = boost_amount * 2 - 1.0;

  /* prescale factor for atan1(x*prescale), ranges from 1/max_boost_factor..max_boost_factor */
  scale = pow (max_boost_factor, tan (boost_amount / 0.75) * recip_tan_1_div_0_75);

  /* atan1_prescale(ba)=100 ** (tan ((ba*2-1) / 0.75) * 0.24202942695518667705824990442766) */

  return scale;
}


/* --- exp2f() approximation taylor coefficients finder --- */
#if 0
#include <stdio.h>
double
exp2coeff (int n)
{
  double r = 1;
  int i;

  for (i = 1; i <= n; i++)
    {
      r *= BSE_LN2;
      r /= i;
    }
  return r;
}
/* generate taylor coefficients */
int
main (int   argc,
      char *argv[])
{
  int i;

  for (i = 0; i < 20; i++)
    printf ("#define EXP2_TAYLOR_COEFF_%u\t(%.40f)\n", i, exp2coeff (i));

  return 0;
}
/* test/bench program */
#define _GNU_SOURCE
#include <math.h> /* for main() in testprogram */
int
main (int   argc,
      char *argv[])
{
  double x, dummy = 0, l = 4;

  if (1)	/* print errors */
    for (x = -3; x < 3.01; x += 0.1)
      {
	g_print ("%+f %+1.20f \t (%.20f - %.20f)\n",
		 x, exp (x * BSE_LN2) - bse_approx5_exp2 (x),
		 exp (x * BSE_LN2), bse_approx5_exp2 (x));
      }

  if (0)	/* bench test */
    for (x = -l; x < l; x += 0.000001)
      {
	dummy += bse_approx5_exp2 (x);
	// dummy += exp2f (x);
      }

  g_print ("%f\r                            \n", dummy);

  return 0;
}
#endif  /* coeff generation */

