static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Segment colors using K-means clustering                                    \n"
" *                                                                            \n"
" * Copyright 2017 Thomas Manni <thomas.manni@free.fr>                         \n"
" *                                                                            \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_int (n_clusters, _(\"Number of clusters\"), 5)                       \n"
" description (_(\"Number of clusters\"))                                      \n"
" value_range (2, 255)                                                         \n"
"    ui_range (2, 30)                                                          \n"
"                                                                              \n"
"property_int (max_iterations, _(\"Max. Iterations\"), 5)                      \n"
" description (_(\"Maximum number of iterations\"))                            \n"
" value_range (1, G_MAXINT)                                                    \n"
"    ui_range (1, 30)                                                          \n"
"                                                                              \n"
"property_seed (seed, _(\"Random seed\"), rand)                                \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_NAME      segment_kmeans                                      \n"
"#define GEGL_OP_C_SOURCE  segment-kmeans.c                                    \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#define MAX_PIXELS 100000                                                     \n"
"                                                                              \n"
"#define POW2(x) ((x)*(x))                                                     \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gfloat center[3];                                                           \n"
"  gfloat sum[3];                                                              \n"
"  glong  count;                                                               \n"
"} Cluster;                                                                    \n"
"                                                                              \n"
"static void                                                                   \n"
"downsample_buffer (GeglBuffer  *input,                                        \n"
"                   GeglBuffer **downsampled)                                  \n"
"{                                                                             \n"
"  gint  w, h;                                                                 \n"
"  glong n_pixels;                                                             \n"
"                                                                              \n"
"  w = gegl_buffer_get_width (input);                                          \n"
"  h = gegl_buffer_get_height (input);                                         \n"
"  n_pixels = w * h;                                                           \n"
"                                                                              \n"
"  if (n_pixels > MAX_PIXELS)                                                  \n"
"    {                                                                         \n"
"      GeglNode *node;                                                         \n"
"      GeglNode *source;                                                       \n"
"      GeglNode *scale;                                                        \n"
"      GeglNode *sink;                                                         \n"
"                                                                              \n"
"      gdouble factor = sqrt (MAX_PIXELS / (gdouble) n_pixels);                \n"
"                                                                              \n"
"      node = gegl_node_new ();                                                \n"
"                                                                              \n"
"      source = gegl_node_new_child (node, \"operation\", \"gegl:buffer-source\",\n"
"                                    \"buffer\", input,                        \n"
"                                    NULL);                                    \n"
"                                                                              \n"
"      scale = gegl_node_new_child (node, \"operation\", \"gegl:scale-ratio\", \n"
"                                   \"x\", factor,                             \n"
"                                   \"y\", factor,                             \n"
"                                   NULL);                                     \n"
"                                                                              \n"
"      sink = gegl_node_new_child (node, \"operation\", \"gegl:buffer-sink\",  \n"
"                                  \"buffer\", downsampled,                    \n"
"                                  NULL);                                      \n"
"                                                                              \n"
"      gegl_node_link_many (source, scale, sink, NULL);                        \n"
"      gegl_node_process (sink);                                               \n"
"      g_object_unref (node);                                                  \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      *downsampled = input;                                                   \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static inline gfloat                                                          \n"
"get_distance (gfloat *c1, gfloat *c2)                                         \n"
"{                                                                             \n"
"  return POW2(c2[0] - c1[0]) +                                                \n"
"         POW2(c2[1] - c1[1]) +                                                \n"
"         POW2(c2[2] - c1[2]);                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"static inline gint                                                            \n"
"find_nearest_cluster (gfloat  *pixel,                                         \n"
"                      Cluster *clusters,                                      \n"
"                      gint     n_clusters)                                    \n"
"{                                                                             \n"
"  gfloat min_distance = G_MAXFLOAT;                                           \n"
"  gint   min_cluster  = 0;                                                    \n"
"  gint   i;                                                                   \n"
"                                                                              \n"
"  for (i = 0; i < n_clusters; i++)                                            \n"
"    {                                                                         \n"
"      gfloat distance = get_distance (clusters[i].center, pixel);             \n"
"                                                                              \n"
"      if (distance < min_distance)                                            \n"
"        {                                                                     \n"
"          min_distance = distance;                                            \n"
"          min_cluster  = i;                                                   \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return min_cluster;                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static Cluster *                                                              \n"
"init_clusters (GeglBuffer     *input,                                         \n"
"               GeglProperties *o)                                             \n"
"{                                                                             \n"
"  Cluster *clusters = g_new0 (Cluster, o->n_clusters);                        \n"
"  GRand   *prg      = g_rand_new_with_seed (o->seed);                         \n"
"                                                                              \n"
"  gint width  = gegl_buffer_get_width (input);                                \n"
"  gint height = gegl_buffer_get_height (input);                               \n"
"  gint i;                                                                     \n"
"                                                                              \n"
"  for (i = 0; i < o->n_clusters; i++)                                         \n"
"    {                                                                         \n"
"      gfloat color[3];                                                        \n"
"      GeglRectangle one_pixel = {0, 0, 1, 1};                                 \n"
"      Cluster *c = clusters + i;                                              \n"
"                                                                              \n"
"      one_pixel.x = g_rand_int_range (prg, 0, width);                         \n"
"      one_pixel.y = g_rand_int_range (prg, 0, height);                        \n"
"                                                                              \n"
"      gegl_buffer_get (input, &one_pixel, 1.0, babl_format (\"CIE Lab float\"),\n"
"                       color, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);          \n"
"                                                                              \n"
"      c->center[0] = color[0];                                                \n"
"      c->center[1] = color[1];                                                \n"
"      c->center[2] = color[2];                                                \n"
"      c->sum[0] = 0.0f;                                                       \n"
"      c->sum[1] = 0.0f;                                                       \n"
"      c->sum[2] = 0.0f;                                                       \n"
"      c->count = 0;                                                           \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_rand_free (prg);                                                          \n"
"                                                                              \n"
"  return clusters;                                                            \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"assign_pixels_to_clusters (GeglBuffer *input,                                 \n"
"                           Cluster    *clusters,                              \n"
"                           gint        n_clusters)                            \n"
"{                                                                             \n"
"  GeglBufferIterator *iter;                                                   \n"
"                                                                              \n"
"  iter = gegl_buffer_iterator_new (input, NULL, 0, babl_format (\"CIE Lab float\"),\n"
"                                   GEGL_ACCESS_READ, GEGL_ABYSS_NONE);        \n"
"                                                                              \n"
"  while (gegl_buffer_iterator_next (iter))                                    \n"
"    {                                                                         \n"
"      gfloat *pixel = iter->data[0];                                          \n"
"      glong   n_pixels = iter->length;                                        \n"
"                                                                              \n"
"      while (n_pixels--)                                                      \n"
"        {                                                                     \n"
"          gint index = find_nearest_cluster (pixel, clusters, n_clusters);    \n"
"                                                                              \n"
"          clusters[index].sum[0] += pixel[0];                                 \n"
"          clusters[index].sum[1] += pixel[1];                                 \n"
"          clusters[index].sum[2] += pixel[2];                                 \n"
"          clusters[index].count++;                                            \n"
"                                                                              \n"
"          pixel += 3;                                                         \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"update_clusters (Cluster  *clusters,                                          \n"
"                 gint      n_clusters)                                        \n"
"{                                                                             \n"
"  gboolean has_changed = FALSE;                                               \n"
"  gint i;                                                                     \n"
"                                                                              \n"
"  for (i = 0; i < n_clusters; i++)                                            \n"
"    {                                                                         \n"
"      gfloat new_center[3];                                                   \n"
"                                                                              \n"
"      if (!clusters[i].count)                                                 \n"
"        continue;                                                             \n"
"                                                                              \n"
"      new_center[0] = clusters[i].sum[0] / clusters[i].count;                 \n"
"      new_center[1] = clusters[i].sum[1] / clusters[i].count;                 \n"
"      new_center[2] = clusters[i].sum[2] / clusters[i].count;                 \n"
"                                                                              \n"
"      if (new_center[0] != clusters[i].center[0] ||                           \n"
"          new_center[1] != clusters[i].center[1] ||                           \n"
"          new_center[2] != clusters[i].center[2])                             \n"
"        has_changed = TRUE;                                                   \n"
"                                                                              \n"
"      clusters[i].center[0] = new_center[0];                                  \n"
"      clusters[i].center[1] = new_center[1];                                  \n"
"      clusters[i].center[2] = new_center[2];                                  \n"
"      clusters[i].sum[0] = 0.0f;                                              \n"
"      clusters[i].sum[1] = 0.0f;                                              \n"
"      clusters[i].sum[2] = 0.0f;                                              \n"
"      clusters[i].count  = 0;                                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  return has_changed;                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"set_output (GeglBuffer *input,                                                \n"
"            GeglBuffer *output,                                               \n"
"            Cluster    *clusters,                                             \n"
"            gint        n_clusters)                                           \n"
"{                                                                             \n"
"  GeglBufferIterator *iter;                                                   \n"
"                                                                              \n"
"  iter = gegl_buffer_iterator_new (output, NULL, 0, babl_format (\"CIE Lab float\"),\n"
"                                   GEGL_ACCESS_WRITE, GEGL_ABYSS_NONE);       \n"
"                                                                              \n"
"  gegl_buffer_iterator_add (iter, input, NULL, 0, babl_format (\"CIE Lab float\"),\n"
"                            GEGL_ACCESS_READ, GEGL_ABYSS_NONE);               \n"
"                                                                              \n"
"  while (gegl_buffer_iterator_next (iter))                                    \n"
"    {                                                                         \n"
"      gfloat *out_pixel = iter->data[0];                                      \n"
"      gfloat *in_pixel  = iter->data[1];                                      \n"
"      glong   n_pixels = iter->length;                                        \n"
"                                                                              \n"
"      while (n_pixels--)                                                      \n"
"        {                                                                     \n"
"          gint index = find_nearest_cluster (in_pixel, clusters, n_clusters); \n"
"                                                                              \n"
"          out_pixel[0] = clusters[index].center[0];                           \n"
"          out_pixel[1] = clusters[index].center[1];                           \n"
"          out_pixel[2] = clusters[index].center[2];                           \n"
"                                                                              \n"
"          out_pixel += 3;                                                     \n"
"          in_pixel  += 3;                                                     \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  const Babl *format = babl_format (\"CIE Lab float\");                       \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"input\",  format);                  \n"
"  gegl_operation_set_format (operation, \"output\", format);                  \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_required_for_output (GeglOperation       *operation,                      \n"
"                         const gchar         *input_pad,                      \n"
"                         const GeglRectangle *roi)                            \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  /* Don't request an infinite plane */                                       \n"
"  if (gegl_rectangle_is_infinite_plane (&result))                             \n"
"    return *roi;                                                              \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_cached_region (GeglOperation       *operation,                            \n"
"                   const GeglRectangle *roi)                                  \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  if (gegl_rectangle_is_infinite_plane (&result))                             \n"
"    return *roi;                                                              \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *result,                                         \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (operation);                            \n"
"  gint            iterations = o->max_iterations;                             \n"
"  Cluster    *clusters;                                                       \n"
"  GeglBuffer *source;                                                         \n"
"                                                                              \n"
"  /* if pixels count of input buffer > MAX_PIXELS, compute a smaller buffer */\n"
"                                                                              \n"
"  downsample_buffer (input, &source);                                         \n"
"                                                                              \n"
"  /* clusters initialization */                                               \n"
"                                                                              \n"
"  clusters = init_clusters (source, o);                                       \n"
"                                                                              \n"
"  /* perform segmentation */                                                  \n"
"                                                                              \n"
"  while (iterations--)                                                        \n"
"    {                                                                         \n"
"      assign_pixels_to_clusters (source, clusters, o->n_clusters);            \n"
"                                                                              \n"
"      if (!update_clusters (clusters, o->n_clusters))                         \n"
"        break;                                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* apply cluster colors to output */                                        \n"
"                                                                              \n"
"  set_output (input, output, clusters, o->n_clusters);                        \n"
"                                                                              \n"
"  g_free (clusters);                                                          \n"
"                                                                              \n"
"  if (source != input)                                                        \n"
"    g_object_unref (source);                                                  \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"operation_process (GeglOperation        *operation,                           \n"
"                   GeglOperationContext *context,                             \n"
"                   const gchar          *output_prop,                         \n"
"                   const GeglRectangle  *result,                              \n"
"                   gint                  level)                               \n"
"{                                                                             \n"
"  GeglOperationClass  *operation_class;                                       \n"
"                                                                              \n"
"  const GeglRectangle *in_rect =                                              \n"
"    gegl_operation_source_get_bounding_box (operation, \"input\");            \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (gegl_op_parent_class);              \n"
"                                                                              \n"
"  if (in_rect && gegl_rectangle_is_infinite_plane (in_rect))                  \n"
"    {                                                                         \n"
"      gpointer in = gegl_operation_context_get_object (context, \"input\");   \n"
"      gegl_operation_context_take_object (context, \"output\",                \n"
"                                          g_object_ref (G_OBJECT (in)));      \n"
"      return TRUE;                                                            \n"
"    }                                                                         \n"
"                                                                              \n"
"  return operation_class->process (operation, context, output_prop, result,   \n"
"                                   gegl_operation_context_get_level (context));\n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  filter_class->process                    = process;                         \n"
"  operation_class->prepare                 = prepare;                         \n"
"  operation_class->process                 = operation_process;               \n"
"  operation_class->get_required_for_output = get_required_for_output;         \n"
"  operation_class->get_cached_region       = get_cached_region;               \n"
"  operation_class->opencl_support          = FALSE;                           \n"
"  operation_class->threaded                = FALSE;                           \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"      \"name\",        \"gegl:segment-kmeans\",                               \n"
"      \"title\",       _(\"K-means Segmentation\"),                           \n"
"      \"categories\",  \"color:segmentation\",                                \n"
"      \"description\", _(\"Segment colors using K-means clustering\"),        \n"
"      NULL);                                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
