static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * TMO:                                                                       \n"
" * Copyright 2010      Danny Robson      <danny@blubinc.net>                  \n"
" * (pfstmo)  2003-2004 Grzegorz Krawczyk <krawczyk@mpi-sb.mpg.de>             \n"
" *                                                                            \n"
" * PDE:                                                                       \n"
" * 2003-2004 Grzegorz Krawczyk  <krawczyk@mpi-sb.mpg.de>                      \n"
" *           Rafal Mantiuk      <mantiuk@mpi-sb.mpg.de>                       \n"
" * Some code from Numerical Recipes in C                                      \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"property_double (alpha, _(\"Alpha\"), 1.0)                                    \n"
"    description(_(\"Gradient threshold for detail enhancement\"))             \n"
"    value_range (0.0, 2.0)                                                    \n"
"                                                                              \n"
"property_double (beta, _(\"Beta\"), 0.9)                                      \n"
"    description(_(\"Strength of local detail enhancement\"))                  \n"
"    value_range (0.1, 2.0)                                                    \n"
"                                                                              \n"
"property_double (saturation, _(\"Saturation\"), 0.8)                          \n"
"    description (_(\"Global color saturation factor\"))                       \n"
"    value_range (0.0, 1.0)                                                    \n"
"                                                                              \n"
"property_double (noise, _(\"Noise\"), 0.0)                                    \n"
"    description (_(\"Gradient threshold for lowering detail enhancement\"))   \n"
"    value_range (0.0, 1.0)                                                    \n"
"                                                                              \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_NAME     fattal02                                             \n"
"#define GEGL_OP_C_SOURCE fattal02.c                                           \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include \"gegl-debug.h\"                                                     \n"
"#include <stdlib.h>                                                           \n"
"                                                                              \n"
"static const gchar *OUTPUT_FORMAT   = \"RGB float\";                          \n"
"static const gint   MINIMUM_PYRAMID = 32;                                     \n"
"                                                                              \n"
"/* I: pixel buffer, luminance with stride of 1                                \n"
" * R: rectangle, describes the buffer extent                                  \n"
" * X: width coordinate                                                        \n"
" * Y: height coordinate                                                       \n"
" */                                                                           \n"
"#define _P(I,R,X,Y) ((I)[(Y) * (R)->width + (X)])                             \n"
"                                                                              \n"
"/* The width/height of the pyramid at a level */                              \n"
"#define LEVEL_WIDTH(extent, level)  ((extent)->width  / (1 << (level)))       \n"
"#define LEVEL_HEIGHT(extent, level) ((extent)->height / (1 << (level)))       \n"
"#define LEVEL_EXTENT(extent, level)       \\                                  \n"
"    ((GeglRectangle){                     \\                                  \n"
"        0, 0,                             \\                                  \n"
"        LEVEL_WIDTH  ((extent), (level)), \\                                  \n"
"        LEVEL_HEIGHT ((extent), (level))  \\                                  \n"
"    })                                                                        \n"
"#define LEVEL_SIZE(extent, level) (LEVEL_EXTENT((extent), (level)).width * \\ \n"
"                                   LEVEL_EXTENT((extent), (level)).height)    \n"
"                                                                              \n"
"#define MODYF 0 /* 1 or 0 (1 is better) */                                    \n"
"#define MINS 16	/* minimum size 4 6 or 100 */                                 \n"
"                                                                              \n"
"/* #define MODYF_SQRT -1.0f *//* -1 or 0 */                                   \n"
"#define SMOOTH_IT 1 /* minimum 1  */                                          \n"
"#define BCG_STEPS 20                                                          \n"
"#define V_CYCLE 2 /* number of v-cycles  */                                   \n"
"                                                                              \n"
"/* precision */                                                               \n"
"#define EPS 1.0e-12                                                           \n"
"                                                                              \n"
"static void                                                                   \n"
"linbcg (guint   rows,                                                         \n"
"        guint   cols,                                                         \n"
"        gfloat  b[],                                                          \n"
"        gfloat  x[],                                                          \n"
"        gint    itol,                                                         \n"
"        gfloat  tol,                                                          \n"
"        gint    itmax,                                                        \n"
"        gint   *iter,                                                         \n"
"        gfloat *err);                                                         \n"
"                                                                              \n"
"                                                                              \n"
"/**                                                                           \n"
" * Set all elements of the array to a give value.                             \n"
" *                                                                            \n"
" * @param array array to modify                                               \n"
" * @param value all elements of the array will be set to this value           \n"
" */                                                                           \n"
"static inline void                                                            \n"
"fattal02_set_array (gfloat *array,                                            \n"
"                    guint   size,                                             \n"
"                    gfloat  value)                                            \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"  for (i = 0; i < size; ++i)                                                  \n"
"    array[i] = value;                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static inline void                                                            \n"
"fattal02_add_array (gfloat       *accum,                                      \n"
"                    guint         size,                                       \n"
"                    const gfloat *input)                                      \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"  for (i = 0; i < size; ++i)                                                  \n"
"    accum[i] += input[i];                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static inline void                                                            \n"
"fattal02_copy_array (const gfloat *input,                                     \n"
"                     gsize         size,                                      \n"
"                     gfloat       *output)                                    \n"
"{                                                                             \n"
"  memcpy (output, input, size * sizeof (input[0]));                           \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/*                                                                            \n"
" * Full Multigrid Algorithm for solving partial differential equations        \n"
" */                                                                           \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_restrict (const gfloat        *input,                                \n"
"                   const GeglRectangle *extent_i,                             \n"
"                   gfloat              *output,                               \n"
"                   const GeglRectangle *extent_o)                             \n"
"{                                                                             \n"
"  const guint inRows = extent_i->height,                                      \n"
"              inCols = extent_i->width;                                       \n"
"                                                                              \n"
"  const guint outRows = extent_o->height,                                     \n"
"              outCols = extent_o->width;                                      \n"
"                                                                              \n"
"  const gfloat dx = (gfloat)inCols / (gfloat)outCols,                         \n"
"               dy = (gfloat)inRows / (gfloat)outRows;                         \n"
"                                                                              \n"
"  const gfloat filterSize = 0.5;                                              \n"
"                                                                              \n"
"  gfloat sx, sy;                                                              \n"
"  guint   x,  y;                                                              \n"
"                                                                              \n"
"  for (y = 0, sy = dy / 2 - 0.5; y < outRows; ++y, sy += dy)                  \n"
"    {                                                                         \n"
"      for (x = 0, sx = dx / 2 - 0.5; x < outCols; ++x, sx += dx )             \n"
"        {                                                                     \n"
"          gfloat pixVal = 0;                                                  \n"
"          gfloat w      = 0;                                                  \n"
"          gint   ix, iy;                                                      \n"
"                                                                              \n"
"          for (ix  = MAX (0, ceilf (sx - dx * filterSize));                   \n"
"               ix <= MIN (floorf (sx + dx * filterSize), inCols - 1);         \n"
"               ++ix)                                                          \n"
"            {                                                                 \n"
"              for (iy  = MAX (0, ceilf (sy - dx * filterSize));               \n"
"                   iy <= MIN (floorf (sy + dx * filterSize), inRows - 1);     \n"
"                   ++iy)                                                      \n"
"                {                                                             \n"
"                  pixVal += input[ix + iy * inCols];                          \n"
"                  w      += 1;                                                \n"
"                }                                                             \n"
"            }                                                                 \n"
"                                                                              \n"
"          output[x + y * outCols] = pixVal / w;                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_prolongate (const gfloat        *input,                              \n"
"                     const GeglRectangle *extent_i,                           \n"
"                     gfloat              *output,                             \n"
"                     const GeglRectangle *extent_o)                           \n"
"{                                                                             \n"
"  gfloat dx = (gfloat)extent_i->width  / (gfloat)extent_o->width,             \n"
"         dy = (gfloat)extent_i->height / (gfloat)extent_o->height;            \n"
"                                                                              \n"
"  const guint outRows = extent_o->height,                                     \n"
"              outCols = extent_o->width;                                      \n"
"                                                                              \n"
"  const gfloat inRows = extent_i->height,                                     \n"
"               inCols = extent_i->width;                                      \n"
"                                                                              \n"
"  const float filterSize = 1;                                                 \n"
"                                                                              \n"
"  gfloat sx, sy;                                                              \n"
"  guint   x,  y;                                                              \n"
"                                                                              \n"
"  for (y = 0, sy = -dy / 2; y < outRows; ++y, sy += dy)                       \n"
"    {                                                                         \n"
"      for (x = 0, sx = -dx / 2; x < outCols; ++x, sx += dx )                  \n"
"        {                                                                     \n"
"          gfloat pixVal = 0;                                                  \n"
"          gfloat weight = 0;                                                  \n"
"          gfloat ix, iy;                                                      \n"
"                                                                              \n"
"          for (ix  = MAX (0, ceilf (sx - filterSize));                        \n"
"               ix <= MIN (floorf (sx + filterSize), inCols - 1);              \n"
"               ++ix)                                                          \n"
"            {                                                                 \n"
"              for (iy  = MAX (0, ceilf (sy - filterSize));                    \n"
"                   iy <= MIN (floorf (sy + filterSize), inRows - 1);          \n"
"                   ++iy)                                                      \n"
"                {                                                             \n"
"                  const gfloat fx   = fabs (sx - ix),                         \n"
"                               fy   = fabs (sy - iy),                         \n"
"                               fval = (1 - fx) * (1 - fy);                    \n"
"                                                                              \n"
"                  pixVal += input[(guint)ix + (guint)iy * (guint)inCols] * fval;\n"
"                  weight += fval;                                             \n"
"                }                                                             \n"
"            }                                                                 \n"
"                                                                              \n"
"          g_return_if_fail (weight != 0);                                     \n"
"                                                                              \n"
"          output [x + y * outCols] = pixVal / weight;                         \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_exact_solution (gfloat              *F,                              \n"
"                         const GeglRectangle *extent_f,                       \n"
"                         gfloat              *U,                              \n"
"                         const GeglRectangle *extent_u)                       \n"
"{                                                                             \n"
"  /* pfstmo suggests that successive over-relaxation should be used here,     \n"
"   * followed by scaling by the square of the inverse of the sqrt of the array\n"
"   * length. However it was commented out due to 'incorrect results', and the \n"
"   * array zeroing was used in its place.                                     \n"
"   */                                                                         \n"
"  fattal02_set_array (U, extent_u->width * extent_u->height, 0.0f);           \n"
"  return;                                                                     \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* smooth u using f at level */                                               \n"
"static void                                                                   \n"
"fattal02_smooth (gfloat              *U,                                      \n"
"                 const GeglRectangle *extent_u,                               \n"
"                 gfloat              *F,                                      \n"
"                 const GeglRectangle *extent_f)                               \n"
"{                                                                             \n"
"  gint   iter;                                                                \n"
"  gfloat err;                                                                 \n"
"                                                                              \n"
"  linbcg (extent_u->height,                                                   \n"
"          extent_u->width,                                                    \n"
"          F, U, 1, 0.001,                                                     \n"
"          BCG_STEPS, &iter, &err);                                            \n"
"                                                                              \n"
"  /* pfstmo notes here that 'gauss relaxation is too slow'. */                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_calculate_defect (gfloat              *D,                            \n"
"                           const GeglRectangle *extent_d,                     \n"
"                           gfloat              *U,                            \n"
"                           const GeglRectangle *extent_u,                     \n"
"                           gfloat              *F,                            \n"
"                           const GeglRectangle *extent_f)                     \n"
"{                                                                             \n"
"  guint sx = extent_f->width,                                                 \n"
"        sy = extent_f->height;                                                \n"
"  guint x, y;                                                                 \n"
"                                                                              \n"
"  for (y = 0; y < sy; ++y)                                                    \n"
"    {                                                                         \n"
"      for (x = 0; x < sx; ++x)                                                \n"
"        {                                                                     \n"
"          guint w = (x     ==  0 ? 0 : x - 1),                                \n"
"                n = (y     ==  0 ? 0 : y - 1),                                \n"
"                s = (y + 1 == sy ? y : y + 1),                                \n"
"                e = (x + 1 == sx ? x : x + 1);                                \n"
"                                                                              \n"
"          _P (D, extent_d, x, y) = _P (F, extent_f, x, y) - (                 \n"
"                                      _P (U, extent_u, e, y) +                \n"
"                                      _P (U, extent_u, w, y) +                \n"
"                                      _P (U, extent_u, x, n) +                \n"
"                                      _P (U, extent_u, x, s) -                \n"
"                                      4.0 * _P (U, extent_u, x, y)            \n"
"                                  );                                          \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_solve_pde_multigrid (gfloat              *F,                         \n"
"                              const GeglRectangle *extent_f,                  \n"
"                              gfloat              *U,                         \n"
"                              const GeglRectangle *extent_u)                  \n"
"{                                                                             \n"
"  guint xmax = extent_f->width,                                               \n"
"        ymax = extent_f->height;                                              \n"
"                                                                              \n"
"  gint i,	/* index for simple loops */                                        \n"
"       k,	/* index for iterating through levels */                            \n"
"       k2;	/* index for iterating through levels in V-cycles */               \n"
"                                                                              \n"
"  gint levels;                                                                \n"
"                                                                              \n"
"  gfloat **RHS, /* given function f restricted on levels */                   \n"
"         **IU,  /* approximate initial sollutions on levels */                \n"
"         **VF;  /* target functions in cycles (approximate sollution error (uh - ~uh) ) */\n"
"                                                                              \n"
"  /* 1. restrict f to coarse-grid (by the way count the number of levels)     \n"
"   *	  k=0: fine-grid = f                                                     \n"
"   *	  k=levels: coarsest-grid                                                \n"
"   */                                                                         \n"
"  {                                                                           \n"
"    guint mins = MIN (xmax, ymax);                                            \n"
"    levels = 0;                                                               \n"
"                                                                              \n"
"    while (mins >= MINS)                                                      \n"
"      {                                                                       \n"
"        levels++;                                                             \n"
"        mins = mins / 2 + MODYF;                                              \n"
"      }                                                                       \n"
"  }                                                                           \n"
"                                                                              \n"
"  RHS = g_new (gfloat*, levels + 1);                                          \n"
"   IU = g_new (gfloat*, levels + 1);                                          \n"
"   VF = g_new (gfloat*, levels + 1);                                          \n"
"                                                                              \n"
"  RHS[0] = F;                                                                 \n"
"   VF[0] = g_new (gfloat, xmax * ymax);                                       \n"
"   IU[0] = g_new (gfloat, xmax * ymax);                                       \n"
"  fattal02_copy_array (U, xmax * ymax, IU[0]);                                \n"
"                                                                              \n"
"  for (k = 0; k < levels; ++k)                                                \n"
"    {                                                                         \n"
"      RHS[k + 1] = g_new (gfloat, LEVEL_SIZE (extent_f, k + 1));              \n"
"       IU[k + 1] = g_new (gfloat, LEVEL_SIZE (extent_f, k + 1));              \n"
"       VF[k + 1] = g_new (gfloat, LEVEL_SIZE (extent_f, k + 1));              \n"
"                                                                              \n"
"      /* restrict from level k to level k+1 (coarser-grid) */                 \n"
"      fattal02_restrict (RHS[k    ], &LEVEL_EXTENT (extent_f, k     ),        \n"
"                         RHS[k + 1], &LEVEL_EXTENT (extent_f, k + 1));        \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* 2. find exact solution at the coarsest-grid (k=levels) */                \n"
"  fattal02_exact_solution (RHS[levels], &LEVEL_EXTENT (extent_f, levels),     \n"
"                            IU[levels], &LEVEL_EXTENT (extent_f, levels));    \n"
"                                                                              \n"
"  /* 3. nested iterations */                                                  \n"
"  for (k = levels - 1; k >= 0; --k)                                           \n"
"    {                                                                         \n"
"      guint cycle;                                                            \n"
"                                                                              \n"
"      /* 4. interpolate sollution from last coarse-grid to finer-grid         \n"
"       * interpolate from level k+1 to level k (finer-grid)                   \n"
"       */                                                                     \n"
"      fattal02_prolongate (IU[k + 1], &LEVEL_EXTENT (extent_f, k + 1),        \n"
"                           IU[k    ], &LEVEL_EXTENT (extent_f, k    ));       \n"
"                                                                              \n"
"      /* 4.1. first target function is the equation target function           \n"
"       *      (following target functions are the defect)                     \n"
"       */                                                                     \n"
"      fattal02_copy_array (RHS[k], LEVEL_SIZE (extent_f, k), VF[k]);          \n"
"                                                                              \n"
"      /* 5. V-cycle (twice repeated) */                                       \n"
"      for (cycle = 0; cycle < V_CYCLE; ++cycle)                               \n"
"        {                                                                     \n"
"          /* 6. downward stroke of V */                                       \n"
"          for (k2 = k; k2 < levels; ++k2)                                     \n"
"            {                                                                 \n"
"              gfloat *D;                                                      \n"
"                                                                              \n"
"              /* 7. pre-smoothing of initial sollution using target function  \n"
"               *    zero for initial guess at smoothing                       \n"
"               *    (except for level k when iu contains prolongated result)  \n"
"               */                                                             \n"
"              if (k2 != k)                                                    \n"
"                {                                                             \n"
"                  fattal02_set_array (IU[k2], LEVEL_SIZE (extent_f, k2), 0.0f);\n"
"                }                                                             \n"
"                                                                              \n"
"              for (i = 0; i < SMOOTH_IT; ++i)                                 \n"
"                {                                                             \n"
"                  fattal02_smooth (IU[k2], &LEVEL_EXTENT (extent_f, k2),      \n"
"                                   VF[k2], &LEVEL_EXTENT (extent_f, k2));     \n"
"                }                                                             \n"
"                                                                              \n"
"              /* 8. calculate defect at level                                 \n"
"               *    d[k2] = Lh * ~u[k2] - f[k2]                               \n"
"               */                                                             \n"
"              D = g_new (gfloat, LEVEL_SIZE (extent_f, k2));                  \n"
"              fattal02_calculate_defect (     D, &LEVEL_EXTENT (extent_f, k2),\n"
"                                         IU[k2], &LEVEL_EXTENT (extent_f, k2),\n"
"                                         VF[k2], &LEVEL_EXTENT (extent_f, k2));\n"
"                                                                              \n"
"              /* 9. restrict deffect as target function for next coarser-grid \n"
"               *    def -> f[k2+1]                                            \n"
"               */                                                             \n"
"              fattal02_restrict (         D, &LEVEL_EXTENT (extent_f, k2    ),\n"
"                                 VF[k2 + 1], &LEVEL_EXTENT (extent_f, k2 + 1));\n"
"              g_free (D);                                                     \n"
"            }                                                                 \n"
"                                                                              \n"
"          /* 10. solve on coarsest-grid (target function is the deffect)      \n"
"           *     iu[levels] should contain sollution for                      \n"
"           *     the f[levels] - last deffect, iu will now be the correction  \n"
"           */                                                                 \n"
"          fattal02_exact_solution (VF[levels], &LEVEL_EXTENT (extent_f, levels),\n"
"                                   IU[levels], &LEVEL_EXTENT (extent_f, levels));\n"
"                                                                              \n"
"          /* 11. upward stroke of V */                                        \n"
"          for (k2 = levels - 1; k2 >= k; --k2)                                \n"
"            {                                                                 \n"
"              /* 12. interpolate correction from last coarser-grid to finer-grid\n"
"               *     iu[k2+1] -> cor                                          \n"
"               */                                                             \n"
"              gfloat *C = g_new (gfloat, LEVEL_SIZE (extent_f, k2));          \n"
"              fattal02_prolongate (IU[k2 + 1], &LEVEL_EXTENT (extent_f, k2 + 1),\n"
"                                            C, &LEVEL_EXTENT (extent_f, k2    ));\n"
"                                                                              \n"
"              /* 13. add interpolated correction to initial sollution at level k2 */\n"
"              fattal02_add_array (IU[k2], LEVEL_SIZE (extent_f, k2), C);      \n"
"              g_free (C);                                                     \n"
"                                                                              \n"
"              /* 14. post-smoothing of current sollution using target function */\n"
"              for (i = 0; i < SMOOTH_IT; ++i)                                 \n"
"                  fattal02_smooth (IU[k2], &LEVEL_EXTENT (extent_f, k2),      \n"
"                                   VF[k2], &LEVEL_EXTENT (extent_f, k2));     \n"
"            }                                                                 \n"
"                                                                              \n"
"        } /*--- end of V-cycle */                                             \n"
"                                                                              \n"
"    } /*--- end of nested iteration */                                        \n"
"                                                                              \n"
"  /* 15. final sollution                                                      \n"
"   *     IU[0] contains the final sollution                                   \n"
"   */                                                                         \n"
"                                                                              \n"
"  fattal02_copy_array (IU[0], extent_f->width * extent_f->height, U);         \n"
"                                                                              \n"
"  g_free (VF[0]);                                                             \n"
"  g_free (IU[0]);                                                             \n"
"                                                                              \n"
"  for (k = 1; k <= levels; ++k)                                               \n"
"    {                                                                         \n"
"      g_free (RHS[k]);                                                        \n"
"      g_free ( IU[k]);                                                        \n"
"      g_free ( VF[k]);                                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (RHS);                                                               \n"
"  g_free ( IU);                                                               \n"
"  g_free ( VF);                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"asolve (gulong n,                                                             \n"
"        gfloat b[],                                                           \n"
"        gfloat x[],                                                           \n"
"        gint   itrnsp)                                                        \n"
"{                                                                             \n"
"  guint i;                                                                    \n"
"                                                                              \n"
"  for (i = 0; i < n; ++i)                                                     \n"
"    x[i] = -4 * b[i];                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"atimes (guint  rows,                                                          \n"
"        guint  cols,                                                          \n"
"        gfloat x[],                                                           \n"
"        gfloat res[],                                                         \n"
"        gint   itrnsp)                                                        \n"
"{                                                                             \n"
"  guint r, c;                                                                 \n"
"                                                                              \n"
"#define IDX(R,C) ((R) * cols + (C))                                           \n"
"                                                                              \n"
"  for (r = 1; r < rows - 1; ++r)                                              \n"
"    {                                                                         \n"
"      for (c = 1; c < cols - 1; ++c)                                          \n"
"        {                                                                     \n"
"          res[IDX (r,c)] = x[IDX (r-1,c)] + x[IDX (r+1,c)] +                  \n"
"            x[IDX (r,c-1)] + x[IDX (r,c+1)] - 4*x[IDX (r,c)];                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (r = 1; r < rows - 1; ++r)                                              \n"
"    {                                                                         \n"
"      res[IDX (r, 0)] =     x[IDX (r - 1, 0)] +                               \n"
"                            x[IDX (r + 1, 0)] +                               \n"
"                            x[IDX (r    , 1)] -                               \n"
"                        3 * x[IDX (r    , 0)];                                \n"
"                                                                              \n"
"      res[IDX (r, cols - 1)] =     x[IDX (r - 1, cols - 1)] +                 \n"
"                                   x[IDX (r + 1, cols - 1)] +                 \n"
"                                   x[IDX (r    , cols - 2)] -                 \n"
"                               3 * x[IDX (r    , cols - 1)];                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (c = 1; c < cols - 1; ++c)                                              \n"
"    {                                                                         \n"
"      res[IDX (0, c)] =     x[IDX (1, c    )] +                               \n"
"                            x[IDX (0, c - 1)] +                               \n"
"                            x[IDX (0, c + 1)] -                               \n"
"                        3 * x[IDX (0, c     )];                               \n"
"                                                                              \n"
"      res[IDX (rows - 1, c)] =     x[IDX (rows - 2, c    )] +                 \n"
"                                   x[IDX (rows - 1, c - 1)] +                 \n"
"                                   x[IDX (rows - 1, c + 1)] -                 \n"
"                               3 * x[IDX (rows - 1, c    )];                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  res[IDX (0       ,        0)] =     x[IDX (1       ,        0)] +           \n"
"                                      x[IDX (0       ,        1)] -           \n"
"                                  2 * x[IDX (0       ,        0)];            \n"
"  res[IDX (rows - 1,        0)] =     x[IDX (rows - 2,        0)] +           \n"
"                                      x[IDX (rows - 1,        1)] -           \n"
"                                  2 * x[IDX (rows - 1,        0)];            \n"
"  res[IDX (0       , cols - 1)] =     x[IDX (1       , cols - 1)] +           \n"
"                                      x[IDX (0       , cols - 2)] -           \n"
"                                  2 * x[IDX (0       , cols - 1)];            \n"
"  res[IDX (rows - 1, cols - 1)] =     x[IDX (rows - 2, cols - 1)] +           \n"
"                                      x[IDX (rows - 1, cols - 2)] -           \n"
"                                  2 * x[IDX (rows - 1, cols - 1)];            \n"
"}                                                                             \n"
"                                                                              \n"
"static gfloat                                                                 \n"
"snrm (gulong n,                                                               \n"
"      gfloat sx[],                                                            \n"
"      gint   itol)                                                            \n"
"{                                                                             \n"
"  gulong i;                                                                   \n"
"                                                                              \n"
"  if (itol <= 3)                                                              \n"
"    {                                                                         \n"
"      gfloat ans = 0.0;                                                       \n"
"      for (i = 0; i < n; ++i)                                                 \n"
"          ans += sx[i] * sx[i];                                               \n"
"      return sqrtf (ans);                                                     \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      gulong isamax = 0;                                                      \n"
"      for (i = 0; i < n; ++i)                                                 \n"
"        if (fabs (sx[i]) > fabs (sx[isamax]))                                 \n"
"            isamax = i;                                                       \n"
"      return fabs (sx[isamax]);                                               \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/**                                                                           \n"
" * Biconjugate Gradient Method                                                \n"
" * from Numerical Recipes in C                                                \n"
" */                                                                           \n"
"static void                                                                   \n"
"linbcg (guint   rows,                                                         \n"
"        guint   cols,                                                         \n"
"        gfloat  b[],                                                          \n"
"        gfloat  x[],                                                          \n"
"        gint    itol,                                                         \n"
"        gfloat  tol,                                                          \n"
"        gint    itmax,                                                        \n"
"        gint   *iter,                                                         \n"
"        gfloat *err)                                                          \n"
"{                                                                             \n"
"  guint  n = rows * cols;                                                     \n"
"                                                                              \n"
"  gulong j;                                                                   \n"
"  gfloat ak,akden,bk,bkden,bknum,bnrm,dxnrm,xnrm,zm1nrm,znrm;                 \n"
"  gfloat *p,*pp,*r,*rr,*z,*zz;                                                \n"
"                                                                              \n"
"  /* To remove warning about potetial uninitialized use */                    \n"
"  bkden = 1;                                                                  \n"
"                                                                              \n"
"  p  = g_new (gfloat, n);                                                     \n"
"  pp = g_new (gfloat, n);                                                     \n"
"  r  = g_new (gfloat, n);                                                     \n"
"  rr = g_new (gfloat, n);                                                     \n"
"  z  = g_new (gfloat, n);                                                     \n"
"  zz = g_new (gfloat, n);                                                     \n"
"                                                                              \n"
"  *iter=0;                                                                    \n"
"  atimes (rows, cols, x, r, 0);                                               \n"
"  for (j = 0; j < n; ++j)                                                     \n"
"    {                                                                         \n"
"       r[j] = b[j] - r[j];                                                    \n"
"      rr[j] = r[j];                                                           \n"
"    }                                                                         \n"
"                                                                              \n"
"  atimes (rows, cols, r, rr, 0);       /* minimum residual */                 \n"
"  znrm = 1.0;                                                                 \n"
"                                                                              \n"
"  if (itol == 1)                                                              \n"
"    {                                                                         \n"
"      bnrm = snrm (n, b, itol);                                               \n"
"    }                                                                         \n"
"  else if (itol == 2)                                                         \n"
"    {                                                                         \n"
"      asolve (n, b, z, 0);                                                    \n"
"      bnrm = snrm (n, z, itol);                                               \n"
"    }                                                                         \n"
"  else if (itol == 3 || itol == 4)                                            \n"
"    {                                                                         \n"
"      asolve (n, b, z, 0);                                                    \n"
"      bnrm = snrm (n, z, itol);                                               \n"
"      asolve (n, r, z, 0);                                                    \n"
"      znrm = snrm (n, z, itol);                                               \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      g_warning (\"illegal itol in linbcg\");                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  asolve (n, r, z, 0);                                                        \n"
"                                                                              \n"
"  while (*iter <= itmax)                                                      \n"
"    {                                                                         \n"
"      ++(*iter);                                                              \n"
"                                                                              \n"
"      zm1nrm = znrm;                                                          \n"
"      asolve (n, rr, zz, 1);                                                  \n"
"      for (bknum = 0.0, j = 0; j < n; ++j)                                    \n"
"        {                                                                     \n"
"          bknum += z[j] * rr[j];                                              \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (*iter == 1)                                                         \n"
"        {                                                                     \n"
"          for (j = 0; j < n; ++j)                                             \n"
"            {                                                                 \n"
"               p[j] =  z[j];                                                  \n"
"              pp[j] = zz[j];                                                  \n"
"            }                                                                 \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          bk = bknum / bkden;                                                 \n"
"                                                                              \n"
"          for (j = 0; j < n; ++j)                                             \n"
"            {                                                                 \n"
"               p[j] = bk *  p[j] +  z[j];                                     \n"
"              pp[j] = bk * pp[j] + zz[j];                                     \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      bkden = bknum;                                                          \n"
"      atimes (rows, cols, p, z, 0);                                           \n"
"                                                                              \n"
"      for (akden = 0.0, j = 0; j < n; ++j)                                    \n"
"        {                                                                     \n"
"          akden += z[j] * pp[j];                                              \n"
"        }                                                                     \n"
"                                                                              \n"
"      ak = bknum / akden;                                                     \n"
"      atimes (rows, cols, pp, zz, 1);                                         \n"
"                                                                              \n"
"      for (j = 0; j < n; ++j)                                                 \n"
"        {                                                                     \n"
"           x[j] += ak *  p[j];                                                \n"
"           r[j] -= ak *  z[j];                                                \n"
"          rr[j] -= ak * zz[j];                                                \n"
"        }                                                                     \n"
"                                                                              \n"
"      asolve (n, r, z, 0);                                                    \n"
"                                                                              \n"
"      if (itol == 1 || itol == 2)                                             \n"
"        {                                                                     \n"
"          znrm = 1.0;                                                         \n"
"          *err = snrm (n, r, itol) / bnrm;                                    \n"
"        }                                                                     \n"
"      else if (itol == 3 || itol == 4)                                        \n"
"        {                                                                     \n"
"          znrm = snrm (n, z, itol);                                           \n"
"                                                                              \n"
"          if (fabs (zm1nrm - znrm) > EPS * znrm)                              \n"
"            {                                                                 \n"
"              dxnrm = fabs (ak) * snrm (n, p, itol);                          \n"
"              *err = znrm / fabs (zm1nrm - znrm) * dxnrm;                     \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              *err = znrm / bnrm;                                             \n"
"              continue;                                                       \n"
"            }                                                                 \n"
"                                                                              \n"
"          xnrm = snrm (n, x, itol);                                           \n"
"          if (*err <= 0.5 * xnrm)                                             \n"
"            {                                                                 \n"
"              *err /= xnrm;                                                   \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              *err=znrm/bnrm;                                                 \n"
"              continue;                                                       \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (*err <= tol)                                                        \n"
"        break;                                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (p);                                                                 \n"
"  g_free (pp);                                                                \n"
"  g_free (r);                                                                 \n"
"  g_free (rr);                                                                \n"
"  g_free (z);                                                                 \n"
"  g_free (zz);                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Downscale the input buffer by a factor of two. Extent describes the input  \n"
" * buffer. Assumes a pixel stride of 1, as we're really only dealing with     \n"
" * luminance. Output should be preallocated with a size that is half of the   \n"
" * input.                                                                     \n"
" */                                                                           \n"
"static void                                                                   \n"
"fattal02_downsample (const gfloat        *input,                              \n"
"                     const GeglRectangle *extent,                             \n"
"                     gfloat              *output)                             \n"
"{                                                                             \n"
"  guint x, y, width, height;                                                  \n"
"  g_return_if_fail (input);                                                   \n"
"  g_return_if_fail (extent);                                                  \n"
"  g_return_if_fail (output);                                                  \n"
"                                                                              \n"
"  width  = extent->width  / 2,                                                \n"
"  height = extent->height / 2;                                                \n"
"                                                                              \n"
"  g_return_if_fail (width  > 0);                                              \n"
"  g_return_if_fail (height > 0);                                              \n"
"                                                                              \n"
"  for (y = 0; y < height; ++y)                                                \n"
"    {                                                                         \n"
"      for (x = 0; x < width; ++x)                                             \n"
"        {                                                                     \n"
"          gfloat p = 0.0f;                                                    \n"
"                                                                              \n"
"          /* Sum the 4 pixels from the input which directly contribute to the \n"
"           * output, and divide by four.                                      \n"
"           */                                                                 \n"
"          p += input[(2 * y + 0) * extent->width + (2 * x + 0)];              \n"
"          p += input[(2 * y + 0) * extent->width + (2 * x + 1)];              \n"
"          p += input[(2 * y + 1) * extent->width + (2 * x + 0)];              \n"
"          p += input[(2 * y + 1) * extent->width + (2 * x + 1)];              \n"
"                                                                              \n"
"          output [y * width + x] = p / 4.0f;                                  \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/* Blur the input buffer with a one pixel radius. Output should be            \n"
" * preallocated with the same size as the input buffer. This must perform     \n"
" * correctly when input and output alias.                                     \n"
" */                                                                           \n"
"static void                                                                   \n"
"fattal02_gaussian_blur (const gfloat        *input,                           \n"
"                        const GeglRectangle *extent,                          \n"
"                        gfloat              *output)                          \n"
"{                                                                             \n"
"  const guint  width  = extent->width,                                        \n"
"               height = extent->height,                                       \n"
"               size   = width * height;                                       \n"
"  guint        x, y;                                                          \n"
"  gfloat      *temp;                                                          \n"
"                                                                              \n"
"  g_return_if_fail (input);                                                   \n"
"  g_return_if_fail (extent);                                                  \n"
"  g_return_if_fail (output);                                                  \n"
"  g_return_if_fail (size > 0);                                                \n"
"                                                                              \n"
"  temp   = g_new (gfloat, size);                                              \n"
"                                                                              \n"
"  /* horizontal blur */                                                       \n"
"  for (y = 0; y < height; ++y)                                                \n"
"    {                                                                         \n"
"      for (x = 1; x < width - 1; ++x)                                         \n"
"        {                                                                     \n"
"          gfloat p  = 2 * input[x     + y * width];                           \n"
"          p        +=     input[x - 1 + y * width];                           \n"
"          p        +=     input[x + 1 + y * width];                           \n"
"                                                                              \n"
"          temp[x + y * extent->width] = p / 4.0f;                             \n"
"        }                                                                     \n"
"                                                                              \n"
"      temp[0         + y * width] = (3 * input[0         + y * width] +       \n"
"                                         input[1         + y * width]) / 4.0f;\n"
"      temp[width - 1 + y * width] = (3 * input[width - 1 + y * width] +       \n"
"                                         input[width - 2 + y * width]) / 4.0f;\n"
"    }                                                                         \n"
"                                                                              \n"
"  /* vertical blur */                                                         \n"
"  for (x = 0; x < width; ++x)                                                 \n"
"    {                                                                         \n"
"      for (y = 1; y < height - 1; ++y)                                        \n"
"        {                                                                     \n"
"          gfloat p  = 2 * temp[x +      y  * width];                          \n"
"          p        +=     temp[x + (y - 1) * width];                          \n"
"          p        +=     temp[x + (y + 1) * width];                          \n"
"                                                                              \n"
"          output[x + y * width] = p / 4.0f;                                   \n"
"        }                                                                     \n"
"                                                                              \n"
"      output[x +           0  * width] = (3 * temp[x +           0  * width] +\n"
"                                              temp[x +           1  * width]) / 4.0f;\n"
"      output[x + (height - 1) * width] = (3 * temp[x + (height - 1) * width] +\n"
"                                              temp[x + (height - 2) * width]) / 4.0f;\n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (temp);                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_create_gaussian_pyramids (const gfloat         *zero,                \n"
"                                   const GeglRectangle  *extent,              \n"
"                                   gfloat              **pyramid,             \n"
"                                   gint                  levels)              \n"
"{                                                                             \n"
"  gint           i;                                                           \n"
"  gfloat        *blur;                                                        \n"
"  GeglRectangle  level_extent = *extent;                                      \n"
"                                                                              \n"
"  /* Copy the first level of the pyramid into place */                        \n"
"  pyramid[0] = g_new (gfloat, level_extent.width * level_extent.height);      \n"
"  for (i = 0; i < level_extent.width * level_extent.height; ++i)              \n"
"    {                                                                         \n"
"      pyramid[0][i] = zero[i];                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Establish a temporary blur buffer. The allocated memory will be used for \n"
"   * progressively smaller levels, and we don't free this until the end.      \n"
"   */                                                                         \n"
"  blur = g_new (gfloat, level_extent.width * level_extent.height);            \n"
"  fattal02_gaussian_blur (pyramid[0], &level_extent, blur);                   \n"
"                                                                              \n"
"  for (i = 1; i < levels; ++i)                                                \n"
"    {                                                                         \n"
"      level_extent.width  /= 2;                                               \n"
"      level_extent.height /= 2;                                               \n"
"                                                                              \n"
"      g_return_if_fail (level_extent.width  >= MINIMUM_PYRAMID);              \n"
"      g_return_if_fail (level_extent.height >= MINIMUM_PYRAMID);              \n"
"                                                                              \n"
"      /* Downsample the blurred buffer into the pyramid */                    \n"
"      pyramid[i] = g_new (gfloat, LEVEL_SIZE (extent, i));                    \n"
"      fattal02_downsample (blur, &LEVEL_EXTENT (extent, i - 1), pyramid[i]);  \n"
"                                                                              \n"
"      /* Blur the current level over the blur buffer */                       \n"
"      fattal02_gaussian_blur (pyramid[i], &level_extent, blur);               \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (blur);                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/********************************************************************/        \n"
"                                                                              \n"
"static gfloat                                                                 \n"
"fattal02_calculate_gradients (const gfloat        *input,   /* H */           \n"
"                              const GeglRectangle *extent,  /*  */            \n"
"                              gfloat              *output,  /* G */           \n"
"                              gint                 k)                         \n"
"{                                                                             \n"
"  guint  width   = extent->width,                                             \n"
"         height  = extent->height;                                            \n"
"  gfloat divider = powf (2.0f, k + 1),                                        \n"
"         average = 0.0f;                                                      \n"
"                                                                              \n"
"  guint  x, y;                                                                \n"
"                                                                              \n"
"  for (y = 0; y < height; ++y)                                                \n"
"    {                                                                         \n"
"      for (x = 0; x < width; ++x)                                             \n"
"        {                                                                     \n"
"          gfloat gx, gy;                                                      \n"
"          gint   w, n, e, s;                                                  \n"
"                                                                              \n"
"          w = (x     ==      0 ? 0 : x - 1);                                  \n"
"          n = (y     ==      0 ? 0 : y - 1);                                  \n"
"          s = (y + 1 == height ? y : y + 1);                                  \n"
"          e = (x + 1 == width  ? x : x + 1);                                  \n"
"                                                                              \n"
"          gx = (input[w + y * width] - input[e + y * width]) / divider;       \n"
"          gy = (input[x + s * width] - input[x + n * width]) / divider;       \n"
"                                                                              \n"
"          output[x + y * width] = sqrtf (gx * gx + gy * gy);                  \n"
"          average += output[x + y * width];                                   \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return average / (width * height);                                          \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/********************************************************************/        \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_upsample (const gfloat        *input,                                \n"
"                   const GeglRectangle *extent,                               \n"
"                   gfloat              *output)                               \n"
"{                                                                             \n"
"  guint  width_i = extent->width,                                             \n"
"        height_i = extent->height,                                            \n"
"         width_o =  width_i * 2,                                              \n"
"        height_o = height_i * 2;                                              \n"
"  guint x_o, y_o;                                                             \n"
"                                                                              \n"
"  for (y_o = 0; y_o < height_o; ++y_o)                                        \n"
"    {                                                                         \n"
"      for (x_o = 0; x_o < width_o; ++x_o)                                     \n"
"        {                                                                     \n"
"          guint x_i = x_o / 2,                                                \n"
"            y_i = y_o / 2;                                                    \n"
"                                                                              \n"
"          x_i = (x_i <  width_i) ? x_i :  width_i - 1;                        \n"
"          y_i = (y_i < height_i) ? y_i : height_i - 1;                        \n"
"                                                                              \n"
"          output[x_o + y_o * width_o] = input[x_i + y_i * width_i];           \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_FI_matrix (gfloat               *FI,                                 \n"
"                    const GeglRectangle  *extent,                             \n"
"                    gfloat              **gradients,                          \n"
"                    const gfloat         *averages,                           \n"
"                    const gint            levels,                             \n"
"                    const gfloat          alfa,                               \n"
"                    const gfloat          beta,                               \n"
"                    const gfloat          noise)                              \n"
"{                                                                             \n"
"  GeglRectangle   level_extent = *extent;                                     \n"
"  gint            i;                                                          \n"
"  gfloat        **fi;                                                         \n"
"                                                                              \n"
"  level_extent.width  = LEVEL_WIDTH  (extent, levels - 1);                    \n"
"  level_extent.height = LEVEL_HEIGHT (extent, levels - 1);                    \n"
"                                                                              \n"
"  fi = g_new (gfloat*, levels);                                               \n"
"                                                                              \n"
"  fi[levels - 1] = g_new (gfloat, level_extent.width * level_extent.height);  \n"
"                                                                              \n"
"  for (i = 0; i < level_extent.width * level_extent.height; ++i)              \n"
"    {                                                                         \n"
"      fi[levels - 1][i] = 1.0f;                                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (i = levels - 1; i >= 0; --i)                                           \n"
"    {                                                                         \n"
"      gint x, y;                                                              \n"
"                                                                              \n"
"      level_extent.width  = LEVEL_WIDTH  (extent, i);                         \n"
"      level_extent.height = LEVEL_HEIGHT (extent, i);                         \n"
"                                                                              \n"
"      for (y = 0; y < level_extent.height; ++y)                               \n"
"          for (x = 0; x < level_extent.width; ++x)                            \n"
"            {                                                                 \n"
"              gfloat grad  = gradients[i][x + y * level_extent.width],        \n"
"                     a     = alfa * averages[i],                              \n"
"                     value = 1.0f;                                            \n"
"                                                                              \n"
"              if (grad > 1e-4f)                                               \n"
"                value = a / (grad + noise) * powf ((grad + noise) / a, beta); \n"
"              fi[i][x + y * level_extent.width] *= value;                     \n"
"            }                                                                 \n"
"                                                                              \n"
"      /* create next level */                                                 \n"
"      if (i > 1)                                                              \n"
"        {                                                                     \n"
"          level_extent.width  = LEVEL_WIDTH  (extent, i - 1);                 \n"
"          level_extent.height = LEVEL_HEIGHT (extent, i - 1);                 \n"
"                                                                              \n"
"          fi[i - 1] = g_new (gfloat,                                          \n"
"                             level_extent.width * level_extent.height);       \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          fi[0] = FI;               /* highest level -> result */             \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (i > 0)                                                              \n"
"        {                                                                     \n"
"          /* upsample to next level */                                        \n"
"          fattal02_upsample      (fi[i    ], &LEVEL_EXTENT (extent, i    ), fi[i - 1]);\n"
"          fattal02_gaussian_blur (fi[i - 1], &LEVEL_EXTENT (extent, i - 1), fi[i - 1]);\n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Careful not to delete the result memory in fi[0] */                      \n"
"  for (i = 1; i < levels; ++i)                                                \n"
"    g_free (fi[i]);                                                           \n"
"                                                                              \n"
"  g_free (fi);                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"/********************************************************************/        \n"
"                                                                              \n"
"                                                                              \n"
"static int                                                                    \n"
"fattal02_float_cmp (const void *_a,                                           \n"
"                    const void *_b)                                           \n"
"{                                                                             \n"
"  const gfloat a = *(gfloat *)_a,                                             \n"
"               b = *(gfloat *)_b;                                             \n"
"                                                                              \n"
"  if (a < b)                                                                  \n"
"    return -1;                                                                \n"
"  if (a > b)                                                                  \n"
"    return  1;                                                                \n"
"  return  0;                                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_find_percentiles (const gfloat *array,                               \n"
"                           const guint   size,                                \n"
"                           const gfloat  min_percent,                         \n"
"                           gfloat       *min_value,                           \n"
"                           const gfloat  max_percent,                         \n"
"                           gfloat       *max_value)                           \n"
"{                                                                             \n"
"  guint   i;                                                                  \n"
"  gfloat *sorting;                                                            \n"
"                                                                              \n"
"  g_return_if_fail (min_percent <= max_percent);                              \n"
"  g_return_if_fail (min_percent >= 0.0f && min_percent <= 1.0f);              \n"
"  g_return_if_fail (max_percent >= 0.0f && max_percent <= 1.0f);              \n"
"                                                                              \n"
"  sorting = g_new (gfloat, size);                                             \n"
"  for (i = 0; i < size; ++i)                                                  \n"
"    {                                                                         \n"
"      sorting[i] = array[i];                                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  qsort (sorting, size, sizeof (sorting[0]), fattal02_float_cmp);             \n"
"                                                                              \n"
"  *min_value = sorting[(guint)(min_percent * size)];                          \n"
"  *max_value = sorting[(guint)(max_percent * size)];                          \n"
"  g_free (sorting);                                                           \n"
"}                                                                             \n"
"                                                                              \n"
"/********************************************************************/        \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_tonemap (const gfloat        *input,   /* Y */                       \n"
"                  const GeglRectangle *extent,                                \n"
"                  gfloat              *output,  /* L */                       \n"
"                  gfloat               alfa,                                  \n"
"                  gfloat               beta,                                  \n"
"                  gfloat               noise)                                 \n"
"{                                                                             \n"
"  gint     height = extent->height,                                           \n"
"           width  = extent->width,                                            \n"
"           size   = height * width;                                           \n"
"  gint     x, y, i;                                                           \n"
"  gfloat  *H, *FI, *Gx, *Gy, *divergence, *U;                                 \n"
"  gint     levels;                                                            \n"
"  gfloat **pyramid;                                                           \n"
"  gfloat **gradient,                                                          \n"
"          *averages;                                                          \n"
"                                                                              \n"
"  /* find max & min values, normalize to range 0..100 and take logarithm */   \n"
"  {                                                                           \n"
"    gfloat min_input = G_MAXFLOAT,                                            \n"
"           max_input = G_MINFLOAT;                                            \n"
"                                                                              \n"
"    for (i = 0; i < size; ++i)                                                \n"
"      {                                                                       \n"
"        min_input = MIN (min_input, input[i]);                                \n"
"        max_input = MAX (max_input, input[i]);                                \n"
"      }                                                                       \n"
"    g_return_if_fail (min_input <= max_input);                                \n"
"                                                                              \n"
"    H = g_new (gfloat, size);                                                 \n"
"    for (i = 0; i < size; ++i)                                                \n"
"      {                                                                       \n"
"        H[i] = log (100.0f * input[i] / max_input + 1e-4f);                   \n"
"      }                                                                       \n"
"  }                                                                           \n"
"                                                                              \n"
"  GEGL_NOTE (GEGL_DEBUG_PROCESS, \"calculating attenuation matrix\");         \n"
"                                                                              \n"
"  /* create gaussian pyramids */                                              \n"
"  {                                                                           \n"
"    gint min_size = MIN (extent->width, extent->height);                      \n"
"                                                                              \n"
"    for (levels = 0; min_size / 2 >= MINIMUM_PYRAMID; )                       \n"
"      {                                                                       \n"
"        ++levels;                                                             \n"
"        min_size /= 2;                                                        \n"
"      }                                                                       \n"
"                                                                              \n"
"    pyramid = g_new (gfloat*, levels);                                        \n"
"    fattal02_create_gaussian_pyramids (H, extent, pyramid, levels);           \n"
"  }                                                                           \n"
"                                                                              \n"
"  /* calculate gradients and its average values on pyramid levels */          \n"
"  gradient = g_new (gfloat*, levels);                                         \n"
"  averages = g_new (gfloat,  levels);                                         \n"
"                                                                              \n"
"  for (i = 0; i < levels; ++i)                                                \n"
"    {                                                                         \n"
"      gradient[i] = g_new (gfloat, LEVEL_SIZE (extent, i));                   \n"
"      averages[i] = fattal02_calculate_gradients (pyramid[i],                 \n"
"                                                  &LEVEL_EXTENT (extent, i),  \n"
"                                                  gradient[i],                \n"
"                                                  i);                         \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* calculate fi matrix */                                                   \n"
"  FI = g_new (gfloat, size);                                                  \n"
"  fattal02_FI_matrix (FI, extent, gradient, averages, levels,                 \n"
"                      alfa, beta, noise);                                     \n"
"                                                                              \n"
"  /* attenuate gradients */                                                   \n"
"  Gx = g_new (gfloat, size);                                                  \n"
"  Gy = g_new (gfloat, size);                                                  \n"
"                                                                              \n"
"  for (y = 0; y < extent->height; ++y)                                        \n"
"    {                                                                         \n"
"      for (x = 0; x < extent->width; ++x)                                     \n"
"        {                                                                     \n"
"          guint s = (y + 1 == height ? y : y + 1),                            \n"
"            e = (x + 1 ==  width ? x : x + 1);                                \n"
"                                                                              \n"
"          Gx[x + y * width] = ( H[e + y * width] - H[x + y * width]) *        \n"
"                               FI[x + y * width];                             \n"
"          Gy[x + y * width] = ( H[x + s * width] - H[x + y * width]) *        \n"
"                               FI[x + y * width];                             \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  GEGL_NOTE (GEGL_DEBUG_PROCESS, \"compressing gradients\");                  \n"
"                                                                              \n"
"  /* calculate divergence */                                                  \n"
"  divergence = g_new (gfloat, size);                                          \n"
"  for (y = 0; y < height; ++y)                                                \n"
"    {                                                                         \n"
"      for (x = 0; x < width; ++x)                                             \n"
"        {                                                                     \n"
"          divergence[x + y * width] = Gx[x + y * width] + Gy[x + y * width];  \n"
"          if (x > 0) divergence[x + y * width] -= Gx[x - 1 + (y    ) * width];\n"
"          if (y > 0) divergence[x + y * width] -= Gy[x     + (y - 1) * width];\n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  GEGL_NOTE (GEGL_DEBUG_PROCESS, \"recovering image\");                       \n"
"                                                                              \n"
"  /* solve pde and exponentiate (ie recover compressed image) */              \n"
"  U = g_new (gfloat, size);                                                   \n"
"  fattal02_solve_pde_multigrid (divergence, extent, U, extent);               \n"
"                                                                              \n"
"  for (i = 0; i < size; ++i)                                                  \n"
"    output[i] = expf (U[i]) - 1e-4f;                                          \n"
"                                                                              \n"
"  {                                                                           \n"
"    gfloat min, max, range;                                                   \n"
"                                                                              \n"
"    /* remove percentile of min and max values and renormalize */             \n"
"    fattal02_find_percentiles (output, size,                                  \n"
"                               0.001f, &min,                                  \n"
"                               0.995f, &max);                                 \n"
"    range = max - min;                                                        \n"
"                                                                              \n"
"    for (i = 0; i < size; ++i)                                                \n"
"      {                                                                       \n"
"        output[i] = (output[i] - min) / range;                                \n"
"        if (output[i] <= 0.0f)                                                \n"
"            output[i] = 1e-4f;                                                \n"
"      }                                                                       \n"
"  }                                                                           \n"
"                                                                              \n"
"  /* clean up */                                                              \n"
"  g_free (H);                                                                 \n"
"  for (i = 0; i < levels; ++i)                                                \n"
"    {                                                                         \n"
"      g_free (  pyramid[i]);                                                  \n"
"      g_free (gradient[i]);                                                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (pyramid);                                                           \n"
"  g_free (gradient);                                                          \n"
"  g_free (averages);                                                          \n"
"  g_free (FI);                                                                \n"
"  g_free (Gx);                                                                \n"
"  g_free (Gy);                                                                \n"
"  g_free (divergence);                                                        \n"
"  g_free (U);                                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static void                                                                   \n"
"fattal02_prepare (GeglOperation *operation)                                   \n"
"{                                                                             \n"
"  gegl_operation_set_format (operation, \"input\",  babl_format (OUTPUT_FORMAT));\n"
"  gegl_operation_set_format (operation, \"output\", babl_format (OUTPUT_FORMAT));\n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"fattal02_get_required_for_output (GeglOperation       *operation,             \n"
"                                  const gchar         *input_pad,             \n"
"                                  const GeglRectangle *roi)                   \n"
"{                                                                             \n"
"  GeglRectangle result = *gegl_operation_source_get_bounding_box (operation,  \n"
"                                                                  \"input\"); \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"fattal02_get_cached_region (GeglOperation       *operation,                   \n"
"                            const GeglRectangle *roi)                         \n"
"{                                                                             \n"
"  return *gegl_operation_source_get_bounding_box (operation, \"input\");      \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"static gboolean                                                               \n"
"fattal02_process (GeglOperation       *operation,                             \n"
"                  GeglBuffer          *input,                                 \n"
"                  GeglBuffer          *output,                                \n"
"                  const GeglRectangle *result,                                \n"
"                  gint                 level)                                 \n"
"{                                                                             \n"
"  const GeglProperties *o     = GEGL_PROPERTIES (operation);                  \n"
"  gfloat            noise;                                                    \n"
"                                                                              \n"
"  const gint  pix_stride = 3; /* RGBA */                                      \n"
"  gfloat     *lum_in,                                                         \n"
"             *lum_out,                                                        \n"
"             *pix;                                                            \n"
"  gint        i;                                                              \n"
"                                                                              \n"
"  g_return_val_if_fail (operation, FALSE);                                    \n"
"  g_return_val_if_fail (input, FALSE);                                        \n"
"  g_return_val_if_fail (output, FALSE);                                       \n"
"  g_return_val_if_fail (result, FALSE);                                       \n"
"                                                                              \n"
"  g_return_val_if_fail (babl_format_get_n_components (babl_format (OUTPUT_FORMAT)) == pix_stride, FALSE);\n"
"                                                                              \n"
"  /* Adjust noise floor if not set by the user */                             \n"
"  if (o->noise == 0.0)                                                        \n"
"    {                                                                         \n"
"      noise = o->alpha * 0.1;                                                 \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      noise = o->noise;                                                       \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Obtain the pixel data */                                                 \n"
"  lum_in  = g_new (gfloat, result->width * result->height);                   \n"
"  lum_out = g_new (gfloat, result->width * result->height);                   \n"
"                                                                              \n"
"  gegl_buffer_get (input, result, 1.0, babl_format (\"Y float\"),             \n"
"                   lum_in, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);             \n"
"                                                                              \n"
"  pix = g_new (gfloat, result->width * result->height * pix_stride);          \n"
"  gegl_buffer_get (input, result, 1.0, babl_format (OUTPUT_FORMAT),           \n"
"                   pix, GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                \n"
"                                                                              \n"
"  fattal02_tonemap (lum_in, result, lum_out, o->alpha, o->beta, noise);       \n"
"                                                                              \n"
"  for (i = 0; i < result->width * result->height * pix_stride; ++i)           \n"
"    {                                                                         \n"
"      pix[i] = (powf (pix[i] / lum_in[i / pix_stride],                        \n"
"                      o->saturation) *                                        \n"
"                lum_out[i / pix_stride]);                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_buffer_set (output, result, 0, babl_format (OUTPUT_FORMAT), pix,       \n"
"                   GEGL_AUTO_ROWSTRIDE);                                      \n"
"  g_free (pix);                                                               \n"
"  g_free (lum_out);                                                           \n"
"  g_free (lum_in);                                                            \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/*                                                                            \n"
" */                                                                           \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  filter_class->process = fattal02_process;                                   \n"
"                                                                              \n"
"  operation_class->prepare                 = fattal02_prepare;                \n"
"  operation_class->get_required_for_output = fattal02_get_required_for_output;\n"
"  operation_class->get_cached_region       = fattal02_get_cached_region;      \n"
"  operation_class->threaded                = FALSE;                           \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"  \"name\"       , \"gegl:fattal02\",                                         \n"
"  \"title\",       _(\"Fattal et al. 2002 Tone Mapping\"),                    \n"
"  \"categories\" , \"tonemapping:enhance\",                                   \n"
"  \"description\",                                                            \n"
"        _(\"Adapt an image, which may have a high dynamic range, for \"       \n"
"	  \"presentation using a low dynamic range. This operator attenuates \"      \n"
"          \"the magnitudes of local image gradients, producing luminance \"   \n"
"          \"within the range 0.0-1.0. This tonemapping approach was originally presented by Raanan Fattal, in the 2002 SIGGRAPH paper: Gradient Domain High Dynamic Range Compression.\"),\n"
"        NULL);                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
