static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2011 Michael Muré <batolettre@gmail.com>                        \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"enum_start (gegl_warp_behavior)                                               \n"
"  enum_value (GEGL_WARP_BEHAVIOR_MOVE,      \"move\",      N_(\"Move pixels\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_GROW,      \"grow\",      N_(\"Grow area\")) \n"
"  enum_value (GEGL_WARP_BEHAVIOR_SHRINK,    \"shrink\",    N_(\"Shrink area\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_SWIRL_CW,  \"swirl-cw\",  N_(\"Swirl clockwise\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_SWIRL_CCW, \"swirl-ccw\", N_(\"Swirl counter-clockwise\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_ERASE,     \"erase\",     N_(\"Erase warping\"))\n"
"  enum_value (GEGL_WARP_BEHAVIOR_SMOOTH,    \"smooth\",    N_(\"Smooth warping\"))\n"
"enum_end (GeglWarpBehavior)                                                   \n"
"                                                                              \n"
"property_double (strength, _(\"Strength\"), 50)                               \n"
"  value_range (0, 100)                                                        \n"
"                                                                              \n"
"property_double (size, _(\"Size\"), 40.0)                                     \n"
"  value_range (1.0, 10000.0)                                                  \n"
"                                                                              \n"
"property_double (hardness, _(\"Hardness\"), 0.5)                              \n"
"  value_range (0.0, 1.0)                                                      \n"
"                                                                              \n"
"property_double (spacing, _(\"Spacing\"), 0.01)                               \n"
"  value_range (0.0, 100.0)                                                    \n"
"                                                                              \n"
"property_path   (stroke, _(\"Stroke\"), NULL)                                 \n"
"                                                                              \n"
"property_enum   (behavior, _(\"Behavior\"),                                   \n"
"                   GeglWarpBehavior, gegl_warp_behavior,                      \n"
"                   GEGL_WARP_BEHAVIOR_MOVE)                                   \n"
"  description   (_(\"Behavior of the op\"))                                   \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_C_SOURCE warp.c                                               \n"
"#define GEGL_OP_NAME     warp                                                 \n"
"                                                                              \n"
"#include \"gegl-plugin.h\"                                                    \n"
"#include \"gegl-path.h\"                                                      \n"
"                                                                              \n"
"static void node_invalidated (GeglNode            *node,                      \n"
"                              const GeglRectangle *roi,                       \n"
"                              GeglOperation       *operation);                \n"
"                                                                              \n"
"static void path_changed     (GeglPath            *path,                      \n"
"                              const GeglRectangle *roi,                       \n"
"                              GeglOperation       *operation);                \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"                                                                              \n"
"#define HARDNESS_EPSILON 0.0000004                                            \n"
"                                                                              \n"
"typedef struct WarpPointList                                                  \n"
"{                                                                             \n"
"  GeglPathPoint         point;                                                \n"
"  struct WarpPointList *next;                                                 \n"
"} WarpPointList;                                                              \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gfloat         *lookup;                                                     \n"
"  GeglBuffer     *buffer;                                                     \n"
"  WarpPointList  *processed_stroke;                                           \n"
"  WarpPointList **processed_stroke_tail;                                      \n"
"  gboolean        processed_stroke_valid;                                     \n"
"  GeglPathList   *remaining_stroke;                                           \n"
"  gfloat          last_x;                                                     \n"
"  gfloat          last_y;                                                     \n"
"} WarpPrivate;                                                                \n"
"                                                                              \n"
"static void                                                                   \n"
"clear_cache (GeglProperties *o)                                               \n"
"{                                                                             \n"
"  WarpPrivate *priv = (WarpPrivate *) o->user_data;                           \n"
"                                                                              \n"
"  if (! priv)                                                                 \n"
"    return;                                                                   \n"
"                                                                              \n"
"  if (priv->lookup)                                                           \n"
"    {                                                                         \n"
"      g_free (priv->lookup);                                                  \n"
"                                                                              \n"
"      priv->lookup = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (priv->buffer)                                                           \n"
"    {                                                                         \n"
"      g_object_unref (priv->buffer);                                          \n"
"                                                                              \n"
"      priv->buffer = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  while (priv->processed_stroke)                                              \n"
"    {                                                                         \n"
"      WarpPointList *next = priv->processed_stroke->next;                     \n"
"                                                                              \n"
"      g_slice_free (WarpPointList, priv->processed_stroke);                   \n"
"                                                                              \n"
"      priv->processed_stroke = next;                                          \n"
"    }                                                                         \n"
"                                                                              \n"
"  priv->processed_stroke_tail = &priv->processed_stroke;                      \n"
"                                                                              \n"
"  priv->processed_stroke_valid = TRUE;                                        \n"
"                                                                              \n"
"  priv->remaining_stroke = o->stroke ? gegl_path_get_path (o->stroke) : NULL; \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"validate_processed_stroke (GeglProperties *o)                                 \n"
"{                                                                             \n"
"  WarpPrivate   *priv = (WarpPrivate *) o->user_data;                         \n"
"  GeglPathList  *event;                                                       \n"
"  WarpPointList *processed_event;                                             \n"
"                                                                              \n"
"  if (priv->processed_stroke_valid)                                           \n"
"    return;                                                                   \n"
"                                                                              \n"
"  /* check if the previously processed stroke is an initial segment of the    \n"
"   * current stroke.                                                          \n"
"   */                                                                         \n"
"  for (event           = o->stroke ? gegl_path_get_path (o->stroke) : NULL,   \n"
"       processed_event = priv->processed_stroke;                              \n"
"                                                                              \n"
"       event && processed_event;                                              \n"
"                                                                              \n"
"       event           = event->next,                                         \n"
"       processed_event = processed_event->next)                               \n"
"    {                                                                         \n"
"      if (event->d.point[0].x != processed_event->point.x ||                  \n"
"          event->d.point[0].y != processed_event->point.y)                    \n"
"        {                                                                     \n"
"          break;                                                              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (! processed_event)                                                      \n"
"    {                                                                         \n"
"      /* it is.  prepare for processing the remaining portion of the stroke on\n"
"       * the next call to process().                                          \n"
"       */                                                                     \n"
"      priv->remaining_stroke = event;                                         \n"
"                                                                              \n"
"      priv->processed_stroke_valid = TRUE;                                    \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      /* it isn't.  clear the cache so that we start from scratch. */         \n"
"      clear_cache (o);                                                        \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"node_invalidated (GeglNode            *node,                                  \n"
"                  const GeglRectangle *rect,                                  \n"
"                  GeglOperation       *operation)                             \n"
"{                                                                             \n"
"  /* if the node is invalidated, clear all cached data.  in particular, redraw\n"
"   * the entire stroke upon the next call to process().                       \n"
"   */                                                                         \n"
"  clear_cache (GEGL_PROPERTIES (operation));                                  \n"
"}                                                                             \n"
"                                                                              \n"
"/* return the smallest range of pixels [min_pixel, max_pixel], whose centers  \n"
" * are inside the range [min_coord, max_coord].                               \n"
" */                                                                           \n"
"static inline void                                                            \n"
"pixel_range (gfloat  min_coord,                                               \n"
"             gfloat  max_coord,                                               \n"
"             gint   *min_pixel,                                               \n"
"             gint   *max_pixel)                                               \n"
"{                                                                             \n"
"  *min_pixel = ceilf  (min_coord - 0.5f);                                     \n"
"  *max_pixel = floorf (max_coord - 0.5f);                                     \n"
"}                                                                             \n"
"                                                                              \n"
"/* return the smallest rectangle of pixels, whose centers are inside the      \n"
" * horizontal range [min_x, max_x] and the vertical range [min_y, max_y].     \n"
" */                                                                           \n"
"static inline GeglRectangle                                                   \n"
"pixel_extent (gfloat min_x,                                                   \n"
"              gfloat max_x,                                                   \n"
"              gfloat min_y,                                                   \n"
"              gfloat max_y)                                                   \n"
"{                                                                             \n"
"  GeglRectangle result;                                                       \n"
"                                                                              \n"
"  pixel_range (min_x,     max_x,                                              \n"
"               &result.x, &result.width);                                     \n"
"  result.width -= result.x - 1;                                               \n"
"                                                                              \n"
"  pixel_range (min_y,     max_y,                                              \n"
"               &result.y, &result.height);                                    \n"
"  result.height -= result.y - 1;                                              \n"
"                                                                              \n"
"  return result;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"path_changed (GeglPath            *path,                                      \n"
"              const GeglRectangle *roi,                                       \n"
"              GeglOperation       *operation)                                 \n"
"{                                                                             \n"
"  GeglRectangle   rect;                                                       \n"
"  GeglProperties *o    = GEGL_PROPERTIES (operation);                         \n"
"  WarpPrivate    *priv = (WarpPrivate *) o->user_data;                        \n"
"                                                                              \n"
"  /* mark the previously processed stroke as invalid, so that we check it     \n"
"   * against the new stroke before processing.                                \n"
"   */                                                                         \n"
"  if (priv)                                                                   \n"
"    priv->processed_stroke_valid = FALSE;                                     \n"
"                                                                              \n"
"  /* invalidate the incoming rectangle */                                     \n"
"                                                                              \n"
"  rect = pixel_extent (roi->x               - o->size / 2.0,                  \n"
"                       roi->x + roi->width  + o->size / 2.0,                  \n"
"                       roi->y               - o->size / 2.0,                  \n"
"                       roi->y + roi->height + o->size / 2.0);                 \n"
"                                                                              \n"
"  /* avoid clearing the cache.  it will be cleared, if necessary, when        \n"
"   * validating the stroke.                                                   \n"
"   */                                                                         \n"
"  g_signal_handlers_block_by_func (operation->node,                           \n"
"                                   node_invalidated, operation);              \n"
"                                                                              \n"
"  gegl_operation_invalidate (operation, &rect, FALSE);                        \n"
"                                                                              \n"
"  g_signal_handlers_unblock_by_func (operation->node,                         \n"
"                                     node_invalidated, operation);            \n"
"}                                                                             \n"
"                                                                              \n"
"static gdouble                                                                \n"
"gauss (gdouble f)                                                             \n"
"{                                                                             \n"
"  /* This is not a real gauss function. */                                    \n"
"  /* Approximation is valid if -1 < f < 1 */                                  \n"
"  if (f < -1.0)                                                               \n"
"    return 0.0;                                                               \n"
"                                                                              \n"
"  if (f < -0.5)                                                               \n"
"    {                                                                         \n"
"      f = -1.0 - f;                                                           \n"
"      return (2.0 * f*f);                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (f < 0.5)                                                                \n"
"    return (1.0 - 2.0 * f*f);                                                 \n"
"                                                                              \n"
"  if (f < 1.0)                                                                \n"
"    {                                                                         \n"
"      f = 1.0 - f;                                                            \n"
"      return (2.0 * f*f);                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return 0.0;                                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"/* set up lookup table */                                                     \n"
"static void                                                                   \n"
"calc_lut (GeglProperties  *o)                                                 \n"
"{                                                                             \n"
"  WarpPrivate  *priv = (WarpPrivate*) o->user_data;                           \n"
"  gdouble       radius;                                                       \n"
"  gint          length;                                                       \n"
"  gint          x;                                                            \n"
"  gdouble       exponent;                                                     \n"
"                                                                              \n"
"  if (priv->lookup)                                                           \n"
"    return;                                                                   \n"
"                                                                              \n"
"  radius = o->size / 2.0;                                                     \n"
"                                                                              \n"
"  length = floor (radius) + 3;                                                \n"
"                                                                              \n"
"  priv->lookup = g_new (gfloat, length);                                      \n"
"                                                                              \n"
"  if (1.0 - o->hardness > HARDNESS_EPSILON)                                   \n"
"    {                                                                         \n"
"      exponent = 0.4 / (1.0 - o->hardness);                                   \n"
"                                                                              \n"
"      for (x = 0; x < length; x++)                                            \n"
"        {                                                                     \n"
"          priv->lookup[x] = gauss (pow (x / radius, exponent));               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      for (x = 0; x < length; x++)                                            \n"
"        {                                                                     \n"
"          priv->lookup[x] = 1.0f;                                             \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"attach (GeglOperation *operation)                                             \n"
"{                                                                             \n"
"  GEGL_OPERATION_CLASS (gegl_op_parent_class)->attach (operation);            \n"
"                                                                              \n"
"  g_signal_connect_object (operation->node, \"invalidated\",                  \n"
"                           G_CALLBACK (node_invalidated), operation, 0);      \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (operation);                            \n"
"                                                                              \n"
"  const Babl *format = babl_format_n (babl_type (\"float\"), 2);              \n"
"  gegl_operation_set_format (operation, \"input\", format);                   \n"
"  gegl_operation_set_format (operation, \"output\", format);                  \n"
"                                                                              \n"
"  if (!o->user_data)                                                          \n"
"    {                                                                         \n"
"      o->user_data = g_slice_new0 (WarpPrivate);                              \n"
"                                                                              \n"
"      clear_cache (o);                                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  validate_processed_stroke (o);                                              \n"
"  calc_lut (o);                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_required_for_output (GeglOperation       *operation,                      \n"
"                         const gchar         *input_pad,                      \n"
"                         const GeglRectangle *output_roi)                     \n"
"{                                                                             \n"
"  GeglProperties *o    = GEGL_PROPERTIES (operation);                         \n"
"  WarpPrivate    *priv = (WarpPrivate*) o->user_data;                         \n"
"  GeglRectangle   rect = {0, 0, 0, 0};                                        \n"
"                                                                              \n"
"  /* we only need the input if we don't have a cached buffer already. */      \n"
"  if (! priv->buffer)                                                         \n"
"    rect = *gegl_operation_source_get_bounding_box (operation, input_pad);    \n"
"                                                                              \n"
"  return rect;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"finalize (GObject *object)                                                    \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (object);                               \n"
"                                                                              \n"
"  if (o->user_data)                                                           \n"
"    {                                                                         \n"
"      clear_cache (o);                                                        \n"
"                                                                              \n"
"      g_slice_free (WarpPrivate, o->user_data);                               \n"
"      o->user_data = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  G_OBJECT_CLASS (gegl_op_parent_class)->finalize (object);                   \n"
"}                                                                             \n"
"                                                                              \n"
"static inline gfloat                                                          \n"
"get_stamp_force (gfloat        x,                                             \n"
"                 gfloat        y,                                             \n"
"                 const gfloat *lookup)                                        \n"
"{                                                                             \n"
"  gfloat radius;                                                              \n"
"  gint   a;                                                                   \n"
"  gfloat ratio;                                                               \n"
"  gfloat before, after;                                                       \n"
"                                                                              \n"
"  radius = sqrtf (x * x + y * y);                                             \n"
"                                                                              \n"
"  /* linear interpolation */                                                  \n"
"  a = (gint) radius;                                                          \n"
"  ratio = (radius - a);                                                       \n"
"                                                                              \n"
"  before = lookup[a];                                                         \n"
"  after = lookup[a + 1];                                                      \n"
"                                                                              \n"
"  return before + ratio * (after - before);                                   \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"stamp (GeglProperties      *o,                                                \n"
"       gfloat              *srcbuf,                                           \n"
"       gint                 srcbuf_stride,                                    \n"
"       const GeglRectangle *srcbuf_extent,                                    \n"
"       gfloat               x,                                                \n"
"       gfloat               y)                                                \n"
"{                                                                             \n"
"  WarpPrivate   *priv = (WarpPrivate*) o->user_data;                          \n"
"  gfloat         stamp_force, influence;                                      \n"
"  gfloat         x_mean = 0.0;                                                \n"
"  gfloat         y_mean = 0.0;                                                \n"
"  gint           x_iter, y_iter;                                              \n"
"  gfloat         xi, yi;                                                      \n"
"  GeglRectangle  area;                                                        \n"
"  gint           sample_min_x, sample_max_x;                                  \n"
"  gint           sample_min_y, sample_max_y;                                  \n"
"  gfloat        *stampbuf;                                                    \n"
"  gfloat        *vals;                                                        \n"
"  gfloat        *srcvals;                                                     \n"
"  gfloat         stamp_radius_sq = 0.25 * o->size * o->size;                  \n"
"  gfloat         strength = 0.01 * o->strength;                               \n"
"  const gfloat  *lookup = priv->lookup;                                       \n"
"  gfloat         s = 0, c = 0;                                                \n"
"  gfloat         motion_x, motion_y;                                          \n"
"  gfloat         lim;                                                         \n"
"  gint           min_x, max_x;                                                \n"
"                                                                              \n"
"  motion_x = priv->last_x - x;                                                \n"
"  motion_y = priv->last_y - y;                                                \n"
"                                                                              \n"
"  /* Memorize the stamp location for movement dependant behavior like move */ \n"
"  priv->last_x = x;                                                           \n"
"  priv->last_y = y;                                                           \n"
"                                                                              \n"
"  if (o->behavior == GEGL_WARP_BEHAVIOR_MOVE &&                               \n"
"      motion_x == 0.0f && motion_y == 0.0f)                                   \n"
"    {                                                                         \n"
"      return;                                                                 \n"
"    }                                                                         \n"
"                                                                              \n"
"  area = pixel_extent (x - o->size / 2.0, x + o->size / 2.0,                  \n"
"                       y - o->size / 2.0, y + o->size / 2.0);                 \n"
"                                                                              \n"
"  if (! gegl_rectangle_intersect (&area, &area, srcbuf_extent))               \n"
"    return;                                                                   \n"
"                                                                              \n"
"  /* Shift the coordiantes so that they're relative to the top-left corner of \n"
"   * the stamped area.                                                        \n"
"   */                                                                         \n"
"  x -= area.x;                                                                \n"
"  y -= area.y;                                                                \n"
"                                                                              \n"
"  /* Shift the stamped area so that it's relative to the top-left corner of the\n"
"   * source buffer.                                                           \n"
"   */                                                                         \n"
"  area.x -= srcbuf_extent->x;                                                 \n"
"  area.y -= srcbuf_extent->y;                                                 \n"
"                                                                              \n"
"  /* Align the source buffer with the stamped area */                         \n"
"  srcbuf += srcbuf_stride * area.y + 2 * area.x;                              \n"
"                                                                              \n"
"  /* Calculate the sample bounds.  We clamp the coordinates of pixels sampled \n"
"   * from the source buffer to these limits.                                  \n"
"   */                                                                         \n"
"  sample_min_x = -area.x;                                                     \n"
"  sample_max_x = -area.x + srcbuf_extent->width - 1;                          \n"
"  sample_min_y = -area.y;                                                     \n"
"  sample_max_y = -area.y + srcbuf_extent->height - 1;                         \n"
"                                                                              \n"
"  /* If needed, compute the mean deformation */                               \n"
"  if (o->behavior == GEGL_WARP_BEHAVIOR_SMOOTH)                               \n"
"    {                                                                         \n"
"      gfloat total_weight = 0.0f;                                             \n"
"                                                                              \n"
"      yi = -y + 0.5f;                                                         \n"
"                                                                              \n"
"      for (y_iter = 0; y_iter < area.height; y_iter++, yi++)                  \n"
"        {                                                                     \n"
"          lim = stamp_radius_sq - yi * yi;                                    \n"
"                                                                              \n"
"          if (lim < 0.0f)                                                     \n"
"            continue;                                                         \n"
"                                                                              \n"
"          lim = sqrtf (lim);                                                  \n"
"                                                                              \n"
"          pixel_range (x - lim, x + lim,                                      \n"
"                       &min_x,  &max_x);                                      \n"
"                                                                              \n"
"          if (max_x < 0 || min_x >= area.width)                               \n"
"            continue;                                                         \n"
"                                                                              \n"
"          min_x = CLAMP (min_x, 0, area.width - 1);                           \n"
"          max_x = CLAMP (max_x, 0, area.width - 1);                           \n"
"                                                                              \n"
"          srcvals = srcbuf + srcbuf_stride * y_iter + 2 * min_x;              \n"
"                                                                              \n"
"          xi = -x + min_x + 0.5f;                                             \n"
"                                                                              \n"
"          for (x_iter  = min_x;                                               \n"
"               x_iter <= max_x;                                               \n"
"               x_iter++, xi++, srcvals += 2)                                  \n"
"            {                                                                 \n"
"              stamp_force = get_stamp_force (xi, yi, lookup);                 \n"
"                                                                              \n"
"              x_mean += stamp_force * srcvals[0];                             \n"
"              y_mean += stamp_force * srcvals[1];                             \n"
"                                                                              \n"
"              total_weight += stamp_force;                                    \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      x_mean /= total_weight;                                                 \n"
"      y_mean /= total_weight;                                                 \n"
"    }                                                                         \n"
"  else if (o->behavior == GEGL_WARP_BEHAVIOR_GROW ||                          \n"
"           o->behavior == GEGL_WARP_BEHAVIOR_SHRINK)                          \n"
"    {                                                                         \n"
"      strength *= 0.1f;                                                       \n"
"                                                                              \n"
"      if (o->behavior == GEGL_WARP_BEHAVIOR_GROW)                             \n"
"        strength = -strength;                                                 \n"
"    }                                                                         \n"
"  else if (o->behavior == GEGL_WARP_BEHAVIOR_SWIRL_CW ||                      \n"
"           o->behavior == GEGL_WARP_BEHAVIOR_SWIRL_CCW)                       \n"
"    {                                                                         \n"
"      /* swirl by 5 degrees per stamp (for strength 100).                     \n"
"       * not exactly sin/cos factors,                                         \n"
"       * since we calculate an off-center offset-vector */                    \n"
"                                                                              \n"
"      /* note that this is fudged for stamp_force < 1.0 and                   \n"
"       * results in a slight upscaling there. It is a compromise              \n"
"       * between exactness and calculation speed. */                          \n"
"      s = sin (0.01 * o->strength * 5.0 / 180 * G_PI);                        \n"
"      c = cos (0.01 * o->strength * 5.0 / 180 * G_PI) - 1.0;                  \n"
"                                                                              \n"
"      if (o->behavior == GEGL_WARP_BEHAVIOR_SWIRL_CW)                         \n"
"        s = -s;                                                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* We render the stamp into a temporary buffer, to avoid overwriting data   \n"
"   * that is still needed.                                                    \n"
"   */                                                                         \n"
"  stampbuf = g_new (gfloat, 2 * area.height * area.width);                    \n"
"                                                                              \n"
"  yi = -y + 0.5f;                                                             \n"
"                                                                              \n"
"  for (y_iter = 0; y_iter < area.height; y_iter++, yi++)                      \n"
"    {                                                                         \n"
"      lim = stamp_radius_sq - yi * yi;                                        \n"
"                                                                              \n"
"      if (lim < 0.0f)                                                         \n"
"        continue;                                                             \n"
"                                                                              \n"
"      lim = sqrtf (lim);                                                      \n"
"                                                                              \n"
"      pixel_range (x - lim, x + lim,                                          \n"
"                   &min_x,  &max_x);                                          \n"
"                                                                              \n"
"      if (max_x < 0 || min_x >= area.width)                                   \n"
"        continue;                                                             \n"
"                                                                              \n"
"      min_x = CLAMP (min_x, 0, area.width - 1);                               \n"
"      max_x = CLAMP (max_x, 0, area.width - 1);                               \n"
"                                                                              \n"
"      vals    = stampbuf + 2 * area.width * y_iter + 2 * min_x;               \n"
"      srcvals = srcbuf   + srcbuf_stride  * y_iter + 2 * min_x;               \n"
"                                                                              \n"
"      xi = -x + min_x + 0.5f;                                                 \n"
"                                                                              \n"
"      for (x_iter  = min_x;                                                   \n"
"           x_iter <= max_x;                                                   \n"
"           x_iter++, xi++, vals += 2, srcvals += 2)                           \n"
"        {                                                                     \n"
"          gfloat nvx, nvy;                                                    \n"
"          gfloat fx, fy;                                                      \n"
"          gint   dx, dy;                                                      \n"
"          gfloat weight_x, weight_y;                                          \n"
"          gfloat a0, b0;                                                      \n"
"          gfloat a1, b1;                                                      \n"
"          gfloat *srcptr;                                                     \n"
"                                                                              \n"
"          stamp_force = get_stamp_force (xi, yi, lookup);                     \n"
"          influence   = strength * stamp_force;                               \n"
"                                                                              \n"
"          switch (o->behavior)                                                \n"
"            {                                                                 \n"
"              case GEGL_WARP_BEHAVIOR_MOVE:                                   \n"
"                nvx = influence * motion_x;                                   \n"
"                nvy = influence * motion_y;                                   \n"
"                break;                                                        \n"
"              case GEGL_WARP_BEHAVIOR_GROW:                                   \n"
"              case GEGL_WARP_BEHAVIOR_SHRINK:                                 \n"
"                nvx = influence * xi;                                         \n"
"                nvy = influence * yi;                                         \n"
"                break;                                                        \n"
"              case GEGL_WARP_BEHAVIOR_SWIRL_CW:                               \n"
"              case GEGL_WARP_BEHAVIOR_SWIRL_CCW:                              \n"
"                nvx = stamp_force * ( c * xi - s * yi);                       \n"
"                nvy = stamp_force * ( s * xi + c * yi);                       \n"
"                break;                                                        \n"
"              case GEGL_WARP_BEHAVIOR_ERASE:                                  \n"
"              case GEGL_WARP_BEHAVIOR_SMOOTH:                                 \n"
"              default:                                                        \n"
"                /* shut up, gcc */                                            \n"
"                nvx = 0.0f;                                                   \n"
"                nvy = 0.0f;                                                   \n"
"                break;                                                        \n"
"            }                                                                 \n"
"                                                                              \n"
"          if (o->behavior == GEGL_WARP_BEHAVIOR_ERASE)                        \n"
"            {                                                                 \n"
"              vals[0] = srcvals[0] * (1.0f - influence);                      \n"
"              vals[1] = srcvals[1] * (1.0f - influence);                      \n"
"            }                                                                 \n"
"          else if (o->behavior == GEGL_WARP_BEHAVIOR_SMOOTH)                  \n"
"            {                                                                 \n"
"              vals[0] = srcvals[0] + influence * (x_mean - srcvals[0]);       \n"
"              vals[1] = srcvals[1] + influence * (y_mean - srcvals[1]);       \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              fx = floorf (nvx);                                              \n"
"              fy = floorf (nvy);                                              \n"
"                                                                              \n"
"              weight_x = nvx - fx;                                            \n"
"              weight_y = nvy - fy;                                            \n"
"                                                                              \n"
"              dx = fx;                                                        \n"
"              dy = fy;                                                        \n"
"                                                                              \n"
"              dx += x_iter;                                                   \n"
"              dy += y_iter;                                                   \n"
"                                                                              \n"
"              /* clamp the sampled coordinates to the sample bounds */        \n"
"              if (dx < sample_min_x || dx >= sample_max_x ||                  \n"
"                  dy < sample_min_y || dy >= sample_max_y)                    \n"
"                {                                                             \n"
"                  if (dx < sample_min_x)                                      \n"
"                    {                                                         \n"
"                      dx = sample_min_x;                                      \n"
"                      weight_x = 0.0f;                                        \n"
"                    }                                                         \n"
"                  else if (dx >= sample_max_x)                                \n"
"                    {                                                         \n"
"                      dx = sample_max_x;                                      \n"
"                      weight_x = 0.0f;                                        \n"
"                    }                                                         \n"
"                                                                              \n"
"                  if (dy < sample_min_y)                                      \n"
"                    {                                                         \n"
"                      dy = sample_min_y;                                      \n"
"                      weight_y = 0.0f;                                        \n"
"                    }                                                         \n"
"                  else if (dy >= sample_max_y)                                \n"
"                    {                                                         \n"
"                      dy = sample_max_y;                                      \n"
"                      weight_y = 0.0f;                                        \n"
"                    }                                                         \n"
"                }                                                             \n"
"                                                                              \n"
"              srcptr = srcbuf + srcbuf_stride * dy + 2 * dx;                  \n"
"                                                                              \n"
"              /* bilinear interpolation of the vectors */                     \n"
"                                                                              \n"
"              a0 = srcptr[0] + (srcptr[2] - srcptr[0]) * weight_x;            \n"
"              b0 = srcptr[srcbuf_stride + 0] +                                \n"
"                   (srcptr[srcbuf_stride + 2] - srcptr[srcbuf_stride + 0]) *  \n"
"                   weight_x;                                                  \n"
"                                                                              \n"
"              a1 = srcptr[1] + (srcptr[3] - srcptr[1]) * weight_x;            \n"
"              b1 = srcptr[srcbuf_stride + 1] +                                \n"
"                   (srcptr[srcbuf_stride + 3] - srcptr[srcbuf_stride + 1]) *  \n"
"                   weight_x;                                                  \n"
"                                                                              \n"
"              vals[0] = a0 + (b0 - a0) * weight_y;                            \n"
"              vals[1] = a1 + (b1 - a1) * weight_y;                            \n"
"                                                                              \n"
"              vals[0] += nvx;                                                 \n"
"              vals[1] += nvy;                                                 \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* Paste the stamp into the source buffer. */                               \n"
"  yi = -y + 0.5f;                                                             \n"
"                                                                              \n"
"  for (y_iter = 0; y_iter < area.height; y_iter++, yi++)                      \n"
"    {                                                                         \n"
"      lim = stamp_radius_sq - yi * yi;                                        \n"
"                                                                              \n"
"      if (lim < 0.0f)                                                         \n"
"        continue;                                                             \n"
"                                                                              \n"
"      lim = sqrtf (lim);                                                      \n"
"                                                                              \n"
"      pixel_range (x - lim, x + lim,                                          \n"
"                   &min_x,  &max_x);                                          \n"
"                                                                              \n"
"      if (max_x < 0 || min_x >= area.width)                                   \n"
"        continue;                                                             \n"
"                                                                              \n"
"      min_x = CLAMP (min_x, 0, area.width - 1);                               \n"
"      max_x = CLAMP (max_x, 0, area.width - 1);                               \n"
"                                                                              \n"
"      vals    = stampbuf + 2 * area.width * y_iter + 2 * min_x;               \n"
"      srcvals = srcbuf   + srcbuf_stride  * y_iter + 2 * min_x;               \n"
"                                                                              \n"
"      memcpy (srcvals, vals, 2 * sizeof (gfloat) * (max_x - min_x + 1));      \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (stampbuf);                                                          \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation        *operation,                                     \n"
"         GeglOperationContext *context,                                       \n"
"         const gchar          *output_prop,                                   \n"
"         const GeglRectangle  *result,                                        \n"
"         gint                  level)                                         \n"
"{                                                                             \n"
"  GeglProperties *o    = GEGL_PROPERTIES (operation);                         \n"
"  WarpPrivate    *priv = (WarpPrivate*) o->user_data;                         \n"
"                                                                              \n"
"  GObject        *input;                                                      \n"
"  GObject        *output;                                                     \n"
"                                                                              \n"
"  gdouble         spacing = MAX (o->size * o->spacing, 0.5);                  \n"
"  gdouble         dist;                                                       \n"
"  gint            stamps;                                                     \n"
"  gint            i;                                                          \n"
"  gdouble         t;                                                          \n"
"                                                                              \n"
"  gdouble         min_x, max_x;                                               \n"
"  gdouble         min_y, max_y;                                               \n"
"                                                                              \n"
"  gfloat         *srcbuf;                                                     \n"
"  gint            srcbuf_stride;                                              \n"
"  gint            srcbuf_padding;                                             \n"
"  GeglRectangle   srcbuf_extent;                                              \n"
"                                                                              \n"
"  GeglPathPoint   prev, next, lerp;                                           \n"
"  GeglPathList   *event;                                                      \n"
"  WarpPointList  *processed_event;                                            \n"
"                                                                              \n"
"  if (!o->stroke || strcmp (output_prop, \"output\"))                         \n"
"    return FALSE;                                                             \n"
"                                                                              \n"
"  event = priv->remaining_stroke;                                             \n"
"                                                                              \n"
"  /* if there is no stroke data left to process, pass the cached buffer right \n"
"   * away, or, if we don't have a cacehd buffer, pass the input buffer        \n"
"   * directly.                                                                \n"
"   *                                                                          \n"
"   * altenatively, if the stroke's strength is 0, the stroke has no effect.  do\n"
"   * the same.                                                                \n"
"   */                                                                         \n"
"  if (! event || o->strength == 0.0)                                          \n"
"    {                                                                         \n"
"      if (priv->buffer)                                                       \n"
"        output = G_OBJECT (priv->buffer);                                     \n"
"      else                                                                    \n"
"        output = gegl_operation_context_get_object (context, \"input\");      \n"
"                                                                              \n"
"      gegl_operation_context_set_object (context, \"output\", output);        \n"
"                                                                              \n"
"      return TRUE;                                                            \n"
"    }                                                                         \n"
"  /* otherwise, we process the remaining stroke on top of the previously-     \n"
"   * processed buffer.                                                        \n"
"   */                                                                         \n"
"                                                                              \n"
"  /* intialize the cached buffer if we don't already have one. */             \n"
"  if (! priv->buffer)                                                         \n"
"    {                                                                         \n"
"      input = gegl_operation_context_get_object (context, \"input\");         \n"
"                                                                              \n"
"      priv->buffer = gegl_buffer_dup (GEGL_BUFFER (input));                   \n"
"                                                                              \n"
"      /* we pass the buffer as output directly while keeping it cached, so mark\n"
"       * it as forked.                                                        \n"
"       */                                                                     \n"
"      gegl_object_set_has_forked (G_OBJECT (priv->buffer));                   \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* is this the first event of the stroke? */                                \n"
"  if (! priv->processed_stroke)                                               \n"
"    {                                                                         \n"
"      prev = *(event->d.point);                                               \n"
"                                                                              \n"
"      priv->last_x = prev.x;                                                  \n"
"      priv->last_y = prev.y;                                                  \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      prev.x = priv->last_x;                                                  \n"
"      prev.y = priv->last_y;                                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* find the bounding box of the portion of the stroke we're about to        \n"
"   * process.                                                                 \n"
"   */                                                                         \n"
"  min_x = max_x = prev.x;                                                     \n"
"  min_y = max_y = prev.y;                                                     \n"
"                                                                              \n"
"  for (event = priv->remaining_stroke; event; event = event->next)            \n"
"    {                                                                         \n"
"      min_x = MIN (min_x, event->d.point[0].x);                               \n"
"      max_x = MAX (max_x, event->d.point[0].x);                               \n"
"                                                                              \n"
"      min_y = MIN (min_y, event->d.point[0].y);                               \n"
"      max_y = MAX (max_y, event->d.point[0].y);                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  srcbuf_extent.x      = floor (min_x - o->size / 2.0) - 1;                   \n"
"  srcbuf_extent.y      = floor (min_y - o->size / 2.0) - 1;                   \n"
"  srcbuf_extent.width  = ceil (max_x + o->size / 2.0) + 1 - srcbuf_extent.x;  \n"
"  srcbuf_extent.height = ceil (max_y + o->size / 2.0) + 1 - srcbuf_extent.y;  \n"
"                                                                              \n"
"  if (gegl_rectangle_intersect (&srcbuf_extent,                               \n"
"                                &srcbuf_extent,                               \n"
"                                gegl_buffer_get_extent (priv->buffer)))       \n"
"    {                                                                         \n"
"      /* we allocate a buffer, referred to as the source buffer, into which we\n"
"       * read the necessary portion of the input buffer, and consecutively    \n"
"       * write the stroke results.                                            \n"
"       */                                                                     \n"
"                                                                              \n"
"      srcbuf_stride = 2 * srcbuf_extent.width;                                \n"
"                                                                              \n"
"      /* the source buffer is padded at the back by enough elements to make   \n"
"       * pointers to out-of-bounds pixels, adjacent to the right and bottom   \n"
"       * edges of the buffer, valid; such pointers may be formed as part of   \n"
"       * sampling.  the value of these elements is irrelevant, as long as     \n"
"       * they're finite.                                                      \n"
"       */                                                                     \n"
"      srcbuf_padding = srcbuf_stride + 2;                                     \n"
"                                                                              \n"
"      srcbuf = g_new (gfloat, srcbuf_stride * srcbuf_extent.height +          \n"
"                              srcbuf_padding);                                \n"
"                                                                              \n"
"      /* fill the padding with (0, 0) vectors */                              \n"
"      memset (srcbuf + srcbuf_stride * srcbuf_extent.height,                  \n"
"              0, sizeof (gfloat) * srcbuf_padding);                           \n"
"                                                                              \n"
"      /* read the input data from the cached buffer */                        \n"
"      gegl_buffer_get (priv->buffer, &srcbuf_extent, 1.0, NULL,               \n"
"                       srcbuf, sizeof (gfloat) * srcbuf_stride,               \n"
"                       GEGL_ABYSS_NONE);                                      \n"
"                                                                              \n"
"      /* process the remaining stroke */                                      \n"
"      for (event = priv->remaining_stroke; event; event = event->next)        \n"
"        {                                                                     \n"
"          next = *(event->d.point);                                           \n"
"          dist = gegl_path_point_dist (&next, &prev);                         \n"
"          stamps = floor (dist / spacing) + 1;                                \n"
"                                                                              \n"
"          /* stroke the current segment, such that there's always a stamp at  \n"
"           * its final endpoint, and at positive integer multiples of         \n"
"           * `spacing` away from it.                                          \n"
"           */                                                                 \n"
"                                                                              \n"
"          if (stamps == 1)                                                    \n"
"            {                                                                 \n"
"              stamp (o,                                                       \n"
"                     srcbuf, srcbuf_stride, &srcbuf_extent,                   \n"
"                     next.x, next.y);                                         \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              for (i = 0; i < stamps; i++)                                    \n"
"                {                                                             \n"
"                  t = 1.0 - ((stamps - i - 1) * spacing) / dist;              \n"
"                                                                              \n"
"                  gegl_path_point_lerp (&lerp, &prev, &next, t);              \n"
"                  stamp (o,                                                   \n"
"                         srcbuf, srcbuf_stride, &srcbuf_extent,               \n"
"                         lerp.x, lerp.y);                                     \n"
"                }                                                             \n"
"            }                                                                 \n"
"                                                                              \n"
"          prev = next;                                                        \n"
"                                                                              \n"
"          /* append the current event to the processed stroke. */             \n"
"          processed_event        = g_slice_new (WarpPointList);               \n"
"          processed_event->point = next;                                      \n"
"                                                                              \n"
"          *priv->processed_stroke_tail = processed_event;                     \n"
"          priv->processed_stroke_tail  = &processed_event->next;              \n"
"        }                                                                     \n"
"                                                                              \n"
"      /* write the result back to the cached buffer */                        \n"
"      gegl_buffer_set (priv->buffer, &srcbuf_extent, 0, NULL,                 \n"
"                       srcbuf, sizeof (gfloat) * srcbuf_stride);              \n"
"                                                                              \n"
"      g_free (srcbuf);                                                        \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      /* if the remaining stroke is completely out of bounds, just append it to\n"
"       * the processed stroke.                                                \n"
"       */                                                                     \n"
"      for (event = priv->remaining_stroke; event; event = event->next)        \n"
"        {                                                                     \n"
"          next = *(event->d.point);                                           \n"
"                                                                              \n"
"          priv->last_x = next.x;                                              \n"
"          priv->last_y = next.y;                                              \n"
"                                                                              \n"
"          /* append the current event to the processed stroke. */             \n"
"          processed_event        = g_slice_new (WarpPointList);               \n"
"          processed_event->point = next;                                      \n"
"                                                                              \n"
"          *priv->processed_stroke_tail = processed_event;                     \n"
"          priv->processed_stroke_tail  = &processed_event->next;              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  *priv->processed_stroke_tail = NULL;                                        \n"
"  priv->remaining_stroke       = NULL;                                        \n"
"                                                                              \n"
"  /* pass the processed buffer as output */                                   \n"
"  gegl_operation_context_set_object (context,                                 \n"
"                                     \"output\", G_OBJECT (priv->buffer));    \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GObjectClass       *object_class    = G_OBJECT_CLASS (klass);               \n"
"  GeglOperationClass *operation_class = GEGL_OPERATION_CLASS (klass);         \n"
"                                                                              \n"
"  object_class->finalize                   = finalize;                        \n"
"  operation_class->attach                  = attach;                          \n"
"  operation_class->prepare                 = prepare;                         \n"
"  operation_class->get_required_for_output = get_required_for_output;         \n"
"  operation_class->process                 = process;                         \n"
"  operation_class->no_cache = TRUE; /* we're effectively doing the caching    \n"
"                                     * ourselves.                             \n"
"                                     */                                       \n"
"  operation_class->threaded = FALSE;                                          \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",               \"gegl:warp\",                                    \n"
"    \"categories\",         \"transform\",                                    \n"
"    \"title\",              _(\"Warp\"),                                      \n"
"    \"position-dependent\", \"true\",                                         \n"
"    \"description\", _(\"Compute a relative displacement mapping from a stroke\"),\n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"#endif                                                                        \n"
;
