/* GDK - The GIMP Drawing Kit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * BeOS Port
 * Copyright (C) 1999 EventLoop, Inc.
 *   Shawn T. Amundson <amundson@gtk.org>
 *   James Mitchell <mitchell@eventloop.com>
 *         
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-1999.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

#include "config.h"

#include <View.h>
#include <Window.h>
#include <Polygon.h>

#include <math.h>
#include <gdk/gdk.h>

#include "gdkconfig.h"
#include "gdkprivate.h"

#ifndef M_TWOPI
#define M_TWOPI         (2.0 * 3.14159265358979323846)
#endif


void
gdk_draw_point (GdkDrawable *drawable,
                GdkGC       *gc,
                gint         x,
                gint         y)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;
  BView *view;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;
  gc_private = (GdkGCPrivate*) gc;

  view = (BView*) drawable_private->xwindow;
  gdk_gc_predraw ((GdkBView*)view, gc_private);

  if (view->Window()->Lock())
    {
      view->StrokeLine(BPoint(x, y), BPoint(x, y));
      view->Window()->Unlock();
    }

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}

void
gdk_draw_line (GdkDrawable *drawable,
	       GdkGC       *gc,
	       gint         x1,
	       gint         y1,
	       gint         x2,
	       gint         y2)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;
  BView *view;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;
  gc_private = (GdkGCPrivate*) gc;

  view = (BView*) drawable_private->xwindow;
  gdk_gc_predraw ((GdkBView*)view, gc_private);
    
  if (view->Window()->Lock())
    {
      view->StrokeLine(BPoint(x1, y1), BPoint(x2, y2));
      view->Window()->Unlock();
    }

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}

void
gdk_draw_rectangle (GdkDrawable *drawable,
		    GdkGC       *gc,
		    gint         filled,
		    gint         x,
		    gint         y,
		    gint         width,
		    gint         height)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;
  BView *view;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;
  gc_private = (GdkGCPrivate*) gc;

  if (width == -1)
    width = drawable_private->width;
  if (height == -1)
    height = drawable_private->height;

  view = (BView*) drawable_private->xwindow;
  gdk_gc_predraw ((GdkBView*)view, gc_private);

  if (view->Window()->Lock())
    {
      if (filled)
        view->FillRect(BRect(x, y, x+width, y+height));
      else
        view->StrokeRect(BRect(x, y, x+width, y+height));
    
      view->Window()->Unlock();
    }

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}

void
gdk_draw_arc (GdkDrawable *drawable,
	      GdkGC       *gc,
	      gint         filled,
	      gint         x,
	      gint         y,
	      gint         width,
	      gint         height,
	      gint         angle1,
	      gint         angle2)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;
  BView *view;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;
  gc_private = (GdkGCPrivate*) gc;

  if (width == -1)
    width = drawable_private->width;
  if (height == -1)
    height = drawable_private->height;

  view = (BView*) drawable_private->xwindow;
  gdk_gc_predraw ((GdkBView*)view, gc_private);

  if (view->Window()->Lock())
    {
      if (filled)
        view->FillArc(BRect(x,y,x+width-1,y+height-1),(float)angle1,(float)angle2);
      else
        view->StrokeArc(BRect(x,y,x+width-1,y+height-1),(float)angle1,(float)angle2);
      view->Window()->Unlock();
    }

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}

void
gdk_draw_polygon (GdkDrawable *drawable,
		  GdkGC       *gc,
		  gint         filled,
		  GdkPoint    *points,
		  gint         npoints)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;
  BView *view;
  BPolygon *poly;
  int i;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;
  gc_private = (GdkGCPrivate*) gc;


  view = (BView*) drawable_private->xwindow;
  gdk_gc_predraw ((GdkBView*)view, gc_private);

  poly = new BPolygon();

  for (i = 0; i < npoints; i++)
    {
      poly->AddPoints(new BPoint(points[i].x, points[i].y), 1);
    }
  
  if ((points[0].x != points[npoints-1].x) ||
      (points[0].y != points[npoints-1].y)) 
    {
      poly->AddPoints(new BPoint(points[0].x, points[0].y), 1);
    }

  if (view->Window()->Lock())
    {
      if (filled)
        view->FillPolygon(poly, B_SOLID_HIGH);  
      else
        view->StrokePolygon(poly, TRUE, B_SOLID_HIGH);
 
      view->Window()->Unlock();
    }

  gdk_gc_postdraw ((GdkBView*)view, gc_private);

  delete poly;
}


/* gdk_draw_string
 */
void
gdk_draw_string (GdkDrawable *drawable,
		 GdkFont     *font,
		 GdkGC       *gc,
		 gint         x,
		 gint         y,
		 const gchar *string)
{
  gdk_draw_text (drawable, font, gc, x, y, string, strlen (string));
}

/* gdk_draw_text
 *
 * Modified by Li-Da Lho to draw 16 bits and Multibyte strings
 *
 * Interface changed: add "GdkFont *font" to specify font or fontset explicitely
 */
void
gdk_draw_text (GdkDrawable *drawable,
	       GdkFont     *font,
	       GdkGC       *gc,
	       gint         x,
	       gint         y,
	       const gchar *text,
	       gint         text_length)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;

  BView *view;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (font != NULL);
  g_return_if_fail (gc != NULL);
  g_return_if_fail (text != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;

  gc_private = (GdkGCPrivate*) gc;

  view = (BView*) drawable_private->xwindow;

  if (font->type == GDK_FONT_FONT)
    {
      gdk_gc_set_font(gc, font);
      gdk_gc_predraw ((GdkBView*)view, gc_private);

      if (view->Window()->Lock())
        {
	  view->SetDrawingMode(B_OP_OVER);
          view->MovePenTo(x, y); 
          view->DrawString(text, text_length);      
        
          view->Window()->Unlock();
        }

      gdk_gc_postdraw ((GdkBView*)view, gc_private);
    }
  else
    g_error ("undefined font type");
}


void
gdk_draw_text_wc (GdkDrawable	 *drawable,
		  GdkFont	 *font,
		  GdkGC		 *gc,
		  gint		  x,
		  gint		  y,
		  const GdkWChar *text,
		  gint		  text_length)
{
   gchar *string;

   string = gdk_wcstombs(text);

   gdk_draw_text(drawable, font, gc, x, y, string, strlen(string)); 
}

void
gdk_draw_pixmap (GdkDrawable *drawable,
		 GdkGC       *gc,
		 GdkPixmap   *src,
		 gint         xsrc,
		 gint         ysrc,
		 gint         xdest,
		 gint         ydest,
		 gint         width,
		 gint         height)
{
  GdkWindowPrivate *drawable_private;
  GdkWindowPrivate *src_private;
  GdkGCPrivate *gc_private;
  BView *view;
  BBitmap *pixmap;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (src != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  src_private = (GdkPixmapPrivate*) src;
  if (drawable_private->destroyed || src_private->destroyed)
    return;
  gc_private = (GdkGCPrivate*) gc;

  if (width == -1)
    width = src_private->width;
  if (height == -1)
    height = src_private->height;

  view = (BView*) drawable_private->xwindow;
  pixmap = (BBitmap*) src_private->pixmap;

  g_return_if_fail (view != NULL);

  if (pixmap == NULL)
    {
      /* Assume we are working with GdkWindows. */
      gdk_window_copy_area(drawable, gc, xdest, ydest, src, 
                           xsrc, ysrc, width, height);

      return;
    }

  gdk_gc_predraw ((GdkBView*)view, gc_private);
 
  if (view->Window()->Lock())
    {
      view->Sync ();
      view->DrawBitmap(pixmap, BRect(xsrc, ysrc, xsrc+width-1, ysrc+height-1),
		       BRect(xdest, ydest, xdest+width-1, ydest+height-1)); 
      view->Window()->Unlock();
    }

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}


void
gdk_draw_image (GdkDrawable *drawable,
		GdkGC       *gc,
		GdkImage    *image,
		gint         xsrc,
		gint         ysrc,
		gint         xdest,
		gint         ydest,
		gint         width,
		gint         height)
{
  GdkImagePrivate *image_private;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (image != NULL);
  g_return_if_fail (gc != NULL);

  image_private = (GdkImagePrivate*) image;

  g_return_if_fail (image_private->image_put != NULL);

  if (width == -1)
    width = image->width;
  if (height == -1)
    height = image->height;

  (* image_private->image_put) (drawable, gc, image, xsrc, ysrc,
				xdest, ydest, width, height);
}

void
gdk_draw_points (GdkDrawable *drawable,
		 GdkGC       *gc,
		 GdkPoint    *points,
		 gint         npoints)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;
  BView *view;
  int i;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail ((points != NULL) && (npoints > 0));
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;
  gc_private = (GdkGCPrivate*) gc;
  
  view = (BView*) drawable_private->xwindow;
  gdk_gc_predraw ((GdkBView*)view, gc_private);
 
  if (view->Window()->Lock())
    {
      for (i = 0; i < npoints; i++)
         view->StrokeLine(BPoint(points[i].x, points[i].y),
		          BPoint(points[i].x, points[i].y));

      view->Window()->Unlock();
    }        

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}

void
gdk_draw_segments (GdkDrawable *drawable,
		   GdkGC       *gc,
		   GdkSegment  *segs,
		   gint         nsegs)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;
  BView *view;
  int i;

  if (nsegs <= 0)
    return;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (segs != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  if (drawable_private->destroyed)
    return;
  gc_private = (GdkGCPrivate*) gc;

  view = (BView*) drawable_private->xwindow;
  gdk_gc_predraw ((GdkBView*)view, gc_private);

  if (view->Window()->Lock())
    {
      for (i = 0; i < nsegs; i++)
          view->StrokeLine(BPoint(segs[i].x1, segs[i].y1),
	  	           BPoint(segs[i].x2, segs[i].y2));
      view->Window()->Unlock();
    }

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}

void
gdk_draw_lines (GdkDrawable *drawable,
		GdkGC       *gc,
		GdkPoint    *points,
		gint         npoints)
{
  GdkWindowPrivate *drawable_private;
  GdkGCPrivate *gc_private;
  BView *view;
  int i;

  /* jsm: changed from 0 to 1. see below. This may affect behavior */

  if (npoints <= 1)
    return;

  g_return_if_fail (drawable != NULL);
  g_return_if_fail (points != NULL);
  g_return_if_fail (gc != NULL);

  drawable_private = (GdkWindowPrivate*) drawable;
  gc_private = (GdkGCPrivate*) gc;

  view = (BView*) drawable_private->xwindow;
  gdk_gc_predraw ((GdkBView*)view, gc_private);

  view->MovePenTo(points[0].x, points[0].y);

  for (i = 1; i < npoints; i++)
    view->StrokeLine(BPoint(points[i].x, points[i].y));

  gdk_gc_postdraw ((GdkBView*)view, gc_private);
}
