/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer_flat.h"


#define DATA   (&flat_buffer->data)
#define VALID  (flat_buffer->valid)
#define VALID2 (flat_buffer->validating)



static void
gimp_flat_buffer_delete (GimpBuffer * buffer)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  d_uninit (DATA);
  g_free (flat_buffer);
}


static gboolean
gimp_flat_buffer_alloc (GimpBuffer   *buffer,
                        GimpArea     *area,
                        Alloc         how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);
  gint n = (buffer->_x.image.size *
            buffer->_y.image.size *
            buffer->_z.bpp);

  area = NULL;
  
  switch (how)
    {
    case ALLOC_ALLOC:
      g_return_val_if_fail (d_alloc (DATA, n) == TRUE, FALSE);
      return TRUE;

    case ALLOC_UNALLOC:
      g_return_val_if_fail (d_unalloc (DATA) == TRUE, FALSE);
      VALID = FALSE;
      return TRUE;

    case ALLOC_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}


static gboolean
gimp_flat_buffer_validate (GimpBuffer  *buffer,
                           GimpArea    *area, 
                           Validate     how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  switch (how)
    {
    case VALIDATE_VALIDATE:
      if (VALID == TRUE)
        return TRUE;

      if (! d_is_alloced (DATA))
        {
          if (gimp_flat_buffer_alloc (buffer, NULL,
                                      ALLOC_ALLOC) != TRUE)
            {
              g_warning ("flatbuffer autoalloc failed");
              return FALSE;
            }
        }

      g_return_val_if_fail (d_write (DATA) == TRUE, FALSE);
      if (buffer->vfunc != NULL)
        {
          if (VALID2 == TRUE)
            {
              g_warning ("recursive validate");
            }
          else
            {
              GimpArea a;
              a.a.x = 0;
              a.a.y = 0;
              a.b.x = buffer->_x.image.size;
              a.b.y = buffer->_y.image.size;
              
              VALID2 = TRUE;
              buffer->vfunc (buffer, &a, guchar_d (DATA));
              VALID2 = FALSE;
            }
        }
      else
        {
          memset (guchar_d (DATA),
                  128,
                  (buffer->_x.image.size *
                   buffer->_y.image.size *
                   buffer->_z.bpp));
        }
      g_return_val_if_fail (d_release (DATA) == TRUE, FALSE);

      VALID = TRUE;
      return TRUE;


    case VALIDATE_INVALIDATE:
      g_return_val_if_fail (d_usecount (DATA) == 0, FALSE);
      VALID = FALSE;
      return TRUE;


    case VALIDATE_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}



static GimpMemPtr *
gimp_flat_buffer_use (GimpBuffer  *buffer,
                      GimpPoint   *tile, 
                      Use          how)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  if ((VALID != TRUE) && ((how == USE_READ) ||
                          (how == USE_UPDATE) ||
                          (how == USE_WRITE)))
    {
      if (gimp_flat_buffer_validate (buffer, NULL,
                                     VALIDATE_VALIDATE) != TRUE)
        {
          g_warning ("flatbuffer autovalidate failed");
          return NULL;
        }
    }

  switch (how)
    {
    case USE_READ:
      g_return_val_if_fail (d_read (DATA) == TRUE, NULL);
      break;

    case USE_UPDATE:
      g_return_val_if_fail (d_update (DATA) == TRUE, NULL);
      break;

    case USE_WRITE:
      g_return_val_if_fail (d_write (DATA) == TRUE, NULL);
      break;

    case USE_RELEASE:
      g_return_val_if_fail (d_release (DATA) == TRUE, NULL);
      break;

    case USE_NONE:
      break;
    }

  return DATA;
}


static gboolean
gimp_flat_buffer_query (GimpBuffer    *buffer,
                        GimpPoint     *tile)
{
  GimpFlatBuffer * flat_buffer = GIMP_FLAT_BUFFER (buffer);

  return VALID;
}



GimpFlatBuffer *
gimp_flat_buffer_new (Tag   tag,
                      gint  width,
                      gint  height)
{
  GimpFlatBuffer * flat_buffer;
  GimpBuffer * buffer;

  static GimpFlatBufferClass my_class =
  {
    {
      BUFFER_FLAT,
      gimp_flat_buffer_delete,
      gimp_flat_buffer_alloc,
      NULL, /* map */
      gimp_flat_buffer_validate,
      gimp_flat_buffer_use,
      gimp_flat_buffer_query
    }
  };

  flat_buffer = g_new (GimpFlatBuffer, 1);
  buffer = GIMP_BUFFER (flat_buffer);
  
  buffer->klass = (void*) &my_class;

  buffer->vfunc = NULL;
  buffer->vdata = NULL;
  
  buffer->_x.tile.count = 1;
  buffer->_x.tile.size = width;
  buffer->_x.image.size = width;
  buffer->_x.image.offset = 0;

  buffer->_y.tile.count = 1;
  buffer->_y.tile.size = height;
  buffer->_y.image.size = height;
  buffer->_y.image.offset = 0;

  buffer->_z.tag = tag;
  buffer->_z.bpp = tag_bytes (tag);

  d_init (DATA);
  VALID = FALSE;
  VALID2 = FALSE;

  buffer->qq_x = 0;
  buffer->qq_y = 0;

  return flat_buffer;
}


