/* Copyright 2016 The Chromium OS Authors. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 *
 * Register map for STM32F446 USB
 */

#ifndef __CHIP_STM32_USB_DWC_REGISTERS_H
#define __CHIP_STM32_USB_DWC_REGISTERS_H

/* Endpoint state */
struct dwc_usb_ep {
	int max_packet;
	int tx_fifo;

	int out_pending;
	int out_expected;
	uint8_t *out_data;
	uint8_t *out_databuffer;
	int out_databuffer_max;
	const struct deferred_data *rx_deferred;

	int in_packets;
	int in_pending;
	uint8_t *in_data;
	uint8_t *in_databuffer;
	int in_databuffer_max;
	const struct deferred_data *tx_deferred;
};

/* USB state */
enum dwc_usb_speed {
	USB_SPEED_FS = 0,
	USB_SPEED_HS,
};

enum dwc_usb_phy {
	USB_PHY_INTERNAL = 0,
	USB_PHY_ULPI,
};

struct dwc_usb {
	struct dwc_usb_ep *ep[USB_EP_COUNT];
	enum dwc_usb_speed speed;
	enum dwc_usb_phy phy_type;
	int dma_en;
	/* IRQ must be STM32_IRQ_OTG_FS / STM32_IRQ_OTG_HS */
	int irq;
};

extern struct dwc_usb_ep ep0_ctl;
extern struct dwc_usb usb_ctl;

/*
 * Added Alias Module Family Base Address to 0-instance Module Base Address
 * Simplify GBASE(mname) macro
 */
#define GC_MODULE_OFFSET         0x10000

#define GBASE(mname)      \
	GC_ ## mname ## _BASE_ADDR
#define GOFFSET(mname, rname)  \
	GC_ ## mname ## _ ## rname ## _OFFSET

#define GREG8(mname, rname) \
	REG8(GBASE(mname) + GOFFSET(mname, rname))
#define GREG32(mname, rname) \
	REG32(GBASE(mname) + GOFFSET(mname, rname))
#define GREG32_ADDR(mname, rname) \
	REG32_ADDR(GBASE(mname) + GOFFSET(mname, rname))
#define GWRITE(mname, rname, value) (GREG32(mname, rname) = (value))
#define GREAD(mname, rname)	    GREG32(mname, rname)

#define GFIELD_MASK(mname, rname, fname) \
	GC_ ## mname ## _ ## rname ## _ ## fname ## _MASK

#define GFIELD_LSB(mname, rname, fname)  \
	GC_ ## mname ## _ ## rname ## _ ## fname ## _LSB

#define GREAD_FIELD(mname, rname, fname) \
	((GREG32(mname, rname) & GFIELD_MASK(mname, rname, fname)) \
		>> GFIELD_LSB(mname, rname, fname))

#define GWRITE_FIELD(mname, rname, fname, fval) \
	(GREG32(mname, rname) = \
	((GREG32(mname, rname) & (~GFIELD_MASK(mname, rname, fname))) | \
	(((fval) << GFIELD_LSB(mname, rname, fname)) & \
		GFIELD_MASK(mname, rname, fname))))


#define GBASE_I(mname, i)     (GBASE(mname) + i*GC_MODULE_OFFSET)

#define GREG32_I(mname, i, rname) \
		REG32(GBASE_I(mname, i) + GOFFSET(mname, rname))

#define GREG32_ADDR_I(mname, i, rname) \
		REG32_ADDR(GBASE_I(mname, i) + GOFFSET(mname, rname))

#define GWRITE_I(mname, i, rname, value) (GREG32_I(mname, i, rname) = (value))
#define GREAD_I(mname, i, rname)	        GREG32_I(mname, i, rname)

#define GREAD_FIELD_I(mname, i, rname, fname) \
	((GREG32_I(mname, i, rname) & GFIELD_MASK(mname, rname, fname)) \
		>> GFIELD_LSB(mname, rname, fname))

#define GWRITE_FIELD_I(mname, i, rname, fname, fval) \
	(GREG32_I(mname, i, rname) = \
	((GREG32_I(mname, i, rname) & (~GFIELD_MASK(mname, rname, fname))) | \
	(((fval) << GFIELD_LSB(mname, rname, fname)) & \
		GFIELD_MASK(mname, rname, fname))))

/* Replace masked bits with val << lsb */
#define REG_WRITE_MLV(reg, mask, lsb, val) \
		(reg = ((reg & ~mask) | ((val << lsb) & mask)))


/* USB device controller */
#define GR_USB_REG(off)               REG32(GC_USB_BASE_ADDR + (off))
#define GR_USB_GOTGCTL                GR_USB_REG(GC_USB_GOTGCTL_OFFSET)
#define GR_USB_GOTGINT                GR_USB_REG(GC_USB_GOTGINT_OFFSET)
#define GR_USB_GAHBCFG                GR_USB_REG(GC_USB_GAHBCFG_OFFSET)
#define GR_USB_GUSBCFG                GR_USB_REG(GC_USB_GUSBCFG_OFFSET)
#define GR_USB_GRSTCTL                GR_USB_REG(GC_USB_GRSTCTL_OFFSET)
#define GR_USB_GINTSTS                GR_USB_REG(GC_USB_GINTSTS_OFFSET)
#define   GINTSTS(bit)                (1 << GC_USB_GINTSTS_ ## bit ## _LSB)
#define GR_USB_GINTMSK                GR_USB_REG(GC_USB_GINTMSK_OFFSET)
#define   GINTMSK(bit)                (1 << GC_USB_GINTMSK_ ## bit ## MSK_LSB)
#define GR_USB_GRXSTSR                GR_USB_REG(GC_USB_GRXSTSR_OFFSET)
#define GR_USB_GRXSTSP                GR_USB_REG(GC_USB_GRXSTSP_OFFSET)
#define GR_USB_GRXFSIZ                GR_USB_REG(GC_USB_GRXFSIZ_OFFSET)
#define GR_USB_GNPTXFSIZ              GR_USB_REG(GC_USB_GNPTXFSIZ_OFFSET)
/*#define GR_USB_GGPIO                  GR_USB_REG(GC_USB_GGPIO_OFFSET)*/
#define GR_USB_GCCFG                  GR_USB_REG(GC_USB_GCCFG_OFFSET)
#define  GCCFG_VBDEN            BIT(21)
#define  GCCFG_PWRDWN           BIT(16)
#define GR_USB_PCGCCTL                GR_USB_REG(GC_USB_PCGCCTL_OFFSET)

#define GR_USB_GSNPSID                GR_USB_REG(GC_USB_GSNPSID_OFFSET)
#define GR_USB_GHWCFG1                GR_USB_REG(GC_USB_GHWCFG1_OFFSET)
#define GR_USB_GHWCFG2                GR_USB_REG(GC_USB_GHWCFG2_OFFSET)
#define GR_USB_GHWCFG3                GR_USB_REG(GC_USB_GHWCFG3_OFFSET)
#define GR_USB_GHWCFG4                GR_USB_REG(GC_USB_GHWCFG4_OFFSET)
#define GR_USB_GDFIFOCFG              GR_USB_REG(GC_USB_GDFIFOCFG_OFFSET)
#define GR_USB_DIEPTXF(n)	\
	GR_USB_REG(GC_USB_DIEPTXF1_OFFSET - 4 + (n)*4)
#define GR_USB_DCFG                   GR_USB_REG(GC_USB_DCFG_OFFSET)
#define GR_USB_DCTL                   GR_USB_REG(GC_USB_DCTL_OFFSET)
#define GR_USB_DSTS                   GR_USB_REG(GC_USB_DSTS_OFFSET)
#define GR_USB_DIEPMSK                GR_USB_REG(GC_USB_DIEPMSK_OFFSET)
#define GR_USB_DOEPMSK                GR_USB_REG(GC_USB_DOEPMSK_OFFSET)
#define GR_USB_DAINT                  GR_USB_REG(GC_USB_DAINT_OFFSET)
#define GR_USB_DAINTMSK               GR_USB_REG(GC_USB_DAINTMSK_OFFSET)
#define  DAINT_INEP(ep)               (1 << (ep + GC_USB_DAINTMSK_INEPMSK0_LSB))
#define  DAINT_OUTEP(ep)	\
	(1 << (ep + GC_USB_DAINTMSK_OUTEPMSK0_LSB))
#define GR_USB_DTHRCTL                GR_USB_REG(GC_USB_DTHRCTL_OFFSET)
#define  DTHRCTL_TXTHRLEN_6           (0x40 << 2)
#define  DTHRCTL_RXTHRLEN_6           (0x40 << 17)
#define  DTHRCTL_RXTHREN              BIT(16)
#define  DTHRCTL_ISOTHREN             BIT(1)
#define  DTHRCTL_NONISOTHREN          BIT(0)
#define GR_USB_DIEPEMPMSK             GR_USB_REG(GC_USB_DIEPEMPMSK_OFFSET)

#define GR_USB_EPIREG(off, n)         GR_USB_REG(0x900 + (n) * 0x20 + (off))
#define GR_USB_EPOREG(off, n)         GR_USB_REG(0xb00 + (n) * 0x20 + (off))
#define GR_USB_DIEPCTL(n)             GR_USB_EPIREG(0x00, n)
#define GR_USB_DIEPINT(n)             GR_USB_EPIREG(0x08, n)
#define GR_USB_DIEPTSIZ(n)            GR_USB_EPIREG(0x10, n)
#define GR_USB_DIEPDMA(n)             GR_USB_EPIREG(0x14, n)
#define GR_USB_DTXFSTS(n)             GR_USB_EPIREG(0x18, n)
#define GR_USB_DIEPDMAB(n)            GR_USB_EPIREG(0x1c, n)
#define GR_USB_DOEPCTL(n)             GR_USB_EPOREG(0x00, n)
#define GR_USB_DOEPINT(n)             GR_USB_EPOREG(0x08, n)
#define GR_USB_DOEPTSIZ(n)            GR_USB_EPOREG(0x10, n)
#define GR_USB_DOEPDMA(n)             GR_USB_EPOREG(0x14, n)
#define GR_USB_DOEPDMAB(n)            GR_USB_EPOREG(0x1c, n)

#define GOTGCTL_BVALOEN               BIT(GC_USB_GOTGCTL_BVALIDOVEN_LSB)
#define GOTGCTL_BVALOVAL              BIT(7)

/* Bit 5 */
#define GAHBCFG_DMA_EN                BIT(GC_USB_GAHBCFG_DMAEN_LSB)
/* Bit 1 */
#define GAHBCFG_GLB_INTR_EN           BIT(GC_USB_GAHBCFG_GLBLINTRMSK_LSB)
/* HS Burst Len */
#define GAHBCFG_HBSTLEN_INCR4         (3 << GC_USB_GAHBCFG_HBSTLEN_LSB)
/* Bit 7 */
#define GAHBCFG_NP_TXF_EMP_LVL        (1 <<  GC_USB_GAHBCFG_NPTXFEMPLVL_LSB)
#define GAHBCFG_TXFELVL               GAHBCFG_NP_TXF_EMP_LVL
#define GAHBCFG_PTXFELVL              BIT(8)

#define GUSBCFG_TOUTCAL(n)            (((n) << GC_USB_GUSBCFG_TOUTCAL_LSB) \
				       & GC_USB_GUSBCFG_TOUTCAL_MASK)
#define GUSBCFG_USBTRDTIM(n)          (((n) << GC_USB_GUSBCFG_USBTRDTIM_LSB) \
				       & GC_USB_GUSBCFG_USBTRDTIM_MASK)
/* Force device mode */
#define GUSBCFG_FDMOD                 BIT(GC_USB_GUSBCFG_FDMOD_LSB)
#define GUSBCFG_PHYSEL                BIT(6)
#define GUSBCFG_SRPCAP                BIT(8)
#define GUSBCFG_HNPCAP                BIT(9)
#define GUSBCFG_ULPIFSLS              BIT(17)
#define GUSBCFG_ULPIAR                BIT(18)
#define GUSBCFG_ULPICSM               BIT(19)
#define GUSBCFG_ULPIEVBUSD            BIT(20)
#define GUSBCFG_ULPIEVBUSI            BIT(21)
#define GUSBCFG_TSDPS                 BIT(22)
#define GUSBCFG_PCCI                  BIT(23)
#define GUSBCFG_PTCI                  BIT(24)
#define GUSBCFG_ULPIIPD               BIT(25)
#define GUSBCFG_TSDPS                 BIT(22)


#define GRSTCTL_CSFTRST               BIT(GC_USB_GRSTCTL_CSFTRST_LSB)
#define GRSTCTL_AHBIDLE               BIT(GC_USB_GRSTCTL_AHBIDLE_LSB)
#define GRSTCTL_TXFFLSH               BIT(GC_USB_GRSTCTL_TXFFLSH_LSB)
#define GRSTCTL_RXFFLSH               BIT(GC_USB_GRSTCTL_RXFFLSH_LSB)
#define GRSTCTL_TXFNUM(n)	\
	(((n) << GC_USB_GRSTCTL_TXFNUM_LSB) & GC_USB_GRSTCTL_TXFNUM_MASK)

#define DCFG_DEVSPD_HSULPI            (0 << GC_USB_DCFG_DEVSPD_LSB)
#define DCFG_DEVSPD_FSULPI            BIT(GC_USB_DCFG_DEVSPD_LSB)
#define DCFG_DEVSPD_FS48              (3 << GC_USB_DCFG_DEVSPD_LSB)
#define DCFG_DEVADDR(a)	\
	(((a) << GC_USB_DCFG_DEVADDR_LSB) & GC_USB_DCFG_DEVADDR_MASK)
#define DCFG_NZLSOHSK                 BIT(GC_USB_DCFG_NZSTSOUTHSHK_LSB)

#define DCTL_SFTDISCON                BIT(GC_USB_DCTL_SFTDISCON_LSB)
#define DCTL_CGOUTNAK                 BIT(GC_USB_DCTL_CGOUTNAK_LSB)
#define DCTL_CGNPINNAK                BIT(GC_USB_DCTL_CGNPINNAK_LSB)
#define DCTL_PWRONPRGDONE             BIT(GC_USB_DCTL_PWRONPRGDONE_LSB)

/* Device Endpoint Common IN Interrupt Mask bits */
#define DIEPMSK_AHBERRMSK             BIT(GC_USB_DIEPMSK_AHBERRMSK_LSB)
#define DIEPMSK_BNAININTRMSK          BIT(GC_USB_DIEPMSK_BNAININTRMSK_LSB)
#define DIEPMSK_EPDISBLDMSK           BIT(GC_USB_DIEPMSK_EPDISBLDMSK_LSB)
#define DIEPMSK_INEPNAKEFFMSK         BIT(GC_USB_DIEPMSK_INEPNAKEFFMSK_LSB)
#define DIEPMSK_INTKNEPMISMSK         BIT(GC_USB_DIEPMSK_INTKNEPMISMSK_LSB)
#define DIEPMSK_INTKNTXFEMPMSK        BIT(GC_USB_DIEPMSK_INTKNTXFEMPMSK_LSB)
#define DIEPMSK_NAKMSK                BIT(GC_USB_DIEPMSK_NAKMSK_LSB)
#define DIEPMSK_TIMEOUTMSK            BIT(GC_USB_DIEPMSK_TIMEOUTMSK_LSB)
#define DIEPMSK_TXFIFOUNDRNMSK        BIT(GC_USB_DIEPMSK_TXFIFOUNDRNMSK_LSB)
#define DIEPMSK_XFERCOMPLMSK          BIT(GC_USB_DIEPMSK_XFERCOMPLMSK_LSB)

/* Device Endpoint Common OUT Interrupt Mask bits */
#define DOEPMSK_AHBERRMSK             BIT(GC_USB_DOEPMSK_AHBERRMSK_LSB)
#define DOEPMSK_BBLEERRMSK            BIT(GC_USB_DOEPMSK_BBLEERRMSK_LSB)
#define DOEPMSK_BNAOUTINTRMSK         BIT(GC_USB_DOEPMSK_BNAOUTINTRMSK_LSB)
#define DOEPMSK_EPDISBLDMSK           BIT(GC_USB_DOEPMSK_EPDISBLDMSK_LSB)
#define DOEPMSK_NAKMSK                BIT(GC_USB_DOEPMSK_NAKMSK_LSB)
#define DOEPMSK_NYETMSK               BIT(GC_USB_DOEPMSK_NYETMSK_LSB)
#define DOEPMSK_OUTPKTERRMSK          BIT(GC_USB_DOEPMSK_OUTPKTERRMSK_LSB)
#define DOEPMSK_OUTTKNEPDISMSK        BIT(GC_USB_DOEPMSK_OUTTKNEPDISMSK_LSB)
#define DOEPMSK_SETUPMSK              BIT(GC_USB_DOEPMSK_SETUPMSK_LSB)
#define DOEPMSK_STSPHSERCVDMSK        BIT(GC_USB_DOEPMSK_STSPHSERCVDMSK_LSB)
#define DOEPMSK_XFERCOMPLMSK          BIT(GC_USB_DOEPMSK_XFERCOMPLMSK_LSB)

/* Device Endpoint-n IN Interrupt Register bits */
#define DIEPINT_AHBERR             BIT(GC_USB_DIEPINT0_AHBERR_LSB)
#define DIEPINT_BBLEERR            BIT(GC_USB_DIEPINT0_BBLEERR_LSB)
#define DIEPINT_BNAINTR            BIT(GC_USB_DIEPINT0_BNAINTR_LSB)
#define DIEPINT_EPDISBLD           BIT(GC_USB_DIEPINT0_EPDISBLD_LSB)
#define DIEPINT_INEPNAKEFF         BIT(GC_USB_DIEPINT0_INEPNAKEFF_LSB)
#define DIEPINT_INTKNEPMIS         BIT(GC_USB_DIEPINT0_INTKNEPMIS_LSB)
#define DIEPINT_INTKNTXFEMP        BIT(GC_USB_DIEPINT0_INTKNTXFEMP_LSB)
#define DIEPINT_NAKINTRPT          BIT(GC_USB_DIEPINT0_NAKINTRPT_LSB)
#define DIEPINT_NYETINTRPT         BIT(GC_USB_DIEPINT0_NYETINTRPT_LSB)
#define DIEPINT_PKTDRPSTS          BIT(GC_USB_DIEPINT0_PKTDRPSTS_LSB)
#define DIEPINT_TIMEOUT            BIT(GC_USB_DIEPINT0_TIMEOUT_LSB)
#define DIEPINT_TXFEMP             BIT(GC_USB_DIEPINT0_TXFEMP_LSB)
#define DIEPINT_TXFIFOUNDRN        BIT(GC_USB_DIEPINT0_TXFIFOUNDRN_LSB)
#define DIEPINT_XFERCOMPL          BIT(GC_USB_DIEPINT0_XFERCOMPL_LSB)

/* Device Endpoint-n OUT Interrupt Register bits */
#define DOEPINT_AHBERR             BIT(GC_USB_DOEPINT0_AHBERR_LSB)
#define DOEPINT_BACK2BACKSETUP     BIT(GC_USB_DOEPINT0_BACK2BACKSETUP_LSB)
#define DOEPINT_BBLEERR            BIT(GC_USB_DOEPINT0_BBLEERR_LSB)
#define DOEPINT_BNAINTR            BIT(GC_USB_DOEPINT0_BNAINTR_LSB)
#define DOEPINT_EPDISBLD           BIT(GC_USB_DOEPINT0_EPDISBLD_LSB)
#define DOEPINT_NAKINTRPT          BIT(GC_USB_DOEPINT0_NAKINTRPT_LSB)
#define DOEPINT_NYETINTRPT         BIT(GC_USB_DOEPINT0_NYETINTRPT_LSB)
#define DOEPINT_OUTPKTERR          BIT(GC_USB_DOEPINT0_OUTPKTERR_LSB)
#define DOEPINT_OUTTKNEPDIS        BIT(GC_USB_DOEPINT0_OUTTKNEPDIS_LSB)
#define DOEPINT_PKTDRPSTS          BIT(GC_USB_DOEPINT0_PKTDRPSTS_LSB)
#define DOEPINT_SETUP              BIT(GC_USB_DOEPINT0_SETUP_LSB)
#define DOEPINT_STSPHSERCVD        BIT(GC_USB_DOEPINT0_STSPHSERCVD_LSB)
#define DOEPINT_STUPPKTRCVD        BIT(GC_USB_DOEPINT0_STUPPKTRCVD_LSB)
#define DOEPINT_XFERCOMPL          BIT(GC_USB_DOEPINT0_XFERCOMPL_LSB)

#define DXEPCTL_EPTYPE_CTRL           (0 << GC_USB_DIEPCTL0_EPTYPE_LSB)
#define DXEPCTL_EPTYPE_ISO            (1 << GC_USB_DIEPCTL0_EPTYPE_LSB)
#define DXEPCTL_EPTYPE_BULK           (2 << GC_USB_DIEPCTL0_EPTYPE_LSB)
#define DXEPCTL_EPTYPE_INT            (3 << GC_USB_DIEPCTL0_EPTYPE_LSB)
#define DXEPCTL_EPTYPE_MASK           GC_USB_DIEPCTL0_EPTYPE_MASK
#define DXEPCTL_TXFNUM(n)             ((n) << GC_USB_DIEPCTL1_TXFNUM_LSB)
#define DXEPCTL_STALL                 BIT(GC_USB_DIEPCTL0_STALL_LSB)
#define DXEPCTL_CNAK                  BIT(GC_USB_DIEPCTL0_CNAK_LSB)
#define DXEPCTL_DPID                  BIT(GC_USB_DIEPCTL0_DPID_LSB)
#define DXEPCTL_SNAK                  BIT(GC_USB_DIEPCTL0_SNAK_LSB)
#define DXEPCTL_NAKSTS                BIT(GC_USB_DIEPCTL0_NAKSTS_LSB)
#define DXEPCTL_EPENA                 BIT(GC_USB_DIEPCTL0_EPENA_LSB)
#define DXEPCTL_EPDIS                 BIT(GC_USB_DIEPCTL0_EPDIS_LSB)
#define DXEPCTL_USBACTEP              BIT(GC_USB_DIEPCTL0_USBACTEP_LSB)
#define DXEPCTL_MPS64                 (0 << GC_USB_DIEPCTL0_MPS_LSB)
#define DXEPCTL_MPS(cnt)              ((cnt) << GC_USB_DIEPCTL1_MPS_LSB)

#define DXEPTSIZ_SUPCNT(n)            ((n) << GC_USB_DOEPTSIZ0_SUPCNT_LSB)
#define DXEPTSIZ_PKTCNT(n)            ((n) << GC_USB_DIEPTSIZ0_PKTCNT_LSB)
#define DXEPTSIZ_XFERSIZE(n)          ((n) << GC_USB_DIEPTSIZ0_XFERSIZE_LSB)

#define DOEPDMA_BS_HOST_RDY           (0 << 30)
#define DOEPDMA_BS_DMA_BSY            (1 << 30)
#define DOEPDMA_BS_DMA_DONE           (2 << 30)
#define DOEPDMA_BS_HOST_BSY           (3 << 30)
#define DOEPDMA_BS_MASK               (3 << 30)
#define DOEPDMA_RXSTS_MASK            (3 << 28)
#define DOEPDMA_LAST                  BIT(27)
#define DOEPDMA_SP                    BIT(26)
#define DOEPDMA_IOC                   BIT(25)
#define DOEPDMA_SR                    BIT(24)
#define DOEPDMA_MTRF                  BIT(23)
#define DOEPDMA_NAK                   BIT(16)
#define DOEPDMA_RXBYTES(n)            (((n) & 0xFFFF) << 0)
#define DOEPDMA_RXBYTES_MASK          (0xFFFF << 0)

#define DIEPDMA_BS_HOST_RDY           (0 << 30)
#define DIEPDMA_BS_DMA_BSY            (1 << 30)
#define DIEPDMA_BS_DMA_DONE           (2 << 30)
#define DIEPDMA_BS_HOST_BSY           (3 << 30)
#define DIEPDMA_BS_MASK               (3 << 30)
#define DIEPDMA_TXSTS_MASK            (3 << 28)
#define DIEPDMA_LAST                  BIT(27)
#define DIEPDMA_SP                    BIT(26)
#define DIEPDMA_IOC                   BIT(25)
#define DIEPDMA_TXBYTES(n)            (((n) & 0xFFFF) << 0)
#define DIEPDMA_TXBYTES_MASK          (0xFFFF << 0)



/* Register defs referenced from DWC block in CR50. This is not a native
 * ST block, so we'll use this modified regdefs list.
 */

#define GC_USB_FS_BASE_ADDR                        0x50000000
#define GC_USB_HS_BASE_ADDR                        0x40040000
#ifdef CONFIG_USB_DWC_FS
#define GC_USB_BASE_ADDR                         GC_USB_FS_BASE_ADDR
#else
#define GC_USB_BASE_ADDR                         GC_USB_HS_BASE_ADDR
#endif

#define GC_USB_GOTGCTL_OFFSET                    0x0
#define GC_USB_GOTGCTL_DEFAULT                   0x0
#define GC_USB_GOTGINT_OFFSET                    0x4
#define GC_USB_GOTGINT_DEFAULT                   0x0
#define GC_USB_GAHBCFG_OFFSET                    0x8
#define GC_USB_GAHBCFG_DEFAULT                   0x0
#define GC_USB_GUSBCFG_OFFSET                    0xc
#define GC_USB_GUSBCFG_DEFAULT                   0x0
#define GC_USB_GRSTCTL_OFFSET                    0x10
#define GC_USB_GRSTCTL_DEFAULT                   0x0
#define GC_USB_GINTSTS_OFFSET                    0x14
#define GC_USB_GINTSTS_DEFAULT                   0x0
#define GC_USB_GINTMSK_OFFSET                    0x18
#define GC_USB_GINTMSK_DEFAULT                   0x0
#define GC_USB_GRXSTSR_OFFSET                    0x1c
#define GC_USB_GRXSTSR_DEFAULT                   0x0
#define GC_USB_GRXSTSP_OFFSET                    0x20
#define GC_USB_GRXSTSP_DEFAULT                   0x0
#define GC_USB_GRXFSIZ_OFFSET                    0x24
#define GC_USB_GRXFSIZ_DEFAULT                   0x0
#define GC_USB_GNPTXFSIZ_OFFSET                  0x28
#define GC_USB_GNPTXFSIZ_DEFAULT                 0x0

#define GC_USB_GCCFG_OFFSET                      0x38
#define GC_USB_GCCFG_DEFAULT                     0x0
#define GC_USB_GUID_OFFSET                       0x3c
#define GC_USB_GUID_DEFAULT                      0x0
#define GC_USB_GSNPSID_OFFSET                    0x40
#define GC_USB_GSNPSID_DEFAULT                   0x0
#define GC_USB_GHWCFG1_OFFSET                    0x44
#define GC_USB_GHWCFG1_DEFAULT                   0x0
#define GC_USB_GHWCFG2_OFFSET                    0x48
#define GC_USB_GHWCFG2_DEFAULT                   0x0
#define GC_USB_GHWCFG3_OFFSET                    0x4c
#define GC_USB_GHWCFG3_DEFAULT                   0x0
#define GC_USB_GHWCFG4_OFFSET                    0x50
#define GC_USB_GHWCFG4_DEFAULT                   0x0
#define GC_USB_GDFIFOCFG_OFFSET                  0x5c
#define GC_USB_GDFIFOCFG_DEFAULT                 0x0
#define GC_USB_DIEPTXF1_OFFSET                   0x104
#define GC_USB_DIEPTXF1_DEFAULT                  0x1000
#define GC_USB_DIEPTXF2_OFFSET                   0x108
#define GC_USB_DIEPTXF2_DEFAULT                  0x0
#define GC_USB_DIEPTXF3_OFFSET                   0x10c
#define GC_USB_DIEPTXF3_DEFAULT                  0x0
#define GC_USB_DIEPTXF4_OFFSET                   0x110
#define GC_USB_DIEPTXF4_DEFAULT                  0x0
#define GC_USB_DIEPTXF5_OFFSET                   0x114
#define GC_USB_DIEPTXF5_DEFAULT                  0x0
#define GC_USB_DIEPTXF6_OFFSET                   0x118
#define GC_USB_DIEPTXF6_DEFAULT                  0x0
#define GC_USB_DIEPTXF7_OFFSET                   0x11c
#define GC_USB_DIEPTXF7_DEFAULT                  0x0
#define GC_USB_DIEPTXF8_OFFSET                   0x120
#define GC_USB_DIEPTXF8_DEFAULT                  0x0
#define GC_USB_DIEPTXF9_OFFSET                   0x124
#define GC_USB_DIEPTXF9_DEFAULT                  0x0
#define GC_USB_DIEPTXF10_OFFSET                  0x128
#define GC_USB_DIEPTXF10_DEFAULT                 0x0
#define GC_USB_DIEPTXF11_OFFSET                  0x12c
#define GC_USB_DIEPTXF11_DEFAULT                 0x0
#define GC_USB_DIEPTXF12_OFFSET                  0x130
#define GC_USB_DIEPTXF12_DEFAULT                 0x0
#define GC_USB_DIEPTXF13_OFFSET                  0x134
#define GC_USB_DIEPTXF13_DEFAULT                 0x0
#define GC_USB_DIEPTXF14_OFFSET                  0x138
#define GC_USB_DIEPTXF14_DEFAULT                 0x0
#define GC_USB_DIEPTXF15_OFFSET                  0x13c
#define GC_USB_DIEPTXF15_DEFAULT                 0x0
#define GC_USB_DCFG_OFFSET                       0x800
#define GC_USB_DCFG_DEFAULT                      0x8000000
#define GC_USB_DCTL_OFFSET                       0x804
#define GC_USB_DCTL_DEFAULT                      0x0
#define GC_USB_DSTS_OFFSET                       0x808
#define GC_USB_DSTS_DEFAULT                      0x0
#define GC_USB_DIEPMSK_OFFSET                    0x810
#define GC_USB_DIEPMSK_DEFAULT                   0x80
#define GC_USB_DOEPMSK_OFFSET                    0x814
#define GC_USB_DOEPMSK_DEFAULT                   0x0
#define GC_USB_DAINT_OFFSET                      0x818
#define GC_USB_DAINT_DEFAULT                     0x0
#define GC_USB_DAINTMSK_OFFSET                   0x81c
#define GC_USB_DAINTMSK_DEFAULT                  0x0
#define GC_USB_DVBUSDIS_OFFSET                   0x828
#define GC_USB_DVBUSDIS_DEFAULT                  0x0
#define GC_USB_DVBUSPULSE_OFFSET                 0x82c
#define GC_USB_DVBUSPULSE_DEFAULT                0x0
#define GC_USB_DTHRCTL_OFFSET                    0x830
#define GC_USB_DTHRCTL_DEFAULT                   0x0
#define GC_USB_DIEPEMPMSK_OFFSET                 0x834
#define GC_USB_DIEPEMPMSK_DEFAULT                0x0
#define GC_USB_DIEPCTL0_OFFSET                   0x900
#define GC_USB_DIEPCTL0_DEFAULT                  0x0
#define GC_USB_DIEPINT0_OFFSET                   0x908
#define GC_USB_DIEPINT0_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ0_OFFSET                  0x910
#define GC_USB_DIEPTSIZ0_DEFAULT                 0x0
#define GC_USB_DIEPDMA0_OFFSET                   0x914
#define GC_USB_DIEPDMA0_DEFAULT                  0x0
#define GC_USB_DTXFSTS0_OFFSET                   0x918
#define GC_USB_DTXFSTS0_DEFAULT                  0x0
#define GC_USB_DIEPDMAB0_OFFSET                  0x91c
#define GC_USB_DIEPDMAB0_DEFAULT                 0x0
#define GC_USB_DIEPCTL1_OFFSET                   0x920
#define GC_USB_DIEPCTL1_DEFAULT                  0x0
#define GC_USB_DIEPINT1_OFFSET                   0x928
#define GC_USB_DIEPINT1_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ1_OFFSET                  0x930
#define GC_USB_DIEPTSIZ1_DEFAULT                 0x0
#define GC_USB_DIEPDMA1_OFFSET                   0x934
#define GC_USB_DIEPDMA1_DEFAULT                  0x0
#define GC_USB_DTXFSTS1_OFFSET                   0x938
#define GC_USB_DTXFSTS1_DEFAULT                  0x0
#define GC_USB_DIEPDMAB1_OFFSET                  0x93c
#define GC_USB_DIEPDMAB1_DEFAULT                 0x0
#define GC_USB_DIEPCTL2_OFFSET                   0x940
#define GC_USB_DIEPCTL2_DEFAULT                  0x0
#define GC_USB_DIEPINT2_OFFSET                   0x948
#define GC_USB_DIEPINT2_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ2_OFFSET                  0x950
#define GC_USB_DIEPTSIZ2_DEFAULT                 0x0
#define GC_USB_DIEPDMA2_OFFSET                   0x954
#define GC_USB_DIEPDMA2_DEFAULT                  0x0
#define GC_USB_DTXFSTS2_OFFSET                   0x958
#define GC_USB_DTXFSTS2_DEFAULT                  0x0
#define GC_USB_DIEPDMAB2_OFFSET                  0x95c
#define GC_USB_DIEPDMAB2_DEFAULT                 0x0
#define GC_USB_DIEPCTL3_OFFSET                   0x960
#define GC_USB_DIEPCTL3_DEFAULT                  0x0
#define GC_USB_DIEPINT3_OFFSET                   0x968
#define GC_USB_DIEPINT3_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ3_OFFSET                  0x970
#define GC_USB_DIEPTSIZ3_DEFAULT                 0x0
#define GC_USB_DIEPDMA3_OFFSET                   0x974
#define GC_USB_DIEPDMA3_DEFAULT                  0x0
#define GC_USB_DTXFSTS3_OFFSET                   0x978
#define GC_USB_DTXFSTS3_DEFAULT                  0x0
#define GC_USB_DIEPDMAB3_OFFSET                  0x97c
#define GC_USB_DIEPDMAB3_DEFAULT                 0x0
#define GC_USB_DIEPCTL4_OFFSET                   0x980
#define GC_USB_DIEPCTL4_DEFAULT                  0x0
#define GC_USB_DIEPINT4_OFFSET                   0x988
#define GC_USB_DIEPINT4_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ4_OFFSET                  0x990
#define GC_USB_DIEPTSIZ4_DEFAULT                 0x0
#define GC_USB_DIEPDMA4_OFFSET                   0x994
#define GC_USB_DIEPDMA4_DEFAULT                  0x0
#define GC_USB_DTXFSTS4_OFFSET                   0x998
#define GC_USB_DTXFSTS4_DEFAULT                  0x0
#define GC_USB_DIEPDMAB4_OFFSET                  0x99c
#define GC_USB_DIEPDMAB4_DEFAULT                 0x0
#define GC_USB_DIEPCTL5_OFFSET                   0x9a0
#define GC_USB_DIEPCTL5_DEFAULT                  0x0
#define GC_USB_DIEPINT5_OFFSET                   0x9a8
#define GC_USB_DIEPINT5_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ5_OFFSET                  0x9b0
#define GC_USB_DIEPTSIZ5_DEFAULT                 0x0
#define GC_USB_DIEPDMA5_OFFSET                   0x9b4
#define GC_USB_DIEPDMA5_DEFAULT                  0x0
#define GC_USB_DTXFSTS5_OFFSET                   0x9b8
#define GC_USB_DTXFSTS5_DEFAULT                  0x0
#define GC_USB_DIEPDMAB5_OFFSET                  0x9bc
#define GC_USB_DIEPDMAB5_DEFAULT                 0x0
#define GC_USB_DIEPCTL6_OFFSET                   0x9c0
#define GC_USB_DIEPCTL6_DEFAULT                  0x0
#define GC_USB_DIEPINT6_OFFSET                   0x9c8
#define GC_USB_DIEPINT6_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ6_OFFSET                  0x9d0
#define GC_USB_DIEPTSIZ6_DEFAULT                 0x0
#define GC_USB_DIEPDMA6_OFFSET                   0x9d4
#define GC_USB_DIEPDMA6_DEFAULT                  0x0
#define GC_USB_DTXFSTS6_OFFSET                   0x9d8
#define GC_USB_DTXFSTS6_DEFAULT                  0x0
#define GC_USB_DIEPDMAB6_OFFSET                  0x9dc
#define GC_USB_DIEPDMAB6_DEFAULT                 0x0
#define GC_USB_DIEPCTL7_OFFSET                   0x9e0
#define GC_USB_DIEPCTL7_DEFAULT                  0x0
#define GC_USB_DIEPINT7_OFFSET                   0x9e8
#define GC_USB_DIEPINT7_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ7_OFFSET                  0x9f0
#define GC_USB_DIEPTSIZ7_DEFAULT                 0x0
#define GC_USB_DIEPDMA7_OFFSET                   0x9f4
#define GC_USB_DIEPDMA7_DEFAULT                  0x0
#define GC_USB_DTXFSTS7_OFFSET                   0x9f8
#define GC_USB_DTXFSTS7_DEFAULT                  0x0
#define GC_USB_DIEPDMAB7_OFFSET                  0x9fc
#define GC_USB_DIEPDMAB7_DEFAULT                 0x0
#define GC_USB_DIEPCTL8_OFFSET                   0xa00
#define GC_USB_DIEPCTL8_DEFAULT                  0x0
#define GC_USB_DIEPINT8_OFFSET                   0xa08
#define GC_USB_DIEPINT8_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ8_OFFSET                  0xa10
#define GC_USB_DIEPTSIZ8_DEFAULT                 0x0
#define GC_USB_DIEPDMA8_OFFSET                   0xa14
#define GC_USB_DIEPDMA8_DEFAULT                  0x0
#define GC_USB_DTXFSTS8_OFFSET                   0xa18
#define GC_USB_DTXFSTS8_DEFAULT                  0x0
#define GC_USB_DIEPDMAB8_OFFSET                  0xa1c
#define GC_USB_DIEPDMAB8_DEFAULT                 0x0
#define GC_USB_DIEPCTL9_OFFSET                   0xa20
#define GC_USB_DIEPCTL9_DEFAULT                  0x0
#define GC_USB_DIEPINT9_OFFSET                   0xa28
#define GC_USB_DIEPINT9_DEFAULT                  0x0
#define GC_USB_DIEPTSIZ9_OFFSET                  0xa30
#define GC_USB_DIEPTSIZ9_DEFAULT                 0x0
#define GC_USB_DIEPDMA9_OFFSET                   0xa34
#define GC_USB_DIEPDMA9_DEFAULT                  0x0
#define GC_USB_DTXFSTS9_OFFSET                   0xa38
#define GC_USB_DTXFSTS9_DEFAULT                  0x0
#define GC_USB_DIEPDMAB9_OFFSET                  0xa3c
#define GC_USB_DIEPDMAB9_DEFAULT                 0x0
#define GC_USB_DIEPCTL10_OFFSET                  0xa40
#define GC_USB_DIEPCTL10_DEFAULT                 0x0
#define GC_USB_DIEPINT10_OFFSET                  0xa48
#define GC_USB_DIEPINT10_DEFAULT                 0x0
#define GC_USB_DIEPTSIZ10_OFFSET                 0xa50
#define GC_USB_DIEPTSIZ10_DEFAULT                0x0
#define GC_USB_DIEPDMA10_OFFSET                  0xa54
#define GC_USB_DIEPDMA10_DEFAULT                 0x0
#define GC_USB_DTXFSTS10_OFFSET                  0xa58
#define GC_USB_DTXFSTS10_DEFAULT                 0x0
#define GC_USB_DIEPDMAB10_OFFSET                 0xa5c
#define GC_USB_DIEPDMAB10_DEFAULT                0x0
#define GC_USB_DIEPCTL11_OFFSET                  0xa60
#define GC_USB_DIEPCTL11_DEFAULT                 0x0
#define GC_USB_DIEPINT11_OFFSET                  0xa68
#define GC_USB_DIEPINT11_DEFAULT                 0x0
#define GC_USB_DIEPTSIZ11_OFFSET                 0xa70
#define GC_USB_DIEPTSIZ11_DEFAULT                0x0
#define GC_USB_DIEPDMA11_OFFSET                  0xa74
#define GC_USB_DIEPDMA11_DEFAULT                 0x0
#define GC_USB_DTXFSTS11_OFFSET                  0xa78
#define GC_USB_DTXFSTS11_DEFAULT                 0x0
#define GC_USB_DIEPDMAB11_OFFSET                 0xa7c
#define GC_USB_DIEPDMAB11_DEFAULT                0x0
#define GC_USB_DIEPCTL12_OFFSET                  0xa80
#define GC_USB_DIEPCTL12_DEFAULT                 0x0
#define GC_USB_DIEPINT12_OFFSET                  0xa88
#define GC_USB_DIEPINT12_DEFAULT                 0x0
#define GC_USB_DIEPTSIZ12_OFFSET                 0xa90
#define GC_USB_DIEPTSIZ12_DEFAULT                0x0
#define GC_USB_DIEPDMA12_OFFSET                  0xa94
#define GC_USB_DIEPDMA12_DEFAULT                 0x0
#define GC_USB_DTXFSTS12_OFFSET                  0xa98
#define GC_USB_DTXFSTS12_DEFAULT                 0x0
#define GC_USB_DIEPDMAB12_OFFSET                 0xa9c
#define GC_USB_DIEPDMAB12_DEFAULT                0x0
#define GC_USB_DIEPCTL13_OFFSET                  0xaa0
#define GC_USB_DIEPCTL13_DEFAULT                 0x0
#define GC_USB_DIEPINT13_OFFSET                  0xaa8
#define GC_USB_DIEPINT13_DEFAULT                 0x0
#define GC_USB_DIEPTSIZ13_OFFSET                 0xab0
#define GC_USB_DIEPTSIZ13_DEFAULT                0x0
#define GC_USB_DIEPDMA13_OFFSET                  0xab4
#define GC_USB_DIEPDMA13_DEFAULT                 0x0
#define GC_USB_DTXFSTS13_OFFSET                  0xab8
#define GC_USB_DTXFSTS13_DEFAULT                 0x0
#define GC_USB_DIEPDMAB13_OFFSET                 0xabc
#define GC_USB_DIEPDMAB13_DEFAULT                0x0
#define GC_USB_DIEPCTL14_OFFSET                  0xac0
#define GC_USB_DIEPCTL14_DEFAULT                 0x0
#define GC_USB_DIEPINT14_OFFSET                  0xac8
#define GC_USB_DIEPINT14_DEFAULT                 0x0
#define GC_USB_DIEPTSIZ14_OFFSET                 0xad0
#define GC_USB_DIEPTSIZ14_DEFAULT                0x0
#define GC_USB_DIEPDMA14_OFFSET                  0xad4
#define GC_USB_DIEPDMA14_DEFAULT                 0x0
#define GC_USB_DTXFSTS14_OFFSET                  0xad8
#define GC_USB_DTXFSTS14_DEFAULT                 0x0
#define GC_USB_DIEPDMAB14_OFFSET                 0xadc
#define GC_USB_DIEPDMAB14_DEFAULT                0x0
#define GC_USB_DIEPCTL15_OFFSET                  0xae0
#define GC_USB_DIEPCTL15_DEFAULT                 0x0
#define GC_USB_DIEPINT15_OFFSET                  0xae8
#define GC_USB_DIEPINT15_DEFAULT                 0x0
#define GC_USB_DIEPTSIZ15_OFFSET                 0xaf0
#define GC_USB_DIEPTSIZ15_DEFAULT                0x0
#define GC_USB_DIEPDMA15_OFFSET                  0xaf4
#define GC_USB_DIEPDMA15_DEFAULT                 0x0
#define GC_USB_DTXFSTS15_OFFSET                  0xaf8
#define GC_USB_DTXFSTS15_DEFAULT                 0x0
#define GC_USB_DIEPDMAB15_OFFSET                 0xafc
#define GC_USB_DIEPDMAB15_DEFAULT                0x0
#define GC_USB_DOEPCTL0_OFFSET                   0xb00
#define GC_USB_DOEPCTL0_DEFAULT                  0x0
#define GC_USB_DOEPINT0_OFFSET                   0xb08
#define GC_USB_DOEPINT0_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ0_OFFSET                  0xb10
#define GC_USB_DOEPTSIZ0_DEFAULT                 0x0
#define GC_USB_DOEPDMA0_OFFSET                   0xb14
#define GC_USB_DOEPDMA0_DEFAULT                  0x0
#define GC_USB_DOEPDMAB0_OFFSET                  0xb1c
#define GC_USB_DOEPDMAB0_DEFAULT                 0x0
#define GC_USB_DOEPCTL1_OFFSET                   0xb20
#define GC_USB_DOEPCTL1_DEFAULT                  0x0
#define GC_USB_DOEPINT1_OFFSET                   0xb28
#define GC_USB_DOEPINT1_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ1_OFFSET                  0xb30
#define GC_USB_DOEPTSIZ1_DEFAULT                 0x0
#define GC_USB_DOEPDMA1_OFFSET                   0xb34
#define GC_USB_DOEPDMA1_DEFAULT                  0x0
#define GC_USB_DOEPDMAB1_OFFSET                  0xb3c
#define GC_USB_DOEPDMAB1_DEFAULT                 0x0
#define GC_USB_DOEPCTL2_OFFSET                   0xb40
#define GC_USB_DOEPCTL2_DEFAULT                  0x0
#define GC_USB_DOEPINT2_OFFSET                   0xb48
#define GC_USB_DOEPINT2_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ2_OFFSET                  0xb50
#define GC_USB_DOEPTSIZ2_DEFAULT                 0x0
#define GC_USB_DOEPDMA2_OFFSET                   0xb54
#define GC_USB_DOEPDMA2_DEFAULT                  0x0
#define GC_USB_DOEPDMAB2_OFFSET                  0xb5c
#define GC_USB_DOEPDMAB2_DEFAULT                 0x0
#define GC_USB_DOEPCTL3_OFFSET                   0xb60
#define GC_USB_DOEPCTL3_DEFAULT                  0x0
#define GC_USB_DOEPINT3_OFFSET                   0xb68
#define GC_USB_DOEPINT3_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ3_OFFSET                  0xb70
#define GC_USB_DOEPTSIZ3_DEFAULT                 0x0
#define GC_USB_DOEPDMA3_OFFSET                   0xb74
#define GC_USB_DOEPDMA3_DEFAULT                  0x0
#define GC_USB_DOEPDMAB3_OFFSET                  0xb7c
#define GC_USB_DOEPDMAB3_DEFAULT                 0x0
#define GC_USB_DOEPCTL4_OFFSET                   0xb80
#define GC_USB_DOEPCTL4_DEFAULT                  0x0
#define GC_USB_DOEPINT4_OFFSET                   0xb88
#define GC_USB_DOEPINT4_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ4_OFFSET                  0xb90
#define GC_USB_DOEPTSIZ4_DEFAULT                 0x0
#define GC_USB_DOEPDMA4_OFFSET                   0xb94
#define GC_USB_DOEPDMA4_DEFAULT                  0x0
#define GC_USB_DOEPDMAB4_OFFSET                  0xb9c
#define GC_USB_DOEPDMAB4_DEFAULT                 0x0
#define GC_USB_DOEPCTL5_OFFSET                   0xba0
#define GC_USB_DOEPCTL5_DEFAULT                  0x0
#define GC_USB_DOEPINT5_OFFSET                   0xba8
#define GC_USB_DOEPINT5_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ5_OFFSET                  0xbb0
#define GC_USB_DOEPTSIZ5_DEFAULT                 0x0
#define GC_USB_DOEPDMA5_OFFSET                   0xbb4
#define GC_USB_DOEPDMA5_DEFAULT                  0x0
#define GC_USB_DOEPDMAB5_OFFSET                  0xbbc
#define GC_USB_DOEPDMAB5_DEFAULT                 0x0
#define GC_USB_DOEPCTL6_OFFSET                   0xbc0
#define GC_USB_DOEPCTL6_DEFAULT                  0x0
#define GC_USB_DOEPINT6_OFFSET                   0xbc8
#define GC_USB_DOEPINT6_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ6_OFFSET                  0xbd0
#define GC_USB_DOEPTSIZ6_DEFAULT                 0x0
#define GC_USB_DOEPDMA6_OFFSET                   0xbd4
#define GC_USB_DOEPDMA6_DEFAULT                  0x0
#define GC_USB_DOEPDMAB6_OFFSET                  0xbdc
#define GC_USB_DOEPDMAB6_DEFAULT                 0x0
#define GC_USB_DOEPCTL7_OFFSET                   0xbe0
#define GC_USB_DOEPCTL7_DEFAULT                  0x0
#define GC_USB_DOEPINT7_OFFSET                   0xbe8
#define GC_USB_DOEPINT7_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ7_OFFSET                  0xbf0
#define GC_USB_DOEPTSIZ7_DEFAULT                 0x0
#define GC_USB_DOEPDMA7_OFFSET                   0xbf4
#define GC_USB_DOEPDMA7_DEFAULT                  0x0
#define GC_USB_DOEPDMAB7_OFFSET                  0xbfc
#define GC_USB_DOEPDMAB7_DEFAULT                 0x0
#define GC_USB_DOEPCTL8_OFFSET                   0xc00
#define GC_USB_DOEPCTL8_DEFAULT                  0x0
#define GC_USB_DOEPINT8_OFFSET                   0xc08
#define GC_USB_DOEPINT8_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ8_OFFSET                  0xc10
#define GC_USB_DOEPTSIZ8_DEFAULT                 0x0
#define GC_USB_DOEPDMA8_OFFSET                   0xc14
#define GC_USB_DOEPDMA8_DEFAULT                  0x0
#define GC_USB_DOEPDMAB8_OFFSET                  0xc1c
#define GC_USB_DOEPDMAB8_DEFAULT                 0x0
#define GC_USB_DOEPCTL9_OFFSET                   0xc20
#define GC_USB_DOEPCTL9_DEFAULT                  0x0
#define GC_USB_DOEPINT9_OFFSET                   0xc28
#define GC_USB_DOEPINT9_DEFAULT                  0x0
#define GC_USB_DOEPTSIZ9_OFFSET                  0xc30
#define GC_USB_DOEPTSIZ9_DEFAULT                 0x0
#define GC_USB_DOEPDMA9_OFFSET                   0xc34
#define GC_USB_DOEPDMA9_DEFAULT                  0x0
#define GC_USB_DOEPDMAB9_OFFSET                  0xc3c
#define GC_USB_DOEPDMAB9_DEFAULT                 0x0
#define GC_USB_DOEPCTL10_OFFSET                  0xc40
#define GC_USB_DOEPCTL10_DEFAULT                 0x0
#define GC_USB_DOEPINT10_OFFSET                  0xc48
#define GC_USB_DOEPINT10_DEFAULT                 0x0
#define GC_USB_DOEPTSIZ10_OFFSET                 0xc50
#define GC_USB_DOEPTSIZ10_DEFAULT                0x0
#define GC_USB_DOEPDMA10_OFFSET                  0xc54
#define GC_USB_DOEPDMA10_DEFAULT                 0x0
#define GC_USB_DOEPDMAB10_OFFSET                 0xc5c
#define GC_USB_DOEPDMAB10_DEFAULT                0x0
#define GC_USB_DOEPCTL11_OFFSET                  0xc60
#define GC_USB_DOEPCTL11_DEFAULT                 0x0
#define GC_USB_DOEPINT11_OFFSET                  0xc68
#define GC_USB_DOEPINT11_DEFAULT                 0x0
#define GC_USB_DOEPTSIZ11_OFFSET                 0xc70
#define GC_USB_DOEPTSIZ11_DEFAULT                0x0
#define GC_USB_DOEPDMA11_OFFSET                  0xc74
#define GC_USB_DOEPDMA11_DEFAULT                 0x0
#define GC_USB_DOEPDMAB11_OFFSET                 0xc7c
#define GC_USB_DOEPDMAB11_DEFAULT                0x0
#define GC_USB_DOEPCTL12_OFFSET                  0xc80
#define GC_USB_DOEPCTL12_DEFAULT                 0x0
#define GC_USB_DOEPINT12_OFFSET                  0xc88
#define GC_USB_DOEPINT12_DEFAULT                 0x0
#define GC_USB_DOEPTSIZ12_OFFSET                 0xc90
#define GC_USB_DOEPTSIZ12_DEFAULT                0x0
#define GC_USB_DOEPDMA12_OFFSET                  0xc94
#define GC_USB_DOEPDMA12_DEFAULT                 0x0
#define GC_USB_DOEPDMAB12_OFFSET                 0xc9c
#define GC_USB_DOEPDMAB12_DEFAULT                0x0
#define GC_USB_DOEPCTL13_OFFSET                  0xca0
#define GC_USB_DOEPCTL13_DEFAULT                 0x0
#define GC_USB_DOEPINT13_OFFSET                  0xca8
#define GC_USB_DOEPINT13_DEFAULT                 0x0
#define GC_USB_DOEPTSIZ13_OFFSET                 0xcb0
#define GC_USB_DOEPTSIZ13_DEFAULT                0x0
#define GC_USB_DOEPDMA13_OFFSET                  0xcb4
#define GC_USB_DOEPDMA13_DEFAULT                 0x0
#define GC_USB_DOEPDMAB13_OFFSET                 0xcbc
#define GC_USB_DOEPDMAB13_DEFAULT                0x0
#define GC_USB_DOEPCTL14_OFFSET                  0xcc0
#define GC_USB_DOEPCTL14_DEFAULT                 0x0
#define GC_USB_DOEPINT14_OFFSET                  0xcc8
#define GC_USB_DOEPINT14_DEFAULT                 0x0
#define GC_USB_DOEPTSIZ14_OFFSET                 0xcd0
#define GC_USB_DOEPTSIZ14_DEFAULT                0x0
#define GC_USB_DOEPDMA14_OFFSET                  0xcd4
#define GC_USB_DOEPDMA14_DEFAULT                 0x0
#define GC_USB_DOEPDMAB14_OFFSET                 0xcdc
#define GC_USB_DOEPDMAB14_DEFAULT                0x0
#define GC_USB_DOEPCTL15_OFFSET                  0xce0
#define GC_USB_DOEPCTL15_DEFAULT                 0x0
#define GC_USB_DOEPINT15_OFFSET                  0xce8
#define GC_USB_DOEPINT15_DEFAULT                 0x0
#define GC_USB_DOEPTSIZ15_OFFSET                 0xcf0
#define GC_USB_DOEPTSIZ15_DEFAULT                0x0
#define GC_USB_DOEPDMA15_OFFSET                  0xcf4
#define GC_USB_DOEPDMA15_DEFAULT                 0x0
#define GC_USB_DOEPDMAB15_OFFSET                 0xcfc
#define GC_USB_DOEPDMAB15_DEFAULT                0x0
#define GC_USB_PCGCCTL_OFFSET                    0xe00
#define GC_USB_PCGCCTL_DEFAULT                   0x0
#define GC_USB_DFIFO_OFFSET                      0x20000
#define GC_USB_GOTGCTL_BVALIDOVEN_LSB            0x6
#define GC_USB_GOTGCTL_BVALIDOVEN_MASK           0x40
#define GC_USB_GOTGCTL_BVALIDOVEN_SIZE           0x1
#define GC_USB_GOTGCTL_BVALIDOVEN_DEFAULT        0x0
#define GC_USB_GOTGCTL_BVALIDOVEN_OFFSET         0x0
#define GC_USB_GOTGCTL_BVALIDOVVAL_LSB           0x7
#define GC_USB_GOTGCTL_BVALIDOVVAL_MASK          0x80
#define GC_USB_GOTGCTL_BVALIDOVVAL_SIZE          0x1
#define GC_USB_GOTGCTL_BVALIDOVVAL_DEFAULT       0x0
#define GC_USB_GOTGCTL_BVALIDOVVAL_OFFSET        0x0
#define GC_USB_GOTGCTL_CONIDSTS_LSB              0x10
#define GC_USB_GOTGCTL_CONIDSTS_MASK             0x10000
#define GC_USB_GOTGCTL_CONIDSTS_SIZE             0x1
#define GC_USB_GOTGCTL_CONIDSTS_DEFAULT          0x0
#define GC_USB_GOTGCTL_CONIDSTS_OFFSET           0x0
#define GC_USB_GOTGCTL_BSESVLD_LSB               0x13
#define GC_USB_GOTGCTL_BSESVLD_MASK              0x80000
#define GC_USB_GOTGCTL_BSESVLD_SIZE              0x1
#define GC_USB_GOTGCTL_BSESVLD_DEFAULT           0x0
#define GC_USB_GOTGCTL_BSESVLD_OFFSET            0x0
#define GC_USB_GOTGCTL_OTGVER_LSB                0x14
#define GC_USB_GOTGCTL_OTGVER_MASK               0x100000
#define GC_USB_GOTGCTL_OTGVER_SIZE               0x1
#define GC_USB_GOTGCTL_OTGVER_DEFAULT            0x0
#define GC_USB_GOTGCTL_OTGVER_OFFSET             0x0
#define GC_USB_GOTGCTL_CURMOD_LSB                0x15
#define GC_USB_GOTGCTL_CURMOD_MASK               0x200000
#define GC_USB_GOTGCTL_CURMOD_SIZE               0x1
#define GC_USB_GOTGCTL_CURMOD_DEFAULT            0x0
#define GC_USB_GOTGCTL_CURMOD_OFFSET             0x0
#define GC_USB_GOTGINT_SESENDDET_LSB             0x2
#define GC_USB_GOTGINT_SESENDDET_MASK            0x4
#define GC_USB_GOTGINT_SESENDDET_SIZE            0x1
#define GC_USB_GOTGINT_SESENDDET_DEFAULT         0x0
#define GC_USB_GOTGINT_SESENDDET_OFFSET          0x4
#define GC_USB_GOTGINT_SESREQSUCSTSCHNG_LSB      0x8
#define GC_USB_GOTGINT_SESREQSUCSTSCHNG_MASK     0x100
#define GC_USB_GOTGINT_SESREQSUCSTSCHNG_SIZE     0x1
#define GC_USB_GOTGINT_SESREQSUCSTSCHNG_DEFAULT  0x0
#define GC_USB_GOTGINT_SESREQSUCSTSCHNG_OFFSET   0x4
#define GC_USB_GOTGINT_HSTNEGSUCSTSCHNG_LSB      0x9
#define GC_USB_GOTGINT_HSTNEGSUCSTSCHNG_MASK     0x200
#define GC_USB_GOTGINT_HSTNEGSUCSTSCHNG_SIZE     0x1
#define GC_USB_GOTGINT_HSTNEGSUCSTSCHNG_DEFAULT  0x0
#define GC_USB_GOTGINT_HSTNEGSUCSTSCHNG_OFFSET   0x4
#define GC_USB_GOTGINT_HSTNEGDET_LSB             0x11
#define GC_USB_GOTGINT_HSTNEGDET_MASK            0x20000
#define GC_USB_GOTGINT_HSTNEGDET_SIZE            0x1
#define GC_USB_GOTGINT_HSTNEGDET_DEFAULT         0x0
#define GC_USB_GOTGINT_HSTNEGDET_OFFSET          0x4
#define GC_USB_GOTGINT_ADEVTOUTCHG_LSB           0x12
#define GC_USB_GOTGINT_ADEVTOUTCHG_MASK          0x40000
#define GC_USB_GOTGINT_ADEVTOUTCHG_SIZE          0x1
#define GC_USB_GOTGINT_ADEVTOUTCHG_DEFAULT       0x0
#define GC_USB_GOTGINT_ADEVTOUTCHG_OFFSET        0x4
#define GC_USB_GAHBCFG_GLBLINTRMSK_LSB           0x0
#define GC_USB_GAHBCFG_GLBLINTRMSK_MASK          0x1
#define GC_USB_GAHBCFG_GLBLINTRMSK_SIZE          0x1
#define GC_USB_GAHBCFG_GLBLINTRMSK_DEFAULT       0x0
#define GC_USB_GAHBCFG_GLBLINTRMSK_OFFSET        0x8
#define GC_USB_GAHBCFG_HBSTLEN_LSB               0x1
#define GC_USB_GAHBCFG_HBSTLEN_MASK              0x1e
#define GC_USB_GAHBCFG_HBSTLEN_SIZE              0x4
#define GC_USB_GAHBCFG_HBSTLEN_DEFAULT           0x0
#define GC_USB_GAHBCFG_HBSTLEN_OFFSET            0x8
#define GC_USB_GAHBCFG_DMAEN_LSB                 0x5
#define GC_USB_GAHBCFG_DMAEN_MASK                0x20
#define GC_USB_GAHBCFG_DMAEN_SIZE                0x1
#define GC_USB_GAHBCFG_DMAEN_DEFAULT             0x0
#define GC_USB_GAHBCFG_DMAEN_OFFSET              0x8

#define GC_USB_GAHBCFG_NPTXFEMPLVL_LSB           0x7
#define GC_USB_GAHBCFG_NPTXFEMPLVL_MASK          0x80
#define GC_USB_GAHBCFG_NPTXFEMPLVL_SIZE          0x1
#define GC_USB_GAHBCFG_NPTXFEMPLVL_DEFAULT       0x0
#define GC_USB_GAHBCFG_NPTXFEMPLVL_OFFSET        0x8

#define GC_USB_GAHBCFG_REMMEMSUPP_LSB            0x15
#define GC_USB_GAHBCFG_REMMEMSUPP_MASK           0x200000
#define GC_USB_GAHBCFG_REMMEMSUPP_SIZE           0x1
#define GC_USB_GAHBCFG_REMMEMSUPP_DEFAULT        0x0
#define GC_USB_GAHBCFG_REMMEMSUPP_OFFSET         0x8
#define GC_USB_GAHBCFG_NOTIALLDMAWRIT_LSB        0x16
#define GC_USB_GAHBCFG_NOTIALLDMAWRIT_MASK       0x400000
#define GC_USB_GAHBCFG_NOTIALLDMAWRIT_SIZE       0x1
#define GC_USB_GAHBCFG_NOTIALLDMAWRIT_DEFAULT    0x0
#define GC_USB_GAHBCFG_NOTIALLDMAWRIT_OFFSET     0x8
#define GC_USB_GAHBCFG_AHBSINGLE_LSB             0x17
#define GC_USB_GAHBCFG_AHBSINGLE_MASK            0x800000
#define GC_USB_GAHBCFG_AHBSINGLE_SIZE            0x1
#define GC_USB_GAHBCFG_AHBSINGLE_DEFAULT         0x0
#define GC_USB_GAHBCFG_AHBSINGLE_OFFSET          0x8
#define GC_USB_GAHBCFG_INVDESCENDIANESS_LSB      0x18
#define GC_USB_GAHBCFG_INVDESCENDIANESS_MASK     0x1000000
#define GC_USB_GAHBCFG_INVDESCENDIANESS_SIZE     0x1
#define GC_USB_GAHBCFG_INVDESCENDIANESS_DEFAULT  0x0
#define GC_USB_GAHBCFG_INVDESCENDIANESS_OFFSET   0x8
#define GC_USB_GUSBCFG_TOUTCAL_LSB               0x0
#define GC_USB_GUSBCFG_TOUTCAL_MASK              0x7
#define GC_USB_GUSBCFG_TOUTCAL_SIZE              0x3
#define GC_USB_GUSBCFG_TOUTCAL_DEFAULT           0x0
#define GC_USB_GUSBCFG_TOUTCAL_OFFSET            0xc

#define GC_USB_GUSBCFG_USBTRDTIM_LSB             0xa
#define GC_USB_GUSBCFG_USBTRDTIM_MASK            0x3c00
#define GC_USB_GUSBCFG_USBTRDTIM_SIZE            0x4
#define GC_USB_GUSBCFG_USBTRDTIM_DEFAULT         0x0
#define GC_USB_GUSBCFG_USBTRDTIM_OFFSET          0xc
#define GC_USB_GUSBCFG_PHYLPWRCLKSEL_LSB         15
#define GC_USB_GUSBCFG_PHYLPWRCLKSEL_MASK        0x8000
#define GC_USB_GUSBCFG_PHYLPWRCLKSEL_SIZE        0x1
#define GC_USB_GUSBCFG_PHYLPWRCLKSEL_DEFAULT     0x0
#define GC_USB_GUSBCFG_PHYLPWRCLKSEL_OFFSET      0xc
#define GC_USB_GUSBCFG_ULPIFSLS_LSB              17
#define GC_USB_GUSBCFG_ULPIFSLS_MASK             0x20000
#define GC_USB_GUSBCFG_ULPIFSLS_SIZE             0x1
#define GC_USB_GUSBCFG_ULPIFSLS_DEFAULT          0x0
#define GC_USB_GUSBCFG_ULPIFSLS_OFFSET           0xc
#define GC_USB_GUSBCFG_ULPIAUTORES_LSB           18
#define GC_USB_GUSBCFG_ULPIAUTORES_MASK          0x40000
#define GC_USB_GUSBCFG_ULPIAUTORES_SIZE          0x1
#define GC_USB_GUSBCFG_ULPIAUTORES_DEFAULT       0x0
#define GC_USB_GUSBCFG_ULPIAUTORES_OFFSET        0xc
#define GC_USB_GUSBCFG_ULPICLKSUSM_LSB           19
#define GC_USB_GUSBCFG_ULPICLKSUSM_MASK          0x80000
#define GC_USB_GUSBCFG_ULPICLKSUSM_SIZE          0x1
#define GC_USB_GUSBCFG_ULPICLKSUSM_DEFAULT       0x0
#define GC_USB_GUSBCFG_ULPICLKSUSM_OFFSET        0xc

#define GC_USB_GUSBCFG_ULPIEVBUSD_LSB            20
#define GC_USB_GUSBCFG_ULPIEVBUSD_MASK           0x100000
#define GC_USB_GUSBCFG_ULPIEVBUSD_SIZE           0x1
#define GC_USB_GUSBCFG_ULPIEVBUSD_DEFAULT        0x0
#define GC_USB_GUSBCFG_ULPIEVBUSD_OFFSET         0xc

#define GC_USB_GUSBCFG_ULPIEVBUSI_LSB            21
#define GC_USB_GUSBCFG_ULPIEVBUSI_MASK           0x200000
#define GC_USB_GUSBCFG_ULPIEVBUSI_SIZE           0x1
#define GC_USB_GUSBCFG_ULPIEVBUSI_DEFAULT        0x0
#define GC_USB_GUSBCFG_ULPIEVBUSI_OFFSET         0xc

#define GC_USB_GUSBCFG_TERMSELDLPULSE_LSB        22
#define GC_USB_GUSBCFG_TERMSELDLPULSE_MASK       0x400000
#define GC_USB_GUSBCFG_TERMSELDLPULSE_SIZE       0x1
#define GC_USB_GUSBCFG_TERMSELDLPULSE_DEFAULT    0x0
#define GC_USB_GUSBCFG_TERMSELDLPULSE_OFFSET     0xc

#define GC_USB_GUSBCFG_PCCI_LSB                  23
#define GC_USB_GUSBCFG_PCCI_MASK                 BIT(23)
#define GC_USB_GUSBCFG_PCCI_SIZE                 0x1
#define GC_USB_GUSBCFG_PCCI_DEFAULT              0x0
#define GC_USB_GUSBCFG_PCCI_OFFSET               0xc

#define GC_USB_GUSBCFG_PTCI_LSB                  24
#define GC_USB_GUSBCFG_PTCI_MASK                 BIT(24)
#define GC_USB_GUSBCFG_PTCI_SIZE                 0x1
#define GC_USB_GUSBCFG_PTCI_DEFAULT              0x0
#define GC_USB_GUSBCFG_PTCI_OFFSET               0xc

#define GC_USB_GUSBCFG_ULPIIPD_LSB               25
#define GC_USB_GUSBCFG_ULPIIPD_MASK              BIT(25)
#define GC_USB_GUSBCFG_ULPIIPD_SIZE              0x1
#define GC_USB_GUSBCFG_ULPIIPD_DEFAULT           0x0
#define GC_USB_GUSBCFG_ULPIIPD_OFFSET            0xc

#define GC_USB_GUSBCFG_FHMOD_LSB                 29
#define GC_USB_GUSBCFG_FHMOD_MASK                BIT(29)
#define GC_USB_GUSBCFG_FHMOD_SIZE                0x1
#define GC_USB_GUSBCFG_FHMOD_DEFAULT             0x0
#define GC_USB_GUSBCFG_FHMOD_OFFSET              0xc

#define GC_USB_GUSBCFG_FDMOD_LSB                 30
#define GC_USB_GUSBCFG_FDMOD_MASK                BIT(30)
#define GC_USB_GUSBCFG_FDMOD_SIZE                0x1
#define GC_USB_GUSBCFG_FDMOD_DEFAULT             0x0
#define GC_USB_GUSBCFG_FDMOD_OFFSET              0xc

#define GC_USB_GRSTCTL_CSFTRST_LSB               0x0
#define GC_USB_GRSTCTL_CSFTRST_MASK              0x1
#define GC_USB_GRSTCTL_CSFTRST_SIZE              0x1
#define GC_USB_GRSTCTL_CSFTRST_DEFAULT           0x0
#define GC_USB_GRSTCTL_CSFTRST_OFFSET            0x10
#define GC_USB_GRSTCTL_PIUFSSFTRST_LSB           0x1
#define GC_USB_GRSTCTL_PIUFSSFTRST_MASK          0x2
#define GC_USB_GRSTCTL_PIUFSSFTRST_SIZE          0x1
#define GC_USB_GRSTCTL_PIUFSSFTRST_DEFAULT       0x0
#define GC_USB_GRSTCTL_PIUFSSFTRST_OFFSET        0x10
#define GC_USB_GRSTCTL_RXFFLSH_LSB               0x4
#define GC_USB_GRSTCTL_RXFFLSH_MASK              0x10
#define GC_USB_GRSTCTL_RXFFLSH_SIZE              0x1
#define GC_USB_GRSTCTL_RXFFLSH_DEFAULT           0x0
#define GC_USB_GRSTCTL_RXFFLSH_OFFSET            0x10
#define GC_USB_GRSTCTL_TXFFLSH_LSB               0x5
#define GC_USB_GRSTCTL_TXFFLSH_MASK              0x20
#define GC_USB_GRSTCTL_TXFFLSH_SIZE              0x1
#define GC_USB_GRSTCTL_TXFFLSH_DEFAULT           0x0
#define GC_USB_GRSTCTL_TXFFLSH_OFFSET            0x10
#define GC_USB_GRSTCTL_TXFNUM_LSB                0x6
#define GC_USB_GRSTCTL_TXFNUM_MASK               0x7c0
#define GC_USB_GRSTCTL_TXFNUM_SIZE               0x5
#define GC_USB_GRSTCTL_TXFNUM_DEFAULT            0x0
#define GC_USB_GRSTCTL_TXFNUM_OFFSET             0x10
#define GC_USB_GRSTCTL_DMAREQ_LSB                0x1e
#define GC_USB_GRSTCTL_DMAREQ_MASK               0x40000000
#define GC_USB_GRSTCTL_DMAREQ_SIZE               0x1
#define GC_USB_GRSTCTL_DMAREQ_DEFAULT            0x0
#define GC_USB_GRSTCTL_DMAREQ_OFFSET             0x10
#define GC_USB_GRSTCTL_AHBIDLE_LSB               0x1f
#define GC_USB_GRSTCTL_AHBIDLE_MASK              0x80000000
#define GC_USB_GRSTCTL_AHBIDLE_SIZE              0x1
#define GC_USB_GRSTCTL_AHBIDLE_DEFAULT           0x0
#define GC_USB_GRSTCTL_AHBIDLE_OFFSET            0x10
#define GC_USB_GINTSTS_CURMOD_LSB                0x0
#define GC_USB_GINTSTS_CURMOD_MASK               0x1
#define GC_USB_GINTSTS_CURMOD_SIZE               0x1
#define GC_USB_GINTSTS_CURMOD_DEFAULT            0x0
#define GC_USB_GINTSTS_CURMOD_OFFSET             0x14
#define GC_USB_GINTSTS_MODEMIS_LSB               0x1
#define GC_USB_GINTSTS_MODEMIS_MASK              0x2
#define GC_USB_GINTSTS_MODEMIS_SIZE              0x1
#define GC_USB_GINTSTS_MODEMIS_DEFAULT           0x0
#define GC_USB_GINTSTS_MODEMIS_OFFSET            0x14
#define GC_USB_GINTSTS_OTGINT_LSB                0x2
#define GC_USB_GINTSTS_OTGINT_MASK               0x4
#define GC_USB_GINTSTS_OTGINT_SIZE               0x1
#define GC_USB_GINTSTS_OTGINT_DEFAULT            0x0
#define GC_USB_GINTSTS_OTGINT_OFFSET             0x14
#define GC_USB_GINTSTS_SOF_LSB                   0x3
#define GC_USB_GINTSTS_SOF_MASK                  0x8
#define GC_USB_GINTSTS_SOF_SIZE                  0x1
#define GC_USB_GINTSTS_SOF_DEFAULT               0x0
#define GC_USB_GINTSTS_SOF_OFFSET                0x14
#define GC_USB_GINTSTS_RXFLVL_LSB                0x4
#define GC_USB_GINTSTS_RXFLVL_MASK               0x10
#define GC_USB_GINTSTS_RXFLVL_SIZE               0x1
#define GC_USB_GINTSTS_RXFLVL_DEFAULT            0x0
#define GC_USB_GINTSTS_RXFLVL_OFFSET             0x14
#define GC_USB_GINTSTS_GINNAKEFF_LSB             0x6
#define GC_USB_GINTSTS_GINNAKEFF_MASK            0x40
#define GC_USB_GINTSTS_GINNAKEFF_SIZE            0x1
#define GC_USB_GINTSTS_GINNAKEFF_DEFAULT         0x0
#define GC_USB_GINTSTS_GINNAKEFF_OFFSET          0x14
#define GC_USB_GINTSTS_GOUTNAKEFF_LSB            0x7
#define GC_USB_GINTSTS_GOUTNAKEFF_MASK           0x80
#define GC_USB_GINTSTS_GOUTNAKEFF_SIZE           0x1
#define GC_USB_GINTSTS_GOUTNAKEFF_DEFAULT        0x0
#define GC_USB_GINTSTS_GOUTNAKEFF_OFFSET         0x14
#define GC_USB_GINTSTS_ERLYSUSP_LSB              0xa
#define GC_USB_GINTSTS_ERLYSUSP_MASK             0x400
#define GC_USB_GINTSTS_ERLYSUSP_SIZE             0x1
#define GC_USB_GINTSTS_ERLYSUSP_DEFAULT          0x0
#define GC_USB_GINTSTS_ERLYSUSP_OFFSET           0x14
#define GC_USB_GINTSTS_USBSUSP_LSB               0xb
#define GC_USB_GINTSTS_USBSUSP_MASK              0x800
#define GC_USB_GINTSTS_USBSUSP_SIZE              0x1
#define GC_USB_GINTSTS_USBSUSP_DEFAULT           0x0
#define GC_USB_GINTSTS_USBSUSP_OFFSET            0x14
#define GC_USB_GINTSTS_USBRST_LSB                0xc
#define GC_USB_GINTSTS_USBRST_MASK               0x1000
#define GC_USB_GINTSTS_USBRST_SIZE               0x1
#define GC_USB_GINTSTS_USBRST_DEFAULT            0x0
#define GC_USB_GINTSTS_USBRST_OFFSET             0x14
#define GC_USB_GINTSTS_ENUMDONE_LSB              0xd
#define GC_USB_GINTSTS_ENUMDONE_MASK             0x2000
#define GC_USB_GINTSTS_ENUMDONE_SIZE             0x1
#define GC_USB_GINTSTS_ENUMDONE_DEFAULT          0x0
#define GC_USB_GINTSTS_ENUMDONE_OFFSET           0x14
#define GC_USB_GINTSTS_ISOOUTDROP_LSB            0xe
#define GC_USB_GINTSTS_ISOOUTDROP_MASK           0x4000
#define GC_USB_GINTSTS_ISOOUTDROP_SIZE           0x1
#define GC_USB_GINTSTS_ISOOUTDROP_DEFAULT        0x0
#define GC_USB_GINTSTS_ISOOUTDROP_OFFSET         0x14
#define GC_USB_GINTSTS_EOPF_LSB                  0xf
#define GC_USB_GINTSTS_EOPF_MASK                 0x8000
#define GC_USB_GINTSTS_EOPF_SIZE                 0x1
#define GC_USB_GINTSTS_EOPF_DEFAULT              0x0
#define GC_USB_GINTSTS_EOPF_OFFSET               0x14
#define GC_USB_GINTSTS_EPMIS_LSB                 0x11
#define GC_USB_GINTSTS_EPMIS_MASK                0x20000
#define GC_USB_GINTSTS_EPMIS_SIZE                0x1
#define GC_USB_GINTSTS_EPMIS_DEFAULT             0x0
#define GC_USB_GINTSTS_EPMIS_OFFSET              0x14
#define GC_USB_GINTSTS_IEPINT_LSB                0x12
#define GC_USB_GINTSTS_IEPINT_MASK               0x40000
#define GC_USB_GINTSTS_IEPINT_SIZE               0x1
#define GC_USB_GINTSTS_IEPINT_DEFAULT            0x0
#define GC_USB_GINTSTS_IEPINT_OFFSET             0x14
#define GC_USB_GINTSTS_OEPINT_LSB                0x13
#define GC_USB_GINTSTS_OEPINT_MASK               0x80000
#define GC_USB_GINTSTS_OEPINT_SIZE               0x1
#define GC_USB_GINTSTS_OEPINT_DEFAULT            0x0
#define GC_USB_GINTSTS_OEPINT_OFFSET             0x14
#define GC_USB_GINTSTS_INCOMPISOIN_LSB           0x14
#define GC_USB_GINTSTS_INCOMPISOIN_MASK          0x100000
#define GC_USB_GINTSTS_INCOMPISOIN_SIZE          0x1
#define GC_USB_GINTSTS_INCOMPISOIN_DEFAULT       0x0
#define GC_USB_GINTSTS_INCOMPISOIN_OFFSET        0x14
#define GC_USB_GINTSTS_INCOMPLP_LSB              0x15
#define GC_USB_GINTSTS_INCOMPLP_MASK             0x200000
#define GC_USB_GINTSTS_INCOMPLP_SIZE             0x1
#define GC_USB_GINTSTS_INCOMPLP_DEFAULT          0x0
#define GC_USB_GINTSTS_INCOMPLP_OFFSET           0x14
#define GC_USB_GINTSTS_FETSUSP_LSB               0x16
#define GC_USB_GINTSTS_FETSUSP_MASK              0x400000
#define GC_USB_GINTSTS_FETSUSP_SIZE              0x1
#define GC_USB_GINTSTS_FETSUSP_DEFAULT           0x0
#define GC_USB_GINTSTS_FETSUSP_OFFSET            0x14
#define GC_USB_GINTSTS_RESETDET_LSB              0x17
#define GC_USB_GINTSTS_RESETDET_MASK             0x800000
#define GC_USB_GINTSTS_RESETDET_SIZE             0x1
#define GC_USB_GINTSTS_RESETDET_DEFAULT          0x0
#define GC_USB_GINTSTS_RESETDET_OFFSET           0x14
#define GC_USB_GINTSTS_CONIDSTSCHNG_LSB          0x1c
#define GC_USB_GINTSTS_CONIDSTSCHNG_MASK         0x10000000
#define GC_USB_GINTSTS_CONIDSTSCHNG_SIZE         0x1
#define GC_USB_GINTSTS_CONIDSTSCHNG_DEFAULT      0x0
#define GC_USB_GINTSTS_CONIDSTSCHNG_OFFSET       0x14
#define GC_USB_GINTSTS_SESSREQINT_LSB            0x1e
#define GC_USB_GINTSTS_SESSREQINT_MASK           0x40000000
#define GC_USB_GINTSTS_SESSREQINT_SIZE           0x1
#define GC_USB_GINTSTS_SESSREQINT_DEFAULT        0x0
#define GC_USB_GINTSTS_SESSREQINT_OFFSET         0x14
#define GC_USB_GINTSTS_WKUPINT_LSB               0x1f
#define GC_USB_GINTSTS_WKUPINT_MASK              0x80000000
#define GC_USB_GINTSTS_WKUPINT_SIZE              0x1
#define GC_USB_GINTSTS_WKUPINT_DEFAULT           0x0
#define GC_USB_GINTSTS_WKUPINT_OFFSET            0x14
#define GC_USB_GINTMSK_MODEMISMSK_LSB            0x1
#define GC_USB_GINTMSK_MODEMISMSK_MASK           0x2
#define GC_USB_GINTMSK_MODEMISMSK_SIZE           0x1
#define GC_USB_GINTMSK_MODEMISMSK_DEFAULT        0x0
#define GC_USB_GINTMSK_MODEMISMSK_OFFSET         0x18
#define GC_USB_GINTMSK_OTGINTMSK_LSB             0x2
#define GC_USB_GINTMSK_OTGINTMSK_MASK            0x4
#define GC_USB_GINTMSK_OTGINTMSK_SIZE            0x1
#define GC_USB_GINTMSK_OTGINTMSK_DEFAULT         0x0
#define GC_USB_GINTMSK_OTGINTMSK_OFFSET          0x18
#define GC_USB_GINTMSK_SOFMSK_LSB                0x3
#define GC_USB_GINTMSK_SOFMSK_MASK               0x8
#define GC_USB_GINTMSK_SOFMSK_SIZE               0x1
#define GC_USB_GINTMSK_SOFMSK_DEFAULT            0x0
#define GC_USB_GINTMSK_SOFMSK_OFFSET             0x18
#define GC_USB_GINTMSK_RXFLVLMSK_LSB             0x4
#define GC_USB_GINTMSK_RXFLVLMSK_MASK            0x10
#define GC_USB_GINTMSK_RXFLVLMSK_SIZE            0x1
#define GC_USB_GINTMSK_RXFLVLMSK_DEFAULT         0x0
#define GC_USB_GINTMSK_RXFLVLMSK_OFFSET          0x18
#define GC_USB_GINTMSK_UNKNOWN5_LSB              0x5
#define GC_USB_GINTMSK_UNKNOWN5_MASK             0x20
#define GC_USB_GINTMSK_UNKNOWN5_SIZE             0x1
#define GC_USB_GINTMSK_UNKNOWN5_DEFAULT          0x0
#define GC_USB_GINTMSK_UNKNOWN5_OFFSET           0x18
#define GC_USB_GINTMSK_GINNAKEFFMSK_LSB          0x6
#define GC_USB_GINTMSK_GINNAKEFFMSK_MASK         0x40
#define GC_USB_GINTMSK_GINNAKEFFMSK_SIZE         0x1
#define GC_USB_GINTMSK_GINNAKEFFMSK_DEFAULT      0x0
#define GC_USB_GINTMSK_GINNAKEFFMSK_OFFSET       0x18
#define GC_USB_GINTMSK_GOUTNAKEFFMSK_LSB         0x7
#define GC_USB_GINTMSK_GOUTNAKEFFMSK_MASK        0x80
#define GC_USB_GINTMSK_GOUTNAKEFFMSK_SIZE        0x1
#define GC_USB_GINTMSK_GOUTNAKEFFMSK_DEFAULT     0x0
#define GC_USB_GINTMSK_GOUTNAKEFFMSK_OFFSET      0x18
#define GC_USB_GINTMSK_ERLYSUSPMSK_LSB           0xa
#define GC_USB_GINTMSK_ERLYSUSPMSK_MASK          0x400
#define GC_USB_GINTMSK_ERLYSUSPMSK_SIZE          0x1
#define GC_USB_GINTMSK_ERLYSUSPMSK_DEFAULT       0x0
#define GC_USB_GINTMSK_ERLYSUSPMSK_OFFSET        0x18
#define GC_USB_GINTMSK_USBSUSPMSK_LSB            0xb
#define GC_USB_GINTMSK_USBSUSPMSK_MASK           0x800
#define GC_USB_GINTMSK_USBSUSPMSK_SIZE           0x1
#define GC_USB_GINTMSK_USBSUSPMSK_DEFAULT        0x0
#define GC_USB_GINTMSK_USBSUSPMSK_OFFSET         0x18
#define GC_USB_GINTMSK_USBRSTMSK_LSB             0xc
#define GC_USB_GINTMSK_USBRSTMSK_MASK            0x1000
#define GC_USB_GINTMSK_USBRSTMSK_SIZE            0x1
#define GC_USB_GINTMSK_USBRSTMSK_DEFAULT         0x0
#define GC_USB_GINTMSK_USBRSTMSK_OFFSET          0x18
#define GC_USB_GINTMSK_ENUMDONEMSK_LSB           0xd
#define GC_USB_GINTMSK_ENUMDONEMSK_MASK          0x2000
#define GC_USB_GINTMSK_ENUMDONEMSK_SIZE          0x1
#define GC_USB_GINTMSK_ENUMDONEMSK_DEFAULT       0x0
#define GC_USB_GINTMSK_ENUMDONEMSK_OFFSET        0x18
#define GC_USB_GINTMSK_ISOOUTDROPMSK_LSB         0xe
#define GC_USB_GINTMSK_ISOOUTDROPMSK_MASK        0x4000
#define GC_USB_GINTMSK_ISOOUTDROPMSK_SIZE        0x1
#define GC_USB_GINTMSK_ISOOUTDROPMSK_DEFAULT     0x0
#define GC_USB_GINTMSK_ISOOUTDROPMSK_OFFSET      0x18
#define GC_USB_GINTMSK_EOPFMSK_LSB               0xf
#define GC_USB_GINTMSK_EOPFMSK_MASK              0x8000
#define GC_USB_GINTMSK_EOPFMSK_SIZE              0x1
#define GC_USB_GINTMSK_EOPFMSK_DEFAULT           0x0
#define GC_USB_GINTMSK_EOPFMSK_OFFSET            0x18
#define GC_USB_GINTMSK_UNKNOWN16_LSB             0x10
#define GC_USB_GINTMSK_UNKNOWN16_MASK            0x10000
#define GC_USB_GINTMSK_UNKNOWN16_SIZE            0x1
#define GC_USB_GINTMSK_UNKNOWN16_DEFAULT         0x0
#define GC_USB_GINTMSK_UNKNOWN16_OFFSET          0x18
#define GC_USB_GINTMSK_EPMISMSK_LSB              0x11
#define GC_USB_GINTMSK_EPMISMSK_MASK             0x20000
#define GC_USB_GINTMSK_EPMISMSK_SIZE             0x1
#define GC_USB_GINTMSK_EPMISMSK_DEFAULT          0x0
#define GC_USB_GINTMSK_EPMISMSK_OFFSET           0x18
#define GC_USB_GINTMSK_IEPINTMSK_LSB             0x12
#define GC_USB_GINTMSK_IEPINTMSK_MASK            0x40000
#define GC_USB_GINTMSK_IEPINTMSK_SIZE            0x1
#define GC_USB_GINTMSK_IEPINTMSK_DEFAULT         0x0
#define GC_USB_GINTMSK_IEPINTMSK_OFFSET          0x18
#define GC_USB_GINTMSK_OEPINTMSK_LSB             0x13
#define GC_USB_GINTMSK_OEPINTMSK_MASK            0x80000
#define GC_USB_GINTMSK_OEPINTMSK_SIZE            0x1
#define GC_USB_GINTMSK_OEPINTMSK_DEFAULT         0x0
#define GC_USB_GINTMSK_OEPINTMSK_OFFSET          0x18
#define GC_USB_GINTMSK_INCOMPISOINMSK_LSB        0x14
#define GC_USB_GINTMSK_INCOMPISOINMSK_MASK       0x100000
#define GC_USB_GINTMSK_INCOMPISOINMSK_SIZE       0x1
#define GC_USB_GINTMSK_INCOMPISOINMSK_DEFAULT    0x0
#define GC_USB_GINTMSK_INCOMPISOINMSK_OFFSET     0x18
#define GC_USB_GINTMSK_INCOMPLISOOUTMSK_LSB      0x15
#define GC_USB_GINTMSK_INCOMPLISOOUTMSK_MASK     0x200000
#define GC_USB_GINTMSK_INCOMPLISOOUTMSK_SIZE     0x1
#define GC_USB_GINTMSK_INCOMPLISOOUTMSK_DEFAULT  0x0
#define GC_USB_GINTMSK_INCOMPLISOOUTMSK_OFFSET   0x18
#define GC_USB_GINTMSK_FETSUSPMSK_LSB            0x16
#define GC_USB_GINTMSK_FETSUSPMSK_MASK           0x400000
#define GC_USB_GINTMSK_FETSUSPMSK_SIZE           0x1
#define GC_USB_GINTMSK_FETSUSPMSK_DEFAULT        0x0
#define GC_USB_GINTMSK_FETSUSPMSK_OFFSET         0x18
#define GC_USB_GINTMSK_RESETDETMSK_LSB           0x17
#define GC_USB_GINTMSK_RESETDETMSK_MASK          0x800000
#define GC_USB_GINTMSK_RESETDETMSK_SIZE          0x1
#define GC_USB_GINTMSK_RESETDETMSK_DEFAULT       0x0
#define GC_USB_GINTMSK_RESETDETMSK_OFFSET        0x18
#define GC_USB_GINTMSK_CONIDSTSCHNGMSK_LSB       0x1c
#define GC_USB_GINTMSK_CONIDSTSCHNGMSK_MASK      0x10000000
#define GC_USB_GINTMSK_CONIDSTSCHNGMSK_SIZE      0x1
#define GC_USB_GINTMSK_CONIDSTSCHNGMSK_DEFAULT   0x0
#define GC_USB_GINTMSK_CONIDSTSCHNGMSK_OFFSET    0x18
#define GC_USB_GINTMSK_DISCONNINTMSK_LSB         0x1d
#define GC_USB_GINTMSK_DISCONNINTMSK_MASK        0x20000000
#define GC_USB_GINTMSK_DISCONNINTMSK_SIZE        0x1
#define GC_USB_GINTMSK_DISCONNINTMSK_DEFAULT     0x0
#define GC_USB_GINTMSK_DISCONNINTMSK_OFFSET      0x18
#define GC_USB_GINTMSK_SESSREQINTMSK_LSB         0x1e
#define GC_USB_GINTMSK_SESSREQINTMSK_MASK        0x40000000
#define GC_USB_GINTMSK_SESSREQINTMSK_SIZE        0x1
#define GC_USB_GINTMSK_SESSREQINTMSK_DEFAULT     0x0
#define GC_USB_GINTMSK_SESSREQINTMSK_OFFSET      0x18
#define GC_USB_GINTMSK_WKUPINTMSK_LSB            0x1f
#define GC_USB_GINTMSK_WKUPINTMSK_MASK           0x80000000
#define GC_USB_GINTMSK_WKUPINTMSK_SIZE           0x1
#define GC_USB_GINTMSK_WKUPINTMSK_DEFAULT        0x0
#define GC_USB_GINTMSK_WKUPINTMSK_OFFSET         0x18
#define GC_USB_GRXSTSR_CHNUM_LSB                 0x0
#define GC_USB_GRXSTSR_CHNUM_MASK                0xf
#define GC_USB_GRXSTSR_CHNUM_SIZE                0x4
#define GC_USB_GRXSTSR_CHNUM_DEFAULT             0x0
#define GC_USB_GRXSTSR_CHNUM_OFFSET              0x1c
#define GC_USB_GRXSTSR_BCNT_LSB                  0x4
#define GC_USB_GRXSTSR_BCNT_MASK                 0x7ff0
#define GC_USB_GRXSTSR_BCNT_SIZE                 0xb
#define GC_USB_GRXSTSR_BCNT_DEFAULT              0x0
#define GC_USB_GRXSTSR_BCNT_OFFSET               0x1c
#define GC_USB_GRXSTSR_DPID_LSB                  0xf
#define GC_USB_GRXSTSR_DPID_MASK                 0x18000
#define GC_USB_GRXSTSR_DPID_SIZE                 0x2
#define GC_USB_GRXSTSR_DPID_DEFAULT              0x0
#define GC_USB_GRXSTSR_DPID_OFFSET               0x1c
#define GC_USB_GRXSTSR_PKTSTS_LSB                0x11
#define GC_USB_GRXSTSR_PKTSTS_MASK               0x1e0000
#define GC_USB_GRXSTSR_PKTSTS_SIZE               0x4
#define GC_USB_GRXSTSR_PKTSTS_DEFAULT            0x0
#define GC_USB_GRXSTSR_PKTSTS_OFFSET             0x1c
#define GC_USB_GRXSTSR_FN_LSB                    0x15
#define GC_USB_GRXSTSR_FN_MASK                   0x1e00000
#define GC_USB_GRXSTSR_FN_SIZE                   0x4
#define GC_USB_GRXSTSR_FN_DEFAULT                0x0
#define GC_USB_GRXSTSR_FN_OFFSET                 0x1c
#define GC_USB_GRXSTSP_CHNUM_LSB                 0x0
#define GC_USB_GRXSTSP_CHNUM_MASK                0xf
#define GC_USB_GRXSTSP_CHNUM_SIZE                0x4
#define GC_USB_GRXSTSP_CHNUM_DEFAULT             0x0
#define GC_USB_GRXSTSP_CHNUM_OFFSET              0x20
#define GC_USB_GRXSTSP_BCNT_LSB                  0x4
#define GC_USB_GRXSTSP_BCNT_MASK                 0x7ff0
#define GC_USB_GRXSTSP_BCNT_SIZE                 0xb
#define GC_USB_GRXSTSP_BCNT_DEFAULT              0x0
#define GC_USB_GRXSTSP_BCNT_OFFSET               0x20
#define GC_USB_GRXSTSP_DPID_LSB                  0xf
#define GC_USB_GRXSTSP_DPID_MASK                 0x18000
#define GC_USB_GRXSTSP_DPID_SIZE                 0x2
#define GC_USB_GRXSTSP_DPID_DEFAULT              0x0
#define GC_USB_GRXSTSP_DPID_OFFSET               0x20
#define GC_USB_GRXSTSP_PKTSTS_LSB                0x11
#define GC_USB_GRXSTSP_PKTSTS_MASK               0x1e0000
#define GC_USB_GRXSTSP_PKTSTS_SIZE               0x4
#define GC_USB_GRXSTSP_PKTSTS_DEFAULT            0x0
#define GC_USB_GRXSTSP_PKTSTS_OFFSET             0x20
#define GC_USB_GRXSTSP_FN_LSB                    0x15
#define GC_USB_GRXSTSP_FN_MASK                   0x1e00000
#define GC_USB_GRXSTSP_FN_SIZE                   0x4
#define GC_USB_GRXSTSP_FN_DEFAULT                0x0
#define GC_USB_GRXSTSP_FN_OFFSET                 0x20
#define GC_USB_GRXFSIZ_RXFDEP_LSB                0x0
#define GC_USB_GRXFSIZ_RXFDEP_MASK               0x7ff
#define GC_USB_GRXFSIZ_RXFDEP_SIZE               0xb
#define GC_USB_GRXFSIZ_RXFDEP_DEFAULT            0x0
#define GC_USB_GRXFSIZ_RXFDEP_OFFSET             0x24
#define GC_USB_GNPTXFSIZ_INEPTXF0STADDR_LSB      0x0
#define GC_USB_GNPTXFSIZ_INEPTXF0STADDR_MASK     0xffff
#define GC_USB_GNPTXFSIZ_INEPTXF0STADDR_SIZE     0x10
#define GC_USB_GNPTXFSIZ_INEPTXF0STADDR_DEFAULT  0x0
#define GC_USB_GNPTXFSIZ_INEPTXF0STADDR_OFFSET   0x28
#define GC_USB_GNPTXFSIZ_INEPTXF0DEP_LSB         0x10
#define GC_USB_GNPTXFSIZ_INEPTXF0DEP_MASK        0xffff0000
#define GC_USB_GNPTXFSIZ_INEPTXF0DEP_SIZE        0x10
#define GC_USB_GNPTXFSIZ_INEPTXF0DEP_DEFAULT     0x0
#define GC_USB_GNPTXFSIZ_INEPTXF0DEP_OFFSET      0x28

#define GC_USB_GUID_GUID_LSB                     0x0
#define GC_USB_GUID_GUID_MASK                    0xffffffff
#define GC_USB_GUID_GUID_SIZE                    0x20
#define GC_USB_GUID_GUID_DEFAULT                 0x0
#define GC_USB_GUID_GUID_OFFSET                  0x3c
#define GC_USB_GSNPSID_SYNOPSYSID_LSB            0x0
#define GC_USB_GSNPSID_SYNOPSYSID_MASK           0xffffffff
#define GC_USB_GSNPSID_SYNOPSYSID_SIZE           0x20
#define GC_USB_GSNPSID_SYNOPSYSID_DEFAULT        0x0
#define GC_USB_GSNPSID_SYNOPSYSID_OFFSET         0x40
#define GC_USB_GHWCFG1_EPDIR_LSB                 0x0
#define GC_USB_GHWCFG1_EPDIR_MASK                0xffffffff
#define GC_USB_GHWCFG1_EPDIR_SIZE                0x20
#define GC_USB_GHWCFG1_EPDIR_DEFAULT             0x0
#define GC_USB_GHWCFG1_EPDIR_OFFSET              0x44
#define GC_USB_GHWCFG2_OTGMODE_LSB               0x0
#define GC_USB_GHWCFG2_OTGMODE_MASK              0x7
#define GC_USB_GHWCFG2_OTGMODE_SIZE              0x3
#define GC_USB_GHWCFG2_OTGMODE_DEFAULT           0x0
#define GC_USB_GHWCFG2_OTGMODE_OFFSET            0x48
#define GC_USB_GHWCFG2_OTGARCH_LSB               0x3
#define GC_USB_GHWCFG2_OTGARCH_MASK              0x18
#define GC_USB_GHWCFG2_OTGARCH_SIZE              0x2
#define GC_USB_GHWCFG2_OTGARCH_DEFAULT           0x0
#define GC_USB_GHWCFG2_OTGARCH_OFFSET            0x48
#define GC_USB_GHWCFG2_SINGPNT_LSB               0x5
#define GC_USB_GHWCFG2_SINGPNT_MASK              0x20
#define GC_USB_GHWCFG2_SINGPNT_SIZE              0x1
#define GC_USB_GHWCFG2_SINGPNT_DEFAULT           0x0
#define GC_USB_GHWCFG2_SINGPNT_OFFSET            0x48
#define GC_USB_GHWCFG2_HSPHYTYPE_LSB             0x6
#define GC_USB_GHWCFG2_HSPHYTYPE_MASK            0xc0
#define GC_USB_GHWCFG2_HSPHYTYPE_SIZE            0x2
#define GC_USB_GHWCFG2_HSPHYTYPE_DEFAULT         0x0
#define GC_USB_GHWCFG2_HSPHYTYPE_OFFSET          0x48
#define GC_USB_GHWCFG2_FSPHYTYPE_LSB             0x8
#define GC_USB_GHWCFG2_FSPHYTYPE_MASK            0x300
#define GC_USB_GHWCFG2_FSPHYTYPE_SIZE            0x2
#define GC_USB_GHWCFG2_FSPHYTYPE_DEFAULT         0x0
#define GC_USB_GHWCFG2_FSPHYTYPE_OFFSET          0x48
#define GC_USB_GHWCFG2_NUMDEVEPS_LSB             0xa
#define GC_USB_GHWCFG2_NUMDEVEPS_MASK            0x3c00
#define GC_USB_GHWCFG2_NUMDEVEPS_SIZE            0x4
#define GC_USB_GHWCFG2_NUMDEVEPS_DEFAULT         0x0
#define GC_USB_GHWCFG2_NUMDEVEPS_OFFSET          0x48
#define GC_USB_GHWCFG2_NUMHSTCHNL_LSB            0xe
#define GC_USB_GHWCFG2_NUMHSTCHNL_MASK           0x3c000
#define GC_USB_GHWCFG2_NUMHSTCHNL_SIZE           0x4
#define GC_USB_GHWCFG2_NUMHSTCHNL_DEFAULT        0x0
#define GC_USB_GHWCFG2_NUMHSTCHNL_OFFSET         0x48
#define GC_USB_GHWCFG2_PERIOSUPPORT_LSB          0x12
#define GC_USB_GHWCFG2_PERIOSUPPORT_MASK         0x40000
#define GC_USB_GHWCFG2_PERIOSUPPORT_SIZE         0x1
#define GC_USB_GHWCFG2_PERIOSUPPORT_DEFAULT      0x0
#define GC_USB_GHWCFG2_PERIOSUPPORT_OFFSET       0x48
#define GC_USB_GHWCFG2_DYNFIFOSIZING_LSB         0x13
#define GC_USB_GHWCFG2_DYNFIFOSIZING_MASK        0x80000
#define GC_USB_GHWCFG2_DYNFIFOSIZING_SIZE        0x1
#define GC_USB_GHWCFG2_DYNFIFOSIZING_DEFAULT     0x0
#define GC_USB_GHWCFG2_DYNFIFOSIZING_OFFSET      0x48
#define GC_USB_GHWCFG2_MULTIPROCINTRPT_LSB       0x14
#define GC_USB_GHWCFG2_MULTIPROCINTRPT_MASK      0x100000
#define GC_USB_GHWCFG2_MULTIPROCINTRPT_SIZE      0x1
#define GC_USB_GHWCFG2_MULTIPROCINTRPT_DEFAULT   0x0
#define GC_USB_GHWCFG2_MULTIPROCINTRPT_OFFSET    0x48
#define GC_USB_GHWCFG2_NPTXQDEPTH_LSB            0x16
#define GC_USB_GHWCFG2_NPTXQDEPTH_MASK           0xc00000
#define GC_USB_GHWCFG2_NPTXQDEPTH_SIZE           0x2
#define GC_USB_GHWCFG2_NPTXQDEPTH_DEFAULT        0x0
#define GC_USB_GHWCFG2_NPTXQDEPTH_OFFSET         0x48
#define GC_USB_GHWCFG2_PTXQDEPTH_LSB             0x18
#define GC_USB_GHWCFG2_PTXQDEPTH_MASK            0x3000000
#define GC_USB_GHWCFG2_PTXQDEPTH_SIZE            0x2
#define GC_USB_GHWCFG2_PTXQDEPTH_DEFAULT         0x0
#define GC_USB_GHWCFG2_PTXQDEPTH_OFFSET          0x48
#define GC_USB_GHWCFG2_TKNQDEPTH_LSB             0x1a
#define GC_USB_GHWCFG2_TKNQDEPTH_MASK            0x7c000000
#define GC_USB_GHWCFG2_TKNQDEPTH_SIZE            0x5
#define GC_USB_GHWCFG2_TKNQDEPTH_DEFAULT         0x0
#define GC_USB_GHWCFG2_TKNQDEPTH_OFFSET          0x48
#define GC_USB_GHWCFG3_XFERSIZEWIDTH_LSB         0x0
#define GC_USB_GHWCFG3_XFERSIZEWIDTH_MASK        0xf
#define GC_USB_GHWCFG3_XFERSIZEWIDTH_SIZE        0x4
#define GC_USB_GHWCFG3_XFERSIZEWIDTH_DEFAULT     0x0
#define GC_USB_GHWCFG3_XFERSIZEWIDTH_OFFSET      0x4c
#define GC_USB_GHWCFG3_PKTSIZEWIDTH_LSB          0x4
#define GC_USB_GHWCFG3_PKTSIZEWIDTH_MASK         0x70
#define GC_USB_GHWCFG3_PKTSIZEWIDTH_SIZE         0x3
#define GC_USB_GHWCFG3_PKTSIZEWIDTH_DEFAULT      0x0
#define GC_USB_GHWCFG3_PKTSIZEWIDTH_OFFSET       0x4c
#define GC_USB_GHWCFG3_OTGEN_LSB                 0x7
#define GC_USB_GHWCFG3_OTGEN_MASK                0x80
#define GC_USB_GHWCFG3_OTGEN_SIZE                0x1
#define GC_USB_GHWCFG3_OTGEN_DEFAULT             0x0
#define GC_USB_GHWCFG3_OTGEN_OFFSET              0x4c
#define GC_USB_GHWCFG3_I2CINTSEL_LSB             0x8
#define GC_USB_GHWCFG3_I2CINTSEL_MASK            0x100
#define GC_USB_GHWCFG3_I2CINTSEL_SIZE            0x1
#define GC_USB_GHWCFG3_I2CINTSEL_DEFAULT         0x0
#define GC_USB_GHWCFG3_I2CINTSEL_OFFSET          0x4c
#define GC_USB_GHWCFG3_VNDCTLSUPT_LSB            0x9
#define GC_USB_GHWCFG3_VNDCTLSUPT_MASK           0x200
#define GC_USB_GHWCFG3_VNDCTLSUPT_SIZE           0x1
#define GC_USB_GHWCFG3_VNDCTLSUPT_DEFAULT        0x0
#define GC_USB_GHWCFG3_VNDCTLSUPT_OFFSET         0x4c
#define GC_USB_GHWCFG3_OPTFEATURE_LSB            0xa
#define GC_USB_GHWCFG3_OPTFEATURE_MASK           0x400
#define GC_USB_GHWCFG3_OPTFEATURE_SIZE           0x1
#define GC_USB_GHWCFG3_OPTFEATURE_DEFAULT        0x0
#define GC_USB_GHWCFG3_OPTFEATURE_OFFSET         0x4c
#define GC_USB_GHWCFG3_RSTTYPE_LSB               0xb
#define GC_USB_GHWCFG3_RSTTYPE_MASK              0x800
#define GC_USB_GHWCFG3_RSTTYPE_SIZE              0x1
#define GC_USB_GHWCFG3_RSTTYPE_DEFAULT           0x0
#define GC_USB_GHWCFG3_RSTTYPE_OFFSET            0x4c
#define GC_USB_GHWCFG3_ADPSUPPORT_LSB            0xc
#define GC_USB_GHWCFG3_ADPSUPPORT_MASK           0x1000
#define GC_USB_GHWCFG3_ADPSUPPORT_SIZE           0x1
#define GC_USB_GHWCFG3_ADPSUPPORT_DEFAULT        0x0
#define GC_USB_GHWCFG3_ADPSUPPORT_OFFSET         0x4c
#define GC_USB_GHWCFG3_HSICMODE_LSB              0xd
#define GC_USB_GHWCFG3_HSICMODE_MASK             0x2000
#define GC_USB_GHWCFG3_HSICMODE_SIZE             0x1
#define GC_USB_GHWCFG3_HSICMODE_DEFAULT          0x0
#define GC_USB_GHWCFG3_HSICMODE_OFFSET           0x4c
#define GC_USB_GHWCFG3_BCSUPPORT_LSB             0xe
#define GC_USB_GHWCFG3_BCSUPPORT_MASK            0x4000
#define GC_USB_GHWCFG3_BCSUPPORT_SIZE            0x1
#define GC_USB_GHWCFG3_BCSUPPORT_DEFAULT         0x0
#define GC_USB_GHWCFG3_BCSUPPORT_OFFSET          0x4c
#define GC_USB_GHWCFG3_LPMMODE_LSB               0xf
#define GC_USB_GHWCFG3_LPMMODE_MASK              0x8000
#define GC_USB_GHWCFG3_LPMMODE_SIZE              0x1
#define GC_USB_GHWCFG3_LPMMODE_DEFAULT           0x0
#define GC_USB_GHWCFG3_LPMMODE_OFFSET            0x4c
#define GC_USB_GHWCFG3_DFIFODEPTH_LSB            0x10
#define GC_USB_GHWCFG3_DFIFODEPTH_MASK           0xffff0000
#define GC_USB_GHWCFG3_DFIFODEPTH_SIZE           0x10
#define GC_USB_GHWCFG3_DFIFODEPTH_DEFAULT        0x0
#define GC_USB_GHWCFG3_DFIFODEPTH_OFFSET         0x4c
#define GC_USB_GHWCFG4_NUMDEVPERIOEPS_LSB        0x0
#define GC_USB_GHWCFG4_NUMDEVPERIOEPS_MASK       0xf
#define GC_USB_GHWCFG4_NUMDEVPERIOEPS_SIZE       0x4
#define GC_USB_GHWCFG4_NUMDEVPERIOEPS_DEFAULT    0x0
#define GC_USB_GHWCFG4_NUMDEVPERIOEPS_OFFSET     0x50
#define GC_USB_GHWCFG4_PARTIALPWRDN_LSB          0x4
#define GC_USB_GHWCFG4_PARTIALPWRDN_MASK         0x10
#define GC_USB_GHWCFG4_PARTIALPWRDN_SIZE         0x1
#define GC_USB_GHWCFG4_PARTIALPWRDN_DEFAULT      0x0
#define GC_USB_GHWCFG4_PARTIALPWRDN_OFFSET       0x50
#define GC_USB_GHWCFG4_AHBFREQ_LSB               0x5
#define GC_USB_GHWCFG4_AHBFREQ_MASK              0x20
#define GC_USB_GHWCFG4_AHBFREQ_SIZE              0x1
#define GC_USB_GHWCFG4_AHBFREQ_DEFAULT           0x0
#define GC_USB_GHWCFG4_AHBFREQ_OFFSET            0x50
#define GC_USB_GHWCFG4_HIBERNATION_LSB           0x6
#define GC_USB_GHWCFG4_HIBERNATION_MASK          0x40
#define GC_USB_GHWCFG4_HIBERNATION_SIZE          0x1
#define GC_USB_GHWCFG4_HIBERNATION_DEFAULT       0x0
#define GC_USB_GHWCFG4_HIBERNATION_OFFSET        0x50
#define GC_USB_GHWCFG4_EXTENDEDHIBERNATION_LSB   0x7
#define GC_USB_GHWCFG4_EXTENDEDHIBERNATION_MASK  0x80
#define GC_USB_GHWCFG4_EXTENDEDHIBERNATION_SIZE  0x1
#define GC_USB_GHWCFG4_EXTENDEDHIBERNATION_DEFAULT 0x0
#define GC_USB_GHWCFG4_EXTENDEDHIBERNATION_OFFSET 0x50
#define GC_USB_GHWCFG4_PHYDATAWIDTH_LSB          0xe
#define GC_USB_GHWCFG4_PHYDATAWIDTH_MASK         0xc000
#define GC_USB_GHWCFG4_PHYDATAWIDTH_SIZE         0x2
#define GC_USB_GHWCFG4_PHYDATAWIDTH_DEFAULT      0x0
#define GC_USB_GHWCFG4_PHYDATAWIDTH_OFFSET       0x50
#define GC_USB_GHWCFG4_NUMCTLEPS_LSB             0x10
#define GC_USB_GHWCFG4_NUMCTLEPS_MASK            0xf0000
#define GC_USB_GHWCFG4_NUMCTLEPS_SIZE            0x4
#define GC_USB_GHWCFG4_NUMCTLEPS_DEFAULT         0x0
#define GC_USB_GHWCFG4_NUMCTLEPS_OFFSET          0x50
#define GC_USB_GHWCFG4_IDDGFLTR_LSB              0x14
#define GC_USB_GHWCFG4_IDDGFLTR_MASK             0x100000
#define GC_USB_GHWCFG4_IDDGFLTR_SIZE             0x1
#define GC_USB_GHWCFG4_IDDGFLTR_DEFAULT          0x0
#define GC_USB_GHWCFG4_IDDGFLTR_OFFSET           0x50
#define GC_USB_GHWCFG4_VBUSVALIDFLTR_LSB         0x15
#define GC_USB_GHWCFG4_VBUSVALIDFLTR_MASK        0x200000
#define GC_USB_GHWCFG4_VBUSVALIDFLTR_SIZE        0x1
#define GC_USB_GHWCFG4_VBUSVALIDFLTR_DEFAULT     0x0
#define GC_USB_GHWCFG4_VBUSVALIDFLTR_OFFSET      0x50
#define GC_USB_GHWCFG4_AVALIDFLTR_LSB            0x16
#define GC_USB_GHWCFG4_AVALIDFLTR_MASK           0x400000
#define GC_USB_GHWCFG4_AVALIDFLTR_SIZE           0x1
#define GC_USB_GHWCFG4_AVALIDFLTR_DEFAULT        0x0
#define GC_USB_GHWCFG4_AVALIDFLTR_OFFSET         0x50
#define GC_USB_GHWCFG4_BVALIDFLTR_LSB            0x17
#define GC_USB_GHWCFG4_BVALIDFLTR_MASK           0x800000
#define GC_USB_GHWCFG4_BVALIDFLTR_SIZE           0x1
#define GC_USB_GHWCFG4_BVALIDFLTR_DEFAULT        0x0
#define GC_USB_GHWCFG4_BVALIDFLTR_OFFSET         0x50
#define GC_USB_GHWCFG4_SESSENDFLTR_LSB           0x18
#define GC_USB_GHWCFG4_SESSENDFLTR_MASK          0x1000000
#define GC_USB_GHWCFG4_SESSENDFLTR_SIZE          0x1
#define GC_USB_GHWCFG4_SESSENDFLTR_DEFAULT       0x0
#define GC_USB_GHWCFG4_SESSENDFLTR_OFFSET        0x50
#define GC_USB_GHWCFG4_DEDFIFOMODE_LSB           0x19
#define GC_USB_GHWCFG4_DEDFIFOMODE_MASK          0x2000000
#define GC_USB_GHWCFG4_DEDFIFOMODE_SIZE          0x1
#define GC_USB_GHWCFG4_DEDFIFOMODE_DEFAULT       0x0
#define GC_USB_GHWCFG4_DEDFIFOMODE_OFFSET        0x50
#define GC_USB_GHWCFG4_INEPS_LSB                 0x1a
#define GC_USB_GHWCFG4_INEPS_MASK                0x3c000000
#define GC_USB_GHWCFG4_INEPS_SIZE                0x4
#define GC_USB_GHWCFG4_INEPS_DEFAULT             0x0
#define GC_USB_GHWCFG4_INEPS_OFFSET              0x50
#define GC_USB_GHWCFG4_DESCDMAENABLED_LSB        0x1e
#define GC_USB_GHWCFG4_DESCDMAENABLED_MASK       0x40000000
#define GC_USB_GHWCFG4_DESCDMAENABLED_SIZE       0x1
#define GC_USB_GHWCFG4_DESCDMAENABLED_DEFAULT    0x0
#define GC_USB_GHWCFG4_DESCDMAENABLED_OFFSET     0x50
#define GC_USB_GHWCFG4_DESCDMA_LSB               0x1f
#define GC_USB_GHWCFG4_DESCDMA_MASK              0x80000000
#define GC_USB_GHWCFG4_DESCDMA_SIZE              0x1
#define GC_USB_GHWCFG4_DESCDMA_DEFAULT           0x0
#define GC_USB_GHWCFG4_DESCDMA_OFFSET            0x50
#define GC_USB_GDFIFOCFG_GDFIFOCFG_LSB           0x0
#define GC_USB_GDFIFOCFG_GDFIFOCFG_MASK          0xffff
#define GC_USB_GDFIFOCFG_GDFIFOCFG_SIZE          0x10
#define GC_USB_GDFIFOCFG_GDFIFOCFG_DEFAULT       0x0
#define GC_USB_GDFIFOCFG_GDFIFOCFG_OFFSET        0x5c
#define GC_USB_GDFIFOCFG_EPINFOBASEADDR_LSB      0x10
#define GC_USB_GDFIFOCFG_EPINFOBASEADDR_MASK     0xffff0000
#define GC_USB_GDFIFOCFG_EPINFOBASEADDR_SIZE     0x10
#define GC_USB_GDFIFOCFG_EPINFOBASEADDR_DEFAULT  0x0
#define GC_USB_GDFIFOCFG_EPINFOBASEADDR_OFFSET   0x5c
#define GC_USB_DIEPTXF1_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF1_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF1_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF1_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF1_INEPNTXFSTADDR_OFFSET    0x104
#define GC_USB_DIEPTXF1_RESERVED11_LSB           0xc
#define GC_USB_DIEPTXF1_RESERVED11_MASK          0x1000
#define GC_USB_DIEPTXF1_RESERVED11_SIZE          0x1
#define GC_USB_DIEPTXF1_RESERVED11_DEFAULT       0x1
#define GC_USB_DIEPTXF1_RESERVED11_OFFSET        0x104
#define GC_USB_DIEPTXF1_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF1_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF1_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF1_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF1_INEPNTXFDEP_OFFSET       0x104
#define GC_USB_DIEPTXF2_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF2_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF2_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF2_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF2_INEPNTXFSTADDR_OFFSET    0x108
#define GC_USB_DIEPTXF2_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF2_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF2_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF2_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF2_INEPNTXFDEP_OFFSET       0x108
#define GC_USB_DIEPTXF3_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF3_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF3_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF3_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF3_INEPNTXFSTADDR_OFFSET    0x10c
#define GC_USB_DIEPTXF3_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF3_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF3_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF3_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF3_INEPNTXFDEP_OFFSET       0x10c
#define GC_USB_DIEPTXF4_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF4_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF4_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF4_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF4_INEPNTXFSTADDR_OFFSET    0x110
#define GC_USB_DIEPTXF4_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF4_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF4_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF4_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF4_INEPNTXFDEP_OFFSET       0x110
#define GC_USB_DIEPTXF5_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF5_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF5_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF5_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF5_INEPNTXFSTADDR_OFFSET    0x114
#define GC_USB_DIEPTXF5_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF5_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF5_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF5_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF5_INEPNTXFDEP_OFFSET       0x114
#define GC_USB_DIEPTXF6_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF6_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF6_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF6_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF6_INEPNTXFSTADDR_OFFSET    0x118
#define GC_USB_DIEPTXF6_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF6_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF6_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF6_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF6_INEPNTXFDEP_OFFSET       0x118
#define GC_USB_DIEPTXF7_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF7_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF7_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF7_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF7_INEPNTXFSTADDR_OFFSET    0x11c
#define GC_USB_DIEPTXF7_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF7_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF7_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF7_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF7_INEPNTXFDEP_OFFSET       0x11c
#define GC_USB_DIEPTXF8_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF8_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF8_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF8_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF8_INEPNTXFSTADDR_OFFSET    0x120
#define GC_USB_DIEPTXF8_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF8_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF8_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF8_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF8_INEPNTXFDEP_OFFSET       0x120
#define GC_USB_DIEPTXF9_INEPNTXFSTADDR_LSB       0x0
#define GC_USB_DIEPTXF9_INEPNTXFSTADDR_MASK      0x7ff
#define GC_USB_DIEPTXF9_INEPNTXFSTADDR_SIZE      0xb
#define GC_USB_DIEPTXF9_INEPNTXFSTADDR_DEFAULT   0x0
#define GC_USB_DIEPTXF9_INEPNTXFSTADDR_OFFSET    0x124
#define GC_USB_DIEPTXF9_INEPNTXFDEP_LSB          0x10
#define GC_USB_DIEPTXF9_INEPNTXFDEP_MASK         0x3f0000
#define GC_USB_DIEPTXF9_INEPNTXFDEP_SIZE         0x6
#define GC_USB_DIEPTXF9_INEPNTXFDEP_DEFAULT      0x0
#define GC_USB_DIEPTXF9_INEPNTXFDEP_OFFSET       0x124
#define GC_USB_DIEPTXF10_INEPNTXFSTADDR_LSB      0x0
#define GC_USB_DIEPTXF10_INEPNTXFSTADDR_MASK     0x7ff
#define GC_USB_DIEPTXF10_INEPNTXFSTADDR_SIZE     0xb
#define GC_USB_DIEPTXF10_INEPNTXFSTADDR_DEFAULT  0x0
#define GC_USB_DIEPTXF10_INEPNTXFSTADDR_OFFSET   0x128
#define GC_USB_DIEPTXF10_INEPNTXFDEP_LSB         0x10
#define GC_USB_DIEPTXF10_INEPNTXFDEP_MASK        0x3f0000
#define GC_USB_DIEPTXF10_INEPNTXFDEP_SIZE        0x6
#define GC_USB_DIEPTXF10_INEPNTXFDEP_DEFAULT     0x0
#define GC_USB_DIEPTXF10_INEPNTXFDEP_OFFSET      0x128
#define GC_USB_DIEPTXF11_INEPNTXFSTADDR_LSB      0x0
#define GC_USB_DIEPTXF11_INEPNTXFSTADDR_MASK     0x7ff
#define GC_USB_DIEPTXF11_INEPNTXFSTADDR_SIZE     0xb
#define GC_USB_DIEPTXF11_INEPNTXFSTADDR_DEFAULT  0x0
#define GC_USB_DIEPTXF11_INEPNTXFSTADDR_OFFSET   0x12c
#define GC_USB_DIEPTXF11_INEPNTXFDEP_LSB         0x10
#define GC_USB_DIEPTXF11_INEPNTXFDEP_MASK        0x3f0000
#define GC_USB_DIEPTXF11_INEPNTXFDEP_SIZE        0x6
#define GC_USB_DIEPTXF11_INEPNTXFDEP_DEFAULT     0x0
#define GC_USB_DIEPTXF11_INEPNTXFDEP_OFFSET      0x12c
#define GC_USB_DIEPTXF12_INEPNTXFSTADDR_LSB      0x0
#define GC_USB_DIEPTXF12_INEPNTXFSTADDR_MASK     0x7ff
#define GC_USB_DIEPTXF12_INEPNTXFSTADDR_SIZE     0xb
#define GC_USB_DIEPTXF12_INEPNTXFSTADDR_DEFAULT  0x0
#define GC_USB_DIEPTXF12_INEPNTXFSTADDR_OFFSET   0x130
#define GC_USB_DIEPTXF12_INEPNTXFDEP_LSB         0x10
#define GC_USB_DIEPTXF12_INEPNTXFDEP_MASK        0x3f0000
#define GC_USB_DIEPTXF12_INEPNTXFDEP_SIZE        0x6
#define GC_USB_DIEPTXF12_INEPNTXFDEP_DEFAULT     0x0
#define GC_USB_DIEPTXF12_INEPNTXFDEP_OFFSET      0x130
#define GC_USB_DIEPTXF13_INEPNTXFSTADDR_LSB      0x0
#define GC_USB_DIEPTXF13_INEPNTXFSTADDR_MASK     0x7ff
#define GC_USB_DIEPTXF13_INEPNTXFSTADDR_SIZE     0xb
#define GC_USB_DIEPTXF13_INEPNTXFSTADDR_DEFAULT  0x0
#define GC_USB_DIEPTXF13_INEPNTXFSTADDR_OFFSET   0x134
#define GC_USB_DIEPTXF13_INEPNTXFDEP_LSB         0x10
#define GC_USB_DIEPTXF13_INEPNTXFDEP_MASK        0x3f0000
#define GC_USB_DIEPTXF13_INEPNTXFDEP_SIZE        0x6
#define GC_USB_DIEPTXF13_INEPNTXFDEP_DEFAULT     0x0
#define GC_USB_DIEPTXF13_INEPNTXFDEP_OFFSET      0x134
#define GC_USB_DIEPTXF14_INEPNTXFSTADDR_LSB      0x0
#define GC_USB_DIEPTXF14_INEPNTXFSTADDR_MASK     0x7ff
#define GC_USB_DIEPTXF14_INEPNTXFSTADDR_SIZE     0xb
#define GC_USB_DIEPTXF14_INEPNTXFSTADDR_DEFAULT  0x0
#define GC_USB_DIEPTXF14_INEPNTXFSTADDR_OFFSET   0x138
#define GC_USB_DIEPTXF14_INEPNTXFDEP_LSB         0x10
#define GC_USB_DIEPTXF14_INEPNTXFDEP_MASK        0x3f0000
#define GC_USB_DIEPTXF14_INEPNTXFDEP_SIZE        0x6
#define GC_USB_DIEPTXF14_INEPNTXFDEP_DEFAULT     0x0
#define GC_USB_DIEPTXF14_INEPNTXFDEP_OFFSET      0x138
#define GC_USB_DIEPTXF15_INEPNTXFSTADDR_LSB      0x0
#define GC_USB_DIEPTXF15_INEPNTXFSTADDR_MASK     0x7ff
#define GC_USB_DIEPTXF15_INEPNTXFSTADDR_SIZE     0xb
#define GC_USB_DIEPTXF15_INEPNTXFSTADDR_DEFAULT  0x0
#define GC_USB_DIEPTXF15_INEPNTXFSTADDR_OFFSET   0x13c
#define GC_USB_DIEPTXF15_INEPNTXFDEP_LSB         0x10
#define GC_USB_DIEPTXF15_INEPNTXFDEP_MASK        0x3f0000
#define GC_USB_DIEPTXF15_INEPNTXFDEP_SIZE        0x6
#define GC_USB_DIEPTXF15_INEPNTXFDEP_DEFAULT     0x0
#define GC_USB_DIEPTXF15_INEPNTXFDEP_OFFSET      0x13c
#define GC_USB_DCFG_DEVSPD_LSB                   0x0
#define GC_USB_DCFG_DEVSPD_MASK                  0x3
#define GC_USB_DCFG_DEVSPD_SIZE                  0x2
#define GC_USB_DCFG_DEVSPD_DEFAULT               0x0
#define GC_USB_DCFG_DEVSPD_OFFSET                0x800
#define GC_USB_DCFG_NZSTSOUTHSHK_LSB             0x2
#define GC_USB_DCFG_NZSTSOUTHSHK_MASK            0x4
#define GC_USB_DCFG_NZSTSOUTHSHK_SIZE            0x1
#define GC_USB_DCFG_NZSTSOUTHSHK_DEFAULT         0x0
#define GC_USB_DCFG_NZSTSOUTHSHK_OFFSET          0x800
#define GC_USB_DCFG_ENA32KHZSUSP_LSB             0x3
#define GC_USB_DCFG_ENA32KHZSUSP_MASK            0x8
#define GC_USB_DCFG_ENA32KHZSUSP_SIZE            0x1
#define GC_USB_DCFG_ENA32KHZSUSP_DEFAULT         0x0
#define GC_USB_DCFG_ENA32KHZSUSP_OFFSET          0x800
#define GC_USB_DCFG_DEVADDR_LSB                  0x4
#define GC_USB_DCFG_DEVADDR_MASK                 0x7f0
#define GC_USB_DCFG_DEVADDR_SIZE                 0x7
#define GC_USB_DCFG_DEVADDR_DEFAULT              0x0
#define GC_USB_DCFG_DEVADDR_OFFSET               0x800
#define GC_USB_DCFG_PERFRINT_LSB                 0xb
#define GC_USB_DCFG_PERFRINT_MASK                0x1800
#define GC_USB_DCFG_PERFRINT_SIZE                0x2
#define GC_USB_DCFG_PERFRINT_DEFAULT             0x0
#define GC_USB_DCFG_PERFRINT_OFFSET              0x800
#define GC_USB_DCFG_ENDEVOUTNAK_LSB              0xd
#define GC_USB_DCFG_ENDEVOUTNAK_MASK             0x2000
#define GC_USB_DCFG_ENDEVOUTNAK_SIZE             0x1
#define GC_USB_DCFG_ENDEVOUTNAK_DEFAULT          0x0
#define GC_USB_DCFG_ENDEVOUTNAK_OFFSET           0x800
#define GC_USB_DCFG_XCVRDLY_LSB                  0xe
#define GC_USB_DCFG_XCVRDLY_MASK                 0x4000
#define GC_USB_DCFG_XCVRDLY_SIZE                 0x1
#define GC_USB_DCFG_XCVRDLY_DEFAULT              0x0
#define GC_USB_DCFG_XCVRDLY_OFFSET               0x800
#define GC_USB_DCFG_ERRATICINTMSK_LSB            0xf
#define GC_USB_DCFG_ERRATICINTMSK_MASK           0x8000
#define GC_USB_DCFG_ERRATICINTMSK_SIZE           0x1
#define GC_USB_DCFG_ERRATICINTMSK_DEFAULT        0x0
#define GC_USB_DCFG_ERRATICINTMSK_OFFSET         0x800
#define GC_USB_DCFG_DESCDMA_LSB                  0x17
#define GC_USB_DCFG_DESCDMA_MASK                 0x800000
#define GC_USB_DCFG_DESCDMA_SIZE                 0x1
#define GC_USB_DCFG_DESCDMA_DEFAULT              0x0
#define GC_USB_DCFG_DESCDMA_OFFSET               0x800
#define GC_USB_DCFG_PERSCHINTVL_LSB              0x18
#define GC_USB_DCFG_PERSCHINTVL_MASK             0x3000000
#define GC_USB_DCFG_PERSCHINTVL_SIZE             0x2
#define GC_USB_DCFG_PERSCHINTVL_DEFAULT          0x0
#define GC_USB_DCFG_PERSCHINTVL_OFFSET           0x800
#define GC_USB_DCFG_RESVALID_LSB                 0x1a
#define GC_USB_DCFG_RESVALID_MASK                0xfc000000
#define GC_USB_DCFG_RESVALID_SIZE                0x6
#define GC_USB_DCFG_RESVALID_DEFAULT             0x2
#define GC_USB_DCFG_RESVALID_OFFSET              0x800
#define GC_USB_DCTL_RMTWKUPSIG_LSB               0x0
#define GC_USB_DCTL_RMTWKUPSIG_MASK              0x1
#define GC_USB_DCTL_RMTWKUPSIG_SIZE              0x1
#define GC_USB_DCTL_RMTWKUPSIG_DEFAULT           0x0
#define GC_USB_DCTL_RMTWKUPSIG_OFFSET            0x804
#define GC_USB_DCTL_SFTDISCON_LSB                0x1
#define GC_USB_DCTL_SFTDISCON_MASK               0x2
#define GC_USB_DCTL_SFTDISCON_SIZE               0x1
#define GC_USB_DCTL_SFTDISCON_DEFAULT            0x0
#define GC_USB_DCTL_SFTDISCON_OFFSET             0x804
#define GC_USB_DCTL_GNPINNAKSTS_LSB              0x2
#define GC_USB_DCTL_GNPINNAKSTS_MASK             0x4
#define GC_USB_DCTL_GNPINNAKSTS_SIZE             0x1
#define GC_USB_DCTL_GNPINNAKSTS_DEFAULT          0x0
#define GC_USB_DCTL_GNPINNAKSTS_OFFSET           0x804
#define GC_USB_DCTL_GOUTNAKSTS_LSB               0x3
#define GC_USB_DCTL_GOUTNAKSTS_MASK              0x8
#define GC_USB_DCTL_GOUTNAKSTS_SIZE              0x1
#define GC_USB_DCTL_GOUTNAKSTS_DEFAULT           0x0
#define GC_USB_DCTL_GOUTNAKSTS_OFFSET            0x804
#define GC_USB_DCTL_TSTCTL_LSB                   0x4
#define GC_USB_DCTL_TSTCTL_MASK                  0x70
#define GC_USB_DCTL_TSTCTL_SIZE                  0x3
#define GC_USB_DCTL_TSTCTL_DEFAULT               0x0
#define GC_USB_DCTL_TSTCTL_OFFSET                0x804
#define GC_USB_DCTL_SGNPINNAK_LSB                0x7
#define GC_USB_DCTL_SGNPINNAK_MASK               0x80
#define GC_USB_DCTL_SGNPINNAK_SIZE               0x1
#define GC_USB_DCTL_SGNPINNAK_DEFAULT            0x0
#define GC_USB_DCTL_SGNPINNAK_OFFSET             0x804
#define GC_USB_DCTL_CGNPINNAK_LSB                0x8
#define GC_USB_DCTL_CGNPINNAK_MASK               0x100
#define GC_USB_DCTL_CGNPINNAK_SIZE               0x1
#define GC_USB_DCTL_CGNPINNAK_DEFAULT            0x0
#define GC_USB_DCTL_CGNPINNAK_OFFSET             0x804
#define GC_USB_DCTL_SGOUTNAK_LSB                 0x9
#define GC_USB_DCTL_SGOUTNAK_MASK                0x200
#define GC_USB_DCTL_SGOUTNAK_SIZE                0x1
#define GC_USB_DCTL_SGOUTNAK_DEFAULT             0x0
#define GC_USB_DCTL_SGOUTNAK_OFFSET              0x804
#define GC_USB_DCTL_CGOUTNAK_LSB                 0xa
#define GC_USB_DCTL_CGOUTNAK_MASK                0x400
#define GC_USB_DCTL_CGOUTNAK_SIZE                0x1
#define GC_USB_DCTL_CGOUTNAK_DEFAULT             0x0
#define GC_USB_DCTL_CGOUTNAK_OFFSET              0x804
#define GC_USB_DCTL_PWRONPRGDONE_LSB             0xb
#define GC_USB_DCTL_PWRONPRGDONE_MASK            0x800
#define GC_USB_DCTL_PWRONPRGDONE_SIZE            0x1
#define GC_USB_DCTL_PWRONPRGDONE_DEFAULT         0x0
#define GC_USB_DCTL_PWRONPRGDONE_OFFSET          0x804
#define GC_USB_DCTL_GMC_LSB                      0xd
#define GC_USB_DCTL_GMC_MASK                     0x6000
#define GC_USB_DCTL_GMC_SIZE                     0x2
#define GC_USB_DCTL_GMC_DEFAULT                  0x0
#define GC_USB_DCTL_GMC_OFFSET                   0x804
#define GC_USB_DCTL_IGNRFRMNUM_LSB               0xf
#define GC_USB_DCTL_IGNRFRMNUM_MASK              0x8000
#define GC_USB_DCTL_IGNRFRMNUM_SIZE              0x1
#define GC_USB_DCTL_IGNRFRMNUM_DEFAULT           0x0
#define GC_USB_DCTL_IGNRFRMNUM_OFFSET            0x804
#define GC_USB_DCTL_NAKONBBLE_LSB                0x10
#define GC_USB_DCTL_NAKONBBLE_MASK               0x10000
#define GC_USB_DCTL_NAKONBBLE_SIZE               0x1
#define GC_USB_DCTL_NAKONBBLE_DEFAULT            0x0
#define GC_USB_DCTL_NAKONBBLE_OFFSET             0x804
#define GC_USB_DCTL_ENCONTONBNA_LSB              0x11
#define GC_USB_DCTL_ENCONTONBNA_MASK             0x20000
#define GC_USB_DCTL_ENCONTONBNA_SIZE             0x1
#define GC_USB_DCTL_ENCONTONBNA_DEFAULT          0x0
#define GC_USB_DCTL_ENCONTONBNA_OFFSET           0x804
#define GC_USB_DSTS_SUSPSTS_LSB                  0x0
#define GC_USB_DSTS_SUSPSTS_MASK                 0x1
#define GC_USB_DSTS_SUSPSTS_SIZE                 0x1
#define GC_USB_DSTS_SUSPSTS_DEFAULT              0x0
#define GC_USB_DSTS_SUSPSTS_OFFSET               0x808
#define GC_USB_DSTS_ENUMSPD_LSB                  0x1
#define GC_USB_DSTS_ENUMSPD_MASK                 0x6
#define GC_USB_DSTS_ENUMSPD_SIZE                 0x2
#define GC_USB_DSTS_ENUMSPD_DEFAULT              0x0
#define GC_USB_DSTS_ENUMSPD_OFFSET               0x808
#define GC_USB_DSTS_ERRTICERR_LSB                0x3
#define GC_USB_DSTS_ERRTICERR_MASK               0x8
#define GC_USB_DSTS_ERRTICERR_SIZE               0x1
#define GC_USB_DSTS_ERRTICERR_DEFAULT            0x0
#define GC_USB_DSTS_ERRTICERR_OFFSET             0x808
#define GC_USB_DSTS_SOFFN_LSB                    0x8
#define GC_USB_DSTS_SOFFN_MASK                   0x3fff00
#define GC_USB_DSTS_SOFFN_SIZE                   0xe
#define GC_USB_DSTS_SOFFN_DEFAULT                0x0
#define GC_USB_DSTS_SOFFN_OFFSET                 0x808
#define GC_USB_DSTS_DEVLNSTS_LSB                 0x16
#define GC_USB_DSTS_DEVLNSTS_MASK                0xc00000
#define GC_USB_DSTS_DEVLNSTS_SIZE                0x2
#define GC_USB_DSTS_DEVLNSTS_DEFAULT             0x0
#define GC_USB_DSTS_DEVLNSTS_OFFSET              0x808
#define GC_USB_DIEPMSK_XFERCOMPLMSK_LSB          0x0
#define GC_USB_DIEPMSK_XFERCOMPLMSK_MASK         0x1
#define GC_USB_DIEPMSK_XFERCOMPLMSK_SIZE         0x1
#define GC_USB_DIEPMSK_XFERCOMPLMSK_DEFAULT      0x0
#define GC_USB_DIEPMSK_XFERCOMPLMSK_OFFSET       0x810
#define GC_USB_DIEPMSK_EPDISBLDMSK_LSB           0x1
#define GC_USB_DIEPMSK_EPDISBLDMSK_MASK          0x2
#define GC_USB_DIEPMSK_EPDISBLDMSK_SIZE          0x1
#define GC_USB_DIEPMSK_EPDISBLDMSK_DEFAULT       0x0
#define GC_USB_DIEPMSK_EPDISBLDMSK_OFFSET        0x810
#define GC_USB_DIEPMSK_AHBERRMSK_LSB             0x2
#define GC_USB_DIEPMSK_AHBERRMSK_MASK            0x4
#define GC_USB_DIEPMSK_AHBERRMSK_SIZE            0x1
#define GC_USB_DIEPMSK_AHBERRMSK_DEFAULT         0x0
#define GC_USB_DIEPMSK_AHBERRMSK_OFFSET          0x810
#define GC_USB_DIEPMSK_TIMEOUTMSK_LSB            0x3
#define GC_USB_DIEPMSK_TIMEOUTMSK_MASK           0x8
#define GC_USB_DIEPMSK_TIMEOUTMSK_SIZE           0x1
#define GC_USB_DIEPMSK_TIMEOUTMSK_DEFAULT        0x0
#define GC_USB_DIEPMSK_TIMEOUTMSK_OFFSET         0x810
#define GC_USB_DIEPMSK_INTKNTXFEMPMSK_LSB        0x4
#define GC_USB_DIEPMSK_INTKNTXFEMPMSK_MASK       0x10
#define GC_USB_DIEPMSK_INTKNTXFEMPMSK_SIZE       0x1
#define GC_USB_DIEPMSK_INTKNTXFEMPMSK_DEFAULT    0x0
#define GC_USB_DIEPMSK_INTKNTXFEMPMSK_OFFSET     0x810
#define GC_USB_DIEPMSK_INTKNEPMISMSK_LSB         0x5
#define GC_USB_DIEPMSK_INTKNEPMISMSK_MASK        0x20
#define GC_USB_DIEPMSK_INTKNEPMISMSK_SIZE        0x1
#define GC_USB_DIEPMSK_INTKNEPMISMSK_DEFAULT     0x0
#define GC_USB_DIEPMSK_INTKNEPMISMSK_OFFSET      0x810
#define GC_USB_DIEPMSK_INEPNAKEFFMSK_LSB         0x6
#define GC_USB_DIEPMSK_INEPNAKEFFMSK_MASK        0x40
#define GC_USB_DIEPMSK_INEPNAKEFFMSK_SIZE        0x1
#define GC_USB_DIEPMSK_INEPNAKEFFMSK_DEFAULT     0x0
#define GC_USB_DIEPMSK_INEPNAKEFFMSK_OFFSET      0x810
#define GC_USB_DIEPMSK_RESERVED7_LSB             0x7
#define GC_USB_DIEPMSK_RESERVED7_MASK            0x80
#define GC_USB_DIEPMSK_RESERVED7_SIZE            0x1
#define GC_USB_DIEPMSK_RESERVED7_DEFAULT         0x1
#define GC_USB_DIEPMSK_RESERVED7_OFFSET          0x810
#define GC_USB_DIEPMSK_TXFIFOUNDRNMSK_LSB        0x8
#define GC_USB_DIEPMSK_TXFIFOUNDRNMSK_MASK       0x100
#define GC_USB_DIEPMSK_TXFIFOUNDRNMSK_SIZE       0x1
#define GC_USB_DIEPMSK_TXFIFOUNDRNMSK_DEFAULT    0x0
#define GC_USB_DIEPMSK_TXFIFOUNDRNMSK_OFFSET     0x810
#define GC_USB_DIEPMSK_BNAININTRMSK_LSB          0x9
#define GC_USB_DIEPMSK_BNAININTRMSK_MASK         0x200
#define GC_USB_DIEPMSK_BNAININTRMSK_SIZE         0x1
#define GC_USB_DIEPMSK_BNAININTRMSK_DEFAULT      0x0
#define GC_USB_DIEPMSK_BNAININTRMSK_OFFSET       0x810
#define GC_USB_DIEPMSK_NAKMSK_LSB                0xd
#define GC_USB_DIEPMSK_NAKMSK_MASK               0x2000
#define GC_USB_DIEPMSK_NAKMSK_SIZE               0x1
#define GC_USB_DIEPMSK_NAKMSK_DEFAULT            0x0
#define GC_USB_DIEPMSK_NAKMSK_OFFSET             0x810
#define GC_USB_DOEPMSK_XFERCOMPLMSK_LSB          0x0
#define GC_USB_DOEPMSK_XFERCOMPLMSK_MASK         0x1
#define GC_USB_DOEPMSK_XFERCOMPLMSK_SIZE         0x1
#define GC_USB_DOEPMSK_XFERCOMPLMSK_DEFAULT      0x0
#define GC_USB_DOEPMSK_XFERCOMPLMSK_OFFSET       0x814
#define GC_USB_DOEPMSK_EPDISBLDMSK_LSB           0x1
#define GC_USB_DOEPMSK_EPDISBLDMSK_MASK          0x2
#define GC_USB_DOEPMSK_EPDISBLDMSK_SIZE          0x1
#define GC_USB_DOEPMSK_EPDISBLDMSK_DEFAULT       0x0
#define GC_USB_DOEPMSK_EPDISBLDMSK_OFFSET        0x814
#define GC_USB_DOEPMSK_AHBERRMSK_LSB             0x2
#define GC_USB_DOEPMSK_AHBERRMSK_MASK            0x4
#define GC_USB_DOEPMSK_AHBERRMSK_SIZE            0x1
#define GC_USB_DOEPMSK_AHBERRMSK_DEFAULT         0x0
#define GC_USB_DOEPMSK_AHBERRMSK_OFFSET          0x814
#define GC_USB_DOEPMSK_SETUPMSK_LSB              0x3
#define GC_USB_DOEPMSK_SETUPMSK_MASK             0x8
#define GC_USB_DOEPMSK_SETUPMSK_SIZE             0x1
#define GC_USB_DOEPMSK_SETUPMSK_DEFAULT          0x0
#define GC_USB_DOEPMSK_SETUPMSK_OFFSET           0x814
#define GC_USB_DOEPMSK_OUTTKNEPDISMSK_LSB        0x4
#define GC_USB_DOEPMSK_OUTTKNEPDISMSK_MASK       0x10
#define GC_USB_DOEPMSK_OUTTKNEPDISMSK_SIZE       0x1
#define GC_USB_DOEPMSK_OUTTKNEPDISMSK_DEFAULT    0x0
#define GC_USB_DOEPMSK_OUTTKNEPDISMSK_OFFSET     0x814
#define GC_USB_DOEPMSK_STSPHSERCVDMSK_LSB        0x5
#define GC_USB_DOEPMSK_STSPHSERCVDMSK_MASK       0x20
#define GC_USB_DOEPMSK_STSPHSERCVDMSK_SIZE       0x1
#define GC_USB_DOEPMSK_STSPHSERCVDMSK_DEFAULT    0x0
#define GC_USB_DOEPMSK_STSPHSERCVDMSK_OFFSET     0x814
#define GC_USB_DOEPMSK_BACK2BACKSETUP_LSB        0x6
#define GC_USB_DOEPMSK_BACK2BACKSETUP_MASK       0x40
#define GC_USB_DOEPMSK_BACK2BACKSETUP_SIZE       0x1
#define GC_USB_DOEPMSK_BACK2BACKSETUP_DEFAULT    0x0
#define GC_USB_DOEPMSK_BACK2BACKSETUP_OFFSET     0x814
#define GC_USB_DOEPMSK_OUTPKTERRMSK_LSB          0x8
#define GC_USB_DOEPMSK_OUTPKTERRMSK_MASK         0x100
#define GC_USB_DOEPMSK_OUTPKTERRMSK_SIZE         0x1
#define GC_USB_DOEPMSK_OUTPKTERRMSK_DEFAULT      0x0
#define GC_USB_DOEPMSK_OUTPKTERRMSK_OFFSET       0x814
#define GC_USB_DOEPMSK_BNAOUTINTRMSK_LSB         0x9
#define GC_USB_DOEPMSK_BNAOUTINTRMSK_MASK        0x200
#define GC_USB_DOEPMSK_BNAOUTINTRMSK_SIZE        0x1
#define GC_USB_DOEPMSK_BNAOUTINTRMSK_DEFAULT     0x0
#define GC_USB_DOEPMSK_BNAOUTINTRMSK_OFFSET      0x814
#define GC_USB_DOEPMSK_BBLEERRMSK_LSB            0xc
#define GC_USB_DOEPMSK_BBLEERRMSK_MASK           0x1000
#define GC_USB_DOEPMSK_BBLEERRMSK_SIZE           0x1
#define GC_USB_DOEPMSK_BBLEERRMSK_DEFAULT        0x0
#define GC_USB_DOEPMSK_BBLEERRMSK_OFFSET         0x814
#define GC_USB_DOEPMSK_NAKMSK_LSB                0xd
#define GC_USB_DOEPMSK_NAKMSK_MASK               0x2000
#define GC_USB_DOEPMSK_NAKMSK_SIZE               0x1
#define GC_USB_DOEPMSK_NAKMSK_DEFAULT            0x0
#define GC_USB_DOEPMSK_NAKMSK_OFFSET             0x814
#define GC_USB_DOEPMSK_NYETMSK_LSB               0xe
#define GC_USB_DOEPMSK_NYETMSK_MASK              0x4000
#define GC_USB_DOEPMSK_NYETMSK_SIZE              0x1
#define GC_USB_DOEPMSK_NYETMSK_DEFAULT           0x0
#define GC_USB_DOEPMSK_NYETMSK_OFFSET            0x814
#define GC_USB_DAINT_INEPINT0_LSB                0x0
#define GC_USB_DAINT_INEPINT0_MASK               0x1
#define GC_USB_DAINT_INEPINT0_SIZE               0x1
#define GC_USB_DAINT_INEPINT0_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT0_OFFSET             0x818
#define GC_USB_DAINT_INEPINT1_LSB                0x1
#define GC_USB_DAINT_INEPINT1_MASK               0x2
#define GC_USB_DAINT_INEPINT1_SIZE               0x1
#define GC_USB_DAINT_INEPINT1_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT1_OFFSET             0x818
#define GC_USB_DAINT_INEPINT2_LSB                0x2
#define GC_USB_DAINT_INEPINT2_MASK               0x4
#define GC_USB_DAINT_INEPINT2_SIZE               0x1
#define GC_USB_DAINT_INEPINT2_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT2_OFFSET             0x818
#define GC_USB_DAINT_INEPINT3_LSB                0x3
#define GC_USB_DAINT_INEPINT3_MASK               0x8
#define GC_USB_DAINT_INEPINT3_SIZE               0x1
#define GC_USB_DAINT_INEPINT3_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT3_OFFSET             0x818
#define GC_USB_DAINT_INEPINT4_LSB                0x4
#define GC_USB_DAINT_INEPINT4_MASK               0x10
#define GC_USB_DAINT_INEPINT4_SIZE               0x1
#define GC_USB_DAINT_INEPINT4_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT4_OFFSET             0x818
#define GC_USB_DAINT_INEPINT5_LSB                0x5
#define GC_USB_DAINT_INEPINT5_MASK               0x20
#define GC_USB_DAINT_INEPINT5_SIZE               0x1
#define GC_USB_DAINT_INEPINT5_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT5_OFFSET             0x818
#define GC_USB_DAINT_INEPINT6_LSB                0x6
#define GC_USB_DAINT_INEPINT6_MASK               0x40
#define GC_USB_DAINT_INEPINT6_SIZE               0x1
#define GC_USB_DAINT_INEPINT6_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT6_OFFSET             0x818
#define GC_USB_DAINT_INEPINT7_LSB                0x7
#define GC_USB_DAINT_INEPINT7_MASK               0x80
#define GC_USB_DAINT_INEPINT7_SIZE               0x1
#define GC_USB_DAINT_INEPINT7_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT7_OFFSET             0x818
#define GC_USB_DAINT_INEPINT8_LSB                0x8
#define GC_USB_DAINT_INEPINT8_MASK               0x100
#define GC_USB_DAINT_INEPINT8_SIZE               0x1
#define GC_USB_DAINT_INEPINT8_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT8_OFFSET             0x818
#define GC_USB_DAINT_INEPINT9_LSB                0x9
#define GC_USB_DAINT_INEPINT9_MASK               0x200
#define GC_USB_DAINT_INEPINT9_SIZE               0x1
#define GC_USB_DAINT_INEPINT9_DEFAULT            0x0
#define GC_USB_DAINT_INEPINT9_OFFSET             0x818
#define GC_USB_DAINT_INEPINT10_LSB               0xa
#define GC_USB_DAINT_INEPINT10_MASK              0x400
#define GC_USB_DAINT_INEPINT10_SIZE              0x1
#define GC_USB_DAINT_INEPINT10_DEFAULT           0x0
#define GC_USB_DAINT_INEPINT10_OFFSET            0x818
#define GC_USB_DAINT_INEPINT11_LSB               0xb
#define GC_USB_DAINT_INEPINT11_MASK              0x800
#define GC_USB_DAINT_INEPINT11_SIZE              0x1
#define GC_USB_DAINT_INEPINT11_DEFAULT           0x0
#define GC_USB_DAINT_INEPINT11_OFFSET            0x818
#define GC_USB_DAINT_INEPINT12_LSB               0xc
#define GC_USB_DAINT_INEPINT12_MASK              0x1000
#define GC_USB_DAINT_INEPINT12_SIZE              0x1
#define GC_USB_DAINT_INEPINT12_DEFAULT           0x0
#define GC_USB_DAINT_INEPINT12_OFFSET            0x818
#define GC_USB_DAINT_INEPINT13_LSB               0xd
#define GC_USB_DAINT_INEPINT13_MASK              0x2000
#define GC_USB_DAINT_INEPINT13_SIZE              0x1
#define GC_USB_DAINT_INEPINT13_DEFAULT           0x0
#define GC_USB_DAINT_INEPINT13_OFFSET            0x818
#define GC_USB_DAINT_INEPINT14_LSB               0xe
#define GC_USB_DAINT_INEPINT14_MASK              0x4000
#define GC_USB_DAINT_INEPINT14_SIZE              0x1
#define GC_USB_DAINT_INEPINT14_DEFAULT           0x0
#define GC_USB_DAINT_INEPINT14_OFFSET            0x818
#define GC_USB_DAINT_INEPINT15_LSB               0xf
#define GC_USB_DAINT_INEPINT15_MASK              0x8000
#define GC_USB_DAINT_INEPINT15_SIZE              0x1
#define GC_USB_DAINT_INEPINT15_DEFAULT           0x0
#define GC_USB_DAINT_INEPINT15_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT0_LSB               0x10
#define GC_USB_DAINT_OUTEPINT0_MASK              0x10000
#define GC_USB_DAINT_OUTEPINT0_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT0_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT0_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT1_LSB               0x11
#define GC_USB_DAINT_OUTEPINT1_MASK              0x20000
#define GC_USB_DAINT_OUTEPINT1_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT1_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT1_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT2_LSB               0x12
#define GC_USB_DAINT_OUTEPINT2_MASK              0x40000
#define GC_USB_DAINT_OUTEPINT2_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT2_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT2_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT3_LSB               0x13
#define GC_USB_DAINT_OUTEPINT3_MASK              0x80000
#define GC_USB_DAINT_OUTEPINT3_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT3_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT3_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT4_LSB               0x14
#define GC_USB_DAINT_OUTEPINT4_MASK              0x100000
#define GC_USB_DAINT_OUTEPINT4_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT4_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT4_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT5_LSB               0x15
#define GC_USB_DAINT_OUTEPINT5_MASK              0x200000
#define GC_USB_DAINT_OUTEPINT5_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT5_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT5_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT6_LSB               0x16
#define GC_USB_DAINT_OUTEPINT6_MASK              0x400000
#define GC_USB_DAINT_OUTEPINT6_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT6_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT6_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT7_LSB               0x17
#define GC_USB_DAINT_OUTEPINT7_MASK              0x800000
#define GC_USB_DAINT_OUTEPINT7_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT7_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT7_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT8_LSB               0x18
#define GC_USB_DAINT_OUTEPINT8_MASK              0x1000000
#define GC_USB_DAINT_OUTEPINT8_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT8_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT8_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT9_LSB               0x19
#define GC_USB_DAINT_OUTEPINT9_MASK              0x2000000
#define GC_USB_DAINT_OUTEPINT9_SIZE              0x1
#define GC_USB_DAINT_OUTEPINT9_DEFAULT           0x0
#define GC_USB_DAINT_OUTEPINT9_OFFSET            0x818
#define GC_USB_DAINT_OUTEPINT10_LSB              0x1a
#define GC_USB_DAINT_OUTEPINT10_MASK             0x4000000
#define GC_USB_DAINT_OUTEPINT10_SIZE             0x1
#define GC_USB_DAINT_OUTEPINT10_DEFAULT          0x0
#define GC_USB_DAINT_OUTEPINT10_OFFSET           0x818
#define GC_USB_DAINT_OUTEPINT11_LSB              0x1b
#define GC_USB_DAINT_OUTEPINT11_MASK             0x8000000
#define GC_USB_DAINT_OUTEPINT11_SIZE             0x1
#define GC_USB_DAINT_OUTEPINT11_DEFAULT          0x0
#define GC_USB_DAINT_OUTEPINT11_OFFSET           0x818
#define GC_USB_DAINT_OUTEPINT12_LSB              0x1c
#define GC_USB_DAINT_OUTEPINT12_MASK             0x10000000
#define GC_USB_DAINT_OUTEPINT12_SIZE             0x1
#define GC_USB_DAINT_OUTEPINT12_DEFAULT          0x0
#define GC_USB_DAINT_OUTEPINT12_OFFSET           0x818
#define GC_USB_DAINT_OUTEPINT13_LSB              0x1d
#define GC_USB_DAINT_OUTEPINT13_MASK             0x20000000
#define GC_USB_DAINT_OUTEPINT13_SIZE             0x1
#define GC_USB_DAINT_OUTEPINT13_DEFAULT          0x0
#define GC_USB_DAINT_OUTEPINT13_OFFSET           0x818
#define GC_USB_DAINT_OUTEPINT14_LSB              0x1e
#define GC_USB_DAINT_OUTEPINT14_MASK             0x40000000
#define GC_USB_DAINT_OUTEPINT14_SIZE             0x1
#define GC_USB_DAINT_OUTEPINT14_DEFAULT          0x0
#define GC_USB_DAINT_OUTEPINT14_OFFSET           0x818
#define GC_USB_DAINT_OUTEPINT15_LSB              0x1f
#define GC_USB_DAINT_OUTEPINT15_MASK             0x80000000
#define GC_USB_DAINT_OUTEPINT15_SIZE             0x1
#define GC_USB_DAINT_OUTEPINT15_DEFAULT          0x0
#define GC_USB_DAINT_OUTEPINT15_OFFSET           0x818
#define GC_USB_DAINTMSK_INEPMSK0_LSB             0x0
#define GC_USB_DAINTMSK_INEPMSK0_MASK            0x1
#define GC_USB_DAINTMSK_INEPMSK0_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK0_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK0_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK1_LSB             0x1
#define GC_USB_DAINTMSK_INEPMSK1_MASK            0x2
#define GC_USB_DAINTMSK_INEPMSK1_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK1_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK1_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK2_LSB             0x2
#define GC_USB_DAINTMSK_INEPMSK2_MASK            0x4
#define GC_USB_DAINTMSK_INEPMSK2_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK2_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK2_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK3_LSB             0x3
#define GC_USB_DAINTMSK_INEPMSK3_MASK            0x8
#define GC_USB_DAINTMSK_INEPMSK3_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK3_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK3_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK4_LSB             0x4
#define GC_USB_DAINTMSK_INEPMSK4_MASK            0x10
#define GC_USB_DAINTMSK_INEPMSK4_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK4_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK4_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK5_LSB             0x5
#define GC_USB_DAINTMSK_INEPMSK5_MASK            0x20
#define GC_USB_DAINTMSK_INEPMSK5_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK5_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK5_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK6_LSB             0x6
#define GC_USB_DAINTMSK_INEPMSK6_MASK            0x40
#define GC_USB_DAINTMSK_INEPMSK6_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK6_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK6_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK7_LSB             0x7
#define GC_USB_DAINTMSK_INEPMSK7_MASK            0x80
#define GC_USB_DAINTMSK_INEPMSK7_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK7_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK7_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK8_LSB             0x8
#define GC_USB_DAINTMSK_INEPMSK8_MASK            0x100
#define GC_USB_DAINTMSK_INEPMSK8_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK8_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK8_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK9_LSB             0x9
#define GC_USB_DAINTMSK_INEPMSK9_MASK            0x200
#define GC_USB_DAINTMSK_INEPMSK9_SIZE            0x1
#define GC_USB_DAINTMSK_INEPMSK9_DEFAULT         0x0
#define GC_USB_DAINTMSK_INEPMSK9_OFFSET          0x81c
#define GC_USB_DAINTMSK_INEPMSK10_LSB            0xa
#define GC_USB_DAINTMSK_INEPMSK10_MASK           0x400
#define GC_USB_DAINTMSK_INEPMSK10_SIZE           0x1
#define GC_USB_DAINTMSK_INEPMSK10_DEFAULT        0x0
#define GC_USB_DAINTMSK_INEPMSK10_OFFSET         0x81c
#define GC_USB_DAINTMSK_INEPMSK11_LSB            0xb
#define GC_USB_DAINTMSK_INEPMSK11_MASK           0x800
#define GC_USB_DAINTMSK_INEPMSK11_SIZE           0x1
#define GC_USB_DAINTMSK_INEPMSK11_DEFAULT        0x0
#define GC_USB_DAINTMSK_INEPMSK11_OFFSET         0x81c
#define GC_USB_DAINTMSK_INEPMSK12_LSB            0xc
#define GC_USB_DAINTMSK_INEPMSK12_MASK           0x1000
#define GC_USB_DAINTMSK_INEPMSK12_SIZE           0x1
#define GC_USB_DAINTMSK_INEPMSK12_DEFAULT        0x0
#define GC_USB_DAINTMSK_INEPMSK12_OFFSET         0x81c
#define GC_USB_DAINTMSK_INEPMSK13_LSB            0xd
#define GC_USB_DAINTMSK_INEPMSK13_MASK           0x2000
#define GC_USB_DAINTMSK_INEPMSK13_SIZE           0x1
#define GC_USB_DAINTMSK_INEPMSK13_DEFAULT        0x0
#define GC_USB_DAINTMSK_INEPMSK13_OFFSET         0x81c
#define GC_USB_DAINTMSK_INEPMSK14_LSB            0xe
#define GC_USB_DAINTMSK_INEPMSK14_MASK           0x4000
#define GC_USB_DAINTMSK_INEPMSK14_SIZE           0x1
#define GC_USB_DAINTMSK_INEPMSK14_DEFAULT        0x0
#define GC_USB_DAINTMSK_INEPMSK14_OFFSET         0x81c
#define GC_USB_DAINTMSK_INEPMSK15_LSB            0xf
#define GC_USB_DAINTMSK_INEPMSK15_MASK           0x8000
#define GC_USB_DAINTMSK_INEPMSK15_SIZE           0x1
#define GC_USB_DAINTMSK_INEPMSK15_DEFAULT        0x0
#define GC_USB_DAINTMSK_INEPMSK15_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK0_LSB            0x10
#define GC_USB_DAINTMSK_OUTEPMSK0_MASK           0x10000
#define GC_USB_DAINTMSK_OUTEPMSK0_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK0_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK0_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK1_LSB            0x11
#define GC_USB_DAINTMSK_OUTEPMSK1_MASK           0x20000
#define GC_USB_DAINTMSK_OUTEPMSK1_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK1_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK1_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK2_LSB            0x12
#define GC_USB_DAINTMSK_OUTEPMSK2_MASK           0x40000
#define GC_USB_DAINTMSK_OUTEPMSK2_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK2_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK2_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK3_LSB            0x13
#define GC_USB_DAINTMSK_OUTEPMSK3_MASK           0x80000
#define GC_USB_DAINTMSK_OUTEPMSK3_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK3_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK3_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK4_LSB            0x14
#define GC_USB_DAINTMSK_OUTEPMSK4_MASK           0x100000
#define GC_USB_DAINTMSK_OUTEPMSK4_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK4_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK4_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK5_LSB            0x15
#define GC_USB_DAINTMSK_OUTEPMSK5_MASK           0x200000
#define GC_USB_DAINTMSK_OUTEPMSK5_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK5_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK5_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK6_LSB            0x16
#define GC_USB_DAINTMSK_OUTEPMSK6_MASK           0x400000
#define GC_USB_DAINTMSK_OUTEPMSK6_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK6_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK6_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK7_LSB            0x17
#define GC_USB_DAINTMSK_OUTEPMSK7_MASK           0x800000
#define GC_USB_DAINTMSK_OUTEPMSK7_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK7_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK7_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK8_LSB            0x18
#define GC_USB_DAINTMSK_OUTEPMSK8_MASK           0x1000000
#define GC_USB_DAINTMSK_OUTEPMSK8_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK8_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK8_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK9_LSB            0x19
#define GC_USB_DAINTMSK_OUTEPMSK9_MASK           0x2000000
#define GC_USB_DAINTMSK_OUTEPMSK9_SIZE           0x1
#define GC_USB_DAINTMSK_OUTEPMSK9_DEFAULT        0x0
#define GC_USB_DAINTMSK_OUTEPMSK9_OFFSET         0x81c
#define GC_USB_DAINTMSK_OUTEPMSK10_LSB           0x1a
#define GC_USB_DAINTMSK_OUTEPMSK10_MASK          0x4000000
#define GC_USB_DAINTMSK_OUTEPMSK10_SIZE          0x1
#define GC_USB_DAINTMSK_OUTEPMSK10_DEFAULT       0x0
#define GC_USB_DAINTMSK_OUTEPMSK10_OFFSET        0x81c
#define GC_USB_DAINTMSK_OUTEPMSK11_LSB           0x1b
#define GC_USB_DAINTMSK_OUTEPMSK11_MASK          0x8000000
#define GC_USB_DAINTMSK_OUTEPMSK11_SIZE          0x1
#define GC_USB_DAINTMSK_OUTEPMSK11_DEFAULT       0x0
#define GC_USB_DAINTMSK_OUTEPMSK11_OFFSET        0x81c
#define GC_USB_DAINTMSK_OUTEPMSK12_LSB           0x1c
#define GC_USB_DAINTMSK_OUTEPMSK12_MASK          0x10000000
#define GC_USB_DAINTMSK_OUTEPMSK12_SIZE          0x1
#define GC_USB_DAINTMSK_OUTEPMSK12_DEFAULT       0x0
#define GC_USB_DAINTMSK_OUTEPMSK12_OFFSET        0x81c
#define GC_USB_DAINTMSK_OUTEPMSK13_LSB           0x1d
#define GC_USB_DAINTMSK_OUTEPMSK13_MASK          0x20000000
#define GC_USB_DAINTMSK_OUTEPMSK13_SIZE          0x1
#define GC_USB_DAINTMSK_OUTEPMSK13_DEFAULT       0x0
#define GC_USB_DAINTMSK_OUTEPMSK13_OFFSET        0x81c
#define GC_USB_DAINTMSK_OUTEPMSK14_LSB           0x1e
#define GC_USB_DAINTMSK_OUTEPMSK14_MASK          0x40000000
#define GC_USB_DAINTMSK_OUTEPMSK14_SIZE          0x1
#define GC_USB_DAINTMSK_OUTEPMSK14_DEFAULT       0x0
#define GC_USB_DAINTMSK_OUTEPMSK14_OFFSET        0x81c
#define GC_USB_DAINTMSK_OUTEPMSK15_LSB           0x1f
#define GC_USB_DAINTMSK_OUTEPMSK15_MASK          0x80000000
#define GC_USB_DAINTMSK_OUTEPMSK15_SIZE          0x1
#define GC_USB_DAINTMSK_OUTEPMSK15_DEFAULT       0x0
#define GC_USB_DAINTMSK_OUTEPMSK15_OFFSET        0x81c
#define GC_USB_DVBUSDIS_DVBUSDIS_LSB             0x0
#define GC_USB_DVBUSDIS_DVBUSDIS_MASK            0xffff
#define GC_USB_DVBUSDIS_DVBUSDIS_SIZE            0x10
#define GC_USB_DVBUSDIS_DVBUSDIS_DEFAULT         0x0
#define GC_USB_DVBUSDIS_DVBUSDIS_OFFSET          0x828
#define GC_USB_DVBUSPULSE_DVBUSPULSE_LSB         0x0
#define GC_USB_DVBUSPULSE_DVBUSPULSE_MASK        0xfff
#define GC_USB_DVBUSPULSE_DVBUSPULSE_SIZE        0xc
#define GC_USB_DVBUSPULSE_DVBUSPULSE_DEFAULT     0x0
#define GC_USB_DVBUSPULSE_DVBUSPULSE_OFFSET      0x82c
#define GC_USB_DTHRCTL_NONISOTHREN_LSB           0x0
#define GC_USB_DTHRCTL_NONISOTHREN_MASK          0x1
#define GC_USB_DTHRCTL_NONISOTHREN_SIZE          0x1
#define GC_USB_DTHRCTL_NONISOTHREN_DEFAULT       0x0
#define GC_USB_DTHRCTL_NONISOTHREN_OFFSET        0x830
#define GC_USB_DTHRCTL_ISOTHREN_LSB              0x1
#define GC_USB_DTHRCTL_ISOTHREN_MASK             0x2
#define GC_USB_DTHRCTL_ISOTHREN_SIZE             0x1
#define GC_USB_DTHRCTL_ISOTHREN_DEFAULT          0x0
#define GC_USB_DTHRCTL_ISOTHREN_OFFSET           0x830
#define GC_USB_DTHRCTL_TXTHRLEN_LSB              0x2
#define GC_USB_DTHRCTL_TXTHRLEN_MASK             0x7fc
#define GC_USB_DTHRCTL_TXTHRLEN_SIZE             0x9
#define GC_USB_DTHRCTL_TXTHRLEN_DEFAULT          0x0
#define GC_USB_DTHRCTL_TXTHRLEN_OFFSET           0x830
#define GC_USB_DTHRCTL_AHBTHRRATIO_LSB           0xb
#define GC_USB_DTHRCTL_AHBTHRRATIO_MASK          0x1800
#define GC_USB_DTHRCTL_AHBTHRRATIO_SIZE          0x2
#define GC_USB_DTHRCTL_AHBTHRRATIO_DEFAULT       0x0
#define GC_USB_DTHRCTL_AHBTHRRATIO_OFFSET        0x830
#define GC_USB_DTHRCTL_RXTHREN_LSB               0x10
#define GC_USB_DTHRCTL_RXTHREN_MASK              0x10000
#define GC_USB_DTHRCTL_RXTHREN_SIZE              0x1
#define GC_USB_DTHRCTL_RXTHREN_DEFAULT           0x0
#define GC_USB_DTHRCTL_RXTHREN_OFFSET            0x830
#define GC_USB_DTHRCTL_RXTHRLEN_LSB              0x11
#define GC_USB_DTHRCTL_RXTHRLEN_MASK             0x3fe0000
#define GC_USB_DTHRCTL_RXTHRLEN_SIZE             0x9
#define GC_USB_DTHRCTL_RXTHRLEN_DEFAULT          0x0
#define GC_USB_DTHRCTL_RXTHRLEN_OFFSET           0x830
#define GC_USB_DTHRCTL_ARBPRKEN_LSB              0x1b
#define GC_USB_DTHRCTL_ARBPRKEN_MASK             0x8000000
#define GC_USB_DTHRCTL_ARBPRKEN_SIZE             0x1
#define GC_USB_DTHRCTL_ARBPRKEN_DEFAULT          0x0
#define GC_USB_DTHRCTL_ARBPRKEN_OFFSET           0x830
#define GC_USB_DIEPEMPMSK_INEPTXFEMPMSK_LSB      0x0
#define GC_USB_DIEPEMPMSK_INEPTXFEMPMSK_MASK     0xffff
#define GC_USB_DIEPEMPMSK_INEPTXFEMPMSK_SIZE     0x10
#define GC_USB_DIEPEMPMSK_INEPTXFEMPMSK_DEFAULT  0x0
#define GC_USB_DIEPEMPMSK_INEPTXFEMPMSK_OFFSET   0x834
#define GC_USB_DIEPCTL0_MPS_LSB                  0x0
#define GC_USB_DIEPCTL0_MPS_MASK                 0x3
#define GC_USB_DIEPCTL0_MPS_SIZE                 0x2
#define GC_USB_DIEPCTL0_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL0_MPS_OFFSET               0x900
#define GC_USB_DIEPCTL0_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL0_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL0_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL0_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL0_USBACTEP_OFFSET          0x900
#define GC_USB_DIEPCTL0_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL0_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL0_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL0_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL0_NAKSTS_OFFSET            0x900
#define GC_USB_DIEPCTL0_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL0_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL0_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL0_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL0_EPTYPE_OFFSET            0x900
#define GC_USB_DIEPCTL0_STALL_LSB                0x15
#define GC_USB_DIEPCTL0_STALL_MASK               0x200000
#define GC_USB_DIEPCTL0_STALL_SIZE               0x1
#define GC_USB_DIEPCTL0_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL0_STALL_OFFSET             0x900
#define GC_USB_DIEPCTL0_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL0_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL0_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL0_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL0_TXFNUM_OFFSET            0x900
#define GC_USB_DIEPCTL0_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL0_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL0_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL0_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL0_CNAK_OFFSET              0x900
#define GC_USB_DIEPCTL0_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL0_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL0_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL0_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL0_SNAK_OFFSET              0x900
#define GC_USB_DIEPCTL0_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL0_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL0_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL0_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL0_EPDIS_OFFSET             0x900
#define GC_USB_DIEPCTL0_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL0_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL0_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL0_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL0_EPENA_OFFSET             0x900
#define GC_USB_DIEPINT0_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT0_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT0_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT0_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT0_XFERCOMPL_OFFSET         0x908
#define GC_USB_DIEPINT0_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT0_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT0_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT0_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT0_EPDISBLD_OFFSET          0x908
#define GC_USB_DIEPINT0_AHBERR_LSB               0x2
#define GC_USB_DIEPINT0_AHBERR_MASK              0x4
#define GC_USB_DIEPINT0_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT0_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT0_AHBERR_OFFSET            0x908
#define GC_USB_DIEPINT0_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT0_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT0_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT0_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT0_TIMEOUT_OFFSET           0x908
#define GC_USB_DIEPINT0_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT0_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT0_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT0_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT0_INTKNTXFEMP_OFFSET       0x908
#define GC_USB_DIEPINT0_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT0_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT0_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT0_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT0_INTKNEPMIS_OFFSET        0x908
#define GC_USB_DIEPINT0_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT0_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT0_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT0_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT0_INEPNAKEFF_OFFSET        0x908
#define GC_USB_DIEPINT0_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT0_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT0_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT0_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT0_TXFEMP_OFFSET            0x908
#define GC_USB_DIEPINT0_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT0_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT0_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT0_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT0_TXFIFOUNDRN_OFFSET       0x908
#define GC_USB_DIEPINT0_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT0_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT0_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT0_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT0_BNAINTR_OFFSET           0x908
#define GC_USB_DIEPINT0_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT0_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT0_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT0_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT0_PKTDRPSTS_OFFSET         0x908
#define GC_USB_DIEPINT0_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT0_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT0_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT0_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT0_BBLEERR_OFFSET           0x908
#define GC_USB_DIEPINT0_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT0_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT0_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT0_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT0_NAKINTRPT_OFFSET         0x908
#define GC_USB_DIEPINT0_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT0_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT0_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT0_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT0_NYETINTRPT_OFFSET        0x908
#define GC_USB_DIEPTSIZ0_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ0_XFERSIZE_MASK           0x7f
#define GC_USB_DIEPTSIZ0_XFERSIZE_SIZE           0x7
#define GC_USB_DIEPTSIZ0_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ0_XFERSIZE_OFFSET         0x910
#define GC_USB_DIEPTSIZ0_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ0_PKTCNT_MASK             0x180000
#define GC_USB_DIEPTSIZ0_PKTCNT_SIZE             0x2
#define GC_USB_DIEPTSIZ0_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ0_PKTCNT_OFFSET           0x910
#define GC_USB_DIEPDMA0_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA0_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA0_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA0_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA0_DMAADDR_OFFSET           0x914
#define GC_USB_DTXFSTS0_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS0_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS0_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS0_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS0_INEPTXFSPCAVAIL_OFFSET   0x918
#define GC_USB_DIEPDMAB0_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB0_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB0_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB0_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB0_DMABUFFERADDR_OFFSET    0x91c
#define GC_USB_DIEPCTL1_MPS_LSB                  0x0
#define GC_USB_DIEPCTL1_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL1_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL1_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL1_MPS_OFFSET               0x920
#define GC_USB_DIEPCTL1_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL1_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL1_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL1_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL1_USBACTEP_OFFSET          0x920
#define GC_USB_DIEPCTL1_DPID_LSB                 0x10
#define GC_USB_DIEPCTL1_DPID_MASK                0x10000
#define GC_USB_DIEPCTL1_DPID_SIZE                0x1
#define GC_USB_DIEPCTL1_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL1_DPID_OFFSET              0x920
#define GC_USB_DIEPCTL1_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL1_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL1_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL1_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL1_NAKSTS_OFFSET            0x920
#define GC_USB_DIEPCTL1_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL1_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL1_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL1_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL1_EPTYPE_OFFSET            0x920
#define GC_USB_DIEPCTL1_STALL_LSB                0x15
#define GC_USB_DIEPCTL1_STALL_MASK               0x200000
#define GC_USB_DIEPCTL1_STALL_SIZE               0x1
#define GC_USB_DIEPCTL1_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL1_STALL_OFFSET             0x920
#define GC_USB_DIEPCTL1_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL1_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL1_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL1_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL1_TXFNUM_OFFSET            0x920
#define GC_USB_DIEPCTL1_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL1_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL1_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL1_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL1_CNAK_OFFSET              0x920
#define GC_USB_DIEPCTL1_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL1_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL1_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL1_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL1_SNAK_OFFSET              0x920
#define GC_USB_DIEPCTL1_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL1_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL1_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL1_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL1_SETD0PID_OFFSET          0x920
#define GC_USB_DIEPCTL1_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL1_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL1_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL1_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL1_SETD1PID_OFFSET          0x920
#define GC_USB_DIEPCTL1_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL1_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL1_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL1_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL1_EPDIS_OFFSET             0x920
#define GC_USB_DIEPCTL1_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL1_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL1_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL1_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL1_EPENA_OFFSET             0x920
#define GC_USB_DIEPINT1_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT1_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT1_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT1_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT1_XFERCOMPL_OFFSET         0x928
#define GC_USB_DIEPINT1_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT1_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT1_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT1_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT1_EPDISBLD_OFFSET          0x928
#define GC_USB_DIEPINT1_AHBERR_LSB               0x2
#define GC_USB_DIEPINT1_AHBERR_MASK              0x4
#define GC_USB_DIEPINT1_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT1_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT1_AHBERR_OFFSET            0x928
#define GC_USB_DIEPINT1_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT1_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT1_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT1_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT1_TIMEOUT_OFFSET           0x928
#define GC_USB_DIEPINT1_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT1_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT1_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT1_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT1_INTKNTXFEMP_OFFSET       0x928
#define GC_USB_DIEPINT1_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT1_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT1_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT1_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT1_INTKNEPMIS_OFFSET        0x928
#define GC_USB_DIEPINT1_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT1_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT1_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT1_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT1_INEPNAKEFF_OFFSET        0x928
#define GC_USB_DIEPINT1_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT1_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT1_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT1_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT1_TXFEMP_OFFSET            0x928
#define GC_USB_DIEPINT1_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT1_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT1_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT1_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT1_TXFIFOUNDRN_OFFSET       0x928
#define GC_USB_DIEPINT1_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT1_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT1_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT1_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT1_BNAINTR_OFFSET           0x928
#define GC_USB_DIEPINT1_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT1_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT1_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT1_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT1_PKTDRPSTS_OFFSET         0x928
#define GC_USB_DIEPINT1_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT1_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT1_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT1_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT1_BBLEERR_OFFSET           0x928
#define GC_USB_DIEPINT1_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT1_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT1_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT1_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT1_NAKINTRPT_OFFSET         0x928
#define GC_USB_DIEPINT1_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT1_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT1_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT1_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT1_NYETINTRPT_OFFSET        0x928
#define GC_USB_DIEPTSIZ1_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ1_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ1_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ1_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ1_XFERSIZE_OFFSET         0x930
#define GC_USB_DIEPTSIZ1_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ1_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ1_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ1_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ1_PKTCNT_OFFSET           0x930
#define GC_USB_DIEPTSIZ1_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ1_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ1_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ1_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ1_MC_OFFSET               0x930
#define GC_USB_DIEPDMA1_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA1_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA1_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA1_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA1_DMAADDR_OFFSET           0x934
#define GC_USB_DTXFSTS1_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS1_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS1_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS1_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS1_INEPTXFSPCAVAIL_OFFSET   0x938
#define GC_USB_DIEPDMAB1_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB1_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB1_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB1_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB1_DMABUFFERADDR_OFFSET    0x93c
#define GC_USB_DIEPCTL2_MPS_LSB                  0x0
#define GC_USB_DIEPCTL2_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL2_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL2_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL2_MPS_OFFSET               0x940
#define GC_USB_DIEPCTL2_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL2_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL2_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL2_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL2_USBACTEP_OFFSET          0x940
#define GC_USB_DIEPCTL2_DPID_LSB                 0x10
#define GC_USB_DIEPCTL2_DPID_MASK                0x10000
#define GC_USB_DIEPCTL2_DPID_SIZE                0x1
#define GC_USB_DIEPCTL2_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL2_DPID_OFFSET              0x940
#define GC_USB_DIEPCTL2_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL2_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL2_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL2_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL2_NAKSTS_OFFSET            0x940
#define GC_USB_DIEPCTL2_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL2_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL2_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL2_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL2_EPTYPE_OFFSET            0x940
#define GC_USB_DIEPCTL2_STALL_LSB                0x15
#define GC_USB_DIEPCTL2_STALL_MASK               0x200000
#define GC_USB_DIEPCTL2_STALL_SIZE               0x1
#define GC_USB_DIEPCTL2_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL2_STALL_OFFSET             0x940
#define GC_USB_DIEPCTL2_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL2_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL2_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL2_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL2_TXFNUM_OFFSET            0x940
#define GC_USB_DIEPCTL2_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL2_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL2_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL2_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL2_CNAK_OFFSET              0x940
#define GC_USB_DIEPCTL2_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL2_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL2_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL2_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL2_SNAK_OFFSET              0x940
#define GC_USB_DIEPCTL2_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL2_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL2_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL2_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL2_SETD0PID_OFFSET          0x940
#define GC_USB_DIEPCTL2_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL2_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL2_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL2_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL2_SETD1PID_OFFSET          0x940
#define GC_USB_DIEPCTL2_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL2_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL2_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL2_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL2_EPDIS_OFFSET             0x940
#define GC_USB_DIEPCTL2_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL2_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL2_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL2_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL2_EPENA_OFFSET             0x940
#define GC_USB_DIEPINT2_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT2_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT2_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT2_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT2_XFERCOMPL_OFFSET         0x948
#define GC_USB_DIEPINT2_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT2_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT2_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT2_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT2_EPDISBLD_OFFSET          0x948
#define GC_USB_DIEPINT2_AHBERR_LSB               0x2
#define GC_USB_DIEPINT2_AHBERR_MASK              0x4
#define GC_USB_DIEPINT2_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT2_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT2_AHBERR_OFFSET            0x948
#define GC_USB_DIEPINT2_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT2_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT2_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT2_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT2_TIMEOUT_OFFSET           0x948
#define GC_USB_DIEPINT2_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT2_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT2_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT2_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT2_INTKNTXFEMP_OFFSET       0x948
#define GC_USB_DIEPINT2_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT2_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT2_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT2_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT2_INTKNEPMIS_OFFSET        0x948
#define GC_USB_DIEPINT2_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT2_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT2_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT2_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT2_INEPNAKEFF_OFFSET        0x948
#define GC_USB_DIEPINT2_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT2_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT2_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT2_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT2_TXFEMP_OFFSET            0x948
#define GC_USB_DIEPINT2_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT2_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT2_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT2_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT2_TXFIFOUNDRN_OFFSET       0x948
#define GC_USB_DIEPINT2_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT2_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT2_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT2_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT2_BNAINTR_OFFSET           0x948
#define GC_USB_DIEPINT2_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT2_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT2_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT2_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT2_PKTDRPSTS_OFFSET         0x948
#define GC_USB_DIEPINT2_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT2_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT2_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT2_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT2_BBLEERR_OFFSET           0x948
#define GC_USB_DIEPINT2_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT2_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT2_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT2_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT2_NAKINTRPT_OFFSET         0x948
#define GC_USB_DIEPINT2_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT2_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT2_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT2_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT2_NYETINTRPT_OFFSET        0x948
#define GC_USB_DIEPTSIZ2_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ2_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ2_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ2_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ2_XFERSIZE_OFFSET         0x950
#define GC_USB_DIEPTSIZ2_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ2_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ2_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ2_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ2_PKTCNT_OFFSET           0x950
#define GC_USB_DIEPTSIZ2_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ2_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ2_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ2_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ2_MC_OFFSET               0x950
#define GC_USB_DIEPDMA2_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA2_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA2_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA2_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA2_DMAADDR_OFFSET           0x954
#define GC_USB_DTXFSTS2_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS2_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS2_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS2_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS2_INEPTXFSPCAVAIL_OFFSET   0x958
#define GC_USB_DIEPDMAB2_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB2_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB2_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB2_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB2_DMABUFFERADDR_OFFSET    0x95c
#define GC_USB_DIEPCTL3_MPS_LSB                  0x0
#define GC_USB_DIEPCTL3_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL3_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL3_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL3_MPS_OFFSET               0x960
#define GC_USB_DIEPCTL3_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL3_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL3_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL3_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL3_USBACTEP_OFFSET          0x960
#define GC_USB_DIEPCTL3_DPID_LSB                 0x10
#define GC_USB_DIEPCTL3_DPID_MASK                0x10000
#define GC_USB_DIEPCTL3_DPID_SIZE                0x1
#define GC_USB_DIEPCTL3_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL3_DPID_OFFSET              0x960
#define GC_USB_DIEPCTL3_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL3_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL3_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL3_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL3_NAKSTS_OFFSET            0x960
#define GC_USB_DIEPCTL3_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL3_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL3_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL3_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL3_EPTYPE_OFFSET            0x960
#define GC_USB_DIEPCTL3_STALL_LSB                0x15
#define GC_USB_DIEPCTL3_STALL_MASK               0x200000
#define GC_USB_DIEPCTL3_STALL_SIZE               0x1
#define GC_USB_DIEPCTL3_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL3_STALL_OFFSET             0x960
#define GC_USB_DIEPCTL3_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL3_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL3_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL3_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL3_TXFNUM_OFFSET            0x960
#define GC_USB_DIEPCTL3_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL3_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL3_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL3_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL3_CNAK_OFFSET              0x960
#define GC_USB_DIEPCTL3_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL3_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL3_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL3_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL3_SNAK_OFFSET              0x960
#define GC_USB_DIEPCTL3_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL3_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL3_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL3_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL3_SETD0PID_OFFSET          0x960
#define GC_USB_DIEPCTL3_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL3_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL3_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL3_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL3_SETD1PID_OFFSET          0x960
#define GC_USB_DIEPCTL3_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL3_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL3_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL3_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL3_EPDIS_OFFSET             0x960
#define GC_USB_DIEPCTL3_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL3_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL3_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL3_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL3_EPENA_OFFSET             0x960
#define GC_USB_DIEPINT3_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT3_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT3_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT3_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT3_XFERCOMPL_OFFSET         0x968
#define GC_USB_DIEPINT3_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT3_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT3_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT3_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT3_EPDISBLD_OFFSET          0x968
#define GC_USB_DIEPINT3_AHBERR_LSB               0x2
#define GC_USB_DIEPINT3_AHBERR_MASK              0x4
#define GC_USB_DIEPINT3_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT3_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT3_AHBERR_OFFSET            0x968
#define GC_USB_DIEPINT3_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT3_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT3_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT3_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT3_TIMEOUT_OFFSET           0x968
#define GC_USB_DIEPINT3_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT3_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT3_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT3_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT3_INTKNTXFEMP_OFFSET       0x968
#define GC_USB_DIEPINT3_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT3_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT3_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT3_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT3_INTKNEPMIS_OFFSET        0x968
#define GC_USB_DIEPINT3_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT3_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT3_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT3_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT3_INEPNAKEFF_OFFSET        0x968
#define GC_USB_DIEPINT3_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT3_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT3_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT3_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT3_TXFEMP_OFFSET            0x968
#define GC_USB_DIEPINT3_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT3_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT3_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT3_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT3_TXFIFOUNDRN_OFFSET       0x968
#define GC_USB_DIEPINT3_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT3_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT3_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT3_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT3_BNAINTR_OFFSET           0x968
#define GC_USB_DIEPINT3_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT3_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT3_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT3_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT3_PKTDRPSTS_OFFSET         0x968
#define GC_USB_DIEPINT3_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT3_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT3_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT3_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT3_BBLEERR_OFFSET           0x968
#define GC_USB_DIEPINT3_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT3_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT3_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT3_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT3_NAKINTRPT_OFFSET         0x968
#define GC_USB_DIEPINT3_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT3_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT3_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT3_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT3_NYETINTRPT_OFFSET        0x968
#define GC_USB_DIEPTSIZ3_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ3_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ3_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ3_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ3_XFERSIZE_OFFSET         0x970
#define GC_USB_DIEPTSIZ3_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ3_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ3_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ3_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ3_PKTCNT_OFFSET           0x970
#define GC_USB_DIEPTSIZ3_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ3_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ3_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ3_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ3_MC_OFFSET               0x970
#define GC_USB_DIEPDMA3_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA3_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA3_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA3_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA3_DMAADDR_OFFSET           0x974
#define GC_USB_DTXFSTS3_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS3_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS3_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS3_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS3_INEPTXFSPCAVAIL_OFFSET   0x978
#define GC_USB_DIEPDMAB3_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB3_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB3_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB3_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB3_DMABUFFERADDR_OFFSET    0x97c
#define GC_USB_DIEPCTL4_MPS_LSB                  0x0
#define GC_USB_DIEPCTL4_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL4_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL4_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL4_MPS_OFFSET               0x980
#define GC_USB_DIEPCTL4_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL4_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL4_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL4_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL4_USBACTEP_OFFSET          0x980
#define GC_USB_DIEPCTL4_DPID_LSB                 0x10
#define GC_USB_DIEPCTL4_DPID_MASK                0x10000
#define GC_USB_DIEPCTL4_DPID_SIZE                0x1
#define GC_USB_DIEPCTL4_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL4_DPID_OFFSET              0x980
#define GC_USB_DIEPCTL4_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL4_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL4_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL4_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL4_NAKSTS_OFFSET            0x980
#define GC_USB_DIEPCTL4_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL4_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL4_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL4_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL4_EPTYPE_OFFSET            0x980
#define GC_USB_DIEPCTL4_STALL_LSB                0x15
#define GC_USB_DIEPCTL4_STALL_MASK               0x200000
#define GC_USB_DIEPCTL4_STALL_SIZE               0x1
#define GC_USB_DIEPCTL4_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL4_STALL_OFFSET             0x980
#define GC_USB_DIEPCTL4_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL4_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL4_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL4_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL4_TXFNUM_OFFSET            0x980
#define GC_USB_DIEPCTL4_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL4_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL4_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL4_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL4_CNAK_OFFSET              0x980
#define GC_USB_DIEPCTL4_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL4_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL4_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL4_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL4_SNAK_OFFSET              0x980
#define GC_USB_DIEPCTL4_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL4_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL4_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL4_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL4_SETD0PID_OFFSET          0x980
#define GC_USB_DIEPCTL4_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL4_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL4_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL4_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL4_SETD1PID_OFFSET          0x980
#define GC_USB_DIEPCTL4_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL4_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL4_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL4_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL4_EPDIS_OFFSET             0x980
#define GC_USB_DIEPCTL4_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL4_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL4_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL4_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL4_EPENA_OFFSET             0x980
#define GC_USB_DIEPINT4_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT4_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT4_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT4_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT4_XFERCOMPL_OFFSET         0x988
#define GC_USB_DIEPINT4_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT4_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT4_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT4_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT4_EPDISBLD_OFFSET          0x988
#define GC_USB_DIEPINT4_AHBERR_LSB               0x2
#define GC_USB_DIEPINT4_AHBERR_MASK              0x4
#define GC_USB_DIEPINT4_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT4_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT4_AHBERR_OFFSET            0x988
#define GC_USB_DIEPINT4_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT4_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT4_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT4_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT4_TIMEOUT_OFFSET           0x988
#define GC_USB_DIEPINT4_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT4_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT4_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT4_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT4_INTKNTXFEMP_OFFSET       0x988
#define GC_USB_DIEPINT4_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT4_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT4_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT4_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT4_INTKNEPMIS_OFFSET        0x988
#define GC_USB_DIEPINT4_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT4_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT4_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT4_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT4_INEPNAKEFF_OFFSET        0x988
#define GC_USB_DIEPINT4_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT4_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT4_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT4_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT4_TXFEMP_OFFSET            0x988
#define GC_USB_DIEPINT4_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT4_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT4_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT4_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT4_TXFIFOUNDRN_OFFSET       0x988
#define GC_USB_DIEPINT4_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT4_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT4_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT4_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT4_BNAINTR_OFFSET           0x988
#define GC_USB_DIEPINT4_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT4_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT4_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT4_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT4_PKTDRPSTS_OFFSET         0x988
#define GC_USB_DIEPINT4_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT4_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT4_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT4_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT4_BBLEERR_OFFSET           0x988
#define GC_USB_DIEPINT4_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT4_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT4_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT4_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT4_NAKINTRPT_OFFSET         0x988
#define GC_USB_DIEPINT4_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT4_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT4_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT4_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT4_NYETINTRPT_OFFSET        0x988
#define GC_USB_DIEPTSIZ4_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ4_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ4_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ4_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ4_XFERSIZE_OFFSET         0x990
#define GC_USB_DIEPTSIZ4_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ4_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ4_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ4_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ4_PKTCNT_OFFSET           0x990
#define GC_USB_DIEPTSIZ4_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ4_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ4_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ4_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ4_MC_OFFSET               0x990
#define GC_USB_DIEPDMA4_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA4_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA4_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA4_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA4_DMAADDR_OFFSET           0x994
#define GC_USB_DTXFSTS4_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS4_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS4_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS4_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS4_INEPTXFSPCAVAIL_OFFSET   0x998
#define GC_USB_DIEPDMAB4_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB4_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB4_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB4_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB4_DMABUFFERADDR_OFFSET    0x99c
#define GC_USB_DIEPCTL5_MPS_LSB                  0x0
#define GC_USB_DIEPCTL5_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL5_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL5_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL5_MPS_OFFSET               0x9a0
#define GC_USB_DIEPCTL5_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL5_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL5_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL5_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL5_USBACTEP_OFFSET          0x9a0
#define GC_USB_DIEPCTL5_DPID_LSB                 0x10
#define GC_USB_DIEPCTL5_DPID_MASK                0x10000
#define GC_USB_DIEPCTL5_DPID_SIZE                0x1
#define GC_USB_DIEPCTL5_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL5_DPID_OFFSET              0x9a0
#define GC_USB_DIEPCTL5_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL5_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL5_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL5_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL5_NAKSTS_OFFSET            0x9a0
#define GC_USB_DIEPCTL5_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL5_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL5_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL5_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL5_EPTYPE_OFFSET            0x9a0
#define GC_USB_DIEPCTL5_STALL_LSB                0x15
#define GC_USB_DIEPCTL5_STALL_MASK               0x200000
#define GC_USB_DIEPCTL5_STALL_SIZE               0x1
#define GC_USB_DIEPCTL5_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL5_STALL_OFFSET             0x9a0
#define GC_USB_DIEPCTL5_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL5_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL5_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL5_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL5_TXFNUM_OFFSET            0x9a0
#define GC_USB_DIEPCTL5_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL5_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL5_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL5_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL5_CNAK_OFFSET              0x9a0
#define GC_USB_DIEPCTL5_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL5_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL5_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL5_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL5_SNAK_OFFSET              0x9a0
#define GC_USB_DIEPCTL5_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL5_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL5_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL5_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL5_SETD0PID_OFFSET          0x9a0
#define GC_USB_DIEPCTL5_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL5_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL5_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL5_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL5_SETD1PID_OFFSET          0x9a0
#define GC_USB_DIEPCTL5_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL5_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL5_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL5_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL5_EPDIS_OFFSET             0x9a0
#define GC_USB_DIEPCTL5_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL5_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL5_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL5_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL5_EPENA_OFFSET             0x9a0
#define GC_USB_DIEPINT5_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT5_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT5_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT5_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT5_XFERCOMPL_OFFSET         0x9a8
#define GC_USB_DIEPINT5_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT5_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT5_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT5_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT5_EPDISBLD_OFFSET          0x9a8
#define GC_USB_DIEPINT5_AHBERR_LSB               0x2
#define GC_USB_DIEPINT5_AHBERR_MASK              0x4
#define GC_USB_DIEPINT5_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT5_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT5_AHBERR_OFFSET            0x9a8
#define GC_USB_DIEPINT5_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT5_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT5_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT5_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT5_TIMEOUT_OFFSET           0x9a8
#define GC_USB_DIEPINT5_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT5_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT5_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT5_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT5_INTKNTXFEMP_OFFSET       0x9a8
#define GC_USB_DIEPINT5_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT5_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT5_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT5_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT5_INTKNEPMIS_OFFSET        0x9a8
#define GC_USB_DIEPINT5_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT5_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT5_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT5_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT5_INEPNAKEFF_OFFSET        0x9a8
#define GC_USB_DIEPINT5_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT5_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT5_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT5_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT5_TXFEMP_OFFSET            0x9a8
#define GC_USB_DIEPINT5_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT5_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT5_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT5_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT5_TXFIFOUNDRN_OFFSET       0x9a8
#define GC_USB_DIEPINT5_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT5_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT5_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT5_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT5_BNAINTR_OFFSET           0x9a8
#define GC_USB_DIEPINT5_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT5_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT5_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT5_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT5_PKTDRPSTS_OFFSET         0x9a8
#define GC_USB_DIEPINT5_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT5_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT5_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT5_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT5_BBLEERR_OFFSET           0x9a8
#define GC_USB_DIEPINT5_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT5_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT5_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT5_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT5_NAKINTRPT_OFFSET         0x9a8
#define GC_USB_DIEPINT5_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT5_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT5_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT5_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT5_NYETINTRPT_OFFSET        0x9a8
#define GC_USB_DIEPTSIZ5_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ5_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ5_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ5_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ5_XFERSIZE_OFFSET         0x9b0
#define GC_USB_DIEPTSIZ5_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ5_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ5_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ5_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ5_PKTCNT_OFFSET           0x9b0
#define GC_USB_DIEPTSIZ5_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ5_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ5_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ5_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ5_MC_OFFSET               0x9b0
#define GC_USB_DIEPDMA5_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA5_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA5_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA5_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA5_DMAADDR_OFFSET           0x9b4
#define GC_USB_DTXFSTS5_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS5_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS5_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS5_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS5_INEPTXFSPCAVAIL_OFFSET   0x9b8
#define GC_USB_DIEPDMAB5_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB5_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB5_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB5_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB5_DMABUFFERADDR_OFFSET    0x9bc
#define GC_USB_DIEPCTL6_MPS_LSB                  0x0
#define GC_USB_DIEPCTL6_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL6_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL6_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL6_MPS_OFFSET               0x9c0
#define GC_USB_DIEPCTL6_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL6_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL6_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL6_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL6_USBACTEP_OFFSET          0x9c0
#define GC_USB_DIEPCTL6_DPID_LSB                 0x10
#define GC_USB_DIEPCTL6_DPID_MASK                0x10000
#define GC_USB_DIEPCTL6_DPID_SIZE                0x1
#define GC_USB_DIEPCTL6_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL6_DPID_OFFSET              0x9c0
#define GC_USB_DIEPCTL6_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL6_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL6_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL6_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL6_NAKSTS_OFFSET            0x9c0
#define GC_USB_DIEPCTL6_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL6_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL6_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL6_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL6_EPTYPE_OFFSET            0x9c0
#define GC_USB_DIEPCTL6_STALL_LSB                0x15
#define GC_USB_DIEPCTL6_STALL_MASK               0x200000
#define GC_USB_DIEPCTL6_STALL_SIZE               0x1
#define GC_USB_DIEPCTL6_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL6_STALL_OFFSET             0x9c0
#define GC_USB_DIEPCTL6_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL6_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL6_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL6_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL6_TXFNUM_OFFSET            0x9c0
#define GC_USB_DIEPCTL6_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL6_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL6_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL6_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL6_CNAK_OFFSET              0x9c0
#define GC_USB_DIEPCTL6_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL6_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL6_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL6_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL6_SNAK_OFFSET              0x9c0
#define GC_USB_DIEPCTL6_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL6_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL6_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL6_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL6_SETD0PID_OFFSET          0x9c0
#define GC_USB_DIEPCTL6_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL6_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL6_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL6_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL6_SETD1PID_OFFSET          0x9c0
#define GC_USB_DIEPCTL6_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL6_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL6_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL6_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL6_EPDIS_OFFSET             0x9c0
#define GC_USB_DIEPCTL6_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL6_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL6_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL6_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL6_EPENA_OFFSET             0x9c0
#define GC_USB_DIEPINT6_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT6_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT6_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT6_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT6_XFERCOMPL_OFFSET         0x9c8
#define GC_USB_DIEPINT6_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT6_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT6_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT6_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT6_EPDISBLD_OFFSET          0x9c8
#define GC_USB_DIEPINT6_AHBERR_LSB               0x2
#define GC_USB_DIEPINT6_AHBERR_MASK              0x4
#define GC_USB_DIEPINT6_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT6_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT6_AHBERR_OFFSET            0x9c8
#define GC_USB_DIEPINT6_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT6_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT6_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT6_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT6_TIMEOUT_OFFSET           0x9c8
#define GC_USB_DIEPINT6_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT6_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT6_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT6_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT6_INTKNTXFEMP_OFFSET       0x9c8
#define GC_USB_DIEPINT6_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT6_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT6_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT6_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT6_INTKNEPMIS_OFFSET        0x9c8
#define GC_USB_DIEPINT6_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT6_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT6_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT6_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT6_INEPNAKEFF_OFFSET        0x9c8
#define GC_USB_DIEPINT6_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT6_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT6_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT6_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT6_TXFEMP_OFFSET            0x9c8
#define GC_USB_DIEPINT6_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT6_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT6_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT6_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT6_TXFIFOUNDRN_OFFSET       0x9c8
#define GC_USB_DIEPINT6_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT6_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT6_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT6_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT6_BNAINTR_OFFSET           0x9c8
#define GC_USB_DIEPINT6_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT6_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT6_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT6_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT6_PKTDRPSTS_OFFSET         0x9c8
#define GC_USB_DIEPINT6_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT6_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT6_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT6_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT6_BBLEERR_OFFSET           0x9c8
#define GC_USB_DIEPINT6_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT6_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT6_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT6_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT6_NAKINTRPT_OFFSET         0x9c8
#define GC_USB_DIEPINT6_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT6_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT6_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT6_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT6_NYETINTRPT_OFFSET        0x9c8
#define GC_USB_DIEPTSIZ6_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ6_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ6_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ6_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ6_XFERSIZE_OFFSET         0x9d0
#define GC_USB_DIEPTSIZ6_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ6_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ6_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ6_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ6_PKTCNT_OFFSET           0x9d0
#define GC_USB_DIEPTSIZ6_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ6_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ6_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ6_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ6_MC_OFFSET               0x9d0
#define GC_USB_DIEPDMA6_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA6_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA6_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA6_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA6_DMAADDR_OFFSET           0x9d4
#define GC_USB_DTXFSTS6_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS6_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS6_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS6_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS6_INEPTXFSPCAVAIL_OFFSET   0x9d8
#define GC_USB_DIEPDMAB6_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB6_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB6_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB6_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB6_DMABUFFERADDR_OFFSET    0x9dc
#define GC_USB_DIEPCTL7_MPS_LSB                  0x0
#define GC_USB_DIEPCTL7_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL7_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL7_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL7_MPS_OFFSET               0x9e0
#define GC_USB_DIEPCTL7_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL7_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL7_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL7_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL7_USBACTEP_OFFSET          0x9e0
#define GC_USB_DIEPCTL7_DPID_LSB                 0x10
#define GC_USB_DIEPCTL7_DPID_MASK                0x10000
#define GC_USB_DIEPCTL7_DPID_SIZE                0x1
#define GC_USB_DIEPCTL7_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL7_DPID_OFFSET              0x9e0
#define GC_USB_DIEPCTL7_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL7_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL7_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL7_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL7_NAKSTS_OFFSET            0x9e0
#define GC_USB_DIEPCTL7_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL7_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL7_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL7_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL7_EPTYPE_OFFSET            0x9e0
#define GC_USB_DIEPCTL7_STALL_LSB                0x15
#define GC_USB_DIEPCTL7_STALL_MASK               0x200000
#define GC_USB_DIEPCTL7_STALL_SIZE               0x1
#define GC_USB_DIEPCTL7_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL7_STALL_OFFSET             0x9e0
#define GC_USB_DIEPCTL7_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL7_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL7_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL7_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL7_TXFNUM_OFFSET            0x9e0
#define GC_USB_DIEPCTL7_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL7_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL7_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL7_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL7_CNAK_OFFSET              0x9e0
#define GC_USB_DIEPCTL7_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL7_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL7_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL7_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL7_SNAK_OFFSET              0x9e0
#define GC_USB_DIEPCTL7_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL7_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL7_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL7_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL7_SETD0PID_OFFSET          0x9e0
#define GC_USB_DIEPCTL7_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL7_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL7_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL7_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL7_SETD1PID_OFFSET          0x9e0
#define GC_USB_DIEPCTL7_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL7_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL7_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL7_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL7_EPDIS_OFFSET             0x9e0
#define GC_USB_DIEPCTL7_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL7_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL7_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL7_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL7_EPENA_OFFSET             0x9e0
#define GC_USB_DIEPINT7_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT7_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT7_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT7_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT7_XFERCOMPL_OFFSET         0x9e8
#define GC_USB_DIEPINT7_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT7_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT7_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT7_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT7_EPDISBLD_OFFSET          0x9e8
#define GC_USB_DIEPINT7_AHBERR_LSB               0x2
#define GC_USB_DIEPINT7_AHBERR_MASK              0x4
#define GC_USB_DIEPINT7_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT7_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT7_AHBERR_OFFSET            0x9e8
#define GC_USB_DIEPINT7_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT7_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT7_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT7_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT7_TIMEOUT_OFFSET           0x9e8
#define GC_USB_DIEPINT7_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT7_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT7_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT7_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT7_INTKNTXFEMP_OFFSET       0x9e8
#define GC_USB_DIEPINT7_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT7_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT7_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT7_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT7_INTKNEPMIS_OFFSET        0x9e8
#define GC_USB_DIEPINT7_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT7_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT7_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT7_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT7_INEPNAKEFF_OFFSET        0x9e8
#define GC_USB_DIEPINT7_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT7_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT7_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT7_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT7_TXFEMP_OFFSET            0x9e8
#define GC_USB_DIEPINT7_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT7_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT7_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT7_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT7_TXFIFOUNDRN_OFFSET       0x9e8
#define GC_USB_DIEPINT7_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT7_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT7_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT7_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT7_BNAINTR_OFFSET           0x9e8
#define GC_USB_DIEPINT7_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT7_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT7_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT7_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT7_PKTDRPSTS_OFFSET         0x9e8
#define GC_USB_DIEPINT7_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT7_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT7_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT7_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT7_BBLEERR_OFFSET           0x9e8
#define GC_USB_DIEPINT7_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT7_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT7_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT7_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT7_NAKINTRPT_OFFSET         0x9e8
#define GC_USB_DIEPINT7_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT7_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT7_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT7_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT7_NYETINTRPT_OFFSET        0x9e8
#define GC_USB_DIEPTSIZ7_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ7_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ7_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ7_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ7_XFERSIZE_OFFSET         0x9f0
#define GC_USB_DIEPTSIZ7_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ7_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ7_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ7_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ7_PKTCNT_OFFSET           0x9f0
#define GC_USB_DIEPTSIZ7_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ7_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ7_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ7_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ7_MC_OFFSET               0x9f0
#define GC_USB_DIEPDMA7_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA7_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA7_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA7_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA7_DMAADDR_OFFSET           0x9f4
#define GC_USB_DTXFSTS7_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS7_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS7_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS7_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS7_INEPTXFSPCAVAIL_OFFSET   0x9f8
#define GC_USB_DIEPDMAB7_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB7_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB7_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB7_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB7_DMABUFFERADDR_OFFSET    0x9fc
#define GC_USB_DIEPCTL8_MPS_LSB                  0x0
#define GC_USB_DIEPCTL8_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL8_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL8_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL8_MPS_OFFSET               0xa00
#define GC_USB_DIEPCTL8_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL8_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL8_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL8_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL8_USBACTEP_OFFSET          0xa00
#define GC_USB_DIEPCTL8_DPID_LSB                 0x10
#define GC_USB_DIEPCTL8_DPID_MASK                0x10000
#define GC_USB_DIEPCTL8_DPID_SIZE                0x1
#define GC_USB_DIEPCTL8_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL8_DPID_OFFSET              0xa00
#define GC_USB_DIEPCTL8_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL8_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL8_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL8_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL8_NAKSTS_OFFSET            0xa00
#define GC_USB_DIEPCTL8_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL8_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL8_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL8_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL8_EPTYPE_OFFSET            0xa00
#define GC_USB_DIEPCTL8_STALL_LSB                0x15
#define GC_USB_DIEPCTL8_STALL_MASK               0x200000
#define GC_USB_DIEPCTL8_STALL_SIZE               0x1
#define GC_USB_DIEPCTL8_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL8_STALL_OFFSET             0xa00
#define GC_USB_DIEPCTL8_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL8_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL8_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL8_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL8_TXFNUM_OFFSET            0xa00
#define GC_USB_DIEPCTL8_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL8_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL8_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL8_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL8_CNAK_OFFSET              0xa00
#define GC_USB_DIEPCTL8_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL8_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL8_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL8_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL8_SNAK_OFFSET              0xa00
#define GC_USB_DIEPCTL8_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL8_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL8_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL8_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL8_SETD0PID_OFFSET          0xa00
#define GC_USB_DIEPCTL8_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL8_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL8_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL8_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL8_SETD1PID_OFFSET          0xa00
#define GC_USB_DIEPCTL8_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL8_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL8_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL8_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL8_EPDIS_OFFSET             0xa00
#define GC_USB_DIEPCTL8_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL8_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL8_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL8_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL8_EPENA_OFFSET             0xa00
#define GC_USB_DIEPINT8_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT8_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT8_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT8_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT8_XFERCOMPL_OFFSET         0xa08
#define GC_USB_DIEPINT8_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT8_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT8_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT8_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT8_EPDISBLD_OFFSET          0xa08
#define GC_USB_DIEPINT8_AHBERR_LSB               0x2
#define GC_USB_DIEPINT8_AHBERR_MASK              0x4
#define GC_USB_DIEPINT8_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT8_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT8_AHBERR_OFFSET            0xa08
#define GC_USB_DIEPINT8_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT8_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT8_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT8_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT8_TIMEOUT_OFFSET           0xa08
#define GC_USB_DIEPINT8_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT8_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT8_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT8_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT8_INTKNTXFEMP_OFFSET       0xa08
#define GC_USB_DIEPINT8_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT8_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT8_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT8_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT8_INTKNEPMIS_OFFSET        0xa08
#define GC_USB_DIEPINT8_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT8_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT8_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT8_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT8_INEPNAKEFF_OFFSET        0xa08
#define GC_USB_DIEPINT8_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT8_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT8_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT8_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT8_TXFEMP_OFFSET            0xa08
#define GC_USB_DIEPINT8_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT8_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT8_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT8_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT8_TXFIFOUNDRN_OFFSET       0xa08
#define GC_USB_DIEPINT8_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT8_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT8_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT8_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT8_BNAINTR_OFFSET           0xa08
#define GC_USB_DIEPINT8_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT8_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT8_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT8_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT8_PKTDRPSTS_OFFSET         0xa08
#define GC_USB_DIEPINT8_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT8_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT8_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT8_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT8_BBLEERR_OFFSET           0xa08
#define GC_USB_DIEPINT8_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT8_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT8_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT8_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT8_NAKINTRPT_OFFSET         0xa08
#define GC_USB_DIEPINT8_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT8_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT8_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT8_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT8_NYETINTRPT_OFFSET        0xa08
#define GC_USB_DIEPTSIZ8_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ8_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ8_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ8_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ8_XFERSIZE_OFFSET         0xa10
#define GC_USB_DIEPTSIZ8_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ8_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ8_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ8_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ8_PKTCNT_OFFSET           0xa10
#define GC_USB_DIEPTSIZ8_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ8_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ8_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ8_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ8_MC_OFFSET               0xa10
#define GC_USB_DIEPDMA8_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA8_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA8_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA8_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA8_DMAADDR_OFFSET           0xa14
#define GC_USB_DTXFSTS8_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS8_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS8_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS8_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS8_INEPTXFSPCAVAIL_OFFSET   0xa18
#define GC_USB_DIEPDMAB8_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB8_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB8_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB8_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB8_DMABUFFERADDR_OFFSET    0xa1c
#define GC_USB_DIEPCTL9_MPS_LSB                  0x0
#define GC_USB_DIEPCTL9_MPS_MASK                 0x7ff
#define GC_USB_DIEPCTL9_MPS_SIZE                 0xb
#define GC_USB_DIEPCTL9_MPS_DEFAULT              0x0
#define GC_USB_DIEPCTL9_MPS_OFFSET               0xa20
#define GC_USB_DIEPCTL9_USBACTEP_LSB             0xf
#define GC_USB_DIEPCTL9_USBACTEP_MASK            0x8000
#define GC_USB_DIEPCTL9_USBACTEP_SIZE            0x1
#define GC_USB_DIEPCTL9_USBACTEP_DEFAULT         0x0
#define GC_USB_DIEPCTL9_USBACTEP_OFFSET          0xa20
#define GC_USB_DIEPCTL9_DPID_LSB                 0x10
#define GC_USB_DIEPCTL9_DPID_MASK                0x10000
#define GC_USB_DIEPCTL9_DPID_SIZE                0x1
#define GC_USB_DIEPCTL9_DPID_DEFAULT             0x0
#define GC_USB_DIEPCTL9_DPID_OFFSET              0xa20
#define GC_USB_DIEPCTL9_NAKSTS_LSB               0x11
#define GC_USB_DIEPCTL9_NAKSTS_MASK              0x20000
#define GC_USB_DIEPCTL9_NAKSTS_SIZE              0x1
#define GC_USB_DIEPCTL9_NAKSTS_DEFAULT           0x0
#define GC_USB_DIEPCTL9_NAKSTS_OFFSET            0xa20
#define GC_USB_DIEPCTL9_EPTYPE_LSB               0x12
#define GC_USB_DIEPCTL9_EPTYPE_MASK              0xc0000
#define GC_USB_DIEPCTL9_EPTYPE_SIZE              0x2
#define GC_USB_DIEPCTL9_EPTYPE_DEFAULT           0x0
#define GC_USB_DIEPCTL9_EPTYPE_OFFSET            0xa20
#define GC_USB_DIEPCTL9_STALL_LSB                0x15
#define GC_USB_DIEPCTL9_STALL_MASK               0x200000
#define GC_USB_DIEPCTL9_STALL_SIZE               0x1
#define GC_USB_DIEPCTL9_STALL_DEFAULT            0x0
#define GC_USB_DIEPCTL9_STALL_OFFSET             0xa20
#define GC_USB_DIEPCTL9_TXFNUM_LSB               0x16
#define GC_USB_DIEPCTL9_TXFNUM_MASK              0x3c00000
#define GC_USB_DIEPCTL9_TXFNUM_SIZE              0x4
#define GC_USB_DIEPCTL9_TXFNUM_DEFAULT           0x0
#define GC_USB_DIEPCTL9_TXFNUM_OFFSET            0xa20
#define GC_USB_DIEPCTL9_CNAK_LSB                 0x1a
#define GC_USB_DIEPCTL9_CNAK_MASK                0x4000000
#define GC_USB_DIEPCTL9_CNAK_SIZE                0x1
#define GC_USB_DIEPCTL9_CNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL9_CNAK_OFFSET              0xa20
#define GC_USB_DIEPCTL9_SNAK_LSB                 0x1b
#define GC_USB_DIEPCTL9_SNAK_MASK                0x8000000
#define GC_USB_DIEPCTL9_SNAK_SIZE                0x1
#define GC_USB_DIEPCTL9_SNAK_DEFAULT             0x0
#define GC_USB_DIEPCTL9_SNAK_OFFSET              0xa20
#define GC_USB_DIEPCTL9_SETD0PID_LSB             0x1c
#define GC_USB_DIEPCTL9_SETD0PID_MASK            0x10000000
#define GC_USB_DIEPCTL9_SETD0PID_SIZE            0x1
#define GC_USB_DIEPCTL9_SETD0PID_DEFAULT         0x0
#define GC_USB_DIEPCTL9_SETD0PID_OFFSET          0xa20
#define GC_USB_DIEPCTL9_SETD1PID_LSB             0x1d
#define GC_USB_DIEPCTL9_SETD1PID_MASK            0x20000000
#define GC_USB_DIEPCTL9_SETD1PID_SIZE            0x1
#define GC_USB_DIEPCTL9_SETD1PID_DEFAULT         0x0
#define GC_USB_DIEPCTL9_SETD1PID_OFFSET          0xa20
#define GC_USB_DIEPCTL9_EPDIS_LSB                0x1e
#define GC_USB_DIEPCTL9_EPDIS_MASK               0x40000000
#define GC_USB_DIEPCTL9_EPDIS_SIZE               0x1
#define GC_USB_DIEPCTL9_EPDIS_DEFAULT            0x0
#define GC_USB_DIEPCTL9_EPDIS_OFFSET             0xa20
#define GC_USB_DIEPCTL9_EPENA_LSB                0x1f
#define GC_USB_DIEPCTL9_EPENA_MASK               0x80000000
#define GC_USB_DIEPCTL9_EPENA_SIZE               0x1
#define GC_USB_DIEPCTL9_EPENA_DEFAULT            0x0
#define GC_USB_DIEPCTL9_EPENA_OFFSET             0xa20
#define GC_USB_DIEPINT9_XFERCOMPL_LSB            0x0
#define GC_USB_DIEPINT9_XFERCOMPL_MASK           0x1
#define GC_USB_DIEPINT9_XFERCOMPL_SIZE           0x1
#define GC_USB_DIEPINT9_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DIEPINT9_XFERCOMPL_OFFSET         0xa28
#define GC_USB_DIEPINT9_EPDISBLD_LSB             0x1
#define GC_USB_DIEPINT9_EPDISBLD_MASK            0x2
#define GC_USB_DIEPINT9_EPDISBLD_SIZE            0x1
#define GC_USB_DIEPINT9_EPDISBLD_DEFAULT         0x0
#define GC_USB_DIEPINT9_EPDISBLD_OFFSET          0xa28
#define GC_USB_DIEPINT9_AHBERR_LSB               0x2
#define GC_USB_DIEPINT9_AHBERR_MASK              0x4
#define GC_USB_DIEPINT9_AHBERR_SIZE              0x1
#define GC_USB_DIEPINT9_AHBERR_DEFAULT           0x0
#define GC_USB_DIEPINT9_AHBERR_OFFSET            0xa28
#define GC_USB_DIEPINT9_TIMEOUT_LSB              0x3
#define GC_USB_DIEPINT9_TIMEOUT_MASK             0x8
#define GC_USB_DIEPINT9_TIMEOUT_SIZE             0x1
#define GC_USB_DIEPINT9_TIMEOUT_DEFAULT          0x0
#define GC_USB_DIEPINT9_TIMEOUT_OFFSET           0xa28
#define GC_USB_DIEPINT9_INTKNTXFEMP_LSB          0x4
#define GC_USB_DIEPINT9_INTKNTXFEMP_MASK         0x10
#define GC_USB_DIEPINT9_INTKNTXFEMP_SIZE         0x1
#define GC_USB_DIEPINT9_INTKNTXFEMP_DEFAULT      0x0
#define GC_USB_DIEPINT9_INTKNTXFEMP_OFFSET       0xa28
#define GC_USB_DIEPINT9_INTKNEPMIS_LSB           0x5
#define GC_USB_DIEPINT9_INTKNEPMIS_MASK          0x20
#define GC_USB_DIEPINT9_INTKNEPMIS_SIZE          0x1
#define GC_USB_DIEPINT9_INTKNEPMIS_DEFAULT       0x0
#define GC_USB_DIEPINT9_INTKNEPMIS_OFFSET        0xa28
#define GC_USB_DIEPINT9_INEPNAKEFF_LSB           0x6
#define GC_USB_DIEPINT9_INEPNAKEFF_MASK          0x40
#define GC_USB_DIEPINT9_INEPNAKEFF_SIZE          0x1
#define GC_USB_DIEPINT9_INEPNAKEFF_DEFAULT       0x0
#define GC_USB_DIEPINT9_INEPNAKEFF_OFFSET        0xa28
#define GC_USB_DIEPINT9_TXFEMP_LSB               0x7
#define GC_USB_DIEPINT9_TXFEMP_MASK              0x80
#define GC_USB_DIEPINT9_TXFEMP_SIZE              0x1
#define GC_USB_DIEPINT9_TXFEMP_DEFAULT           0x0
#define GC_USB_DIEPINT9_TXFEMP_OFFSET            0xa28
#define GC_USB_DIEPINT9_TXFIFOUNDRN_LSB          0x8
#define GC_USB_DIEPINT9_TXFIFOUNDRN_MASK         0x100
#define GC_USB_DIEPINT9_TXFIFOUNDRN_SIZE         0x1
#define GC_USB_DIEPINT9_TXFIFOUNDRN_DEFAULT      0x0
#define GC_USB_DIEPINT9_TXFIFOUNDRN_OFFSET       0xa28
#define GC_USB_DIEPINT9_BNAINTR_LSB              0x9
#define GC_USB_DIEPINT9_BNAINTR_MASK             0x200
#define GC_USB_DIEPINT9_BNAINTR_SIZE             0x1
#define GC_USB_DIEPINT9_BNAINTR_DEFAULT          0x0
#define GC_USB_DIEPINT9_BNAINTR_OFFSET           0xa28
#define GC_USB_DIEPINT9_PKTDRPSTS_LSB            0xb
#define GC_USB_DIEPINT9_PKTDRPSTS_MASK           0x800
#define GC_USB_DIEPINT9_PKTDRPSTS_SIZE           0x1
#define GC_USB_DIEPINT9_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DIEPINT9_PKTDRPSTS_OFFSET         0xa28
#define GC_USB_DIEPINT9_BBLEERR_LSB              0xc
#define GC_USB_DIEPINT9_BBLEERR_MASK             0x1000
#define GC_USB_DIEPINT9_BBLEERR_SIZE             0x1
#define GC_USB_DIEPINT9_BBLEERR_DEFAULT          0x0
#define GC_USB_DIEPINT9_BBLEERR_OFFSET           0xa28
#define GC_USB_DIEPINT9_NAKINTRPT_LSB            0xd
#define GC_USB_DIEPINT9_NAKINTRPT_MASK           0x2000
#define GC_USB_DIEPINT9_NAKINTRPT_SIZE           0x1
#define GC_USB_DIEPINT9_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DIEPINT9_NAKINTRPT_OFFSET         0xa28
#define GC_USB_DIEPINT9_NYETINTRPT_LSB           0xe
#define GC_USB_DIEPINT9_NYETINTRPT_MASK          0x4000
#define GC_USB_DIEPINT9_NYETINTRPT_SIZE          0x1
#define GC_USB_DIEPINT9_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT9_NYETINTRPT_OFFSET        0xa28
#define GC_USB_DIEPTSIZ9_XFERSIZE_LSB            0x0
#define GC_USB_DIEPTSIZ9_XFERSIZE_MASK           0x7ffff
#define GC_USB_DIEPTSIZ9_XFERSIZE_SIZE           0x13
#define GC_USB_DIEPTSIZ9_XFERSIZE_DEFAULT        0x0
#define GC_USB_DIEPTSIZ9_XFERSIZE_OFFSET         0xa30
#define GC_USB_DIEPTSIZ9_PKTCNT_LSB              0x13
#define GC_USB_DIEPTSIZ9_PKTCNT_MASK             0x1ff80000
#define GC_USB_DIEPTSIZ9_PKTCNT_SIZE             0xa
#define GC_USB_DIEPTSIZ9_PKTCNT_DEFAULT          0x0
#define GC_USB_DIEPTSIZ9_PKTCNT_OFFSET           0xa30
#define GC_USB_DIEPTSIZ9_MC_LSB                  0x1d
#define GC_USB_DIEPTSIZ9_MC_MASK                 0x60000000
#define GC_USB_DIEPTSIZ9_MC_SIZE                 0x2
#define GC_USB_DIEPTSIZ9_MC_DEFAULT              0x0
#define GC_USB_DIEPTSIZ9_MC_OFFSET               0xa30
#define GC_USB_DIEPDMA9_DMAADDR_LSB              0x0
#define GC_USB_DIEPDMA9_DMAADDR_MASK             0xffffffff
#define GC_USB_DIEPDMA9_DMAADDR_SIZE             0x20
#define GC_USB_DIEPDMA9_DMAADDR_DEFAULT          0x0
#define GC_USB_DIEPDMA9_DMAADDR_OFFSET           0xa34
#define GC_USB_DTXFSTS9_INEPTXFSPCAVAIL_LSB      0x0
#define GC_USB_DTXFSTS9_INEPTXFSPCAVAIL_MASK     0xffff
#define GC_USB_DTXFSTS9_INEPTXFSPCAVAIL_SIZE     0x10
#define GC_USB_DTXFSTS9_INEPTXFSPCAVAIL_DEFAULT  0x0
#define GC_USB_DTXFSTS9_INEPTXFSPCAVAIL_OFFSET   0xa38
#define GC_USB_DIEPDMAB9_DMABUFFERADDR_LSB       0x0
#define GC_USB_DIEPDMAB9_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DIEPDMAB9_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DIEPDMAB9_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DIEPDMAB9_DMABUFFERADDR_OFFSET    0xa3c
#define GC_USB_DIEPCTL10_MPS_LSB                 0x0
#define GC_USB_DIEPCTL10_MPS_MASK                0x7ff
#define GC_USB_DIEPCTL10_MPS_SIZE                0xb
#define GC_USB_DIEPCTL10_MPS_DEFAULT             0x0
#define GC_USB_DIEPCTL10_MPS_OFFSET              0xa40
#define GC_USB_DIEPCTL10_USBACTEP_LSB            0xf
#define GC_USB_DIEPCTL10_USBACTEP_MASK           0x8000
#define GC_USB_DIEPCTL10_USBACTEP_SIZE           0x1
#define GC_USB_DIEPCTL10_USBACTEP_DEFAULT        0x0
#define GC_USB_DIEPCTL10_USBACTEP_OFFSET         0xa40
#define GC_USB_DIEPCTL10_DPID_LSB                0x10
#define GC_USB_DIEPCTL10_DPID_MASK               0x10000
#define GC_USB_DIEPCTL10_DPID_SIZE               0x1
#define GC_USB_DIEPCTL10_DPID_DEFAULT            0x0
#define GC_USB_DIEPCTL10_DPID_OFFSET             0xa40
#define GC_USB_DIEPCTL10_NAKSTS_LSB              0x11
#define GC_USB_DIEPCTL10_NAKSTS_MASK             0x20000
#define GC_USB_DIEPCTL10_NAKSTS_SIZE             0x1
#define GC_USB_DIEPCTL10_NAKSTS_DEFAULT          0x0
#define GC_USB_DIEPCTL10_NAKSTS_OFFSET           0xa40
#define GC_USB_DIEPCTL10_EPTYPE_LSB              0x12
#define GC_USB_DIEPCTL10_EPTYPE_MASK             0xc0000
#define GC_USB_DIEPCTL10_EPTYPE_SIZE             0x2
#define GC_USB_DIEPCTL10_EPTYPE_DEFAULT          0x0
#define GC_USB_DIEPCTL10_EPTYPE_OFFSET           0xa40
#define GC_USB_DIEPCTL10_STALL_LSB               0x15
#define GC_USB_DIEPCTL10_STALL_MASK              0x200000
#define GC_USB_DIEPCTL10_STALL_SIZE              0x1
#define GC_USB_DIEPCTL10_STALL_DEFAULT           0x0
#define GC_USB_DIEPCTL10_STALL_OFFSET            0xa40
#define GC_USB_DIEPCTL10_TXFNUM_LSB              0x16
#define GC_USB_DIEPCTL10_TXFNUM_MASK             0x3c00000
#define GC_USB_DIEPCTL10_TXFNUM_SIZE             0x4
#define GC_USB_DIEPCTL10_TXFNUM_DEFAULT          0x0
#define GC_USB_DIEPCTL10_TXFNUM_OFFSET           0xa40
#define GC_USB_DIEPCTL10_CNAK_LSB                0x1a
#define GC_USB_DIEPCTL10_CNAK_MASK               0x4000000
#define GC_USB_DIEPCTL10_CNAK_SIZE               0x1
#define GC_USB_DIEPCTL10_CNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL10_CNAK_OFFSET             0xa40
#define GC_USB_DIEPCTL10_SNAK_LSB                0x1b
#define GC_USB_DIEPCTL10_SNAK_MASK               0x8000000
#define GC_USB_DIEPCTL10_SNAK_SIZE               0x1
#define GC_USB_DIEPCTL10_SNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL10_SNAK_OFFSET             0xa40
#define GC_USB_DIEPCTL10_SETD0PID_LSB            0x1c
#define GC_USB_DIEPCTL10_SETD0PID_MASK           0x10000000
#define GC_USB_DIEPCTL10_SETD0PID_SIZE           0x1
#define GC_USB_DIEPCTL10_SETD0PID_DEFAULT        0x0
#define GC_USB_DIEPCTL10_SETD0PID_OFFSET         0xa40
#define GC_USB_DIEPCTL10_SETD1PID_LSB            0x1d
#define GC_USB_DIEPCTL10_SETD1PID_MASK           0x20000000
#define GC_USB_DIEPCTL10_SETD1PID_SIZE           0x1
#define GC_USB_DIEPCTL10_SETD1PID_DEFAULT        0x0
#define GC_USB_DIEPCTL10_SETD1PID_OFFSET         0xa40
#define GC_USB_DIEPCTL10_EPDIS_LSB               0x1e
#define GC_USB_DIEPCTL10_EPDIS_MASK              0x40000000
#define GC_USB_DIEPCTL10_EPDIS_SIZE              0x1
#define GC_USB_DIEPCTL10_EPDIS_DEFAULT           0x0
#define GC_USB_DIEPCTL10_EPDIS_OFFSET            0xa40
#define GC_USB_DIEPCTL10_EPENA_LSB               0x1f
#define GC_USB_DIEPCTL10_EPENA_MASK              0x80000000
#define GC_USB_DIEPCTL10_EPENA_SIZE              0x1
#define GC_USB_DIEPCTL10_EPENA_DEFAULT           0x0
#define GC_USB_DIEPCTL10_EPENA_OFFSET            0xa40
#define GC_USB_DIEPINT10_XFERCOMPL_LSB           0x0
#define GC_USB_DIEPINT10_XFERCOMPL_MASK          0x1
#define GC_USB_DIEPINT10_XFERCOMPL_SIZE          0x1
#define GC_USB_DIEPINT10_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DIEPINT10_XFERCOMPL_OFFSET        0xa48
#define GC_USB_DIEPINT10_EPDISBLD_LSB            0x1
#define GC_USB_DIEPINT10_EPDISBLD_MASK           0x2
#define GC_USB_DIEPINT10_EPDISBLD_SIZE           0x1
#define GC_USB_DIEPINT10_EPDISBLD_DEFAULT        0x0
#define GC_USB_DIEPINT10_EPDISBLD_OFFSET         0xa48
#define GC_USB_DIEPINT10_AHBERR_LSB              0x2
#define GC_USB_DIEPINT10_AHBERR_MASK             0x4
#define GC_USB_DIEPINT10_AHBERR_SIZE             0x1
#define GC_USB_DIEPINT10_AHBERR_DEFAULT          0x0
#define GC_USB_DIEPINT10_AHBERR_OFFSET           0xa48
#define GC_USB_DIEPINT10_TIMEOUT_LSB             0x3
#define GC_USB_DIEPINT10_TIMEOUT_MASK            0x8
#define GC_USB_DIEPINT10_TIMEOUT_SIZE            0x1
#define GC_USB_DIEPINT10_TIMEOUT_DEFAULT         0x0
#define GC_USB_DIEPINT10_TIMEOUT_OFFSET          0xa48
#define GC_USB_DIEPINT10_INTKNTXFEMP_LSB         0x4
#define GC_USB_DIEPINT10_INTKNTXFEMP_MASK        0x10
#define GC_USB_DIEPINT10_INTKNTXFEMP_SIZE        0x1
#define GC_USB_DIEPINT10_INTKNTXFEMP_DEFAULT     0x0
#define GC_USB_DIEPINT10_INTKNTXFEMP_OFFSET      0xa48
#define GC_USB_DIEPINT10_INTKNEPMIS_LSB          0x5
#define GC_USB_DIEPINT10_INTKNEPMIS_MASK         0x20
#define GC_USB_DIEPINT10_INTKNEPMIS_SIZE         0x1
#define GC_USB_DIEPINT10_INTKNEPMIS_DEFAULT      0x0
#define GC_USB_DIEPINT10_INTKNEPMIS_OFFSET       0xa48
#define GC_USB_DIEPINT10_INEPNAKEFF_LSB          0x6
#define GC_USB_DIEPINT10_INEPNAKEFF_MASK         0x40
#define GC_USB_DIEPINT10_INEPNAKEFF_SIZE         0x1
#define GC_USB_DIEPINT10_INEPNAKEFF_DEFAULT      0x0
#define GC_USB_DIEPINT10_INEPNAKEFF_OFFSET       0xa48
#define GC_USB_DIEPINT10_TXFEMP_LSB              0x7
#define GC_USB_DIEPINT10_TXFEMP_MASK             0x80
#define GC_USB_DIEPINT10_TXFEMP_SIZE             0x1
#define GC_USB_DIEPINT10_TXFEMP_DEFAULT          0x0
#define GC_USB_DIEPINT10_TXFEMP_OFFSET           0xa48
#define GC_USB_DIEPINT10_TXFIFOUNDRN_LSB         0x8
#define GC_USB_DIEPINT10_TXFIFOUNDRN_MASK        0x100
#define GC_USB_DIEPINT10_TXFIFOUNDRN_SIZE        0x1
#define GC_USB_DIEPINT10_TXFIFOUNDRN_DEFAULT     0x0
#define GC_USB_DIEPINT10_TXFIFOUNDRN_OFFSET      0xa48
#define GC_USB_DIEPINT10_BNAINTR_LSB             0x9
#define GC_USB_DIEPINT10_BNAINTR_MASK            0x200
#define GC_USB_DIEPINT10_BNAINTR_SIZE            0x1
#define GC_USB_DIEPINT10_BNAINTR_DEFAULT         0x0
#define GC_USB_DIEPINT10_BNAINTR_OFFSET          0xa48
#define GC_USB_DIEPINT10_PKTDRPSTS_LSB           0xb
#define GC_USB_DIEPINT10_PKTDRPSTS_MASK          0x800
#define GC_USB_DIEPINT10_PKTDRPSTS_SIZE          0x1
#define GC_USB_DIEPINT10_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DIEPINT10_PKTDRPSTS_OFFSET        0xa48
#define GC_USB_DIEPINT10_BBLEERR_LSB             0xc
#define GC_USB_DIEPINT10_BBLEERR_MASK            0x1000
#define GC_USB_DIEPINT10_BBLEERR_SIZE            0x1
#define GC_USB_DIEPINT10_BBLEERR_DEFAULT         0x0
#define GC_USB_DIEPINT10_BBLEERR_OFFSET          0xa48
#define GC_USB_DIEPINT10_NAKINTRPT_LSB           0xd
#define GC_USB_DIEPINT10_NAKINTRPT_MASK          0x2000
#define GC_USB_DIEPINT10_NAKINTRPT_SIZE          0x1
#define GC_USB_DIEPINT10_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT10_NAKINTRPT_OFFSET        0xa48
#define GC_USB_DIEPINT10_NYETINTRPT_LSB          0xe
#define GC_USB_DIEPINT10_NYETINTRPT_MASK         0x4000
#define GC_USB_DIEPINT10_NYETINTRPT_SIZE         0x1
#define GC_USB_DIEPINT10_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DIEPINT10_NYETINTRPT_OFFSET       0xa48
#define GC_USB_DIEPTSIZ10_XFERSIZE_LSB           0x0
#define GC_USB_DIEPTSIZ10_XFERSIZE_MASK          0x7ffff
#define GC_USB_DIEPTSIZ10_XFERSIZE_SIZE          0x13
#define GC_USB_DIEPTSIZ10_XFERSIZE_DEFAULT       0x0
#define GC_USB_DIEPTSIZ10_XFERSIZE_OFFSET        0xa50
#define GC_USB_DIEPTSIZ10_PKTCNT_LSB             0x13
#define GC_USB_DIEPTSIZ10_PKTCNT_MASK            0x1ff80000
#define GC_USB_DIEPTSIZ10_PKTCNT_SIZE            0xa
#define GC_USB_DIEPTSIZ10_PKTCNT_DEFAULT         0x0
#define GC_USB_DIEPTSIZ10_PKTCNT_OFFSET          0xa50
#define GC_USB_DIEPTSIZ10_MC_LSB                 0x1d
#define GC_USB_DIEPTSIZ10_MC_MASK                0x60000000
#define GC_USB_DIEPTSIZ10_MC_SIZE                0x2
#define GC_USB_DIEPTSIZ10_MC_DEFAULT             0x0
#define GC_USB_DIEPTSIZ10_MC_OFFSET              0xa50
#define GC_USB_DIEPDMA10_DMAADDR_LSB             0x0
#define GC_USB_DIEPDMA10_DMAADDR_MASK            0xffffffff
#define GC_USB_DIEPDMA10_DMAADDR_SIZE            0x20
#define GC_USB_DIEPDMA10_DMAADDR_DEFAULT         0x0
#define GC_USB_DIEPDMA10_DMAADDR_OFFSET          0xa54
#define GC_USB_DTXFSTS10_INEPTXFSPCAVAIL_LSB     0x0
#define GC_USB_DTXFSTS10_INEPTXFSPCAVAIL_MASK    0xffff
#define GC_USB_DTXFSTS10_INEPTXFSPCAVAIL_SIZE    0x10
#define GC_USB_DTXFSTS10_INEPTXFSPCAVAIL_DEFAULT 0x0
#define GC_USB_DTXFSTS10_INEPTXFSPCAVAIL_OFFSET  0xa58
#define GC_USB_DIEPDMAB10_DMABUFFERADDR_LSB      0x0
#define GC_USB_DIEPDMAB10_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DIEPDMAB10_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DIEPDMAB10_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DIEPDMAB10_DMABUFFERADDR_OFFSET   0xa5c
#define GC_USB_DIEPCTL11_MPS_LSB                 0x0
#define GC_USB_DIEPCTL11_MPS_MASK                0x7ff
#define GC_USB_DIEPCTL11_MPS_SIZE                0xb
#define GC_USB_DIEPCTL11_MPS_DEFAULT             0x0
#define GC_USB_DIEPCTL11_MPS_OFFSET              0xa60
#define GC_USB_DIEPCTL11_USBACTEP_LSB            0xf
#define GC_USB_DIEPCTL11_USBACTEP_MASK           0x8000
#define GC_USB_DIEPCTL11_USBACTEP_SIZE           0x1
#define GC_USB_DIEPCTL11_USBACTEP_DEFAULT        0x0
#define GC_USB_DIEPCTL11_USBACTEP_OFFSET         0xa60
#define GC_USB_DIEPCTL11_DPID_LSB                0x10
#define GC_USB_DIEPCTL11_DPID_MASK               0x10000
#define GC_USB_DIEPCTL11_DPID_SIZE               0x1
#define GC_USB_DIEPCTL11_DPID_DEFAULT            0x0
#define GC_USB_DIEPCTL11_DPID_OFFSET             0xa60
#define GC_USB_DIEPCTL11_NAKSTS_LSB              0x11
#define GC_USB_DIEPCTL11_NAKSTS_MASK             0x20000
#define GC_USB_DIEPCTL11_NAKSTS_SIZE             0x1
#define GC_USB_DIEPCTL11_NAKSTS_DEFAULT          0x0
#define GC_USB_DIEPCTL11_NAKSTS_OFFSET           0xa60
#define GC_USB_DIEPCTL11_EPTYPE_LSB              0x12
#define GC_USB_DIEPCTL11_EPTYPE_MASK             0xc0000
#define GC_USB_DIEPCTL11_EPTYPE_SIZE             0x2
#define GC_USB_DIEPCTL11_EPTYPE_DEFAULT          0x0
#define GC_USB_DIEPCTL11_EPTYPE_OFFSET           0xa60
#define GC_USB_DIEPCTL11_STALL_LSB               0x15
#define GC_USB_DIEPCTL11_STALL_MASK              0x200000
#define GC_USB_DIEPCTL11_STALL_SIZE              0x1
#define GC_USB_DIEPCTL11_STALL_DEFAULT           0x0
#define GC_USB_DIEPCTL11_STALL_OFFSET            0xa60
#define GC_USB_DIEPCTL11_TXFNUM_LSB              0x16
#define GC_USB_DIEPCTL11_TXFNUM_MASK             0x3c00000
#define GC_USB_DIEPCTL11_TXFNUM_SIZE             0x4
#define GC_USB_DIEPCTL11_TXFNUM_DEFAULT          0x0
#define GC_USB_DIEPCTL11_TXFNUM_OFFSET           0xa60
#define GC_USB_DIEPCTL11_CNAK_LSB                0x1a
#define GC_USB_DIEPCTL11_CNAK_MASK               0x4000000
#define GC_USB_DIEPCTL11_CNAK_SIZE               0x1
#define GC_USB_DIEPCTL11_CNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL11_CNAK_OFFSET             0xa60
#define GC_USB_DIEPCTL11_SNAK_LSB                0x1b
#define GC_USB_DIEPCTL11_SNAK_MASK               0x8000000
#define GC_USB_DIEPCTL11_SNAK_SIZE               0x1
#define GC_USB_DIEPCTL11_SNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL11_SNAK_OFFSET             0xa60
#define GC_USB_DIEPCTL11_SETD0PID_LSB            0x1c
#define GC_USB_DIEPCTL11_SETD0PID_MASK           0x10000000
#define GC_USB_DIEPCTL11_SETD0PID_SIZE           0x1
#define GC_USB_DIEPCTL11_SETD0PID_DEFAULT        0x0
#define GC_USB_DIEPCTL11_SETD0PID_OFFSET         0xa60
#define GC_USB_DIEPCTL11_SETD1PID_LSB            0x1d
#define GC_USB_DIEPCTL11_SETD1PID_MASK           0x20000000
#define GC_USB_DIEPCTL11_SETD1PID_SIZE           0x1
#define GC_USB_DIEPCTL11_SETD1PID_DEFAULT        0x0
#define GC_USB_DIEPCTL11_SETD1PID_OFFSET         0xa60
#define GC_USB_DIEPCTL11_EPDIS_LSB               0x1e
#define GC_USB_DIEPCTL11_EPDIS_MASK              0x40000000
#define GC_USB_DIEPCTL11_EPDIS_SIZE              0x1
#define GC_USB_DIEPCTL11_EPDIS_DEFAULT           0x0
#define GC_USB_DIEPCTL11_EPDIS_OFFSET            0xa60
#define GC_USB_DIEPCTL11_EPENA_LSB               0x1f
#define GC_USB_DIEPCTL11_EPENA_MASK              0x80000000
#define GC_USB_DIEPCTL11_EPENA_SIZE              0x1
#define GC_USB_DIEPCTL11_EPENA_DEFAULT           0x0
#define GC_USB_DIEPCTL11_EPENA_OFFSET            0xa60
#define GC_USB_DIEPINT11_XFERCOMPL_LSB           0x0
#define GC_USB_DIEPINT11_XFERCOMPL_MASK          0x1
#define GC_USB_DIEPINT11_XFERCOMPL_SIZE          0x1
#define GC_USB_DIEPINT11_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DIEPINT11_XFERCOMPL_OFFSET        0xa68
#define GC_USB_DIEPINT11_EPDISBLD_LSB            0x1
#define GC_USB_DIEPINT11_EPDISBLD_MASK           0x2
#define GC_USB_DIEPINT11_EPDISBLD_SIZE           0x1
#define GC_USB_DIEPINT11_EPDISBLD_DEFAULT        0x0
#define GC_USB_DIEPINT11_EPDISBLD_OFFSET         0xa68
#define GC_USB_DIEPINT11_AHBERR_LSB              0x2
#define GC_USB_DIEPINT11_AHBERR_MASK             0x4
#define GC_USB_DIEPINT11_AHBERR_SIZE             0x1
#define GC_USB_DIEPINT11_AHBERR_DEFAULT          0x0
#define GC_USB_DIEPINT11_AHBERR_OFFSET           0xa68
#define GC_USB_DIEPINT11_TIMEOUT_LSB             0x3
#define GC_USB_DIEPINT11_TIMEOUT_MASK            0x8
#define GC_USB_DIEPINT11_TIMEOUT_SIZE            0x1
#define GC_USB_DIEPINT11_TIMEOUT_DEFAULT         0x0
#define GC_USB_DIEPINT11_TIMEOUT_OFFSET          0xa68
#define GC_USB_DIEPINT11_INTKNTXFEMP_LSB         0x4
#define GC_USB_DIEPINT11_INTKNTXFEMP_MASK        0x10
#define GC_USB_DIEPINT11_INTKNTXFEMP_SIZE        0x1
#define GC_USB_DIEPINT11_INTKNTXFEMP_DEFAULT     0x0
#define GC_USB_DIEPINT11_INTKNTXFEMP_OFFSET      0xa68
#define GC_USB_DIEPINT11_INTKNEPMIS_LSB          0x5
#define GC_USB_DIEPINT11_INTKNEPMIS_MASK         0x20
#define GC_USB_DIEPINT11_INTKNEPMIS_SIZE         0x1
#define GC_USB_DIEPINT11_INTKNEPMIS_DEFAULT      0x0
#define GC_USB_DIEPINT11_INTKNEPMIS_OFFSET       0xa68
#define GC_USB_DIEPINT11_INEPNAKEFF_LSB          0x6
#define GC_USB_DIEPINT11_INEPNAKEFF_MASK         0x40
#define GC_USB_DIEPINT11_INEPNAKEFF_SIZE         0x1
#define GC_USB_DIEPINT11_INEPNAKEFF_DEFAULT      0x0
#define GC_USB_DIEPINT11_INEPNAKEFF_OFFSET       0xa68
#define GC_USB_DIEPINT11_TXFEMP_LSB              0x7
#define GC_USB_DIEPINT11_TXFEMP_MASK             0x80
#define GC_USB_DIEPINT11_TXFEMP_SIZE             0x1
#define GC_USB_DIEPINT11_TXFEMP_DEFAULT          0x0
#define GC_USB_DIEPINT11_TXFEMP_OFFSET           0xa68
#define GC_USB_DIEPINT11_TXFIFOUNDRN_LSB         0x8
#define GC_USB_DIEPINT11_TXFIFOUNDRN_MASK        0x100
#define GC_USB_DIEPINT11_TXFIFOUNDRN_SIZE        0x1
#define GC_USB_DIEPINT11_TXFIFOUNDRN_DEFAULT     0x0
#define GC_USB_DIEPINT11_TXFIFOUNDRN_OFFSET      0xa68
#define GC_USB_DIEPINT11_BNAINTR_LSB             0x9
#define GC_USB_DIEPINT11_BNAINTR_MASK            0x200
#define GC_USB_DIEPINT11_BNAINTR_SIZE            0x1
#define GC_USB_DIEPINT11_BNAINTR_DEFAULT         0x0
#define GC_USB_DIEPINT11_BNAINTR_OFFSET          0xa68
#define GC_USB_DIEPINT11_PKTDRPSTS_LSB           0xb
#define GC_USB_DIEPINT11_PKTDRPSTS_MASK          0x800
#define GC_USB_DIEPINT11_PKTDRPSTS_SIZE          0x1
#define GC_USB_DIEPINT11_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DIEPINT11_PKTDRPSTS_OFFSET        0xa68
#define GC_USB_DIEPINT11_BBLEERR_LSB             0xc
#define GC_USB_DIEPINT11_BBLEERR_MASK            0x1000
#define GC_USB_DIEPINT11_BBLEERR_SIZE            0x1
#define GC_USB_DIEPINT11_BBLEERR_DEFAULT         0x0
#define GC_USB_DIEPINT11_BBLEERR_OFFSET          0xa68
#define GC_USB_DIEPINT11_NAKINTRPT_LSB           0xd
#define GC_USB_DIEPINT11_NAKINTRPT_MASK          0x2000
#define GC_USB_DIEPINT11_NAKINTRPT_SIZE          0x1
#define GC_USB_DIEPINT11_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT11_NAKINTRPT_OFFSET        0xa68
#define GC_USB_DIEPINT11_NYETINTRPT_LSB          0xe
#define GC_USB_DIEPINT11_NYETINTRPT_MASK         0x4000
#define GC_USB_DIEPINT11_NYETINTRPT_SIZE         0x1
#define GC_USB_DIEPINT11_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DIEPINT11_NYETINTRPT_OFFSET       0xa68
#define GC_USB_DIEPTSIZ11_XFERSIZE_LSB           0x0
#define GC_USB_DIEPTSIZ11_XFERSIZE_MASK          0x7ffff
#define GC_USB_DIEPTSIZ11_XFERSIZE_SIZE          0x13
#define GC_USB_DIEPTSIZ11_XFERSIZE_DEFAULT       0x0
#define GC_USB_DIEPTSIZ11_XFERSIZE_OFFSET        0xa70
#define GC_USB_DIEPTSIZ11_PKTCNT_LSB             0x13
#define GC_USB_DIEPTSIZ11_PKTCNT_MASK            0x1ff80000
#define GC_USB_DIEPTSIZ11_PKTCNT_SIZE            0xa
#define GC_USB_DIEPTSIZ11_PKTCNT_DEFAULT         0x0
#define GC_USB_DIEPTSIZ11_PKTCNT_OFFSET          0xa70
#define GC_USB_DIEPTSIZ11_MC_LSB                 0x1d
#define GC_USB_DIEPTSIZ11_MC_MASK                0x60000000
#define GC_USB_DIEPTSIZ11_MC_SIZE                0x2
#define GC_USB_DIEPTSIZ11_MC_DEFAULT             0x0
#define GC_USB_DIEPTSIZ11_MC_OFFSET              0xa70
#define GC_USB_DIEPDMA11_DMAADDR_LSB             0x0
#define GC_USB_DIEPDMA11_DMAADDR_MASK            0xffffffff
#define GC_USB_DIEPDMA11_DMAADDR_SIZE            0x20
#define GC_USB_DIEPDMA11_DMAADDR_DEFAULT         0x0
#define GC_USB_DIEPDMA11_DMAADDR_OFFSET          0xa74
#define GC_USB_DTXFSTS11_INEPTXFSPCAVAIL_LSB     0x0
#define GC_USB_DTXFSTS11_INEPTXFSPCAVAIL_MASK    0xffff
#define GC_USB_DTXFSTS11_INEPTXFSPCAVAIL_SIZE    0x10
#define GC_USB_DTXFSTS11_INEPTXFSPCAVAIL_DEFAULT 0x0
#define GC_USB_DTXFSTS11_INEPTXFSPCAVAIL_OFFSET  0xa78
#define GC_USB_DIEPDMAB11_DMABUFFERADDR_LSB      0x0
#define GC_USB_DIEPDMAB11_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DIEPDMAB11_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DIEPDMAB11_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DIEPDMAB11_DMABUFFERADDR_OFFSET   0xa7c
#define GC_USB_DIEPCTL12_MPS_LSB                 0x0
#define GC_USB_DIEPCTL12_MPS_MASK                0x7ff
#define GC_USB_DIEPCTL12_MPS_SIZE                0xb
#define GC_USB_DIEPCTL12_MPS_DEFAULT             0x0
#define GC_USB_DIEPCTL12_MPS_OFFSET              0xa80
#define GC_USB_DIEPCTL12_USBACTEP_LSB            0xf
#define GC_USB_DIEPCTL12_USBACTEP_MASK           0x8000
#define GC_USB_DIEPCTL12_USBACTEP_SIZE           0x1
#define GC_USB_DIEPCTL12_USBACTEP_DEFAULT        0x0
#define GC_USB_DIEPCTL12_USBACTEP_OFFSET         0xa80
#define GC_USB_DIEPCTL12_DPID_LSB                0x10
#define GC_USB_DIEPCTL12_DPID_MASK               0x10000
#define GC_USB_DIEPCTL12_DPID_SIZE               0x1
#define GC_USB_DIEPCTL12_DPID_DEFAULT            0x0
#define GC_USB_DIEPCTL12_DPID_OFFSET             0xa80
#define GC_USB_DIEPCTL12_NAKSTS_LSB              0x11
#define GC_USB_DIEPCTL12_NAKSTS_MASK             0x20000
#define GC_USB_DIEPCTL12_NAKSTS_SIZE             0x1
#define GC_USB_DIEPCTL12_NAKSTS_DEFAULT          0x0
#define GC_USB_DIEPCTL12_NAKSTS_OFFSET           0xa80
#define GC_USB_DIEPCTL12_EPTYPE_LSB              0x12
#define GC_USB_DIEPCTL12_EPTYPE_MASK             0xc0000
#define GC_USB_DIEPCTL12_EPTYPE_SIZE             0x2
#define GC_USB_DIEPCTL12_EPTYPE_DEFAULT          0x0
#define GC_USB_DIEPCTL12_EPTYPE_OFFSET           0xa80
#define GC_USB_DIEPCTL12_STALL_LSB               0x15
#define GC_USB_DIEPCTL12_STALL_MASK              0x200000
#define GC_USB_DIEPCTL12_STALL_SIZE              0x1
#define GC_USB_DIEPCTL12_STALL_DEFAULT           0x0
#define GC_USB_DIEPCTL12_STALL_OFFSET            0xa80
#define GC_USB_DIEPCTL12_TXFNUM_LSB              0x16
#define GC_USB_DIEPCTL12_TXFNUM_MASK             0x3c00000
#define GC_USB_DIEPCTL12_TXFNUM_SIZE             0x4
#define GC_USB_DIEPCTL12_TXFNUM_DEFAULT          0x0
#define GC_USB_DIEPCTL12_TXFNUM_OFFSET           0xa80
#define GC_USB_DIEPCTL12_CNAK_LSB                0x1a
#define GC_USB_DIEPCTL12_CNAK_MASK               0x4000000
#define GC_USB_DIEPCTL12_CNAK_SIZE               0x1
#define GC_USB_DIEPCTL12_CNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL12_CNAK_OFFSET             0xa80
#define GC_USB_DIEPCTL12_SNAK_LSB                0x1b
#define GC_USB_DIEPCTL12_SNAK_MASK               0x8000000
#define GC_USB_DIEPCTL12_SNAK_SIZE               0x1
#define GC_USB_DIEPCTL12_SNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL12_SNAK_OFFSET             0xa80
#define GC_USB_DIEPCTL12_SETD0PID_LSB            0x1c
#define GC_USB_DIEPCTL12_SETD0PID_MASK           0x10000000
#define GC_USB_DIEPCTL12_SETD0PID_SIZE           0x1
#define GC_USB_DIEPCTL12_SETD0PID_DEFAULT        0x0
#define GC_USB_DIEPCTL12_SETD0PID_OFFSET         0xa80
#define GC_USB_DIEPCTL12_SETD1PID_LSB            0x1d
#define GC_USB_DIEPCTL12_SETD1PID_MASK           0x20000000
#define GC_USB_DIEPCTL12_SETD1PID_SIZE           0x1
#define GC_USB_DIEPCTL12_SETD1PID_DEFAULT        0x0
#define GC_USB_DIEPCTL12_SETD1PID_OFFSET         0xa80
#define GC_USB_DIEPCTL12_EPDIS_LSB               0x1e
#define GC_USB_DIEPCTL12_EPDIS_MASK              0x40000000
#define GC_USB_DIEPCTL12_EPDIS_SIZE              0x1
#define GC_USB_DIEPCTL12_EPDIS_DEFAULT           0x0
#define GC_USB_DIEPCTL12_EPDIS_OFFSET            0xa80
#define GC_USB_DIEPCTL12_EPENA_LSB               0x1f
#define GC_USB_DIEPCTL12_EPENA_MASK              0x80000000
#define GC_USB_DIEPCTL12_EPENA_SIZE              0x1
#define GC_USB_DIEPCTL12_EPENA_DEFAULT           0x0
#define GC_USB_DIEPCTL12_EPENA_OFFSET            0xa80
#define GC_USB_DIEPINT12_XFERCOMPL_LSB           0x0
#define GC_USB_DIEPINT12_XFERCOMPL_MASK          0x1
#define GC_USB_DIEPINT12_XFERCOMPL_SIZE          0x1
#define GC_USB_DIEPINT12_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DIEPINT12_XFERCOMPL_OFFSET        0xa88
#define GC_USB_DIEPINT12_EPDISBLD_LSB            0x1
#define GC_USB_DIEPINT12_EPDISBLD_MASK           0x2
#define GC_USB_DIEPINT12_EPDISBLD_SIZE           0x1
#define GC_USB_DIEPINT12_EPDISBLD_DEFAULT        0x0
#define GC_USB_DIEPINT12_EPDISBLD_OFFSET         0xa88
#define GC_USB_DIEPINT12_AHBERR_LSB              0x2
#define GC_USB_DIEPINT12_AHBERR_MASK             0x4
#define GC_USB_DIEPINT12_AHBERR_SIZE             0x1
#define GC_USB_DIEPINT12_AHBERR_DEFAULT          0x0
#define GC_USB_DIEPINT12_AHBERR_OFFSET           0xa88
#define GC_USB_DIEPINT12_TIMEOUT_LSB             0x3
#define GC_USB_DIEPINT12_TIMEOUT_MASK            0x8
#define GC_USB_DIEPINT12_TIMEOUT_SIZE            0x1
#define GC_USB_DIEPINT12_TIMEOUT_DEFAULT         0x0
#define GC_USB_DIEPINT12_TIMEOUT_OFFSET          0xa88
#define GC_USB_DIEPINT12_INTKNTXFEMP_LSB         0x4
#define GC_USB_DIEPINT12_INTKNTXFEMP_MASK        0x10
#define GC_USB_DIEPINT12_INTKNTXFEMP_SIZE        0x1
#define GC_USB_DIEPINT12_INTKNTXFEMP_DEFAULT     0x0
#define GC_USB_DIEPINT12_INTKNTXFEMP_OFFSET      0xa88
#define GC_USB_DIEPINT12_INTKNEPMIS_LSB          0x5
#define GC_USB_DIEPINT12_INTKNEPMIS_MASK         0x20
#define GC_USB_DIEPINT12_INTKNEPMIS_SIZE         0x1
#define GC_USB_DIEPINT12_INTKNEPMIS_DEFAULT      0x0
#define GC_USB_DIEPINT12_INTKNEPMIS_OFFSET       0xa88
#define GC_USB_DIEPINT12_INEPNAKEFF_LSB          0x6
#define GC_USB_DIEPINT12_INEPNAKEFF_MASK         0x40
#define GC_USB_DIEPINT12_INEPNAKEFF_SIZE         0x1
#define GC_USB_DIEPINT12_INEPNAKEFF_DEFAULT      0x0
#define GC_USB_DIEPINT12_INEPNAKEFF_OFFSET       0xa88
#define GC_USB_DIEPINT12_TXFEMP_LSB              0x7
#define GC_USB_DIEPINT12_TXFEMP_MASK             0x80
#define GC_USB_DIEPINT12_TXFEMP_SIZE             0x1
#define GC_USB_DIEPINT12_TXFEMP_DEFAULT          0x0
#define GC_USB_DIEPINT12_TXFEMP_OFFSET           0xa88
#define GC_USB_DIEPINT12_TXFIFOUNDRN_LSB         0x8
#define GC_USB_DIEPINT12_TXFIFOUNDRN_MASK        0x100
#define GC_USB_DIEPINT12_TXFIFOUNDRN_SIZE        0x1
#define GC_USB_DIEPINT12_TXFIFOUNDRN_DEFAULT     0x0
#define GC_USB_DIEPINT12_TXFIFOUNDRN_OFFSET      0xa88
#define GC_USB_DIEPINT12_BNAINTR_LSB             0x9
#define GC_USB_DIEPINT12_BNAINTR_MASK            0x200
#define GC_USB_DIEPINT12_BNAINTR_SIZE            0x1
#define GC_USB_DIEPINT12_BNAINTR_DEFAULT         0x0
#define GC_USB_DIEPINT12_BNAINTR_OFFSET          0xa88
#define GC_USB_DIEPINT12_PKTDRPSTS_LSB           0xb
#define GC_USB_DIEPINT12_PKTDRPSTS_MASK          0x800
#define GC_USB_DIEPINT12_PKTDRPSTS_SIZE          0x1
#define GC_USB_DIEPINT12_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DIEPINT12_PKTDRPSTS_OFFSET        0xa88
#define GC_USB_DIEPINT12_BBLEERR_LSB             0xc
#define GC_USB_DIEPINT12_BBLEERR_MASK            0x1000
#define GC_USB_DIEPINT12_BBLEERR_SIZE            0x1
#define GC_USB_DIEPINT12_BBLEERR_DEFAULT         0x0
#define GC_USB_DIEPINT12_BBLEERR_OFFSET          0xa88
#define GC_USB_DIEPINT12_NAKINTRPT_LSB           0xd
#define GC_USB_DIEPINT12_NAKINTRPT_MASK          0x2000
#define GC_USB_DIEPINT12_NAKINTRPT_SIZE          0x1
#define GC_USB_DIEPINT12_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT12_NAKINTRPT_OFFSET        0xa88
#define GC_USB_DIEPINT12_NYETINTRPT_LSB          0xe
#define GC_USB_DIEPINT12_NYETINTRPT_MASK         0x4000
#define GC_USB_DIEPINT12_NYETINTRPT_SIZE         0x1
#define GC_USB_DIEPINT12_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DIEPINT12_NYETINTRPT_OFFSET       0xa88
#define GC_USB_DIEPTSIZ12_XFERSIZE_LSB           0x0
#define GC_USB_DIEPTSIZ12_XFERSIZE_MASK          0x7ffff
#define GC_USB_DIEPTSIZ12_XFERSIZE_SIZE          0x13
#define GC_USB_DIEPTSIZ12_XFERSIZE_DEFAULT       0x0
#define GC_USB_DIEPTSIZ12_XFERSIZE_OFFSET        0xa90
#define GC_USB_DIEPTSIZ12_PKTCNT_LSB             0x13
#define GC_USB_DIEPTSIZ12_PKTCNT_MASK            0x1ff80000
#define GC_USB_DIEPTSIZ12_PKTCNT_SIZE            0xa
#define GC_USB_DIEPTSIZ12_PKTCNT_DEFAULT         0x0
#define GC_USB_DIEPTSIZ12_PKTCNT_OFFSET          0xa90
#define GC_USB_DIEPTSIZ12_MC_LSB                 0x1d
#define GC_USB_DIEPTSIZ12_MC_MASK                0x60000000
#define GC_USB_DIEPTSIZ12_MC_SIZE                0x2
#define GC_USB_DIEPTSIZ12_MC_DEFAULT             0x0
#define GC_USB_DIEPTSIZ12_MC_OFFSET              0xa90
#define GC_USB_DIEPDMA12_DMAADDR_LSB             0x0
#define GC_USB_DIEPDMA12_DMAADDR_MASK            0xffffffff
#define GC_USB_DIEPDMA12_DMAADDR_SIZE            0x20
#define GC_USB_DIEPDMA12_DMAADDR_DEFAULT         0x0
#define GC_USB_DIEPDMA12_DMAADDR_OFFSET          0xa94
#define GC_USB_DTXFSTS12_INEPTXFSPCAVAIL_LSB     0x0
#define GC_USB_DTXFSTS12_INEPTXFSPCAVAIL_MASK    0xffff
#define GC_USB_DTXFSTS12_INEPTXFSPCAVAIL_SIZE    0x10
#define GC_USB_DTXFSTS12_INEPTXFSPCAVAIL_DEFAULT 0x0
#define GC_USB_DTXFSTS12_INEPTXFSPCAVAIL_OFFSET  0xa98
#define GC_USB_DIEPDMAB12_DMABUFFERADDR_LSB      0x0
#define GC_USB_DIEPDMAB12_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DIEPDMAB12_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DIEPDMAB12_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DIEPDMAB12_DMABUFFERADDR_OFFSET   0xa9c
#define GC_USB_DIEPCTL13_MPS_LSB                 0x0
#define GC_USB_DIEPCTL13_MPS_MASK                0x7ff
#define GC_USB_DIEPCTL13_MPS_SIZE                0xb
#define GC_USB_DIEPCTL13_MPS_DEFAULT             0x0
#define GC_USB_DIEPCTL13_MPS_OFFSET              0xaa0
#define GC_USB_DIEPCTL13_USBACTEP_LSB            0xf
#define GC_USB_DIEPCTL13_USBACTEP_MASK           0x8000
#define GC_USB_DIEPCTL13_USBACTEP_SIZE           0x1
#define GC_USB_DIEPCTL13_USBACTEP_DEFAULT        0x0
#define GC_USB_DIEPCTL13_USBACTEP_OFFSET         0xaa0
#define GC_USB_DIEPCTL13_DPID_LSB                0x10
#define GC_USB_DIEPCTL13_DPID_MASK               0x10000
#define GC_USB_DIEPCTL13_DPID_SIZE               0x1
#define GC_USB_DIEPCTL13_DPID_DEFAULT            0x0
#define GC_USB_DIEPCTL13_DPID_OFFSET             0xaa0
#define GC_USB_DIEPCTL13_NAKSTS_LSB              0x11
#define GC_USB_DIEPCTL13_NAKSTS_MASK             0x20000
#define GC_USB_DIEPCTL13_NAKSTS_SIZE             0x1
#define GC_USB_DIEPCTL13_NAKSTS_DEFAULT          0x0
#define GC_USB_DIEPCTL13_NAKSTS_OFFSET           0xaa0
#define GC_USB_DIEPCTL13_EPTYPE_LSB              0x12
#define GC_USB_DIEPCTL13_EPTYPE_MASK             0xc0000
#define GC_USB_DIEPCTL13_EPTYPE_SIZE             0x2
#define GC_USB_DIEPCTL13_EPTYPE_DEFAULT          0x0
#define GC_USB_DIEPCTL13_EPTYPE_OFFSET           0xaa0
#define GC_USB_DIEPCTL13_STALL_LSB               0x15
#define GC_USB_DIEPCTL13_STALL_MASK              0x200000
#define GC_USB_DIEPCTL13_STALL_SIZE              0x1
#define GC_USB_DIEPCTL13_STALL_DEFAULT           0x0
#define GC_USB_DIEPCTL13_STALL_OFFSET            0xaa0
#define GC_USB_DIEPCTL13_TXFNUM_LSB              0x16
#define GC_USB_DIEPCTL13_TXFNUM_MASK             0x3c00000
#define GC_USB_DIEPCTL13_TXFNUM_SIZE             0x4
#define GC_USB_DIEPCTL13_TXFNUM_DEFAULT          0x0
#define GC_USB_DIEPCTL13_TXFNUM_OFFSET           0xaa0
#define GC_USB_DIEPCTL13_CNAK_LSB                0x1a
#define GC_USB_DIEPCTL13_CNAK_MASK               0x4000000
#define GC_USB_DIEPCTL13_CNAK_SIZE               0x1
#define GC_USB_DIEPCTL13_CNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL13_CNAK_OFFSET             0xaa0
#define GC_USB_DIEPCTL13_SNAK_LSB                0x1b
#define GC_USB_DIEPCTL13_SNAK_MASK               0x8000000
#define GC_USB_DIEPCTL13_SNAK_SIZE               0x1
#define GC_USB_DIEPCTL13_SNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL13_SNAK_OFFSET             0xaa0
#define GC_USB_DIEPCTL13_SETD0PID_LSB            0x1c
#define GC_USB_DIEPCTL13_SETD0PID_MASK           0x10000000
#define GC_USB_DIEPCTL13_SETD0PID_SIZE           0x1
#define GC_USB_DIEPCTL13_SETD0PID_DEFAULT        0x0
#define GC_USB_DIEPCTL13_SETD0PID_OFFSET         0xaa0
#define GC_USB_DIEPCTL13_SETD1PID_LSB            0x1d
#define GC_USB_DIEPCTL13_SETD1PID_MASK           0x20000000
#define GC_USB_DIEPCTL13_SETD1PID_SIZE           0x1
#define GC_USB_DIEPCTL13_SETD1PID_DEFAULT        0x0
#define GC_USB_DIEPCTL13_SETD1PID_OFFSET         0xaa0
#define GC_USB_DIEPCTL13_EPDIS_LSB               0x1e
#define GC_USB_DIEPCTL13_EPDIS_MASK              0x40000000
#define GC_USB_DIEPCTL13_EPDIS_SIZE              0x1
#define GC_USB_DIEPCTL13_EPDIS_DEFAULT           0x0
#define GC_USB_DIEPCTL13_EPDIS_OFFSET            0xaa0
#define GC_USB_DIEPCTL13_EPENA_LSB               0x1f
#define GC_USB_DIEPCTL13_EPENA_MASK              0x80000000
#define GC_USB_DIEPCTL13_EPENA_SIZE              0x1
#define GC_USB_DIEPCTL13_EPENA_DEFAULT           0x0
#define GC_USB_DIEPCTL13_EPENA_OFFSET            0xaa0
#define GC_USB_DIEPINT13_XFERCOMPL_LSB           0x0
#define GC_USB_DIEPINT13_XFERCOMPL_MASK          0x1
#define GC_USB_DIEPINT13_XFERCOMPL_SIZE          0x1
#define GC_USB_DIEPINT13_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DIEPINT13_XFERCOMPL_OFFSET        0xaa8
#define GC_USB_DIEPINT13_EPDISBLD_LSB            0x1
#define GC_USB_DIEPINT13_EPDISBLD_MASK           0x2
#define GC_USB_DIEPINT13_EPDISBLD_SIZE           0x1
#define GC_USB_DIEPINT13_EPDISBLD_DEFAULT        0x0
#define GC_USB_DIEPINT13_EPDISBLD_OFFSET         0xaa8
#define GC_USB_DIEPINT13_AHBERR_LSB              0x2
#define GC_USB_DIEPINT13_AHBERR_MASK             0x4
#define GC_USB_DIEPINT13_AHBERR_SIZE             0x1
#define GC_USB_DIEPINT13_AHBERR_DEFAULT          0x0
#define GC_USB_DIEPINT13_AHBERR_OFFSET           0xaa8
#define GC_USB_DIEPINT13_TIMEOUT_LSB             0x3
#define GC_USB_DIEPINT13_TIMEOUT_MASK            0x8
#define GC_USB_DIEPINT13_TIMEOUT_SIZE            0x1
#define GC_USB_DIEPINT13_TIMEOUT_DEFAULT         0x0
#define GC_USB_DIEPINT13_TIMEOUT_OFFSET          0xaa8
#define GC_USB_DIEPINT13_INTKNTXFEMP_LSB         0x4
#define GC_USB_DIEPINT13_INTKNTXFEMP_MASK        0x10
#define GC_USB_DIEPINT13_INTKNTXFEMP_SIZE        0x1
#define GC_USB_DIEPINT13_INTKNTXFEMP_DEFAULT     0x0
#define GC_USB_DIEPINT13_INTKNTXFEMP_OFFSET      0xaa8
#define GC_USB_DIEPINT13_INTKNEPMIS_LSB          0x5
#define GC_USB_DIEPINT13_INTKNEPMIS_MASK         0x20
#define GC_USB_DIEPINT13_INTKNEPMIS_SIZE         0x1
#define GC_USB_DIEPINT13_INTKNEPMIS_DEFAULT      0x0
#define GC_USB_DIEPINT13_INTKNEPMIS_OFFSET       0xaa8
#define GC_USB_DIEPINT13_INEPNAKEFF_LSB          0x6
#define GC_USB_DIEPINT13_INEPNAKEFF_MASK         0x40
#define GC_USB_DIEPINT13_INEPNAKEFF_SIZE         0x1
#define GC_USB_DIEPINT13_INEPNAKEFF_DEFAULT      0x0
#define GC_USB_DIEPINT13_INEPNAKEFF_OFFSET       0xaa8
#define GC_USB_DIEPINT13_TXFEMP_LSB              0x7
#define GC_USB_DIEPINT13_TXFEMP_MASK             0x80
#define GC_USB_DIEPINT13_TXFEMP_SIZE             0x1
#define GC_USB_DIEPINT13_TXFEMP_DEFAULT          0x0
#define GC_USB_DIEPINT13_TXFEMP_OFFSET           0xaa8
#define GC_USB_DIEPINT13_TXFIFOUNDRN_LSB         0x8
#define GC_USB_DIEPINT13_TXFIFOUNDRN_MASK        0x100
#define GC_USB_DIEPINT13_TXFIFOUNDRN_SIZE        0x1
#define GC_USB_DIEPINT13_TXFIFOUNDRN_DEFAULT     0x0
#define GC_USB_DIEPINT13_TXFIFOUNDRN_OFFSET      0xaa8
#define GC_USB_DIEPINT13_BNAINTR_LSB             0x9
#define GC_USB_DIEPINT13_BNAINTR_MASK            0x200
#define GC_USB_DIEPINT13_BNAINTR_SIZE            0x1
#define GC_USB_DIEPINT13_BNAINTR_DEFAULT         0x0
#define GC_USB_DIEPINT13_BNAINTR_OFFSET          0xaa8
#define GC_USB_DIEPINT13_PKTDRPSTS_LSB           0xb
#define GC_USB_DIEPINT13_PKTDRPSTS_MASK          0x800
#define GC_USB_DIEPINT13_PKTDRPSTS_SIZE          0x1
#define GC_USB_DIEPINT13_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DIEPINT13_PKTDRPSTS_OFFSET        0xaa8
#define GC_USB_DIEPINT13_BBLEERR_LSB             0xc
#define GC_USB_DIEPINT13_BBLEERR_MASK            0x1000
#define GC_USB_DIEPINT13_BBLEERR_SIZE            0x1
#define GC_USB_DIEPINT13_BBLEERR_DEFAULT         0x0
#define GC_USB_DIEPINT13_BBLEERR_OFFSET          0xaa8
#define GC_USB_DIEPINT13_NAKINTRPT_LSB           0xd
#define GC_USB_DIEPINT13_NAKINTRPT_MASK          0x2000
#define GC_USB_DIEPINT13_NAKINTRPT_SIZE          0x1
#define GC_USB_DIEPINT13_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT13_NAKINTRPT_OFFSET        0xaa8
#define GC_USB_DIEPINT13_NYETINTRPT_LSB          0xe
#define GC_USB_DIEPINT13_NYETINTRPT_MASK         0x4000
#define GC_USB_DIEPINT13_NYETINTRPT_SIZE         0x1
#define GC_USB_DIEPINT13_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DIEPINT13_NYETINTRPT_OFFSET       0xaa8
#define GC_USB_DIEPTSIZ13_XFERSIZE_LSB           0x0
#define GC_USB_DIEPTSIZ13_XFERSIZE_MASK          0x7ffff
#define GC_USB_DIEPTSIZ13_XFERSIZE_SIZE          0x13
#define GC_USB_DIEPTSIZ13_XFERSIZE_DEFAULT       0x0
#define GC_USB_DIEPTSIZ13_XFERSIZE_OFFSET        0xab0
#define GC_USB_DIEPTSIZ13_PKTCNT_LSB             0x13
#define GC_USB_DIEPTSIZ13_PKTCNT_MASK            0x1ff80000
#define GC_USB_DIEPTSIZ13_PKTCNT_SIZE            0xa
#define GC_USB_DIEPTSIZ13_PKTCNT_DEFAULT         0x0
#define GC_USB_DIEPTSIZ13_PKTCNT_OFFSET          0xab0
#define GC_USB_DIEPTSIZ13_MC_LSB                 0x1d
#define GC_USB_DIEPTSIZ13_MC_MASK                0x60000000
#define GC_USB_DIEPTSIZ13_MC_SIZE                0x2
#define GC_USB_DIEPTSIZ13_MC_DEFAULT             0x0
#define GC_USB_DIEPTSIZ13_MC_OFFSET              0xab0
#define GC_USB_DIEPDMA13_DMAADDR_LSB             0x0
#define GC_USB_DIEPDMA13_DMAADDR_MASK            0xffffffff
#define GC_USB_DIEPDMA13_DMAADDR_SIZE            0x20
#define GC_USB_DIEPDMA13_DMAADDR_DEFAULT         0x0
#define GC_USB_DIEPDMA13_DMAADDR_OFFSET          0xab4
#define GC_USB_DTXFSTS13_INEPTXFSPCAVAIL_LSB     0x0
#define GC_USB_DTXFSTS13_INEPTXFSPCAVAIL_MASK    0xffff
#define GC_USB_DTXFSTS13_INEPTXFSPCAVAIL_SIZE    0x10
#define GC_USB_DTXFSTS13_INEPTXFSPCAVAIL_DEFAULT 0x0
#define GC_USB_DTXFSTS13_INEPTXFSPCAVAIL_OFFSET  0xab8
#define GC_USB_DIEPDMAB13_DMABUFFERADDR_LSB      0x0
#define GC_USB_DIEPDMAB13_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DIEPDMAB13_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DIEPDMAB13_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DIEPDMAB13_DMABUFFERADDR_OFFSET   0xabc
#define GC_USB_DIEPCTL14_MPS_LSB                 0x0
#define GC_USB_DIEPCTL14_MPS_MASK                0x7ff
#define GC_USB_DIEPCTL14_MPS_SIZE                0xb
#define GC_USB_DIEPCTL14_MPS_DEFAULT             0x0
#define GC_USB_DIEPCTL14_MPS_OFFSET              0xac0
#define GC_USB_DIEPCTL14_USBACTEP_LSB            0xf
#define GC_USB_DIEPCTL14_USBACTEP_MASK           0x8000
#define GC_USB_DIEPCTL14_USBACTEP_SIZE           0x1
#define GC_USB_DIEPCTL14_USBACTEP_DEFAULT        0x0
#define GC_USB_DIEPCTL14_USBACTEP_OFFSET         0xac0
#define GC_USB_DIEPCTL14_DPID_LSB                0x10
#define GC_USB_DIEPCTL14_DPID_MASK               0x10000
#define GC_USB_DIEPCTL14_DPID_SIZE               0x1
#define GC_USB_DIEPCTL14_DPID_DEFAULT            0x0
#define GC_USB_DIEPCTL14_DPID_OFFSET             0xac0
#define GC_USB_DIEPCTL14_NAKSTS_LSB              0x11
#define GC_USB_DIEPCTL14_NAKSTS_MASK             0x20000
#define GC_USB_DIEPCTL14_NAKSTS_SIZE             0x1
#define GC_USB_DIEPCTL14_NAKSTS_DEFAULT          0x0
#define GC_USB_DIEPCTL14_NAKSTS_OFFSET           0xac0
#define GC_USB_DIEPCTL14_EPTYPE_LSB              0x12
#define GC_USB_DIEPCTL14_EPTYPE_MASK             0xc0000
#define GC_USB_DIEPCTL14_EPTYPE_SIZE             0x2
#define GC_USB_DIEPCTL14_EPTYPE_DEFAULT          0x0
#define GC_USB_DIEPCTL14_EPTYPE_OFFSET           0xac0
#define GC_USB_DIEPCTL14_STALL_LSB               0x15
#define GC_USB_DIEPCTL14_STALL_MASK              0x200000
#define GC_USB_DIEPCTL14_STALL_SIZE              0x1
#define GC_USB_DIEPCTL14_STALL_DEFAULT           0x0
#define GC_USB_DIEPCTL14_STALL_OFFSET            0xac0
#define GC_USB_DIEPCTL14_TXFNUM_LSB              0x16
#define GC_USB_DIEPCTL14_TXFNUM_MASK             0x3c00000
#define GC_USB_DIEPCTL14_TXFNUM_SIZE             0x4
#define GC_USB_DIEPCTL14_TXFNUM_DEFAULT          0x0
#define GC_USB_DIEPCTL14_TXFNUM_OFFSET           0xac0
#define GC_USB_DIEPCTL14_CNAK_LSB                0x1a
#define GC_USB_DIEPCTL14_CNAK_MASK               0x4000000
#define GC_USB_DIEPCTL14_CNAK_SIZE               0x1
#define GC_USB_DIEPCTL14_CNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL14_CNAK_OFFSET             0xac0
#define GC_USB_DIEPCTL14_SNAK_LSB                0x1b
#define GC_USB_DIEPCTL14_SNAK_MASK               0x8000000
#define GC_USB_DIEPCTL14_SNAK_SIZE               0x1
#define GC_USB_DIEPCTL14_SNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL14_SNAK_OFFSET             0xac0
#define GC_USB_DIEPCTL14_SETD0PID_LSB            0x1c
#define GC_USB_DIEPCTL14_SETD0PID_MASK           0x10000000
#define GC_USB_DIEPCTL14_SETD0PID_SIZE           0x1
#define GC_USB_DIEPCTL14_SETD0PID_DEFAULT        0x0
#define GC_USB_DIEPCTL14_SETD0PID_OFFSET         0xac0
#define GC_USB_DIEPCTL14_SETD1PID_LSB            0x1d
#define GC_USB_DIEPCTL14_SETD1PID_MASK           0x20000000
#define GC_USB_DIEPCTL14_SETD1PID_SIZE           0x1
#define GC_USB_DIEPCTL14_SETD1PID_DEFAULT        0x0
#define GC_USB_DIEPCTL14_SETD1PID_OFFSET         0xac0
#define GC_USB_DIEPCTL14_EPDIS_LSB               0x1e
#define GC_USB_DIEPCTL14_EPDIS_MASK              0x40000000
#define GC_USB_DIEPCTL14_EPDIS_SIZE              0x1
#define GC_USB_DIEPCTL14_EPDIS_DEFAULT           0x0
#define GC_USB_DIEPCTL14_EPDIS_OFFSET            0xac0
#define GC_USB_DIEPCTL14_EPENA_LSB               0x1f
#define GC_USB_DIEPCTL14_EPENA_MASK              0x80000000
#define GC_USB_DIEPCTL14_EPENA_SIZE              0x1
#define GC_USB_DIEPCTL14_EPENA_DEFAULT           0x0
#define GC_USB_DIEPCTL14_EPENA_OFFSET            0xac0
#define GC_USB_DIEPINT14_XFERCOMPL_LSB           0x0
#define GC_USB_DIEPINT14_XFERCOMPL_MASK          0x1
#define GC_USB_DIEPINT14_XFERCOMPL_SIZE          0x1
#define GC_USB_DIEPINT14_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DIEPINT14_XFERCOMPL_OFFSET        0xac8
#define GC_USB_DIEPINT14_EPDISBLD_LSB            0x1
#define GC_USB_DIEPINT14_EPDISBLD_MASK           0x2
#define GC_USB_DIEPINT14_EPDISBLD_SIZE           0x1
#define GC_USB_DIEPINT14_EPDISBLD_DEFAULT        0x0
#define GC_USB_DIEPINT14_EPDISBLD_OFFSET         0xac8
#define GC_USB_DIEPINT14_AHBERR_LSB              0x2
#define GC_USB_DIEPINT14_AHBERR_MASK             0x4
#define GC_USB_DIEPINT14_AHBERR_SIZE             0x1
#define GC_USB_DIEPINT14_AHBERR_DEFAULT          0x0
#define GC_USB_DIEPINT14_AHBERR_OFFSET           0xac8
#define GC_USB_DIEPINT14_TIMEOUT_LSB             0x3
#define GC_USB_DIEPINT14_TIMEOUT_MASK            0x8
#define GC_USB_DIEPINT14_TIMEOUT_SIZE            0x1
#define GC_USB_DIEPINT14_TIMEOUT_DEFAULT         0x0
#define GC_USB_DIEPINT14_TIMEOUT_OFFSET          0xac8
#define GC_USB_DIEPINT14_INTKNTXFEMP_LSB         0x4
#define GC_USB_DIEPINT14_INTKNTXFEMP_MASK        0x10
#define GC_USB_DIEPINT14_INTKNTXFEMP_SIZE        0x1
#define GC_USB_DIEPINT14_INTKNTXFEMP_DEFAULT     0x0
#define GC_USB_DIEPINT14_INTKNTXFEMP_OFFSET      0xac8
#define GC_USB_DIEPINT14_INTKNEPMIS_LSB          0x5
#define GC_USB_DIEPINT14_INTKNEPMIS_MASK         0x20
#define GC_USB_DIEPINT14_INTKNEPMIS_SIZE         0x1
#define GC_USB_DIEPINT14_INTKNEPMIS_DEFAULT      0x0
#define GC_USB_DIEPINT14_INTKNEPMIS_OFFSET       0xac8
#define GC_USB_DIEPINT14_INEPNAKEFF_LSB          0x6
#define GC_USB_DIEPINT14_INEPNAKEFF_MASK         0x40
#define GC_USB_DIEPINT14_INEPNAKEFF_SIZE         0x1
#define GC_USB_DIEPINT14_INEPNAKEFF_DEFAULT      0x0
#define GC_USB_DIEPINT14_INEPNAKEFF_OFFSET       0xac8
#define GC_USB_DIEPINT14_TXFEMP_LSB              0x7
#define GC_USB_DIEPINT14_TXFEMP_MASK             0x80
#define GC_USB_DIEPINT14_TXFEMP_SIZE             0x1
#define GC_USB_DIEPINT14_TXFEMP_DEFAULT          0x0
#define GC_USB_DIEPINT14_TXFEMP_OFFSET           0xac8
#define GC_USB_DIEPINT14_TXFIFOUNDRN_LSB         0x8
#define GC_USB_DIEPINT14_TXFIFOUNDRN_MASK        0x100
#define GC_USB_DIEPINT14_TXFIFOUNDRN_SIZE        0x1
#define GC_USB_DIEPINT14_TXFIFOUNDRN_DEFAULT     0x0
#define GC_USB_DIEPINT14_TXFIFOUNDRN_OFFSET      0xac8
#define GC_USB_DIEPINT14_BNAINTR_LSB             0x9
#define GC_USB_DIEPINT14_BNAINTR_MASK            0x200
#define GC_USB_DIEPINT14_BNAINTR_SIZE            0x1
#define GC_USB_DIEPINT14_BNAINTR_DEFAULT         0x0
#define GC_USB_DIEPINT14_BNAINTR_OFFSET          0xac8
#define GC_USB_DIEPINT14_PKTDRPSTS_LSB           0xb
#define GC_USB_DIEPINT14_PKTDRPSTS_MASK          0x800
#define GC_USB_DIEPINT14_PKTDRPSTS_SIZE          0x1
#define GC_USB_DIEPINT14_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DIEPINT14_PKTDRPSTS_OFFSET        0xac8
#define GC_USB_DIEPINT14_BBLEERR_LSB             0xc
#define GC_USB_DIEPINT14_BBLEERR_MASK            0x1000
#define GC_USB_DIEPINT14_BBLEERR_SIZE            0x1
#define GC_USB_DIEPINT14_BBLEERR_DEFAULT         0x0
#define GC_USB_DIEPINT14_BBLEERR_OFFSET          0xac8
#define GC_USB_DIEPINT14_NAKINTRPT_LSB           0xd
#define GC_USB_DIEPINT14_NAKINTRPT_MASK          0x2000
#define GC_USB_DIEPINT14_NAKINTRPT_SIZE          0x1
#define GC_USB_DIEPINT14_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT14_NAKINTRPT_OFFSET        0xac8
#define GC_USB_DIEPINT14_NYETINTRPT_LSB          0xe
#define GC_USB_DIEPINT14_NYETINTRPT_MASK         0x4000
#define GC_USB_DIEPINT14_NYETINTRPT_SIZE         0x1
#define GC_USB_DIEPINT14_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DIEPINT14_NYETINTRPT_OFFSET       0xac8
#define GC_USB_DIEPTSIZ14_XFERSIZE_LSB           0x0
#define GC_USB_DIEPTSIZ14_XFERSIZE_MASK          0x7ffff
#define GC_USB_DIEPTSIZ14_XFERSIZE_SIZE          0x13
#define GC_USB_DIEPTSIZ14_XFERSIZE_DEFAULT       0x0
#define GC_USB_DIEPTSIZ14_XFERSIZE_OFFSET        0xad0
#define GC_USB_DIEPTSIZ14_PKTCNT_LSB             0x13
#define GC_USB_DIEPTSIZ14_PKTCNT_MASK            0x1ff80000
#define GC_USB_DIEPTSIZ14_PKTCNT_SIZE            0xa
#define GC_USB_DIEPTSIZ14_PKTCNT_DEFAULT         0x0
#define GC_USB_DIEPTSIZ14_PKTCNT_OFFSET          0xad0
#define GC_USB_DIEPTSIZ14_MC_LSB                 0x1d
#define GC_USB_DIEPTSIZ14_MC_MASK                0x60000000
#define GC_USB_DIEPTSIZ14_MC_SIZE                0x2
#define GC_USB_DIEPTSIZ14_MC_DEFAULT             0x0
#define GC_USB_DIEPTSIZ14_MC_OFFSET              0xad0
#define GC_USB_DIEPDMA14_DMAADDR_LSB             0x0
#define GC_USB_DIEPDMA14_DMAADDR_MASK            0xffffffff
#define GC_USB_DIEPDMA14_DMAADDR_SIZE            0x20
#define GC_USB_DIEPDMA14_DMAADDR_DEFAULT         0x0
#define GC_USB_DIEPDMA14_DMAADDR_OFFSET          0xad4
#define GC_USB_DTXFSTS14_INEPTXFSPCAVAIL_LSB     0x0
#define GC_USB_DTXFSTS14_INEPTXFSPCAVAIL_MASK    0xffff
#define GC_USB_DTXFSTS14_INEPTXFSPCAVAIL_SIZE    0x10
#define GC_USB_DTXFSTS14_INEPTXFSPCAVAIL_DEFAULT 0x0
#define GC_USB_DTXFSTS14_INEPTXFSPCAVAIL_OFFSET  0xad8
#define GC_USB_DIEPDMAB14_DMABUFFERADDR_LSB      0x0
#define GC_USB_DIEPDMAB14_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DIEPDMAB14_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DIEPDMAB14_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DIEPDMAB14_DMABUFFERADDR_OFFSET   0xadc
#define GC_USB_DIEPCTL15_MPS_LSB                 0x0
#define GC_USB_DIEPCTL15_MPS_MASK                0x7ff
#define GC_USB_DIEPCTL15_MPS_SIZE                0xb
#define GC_USB_DIEPCTL15_MPS_DEFAULT             0x0
#define GC_USB_DIEPCTL15_MPS_OFFSET              0xae0
#define GC_USB_DIEPCTL15_USBACTEP_LSB            0xf
#define GC_USB_DIEPCTL15_USBACTEP_MASK           0x8000
#define GC_USB_DIEPCTL15_USBACTEP_SIZE           0x1
#define GC_USB_DIEPCTL15_USBACTEP_DEFAULT        0x0
#define GC_USB_DIEPCTL15_USBACTEP_OFFSET         0xae0
#define GC_USB_DIEPCTL15_DPID_LSB                0x10
#define GC_USB_DIEPCTL15_DPID_MASK               0x10000
#define GC_USB_DIEPCTL15_DPID_SIZE               0x1
#define GC_USB_DIEPCTL15_DPID_DEFAULT            0x0
#define GC_USB_DIEPCTL15_DPID_OFFSET             0xae0
#define GC_USB_DIEPCTL15_NAKSTS_LSB              0x11
#define GC_USB_DIEPCTL15_NAKSTS_MASK             0x20000
#define GC_USB_DIEPCTL15_NAKSTS_SIZE             0x1
#define GC_USB_DIEPCTL15_NAKSTS_DEFAULT          0x0
#define GC_USB_DIEPCTL15_NAKSTS_OFFSET           0xae0
#define GC_USB_DIEPCTL15_EPTYPE_LSB              0x12
#define GC_USB_DIEPCTL15_EPTYPE_MASK             0xc0000
#define GC_USB_DIEPCTL15_EPTYPE_SIZE             0x2
#define GC_USB_DIEPCTL15_EPTYPE_DEFAULT          0x0
#define GC_USB_DIEPCTL15_EPTYPE_OFFSET           0xae0
#define GC_USB_DIEPCTL15_STALL_LSB               0x15
#define GC_USB_DIEPCTL15_STALL_MASK              0x200000
#define GC_USB_DIEPCTL15_STALL_SIZE              0x1
#define GC_USB_DIEPCTL15_STALL_DEFAULT           0x0
#define GC_USB_DIEPCTL15_STALL_OFFSET            0xae0
#define GC_USB_DIEPCTL15_TXFNUM_LSB              0x16
#define GC_USB_DIEPCTL15_TXFNUM_MASK             0x3c00000
#define GC_USB_DIEPCTL15_TXFNUM_SIZE             0x4
#define GC_USB_DIEPCTL15_TXFNUM_DEFAULT          0x0
#define GC_USB_DIEPCTL15_TXFNUM_OFFSET           0xae0
#define GC_USB_DIEPCTL15_CNAK_LSB                0x1a
#define GC_USB_DIEPCTL15_CNAK_MASK               0x4000000
#define GC_USB_DIEPCTL15_CNAK_SIZE               0x1
#define GC_USB_DIEPCTL15_CNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL15_CNAK_OFFSET             0xae0
#define GC_USB_DIEPCTL15_SNAK_LSB                0x1b
#define GC_USB_DIEPCTL15_SNAK_MASK               0x8000000
#define GC_USB_DIEPCTL15_SNAK_SIZE               0x1
#define GC_USB_DIEPCTL15_SNAK_DEFAULT            0x0
#define GC_USB_DIEPCTL15_SNAK_OFFSET             0xae0
#define GC_USB_DIEPCTL15_SETD0PID_LSB            0x1c
#define GC_USB_DIEPCTL15_SETD0PID_MASK           0x10000000
#define GC_USB_DIEPCTL15_SETD0PID_SIZE           0x1
#define GC_USB_DIEPCTL15_SETD0PID_DEFAULT        0x0
#define GC_USB_DIEPCTL15_SETD0PID_OFFSET         0xae0
#define GC_USB_DIEPCTL15_SETD1PID_LSB            0x1d
#define GC_USB_DIEPCTL15_SETD1PID_MASK           0x20000000
#define GC_USB_DIEPCTL15_SETD1PID_SIZE           0x1
#define GC_USB_DIEPCTL15_SETD1PID_DEFAULT        0x0
#define GC_USB_DIEPCTL15_SETD1PID_OFFSET         0xae0
#define GC_USB_DIEPCTL15_EPDIS_LSB               0x1e
#define GC_USB_DIEPCTL15_EPDIS_MASK              0x40000000
#define GC_USB_DIEPCTL15_EPDIS_SIZE              0x1
#define GC_USB_DIEPCTL15_EPDIS_DEFAULT           0x0
#define GC_USB_DIEPCTL15_EPDIS_OFFSET            0xae0
#define GC_USB_DIEPCTL15_EPENA_LSB               0x1f
#define GC_USB_DIEPCTL15_EPENA_MASK              0x80000000
#define GC_USB_DIEPCTL15_EPENA_SIZE              0x1
#define GC_USB_DIEPCTL15_EPENA_DEFAULT           0x0
#define GC_USB_DIEPCTL15_EPENA_OFFSET            0xae0
#define GC_USB_DIEPINT15_XFERCOMPL_LSB           0x0
#define GC_USB_DIEPINT15_XFERCOMPL_MASK          0x1
#define GC_USB_DIEPINT15_XFERCOMPL_SIZE          0x1
#define GC_USB_DIEPINT15_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DIEPINT15_XFERCOMPL_OFFSET        0xae8
#define GC_USB_DIEPINT15_EPDISBLD_LSB            0x1
#define GC_USB_DIEPINT15_EPDISBLD_MASK           0x2
#define GC_USB_DIEPINT15_EPDISBLD_SIZE           0x1
#define GC_USB_DIEPINT15_EPDISBLD_DEFAULT        0x0
#define GC_USB_DIEPINT15_EPDISBLD_OFFSET         0xae8
#define GC_USB_DIEPINT15_AHBERR_LSB              0x2
#define GC_USB_DIEPINT15_AHBERR_MASK             0x4
#define GC_USB_DIEPINT15_AHBERR_SIZE             0x1
#define GC_USB_DIEPINT15_AHBERR_DEFAULT          0x0
#define GC_USB_DIEPINT15_AHBERR_OFFSET           0xae8
#define GC_USB_DIEPINT15_TIMEOUT_LSB             0x3
#define GC_USB_DIEPINT15_TIMEOUT_MASK            0x8
#define GC_USB_DIEPINT15_TIMEOUT_SIZE            0x1
#define GC_USB_DIEPINT15_TIMEOUT_DEFAULT         0x0
#define GC_USB_DIEPINT15_TIMEOUT_OFFSET          0xae8
#define GC_USB_DIEPINT15_INTKNTXFEMP_LSB         0x4
#define GC_USB_DIEPINT15_INTKNTXFEMP_MASK        0x10
#define GC_USB_DIEPINT15_INTKNTXFEMP_SIZE        0x1
#define GC_USB_DIEPINT15_INTKNTXFEMP_DEFAULT     0x0
#define GC_USB_DIEPINT15_INTKNTXFEMP_OFFSET      0xae8
#define GC_USB_DIEPINT15_INTKNEPMIS_LSB          0x5
#define GC_USB_DIEPINT15_INTKNEPMIS_MASK         0x20
#define GC_USB_DIEPINT15_INTKNEPMIS_SIZE         0x1
#define GC_USB_DIEPINT15_INTKNEPMIS_DEFAULT      0x0
#define GC_USB_DIEPINT15_INTKNEPMIS_OFFSET       0xae8
#define GC_USB_DIEPINT15_INEPNAKEFF_LSB          0x6
#define GC_USB_DIEPINT15_INEPNAKEFF_MASK         0x40
#define GC_USB_DIEPINT15_INEPNAKEFF_SIZE         0x1
#define GC_USB_DIEPINT15_INEPNAKEFF_DEFAULT      0x0
#define GC_USB_DIEPINT15_INEPNAKEFF_OFFSET       0xae8
#define GC_USB_DIEPINT15_TXFEMP_LSB              0x7
#define GC_USB_DIEPINT15_TXFEMP_MASK             0x80
#define GC_USB_DIEPINT15_TXFEMP_SIZE             0x1
#define GC_USB_DIEPINT15_TXFEMP_DEFAULT          0x0
#define GC_USB_DIEPINT15_TXFEMP_OFFSET           0xae8
#define GC_USB_DIEPINT15_TXFIFOUNDRN_LSB         0x8
#define GC_USB_DIEPINT15_TXFIFOUNDRN_MASK        0x100
#define GC_USB_DIEPINT15_TXFIFOUNDRN_SIZE        0x1
#define GC_USB_DIEPINT15_TXFIFOUNDRN_DEFAULT     0x0
#define GC_USB_DIEPINT15_TXFIFOUNDRN_OFFSET      0xae8
#define GC_USB_DIEPINT15_BNAINTR_LSB             0x9
#define GC_USB_DIEPINT15_BNAINTR_MASK            0x200
#define GC_USB_DIEPINT15_BNAINTR_SIZE            0x1
#define GC_USB_DIEPINT15_BNAINTR_DEFAULT         0x0
#define GC_USB_DIEPINT15_BNAINTR_OFFSET          0xae8
#define GC_USB_DIEPINT15_PKTDRPSTS_LSB           0xb
#define GC_USB_DIEPINT15_PKTDRPSTS_MASK          0x800
#define GC_USB_DIEPINT15_PKTDRPSTS_SIZE          0x1
#define GC_USB_DIEPINT15_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DIEPINT15_PKTDRPSTS_OFFSET        0xae8
#define GC_USB_DIEPINT15_BBLEERR_LSB             0xc
#define GC_USB_DIEPINT15_BBLEERR_MASK            0x1000
#define GC_USB_DIEPINT15_BBLEERR_SIZE            0x1
#define GC_USB_DIEPINT15_BBLEERR_DEFAULT         0x0
#define GC_USB_DIEPINT15_BBLEERR_OFFSET          0xae8
#define GC_USB_DIEPINT15_NAKINTRPT_LSB           0xd
#define GC_USB_DIEPINT15_NAKINTRPT_MASK          0x2000
#define GC_USB_DIEPINT15_NAKINTRPT_SIZE          0x1
#define GC_USB_DIEPINT15_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DIEPINT15_NAKINTRPT_OFFSET        0xae8
#define GC_USB_DIEPINT15_NYETINTRPT_LSB          0xe
#define GC_USB_DIEPINT15_NYETINTRPT_MASK         0x4000
#define GC_USB_DIEPINT15_NYETINTRPT_SIZE         0x1
#define GC_USB_DIEPINT15_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DIEPINT15_NYETINTRPT_OFFSET       0xae8
#define GC_USB_DIEPTSIZ15_XFERSIZE_LSB           0x0
#define GC_USB_DIEPTSIZ15_XFERSIZE_MASK          0x7ffff
#define GC_USB_DIEPTSIZ15_XFERSIZE_SIZE          0x13
#define GC_USB_DIEPTSIZ15_XFERSIZE_DEFAULT       0x0
#define GC_USB_DIEPTSIZ15_XFERSIZE_OFFSET        0xaf0
#define GC_USB_DIEPTSIZ15_PKTCNT_LSB             0x13
#define GC_USB_DIEPTSIZ15_PKTCNT_MASK            0x1ff80000
#define GC_USB_DIEPTSIZ15_PKTCNT_SIZE            0xa
#define GC_USB_DIEPTSIZ15_PKTCNT_DEFAULT         0x0
#define GC_USB_DIEPTSIZ15_PKTCNT_OFFSET          0xaf0
#define GC_USB_DIEPTSIZ15_MC_LSB                 0x1d
#define GC_USB_DIEPTSIZ15_MC_MASK                0x60000000
#define GC_USB_DIEPTSIZ15_MC_SIZE                0x2
#define GC_USB_DIEPTSIZ15_MC_DEFAULT             0x0
#define GC_USB_DIEPTSIZ15_MC_OFFSET              0xaf0
#define GC_USB_DIEPDMA15_DMAADDR_LSB             0x0
#define GC_USB_DIEPDMA15_DMAADDR_MASK            0xffffffff
#define GC_USB_DIEPDMA15_DMAADDR_SIZE            0x20
#define GC_USB_DIEPDMA15_DMAADDR_DEFAULT         0x0
#define GC_USB_DIEPDMA15_DMAADDR_OFFSET          0xaf4
#define GC_USB_DTXFSTS15_INEPTXFSPCAVAIL_LSB     0x0
#define GC_USB_DTXFSTS15_INEPTXFSPCAVAIL_MASK    0xffff
#define GC_USB_DTXFSTS15_INEPTXFSPCAVAIL_SIZE    0x10
#define GC_USB_DTXFSTS15_INEPTXFSPCAVAIL_DEFAULT 0x0
#define GC_USB_DTXFSTS15_INEPTXFSPCAVAIL_OFFSET  0xaf8
#define GC_USB_DIEPDMAB15_DMABUFFERADDR_LSB      0x0
#define GC_USB_DIEPDMAB15_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DIEPDMAB15_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DIEPDMAB15_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DIEPDMAB15_DMABUFFERADDR_OFFSET   0xafc
#define GC_USB_DOEPCTL0_MPS_LSB                  0x0
#define GC_USB_DOEPCTL0_MPS_MASK                 0x3
#define GC_USB_DOEPCTL0_MPS_SIZE                 0x2
#define GC_USB_DOEPCTL0_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL0_MPS_OFFSET               0xb00
#define GC_USB_DOEPCTL0_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL0_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL0_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL0_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL0_USBACTEP_OFFSET          0xb00
#define GC_USB_DOEPCTL0_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL0_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL0_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL0_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL0_NAKSTS_OFFSET            0xb00
#define GC_USB_DOEPCTL0_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL0_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL0_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL0_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL0_EPTYPE_OFFSET            0xb00
#define GC_USB_DOEPCTL0_SNP_LSB                  0x14
#define GC_USB_DOEPCTL0_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL0_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL0_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL0_SNP_OFFSET               0xb00
#define GC_USB_DOEPCTL0_STALL_LSB                0x15
#define GC_USB_DOEPCTL0_STALL_MASK               0x200000
#define GC_USB_DOEPCTL0_STALL_SIZE               0x1
#define GC_USB_DOEPCTL0_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL0_STALL_OFFSET             0xb00
#define GC_USB_DOEPCTL0_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL0_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL0_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL0_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL0_CNAK_OFFSET              0xb00
#define GC_USB_DOEPCTL0_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL0_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL0_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL0_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL0_SNAK_OFFSET              0xb00
#define GC_USB_DOEPCTL0_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL0_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL0_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL0_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL0_EPDIS_OFFSET             0xb00
#define GC_USB_DOEPCTL0_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL0_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL0_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL0_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL0_EPENA_OFFSET             0xb00
#define GC_USB_DOEPINT0_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT0_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT0_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT0_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT0_XFERCOMPL_OFFSET         0xb08
#define GC_USB_DOEPINT0_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT0_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT0_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT0_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT0_EPDISBLD_OFFSET          0xb08
#define GC_USB_DOEPINT0_AHBERR_LSB               0x2
#define GC_USB_DOEPINT0_AHBERR_MASK              0x4
#define GC_USB_DOEPINT0_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT0_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT0_AHBERR_OFFSET            0xb08
#define GC_USB_DOEPINT0_SETUP_LSB                0x3
#define GC_USB_DOEPINT0_SETUP_MASK               0x8
#define GC_USB_DOEPINT0_SETUP_SIZE               0x1
#define GC_USB_DOEPINT0_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT0_SETUP_OFFSET             0xb08
#define GC_USB_DOEPINT0_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT0_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT0_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT0_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT0_OUTTKNEPDIS_OFFSET       0xb08
#define GC_USB_DOEPINT0_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT0_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT0_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT0_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT0_STSPHSERCVD_OFFSET       0xb08
#define GC_USB_DOEPINT0_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT0_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT0_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT0_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT0_BACK2BACKSETUP_OFFSET    0xb08
#define GC_USB_DOEPINT0_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT0_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT0_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT0_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT0_OUTPKTERR_OFFSET         0xb08
#define GC_USB_DOEPINT0_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT0_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT0_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT0_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT0_BNAINTR_OFFSET           0xb08
#define GC_USB_DOEPINT0_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT0_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT0_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT0_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT0_PKTDRPSTS_OFFSET         0xb08
#define GC_USB_DOEPINT0_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT0_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT0_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT0_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT0_BBLEERR_OFFSET           0xb08
#define GC_USB_DOEPINT0_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT0_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT0_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT0_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT0_NAKINTRPT_OFFSET         0xb08
#define GC_USB_DOEPINT0_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT0_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT0_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT0_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT0_NYETINTRPT_OFFSET        0xb08
#define GC_USB_DOEPINT0_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT0_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT0_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT0_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT0_STUPPKTRCVD_OFFSET       0xb08
#define GC_USB_DOEPTSIZ0_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ0_XFERSIZE_MASK           0x7f
#define GC_USB_DOEPTSIZ0_XFERSIZE_SIZE           0x7
#define GC_USB_DOEPTSIZ0_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ0_XFERSIZE_OFFSET         0xb10
#define GC_USB_DOEPTSIZ0_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ0_PKTCNT_MASK             0x80000
#define GC_USB_DOEPTSIZ0_PKTCNT_SIZE             0x1
#define GC_USB_DOEPTSIZ0_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ0_PKTCNT_OFFSET           0xb10
#define GC_USB_DOEPTSIZ0_SUPCNT_LSB              0x1d
#define GC_USB_DOEPTSIZ0_SUPCNT_MASK             0x60000000
#define GC_USB_DOEPTSIZ0_SUPCNT_SIZE             0x2
#define GC_USB_DOEPTSIZ0_SUPCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ0_SUPCNT_OFFSET           0xb10
#define GC_USB_DOEPDMA0_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA0_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA0_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA0_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA0_DMAADDR_OFFSET           0xb14
#define GC_USB_DOEPDMAB0_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB0_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB0_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB0_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB0_DMABUFFERADDR_OFFSET    0xb1c
#define GC_USB_DOEPCTL1_MPS_LSB                  0x0
#define GC_USB_DOEPCTL1_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL1_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL1_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL1_MPS_OFFSET               0xb20
#define GC_USB_DOEPCTL1_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL1_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL1_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL1_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL1_USBACTEP_OFFSET          0xb20
#define GC_USB_DOEPCTL1_DPID_LSB                 0x10
#define GC_USB_DOEPCTL1_DPID_MASK                0x10000
#define GC_USB_DOEPCTL1_DPID_SIZE                0x1
#define GC_USB_DOEPCTL1_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL1_DPID_OFFSET              0xb20
#define GC_USB_DOEPCTL1_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL1_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL1_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL1_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL1_NAKSTS_OFFSET            0xb20
#define GC_USB_DOEPCTL1_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL1_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL1_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL1_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL1_EPTYPE_OFFSET            0xb20
#define GC_USB_DOEPCTL1_SNP_LSB                  0x14
#define GC_USB_DOEPCTL1_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL1_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL1_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL1_SNP_OFFSET               0xb20
#define GC_USB_DOEPCTL1_STALL_LSB                0x15
#define GC_USB_DOEPCTL1_STALL_MASK               0x200000
#define GC_USB_DOEPCTL1_STALL_SIZE               0x1
#define GC_USB_DOEPCTL1_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL1_STALL_OFFSET             0xb20
#define GC_USB_DOEPCTL1_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL1_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL1_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL1_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL1_CNAK_OFFSET              0xb20
#define GC_USB_DOEPCTL1_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL1_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL1_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL1_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL1_SNAK_OFFSET              0xb20
#define GC_USB_DOEPCTL1_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL1_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL1_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL1_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL1_SETD0PID_OFFSET          0xb20
#define GC_USB_DOEPCTL1_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL1_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL1_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL1_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL1_SETD1PID_OFFSET          0xb20
#define GC_USB_DOEPCTL1_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL1_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL1_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL1_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL1_EPDIS_OFFSET             0xb20
#define GC_USB_DOEPCTL1_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL1_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL1_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL1_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL1_EPENA_OFFSET             0xb20
#define GC_USB_DOEPINT1_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT1_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT1_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT1_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT1_XFERCOMPL_OFFSET         0xb28
#define GC_USB_DOEPINT1_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT1_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT1_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT1_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT1_EPDISBLD_OFFSET          0xb28
#define GC_USB_DOEPINT1_AHBERR_LSB               0x2
#define GC_USB_DOEPINT1_AHBERR_MASK              0x4
#define GC_USB_DOEPINT1_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT1_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT1_AHBERR_OFFSET            0xb28
#define GC_USB_DOEPINT1_SETUP_LSB                0x3
#define GC_USB_DOEPINT1_SETUP_MASK               0x8
#define GC_USB_DOEPINT1_SETUP_SIZE               0x1
#define GC_USB_DOEPINT1_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT1_SETUP_OFFSET             0xb28
#define GC_USB_DOEPINT1_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT1_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT1_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT1_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT1_OUTTKNEPDIS_OFFSET       0xb28
#define GC_USB_DOEPINT1_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT1_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT1_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT1_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT1_STSPHSERCVD_OFFSET       0xb28
#define GC_USB_DOEPINT1_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT1_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT1_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT1_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT1_BACK2BACKSETUP_OFFSET    0xb28
#define GC_USB_DOEPINT1_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT1_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT1_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT1_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT1_OUTPKTERR_OFFSET         0xb28
#define GC_USB_DOEPINT1_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT1_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT1_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT1_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT1_BNAINTR_OFFSET           0xb28
#define GC_USB_DOEPINT1_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT1_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT1_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT1_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT1_PKTDRPSTS_OFFSET         0xb28
#define GC_USB_DOEPINT1_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT1_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT1_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT1_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT1_BBLEERR_OFFSET           0xb28
#define GC_USB_DOEPINT1_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT1_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT1_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT1_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT1_NAKINTRPT_OFFSET         0xb28
#define GC_USB_DOEPINT1_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT1_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT1_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT1_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT1_NYETINTRPT_OFFSET        0xb28
#define GC_USB_DOEPINT1_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT1_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT1_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT1_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT1_STUPPKTRCVD_OFFSET       0xb28
#define GC_USB_DOEPTSIZ1_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ1_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ1_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ1_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ1_XFERSIZE_OFFSET         0xb30
#define GC_USB_DOEPTSIZ1_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ1_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ1_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ1_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ1_PKTCNT_OFFSET           0xb30
#define GC_USB_DOEPTSIZ1_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ1_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ1_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ1_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ1_RXDPID_OFFSET           0xb30
#define GC_USB_DOEPDMA1_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA1_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA1_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA1_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA1_DMAADDR_OFFSET           0xb34
#define GC_USB_DOEPDMAB1_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB1_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB1_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB1_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB1_DMABUFFERADDR_OFFSET    0xb3c
#define GC_USB_DOEPCTL2_MPS_LSB                  0x0
#define GC_USB_DOEPCTL2_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL2_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL2_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL2_MPS_OFFSET               0xb40
#define GC_USB_DOEPCTL2_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL2_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL2_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL2_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL2_USBACTEP_OFFSET          0xb40
#define GC_USB_DOEPCTL2_DPID_LSB                 0x10
#define GC_USB_DOEPCTL2_DPID_MASK                0x10000
#define GC_USB_DOEPCTL2_DPID_SIZE                0x1
#define GC_USB_DOEPCTL2_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL2_DPID_OFFSET              0xb40
#define GC_USB_DOEPCTL2_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL2_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL2_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL2_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL2_NAKSTS_OFFSET            0xb40
#define GC_USB_DOEPCTL2_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL2_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL2_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL2_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL2_EPTYPE_OFFSET            0xb40
#define GC_USB_DOEPCTL2_SNP_LSB                  0x14
#define GC_USB_DOEPCTL2_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL2_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL2_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL2_SNP_OFFSET               0xb40
#define GC_USB_DOEPCTL2_STALL_LSB                0x15
#define GC_USB_DOEPCTL2_STALL_MASK               0x200000
#define GC_USB_DOEPCTL2_STALL_SIZE               0x1
#define GC_USB_DOEPCTL2_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL2_STALL_OFFSET             0xb40
#define GC_USB_DOEPCTL2_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL2_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL2_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL2_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL2_CNAK_OFFSET              0xb40
#define GC_USB_DOEPCTL2_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL2_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL2_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL2_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL2_SNAK_OFFSET              0xb40
#define GC_USB_DOEPCTL2_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL2_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL2_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL2_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL2_SETD0PID_OFFSET          0xb40
#define GC_USB_DOEPCTL2_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL2_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL2_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL2_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL2_SETD1PID_OFFSET          0xb40
#define GC_USB_DOEPCTL2_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL2_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL2_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL2_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL2_EPDIS_OFFSET             0xb40
#define GC_USB_DOEPCTL2_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL2_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL2_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL2_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL2_EPENA_OFFSET             0xb40
#define GC_USB_DOEPINT2_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT2_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT2_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT2_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT2_XFERCOMPL_OFFSET         0xb48
#define GC_USB_DOEPINT2_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT2_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT2_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT2_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT2_EPDISBLD_OFFSET          0xb48
#define GC_USB_DOEPINT2_AHBERR_LSB               0x2
#define GC_USB_DOEPINT2_AHBERR_MASK              0x4
#define GC_USB_DOEPINT2_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT2_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT2_AHBERR_OFFSET            0xb48
#define GC_USB_DOEPINT2_SETUP_LSB                0x3
#define GC_USB_DOEPINT2_SETUP_MASK               0x8
#define GC_USB_DOEPINT2_SETUP_SIZE               0x1
#define GC_USB_DOEPINT2_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT2_SETUP_OFFSET             0xb48
#define GC_USB_DOEPINT2_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT2_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT2_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT2_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT2_OUTTKNEPDIS_OFFSET       0xb48
#define GC_USB_DOEPINT2_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT2_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT2_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT2_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT2_STSPHSERCVD_OFFSET       0xb48
#define GC_USB_DOEPINT2_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT2_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT2_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT2_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT2_BACK2BACKSETUP_OFFSET    0xb48
#define GC_USB_DOEPINT2_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT2_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT2_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT2_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT2_OUTPKTERR_OFFSET         0xb48
#define GC_USB_DOEPINT2_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT2_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT2_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT2_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT2_BNAINTR_OFFSET           0xb48
#define GC_USB_DOEPINT2_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT2_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT2_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT2_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT2_PKTDRPSTS_OFFSET         0xb48
#define GC_USB_DOEPINT2_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT2_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT2_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT2_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT2_BBLEERR_OFFSET           0xb48
#define GC_USB_DOEPINT2_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT2_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT2_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT2_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT2_NAKINTRPT_OFFSET         0xb48
#define GC_USB_DOEPINT2_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT2_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT2_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT2_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT2_NYETINTRPT_OFFSET        0xb48
#define GC_USB_DOEPINT2_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT2_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT2_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT2_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT2_STUPPKTRCVD_OFFSET       0xb48
#define GC_USB_DOEPTSIZ2_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ2_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ2_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ2_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ2_XFERSIZE_OFFSET         0xb50
#define GC_USB_DOEPTSIZ2_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ2_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ2_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ2_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ2_PKTCNT_OFFSET           0xb50
#define GC_USB_DOEPTSIZ2_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ2_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ2_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ2_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ2_RXDPID_OFFSET           0xb50
#define GC_USB_DOEPDMA2_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA2_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA2_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA2_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA2_DMAADDR_OFFSET           0xb54
#define GC_USB_DOEPDMAB2_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB2_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB2_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB2_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB2_DMABUFFERADDR_OFFSET    0xb5c
#define GC_USB_DOEPCTL3_MPS_LSB                  0x0
#define GC_USB_DOEPCTL3_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL3_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL3_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL3_MPS_OFFSET               0xb60
#define GC_USB_DOEPCTL3_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL3_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL3_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL3_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL3_USBACTEP_OFFSET          0xb60
#define GC_USB_DOEPCTL3_DPID_LSB                 0x10
#define GC_USB_DOEPCTL3_DPID_MASK                0x10000
#define GC_USB_DOEPCTL3_DPID_SIZE                0x1
#define GC_USB_DOEPCTL3_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL3_DPID_OFFSET              0xb60
#define GC_USB_DOEPCTL3_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL3_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL3_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL3_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL3_NAKSTS_OFFSET            0xb60
#define GC_USB_DOEPCTL3_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL3_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL3_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL3_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL3_EPTYPE_OFFSET            0xb60
#define GC_USB_DOEPCTL3_SNP_LSB                  0x14
#define GC_USB_DOEPCTL3_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL3_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL3_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL3_SNP_OFFSET               0xb60
#define GC_USB_DOEPCTL3_STALL_LSB                0x15
#define GC_USB_DOEPCTL3_STALL_MASK               0x200000
#define GC_USB_DOEPCTL3_STALL_SIZE               0x1
#define GC_USB_DOEPCTL3_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL3_STALL_OFFSET             0xb60
#define GC_USB_DOEPCTL3_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL3_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL3_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL3_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL3_CNAK_OFFSET              0xb60
#define GC_USB_DOEPCTL3_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL3_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL3_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL3_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL3_SNAK_OFFSET              0xb60
#define GC_USB_DOEPCTL3_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL3_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL3_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL3_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL3_SETD0PID_OFFSET          0xb60
#define GC_USB_DOEPCTL3_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL3_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL3_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL3_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL3_SETD1PID_OFFSET          0xb60
#define GC_USB_DOEPCTL3_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL3_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL3_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL3_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL3_EPDIS_OFFSET             0xb60
#define GC_USB_DOEPCTL3_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL3_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL3_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL3_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL3_EPENA_OFFSET             0xb60
#define GC_USB_DOEPINT3_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT3_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT3_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT3_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT3_XFERCOMPL_OFFSET         0xb68
#define GC_USB_DOEPINT3_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT3_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT3_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT3_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT3_EPDISBLD_OFFSET          0xb68
#define GC_USB_DOEPINT3_AHBERR_LSB               0x2
#define GC_USB_DOEPINT3_AHBERR_MASK              0x4
#define GC_USB_DOEPINT3_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT3_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT3_AHBERR_OFFSET            0xb68
#define GC_USB_DOEPINT3_SETUP_LSB                0x3
#define GC_USB_DOEPINT3_SETUP_MASK               0x8
#define GC_USB_DOEPINT3_SETUP_SIZE               0x1
#define GC_USB_DOEPINT3_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT3_SETUP_OFFSET             0xb68
#define GC_USB_DOEPINT3_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT3_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT3_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT3_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT3_OUTTKNEPDIS_OFFSET       0xb68
#define GC_USB_DOEPINT3_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT3_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT3_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT3_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT3_STSPHSERCVD_OFFSET       0xb68
#define GC_USB_DOEPINT3_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT3_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT3_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT3_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT3_BACK2BACKSETUP_OFFSET    0xb68
#define GC_USB_DOEPINT3_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT3_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT3_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT3_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT3_OUTPKTERR_OFFSET         0xb68
#define GC_USB_DOEPINT3_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT3_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT3_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT3_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT3_BNAINTR_OFFSET           0xb68
#define GC_USB_DOEPINT3_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT3_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT3_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT3_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT3_PKTDRPSTS_OFFSET         0xb68
#define GC_USB_DOEPINT3_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT3_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT3_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT3_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT3_BBLEERR_OFFSET           0xb68
#define GC_USB_DOEPINT3_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT3_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT3_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT3_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT3_NAKINTRPT_OFFSET         0xb68
#define GC_USB_DOEPINT3_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT3_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT3_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT3_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT3_NYETINTRPT_OFFSET        0xb68
#define GC_USB_DOEPINT3_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT3_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT3_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT3_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT3_STUPPKTRCVD_OFFSET       0xb68
#define GC_USB_DOEPTSIZ3_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ3_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ3_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ3_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ3_XFERSIZE_OFFSET         0xb70
#define GC_USB_DOEPTSIZ3_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ3_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ3_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ3_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ3_PKTCNT_OFFSET           0xb70
#define GC_USB_DOEPTSIZ3_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ3_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ3_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ3_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ3_RXDPID_OFFSET           0xb70
#define GC_USB_DOEPDMA3_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA3_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA3_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA3_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA3_DMAADDR_OFFSET           0xb74
#define GC_USB_DOEPDMAB3_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB3_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB3_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB3_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB3_DMABUFFERADDR_OFFSET    0xb7c
#define GC_USB_DOEPCTL4_MPS_LSB                  0x0
#define GC_USB_DOEPCTL4_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL4_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL4_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL4_MPS_OFFSET               0xb80
#define GC_USB_DOEPCTL4_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL4_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL4_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL4_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL4_USBACTEP_OFFSET          0xb80
#define GC_USB_DOEPCTL4_DPID_LSB                 0x10
#define GC_USB_DOEPCTL4_DPID_MASK                0x10000
#define GC_USB_DOEPCTL4_DPID_SIZE                0x1
#define GC_USB_DOEPCTL4_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL4_DPID_OFFSET              0xb80
#define GC_USB_DOEPCTL4_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL4_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL4_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL4_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL4_NAKSTS_OFFSET            0xb80
#define GC_USB_DOEPCTL4_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL4_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL4_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL4_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL4_EPTYPE_OFFSET            0xb80
#define GC_USB_DOEPCTL4_SNP_LSB                  0x14
#define GC_USB_DOEPCTL4_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL4_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL4_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL4_SNP_OFFSET               0xb80
#define GC_USB_DOEPCTL4_STALL_LSB                0x15
#define GC_USB_DOEPCTL4_STALL_MASK               0x200000
#define GC_USB_DOEPCTL4_STALL_SIZE               0x1
#define GC_USB_DOEPCTL4_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL4_STALL_OFFSET             0xb80
#define GC_USB_DOEPCTL4_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL4_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL4_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL4_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL4_CNAK_OFFSET              0xb80
#define GC_USB_DOEPCTL4_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL4_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL4_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL4_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL4_SNAK_OFFSET              0xb80
#define GC_USB_DOEPCTL4_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL4_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL4_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL4_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL4_SETD0PID_OFFSET          0xb80
#define GC_USB_DOEPCTL4_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL4_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL4_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL4_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL4_SETD1PID_OFFSET          0xb80
#define GC_USB_DOEPCTL4_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL4_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL4_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL4_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL4_EPDIS_OFFSET             0xb80
#define GC_USB_DOEPCTL4_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL4_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL4_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL4_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL4_EPENA_OFFSET             0xb80
#define GC_USB_DOEPINT4_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT4_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT4_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT4_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT4_XFERCOMPL_OFFSET         0xb88
#define GC_USB_DOEPINT4_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT4_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT4_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT4_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT4_EPDISBLD_OFFSET          0xb88
#define GC_USB_DOEPINT4_AHBERR_LSB               0x2
#define GC_USB_DOEPINT4_AHBERR_MASK              0x4
#define GC_USB_DOEPINT4_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT4_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT4_AHBERR_OFFSET            0xb88
#define GC_USB_DOEPINT4_SETUP_LSB                0x3
#define GC_USB_DOEPINT4_SETUP_MASK               0x8
#define GC_USB_DOEPINT4_SETUP_SIZE               0x1
#define GC_USB_DOEPINT4_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT4_SETUP_OFFSET             0xb88
#define GC_USB_DOEPINT4_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT4_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT4_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT4_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT4_OUTTKNEPDIS_OFFSET       0xb88
#define GC_USB_DOEPINT4_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT4_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT4_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT4_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT4_STSPHSERCVD_OFFSET       0xb88
#define GC_USB_DOEPINT4_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT4_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT4_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT4_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT4_BACK2BACKSETUP_OFFSET    0xb88
#define GC_USB_DOEPINT4_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT4_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT4_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT4_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT4_OUTPKTERR_OFFSET         0xb88
#define GC_USB_DOEPINT4_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT4_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT4_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT4_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT4_BNAINTR_OFFSET           0xb88
#define GC_USB_DOEPINT4_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT4_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT4_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT4_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT4_PKTDRPSTS_OFFSET         0xb88
#define GC_USB_DOEPINT4_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT4_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT4_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT4_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT4_BBLEERR_OFFSET           0xb88
#define GC_USB_DOEPINT4_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT4_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT4_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT4_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT4_NAKINTRPT_OFFSET         0xb88
#define GC_USB_DOEPINT4_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT4_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT4_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT4_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT4_NYETINTRPT_OFFSET        0xb88
#define GC_USB_DOEPINT4_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT4_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT4_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT4_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT4_STUPPKTRCVD_OFFSET       0xb88
#define GC_USB_DOEPTSIZ4_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ4_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ4_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ4_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ4_XFERSIZE_OFFSET         0xb90
#define GC_USB_DOEPTSIZ4_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ4_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ4_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ4_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ4_PKTCNT_OFFSET           0xb90
#define GC_USB_DOEPTSIZ4_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ4_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ4_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ4_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ4_RXDPID_OFFSET           0xb90
#define GC_USB_DOEPDMA4_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA4_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA4_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA4_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA4_DMAADDR_OFFSET           0xb94
#define GC_USB_DOEPDMAB4_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB4_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB4_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB4_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB4_DMABUFFERADDR_OFFSET    0xb9c
#define GC_USB_DOEPCTL5_MPS_LSB                  0x0
#define GC_USB_DOEPCTL5_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL5_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL5_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL5_MPS_OFFSET               0xba0
#define GC_USB_DOEPCTL5_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL5_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL5_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL5_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL5_USBACTEP_OFFSET          0xba0
#define GC_USB_DOEPCTL5_DPID_LSB                 0x10
#define GC_USB_DOEPCTL5_DPID_MASK                0x10000
#define GC_USB_DOEPCTL5_DPID_SIZE                0x1
#define GC_USB_DOEPCTL5_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL5_DPID_OFFSET              0xba0
#define GC_USB_DOEPCTL5_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL5_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL5_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL5_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL5_NAKSTS_OFFSET            0xba0
#define GC_USB_DOEPCTL5_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL5_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL5_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL5_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL5_EPTYPE_OFFSET            0xba0
#define GC_USB_DOEPCTL5_SNP_LSB                  0x14
#define GC_USB_DOEPCTL5_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL5_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL5_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL5_SNP_OFFSET               0xba0
#define GC_USB_DOEPCTL5_STALL_LSB                0x15
#define GC_USB_DOEPCTL5_STALL_MASK               0x200000
#define GC_USB_DOEPCTL5_STALL_SIZE               0x1
#define GC_USB_DOEPCTL5_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL5_STALL_OFFSET             0xba0
#define GC_USB_DOEPCTL5_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL5_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL5_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL5_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL5_CNAK_OFFSET              0xba0
#define GC_USB_DOEPCTL5_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL5_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL5_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL5_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL5_SNAK_OFFSET              0xba0
#define GC_USB_DOEPCTL5_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL5_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL5_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL5_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL5_SETD0PID_OFFSET          0xba0
#define GC_USB_DOEPCTL5_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL5_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL5_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL5_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL5_SETD1PID_OFFSET          0xba0
#define GC_USB_DOEPCTL5_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL5_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL5_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL5_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL5_EPDIS_OFFSET             0xba0
#define GC_USB_DOEPCTL5_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL5_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL5_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL5_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL5_EPENA_OFFSET             0xba0
#define GC_USB_DOEPINT5_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT5_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT5_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT5_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT5_XFERCOMPL_OFFSET         0xba8
#define GC_USB_DOEPINT5_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT5_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT5_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT5_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT5_EPDISBLD_OFFSET          0xba8
#define GC_USB_DOEPINT5_AHBERR_LSB               0x2
#define GC_USB_DOEPINT5_AHBERR_MASK              0x4
#define GC_USB_DOEPINT5_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT5_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT5_AHBERR_OFFSET            0xba8
#define GC_USB_DOEPINT5_SETUP_LSB                0x3
#define GC_USB_DOEPINT5_SETUP_MASK               0x8
#define GC_USB_DOEPINT5_SETUP_SIZE               0x1
#define GC_USB_DOEPINT5_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT5_SETUP_OFFSET             0xba8
#define GC_USB_DOEPINT5_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT5_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT5_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT5_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT5_OUTTKNEPDIS_OFFSET       0xba8
#define GC_USB_DOEPINT5_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT5_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT5_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT5_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT5_STSPHSERCVD_OFFSET       0xba8
#define GC_USB_DOEPINT5_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT5_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT5_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT5_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT5_BACK2BACKSETUP_OFFSET    0xba8
#define GC_USB_DOEPINT5_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT5_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT5_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT5_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT5_OUTPKTERR_OFFSET         0xba8
#define GC_USB_DOEPINT5_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT5_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT5_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT5_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT5_BNAINTR_OFFSET           0xba8
#define GC_USB_DOEPINT5_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT5_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT5_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT5_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT5_PKTDRPSTS_OFFSET         0xba8
#define GC_USB_DOEPINT5_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT5_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT5_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT5_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT5_BBLEERR_OFFSET           0xba8
#define GC_USB_DOEPINT5_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT5_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT5_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT5_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT5_NAKINTRPT_OFFSET         0xba8
#define GC_USB_DOEPINT5_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT5_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT5_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT5_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT5_NYETINTRPT_OFFSET        0xba8
#define GC_USB_DOEPINT5_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT5_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT5_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT5_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT5_STUPPKTRCVD_OFFSET       0xba8
#define GC_USB_DOEPTSIZ5_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ5_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ5_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ5_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ5_XFERSIZE_OFFSET         0xbb0
#define GC_USB_DOEPTSIZ5_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ5_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ5_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ5_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ5_PKTCNT_OFFSET           0xbb0
#define GC_USB_DOEPTSIZ5_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ5_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ5_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ5_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ5_RXDPID_OFFSET           0xbb0
#define GC_USB_DOEPDMA5_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA5_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA5_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA5_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA5_DMAADDR_OFFSET           0xbb4
#define GC_USB_DOEPDMAB5_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB5_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB5_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB5_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB5_DMABUFFERADDR_OFFSET    0xbbc
#define GC_USB_DOEPCTL6_MPS_LSB                  0x0
#define GC_USB_DOEPCTL6_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL6_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL6_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL6_MPS_OFFSET               0xbc0
#define GC_USB_DOEPCTL6_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL6_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL6_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL6_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL6_USBACTEP_OFFSET          0xbc0
#define GC_USB_DOEPCTL6_DPID_LSB                 0x10
#define GC_USB_DOEPCTL6_DPID_MASK                0x10000
#define GC_USB_DOEPCTL6_DPID_SIZE                0x1
#define GC_USB_DOEPCTL6_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL6_DPID_OFFSET              0xbc0
#define GC_USB_DOEPCTL6_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL6_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL6_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL6_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL6_NAKSTS_OFFSET            0xbc0
#define GC_USB_DOEPCTL6_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL6_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL6_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL6_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL6_EPTYPE_OFFSET            0xbc0
#define GC_USB_DOEPCTL6_SNP_LSB                  0x14
#define GC_USB_DOEPCTL6_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL6_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL6_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL6_SNP_OFFSET               0xbc0
#define GC_USB_DOEPCTL6_STALL_LSB                0x15
#define GC_USB_DOEPCTL6_STALL_MASK               0x200000
#define GC_USB_DOEPCTL6_STALL_SIZE               0x1
#define GC_USB_DOEPCTL6_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL6_STALL_OFFSET             0xbc0
#define GC_USB_DOEPCTL6_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL6_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL6_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL6_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL6_CNAK_OFFSET              0xbc0
#define GC_USB_DOEPCTL6_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL6_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL6_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL6_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL6_SNAK_OFFSET              0xbc0
#define GC_USB_DOEPCTL6_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL6_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL6_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL6_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL6_SETD0PID_OFFSET          0xbc0
#define GC_USB_DOEPCTL6_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL6_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL6_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL6_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL6_SETD1PID_OFFSET          0xbc0
#define GC_USB_DOEPCTL6_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL6_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL6_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL6_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL6_EPDIS_OFFSET             0xbc0
#define GC_USB_DOEPCTL6_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL6_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL6_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL6_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL6_EPENA_OFFSET             0xbc0
#define GC_USB_DOEPINT6_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT6_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT6_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT6_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT6_XFERCOMPL_OFFSET         0xbc8
#define GC_USB_DOEPINT6_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT6_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT6_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT6_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT6_EPDISBLD_OFFSET          0xbc8
#define GC_USB_DOEPINT6_AHBERR_LSB               0x2
#define GC_USB_DOEPINT6_AHBERR_MASK              0x4
#define GC_USB_DOEPINT6_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT6_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT6_AHBERR_OFFSET            0xbc8
#define GC_USB_DOEPINT6_SETUP_LSB                0x3
#define GC_USB_DOEPINT6_SETUP_MASK               0x8
#define GC_USB_DOEPINT6_SETUP_SIZE               0x1
#define GC_USB_DOEPINT6_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT6_SETUP_OFFSET             0xbc8
#define GC_USB_DOEPINT6_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT6_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT6_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT6_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT6_OUTTKNEPDIS_OFFSET       0xbc8
#define GC_USB_DOEPINT6_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT6_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT6_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT6_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT6_STSPHSERCVD_OFFSET       0xbc8
#define GC_USB_DOEPINT6_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT6_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT6_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT6_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT6_BACK2BACKSETUP_OFFSET    0xbc8
#define GC_USB_DOEPINT6_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT6_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT6_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT6_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT6_OUTPKTERR_OFFSET         0xbc8
#define GC_USB_DOEPINT6_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT6_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT6_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT6_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT6_BNAINTR_OFFSET           0xbc8
#define GC_USB_DOEPINT6_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT6_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT6_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT6_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT6_PKTDRPSTS_OFFSET         0xbc8
#define GC_USB_DOEPINT6_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT6_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT6_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT6_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT6_BBLEERR_OFFSET           0xbc8
#define GC_USB_DOEPINT6_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT6_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT6_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT6_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT6_NAKINTRPT_OFFSET         0xbc8
#define GC_USB_DOEPINT6_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT6_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT6_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT6_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT6_NYETINTRPT_OFFSET        0xbc8
#define GC_USB_DOEPINT6_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT6_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT6_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT6_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT6_STUPPKTRCVD_OFFSET       0xbc8
#define GC_USB_DOEPTSIZ6_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ6_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ6_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ6_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ6_XFERSIZE_OFFSET         0xbd0
#define GC_USB_DOEPTSIZ6_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ6_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ6_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ6_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ6_PKTCNT_OFFSET           0xbd0
#define GC_USB_DOEPTSIZ6_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ6_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ6_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ6_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ6_RXDPID_OFFSET           0xbd0
#define GC_USB_DOEPDMA6_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA6_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA6_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA6_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA6_DMAADDR_OFFSET           0xbd4
#define GC_USB_DOEPDMAB6_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB6_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB6_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB6_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB6_DMABUFFERADDR_OFFSET    0xbdc
#define GC_USB_DOEPCTL7_MPS_LSB                  0x0
#define GC_USB_DOEPCTL7_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL7_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL7_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL7_MPS_OFFSET               0xbe0
#define GC_USB_DOEPCTL7_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL7_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL7_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL7_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL7_USBACTEP_OFFSET          0xbe0
#define GC_USB_DOEPCTL7_DPID_LSB                 0x10
#define GC_USB_DOEPCTL7_DPID_MASK                0x10000
#define GC_USB_DOEPCTL7_DPID_SIZE                0x1
#define GC_USB_DOEPCTL7_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL7_DPID_OFFSET              0xbe0
#define GC_USB_DOEPCTL7_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL7_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL7_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL7_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL7_NAKSTS_OFFSET            0xbe0
#define GC_USB_DOEPCTL7_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL7_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL7_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL7_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL7_EPTYPE_OFFSET            0xbe0
#define GC_USB_DOEPCTL7_SNP_LSB                  0x14
#define GC_USB_DOEPCTL7_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL7_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL7_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL7_SNP_OFFSET               0xbe0
#define GC_USB_DOEPCTL7_STALL_LSB                0x15
#define GC_USB_DOEPCTL7_STALL_MASK               0x200000
#define GC_USB_DOEPCTL7_STALL_SIZE               0x1
#define GC_USB_DOEPCTL7_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL7_STALL_OFFSET             0xbe0
#define GC_USB_DOEPCTL7_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL7_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL7_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL7_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL7_CNAK_OFFSET              0xbe0
#define GC_USB_DOEPCTL7_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL7_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL7_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL7_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL7_SNAK_OFFSET              0xbe0
#define GC_USB_DOEPCTL7_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL7_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL7_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL7_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL7_SETD0PID_OFFSET          0xbe0
#define GC_USB_DOEPCTL7_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL7_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL7_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL7_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL7_SETD1PID_OFFSET          0xbe0
#define GC_USB_DOEPCTL7_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL7_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL7_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL7_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL7_EPDIS_OFFSET             0xbe0
#define GC_USB_DOEPCTL7_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL7_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL7_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL7_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL7_EPENA_OFFSET             0xbe0
#define GC_USB_DOEPINT7_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT7_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT7_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT7_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT7_XFERCOMPL_OFFSET         0xbe8
#define GC_USB_DOEPINT7_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT7_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT7_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT7_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT7_EPDISBLD_OFFSET          0xbe8
#define GC_USB_DOEPINT7_AHBERR_LSB               0x2
#define GC_USB_DOEPINT7_AHBERR_MASK              0x4
#define GC_USB_DOEPINT7_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT7_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT7_AHBERR_OFFSET            0xbe8
#define GC_USB_DOEPINT7_SETUP_LSB                0x3
#define GC_USB_DOEPINT7_SETUP_MASK               0x8
#define GC_USB_DOEPINT7_SETUP_SIZE               0x1
#define GC_USB_DOEPINT7_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT7_SETUP_OFFSET             0xbe8
#define GC_USB_DOEPINT7_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT7_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT7_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT7_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT7_OUTTKNEPDIS_OFFSET       0xbe8
#define GC_USB_DOEPINT7_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT7_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT7_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT7_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT7_STSPHSERCVD_OFFSET       0xbe8
#define GC_USB_DOEPINT7_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT7_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT7_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT7_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT7_BACK2BACKSETUP_OFFSET    0xbe8
#define GC_USB_DOEPINT7_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT7_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT7_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT7_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT7_OUTPKTERR_OFFSET         0xbe8
#define GC_USB_DOEPINT7_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT7_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT7_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT7_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT7_BNAINTR_OFFSET           0xbe8
#define GC_USB_DOEPINT7_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT7_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT7_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT7_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT7_PKTDRPSTS_OFFSET         0xbe8
#define GC_USB_DOEPINT7_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT7_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT7_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT7_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT7_BBLEERR_OFFSET           0xbe8
#define GC_USB_DOEPINT7_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT7_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT7_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT7_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT7_NAKINTRPT_OFFSET         0xbe8
#define GC_USB_DOEPINT7_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT7_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT7_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT7_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT7_NYETINTRPT_OFFSET        0xbe8
#define GC_USB_DOEPINT7_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT7_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT7_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT7_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT7_STUPPKTRCVD_OFFSET       0xbe8
#define GC_USB_DOEPTSIZ7_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ7_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ7_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ7_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ7_XFERSIZE_OFFSET         0xbf0
#define GC_USB_DOEPTSIZ7_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ7_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ7_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ7_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ7_PKTCNT_OFFSET           0xbf0
#define GC_USB_DOEPTSIZ7_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ7_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ7_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ7_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ7_RXDPID_OFFSET           0xbf0
#define GC_USB_DOEPDMA7_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA7_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA7_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA7_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA7_DMAADDR_OFFSET           0xbf4
#define GC_USB_DOEPDMAB7_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB7_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB7_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB7_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB7_DMABUFFERADDR_OFFSET    0xbfc
#define GC_USB_DOEPCTL8_MPS_LSB                  0x0
#define GC_USB_DOEPCTL8_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL8_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL8_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL8_MPS_OFFSET               0xc00
#define GC_USB_DOEPCTL8_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL8_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL8_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL8_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL8_USBACTEP_OFFSET          0xc00
#define GC_USB_DOEPCTL8_DPID_LSB                 0x10
#define GC_USB_DOEPCTL8_DPID_MASK                0x10000
#define GC_USB_DOEPCTL8_DPID_SIZE                0x1
#define GC_USB_DOEPCTL8_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL8_DPID_OFFSET              0xc00
#define GC_USB_DOEPCTL8_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL8_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL8_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL8_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL8_NAKSTS_OFFSET            0xc00
#define GC_USB_DOEPCTL8_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL8_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL8_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL8_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL8_EPTYPE_OFFSET            0xc00
#define GC_USB_DOEPCTL8_SNP_LSB                  0x14
#define GC_USB_DOEPCTL8_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL8_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL8_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL8_SNP_OFFSET               0xc00
#define GC_USB_DOEPCTL8_STALL_LSB                0x15
#define GC_USB_DOEPCTL8_STALL_MASK               0x200000
#define GC_USB_DOEPCTL8_STALL_SIZE               0x1
#define GC_USB_DOEPCTL8_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL8_STALL_OFFSET             0xc00
#define GC_USB_DOEPCTL8_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL8_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL8_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL8_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL8_CNAK_OFFSET              0xc00
#define GC_USB_DOEPCTL8_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL8_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL8_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL8_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL8_SNAK_OFFSET              0xc00
#define GC_USB_DOEPCTL8_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL8_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL8_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL8_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL8_SETD0PID_OFFSET          0xc00
#define GC_USB_DOEPCTL8_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL8_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL8_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL8_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL8_SETD1PID_OFFSET          0xc00
#define GC_USB_DOEPCTL8_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL8_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL8_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL8_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL8_EPDIS_OFFSET             0xc00
#define GC_USB_DOEPCTL8_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL8_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL8_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL8_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL8_EPENA_OFFSET             0xc00
#define GC_USB_DOEPINT8_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT8_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT8_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT8_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT8_XFERCOMPL_OFFSET         0xc08
#define GC_USB_DOEPINT8_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT8_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT8_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT8_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT8_EPDISBLD_OFFSET          0xc08
#define GC_USB_DOEPINT8_AHBERR_LSB               0x2
#define GC_USB_DOEPINT8_AHBERR_MASK              0x4
#define GC_USB_DOEPINT8_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT8_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT8_AHBERR_OFFSET            0xc08
#define GC_USB_DOEPINT8_SETUP_LSB                0x3
#define GC_USB_DOEPINT8_SETUP_MASK               0x8
#define GC_USB_DOEPINT8_SETUP_SIZE               0x1
#define GC_USB_DOEPINT8_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT8_SETUP_OFFSET             0xc08
#define GC_USB_DOEPINT8_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT8_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT8_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT8_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT8_OUTTKNEPDIS_OFFSET       0xc08
#define GC_USB_DOEPINT8_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT8_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT8_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT8_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT8_STSPHSERCVD_OFFSET       0xc08
#define GC_USB_DOEPINT8_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT8_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT8_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT8_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT8_BACK2BACKSETUP_OFFSET    0xc08
#define GC_USB_DOEPINT8_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT8_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT8_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT8_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT8_OUTPKTERR_OFFSET         0xc08
#define GC_USB_DOEPINT8_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT8_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT8_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT8_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT8_BNAINTR_OFFSET           0xc08
#define GC_USB_DOEPINT8_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT8_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT8_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT8_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT8_PKTDRPSTS_OFFSET         0xc08
#define GC_USB_DOEPINT8_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT8_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT8_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT8_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT8_BBLEERR_OFFSET           0xc08
#define GC_USB_DOEPINT8_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT8_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT8_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT8_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT8_NAKINTRPT_OFFSET         0xc08
#define GC_USB_DOEPINT8_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT8_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT8_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT8_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT8_NYETINTRPT_OFFSET        0xc08
#define GC_USB_DOEPINT8_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT8_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT8_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT8_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT8_STUPPKTRCVD_OFFSET       0xc08
#define GC_USB_DOEPTSIZ8_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ8_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ8_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ8_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ8_XFERSIZE_OFFSET         0xc10
#define GC_USB_DOEPTSIZ8_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ8_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ8_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ8_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ8_PKTCNT_OFFSET           0xc10
#define GC_USB_DOEPTSIZ8_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ8_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ8_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ8_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ8_RXDPID_OFFSET           0xc10
#define GC_USB_DOEPDMA8_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA8_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA8_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA8_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA8_DMAADDR_OFFSET           0xc14
#define GC_USB_DOEPDMAB8_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB8_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB8_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB8_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB8_DMABUFFERADDR_OFFSET    0xc1c
#define GC_USB_DOEPCTL9_MPS_LSB                  0x0
#define GC_USB_DOEPCTL9_MPS_MASK                 0x7ff
#define GC_USB_DOEPCTL9_MPS_SIZE                 0xb
#define GC_USB_DOEPCTL9_MPS_DEFAULT              0x0
#define GC_USB_DOEPCTL9_MPS_OFFSET               0xc20
#define GC_USB_DOEPCTL9_USBACTEP_LSB             0xf
#define GC_USB_DOEPCTL9_USBACTEP_MASK            0x8000
#define GC_USB_DOEPCTL9_USBACTEP_SIZE            0x1
#define GC_USB_DOEPCTL9_USBACTEP_DEFAULT         0x0
#define GC_USB_DOEPCTL9_USBACTEP_OFFSET          0xc20
#define GC_USB_DOEPCTL9_DPID_LSB                 0x10
#define GC_USB_DOEPCTL9_DPID_MASK                0x10000
#define GC_USB_DOEPCTL9_DPID_SIZE                0x1
#define GC_USB_DOEPCTL9_DPID_DEFAULT             0x0
#define GC_USB_DOEPCTL9_DPID_OFFSET              0xc20
#define GC_USB_DOEPCTL9_NAKSTS_LSB               0x11
#define GC_USB_DOEPCTL9_NAKSTS_MASK              0x20000
#define GC_USB_DOEPCTL9_NAKSTS_SIZE              0x1
#define GC_USB_DOEPCTL9_NAKSTS_DEFAULT           0x0
#define GC_USB_DOEPCTL9_NAKSTS_OFFSET            0xc20
#define GC_USB_DOEPCTL9_EPTYPE_LSB               0x12
#define GC_USB_DOEPCTL9_EPTYPE_MASK              0xc0000
#define GC_USB_DOEPCTL9_EPTYPE_SIZE              0x2
#define GC_USB_DOEPCTL9_EPTYPE_DEFAULT           0x0
#define GC_USB_DOEPCTL9_EPTYPE_OFFSET            0xc20
#define GC_USB_DOEPCTL9_SNP_LSB                  0x14
#define GC_USB_DOEPCTL9_SNP_MASK                 0x100000
#define GC_USB_DOEPCTL9_SNP_SIZE                 0x1
#define GC_USB_DOEPCTL9_SNP_DEFAULT              0x0
#define GC_USB_DOEPCTL9_SNP_OFFSET               0xc20
#define GC_USB_DOEPCTL9_STALL_LSB                0x15
#define GC_USB_DOEPCTL9_STALL_MASK               0x200000
#define GC_USB_DOEPCTL9_STALL_SIZE               0x1
#define GC_USB_DOEPCTL9_STALL_DEFAULT            0x0
#define GC_USB_DOEPCTL9_STALL_OFFSET             0xc20
#define GC_USB_DOEPCTL9_CNAK_LSB                 0x1a
#define GC_USB_DOEPCTL9_CNAK_MASK                0x4000000
#define GC_USB_DOEPCTL9_CNAK_SIZE                0x1
#define GC_USB_DOEPCTL9_CNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL9_CNAK_OFFSET              0xc20
#define GC_USB_DOEPCTL9_SNAK_LSB                 0x1b
#define GC_USB_DOEPCTL9_SNAK_MASK                0x8000000
#define GC_USB_DOEPCTL9_SNAK_SIZE                0x1
#define GC_USB_DOEPCTL9_SNAK_DEFAULT             0x0
#define GC_USB_DOEPCTL9_SNAK_OFFSET              0xc20
#define GC_USB_DOEPCTL9_SETD0PID_LSB             0x1c
#define GC_USB_DOEPCTL9_SETD0PID_MASK            0x10000000
#define GC_USB_DOEPCTL9_SETD0PID_SIZE            0x1
#define GC_USB_DOEPCTL9_SETD0PID_DEFAULT         0x0
#define GC_USB_DOEPCTL9_SETD0PID_OFFSET          0xc20
#define GC_USB_DOEPCTL9_SETD1PID_LSB             0x1d
#define GC_USB_DOEPCTL9_SETD1PID_MASK            0x20000000
#define GC_USB_DOEPCTL9_SETD1PID_SIZE            0x1
#define GC_USB_DOEPCTL9_SETD1PID_DEFAULT         0x0
#define GC_USB_DOEPCTL9_SETD1PID_OFFSET          0xc20
#define GC_USB_DOEPCTL9_EPDIS_LSB                0x1e
#define GC_USB_DOEPCTL9_EPDIS_MASK               0x40000000
#define GC_USB_DOEPCTL9_EPDIS_SIZE               0x1
#define GC_USB_DOEPCTL9_EPDIS_DEFAULT            0x0
#define GC_USB_DOEPCTL9_EPDIS_OFFSET             0xc20
#define GC_USB_DOEPCTL9_EPENA_LSB                0x1f
#define GC_USB_DOEPCTL9_EPENA_MASK               0x80000000
#define GC_USB_DOEPCTL9_EPENA_SIZE               0x1
#define GC_USB_DOEPCTL9_EPENA_DEFAULT            0x0
#define GC_USB_DOEPCTL9_EPENA_OFFSET             0xc20
#define GC_USB_DOEPINT9_XFERCOMPL_LSB            0x0
#define GC_USB_DOEPINT9_XFERCOMPL_MASK           0x1
#define GC_USB_DOEPINT9_XFERCOMPL_SIZE           0x1
#define GC_USB_DOEPINT9_XFERCOMPL_DEFAULT        0x0
#define GC_USB_DOEPINT9_XFERCOMPL_OFFSET         0xc28
#define GC_USB_DOEPINT9_EPDISBLD_LSB             0x1
#define GC_USB_DOEPINT9_EPDISBLD_MASK            0x2
#define GC_USB_DOEPINT9_EPDISBLD_SIZE            0x1
#define GC_USB_DOEPINT9_EPDISBLD_DEFAULT         0x0
#define GC_USB_DOEPINT9_EPDISBLD_OFFSET          0xc28
#define GC_USB_DOEPINT9_AHBERR_LSB               0x2
#define GC_USB_DOEPINT9_AHBERR_MASK              0x4
#define GC_USB_DOEPINT9_AHBERR_SIZE              0x1
#define GC_USB_DOEPINT9_AHBERR_DEFAULT           0x0
#define GC_USB_DOEPINT9_AHBERR_OFFSET            0xc28
#define GC_USB_DOEPINT9_SETUP_LSB                0x3
#define GC_USB_DOEPINT9_SETUP_MASK               0x8
#define GC_USB_DOEPINT9_SETUP_SIZE               0x1
#define GC_USB_DOEPINT9_SETUP_DEFAULT            0x0
#define GC_USB_DOEPINT9_SETUP_OFFSET             0xc28
#define GC_USB_DOEPINT9_OUTTKNEPDIS_LSB          0x4
#define GC_USB_DOEPINT9_OUTTKNEPDIS_MASK         0x10
#define GC_USB_DOEPINT9_OUTTKNEPDIS_SIZE         0x1
#define GC_USB_DOEPINT9_OUTTKNEPDIS_DEFAULT      0x0
#define GC_USB_DOEPINT9_OUTTKNEPDIS_OFFSET       0xc28
#define GC_USB_DOEPINT9_STSPHSERCVD_LSB          0x5
#define GC_USB_DOEPINT9_STSPHSERCVD_MASK         0x20
#define GC_USB_DOEPINT9_STSPHSERCVD_SIZE         0x1
#define GC_USB_DOEPINT9_STSPHSERCVD_DEFAULT      0x0
#define GC_USB_DOEPINT9_STSPHSERCVD_OFFSET       0xc28
#define GC_USB_DOEPINT9_BACK2BACKSETUP_LSB       0x6
#define GC_USB_DOEPINT9_BACK2BACKSETUP_MASK      0x40
#define GC_USB_DOEPINT9_BACK2BACKSETUP_SIZE      0x1
#define GC_USB_DOEPINT9_BACK2BACKSETUP_DEFAULT   0x0
#define GC_USB_DOEPINT9_BACK2BACKSETUP_OFFSET    0xc28
#define GC_USB_DOEPINT9_OUTPKTERR_LSB            0x8
#define GC_USB_DOEPINT9_OUTPKTERR_MASK           0x100
#define GC_USB_DOEPINT9_OUTPKTERR_SIZE           0x1
#define GC_USB_DOEPINT9_OUTPKTERR_DEFAULT        0x0
#define GC_USB_DOEPINT9_OUTPKTERR_OFFSET         0xc28
#define GC_USB_DOEPINT9_BNAINTR_LSB              0x9
#define GC_USB_DOEPINT9_BNAINTR_MASK             0x200
#define GC_USB_DOEPINT9_BNAINTR_SIZE             0x1
#define GC_USB_DOEPINT9_BNAINTR_DEFAULT          0x0
#define GC_USB_DOEPINT9_BNAINTR_OFFSET           0xc28
#define GC_USB_DOEPINT9_PKTDRPSTS_LSB            0xb
#define GC_USB_DOEPINT9_PKTDRPSTS_MASK           0x800
#define GC_USB_DOEPINT9_PKTDRPSTS_SIZE           0x1
#define GC_USB_DOEPINT9_PKTDRPSTS_DEFAULT        0x0
#define GC_USB_DOEPINT9_PKTDRPSTS_OFFSET         0xc28
#define GC_USB_DOEPINT9_BBLEERR_LSB              0xc
#define GC_USB_DOEPINT9_BBLEERR_MASK             0x1000
#define GC_USB_DOEPINT9_BBLEERR_SIZE             0x1
#define GC_USB_DOEPINT9_BBLEERR_DEFAULT          0x0
#define GC_USB_DOEPINT9_BBLEERR_OFFSET           0xc28
#define GC_USB_DOEPINT9_NAKINTRPT_LSB            0xd
#define GC_USB_DOEPINT9_NAKINTRPT_MASK           0x2000
#define GC_USB_DOEPINT9_NAKINTRPT_SIZE           0x1
#define GC_USB_DOEPINT9_NAKINTRPT_DEFAULT        0x0
#define GC_USB_DOEPINT9_NAKINTRPT_OFFSET         0xc28
#define GC_USB_DOEPINT9_NYETINTRPT_LSB           0xe
#define GC_USB_DOEPINT9_NYETINTRPT_MASK          0x4000
#define GC_USB_DOEPINT9_NYETINTRPT_SIZE          0x1
#define GC_USB_DOEPINT9_NYETINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT9_NYETINTRPT_OFFSET        0xc28
#define GC_USB_DOEPINT9_STUPPKTRCVD_LSB          0xf
#define GC_USB_DOEPINT9_STUPPKTRCVD_MASK         0x8000
#define GC_USB_DOEPINT9_STUPPKTRCVD_SIZE         0x1
#define GC_USB_DOEPINT9_STUPPKTRCVD_DEFAULT      0x0
#define GC_USB_DOEPINT9_STUPPKTRCVD_OFFSET       0xc28
#define GC_USB_DOEPTSIZ9_XFERSIZE_LSB            0x0
#define GC_USB_DOEPTSIZ9_XFERSIZE_MASK           0x7ffff
#define GC_USB_DOEPTSIZ9_XFERSIZE_SIZE           0x13
#define GC_USB_DOEPTSIZ9_XFERSIZE_DEFAULT        0x0
#define GC_USB_DOEPTSIZ9_XFERSIZE_OFFSET         0xc30
#define GC_USB_DOEPTSIZ9_PKTCNT_LSB              0x13
#define GC_USB_DOEPTSIZ9_PKTCNT_MASK             0x1ff80000
#define GC_USB_DOEPTSIZ9_PKTCNT_SIZE             0xa
#define GC_USB_DOEPTSIZ9_PKTCNT_DEFAULT          0x0
#define GC_USB_DOEPTSIZ9_PKTCNT_OFFSET           0xc30
#define GC_USB_DOEPTSIZ9_RXDPID_LSB              0x1d
#define GC_USB_DOEPTSIZ9_RXDPID_MASK             0x60000000
#define GC_USB_DOEPTSIZ9_RXDPID_SIZE             0x2
#define GC_USB_DOEPTSIZ9_RXDPID_DEFAULT          0x0
#define GC_USB_DOEPTSIZ9_RXDPID_OFFSET           0xc30
#define GC_USB_DOEPDMA9_DMAADDR_LSB              0x0
#define GC_USB_DOEPDMA9_DMAADDR_MASK             0xffffffff
#define GC_USB_DOEPDMA9_DMAADDR_SIZE             0x20
#define GC_USB_DOEPDMA9_DMAADDR_DEFAULT          0x0
#define GC_USB_DOEPDMA9_DMAADDR_OFFSET           0xc34
#define GC_USB_DOEPDMAB9_DMABUFFERADDR_LSB       0x0
#define GC_USB_DOEPDMAB9_DMABUFFERADDR_MASK      0xffffffff
#define GC_USB_DOEPDMAB9_DMABUFFERADDR_SIZE      0x20
#define GC_USB_DOEPDMAB9_DMABUFFERADDR_DEFAULT   0x0
#define GC_USB_DOEPDMAB9_DMABUFFERADDR_OFFSET    0xc3c
#define GC_USB_DOEPCTL10_MPS_LSB                 0x0
#define GC_USB_DOEPCTL10_MPS_MASK                0x7ff
#define GC_USB_DOEPCTL10_MPS_SIZE                0xb
#define GC_USB_DOEPCTL10_MPS_DEFAULT             0x0
#define GC_USB_DOEPCTL10_MPS_OFFSET              0xc40
#define GC_USB_DOEPCTL10_USBACTEP_LSB            0xf
#define GC_USB_DOEPCTL10_USBACTEP_MASK           0x8000
#define GC_USB_DOEPCTL10_USBACTEP_SIZE           0x1
#define GC_USB_DOEPCTL10_USBACTEP_DEFAULT        0x0
#define GC_USB_DOEPCTL10_USBACTEP_OFFSET         0xc40
#define GC_USB_DOEPCTL10_DPID_LSB                0x10
#define GC_USB_DOEPCTL10_DPID_MASK               0x10000
#define GC_USB_DOEPCTL10_DPID_SIZE               0x1
#define GC_USB_DOEPCTL10_DPID_DEFAULT            0x0
#define GC_USB_DOEPCTL10_DPID_OFFSET             0xc40
#define GC_USB_DOEPCTL10_NAKSTS_LSB              0x11
#define GC_USB_DOEPCTL10_NAKSTS_MASK             0x20000
#define GC_USB_DOEPCTL10_NAKSTS_SIZE             0x1
#define GC_USB_DOEPCTL10_NAKSTS_DEFAULT          0x0
#define GC_USB_DOEPCTL10_NAKSTS_OFFSET           0xc40
#define GC_USB_DOEPCTL10_EPTYPE_LSB              0x12
#define GC_USB_DOEPCTL10_EPTYPE_MASK             0xc0000
#define GC_USB_DOEPCTL10_EPTYPE_SIZE             0x2
#define GC_USB_DOEPCTL10_EPTYPE_DEFAULT          0x0
#define GC_USB_DOEPCTL10_EPTYPE_OFFSET           0xc40
#define GC_USB_DOEPCTL10_SNP_LSB                 0x14
#define GC_USB_DOEPCTL10_SNP_MASK                0x100000
#define GC_USB_DOEPCTL10_SNP_SIZE                0x1
#define GC_USB_DOEPCTL10_SNP_DEFAULT             0x0
#define GC_USB_DOEPCTL10_SNP_OFFSET              0xc40
#define GC_USB_DOEPCTL10_STALL_LSB               0x15
#define GC_USB_DOEPCTL10_STALL_MASK              0x200000
#define GC_USB_DOEPCTL10_STALL_SIZE              0x1
#define GC_USB_DOEPCTL10_STALL_DEFAULT           0x0
#define GC_USB_DOEPCTL10_STALL_OFFSET            0xc40
#define GC_USB_DOEPCTL10_CNAK_LSB                0x1a
#define GC_USB_DOEPCTL10_CNAK_MASK               0x4000000
#define GC_USB_DOEPCTL10_CNAK_SIZE               0x1
#define GC_USB_DOEPCTL10_CNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL10_CNAK_OFFSET             0xc40
#define GC_USB_DOEPCTL10_SNAK_LSB                0x1b
#define GC_USB_DOEPCTL10_SNAK_MASK               0x8000000
#define GC_USB_DOEPCTL10_SNAK_SIZE               0x1
#define GC_USB_DOEPCTL10_SNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL10_SNAK_OFFSET             0xc40
#define GC_USB_DOEPCTL10_SETD0PID_LSB            0x1c
#define GC_USB_DOEPCTL10_SETD0PID_MASK           0x10000000
#define GC_USB_DOEPCTL10_SETD0PID_SIZE           0x1
#define GC_USB_DOEPCTL10_SETD0PID_DEFAULT        0x0
#define GC_USB_DOEPCTL10_SETD0PID_OFFSET         0xc40
#define GC_USB_DOEPCTL10_SETD1PID_LSB            0x1d
#define GC_USB_DOEPCTL10_SETD1PID_MASK           0x20000000
#define GC_USB_DOEPCTL10_SETD1PID_SIZE           0x1
#define GC_USB_DOEPCTL10_SETD1PID_DEFAULT        0x0
#define GC_USB_DOEPCTL10_SETD1PID_OFFSET         0xc40
#define GC_USB_DOEPCTL10_EPDIS_LSB               0x1e
#define GC_USB_DOEPCTL10_EPDIS_MASK              0x40000000
#define GC_USB_DOEPCTL10_EPDIS_SIZE              0x1
#define GC_USB_DOEPCTL10_EPDIS_DEFAULT           0x0
#define GC_USB_DOEPCTL10_EPDIS_OFFSET            0xc40
#define GC_USB_DOEPCTL10_EPENA_LSB               0x1f
#define GC_USB_DOEPCTL10_EPENA_MASK              0x80000000
#define GC_USB_DOEPCTL10_EPENA_SIZE              0x1
#define GC_USB_DOEPCTL10_EPENA_DEFAULT           0x0
#define GC_USB_DOEPCTL10_EPENA_OFFSET            0xc40
#define GC_USB_DOEPINT10_XFERCOMPL_LSB           0x0
#define GC_USB_DOEPINT10_XFERCOMPL_MASK          0x1
#define GC_USB_DOEPINT10_XFERCOMPL_SIZE          0x1
#define GC_USB_DOEPINT10_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DOEPINT10_XFERCOMPL_OFFSET        0xc48
#define GC_USB_DOEPINT10_EPDISBLD_LSB            0x1
#define GC_USB_DOEPINT10_EPDISBLD_MASK           0x2
#define GC_USB_DOEPINT10_EPDISBLD_SIZE           0x1
#define GC_USB_DOEPINT10_EPDISBLD_DEFAULT        0x0
#define GC_USB_DOEPINT10_EPDISBLD_OFFSET         0xc48
#define GC_USB_DOEPINT10_AHBERR_LSB              0x2
#define GC_USB_DOEPINT10_AHBERR_MASK             0x4
#define GC_USB_DOEPINT10_AHBERR_SIZE             0x1
#define GC_USB_DOEPINT10_AHBERR_DEFAULT          0x0
#define GC_USB_DOEPINT10_AHBERR_OFFSET           0xc48
#define GC_USB_DOEPINT10_SETUP_LSB               0x3
#define GC_USB_DOEPINT10_SETUP_MASK              0x8
#define GC_USB_DOEPINT10_SETUP_SIZE              0x1
#define GC_USB_DOEPINT10_SETUP_DEFAULT           0x0
#define GC_USB_DOEPINT10_SETUP_OFFSET            0xc48
#define GC_USB_DOEPINT10_OUTTKNEPDIS_LSB         0x4
#define GC_USB_DOEPINT10_OUTTKNEPDIS_MASK        0x10
#define GC_USB_DOEPINT10_OUTTKNEPDIS_SIZE        0x1
#define GC_USB_DOEPINT10_OUTTKNEPDIS_DEFAULT     0x0
#define GC_USB_DOEPINT10_OUTTKNEPDIS_OFFSET      0xc48
#define GC_USB_DOEPINT10_STSPHSERCVD_LSB         0x5
#define GC_USB_DOEPINT10_STSPHSERCVD_MASK        0x20
#define GC_USB_DOEPINT10_STSPHSERCVD_SIZE        0x1
#define GC_USB_DOEPINT10_STSPHSERCVD_DEFAULT     0x0
#define GC_USB_DOEPINT10_STSPHSERCVD_OFFSET      0xc48
#define GC_USB_DOEPINT10_BACK2BACKSETUP_LSB      0x6
#define GC_USB_DOEPINT10_BACK2BACKSETUP_MASK     0x40
#define GC_USB_DOEPINT10_BACK2BACKSETUP_SIZE     0x1
#define GC_USB_DOEPINT10_BACK2BACKSETUP_DEFAULT  0x0
#define GC_USB_DOEPINT10_BACK2BACKSETUP_OFFSET   0xc48
#define GC_USB_DOEPINT10_OUTPKTERR_LSB           0x8
#define GC_USB_DOEPINT10_OUTPKTERR_MASK          0x100
#define GC_USB_DOEPINT10_OUTPKTERR_SIZE          0x1
#define GC_USB_DOEPINT10_OUTPKTERR_DEFAULT       0x0
#define GC_USB_DOEPINT10_OUTPKTERR_OFFSET        0xc48
#define GC_USB_DOEPINT10_BNAINTR_LSB             0x9
#define GC_USB_DOEPINT10_BNAINTR_MASK            0x200
#define GC_USB_DOEPINT10_BNAINTR_SIZE            0x1
#define GC_USB_DOEPINT10_BNAINTR_DEFAULT         0x0
#define GC_USB_DOEPINT10_BNAINTR_OFFSET          0xc48
#define GC_USB_DOEPINT10_PKTDRPSTS_LSB           0xb
#define GC_USB_DOEPINT10_PKTDRPSTS_MASK          0x800
#define GC_USB_DOEPINT10_PKTDRPSTS_SIZE          0x1
#define GC_USB_DOEPINT10_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DOEPINT10_PKTDRPSTS_OFFSET        0xc48
#define GC_USB_DOEPINT10_BBLEERR_LSB             0xc
#define GC_USB_DOEPINT10_BBLEERR_MASK            0x1000
#define GC_USB_DOEPINT10_BBLEERR_SIZE            0x1
#define GC_USB_DOEPINT10_BBLEERR_DEFAULT         0x0
#define GC_USB_DOEPINT10_BBLEERR_OFFSET          0xc48
#define GC_USB_DOEPINT10_NAKINTRPT_LSB           0xd
#define GC_USB_DOEPINT10_NAKINTRPT_MASK          0x2000
#define GC_USB_DOEPINT10_NAKINTRPT_SIZE          0x1
#define GC_USB_DOEPINT10_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT10_NAKINTRPT_OFFSET        0xc48
#define GC_USB_DOEPINT10_NYETINTRPT_LSB          0xe
#define GC_USB_DOEPINT10_NYETINTRPT_MASK         0x4000
#define GC_USB_DOEPINT10_NYETINTRPT_SIZE         0x1
#define GC_USB_DOEPINT10_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DOEPINT10_NYETINTRPT_OFFSET       0xc48
#define GC_USB_DOEPINT10_STUPPKTRCVD_LSB         0xf
#define GC_USB_DOEPINT10_STUPPKTRCVD_MASK        0x8000
#define GC_USB_DOEPINT10_STUPPKTRCVD_SIZE        0x1
#define GC_USB_DOEPINT10_STUPPKTRCVD_DEFAULT     0x0
#define GC_USB_DOEPINT10_STUPPKTRCVD_OFFSET      0xc48
#define GC_USB_DOEPTSIZ10_XFERSIZE_LSB           0x0
#define GC_USB_DOEPTSIZ10_XFERSIZE_MASK          0x7ffff
#define GC_USB_DOEPTSIZ10_XFERSIZE_SIZE          0x13
#define GC_USB_DOEPTSIZ10_XFERSIZE_DEFAULT       0x0
#define GC_USB_DOEPTSIZ10_XFERSIZE_OFFSET        0xc50
#define GC_USB_DOEPTSIZ10_PKTCNT_LSB             0x13
#define GC_USB_DOEPTSIZ10_PKTCNT_MASK            0x1ff80000
#define GC_USB_DOEPTSIZ10_PKTCNT_SIZE            0xa
#define GC_USB_DOEPTSIZ10_PKTCNT_DEFAULT         0x0
#define GC_USB_DOEPTSIZ10_PKTCNT_OFFSET          0xc50
#define GC_USB_DOEPTSIZ10_RXDPID_LSB             0x1d
#define GC_USB_DOEPTSIZ10_RXDPID_MASK            0x60000000
#define GC_USB_DOEPTSIZ10_RXDPID_SIZE            0x2
#define GC_USB_DOEPTSIZ10_RXDPID_DEFAULT         0x0
#define GC_USB_DOEPTSIZ10_RXDPID_OFFSET          0xc50
#define GC_USB_DOEPDMA10_DMAADDR_LSB             0x0
#define GC_USB_DOEPDMA10_DMAADDR_MASK            0xffffffff
#define GC_USB_DOEPDMA10_DMAADDR_SIZE            0x20
#define GC_USB_DOEPDMA10_DMAADDR_DEFAULT         0x0
#define GC_USB_DOEPDMA10_DMAADDR_OFFSET          0xc54
#define GC_USB_DOEPDMAB10_DMABUFFERADDR_LSB      0x0
#define GC_USB_DOEPDMAB10_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DOEPDMAB10_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DOEPDMAB10_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DOEPDMAB10_DMABUFFERADDR_OFFSET   0xc5c
#define GC_USB_DOEPCTL11_MPS_LSB                 0x0
#define GC_USB_DOEPCTL11_MPS_MASK                0x7ff
#define GC_USB_DOEPCTL11_MPS_SIZE                0xb
#define GC_USB_DOEPCTL11_MPS_DEFAULT             0x0
#define GC_USB_DOEPCTL11_MPS_OFFSET              0xc60
#define GC_USB_DOEPCTL11_USBACTEP_LSB            0xf
#define GC_USB_DOEPCTL11_USBACTEP_MASK           0x8000
#define GC_USB_DOEPCTL11_USBACTEP_SIZE           0x1
#define GC_USB_DOEPCTL11_USBACTEP_DEFAULT        0x0
#define GC_USB_DOEPCTL11_USBACTEP_OFFSET         0xc60
#define GC_USB_DOEPCTL11_DPID_LSB                0x10
#define GC_USB_DOEPCTL11_DPID_MASK               0x10000
#define GC_USB_DOEPCTL11_DPID_SIZE               0x1
#define GC_USB_DOEPCTL11_DPID_DEFAULT            0x0
#define GC_USB_DOEPCTL11_DPID_OFFSET             0xc60
#define GC_USB_DOEPCTL11_NAKSTS_LSB              0x11
#define GC_USB_DOEPCTL11_NAKSTS_MASK             0x20000
#define GC_USB_DOEPCTL11_NAKSTS_SIZE             0x1
#define GC_USB_DOEPCTL11_NAKSTS_DEFAULT          0x0
#define GC_USB_DOEPCTL11_NAKSTS_OFFSET           0xc60
#define GC_USB_DOEPCTL11_EPTYPE_LSB              0x12
#define GC_USB_DOEPCTL11_EPTYPE_MASK             0xc0000
#define GC_USB_DOEPCTL11_EPTYPE_SIZE             0x2
#define GC_USB_DOEPCTL11_EPTYPE_DEFAULT          0x0
#define GC_USB_DOEPCTL11_EPTYPE_OFFSET           0xc60
#define GC_USB_DOEPCTL11_SNP_LSB                 0x14
#define GC_USB_DOEPCTL11_SNP_MASK                0x100000
#define GC_USB_DOEPCTL11_SNP_SIZE                0x1
#define GC_USB_DOEPCTL11_SNP_DEFAULT             0x0
#define GC_USB_DOEPCTL11_SNP_OFFSET              0xc60
#define GC_USB_DOEPCTL11_STALL_LSB               0x15
#define GC_USB_DOEPCTL11_STALL_MASK              0x200000
#define GC_USB_DOEPCTL11_STALL_SIZE              0x1
#define GC_USB_DOEPCTL11_STALL_DEFAULT           0x0
#define GC_USB_DOEPCTL11_STALL_OFFSET            0xc60
#define GC_USB_DOEPCTL11_CNAK_LSB                0x1a
#define GC_USB_DOEPCTL11_CNAK_MASK               0x4000000
#define GC_USB_DOEPCTL11_CNAK_SIZE               0x1
#define GC_USB_DOEPCTL11_CNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL11_CNAK_OFFSET             0xc60
#define GC_USB_DOEPCTL11_SNAK_LSB                0x1b
#define GC_USB_DOEPCTL11_SNAK_MASK               0x8000000
#define GC_USB_DOEPCTL11_SNAK_SIZE               0x1
#define GC_USB_DOEPCTL11_SNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL11_SNAK_OFFSET             0xc60
#define GC_USB_DOEPCTL11_SETD0PID_LSB            0x1c
#define GC_USB_DOEPCTL11_SETD0PID_MASK           0x10000000
#define GC_USB_DOEPCTL11_SETD0PID_SIZE           0x1
#define GC_USB_DOEPCTL11_SETD0PID_DEFAULT        0x0
#define GC_USB_DOEPCTL11_SETD0PID_OFFSET         0xc60
#define GC_USB_DOEPCTL11_SETD1PID_LSB            0x1d
#define GC_USB_DOEPCTL11_SETD1PID_MASK           0x20000000
#define GC_USB_DOEPCTL11_SETD1PID_SIZE           0x1
#define GC_USB_DOEPCTL11_SETD1PID_DEFAULT        0x0
#define GC_USB_DOEPCTL11_SETD1PID_OFFSET         0xc60
#define GC_USB_DOEPCTL11_EPDIS_LSB               0x1e
#define GC_USB_DOEPCTL11_EPDIS_MASK              0x40000000
#define GC_USB_DOEPCTL11_EPDIS_SIZE              0x1
#define GC_USB_DOEPCTL11_EPDIS_DEFAULT           0x0
#define GC_USB_DOEPCTL11_EPDIS_OFFSET            0xc60
#define GC_USB_DOEPCTL11_EPENA_LSB               0x1f
#define GC_USB_DOEPCTL11_EPENA_MASK              0x80000000
#define GC_USB_DOEPCTL11_EPENA_SIZE              0x1
#define GC_USB_DOEPCTL11_EPENA_DEFAULT           0x0
#define GC_USB_DOEPCTL11_EPENA_OFFSET            0xc60
#define GC_USB_DOEPINT11_XFERCOMPL_LSB           0x0
#define GC_USB_DOEPINT11_XFERCOMPL_MASK          0x1
#define GC_USB_DOEPINT11_XFERCOMPL_SIZE          0x1
#define GC_USB_DOEPINT11_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DOEPINT11_XFERCOMPL_OFFSET        0xc68
#define GC_USB_DOEPINT11_EPDISBLD_LSB            0x1
#define GC_USB_DOEPINT11_EPDISBLD_MASK           0x2
#define GC_USB_DOEPINT11_EPDISBLD_SIZE           0x1
#define GC_USB_DOEPINT11_EPDISBLD_DEFAULT        0x0
#define GC_USB_DOEPINT11_EPDISBLD_OFFSET         0xc68
#define GC_USB_DOEPINT11_AHBERR_LSB              0x2
#define GC_USB_DOEPINT11_AHBERR_MASK             0x4
#define GC_USB_DOEPINT11_AHBERR_SIZE             0x1
#define GC_USB_DOEPINT11_AHBERR_DEFAULT          0x0
#define GC_USB_DOEPINT11_AHBERR_OFFSET           0xc68
#define GC_USB_DOEPINT11_SETUP_LSB               0x3
#define GC_USB_DOEPINT11_SETUP_MASK              0x8
#define GC_USB_DOEPINT11_SETUP_SIZE              0x1
#define GC_USB_DOEPINT11_SETUP_DEFAULT           0x0
#define GC_USB_DOEPINT11_SETUP_OFFSET            0xc68
#define GC_USB_DOEPINT11_OUTTKNEPDIS_LSB         0x4
#define GC_USB_DOEPINT11_OUTTKNEPDIS_MASK        0x10
#define GC_USB_DOEPINT11_OUTTKNEPDIS_SIZE        0x1
#define GC_USB_DOEPINT11_OUTTKNEPDIS_DEFAULT     0x0
#define GC_USB_DOEPINT11_OUTTKNEPDIS_OFFSET      0xc68
#define GC_USB_DOEPINT11_STSPHSERCVD_LSB         0x5
#define GC_USB_DOEPINT11_STSPHSERCVD_MASK        0x20
#define GC_USB_DOEPINT11_STSPHSERCVD_SIZE        0x1
#define GC_USB_DOEPINT11_STSPHSERCVD_DEFAULT     0x0
#define GC_USB_DOEPINT11_STSPHSERCVD_OFFSET      0xc68
#define GC_USB_DOEPINT11_BACK2BACKSETUP_LSB      0x6
#define GC_USB_DOEPINT11_BACK2BACKSETUP_MASK     0x40
#define GC_USB_DOEPINT11_BACK2BACKSETUP_SIZE     0x1
#define GC_USB_DOEPINT11_BACK2BACKSETUP_DEFAULT  0x0
#define GC_USB_DOEPINT11_BACK2BACKSETUP_OFFSET   0xc68
#define GC_USB_DOEPINT11_OUTPKTERR_LSB           0x8
#define GC_USB_DOEPINT11_OUTPKTERR_MASK          0x100
#define GC_USB_DOEPINT11_OUTPKTERR_SIZE          0x1
#define GC_USB_DOEPINT11_OUTPKTERR_DEFAULT       0x0
#define GC_USB_DOEPINT11_OUTPKTERR_OFFSET        0xc68
#define GC_USB_DOEPINT11_BNAINTR_LSB             0x9
#define GC_USB_DOEPINT11_BNAINTR_MASK            0x200
#define GC_USB_DOEPINT11_BNAINTR_SIZE            0x1
#define GC_USB_DOEPINT11_BNAINTR_DEFAULT         0x0
#define GC_USB_DOEPINT11_BNAINTR_OFFSET          0xc68
#define GC_USB_DOEPINT11_PKTDRPSTS_LSB           0xb
#define GC_USB_DOEPINT11_PKTDRPSTS_MASK          0x800
#define GC_USB_DOEPINT11_PKTDRPSTS_SIZE          0x1
#define GC_USB_DOEPINT11_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DOEPINT11_PKTDRPSTS_OFFSET        0xc68
#define GC_USB_DOEPINT11_BBLEERR_LSB             0xc
#define GC_USB_DOEPINT11_BBLEERR_MASK            0x1000
#define GC_USB_DOEPINT11_BBLEERR_SIZE            0x1
#define GC_USB_DOEPINT11_BBLEERR_DEFAULT         0x0
#define GC_USB_DOEPINT11_BBLEERR_OFFSET          0xc68
#define GC_USB_DOEPINT11_NAKINTRPT_LSB           0xd
#define GC_USB_DOEPINT11_NAKINTRPT_MASK          0x2000
#define GC_USB_DOEPINT11_NAKINTRPT_SIZE          0x1
#define GC_USB_DOEPINT11_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT11_NAKINTRPT_OFFSET        0xc68
#define GC_USB_DOEPINT11_NYETINTRPT_LSB          0xe
#define GC_USB_DOEPINT11_NYETINTRPT_MASK         0x4000
#define GC_USB_DOEPINT11_NYETINTRPT_SIZE         0x1
#define GC_USB_DOEPINT11_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DOEPINT11_NYETINTRPT_OFFSET       0xc68
#define GC_USB_DOEPINT11_STUPPKTRCVD_LSB         0xf
#define GC_USB_DOEPINT11_STUPPKTRCVD_MASK        0x8000
#define GC_USB_DOEPINT11_STUPPKTRCVD_SIZE        0x1
#define GC_USB_DOEPINT11_STUPPKTRCVD_DEFAULT     0x0
#define GC_USB_DOEPINT11_STUPPKTRCVD_OFFSET      0xc68
#define GC_USB_DOEPTSIZ11_XFERSIZE_LSB           0x0
#define GC_USB_DOEPTSIZ11_XFERSIZE_MASK          0x7ffff
#define GC_USB_DOEPTSIZ11_XFERSIZE_SIZE          0x13
#define GC_USB_DOEPTSIZ11_XFERSIZE_DEFAULT       0x0
#define GC_USB_DOEPTSIZ11_XFERSIZE_OFFSET        0xc70
#define GC_USB_DOEPTSIZ11_PKTCNT_LSB             0x13
#define GC_USB_DOEPTSIZ11_PKTCNT_MASK            0x1ff80000
#define GC_USB_DOEPTSIZ11_PKTCNT_SIZE            0xa
#define GC_USB_DOEPTSIZ11_PKTCNT_DEFAULT         0x0
#define GC_USB_DOEPTSIZ11_PKTCNT_OFFSET          0xc70
#define GC_USB_DOEPTSIZ11_RXDPID_LSB             0x1d
#define GC_USB_DOEPTSIZ11_RXDPID_MASK            0x60000000
#define GC_USB_DOEPTSIZ11_RXDPID_SIZE            0x2
#define GC_USB_DOEPTSIZ11_RXDPID_DEFAULT         0x0
#define GC_USB_DOEPTSIZ11_RXDPID_OFFSET          0xc70
#define GC_USB_DOEPDMA11_DMAADDR_LSB             0x0
#define GC_USB_DOEPDMA11_DMAADDR_MASK            0xffffffff
#define GC_USB_DOEPDMA11_DMAADDR_SIZE            0x20
#define GC_USB_DOEPDMA11_DMAADDR_DEFAULT         0x0
#define GC_USB_DOEPDMA11_DMAADDR_OFFSET          0xc74
#define GC_USB_DOEPDMAB11_DMABUFFERADDR_LSB      0x0
#define GC_USB_DOEPDMAB11_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DOEPDMAB11_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DOEPDMAB11_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DOEPDMAB11_DMABUFFERADDR_OFFSET   0xc7c
#define GC_USB_DOEPCTL12_MPS_LSB                 0x0
#define GC_USB_DOEPCTL12_MPS_MASK                0x7ff
#define GC_USB_DOEPCTL12_MPS_SIZE                0xb
#define GC_USB_DOEPCTL12_MPS_DEFAULT             0x0
#define GC_USB_DOEPCTL12_MPS_OFFSET              0xc80
#define GC_USB_DOEPCTL12_USBACTEP_LSB            0xf
#define GC_USB_DOEPCTL12_USBACTEP_MASK           0x8000
#define GC_USB_DOEPCTL12_USBACTEP_SIZE           0x1
#define GC_USB_DOEPCTL12_USBACTEP_DEFAULT        0x0
#define GC_USB_DOEPCTL12_USBACTEP_OFFSET         0xc80
#define GC_USB_DOEPCTL12_DPID_LSB                0x10
#define GC_USB_DOEPCTL12_DPID_MASK               0x10000
#define GC_USB_DOEPCTL12_DPID_SIZE               0x1
#define GC_USB_DOEPCTL12_DPID_DEFAULT            0x0
#define GC_USB_DOEPCTL12_DPID_OFFSET             0xc80
#define GC_USB_DOEPCTL12_NAKSTS_LSB              0x11
#define GC_USB_DOEPCTL12_NAKSTS_MASK             0x20000
#define GC_USB_DOEPCTL12_NAKSTS_SIZE             0x1
#define GC_USB_DOEPCTL12_NAKSTS_DEFAULT          0x0
#define GC_USB_DOEPCTL12_NAKSTS_OFFSET           0xc80
#define GC_USB_DOEPCTL12_EPTYPE_LSB              0x12
#define GC_USB_DOEPCTL12_EPTYPE_MASK             0xc0000
#define GC_USB_DOEPCTL12_EPTYPE_SIZE             0x2
#define GC_USB_DOEPCTL12_EPTYPE_DEFAULT          0x0
#define GC_USB_DOEPCTL12_EPTYPE_OFFSET           0xc80
#define GC_USB_DOEPCTL12_SNP_LSB                 0x14
#define GC_USB_DOEPCTL12_SNP_MASK                0x100000
#define GC_USB_DOEPCTL12_SNP_SIZE                0x1
#define GC_USB_DOEPCTL12_SNP_DEFAULT             0x0
#define GC_USB_DOEPCTL12_SNP_OFFSET              0xc80
#define GC_USB_DOEPCTL12_STALL_LSB               0x15
#define GC_USB_DOEPCTL12_STALL_MASK              0x200000
#define GC_USB_DOEPCTL12_STALL_SIZE              0x1
#define GC_USB_DOEPCTL12_STALL_DEFAULT           0x0
#define GC_USB_DOEPCTL12_STALL_OFFSET            0xc80
#define GC_USB_DOEPCTL12_CNAK_LSB                0x1a
#define GC_USB_DOEPCTL12_CNAK_MASK               0x4000000
#define GC_USB_DOEPCTL12_CNAK_SIZE               0x1
#define GC_USB_DOEPCTL12_CNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL12_CNAK_OFFSET             0xc80
#define GC_USB_DOEPCTL12_SNAK_LSB                0x1b
#define GC_USB_DOEPCTL12_SNAK_MASK               0x8000000
#define GC_USB_DOEPCTL12_SNAK_SIZE               0x1
#define GC_USB_DOEPCTL12_SNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL12_SNAK_OFFSET             0xc80
#define GC_USB_DOEPCTL12_SETD0PID_LSB            0x1c
#define GC_USB_DOEPCTL12_SETD0PID_MASK           0x10000000
#define GC_USB_DOEPCTL12_SETD0PID_SIZE           0x1
#define GC_USB_DOEPCTL12_SETD0PID_DEFAULT        0x0
#define GC_USB_DOEPCTL12_SETD0PID_OFFSET         0xc80
#define GC_USB_DOEPCTL12_SETD1PID_LSB            0x1d
#define GC_USB_DOEPCTL12_SETD1PID_MASK           0x20000000
#define GC_USB_DOEPCTL12_SETD1PID_SIZE           0x1
#define GC_USB_DOEPCTL12_SETD1PID_DEFAULT        0x0
#define GC_USB_DOEPCTL12_SETD1PID_OFFSET         0xc80
#define GC_USB_DOEPCTL12_EPDIS_LSB               0x1e
#define GC_USB_DOEPCTL12_EPDIS_MASK              0x40000000
#define GC_USB_DOEPCTL12_EPDIS_SIZE              0x1
#define GC_USB_DOEPCTL12_EPDIS_DEFAULT           0x0
#define GC_USB_DOEPCTL12_EPDIS_OFFSET            0xc80
#define GC_USB_DOEPCTL12_EPENA_LSB               0x1f
#define GC_USB_DOEPCTL12_EPENA_MASK              0x80000000
#define GC_USB_DOEPCTL12_EPENA_SIZE              0x1
#define GC_USB_DOEPCTL12_EPENA_DEFAULT           0x0
#define GC_USB_DOEPCTL12_EPENA_OFFSET            0xc80
#define GC_USB_DOEPINT12_XFERCOMPL_LSB           0x0
#define GC_USB_DOEPINT12_XFERCOMPL_MASK          0x1
#define GC_USB_DOEPINT12_XFERCOMPL_SIZE          0x1
#define GC_USB_DOEPINT12_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DOEPINT12_XFERCOMPL_OFFSET        0xc88
#define GC_USB_DOEPINT12_EPDISBLD_LSB            0x1
#define GC_USB_DOEPINT12_EPDISBLD_MASK           0x2
#define GC_USB_DOEPINT12_EPDISBLD_SIZE           0x1
#define GC_USB_DOEPINT12_EPDISBLD_DEFAULT        0x0
#define GC_USB_DOEPINT12_EPDISBLD_OFFSET         0xc88
#define GC_USB_DOEPINT12_AHBERR_LSB              0x2
#define GC_USB_DOEPINT12_AHBERR_MASK             0x4
#define GC_USB_DOEPINT12_AHBERR_SIZE             0x1
#define GC_USB_DOEPINT12_AHBERR_DEFAULT          0x0
#define GC_USB_DOEPINT12_AHBERR_OFFSET           0xc88
#define GC_USB_DOEPINT12_SETUP_LSB               0x3
#define GC_USB_DOEPINT12_SETUP_MASK              0x8
#define GC_USB_DOEPINT12_SETUP_SIZE              0x1
#define GC_USB_DOEPINT12_SETUP_DEFAULT           0x0
#define GC_USB_DOEPINT12_SETUP_OFFSET            0xc88
#define GC_USB_DOEPINT12_OUTTKNEPDIS_LSB         0x4
#define GC_USB_DOEPINT12_OUTTKNEPDIS_MASK        0x10
#define GC_USB_DOEPINT12_OUTTKNEPDIS_SIZE        0x1
#define GC_USB_DOEPINT12_OUTTKNEPDIS_DEFAULT     0x0
#define GC_USB_DOEPINT12_OUTTKNEPDIS_OFFSET      0xc88
#define GC_USB_DOEPINT12_STSPHSERCVD_LSB         0x5
#define GC_USB_DOEPINT12_STSPHSERCVD_MASK        0x20
#define GC_USB_DOEPINT12_STSPHSERCVD_SIZE        0x1
#define GC_USB_DOEPINT12_STSPHSERCVD_DEFAULT     0x0
#define GC_USB_DOEPINT12_STSPHSERCVD_OFFSET      0xc88
#define GC_USB_DOEPINT12_BACK2BACKSETUP_LSB      0x6
#define GC_USB_DOEPINT12_BACK2BACKSETUP_MASK     0x40
#define GC_USB_DOEPINT12_BACK2BACKSETUP_SIZE     0x1
#define GC_USB_DOEPINT12_BACK2BACKSETUP_DEFAULT  0x0
#define GC_USB_DOEPINT12_BACK2BACKSETUP_OFFSET   0xc88
#define GC_USB_DOEPINT12_OUTPKTERR_LSB           0x8
#define GC_USB_DOEPINT12_OUTPKTERR_MASK          0x100
#define GC_USB_DOEPINT12_OUTPKTERR_SIZE          0x1
#define GC_USB_DOEPINT12_OUTPKTERR_DEFAULT       0x0
#define GC_USB_DOEPINT12_OUTPKTERR_OFFSET        0xc88
#define GC_USB_DOEPINT12_BNAINTR_LSB             0x9
#define GC_USB_DOEPINT12_BNAINTR_MASK            0x200
#define GC_USB_DOEPINT12_BNAINTR_SIZE            0x1
#define GC_USB_DOEPINT12_BNAINTR_DEFAULT         0x0
#define GC_USB_DOEPINT12_BNAINTR_OFFSET          0xc88
#define GC_USB_DOEPINT12_PKTDRPSTS_LSB           0xb
#define GC_USB_DOEPINT12_PKTDRPSTS_MASK          0x800
#define GC_USB_DOEPINT12_PKTDRPSTS_SIZE          0x1
#define GC_USB_DOEPINT12_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DOEPINT12_PKTDRPSTS_OFFSET        0xc88
#define GC_USB_DOEPINT12_BBLEERR_LSB             0xc
#define GC_USB_DOEPINT12_BBLEERR_MASK            0x1000
#define GC_USB_DOEPINT12_BBLEERR_SIZE            0x1
#define GC_USB_DOEPINT12_BBLEERR_DEFAULT         0x0
#define GC_USB_DOEPINT12_BBLEERR_OFFSET          0xc88
#define GC_USB_DOEPINT12_NAKINTRPT_LSB           0xd
#define GC_USB_DOEPINT12_NAKINTRPT_MASK          0x2000
#define GC_USB_DOEPINT12_NAKINTRPT_SIZE          0x1
#define GC_USB_DOEPINT12_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT12_NAKINTRPT_OFFSET        0xc88
#define GC_USB_DOEPINT12_NYETINTRPT_LSB          0xe
#define GC_USB_DOEPINT12_NYETINTRPT_MASK         0x4000
#define GC_USB_DOEPINT12_NYETINTRPT_SIZE         0x1
#define GC_USB_DOEPINT12_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DOEPINT12_NYETINTRPT_OFFSET       0xc88
#define GC_USB_DOEPINT12_STUPPKTRCVD_LSB         0xf
#define GC_USB_DOEPINT12_STUPPKTRCVD_MASK        0x8000
#define GC_USB_DOEPINT12_STUPPKTRCVD_SIZE        0x1
#define GC_USB_DOEPINT12_STUPPKTRCVD_DEFAULT     0x0
#define GC_USB_DOEPINT12_STUPPKTRCVD_OFFSET      0xc88
#define GC_USB_DOEPTSIZ12_XFERSIZE_LSB           0x0
#define GC_USB_DOEPTSIZ12_XFERSIZE_MASK          0x7ffff
#define GC_USB_DOEPTSIZ12_XFERSIZE_SIZE          0x13
#define GC_USB_DOEPTSIZ12_XFERSIZE_DEFAULT       0x0
#define GC_USB_DOEPTSIZ12_XFERSIZE_OFFSET        0xc90
#define GC_USB_DOEPTSIZ12_PKTCNT_LSB             0x13
#define GC_USB_DOEPTSIZ12_PKTCNT_MASK            0x1ff80000
#define GC_USB_DOEPTSIZ12_PKTCNT_SIZE            0xa
#define GC_USB_DOEPTSIZ12_PKTCNT_DEFAULT         0x0
#define GC_USB_DOEPTSIZ12_PKTCNT_OFFSET          0xc90
#define GC_USB_DOEPTSIZ12_RXDPID_LSB             0x1d
#define GC_USB_DOEPTSIZ12_RXDPID_MASK            0x60000000
#define GC_USB_DOEPTSIZ12_RXDPID_SIZE            0x2
#define GC_USB_DOEPTSIZ12_RXDPID_DEFAULT         0x0
#define GC_USB_DOEPTSIZ12_RXDPID_OFFSET          0xc90
#define GC_USB_DOEPDMA12_DMAADDR_LSB             0x0
#define GC_USB_DOEPDMA12_DMAADDR_MASK            0xffffffff
#define GC_USB_DOEPDMA12_DMAADDR_SIZE            0x20
#define GC_USB_DOEPDMA12_DMAADDR_DEFAULT         0x0
#define GC_USB_DOEPDMA12_DMAADDR_OFFSET          0xc94
#define GC_USB_DOEPDMAB12_DMABUFFERADDR_LSB      0x0
#define GC_USB_DOEPDMAB12_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DOEPDMAB12_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DOEPDMAB12_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DOEPDMAB12_DMABUFFERADDR_OFFSET   0xc9c
#define GC_USB_DOEPCTL13_MPS_LSB                 0x0
#define GC_USB_DOEPCTL13_MPS_MASK                0x7ff
#define GC_USB_DOEPCTL13_MPS_SIZE                0xb
#define GC_USB_DOEPCTL13_MPS_DEFAULT             0x0
#define GC_USB_DOEPCTL13_MPS_OFFSET              0xca0
#define GC_USB_DOEPCTL13_USBACTEP_LSB            0xf
#define GC_USB_DOEPCTL13_USBACTEP_MASK           0x8000
#define GC_USB_DOEPCTL13_USBACTEP_SIZE           0x1
#define GC_USB_DOEPCTL13_USBACTEP_DEFAULT        0x0
#define GC_USB_DOEPCTL13_USBACTEP_OFFSET         0xca0
#define GC_USB_DOEPCTL13_DPID_LSB                0x10
#define GC_USB_DOEPCTL13_DPID_MASK               0x10000
#define GC_USB_DOEPCTL13_DPID_SIZE               0x1
#define GC_USB_DOEPCTL13_DPID_DEFAULT            0x0
#define GC_USB_DOEPCTL13_DPID_OFFSET             0xca0
#define GC_USB_DOEPCTL13_NAKSTS_LSB              0x11
#define GC_USB_DOEPCTL13_NAKSTS_MASK             0x20000
#define GC_USB_DOEPCTL13_NAKSTS_SIZE             0x1
#define GC_USB_DOEPCTL13_NAKSTS_DEFAULT          0x0
#define GC_USB_DOEPCTL13_NAKSTS_OFFSET           0xca0
#define GC_USB_DOEPCTL13_EPTYPE_LSB              0x12
#define GC_USB_DOEPCTL13_EPTYPE_MASK             0xc0000
#define GC_USB_DOEPCTL13_EPTYPE_SIZE             0x2
#define GC_USB_DOEPCTL13_EPTYPE_DEFAULT          0x0
#define GC_USB_DOEPCTL13_EPTYPE_OFFSET           0xca0
#define GC_USB_DOEPCTL13_SNP_LSB                 0x14
#define GC_USB_DOEPCTL13_SNP_MASK                0x100000
#define GC_USB_DOEPCTL13_SNP_SIZE                0x1
#define GC_USB_DOEPCTL13_SNP_DEFAULT             0x0
#define GC_USB_DOEPCTL13_SNP_OFFSET              0xca0
#define GC_USB_DOEPCTL13_STALL_LSB               0x15
#define GC_USB_DOEPCTL13_STALL_MASK              0x200000
#define GC_USB_DOEPCTL13_STALL_SIZE              0x1
#define GC_USB_DOEPCTL13_STALL_DEFAULT           0x0
#define GC_USB_DOEPCTL13_STALL_OFFSET            0xca0
#define GC_USB_DOEPCTL13_CNAK_LSB                0x1a
#define GC_USB_DOEPCTL13_CNAK_MASK               0x4000000
#define GC_USB_DOEPCTL13_CNAK_SIZE               0x1
#define GC_USB_DOEPCTL13_CNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL13_CNAK_OFFSET             0xca0
#define GC_USB_DOEPCTL13_SNAK_LSB                0x1b
#define GC_USB_DOEPCTL13_SNAK_MASK               0x8000000
#define GC_USB_DOEPCTL13_SNAK_SIZE               0x1
#define GC_USB_DOEPCTL13_SNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL13_SNAK_OFFSET             0xca0
#define GC_USB_DOEPCTL13_SETD0PID_LSB            0x1c
#define GC_USB_DOEPCTL13_SETD0PID_MASK           0x10000000
#define GC_USB_DOEPCTL13_SETD0PID_SIZE           0x1
#define GC_USB_DOEPCTL13_SETD0PID_DEFAULT        0x0
#define GC_USB_DOEPCTL13_SETD0PID_OFFSET         0xca0
#define GC_USB_DOEPCTL13_SETD1PID_LSB            0x1d
#define GC_USB_DOEPCTL13_SETD1PID_MASK           0x20000000
#define GC_USB_DOEPCTL13_SETD1PID_SIZE           0x1
#define GC_USB_DOEPCTL13_SETD1PID_DEFAULT        0x0
#define GC_USB_DOEPCTL13_SETD1PID_OFFSET         0xca0
#define GC_USB_DOEPCTL13_EPDIS_LSB               0x1e
#define GC_USB_DOEPCTL13_EPDIS_MASK              0x40000000
#define GC_USB_DOEPCTL13_EPDIS_SIZE              0x1
#define GC_USB_DOEPCTL13_EPDIS_DEFAULT           0x0
#define GC_USB_DOEPCTL13_EPDIS_OFFSET            0xca0
#define GC_USB_DOEPCTL13_EPENA_LSB               0x1f
#define GC_USB_DOEPCTL13_EPENA_MASK              0x80000000
#define GC_USB_DOEPCTL13_EPENA_SIZE              0x1
#define GC_USB_DOEPCTL13_EPENA_DEFAULT           0x0
#define GC_USB_DOEPCTL13_EPENA_OFFSET            0xca0
#define GC_USB_DOEPINT13_XFERCOMPL_LSB           0x0
#define GC_USB_DOEPINT13_XFERCOMPL_MASK          0x1
#define GC_USB_DOEPINT13_XFERCOMPL_SIZE          0x1
#define GC_USB_DOEPINT13_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DOEPINT13_XFERCOMPL_OFFSET        0xca8
#define GC_USB_DOEPINT13_EPDISBLD_LSB            0x1
#define GC_USB_DOEPINT13_EPDISBLD_MASK           0x2
#define GC_USB_DOEPINT13_EPDISBLD_SIZE           0x1
#define GC_USB_DOEPINT13_EPDISBLD_DEFAULT        0x0
#define GC_USB_DOEPINT13_EPDISBLD_OFFSET         0xca8
#define GC_USB_DOEPINT13_AHBERR_LSB              0x2
#define GC_USB_DOEPINT13_AHBERR_MASK             0x4
#define GC_USB_DOEPINT13_AHBERR_SIZE             0x1
#define GC_USB_DOEPINT13_AHBERR_DEFAULT          0x0
#define GC_USB_DOEPINT13_AHBERR_OFFSET           0xca8
#define GC_USB_DOEPINT13_SETUP_LSB               0x3
#define GC_USB_DOEPINT13_SETUP_MASK              0x8
#define GC_USB_DOEPINT13_SETUP_SIZE              0x1
#define GC_USB_DOEPINT13_SETUP_DEFAULT           0x0
#define GC_USB_DOEPINT13_SETUP_OFFSET            0xca8
#define GC_USB_DOEPINT13_OUTTKNEPDIS_LSB         0x4
#define GC_USB_DOEPINT13_OUTTKNEPDIS_MASK        0x10
#define GC_USB_DOEPINT13_OUTTKNEPDIS_SIZE        0x1
#define GC_USB_DOEPINT13_OUTTKNEPDIS_DEFAULT     0x0
#define GC_USB_DOEPINT13_OUTTKNEPDIS_OFFSET      0xca8
#define GC_USB_DOEPINT13_STSPHSERCVD_LSB         0x5
#define GC_USB_DOEPINT13_STSPHSERCVD_MASK        0x20
#define GC_USB_DOEPINT13_STSPHSERCVD_SIZE        0x1
#define GC_USB_DOEPINT13_STSPHSERCVD_DEFAULT     0x0
#define GC_USB_DOEPINT13_STSPHSERCVD_OFFSET      0xca8
#define GC_USB_DOEPINT13_BACK2BACKSETUP_LSB      0x6
#define GC_USB_DOEPINT13_BACK2BACKSETUP_MASK     0x40
#define GC_USB_DOEPINT13_BACK2BACKSETUP_SIZE     0x1
#define GC_USB_DOEPINT13_BACK2BACKSETUP_DEFAULT  0x0
#define GC_USB_DOEPINT13_BACK2BACKSETUP_OFFSET   0xca8
#define GC_USB_DOEPINT13_OUTPKTERR_LSB           0x8
#define GC_USB_DOEPINT13_OUTPKTERR_MASK          0x100
#define GC_USB_DOEPINT13_OUTPKTERR_SIZE          0x1
#define GC_USB_DOEPINT13_OUTPKTERR_DEFAULT       0x0
#define GC_USB_DOEPINT13_OUTPKTERR_OFFSET        0xca8
#define GC_USB_DOEPINT13_BNAINTR_LSB             0x9
#define GC_USB_DOEPINT13_BNAINTR_MASK            0x200
#define GC_USB_DOEPINT13_BNAINTR_SIZE            0x1
#define GC_USB_DOEPINT13_BNAINTR_DEFAULT         0x0
#define GC_USB_DOEPINT13_BNAINTR_OFFSET          0xca8
#define GC_USB_DOEPINT13_PKTDRPSTS_LSB           0xb
#define GC_USB_DOEPINT13_PKTDRPSTS_MASK          0x800
#define GC_USB_DOEPINT13_PKTDRPSTS_SIZE          0x1
#define GC_USB_DOEPINT13_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DOEPINT13_PKTDRPSTS_OFFSET        0xca8
#define GC_USB_DOEPINT13_BBLEERR_LSB             0xc
#define GC_USB_DOEPINT13_BBLEERR_MASK            0x1000
#define GC_USB_DOEPINT13_BBLEERR_SIZE            0x1
#define GC_USB_DOEPINT13_BBLEERR_DEFAULT         0x0
#define GC_USB_DOEPINT13_BBLEERR_OFFSET          0xca8
#define GC_USB_DOEPINT13_NAKINTRPT_LSB           0xd
#define GC_USB_DOEPINT13_NAKINTRPT_MASK          0x2000
#define GC_USB_DOEPINT13_NAKINTRPT_SIZE          0x1
#define GC_USB_DOEPINT13_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT13_NAKINTRPT_OFFSET        0xca8
#define GC_USB_DOEPINT13_NYETINTRPT_LSB          0xe
#define GC_USB_DOEPINT13_NYETINTRPT_MASK         0x4000
#define GC_USB_DOEPINT13_NYETINTRPT_SIZE         0x1
#define GC_USB_DOEPINT13_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DOEPINT13_NYETINTRPT_OFFSET       0xca8
#define GC_USB_DOEPINT13_STUPPKTRCVD_LSB         0xf
#define GC_USB_DOEPINT13_STUPPKTRCVD_MASK        0x8000
#define GC_USB_DOEPINT13_STUPPKTRCVD_SIZE        0x1
#define GC_USB_DOEPINT13_STUPPKTRCVD_DEFAULT     0x0
#define GC_USB_DOEPINT13_STUPPKTRCVD_OFFSET      0xca8
#define GC_USB_DOEPTSIZ13_XFERSIZE_LSB           0x0
#define GC_USB_DOEPTSIZ13_XFERSIZE_MASK          0x7ffff
#define GC_USB_DOEPTSIZ13_XFERSIZE_SIZE          0x13
#define GC_USB_DOEPTSIZ13_XFERSIZE_DEFAULT       0x0
#define GC_USB_DOEPTSIZ13_XFERSIZE_OFFSET        0xcb0
#define GC_USB_DOEPTSIZ13_PKTCNT_LSB             0x13
#define GC_USB_DOEPTSIZ13_PKTCNT_MASK            0x1ff80000
#define GC_USB_DOEPTSIZ13_PKTCNT_SIZE            0xa
#define GC_USB_DOEPTSIZ13_PKTCNT_DEFAULT         0x0
#define GC_USB_DOEPTSIZ13_PKTCNT_OFFSET          0xcb0
#define GC_USB_DOEPTSIZ13_RXDPID_LSB             0x1d
#define GC_USB_DOEPTSIZ13_RXDPID_MASK            0x60000000
#define GC_USB_DOEPTSIZ13_RXDPID_SIZE            0x2
#define GC_USB_DOEPTSIZ13_RXDPID_DEFAULT         0x0
#define GC_USB_DOEPTSIZ13_RXDPID_OFFSET          0xcb0
#define GC_USB_DOEPDMA13_DMAADDR_LSB             0x0
#define GC_USB_DOEPDMA13_DMAADDR_MASK            0xffffffff
#define GC_USB_DOEPDMA13_DMAADDR_SIZE            0x20
#define GC_USB_DOEPDMA13_DMAADDR_DEFAULT         0x0
#define GC_USB_DOEPDMA13_DMAADDR_OFFSET          0xcb4
#define GC_USB_DOEPDMAB13_DMABUFFERADDR_LSB      0x0
#define GC_USB_DOEPDMAB13_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DOEPDMAB13_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DOEPDMAB13_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DOEPDMAB13_DMABUFFERADDR_OFFSET   0xcbc
#define GC_USB_DOEPCTL14_MPS_LSB                 0x0
#define GC_USB_DOEPCTL14_MPS_MASK                0x7ff
#define GC_USB_DOEPCTL14_MPS_SIZE                0xb
#define GC_USB_DOEPCTL14_MPS_DEFAULT             0x0
#define GC_USB_DOEPCTL14_MPS_OFFSET              0xcc0
#define GC_USB_DOEPCTL14_USBACTEP_LSB            0xf
#define GC_USB_DOEPCTL14_USBACTEP_MASK           0x8000
#define GC_USB_DOEPCTL14_USBACTEP_SIZE           0x1
#define GC_USB_DOEPCTL14_USBACTEP_DEFAULT        0x0
#define GC_USB_DOEPCTL14_USBACTEP_OFFSET         0xcc0
#define GC_USB_DOEPCTL14_DPID_LSB                0x10
#define GC_USB_DOEPCTL14_DPID_MASK               0x10000
#define GC_USB_DOEPCTL14_DPID_SIZE               0x1
#define GC_USB_DOEPCTL14_DPID_DEFAULT            0x0
#define GC_USB_DOEPCTL14_DPID_OFFSET             0xcc0
#define GC_USB_DOEPCTL14_NAKSTS_LSB              0x11
#define GC_USB_DOEPCTL14_NAKSTS_MASK             0x20000
#define GC_USB_DOEPCTL14_NAKSTS_SIZE             0x1
#define GC_USB_DOEPCTL14_NAKSTS_DEFAULT          0x0
#define GC_USB_DOEPCTL14_NAKSTS_OFFSET           0xcc0
#define GC_USB_DOEPCTL14_EPTYPE_LSB              0x12
#define GC_USB_DOEPCTL14_EPTYPE_MASK             0xc0000
#define GC_USB_DOEPCTL14_EPTYPE_SIZE             0x2
#define GC_USB_DOEPCTL14_EPTYPE_DEFAULT          0x0
#define GC_USB_DOEPCTL14_EPTYPE_OFFSET           0xcc0
#define GC_USB_DOEPCTL14_SNP_LSB                 0x14
#define GC_USB_DOEPCTL14_SNP_MASK                0x100000
#define GC_USB_DOEPCTL14_SNP_SIZE                0x1
#define GC_USB_DOEPCTL14_SNP_DEFAULT             0x0
#define GC_USB_DOEPCTL14_SNP_OFFSET              0xcc0
#define GC_USB_DOEPCTL14_STALL_LSB               0x15
#define GC_USB_DOEPCTL14_STALL_MASK              0x200000
#define GC_USB_DOEPCTL14_STALL_SIZE              0x1
#define GC_USB_DOEPCTL14_STALL_DEFAULT           0x0
#define GC_USB_DOEPCTL14_STALL_OFFSET            0xcc0
#define GC_USB_DOEPCTL14_CNAK_LSB                0x1a
#define GC_USB_DOEPCTL14_CNAK_MASK               0x4000000
#define GC_USB_DOEPCTL14_CNAK_SIZE               0x1
#define GC_USB_DOEPCTL14_CNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL14_CNAK_OFFSET             0xcc0
#define GC_USB_DOEPCTL14_SNAK_LSB                0x1b
#define GC_USB_DOEPCTL14_SNAK_MASK               0x8000000
#define GC_USB_DOEPCTL14_SNAK_SIZE               0x1
#define GC_USB_DOEPCTL14_SNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL14_SNAK_OFFSET             0xcc0
#define GC_USB_DOEPCTL14_SETD0PID_LSB            0x1c
#define GC_USB_DOEPCTL14_SETD0PID_MASK           0x10000000
#define GC_USB_DOEPCTL14_SETD0PID_SIZE           0x1
#define GC_USB_DOEPCTL14_SETD0PID_DEFAULT        0x0
#define GC_USB_DOEPCTL14_SETD0PID_OFFSET         0xcc0
#define GC_USB_DOEPCTL14_SETD1PID_LSB            0x1d
#define GC_USB_DOEPCTL14_SETD1PID_MASK           0x20000000
#define GC_USB_DOEPCTL14_SETD1PID_SIZE           0x1
#define GC_USB_DOEPCTL14_SETD1PID_DEFAULT        0x0
#define GC_USB_DOEPCTL14_SETD1PID_OFFSET         0xcc0
#define GC_USB_DOEPCTL14_EPDIS_LSB               0x1e
#define GC_USB_DOEPCTL14_EPDIS_MASK              0x40000000
#define GC_USB_DOEPCTL14_EPDIS_SIZE              0x1
#define GC_USB_DOEPCTL14_EPDIS_DEFAULT           0x0
#define GC_USB_DOEPCTL14_EPDIS_OFFSET            0xcc0
#define GC_USB_DOEPCTL14_EPENA_LSB               0x1f
#define GC_USB_DOEPCTL14_EPENA_MASK              0x80000000
#define GC_USB_DOEPCTL14_EPENA_SIZE              0x1
#define GC_USB_DOEPCTL14_EPENA_DEFAULT           0x0
#define GC_USB_DOEPCTL14_EPENA_OFFSET            0xcc0
#define GC_USB_DOEPINT14_XFERCOMPL_LSB           0x0
#define GC_USB_DOEPINT14_XFERCOMPL_MASK          0x1
#define GC_USB_DOEPINT14_XFERCOMPL_SIZE          0x1
#define GC_USB_DOEPINT14_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DOEPINT14_XFERCOMPL_OFFSET        0xcc8
#define GC_USB_DOEPINT14_EPDISBLD_LSB            0x1
#define GC_USB_DOEPINT14_EPDISBLD_MASK           0x2
#define GC_USB_DOEPINT14_EPDISBLD_SIZE           0x1
#define GC_USB_DOEPINT14_EPDISBLD_DEFAULT        0x0
#define GC_USB_DOEPINT14_EPDISBLD_OFFSET         0xcc8
#define GC_USB_DOEPINT14_AHBERR_LSB              0x2
#define GC_USB_DOEPINT14_AHBERR_MASK             0x4
#define GC_USB_DOEPINT14_AHBERR_SIZE             0x1
#define GC_USB_DOEPINT14_AHBERR_DEFAULT          0x0
#define GC_USB_DOEPINT14_AHBERR_OFFSET           0xcc8
#define GC_USB_DOEPINT14_SETUP_LSB               0x3
#define GC_USB_DOEPINT14_SETUP_MASK              0x8
#define GC_USB_DOEPINT14_SETUP_SIZE              0x1
#define GC_USB_DOEPINT14_SETUP_DEFAULT           0x0
#define GC_USB_DOEPINT14_SETUP_OFFSET            0xcc8
#define GC_USB_DOEPINT14_OUTTKNEPDIS_LSB         0x4
#define GC_USB_DOEPINT14_OUTTKNEPDIS_MASK        0x10
#define GC_USB_DOEPINT14_OUTTKNEPDIS_SIZE        0x1
#define GC_USB_DOEPINT14_OUTTKNEPDIS_DEFAULT     0x0
#define GC_USB_DOEPINT14_OUTTKNEPDIS_OFFSET      0xcc8
#define GC_USB_DOEPINT14_STSPHSERCVD_LSB         0x5
#define GC_USB_DOEPINT14_STSPHSERCVD_MASK        0x20
#define GC_USB_DOEPINT14_STSPHSERCVD_SIZE        0x1
#define GC_USB_DOEPINT14_STSPHSERCVD_DEFAULT     0x0
#define GC_USB_DOEPINT14_STSPHSERCVD_OFFSET      0xcc8
#define GC_USB_DOEPINT14_BACK2BACKSETUP_LSB      0x6
#define GC_USB_DOEPINT14_BACK2BACKSETUP_MASK     0x40
#define GC_USB_DOEPINT14_BACK2BACKSETUP_SIZE     0x1
#define GC_USB_DOEPINT14_BACK2BACKSETUP_DEFAULT  0x0
#define GC_USB_DOEPINT14_BACK2BACKSETUP_OFFSET   0xcc8
#define GC_USB_DOEPINT14_OUTPKTERR_LSB           0x8
#define GC_USB_DOEPINT14_OUTPKTERR_MASK          0x100
#define GC_USB_DOEPINT14_OUTPKTERR_SIZE          0x1
#define GC_USB_DOEPINT14_OUTPKTERR_DEFAULT       0x0
#define GC_USB_DOEPINT14_OUTPKTERR_OFFSET        0xcc8
#define GC_USB_DOEPINT14_BNAINTR_LSB             0x9
#define GC_USB_DOEPINT14_BNAINTR_MASK            0x200
#define GC_USB_DOEPINT14_BNAINTR_SIZE            0x1
#define GC_USB_DOEPINT14_BNAINTR_DEFAULT         0x0
#define GC_USB_DOEPINT14_BNAINTR_OFFSET          0xcc8
#define GC_USB_DOEPINT14_PKTDRPSTS_LSB           0xb
#define GC_USB_DOEPINT14_PKTDRPSTS_MASK          0x800
#define GC_USB_DOEPINT14_PKTDRPSTS_SIZE          0x1
#define GC_USB_DOEPINT14_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DOEPINT14_PKTDRPSTS_OFFSET        0xcc8
#define GC_USB_DOEPINT14_BBLEERR_LSB             0xc
#define GC_USB_DOEPINT14_BBLEERR_MASK            0x1000
#define GC_USB_DOEPINT14_BBLEERR_SIZE            0x1
#define GC_USB_DOEPINT14_BBLEERR_DEFAULT         0x0
#define GC_USB_DOEPINT14_BBLEERR_OFFSET          0xcc8
#define GC_USB_DOEPINT14_NAKINTRPT_LSB           0xd
#define GC_USB_DOEPINT14_NAKINTRPT_MASK          0x2000
#define GC_USB_DOEPINT14_NAKINTRPT_SIZE          0x1
#define GC_USB_DOEPINT14_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT14_NAKINTRPT_OFFSET        0xcc8
#define GC_USB_DOEPINT14_NYETINTRPT_LSB          0xe
#define GC_USB_DOEPINT14_NYETINTRPT_MASK         0x4000
#define GC_USB_DOEPINT14_NYETINTRPT_SIZE         0x1
#define GC_USB_DOEPINT14_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DOEPINT14_NYETINTRPT_OFFSET       0xcc8
#define GC_USB_DOEPINT14_STUPPKTRCVD_LSB         0xf
#define GC_USB_DOEPINT14_STUPPKTRCVD_MASK        0x8000
#define GC_USB_DOEPINT14_STUPPKTRCVD_SIZE        0x1
#define GC_USB_DOEPINT14_STUPPKTRCVD_DEFAULT     0x0
#define GC_USB_DOEPINT14_STUPPKTRCVD_OFFSET      0xcc8
#define GC_USB_DOEPTSIZ14_XFERSIZE_LSB           0x0
#define GC_USB_DOEPTSIZ14_XFERSIZE_MASK          0x7ffff
#define GC_USB_DOEPTSIZ14_XFERSIZE_SIZE          0x13
#define GC_USB_DOEPTSIZ14_XFERSIZE_DEFAULT       0x0
#define GC_USB_DOEPTSIZ14_XFERSIZE_OFFSET        0xcd0
#define GC_USB_DOEPTSIZ14_PKTCNT_LSB             0x13
#define GC_USB_DOEPTSIZ14_PKTCNT_MASK            0x1ff80000
#define GC_USB_DOEPTSIZ14_PKTCNT_SIZE            0xa
#define GC_USB_DOEPTSIZ14_PKTCNT_DEFAULT         0x0
#define GC_USB_DOEPTSIZ14_PKTCNT_OFFSET          0xcd0
#define GC_USB_DOEPTSIZ14_RXDPID_LSB             0x1d
#define GC_USB_DOEPTSIZ14_RXDPID_MASK            0x60000000
#define GC_USB_DOEPTSIZ14_RXDPID_SIZE            0x2
#define GC_USB_DOEPTSIZ14_RXDPID_DEFAULT         0x0
#define GC_USB_DOEPTSIZ14_RXDPID_OFFSET          0xcd0
#define GC_USB_DOEPDMA14_DMAADDR_LSB             0x0
#define GC_USB_DOEPDMA14_DMAADDR_MASK            0xffffffff
#define GC_USB_DOEPDMA14_DMAADDR_SIZE            0x20
#define GC_USB_DOEPDMA14_DMAADDR_DEFAULT         0x0
#define GC_USB_DOEPDMA14_DMAADDR_OFFSET          0xcd4
#define GC_USB_DOEPDMAB14_DMABUFFERADDR_LSB      0x0
#define GC_USB_DOEPDMAB14_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DOEPDMAB14_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DOEPDMAB14_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DOEPDMAB14_DMABUFFERADDR_OFFSET   0xcdc
#define GC_USB_DOEPCTL15_MPS_LSB                 0x0
#define GC_USB_DOEPCTL15_MPS_MASK                0x7ff
#define GC_USB_DOEPCTL15_MPS_SIZE                0xb
#define GC_USB_DOEPCTL15_MPS_DEFAULT             0x0
#define GC_USB_DOEPCTL15_MPS_OFFSET              0xce0
#define GC_USB_DOEPCTL15_USBACTEP_LSB            0xf
#define GC_USB_DOEPCTL15_USBACTEP_MASK           0x8000
#define GC_USB_DOEPCTL15_USBACTEP_SIZE           0x1
#define GC_USB_DOEPCTL15_USBACTEP_DEFAULT        0x0
#define GC_USB_DOEPCTL15_USBACTEP_OFFSET         0xce0
#define GC_USB_DOEPCTL15_DPID_LSB                0x10
#define GC_USB_DOEPCTL15_DPID_MASK               0x10000
#define GC_USB_DOEPCTL15_DPID_SIZE               0x1
#define GC_USB_DOEPCTL15_DPID_DEFAULT            0x0
#define GC_USB_DOEPCTL15_DPID_OFFSET             0xce0
#define GC_USB_DOEPCTL15_NAKSTS_LSB              0x11
#define GC_USB_DOEPCTL15_NAKSTS_MASK             0x20000
#define GC_USB_DOEPCTL15_NAKSTS_SIZE             0x1
#define GC_USB_DOEPCTL15_NAKSTS_DEFAULT          0x0
#define GC_USB_DOEPCTL15_NAKSTS_OFFSET           0xce0
#define GC_USB_DOEPCTL15_EPTYPE_LSB              0x12
#define GC_USB_DOEPCTL15_EPTYPE_MASK             0xc0000
#define GC_USB_DOEPCTL15_EPTYPE_SIZE             0x2
#define GC_USB_DOEPCTL15_EPTYPE_DEFAULT          0x0
#define GC_USB_DOEPCTL15_EPTYPE_OFFSET           0xce0
#define GC_USB_DOEPCTL15_SNP_LSB                 0x14
#define GC_USB_DOEPCTL15_SNP_MASK                0x100000
#define GC_USB_DOEPCTL15_SNP_SIZE                0x1
#define GC_USB_DOEPCTL15_SNP_DEFAULT             0x0
#define GC_USB_DOEPCTL15_SNP_OFFSET              0xce0
#define GC_USB_DOEPCTL15_STALL_LSB               0x15
#define GC_USB_DOEPCTL15_STALL_MASK              0x200000
#define GC_USB_DOEPCTL15_STALL_SIZE              0x1
#define GC_USB_DOEPCTL15_STALL_DEFAULT           0x0
#define GC_USB_DOEPCTL15_STALL_OFFSET            0xce0
#define GC_USB_DOEPCTL15_CNAK_LSB                0x1a
#define GC_USB_DOEPCTL15_CNAK_MASK               0x4000000
#define GC_USB_DOEPCTL15_CNAK_SIZE               0x1
#define GC_USB_DOEPCTL15_CNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL15_CNAK_OFFSET             0xce0
#define GC_USB_DOEPCTL15_SNAK_LSB                0x1b
#define GC_USB_DOEPCTL15_SNAK_MASK               0x8000000
#define GC_USB_DOEPCTL15_SNAK_SIZE               0x1
#define GC_USB_DOEPCTL15_SNAK_DEFAULT            0x0
#define GC_USB_DOEPCTL15_SNAK_OFFSET             0xce0
#define GC_USB_DOEPCTL15_SETD0PID_LSB            0x1c
#define GC_USB_DOEPCTL15_SETD0PID_MASK           0x10000000
#define GC_USB_DOEPCTL15_SETD0PID_SIZE           0x1
#define GC_USB_DOEPCTL15_SETD0PID_DEFAULT        0x0
#define GC_USB_DOEPCTL15_SETD0PID_OFFSET         0xce0
#define GC_USB_DOEPCTL15_SETD1PID_LSB            0x1d
#define GC_USB_DOEPCTL15_SETD1PID_MASK           0x20000000
#define GC_USB_DOEPCTL15_SETD1PID_SIZE           0x1
#define GC_USB_DOEPCTL15_SETD1PID_DEFAULT        0x0
#define GC_USB_DOEPCTL15_SETD1PID_OFFSET         0xce0
#define GC_USB_DOEPCTL15_EPDIS_LSB               0x1e
#define GC_USB_DOEPCTL15_EPDIS_MASK              0x40000000
#define GC_USB_DOEPCTL15_EPDIS_SIZE              0x1
#define GC_USB_DOEPCTL15_EPDIS_DEFAULT           0x0
#define GC_USB_DOEPCTL15_EPDIS_OFFSET            0xce0
#define GC_USB_DOEPCTL15_EPENA_LSB               0x1f
#define GC_USB_DOEPCTL15_EPENA_MASK              0x80000000
#define GC_USB_DOEPCTL15_EPENA_SIZE              0x1
#define GC_USB_DOEPCTL15_EPENA_DEFAULT           0x0
#define GC_USB_DOEPCTL15_EPENA_OFFSET            0xce0
#define GC_USB_DOEPINT15_XFERCOMPL_LSB           0x0
#define GC_USB_DOEPINT15_XFERCOMPL_MASK          0x1
#define GC_USB_DOEPINT15_XFERCOMPL_SIZE          0x1
#define GC_USB_DOEPINT15_XFERCOMPL_DEFAULT       0x0
#define GC_USB_DOEPINT15_XFERCOMPL_OFFSET        0xce8
#define GC_USB_DOEPINT15_EPDISBLD_LSB            0x1
#define GC_USB_DOEPINT15_EPDISBLD_MASK           0x2
#define GC_USB_DOEPINT15_EPDISBLD_SIZE           0x1
#define GC_USB_DOEPINT15_EPDISBLD_DEFAULT        0x0
#define GC_USB_DOEPINT15_EPDISBLD_OFFSET         0xce8
#define GC_USB_DOEPINT15_AHBERR_LSB              0x2
#define GC_USB_DOEPINT15_AHBERR_MASK             0x4
#define GC_USB_DOEPINT15_AHBERR_SIZE             0x1
#define GC_USB_DOEPINT15_AHBERR_DEFAULT          0x0
#define GC_USB_DOEPINT15_AHBERR_OFFSET           0xce8
#define GC_USB_DOEPINT15_SETUP_LSB               0x3
#define GC_USB_DOEPINT15_SETUP_MASK              0x8
#define GC_USB_DOEPINT15_SETUP_SIZE              0x1
#define GC_USB_DOEPINT15_SETUP_DEFAULT           0x0
#define GC_USB_DOEPINT15_SETUP_OFFSET            0xce8
#define GC_USB_DOEPINT15_OUTTKNEPDIS_LSB         0x4
#define GC_USB_DOEPINT15_OUTTKNEPDIS_MASK        0x10
#define GC_USB_DOEPINT15_OUTTKNEPDIS_SIZE        0x1
#define GC_USB_DOEPINT15_OUTTKNEPDIS_DEFAULT     0x0
#define GC_USB_DOEPINT15_OUTTKNEPDIS_OFFSET      0xce8
#define GC_USB_DOEPINT15_STSPHSERCVD_LSB         0x5
#define GC_USB_DOEPINT15_STSPHSERCVD_MASK        0x20
#define GC_USB_DOEPINT15_STSPHSERCVD_SIZE        0x1
#define GC_USB_DOEPINT15_STSPHSERCVD_DEFAULT     0x0
#define GC_USB_DOEPINT15_STSPHSERCVD_OFFSET      0xce8
#define GC_USB_DOEPINT15_BACK2BACKSETUP_LSB      0x6
#define GC_USB_DOEPINT15_BACK2BACKSETUP_MASK     0x40
#define GC_USB_DOEPINT15_BACK2BACKSETUP_SIZE     0x1
#define GC_USB_DOEPINT15_BACK2BACKSETUP_DEFAULT  0x0
#define GC_USB_DOEPINT15_BACK2BACKSETUP_OFFSET   0xce8
#define GC_USB_DOEPINT15_OUTPKTERR_LSB           0x8
#define GC_USB_DOEPINT15_OUTPKTERR_MASK          0x100
#define GC_USB_DOEPINT15_OUTPKTERR_SIZE          0x1
#define GC_USB_DOEPINT15_OUTPKTERR_DEFAULT       0x0
#define GC_USB_DOEPINT15_OUTPKTERR_OFFSET        0xce8
#define GC_USB_DOEPINT15_BNAINTR_LSB             0x9
#define GC_USB_DOEPINT15_BNAINTR_MASK            0x200
#define GC_USB_DOEPINT15_BNAINTR_SIZE            0x1
#define GC_USB_DOEPINT15_BNAINTR_DEFAULT         0x0
#define GC_USB_DOEPINT15_BNAINTR_OFFSET          0xce8
#define GC_USB_DOEPINT15_PKTDRPSTS_LSB           0xb
#define GC_USB_DOEPINT15_PKTDRPSTS_MASK          0x800
#define GC_USB_DOEPINT15_PKTDRPSTS_SIZE          0x1
#define GC_USB_DOEPINT15_PKTDRPSTS_DEFAULT       0x0
#define GC_USB_DOEPINT15_PKTDRPSTS_OFFSET        0xce8
#define GC_USB_DOEPINT15_BBLEERR_LSB             0xc
#define GC_USB_DOEPINT15_BBLEERR_MASK            0x1000
#define GC_USB_DOEPINT15_BBLEERR_SIZE            0x1
#define GC_USB_DOEPINT15_BBLEERR_DEFAULT         0x0
#define GC_USB_DOEPINT15_BBLEERR_OFFSET          0xce8
#define GC_USB_DOEPINT15_NAKINTRPT_LSB           0xd
#define GC_USB_DOEPINT15_NAKINTRPT_MASK          0x2000
#define GC_USB_DOEPINT15_NAKINTRPT_SIZE          0x1
#define GC_USB_DOEPINT15_NAKINTRPT_DEFAULT       0x0
#define GC_USB_DOEPINT15_NAKINTRPT_OFFSET        0xce8
#define GC_USB_DOEPINT15_NYETINTRPT_LSB          0xe
#define GC_USB_DOEPINT15_NYETINTRPT_MASK         0x4000
#define GC_USB_DOEPINT15_NYETINTRPT_SIZE         0x1
#define GC_USB_DOEPINT15_NYETINTRPT_DEFAULT      0x0
#define GC_USB_DOEPINT15_NYETINTRPT_OFFSET       0xce8
#define GC_USB_DOEPINT15_STUPPKTRCVD_LSB         0xf
#define GC_USB_DOEPINT15_STUPPKTRCVD_MASK        0x8000
#define GC_USB_DOEPINT15_STUPPKTRCVD_SIZE        0x1
#define GC_USB_DOEPINT15_STUPPKTRCVD_DEFAULT     0x0
#define GC_USB_DOEPINT15_STUPPKTRCVD_OFFSET      0xce8
#define GC_USB_DOEPTSIZ15_XFERSIZE_LSB           0x0
#define GC_USB_DOEPTSIZ15_XFERSIZE_MASK          0x7ffff
#define GC_USB_DOEPTSIZ15_XFERSIZE_SIZE          0x13
#define GC_USB_DOEPTSIZ15_XFERSIZE_DEFAULT       0x0
#define GC_USB_DOEPTSIZ15_XFERSIZE_OFFSET        0xcf0
#define GC_USB_DOEPTSIZ15_PKTCNT_LSB             0x13
#define GC_USB_DOEPTSIZ15_PKTCNT_MASK            0x1ff80000
#define GC_USB_DOEPTSIZ15_PKTCNT_SIZE            0xa
#define GC_USB_DOEPTSIZ15_PKTCNT_DEFAULT         0x0
#define GC_USB_DOEPTSIZ15_PKTCNT_OFFSET          0xcf0
#define GC_USB_DOEPTSIZ15_RXDPID_LSB             0x1d
#define GC_USB_DOEPTSIZ15_RXDPID_MASK            0x60000000
#define GC_USB_DOEPTSIZ15_RXDPID_SIZE            0x2
#define GC_USB_DOEPTSIZ15_RXDPID_DEFAULT         0x0
#define GC_USB_DOEPTSIZ15_RXDPID_OFFSET          0xcf0
#define GC_USB_DOEPDMA15_DMAADDR_LSB             0x0
#define GC_USB_DOEPDMA15_DMAADDR_MASK            0xffffffff
#define GC_USB_DOEPDMA15_DMAADDR_SIZE            0x20
#define GC_USB_DOEPDMA15_DMAADDR_DEFAULT         0x0
#define GC_USB_DOEPDMA15_DMAADDR_OFFSET          0xcf4
#define GC_USB_DOEPDMAB15_DMABUFFERADDR_LSB      0x0
#define GC_USB_DOEPDMAB15_DMABUFFERADDR_MASK     0xffffffff
#define GC_USB_DOEPDMAB15_DMABUFFERADDR_SIZE     0x20
#define GC_USB_DOEPDMAB15_DMABUFFERADDR_DEFAULT  0x0
#define GC_USB_DOEPDMAB15_DMABUFFERADDR_OFFSET   0xcfc
#define GC_USB_PCGCCTL_STOPPCLK_LSB              0x0
#define GC_USB_PCGCCTL_STOPPCLK_MASK             0x1
#define GC_USB_PCGCCTL_STOPPCLK_SIZE             0x1
#define GC_USB_PCGCCTL_STOPPCLK_DEFAULT          0x0
#define GC_USB_PCGCCTL_STOPPCLK_OFFSET           0xe00
#define GC_USB_PCGCCTL_GATEHCLK_LSB              0x1
#define GC_USB_PCGCCTL_GATEHCLK_MASK             0x2
#define GC_USB_PCGCCTL_GATEHCLK_SIZE             0x1
#define GC_USB_PCGCCTL_GATEHCLK_DEFAULT          0x0
#define GC_USB_PCGCCTL_GATEHCLK_OFFSET           0xe00
#define GC_USB_PCGCCTL_PWRCLMP_LSB               0x2
#define GC_USB_PCGCCTL_PWRCLMP_MASK              0x4
#define GC_USB_PCGCCTL_PWRCLMP_SIZE              0x1
#define GC_USB_PCGCCTL_PWRCLMP_DEFAULT           0x0
#define GC_USB_PCGCCTL_PWRCLMP_OFFSET            0xe00
#define GC_USB_PCGCCTL_RSTPDWNMODULE_LSB         0x3
#define GC_USB_PCGCCTL_RSTPDWNMODULE_MASK        0x8
#define GC_USB_PCGCCTL_RSTPDWNMODULE_SIZE        0x1
#define GC_USB_PCGCCTL_RSTPDWNMODULE_DEFAULT     0x0
#define GC_USB_PCGCCTL_RSTPDWNMODULE_OFFSET      0xe00
#define GC_USB_PCGCCTL_PHYSLEEP_LSB              0x6
#define GC_USB_PCGCCTL_PHYSLEEP_MASK             0x40
#define GC_USB_PCGCCTL_PHYSLEEP_SIZE             0x1
#define GC_USB_PCGCCTL_PHYSLEEP_DEFAULT          0x0
#define GC_USB_PCGCCTL_PHYSLEEP_OFFSET           0xe00
#define GC_USB_PCGCCTL_L1SUSPENDED_LSB           0x7
#define GC_USB_PCGCCTL_L1SUSPENDED_MASK          0x80
#define GC_USB_PCGCCTL_L1SUSPENDED_SIZE          0x1
#define GC_USB_PCGCCTL_L1SUSPENDED_DEFAULT       0x0
#define GC_USB_PCGCCTL_L1SUSPENDED_OFFSET        0xe00
#define GC_USB_DFIFO_SIZE                        0x1000


#endif /* __CHIP_STM32_USB_DWC_REGISTERS_H */
