/* -*- c++ -*- */
/*
 * Copyright 2003 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gr_FlowGraphImpl.h>
#include <iostream>
#include <VrSigProc.h>
#include <VrMultiTask.h>
#include <algorithm>

using std::cerr;


static void *
executor_loop (void *arg)
{
  gr_FlowGraphImpl	*fg = (gr_FlowGraphImpl *) arg;
  VrMultiTask		*mt = fg->mt ();
  
  try {
    while (fg->isRunning ()){
      mt->process ();
    }
  }
  catch (std::exception &e){
    std::cerr << "gr_FlowGraph: std library exception: " << e.what () << endl;
  }
  catch (...){
    std::cerr << "gr_FlowGraph: uncaught exception\n";
  }

  return 0;
}

// ----------------------------------------------------------------

gr_FlowGraphImpl::gr_FlowGraphImpl ()
{
  d_run = false;
  d_has_been_started = false;
  d_thread = 0;
  d_mt = new VrMultiTask ();
  d_sigprocs.resize (0);
}

gr_FlowGraphImpl::~gr_FlowGraphImpl ()
{
  stop ();

  // FIXME delete all the VrSigProcs and associated buffers and connectors

  delete d_mt;
}

bool
gr_FlowGraphImpl::start ()
{
  omni_mutex_lock l(d_mutex);
  
  if (isRunning ()){
    cerr << "gr_FlowGraph: start: already running\n";
    return false;
  }

  if (!d_has_been_started){
    d_has_been_started = true;
    d_mt->start ();		// only call VrMultiTask's start the first time through
  }

  // fork joinable thread
  
  d_run = true;
  d_thread = new omni_thread (executor_loop, this);	// create thread
  d_thread->start ();					// start it running

  return true;
}

void
gr_FlowGraphImpl::stop ()
{
  omni_mutex_lock l(d_mutex);

  if (!isRunning ()){
    cerr << "gr_FlowGraph: stop: already stopped\n";
    return;
  }

  d_run = false;	// signal executor_loop to stop looping...
  d_thread->join (0);	// deletes d_thread
  d_thread = 0;
}

bool
gr_FlowGraphImpl::connect (VrSigProc *upstream, VrSigProc *downstream)
{
  return connect (upstream, 0, downstream);
}

bool
gr_FlowGraphImpl::connect (VrSigProc *upstream, int upstream_port,
			   VrSigProc *downstream)
{
  omni_mutex_lock l(d_mutex);

  if (isRunning ()){
    cerr << "gr_FlowGraph: connect: currently running\n";
    return false;
  }

  if (d_has_been_started){
    cerr << "gr_FlowGraph: connect: sorry, can't call connect (...) after start ()\n";
    return false;
  }

  NWO_CONNECTN (upstream, upstream_port, downstream);

  add_sigproc (upstream);
  if (add_sigproc (downstream) && downstream->isSink ())
    d_mt->add (downstream);

  return true;
}

bool
gr_FlowGraphImpl::isRunning ()
{
  return d_run;
}

void
gr_FlowGraphImpl::wait ()
{
  // FIXME really ought to wait on a condition variable, but for now
  // this will work...

  while (isRunning ())
    sleep (1);
}

// if we haven't seen this VrSigProc yet add it to our vector and return true

bool 
gr_FlowGraphImpl::add_sigproc (VrSigProc *sp)
{
  std::vector<VrSigProc *>::iterator
    it = std::find (d_sigprocs.begin (), d_sigprocs.end (), sp);

  if (it == d_sigprocs.end ()){		// first time we've seen sp
    d_sigprocs.push_back (sp);
    return true;
  }

  return false;				// already seen sp
}
